# @source - Source Code Reference

Specify the original source location for generated or imported code.

## Syntax

```lua
---@source <source_location>
```

## Examples

```lua
-- Reference to original file
---@source lib/external/math_utils.lua
function calculateDistance(x1, y1, x2, y2)
    return math.sqrt((x2 - x1)^2 + (y2 - y1)^2)
end

-- Reference to specific line in source
---@source utils/helpers.lua:42
function formatCurrency(amount)
    return string.format("$%.2f", amount)
end

-- Reference to external library
---@source npm:lodash/isEqual
function deepEqual(a, b)
    -- Implementation copied from lodash
    return isEqual(a, b)
end

-- Reference to code generation tool
---@source generated/api_client.lua
---@source generator: openapi-codegen v1.2.3
---@class APIClient
local APIClient = {}

---@param endpoint string API endpoint
---@return table Response data
function APIClient:get(endpoint)
    return http.get(self.baseUrl .. endpoint)
end

-- Reference to template source
---@source templates/model.lua.template
---@class Model
local Model = {}

function Model:save()
    return database.save(self)
end

function Model:delete()
    return database.delete(self)
end

-- Reference to macro expansion
---@source macros/property.lua
---@macro PROPERTY(name, type)
function defineProperty(obj, name, type)
    obj["get" .. name] = function(self)
        return self["_" .. name]
    end
    obj["set" .. name] = function(self, value)
        self["_" .. name] = value
    end
end

-- Reference to third-party source
---@source github:user/repo/file.lua
---@source commit: abc123def456
function externalFunction()
    -- Code from external repository
end

-- Reference to copied implementation
---@source Stack Overflow answer #12345678
---@source https://stackoverflow.com/questions/12345678
function quickSort(arr, low, high)
    if low < high then
        local pi = partition(arr, low, high)
        quickSort(arr, low, pi - 1)
        quickSort(arr, pi + 1, high)
    end
end

-- Reference to specification implementation
---@source RFC 3986 URI Generic Syntax
---@source https://tools.ietf.org/html/rfc3986
function parseURI(uri)
    -- URI parsing implementation
    local scheme, authority, path, query, fragment = uri:match("^([^:/?#]+):?([^/?#]*)([^?#]*)%??([^#]*)#?(.*)")
    return {
        scheme = scheme,
        authority = authority,
        path = path,
        query = query,
        fragment = fragment
    }
end

-- Reference to auto-generated code
---@source auto-generated by protoc
---@source proto/user.proto
---@class UserProto
---@field id number
---@field name string
---@field email string
local UserProto = {}

-- Reference to transpiled code
---@source TypeScript: src/utils.ts
---@source transpiler: lua-typescript v2.1.0
function forEach(array, callback)
    for i, item in ipairs(array) do
        callback(item, i, array)
    end
end

-- Reference to vendor code
---@source vendor/json.lua
---@source version: 1.4.2
---@source license: MIT
local json = require("json")

-- Reference to build-time generation
---@source build/generated/constants.lua
---@source build script: scripts/generate_constants.sh
---@source build time: 2024-01-15 14:30:22
local constants = {
    VERSION = "1.0.0",
    BUILD_NUMBER = 42,
    COMMIT_HASH = "abc123def"
}

-- Reference to AI-generated code
---@source AI Assistant: GPT-4
---@source prompt: "Generate a binary search function"
function binarySearch(arr, target)
    local left, right = 1, #arr
    while left <= right do
        local mid = math.floor((left + right) / 2)
        if arr[mid] == target then
            return mid
        elseif arr[mid] < target then
            left = mid + 1
        else
            right = mid - 1
        end
    end
    return nil
end

-- Reference to migration source
---@source migration: v1.2.3 -> v1.3.0
---@source migration script: scripts/migrate_config.lua
function migrateConfig(oldConfig)
    local newConfig = {}
    newConfig.database = oldConfig.db
    newConfig.server = {
        port = oldConfig.port or 8080,
        host = oldConfig.host or "localhost"
    }
    return newConfig
end
```

## Features

1. **Source location tracking**
2. **Code provenance documentation**
3. **Generated code attribution**
4. **External reference linking**
5. **Build process tracking**
