/*
 * Copyright © Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 or 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef MIR_TEST_DOUBLES_FAKE_RENDERABLE_H_
#define MIR_TEST_DOUBLES_FAKE_RENDERABLE_H_

#include "stub_buffer.h"
#include <mir/graphics/renderable.h>
#define GLM_FORCE_RADIANS
#include <glm/gtc/matrix_transform.hpp>
#include <gmock/gmock.h>

namespace mir
{
namespace test
{
namespace doubles
{

class FakeRenderable : public graphics::Renderable
{
public:
    FakeRenderable(int x, int y, int width, int height)
        : FakeRenderable{geometry::Rectangle{{x,y},{width,height}}}
    {
    }
    FakeRenderable(geometry::Rectangle display_area)
        : FakeRenderable{display_area, 1.0f, true, std::nullopt}
    {
    }

    FakeRenderable(geometry::Rectangle display_area,
                   float opacity)
        : FakeRenderable{display_area, opacity, true, std::nullopt}
    {
    }

    FakeRenderable(geometry::Rectangle display_area,
                   geometry::Rectangles opaque_region)
        : FakeRenderable{display_area, 0.5f, true, opaque_region}
    {
        assert(
            display_area.contains(opaque_region.bounding_rectangle()) &&
            "Opaque regions cannot be larger than the display area");
    }

    FakeRenderable(
        geometry::Rectangle display_area,
        float opacity,
        bool rectangular,
        std::optional<geometry::Rectangles> opaque_region) :
        buf{std::make_shared<StubBuffer>()},
        rect(display_area),
        opacity(opacity),
        rectangular(rectangular),
        opaque_region_(opaque_region)
    {
    }

    ID id() const override
    {
        return this;
    }

    float alpha() const override
    {
        return opacity;
    }

    glm::mat4 transformation() const override
    {
        return glm::mat4(1);
    }

    MirOrientation orientation() const override
    {
        return mir_orientation_normal;
    }

    MirMirrorMode mirror_mode() const override
    {
        return mir_mirror_mode_none;
    }

    bool shaped() const override
    {
        return !rectangular;
    }

    void set_buffer(std::shared_ptr<graphics::Buffer> b)
    {
        buf = b;
    }

    std::shared_ptr<graphics::Buffer> buffer() const override
    {
        return buf;
    }

    geometry::Rectangle screen_position() const override
    {
        return rect;
    }

    geometry::RectangleD src_bounds() const override
    {
        return {{0, 0}, buf->size()};
    }

    std::optional<geometry::Rectangle> clip_area() const override
    {
        return std::optional<geometry::Rectangle>();
    }

    auto surface_if_any() const
        -> std::optional<mir::scene::Surface const*> override
    {
        return std::nullopt;
    }

    auto opaque_region() const -> std::optional<geometry::Rectangles> override
    {
        return opaque_region_;
    }

private:
    std::shared_ptr<graphics::Buffer> buf;
    mir::geometry::Rectangle rect;
    float opacity;
    bool rectangular;
    std::optional<mir::geometry::Rectangles> const opaque_region_;
};

} // namespace doubles
} // namespace test
} // namespace mir
#endif // MIR_TEST_DOUBLES_FAKE_RENDERABLE_H_
