/* devd-version.h.in
 *
 * Copyright 2018 Christian Hergert <chergert@redhat.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma once

#if !defined(DEVICED_INSIDE) && !defined(DEVICED_COMPILATION)
# error "Only <libdeviced.h> can be included directly."
#endif

/**
 * SECTION:devd-version
 * @title: Version Checking
 * @short_description: Conditionally include code based on Builder version
 *
 * This module provides various macros that may be used to build code based
 * on the version of Builder at build time.
 */

/**
 * DEVD_MAJOR_VERSION:
 *
 * libdeviced major version component (e.g. 1 if %DEVD_VERSION is 1.2.3)
 */
#define DEVD_MAJOR_VERSION (3)

/**
 * DEVD_MINOR_VERSION:
 *
 * libdeviced minor version component (e.g. 2 if %DEVD_VERSION is 1.2.3)
 */
#define DEVD_MINOR_VERSION (27)

/**
 * DEVD_MICRO_VERSION:
 *
 * libdeviced micro version component (e.g. 3 if %DEVD_VERSION is 1.2.3)
 */
#define DEVD_MICRO_VERSION (4)

/**
 * DEVD_VERSION
 *
 * libdeviced version.
 */
#define DEVD_VERSION (3.27.4)

/**
 * DEVD_VERSION_S:
 *
 * libdeviced version, encoded as a string, useful for printing and
 * concatenation.
 */
#define DEVD_VERSION_S "3.27.4"

#define DEVD_ENCODE_VERSION(major,minor,micro) \
        ((major) << 24 | (minor) << 16 | (micro) << 8)

/**
 * DEVD_VERSION_HEX:
 *
 * libdeviced version, encoded as an hexadecimal number, useful for
 * integer comparisons.
 */
#define DEVD_VERSION_HEX \
        (DEVD_ENCODE_VERSION (DEVD_MAJOR_VERSION, DEVD_MINOR_VERSION, DEVD_MICRO_VERSION))

/**
 * DEVD_CHECK_VERSION:
 * @major: required major version
 * @minor: required minor version
 * @micro: required micro version
 *
 * Compile-time version checking. Evaluates to %TRUE if the version
 * of libdeviced is greater than the required one.
 */
#define DEVD_CHECK_VERSION(major,minor,micro)   \
        (DEVD_MAJOR_VERSION > (major) || \
         (DEVD_MAJOR_VERSION == (major) && DEVD_MINOR_VERSION > (minor)) || \
         (DEVD_MAJOR_VERSION == (major) && DEVD_MINOR_VERSION == (minor) && \
          DEVD_MICRO_VERSION >= (micro)))

