/*
 * Project Ginger Base
 *
 * Copyright IBM Corp, 2015-2016
 *
 * Code derived from Project Kimchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var gingerbase = {

    widget: {},

    trackingTasks: [],

    /**
     *
     * Get host capabilities
     * suc: callback if succeed err: callback if failed
     */
    getCapabilities : function(suc, err, done) {
        done = typeof done !== 'undefined' ? done: function(){};
        wok.requestJSON({
            url : "plugins/gingerbase/host/capabilities",
            type : "GET",
            contentType : "application/json",
            dataType : "json",
            success: suc,
            error: err,
            complete: done
        });
    },

    /**
     * Get the i18 strings.
     */
    getI18n: function(suc, err, url, sync) {
        wok.requestJSON({
            url : url ? url : 'plugins/gingerbase/i18n.json',
            type : 'GET',
            resend: true,
            dataType : 'json',
            async : !sync,
            success : suc,
            error: err
        });
    },

    /**
     * Get the host static information.
     */
    getHost: function(suc, err) {
        wok.requestJSON({
            url : 'plugins/gingerbase/host',
            type : 'GET',
            resend: true,
            contentType : 'application/json',
            dataType : 'json',
            success : suc,
            error: err
        });
    },

    /**
     * Get the dynamic host stats (usually used for monitoring).
     */
    getHostStats : function(suc, err) {
        wok.requestJSON({
            url : 'plugins/gingerbase/host/stats',
            type : 'GET',
            contentType : 'application/json',
            headers: {'Wok-Robot': 'wok-robot'},
            dataType : 'json',
            success : suc,
            error: err
        });
    },

    /**
     * Get the historic host stats.
     */
    getHostStatsHistory : function(suc, err) {
        wok.requestJSON({
            url : 'plugins/gingerbase/host/stats/history',
            type : 'GET',
            resend: true,
            contentType : 'application/json',
            headers: {'Wok-Robot': 'wok-robot'},
            dataType : 'json',
            success : suc,
            error: err
        });
    },

    getTask : function(taskId, suc, err) {
        wok.requestJSON({
            url : 'plugins/gingerbase/tasks/' + encodeURIComponent(taskId),
            type : 'GET',
            async: false,
            contentType : 'application/json',
            dataType : 'json',
            success : suc,
            error : err
        });
    },

    getTasksByFilter : function(filter, suc, err, sync) {
        wok.requestJSON({
            url : 'plugins/gingerbase/tasks?' + filter,
            type : 'GET',
            contentType : 'application/json',
            dataType : 'json',
            async : !sync,
            success : suc,
            error : err
        });
    },

    listReports : function(suc, err) {
        wok.requestJSON({
            url : 'plugins/gingerbase/debugreports',
            type : 'GET',
            contentType : 'application/json',
            dataType : 'json',
            resend: true,
            success : suc,
            error : err
        });
    },

    trackTask : function(taskID, suc, err, progress) {
        var onTaskResponse = function(result) {
            var taskStatus = result['status'];
            switch(taskStatus) {
            case 'running':
                progress && progress(result);
                setTimeout(function() {
                    gingerbase.trackTask(taskID, suc, err, progress);
                }, 2000);
                break;
            case 'finished':
                suc && suc(result);
                break;
            case 'failed':
                err && err(result);
                break;
            default:
                break;
            }
        };

        gingerbase.getTask(taskID, onTaskResponse, err);
        if(gingerbase.trackingTasks.indexOf(taskID) < 0)
            gingerbase.trackingTasks.push(taskID);
    },

    createReport: function(settings, suc, err, progress) {
        var onResponse = function(data) {
            taskID = data['id'];
            gingerbase.trackTask(taskID, suc, err, progress);
        };

        wok.requestJSON({
            url : 'plugins/gingerbase/debugreports',
            type : "POST",
            contentType : "application/json",
            data : JSON.stringify(settings),
            dataType : "json",
            success : onResponse,
            error : err
        });
    },

    renameReport : function(name, settings, suc, err) {
        $.ajax({
            url : "plugins/gingerbase/debugreports/" + encodeURIComponent(name),
            type : 'PUT',
            contentType : 'application/json',
            data : JSON.stringify(settings),
            dataType : 'json',
            success: suc,
            error: err
        });
    },

    deleteReport: function(settings, suc, err) {
        var reportName = encodeURIComponent(settings['name']);
        wok.requestJSON({
            url : 'plugins/gingerbase/debugreports/' + reportName,
            type : 'DELETE',
            contentType : 'application/json',
            dataType : 'json',
            success : suc,
            error : err
        });
    },

    downloadReport: function(settings, suc, err) {
        window.open(settings['file'],'_blank');
    },

    shutdown: function(settings, suc, err) {
        var reboot = settings && settings['reboot'] === true;
        var url = 'plugins/gingerbase/host/' + (reboot ? 'reboot' : 'shutdown');
        wok.requestJSON({
            url : url,
            type : 'POST',
            contentType : 'application/json',
            dataType : 'json',
            success : suc,
            error : err
        });
    },

    listSoftwareUpdates : function(suc, err) {
        wok.requestJSON({
            url : 'plugins/gingerbase/host/packagesupdate',
            type : 'GET',
            contentType : 'application/json',
            dataType : 'json',
            resend: true,
            success : suc,
            error : err
        });
    },

    getPackageDeps : function(pkg, suc, err) {
        var pkg_name = encodeURIComponent(pkg);
        wok.requestJSON({
            url : "plugins/gingerbase/host/packagesupdate/" + pkg_name + "/deps",
            type : 'GET',
            contentType : 'application/json',
            dataType : 'json',
            async: false,
            success : suc,
            error : err
        });
    },

    softwareUpdateProgress : function(suc, err, progress) {
        var taskID = -1;
        var onResponse = function(data) {
            taskID = data['id'];
            trackTask();
        };

        var trackTask = function() {
            gingerbase.getTask(taskID, onTaskResponse, err);
        };

        var onTaskResponse = function(result) {
            var taskStatus = result['status'];
            switch(taskStatus) {
            case 'running':
                progress && progress(result);
                setTimeout(function() {
                    trackTask();
                }, 1000);
                break;
            case 'finished':
            case 'failed':
                suc(result);
                break;
            default:
                break;
            }
        };

        wok.requestJSON({
            url : 'plugins/gingerbase/host/swupdateprogress',
            type : "GET",
            contentType : "application/json",
            dataType : "json",
            success : onResponse,
            error : err
        });
    },

    updateSoftware : function(pack, suc, err, progress) {
        var taskID = -1;
        var onResponse = function(data) {
            taskID = data['id'];
            trackTask();
        };

        var trackTask = function() {
            gingerbase.getTask(taskID, onTaskResponse, err);
        };

        var onTaskResponse = function(result) {
            var taskStatus = result['status'];
            switch(taskStatus) {
            case 'running':
                progress();
                setTimeout(function() {
                    trackTask();
                }, 1000);
                break;
            case 'finished':
            case 'failed':
                suc(result);
                break;
            default:
                break;
            }
        };

        wok.requestJSON({
            url : 'plugins/gingerbase/host/packagesupdate/' + pack + '/upgrade',
            type : "POST",
            async: false,
            contentType : "application/json",
            dataType : "json",
            success : onResponse,
            error : err
        });
    },

    updateAllSoftware : function(suc, err, progress) {
        var taskID = -1;
        var onResponse = function(data) {
            taskID = data['id'];
            trackTask();
        };

        var trackTask = function() {
            gingerbase.getTask(taskID, onTaskResponse, err);
        };

        var onTaskResponse = function(result) {
            var taskStatus = result['status'];
            switch(taskStatus) {
            case 'running':
                progress && progress(result);
                setTimeout(function() {
                    trackTask();
                }, 700);
                break;
            case 'finished':
            case 'failed':
                suc(result);
                break;
            default:
                break;
            }
        };

        wok.requestJSON({
            url : 'plugins/gingerbase/host/swupdate',
            type : "POST",
            contentType : "application/json",
            dataType : "json",
            success : onResponse,
            error : err
        });
    },

    createRepository : function(settings, suc, err) {
        wok.requestJSON({
            url : "plugins/gingerbase/host/repositories",
            type : "POST",
            contentType : "application/json",
            data : JSON.stringify(settings),
            dataType : "json",
            success: suc,
            error: err
        });
    },

    retrieveRepository : function(repository, suc, err) {
        var reposID = encodeURIComponent(repository);
        wok.requestJSON({
            url : "plugins/gingerbase/host/repositories/" + reposID,
            type : 'GET',
            contentType : 'application/json',
            dataType : 'json',
            success : suc,
            error : err
        });
    },

    updateRepository : function(name, settings, suc, err) {
        var reposID = encodeURIComponent(name);
        $.ajax({
            url : "plugins/gingerbase/host/repositories/" + reposID,
            type : 'PUT',
            contentType : 'application/json',
            data : JSON.stringify(settings),
            dataType : 'json',
            success : suc,
            error : err
        });
    },

    enableRepository : function(name, enable, suc, err) {
        var reposID = encodeURIComponent(name);
        $.ajax({
            url : "plugins/gingerbase/host/repositories/" + reposID +
                  '/' + (enable === true ? 'enable' : 'disable'),
            type : 'POST',
            contentType : 'application/json',
            dataType : 'json',
            success : suc,
            error : err
        });
    },

    deleteRepository : function(repository, suc, err) {
        var reposID = encodeURIComponent(repository);
        wok.requestJSON({
            url : 'plugins/gingerbase/host/repositories/' + reposID,
            type : 'DELETE',
            contentType : 'application/json',
            dataType : 'json',
            success : suc,
            error : err
        });
    },

    listRepositories : function(suc, err) {
        wok.requestJSON({
            url : 'plugins/gingerbase/host/repositories',
            type : 'GET',
            contentType : 'application/json',
            dataType : 'json',
            resend: true,
            success : suc,
            error : err
        });
    },

    getCPUInfo : function(suc, err) {
        wok.requestJSON({
            url : 'plugins/gingerbase/host/cpuinfo',
            type : 'GET',
            contentType : 'application/json',
            dataType : 'json',
            resend : true,
            success : suc,
            error : err ? err : function(data) {
                wok.message.error(data.responseJSON.reason);
            }
        });
    },
    // get smt data
    getSMT: function(suc, err) {
        wok.requestJSON({
            url : 'plugins/gingerbase/host/smt',
            type : 'GET',
            resend: true,
            contentType : 'application/json',
            dataType : 'json',
            resend : true,
            success : suc,
            error: err
        });
    },
    enablesmt : function(value, suc, err) {
        wok.requestJSON({
            url : "plugins/gingerbase/host/smt/enable",
            type : "POST",
            contentType : "application/json",
            data : JSON.stringify(value),
            dataType : "json",
            success: suc,
            error: err
        });
    },
    disablesmt : function(value, suc, err) {
        wok.requestJSON({
            url : "plugins/gingerbase/host/smt/disable",
            type : "POST",
            contentType : "application/json",
            data : JSON.stringify(value),
            dataType : "json",
            success: suc,
            error: err
        });
    },
    capabilities: undefined
};
/*
 * Project Ginger Base
 *
 * Copyright IBM Corp, 2015-2016
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
gingerbase.host = {};
var arch;

gingerbase.host_dashboard = function() {
  gingerbase.getCapabilities(function(result) {
    gingerbase.capabilities = result;
    gingerbase.init_dashboard();
  }, function() {
    gingerbase.init_dashboard();
  })
};

gingerbase.init_dashboard = function() {
    "use strict";
    var reportGridID = 'available-reports-grid';
    var reportGrid = null;
    var enableReportButtons = function(toEnable) {
        // available-reports-grid-action-group
        if(toEnable === 'all'){
            $.each($('#'+reportGridID+'-action-group ul.dropdown-menu .btn'), function(i,button){;
                $(this).attr('disabled', false);
            });
        }else if(toEnable === 'some'){
            $.each($('#'+reportGridID+'-action-group ul.dropdown-menu .btn'), function(i,button){
                if($(this).attr('id') === 'available-reports-grid-rename-button'){
                    $(this).attr('disabled', true);
                }else {
                    $(this).attr('disabled', false);
                }
            });
        }else {
            $.each($('#'+reportGridID+'-action-group ul.dropdown-menu .btn'), function(i,button){
                if($(this).attr('id') === 'available-reports-grid-generate-button'){
                    $(this).attr('disabled', false);
                }else {
                    $(this).attr('disabled', true);
                }
            });
        }
    };
    var initReportGrid = function(reports) {
        reportGrid = new wok.widget.List({
            container: 'debug-report-section',
            id: reportGridID,
            title: i18n['GGBDR6002M'],
            toolbarButtons: [{
                id: reportGridID + '-generate-button',
                class: 'fa fa-plus-circle',
                label: i18n['GGBDR6006M'],
                disabled: false,
                onClick: function(event) {
                    wok.window.open('plugins/gingerbase/report-add.html');
                }
            }, {
                id: reportGridID + '-rename-button',
                class: 'fa fa-pencil',
                label: i18n['GGBDR6008M'],
                disabled: true,
                onClick: function(event) {
                    var report = reportGrid.getSelected();
                    if (!report || report.length > 1){
                        return;
                    }
                    gingerbase.selectedReport = report[0]['name'];
                    wok.window.open('plugins/gingerbase/report-rename.html');
                }
            }, {
                id: reportGridID + '-download-button',
                label: i18n['GGBDR6010M'],
                class: 'fa fa-download',
                disabled: true,
                onClick: function(event) {
                    var report = reportGrid.getSelected();
                    if (!report) {
                        return;
                    }

                    for(var i = 0; i < report.length; i++){
                        gingerbase.downloadReport({
                            file: report[i]['uri']
                        });
                    }
                }
            }, {
                id: reportGridID + '-remove-button',
                class: 'fa fa-minus-circle',
                label: i18n['GGBDR6009M'],
                critical: true,
                disabled: true,
                onClick: function(event) {
                    event.preventDefault();
                    if($(this).find('.btn').attr('disabled') !== 'disabled'){
                        var report = reportGrid.getSelected();
                        if (!report) {
                            return;
                        }

                        if(report.length > 1) {

                            var settings = {
                                    title: i18n['GGBDR6016M'],
                                    content: i18n['GGBDR6014M'],
                                    confirm: i18n['GGBAPI6002M'],
                                    cancel: i18n['GGBAPI6003M']
                                };

                        }else {

                            var settings = {
                                title: i18n['GGBDR6015M'],
                                content: i18n['GGBDR6001M'].replace("%1", '<strong>'+report[0]['name']+'</strong>'),
                                confirm: i18n['GGBAPI6002M'],
                                cancel: i18n['GGBAPI6003M']
                            };

                        }

                        wok.confirm(settings, function() {
                            for(var i = 0; i < report.length; i++){
                              gingerbase.deleteReport({
                                name: report[i]['name']
                              }, function(result) {
                                listDebugReports();

                              }, function(error) {
                                wok.message.error(error.responseJSON.reason);
                              });
                            }
                        });
                    }else {
                        return false;
                    }
                }
            }],
            onRowSelected: function(row) {
                var report = reportGrid.getSelected();
                if (report.length <= 0) {
                    enableReportButtons(false);
                }else if (report.length === 1) {
                    enableReportButtons('all');
                } else {
                    enableReportButtons('some');
                }
            },
            frozenFields: [],
            fields: [{
                name: 'name',
                label: i18n['GGBDR6003M'],
                cssClass: 'debug-report-name',
                type: 'name'
            }, {
                name: 'time',
                label: i18n['GGBDR6005M'],
                cssClass: 'debug-report-time',
                type: 'description',
                converter: 'datetime-locale-converter'
            }],
            converters: wok.localeConverters,
            data: reports
        });
    };

    var getPendingReports = function() {
        var reports = [];
        var filter = 'status=running&target_uri=' + encodeURIComponent('^/plugins/gingerbase/debugreports/*');

        gingerbase.getTasksByFilter(filter, function(tasks) {
            for (var i = 0; i < tasks.length; i++) {
                var reportName = tasks[i].target_uri.replace(/^\/plugins\/gingerbase\/debugreports\//, '') || i18n['GGBDR6012M'];
                reports.push({
                    'name': reportName,
                    'time': i18n['GGBDR6007M']
                });

                if (gingerbase.trackingTasks.indexOf(tasks[i].id) >= 0) {
                    continue;
                }

                gingerbase.trackTask(tasks[i].id, function(result) {
                    wok.topic('gingerbase/debugReportAdded').publish();
                }, function(result) {
                    // Error message from Async Task status
                    if (result['message']) {
                        var errText = result['message'];
                    }
                    // Error message from standard gingerbase exception
                    else {
                        var errText = result['responseJSON']['reason'];
                    }
                    result && wok.message.error(errText);
                    wok.topic('gingerbase/debugReportAdded').publish();
                }, null);
            }
        }, null, true);

        return reports;
    };

    var listDebugReports = function() {
        gingerbase.listReports(function(reports) {
            var pendingReports = getPendingReports();
            var allReports = pendingReports.concat(reports);
            $('#debug-report-section').removeClass('hidden');

            // Row selection will be cleared so disable buttons here
            enableReportButtons(false);

            if (reportGrid) {
                reportGrid.setData(allReports);
            } else {
                initReportGrid(allReports);
            }

            // Set id-debug-img to pending reports
            // It will display a loading icon
            var gridElement = $('#' + reportGridID);
            var gridButtonContainer = $('#' + reportGridID+'-action-group');
            // Clean-up selected Item count on mobile
            $('.mobile-action-count', gridButtonContainer).remove();
            //  "Generating..."
            $.each($('li label span.debug-report-time', gridElement), function(index, row) {
                if($(row).text() ===  i18n['GGBDR6007M']){
                    $(row).parent().parent().addClass('generating');
                    $(row).parent().parent().find('input[type="checkbox"]').prop('disabled', true);
                }else {
                    $(row).parent().parent().find('input[type="checkbox"]').prop('disabled', false);
                }
            });
        }, function(error) {
            if (error['status'] === 403) {
                $('#debug-report-section').addClass('hidden');
                return;
            }
            $('#debug-report-section').removeClass('hidden');
        });
    };

    var shutdownButtonID = '#host-button-shutdown';
    var restartButtonID = '#host-button-restart';
    var shutdownHost = function(params) {
        var settings = {
            content: i18n['GGBHOST6008M'],
            confirm: i18n['GGBAPI6002M'],
            cancel: i18n['GGBAPI6003M']
        };

        wok.confirm(settings, function() {
            $(shutdownButtonID).prop('disabled', true);
            $(restartButtonID).prop('disabled', true);
            wok.message.warn(i18n['GGBHOST6003E'],null,true);
            // Check if there is any VM is running.
            // Based on the success will shutdown/reboot
            gingerbase.shutdown(params, function(success) {
                wok.message.success(i18n['GGBHOST6009M'])
                $(shutdownButtonID).prop('disabled', false);
                $(restartButtonID).prop('disabled', false);
                return;
            }, function(error) {
                var status = error.status;
                if(status===502) {
                    // Gateway server is not able to get a valid
                    // response from upstream server.
                    wok.message.error(i18n['GGBHOST6002E']);
                    setTimeout(function() {
                        location.reload(true);
                    },1000);
                } else {
                    // Looks like VMs are running.
                    if(status !== 0){
                        wok.message.error(i18n['GGBHOST6001E']);
                    }
                }
                $(shutdownButtonID).prop('disabled', false);
                $(restartButtonID).prop('disabled', false);
            });
        }, function() {
        });
    };

    var initPage = function() {
        var toolsHtml = '<li><a id="host-button-restart" href="javascript:void(0);">'
        toolsHtml += '<i class="fa fa-undo"></i><span>' + i18n['GGBHOST6022M'] + '</span></a></li>'
        toolsHtml += '<li><a id="host-button-shutdown" href="javascript:void(0);">'
        toolsHtml += '<i class="fa fa-ban"></i><span>' + i18n['GGBHOST6023M'] + '</span></a></li>'

        if(wok.tabMode["dashboard"] === "admin") {
            $('#toolbar ul.tools').html(toolsHtml);
        }

        $('#host-button-shutdown').on('click', function(event) {
            event.preventDefault();
            shutdownHost(null);
        });

        $('#host-button-restart').on('click', function(event) {
            event.preventDefault();
            shutdownHost({
                reboot: true
            });
        });

        var setupUI = function() {
            if (gingerbase.capabilities === undefined) {
                setTimeout(setupUI, 2000);
                return;
            }

            if (gingerbase.capabilities['system_report_tool']) {
                listDebugReports();
                wok.topic('gingerbase/debugReportAdded')
                    .subscribe(listDebugReports);
                wok.topic('gingerbase/debugReportRenamed')
                    .subscribe(listDebugReports);
            }
        };
        setupUI();
    };

    gingerbase.getHost(function(data) {
        var htmlTmpl = $('#host-dashboard-tmpl').html();
        var memory = null
        var cpus = null
        var cpu_threads = null
        data['logo'] = data['logo'] || '';
        // fetch online and offline memory details
        data['memory']['online'] = wok.formatMeasurement(data['memory']['online'], {
            fixed: 2, converter: wok.localeConverters["number-locale-converter"]
        });
        data['memory']['offline'] = wok.formatMeasurement(data['memory']['offline'], {
            fixed: 2, converter: wok.localeConverters["number-locale-converter"]
        });
        memory =  i18n["GGBHOST6010M"] + data['memory']['online'] + "\xa0\xa0\xa0\xa0" +
                  i18n["GGBHOST6011M"] + data['memory']['offline'];
        // fetch online and offline cpu details
        cpus = i18n["GGBHOST6010M"] + data['cpus']['online'] + "\xa0\xa0\xa0\xa0" +
               i18n["GGBHOST6011M"] + data['cpus']['offline'];
        // fetch socket(s), core(s) per socket and thread(s) per core details
        cpu_threads = i18n["GGBHOST6015M"] + data['cpu_threads']['sockets'] + "\xa0\xa0\xa0\xa0" +
                      i18n["GGBHOST6016M"] + data['cpu_threads']['cores_per_socket'] + "\xa0\xa0\xa0\xa0" +
                      i18n["GGBHOST6017M"] + data['cpu_threads']['threads_per_core'];
        // This code is only for s390x architecture where hypervisor details required.
        arch = data['architecture'];
        if (data['architecture'] == 's390x'){
            // cores_info is total shared and dedicated cpu cores for s390x
            data['cores_info'] = i18n["GGBHOST6012M"] + data['cpus']['shared'] + "\xa0\xa0\xa0\xa0" +
                                 i18n["GGBHOST6013M"] + data['cpus']['dedicated'];
            //prepend book(s) details to cpu_threads
            cpu_threads = i18n["GGBHOST6014M"] + data['cpu_threads']['books'] + "\xa0\xa0\xa0\xa0" + cpu_threads
            data['lpar_details'] = i18n["GGBHOST6019M"] + data['virtualization']['lpar_name'] + '\xa0\xa0\xa0\xa0' + i18n["GGBHOST6020M"] + data['virtualization']['lpar_number'];
            data['hypervisor_details'] = i18n["GGBHOST6019M"] + data['virtualization']['hypervisor'] + '\xa0\xa0\xa0\xa0' + i18n["GGBHOST6021M"] + data['virtualization']['hypervisor_vendor'];
        }
        data['memory'] = memory
        data['cpus'] = cpus
        data['cpu_threads'] = cpu_threads
        var templated = wok.substitute(htmlTmpl, data);
        $('#host-content-container').html(templated);

        initPage();
        initTracker();
        // Enable cores details, hypervisor details and LPAR details on s390x architechture
        if (data['architecture'] == 's390x'){
            $('#s390x-cores-info').removeClass('hidden');
            $('#s390x-info').removeClass('hidden');
        }
    });

    var StatsMgr = function() {
      if(gingerbase.capabilities['smt']){
        $('#smt_available').show();
      var smt_status;
        gingerbase.getSMT(function suc(result) {
          smt_status = result['current_smt_settings'];
          $("#smt_cstatus").text(smt_status['status']);
        }, function(error) {
        wok.message.error(error.responseJSON.reason);
        });
      }
        var statsArray = {
            cpu: {
                u: {
                    type: 'percent',
                    legend: i18n['GGBHOST6002M'],
                    points: [],
                    converter: 'number-locale-converter'
                }
            },
            memory: {
                u: {
                    type: 'value',
                    base: 2,
                    fixed: 2,
                    locale: wok.lang.get_locale(),
                    legend: i18n['GGBHOST6003M'],
                    points: [],
                    converter: 'number-locale-converter'
                }
            },
            diskIO: {
                w: {
                    type: 'value',
                    base: 2,
                    fixed: 2,
                    locale: wok.lang.get_locale(),
                    unit: i18n['GGBHOST6018M'],
                    legend: i18n['GGBHOST6005M'],
                    'class': 'disk-write',
                    points: [],
                    converter: 'number-locale-converter'
                },
                r: {
                    type: 'value',
                    base: 2,
                    fixed: 2,
                    locale: wok.lang.get_locale(),
                    unit: i18n['GGBHOST6018M'],
                    legend: i18n['GGBHOST6004M'],
                    points: [],
                    converter: 'number-locale-converter'
                }
            },
            networkIO: {
                s: {
                    type: 'value',
                    base: 2,
                    fixed: 2,
                    locale: wok.lang.get_locale(),
                    unit: i18n['GGBHOST6018M'],
                    legend: i18n['GGBHOST6007M'],
                    'class': 'network-sent',
                    points: [],
                    converter: 'number-locale-converter'
                },
                r: {
                    type: 'value',
                    base: 2,
                    fixed: 2,
                    locale: wok.lang.get_locale(),
                    unit: i18n['GGBHOST6018M'],
                    legend: i18n['GGBHOST6006M'],
                    points: [],
                    converter: 'number-locale-converter'
                }
            }
        };

        var SIZE = 20;
        var cursor = SIZE;

        var add = function(stats) {
            for (var key in stats) {
                var item = stats[key];
                for (var metrics in item) {
                    var value = item[metrics]['v'];
                    var max = item[metrics]['max'];
                    var unifiedMetrics = statsArray[key][metrics];
                    var ps = unifiedMetrics['points'];
                    if (!Array.isArray(value)) {
                        ps.push(value);
                        if (ps.length > SIZE + 1) {
                            ps.shift();
                        }
                    } else {
                        ps = ps.concat(value);
                        ps.splice(0, ps.length - SIZE - 1);
                        unifiedMetrics['points'] = ps;
                    }
                    if (max !== undefined) {
                        unifiedMetrics['max'] = max;
                    } else {
                        if (unifiedMetrics['type'] !== 'value') {
                            continue;
                        }
                        max = -Infinity;
                        $.each(ps, function(i, value) {
                            if (value > max) {
                                max = value;
                            }
                        });
                        if (max === 0) {
                            ++max;
                        }
                        max *= 1.1;
                        unifiedMetrics['max'] = max;
                    }
                }
            }
            cursor++;
        };

        var get = function(which) {
            var stats = statsArray[which];
            var lines = [];
            for (var k in stats) {
                var obj = stats[k];
                var line = {
                    type: obj['type'],
                    base: obj['base'],
                    unit: obj['unit'],
                    fixed: obj['fixed'],
                    legend: obj['legend']
                };
                if (obj['max']) {
                    line['max'] = obj['max'];
                }
                if (obj['class']) {
                    line['class'] = obj['class'];
                }
                if (obj['converter']) {
                    line['converter'] = obj['converter'];
                }
                var ps = obj['points'];
                var numStats = ps.length;
                var unifiedPoints = [];
                $.each(ps, function(i, value) {
                    unifiedPoints.push({
                        x: cursor - numStats + i,
                        y: value
                    });
                });
                line['points'] = unifiedPoints;
                lines.push(line);
            }
            return lines;
        };

        return {
            add: add,
            get: get
        };
    };

    var Tracker = function(charts) {
        var charts = charts;
        var timer = null;
        var statsPool = new StatsMgr();
        var setCharts = function(newCharts) {
            charts = newCharts;
            for (var key in charts) {
                var chart = charts[key];
                chart.updateUI(statsPool.get(key));
            }
        };

        var self = this;

        var UnifyStats = function(stats) {
            var result = {
                cpu: {
                    u: {
                        v: stats['cpu_utilization']
                    }
                },
                memory: {
                    u: {}
                },
                diskIO: {
                    w: {
                        v: stats['disk_write_rate']
                    },
                    r: {
                        v: stats['disk_read_rate']
                    }
                },
                networkIO: {
                    s: {
                        v: stats['net_sent_rate']
                    },
                    r: {
                        v: stats['net_recv_rate']
                    }
                }
            };

            if (Array.isArray(stats['memory'])) {
                result.memory.u['v'] = [];
                result.memory.u['max'] = -Infinity;
                for (var i = 0; i < stats['memory'].length; i++) {
                    result.memory.u['v'].push(stats['memory'][i]['avail']);
                    result.memory.u['max'] = Math.max(result.memory.u['max'], stats['memory'][i]['total']);
                }
            } else {
                result.memory.u['v'] = stats['memory']['avail'],
                    result.memory.u['max'] = stats['memory']['total']
            }
            return (result);
        };


        var statsCallback = function(stats) {
            var unifiedStats = UnifyStats(stats);
            statsPool.add(unifiedStats);
            for (var key in charts) {
                var chart = charts[key];
                chart.updateUI(statsPool.get(key));
            }
            timer = setTimeout(function() {
                continueTrack();
            }, 1000);
        };

        var track = function() {
            gingerbase.getHostStatsHistory(statsCallback,
                function() {
                    continueTrack();
                });
        };

        var continueTrack = function() {
            gingerbase.getHostStats(statsCallback,
                function() {
                    continueTrack();
                });
        };

        var destroy = function() {
            timer && clearTimeout(timer);
            timer = null;
        };

        return {
            setCharts: setCharts,
            start: track,
            stop: destroy
        };
    };

    var initTracker = function() {
        // TODO: Extend tabs with onUnload event to unregister timers.
        if (gingerbase.hostTimer) {
            gingerbase.hostTimer.stop();
            delete gingerbase.hostTimer;
        }

        var trackedCharts = {
            cpu: new wok.widget.LineChart({
                id: 'chart-cpu',
                node: 'container-chart-cpu',
                type: 'percent',
                converters: wok.localeConverters
            }),
            memory: new wok.widget.LineChart({
                id: 'chart-memory',
                node: 'container-chart-memory',
                type: 'value',
                converters: wok.localeConverters
            }),
            diskIO: new wok.widget.LineChart({
                id: 'chart-disk-io',
                node: 'container-chart-disk-io',
                type: 'value',
                converters: wok.localeConverters
            }),
            networkIO: new wok.widget.LineChart({
                id: 'chart-network-io',
                node: 'container-chart-network-io',
                type: 'value',
                converters: wok.localeConverters
            })
        };

        if (gingerbase.hostTimer) {
            gingerbase.hostTimer.setCharts(trackedCharts);
        } else {
            gingerbase.hostTimer = new Tracker(trackedCharts);
            gingerbase.hostTimer.start();
        }
    };

    $('#host-root-container').on('remove', function() {
        if (gingerbase.hostTimer) {
            gingerbase.hostTimer.stop();
            delete gingerbase.hostTimer;
        }

        reportGrid && reportGrid.destroy();
        wok.topic('gingerbase/debugReportAdded').unsubscribe(listDebugReports);
        wok.topic('gingerbase/debugReportRenamed').unsubscribe(listDebugReports);
    });

};
gingerbase.getsmtstatus = function() {
    $('#smt-submit').off();
    $('.selectpicker').selectpicker();
    $("input.make-switch").bootstrapSwitch();
    $('#smt-value').show();
    $('#no-smt-value').hide();
    $('#smt-submit').prop("disabled", true);
    gingerbase.getSMT(function suc(result) {
        var res = result['current_smt_settings'];
        $("#smtstatus-textbox").text(res['status']);
        $("#smtvalue-textbox").text(res['smt']);
        var persist_re = result['persisted_smt_settings'];
        if (persist_re['status'] == "enabled") {
            $('#no-smt-value').hide();
            $('#smt-value').show();
            $('#smt-status-change').bootstrapSwitch('state', true);
            $('#smtTypeenabled').selectpicker("val", persist_re['smt']);
        } else {
            $('#smt-value').hide();
            $('#no-smt-value').show();
            $('#smt-status-change').bootstrapSwitch('state', false);
            $('#smtTypedisabled').val(persist_re['smt']);
        }
    }, function(error) {
        wok.message.error(error.responseJSON.reason);
    });
    $('.selectpicker').on('change', function() {
        $('#smt-submit').prop("disabled", false);
    });
    $('#smt-status-change').on('switchChange.bootstrapSwitch', function(event, state) {
        $('#smt-submit').prop("disabled", false);
        if (state) {
            $('#no-smt-value').hide();
            $('#smt-value').show();
        } else {
            $('#smt-value').hide();
            $('#no-smt-value').show();
        }
    });
    $('#smt-submit').on('click', function(event) {
        var smtval = {};
        if ($('#smt-status-change').bootstrapSwitch('state')) {
            smtval['smt_val'] = $('#smtTypeenabled').val();
            var settings = {
                content: i18n["GGBHSMT0001M"],
                confirm: i18n["GINNET0015M"]
            };
            wok.confirm(settings, function() {
                gingerbase.enablesmt(smtval, function(result) {
                    var settings = {
                        content: i18n["GGBHSMT0002M"],
                        confirm: i18n["GGBHSMT0003M"],
                        cancel: i18n["GGBHSMT0004M"]
                    };
                    wok.confirm(settings, function() {
                        var params = {};
                        params['reboot'] = true;
                        gingerbase.shutdown(params, function(success) {
                            wok.message.success(i18n['GGBHOST6009M'])
                        }, function(error) {
                            var status = error.status;
                            if (status === 502) {
                                wok.message.error(i18n['GGBHOST6002E']);
                                setTimeout(function() {
                                    location.reload(true);
                                }, 1000);
                            } else {
                                if (status !== 0) {
                                    wok.message.error(i18n['GGBHOST6001E']);
                                }
                            }
                        });
                    }, function() {
                        $('#smtinfo').modal('hide');
                    });
                    $('#smtinfo').modal('hide');
                    wok.message.success(i18n['GGBHSMT0005M'], '#smt-load-message');
                }, function(error) {
                    wok.message.error(error.responseJSON.reason, '#smt-message');
                });
            });
        } else {
            smtval = {};
            var settings = {
                content: i18n["GGBHSMT0006M"],
                confirm: i18n["GINNET0015M"]
            };
            wok.confirm(settings, function() {
                gingerbase.disablesmt(smtval, function(result) {
                    var settings = {
                        content: i18n["GGBHSMT0002M"],
                        confirm: i18n["GGBHSMT0003M"],
                        cancel: i18n["GGBHSMT0004M"]
                    };
                    wok.confirm(settings, function() {
                        var params = {};
                        params['reboot'] = true;
                        gingerbase.shutdown(params, function(success) {
                            wok.message.success(i18n['GGBHOST6009M'])
                        }, function(error) {
                            var status = error.status;
                            if (status === 502) {
                                wok.message.error(i18n['GGBHOST6002E']);
                                setTimeout(function() {
                                    location.reload(true);
                                }, 1000);
                            } else {
                                if (status !== 0) {
                                    wok.message.error(i18n['GGBHOST6001E']);
                                }
                            }
                        });
                    }, function() {
                        $('#smtinfo').modal('hide');
                    });
                    $('#smtinfo').modal('hide');
                    wok.message.success(i18n['GGBHSMT0007M'], '#smt-load-message');
                }, function(error) {
                    wok.message.error(error.responseJSON.reason, '#smt-message');
                });
            });
        }
    });
}
/*
 * Project Ginger Base
 *
 * Copyright IBM Corp, 2015-2017
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
gingerbase.host = {};
gingerbase.arrayOfPackagesToKeepIcon = [];
gingerbase.host_update = function() {

  gingerbase.getCapabilities(function(result) {
    gingerbase.capabilities = result;
    gingerbase.init_update();
  }, function() {
    gingerbase.init_update();
  })
};

gingerbase.init_update = function() {
    "use strict";
    var repositoriesGrid = null;
    var enableRepositoryButtons = function(toEnable) {
        // available-reports-grid-action-group
        if(toEnable === 'all'){
            $.each($('#'+repositoriesGrid.selectButtonContainer[0].id+' ul.dropdown-menu .btn'), function(i,button){;
                $(this).attr('disabled', false);
            });
        }else if(toEnable === 'some'){
            $.each($('#'+repositoriesGrid.selectButtonContainer[0].id+' ul.dropdown-menu .btn'), function(i,button){
                if($(this).attr('id') === 'repositories-grid-edit-button' || $(this).attr('id') === 'repositories-grid-enable-button'){
                    $(this).attr('disabled', true);
                }else {
                    $(this).attr('disabled', false);
                }
            });
        }else if(toEnable === 'enable'){
            $.each($('#'+repositoriesGrid.selectButtonContainer[0].id+' ul.dropdown-menu .btn'), function(i,button){
                if($(this).attr('id') === 'repositories-grid-edit-button'){
                    $(this).attr('disabled', true);
                }else {
                    $(this).attr('disabled', false);
                }
            });
        }else {
            $.each($('#'+repositoriesGrid.selectButtonContainer[0].id+' ul.dropdown-menu .btn'), function(i,button){
                if($(this).attr('id') === 'repositories-grid-add-button'){
                    $(this).attr('disabled', false);
                }else {
                    $(this).attr('disabled', true);
                }
            });
        }
    };
    var initRepositoriesGrid = function(repo_type) {
        var gridFields = [];
        if (repo_type === "yum") {
            gridFields = [{
                name: 'repo_id',
                label: i18n['GGBREPO6004M'],
                cssClass: 'repository-id',
                type: 'name'
            }, {
                name: 'config[display_repo_name]',
                label: i18n['GGBREPO6005M'],
                cssClass: 'repository-name',
                type: 'description'
            }, {
                name: 'enabled',
                label: i18n['GGBREPO6009M'],
                cssClass: 'repository-enabled',
                type: 'status'
            }];
        } else if (repo_type === "deb") {
            gridFields = [{
                name: 'baseurl',
                label: i18n['GGBREPO6006M'],
                makeTitle: true,
                cssClass: 'repository-baseurl deb',
                type: 'description'
            }, {
                name: 'enabled',
                label: i18n['GGBREPO6009M'],
                cssClass: 'repository-enabled deb',
                type: 'status'
            }, {
                name: 'config[dist]',
                label: i18n['GGBREPO6018M'],
                cssClass: 'repository-gpgcheck deb'
            }, {
                name: 'config[comps]',
                label: i18n['GGBREPO6019M'],
                cssClass: 'repository-gpgcheck deb'
            }];
        } else {
            gridFields = [{
                name: 'repo_id',
                label: i18n['GGBREPO6004M'],
                cssClass: 'repository-id',
                type: 'name'
            }, {
                name: 'enabled',
                label: i18n['GGBREPO6009M'],
                cssClass: 'repository-enabled',
                type: 'status'
            }, {
                name: 'baseurl',
                label: i18n['GGBREPO6006M'],
                makeTitle: true,
                cssClass: 'repository-baseurl',
                type: 'description'
            }];
        }
        repositoriesGrid = new wok.widget.List({
            container: 'repositories-section',
            id: 'repositories-grid',
            title: i18n['GGBREPO6003M'],
            toolbarButtons: [{
                id: 'repositories-grid-add-button',
                label: i18n['GGBREPO6012M'],
                class: 'fa fa-plus-circle',
                onClick: function(event) {
                    event.preventDefault();
                    wok.window.open({
                        url: 'plugins/gingerbase/repository-add.html',
                        class: repo_type
                    });
                }
            }, {
                id: 'repositories-grid-enable-button',
                label: i18n['GGBREPO6016M'],
                class: 'fa fa-play-circle-o',
                disabled: true,
                onClick: function(event) {
                    event.preventDefault();
                    if(!$("#repositories-grid-enable-button").prop("disabled")){
                        var repository = repositoriesGrid.getSelected();
                        if (!repository) {
                            return;
                        }

                        $.each(repository, function(index, repo){
                            var name = repo['repo_id'];
                            var enable = !repo['enabled'];
                            enableRepositoryButtons(false);
                            gingerbase.enableRepository(name, enable, function() {
                                wok.topic('gingerbase/repositoryUpdated').publish();
                            });
                        });
                    } else {
                        return false;
                    }
                }
            }, {
                id: 'repositories-grid-edit-button',
                label: i18n['GGBREPO6013M'],
                class: 'fa fa-pencil',
                disabled: true,
                onClick: function(event) {
                    event.preventDefault();
                    if(!$("#repositories-grid-edit-button").prop("disabled")){
                        var repository = repositoriesGrid.getSelected();
                        if (!repository) {
                            return;
                        }
                        gingerbase.selectedRepository = repository[0]['repo_id'];
                        wok.window.open({
                            url: 'plugins/gingerbase/repository-edit.html',
                            class: repo_type
                        });
                    } else {
                        return false;
                    }
                }
            }, {
                id: 'repositories-grid-remove-button',
                label: i18n['GGBREPO6014M'],
                class: 'fa fa-minus-circle',
                critical: true,
                disabled: true,
                onClick: function(event) {
                    event.preventDefault();
                    if(!$("#repositories-grid-remove-button").prop('disabled')){
                        var repository = repositoriesGrid.getSelected();
                        if (!repository) {
                            return;
                        }

                        if(repository.length > 1) {

                          var settings = {
                                    title: i18n['GGBREPO6020M'],
                                    content: i18n['GGBREPO6021M'],
                                    confirm: i18n['GGBAPI6002M'],
                                    cancel: i18n['GGBAPI6003M']
                          };

                        }else {

                            var settings = {
                                title: i18n['GGBREPO6001M'],
                                content: i18n['GGBREPO6002M'].replace("%1", '<strong>'+repository[0]['repo_id']+'</strong>'),
                                confirm: i18n['GGBAPI6002M'],
                                cancel: i18n['GGBAPI6003M']
                            };

                        }

                        wok.confirm(settings, function() {
                            for(var i = 0; i < repository.length; i++){
                              gingerbase.deleteRepository(
                                repository[i]['repo_id'],
                                function(result) {
                                    listRepositories();
                                    wok.topic('gingerbase/repositoryDeleted').publish(result);
                                }, function(error) {
                                    wok.message.error(error.responseJSON.reason);
                                });
                                }
                                enableRepositoryButtons(false);
                        });
                    }else {
                        return false;
                    }
                }
            }],
            onRowSelected: function(row) {
                var repository = repositoriesGrid.getSelected();
                var actionHtml,actionText,actionIcon ='';
                if (!repository) {
                    return;
                }
                if (repository.length <= 0) {
                    enableRepositoryButtons(false);
                    actionText= i18n['GGBREPO6016M'];
                    actionIcon = 'fa-play-circle-o';
                    actionHtml = ['<i class="fa',' ',actionIcon,'"></i>',' ',actionText].join('');
                    $('#repositories-grid-enable-button').html(actionHtml);
                }else if (repository.length === 1) {
                    enableRepositoryButtons('all');
                    var enabled = repository[0]['enabled'];
                    if(enabled){
                        actionText= i18n['GGBREPO6017M'];
                        actionIcon = 'fa-pause';
                    }else{
                        actionText= i18n['GGBREPO6016M'];
                        actionIcon = 'fa-play-circle-o';
                    }
                    actionHtml = ['<i class="fa',' ',actionIcon,'"></i>',' ',actionText].join('');
                    $('#repositories-grid-enable-button').html(actionHtml);
                } else {
                    var repoState = repository[0]['enabled'];
                    var diff = false;
                    $.each(repository, function(index, repo){
                        if (repo['enabled'] != repoState) {
                            diff = true;
                            return false;
                        }
                    });
                    if (diff) {
                        enableRepositoryButtons('some');
                    }
                    else {
                        enableRepositoryButtons('enable');
                    }
                }
            },
            frozenFields: [],
            fields: gridFields
        });

        var header = "#list";
        var list = "#repositories-grid";

        // create and add the filter form to the header
        var input = $("<input>").attr({"class":"filter form-control search",
            "type":"text","placeholder": i18n["GGBREPO6022M"],
            "id":"searchButton"});
        $(input).prependTo(header);

        // deal with input
        $(input).change( function () {
            var filter = $(this).val();

            // something to search: do it
            if (filter) {
                $(list).children().find("li:not(:contains(" + filter + "))").slideUp();
                $(list).children().find("li:contains(" + filter + ")").slideDown();

            // nothing: clean
            } else {
                $(list).children().find("li").slideDown();
            }
        }).keyup( function () {
            $(this).change();
        });
    };

    var listRepositories = function(gridCallback) {
        $("#searchButton").val("");
        repositoriesGrid.maskNode.removeClass('hidden');
        gingerbase.listRepositories(function(repositories) {
                if ($.isFunction(gridCallback)) {
                    gridCallback(repositories);
                } else {
                    if (repositoriesGrid) {
                        repositoriesGrid.setData(repositories);
                    } else {
                        initRepositoriesGrid();
                        repositoriesGrid.setData(repositories);
                    }
                }
                repositoriesGrid.maskNode.addClass('hidden');
            },
            function(error) {
                repositoriesGrid.maskNode.addClass('hidden');
                var message = error && error['responseJSON'] && error['responseJSON']['reason'];

                if ($.isFunction(gridCallback)) {
                    gridCallback([]);
                }
                repositoriesGrid &&
                    repositoriesGrid.showMessage(message || i18n['GGBUPD6008M']);
            });

        $('#repositories-grid-remove-button').prop('disabled', true);
        $('#repositories-grid-edit-button').prop('disabled', true);
        $('#repositories-grid-enable-button').prop('disabled', true);
    };

    var softwareUpdatesGridID = 'software-updates-grid';
    var softwareUpdatesGrid = null;
    var progressAreaID = 'software-updates-progress-textarea';
    var textMessage = "";
    var reloadProgressArea = function(result) {
        $("#updates-accordion").show(500);
        var progressArea = $('#' + progressAreaID)[0];
        textMessage += result['message'];
        if (result['status'] == 'finished') {
            textMessage += i18n['GGBUPD6015M'];
            gingerbase.init_update_packages();
        }
        $(progressArea).text(textMessage);
        var scrollTop = $(progressArea).prop('scrollHeight');
        $(progressArea).prop('scrollTop', scrollTop);
    };

    $("#update-all-packages").click(function() {
        $("#update-all-packages").prop('disabled', true);
        $("#update-accordion").show(500);
        $("#software-updates-progress-textarea").text("Processing...");
        gingerbase.updateAllSoftware(function(result) {
            $("#update-all-packages").prop('disabled', true);
            reloadProgressArea(result);
            wok.topic('gingerbase/softwareUpdated').publish({
                result: result
            });
            $("#update-all-packages").prop('disabled', false);
        }, function(error) {
            var message = error && error['responseJSON'] && error['responseJSON']['reason'];
            wok.message.error(message || i18n['GGBUPD6009M']);
            $("#update-all-packages").prop('disabled', false);
        }, reloadProgressArea);
    });

    var startSoftwareUpdateProgress = function() {
        var progressArea = $('#' + progressAreaID)[0];
        $('#software-updates-progress-container').removeClass('hidden');
        $(progressArea).text('');
        var filter = 'status=running&target_uri=' + encodeURIComponent('^/plugins/gingerbase/host/swupdate/*');
            gingerbase.getTasksByFilter(filter, function(tasks) {
                var result = {};
                if (tasks.length > 0) {
                    gingerbase.getTask(tasks[0].id, function(task){
                        result = task;
                    }, function(error){});
                }
                if (result['status'] == 'running') {
                    reloadProgressArea(result);
                    $(".wok-mask").fadeOut(300, function() {});
                } else {
                    gingerbase.init_update_packages();
                }
            }, function(error) {
                wok.message.error(i18n['GGBUPD6011M']);
            }, reloadProgressArea);
    };

    var initPage = function() {

        var setupUI = function() {
            if (gingerbase.capabilities === undefined) {
                setTimeout(setupUI, 2000);
                return;
            }

            if (gingerbase.capabilities['update_tool']) {
                $('#software-update-section').removeClass('hidden');
                $('#software-update-grid-section').on("shown.bs.collapse", function(){
                    startSoftwareUpdateProgress();
                });
            }

            if ((gingerbase.capabilities['repo_mngt_tool']) && (gingerbase.capabilities['repo_mngt_tool'] !== "None")) {
                $('#repositories-section').on("shown.bs.collapse", function(){
                    listRepositories();
                });

                initRepositoriesGrid(gingerbase.capabilities['repo_mngt_tool']);
                $('div#repositories-grid-section').removeClass('panel panel-default').addClass('panel-group accordion');
                $('div#repositories-grid-section').prepend($('h3.panel-title'));
                $('h3.panel-title').removeClass('panel-title').wrapInner('<span class="accordion-text" id="repositories-header"></span>').wrapInner('<a role="button" data-toggle="collapse" data-parent="#repositories-grid-section" href="#content-repositories-grid" aria-expanded="false" aria-controls="content-repositories-grid" />');
                $('div#repositories-grid-section .panel-heading').remove();
                $('div#repositories-grid-section .accordion-text').before('<span class="accordion-icon" />');
                $('div#content-repositories-grid').removeClass('panel-body').addClass('panel-collapse collapse');
                $('#repositories-section').switchClass('hidden', gingerbase.capabilities['repo_mngt_tool']);
                $('#content-sys-info').removeClass('col-md-12', gingerbase.capabilities['repo_mngt_tool']);
                $('#content-sys-info').addClass('col-md-4', gingerbase.capabilities['repo_mngt_tool']);
                wok.topic('gingerbase/repositoryAdded')
                    .subscribe(listRepositories);
                wok.topic('gingerbase/repositoryUpdated')
                    .subscribe(listRepositories);
                wok.topic('gingerbase/repositoryDeleted')
                    .subscribe(listRepositories);

            }
        };
        setupUI();
    };

    initPage();

    $('#host-root-container').on('remove', function() {
        if (gingerbase.hostTimer) {
            gingerbase.hostTimer.stop();
            delete gingerbase.hostTimer;
        }

        repositoriesGrid && repositoriesGrid.destroy();
        wok.topic('gingerbase/repositoryAdded')
            .unsubscribe(listRepositories);
        wok.topic('gingerbase/repositoryUpdated')
            .unsubscribe(listRepositories);
        wok.topic('gingerbase/repositoryDeleted')
            .unsubscribe(listRepositories);

        reportGrid && reportGrid.destroy();
        wok.topic('gingerbase/debugReportAdded').unsubscribe(listDebugReports);
        wok.topic('gingerbase/debugReportRenamed').unsubscribe(listDebugReports);
    });

     $('#host-root-container').on('remove', function() {
        if (gingerbase.hostTimer) {
            gingerbase.hostTimer.stop();
            delete gingerbase.hostTimer;
        }

        softwareUpdatesGrid && softwareUpdatesGrid.destroy();
    });
};

gingerbase.isDependOnPackageList = function(depend, packageList) {
    var result = false;
    $.each(packageList, function(index, pack){
        if (pack == depend) {
            result = true;
            return false;
        }
    });
    return result;
};

gingerbase.setUpdateStatusIcon = function(arrayPackages) {
    if (arrayPackages == undefined) {
        arrayPackages = gingerbase.arrayOfPackagesToKeepIcon;
    }
    $.each(arrayPackages, function(index, value){
        var pacakgeNameEscaped = (value.package).replace(/\./g, '\\.');
        if(value.status == 'finished') {
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "]").on('click', function(){return false});
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "] td:nth-child(1) input").attr("disabled", true);
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "] td:nth-child(3)").empty();
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "] td:nth-child(3)").append('<span class="specialClass"><i class="fa fa-check" aria-hidden="true" data-toggle="tooltip" title="'+ i18n['GGBUPD6013M'] +'"></i></span>');
            gingerbase.setUpdateStatusIconForDependecies(value.dependsNotSelected, 'finished');
        } else if (value.status == 'failed') {
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "]").on('click', function(){return false});
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "] td:nth-child(1) input").attr("disabled", true);
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "] td:nth-child(3)").empty();
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "] td:nth-child(3)").append('<span class="specialClass"><i class="fa fa-times" aria-hidden="true" data-toggle="tooltip" title="'+ i18n['GGBUPD6011M'] +'"></i></span>');
            gingerbase.setUpdateStatusIconForDependecies(value.dependsNotSelected, 'failed');
        } else if(value.status == 'running'){
            $("#update-packages").prop('disabled', true);
            $("#update-all-packages").prop('disabled', true);
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "]").on('click', function(){return false});
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "] td:nth-child(1) input").attr("disabled", true);
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "] td:nth-child(3)").empty();
            $("#grid-basic tr[data-row-id=" + pacakgeNameEscaped + "] td:nth-child(3)").append('<span class="specialClass"><i class="fa fa-spinner fa-spin fa-fw" aria-hidden="true" data-toggle="tooltip" title="'+ i18n['GGBUPD6012M'] +'"></i></span>');
            gingerbase.setUpdateStatusIconForDependecies(value.dependsNotSelected, 'running');
            $('[data-toggle="tooltip"]').tooltip();
        }
    });
    $('[data-toggle="tooltip"]').tooltip();
};

gingerbase.setUpdateStatusIconForDependecies = function(arrayDependecies, status) {
    $.each(arrayDependecies, function(index, value){
        var dependencieNameEscaped = (value).replace(/\./g, '\\.');
        if(status == 'finished') {
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "]").on('click', function(){return false});
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(1) input").prop("checked", true);
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(1) input").attr("disabled", true);
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(3)").empty();
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(3)").append('<span class="specialClass"><i class="fa fa-check" aria-hidden="true" data-toggle="tooltip" title="'+ i18n['GGBUPD6013M'] +'"></i></span>');
        } else if(status == 'failed') {
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "]").on('click', function(){return false});
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(1) input").prop("checked", true);
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(1) input").attr("disabled", true);
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(3)").empty();
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(3)").append('<span class="specialClass"><i class="fa fa-times" aria-hidden="true" data-toggle="tooltip" title="'+ i18n['GGBUPD6011M'] +'"></i></span>');
        } else if(status == 'running') {
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "]").on('click', function(){return false});
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(1) input").prop("checked", true);
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(1) input").attr("disabled", true);
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(3)").empty();
            $("#grid-basic tr[data-row-id=" + dependencieNameEscaped + "] td:nth-child(3)").append('<span class="specialClass"><i class="fa fa-spinner fa-spin fa-fw" aria-hidden="true" data-toggle="tooltip" title="'+ i18n['GGBUPD6012M'] +'"></i></span>');
        }
    });
    $('[data-toggle="tooltip"]').tooltip();
}

gingerbase.message = '';
gingerbase.syncUpdatePackages = function(arrayPackages, position) {
    setTimeout(function(){
        gingerbase.arrayOfPackagesToKeepIcon = arrayPackages;
        var count = position + 1;
        $("#update-packages").prop('disabled', true);
        $("#update-all-packages").prop('disabled', true);
        if (arrayPackages.length !== position) {
            if (!arrayPackages[position].isDepend) {
                gingerbase.updateSoftware(arrayPackages[position].package, function(result){
                    $("#update-packages").prop('disabled', true);
                    $("#update-all-packages").prop('disabled', true);
                    arrayPackages[position].status = result['status'];
                    if (result['status'] == 'failed') {
                        $("#update-packages").prop('disabled', true);
                        $("#update-all-packages").prop('disabled', true);
                        $("#update-accordion").show(500);
                        gingerbase.message += arrayPackages[position].package + '   ' + result['message'];
                        $("#software-updates-progress-textarea").text(gingerbase.message);
                    }
                    gingerbase.arrayOfPackagesToKeepIcon = arrayPackages;
                    gingerbase.setUpdateStatusIcon(arrayPackages);
                    gingerbase.syncUpdatePackages(arrayPackages, count);
                    $("#update-packages").prop('disabled', false);
                    $("#update-all-packages").prop('disabled', false);
                }, function(err){
                    wok.message.error(err.responseJSON.reason);
                }, gingerbase.setUpdateStatusIcon);
            } else {
                gingerbase.arrayOfPackagesToKeepIcon = arrayPackages;
                gingerbase.setUpdateStatusIcon(arrayPackages);
                gingerbase.syncUpdatePackages(arrayPackages, count);
                $("#update-packages").prop('disabled', false);
                $("#update-all-packages").prop('disabled', false);
            }
        } else {
            $("#update-all-packages").prop('disabled', false);
            gingerbase.arrayOfPackagesToKeepIcon = [];
            gingerbase.init_update_packages();
        };
    },1000);
};

gingerbase.init_update_packages = function(){
        $("#update-packages").unbind("click");
        $(".wok-mask").fadeIn(300, function() {});
        var packageList = [];
        var packageListNames = [];
        var packagesSelected = [];
        $("#grid-basic tbody tr").remove();
        gingerbase.listSoftwareUpdates(function(softwareUpdates) {
            $(".wok-mask").fadeOut(300, function() {});
            $("#update-packages").prop('disabled', true);
            packageList = softwareUpdates;
            if (packageList.length == 0) {
                $("#update-all-packages").prop('disabled', true);
            }
            var htmlRow = "";
            $.each( softwareUpdates, function( key, value ) {
                htmlRow += "<tr><td>" + value.package_name + "</td><td></td><td>" + value.version + "</td><td>" + value.arch + "</td><td>" + value.repository + "</td></tr>";
                packageListNames.push(value.package_name);
            });
            $("#grid-basic").bootgrid('destroy');
            $("#grid-basic tbody").html(htmlRow);
            $("#grid-basic").bootgrid('destroy');
            $("#grid-basic").bootgrid({
                selection: true,
                multiSelect: true,
                rowSelect: true,
                keepSelection: true,
                sorting: true,
                multiSort: true,
                css: {
                  iconDown: "fa fa-sort-desc",
                  iconUp: "fa fa-sort-asc",
                  center: "text-center"
                },
                labels: {
                  search: i18n['WOKSETT0008M'],
                  infos: i18n['WOKSETT0009M']
                }
            }).on("loaded.rs.jquery.bootgrid", function(e) {
                $('.input-group .glyphicon-search').remove();
                $(".pagination li a").click(function() {
                    setTimeout(function() {
                        gingerbase.setUpdateStatusIcon(gingerbase.arrayOfPackagesToKeepIcon);
                    }, 700);
                });
            }).on("selected.rs.jquery.bootgrid", function(e, rows){
                for (var i = 0; i < rows.length; i++) {
                    packagesSelected.push(rows[i].package_name);
                }
                if (packagesSelected.length > 0) {
                    $("#update-packages").prop('disabled', false);
                } else {
                    $("#update-packages").prop('disabled', true);
                }
            }).on("deselected.rs.jquery.bootgrid", function(e, rows){
                packagesSelected = $.grep(packagesSelected, function(value) {
                    return value != rows[0].package_name;
                });
                if (packagesSelected.length > 0) {
                    $("#update-packages").prop('disabled', false);
                } else {
                    $("#update-packages").prop('disabled', true);
                }
            });

            $("#grid-basic thead .select-box").remove();
        }, function(error){
            wok.message.error(error.responseJSON.reason, '#message-container-area');
        });

        $("#update-packages").on("click", function(evt) {
            evt.preventDefault();
            evt.stopPropagation();
            var resultList = [];

            $("#update-packages").prop('disabled', true);
            $("#update-all-packages").prop('disabled', true);

            $.each(packagesSelected, function( indice, pack ) {
                var resultObject = {
                        package: pack,
                        status: 'running',
                        dependsNotSelected: [],
                        isDepend: false,
                        loopFlag: false
                }

                $("#grid-basic tr[data-row-id=" + pack + "] td:nth-child(3)").empty();
                $("#grid-basic tr[data-row-id=" + pack + "] td:nth-child(3)").append('<span class="specialClass"><i class="fa fa-spinner fa-spin fa-fw" aria-hidden="true" data-toggle="tooltip" title="'+ i18n['GGBUPD6012M'] +'"></i></span>');

                gingerbase.getPackageDeps(pack, function(deplist){
                    $.each(deplist, function(index, depend){
                        if (gingerbase.isDependOnPackageList(depend, packageListNames)) {
                            resultObject.dependsNotSelected.push(depend);
                        }
                    });
                    resultList.push(resultObject);
                }, null);
            });

            $.each(resultList, function(index, packObj){
                packObj.loopFlag = true;
                gingerbase.getPackageDeps(packObj.package, function(deplist){
                    $.each(deplist, function(index2, depend){
                        if (gingerbase.isDependOnPackageList(depend, packagesSelected)) {
                            $.each(resultList, function(index3, packObj2){
                                if (packObj2.package == depend && !packObj2.loopFlag) {
                                    packObj2.isDepend = true;
                                }
                            });
                        }
                    });
                }, null);
            });

            var content = '';
            var modalFlag = false;
            $.each(resultList, function(index, value){
                var len = value.dependsNotSelected.length;
                if (len > 0) {
                    modalFlag = true;
                    content += '<b>' + value.package + ': </b>';
                    $.each(value.dependsNotSelected, function(index2, value2){
                        content += value2;
                        if (index2 != len - 1) {
                            content += ", ";
                        }
                    });
                    content += '<br />';
                }
            });
            if (modalFlag) {
                var settings = {
                    title : i18n['GGBUPD6014M'],
                    content : content,
                    confirm : 'Yes',
                    cancel : 'No'
                };
                wok.confirm(settings, function() {
                    wok.window.close();
                    setTimeout(function() {
                        gingerbase.message = '';
                        gingerbase.setUpdateStatusIcon(resultList);
                        gingerbase.syncUpdatePackages(resultList, 0);
                    }, 400);

                },function(){
                    $.each(packagesSelected, function( indice, pack ) {
                        $("#grid-basic tr[data-row-id=" + pack + "] td:nth-child(3)").empty();
                    });

                    $("#update-packages").prop('disabled', false);
                    $("#update-all-packages").prop('disabled', false);
                });
            } else if(packagesSelected.length > 0) {
                gingerbase.message = '';
                gingerbase.setUpdateStatusIcon(resultList);
                gingerbase.syncUpdatePackages(resultList, 0);
            }
        });
};
/*
 * Project Ginger Base
 *
 * Copyright IBM Corp, 2015-2016
 *
 * Code derived from Project Kimchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
gingerbase.report_add_main = function() {
    var reportGridID = 'available-reports-grid';
    var addReportForm = $('#form-report-add');
    var submitButton = $('#button-report-add');
    var nameTextbox = $('input[name="name"]', addReportForm);
    nameTextbox.select();

    var submitForm = function(event) {
        if(submitButton.prop('disabled')) {
            return false;
        }
        var reportName = nameTextbox.val();
        var validator = RegExp("^[A-Za-z0-9-]*$");
        if (!validator.test(reportName)) {
            wok.message.error.code('GGBDR6011M','#alert-modal-debugreportadd-container', true);
            return false;
        }
        var formData = addReportForm.serializeObject();
        var taskAccepted = false;
        var onTaskAccepted = function() {
            if(taskAccepted) {
                return;
            }
            taskAccepted = true;
            wok.topic('gingerbase/debugReportAdded').publish();
            $('#button-report-cancel').trigger('click');
        };

        gingerbase.createReport(formData, function(result) {
            onTaskAccepted();
            wok.topic('gingerbase/debugReportAdded').publish();
             $('#button-report-cancel').trigger('click');
        }, function(result) {
            // Error message from Async Task status
            if (result['message']) {
                var errText = result['message'];
            }
            // Error message from standard gingerbase exception
            else {
                var errText = result['responseJSON']['reason'];
            }
            result && wok.message.error(errText,'#alert-modal-debugreportadd-container', true);

            taskAccepted &&
                $('.grid-body-view table tr:first-child',
                    '#' + reportGridID).remove();
            submitButton.prop('disabled', false);
            nameTextbox.select();
        }, onTaskAccepted);

        event.preventDefault();
    };

    addReportForm.on('submit', submitForm);
    submitButton.on('click', submitForm);
};
/*
 * Project Ginger Base
 *
 * Copyright IBM Corp, 2015-2016
 *
 * Code derived from Project Kimchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
gingerbase.report_rename_main = function() {
    var renameReportForm = $('#form-report-rename');
    var submitButton = $('#button-report-rename');
    var nameTextbox = $('input[name="name"]', renameReportForm);
    var submitForm = function(event) {
        if(submitButton.prop('disabled')) {
            return false;
        }
        var reportName = nameTextbox.val();

        // if the user hasn't changed the report's name,
        // nothing should be done.
        if (reportName == gingerbase.selectedReport) {
        wok.message.error.code('GGBDR6013M','#alert-modal-debugreportrename-container', true);
        return false;
        }

        var validator = RegExp("^[A-Za-z0-9-]*$");
        if (!validator.test(reportName)) {
            wok.message.error.code('GGBDR6011M','#alert-modal-debugreportrename-container', true);
            return false;
        }
        var formData = renameReportForm.serializeObject();
        submitButton.prop('disabled', true);
        nameTextbox.prop('disabled', true);
        gingerbase.renameReport(gingerbase.selectedReport, formData, function(result) {
            submitButton.prop('disabled', false);
            nameTextbox.prop('disabled', false);
            wok.window.close();
            wok.topic('gingerbase/debugReportRenamed').publish({
                result: result
            });
        }, function(result) {
            var errText = result &&
                result['responseJSON'] &&
                result['responseJSON']['reason'];
            wok.message.error(errText,'#alert-modal-debugreportrename-container', true);
            submitButton.prop('disabled', false);
            nameTextbox.prop('disabled', false).focus();
        });

        event.preventDefault();
    };

    renameReportForm.on('submit', submitForm);
    submitButton.on('click', submitForm);

    nameTextbox.val(gingerbase.selectedReport).select();
};
/*
 * Project Ginger Base
 *
 * Copyright IBM Corp, 2015-2016
 *
 * Code derived from Project Kimchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
gingerbase.repository_add_main = function() {

    var addForm = $('#form-repository-add');
    var addButton = $('#button-repository-add');

    if(gingerbase.capabilities['repo_mngt_tool']=="yum") {
        addForm.find('div.deb').hide();
    }
    else if(gingerbase.capabilities['repo_mngt_tool']=="deb") {
        addForm.find('div.yum').hide();
    }

    var validateField = function(event) {
        var valid=($(this).val()!=='');
        $(addButton).prop('disabled', !valid);
        return(valid);
    };

    var validateForm = function(event) {
        var valid=false;
        addForm.find('input.required').each( function() {
            valid=($(this).val()!=='');
            return(!valid);
        });
        return(valid);
    }

    addForm.find('input.required').on('input propertychange', validateField);

    var weedObject = function(obj) {
        for (var key in obj) {
            if (obj.hasOwnProperty(key)) {
                if((typeof(obj[key])==="object") && !Array.isArray(obj[key])) {
                    weedObject(obj[key]);
                }
                else if(obj[key] == '') {
                    delete obj[key];
                }
            }
        }
    }

    var addRepository = function(event) {
        var valid = validateForm();
        if(!valid) {
            return false;
        }

        var formData = $(addForm).serializeObject();

        if (formData && formData.isMirror!=undefined) {
            formData.isMirror=(String(formData.isMirror).toLowerCase() === 'true');
        }
        if(formData.isMirror) {
            if(formData.config==undefined) {
                formData.config=new Object();
            }
            formData.config.mirrorlist=formData.baseurl;
            delete formData.baseurl;
            delete formData.isMirror;
        }
        weedObject(formData);
        if(formData.config && formData.config.comps) {
            formData.config.comps=formData.config.comps.split(/[,\s]/);
            for(var i=0; i>formData.config.comps.length; i++) {
                formData.config.comps[i]=formData.config.comps[i].trim();
            }
            for (var j=formData.config.comps.indexOf(""); j!=-1; j=formData.config.comps.indexOf("")) {
                formData.config.comps.splice(j, 1);
            }
        }

        gingerbase.createRepository(formData, function() {
            wok.topic('gingerbase/repositoryAdded').publish();
            $("#repositories-grid-enable-button").attr('disabled', true);
            $("#repositories-grid-edit-button").attr('disabled', true);
            $("#repositories-grid-remove-button").attr('disabled', true);
            wok.window.close();
        }, function(jqXHR, textStatus, errorThrown) {
            var reason = jqXHR &&
                jqXHR['responseJSON'] &&
                jqXHR['responseJSON']['reason'];
            wok.message.error(reason, '#alert-modal-container');
        });
        return false;
    };

    $(addForm).on('submit', function(e){
        e.preventDefault();
        e.stopPropagation();
        addRepository();
    });

    $(addButton).on('click',function(e){
        e.preventDefault();
        e.stopPropagation();
        $(addForm).submit();
    });
};
/*
 * Project Ginger Base
 *
 * Copyright IBM Corp, 2015-2016
 *
 * Code derived from Project Kimchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
gingerbase.repository_edit_main = function() {

    var editForm = $('#form-repository-edit');
    var saveButton = $('#repository-edit-button-save');

    if(gingerbase.capabilities['repo_mngt_tool']=="yum") {
        editForm.find('div.deb').hide();
    }
    else if(gingerbase.capabilities['repo_mngt_tool']=="deb") {
        editForm.find('div.yum').hide();
    }

    gingerbase.retrieveRepository(gingerbase.selectedRepository, function(repository) {
        editForm.fillWithObject(repository);

        $('input', editForm).on('input propertychange', function(event) {
            if($(this).val() !== '') {
                $(saveButton).prop('disabled', false);
            }
        });
    });


    var editRepository = function(event) {
        var formData = $(':input:visible', $(editForm)).serializeObject();

        if (formData && formData.config) {
            formData.config.gpgcheck=(String(formData.config.gpgcheck).toLowerCase() === 'true');
        }

        if(formData.config && formData.config.comps) {
            formData.config.comps=formData.config.comps.split(/[,\s]/);
            for(var i=0; i>formData.config.comps.length; i++) {
                formData.config.comps[i]=formData.config.comps[i].trim();
            }
            for (var j=formData.config.comps.indexOf(""); j!=-1; j=formData.config.comps.indexOf("")) {
                formData.config.comps.splice(j, 1);
            }
        }

        gingerbase.updateRepository(gingerbase.selectedRepository, formData, function() {
            wok.topic('gingerbase/repositoryUpdated').publish();
            $("#repositories-grid-enable-button").attr('disabled', true);
            $("#repositories-grid-edit-button").attr('disabled', true);
            $("#repositories-grid-remove-button").attr('disabled', true);
            wok.window.close();
        }, function(jqXHR, textStatus, errorThrown) {
            var reason = jqXHR &&
                jqXHR['responseJSON'] &&
                jqXHR['responseJSON']['reason'];
            wok.message.error(reason, '#alert-modal-container');
        });

        return false;
    };

    $(editForm).on('submit', editRepository);
    $(saveButton).on('click', editRepository);
};
