local mymodule = {}

-- Load libraries
modelfunctions = require("modelfunctions")
posix = require("posix")
fs = require("acf.fs")
format = require("acf.format")
validator = require("acf.validator")
db = require("acf.db")
session = require("session")
socket = require("socket")

-- Set variables
local configfile = "/etc/freeswitchvmail.conf"
local db_path = "/var/lib/freeswitch/db/"
local configcontent = fs.read_file(configfile) or ""
local config = format.parse_ini_file(configcontent, "") or {}
config.database = config.database or db_path.."voicemail_default.db"
config.domain = config.domain or "voicemail"
config.event_socket_ip = config.event_socket_ip or "127.0.0.1"
config.event_socket_port = config.event_socket_port or "8021"
config.event_socket_password = config.event_socket_password or "ClueCon"
config.callback_command = config.callback_command or "originate {ignore_early_media=true,origination_caller_id_name='Voicemail',origination_caller_id_number='Voicemail'}sofia/gateway/asterlink.com/$1 &playback($2)"

local vmaildb
local dbengine = db.engine.sqlite3
if not config.dsn then
	-- Backward compatible assumed sqlite3 with full path in config.database
	config.dsn = "sqlite://"..config.database
end
if string.find(config.dsn, "^sqlite://") then
	local params = string.match(config.dsn, "^sqlite://(.*)") or ""
	if not string.find(params, "^/") then
		params = db_path..params
	end
	vmaildb = db.create(dbengine, params)
elseif string.find(config.dsn, "^pgsql://") then
	local params = string.match(config.dsn, "^pgsql://(.*)") or ""
	dbengine = db.engine.postgresql
	-- Sample params: hostaddr=127.0.0.1 dbname=freeswitch user=freeswitch password= options='-c client_min_messages=NOTICE' application_name='freeswitch'
	-- Second sample: hostaddr=127.0.0.1 dbname=voicemail user=postgres password= options=''
	local hostaddr = string.match(params, "hostaddr=(%S*)") or "127.0.0.1"
	local dbname = string.match(params, "dbname=(%S*)") or "freeswitch"
	local user = string.match(params, "user=(%S*)")
	local password = string.match(params, "password=(%S*)")
	vmaildb = db.create(dbengine, dbname, user, password, hostaddr)
else
	-- Failure - we do not support ODBC database
end

vmaildb.table_creation_scripts = {
	voicemail_values = {
		"CREATE TABLE voicemail_values (uid INTEGER, nid INTEGER, value VARCHAR(255), PRIMARY KEY(uid, nid))",
	},
	voicemail_folders = {
		"CREATE TABLE voicemail_folders (in_folder VARCHAR(255) PRIMARY KEY, label VARCHAR(255))",
		"INSERT INTO voicemail_folders VALUES('inbox', 'Inbox')",
	},
	-- The voicemail_prefs table is created by Freeswitch mod_voicemail.c, but we duplicate here in case not already created
	voicemail_prefs = {"CREATE TABLE voicemail_prefs (username VARCHAR(255), domain VARCHAR(255), name_path VARCHAR(255), greeting_path VARCHAR(255), password VARCHAR(255))",
		"create index voicemail_prefs_idx1 on voicemail_prefs(username)",
		"create index voicemail_prefs_idx2 on voicemail_prefs(domain)",
	},
}

if dbengine == db.engine.postgresql then
	vmaildb.table_creation_scripts.voicemail_users = {
		"CREATE TABLE voicemail_users (uid SERIAL PRIMARY KEY, username VARCHAR(255) UNIQUE)",
		"CREATE INDEX users_username_idx ON voicemail_users (username)",
	}
	vmaildb.table_creation_scripts.voicemail_params = {
		"CREATE TABLE voicemail_params (nid SERIAL PRIMARY KEY, name VARCHAR(255) UNIQUE, type VARCHAR(255), label VARCHAR(255), descr TEXT, value VARCHAR(255), seq INTEGER)",
		"CREATE INDEX params_name_idx ON voicemail_params (name)",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'username', 'text', 'Extension', '', '', '1')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'firstname', 'text', 'User First Name', '', '', '2')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'lastname', 'text', 'User Last Name', '', '', '3')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'vm-password', 'password', 'Voicemail Password', 'Passwords must be all numbers and at least three digits', '', '4')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'vm-password-confirm', 'password', 'Enter again to confirm', '', '', '5')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'vm-mailto', 'text', 'Email Address', 'Email a notification, including audio file if enabled', '', '6')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'vm-email-all-messages', 'boolean', 'Email Enable', '', 'false', '7')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'vm-attach-file', 'boolean', 'Attach voicemail to email', 'Option to attach audio file to emaie', 'false', '8')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'vm-keep-local-after-email', 'boolean', 'Keep voicemail', 'When disabled, the message will be deleted from the voicemailbox after any notification email is sent (whether or not email is enabled). This allows receiving voicemail via email alone, rather than having the voicemail available from the Web interface or by telephone. CAUTION: \"Email Address\" must be valid and \"Email Enable\" and \"Attach voicemail to email\" must be enabled, OTHERWISE YOUR MESSAGES WILL BE LOST FOREVER.', 'true', '9')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'vm-notify-mailto', 'text', 'Pager Email Address', 'Email a short notification', '', '10')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'vm-notify-email-all-messages', 'boolean', 'Pager Email Enable', '', 'false', '11')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'callmenumber', 'text', 'Call Me Number', '', '', '12')",
		"INSERT INTO voicemail_params VALUES(DEFAULT, 'timezone', 'text', 'Timezone Settings', 'Linux timezone format (ex. America/New_York)', '', '13')",
	}
else
	vmaildb.table_creation_scripts.voicemail_users = {
		"CREATE TABLE voicemail_users (uid INTEGER PRIMARY KEY, username VARCHAR(255) UNIQUE)",
		"CREATE INDEX users_username_idx ON voicemail_users (username)",
	}
	vmaildb.table_creation_scripts.voicemail_params = {
		"CREATE TABLE voicemail_params (nid INTEGER PRIMARY KEY, name VARCHAR(255) UNIQUE, type VARCHAR(255), label VARCHAR(255), descr TEXT, value VARCHAR(255), seq INTEGER)",
		"CREATE INDEX params_name_idx ON voicemail_params (name)",
		"INSERT INTO voicemail_params VALUES(null, 'username', 'text', 'Extension', '', '', '1')",
		"INSERT INTO voicemail_params VALUES(null, 'firstname', 'text', 'User First Name', '', '', '2')",
		"INSERT INTO voicemail_params VALUES(null, 'lastname', 'text', 'User Last Name', '', '', '3')",
		"INSERT INTO voicemail_params VALUES(null, 'vm-password', 'password', 'Voicemail Password', 'Passwords must be all numbers and at least three digits', '', '4')",
		"INSERT INTO voicemail_params VALUES(null, 'vm-password-confirm', 'password', 'Enter again to confirm', '', '', '5')",
		"INSERT INTO voicemail_params VALUES(null, 'vm-mailto', 'text', 'Email Address', 'Email a notification, including audio file if enabled', '', '6')",
		"INSERT INTO voicemail_params VALUES(null, 'vm-email-all-messages', 'boolean', 'Email Enable', '', 'false', '7')",
		"INSERT INTO voicemail_params VALUES(null, 'vm-attach-file', 'boolean', 'Attach voicemail to email', 'Option to attach audio file to emaie', 'false', '8')",
		"INSERT INTO voicemail_params VALUES(null, 'vm-keep-local-after-email', 'boolean', 'Keep voicemail', 'When disabled, the message will be deleted from the voicemailbox after any notification email is sent (whether or not email is enabled). This allows receiving voicemail via email alone, rather than having the voicemail available from the Web interface or by telephone. CAUTION: \"Email Address\" must be valid and \"Email Enable\" and \"Attach voicemail to email\" must be enabled, OTHERWISE YOUR MESSAGES WILL BE LOST FOREVER.', 'true', '9')",
		"INSERT INTO voicemail_params VALUES(null, 'vm-notify-mailto', 'text', 'Pager Email Address', 'Email a short notification', '', '10')",
		"INSERT INTO voicemail_params VALUES(null, 'vm-notify-email-all-messages', 'boolean', 'Pager Email Enable', '', 'false', '11')",
		"INSERT INTO voicemail_params VALUES(null, 'callmenumber', 'text', 'Call Me Number', '', '', '12')",
		"INSERT INTO voicemail_params VALUES(null, 'timezone', 'text', 'Timezone Settings', 'Linux timezone format (ex. America/New_York)', '', '13')",
	}
end

-- ################################################################################
-- LOCAL FUNCTIONS

-- Use socket.protect to create a function that will not throw exceptions and cleans itself up
local send_to_event_socket = socket.protect(function(cmd)
	-- connect to freeswitch
	local conn = socket.try(socket.connect(config.event_socket_ip, config.event_socket_port))
	-- create a try function that closes 'conn' on error
	local try = socket.newtry(function() conn:close() end)
	-- do everything reassured conn will be closed
	local out = {}
	repeat
		out[#out+1] = try(conn:receive())
	until out[#out] == ""
	for i,c in ipairs(cmd) do
		posix.sleep(0)
		try(conn:send(c.."\n\n"))
		repeat
			out[#out+1] = try(conn:receive())
		until out[#out] == ""
	end
	conn:close()
	return table.concat(out, "\n") or ""
end)

local function voicemail_inject(user, domain, sound_file, cid_num, cid_name)
	local cmd = {"auth "..config.event_socket_password, "api voicemail_inject "..user.."@"..domain.." "..sound_file.." "..cid_num.." "..string.gsub(cid_name, " ", "%%20"), "exit"}
	return send_to_event_socket(cmd)
end

local function voicemail_callback(extension, sound_file, username)
	local c = config.callback_command
	c = c:gsub("%$1", extension)
	c = c:gsub("%$2", sound_file)
	c = c:gsub("%$3", username)
	local cmd = {"auth "..config.event_socket_password, "bgapi "..c, "exit"}
	return send_to_event_socket(cmd)
end

-- Need to update the voicemail module to turn off MWI
local function voicemail_update(user, domain)
	local cmd = {"auth "..config.event_socket_password, "api vm_list "..user.."@"..domain, "exit"}
	return send_to_event_socket(cmd)
end

local function voicemail_read(user, domain, message)
	local cmd = {"auth "..config.event_socket_password, "api vm_read "..user.."@"..domain.." read "..message, "exit"}
	return send_to_event_socket(cmd)
end

local databasedisconnect = function()
	vmaildb.databasedisconnect()
	if dbengine == db.engine.sqlite3 then
		posix.chown(vmaildb.database, posix.getpasswd("freeswitch", "uid") or 0, posix.getpasswd("freeswitch", "gid") or 0)
	end
end

local generatewhereclause = function(username, message, foldername, uid)
        local sql = ""
	local where = {}
	if username and username ~= "" then
		where[#where+1] = "username = '"..vmaildb.escape(username).."'"
	end
	if message and type(message) == "string" and message ~= "" then
		where[#where+1] = "uuid = '"..vmaildb.escape(message).."'"
	elseif message and type(message) == "table" and #message > 0 then
		local where2 = {}
		for i,m in ipairs(message) do
			where2[#where2+1] = "uuid = '"..vmaildb.escape(m).."'"
		end
		where[#where+1] = "(" .. table.concat(where2, " OR ") .. ")"
	end
	if foldername and foldername ~= "" then
		where[#where+1] = "in_folder = '"..vmaildb.escape(foldername).."'"
	end
	if uid and uid ~= "" then
		where[#where+1] = "uid = '"..vmaildb.escape(uid).."'"
	end
	if #where > 0 then
		sql = " WHERE " .. table.concat(where, " AND ")
	end
	return sql
end

-- These funtions access the new voicemail tables added for ACF

local listfolders = function(foldername)
	local sql = "SELECT * FROM voicemail_folders" .. generatewhereclause(nil, nil, foldername).." ORDER BY label"
	return vmaildb.getselectresponse(sql)
end

local validfolder = function(foldername)
	return foldername and (foldername ~= "") and (#listfolders(foldername) > 0)
end

local listusers = function(username)
	local sql = "SELECT * FROM voicemail_users" .. generatewhereclause(username).." ORDER BY username"
	return vmaildb.getselectresponse(sql)
end

local validuser = function(username)
	return username and (username ~= "") and (#listusers(username) > 0)
end

local getuserparams = function(username)
	local retval = {}
	local sql = "SELECT * FROM voicemail_params"
	local params = vmaildb.getselectresponse(sql)
	local reverse_nids = {}
	for i,parm in ipairs(params) do
		if parm.name then
			reverse_nids[parm.nid] = parm.name
			retval[parm.name] = {}
			for n,v in pairs(parm) do
				retval[parm.name][n] = v
			end
			if retval[parm.name].type == "boolean" then
				retval[parm.name].value = (retval[parm.name].value == "true")
			end
		end
	end
	if retval.username and username then
		retval.username.value = username
		retval.username.readonly = true

		local users = listusers(username)
		if #users == 1 then
			-- Get password from voicemail_prefs
			sql = "SELECT password FROM voicemail_prefs"..generatewhereclause(username)
			local password = vmaildb.getselectresponse(sql)
			if retval["vm-password"] and password[1] then
				retval["vm-password"].value = password[1].password
			end

			local uid = users[1].uid
			if uid then
				-- Get other parameters from voicemail_values
				sql = "SELECT * FROM voicemail_values"..generatewhereclause(nil, nil, nil, uid)
				local params = vmaildb.getselectresponse(sql)
				for i,param in ipairs(params) do
					if param.nid and reverse_nids[param.nid] and retval[reverse_nids[param.nid]] and param.value then
						if retval[reverse_nids[param.nid]].type == "boolean" then
							param.value = (param.value == "true")
						end
						retval[reverse_nids[param.nid]].value = param.value
					end
				end
			end
		end
	end
	return retval
end

local setuserparams = function(userparams)
	if not userparams.username or not userparams.username.value or not validuser(userparams.username.value) then
		return false, "Invalid User"
	end
	local success = true
	local sql = "SELECT * FROM voicemail_params"
	local params = vmaildb.getselectresponse(sql)
	-- Get the uid that corresponds to this username
	sql = "SELECT uid FROM voicemail_users"..generatewhereclause(userparams.username.value)
	local uid = vmaildb.getselectresponse(sql)
	if #uid == 1 then
		-- There are a few params not to put in the voicemail_values table
		local ignoreparam = { username=true, ["vm-password"]=true, ["vm-password-confirm"]=true }
		vmaildb.runsqlcommand("BEGIN TRANSACTION")
		sql = "DELETE FROM voicemail_values"..generatewhereclause(nil, nil, nil, uid[1].uid).." AND nid IN (SELECT nid FROM voicemail_params)"
		vmaildb.runsqlcommand(sql, true)
		for i,parm in ipairs(params) do
			if parm.name and not ignoreparam[parm.name] then
				if userparams[parm.name] and (userparams[parm.name].value ~= nil) and tostring(userparams[parm.name].value) ~= parm.value then
					sql = "INSERT INTO voicemail_values VALUES('"..vmaildb.escape(uid[1].uid).."', '"..vmaildb.escape(parm.nid).."', '"..vmaildb.escape(tostring(userparams[parm.name].value)).."')"
					vmaildb.runsqlcommand(sql, true)
				end
			end
		end
	end
	-- Set password to voicemail_prefs
	if userparams["vm-password"] and userparams["vm-password"].value and userparams["vm-password"].value ~= "" then
		sql = "SELECT password FROM voicemail_prefs"..generatewhereclause(userparams.username.value)
		local password = vmaildb.getselectresponse(sql, true)
		if #password > 0 then
			-- update
			sql = "UPDATE voicemail_prefs SET password='"..vmaildb.escape(userparams["vm-password"].value).."'"..generatewhereclause(userparams.username.value)
		else
			-- insert
			sql = "INSERT INTO voicemail_prefs (username, domain, password) VALUES ('"..vmaildb.escape(userparams.username.value).."', '"..vmaildb.escape(config.domain).."', '"..vmaildb.escape(userparams["vm-password"].value).."')"
		end
		vmaildb.runsqlcommand(sql, true)
	end
	vmaildb.runsqlcommand("COMMIT")
	return success
end

local function validateconfig(newconfig)
	local success = true
	if newconfig.value.domain.value == "" then
		newconfig.value.domain.errtxt = "Cannot be blank"
		success = false
	end
	if newconfig.value.dsn.value == "" then
		newconfig.value.dsn.errtxt = "Cannot be blank"
		success = false
	end
	if newconfig.value.event_socket_ip.value == "" then
		newconfig.value.event_socket_ip.errtxt = "Cannot be blank"
		success = false
	end
	if newconfig.value.event_socket_port.value == "" then
		newconfig.value.event_socket_port.errtxt = "Cannot be blank"
		success = false
	end
	if newconfig.value.event_socket_password.value == "" then
		newconfig.value.event_socket_password.errtxt = "Cannot be blank"
		success = false
	end
	if newconfig.value.callback_command.value == "" then
		newconfig.value.callback_command.errtxt = "Cannot be blank"
		success = false
	end
	return success, newconfig
end

local delete_message = function(messages, username)
	local retval = ""

	local sql = "SELECT * FROM voicemail_msgs"
	sql = sql .. generatewhereclause(username, messages)
	local tmp = vmaildb.getselectresponse(sql)
	if #tmp == #messages then
		sql = "DELETE FROM voicemail_msgs" .. generatewhereclause(username, messages)
		vmaildb.runsqlcommand(sql)
		for i,t in ipairs(tmp) do
			os.remove(t.file_path)
		end
		if #messages == 1 then
			retval = "Deleted message"
		else
			retval = "Deleted "..#messages.." messages"
		end
		if username then
			voicemail_update(username, config.domain)
		else
			for i,m in ipairs(tmp) do
				voicemail_update(m.username, m.domain)
			end
		end
	end

	if not res and err then
		return false, err
	end

	return retval
end

local delete_user = function(username)
	local users = listusers(username)
	if #users == 0 then
		return false, "User does not exist"
	else
		local recording_path

		-- Delete all of the user's voicemails
		local messages = mymodule.list_messages(username)
		if #messages.value > 0 then
			recording_path = posix.dirname(messages.value[1].file_path)
			local uuids = {}
			for i,m in ipairs(messages.value) do
				uuids[#uuids+1] = m.uuid
			end
			delete_message(uuids)
		end
		-- Remove the greetings
		if not recording_path then
			local sql = "SELECT * FROM voicemail_prefs"..generatewhereclause(username)
			local prefs = vmaildb.getselectresponse(sql)
			if prefs and prefs[1] then
				if prefs[1].greeting_path then
					recording_path = posix.dirname(prefs[1].greeting_path)
				elseif prefs[1].name_path then
					recording_path = posix.dirname(prefs[1].name_path)
				end
			end
		end
		if recording_path then
			fs.remove_directory(recording_path)
		end
		-- Remove the user parameters
		local sql = "DELETE FROM voicemail_values " .. generatewhereclause(nil, nil, nil, users[1].uid)
		vmaildb.runsqlcommand(sql)
		-- Remove the user password
		sql = "DELETE FROM voicemail_prefs " .. generatewhereclause(username)
		vmaildb.runsqlcommand(sql)
		-- Remove the user
		sql = "DELETE FROM voicemail_users " .. generatewhereclause(nil, nil, nil, users[1].uid)
		vmaildb.runsqlcommand(sql)
		result = "Voicemail User Deleted"
	end
	return true
end

local validateentry = function(entry)
	local success = true
	-- Validate the settings
	if entry.value["vm-password"] and not string.match(entry.value["vm-password"].value, "^%d%d%d+$") then
		success = false
		entry.value["vm-password"].errtxt = "Passwords must be all numbers and at least three digits"
	end
	if entry.value["vm-password"] and entry.value["vm-password-confirm"] and entry.value["vm-password"].value ~= entry.value["vm-password-confirm"].value then
		success = false
		entry.value["vm-password-confirm"].errtxt = "Password does not match"
	end
	return success
end

-- ################################################################################
-- PUBLIC FUNCTIONS

mymodule.get_config = function()
	local result = {}
	result.domain = cfe({ value=config.domain, label="Domain" })
	result.dsn = cfe({ value=config.dsn, label="Data Source Name (DSN)" })
	result.event_socket_ip = cfe({ value=config.event_socket_ip, label="FS Event Socket IP" })
	result.event_socket_port = cfe({ value=config.event_socket_port, label="FS Event Socket Port" })
	result.event_socket_password = cfe({ value=config.event_socket_password, label="FS Event Socket Password" })
	result.callback_command = cfe({ value=config.callback_command, label="FS API Command for Callback", desc="Use $1 for extension, $2 for audio filename, and $3 for originating user. No other parameters allowed." })
	return cfe({ type="group", value=result, label="Voicemail Config" })
end

mymodule.update_config = function(self, newconfig)
	local success = validateconfig(newconfig)
	if success then
		for name,val in pairs(newconfig.value) do
			configcontent = format.update_ini_file(configcontent, "", name, tostring(val.value))
		end

		fs.write_file(configfile, configcontent)
		config = format.parse_ini_file(configcontent, "") or {}
	else
		newconfig.errtxt = "Failed to update config"
	end

	return newconfig
end

mymodule.list_messages = function(username)
	local retval = {}
	local errtxt
        local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()
		local sql = "SELECT * FROM voicemail_msgs"
		sql = sql .. generatewhereclause(username)
		sql = sql .. " ORDER BY username ASC, created_epoch ASC"
		retval = vmaildb.getselectresponse(sql)
		if connected then databasedisconnect() end
	end)
	if not res and err then
		errtxt = err
	end
	return cfe({ type="structure", value=retval, label="Messages", errtxt=errtxt })
end

mymodule.get_download_message = function(self, clientdata)
	local result = {}
	result.message = cfe({ label="Message" })
	result.username = cfe({ label="User Name" })
	return cfe({ type="group", value=result, label="Download Message" })
end

mymodule.download_message = function(self, downloadrequest)
	local file = cfe({ type="raw", label="error", option="audio/x-wav" })
        local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()
		local sql = "SELECT username, file_path FROM voicemail_msgs"
		sql = sql .. generatewhereclause(downloadrequest.value.username.value, downloadrequest.value.message.value)
		local tmp = vmaildb.getselectresponse(sql)
		if connected then databasedisconnect() end
		if #tmp == 0 then
			downloadrequest.errtxt = "Invalid message"
		else
			file.label = posix.basename(tmp[1].file_path)
			file.value = fs.read_file(tmp[1].file_path)
			file.length = #file.value
			local option = string.match(file.label, "[^.]*$")
			if "wav" == option then option = "x-wav"
			elseif "mp3" == option then option = "mpeg"
			end
			file.option = "audio/"..option
			-- Mark the message as read
			voicemail_read(tmp[1].username, config.domain, downloadrequest.value.message.value)
		end
		downloadrequest.value.file = file
	end)
	if not res and err then
		downloadrequest.errtxt = err
	end
	return downloadrequest
end

mymodule.get_delete_message = function(self, clientdata)
	local result = {}
	result.message = cfe({ value=clientdata.message or "", label="Message" })
	result.username = cfe({ value=clientdata.username or "", label="User Name" })
	return cfe({ type="group", value=result, label="Delete Message" })
end

mymodule.set_delete_message = function(self, deleterequest)
	if not deleterequest.value.message.value or deleterequest.value.message.value == "" then
		deleterequest.errtxt = "Failed to delete message - message not found"
		return deleterequest
	end
	local messages = format.string_to_table(deleterequest.value.message.value, "%s*,%s*")

	local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()
		res, err = delete_message(messages, deleterequest.value.username.value)
		if connected then databasedisconnect() end
	end)

	if res then
		deleterequest.descr = res
	end

	if not res and err then
		deleterequest.errtxt = err
	end

	return deleterequest
end

mymodule.get_forward_message = function(self, clientdata)
	local result = {}
	result.message = cfe({ value=clientdata.message or "", label="Message" })
	result.username = cfe({ value=clientdata.username or "", label="User Name" })
	result.newuser = cfe({ value=clientdata.newuser or "", label="New User" })
	return cfe({ type="group", value=result, label="Forward Message" })
end

mymodule.forward_message = function(self, forwardrequest)
	local messages = format.string_to_table(forwardrequest.value.message.value, "%s*,%s*")
        local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()
		-- Check if message exists
		local sql = "SELECT * FROM voicemail_msgs" .. generatewhereclause(forwardrequest.value.username.value, messages)
		local mess = vmaildb.getselectresponse(sql)
		if #mess == #messages then
			-- Check if newuser exists
			if validuser(forwardrequest.value.newuser.value) then
				for i,m in ipairs(mess) do
					-- Forward message using mod_voicemail API
					-- doesn't seem like there's any way to tell whether or not it worked
					voicemail_inject(forwardrequest.value.newuser.value, config.domain, m.file_path, m.cid_number, m.cid_name)
				end
				if #mess == 1 then
					forwardrequest.descr = "Forwarded message"
				else
					forwardrequest.descr = "Forwarded "..#mess.." messages"
				end
			else
				forwardrequest.errtxt = "Failed to forward message - invalid user"
			end
		else
			forwardrequest.errtxt = "Failed to forward message - message not found"
		end
		if connected then databasedisconnect() end
	end)
	if not res and err then
		forwardrequest.errtxt = err
	end

	return forwardrequest
end

mymodule.get_email_message = function(self, clientdata)
	local result = {}
	result.message = cfe({ value=clientdata.message or "", label="Message" })
	result.address = cfe({ value=clientdata.newuser or "", label="Address" })
	result.username = cfe({ value=clientdata.username or "", label="User Name" })
	return cfe({ type="group", value=result, label="Email Message" })
end

mymodule.email_message = function(self, emailrequest)
	local messages = format.string_to_table(emailrequest.value.message.value, "%s*,%s*")
	if emailrequest.value.address.value == "" or string.find(emailrequest.value.address.value, "%s") or not string.find(emailrequest.value.address.value, "@") then
		emailrequest.errtxt = "Failed to e-mail message - invalid address"
	else
        	local res, err = pcall(function()
			local connected = vmaildb.databaseconnect()
			-- Check if message exists
			local sql = "SELECT * FROM voicemail_msgs" .. generatewhereclause(emailrequest.value.username.value, messages)
			local mess = vmaildb.getselectresponse(sql)
			if #mess == #messages then
				-- Create a temporary user and settings
				local newuser = "tempuser"..session.random_hash(128)
				while validuser(newuser) do
					newuser = "tempuser"..session.random_hash(128)
				end
				local settings = mymodule.get_usersettings(self, {username=newuser})
				if settings.value["vm-mailto"] and settings.value["vm-email-all-messages"] and settings.value["vm-attach-file"] and settings.value["vm-keep-local-after-email"] then
					settings.value["vm-mailto"].value = emailrequest.value.address.value
					settings.value["vm-email-all-messages"].value = true
					settings.value["vm-attach-file"].value = true
					settings.value["vm-keep-local-after-email"].value = true -- this allows us to delete the folder in delete_user
					if settings.value["vm-password"] then settings.value["vm-password"].value = "1234" end
					if settings.value["vm-password-confirm"] then settings.value["vm-password-confirm"].value = "1234" end
					settings = mymodule.create_usersettings(self, settings)
					if not settings.errtxt then
						for i,m in ipairs(mess) do
							-- E-mail message using mod_voicemail API
							-- doesn't seem like there's any way to tell whether or not it worked
							voicemail_inject(newuser, config.domain, m.file_path, m.cid_number, m.cid_name)
						end
						if #mess == 1 then
							emailrequest.descr = "E-mailed message"
						else
							emailrequest.descr = "E-mailed "..#mess.." messages"
						end
						-- Now, delete the temporary user
						delete_user(newuser)
					else
						emailrequest.errtxt = "Failed to e-mail message - "..settings.errtxt
					end
				else
					emailrequest.errtxt = "Failed to e-mail message - unsupported"
				end
			else
				emailrequest.errtxt = "Failed to e-mail message - message not found"
			end
			if connected then databasedisconnect() end
		end)
		if not res and err then
			emailrequest.errtxt = err
		end
	end

	return emailrequest
end

mymodule.get_move_message = function(self, clientdata)
	local result = {}
	result.message = cfe({ value=clientdata.message or "", label="Message" })
	result.newfolder = cfe({ value=clientdata.newfolder or "", label="New Folder" })
	result.username = cfe({ value=clientdata.username or "", label="User Name" })
	return cfe({ type="group", value=result, label="Move Message" })
end

mymodule.move_message = function(self, moverequest)
	local messages = format.string_to_table(moverequest.value.message.value, "%s*,%s*")
        local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()
		-- Check if message exists
		local sql = "SELECT * FROM voicemail_msgs" .. generatewhereclause(moverequest.value.username.value, messages)
		local mess = vmaildb.getselectresponse(sql)
		if #mess == #messages then
			-- Check if newfolder exists
			if validfolder(moverequest.value.newfolder.value) then
				for i,m in ipairs(mess) do
					local sql = "UPDATE voicemail_msgs SET in_folder='"..vmaildb.escape(moverequest.value.newfolder.value).."'" .. generatewhereclause(moverequest.value.username.value, messages)
					vmaildb.runsqlcommand(sql)
				end
				if #mess == 1 then
					moverequest.descr = "Moved message"
				else
					moverequest.descr = "Moved "..#mess.." messages"
				end
				if moverequest.value.username.value then
					voicemail_update(moverequest.value.username.value, config.domain)
				else
					for i,m in ipairs(mess) do
						voicemail_update(m.username, m.domain)
					end
				end
			else
				moverequest.errtxt = "Failed to move message - invalid folder"
			end
		else
			moverequest.errtxt = "Failed to move message - message not found"
		end
		if connected then databasedisconnect() end
	end)
	if not res and err then
		moverequest.errtxt = err
	end

	return moverequest
end

mymodule.get_callback_message = function(self, clientdata)
	local result = {}
	result.message = cfe({ value=clientdata.message or "", label="Message" })
	result.extension = cfe({ value=clientdata.extension or "", label="Extension" })
	result.username = cfe({ value=clientdata.username or "", label="User Name" })
	return cfe({ type="group", value=result, label="Callback Message" })
end

mymodule.callback_message = function(self, callbackrequest)
	if string.find(callbackrequest.value.message.value, ",") then
		callbackrequest.errtxt = "Failed to callback message - can only callback one message at a time"
	elseif callbackrequest.value.extension.value == "" or string.find(callbackrequest.value.extension.value, "[%s@]") then
		callbackrequest.errtxt = "Failed to callback message - invalid extension"
	else
        	local res, err = pcall(function()
			local connected = vmaildb.databaseconnect()
			-- Check if message exists
			local sql = "SELECT * FROM voicemail_msgs" .. generatewhereclause(callbackrequest.value.username.value, callbackrequest.value.message.value)
			local mess = vmaildb.getselectresponse(sql)
			if #mess == 1 then
				-- Mark the message as read
				voicemail_read(mess[1].username, config.domain, callbackrequest.value.message.value)
				-- Initiate the call to the extension
				voicemail_callback(callbackrequest.value.extension.value, mess[1].file_path, mess[1].username)
				callbackrequest.descr = "Initiated callback"
			else
				callbackrequest.errtxt = "Failed to callback message - message not found"
			end
			if connected then databasedisconnect() end
		end)
		if not res and err then
			callbackrequest.errtxt = err
		end
	end

	return callbackrequest
end

mymodule.list_folders = function()
	local errtxt
	local folders = {}
        local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()
		folders = listfolders()
		if connected then databasedisconnect() end
	end)
	if not res and err then
		errtxt = err
	end

	return cfe({ type="structure", value=folders, label="Voicemail Folders", errtxt=errtxt })
end

mymodule.list_passwords = function(username)
	local errtxt
	local users = {}
        local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()
		local sql = "SELECT username, password FROM voicemail_prefs"..generatewhereclause(username)
		users = vmaildb.getselectresponse(sql)
		if connected then databasedisconnect() end
	end)
	if not res and err then
		errtxt = err
	end

	return cfe({ type="structure", value=users, label="Voicemail User Passwords", errtxt=errtxt })
end

mymodule.list_users = function(self, clientdata)
	local retval = cfe({ type="group", value={}, label="Voicemail Users" })
	retval.value.page = cfe({ value=0, label="Page Number", descr="0 indicates ALL", key=true })
	retval.value.pagesize = cfe({ value=10, label="Page Size", key=true })
	retval.value.rowcount = cfe({ value=0, label="Row Count" })
	-- orderby must be an array of tables with column name and direction
	retval.value.orderby = cfe({ type="structure", value={{column="username", direction="asc"}}, label="Order By", key=true })
	-- filter is a table with a string filter for each column
	retval.value.filter = cfe({ type="structure", value={username="", firstname="", lastname=""}, label="Filter", key=true })
	self.handle_clientdata(retval, clientdata)
	retval.value.result = cfe({ type="structure", value={}, label="Voicemail Users" })

	-- Process the incoming page data
	local page = tonumber(retval.value.page.value) or 0
	retval.value.page.value = page
	local pagesize = tonumber(retval.value.pagesize.value) or 10
	retval.value.pagesize.value = pagesize
	local orderby = {}
	local columns = {username="u.username", lastname="v1.value", firstname="v2.value"}
	local directions = {asc="ASC", desc="DESC", ASC="ASC", DESC="DESC"}
	for i,o in ipairs(retval.value.orderby.value) do
		if columns[o.column] and directions[o.direction] then
			orderby[#orderby+1] = columns[o.column].." "..directions[o.direction]
		end
	end
	if #orderby == 0 then
		orderby[#orderby+1] = "username ASC"
	end

        local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()

		local filter = {}
		for c,f in pairs(retval.value.filter.value) do
			if columns[c] and f ~= "" then
				filter[#filter+1] = columns[c].."~'"..vmaildb.escape(f).."'"
			end
		end
		-- This crazy query gets the username from voicemail_users, the firstname and lastname from two instances of voicemail_values (using voicemail_params to determine the corresponding nid values) drops usernames starting with "tempuser"
		local sql = " FROM voicemail_users u LEFT OUTER JOIN voicemail_values v1 ON u.uid = v1.uid AND v1.nid=(SELECT nid FROM voicemail_params WHERE name='lastname') LEFT OUTER JOIN voicemail_values v2 ON u.uid = v2.uid AND v2.nid=(SELECT nid FROM voicemail_params WHERE name='firstname') WHERE u.username NOT LIKE 'tempuser%'"
		if #filter>0 then
			sql = sql.." AND "..table.concat(filter, " AND ")
		end
		if page > 0 then
			local count = vmaildb.getselectresponse("SELECT count(*) AS count"..sql)
			retval.value.rowcount.value = count[1].count
		end
		sql = sql.." ORDER BY "..table.concat(orderby, ", ")
		if page > 0 then
			sql = sql.." LIMIT "..pagesize.." OFFSET "..(page - 1)*pagesize
		end
		retval.value.result.value = vmaildb.getselectresponse("SELECT u.username, v1.value lastname, v2.value firstname"..sql) or {}
		if page <= 0 then
			retval.value.rowcount.value = #retval.value.result.value
		end
		if connected then databasedisconnect() end
	end)
	if not res and err then
		retval.errtxt = err
	end

	return retval
end

mymodule.get_delete_user = function(self, clientdata)
	local result = {}
	result.username = cfe({ value=clientdata.username or "", label="User Name" })
	return cfe({ type="group", value=result, label="Delete User" })
end

mymodule.set_delete_user = function(self, deleterequest)
	local result = ""
	local success = true
	if deleterequest.value.username.value == "" then
		deleterequest.value.username.errtxt = "User does not exist"
		success = false
	end
	if success then
	       	local res, err = pcall(function()
			local connected = vmaildb.databaseconnect()
			success, deleterequest.errtxt = delete_user(deleterequest.value.username.value)
			if connected then databasedisconnect() end
		end)
		if not res and err then
			deleterequest.errtxt = err
		end
	else
		deleterequest.errtxt = "Failed to delete user"
	end

	return deleterequest
end

mymodule.get_bunchsettings = function (self)
	local bunch = cfe({ name="list", label="List", type="longtext", descr="Each line will be seen as a new extension. To separate fields make use of colons. Follow the format:\n\nExtension:Name:Lastname:Password"})

	return cfe({ type="group", value={bunch=bunch}, label="Voicemail Users List" })
end

mymodule.set_bunchsettings = function (self, bunchdata)
        local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()

		local entry = {}
		entry.value = getuserparams()
		if not entry.value or not entry.value.username or not entry.value.firstname or not entry.value.lastname or not entry.value["vm-password"] then
			error("Database parameters missing", 0)
		end
		entry.value["vm-password-confirm"] = nil

		local sql = "BEGIN TRANSACTION"
		vmaildb.runsqlcommand(sql)

		for i,line in ipairs(format.string_to_table(format.dostounix(bunchdata.value.bunch.value), '\n')) do
			if string.find(line, "%S") then
				local username, firstname, lastname, password = string.match(line, "(%w+):(%w+):(%w+):(%w+)")
				if not username then
					bunchdata.value.bunch.errtxt = "Invalid syntax on line "..i
					bunchdata.errtxt = "Failed to create users"
					vmaildb.runsqlcommand("ROLLBACK")
					break
				elseif validuser(username) then
					bunchdata.value.bunch.errtxt = "Username already exists on line "..i
					bunchdata.errtxt = "Failed to create users"
					vmaildb.runsqlcommand("ROLLBACK")
					break
				else
					entry.value.username.value = username
					entry.value.firstname.value=firstname
					entry.value.lastname.value=lastname
					entry.value["vm-password"].value=password
					if validateentry(entry) then
						if dbengine == db.engine.postgresql then
							sql = "INSERT INTO voicemail_users VALUES(default, '"..vmaildb.escape(username).."')"
						else
							sql = "INSERT INTO voicemail_users VALUES(null, '"..vmaildb.escape(username).."')"
						end
						vmaildb.runsqlcommand(sql)
						sql = "SELECT uid FROM voicemail_users where username ='"..vmaildb.escape(username).."'";
						uid = vmaildb.getselectresponse(sql)
						sql = "INSERT INTO voicemail_values VALUES('"..vmaildb.escape(uid[1].uid).."', '"..vmaildb.escape(entry.value.firstname.nid).."', '"..vmaildb.escape(tostring(firstname)).."')"
						vmaildb.runsqlcommand(sql)
						sql = "INSERT INTO voicemail_values VALUES('"..vmaildb.escape(uid[1].uid).."', '"..vmaildb.escape(entry.value.lastname.nid).."', '"..vmaildb.escape(tostring(lastname)).."')"
						vmaildb.runsqlcommand(sql)

						sql = "INSERT INTO voicemail_prefs (username, domain, password) VALUES ('"..vmaildb.escape(username).."', '"..config.domain.."', '"..vmaildb.escape(password).."')"
						vmaildb.runsqlcommand(sql)
					else
						bunchdata.value.bunch.errtxt = {"Error on line "..i}
						for n,v in pairs(entry.value) do
							bunchdata.value.bunch.errtxt[#bunchdata.value.bunch.errtxt+1] = v.errtxt
						end
						bunchdata.value.bunch.errtxt = table.concat(bunchdata.value.bunch.errtxt, '\n')
						bunchdata.errtxt = "Failed to create users"
						vmaildb.runsqlcommand("ROLLBACK")
						break
					end
				end
			end
		end
		if not bunchdata.errtxt then
			vmaildb.runsqlcommand("COMMIT")
		end

		if connected then vmaildb.databasedisconnect() end
	end)
	if not res and err then
		bunchdata.errtxt = err
	end

	return bunchdata
end

mymodule.get_usersettings = function(self, clientdata)
	local retval = {}
	local errtxt
        local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()
		retval = getuserparams(clientdata.username)
		if connected then databasedisconnect() end
	end)
	if not res and err then
		errtxt = err
	end
	if retval["vm-password"] and retval["vm-password-confirm"] then retval["vm-password-confirm"].value = retval["vm-password"].value end

	return cfe({ type="group", value=retval, label="Voicemail User Settings", errtxt=errtxt })
end

mymodule.create_usersettings = function(self, usersettings, action)
	return mymodule.update_usersettings(self, usersettings, action, true)
end

mymodule.update_usersettings = function(self, usersettings, action, create)
	local errtxt
	local success = validateentry(usersettings)
	if success then
        	local res, err = pcall(function()
			local connected = vmaildb.databaseconnect()
			local u = listusers(usersettings.value.username.value)
			if create and #u > 0 then
				success = false
				errtxt = "User already exists"
			elseif not create and #u == 0 then
				success = false
				errtxt = "User does not exist"
			else
				if create then
					if dbengine == db.engine.postgresql then
						sql = "INSERT INTO voicemail_users VALUES(default, '"..vmaildb.escape(usersettings.value.username.value).."')"
					else
						sql = "INSERT INTO voicemail_users VALUES(null, '"..vmaildb.escape(usersettings.value.username.value).."')"
					end
					vmaildb.runsqlcommand(sql)
				end
				success,errtxt = setuserparams(usersettings.value)
			end
			if connected then databasedisconnect() end
		end)
		if not res and err then
			success = false
			errtxt = err
		end
	end
	if not success then
		if create then
			usersettings.errtxt = errtxt or "Failed to create user"
		else
			usersettings.errtxt = errtxt or "Failed to save settings"
		end
	end
	return usersettings
end

mymodule.process_directory_xml_request = function(input)
	local output = {}
	local errtxt
        local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()
		if validuser(input.user) then
			output = getuserparams(input.user)
			-- Add the domain
			output.domain = cfe({ value=input.domain })
		else
			errtxt = "User not found"
		end
		if connected then databasedisconnect() end
	end)
	if not res and err then
		errtxt = err
	end
	return cfe({ type="group", value=output, label="Directory Data", errtxt=errtxt })
end

mymodule.process_dialplan_xml_request = function(input)
	local output = {}
	local errtxt
        local res, err = pcall(function()
		local connected = vmaildb.databaseconnect()
		if validuser(input["Caller-Destination-Number"]) then
			output.domain = cfe({ value=config.domain })
			output.username = cfe({ value=input["Caller-Destination-Number"] })
		else
			errtxt = "User not found"
		end
		if connected then databasedisconnect() end
	end)
	if not res and err then
		errtxt = err
	end
	return cfe({ type="group", value=output, label="Dialplan Data", errtxt=errtxt })
end

return mymodule
