#! /usr/bin/env python

import argparse
from pmx.model import Model, double_box
from pmx.scripts.cli import check_unknown_cmd


def parse_options():
    parser = argparse.ArgumentParser(description='''Places two structures
into a single box. The box is a rectangular cuboid in which the two structures
are placed in such a way as to minimise the box volume. You can use this script
to help in the setup of a calculation using the single-box double-system approach.
''')
    parser.add_argument('-f1',
                        metavar='',
                        dest='file1',
                        type=str,
                        help='First structure in PDB or GRO format.',
                        required=True)
    parser.add_argument('-f2',
                        metavar='',
                        dest='file2',
                        type=str,
                        help='Second structure in PDB or GRO format.',
                        required=True)
    parser.add_argument('-o',
                        metavar='',
                        dest='outfname',
                        type=str,
                        help='Name of output file. Default is "doublebox.pdb".',
                        default='doublebox.pdb')
    parser.add_argument('-r',
                        metavar='',
                        dest='r',
                        type=float,
                        help='Distance between the two structures (nm). '
                        'Default is 2.5 nm.',
                        default=2.5)
    parser.add_argument('-d',
                        metavar='',
                        dest='d',
                        type=float,
                        help='Distance to the box wall (nm). '
                        'Default is 1.5 nm.',
                        default=1.5)
    parser.add_argument('--longest_axis',
                        dest='longest_axis',
                        help='Whether to just place structures along the '
                        'longest axis, rather then minimising the volume. '
                        'Default is False.',
                        default=False,
                        action='store_true')

    args, unknown = parser.parse_known_args()
    check_unknown_cmd(unknown)

    return args


def main(args):

    m1 = Model(args.file1, bPDBTER=True, renumber_residues=False)
    m2 = Model(args.file2, bPDBTER=True, renumber_residues=False)

    mout = double_box(m1=m1, m2=m2, r=args.r, d=args.d,
                      bLongestAxis=args.longest_axis, verbose=True)
    mout.write(args.outfname, bPDBTER=True)


def entry_point():
    args = parse_options()
    main(args)


if __name__ == '__main__':
    entry_point()
