import logging
import subprocess
import shutil
from pathlib import Path
from typing import List, Dict, Tuple
from . import fasta_processor

# Initialize logger once at module level
log = logging.getLogger(__name__)


def perform_self_blast(
    input_files: List[Path], output_dir: Path, evalue: float, min_pident: float
) -> Tuple[Path, Dict[str, int]]:
    """
    Orchestrates the all-vs-all BLAST search.

    Args:
        input_files (List[Path]): List of input FASTA files.
        output_dir (Path): Directory to store results and temp files.
        evalue (float): E-value cutoff.
        min_pident (float): Minimum percent identity.

    Returns:
        Tuple[Path, Dict[str, int]]:
            1. Path to the generated BLAST results file (tab-delimited).
            2. A dictionary mapping original headers to internal IDs (if fasta_processor does this).

    Raises:
        FileNotFoundError: If BLAST+ tools are missing.
        subprocess.CalledProcessError: If BLAST commands fail.
    """

    # --- 1. Check for BLAST+ dependencies ---
    if shutil.which("makeblastdb") is None or shutil.which("blastn") is None:
        log.error("BLAST+ binaries not found in system PATH.")
        raise FileNotFoundError(
            "'makeblastdb' or 'blastn' not found. "
            "Please install NCBI BLAST+ and ensure it's in your PATH."
        )

    # --- 2. Process FASTAs and create header map ---
    combined_fasta_file = output_dir / "combined_query.fasta"
    try:
        # Ensure fasta_processor is implemented to return the map
        header_map = fasta_processor.process_fastas(input_files, combined_fasta_file)
    except Exception as e:
        log.error(f"Failed during FASTA processing: {e}")
        raise

    # --- 3. Create BLAST Database ---
    db_name = output_dir / "blast_db"
    try:
        log.info("Creating BLAST database...")
        _run_makeblastdb(combined_fasta_file, db_name)
        log.info("BLAST database created successfully.")
    except subprocess.CalledProcessError as e:
        log.error(f"makeblastdb failed:\n{e.stderr}")
        raise

    # --- 4. Run blastn ---
    blast_result_file = output_dir / "blast_hits.txt"
    try:
        log.info("Running blastn all-vs-all search...")
        _run_blastn(combined_fasta_file, db_name, blast_result_file, evalue, min_pident)
        log.info(f"BLAST search complete. Results: {blast_result_file.name}")
    except subprocess.CalledProcessError as e:
        log.error(f"blastn failed:\n{e.stderr}")
        raise

    # --- 5. Cleanup Temporary Files ---
    log.info("Cleaning up temporary BLAST files...")
    try:
        if combined_fasta_file.exists():
            combined_fasta_file.unlink()

        # Clean up the various BLAST DB files (.nhr, .nin, .nsq, etc.)
        for db_file in output_dir.glob(f"{db_name.name}.*"):
            db_file.unlink()

        log.info("Cleanup successful.")
    except OSError as e:
        log.warning(f"Could not delete temporary files: {e}")

    return blast_result_file, header_map


def _run_makeblastdb(fasta_file: Path, db_name: Path) -> None:
    """
    Runs makeblastdb to create a nucleotide database.

    NOTE: We force -blastdb_version 4 because Version 5 uses LMDB,
    which often crashes on network drives (SMB/NFS) or shared volumes
    with 'mdb_env_open: Operation not supported'.
    """
    cmd = [
        "makeblastdb",
        "-in",
        str(fasta_file),
        "-dbtype",
        "nucl",
        "-out",
        str(db_name),
        "-blastdb_version",
        "4",
    ]
    subprocess.run(cmd, check=True, capture_output=True, text=True)


def _run_blastn(
    query_file: Path, db_name: Path, output_file: Path, evalue: float, min_pident: float
) -> None:
    """
    Runs blastn with output format 6 (tabular).

    Flags used:
      -ungapped: Perform ungapped alignment (often better for synteny/exact matches).
      -outfmt 6: Tabular (qseqid sseqid pident length mismatch gapopen qstart qend sstart send evalue bitscore)
    """
    cmd = [
        "blastn",
        "-query",
        str(query_file),
        "-db",
        str(db_name),
        "-out",
        str(output_file),
        "-outfmt",
        "6",
        "-evalue",
        str(evalue),
        "-perc_identity",
        str(min_pident),
        "-ungapped",
    ]
    subprocess.run(cmd, check=True, capture_output=True, text=True)
