import argparse
import logging
import sys
from importlib.metadata import version, PackageNotFoundError
from pathlib import Path
from . import blast_runner
from .parse_blast_and_shatter import parse_blast_and_shatter
from .trim_blast import trim_blast
from .grouping import hit_grouping
from .visualize_groups import visualize_groups
from .dotplot import generate_dotplots
from .summary import generate_summary_plots


def parse_arguments():
    """Parses command-line arguments using argparse."""
    parser = argparse.ArgumentParser(
        description="DaisyBlast: A tool to find and visualize synteny blocks from a single multi-FASTA file."
    )

    try:
        # This fetches the version from installed package metadata
        tool_version = version("DaisyBlast")
    except PackageNotFoundError:
        # Fallback if the tool isn't installed via pip yet
        tool_version = "local-dev"

    parser.add_argument(
        "-i",
        "--input",
        type=Path,
        required=True,
        nargs="+",
        help="One or more input FASTA files (e.g., contig1.fa contig2.fa).",
    )

    parser.add_argument(
        "-o",
        "--output_dir",
        type=Path,
        default=Path("daisyblast_results"),
        help="Directory to save output .bed and .png files. (Default: daisyblast_results)",
    )

    parser.add_argument(
        "-e",
        "--evalue",
        type=float,
        default=1e-10,
        help="E-value cutoff for the self-BLAST search. (Default: 1e-10)",
    )

    parser.add_argument(
        "--min_pident",
        type=float,
        default=90.0,
        help="Minimum percent identity for a BLAST hit. (Default: 90.0)",
    )

    parser.add_argument(
        "--min_length",
        type=int,
        default=500,
        help="Minimum alignment length *after* splitting hits. (Default: 500)",
    )

    parser.add_argument(
        "-n",
        "--num_groups",
        type=int,
        default=20,
        help="Maximum number of groups in final bedfile (Default: 20)",
    )

    parser.add_argument(
        "-v",
        "--version",
        action="version",
        version=f"%(prog)s {tool_version}",
        help="Show the version number and exit.",
    )

    return parser.parse_args()


def main():
    """Main entry point for the DaisyBlast application."""
    args = parse_arguments()

    # --- 1. Setup Logging ---
    logging.basicConfig(
        level=logging.INFO,
        format="%(asctime)s [%(levelname)-5.5s]  %(message)s",
        handlers=[logging.StreamHandler(sys.stdout)],
    )

    log = logging.getLogger(__name__)
    log.info(f"Starting DaisyBlast analysis for {len(args.input)} input file(s).")

    try:
        args.output_dir.mkdir(parents=True, exist_ok=True)
        log.info(f"Output directory set to: {args.output_dir.resolve()}")
    except OSError as e:
        log.error(f"Failed to create output directory {args.output_dir}: {e}")
        sys.exit(1)

    # --- 2. Run Self-BLAST ---
    blast_result_file, header_map = blast_runner.perform_self_blast(
        args.input, args.output_dir, args.evalue, args.min_pident
    )

    # --- 3. Parse BLAST Results ---
    log.info(f"Parsing BLAST results from: {blast_result_file.name}")

    shattered_file = parse_blast_and_shatter(
        blast_file=blast_result_file,
        outdir=args.output_dir,
        min_len=args.min_length,
        identity_cutoff=args.min_pident,
    )

    # --- 4. Resolve Hits & Generate Files ---
    log.info("Resolving hits, finding matching groups, and generating BED file...")
    trimmed_file = trim_blast(blast_result_file, shattered_file, args.output_dir)

    # --- 4. Chain hits into Groups ---
    log.info("Reciprocally grouping hits...")
    grouped_file = hit_grouping(
        shattered_file, trimmed_file, args.output_dir, args.num_groups
    )

    # --- 5. Generate Plots ---
    log.info("Generating linear plots for contigs...")
    visualize_groups(grouped_file, args.output_dir)

    # --- 6. Generate Dotplots (NEW) ---
    # We use the raw blast_result_file here because dotplots
    # are best for seeing the raw alignment structure before we shatter it.
    generate_dotplots(blast_result_file, args.output_dir)

    # --- 7. Generate Summary Plots ---
    # Creates the NCBI-style stacked hit bars
    generate_summary_plots(blast_result_file, args.output_dir)

    log.info("DaisyBlast analysis complete.")


if __name__ == "__main__":
    main()
