import logging
from pathlib import Path
from typing import List, Dict
from Bio import SeqIO

log = logging.getLogger(__name__)


def process_fastas(
    input_files: List[Path], combined_fasta_file: Path
) -> Dict[str, int]:
    """
    Combines multiple FASTA files into one.

    Critically, it renames headers to be unique, e.g.,
    '>contig1' in 'file1.fa' becomes '>file1.fa___contig1'

    Returns a dictionary mapping the new unique ID to its sequence length.
    This map is essential for the plotting step.
    """
    log.info(f"Combining {len(input_files)} FASTA file(s) and standardizing headers...")
    header_map = {}  # {new_id: length}

    with open(combined_fasta_file, "w") as out_f:
        for fasta_file in input_files:
            try:
                for record in SeqIO.parse(fasta_file, "fasta"):
                    original_id = record.id

                    # Create a new, unique ID based on filename and original ID
                    new_id = f"{fasta_file.name}___{original_id}"

                    if new_id in header_map:
                        # This case is unlikely but important to handle
                        log.warning(
                            f"Duplicate header generated: {new_id}. "
                            "Ensure contig names are unique within each file."
                        )

                    # Store the length for plotting
                    header_map[new_id] = len(record.seq)

                    # Update the record for the new combined FASTA
                    record.id = new_id
                    record.description = ""  # Keep headers clean
                    SeqIO.write(record, out_f, "fasta")

            except FileNotFoundError:
                log.error(f"Input file not found: {fasta_file}")
                raise
            except Exception as e:
                log.error(f"Error parsing FASTA file {fasta_file}: {e}")
                raise

    log.info(f"Combined {len(header_map)} sequences into {combined_fasta_file.name}.")
    return header_map
