#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of candidate fusion genes. This list is hard coded
in here and it is manually curated from:


B. Alaei-Mahabadia et al., Global analysis of somatic structural genomic 
alterations and their impact on gene expression in diverse human cancers, PNAS,
Nov. 2016, http://dx.doi.org/10.1073/pnas.1606220113



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of pre-candidate fusion genes from TCGA project."""
    version = "%prog 0.12 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of allowed candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of allowed candidate fusion genes is generated. Default is '%default'.""")

    parser.add_option("--skip-filter-overlap",
                      action="store_true",
                      dest="skip_filter_overlap",
                      default = False,
                      help="""If set then it filters out the known fusion genes where the (i) genes are fully overlapping, or (ii) the genes are partially overlapping and are on the same strand. Default is '%default'.""")

    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    print "Generating the list of TCGA fusion genes..."
    fusions = dict()

    # manual curation from papers

    fusions['rattus_norvegicus'] = []

    fusions['mus_musculus'] = []

    fusions['canis_lupus_familiaris'] = []

    fusions['homo_sapiens'] = [
        ['A2M','CSRP1'],
        ['A2M','FXYD5'],
        ['A2ML1','IGKC'],
        ['AAK1','GALNT14'],
        ['AAK1','KCMF1'],
        ['AARD','CD24'],
        ['AARS','COG4'],
        ['AARS','ST3GAL2'],
        ['AATF','SNAP25-AS1'],
        ['ABAT','C16ORF62'],
        ['ABCA2','SLC4A1'],
        ['ABCA7','COL5A3'],
        ['ABCA9','CEP112'],
        ['ABCA9','TANC2'],
        ['ABCB9','EIF2B1'],
        ['ABCC4','ABCB4'],
        ['ABCC4','KLK3'],
        ['ABCC9','RASSF8'],
        ['ABCG1','CCDC134'],
        ['ABCG1','MEI1'],
        ['ABHD17B','TMEM2'],
        ['ABHD2','PDLIM5'],
        ['ABI2','CYP20A1'],
        ['ABR','IGHG1'],
        ['ABR','STAP2'],
        ['ABTB2','TYRL'],
        ['AC002116.7','WDR62'],
        ['AC004019.10','TERF2IP'],
        ['AC004066.3','PPA2'],
        ['AC004549.6','HIBADH'],
        ['AC004967.7','OR7E38P'],
        ['AC005013.5','CTB-113D17.1'],
        ['AC005077.12','LINC01002'],
        ['AC005534.8','LINC01006'],
        ['AC005618.6','PCDHGA2'],
        ['AC005775.2','ODF3L2'],
        ['AC005884.1','RPL23AP64'],
        ['AC006129.1','LINC01480'],
        ['AC007405.4','AC007277.3'],
        ['AC007950.2','HERC1'],
        ['AC008271.1','DDX1'],
        ['AC008440.5','2-Sep'],
        ['AC011523.2','KLK3'],
        ['AC013275.2','CFAP221'],
        ['AC015849.16','CCL5'],
        ['AC015977.6','CIB4'],
        ['AC016700.3','PCBP1-AS1'],
        ['AC019349.5','KRT14'],
        ['AC019349.5','KRT15'],
        ['AC019349.5','KRT16'],
        ['AC019349.5','KRT5'],
        ['AC019349.5','KRT6A'],
        ['AC019349.5','KRT7'],
        ['AC019349.5','PLEC'],
        ['AC024084.1','AC009784.3'],
        ['AC027119.1','AC026167.1'],
        ['AC064834.1','TAPBP'],
        ['AC078842.3','PTN'],
        ['AC090283.3','CCR7'],
        ['AC092415.1','RPA3OS'],
        ['AC092635.1','AC016730.1'],
        ['AC098828.2','LAPTM4A'],
        ['AC099535.4','UBA52'],
        ['AC138430.4','FAM129A'],
        ['ACACA','ACPP'],
        ['ACACA','RBM3'],
        ['ACAD11','SOD2'],
        ['ACAP2','ATP13A3'],
        ['ACE2','PIR'],
        ['ACER3','NARS2'],
        ['ACER3','PMEL'],
        ['ACER3','UVRAG'],
        ['ACIN1','HAUS4'],
        ['ACKR2','POMGNT2'],
        ['ACOT9','APOO'],
        ['ACPP','A2M'],
        ['ACPP','ADD1'],
        ['ACPP','AHNAK'],
        ['ACPP','ALOX15B'],
        ['ACPP','AZGP1'],
        ['ACPP','BCAM'],
        ['ACPP','BMPR1B'],
        ['ACPP','CLSTN1'],
        ['ACPP','CNNM2'],
        ['ACPP','COPA'],
        ['ACPP','EIF3A'],
        ['ACPP','FAM129B'],
        ['ACPP','FBLN1'],
        ['ACPP','GCNT2'],
        ['ACPP','GLUD1'],
        ['ACPP','KIAA1324'],
        ['ACPP','KLK3'],
        ['ACPP','MYH11'],
        ['ACPP','MYH9'],
        ['ACPP','NCL'],
        ['ACPP','NUCB1'],
        ['ACPP','PLEKHB2'],
        ['ACPP','PPP3CA'],
        ['ACPP','PTRF'],
        ['ACPP','RPL30'],
        ['ACPP','SAFB2'],
        ['ACPP','SASH1'],
        ['ACPP','SERINC5'],
        ['ACPP','SGK1'],
        ['ACPP','SLC4A4'],
        ['ACPP','SMIM14'],
        ['ACPP','SRSF5'],
        ['ACPP','TBC1D16'],
        ['ACPP','TGM4'],
        ['ACPP','TPD52'],
        ['ACPP','TSC22D1'],
        ['ACPP','UBC'],
        ['ACSF2','SPAG9'],
        ['ACSF3','MYH7B'],
        ['ACSL3','SGPP2'],
        ['ACSS1','APMAP'],
        ['ACSS1','KRT18P2'],
        ['ACTB','ALG8'],
        ['ACTB','CHI3L1'],
        ['ACTB','COPA'],
        ['ACTB','GFAP'],
        ['ACTB','IGHG1'],
        ['ACTB','KRT14'],
        ['ACTB','PABPC1'],
        ['ACTB','PARP9'],
        ['ACTB','S100A11'],
        ['ACTB','SERPINF1'],
        ['ACTB','TAPBP'],
        ['ACTB','TNFSF10'],
        ['ACTG1','CCT6A'],
        ['ACTL6A','NDUFB5'],
        ['ACTN1','MYH11'],
        ['ACTN1','TPM4'],
        ['ACTN2','CHRM3'],
        ['ACTN4','CAPZB'],
        ['ACTN4','IGFBP5'],
        ['ACTN4','KRT15'],
        ['ACTN4','NIPAL2'],
        ['ACTR2','TSGA10'],
        ['ACTR3','ACTR3P2'],
        ['ACTR3','LINC01191'],
        ['ACTR3P2','ACTR3'],
        ['ACVR2A','CASP8'],
        ['ACVR2A','SUMO1'],
        ['ADAM17','NHEJ1'],
        ['ADAM23','AC010731.4'],
        ['ADAM9','ADAM32'],
        ['ADAM9','CTA-293F17.1'],
        ['ADAM9','HTRA4'],
        ['ADAM9','RNF170'],
        ['ADAMTS12','LIFR-AS1'],
        ['ADAMTS17','LPAR1'],
        ['ADAMTS7','PEAK1'],
        ['ADAMTS9-AS2','PTPRG'],
        ['ADAMTS9','ULK4'],
        ['ADAP2','RNF135'],
        ['ADCK2','DPP6'],
        ['ADCK4','AKAP8'],
        ['ADCY3','PTRHD1'],
        ['ADCYAP1R1','BCAN'],
        ['ADD1','FAM193A'],
        ['ADD3','COL3A1'],
        ['ADI1','AC108488.4'],
        ['ADIPOR2','FOXJ2'],
        ['ADIPOR2','SLC2A13'],
        ['ADK','LRRTM3'],
        ['ADNP','DPM1'],
        ['ADRBK2','AP1B1'],
        ['ADSS','BPNT1'],
        ['AEBP1','AC008753.6'],
        ['AEBP1','MGP'],
        ['AEBP1','MYL9'],
        ['AF064860.7','IGSF5'],
        ['AFAP1','ABLIM2'],
        ['AFF4','AC004041.2'],
        ['AFF4','FN1'],
        ['AFG3L1P','TUBB3'],
        ['AFG3L2','ETV4'],
        ['AFMID','TNRC6C'],
        ['AGAP1','AC079135.1'],
        ['AGAP3','JAZF1-AS1'],
        ['AGAP3','RELN'],
        ['AGO2','MROH5'],
        ['AGO2','PTK2'],
        ['AGPAT6','LY6H'],
        ['AHCTF1','SMYD3'],
        ['AHCYL2','SFTPB'],
        ['AHDC1','COL1A1'],
        ['AHNAK2','IGKC'],
        ['AHNAK2','KRT14'],
        ['AHNAK','CTSB'],
        ['AHRR','AVIL'],
        ['AHSA1','TG'],
        ['AKAP1','ANKFN1'],
        ['AKAP1','C17ORF67'],
        ['AKAP1','GGA3'],
        ['AKAP9','PON2'],
        ['AKR1B10','CALU'],
        ['AKT1','KLK3'],
        ['AL121578.2','SYTL5'],
        ['ALCAM','EIF2A'],
        ['ALDH3A1','ULK2'],
        ['ALDH3B1','SFTPB'],
        ['ALDOA','IGKC'],
        ['ALDOA','SCIN'],
        ['ALG14','C1ORF194'],
        ['ALG14','CNN3'],
        ['ALG1L2','ALG1L6P'],
        ['ALG8','NDUFC2'],
        ['ALG8','USP32'],
        ['ALOX12','RNASEK-C17ORF49'],
        ['AMACR','SLC45A2'],
        ['AMBRA1','HARBI1'],
        ['AMD1','IGKC'],
        ['AMMECR1L','SAP130'],
        ['AMOTL1','C11ORF97'],
        ['ANAPC15','LAMTOR1'],
        ['ANAPC5','KDM2B'],
        ['ANKAR','OSGEPL1-AS1'],
        ['ANKHD1','CYSTM1'],
        ['ANKH','DNAH5'],
        ['ANKRD11','CHMP1A'],
        ['ANKRD11','CPNE7'],
        ['ANKRD11','LINGO1'],
        ['ANKRD44','LINC01473'],
        ['ANKRD46','TMEM222'],
        ['ANO1','COL1A1'],
        ['ANO7','2-Sep'],
        ['ANP32B','APP'],
        ['ANXA1','ENO1'],
        ['AP000230.1','AP001596.6'],
        ['AP001462.6','CYP4F22'],
        ['AP1B1','ADRBK2'],
        ['AP1M1','CIB3'],
        ['AP2A1','COL3A1'],
        ['AP2A2','TALDO1'],
        ['AP2A2','VPS26A'],
        ['AP2B1','PIGR'],
        ['AP2B1','PPM1E'],
        ['AP2M1','CALR'],
        ['AP2M1','PHF2'],
        ['APBA2','CHD4'],
        ['APBA2','GABRB3'],
        ['APBB2','TG'],
        ['APC','KIAA0825'],
        ['APC','NREP-AS1'],
        ['APH1A','GNB2L1'],
        ['APH1A','PMEL'],
        ['APH1A','PTPN9'],
        ['APH1A','RGS4'],
        ['APLP2','CANX'],
        ['APLP2','CTSB'],
        ['APLP2','GPNMB'],
        ['APLP2','NCL'],
        ['APLP2','PIGR'],
        ['APLP2','TMEM213'],
        ['APOD','9-Sep'],
        ['APOD','CABLES1'],
        ['APOD','CDCP1'],
        ['APOD','ERBB2'],
        ['APOD','LTF'],
        ['APOD','PMEL'],
        ['APOD','PRKCSH'],
        ['APOD','TIMP3'],
        ['APOE','NME4'],
        ['APOL1','CD74'],
        ['APOL1','KIAA0020'],
        ['APOL1','PKM'],
        ['APOL1','SOD2'],
        ['APOLD1','CDKN1B'],
        ['APOPT1','MARK3'],
        ['APP','ABHD2'],
        ['APP','ANP32B'],
        ['APPBP2','CA10'],
        ['APP','C7'],
        ['APP','CANX'],
        ['APP','COL5A2'],
        ['APP','DDX17'],
        ['APP','IGHG1'],
        ['APPL2','DDHD2'],
        ['APPL2','TUBA1C'],
        ['APP','PTMS'],
        ['APP','SPARC'],
        ['APP','TMEM213'],
        ['APTX','GNE'],
        ['AQP3','IGF2'],
        ['ARAP1','IGF2'],
        ['ARF1','ACPP'],
        ['ARFGAP2','CELF1'],
        ['ARFGEF2','CNIH2'],
        ['ARHGAP10','EDNRA'],
        ['ARHGAP11A','SCG5'],
        ['ARHGAP15','KYNU'],
        ['ARHGAP1','MGP'],
        ['ARHGAP21','ENKUR'],
        ['ARHGAP4','TBC1D30'],
        ['ARHGAP8','LINC00511'],
        ['ARHGDIB','OVCH1-AS1'],
        ['ARHGEF11','CD48'],
        ['ARHGEF12','IGKC'],
        ['ARHGEF17','COA4'],
        ['ARHGEF19','ANO7P1'],
        ['ARHGEF1','PMEL'],
        ['ARHGEF1','RPS19'],
        ['ARHGEF26','LINC00880'],
        ['ARHGEF2','SSR2'],
        ['ARHGEF3','ACP6'],
        ['ARHGEF4','KRT13'],
        ['ARHGEF7','EXOC4'],
        ['ARHGEF7','TEX29'],
        ['ARID1A','DHDDS'],
        ['ARID1A','RNF31'],
        ['ARID1A','RPL4'],
        ['ARID1B','NKAIN2'],
        ['ARID1B','TIAM2'],
        ['ARID2','SPSB2'],
        ['ARIH2','CAMP'],
        ['ARIH2','DHX30'],
        ['ARIH2','TRIO'],
        ['ARL17A','KANSL1'],
        ['ARL1','TG'],
        ['ARL5B','NSUN6'],
        ['ARMC12','ERGIC3'],
        ['ARMC6','SLC25A42'],
        ['ARMC9','AC017104.6'],
        ['ARMC9','B3GNT7'],
        ['ARMCX3-AS1','ARMCX6'],
        ['ARNT','F5'],
        ['ARNTL','RIC8B'],
        ['ARRB1','GAB2'],
        ['ARRB1','UVRAG'],
        ['ARSA','PMEL'],
        ['ART3','NUP54'],
        ['ART4','SMCO3'],
        ['ASAP3','TCEA3'],
        ['ASB4','GS1-306C12.1'],
        ['ASB4','RAI14'],
        ['ASCC1','TG'],
        ['ASH1L','ARHGEF2'],
        ['ASH2L','CASS4'],
        ['ASLP1','AP000347.4'],
        ['ASMTL','UGT1A6'],
        ['ASNSD1','ANKAR'],
        ['ASPH','GFAP'],
        ['ASPH','KRT5'],
        ['ASPH','VIM'],
        ['ASPM','PIGC'],
        ['ASRGL1','CPSF7'],
        ['ASRGL1','FLRT1'],
        ['ASTN2','ARHGDIG'],
        ['ASTN2','GFAP'],
        ['ASXL1','TM9SF4'],
        ['ATAD2','NCL'],
        ['ATAD2','WDYHV1'],
        ['ATF6','GJA5'],
        ['ATF6','SH2D1B'],
        ['ATF7','COPZ1'],
        ['ATF7','TG'],
        ['ATG101','CSRNP2'],
        ['ATL2','TG'],
        ['ATN1','APOD'],
        ['ATN1','CD27-AS1'],
        ['ATP11A','GAS6'],
        ['ATP11A','LAMP1'],
        ['ATP11A','SFTPB'],
        ['ATP11C','XXYAC-YR29IB3.1'],
        ['ATP13A1','TG'],
        ['ATP1A1','CTSB'],
        ['ATP1A2','GPRC5B'],
        ['ATP1A2','NDRG2'],
        ['ATP1B1','CTSB'],
        ['ATP1B3','RNF7'],
        ['ATP2A2','SFTPB'],
        ['ATP2B1','CCDC38'],
        ['ATP2B1','DCDC2'],
        ['ATP2B4','ERBB4'],
        ['ATP2B4','NFASC'],
        ['ATP5B','ORMDL2'],
        ['ATP5E','CTSZ'],
        ['ATP6V0A4','ATP6V0D2'],
        ['ATP6V0D2','ALDOA'],
        ['ATP6V0D2','ATP6V0A4'],
        ['ATP6V0D2','GLTP'],
        ['ATP6V0D2','TBC1D14'],
        ['ATP6V0D2','TMEM213'],
        ['ATP6V0D2','ZNF664'],
        ['ATP6V1A','GPNMB'],
        ['ATP6V1E1','TG'],
        ['ATP6V1G2-DDX39B','IGKC'],
        ['ATP9A','BCAN'],
        ['ATP9A','CTSB'],
        ['ATP9A','SLC34A2'],
        ['ATP9B','LUC7L'],
        ['ATPIF1','PMEL'],
        ['ATXN10','KRTAP9-1'],
        ['ATXN1','DNM3'],
        ['ATXN1','SLC24A1'],
        ['AXIN1','ITFG3'],
        ['AZIN1','TG'],
        ['B2M','AES'],
        ['B2M','COL1A2'],
        ['B2M','ITGB5'],
        ['B2M','SH3BP2'],
        ['B3GALT5','IGSF5'],
        ['B3GALTL','TEX26'],
        ['B3GNT5','IGF2'],
        ['B4GALNT1','RPS6P22'],
        ['B4GALNT1','TSFM'],
        ['B4GALT1','SFTPB'],
        ['B4GALT3','ERO1LB'],
        ['B4GALT3','NDUFS2'],
        ['BAALC-AS1','BAALCOS'],
        ['BACE2','FAM3B'],
        ['BAG4','NEDD8'],
        ['BAHCC1','CD44'],
        ['BAIAP2L2','SLC16A8'],
        ['BAIAP2','TG'],
        ['BAZ1B','PIGR'],
        ['BAZ1B','SERPINE2'],
        ['BBOX1-AS1','SLC5A12'],
        ['BCAM','KLK4'],
        ['BCAN','FYN'],
        ['BCAN','NDRG2'],
        ['BCAP29','SLC26A4'],
        ['BCAP31','PDZD4'],
        ['BCAS3','CASC3'],
        ['BCAS3','GH2'],
        ['BCAS3','GIT1'],
        ['BCAS3','RNF169'],
        ['BCAS3','SHANK2'],
        ['BCAS3','VMP1'],
        ['BCKDHA','TG'],
        ['BCL11B','MIPOL1'],
        ['BCL9','BEST3'],
        ['BCL9','NAV3'],
        ['BCR','MRVI1'],
        ['BET1L','COL1A2'],
        ['BGN','PTMS'],
        ['BICC1','LINC00844'],
        ['BICD1','PMEL'],
        ['BICD2','IPPK'],
        ['BIRC2','TMEM123'],
        ['BIRC6','TTC27'],
        ['BLACE','C1ORF186'],
        ['BLOC1S1','CPM'],
        ['BMPR1A','TMPO'],
        ['BMPR2','4-Mar'],
        ['BOLA2B','SMG1'],
        ['BOLA2B','SMG1P1'],
        ['BOLA2B','SMG1P3'],
        ['BOLA2B','SMG1P4'],
        ['BOLA2B','SMG1P5'],
        ['BOLA2B','SMG1P6'],
        ['BPIFA1','GAPDH'],
        ['BPIFB1','OR2H1'],
        ['BPIFB1','SFTPB'],
        ['BRAF','AP3B1'],
        ['BRAF','DENND2A'],
        ['BRAF','MRPS33'],
        ['BRD1','SLC3A2'],
        ['BRD8','IGF2'],
        ['BRDT','EPHX4'],
        ['BRE','GPN1'],
        ['BRI3BP','KRT5'],
        ['BRMS1','IGF2'],
        ['BROX','SUSD4'],
        ['BSG','PIGR'],
        ['BSG','ZBTB44'],
        ['BTBD11','CMKLR1'],
        ['BTBD2','KIAA1109'],
        ['BTD','BCAS3'],
        ['BZW2','CLCN3'],
        ['C11ORF70','YAP1'],
        ['C11ORF80','KDM2A'],
        ['C11ORF84','RTN3'],
        ['C12ORF55','SLC5A8'],
        ['C12ORF5','CRP'],
        ['C12ORF5','TSPAN9'],
        ['C12ORF65','EIF4EBP1'],
        ['C12ORF66','CPM'],
        ['C14ORF37','COL1A1'],
        ['C17ORF107','GP1BA'],
        ['C17ORF75','TOP2A'],
        ['C17ORF99','SYNGR2'],
        ['C19ORF25','MED1'],
        ['C19ORF47','AKT2'],
        ['C19ORF47','PTMS'],
        ['C1GALT1','AC005532.5'],
        ['C1ORF186','LINC01320'],
        ['C1ORF21','INTS7'],
        ['C1ORF43','CTSB'],
        ['C1ORF43','ERBB2'],
        ['C1ORF61','CCT3'],
        ['C1QTNF6','RAC2'],
        ['C1QTNF9B','MIPEP'],
        ['C1S','CTDSP2'],
        ['C1S','SFTPB'],
        ['C20ORF197','MIR646HG'],
        ['C21ORF91','COLGALT2'],
        ['C2CD2','AP000695.4'],
        ['C2CD3','P2RY6'],
        ['C2CD5','ITPR2'],
        ['C2ORF42','TGFA'],
        ['C3ORF55','PTX3'],
        ['C5ORF46','SPINK1'],
        ['C6ORF106','SPDEF'],
        ['C6ORF106','TG'],
        ['C6ORF132','KRT5'],
        ['C6ORF183','CCDC162P'],
        ['C6ORF62','FAM65B'],
        ['C6ORF62','KIAA0319'],
        ['C6ORF62','MGP'],
        ['C7','CANX'],
        ['C7ORF50','CD177'],
        ['C9ORF3','VCAN'],
        ['C9ORF41','C9ORF40'],
        ['C9ORF85','FXN'],
        ['CABLES1','GNAS'],
        ['CACHD1','CLIC4'],
        ['CACNA1C','EEA1'],
        ['CACNA1C','SLC6A15'],
        ['CACNA1C','TSPAN8'],
        ['CACNA2D2','THRSP'],
        ['CACNB1','MLLT6'],
        ['CACNG4','CACNG1'],
        ['CADM1','GAS7'],
        ['CADM2','ROBO1'],
        ['CADM4','ZNF428'],
        ['CALCOCO2','TG'],
        ['CALD1','SLC41A1'],
        ['CALM1','IGF2'],
        ['CALM3','CHRNA4'],
        ['CALR','MYH11'],
        ['CAMKK2','DIP2B'],
        ['CAMKK2','PMCH'],
        ['CAMSAP1','LCN8'],
        ['CAMSAP2','BCAR3'],
        ['CANX','COL4A2'],
        ['CANX','ERBB2'],
        ['CANX','KDELR2'],
        ['CANX','RUFY1'],
        ['CANX','TIMP3'],
        ['CANX','TMEFF2'],
        ['CAP1','PPIE'],
        ['CAPN1','TG'],
        ['CAPN8','HHAT'],
        ['CAPRIN1','CAT'],
        ['CAPZB','APITD1-CORT'],
        ['CARD8','POLR3H'],
        ['CARD8','SULT2B1'],
        ['CARD8','TG'],
        ['CASC3','HNF1B'],
        ['CASC3','IGKC'],
        ['CASC4','TMEM213'],
        ['CASD1','VWC2'],
        ['CASKIN2','MYH11'],
        ['CASP8','AC017104.2'],
        ['CAV2','TG'],
        ['CBL','CLNS1A'],
        ['CBR4','GALNTL6'],
        ['CBX1','HOXB8'],
        ['CBX1','SKAP1'],
        ['CBY1','GTPBP1'],
        ['CC2D1A','RNF207'],
        ['CCAT1','CASC19'],
        ['CCDC117','CTA-292E10.6'],
        ['CCDC124','PCBP1-AS1'],
        ['CCDC127','MUCL1'],
        ['CCDC142','TTC31'],
        ['CCDC152','TG'],
        ['CCDC15','SLC37A2'],
        ['CCDC26','LINC00977'],
        ['CCDC64B','HCFC1R1'],
        ['CCDC6','LINC00948'],
        ['CCDC6','RET'],
        ['CCDC80','LINC01279'],
        ['CCDC88B','RELT'],
        ['CCDC88B','TEX40'],
        ['CCND2','CASC1'],
        ['CCNI','CTB-179I1.2'],
        ['CCNY','DENND1A'],
        ['CCT2','KCNC2'],
        ['CCT2','OR8S1'],
        ['CD24','COL4A1'],
        ['CD2AP','MGAT4C'],
        ['CD302','ALPL'],
        ['CD36','COL1A1'],
        ['CD44','CASC11'],
        ['CD44','COPA'],
        ['CD44','HSP90B1'],
        ['CD55','CENPL'],
        ['CD55','PLD5'],
        ['CD55','RGS21'],
        ['CD55','UBA52'],
        ['CD59','CTSB'],
        ['CD59','FBXO3'],
        ['CD59','MYH11'],
        ['CD74','COL5A1'],
        ['CD74','FAM129B'],
        ['CD74','GAPDH'],
        ['CD74','LAMP3'],
        ['CD74','ROS1'],
        ['CD81-AS1','CARS'],
        ['CD82','TAPBP'],
        ['CDC27','BRAF'],
        ['CDC42BPA','EPRS'],
        ['CDC42EP2','CAPN1'],
        ['CDC5L','SPATS1'],
        ['CDC6','RNFT1'],
        ['CDC73','SWT1'],
        ['CDH1','FLNB'],
        ['CDH1','FTL'],
        ['CDH1','IYD'],
        ['CDH1','KLK2'],
        ['CDH1','LYZ'],
        ['CDH1','PRR11'],
        ['CDH1','SLC4A1'],
        ['CDK12','PPM1E'],
        ['CDK2AP1','SBNO1'],
        ['CDK2','ERBB3'],
        ['CDK2','RAB5B'],
        ['CDK8','IKZF3'],
        ['CDKAL1','RBM24'],
        ['CDPF1','AC012358.4'],
        ['CDV3','TMEM108'],
        ['CDYL2','LINC01227'],
        ['CDYL2','VEPH1'],
        ['CDYL','EEF1E1'],
        ['CEACAM5','PSAP'],
        ['CEACAM5','SFTPB'],
        ['CELF1','SPI1'],
        ['CELSR1','CD74'],
        ['CELSR1','CERK'],
        ['CENPV','FERMT1'],
        ['CEP192','AQP4-AS1'],
        ['CEP290','VEZT'],
        ['CEP85L','CCDC53'],
        ['CEP85L','SYTL3'],
        ['CEP85L','TMEM120B'],
        ['CERS6-AS1','TG'],
        ['CERS6','GFAP'],
        ['CES2','GPATCH2'],
        ['CFAP69','AC002064.5'],
        ['CFAP97','CCDC110'],
        ['CFDP1','BCAR1'],
        ['CFH','ASPM'],
        ['CFLAR','B2M'],
        ['CH17-140K24.5','PCDHB15'],
        ['CH17-260O16.1','CH17-129M10.1'],
        ['CH17-260O16.1','TBC1D3'],
        ['CH17-260O16.1','TBC1D3B'],
        ['CH17-260O16.1','TBC1D3C'],
        ['CH17-260O16.1','TBC1D3F'],
        ['CH17-260O16.1','TBC1D3G'],
        ['CHCHD10','VPREB3'],
        ['CHCHD2','PHKG1'],
        ['CHCHD3','CSMD1'],
        ['CHD2','DNM3OS'],
        ['CHD4','CPM'],
        ['CHD4','NOP2'],
        ['CHD9','RAD51B'],
        ['CHIA','PIFO'],
        ['CHIC2','SIPA1L3'],
        ['CHID1','PMEL'],
        ['CHIT1','SFTPB'],
        ['CHMP1A','SPIRE2'],
        ['CHMP3','SFTPB'],
        ['CHMP4B','CBFA2T2'],
        ['CHMP5','VMP1'],
        ['CHN2','TSPAN33'],
        ['CHODL','AL109761.5'],
        ['CHPT1','ATXN2'],
        ['CHPT1','TXNRD1'],
        ['CHR22-38_28785274-29006793.1','IGKC'],
        ['CHR22-38_28785274-29006793.1','PMEL'],
        ['CHST11','PTPRR'],
        ['CHST12','AC004840.9'],
        ['CHST6','TMEM170A'],
        ['CIC','ADCK4'],
        ['CIPC','LRRC74A'],
        ['CKAP4','ERBB2'],
        ['CLCN6','RAF1'],
        ['CLEC16A','TXNDC11'],
        ['CLIC1','MSH5'],
        ['CLIC2','VPS39'],
        ['CLIC4','GRHL3'],
        ['CLIC4','MYH11'],
        ['CLIP2','PTPRZ1'],
        ['CLK2','MARS'],
        ['CLN5','MYCBP2-AS1'],
        ['CLN6','C15ORF59'],
        ['CLPB','KIAA2026'],
        ['CLPB','STARD10'],
        ['CLPTM1L','TERT'],
        ['CLSTN1','CTSS'],
        ['CLSTN1','SFTPB'],
        ['CLTB','KRT14'],
        ['CLTC','MYH11'],
        ['CLTC','ROS1'],
        ['CLTC','SFTPB'],
        ['CLTC','VMP1'],
        ['CLTC','WFDC21P'],
        ['CLU','GPR56'],
        ['CLUHP3','ZNF720'],
        ['CLU','OS9'],
        ['CLU','TAGLN'],
        ['CLU','VAMP2'],
        ['CMC1','WDR72'],
        ['CMSS1','TBKBP1'],
        ['CMTM8','CMTM7'],
        ['CNDP2','CD74'],
        ['CNDP2','GNB1'],
        ['CNN1','MRVI1'],
        ['CNOT2','CRADD'],
        ['CNOT2','RAB3IP'],
        ['CNP','MBP'],
        ['CNRIP1','WDR92'],
        ['CNTN2','PKP1'],
        ['COA1','HECW1'],
        ['COIL','CDR2L'],
        ['COL14A1','COL3A1'],
        ['COL15A1','COL1A2'],
        ['COL17A1','KRT6A'],
        ['COL18A1','COL3A1'],
        ['COL18A1','CSPG4'],
        ['COL18A1','IGF2'],
        ['COL1A1','AGAP2'],
        ['COL1A1','AKAP13'],
        ['COL1A1','AKAP2'],
        ['COL1A1','ANAPC16'],
        ['COL1A1','AQP1'],
        ['COL1A1','ATXN1'],
        ['COL1A1','CACNG4'],
        ['COL1A1','CD36'],
        ['COL1A1','CEPT1'],
        ['COL1A1','COL9A3'],
        ['COL1A1','COLEC12'],
        ['COL1A1','ESYT2'],
        ['COL1A1','FNDC1'],
        ['COL1A1','GDF11'],
        ['COL1A1','GOLM1'],
        ['COL1A1','HTRA3'],
        ['COL1A1','IGHG1'],
        ['COL1A1','IGHM'],
        ['COL1A1','KIAA1217'],
        ['COL1A1','MMP9'],
        ['COL1A1','NUDT3'],
        ['COL1A1','OBSL1'],
        ['COL1A1','PMEPA1'],
        ['COL1A1','PTPRK'],
        ['COL1A1','SASH1'],
        ['COL1A1','SCARB2'],
        ['COL1A1','SULF2'],
        ['COL1A1','TAF15'],
        ['COL1A1','TUBGCP2'],
        ['COL1A2','AGAP1'],
        ['COL1A2','CDH11'],
        ['COL1A2','COL16A1'],
        ['COL1A2','COL18A1'],
        ['COL1A2','DDAH1'],
        ['COL1A2','DPYSL3'],
        ['COL1A2','HMCN1'],
        ['COL1A2','JUP'],
        ['COL1A2','KITLG'],
        ['COL1A2','LGALS3'],
        ['COL1A2','LPP'],
        ['COL1A2','MYL12B'],
        ['COL1A2','POSTN'],
        ['COL1A2','SORL1'],
        ['COL1A2','SPARCL1'],
        ['COL1A2','UFC1'],
        ['COL22A1','FAM135B'],
        ['COL3A1','6-Mar'],
        ['COL3A1','APP'],
        ['COL3A1','CCDC152'],
        ['COL3A1','CD81'],
        ['COL3A1','CFL1'],
        ['COL3A1','CKAP4'],
        ['COL3A1','CLDN4'],
        ['COL3A1','COL12A1'],
        ['COL3A1','COL4A2'],
        ['COL3A1','COLGALT1'],
        ['COL3A1','CPSF6'],
        ['COL3A1','CTNNB1'],
        ['COL3A1','EMILIN1'],
        ['COL3A1','GPC6'],
        ['COL3A1','HSPG2'],
        ['COL3A1','IGFBP7'],
        ['COL3A1','IGHG1'],
        ['COL3A1','KDELR2'],
        ['COL3A1','MMP2'],
        ['COL3A1','PHF2'],
        ['COL3A1','PSAP'],
        ['COL3A1','PUM2'],
        ['COL3A1','RAB11A'],
        ['COL3A1','SON'],
        ['COL3A1','SPTBN1'],
        ['COL3A1','TIMP3'],
        ['COL3A1','WTAP'],
        ['COL4A2','9-Sep'],
        ['COL4A2','GAPDH'],
        ['COL4A3BP','ELOVL7'],
        ['COL5A1','CD74'],
        ['COL5A1','IGF2'],
        ['COL5A1','IGKC'],
        ['COL6A2','CLU'],
        ['COL6A2','CRTAP'],
        ['COMMD1','B3GNT2'],
        ['COMMD9','CD59'],
        ['COPA','AZGP1'],
        ['COPA','MDM2'],
        ['COPA','PVRL4'],
        ['COPE','GDF1'],
        ['COQ4','DNM1'],
        ['COQ9','POLR2C'],
        ['CORO1B','ERBB2'],
        ['COX14','SPATS2'],
        ['COX6B1','UPK1A'],
        ['CPB1','AMFR'],
        ['CPB1','LIMCH1'],
        ['CPLX2','ANKRD55'],
        ['CPM','CCT2'],
        ['CPM','EIF2S3'],
        ['CPM','MDM2'],
        ['CPM','MON2'],
        ['CPM','PDZRN4'],
        ['CPNE2','PMEL'],
        ['CPQ','ANGPT1'],
        ['CPQ','MATN2'],
        ['CPSF6','C9ORF3'],
        ['CPSF6','COL1A2'],
        ['CPSF6','COL9A1'],
        ['CPSF6','GFPT2'],
        ['CPSF6','LYZ'],
        ['CPSF6','MYF6'],
        ['CPSF6','TMBIM4'],
        ['CPSF7','SMR3B'],
        ['CPT1A','NANOGNBP1'],
        ['CPT1A','PABPC1'],
        ['CPTP','TAS1R3'],
        ['CRADD','GRIP1'],
        ['CRAT','TTLL11'],
        ['CRAT','TTLL11-IT1'],
        ['CRB3','RTP1'],
        ['CREB3L2','SFTPB'],
        ['CREB5','KCNH2'],
        ['CREBBP','IGF2'],
        ['CREBBP','LA16C-325D7.2'],
        ['CREBBP','TRAP1'],
        ['CRELD1','PRRT3-AS1'],
        ['CRNN','KRT13'],
        ['CRTAP','CD74'],
        ['CRTAP','ENO1'],
        ['CRTAP','GNS'],
        ['CRTAP','MYH11'],
        ['CRTAP','PLEC'],
        ['CRTAP','PMEL'],
        ['CRTAP','TG'],
        ['CRTC1','KLHL26'],
        ['CRYAB','GFAP'],
        ['CSAG1','CSAG2'],
        ['CSAG2','CSAG1'],
        ['CSDE1','CTB-171A8.1'],
        ['CSDE1','RBM3'],
        ['CSN3','FDCSP'],
        ['CSNK1D','SECTM1'],
        ['CSNK1G1','AP4E1'],
        ['CSNK1G2','PDE5A'],
        ['CSNK2A1','SMARCA2'],
        ['CSPG4','ACVR1'],
        ['CSPG4','APP'],
        ['CSPG4','CAPN2'],
        ['CSPG4','GADD45GIP1'],
        ['CSPG4','LAMA4'],
        ['CSPG4','MED15'],
        ['CSPG4','MET'],
        ['CSPG4','MSH6'],
        ['CSPG4','RPL37A'],
        ['CSPG4','VIM'],
        ['CSPG5','PTPRZ1'],
        ['CSRP1','IGF2'],
        ['CSRP2','SNRPF'],
        ['CTA-221G9.7','CRYBB2'],
        ['CTA-246H3.11','PHF10'],
        ['CTA-253N17.1','RPL22'],
        ['CTA-292E10.9','AZGP1'],
        ['CTA-292E10.9','TG'],
        ['CTA-392C11.1','ADAM9'],
        ['CTA-392E5.1','ATP6V0D2'],
        ['CTB-114C7.4','LCP2'],
        ['CTB-131B5.5','CYSTM1'],
        ['CTB-186G2.1','EIF3K'],
        ['CTB-35F21.1','PSD2'],
        ['CTB-47B8.1','RPLP2'],
        ['CTB-60B18.15','CTB-60B18.6'],
        ['CTBP2','ADAM12'],
        ['CTDSP1','C9ORF3'],
        ['CTDSP2','AUH'],
        ['CTDSP2','AVIL'],
        ['CTDSP2','COL15A1'],
        ['CTDSP2','CPM'],
        ['CTDSP2','GPR84'],
        ['CTDSP2','LRP1'],
        ['CTDSP2','NFE2'],
        ['CTDSP2','RALB'],
        ['CTDSP2','RAP1B'],
        ['CTIF','MOCOS'],
        ['CTNNA1','CTB-46B19.2'],
        ['CTNNA1','IGF2'],
        ['CTNNA1','PMEL'],
        ['CTNNA1','SFTPB'],
        ['CTNNA1','TSPAN31'],
        ['CTNNA2','SPARC'],
        ['CTNNBL1','VSTM2L'],
        ['CTNND2','NANS'],
        ['CTSB','AARS'],
        ['CTSB','ABR'],
        ['CTSB','ACTN4'],
        ['CTSB','AHNAK'],
        ['CTSB','ALDH2'],
        ['CTSB','ALDOA'],
        ['CTSB','AP2B1'],
        ['CTSB','APLP2'],
        ['CTSB','APOD'],
        ['CTSB','ARFGEF2'],
        ['CTSB','ATP9A'],
        ['CTSB','BRPF3'],
        ['CTSB','C1QC'],
        ['CTSB','C7ORF55-LUC7L2'],
        ['CTSB','CADM1'],
        ['CTSB','CANX'],
        ['CTSB','CAPG'],
        ['CTSB','CDH1'],
        ['CTSB','CLDN4'],
        ['CTSB','CPAMD8'],
        ['CTSB','CRYAB'],
        ['CTSB','CTNNA1'],
        ['CTSB','DAG1'],
        ['CTSB','DAZAP2'],
        ['CTSB','DDB1'],
        ['CTSB','DEFB136'],
        ['CTSB','DTNA'],
        ['CTSB','EIF5B'],
        ['CTSB','EMC10'],
        ['CTSB','EMID1'],
        ['CTSB','ESYT2'],
        ['CTSB','FAM129B'],
        ['CTSB','GATAD2A'],
        ['CTSB','GNB1'],
        ['CTSB','GPI'],
        ['CTSB','GSN'],
        ['CTSB','HP1BP3'],
        ['CTSB','IGHG1'],
        ['CTSB','IGHG2'],
        ['CTSB','ITM2B'],
        ['CTSB','KANK2'],
        ['CTSB','KCTD10'],
        ['CTSB','KDM2A'],
        ['CTSB','KIAA1244'],
        ['CTSB','LAMP1'],
        ['CTSB','LMNA'],
        ['CTSB','LTF'],
        ['CTSB','MANBA'],
        ['CTSB','MAP4'],
        ['CTSB','MFSD12'],
        ['CTSB','MGST1'],
        ['CTSB','MRPL30'],
        ['CTSB','MSN'],
        ['CTSB','MYH14'],
        ['CTSB','MYO9B'],
        ['CTSB','NAV2'],
        ['CTSB','NCL'],
        ['CTSB','NUCKS1'],
        ['CTSB','NUDT3'],
        ['CTSB','PAX8'],
        ['CTSB','PEBP1'],
        ['CTSB','PGAM5'],
        ['CTSB','PI4KA'],
        ['CTSB','PIGR'],
        ['CTSB','PLD3'],
        ['CTSB','POSTN'],
        ['CTSB','PPAP2B'],
        ['CTSB','PPL'],
        ['CTSB','PROS1'],
        ['CTSB','PRRC2B'],
        ['CTSB','PTTG1IP'],
        ['CTSB','RAB7A'],
        ['CTSB','RAP1GAP'],
        ['CTSB','RHOA'],
        ['CTSB','RNF213'],
        ['CTSB','RSPRY1'],
        ['CTSB','SCARB2'],
        ['CTSB','SDC2'],
        ['CTSB','SH3BP5'],
        ['CTSB','SHISA5'],
        ['CTSB','SLC1A5'],
        ['CTSB','SLC25A29'],
        ['CTSB','SNTA1'],
        ['CTSB','SNX1'],
        ['CTSB','SNX4'],
        ['CTSB','SPINT2'],
        ['CTSB','SPRED1'],
        ['CTSB','SPTBN1'],
        ['CTSB','ST6GAL1'],
        ['CTSB','STAG2'],
        ['CTSB','STK10'],
        ['CTSB','STOM'],
        ['CTSB','STRN4'],
        ['CTSB','SUN1'],
        ['CTSB','SYNM'],
        ['CTSB','TAF15'],
        ['CTSB','TBC1D16'],
        ['CTSB','TIMP2'],
        ['CTSB','TMBIM6'],
        ['CTSB','TSPAN3'],
        ['CTSB','UBE2H'],
        ['CTSB','VCL'],
        ['CTSB','WDTC1'],
        ['CTSB','YWHAE'],
        ['CTSD','ACTG1'],
        ['CTTN','ANO1'],
        ['CUL3','ACPP'],
        ['CUL4A','F7'],
        ['CUL4A','PCID2'],
        ['CUL5','ACAT1'],
        ['CUX1','SFTPB'],
        ['CWC25','B4GALNT2'],
        ['CWC25','PIP4K2B'],
        ['CXCL5','GLYR1'],
        ['CYBRD1','MYH11'],
        ['CYFIP1','ACPP'],
        ['CYP11B2','CYP11B1'],
        ['CYP20A1','ABI2'],
        ['CYP2T3P','CYP2T2P'],
        ['CYP4F11','CYP4F23P'],
        ['CYP4F12','CYP4F23P'],
        ['CYP4F12','CYP4F24P'],
        ['CYP4F24P','CYP4F12'],
        ['CYP4F8','CYP4F23P'],
        ['CYP4Z2P','TRABD2B'],
        ['CYSLTR2','DLEU1'],
        ['DAGLA','COL1A2'],
        ['DAND5','NFIX'],
        ['DAP3','TG'],
        ['DAP','SEMA5A'],
        ['DAP','SNX9'],
        ['DARS-AS1','AC068492.1'],
        ['DBF4','VAMP4'],
        ['DBNDD1','TPGS1'],
        ['DCAF10','NR6A1'],
        ['DCAF6','HMCN1'],
        ['DCAF6','MPZL1'],
        ['DCAF6','UFC1'],
        ['DCAF6','YEATS4'],
        ['DCAF7','VMP1'],
        ['DCAF8','CD1C'],
        ['DCAKD','PLCD3'],
        ['DCBLD2','ST3GAL6-AS1'],
        ['DCD','LACRT'],
        ['DCN','ANK2'],
        ['DCP1A','LINC00681'],
        ['DCPS','FGF3'],
        ['DCT','COL4A1'],
        ['DCT','DDX21'],
        ['DCTN1','KRT5'],
        ['DCTN2','IFNG-AS1'],
        ['DCTN2','NAV3'],
        ['DCTN2','SRGAP1'],
        ['DCTN2','SYT1'],
        ['DDB1','TMEM258'],
        ['DDHD2','APPL2'],
        ['DDR1','MRPS18B'],
        ['DDR1','XXBAC-BPG252P9.10'],
        ['DDX17','CLU'],
        ['DDX21','SFTPB'],
        ['DDX23','PMEL'],
        ['DDX39B','PRPH2'],
        ['DDX42','BRIP1'],
        ['DDX46','TPO'],
        ['DDX52','PSMD3'],
        ['DDX56','NPC1L1'],
        ['DDX5','POLG2'],
        ['DEAF1','SCT'],
        ['DENND1B','TSPAN31'],
        ['DENND6B','MAPK11'],
        ['DEPDC1B','PDE4D'],
        ['DERA','DNAJA2'],
        ['DES','KLK3'],
        ['DES','KRT7'],
        ['DGCR2','CLTCL1'],
        ['DGCR2','GNB1L'],
        ['DGKD','NDUFA10'],
        ['DGUOK','AC073046.25'],
        ['DGUOK','ACTG2'],
        ['DHDDS','LINC00505'],
        ['DHFRL1','STX19'],
        ['DHRS2','GSTM2'],
        ['DHX15','CCND3'],
        ['DHX15','RBPJ'],
        ['DIABLO','TG'],
        ['DIAPH2','RAB3IP'],
        ['DIDO1','RPS4X'],
        ['DIP2A','PRH1'],
        ['DIP2B','ATF1'],
        ['DIP2B','RTTN'],
        ['DLEU2','LINC00462'],
        ['DLG3','UNC5D'],
        ['DLGAP1','ARHGAP28'],
        ['DLGAP4','KRT14'],
        ['DMC1','FAM227A'],
        ['DMKN','KRT1'],
        ['DNAAF1','ADIPOR2'],
        ['DNAJC16','FHAD1'],
        ['DNAJC5','SFTPB'],
        ['DNASE2','TG'],
        ['DNM1L','BICD1'],
        ['DNM2','GDF15'],
        ['DNM2','SMARCA4'],
        ['DNM3','KERA'],
        ['DNM3OS','TMTC2'],
        ['DNM3','TPH2'],
        ['DOCK1','DPCD'],
        ['DOCK4','SND1'],
        ['DOCK7','INADL'],
        ['DOCK8','TG'],
        ['DOCK9','SLC34A2'],
        ['DOPEY1','RNF144B'],
        ['DPP3','ERBB2'],
        ['DPP3','FEZ1'],
        ['DPP3','PACS1'],
        ['DPP9','KCTD21-AS1'],
        ['DPP9','PMEL'],
        ['DPT','RAP1B'],
        ['DRAM1','GNPTAB'],
        ['DRAM1','SFTPB'],
        ['DRAXIN','AGTRAP'],
        ['DRICH1','KB-208E9.1'],
        ['DSCC1','TAF2'],
        ['DSCR3','IGSF5'],
        ['DSCR3','MACROD2'],
        ['DSCR4','KCNJ6'],
        ['DSE','FAM26F'],
        ['DSG3','PERP'],
        ['DSP','MYH11'],
        ['DST','IGKC'],
        ['DST','KRT4'],
        ['DST','LAMC2'],
        ['DST','MLIP'],
        ['DST','MYH9'],
        ['DSTYK','NAV1'],
        ['DTD2','BBS5'],
        ['DTNA','GFAP'],
        ['DUOXA1','DUOX2'],
        ['DUS1L','FASN'],
        ['DUS1L','RFNG'],
        ['DUSP14','EME1'],
        ['DUSP3','PMEL'],
        ['DUXAP10','C2ORF27A'],
        ['DUXAP8','C2ORF27A'],
        ['DYNC1H1','RBM3'],
        ['DYNC1H1','SPARC'],
        ['DYNC1I2','AC106900.6'],
        ['DYRK1A','APP'],
        ['DYRK1A','TTC3'],
        ['DYRK1A','ZBTB46'],
        ['E2F1','NECAB3'],
        ['E2F3','AKIRIN2'],
        ['E2F7','PTPRR'],
        ['EAF2','RGS9'],
        ['ECE1','NIPAL3'],
        ['ECT2','EGFEM1P'],
        ['EDA','TEX11'],
        ['EEA1','COL1A2'],
        ['EEF1D','KRT14'],
        ['EEF1DP3','CLDN10'],
        ['EEF1D','TG'],
        ['EEF1G','GFAP'],
        ['EEF2','HNRNPH2'],
        ['EEF2K','UQCRC2'],
        ['EEF2','PEA15'],
        ['EFCAB13','MYL4'],
        ['EFCAB7','PGM1'],
        ['EFHD1','EIF4E2'],
        ['EFNA3','PIK3C2G'],
        ['EFNA4','C20ORF96'],
        ['EFR3A','CASC21'],
        ['EFTUD2','PMEL'],
        ['EGFR','AARS2'],
        ['EGFR','ERP44'],
        ['EGFR','GNS'],
        ['EGFR','KRT5'],
        ['EGFR','LANCL2'],
        ['EGFR','PPM1H'],
        ['EGFR','SEC61G'],
        ['EGR1','SLC45A3'],
        ['EHBP1L1','FRMD8'],
        ['EHD4','SPTBN5'],
        ['EHF','CAT'],
        ['EHF','Z83001.1'],
        ['EHMT2','SLC44A4'],
        ['EIF2AK1','SLC34A2'],
        ['EIF2AK2','CEBPZ'],
        ['EIF2AK4','BUB1B'],
        ['EIF2A','LINC01213'],
        ['EIF3B','PHGDH'],
        ['EIF3E','CNN2'],
        ['EIF3I','MGP'],
        ['EIF3I','TG'],
        ['EIF3K','ACTN4'],
        ['EIF3K','CIC'],
        ['EIF3L','ORAOV1'],
        ['EIF3M','UEVLD'],
        ['EIF4A1','PHOSPHO1'],
        ['EIF4G2','IGKC'],
        ['EIF4G3','SPSB1'],
        ['EIF4H','ELN'],
        ['ELF2','NR1I3'],
        ['ELF5','CAT'],
        ['ELK3','CDK17'],
        ['ELK3','KRT78'],
        ['ELK3','PMEL'],
        ['ELK3','SBNO1'],
        ['ELMSAN1','ENTPD5'],
        ['ELN','FKBP14'],
        ['ELP2','MAPRE2'],
        ['EMC3','ERBB2'],
        ['EMC3','PRRT3'],
        ['EML1','BASP1'],
        ['EML4','ALK'],
        ['EMP2','APLP2'],
        ['EMP2','PSAP'],
        ['ENG','AK1'],
        ['ENG','TG'],
        ['ENO1','TAPBP'],
        ['ENOX1','FNDC3A'],
        ['ENPP1','CTSB'],
        ['ENSA','KPRP'],
        ['ENTPD5','AHNAK'],
        ['ENTPD5','FAM161B'],
        ['EPB41L4A','MCC'],
        ['EPB41L4B','PTPN3'],
        ['EPB41','VPS13B'],
        ['EPCAM','ACPP'],
        ['EPN2','KCNJ18'],
        ['EPS15L1','CLNS1A'],
        ['EPS15L1','MED26'],
        ['EPS8L2','TALDO1'],
        ['ERBB2','ARL5C'],
        ['ERBB2','CD82'],
        ['ERBB2','CTA-292E10.9'],
        ['ERBB2','CTTN'],
        ['ERBB2','GRB7'],
        ['ERBB2','IGKC'],
        ['ERBB2IP','ARL15'],
        ['ERBB2IP','SREK1'],
        ['ERBB2','PBX1'],
        ['ERBB2','PGAP3'],
        ['ERBB2','SELENBP1'],
        ['ERBB3','BCAR4'],
        ['ERBB3','RPL8'],
        ['ERBB3','TMEM198B'],
        ['ERC1','RET'],
        ['ERF','GSK3A'],
        ['ERGIC1','KLK3'],
        ['ERICH1','IYD'],
        ['ERO1L','FN1'],
        ['ERRFI1','VAMP3'],
        ['ERVK13-1','PDPK1'],
        ['ESCO1','ABHD3'],
        ['ESR1','CCDC170'],
        ['ESR1','SYNE1'],
        ['ESRP1','DPY19L4'],
        ['ESRP1','TRIQK'],
        ['ESRP2','SLC7A6'],
        ['ESRRG','ADCY10'],
        ['ESYT2','NCAPG2'],
        ['ETF1','CD24'],
        ['ETFA','IMP3'],
        ['ETFB','LIM2'],
        ['ETS1','UBE2V2'],
        ['ETS2','IGHG1'],
        ['ETV6','NTRK3'],
        ['ETV6','YTHDF2'],
        ['EVA1C','MRAP'],
        ['EVPL','COL3A1'],
        ['EVPL','MAP2K6'],
        ['EXD2','TG'],
        ['EXOC4','COL1A1'],
        ['EXOC4','LRGUK'],
        ['EXOC4','PTAFR'],
        ['EXOC4','SLC25A13'],
        ['EXOC6B','SFXN5'],
        ['EXOC7','RNF157'],
        ['EXOSC10','MYH11'],
        ['EXT1','OC90'],
        ['EXT2','MIR670HG'],
        ['EZR','ROS1'],
        ['EZR','UPK1B'],
        ['F11R','NOS1AP'],
        ['F3','LINC01057'],
        ['F3','SFTPB'],
        ['FAAHP1','LINC00505'],
        ['FABP2','C4ORF3'],
        ['FADS1','TMEM258'],
        ['FAF1','NFIA'],
        ['FAM117A','SLC35B1'],
        ['FAM129A','AC138430.4'],
        ['FAM129B','C16ORF62'],
        ['FAM129B','IGHM'],
        ['FAM13C','ZNF365'],
        ['FAM168A','PTCRA'],
        ['FAM168A','PXDNL'],
        ['FAM168A','SNX29'],
        ['FAM172A','FHIT'],
        ['FAM172A','NR2F1-AS1'],
        ['FAM174B','IGKC'],
        ['FAM174B','SFTPB'],
        ['FAM193A','TNIP2'],
        ['FAM19A2','TMBIM4'],
        ['FAM213A','CD24'],
        ['FAM217B','CDH26'],
        ['FAM219A','PAX5'],
        ['FAM219A','POLN'],
        ['FAM21A','FAM21EP'],
        ['FAM21EP','FAM21A'],
        ['FAM222B','CRLF3'],
        ['FAM222B','TLCD1'],
        ['FAM222B','USP32'],
        ['FAM65C','TMEM189-UBE2V1'],
        ['FAM66A','FAM66C'],
        ['FAM78B','RAB3IP'],
        ['FAM85A','FAM85B'],
        ['FAM89B','EHBP1L1'],
        ['FANCD2','BRK1'],
        ['FAR1','C3ORF55'],
        ['FAR2','OVCH1-AS1'],
        ['FARP1','SFTPB'],
        ['FARS2','SYCP2L'],
        ['FARSA','SYCE2'],
        ['FASN','KLK3'],
        ['FAT1','F11-AS1'],
        ['FBF1','HELZ'],
        ['FBLN2','COL3A1'],
        ['FBL','RASGRP4'],
        ['FBL','RYR1'],
        ['FBN1','COL3A1'],
        ['FBN2','IGKC'],
        ['FBXL20','CEP112'],
        ['FBXL20','ERBB2'],
        ['FBXL20','GDPD5'],
        ['FBXL20','MIEN1'],
        ['FBXL20','MYO1D'],
        ['FBXL20','SIRT6'],
        ['FBXL20','TLK2'],
        ['FBXL5','ERBB2'],
        ['FBXL7','CCT5'],
        ['FBXL7','NRD1'],
        ['FBXO31','C16ORF95'],
        ['FBXO9','CD74'],
        ['FBXW4','MGEA5'],
        ['FCGRT','BCAT2'],
        ['FCHSD2','ATP6V1H'],
        ['FCHSD2','THRA'],
        ['FCHSD2','UQCC3'],
        ['FDCSP','CALM2'],
        ['FDCSP','CSN1S2AP'],
        ['FDCSP','MDH2'],
        ['FDCSP','PDCD11'],
        ['FDCSP','PRR27'],
        ['FDFT1','CTSB'],
        ['FDFT1','INTS9'],
        ['FDFT1','NEIL2'],
        ['FECH','NCMAP'],
        ['FER1L4','COL4A2'],
        ['FERMT1','CDRT4'],
        ['FER','TG'],
        ['FGD4','ABCC9'],
        ['FGD4','DNM1L'],
        ['FGFR3','TACC3'],
        ['FITM2','OSER1'],
        ['FKBP14','SCRN1'],
        ['FKBP2','C11ORF84'],
        ['FKBP8','SLC1A6'],
        ['FLJ16171','SKAP2'],
        ['FLJ21408','IL4R'],
        ['FLNA','WI2-89031B12.1'],
        ['FLNB','CPB1'],
        ['FLNB','PDLIM5'],
        ['FLOT2','PEX12'],
        ['FLYWCH1','FAR1'],
        ['FMN1','LGALS3'],
        ['FMNL3','CPNE8'],
        ['FMNL3','HECTD4'],
        ['FN1','AKAP13'],
        ['FN1','ATG7'],
        ['FN1','COL15A1'],
        ['FN1','EML4'],
        ['FN1','GPNMB'],
        ['FN1','GPR56'],
        ['FN1','GPX3'],
        ['FN1','ITGAV'],
        ['FN1','KIAA1244'],
        ['FN1','KRT17'],
        ['FN1','MORF4L2'],
        ['FN1','PINK1-AS'],
        ['FN1','TAPBP'],
        ['FOS','ACPP'],
        ['FOXA2','LINC00261'],
        ['FOXJ3','ZFYVE9'],
        ['FOXP4','PIGR'],
        ['FOXRED2','KIAA0513'],
        ['FOXRED2','TXN2'],
        ['FREM2','MTRF1'],
        ['FRMD5','MFAP1'],
        ['FRMD8','MAST1'],
        ['FRMD8','POLA2'],
        ['FRS2','ATXN1'],
        ['FRS2','C12ORF55'],
        ['FRS2','CPM'],
        ['FRS2','HMGA2'],
        ['FRS2','PLEKHG7'],
        ['FRS2','SBF2'],
        ['FRS2','TMTC2'],
        ['FSCN1','AHNAK'],
        ['FTH1','COL3A1'],
        ['FTL','CDH1'],
        ['FTL','KLK3'],
        ['FTL','SPP1'],
        ['FURIN','SMR3B'],
        ['FUT6','DUS3L'],
        ['FXN','FPR2'],
        ['FXN','SMC5'],
        ['FXR1','NAALADL2'],
        ['FXR1','PMEL'],
        ['FXR1','SOX2-OT'],
        ['FXR1','VGLL4'],
        ['FXYD3','KRT6A'],
        ['FXYD6','DSCAML1'],
        ['FXYD6','KIF1B'],
        ['FYN','PMEL'],
        ['FZR1','MAN2B1'],
        ['G3BP2','IGF2'],
        ['G6PD','WI2-89031B12.1'],
        ['GAB2','ALG8'],
        ['GAB2','FGF7'],
        ['GAB2','GAS7'],
        ['GAB2','KCTD21'],
        ['GAB2','RBM3'],
        ['GAB2','TENM4'],
        ['GABPB2','OR10J1'],
        ['GABRP','KCNIP1'],
        ['GALNS','APRT'],
        ['GALNT11','EZH2'],
        ['GALNT14','SF3B6'],
        ['GALNT18','PARVA'],
        ['GALNT18','PPP1R1B'],
        ['GALNT2','PTPRZ1'],
        ['GANAB','ZNF76'],
        ['GAS2L3','KRT2'],
        ['GAS5','ACP6'],
        ['GAS7','ZNF106'],
        ['GAS8','WDR59'],
        ['GATAD2A','MAU2'],
        ['GATAD2B','GRM8'],
        ['GBA','SCAMP3'],
        ['GBF1','GFAP'],
        ['GBP3','RBMXL1'],
        ['GCNT2','TXNDC5'],
        ['GCSAML','FAM129A'],
        ['GDI2','CPVL'],
        ['GDPGP1','TTLL13'],
        ['GFAP','ACTB'],
        ['GFAP','ACTN4'],
        ['GFAP','ALDOA'],
        ['GFAP','APOE'],
        ['GFAP','APP'],
        ['GFAP','ASPH'],
        ['GFAP','BCAN'],
        ['GFAP','BSCL2'],
        ['GFAP','CALR'],
        ['GFAP','CAPNS1'],
        ['GFAP','CERS6'],
        ['GFAP','CKAP5'],
        ['GFAP','CNDP2'],
        ['GFAP','CNTN1'],
        ['GFAP','COPA'],
        ['GFAP','DLGAP4'],
        ['GFAP','DLL3'],
        ['GFAP','DTNA'],
        ['GFAP','EEF2'],
        ['GFAP','EGFR'],
        ['GFAP','FCGBP'],
        ['GFAP','GBF1'],
        ['GFAP','HIATL1'],
        ['GFAP','ITM2B'],
        ['GFAP','LTF'],
        ['GFAP','MAP4'],
        ['GFAP','MPRIP'],
        ['GFAP','MTSS1'],
        ['GFAP','PDE4B'],
        ['GFAP','SFRP4'],
        ['GFAP','SH3BP2'],
        ['GFAP','SOD2'],
        ['GFAP','SPARC'],
        ['GFAP','SPP1'],
        ['GFAP','SRCAP'],
        ['GFAP','TRIM9'],
        ['GFAP','TSPAN31'],
        ['GFAP','VIM-AS1'],
        ['GFM1','MLF1'],
        ['GFOD2','TRIO'],
        ['GFPT1','NFU1'],
        ['GGA1','PMEL'],
        ['GGA1','TRIOBP'],
        ['GGNBP2','APOL4'],
        ['GGNBP2','ZMAT4'],
        ['GGT5','TG'],
        ['GGT6','MYBBP1A'],
        ['GIGYF2','CHRND'],
        ['GIGYF2','EFHD1'],
        ['GIPC1','PMEL'],
        ['GIPC1','SLC34A2'],
        ['GIPC1','TG'],
        ['GJB3','KRT5'],
        ['GLCE','BCL7C'],
        ['GLI3','HIBADH'],
        ['GLIPR1L1','RAB3IP'],
        ['GLIPR2','RECK'],
        ['GLT8D2','YY1'],
        ['GLTP','ATP6V0D2'],
        ['GLYR1','AZGP1'],
        ['GLYR1','CXCL5'],
        ['GMDS','METTL24'],
        ['GMPPA','ASIC4'],
        ['GMPS','C3ORF55'],
        ['GNA11','AC005264.2'],
        ['GNA13','CCDC171'],
        ['GNAI3','CTTNBP2NL'],
        ['GNAI3','SARS'],
        ['GNAQ','IGFBP5'],
        ['GNAS','CA12'],
        ['GNAS','CEACAM5'],
        ['GNAS','ENO1'],
        ['GNAS','HSP90B1'],
        ['GNAS','MGP'],
        ['GNAS','RTFDC1'],
        ['GNB1','IGHG1'],
        ['GNB1','NADK'],
        ['GNL1','REPS1'],
        ['GNPTAB','DRAM1'],
        ['GNPTAB','TRHDE'],
        ['GNS','APOE'],
        ['GNS','COL22A1'],
        ['GNS','DDOST'],
        ['GNS','EGFR'],
        ['GNS','GRSF1'],
        ['GNS','NUP107'],
        ['GNS','P4HB'],
        ['GNS','RFC2'],
        ['GNS','RNF41'],
        ['GOLGA2P8','ZNF774'],
        ['GOLGA3','IGKC'],
        ['GOLGA4','HSPA8'],
        ['GOLGA4','MLH1'],
        ['GOLGB1','CCDC58'],
        ['GOLM1','ACPP'],
        ['GOLM1','AZGP1'],
        ['GOLM1','ERBB2'],
        ['GOLM1','SLC44A4'],
        ['GOLT1A','KISS1'],
        ['GON4L','SSR2'],
        ['GOPC','ABLIM1'],
        ['GOT2P3','GOT2'],
        ['GPATCH11','DYM'],
        ['GPBP1L1','CCDC17'],
        ['GPC6','DCN'],
        ['GPD2','GALNT5'],
        ['GPHN','EIF2S1'],
        ['GPI','ALDOA'],
        ['GPI','APP'],
        ['GPI','C6ORF89'],
        ['GPI','IGFBP2'],
        ['GPI','PKP1'],
        ['GPI','PMEL'],
        ['GPI','RGS5'],
        ['GPM6B','GFAP'],
        ['GPNMB','CA12'],
        ['GPNMB','RBM3'],
        ['GPR107','TG'],
        ['GPR116','MEP1A'],
        ['GPR124','GSR'],
        ['GPR137C','FERMT2'],
        ['GPR141','NME8'],
        ['GPR39','AC010974.3'],
        ['GPR56','CSTB'],
        ['GPR56','CTSB'],
        ['GPR56','SLC34A2'],
        ['GPR56','TG'],
        ['GPR56','TGM4'],
        ['GPRC5A','APLP2'],
        ['GPRC5A','CTSB'],
        ['GPRC5A','FAM83A'],
        ['GPRC5A','TMBIM6'],
        ['GPSM2','GFAP'],
        ['GPX3','IGHG1'],
        ['GPX7','FAM159A'],
        ['GRAP2','BRD1'],
        ['GRB10','GTPBP10'],
        ['GRB2','SS18L1'],
        ['GRB7','ERBB2'],
        ['GRHL2','NACAP1'],
        ['GRM1','CNOT2'],
        ['GRPR','MAGEB17'],
        ['GSK3B','FSTL1'],
        ['GSN','ACPP'],
        ['GSN','CSTB'],
        ['GSN','ERBB2'],
        ['GSN','MYH11'],
        ['GSN','SFTPB'],
        ['GSN','SLC34A2'],
        ['GSN','TGM4'],
        ['GSTP1','MSLN'],
        ['GSTP1','MYO10'],
        ['GTF2E1','RABL3'],
        ['GTF2H3','MDM2'],
        ['GTF2IRD1','LINC00881'],
        ['GTPBP1','TOMM22'],
        ['GUCY2C','PLBD1'],
        ['GUK1','TARBP1'],
        ['GUK1','TPPP'],
        ['H1FX','GFAP'],
        ['H2AFY','CNOT6'],
        ['HADHB','TG'],
        ['HAT1','EXOC6B'],
        ['HAVCR2','IGFBP3'],
        ['HAX1','U40455.1'],
        ['HDAC11','CDH18'],
        ['HDAC1','KPNA6'],
        ['HDAC4','PDE11A'],
        ['HDAC4','SATB2'],
        ['HDAC6','ERAS'],
        ['HDAC7','SRGAP1'],
        ['HDAC8','CITED1'],
        ['HDDC2','TPD52L1'],
        ['HDGF','IGHG1'],
        ['HDLBP','KRT17'],
        ['HDX','ERICH1'],
        ['HEATR6','MIEN1'],
        ['HEATR6','PPME1'],
        ['HECTD4','AMDHD1'],
        ['HECTD4','LTA4H'],
        ['HECW1','COA1'],
        ['HEG1','CTSB'],
        ['HELZ','CACNG4'],
        ['HEPHL1','PANX1'],
        ['HERC1','TG'],
        ['HERC2P10','HERC2'],
        ['HERC3','FAM13A-AS1'],
        ['HEXA','MYO9A'],
        ['HEXDC','C17ORF62'],
        ['HEYL','COL9A2'],
        ['HGD','IGKC'],
        ['HIBADH','SUGCT'],
        ['HID1','ARHGEF26-AS1'],
        ['HIRA','BCL2L13'],
        ['HIRA','TPO'],
        ['HIST4H4','MGP'],
        ['HIVEP1','TXNDC5'],
        ['HIVEP3','IGKC'],
        ['HK1','ACPP'],
        ['HK1','CTSB'],
        ['HK1','SUPV3L1'],
        ['HM13','KLK3'],
        ['HMBOX1','BNIP3L'],
        ['HMGA2','ARFGAP2'],
        ['HMGA2','DNM3OS'],
        ['HMGA2','NR2C1'],
        ['HMGA2','PPFIA2'],
        ['HMGB1','CRYL1'],
        ['HMGN2P46','ACPP'],
        ['HN1','NT5C'],
        ['HNRNPC','NDRG2'],
        ['HNRNPH1','IGFBP5'],
        ['HOMER2','TG'],
        ['HOOK1','FGGY'],
        ['HOOK1','RAB6A'],
        ['HOOK2','SFTPB'],
        ['HOOK3','IKBKB'],
        ['HOXB3','TCF12'],
        ['HOXC10','WIBG'],
        ['HOXC6','FLJ12825'],
        ['HOXC6','WIBG'],
        ['HOXC9','WIBG'],
        ['HP1BP3','ZMYM4'],
        ['HPN-AS1','LGI4'],
        ['HPN','PPP2R5C'],
        ['HPS1','PYROXD2'],
        ['HPS3','SFTPB'],
        ['HRASLS','ATP13A5-AS1'],
        ['HSD17B4','MSMB'],
        ['HSF1','ERBB2'],
        ['HSP90AA1','MBP'],
        ['HSP90AB1','PTMS'],
        ['HSP90AB1','SLC29A1'],
        ['HSP90B1','PTMS'],
        ['HSP90B1','TMEM19'],
        ['HSPB7','CTB-129P6.7'],
        ['HSPG2','CSPG4'],
        ['HSPG2','IGF2'],
        ['HUWE1','PTPRZ1'],
        ['IARS2','PIGR'],
        ['ICMT','ACOT7'],
        ['IDS','SLC3A1'],
        ['IFFO2','UBR4'],
        ['IFI6','CD74'],
        ['IFI6','PTP4A2'],
        ['IFI6','SPDEF'],
        ['IFI6','TIMP2'],
        ['IGF2','9-Sep'],
        ['IGF2','ABI2'],
        ['IGF2','AC007563.5'],
        ['IGF2','ANO6'],
        ['IGF2','BCL2L1'],
        ['IGF2','BRD8'],
        ['IGF2','CHD9'],
        ['IGF2','COL18A1'],
        ['IGF2','COL3A1'],
        ['IGF2','COL4A2'],
        ['IGF2','FAM83A'],
        ['IGF2','FOXP4'],
        ['IGF2','ICAM1'],
        ['IGF2','KIF1C'],
        ['IGF2','LAMA5'],
        ['IGF2','NCOA6'],
        ['IGF2','PIK3R1'],
        ['IGF2','PTPRD'],
        ['IGF2','TMEM184A'],
        ['IGF2','YES1'],
        ['IGF2','ZCCHC24'],
        ['IGFBP2','SMARCAL1'],
        ['IGFBP5','AHR'],
        ['IGFBP5','CDKN1B'],
        ['IGFBP5','LSAMP'],
        ['IGFBP5','MYH11'],
        ['IGFBP5','RPN1'],
        ['IGFBP5','SNRNP70'],
        ['IGFBP7','MYLK'],
        ['IGFBP7','SFTPB'],
        ['IGFBP7','SPINK2'],
        ['IGFL1P1','DKFZP434J0226'],
        ['IGHA1','MUC1'],
        ['IGHA1','TSC22D1'],
        ['IGHG1','ACTB'],
        ['IGHG1','AKR1C1'],
        ['IGHG1','APOL1'],
        ['IGHG1','CLDN1'],
        ['IGHG1','COL1A2'],
        ['IGHG1','COL3A1'],
        ['IGHG1','COL4A1'],
        ['IGHG1','CTSB'],
        ['IGHG1','DYM'],
        ['IGHG1','EEF1G'],
        ['IGHG1','EEF2'],
        ['IGHG1','FLNA'],
        ['IGHG1','HMGA1'],
        ['IGHG1','IGF1R'],
        ['IGHG1','IGLL5'],
        ['IGHG1','IGLV4-69'],
        ['IGHG1','JUP'],
        ['IGHG1','KDM4C'],
        ['IGHG1','MET'],
        ['IGHG1','MYH9'],
        ['IGHG1','NONO'],
        ['IGHG1','PGRMC1'],
        ['IGHG1','PRRC2B'],
        ['IGHG1','PSMC4'],
        ['IGHG1','RPS8'],
        ['IGHG1','SFTPA2'],
        ['IGHG1','SOD2'],
        ['IGHG1','SRRM2'],
        ['IGHM','EEF1G'],
        ['IGHM','FOS'],
        ['IGHM','MYH9'],
        ['IGHM','PAPSS2'],
        ['IGHM','SPINK1'],
        ['IGKC','ALB'],
        ['IGKC','ATG2A'],
        ['IGKC','ATG7'],
        ['IGKC','BCAM'],
        ['IGKC','CAP1'],
        ['IGKC','CD2AP'],
        ['IGKC','CFL1'],
        ['IGKC','CLIP1'],
        ['IGKC','CTSH'],
        ['IGKC','DAB2IP'],
        ['IGKC','DDOST'],
        ['IGKC','DNAH17'],
        ['IGKC','EEF2'],
        ['IGKC','EGFR'],
        ['IGKC','EIF3E'],
        ['IGKC','FBXO7'],
        ['IGKC','GABRP'],
        ['IGKC','GPRC5B'],
        ['IGKC','GRB2'],
        ['IGKC','HDAC5'],
        ['IGKC','HMCN1'],
        ['IGKC','HOPX'],
        ['IGKC','INF2'],
        ['IGKC','INTS1'],
        ['IGKC','IRF9'],
        ['IGKC','KRT6A'],
        ['IGKC','LAMA5'],
        ['IGKC','LAMC1'],
        ['IGKC','LAMP1'],
        ['IGKC','MDH2'],
        ['IGKC','MED15'],
        ['IGKC','MSLN'],
        ['IGKC','MTSS1'],
        ['IGKC','MUC6'],
        ['IGKC','MYO18A'],
        ['IGKC','NPC2'],
        ['IGKC','NUP210'],
        ['IGKC','PARM1'],
        ['IGKC','PARP1'],
        ['IGKC','PERP'],
        ['IGKC','PGM1'],
        ['IGKC','PHF3'],
        ['IGKC','PKMYT1'],
        ['IGKC','PLCH2'],
        ['IGKC','PPP4R1'],
        ['IGKC','PRRX1'],
        ['IGKC','PTGFRN'],
        ['IGKC','RAB7A'],
        ['IGKC','RHOA'],
        ['IGKC','RNF213'],
        ['IGKC','ROBO1'],
        ['IGKC','RPL35A'],
        ['IGKC','S100A11'],
        ['IGKC','SFRP4'],
        ['IGKC','SH3PXD2A'],
        ['IGKC','SLC6A17'],
        ['IGKC','SMARCC1'],
        ['IGKC','SUN1'],
        ['IGKC','TPD52'],
        ['IGKC','WARS'],
        ['IGKC','WASF2'],
        ['IGKV3-11','FLNA'],
        ['IGSF3','CD58'],
        ['IGSF8','COL3A1'],
        ['IKBKB','FADD'],
        ['IKZF1','LRBA'],
        ['IL18BP','TG'],
        ['IL1RAPL1','KERA'],
        ['IL2RB','IGKC'],
        ['IL32','CRYAB'],
        ['ILF3','QTRT1'],
        ['ILF3','SLC44A2'],
        ['ILKAP','PER2'],
        ['ILVBL','SLC1A6'],
        ['ILVBL','SYDE1'],
        ['IMMP2L','SMC5'],
        ['INADL','HSD52'],
        ['INADL','ST6GALNAC5'],
        ['INHA','STK11IP'],
        ['INO80D','MYH11'],
        ['INPP4A','EEF1E1'],
        ['INPP5F','GRK5'],
        ['INTS4L1','INTS4'],
        ['INTS9','PHYHIP'],
        ['IPO11','TNFAIP8'],
        ['IPO8','CDH13'],
        ['IQCB1','CACNG4'],
        ['IQCH','CTNND2'],
        ['IRAK2','BRK1'],
        ['IRAK3','DYRK2'],
        ['IRAK3','GS1-204I12.4'],
        ['IREB2','DIAPH3'],
        ['IRF6','ACTB'],
        ['IRF6','C1ORF21'],
        ['ISL2','PEAK1'],
        ['IST1','AC004158.3'],
        ['IST1','TG'],
        ['ITCH','AHCY'],
        ['ITCH','DEFB118'],
        ['ITCH','HM13'],
        ['ITFG2','TULP3'],
        ['ITGA7','MYH11'],
        ['ITGA7','MYRFL'],
        ['ITGB1','MYH11'],
        ['ITGB4','RPS19'],
        ['ITGB5','PARP14'],
        ['ITGB5','S100A9'],
        ['ITGB5','WAC-AS1'],
        ['ITGB8','IGKC'],
        ['ITM2B','RB1'],
        ['ITPK1','BTBD7'],
        ['ITPK1','IGKC'],
        ['ITPKB','ZC3HAV1'],
        ['ITPKC','RAB4B'],
        ['ITPR1','SUMF1'],
        ['JAK2','RIC1'],
        ['JAK3','INSL3'],
        ['JAZF1','CPT1C'],
        ['JAZF1','EVX1'],
        ['JDP2','FLVCR2'],
        ['JMJD1C-AS1','REEP3'],
        ['JUP','LEPREL4'],
        ['KALRN','OPA1-AS1'],
        ['KANK2','MYH11'],
        ['KANSL1','ARL17A'],
        ['KAT6B','ADK'],
        ['KATNBL1','TNPO3'],
        ['KAZN','ST3GAL3'],
        ['KB-1125A3.11','AGT'],
        ['KB-1507C5.2','ATP6V1C1'],
        ['KB-1572G7.2','AP000347.4'],
        ['KC6','LAMA3'],
        ['KCNAB2','CTSB'],
        ['KCNIP2','MGEA5'],
        ['KCNK5','SAYSD1'],
        ['KCNN3','YEATS4'],
        ['KCNQ5','DST'],
        ['KCTD20','JARID2'],
        ['KDELR2','CTSB'],
        ['KDELR2','IGHG1'],
        ['KDELR2','IGKC'],
        ['KDM2A','RHOD'],
        ['KDM2A','RPS6KB2'],
        ['KDM2A','SYT12'],
        ['KDM2B','COL1A1'],
        ['KDM2B','LRRIQ1'],
        ['KDM4A','PTCH2'],
        ['KDM4B','AKAP8L'],
        ['KDM4B','UHRF1'],
        ['KDM5A','NINJ2'],
        ['KDM6A','KAL1'],
        ['KDM6B','TMEM95'],
        ['KIAA0100','ALKBH5'],
        ['KIAA0319L','PSMB2'],
        ['KIAA0319L','YTHDF2'],
        ['KIAA0907','ACBD3'],
        ['KIAA0930','CD74'],
        ['KIAA1109','GPR125'],
        ['KIAA1217','COL1A1'],
        ['KIAA1217','TG'],
        ['KIAA1244','ACPP'],
        ['KIAA1324','ACPP'],
        ['KIAA1324','SFTPB'],
        ['KIAA1324','TGM4'],
        ['KIAA1429','CAMK2A'],
        ['KIAA1468','NFATC1'],
        ['KIAA1598','EMX2OS'],
        ['KIAA1614','F11R'],
        ['KIAA1671','CRYBB2'],
        ['KIAA1671','CRYBB3'],
        ['KIAA1671','KRT13'],
        ['KIAA1755','HS6ST2'],
        ['KIDINS220','C3ORF70'],
        ['KIF13A','GFAP'],
        ['KIF1B','UBE4B'],
        ['KIF1C','ACPP'],
        ['KIF5A','PIP4K2C'],
        ['KIN','ITIH5'],
        ['KIRREL','PLBD1-AS1'],
        ['KLC1','MARK3'],
        ['KLC4','RPL7L1'],
        ['KLF10','PSAP'],
        ['KLF3','SCFD2'],
        ['KLF9','PTAR1'],
        ['KLHL18','TEX264'],
        ['KLHL8','CD24'],
        ['KLK2','MYH11'],
        ['KLK3','B2M'],
        ['KLK3','CDH1'],
        ['KLK3','FCGBP'],
        ['KLK3','MSMB'],
        ['KLK3','PAPSS1'],
        ['KLK3','PRKCSH'],
        ['KLK3','PTPRN2'],
        ['KLK3','RPL38'],
        ['KLK3','SF1'],
        ['KLK3','SLC4A4'],
        ['KLK3','SMARCC2'],
        ['KLK3','SRPK2'],
        ['KLK3','STT3B'],
        ['KLK3','TMPRSS2'],
        ['KLK3','TSC22D3'],
        ['KLK4','BCAM'],
        ['KLK4','KLKP1'],
        ['KMT2C','CALN1'],
        ['KMT2E','LHFPL3'],
        ['KNTC1','BTBD11'],
        ['KPNA6','TMEM39B'],
        ['KRIT1','LRRD1'],
        ['KRR1','CDK17'],
        ['KRT13','ATP2C1'],
        ['KRT13','GRN'],
        ['KRT13','KRT1'],
        ['KRT13','KRT3'],
        ['KRT13','KRT4'],
        ['KRT13','KRT5'],
        ['KRT13','KRT6A'],
        ['KRT13','KRT6B'],
        ['KRT13','KRT7'],
        ['KRT13','LMNA'],
        ['KRT13','PPP1R14C'],
        ['KRT14','AGRN'],
        ['KRT14','DST'],
        ['KRT14','EGR1'],
        ['KRT14','IFI6'],
        ['KRT14','KRT1'],
        ['KRT14','KRT4'],
        ['KRT14','KRT5'],
        ['KRT14','KRT6A'],
        ['KRT14','MSN'],
        ['KRT14','MYH9'],
        ['KRT14','SGSH'],
        ['KRT15','BCL2L1'],
        ['KRT16','GSN'],
        ['KRT16','KRT1'],
        ['KRT16','KRT3'],
        ['KRT16','KRT5'],
        ['KRT16','MYH9'],
        ['KRT16','S100A8'],
        ['KRT17','KRT5'],
        ['KRT17','KRT6A'],
        ['KRT17','LAMB3'],
        ['KRT17P5','KRT42P'],
        ['KRT17','S100A9'],
        ['KRT17','TAGLN2'],
        ['KRT17','TGFBI'],
        ['KRT18P59','PKNOX2'],
        ['KRT1','ATP5G2'],
        ['KRT1','FUS'],
        ['KRT1','KRT10'],
        ['KRT1','KRT14'],
        ['KRT3','KRT13'],
        ['KRT42P','KRT17P5'],
        ['KRT4','KRT13'],
        ['KRT4','KRT78'],
        ['KRT5','ARHGAP35'],
        ['KRT5','ASPH'],
        ['KRT5','BAIAP2'],
        ['KRT5','DLG5'],
        ['KRT5','EGFR'],
        ['KRT5','H2AFY'],
        ['KRT5','HSPB1'],
        ['KRT5','IGHG2'],
        ['KRT5','IGLL5'],
        ['KRT5','KRT13'],
        ['KRT5','KRT14'],
        ['KRT5','KRT15'],
        ['KRT5','KRT17'],
        ['KRT5','KRT80'],
        ['KRT5','MED15'],
        ['KRT5','MYH9'],
        ['KRT5','NDRG1'],
        ['KRT5','SGK1'],
        ['KRT5','SRSF5'],
        ['KRT5','TAF15'],
        ['KRT5','VIM'],
        ['KRT5','VIM-AS1'],
        ['KRT5','WDR13'],
        ['KRT6A','ANXA5'],
        ['KRT6A','COL1A1'],
        ['KRT6A','COL6A2'],
        ['KRT6A','CTDSP2'],
        ['KRT6A','DES'],
        ['KRT6A','KRT17'],
        ['KRT6A','RPL8'],
        ['KRT6A','SBSN'],
        ['KRT6A','TNFRSF12A'],
        ['KRT6A','YWHAE'],
        ['KRT6B','KRT17'],
        ['KRT6C','KRT13'],
        ['KRT7','FN1'],
        ['KRT7','LMNA'],
        ['KRT80','KRT5'],
        ['KRT8','LYZ'],
        ['KRT8','SFTPB'],
        ['L2HGDH','LMBRD1'],
        ['LA16C-306E5.2','CLUAP1'],
        ['LAD1','COL3A1'],
        ['LAD1','PKP1'],
        ['LAMA2','UQCRB'],
        ['LAMA3','KRT14'],
        ['LAMA5','CABLES2'],
        ['LAMA5','CTSB'],
        ['LAMA5','UBN1'],
        ['LAMC1','CD24'],
        ['LAMC1','KRT5'],
        ['LAMC1','SUSD4'],
        ['LAMC2','COL1A1'],
        ['LAMP1','CSPG4'],
        ['LAMP1','JAG2'],
        ['LAMP1','PTRF'],
        ['LAMP1','S100A11'],
        ['LAMP1','SORL1'],
        ['LAMP2','IGKC'],
        ['LAMTOR4','AP4M1'],
        ['LANCL2','14-Sep'],
        ['LANCL2','GFAP'],
        ['LANCL2','GS1-18A18.2'],
        ['LANCL2','PEX1'],
        ['LAPTM4B','SPINK1'],
        ['LARP1','CNOT8'],
        ['LARP4','MUCL1'],
        ['LARS2','LIMD1'],
        ['LARS','TG'],
        ['LASP1','ARHGAP23'],
        ['LASP1','ERBB2'],
        ['LATS1','LACE1'],
        ['LCAT','PSMB10'],
        ['LCLAT1','AC109642.1'],
        ['LCOR','CPNE8'],
        ['LDHB','ATP6V1A'],
        ['LDLRAD3','MOB2'],
        ['LDLR','HSH2D'],
        ['LDLR','KRT5'],
        ['LEPRE1','ERI3'],
        ['LEPROT','SGIP1'],
        ['LGALS13','AC005176.2'],
        ['LGALS17A','AC093063.2'],
        ['LGALS8','ERBB2'],
        ['LGMN','PMEL'],
        ['LGR5','TSPAN8'],
        ['LHPP','FANK1'],
        ['LHX6','NDUFA8'],
        ['LIFR','BCAS3'],
        ['LIFR','C9'],
        ['LIF','SEC61A1'],
        ['LILRP1','AC098784.1'],
        ['LILRP2','LILRB2'],
        ['LIMK2','OSBP2'],
        ['LIN7A','RPSAP52'],
        ['LINC00263','CALCOCO2'],
        ['LINC00265','ANLN'],
        ['LINC00317','AF241725.6'],
        ['LINC00482','SERINC2'],
        ['LINC00619','LINC00840'],
        ['LINC00620','FBLN2'],
        ['LINC00665','HKR1'],
        ['LINC00669','CGGBP1'],
        ['LINC00886','TIPARP-AS1'],
        ['LINC00993','TACC1P1'],
        ['LINC00994','PSMD6'],
        ['LINC01031','IL15'],
        ['LINC01158','LINC01114'],
        ['LINC01169','SMAD6'],
        ['LINC01198','TPT1-AS1'],
        ['LINC01252','ETV6'],
        ['LINC01279','CCDC80'],
        ['LINC01291','POLE4'],
        ['LINC01293','POLE4'],
        ['LINC01296','C2ORF27A'],
        ['LINC01305','SP9'],
        ['LINC01315','TCF20'],
        ['LINC01320','C1ORF186'],
        ['LINC01385','BTF3'],
        ['LINC01481','SMC2'],
        ['LINC01515','C10ORF11'],
        ['LINC01516','GPR158'],
        ['LINC-PINT','C1ORF61'],
        ['LINC-PINT','CCT3'],
        ['LINC-PINT','FMO3'],
        ['LIPE','CNFN'],
        ['LITAF','BCAR4'],
        ['LLGL2','CA10'],
        ['LLGL2','ERBB2'],
        ['LMAN2','F12'],
        ['LMAN2','SFTPB'],
        ['LMBR1','PTPRN2'],
        ['LMNA','ACPP'],
        ['LMNA','IGHG1'],
        ['LMNA','KRT5'],
        ['LMO1','RIC3'],
        ['LNPEP','CAST'],
        ['LONP1','C19ORF70'],
        ['LONP1','ELOF1'],
        ['LONP1','MAP2K7'],
        ['LONP1','ZNF780B'],
        ['LPCAT1','CLPTM1L'],
        ['LPIN1','MYH11'],
        ['LPP','MYH11'],
        ['LRAT','RBM46'],
        ['LRBA','SH3D19'],
        ['LRCH3','FYTTD1'],
        ['LRCH4','NCEH1'],
        ['LRIG3','R3HDM2'],
        ['LRIG3','SLC16A7'],
        ['LRP10','GPR56'],
        ['LRP12','CCL28'],
        ['LRP12','RALGPS1'],
        ['LRP12','TRAPPC9'],
        ['LRP1','CAND1'],
        ['LRP1','MARS'],
        ['LRP1','XPO5'],
        ['LRP5','SFTPB'],
        ['LRP8','MAGOH'],
        ['LRRC17','NFE4'],
        ['LRRC47','RBM3'],
        ['LRRC4','LTA4H'],
        ['LRRC59','B4GALNT2'],
        ['LRRC75A','APOL1'],
        ['LRRC75A-AS1','MYH11'],
        ['LRRC75A-AS1','SFTPB'],
        ['LRRC75A-AS1','SOD2'],
        ['LRRC75A','PDLIM5'],
        ['LRRC75A','SFTPB'],
        ['LRRC75A','TMBIM6'],
        ['LRRC75A','TMEM213'],
        ['LRRC8E','MFSD12'],
        ['LRRFIP1','KCNMB3'],
        ['LRTOMT','UVRAG'],
        ['LSP1','TNNT3'],
        ['LSR','CPM'],
        ['LTA4H','BTBD11'],
        ['LTBP2','TNFSF10'],
        ['LTBP4','MYO1B'],
        ['LTBP4','TG'],
        ['LTF','EGR1'],
        ['LUC7L3','C17ORF64'],
        ['LUM','ATP2C1'],
        ['LY6D','KRT6A'],
        ['LYG2','MITD1'],
        ['LYRM4','PHACTR1'],
        ['LYSMD1','SEMA6C'],
        ['LYZ','ACTB'],
        ['LYZ','AHNAK'],
        ['LYZ','ATP1A1'],
        ['LYZ','CANX'],
        ['LYZ','CAPRIN1'],
        ['LYZ','CMBL'],
        ['LYZ','CTSB'],
        ['LYZ','EIF4G1'],
        ['LYZ','P4HB'],
        ['MACROD1','BAD'],
        ['MAEA','TMEM71'],
        ['MAGEC3','MAGEC2'],
        ['MAGI3','MROH3P'],
        ['MAGIX','PLP2'],
        ['MAGOHB','KLRAP1'],
        ['MAL2','ARFGEF1'],
        ['MALAT1','KMT2C'],
        ['MALAT1','PRRC2C'],
        ['MALRD1','PLXDC2'],
        ['MAML2','CNTN5'],
        ['MAML2','GFAP'],
        ['MAN1A1','FMNL3'],
        ['MAN1A2','TTF2'],
        ['MAN2B2','TG'],
        ['MANBA','TG'],
        ['MANF','SETD2'],
        ['MAP2','CTSB'],
        ['MAP2K4','DNAH9'],
        ['MAP2K5','MYRFL'],
        ['MAP3K11','KCNK7'],
        ['MAP3K13','KRT5'],
        ['MAP3K5','NKAIN2'],
        ['MAP3K8','MROH2B'],
        ['MAP4','CTSB'],
        ['MAP4','GFAP'],
        ['MAP4','IGKC'],
        ['MAP4','KIF9-AS1'],
        ['MAP7','ARG1'],
        ['MAPK1IP1L','ARF6'],
        ['MAPK1','KCNJ4'],
        ['MAPK7','RNF112'],
        ['MAPK8','ARFIP1'],
        ['MAPKAPK5','ACAD10'],
        ['MAPKAPK5','ADAM1A'],
        ['MAPT','BCAN'],
        ['MAPT','GPM6A'],
        ['MARCKSL1','IGHG1'],
        ['MARK2','PMEL'],
        ['MARK3','TDRD9'],
        ['MARS','MSRB3'],
        ['MAST2','LINC00505'],
        ['MAST4','NLN'],
        ['MATN2','IGF2'],
        ['MATN2','POP1'],
        ['MATR3','CTNNA1'],
        ['MAVS','PANK2'],
        ['MB21D2','FGF12'],
        ['MBD1','CCDC11'],
        ['MBD2','DOCK2'],
        ['MBNL1','SFTPB'],
        ['MBP','ASTN1'],
        ['MBP','CRYAB'],
        ['MBP','SERINC1'],
        ['MBTD1','EFCAB3'],
        ['MBTD1','PART1'],
        ['MCCC1','DCUN1D1'],
        ['MCCC2','NUMA1'],
        ['MCC','PSMD6'],
        ['MCFD2','TG'],
        ['MCFD2','TTC7A'],
        ['MCL1','COL4A2'],
        ['MCM3AP','LINC00315'],
        ['MCM4','IGKC'],
        ['MCM5','HMOX1'],
        ['MCRS1','SRRM4'],
        ['MCU','CDH23'],
        ['MDM1','NAV3'],
        ['MDM1','RBMS2'],
        ['MDM2','AC079354.1'],
        ['MDM2','ATF7'],
        ['MDM2','CAND2'],
        ['MDM2','COPA'],
        ['MDM2','CPM'],
        ['MDM2','EGFR'],
        ['MDM2','FMO6P'],
        ['MDM2','LAMC1'],
        ['MDM2','LINC00615'],
        ['MDM2','MPHOSPH9'],
        ['MDM2','PDZRN4'],
        ['MDM2','PPM1H'],
        ['MDM2','RAP1B'],
        ['MDM2','SLAMF9'],
        ['MDM2','SYK'],
        ['MDM2','TSPAN31'],
        ['MDM4','TTC7B'],
        ['MEA1','LRRC1'],
        ['MECP2','PMEL'],
        ['MED12L','TSHZ2'],
        ['MED13','BRIP1'],
        ['MED13','DDX42'],
        ['MED13','VMP1'],
        ['MED15','PI4KA'],
        ['MED15','PMEL'],
        ['MED15','TIMP3'],
        ['MED16','COL1A1'],
        ['MED1','ACTB'],
        ['MED1','GDPD1'],
        ['MED1','KRT222'],
        ['MED1','STAC2'],
        ['MED1','TMPRSS9'],
        ['MED20','RNF182'],
        ['MED20','TLL2'],
        ['MED24','IKZF3'],
        ['MED24','TG'],
        ['MED25','AP2A1'],
        ['MED31','DNM3OS'],
        ['MED31','KIAA0753'],
        ['MEF2A','WDR61'],
        ['MEF2D','C1ORF61'],
        ['MEF2D','CSPG4'],
        ['MEGF8','PRR19'],
        ['MEGF9','GFAP'],
        ['MEGF9','SFTPB'],
        ['MELK','ZMIZ1-AS1'],
        ['MEMO1P1','RPS9P1'],
        ['MET','APP'],
        ['MET','C3'],
        ['MET','CAPZA2'],
        ['MET','CAV1'],
        ['MET','TFG'],
        ['MET','TG'],
        ['METTL12','DPF2'],
        ['METTL16','MNT'],
        ['METTL16','VPS53'],
        ['METTL21B','INHBE'],
        ['METTL21B','METTL1'],
        ['METTL23','MFSD11'],
        ['METTL24','SLC22A16'],
        ['METTL25','GALNT9'],
        ['METTL25','GTPBP10'],
        ['METTL2B','TSPAN31'],
        ['MFI2','AC011322.1'],
        ['MFSD12','PMEL'],
        ['MFSD12','RPL36'],
        ['MFSD3','LRRC14'],
        ['MGP','APP'],
        ['MGP','CD24'],
        ['MGP','IGFBP5'],
        ['MGP','MYH9'],
        ['MGP','PPP1R12A'],
        ['MGST1','ERBB2'],
        ['MGST1','KRT7'],
        ['MGST3','SLAMF7'],
        ['MGST3','UCK2'],
        ['MICAL2','NELL1'],
        ['MICAL2','TG'],
        ['MICALL1','POLR2F'],
        ['MINOS1','DENND2C'],
        ['MINPP1','LIPJ'],
        ['MIPOL1','SKIL'],
        ['MIR205HG','C9ORF3'],
        ['MIRLET7BHG','ATXN10'],
        ['MIRLET7BHG','COL4A1'],
        ['MITF','MTERF3'],
        ['MKI67','NIM1K'],
        ['MKL2','SNX29'],
        ['MKLN1','DNM3'],
        ['MKLN1','NRF1'],
        ['MLANA','AP1S2'],
        ['MLANA','LGALS3BP'],
        ['MLANA','MAP4'],
        ['MLANA','PMEL'],
        ['MLANA','SERPINF1'],
        ['MLXIP','ASH2L'],
        ['MLXIP','TSPAN8'],
        ['MMACHC','IGHG1'],
        ['MMP14','CRYAB'],
        ['MMP14','SPARC'],
        ['MMP17','NOC4L'],
        ['MMP25','IL32'],
        ['MMP2','IGKC'],
        ['MMP7','CD24'],
        ['MMS19','EXOSC1'],
        ['MND1','ARFIP1'],
        ['MOB2','KIAA1549L'],
        ['MOB3B','EQTN'],
        ['MOK','MUCL1'],
        ['MON2','TBC1D15'],
        ['MORF4L2-AS1','TMEM31'],
        ['MOV10','ERBB2'],
        ['MPC2','ADCY10'],
        ['MPDZ','NFIB'],
        ['MPHOSPH9','ELK3'],
        ['MPHOSPH9','MDM2'],
        ['MPI','KRT6A'],
        ['MPP5','GPHN'],
        ['MPP5','RDH11'],
        ['MPRIP','CTB-41I6.2'],
        ['MPRIP','TNFRSF13B'],
        ['MPZ','IGHG1'],
        ['MPZL1','MPC2'],
        ['MPZL1','TNFSF4'],
        ['MREG','SPAG17'],
        ['MROH1','ZNF623'],
        ['MROH2B','MRPS30'],
        ['MROH9','FMO3'],
        ['MRPL11','ST8SIA1'],
        ['MRPL48','GDPD5'],
        ['MRPL50','ERP44'],
        ['MRPS16','TTC18'],
        ['MRPS35','CCDC91'],
        ['MRPS6','AP000318.2'],
        ['MRVI1','MYH11'],
        ['MSH2','TTC7A'],
        ['MSI1','KRT78'],
        ['MSI2','PCTP'],
        ['MSI2','WIPF2'],
        ['MSL2','STAG1'],
        ['MSLN','AKR1C1'],
        ['MSLN','EEF2'],
        ['MSMB','MYH11'],
        ['MSN','IGKC'],
        ['MT1E','MT3'],
        ['MT1F','MT3'],
        ['MT2A','MT3'],
        ['MT3','IGFBP7'],
        ['MT3','MT1F'],
        ['MT3','MT2A'],
        ['MTA1','PACS2'],
        ['MTDH','MATN2'],
        ['MTDH','SFTPB'],
        ['MTERF3','LINC01111'],
        ['MTF2','BRDT'],
        ['MTF2','FNBP1L'],
        ['MTF2','RPL5'],
        ['MTG1','SCART1'],
        ['MTL5','CPT1A'],
        ['MTL5','PPFIA1'],
        ['MTOR','ZMYM4-AS1'],
        ['MTURN','NDRG2'],
        ['MTURN','ZNRF2'],
        ['MUC16','DDOST'],
        ['MUC1','GBA'],
        ['MUC20','ATP1B1'],
        ['MUC20','RHCG'],
        ['MUC20','TBC1D14'],
        ['MUC5AC','SFTPB'],
        ['MUT','SUPT3H'],
        ['MVB12B','MAMDC2'],
        ['MVB12B','TG'],
        ['MXD3','COL3A1'],
        ['MXRA7','SPARCL1'],
        ['MXRA8','CDK11A'],
        ['MXRA8','CDK11B'],
        ['MYBBP1A','ANKFY1'],
        ['MYBPC1','CHPT1'],
        ['MYH11','AP000688.14'],
        ['MYH11','AZGP1'],
        ['MYH11','CANX'],
        ['MYH11','EXOSC10'],
        ['MYH11','FBLIM1'],
        ['MYH11','FILIP1L'],
        ['MYH11','GOLPH3'],
        ['MYH11','GPR157'],
        ['MYH11','HDGFRP2'],
        ['MYH11','KANK2'],
        ['MYH11','MACF1'],
        ['MYH11','MINOS1-NBL1'],
        ['MYH11','NAV2'],
        ['MYH11','NEFH'],
        ['MYH11','PDE4D'],
        ['MYH11','PDLIM5'],
        ['MYH11','PPP1R12B'],
        ['MYH11','PTRF'],
        ['MYH11','ST5'],
        ['MYH11','TMPRSS2'],
        ['MYH11','ZAK'],
        ['MYH14','CTSB'],
        ['MYH9','6-Mar'],
        ['MYH9','APOD'],
        ['MYH9','CACNG2'],
        ['MYH9','CCNDBP1'],
        ['MYH9','GPNMB'],
        ['MYH9','IGHG1'],
        ['MYH9','KLK3'],
        ['MYH9','KRT14'],
        ['MYH9','KRT17'],
        ['MYH9','NFYC'],
        ['MYH9','ORAOV1'],
        ['MYH9','TXN2'],
        ['MYH9','USP5'],
        ['MYL12B','LINC00470'],
        ['MYL9','IGKC'],
        ['MYLK','MACF1'],
        ['MYLK','MYH11'],
        ['MYLK','NXPH1'],
        ['MYLK','TNS1'],
        ['MYLK','ZC3H14'],
        ['MYO10','AC091878.1'],
        ['MYO10','FAM134B'],
        ['MYO10','SARNP'],
        ['MYO1B','COL3A1'],
        ['MYO1C','PMEL'],
        ['MYO1D','FAIM3'],
        ['MYO1D','KLK3'],
        ['MYO1D','SSH2'],
        ['MYO9A','CELF6'],
        ['NAALADL2','PEX5L'],
        ['NAALADL2','VPS13C'],
        ['NADK','KIAA1751'],
        ['NADSYN1','KRTAP5-10'],
        ['NADSYN1','KRTAP5-8'],
        ['NAP1L1','PHLDA1'],
        ['NAP1L1','PTPRR'],
        ['NAP1L4','RPAP1'],
        ['NAPSA','ENO1'],
        ['NAPSA','PIGR'],
        ['NARF','OLFML2B'],
        ['NARF','USP36'],
        ['NASP','AC022182.2'],
        ['NAT1','NATP'],
        ['NAT8L','DBNDD1'],
        ['NAV1','LAMP1'],
        ['NAV2','CTSB'],
        ['NAV3','BRD4'],
        ['NAV3','DCN'],
        ['NAV3','DOK3'],
        ['NAV3','EVX1'],
        ['NAV3','SLC6A15'],
        ['NBEAL1','MLLT3'],
        ['NBEA','NDFIP2'],
        ['NBR1','FAM213A'],
        ['NCAPD2','VWF'],
        ['NCCRP1','RRP12'],
        ['NCEH1','TNFSF10'],
        ['NCL','APP'],
        ['NCL','OS9'],
        ['NCOA2','ST18'],
        ['NCOA4','CTSB'],
        ['NCOA4','PARPG1'],
        ['NCOR2','KRT13'],
        ['NCSTN','NHLH1'],
        ['NDE1','KLK3'],
        ['NDE1','TGM4'],
        ['NDFIP2','LINC01068'],
        ['NDRG1','SNTB1'],
        ['NDRG3','MYH11'],
        ['NDRG4','KRT13'],
        ['NDRG4','SETD6'],
        ['NDUFA11','ZNF846'],
        ['NDUFA12','ST6GALNAC6'],
        ['NDUFA3','TG'],
        ['NDUFA7','TG'],
        ['NDUFA9','C12ORF4'],
        ['NDUFB3','GS1-164F24.1'],
        ['NDUFB9','CSMD3'],
        ['NDUFB9','FN1'],
        ['NDUFS1','GFAP'],
        ['NDUFS2','SLC24A2'],
        ['NDUFS4','HMGCR'],
        ['NDUFV3','PKNOX1'],
        ['NEAT1','LRBA'],
        ['NEBL','FBXO11'],
        ['NEDD1','TMTC2'],
        ['NEDD4','GABRA5'],
        ['NEFH','ACPP'],
        ['NEK9','TMED10'],
        ['NELL2','OS9'],
        ['NET1','ADARB2-AS1'],
        ['NETO2','CHD9'],
        ['NF1','AP002381.2'],
        ['NF1','ASIC2'],
        ['NF1','CDK5RAP3'],
        ['NF1','NLE1'],
        ['NF1','RAB11FIP4'],
        ['NF2','CLTCL1'],
        ['NFASC','NTRK1'],
        ['NFASC','PRELP'],
        ['NFASC','SOX13'],
        ['NFAT5','DENND2C'],
        ['NFE2L2','PDE11A'],
        ['NFE2L2','TG'],
        ['NFE2L2','ZC3H13'],
        ['NFIA','LTV1'],
        ['NFIX','CALR'],
        ['NFIX','DAND5'],
        ['NFIX','FARSA-AS1'],
        ['NFIX','GATAD2A'],
        ['NFIX','NACC1'],
        ['NFX1','UBE2R2'],
        ['NHP2L1','LLPH'],
        ['NHS','F11R'],
        ['NHSL1','CDIP1'],
        ['NID1','SERPINE2'],
        ['NIN','PMEL'],
        ['NIPBL','CLSPN'],
        ['NIPBL','SPEF2'],
        ['NLGN4Y','TXLNGY'],
        ['NME1-NME2','CEP112'],
        ['NOL4L','COMMD7'],
        ['NOP9','LTB4R'],
        ['NOS1AP','MROH9'],
        ['NOTCH1','IGKC'],
        ['NOTCH1','SEC16A'],
        ['NOTCH2','ADAM30'],
        ['NOTCH2','PMEL'],
        ['NPC1','FGD4'],
        ['NPC2','PGC'],
        ['NPC2','TG'],
        ['NPDC1','COL5A1'],
        ['NPEPL1','EVI5'],
        ['NPEPPS','CH17-129M10.1'],
        ['NPEPPS','TBC1D3'],
        ['NPEPPS','TBC1D3B'],
        ['NPEPPS','TBC1D3C'],
        ['NPEPPS','TBC1D3G'],
        ['NPEPPS','TBC1D3H'],
        ['NPIPP1','PDXDC2P'],
        ['NPL','DHX9'],
        ['NPLOC4','P4HB'],
        ['NPTN','CLUAP1'],
        ['NPTN','HMG20A'],
        ['NPTXR','DNAL4'],
        ['NQO2','PMEL'],
        ['NR3C1','HMHB1'],
        ['NRCAM','SHFM1'],
        ['NRL','DHRS4-AS1'],
        ['NRP2','PARD3B'],
        ['NRSN2-AS1','C20ORF96'],
        ['NSA2','HEXB'],
        ['NSD1','ZNF346'],
        ['NT5DC3','PTPRR'],
        ['NTN4','KIFAP3'],
        ['NTN4','NUP107'],
        ['NTN4','RAP1B'],
        ['NTN4','TRHDE'],
        ['NTRK1','PEAR1'],
        ['NUAK1','COL1A1'],
        ['NUB1','TG'],
        ['NUBP2','MAPK8IP3'],
        ['NUCKS1','SFTPB'],
        ['NUDCD3','IGKC'],
        ['NUDT2','GLIPR2'],
        ['NUDT3','CTSB'],
        ['NUDT9','FAM13A'],
        ['NUMA1','AATF'],
        ['NUMA1','ITPR3'],
        ['NUMA1','MSMB'],
        ['NUMA1','MYH11'],
        ['NUMA1','PMEL'],
        ['NUMA1','TGM4'],
        ['NUP107','CYP27B1'],
        ['NUP107','PAAF1'],
        ['NUP153','RNF2'],
        ['NUP155','ZFYVE26'],
        ['NUP210','FBLN2'],
        ['NUP210','IGKC'],
        ['NXN','IGKC'],
        ['NXPH4','SHMT2'],
        ['NXT1','VPS16'],
        ['OAF','POU2F3'],
        ['OBSL1','TG'],
        ['OGDH','CA12'],
        ['OLFM4','CEACAM5'],
        ['OLFM4','TKT'],
        ['OPA1','PPP2R3A'],
        ['OR2H1','BPIFB1'],
        ['OR51E2','SLC44A4'],
        ['OR7A5','EMR2'],
        ['OR9G1','OR9G3P'],
        ['ORMDL2','ATP5B'],
        ['ORMDL3','GSDMB'],
        ['OS9','CD48'],
        ['OS9','CEP85L'],
        ['OS9','COLEC12'],
        ['OS9','KIF5A'],
        ['OS9','NCL'],
        ['OS9','TSPAN31'],
        ['OSBPL2','EDEM2'],
        ['OSBPL2','HM13'],
        ['OSBPL8','BBS10'],
        ['OSBPL9','SERINC5'],
        ['OSCP1','SOX2-OT'],
        ['OSGIN1','NECAB2'],
        ['OTOGL','HMGA2'],
        ['OTOS','MYEOV2'],
        ['OTUD7B','CTSS'],
        ['OTUD7B','HORMAD1'],
        ['OXR1','TG'],
        ['P2RX1','CAMKK1'],
        ['P2RY6','ARHGEF17'],
        ['P4HB','ERBB2'],
        ['P4HB','IFI6'],
        ['P4HB','IYD'],
        ['P4HB','MYH11'],
        ['P4HB','RAB3D'],
        ['PA2G4','MARK1'],
        ['PABPC1','MYH11'],
        ['PABPC1','ZFAND6'],
        ['PACS1','TSGA10IP'],
        ['PACS2','RAD51B'],
        ['PACSIN2','SFTPB'],
        ['PACSIN2','TTLL1'],
        ['PACSIN2','ZNF823'],
        ['PADI2','ATP13A2'],
        ['PAPD5','SHANK2'],
        ['PAPLN','TG'],
        ['PAPPA','SPARC'],
        ['PARD6B','BCAS4'],
        ['PARL','MAP6D1'],
        ['PARM1','SFTPB'],
        ['PATL1','TECTA'],
        ['PAX8','NFE2L2'],
        ['PBRM1','NT5DC2'],
        ['PBRM1','TKT'],
        ['PBX1','CTNNA3'],
        ['PBX1','DISC1'],
        ['PBX1','IGF2'],
        ['PCAT18','KCTD1'],
        ['PCAT1','CASC21'],
        ['PCBP1-AS1','ERBB2'],
        ['PCBP1-AS1','TGM4'],
        ['PCBP2','AMHR2'],
        ['PCCB','TG'],
        ['PCDHB16','CH17-140K24.8'],
        ['PCGF2','CTB-58E17.9'],
        ['PCMTD1','ST18'],
        ['PCNXL2','PTPN14'],
        ['PCSK6','GFAP'],
        ['PCSK7','LMOD1'],
        ['PDCD6','FRS2'],
        ['PDCD7','UBAP1L'],
        ['PDE10A','ZNF670'],
        ['PDE4B','MOB3C'],
        ['PDE4D','DEPDC1B'],
        ['PDE4D','LINC01511'],
        ['PDE5A','CASC14'],
        ['PDE7A','TRIM55'],
        ['PDE8A','SLCO3A1'],
        ['PDGFRA','COL3A1'],
        ['PDHA1','VCP'],
        ['PDIA4','AC060834.3'],
        ['PDIA4','CSRP1'],
        ['PDIA6','PDIA4'],
        ['PDK4','SOD2'],
        ['PDLIM5','ACPP'],
        ['PDLIM5','FASN'],
        ['PDLIM5','MYH11'],
        ['PDLIM5','SLC39A6'],
        ['PDPK1','ERVK13-1'],
        ['PDXK','COL1A1'],
        ['PDZD8','ENO4'],
        ['PDZRN4','CPM'],
        ['PDZRN4','MDM2'],
        ['PEAK1','PSTPIP1'],
        ['PEAK1','TSPAN3'],
        ['PEBP1','CTSB'],
        ['PEBP4','SFTPB'],
        ['PECAM1','TG'],
        ['PELP1','CHRNE'],
        ['PERP','COL1A2'],
        ['PERP','NGRN'],
        ['PERP','SPAG9'],
        ['PES1','GAL3ST1'],
        ['PFKFB3','ERBB2'],
        ['PFKFB4','SHISA5'],
        ['PGAM5','INTS6'],
        ['PGC','HDLBP'],
        ['PGD','CAMTA1'],
        ['PGD','SFTPB'],
        ['PGM1','COL1A1'],
        ['PGS1','AC061992.2'],
        ['PGS1','BIRC5'],
        ['PGS1','KIF22'],
        ['PHACTR2','PLAGL1'],
        ['PHC3','FNDC3B'],
        ['PHF12','OR2AM1P'],
        ['PHF12','PLXDC1'],
        ['PHGDH','GAPDH'],
        ['PHGDH','HMGCS2'],
        ['PHGDH','LINC00869'],
        ['PHKB','SCN8A'],
        ['PHYHIPL','LINC00844'],
        ['PI3','KRT14'],
        ['PI3','KRT17'],
        ['PI3','S100A8'],
        ['PI4K2A','ZDHHC16'],
        ['PIAS2','ST8SIA5'],
        ['PIAS4','GLYATL2'],
        ['PICK1','POLR2F'],
        ['PIEZO1','KRT14'],
        ['PIGC','KERA'],
        ['PIGL','TBC1D26'],
        ['PIGR','ACPP'],
        ['PIGR','ACTN4'],
        ['PIGR','C5ORF45'],
        ['PIGR','CD82'],
        ['PIGR','CTSB'],
        ['PIGR','FN1'],
        ['PIGR','GDI2'],
        ['PIGR','IGKC'],
        ['PIGR','LIN52'],
        ['PIGR','LTF'],
        ['PIGR','MYH9'],
        ['PIGR','OLFM4'],
        ['PIGR','SF3B1'],
        ['PIGX','PAK2'],
        ['PIK3C3','TRPC6'],
        ['PIK3IP1-AS1','LIMK2'],
        ['PIP5K1A','MLLT11'],
        ['PIP5K1C','CTB-131K11.1'],
        ['PIP5K1C','IGKC'],
        ['PIP5K1C','MATK'],
        ['PITPNA','NXN'],
        ['PITPNC1','BPTF'],
        ['PITPNM2','ARHGEF25'],
        ['PKD1','NTHL1'],
        ['PKHD1L1','EBAG9'],
        ['PKN1','IGF2'],
        ['PKP1','ALDH3A1'],
        ['PLA2G12A','CASP6'],
        ['PLAT','MYH9'],
        ['PLBD1','CREBL2'],
        ['PLCG2','SLC26A7'],
        ['PLCH2','KRT5'],
        ['PLD3','ACTN4'],
        ['PLD3','HIPK4'],
        ['PLEKHA3','AC009948.5'],
        ['PLEKHA3','PLEKHA3P1'],
        ['PLEKHA5','CPM'],
        ['PLEKHA6','DTX3'],
        ['PLEKHA6','IGHG1'],
        ['PLEKHA6','LRRN2'],
        ['PLEKHF2','NDUFAF6'],
        ['PLEKHG4B','LRRC14B'],
        ['PLEKHG7','MRPL42'],
        ['PLEKHH2','DYNC2LI1'],
        ['PLEKHM2','SLC25A34'],
        ['PLLP','KRT5'],
        ['PLVAP','CANX'],
        ['PLVAP','HSPG2'],
        ['PLVAP','RGS5'],
        ['PLVAP','VWF'],
        ['PLXNA2','DUSP10'],
        ['PLXNB2','IGHM'],
        ['PLXNC1','CEP83'],
        ['PLXNC1','TXNRD1'],
        ['PMEL','ARPC2'],
        ['PMEL','C7ORF50'],
        ['PMEL','CABLES1'],
        ['PMEL','CD74'],
        ['PMEL','GNB1'],
        ['PMEL','INPP4B'],
        ['PMEL','LHFPL2'],
        ['PMEL','MBP'],
        ['PMEL','MYADM'],
        ['PMEL','MYO1C'],
        ['PMEL','MYO1D'],
        ['PMEL','NFE2L2'],
        ['PMEL','PABPC4'],
        ['PMEL','PLS3'],
        ['PMEL','RBM3'],
        ['PMEL','SQSTM1'],
        ['PMEL','SYNE2'],
        ['PMEL','TAPBP'],
        ['PMEL','TNS1'],
        ['PMEL','UBA52'],
        ['PMEPA1','PTGIS'],
        ['PMS2CL','AC073343.13'],
        ['PNCK','HCFC1'],
        ['PNPLA7','DPH7'],
        ['POC1B','DUSP6'],
        ['PODXL','CSPG4'],
        ['PODXL','TG'],
        ['POGZ','HHIPL2'],
        ['POLE','ANKS1B'],
        ['POLR1A','PMEL'],
        ['POLR2B','FIP1L1'],
        ['POLR2F','MICALL1'],
        ['POLR3C','NUDT17'],
        ['POLR3K','SNRNP25'],
        ['POSTN','WWTR1'],
        ['POU2AF1','COLCA1'],
        ['POU2F1','COPA'],
        ['PPAP2A','ACPP'],
        ['PPFIA1','FRMD8'],
        ['PPFIA1','HSD3B2'],
        ['PPFIA1','TSKU'],
        ['PPFIA2','HMGA2'],
        ['PPFIA2','OTOGL'],
        ['PPIEL','PPIE'],
        ['PPIE','PPIEL'],
        ['PPL','TG'],
        ['PPM1D','CCND1'],
        ['PPM1G','COL1A1'],
        ['PPM1H','IFNG-AS1'],
        ['PPM1H','MDM2'],
        ['PPM1H','RAP1B'],
        ['PPM1H','SRGAP1'],
        ['PPM1L','B3GNT5'],
        ['PPP1CB','BRE'],
        ['PPP1R12C','PPP6R1'],
        ['PPP1R13B','WIBG'],
        ['PPP1R1B','PIPOX'],
        ['PPP1R1B','STARD3'],
        ['PPP1R21','STON1'],
        ['PPP2CB','TG'],
        ['PPP2R1A','ACPP'],
        ['PPP2R2A','APP'],
        ['PPP2R2A','SMIM19'],
        ['PPP2R2B','GFAP'],
        ['PPP2R3A','PCCB'],
        ['PPP2R4','FLNA'],
        ['PPP2R4','NAP1L1'],
        ['PPP2R4','PMEL'],
        ['PPP2R4','TG'],
        ['PPP3R1','AC010987.5'],
        ['PPP3R1','AC010987.6'],
        ['PPP6R2','TRABD'],
        ['PPTC7','ARPC3'],
        ['PPWD1','NLN'],
        ['PQBP1','ERAS'],
        ['PRCC','MPTX1'],
        ['PRCP','RAB30'],
        ['PRDM11','ACCS'],
        ['PRDX6','IGKC'],
        ['PREP','PLN'],
        ['PRKAR1A','ZBTB20'],
        ['PRKAR2A','SPINK8'],
        ['PRKCA','PITPNC1'],
        ['PRKCH','FLJ22447'],
        ['PRKCSH','LAMA3'],
        ['PRKCSH','NEFH'],
        ['PRKCZ','TG'],
        ['PRKDC','IGF2'],
        ['PRKDC','TINAGL1'],
        ['PROM2','MYH9'],
        ['PRPF18','CACNB2'],
        ['PRPF39','FAM179B'],
        ['PRPF4B','F13A1'],
        ['PRR11','CDH1'],
        ['PRR11','RARA'],
        ['PRR11','SMG8'],
        ['PRR19','PPP1R37'],
        ['PRRC2B','SET'],
        ['PRRC2C','RSF1'],
        ['PRRG4','ZNF215'],
        ['PRSS23','IGKC'],
        ['PRTFDC1','GPR158-AS1'],
        ['PSAP','APP'],
        ['PSAP','PGC'],
        ['PSAP','PHC2'],
        ['PSAP','SOD2'],
        ['PSCA','EIF4G2'],
        ['PSD4','IL1RN'],
        ['PSD4','PAX8-AS1'],
        ['PSMA1','SERGEF'],
        ['PSMB2','FAM212B'],
        ['PSMB6','ENO3'],
        ['PSMD11','CDK12'],
        ['PSMD12','PITPNC1'],
        ['PSPC1','MPHOSPH8'],
        ['PTBP2','KAZN'],
        ['PTBP3','C9ORF147'],
        ['PTGES2','SLC25A25-AS1'],
        ['PTGFRN','NOTCH2'],
        ['PTK2','SLC45A4'],
        ['PTK2','TRAPPC9'],
        ['PTK2','ZFPM2-AS1'],
        ['PTK7','DLK2'],
        ['PTK7','SFTPB'],
        ['PTPLAD1','ANKDD1A'],
        ['PTPN11','EPYC'],
        ['PTPN11','MYRFL'],
        ['PTPN11','OAS1'],
        ['PTPN12','CCDC146'],
        ['PTPN12','PHTF2'],
        ['PTPN14','NSL1'],
        ['PTPN1','PREX1'],
        ['PTPN2','FTL'],
        ['PTPRA','N4BP1'],
        ['PTPRD','IGF2'],
        ['PTPRF','AZGP1'],
        ['PTPRF','EEF2'],
        ['PTPRF','IGF2'],
        ['PTPRG','SYNPR'],
        ['PTPRJ','TG'],
        ['PTPRK','AC007246.3'],
        ['PTPRK','IGKC'],
        ['PTPRM','TG'],
        ['PTPRR','NAV3'],
        ['PTPRR','SLC6A15'],
        ['PTPRR','SNRPF'],
        ['PTPRZ1','CCT3'],
        ['PTPRZ1','EPN2'],
        ['PTPRZ1','FYN'],
        ['PTPRZ1','GFAP'],
        ['PTPRZ1','GPR56'],
        ['PTPRZ1','SMARCA4'],
        ['PTRF','STAT3'],
        ['PTRH2','RNFT1'],
        ['PUM1','RPH3AL'],
        ['PVRL1','KRT17'],
        ['PVRL4','ARHGAP30'],
        ['PVRL4','PFDN2'],
        ['PVT1_1','CASC21'],
        ['PVT1','C19ORF57'],
        ['PVT1','CASC11'],
        ['PVT1','CASC21'],
        ['PVT1','CS'],
        ['PVT1','IGKC'],
        ['PVT1','MYC'],
        ['PVT1','NABP2'],
        ['PVT1','PCAT1'],
        ['PVT1','TPCN2'],
        ['PWP1','CCDC63'],
        ['PXDC1','AC010890.1'],
        ['PXDNL','USP8'],
        ['PXMP4','E2F1'],
        ['PYGB','ENTPD6'],
        ['PYGM','RBMS2'],
        ['QARS','USP19'],
        ['QKI','AKAP7'],
        ['QKI','GAS7'],
        ['QKI','SPARC'],
        ['R3HDM2','MARS'],
        ['R3HDM2','RBMS2'],
        ['R3HDM2','TPH2'],
        ['R3HDM2','TSFM'],
        ['R3HDM2','XRCC6BP1'],
        ['RAB11A','COL1A1'],
        ['RAB11FIP1','KCNU1'],
        ['RAB11FIP1P1','FTX_1'],
        ['RAB1A','IGKC'],
        ['RAB1C','RAB1B'],
        ['RAB21','DDOST'],
        ['RAB21','FRS2'],
        ['RAB21','MYBPC1'],
        ['RAB27A','ACPP'],
        ['RAB38','CA10'],
        ['RAB3GAP1','ACMSD'],
        ['RAB4B-EGLN2','CYP2B7P'],
        ['RAB4B-EGLN2','PLD3'],
        ['RAB5C','FAM134C'],
        ['RAB9A','OFD1'],
        ['RABEP1','ASPA'],
        ['RABEP1','PIGL'],
        ['RABEP1','RPAIN'],
        ['RABGAP1L','CACYBP'],
        ['RAD21','IGHG1'],
        ['RAD51AP1','CD27-AS1'],
        ['RAD51AP1','DYRK4'],
        ['RAD51B','SIVA1'],
        ['RAD54L2','ERC2'],
        ['RAD9A','IKZF3'],
        ['RAD9A','KDM2A'],
        ['RAF1','TRAK1'],
        ['RAI14','ASB4'],
        ['RALBP1','TWSG1'],
        ['RALGAPB','RPRD1B'],
        ['RALGPS1','MVB12B'],
        ['RALGPS1','TYRP1'],
        ['RALY','ASXL1'],
        ['RALY','CBFA2T2'],
        ['RANBP10','RRP15'],
        ['RANBP10','USP10'],
        ['RANBP17','KCNIP1'],
        ['RAP1B','CPM'],
        ['RAP1B','CTDSP2'],
        ['RAP1B','DPT'],
        ['RAP1B','SCYL2'],
        ['RARA-AS1','POLDIP2'],
        ['RARA','MYO1D'],
        ['RARA','NARS2'],
        ['RARA','PGAP3'],
        ['RARA','PRR11'],
        ['RARA','ZNF595'],
        ['RASA1','RAPGEF6'],
        ['RASSF3','STK10'],
        ['RASSF8','RERGL'],
        ['RB1','CNIH3'],
        ['RBAK-RBAKDN','PMS2CL'],
        ['RBBP4','KIAA1522'],
        ['RBBP7','NDUFAF4'],
        ['RBM17','PFKFB3'],
        ['RBM38','RAE1'],
        ['RBM38','SPO11'],
        ['RBM39','CRYZL1'],
        ['RBM3','GLG1'],
        ['RBM45','LINC01117'],
        ['RBMS2','LITAF'],
        ['RBMY2EP','PRORY'],
        ['RBMY2QP','RBMY2HP'],
        ['RBPMS2','OAZ2'],
        ['RCC1','TRNAU1AP'],
        ['RCC2','PADI4'],
        ['RCCD1','BLM'],
        ['RCOR1','TECPR2'],
        ['RCOR3','TG'],
        ['RCOR3','TRAF5'],
        ['RDH11','CASZ1'],
        ['RDH11','TGM4'],
        ['RECK','ACER3'],
        ['RECQL4','CPSF1'],
        ['RELT','TEX40'],
        ['RERE','ENO1'],
        ['RERE','OOEP'],
        ['REV3L','FYN'],
        ['REV3L','TRAF3IP2'],
        ['RFFL','NR1D1'],
        ['RFPL1','NEFH'],
        ['RFT1','DCP1A'],
        ['RFTN1','DAZL'],
        ['RFTN1P1','TTTY16'],
        ['RFX3','GLDC'],
        ['RFX7','LIPC'],
        ['RHBDD2','POR'],
        ['RHBDL2','GJA9-MYCBP'],
        ['RHEB','PRKAG2'],
        ['RHNO1','ITFG2'],
        ['RHOA','ACPP'],
        ['RHOA','COL4A2'],
        ['RHOA','NICN1'],
        ['RHOBTB2','SFTPA2'],
        ['RHOT1','FGFR1'],
        ['RIC8B','LINC01483'],
        ['RIC8B','LINC01533'],
        ['RIC8B','RFX4'],
        ['RIMKLB','ELP3'],
        ['RIPK1','SERPINB9'],
        ['RMI1','STX17'],
        ['RMND5A','AC125232.1'],
        ['RN7SL8P','LHFPL3'],
        ['RNF11','TMEM61'],
        ['RNF121','CYSLTR2'],
        ['RNF121','ESRRG'],
        ['RNF130','MGAT4B'],
        ['RNF13','TM4SF4'],
        ['RNF157','CA10'],
        ['RNF169','CBX1'],
        ['RNF186','TMCO4'],
        ['RNF19A','SLC41A3'],
        ['RNF19A','SNX31'],
        ['RNF20','TMEM246-AS1'],
        ['RNF213','CTSB'],
        ['RNF38','LCE5A'],
        ['RNF43','CACNA1G-AS1'],
        ['RNF44','IGF2'],
        ['RNGTT','AHI1'],
        ['RNGTT','BACH2'],
        ['RNH1','DEAF1'],
        ['ROBO1','IGKC'],
        ['RORA','CLSTN2'],
        ['ROS1','CLTC'],
        ['RP11-274B21.1','TLK2'],
        ['RP11-356O9.1','TTC6'],
        ['RP11-381K20.2','KLHL3'],
        ['RP11-386M24.4','H2AFV'],
        ['RP2','CCND2-AS1'],
        ['RP2','CCND2-AS2'],
        ['RP9','JAZF1-AS1'],
        ['RPA3OS','BEX5'],
        ['RPE','AC007038.7'],
        ['RPL13A','SFTPB'],
        ['RPL22','MECOM'],
        ['RPL24','ZBTB11'],
        ['RPL30','TG'],
        ['RPL36','MFSD12'],
        ['RPL36','TTYH1'],
        ['RPL37A','SMARCAL1'],
        ['RPL38','TTYH2'],
        ['RPL7L1','FOXP4'],
        ['RPL7L1','KLC4'],
        ['RPLP2','CTB-47B8.1'],
        ['RPLP2','TALDO1'],
        ['RPN1','TYRP1'],
        ['RPN2','ASPH'],
        ['RPN2','RDH10'],
        ['RPP14','PXK'],
        ['RPPH1','CCNB1IP1'],
        ['RPRD2','PRPF3'],
        ['RPRD2','SHC1'],
        ['RPRD2','VPS45'],
        ['RPS15A','ARL6IP1'],
        ['RPS15A','MYH11'],
        ['RPS19','CD79A'],
        ['RPS19','CEACAM6'],
        ['RPS19','FADS2'],
        ['RPS19','PMEL'],
        ['RPS3','TSKU'],
        ['RPS6KB1','VMP1'],
        ['RPS6KC1','CD34'],
        ['RPS6KC1','LYST'],
        ['RPS6KC1','OTX2-AS1'],
        ['RPS8','IGHG1'],
        ['RPSAP52','TSPAN8'],
        ['RPUSD1','MSLNL'],
        ['RQCD1','BCS1L'],
        ['RRBP1','TG'],
        ['RRM1','STIM1'],
        ['RRM2B','PMEL'],
        ['RRP12','EXOSC1'],
        ['RSF1','PRRC2C'],
        ['RSPH3','MIF4GD'],
        ['RSPH6A','DMWD'],
        ['RTCA-AS1','DBT'],
        ['RTFDC1','SFTPB'],
        ['RTN4','DTNB'],
        ['RTN4','SFTPB'],
        ['RUNX1','COL1A1'],
        ['RUNX1','IFNAR1'],
        ['RYBP','TADA3'],
        ['S100A11','ACTB'],
        ['S100A13','S100A16'],
        ['S100A13','S100A6'],
        ['S100A16','IGKC'],
        ['S100A2','LAMC1'],
        ['S100A6','CTSB'],
        ['S100A6','IGKC'],
        ['S100A9','HNRNPH1'],
        ['S100A9','KRT13'],
        ['S100A9','KRT17'],
        ['S100PBP','YARS'],
        ['SACM1L','LIMD1'],
        ['SAMD4B','SIPA1L3'],
        ['SAMD5','SASH1'],
        ['SAR1A','NPFFR1'],
        ['SAR1A','TG'],
        ['SARAF','CTSB'],
        ['SART1','CNTN5'],
        ['SART3','PAH'],
        ['SBF1','TENM4'],
        ['SBF2','MRVI1'],
        ['SBNO2','ARHGEF1'],
        ['SBNO2','TPGS1'],
        ['SCAF1','TRAPPC6A'],
        ['SCAI','PPP6C'],
        ['SCAMP2','SLC12A8'],
        ['SCARA3','COL1A1'],
        ['SCARB1','DHX37'],
        ['SCARB2','CTSB'],
        ['SCARB2','SLC30A4'],
        ['SCCPDH','CNST'],
        ['SCD5','NCAN'],
        ['SCD5','QKI'],
        ['SCD5','RAB6B'],
        ['SCD','GFAP'],
        ['SCD','GPR19'],
        ['SCD','HNRNPC'],
        ['SCIN','ARFGEF2'],
        ['SCIN','EEF2'],
        ['SCNN1A','PGC'],
        ['SCNN1G','COL27A1'],
        ['SCRN1','GFAP'],
        ['SCYL1','FRMD8'],
        ['SCYL1','MALAT1'],
        ['SCYL2','ANKS1B'],
        ['SDC1','MYH9'],
        ['SDC2','COL1A2'],
        ['SDC3','CTSD'],
        ['SDC3','IGF2'],
        ['SDF2L1','DEPDC5'],
        ['SEC11A','VPS33B'],
        ['SEC16A','ACPP'],
        ['SEC16A','QSOX2'],
        ['SEC16A','RXRA'],
        ['SEC22B','CH17-385C13.2'],
        ['SEC61A1','IGKC'],
        ['SEC61A1','SPP1'],
        ['SEC61G','EGFR'],
        ['SEC61G','EGFR-AS1'],
        ['SEC61G','GS1-18A18.2'],
        ['SEC62','RPL22L1'],
        ['SEL1L3','CSPG4'],
        ['SEL1L3','IGKC'],
        ['SEMA3C','CNTNAP2'],
        ['SEMA3D','RBM28'],
        ['SEMA3F','RBM6'],
        ['SEMA4C','ANKRD39'],
        ['SEMA4C','FAM178B'],
        ['SEMA4C','WNT5B'],
        ['SEMA5A','SLC6A19'],
        ['SEMA5A','TGIF2'],
        ['SEPT7-AS1','HERPUD2'],
        ['SERGEF','ANKRD29'],
        ['SERINC3','TG'],
        ['SERINC5','ACPP'],
        ['SERINC5','CDYL2'],
        ['SERINC5','TTC39A'],
        ['SERPINE2','APP'],
        ['SERPINE2','ATP1A1'],
        ['SERPINE2','COL3A1'],
        ['SERPINE2','CSDE1'],
        ['SERPINE2','INPP5F'],
        ['SERPINE2','MAP1B'],
        ['SERPINE2','NOTCH2'],
        ['SERPINE2','PSMF1'],
        ['SERPINE2','RAB7A'],
        ['SERPINE2','SP1'],
        ['SERPINF1','COL3A1'],
        ['SESTD1','ZNF385B'],
        ['SETD8','AL356585.3'],
        ['SET','NCL'],
        ['SF3A1','IGFBP5'],
        ['SFMBT1','MINOS1'],
        ['SFMBT1','WNT7A'],
        ['SFMBT2','AKR1C4'],
        ['SFN','GPRC5A'],
        ['SFN','KRT17'],
        ['SFN','RPS21'],
        ['SFTPA1','CTSB'],
        ['SFTPA1','EPHB4'],
        ['SFTPA2','PHC2'],
        ['SFTPB','6-Mar'],
        ['SFTPB','ADAM17'],
        ['SFTPB','AKR1C2'],
        ['SFTPB','ALDH2'],
        ['SFTPB','ALDOA'],
        ['SFTPB','ALG5'],
        ['SFTPB','APOL1'],
        ['SFTPB','B4GALT1'],
        ['SFTPB','C6ORF48'],
        ['SFTPB','CAST'],
        ['SFTPB','CCDC109B'],
        ['SFTPB','CCND3'],
        ['SFTPB','CD9'],
        ['SFTPB','CD99'],
        ['SFTPB','CFL1'],
        ['SFTPB','CIT'],
        ['SFTPB','CTNNA1'],
        ['SFTPB','CTSC'],
        ['SFTPB','CUEDC2'],
        ['SFTPB','CUX1'],
        ['SFTPB','DNAJC5'],
        ['SFTPB','DNM2'],
        ['SFTPB','DOPEY2'],
        ['SFTPB','DYNC1H1'],
        ['SFTPB','EIF2AK1'],
        ['SFTPB','EIF3E'],
        ['SFTPB','ERGIC1'],
        ['SFTPB','FGD5-AS1'],
        ['SFTPB','GPD1L'],
        ['SFTPB','ITGA3'],
        ['SFTPB','ITGB1'],
        ['SFTPB','ITPR1'],
        ['SFTPB','KIAA0319L'],
        ['SFTPB','KRT8'],
        ['SFTPB','LIMCH1'],
        ['SFTPB','LIPA'],
        ['SFTPB','LSR'],
        ['SFTPB','MAL2'],
        ['SFTPB','MGST1'],
        ['SFTPB','MSLN'],
        ['SFTPB','MVP'],
        ['SFTPB','NDRG1'],
        ['SFTPB','NUCB2'],
        ['SFTPB','PFKP'],
        ['SFTPB','PLD3'],
        ['SFTPB','PLS3'],
        ['SFTPB','PRPF6'],
        ['SFTPB','PSMB1'],
        ['SFTPB','RBM3'],
        ['SFTPB','RGS5'],
        ['SFTPB','RPL28'],
        ['SFTPB','RPN1'],
        ['SFTPB','SERP1'],
        ['SFTPB','SLC44A2'],
        ['SFTPB','SNRPB'],
        ['SFTPB','SNTB1'],
        ['SFTPB','SORCS2'],
        ['SFTPB','SPINT2'],
        ['SFTPB','SUMF1'],
        ['SFTPB','SYNE2'],
        ['SFTPB','TACC2'],
        ['SFTPB','TKT'],
        ['SFTPB','TMEM189-UBE2V1'],
        ['SFTPB','TMPRSS4'],
        ['SFTPB','TRIM59'],
        ['SFTPB','TUBB3'],
        ['SFTPB','TXNRD1'],
        ['SFTPB','VMP1'],
        ['SFTPB','WDR82'],
        ['SFTPB','XRCC5'],
        ['SFTPB','YWHAQ'],
        ['SGCD','PPP1R2P3'],
        ['SGK1','KRT5'],
        ['SGPL1','KRT5'],
        ['SGPP2','ALG14'],
        ['SGTA','TG'],
        ['SH2D3C','TOR2A'],
        ['SH3BP1','LGALS1'],
        ['SH3BP2','ADD1'],
        ['SH3D19','INPP4B'],
        ['SH3PXD2A','LINC01435'],
        ['SH3PXD2A','OBFC1'],
        ['SHANK2','FLJ42102'],
        ['SHANK2','HMGN3'],
        ['SHANK2','PAPD5'],
        ['SHB','LINGO2'],
        ['SHC1','RPRD2'],
        ['SHC2','RNF126'],
        ['SHKBP1','TG'],
        ['SHPRH','B4GALNT1'],
        ['SHPRH','NDUFS5'],
        ['SHROOM3','FABP5'],
        ['SHROOM3','FIP1L1'],
        ['SHROOM3','TRIM28'],
        ['SHROOM4','ARID3A'],
        ['SIAH1','SLC6A15'],
        ['SIL1','AC034243.1'],
        ['SIN3A','LOXL1-AS1'],
        ['SIN3B','ZNF285'],
        ['SIPA1','ANXA1'],
        ['SIPA1L2','ZNF252P'],
        ['SIPA1L3','CEP89'],
        ['SIPA1','RHOD'],
        ['SIRT6','CREB3L3'],
        ['SIX3','AC012354.6'],
        ['SKA2','HEATR6'],
        ['SKIL','GPR160'],
        ['SLBP','TACC3'],
        ['SLC12A7','TMEM117'],
        ['SLC12A8','BCAS1'],
        ['SLC16A12','LIPA'],
        ['SLC16A7','NUP107'],
        ['SLC17A5','NTAN1'],
        ['SLC1A3','C20ORF196'],
        ['SLC1A4','TG'],
        ['SLC22A3','ZDHHC14'],
        ['SLC23A2','LINC00658'],
        ['SLC23A2','PHKB'],
        ['SLC24A3','NAA20'],
        ['SLC25A16','DNA2'],
        ['SLC25A23','SLC2A9'],
        ['SLC25A23','TG'],
        ['SLC25A32','BAALC-AS1'],
        ['SLC25A43','SLC25A5'],
        ['SLC26A7','TG'],
        ['SLC28A1','PDE8A'],
        ['SLC29A1','HSP90AB1'],
        ['SLC2A12','SGK1'],
        ['SLC2A13','ARHGDIB'],
        ['SLC30A6','SPAST'],
        ['SLC34A2','ACTG1'],
        ['SLC34A2','ARPC1A'],
        ['SLC34A2','ATP11A'],
        ['SLC34A2','C3'],
        ['SLC34A2','CFL1'],
        ['SLC34A2','CPM'],
        ['SLC34A2','CTSH'],
        ['SLC34A2','FLNB'],
        ['SLC34A2','HNRNPUL1'],
        ['SLC34A2','P4HB'],
        ['SLC34A2','PIGR'],
        ['SLC34A2','TG'],
        ['SLC34A2','TMBIM6'],
        ['SLC35B1','TG'],
        ['SLC35E3','MYRFL'],
        ['SLC35F2','CCDC162P'],
        ['SLC35G2','NCK1'],
        ['SLC39A11','LINC00511'],
        ['SLC39A11','SOX9-AS1'],
        ['SLC39A3','DDX39A'],
        ['SLC39A3','GNG7'],
        ['SLC39A3','SGTA'],
        ['SLC3A2','GRAP2'],
        ['SLC3A2','TG'],
        ['SLC44A1','RAD23B'],
        ['SLC44A2','SFTPB'],
        ['SLC44A5','PTGFR'],
        ['SLC45A3','AZGP1'],
        ['SLC45A3','ELK4'],
        ['SLC45A3','ERG'],
        ['SLC45A4','MROH5'],
        ['SLC5A3','AP000318.2'],
        ['SLC6A15','MYRFL'],
        ['SLC7A5','KRT4'],
        ['SLC7A6','PLA2G15'],
        ['SLC8A1','SRSF7'],
        ['SLFN5','SYNRG'],
        ['SLFN5','TBC1D15'],
        ['SLIT3','TBC1D30'],
        ['SLPI','RPL28'],
        ['SLTM','UACA'],
        ['SMAD3','DIS3L'],
        ['SMAD3','DLGAP4'],
        ['SMAD3','MAP2K5'],
        ['SMAD4','GAB2'],
        ['SMARCA4','GNAS'],
        ['SMARCA4','ICAM1'],
        ['SMARCA4','LINC00598'],
        ['SMARCA4','MYH11'],
        ['SMARCC1','MAP4'],
        ['SMARCD2','NUFIP2'],
        ['SMC2','TENM1'],
        ['SMC5','AP001255.2'],
        ['SMC5-AS1','MAMDC2-AS1'],
        ['SMC5','IMMP2L'],
        ['SMCHD1','NDC80'],
        ['SMDT1','NDUFA6-AS1'],
        ['SMG1','ARL6IP1'],
        ['SMG1','MYH11'],
        ['SMG7','FTL'],
        ['SMIM14','FN1'],
        ['SMOC1','GFAP'],
        ['SMR3B','ARHGDIA'],
        ['SMR3B','DSE'],
        ['SMR3B','IGF2R'],
        ['SMR3B','PIGS'],
        ['SMR3B','SDC3'],
        ['SMR3B','UBR5'],
        ['SNAPIN','S100A1'],
        ['SNCG','ADIRF'],
        ['SND1','COL3A1'],
        ['SND1','IMMP2L'],
        ['SND1','PMEL'],
        ['SNN','APOE'],
        ['SNRPA','ANKRD24'],
        ['SNRPA','C19ORF54'],
        ['SNRPD2','TG'],
        ['SNX10','AC004540.5'],
        ['SNX22','SFTPB'],
        ['SNX27','ADAR'],
        ['SNX29','FAM168A'],
        ['SNX5','PCSK2'],
        ['SNX9','DAP'],
        ['SOCS7','FLOT2'],
        ['SOD2','ACTB'],
        ['SOD2','APOL1'],
        ['SOD2','ATP1B4'],
        ['SOD2','C3'],
        ['SOD2','CLU'],
        ['SOD2','EEF2'],
        ['SOD2','FN1'],
        ['SOD2','GFAP'],
        ['SOD2','HSP90AA1'],
        ['SOD2','IGFBP4'],
        ['SOD2','KRIT1'],
        ['SOD2','VWF'],
        ['SORBS1','PMEL'],
        ['SORBS1','TGM4'],
        ['SORBS2','AIF1'],
        ['SORBS2','PMEL'],
        ['SORCS2','SFTPB'],
        ['SORT1','COL1A1'],
        ['SORT1','FN1'],
        ['SORT1','MBP'],
        ['SOX2-OT','GFAP'],
        ['SOX8','DCAF8'],
        ['SPAG1','STK3'],
        ['SPAG5','BLMH'],
        ['SPARC','ADAMTS2'],
        ['SPARC','ATP2B2'],
        ['SPARC','BLOC1S5-TXNDC5'],
        ['SPARC','ELK3'],
        ['SPARCL1','ATP6V1A'],
        ['SPARCL1','DCLK2'],
        ['SPARCL1','MBP'],
        ['SPARCL1','SLC4A1'],
        ['SPARCL1','TG'],
        ['SPARCL1','TNR'],
        ['SPARC','NUMA1'],
        ['SPARC','PDLIM5'],
        ['SPARC','PPP1CB'],
        ['SPARC','RTN1'],
        ['SPATA5L1','C15ORF48'],
        ['SPATC1L','FTCD'],
        ['SPECC1L','PPP6R2'],
        ['SPECC1L','SQLE'],
        ['SPECC1L','ZNF572'],
        ['SPECC1','TMEM11'],
        ['SPEN','CERS4'],
        ['SPIN1','NXNL2'],
        ['SPINK1','CPM'],
        ['SPINT1','IGHG2'],
        ['SPINT2','C19ORF33'],
        ['SPON1','COL3A1'],
        ['SPON1','PIK3R3'],
        ['SPOP','IGHV7-81'],
        ['SPOP','MYH11'],
        ['SPOP','XYLT2'],
        ['SPP1','BMPR2'],
        ['SPP1','FTL'],
        ['SPP1','MKLN1'],
        ['SPP1','SLC3A2'],
        ['SPP1','ST3GAL1'],
        ['SPPL2A','PXDNL'],
        ['SPPL2A','TENM4'],
        ['SPRED1','TMCO5B'],
        ['SPRYD4','MDM1'],
        ['SPSB3','MEIOB'],
        ['SPSB4','PXYLP1'],
        ['SPTAN1','MAPKAP1'],
        ['SPTBN2','FGF19'],
        ['SPTLC3','ISM1'],
        ['SPTY2D1','SERGEF'],
        ['SRBD1','CCDC88A'],
        ['SRCAP','GFAP'],
        ['SRCAP','SLC26A4-AS1'],
        ['SREBF1','ELF3'],
        ['SREK1IP1','AP000568.2'],
        ['SRF','FOXP4-AS1'],
        ['SRP68','LPO'],
        ['SRP72','ARL9'],
        ['SRPK2','PUS7'],
        ['SS18L1','CDH4'],
        ['SSC5D','COL1A1'],
        ['SSH2','C10ORF54'],
        ['ST14','ST7'],
        ['ST3GAL3','TAL1'],
        ['ST6GAL1','CTSB'],
        ['ST6GAL1','TG'],
        ['ST6GALNAC6','NUP107'],
        ['ST6GALNAC6','PAWR'],
        ['ST7','CAPZA2'],
        ['ST7-OT4','CAPZA2'],
        ['STAB1','COL1A1'],
        ['STAG1','ISPD-AS1'],
        ['STAG1','MSL2'],
        ['STAG2','GFAP'],
        ['STAG3L5P-PVRIG2P-PILRB','STAG3'],
        ['STAG3L5P','STAG3'],
        ['STAP2','ABR'],
        ['STARD3','GSDMB'],
        ['STARD3','UVRAG'],
        ['STARD7','IGF2'],
        ['STAT2','B4GALNT1'],
        ['STAT3','PTRF'],
        ['STAT4','ACKR3'],
        ['STAT5A','YWHAEP7'],
        ['STAU1','CTSA'],
        ['STEAP2','PDLIM5'],
        ['STK10','GNS'],
        ['STK17B','HECW2'],
        ['STK19','STK19B'],
        ['STK24','FARP1-AS1'],
        ['STK24','TMTC4'],
        ['STK3','COL1A2'],
        ['STK40','MDM4'],
        ['STK40','OSCP1'],
        ['STOM','GGTA1P'],
        ['STON1','PPP1R21'],
        ['STRADA','LIMD2'],
        ['STRIP1','LINC01397'],
        ['STRIP2','SMKR1'],
        ['STRN4','PMEL'],
        ['STRN','ALK'],
        ['STUB1','RAB11FIP3'],
        ['STX17','NR4A3'],
        ['STXBP4','MSI2'],
        ['STXBP4','PPM1D'],
        ['SUCLA2-AS1','NUDT15'],
        ['SUCO','AC025263.3'],
        ['SUCO','C12ORF42'],
        ['SUCO','CDH18'],
        ['SUCO','GLIPR1L1'],
        ['SUCO','MYRFL'],
        ['SUFU','GBF1'],
        ['SUGT1P3','MRPS31'],
        ['SUMF1','SFTPB'],
        ['SUMF2','MYH9'],
        ['SUMF2','NASP'],
        ['SUV420H1','MTL5'],
        ['SUV420H1','TG'],
        ['SUZ12P1','CRLF3'],
        ['SUZ12P1','ORAOV1'],
        ['SUZ12P1','SUZ12'],
        ['SUZ12','SUZ12P1'],
        ['SYMPK','ZNF573'],
        ['SYN3','C6ORF48'],
        ['SYNE1','GFAP'],
        ['SYNE1','TG'],
        ['SYNE2','CTSB'],
        ['SYNE2','IGKC'],
        ['SYNPO2L','MYOZ1'],
        ['SYPL1','AKR1D1'],
        ['SYT11','CLIP2'],
        ['SYT11','TMBIM6'],
        ['SYT12','C11ORF80'],
        ['SYT1','PHKB'],
        ['SYT8','TNNI2'],
        ['SYTL4','TSPAN6'],
        ['TACC1','MYH11'],
        ['TACC2','CASC2'],
        ['TACC2','HS1BP3'],
        ['TACC2','SFTPB'],
        ['TACC2','TG'],
        ['TACSTD2','OMA1'],
        ['TADA1','CADM3'],
        ['TADA2B','SORCS2'],
        ['TAF15','COL1A1'],
        ['TAF15','MAN1A2'],
        ['TAF2','DSCC1'],
        ['TAF3','DNAJC1'],
        ['TAF4','GATA5'],
        ['TAF7','CH17-140K24.2'],
        ['TAF8','CRISP1'],
        ['TAGLN2','COPA'],
        ['TAGLN2','CTA-292E10.9'],
        ['TAL1','PDZK1IP1'],
        ['TALDO1','EPS8L2'],
        ['TANC2','ARHGEF12'],
        ['TANC2','CASC17'],
        ['TANC2','WFDC21P'],
        ['TANGO6','CDH1'],
        ['TAOK1','ACER3'],
        ['TAOK1','ZNF385C'],
        ['TAOK3','ERBB2'],
        ['TAOK3','VSIG10'],
        ['TAPBP','DAXX'],
        ['TARBP1','ARID4B'],
        ['TATDN1','CTB-78F1.1'],
        ['TAX1BP3','C17ORF85'],
        ['TAX1BP3','TRPV1'],
        ['TBC1D14','ATP6V0D2'],
        ['TBC1D15','SLC16A7'],
        ['TBC1D16','CTSB'],
        ['TBC1D1','CTSD'],
        ['TBC1D30','PPM1H'],
        ['TBCD','HLCS'],
        ['TBCD','SLC9A3R1'],
        ['TBCEL','TECTA'],
        ['TBL1XR1','PIK3CA'],
        ['TBL3','GFER'],
        ['TBX15','FMN2'],
        ['TC2N','ATXN3'],
        ['TCEANC','RAB9A'],
        ['TCF12','C7'],
        ['TCF12','CLIP2'],
        ['TCF12','NCOA3'],
        ['TCF12','TEX9'],
        ['TCF12','TMEM255B'],
        ['TCF25','ERBB2'],
        ['TCF25','SPIRE2'],
        ['TCF3','AP3D1'],
        ['TCF3','CTB-25B13.5'],
        ['TCTN2','ATP6V0A2'],
        ['TEAD1','MICAL2'],
        ['TEAD1','PARVA'],
        ['TEAD2','SLC6A16'],
        ['TEAD4','C2CD5'],
        ['TEKT3','MYADM'],
        ['TEKT4P2','BP-21264C1.2'],
        ['TERF2','COG4'],
        ['TESK2','GPR37L1'],
        ['TEX2','WFDC21P'],
        ['TFCP2','BIN2'],
        ['TFDP1','TMCO3'],
        ['TFG','GPR128'],
        ['TFG','MET'],
        ['TFRC','C1ORF43'],
        ['TF','SRPRB'],
        ['TG','2-Sep'],
        ['TG','ACTN4'],
        ['TG','ARCN1'],
        ['TG','ARPC5'],
        ['TG','ASPH'],
        ['TG','ATP13A1'],
        ['TG','ATRX'],
        ['TG','B4GALT1'],
        ['TG','BAG1'],
        ['TG','CAPZB'],
        ['TG','CD24'],
        ['TG','CD55'],
        ['TG','CDKL5'],
        ['TG','CMPK1'],
        ['TG','CNN2'],
        ['TG','COL14A1'],
        ['TG','CPE'],
        ['TG','CPQ'],
        ['TG','CSNK1D'],
        ['TG','CTTN'],
        ['TG','DAP'],
        ['TG','DAP3'],
        ['TG','DBN1'],
        ['TG','DDX23'],
        ['TG','DYSF'],
        ['TG','EFHD2'],
        ['TG','EIF4EBP2'],
        ['TG','EMC10'],
        ['TG','ENSA'],
        ['TG','ERGIC1'],
        ['TG','FAM189A2'],
        ['TGFBR2','XPC'],
        ['TGFBRAP1','FHL2'],
        ['TG','GDI2'],
        ['TG','GSE1'],
        ['TG','GTF2IRD1'],
        ['TG','HDLBP'],
        ['TG','HNRNPUL1'],
        ['TG','INADL'],
        ['TG','KCNAB1'],
        ['TG','KCNQ1'],
        ['TG','KDM2A'],
        ['TG','LAMB1'],
        ['TG','LGI3'],
        ['TG','LIFR'],
        ['TG','LINC00963'],
        ['TG','LONP1'],
        ['TG','LRP2'],
        ['TG','LRPAP1'],
        ['TG','LRRC75A-AS1'],
        ['TG','LRTOMT'],
        ['TG','LTBP4'],
        ['TGM4','ACTB'],
        ['TGM4','QSOX1'],
        ['TGM4','TPM4'],
        ['TG','MACF1'],
        ['TG','MGAT5'],
        ['TG','MTA1'],
        ['TG','MTUS1'],
        ['TG','MYADM'],
        ['TG','NACA'],
        ['TG','NID2'],
        ['TG','NMNAT2'],
        ['TG','NTRK2'],
        ['TG','OCIAD1'],
        ['TG','PACSIN2'],
        ['TG','PERP'],
        ['TG','PINK1-AS'],
        ['TG','PPIB'],
        ['TG','PPP2R1A'],
        ['TG','PRKDC'],
        ['TG','PTPN12'],
        ['TG','PTPRM'],
        ['TG','RFXANK'],
        ['TG','RRBP1'],
        ['TG','RXRA'],
        ['TG','SBF1'],
        ['TG','SERINC1'],
        ['TG','SHC1'],
        ['TG','SLC34A2'],
        ['TG','SLIRP'],
        ['TG','SMAD2'],
        ['TG','SOD3'],
        ['TG','SPRY1'],
        ['TG','TAPBP'],
        ['TG','TBC1D5'],
        ['TG','THADA'],
        ['TG','TPD52'],
        ['TG','TSHR'],
        ['TG','UTRN'],
        ['TG','VGLL4'],
        ['THADA','TG'],
        ['THAP7-AS1','CRKL'],
        ['THBS4','TIMP2'],
        ['THEM4','S100A10'],
        ['THRA','CACNB1'],
        ['THRA','USP32'],
        ['THSD4','ACPP'],
        ['TIMP2','COL6A3'],
        ['TIMP2','CTSB'],
        ['TIMP2','GFAP'],
        ['TIMP2','SERPINE2'],
        ['TIMP3','SCARB2'],
        ['TINAGL1','ADIRF'],
        ['TINAGL1','DNM2'],
        ['TINAGL1','FN1'],
        ['TINF2','NEDD8'],
        ['TIRAP','DCPS'],
        ['TJP2','GFAP'],
        ['TK2','EXOC3L1'],
        ['TKT','ANXA1'],
        ['TKT','PMEL'],
        ['TKT','SFTPB'],
        ['TLE1P1','TLE1'],
        ['TLK1','SSB'],
        ['TLK2','RP11-274B21.1'],
        ['TLN1','AZGP1'],
        ['TLR3','FAM149A'],
        ['TM2D2','ADAM9'],
        ['TM9SF2','CPM'],
        ['TM9SF2','MTUS2'],
        ['TMBIM1','ANXA1'],
        ['TMBIM1','TGM4'],
        ['TMBIM4','GRIP1'],
        ['TMBIM4','NAV3'],
        ['TMBIM6','TMEM213'],
        ['TMED10','EIF4G1'],
        ['TMED10','TTLL5'],
        ['TMED3','KIAA1024'],
        ['TMED4','AC005537.2'],
        ['TMED5','ADORA3'],
        ['TMEM101','PYY'],
        ['TMEM109','IGHG1'],
        ['TMEM119','LMNA'],
        ['TMEM120B','BHLHE40-AS1'],
        ['TMEM120B','CEP85L'],
        ['TMEM120B','GRIP1'],
        ['TMEM123','MMP27'],
        ['TMEM123','MMP7'],
        ['TMEM132E','COL1A1'],
        ['TMEM138','TMEM216'],
        ['TMEM147-AS1','MTMR3'],
        ['TMEM147-AS1','SBSN'],
        ['TMEM14C','PAK1IP1'],
        ['TMEM159','DNAH3'],
        ['TMEM161B','LINC00461'],
        ['TMEM184B','IGKC'],
        ['TMEM189-UBE2V1','PXMP4'],
        ['TMEM189-UBE2V1','SFTPB'],
        ['TMEM192','TMEM184C'],
        ['TMEM213','ATP6V0D2'],
        ['TMEM233','PRKAB1'],
        ['TMEM245','CTNNAL1'],
        ['TMEM255A','FPR2'],
        ['TMEM30A','TG'],
        ['TMEM37','TG'],
        ['TMEM57','RHCE'],
        ['TMEM69','ABHD2'],
        ['TMEM87B','MERTK'],
        ['TMEM9','IGKC'],
        ['TMOD3','KRT14'],
        ['TMPRSS2','ERG'],
        ['TMPRSS7','C3ORF52'],
        ['TMSB4Y','KALP'],
        ['TMTC2','LYZ'],
        ['TMTC2','METTL25'],
        ['TMTC2','MYRFL'],
        ['TMTC2','NBPF10'],
        ['TMTC3','SLC35E3'],
        ['TMTC4','FGF14'],
        ['TMX1','LINC00519'],
        ['TNFAIP2','P4HB'],
        ['TNFRSF21','SOD2'],
        ['TNFSF10','LTBP2'],
        ['TNFSF10','NCEH1'],
        ['TNNC2','WFDC3'],
        ['TNNI1','LAD1'],
        ['TNPO2','DNASE2'],
        ['TNPO2','NFIX'],
        ['TNR','CANX'],
        ['TNR','MBP'],
        ['TNS1','ACTN1'],
        ['TNS3','AC004901.1'],
        ['TNS3','ALG13'],
        ['TNS3','SPARC'],
        ['TNS4','LAMP1'],
        ['TOLLIP','COL1A1'],
        ['TOP3A','MYO18A'],
        ['TOPORS','ADAMTSL1'],
        ['TOX3','CASC16'],
        ['TP53BP1','DMXL2'],
        ['TP53','FXR2'],
        ['TP63','KRT5'],
        ['TP63','KRT6A'],
        ['TPCN1','TG'],
        ['TPD52','ACPP'],
        ['TPD52','KLK3'],
        ['TPD52L1','RNF217'],
        ['TPO','HSP90B1'],
        ['TPRG1','DGKG'],
        ['TPST1','GS1-124K5.4'],
        ['TPST2','TG'],
        ['TPT1-AS1','LINC01198'],
        ['TPTE2P1','CENPJ'],
        ['TPTE2P5','MRPS31'],
        ['TPTEP1','AL078471.5'],
        ['TPX2','HM13'],
        ['TRA2A','HECW1'],
        ['TRA2B','LINC00879'],
        ['TRADD','B3GNT9'],
        ['TRAF3','PMEL'],
        ['TRAK1','RAF1'],
        ['TRAPPC10','BRWD1'],
        ['TRAPPC9','COL22A1'],
        ['TRAPPC9','IGHG1'],
        ['TRDV1','TRAC'],
        ['TRHDE','C12ORF75'],
        ['TRHDE','TBC1D15'],
        ['TRHDE','TSFM'],
        ['TRIB1','NSMCE2'],
        ['TRIM25','TBX4'],
        ['TRIM25','TRIM37'],
        ['TRIM33','CSDE1'],
        ['TRIM37','LINC01483'],
        ['TRIM3','HPX'],
        ['TRIM5','IGKC'],
        ['TRIOBP','UNC13C'],
        ['TRIO','PPP2R5E'],
        ['TRIO','TERT'],
        ['TRIP12','DNER'],
        ['TRIP12','PID1'],
        ['TRIP12','SLC16A14'],
        ['TRIQK','FLJ46284'],
        ['TRIQK','RAD54B'],
        ['TRMT12','RNF139'],
        ['TROVE2','CDC73'],
        ['TRPS1','TBC1D9B'],
        ['TRPS1','ZNRF2'],
        ['TRUB1','FAM160B1'],
        ['TSC1','AK8'],
        ['TSC1','CEL'],
        ['TSC1','DOLPP1'],
        ['TSC22D1','NDUFS7'],
        ['TSC22D3','IGF2'],
        ['TSC2','TMEM204'],
        ['TSC2','XX-DJ76P10__A.2'],
        ['TSEN2','PPARG'],
        ['TSFM','CPM'],
        ['TSFM','INHBE'],
        ['TSFM','NUP107'],
        ['TSFM','RCSD1'],
        ['TSPAN14','DYDC2'],
        ['TSPAN14','TG'],
        ['TSPAN18','DCN'],
        ['TSPAN19','CPSF6'],
        ['TSPAN31','GFAP'],
        ['TSPAN3','CORO2B'],
        ['TSPAN3','PTPN9'],
        ['TSPAN8','ASH2L'],
        ['TSTD1','ARHGAP30'],
        ['TTBK2','SPATA5L1'],
        ['TTC12','C7ORF55-LUC7L2'],
        ['TTC19','TG'],
        ['TTC21B-AS1','AC010127.3'],
        ['TTC21B','SLC38A11'],
        ['TTC22','PARS2'],
        ['TTC39A','PHGDH'],
        ['TTC6','MIPOL1'],
        ['TTC7B','MDM4'],
        ['TTC7B','PMEL'],
        ['TTLL4','CYP27A1'],
        ['TTLL5','TMED10'],
        ['TTN','AC018890.4'],
        ['TUBB3','TCF25'],
        ['TUBB4A','PMEL'],
        ['TUBGCP3','FARP1'],
        ['TUBGCP3','MSMO1'],
        ['TUBGCP3','TFDP1'],
        ['TULP3','C12ORF57'],
        ['TULP3','TEAD4'],
        ['TUT1','BSCL2'],
        ['TXN2','FOXRED2'],
        ['TXNDC11','LITAF'],
        ['TXNDC11','RORA'],
        ['TXNDC15','ARPC2'],
        ['TXNDC16','ARG2'],
        ['TXNRD1','ANXA1'],
        ['TXNRD1','TG'],
        ['TXNRD1','TKT'],
        ['TYK2','CDC37'],
        ['TYRP1','APP'],
        ['TYRP1','NCOR2'],
        ['TYRP1','RERE'],
        ['U73166.2','SEMA3B'],
        ['U82695.9','BGN'],
        ['UBA1','CDK16'],
        ['UBA1','CTSD'],
        ['UBA2','GPI'],
        ['UBA2','WTIP'],
        ['UBA3','FRMD4B'],
        ['UBA52','AC099535.4'],
        ['UBA52','CACNA1C'],
        ['UBAC1','SHANK2'],
        ['UBAP1','KRT6A'],
        ['UBE2E2','AC026167.1'],
        ['UBE2H','IGHM'],
        ['UBE2O','TG'],
        ['UBE2R2','PRSS3'],
        ['UBE2V2','TG'],
        ['UBE3A','DISC1FP1'],
        ['UBE4B','ZNF536'],
        ['UBL3','MTIF3'],
        ['UBN1','LAMA5'],
        ['UBQLN1','GKAP1'],
        ['UBXN2A','GBAS'],
        ['UBXN7','RPL39L'],
        ['UCHL3','LMO7'],
        ['UCHL5','C1ORF112'],
        ['UCK2','DHX9'],
        ['UCK2','SNX27'],
        ['UEVLD','PRMT3'],
        ['UGGT1','COL1A1'],
        ['UGT1A2P','UGT1A8'],
        ['UHRF1BP1L','ZDHHC17'],
        ['UHRF1BP1','MDC1-AS1'],
        ['UHRF1','CTA-292E10.6'],
        ['UNC13D','SRP68'],
        ['UNC45A','BLM'],
        ['UNC45A','KRT13'],
        ['UNK','DPYD'],
        ['UPK1B','UPK2'],
        ['URB1','MIS18A'],
        ['URB2','KB-1562D12.1'],
        ['URI1','ARHGAP33'],
        ['URI1','IFNL1'],
        ['USP10','ATP2C2'],
        ['USP13','PIK3CA'],
        ['USP22','ACPP'],
        ['USP22','AZGP1'],
        ['USP22','GFAP'],
        ['USP22','MYH10'],
        ['USP25','CYYR1'],
        ['USP32','IKZF3'],
        ['USP32','LINC00511'],
        ['USP32','PGS1'],
        ['USP32','RN7SL606P'],
        ['USP34','FAM161A'],
        ['USP36','OFCC1'],
        ['USP39','DNAH6'],
        ['USP43','ENDOV'],
        ['USP54','SCD5'],
        ['USP54','TG'],
        ['USP6NL','ARMC3'],
        ['USP7','CARHSP1'],
        ['UTP11L','VMA21'],
        ['UTRN','ECT2L'],
        ['UTRN','TG'],
        ['UVRAG','LRTOMT'],
        ['UVRAG','PAK1'],
        ['VAC14','WWOX'],
        ['VAMP7','TGFB2'],
        ['VANGL2','IGF2'],
        ['VARS','TG'],
        ['VCAN','MALAT1'],
        ['VDAC2','C10ORF11'],
        ['VEGFA','BLOC1S6'],
        ['VEGFA','UACA'],
        ['VGLL4','FXR1'],
        ['VIM-AS1','KRT5'],
        ['VIM-AS1','PMEL'],
        ['VIM','ASPH'],
        ['VIM','CERS2'],
        ['VIM','KRT5'],
        ['VIM','KRT7'],
        ['VIM','NRXN1'],
        ['VIPAS39','NOXRED1'],
        ['VMP1','RAD51C'],
        ['VPS13B','MYO9B'],
        ['VPS13B','SGCZ'],
        ['VPS13B','STAU2'],
        ['VPS13C','LINC00593'],
        ['VPS13C','RB1CC1'],
        ['VPS13D','GFAP'],
        ['VPS35','SLC6A2'],
        ['VPS45','PLEKHO1'],
        ['VPS53','TXNDC17'],
        ['VPS8','MBOAT2'],
        ['VPS9D1-AS1','ZNF276'],
        ['VRK2','AGO3'],
        ['VWA1','ATAD3C'],
        ['VWF','SOD2'],
        ['WAC','UBAC2'],
        ['WARS','CCNY'],
        ['WARS','MSH6'],
        ['WASH5P','WASH7P'],
        ['WBP11','MGP'],
        ['WBP1L','CYP17A1-AS1'],
        ['WBP1L','SUFU'],
        ['WDFY2','DLGAP2-AS1'],
        ['WDFY3-AS2','ARHGAP24'],
        ['WDR1','IGKC'],
        ['WDR34','ZER1'],
        ['WDR36','NREP'],
        ['WDR62','HCST'],
        ['WDR62','MAVS'],
        ['WDR72','AZIN1'],
        ['WDR81','SERPINF2'],
        ['WDR91','C7ORF49'],
        ['WEE2-AS1','KIAA1147'],
        ['WHSC1','FAM53A'],
        ['WHSC1','GSDMB'],
        ['WHSC1L1','ADAM32'],
        ['WHSC1L1','PSD3'],
        ['WHSC1','TG'],
        ['WIPF2','EZH1'],
        ['WIPF2','USP32'],
        ['WNK1','ERC1'],
        ['WNK1','TKT'],
        ['WNK2','KLK3'],
        ['WRAP53','SLC44A2'],
        ['WWP1','CPB2'],
        ['WWP1','CPNE3'],
        ['WWP2','TSEN54'],
        ['WWTR1','COL5A1'],
        ['WWTR1','IGFBP3'],
        ['WWTR1','KRT5'],
        ['XPO1','ERBB2'],
        ['XPO1','TG'],
        ['XPO1','USP34'],
        ['XPO6','GSG1L'],
        ['XPO7','H2AFY2'],
        ['XPOT','LYZ'],
        ['XPR1','TG'],
        ['XPR1','TNR'],
        ['XRCC2','WASL'],
        ['XRCC5','SFTPB'],
        ['XRN1','SMYD3'],
        ['XXBAC-B444P24.10','PRODH'],
        ['XX-FW83563B9.5','DNASE1L1'],
        ['YAP1','BIRC2'],
        ['YAP1','PTRF'],
        ['YARS','ADM5'],
        ['YEATS4','AGAP2'],
        ['YEATS4','ARHGEF25'],
        ['YEATS4','C12ORF55'],
        ['YEATS4','NME7'],
        ['YIPF4','COMMD1'],
        ['YPEL2','HSPA12B'],
        ['YPEL2','PAK1'],
        ['YWHAE','CRK'],
        ['YWHAE','CTSB'],
        ['YWHAQ','YWHAQP6'],
        ['YWHAZ','GATA3'],
        ['YY1','IGKC'],
        ['YY1','TESPA1'],
        ['ZBED4','CITF22-1A6.3'],
        ['ZBED4','RANGAP1'],
        ['ZBTB1','PPP1R36'],
        ['ZBTB37','FSD1L'],
        ['ZBTB44','FRS2'],
        ['ZBTB7B','ADAM15'],
        ['ZC3H18','CHMP1A'],
        ['ZC3H3','KIAA0907'],
        ['ZC3H3','PYCRL'],
        ['ZC3H7A','LITAF'],
        ['ZC3HAV1','ITPKB'],
        ['ZC3HAV1','ZC3HAV1L'],
        ['ZCCHC11','ORC1'],
        ['ZDHHC11B','TPPP'],
        ['ZDHHC13','MGAT4C'],
        ['ZDHHC14','NOX3'],
        ['ZDHHC17','CSRP2'],
        ['ZDHHC17','NAV3'],
        ['ZDHHC17','NCR2'],
        ['ZDHHC17','RAB3IP'],
        ['ZDHHC1','TPPP3'],
        ['ZDHHC21','PHGDH'],
        ['ZER1','ZDHHC12'],
        ['ZFAND3','BTBD9-AS1'],
        ['ZFAND3','DDHD2'],
        ['ZFAND3','SFTPB'],
        ['ZFAND3','TG'],
        ['ZFC3H1','MON2'],
        ['ZFHX3','ZMAT4'],
        ['ZFP36L2','IGKC'],
        ['ZFP90','CDH1'],
        ['ZFP91','RAB6A'],
        ['ZFYVE16','FAM151B'],
        ['ZHX1','RALYL'],
        ['ZHX2','BAALC-AS1'],
        ['ZHX3','PLCG1-AS1'],
        ['ZKSCAN1','MYH11'],
        ['ZMAT3','LINC01014'],
        ['ZMIZ1','TG'],
        ['ZNF132','LINC00613'],
        ['ZNF136','XAB2'],
        ['ZNF141','ZNF876P'],
        ['ZNF146','ABCA9'],
        ['ZNF148','FHAD1'],
        ['ZNF180','COL27A1'],
        ['ZNF20','TINCR'],
        ['ZNF219','HNRNPC'],
        ['ZNF248','FAM107B'],
        ['ZNF252P','ZNF16'],
        ['ZNF260','HECW1'],
        ['ZNF260','SCGB2B2'],
        ['ZNF264','AURKC'],
        ['ZNF271','ZNF397'],
        ['ZNF282','AC015987.1'],
        ['ZNF285','ZNF806'],
        ['ZNF292','FIG4'],
        ['ZNF318','CD2AP'],
        ['ZNF343','CSNK2A1'],
        ['ZNF345','TRIM21'],
        ['ZNF385A','NAV3'],
        ['ZNF385B','SCG5'],
        ['ZNF395','SLC25A44'],
        ['ZNF417','BAX'],
        ['ZNF429','ZNF738'],
        ['ZNF431','RN7SL364P'],
        ['ZNF44','MYRFL'],
        ['ZNF44','ZNF833P'],
        ['ZNF451','BAG2'],
        ['ZNF462','DNAJC1'],
        ['ZNF470','ZNF71'],
        ['ZNF490','FARSA'],
        ['ZNF501','KIF15'],
        ['ZNF528','ZNF534'],
        ['ZNF529-AS1','ZNF382'],
        ['ZNF532','MALT1'],
        ['ZNF544','AP2A1'],
        ['ZNF544','SENP6'],
        ['ZNF550','CXCL17'],
        ['ZNF552','ZNF814'],
        ['ZNF562','DTX3'],
        ['ZNF577','ZNF649'],
        ['ZNF57','NANOS3'],
        ['ZNF587B','ZSCAN4'],
        ['ZNF606','GDAP1'],
        ['ZNF607','CAPN12'],
        ['ZNF609','KRT5'],
        ['ZNF610','RALA'],
        ['ZNF614','ZNF649'],
        ['ZNF618','PARP4'],
        ['ZNF649','ZNF577'],
        ['ZNF665','RANBP3L'],
        ['ZNF702P','ZNF701'],
        ['ZNF704','IMPA1'],
        ['ZNF717','BP-21264C1.2'],
        ['ZNF717','TMEM254'],
        ['ZNF718','ZNF876P'],
        ['ZNF71','HSPB8'],
        ['ZNF71','SMIM17'],
        ['ZNF732','DAB1'],
        ['ZNF738','ZNF429'],
        ['ZNF765','CACNG6'],
        ['ZNF765','ZNF331'],
        ['ZNF771','ZNF48'],
        ['ZNF772','ERCC2'],
        ['ZNF79','SLC2A8'],
        ['ZNF814','EIF3B'],
        ['ZNF816-ZNF321P','ERGIC3'],
        ['ZNF823','VAV1'],
        ['ZNF833P','ZNF44'],
        ['ZNF876P','ZNF141'],
        ['ZNF91','RNF38'],
        ['ZNRF3','ASCC2'],
        ['ZNRF3','FAM19A5'],
        ['ZSCAN26','PGBD1'],
        ['ZSWIM4','C19ORF53'],
        ['ZSWIM4','CCDC130']


]



    data = fusions.get(options.organism.lower(),[])
    if data:

        #file_symbols = os.path.join(options.output_directory,'genes_symbols.txt')
        file_symbols = os.path.join(options.output_directory,'synonyms.txt')
        loci = symbols.generate_loci(file_symbols)

        genes = symbols.read_genes_symbols(file_symbols)

        d = []
        for (g1,g2) in data:
            if g1.upper() != g2.upper():
                ens1 = symbols.ensembl(g1.upper(),genes,loci)
                ens2 = symbols.ensembl(g2.upper(),genes,loci)
                if ens1 and ens2:
                    for e1 in ens1:
                        for e2 in ens2:
                            if e1 != e2:
                                d.append([e1,e2])

        data = ['\t'.join(sorted(line)) + '\n' for line in d]
        data = list(set(data))

        print "%d known fusion genes found in manually currated database" % (len(data),)

        if not options.skip_filter_overlap:
            d1 = []
            overlappings = ['ensembl_fully_overlapping_genes.txt',
                            'ensembl_same_strand_overlapping_genes.txt',
#                            'refseq_fully_overlapping_genes.txt',
#                            'refseq_same_strand_overlapping_genes.txt',
#                            'ucsc_fully_overlapping_genes.txt',
#                            'ucsc_same_strand_overlapping_genes.txt',
#                            'pairs_pseudogenes.txt',
#                            'paralogs.txt'
                            ]
            for ov in overlappings:
                p = os.path.join(options.output_directory,ov)
                print "Parsing file:",p
                if os.path.isfile(p):
                    d2 = sorted(set([tuple(sorted(line.rstrip('\r\n').split('\t'))) for line in file(p,'r').readlines() if line.rstrip('\r\n')]))
                d1.extend(d2)
            d = set()
            for line in d1:
                (a,b) = (line[0],line[1])
                if a > b:
                    (a,b) = (b,a)
                d.add("%s\t%s\n" % (a,b))
            skipped = [line for line in data if line in d]
            data = [line for line in data if not line in d]
            file(os.path.join(options.output_directory,'18cancers_known_but_overlapping.txt'),'w').writelines(sorted(skipped))

            print "%d known fusion genes left after removing the overlappings" % (len(data),)

    file(os.path.join(options.output_directory,'18cancers.txt'),'w').writelines(sorted(data))
    #
