#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of banned candidate fusion genes. This list is hard coded inhere.



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of banned candidate fusion genes."""
    version = "%prog 0.10 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of banned candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of banned candidate fusion genes is generated. Default is '%default'.""")


    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    print "Generating the list of banned fusion genes..."
    fusions = dict()

    fusions['rattus_norvegicus'] = [
        ['BIN2','CELA1'],
        ['ALB','FETUB'],
        ['C8G','LCN12'],
        ['FETUB','TF'],
        ['ALB','TF'],
        ['APOC1','APOC4'],
        ['GGH','TTPA'],
        ['GM7694','NOS1AP'],
        ['ALB','C3'],
        ['APOC2','APOC4'],
        ['DDT','GSTT3'],
        ['CES1H','CES2I'],
        ['PHYH','TF'],
        ['AHSG','ALB'],
        ['ALB','APOE'],
        ['ALB','SERPINF1'],
        ['NDUFB9','RNF139'],
        ['PROC','TF'],
        ['ALB','CTSH'],
        ['ALB','FGA'],
        ['ALB','SERPINA1'],
        ['ALB','SERPINA3K'],
        ['CRYGA','D630023F18RIK'],
        ['RBP4','SERPINA1'],
        ['ACOX2','TF'],
        ['ALB','APLP2'],
        ['ALB','HP'],
        ['ALB','PHYH'],
        ['ALB','TTR'],
        ['ALB','VTN'],
        ['ALDOB','MRPL50'],
        ['APOA5','TF'],
        ['ATP5L','UBE4A'],
        ['CRP','TF'],
        ['FABP7','SMPDL3A'],
        ['GOLT1A','KISS1'],
        ['GSTA2','TF'],
        ['IGKC','RGD1559916'],
        ['SERPINF2','WDR81'],
        ['TF','TTR'],
        ['ADNP','DPM1'],
        ['AHCY','ALB'],
        ['AHSG','HP'],
        ['AHSG','RBP4'],
        ['ALB','ALDH2'],
        ['ALB','AMACR'],
        ['ALB','AMBP'],
        ['ALB','APOA1'],
        ['ALB','C2'],
        ['ALB','CES1C'],
        ['ALB','CLU'],
        ['ALB','CPS1'],
        ['ALB','CRP'],
        ['ALB','CYP2E1'],
        ['ALB','FGB'],
        ['ALB','GLUL'],
        ['ALB','HMGCS2'],
        ['ALB','HPX'],
        ['ALB','MAT1A'],
        ['ALB','SERPINF2'],
        ['ALB','SLCO1B3'],
        ['AMACR','C1QTNF3'],
        ['AMBP','TF'],
        ['APOB','FETUB'],
        ['APOE','SERPINF1'],
        ['APOH','CRP'],
        ['APOH','HP'],
        ['ATP5B','SLC34A1'],
        ['C2','PHYH'],
        ['CAR3','TF'],
        ['CEACAM18-PS1','ZFP819'],
        ['CYP1A2','SERPINA1'],
        ['FGB','SERPINA1'],
        ['GC','RBP4'],
        ['GPD1','SERPINA1'],
        ['IGFBP5','ITM2B'],
        ['KDELC1','TEX30'],
        ['METTL21B','TSFM'],
        ['MIOX','SLC34A1'],
        ['SERPINA1','TTR'],
        ['ALB','ENSRNOG00000005599'],
        ['ALB','ENSRNOG00000009273'],
        ['ALB','ENSRNOG00000050117'],
        ['ARHGAP5','ENSRNOG00000047751'],
        ['CORO7','ENSRNOG00000046980'],
        ['ENSRNOG00000005599','PROC'],
        ['ENSRNOG00000011640','MECOM'],
        ['ENSRNOG00000015605','PTPRK'],
        ['ENSRNOG00000042512','PSMD10'],
        ['ENSRNOG00000047366','RGD1311595'],
        ['ENSRNOG00000048877','PLCB1'],
        ['ENSRNOG00000049177','PSD3'],
        ['ENSRNOG00000050066','IGKC'],
        ['ENSRNOG00000050188','PITPNC1'],
        ['GM10324','GM1976'],
        ['ISY1','RAB43'],
        ['SYS1','DBNDD2'],
        ['GM9900','ZFP672'],
        ['BRDT','EPHX4'],
        ['A930018M24RIK','KLHL33'],
        ['GM13308','GM12394'],
        ['2610524H06RIK','1500011B03RIK'],
        ['GRIA1','KIF5A'],
        ['APP','CPLX2'],
        ['ARF3','SPOCK2'],
        ['GTF2H5','TULP4'],
        ['ATP2B3','CAMK2A'],
        ['GM20687','THTPA'],
        ['PYGO1','GM20510'],
        ['A930011G23RIK','RASGEF1B'],
        ['PIGZ','NCBP2'],
        ['YOD1','AA986860'],
        ['PCSK2','CAMK2A'],
        ['SAMD5','SASH1'],
        ['PARP6','PKM'],
        ['PCP4L1','SDHC'],
        ['PVALB','IFT27'],
        ['C630016N16RIK','SBSN'],
        ['SIN3B','F2RL3'],
        ['TTBK1','KIF5A'],
        ['PHACTR1','CAMK2A'],
        ['NOLC1','GM5124'],
        ['TMEM219','TAOK2'],
        ['6330407J23RIK','9330159F19RIK'],
        ['MBP','KIF5A'],
        ['ZSCAN22','RPS5'],
        ['CPLX2','APP'],
        ['CAMK2A','ARF3'],
        ['VSNL1','CAMK2A'],
        ['SPARCL1','CPLX2'],
        ['GM13832','MSI1'],
        ['AV099323','CDS2'],
        ['PTPRN2','KIF5A'],
        ['A230083G16RIK','MLF2'],
        ['TOP3B','PPM1F'],
        ['METTL10','FAM53B'],
        ['RNF139','NDUFB9'],
        ['SPOP','STX1B'],
        ['PANX2','TRABD'],
        ['GM5124','NOLC1'],
        ['OTUD5','PIM2'],
        ['YWHAG','GNG7'],
        ['ARF3','DZANK1'],
        ['GALNTL6','AC116875.1'],
        ['CAMK2A','SYT1'],
        ['E130111B04RIK','MPP3'],
        ['GEMIN7','PPP1R37'],
        ['PCED1A','CPXM1'],
        ['ARF3','STX1B'],
        ['CAMK2A','YWHAG'],
        ['NAT15','CLUAP1'],
        ['IQCJ','SCHIP1'],
        ['PCDHGA3','CAMK2A'],
        ['SYNA','CLIP2'],
        ['ACSS2','GM14256'],
        ['CNRIP1','PPP3R1'],
        ['FTL','TIMP1'],
        ['MFGE8','HAPLN3'],
        ['CLCF1','POLD4'],
        ['MFSD7','ATP5I']
    ]

    fusions['mus_musculus'] = [
        ['A930011G23RIK','RASGEF1B'],
        ['GM10324','GM1976'],
        ['ISY1','RAB43'],
        ['SYS1','DBNDD2'],
        ['GM9900','ZFP672'],
        ['BRDT','EPHX4'],
        ['A930018M24RIK','KLHL33'],
        ['GM13308','GM12394'],
        ['2610524H06RIK','1500011B03RIK'],
        ['GRIA1','KIF5A'],
        ['APP','CPLX2'],
        ['ARF3','SPOCK2'],
        ['GTF2H5','TULP4'],
        ['ATP2B3','CAMK2A'],
        ['GM20687','THTPA'],
        ['PYGO1','GM20510'],
        ['A930011G23RIK','RASGEF1B'],
        ['PIGZ','NCBP2'],
        ['YOD1','AA986860'],
        ['PCSK2','CAMK2A'],
        ['SAMD5','SASH1'],
        ['PARP6','PKM'],
        ['PCP4L1','SDHC'],
        ['PVALB','IFT27'],
        ['C630016N16RIK','SBSN'],
        ['SIN3B','F2RL3'],
        ['TTBK1','KIF5A'],
        ['PHACTR1','CAMK2A'],
        ['NOLC1','GM5124'],
        ['TMEM219','TAOK2'],
        ['6330407J23RIK','9330159F19RIK'],
        ['MBP','KIF5A'],
        ['ZSCAN22','RPS5'],
        ['CPLX2','APP'],
        ['CAMK2A','ARF3'],
        ['VSNL1','CAMK2A'],
        ['SPARCL1','CPLX2'],
        ['GM13832','MSI1'],
        ['AV099323','CDS2'],
        ['PTPRN2','KIF5A'],
        ['A230083G16RIK','MLF2'],
        ['TOP3B','PPM1F'],
        ['METTL10','FAM53B'],
        ['RNF139','NDUFB9'],
        ['SPOP','STX1B'],
        ['PANX2','TRABD'],
        ['GM5124','NOLC1'],
        ['OTUD5','PIM2'],
        ['YWHAG','GNG7'],
        ['ARF3','DZANK1'],
        ['GALNTL6','AC116875.1'],
        ['CAMK2A','SYT1'],
        ['E130111B04RIK','MPP3'],
        ['GEMIN7','PPP1R37'],
        ['PCED1A','CPXM1'],
        ['ARF3','STX1B'],
        ['CAMK2A','YWHAG'],
        ['NAT15','CLUAP1'],
        ['IQCJ','SCHIP1'],
        ['PCDHGA3','CAMK2A'],
        ['SYNA','CLIP2'],
        ['ACSS2','GM14256'],
        ['CNRIP1','PPP3R1']
    ]

    fusions['canis_lupus_familiaris'] = [
        ['ENSCAFG00000011681','ENSCAFG00000040733'],
        ['ENSCAFG00000028546','ENSCAFG00000000193'],
        ['ENSCAFG00000028546','ENSCAFG00000000193'],
        ['ENSCAFG00000028546','ENSCAFG00000000193'],
        ['ENSCAFG00000038498','ENSCAFG00000004924'],
        ['ENSCAFG00000035698','ENSCAFG00000007015'],
        ['ENSCAFG00000040551','ENSCAFG00000007135']

    ]


    fusions['homo_sapiens'] = [
        ['7SK','CLK4'],
        ['7SK','COG7'],
        ['A2M','ALB'],
        ['A2M','CRP'],
        ['A2M','EPAS1'],
        ['A2M','FGG'],
        ['A2M','HP'],
        ['A2M','MB'],
        ['A2M','SFTPA1'],
        ['A2M','SFTPC'],
        ['A2M','SIK1'],
        ['A2ML1','ABI2'],
        ['A2ML1','ANXA2'],
        ['A2ML1','COA8'],
        ['A2ML1','CRNN'],
        ['A2ML1','GIGYF2'],
        ['A2ML1','KRT4'],
        ['A2ML1','MYH11'],
        ['A2ML1','NDE1'],
        ['A2ML1','PHC1'],
        ['A2ML1','RIMKLB'],
        ['A2ML1','ZNF665'],
        ['A3GALT2','PHC2'],
        ['A4GNT','DZIP1L'],
        ['AACSP1','GBP6'],
        ['AADAC','SUCNR1'],
        ['AADACL3','C1ORF158'],
        ['AAK1','ZNF626'],
        ['AAMP','COL1A1'],
        ['AARS','SMR3B'],
        ['AATF','AC010542.3'],
        ['AATF','TK2'],
        ['AB015752.1','TMEM64'],
        ['AB015752.3','TMEM64'],
        ['ABAT','ALB'],
        ['ABAT','CEL'],
        ['ABBA01031661.1','ABBA01031674.1'],
        ['ABCA10','ABCA6'],
        ['ABCA10','GFOD1'],
        ['ABCA13','AC027088.1'],
        ['ABCA13','LYZ'],
        ['ABCA13','NOX5'],
        ['ABCA13','TRA@'],
        ['ABCA17P','CCNF'],
        ['ABCA3','SFTPC'],
        ['ABCA8','ABCA9'],
        ['ABCA8','LYZ'],
        ['ABCA8','PIGR'],
        ['ABCC6','NEAT1'],
        ['ABCF2','ICA1'],
        ['ABHD11','SNAP23'],
        ['ABHD14B','PNLIP'],
        ['ABHD15-AS1','INTS7'],
        ['ABHD17A','CCDC144NL-AS1'],
        ['ABHD17A','PTMA'],
        ['ABHD17B','CEMIP2'],
        ['ABHD17C','CTGF'],
        ['ABHD2','AC013565.1'],
        ['ABHD2','AC020656.1'],
        ['ABHD2','CCNY'],
        ['ABHD2','FANCI'],
        ['ABHD2','LYZ'],
        ['ABHD2','SMR3B'],
        ['ABI1','LYZ'],
        ['ABI2','CYP20A1'],
        ['ABI2','SMR3B'],
        ['ABI2','TG'],
        ['ABL1','EXOSC2'],
        ['ABL1','SFTPA2'],
        ['ABL1','ZG16B'],
        ['ABL2','SFTPB'],
        ['ABLIM1','BTK'],
        ['ABLIM1','HSF2'],
        ['ABLIM1','HTN3'],
        ['ABLIM1','SFTPB'],
        ['ABLIM1','SMR3B'],
        ['ABO','NLGN4Y'],
        ['ABO','SOX2-OT'],
        ['ABO','TRB@'],
        ['ABR','AC018630.6'],
        ['ABR','CLUH'],
        ['ABR','LYZ'],
        ['ABR','PRB2'],
        ['ABR','PRH1-PRR4'],
        ['ABR','PRH2'],
        ['ABTB2','MMADHC'],
        ['AC000093.1','AC005258.1'],
        ['AC000093.1','OAZ1'],
        ['AC000124.1','ARF5'],
        ['AC002057.1','AC002057.2'],
        ['AC002064.1','CFAP69'],
        ['AC002116.2','WDR62'],
        ['AC002116.7','WDR62'],
        ['AC002398.11','AD000671.6'],
        ['AC002398.12','ACTG1'],
        ['AC002398.12','DES'],
        ['AC002398.12','MYL2'],
        ['AC002398.12','SPARCL1'],
        ['AC002398.2','DES'],
        ['AC002398.2','MYL2'],
        ['AC002398.2','TNNT2'],
        ['AC002480.1','AC002480.2'],
        ['AC002480.2','AC002480.3'],
        ['AC002480.2','STEAP1B'],
        ['AC002480.4','STEAP1B'],
        ['AC002519.6','PPP2CB'],
        ['AC002550.5','IQCK'],
        ['AC003043.2','PLAC8'],
        ['AC003044.1','AC004485.1'],
        ['AC003102.2','YIPF5'],
        ['AC003973.1','AC073539.1'],
        ['AC003973.1','LINC01859'],
        ['AC003973.2','LINC01859'],
        ['AC003973.4','LINC01859'],
        ['AC004066.2','PPA2'],
        ['AC004066.3','PPA2'],
        ['AC004080.5','HOXA5'],
        ['AC004083.1','TMEM64'],
        ['AC004160.1','PHF14'],
        ['AC004237.1','BAIAP2-AS1'],
        ['AC004241.5','BGN'],
        ['AC004381.1','THUMPD1'],
        ['AC004381.7','THUMPD1'],
        ['AC004471.2','DGCR2'],
        ['AC004477.1','COPZ2'],
        ['AC004510.3','ZNF861P'],
        ['AC004520.1','HNRNPA2B1'],
        ['AC004687.2','HSF5'],
        ['AC004687.2','PRH2'],
        ['AC004706.3','FOXP1'],
        ['AC004706.3','MED31'],
        ['AC004846.2','NUMB'],
        ['AC004865.2','PSMB2'],
        ['AC004889.1','AC005304.2'],
        ['AC004889.1','ARHGEF5'],
        ['AC004895.1','USP42'],
        ['AC004895.4','USP42'],
        ['AC004908.2','LINC00115'],
        ['AC004921.2','NPLOC4'],
        ['AC004943.1','AC004943.2'],
        ['AC004951.2','RASA4'],
        ['AC004951.4','RASA4'],
        ['AC004967.1','OR7E38P'],
        ['AC004967.2','CCZ1'],
        ['AC004967.2','CCZ1B'],
        ['AC004967.7','OR7E38P'],
        ['AC004967.7','OR7E7P'],
        ['AC004980.3','AC091390.2'],
        ['AC004980.3','AC211486.3'],
        ['AC004980.3','PMS2'],
        ['AC004980.4','AC091390.2'],
        ['AC004980.4','AC091390.4'],
        ['AC004980.4','AC211486.3'],
        ['AC004980.4','PMS2'],
        ['AC004980.4','PRKRIP1'],
        ['AC004980.4','UPK3B'],
        ['AC004980.8','PMS2'],
        ['AC004980.8','PRKRIP1'],
        ['AC004980.8','UPK3B'],
        ['AC004980.9','PMS2'],
        ['AC005005.4','TXNDC15'],
        ['AC005013.1','AC005162.3'],
        ['AC005035.1','IL1R2'],
        ['AC005037.3','NDUFB3'],
        ['AC005041.1','RTKN'],
        ['AC005071.1','ZCWPW1'],
        ['AC005077.2','LINC00174'],
        ['AC005077.5','SRRM3'],
        ['AC005077.7','GTF2IP7'],
        ['AC005077.7','LINC00174'],
        ['AC005091.1','HIBADH'],
        ['AC005154.1','GGCT'],
        ['AC005154.2','GGCT'],
        ['AC005154.3','GGCT'],
        ['AC005154.7','GGCT'],
        ['AC005176.1','LGALS13'],
        ['AC005224.2','KCNK6'],
        ['AC005224.3','KCNK6'],
        ['AC005229.5','AC006974.2'],
        ['AC005237.4','TG'],
        ['AC005244.1','DYNLT1'],
        ['AC005253.4','CRLF1'],
        ['AC005258.1','BSG'],
        ['AC005258.1','CCNG2'],
        ['AC005258.1','CSNK1G2'],
        ['AC005258.1','DDX5'],
        ['AC005258.1','DOT1L'],
        ['AC005258.1','KLF16'],
        ['AC005258.1','PTMA'],
        ['AC005258.1','SEPT5'],
        ['AC005258.1','SGTA'],
        ['AC005258.1','UBTF'],
        ['AC005261.5','FER'],
        ['AC005261.5','PHACTR3'],
        ['AC005332.4','AMZ2'],
        ['AC005332.4','BPTF'],
        ['AC005332.7','AMZ2'],
        ['AC005332.7','BPTF'],
        ['AC005336.1','IPMK'],
        ['AC005336.4','IPMK'],
        ['AC005336.4','IPMKP1'],
        ['AC005410.1','RPL21'],
        ['AC005479.1','NPC2'],
        ['AC005479.2','NPC2'],
        ['AC005487.1','DNAJB9'],
        ['AC005487.2','DNAJB9'],
        ['AC005532.1','C1GALT1'],
        ['AC005534.6','RPL26'],
        ['AC005540.1','RPL14'],
        ['AC005562.1','AC138207.8'],
        ['AC005562.1','SMURF2'],
        ['AC005578.3','USP34'],
        ['AC005586.2','LRRC61'],
        ['AC005670.3','AC103810.2'],
        ['AC005670.3','ARL17A'],
        ['AC005670.3','ARL17B'],
        ['AC005674.2','WDR1'],
        ['AC005703.1','PMP22'],
        ['AC005703.3','PMP22'],
        ['AC005736.1','AC005736.2'],
        ['AC005740.1','NDUFS1'],
        ['AC005740.3','NDUFS1'],
        ['AC005775.1','ODF3L2'],
        ['AC005775.2','ODF3L2'],
        ['AC005786.7','FZR1'],
        ['AC005829.2','HARS'],
        ['AC005832.1','AC005833.1'],
        ['AC005832.1','NDUFA9'],
        ['AC005832.2','CCDC181'],
        ['AC005856.1','AC005884.1'],
        ['AC005944.2','PLEC'],
        ['AC005944.2','SFTPC'],
        ['AC005993.1','RGS6'],
        ['AC006001.2','RABGEF1'],
        ['AC006001.3','AC027644.4'],
        ['AC006001.3','RABGEF1'],
        ['AC006004.1','FLVCR1'],
        ['AC006008.1','ACTR3C'],
        ['AC006042.2','AC006042.4'],
        ['AC006042.7','AC009473.1'],
        ['AC006050.2','KRT42P'],
        ['AC006059.2','HHATL'],
        ['AC006062.1','AL079338.1'],
        ['AC006064.4','EEF1A1'],
        ['AC006064.4','HBB'],
        ['AC006064.4','NPPA'],
        ['AC006064.6','ERBB2'],
        ['AC006064.6','GTF2I'],
        ['AC006115.2','IGF2'],
        ['AC006116.14','AC006116.22'],
        ['AC006116.15','AC006116.21'],
        ['AC006116.6','AC006116.9'],
        ['AC006230.1','IL6ST'],
        ['AC006262.10','AC006262.5'],
        ['AC006262.10','IGFL2-AS1'],
        ['AC006262.3','IGFL2-AS1'],
        ['AC006269.1','MMD'],
        ['AC006277.2','ZNF77'],
        ['AC006333.1','NDUFA5'],
        ['AC006372.1','AC006372.3'],
        ['AC006453.2','AC245595.1'],
        ['AC006453.2','LINC02802'],
        ['AC006460.2','NEMP2'],
        ['AC006518.7','ARHGAP32'],
        ['AC006518.7','CA6'],
        ['AC006518.7','CEACAM1'],
        ['AC006518.7','CST1'],
        ['AC006518.7','CST4'],
        ['AC006518.7','DMBT1'],
        ['AC006518.7','IGH@'],
        ['AC006518.7','IGK@'],
        ['AC006518.7','LPO'],
        ['AC006518.7','MUC7'],
        ['AC006518.7','PRB1'],
        ['AC006518.7','PRB2'],
        ['AC006518.7','PRB3'],
        ['AC006518.7','PRB4'],
        ['AC006518.7','RBM47'],
        ['AC006518.7','SFRP1'],
        ['AC006518.7','SMR3B'],
        ['AC006518.7','STATH'],
        ['AC006538.1','SLC39A3'],
        ['AC006538.2','SLC39A3'],
        ['AC006547.14','KB-1269D1.8'],
        ['AC006548.3','CCDC90B'],
        ['AC006548.3','DA750114'],
        ['AC006548.3','IGH@'],
        ['AC006548.3','IGK@'],
        ['AC006548.3','TRA@'],
        ['AC006946.2','CECR7'],
        ['AC007000.3','AC211486.3'],
        ['AC007038.1','RPE'],
        ['AC007038.7','RPE'],
        ['AC007099.1','RGS6'],
        ['AC007192.1','AC020656.1'],
        ['AC007192.4','LYZ'],
        ['AC007207.2','AC027288.1'],
        ['AC007207.2','AL117378.1'],
        ['AC007207.2','AP000943.2'],
        ['AC007221.1','AC007221.2'],
        ['AC007221.1','AC012178.1'],
        ['AC007228.11','CFLAR'],
        ['AC007228.9','TOB2'],
        ['AC007229.1','HIKESHI'],
        ['AC007255.8','ZNRF2P2'],
        ['AC007262.2','CCND3'],
        ['AC007342.3','MPHOSPH10'],
        ['AC007342.4','LINC01949'],
        ['AC007342.4','RBL2'],
        ['AC007365.1','AFTPH'],
        ['AC007384.1','AP001360.2'],
        ['AC007384.1','KMT2E'],
        ['AC007388.1','AC020656.1'],
        ['AC007448.2','AC007448.4'],
        ['AC007448.3','AC007448.5'],
        ['AC007529.2','MGST1'],
        ['AC007562.1','UBE2Q2'],
        ['AC007563.4','LINC01921'],
        ['AC007563.5','MYH11'],
        ['AC007566.10','GATAD1'],
        ['AC007663.1','PRODH'],
        ['AC007663.4','DGCR6'],
        ['AC007688.2','AC023790.2'],
        ['AC007834.1','TCHP'],
        ['AC007923.4','SDHA'],
        ['AC007950.1','FBXL22'],
        ['AC008083.2','PCED1B'],
        ['AC008083.3','PCED1B'],
        ['AC008105.3','AC138150.2'],
        ['AC008114.1','PLBD1-AS1'],
        ['AC008119.1','ISCU'],
        ['AC008132.1','AC121320.1'],
        ['AC008132.1','CALM1'],
        ['AC008132.1','RNF223'],
        ['AC008132.13','CALM1'],
        ['AC008132.13','USP18'],
        ['AC008158.1','POLRMT'],
        ['AC008250.1','AC010185.1'],
        ['AC008267.5','TPST1'],
        ['AC008268.2','GPAT2'],
        ['AC008269.1','FASTKD2'],
        ['AC008278.2','DDX1'],
        ['AC008279.1','RPL5'],
        ['AC008280.5','ACYP2'],
        ['AC008427.2','MFF'],
        ['AC008440.5','IGF2'],
        ['AC008440.5','MTCH2'],
        ['AC008440.5','NLRP1'],
        ['AC008440.5','SPARCL1'],
        ['AC008467.1','AC116428.1'],
        ['AC008474.1','NUDT19'],
        ['AC008543.1','ZNF627'],
        ['AC008554.1','ZNF826P'],
        ['AC008554.2','ZNF93'],
        ['AC008555.4','AC008555.5'],
        ['AC008555.4','LINC01801'],
        ['AC008555.5','LINC01801'],
        ['AC008568.1','SPINK1'],
        ['AC008575.1','AC136475.10'],
        ['AC008632.1','AC016573.1'],
        ['AC008676.1','AC016571.1'],
        ['AC008687.7','NTF4'],
        ['AC008746.10','LYZ'],
        ['AC008750.8','NKG7'],
        ['AC008753.1','RPLP1'],
        ['AC008753.3','RPLP1'],
        ['AC008760.2','SERPING1'],
        ['AC008764.1','EPS15L1'],
        ['AC008764.4','SLC35E1'],
        ['AC008770.1','AC011455.2'],
        ['AC008770.1','ZNF625'],
        ['AC008770.1','ZNF625-ZNF20'],
        ['AC008808.2','LINC02106'],
        ['AC008870.1','GTDC1'],
        ['AC008883.3','AC092683.1'],
        ['AC008906.1','CAST'],
        ['AC008993.1','AL928970.1'],
        ['AC008993.2','XXYAC-YRM2039.3'],
        ['AC009014.3','TRPC7'],
        ['AC009022.1','AC009065.3'],
        ['AC009022.1','NPIPB15'],
        ['AC009039.1','LINC02180'],
        ['AC009041.2','LMBR1'],
        ['AC009053.3','AC009060.1'],
        ['AC009065.3','PDXDC1'],
        ['AC009065.3','PDXDC2P'],
        ['AC009070.1','AC092718.2'],
        ['AC009070.1','AC092718.8'],
        ['AC009078.2','AL590556.3'],
        ['AC009078.2','PRSS1'],
        ['AC009078.2','PRSS2'],
        ['AC009078.2','TRB@'],
        ['AC009081.2','APLF'],
        ['AC009081.2','SPTLC1'],
        ['AC009088.2','TNFSF10'],
        ['AC009090.6','RSPRY1'],
        ['AC009093.1','CSF2RA'],
        ['AC009093.10','RRN3P2'],
        ['AC009093.10','SNX29'],
        ['AC009093.2','BANP'],
        ['AC009093.9','AC092338.1'],
        ['AC009093.9','CDR2'],
        ['AC009107.1','CCDC113'],
        ['AC009126.1','ERAP1'],
        ['AC009133.14','SFTPC'],
        ['AC009133.22','LYZ'],
        ['AC009135.2','ASNS'],
        ['AC009163.2','AC009163.5'],
        ['AC009163.5','CHST5'],
        ['AC009171.2','SRL'],
        ['AC009228.1','FAM228A'],
        ['AC009264.1','HACD4'],
        ['AC009264.1','RAB10'],
        ['AC009271.1','LINC01905'],
        ['AC009275.1','FGD4'],
        ['AC009292.2','PIAS1'],
        ['AC009309.1','MRPL35'],
        ['AC009318.1','ERGIC2'],
        ['AC009362.2','RPL27'],
        ['AC009404.1','CRTAP'],
        ['AC009413.2','RPS12'],
        ['AC009704.3','DIPK1C'],
        ['AC009812.4','ZBTB10'],
        ['AC009948.1','DFNB59'],
        ['AC009948.4','THEM4'],
        ['AC009951.1','ZEB2'],
        ['AC009961.1','BAZ2B'],
        ['AC009961.3','BAZ2B'],
        ['AC010086.3','RBMY2EP'],
        ['AC010095.6','COX7B'],
        ['AC010136.2','RUFY4'],
        ['AC010149.4','CLASP2'],
        ['AC010149.4','MORC1'],
        ['AC010168.1','PLBD1-AS1'],
        ['AC010186.2','CLEC2D'],
        ['AC010186.2','DCK'],
        ['AC010186.2','GOT2'],
        ['AC010186.4','DCK'],
        ['AC010186.4','GOT2'],
        ['AC010203.1','DEPDC4'],
        ['AC010247.1','AC010247.2'],
        ['AC010273.2','LINC02198'],
        ['AC010273.3','LINC02198'],
        ['AC010325.1','C19ORF48'],
        ['AC010326.4','ZNF587'],
        ['AC010327.1','CLCN6'],
        ['AC010327.1','DES'],
        ['AC010327.1','NPPA-AS1'],
        ['AC010327.2','CLCN6'],
        ['AC010327.2','DES'],
        ['AC010327.2','MYL7'],
        ['AC010329.1','AC061975.7'],
        ['AC010329.1','AC079804.3'],
        ['AC010329.1','AC127526.5'],
        ['AC010329.1','AL392046.2'],
        ['AC010329.1','ERVE-1'],
        ['AC010329.1','LINC02635'],
        ['AC010329.1','UBXN8'],
        ['AC010359.3','TMSB4X'],
        ['AC010435.1','AMD1'],
        ['AC010504.2','IGF2'],
        ['AC010518.3','LILRA2'],
        ['AC010524.2','TEAD2'],
        ['AC010538.1','UGDH'],
        ['AC010542.1','CMTM4'],
        ['AC010542.5','CMTM4'],
        ['AC010615.1','ZNF738'],
        ['AC010618.1','FAM129C'],
        ['AC010636.1','LINC02701'],
        ['AC010642.1','ZSCAN22'],
        ['AC010643.1','TEAD2'],
        ['AC010677.5','RPL23'],
        ['AC010680.1','MYH6'],
        ['AC010731.4','ADAM23'],
        ['AC010735.2','RHBDD1'],
        ['AC010878.3','STRAP'],
        ['AC011313.1','APPL2'],
        ['AC011317.1','SIAH2'],
        ['AC011346.1','DA750114'],
        ['AC011346.1','IGH@'],
        ['AC011346.1','TRA@'],
        ['AC011379.2','CYSTM1'],
        ['AC011410.1','SMIM23'],
        ['AC011416.3','AC011416.4'],
        ['AC011445.1','GMFG'],
        ['AC011446.2','PRB1'],
        ['AC011450.1','SLC6A16'],
        ['AC011451.1','HCAR1'],
        ['AC011462.1','PTMA'],
        ['AC011468.4','TMEM50B'],
        ['AC011481.3','LINC01602'],
        ['AC011487.1','ZNF765'],
        ['AC011487.3','OSTC'],
        ['AC011491.3','ACER1'],
        ['AC011513.3','LYZ'],
        ['AC011513.3','PIGR'],
        ['AC011523.1','KLK3'],
        ['AC011523.2','KLK3'],
        ['AC011558.5','CTSB'],
        ['AC011591.1','SFTPB'],
        ['AC011603.2','MBP'],
        ['AC011603.3','MBP'],
        ['AC011611.5','DENR'],
        ['AC011752.1','NASP'],
        ['AC011767.1','GOLGA6L2'],
        ['AC011825.1','RPS15A'],
        ['AC011840.1','KRT42P'],
        ['AC011997.1','PLCB1'],
        ['AC012020.1','CD47'],
        ['AC012101.1','AC012101.2'],
        ['AC012103.1','NEAT1'],
        ['AC012170.2','USP50'],
        ['AC012309.5','LINC00665'],
        ['AC012313.1','CNGB3'],
        ['AC012322.1','EBPL'],
        ['AC012349.1','IGH@'],
        ['AC012435.2','UBL7-AS1'],
        ['AC012441.2','AC018638.5'],
        ['AC012442.1','CHCHD5'],
        ['AC012442.6','CHCHD5'],
        ['AC012485.1','CSAD'],
        ['AC012485.1','TARDBP'],
        ['AC012485.2','PER2'],
        ['AC012560.1','UST'],
        ['AC012636.1','SERINC5'],
        ['AC012645.3','PTMA'],
        ['AC013244.1','DIP2B'],
        ['AC013265.1','AC062021.1'],
        ['AC013394.1','B2M'],
        ['AC013394.1','ELANE'],
        ['AC013394.1','ZBTB7A'],
        ['AC013457.1','AP000688.2'],
        ['AC013463.2','AC019181.2'],
        ['AC015468.1','LINC02153'],
        ['AC015468.2','LINC02153'],
        ['AC015631.1','AC079380.1'],
        ['AC015688.3','AC015688.4'],
        ['AC015688.4','AC098850.5'],
        ['AC015688.4','NOS2'],
        ['AC015689.1','RRM1'],
        ['AC015689.2','RDX'],
        ['AC015691.1','HBE1'],
        ['AC015802.4','AC015802.6'],
        ['AC015802.4','ST6GALNAC2'],
        ['AC015802.6','CTSS'],
        ['AC015802.6','CYGB'],
        ['AC015802.7','ST6GALNAC1'],
        ['AC015813.1','SRSF1'],
        ['AC015849.1','TAF15'],
        ['AC015849.16','CCL5'],
        ['AC015849.16','LRRC37A9P'],
        ['AC015849.2','TAF15'],
        ['AC015849.5','LRRC37A9P'],
        ['AC015883.1','CRLF2'],
        ['AC015908.2','TMEM220-AS1'],
        ['AC015911.10','SLFN13'],
        ['AC015911.11','SLFN13'],
        ['AC015922.3','IL6ST'],
        ['AC015922.4','IL6ST'],
        ['AC015971.2','B2M'],
        ['AC015977.2','CIB4'],
        ['AC015987.1','IGF2'],
        ['AC016044.1','ONECUT1'],
        ['AC016257.1','ALDH1L2'],
        ['AC016582.2','WDR87'],
        ['AC016586.1','ZBTB7A'],
        ['AC016588.2','AL078621.3'],
        ['AC016705.1','RNF152'],
        ['AC016717.1','NYAP2'],
        ['AC016722.2','SOCS5'],
        ['AC016730.1','AC073062.1'],
        ['AC016745.1','AL353608.2'],
        ['AC016747.4','AHSA2'],
        ['AC016821.1','TSPAN15'],
        ['AC016831.1','LINC-PINT'],
        ['AC016876.2','DDX5'],
        ['AC016876.2','EIF4A2'],
        ['AC016876.2','IGK@'],
        ['AC016876.2','LYZ'],
        ['AC016876.2','MPDU1'],
        ['AC016907.2','ALK'],
        ['AC016907.2','GSN'],
        ['AC016907.3','ALK'],
        ['AC016912.1','RHOQ'],
        ['AC017002.6','CYTOR'],
        ['AC017015.2','GBE1'],
        ['AC017083.3','C1D'],
        ['AC017083.3','CNRIP1'],
        ['AC017083.4','C1D'],
        ['AC017083.4','CNRIP1'],
        ['AC017091.1','AC097501.1'],
        ['AC017091.1','LINC02619'],
        ['AC017104.1','ARMC9'],
        ['AC017104.5','LINC00471'],
        ['AC017104.6','NMUR1'],
        ['AC018521.5','SP2-AS1'],
        ['AC018616.1','BEST1'],
        ['AC018629.1','AC110769.2'],
        ['AC018630.2','AC020656.1'],
        ['AC018630.2','AMY2B'],
        ['AC018630.2','BPIFA2'],
        ['AC018630.2','CA6'],
        ['AC018630.2','CCL28'],
        ['AC018630.2','CD44'],
        ['AC018630.2','CD74'],
        ['AC018630.2','CST1'],
        ['AC018630.2','CST2'],
        ['AC018630.2','CST4'],
        ['AC018630.2','CX3CL1'],
        ['AC018630.2','DGCR2'],
        ['AC018630.2','DIAPH2'],
        ['AC018630.2','DMBT1'],
        ['AC018630.2','DSG2'],
        ['AC018630.2','DYNC1H1'],
        ['AC018630.2','EEF1A1'],
        ['AC018630.2','EEF2'],
        ['AC018630.2','GAPDH'],
        ['AC018630.2','GCC2'],
        ['AC018630.2','GNAS'],
        ['AC018630.2','GNE'],
        ['AC018630.2','H1F0'],
        ['AC018630.2','HTN3'],
        ['AC018630.2','IGH@'],
        ['AC018630.2','IGHA1'],
        ['AC018630.2','IGK@'],
        ['AC018630.2','IL6ST'],
        ['AC018630.2','KIAA1324'],
        ['AC018630.2','LPO'],
        ['AC018630.2','LYZ'],
        ['AC018630.2','MUC5B'],
        ['AC018630.2','MUC7'],
        ['AC018630.2','MYH9'],
        ['AC018630.2','NCOR2'],
        ['AC018630.2','NFIB'],
        ['AC018630.2','ODAM'],
        ['AC018630.2','PARM1'],
        ['AC018630.2','PER1'],
        ['AC018630.2','PHLDA1'],
        ['AC018630.2','PIGR'],
        ['AC018630.2','PIK3R1'],
        ['AC018630.2','PRR27'],
        ['AC018630.2','PSMF1'],
        ['AC018630.2','RBM47'],
        ['AC018630.2','RPL3'],
        ['AC018630.2','SFRP1'],
        ['AC018630.2','SIX4'],
        ['AC018630.2','SLC20A2'],
        ['AC018630.2','SLC6A8'],
        ['AC018630.2','SMR3B'],
        ['AC018630.2','SND1'],
        ['AC018630.2','SORL1'],
        ['AC018630.2','SQSTM1'],
        ['AC018630.2','STATH'],
        ['AC018630.2','ZG16B'],
        ['AC018630.6','AC020656.1'],
        ['AC018630.6','AMY2B'],
        ['AC018630.6','CA6'],
        ['AC018630.6','CCL28'],
        ['AC018630.6','CD44'],
        ['AC018630.6','CD74'],
        ['AC018630.6','CST1'],
        ['AC018630.6','CST2'],
        ['AC018630.6','CST4'],
        ['AC018630.6','CX3CL1'],
        ['AC018630.6','DGCR2'],
        ['AC018630.6','DSG2'],
        ['AC018630.6','EEF1A1'],
        ['AC018630.6','EEF2'],
        ['AC018630.6','GAPDH'],
        ['AC018630.6','GNAS'],
        ['AC018630.6','GNE'],
        ['AC018630.6','H1F0'],
        ['AC018630.6','IGH@'],
        ['AC018630.6','IGK@'],
        ['AC018630.6','IL6ST'],
        ['AC018630.6','ITPR1'],
        ['AC018630.6','KIAA1324'],
        ['AC018630.6','LPO'],
        ['AC018630.6','LYZ'],
        ['AC018630.6','MLLT6'],
        ['AC018630.6','MUC7'],
        ['AC018630.6','MYH9'],
        ['AC018630.6','ODAM'],
        ['AC018630.6','PER1'],
        ['AC018630.6','PHLDA1'],
        ['AC018630.6','PIGR'],
        ['AC018630.6','PIK3R1'],
        ['AC018630.6','PRB1'],
        ['AC018630.6','PRB2'],
        ['AC018630.6','PRB3'],
        ['AC018630.6','PRB4'],
        ['AC018630.6','PRR27'],
        ['AC018630.6','PSMF1'],
        ['AC018630.6','RBM47'],
        ['AC018630.6','RPL3'],
        ['AC018630.6','SFRP1'],
        ['AC018630.6','SIX4'],
        ['AC018630.6','SLC20A2'],
        ['AC018630.6','SLC6A8'],
        ['AC018630.6','SMR3B'],
        ['AC018630.6','SND1'],
        ['AC018630.6','SORL1'],
        ['AC018630.6','SQSTM1'],
        ['AC018630.6','STATH'],
        ['AC018630.6','TRPV6'],
        ['AC018630.6','ZG16B'],
        ['AC018638.5','AL606534.4'],
        ['AC018638.5','AL606534.5'],
        ['AC018647.3','HERPUD2'],
        ['AC018688.1','ZNF285'],
        ['AC018695.3','COX4I1'],
        ['AC018731.3','NEB'],
        ['AC018737.3','TSN'],
        ['AC018754.1','CCNH'],
        ['AC018797.2','CISD2'],
        ['AC018809.2','MORF4L1'],
        ['AC018845.3','PRXL2A'],
        ['AC018890.6','TG'],
        ['AC019077.1','TJP1'],
        ['AC019131.2','ADH5'],
        ['AC019133.1','LINC02494'],
        ['AC019172.2','DCDC2C'],
        ['AC019349.1','CRNN'],
        ['AC019349.1','GSN'],
        ['AC019349.1','KRT4'],
        ['AC019349.1','KRT6A'],
        ['AC019349.1','KRT6B'],
        ['AC019349.1','S100A9'],
        ['AC019349.1','SPINK5'],
        ['AC019349.5','AL513122.1'],
        ['AC019349.5','AL513122.2'],
        ['AC019349.5','CRNN'],
        ['AC019349.5','FLNA'],
        ['AC019349.5','GSN'],
        ['AC019349.5','KRT15'],
        ['AC019349.5','KRT4'],
        ['AC019349.5','KRT6A'],
        ['AC019349.5','NDE1'],
        ['AC019349.5','S100A9'],
        ['AC019349.5','SPINK5'],
        ['AC020656.1','ACTG1'],
        ['AC020656.1','AHNAK'],
        ['AC020656.1','AKR1B10'],
        ['AC020656.1','AP002956.1'],
        ['AC020656.1','APP'],
        ['AC020656.1','CCL5'],
        ['AC020656.1','CEP68'],
        ['AC020656.1','CLDN18'],
        ['AC020656.1','CRISP3'],
        ['AC020656.1','CTSG'],
        ['AC020656.1','DEPDC4'],
        ['AC020656.1','EFCAB11'],
        ['AC020656.1','FAM120AOS'],
        ['AC020656.1','FAXDC2'],
        ['AC020656.1','FBXO6'],
        ['AC020656.1','FGFR2'],
        ['AC020656.1','FOXP1'],
        ['AC020656.1','GOLGB1'],
        ['AC020656.1','HP1BP3'],
        ['AC020656.1','HSD17B12'],
        ['AC020656.1','IGK@'],
        ['AC020656.1','IL3RA'],
        ['AC020656.1','IVD'],
        ['AC020656.1','LINC01285'],
        ['AC020656.1','LINC01550'],
        ['AC020656.1','LPO'],
        ['AC020656.1','MAP4K3-DT'],
        ['AC020656.1','MBTD1'],
        ['AC020656.1','MCAM'],
        ['AC020656.1','MPPED2'],
        ['AC020656.1','NEBL'],
        ['AC020656.1','NT5C2'],
        ['AC020656.1','PEAK1'],
        ['AC020656.1','PGC'],
        ['AC020656.1','PIK3R2'],
        ['AC020656.1','PKM'],
        ['AC020656.1','POMGNT1'],
        ['AC020656.1','PRH1'],
        ['AC020656.1','PRKAR2A'],
        ['AC020656.1','RPL8'],
        ['AC020656.1','S100A8'],
        ['AC020656.1','SFTPB'],
        ['AC020656.1','SLC25A37'],
        ['AC020656.1','SLC2A3'],
        ['AC020656.1','SLC43A2'],
        ['AC020656.1','SLC4A1'],
        ['AC020656.1','SMARCA4'],
        ['AC020656.1','SOX9-AS1'],
        ['AC020656.1','SPTBN1'],
        ['AC020656.1','TAF15'],
        ['AC020656.1','TARS2'],
        ['AC020656.1','TMBIM6'],
        ['AC020656.1','TMEM123'],
        ['AC020656.1','TRA@'],
        ['AC020656.1','UBE2G2'],
        ['AC020656.1','WWOX'],
        ['AC020656.1','ZKSCAN1'],
        ['AC020765.2','NUPR1'],
        ['AC020915.1','ZSCAN22'],
        ['AC020915.2','ZSCAN22'],
        ['AC020915.5','ZSCAN22'],
        ['AC020915.6','ZSCAN22'],
        ['AC020916.1','PRKCSH'],
        ['AC020928.1','HKR1'],
        ['AC020928.1','LINC00665'],
        ['AC020928.1','ZNF567'],
        ['AC020928.2','ZNF567'],
        ['AC020956.3','LYPD4'],
        ['AC020978.3','ESRP2'],
        ['AC020978.4','ESRP2'],
        ['AC020978.9','NEAT1'],
        ['AC021054.1','CPSF6'],
        ['AC021087.5','AC026412.1'],
        ['AC021087.5','AC073210.3'],
        ['AC021087.5','AC091849.2'],
        ['AC021087.5','AC124944.2'],
        ['AC021087.5','AC132008.2'],
        ['AC021087.5','DES'],
        ['AC021087.5','ERBB2'],
        ['AC021087.5','MUC20-OT1'],
        ['AC021097.2','ICA1'],
        ['AC021171.1','FAM133B'],
        ['AC021517.2','SLC14A2-AS1'],
        ['AC021660.4','CPOX'],
        ['AC021739.2','AC021739.3'],
        ['AC021851.1','NUP35'],
        ['AC022028.2','GRID1-AS1'],
        ['AC022034.1','AC022034.3'],
        ['AC022137.3','ZNF765'],
        ['AC022145.1','STK33'],
        ['AC022167.2','AC022167.4'],
        ['AC022182.2','NASP'],
        ['AC022210.2','KCMF1'],
        ['AC022217.1','FAM58A'],
        ['AC022400.3','BMS1'],
        ['AC022400.3','PARG'],
        ['AC022400.3','TIMM23B'],
        ['AC022400.4','BMS1'],
        ['AC022400.4','TIMM23B-AGAP6'],
        ['AC022400.5','BMS1'],
        ['AC022400.5','LINC00843'],
        ['AC022400.5','PARG'],
        ['AC022400.5','TIMM23B'],
        ['AC022400.6','BMS1'],
        ['AC022400.6','PARG'],
        ['AC022400.6','TIMM23B'],
        ['AC022733.1','AC022733.2'],
        ['AC022733.1','AC087518.1'],
        ['AC023055.1','AC073063.1'],
        ['AC023157.1','IFITM3'],
        ['AC023282.1','PLPP4'],
        ['AC023509.1','SMR3B'],
        ['AC023510.1','CASC1'],
        ['AC023538.1','SMC5'],
        ['AC023644.1','BEST1'],
        ['AC023790.2','GPRC5D-AS1'],
        ['AC023824.5','AC023824.6'],
        ['AC023824.6','AC105916.1'],
        ['AC023906.3','TRIM69'],
        ['AC024028.1','AC073316.1'],
        ['AC024084.1','AC078845.1'],
        ['AC024559.2','AC080129.1'],
        ['AC024560.3','AC024937.3'],
        ['AC024560.3','AC069213.1'],
        ['AC024560.3','AC093642.1'],
        ['AC024560.3','SDHA'],
        ['AC024560.4','AC026412.1'],
        ['AC024560.4','BBS2'],
        ['AC024575.1','TMEM205'],
        ['AC024587.1','USP7'],
        ['AC024592.2','LINC01356'],
        ['AC024937.3','AC026412.1'],
        ['AC024937.3','AC093642.1'],
        ['AC024937.3','PDCD6'],
        ['AC024940.1','AC092821.1'],
        ['AC024940.1','AC092821.3'],
        ['AC024940.2','AC092666.1'],
        ['AC024940.2','AC092821.1'],
        ['AC024940.2','AC092821.3'],
        ['AC024940.2','SSUH2'],
        ['AC025048.2','AC243829.2'],
        ['AC025161.1','RANBP1'],
        ['AC025171.1','KCNK6'],
        ['AC025171.2','ZNF131'],
        ['AC025244.2','AC096759.1'],
        ['AC025262.1','TBC1D30'],
        ['AC025279.1','SNX29'],
        ['AC025279.2','SNX29'],
        ['AC025287.1','CHST5'],
        ['AC025287.1','CHST6'],
        ['AC025335.1','DNAH14'],
        ['AC025335.1','FLG2'],
        ['AC025335.1','PLXDC2'],
        ['AC025566.1','LINC02032'],
        ['AC025575.2','TSPAN8'],
        ['AC025580.3','SMG1'],
        ['AC025594.1','ATP6V1F'],
        ['AC025884.1','RERE'],
        ['AC026150.1','GOLGA8M'],
        ['AC026150.8','GOLGA8M'],
        ['AC026167.1','AC027119.1'],
        ['AC026202.3','TEX101'],
        ['AC026412.1','AC124944.3'],
        ['AC026412.1','PDCD6'],
        ['AC026412.1','SDHA'],
        ['AC026471.2','AHSP'],
        ['AC026704.1','MTREX'],
        ['AC026740.3','BRD9'],
        ['AC026979.4','SLC25A3'],
        ['AC027020.2','PRKX'],
        ['AC027031.2','OXR1'],
        ['AC027088.5','AL355377.4'],
        ['AC027288.3','PAWR'],
        ['AC027290.2','LYZ'],
        ['AC027290.2','OSBPL2'],
        ['AC027343.1','AC027343.2'],
        ['AC027343.1','LINC02123'],
        ['AC027544.2','RPL7L1'],
        ['AC027559.1','AC090825.1'],
        ['AC027612.2','IGK@'],
        ['AC027612.6','IGK@'],
        ['AC027644.4','LINC00174'],
        ['AC027682.1','SFTPC'],
        ['AC034205.2','SLC36A2'],
        ['AC036108.3','SYNM'],
        ['AC037198.1','FSIP1'],
        ['AC037487.4','SLC16A6'],
        ['AC040162.1','CELA2A'],
        ['AC040162.1','CELA2B'],
        ['AC040162.1','CTRB1'],
        ['AC040162.1','CTRB2'],
        ['AC040162.1','PRSS1'],
        ['AC040162.1','PRSS2'],
        ['AC040162.1','TRB@'],
        ['AC040170.1','LINC01081'],
        ['AC044810.2','OVCH2'],
        ['AC044860.1','AC243562.1'],
        ['AC044860.1','GOLGA2P3Y'],
        ['AC044860.1','GOLGA8F'],
        ['AC044860.1','GOLGA8G'],
        ['AC044860.1','GOLGA8S'],
        ['AC048337.1','DGKD'],
        ['AC048338.1','CLIP1'],
        ['AC048338.2','TG'],
        ['AC053503.4','KRT13'],
        ['AC053503.4','MYH6'],
        ['AC053503.4','MYH7'],
        ['AC053503.4','MYL2'],
        ['AC053503.4','NPPA'],
        ['AC053503.4','TNNI3'],
        ['AC053503.4','TPM1'],
        ['AC053503.6','CRNN'],
        ['AC053503.6','DLGAP4-AS1'],
        ['AC053503.6','KRT13'],
        ['AC053503.6','MYH6'],
        ['AC053503.6','MYL2'],
        ['AC053503.6','NPPA'],
        ['AC053503.6','PPP1R12B'],
        ['AC053503.6','SYNPO'],
        ['AC053503.6','TNNC1'],
        ['AC053503.6','TNNI3'],
        ['AC053503.6','TNNT2'],
        ['AC053503.6','TPM1'],
        ['AC053545.1','BBS12'],
        ['AC055713.1','WSB1'],
        ['AC055720.1','SSPN'],
        ['AC055811.1','MPRIP'],
        ['AC058791.1','LINC00513'],
        ['AC060780.1','AC096637.2'],
        ['AC061975.7','AL035661.1'],
        ['AC061975.8','NLK'],
        ['AC061975.8','PYY2'],
        ['AC063926.3','PIWIL1'],
        ['AC063944.1','LINC00882'],
        ['AC063979.2','LINC02056'],
        ['AC064807.1','AC064807.2'],
        ['AC064862.1','AC064862.6'],
        ['AC067930.1','ANKRD19P'],
        ['AC067930.1','HBB'],
        ['AC067930.2','ANKRD19P'],
        ['AC067930.2','HBB'],
        ['AC067930.5','ANKRD19P'],
        ['AC067930.5','HBB'],
        ['AC067930.5','KRT19'],
        ['AC067930.6','ANKRD19P'],
        ['AC067930.6','HBB'],
        ['AC068152.1','AC091132.1'],
        ['AC068152.1','ARL17A'],
        ['AC068152.1','ARL17B'],
        ['AC068228.3','TBC1D31'],
        ['AC068279.2','IGK@'],
        ['AC068389.1','SLC5A12'],
        ['AC068446.1','PAK2'],
        ['AC068491.1','PAFAH1B1'],
        ['AC068580.4','AC132217.2'],
        ['AC068580.4','ALB'],
        ['AC068580.4','IGF2'],
        ['AC068580.4','LSP1'],
        ['AC068580.4','PRB1'],
        ['AC068580.4','PRB2'],
        ['AC068580.4','SFTPA2'],
        ['AC068580.4','SMR3B'],
        ['AC068580.7','AC139143.2'],
        ['AC068587.4','AC079781.5'],
        ['AC068587.4','AC127526.2'],
        ['AC068587.4','LINC00937'],
        ['AC068587.6','AC079781.5'],
        ['AC068587.6','AC127526.4'],
        ['AC068587.6','LINC00937'],
        ['AC068672.2','TRMT12'],
        ['AC068700.2','PKIA-AS1'],
        ['AC068756.1','AC108690.1'],
        ['AC068768.1','MPHOSPH9'],
        ['AC068774.1','DENND5B'],
        ['AC068790.8','KMT5A'],
        ['AC068831.1','CERK'],
        ['AC068831.3','CERK'],
        ['AC068831.7','B2M'],
        ['AC068831.8','RCCD1'],
        ['AC068896.1','MTHFS'],
        ['AC068896.1','ST20-MTHFS'],
        ['AC068987.1','AC068987.3'],
        ['AC068987.2','FIGNL2-DT'],
        ['AC068989.1','DDX60'],
        ['AC068993.2','SYT1'],
        ['AC069208.1','SOX5'],
        ['AC069280.2','UST'],
        ['AC069281.1','AGFG2'],
        ['AC069281.2','CCNG2'],
        ['AC069288.1','MAFK'],
        ['AC069363.1','CCL4'],
        ['AC069503.4','HPD'],
        ['AC072062.1','GOLGB1'],
        ['AC073063.1','SARNP'],
        ['AC073063.10','SARNP'],
        ['AC073073.1','RNF181'],
        ['AC073133.1','LINC01006'],
        ['AC073210.3','SDHA'],
        ['AC073263.1','TET3'],
        ['AC073264.3','AC099548.2'],
        ['AC073264.3','TCAF1'],
        ['AC073270.1','LINC01005'],
        ['AC073270.2','ZNF736'],
        ['AC073288.2','SKIL'],
        ['AC073343.13','IGF2'],
        ['AC073343.13','ZNF316'],
        ['AC073343.2','ZNF316'],
        ['AC073348.2','AL132800.1'],
        ['AC073349.5','AC092161.1'],
        ['AC073365.1','BGN'],
        ['AC073529.1','MID1'],
        ['AC073578.1','LINC01257'],
        ['AC073585.2','CEL'],
        ['AC073585.2','DOCK1'],
        ['AC073585.2','GP2'],
        ['AC073585.2','PNLIP'],
        ['AC073610.3','ARF1'],
        ['AC073610.5','IGH@'],
        ['AC073842.19','LAMTOR4'],
        ['AC074008.1','ALMS1'],
        ['AC074043.1','PTPN11'],
        ['AC074051.2','ASNS'],
        ['AC074099.1','LINC01876'],
        ['AC074117.10','GPD1L'],
        ['AC074141.3','ZNF577'],
        ['AC074250.2','AFM'],
        ['AC078795.1','KRT19'],
        ['AC078842.1','PTN'],
        ['AC078842.3','DGKI'],
        ['AC078842.3','PTN'],
        ['AC078883.1','AC078883.3'],
        ['AC078899.3','ZNF66'],
        ['AC079142.1','HMGXB4'],
        ['AC079193.2','MICU3'],
        ['AC079228.1','RTN3'],
        ['AC079316.1','AL139353.3'],
        ['AC079328.2','STARD5'],
        ['AC079354.5','DAZAP2P1'],
        ['AC079414.3','IFI44L'],
        ['AC079753.5','IL37'],
        ['AC079767.4','PPP1R14BP2'],
        ['AC079781.5','LINC00937'],
        ['AC079781.5','RBFOX1'],
        ['AC079790.2','FMNL2'],
        ['AC079804.3','AC127526.5'],
        ['AC079804.3','LINC02635'],
        ['AC079848.1','AC108134.3'],
        ['AC079921.2','KLHL5'],
        ['AC080128.2','AL512356.3'],
        ['AC080162.1','IAH1'],
        ['AC083841.2','MPHOSPH8'],
        ['AC083862.1','RNF14'],
        ['AC083862.3','TMEM140'],
        ['AC083900.1','PLEKHM3'],
        ['AC083964.2','TDRP'],
        ['AC084024.3','ASH2L'],
        ['AC084032.1','CNOT2'],
        ['AC084082.1','TRIM55'],
        ['AC084219.2','ZNF224'],
        ['AC084734.1','CA2'],
        ['AC084809.1','AC084809.2'],
        ['AC087071.2','SSMEM1'],
        ['AC087190.3','LYZ'],
        ['AC087190.4','LYZ'],
        ['AC087203.3','AF228730.5'],
        ['AC087203.3','AF228730.7'],
        ['AC087280.2','MRPL17'],
        ['AC087283.1','PNO1'],
        ['AC087380.14','UBQLNL'],
        ['AC087457.1','MYH7'],
        ['AC087457.1','NPPA'],
        ['AC087463.4','PWRN1'],
        ['AC087463.4','PWRN3'],
        ['AC087473.1','DA750114'],
        ['AC087473.1','IGH@'],
        ['AC087473.1','PSMB2'],
        ['AC087473.1','TRA@'],
        ['AC087473.1','USP33'],
        ['AC087477.2','ANXA4'],
        ['AC087477.2','NR2F2'],
        ['AC087481.2','FAN1'],
        ['AC087481.2','HERC2'],
        ['AC087500.1','RABEP1'],
        ['AC087533.1','RABGAP1'],
        ['AC087639.1','C1R'],
        ['AC087650.1','RPL29'],
        ['AC087672.2','LY96'],
        ['AC087672.2','STAU2'],
        ['AC087672.3','LY96'],
        ['AC087897.2','CPNE8'],
        ['AC089984.3','CCDC9B'],
        ['AC090001.1','NTN4'],
        ['AC090001.1','RNASE10'],
        ['AC090114.3','TLK2'],
        ['AC090371.1','AC090371.2'],
        ['AC090409.1','AC090409.2'],
        ['AC090409.2','AC105094.2'],
        ['AC090515.4','SLTM'],
        ['AC090517.5','ZNF280D'],
        ['AC091057.1','AC091057.6'],
        ['AC091057.1','OTUD7A'],
        ['AC091057.2','GOLGA8N'],
        ['AC091057.3','GOLGA8N'],
        ['AC091096.1','AC091096.2'],
        ['AC091132.1','ARL17B'],
        ['AC091132.5','LRRC37A3'],
        ['AC091167.3','GDPGP1'],
        ['AC091390.2','AC211486.3'],
        ['AC091390.2','PMS2'],
        ['AC091390.3','STAG3'],
        ['AC091390.3','STAG3L5P'],
        ['AC091390.3','STAG3L5P-PVRIG2P-PILRB'],
        ['AC091390.4','AC211429.1'],
        ['AC091390.4','AC211486.3'],
        ['AC091390.4','PMS2'],
        ['AC091390.4','POM121'],
        ['AC091390.4','POM121C'],
        ['AC091390.4','RBAK'],
        ['AC091493.2','DAZL'],
        ['AC091564.2','TAF10'],
        ['AC091849.2','SDHA'],
        ['AC091951.3','AC138649.1'],
        ['AC091951.3','NIPA1'],
        ['AC091978.1','LINC00680'],
        ['AC091982.1','AC091982.4'],
        ['AC091982.3','RPLP1P6'],
        ['AC092079.1','DDX18'],
        ['AC092128.1','AC138627.1'],
        ['AC092162.2','SPOPL'],
        ['AC092284.1','CRLF3'],
        ['AC092338.1','RRN3P2'],
        ['AC092376.1','MAF'],
        ['AC092418.2','DNAH12'],
        ['AC092431.1','ANXA4'],
        ['AC092435.1','AC092435.2'],
        ['AC092437.1','STX18-AS1'],
        ['AC092506.1','THAP3'],
        ['AC092683.1','TRA@'],
        ['AC092683.1','TRB@'],
        ['AC092691.1','LSAMP'],
        ['AC092691.3','TRIOBP'],
        ['AC092718.3','NDUFS1'],
        ['AC092718.8','CMC2'],
        ['AC092718.8','NDUFS1'],
        ['AC092718.8','TWSG1'],
        ['AC092745.1','AF228730.5'],
        ['AC092745.5','AC108519.1'],
        ['AC092745.5','AC145124.1'],
        ['AC092745.5','AF228730.5'],
        ['AC092746.1','CLEC4E'],
        ['AC092747.2','MED21'],
        ['AC092798.2','RPL32'],
        ['AC092809.4','DEGS1'],
        ['AC092813.2','NEAT1'],
        ['AC092813.2','ST6GALNAC3'],
        ['AC092821.1','AC141557.1'],
        ['AC092821.1','TPT1-AS1'],
        ['AC092821.2','RANBP6'],
        ['AC092821.3','CLEC2D'],
        ['AC092821.3','DCK'],
        ['AC092821.3','GOT2'],
        ['AC092821.3','TPT1-AS1'],
        ['AC092902.2','FAM85B'],
        ['AC092902.6','AC108519.1'],
        ['AC092920.1','LINC01208'],
        ['AC092933.2','SLC37A2'],
        ['AC092933.3','RPS29'],
        ['AC092944.1','AC104411.1'],
        ['AC092958.2','LINC02032'],
        ['AC092958.3','LINC02032'],
        ['AC092969.1','IQUB'],
        ['AC092979.2','GYG1'],
        ['AC093010.2','TMPRSS11B'],
        ['AC093106.5','NDUFB9'],
        ['AC093142.1','RPL34'],
        ['AC093157.1','SLC30A7'],
        ['AC093157.2','SLC30A7'],
        ['AC093162.2','RETSAT'],
        ['AC093392.1','SEPT7'],
        ['AC093484.3','ANKRD1'],
        ['AC093484.3','HBA2'],
        ['AC093484.3','HBB'],
        ['AC093484.3','UBA52'],
        ['AC093484.3','UBC'],
        ['AC093484.4','ANKRD1'],
        ['AC093484.4','HBA2'],
        ['AC093484.4','HBB'],
        ['AC093484.4','UBA52'],
        ['AC093484.4','UBC'],
        ['AC093512.2','PPP4C'],
        ['AC093512.2','SMR3B'],
        ['AC093525.2','DDX5'],
        ['AC093536.1','EBPL'],
        ['AC093627.3','HABP4'],
        ['AC093698.4','CRYGA'],
        ['AC093698.4','CRYGD'],
        ['AC093720.1','UGT2A3'],
        ['AC093752.1','AC138393.1'],
        ['AC093752.1','PSPH'],
        ['AC093752.1','SEPT7'],
        ['AC093787.2','AC233263.6'],
        ['AC093787.2','IGK@'],
        ['AC093821.1','LINC01091'],
        ['AC093827.5','SLC10A6'],
        ['AC093838.1','AL929601.1'],
        ['AC093838.1','AL929601.2'],
        ['AC093838.2','AL929601.1'],
        ['AC093838.2','AL929601.2'],
        ['AC093890.1','GYPE'],
        ['AC093895.1','SPARCL1'],
        ['AC095038.1','ZNF727'],
        ['AC095038.5','ZNF736'],
        ['AC095350.2','RIMBP2'],
        ['AC096574.4','TARDBP'],
        ['AC096579.13','ANPEP'],
        ['AC096579.13','CALD1'],
        ['AC096579.13','CD53'],
        ['AC096579.13','CTGF'],
        ['AC096579.13','DES'],
        ['AC096579.13','DPCR1'],
        ['AC096579.13','EIF2B5'],
        ['AC096579.13','HBB'],
        ['AC096579.13','IGKV1D-39'],
        ['AC096579.13','IGKV2D-29'],
        ['AC096579.13','IGKV2D-30'],
        ['AC096579.13','IGKV3D-11'],
        ['AC096579.13','IGKV3D-20'],
        ['AC096579.13','IGKV4-1'],
        ['AC096579.13','JCHAIN'],
        ['AC096579.13','MUC6'],
        ['AC096579.13','NDE1'],
        ['AC096579.13','PGC'],
        ['AC096579.13','PIGR'],
        ['AC096579.13','PRH1'],
        ['AC096579.13','PRH1-PRR4'],
        ['AC096579.13','PRM2'],
        ['AC096579.13','PRR4'],
        ['AC096579.13','SMR3B'],
        ['AC096579.13','TFF1'],
        ['AC096579.13','TRBC2'],
        ['AC096579.15','IGKV4-1'],
        ['AC096579.7','CD74'],
        ['AC096579.7','HLA@'],
        ['AC096579.7','IGH@'],
        ['AC096579.7','IGK@'],
        ['AC096579.7','IGL@'],
        ['AC096664.1','RPL27'],
        ['AC096664.2','RPL27'],
        ['AC097372.2','LINC01095'],
        ['AC097372.3','LINC01095'],
        ['AC097374.1','AL513478.2'],
        ['AC097374.1','CR392039.5'],
        ['AC097374.1','CR392039.6'],
        ['AC097374.2','BP-2171C21.2'],
        ['AC097468.7','LINC01494'],
        ['AC097500.1','FSIP2-AS1'],
        ['AC097634.4','PROK2'],
        ['AC098483.1','CR589904.2'],
        ['AC098483.1','OR2T2'],
        ['AC098484.2','PPIH'],
        ['AC098588.2','GYPB'],
        ['AC098588.2','GYPE'],
        ['AC098588.3','GYPB'],
        ['AC098588.3','GYPE'],
        ['AC098679.5','TMEM144'],
        ['AC098798.1','PANCR'],
        ['AC098825.1','PARM1'],
        ['AC098826.2','ANKRD30A'],
        ['AC098828.2','LAPTM4A'],
        ['AC098831.1','PYURF'],
        ['AC098848.1','CCDC68'],
        ['AC098850.3','FAM106A'],
        ['AC098850.3','USP32'],
        ['AC098850.4','AC107983.2'],
        ['AC098850.4','NOS2'],
        ['AC098850.4','USP32'],
        ['AC098850.4','USP6'],
        ['AC098850.5','NOS2'],
        ['AC098850.5','USP6'],
        ['AC099329.1','CCDC13'],
        ['AC099343.3','AC099343.4'],
        ['AC099344.3','SLC52A3'],
        ['AC099508.2','ZFP1'],
        ['AC099521.3','GLTP'],
        ['AC099548.2','TCAF1'],
        ['AC099548.2','TCAF2'],
        ['AC099548.2','TCAF2C'],
        ['AC099654.1','AL136102.1'],
        ['AC099794.1','ATG4C'],
        ['AC099805.1','EYA1'],
        ['AC099850.1','PRR11'],
        ['AC100771.1','RPL37A'],
        ['AC100802.3','EMP2'],
        ['AC100810.1','AC100810.3'],
        ['AC100821.2','TCEA1'],
        ['AC100861.1','CHMP7'],
        ['AC103702.2','HOXB9'],
        ['AC103718.1','GSDMC'],
        ['AC103810.1','ARL17B'],
        ['AC103957.1','PRAG1'],
        ['AC103957.2','SSBP2'],
        ['AC104088.1','ITGA6'],
        ['AC104109.3','LINC00174'],
        ['AC104297.1','PPA1'],
        ['AC104316.1','TBC1D31'],
        ['AC104453.1','CD93'],
        ['AC104472.3','UBP1'],
        ['AC104532.2','MKLN1'],
        ['AC104532.2','SURF1'],
        ['AC104581.1','DNAH14'],
        ['AC104699.1','LAMP2'],
        ['AC104758.1','COMMD4'],
        ['AC104794.3','PRELP'],
        ['AC104819.3','SH3D19'],
        ['AC104820.2','ITGA4'],
        ['AC104836.1','C1ORF146'],
        ['AC104837.1','SNW1'],
        ['AC104850.2','CX3CR1'],
        ['AC105046.1','EGR3'],
        ['AC105052.3','RASA4CP'],
        ['AC105052.4','RASA4CP'],
        ['AC105052.5','CCDC146'],
        ['AC105053.3','ATOH8'],
        ['AC105094.2','RNF152'],
        ['AC105342.1','ANKRD11'],
        ['AC105411.1','LINC01227'],
        ['AC105916.1','AC108519.1'],
        ['AC105916.1','LINC02021'],
        ['AC106791.1','KLHL3'],
        ['AC106791.3','KLHL3'],
        ['AC106795.2','AC139493.2'],
        ['AC106795.2','THOC3'],
        ['AC106795.3','AC139493.2'],
        ['AC106865.1','TLR2'],
        ['AC106886.5','MYH11'],
        ['AC107021.1','RAB10'],
        ['AC107032.3','HACD4'],
        ['AC107071.1','ELF1'],
        ['AC107072.2','GK5'],
        ['AC107204.1','DA750114'],
        ['AC107208.1','GUCY1A1'],
        ['AC107223.1','MYH11'],
        ['AC107918.4','DEFB131E'],
        ['AC107983.2','USP32'],
        ['AC107983.2','USP6'],
        ['AC108010.1','AC138035.1'],
        ['AC108010.1','AL133216.2'],
        ['AC108010.1','LINC01002'],
        ['AC108010.1','LINC01347'],
        ['AC108010.1','TLK2'],
        ['AC108024.1','KCNE3'],
        ['AC108063.2','PROM1'],
        ['AC108097.1','LINC01227'],
        ['AC108134.3','ZNF600'],
        ['AC108488.1','ADI1'],
        ['AC108488.2','ADI1'],
        ['AC108488.4','ADI1'],
        ['AC108519.1','AF228730.5'],
        ['AC108688.1','DNAJB6'],
        ['AC108734.3','RNF13'],
        ['AC108751.4','TM4SF1'],
        ['AC109361.1','TBCK'],
        ['AC109466.1','CCDC7'],
        ['AC109466.1','IGH@'],
        ['AC109466.1','TRA@'],
        ['AC109583.3','PRSS46'],
        ['AC110079.1','AC118758.3'],
        ['AC110079.2','NPR1'],
        ['AC110611.1','AC110611.2'],
        ['AC111149.1','RDH10-AS1'],
        ['AC111186.1','SLC16A5'],
        ['AC111188.1','ARPC1B'],
        ['AC112719.2','ART3'],
        ['AC112721.1','COL6A3'],
        ['AC112721.2','COL6A3'],
        ['AC113189.4','TMEM102'],
        ['AC113189.6','C17ORF74'],
        ['AC113189.6','TMEM102'],
        ['AC113189.9','FGF11'],
        ['AC113189.9','SPEM2'],
        ['AC113398.1','RPL28'],
        ['AC113607.1','TMEM18'],
        ['AC114284.1','PRR16'],
        ['AC114490.1','AC114490.2'],
        ['AC114490.1','TMEM35B'],
        ['AC114501.2','INTS4'],
        ['AC114781.2','COQ2'],
        ['AC114964.2','FBXL7'],
        ['AC114971.1','TECRL'],
        ['AC115220.3','LINC01005'],
        ['AC115220.3','MAEL'],
        ['AC116050.1','IGK@'],
        ['AC116366.1','IRF1'],
        ['AC116366.6','IRF1'],
        ['AC116535.2','CTR9'],
        ['AC116563.1','AC131956.2'],
        ['AC116651.1','FBXL5'],
        ['AC117382.2','GM2A'],
        ['AC117386.2','CHURC1'],
        ['AC117386.2','TGOLN2'],
        ['AC117386.2','TRA@'],
        ['AC117409.1','PSMC2'],
        ['AC117440.1','MTCH2'],
        ['AC117834.1','AC117834.2'],
        ['AC118658.1','ANP32A'],
        ['AC118758.2','AC138035.3'],
        ['AC118758.3','LINC00174'],
        ['AC119396.1','ARHGEF18'],
        ['AC119403.1','ZNF77'],
        ['AC119427.2','ACTN2'],
        ['AC119427.2','ANP32B'],
        ['AC119427.2','CLCN6'],
        ['AC119427.2','CRYAB'],
        ['AC119427.2','MYH7'],
        ['AC119427.2','NPPA'],
        ['AC119427.2','TNNT1'],
        ['AC119674.1','TMEM127'],
        ['AC119674.2','TG'],
        ['AC120057.2','CLCN6'],
        ['AC120057.3','NPPA-AS1'],
        ['AC122133.1','SEPT7'],
        ['AC123767.1','ADAM5'],
        ['AC123912.3','ZNF100'],
        ['AC124242.1','AC124242.3'],
        ['AC124312.1','WDR4'],
        ['AC124944.2','MUC20-OT1'],
        ['AC124944.3','LINC00969'],
        ['AC124944.3','MUC20-OT1'],
        ['AC124944.3','PDCD6'],
        ['AC125257.1','KLHL11'],
        ['AC125494.1','RPL14'],
        ['AC125634.1','AL353626.3'],
        ['AC126120.1','PPIA'],
        ['AC126182.3','NDUFB4'],
        ['AC126283.2','ALB'],
        ['AC126335.1','IGK@'],
        ['AC126544.1','BRWD1'],
        ['AC126544.1','HARS'],
        ['AC126544.2','BRWD1'],
        ['AC126755.3','AC136443.3'],
        ['AC127502.1','AC135983.3'],
        ['AC127502.2','AC135983.2'],
        ['AC127502.2','LYZ'],
        ['AC127526.2','AC127526.5'],
        ['AC127526.2','LINC00937'],
        ['AC127526.4','LINC00937'],
        ['AC127526.4','LINC02614'],
        ['AC127526.5','ERVE-1'],
        ['AC127526.5','LINC02635'],
        ['AC127894.1','CPSF6'],
        ['AC129492.1','VAMP2'],
        ['AC131056.5','CH17-431G21.1'],
        ['AC131160.1','AIG1'],
        ['AC131160.1','AL136116.3'],
        ['AC131180.1','NF1'],
        ['AC131254.2','RBM47'],
        ['AC132008.2','SDHA'],
        ['AC132217.2','ADAM12'],
        ['AC132217.2','ADAMTSL4-AS1'],
        ['AC132217.2','AMOTL1'],
        ['AC132217.2','ANXA2'],
        ['AC132217.2','AP000781.2'],
        ['AC132217.2','ATP5F1A'],
        ['AC132217.2','BDH1'],
        ['AC132217.2','CADM3-AS1'],
        ['AC132217.2','CALD1'],
        ['AC132217.2','CBX4'],
        ['AC132217.2','CCDC152'],
        ['AC132217.2','CD93'],
        ['AC132217.2','COL1A2'],
        ['AC132217.2','COL5A1'],
        ['AC132217.2','COLEC12'],
        ['AC132217.2','CSF2RB'],
        ['AC132217.2','CSH1'],
        ['AC132217.2','CYP19A1'],
        ['AC132217.2','DCN'],
        ['AC132217.2','EGFR'],
        ['AC132217.2','EHD3'],
        ['AC132217.2','F13A1'],
        ['AC132217.2','FGF14'],
        ['AC132217.2','FGFRL1'],
        ['AC132217.2','FLT1'],
        ['AC132217.2','FN1'],
        ['AC132217.2','HACD3'],
        ['AC132217.2','INS'],
        ['AC132217.2','KIAA0895L'],
        ['AC132217.2','LGMN'],
        ['AC132217.2','LHFPL6'],
        ['AC132217.2','LRP1'],
        ['AC132217.2','MEG3'],
        ['AC132217.2','MPEG1'],
        ['AC132217.2','MRNIP'],
        ['AC132217.2','NAP1L1'],
        ['AC132217.2','PCBP2'],
        ['AC132217.2','PODN'],
        ['AC132217.2','SEPT9'],
        ['AC132217.2','SPARC'],
        ['AC132217.2','SYNPO'],
        ['AC132217.2','TIMP2'],
        ['AC132217.2','TIMP3'],
        ['AC132217.2','TP53I11'],
        ['AC132217.2','TRIOBP'],
        ['AC132217.2','VWF'],
        ['AC132217.4','SPARC'],
        ['AC133065.2','CIITA'],
        ['AC133065.6','CIITA'],
        ['AC133555.5','SMG1'],
        ['AC133644.3','PLG'],
        ['AC133919.3','AL606534.5'],
        ['AC133919.3','AL606534.6'],
        ['AC133961.1','SEL1L3'],
        ['AC134504.1','PRSS44P'],
        ['AC134511.1','GRIP1'],
        ['AC134772.1','SHISA5'],
        ['AC134878.2','BRWD1'],
        ['AC134878.2','CR381653.1'],
        ['AC134878.2','TEKT4'],
        ['AC134879.2','IGK@'],
        ['AC134980.2','OR4N2'],
        ['AC135983.1','CHRFAM7A'],
        ['AC135983.2','WHAMM'],
        ['AC136475.10','SRP19'],
        ['AC136628.2','C5ORF60'],
        ['AC136698.1','AC243562.3'],
        ['AC136698.1','LINC00933'],
        ['AC136944.2','IGK@'],
        ['AC136944.3','CHEK2'],
        ['AC136944.4','DUSP22'],
        ['AC137056.1','AC141273.1'],
        ['AC138035.1','AL627309.5'],
        ['AC138207.8','LRRC37A3'],
        ['AC138207.8','LRRC37B'],
        ['AC138207.8','NF1'],
        ['AC138207.8','SMURF2'],
        ['AC138207.8','SMURF2P1-LRRC37BP1'],
        ['AC138356.3','ERLIN2'],
        ['AC138393.1','SEPT7'],
        ['AC138393.1','ZSCAN16-AS1'],
        ['AC138409.2','AP000347.1'],
        ['AC138409.2','C1QTNF3'],
        ['AC138409.2','C1QTNF3-AMACR'],
        ['AC138409.2','GTF2H2'],
        ['AC138409.2','GTF2H2B'],
        ['AC138409.2','GTF2H2C'],
        ['AC138409.2','LINC02241'],
        ['AC138409.2','WDR70'],
        ['AC138430.4','FAM129A'],
        ['AC138646.1','MIR100HG'],
        ['AC138649.1','PDCD6IP'],
        ['AC138776.1','ANKRD20A21P'],
        ['AC138811.2','SMG1'],
        ['AC138819.1','AC139493.2'],
        ['AC138915.4','CISD2'],
        ['AC138932.1','NPIPA8'],
        ['AC138932.3','KIAA2013'],
        ['AC139256.1','SNX29'],
        ['AC139256.2','PKD1'],
        ['AC139491.5','FAM153B'],
        ['AC139491.5','FAM153C'],
        ['AC139768.1','SMAGP'],
        ['AC139795.1','SIMC1'],
        ['AC139795.2','CDC6'],
        ['AC139795.2','FOXP1'],
        ['AC139795.2','IBA57'],
        ['AC140134.1','OCLN'],
        ['AC140481.3','FAM95B1'],
        ['AC141002.1','LINC02585'],
        ['AC142381.1','IGH@'],
        ['AC145098.1','DOK3'],
        ['AC145124.1','FAM85B'],
        ['AC145124.1','LINC02018'],
        ['AC145124.2','FAM85B'],
        ['AC145138.1','OCLN'],
        ['AC195454.1','CHRNA9'],
        ['AC209154.1','C17ORF51'],
        ['AC211429.1','PRKRIP1'],
        ['AC211476.3','DTX2P1-UPK3BP1-PMS2P11'],
        ['AC211476.4','DTX2P1-UPK3BP1-PMS2P11'],
        ['AC211486.3','CCDC146'],
        ['AC211486.3','DTX2P1-UPK3BP1-PMS2P11'],
        ['AC211486.3','PMS2'],
        ['AC211486.3','POM121C'],
        ['AC211486.3','PRKRIP1'],
        ['AC211486.3','UPK3B'],
        ['AC226119.4','SFTPB'],
        ['AC226119.5','SFTPB'],
        ['AC231759.2','AC239585.2'],
        ['AC233266.2','TMEM128'],
        ['AC233702.10','FLJ36000'],
        ['AC233702.4','FAM182A'],
        ['AC233702.7','FLJ36000'],
        ['AC234783.1','TMSB15B'],
        ['AC239798.4','HIST2H2BF'],
        ['AC239804.1','LINC01719'],
        ['AC239809.3','HYDIN2'],
        ['AC239860.1','NBPF13P'],
        ['AC241377.4','LINC00869'],
        ['AC241584.1','SSB'],
        ['AC242376.2','HERC2'],
        ['AC242988.2','PLEKHO1'],
        ['AC243562.1','AKAP13'],
        ['AC243562.1','GOLGA2P2Y'],
        ['AC243562.1','GOLGA2P3Y'],
        ['AC243562.3','DNM1'],
        ['AC243829.4','CCL4'],
        ['AC243829.5','AC244154.1'],
        ['AC243829.5','NPEPPS'],
        ['AC243967.3','LYPD4'],
        ['AC244033.2','OTUD7B'],
        ['AC244197.3','LINC00894'],
        ['AC244197.3','SMR3B'],
        ['AC244205.1','IGKV4-1'],
        ['AC244205.2','IGK@'],
        ['AC244226.1','IGH@'],
        ['AC244258.1','PRIM2'],
        ['AC244517.10','AC244517.6'],
        ['AC244517.10','PCDHB9'],
        ['AC244517.11','TAF7'],
        ['AC244517.2','MALL'],
        ['AC245033.1','PNLIP'],
        ['AC245041.2','LINC00842'],
        ['AC245047.1','SSX3'],
        ['AC245060.4','IGLV5-52'],
        ['AC245060.5','IGK@'],
        ['AC245060.6','IGK@'],
        ['AC245060.7','IGLC1'],
        ['AC245060.7','IGLC2'],
        ['AC245060.7','IGLC3'],
        ['AC245060.7','IGLL5'],
        ['AC245060.7','IGLV1-40'],
        ['AC245060.7','IGLV1-41'],
        ['AC245100.4','LINC00869'],
        ['AC245128.3','AL357055.3'],
        ['AC245128.3','ORC4'],
        ['AC245369.1','IGH@'],
        ['AC245369.3','IGH@'],
        ['AC245369.3','IGHA2'],
        ['AC245369.3','IGHG1'],
        ['AC245369.4','IGH@'],
        ['AC245369.7','IGH@'],
        ['AC245452.5','IGLVIV-66-1'],
        ['AC245595.1','IGK@'],
        ['AC245884.11','LILRA6'],
        ['AC245884.11','LILRB3'],
        ['AC245884.4','LYZ'],
        ['AC246787.2','FAM30A'],
        ['AC247036.4','IGH@'],
        ['AC247036.6','IGH@'],
        ['AC253536.1','GSTTP1'],
        ['ACAA1','FBP1'],
        ['ACACA','EIF3G'],
        ['ACAD8','GLB1L3'],
        ['ACADL','KANSL1L'],
        ['ACADSB','ARHGAP32'],
        ['ACADSB','MCFD2'],
        ['ACADVL','LTBP2'],
        ['ACADVL','RACK1'],
        ['ACAT1','CUL5'],
        ['ACBD3','ZMYM4'],
        ['ACCS','EXT2'],
        ['ACE2','PIR'],
        ['ACE2','TMEM27'],
        ['ACER3','POP1'],
        ['ACIN1','MTFR1L'],
        ['ACIN1','MUC7'],
        ['ACIN1','PRH1'],
        ['ACIN1','PRH2'],
        ['ACIN1','RALGAPA1'],
        ['ACIN1','WTIP'],
        ['ACKR2','ALB'],
        ['ACKR2','CALR'],
        ['ACO2','MYH6'],
        ['ACOT11','LINC01445'],
        ['ACOT11','MROH7-TTC4'],
        ['ACOT8','TNNC2'],
        ['ACOX1','DNAL1'],
        ['ACPP','ACTG2'],
        ['ACPP','B2M'],
        ['ACPP','CALD1'],
        ['ACPP','EEF1A1'],
        ['ACPP','ERGIC3'],
        ['ACPP','ITGB1'],
        ['ACPP','MSMB'],
        ['ACPP','MYH11'],
        ['ACPP','MYLK'],
        ['ACPP','NCAPD3'],
        ['ACPP','SGMS1'],
        ['ACR','SHANK3'],
        ['ACRBP','RMI2'],
        ['ACSF2','ALB'],
        ['ACSL1','ALB'],
        ['ACSL1','CENPU'],
        ['ACSL1','COL4A2'],
        ['ACSL1','CT69'],
        ['ACSL1','FTX'],
        ['ACSL1','HBB'],
        ['ACSL1','HTN1'],
        ['ACSL1','HTN3'],
        ['ACSL1','IGH@'],
        ['ACSL1','IPO11'],
        ['ACSL3','CPM'],
        ['ACSL3','HELLPAR'],
        ['ACSL3','WEE1'],
        ['ACSL4','SULT2A1'],
        ['ACSS1','APMAP'],
        ['ACSS2','TMEM154'],
        ['ACTG1','ALB'],
        ['ACTG1','AMY2A'],
        ['ACTG1','CFL1'],
        ['ACTG1','COL1A1'],
        ['ACTG1','COL1A2'],
        ['ACTG1','COL3A1'],
        ['ACTG1','EEF2'],
        ['ACTG1','KRT13'],
        ['ACTG1','LYZ'],
        ['ACTG1','MYL6'],
        ['ACTG1','PCOLCE'],
        ['ACTG1','PLPP3'],
        ['ACTG1','SLC7A5'],
        ['ACTG1','SPARC'],
        ['ACTG1','STAMBPL1'],
        ['ACTG1','TINAGL1'],
        ['ACTG1','TMEM259'],
        ['ACTG1','TMSB10'],
        ['ACTG1','TPM2'],
        ['ACTG1','TRB@'],
        ['ACTG1','XIST'],
        ['ACTG2','AHNAK'],
        ['ACTG2','CALR'],
        ['ACTG2','COL1A2'],
        ['ACTG2','GREB1L'],
        ['ACTG2','MYLK'],
        ['ACTG2','MYLK-AS1'],
        ['ACTG2','NDE1'],
        ['ACTG2','STAMBPL1'],
        ['ACTG2','ZSCAN32'],
        ['ACTN1','FN1'],
        ['ACTN1','HBB'],
        ['ACTN1','IGH@'],
        ['ACTN1','KRT4'],
        ['ACTN1','LASP1'],
        ['ACTN1','MYH9'],
        ['ACTN1','NOC2L'],
        ['ACTN2','CSRP3'],
        ['ACTN2','TNNT2'],
        ['ACTN2','TPM1'],
        ['ACTN4','BCAP31'],
        ['ACTN4','EIF3K'],
        ['ACTN4','FASN'],
        ['ACTN4','IGH@'],
        ['ACTN4','LYZ'],
        ['ACTN4','PLIN4'],
        ['ACTN4','PRSS1'],
        ['ACTN4','PTRF'],
        ['ACTN4','SMR3B'],
        ['ACTN4','STATH'],
        ['ACTN4','TG'],
        ['ACTN4','TRB@'],
        ['ACTR10','CRHR1-IT1_'],
        ['ACTR10','TG'],
        ['ACTR2','LYZ'],
        ['ACTR6','MPO'],
        ['ACTR8','SELK'],
        ['ACVR1B','GRASP'],
        ['ACVR1B','STATH'],
        ['ACVR1B','ZG16B'],
        ['ACVR2B','EXOG'],
        ['ACYP2','GBP6'],
        ['ADAD1','IL21-AS1'],
        ['ADAM10','MRI1'],
        ['ADAM10','PTMS'],
        ['ADAM12','IGF2'],
        ['ADAM23','FAM237A'],
        ['ADAM32','ADAM9'],
        ['ADAM32','BANK1'],
        ['ADAMDEC1','CSNK2A1'],
        ['ADAMDEC1','MYH9'],
        ['ADAMTS16','ICE1'],
        ['ADAMTS17','SPATA41'],
        ['ADAMTS4','MLXIP'],
        ['ADAMTS4','PLIN4'],
        ['ADAMTS4','TRA@'],
        ['ADAMTS6','CENPK'],
        ['ADAMTS7','AL049757.1'],
        ['ADAMTSL3','LYZ'],
        ['ADAMTSL3','SH3GL3'],
        ['ADAMTSL4-AS1','IGF2'],
        ['ADAP1','MAFK'],
        ['ADAP1','SUN1'],
        ['ADAP2','RNF135'],
        ['ADAR','CCDC80'],
        ['ADAR','SFTPB'],
        ['ADCK3','PSEN2'],
        ['ADCK4','NUMBL'],
        ['ADCY1','ZSCAN5A'],
        ['ADCY10','FAM177A1'],
        ['ADCY10','MPC2'],
        ['ADCY10P1','BGN'],
        ['ADCY10P1','NFYA'],
        ['ADCY3','AL160408.3'],
        ['ADCY3','COL1A1'],
        ['ADCY5','NPPA'],
        ['ADD1','ERGIC1'],
        ['ADD1','LYZ'],
        ['ADD1','MYH11'],
        ['ADD1','SH3BP2'],
        ['ADD1','TG'],
        ['ADD3','ENAM'],
        ['ADD3','LPP'],
        ['ADD3','TG'],
        ['ADGRA1','RAB7A'],
        ['ADGRA3','TECRL'],
        ['ADGRE1','VAV1'],
        ['ADGRE2','ADGRE5'],
        ['ADGRE5','AK2'],
        ['ADGRF1','RNF115'],
        ['ADGRF5','AL512625.3'],
        ['ADGRF5','LERFS'],
        ['ADGRG1','CLPS'],
        ['ADGRG1','EEF1A1'],
        ['ADGRG1','FCGBP'],
        ['ADGRG1','TG'],
        ['ADGRG5','HMGB3P32'],
        ['ADGRG7','TFG'],
        ['ADGRL1','SUZ12'],
        ['ADGRL2','FAM134C'],
        ['ADGRL2','ZG16B'],
        ['ADH1A','ADH4'],
        ['ADH1A','TF'],
        ['ADH1B','ADH4'],
        ['ADH1B','ALB'],
        ['ADH1B','C1S'],
        ['ADH1B','HP'],
        ['ADH1B','PTRF'],
        ['ADH1B','TXNL4B'],
        ['ADH4','ADH5'],
        ['ADH4','ALB'],
        ['ADH4','C3'],
        ['ADIG','ARHGAP40'],
        ['ADIPOQ','ANKRD31'],
        ['ADIPOQ','C19MC'],
        ['ADIPOQ','CD36'],
        ['ADIPOQ','CLU'],
        ['ADIPOQ','CPM'],
        ['ADIPOQ','DNAH14'],
        ['ADIPOQ','IGK@'],
        ['ADIPOQ','MRPL10'],
        ['ADIPOQ','TLDC2'],
        ['ADIPOQ','VKORC1L1'],
        ['ADIPOQ-AS1','AL139393.1'],
        ['ADIPOQ-AS1','CWF19L1'],
        ['ADIPOQ-AS1','GNPTG'],
        ['ADIPOQ-AS1','PLIN4'],
        ['ADIPOQ-AS1','PTRF'],
        ['ADIPOR1','CYB5R1'],
        ['ADIRF','SNCG'],
        ['ADK','HIBCH'],
        ['ADK','KAT6B'],
        ['ADM','RIN3'],
        ['ADM','TMEM214'],
        ['ADNP','DPM1'],
        ['ADPGK','AL160408.3'],
        ['ADPRHL1','DCUN1D2'],
        ['ADRBK1','LYZ'],
        ['ADRM1','GSE1'],
        ['ADRM1','SS18L1'],
        ['ADSL','SGSM3'],
        ['ADSL','TNRC6B'],
        ['AEBP2','PLEKHA5'],
        ['AES','COL5A1'],
        ['AES','EPN1'],
        ['AES','FLNC'],
        ['AES','LYZ'],
        ['AES','MUC7'],
        ['AES','PRRX2'],
        ['AES','SFTPC'],
        ['AES','SMR3B'],
        ['AF001548.2','KRT13'],
        ['AF001548.5','KRT13'],
        ['AF038458.2','AF038458.3'],
        ['AF064860.2','IGSF5'],
        ['AF064860.7','IGSF5'],
        ['AF127936.9','HBB'],
        ['AF131217.1','MSMB'],
        ['AF165147.1','FXR1'],
        ['AF228730.5','AP003498.1'],
        ['AF228730.5','AP003498.2'],
        ['AF228730.5','LINC00923'],
        ['AF228730.7','AP003498.1'],
        ['AF228730.7','CECR7'],
        ['AF235103.3','ZNF250'],
        ['AF254983.1','TPTE2'],
        ['AF305872.1','RPL21'],
        ['AFAP1','ANXA7'],
        ['AFAP1L2','TG'],
        ['AFDN-AS1','LINC01558'],
        ['AFF1','PTPN13'],
        ['AFF1','TG'],
        ['AFF3','ITGB1'],
        ['AFF4','F11R'],
        ['AFG3L2P1','LYZ'],
        ['AFM','FCN1'],
        ['AFM','LINC02499'],
        ['AFMID','AP001372.3'],
        ['AFMID','LRG1'],
        ['AFMID','LYZ'],
        ['AFMID','SLC48A1'],
        ['AFP','ALB'],
        ['AFTPH','LINC01805'],
        ['AGAP11','AL512662.2'],
        ['AGAP11','BMS1'],
        ['AGAP3','CUX1'],
        ['AGAP3','SFTPC'],
        ['AGAP4','LINC00843'],
        ['AGAP5','BMS1P4'],
        ['AGAP6','TIMM23B'],
        ['AGBL2','MTCH2'],
        ['AGBL5','HBB'],
        ['AGER','CD74'],
        ['AGER','IFITM2'],
        ['AGER','SFTPA2'],
        ['AGER','SFTPC'],
        ['AGFG1','CFLAR'],
        ['AGGF1','ZBED3-AS1'],
        ['AGK','TG'],
        ['AGL','HIAT1'],
        ['AGMAT','AL121992.1'],
        ['AGMAT','FBXW12'],
        ['AGO2','MRE11'],
        ['AGO3','MPV17L'],
        ['AGO3','STAG3'],
        ['AGPAT2','ALB'],
        ['AGPAT2','LRP1'],
        ['AGPAT3','SOD2'],
        ['AGPAT3','TG'],
        ['AGPS','USP34'],
        ['AGR2','LYZ'],
        ['AGRN','PRNP'],
        ['AGT','ALB'],
        ['AGT','C1ORF198'],
        ['AGT','FGA'],
        ['AGT','FGG'],
        ['AGTPBP1','AL353743.2'],
        ['AGXT','SERPINA1'],
        ['AHCYL2','PRPF4'],
        ['AHNAK','AHNAK2'],
        ['AHNAK','CSRP1'],
        ['AHNAK','FN1'],
        ['AHNAK','HLA@'],
        ['AHNAK','IGF2'],
        ['AHNAK','KRT10'],
        ['AHNAK','KRT14'],
        ['AHNAK','KRT2'],
        ['AHNAK','LIPF'],
        ['AHNAK','LYZ'],
        ['AHNAK','MYH11'],
        ['AHNAK','NEGR1'],
        ['AHNAK','SLC34A2'],
        ['AHNAK','SMR3B'],
        ['AHNAK','TRB@'],
        ['AHR','LYZ'],
        ['AHSA2','C2ORF74'],
        ['AHSA2','TIAL1'],
        ['AHSG','HP'],
        ['AIF1L','NUP214'],
        ['AIFM3','LZTR1'],
        ['AIG1','AL136116.3'],
        ['AIG1','GSK3B'],
        ['AIG1','PHF20L1'],
        ['AIM1','C6ORF203'],
        ['AIM2','CADM3-AS1'],
        ['AIMP1','ZNF605'],
        ['AJUBA','SERPINE1'],
        ['AK4','NF1'],
        ['AK4P4','ERMP1'],
        ['AKAP1','TRB@'],
        ['AKAP10','SFTPB'],
        ['AKAP11','ETS2'],
        ['AKAP12','RBM39'],
        ['AKAP13','CD2AP'],
        ['AKAP13','CLPS'],
        ['AKAP13','CPA1'],
        ['AKAP13','CPB1'],
        ['AKAP13','FGD4'],
        ['AKAP13','LYZ'],
        ['AKAP13','STATH'],
        ['AKAP17A','P2RY8'],
        ['AKAP8L','SMR3B'],
        ['AKIRIN1','LYZ'],
        ['AKNA','AL138895.1'],
        ['AKNA','EMP2'],
        ['AKR1A1','CCND3'],
        ['AKR1A1','NBR2'],
        ['AKR1B1','CYP17A1'],
        ['AKR1B1','GML'],
        ['AKR1B1','STAG3'],
        ['AKR1B10','IGK@'],
        ['AKR1B15','DA750114'],
        ['AKR1C2','AKR1C8P'],
        ['AKR1C5P','AKR1C8P'],
        ['AKR1C7P','AKR1C8P'],
        ['AKR7L','AL035413.2'],
        ['AKT1','AKT2'],
        ['AKT1','COL1A1'],
        ['AKT1','EXT1'],
        ['AKT2','C19ORF47'],
        ['AKT2','TRB@'],
        ['AKT3','TRB@'],
        ['AL008628.1','FAM120B'],
        ['AL021368.4','WDR70'],
        ['AL021707.2','FAM227A'],
        ['AL021920.2','EIF1AX'],
        ['AL022151.1','TRA@'],
        ['AL022238.4','TNRC6B'],
        ['AL022322.2','GALNT4'],
        ['AL022322.2','POC1B-GALNT4'],
        ['AL022324.2','PHF10'],
        ['AL023802.1','MEG3'],
        ['AL024497.1','AL024497.2'],
        ['AL024509.3','CMAHP'],
        ['AL031056.1','MAP3K20'],
        ['AL031259.1','AL669831.3'],
        ['AL031281.2','SLC25A37'],
        ['AL031281.2','TRB@'],
        ['AL031281.2','ZEB2'],
        ['AL031282.2','DSC3'],
        ['AL031282.2','MALL'],
        ['AL031282.2','SLC35E2B'],
        ['AL031289.1','NFYC-AS1'],
        ['AL031289.1','RIMS3'],
        ['AL031315.1','TSPAN15'],
        ['AL031432.4','TG'],
        ['AL031587.6','AL627309.5'],
        ['AL031587.6','LYZ'],
        ['AL031587.6','MICALL1'],
        ['AL031674.1','LINC01524'],
        ['AL031705.1','CCND3'],
        ['AL033527.3','AL033527.4'],
        ['AL033527.3','AL033527.5'],
        ['AL034369.1','PNLIPRP2'],
        ['AL035078.4','RARA'],
        ['AL035412.1','DEPDC1-AS1'],
        ['AL035420.1','EPB41L1'],
        ['AL035425.3','IRS4'],
        ['AL035587.1','CNPY3'],
        ['AL035587.1','GNMT'],
        ['AL035661.1','ERVE-1'],
        ['AL035681.1','CHADL'],
        ['AL035701.1','CLIC5'],
        ['AL035701.1','ENPP5'],
        ['AL049634.2','SIRPA'],
        ['AL049646.1','AL049646.2'],
        ['AL049795.1','TMEM234'],
        ['AL049828.1','KCNJ15'],
        ['AL049830.3','STRN3'],
        ['AL049836.2','LINC02314'],
        ['AL049839.2','FGB'],
        ['AL049839.2','SERPINA1'],
        ['AL049840.4','IGH@'],
        ['AL049869.3','ZBTB25'],
        ['AL049870.2','AL049870.3'],
        ['AL049872.1','CXCR4'],
        ['AL050303.11','LINC01087'],
        ['AL050303.11','LINC01297'],
        ['AL050303.3','LINC01087'],
        ['AL050303.3','LINC01297'],
        ['AL050309.1','KLF8'],
        ['AL050320.1','ISM1-AS1'],
        ['AL078587.2','AL450124.1'],
        ['AL079341.1','TFG'],
        ['AL080275.1','COL9A1'],
        ['AL080317.1','AL080317.4'],
        ['AL080317.1','MFSD4B'],
        ['AL096677.2','CSTL1'],
        ['AL096803.2','ODR4'],
        ['AL096870.2','AL136295.7'],
        ['AL096870.3','AL136295.7'],
        ['AL109804.1','MAVS'],
        ['AL109806.1','SCARB2'],
        ['AL109809.1','AL117335.1'],
        ['AL109809.1','AL121760.1'],
        ['AL109809.5','AL117335.1'],
        ['AL109811.1','AL109811.2'],
        ['AL109811.1','EXOSC10-AS1'],
        ['AL109811.2','TARDBP'],
        ['AL109811.3','TARDBP'],
        ['AL109923.1','DYNLRB1'],
        ['AL109954.2','CSTL1'],
        ['AL110114.1','YWHAB'],
        ['AL110115.2','NPEPPS'],
        ['AL117329.1','ARPC2'],
        ['AL117335.1','AL121760.1'],
        ['AL117350.1','RAB4A'],
        ['AL121578.2','SYTL5'],
        ['AL121578.3','SYTL5'],
        ['AL121656.1','LTBP1'],
        ['AL121656.5','LTBP1'],
        ['AL121761.1','GABRP'],
        ['AL121761.1','RIN2'],
        ['AL121790.1','MIPOL1'],
        ['AL121790.1','TTC6'],
        ['AL121936.1','NORAD'],
        ['AL121985.1','SET'],
        ['AL121987.2','DCAF8'],
        ['AL132639.3','TRAPPC6B'],
        ['AL132639.4','TRAPPC6B'],
        ['AL132656.2','NUTM2A-AS1'],
        ['AL132656.2','NUTM2B-AS1'],
        ['AL132712.2','LINC00637'],
        ['AL132780.3','PRMT5'],
        ['AL132800.1','AL136018.1'],
        ['AL133304.3','LINC00609'],
        ['AL133338.2','MAPK1IP1L'],
        ['AL133353.1','SLC25A28'],
        ['AL133353.2','SLC25A28'],
        ['AL133353.2','TBC1D10B'],
        ['AL133367.1','CKB'],
        ['AL133371.1','AL355922.4'],
        ['AL133371.1','AL355922.5'],
        ['AL133405.1','STMND1'],
        ['AL133410.1','RGP1'],
        ['AL133410.3','TMEM8B'],
        ['AL133415.1','VIM-AS1'],
        ['AL133481.1','PPIF'],
        ['AL133481.2','AL359195.2'],
        ['AL133481.3','AL359195.2'],
        ['AL135905.2','B2M'],
        ['AL135905.2','CXCR4'],
        ['AL135905.2','DDX5'],
        ['AL135905.2','EEF1A1'],
        ['AL135905.2','G3BP2'],
        ['AL135905.2','GABPB1'],
        ['AL135905.2','HIF1A'],
        ['AL135905.2','HNRNPA2B1'],
        ['AL135905.2','HNRNPH1'],
        ['AL135905.2','SRSF5'],
        ['AL135923.1','AL135923.2'],
        ['AL136116.3','PARL'],
        ['AL136133.1','RPL22'],
        ['AL136140.1','CDC5L'],
        ['AL136141.1','PIPOX'],
        ['AL136295.5','THADA'],
        ['AL136295.7','CHMP4A'],
        ['AL136309.2','C6ORF201'],
        ['AL136309.4','C6ORF201'],
        ['AL136520.1','SRSF10'],
        ['AL136982.1','BMS1'],
        ['AL136982.4','BMS1'],
        ['AL136982.6','BMS1'],
        ['AL137003.1','ATXN1'],
        ['AL137026.3','CXCL12'],
        ['AL137058.1','AL158066.1'],
        ['AL137058.1','MRPS31P5'],
        ['AL137072.1','MRPS16'],
        ['AL137145.1','PFKFB3'],
        ['AL137145.2','PFKFB3'],
        ['AL137186.2','SPAG9'],
        ['AL137786.1','PAPOLA'],
        ['AL137796.1','C1ORF220'],
        ['AL137802.3','BX284668.5'],
        ['AL137803.1','SLC44A5'],
        ['AL137856.1','AP4B1-AS1'],
        ['AL138830.1','AL138830.2'],
        ['AL138885.2','LINC00518'],
        ['AL138963.3','APOH'],
        ['AL138963.3','HP'],
        ['AL138963.3','PNLIP'],
        ['AL138963.3','PRB1'],
        ['AL138963.3','PRB2'],
        ['AL138963.3','PRB3'],
        ['AL138963.3','PRSS1'],
        ['AL138963.3','SMR3B'],
        ['AL138963.3','TRB@'],
        ['AL138963.4','APOH'],
        ['AL138963.4','HP'],
        ['AL138963.4','PNLIP'],
        ['AL138963.4','PRSS1'],
        ['AL138963.4','SMR3B'],
        ['AL138963.4','TRB@'],
        ['AL138999.2','C9ORF78'],
        ['AL139022.1','FNTB'],
        ['AL139106.1','SNX1'],
        ['AL139142.1','BX248409.2'],
        ['AL139142.1','TEX50'],
        ['AL139158.2','AL390839.2'],
        ['AL139174.1','MMADHC'],
        ['AL139220.2','KLF17'],
        ['AL139260.3','RHBDL2'],
        ['AL139300.1','AL139300.2'],
        ['AL139300.2','APOPT1'],
        ['AL139353.3','CHST11'],
        ['AL139383.1','LINC02344'],
        ['AL139383.1','STARD13'],
        ['AL139384.2','TUBGCP3'],
        ['AL139412.1','C1ORF61'],
        ['AL139819.1','PKD2L1'],
        ['AL157400.3','PANK1'],
        ['AL157714.2','RXRG'],
        ['AL157762.1','KLHDC10'],
        ['AL157791.1','COIL'],
        ['AL157834.1','CYP2C8'],
        ['AL157834.2','PDLIM1'],
        ['AL157871.1','YY1'],
        ['AL157893.1','NUTM2B-AS1'],
        ['AL157935.2','ST6GALNAC4'],
        ['AL157955.1','AL157955.2'],
        ['AL157955.1','LINC01147'],
        ['AL158066.1','MRPS31P4'],
        ['AL158066.1','SUGT1P4-STRA6LP'],
        ['AL158066.2','SUGT1P4-STRA6LP'],
        ['AL158154.2','UPP1'],
        ['AL158154.3','ATG14'],
        ['AL158154.3','MRE11'],
        ['AL158154.3','SOD2'],
        ['AL158154.3','TTPA'],
        ['AL158154.3','UPP1'],
        ['AL158212.1','VTI1A'],
        ['AL160154.1','LINC00375'],
        ['AL160291.1','ARMC4'],
        ['AL160313.1','HHIPL1'],
        ['AL160408.2','HBB'],
        ['AL161431.1','LINC00882'],
        ['AL161457.2','FRG1DP'],
        ['AL161457.2','FRG1JP'],
        ['AL161621.1','MRAP2'],
        ['AL161669.1','EXOC3L4'],
        ['AL161782.1','LINC01507'],
        ['AL162258.2','DES'],
        ['AL162377.3','ATP7B'],
        ['AL162400.2','LINC01739'],
        ['AL162414.1','CLPB'],
        ['AL162417.1','METRNL'],
        ['AL353148.1','LYZ'],
        ['AL353572.2','CDK20'],
        ['AL353572.2','SMR3B'],
        ['AL353608.3','PIP5K1B'],
        ['AL353660.1','LINC00402'],
        ['AL353693.1','AL591135.2'],
        ['AL353743.1','KIF27'],
        ['AL354710.2','GAPVD1'],
        ['AL354726.1','HEMGN'],
        ['AL354740.1','METTL7A'],
        ['AL354751.1','PTPRVP'],
        ['AL354760.1','ST7L'],
        ['AL354892.2','TCTE3'],
        ['AL354892.3','MOB1A'],
        ['AL354892.3','MORF4L1'],
        ['AL354953.1','LAMC2'],
        ['AL354956.1','TCAM1P'],
        ['AL355312.2','AL355312.6'],
        ['AL355312.2','LRP11'],
        ['AL355376.1','LINC01517'],
        ['AL355499.1','AL591485.1'],
        ['AL355499.1','HULC'],
        ['AL355916.1','PRKCH'],
        ['AL356235.1','ZC3H12B'],
        ['AL356275.1','AL356275.2'],
        ['AL356275.1','LINC02767'],
        ['AL356289.2','FUT4'],
        ['AL356489.2','TRB@'],
        ['AL356489.3','TRB@'],
        ['AL356489.4','PRSS3'],
        ['AL356585.4','ANKRD36B'],
        ['AL356867.1','SPRR3'],
        ['AL357075.4','RELA'],
        ['AL357078.1','JAK1'],
        ['AL357127.2','LINC02641'],
        ['AL358113.1','LYZ'],
        ['AL358777.1','PAK1IP1'],
        ['AL358790.1','JPX'],
        ['AL359075.1','SEC16B'],
        ['AL359237.1','LINC02330'],
        ['AL359715.1','AL359715.4'],
        ['AL359715.1','BCKDHB'],
        ['AL359715.2','AL359715.5'],
        ['AL359715.2','BCKDHB'],
        ['AL359762.1','SPATA1'],
        ['AL360181.3','SCART1'],
        ['AL360270.1','AL360270.3'],
        ['AL360270.1','CD53'],
        ['AL360295.1','C1ORF168'],
        ['AL365203.2','ITGB1'],
        ['AL365203.3','ITGB1'],
        ['AL365277.1','PPIE'],
        ['AL365357.1','RPS14'],
        ['AL365394.1','SLC5A12'],
        ['AL390036.1','VAV3-AS1'],
        ['AL390195.1','OVGP1'],
        ['AL390726.4','CYP4F26P'],
        ['AL390726.6','ANKRD18B'],
        ['AL390726.6','CYP4F26P'],
        ['AL390728.4','RPGRIP1L'],
        ['AL390728.5','ZNF496'],
        ['AL390730.1','AL390730.2'],
        ['AL390763.1','ANKRD7'],
        ['AL390838.1','RMI1'],
        ['AL390839.1','AL390839.2'],
        ['AL390860.1','TRIM58'],
        ['AL390961.1','FAM238B'],
        ['AL390961.1','FAM238C'],
        ['AL390961.2','FAM238B'],
        ['AL390961.2','FAM238C'],
        ['AL391832.2','AL391832.3'],
        ['AL391839.3','PAQR3'],
        ['AL391840.3','SH3BGRL2'],
        ['AL392046.1','CUL2'],
        ['AL392086.1','LINP1'],
        ['AL392086.2','LINP1'],
        ['AL392086.3','ZNF585A'],
        ['AL392172.1','DISP1'],
        ['AL445238.1','OR7E156P'],
        ['AL445430.1','AL445430.2'],
        ['AL445472.1','KLF8'],
        ['AL445524.2','SCAND2P'],
        ['AL445623.2','ELAVL2'],
        ['AL449043.1','LINC01388'],
        ['AL450124.1','FAM182B'],
        ['AL450322.2','MYH11'],
        ['AL450992.1','NBPF15'],
        ['AL450992.1','NOTCH2NL'],
        ['AL450992.1','TRAF3IP2'],
        ['AL451042.1','AL451042.2'],
        ['AL451085.2','CKS1B'],
        ['AL451136.1','CCDC163'],
        ['AL512310.11','NF1'],
        ['AL512310.12','NF1'],
        ['AL512356.3','TF'],
        ['AL512625.3','GLIDR'],
        ['AL512662.2','BMS1'],
        ['AL513122.1','KRT13'],
        ['AL513122.1','PTRF'],
        ['AL513122.2','KRT13'],
        ['AL513283.1','MYO1D'],
        ['AL513314.2','AL627309.5'],
        ['AL513314.2','LINC00174'],
        ['AL513314.2','LINC01002'],
        ['AL513478.2','PARG'],
        ['AL513548.1','AL513548.4'],
        ['AL583722.1','INF2'],
        ['AL589843.2','MKRN1'],
        ['AL589935.1','LINC00472'],
        ['AL590556.3','AMY2A'],
        ['AL590556.3','CEL'],
        ['AL590556.3','CELA2A'],
        ['AL590556.3','CELA2B'],
        ['AL590556.3','CLPS'],
        ['AL590556.3','CPA1'],
        ['AL590556.3','CPA2'],
        ['AL590556.3','CTRB1'],
        ['AL590556.3','CTRB2'],
        ['AL590556.3','CTRC'],
        ['AL590556.3','EIF4A2'],
        ['AL590556.3','GP2'],
        ['AL590556.3','KRT8'],
        ['AL590556.3','LHFPL5'],
        ['AL590556.3','PNLIP'],
        ['AL590556.3','PRSS1'],
        ['AL590556.3','PRSS2'],
        ['AL590556.3','PRSS3'],
        ['AL590556.3','TRB@'],
        ['AL590617.1','MTCH1'],
        ['AL590705.5','CCDC180'],
        ['AL590762.3','ZCRB1'],
        ['AL590787.1','MTMR6'],
        ['AL590807.1','TRA@'],
        ['AL590867.1','TRA@'],
        ['AL591242.1','PLA2G7'],
        ['AL591485.1','DPY19L2'],
        ['AL591485.1','PLCB1'],
        ['AL591684.2','BMS1'],
        ['AL591806.3','C21ORF62'],
        ['AL591806.3','CCND3'],
        ['AL591848.2','AL591848.3'],
        ['AL591926.2','LINC01597'],
        ['AL592310.1','FAM133B'],
        ['AL606491.1','LDLRAP1'],
        ['AL627171.2','RN7SL151P'],
        ['AL627309.5','AL732372.2'],
        ['AL645568.1','TNFSF4'],
        ['AL645929.1','AL671277.1'],
        ['AL645929.1','AL671883.3'],
        ['AL645937.2','AL672167.1'],
        ['AL645937.4','AL672167.1'],
        ['AL645937.4','AL672167.2'],
        ['AL662791.2','OR2J3'],
        ['AL662791.5','OR2J3'],
        ['AL662899.1','LY6G5C'],
        ['AL669831.4','LINC01881'],
        ['AL671277.1','AL671883.3'],
        ['AL671710.1','ZBED4'],
        ['AL683807.1','AL683807.2'],
        ['AL691447.2','MFSD14C'],
        ['AL731559.1','LINC02306'],
        ['AL732437.3','CALML3-AS1'],
        ['AL807752.7','COL1A1'],
        ['AL845552.1','HMGN1'],
        ['AL845552.2','WDR25'],
        ['AL928654.3','CRIP2'],
        ['AL928768.1','IGHA1'],
        ['AL928768.3','IGHA1'],
        ['AL953883.1','FAM27C'],
        ['ALAS2','MPO'],
        ['ALB','ALDH2'],
        ['ALB','ALDOB'],
        ['ALB','AMBP'],
        ['ALB','APLP2'],
        ['ALB','APOA1-AS'],
        ['ALB','APOB'],
        ['ALB','APOC3'],
        ['ALB','APOH'],
        ['ALB','BAAT'],
        ['ALB','C3'],
        ['ALB','C4BPA'],
        ['ALB','C9'],
        ['ALB','CANX'],
        ['ALB','CCDC152'],
        ['ALB','CCNY'],
        ['ALB','CES1'],
        ['ALB','CFH'],
        ['ALB','CHD4'],
        ['ALB','CLU'],
        ['ALB','CP'],
        ['ALB','CPS1'],
        ['ALB','CRP'],
        ['ALB','CTRB1'],
        ['ALB','CTSB'],
        ['ALB','CTSD'],
        ['ALB','CYP1A2'],
        ['ALB','CYP2A6'],
        ['ALB','CYP2C9'],
        ['ALB','CYP2E1'],
        ['ALB','CYP4A11'],
        ['ALB','CYP4A22'],
        ['ALB','CYP4F2'],
        ['ALB','CYP51A1'],
        ['ALB','DMGDH'],
        ['ALB','EEF1A1'],
        ['ALB','EIF2B5'],
        ['ALB','EIF3A'],
        ['ALB','F2'],
        ['ALB','F9'],
        ['ALB','FFAR4'],
        ['ALB','FGA'],
        ['ALB','FGB'],
        ['ALB','FGG'],
        ['ALB','FGL1'],
        ['ALB','FMO3'],
        ['ALB','FMO5'],
        ['ALB','FN1'],
        ['ALB','GAPDH'],
        ['ALB','GATM'],
        ['ALB','GP2'],
        ['ALB','GSTA2'],
        ['ALB','HAMP'],
        ['ALB','HP'],
        ['ALB','HPS3'],
        ['ALB','HPX'],
        ['ALB','IGHG1'],
        ['ALB','IGK@'],
        ['ALB','IGKC'],
        ['ALB','INHBE'],
        ['ALB','ITIH2'],
        ['ALB','ITM2B'],
        ['ALB','ITPR2'],
        ['ALB','KRBOX1'],
        ['ALB','LBP'],
        ['ALB','LIPA'],
        ['ALB','MAOB'],
        ['ALB','MAT1A'],
        ['ALB','MGST1'],
        ['ALB','MT2A'],
        ['ALB','MYO1B'],
        ['ALB','NDUFS6'],
        ['ALB','NNMT'],
        ['ALB','NUCB1'],
        ['ALB','OPN3'],
        ['ALB','ORM1'],
        ['ALB','ORM2'],
        ['ALB','PCBP1-AS1'],
        ['ALB','PCSK6'],
        ['ALB','PDIA4'],
        ['ALB','PIPOX'],
        ['ALB','PLG'],
        ['ALB','PRAP1'],
        ['ALB','PRDX1'],
        ['ALB','PROC'],
        ['ALB','PTGR1'],
        ['ALB','RRBP1'],
        ['ALB','SAA1'],
        ['ALB','SAA2'],
        ['ALB','SARM1'],
        ['ALB','SCD'],
        ['ALB','SCP2'],
        ['ALB','SERPINA1'],
        ['ALB','SERPINA3'],
        ['ALB','SERPIND1'],
        ['ALB','SERPINE1'],
        ['ALB','SERPING1'],
        ['ALB','SNAP91'],
        ['ALB','ST6GAL1'],
        ['ALB','STAU1'],
        ['ALB','STOM'],
        ['ALB','TAT'],
        ['ALB','TAT-AS1'],
        ['ALB','TF'],
        ['ALB','TMBIM6'],
        ['ALB','TXNL4B'],
        ['ALB','UGT1A8'],
        ['ALB','UGT2B11'],
        ['ALB','ZNF511-PRAP1'],
        ['ALDH18A1','SMR3B'],
        ['ALDH1L1','HTN3'],
        ['ALDH1L1','SMR3B'],
        ['ALDH2','ANKHD1'],
        ['ALDH2','ANKHD1-EIF4EBP3'],
        ['ALDH2','LYZ'],
        ['ALDH2','SFTPB'],
        ['ALDH3A2','SLC47A1'],
        ['ALDH3B2','TMEM99'],
        ['ALDH5A1','LYZ'],
        ['ALDH6A1','C3ORF70'],
        ['ALDH6A1','LRG1'],
        ['ALDH6A1','SFTPB'],
        ['ALDH6A1','SPAST'],
        ['ALDH7A1','CLPS'],
        ['ALDH9A1','USF3'],
        ['ALDOA','IGF2'],
        ['ALDOA','MUC5AC'],
        ['ALDOA','NUDC'],
        ['ALDOA','PIGR'],
        ['ALDOA','PPP4C'],
        ['ALDOA','SMR3B'],
        ['ALDOA','TPM2'],
        ['ALDOA','VIM'],
        ['ALDOB','ANPEP'],
        ['ALDOB','APOC3'],
        ['ALDOB','CRP'],
        ['ALDOB','ENO1'],
        ['ALDOB','HRG'],
        ['ALDOB','MYO1A'],
        ['ALDOB','PIGR'],
        ['ALDOB','REG3A'],
        ['ALDOB','TXNL4B'],
        ['ALG11','YES1'],
        ['ALG14','CNN3'],
        ['ALG1L2','ALG1L6P'],
        ['ALG1L5P','TSSC2'],
        ['ALG9','OSMR'],
        ['ALK','CDK12'],
        ['ALK','DA750114'],
        ['ALK','ETS1'],
        ['ALK','ETV6'],
        ['ALK','IGH@'],
        ['ALK','IGK@'],
        ['ALK','LYZ'],
        ['ALK','TMED2'],
        ['ALK','TRA@'],
        ['ALKBH5','HBB'],
        ['ALKBH5','LLGL1'],
        ['ALMS1','ALMS1-IT1'],
        ['ALMS1','SMOC2'],
        ['ALOX12','RNASEK'],
        ['ALOX12','RNASEK-C17ORF49'],
        ['ALOX12-AS1','MIR497HG'],
        ['ALOX15','C17ORF100'],
        ['ALS2CR12','TRAK2'],
        ['ALX1','LRRIQ1'],
        ['AMACR','C1QTNF3'],
        ['AMACR','SLC45A2'],
        ['AMBP','FGA'],
        ['AMBP','SERPINA1'],
        ['AMBRA1','CHRM4'],
        ['AMBRA1','GFAP'],
        ['AMBRA1','HARBI1'],
        ['AMD1','EEF1A1'],
        ['AMD1','QRSL1'],
        ['AMMECR1L','NUTM2B-AS1'],
        ['AMN1','KIAA1551'],
        ['AMN1','RESF1'],
        ['AMOTL1','IGF2'],
        ['AMOTL1','ZBTB18'],
        ['AMPD3','ETS1'],
        ['AMY2A','CELA2A'],
        ['AMY2A','CELA3A'],
        ['AMY2A','CELA3B'],
        ['AMY2A','CLPS'],
        ['AMY2A','CPA1'],
        ['AMY2A','CPA2'],
        ['AMY2A','CTRB1'],
        ['AMY2A','PNLIP'],
        ['AMY2A','PRSS1'],
        ['AMY2A','PRSS2'],
        ['AMY2A','TRB@'],
        ['AMY2B','CLPS'],
        ['AMY2B','PNLIP'],
        ['AMY2B','PRH1'],
        ['AMY2B','PRH1-PRR4'],
        ['AMY2B','PRH2'],
        ['AMY2B','PRSS1'],
        ['AMY2B','PRSS2'],
        ['AMY2B','PRSS3'],
        ['AMY2B','TRB@'],
        ['AMZ1','APOLD1'],
        ['AMZ1','IGF2'],
        ['AMZ2','BPTF'],
        ['ANAPC1','RMND5A'],
        ['ANAPC15','LAMTOR1'],
        ['ANAPC16','CNOT10'],
        ['ANAPC16','CRLF2'],
        ['ANAPC16','TG'],
        ['ANAPC16','UGDH'],
        ['ANAPC7','KDSR'],
        ['ANGPTL4','PTRF'],
        ['ANK1','AZU1'],
        ['ANK1','LYZ'],
        ['ANK1','S100A9'],
        ['ANK2','CAMK2D'],
        ['ANK2','HIST3H2BB'],
        ['ANK3','CCDC6'],
        ['ANK3','TG'],
        ['ANK3-DT','CDK1'],
        ['ANKAR','ASNSD1'],
        ['ANKAR','ZDHHC7'],
        ['ANKDD1A','PLEKHO2'],
        ['ANKEF1','PDE1C'],
        ['ANKEF1','TACC2'],
        ['ANKH','MORF4L1'],
        ['ANKIB1','CYP51A1-AS1'],
        ['ANKLE2','GOLGA3'],
        ['ANKLE2','POLE'],
        ['ANKMY2','ZNF708'],
        ['ANKRD1','LDB3'],
        ['ANKRD11','HNRNPK'],
        ['ANKRD11','LYZ'],
        ['ANKRD11','STOM'],
        ['ANKRD11','TRB@'],
        ['ANKRD12','PTPRM'],
        ['ANKRD13A','CRISPLD2'],
        ['ANKRD13D','LINC00337'],
        ['ANKRD17','SMR3B'],
        ['ANKRD18B','FAM95C'],
        ['ANKRD18CP','BX664727.3'],
        ['ANKRD18CP','ZNF322'],
        ['ANKRD20A11P','AP005212.4'],
        ['ANKRD20A11P','AP005212.5'],
        ['ANKRD20A17P','CR381670.1'],
        ['ANKRD20A21P','BP-2189O9.2'],
        ['ANKRD20A21P','CR381670.1'],
        ['ANKRD28','CAST'],
        ['ANKRD28','TRB@'],
        ['ANKRD30B','AP005901.1'],
        ['ANKRD34A','LIX1L'],
        ['ANKRD36B','LINC00342'],
        ['ANKRD36C','LINC00342'],
        ['ANKRD42','RPL32'],
        ['ANKRD52','MUC7'],
        ['ANKRD54','LYZ'],
        ['ANKRD62','VWF'],
        ['ANKS1B','APOL1'],
        ['ANO1','AP003555.2'],
        ['ANO1','AP003555.3'],
        ['ANO10','NEK7'],
        ['ANO10','TG'],
        ['ANO2','VWF'],
        ['ANO3','REEP5'],
        ['ANO6','BROX'],
        ['ANOS2P','TMSB4Y'],
        ['ANP32B','CTSS'],
        ['ANP32B','FN1'],
        ['ANP32B','HBB'],
        ['ANP32B','HMGB2'],
        ['ANP32B','IGH@'],
        ['ANP32B','NCL'],
        ['ANP32B','NPM1'],
        ['ANP32B','PTMA'],
        ['ANP32B','TMOD1'],
        ['ANP32B','TNNT2'],
        ['ANP32E','HMGB1'],
        ['ANP32E','NCL'],
        ['ANPEP','CELA2A'],
        ['ANPEP','COL6A2'],
        ['ANPEP','FN1'],
        ['ANPEP','HBB'],
        ['ANPEP','IGH@'],
        ['ANPEP','IGK@'],
        ['ANPEP','IGKC'],
        ['ANPEP','LYZ'],
        ['ANPEP','PIGR'],
        ['ANPEP','REG3A'],
        ['ANPEP','TRB@'],
        ['ANTXR1','IGK@'],
        ['ANTXR2','MORC2-AS1'],
        ['ANXA1','ANXA2'],
        ['ANXA1','COL6A1'],
        ['ANXA1','CRNN'],
        ['ANXA1','HBB'],
        ['ANXA1','HTN3'],
        ['ANXA1','KRT4'],
        ['ANXA1','MAGT1'],
        ['ANXA1','MGLL'],
        ['ANXA1','SMR3B'],
        ['ANXA1','STATH'],
        ['ANXA1','TAGLN'],
        ['ANXA1','TG'],
        ['ANXA1','ZNF207'],
        ['ANXA11','PTRF'],
        ['ANXA11','TRB@'],
        ['ANXA2','CELA2A'],
        ['ANXA2','CRYBG1'],
        ['ANXA2','ERBB2'],
        ['ANXA2','IGF2'],
        ['ANXA2','KRT10'],
        ['ANXA2','PNLIP'],
        ['ANXA2','PRB2'],
        ['ANXA2','RNF111'],
        ['ANXA2','S100A9'],
        ['ANXA3','ZNF350'],
        ['ANXA4','PRSS1'],
        ['ANXA5','CLEC4E'],
        ['ANXA5','PCSK7'],
        ['ANXA6','PLIN1'],
        ['ANXA6','TNIP1'],
        ['AOAH','CXCL8'],
        ['AOC3','FTL'],
        ['AOPEP','CPSF6'],
        ['AOPEP','MIR205HG'],
        ['AOX1','PRSS1'],
        ['AOX1','SPTLC2'],
        ['AOX1','TRB@'],
        ['AP000295.9','ZNF595'],
        ['AP000304.12','HTN1'],
        ['AP000317.1','MRPS6'],
        ['AP000317.1','SLC5A3'],
        ['AP000322.53','KCNE1'],
        ['AP000344.1','LINC01659'],
        ['AP000344.4','LINC01659'],
        ['AP000345.2','DRICH1'],
        ['AP000345.3','DRICH1'],
        ['AP000346.1','AP000347.1'],
        ['AP000346.1','AP000347.2'],
        ['AP000346.1','IGLL3P'],
        ['AP000346.2','AP000347.1'],
        ['AP000346.2','AP000347.2'],
        ['AP000347.2','KB-1572G7.3'],
        ['AP000350.4','SFTPA1'],
        ['AP000351.4','GSTTP1'],
        ['AP000356.2','POM121L4P'],
        ['AP000356.2','POM121L7P'],
        ['AP000356.2','USP18'],
        ['AP000356.3','BCR'],
        ['AP000438.2','CYP4F3'],
        ['AP000473.1','MIR99AHG'],
        ['AP000527.1','FP475955.3'],
        ['AP000527.1','LINC01297'],
        ['AP000542.3','DA750114'],
        ['AP000553.4','PPIL2'],
        ['AP000553.5','PPIL2'],
        ['AP000662.1','CTSS'],
        ['AP000688.14','PIGR'],
        ['AP000695.6','CHAF1B'],
        ['AP000781.2','IGF2'],
        ['AP000781.2','TG'],
        ['AP000790.1','YWHAZ'],
        ['AP000790.2','YWHAZ'],
        ['AP000812.5','FOLR3'],
        ['AP000860.2','SLC25A25-AS1'],
        ['AP000880.1','NCAM1-AS1'],
        ['AP000915.2','PIPOX'],
        ['AP000924.1','FDX1'],
        ['AP000936.1','PCSK7'],
        ['AP000943.1','AP000943.2'],
        ['AP000943.2','MRE11'],
        ['AP000943.3','MRE11'],
        ['AP000943.4','FUT4'],
        ['AP001005.3','TUBB8'],
        ['AP001005.4','TUBB8'],
        ['AP001011.1','SGCB'],
        ['AP001107.1','EMP2'],
        ['AP001350.2','CNTF'],
        ['AP001350.2','ZFP91-CNTF'],
        ['AP001350.4','CNTF'],
        ['AP001350.4','ZFP91-CNTF'],
        ['AP001372.3','SERPINA10'],
        ['AP001372.3','TCF3'],
        ['AP001372.3','ZNF726'],
        ['AP001528.1','FZD4'],
        ['AP001528.1','TMEM135'],
        ['AP001528.2','FZD4'],
        ['AP001931.1','TMX2'],
        ['AP001979.2','SPATA19'],
        ['AP002008.1','BTG4'],
        ['AP002495.1','FBXO42'],
        ['AP002748.1','CTSF'],
        ['AP002748.2','CTSF'],
        ['AP002852.1','ODF1'],
        ['AP002884.2','IL18'],
        ['AP002884.4','IL18'],
        ['AP002956.1','EMC10'],
        ['AP002956.1','KRT13'],
        ['AP002957.1','FNDC3B'],
        ['AP002990.1','HBB'],
        ['AP003066.1','AP003781.1'],
        ['AP003066.1','LINC02737'],
        ['AP003068.18','AP003068.23'],
        ['AP003084.1','EED'],
        ['AP003086.1','GAB2'],
        ['AP003354.2','ATP6V1C1'],
        ['AP003400.1','NOX4'],
        ['AP003472.1','AP003472.2'],
        ['AP003497.1','HMGB3'],
        ['AP005121.1','LINC01906'],
        ['AP005242.5','LINC00349'],
        ['AP005329.1','MYOM1'],
        ['AP005435.2','EMB'],
        ['AP005482.3','ZNF486'],
        ['AP005530.1','C2ORF27B'],
        ['AP005671.1','EPB41L3'],
        ['AP005901.3','AP005901.5'],
        ['AP006219.2','AP006219.3'],
        ['AP006222.1','DUX4'],
        ['AP006621.8','PRH1'],
        ['AP006621.8','PRH1-PRR4'],
        ['AP006621.8','PRR4'],
        ['AP1B1','QRICH1'],
        ['AP1M1','KLHL36'],
        ['AP2M1','FN1'],
        ['AP2M1','VIM'],
        ['AP3B2','SRSF9'],
        ['AP3D1','FLNA'],
        ['AP3D1','KRT4'],
        ['AP3D1','RNF10'],
        ['AP3D1','SMR3B'],
        ['AP3S1','COMMD10'],
        ['AP3S2','YPEL5'],
        ['AP4B1','HIPK1-AS1'],
        ['AP4B1','PTPN22'],
        ['AP5B1','GLTP'],
        ['AP5B1','LYZ'],
        ['APBB1IP','FAM238A'],
        ['APBB2','BRD4'],
        ['APBB2','NR3C2'],
        ['APBB2','SPAG9'],
        ['APBB2','TG'],
        ['APBB3','NDUFA2'],
        ['APBB3','SRA1'],
        ['APC2','GSR'],
        ['APCDD1L-AS1','VAPB'],
        ['APLP2','CANX'],
        ['APLP2','EEF1A1'],
        ['APLP2','HLA@'],
        ['APLP2','HSP90B1'],
        ['APLP2','LAMB2'],
        ['APLP2','PRSS1'],
        ['APLP2','SFTPA2'],
        ['APLP2','SFTPC'],
        ['APMAP','ID2'],
        ['APOA1','CLU'],
        ['APOA1','FGB'],
        ['APOA1','IGHG1'],
        ['APOA1','P4HB'],
        ['APOA1','PTMS'],
        ['APOA1','SARM1'],
        ['APOA1','SERPINA1'],
        ['APOA1','SIK3'],
        ['APOA1-AS','ATF5'],
        ['APOA1-AS','C3'],
        ['APOA1-AS','FGA'],
        ['APOA1-AS','HRG'],
        ['APOA1-AS','MIDN'],
        ['APOA1-AS','SERPINA1'],
        ['APOA4','IGH@'],
        ['APOA4','IL9R'],
        ['APOA4','PIGR'],
        ['APOB','C3'],
        ['APOB','CRP'],
        ['APOB','IGKC'],
        ['APOB','PIGR'],
        ['APOB','SERPINA1'],
        ['APOB','SERPINA3'],
        ['APOBEC3C','PTRF'],
        ['APOBEC3F','APOBEC3G'],
        ['APOC1','SERPINA1'],
        ['APOC1P1','APOC4-APOC2'],
        ['APOC2','ISOC2'],
        ['APOC3','C3'],
        ['APOC3','FTL'],
        ['APOC3','SERPINA1'],
        ['APOD','MYH11'],
        ['APOE','FFAR4'],
        ['APOE','FGG'],
        ['APOE','RBP4'],
        ['APOE','SCAP'],
        ['APOE','TOMM40'],
        ['APOH','C3'],
        ['APOH','FGB'],
        ['APOH','SERPINA1'],
        ['APOL1','DA750114'],
        ['APOL1','LINC01138'],
        ['APOL1','SCYL3'],
        ['APOL1','SFTPA2'],
        ['APOL1','TNFRSF1B'],
        ['APOL3','MIR100HG'],
        ['APOLD1','DPF3'],
        ['APOLD1','PARVA'],
        ['APOLD1','TG'],
        ['APP','FAM172A'],
        ['APP','GFAP'],
        ['APP','GLUL'],
        ['APP','HBB'],
        ['APP','HLA@'],
        ['APP','HTN3'],
        ['APP','IGF2'],
        ['APP','KRT10'],
        ['APP','KRT13'],
        ['APP','LYZ'],
        ['APP','MUC7'],
        ['APP','MYH11'],
        ['APP','MYH7'],
        ['APP','PIGR'],
        ['APP','PNLIP'],
        ['APP','PRSS1'],
        ['APP','PRSS2'],
        ['APP','PTMS'],
        ['APP','SFTPC'],
        ['APP','SMR3B'],
        ['APP','STATH'],
        ['APP','TRB@'],
        ['APPBP2','USP32'],
        ['APPL1','COPA'],
        ['AQP1','NPPA-AS1'],
        ['AQP2','GPX3'],
        ['AQP3','MYH11'],
        ['AQP3','SMR3B'],
        ['AQP4','CHST9'],
        ['AQP4-AS1','MBP'],
        ['AQP5','AQP6'],
        ['AQP7','CH17-469D17.1'],
        ['AQP8','PRSS1'],
        ['AQP8','PRSS2'],
        ['AQP8','TRB@'],
        ['ARAP1','PRB3'],
        ['ARCN1','CLCN6'],
        ['ARCN1','FN1'],
        ['ARCN1','LRRC28'],
        ['ARCN1','NPPA-AS1'],
        ['ARCN1','PTRF'],
        ['AREL1','TG'],
        ['ARF1','ARF5'],
        ['ARF1','H3F3A'],
        ['ARF1','HMGB1'],
        ['ARF1','IVNS1ABP'],
        ['ARF1','LYZ'],
        ['ARF1','PTMA'],
        ['ARF3','FKBP11'],
        ['ARF3','WNT10B'],
        ['ARF4','DENND6A'],
        ['ARF4-AS1','DENND6A-AS1'],
        ['ARF6','SFTPC'],
        ['ARFGAP1','PNLIP'],
        ['ARFGAP2','CIB1'],
        ['ARFGAP3','PACSIN2'],
        ['ARG1','FGG'],
        ['ARGLU1','IGFBP5'],
        ['ARHGAP1','CPT1A'],
        ['ARHGAP1','FLG2'],
        ['ARHGAP1','NPPA'],
        ['ARHGAP1','PTRF'],
        ['ARHGAP11A','SCG5'],
        ['ARHGAP11B','OTUD7A'],
        ['ARHGAP18','DLGAP5'],
        ['ARHGAP18','PNLIP'],
        ['ARHGAP18','TMEM244'],
        ['ARHGAP23','COL5A1'],
        ['ARHGAP23','MARCKS'],
        ['ARHGAP26','NR3C1'],
        ['ARHGAP27','PLEKHM1'],
        ['ARHGAP27P1-BPTFP1-KPNA2P3','BPTF'],
        ['ARHGAP27P1-BPTFP1-KPNA2P3','KPNA2'],
        ['ARHGAP35','CTSS'],
        ['ARHGAP35','TG'],
        ['ARHGAP4','OPN1LW'],
        ['ARHGAP44','MAGT1'],
        ['ARHGDIA','FLNA'],
        ['ARHGDIA','MUC7'],
        ['ARHGDIB','ERP27'],
        ['ARHGDIB','MYH9'],
        ['ARHGDIG','FAM234A'],
        ['ARHGEF1','NPPA-AS1'],
        ['ARHGEF12','MUC7'],
        ['ARHGEF12','PCBP2'],
        ['ARHGEF12','SMR3B'],
        ['ARHGEF17','P2RY6'],
        ['ARHGEF2','SSR2'],
        ['ARHGEF26-AS1','TRA@'],
        ['ARHGEF28','MTX2'],
        ['ARHGEF3','FGD5-AS1'],
        ['ARHGEF5','OR2A1-AS1'],
        ['ARHGEF7','TEX29'],
        ['ARHGEF9','LINC01278'],
        ['ARID1A','CTSB'],
        ['ARID1A','TG'],
        ['ARID1A','WSB1'],
        ['ARID1B','CD24'],
        ['ARID1B','PIGR'],
        ['ARID1B','SMR3B'],
        ['ARID2','INSR'],
        ['ARID3A','LINC01578'],
        ['ARID4B','B2M'],
        ['ARID4B','LYZ'],
        ['ARIH2','LYZ'],
        ['ARIH2','TG'],
        ['ARL10','KRT13'],
        ['ARL14EP','MAOB'],
        ['ARL15','KLK3'],
        ['ARL17A','KANSL1'],
        ['ARL17B','BPTF'],
        ['ARL17B','KANSL1'],
        ['ARL2-SNX15','WSB1'],
        ['ARL4A','NSUN7'],
        ['ARL6','EPHA6'],
        ['ARL6IP1','LYZ'],
        ['ARL6IP1','RPS15A'],
        ['ARL6IP1','SMG1'],
        ['ARL6IP6','TRNT1'],
        ['ARL8A','PTPN7'],
        ['ARL8B','TASOR'],
        ['ARL9','SRP72'],
        ['ARMC6','SLC25A42'],
        ['ARMCX4','TG'],
        ['ARMH3','LARP4'],
        ['ARMS2','PLEKHA1'],
        ['ARNT','CTSK'],
        ['ARNT','MYH11'],
        ['ARPC1A','MMD'],
        ['ARPC1A','PAPOLA'],
        ['ARPC1B','PDE3A'],
        ['ARPC2','EDC3'],
        ['ARPC2','LCORL'],
        ['ARPC2','LYZ'],
        ['ARPC2','NABP2'],
        ['ARPC3','ARPC3P1'],
        ['ARPC4','OGG1'],
        ['ARPC4','TTLL3'],
        ['ARPC4-TTLL3','CNTN4'],
        ['ARPC4-TTLL3','OGG1'],
        ['ARPC4-TTLL3','SFTPB'],
        ['ARPIN','RAI14'],
        ['ARPP19','RPL37A'],
        ['ARRB2','LYZ'],
        ['ARRB2','WNT4'],
        ['ARRDC1','EHMT1'],
        ['ARRDC1','OSTF1'],
        ['ARRDC3','BCLAF1'],
        ['ARRDC3','HIST1H2BD'],
        ['ARRDC3','MPO'],
        ['ARRDC3','TNRC6B'],
        ['ARSD','BCL2'],
        ['ARSD','EPN1'],
        ['ARSD','PIK3C2B'],
        ['ARSD','SSH2'],
        ['ARSD','ZNF668'],
        ['ARSE','TAF13'],
        ['ARSG','CARNS1'],
        ['ART4','SMCO3'],
        ['AS3MT','CCND3'],
        ['AS3MT','CNNM2'],
        ['ASAH1','B2M'],
        ['ASAH1','PCM1'],
        ['ASAP3','TCEA3'],
        ['ASB11','ASB9'],
        ['ASB12','MTMR8'],
        ['ASB15','LMOD2'],
        ['ASCC1','SPOCK2'],
        ['ASCC1','TG'],
        ['ASCC3','PDE10A'],
        ['ASF1B','PRKACA'],
        ['ASH1L','GABRG2'],
        ['ASH1L','GBP6'],
        ['ASH1L','STATH'],
        ['ASH1L','TG'],
        ['ASIP','EIF5'],
        ['ASL','CRCP'],
        ['ASL','GUSBP11'],
        ['ASLP1','KB-1572G7.3'],
        ['ASMTL','SLC25A6'],
        ['ASNS','RBFOX1'],
        ['ASPG','TDRD9'],
        ['ASPH','PRB3'],
        ['ASPH','TG'],
        ['ASRGL1','SCGB1A1'],
        ['ASS1','ASS1P11'],
        ['ASS1','ASS1P12'],
        ['ASS1','HMCN2'],
        ['ASTN2','DA750114'],
        ['ASTN2','IGH@'],
        ['ASTN2','TRA@'],
        ['ASXL1','KIF3B'],
        ['ASXL1','NOL4L'],
        ['ATAD1','PTEN'],
        ['ATAD2','NCL'],
        ['ATAD2','NPM1'],
        ['ATAD3B','ATAD3C'],
        ['ATF3','HNRNPH1'],
        ['ATF3','TG'],
        ['ATF5','C1S'],
        ['ATF5','CLU'],
        ['ATF5','ORM1'],
        ['ATF5','SERPINA1'],
        ['ATF5','TG'],
        ['ATF5','TXNL4B'],
        ['ATF6','DUSP12'],
        ['ATF6','PNLIP'],
        ['ATF7','SFTPC'],
        ['ATF7IP','IGF2'],
        ['ATF7IP','IL3RA'],
        ['ATF7IP','ZNF432'],
        ['ATG12','CDO1'],
        ['ATG13','NRAS'],
        ['ATG13','SMIM14'],
        ['ATG14','JPT1'],
        ['ATG14','LYZ'],
        ['ATG16L1','INPP5D'],
        ['ATG16L1','SAG'],
        ['ATG16L2','DSC3'],
        ['ATG16L2','IGK@'],
        ['ATG7','RAF1'],
        ['ATHL1','TRB@'],
        ['ATL2','CXCR4'],
        ['ATL2','TG'],
        ['ATL3','RAD17'],
        ['ATL3','TPM4'],
        ['ATM','LINC01091'],
        ['ATM','PRH1'],
        ['ATM','PRH1-PRR4'],
        ['ATM','PRR4'],
        ['ATM','RHOBTB3'],
        ['ATN1','IGFBP5'],
        ['ATN1','SFTPA1'],
        ['ATOH8','TXNDC5'],
        ['ATP10B','LINC02159'],
        ['ATP10B','UPK1A'],
        ['ATP10D','USP32'],
        ['ATP11B','MYH11'],
        ['ATP12A','TUBGCP2'],
        ['ATP13A1','TG'],
        ['ATP13A3','UBR3'],
        ['ATP13A5-AS1','HRASLS'],
        ['ATP1A1','CTSB'],
        ['ATP1A1','GP2'],
        ['ATP1A1','IGH@'],
        ['ATP1A1','LYZ'],
        ['ATP1A1','PIGR'],
        ['ATP1A1','SUN1'],
        ['ATP1A1-AS1','ATP1A3'],
        ['ATP1A1-AS1','IGH@'],
        ['ATP1A1-AS1','LINC01762'],
        ['ATP1A1-AS1','MUC7'],
        ['ATP1A1-AS1','TG'],
        ['ATP1A2','GFAP'],
        ['ATP1A2','NRXN3'],
        ['ATP1B1','CEL'],
        ['ATP1B1','CRLF2'],
        ['ATP1B1','CYP11A1'],
        ['ATP1B1','DTWD1'],
        ['ATP1B1','FNTA'],
        ['ATP1B2','LINC00649'],
        ['ATP2A1','LRP1'],
        ['ATP2A2','LYZ'],
        ['ATP2A2','MMP14'],
        ['ATP2A2','MYH6'],
        ['ATP2A2','MYH9'],
        ['ATP2A3','GANAB'],
        ['ATP2A3','GP2'],
        ['ATP2A3','HBB'],
        ['ATP2A3','LYZ'],
        ['ATP2A3','PNLIP'],
        ['ATP2B2','PRB3'],
        ['ATP2B4','NPPA-AS1'],
        ['ATP2B4','TRB@'],
        ['ATP2C1','MBNL1'],
        ['ATP2C2-AS1','MEAK7'],
        ['ATP5A1','IGF2'],
        ['ATP5A1','LYZ'],
        ['ATP5A1','NPPA-AS1'],
        ['ATP5B','FN1'],
        ['ATP5B','KRT13'],
        ['ATP5B','MIR762HG'],
        ['ATP5C1','ITIH2'],
        ['ATP5D','FAM96B'],
        ['ATP5F1E','CTSZ'],
        ['ATP5G2','TFRC'],
        ['ATP5G3','NPPA-AS1'],
        ['ATP5H','TG'],
        ['ATP5I','MFSD7'],
        ['ATP5L','SLC25A37'],
        ['ATP5MC3','CLCN6'],
        ['ATP5ME','SLC49A3'],
        ['ATP5MG','KMT2A'],
        ['ATP6AP1','GNAS'],
        ['ATP6AP1','PTMS'],
        ['ATP6AP2','SORL1'],
        ['ATP6V0A1','YBX3'],
        ['ATP6V0A2','TCTN2'],
        ['ATP6V0A4','SVOPL'],
        ['ATP6V0B','DPH2'],
        ['ATP6V0C','DDX5'],
        ['ATP6V0C','EEF1A1'],
        ['ATP6V0C','ZEB2'],
        ['ATP6V0E1','GSN'],
        ['ATP6V1A','MTREX'],
        ['ATP6V1B1','VAX2'],
        ['ATP6V1C1','AZIN1-AS1'],
        ['ATP6V1C1','ECE1'],
        ['ATP6V1C1','KB-1507C5.4'],
        ['ATP6V1F','FLNC'],
        ['ATP6V1G2-DDX39B','DDX39A'],
        ['ATP6V1G2-DDX39B','LYZ'],
        ['ATP6V1G2-DDX39B','XXBAC-BPG299F13.16'],
        ['ATP6V1H','STARD4-AS1'],
        ['ATP7A','LYZ'],
        ['ATP7A','ZC3H12B'],
        ['ATP7B','LINC00282'],
        ['ATP8A2','NUP58'],
        ['ATP8B1','AZGP1'],
        ['ATP8B1','TG'],
        ['ATP9A','NFATC2'],
        ['ATPIF1','HBB'],
        ['ATRAID','GTF3C2'],
        ['ATXN1','EEF1A1'],
        ['ATXN1','FABP3'],
        ['ATXN2','PECAM1'],
        ['ATXN2','PEX26'],
        ['ATXN3','THAP11'],
        ['AUH','RNF4'],
        ['AURKA','ERBB2'],
        ['AURKA','RAB3GAP2'],
        ['AURKC','ZNF264'],
        ['AUTS2','LYZ'],
        ['AUTS2','PRB3'],
        ['AUTS2','PRH2'],
        ['AUTS2','SMR3B'],
        ['AVEN','CCDC69'],
        ['AXDND1','EPHB1'],
        ['AXDND1','UPRT'],
        ['AXL','SFTPB'],
        ['AZGP1','CA6'],
        ['AZGP1','CD36'],
        ['AZGP1','GJC3'],
        ['AZGP1','LYZ'],
        ['AZGP1','MYH11'],
        ['AZGP1','PRB1'],
        ['AZGP1','PRB2'],
        ['AZGP1','PRB4'],
        ['AZGP1','PRH1'],
        ['AZGP1','PRH1-PRR4'],
        ['AZGP1','PRR4'],
        ['AZGP1','SMR3B'],
        ['AZGP1','TGM4'],
        ['AZIN1','KLF10'],
        ['AZIN1','SMR3B'],
        ['AZIN1','TG'],
        ['AZIN1','TRB@'],
        ['AZIN1','YWHAZ'],
        ['AZU1','CLTC'],
        ['AZU1','DDX39B'],
        ['AZU1','DNM2'],
        ['AZU1','EEF1A1'],
        ['AZU1','ELANE'],
        ['AZU1','IGHG1'],
        ['AZU1','ILF3'],
        ['AZU1','LCP1'],
        ['AZU1','MYH9'],
        ['AZU1','NARF'],
        ['AZU1','NEMP1'],
        ['AZU1','NR3C1'],
        ['AZU1','OOEP-AS1'],
        ['AZU1','PLEK'],
        ['AZU1','PTBP1'],
        ['AZU1','PTP4A1'],
        ['AZU1','RBM3'],
        ['AZU1','S100A8'],
        ['AZU1','S100A9'],
        ['AZU1','SRGN'],
        ['AZU1','TUBGCP3'],
        ['AZU1','ZMYM1'],
        ['B2M','CALM1'],
        ['B2M','CD36'],
        ['B2M','CD99'],
        ['B2M','CDK6'],
        ['B2M','CELA3A'],
        ['B2M','CSNK1E'],
        ['B2M','CXCR4'],
        ['B2M','EEF1A1'],
        ['B2M','ERV3-1'],
        ['B2M','FBN1'],
        ['B2M','FGA'],
        ['B2M','FUS'],
        ['B2M','G3BP2'],
        ['B2M','GNAS'],
        ['B2M','HBA2'],
        ['B2M','HMGB1'],
        ['B2M','HMGB2'],
        ['B2M','HSPA4'],
        ['B2M','HSPA9'],
        ['B2M','IGH@'],
        ['B2M','IGK@'],
        ['B2M','KIF2A'],
        ['B2M','KLF13'],
        ['B2M','LINC01578'],
        ['B2M','MED13L'],
        ['B2M','MYCBP'],
        ['B2M','PFKFB3'],
        ['B2M','PTMA'],
        ['B2M','PTP4A1'],
        ['B2M','RASA3'],
        ['B2M','RBBP6'],
        ['B2M','REG3A'],
        ['B2M','RNF166'],
        ['B2M','RUNX1'],
        ['B2M','S100A8'],
        ['B2M','SEPTIN9'],
        ['B2M','SF1'],
        ['B2M','SQRDL'],
        ['B2M','SUB1'],
        ['B2M','SYN3'],
        ['B2M','TLN1'],
        ['B2M','TMEM91'],
        ['B2M','TMSB4X'],
        ['B2M','TRAK2'],
        ['B2M','UBB'],
        ['B2M','UBC'],
        ['B2M','UBE3A'],
        ['B2M','WIPF1'],
        ['B3GAT3','FCGBP'],
        ['B3GNT2','COMMD1'],
        ['B3GNT5','KLHL24'],
        ['B3GNT5','TIPARP'],
        ['B3GNTL1','METRNL'],
        ['B3GNTL1','TRAF6'],
        ['B4GALNT2','EIF3I'],
        ['B4GALNT3','NCL'],
        ['B4GALT4','IGSF11'],
        ['BAAT','TRMT112P4'],
        ['BABAM2','RBKS'],
        ['BACE1','GP2'],
        ['BACE1-AS','CLPS'],
        ['BACE1-AS','PRSS1'],
        ['BACE2','IGF2'],
        ['BACH1','BACH1-IT1'],
        ['BACH2','CPA4'],
        ['BACH2','DA750114'],
        ['BACH2','EEF1A1'],
        ['BACH2','IGH@'],
        ['BACH2','IGK@'],
        ['BACH2','LFNG'],
        ['BACH2','TRA@'],
        ['BAG1','LYZ'],
        ['BAG2','ZNF451'],
        ['BAG6','C6ORF47'],
        ['BAG6','FN1'],
        ['BAG6','SFTPC'],
        ['BAG6','SMR3B'],
        ['BAG6','SOD2'],
        ['BAG6','TG'],
        ['BAGE2','GBP6'],
        ['BAHD1','IVD'],
        ['BAIAP2','LYZ'],
        ['BAIAP2','TFB2M'],
        ['BAIAP2-DT','BPTF'],
        ['BAIAP2L2','SLC16A8'],
        ['BANK1','BRI3BP'],
        ['BANK1','CTSB'],
        ['BANK1','DA750114'],
        ['BANK1','IGH@'],
        ['BANK1','IGK@'],
        ['BANK1','LEPROT'],
        ['BANK1','LYZ'],
        ['BANK1','OSGEP'],
        ['BANK1','RNF152'],
        ['BANK1','TRA@'],
        ['BANK1','TRB@'],
        ['BANP','RRN3'],
        ['BASP1','DSE'],
        ['BASP1','LINC01435'],
        ['BASP1','Z84488.2'],
        ['BAZ2B','MARCH7'],
        ['BBOX1-AS1','SLC5A12'],
        ['BBS2','NUDT21'],
        ['BBS5','KLHL41'],
        ['BCAM','LYZ'],
        ['BCAM','SFTPA2'],
        ['BCAN','PTPRZ1'],
        ['BCAP29','DUS4L'],
        ['BCAP29','SLC26A4'],
        ['BCAR1','CPA1'],
        ['BCAS1','LYZ'],
        ['BCAS1','MID1IP1'],
        ['BCAS1','PRKAA1'],
        ['BCAS2','DENND2C'],
        ['BCAS3','TG'],
        ['BCKDHA','TMEM91'],
        ['BCKDHB','DENND1A'],
        ['BCKDK','KAT8'],
        ['BCL10','NIPAL1'],
        ['BCL11A','ITM2A'],
        ['BCL2','LYZ'],
        ['BCL2','NASP'],
        ['BCL2','NEAT1'],
        ['BCL2','RNF152'],
        ['BCL2L11','CXCR4'],
        ['BCL2L12','PRMT1'],
        ['BCL2L13','LYZ'],
        ['BCL2L15','PTPN22'],
        ['BCL2L2-PABPN1','KRT1'],
        ['BCL6','NAMPT'],
        ['BCL6','SERPINA1'],
        ['BCL7A','RBM38'],
        ['BCL9','SH3PXD2A'],
        ['BCL9','TSPAN14'],
        ['BCL9L','LYZ'],
        ['BCL9L','SMR3B'],
        ['BCLAF1','EEF1A1'],
        ['BCLAF1','LYZ'],
        ['BCO1','TRIM56'],
        ['BDH1','CTSB'],
        ['BDH1','IGF2'],
        ['BDKRB1','BDKRB2'],
        ['BEAN1-AS1','PIGR'],
        ['BEST1','EEF2'],
        ['BEST1','HBB'],
        ['BEST1','IGF2'],
        ['BEST1','LYZ'],
        ['BEST1','MYH6'],
        ['BEST1','TG'],
        ['BFAR','SYN3'],
        ['BFSP2','KRT13'],
        ['BFSP2','KRT14'],
        ['BGLAP','PMF1'],
        ['BGN','CKMT2-AS1'],
        ['BGN','CLCN6'],
        ['BGN','FAM102B'],
        ['BGN','GADL1'],
        ['BGN','IGHG1'],
        ['BGN','IGHM'],
        ['BGN','NPPA-AS1'],
        ['BGN','PRKCH'],
        ['BHLHB9','LINC00630'],
        ['BHLHE40','FOXL2NB'],
        ['BHLHE40','TRB@'],
        ['BHLHE40','WDR3'],
        ['BHLHE41','CTR9'],
        ['BHMT','ZNF414'],
        ['BHMT2','CYP2B6'],
        ['BHMT2','EBF1'],
        ['BICD1','DES'],
        ['BICD2','IPPK'],
        ['BICD2','PTMS'],
        ['BID','SLC16A3'],
        ['BIN2','CACNB2'],
        ['BIRC6','TG'],
        ['BIRC6','TRB@'],
        ['BIRC6','YIPF4'],
        ['BLOC1S3','PTRF'],
        ['BLOC1S5-TXNDC5','HTN3'],
        ['BLOC1S5-TXNDC5','SMR3B'],
        ['BLOC1S6','GNG4'],
        ['BLOC1S6','LYZ'],
        ['BLOC1S6','SMAD4'],
        ['BLOC1S6','SQOR'],
        ['BLVRA','VMA21'],
        ['BMI1','DDX5'],
        ['BMP10','CLCN6'],
        ['BMP10','NPPA-AS1'],
        ['BMP10','TNNC1'],
        ['BMP7','ID4'],
        ['BMPR1A','MIR100HG'],
        ['BMPR1B','PDLIM5'],
        ['BMPR2','SFTPB'],
        ['BMPR2','TG'],
        ['BMS1','BMS1P4'],
        ['BMS1','BMS1P4-AGAP5'],
        ['BMS1','LINC00843'],
        ['BMS1','MBL1P'],
        ['BMS1P10','LL22NC03-80A10.6'],
        ['BMS1P20','LL22NC03-2H8.4'],
        ['BMS1P4','PARG'],
        ['BMS1P4','TIMM23B'],
        ['BMS1P4','TIMM23B-AGAP6'],
        ['BMS1P4-AGAP5','PARG'],
        ['BNIP2','PRAMEF17'],
        ['BNIP3L','HBB'],
        ['BNIP3L','HTN1'],
        ['BNIP3L','STATH'],
        ['BNIP3L','TG'],
        ['BNIPL','FLG2'],
        ['BNIPL','IGH@'],
        ['BNIPL','PSPH'],
        ['BOLA2','SMG1'],
        ['BOLA2-SMG1P6','SMG1'],
        ['BOLA2B','SMG1'],
        ['BORCS7-ASMT','CCND3'],
        ['BP-21264C1.2','MAFIP'],
        ['BP-21264C1.2','TEKT4P2'],
        ['BP-21264C1.2','ZNF717'],
        ['BPGM','CALD1'],
        ['BPGM','HBB'],
        ['BPI','CFL1'],
        ['BPI','NUMA1'],
        ['BPI','S100A9'],
        ['BPIFA2','HTN1'],
        ['BPIFA2','LYZ'],
        ['BPIFA2','MOCOS'],
        ['BPIFA2','MUC7'],
        ['BPIFA2','PIGR'],
        ['BPIFA2','PRH2'],
        ['BPIFA2','SMR3B'],
        ['BPIFB1','DDX17'],
        ['BPIFB1','IGH@'],
        ['BPIFB1','LYZ'],
        ['BPIFB2','SMR3B'],
        ['BPIFC','RTCB'],
        ['BPNT1','PDXDC2P'],
        ['BPNT1','TSPAN31'],
        ['BPTF','KPNA2'],
        ['BPTF','LRRC37A2'],
        ['BRD1','C22ORF34'],
        ['BRD2','SRSF5'],
        ['BRD3','TG'],
        ['BRD4','GNAO1'],
        ['BRD4','TG'],
        ['BRD7','SLC22A10'],
        ['BRD9P2','TPPP'],
        ['BRDT','EPHX4'],
        ['BRF1','TRIM22'],
        ['BRI3BP','MYO1D'],
        ['BRK1','VHL'],
        ['BRS3','HTATSF1'],
        ['BSCL2','LYZ'],
        ['BSDC1','FAM229A'],
        ['BSDC1','HDAC1'],
        ['BSG','CDC34'],
        ['BSG','ELANE'],
        ['BSG','LYZ'],
        ['BSG','NR1H2'],
        ['BSG','OAZ1'],
        ['BSG','PTBP1'],
        ['BSG','R3HDM4'],
        ['BSG','RNF126'],
        ['BSG','SBNO2'],
        ['BSG','TCF3'],
        ['BSG','UBB'],
        ['BST1','FAM200B'],
        ['BTBD1','SPP1'],
        ['BTBD2','CAPNS1'],
        ['BTBD6','LOXL1'],
        ['BTBD8','C1ORF146'],
        ['BTBD8','KIAA1107'],
        ['BTG1','HBB'],
        ['BTG2','CEL'],
        ['BTG2','GP2'],
        ['BTG2','IGF2'],
        ['BTG2','SRGN'],
        ['BTG2','TG'],
        ['BTG4','POU2AF1'],
        ['BTN2A1','BTN2A2'],
        ['BTN2A1','BTN2A3P'],
        ['BTN2A3P','BTN3A2'],
        ['BTN3A2','SCAMP1-AS1'],
        ['BTNL9','POLR1A'],
        ['BVES','EMC2'],
        ['BVES','NUBPL'],
        ['BX004807.1','EFCAB7'],
        ['BX072579.2','LINC00452'],
        ['BX276092.10','BX276092.7'],
        ['BX276092.7','BX276092.9'],
        ['BX284632.1','IGK@'],
        ['BX284668.3','EIF1AX'],
        ['BX322557.2','LINC00205'],
        ['BX322639.1','ZNF355P'],
        ['BX469938.1','CPXM2'],
        ['BX470102.3','S100A6'],
        ['BX547991.1','LINC00864'],
        ['BZW1','CBL'],
        ['BZW1','NORAD'],
        ['BZW1','SNX1'],
        ['C10ORF10','PNLIP'],
        ['C10ORF10','SFTPA2'],
        ['C10ORF10','SFTPC'],
        ['C10ORF25','CEP164P1'],
        ['C10ORF32-ASMT','CNNM2'],
        ['C10ORF32-ASMT','LYZ'],
        ['C10ORF67','LINC01552'],
        ['C10ORF90','LINC00601'],
        ['C11ORF54','SP3'],
        ['C11ORF57','DLAT'],
        ['C11ORF63','GOLGA4'],
        ['C11ORF70','YAP1'],
        ['C11ORF80','SF1'],
        ['C12ORF45','DA750114'],
        ['C12ORF45','IGH@'],
        ['C12ORF45','LYRM2'],
        ['C12ORF45','PRKCSH'],
        ['C12ORF49','TG'],
        ['C12ORF60','TG'],
        ['C12ORF65','NIPAL1'],
        ['C12ORF73','HSP90B1'],
        ['C12ORF75','PTOV1'],
        ['C13ORF45','LMO7'],
        ['C14ORF166','GNG2'],
        ['C14ORF166','TG'],
        ['C15ORF26','IL16'],
        ['C15ORF38-AP3S2','RAI14'],
        ['C15ORF38-AP3S2','YPEL5'],
        ['C15ORF48','SPATA5L1'],
        ['C15ORF52','PGPEP1'],
        ['C15ORF57','CBX3'],
        ['C16ORF45','CBX3'],
        ['C16ORF46','GCSH'],
        ['C16ORF52','VWA3A'],
        ['C16ORF62','CCP110'],
        ['C16ORF62','PNLIP'],
        ['C16ORF72','GTPBP10'],
        ['C16ORF72','HOOK2'],
        ['C16ORF72','LYZ'],
        ['C16ORF87','COL14A1'],
        ['C16ORF89','SFTPB'],
        ['C17ORF104','DBF4B'],
        ['C17ORF108','NOS2'],
        ['C17ORF51','FLJ36000'],
        ['C17ORF74','TMEM102'],
        ['C17ORF99','SYNGR2'],
        ['C18ORF8','CAPZA2'],
        ['C19MC','CD44'],
        ['C19MC','CTSB'],
        ['C19MC','ETS2'],
        ['C19MC','MDM4'],
        ['C19MC','METTL7A'],
        ['C19MC','RPLP1'],
        ['C19MC','RPS19'],
        ['C19MC','SYT11'],
        ['C19MC','TIMM50'],
        ['C19ORF10','FLNA'],
        ['C19ORF18','ZNF606'],
        ['C19ORF24','CIRBP'],
        ['C19ORF33','SPINT2'],
        ['C1D','HZGJ'],
        ['C1D','WDR92'],
        ['C1GALT1C1','CSNK1D'],
        ['C1GALT1C1','NDST4'],
        ['C1ORF105','DNM3'],
        ['C1ORF146','KIAA1107'],
        ['C1ORF167','TG'],
        ['C1ORF194','UQCR10'],
        ['C1ORF226','NOS1AP'],
        ['C1ORF43','KRT1'],
        ['C1ORF43','MB'],
        ['C1ORF43','PNLIP'],
        ['C1ORF61','IFNAR2'],
        ['C1ORF74','IRF6'],
        ['C1QC','GML'],
        ['C1QC','SFTPA1'],
        ['C1QTNF3-AMACR','SLC45A2'],
        ['C1QTNF9B','MIPEP'],
        ['C1R','HP'],
        ['C1R','IGH@'],
        ['C1R','SERPINA1'],
        ['C1R','SERPING1'],
        ['C1RL-AS1','CLSTN3'],
        ['C1S','HLA-E'],
        ['C1S','IGFBP5'],
        ['C1S','KRT10'],
        ['C1S','ORM2'],
        ['C20ORF197','MIR646HG'],
        ['C20ORF202','PSMF1'],
        ['C20ORF203','COMMD7'],
        ['C20ORF62','LINC01430'],
        ['C21ORF58','PIK3CG'],
        ['C21ORF62-AS1','HOMER1'],
        ['C22ORF46','MEI1'],
        ['C2CD3','UCP3'],
        ['C2CD5','ST8SIA1'],
        ['C2ORF27A','DUXAP8'],
        ['C2ORF27B','LINC02564'],
        ['C2ORF48','RRM2'],
        ['C2ORF49','RN7SL619P'],
        ['C2ORF50','PECAM1'],
        ['C2ORF72','SPATA3'],
        ['C2ORF81','HMGA1'],
        ['C2ORF81','RTKN'],
        ['C3','CLU'],
        ['C3','CREB3L3'],
        ['C3','CRP'],
        ['C3','DHCR24'],
        ['C3','EMP2'],
        ['C3','FGA'],
        ['C3','FGB'],
        ['C3','FGG'],
        ['C3','GML'],
        ['C3','HLA@'],
        ['C3','HP'],
        ['C3','HRG'],
        ['C3','ITIH1'],
        ['C3','PTMS'],
        ['C3','RBP4'],
        ['C3','SERPINA1'],
        ['C3','SERPING1'],
        ['C3','TMEM176B'],
        ['C3','TNFSF14'],
        ['C3','TXNL4B'],
        ['C3','ZFP36L1'],
        ['C3ORF22','TXNRD3'],
        ['C3ORF67','FAM3D'],
        ['C4BPB','PFKFB2'],
        ['C4ORF3','CD36'],
        ['C4ORF3','FABP2'],
        ['C4ORF3','LYZ'],
        ['C4ORF3','MYH11'],
        ['C4ORF3','TRA@'],
        ['C4ORF48','EIF5A'],
        ['C4ORF51','CALB1'],
        ['C5','LRG1'],
        ['C5','SERPINA1'],
        ['C5AR1','INAFM1'],
        ['C5AR1','LYZ'],
        ['C5AR1','NDUFA7'],
        ['C5AR1','SEPT9'],
        ['C5ORF24','TG'],
        ['C5ORF34','EEF1A1'],
        ['C5ORF4','CCDC106'],
        ['C5ORF46','SPINK1'],
        ['C5ORF58','LCP2'],
        ['C5ORF64','RRM1'],
        ['C5ORF66','KRT13'],
        ['C6ORF106','SPDEF'],
        ['C6ORF141','CLEC4M'],
        ['C6ORF141','KIAA1671'],
        ['C6ORF183','CCDC162P'],
        ['C6ORF183','ZNF528-AS1'],
        ['C6ORF25','CCL5'],
        ['C6ORF25','F2R'],
        ['C6ORF25','LIMS1'],
        ['C6ORF25','PLEKHB2'],
        ['C6ORF25','SNAP29'],
        ['C6ORF25','STAT3'],
        ['C6ORF25','TRIM58'],
        ['C6ORF48','HSPA1A'],
        ['C6ORF58','LYZ'],
        ['C6ORF58','PGC'],
        ['C6ORF62','LYZ'],
        ['C6ORF89','LYZ'],
        ['C7','CHGB'],
        ['C7','COL1A2'],
        ['C7','HNRNPU'],
        ['C7','IGF2'],
        ['C7','IGH@'],
        ['C7','JAK1'],
        ['C7','MYH9'],
        ['C7','PNLIP'],
        ['C7','RPS3'],
        ['C7','RPS4X'],
        ['C7','STAR'],
        ['C7','TBC1D2B'],
        ['C7','TIMP2'],
        ['C7','TRA@'],
        ['C7ORF49','WDR91'],
        ['C7ORF50','COX19'],
        ['C7ORF50','PNLIP'],
        ['C7ORF55','TXNDC17'],
        ['C7ORF55-LUC7L2','TXNDC17'],
        ['C7ORF60','HRAT17'],
        ['C7ORF73','NSUN5'],
        ['C8ORF89','RPL7'],
        ['C9ORF24','KIF24'],
        ['C9ORF3','IGF2'],
        ['C9ORF3','MIR205HG'],
        ['C9ORF3','PIGR'],
        ['C9ORF43','RGS3'],
        ['C9ORF50','PIGR'],
        ['C9ORF72','MOB3B'],
        ['C9ORF78','HBB'],
        ['C9ORF78','LYZ'],
        ['CA1','MGEA5'],
        ['CA12','FLG-AS1'],
        ['CA2','MEIS1'],
        ['CA5B','CA5BP1'],
        ['CA5B','NT5C'],
        ['CA6','JCHAIN'],
        ['CA6','LYZ'],
        ['CA6','MUC7'],
        ['CA6','PRB2'],
        ['CA6','PRB3'],
        ['CA6','PRB4'],
        ['CA6','PRH1'],
        ['CA6','PRH1-PRR4'],
        ['CA6','PRR27'],
        ['CA6','SMR3B'],
        ['CA6','STATH'],
        ['CA6','ZG16B'],
        ['CABIN1','CBFA2T2'],
        ['CABP7','NF2'],
        ['CACNA1A','LRRFIP1'],
        ['CACNA1C','LINC02371'],
        ['CACNA1H','TG'],
        ['CACNA2D3','RPS15'],
        ['CACNB1','MRC2'],
        ['CACNG1','CACNG4'],
        ['CACUL1','SND1'],
        ['CADM1','ST3GAL2'],
        ['CADM3-AS1','IGF2'],
        ['CADM4','PLAUR'],
        ['CADM4','ZNF428'],
        ['CADPS2','RNF148'],
        ['CALCB','INSC'],
        ['CALCOCO1','KRT13'],
        ['CALCRL','NBEAL1'],
        ['CALD1','CTSB'],
        ['CALD1','IGF2'],
        ['CALD1','KRT13'],
        ['CALD1','KRT4'],
        ['CALD1','MYH11'],
        ['CALD1','PNLIP'],
        ['CALD1','SFTPC'],
        ['CALM1','CXCR4'],
        ['CALM1','EEF1A1'],
        ['CALM1','SPG7'],
        ['CALM1','YWHAE'],
        ['CALM2','PIGF'],
        ['CALM2','PNLIP'],
        ['CALM2','STATH'],
        ['CALM3','CAMK2N1'],
        ['CALM3','GMPR2'],
        ['CALM3','MYL4'],
        ['CALM3','NRGN'],
        ['CALM3','TG'],
        ['CALM3','VAMP2'],
        ['CALML4','CLN6'],
        ['CALML5','SMR3B'],
        ['CALN1','CD24'],
        ['CALR','COL1A1'],
        ['CALR','EEF2'],
        ['CALR','FGA'],
        ['CALR','H19'],
        ['CALR','KRBOX1'],
        ['CALR','KRT13'],
        ['CALR','MDN1'],
        ['CALR','PLIN1'],
        ['CALR','SERPINA1'],
        ['CALR','SFTPC'],
        ['CALR','SMR3B'],
        ['CALR','UBXN11'],
        ['CALR','ZNF554'],
        ['CALU','FAM71F1'],
        ['CAMK1D','IL1RAPL1'],
        ['CAMK2A','CCNI'],
        ['CAMK2A','TRA@'],
        ['CAMK2D','GNAS'],
        ['CAMK2N1','MBP'],
        ['CAMK2N1','VSNL1'],
        ['CAMKK1','P2RX1'],
        ['CAMLG','SEC24A'],
        ['CAMTA1','CAMTA1-IT1'],
        ['CAMTA1','CRYL1'],
        ['CAMTA1','VAMP3'],
        ['CAMTA2','FBXW8'],
        ['CAND1','PTMS'],
        ['CANT1','FSCN1'],
        ['CANX','ERBB2'],
        ['CANX','HMGB3'],
        ['CANX','LYZ'],
        ['CANX','MAML1'],
        ['CANX','MUC7'],
        ['CANX','PNLIP'],
        ['CANX','VEGFA'],
        ['CAP1','LHX4'],
        ['CAP1','SERPINA1'],
        ['CAP1','SMR3B'],
        ['CAPN1','SMR3B'],
        ['CAPN1','TG'],
        ['CAPN2','IGKC'],
        ['CAPN2','IRF2BP2'],
        ['CAPN2','SFTPC'],
        ['CAPN2','TG'],
        ['CAPN3','GANC'],
        ['CAPN8','IGH@'],
        ['CAPN8','LYZ'],
        ['CAPNS1','PTMS'],
        ['CAPNS1','VEGFB'],
        ['CAPRIN1','KIF5B'],
        ['CAPZA2','SLC38A2'],
        ['CAPZB','EEF2'],
        ['CAPZB','IGF2'],
        ['CAPZB','SFTPB'],
        ['CAPZB','SMR3B'],
        ['CARD17','CASP1'],
        ['CARD8','TG'],
        ['CARD8','TOR1AIP2'],
        ['CARF','IQUB'],
        ['CARF','NBEAL1'],
        ['CARHSP1','PTRF'],
        ['CARM1P1','FZD4'],
        ['CASC15','CTRB1'],
        ['CASC15','PRSS23'],
        ['CASC3','TG'],
        ['CASC4','PTRF'],
        ['CASC4','TPM4'],
        ['CASP14','TMEM99'],
        ['CASP6','PLA2G12A'],
        ['CASP6','TRA2A'],
        ['CASP8','FAM210B'],
        ['CASP8','LONP2'],
        ['CASP9','PNLIP'],
        ['CASQ2','MYH7'],
        ['CASR','LPP'],
        ['CAST','HBB'],
        ['CAST','KRT4'],
        ['CAST','MYH11'],
        ['CASTOR2','FURIN'],
        ['CASTOR2','GTF2I'],
        ['CASTOR2','SOD2'],
        ['CASTOR3','GTF2I'],
        ['CAT','OGFOD1'],
        ['CATSPER2','PPIP5K1'],
        ['CATSPERB','POLR3G'],
        ['CAV1','CLU'],
        ['CAV1','COL6A1'],
        ['CAV2','FGFR2'],
        ['CAV2','LINC01923'],
        ['CAV2','NDUFV1'],
        ['CAV2','TG'],
        ['CAVIN1','KRT6A'],
        ['CAVIN1','MCAM'],
        ['CAVIN1','RBMS3'],
        ['CAVIN1','STAT3'],
        ['CBFA2T3','CTSS'],
        ['CBR4','DES'],
        ['CBWD2','DOCK8'],
        ['CBX3','CCDC32'],
        ['CBX3','FLG-AS1'],
        ['CBX4','IGF2'],
        ['CBX4','SMR3B'],
        ['CBX5','PTRF'],
        ['CBX5','SFTPB'],
        ['CBX5','TG'],
        ['CCAR1','SP140L'],
        ['CCAR2','PNLIP'],
        ['CCBE1','GP2'],
        ['CCDC11','MBD1'],
        ['CCDC113','CEP70'],
        ['CCDC117','Z93930.2'],
        ['CCDC142','MOGS'],
        ['CCDC144A','USP32'],
        ['CCDC144NL','NBEA'],
        ['CCDC144NL-AS1','SPECC1'],
        ['CCDC144NL-AS1','TBC1D27'],
        ['CCDC146','PMS2'],
        ['CCDC15','SLC37A2'],
        ['CCDC15-DT','HEPACAM'],
        ['CCDC152','EEF1A1'],
        ['CCDC152','HP'],
        ['CCDC152','IGF2'],
        ['CCDC152','SFTPC'],
        ['CCDC152','SMR3B'],
        ['CCDC152','TF'],
        ['CCDC152','TG'],
        ['CCDC163P','TESK2'],
        ['CCDC169','SOHLH2'],
        ['CCDC17','GPBP1L1'],
        ['CCDC170','ESR1'],
        ['CCDC175','L3HYPDH'],
        ['CCDC176','LIN52'],
        ['CCDC180','SUGT1P4-STRA6LP'],
        ['CCDC183','TMEM141'],
        ['CCDC25','PPIE'],
        ['CCDC26','IGH@'],
        ['CCDC27','SMIM1'],
        ['CCDC28A','ECT2L'],
        ['CCDC32','CEACAM8'],
        ['CCDC34','LGR4'],
        ['CCDC47','STRADA'],
        ['CCDC6','MRLN'],
        ['CCDC62','SLC66A2'],
        ['CCDC69','TPMT'],
        ['CCDC80','IGFBP5'],
        ['CCDC80','LINC01279'],
        ['CCDC80','MYH9'],
        ['CCDC82','JRKL-AS1'],
        ['CCDC82','NCKAP1L'],
        ['CCDC83','SLC25A1P1'],
        ['CCDC85B','EVA1B'],
        ['CCDC88C','HBB'],
        ['CCDC9B','STRN3'],
        ['CCIN','GLIPR2'],
        ['CCL28','LYZ'],
        ['CCL28','ORAI2'],
        ['CCL28','PRH1'],
        ['CCL28','PRH1-PRR4'],
        ['CCL28','SMR3B'],
        ['CCL5','LYZ'],
        ['CCL5','TMED4'],
        ['CCM2L','HCK'],
        ['CCNB1IP1','RPPH1'],
        ['CCND1','COL1A1'],
        ['CCND1','CXCL12'],
        ['CCND1','CYR61'],
        ['CCND1','TG'],
        ['CCND3','F11R'],
        ['CCND3','FCAR'],
        ['CCND3','GRB2'],
        ['CCND3','PIGR'],
        ['CCND3','PSMD6-AS1'],
        ['CCND3','SIGLEC6'],
        ['CCND3','UGDH'],
        ['CCND3','ZNF273'],
        ['CCNG2','LRCH4'],
        ['CCNG2','OAZ1'],
        ['CCNG2','S100A9'],
        ['CCNG2','VAV1'],
        ['CCNI','CD74'],
        ['CCNI','FLNA'],
        ['CCNI','SFTPC'],
        ['CCNL1','SMG1'],
        ['CCNL1','TOPBP1'],
        ['CCNO','DHX29'],
        ['CCNT2-AS1','TMEM163'],
        ['CCNY','CREM'],
        ['CCP110','TMC5'],
        ['CCPG1','PIGBOS1'],
        ['CCPG1','PNLIP'],
        ['CCPG1','RAB27A'],
        ['CCR10','EZH1'],
        ['CCSER1','FUBP1'],
        ['CCT3','PRB3'],
        ['CCT4','FAM161A'],
        ['CCT6A','CCT6P1'],
        ['CCT6A','VKORC1L1'],
        ['CCT6P1','VKORC1L1'],
        ['CD101','PTGFRN'],
        ['CD163','DA750114'],
        ['CD164','WWOX'],
        ['CD22','MAG'],
        ['CD22','PLAT'],
        ['CD22','U62631.5'],
        ['CD24','CEBPD'],
        ['CD24','CNTN4'],
        ['CD24','COL6A2'],
        ['CD24','CTSB'],
        ['CD24','ERBB2'],
        ['CD24','ERCC1'],
        ['CD24','ESPL1'],
        ['CD24','FLNA'],
        ['CD24','FOXP1'],
        ['CD24','GLTP'],
        ['CD24','IGH@'],
        ['CD24','IL3RA'],
        ['CD24','IYD'],
        ['CD24','KRT13'],
        ['CD24','KRT4'],
        ['CD24','MACROD2'],
        ['CD24','MUC7'],
        ['CD24','PCYOX1'],
        ['CD24','PRKCA'],
        ['CD24','PVRL2'],
        ['CD24','RACK1'],
        ['CD24','SAP30L-AS1'],
        ['CD24','SMIM14'],
        ['CD24','TG'],
        ['CD24','ZNF773'],
        ['CD248','COL1A1'],
        ['CD248','FN1'],
        ['CD248','SPARC'],
        ['CD248','VIM'],
        ['CD27-AS1','VAMP1'],
        ['CD2AP','DBT'],
        ['CD2AP','FOXP1'],
        ['CD2AP','GTF3C2'],
        ['CD302','TRIP12'],
        ['CD34','NPPA'],
        ['CD36','CDK6'],
        ['CD36','EHD2'],
        ['CD36','IGF2'],
        ['CD36','MB'],
        ['CD44','CTSS'],
        ['CD44','FLG-AS1'],
        ['CD44','HTN1'],
        ['CD44','IGKC'],
        ['CD44','PRB1'],
        ['CD44','PRH1'],
        ['CD44','PRH1-PRR4'],
        ['CD44','PRH2'],
        ['CD44','PRR4'],
        ['CD44','SAR1B'],
        ['CD44','SMR3B'],
        ['CD44','TG'],
        ['CD44','ZEB2'],
        ['CD46','KRT4'],
        ['CD46','SMR3B'],
        ['CD55','SMR3B'],
        ['CD59','HTN3'],
        ['CD59','MUC7'],
        ['CD59','MYH11'],
        ['CD59','TIMP3'],
        ['CD5L','EOGT'],
        ['CD63','CPB1'],
        ['CD63','EEF1A1'],
        ['CD63','HBB'],
        ['CD63','HTN3'],
        ['CD63','PNLIP'],
        ['CD63','PRSS1'],
        ['CD63','SMR3B'],
        ['CD63','STATH'],
        ['CD63','TAGLN'],
        ['CD63','TMBIM6'],
        ['CD68','MPDU1'],
        ['CD68','SCUBE3'],
        ['CD68','SFTPC'],
        ['CD74','FLNA'],
        ['CD74','FOS'],
        ['CD74','GNAS'],
        ['CD74','HLA@'],
        ['CD74','IGH@'],
        ['CD74','IGK@'],
        ['CD74','LAMP1'],
        ['CD74','LIPF'],
        ['CD74','LYZ'],
        ['CD74','PIGR'],
        ['CD74','PRH1'],
        ['CD74','PRH1-PRR4'],
        ['CD74','PRH2'],
        ['CD74','PSCA'],
        ['CD74','SFTPA1'],
        ['CD74','SFTPA2'],
        ['CD74','SFTPB'],
        ['CD74','SMR3B'],
        ['CD74','SOD2'],
        ['CD74','STATH'],
        ['CD74','SYNRG'],
        ['CD74','TG'],
        ['CD74','TNS1'],
        ['CD79A','RPS19'],
        ['CD81','KIAA2013'],
        ['CD81','PRSS2'],
        ['CD81','TSSC4'],
        ['CD81','YKT6'],
        ['CD82','COLEC12'],
        ['CD83','JARID2'],
        ['CD84','SLC26A2'],
        ['CD8B','CD8B2'],
        ['CD9','CST4'],
        ['CD9','NDE1'],
        ['CD93','IGF2'],
        ['CD93','SYN3'],
        ['CD93','SYNPO'],
        ['CD99','KANK2'],
        ['CD99','MYH9'],
        ['CDA','PINK1'],
        ['CDADC1','GTF2IP23'],
        ['CDADC1','LYZ'],
        ['CDC123','SEC61A2'],
        ['CDC16','UPF3A'],
        ['CDC34','GZMM'],
        ['CDC34','PTMA'],
        ['CDC37','DDX42'],
        ['CDC40','LYZ'],
        ['CDC40','WDR78'],
        ['CDC42','S100A9'],
        ['CDC42','SLC25A37'],
        ['CDC42','TRB@'],
        ['CDC42','ZEB2'],
        ['CDC42BPA','FGD5-AS1'],
        ['CDC42BPG','EHD1'],
        ['CDC42EP1','TMEM200A'],
        ['CDC42EP2','POLA2'],
        ['CDC42EP3','LINC00211'],
        ['CDC42EP4','CPSF4L'],
        ['CDCA7L','SFTPB'],
        ['CDH12','LINC02197'],
        ['CDH17','IGHA1'],
        ['CDH2','FN1'],
        ['CDH23','CXCL16'],
        ['CDH24','GPC1'],
        ['CDH5','TG'],
        ['CDH8','SYNPO'],
        ['CDHR1','FLG-AS1'],
        ['CDHR4','IP6K1'],
        ['CDHR5','GAPDH'],
        ['CDIPT','RTN4'],
        ['CDK10','SPATA33'],
        ['CDK11B','IGHG1'],
        ['CDK11B','SLC35E2'],
        ['CDK11B','SLC35E2A'],
        ['CDK12','HBB'],
        ['CDK12','MGEA5'],
        ['CDK14','CLDN12'],
        ['CDK16','TXNDC5'],
        ['CDK19','REV3L'],
        ['CDK2','RAB5B'],
        ['CDK2AP1','MPHOSPH9'],
        ['CDK2AP2','TG'],
        ['CDK5RAP2','MEGF9'],
        ['CDK5RAP3','PNPO'],
        ['CDK6','HTN3'],
        ['CDK6','PRELP'],
        ['CDK6','RPL37'],
        ['CDK6','STATH'],
        ['CDKAL1','E2F3'],
        ['CDKL4','MAP4K3'],
        ['CDKN1A','MYH9'],
        ['CDKN1A','RAB44'],
        ['CDKN1C','FN1'],
        ['CDKN2B-AS1','MTAP'],
        ['CDKN2B-AS1','UBA52'],
        ['CDR1','LINC00632'],
        ['CDR1-AS','LINC00632'],
        ['CDR2','RRN3'],
        ['CDR2','RRN3P2'],
        ['CDR2L','HID1-AS1'],
        ['CDR2L','ICT1'],
        ['CDRT15L2','TBC1D26'],
        ['CDRT4','FAM18B2'],
        ['CDS2','PCBP2'],
        ['CDT1','HBB'],
        ['CDV3','CDV3P1'],
        ['CEACAM1','MUC2'],
        ['CEACAM1','PRH1'],
        ['CEACAM1','PRH1-PRR4'],
        ['CEACAM20','CEACAM22P'],
        ['CEACAM20','ZNF180'],
        ['CEACAM5','ENAM'],
        ['CEACAM5','IGH@'],
        ['CEACAM5','IGK@'],
        ['CEACAM5','IGKC'],
        ['CEACAM5','OPA3'],
        ['CEACAM5','PIGR'],
        ['CEBPB','ITGB2'],
        ['CECR2','TERF2IP'],
        ['CECR7','IL17RA'],
        ['CEL','CELA2A'],
        ['CEL','CELA2B'],
        ['CEL','CELA3A'],
        ['CEL','CELA3B'],
        ['CEL','CELP'],
        ['CEL','CLPS'],
        ['CEL','CPA1'],
        ['CEL','CPA2'],
        ['CEL','CPB1'],
        ['CEL','CTRB1'],
        ['CEL','CTRB2'],
        ['CEL','CTRC'],
        ['CEL','CUZD1'],
        ['CEL','GP2'],
        ['CEL','HDLBP'],
        ['CEL','LHFPL5'],
        ['CEL','MT1A'],
        ['CEL','PLA2G1B'],
        ['CEL','PNLIP'],
        ['CEL','PNLIPRP1'],
        ['CEL','PNLIPRP2'],
        ['CEL','PRSS1'],
        ['CEL','PRSS2'],
        ['CEL','PRSS3'],
        ['CEL','SLC7A2'],
        ['CEL','SPINK1'],
        ['CEL','SYCN'],
        ['CEL','TMEM59'],
        ['CEL','TRB@'],
        ['CELA2A','CFTR'],
        ['CELA2A','CLPS'],
        ['CELA2A','CPA1'],
        ['CELA2A','CPA2'],
        ['CELA2A','CPB1'],
        ['CELA2A','CTRB1'],
        ['CELA2A','CTRB2'],
        ['CELA2A','CTRC'],
        ['CELA2A','CTRL'],
        ['CELA2A','GP2'],
        ['CELA2A','LHFPL5'],
        ['CELA2A','PNLIP'],
        ['CELA2A','PRSS1'],
        ['CELA2A','PRSS2'],
        ['CELA2A','PRSS3'],
        ['CELA2A','RRBP1'],
        ['CELA2A','SYCN'],
        ['CELA2A','TRB@'],
        ['CELA2A','UBE2R2-AS1'],
        ['CELA2B','CLPS'],
        ['CELA2B','CPA1'],
        ['CELA2B','CPA2'],
        ['CELA2B','CPB1'],
        ['CELA2B','CTRB1'],
        ['CELA2B','CTRB2'],
        ['CELA2B','CTRC'],
        ['CELA2B','CTRL'],
        ['CELA2B','GP2'],
        ['CELA2B','PDIA2'],
        ['CELA2B','PNLIP'],
        ['CELA2B','PRSS1'],
        ['CELA2B','PRSS2'],
        ['CELA2B','SYCN'],
        ['CELA2B','TRB@'],
        ['CELA3A','CLPS'],
        ['CELA3A','CPA1'],
        ['CELA3A','CPA2'],
        ['CELA3A','CTRB1'],
        ['CELA3A','CTRB2'],
        ['CELA3A','CTRC'],
        ['CELA3A','FLNA'],
        ['CELA3A','GP2'],
        ['CELA3A','KLK1'],
        ['CELA3A','LHFPL5'],
        ['CELA3A','LINC00339'],
        ['CELA3A','NEAT1'],
        ['CELA3A','PNLIP'],
        ['CELA3A','PRSS1'],
        ['CELA3A','PRSS2'],
        ['CELA3A','PRSS3'],
        ['CELA3A','TRB@'],
        ['CELA3A','UBE2R2-AS1'],
        ['CELA3B','CLPS'],
        ['CELA3B','CPA1'],
        ['CELA3B','CTRB1'],
        ['CELA3B','CTRB2'],
        ['CELA3B','CTRC'],
        ['CELA3B','GP2'],
        ['CELA3B','LHFPL5'],
        ['CELA3B','PNLIP'],
        ['CELA3B','PRSS1'],
        ['CELA3B','PRSS2'],
        ['CELA3B','PRSS3'],
        ['CELA3B','TPT1'],
        ['CELA3B','TRB@'],
        ['CELF4','SOX4'],
        ['CELP','CPA1'],
        ['CELP','CPB1'],
        ['CELP','CTRC'],
        ['CELP','PLA2G1B'],
        ['CELP','PNLIP'],
        ['CELP','PRSS1'],
        ['CELP','PRSS3'],
        ['CELP','TRB@'],
        ['CELSR1','SMR3B'],
        ['CELSR3','NCKIPSD'],
        ['CEMIP2','NMRK1'],
        ['CENPB','GRWD1'],
        ['CENPB','VEGFB'],
        ['CENPC','LEPROT'],
        ['CENPC','NR2F2-AS1'],
        ['CENPM','TNFRSF13C'],
        ['CENPT','RPL14'],
        ['CENPU','KIAA0232'],
        ['CEP104','GRK4'],
        ['CEP104','IL6ST'],
        ['CEP152','FBN1'],
        ['CEP170','RAD51B'],
        ['CEP192','SEH1L'],
        ['CEP250','ERBB2'],
        ['CEP63','EPHB1'],
        ['CEP76','GP2'],
        ['CEP78','PSAT1'],
        ['CEP97','PDCL3'],
        ['CERK','LCN10'],
        ['CERK','SLC25A51'],
        ['CERS2','FTH1'],
        ['CERS6','SYTL2'],
        ['CES1','CES1P1'],
        ['CES2','H6PD'],
        ['CES2','PIGR'],
        ['CES4A','TRA@'],
        ['CFAP157','STXBP1'],
        ['CFAP20','CSNK2A2'],
        ['CFAP221','TMEM177'],
        ['CFAP300','YAP1'],
        ['CFAP54','NEDD1'],
        ['CFAP58','ITPRIP-AS1'],
        ['CFB','TXNL4B'],
        ['CFD','ELANE'],
        ['CFD','ICA1'],
        ['CFH','CFHR1'],
        ['CFHR2','CFHR5'],
        ['CFI','LRG1'],
        ['CFI','SULT2A1'],
        ['CFL1','COL1A1'],
        ['CFL1','EFEMP2'],
        ['CFL1','HBB'],
        ['CFL1','LDHA'],
        ['CFL1','LYZ'],
        ['CFL1','MKNK2'],
        ['CFL1','MUC7'],
        ['CFL1','PIGR'],
        ['CFL1','PTMA'],
        ['CFL1','SF1'],
        ['CFL1','TG'],
        ['CFL1','TPM2'],
        ['CFL1','TRB@'],
        ['CFLAR','ERCC4'],
        ['CFLAR','METTL7A'],
        ['CFLAR','PKP2'],
        ['CFLAR','PTH2R'],
        ['CFLAR','PTMA'],
        ['CFTR','HTN3'],
        ['CGNL1','PRSS1'],
        ['CGNL1','PRSS2'],
        ['CGNL1','SFTPA1'],
        ['CH17-140K24.2','TAF7'],
        ['CH17-164J11.1','IGK@'],
        ['CH17-212P11.4','IGH@'],
        ['CH17-212P11.5','IGH@'],
        ['CH17-212P11.5','IGHG1'],
        ['CH17-224D4.1','IGH@'],
        ['CH17-224D4.2','IGH@'],
        ['CH17-264L24.1','IGK@'],
        ['CH17-53B9.2','NCOR1'],
        ['CHCHD10','VPREB3'],
        ['CHCHD2','PHKG1'],
        ['CHCHD2','TCEA3'],
        ['CHCHD2P6','DDI2'],
        ['CHCHD3','SRSF2'],
        ['CHD1L','GAMT'],
        ['CHD2','COL1A1'],
        ['CHD2','TRIM52'],
        ['CHD4','GTF2I'],
        ['CHD4','LYZ'],
        ['CHD4','NOP2'],
        ['CHD8','SMR3B'],
        ['CHFR','YKT6'],
        ['CHGA','CHGB'],
        ['CHGA','NR4A1'],
        ['CHGA','PRSS2'],
        ['CHGB','CYP11B1'],
        ['CHGB','HNRNPA2B1'],
        ['CHGB','IGH@'],
        ['CHGB','MEG3'],
        ['CHI3L2','SMR3B'],
        ['CHIC2','FIP1L1'],
        ['CHIC2','ROCK2'],
        ['CHIC2','STIM2'],
        ['CHIC2','WWOX'],
        ['CHMP1A','CPSF6'],
        ['CHMP3','LYZ'],
        ['CHMP3','SMR3B'],
        ['CHODL','DIAPH2'],
        ['CHP1','FBXO27'],
        ['CHP1','LRG1'],
        ['CHP1','NAA40'],
        ['CHP1','SULT2A1'],
        ['CHPF','COL6A2'],
        ['CHPF','OBSL1'],
        ['CHPT1','LYZ'],
        ['CHRAC1','PAXIP1-AS1'],
        ['CHRDL1','CHST11'],
        ['CHRDL1','NUTM2A-AS1'],
        ['CHRM3','PRB3'],
        ['CHRNA3','CHRNB4'],
        ['CHRNA9','LINC02265'],
        ['CHST11','DA750114'],
        ['CHST11','HIF1A'],
        ['CHST11','IGF2'],
        ['CHST11','IGH@'],
        ['CHST11','NIBAN3'],
        ['CHST11','TRA@'],
        ['CHST5','TMEM231'],
        ['CHST6','TMEM231'],
        ['CHST7','ZNF674-AS1'],
        ['CHSY3','ROCK2'],
        ['CHURC1','GLMP'],
        ['CIAO1','ZNF264'],
        ['CIDEC','EMC3'],
        ['CIITA','CLEC16A'],
        ['CIITA','PIK3C2A'],
        ['CIPC','CSAD'],
        ['CIPC','SSB'],
        ['CIRBP','FAM174C'],
        ['CIRBP','MUC7'],
        ['CIRBP','PNLIP'],
        ['CIRBP','SMR3B'],
        ['CIRH1A','PHACTR4'],
        ['CISD2','UBE2D3-AS1'],
        ['CITED1','HDAC8'],
        ['CITED2','ITGA5'],
        ['CITED2','PTMS'],
        ['CITED2','RGMB'],
        ['CITF22-1A6.3','ZBED4'],
        ['CIZ1','PNLIP'],
        ['CIZ1','SMR3B'],
        ['CKAP4','HS3ST3B1'],
        ['CKAP4','MT2A'],
        ['CKAP4','TWIST1'],
        ['CKM','CLCN6'],
        ['CKM','CTSB'],
        ['CKM','DAB2'],
        ['CKM','MYH6'],
        ['CKM','NPPA'],
        ['CKM','NPPA-AS1'],
        ['CKM','NUCB1'],
        ['CKM','SYNPO2'],
        ['CKMT2','ZCCHC9'],
        ['CKMT2-AS1','LYZ'],
        ['CLASP2','HBB'],
        ['CLCA1','PIGR'],
        ['CLCA2','RAPH1'],
        ['CLCC1','KLHDC4'],
        ['CLCC1','ZNF33A'],
        ['CLCF1','POLD4'],
        ['CLCN3','EGF'],
        ['CLCN3','LIPG'],
        ['CLCN3','PRSS1'],
        ['CLCN3','RGS18'],
        ['CLCN3','TG'],
        ['CLCN3P1','TTC39B'],
        ['CLCN6','CLEC3B'],
        ['CLCN6','CLIC4'],
        ['CLCN6','DES'],
        ['CLCN6','FLOT2'],
        ['CLCN6','GABARAP'],
        ['CLCN6','GABARAPL1'],
        ['CLCN6','HNRNPU'],
        ['CLCN6','ITGA7'],
        ['CLCN6','LRRFIP1'],
        ['CLCN6','MB'],
        ['CLCN6','MFGE8'],
        ['CLCN6','MORF4L1'],
        ['CLCN6','MSN'],
        ['CLCN6','MYH6'],
        ['CLCN6','MYH7'],
        ['CLCN6','NDUFV1'],
        ['CLCN6','NFE2L1'],
        ['CLCN6','PSAP'],
        ['CLCN6','S100A1'],
        ['CLCN6','SPARC'],
        ['CLCN6','SRRM2'],
        ['CLCN6','TGOLN2'],
        ['CLCN6','TNNC1'],
        ['CLCN6','TNNI3'],
        ['CLCN6','TNNT2'],
        ['CLCN6','TPM1'],
        ['CLCN6','USB1'],
        ['CLCN7','TG'],
        ['CLDN11','SERPINE1'],
        ['CLDN18','ICAM1'],
        ['CLDN18','LYZ'],
        ['CLDN18','PGC'],
        ['CLDN4','SMR3B'],
        ['CLEC11A','S100A9'],
        ['CLEC12A','CLEC12B'],
        ['CLEC16A','EMP2'],
        ['CLEC2B','ZNF708'],
        ['CLEC2D','IGH@'],
        ['CLEC2D','USP24'],
        ['CLEC3B','NPPA-AS1'],
        ['CLEC4A','NECAP1'],
        ['CLEC4F','FIGLA'],
        ['CLEC6A','PIGR'],
        ['CLEC6A','UTP23'],
        ['CLEC7A','ERC1'],
        ['CLHC1','LSMEM1'],
        ['CLIC1','HBB'],
        ['CLIC2','F8'],
        ['CLIC6','MIR100HG'],
        ['CLIP1','IGH@'],
        ['CLIP1','VHL'],
        ['CLIP1','VPS33A'],
        ['CLIP1','ZCCHC8'],
        ['CLK1','CXCR4'],
        ['CLK1','PPIL3'],
        ['CLK3','FOXN3'],
        ['CLK3','H2AFZP5'],
        ['CLK3','HEATR4'],
        ['CLK3','MPO'],
        ['CLMN','TRB@'],
        ['CLMP','FOXO4'],
        ['CLMP','HSPA8'],
        ['CLNK','ZNF518B'],
        ['CLPS','CPA1'],
        ['CLPS','CPA2'],
        ['CLPS','CPB1'],
        ['CLPS','CTRB1'],
        ['CLPS','CTRB2'],
        ['CLPS','CTRC'],
        ['CLPS','CUZD1'],
        ['CLPS','GP2'],
        ['CLPS','PLA2G1B'],
        ['CLPS','PNLIP'],
        ['CLPS','PNLIPRP2'],
        ['CLPS','PRSS1'],
        ['CLPS','PRSS2'],
        ['CLPS','PRSS3'],
        ['CLPS','TPM3'],
        ['CLPS','TRB@'],
        ['CLPS','UBE2R2-AS1'],
        ['CLSTN1','CTNNBIP1'],
        ['CLSTN1','EIF3B'],
        ['CLSTN1','IGFBP5'],
        ['CLSTN1','KRT10'],
        ['CLSTN1','MARCKS'],
        ['CLSTN1','PRKCDBP'],
        ['CLSTN1','PRSS1'],
        ['CLSTN1','TG'],
        ['CLSTN1','TRB@'],
        ['CLTB','KRT4'],
        ['CLTC','PPP1R12B'],
        ['CLTC','PTRF'],
        ['CLTC','SFTPC'],
        ['CLTC','VMP1'],
        ['CLTCL1','HIRA'],
        ['CLU','CRYAB'],
        ['CLU','CTRB2'],
        ['CLU','CYP4A11'],
        ['CLU','EEF1A1'],
        ['CLU','EIF2B5'],
        ['CLU','GAPDH'],
        ['CLU','GFAP'],
        ['CLU','GRINA'],
        ['CLU','NRGN'],
        ['CLU','ORM1'],
        ['CLU','PLIN1'],
        ['CLU','POLR2A'],
        ['CLU','PRSS2'],
        ['CLU','PTGDS'],
        ['CLU','RMI2'],
        ['CLU','SERPINA1'],
        ['CLU','SPARCL1'],
        ['CLU','TRB@'],
        ['CLU','TRBC2'],
        ['CLU','TRBV25-1'],
        ['CLU','TUBB4A'],
        ['CLUHP3','ZNF720'],
        ['CLUL1','DCUN1D2'],
        ['CLVS1','KRT78'],
        ['CLYBL','PIGR'],
        ['CMAHP','FAM65B'],
        ['CMBL','CNN2'],
        ['CMBL','LIG3'],
        ['CMBL','WWOX'],
        ['CMIP','LITAF'],
        ['CMIP','PNLIP'],
        ['CMSS1','COL8A1'],
        ['CMSS1','TBC1D23'],
        ['CMTM7','CMTM8'],
        ['CMYA5','TRDN'],
        ['CNBD2','PHF20'],
        ['CNDP2','LYZ'],
        ['CNDP2','TG'],
        ['CNGA3','VWA3B'],
        ['CNKSR3','IGSF6'],
        ['CNKSR3','IPCEF1'],
        ['CNN2','ECE1'],
        ['CNN2','FBLIM1'],
        ['CNN2','FURIN'],
        ['CNN2','GALNT6'],
        ['CNN2','KIAA1191'],
        ['CNN2','LEPREL4'],
        ['CNN2','PPP1R12C'],
        ['CNN2','PXDC1'],
        ['CNN2','RBM3'],
        ['CNN2','RFC2'],
        ['CNN2','RNF126'],
        ['CNN2','S100A9'],
        ['CNN2','SBNO2'],
        ['CNN2','SC5D'],
        ['CNN2','SFTPB'],
        ['CNN2','SLC25A37'],
        ['CNN3','PFDN5'],
        ['CNN3','PPP1R12B'],
        ['CNN3','TG'],
        ['CNNM2','LYZ'],
        ['CNNM3','CNNM4'],
        ['CNOT1','IGF2'],
        ['CNOT1','PNLIP'],
        ['CNOT10','HBB'],
        ['CNOT2','LINC02821'],
        ['CNOT3','SMR3B'],
        ['CNOT4','SPTLC2'],
        ['CNPY2','PARP3'],
        ['CNRIP1','HZGJ'],
        ['CNRIP1','PPP3R1'],
        ['CNRIP1','WDR92'],
        ['CNST','MOB1A'],
        ['CNTN4','TTLL3'],
        ['CNTNAP2','RN7SL72P'],
        ['CNTROB','PET100'],
        ['COA1','EOGT'],
        ['COASY','HSD17B1'],
        ['COBL','MUC7'],
        ['COCH','GP2'],
        ['COG5','CPD'],
        ['COG5','TRB@'],
        ['COL12A1','COL1A1'],
        ['COL14A1','DEPTOR'],
        ['COL14A1','IGF2'],
        ['COL14A1','IGFBP5'],
        ['COL14A1','TG'],
        ['COL15A1','IGF2'],
        ['COL15A1','TG'],
        ['COL16A1','COL1A1'],
        ['COL16A1','DES'],
        ['COL16A1','EMILIN1'],
        ['COL17A1','FLG2'],
        ['COL17A1','KRT1'],
        ['COL18A1','COL3A1'],
        ['COL18A1','IGF2'],
        ['COL18A1','PRSS1'],
        ['COL18A1','SPARC'],
        ['COL18A1','TRB@'],
        ['COL1A1','COL6A1'],
        ['COL1A1','COL6A2'],
        ['COL1A1','COL6A3'],
        ['COL1A1','CPSF6'],
        ['COL1A1','CTSB'],
        ['COL1A1','DLGAP4'],
        ['COL1A1','EFEMP1'],
        ['COL1A1','EMILIN1'],
        ['COL1A1','FAM129B'],
        ['COL1A1','FTL'],
        ['COL1A1','GAPDH'],
        ['COL1A1','GPC1'],
        ['COL1A1','IGK@'],
        ['COL1A1','KLHL22'],
        ['COL1A1','KMT2D'],
        ['COL1A1','KRT18'],
        ['COL1A1','LAMB2'],
        ['COL1A1','LEPRE1'],
        ['COL1A1','LGALS1'],
        ['COL1A1','MAP4K4'],
        ['COL1A1','MCOLN3'],
        ['COL1A1','MDH2'],
        ['COL1A1','MMP14'],
        ['COL1A1','MYH11'],
        ['COL1A1','MYL9'],
        ['COL1A1','MYO1C'],
        ['COL1A1','ORAOV1'],
        ['COL1A1','P4HB'],
        ['COL1A1','PCSK7'],
        ['COL1A1','PIGR'],
        ['COL1A1','PKM'],
        ['COL1A1','PLEC'],
        ['COL1A1','PLOD1'],
        ['COL1A1','PRRC2B'],
        ['COL1A1','PTGDS'],
        ['COL1A1','PTMS'],
        ['COL1A1','RAB7A'],
        ['COL1A1','SFTPB'],
        ['COL1A1','SH3BGRL3'],
        ['COL1A1','SHC1'],
        ['COL1A1','SMTN'],
        ['COL1A1','SPARC'],
        ['COL1A1','STIM1'],
        ['COL1A1','SYMPK'],
        ['COL1A1','SYN3'],
        ['COL1A1','TCOF1'],
        ['COL1A1','TGFBI'],
        ['COL1A1','THOP1'],
        ['COL1A1','TLN1'],
        ['COL1A1','TRAK1'],
        ['COL1A1','TRIP12'],
        ['COL1A1','TSPO'],
        ['COL1A1','UNC45A'],
        ['COL1A1','VCL'],
        ['COL1A1','VIM-AS1'],
        ['COL1A1','XXBAC-BPG252P9.9'],
        ['COL1A2','COL6A3'],
        ['COL1A2','KRT10'],
        ['COL1A2','NKX2-2'],
        ['COL1A2','NOTCH3'],
        ['COL1A2','PLEKHM2'],
        ['COL1A2','RPL4'],
        ['COL1A2','SMR3B'],
        ['COL1A2','SPTBN1'],
        ['COL1A2','TPM4'],
        ['COL1A2','VIM-AS1'],
        ['COL23A1','TG'],
        ['COL3A1','COL4A2'],
        ['COL3A1','COL5A2'],
        ['COL3A1','COL6A2'],
        ['COL3A1','GNB2L1'],
        ['COL3A1','KRT4'],
        ['COL3A1','RPL35A'],
        ['COL3A1','SFTPA2'],
        ['COL3A1','TNNT3'],
        ['COL3A1','UBC'],
        ['COL4A1','IGF2'],
        ['COL4A1','MEG3'],
        ['COL4A1','SLC29A1'],
        ['COL4A1','SYN3'],
        ['COL4A2','CTGF'],
        ['COL4A2','H19'],
        ['COL4A2','IGF2'],
        ['COL4A2','PRH1'],
        ['COL4A2','PRH1-PRR4'],
        ['COL4A2','PRR4'],
        ['COL4A2','SPARC'],
        ['COL4A3BP','NSA2'],
        ['COL4A3BP','PNLIP'],
        ['COL4A5','COL4A6'],
        ['COL5A1','EEF2'],
        ['COL5A1','EMILIN1'],
        ['COL5A1','IGF2'],
        ['COL5A1','ITGB5'],
        ['COL5A1','LRP1'],
        ['COL5A1','MXRA8'],
        ['COL5A1','PTPRS'],
        ['COL5A1','TMEM99'],
        ['COL5A1','VIM-AS1'],
        ['COL5A3','COL6A2'],
        ['COL6A1','CSH1'],
        ['COL6A1','FN1'],
        ['COL6A1','HSPB6'],
        ['COL6A1','LMNA'],
        ['COL6A1','MAZ'],
        ['COL6A1','MRC2'],
        ['COL6A1','NID2'],
        ['COL6A1','PLXNB2'],
        ['COL6A1','SEC16A'],
        ['COL6A1','SEPT9'],
        ['COL6A1','SERPINE2'],
        ['COL6A1','SPARC'],
        ['COL6A1','VIM'],
        ['COL6A2','CTSB'],
        ['COL6A2','FBN2'],
        ['COL6A2','FLNC'],
        ['COL6A2','GNAI2'],
        ['COL6A2','HNRNPK'],
        ['COL6A2','IGF2'],
        ['COL6A2','KRT14'],
        ['COL6A2','MUC7'],
        ['COL6A2','MXRA8'],
        ['COL6A2','PRKACA'],
        ['COL6A2','SFTPB'],
        ['COL6A2','SH3BP2'],
        ['COL6A2','TAF10'],
        ['COL6A2','TLN1'],
        ['COL6A2','VIM'],
        ['COL6A2','WNT5B'],
        ['COL6A3','LAMB2'],
        ['COL6A3','PLEC'],
        ['COL6A3','PTK7'],
        ['COL6A3','RTN4'],
        ['COL6A3','THY1'],
        ['COL6A3','VIM'],
        ['COL7A1','KRT14'],
        ['COL7A1','UCN2'],
        ['COLCA1','POU2AF1'],
        ['COLEC10','SAMD12-AS1'],
        ['COLEC12','IGF2'],
        ['COLGALT1','FAM129C'],
        ['COLGALT1','PTRF'],
        ['COLQ','HACL1'],
        ['COMMD10','FBXL17'],
        ['COMMD2','ICK'],
        ['COMMD2','WWTR1'],
        ['COMMD3-BMI1','DDX5'],
        ['COMMD6','LYZ'],
        ['COMMD7','FTH1'],
        ['COPA','FGFR2'],
        ['COPA','GOLGA8R'],
        ['COPA','HTN3'],
        ['COPA','SLC16A1-AS1'],
        ['COPA','SUGP2'],
        ['COPB1','MUC7'],
        ['COPE','GDF1'],
        ['COPS3','GTF3C2-AS1'],
        ['COPS5','TCF24'],
        ['COPS7A','FN1'],
        ['COQ10B','LYZ'],
        ['COQ10B','R3HDM2'],
        ['COQ4','SLC27A4'],
        ['COQ4','SWI5'],
        ['COQ8B','NUMBL'],
        ['COQ9','POLR2C'],
        ['CORO1A','EEF1A1'],
        ['COTL1','HNRNPA0'],
        ['COTL1','SCUBE3'],
        ['COTL1','SFTPB'],
        ['COX10-AS1','GHR'],
        ['COX10-AS1','SH2D4A'],
        ['COX18','DSG3'],
        ['COX19','LINC01002'],
        ['COX19','MAFK'],
        ['COX20','TET1'],
        ['COX4I1','TG'],
        ['COX6B1','ETV2'],
        ['COX6B1','MDK'],
        ['COX6C','HMGB1'],
        ['COX6C','LYZ'],
        ['COX6C','SMR3B'],
        ['COX7A2','MINDY3'],
        ['COX7A2','TMEM30A'],
        ['COX7B','TCEANC2'],
        ['COX7B','XPR1'],
        ['COX7C','KDM5B'],
        ['COX7C','ZG16B'],
        ['COX8A','NAA40'],
        ['CPA1','CPA2'],
        ['CPA1','CPB1'],
        ['CPA1','CTRB1'],
        ['CPA1','CTRB2'],
        ['CPA1','CTRC'],
        ['CPA1','CTRL'],
        ['CPA1','DDIT4'],
        ['CPA1','DPEP1'],
        ['CPA1','EPAS1'],
        ['CPA1','ERN1'],
        ['CPA1','EZR'],
        ['CPA1','FAM162A'],
        ['CPA1','FURIN'],
        ['CPA1','GLTSCR2'],
        ['CPA1','GP2'],
        ['CPA1','GRB10'],
        ['CPA1','GTF2I'],
        ['CPA1','HUWE1'],
        ['CPA1','LAMB2'],
        ['CPA1','LARP4B'],
        ['CPA1','LHFPL5'],
        ['CPA1','MOB3A'],
        ['CPA1','MPRIP'],
        ['CPA1','PDIA3'],
        ['CPA1','PHLDA1'],
        ['CPA1','PLEC'],
        ['CPA1','PNLIP'],
        ['CPA1','PNLIPRP2'],
        ['CPA1','PRSS1'],
        ['CPA1','PRSS2'],
        ['CPA1','PRSS3'],
        ['CPA1','PTPRN2'],
        ['CPA1','RBM3'],
        ['CPA1','RBPJL'],
        ['CPA1','REG1A'],
        ['CPA1','RGL3'],
        ['CPA1','RNASE1'],
        ['CPA1','SLC3A1'],
        ['CPA1','SLC4A4'],
        ['CPA1','SPTBN1'],
        ['CPA1','TIMP3'],
        ['CPA1','TRB@'],
        ['CPA1','TSC2'],
        ['CPA1','UBE2R2-AS1'],
        ['CPA2','CPA4'],
        ['CPA2','GP2'],
        ['CPA2','LHFPL5'],
        ['CPA2','PNLIP'],
        ['CPA2','PRSS1'],
        ['CPA2','PRSS2'],
        ['CPA2','TRB@'],
        ['CPA2','UBE2R2-AS1'],
        ['CPA3','CPB1'],
        ['CPB1','CTRB1'],
        ['CPB1','CTRB2'],
        ['CPB1','CYB5A'],
        ['CPB1','DDX24'],
        ['CPB1','GOLGA8B'],
        ['CPB1','GP2'],
        ['CPB1','LHFPL5'],
        ['CPB1','MAN1A2'],
        ['CPB1','MYH9'],
        ['CPB1','PNLIP'],
        ['CPB1','PNLIPRP2'],
        ['CPB1','PPIB'],
        ['CPB1','PRSS1'],
        ['CPB1','PRSS2'],
        ['CPB1','PRSS3'],
        ['CPB1','REG1A'],
        ['CPB1','RNASE1'],
        ['CPB1','STAT6'],
        ['CPB1','TRB@'],
        ['CPB1','UBA52'],
        ['CPB1','UBE2R2-AS1'],
        ['CPD','HTN3'],
        ['CPD','STATH'],
        ['CPEB2-AS1','LINC00504'],
        ['CPED1','GBP6'],
        ['CPED1','WNT16'],
        ['CPLX2','GLUL'],
        ['CPM','LYZ'],
        ['CPM','TMED9'],
        ['CPM','USP9Y'],
        ['CPNE1','GP2'],
        ['CPQ','PNLIP'],
        ['CPS1','TF'],
        ['CPSF6','EEF1A1'],
        ['CPSF6','LYZ'],
        ['CPT1A','LYZ'],
        ['CPT1A','TG'],
        ['CPT1B','NPPA-AS1'],
        ['CR1','CR1L'],
        ['CR2','HLA-E'],
        ['CR769776.3','PARG'],
        ['CR786580.1','LINC01410'],
        ['CRBN','TG'],
        ['CRCP','LYZ'],
        ['CRCP','TPST1'],
        ['CREB3','GRB2'],
        ['CREB3L1','PRSS1'],
        ['CREB3L1','PSPC1'],
        ['CREB3L1','TRB@'],
        ['CREB3L2','IGF2'],
        ['CREB3L2','TPO'],
        ['CREBBP','EP300'],
        ['CREBBP','TRB@'],
        ['CREBRF','NSD1'],
        ['CREBZF','LYZ'],
        ['CRELD1','PRRT3-AS1'],
        ['CRELD2','DPF2'],
        ['CRHR1-IT1_','YIPF5'],
        ['CRIM1','GNAS'],
        ['CRIM1','MARCKS'],
        ['CRIM1','TG'],
        ['CRIPAK','UVSSA'],
        ['CRISP3','HTN1'],
        ['CRISP3','HTN3'],
        ['CRISP3','SMR3B'],
        ['CRISPLD2','GPNMB'],
        ['CRK','FAHD1'],
        ['CRK','YWHAE'],
        ['CRLF1','HBB'],
        ['CRLF2','CSF2RA'],
        ['CRLF2','IDS'],
        ['CRLF2','LYZ'],
        ['CRLF2','METTL7A'],
        ['CRLF2','RBM47'],
        ['CRLS1','MCM8'],
        ['CRNN','FLNA'],
        ['CRNN','HSPB1'],
        ['CRNN','KRT13'],
        ['CRNN','KRT4'],
        ['CRNN','KRT6A'],
        ['CRNN','MYH11'],
        ['CRNN','PPL'],
        ['CRNN','S100A9'],
        ['CRNN','TAGLN'],
        ['CRNN','TGM3'],
        ['CRP','CYP2E1'],
        ['CRP','FGA'],
        ['CRP','FGB'],
        ['CRP','FGG'],
        ['CRP','HP'],
        ['CRP','ORM1'],
        ['CRP','RDH11'],
        ['CRP','SERPINA1'],
        ['CRP','SERPINA3'],
        ['CRP','TXNL4B'],
        ['CRTAP','MCAM'],
        ['CRTAP','MYO1C'],
        ['CRTAP','SMAD3'],
        ['CRTAP','SP4'],
        ['CRTAP','TG'],
        ['CRTAP','TPM4'],
        ['CRTC1','FZD4'],
        ['CRTC1','KLHL26'],
        ['CRYAB','DGAT2'],
        ['CRYAB','GFAP'],
        ['CRYAB','HSPB6'],
        ['CRYAB','MBP'],
        ['CRYAB','MYL2'],
        ['CRYAB','PLIN4'],
        ['CRYAB','TNNT2'],
        ['CRYBA4','Z95115.2'],
        ['CRYBB2','Z99916.1'],
        ['CRYBG2','ZNF683'],
        ['CRYL1','TG'],
        ['CRYZL2P','SEC16B'],
        ['CS','LYZ'],
        ['CSAG1','CSAG2'],
        ['CSAG1','CSAG3'],
        ['CSAG2','CSAG3'],
        ['CSDC2','HELLPAR'],
        ['CSDE1','DES'],
        ['CSDE1','LYZ'],
        ['CSDE1','STATH'],
        ['CSF2RA','IL3RA'],
        ['CSF2RA','LYZ'],
        ['CSF2RA','PLIN3'],
        ['CSF2RA','TMEM220'],
        ['CSF2RB','IGF2'],
        ['CSF3R','LYZ'],
        ['CSF3R','MRPS15'],
        ['CSGALNACT1','CUTALP'],
        ['CSGALNACT1','GYG2'],
        ['CSH1','H19'],
        ['CSH1','IGF2'],
        ['CSH1','MEG3'],
        ['CSH1','NEAT1'],
        ['CSH1','STAT6'],
        ['CSH2','H19'],
        ['CSH2','IGF2'],
        ['CSNK1A1','LYZ'],
        ['CSNK1D','METRNL'],
        ['CSNK1D','TG'],
        ['CSNK1E','FBXO21'],
        ['CSNK1E','TPTE2'],
        ['CSNK1G2','LLFOS-48D6.2'],
        ['CSNK1G2','OAZ1'],
        ['CSNK2A2','TSHZ2'],
        ['CSPG4','SDC3'],
        ['CSPP1','MCMDC2'],
        ['CSRNP1','SFTPC'],
        ['CSRP1','MSLN'],
        ['CSRP1','NRGN'],
        ['CSRP1','SMR3B'],
        ['CST1','CST3'],
        ['CST1','IGH@'],
        ['CST1','LPO'],
        ['CST1','MUC7'],
        ['CST1','PRH1'],
        ['CST1','PRH1-PRR4'],
        ['CST1','PRH2'],
        ['CST1','PRR4'],
        ['CST1','SMR3A'],
        ['CST1','SMR3B'],
        ['CST1','STATH'],
        ['CST2','IGK@'],
        ['CST2','MUC7'],
        ['CST2','PRH1'],
        ['CST2','PRH1-PRR4'],
        ['CST2','PRH2'],
        ['CST2','PRR4'],
        ['CST2','SMR3B'],
        ['CST2','STATH'],
        ['CST3','CST4'],
        ['CST3','FN1'],
        ['CST3','PRB3'],
        ['CST3','SMR3B'],
        ['CST3','TNFAIP2'],
        ['CST4','IGLC2'],
        ['CST4','MUC7'],
        ['CST4','PRH1'],
        ['CST4','PRH1-PRR4'],
        ['CST4','PRH2'],
        ['CST4','PRR4'],
        ['CST4','SMR3B'],
        ['CST4','STATH'],
        ['CST4','TNS1'],
        ['CST5','MUC7'],
        ['CST5','PRH2'],
        ['CST5','SMR3B'],
        ['CSTA','FGD5-AS1'],
        ['CSTB','KRT5'],
        ['CSTB','UBE2L3'],
        ['CSTF3-AS1','PIGR'],
        ['CSTP1','MUC7'],
        ['CTGF','IGHG1'],
        ['CTGF','IGHM'],
        ['CTGF','IGK@'],
        ['CTGF','IGKC'],
        ['CTGF','IGKV1-8'],
        ['CTGF','IGKV3-11'],
        ['CTH','HHLA3'],
        ['CTIF','SLC25A3'],
        ['CTNNA1','ERBB2'],
        ['CTNNA1','KRT4'],
        ['CTNNA1','MATR3'],
        ['CTNNA3','KRT8'],
        ['CTNND1','LYZ'],
        ['CTNND1','MAGED1'],
        ['CTNND1','TMEM135'],
        ['CTNND1','TMX2'],
        ['CTNND2','MEX3C'],
        ['CTRB1','CTRL'],
        ['CTRB1','GATM'],
        ['CTRB1','GP2'],
        ['CTRB1','KPNB1'],
        ['CTRB1','LHFPL5'],
        ['CTRB1','MARK3'],
        ['CTRB1','P4HB'],
        ['CTRB1','PLA2G1B'],
        ['CTRB1','PNLIP'],
        ['CTRB1','PNLIPRP2'],
        ['CTRB1','PRSS1'],
        ['CTRB1','PRSS2'],
        ['CTRB1','PRSS3'],
        ['CTRB1','SYCN'],
        ['CTRB1','TMBIM6'],
        ['CTRB1','TRB@'],
        ['CTRB1','UBE2R2-AS1'],
        ['CTRB2','CTRL'],
        ['CTRB2','GNAS'],
        ['CTRB2','LHFPL5'],
        ['CTRB2','P4HB'],
        ['CTRB2','PNLIP'],
        ['CTRB2','PRSS1'],
        ['CTRB2','PRSS2'],
        ['CTRB2','PRSS3'],
        ['CTRB2','TRB@'],
        ['CTRB2','UBE2R2-AS1'],
        ['CTRC','GP2'],
        ['CTRC','LHFPL5'],
        ['CTRC','PACSIN2'],
        ['CTRC','PNLIP'],
        ['CTRC','PNLIPRP2'],
        ['CTRC','PRSS1'],
        ['CTRC','PRSS2'],
        ['CTRC','PRSS3'],
        ['CTRC','RPL30'],
        ['CTRC','TRB@'],
        ['CTRC','UBE2R2-AS1'],
        ['CTRL','PRSS1'],
        ['CTRL','PRSS2'],
        ['CTRL','TRB@'],
        ['CTSA','PLTP'],
        ['CTSA','TERF2IP'],
        ['CTSA','TJP2'],
        ['CTSB','DNAJC7'],
        ['CTSB','EEF1A1'],
        ['CTSB','EHD2'],
        ['CTSB','EML4'],
        ['CTSB','FGFR2'],
        ['CTSB','GCSH'],
        ['CTSB','GPD2'],
        ['CTSB','GTF3C1'],
        ['CTSB','HNRNPH1'],
        ['CTSB','HOOK2'],
        ['CTSB','IYD'],
        ['CTSB','LARP1'],
        ['CTSB','LINC02245'],
        ['CTSB','LOXL2'],
        ['CTSB','LTBP2'],
        ['CTSB','LYZ'],
        ['CTSB','MT1A'],
        ['CTSB','MYOZ1'],
        ['CTSB','NEAT1'],
        ['CTSB','NUCB2'],
        ['CTSB','P4HB'],
        ['CTSB','PAX5'],
        ['CTSB','PIEZO1'],
        ['CTSB','PNPLA6'],
        ['CTSB','PPP1R36'],
        ['CTSB','PTRF'],
        ['CTSB','RAP1GAP'],
        ['CTSB','SCUBE3'],
        ['CTSB','SLC35E3'],
        ['CTSB','TG'],
        ['CTSB','THBS1'],
        ['CTSB','TMED10'],
        ['CTSB','TPT1'],
        ['CTSB','TRA@'],
        ['CTSB','TRAPPC12'],
        ['CTSB','VSTM1'],
        ['CTSB','WWP2'],
        ['CTSC','RAB38'],
        ['CTSD','IFITM10'],
        ['CTSD','IGF2'],
        ['CTSD','LSP1'],
        ['CTSD','PRB1'],
        ['CTSD','PRB2'],
        ['CTSD','SFTPA1'],
        ['CTSD','SMR3B'],
        ['CTSD','ZYX'],
        ['CTSE','DPCR1'],
        ['CTSE','LYZ'],
        ['CTSE','MUC6'],
        ['CTSE','PGC'],
        ['CTSG','HOOK2'],
        ['CTSG','IGH@'],
        ['CTSG','ITGB2'],
        ['CTSG','S100A9'],
        ['CTSL','CTSL3P'],
        ['CTSL','ZC3H14'],
        ['CTSS','GPRC5A'],
        ['CTSS','HLA@'],
        ['CTSS','HORMAD1'],
        ['CTSS','IGH@'],
        ['CTSS','KAT6A'],
        ['CTSS','KIRREL3'],
        ['CTSS','LYZ'],
        ['CTSS','MAN1C1'],
        ['CTSS','MIR100HG'],
        ['CTSS','OR1M1'],
        ['CTSS','P2RX7'],
        ['CTSS','SYT14'],
        ['CTSS','TBXAS1'],
        ['CTSS','VPRBP'],
        ['CTSS','WWOX'],
        ['CTSZ','PTOV1'],
        ['CTTN','IGK@'],
        ['CTTN','RACK1'],
        ['CU633967.1','CU638689.4'],
        ['CU634019.5','CU638689.4'],
        ['CU634019.6','CU638689.4'],
        ['CUBN','RSU1'],
        ['CUBN','TMEM168'],
        ['CUEDC1','MALAT1'],
        ['CUL1','EZH2'],
        ['CUL3','SMR3B'],
        ['CUL4B','EMP2'],
        ['CUL4B','MBD3'],
        ['CUX1','HBB'],
        ['CUX1','KLF10'],
        ['CUX1','PRKRIP1'],
        ['CUZD1','GP2'],
        ['CUZD1','PNLIP'],
        ['CWC25','LSINCT5'],
        ['CWC25','LYZ'],
        ['CWC25','OSBPL2'],
        ['CWC25','PIP4K2B'],
        ['CWF19L1','PTRF'],
        ['CWF19L1','RFWD3'],
        ['CWF19L1','WWOX'],
        ['CWF19L2','TMEM41A'],
        ['CX3CL1','PRH1'],
        ['CX3CL1','PRH1-PRR4'],
        ['CX3CL1','PRH2'],
        ['CXADR','FP475955.3'],
        ['CXADR','GABPB2'],
        ['CXADR','PCLAF'],
        ['CXADR','PRKAR2A'],
        ['CXCL12','PXDC1'],
        ['CXCL14','TMEM99'],
        ['CXCL16','CYB561A3'],
        ['CXCL5','GLYR1'],
        ['CXCR2','WSB1'],
        ['CXCR4','EEF1A1'],
        ['CXCR4','EIF4G2'],
        ['CXCR4','HMGB1'],
        ['CXCR4','HNRNPA2B1'],
        ['CXCR4','HNRNPH1'],
        ['CXCR4','METRNL'],
        ['CXCR4','MYO15B'],
        ['CXCR4','PTMA'],
        ['CXCR4','PTP4A1'],
        ['CXCR4','RARA'],
        ['CXCR4','RBM38'],
        ['CXCR4','SF1'],
        ['CXCR4','SP3'],
        ['CXCR4','THAP4'],
        ['CXCR4','UBB'],
        ['CXCR4','UBC'],
        ['CXCR4','UBXN4'],
        ['CXCR4','YPEL5'],
        ['CXCR4','ZBTB18'],
        ['CXCR4','ZEB2'],
        ['CXORF56','TECRL'],
        ['CXXC5','TG'],
        ['CXXC5','UBE2D2'],
        ['CYB561D1','VPS53'],
        ['CYB5R2','OVCH2'],
        ['CYB5R3','FTH1'],
        ['CYB5R3','HLA@'],
        ['CYB5R3','POLDIP3'],
        ['CYB5R3','TNNC2'],
        ['CYBA','EML4'],
        ['CYBA','WWOX'],
        ['CYBB','XK'],
        ['CYBRD1','MYH11'],
        ['CYC1','FN1'],
        ['CYCS','ZNF561'],
        ['CYGB','ST6GALNAC2'],
        ['CYP11B1','DLK1'],
        ['CYP17A1-AS1','WBP1L'],
        ['CYP19A1','IGF2'],
        ['CYP1A2','IL3RA'],
        ['CYP1A2','KAT6A'],
        ['CYP1A2','MICAL2'],
        ['CYP1B1-AS1','RIN2'],
        ['CYP27A1','TTLL4'],
        ['CYP2A6','CYP2A7'],
        ['CYP2A6','LRP1'],
        ['CYP2B6','ECE1'],
        ['CYP2B6','LRG1'],
        ['CYP2B6','QPRT'],
        ['CYP2B6','SULT2A1'],
        ['CYP2B7P1','LRG1'],
        ['CYP2C8','CYP2E1'],
        ['CYP2C8','SERPINA1'],
        ['CYP2D7','CYP2D8P'],
        ['CYP2E1','FGB'],
        ['CYP2E1','FGG'],
        ['CYP2E1','HP'],
        ['CYP2E1','SERPINA1'],
        ['CYP2G1P','CYP2G2P'],
        ['CYP2U1','HADH'],
        ['CYP3A4','HECW1'],
        ['CYP3A4','SEPP1'],
        ['CYP3A5','LINC00662'],
        ['CYP3A5','TATDN2P2'],
        ['CYP4F11','DNM1P41'],
        ['CYP4F12','CYP4F23P'],
        ['CYP4F12','CYP4F24P'],
        ['CYP4F26P','FAM95C'],
        ['CYP4F3','LA16C-83F12.6'],
        ['CYP4F3','PLCH1'],
        ['CYP4F3','QRFPR'],
        ['CYP4F3','RSPH14'],
        ['CYP4F3','ZNF536'],
        ['CYP4F32P','SNX18P14'],
        ['CYP4F34P','CYP4F35P'],
        ['CYP4V2','NSD2'],
        ['CYP4X1','CYP4Z2P'],
        ['CYP8B1','HRG'],
        ['CYR61','DUSP7'],
        ['CYR61','HLA-B'],
        ['CYR61','IGHG1'],
        ['CYTH1','DIAPH1'],
        ['CYTH1','UBB'],
        ['CYTH1','USP36'],
        ['CYTIP','ERMN'],
        ['D2HGDH','GAL3ST2'],
        ['DA750114','HELLPAR'],
        ['DA750114','IGH@'],
        ['DA750114','IGK@'],
        ['DA750114','IRX2'],
        ['DA750114','JAK2'],
        ['DA750114','LINC00504'],
        ['DA750114','LINC00506'],
        ['DA750114','LINC00939'],
        ['DA750114','LINC02503'],
        ['DA750114','LMO7'],
        ['DA750114','MIPOL1'],
        ['DA750114','MIR100HG'],
        ['DA750114','NMNAT3'],
        ['DA750114','OR14J1'],
        ['DA750114','PIGN'],
        ['DA750114','SOD2'],
        ['DA750114','SULT1B1'],
        ['DA750114','TBC1D32'],
        ['DA750114','TMEM45A'],
        ['DA750114','TRA@'],
        ['DAB2','ENAM'],
        ['DAB2','MEG3'],
        ['DAB2IP','LYZ'],
        ['DAD1','KIAA0922'],
        ['DAGLB','PNLIP'],
        ['DAND5','NFIX'],
        ['DAP','HBB'],
        ['DAP3','UBAP2L'],
        ['DAPP1','PTP4A2'],
        ['DAPP1','SNAP23'],
        ['DAZAP1','GLUL'],
        ['DAZAP2','LYZ'],
        ['DBN1','MYH9'],
        ['DBN1','NUP62'],
        ['DBN1','TBC1D10B'],
        ['DBNDD2','SYS1'],
        ['DBT','RTCA-AS1'],
        ['DBT','SFXN2'],
        ['DCAF11','PSME1'],
        ['DCAF12','HBA2'],
        ['DCAF13','RGS10'],
        ['DCAF7','LYZ'],
        ['DCAF8','SFTPC'],
        ['DCBLD2','PCBP2'],
        ['DCLRE1A','IKZF5'],
        ['DCN','IGF2'],
        ['DCN','LUM'],
        ['DCN','TIMP3'],
        ['DCN','UBE2N'],
        ['DCTN1','FN1'],
        ['DCTN2','PRB2'],
        ['DCTN5','GTDC1'],
        ['DCTN5','UBFD1'],
        ['DCTN6','LYZ'],
        ['DCUN1D1','MCCC1'],
        ['DDA1','MRPL34'],
        ['DDI2','SUMF2'],
        ['DDIT4','TRB@'],
        ['DDOST','LYZ'],
        ['DDOST','SMR3B'],
        ['DDOST','TAS2R4'],
        ['DDOST','TTL'],
        ['DDR1','SFTPA2'],
        ['DDR1-AS1','LINC00243'],
        ['DDR2','NPPA'],
        ['DDR2','UAP1'],
        ['DDRGK1','PROSAPIP1'],
        ['DDX1','NBAS'],
        ['DDX11','DDX12P'],
        ['DDX17','KHDRBS3'],
        ['DDX17','LYZ'],
        ['DDX17','MYH9'],
        ['DDX17','NQO2'],
        ['DDX17','RAB18'],
        ['DDX17','SFTPB'],
        ['DDX17','SFTPC'],
        ['DDX17','SIAH3'],
        ['DDX17','UBC'],
        ['DDX21','KIF1BP'],
        ['DDX24','HBB'],
        ['DDX25','HYLS1'],
        ['DDX27','PARP15'],
        ['DDX39A','JMJD1C'],
        ['DDX39B','LYZ'],
        ['DDX39B','XXBAC-BPG299F13.16'],
        ['DDX3X','EEF1A1'],
        ['DDX3X','LYZ'],
        ['DDX3X','NPPA'],
        ['DDX3X','PNLIP'],
        ['DDX3X','SFTPA1'],
        ['DDX3X','UBC'],
        ['DDX3Y','HBB'],
        ['DDX42','MAP3K3'],
        ['DDX5','EEF1A1'],
        ['DDX5','EEF1D'],
        ['DDX5','EIF4A1'],
        ['DDX5','FUS'],
        ['DDX5','GRK2'],
        ['DDX5','HBB'],
        ['DDX5','HBD'],
        ['DDX5','HMGB1'],
        ['DDX5','HMGN2'],
        ['DDX5','HNRNPH1'],
        ['DDX5','HNRNPU'],
        ['DDX5','IGH@'],
        ['DDX5','LIMS2'],
        ['DDX5','METRNL'],
        ['DDX5','NFE2L1'],
        ['DDX5','OAZ1'],
        ['DDX5','POLG2'],
        ['DDX5','PRSS1'],
        ['DDX5','PTMA'],
        ['DDX5','PTP4A1'],
        ['DDX5','SENP3-EIF4A1'],
        ['DDX5','TEX2'],
        ['DDX5','TPD52L2'],
        ['DDX5','UGCG'],
        ['DDX58','TECRL'],
        ['DDX58','TOPORS'],
        ['DDX6','WAC-AS1'],
        ['DEF6','PPARD'],
        ['DEFA4','DEFA8P'],
        ['DEFA4','HBB'],
        ['DEFA6','IGH@'],
        ['DEFA6','REG3A'],
        ['DEFB114','DEFB133'],
        ['DEFB122','VCL'],
        ['DEGS1','FN1'],
        ['DEGS2','PALLD'],
        ['DENND3','HBA2'],
        ['DENND3','HBB'],
        ['DENND4A','LYZ'],
        ['DENND5A','PCBP1'],
        ['DENND6A-AS1','PDE12'],
        ['DEPDC1B','PDE4D'],
        ['DEPDC1B','PSMD12'],
        ['DEPDC5','LYZ'],
        ['DEPTOR','TG'],
        ['DERL2','DHX33'],
        ['DES','DLGAP4-AS1'],
        ['DES','DUSP3'],
        ['DES','EEF1A2'],
        ['DES','FABP3'],
        ['DES','FHL1'],
        ['DES','FKBP8'],
        ['DES','FLG-AS1'],
        ['DES','FLNA'],
        ['DES','FTH1'],
        ['DES','HHATL'],
        ['DES','HSPB6'],
        ['DES','HTRA1'],
        ['DES','IGH@'],
        ['DES','KLK2'],
        ['DES','KRT1'],
        ['DES','KRT13'],
        ['DES','KRT14'],
        ['DES','KRT16'],
        ['DES','KRT4'],
        ['DES','KRT5'],
        ['DES','KRT6A'],
        ['DES','KRT7'],
        ['DES','LDHB'],
        ['DES','LGMN'],
        ['DES','MB'],
        ['DES','NAP1L4'],
        ['DES','NPPA'],
        ['DES','NPPA-AS1'],
        ['DES','NPPB'],
        ['DES','OGN'],
        ['DES','PARM1'],
        ['DES','PIK3IP1'],
        ['DES','S100A13'],
        ['DES','S100A9'],
        ['DES','SRL'],
        ['DES','SRRM2'],
        ['DES','SYNPO'],
        ['DES','TCF4'],
        ['DES','TIMP2'],
        ['DES','TIMP3'],
        ['DES','TNNI3'],
        ['DES','TNNT2'],
        ['DES','VIM'],
        ['DES','ZBTB38'],
        ['DET1','MRPL46'],
        ['DEXI','FAM102B'],
        ['DFNA5','OSBPL3'],
        ['DGAT2','PSAP'],
        ['DGAT2','RN7SL786P'],
        ['DGCR2','PRH1-PRR4'],
        ['DGCR2','PRH2'],
        ['DGCR6','XXBAC-B444P24.13'],
        ['DGCR8','TANGO2'],
        ['DGKB','TRIOBP'],
        ['DGKD','HBB'],
        ['DGKD','LYZ'],
        ['DGKG','IGF2'],
        ['DGKH','GRB2'],
        ['DGKI','TG'],
        ['DGKZ','LYZ'],
        ['DGKZ','PRB3'],
        ['DGLUCY','ERBB2'],
        ['DHCR24','KRT4'],
        ['DHCR24','LYZ'],
        ['DHCR24','STAR'],
        ['DHCR24','TG'],
        ['DHFR2','TTPA'],
        ['DHODH','MYSM1'],
        ['DHRS1','RABGGTA'],
        ['DHRS12','LINC00282'],
        ['DHRS2','LINC01239'],
        ['DHRS2','LINC01892'],
        ['DHRS2','ZBTB40'],
        ['DHRS3','MAPK1'],
        ['DHRS4-AS1','NRL'],
        ['DHRS7C','GSG1L2'],
        ['DHRSX','SORBS2'],
        ['DHTKD1','LYZ'],
        ['DHX34','IGH@'],
        ['DHX35','FAM83D'],
        ['DHX37','UBC'],
        ['DHX40','RNFT1'],
        ['DHX40','TBC1D3P1-DHX40P1'],
        ['DHX9','GLUL'],
        ['DHX9','NPL'],
        ['DIABLO','TG'],
        ['DIAPH1','HDAC3'],
        ['DIAPH1','TG'],
        ['DIAPH2','MIR100HG'],
        ['DIAPH2','SSR1'],
        ['DICER1','HBB'],
        ['DIP2A','DIP2A-IT1'],
        ['DIP2A','LYZ'],
        ['DIP2B','TMBIM6'],
        ['DIRAS3','WLS'],
        ['DIS3L2','SMR3B'],
        ['DISP2','KNSTRN'],
        ['DIXDC1','DLAT'],
        ['DKK3','MYH6'],
        ['DKK3','MYH7'],
        ['DKK3','MYL4'],
        ['DLD','HNRNPK'],
        ['DLEU1','RNASEH2B'],
        ['DLEU2','HMGB1'],
        ['DLEU2','KLF5'],
        ['DLEU2','SPRYD7'],
        ['DLEU7-AS1','RNASEH2B'],
        ['DLG1','MYH7'],
        ['DLG2','PIGR'],
        ['DLGAP1-AS1','DLGAP1-AS2'],
        ['DLGAP4','ITGAM'],
        ['DLGAP4','LOXL2'],
        ['DLGAP4','MUC7'],
        ['DLGAP4','TRB@'],
        ['DLGAP4-AS1','FLG-AS1'],
        ['DLGAP4-AS1','KRT13'],
        ['DLGAP4-AS1','KRT4'],
        ['DLGAP4-AS1','MYH11'],
        ['DLGAP4-AS1','MYL12A'],
        ['DLGAP4-AS1','NDE1'],
        ['DLGAP4-AS1','TPM2'],
        ['DLGAP4-AS1','WWTR1'],
        ['DLK1','MEG3'],
        ['DMBT1','IGH@'],
        ['DMBT1','LYZ'],
        ['DMBT1','PIGR'],
        ['DMBT1','PRB1'],
        ['DMBT1','PRB2'],
        ['DMBT1','PRH1'],
        ['DMBT1','PRH1-PRR4'],
        ['DMBT1','PRR4'],
        ['DMD','STATH'],
        ['DMKN','EDEM2'],
        ['DMKN','KRT10'],
        ['DMKN','KRTDAP'],
        ['DMKN','SBSN'],
        ['DMTN','LYZ'],
        ['DMTN','TG'],
        ['DMWD','RSPH6A'],
        ['DNA2','SLC25A16'],
        ['DNAAF1','LINC00470'],
        ['DNAAF3','SYT5'],
        ['DNAAF5','LYZ'],
        ['DNAH1','GLYCTK'],
        ['DNAH14','RNF227'],
        ['DNAJA1','POLR2A'],
        ['DNAJB1','TG'],
        ['DNAJB14','TRA@'],
        ['DNAJB2','PNLIP'],
        ['DNAJB2','PRH1'],
        ['DNAJB2','TUBA4B'],
        ['DNAJB6','LYZ'],
        ['DNAJB6','MPO'],
        ['DNAJC1','MIGA1'],
        ['DNAJC15','LINC00400'],
        ['DNAJC18','ECSCR'],
        ['DNAJC25-GNG10','UGCG'],
        ['DNAJC3','HTN3'],
        ['DNAJC5','TPD52L2'],
        ['DNAJC7','LYZ'],
        ['DNAJC9','MRPS16'],
        ['DNASE1','UGDH'],
        ['DNASE2','KLF1'],
        ['DNASE2B','SAMD13'],
        ['DNER','GFAP'],
        ['DNER','SMR3B'],
        ['DNLZ','ITGB5'],
        ['DNM1','DNM1P33'],
        ['DNM1L','FBLIM1'],
        ['DNM1L','GIT2'],
        ['DNM1L','LYZ'],
        ['DNM1L','STX3'],
        ['DNM2','SMR3B'],
        ['DNM2','ZG16B'],
        ['DNM3','EEF1AKNMT'],
        ['DNMT1','HBB'],
        ['DNMT1','LYZ'],
        ['DNMT1','PRSS1'],
        ['DNTTIP2','GNL2'],
        ['DNTTIP2','TRA@'],
        ['DOC2B','LINC02091'],
        ['DOCK1','FAM24B'],
        ['DOCK1','FUS'],
        ['DOCK1','TG'],
        ['DOCK3','MANF'],
        ['DOCK9','IGF2'],
        ['DOCK9','SMR3B'],
        ['DOCK9','STK24'],
        ['DOPEY2','LYZ'],
        ['DOT1L','LLFOS-48D6.2'],
        ['DOT1L','OAZ1'],
        ['DOT1L','PTBP1'],
        ['DOT1L','ZBTB7A'],
        ['DPCR1','IGH@'],
        ['DPCR1','IGK@'],
        ['DPCR1','IGKC'],
        ['DPCR1','IGKV1-8'],
        ['DPCR1','IGKV3-11'],
        ['DPCR1','MUC5AC'],
        ['DPCR1','MUC6'],
        ['DPH5','TG'],
        ['DPH6-AS1','TAP2'],
        ['DPH6-AS1','XXBAC-BPG246D15.9'],
        ['DPH7','PNPLA7'],
        ['DPM2','PIP5KL1'],
        ['DPP3','LYZ'],
        ['DPP3','PELI3'],
        ['DPP4','IGF2'],
        ['DPP6','PABPC1'],
        ['DPP9','TG'],
        ['DPY19L1','KCNK3'],
        ['DPY19L2','DPY19L2P2'],
        ['DPY19L2','HULC'],
        ['DPY19L2','MAP4K3-DT'],
        ['DPY19L2','MRPS23'],
        ['DPY19L2P2','NAPEPLD'],
        ['DPY19L4','INTS8'],
        ['DPY30','RPS9P1'],
        ['DPYSL2','KDELR1'],
        ['DPYSL2','SPOCK1'],
        ['DPYSL3','KRT13'],
        ['DRAIC','PCAT29'],
        ['DRAM1','NR3C1'],
        ['DRAP1','GPR124'],
        ['DRAP1','PALLD'],
        ['DRG1','TG'],
        ['DRICH1','KB-208E9.1'],
        ['DROSHA','TG'],
        ['DSC1','KRT1'],
        ['DSC3','FLG2'],
        ['DSC3','RFX4'],
        ['DSC3','SKP2'],
        ['DSC3','TMEM269'],
        ['DSC3','TMEM99'],
        ['DSCAML1','FXYD6-FXYD2'],
        ['DSCAS','KRT10'],
        ['DSCC1','KB-1471A8.1'],
        ['DSCR3','PTRF'],
        ['DSE','TG'],
        ['DSG2','LYZ'],
        ['DSG2','PRH1'],
        ['DSG2','PRH1-PRR4'],
        ['DSG3','IL1RAPL1'],
        ['DSG3','NXN'],
        ['DSG3','ZBTB11'],
        ['DSP','EEF1A1'],
        ['DSP','IGF2'],
        ['DSP','KRT13'],
        ['DSP','MUC7'],
        ['DSP','TMEM99'],
        ['DSP','TRB@'],
        ['DST','KRT10'],
        ['DSTYK','LYZ'],
        ['DTD1','MKL2'],
        ['DTD2','GRB2'],
        ['DTD2','SS18'],
        ['DTNBP1','JARID2'],
        ['DTX2','UPK3B'],
        ['DTX2P1-UPK3BP1-PMS2P11','PMS2'],
        ['DUOX1','SFTPB'],
        ['DUS1L','FASN'],
        ['DUS3L','FUT6'],
        ['DUSP1','HBB'],
        ['DUSP1','SFTPC'],
        ['DUSP11','TPRKB'],
        ['DUSP16','GPR19'],
        ['DUSP22','MYH11'],
        ['DUSP3','FGG'],
        ['DUSP6','POC1B'],
        ['DUT','ZNF534'],
        ['DUX4L26','DUX4L50'],
        ['DUX4L50','LINC01410'],
        ['DUXAP8','LINC01297'],
        ['DUXAP8','NBEA'],
        ['DVL3','IRF2BPL'],
        ['DYDC1','MAT1A'],
        ['DYM','STATH'],
        ['DYNC1H1','HBB'],
        ['DYNC1H1','SERINC3'],
        ['DYNC1H1','SFTPA1'],
        ['DYNC1H1','SFTPA2'],
        ['DYNC1H1','TBL2'],
        ['DYNC1H1','TKT'],
        ['DYNLL2','PTRF'],
        ['DYNLRB1','ITCH'],
        ['DYNLT1','TATDN2P2'],
        ['DYRK1A','TG'],
        ['DYRK2','EIF1'],
        ['DYRK4','RAD51AP1'],
        ['DYX1C1-CCPG1','PIGBOS1'],
        ['DYX1C1-CCPG1','RAB27A'],
        ['E2F1','NECAB3'],
        ['E2F1','PXMP4'],
        ['E2F3','VPS53'],
        ['E2F3P2','MRE11A'],
        ['E2F4','RPL14'],
        ['E2F4','RPL14P1'],
        ['EAF2','SLC15A2'],
        ['EBF1','IGH@'],
        ['EBF1','JAK2'],
        ['EBLN3P','ZCCHC7'],
        ['ECE1','KRT13'],
        ['ECE1','LYZ'],
        ['ECE1','SFTPC'],
        ['ECE1','TRB@'],
        ['ECH1','HNRNPL'],
        ['ECHDC2','GP2'],
        ['ECHDC2','PNLIP'],
        ['ECI2','PRSS1'],
        ['ECM1','KRT13'],
        ['EDA','SLC35E1'],
        ['EDA2R','ROCK2'],
        ['EDARADD','ENO1'],
        ['EED','HIKESHI'],
        ['EEF1A1','EEF1A2'],
        ['EEF1A1','EIF4B'],
        ['EEF1A1','FOS'],
        ['EEF1A1','G0S2'],
        ['EEF1A1','GP2'],
        ['EEF1A1','HBA1'],
        ['EEF1A1','HBA2'],
        ['EEF1A1','HBB'],
        ['EEF1A1','HMGB1'],
        ['EEF1A1','HMGB2'],
        ['EEF1A1','HNRNPA2B1'],
        ['EEF1A1','HNRNPH1'],
        ['EEF1A1','IGH@'],
        ['EEF1A1','ITGB1'],
        ['EEF1A1','KLK3'],
        ['EEF1A1','KRT13'],
        ['EEF1A1','LYZ'],
        ['EEF1A1','MPO'],
        ['EEF1A1','MUC7'],
        ['EEF1A1','MYADM'],
        ['EEF1A1','MYL2'],
        ['EEF1A1','NFKBIZ'],
        ['EEF1A1','PIGR'],
        ['EEF1A1','PLD3'],
        ['EEF1A1','PRH1'],
        ['EEF1A1','PRH1-PRR4'],
        ['EEF1A1','PRR4'],
        ['EEF1A1','PTMA'],
        ['EEF1A1','PTP4A1'],
        ['EEF1A1','RGS5'],
        ['EEF1A1','RPL4'],
        ['EEF1A1','RPS4X'],
        ['EEF1A1','SCMH1'],
        ['EEF1A1','SERPINA3'],
        ['EEF1A1','SFTPB'],
        ['EEF1A1','SFTPC'],
        ['EEF1A1','SLC25A37'],
        ['EEF1A1','SLC4A1'],
        ['EEF1A1','SREK1'],
        ['EEF1A1','SRSF5'],
        ['EEF1A1','SRSF7'],
        ['EEF1A1','SYNCRIP'],
        ['EEF1A1','TG'],
        ['EEF1A1','TGFBR3'],
        ['EEF1A1','TMBIM6'],
        ['EEF1A1','TPM3'],
        ['EEF1A1','TPM4'],
        ['EEF1A1','TRB@'],
        ['EEF1A1','UBB'],
        ['EEF1A1','UBC'],
        ['EEF1A1','XPR1'],
        ['EEF1A2','MYH7'],
        ['EEF1AKMT3','TSFM'],
        ['EEF1D','KRT8'],
        ['EEF1D','NAPRT'],
        ['EEF1D','PARK2'],
        ['EEF1D','PNLIP'],
        ['EEF1D','SFTPC'],
        ['EEF1D','TG'],
        ['EEF1D','ZC3H3'],
        ['EEF1DP3','FRY'],
        ['EEF2','EGR1'],
        ['EEF2','FASN'],
        ['EEF2','FN1'],
        ['EEF2','GP2'],
        ['EEF2','HBB'],
        ['EEF2','IGK@'],
        ['EEF2','IGKC'],
        ['EEF2','KRT14'],
        ['EEF2','LAMP3'],
        ['EEF2','LGALS1'],
        ['EEF2','LYZ'],
        ['EEF2','MGLL'],
        ['EEF2','MUC7'],
        ['EEF2','NDE1'],
        ['EEF2','PFN1'],
        ['EEF2','PIGR'],
        ['EEF2','PLIN1'],
        ['EEF2','PLIN4'],
        ['EEF2','PRB3'],
        ['EEF2','PRH1'],
        ['EEF2','PRH1-PRR4'],
        ['EEF2','PRSS1'],
        ['EEF2','PRSS2'],
        ['EEF2','TPM2'],
        ['EEF2','TRB@'],
        ['EEF2','VIM-AS1'],
        ['EEF2','WDR1'],
        ['EFCAB10','PUS7'],
        ['EFCAB10','YBX1'],
        ['EFCAB11','SYNC'],
        ['EFCAB14','LYZ'],
        ['EFCAB14','SFTPC'],
        ['EFCAB2','KIF26B'],
        ['EFCAB5','NSRP1'],
        ['EFCAB7','PGM1'],
        ['EGF','SEC24B'],
        ['EGFR','H19'],
        ['EGFR','IGF2'],
        ['EGFR','IGSF6'],
        ['EGFR','IL6ST'],
        ['EGFR','SSR3'],
        ['EGFR','ZG16B'],
        ['EGLN1','NUDT4'],
        ['EGLN1','SFTPB'],
        ['EGR1','KRT13'],
        ['EGR1','MARCKS'],
        ['EGR1','MUC6'],
        ['EGR1','PLIN4'],
        ['EGR1','SERPINA1'],
        ['EGR1','SFTPA2'],
        ['EGR1','SFTPB'],
        ['EGR1','SFTPC'],
        ['EGR1','SPRR1B'],
        ['EGR2','TG'],
        ['EHBP1L1','FAM89B'],
        ['EHBP1L1','SSSCA1'],
        ['EHD1','KLF6'],
        ['EHD1','SF1'],
        ['EHD2','FBLIM1'],
        ['EHD2','FGFR1'],
        ['EHD2','FKBP14'],
        ['EHD2','GLTP'],
        ['EHD2','KIAA1191'],
        ['EHD2','LEPREL4'],
        ['EHD2','MYADM'],
        ['EHD2','PLIN3'],
        ['EHD2','PLIN4'],
        ['EHD2','PTRF'],
        ['EHD2','RGS5'],
        ['EHD2','SFTPB'],
        ['EHD2','SLC2A6'],
        ['EHD2','TPM4'],
        ['EHD2','TRAM2'],
        ['EHD3','IGF2'],
        ['EHMT1','ERBB2'],
        ['EHMT1','SFTPB'],
        ['EHMT2','SLC44A4'],
        ['EI24','UFM1'],
        ['EIF1','EIF1B'],
        ['EIF1','HNRNPA0'],
        ['EIF1','LYZ'],
        ['EIF1','NAMPT'],
        ['EIF1AX','MAP7D2'],
        ['EIF1AX','RPS6KA3'],
        ['EIF1B-AS1','ENTPD3-AS1'],
        ['EIF2AK1','S100A9'],
        ['EIF2AK1','TMSB4X'],
        ['EIF2AK4','ZNF121'],
        ['EIF2B5','KRT1'],
        ['EIF2B5','LYZ'],
        ['EIF2B5','MUC7'],
        ['EIF2B5','MYH7'],
        ['EIF2B5','NPPA-AS1'],
        ['EIF2B5','PRH1'],
        ['EIF2B5','PRH1-PRR4'],
        ['EIF2B5','PRR4'],
        ['EIF2B5','SFTPB'],
        ['EIF2B5','SMR3B'],
        ['EIF2B5','ST3GAL1'],
        ['EIF2B5','TG'],
        ['EIF2D','SFTPC'],
        ['EIF2S1','RBM25'],
        ['EIF2S1','TG'],
        ['EIF2S3','TPM4'],
        ['EIF3B','HTN3'],
        ['EIF3B','ITGAV'],
        ['EIF3D','PTMA'],
        ['EIF3D','TNNT2'],
        ['EIF3E','HTN1'],
        ['EIF3E','PRB3'],
        ['EIF3E','PRH1'],
        ['EIF3E','PRSS1'],
        ['EIF3E','RAB8A'],
        ['EIF3E','SMR3B'],
        ['EIF3E','TG'],
        ['EIF3E','TRB@'],
        ['EIF3F','PNLIP'],
        ['EIF3G','SMR3B'],
        ['EIF3H','HSP90AA1'],
        ['EIF3I','HP1BP3'],
        ['EIF3L','IGF2'],
        ['EIF3M','FDFT1'],
        ['EIF4A1','EIF4A2'],
        ['EIF4A2','SENP3-EIF4A1'],
        ['EIF4B','FNBP1'],
        ['EIF4B','IFI44L'],
        ['EIF4B','TMCO1'],
        ['EIF4E','PIK3C2A'],
        ['EIF4E','TMEM30B'],
        ['EIF4E3','FOXP1'],
        ['EIF4E3','PROK2'],
        ['EIF4G1','FLNC'],
        ['EIF4G1','KRT13'],
        ['EIF4G1','SPARC'],
        ['EIF4G2','PTMA'],
        ['EIF4G3','HP1BP3'],
        ['EIF5','HMGB2'],
        ['EIF5','MARK3'],
        ['EIF5A','FBRS'],
        ['EIF5A','FTH1'],
        ['EIF5A','IRF2BPL'],
        ['EIF5A','PTMS'],
        ['EIF6','LYZ'],
        ['ELANE','HBA2'],
        ['ELANE','HBB'],
        ['ELANE','HBD'],
        ['ELANE','KBTBD11'],
        ['ELANE','LINC01578'],
        ['ELANE','PTBP1'],
        ['ELANE','S100A9'],
        ['ELANE','SF1'],
        ['ELAVL1','HBB'],
        ['ELAVL1','MUC7'],
        ['ELAVL1','PRB1'],
        ['ELAVL1','SMR3B'],
        ['ELAVL1','TIMM44'],
        ['ELAVL1','TRB@'],
        ['ELF1','HMGB1'],
        ['ELF1','TG'],
        ['ELF3','RNPEP'],
        ['ELF5','LYZ'],
        ['ELK1','IGH@'],
        ['ELK2BP','IGH@'],
        ['ELK4','SLC45A3'],
        ['ELL2','PNLIP'],
        ['ELMO1','MPO'],
        ['ELMO1','PNLIP'],
        ['ELN','TNRC18'],
        ['ELOC','MORF4L1'],
        ['ELOVL1','MED8'],
        ['ELOVL2-AS1','SMIM13'],
        ['ELOVL6','MUC5AC'],
        ['ELP2','GLTPD2'],
        ['ELP6','SIK2'],
        ['EMB','TCF3'],
        ['EMB','WWC1'],
        ['EMC10','MCAM'],
        ['EMC2','NUDCD1'],
        ['EMC3','PRRT3'],
        ['EMG1','MIR200CHG'],
        ['EMG1','PPP3CA'],
        ['EMILIN1','EWSR1'],
        ['EMILIN1','GNAS'],
        ['EMILIN1','MVP'],
        ['EMILIN1','TSC22D4'],
        ['EMILIN2','FLNA'],
        ['EML4','PIGR'],
        ['EMP1','EVPL'],
        ['EMP1','MYH11'],
        ['EMP2','F11R'],
        ['EMP2','FN1'],
        ['EMP2','HSPG2'],
        ['EMP2','MIR100HG'],
        ['EMP2','MYH9'],
        ['EMP2','NCOR2'],
        ['EMP2','PTCD1'],
        ['EMP2','SCIMP'],
        ['EMP2','SFTPA1'],
        ['EMP2','VIRMA'],
        ['EMX2','PRKCB'],
        ['EMX2OS','TRA@'],
        ['ENAH','LYZ'],
        ['ENAH','MUC7'],
        ['ENAM','IGFBP5'],
        ['ENAM','KLF6'],
        ['ENAM','LYZ'],
        ['ENAM','PIGR'],
        ['ENAM','ZG16B'],
        ['ENC1','UPK1A'],
        ['ENG','PTRF'],
        ['ENO1','LYZ'],
        ['ENO2','MYBPC3'],
        ['ENO3','NPPA-AS1'],
        ['ENPP1','KCNE3'],
        ['ENPP1','SLC12A8'],
        ['ENSA','MCL1'],
        ['ENTPD1','MCFD2'],
        ['ENTPD1','TG'],
        ['ENTPD1','ZNF521'],
        ['ENTPD1-AS1','TCTN3'],
        ['ENTPD4','IL3RA'],
        ['ENTPD4','LYZ'],
        ['ENTPD4','TRA@'],
        ['ENTPD5','FAM161B'],
        ['ENTPD6','MUC7'],
        ['EOGT','TMF1'],
        ['EOGT','WWOX'],
        ['EPAS1','IGF2'],
        ['EPAS1','MYH9'],
        ['EPAS1','SFTPA2'],
        ['EPAS1','TMEM247'],
        ['EPB41','SLC5A5'],
        ['EPB41L2','IGFBP5'],
        ['EPB41L3','ZBTB14'],
        ['EPB42','LYZ'],
        ['EPCAM','TG'],
        ['EPCAM','TIAL1'],
        ['EPDR1','IGFBP5'],
        ['EPG5','LYZ'],
        ['EPG5','PSTPIP2'],
        ['EPHA1-AS1','ZYX'],
        ['EPHB2','GNAZ'],
        ['EPHB3','HBG1'],
        ['EPHX1','FGG'],
        ['EPHX1','SERPINA1'],
        ['EPHX2','GULOP'],
        ['EPN1','ITGA3'],
        ['EPN1','MMP14'],
        ['EPN2','UBB'],
        ['EPOR','SLC16A12'],
        ['EPOR','SYNC'],
        ['EPRS','HTN3'],
        ['EPRS','LYZ'],
        ['EPS15L1','IGFBP5'],
        ['EPS8L2','TRIP6'],
        ['EPSTI1','LYZ'],
        ['EQTN','MOB3B'],
        ['ERAP1','PNLIP'],
        ['ERBB2','FDPS'],
        ['ERBB2','GIPC1'],
        ['ERBB2','GRB7'],
        ['ERBB2','GSN'],
        ['ERBB2','KRT8'],
        ['ERBB2','LLGL2'],
        ['ERBB2','MCM4'],
        ['ERBB2','PCGF2'],
        ['ERBB2','PIGO'],
        ['ERBB2','PTPA'],
        ['ERBB2','RPL27'],
        ['ERBB2','RPL8'],
        ['ERBB2','RPS19'],
        ['ERBB2','SEC16A'],
        ['ERBB2','SEPTIN9'],
        ['ERBB2','SHROOM3'],
        ['ERBB2','SOCS7'],
        ['ERBB2','SPECC1L-ADORA2A'],
        ['ERBB2','TAF6'],
        ['ERBB2','TCF25'],
        ['ERBB2','VIM'],
        ['ERBB3','GLTP'],
        ['ERBB3','LYZ'],
        ['ERCC1','MYH11'],
        ['ERCC1','S100A9'],
        ['ERCC2','SERPINE1'],
        ['ERCC5','S100A9'],
        ['ERCC6L2','TMEM245'],
        ['ERGIC1','FGB'],
        ['ERGIC1','FGG'],
        ['ERGIC1','FLNA'],
        ['ERGIC1','SMR3B'],
        ['ERGIC1','TRB@'],
        ['ERGIC3','ZG16B'],
        ['ERI3','NOL10'],
        ['ERICH1','MSI2'],
        ['ERICH1','ZNF624'],
        ['ERMN','GFAP'],
        ['ERN1','TEX2'],
        ['ERP27','GP2'],
        ['ERP27','PSMB4'],
        ['ERP44','UBQLN1'],
        ['ERRFI1','PNLIP'],
        ['ERV3-1','SDPR'],
        ['ERVK-28','ZNF585A'],
        ['ERVK13-1','PDPK1'],
        ['ERVK3-1','ZSCAN22'],
        ['ERVK9-11','ZNF433-AS1'],
        ['ERVW-1','LINC01239'],
        ['ERVW-1','PEX1'],
        ['ESAM','TG'],
        ['ESD','HTR2A'],
        ['ESPL1','UGDH'],
        ['ESRP1','WRNIP1'],
        ['ESRP2','FMO9P'],
        ['ESRP2','LYZ'],
        ['ESRRA','KRT18'],
        ['ESRRAP2','TPTE2'],
        ['ESRRG','NEDD9'],
        ['ESRRG','SMR3B'],
        ['ESYT1','PA2G4'],
        ['ESYT2','LYZ'],
        ['ETF1','HSPA9'],
        ['ETF1','LYZ'],
        ['ETFB','NKG7'],
        ['ETFRF1','KRAS'],
        ['ETHE1','XRCC1'],
        ['ETS1','NPAP1'],
        ['ETS2','MMP2'],
        ['ETV6','FOSB'],
        ['ETV6','PIGR'],
        ['ETV6','PIPOX'],
        ['EVI5','PTCD3'],
        ['EVPL','TG'],
        ['EVPLL','TNFRSF13B'],
        ['EWSR1','KRT10'],
        ['EWSR1','LYZ'],
        ['EXO1','WDR64'],
        ['EXOC1','UPK1A'],
        ['EXOC6','HHEX'],
        ['EXOC7','FN1'],
        ['EXOC7','RABEP1'],
        ['EXOSC6','SMG1'],
        ['EXPH5','POGLUT3'],
        ['EXT1','SAMD12'],
        ['EXT2','PPIC'],
        ['EXTL3-AS1','SOD2'],
        ['EYS','LYZ'],
        ['EZH1','TG'],
        ['EZH2','ZNF425'],
        ['EZR','HLA-A'],
        ['EZR','NBEAL2'],
        ['EZR','SFTPC'],
        ['EZR','SMR3B'],
        ['F11-AS1','SLC25A5'],
        ['F11R','FOXM1'],
        ['F11R','LRG1'],
        ['F11R','RBM22P2'],
        ['F11R','TSTD1'],
        ['F13A1','IGF2'],
        ['F2R','LPCAT3'],
        ['F3','PIGS'],
        ['F3','PIN1'],
        ['F5','SELP'],
        ['FA2H','MLKL'],
        ['FABP1','FGA'],
        ['FABP3','NPPA'],
        ['FABP4','FAM213A'],
        ['FABP6','REG3A'],
        ['FABP6','WSB1'],
        ['FADS1','TMEM258'],
        ['FADS3','SLC1A5'],
        ['FAH','ZFAND6'],
        ['FAM102A','IGH@'],
        ['FAM106A','USP32'],
        ['FAM106A','USP6'],
        ['FAM107A','FAM3D'],
        ['FAM107A','SMR3B'],
        ['FAM107B','HBB'],
        ['FAM107B','LYZ'],
        ['FAM110A','JAZF1'],
        ['FAM111A','PROX2'],
        ['FAM117A','SLC35B1'],
        ['FAM117B','YWHAB'],
        ['FAM120AOS','LYST'],
        ['FAM129A','GCSAML'],
        ['FAM129A','HMCN2'],
        ['FAM129A','HTN3'],
        ['FAM129A','MYL2'],
        ['FAM129A','PNLIP'],
        ['FAM129A','TRB@'],
        ['FAM129B','FKBP8'],
        ['FAM129B','KRT13'],
        ['FAM129B','PRSS2'],
        ['FAM129C','KAT6A'],
        ['FAM129C','MMP23B'],
        ['FAM129C','PGLS'],
        ['FAM129C','SSH2'],
        ['FAM132A','UBE2J2'],
        ['FAM133B','FAM133CP'],
        ['FAM134B','ZNF622'],
        ['FAM134C','TEDDM1'],
        ['FAM135A','MAP3K4'],
        ['FAM135A','SDHAF4'],
        ['FAM135A','SSR3'],
        ['FAM136A','FAM136BP'],
        ['FAM13A','PRH1'],
        ['FAM13A','PRH1-PRR4'],
        ['FAM13A','PRR4'],
        ['FAM13A-AS1','HERC3'],
        ['FAM149A','TLR3'],
        ['FAM149B1','PTGR1'],
        ['FAM153A','FAM153CP'],
        ['FAM155A','PCDH12'],
        ['FAM157A','FAM157B'],
        ['FAM157B','FBXO25'],
        ['FAM157C','RPL23AP53'],
        ['FAM160B1','TRUB1'],
        ['FAM162B','ZUFSP'],
        ['FAM163B','LL09NC01-254D11.1'],
        ['FAM167A','TG'],
        ['FAM167A','ZDHHC20'],
        ['FAM167A-AS1','IYD'],
        ['FAM167A-AS1','TG'],
        ['FAM170B','VSTM4'],
        ['FAM172A','KIAA0825'],
        ['FAM174A','ST8SIA4'],
        ['FAM174B','STATH'],
        ['FAM174B','WDR37'],
        ['FAM175A','HELQ'],
        ['FAM177A1','Z99943.2'],
        ['FAM182A','FAM182B'],
        ['FAM185A','FAM185BP'],
        ['FAM193B','LYZ'],
        ['FAM198A','KRBOX1'],
        ['FAM205BP','FAM205C'],
        ['FAM209B','RTFDC1'],
        ['FAM20B','TFDP2'],
        ['FAM210B','LYZ'],
        ['FAM210B','MRPL49'],
        ['FAM211A-AS1','LGALS1'],
        ['FAM212B','PSMB2'],
        ['FAM213A','LIN7C'],
        ['FAM213A','LYZ'],
        ['FAM214A','IL6ST'],
        ['FAM221B','HINT2'],
        ['FAM228B','TRA@'],
        ['FAM230B','SNX1'],
        ['FAM245B','LINC02675'],
        ['FAM3A','SFTPB'],
        ['FAM3D-AS1','SMR3B'],
        ['FAM46A','LYZ'],
        ['FAM46A','SMR3B'],
        ['FAM47E','PTGR1'],
        ['FAM49B','MPO'],
        ['FAM49B','ZFHX3'],
        ['FAM53A','XRRA1'],
        ['FAM53B','METTL10'],
        ['FAM53B','MTHFS'],
        ['FAM53B','ST20-MTHFS'],
        ['FAM53B','TSPAN3'],
        ['FAM53C','KDM3B'],
        ['FAM63B','GTPBP3'],
        ['FAM66A','FAM66C'],
        ['FAM66C','FAM66D'],
        ['FAM71B','MED7'],
        ['FAM71D','MPP5'],
        ['FAM71F1','FAM71F2'],
        ['FAM73A','NEXN'],
        ['FAM83F','TNRC6B'],
        ['FAM85B','LINC02018'],
        ['FAM85B','LINC02614'],
        ['FAM86B1','FAM86B3P'],
        ['FAM86C1','FAM86C2P'],
        ['FAM89B','KLC1'],
        ['FAM8A1','SUMO2P13'],
        ['FAM96B','RRAD'],
        ['FAM98C','MRPS21'],
        ['FAN1','HERC2'],
        ['FARP1','KRT10'],
        ['FARP1','TG'],
        ['FARP2','SEPTIN2'],
        ['FARSA','SYCE2'],
        ['FASN','MGST1'],
        ['FASN','PLIN4'],
        ['FASN','PRB1'],
        ['FASTKD2','ZNF638'],
        ['FBLIM1','NONO'],
        ['FBLIM1','PDGFRA'],
        ['FBLIM1','SCUBE3'],
        ['FBLN1','FSCN1'],
        ['FBLN1','IGF2'],
        ['FBLN1','MEG3'],
        ['FBLN1','MYH11'],
        ['FBLN1','NDE1'],
        ['FBLN1','SERPINH1'],
        ['FBLN2','MBOAT7'],
        ['FBLN5','MYH11'],
        ['FBN1','SERPINE1'],
        ['FBP1','UPF1'],
        ['FBRS','MUC7'],
        ['FBRSL1','NOC4L'],
        ['FBXL16','GFAP'],
        ['FBXL17','PJA2'],
        ['FBXL18','PTGIS'],
        ['FBXL18','TTLL3'],
        ['FBXL19','ORAI2'],
        ['FBXL2','IGF2'],
        ['FBXL20','PTGIR'],
        ['FBXL22','USP3'],
        ['FBXL5','HBB'],
        ['FBXL5','HMGB1'],
        ['FBXL5','LYZ'],
        ['FBXL5','METTL16'],
        ['FBXO11','LYZ'],
        ['FBXO16','HBB'],
        ['FBXO16','STOM'],
        ['FBXO21','NOS1'],
        ['FBXO22','UBE2Q2'],
        ['FBXO25','TDRP'],
        ['FBXO30','SHPRH'],
        ['FBXO32','LIN7C'],
        ['FBXO32','RAB21'],
        ['FBXO42','MORF4L1'],
        ['FBXO43','RGS22'],
        ['FBXO44','MUC7'],
        ['FBXW12','ZNF589'],
        ['FBXW9','TNPO2'],
        ['FCAR','KIAA1211L'],
        ['FCAR','LENG8'],
        ['FCAR','LYZ'],
        ['FCAR','NRF1'],
        ['FCAR','TCF3'],
        ['FCER1G','TOMM40L'],
        ['FCGBP','HNRNPU'],
        ['FCGBP','IGH@'],
        ['FCGBP','IGHA2'],
        ['FCGBP','PAX8-AS1'],
        ['FCGBP','PIGR'],
        ['FCGBP','PNKP'],
        ['FCGBP','SMR3B'],
        ['FCGBP','TPO'],
        ['FCN1','MPO'],
        ['FCN3','MAP3K6'],
        ['FCRL4','IGH@'],
        ['FCRL4','RBM47'],
        ['FCRL4','SAMHD1'],
        ['FDCSP','SMR3B'],
        ['FDFT1','NEIL2'],
        ['FDFT1','SUB1'],
        ['FDPSP7','SLC35E1'],
        ['FDX1','LINC02732'],
        ['FEM1C','HNRNPK'],
        ['FER','MAN2A1'],
        ['FER1L4','IGH@'],
        ['FER1L4','IGHG1'],
        ['FES','MAN2A2'],
        ['FEZ2','IGF2'],
        ['FFAR4','SERPINA1'],
        ['FGA','HP'],
        ['FGA','IGF2'],
        ['FGA','KTN1'],
        ['FGA','P4HB'],
        ['FGA','SAA1'],
        ['FGA','SAA2'],
        ['FGA','SARM1'],
        ['FGA','SERPINA1'],
        ['FGA','SERPINA3'],
        ['FGA','TXNL4B'],
        ['FGB','GC'],
        ['FGB','HP'],
        ['FGB','ORM1'],
        ['FGB','PEBP1'],
        ['FGB','SERPINA1'],
        ['FGB','SERPINA3'],
        ['FGD3','LYZ'],
        ['FGD3','SFTPC'],
        ['FGD5','TPRXL'],
        ['FGD5-AS1','FRRS1L'],
        ['FGD5-AS1','LINC00969'],
        ['FGD5-AS1','MUC20-OT1'],
        ['FGD5-AS1','RCL1'],
        ['FGF14','IGF2'],
        ['FGFR1','PPIC'],
        ['FGFR1OP2','MBNL1'],
        ['FGFR2','PNLIP'],
        ['FGFR2','TMPO'],
        ['FGFR4','ZNF346'],
        ['FGFRL1','IGF2'],
        ['FGFRL1','PLEC'],
        ['FGG','GC'],
        ['FGG','HP'],
        ['FGG','HPX'],
        ['FGG','IGH@'],
        ['FGG','LRG1'],
        ['FGG','PEBP1'],
        ['FGG','RBP4'],
        ['FGG','SERPINA1'],
        ['FGG','SERPINA3'],
        ['FGG','TMBIM6'],
        ['FGG','TXNL4B'],
        ['FGL1','HP'],
        ['FGL1','SERPINA3'],
        ['FHL2','TTN'],
        ['FHL3','SF3A3'],
        ['FHOD3','NPPA'],
        ['FICD','ISCU'],
        ['FIK','ZNF674'],
        ['FIS1','HSPE1'],
        ['FITM2','OSER1'],
        ['FKBP10','JUNB'],
        ['FKBP15','HTN3'],
        ['FKBP15','LYZ'],
        ['FKBP15','PRB3'],
        ['FKBP1A','SDCBP2'],
        ['FKBP3','TOP2B'],
        ['FKBP5','PNLIP'],
        ['FKBP5','SFTPB'],
        ['FKBP5','TARBP1'],
        ['FKBP8','LYZ'],
        ['FKRP','IGF2'],
        ['FKRP','TG'],
        ['FLG-AS1','KRT1'],
        ['FLG-AS1','KRT13'],
        ['FLG-AS1','KRT2'],
        ['FLG-AS1','KRT4'],
        ['FLG-AS1','KRT6A'],
        ['FLG-AS1','NUCKS1'],
        ['FLG-AS1','PROM2'],
        ['FLG-AS1','RNF213'],
        ['FLG-AS1','S100A2'],
        ['FLG-AS1','SBSN'],
        ['FLG-AS1','TGM3'],
        ['FLG-AS1','VAMP2'],
        ['FLG2','GSDMA'],
        ['FLG2','IGK@'],
        ['FLG2','KRT5'],
        ['FLG2','SGPL1'],
        ['FLG2','TSIX'],
        ['FLG2','TTC39C'],
        ['FLG2','TUFT1'],
        ['FLJ22447','PRKCH'],
        ['FLJ37505','LINC00507'],
        ['FLJ46284','TRIQK'],
        ['FLNA','FTH1'],
        ['FLNA','GNAS'],
        ['FLNA','HBB'],
        ['FLNA','HNRNPUL1'],
        ['FLNA','IGF2'],
        ['FLNA','IGH@'],
        ['FLNA','KRT4'],
        ['FLNA','LYZ'],
        ['FLNA','MIF'],
        ['FLNA','NCOR2'],
        ['FLNA','NME4'],
        ['FLNA','NPPA'],
        ['FLNA','PRSS2'],
        ['FLNA','PUF60'],
        ['FLNA','RUSC2'],
        ['FLNA','SFN'],
        ['FLNA','SFTPA2'],
        ['FLNA','SFTPC'],
        ['FLNA','TUBB'],
        ['FLNA','UBC'],
        ['FLNB','PRB3'],
        ['FLNB','SFTPB'],
        ['FLNC','FSTL1'],
        ['FLNC','PLEC'],
        ['FLOT1','HTN3'],
        ['FLOT1','SMR3B'],
        ['FLOT2','NPPA-AS1'],
        ['FLRT2','TRB@'],
        ['FLT1','IGF2'],
        ['FLT1','IGFBP4'],
        ['FLT1','MEG3'],
        ['FMN2','GREM2'],
        ['FMNL1','FMNL3'],
        ['FMNL2','LYZ'],
        ['FMNL2','STATH'],
        ['FMO4','TOP1'],
        ['FN1','HDGF'],
        ['FN1','HLA@'],
        ['FN1','IGH@'],
        ['FN1','IGHG1'],
        ['FN1','INPPL1'],
        ['FN1','KIAA0930'],
        ['FN1','LAMB1'],
        ['FN1','LEPRE1'],
        ['FN1','LGALS1'],
        ['FN1','LMO7'],
        ['FN1','MCAM'],
        ['FN1','MED15'],
        ['FN1','MEF2D'],
        ['FN1','MYADM'],
        ['FN1','NLGN2'],
        ['FN1','NOL6'],
        ['FN1','P4HB'],
        ['FN1','PCSK7'],
        ['FN1','PECAM1'],
        ['FN1','PPP1R18'],
        ['FN1','PRSS2'],
        ['FN1','PTPN23'],
        ['FN1','PTRF'],
        ['FN1','RCN3'],
        ['FN1','RUSC2'],
        ['FN1','SEPT9'],
        ['FN1','SERPINE1'],
        ['FN1','SERPINE2'],
        ['FN1','SETD3'],
        ['FN1','SFTPA2'],
        ['FN1','SH3BGRL3'],
        ['FN1','SIPA1L3'],
        ['FN1','SLC35E1'],
        ['FN1','SLC4A2'],
        ['FN1','SLC6A8'],
        ['FN1','SLC7A5'],
        ['FN1','SLIT2'],
        ['FN1','SMARCA4'],
        ['FN1','SND1'],
        ['FN1','SNX32'],
        ['FN1','STEAP3'],
        ['FN1','TAGLN'],
        ['FN1','THBS1'],
        ['FN1','TIMP1'],
        ['FN1','TUBB'],
        ['FN1','VAMP2'],
        ['FN1','XXBAC-BPG252P9.9'],
        ['FN1','ZBTB4'],
        ['FN1','ZNF282'],
        ['FN1','ZYX'],
        ['FNBP1','LYZ'],
        ['FNBP4','SMR3B'],
        ['FNBP4','Y_RNA'],
        ['FNDC3A','IGF2'],
        ['FNDC3A','TPM4'],
        ['FNIP1','PDE4B'],
        ['FOLR3','LRTOMT'],
        ['FOS','KLF6'],
        ['FOS','KRT13'],
        ['FOS','MMP8'],
        ['FOS','PTRF'],
        ['FOS','SFTPA1'],
        ['FOSB','PPM1N'],
        ['FOSB','RGCC'],
        ['FOSL2','SMR3B'],
        ['FOXC1','PRH1'],
        ['FOXC1','PRH1-PRR4'],
        ['FOXC1','PRR4'],
        ['FOXC2','IGFBP7'],
        ['FOXC2','PLEC'],
        ['FOXJ1','RNF157'],
        ['FOXJ2','TG'],
        ['FOXJ3','TRB@'],
        ['FOXK2','IGFBP3'],
        ['FOXK2','SERPINE2'],
        ['FOXL2NB','MYLK3'],
        ['FOXO1','HMGB1'],
        ['FOXO1','NUP98'],
        ['FOXO3','LYZ'],
        ['FOXP1','GBP6'],
        ['FOXP1','GPRC5A'],
        ['FOXP1','HBB'],
        ['FOXP1','HECA'],
        ['FOXP1','IFNAR2'],
        ['FOXP1','IGH@'],
        ['FOXP1','IGK@'],
        ['FOXP1','LPP'],
        ['FOXP1','PRSS1'],
        ['FOXP1','RYBP'],
        ['FOXP1','TG'],
        ['FOXP1','TMPO'],
        ['FOXP4','PRH1'],
        ['FOXRED2','TXN2'],
        ['FP326651.1','LINC01667'],
        ['FP700111.2','IGK@'],
        ['FPGS','LYZ'],
        ['FRG1','FRG1HP'],
        ['FRG1BP','FRG1HP'],
        ['FRG1CP','FRG1HP'],
        ['FRG1HP','FRG1JP'],
        ['FRG2C','FRG2EP'],
        ['FRK','TRA@'],
        ['FRMD1','SERPINF1'],
        ['FRMD4A','PRB3'],
        ['FRMD4B','MALAT1'],
        ['FRMD6','HERC2'],
        ['FRMD6-AS2','LINC02310'],
        ['FRMPD2','PTPN20'],
        ['FRMPD2','PTPN20CP'],
        ['FRS2','FTL'],
        ['FRS2','ZNF529-AS1'],
        ['FRS3','RBBP4'],
        ['FSCN1','PFKL'],
        ['FSCN1','PHLDA2'],
        ['FSTL1','NCKAP1'],
        ['FSTL1','TG'],
        ['FTCD','SPATC1L'],
        ['FTH1','FTL'],
        ['FTH1','GNB1'],
        ['FTH1','GP2'],
        ['FTH1','HBB'],
        ['FTH1','HNRNPA0'],
        ['FTH1','HNRNPL'],
        ['FTH1','LYZ'],
        ['FTH1','MTCH1'],
        ['FTH1','PDZD2'],
        ['FTH1','POLR2I'],
        ['FTH1','PRB3'],
        ['FTH1','PRELID1'],
        ['FTH1','TSC22D1'],
        ['FTL','HDAC5'],
        ['FTL','S100A9'],
        ['FTL','SFTPA2'],
        ['FTL','TIMP1'],
        ['FTL','TPO'],
        ['FTL','TPT1'],
        ['FTO','FTO-IT1'],
        ['FTO','ZNF587B'],
        ['FTSJ3','SAMD11'],
        ['FTX','KRT10'],
        ['FTX','NIBAN1'],
        ['FTX','PAQR3'],
        ['FTX','TG'],
        ['FURIN','SMR3B'],
        ['FURIN','TG'],
        ['FUS','LGMN'],
        ['FUS','NCL'],
        ['FUS','PTMA'],
        ['FUS','TCF25'],
        ['FUT11','JMJD6'],
        ['FUT2','SEC1P'],
        ['FUT4','PPFIBP1'],
        ['FUT8','STATH'],
        ['FXN','TJP2'],
        ['FXYD2','SMR3B'],
        ['FYTTD1','METRNL'],
        ['FZD4-DT','TMEM135'],
        ['G0S2','MARCH7'],
        ['G0S2','MPO'],
        ['G2E3','SCFD1'],
        ['G3BP1','TNFAIP8'],
        ['G3BP2','PTP4A1'],
        ['G6PC','PLAC8'],
        ['GAA','MUC5AC'],
        ['GAA','SMR3B'],
        ['GAB1','SMARCA5'],
        ['GAB2','LINC02728'],
        ['GABARAP','NPPA-AS1'],
        ['GABARAP','PHF23'],
        ['GABARAPL1','NPPA-AS1'],
        ['GABARAPL2','NPPA-AS1'],
        ['GABPB1','PTP4A1'],
        ['GABPB1-AS1','GBP6'],
        ['GABPB1-AS1','NSD2'],
        ['GABPB2','MLLT11'],
        ['GABPB2','PIGR'],
        ['GABRP','RIN2'],
        ['GABRR2','UBE2J1'],
        ['GAK','ZG16B'],
        ['GAL3ST1','PES1'],
        ['GALK2','SUPT16H'],
        ['GALNT10','SAMD1'],
        ['GALNT10','SFTPA1'],
        ['GALNT15','WWOX'],
        ['GALNT2','LAMP1'],
        ['GALNT6','MLKL'],
        ['GALNTL5','MALT1'],
        ['GAMT','PPP2R5B'],
        ['GAN','TMEM50B'],
        ['GANAB','LYZ'],
        ['GANC','HTN3'],
        ['GAPDH','GAPDHP1'],
        ['GAPDH','GAPDHP14'],
        ['GAPDH','GAPDHP15'],
        ['GAPDH','GAPDHP16'],
        ['GAPDH','GAPDHP17'],
        ['GAPDH','GAPDHP19'],
        ['GAPDH','GAPDHP2'],
        ['GAPDH','GAPDHP20'],
        ['GAPDH','GAPDHP21'],
        ['GAPDH','GAPDHP22'],
        ['GAPDH','GAPDHP23'],
        ['GAPDH','GAPDHP24'],
        ['GAPDH','GAPDHP25'],
        ['GAPDH','GAPDHP26'],
        ['GAPDH','GAPDHP27'],
        ['GAPDH','GAPDHP28'],
        ['GAPDH','GAPDHP29'],
        ['GAPDH','GAPDHP31'],
        ['GAPDH','GAPDHP32'],
        ['GAPDH','GAPDHP33'],
        ['GAPDH','GAPDHP34'],
        ['GAPDH','GAPDHP36'],
        ['GAPDH','GAPDHP37'],
        ['GAPDH','GAPDHP38'],
        ['GAPDH','GAPDHP39'],
        ['GAPDH','GAPDHP40'],
        ['GAPDH','GAPDHP41'],
        ['GAPDH','GAPDHP42'],
        ['GAPDH','GAPDHP44'],
        ['GAPDH','GAPDHP45'],
        ['GAPDH','GAPDHP46'],
        ['GAPDH','GAPDHP47'],
        ['GAPDH','GAPDHP48'],
        ['GAPDH','GAPDHP49'],
        ['GAPDH','GAPDHP50'],
        ['GAPDH','GAPDHP51'],
        ['GAPDH','GAPDHP52'],
        ['GAPDH','GAPDHP53'],
        ['GAPDH','GAPDHP54'],
        ['GAPDH','GAPDHP55'],
        ['GAPDH','GAPDHP56'],
        ['GAPDH','GAPDHP57'],
        ['GAPDH','GAPDHP58'],
        ['GAPDH','GAPDHP59'],
        ['GAPDH','GAPDHP60'],
        ['GAPDH','GAPDHP61'],
        ['GAPDH','GAPDHP62'],
        ['GAPDH','GAPDHP63'],
        ['GAPDH','GAPDHP64'],
        ['GAPDH','GAPDHP65'],
        ['GAPDH','GAPDHP66'],
        ['GAPDH','GAPDHP67'],
        ['GAPDH','GAPDHP68'],
        ['GAPDH','GAPDHP69'],
        ['GAPDH','GAPDHP70'],
        ['GAPDH','GAPDHP71'],
        ['GAPDH','GAPDHP72'],
        ['GAPDH','GAPDHP73'],
        ['GAPDH','GAPDHP74'],
        ['GAPDH','GAPDHP75'],
        ['GAPDH','GAPDHP76'],
        ['GAPDH','GAPDHS'],
        ['GAPDH','H19'],
        ['GAPDH','HBB'],
        ['GAPDH','KRT13'],
        ['GAPDH','MCAM'],
        ['GAPDH','MUC7'],
        ['GAPDH','MYO1A'],
        ['GAPDH','PCSK7'],
        ['GAPDH','PGC'],
        ['GAPDH','PLEC'],
        ['GAPDH','PRH1'],
        ['GAPDH','PRH1-PRR4'],
        ['GAPDH','RMDN3'],
        ['GAPDH','SNX22'],
        ['GAPDH','TIMP3'],
        ['GAPDH','TUBB4B'],
        ['GAPDH','ZNF516'],
        ['GAPDHP38','GAPDHP70'],
        ['GAPLINC','TGIF1'],
        ['GAREM','IGH@'],
        ['GAREM1','TRB@'],
        ['GAS6','PITPNA'],
        ['GAS6','TG'],
        ['GAS7','RCVRN'],
        ['GAST','LIPF'],
        ['GAST','PGC'],
        ['GATA2-AS1','TMED10P2'],
        ['GATAD2A','LYZ'],
        ['GATC','KDM2A'],
        ['GATM','GP2'],
        ['GATM-AS1','SMG1'],
        ['GATSL2','GTF2I'],
        ['GBA','MED4'],
        ['GBA','TG'],
        ['GBGT1','RALGDS'],
        ['GBP1','GBP6'],
        ['GBP2','PTPRC'],
        ['GBP3','KYAT3'],
        ['GBP6','LRP11'],
        ['GBP6','LRP1B'],
        ['GBP6','MYH11'],
        ['GBP6','NRG1'],
        ['GBP6','PTPN12'],
        ['GBP6','RYR2'],
        ['GBP6','SLC27A1'],
        ['GBP6','SLTM'],
        ['GC','SERPINA1'],
        ['GCA','LYZ'],
        ['GCA','SAMSN1'],
        ['GCA','TG'],
        ['GCC1','TG'],
        ['GCC2-AS1','PLGLB1'],
        ['GCLC','YES1'],
        ['GCNT3','PGPEP1'],
        ['GCNT4','LINC01336'],
        ['GCOM1','MARCKS'],
        ['GCSAM','SLC9C1'],
        ['GCSH','NDUFS1'],
        ['GDF7','HSPBP1'],
        ['GDPGP1','TTLL13P'],
        ['GEMIN2','SEC23A-AS1'],
        ['GEMIN6','TSPAN14'],
        ['GFAP','HK1'],
        ['GFAP','IL3RA'],
        ['GFAP','KRT1'],
        ['GFAP','KRT13'],
        ['GFAP','KRT14'],
        ['GFAP','KRT16'],
        ['GFAP','KRT4'],
        ['GFAP','KRT5'],
        ['GFAP','MBP'],
        ['GFAP','MEG3'],
        ['GFAP','NCDN'],
        ['GFAP','NEFM'],
        ['GFAP','PHYHIP'],
        ['GFAP','PLP1'],
        ['GFAP','SERINC1'],
        ['GFAP','SPARC'],
        ['GFAP','SPARCL1'],
        ['GFM1','MFSD1'],
        ['GFOD2','RPL7L1'],
        ['GFOD2','VTI1B'],
        ['GFPT2','SIRT3'],
        ['GFRA1','SAMD14'],
        ['GGA2','ZFAND5'],
        ['GGACT','TMTC4'],
        ['GGACT','ZC3H8'],
        ['GGNBP2','LYZ'],
        ['GGPS1','MAPKAPK5'],
        ['GGT1','IGSF3'],
        ['GGT1','PIGS'],
        ['GGT6','IGH@'],
        ['GHR','GSEC'],
        ['GIMAP8','LINC00996'],
        ['GINM1','LYZ'],
        ['GINM1','RN7SL470P'],
        ['GINM1','STXBP5-AS1'],
        ['GIPC1','IGHA1'],
        ['GIPC1','LINC01484'],
        ['GIPC1','RAB32'],
        ['GIPC2','MYADM'],
        ['GJA9-MYCBP','RHBDL2'],
        ['GK5','LYZ'],
        ['GK5','TFDP2'],
        ['GK5','UXT-AS1'],
        ['GK5','XRN1'],
        ['GKAP1','KIF27'],
        ['GKN1','IGH@'],
        ['GKN1','IGHA1'],
        ['GKN1','IGK@'],
        ['GKN1','IGKC'],
        ['GKN1','LYZ'],
        ['GKN1','NSD1'],
        ['GKN2','IGH@'],
        ['GKN2','LYZ'],
        ['GKN2','PGC'],
        ['GKN2','TFF2'],
        ['GLE1','ODF2'],
        ['GLE1','PWRN1'],
        ['GLG1','HTN3'],
        ['GLI3','GRB10'],
        ['GLIS2','IGF2'],
        ['GLIS3','RFX3'],
        ['GLOD4','NXN'],
        ['GLRX','TG'],
        ['GLRX3','SLC27A2'],
        ['GLT8D1','TG'],
        ['GLTP','GTF2I'],
        ['GLTP','IDE'],
        ['GLTP','LYZ'],
        ['GLTP','MALL'],
        ['GLTP','MAP2K5'],
        ['GLTP','MRGPRX4'],
        ['GLTP','NEK5'],
        ['GLTP','PAFAH1B2'],
        ['GLTP','PCBD2'],
        ['GLTP','POLE2'],
        ['GLTP','SAR1B'],
        ['GLTP','SORBS1'],
        ['GLTP','SUMF2'],
        ['GLTP','TRAPPC9'],
        ['GLTP','U52111.1'],
        ['GLTSCR2','PRSS1'],
        ['GLTSCR2','TRB@'],
        ['GLUL','IGFBP5'],
        ['GLUL','KCNAB2'],
        ['GLUL','PTMA'],
        ['GLUL','S100A9'],
        ['GLUL','SFTPB'],
        ['GLUL','TAGLN2'],
        ['GLYR1','TG'],
        ['GM2A','KAT6A'],
        ['GM2A','LINC00910'],
        ['GMCL1','SNRNP27'],
        ['GMPPB','LRG1'],
        ['GMPPB','TG'],
        ['GMPR','HCG18'],
        ['GMPR','SPTLC2'],
        ['GNAI2','HBB'],
        ['GNAI2','SFTPB'],
        ['GNAO1','MBP'],
        ['GNAQ','LYZ'],
        ['GNAS','GSPT1'],
        ['GNAS','HNRNPA0'],
        ['GNAS','ID4'],
        ['GNAS','IGF2'],
        ['GNAS','IGFBP7'],
        ['GNAS','IGH@'],
        ['GNAS','KRT13'],
        ['GNAS','MAPK1'],
        ['GNAS','MAZ'],
        ['GNAS','MMP2'],
        ['GNAS','MUC7'],
        ['GNAS','MYH11'],
        ['GNAS','OVOL1'],
        ['GNAS','PRB3'],
        ['GNAS','PRH1'],
        ['GNAS','PRH1-PRR4'],
        ['GNAS','PRH2'],
        ['GNAS','PRR4'],
        ['GNAS','PTMA'],
        ['GNAS','RAB21'],
        ['GNAS','RGS5'],
        ['GNAS','SENP3'],
        ['GNAS','SENP3-EIF4A1'],
        ['GNAS','SLA'],
        ['GNAS','TMBIM6'],
        ['GNAS','TRB@'],
        ['GNB1','MYH11'],
        ['GNB1','NADK'],
        ['GNB2L1','LYZ'],
        ['GNE','ITGA10'],
        ['GNE','LYZ'],
        ['GNE','PRH1'],
        ['GNE','PRH1-PRR4'],
        ['GNE','SMR3B'],
        ['GNG10','UGCG'],
        ['GNG12','SMR3B'],
        ['GNG12-AS1','PPP4R3A'],
        ['GNG7','RAB21'],
        ['GNG7','TPM1'],
        ['GNPTG','ZNF502'],
        ['GNS','IGH@'],
        ['GNS','PTRF'],
        ['GOLGA2','SMR3B'],
        ['GOLGA2P8','ZNF774'],
        ['GOLGA3','SFTPC'],
        ['GOLGA4','TG'],
        ['GOLGA4','TJP1'],
        ['GOLGA6L2','GOLGA6L7P'],
        ['GOLGB1','MAGI2'],
        ['GOLIM4','MYH9'],
        ['GOLM1','SOCS6'],
        ['GOLT1A','KISS1'],
        ['GON4L','TG'],
        ['GORASP2','PRH1'],
        ['GORASP2','PRH1-PRR4'],
        ['GORASP2','PRR4'],
        ['GOT2','GOT2P2'],
        ['GOT2','GOT2P3'],
        ['GOT2','TG'],
        ['GP2','GPT2'],
        ['GP2','GSN'],
        ['GP2','HNRNPH1'],
        ['GP2','LMNA'],
        ['GP2','MAN1B1-AS1'],
        ['GP2','MEG3'],
        ['GP2','NPHP3-ACAD11'],
        ['GP2','NTN4'],
        ['GP2','NUPR1'],
        ['GP2','P4HB'],
        ['GP2','PLA2G1B'],
        ['GP2','PNLIP'],
        ['GP2','PNLIPRP1'],
        ['GP2','PNLIPRP2'],
        ['GP2','PRSS1'],
        ['GP2','PRSS2'],
        ['GP2','PRSS3'],
        ['GP2','REG1A'],
        ['GP2','SEL1L'],
        ['GP2','SETD2'],
        ['GP2','SLC4A4'],
        ['GP2','TEX11'],
        ['GP2','TOLLIP'],
        ['GP2','TRB@'],
        ['GP2','UBE2R2-AS1'],
        ['GPBP1','STATH'],
        ['GPBP1L1','SEMA4C'],
        ['GPC1','MZT2B'],
        ['GPC5','GPC5-IT1'],
        ['GPD1','GSN'],
        ['GPD1','MCAM'],
        ['GPI','LRG1'],
        ['GPM6B','PIGR'],
        ['GPR107','TG'],
        ['GPR137B','LINC00513'],
        ['GPR141','NME8'],
        ['GPR155','OTUB1'],
        ['GPR156','LRRC58'],
        ['GPR35','PTMA'],
        ['GPR65','LINC01146'],
        ['GPRC5A','MIR100HG'],
        ['GPRC5A','OGFRL1'],
        ['GPRC5A','SFTPB'],
        ['GPRIN1','SNCB'],
        ['GPT','PPP1R16A'],
        ['GPX3','HNF4A'],
        ['GPX3','KIAA0319L'],
        ['GRAMD1A','TG'],
        ['GRAMD1B','MTHFR'],
        ['GRAP2','TMA7'],
        ['GRB10','NEAT1'],
        ['GRB10','RNF152'],
        ['GRB2','IGF2BP1'],
        ['GRB2','LYZ'],
        ['GRB2','MPO'],
        ['GRB2','RNF130'],
        ['GRB2','SCUBE3'],
        ['GRB2','SFTPB'],
        ['GREB1','MARK4'],
        ['GREB1','PCMT1'],
        ['GREM1','QKI'],
        ['GRHL1','LRRFIP1'],
        ['GRIN2D','KDELR1'],
        ['GRIPAP1','TRA@'],
        ['GRK2','IGH@'],
        ['GRK4','HTT'],
        ['GRK6','PRR7'],
        ['GRN','IGH@'],
        ['GRN','LYZ'],
        ['GRSF1','RSRP1'],
        ['GRWD1','PTRF'],
        ['GS1-124K5.11','GS1-124K5.12'],
        ['GS1-124K5.12','RABGEF1'],
        ['GS1-124K5.2','RABGEF1'],
        ['GS1-124K5.2','SKP1'],
        ['GS1-124K5.4','TPST1'],
        ['GS1-259H13.11','LYZ'],
        ['GS1-259H13.2','ZNF655'],
        ['GS1-259H13.2','ZSCAN25'],
        ['GSDMB','ORMDL3'],
        ['GSE1','SLC7A5'],
        ['GSE1','TG'],
        ['GSN','ITGB1'],
        ['GSN','KRT13'],
        ['GSN','LYZ'],
        ['GSN','MMP25'],
        ['GSN','NPPA'],
        ['GSN','PECAM1'],
        ['GSN','PLIN4'],
        ['GSN','PNLIP'],
        ['GSN','PRSS1'],
        ['GSN','PRSS2'],
        ['GSN','PTRF'],
        ['GSN','S100A9'],
        ['GSN','SMR3B'],
        ['GSN','SPTBN1'],
        ['GSN','TIMP3'],
        ['GSN','TRB@'],
        ['GSN','TRBC2'],
        ['GSN','TRBV25-1'],
        ['GSN','ZNF106'],
        ['GSR','GTF2E2'],
        ['GSR','ZNF91'],
        ['GSTA1','GSTA7P'],
        ['GSTA5','GSTA7P'],
        ['GSTK1','UBA52'],
        ['GSTM3','NFIC'],
        ['GSTM4','GSTM5'],
        ['GSTM4','MUC7'],
        ['GSTO2','PRRC2C'],
        ['GSTP1','HBB'],
        ['GSTP1','PIGR'],
        ['GSTT4','KLHL20'],
        ['GTF2F2','GTF2F2P1'],
        ['GTF2H2','NAIP'],
        ['GTF2H2B','NAIP'],
        ['GTF2H2C','NAIP'],
        ['GTF2H5','SORBS2'],
        ['GTF2H5','SOX2-OT'],
        ['GTF2I','HBB'],
        ['GTF2I','HTN1'],
        ['GTF2I','HTN3'],
        ['GTF2I','PNLIP'],
        ['GTF2I','PRSS1'],
        ['GTF2I','RNF10'],
        ['GTF2I','SMR3B'],
        ['GTF2I','STATH'],
        ['GTF2IP7','LINC00174'],
        ['GTF2IRD2','LYZ'],
        ['GTPBP1','LAMC1'],
        ['GTPBP1','TG'],
        ['GUCA1B','KRT13'],
        ['GUCA1B','KRT14'],
        ['GUCA1B','KRT16'],
        ['GUCA1C','MORC1'],
        ['GUCD1','LDLR'],
        ['GUCD1','LRG1'],
        ['GUCD1','RFC2'],
        ['GUCY2C','PLBD1'],
        ['GULP1','SCN9A'],
        ['GUSBP11','LYZ'],
        ['GUSBP4','LINC00680'],
        ['GVQW3','PCYOX1'],
        ['GVQW3','VCL'],
        ['GYG2','TG'],
        ['GYG2','XG'],
        ['GYPA','GYPB'],
        ['GYPA','GYPE'],
        ['GYPA','HBB'],
        ['GYPB','GYPE'],
        ['GYPB','SAV1'],
        ['GYS1','MYL2'],
        ['H19','NRP1'],
        ['H19','PFN1'],
        ['H19','PLAC4'],
        ['H19','PLPPR2'],
        ['H19','QSOX1'],
        ['H19','TBC1D14'],
        ['H19','TPT1'],
        ['H1F0','HBB'],
        ['H1F0','LYZ'],
        ['H1F0','PRB1'],
        ['H1F0','PRB2'],
        ['H1F0','PRH1'],
        ['H1F0','PRH1-PRR4'],
        ['H1F0','S100A9'],
        ['H1FX','RABL6'],
        ['H1FX-AS1','HBB'],
        ['H1FX-AS1','S100A8'],
        ['H2AFJ','HBA1'],
        ['H2AFJ','HIST1H3J'],
        ['H2AFV','HMGCL'],
        ['H2AFV','RNF128'],
        ['H2AFY','SFTPA2'],
        ['H3F3A','LYZ'],
        ['H3F3A','S100A9'],
        ['H3F3A','WWOX'],
        ['H3F3B','HBB'],
        ['H3F3B','LYZ'],
        ['H3F3B','MPO'],
        ['H3F3B','ZG16B'],
        ['H6PD','LRG1'],
        ['H6PD','LYZ'],
        ['H6PD','PMPCA'],
        ['H6PD','SMR3B'],
        ['H6PD','SPSB1'],
        ['HABP4','LINC01545'],
        ['HACD1','PLS3'],
        ['HACD3','IGF2'],
        ['HADH','SGMS2'],
        ['HADHA','HBB'],
        ['HAIR','KRT13'],
        ['HAIR','KRT14'],
        ['HAND2-AS1','NPPA-AS1'],
        ['HAPLN3','MFGE8'],
        ['HARS2','ZMAT2'],
        ['HAS3','UTP4'],
        ['HAUS4','PRMT5'],
        ['HAX1','UBAP2L'],
        ['HBA1','HBB'],
        ['HBA1','MCL1'],
        ['HBA1','RPS27'],
        ['HBA1','SLC4A1'],
        ['HBA1','SNX22'],
        ['HBA2','HBB'],
        ['HBA2','HBD'],
        ['HBA2','HBM'],
        ['HBA2','HERPUD1'],
        ['HBA2','MPO'],
        ['HBA2','NFE2L1'],
        ['HBA2','SLC4A1'],
        ['HBA2','TUBB4B'],
        ['HBB','HBM'],
        ['HBB','HGS'],
        ['HBB','HIPK3'],
        ['HBB','HLA-A'],
        ['HBB','HLA-C'],
        ['HBB','HLA-DRA'],
        ['HBB','HLA@'],
        ['HBB','HNRNPAB'],
        ['HBB','HOOK2'],
        ['HBB','ICE1'],
        ['HBB','IER2'],
        ['HBB','IGH@'],
        ['HBB','IGK@'],
        ['HBB','IGKC'],
        ['HBB','ILF3'],
        ['HBB','IQSEC1'],
        ['HBB','KDM3A'],
        ['HBB','KLF6'],
        ['HBB','LAPTM5'],
        ['HBB','LMNA'],
        ['HBB','LTF'],
        ['HBB','LYST'],
        ['HBB','MBNL3'],
        ['HBB','MCL1'],
        ['HBB','MICAL2'],
        ['HBB','MOB3A'],
        ['HBB','MPO'],
        ['HBB','MTMR14'],
        ['HBB','MYH10'],
        ['HBB','MYO9B'],
        ['HBB','NCOA6'],
        ['HBB','NLRC5'],
        ['HBB','NOP56'],
        ['HBB','NSD1'],
        ['HBB','NSUN3'],
        ['HBB','NUDT4'],
        ['HBB','PAFAH1B1'],
        ['HBB','PDE4B'],
        ['HBB','PDIA3'],
        ['HBB','PGK1'],
        ['HBB','PHLDA1'],
        ['HBB','PINK1'],
        ['HBB','PKM'],
        ['HBB','PLBD1'],
        ['HBB','PLXND1'],
        ['HBB','PNP'],
        ['HBB','POLDIP2'],
        ['HBB','PRR13'],
        ['HBB','PRTN3'],
        ['HBB','PTMA'],
        ['HBB','RASA3'],
        ['HBB','RBMS1'],
        ['HBB','RNF216'],
        ['HBB','RPL3'],
        ['HBB','RPL37A'],
        ['HBB','RPL8'],
        ['HBB','RPLP1'],
        ['HBB','RPLP2'],
        ['HBB','RPS14'],
        ['HBB','RPS27'],
        ['HBB','RPS3'],
        ['HBB','RPS7'],
        ['HBB','RPS8'],
        ['HBB','RRP12'],
        ['HBB','RUNX3'],
        ['HBB','S100A9'],
        ['HBB','SDHA'],
        ['HBB','SEPT2'],
        ['HBB','SESN3'],
        ['HBB','SFI1'],
        ['HBB','SFTPC'],
        ['HBB','SGMS1'],
        ['HBB','SLC6A9'],
        ['HBB','SMARCA2'],
        ['HBB','SNCA'],
        ['HBB','SNRPB2'],
        ['HBB','SNX32'],
        ['HBB','SP100'],
        ['HBB','SPPL2B'],
        ['HBB','SPTA1'],
        ['HBB','STAT6'],
        ['HBB','STK35'],
        ['HBB','SULF2'],
        ['HBB','SYNE1'],
        ['HBB','TACC3'],
        ['HBB','TET3'],
        ['HBB','TMSB4X'],
        ['HBB','TNFAIP3'],
        ['HBB','TPM3'],
        ['HBB','TRAK2'],
        ['HBB','TSC2'],
        ['HBB','TTC7A'],
        ['HBB','TTN'],
        ['HBB','TXNIP'],
        ['HBB','UBE2Z'],
        ['HBB','UNC13D'],
        ['HBB','USP15'],
        ['HBB','VIM-AS1'],
        ['HBB','VPS13A'],
        ['HBB','VPS28'],
        ['HBB','VTI1A'],
        ['HBB','XPO6'],
        ['HBB','ZBTB1'],
        ['HBB','ZFAND5'],
        ['HBB','ZNF395'],
        ['HBB','ZNF726'],
        ['HBB','ZWINT'],
        ['HBD','LTF'],
        ['HBG2','IGF2'],
        ['HBG2','TRIM5'],
        ['HBP1','IMMP2L'],
        ['HBP1','KMT2E'],
        ['HCG18','NPM1'],
        ['HCG21','IGH@'],
        ['HCG21','IGHG1'],
        ['HCG21','SFTA2'],
        ['HCG23','XXBAC-BPG154L12.5'],
        ['HCG4','XXBAC-BPG248L24.12'],
        ['HCG4B','XXBAC-BPG248L24.12'],
        ['HCG4P5','HLA@'],
        ['HCLS1','TMEM9B'],
        ['HDAC1','RNF123'],
        ['HDAC2','LYZ'],
        ['HDAC2','NPPA-AS1'],
        ['HDAC4','PTMA'],
        ['HDGFRP2','TAOK2'],
        ['HDLBP','IGF2'],
        ['HDLBP','IGFBP5'],
        ['HDLBP','KAT6A'],
        ['HDLBP','LYZ'],
        ['HDLBP','MYH11'],
        ['HDLBP','PGC'],
        ['HDLBP','PIGR'],
        ['HDLBP','PNLIP'],
        ['HDLBP','PTMA'],
        ['HEATR1','PRSS1'],
        ['HEATR1','PRSS2'],
        ['HEATR1','TRBC2'],
        ['HEBP2','SMR3B'],
        ['HECA','SLC48A1'],
        ['HECA','XXBAC-BPG32J3.22'],
        ['HECW2','LINC00381'],
        ['HEG1','MAZ'],
        ['HEIH','ZFP62'],
        ['HELLPAR','IGH@'],
        ['HELLPAR','IGK@'],
        ['HELLPAR','TRA@'],
        ['HELLS','MASP2'],
        ['HELZ2','SFTPA1'],
        ['HELZ2','SFTPC'],
        ['HEPHL1','PANX1'],
        ['HERC1','LYZ'],
        ['HERC1','TG'],
        ['HERC1','USP3-AS1'],
        ['HERC2','PWRN1'],
        ['HERPUD1','SLC12A3'],
        ['HERPUD2','LYZ'],
        ['HERPUD2','MSMO1'],
        ['HERPUD2','SEPT7-AS1'],
        ['HEXIM1','SNRNP40'],
        ['HGF','IGF2'],
        ['HGSNAT','TG'],
        ['HHIPL2','TAF1A'],
        ['HHLA3','TESPA1'],
        ['HIAT1','SLC35A3'],
        ['HIC2','PI4KA'],
        ['HIF1A','LYZ'],
        ['HIF1A','NEK9'],
        ['HIF1A','PTP4A1'],
        ['HIF1A','RNPC3'],
        ['HIF1A','S100A8'],
        ['HIF1A','SF3B1'],
        ['HIF1A','STAG2'],
        ['HIF1A-AS2','MYO1F'],
        ['HIF1A-AS2','SFT2D2'],
        ['HIGD1C','METTL7A'],
        ['HILPDA','RCAN3'],
        ['HIP1','IGF2'],
        ['HIPK1','HMGB1'],
        ['HIPK1','OLFML3'],
        ['HIPK2','METTL7A'],
        ['HIPK2','PRB3'],
        ['HIPK2','TG'],
        ['HIST1H2AA','SLC17A1'],
        ['HIST1H2AG','HIST1H4I'],
        ['HIST1H2AG','TSC22D1'],
        ['HIST1H2APS1','HIST1H2BA'],
        ['HIST1H2BF','HIST1H4E'],
        ['HIST1H2BH','TRIM13'],
        ['HKR1','LINC00665'],
        ['HKR1','TG'],
        ['HKR1','ZNF790-AS1'],
        ['HLA-A','HLA-B'],
        ['HLA-A','HLA-C'],
        ['HLA-A','HLA-DRA'],
        ['HLA-A','HLA-E'],
        ['HLA-B','HLA-C'],
        ['HLA-B','HLA-DRA'],
        ['HLA-B','HLA-E'],
        ['HLA-B','SFTPC'],
        ['HLA-C','HLA-DRA'],
        ['HLA-C','HLA-E'],
        ['HLA-C','LTF'],
        ['HLA-C','LYZ'],
        ['HLA-DRA','TGFBI'],
        ['HLA-DRB1','HLA-DRB6'],
        ['HLA-DRB5','HLA-DRB6'],
        ['HLA-E','MPO'],
        ['HLA-E','PLIN4'],
        ['HLA-E','ZNF638'],
        ['HLA@','IGH@'],
        ['HLA@','IGK@'],
        ['HLA@','INMT'],
        ['HLA@','LTF'],
        ['HLA@','MACF1'],
        ['HLA@','MPO'],
        ['HLA@','PIGR'],
        ['HLA@','RAP1A'],
        ['HLA@','SFTPA2'],
        ['HLA@','SFTPC'],
        ['HLA@','SPTBN1'],
        ['HLA@','TGFBI'],
        ['HLA@','XXBAC-BPG254F23.6'],
        ['HLA@','ZNF638'],
        ['HLX','MARC1'],
        ['HM13','SMR3B'],
        ['HMBS','SBNO2'],
        ['HMG20A','SRSF8'],
        ['HMG20B','TAX1BP3'],
        ['HMGA1','SRGN'],
        ['HMGA1P4','WDR34'],
        ['HMGB1','HMGB2'],
        ['HMGB1','HMGN2'],
        ['HMGB1','HNRNPH1'],
        ['HMGB1','HSP90AB1'],
        ['HMGB1','IGH@'],
        ['HMGB1','MXD4'],
        ['HMGB1','N4BP2L1'],
        ['HMGB1','NCL'],
        ['HMGB1','PNLIP'],
        ['HMGB1','SRSF5'],
        ['HMGB1','TMEM18'],
        ['HMGB1','YPEL5'],
        ['HMGB1','ZBED3-AS1'],
        ['HMGB2','IVNS1ABP'],
        ['HMGB2','NCL'],
        ['HMGB2','PTMA'],
        ['HMGB2','TMSB4X'],
        ['HMGN2','NECAB3'],
        ['HMGN2','PTMA'],
        ['HMGN2','SEPT5'],
        ['HMGN3','SFTPB'],
        ['HMGXB4','LINC02321'],
        ['HMGXB4','TOM1'],
        ['HMHA1','TFE3'],
        ['HMOX1','SFTPC'],
        ['HMP19','METTL7A'],
        ['HNF1A-AS1','LYZ'],
        ['HNRNPA1','KRT13'],
        ['HNRNPA1','KRT4'],
        ['HNRNPA1L2','MRPS31P4'],
        ['HNRNPA1P27','XKRX'],
        ['HNRNPA2B1','IGH@'],
        ['HNRNPA2B1','LYZ'],
        ['HNRNPA2B1','PTMA'],
        ['HNRNPA2B1','PTP4A1'],
        ['HNRNPA2B1','SKAP2'],
        ['HNRNPA2B1','UBB'],
        ['HNRNPA3','LYZ'],
        ['HNRNPA3','SFTPC'],
        ['HNRNPAB','TPBG'],
        ['HNRNPC','KIAA0586'],
        ['HNRNPC','LYZ'],
        ['HNRNPC','PTP4A1'],
        ['HNRNPD','IGFBP4'],
        ['HNRNPD','RCC1'],
        ['HNRNPD','SMR3B'],
        ['HNRNPDL','RNF128'],
        ['HNRNPDL','TUBA1B'],
        ['HNRNPF','ZCCHC24'],
        ['HNRNPH1','ITPRIP'],
        ['HNRNPH1','KRT4'],
        ['HNRNPH1','LYZ'],
        ['HNRNPH1','PTP4A1'],
        ['HNRNPH1','TG'],
        ['HNRNPK','KIF2A'],
        ['HNRNPK','NFIL3'],
        ['HNRNPK','PRB3'],
        ['HNRNPK','UBQLN1'],
        ['HNRNPL','SPP1'],
        ['HNRNPM','LYZ'],
        ['HNRNPM','MARCH2'],
        ['HNRNPU','HSP90AA1'],
        ['HNRNPU-AS1','TG'],
        ['HNRNPUL1','MAEA'],
        ['HNRNPUL1','PRB3'],
        ['HNRNPUL1','SMR3B'],
        ['HNRNPUL2-BSCL2','LYZ'],
        ['HOOK2','HTN3'],
        ['HOOK2','KLK2'],
        ['HOOK2','PNLIP'],
        ['HOOK2','VIM'],
        ['HOOK3','PPM1A'],
        ['HORMAD1','SLC22A20'],
        ['HORMAD1','SLC22A20P'],
        ['HOXC8','HOXC9'],
        ['HOXD11','NFE2L2'],
        ['HP','ORM1'],
        ['HP','PLG'],
        ['HP','SAA1'],
        ['HP','SDHA'],
        ['HP','SERPINA1'],
        ['HP','SERPINA3'],
        ['HP','TF'],
        ['HP1BP3','LYZ'],
        ['HPCAL1','LYZ'],
        ['HPGD','HSPA8'],
        ['HPN','TRB@'],
        ['HPR','SERPINA1'],
        ['HPS1','LYZ'],
        ['HPS1','SMG1'],
        ['HPS5','SAA2'],
        ['HPX','SERPINA1'],
        ['HPX','TRIM3'],
        ['HRG','ORM1'],
        ['HRG','ORM2'],
        ['HRG','RBP4'],
        ['HRG','SERPINA1'],
        ['HRH4','IMPACT'],
        ['HS1BP3','HS1BP3-IT1'],
        ['HS1BP3','PDF'],
        ['HS2ST1','LINC01140'],
        ['HSBP1L1','ZNF554'],
        ['HSD17B11','HSD17B13'],
        ['HSD17B14','PLEKHA4'],
        ['HSD17B2','LINC01146'],
        ['HSD17B2','LINC01920'],
        ['HSD17B4','TNFAIP8'],
        ['HSDL2','KIAA1958'],
        ['HSF4','NOL3'],
        ['HSF5','RNF43'],
        ['HSP90AA1','HSP90AA4P'],
        ['HSP90AA1','IGH@'],
        ['HSP90AA1','LYZ'],
        ['HSP90AB1','SLC29A1'],
        ['HSP90AB3P','SLC29A1'],
        ['HSP90B1','HSP90B3P'],
        ['HSP90B1','PTMS'],
        ['HSP90B1','SYN3'],
        ['HSPA1B','TPT1'],
        ['HSPA2','ZBTB1'],
        ['HSPA5','MPO'],
        ['HSPA5','TG'],
        ['HSPA6','HSPA7'],
        ['HSPA8','MALAT1'],
        ['HSPA9','UBE2D2'],
        ['HSPB1','TIMP3'],
        ['HSPB11','YIPF1'],
        ['HSPB6','PLEKHA4'],
        ['HSPB6','PLIN1'],
        ['HSPB6','PLIN4'],
        ['HSPB7','MYL7'],
        ['HSPB7','NPPA-AS1'],
        ['HSPB8','NPPA'],
        ['HSPBP1','ZSWIM6'],
        ['HSPE1','MOB4'],
        ['HSPE1P22','ZNF512B'],
        ['HSPG2','MYH6'],
        ['HSPG2','PCBP2'],
        ['HSPG2','SCD'],
        ['HSPG2','SFTPB'],
        ['HSPG2','TG'],
        ['HSPG2','TLN1'],
        ['HTN1','IGH@'],
        ['HTN1','IGK@'],
        ['HTN1','IGKC'],
        ['HTN1','IGKV1-8'],
        ['HTN1','LPO'],
        ['HTN1','MYH11'],
        ['HTN1','NAALADL2'],
        ['HTN1','PDIA6'],
        ['HTN1','PLA2R1'],
        ['HTN1','PRB2'],
        ['HTN1','PRH1'],
        ['HTN1','PRR27'],
        ['HTN1','SH3BGRL2'],
        ['HTN1','SMR3B'],
        ['HTN1','TTC3'],
        ['HTN1','WWC1'],
        ['HTN1','ZG16B'],
        ['HTN3','IGH@'],
        ['HTN3','IGK@'],
        ['HTN3','IGKC'],
        ['HTN3','IGKV1-8'],
        ['HTN3','IGKV3-11'],
        ['HTN3','IL6ST'],
        ['HTN3','ITPR2'],
        ['HTN3','LARP1'],
        ['HTN3','LGR4'],
        ['HTN3','LRRC26'],
        ['HTN3','LYZ'],
        ['HTN3','MAOA'],
        ['HTN3','MAPK1'],
        ['HTN3','MBNL1'],
        ['HTN3','MCFD2'],
        ['HTN3','MGEA5'],
        ['HTN3','MGP'],
        ['HTN3','MSN'],
        ['HTN3','NBAS'],
        ['HTN3','PDIA3'],
        ['HTN3','PIK3R1'],
        ['HTN3','PITPNA'],
        ['HTN3','PRB3'],
        ['HTN3','PRH1'],
        ['HTN3','PRH1-PRR4'],
        ['HTN3','PRH2'],
        ['HTN3','PRR27'],
        ['HTN3','PRR4'],
        ['HTN3','PTK7'],
        ['HTN3','RNF150'],
        ['HTN3','SLC12A2'],
        ['HTN3','SMR3B'],
        ['HTN3','SSBP2'],
        ['HTN3','TACC2'],
        ['HTN3','TCN1'],
        ['HTN3','TMEM123'],
        ['HTN3','TOPORS-AS1'],
        ['HTN3','TRAM1'],
        ['HTN3','TRIM2'],
        ['HTN3','TSC2'],
        ['HTN3','USO1'],
        ['HTN3','VPS13B'],
        ['HTRA1','LYZ'],
        ['HTRA1','PLIN4'],
        ['HTT','TPO'],
        ['HULC','PLCB1'],
        ['HUWE1','RPS2'],
        ['HUWE1','TG'],
        ['HVCN1','METTL7A'],
        ['HVCN1','PPP1CC'],
        ['HYDIN2','LYZ'],
        ['HYKK','IREB2'],
        ['HYKK','PSMA4'],
        ['HYPK','WSB1'],
        ['IBA57','SORBS1'],
        ['ICAM1','SFTPB'],
        ['ICAM1','TNXB'],
        ['ID4','IGH@'],
        ['ID4','PTMS'],
        ['ID4','TMEM158'],
        ['IDI2-AS1','WDR37'],
        ['IDS','LINC00893'],
        ['IDS','ZMAT4'],
        ['IER2','OSCAR'],
        ['IFI44L','WWOX'],
        ['IFITM2','SFTPC'],
        ['IFITM3','SFTPA2'],
        ['IFNAR2','NFX1'],
        ['IFNAR2','ZNF595'],
        ['IFNG-AS1','LINC01479'],
        ['IFRD1','UBE2H'],
        ['IFT122','RYK'],
        ['IFT20','PNLIP'],
        ['IFT80','TMEM87A'],
        ['IGF2','IGFBP5'],
        ['IGF2','IL6ST'],
        ['IGF2','INS'],
        ['IGF2','ISM2'],
        ['IGF2','ITGB1'],
        ['IGF2','KDM5B'],
        ['IGF2','KIAA0895L'],
        ['IGF2','KLF7'],
        ['IGF2','LAMB1'],
        ['IGF2','LGMN'],
        ['IGF2','LHFP'],
        ['IGF2','MACF1'],
        ['IGF2','MAF'],
        ['IGF2','MBNL1'],
        ['IGF2','MED15'],
        ['IGF2','MGRN1'],
        ['IGF2','MPEG1'],
        ['IGF2','MPRIP'],
        ['IGF2','MSN'],
        ['IGF2','MYH10'],
        ['IGF2','MYH11'],
        ['IGF2','MYH9'],
        ['IGF2','NAP1L1'],
        ['IGF2','NID1'],
        ['IGF2','NPPA-AS1'],
        ['IGF2','OLR1'],
        ['IGF2','OS9'],
        ['IGF2','OSBPL9'],
        ['IGF2','PAFAH1B1'],
        ['IGF2','PALM2-AKAP2'],
        ['IGF2','PAPPA'],
        ['IGF2','PCBP2'],
        ['IGF2','PECAM1'],
        ['IGF2','PLPP3'],
        ['IGF2','PODN'],
        ['IGF2','PRG2'],
        ['IGF2','PTPRF'],
        ['IGF2','RAB7A'],
        ['IGF2','RAP2C-AS1'],
        ['IGF2','RBM3'],
        ['IGF2','RBM6'],
        ['IGF2','REEP5'],
        ['IGF2','RHOA'],
        ['IGF2','SCARA3'],
        ['IGF2','SCARB1'],
        ['IGF2','SEC31A'],
        ['IGF2','SEPP1'],
        ['IGF2','SEPT9'],
        ['IGF2','SERPINE2'],
        ['IGF2','SFTPA1'],
        ['IGF2','SLC16A3'],
        ['IGF2','SNX2'],
        ['IGF2','SNX24'],
        ['IGF2','SPINT2'],
        ['IGF2','STAU1'],
        ['IGF2','SYN3'],
        ['IGF2','SYNPO'],
        ['IGF2','TIMP2'],
        ['IGF2','TIMP3'],
        ['IGF2','TP53I11'],
        ['IGF2','TREML2'],
        ['IGF2','TRIOBP'],
        ['IGF2','TXNL4B'],
        ['IGF2','VSNL1'],
        ['IGF2','VWF'],
        ['IGF2','ZFAT'],
        ['IGF2BP3','RPS6KA2'],
        ['IGF2R','IGFBP4'],
        ['IGFBP2','LAMC1'],
        ['IGFBP2','RPL37A'],
        ['IGFBP2','TRIM28'],
        ['IGFBP3','MYH9'],
        ['IGFBP4','KLF2'],
        ['IGFBP4','NELFB'],
        ['IGFBP4','PLEC'],
        ['IGFBP4','SCARF2'],
        ['IGFBP4','SERPINA1'],
        ['IGFBP4','SPIRE1'],
        ['IGFBP4','SPTBN1'],
        ['IGFBP4','TRIM28'],
        ['IGFBP4','TSR3'],
        ['IGFBP5','IGH@'],
        ['IGFBP5','NCOR2'],
        ['IGFBP5','NDE1'],
        ['IGFBP5','NFIX'],
        ['IGFBP5','PITHD1'],
        ['IGFBP5','RPL30'],
        ['IGFBP5','SMR3B'],
        ['IGFBP5','VCL'],
        ['IGFBP6','PLEC'],
        ['IGFBP7','NOA1'],
        ['IGFBP7','PRSS2'],
        ['IGFBP7','TRB@'],
        ['IGFBP7-AS1','NPPA'],
        ['IGFBP7-AS1','TNNT2'],
        ['IGH@','IGHA1'],
        ['IGH@','IGHA2'],
        ['IGH@','IGHG1'],
        ['IGH@','IGHG3'],
        ['IGH@','IGHV2-70'],
        ['IGH@','IGHV3-21'],
        ['IGH@','IGK@'],
        ['IGH@','JAK2'],
        ['IGH@','JCHAIN'],
        ['IGH@','KCNK6'],
        ['IGH@','KCNQ1OT1'],
        ['IGH@','KRT13'],
        ['IGH@','KRT19'],
        ['IGH@','KRT4'],
        ['IGH@','LAPTM5'],
        ['IGH@','LCN2'],
        ['IGH@','LCT'],
        ['IGH@','LDHA'],
        ['IGH@','LGALS4'],
        ['IGH@','LINC00221'],
        ['IGH@','LINC00342'],
        ['IGH@','LINC00506'],
        ['IGH@','LINC00926'],
        ['IGH@','LYZ'],
        ['IGH@','MAP3K20'],
        ['IGH@','MEP1B'],
        ['IGH@','MIR100HG'],
        ['IGH@','MMP2'],
        ['IGH@','MUC1'],
        ['IGH@','MUC12'],
        ['IGH@','MUC13'],
        ['IGH@','MUC2'],
        ['IGH@','MUC5AC'],
        ['IGH@','MUC5B'],
        ['IGH@','MUC6'],
        ['IGH@','MUC7'],
        ['IGH@','MXI1'],
        ['IGH@','MXRA7'],
        ['IGH@','MYH9'],
        ['IGH@','MYL2'],
        ['IGH@','NCL'],
        ['IGH@','NDRG1'],
        ['IGH@','NEAT1'],
        ['IGH@','NFKB2'],
        ['IGH@','NPM1'],
        ['IGH@','OLMALINC'],
        ['IGH@','PCSK7'],
        ['IGH@','PGA3'],
        ['IGH@','PGA5'],
        ['IGH@','PGC'],
        ['IGH@','PIGR'],
        ['IGH@','PPIA'],
        ['IGH@','PPP1R12B'],
        ['IGH@','PPP6R2'],
        ['IGH@','PRB1'],
        ['IGH@','PRB2'],
        ['IGH@','PRB3'],
        ['IGH@','PRB4'],
        ['IGH@','PRH1'],
        ['IGH@','PRH1-PRR4'],
        ['IGH@','PRH2'],
        ['IGH@','PRR4'],
        ['IGH@','PRSS2'],
        ['IGH@','PSAP'],
        ['IGH@','PTMA'],
        ['IGH@','PTPRF'],
        ['IGH@','PTRF'],
        ['IGH@','RAP1GAP'],
        ['IGH@','RBM47'],
        ['IGH@','REG4'],
        ['IGH@','RPAP2'],
        ['IGH@','RPS19'],
        ['IGH@','RPS8'],
        ['IGH@','RRBP1'],
        ['IGH@','RTCA-AS1'],
        ['IGH@','S100A8'],
        ['IGH@','SDC1'],
        ['IGH@','SFTPB'],
        ['IGH@','SFTPC'],
        ['IGH@','SH3BGRL3'],
        ['IGH@','SIAE'],
        ['IGH@','SLC25A6'],
        ['IGH@','SMR3B'],
        ['IGH@','SOD2'],
        ['IGH@','SPTBN1'],
        ['IGH@','SRRM2'],
        ['IGH@','SULT1C2'],
        ['IGH@','TAGLN2'],
        ['IGH@','TFF2'],
        ['IGH@','TG'],
        ['IGH@','TMSB4X'],
        ['IGH@','TRA@'],
        ['IGH@','TRB@'],
        ['IGH@','USP48'],
        ['IGH@','VILL'],
        ['IGH@','WDFY4'],
        ['IGH@','Z99755.3'],
        ['IGH@','ZFP36L1'],
        ['IGH@','ZG16B'],
        ['IGHA1','LTF'],
        ['IGHA1','MPO'],
        ['IGHA1','MUC1'],
        ['IGHA1','MUC6'],
        ['IGHA1','MUC7'],
        ['IGHA1','MVP'],
        ['IGHA1','MYO1A'],
        ['IGHA1','PFN1'],
        ['IGHA1','PGA3'],
        ['IGHA1','PGC'],
        ['IGHA1','PIGR'],
        ['IGHA1','PRB1'],
        ['IGHA1','PRB2'],
        ['IGHA1','REG3A'],
        ['IGHA1','SFTPA1'],
        ['IGHA1','SFTPC'],
        ['IGHA1','SMR3B'],
        ['IGHA1','SRRM2'],
        ['IGHA2','MUC6'],
        ['IGHA2','MUC7'],
        ['IGHA2','SMR3B'],
        ['IGHG1','IGLC1'],
        ['IGHG1','LCT'],
        ['IGHG1','LINC01578'],
        ['IGHG1','LIPF'],
        ['IGHG1','MUC7'],
        ['IGHG1','OLFM4'],
        ['IGHG1','PRH1'],
        ['IGHG1','PRH1-PRR4'],
        ['IGHG3','PRH1'],
        ['IGHG3','PRH1-PRR4'],
        ['IGHG3','SFTPB'],
        ['IGHM','JCHAIN'],
        ['IGHM','MUC5AC'],
        ['IGHM','MUC7'],
        ['IGHM','PRH1'],
        ['IGHM','PRH1-PRR4'],
        ['IGHM','PTPRF'],
        ['IGHM','STAT6'],
        ['IGK@','IGL@'],
        ['IGK@','JAK2'],
        ['IGK@','JCHAIN'],
        ['IGK@','KMT2C'],
        ['IGK@','KNL1'],
        ['IGK@','LAPTM5'],
        ['IGK@','LINC00504'],
        ['IGK@','LINC01106'],
        ['IGK@','LINC01123'],
        ['IGK@','LINC02802'],
        ['IGK@','LL22NC03-2H8.5'],
        ['IGK@','LRRK1'],
        ['IGK@','LYZ'],
        ['IGK@','MUC5AC'],
        ['IGK@','MUC6'],
        ['IGK@','MUC7'],
        ['IGK@','NCK2'],
        ['IGK@','NDE1'],
        ['IGK@','OLMALINC'],
        ['IGK@','PGC'],
        ['IGK@','PIGR'],
        ['IGK@','PNMA8A'],
        ['IGK@','PRH1'],
        ['IGK@','PRH1-PRR4'],
        ['IGK@','PRH2'],
        ['IGK@','PRR4'],
        ['IGK@','QSOX1'],
        ['IGK@','RNF152'],
        ['IGK@','RYBP'],
        ['IGK@','SAMHD1'],
        ['IGK@','SCAMP4'],
        ['IGK@','SFTPB'],
        ['IGK@','SLC31A1'],
        ['IGK@','SMR3B'],
        ['IGK@','SOD2'],
        ['IGK@','SRGAP2C'],
        ['IGK@','STAG3'],
        ['IGK@','STATH'],
        ['IGK@','SYNPO2'],
        ['IGK@','TFF1'],
        ['IGK@','TG'],
        ['IGK@','TMEM38B'],
        ['IGK@','TRA@'],
        ['IGKC','ILF3'],
        ['IGKC','JCHAIN'],
        ['IGKC','LAPTM5'],
        ['IGKC','LYZ'],
        ['IGKC','MUC5AC'],
        ['IGKC','MUC6'],
        ['IGKC','MUC7'],
        ['IGKC','NDE1'],
        ['IGKC','PGC'],
        ['IGKC','PIGR'],
        ['IGKC','PRH1'],
        ['IGKC','PRH1-PRR4'],
        ['IGKC','PRR27'],
        ['IGKC','PRR4'],
        ['IGKC','REG3A'],
        ['IGKC','SEPT9'],
        ['IGKC','SMR3B'],
        ['IGKC','TFF1'],
        ['IGKC','TMBIM6'],
        ['IGKC','TMED5'],
        ['IGKV1-12','MUC5AC'],
        ['IGKV1-8','MUC5AC'],
        ['IGKV1-8','NDRG2'],
        ['IGKV1-8','PIGR'],
        ['IGKV1-8','PRH1'],
        ['IGKV1-8','PRH1-PRR4'],
        ['IGKV1-8','PRR4'],
        ['IGKV1-8','TFF1'],
        ['IGKV3-11','LYZ'],
        ['IGKV3-11','PIGR'],
        ['IGKV3-11','PRH1'],
        ['IGKV3-11','PRH1-PRR4'],
        ['IGKV3-11','PRR4'],
        ['IGKV3-11','TFF1'],
        ['IGLC1','SMR3B'],
        ['IGLC2','MUCL3'],
        ['IGLL5','SMR3B'],
        ['IGLV5-52','LL22NC03-80A10.6'],
        ['IGLVIV-64','LL22NC03-23C6.13'],
        ['IGLVIV-65','LL22NC03-23C6.13'],
        ['IGLVIV-66-1','LL22NC03-23C6.13'],
        ['IGSF10','MED12L'],
        ['IGSF5','PCP4'],
        ['IKZF1','LYZ'],
        ['IL11','PTRF'],
        ['IL13RA1','TG'],
        ['IL17RA','LYZ'],
        ['IL18','LITAF'],
        ['IL1R1','MRPS27'],
        ['IL1R1','PNLIP'],
        ['IL1R2','LINC01127'],
        ['IL31RA','PRKCSH'],
        ['IL36A','MRPL11'],
        ['IL3RA','PTRF'],
        ['IL3RA','YIPF5'],
        ['IL6R','LYZ'],
        ['IL6ST','LYZ'],
        ['IL6ST','MON1B'],
        ['IL6ST','PRH1'],
        ['IL6ST','PRH1-PRR4'],
        ['IL6ST','SLAMF7'],
        ['IL6ST','SLC7A11'],
        ['IL6ST','SOX5'],
        ['IL6ST','TRIM2'],
        ['IL6ST','VPS13C'],
        ['IL6ST','ZEB2'],
        ['IL9R','RPL23AP82'],
        ['ILF3','LYZ'],
        ['ILRUN','SPDEF'],
        ['IMMP2L','PIGR'],
        ['IMMP2L','TG'],
        ['IMP3','PTRF'],
        ['IMP3','SNUPN'],
        ['IMPA1','SLC10A5'],
        ['IMPDH1','IMPDH1P4'],
        ['INA','KRT13'],
        ['INA','KRT14'],
        ['INA','KRT16'],
        ['INA','KRT4'],
        ['INA','KRT5'],
        ['INA','KRT6A'],
        ['INE1','UBA1'],
        ['INHBC','INHBE'],
        ['INHBE','LRG1'],
        ['INO80B-WBP1','MYL12B'],
        ['INPP1','MFSD6'],
        ['INPP4B','PLCZ1'],
        ['INPP5A','PWWP2B'],
        ['INPP5B','LYZ'],
        ['INPP5B','NKIRAS2'],
        ['INPP5B','SFTPB'],
        ['INPP5B','WHAMM'],
        ['INPP5D','LYZ'],
        ['INPP5K','MYO1C'],
        ['INPPL1','UBA1'],
        ['INS','TRB@'],
        ['INSL3','JAK3'],
        ['INSR','LITAF'],
        ['INTS1','TG'],
        ['INTS2','MED13'],
        ['INTS3','SLC27A3'],
        ['INTS6-AS1','WDFY2'],
        ['INTS7','LYZ'],
        ['IQCG','POLG2'],
        ['IQGAP1','LYZ'],
        ['IQGAP1','NSMCE1'],
        ['IQGAP1','ZNF774'],
        ['IQSEC1','SFTPC'],
        ['IQUB','NDUFA5'],
        ['IRF1','LYZ'],
        ['IRF1','PLCL1'],
        ['IRS2','KIDINS220'],
        ['ISG20L2','SNAP23'],
        ['ISPD','SOSTDC1'],
        ['IST1','LYZ'],
        ['IST1','TG'],
        ['ITGA2B','LYZ'],
        ['ITGA3','SERINC5'],
        ['ITGA3','SFTPB'],
        ['ITGA4','LINC01934'],
        ['ITGA5','MAZ'],
        ['ITGA5','MYH9'],
        ['ITGA5','PTOV1'],
        ['ITGA5','VPS37B'],
        ['ITGA5','ZFP36L2'],
        ['ITGA6','PRB3'],
        ['ITGA6','PRH2'],
        ['ITGA7','NPPA-AS1'],
        ['ITGA7','PPP1R1A'],
        ['ITGAD','RPL37A'],
        ['ITGAE','NCBP3'],
        ['ITGAL','LYZ'],
        ['ITGAM','LYZ'],
        ['ITGAV','PXK'],
        ['ITGAX','SFTPB'],
        ['ITGB1','MYH11'],
        ['ITGB1','PLEC'],
        ['ITGB1','SMR3B'],
        ['ITGB1','TG'],
        ['ITGB2','PICALM'],
        ['ITGB2','POFUT2'],
        ['ITGB2','SUMO3'],
        ['ITGB4','LYZ'],
        ['ITGB4','SMR3B'],
        ['ITGB6','PPP4R1'],
        ['ITIH1','SERPINA1'],
        ['ITM2B','LINC00534'],
        ['ITM2B','LYZ'],
        ['ITM2B','MAPRE2'],
        ['ITM2B','RB1'],
        ['ITM2B','S100A9'],
        ['ITPKB','SFTPA2'],
        ['ITPKB','SMR3B'],
        ['ITPKC','SNRPA'],
        ['ITPR1','TG'],
        ['ITPR2','STATH'],
        ['IVD','KNSTRN'],
        ['IVD','TG'],
        ['IWS1','NCL'],
        ['IWS1','SDF4'],
        ['IYD','LAMC1'],
        ['IYD','METTL7A'],
        ['IYD','SLA'],
        ['IYD','TPO'],
        ['JADE1','LINC00639'],
        ['JADE1','SOX4'],
        ['JADRR','KCNE3'],
        ['JAG2','NUDT14'],
        ['JAK1','PECAM1'],
        ['JAK2','LITAF'],
        ['JAK2','RCL1'],
        ['JAK2','TRA@'],
        ['JAK2','TRB@'],
        ['JAK3','METTL7A'],
        ['JAM3','LINC02731'],
        ['JARID2','LINC01578'],
        ['JARID2','MGAT4A'],
        ['JAZF1','RNPS1'],
        ['JAZF1','ZNF777'],
        ['JCHAIN','LYZ'],
        ['JCHAIN','PGC'],
        ['JCHAIN','PIGR'],
        ['JCHAIN','RPAIN'],
        ['JKAMP','PCNX4'],
        ['JMJD6','SRSF2'],
        ['JMJD7','TRAK1'],
        ['JUN','SLAIN2'],
        ['JUND','KIAA1683'],
        ['JUND','NPAS1'],
        ['JUND','SERPINE1'],
        ['JUND','SFTPA1'],
        ['JUND','SFTPC'],
        ['JUP','KRT13'],
        ['JUP','KRT14'],
        ['JUP','KRT16'],
        ['JUP','KRT5'],
        ['JUP','LCE1C'],
        ['JUP','MYH11'],
        ['JUP','SFTPB'],
        ['JUP','TMEM99'],
        ['K14','KRT13'],
        ['K14','KRT14'],
        ['K14','KRT16'],
        ['KALRN','STOM'],
        ['KANSL1','LINC02210'],
        ['KANSL1','LRRC37A3'],
        ['KANSL2','LYZ'],
        ['KANSL3','RIPK4'],
        ['KANSL3','SPN'],
        ['KANSL3','TMCO4'],
        ['KARS','SMR3B'],
        ['KAT2B','SGO1-AS1'],
        ['KAT6A','RAB7A'],
        ['KAT8','MYH9'],
        ['KAT8','SNX24'],
        ['KATNAL2','PAX8-AS1'],
        ['KB-1980E6.3','ODF1'],
        ['KCNE1','SMIM34B'],
        ['KCNE2','SMIM11A'],
        ['KCNE2','SMIM11B'],
        ['KCNE3','ZNF346'],
        ['KCNH8','PRKDC'],
        ['KCNIP3','PROM2'],
        ['KCNIP4','UBE2V2'],
        ['KCNJ15','TG'],
        ['KCNK6','SAMHD1'],
        ['KCNK6','TRA@'],
        ['KCNK7','MAP3K11'],
        ['KCNMA1','KRT13'],
        ['KCNMA1','NDE1'],
        ['KCNMA1','SMR3B'],
        ['KCNMB3','LRRFIP1'],
        ['KCNN4','LYPD5'],
        ['KCNN4','MUC7'],
        ['KCNQ1','TG'],
        ['KCNQ1OT1','TRA@'],
        ['KCP','SENP1'],
        ['KDELC1P1','POGLUT2'],
        ['KDELR2','TRB@'],
        ['KDM1B','LYZ'],
        ['KDM2A','LYZ'],
        ['KDM2A','NCALD'],
        ['KDM2A','SF1'],
        ['KDM2B','STOM'],
        ['KDM4B','SAFB2'],
        ['KDM4C','TMEM50B'],
        ['KDM4C','UHRF2'],
        ['KDM5A','PLD5'],
        ['KDM5A','SLC6A13'],
        ['KDM5B','PPP1R12B'],
        ['KDM5C','KDM5C-IT1'],
        ['KDM7A','LAPTM5'],
        ['KDSR','STAG1'],
        ['KHDC4','SPG7'],
        ['KHDRBS1','KIAA0319L'],
        ['KHK','PDZK1IP1'],
        ['KIAA0020','SSR3'],
        ['KIAA0196','VGLL4'],
        ['KIAA0319L','LYZ'],
        ['KIAA0355','TRB@'],
        ['KIAA0368','N4BP2L2'],
        ['KIAA0391','NSD1'],
        ['KIAA0430','PAX8-AS1'],
        ['KIAA0513','ORAI2'],
        ['KIAA0586','TOMM20L'],
        ['KIAA0753','MED31'],
        ['KIAA0753','PITPNM3'],
        ['KIAA0754','MACF1'],
        ['KIAA1024','TMED3'],
        ['KIAA1109','LYZ'],
        ['KIAA1147','WEE2-AS1'],
        ['KIAA1191','LYZ'],
        ['KIAA1191','PRKCSH'],
        ['KIAA1217','PPFIA1'],
        ['KIAA1217','PRB3'],
        ['KIAA1217','SMR3B'],
        ['KIAA1217','TG'],
        ['KIAA1324','PAGE4'],
        ['KIAA1324','PRH1'],
        ['KIAA1324','PRH1-PRR4'],
        ['KIAA1324','PRR4'],
        ['KIAA1324','PRSS1'],
        ['KIAA1324','TRB@'],
        ['KIAA1462','SFTPB'],
        ['KIAA1522','SMR3B'],
        ['KIAA1671','TG'],
        ['KIAA1731','SCARNA9'],
        ['KIAA1841','PEX13'],
        ['KIAA2012','PEX26'],
        ['KIAA2026','UHRF2'],
        ['KIDINS220','PTH'],
        ['KIDINS220','TG'],
        ['KIF13A','RPL8'],
        ['KIF13B','SMR3B'],
        ['KIF15','ZNF501'],
        ['KIF1C','KRT13'],
        ['KIF1C','SFTPB'],
        ['KIF1C','SMR3B'],
        ['KIF22','MCM3AP'],
        ['KIF26B','PIGR'],
        ['KIF2A','NDUFS4'],
        ['KIF5A','PCNT'],
        ['KIFAP3','TG'],
        ['KIFC1','PHF1'],
        ['KIFC1','RNH1'],
        ['KIFC3','TG'],
        ['KIRREL3-AS2','MORF4L1'],
        ['KIT','TG'],
        ['KLC1','SMR3B'],
        ['KLF10','PTMA'],
        ['KLF16','LLFOS-48D6.2'],
        ['KLF16','OAZ1'],
        ['KLF16','PTBP1'],
        ['KLF16','REXO1'],
        ['KLF2','LOXL1'],
        ['KLF2','MAZ'],
        ['KLF2','OAZ1'],
        ['KLF3','PNLIP'],
        ['KLF4','LINC01509'],
        ['KLF6','METAP1'],
        ['KLF6','NAMPT'],
        ['KLF6','PDK1'],
        ['KLF6','PRLHR'],
        ['KLF6','TG'],
        ['KLF7P1','U1'],
        ['KLHL23','SSB'],
        ['KLHL33','TEP1'],
        ['KLHL36','LINC02680'],
        ['KLHL36','ZDHHC7'],
        ['KLHL42','ZSWIM8'],
        ['KLK1','PRSS1'],
        ['KLK1','PRSS2'],
        ['KLK1','PRSS3'],
        ['KLK1','SMR3B'],
        ['KLK1','TRB@'],
        ['KLK11','PRSS30P'],
        ['KLK2','NDE1'],
        ['KLK2','STAMBPL1'],
        ['KLK2','THEM4'],
        ['KLK3','LLFOS-48D6.2'],
        ['KLK3','MALAT1'],
        ['KLK3','MYLK'],
        ['KLK3','NR4A1'],
        ['KLK3','PLPP3'],
        ['KLK4','KLKP1'],
        ['KLK7','KLK8'],
        ['KLRA1P','TRA@'],
        ['KLRAP1','MAGOHB'],
        ['KMT2C','LYZ'],
        ['KMT2C','TPTE'],
        ['KMT2D','S100A8'],
        ['KMT2E','SRSF7'],
        ['KMT5A','KRT1'],
        ['KMT5A','RFLNA'],
        ['KMT5A','ZNF664'],
        ['KNL1','PIGR'],
        ['KNOP1P1','TRIM69'],
        ['KPNA1','PARP9'],
        ['KRAS','SOX5'],
        ['KRBOX1','ZNF662'],
        ['KRBOX4','ZNF674'],
        ['KREMEN1','LYZ'],
        ['KRIT1','LRRD1'],
        ['KRT1','KRT10'],
        ['KRT1','KRT12'],
        ['KRT1','KRT13'],
        ['KRT1','KRT14'],
        ['KRT1','KRT15'],
        ['KRT1','KRT16'],
        ['KRT1','KRT17'],
        ['KRT1','KRT18'],
        ['KRT1','KRT19'],
        ['KRT1','KRT1B'],
        ['KRT1','KRT2'],
        ['KRT1','KRT24'],
        ['KRT1','KRT3'],
        ['KRT1','KRT4'],
        ['KRT1','KRT5'],
        ['KRT1','KRT6A'],
        ['KRT1','KRT6B'],
        ['KRT1','KRT6C'],
        ['KRT1','KRT7'],
        ['KRT1','KRT71'],
        ['KRT1','KRT72'],
        ['KRT1','KRT73'],
        ['KRT1','KRT73-AS1'],
        ['KRT1','KRT74'],
        ['KRT1','KRT75'],
        ['KRT1','KRT76'],
        ['KRT1','KRT77'],
        ['KRT1','KRT78'],
        ['KRT1','KRT79'],
        ['KRT1','KRT8'],
        ['KRT1','KRT80'],
        ['KRT1','KRT81'],
        ['KRT1','KRT82'],
        ['KRT1','KRT83'],
        ['KRT1','KRT84'],
        ['KRT1','KRT85'],
        ['KRT1','KRT86'],
        ['KRT1','KRT9'],
        ['KRT1','KRTHB1'],
        ['KRT1','KRTHB2'],
        ['KRT1','KRTHB6'],
        ['KRT1','LYPD3'],
        ['KRT1','MACF1'],
        ['KRT1','MYB'],
        ['KRT1','MYH9'],
        ['KRT1','NEFL'],
        ['KRT1','PCGF3'],
        ['KRT1','PERP'],
        ['KRT1','PITPNM3'],
        ['KRT1','POLR2A'],
        ['KRT1','PRPH'],
        ['KRT1','PTPRF'],
        ['KRT1','SBSN'],
        ['KRT1','SRGAP2'],
        ['KRT1','TM6SF1'],
        ['KRT1','TMEM99'],
        ['KRT1','TTBK2'],
        ['KRT1','VIM'],
        ['KRT1','ZFP36L1'],
        ['KRT10','KRT13'],
        ['KRT10','KRT14'],
        ['KRT10','KRT15'],
        ['KRT10','KRT16'],
        ['KRT10','KRT4'],
        ['KRT10','KRT5'],
        ['KRT10','KRT6A'],
        ['KRT10','PSORS1C1'],
        ['KRT10','PVRL4'],
        ['KRT10','VIM-AS1'],
        ['KRT10','WASF2'],
        ['KRT12','KRT13'],
        ['KRT12','KRT14'],
        ['KRT12','KRT16'],
        ['KRT12','KRT4'],
        ['KRT121P','KRT13'],
        ['KRT121P','KRT4'],
        ['KRT121P','KRT5'],
        ['KRT121P','KRT6A'],
        ['KRT128P','KRT2'],
        ['KRT128P','KRT73'],
        ['KRT13','KRT14'],
        ['KRT13','KRT15'],
        ['KRT13','KRT16'],
        ['KRT13','KRT17'],
        ['KRT13','KRT18'],
        ['KRT13','KRT19'],
        ['KRT13','KRT2'],
        ['KRT13','KRT20'],
        ['KRT13','KRT23'],
        ['KRT13','KRT24'],
        ['KRT13','KRT25'],
        ['KRT13','KRT26'],
        ['KRT13','KRT27'],
        ['KRT13','KRT28'],
        ['KRT13','KRT3'],
        ['KRT13','KRT31'],
        ['KRT13','KRT32'],
        ['KRT13','KRT33A'],
        ['KRT13','KRT33B'],
        ['KRT13','KRT34'],
        ['KRT13','KRT35'],
        ['KRT13','KRT36'],
        ['KRT13','KRT37'],
        ['KRT13','KRT38'],
        ['KRT13','KRT39'],
        ['KRT13','KRT4'],
        ['KRT13','KRT40'],
        ['KRT13','KRT5'],
        ['KRT13','KRT6B'],
        ['KRT13','KRT6C'],
        ['KRT13','KRT7'],
        ['KRT13','KRT73'],
        ['KRT13','KRT75'],
        ['KRT13','KRT76'],
        ['KRT13','KRT77'],
        ['KRT13','KRT78'],
        ['KRT13','KRT79'],
        ['KRT13','KRT8'],
        ['KRT13','KRT83'],
        ['KRT13','KRT85'],
        ['KRT13','KRT9'],
        ['KRT13','KRTHA7'],
        ['KRT13','KRTHB1'],
        ['KRT13','MCAM'],
        ['KRT13','MGLL'],
        ['KRT13','MINK1'],
        ['KRT13','MUC21'],
        ['KRT13','MUC4'],
        ['KRT13','MYH11'],
        ['KRT13','MYLK'],
        ['KRT13','NDE1'],
        ['KRT13','NEFL'],
        ['KRT13','NEFM'],
        ['KRT13','PALLD'],
        ['KRT13','PARD3'],
        ['KRT13','PARP9'],
        ['KRT13','PFKL'],
        ['KRT13','PKP1'],
        ['KRT13','PPP1CB'],
        ['KRT13','PPP2R1A'],
        ['KRT13','PRMT2'],
        ['KRT13','PRPH'],
        ['KRT13','PTPRF'],
        ['KRT13','RGS12'],
        ['KRT13','RHCG'],
        ['KRT13','S100A14'],
        ['KRT13','S100A8'],
        ['KRT13','S100A9'],
        ['KRT13','SCP2'],
        ['KRT13','SF3B2'],
        ['KRT13','SFN'],
        ['KRT13','SH3PXD2A'],
        ['KRT13','SNX32'],
        ['KRT13','SORBS1'],
        ['KRT13','SORT1'],
        ['KRT13','SPARCL1'],
        ['KRT13','SPRR1B'],
        ['KRT13','SREBF2'],
        ['KRT13','SRPK2'],
        ['KRT13','TACSTD2'],
        ['KRT13','TAGLN'],
        ['KRT13','TBCD'],
        ['KRT13','TGFB1'],
        ['KRT13','TIMP2'],
        ['KRT13','TMBIM1'],
        ['KRT13','TNS1'],
        ['KRT13','TPM2'],
        ['KRT13','TPM4'],
        ['KRT13','TRIM29'],
        ['KRT13','TYPE'],
        ['KRT13','VIM'],
        ['KRT13','YWHAE'],
        ['KRT13','ZBTB16'],
        ['KRT13','ZBTB4'],
        ['KRT14','KRT15'],
        ['KRT14','KRT16'],
        ['KRT14','KRT17'],
        ['KRT14','KRT18'],
        ['KRT14','KRT19'],
        ['KRT14','KRT2'],
        ['KRT14','KRT20'],
        ['KRT14','KRT23'],
        ['KRT14','KRT24'],
        ['KRT14','KRT25'],
        ['KRT14','KRT26'],
        ['KRT14','KRT27'],
        ['KRT14','KRT28'],
        ['KRT14','KRT31'],
        ['KRT14','KRT32'],
        ['KRT14','KRT33A'],
        ['KRT14','KRT33B'],
        ['KRT14','KRT34'],
        ['KRT14','KRT35'],
        ['KRT14','KRT36'],
        ['KRT14','KRT37'],
        ['KRT14','KRT38'],
        ['KRT14','KRT39'],
        ['KRT14','KRT4'],
        ['KRT14','KRT40'],
        ['KRT14','KRT5'],
        ['KRT14','KRT7'],
        ['KRT14','KRT77'],
        ['KRT14','KRT8'],
        ['KRT14','KRT9'],
        ['KRT14','KRTHA7'],
        ['KRT14','MYH9'],
        ['KRT14','NEAT1'],
        ['KRT14','NEFM'],
        ['KRT14','PRPH'],
        ['KRT14','TMEM99'],
        ['KRT14','TYPE'],
        ['KRT14','VIM'],
        ['KRT15','KRT16'],
        ['KRT15','KRT19'],
        ['KRT15','KRT4'],
        ['KRT15','KRT5'],
        ['KRT15','TMEM99'],
        ['KRT16','KRT17'],
        ['KRT16','KRT18'],
        ['KRT16','KRT19'],
        ['KRT16','KRT2'],
        ['KRT16','KRT20'],
        ['KRT16','KRT23'],
        ['KRT16','KRT24'],
        ['KRT16','KRT25'],
        ['KRT16','KRT26'],
        ['KRT16','KRT27'],
        ['KRT16','KRT28'],
        ['KRT16','KRT3'],
        ['KRT16','KRT31'],
        ['KRT16','KRT32'],
        ['KRT16','KRT33A'],
        ['KRT16','KRT33B'],
        ['KRT16','KRT34'],
        ['KRT16','KRT35'],
        ['KRT16','KRT36'],
        ['KRT16','KRT37'],
        ['KRT16','KRT38'],
        ['KRT16','KRT39'],
        ['KRT16','KRT4'],
        ['KRT16','KRT40'],
        ['KRT16','KRT5'],
        ['KRT16','KRT6B'],
        ['KRT16','KRT6C'],
        ['KRT16','KRT7'],
        ['KRT16','KRT75'],
        ['KRT16','KRT76'],
        ['KRT16','KRT77'],
        ['KRT16','KRT8'],
        ['KRT16','KRT9'],
        ['KRT16','KRTHA7'],
        ['KRT16','NEFL'],
        ['KRT16','NEFM'],
        ['KRT16','PRPH'],
        ['KRT16','VIM'],
        ['KRT17','KRT4'],
        ['KRT17','KRT5'],
        ['KRT18','KRT4'],
        ['KRT18','KRT5'],
        ['KRT18P59','PKNOX2'],
        ['KRT19','KRT4'],
        ['KRT19','LYZ'],
        ['KRT19','MFN2'],
        ['KRT19','MUC5AC'],
        ['KRT19','TOP1'],
        ['KRT1B','KRT4'],
        ['KRT1B','KRT5'],
        ['KRT1B','KRT6A'],
        ['KRT2','KRT4'],
        ['KRT2','KRT5'],
        ['KRT2','KRT6A'],
        ['KRT2','KRT77'],
        ['KRT2','SRSF5'],
        ['KRT2','TMEM99'],
        ['KRT20','PIGR'],
        ['KRT222','KRT5'],
        ['KRT23','KRT5'],
        ['KRT23','SMR3B'],
        ['KRT24','KRT5'],
        ['KRT25','KRT5'],
        ['KRT26','KRT5'],
        ['KRT27','KRT5'],
        ['KRT28','KRT5'],
        ['KRT3','KRT4'],
        ['KRT3','KRT5'],
        ['KRT3','KRT6A'],
        ['KRT35','KRT5'],
        ['KRT37','KRT4'],
        ['KRT37','KRT5'],
        ['KRT38','KRT5'],
        ['KRT4','KRT5'],
        ['KRT4','KRT6A'],
        ['KRT4','KRT6B'],
        ['KRT4','KRT6C'],
        ['KRT4','KRT7'],
        ['KRT4','KRT71'],
        ['KRT4','KRT72'],
        ['KRT4','KRT73'],
        ['KRT4','KRT74'],
        ['KRT4','KRT75'],
        ['KRT4','KRT76'],
        ['KRT4','KRT77'],
        ['KRT4','KRT78'],
        ['KRT4','KRT79'],
        ['KRT4','KRT8'],
        ['KRT4','KRT80'],
        ['KRT4','KRT81'],
        ['KRT4','KRT82'],
        ['KRT4','KRT83'],
        ['KRT4','KRT84'],
        ['KRT4','KRT85'],
        ['KRT4','KRT86'],
        ['KRT4','KRT9'],
        ['KRT4','KRTHA7'],
        ['KRT4','KRTHB1'],
        ['KRT4','KRTHB2'],
        ['KRT4','KRTHB6'],
        ['KRT4','LMNA'],
        ['KRT4','LMNB2'],
        ['KRT4','LY6D'],
        ['KRT4','MCM7'],
        ['KRT4','MYH11'],
        ['KRT4','MYH9'],
        ['KRT4','MYLK'],
        ['KRT4','MYO5B'],
        ['KRT4','NEFL'],
        ['KRT4','NEFM'],
        ['KRT4','NPEPPS'],
        ['KRT4','PERP'],
        ['KRT4','PPP1R12B'],
        ['KRT4','PRPH'],
        ['KRT4','RPS11'],
        ['KRT4','RPS25'],
        ['KRT4','S100A14'],
        ['KRT4','SORBS1'],
        ['KRT4','SPRR3'],
        ['KRT4','SUN2'],
        ['KRT4','SVIL'],
        ['KRT4','SYN3'],
        ['KRT4','SYNPO2'],
        ['KRT4','TBC1D14'],
        ['KRT4','TGM3'],
        ['KRT4','TNFAIP2'],
        ['KRT4','TRIM29'],
        ['KRT4','TTBK2'],
        ['KRT4','VIM'],
        ['KRT4','ZMIZ1'],
        ['KRT40','KRT5'],
        ['KRT42P','USP6'],
        ['KRT5','KRT6A'],
        ['KRT5','KRT6B'],
        ['KRT5','KRT6C'],
        ['KRT5','KRT7'],
        ['KRT5','KRT71'],
        ['KRT5','KRT72'],
        ['KRT5','KRT73'],
        ['KRT5','KRT74'],
        ['KRT5','KRT75'],
        ['KRT5','KRT76'],
        ['KRT5','KRT77'],
        ['KRT5','KRT78'],
        ['KRT5','KRT79'],
        ['KRT5','KRT8'],
        ['KRT5','KRT80'],
        ['KRT5','KRT81'],
        ['KRT5','KRT82'],
        ['KRT5','KRT83'],
        ['KRT5','KRT84'],
        ['KRT5','KRT85'],
        ['KRT5','KRT86'],
        ['KRT5','KRT9'],
        ['KRT5','KRTHA7'],
        ['KRT5','KRTHB1'],
        ['KRT5','KRTHB2'],
        ['KRT5','KRTHB6'],
        ['KRT5','LMNA'],
        ['KRT5','LMNB2'],
        ['KRT5','LYPD3'],
        ['KRT5','MYH9'],
        ['KRT5','NEFH'],
        ['KRT5','NEFL'],
        ['KRT5','NEFM'],
        ['KRT5','PRPH'],
        ['KRT5','TMEM99'],
        ['KRT5','TTBK2'],
        ['KRT5','VIM'],
        ['KRT6A','KRT6B'],
        ['KRT6A','KRT6C'],
        ['KRT6A','KRT7'],
        ['KRT6A','KRT71'],
        ['KRT6A','KRT72'],
        ['KRT6A','KRT73'],
        ['KRT6A','KRT74'],
        ['KRT6A','KRT75'],
        ['KRT6A','KRT76'],
        ['KRT6A','KRT77'],
        ['KRT6A','KRT78'],
        ['KRT6A','KRT79'],
        ['KRT6A','KRT8'],
        ['KRT6A','KRT80'],
        ['KRT6A','KRT81'],
        ['KRT6A','KRT82'],
        ['KRT6A','KRT83'],
        ['KRT6A','KRT84'],
        ['KRT6A','KRT85'],
        ['KRT6A','KRT86'],
        ['KRT6A','KRT9'],
        ['KRT6A','KRTHB1'],
        ['KRT6A','KRTHB2'],
        ['KRT6A','MYH11'],
        ['KRT6A','NEFL'],
        ['KRT6A','PPL'],
        ['KRT6A','PRPH'],
        ['KRT6A','RHCG'],
        ['KRT6A','TTBK2'],
        ['KRT6A','VIM'],
        ['KRT6B','MYH11'],
        ['KRT7','KRT8'],
        ['KRT7','P2RX2'],
        ['KRT7','SFTPB'],
        ['KRT7','TG'],
        ['KRT72','KRT73'],
        ['KRT78','MYH11'],
        ['KRT78','TNNT1'],
        ['KRT8','LYZ'],
        ['KRT8','MALAT1'],
        ['KRT8','MUC7'],
        ['KRT8','PIGR'],
        ['KRT8','RAB5C'],
        ['KRT8','VIM'],
        ['KRTAP5-8','NADSYN1'],
        ['KSR1','LGALS9'],
        ['KSR1','TRB@'],
        ['KTN1','TG'],
        ['KXD1','UBA52'],
        ['KYNU','ZNF528-AS1'],
        ['L2HGDH','SOS2'],
        ['LACC1','LINC00284'],
        ['LAD1','TNNI1'],
        ['LAIR1','LYZ'],
        ['LAMA1','TMPO'],
        ['LAMA3','RBBP8'],
        ['LAMA5','PNLIP'],
        ['LAMB1','TG'],
        ['LAMB2','TG'],
        ['LAMC1','PPP1R14B'],
        ['LAMP1','MAMDC2'],
        ['LAMP2','LYZ'],
        ['LAMP2','PRRG3'],
        ['LAMP3','SFTPA1'],
        ['LAMP3','SFTPB'],
        ['LAMP3','SLC22A5'],
        ['LAMTOR1','NUMA1'],
        ['LAPTM4A','PUM2'],
        ['LAPTM4B','SFTPC'],
        ['LAPTM5','RIPOR1'],
        ['LAPTM5','SFTPA2'],
        ['LARP1','MUC7'],
        ['LARP1','NR3C1'],
        ['LARP1','SMR3B'],
        ['LARP1','TRB@'],
        ['LARS','MUC7'],
        ['LARS','NDUFC2'],
        ['LARS','PLAC8L1'],
        ['LARS','TG'],
        ['LASP1','TAGLN'],
        ['LATS1','LYZ'],
        ['LATS1','NUP43'],
        ['LAX1','TRA@'],
        ['LBH','LINC01936'],
        ['LBP','RPN2'],
        ['LBR','LYZ'],
        ['LCAT','PSMB10'],
        ['LCLAT1','MLIP'],
        ['LCP1','LYZ'],
        ['LCP1','SRGN'],
        ['LDAH','PUM2'],
        ['LDB3','SAMD4B'],
        ['LDHB','TG'],
        ['LDLR','LYZ'],
        ['LDLR','TAPBP'],
        ['LDLR','TNRC18'],
        ['LENG8','PAX8-AS1'],
        ['LENG8','PRSS1'],
        ['LEP','SPTBN1'],
        ['LEPRE1','TRNP1'],
        ['LEPREL1','NCOR2'],
        ['LEPREL4','SCUBE3'],
        ['LEPROT','THOC2'],
        ['LEPROTL1','LYZ'],
        ['LGALS1','PDXP'],
        ['LGALS8','SFTPB'],
        ['LGMN','MYH11'],
        ['LGR4','LIN7C'],
        ['LHFPL3','METAZOA_SRP'],
        ['LHFPL3','RN7SL8P'],
        ['LHFPL4','NAP1L4'],
        ['LHFPL5','PNLIP'],
        ['LHFPL5','TRB@'],
        ['LIFR','SMR3B'],
        ['LIFR-AS1','SPINK1'],
        ['LIG3','ZNF830'],
        ['LILRA1','LILRP2'],
        ['LILRB2','LILRP2'],
        ['LIMCH1','SFTPB'],
        ['LIMD1','PKN1'],
        ['LIMD1','PTCSC2'],
        ['LIMK2','LYZ'],
        ['LIMS1','LYZ'],
        ['LIN54','SEC31A'],
        ['LIN54','THAP9-AS1'],
        ['LINC-PINT','TG'],
        ['LINC00174','RABGEF1'],
        ['LINC00174','SKP1'],
        ['LINC00202-1','LINC00264'],
        ['LINC00202-1','LINC00614'],
        ['LINC00266-1','LINC01881'],
        ['LINC00271','PDE7B'],
        ['LINC00309','PELI1'],
        ['LINC00354','SOX1-OT'],
        ['LINC00378','LINC01442'],
        ['LINC00452','LINC00453'],
        ['LINC00467','TRAF5'],
        ['LINC00472','RALGPS2'],
        ['LINC00491','LINC01807'],
        ['LINC00494','LINC01522'],
        ['LINC00504','TRA@'],
        ['LINC00521','OTUB2'],
        ['LINC00535','UBE2V2'],
        ['LINC00545','SLIRP'],
        ['LINC00665','LINC01535'],
        ['LINC00665','TRG@'],
        ['LINC00665','ZNF790-AS1'],
        ['LINC00670','MYOCD'],
        ['LINC00680','WDR70'],
        ['LINC00694','ZNF445'],
        ['LINC00707','LINP1'],
        ['LINC00840','LINC00841'],
        ['LINC00843','PARG'],
        ['LINC00843','TIMM23'],
        ['LINC00861','LYZ'],
        ['LINC00879','RPS18P6'],
        ['LINC00886','TIPARP-AS1'],
        ['LINC00893','LINC00894'],
        ['LINC00950','TMEM8B'],
        ['LINC00963','LINC02511'],
        ['LINC00963','RGMA'],
        ['LINC00963','SLC12A6'],
        ['LINC00964','ZNF572'],
        ['LINC00969','LYZ'],
        ['LINC00969','PYGL'],
        ['LINC00969','SDHAP3'],
        ['LINC00989','LINC01088'],
        ['LINC00994','PSMD6'],
        ['LINC01068','NDFIP2'],
        ['LINC01087','POTEM'],
        ['LINC01091','LINC02435'],
        ['LINC01115','TMEM18'],
        ['LINC01119','SOCS5'],
        ['LINC01135','LINC01358'],
        ['LINC01169','SMAD6'],
        ['LINC01186','TMEM178B'],
        ['LINC01198','TPT1-AS1'],
        ['LINC01251','TRB@'],
        ['LINC01251','UBE2R2-AS1'],
        ['LINC01278','MUC7'],
        ['LINC01281','LINC01282'],
        ['LINC01291','POLE4'],
        ['LINC01297','POTEM'],
        ['LINC01317','LYZ'],
        ['LINC01324','PEX5L'],
        ['LINC01347','NUPR2'],
        ['LINC01358','TMBIM6'],
        ['LINC01372','MEF2C'],
        ['LINC01378','SCD5'],
        ['LINC01422','Z97353.2'],
        ['LINC01470','NEDD1'],
        ['LINC01492','ST6GALNAC2'],
        ['LINC01496','NUDT11'],
        ['LINC01498','SART3'],
        ['LINC01513','ROPN1L'],
        ['LINC01518','ZNF33B'],
        ['LINC01524','ZNF281'],
        ['LINC01537','TMPRSS11B'],
        ['LINC01578','YPEL5'],
        ['LINC01578','ZBTB7A'],
        ['LINC01595','SULT2A1'],
        ['LINC01608','LINC01609'],
        ['LINC01619','LINC02391'],
        ['LINC01641','RPS18'],
        ['LINC01649','SLC22A15'],
        ['LINC01655','LINC02257'],
        ['LINC01705','LINC02257'],
        ['LINC01719','PDE4DIP'],
        ['LINC01740','NENF'],
        ['LINC01761','ZNF528-AS1'],
        ['LINC01764','ZNF861P'],
        ['LINC01765','NFIX'],
        ['LINC01808','SCARA5'],
        ['LINC01823','TSN'],
        ['LINC01859','ZNF728'],
        ['LINC01872','SIGLECL1'],
        ['LINC01962','TRIM7'],
        ['LINC01979','TBC1D16'],
        ['LINC02009','LTF'],
        ['LINC02019','MAPKAPK3'],
        ['LINC02023','SI'],
        ['LINC02027','LINC02050'],
        ['LINC02087','LINC02088'],
        ['LINC02185','LINC02186'],
        ['LINC02193','RPL23AP82'],
        ['LINC02219','PIK3R1'],
        ['LINC02245','PITPNM3'],
        ['LINC02246','NRIP1'],
        ['LINC02345','TMCO5A'],
        ['LINC02558','SLC5A1'],
        ['LINC02582','LINC02700'],
        ['LINC02583','RHOQ'],
        ['LINC02583','TMEM247'],
        ['LINC02624','LYZ'],
        ['LINC02649','PFKFB3'],
        ['LINC02687','MS4A10'],
        ['LINC02743','SPATA19'],
        ['LINC02749','RRM1'],
        ['LINC02785','VAV3-AS1'],
        ['LINC02857','MRAP2'],
        ['LINCADL','LVRN'],
        ['LINCR-0001','MSRA'],
        ['LIPA','LYZ'],
        ['LIPF','LYZ'],
        ['LIPF','PIGR'],
        ['LIPF','RPL3'],
        ['LITAF','SMR3B'],
        ['LITAF','TMBIM6'],
        ['LIX1L','PTRF'],
        ['LIX1L','UGT8'],
        ['LIX1L-AS1','POLR3GL'],
        ['LL0XNC01-16G2.1','PNMA3'],
        ['LLFOS-48D6.2','PTMA'],
        ['LLFOS-48D6.2','SGTA'],
        ['LLGL1','MYO1C'],
        ['LLGL2','LYZ'],
        ['LLPH','LYZ'],
        ['LLPH','Z83840.1'],
        ['LMAN1L','TMOD3'],
        ['LMAN2','MXD3'],
        ['LMCD1-AS1','SSUH2'],
        ['LMF1','SMR3B'],
        ['LMNA','PNLIP'],
        ['LMNA','SLC10A3'],
        ['LMNA','TLN1'],
        ['LMNA','WIZ'],
        ['LMNTD1','TRIM36'],
        ['LMO1','RIC3'],
        ['LMO7','SFTPC'],
        ['LMO7','TG'],
        ['LMO7','UCHL3'],
        ['LMOD2','NPPA-AS1'],
        ['LMX1B','PAPOLG'],
        ['LNP1','TMEM45A'],
        ['LNPK','NFIA'],
        ['LNX2','MTHFR'],
        ['LONP1','SCAF1'],
        ['LONRF2','RAPH1'],
        ['LOX','SH3GL1'],
        ['LOXL1','MAFF'],
        ['LOXL1','MARVELD1'],
        ['LOXL1','ZFP36L2'],
        ['LOXL2','PKN1'],
        ['LOXL2','PTRF'],
        ['LPAR2','PBX4'],
        ['LPCAT1','SLC6A3'],
        ['LPCAT2','MMP2'],
        ['LPCAT3','LRG1'],
        ['LPCAT3','PHB2'],
        ['LPGAT1','RPS6KC1'],
        ['LPGAT1','TG'],
        ['LPHN1','SUZ12P'],
        ['LPL','VIM'],
        ['LPL','ZFP36L1'],
        ['LPO','LYZ'],
        ['LPO','PRH1'],
        ['LPO','PRH1-PRR4'],
        ['LPO','PRH2'],
        ['LPO','PRR4'],
        ['LPO','SMR3B'],
        ['LPP','MALAT1'],
        ['LPP','MYH11'],
        ['LPP','RNF128'],
        ['LPP','STXBP4'],
        ['LPP','SUGCT'],
        ['LRBA','TG'],
        ['LRCH3','PTRF'],
        ['LRG1','LYZ'],
        ['LRG1','SAA2'],
        ['LRG1','SS18'],
        ['LRG1','TAT'],
        ['LRIG1','PRSS1'],
        ['LRIG1','PRSS2'],
        ['LRIG1','TRB@'],
        ['LRIG2','SORL1'],
        ['LRIG2','ZNF106'],
        ['LRP1','PECAM1'],
        ['LRP10','LYZ'],
        ['LRP10','REM2'],
        ['LRP11','TTYH3'],
        ['LRP12','PKP4'],
        ['LRP2','TG'],
        ['LRP2BP','UFSP2'],
        ['LRP5','LRP5L'],
        ['LRP5','NPPA'],
        ['LRP5','TG'],
        ['LRPAP1','WDR1'],
        ['LRPPRC','PNLIP'],
        ['LRRC14B','PLEKHG4B'],
        ['LRRC37A15P','MANBA'],
        ['LRRC37A16P','LRRC37B'],
        ['LRRC37A3','NSF'],
        ['LRRC37B','SMURF2'],
        ['LRRC49','PIGR'],
        ['LRRC52-AS1','RXRG'],
        ['LRRC56','QRSL1'],
        ['LRRC57','PLEKHA1'],
        ['LRRC59','MRPL27'],
        ['LRRC75A','METTL7A'],
        ['LRRC75A-AS1','LYZ'],
        ['LRRC75A-AS1','SYN3'],
        ['LRRC8B','LRRC8C'],
        ['LRRFIP1','LYZ'],
        ['LRRFIP1','PCBD2'],
        ['LRRFIP1','SMR3B'],
        ['LRRK1','OLMALINC'],
        ['LRRK2','MUC19'],
        ['LSP1','TNNT3'],
        ['LTA','NFKBIL1'],
        ['LTBP2','PRH2'],
        ['LTBR','SPEF2'],
        ['LTF','LYZ'],
        ['LTF','NFKBIZ'],
        ['LTF','RBM25'],
        ['LTF','SMR3B'],
        ['LTF','STATH'],
        ['LUCAT1','PIGL'],
        ['LY6K','THEM6'],
        ['LYG1','TXNDC9'],
        ['LYNX1','SLURP2'],
        ['LYPLA1','TCEA1'],
        ['LYPLAL1','SMYD3'],
        ['LYPLAL1-AS1','MYO1D'],
        ['LYRM7','LYZ'],
        ['LYRM7','NUCKS1'],
        ['LYSMD1','SEMA6C'],
        ['LYST','LYZ'],
        ['LYST','SETX'],
        ['LYZ','MAFF'],
        ['LYZ','MAGED1'],
        ['LYZ','MAP3K5'],
        ['LYZ','MARF1'],
        ['LYZ','MAU2'],
        ['LYZ','MCM4'],
        ['LYZ','MEI1'],
        ['LYZ','MKNK2'],
        ['LYZ','MLLT6'],
        ['LYZ','MLXIP'],
        ['LYZ','MOGS'],
        ['LYZ','MON1B'],
        ['LYZ','MORF4L1'],
        ['LYZ','MROH1'],
        ['LYZ','MUC20-OT1'],
        ['LYZ','MUC7'],
        ['LYZ','MYH11'],
        ['LYZ','MYH9'],
        ['LYZ','MYO18A'],
        ['LYZ','MYO1A'],
        ['LYZ','MYO1F'],
        ['LYZ','NARF'],
        ['LYZ','NCL'],
        ['LYZ','NCOA6'],
        ['LYZ','NDRG2'],
        ['LYZ','NEAT1'],
        ['LYZ','NEBL'],
        ['LYZ','NF1'],
        ['LYZ','NFATC2IP'],
        ['LYZ','NFKBIB'],
        ['LYZ','NFKBIZ'],
        ['LYZ','NHSL2'],
        ['LYZ','NIPAL2'],
        ['LYZ','NMD3'],
        ['LYZ','NONO'],
        ['LYZ','NPC1'],
        ['LYZ','NRAS'],
        ['LYZ','NT5C2'],
        ['LYZ','NXN'],
        ['LYZ','OBSL1'],
        ['LYZ','OPA1'],
        ['LYZ','ORAI2'],
        ['LYZ','OSBPL2'],
        ['LYZ','OSBPL9'],
        ['LYZ','OTUD5'],
        ['LYZ','P4HB'],
        ['LYZ','PABPC1'],
        ['LYZ','PAM'],
        ['LYZ','PARP8'],
        ['LYZ','PAXBP1-AS1'],
        ['LYZ','PCBP1-AS1'],
        ['LYZ','PCBP2'],
        ['LYZ','PCDHA1'],
        ['LYZ','PCDHA10'],
        ['LYZ','PCDHA11'],
        ['LYZ','PCDHA12'],
        ['LYZ','PCDHA13'],
        ['LYZ','PCDHA2'],
        ['LYZ','PCDHA3'],
        ['LYZ','PCDHA4'],
        ['LYZ','PCDHA5'],
        ['LYZ','PCDHA7'],
        ['LYZ','PCDHA8'],
        ['LYZ','PCDHA9'],
        ['LYZ','PCDHAC1'],
        ['LYZ','PCDHAC2'],
        ['LYZ','PCYOX1'],
        ['LYZ','PDLIM5'],
        ['LYZ','PET100'],
        ['LYZ','PGC'],
        ['LYZ','PHACTR4'],
        ['LYZ','PHF2'],
        ['LYZ','PIGQ'],
        ['LYZ','PIK3R2'],
        ['LYZ','PIP'],
        ['LYZ','PITPNA'],
        ['LYZ','PLAUR'],
        ['LYZ','PLBD1'],
        ['LYZ','PLD1'],
        ['LYZ','PLEC'],
        ['LYZ','PLEKHB2'],
        ['LYZ','PLEKHM2'],
        ['LYZ','PLEKHS1'],
        ['LYZ','PLIN3'],
        ['LYZ','PLPP3'],
        ['LYZ','PPIA'],
        ['LYZ','PPIB'],
        ['LYZ','PPP1CB'],
        ['LYZ','PPP2CA'],
        ['LYZ','PRAM1'],
        ['LYZ','PRB1'],
        ['LYZ','PRB2'],
        ['LYZ','PRELP'],
        ['LYZ','PRH1'],
        ['LYZ','PRH1-PRR4'],
        ['LYZ','PRKX'],
        ['LYZ','PRPF8'],
        ['LYZ','PRR27'],
        ['LYZ','PRR4'],
        ['LYZ','PRRC2B'],
        ['LYZ','PRTN3'],
        ['LYZ','PSAP'],
        ['LYZ','PSKH1'],
        ['LYZ','PSMA3-AS1'],
        ['LYZ','PSMB5'],
        ['LYZ','PSMC6'],
        ['LYZ','PSME4'],
        ['LYZ','PTAFR'],
        ['LYZ','PTP4A2'],
        ['LYZ','PTPN2'],
        ['LYZ','PTPRA'],
        ['LYZ','PTPRC'],
        ['LYZ','PTPRN2'],
        ['LYZ','PTRF'],
        ['LYZ','R3HDM2'],
        ['LYZ','RAB11FIP1'],
        ['LYZ','RAB18'],
        ['LYZ','RAB1A'],
        ['LYZ','RAB8B'],
        ['LYZ','RABGEF1'],
        ['LYZ','RAD23B'],
        ['LYZ','RANBP10'],
        ['LYZ','RBM3'],
        ['LYZ','RBM39'],
        ['LYZ','RBM5'],
        ['LYZ','RBSN'],
        ['LYZ','REEP5'],
        ['LYZ','REV1'],
        ['LYZ','REXO2'],
        ['LYZ','RF00017'],
        ['LYZ','RFWD3'],
        ['LYZ','RGCC'],
        ['LYZ','RIOK3'],
        ['LYZ','RNF115'],
        ['LYZ','RPL26'],
        ['LYZ','RPL30'],
        ['LYZ','RPL37A'],
        ['LYZ','RPL4'],
        ['LYZ','RPL7L1'],
        ['LYZ','RPS10-NUDT3'],
        ['LYZ','RPS14'],
        ['LYZ','RPS23'],
        ['LYZ','RPS24'],
        ['LYZ','RSPH6A'],
        ['LYZ','RTF1'],
        ['LYZ','RUNDC1'],
        ['LYZ','SAMD4B'],
        ['LYZ','SAMHD1'],
        ['LYZ','SAT1'],
        ['LYZ','SBF2'],
        ['LYZ','SCAF11'],
        ['LYZ','SCAMP1'],
        ['LYZ','SCYL1'],
        ['LYZ','SENP3-EIF4A1'],
        ['LYZ','SETX'],
        ['LYZ','SF3B1'],
        ['LYZ','SFPQ'],
        ['LYZ','SFTPB'],
        ['LYZ','SFXN1'],
        ['LYZ','SGSM3'],
        ['LYZ','SGTA'],
        ['LYZ','SLC12A6'],
        ['LYZ','SLC20A2'],
        ['LYZ','SLC25A37'],
        ['LYZ','SLC25A39'],
        ['LYZ','SLC2A3'],
        ['LYZ','SLC43A2'],
        ['LYZ','SLC4A1'],
        ['LYZ','SLC7A5'],
        ['LYZ','SMG1'],
        ['LYZ','SMIM14'],
        ['LYZ','SMR3B'],
        ['LYZ','SNRNP200'],
        ['LYZ','SNTB2'],
        ['LYZ','SOCS7'],
        ['LYZ','SP3'],
        ['LYZ','SPARC'],
        ['LYZ','SPATA13'],
        ['LYZ','SPATS2L'],
        ['LYZ','SPC24'],
        ['LYZ','SPINT2'],
        ['LYZ','SPN'],
        ['LYZ','SPTBN1'],
        ['LYZ','SPTY2D1'],
        ['LYZ','SRGN'],
        ['LYZ','SRRM2'],
        ['LYZ','SRSF10'],
        ['LYZ','SRSF5'],
        ['LYZ','SSH1'],
        ['LYZ','STAT1'],
        ['LYZ','STAT3'],
        ['LYZ','STAT6'],
        ['LYZ','STK32C'],
        ['LYZ','STK39'],
        ['LYZ','STT3B'],
        ['LYZ','STX16'],
        ['LYZ','STX16-NPEPL1'],
        ['LYZ','STX8'],
        ['LYZ','SUGCT'],
        ['LYZ','SUGP2'],
        ['LYZ','SYAP1'],
        ['LYZ','SYNE3'],
        ['LYZ','TAF15'],
        ['LYZ','TAOK3'],
        ['LYZ','TAPBP'],
        ['LYZ','TATDN2'],
        ['LYZ','TBRG1'],
        ['LYZ','TEF'],
        ['LYZ','TENM3'],
        ['LYZ','TFCP2L1'],
        ['LYZ','TFRC'],
        ['LYZ','TGFBRAP1'],
        ['LYZ','THEMIS2'],
        ['LYZ','TIA1'],
        ['LYZ','TIGD6'],
        ['LYZ','TIMP2'],
        ['LYZ','TJP2'],
        ['LYZ','TMBIM6'],
        ['LYZ','TMED3'],
        ['LYZ','TMEM184C'],
        ['LYZ','TMEM220-AS1'],
        ['LYZ','TMEM56'],
        ['LYZ','TMEM87A'],
        ['LYZ','TNS3'],
        ['LYZ','TPD52L1'],
        ['LYZ','TPM4'],
        ['LYZ','TPT1'],
        ['LYZ','TRA2A'],
        ['LYZ','TRB@'],
        ['LYZ','TRIM58'],
        ['LYZ','TRIP12'],
        ['LYZ','TSC2'],
        ['LYZ','TSG101'],
        ['LYZ','TSN'],
        ['LYZ','TXNL4A'],
        ['LYZ','TYK2'],
        ['LYZ','UBE2D3'],
        ['LYZ','UBE2H'],
        ['LYZ','UBXN2A'],
        ['LYZ','UBXN4'],
        ['LYZ','UBXN7'],
        ['LYZ','UFD1L'],
        ['LYZ','UGDH-AS1'],
        ['LYZ','USP4'],
        ['LYZ','VPS16'],
        ['LYZ','VPS4B'],
        ['LYZ','VPS53'],
        ['LYZ','VTI1A'],
        ['LYZ','WAC-AS1'],
        ['LYZ','WDR1'],
        ['LYZ','WDR82'],
        ['LYZ','WIPF2'],
        ['LYZ','WSB1'],
        ['LYZ','WWOX'],
        ['LYZ','XIST'],
        ['LYZ','XRCC6'],
        ['LYZ','YIPF4'],
        ['LYZ','YIPF5'],
        ['LYZ','YME1L1'],
        ['LYZ','Z99129.3'],
        ['LYZ','ZBTB43'],
        ['LYZ','ZG16B'],
        ['LYZ','ZKSCAN1'],
        ['LYZ','ZMYND8'],
        ['LYZ','ZNF251'],
        ['LYZ','ZNF394'],
        ['LYZ','ZNF431'],
        ['LYZ','ZNF655'],
        ['MAATS1','NR1I2'],
        ['MACC1','TAC4'],
        ['MACF1','PAX8'],
        ['MACF1','SFTPB'],
        ['MACF1','TG'],
        ['MAD1L1','MAFK'],
        ['MAD1L1','MROH3P'],
        ['MAD2L1','MRI1'],
        ['MAEA','UVSSA'],
        ['MAFF','STAP2'],
        ['MAFG','NUCKS1'],
        ['MALAT1','MTRNR2L12'],
        ['MALAT1','MTRNR2L8'],
        ['MALAT1','RPPH1'],
        ['MALAT1','SOD2'],
        ['MALAT1','TNN'],
        ['MALAT1','VPS36'],
        ['MALAT1','WDR74'],
        ['MALRD1','PLXDC2'],
        ['MALT1','RFWD3'],
        ['MAMDC2-AS1','SMC5-AS1'],
        ['MAML1','SQSTM1'],
        ['MAML1','TG'],
        ['MAML2','MTMR2'],
        ['MAN1A2','STATH'],
        ['MAN1B1-AS1','XIAP'],
        ['MAN2A1','UIMC1'],
        ['MAN2C1','PLA2G4F'],
        ['MANBA','UBE2D3'],
        ['MANBAL','SRC'],
        ['MAOA','MUC7'],
        ['MAOB','PRH2'],
        ['MAP1A','VIM'],
        ['MAP2K2','PIGR'],
        ['MAP2K2','PRB3'],
        ['MAP2K5','SKOR1'],
        ['MAP3K20','SSTR2'],
        ['MAP3K7CL','PPBP'],
        ['MAP3K8','PRKG1'],
        ['MAP3K9','PGD'],
        ['MAP4K4','TG'],
        ['MAP6D1','PARL'],
        ['MAP7D1','PRELID1'],
        ['MAP7D3','PIGR'],
        ['MAPK1','TRB@'],
        ['MAPK10','PTN'],
        ['MAPK10','TRIM36'],
        ['MAPK10','WDR44'],
        ['MAPK6','PRSS23'],
        ['MAPK7','RNF112'],
        ['MAPKAPK2','MPO'],
        ['MAPKBP1','MGA'],
        ['MAPRE1','MAPRE1P1'],
        ['MARC2','ZFP91-CNTF'],
        ['MARCH1','SNRPA1'],
        ['MARCH6','MUC7'],
        ['MARCH6','TG'],
        ['MARCKS','PTMA'],
        ['MARCKS','RCC2'],
        ['MARCKSL1','PTMS'],
        ['MARK2','ZDHHC24'],
        ['MAST2','TNNT2'],
        ['MAT1A','P4HA2'],
        ['MAT2A','TG'],
        ['MAT2B','STATH'],
        ['MATN2','TG'],
        ['MATN4','SLPI'],
        ['MATR3','PAIP2'],
        ['MAVS','PANK2'],
        ['MAVS','ZNF444'],
        ['MAX','SFI1'],
        ['MAZ','PTMS'],
        ['MAZ','TMEM160'],
        ['MB','NPPA-AS1'],
        ['MB','TTN-AS1'],
        ['MBD3','TG'],
        ['MBD5','ORC4'],
        ['MBNL1','SMR3B'],
        ['MBP','RTN3'],
        ['MBP','SET'],
        ['MBP','SYT11'],
        ['MBP','TMSB4X'],
        ['MBP','TUBA1A'],
        ['MBP','TUBB4A'],
        ['MBP','VAMP2'],
        ['MC2R','SH3PXD2A'],
        ['MCAM','MYL7'],
        ['MCAM','RPS24'],
        ['MCAM','RPS29'],
        ['MCC','YWHAZ'],
        ['MCCC2','SMIM14'],
        ['MCFD2','TG'],
        ['MCL1','PNLIP'],
        ['MCL1','S100A9'],
        ['MCM3AP-AS1','SSR3'],
        ['MCM4','PGAM5'],
        ['MCPH1','PIGR'],
        ['MCUR1','SRSF10'],
        ['MDC1','SCARB2'],
        ['MDH1','TMBIM6'],
        ['MDM1','SFTPB'],
        ['MDM2','NUP107'],
        ['MDM2','VPS4B'],
        ['MDM4','RAB30-AS1'],
        ['MDM4','TRA@'],
        ['MECOM','RPL22'],
        ['MECP2','TG'],
        ['MED12','NUMBL'],
        ['MED13L','PNLIP'],
        ['MED13L','SMR3B'],
        ['MED13L','TG'],
        ['MED15','ZNF890P'],
        ['MED21','PPFIBP1'],
        ['MED22','SURF6'],
        ['MED25','PDLIM1'],
        ['MED25','PTOV1'],
        ['MED29','TG'],
        ['MED9','SLC37A2'],
        ['MEF2C','TMEM161B'],
        ['MEFV','PLEKHH2'],
        ['MEG3','MYH9'],
        ['MEG3','SCARB1'],
        ['MEG3','SLC16A3'],
        ['MEG3','SNRNP200'],
        ['MEG3','SPARC'],
        ['MEG3','STAR'],
        ['MEG3','TFAP2D'],
        ['MEG3','TIMP3'],
        ['MEIS1','STATH'],
        ['MEIS2','MORF4L1'],
        ['MEIS2','MUC7'],
        ['MEMO1','RPS9P1'],
        ['MEMO1P1','RPS9P1'],
        ['METAZOA_SRP','PAFAH1B1'],
        ['METRNL','RALGDS'],
        ['METRNL','RASA3'],
        ['METRNL','TBCD'],
        ['METTL14-DT','NPR1'],
        ['METTL21B','TSFM'],
        ['METTL23','MFSD11'],
        ['METTL7A','PCAT7'],
        ['METTL7A','PLCG1-AS1'],
        ['METTL7A','USP37'],
        ['METTL9','SRM'],
        ['MFAP5','PIGR'],
        ['MFGE8','NPPA-AS1'],
        ['MFSD11','RND3'],
        ['MFSD13A','SLC25A29'],
        ['MFSD14A','SLC35A3'],
        ['MFSD14C','ZNF782'],
        ['MGAM','MGAM2'],
        ['MGAT4A','PRSS2'],
        ['MGEA5','S100A9'],
        ['MGLL','PPP1R1A'],
        ['MGRN1','NUDT16L1'],
        ['MICA','ZDHHC20P2'],
        ['MICAL2','TEAD1'],
        ['MICAL3','SPAG9'],
        ['MICAL3','TLR6'],
        ['MICALL1','POLR2F'],
        ['MICU1','UPK1A'],
        ['MICU2','ZDHHC20'],
        ['MICU3','SERPINB1'],
        ['MID2','PTK2'],
        ['MIF','SLC2A11'],
        ['MIF4GD','NFKBIA'],
        ['MINK1','SFTPB'],
        ['MINK1','SMR3B'],
        ['MINOS1','MMD'],
        ['MINOS1','NEK10'],
        ['MINOS1-NBL1','NEK10'],
        ['MINPP1','NUTM2A-AS1'],
        ['MIPOL1','TRA@'],
        ['MIR-371-CLUSTER','RPRD1B'],
        ['MIR100HG','PIGR'],
        ['MIR100HG','SSR3'],
        ['MIR100HG','STAG1'],
        ['MIR100HG','TG'],
        ['MIR100HG','TRA@'],
        ['MIR100HG','TRB@'],
        ['MIR22HG','TLCD2'],
        ['MIR4500HG','SAMHD1'],
        ['MIR497HG','SFTPC'],
        ['MIR99AHG','NRIP1'],
        ['MKL2','PARN'],
        ['MKLN1','VMAC'],
        ['MKNK2','TG'],
        ['MKRN2OS','NONO'],
        ['MLEC','UNC119B'],
        ['MLF2','TG'],
        ['MLIP','TINAG'],
        ['MLLT10','TG'],
        ['MLLT10P1','NCOA3'],
        ['MLLT4','PRSS1'],
        ['MLLT6','SMR3B'],
        ['MLLT6','TRB@'],
        ['MLPH','SMR3B'],
        ['MLPH','VPS28'],
        ['MLYCD','UGCG'],
        ['MMD','SMIM36'],
        ['MMD','TG'],
        ['MMP2','MRC2'],
        ['MMP2','SFTPB'],
        ['MN1','NCOA3'],
        ['MNAT1','SMR3B'],
        ['MND1','TRIM2'],
        ['MNS1','TEX9'],
        ['MOB1B','VLDLR-AS1'],
        ['MOB2','MUC7'],
        ['MOCS1','TP53I3'],
        ['MORC2-AS1','OSBP2'],
        ['MORF4L1','SEMA6A'],
        ['MORF4L1','SLC48A1'],
        ['MORF4L1','TRIM38'],
        ['MOV10L1','TYK2'],
        ['MPEG1','WWOX'],
        ['MPEG1','ZDHHC20'],
        ['MPI','SPINK1'],
        ['MPO','NUP210L'],
        ['MPO','PDZD2'],
        ['MPO','PIEZO1'],
        ['MPO','RNASE2'],
        ['MPO','RPS21'],
        ['MPO','SNX32'],
        ['MPO','TBC1D9B'],
        ['MPO','ZFP36'],
        ['MPP7','RAB18'],
        ['MPRIP','NPPB'],
        ['MPV17L','ZNF7'],
        ['MRC1','PICALM'],
        ['MRC1','TMEM236'],
        ['MRNIP','MUC5AC'],
        ['MRNIP','TRA@'],
        ['MRPL10','SCRN2'],
        ['MRPL14','RAB40C'],
        ['MRPL28','VIM'],
        ['MRPL30','RN7SL377P'],
        ['MRPL30P1','MS4A8'],
        ['MRPL42P3','STX11'],
        ['MRPL51','RBM3'],
        ['MRPL9','PI4KA'],
        ['MRPS10','PPM1B'],
        ['MRPS16','TM4SF19-AS1'],
        ['MRPS27','SOCS2-AS1'],
        ['MRPS27','SULT2A1'],
        ['MRPS28','TPD52'],
        ['MRPS31P2','TPTE2'],
        ['MRPS31P5','SUGT1P4-STRA6LP'],
        ['MS4A10','SP140L'],
        ['MS4A10','ZNF512'],
        ['MS4A12','THCAT158'],
        ['MS4A7','TCTN1'],
        ['MSH2','TTC7A'],
        ['MSI1','PLA2G1B'],
        ['MSL2','STAG1'],
        ['MSMB','NCOA4'],
        ['MSN','MUC7'],
        ['MSN','NPPA-AS1'],
        ['MSN','SFTPC'],
        ['MSRB3','TMBIM4'],
        ['MSX2','OR4D1'],
        ['MT1A','RSG1'],
        ['MT1A','TG'],
        ['MT1E','MT1L'],
        ['MT1M','MT2P1'],
        ['MT1X','RSG1'],
        ['MTA3','PNLIP'],
        ['MTA3','THADA'],
        ['MTATP6P1','MTRNR2L8'],
        ['MTERF4','PASK'],
        ['MTFMT','XXBAC-BPG299F13.17'],
        ['MTG1','NSD2'],
        ['MTG1','SCART1'],
        ['MTHFD1L','PLEKHG1'],
        ['MTHFSD','STIM2'],
        ['MTMR10','TRPM1'],
        ['MTMR12','SFTPB'],
        ['MTMR14','TMPRSS11B'],
        ['MTMR9','YIPF5'],
        ['MTO1','PIGR'],
        ['MTR','RYR2'],
        ['MTRNR2L12','MUC16'],
        ['MTRNR2L12','MUC2'],
        ['MTRNR2L12','MUC5AC'],
        ['MTRNR2L8','MUC5AC'],
        ['MTSS1','TATDN1'],
        ['MTSS1L','RANBP3'],
        ['MTSS1L','VAC14'],
        ['MTURN','PLEKHA8'],
        ['MTURN','ZNRF2'],
        ['MTUS1','PNLIP'],
        ['MTX2','SATB2'],
        ['MUC16','MUC4'],
        ['MUC16','MUC5AC'],
        ['MUC2','MUC5AC'],
        ['MUC2','MUC5B'],
        ['MUC20-OT1','SDHA'],
        ['MUC21','MYH11'],
        ['MUC21','PPL'],
        ['MUC5AC','MUC5B'],
        ['MUC5AC','MUC6'],
        ['MUC5AC','PGC'],
        ['MUC5AC','PTBP1'],
        ['MUC5B','MUC7'],
        ['MUC5B','SMR3B'],
        ['MUC5B','STATH'],
        ['MUC6','PGC'],
        ['MUC7','NDRG2'],
        ['MUC7','NEBL'],
        ['MUC7','NFIC'],
        ['MUC7','PABPC1'],
        ['MUC7','PDLIM5'],
        ['MUC7','PFN1'],
        ['MUC7','PHLDA1'],
        ['MUC7','PIGR'],
        ['MUC7','POLR2A'],
        ['MUC7','PRB1'],
        ['MUC7','PRB2'],
        ['MUC7','PRB3'],
        ['MUC7','PRH1'],
        ['MUC7','PRH1-PRR4'],
        ['MUC7','PRH2'],
        ['MUC7','PRR4'],
        ['MUC7','PSAP'],
        ['MUC7','PTPRF'],
        ['MUC7','RPL8'],
        ['MUC7','RPS11'],
        ['MUC7','RPS21'],
        ['MUC7','RRBP1'],
        ['MUC7','RUNX1'],
        ['MUC7','SCARB2'],
        ['MUC7','SECISBP2L'],
        ['MUC7','SLC35E2B'],
        ['MUC7','SMR3A'],
        ['MUC7','SQSTM1'],
        ['MUC7','ST3GAL1'],
        ['MUC7','STATH'],
        ['MUC7','SYN3'],
        ['MUC7','TMED10'],
        ['MUC7','TNS1'],
        ['MUC7','TNS4'],
        ['MUC7','TSHZ2'],
        ['MUC7','WNK2'],
        ['MUC7','ZG16B'],
        ['MUM1','MYH11'],
        ['MUM1','NDUFS7'],
        ['MVD','TTC7B'],
        ['MXRA7','PTRF'],
        ['MXRA7','RBMS2'],
        ['MXRA7','TRA@'],
        ['MXRA8','SBF1'],
        ['MYADM','NEDD4L'],
        ['MYADM','PARN'],
        ['MYADM','SAMHD1'],
        ['MYADM','SEC11A'],
        ['MYADM','TG'],
        ['MYADM','TNS1'],
        ['MYBPC1','PRKAR2A'],
        ['MYBPC1','RYR1'],
        ['MYBPC1','SOD2'],
        ['MYBPC3','PPP1R12B'],
        ['MYH10','SFTPC'],
        ['MYH10','TRB@'],
        ['MYH11','MYH6'],
        ['MYH11','NUMA1'],
        ['MYH11','P4HB'],
        ['MYH11','PABPC1'],
        ['MYH11','PCDH15'],
        ['MYH11','PIGR'],
        ['MYH11','PPL'],
        ['MYH11','PPT1'],
        ['MYH11','RET'],
        ['MYH11','RHCG'],
        ['MYH11','SCARB2'],
        ['MYH11','SMR3B'],
        ['MYH11','SPINK5'],
        ['MYH11','SPTBN1'],
        ['MYH11','SRCAP'],
        ['MYH11','SYN3'],
        ['MYH11','SYNM'],
        ['MYH11','SYNPO2'],
        ['MYH11','TGM3'],
        ['MYH11','TPM1'],
        ['MYH11','TRB@'],
        ['MYH11','TRIM29'],
        ['MYH11','UBE2N'],
        ['MYH11','WWOX'],
        ['MYH11','YWHAZ'],
        ['MYH16','SCD5'],
        ['MYH3','SCO1'],
        ['MYH6','NEXN'],
        ['MYH6','NPPA'],
        ['MYH6','NPPA-AS1'],
        ['MYH6','PLIN4'],
        ['MYH6','SRRM2'],
        ['MYH6','TIMP3'],
        ['MYH6','TPT1'],
        ['MYH6','TTN'],
        ['MYH6','ZNF106'],
        ['MYH7','MYH7B'],
        ['MYH7','MYH9'],
        ['MYH7','NPPA-AS1'],
        ['MYH7','PSMC4'],
        ['MYH7','SLC4A3'],
        ['MYH7','TNNC1'],
        ['MYH7','TNS1'],
        ['MYH9','NDE1'],
        ['MYH9','PLIN4'],
        ['MYH9','PRH1'],
        ['MYH9','PRH1-PRR4'],
        ['MYH9','PRH2'],
        ['MYH9','PXDN'],
        ['MYH9','QSOX1'],
        ['MYH9','RPL13A'],
        ['MYH9','SFTPA2'],
        ['MYH9','SFTPC'],
        ['MYH9','SMR3B'],
        ['MYH9','TAGLN'],
        ['MYH9','TPM2'],
        ['MYH9','TRB@'],
        ['MYL2','MYL7'],
        ['MYL2','MYL9'],
        ['MYL2','NCOA6'],
        ['MYL2','NDRG1'],
        ['MYL2','NPPA-AS1'],
        ['MYL2','PALM2-AKAP2'],
        ['MYL3','PRSS42'],
        ['MYL3','PRSS44'],
        ['MYL3','PRSS45'],
        ['MYL3','PRSS50'],
        ['MYL3','TNNC1'],
        ['MYL4','NPPA'],
        ['MYL6','MYL6P1'],
        ['MYL6','MYL6P2'],
        ['MYL6','MYL6P3'],
        ['MYL6','MYL6P4'],
        ['MYL6','MYL6P5'],
        ['MYL6','SPARC'],
        ['MYL6','TG'],
        ['MYL7','MYLK3'],
        ['MYL7','NPPA'],
        ['MYL7','PPP1R12B'],
        ['MYL7','RYR2'],
        ['MYL9','TGIF2'],
        ['MYLK','PRB3'],
        ['MYO10','TSPAN18'],
        ['MYO19','ZG16B'],
        ['MYO1C','TAPBP'],
        ['MYO1F','ZNF414'],
        ['MYO5A','MYO5C'],
        ['MYO5A','SFTPB'],
        ['MYO5C','NDUFS6'],
        ['MYO6','PNLIP'],
        ['MYO9B','PTRF'],
        ['MYOZ1','SYNPO2L'],
        ['MYRFL','RAB3IP'],
        ['MYT1','OPRL1'],
        ['N4BP2','PNPT1'],
        ['N4BP2L2','RTN3'],
        ['NAA16','NKIRAS2'],
        ['NAA20','RIN2'],
        ['NAA35','RPL7L1'],
        ['NACA','NPPA-AS1'],
        ['NACA','PRIM1'],
        ['NACA','RAB3IL1'],
        ['NACC1','NFX1'],
        ['NADKD1','SDK2'],
        ['NAIP','OCLN'],
        ['NANOGNBP2','ZNF605'],
        ['NANP','NINL'],
        ['NAP1L1','SEMA4D'],
        ['NBEAL2','S100A9'],
        ['NBPF10','NOTCH2NL'],
        ['NBPF13P','PDE4DIP'],
        ['NBPF20','NOTCH2NL'],
        ['NBR1','NBR2'],
        ['NCAM1','VSNL1'],
        ['NCBP2','NUDT19'],
        ['NCBP2','PIGZ'],
        ['NCBP2-AS1','SENP5'],
        ['NCBP3','PTGES'],
        ['NCCRP1','TMF1'],
        ['NCDN','VSNL1'],
        ['NCKAP1','PNLIP'],
        ['NCKAP1','SMR3B'],
        ['NCKAP1L','PDE1B'],
        ['NCKAP5','R3HDM1'],
        ['NCL','NEFM'],
        ['NCL','NPM1'],
        ['NCL','PLBD2'],
        ['NCL','PTMA'],
        ['NCL','SFTPC'],
        ['NCL','TRB@'],
        ['NCLN','S1PR4'],
        ['NCOA3','PNLIP'],
        ['NCOA6','NPPA'],
        ['NCOA6','PIGU'],
        ['NCOR2','OAF'],
        ['NCOR2','PLAUR'],
        ['NCOR2','SFTPA2'],
        ['NCOR2','SMR3B'],
        ['NCOR2','UBC'],
        ['NCR3LG1','NUCB2'],
        ['NCSTN','NHLH1'],
        ['NDE1','PGR'],
        ['NDE1','STAMBPL1'],
        ['NDEL1','UBB'],
        ['NDFIP1','SYNPO2'],
        ['NDRG1','ST3GAL1'],
        ['NDRG2','PRH1'],
        ['NDRG2','PRH1-PRR4'],
        ['NDRG2','SMR3B'],
        ['NDRG3','ZFAND6'],
        ['NDRG3','ZNF829'],
        ['NDST1','ZG16B'],
        ['NDUFA10','TOX4'],
        ['NDUFA6-AS1','SMDT1'],
        ['NDUFAB1','PALB2'],
        ['NDUFB10','USO1'],
        ['NDUFB8','SEC31B'],
        ['NDUFS7','SCD5'],
        ['NDUFS8','PNLIP'],
        ['NDUFV1','NPPA-AS1'],
        ['NDUFV3','PKNOX1'],
        ['NEAT1','OCLN'],
        ['NEAT1','PIGR'],
        ['NEAT1','SYN3'],
        ['NEAT1','TFF2'],
        ['NEAT1','TMEM178B'],
        ['NEAT1','TMEM99'],
        ['NEAT1','TRA@'],
        ['NEAT1','WWOX'],
        ['NEAT1','ZFHX3'],
        ['NEBL','SMR3B'],
        ['NECAB2','OSGIN1'],
        ['NECAP2','SZRD1'],
        ['NEDD4L','TRB@'],
        ['NEDD8','TINF2'],
        ['NEDD8','UBA52'],
        ['NEDD8-MDP1','TINF2'],
        ['NEFH','RFPL1'],
        ['NEGR1','RBL1'],
        ['NEIL1','ZNF557'],
        ['NEK10','SLC4A7'],
        ['NEK7','PIGR'],
        ['NEK9','TMED10'],
        ['NEMP1','PLEKHA2'],
        ['NEU3','SPCS2'],
        ['NEXN','TGM4'],
        ['NF2','TG'],
        ['NFATC1','PQLC1'],
        ['NFATC2IP','SPNS1'],
        ['NFATC3','PLA2G15'],
        ['NFE2L1','NPPA-AS1'],
        ['NFE2L2','PTMA'],
        ['NFE2L2','TG'],
        ['NFIB','PRH1'],
        ['NFIB','PRH1-PRR4'],
        ['NFIB','RAC1'],
        ['NFIB','VMP1'],
        ['NFIC','PNLIP'],
        ['NFIC','PRB2'],
        ['NFIX','PRB3'],
        ['NFIX','PRH2'],
        ['NFIX','PTMS'],
        ['NFIX','SMR3B'],
        ['NFIX','VPS26B'],
        ['NFKBIZ','SRGN'],
        ['NFKBIZ','TGM4'],
        ['NFYC','TG'],
        ['NIBAN1','PLEKHA1'],
        ['NIF3L1','ORC4'],
        ['NIFK-AS1','TERB1'],
        ['NIN','SAV1'],
        ['NIPBL','TRB@'],
        ['NIT2','TBC1D23'],
        ['NKAPP1','RHOXF1'],
        ['NKIRAS1','SMR3B'],
        ['NKRF','SEPT6'],
        ['NKTR','TMEM99'],
        ['NKX2-1','SFTA3'],
        ['NME2','SLK'],
        ['NME4','PCSK7'],
        ['NME4','Z97634.1'],
        ['NMNAT1','RBP7'],
        ['NMNAT2','VSNL1'],
        ['NMNAT3','TRA@'],
        ['NMT1','PABPC1'],
        ['NMU','PDCL2'],
        ['NNT-AS1','TPT1P2'],
        ['NOL11','PITPNC1'],
        ['NOL6','SUGT1'],
        ['NOL6','SUGT1P'],
        ['NOL9','PPP1R15B'],
        ['NONO','PTRF'],
        ['NONO','SFTPB'],
        ['NOP9','TF'],
        ['NORAD','TRB@'],
        ['NOS1','PKP1'],
        ['NOTCH2','SMR3B'],
        ['NOTCH3','PTMS'],
        ['NOTCH3','TG'],
        ['NOVA1','SDC3'],
        ['NOXRED1','TMED8'],
        ['NOXRED1','VIPAS39'],
        ['NPC2','SFTPB'],
        ['NPC2','SYNDIG1L'],
        ['NPEPL1','STX16'],
        ['NPEPPS','TBC1D3'],
        ['NPEPPS','TBC1D3B'],
        ['NPEPPS','TBC1D3C'],
        ['NPEPPS','TBC1D3D'],
        ['NPEPPS','TBC1D3E'],
        ['NPEPPS','TBC1D3F'],
        ['NPEPPS','TBC1D3G'],
        ['NPEPPS','TBC1D3H'],
        ['NPEPPS','TBC1D3I'],
        ['NPEPPS','TBC1D3K'],
        ['NPEPPS','TBC1D3L'],
        ['NPHS1','PRODH2'],
        ['NPIPB15','PDXDC2P-NPIPB14P'],
        ['NPIPB3','SMG1'],
        ['NPIPB4','SMG1'],
        ['NPIPB5','POLR3K'],
        ['NPIPB5','SMG1'],
        ['NPIPP1','PKD1'],
        ['NPM','NPM1P43'],
        ['NPM1','NPM1P51'],
        ['NPM1','PTMA'],
        ['NPM1','TJP1'],
        ['NPPA','P4HB'],
        ['NPPA','POLD2'],
        ['NPPA','PPP1R12B'],
        ['NPPA','PRRC2B'],
        ['NPPA','RAPGEF1'],
        ['NPPA','S100A13'],
        ['NPPA','SF3B1'],
        ['NPPA','SORBS2'],
        ['NPPA','SPOCK1'],
        ['NPPA','STAMBPL1'],
        ['NPPA','SYNPO'],
        ['NPPA','TECRL'],
        ['NPPA','TGM2'],
        ['NPPA','TIMP3'],
        ['NPPA','TNNT2'],
        ['NPPA','TNS1'],
        ['NPPA-AS1','NPPB'],
        ['NPPA-AS1','NUCB2'],
        ['NPPA-AS1','PBX1'],
        ['NPPA-AS1','PLEC'],
        ['NPPA-AS1','PSAP'],
        ['NPPA-AS1','RAB11B'],
        ['NPPA-AS1','RCAN2'],
        ['NPPA-AS1','RILPL1'],
        ['NPPA-AS1','S100A1'],
        ['NPPA-AS1','SGCD'],
        ['NPPA-AS1','SPARC'],
        ['NPPA-AS1','SQSTM1'],
        ['NPPA-AS1','SRRM2'],
        ['NPPA-AS1','SYNPO'],
        ['NPPA-AS1','TGOLN2'],
        ['NPPA-AS1','TNNC1'],
        ['NPPA-AS1','TNNI3'],
        ['NPPA-AS1','TNNT2'],
        ['NPPA-AS1','TNS1'],
        ['NPPA-AS1','TP53INP2'],
        ['NPPA-AS1','TPM1'],
        ['NPPA-AS1','TTN-AS1'],
        ['NPPA-AS1','TXNIP'],
        ['NPPA-AS1','USB1'],
        ['NPPA-AS1','WDPCP'],
        ['NPTXR','PPP1R12B'],
        ['NPTXR','RAD52'],
        ['NQO1','THRIL'],
        ['NQO1','ZNF486'],
        ['NQO2','PABPC1L'],
        ['NR2F1','NR2F2'],
        ['NR4A1','STAT3'],
        ['NRAV','SRSF9'],
        ['NRBP1','TG'],
        ['NRBP2','PUF60'],
        ['NRGN','TMSB10'],
        ['NRIP1','USP25'],
        ['NRIP3','TMEM9B'],
        ['NRSN2-AS1','TRB@'],
        ['NSD1','PNLIP'],
        ['NSD1','PWWP2A'],
        ['NSD1','SFTPB'],
        ['NSD1','SMR3B'],
        ['NSF','PNLIP'],
        ['NSFL1C','SIRPB2'],
        ['NSMAF','SPTLC3'],
        ['NSMCE2','PLCL1'],
        ['NSUN5','TRIM50'],
        ['NSUN5','TRIM73'],
        ['NT5C2','PRICKLE1'],
        ['NT5C2','SNAP23'],
        ['NT5C3B','ZNF814'],
        ['NT5DC2','PTRF'],
        ['NTN4','TRB@'],
        ['NTN4','VASN'],
        ['NTRK2','SMR3B'],
        ['NTRK2','TG'],
        ['NUCB1','SFTPC'],
        ['NUCB1-AS1','TULP2'],
        ['NUCB2','SMR3B'],
        ['NUCB2','ZG16B'],
        ['NUDCD1','SNX16'],
        ['NUDCD3','TG'],
        ['NUDT10','PTMS'],
        ['NUDT15','SUCLA2-AS1'],
        ['NUDT17','POLR3C'],
        ['NUDT4','PNLIP'],
        ['NUMA1','SMR3B'],
        ['NUP160','PNLIP'],
        ['NUP188','PHYHD1'],
        ['NUP210','STOM'],
        ['NUP88','TG'],
        ['NUPR1','RBM47'],
        ['NUSAP1','OIP5-AS1'],
        ['NUTM2A','NUTM2B'],
        ['NUTM2A-AS1','SYT15'],
        ['NUTM2B-AS1','SYT15'],
        ['NWD1','SLC25A15'],
        ['NXN','PTMS'],
        ['NXN','TPM4'],
        ['OAZ1','PFKFB3'],
        ['OAZ1','PNLIP'],
        ['OAZ1','PTBP1'],
        ['OAZ1','PTMA'],
        ['OAZ1','RAC2'],
        ['OAZ1','SEPT5'],
        ['OAZ1','SEPT5-GP1BB'],
        ['OAZ1','SGTA'],
        ['OAZ1','TCF3'],
        ['OAZ1','UBTF'],
        ['OAZ1','VIM-AS1'],
        ['OAZ2','RBPMS2'],
        ['OAZ2','SIK2'],
        ['OBFC1','SH3PXD2A'],
        ['OCEL1','TMEM97'],
        ['OCIAD1','TG'],
        ['ODAM','PRH1'],
        ['ODAM','PRH1-PRR4'],
        ['ODAM','PRH2'],
        ['ODF3B','SCO2'],
        ['OGDH','PAX8-AS1'],
        ['OGG1','TMBIM6'],
        ['OIP5-AS1','RSU1'],
        ['OLFM5P','UBQLNL'],
        ['OLMALINC','TRA@'],
        ['ONECUT1','RAB7A'],
        ['OOEP-AS1','PTMA'],
        ['OPA3','ZNF451'],
        ['OPRPN','SMR3B'],
        ['OR14J1','TRA@'],
        ['OR2J3','XXBAC-BPG258E24.10'],
        ['OR2T32P','TRIM58'],
        ['OR2T8','TRIM58'],
        ['OR52L1','OR56A4'],
        ['OR52M2P','TRIM21'],
        ['OR6D1P','SMG1'],
        ['OR6K3','OR6K4P'],
        ['OR6K5P','OR6N2'],
        ['OR6M3P','TMEM225'],
        ['OR7D1P','OR7E24'],
        ['OR7E161P','OR7E91P'],
        ['OR8T1P','ZNF641'],
        ['ORAI2','PRKRIP1'],
        ['ORAOV1','PTMS'],
        ['ORC1','ZCCHC11'],
        ['ORM1','SERPINA1'],
        ['ORM2','SERPINA1'],
        ['OS9','TRB@'],
        ['OSBPL8','UBAC2'],
        ['OSMR','RICTOR'],
        ['OSTF1','SSR3'],
        ['OXR1','PNLIP'],
        ['OXTR','SSUH2'],
        ['P2RX4','SORBS1'],
        ['P4HB','SMR3B'],
        ['PA2G4','ZC3H10'],
        ['PABPC1','RLIM'],
        ['PABPC1','RPS19'],
        ['PABPC1L','YWHAB'],
        ['PACRG','TMEM242'],
        ['PACSIN2','TOB1'],
        ['PADI4','TFRC'],
        ['PAFAH1B1','TG'],
        ['PAFAH1B2','SUGP2'],
        ['PAFAH1B2','USP54'],
        ['PAFAH1B2','YIPF5'],
        ['PAFAH2','TMBIM6'],
        ['PAICS','SRP72'],
        ['PAK2','SMIM14'],
        ['PAK3','TRB@'],
        ['PAK4','SMR3B'],
        ['PALM','UBE2J2'],
        ['PAM','PNLIP'],
        ['PAM','TRB@'],
        ['PAN2','STAT2'],
        ['PANK1','SLC16A12'],
        ['PANK3','PIGR'],
        ['PANK3','TRIM68'],
        ['PAPPA','TIMP3'],
        ['PAQR6','SMG5'],
        ['PAQR7','RPS23'],
        ['PAQR7','STMN1'],
        ['PARG','TIMM23B'],
        ['PARK2','UBA52'],
        ['PARM1','PRH1'],
        ['PARM1','PRH1-PRR4'],
        ['PARP10','SFTPC'],
        ['PATE2','PKP2'],
        ['PATJ','TG'],
        ['PATL2','SPG11'],
        ['PATZ1','PIK3IP1'],
        ['PAX5','RHOBTB3'],
        ['PAX5','ZNF451'],
        ['PAX8-AS1','TG'],
        ['PAX8-AS1','TPO'],
        ['PAXBP1','PCDH17'],
        ['PBX1','PRB2'],
        ['PBX1','TG'],
        ['PBX3','ZNF518A'],
        ['PBXIP1','PMVK'],
        ['PCBD1','PRSS1'],
        ['PCBP1-AS1','S100A9'],
        ['PCBP1-AS1','SRGN'],
        ['PCBP1-AS1','STATH'],
        ['PCBP1-AS1','TG'],
        ['PCBP2','PHF7'],
        ['PCBP2','SMR3B'],
        ['PCBP2','TG'],
        ['PCBP3','PTTG1IP'],
        ['PCCA','WDR60'],
        ['PCCB','PDS5B'],
        ['PCDH7','RAB29'],
        ['PCDHB13','PCDHB8'],
        ['PCDHB15','PCDHB19P'],
        ['PCDHGA12','PCDHGB8P'],
        ['PCDHGB8P','PCDHGC3'],
        ['PCDHGB8P','PCDHGC4'],
        ['PCDHGB8P','PCDHGC5'],
        ['PCLO','UCP2'],
        ['PCM1','SPIDR'],
        ['PCMTD1','PXDNL'],
        ['PCNA','TMEM230'],
        ['PCNP','PCNPP3'],
        ['PCNP','ZNF26'],
        ['PCOLCE','PLEC'],
        ['PCP4L1','SDHC'],
        ['PCSK7','PKM'],
        ['PCSK7','RRBP1'],
        ['PCSK7','TAF15'],
        ['PCSK7','TGFBI'],
        ['PCYOX1','SOD2'],
        ['PDCD11','PNLIP'],
        ['PDCD2L','UBA2'],
        ['PDCD7','UBAP1L'],
        ['PDE1C','SNX1'],
        ['PDE4B','SMR3B'],
        ['PDE8B','TG'],
        ['PDE8B','ZBED3-AS1'],
        ['PDHA1','STATH'],
        ['PDIA2','TRB@'],
        ['PDIA3','PIGR'],
        ['PDIA3','PNLIP'],
        ['PDIA3','PRSS1'],
        ['PDIA3','SMR3B'],
        ['PDIA3','SPP1'],
        ['PDIA3','STATH'],
        ['PDIA3','TG'],
        ['PDIA6','PRSS2'],
        ['PDIA6','SMR3B'],
        ['PDIA6','STATH'],
        ['PDK4','TRB@'],
        ['PDLIM1','TG'],
        ['PDS5A','TG'],
        ['PDXDC1','PNLIP'],
        ['PDXDC1','PRSS1'],
        ['PDXDC1','RAC1'],
        ['PDXDC1','SMR3B'],
        ['PDXK','SFTPB'],
        ['PDXK','ZNF593'],
        ['PDZD2','PVRL4'],
        ['PDZK1IP1','TAL1'],
        ['PDZRN3','STATH'],
        ['PEA15','S100A11'],
        ['PEA15','SMR3B'],
        ['PEBP1','SERPINA1'],
        ['PECAM1','PTRF'],
        ['PECAM1','SFTPB'],
        ['PECAM1','SMARCA4'],
        ['PECAM1','TG'],
        ['PECAM1','ZG16B'],
        ['PEMT','UBB'],
        ['PER1','PRH1'],
        ['PER1','PRH1-PRR4'],
        ['PER1','PRR4'],
        ['PEX11A','PLIN1'],
        ['PEX26','PLIN3'],
        ['PFDN1','SKP1'],
        ['PFDN6','RNF19B'],
        ['PFKFB3','S100A9'],
        ['PFKFB4','SHISA5'],
        ['PFKL','TG'],
        ['PFN1','RANBP3'],
        ['PGA3','PGC'],
        ['PGA3','POLR2A'],
        ['PGA5','PGC'],
        ['PGAM1P5','PRSS2'],
        ['PGAM1P5','TRB@'],
        ['PGAM1P5','TRBC2'],
        ['PGAM1P5','TRBV25-1'],
        ['PGBD2','RPL23AP7'],
        ['PGC','PIGR'],
        ['PGC','POLR2A'],
        ['PGK1','SMR3B'],
        ['PGLYRP3','PGLYRP4'],
        ['PGM5','TGM4'],
        ['PGM5','WASHC1'],
        ['PHACTR4','RCC1'],
        ['PHACTR4','SNTB2'],
        ['PHB','ZNF607'],
        ['PHC1','PRB3'],
        ['PHF14','SMR3B'],
        ['PHF2','SFTPC'],
        ['PHF6','PPP1CB'],
        ['PHF8','PRH2'],
        ['PHGDH','SCD'],
        ['PHLDA1','PRH1'],
        ['PHLDA1','PRH1-PRR4'],
        ['PIAS1','SMR3B'],
        ['PIAS2','ST8SIA5'],
        ['PICALM','PRB3'],
        ['PICALM','SLC38A1'],
        ['PID1','SFTPA1'],
        ['PIEZO1','SFTPA2'],
        ['PIGL','UBB'],
        ['PIGN','PNLIP'],
        ['PIGN','TRA@'],
        ['PIGO','TG'],
        ['PIGR','PIP'],
        ['PIGR','PLA2G2A'],
        ['PIGR','POC5'],
        ['PIGR','PRB1'],
        ['PIGR','PRB2'],
        ['PIGR','PRH1'],
        ['PIGR','PRH1-PRR4'],
        ['PIGR','PRR4'],
        ['PIGR','RAD51C'],
        ['PIGR','RBM20'],
        ['PIGR','RBM3'],
        ['PIGR','RPLP1'],
        ['PIGR','SETD4'],
        ['PIGR','SFTPB'],
        ['PIGR','SLC44A4'],
        ['PIGR','SLC5A1'],
        ['PIGR','SMU1'],
        ['PIGR','SPG7'],
        ['PIGR','SPTBN1'],
        ['PIGR','STAT1'],
        ['PIGR','STMND1'],
        ['PIGR','SUSD1'],
        ['PIGR','SYNPO2'],
        ['PIGR','TMBIM6'],
        ['PIGR','TRA@'],
        ['PIGR','VAC14-AS1'],
        ['PIGR','ZG16B'],
        ['PIGR','ZKSCAN1'],
        ['PIGR','ZNF232'],
        ['PIK3C2A','PNLIP'],
        ['PIK3R1','PRH1'],
        ['PIK3R1','PRH1-PRR4'],
        ['PIK3R1','TPM1'],
        ['PILRB','WDR66'],
        ['PIM2','SLC35A2'],
        ['PIM3','SCO2'],
        ['PIM3','TRABD'],
        ['PIN4','PNLIP'],
        ['PINK1-AS','TAS2R4'],
        ['PINK1-AS','TTL'],
        ['PIP4K2A','TRG-AS1'],
        ['PIP4K2A','TRG@'],
        ['PIPOX','SOD2'],
        ['PITPNA','SMR3B'],
        ['PITPNA','TG'],
        ['PITPNA','YWHAE'],
        ['PITPNC1','PRKCA'],
        ['PITPNC1','SLC25A3'],
        ['PIWIL2','STAR'],
        ['PIWIL3','ZFYVE19'],
        ['PJA2','SFTPB'],
        ['PJVK','PRKRA-AS1'],
        ['PKD1','PKD1P6-NPIPP1'],
        ['PKD1','PRB3'],
        ['PKHD1L1','STX8'],
        ['PKHD1L1','TG'],
        ['PKIB','SIGLEC6'],
        ['PKM','RHOC'],
        ['PKM','SERPINE1'],
        ['PKM','THBS1'],
        ['PKN1','UBE2M'],
        ['PKP1','TMEM99'],
        ['PKP4','PRB3'],
        ['PLA2G15','WBP1L'],
        ['PLA2G1B','PNLIP'],
        ['PLA2G1B','PRSS1'],
        ['PLA2G1B','PRSS2'],
        ['PLA2G1B','TRB@'],
        ['PLA2G6','TMEM184B'],
        ['PLAC8','SNX5'],
        ['PLAT','SPTLC2'],
        ['PLCB1','STATH'],
        ['PLCB3','TXLNA'],
        ['PLCB4','SMR3B'],
        ['PLCD3','PTRF'],
        ['PLCG1','TG'],
        ['PLEC','PLOD3'],
        ['PLEC','SMR3B'],
        ['PLEC','TFPI2'],
        ['PLEC','TNC'],
        ['PLEC','VIM-AS1'],
        ['PLEKHA1','RAB17'],
        ['PLEKHA5','SRSF11'],
        ['PLEKHB1','VAMP2'],
        ['PLEKHB2','SETBP1'],
        ['PLEKHB2','ZNF727'],
        ['PLEKHG2','ZFP36'],
        ['PLEKHG4B','STX17'],
        ['PLEKHH1','TG'],
        ['PLEKHM1','TG'],
        ['PLEKHO1','VPS45'],
        ['PLIN1','WWOX'],
        ['PLIN3','SCUBE3'],
        ['PLIN4','PSAP'],
        ['PLIN4','SERPING1'],
        ['PLIN4','SPARC'],
        ['PLIN4','SPTBN1'],
        ['PLIN4','SYN3'],
        ['PLIN4','TBX1'],
        ['PLIN5','SAA2'],
        ['PLPP1','RBM47'],
        ['PLSCR4','RNF13'],
        ['PLVAP','RAD23A'],
        ['PLXNA1','SFTPC'],
        ['PLXNA2','PRSS1'],
        ['PLXNA2','PRSS2'],
        ['PLXNB2','SFTPB'],
        ['PM20D2','WDFY3'],
        ['PMCH','PMCHL1'],
        ['PMCH','PMCHL2'],
        ['PMCHL1','PMCHL2'],
        ['PMEPA1','PNLIP'],
        ['PML','SDC3'],
        ['PMPCB','UBE2L6'],
        ['PMS2','PMS2CL'],
        ['PMS2','POLR2J3'],
        ['PMS2CL','RBAK'],
        ['PMS2CL','RBAK-RBAKDN'],
        ['PNLIP','PNLIPRP1'],
        ['PNLIP','PNLIPRP2'],
        ['PNLIP','POM121'],
        ['PNLIP','PPARA'],
        ['PNLIP','PRSS1'],
        ['PNLIP','PRSS2'],
        ['PNLIP','PRSS3'],
        ['PNLIP','PRSS3P1'],
        ['PNLIP','PTPRN2'],
        ['PNLIP','RALY'],
        ['PNLIP','RAPGEF1'],
        ['PNLIP','RNF10'],
        ['PNLIP','RPL30'],
        ['PNLIP','RPS20'],
        ['PNLIP','RPS29'],
        ['PNLIP','SCAMP5'],
        ['PNLIP','SCARB2'],
        ['PNLIP','SDHA'],
        ['PNLIP','SEC23B'],
        ['PNLIP','SERF2'],
        ['PNLIP','SERPINI2'],
        ['PNLIP','SGK1'],
        ['PNLIP','SLC25A26'],
        ['PNLIP','SLC37A4'],
        ['PNLIP','SLC38A10'],
        ['PNLIP','SLC4A4'],
        ['PNLIP','SLIT2'],
        ['PNLIP','SMIM19'],
        ['PNLIP','SND1'],
        ['PNLIP','SNTG2'],
        ['PNLIP','SOD2'],
        ['PNLIP','SPATS2L'],
        ['PNLIP','SPINT2'],
        ['PNLIP','SPTBN1'],
        ['PNLIP','SRRM2'],
        ['PNLIP','ST6GALNAC6'],
        ['PNLIP','STAG3L5P'],
        ['PNLIP','SYCN'],
        ['PNLIP','SYN3'],
        ['PNLIP','TCIRG1'],
        ['PNLIP','TM9SF3'],
        ['PNLIP','TMBIM6'],
        ['PNLIP','TMED3'],
        ['PNLIP','TMEM179B'],
        ['PNLIP','TMSB4X'],
        ['PNLIP','TRAM1'],
        ['PNLIP','TRB@'],
        ['PNLIP','TRHDE'],
        ['PNLIP','TRIM28'],
        ['PNLIP','TRIM44'],
        ['PNLIP','TSC22D3'],
        ['PNLIP','UBA52'],
        ['PNLIP','UBE2R2-AS1'],
        ['PNLIP','USP7'],
        ['PNLIP','VPS28'],
        ['PNLIP','WLS'],
        ['PNLIP','ZNF664'],
        ['PNLIPRP1','PNLIPRP2'],
        ['PNLIPRP2','PRSS1'],
        ['PNLIPRP2','PRSS2'],
        ['PNLIPRP2','TRB@'],
        ['PNPLA6','TG'],
        ['PNPLA7','TG'],
        ['PNPLA8','THAP5P1'],
        ['PNPO','TBC1D31'],
        ['PNRC1','ZNF292'],
        ['PODXL','TG'],
        ['POLR2A','PRM2'],
        ['POLR2A','SMR3B'],
        ['POLR2A','TMEM99'],
        ['POLR2J3','RASA4'],
        ['POLR2J3','RASA4B'],
        ['POLR2J4','RASA4'],
        ['POLR3H','PRPSAP2'],
        ['POM121','PRKRIP1'],
        ['POM121C','PRKRIP1'],
        ['POMGNT1','TG'],
        ['POU2F1','WDR59'],
        ['POU2F3','TMEM99'],
        ['POU6F1','SMAGP'],
        ['PPAP2A','RNF138'],
        ['PPARA','STATH'],
        ['PPARG','TSEN2'],
        ['PPARGC1A','RSF1'],
        ['PPDPF','SDC3'],
        ['PPIB','PRH1-PRR4'],
        ['PPIL2','TG'],
        ['PPIL6','TRB@'],
        ['PPIP5K1','STRC'],
        ['PPM1B','SLC3A1'],
        ['PPM1H','USP15'],
        ['PPP1CB','TG'],
        ['PPP1CB','WSB1'],
        ['PPP1CB','YPEL5'],
        ['PPP1R12B','SFTPB'],
        ['PPP1R12B','SOX2-OT'],
        ['PPP1R12B','STT3A'],
        ['PPP1R12B','TPM1'],
        ['PPP1R12B','USHBP1'],
        ['PPP1R12B','ZNF623'],
        ['PPP1R12C','TSPAN4'],
        ['PPP1R18','SFTPA2'],
        ['PPP1R1B','STARD3'],
        ['PPP1R21','STON1'],
        ['PPP1R36','ZBTB1'],
        ['PPP1R8','STX12'],
        ['PPP2CB','TG'],
        ['PPP2CB','TRIM28'],
        ['PPP2R1A','PRB3'],
        ['PPP3CB-AS1','TG'],
        ['PPP3CC','SORBS3'],
        ['PPP3R1','TMEM38B'],
        ['PPP6R1','TG'],
        ['PPTC7','VPS29'],
        ['PRAF2','TFE3'],
        ['PRB1','PRH1'],
        ['PRB1','PRH1-PRR4'],
        ['PRB1','PRH2'],
        ['PRB1','PRR4'],
        ['PRB1','PRSS8'],
        ['PRB1','PSAP'],
        ['PRB1','PTCH2'],
        ['PRB1','SFRP1'],
        ['PRB1','SMR3B'],
        ['PRB1','STATH'],
        ['PRB1','ZG16B'],
        ['PRB2','PRH1'],
        ['PRB2','PRH1-PRR4'],
        ['PRB2','PRH2'],
        ['PRB2','PRR4'],
        ['PRB2','SMR3B'],
        ['PRB2','STATH'],
        ['PRB2','SUN2'],
        ['PRB2','ZG16B'],
        ['PRB3','PRH1'],
        ['PRB3','PRH1-PRR4'],
        ['PRB3','PRH2'],
        ['PRB3','PRR4'],
        ['PRB3','SCARB2'],
        ['PRB3','SERF2'],
        ['PRB3','SLPI'],
        ['PRB3','SMR3A'],
        ['PRB3','SMR3B'],
        ['PRB3','SOAT1'],
        ['PRB3','STATH'],
        ['PRB3','SYVN1'],
        ['PRB4','PRH1'],
        ['PRB4','PRH1-PRR4'],
        ['PRB4','PRH2'],
        ['PRB4','PRR4'],
        ['PRB4','SLPI'],
        ['PRB4','SMR3B'],
        ['PRB4','STATH'],
        ['PRC1','VPS33B'],
        ['PRCP','RAB30'],
        ['PRDX1','TESK2'],
        ['PRELP','SFTPA2'],
        ['PRELP','TRG@'],
        ['PRH1','PRKCSH'],
        ['PRH1','PRR27'],
        ['PRH1','PSMF1'],
        ['PRH1','RBM47'],
        ['PRH1','RPL3'],
        ['PRH1','RPS19'],
        ['PRH1','RRBP1'],
        ['PRH1','SFRP1'],
        ['PRH1','SIX4'],
        ['PRH1','SLC6A8'],
        ['PRH1','SMR3B'],
        ['PRH1','SND1'],
        ['PRH1','SORL1'],
        ['PRH1','SQSTM1'],
        ['PRH1','STATH'],
        ['PRH1','TMBIM6'],
        ['PRH1','TNS1'],
        ['PRH1','UBC'],
        ['PRH1','ZG16B'],
        ['PRH1-PRR4','PRKCSH'],
        ['PRH1-PRR4','PRR27'],
        ['PRH1-PRR4','PSMF1'],
        ['PRH1-PRR4','RBM47'],
        ['PRH1-PRR4','RPL3'],
        ['PRH1-PRR4','RPS19'],
        ['PRH1-PRR4','SFRP1'],
        ['PRH1-PRR4','SIX4'],
        ['PRH1-PRR4','SLC20A2'],
        ['PRH1-PRR4','SLC6A8'],
        ['PRH1-PRR4','SMR3B'],
        ['PRH1-PRR4','SORL1'],
        ['PRH1-PRR4','SQSTM1'],
        ['PRH1-PRR4','STATH'],
        ['PRH1-PRR4','TMBIM6'],
        ['PRH1-PRR4','TNS1'],
        ['PRH1-PRR4','UBC'],
        ['PRH1-PRR4','ZG16B'],
        ['PRH2','PRR4'],
        ['PRH2','PTPRF'],
        ['PRH2','RBM47'],
        ['PRH2','SASH1'],
        ['PRH2','SF3B1'],
        ['PRH2','SFRP1'],
        ['PRH2','SH3BP4'],
        ['PRH2','SLC6A8'],
        ['PRH2','SMR3B'],
        ['PRH2','SQSTM1'],
        ['PRH2','STATH'],
        ['PRH2','TACC2'],
        ['PRICKLE2','RPRD1A'],
        ['PRIM2','SCARB2'],
        ['PRIM2','SERPINB9P1'],
        ['PRKAA1','TTC33'],
        ['PRKAB1','TMEM233'],
        ['PRKACB','SAMD13'],
        ['PRKACB','TXN2'],
        ['PRKAG2','PSD3'],
        ['PRKAG2','RHEB'],
        ['PRKAR1B','RASA3'],
        ['PRKCA','TPM4'],
        ['PRKCD','TIMM50'],
        ['PRKCH','PRPS2'],
        ['PRKCH','TG'],
        ['PRKCSH','PRR4'],
        ['PRKCSH','RAB11FIP1'],
        ['PRKCSH','TG'],
        ['PRKCZ','SKI'],
        ['PRKRIP1','RBAK'],
        ['PRKX','TG'],
        ['PRMT7','SLC7A6'],
        ['PRPF39','TOGARAM1'],
        ['PRPF8','PRSS1'],
        ['PRPF8','S100A9'],
        ['PRPSAP1','QRICH2'],
        ['PRR11','ZNF197'],
        ['PRR18','SFT2D1'],
        ['PRR27','SMR3B'],
        ['PRR4','RPS19'],
        ['PRR4','SH3BP4'],
        ['PRR4','SMR3B'],
        ['PRR4','STATH'],
        ['PRR4','TACC2'],
        ['PRR4','TAS2R15'],
        ['PRR4','TMBIM6'],
        ['PRR4','UBC'],
        ['PRR4','ZG16B'],
        ['PRRC2A','PTMS'],
        ['PRRC2A','SMR3B'],
        ['PRRX2','YKT6'],
        ['PRSS1','RNASE1'],
        ['PRSS1','RPL3'],
        ['PRSS1','RPS19'],
        ['PRSS1','RPS9'],
        ['PRSS1','SERPINE1'],
        ['PRSS1','SLC41A1'],
        ['PRSS1','SMARCB1'],
        ['PRSS1','SNRNP200'],
        ['PRSS1','SYCN'],
        ['PRSS1','TIMP2'],
        ['PRSS1','TPT1'],
        ['PRSS1','TSC2'],
        ['PRSS1','TUBGCP6'],
        ['PRSS1','UBE2R2-AS1'],
        ['PRSS1','USP9Y'],
        ['PRSS1','ZNF706'],
        ['PRSS2','RNASE1'],
        ['PRSS2','RRBP1'],
        ['PRSS2','RUVBL1'],
        ['PRSS2','SERPINE1'],
        ['PRSS2','SH3BP4'],
        ['PRSS2','SLC41A1'],
        ['PRSS2','SNRNP200'],
        ['PRSS2','SYCN'],
        ['PRSS2','TNS3'],
        ['PRSS2','TPT1'],
        ['PRSS2','UBE2R2-AS1'],
        ['PRSS3','TMEM45B'],
        ['PRSS3','TRB@'],
        ['PRSS3','UBE2R2'],
        ['PRSS42','PRSS50'],
        ['PRSS51','TNKS'],
        ['PRX','SFTPC'],
        ['PSAP','SFTPC'],
        ['PSAP','SMR3B'],
        ['PSAP','SPARC'],
        ['PSD4','SMR3B'],
        ['PSMA5','SORT1'],
        ['PSMA7','SS18L1'],
        ['PSMB7','SMR3B'],
        ['PSMB8','TAP1'],
        ['PSMD11','ZNF207'],
        ['PSMD12P','SHQ1'],
        ['PSME4','TG'],
        ['PSMF1','SIPA1L1'],
        ['PSMG3','PTRF'],
        ['PSPC1','XPO4'],
        ['PSPC1','ZMYM2'],
        ['PSPC1','ZMYM5'],
        ['PSPH','SEPT7P2'],
        ['PTBP1','RNF126'],
        ['PTBP1','SBNO2'],
        ['PTBP3','SUSD1'],
        ['PTCSC2','RNF144B'],
        ['PTCSC2','TRG@'],
        ['PTCSC2','XPA'],
        ['PTEN','RPL11'],
        ['PTGDS','VAMP2'],
        ['PTGR1','RNASE10'],
        ['PTHLH','TGM1'],
        ['PTK6','ZNF611'],
        ['PTK7','ZG16B'],
        ['PTMA','RNF44'],
        ['PTMA','S100A9'],
        ['PTMA','SLC38A2'],
        ['PTMA','SMR3B'],
        ['PTMA','SRSF5'],
        ['PTMA','SRSF7'],
        ['PTMA','TCF25'],
        ['PTMA','TG'],
        ['PTMA','TGFB1'],
        ['PTMA','TMC5'],
        ['PTMA','WDR1'],
        ['PTMA','ZBTB2'],
        ['PTMA','ZEB2'],
        ['PTMS','R3HDM4'],
        ['PTMS','RPS2'],
        ['PTMS','RRP36'],
        ['PTMS','SEPT9'],
        ['PTMS','SULF2'],
        ['PTMS','TMED2'],
        ['PTMS','TMEM99'],
        ['PTMS','TRIM28'],
        ['PTMS','UBE2M'],
        ['PTMS','VIM'],
        ['PTN','TPK1'],
        ['PTOV1','TG'],
        ['PTP4A1','SRSF5'],
        ['PTPN14','SMYD2'],
        ['PTPN22','RSBN1'],
        ['PTPN3','YBX1P6'],
        ['PTPN4','SP140L'],
        ['PTPRF','PTPRS'],
        ['PTPRJ','TG'],
        ['PTPRM','TG'],
        ['PTPRN2','TRB@'],
        ['PTPRS','SOX4'],
        ['PTPRS','STK24'],
        ['PTRF','SCARB2'],
        ['PTRF','SCUBE3'],
        ['PTRF','SLC35E2B'],
        ['PTRF','SLX4'],
        ['PTRF','SNX15'],
        ['PTRF','SOD2'],
        ['PTRF','SPG7'],
        ['PTRF','SYN3'],
        ['PTRF','TAPBP'],
        ['PTRF','TMOD3'],
        ['PTRF','TNFRSF10B'],
        ['PTRF','TTL'],
        ['PTRF','UAP1L1'],
        ['PTRF','ZSCAN16-AS1'],
        ['PUM1','SDC3'],
        ['PUM1','SMR3B'],
        ['PVR','SERPINE1'],
        ['PVRIG','STAG3L5P'],
        ['PVRIG','STAG3L5P-PVRIG2P-PILRB'],
        ['PVRIG2P','STAG3'],
        ['PVRL1','TMEM99'],
        ['PVRL2','SERINC5'],
        ['PVRL2','TOMM40'],
        ['PWRN1','PWRN3'],
        ['PWRN1','PWRN4'],
        ['PWWP2B','SKI'],
        ['PXK','RPP14'],
        ['PXT1','STK38'],
        ['PXYLP1','SPSB4'],
        ['PYGB','TRB@'],
        ['PYGM','SF1'],
        ['PYY','TMEM101'],
        ['QPRT','TMED4'],
        ['QRICH1','SOD2'],
        ['R3HDM1','TRAPPC12'],
        ['R3HDM1','UBXN4'],
        ['R3HDM2','STAC3'],
        ['RAB11B','SEPHS1'],
        ['RAB11B','TG'],
        ['RAB11FIP1','ZMAT3'],
        ['RAB1A','SFTPB'],
        ['RAB1A','STAT1'],
        ['RAB1A','TG'],
        ['RAB1B','RAB1C'],
        ['RAB21','RNF24'],
        ['RAB27A','RSL24D1'],
        ['RAB29','TRABD2A'],
        ['RAB2A','SGMS1'],
        ['RAB2B','TOP3A'],
        ['RAB31','RASA3'],
        ['RAB31','VAPA'],
        ['RAB35','TG'],
        ['RAB35','THEMIS2'],
        ['RAB3D','VMAC'],
        ['RAB40C','RNF166'],
        ['RAB40C','TECPR1'],
        ['RAB43P1','RABGAP1'],
        ['RAB7A','SAMD14'],
        ['RAB7A','ZNF793'],
        ['RAB9A','TCEANC'],
        ['RAC1','TENM1'],
        ['RAC1','TG'],
        ['RACK1','SRSF5'],
        ['RACK1','TRIM52'],
        ['RAD23A','UBA52'],
        ['RAD51B','SMR3B'],
        ['RAI14','RTN4'],
        ['RALY-AS1','STAG3'],
        ['RAP1B','RBPJ'],
        ['RAPGEF1','TRB@'],
        ['RARA','RCN1'],
        ['RASA3','TFDP1'],
        ['RASSF3','ZG16B'],
        ['RASSF4','ZNF22'],
        ['RASSF8','SSPN'],
        ['RBAK','RNF216'],
        ['RBAK-RBAKDN','RNF216'],
        ['RBBP4','TRA@'],
        ['RBBP9','TRA@'],
        ['RBFA','ZMYND11'],
        ['RBFOX2','RPL41'],
        ['RBM14','RBM4'],
        ['RBM14','RBM4B'],
        ['RBM15B','RPLP1'],
        ['RBM25','S100A9'],
        ['RBM25','STAT5B'],
        ['RBM3','TBC1D25'],
        ['RBM38','SUMO2'],
        ['RBM5','RBM6'],
        ['RBM5','RHOA'],
        ['RBM6','RHOA'],
        ['RBM8A','SEH1L'],
        ['RBMS2','SPRYD4'],
        ['RBMS2','TG'],
        ['RBP4','SERPINA1'],
        ['RBP4','TXNL4B'],
        ['RBP7','VSIG1'],
        ['RBPJL','TRB@'],
        ['RBPMS','TUBBP1'],
        ['RBPMS-AS1','TNNT2'],
        ['RCAN1','TG'],
        ['RCBTB1','SMARCA4'],
        ['RCOR3','TG'],
        ['RDH11','TRG@'],
        ['RDH11','VTI1B'],
        ['RDH16','SDR9C7'],
        ['RDX','SS18'],
        ['RECQL5','SENP6'],
        ['REG1A','REG3A'],
        ['REG1A','TRB@'],
        ['REG1A','UBE2R2-AS1'],
        ['REPS2','SETDB1'],
        ['RET','ZNF750'],
        ['REXO2','RPUSD4'],
        ['RFC2','TPM1'],
        ['RFFL','RUBCN'],
        ['RFFL','TG'],
        ['RFPL3S','RTCB'],
        ['RFWD3','RSL1D1'],
        ['RFX1','TOB1'],
        ['RGCC','SRGN'],
        ['RGS17','SORL1'],
        ['RHBDF1','RHBDF1P1'],
        ['RHBG','TSACC'],
        ['RHCG','SPRR3'],
        ['RHEB','SHLD2'],
        ['RHEX','ZNF888'],
        ['RHOA','TKT'],
        ['RHOF','UBC'],
        ['RHOH','SLC31A1'],
        ['RHOH','TPM4'],
        ['RHOQ','TMEM247'],
        ['RIOK3','SRGN'],
        ['RIPK1','SERPINB9'],
        ['RLN1','RLN2'],
        ['RMI1','U62317.4'],
        ['RMND1','TBPL1'],
        ['RMND1','ZBTB2'],
        ['RN7SL151P','RN7SL2'],
        ['RN7SL2','RN7SL381P'],
        ['RNASE1','TNS1'],
        ['RNASE1','TRB@'],
        ['RNASEH2B','TG'],
        ['RNASET2','SRGN'],
        ['RNF125','RNF138'],
        ['RNF130','RTN4RL1'],
        ['RNF139','TRMT12'],
        ['RNF150','TPM1'],
        ['RNF151','TBL3'],
        ['RNF152','TAFA2'],
        ['RNF168','UBXN7'],
        ['RNF169','TCIM'],
        ['RNF185','UQCR10'],
        ['RNF19A','TG'],
        ['RNF20','TMEM246-AS1'],
        ['RNF212','TMED11P'],
        ['RNF212B','TMEM184A'],
        ['RNF213','SFTPC'],
        ['RNF216','XKR8'],
        ['RNF216P1','XKR8'],
        ['RNF220','TRIM28'],
        ['RNF24','SLC35E1'],
        ['RNF31','THADA'],
        ['RNF4','TG'],
        ['RNF40','ZG16B'],
        ['RNF5','SFTPA1'],
        ['RNF5','SFTPC'],
        ['RNGTT','UTRN'],
        ['RNH1','VAMP2'],
        ['RNPS1','SRSF5'],
        ['RNPS1','YWHAE'],
        ['ROCK2','SFTPB'],
        ['RPAP2','TRA@'],
        ['RPL11','TCEB3'],
        ['RPL13','SPG7'],
        ['RPL13A','TG'],
        ['RPL14','S100A16'],
        ['RPL14','SRP14'],
        ['RPL14','VSIG1'],
        ['RPL15','SMR3B'],
        ['RPL15','TG'],
        ['RPL18A','SMR3B'],
        ['RPL18A','TRB@'],
        ['RPL24','ZBTB11'],
        ['RPL27','RUNDC1'],
        ['RPL28','TMEM190'],
        ['RPL3','TG'],
        ['RPL3','TRB@'],
        ['RPL30','TG'],
        ['RPL37','TNFSF10'],
        ['RPL37A','SRGN'],
        ['RPL37A','TRB@'],
        ['RPL37A','VEGFA'],
        ['RPL38','TTYH2'],
        ['RPL4','TG'],
        ['RPL7L1','RPS2P55'],
        ['RPL7L1','SNX8'],
        ['RPL7L1','WBP2NL'],
        ['RPL8','SFTPC'],
        ['RPS11','SMR3B'],
        ['RPS12','S100A9'],
        ['RPS14','S100A9'],
        ['RPS18','TG'],
        ['RPS19','SNUPN'],
        ['RPS19','SNX6'],
        ['RPS19','STATH'],
        ['RPS19','TG'],
        ['RPS2','STK39'],
        ['RPS2','YWHAQ'],
        ['RPS20','RSF1'],
        ['RPS23','ZNF888'],
        ['RPS25','SMR3B'],
        ['RPS27A','UBA52'],
        ['RPS29','SLC34A2'],
        ['RPS29','SYN3'],
        ['RPS4X','RPS4XP1'],
        ['RPS4X','RPS4XP19'],
        ['RPS4X','RPS4XP2'],
        ['RPS4X','RPS4XP5'],
        ['RPS6KA2','TG'],
        ['RPS9','TRB@'],
        ['RPS9','UBB'],
        ['RRBP1','TG'],
        ['RRN3P2','SNN'],
        ['RSPH1','VPS13A'],
        ['RSPH4A','RWDD1'],
        ['RUSC2','SMR3B'],
        ['RUVBL1','SMR3B'],
        ['RUVBL1','TRB@'],
        ['RUVBL1','TRBC2'],
        ['RUVBL1','TRBV25-1'],
        ['RUVBL2','SMR3B'],
        ['RWDD3','TLCD4'],
        ['RWDD3','TMEM56'],
        ['RXRA','TG'],
        ['RYBP','SFTPC'],
        ['RYR2','TG'],
        ['S100A10','TNS3'],
        ['S100A13','S100A16'],
        ['S100A13','TNNT2'],
        ['S100A16','S100A2'],
        ['S100A9','SLC25A37'],
        ['S100A9','SLC4A1'],
        ['S100A9','TKT'],
        ['S100A9','TOPBP1'],
        ['S100A9','YWHAZ'],
        ['S100P','YTHDF2'],
        ['SAA1','SERPINA1'],
        ['SAA1','TIMP3'],
        ['SAA2','SERPINA1'],
        ['SAA2','TMBIM6'],
        ['SAA2-SAA4','SERPINA1'],
        ['SAA4','SERPINA1'],
        ['SAFB','SAFB2'],
        ['SAG','VPS13C'],
        ['SAMD5','SASH1'],
        ['SAMD8','SELT'],
        ['SAMD8','VDAC2'],
        ['SAMHD1','TRA@'],
        ['SAMSN1','USP25'],
        ['SAR1A','SAR1AP4'],
        ['SAR1A','TG'],
        ['SAR1A','TYSND1'],
        ['SARS','SMR3B'],
        ['SARS','TLR8-AS1'],
        ['SBF2','TG'],
        ['SCAF4','SERGEF'],
        ['SCAMP4','SFTPB'],
        ['SCAND2P','ZSCAN2'],
        ['SCARB1','UBC'],
        ['SCD','SORBS1'],
        ['SCD5','SDC3'],
        ['SCD5','SEPT4-AS1'],
        ['SCD5','TPM4'],
        ['SCG5','TRB@'],
        ['SCGB1B2P','SCGB2B2'],
        ['SCNN1A','TNFRSF1A'],
        ['SCP2','TUBB'],
        ['SCUBE3','TG'],
        ['SCUBE3','ZNF76'],
        ['SDC2','VMP1'],
        ['SDC3','WWOX'],
        ['SDCBP2-AS1','ZNF587B'],
        ['SDF4','TNFRSF4'],
        ['SDHA','SMR3B'],
        ['SEC11A','SSR3'],
        ['SEC1P','ZNF826P'],
        ['SEC24C','SPAG9'],
        ['SEC61A1','TRB@'],
        ['SEC62','SMR3B'],
        ['SEL1L','SFTPB'],
        ['SELENOP','TPT1'],
        ['SELPLG','SFTPA1'],
        ['SELPLG','TMEM119'],
        ['SEMA3B','U73166.1'],
        ['SEMA3B','U73166.2'],
        ['SEMA6A-AS1','SEMA6A-AS2'],
        ['SENP1','TG'],
        ['SENP3-EIF4A1','TNFSF13'],
        ['SENP3-EIF4A1','UBL5'],
        ['SEPN1','SFTPB'],
        ['SEPN1','TRB@'],
        ['SEPT11','SERPINE1'],
        ['SEPT2','SKIL'],
        ['SEPT2','TG'],
        ['SEPT9','SFTPC'],
        ['SEPT9','SMR3B'],
        ['SEPT9','ZG16B'],
        ['SERF1A','SMN1'],
        ['SERINC1','SUMF2'],
        ['SERINC2','SMR3B'],
        ['SERINC3','TG'],
        ['SERP1','SMR3B'],
        ['SERP1','TG'],
        ['SERPINA1','SERPINA11'],
        ['SERPINA1','SERPINA5'],
        ['SERPINA1','SERPINC1'],
        ['SERPINA1','SERPING1'],
        ['SERPINA1','TAT'],
        ['SERPINA1','TGM2'],
        ['SERPINA1','TMBIM6'],
        ['SERPINA1','TXNL4B'],
        ['SERPINA1','UGT2B4'],
        ['SERPINB3','SERPINB4'],
        ['SERPINB6','TKT'],
        ['SERPINE1','SHROOM3'],
        ['SERPINE1','TRB@'],
        ['SERPINE1','TRBV25-1'],
        ['SERPINE1','TXNDC5'],
        ['SERPINF1','TRPM8'],
        ['SERPINF2','WDR81'],
        ['SERPING1','TXNL4B'],
        ['SET','SETP11'],
        ['SET','SETP3'],
        ['SF1','TRB@'],
        ['SF3B1','WWOX'],
        ['SFI1','SMR3B'],
        ['SFN','SLFN5'],
        ['SFN','TNS1'],
        ['SFPQ','SMR3B'],
        ['SFPQ','SRGN'],
        ['SFPQ','SUGP2'],
        ['SFRP1','SMAD6'],
        ['SFRP1','SMR3B'],
        ['SFT2D2','TRA@'],
        ['SFTPA1','SFTPB'],
        ['SFTPA1','SFTPC'],
        ['SFTPA1','SPTBN1'],
        ['SFTPA1','SQSTM1'],
        ['SFTPA1','SUSD2'],
        ['SFTPA1','TAGLN'],
        ['SFTPA1','TMBIM6'],
        ['SFTPA2','SFTPC'],
        ['SFTPA2','SLK'],
        ['SFTPA2','SOCS3'],
        ['SFTPA2','STAT3'],
        ['SFTPA2','SYN3'],
        ['SFTPA2','USP36'],
        ['SFTPA2','ZFP36'],
        ['SFTPA2','ZMIZ1'],
        ['SFTPB','SFTPC'],
        ['SFTPB','SPTBN1'],
        ['SFTPB','SQSTM1'],
        ['SFTPB','SUMF2'],
        ['SFTPB','SYN3'],
        ['SFTPB','SYNE1'],
        ['SFTPB','TBCEL'],
        ['SFTPB','TFPI'],
        ['SFTPB','THAP5'],
        ['SFTPB','TIMM10B'],
        ['SFTPB','TMEM119'],
        ['SFTPB','TMEM50B'],
        ['SFTPB','TNFRSF10B'],
        ['SFTPB','TTLL3'],
        ['SFTPB','TYROBP'],
        ['SFTPB','WARS'],
        ['SFTPB','WFS1'],
        ['SFTPB','XXYLT1'],
        ['SFTPB','ZFP36'],
        ['SFTPB','ZFP36L1'],
        ['SFTPB','ZFP36L2'],
        ['SFTPB','ZNF562'],
        ['SFTPC','SH3D19'],
        ['SFTPC','SLC34A2'],
        ['SFTPC','SLCO2A1'],
        ['SFTPC','SPARC'],
        ['SFTPC','SPOCK2'],
        ['SFTPC','SUPT5H'],
        ['SFTPC','TACSTD2'],
        ['SFTPC','TSIX'],
        ['SFTPC','VIM'],
        ['SFTPC','ZFP36'],
        ['SFTPC','ZNF609'],
        ['SFXN3','TPM4'],
        ['SH3BGR','WRB'],
        ['SH3BGRL2','SMR3B'],
        ['SH3BP4','TRB@'],
        ['SH3BP4','TRBC2'],
        ['SH3BP4','TRBV25-1'],
        ['SH3GLB1','WSB1'],
        ['SH3KBP1','TYROBP'],
        ['SHANK2-AS1','SHANK2-AS2'],
        ['SHISA7','UBE2S'],
        ['SHISA9','U91319.1'],
        ['SHQ1','TG'],
        ['SIDT2','TAGLN'],
        ['SIGLEC16','SIGLEC17P'],
        ['SIGLEC16','WDR35'],
        ['SIGLEC6','Z99129.3'],
        ['SIPA1L3','TG'],
        ['SIPA1L3','TRB@'],
        ['SIRPA','SMR3B'],
        ['SIRT2','XPNPEP1'],
        ['SKA2','TRIM37'],
        ['SLAIN2','SLC10A4'],
        ['SLAIN2','ST20'],
        ['SLAIN2','ST20-MTHFS'],
        ['SLC13A3','TP53RK'],
        ['SLC14A1','SORCS1'],
        ['SLC16A3','TG'],
        ['SLC17A1','SLC17A3'],
        ['SLC19A1','SUMO3'],
        ['SLC1A4','TG'],
        ['SLC20A2','WDR45'],
        ['SLC25A23','SLC25A41'],
        ['SLC25A23','TG'],
        ['SLC25A29','TPO'],
        ['SLC25A29','YY1'],
        ['SLC25A3','SRGN'],
        ['SLC25A3','SUGCT'],
        ['SLC25A3','TET1'],
        ['SLC25A3','TOM1'],
        ['SLC25A32','UBR5'],
        ['SLC25A38','TG'],
        ['SLC25A4','SLC25A6'],
        ['SLC25A5','SLC25A6'],
        ['SLC25A5','TRB@'],
        ['SLC26A4','SLC26A4-AS1'],
        ['SLC26A4','TG'],
        ['SLC26A7','TG'],
        ['SLC26A8','SNX29'],
        ['SLC29A3','UNC5B'],
        ['SLC2A11','SMARCB1'],
        ['SLC2A2','TNIK'],
        ['SLC2A6','SMC6'],
        ['SLC30A9','TG'],
        ['SLC33A1','UBP1'],
        ['SLC35E3','THNSL1'],
        ['SLC35E3','ZKSCAN1'],
        ['SLC35F6','TPM4'],
        ['SLC39A10','TG'],
        ['SLC39A8','TRB@'],
        ['SLC3A2','TG'],
        ['SLC41A1','TRB@'],
        ['SLC41A3','TG'],
        ['SLC43A3','TG'],
        ['SLC44A1','TG'],
        ['SLC4A1','SORL1'],
        ['SLC5A12','SOX2-OT'],
        ['SLC5A2','TG'],
        ['SLC5A5','SMR3B'],
        ['SLC6A14','SOCS2'],
        ['SLC6A16','TEAD2'],
        ['SLC7A10','TAPBP'],
        ['SLC7A5','SMG1'],
        ['SLC8A1','TPM1'],
        ['SLC9A7','VPS41'],
        ['SLC9A8','ZNFX1-AS1'],
        ['SLFN11','TG'],
        ['SLPI','SMR3B'],
        ['SLTM','TG'],
        ['SMAD2','TG'],
        ['SMAD2','TRA@'],
        ['SMAD2','TRB@'],
        ['SMAD4','SSB'],
        ['SMAP2','ZFP69B'],
        ['SMARCA2','TG'],
        ['SMARCB1','TRB@'],
        ['SMG1','ZNF623'],
        ['SMR3B','SNTB1'],
        ['SMR3B','SPINT2'],
        ['SMR3B','SPSB3'],
        ['SMR3B','SPTBN1'],
        ['SMR3B','SREBF2'],
        ['SMR3B','SRRM2'],
        ['SMR3B','STAT6'],
        ['SMR3B','STATH'],
        ['SMR3B','STIM1'],
        ['SMR3B','SUPT5H'],
        ['SMR3B','SURF4'],
        ['SMR3B','SYN3'],
        ['SMR3B','TBL1XR1'],
        ['SMR3B','TGOLN2'],
        ['SMR3B','THRA'],
        ['SMR3B','TIMP3'],
        ['SMR3B','TLN2'],
        ['SMR3B','TMED10'],
        ['SMR3B','TMEM120B'],
        ['SMR3B','TNS1'],
        ['SMR3B','TOB1'],
        ['SMR3B','TPD52L1'],
        ['SMR3B','TRB@'],
        ['SMR3B','TRIM29'],
        ['SMR3B','TRPC4AP'],
        ['SMR3B','TSC2'],
        ['SMR3B','TSIX'],
        ['SMR3B','TXNDC5'],
        ['SMR3B','UQCRB'],
        ['SMR3B','USP34'],
        ['SMR3B','UVSSA'],
        ['SMR3B','VIM'],
        ['SMR3B','WFDC2'],
        ['SMR3B','WNK2'],
        ['SMR3B','WWC1'],
        ['SMR3B','YBX3'],
        ['SMR3B','ZCCHC14'],
        ['SMR3B','ZFP36L1'],
        ['SMR3B','ZG16B'],
        ['SMR3B','ZNF827'],
        ['SMTN','VARS2'],
        ['SMURF2','SMURF2P1-LRRC37BP1'],
        ['SNAP23','TRIM58'],
        ['SNAP25','VSNL1'],
        ['SNAP25-AS1','SYN2'],
        ['SND1','ZNF704'],
        ['SNHG14','SNURF'],
        ['SNRPB','SNRPN'],
        ['SNRPB','SNURF'],
        ['SNRPD2','TG'],
        ['SNRPN','WDR4'],
        ['SNTB2','VPS4A'],
        ['SNURF','WDR4'],
        ['SNX2','SNX24'],
        ['SNX29','TG'],
        ['SNX9','SYNJ2'],
        ['SOD2','SOGA1'],
        ['SOD2','TRA@'],
        ['SOD2','TRB@'],
        ['SOD2','TTN'],
        ['SOGA1','TG'],
        ['SORBS2','TG'],
        ['SORL1','TFDP2'],
        ['SORL1','ZKSCAN1'],
        ['SP100','SP140L'],
        ['SPAG9','ZNFX1'],
        ['SPARC','TG'],
        ['SPARC','TNS1'],
        ['SPARC','TPM2'],
        ['SPARC','VWF'],
        ['SPATA3','ZNF483'],
        ['SPATS2','TMEM99'],
        ['SPEM2','SPEM3'],
        ['SPEM3','TMEM102'],
        ['SPG7','TMEM189'],
        ['SPIN1','TG'],
        ['SPINK1','TRB@'],
        ['SPINK5','SPRR3'],
        ['SPP1','UBE2D1'],
        ['SPPL3','TG'],
        ['SPRN','SYCE1'],
        ['SPRR1A','SPRR3'],
        ['SPRR1B','SPRR3'],
        ['SPRY4','TG'],
        ['SPSB3','TAGLN'],
        ['SQLE','ZNF572'],
        ['SQSTM1','STEAP2'],
        ['SREBF2','STATH'],
        ['SREBF2','TG'],
        ['SRGAP2B','SRGAP2C'],
        ['SRGN','TKT'],
        ['SRGN','USP15'],
        ['SRGN','XIRP2'],
        ['SRI','STEAP4'],
        ['SRP19','ZRSR2'],
        ['SRRM2','TIMP3'],
        ['SRRM2','TMEM99'],
        ['SRSF10','ZFP69'],
        ['SRSF4','TMBIM6'],
        ['SSBP1','WEE2'],
        ['SSH2','STK4'],
        ['SSR2','TRIM25'],
        ['SSR2','UBQLN4'],
        ['SSR3','WIPI2'],
        ['SSR3','ZNF684'],
        ['ST3GAL1','TCEA1'],
        ['ST3GAL1','TPO'],
        ['ST3GAL1','UBB'],
        ['ST3GAL3','TG'],
        ['ST6GAL1','TG'],
        ['STAG3','STAG3L5P'],
        ['STAG3','STAG3L5P-PVRIG2P-PILRB'],
        ['STAG3','WWOX'],
        ['STARD7','TPT1'],
        ['STARD7','ZFHX3'],
        ['STAT3','TRA@'],
        ['STAT3','TRB@'],
        ['STATH','TMED3'],
        ['STATH','TSC22D1'],
        ['STATH','UBE2Z'],
        ['STATH','UBXN4'],
        ['STATH','WASHC4'],
        ['STC1','TG'],
        ['STIP1','TAPBP'],
        ['STK10','UBE2D2'],
        ['STK3','VPS13B'],
        ['STK4-AS1','TOMM34'],
        ['STMN4','TRIM35'],
        ['STRA6LP','SUGT1'],
        ['STXBP4','TOM1L1'],
        ['STYXL1','TMEM120A'],
        ['SUGT1','SUGT1P4-STRA6LP'],
        ['SULT2A1','TAT'],
        ['SUMF2','TG'],
        ['SUMO1','SUMO1P4'],
        ['SUMO3','UBE2G2'],
        ['SURF1','SURF4'],
        ['SVEP1','TIMP3'],
        ['SVEP1','TXNDC8'],
        ['SVIL','ZG16B'],
        ['SYBU','TRB@'],
        ['SYCN','TRB@'],
        ['SYN1','TUBB4A'],
        ['SYN3','TG'],
        ['SYN3','TNS1'],
        ['SYNC','TRA@'],
        ['SYNE1','TG'],
        ['SYT8','TNNI2'],
        ['SYTL4','TG'],
        ['SYTL4','TSPAN6'],
        ['TAB2','TG'],
        ['TACC2','TG'],
        ['TAMM41','VGLL4'],
        ['TAPBP','TTC39B'],
        ['TAPBP','WWOX'],
        ['TAS2R5','TAS2R6'],
        ['TAT-AS1','TF'],
        ['TBC1D23','TMEM30C'],
        ['TBC1D29','USP6'],
        ['TBC1D9','TM4SF1'],
        ['TBC1D9','TNRC18P1'],
        ['TBCD','TBL1XR1'],
        ['TBCE','TG'],
        ['TBX15','WARS2'],
        ['TCF25','TRB@'],
        ['TDRD10','UBE2Q1-AS1'],
        ['TDRP','TG'],
        ['TECPR1','TRB@'],
        ['TELO2','UBE2I'],
        ['TESC','ZG16B'],
        ['TEX41','ZNF417'],
        ['TF','TXNL4B'],
        ['TFDP1','TMCO3'],
        ['TFDP1','ZEB2'],
        ['TFDP2','XRN1'],
        ['TFF1','TFF2'],
        ['TG','TIE1'],
        ['TG','TM2D3'],
        ['TG','TMCC3'],
        ['TG','TMED10'],
        ['TG','TPCN1'],
        ['TG','TSPAN14'],
        ['TG','TTC19'],
        ['TG','UBA1'],
        ['TG','UBE2H'],
        ['TG','UBE2Z'],
        ['TG','UQCC2'],
        ['TG','USP39'],
        ['TG','USP40'],
        ['TG','USP54'],
        ['TG','UTRN'],
        ['TG','VAC14'],
        ['TG','VAV3'],
        ['TG','VMP1'],
        ['TG','VPS45'],
        ['TG','VWF'],
        ['TG','WAC'],
        ['TG','WDR86'],
        ['TG','WHSC1'],
        ['TG','XPO1'],
        ['TG','XRCC5'],
        ['TG','YWHAB'],
        ['TG','ZFAND3'],
        ['TG','ZNF12'],
        ['TG','ZNF704'],
        ['TG','ZSCAN32'],
        ['TGM4','WDR6'],
        ['THSD4','TPD52'],
        ['THY1','VIM-AS1'],
        ['TIMP1','TIMP3'],
        ['TIMP1','VIM'],
        ['TIMP3','TPM2'],
        ['TLE1','TLE1P1'],
        ['TLK2','ZNF880'],
        ['TLN1','TRB@'],
        ['TMC6','TRIM69'],
        ['TMCO2','WDR60'],
        ['TMED2-DT','WSB1'],
        ['TMED4','ZNFX1'],
        ['TMEM131','YIPF5'],
        ['TMEM189-UBE2V1','UBE2V1P6'],
        ['TMEM198B','YWHAB'],
        ['TMEM220','TMEM97'],
        ['TMEM225B','ZNF655'],
        ['TMEM225B','ZSCAN25'],
        ['TMEM38B','TRA@'],
        ['TMEM38B','TRB@'],
        ['TMEM41B','ZNF217'],
        ['TMEM45A','TRA@'],
        ['TMEM45B','TRA@'],
        ['TMPO','TRB@'],
        ['TMPRSS5','ZW10'],
        ['TMSB10','TMSB4X'],
        ['TMSB10','TPM2'],
        ['TNFSF10','TVP23B'],
        ['TNNT1','TNNT2'],
        ['TNS3','TRB@'],
        ['TNS3','TRBC2'],
        ['TNS3','TRBV25-1'],
        ['TOGARAM1','TSTD3'],
        ['TOMM70','ZNF490'],
        ['TOP1','TPRXL'],
        ['TPI1','TRIM27'],
        ['TPM1','UQCRB'],
        ['TPM1','XIRP2'],
        ['TPM1','ZNF284'],
        ['TPM3P9','ZNF765'],
        ['TPM4','UBXN2B'],
        ['TPO','VWF'],
        ['TPPP','ZDHHC11B'],
        ['TPPP3','ZDHHC1'],
        ['TPST2','TRB@'],
        ['TPT1','TRB@'],
        ['TPTE2','TPTEP2-CSNK1E'],
        ['TRA@','TRIM69'],
        ['TRA@','TXNDC15'],
        ['TRA@','UBE2V2'],
        ['TRA@','USP33'],
        ['TRA@','Z99755.3'],
        ['TRAC','TRAJ5'],
        ['TRAC','TRAV10'],
        ['TRAC','TRAV12-1'],
        ['TRAC','TRAV12-2'],
        ['TRAC','TRAV12-3'],
        ['TRAC','TRAV14DV4'],
        ['TRAC','TRAV17'],
        ['TRAC','TRAV19'],
        ['TRAC','TRAV21'],
        ['TRAC','TRAV22'],
        ['TRAC','TRAV25'],
        ['TRAC','TRAV26-1'],
        ['TRAC','TRAV35'],
        ['TRAC','TRAV9-2'],
        ['TRAJ39','TRAV12-2'],
        ['TRAJ5','TRAV19'],
        ['TRAV18','TRAV19'],
        ['TRB@','TRBC2'],
        ['TRB@','TREH'],
        ['TRB@','TUBB'],
        ['TRB@','UBE2R2-AS1'],
        ['TRB@','ULK1'],
        ['TRB@','XIAP'],
        ['TRB@','ZFP36L1'],
        ['TRBC2','TRBV10-2'],
        ['TRBC2','TRBV19'],
        ['TRBC2','TRBV2'],
        ['TRBC2','TRBV20-1'],
        ['TRBC2','TRBV28'],
        ['TRBC2','TRBV3-1'],
        ['TRBC2','TRBV30'],
        ['TRBC2','TRBV4-1'],
        ['TRBC2','TRBV4-2'],
        ['TRBC2','TRBV5-1'],
        ['TRBC2','TRBV6-5'],
        ['TRBC2','TRBV7-3'],
        ['TRBC2','TRBV9'],
        ['TREM1','TREML3P'],
        ['TRIM11','TRIM17'],
        ['TRIM22','TRIM6-TRIM34'],
        ['TRIM8','YIPF5'],
        ['TRPM7','USP50'],
        ['TTC32','WDR35'],
        ['TTC38','UQCRB'],
        ['TTTY15','USP9Y'],
        ['TUBA1C','TUBA3C'],
        ['TUBA1C','TUBAP2'],
        ['TUBA4A','TUBAP2'],
        ['TUBB','TUBB4B'],
        ['TUBB','TUBB6'],
        ['TUBB7P','TUBB8'],
        ['TXNL4B','XXBAC-BPG116M5.17'],
        ['TYW1','TYW1B'],
        ['UBA2','WTIP'],
        ['UBA52','UBB'],
        ['UBA52','UBBP4'],
        ['UBA52','UBC'],
        ['UBA52','UBL4A'],
        ['UBA52','ZFAND4'],
        ['UBE2I','UNKL'],
        ['UGDH','YWHAZ'],
        ['UGT1A1','UGT1A2P'],
        ['UMODL1','ZNF295-AS1'],
        ['UPP2','UPP2-IT1'],
        ['UQCRC2','ZSWIM7'],
        ['USP10','ZDHHC7'],
        ['USP32','USP6'],
        ['USP34','XPO1'],
        ['USP9X','ZNF268'],
        ['VN1R37P','VN1R40P'],
        ['VPS53','WDR70'],
        ['VPS8','ZNF699'],
        ['VRK2','WDPCP'],
        ['VWF','ZG16B'],
        ['WAC-AS1','Z99572.1'],
        ['WASH2P','WASH5P'],
        ['WASH5P','WASH6P'],
        ['WDR41','WWTR1'],
        ['WSB1','ZNF264'],
        ['XPOT','ZNF271P'],
        ['YPEL5','ZEB2'],
        ['Z93930.2','ZNRF3'],
        ['ZBTB8A','ZNF440'],
        ['ZC2HC1A','ZFX'],
        ['ZC3H18','ZFPM1'],
        ['ZDHHC12','ZER1'],
        ['ZFP30','ZNF571'],
        ['ZFP30','ZNF607'],
        ['ZFP41','ZNF674'],
        ['ZFP90','ZNF674'],
        ['ZFS-4','ZNF674'],
        ['ZKSCAN5','ZNF674'],
        ['ZKSCAN7','ZNF852'],
        ['ZKSCAN8','ZNF192P1'],
        ['ZNF137P','ZNF701'],
        ['ZNF141','ZNF876P'],
        ['ZNF160','ZNF347'],
        ['ZNF208','ZNF676'],
        ['ZNF212','ZNF783'],
        ['ZNF224','ZNF225'],
        ['ZNF226','ZNF234'],
        ['ZNF227','ZNF233'],
        ['ZNF229','ZNF285'],
        ['ZNF232','ZNF674'],
        ['ZNF257','ZNF429'],
        ['ZNF267','ZNF720'],
        ['ZNF28','ZNF320'],
        ['ZNF28','ZNF765'],
        ['ZNF28','ZNF845'],
        ['ZNF28','ZNF860'],
        ['ZNF32','ZNF674'],
        ['ZNF32-AS3','ZNF485'],
        ['ZNF331','ZNF765'],
        ['ZNF33A','ZNF33B'],
        ['ZNF345','ZNF790-AS1'],
        ['ZNF350','ZNF674'],
        ['ZNF354C','ZNF879'],
        ['ZNF37A','ZNF37BP'],
        ['ZNF415','ZNF674'],
        ['ZNF429','ZNF738'],
        ['ZNF430','ZNF738'],
        ['ZNF431','ZNF726'],
        ['ZNF433-AS1','ZNF625'],
        ['ZNF433-AS1','ZNF625-ZNF20'],
        ['ZNF439','ZNF69'],
        ['ZNF439','ZNF700'],
        ['ZNF440','ZNF69'],
        ['ZNF440','ZNF700'],
        ['ZNF486','ZNF826P'],
        ['ZNF517','ZNF7'],
        ['ZNF525','ZNF83'],
        ['ZNF525','ZNF845'],
        ['ZNF573','ZNF607'],
        ['ZNF578','ZNF611'],
        ['ZNF586','ZNF587'],
        ['ZNF586','ZNF776'],
        ['ZNF587','ZNF587B'],
        ['ZNF587B','ZNF880'],
        ['ZNF610','ZNF880'],
        ['ZNF616','ZNF836'],
        ['ZNF620','ZNF621'],
        ['ZNF621','ZNF674'],
        ['ZNF674','ZNF675'],
        ['ZNF674','ZNF702P'],
        ['ZNF674','ZNF78L1'],
        ['ZNF674','ZNF81'],
        ['ZNF674','ZNF93'],
        ['ZNF674','ZSCAN30'],
        ['ZNF702P','ZNF816'],
        ['ZNF702P','ZNF816-ZNF321P'],
        ['ZNF708','ZNF738'],
        ['ZNF708','ZNF85'],
        ['ZNF718','ZNF876P'],
        ['ZNF737','ZNF826P'],
        ['ZNF746','ZNF767P'],
        ['ZNF761','ZNF765'],
        ['ZNF765','ZNF845'],
        ['ZNF780A','ZNF780B'],
        ['ZNF815P','ZNF890P'],
        ['ZNF826P','ZNF90'],
        ['ZSCAN5A','ZSCAN5B'],
        ['AC089984.2','CCDC9B'],
        ['AC091390.3','PMS2'],
        ['AC093307.1','KIAA1217'],
        ['AC114402.1','CHMP5'],
        ['AC140479.3','MALL'],
        ['ACTB','PECAM1'],
        ['ACTB','SDHA'],
        ['ACTB','STAT6'],
        ['ACTB','ZNNT1'],
        ['ADGRG1','MYH11'],
        ['ANAPC1','RMND5A'],
        ['AP3B1','TG'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ATP1A1-AS1','HSP90B1'],
        ['ATRX','CIRBP'],
        ['AZGP1','GJC3'],
        ['B2M','ERBB3'],
        ['BLNK','NPM1'],
        ['BRS3','HTATSF1'],
        ['C19MC','RPLP1'],
        ['C7','EEF1A1'],
        ['CACNG2','PDE4D'],
        ['CATSPER2','PPIP5K1'],
        ['CENPW','TRMT11'],
        ['COPDA1','IGHG2'],
        ['COQ8B','NUMBL'],
        ['CSNK1G1','TG'],
        ['CTBS','GNG5'],
        ['DDX23','TG'],
        ['DDX24','EEF1A1'],
        ['DEPP1','FASN'],
        ['DNAJA1','TG'],
        ['DYSF','TG'],
        ['EEF1A1','FASN'],
        ['ELOVL1','MED8'],
        ['EML1','TFAM'],
        ['EML2','TG'],
        ['ETV6','SPRY4'],
        ['FAM157B','FBXO25'],
        ['FGFR2','MKNK2'],
        ['GCLC','TG'],
        ['GKAP1','KIF27'],
        ['GLG1','HNRNPA2B1'],
        ['GRB2','NME6'],
        ['HBM','SPOP'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['IGHM','UNC13D'],
        ['LARS1','TAT'],
        ['LHFPL5','MYH11'],
        ['LINC02379','RNF212B'],
        ['MAPK6','PECAM1'],
        ['MECOM','RPL22'],
        ['MGRN1','PPP1R1A'],
        ['MICOS10','TG'],
        ['MIF','SLC2A11'],
        ['MRPS31P5','THSD1'],
        ['MYH11','SERF2'],
        ['MYH11','TTN'],
        ['NAIP','OCLN'],
        ['NEK9','RAB7A'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['OBSCN','TG'],
        ['PDXDC1','SCD'],
        ['PKD1','TG'],
        ['PMS2','POLR2J3'],
        ['PPP2R5A','ZNF680'],
        ['RANBP3','TG'],
        ['RFC2','TG'],
        ['SCP2','SERPINA1'],
        ['SENP3','TG'],
        ['SERF1A','SMN1'],
        ['SIDT2','TAGLN'],
        ['STAT6','TMSB4X'],
        ['TAPBP','TG'],
        ['TMEM8B','TXN2'],
        ['ABL1','EXOSC2'],
        ['ABR','NXN'],
        ['AC004066.1','PPA2'],
        ['AC004951.1','DTX2P1-UPK3BP1-PMS2P11'],
        ['AC004980.4','AC091390.3'],
        ['AC005674.1','WDR1'],
        ['AC006064.6','LYZ'],
        ['AC008750.7','NKG7'],
        ['AC008993.1','MIR1302-9HG'],
        ['AC009065.3','PDXDC2P-NPIPB14P'],
        ['AC019205.1','EIF3E'],
        ['AC020656.1','POLH'],
        ['AC020656.1','RPL37'],
        ['AC020656.1','XACT'],
        ['AC020907.3','SYT11'],
        ['AC021739.2','LINC02883'],
        ['AC022210.1','KCMF1'],
        ['AC083899.1','AC133644.2'],
        ['AC091057.1','ARHGAP11B'],
        ['AC091390.3','AC211429.1'],
        ['AC091390.3','PMS2'],
        ['AC091390.3','POM121C'],
        ['AC093010.1','TMPRSS11B'],
        ['AC093162.1','RETSAT'],
        ['AC093525.2','ZEB2'],
        ['AC093787.2','AC233263.7'],
        ['AC133644.2','ANAPC1'],
        ['AC138123.2','CAP1'],
        ['AC138409.2','AC138866.1'],
        ['AC138409.2','NAIP'],
        ['AC138866.1','NAIP'],
        ['AC138969.1','PKD1'],
        ['AC211476.4','PMS2'],
        ['AC211476.5','PMS2'],
        ['AC239859.5','LINC02802'],
        ['AC244197.3','ZMAT4'],
        ['AC245748.1','ZNF806'],
        ['ACAA1','CTSS'],
        ['ACAD8','GLB1L3'],
        ['ACBD5','SMCHD1'],
        ['ACCS','EXT2'],
        ['ACSS1','APMAP'],
        ['ACTB','HMGB1'],
        ['ACTN4','EIF3K'],
        ['ADAP1','SUN1'],
        ['ADGRE2','ADGRE5'],
        ['ADGRG7','TFG'],
        ['ADSL','PGK1'],
        ['AHCTF1','KMT2E'],
        ['AKAP17A','FOS'],
        ['AKT2','C19ORF47'],
        ['AL022238.3','TNRC6B'],
        ['AL109809.4','AL117335.1'],
        ['AL121578.1','SYTL5'],
        ['AL133351.4','NQO2'],
        ['AL138963.3','HBA2'],
        ['AL158066.1','THSD1'],
        ['AL512662.1','BMS1'],
        ['ALG14','CNN3'],
        ['ANAPC1','ANAPC1P2'],
        ['ANAPC1','RMND5A'],
        ['ANXA1','DDX5'],
        ['ANXA1','RBM39'],
        ['AP000943.3','FUT4'],
        ['ARF1','MCM3AP'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ASXL1','NOL4L'],
        ['ATG16L1','INPP5D'],
        ['ATP5ME','SLC49A3'],
        ['ATP5MF','ZNF394'],
        ['ATP8B4','ETS2'],
        ['ATXN3','THAP11'],
        ['AZGP1','GJC3'],
        ['AZU1','ZNF540'],
        ['B2M','DDX5'],
        ['B3GNTL1','METRNL'],
        ['BAG2','ZNF451'],
        ['BAG6','C6ORF47'],
        ['BANK1','IGH@'],
        ['BASP1','Z84488.1'],
        ['BCL11B','PIGR'],
        ['BCL11B','RPL3'],
        ['BCLAF1','LGALSL'],
        ['BHLHE40','LTF'],
        ['BLNK','NPM1'],
        ['BLOC1S6','SQOR'],
        ['BPTF','LRRC37A2'],
        ['BRD2','RACK1'],
        ['BSDC1','FAM229A'],
        ['C12ORF73','HSP90B1'],
        ['C19MC','RPLP1'],
        ['CABP7','NF2'],
        ['CALM1','HIF1A'],
        ['CALML4','CLN6'],
        ['CATSPER2','PPIP5K1'],
        ['CBWD2','DOCK8'],
        ['CBX3','CCDC32'],
        ['CCM2L','HCK'],
        ['CD24','LINC02321'],
        ['CD81','TSSC4'],
        ['CD83','JARID2'],
        ['CDC42EP2','POLA2'],
        ['CDK2','RAB5B'],
        ['CEP104','LINC01031'],
        ['CFAP300','YAP1'],
        ['CFAP54','NEDD1'],
        ['CHCHD10','VPREB3'],
        ['CHMP3','EEF1A1'],
        ['CIRBP','FAM174C'],
        ['CLSTN1','CTNNBIP1'],
        ['COLQ','HACL1'],
        ['COPB2','RNF130'],
        ['COQ8B','NUMBL'],
        ['CPM','LYZ'],
        ['CRCP','TPST1'],
        ['CSDC2','LINC02348'],
        ['CSF2RA','IL3RA'],
        ['CSNK1G2','OAZ1'],
        ['CTBS','GNG5'],
        ['CTDP1','SLC66A2'],
        ['CTSC','RAB38'],
        ['CWF19L1','DA750114'],
        ['CXCR4','UBXN4'],
        ['CXCR4','ZEB2'],
        ['CYTIP','ERMN'],
        ['DCUN1D1','MCCC1'],
        ['DDX17','EEF1A1'],
        ['DDX42','MAP3K3'],
        ['DHX35','FAM83D'],
        ['DHX9','NPL'],
        ['DOCK9','STK24'],
        ['DSG3','SYNRG'],
        ['EDEM3','MNDA'],
        ['EEF1A1','ELANE'],
        ['EEF1A1','FUS'],
        ['EEF1A1','HNRNPC'],
        ['EEF1A1','RPS3A'],
        ['EEF1A1','SCD5'],
        ['EEF1A1','SFPQ'],
        ['EEF1A1','SH3D19'],
        ['EEF1G','HBB'],
        ['EIF4E3','FOXP1'],
        ['ELOVL1','MED8'],
        ['ENTPD5','FAM161B'],
        ['EXOC8','PIGH'],
        ['FAM13A-AS1','HERC3'],
        ['FAM157B','FBXO25'],
        ['FAT4','LYZ'],
        ['FCAR','MARK3'],
        ['FES','MAN2A2'],
        ['FOSB','HMGB1'],
        ['FOXO1','LINC00598'],
        ['FOXP1','RYBP'],
        ['GAS7','RCVRN'],
        ['GGT1','IGSF3'],
        ['GKAP1','KIF27'],
        ['GOLT1A','KISS1'],
        ['GPR160','HBB'],
        ['GSR','GTF2E2'],
        ['HAPLN3','MFGE8'],
        ['HBB','PLEKHB2'],
        ['HBB','RPS4X'],
        ['HIBADH','TAX1BP1-AS1'],
        ['HORMAD1','SLC22A20P'],
        ['HPX','TRIM3'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['IDS','LINC00893'],
        ['IFI44L','PDGFA'],
        ['IGBP1','KLF8'],
        ['IGH@','LAPTM5'],
        ['IGH@','NFKB2'],
        ['IGH@','PCSK7'],
        ['IGH@','TUBA1B'],
        ['IGH@','ZFP36L1'],
        ['ILRUN','SPDEF'],
        ['INSL3','JAK3'],
        ['IQGAP1','ZNF774'],
        ['ITM2B','RB1'],
        ['KCNK3','NEB'],
        ['KIAA2026','UHRF2'],
        ['KLF2','OAZ1'],
        ['KLK4','KLKP1'],
        ['LINC01513','ROPN1L'],
        ['LPGAT1','RPS6KC1'],
        ['LRRC37A3','NSF'],
        ['LYZ','NCAPH2'],
        ['LYZ','XPO1'],
        ['MACF1','TAGLN'],
        ['MALT1','STAR'],
        ['MANBAL','SRC'],
        ['MAPKBP1','MGA'],
        ['METTL23','MFSD11'],
        ['MICAL2','TEAD1'],
        ['MKNK2','PTBP1'],
        ['MRPL51','NCAPD2'],
        ['MRPS31P5','THSD1'],
        ['MT1E','MT1M'],
        ['MTG1','SCART1'],
        ['MTSS1','TATDN1'],
        ['NAIP','OCLN'],
        ['NCOA6','PIGU'],
        ['NCOR2','UBC'],
        ['NDRG1','ST3GAL1'],
        ['NFATC3','PLA2G15'],
        ['NONO','S100A9'],
        ['NPEPL1','STX16'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['NPM1','SET'],
        ['NPNT','PPA2'],
        ['OAZ1','PTMA'],
        ['PADI4','S100A9'],
        ['PBXIP1','PMVK'],
        ['PIGR','SI'],
        ['PIM3','SCO2'],
        ['PLEKHO1','VPS45'],
        ['PMS2','POLR2J3'],
        ['PRDX1','TESK2'],
        ['PRKAA1','TTC33'],
        ['PRKAG2','RHEB'],
        ['PSMB2','TFAP2E-AS1'],
        ['PXK','RPP14'],
        ['RAB9B','TMSB15B-AS1'],
        ['RACK1','TRIM52'],
        ['RBFOX1','SDC3'],
        ['RBM39','ZNF791'],
        ['RBMS2','SPRYD4'],
        ['RHOA','TKT'],
        ['RIPK1','SERPINB9'],
        ['RPL14','SRP14'],
        ['RPL38','TTYH2'],
        ['S100A9','SPEN'],
        ['S100A9','SRSF3'],
        ['SAMD5','SASH1'],
        ['SCNN1A','TNFRSF1A'],
        ['SERF1A','SMN1'],
        ['SERF2','WSB1'],
        ['SIDT2','TAGLN'],
        ['SMAP2','TNK2'],
        ['SNTB2','VPS4A'],
        ['SSBP1','WEE2'],
        ['STARD3','TCAP'],
        ['STK10','UBE2D2'],
        ['SYT8','TNNI2'],
        ['TFDP2','XRN1'],
        ['TMOD4','VPS72'],
        ['TMSB15B','TMSB15B'],
        ['TRPM2','ZNF511'],
        ['UBA2','WTIP'],
        ['UBC','YWHAZ'],
        ['USP34','XPO1'],
        ['ZNF285','ZNF806'],
        ['ABR','NXN'],
        ['AC004690.2','SPAM1'],
        ['AC004946.1','UBC'],
        ['AC004951.1','DTX2P1-UPK3BP1-PMS2P11'],
        ['AC004980.4','AC091390.3'],
        ['AC005013.1','AC005162.2'],
        ['AC005674.1','WDR1'],
        ['AC005696.3','PAFAH1B1'],
        ['AC006042.3','GLCCI1-DT'],
        ['AC006064.6','LYZ'],
        ['AC006116.6','AC006116.8'],
        ['AC008686.1','OLFM1'],
        ['AC008750.7','NKG7'],
        ['AC008993.1','MIR1302-9HG'],
        ['AC009065.3','PDXDC2P-NPIPB14P'],
        ['AC012485.1','ANXA1'],
        ['AC015911.9','SLFN13'],
        ['AC015977.1','CIB4'],
        ['AC016588.2','AL078621.1'],
        ['AC016907.2','AC106870.3'],
        ['AC016912.2','RHOQ'],
        ['AC017074.1','ACR'],
        ['AC017104.4','LINC00471'],
        ['AC017104.5','NMUR1'],
        ['AC018630.4','AMY2B'],
        ['AC018630.4','BHLHA15'],
        ['AC018630.4','CA6'],
        ['AC018630.4','CCL28'],
        ['AC018630.4','CD44'],
        ['AC018630.4','CD74'],
        ['AC018630.4','CST1'],
        ['AC018630.4','CST2'],
        ['AC018630.4','CST4'],
        ['AC018630.4','DDX17'],
        ['AC018630.4','EEF1A1'],
        ['AC018630.4','EEF2'],
        ['AC018630.4','FKBP8'],
        ['AC018630.4','GAPDH'],
        ['AC018630.4','GNAS'],
        ['AC018630.4','GNE'],
        ['AC018630.4','H1-0'],
        ['AC018630.4','IGH@'],
        ['AC018630.4','IGK@'],
        ['AC018630.4','IL6ST'],
        ['AC018630.4','ITPR1'],
        ['AC018630.4','MUC7'],
        ['AC018630.4','MYH9'],
        ['AC018630.4','PIGR'],
        ['AC018630.4','PIK3R1'],
        ['AC018630.4','PRR27'],
        ['AC018630.4','RBM47'],
        ['AC018630.4','RPL3'],
        ['AC018630.4','SFRP1'],
        ['AC018630.4','SLC20A2'],
        ['AC018630.4','SLC6A8'],
        ['AC018630.4','SMR3B'],
        ['AC018630.4','SORL1'],
        ['AC018630.4','SQSTM1'],
        ['AC018630.4','STATH'],
        ['AC018630.4','ZG16B'],
        ['AC018809.3','PRRT3-AS1'],
        ['AC019205.1','EIF3E'],
        ['AC020656.1','CEACAM5'],
        ['AC020656.1','KHDC4'],
        ['AC020656.1','MYLK'],
        ['AC020656.1','POLH'],
        ['AC020656.1','RPL37'],
        ['AC020656.1','XACT'],
        ['AC020907.3','SYT11'],
        ['AC021517.3','SLC14A2-AS1'],
        ['AC021739.2','LINC02883'],
        ['AC022210.1','KCMF1'],
        ['AC022733.1','LINC02866'],
        ['AC023794.1','LINC02381'],
        ['AC024560.1','AC026412.4'],
        ['AC024940.1','AC092666.1'],
        ['AC024940.1','SSUH2'],
        ['AC053503.3','MYH6'],
        ['AC053503.3','MYL2'],
        ['AC053503.3','NPPA'],
        ['AC053503.3','TPM1'],
        ['AC055876.5','LINC02256'],
        ['AC063979.1','LINC02056'],
        ['AC068643.1','C12ORF42'],
        ['AC073585.1','DOCK1'],
        ['AC073585.1','GP2'],
        ['AC073585.1','PNLIP'],
        ['AC083899.1','AC133644.2'],
        ['AC087518.1','LINC02866'],
        ['AC091057.1','ARHGAP11B'],
        ['AC091167.1','PERP'],
        ['AC091390.3','AC211429.1'],
        ['AC091390.3','AC211486.3'],
        ['AC091390.3','PMS2'],
        ['AC091390.3','POM121'],
        ['AC091390.3','POM121C'],
        ['AC091390.3','RBAK'],
        ['AC091982.1','G3BP1'],
        ['AC093010.1','TMPRSS11B'],
        ['AC093162.1','RETSAT'],
        ['AC093512.2','ALDOB'],
        ['AC093512.2','ALDOC'],
        ['AC093525.2','ZEB2'],
        ['AC093787.2','AC233263.7'],
        ['AC098828.1','LAPTM4A'],
        ['AC109583.1','PRSS42P'],
        ['AC109583.4','PRSS46P'],
        ['AC113189.12','SPEM2'],
        ['AC115220.2','LINC01005'],
        ['AC115220.2','MAEL'],
        ['AC118758.2','LINC00174'],
        ['AC122688.3','H2AZ2'],
        ['AC132217.2','COL4A2'],
        ['AC132217.2','H19'],
        ['AC133644.2','ANAPC1'],
        ['AC133644.2','PLG'],
        ['AC134980.1','OR4N2'],
        ['AC138123.2','CAP1'],
        ['AC138409.2','AC138866.1'],
        ['AC138409.2','NAIP'],
        ['AC138866.1','AL021368.4'],
        ['AC138866.1','AL513548.1'],
        ['AC138866.1','LINC00680'],
        ['AC138866.1','NAIP'],
        ['AC138866.1','WDR70'],
        ['AC138969.1','PKD1'],
        ['AC140479.4','MALL'],
        ['AC211476.4','CCDC146'],
        ['AC211476.4','PMS2'],
        ['AC211476.4','UPK3B'],
        ['AC211476.5','PMS2'],
        ['AC239859.5','LINC02802'],
        ['AC244197.3','ZMAT4'],
        ['AC245452.4','IGLVIV-66-1'],
        ['AC245748.1','ZNF806'],
        ['ACAA1','CTSS'],
        ['ACAP2','KPNA4'],
        ['ACBD5','LRRC37A6P'],
        ['ACBD5','SMCHD1'],
        ['ACTB','HMGB1'],
        ['ACTR1A','GBF1'],
        ['ADAM3A','NUP153'],
        ['ADSL','PGK1'],
        ['AGAP11','AL512662.1'],
        ['AHCTF1','KMT2E'],
        ['AKAP17A','FOS'],
        ['AL022238.3','TNRC6B'],
        ['AL033527.2','AL033527.3'],
        ['AL049828.1','MIA2'],
        ['AL049869.2','ZBTB25'],
        ['AL050303.2','LINC01087'],
        ['AL050303.2','LINC01297'],
        ['AL109809.4','AL117335.1'],
        ['AL121578.1','SYTL5'],
        ['AL133351.4','NQO2'],
        ['AL136309.3','C6ORF201'],
        ['AL138963.3','HBA2'],
        ['AL157834.4','CYP2C8'],
        ['AL158066.1','THSD1'],
        ['AL355075.4','CCNB1IP1'],
        ['AL355312.2','AL355312.5'],
        ['AL358777.3','PAK1IP1'],
        ['AL365295.1','SENP6'],
        ['AL391628.1','TRB@'],
        ['AL445669.1','PPIH'],
        ['AL512590.2','CCDC180'],
        ['AL512662.1','BMS1'],
        ['AL513217.1','SHISA4'],
        ['AL590004.3','ETS2'],
        ['AL590556.3','ATP1A1-AS1'],
        ['AL590556.3','INS'],
        ['AL590666.1','CRABP2'],
        ['AL591848.1','AL591848.2'],
        ['ANAPC1','ANAPC1P2'],
        ['ANXA1','DDX5'],
        ['ANXA1','RBM39'],
        ['AP000356.5','PIGS'],
        ['AP000812.4','FOLR3'],
        ['AP000943.3','FUT4'],
        ['APLF','RTRAF'],
        ['APOA1','CALR'],
        ['APOA1','CYP2E1'],
        ['APOA1','DPYS'],
        ['APOA1','HP'],
        ['APOA1','REG3A'],
        ['APPL2','KCCAT198'],
        ['ARF1','MCM3AP'],
        ['ART3','SDAD1-AS1'],
        ['ASB15-AS1','NDUFA5'],
        ['ASS1','RIPOR2'],
        ['ATP1B1','USP7'],
        ['ATP5MF','ZNF394'],
        ['ATP6V1E1','TG'],
        ['ATP8B4','ETS2'],
        ['AZU1','ZNF540'],
        ['B2M','DDX5'],
        ['B4GALT1','SMU1'],
        ['BAIAP2-DT','TCF3'],
        ['BASP1','Z84488.1'],
        ['BCL11B','PIGR'],
        ['BCL11B','PRH1'],
        ['BCL11B','PTP4A2'],
        ['BCL11B','RPL3'],
        ['BCL11B','ZNF880'],
        ['BCLAF1','LGALSL'],
        ['BHLHA15','PRH2'],
        ['BHLHE40','LTF'],
        ['BLNK','NPM1'],
        ['BRD1','G3BP1'],
        ['BRD2','RACK1'],
        ['C19MC','SERINC3'],
        ['C1ORF116','YOD1'],
        ['CALM1','HIF1A'],
        ['CAPZA1','MOV10'],
        ['CAPZA2','H2AZ2'],
        ['CCDC81','HIKESHI'],
        ['CCDC88C','MUC7'],
        ['CCND3','SDC3'],
        ['CD24','EEF1A1'],
        ['CD24','LINC02321'],
        ['CDK10','LYZ'],
        ['CDRT15L2','TBC1D28'],
        ['CEACAM5','LINC01553'],
        ['CELA2A','TRIP12'],
        ['CEP104','LINC01031'],
        ['CGB7','NTF4'],
        ['CHCHD1','FUT11'],
        ['CHEK2','NAIP'],
        ['CHMP1A','TG'],
        ['CHMP3','EEF1A1'],
        ['CHMP4B','TRB@'],
        ['CLCC1','GBP6'],
        ['CLCC1','WFDC3'],
        ['CLCN6','EXOSC7'],
        ['CLCN6','PBX1'],
        ['CLEC2D','NPM1'],
        ['COL1A2','IGF2'],
        ['COPB1','RRAS2'],
        ['COPB2','RNF130'],
        ['COPZ2','NFE2L1-DT'],
        ['CPA1','FTL'],
        ['CPSF6','NINJ2-AS1'],
        ['CSDC2','LINC02348'],
        ['CSHL1','H19'],
        ['CST4','PPP1R1B'],
        ['CTRL','PRSS3'],
        ['CTSB','LIPG'],
        ['CWF19L1','DA750114'],
        ['DCAF1','ITGB1'],
        ['DCAF11','PCK2'],
        ['DDX17','EEF1A1'],
        ['DDX17','USP15'],
        ['DPY19L2','SORD'],
        ['DSG3','SYNRG'],
        ['DTX2P1-UPK3BP1-PMS2P11','EEF1A1'],
        ['DUS4L-BCAP29','SLC26A4'],
        ['DYNLL2','PABPC1'],
        ['EDEM3','MNDA'],
        ['EEF1A1','ELANE'],
        ['EEF1A1','FUS'],
        ['EEF1A1','HNRNPC'],
        ['EEF1A1','OBSL1'],
        ['EEF1A1','PECAM1'],
        ['EEF1A1','PRB3'],
        ['EEF1A1','RPS3A'],
        ['EEF1A1','SCD5'],
        ['EEF1A1','SFPQ'],
        ['EEF1A1','SH3D19'],
        ['EEF1A1','TNP1'],
        ['EEF1A1','TRIM22'],
        ['EEF1G','HBB'],
        ['EIF3E','HTN3'],
        ['ETS2','NOVA1'],
        ['EXOC6','SHROOM3'],
        ['EXOC8','PIGH'],
        ['FAM133B','FAM133EP'],
        ['FAM133B','FAM133FP'],
        ['FASN','PRB2'],
        ['FAT4','LYZ'],
        ['FCAR','MARK3'],
        ['FOSB','HMGB1'],
        ['FP475955.3','PSLNR'],
        ['FTX','STATH'],
        ['GBF1','TG'],
        ['GMFG','LRFN1'],
        ['GP2','PPM1G'],
        ['GPR160','HBB'],
        ['H2BW3P','TMSB15B'],
        ['HBB','PLEKHB2'],
        ['HBB','RPS4X'],
        ['HGSNAT','NDUFS1'],
        ['HIBADH','TAX1BP1-AS1'],
        ['HNRNPH1','STATH'],
        ['HP','HPX'],
        ['HSPD1','TG'],
        ['HSPG2','KRT4'],
        ['HTN3','TMED10'],
        ['IFI44L','PDGFA'],
        ['IFT172','TG'],
        ['IFT80','MOB1A'],
        ['IGBP1','KLF8'],
        ['IGF2BP1','LYZ'],
        ['IGH@','PCBP2'],
        ['IGH@','PGA4'],
        ['IGH@','TUBA1B'],
        ['IGHA1','PGA4'],
        ['IGK@','PIPOX'],
        ['IL9R','RPL23AP53'],
        ['ITM2B','SI'],
        ['KANSL1L-AS1','RPE'],
        ['KCNK3','NEB'],
        ['KDM6B','SMR3B'],
        ['KIR3DX1','UGDH'],
        ['LAMP2','SMAD1'],
        ['LCOR','TG'],
        ['LINC01297','PSLNR'],
        ['LINC01923','NDUFV1'],
        ['LINC02210_','MORC2-AS1'],
        ['LMX1A-AS1','LMX1A-AS2'],
        ['LYZ','NCAPH2'],
        ['LYZ','XPO1'],
        ['MACF1','TAGLN'],
        ['MALT1','STAR'],
        ['MAP3K20','XPO1'],
        ['MARK3','TG'],
        ['MB','MFN2'],
        ['MB','NEK9'],
        ['MB','RPS4X'],
        ['METTL6','NUP58'],
        ['MFF','PDE6A'],
        ['MFN2','TG'],
        ['MICALL1','RBM39'],
        ['MIR100HG','RAB29'],
        ['MKNK2','PTBP1'],
        ['MRPS31P5','THSD1'],
        ['MT1E','MT1M'],
        ['MTHFD2','STAG3'],
        ['MTR','TG'],
        ['MUC7','NFIB'],
        ['NDRG2','PRR27'],
        ['NEK4','ZBTB5'],
        ['NEK9','SMR3B'],
        ['NONO','S100A9'],
        ['NPM1','SET'],
        ['NPNT','PPA2'],
        ['NT5C2','TG'],
        ['NTRK2','PNLIP'],
        ['NUTM2A','NUTM2B'],
        ['PADI4','S100A9'],
        ['PDCD6IP','STATH'],
        ['PGBD2','RPL23AP25'],
        ['PGK1','POLA1'],
        ['PIGR','SI'],
        ['PIP5K1B','ZFAND5'],
        ['PMS2','POM121C'],
        ['PMS2','UPK3B'],
        ['PNLIP','SERPING1'],
        ['PRSS43P','PRSS44P'],
        ['PSMB2','TFAP2E-AS1'],
        ['PTGES2','TGFBR1'],
        ['PTPN14','TSR1'],
        ['RAB29','ZNF665'],
        ['RAB9B','TMSB15B-AS1'],
        ['RBAK','RNF216P1'],
        ['RBFOX1','SDC3'],
        ['RBFOX1','UBE2D3'],
        ['RBM39','ZNF791'],
        ['RMI1','U62317.3'],
        ['RNF170','THAP1'],
        ['RNF38','TG'],
        ['RPS19','TIGD1'],
        ['S100A9','SPEN'],
        ['S100A9','SRSF3'],
        ['SEMA6A','SMR3B'],
        ['SERF2','WSB1'],
        ['SIPA1L3','SMR3B'],
        ['SMAP2','TNK2'],
        ['SMARCE1','TG'],
        ['SMKR1','STRIP2'],
        ['SMR3B','SVIL'],
        ['SPTBN2','WWOX'],
        ['SSB','SSBL4P'],
        ['SSX1','SSX8P'],
        ['STARD3','TCAP'],
        ['STAU1','TG'],
        ['TAF1L','TMEM154'],
        ['TG','TRIM2'],
        ['TG','UGP2'],
        ['TG','VAMP7'],
        ['TGM4','XPO1'],
        ['TMSB15B','TMSB15B'],
        ['TRPM2','ZNF511'],
        ['UBC','YWHAZ'],
        ['ZNF285','ZNF806'],
        ['TMEM256','PLSCR3']

        ]
    # up is human



    data = fusions.get(options.organism.lower(),[])
    if data:

        file_symbols = os.path.join(options.output_directory,'synonyms.txt')
        loci = symbols.generate_loci(file_symbols)

        genes = symbols.read_genes_symbols(file_symbols)
        d = []
        for gg in data:
            if len(gg)>2 or len(gg)<2:
                print "ERROR:",gg
                sys.exit(1)
            g1 = gg[0]
            g2 = gg[1]
            if g1.upper() != g2.upper():
                #print g1,g2
                ens1 = symbols.ensembl(g1.upper(),genes,loci)
                ens2 = symbols.ensembl(g2.upper(),genes,loci)
                if ens1 and ens2:
                    for e1 in ens1:
                        for e2 in ens2:
                            if e1 != e2:
                                d.append([e1,e2])
                                #print '-->',e1,e2
        # create a banned list of gene based on loci
        for v in loci.values():
            if v:
                n = len(v)
                if n > 1:
                    for i in xrange(n-1):
                        for j in xrange(i+1,n):
                            if v[i].upper() != v[j].upper():
                                ens1 = symbols.ensembl(v[i].upper(),genes,loci)
                                ens2 = symbols.ensembl(v[j].upper(),genes,loci)
                                if ens1 and ens2:
                                    for e1 in ens1:
                                        for e2 in ens2:
                                            if e1 != e2:
                                                d.append([e1,e2])
        data = ['\t'.join(sorted(line)) + '\n' for line in d]
        data = sorted(set(data))

    print "List of banned fusions has ",len(data),'fusions.'
    file(os.path.join(options.output_directory,'banned.txt'),'w').writelines(data)
    #
