#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of banned fusion genes sequences. This list is hard coded inhere.



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of banned fusion genes sequences. This list is hard coded inhere."""
    version = "%prog 0.10 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of banned candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of banned candidate fusion genes is generated. Default is '%default'.""")


    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #
    stub = """>abc--def
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
"""

    print "Generating the list of banned fusion genes..."
    fusions = dict()

    fusions['rattus_norvegicus'] = """>abc--def
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
"""

    fusions['mus_musculus'] = """>abc--def
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
"""

    fusions['canis_lupus_familiaris'] = """>abc--def
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
"""


    fusions['homo_sapiens'] = """>AC005670.3--ARL17A
AGCACAAGGCTCCCGCTCTCCTCGTCCTCGG*GAGAAAATACATCTTTTGAAAACACGGCCAC
>AC008119.1--ISCU
GCTGGGGGTCCCACCTTCTGTCCTCTCCATG*GTTGTTGATCATTATGAAAATCCTAGAAACG
>AC022034.1--AC022034.3
GCTTTAATTTGGGCCTTAGAACTCAGTTAAG*GCACAGCTGCTTAAAGACTGTTACCGGAAAG
>AC104411.1--AC092944.1
AAAAATACCATCCAAAGCTACATGACTTGAG*AGACAAGGTCTCACTATGTTGCCCAGGCTGG
>AC110611.2--AC110611.1
GACTTCTTAATTTCCCCCTGGATAGAGCAGG*ATTCGATTCAGTTGGAAAGACATGGTTAATG
>AC126182.3--NDUFB4
ATGGGATTTCACATCTCAAGACATCTCCCAG*GAAAATCCTGCCTTGCTTCGTTGGGCCTATG
>AC145124.1--LINC02018
TGAAGGAGACAAGAAAAACCAACGGGAACAG*CTGGGGAAACTGAGTTACAGAGCTTTTCTGC
>AL096803.2--KCTD10
TCATTGCAAGCTCCGCCTCCTGGGTTCACGC*ATTTTATACTGTTTAAGTGTTAAACACCAAC
>AL117335.1--AL109809.5
CCCCAAGAGTTGCTGCTCACCAGGCTGGAAG*AAAATTCTCAAACTACAGGAAGTGGAGAGAA
>AL133405.1--STMND1
AATAATATTGATGATCCTGACCTGTGTAAAG*GCTGATGTCAGTGTGCCTCATACTGGGGAAA
>AL139393.1--ADIPOQ-AS1
GTGAGACTCCATCTCAAAAAAAAAAAAAAGT*TGGATATCACATCAGAGAAGCTCCTGGCCAG
>AL391840.3--SH3BGRL2
TCATTCCATCTTCATTCATGACTCCAGACAG*ATAAAGAAGAAGCAGCAAGATGTGGTTAGAT
>AL445524.2--SCAND2P
CCTAGGCTGGGGTTTAAGTTAAATTTTTTAA*AAAAACTAAAGTGACTGGCACTAAGTGAACT
>AL445524.2--SCAND2P
CTGGATATGAAATAAGTGCCCTGTGTAGAAT*TTTTTTTCATTCTTATATTTTGCCAGATCTG
>CTSB--LINC02245
CTTTGGGAGGCCAAGGTGGGAGGATGACAAG*GTCGGGAGTTTGGGACCAGCCTGACCAACAT
>FAM27C--AL953883.1
TCAGTCCATCCTTGGCTGTTCAAGTGTAGAG*AAATAAAAACATCAGTAGGAGGGGAAGGAGG
>IGLC1--AC245060.7
CCGTCAAGGCGGGAGTGGAGACCACCAAACC*AGGCCGATTATTACTGCGGAACATGGGATAG
>LAMC2--AL354953.1
AACACATTAGACGGCCTCCTGCATCTGATGG*ACCCTGCACTTGATGGACCAGCTGGCACCAC
>LAMC2--AL354953.1
AACACATTAGACGGCCTCCTGCATCTGATGG*GAGCCCATGGAGGAGAGGCACCAACAGGTAT
>LBH--LINC01936
CAGCCCCCGCAAGGATGGCCTTTCCTACCAG*ATGAGAAGAGAATGAGCCATGTGTTTCCTGA
>LINC00402--AL353660.1
AAGAGCCTCTTTCCTTCATACCATCTTCAGA*GTATAACTCCTACTTGAAGAGTGTGAGAGAA
>LINCADL--LVRN
GCTGTGATGAATGCTGCAGAGGCCAAATAAG*GGCCCTGTTAGCGTCCCAGCTGGAACCAACA
>MYH7--DLG1
GCGGGCCAAGAGCCGTGACATTGGCACGAAG*AGATGGGGTTTCACCATGTTCCACAGGCTGG
>RRM1--AC015689.1
AAATAAGGAGAAGCTAAAAGATAAAGAAAAG*GATGACACAGCAAGAAGGCCCTCACCAGATG
>RRM1--AC015689.1
AGTAGCCAAACTACTTCTTGAGCATAGATAG*GATGACACAGCAAGAAGGCCCTCACCAGATG
>AC021087.5--MUC20-OT1
CCTGTCCTATGTGGACGTTGGCACTGGGAAG*TGCACAGCCAGAGGACAACACATCACACACA
>AC132008.2--AC021087.5
TCGGCGCCTGGCGCTGGCGCTGGCCAAGGCT*ATTTCTGCTCAGTATCCAGTAGTGGATCATG
>AC138649.1--AC091951.3
CAGCAACAAATCTGATTTAAGCTTCACACAT*AATCAAGGAAAACTGATGCTTCAGAAAGATG
>AC245060.7--IGLC2
CTCCTTCTCGTGCTCCTCTCTCACTGCACAG*GTCAGCCCAAGGCTGCCCCCTCGGTCACTCT
>AC245060.7--IGLL5
ACTGCGGAACATGGGATAGCAGCCTGAGTGC*GTCTTCGGAACTGGGACCAAGGTCACCGTCC
>AC245060.7--IGLL5
ACTGCGGAACATGGGATAGCAGCCTGAGTGC*TTATGTCTTCGGAACTGGGACCAAGGTCACC
>AC245060.7--IGLL5
TACTGCGGAACATGGGATAGCAGCCTGAGTG*TTATGTCTTCGGAACTGGGACCAAGGTCACC
>ADD1--MYH11
GCCTCAGCCTCCTGAGTAGCTGAGATTACAG*AGGGCCCCCCCCACAGGAAACTTCGCAGTGA
>CYP2B6--BHMT2
ATACAAAATCATATATATATATATGTTCTTG*TTTTTTTGAGACAGAGTCTCACTCTGTTGCC
>EMC10--AP002956.1
ACGGTGAAACCCTGTCTCTACTAAAAATACA*AAAAAAAAATTAGCTAGGCGTAGTGGTTGGC
>IGLL5--AC245060.7
CCGTCAAGGCGGGAGTGGAGACCACCAAACC*AGGCCGATTATTACTGCGGAACATGGGATAG
>LARP1--CTSB
GGCTGGTCTCAAACTCCTGACCTCAGGTGAT*CTGCCTGTATTTTTTATCAGCCACAAATCCA
>MYH11--ADD1
GCGAGGTGAACGCACTCAAGAGCAAGCTCAG*ACGGAGTCTTACTCTGTTGTTCAGGCTGGAG
>PARG--AC022400.6
CTGGATCACAATGAATGTCTAATTATCACAG*TGTCAGAGATGGCTTTGGATTCCCCGTTCTG
>PKD1--AC139256.2
AGAGTGACTTGTTTCTGGATGATTCTAAGAG*GTTATCAATACTCTGGCTGACCATCGTCATC
>TG--HECTD1
TACCTTCAATAAAGTATCTACATGCGGTGAA*GAAAAAAAGAAAAAAAAAAAGAATCTTTGAT
>YWHAZ--AP000790.2
CAAGCATACCAAGAAGCTTTTGAAATCAGCA*AAAAAGGAAATGCAACCAACAGATCCTATCA
>AC002057.2--AC002057.1
GGATAATTTAGGATAACCTCCTTATTTTAAG*ACACAGTAATTTTAGCAGAATGCTAATAGGA
>AC003044.1--AC004485.1
AAACGTCCTTTTGCTGAATATTCTACCTACG*GAAGAAGCAAAGAAGTCTAAGTTCTTTGGAG
>AC003044.1--AC004485.1
AAACGTCCTTTTGCTGAATATTCTACCTACG*GATGACTGATGTACCTCCAGCATCAAGCTCA
>AC003044.1--AC004485.1
CAGTCTATGGCTCCAGTTGCTTCTGTTCCAG*GAAGAAGCAAAGAAGTCTAAGTTCTTTGGAG
>AC005258.1--AC000093.1
TGCACAGCCGCGGCGGCAGCAGCAGTGAGAG*GACATTGACAAGCAGTACGTGGGCTTCGCCA
>AC005670.3--AC103810.2
TTCAGGGGCAAAATCTAGAGAATCTGAAACG*AGCTATCAGCTTCCCAGTTTGCACAATTCAT
>AC005670.3--ARL17A
AATCTGAAAATGCCCCTGGGGAGACACATGC*CAAGACAGTGAGTGATGCAGCCGTTTCCCAC
>AC005670.3--ARL17A
AATGCCCCTGGGGAGACACATGCACAAGACA*AGCTATCAGCTTCCCAGTTTGCACAATTCAT
>AC005670.3--ARL17A
AATGCCCCTGGGGAGACACATGCACAAGACA*CCTGATCTAGTTGAATCTGAAGATGTGGACC
>AC005670.3--ARL17A
AGAGAGGAGACACACACACCCAATCCTAAAC*CACAATGAGGCAGAAAAGTATAATCAAAATC
>AC005670.3--ARL17A
CCAGCTCCCTTGGGGCTGGACTCCCGAGCCT*TTTTTTCTTCGGCAGCGTTCTCCACAGATGC
>AC005670.3--ARL17A
CGTGTTCCTGCAAAAGACATGATCTCGCTGT*TTTTTTATGGTTGCACATCTTTTGTTCTAAC
>AC005670.3--ARL17A
GGCTTTCACTATGTTGGCCAAGCTGGTCTCA*AAACTCCTGACTTCAAGTAATCCACCCACCT
>AC005670.3--ARL17A
TTCAGGGGCAAAATCTAGAGAATCTGAAACG*AGCTATCAGCTTCCCAGTTTGCACAATTCAT
>AC005670.3--ARL17A
TTCTTGGGGAGAAGGGAATAATATAATTGGT*AAAAAAAAAAAAAAAAAAGTTTTGAAAAATA
>AC005670.3--ARL17B
AGAGAGGAGACACACACACCCAATCCTAAAC*CACAATGAGGCAGAAAAGTATAATCAAAATC
>AC005670.3--ARL17B
GGCTTTCACTATGTTGGCCAAGCTGGTCTCA*AAACTCCTGACTTCAAGTAATCCACCCACCT
>AC005670.3--ARL17B
TTCTTGGGGAGAAGGGAATAATATAATTGGT*AAAAAAAAAAAAAAAAAAAAGTTTTGAAAAA
>AC005670.3--ARL17B
TTCTTGGGGAGAAGGGAATAATATAATTGGT*AAAAAAAAAAAAAAAAAAGTTTTGAAAAATA
>AC006518.7--CA6
ATCTCAACCCTCTGCTGGTGATGGGAACCAG*CAGAGTGTTGAGATTACAAGCGTGAGCCACT
>AC007563.4--LINC01921
CTGGGCAGGGCTGTGGCTGTTTCAGGGCAAG*CCATTGTGACAAGGCACCATCAATCATCATG
>AC008114.1--PLBD1-AS1
AAAACCACAATGTTTCACAAAGTCAATAAAG*GTAGAAGCTCCCTGGAACAACTGCTGGAATA
>AC008119.1--ISCU
GCTGGGGGTCCCACCTTCTGTCCTCTCCATG*GTTGTTGATCATTATGAAAATCCTAGAAACG
>AC008467.1--AC116428.1
CTCTGGACTTCCCATCGGCCCTCAGGGACGT*GGTACGGACAAACCCCTAAGCTGAAGGTCTC
>AC008554.2--ZNF93
GGAGCCCAGACTGTTGCTTCCTGGTCTGGTG*ATGAGTGGCCGACTCTGGAGCCCAGGTTGTT
>AC008555.5--AC008555.4
ACTCCACAGCCAGCTAGATTTGGATAAAATG*AAACAAGCCCTGGAAAAACAAAGAATGGAAC
>AC009053.3--AC009060.1
CCAACTCCGTGGAAGCTCTGCTGAGATGCAG*ATGGCTTTGGAGGGGTGCAAATGCTACGTGT
>AC009053.3--AC009060.1
CCAACTCCGTGGAAGCTCTGCTGAGATGCAG*GTTCTCAAAAAGTGGTGTAGACTCAGGCAAA
>AC009081.2--APLF
AGCCTCAGGGACGTCCTGCTCCTGGGGGCAG*GAAATGCCACAATCATTTTCTGCAATCACAT
>AC009081.2--APLF
TGTTCTTTCTCGTCTGATGGGAAGGATCCAA*GAAATGCCACAATCATTTTCTGCAATCACAT
>AC009704.3--DIPK1C
ATGAACTTCAGGTGATAATGACGTGCCAAGT*TGCCAGGACTACCAGGGCGGCACGCTGGCCG
>AC010086.3--RBMY2EP
CCTGAATATGTTTTCACTTCAAGCTCTACAG*AAATCATCCAAGCTCCCAAGAAACCAGGGAT
>AC010327.2--ACTA1
TCCTCCAACTACCGCGCTTATGCCACGGAGC*CGCACGCCATCATGCGCCTGGACCTGGCGGG
>AC010327.2--DES
CGGGCGGCCCCCTCACTGACCCTCCAAACGC*CCCTGTCCTCACTGCTCCCTGAAGCCAGCCT
>AC010329.1--AC079804.3
TAACTCGTGTGCTTCTTCGAGATCTTATTCC*TAGATTTGGACTGCCCTTATGGATCAGCTCA
>AC010329.1--AC127526.5
ATGTGTGTTATAACCCATCTGAGCCTCCTGC*ACCACCGTTTTTGAAATAAGAATAAGAACTG
>AC012101.2--AC012101.1
GTGCTGAATAAAAGAAGCCAGACATGAAAGA*TAAGGATGCTCTGGCTGACTAACGTTCTCTT
>AC012178.1--AC007221.1
CATGCAGAGCCAGTTGGCCACATCCTTTGGG*ATGAGCATCTCAGCATCTCAGAGGTGAAGAA
>AC013265.1--AC062021.1
ACAATGGACAACCTAGAAGCAGATGTGCCTG*GAAATGCAATGAGTCAATATACTTACTGAGA
>AC013265.1--AC062021.1
CCAGCTTCAAGTATTATACCAACAAAAAAAA*GAAATGCAATGAGTCAATATACTTACTGAGA
>AC013265.1--AC062021.1
GGAAGAGCATCAGAAAAATGAATAAGTGAAG*GAAATGCAATGAGTCAATATACTTACTGAGA
>AC015631.1--AC079380.1
CCTTGATCTTCAACTTCTGGCATCCAAAACT*GCTATCATTGCCAACTGTATTCAGCGGATCC
>AC015911.10--SLFN13
GTGAAAGTCAGTTTCTGAGGAATTTCATCAG*GTCTTTCAGGACTGATTTTGGTGGAAGAAGA
>AC016573.1--AC008632.1
TCACAGTCCCTTCCAAGAAGCTCCACAATGG*AGACAGCAGGGTTCAGGCTTCGAGTTGTGGT
>AC018630.2--BPIFA2
ATCTCAACCCTCTGCTGGTGATGGGAACCAG*CAGAGGGGTGGGATTGCTTGGGTTAGCTAAT
>AC020656.1--FOXP1
CGAGGTGGGTGGATCACAAGGTCAGGAGATC*GACCATCCTGGCTAACACGGTGAAACCCTGT
>AC020656.1--FOXP1
TTCTTTGATAAGAACTGAATGTGGCCGGGCG*CCCGCCACCATGCCCGGCTAATTTTTTGTAT
>AC020656.1--SOX9-AS1
CTTTGGGAGGCCGAGGTGGGTGGATCACAAG*GTCAGGAGATCGAGACCATCCTGGCCAACAC
>AC020656.1--TMBIM6
TCGAGACCATCCTGGCTAACACGGTGAAACC*CCGCCACCACGCCCAGCTAATTTTTTGTATT
>AC021087.5--AC073210.3
GAGGAAGAGATTCTGGGAGGATCCCGAGGAC*CCCCCCAGTTGCAGCCATGTCAAGACTGATG
>AC023824.5--AC023824.6
CTTGACTCAGCCTCCTGCGATTGGACTACAG*GGACCTGCCTTTGCCTCCTGCTCGAAGGAAG
>AC024559.2--AC080129.1
GGAAGAAAATCCAAGACTTCTCTGGAGAGAG*ATACCAGCCTGGAAGACAACGGAGCTGTCAT
>AC025171.1--KCNK6
CCTGGGCGACAGAGCGAGACTCCGTCTCAGA*AAAAAAAAAAAAAAGAAAAAAGTGTGTGCTA
>AC026412.1--AC024560.4
ACTTCCTTCTCCTCTCTGAATTTCTTTATTT*AAAAAAAAAATTAGAGATGGAGTCTCACTGT
>AC053545.1--BBS12
TCCGGATTCTGTTGTCCTTTGCTGCCTGTTG*ATCATGATACATGGTGATGGCTTGCAGAGTC
>AC061975.8--PYY2
ATTTTTATTTGTCCATTATTCCTGAATAAAG*GACGTCTCCCAGGCTTGCCTCCCCGGATCGG
>AC067930.1--HBB
CAGGTCAATGTCATCATCCTCGTCATCCTCT*GCTGGTGTGGCTAATGCCCTGGCCCACAAGT
>AC068580.4--AC132217.2
GGCCCCTCCCACACACACCCACACACTCGCC*AACACAGCACACACGAACACAGCACACACAG
>AC068700.2--PKIA-AS1
TGCTTATACAACCATTATACCAAAAGTAAAG*TCTGCTGGTTGAATCCATGGATGCAGAACCC
>AC068756.1--AC108690.1
CACCTCAGCCTGCCTAGTACTGGGACTACAG*AAATTGCCACAGTCACCTCAACCTTCAGCAA
>AC068756.1--AC108690.1
TGGGACTACAGGTACGCGCCACCACGCCCAG*AAATTGCCACAGTCACCTCAACCTTCAGCAA
>AC068774.1--DENND5B
CTGTATCAGTGAAGTGCAAAGCTAGTGAAAG*GCGAAAATTTTGACCAGAGTCCTTTGAGAAG
>AC068993.2--SYT1
AGGGGCAACAACTATACAGGAAGGAGAACAG*GCTGCCCCTCCAAGAGCGGAGGCAGCGAGAG
>AC069503.4--HPD
CAGCCCTGGAGGATTTACAAAATACATTAAG*ACGACTTACAGTGACAAAGGGGCAAAGGTAA
>AC069503.4--HPD
CAGCCCTGGAGGATTTACAAAATACATTAAG*CCTGAGAGAGGCCGATTCCTCCACTTCCACT
>AC069503.4--HPD
CAGCCCTGGAGGATTTACAAAATACATTAAG*GCAAGAGTTCTGGAGCCGCCACTGCACTCCA
>AC069503.4--HPD
CAGCCCTGGAGGATTTACAAAATACATTAAG*TGAGCAGCATAGAAGAAATAATGCCTTTTGA
>AC073210.3--AC021087.5
GTCACGAGCAACACCCGTCGAACACAGCCAC*CCACCTGAGGTCAGATCAAGAAGCTGTCACA
>AC073210.3--SDHA
GTCACGAGCAACACCCGTCGAACACAGCCAC*CCACCTGAGGTCAGATCAAGAAGCTGTCACA
>AC073270.1--LINC01005
AATGTTGACATATTCCTGGCCCAGAATTCAG*GTGGTTTGGTGACTCTCAAACCATGATTCAG
>AC073270.1--LINC01005
AATGTTGACATATTCCTGGCCCAGAATTCAG*GTTGTATTGTGCCATATATCTGTGTCCAAAT
>AC073316.1--AC024028.1
GGCTGAGAATGACCTCCATCCGCAGGACGAG*GATCATCCGTCAATGCCAGCAAGTGAAGATA
>AC073529.1--MID1
AGAAGTAGAAGCCTGCACAGTCCACAGAACT*ATAGCTGATCAGCTTCCTTGGGTTTTGCTGA
>AC079804.3--AC010329.1
AGATGATCGAGTGTGGATCAAGGACTGGAAA*GTAGCCTGTTTGTGTCCACGGTGGAAAGGAC
>AC079804.3--AC010329.1
ATGTGTGTTATAACCCATCTGAGCCCCCTGC*ACCACCATTTTTGAAATAAGAATAAGAACTG
>AC079804.3--AC010329.1
GAGTGTGGATCAAGGACTGGAAAGTAGCCTC*TTTGTGTCCACGGTGGAAAGGACCCCAGACT
>AC079804.3--AC127526.5
AACTTGTTTGTATAATGCCACTCAGTACAAG*GTATGTAGCCCGAGAAATGACCGACCTGATG
>AC079804.3--AC127526.5
ATGTGTGTTATAACCCATCTGAGCCCCCTGC*ACCACCGTTTTTGAAATAAGAATAAGAACTG
>AC079804.3--AC127526.5
CATCTGAGCCCCCTGCAACCGCCGTTTTTGA*AATAAGAACTGGCCTTTTCCTAGGTGATACA
>AC079804.3--AC127526.5
CCCTTGCAGAGTGACCCTGAAGAAGACGACA*CCCTGCTCCAGTCACACCCGGAAGCTGACTG
>AC087457.1--ACTA1
ACAAAGGAGTAGCCACGCTCAGTGAGGATCT*CATCATGCGCCTGGACCTGGCGGGCCGCGAT
>AC087457.1--ACTA1
AGAAGCATTTGCGGTGGACAATGGATGGGCC*CTCGCTGTCCACCTTCCAGCAGATGTGGATC
>AC087457.1--ACTA1
AGAAGCATTTGCGGTGGACAATGGATGGGCC*GGCCTCGCTGTCCACCTTCCAGCAGATGTGG
>AC087457.1--ACTA1
AGAAGGTGTGGTGCCAGATCTTCTCCATGTC*ATCCCAAGGCCAACCGCGAGAAGATGACCCA
>AC087457.1--ACTA1
AGAAGGTGTGGTGCCAGATCTTCTCCATGTC*CCTGACCCTGAAGTACCCTATCGAGCACGGC
>AC087457.1--ACTA1
AGCAGGGTGGGGTGCTCCTCGGGAGCCACAC*GAGGCTCAGAGCAAGAGAGGTATCCTGACCC
>AC087457.1--ACTA1
CAGTGAGGATCTTCATGAGGTAGTCAGTGAG*AAGCTGTGCTACGTGGCCCTGGACTTCGAGA
>AC087457.1--ACTA1
CAGTGAGGATCTTCATGAGGTAGTCAGTGAG*ATCGTGCGCGACATCAAGGAGAAGCTGTGCT
>AC087457.1--ACTA1
CATCTTAGAAGCATTTGCGGTGGACAATGGA*GCCGGCCCTTCCATCGTCCACCGCAAATGCT
>AC087457.1--ACTA1
CATGGCAGGGACATTGAAGGTCTCAAACATG*ATCTGGCACCACACCTTCTACAACGAGCTTC
>AC087457.1--ACTA1
CATTGAAGGTCTCAAACATGATCTGAGTCAT*CTCACCGAGGCCCCCCTCAATCCCAAGGCCA
>AC087457.1--ACTA1
CATTGTAGAAGGTGTGGTGCCAGATCTTCTC*CACGGCATCATCACCAACTGGGATGACATGG
>AC087457.1--ACTA1
CCACACGGAGCTCATTGTAGAAGGTGTGGTG*CCAGATCATGTTTGAGACCTTCAACGTGCCC
>AC087457.1--ACTA1
CTGGAAGGTGGACAGAGAGGCCAGGATGGAG*CCGCCGGAGCGCAAATACTCGGTGTGGATCG
>AC087457.1--ACTA1
CTTGCGGATATCAATGTCACACTTCATGATG*CTGTGCTACGTGGCCCTGGACTTCGAGAACG
>AC087457.1--ACTA1
GCCAGATCTTCTCCATGTCGTCCCAGTTGGT*GACATGGAGAAGATCTGGCACCACACCTTCT
>AC087457.1--ACTA1
GCGGGGCCTCTGTGAGCAGGGTGGGGTGCTC*CTCAATCCCAAGGCCAACCGCGAGAAGATGA
>AC087457.1--ACTA1
GCTCATTGTAGAAGGTGTGGTGCCAGATCTT*CACGGCATCATCACCAACTGGGATGACATGG
>AC087457.1--ACTA1
GGAGCTCATTGTAGAAGGTGTGGTGCCAGAT*CTTCTACAACGAGCTTCGCGTGGCTCCCGAG
>AC087457.1--ACTA1
GGCATCTTAGAAGCATTTGCGGTGGACAATG*GAAGGACCTGTATGCCAACAACGTCATGTCG
>AC087457.1--ACTA1
GGTGCCGCGGGCGGCCCACGATGGACGGGAA*CACGGCATCATCACCAACTGGGATGACATGG
>AC087457.1--ACTA1
GTGAGCAGGGTGGGGTGCTCCTCGGGAGCCA*CAGCTGAGCGCGAGATCGTGCGCGACATCAA
>AC087457.1--ACTA1
GTGGGGTGCTCCTCGGGAGCCACACGGAGCT*CGTCATGGTCGGTATGGGTCAGAAAGATTCC
>AC087457.1--ACTA1
TCTGTGAGCAGGGTGGGGTGCTCCTCGGGAG*AAGATGACCCAGATCATGTTTGAGACCTTCA
>AC087457.1--ACTA1
TCTGTGAGCAGGGTGGGGTGCTCCTCGGGAG*CACGGCATCATCACCAACTGGGATGACATGG
>AC087457.1--ACTA1
TCTTCTCCATGTCGTCCCAGTTGGTGATGAT*CGTCATGGTCGGTATGGGTCAGAAAGATTCC
>AC087457.1--ACTA1
TCTTCTCCATGTCGTCCCAGTTGGTGATGAT*CGTGGGCGACGAGGCTCAGAGCAAGAGAGGT
>AC087457.1--ACTA1
TCTTCTCCATGTCGTCCCAGTTGGTGATGAT*GCCGTGCTGTCCCTCTACGCCTCCGGCAGGA
>AC087457.1--ACTA1
TGAGCAGGGTGGGGTGCTCCTCGGGAGCCAC*GAAGATCTGGCACCACACCTTCTACAACGAG
>AC087457.1--ACTA1
TGTAGAAGGTGTGGTGCCAGATCTTCTCCAT*CATGGTCGGTATGGGTCAGAAAGATTCCTAC
>AC087457.1--ACTA1
TTGCTGGAAGGTGGACAGAGAGGCCAGGATG*GAGTACGACGAGGCCGGCCCTTCCATCGTCC
>AC087457.1--ACTA1
TTGCTGGAAGGTGGACAGAGAGGCCAGGATG*TACGACGAGGCCGGCCCTTCCATCGTCCACC
>AC087457.1--ACTA1
TTGTAGAAGGTGTGGTGCCAGATCTTCTCCA*GTCATGGTCGGTATGGGTCAGAAAGATTCCT
>AC087457.1--ACTA2
TGCCTCATCGTACTCTTGCTTGCTAATCCAC*TCTGTCTGGATCGGTGGCTCCATCCTGGCCT
>AC087457.1--ACTA2
TGGAAGACTCCAAGAAGCATAATACCGTCAT*CTGTCTGGATCGGTGGCTCCATCCTGGCCTC
>AC087457.1--MYH7
ATCCTGACTGGAAGGTAGATGGAGAGAGAAG*AAGGTGCGCATGGACCTGGAGCGAGCGAAGC
>AC087457.1--NPPA
CATTTGCTGGAAGGTGGACAGAGAGGCCAGG*AGAGGGGAACCAGAGAGGAACCAGAGGGGAG
>AC087457.1--NPPA
CTCTTGCTTGCTAATCCACATTTGCTGGAAG*GGGCAGGATGGACAGGATTGGAGCCCAGAGC
>AC087457.1--NPPA
TCTTGCTTGCTAATCCACATTTGCTGGAAGG*GGGACAGACGTAGGCCAAGAGAGGGGAACCA
>AC087457.1--NPPA
TTTGCTGGAAGGTGGACAGAGAGGCCAGGAT*GGAGAGACAGAGCAGCAAGCAGTGGATTGCT
>AC087473.1--PSMB2
GGCCGAGGCGGGCGGATCACGAGGTCAGGAG*AGCGAGACCATCCTGACCAACATGGTAAAAC
>AC092161.1--AC073349.5
GATGTGTCAGGCATGCCTAGGCGGTGTGGGG*GTGAAGATGCGGGACCCAAACCACATTTCAC
>AC092718.8--AC009070.1
AGCTCTGTGCTTTATCATAGAAATGGATCTG*GTTGCACTGCTGTGATCCATCCTCATCTCCT
>AC092745.5--AC145124.1
ATCCTGGAAGCTGGGCTTCGGAAGCGTCCAG*CTGTTGTTCCCCACCAGAACCAAATGAGCGC
>AC093787.2--IGK@
ACATATCTGTCTTTTAATATAGGCAGAAACA*TAGATAAAAAAAGGACTTGTCTCAGAGCATG
>AC093787.2--IGK@
ATGTGCTTTAAATATTATGAATAGTGCTTGC*GTGTGTGTGTGTGTGTGTGTGTGTGTCTATA
>AC093787.2--IGK@
ATGTGCTTTAAATATTATGAATAGTGCTTGC*GTGTGTGTGTGTGTGTGTGTGTGTGTGTCTA
>AC093787.2--IGK@
GAGTGGACAAAATAATTATGAGAAGATGAAT*CTTCAAATCAGAAAGAGGGAAAAAAGCTTAT
>AC097500.1--FSIP2-AS1
AAGGCGGGTGATTTCTGCATTTCCAACTGAG*TACATGAATCACTCCTACTAAGAGATCCTGA
>AC098825.1--PARM1
AAGAAATTTATCATGAAGATAAGAAGAAAAG*ATCACCCTTGAAGGCAGACATTGTCCTCTCA
>AC099343.3--AC099343.4
AAAAATGGAAGTATCCAAGAAAACGCGTGTG*AAGAAACTGTTAGAGCACCCTGGGATGTGTG
>AC099654.1--AL136102.1
AAAGTAGAAAAGCAGCCCTGGCTCTTCCCCA*GGTCTCCAGCTCCACAACCCTCAAAGTAGAA
>AC099654.1--AL136102.1
ACCAGCTCCCCCGTGGATCTCCAGCTCAATG*ATCCTCAAACTACAACAGCAGCACCAGCTCC
>AC099654.1--AL136102.1
ACTACAACAGCAGCACCAGCTCCCCCGTGGA*TCTCCAGCTCCACAATCCTCAAACTACAACA
>AC099654.1--AL136102.1
CCAGCTCCATGACCCTCAGATTTGAACAGTG*GCAGCACCGGCACCTCCCCAAGGCTCCAGCT
>AC099654.1--AL136102.1
CCCTGACCCTCAAACTCCAACTGCAGCACCA*GCTCCTCCCCAGATCTCCAGCTCCCTGACCC
>AC099654.1--AL136102.1
CTAGAACAGCAGCACCAGCTACTCCCTGGGT*CTCCAGCTCCACAACCTTCAAACTAGAACAG
>AC099654.1--AL136102.1
TCATTCTACACCAAAACCTCACGGGAAAAAT*GCTCCACGACCCTCAGACTTAAAAAGCCGTA
>AC099654.1--AL136102.1
TTACACGGTGGTAGCACCAGCTCCTCCCCAG*CACCAGCTCCACAACCCTCAACTTACACGGT
>AC104088.1--ITGA6
GATTCACTTCATGTCTCTGAGTCACATTTTG*GTTGCTCGTGGGGGCCCCGCGGGCAGAAGCG
>AC104088.1--ITGA6
GCCTCAGCCTCCTGAGTAGTTACAACTACAG*GTTGCTCGTGGGGGCCCCGCGGGCAGAAGCG
>AC104088.1--ITGA6
TCTCCCTTCCTTCACACTGTGCTTCAGCGAG*GTTGCTCGTGGGGGCCCCGCGGGCAGAAGCG
>AC104411.1--AC092944.1
AAAAATACCATCCAAAGCTACATGACTTGAG*AGACAAGGTCTCACTATGTTGCCCAGGCTGG
>AC104850.2--CX3CR1
ACAGTGAGTCACACTTGTTAAGTACAGTGTG*GCCTTCACCATGGATCAGTTCCCTGAATCAG
>AC105052.5--CCDC146
GCCTAAATAAGGATCAGGACCAAGGGAAGGG*TTACATGCTATGGGAAAACTTCCTGGAACCA
>AC106791.3--KLHL3
AGCTCAAGCCTTCCACTGATTCTACGTTACG*TGTCAAGCTGAGCTCCCAGACTCTGATACAG
>AC108063.2--PROM1
GTGGAAGCAGGGAGAGCAAGCATGTCACAAG*GGATGGTGCCTTGAGTGAATGACCCCCTTGG
>AC108519.1--AF228730.5
ACACTATCAGGGGACCTGCCCTGATAATCAG*TCTACAGGTGTATCCAGCAGCTCCAGAGAGA
>AC108519.1--AF228730.5
ACTTCATTATGACATAGATTCTATTGCTCAC*GTTTGTTGCTGACCTTCTCCTTATTATCACC
>AC110611.2--AC110611.1
AGACTTCTTAATTTCCCCCTGGATAGAGCAG*GTCTTGTTGAACTGGCTAGTGCTTCAACAAC
>AC110611.2--AC110611.1
AGTACCTGTTTAAAAGGCATCCAAAAGATAG*GTCTTGTTGAACTGGCTAGTGCTTCAACAAC
>AC110611.2--AC110611.1
GACTTCTTAATTTCCCCCTGGATAGAGCAGG*ATTCGATTCAGTTGGAAAGACATGGTTAATG
>AC110611.2--AC110611.1
GACTTCTTAATTTCCCCCTGGATAGAGCAGG*GGTACTCCTTAATGACTTTACGAAGTTCTTG
>AC110611.2--AC110611.1
GACTTCTTAATTTCCCCCTGGATAGAGCAGG*GTCTTGTTGAACTGGCTAGTGCTTCAACAAC
>AC115220.3--LINC01005
GCGCTCCAGCTTTGTGGGAGAAACTTTCAAG*ATGGAGTCTTGCTCTGTTGCCCAGGCTGGAA
>AC115220.3--LINC01005
GCGCTCCAGCTTTGTGGGAGAAACTTTCAAG*GTGGTTTGGTGACTCTCAAACCATGATTCAG
>AC115220.3--LINC01005
GGATGGAATGCTCATCACCTACAGCAGTCAG*GTTGTATTGTGCCATATATCTGTGTCCAAAT
>AC117834.2--AC117834.1
AACGCAAAGAAAGAAGAGACCATTCCACTGA*AAACTGTTTTGTACTGGCAAGTCCCAATGAA
>AC117834.2--AC117834.1
AACGCAAAGAAAGAAGAGACCATTCCACTGA*CCTCCTTTGATGTTAGTTGTGGACACATCAC
>AC119427.2--ACTN2
CATGATGCATTTTGGGGGTTACATCCAGAAG*CAGGAGCCCTGGAAGACCAGATGAACCAGCT
>AC119427.2--CRYAB
CAAAGATCTGCTCCTCGCTCGCACCTGCCTC*TGGACATCGCCATCCACCACCCCTGGATCCG
>AC119427.2--MYH7
CATGATGCATTTTGGGGGTTACATCCAGAAG*AGGCCCACCAACAGGCTCTGGATGACCTTCA
>AC119427.2--NPPA
ATGCTGAAGCAGAGGCTGAGACCGAGGAGAC*GGACAGACGTAGGCCAAGAGAGGGGAACCAG
>AC119427.2--TNNT1
CCATTGACCACCTGAATGAAGATCAGCTGAG*GGAGAAAGCCCAGGAGCTGTCGGACTGGATC
>AC119427.2--TNNT1
GATCCCCGATGGAGAGAGAGTGGACTTTGAT*GACATCCACCGCAAGCGCATGGAGAAAGACC
>AC124242.1--AC124242.3
ACAAAAGCAATTAAGAGAGCTCAAACAATGG*ACAGCAAGACCCCCCGCAAGACAGCAAGACC
>AC124242.1--AC124242.3
ACAAAAGCAATTAAGAGAGCTCAAACAATGG*GTCTCAATCTTAACTTCCAGTTTAATGGAAT
>AC126182.3--NDUFB4
ATGGGATTTCACATCTCAAGACATCTCCCAG*GAAAATCCTGCCTTGCTTCGTTGGGCCTATG
>AC126182.3--NDUFB4
GCAGACCTGGAGTTCAAGATGGGCTCCAAAG*GAAAATCCTGCCTTGCTTCGTTGGGCCTATG
>AC127526.2--AC127526.5
ACCTCCCCATCCCACTTCACTGCCATCCCAG*GTGATCGAGTGTGGATCAAGAACTGGAACGT
>AC127526.5--AC010329.1
CCAGCCTGGATCCACCACAGCCATGTAAAAC*CTGCAGTGCCTGAAACCTGGGAGGCAAGACC
>AC127526.5--AC079804.3
CCAGCCTGGATCCACCACAGCCATGTAAAAC*CTGCAGGGCCTGAAACCTGGGAGGCAAGACC
>AC127526.5--AC079804.3
GTGGACTTCACAGAGATGTCAAAGTGTAGAG*ATGATCGAGTGTGGATCAAGGACTGGAAAGT
>AC132217.2--ACTB
GCATTTTTCCTTTTTTTTTTTTTTTTGTTTT*TTTTTTTTTTTTTTTTGGCTTGACTCAGGAT
>AC132217.2--ANXA2
ATGCACACACAGCACACACACAAACGCACAG*CACACACACACACACACCACACACGCACACA
>AC132217.2--CD93
ACACACAACACACAGCACACGCATGAACACA*ACACACACACACACACACACACACACACACA
>AC132217.2--COL5A1
CACATGAACACAGCACACAGCACACACATGC*GTGTGTGCTGTGTGTGCATGCATGGTATATG
>AC132217.2--CSF2RB
GAAGAAGACTGGTCTCTCCCACCACACAGAG*GCCTGGAGGAGGCAGAGGCCAGGAGGGAGAG
>AC132217.2--CSH1
ACAACACACAGCACACGCATGAACACAGCAC*ACAAACTCGCACAACCATGACGCACTGCTCA
>AC132217.2--CSH1
ACTGCCGCAAGTCTGCAGCCCGGCGCCACCA*TCCTGCATGACTCCCAGACCTCCTTCTGCTT
>AC132217.2--CSH1
CCTGGGGCTTCTCCTGACCCAGTCCCCGTGC*CCCAACTCCCCGAACCACTCAGGGTCCTGTG
>AC132217.2--EGFR
AGCACACGCACACACATGCACACACAGCACA*CTAGCACACAGCACACAGCCAATGGCCCAAT
>AC132217.2--FLT1
GACAAAAGCAAGAACGCACGAAGCCAGCCTG*AGGTGTGACTTTTGTTCCAATAATGAAGAGT
>AC132217.2--LGMN
AGCACCCACCTGCACACACACATGCGCACAC*ACACACACACACACACACACCCACCCTGACC
>AC132217.2--MEG3
GAGCACACAGCACACAAACGCACAGCACACG*CACACACACCCCCTCGTGTATCTACCCCACA
>AC132217.2--MPEG1
AACACAGCACACACGAACACAGCACACACAG*CACACACACACACACACGCACACACATACCA
>AC132217.2--SEPT9
AGCACACACACAAACGCACAGCACACACAAA*CACACAACACACACACACACACACACACACA
>AC132217.2--SEPT9
CACACAGATGAGCACACAGCACACACACAAA*CACAACACACACACACACACACACACACACA
>AC132217.2--SPARC
GAAGCCCAAAGAGCCAAATTGTCACAATTGT*AGAACCCACCAACCAGAACGTGGTTTGCCTG
>AC132217.2--SYNPO
ACACAGCACACACAGCACACACACAAACACA*CACACACACACACACACACCACACACACACA
>AC132217.2--TP53I11
CAATTGGCAAAATAAAGGAATTTGGCACTCC*CCACCCCCCATCCTTCTCCCTACACTCCCAG
>AC132217.2--TP53I11
CAGCACACACACAAACACAGCACACACATGC*ACACACAACACACACACATACACACCCCTGA
>AC138409.2--LINC02241
CTAGTCTCAAGGAAAATATATTAATACACAC*ACACACACACACGCATGCACACACACACACA
>AC138409.2--LINC02241
TGGAATTTTGCCGATTTCATGACTAACCAGT*TTCTGCAGGCTGGAAATCCAAGATGAAGTCA
>AC138776.1--ANKRD20A21P
ATGTTGGAATTTCTGATGAATCTGCAGTCAG*CATTTTGCATGAACTGTGTGTGGATTTATTG
>AC145124.1--AC092745.5
TGAAGGAGACAAGAAAAACCAACGGGAACAG*ATAGGGTCTCTCTCTCTGCCACCCAGGCCAG
>AC145124.1--AC092745.5
TGAAGGAGACAAGAAAAACCAACGGGAACAG*CCCAGTAAAGTCGCCATTCCCCATTTTACAG
>AC145124.1--LINC02018
TAAAAGTACGATAACGATATCTACCACGAAG*ATAGGGTCTCTCTCTCTGCCACCCAGGCCAG
>AC145124.1--LINC02018
TGAAGGAGACAAGAAAAACCAACGGGAACAG*ATAGGGTCTCTCTCTCTGCCACCCAGGCCAG
>AC145124.1--LINC02018
TGAAGGAGACAAGAAAAACCAACGGGAACAG*CCCAGTAAAGGCTCCATTCCCCATTTTACAG
>AC145124.1--LINC02018
TGAAGGAGACAAGAAAAACCAACGGGAACAG*CTGGGGAAACTGAGTTACAGAGCTTTTCTGC
>AC233263.6--AC093787.2
ACATACAATGGTAAATCATTCAGCCTTAAAC*GAGGGAAATATTCTGACATATGTTGCAACTT
>AC239585.2--AC231759.2
GTCTGCAACACTGGCATTGCCACGTATCCCG*ATCCCATTGTGGCATTGTGGTCAGAGAATAT
>AC239804.1--LINC01719
AATTCCTTAAATATATAATTTTAATACAATT*TTTTTTTTTTTGAGATGGAGTCTCACTCTGT
>AC244197.3--LINC00894
AGCTCGAGGAAGCAGGAATGCTGGAGATGAG*ACGGAGTTTTGCTCTTGTTGCCCAGGCTGGT
>AC244197.3--LINC00894
CAGGCCCCCTCCTGAAAGAGGCTCCTTGAAG*AGACAGAGAAGAGTCCATGAGAATCAACATT
>AC244197.3--SMR3B
GAAGAGCAAGTAAGGAGGAGGAGGGGGGAGT*CTCCTCCTCAACCTTTTGGCCCAGGATTTGT
>ACTA1--AC087457.1
ACATGGAGAAGATCTGGCACCACACCTTCTA*CATGTCGTCCCAGTTGGTGATGATACCATGC
>ACTA1--AC087457.1
ACCCTGAAGTACCCTATCGAGCACGGCATCA*TCCCAGTTGGTGATGATACCATGCTCGATGG
>ACTA1--AC087457.1
ACCTTCTACAACGAGCTTCGCGTGGCTCCCG*GTGAGCAGGGTGGGGTGCTCCTCGGGAGCCA
>ACTA1--AC087457.1
ATCTCACCGACTACCTGATGAAGATCCTCAC*AAGGAGTAGCCACGCTCAGTGAGGATCTTCA
>ACTA1--AC087457.1
CCCGTCCATCGTGGGCCGCCCCCGACACCAG*GGCGGCCCACGATGGACGGGAAGACAGCGCG
>ACTA1--AC087457.1
CCTTCTACAACGAGCTTCGCGTGGCTCCCGA*GGAGCTCATTGTAGAAGGTGTGGTGCCAGAT
>ACTA1--AC087457.1
CTGTCCCTCTACGCCTCCGGCAGGACCACCG*GCCTGGATGGCCACGTACATGGCAGGGACAT
>ACTA1--AC087457.1
GCACGGCATCATCACCAACTGGGATGACATG*TAGAAGGTGTGGTGCCAGATCTTCTCCATGT
>ACTA1--AC087457.1
GGTGAAAGCCGGCTTCGCCGGGGATGACGCC*CTGGTGCCGCGGGCGGCCCACGATGGACGGG
>ACTA1--AC087457.1
TCCCGTCCATCGTGGGCCGCCCCCGACACCA*GGGCGGCCCACGATGGACGGGAAGACAGCGC
>ACTA2--AC087457.1
GCCGGGCCTTCCATTGTCCACCGCAAATGCT*TCCAAGAAGCATAATACCGTCATCCTGACTG
>ACTA2--AC094105.2
CCATCATCCTCAGCAAACTGTCGCAAGGACA*AAAAAAACCAAACACCGCATGTTCTCACTCA
>ACTR3C--AC006008.1
CCCCATCGATGTGCGGCGTCCGCTGTATAAG*GACGAAAAATGGATCTCTAATGAATACAATC
>ACTR3C--AC006008.1
TATTGAGCCCATGCGTAGCCTTTTATTTGAG*GACGAAAAATGGATCTCTAATGAATACAATC
>ACTR3C--AC006008.1
TATTGAGCCCATGCGTAGCCTTTTATTTGAG*GTACTTGGAGTTCTTCCAGGGCGTTACCGTG
>ADAM12--AC132217.2
TCTAATACATTCCAACTCGTATAGCATGCAT*ACAGCACACGCACACACAGCACACACATGAA
>ADAMTSL4-AS1--AC132217.2
AAGCACACCTGAGGCATATGTGCACACACCC*AAACGCACAGCACACACACGCACACACATGC
>ADGRG1--TG
CTGGGGGGGTCAGATGGTACTCCAAGGGACT*GGTCCCAGGCCATCCAGGTGGGTACCTCATG
>AF228730.5--AC108519.1
ACACTATCGGGGGACCTGCCCTGATAATCAG*TCTACAGTTGTATCCAGCAGCTCCAGAGAGA
>AL031259.1--AL669831.3
ACGCGCTTTCAACCGGCGGAGACACTGGCAG*CTGATCCATATGAATTCCTCTTATTAAGAAA
>AL050309.1--KLF8
GGGAACTGAATGATGAGAACATTTGGACATG*ATATGGATAAACTCATAAACAACTTGGAGGT
>AL078587.2--AL450124.1
TGGCCCCGAGCGGAACTGGAGAAAGCTGAGG*GGGTTGGGCATGTTCCAGGATCTGTCTATCG
>AL080275.1--COL9A1
GACTTCAAGAATGGAGCCGTGGACCTTCGCA*GAAAATTCCAGTTTTCTTCTTTGTGTGCAGT
>AL080317.1--AL080317.4
AAGAGAATGACTGGCTTGCCTAATAAAAAAC*AAATAAAAGAAGTGTGGATACTGAAAGCATC
>AL080317.1--AL080317.4
GATTGTGAAGCCTCCCCAGCCATGTGGAACT*AAATAAAAGAAGTGTGGATACTGAAAGCATC
>AL096870.3--AL136295.7
GTACCTTCTACTCATCTGCCGGCAGGGCCAG*GGATGTTCCTTGCTGAGAAAAAGAATTCAGC
>AL117335.1--AL109809.5
AACCGCTGAATAGACTGCCATTCAAGCCAAG*GATTTGCTGAGCACCTACTATGCATCAGACG
>AL117335.1--AL109809.5
CCCCAAGAGTTGCTGCTCACCAGGCTGGAAG*GATTTGCTGAGCACCTACTATGCATCAGACG
>AL117335.1--AL109809.5
CCCCAAGAGTTGCTGCTCACCAGGCTGGAAG*GTATGCCCATGTAAGTTCTGGCCAATGGAAA
>AL132656.2--NUTM2B-AS1
AAATTTATAGCATTCTTCTACCACCACAGGG*GCTCATATGACATTAACAGACAAGACACTTT
>AL132656.2--NUTM2B-AS1
AGAGAAGCTTTACTGGTGTTCCAGCCCTAAG*AAACTTAAATCCTCAGTGTGACACAACTCAG
>AL132800.1--AC073348.2
CACCTTAAGAGCTGTAACACTCACTGCCAAG*GTCTGCAGCTTCACTCCTGAAGCCAGCGAGA
>AL132800.1--AL136018.1
ACCACATGGACTCACAGAGACTCCCAACCAA*GTCTGCTGCAGTTTGCTGGAGGTCCACTCCA
>AL133304.3--LINC00609
TTCCTCTGCAAGAAACCTGCTCTCTGGGACA*TGTTCGCTTCCTGGATGTTCTGACTAATTTA
>AL133405.1--STMND1
AATAATATTGATGATCCTGACCTGTGTAAAG*GCTGATGTCAGTGTGCCTCATACTGGGGAAA
>AL136102.1--AC099654.1
ACTACAACAGCAGCACCAGCTCCCCTGGGGA*TCTCCAGCTCCATGATCCTCAAACTACAACA
>AL136102.1--AC099654.1
CCAACAACAACAGCAACACCGGCTCCTCCCT*TCTCCAGCTCCACGACCATCAAACTTGAACT
>AL136102.1--AC099654.1
CTTGAGAGGCAGCACCGGCACCTCCCCAAGG*CTCCAGCTCCACAATCCTCCAACAACAACAG
>AL136102.1--AC099654.1
TCATTCTACACCAAACCTCACGGGAAAAATG*CTCCCCGACCCTCAGACTTAAACAGCAGCAC
>AL136102.1--AC099654.1
TCCCTGGAACTCCAGCTCAATGACTCTCAGA*CTAGAACAGCAACACCGGTTCCTCCCTGGGT
>AL138830.1--AL138830.2
AAAACCACTGGGCCACTCTAATGCCTTCAAG*GGACAGATCCCAGGTCTAAACCCTAGCCAAA
>AL139158.2--AL390839.2
GGCACACCTTCTCCCGGGCACTGAGGCCCAG*GTCTGCTTCAGGGGACCCCCAATCTAAGACA
>AL157400.3--PANK1
CCTCTTCTCCAAGATTCTCAGGCAATCAAAG*CATTCCCATGGTTTGGCATGGACATCGGTGG
>AL158066.2--SUGT1P4-STRA6LP
TATTGTCACATTCTTCTTGGGAGTTACTATG*GTGCAGATGCTAATTTCAGTGACTGGATTAA
>AL160154.1--LINC00375
AAGATGAGAGCCATGGTGTTTTGGTAACCTA*ATTCTGATTATCTTGGTGGAATCTTGGAAAA
>AL353743.2--AGTPBP1
ACAGCAACAGAGGAAAGGACAAAAGGAAAAG*ATTATCTGCAATTATGAAATGAAGTAACTCA
>AL353743.2--AGTPBP1
TGGAATCCTGGATCATTGTGGGAAATGAAAA*ATTATCTGCAATTATGAAATGAAGTAACTCA
>AL355377.4--AC027088.5
GTCCTCACCCTCACTCCATGAAGAGATCCAC*CTATGACCTTGGGTCCTCAGACCAACCAGCC
>AL355499.1--HULC
GTGCTCACTTCATGTCTCTGTGCCACATTTG*GTGATGTCATTGAACTCTTAAGTGCAAGATG
>AL356489.4--PRSS3
CTTGGGAGATTTTTAACAGTCCTGGATGGAG*CACCCAGAAGAATCCTGTATTTAATTGGCTC
>AL356489.4--PRSS3
CTTGGGAGATTTTTAACAGTCCTGGATGGAG*TTGCTGTCCCCTTTGACGATGATGACAAGAT
>AL357075.4--RELA
GTGCAGTGGCGCGATCTCAGCTCACTGCAGC*CTCCACCTTCCGGGTTCATGCCATTCTCCTG
>AL390839.1--AL390839.2
TTTCCCATGACTCATCTTGTGGAAGGCTGAG*GTCTGCTTCAGGGGACCCCCAATCTAAGACA
>AL391840.3--SH3BGRL2
ATCTCATTCCATCTTCATTCATGACTCCAGA*ATAAAGAAGAAGCAGCAAGATGTGGTTAGAT
>AL391840.3--SH3BGRL2
TCATTCCATCTTCATTCATGACTCCAGACAG*ATAAAGAAGAAGCAGCAAGATGTGGTTAGAT
>AL445524.2--SCAND2P
GAGCCTTTGATCCAGCAATACCCCACTTCAC*CTACAATAATTGTTGACGCTATTTGTTAATT
>AL445524.2--SCAND2P
GCTGGGATTACAGGCGTGAGCCACCGCGCCC*GGTCTATCTTTAAATTCCTATCTGAATTTGT
>AL445524.2--SCAND2P
GTTCATACTATTAAATTTTGCCCTAAAAGAC*CTCTTAATAATGATTGTTGCCAGTGACTGAT
>AL445524.2--SCAND2P
TGGGGTTTAAGTTAAATTTTTTAAAAAACTA*AAAGTGACTGGCACTAAGTGAACTTGAGATT
>AL591135.2--AL353693.1
AAGATGGGTGATTTCTGCATTTCCAACTGAG*GCAAAGCCCTCCTGTTCCCAGCCCCAAGTCG
>AL645568.1--TNFSF4
CAAAAATTCCCGTACCAATGTCAGTGCTAGG*GTATCACATCGGTATCCTCGAATTCAAAGTA
>AL669831.3--AL031259.1
CTGAGCACCGATACAAAGAAAGACAAACATC*CTGATCCATATGAATTCCTCTTATTAAGAAA
>AL669831.3--AL031259.1
CTGAGCACCGATACAAAGAAAGACAAACATC*GAATTAGAGAAAGAACAAATCAGTGTGAAGG
>AL669831.3--AL031259.1
CTTCCTGAGCTGAATGATAGTTAAGCTGAAT*GAATTAGAGAAAGAACAAATCAGTGTGAAGG
>AL929601.1--AC093838.1
TTTCTTCATCCCTTCATCTATGGATGGACAG*GTTCCACAGTGTTGCCACGGATGACCTGATT
>AL929601.2--AC093838.1
TTGTCCGTATACCAGGTGTAAAAATAAAACG*TTTCTTAATGCACACAGGACATGGAAGGCAA
>ANKRD19P--AC067930.1
GATTCACGGGCGCACACACACACACTCAGGA*TTCAGATCTTGTTGAAAGCTGCGATATCGAC
>ANKRD20A21P--AC138776.1
ATGTTGGAATTTCTGATGAATCTGCAGTCAG*CATTTTGCATGAACTGTGTGTGGATTCATTG
>ANKRD36C--LINC00342
CTCAGGAAAATTGTGAAAATAATCAGCAACC*GTGATGGCATCAGCTGAGAAGGGAAGGAAGC
>ANKRD36C--LINC00342
CTCAGGAAAATTGTGAAAATAATCAGCAACC*TTAGGGAAGAAATATAAACACTGAGATTGCC
>ANKRD36C--LINC00342
CTCAGGAAAATTGTGAAAATAATCAGCAACC*TTTCTCCACAGACACTACCCAAAGCAGTCCT
>ANXA4--AC087477.2
ACTCTGATCTTGACCTAGAGTCATGGCCATG*AAGTTCATGGATTCGAATGCCAAAGACACAA
>AP000781.2--AC132217.2
CCTTGCAGGGAGCTGGAGGAAGAGGAGGAGT*GGGGCAGTGACAAAAGCAAGAACGCACGAAG
>AP005242.5--LINC00349
AAACTAGAAACTGGACTGTAGATGATGCTTG*GTCCATTTCTTATACAAGGAAATGCAGTTAC
>AP005901.3--AP005901.5
CCCGCACCCTTGGTTAAGACATTATACCACC*ATTCTTTTCAATATACCCCATCATGGTTCAA
>AP006219.2--AP006219.3
GATAAAGAAGAGCTTGATGATCTGATGTGAG*GTCTGGAGATCAGAAGTCCAAAAAATGGGTC
>AP006219.2--AP006219.3
GATAAAGAAGAGCTTGATGATCTGATGTGAG*GTTTTAAAATTTGCAGAATCTTGGCCAGGTG
>ARF4-AS1--DENND6A-AS1
CCGGAAAGGCGAGCTGAGCATTATGGGTTAG*AAAACTGCCCTTCAATTCTAATCTCACCTCA
>ARL17A--AC005670.3
AGAGAGGAGAAACACACACCCAATCCTAAAC*CACAATGAGGCAGAAAAGTATAATAAAAATC
>ARL17A--AC005670.3
CTGTGGCAGCATTTTTTCCAGAACACAAAAG*CAATGAATCCTCCAATGTACCTGACTCTCCC
>ARL17A--AC005670.3
CTGTGGCAGCATTTTTTCCAGAACACAAAAG*GTTCGTGAGCCACACATCTCCCCACACCAAG
>ARL17A--AC005670.3
GGCTTTCACTATGTTGGCCAAGCTGGTCTCA*AAACTCCTGACTTCAAGTAATCCACCCACCT
>ARL17A--AC005670.3
GTCTCCGTAGTTAAAATTCTTTTTTGTTGTT*TTTTTTTTTATTTGAGACAGTTTGGCTCGTT
>ARL17B--AC005670.3
AGAGAGGAGAAACACACACCCAATCCTAAAC*CACAATGAGGCAGAAAAGTATAATAAAAATC
>ARL17B--AC005670.3
CTGTGGCAGCATTTTTTCCAGAACACAAAAG*CAATGAATCCTCCAATGTACCTGACTCTCCC
>ARL17B--BPTF
TTGTGGCAGAAGCGAGAAAATTTTGTTTATT*AAAAAAAAGAAAAAGAAAGCAAGAAAAAAAG
>ATP5F1A--AC132217.2
TTTTTTTTTTTTTTTTTTTGTTTTTTTTTTT*CCCCTCCTTAGCTTTATGCGCTCAGAAACCA
>AZU1--S100A8
GTCTCTTGTCAGCTGTCTTTCAGAAGACCTG*GTGGGGCAAGTCCGTGGGCATCATGTTGACC
>BDH1--AC132217.2
TCCCTGCTTGCCTCACGTGGTGCTTACACAC*GCGAACACAGCACACACGAACACAGCACACA
>BX004807.1--EFCAB7
CTGCATTCTGCACAGAACAAGTTGGATAATG*AATGGCGATCAGTCCACGAAGCGATGCAACT
>BX004807.1--EFCAB7
TGCATTCTGCACAGAACAAGTTGGATAATGG*GTCACCCAGGCTGGAATGCAGGTGGGATCTT
>BX469938.1--CPXM2
CAGTTGGCACATACCTGGCTCAGCTCTACAG*AGCTTTATCTCTGCGGACCTGGAGCTGCTGG
>BX469938.1--CPXM2
CAGTTGGCACATACCTGGCTCAGCTCTACAG*GTAAACACAGCAACAAAAAAGTTATGAGAAC
>C1GALT1--AC005532.1
ATCCTTGAAAGAAAATCATGAATGAACAAAG*ACAATGCATAGGAGTGAATGAGACAGAAAGC
>C22ORF46--MEI1
GAGAGGATAGTAGAGAGACCCAGGCTCCAAG*TTAGTGCGCAAGAAGCACATGTTGTCCTGCT
>CA2--AC084734.1
ACTGGCGCCCAGCTCAGCCACTGAAGAACAG*GTATAGAAGACTCTCTATGCTGCCCTCTTGA
>CA2--AC084734.1
CAACTGGCGCCCAGCTCAGCCACTGAAGAAC*AGGTATAGAAGACTCTCTATGCTGCCCTCTT
>CALD1--AC132217.2
TTTCTTTTTGTTTAAAAAAAAACAACCAACA*AAAAAAAATCAATTGGCTAAAAAAAAAAAGT
>CBX4--AC132217.2
CCCCCAGCCACAAGAAGCGGGCAGCCGACGA*GCGCCACCATCCTGCAGCCTCCTCCTGACCA
>CCDC152--AC132217.2
CAATCACCTTTCAGTTGCTCCATCATAAAAA*GCACTCATACTTTATGCATCCCCGCAGCTAC
>CCND3--AKR1A1
GACATCGCGCCACTGCACTCCAGCCTGGGCG*GGTAAAGCGAGACTCCATCTCAAAAAAAAAA
>CCND3--AKR1A1
GAGATCACACCACTGCACTCCAGCCTGGGCG*GGTAAAGCGAGACTCCATCTCAAAAAAAAAA
>CCND3--AKR1A1
GAGATCATGCCACTGCACTCCAGCCTGGGCG*GGTAAAGCGAGACTCCATCTCAAAAAAAAAA
>CCND3--AKR1A1
GAGATCCCGCCACTGCACTCCAGCCTGGGCG*GGTAAAGCGAGACTCCATCTCAAAAAAAAAA
>CCND3--AKR1A1
GAGATCGAGCCACTGCACTCCAGCCTGGGCG*GGTAAAGCGAGACTCCATCTCAAAAAAAAAA
>CCND3--AKR1A1
GAGATCGCACCACTGCACTCCAGCCTGGGCG*GGTAAAGCGAGACTCCATCTCAAAAAAAAAA
>CCND3--AKR1A1
GAGATCGCGCCACTGCACTCCAGCCTGGGCG*GGTAAAGCGAGACTCCATCTCAAAAAAAAAA
>CCND3--SIGLEC6
AATCCCAGCACTTTGGGAGGCTGAGGTGGGT*GTTTCACCGTGTTAGCCAGGATGGTCTTGAT
>CCND3--SIGLEC6
GCAGCTGGGACTACAGGCACCCACCACCACA*GGCTGGAGTGCAGTGGTGCGATCTTGGCTCA
>CDC5L--AL136140.1
CAGAATTGAATCACTTGAAAAGAGGCTCGAG*AGATCCATGTAGCCACCACTGAGAACTTCAG
>CHADL--AL035681.1
CATCAAAGGAAGACAGTGTGGAGCAGATAAG*GGGACACTCTGAAGGCACCTGGCTCAGAACC
>CHMP3--SMR3B
GAAAAGCATTATTTTTAAAGGGAAAAAGGGA*GGTCCAGGGATATTTCCACCACCCCCTCCTC
>CHRDL1--NUTM2A-AS1
CCATCCTCCCTCCTCACTCCCTCCCTCCCTC*CTTCTCTCTCTCTCTCTCTCTCTCTCTCACA
>CHRDL1--NUTM2A-AS1
CCTCCCTCCTTCTCTCTCTCTCTCTCTCTCC*CTCTCACACACACACACACACACACACACAC
>CLCN6--AC010327.2
CTCTCTGGTTCCCCTCTCTTGGCCTACGTCT*CCCTGTCCTCGCCCTGCCTCCTGCCATTCCC
>CLCN6--AC010327.2
TGCTCTGTCTCTCCCCTCTGGTTCCTCTCTG*CCCCCTCACTGACCCTCCAAACGCCCCTGTC
>CLCN6--AC119427.2
AAGGAGGAAGCTCACGGTGGTGGTGGAGAAG*GACCTGAATGAGTTGCAGGCGCTGATCGAGG
>CLCN6--AC119427.2
AAGGAGGAAGCTCACGGTGGTGGTGGAGAAG*GAGGAGGCAGCGGAAGAGGATGCTGAAGCAG
>CLCN6--AC119427.2
CAGTAAAAGGAGGAAGCTCACGGTGGTGGTG*GAGAACAGGAGGAAGGCTGAGGATGAGGCCC
>COL17A1--FLG2
TTTTTAGTAGAGACGGGGTTTCACTATGTTG*GCCAAGATGGGTGGATCACGAAGTCAGGAGT
>COLEC12--AC132217.2
AGTGGTCCTCTGTGTCTTGGGGCTTTTTTCC*TTTTTTTTTTTTTTGTTTTTTTTTTACCCCT
>CPA1--BCAR1
TTCCCCAGCATCCAGGCGGTCAAGATCTTTC*CCAGTGGCCTTCCCCAGCATGCACCACGGGC
>CPA1--MPRIP
GACACGGGCATCCATTCCCGGGAGTGGGTCA*CCCAGAGCCCAGTGGCCTGAGATTCACTGGG
>CPOX--AC021660.4
AGAAGCTGTGGCAGCAGCTCAGAAGAGGACG*GGAGTGGAGGAGGAGAAAGTAAAGAAAAGCA
>CPOX--AC021660.4
TCTGGAAGTTCTACGCCATCCAAGGGACTGG*GGAGTGGAGGAGGAGAAAGTAAAGAAAAGCA
>CR392039.6--AC097374.1
GGATTTTTGGCTGTGCGTTCAGAAGCTAAAG*TTCTGGAGGCCACAAGTCCAAAATCAAGGTG
>CSH1--AC132217.2
AAACTCGCACAACCATGACGCACTGCTCAAG*GCACGGCACACACACGCACACACATGCACAC
>CSH1--AC132217.2
AAGACGGCAGCCGCCGGACTGGGCAGATCCT*AGCACACACAGCACACACACAAACACAGCAC
>CSH1--AC132217.2
AGCAAGTTTGACACAAACTCGCACAACCATG*ACGCACACACAGCACACACACGAGCACACAG
>CSH1--AC132217.2
ATCCTCAAGCAGACCTACAGCAAGTTTGACA*ACACGCACACACATGCACACACAGCACACAC
>CSH1--AC132217.2
CAACCATGACGCACTGCTCAAGAACTACGGG*CACACACGAGCACACAGCACACAAACGCACA
>CSH1--AC132217.2
CACAACCATGACGCACTGCTCAAGAACTACG*GCACACACACGCACACACATGCACACACAGC
>CSH1--AC132217.2
CAGATCCTCAAGCAGACCTACAGCAAGTTTG*ACACACACGCACACACATGCACACACAGCAC
>CSH1--AC132217.2
GACGGCAGCCGCCGGACTGGGCAGATCCTCA*AGCACACACACAAACGCACAGCACACACACG
>CSH1--AC132217.2
GCAGATCCTCAAGCAGACCTACAGCAAGTTT*ACACAAACGCACAGCACACACGCACACACAG
>CSH1--AC132217.2
TCAAGCAGACCTACAGCAAGTTTGACACAAA*CACAGCACACACATGCACACAGCACACGCAC
>CSH1--AC132217.2
TTGACACAAACTCGCACAACCATGACGCACT*GCACACACACAAACGCACAGCACACACAAAC
>CST2--AC018630.2
AGAGGTGGGCCGAACCATATGTACCAAGTCC*CCCAACAAGGAGGCCAGCAGCAACAAGGTCC
>CST2--AC018630.2
AGGTGTCAAGAAGCCTAGGGATCTGTGCCAG*GCCAGCAGCAACAAGGTCCACCACCTCCTCA
>CSTL1--AL096677.2
CTGCCCCCTGCAAAGCAAGAAGCTGAGAAAG*AAATTGCCACAGCCAACCCAACCTTCAGCAA
>CSTL1--AL096677.2
CTGCCCCCTGCAAAGCAAGAAGCTGAGAAAG*CTTTCATGTCTCATGAAAACTGTTCTCCTGG
>CSTL1--AL096677.2
CTGCCCCCTGCAAAGCAAGAAGCTGAGAAAG*GGGCTGCTCCCTCAGCAACACACAGGCTTCT
>CSTL1--AL096677.2
GCCTCAGCCTCCCAAGTAGCTGAGATTACAG*GGGCTGCTCCCTCAGCAACACACAGGCTTCT
>CTRC--PACSIN2
CCAGCCACCATGCCTGGCTAATTTTTTTTTT*TTTTTTGTATTTTTAGTAGAGACGGGGTTTC
>CTRC--PNLIPRP2
TGATGGTGAGTGGAAGCTCTGGTGCAGCATG*ATGCTGCCCCCTTGGACCCTCGGCCTTCTCC
>CTSS--KAT6A
ATTATAGGCGTGCACCACCACGCCCAGCTAA*TTTTTGTATTTTTAGTAGAGACAGGGTTTTG
>CTSS--KAT6A
ATTTAGTTGATTACTTTTCAGTTTTGTTTTG*TTTTTTTGAGACGGAGTCTCACTCTGTCGCC
>CTSS--KAT6A
ATTTAGTTGATTACTTTTCAGTTTTGTTTTG*TTTTTTTGAGACGGAGTCTCACTCTTTGTTG
>CTSS--KAT6A
CAAAAAATTAGCCGAGCACGGTGGTGCATGC*CTGTAGTCCCAGCTACTTGGGAGGCTGAGGC
>CTSS--KAT6A
GCTGGAATTATAGGCGTGCACCACCACGCCC*GGCTAATTTTTGTATTTTTAGTAGAGACGGG
>CTSS--KAT6A
TAGTTGATTACTTTTCAGTTTTGTTTTGTTT*TTTTGAGACGGAGTCTCACTCTGTTGCCAGG
>CTSS--KAT6A
TCAAAACGCCACGTGTAGTGGCTCATGCCTG*TAGTCCCGGCACTTTGGGAGGCCGAGGAGGG
>CTSS--KAT6A
TCAGCCTCCCGAGTAGCTGGAATTATAGGCG*TGCACCACCACGCCCGGCTAATTTTTGTATT
>CXCL12--AL137026.3
TTCAGGAGTACCTGGAGAAAGCTTTAAACAA*TTGCTAGAATATGGCTGTACGGCAATGCTGA
>CYP19A1--AC132217.2
CTTATATATATTAAAAAAAAAACAACCACAA*AAAAAAAATCAATTGGCTAAAAAAAAAAAGT
>CYP4F26P--AL390726.4
GAAACATCCTAACCACCTGACTGCAGAAACA*TCCTAACCACCTGACTGCAGAAACATCCTTA
>DCN--AC132217.2
TGGAGCATTTACACCTTTGGTGAAGTTGGAA*GGAAAAGTCAGGAGGATTGACAGCAGACTTG
>DDX60--AC068989.1
AGGATTTTGCACTCACCATTAAATCTATCAG*CTAATTACCAAGAGCACTTTAAGCACACTGG
>DPY19L2--HULC
CAGCTAACCCTCCCTTATGTAGCGTCCTGCT*TGAGCCGAGATCGTGCCACTGCACTCCAGCC
>DPY19L2--HULC
CAGCTAACCCTCCCTTATGTAGCGTCCTGCT*TGAGCCGAGATTGTGCCACAGCACCCCAGCC
>DPY19L2--HULC
CAGCTAACCCTCCCTTATGTAGCGTCCTGCT*TGAGCCGAGATTGTGCCACTGCACTTCAGCT
>DPY19L2--HULC
CAGCTAACCCTCCCTTATGTAGCGTCCTGCT*TGAGCTGAGATTGTGCCACTGCACTCCAGCC
>DPY19L2--MAP4K3-DT
AATGGATTGGAAATTTTTCTGATTACTAAAA*GGGACGTGGATGAAGCTGGAAGCCATCATCC
>EHD3--AC132217.2
TGACTACACACACACACACACACACACACAC*GCACACAGCACACGCACACACAGCACACACA
>ELF1--AC107071.1
TCAAAAGACGATAGAGAAAGAATACTTGAAG*GAAGGCATGTGCCAGATGCTGGATTACATCA
>ENTPD3-AS1--EIF1B-AS1
ATGGATTCTCGATTTTATGATGCTGGACTGG*ATTGGGTGGATCACAAGAACTAATCTTTCAT
>F13A1--AC132217.2
GTTATACTTAAAAAAAAAAAACAATTTGCCA*AAAAAAAAAAAGTATTAAAAACGAATTGGCT
>FAM129C--KAT6A
TTTCAGGCACCCGCCACCATGCCTGGTTAAT*TGTGTTTTTAGTAGAGACGGGGTTTCTCCAT
>FAM129C--KAT6A
TTTCAGGCACCCGCCACCATGCCTGGTTAAT*TGTGTTTTTAGTAGAGACGGGGTTTTACCAT
>FAM129C--KAT6A
TTTTCCCCAAGGCTTTCTTTATTTTAATTTT*TTTTTTTTTTTTTTGAGACAGAGTCTTGCTC
>FAM182A--AC233702.4
GATGGAGGGGTTCCTGGAACTACTGTGAGGG*AAATGGTGAGAGAACTTTGGATGTGGAATGT
>FAM182A--FAM182B
CCAACGCAATCCACTTTCCACCACTTTCACA*AAAAAACGTTACTGCACAATTATACTATCCT
>FAM182A--FAM182B
CTCGCAGGTCACAAGGATGTTATGAAATCAG*GTAAGATTGGGGAAGATGCCAGGTGATAGAA
>FAM182A--FAM182B
GATGGAGGGGTTCCTGGAACTACTGTGAGGG*AAATGGTGCGAGAACTTTGGATGTGGAACGT
>FAM182B--FAM182A
CCTCAGGGCTGCCCGGACGGTGTTGGTTGGG*ATATATATATATGGAAAAGAGAAGGTCCAAG
>FAM182B--FAM182A
CCTCAGGGCTGCCCGGACGGTGTTGGTTGGG*GTAAGATTGGGGAAGATGCCAGGTGATAGAA
>FAM182B--FAM182A
CTGGCAGGTCACAAGGATGTTATGAAATCAG*GTAAGATTGGGGAAGATGCCAGGTGATAGAA
>FAM182B--FAM182A
GATGGAGGGGTTCCTGGAACTACTGTGAGGG*AAATGGTGAGAGAACTTTGGATGTGGAACGT
>FAM234A--ARHGDIG
TCCCTGTGGCCTTGGTCTCACCTCTCTTCAG*ACCCAAGCCTGCCCAATGTGCAGGTGACCAG
>FAM234A--ARHGDIG
TCCCTGTGGCCTTGGTCTCACCTCTCTTCAG*TCCTCCTGGCTGACAAGGAGGGTGGGCCGCC
>FAM245B--LINC02675
AAGAGCTGACTCGTGGACATTCTCATTGCGG*GTGCAGCTCAGTTGCCGTGATGCTCCTTGAG
>FAM245B--LINC02675
AGAAGCTGACAAGAACCACAGGCTTCTTCAG*GCTCTAGCCTGATTTTCTGGCTCATAAGGCC
>FAM27C--AL953883.1
TCAGTCCATCCTTGGCTGTTCAAGTGTAGAG*AAATAAAAACATCAGTAGGAGGGGAAGGAGG
>FAM66C--FAM66D
CTCCTGAAAGCTGCTGAGTGTCCTGCAGCAG*GCCATTTCCAATCCAGCGGAAGACCTTCTAG
>FAM66C--FAM66D
GGTTGCTGCTCATATGTCAGAGACATATGCA*TTTAAGAAAGGTAGCAGTTCCAATCCTGGTT
>FAM66C--FAM66D
TGTTGGCAAGTGGAGCAGAGTCCCAGGACAG*GTACAGAAACACCCCCAGTTCAAGGATCTTC
>FAM66D--FAM66C
CTACTGAAAGCTGCTGAGTGTCCTGCAGCAG*CCTTTCATCCCAAAGATGCAGCCGACTTGAT
>FAM66D--FAM66C
TGTTGGCAAGTGGAGCAGAGTCCCAGAACAG*GTACAGAAACACCCCCAGTTCAAGGATCTTC
>FAM95C--CYP4F26P
CACCCAACATCTCTCAGGTTGCTGCTCAAAG*GTCACCTCTTGGAGGGGTCTCTCCCTGTGAC
>FASTKD2--AC008269.1
TTTGAATGCAATGGGTTTTCATGTGATCTTG*CATATCAGCAATCATGGTGACGTGCCACTCT
>FASTKD2--AC008269.1
TTTGAATGCAATGGGTTTTCATGTGATCTTG*GTCTGGCTGATTTCACTTAACATAATGTCCT
>FASTKD2--AC008269.1
TTTGAATGCAATGGGTTTTCATGTGATCTTG*TTTTCCTCTTCTCCTGTGTTAGAAAGCGTGA
>FGF14--AC132217.2
ACACACACACACACACACACACACACACACA*GAGCACACAGCACACAAACGCACAGCACACG
>FGF14--AC132217.2
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTC*CCCTCCTTAGCTTTATGCGCTCAGAAACCAA
>FGFRL1--AC132217.2
CAGGCAGTCTGTGTGTGAGGCATAGCCCCTG*CACACACGCACACACAGCACACACACGAGCA
>FLT1--AC132217.2
TACTTTAACCTTGAACACAGCTCAAGCAAAC*CACACAAACACAGCACACACATGCACACAGC
>FN1--AC132217.2
ACCGATGTTCGTGCAGCTGTTTACCAACCGC*CCTCACCCCCAAATCTTACATCTCAATTCCC
>FP326651.1--LINC01667
CCTCAGGCCTGCACAGACGGTGTTGGTTGGG*AAGGTCCTTAGCACTCACCAGGAACAGTGGT
>FP326651.1--LINC01667
CCTCAGGCCTGCACAGACGGTGTTGGTTGGG*CATCTCCTGGACAGTCAGCCGAGTGTTTCCA
>FP326651.1--LINC01667
CCTCAGGCCTGCACAGACGGTGTTGGTTGGG*GGTACCTTGGGACAGAGGGAGTGCAGAGCAG
>FP326651.1--LINC01667
CCTCAGGCCTGCACAGACGGTGTTGGTTGGG*TCACTGTCATGGCTAAGAAAGCAACTCCAAC
>G0S2--MARCH7
TGACCACCATTTTGCATTTTGAAATAAAAAA*CTGTTTCTACCAAAAAAAAAAAAAGAGAGAA
>GLTP--AC099521.3
TGTAGTTCCAGCTACTCAGGAGGCTGAGGCA*GGAGAATCGCTTGAACCCCGGGAGGTGGAGG
>HACD3--AC132217.2
AGTGCAAGACCCTGTCTCAAACCAAACCAAA*CACACAGCACACGCACACACAGCACACACAT
>HACD3--AC132217.2
GTGCAAGACCCTGTCTCAAACCAAACCAAAC*CACACACAGCACACACACAAACACAGCACAC
>HERC2--FAN1
AACACCGTTGCCCCGTCTGTTTCTTGATGAG*TCGGGTAAAAGTTCCAGAAATGTGAAGTACA
>HERC2--FAN1
CAAACAGCTGCTCCACAGCCCATTCTTTAAG*GGGTAAAAGTTCCAGAAATGTGAAGTACAAA
>HERC2--FAN1
CAAACAGCTGCTCCACAGCCCATTCTTTAAG*TCGGGTAAAAGTTCCAGAAATGTGAAGTACA
>HERPUD2--AC018647.3
ACTGTGCCAGCTACAAGGAGGGTCTGACTTC*AGATTTTGCATATAGAAGACGGTAATTTTAA
>HERPUD2--AC018647.3
TGTGCCAGCTACAAGGAGGGTCTGACTTCAG*ATTTTGCATATAGAAGACGGTAATTTTAAAA
>HIF1A--S100A8
GTCTCTTGTCAGCTGTCTTTCAGAAGACCTG*GTGGGGCAAGTCCGTGGGCATCATGTTGACC
>HMGA1P4--WDR34
AGAGAGGTGGACAGATCTGAGAGATACTGAG*AAAAGTTGCCAGACGGCCAGCATTGCCACTG
>HTN3--AC018630.2
TCATACCGCATCACACTACCACTGCTTTTTG*AAGAATTATAAGGCCATGAAGAAAAATTAAA
>HTRA1--DES
TCTCCGCTCACTGCAACCTCCACCTCCAGGA*CCTGCTCAACGTGAAGATGGCCCTGGATGTG
>HULC--DPY19L2
CCAGCCCCCAGCCCGAAGATACAGATTTGAG*GCCTGGTTGCAGTATGCTTGAAATCTGGGAT
>IGF2--AC006115.2
AGACGAATTCTCCCCCCCCCCCCAAAAAAAA*CCACCCCCCCCCCCCCCCGCCAAAAAATGCT
>IGK@--AC093787.2
ACATACAATGGTAAATCATTCAGCCTTAAAC*GAGGGAAATATTCTGACATATGTTGCAACTT
>IGK@--AC093787.2
TCCTTCCAATTTAGTTCATTATTTCCTGCAC*CTTTGATGTCATATACATAAACAGGAAATAA
>IGK@--FLG2
GGGCGCGGTGGCTCACGCCAGTAATCCCAGC*GCCAACATAAACTAGCCCAAGATAAGAACTA
>INS--AC132217.2
AAGACCCGCCGGGAGGCAGAGGACCTGCAGG*CCTCAGCTTTGTCCCTCTCCTCCTCCACGTC
>KAT6A--CTSS
ACCTCAGCCTCCCAAAGTGCTGGGATTACAG*GCGTGAGCCACTGAGCCCAGCCTACTTTTCA
>KAT6A--CTSS
ATGGTGAAACCCCATCTCTACTAAAAATACA*AAAAAATTAGCCGAGCACGGTGGTGCATGCC
>KAT6A--CTSS
CCGGCTAATTTTTGTATTTTTAGTAGAGACG*GGGTTTTGCCACATTAGCCAGGCTGGTCTCA
>KIAA0895L--AC132217.2
GGCACCAATTTTCCCAGCCCCACTCCCATTA*CACACACGCACACACAGCACACACACGAGCA
>KLF4--LINC01509
GACACTGCGTCAAGCAGGTGCCCCGAATAAC*GTTTCCTGCAACCTCACTGGGCATTTTCAGA
>KMT5A--AC068790.8
AGGCGGTGACAGAGTGGAGCCATGGCTAGAG*GGCGCCCTGCGTCCGGCAGAGGAGGCGAGCA
>KRT128P--KRT2
GGCTGAGGCCGAGGCCCTGTACCAGACCAAG*TATGAGGAGCTCCAGGTGACTGTCGGGAGAC
>KRT2--KRT128P
TCTGGAGACCTCAGCAGCAATGTGACTGTGT*GATGTCGACAGCACCTGCTTGGTCCAGGTTG
>LAMC2--AL354953.1
AACACATTAGACGGCCTCCTGCATCTGATGG*ACCCTGCACTTGATGGACCAGCTGGCACCAC
>LAMC2--AL354953.1
AACACATTAGACGGCCTCCTGCATCTGATGG*GAGCCCATGGAGGAGAGGCACCAACAGGTAT
>LAMC2--AL354953.1
CAACACATTAGACGGCCTCCTGCATCTGATG*GAGCCCATGGAGGAGAGGCACCAACAGGTAT
>LHFPL5--CTRB2
CAGGAGGATCAGGATCTTCTCCATGGTGAGT*GGACCTCCGACGTGGTCGTGGCTGGGGAGTT
>LHFPL5--CTRB2
CATGGTGAGTGGGACAGCTGGTGTGGGTGGC*GGGAGTTTGACCAGGGCTCTGACGAGGAGAA
>LHFPL6--AC132217.2
ACACACACACACACACACACACACACACACA*CACACAGCACACACATGCACACACAGCACAC
>LHFPL6--AC132217.2
ACACACACACACACACACACACACACACACA*GCACACACATGCACACACAGCACACACACTA
>LHFPL6--AC132217.2
CACACACACACACACACACACACACACACAC*ACACACAGCACACACATGCACACACAGCACA
>LHFPL6--AC132217.2
CACACACACACACACACACACACACACACAC*ACACAGCACACACATGCACACACAGCACACA
>LINC00271--PDE7B
GAGCCCACACCCTTCATCACTACTGATAAAG*AGGTGTGGCGAAATCTTGTTTGAGAACCCCG
>LINC00354--SOX1-OT
CACAAGCCCGCCCTGCAGAGATCTTCGAAAG*ATACCTCAGTATACACTGCCCACAGTTAAGA
>LINC00354--SOX1-OT
CACAAGCCCGCCCTGCAGAGATCTTCGAAAG*ATGGCTTGGCATCTTCTTCCGAGCAAGCGCA
>LINC00402--AL353660.1
AAGAGCCTCTTTCCTTCATACCATCTTCAGA*GGACTTCAGCCATGAACTTGGTGATGAGCCA
>LINC00402--AL353660.1
AAGAGCCTCTTTCCTTCATACCATCTTCAGA*GTATAACTCCTACTTGAAGAGTGTGAGAGAA
>LINC00662--CYP3A5
AAGATGGATGATTTCTGCATTTCCAACTGAG*GCACCACCTACCTATGATGCCGTGGTACAGA
>LINC00680--AC091978.1
ACAGCTGTGGAAAGAAGTCACAAAACACTGG*TTCAACCCAATGGAAATAGATTAAAAGAGTG
>LINC00680--AC091978.1
ACTTAGGGACAAGACTCCTTAGTGGTGGAAC*CTCAGGCATCCGTGCAGGATGTAAGAAAGCT
>LINC00840--LINC00841
GATGCAGACAAGGTCCACGCATTGTGGTTGG*AATGTGGCAGAATTGGCCGGAAGCAGAATTG
>LINC01005--AC073270.1
ATTTTGATATATCACTGGGCCCAACAGCCAG*GTTGTATTGTGCCATATACCTGGGACCAAAT
>LINC01005--AC073270.1
TGTGAGTTCCCTATCAGAACACAATCTGCAG*GTTGTATTGTGCCATATACCTGGGACCAAAT
>LINC01388--AL449043.1
ACCACGAGAAAAACAGTGGAGTCAAACGGCG*GCGCTGTTGTAGGAGCAAGGTATACAATGAT
>LINC01484--GIPC1
GGAAGATGACACAGCTGTGAGACATTTTGGG*GTGATGTTCTGCACCCTGAACACCCACAAAG
>LINC01522--LINC00494
CAATGACCCTCAATAACACAGCCTCCCACAG*ATAAGAACCAAACGACATCCGGCCTTTGCGT
>LINC01522--LINC00494
CAATGACCCTCAATAACACAGCCTCCCACAG*CTCCCGGCTCACCCACCCAGCAGCTCAGCAC
>LINC01524--AL031674.1
ATGACAATAAACAAGATAAATTAGATCTATG*CAACAAAACCAAGCCTTTGTGCACATCTGTG
>LINC01597--AL591926.2
ACCCCTGGGCCTGCTCCACAAAGTTTGAAGG*GAGCTCAGAGATGCCCGGGAAGACGGGGGGC
>LINC02018--AC145124.1
AGAACCAAATGAGCGCAAGATCTGACAAAGA*AAAAAAAAAGGTTCATCTTTTATTCCTCCAA
>LINC02050--LINC02027
CGCTTTCCTTCTCTCATCCCGAGTCAGTACT*GTGACTGTAACACAGATTGATGAAAGATTCT
>LINC02088--LINC02087
CGTAAGGAAGGATGTCCCAGGTGCAGGCGAA*GTTTTGTTCATGTGTTCCTGGAGGATTTTTC
>LINC02186--LINC02185
TTGCTAAGCAAATGAAAGACTACACGACAAG*GCCGCAACCTGCCATGCTGTTTTTAAATGCC
>LINC02245--CTSB
CGGGCATGATGGCTCAAGCCTGTAATCCCAG*AGTGTTGGGATTACAGGCGTGAGCCACTGGG
>LINC02245--CTSB
GCCTCGGCCTCCCAAGGTGCTGGGATTACAG*GCGTGAGCCACTGGGCCCTGCCTGTATTTTT
>LINC02558--SLC5A1
AAAAGAAATAAAAGCTATCCAATTCAGAAAG*GCTATGTTTTCCACCAATCGTGGGACTGTTG
>LINC02558--SLC5A1
CACCAGGGTCCTTGGTCTTGCGCCAATGTAG*GCTATGTTTTCCACCAATCGTGGGACTGTTG
>LINC02635--AC079804.3
ACAGCCATGTAAAACCTGCAGCGCCTGAAAC*CTGGGAGGCAAGACCAAGCCCAGACAACCCT
>LINC02635--AC079804.3
GTAGATTCCACAGAGATGCCAAAGTGTGGAG*ATGATCGAGTGTGGATCAAGGACTGGAAAGT
>LINC02635--AC127526.5
ACAGCCATGTAAAACCTGCAGCGCCTGAAAC*CTGGGAGGCAAGACCAAGCCCAGACAACCCT
>LINC02641--AL357127.2
TACTTTTGATGTCCACCTAGTCAAAGTTCAG*TCTTTGTAATAGTTAATTTTATGCATCAACT
>LINC02675--FAM245B
AAGAGCTGACTCGTGGACATTCTCATTGTGG*GCGCAGGTCAGTTGCCGTGATGCTCCTTGAG
>LINCADL--LVRN
GCTGTGATGAATGCTGCAGAGGCCAAATAAG*GGCCCTGTTAGCGTCCCAGCTGGAACCAACA
>LINP1--AL392086.1
TCCAGAAGCTGGCCCAGCCGGTCCAGTACAC*AATGATACTGAAAGCATCTCAAAAATAAAAT
>LINP1--AL392086.1
TCCAGAAGCTGGCCCAGCCGGTCCAGTACAC*GTACATGAGGGAAATACACACCCCTCCCAGG
>LYZ--PSMA3-AS1
GGTTATCAAATACATCTCCAGTACATTCCGT*ACTTTTTTTTTTTTTTTTTTTTTTTTAGGTA
>MED31--AC004706.3
CAATGTTTAGCCAACCCAAATTACCTTAATT*GAATGGTTTTAGGGAAGACTGTTGCCGGATG
>MED31--AC004706.3
GGAAAGACCCAGAATATGCCAAGTATCTAAA*GAATGGTTTTAGGGAAGACTGTTGCCGGATG
>MEG3--AC132217.2
ACACACACACACACACCCCCTCGTGTATCTA*CATGCGCAGCACATACATGAACACAGCTCAC
>MEG3--AC132217.2
AGATTAAAACACACACACACACACACACACA*CACACACACGCACACACATGCACACACAGCA
>MEG3--AC132217.2
AGTTTCTTTCATGGTCTAAATGGAGATTAAA*ACACACGCACACACATGCACACACAGCACAC
>MEG3--AC132217.2
CAGTTTCTTTCATGGTCTAAATGGAGATTAA*AGCACACACAGCACACACACAAACACAGCAC
>MEG3--AC132217.2
TTCTTTCATGGTCTAAATGGAGATTAAAACA*CACACACACGAGCACAGCACACACACAAACG
>METTL7A--PLCG1-AS1
CGCTCTTGTTGCCCAGGCTGGAGTGCAATGC*TGTGATCTCGGCTTACTGCAGCCTCCGCCTC
>METTL7A--PLCG1-AS1
GCAAAACCCTACTAAAGATACAAAAAAAAAA*AAAAAATTAGCCAGGCGTGGTGGTGGGTGCC
>MPEG1--AC132217.2
ATATGAACACACACACACACACACACACACA*CACACACACGCACACACATGCACACACAGCA
>MPO--ALAS2
TCATTCGTTCGTCCTCAGTGCAGGGCAACAG*GACTTTAGGTTCAAGATGGTGACTGCAGCCA
>MRNIP--AC132217.2
ACCCAGGAAACATCAGCACACACACACACAG*ACACACAGCACACACGCATGCACAGCACACA
>MRNIP--AC132217.2
CACCCAGGAAACATCAGCACACACACACACA*AGCACACAGCACACACACAAACGCACAGCAC
>MRNIP--AC132217.2
CACCCAGGAAACATCAGCACACACACACACA*GCACACACATGCACACACAGCACACACACTA
>MRNIP--AC132217.2
CCAGGAAACATCAGCACACACACACACAGGG*ACGCACAGCACACACACGCACACACATGCAC
>MRNIP--AC132217.2
CCCAGGAAACATCAGCACACACACACACAGG*ACAGCACACACATGCACACACAGCACACACA
>MRNIP--AC132217.2
CTGCAAGGAGCCAGGGCACCCAGGAAACATC*AGCACACACACAAACGCACAGCACACACACG
>MRNIP--AC132217.2
GCTGCAAGGAGCCAGGGCACCCAGGAAACAT*CAGCACACACACAAAGACACAGCACACACAT
>MUC20-OT1--AC124944.2
AGATTCCAAAGGCACAGGAAGAAGCTTGCAG*GCTCATCAATGCACCGTGTGGCCAAACAGAA
>MUC7--ARHGEF12
CTATCAATTCAATTAATTCACATAAATTCCA*TGTGAACATACATATACATTTCTGCCTAGTT
>MYH7--AC119427.2
CCGCAACGAGGCCCTGAGGGTGAAGAAGAAG*ATTCTGGCTGAGAGGAGGAAGGTGCTGGCCA
>MYH7--AC119427.2
GATCGACAACCTGCAGCGGGTGAAGCAGAAG*CGGGAAAAGAAGAAGAAGATTCTGGCTGAGA
>MYH7--AC119427.2
GTACGAGGAGGAGACGGAGGCCAAGGCCGAG*CAGCAGCGCATCCGGAATGAGCGGGAGAAGG
>NAP1L1--AC132217.2
GTTAAAGGAAATACCTTTTTTTTTTTTTTGT*TTTTTTTTTTACCCCTCCTTAGCTTTATGCG
>NDUFS8--PNLIP
CCAGCGTGTGGAACCTGACGGAACTGCCACG*ATGCTGCCACTTTGGACTCTTTCACTGCTGC
>NLK--AC061975.8
CCACTGGAGTCTGGGATTTGCAATTCTGGAG*GGGAAACTCATCCAGTAAAGTTCAGCCCTTT
>NMUR1--AC017104.6
AGCCGGGGTCTCGCGGGCCGCGGGCCGCATG*ACTTCCAGGTTCTAGACGGCTTTCTCCCAAT
>NMUR1--AC017104.6
CGGGGCCGGAGACAAGTGACCAAGATGCTGT*ACTTCCAGGTTCTAGACGGCTTTCTCCCAAT
>NPLOC4--AC004921.2
ACCTAATAGTTTTTCAGCCCGCCCTGCTCTT*TGTGTGTGTGTGTGTGTGTGTGTGTGTGTGT
>NUTM2A-AS1--CHRDL1
GCAACTTCTCTCTCTCTCTCTCTCTCTCTCT*CACACACACACACACACACACACACACACAC
>NUTM2A-AS1--SYT15
CTTTTTGTCACTTATTTCACTTGCACATTTT*TTTTTTGTTTGTTTGTTTGTTTGTTTGTTTG
>NUTM2A-AS1--SYT15
GCCTCAGCCTCCAAAGTAGCTGAGACTACAG*CAGTTTAAGTATAGTGTGCCTGGGTTGGGTG
>NUTM2A-AS1--SYT15
TCTATGGAAAATCCTAAGGAATCCACACACA*AAAAAAACTACTAGAGCTAATAAGTTTGGCA
>NUTM2A-AS1--SYT15
TTGCTTTTTGTCACTTATTTCACTTGCACAT*TTTTTTGTTTGTTTGTTTGTTTGTTTGTTTG
>NUTM2A-AS1--SYT15
TTTTCCTTCTTTTTCTTTTTTTTTTTTTTTT*TTTTAATTTTTTGAGACGGAGTTTCCCTCTT
>ODR4--AL096803.2
GGTATCTCCTTTCATTACTTCAGTGATTAGG*GAAACCTTAAAGTCTCATCTCAATATGACAT
>PCBP2--AC132217.2
GAACAGAAAAGTGCAGTCTAAGTGGTTTTCT*GTCCTGCCCCCCGCCATTTATCGCCCTGATT
>PDE4DIP--LINC01719
ATGTGTTTAACTTCAAACCCTGGCAGATTAG*GTTTTTCATCCAGAGCCTGTGGCTTCACCAT
>PKIB--SIGLEC6
GTGTTAGCCAGGATGGTCTTGATCTCCTGAC*CTCGTGATCTGCCCTCCTCGGCCTCCCAAAG
>PLAT--CD22
AGCCTGGGCGACAGAGCAAGACTCTGTCTCA*AAAAAAAAAGAAAGAACTCCGTGAGCATCCC
>PLAT--SPTLC2
AAAAAAATTAGCTGGGCGTGGTGGCGGGCAC*CTATAATCCCACCTACTTGGTAGGCTGAGGC
>PLBD1-AS1--AC010168.1
CCAGGCTAGACTATGCGCTGTGGCATATTAG*GTTGAGTTGCTGAAGCGGCTTCATTGTCTGG
>PLPP4--AC023282.1
CGCATGTGCGACTACAAGCATCACTGGCAAG*CTCAACCTACTGCCTGCCCAGGGCCATCAAT
>PODN--AC132217.2
CACCCCTTCCCACGGCGTGTCCCACGGCCAG*CACACGCACACACATGCACACACAGCACACA
>PRB2--SUN2
CAAGGACCACCCCCACAAGGAGACAAGTCCC*GAAAGCCCCACTCTCCTCCAGTGGTCACAGT
>PRSS3--KLK1
CATCATGCTGATCAAACTCTCCTCACCTGCC*CCCCTGGACACCTCTGTCACCATGTGGTTCC
>RALY-AS1--STAG3
GGCAACATAGTGAGACCCTGTCTCTACAAAA*AAAAAAAAAAAAAGACAAATGGAGAAGGATG
>RMI1--AL390838.1
ATTCCAATTCTTCATAGTGATCTTCCTCCAG*GCACTCCTGATACCCACTGGAGGTAATGCTG
>RRM1--AC015689.1
AAATAAGGAGAAGCTAAAAGATAAAGAAAAG*GATGACACAGCAAGAAGGCCCTCACCAGATG
>RRM1--AC015689.1
AGTAGCCAAACTACTTCTTGAGCATAGATAG*GATGACACAGCAAGAAGGCCCTCACCAGATG
>SAMD12-AS1--COLEC10
ACTGCCCTCTGGTAGCACCTGAGGATCATGG*GAGATGATGGTGAAAAAGGAGATCCAGGAGA
>SART3--LINC01498
GCCCCATGCTTTTGCCGCAGACATACGGAGC*GAATCTGTGGCCTGGGAACTCTTCAAGAAAT
>SCAND2P--AL445524.2
AATGGGATAAAGAAAAATAGACAACCAGTTC*CATTTTAATAAGGAAACAGAAACAACTTTTT
>SCAND2P--AL445524.2
CACTAATTTGAACTCTTAAAGATTGCTGCTT*TTTTTTTTTGACATTGTCAATAACGAAACCT
>SCAND2P--AL445524.2
CCTTAAGTTGATTGTACTTCCAAATTTGCTG*TTATGTTTTTTTCCTAATACTGTGATCTATC
>SCAND2P--AL445524.2
CTCTGGATATGAAATAAGTGCCCTGTGTAGA*TTTTTTTCATTCTTATATTTTGCCAGATCTG
>SCAND2P--AL445524.2
CTGGATATGAAATAAGTGCCCTGTGTAGAAT*TTTTTTTCATTCTTATATTTTGCCAGATCTG
>SCAND2P--AL445524.2
CTTTGAGTCAAGTTTTGTTTTGGTTCTTTTG*TTTTTTTTGAGACAGAGTCTCACTCTGTTGC
>SDHA--AC073210.3
AGGAAGAGATTCTGGGAGGATCCCGAGGACC*CCCCCAGTTGCAGCCATGTCAAGACTGATGC
>SEPT9--AC132217.2
TGTGCACGTGTGTGTGCATGTGTGTGCGTGT*GAGCACACAGCACACACACAAACGCACAGCA
>SERINC5--AC012636.1
ATGTGATGATGACCGTCACCAACTGGTTCAA*CGATACTTGAGGTTGGACCAGATGATGTTTT
>SLC39A3--AC006538.2
CCGTCCTGGCCGGGATGGTCTTCCTCAAGTG*ACACACTGTCACCCAATGCTAGGAGCCACTG
>SMR3B--SFRP1
CCCGACCAAGACCCTATCCACCTGGACCTCC*TTCCCGAGAAAGACCCAGGCAGGCCCTTGTT
>SOX9-AS1--AC020656.1
ACGGTGAAACCCTGTCTCTACTAAAAATACA*AAAAAATTAGCCGGGCGTGGTGGCGGGCGCC
>SRP14-AS1--AC021755.3
CCGGCTACCTGTTGGGGAAGAAGAACTTAAG*TGGCTTTTCAAGGTACACGCTGAAGACAGCA
>SSPN--AC055720.1
GCTTTGGGGAGCCCAAGCCAGGAAGCTGGAG*AAGTCCTCATCTCCTGAAGCTGCAAGAAGAG
>SSPN--AC055720.1
GCTTTGGGGAGCCCAAGCCAGGAAGCTGGAG*GTCACCCAGCCAAAATAATCTACTCTGCCTG
>ST6GALNAC1--AC015802.7
GCTGCTCACTGCCCTTCAGCTCTGTGACCAG*ATGGTTTGTACAATAAAGGGGAAGATTAAAG
>STAG3--RALY-AS1
TTGGGAGGCTGAGGTGGGAGGATCCTTTGAG*CCCAGGAGTTTGAGACCAGCTTGGGCAACAT
>STATH--FAM174B
TCTAAAAGCAATAAATTTCAAGCACATTTTT*AAAAAAAAACAAAAAAACAAAACCCCAACAA
>STXBP5-AS1--GINM1
GCCTGGGTGACAGAGCGAGACTCTGTCTCAA*AAAAAAAAAAAAGTTGACCTTATTCTCTAAA
>SUGT1P4-STRA6LP--AL158066.2
CGGCTGCAACAGGACCTGCAGCATCCCAGAG*GAGCTGACTAAGGCTTTGGAACAGAAACCAG
>SULT2A1--LINC01595
CCTTGTACATGTACCTGACTGGGGTCATTGT*GCAGGAAAACAAGAGAAATTCTTGCTCTTGC
>TESC--ZG16B
CCCGCAGGAGGCCTGGAAGGTGTGTGGCCCT*GGAGGCAAGTATTTCAGCACCACTGAAGACT
>TGFB1--PTMA
CATTTGGAGCCTGGACACGCAGTACAGCAAG*GACTTAAAGGAGAAGAAGGAAGTTGTGGAAG
>TIMP2--AC132217.2
CACACACACCCCTGCACACACATGCACACAC*AGCACACACAGCACACAGATGAGCACACAGC
>TIMP3--AC132217.2
TTGGATAAATACACACACCATACACTATCCA*CAATTATCCCCACACATAAAAAATCAAAACA
>TMEM140--AC083862.3
AGTGTTCCTGTGGTCTCTGCACCTCCTTTCT*CTCCGTGTCTGAGAGGTTTTGTCTGTGGCTT
>TMEM144--AC098679.5
AGCAAATGGATCTCAGCCACTGTTGGAGTGG*AGGTGTTCGCCGTGACCGTTACAATGAAGAG
>TMEM144--AC098679.5
AGCAAATGGATCTCAGCCACTGTTGGAGTGG*GACAACATCTGAATTTGCAAATTTAACTTAG
>TMEM144--AC098679.5
AGCAAATGGATCTCAGCCACTGTTGGAGTGG*GTTTTATTTAACTCCTTAGCTAATGAGGAAA
>TMEM184B--PLA2G6
GTCTATGCTGACAAGAGGCTGGACGCACAAG*CTCCTCCTTCTGGCACACGCGCTTCACGCGG
>TMEM64--AC004083.1
TACAACAAGAGGACCCTAACATTTTCTGGAG*GAAGAAACTATAAAGTTCATGGTACTCAAGG
>TMPO--FOXP1
CTTTGGGAGGCCAGGGCGGGTGGATCACGAG*GTCAAGAGATCAAGGCCATCCTGGCCAACAT
>TNNT1--AC119427.2
GAAGAGGAGGCTGCGGAGGAGGAGGAGGAAG*AAGCAGCTGTTGAAGAGCAGGAGGAGGCAGC
>TNNT1--AC119427.2
GATCCCAGAAGGGGAGCGCGTTGACTTCGAT*GACATCCACCGGAAGCGCATGGAGAAGGACC
>TNNT1--AC119427.2
GTGCTGCACTGGGAGGGGGAGCAGCCGGAAG*AAGCAGCTGTTGAAGAGCAGGAGGAGGCAGC
>TRB@--AL356489.3
ACTCATTTAATTGCCACAACAAATATGTAAG*TACTTTTATTATTTCCATTGTTATGGACTGA
>TRB@--PAM
CAGTGGGTGGTGTCAGCAGGTCACTGCTACA*ACTGTGAGGAGAACTCTGCTCAGTAATCGGA
>TRB@--RAPGEF1
CCCCTCCAGGACTTCGATGTTGTGCTCTCCC*GCATCCAGGTGAGGCAGGGGCAGGCATGGGA
>TRIM7--LINC01962
ACCCGAACTTTTTTTGTTTTTGCTTCTGGAA*TTCCTTCTGCGCGCTCCCCGGGAGCGTCTTT
>TRIOBP--AC132217.2
GTGTGTGTGTGTGTGTGTGTGTGTGCTGGGT*ACTCATACTTTATGCATCCCCGCAGCTACAC
>TRIOBP--AC132217.2
TACACACGGATACACACACACACACACACAC*GAGCACACAGCACACACACAAACGCACAGCA
>TRIOBP--AC132217.2
TGTGTGTGTGTGTGTGTGTGTGTGCTGGGTG*CTCACCCCCAAATCTTACATCTCAATTCCCA
>TSPAN15--AC016821.1
CATGTGTGGCTACAAAACTATCGACAAGGAG*GGACATGGATGAAGCTGGAAGCCATCATTCT
>TSPAN15--AC016821.1
GGGCATCCTCCTGGGCATCCTGCTTCCCCAG*GGACATGGATGAAGCTGGAAGCCATCATTCT
>TSPAN15--AC016821.1
GGTGCCCTACACCTGCTGCATCAGGAACACG*GGACATGGATGAAGCTGGAAGCCATCATTCT
>UBE2G2--AC020656.1
ACAGTGAAACCCTGTCTCCACTAAAAATACA*AAAAAATTAGCCGGGCGTGGTGGCGGGCGCC
>WAC-AS1--Z99572.1
AGTGAGCTGGGTGTGGTGGCACGTGCCTGTG*GTCCCAGCTACTCGAGAGGCTGAGGCAGGAG
>Z95115.2--CRYBA4
GAAGGACCATGAGGCAGACAGGAGACATATG*GAAGGGGCCACAATGACCCTGCAATGCACAA
>ZCWPW1--AC005071.1
AGAGTTGGAAAAGGAGGAAGGAGAGAAAACA*GAACCCAGAAATAGTGGCTGGAATCCTTGCA
>ZNF250--AF235103.3
GACGTCAAGAATGAAGCTGCGGACCCTCGTG*GCAGCTCCAGCAAGTATCAATCCCATTGCTG
>ZNF736--AC073270.2
AAGAGACAGGAGGCAGTAGCCAAACACCCAG*GGTCCTGCTCACAGAAAGATTGTGACATCTC
>ZNF736--AC073270.2
AAGAGACAGGAGGCAGTAGCCAAACACCCAG*GTAATATTGTGCCATATACCAGAGACCAGAT
>ZNF736--AC073270.2
GGACGGCGGAACATCTGGAGGCTGGGAAATG*GTAATATTGTGCCATATACCAGAGACCAGAT
>AC005224.2--KCNK6
GAAAGAAAGAAAGAAAGAAAGAAAGAAAGAA*AGAAAGAAAGAAAGAAAGAAAGGGAAAGATG
>AC005670.3--ARL17A
AGTTCTCCGTAGTTAAAATTCTTTTTTGTTG*TTTTTTTTATTTGAGACAGTTTGGCTCATTC
>AC005670.3--ARL17A
GTTCTCCGTAGTTAAAATTCTTTTTTGTTGT*TTTTTTTTTTATTTGAGACAGTTTGGCTCAT
>AC005670.3--ARL17A
TCTAATAAGTCTACAAGACAATACTTCTGTC*TGTCTTTTGGCTCTCTCCTTCCTCTCCAGGG
>AC005670.3--ARL17A
TTCAGGGGCAAAATCTAGAGAATCTGAAACG*AGCTATCAGCTTCCCAGTTTGCACAATTCAT
>AC007342.4--LINC01949
GTGAAGATTTCATGGACATTTATCACTTCCC*CAATCAATACTCTTATAATTTCCTATGCCTG
>AC007342.4--LINC01949
TCACTTCCCCAATCAATACTCTTATAATTTC*CTATGCCTGTCTTTACTTTAATCTCTTAATC
>AC007384.1--KMT2E
GGAAGCGGTCTACTACCAAAGTACAATGAAG*GTCTATTCTGGATAAGCAGGTTTTGTGAAGA
>AC009093.1--CSF2RA
GCCAGGAGTGGTGGCTCACGCCTGTCATCCA*AACACTTTGGGAGGCCGAGGCGGGTGGATCA
>AC009093.1--CSF2RA
GCTCACGCCTGTCATCCAAACACTTTGGGAG*GCCGAGGCGGGTGGATCACGAGGTCAGGAGT
>AC009093.1--CSF2RA
GTGGTGGCTCACGCCTGTCATCCAAACACTT*TGGGAGGCCGAGGCGGGTGGATCACGAGGTC
>AC010329.1--AC127526.5
ATGTGTGTTATAACCCATCTGAGCCTCCTGC*ACCACCGTTTTTGAAATAAGAATAAGAACTG
>AC011346.1--TRA@
GTTTTCCATTTGCTTGGTAGATCTTCCTCCA*CCCTTTATTTTGAGCCTATGTGTGTCTCTGC
>AC011416.4--AC011416.3
GCTGTTATGTGTAATAAACTCCATCACAGGG*CCCTTTTGCACATGACGCGCCTCCCCGGGTC
>AC011416.4--AC011416.3
TTGCTGTTATGTGTAATAAACTCCATCACAG*CCTTTTGCACATGACGCGCCTCCCCGGGTCT
>AC022400.6--TIMM23B
AGGAAGCTACACAAAAGGCAGCAAAGTATTA*TAACTGGTATGAACCCTCTGTGTCCTTATTT
>AC022400.6--TIMM23B
CTCAGAATGAAGGTAATTTTTTTTCTTTTTT*TTTTTTTTTGAGATGGAGTCTTGCTCTGTTG
>AC023824.6--AC105916.1
CTGAAATATGGCCTCGTGGGATGAGAAAGAC*CTGACGGTCCCCCAGCCTGACACCCGTAAAA
>AC079804.3--AC127526.5
ATGTGTGTTATAACCCATCTGAGCCCCCTGC*ACCACCGTTTTTGAAATAAGAATAAGAACTG
>AC079804.3--AC127526.5
ATGTGTGTTATAACCCATCTGAGCCCCCTGC*CACCGTTTTTGAAATAAGAATAAGAACTGGC
>AC079804.3--AC127526.5
GATGTGTGTTATAACCCATCTGAGCCCCCTG*ACCACCGTTTTTGAAATAAGAATAAGAACTG
>AC079804.3--AC127526.5
TGATGTGTGTTATAACCCATCTGAGCCCCCT*ACCACCGTTTTTGAAATAAGAATAAGAACTG
>AC092813.2--NEAT1
AAAGATAGGATGTGTGTGTGTGTGTGTGTGT*GTGTGTGTGTGTGTGTGTGTGTATGATTTTC
>AC092813.2--NEAT1
TGTGTGTGTGTGTTTATGTGTGTGTGTGTGT*TTTTTTTTTTTTTAATTTCAATGGCTTTTGG
>AC092821.3--AC024940.2
TTATGTACCAACATAGAAGCTTGAATATAAG*GTTCCTCAGGCCAGATCTGACCCACTGGCAT
>AC092902.6--AC108519.1
AAGACATAAGAGACTCCATTTTGAAAAAGAC*CTGTACTTTAAACAATTGCTTTGCTGAGATG
>AC106791.3--KLHL3
AGCTCAAGCCTTCCACTGATTCTACGTTACG*TGTCAAGCTGAGCTCCCAGACTCTGATACAG
>AC107204.1--DA750114
TCCAACTATGTGGTCAATTTTGGAATAGGTG*CAGTGTGGTGCTGAGAAGAATGTATATTCTG
>AC108063.2--PROM1
AGGCTGGGCGGCCAGTGATGCTGTGAGTCAG*ATACTTTGAGAAAATTCTTACAGAAGGCATA
>AC108063.2--PROM1
AGGCTGGGCGGCCAGTGATGCTGTGAGTCAG*GGATGGTGCCATGAGTGAAACAAGCTGCTCA
>AC108063.2--PROM1
AGGCTGGGCGGCCAGTGATGCTGTGAGTCAG*GGATGGTGCCTTGAGTGAATGACCCCCTTGG
>AC108063.2--PROM1
GGCTCAGTTTAGCCTCAGCTGGTGACAGCTG*GGATGGTGCCTTGAGTGAATGACCCCCTTGG
>AC108063.2--PROM1
GTGGAAGCAGGGAGAGCAAGCATGTCACAAG*GGATGGTGCCTTGAGTGAATGACCCCCTTGG
>AC108519.1--AC105916.1
ACACTATCAGGGGACCTGCCCTGATAATCAG*TCTACAGGTGTATCCAGCAGCTCCAGAGAGA
>AC108519.1--AC105916.1
ACTTCATTATGACATAGATTCTATTGCTCAC*TTTTGTTGCTGACCTTCTCCTTATTATCACC
>AC108519.1--AC105916.1
AGATTCTATTGCTCACATGTTTGTTGCTGAC*CTTCTCCTTATTATCACCCTGCCCTCCTACT
>AC108519.1--AC105916.1
CTGAAATATGGCCTCGTGGGATGAGAAAGAC*CTGACGGTCCCCCAGCCTGACACCCGTAAAA
>AC108519.1--AF228730.5
ACACTATCAGGGGACCTGCCCTGATAATCAG*TCTACAGGTGTATCCAGCAGCTCCAGAGAGA
>AC108519.1--AF228730.5
AGATGGGAGAAAAACCGCCCTATGGTGGGAG*GCGAGACATGTTTACAGCAATGCTGCCTTGT
>AC127526.5--AC010329.1
GGAGCTGAATGTGCCAGCACCAGAACTTCTG*AAAAAGACAAAAAATTTGTTTCTCCAATTAG
>AC127526.5--AC010329.1
TCTTAACTGAGAAAGGAGCTAAAGAGTAGAA*AAATATGTTTGTCATGAGTCAGGGGTTTGTG
>AC127526.5--AC079804.3
ACAGCCATGTAAAACCTGCAGCGCGTGAAAC*CTGGGAGGCAAGACCAAGCCCAGACAACCCT
>AC127526.5--AC079804.3
CCAGCCTGGATCCACCACAGCCATGTAAAAC*CTGCAGGGCCTGAAACCTGGGAGGCAAGACC
>AC127526.5--AC079804.3
GTGGACTTCACAGAGATGTCAAAGTGTAGAG*ATGATCGAGTGTGGATCAAGGACTGGAAAGT
>AC127526.5--AC079804.3
TGCAAGGTATGTAGCCCGAGAAATGACCGAC*CTGATGTGTGTTATAACCCATCTGAGCCCCC
>AF228730.5--AC108519.1
ACACTATCGGGGGACCTGCCCTGATAATCAG*TCTACAGTTGTATCCAGCAGCTCCAGAGAGA
>AL109811.1--EXOSC10-AS1
AAGAAACTTTGGATTTGCTGTTTCCTGAAAG*GTAAGCTCCTGAAGAAAGCGTCTAGCTCCAA
>AL391840.3--SH3BGRL2
TCATTCCATCTTCATTCATGACTCCAGACAG*AGAACTTGATGCACACCTGGACAAAAGGACG
>AL391840.3--SH3BGRL2
TCATTCCATCTTCATTCATGACTCCAGACAG*ATAAAGAAGAAGCAGCAAGATGTGGTTAGAT
>AL392086.3--ZNF585A
TGTGCCTGTTTTGTCTGTTGTTAGTCTGCAG*GTGTACCCAACAGCTCCAAAGAGACAGCGAC
>AL445524.2--SCAND2P
CCTAGGCTGGGGTTTAAGTTAAATTTTTTAA*AAAAACTAAAGTGACTGGCACTAAGTGAACT
>AL627171.2--RN7SL151P
ACCGCCGGGCGCGGTGGCGCGTGCCTGTAGT*CAGCTACTCAGGAGGCTGAGGCTGGAGGATC
>AL627171.2--RN7SL151P
ACCGCCGGGCGCGGTGGCGCGTGCCTGTAGT*CCAGCTACTCAGGAGGCTGAGGCTGGAGGAT
>AL627171.2--RN7SL151P
ACCTCCCGGGAGCGGGGGACCACCAGGTTGC*AAGGAGGGGTGAACTGGCCCAGGTCGGAAAC
>AL627171.2--RN7SL151P
ACCTCCCGGGAGCGGGGGACCACCAGGTTGC*AGGAGGGGTGAACTGGCCCAGGTCGGAAACG
>AL627171.2--RN7SL151P
ACTAAGTTCGGCATCAATATGGTGACCTCCC*GGGAGCGGGGACCACCAGGTTGCCTAAGGAG
>AL627171.2--RN7SL151P
AGGTTGCCTAAGGAGGGGTGAACCGGCCCAG*CTACTCAGGAGGCTGAGGCTGGAGGATCGCT
>AL627171.2--RN7SL151P
AGTGCGCTATGCCGATCGGGTGTCCGCACTA*AAGTTTGGCATCAATATGGTGACCTCCTGGG
>AL627171.2--RN7SL151P
ATCAATATGGTGACCTCCCGGGAGCGGGGGA*CAGGTTGCCTAAGGAGGGGTGAACTGGCCCA
>AL627171.2--RN7SL151P
CATCAATATGGTGACCTCCCGGGAGCGGGGG*ACCAGGTTGCCTAAGGAGGGGTGAACTGGCC
>AL627171.2--RN7SL151P
CATCAATATGGTGACCTCCCGGGAGCGGGGG*CACCAGGTTGCCTAAGGAGGGGTGAACTGGC
>AL627171.2--RN7SL151P
CCACCAGGTTGCCTAAGGAGGGGTGAACCGG*CCAGGTCGGAAACGGAGCAGGTCGAAACTCC
>AL627171.2--RN7SL151P
CCAGCCTGAGCAACATAGCGAGACCCCGTCT*CTACTCAGGAGGCTGAGGCTGGAGGATCGCT
>AL627171.2--RN7SL151P
CCAGCCTGAGCAACATAGCGAGACCCCGTCT*TACTCAGGAGGCTGAGGCTGGAGGATCGCTT
>AL627171.2--RN7SL151P
CCGACCGCCGGGCGCGGTGGCGCGTGCCTGT*CCAGCTACTCAGGAGGCTGAGGCTGGAGGAT
>AL627171.2--RN7SL151P
CTCCCGGGAGCGGGGGACCACCAGGTTGCCT*AGGAGGGGTGAACTGGCCCAGGTCGGAAACG
>AL627171.2--RN7SL151P
GACCACCAGGTTGCCTAAGGAGGGGTGAACC*CCAGGTCGGAAACGGAGCAGGTCGAAACTCC
>AL627171.2--RN7SL151P
GACCGCCGGGCGCGGTGGCGCGTGCCTGTAG*CCAGCTACTCAGGAGGCTGAGGCTGGAGGAT
>AL627171.2--RN7SL151P
GCAGGTCAAAACTCCCGTGCTGATCAGTAGT*AGGATCGCTTGAGTCCAGGAGTTCTGGGCTG
>AL627171.2--RN7SL151P
GCCTGTAGTCCCAGCTACTCGGGAGGCTGAG*GCTGGAGGATCGCTTGAGTCCAGGAGTTCTG
>AL627171.2--RN7SL151P
GCCTGTAGTCCCAGCTACTCGGGAGGCTGAG*GGAGGATCGCTTGAGTCCAGGAGTTCTGGGC
>AL627171.2--RN7SL151P
GCCTGTAGTCCCAGCTACTCGGGAGGCTGAG*TGGAGGATCGCTTGAGTCCAGGAGTTCTGGG
>AL627171.2--RN7SL151P
GGACCACCAGGTTGCCTAAGGAGGGGTGAAC*CCAGGTCGGAAACGGAGCAGGTCGAAACTCC
>AL627171.2--RN7SL151P
GGACCACCAGGTTGCCTAAGGAGGGGTGAAC*GGCCCAGGTCGGAAACGGAGCAGGTCGAAAC
>AL627171.2--RN7SL151P
GTGCCTGTAGTCCCAGCTACTCGGGAGGCTG*GCTGGAGGATCGCTTGAGTCCAGGAGTTCTG
>AL627171.2--RN7SL151P
TCCGCACTAAGTTCGGCATCAATATGGTGAC*CTCCTGGGAGCGGGGACCACCAGGTTGCCTA
>AL627171.2--RN7SL151P
TGACCTCCCGGGAGCGGGGGACCACCAGGTT*AAGGAGGGGTGAACTGGCCCAGGTCGGAAAC
>ANKRD36C--LINC00342
ATCTCAGGAAAATTGTGAAAATAATCAGCAA*TTAGGGAAGAAATATAAACACTGAGATTGCC
>ANKRD36C--LINC00342
CAAGCTACCATTGACTTTCTTCACAGAATTA*AAAAAAAAAACTACTTTAAATTTCATATGCA
>ANKRD36C--LINC00342
CTCAGGAAAATTGTGAAAATAATCAGCAACC*GCAAGTCTTGCAGGCCTGGGCCTCTACCTAA
>ANKRD36C--LINC00342
CTCAGGAAAATTGTGAAAATAATCAGCAACC*GTGATGGCATCAGCTGAGAAGGGAAGGAAGC
>ANKRD36C--LINC00342
CTCAGGAAAATTGTGAAAATAATCAGCAACC*TTAGGGAAGAAATATAAACACTGAGATTGCC
>ANKRD36C--LINC00342
CTCAGGAAAATTGTGAAAATAATCAGCAACC*TTTCTCCACAGACACTACCCAAAGCAGTCCT
>ANKRD36C--LINC00342
TAGACTACCATTGACTTTCTTCACAGAATTA*AAAAAAAAAACTACTTTAAATTTCATATGCA
>ANKRD36C--LINC00342
TCAAGCTACCATTGACTTTCTTCACAGAATT*AAAAAAAACTACTTTAAATTTCATATGCAAC
>ANKRD36C--LINC00342
TGTTTTTTTTTTTCTTTCCCTCTCTCTCTCT*CACACACACACACACACACAAACTTCAGGCC
>AP003498.2--AF228730.5
GGGAAATCGCTGGTATTAGAAGAGTGAAAAG*AAACTCCTGGAGGGCAGATGCATGCCTGCCA
>ARL17A--AC005670.3
AATGCCCCTGGGGAGACACATGCACAAGACA*GCTTCGAGGGTCTGGCTTGAGCAGCACAGAC
>ARL17A--AC005670.3
AATGCCCCTGGGGAGACACATGCACAAGACA*GTCCACGCTGCTTTTATGAGCTGTAACACTC
>ARL17A--AC005670.3
CTGTGGCAGCATTTTTTCCAGAACACAAAAG*CAATGAATCCTCCAATGTACCTGACTCTCCC
>ARL17A--AC005670.3
GGCTTTCACTATGTTGGCCAAGCTGGTCTCA*AAACTCCTGACTTCAAGTAATCCACCCACCT
>ARL17B--AC103810.1
GTTATACACTTACATTACAAATTCCTTTTCA*TTTTTTTTCTCTTTTTCCCTTTTTACCTACC
>DA750114--AC011346.1
AATGTCTTCTTTTGAGAAGTGTCTGTTCATA*CTTCACCCACTTTTTGATGGGGTTGTTTGTT
>DA750114--LINC02503
CTGAAGGAAATAGAGACACAAAAAACCCTTC*AACCCTTCAAAAAATTAATGAATCCAGGAGC
>FAM153A--FAM153CP
AGACTCCACCATTACAGGCAGCCACCAGCAG*ATGTCAGCAAGTCCTTCCTCTGCACCTGCAG
>FAM153A--FAM153CP
AGTCTCTGAGGAAGCCACAGGGGTTCACATG*ATGCAGGTGGACCCAGCCACACCGGCAAAGA
>FAM153A--FAM153CP
GAAGGCTGTGTTGTTTGTCTACCTGGGACTA*CACCTACTGAAAGAAGTTCTCAAGTTCTGAC
>FAM153A--FAM153CP
GTGAAGACATCCTTGGGAGTTCCACAACGTG*GTGACCTGGAAGACCTGGAGGAGCATCTGCC
>FAM153CP--FAM153A
AGACTCCACCATTACAGGCAGCCACCAGCAG*ATGTCAGCAAGTCCTTCCTCTGCACCTGCAG
>FAM153CP--FAM153A
AGTCTCTGAGGAAGCCACAGGGGTTCACATG*ATGCAGGTGGACCCAGCCACACTGGCAAAGC
>FAM153CP--FAM153A
ATGAAGACATCCTTGGGAGTTCCACAACGTG*GTGACCTGGAAGACCTGGAGGAGCATGTGCC
>FAM153CP--FAM153A
GAAGGCCGTGTTGTTTGTCTACCTGGGACTA*CACCTACTGAAAGAAGTTCTCAAGTTCTGAT
>FAM153CP--FAM153A
GAGCCAGCCACCCTCATAGAGGCCATCCCAG*ATTACGATGAGGACCTGGTGCAGGAAGCTTC
>FAM153CP--FAM153A
TTACAGACATCCTTGGGAGTTCCACAACGTG*GTGACCTGGAAGACCTGGAGGAGCATGGGCC
>FAM27C--AL953883.1
TCAGTCCATCCTTGGCTGTTCAAGTGTAGAG*AAATAAAAACATCAGTAGGAGGGGAAGGAGG
>FASTKD2--AC008269.1
TGGATACAACTTCTGCTACAGATATTCAAAG*CATATCAGCAATCATGGTGACGTGCCACTCT
>FASTKD2--AC008269.1
TTTGAATGCAATGGGTTTTCATGTGATCTTG*CATATCAGCAATCATGGTGACGTGCCACTCT
>FASTKD2--AC008269.1
TTTGAATGCAATGGGTTTTCATGTGATCTTG*GTCTGGCTGATTTCACTTAACATAATGTCCT
>FASTKD2--AC008269.1
TTTGAATGCAATGGGTTTTCATGTGATCTTG*TTTTCCTCTTCTCCTGTGTTAGAAAGCGTGA
>FRG1CP--FRG1HP
TAAAACAGCAGGAGAAGAAGAAATGATCAAG*ATTAGATCCTGTGCTGAAAGAGAAACCAAGA
>FRG1CP--FRG1HP
TCTACCAGAACACAGATCCCTCTTACTAAAG*ATTAGATCCTGTGCTGAAAGAGAAACCAAGA
>FRG1HP--FRG1CP
TAAAACAGCAGGAGAAGAAGAAATGATGAAG*ATTAGATCCTGTGCTGAAAGAGAAACCAAGA
>HULC--PLCB1
GACTCCGTGGGCGTAGGACCCTGCGAGCCAG*GACTCAACTATTGTTACTCCAATTATTTTGA
>IGH@--AC011346.1
AATATACATTCTTTTCAGCACCACACCACAC*CTATTCCAAAATTGACCACATAGTTGGAAGT
>IGH@--AC011346.1
CTCTCTCTCTATATATATATATATATATATA*TACATATATATATACACATACACACACATAT
>IGH@--AC011346.1
TATGACAAACCCACAGCCAATATCATACTGA*GGCAAAAACTGGAAGCATTCCCTTTGAAAAC
>IGH@--AC012349.1
AATATACATTCTTTTCAGCACCACACCACAC*TTATTCCAAAATTGACCACATAGTTGGAAGT
>IGH@--AC109466.1
TCTTTTCAAAAAACCAGCTCCTGGATTCATT*GATTTTTTGAAGGGTTTTTTGTGTCTCTATC
>KCNK6--AC005224.2
AAGAAAGAAAGAAAGAGAGAGAAAGAAAGAA*AGAAAGAAAGAAAGAAAGAAAGAAAGAAAGA
>KCNK6--AC005224.2
CCCTGTCTGAAAGAAAGAGAGAAAGAAAGAA*AGAAAGAGAGAAAGAAAGAAAGAAAGAAAGA
>KCNK6--AC005224.2
CCCTGTCTGAAAGAAAGAGAGAAAGAAAGAA*AGAGAAAGAGAGAAAGAAAGAAAGAAAGAAA
>KCNK6--AC005224.2
CTGAAAGAAAGAGAGAAAGAAAGAAAGAAAG*AGAGAGAAAGAAAGAAAGAAAGAAAGAAAGA
>KDELC1P1--POGLUT2
TCTTCCCTTGCACTTGCCATAGGAAAAAGAC*CAAAGATGAACTCTGATATGCAAAATAACTT
>LAMC2--AL354953.1
AACACATTAGACGGCCTCCTGCATCTGATGG*ACCCTGCACTTGATGGACCAGCTGGCACCAC
>LAMC2--AL354953.1
AACACATTAGACGGCCTCCTGCATCTGATGG*GAGCCCATGGAGGAGAGGCACCAACAGGTAT
>LINC00271--PDE7B
ACCAAATAGAAGAATAAGCAGTGCCTGTGTG*AGGTGTGGCGAAATCTTGTTTGAGAACCCCG
>LINC00271--PDE7B
GAGCCCACACCCTTCATCACTACTGATAAAG*AGGTGTGGCGAAATCTTGTTTGAGAACCCCG
>LINC00271--PDE7B
GAGCCCACACCCTTCATCACTACTGATAAAG*GTGTGGCGAAATCTTGTTTGAGAACCCCGAT
>LINC00271--PDE7B
TTGAAACTGCGGCTGTAAAAAGTTAGTGCTG*AGGTGTGGCGAAATCTTGTTTGAGAACCCCG
>LINC00271--PDE7B
TTGAAACTGCGGCTGTAAAAAGTTAGTGCTG*GTGATTTGGGGTAAGGACTCCAAGGAAACAA
>LINC00342--ANKRD36C
CAATATCGTGAAAATGGCCATACTGCCCAAG*TGTCTTCTCAGAAACAACCAGCTGAGAAGGT
>LINC00342--ANKRD36C
CCATTGACATTCTTCACAGAATTAAAAAAAA*AAAAAACTACTTTAAATTTCATATGGAACCA
>LINC00342--ANKRD36C
CTACCATTGACATTCTTCACAGAATTAAAAA*AAAAAACTACTTTAAATTTCATATGGAACCA
>LINC00342--IGH@
ACTACCATTGACATTCTTCACAGAATTAAAA*AAAAAAACTACTTTAAATTTCATATGGAACC
>LINC00680--AC091978.1
ACTTAGGGACAAGACTCCTTAGTGGTGGAAC*CTCAGGCATCCGTGCAGGATGTAAGAAAGCT
>LINC00680--AC091978.1
CCCTGTCCAGTGTTGGTGCTGTGTGGCATTG*GCTTGTGTCCATAGGAGTCCATCAAGGCAGA
>LINC00680--AC091978.1
TTTGGGACGCAGCACTTGAAGAGAGGTTTAC*CTAAACTTTCCTCCTGGGAGCCTGTGATCAT
>LINC01445--ACOT11
GAAATAGGTAAAATAATAACTTGGGGTAGAG*GTTATGCTTGCGTTTCTCCAGGCCAAAATCA
>LYPLAL1-AS1--MYO1D
GCCTCAGCCTCCCAAGTAGCTGGGATTACAG*ATTTGAAAAAGGGCGCATCTATACGTTCATT
>MIR100HG--AC138646.1
TTCTTTCTTTCTTTCTTTCTTTCTTTCTTTC*TCTCTCTCTCTCTCTCTCTCTCTTTACTAAG
>NUTM2A--NUTM2B
ACCGGATGATCTTCTACGAGATGGCGGAAAA*GTTCCTGGAGTTTGAGGCTGAGGAGGAGATG
>NUTM2A--NUTM2B
CTGACACCGAAGCGCTTTCGTGCTTCCTCAT*CCCAGTTCTCCGATCGCTGGCCCGGCGGAAG
>NUTM2A--NUTM2B
GCTGTGTTCCCAGAAAGACTTCGTCACCAAG*GTGGAGGCCGTCATTCATCCCCAATTCCTGG
>NUTM2A--NUTM2B
GGAGCAGGAGGAAGGACTCACCCTTGCCCAG*CTAGTGGAGAAGCGCCTCCCACCCTTGAAGG
>NUTM2B--NUTM2A
CTGACACCGAAGCGCTTTCGTGCTTCCTCAT*CCCAGTTCTCCGATCCCTGGCCCGGCGGAAG
>NUTM2B--NUTM2A
GCTGTGTTCCCAGAAAGACTTCGTCACCAAG*GTGGAGGCCGTCATTCATCCCCAATTCCTGG
>NUTM2B--NUTM2A
GGAGCAGGAGGAAGGACTCACCCTTGCCCAG*CTAGTGGAGAAGCGCCTCCTACCCTTGAAGG
>PLBD1-AS1--AC010168.1
CCAGGCTAGACTATGCGCTGTGGCATATTAG*GTTGAGTTGCTGAAGCGGCTTCATTGTCTGG
>PLBD1-AS1--AC010168.1
GACAGAAGACAAGACAAAGGCATAACTTTCC*GTTGAGTTGCTGAAGCGGCTTCATTGTCTGG
>RN7SL151P--AL627171.2
ACTGGCCCAGGTCGGAAACGGAGCAGGTCGA*AAAACTCCCGTGCTGATCAGTAGTGGGATCG
>RN7SL151P--AL627171.2
AGTCCAGGAGTTCTGGGCTGTAGTGTGCTAT*CGATCGGGTGTCCGCACTAAGTTCGGCATCA
>RN7SL151P--AL627171.2
AGTCCAGGAGTTCTGGGCTGTAGTGTGCTAT*GATCGGGTGTCCGCACTAAGTTCGGCATCAA
>RN7SL151P--AL627171.2
ATCGCTTGAGTCCAGGAGTTCTGGGCTGTAG*TATGCCGATCGGGTGTCCGCACTAAGTTCGG
>RN7SL151P--AL627171.2
ATGGTGACCTCCTGGGAGCGGGGACCACCAG*GTTGCCTAAGGAGGGGTGAACCGGCCCAGGT
>RN7SL151P--AL627171.2
CAGGAGGCTGAGGCTGGAGGATCGCTTGAGT*AGGAGTTCTGGGCTGTAGTGCGCTATGCCGA
>RN7SL151P--AL627171.2
CAGGAGGCTGAGGCTGGAGGATCGCTTGAGT*CAGGAGTTCTGGGCTGTAGTGCGCTATGCCG
>RN7SL151P--AL627171.2
CAGGAGGCTGAGGCTGGAGGATCGCTTGAGT*GAGTTCTGGGCTGTAGTGCGCTATGCCGATC
>RN7SL151P--AL627171.2
CGCACTAAGTTTGGCATCAATATGGTGACCT*CCGGGCGCGGTGGCGCGTGCCTGTAGTCCCA
>RN7SL151P--AL627171.2
CGCACTAAGTTTGGCATCAATATGGTGACCT*GCCGGGCGCGGTGGCGCGTGCCTGTAGTCCC
>RN7SL151P--AL627171.2
GAACTGGCCCAGGTCGGAAACGGAGCAGGTC*AAAACTCCCGTGCTGATCAGTAGTGGGATCG
>RN7SL151P--AL627171.2
GCTTGAGTCCAGGAGTTCTGGGCTGTAGTGT*GCTATGCCGATCGGGTGTCCGCACTAAGTTC
>RN7SL151P--AL627171.2
GGGACCACCAGGTTGCCTAAGGAGGGGTGAA*CGGCCCAGGTCGGAAACGGAGCAGGTCAAAA
>RN7SL151P--AL627171.2
GGGTGTCCGCACTAAGTTTGGCATCAATATG*GACCTCCCGGGAGCGGGGGACCACCAGGTTG
>RN7SL151P--AL627171.2
GTCCAGGAGTTCTGGGCTGTAGTGTGCTATG*CGATCGGGTGTCCGCACTAAGTTCGGCATCA
>RN7SL151P--AL627171.2
GTCCGCACTAAGTTTGGCATCAATATGGTGA*CTCCCGGGAGCGGGGGACCACCAGGTTGCCT
>RN7SL151P--AL627171.2
TCCGCACTAAGTTTGGCATCAATATGGTGAC*CTCCCGGGAGCGGGGGACCACCAGGTTGCCT
>RN7SL151P--AL627171.2
TCCGCACTAAGTTTGGCATCAATATGGTGAC*GCCGGGCGCGGTGGCGCGTGCCTGTAGTCCC
>RN7SL151P--AL627171.2
TCGCTTGAGTCCAGGAGTTCTGGGCTGTAGT*GCTATGCCGATCGGGTGTCCGCACTAAGTTC
>RTCA-AS1--IGH@
AGGTACAAAAGTACAAAGATCACATGCTTCA*AAAGGCAAAAAGCAGAACTACTAATAAGGGT
>SCAND2P--AL445524.2
TAAGTTGATTGTACTTCCAAATTTGCTGTTA*TGTTGCCAGTGACTGATGATTAATTTTATTT
>ST6GALNAC1--AC015802.7
ACCCAGACTTTCTCCGATACATGAAGAACAG*ATGGTTTGTACAATAAAGGGGAAGATTAAAG
>ST6GALNAC1--AC015802.7
GCTGCTCACTGCCCTTCAGCTCTGTGACCAG*ATGGTTTGTACAATAAAGGGGAAGATTAAAG
>TIMM23B--AC022400.6
AAGAAGCTACACAAAAGGCAGCAAAGTATTA*GTGAGTGGTGTCATTGAGCAAACACGAAGTC
>TRA@--AC109466.1
ATCAGAGACTAGGATTGCAACCCCTGCCTTT*TTTTTTTTGCTTTCCATTTGCTTGGTAAATA
>TRA@--AL022151.1
TATCGGAGACTAGGATTGCAACCCCTGCCTT*TTTTTTTGTTTTCCATTTGCTTGGTAGATCT
>TRA@--AL590807.1
CTACAAAGAACTTAAACAAATTTACAAGAAA*AAACAACCCCATCAAAAAGTGGGCGAAGGAT
>ZNF585A--AL392086.3
TCCAGCACCTGTTGTTTCCTGACTTTTTAAT*GATTGCCATTCTAACTGGTGTGAGATGGTAT
>NPM1--ATAD2
GATGAAGATGATGACGATGATGATGAAGAGG*ATGATGATGATGATGACGATGATGATGATGA
>ABCC6--NEAT1
AGCCTGGGTGACAGAGTAAGACTCCGTCTCA*AAAAAAACCAAAAGACTTTATCTTATTTCCT
>ABCC6--NEAT1
GTGTGTGTGTGTGTGTGTATTTTTTTTTTTT*ATTTCAATGGCTTTTGGGGTACAAATGGCTT
>ABHD17A--CCDC144NL-AS1
TCGACGCCGCCTGGCAGGCCCTGCGCACCAG*TGACACAAATCACTATGTAATATAGTTAGCC
>AC007262.2--CCND3
TATTCTGTGTTCATGAATTGGAAGAATTAAT*ATGTTTAAATTAATTATTACCTGACGTTACC
>AC008883.3--AC092683.1
AAACCACACAACTACATGGAAATTGAACAAC*GTGCTCCTGAGTGACTTCTGGGTAAATAATA
>AC009948.4--THEM4
ACTCTGTCTCAAAACAAACAAACAAACAAAC*AAAAAAAAACTCTTGTCTCCTTAGGATATGT
>AC009948.4--THEM4
CTCTGTCTCAAAACAAACAAACAAACAAACA*AAAAAAAAAAAACTCTTGTCTCCTTAGGATA
>AC009948.4--THEM4
CTGTCTCAAAACAAACAAACAAACAAACAAA*AAAAAAAAAAAACTCTTGTCTCCTTAGGATA
>AC009948.4--THEM4
GTCTCAAAACAAACAAACAAACAAACAAACA*AAAAAAAAAAAACTCTTGTCTCCTTAGGATA
>AC009948.4--THEM4
TCTCAAAACAAACAAACAAACAAACAAACAA*AAAAAAAAAAAACTCTTGTCTCCTTAGGATA
>AC009948.4--THEM4
TCTGTCTCAAAACAAACAAACAAACAAACAA*AAAAAAAAAAAACTCTTGTCTCCTTAGGATA
>AC009948.4--THEM4
TGTCTCAAAACAAACAAACAAACAAACAAAC*AAAAAAAAACTCTTGTCTCCTTAGGATATGT
>AC012103.1--NEAT1
GTGATGCATGACTTTGTGTGTGTGTGTGCGT*GTGTGTGTGTGTGTGTGTGTGTATGATTTTC
>AC015883.1--CRLF2
ACTGGAAATGGTATTCCCCAGCTTAAAGAAA*GAAAGAAAGAAAGAAGGAAAGAAAGAAAGAA
>AC015883.1--CRLF2
ATTCCCCAGCTTAAAGAAAGAAAGAAAGAAA*GAAGGAAAGAAAGAAAGAAAGAAAGAAAGAA
>AC022400.4--TIMM23B-AGAP6
AGGAAGCTACACAAAAGGCAGCAAAGTATTA*CTTTGGAGTTTAACCTTTCTGCCAATCCAGA
>AC022400.4--TIMM23B-AGAP6
AGGAAGCTACACAAAAGGCAGCAAAGTATTA*GTGAGTAGTGTCATTGAGCAAACACAAAGTC
>AC024560.4--BBS2
TAAAAAATAATAATTCTTACATTCGTTTGAA*GGTTTTTTGTTTGTTTGTTTGTTTGTTTTGA
>AC026704.1--MTREX
AACTGATGAACTGACTAAAGGAAAAACAGAG*GAAACGTTTTGATGGTAAATTACAATCAGAA
>AC044860.1--GOLGA8F
CGGCCCGCGGGGGAGCCCCGGGAGCCGCACG*GAGATGGAGGAGGACATCTGGACAGTGAGGG
>AC044860.1--GOLGA8G
CGGCCCGCGGGGGAGCCCCGGGAGCCGCACG*GAGATGGAGGAGGACATCTGGACAGTGAGGG
>AC044860.1--GOLGA8S
CGGCCCGCGGGGGAGCCCCGGGAGCCGCACG*GAGATGGAGGAGGACATCTGGACAGTGAGGG
>AC078795.1--KRT19
CTTTGGAAGGCGGAGGCGGGCGGATCACGAG*ATTCTTGGTGCCACCATTGAGAACTCCAGGA
>AC092418.2--DNAH12
TATCTTATTTAAGCCTCATAGTTCTTATGAG*AGTACAGAGTATTTTAATCTTTAGGGGATCA
>AC092683.1--AC008883.3
ATAACAATCTCTCAGACCACAGTGCAATCAA*ATTAGAACTCAAGATTAAGAAACTCACTCAA
>AC096759.1--AC025244.2
AGAAGGGATCTACCCCTGGACTTTTTCCTGG*ACAAAATAGCTCACTTGCTGGAACCTGCTGA
>AC127526.4--LINC02614
CCTTTGCAGTGGATCACAGCTACCTGCTGAG*AAACTCCTGGAGGGCAGAGCCATGCCTGCCA
>AC139795.2--CDC6
CTCCTTCCCTCCCTTCTTTCTTTCTTCCTTC*TTTTGTTTGTTTTTTTTGTTGTTGTTGTTTT
>AC139795.2--IBA57
TCCCTTCTTTCTTTCTTCCTTCTTTTGTTTG*TTTTTTTTTTGTTGTTGTTGTTGTTGTTTTT
>AGO3--MPV17L
AGCCTGGGTGGCAGAGTGAGACTCCGTCTCA*AAAAAAAAAAAGTTTCAGTAAATTCCACCTA
>AL031705.1--CCND3
GACACTCCATCTCAAATAAATAAATAAATAA*ATAAATAAATAAATAAATAAAAGGAAAGAAA
>AL049840.4--IGH@
TGGATGGGATTAAAAGAAAAAATGTTAAATT*ATATATATATATATATGGCTGGGTGCAGTGG
>AL590617.1--MTCH1
TTCCACTGGGTGTCATCAGGATCATCTTTCA*CCTGGAGTAACCTGAATCATCTAAAAAACAC
>ANK3-DT--CDK1
CACAGAGCAGGCTCCAGGCTCAAAGCCCTAG*GATCTACCATACCCATTGACTAACTATGGAA
>ANKRD36B--LINC00342
CTCAGGAAAATTGTGAAAATAATCAGCAGCC*GTGATGGCATCAGCTGAGAAGGGAAGGAAGC
>ANKRD36B--LINC00342
CTCAGGAAAATTGTGAAAATAATCAGCAGCC*TTTCTCCACAGACACTACCCAAAGCAGTCCT
>AP002852.1--ODF1
AACCCAGGAGGCGCCTTCCGCCCACCCACAG*ACTGAGAAGAACAACAAATAGAATTCTGGCT
>APC2--GSR
CAGTCTGGACAACAGAGCGAGACCCTGTCTC*AAAAAAAAAAAATATATATATATATATATAT
>ARHGAP1--CPT1A
CTAGCCTGGGCGACAGAGCGAGACTCCGTCT*AAAAAAAAAGCAAAAACAAACCAACAACAAA
>ARSD--BCL2
CTGGACTGAGAGTGGGTCCCCACTTTCTTTC*TTTCTTTCTTTCTTTTTTTTTTGAGACAGAG
>B4GALT4--IGSF11
GAGACAAAGGCAATGAGGTGAACGCAGAACG*ATGAATGGATAAAGAAAATGCAGGACATATG
>BANK1--RNF152
GTGTTTGTGTGTGTGTGTGTGTGAGAGAGAG*AGAGAGAGAGAGAGAGAGAATCTCAGTCTCC
>BANK1--RNF152
GTTTGTGTGTGTGTGTGTGTGAGAGAGAGAG*AGAGAGAGAGAGAGAGAGAATCTCAGTCTCC
>BANK1--RNF152
TGTGTGTTTGTGTGTGTGTGTGTGTGAGAGA*GAGAGAGAGAGAGAGAGAGAGAATCTCAGTC
>BCL2--NEAT1
GAAAATATCCTTCGTGTGTGTGTGTGTGCGT*GTGTGTGTGTGTGTGTGTGTGTATGATTTTC
>BCL2--NEAT1
TATGTGTGTGTATGTGTGTGTGTGTGTGCGT*GTGTGTGTGTGTGTGTGTGTGTATGATTTTC
>BCL2--NEAT1
TGTGTGTGTGTTTATGTGTGTGTGTGTGTGT*TTTTTTTTTTTTTAATTTCAATGGCTTTTGG
>BCL2--NEAT1
TTGAAAATATCCTTCGTGTGTGTGTGTGTGC*GTGTGTGTGTGTGTGTGTGTGTATGATTTTC
>BCL2--RNF152
TATATTTTTGTGTGTGTGTGTGTGTGTGAGA*GAGAGAGAGAGAGAGAGAGAGAATCTCAGTC
>BCL2--RNF152
TATTTTTGTGTGTGTGTGTGTGTGTGAGAGA*GAGAGAGAGAGAGAGAGAGAGAATCTCAGTC
>BCL2--RNF152
TTTTGTGTGTGTGTGTGTGTGTGAGAGAGAG*AGAGAGAGAGAGAGAGAGAATCTCAGTCTCC
>C21ORF58--PIK3CG
ACTGCACTATAGCCTGGACCACAGAGCAGGC*CCCTGTCTCAAAAAAATAAAATAAAAAATAA
>CAPN2--IRF2BP2
AGGCATTTGTGTTTTTTGGTAGAAAGTTTGC*TTTTGTTTTTTTTTTTTTAAGACAAAGTTGA
>CAPN2--IRF2BP2
CAAACTTCTGAGGCAGGCACTTTCTTATCAC*TGGAATTTGACCTCACAGATGTTACAACTTG
>CFAP221--TMEM177
TACCTGATTCTAGAATGAAAGTCACCAGTAG*GAAAGGGAACATCACAGAAAGGTGAAGAAAT
>CFAP69--AC002064.1
TTGTGGATGCACCAAAAAAGAGTATTCCTAC*GAGCTGGTTGGAGAAGATGAACGTAGACATA
>CHODL--DIAPH2
TGCTTGGCTCTCTAAATTGACTCAGCTCCAG*ATCAGAGATACTAAATCAGCGGATCAAAAAA
>CIPC--CSAD
ATGGTCAGCTTCATCTTCCTTTTTTTTTTTT*GAGATGGAGTTTCGCTCTTGTAGCCCAGGCT
>CIPC--SSB
TTCTATATGGTCAGCTTCATCTTCCTTTTTT*TTTTTTGAGATGGAGTTTCGCTCTTGCTGCC
>CTCF--RBBP4
TGCGCTCCAGCCTGGGTGGCAGAGCGAGACT*CTGTCTCAAAAAAAAAAGAAAGTTGTGAATT
>CXADR--GABPB2
TGTCTTAGTTTTGTGAAGGTGATTTATTCTT*AAAAAAAAAAAAAGAAAGAAAAAGGAAAAAA
>CXADR--GABPB2
TGTCTTAGTTTTGTGAAGGTGATTTATTCTT*AAAAAAAAAAAGAAAGAAAAAGGAAAAAAAG
>CXADR--GABPB2
TTGTGAAGGTGATTTATTCTTAAAAAAAAAA*AAAGAAAGAAAAAGAAAATATTTTCGACCTA
>CXADR--PCLAF
AGTTTGTCTTAGTTTTGTGAAGGTGATTTAT*TCTTAAAAAAAAAAAAAGAAAGAAAAGAAAA
>CXADR--PRKAR2A
CTTAGTTTTGTGAAGGTGATTTATTCTTAAA*AAAAAAAAAAAAAAAGAAAAAGGAAAAACCC
>CXADR--PRKAR2A
GTGAAGGTGATTTATTCTTAAAAAAAAAAAA*AATAAGAAAAAGATAAAAACCCTAATAGAAA
>CYP1B1-AS1--RIN2
GAGTTCCAGGGGCTCTGGGAGTGGCTGCCAG*ATGGAGACGAGAGATCTGGACTTCTGAACAT
>CYP1B1-AS1--RIN2
GTCTGTGAAGCCTTGCGGCAGTACAGCCCAG*GATTGATTGCTGATTACCATGGAGAAGCAGG
>EGFR--IL6ST
TATATATATATATATATATATATACACACAC*ATATACACACACATATCATTGACTTTTCTTA
>EIF1AX--BX284668.3
GTCGCCGCCACCGCCATGCCCAAGAATAAAG*GAAATATGATTCCCTGATTCAGGATCAGCCC
>F3--PIGS
ATCCTGGGTGGACTTTTGAAAAGCTTTTTTT*TTTTTTTTTTTGAGACGGAGTCTTGCTCTGT
>FAM83F--TNRC6B
CCCAACGAGAATTCCAGTGCTGACATCTCAG*GGTTGTGGTTTTTTGCACTGAATGTAACCTA
>FER--AC005261.5
CACTAATTTTCAGGGAAATGCAAATCAAAAC*CACAATGAGATATAATTTCACCTCAGTTAGG
>FOXL2NB--BHLHE40
AAGATAGGGTGTGTGTGTGTGTGTGTGTGTA*TGTGTATGTGCGTGTGCGTGCACATGTGTGC
>FOXP1--AC139795.2
ATAAACTTACAATTTTTGTCTGTTTTTTTTT*TTGTTGTTGTTGTTGTTGTTTTAGACAGAGT
>FOXP1--AC139795.2
TTTTGTTTTTTTTTTTTGTTTGTTTTTTTTT*TTGTTGTTGTTGTTGTTGTTTTAGACAGAGT
>FOXP1--CD2AP
GTGTGTGTGTGTGTGTGTGTGTGTGTGTGTG*TATATATATTTTTACTTTTATCCTCTTACCG
>FOXP1--CD2AP
TACAGGTTTTGTGTGTGTGTGTGTGTGTGTG*TATATATATTTTTACTTTTATCCTCTTACCG
>FRMPD2--PTPN20
ATTCAGTTTGTGATATCATGAGACTTGGAAG*GTTTTTGAAAATAAAGTTAATTCAGAGAAGG
>FRMPD2--PTPN20
ATTCAGTTTGTGATATCATGAGACTTGGAAG*TGGCAGTGATCCCAGCATGTGGACAGCCAGA
>FRMPD2--PTPN20
GTCAGAACATATTGATGAGGACACAGAAAAA*GTTTTTGAAAATAAAGTTAATTCAGAGAAGG
>FRMPD2--PTPN20CP
ATTCAGTTTGTGATATCATGAGACTTGGAAG*GTTTTTGAAAATAAAGTTAATTCAGAGAAGG
>FRMPD2--PTPN20CP
ATTCAGTTTGTGATATCATGAGACTTGGAAG*TGGCAGTGATCCCAGCATGTGGACAGCCAGA
>FRMPD2--PTPN20CP
GTCAGAACATATTGATGAGGACACAGAAAAA*GTTTTTGAAAATAAAGTTAATTCAGAGAAGG
>GK5--AC107072.2
CAGCCTGGGTGAGAGAGCGAGACTCTGTCTC*AAAAAAAAAAAAAGAGAGAGAGAGAAAGAGA
>GK5--AC107072.2
CAGCCTGGGTGAGAGAGCGAGACTCTGTCTC*AAAAAAAAAAAGAGAGAGAGAGAAAGAGAGA
>GK5--TFDP2
CCTGGGTGAGAGAGCGAGACTCTGTCTCAAA*AAAAAAAAAAGAGAGAGAGAGAGAAAGAAAA
>GK5--TFDP2
CTGAGGCAGGAGAATTGCTTGAATCTGGGGG*GTGGAGCTTGCAGTGAGCCGAGATCGCGCCA
>GK5--TFDP2
GGGCCAAAGCAGTGAAACGCTCCATGAATTG*GTTGTTCTTTTAAAGAATTATTGAAGACGAA
>GRB10--NEAT1
GATATGTGCAATGGTGTGTGTGTGTGTGTGT*GTGTGTGTGTGTGTGTGTGTGTATGATTTTC
>GRB10--RNF152
TGTGTGTGTGTGTGTGTGTGTGTGTGTAAGA*GAGAGAGAGAGAGAGAGAGAGAATCTCAGTC
>GRWD1--CENPB
GCATGGGACAAAGCCCCCACCCTCAGAGGGC*AGTGATGAAGAAGAGGAGGAAGATGAGGAGA
>GSTA5--GSTA7P
TCTGCAGAAGATTTGGACAAGTTAAGAAATG*ATGGGAGTTTGATGTTCCAGCTAGTGTCAAT
>GSTA5--GSTA7P
TCTGCAGAAGATTTGGACAAGTTAAGAAATG*GATTGATATGTATATAGAAGGTATGGCAGAT
>HIPK2--METTL7A
CCAGGAACTTTTTTCTTTCTGTTAAAAAAAA*AAAAAAAAAAAAAAAAAAAAAAAGAAGTAGA
>IGH@--USP48
GTTTATCCCAGGGATGCAAGGATGGTTCAAC*ATATGCAAACCAATACATCACATCAGCAGAA
>IGK@--RNF152
GTGTGTGTGAGAGAGAGAGAGAGAGAGAGAG*AGAATCTCAGTCTCCCAGATAGAATTTTATA
>IGK@--RNF152
GTGTGTGTGAGAGAGAGAGAGAGAGAGAGAG*AGAGAATCTCAGTCTCCCAGATAGAATTTTA
>IGK@--RNF152
TGTGTGTGTGTGTGTGTGTGTGTGTGAGAGA*GAGAGAGAGAGAGAGAGAGAGAATCTCAGTC
>IGK@--SRGAP2C
TTCCCCCTCCTCTTTACTTTTCTTCCTCTTC*CTCTTCTTCTTTCTCGCCTTCATTATCCCTT
>INSR--ARID2
CTTTGGGGATGATTTTTTTCTTTTTTTTTTC*TTTTTTTTTTGTATTATACACCTTGTAGAAC
>IRF1--PLCL1
AATCTCTATGGCAGATAGGTCCACCGGGAGG*GTGTGTGTGTGTGTGTGTGTGTGTGAGAGAG
>JAK2--LITAF
CCCTTTTTTCCACTTAAAAAAAAAAAAAAAA*AAAAAGAAATGGATAATTTCATCTACTGCCT
>JAK2--LITAF
TTTCCACTTAAAAAAAAAAAAAAAAAAAAAA*AAATGGATAATTTCATCTACTGCCTTTACTT
>KCTD20--MSH3
CAGCCTGGGCAACAGAGCAAGACTCCATCTC*AAAAAAGAAAAAAGAAAAGAAATAGAATTAT
>KHDC4--SPG7
CACCACTGCATTCCAGCCTGGGCACTAGAGT*GAGACTCTTGTCTCAAAAAAAAAGAAAAAAG
>KLHL20--GSTT4
CTCGCACTAACCACCAGAAGACTTGAAAATC*TTAAAAAAAAAAAAACAAAAAAACAAAAAAA
>KNL1--PIGR
ACCTCTGTCCCCTGGGTTCAAGCAATTCTCC*TGCCTCAGCCTCTCAAGTAGCTGGGACTACA
>KNL1--PIGR
TGTTGCCCAGGCTGGAGTGCAGTGGCGCAAT*CTCGAACTCCTGGCCTCAAGTGATCTGCCGC
>LINC02306--AL731559.1
AACTTCTGGTGTCTCTCTCTCTCTCTGTCTG*TCTCTCTCTCTCTCTCTCTCTCTCATAGGCT
>LINC02306--AL731559.1
AACTTCTGGTGTCTCTCTCTCTCTCTGTCTG*TCTCTCTCTCTCTCTCTCTCTCTCTCATAGG
>LINC02511--LINC00963
ACCACCACCAAAAAAAAAAAAAAAAAAAAAA*AAAGATGATGTGATACATATCACATATTTCC
>MAPK6--PRSS23
AAAAAAATACACACACACACACACACACACA*CACACGTGTGTGTGTGTGTGTGTGTGTGTGT
>MAPK6--PRSS23
ATATATATACACACACACACACACACACACA*CACACGTGTGTGTGTGTGTGTGTGTGTGTGT
>MIR100HG--DIAPH2
CAGCCACAGCCCACTGGACAGCTCTTTTTTT*TTTTTTTTGAGATGGAGTCTCACTCTGTCAC
>MIR100HG--DIAPH2
GTTTTATCAGAGACTAGGATTGCAACCCCTG*TTTTTTGCTTTCCATTTGCTTGGTAGATCTT
>MIR100HG--DIAPH2
TCTCCACATCACTTTCAGGTACACCAATCAG*CACCAATCAGACGTAGATTTGGTCTTTTCAC
>MIR100HG--DIAPH2
TTCTGTTCCATTGATCTATATCTCTGTTTTG*GTACCAGTACCATGCTGTTTTGGTTACTGTA
>MTG1--NSD2
CTGGGGCCCCTCCTCCTAGTCACCTCATTTA*AAAAAAAAACAAACAAAAAAACCCCCAAACA
>MTG1--NSD2
GGGCCCCTCCTCCTAGTCACCTCATTTAAAA*AAAAAAAAACAAACAAAAAAACCCCCAAACA
>MTHFR--GRAMD1B
TCTGATTACTTGTTTCTGTTTTTTGTTTTTG*TTTTTTTTTTTTGAGATGGAGTTTTGCTTTT
>MTRNR2L12--MALAT1
AAATAACCTTAACCAAACCATTTACCCAAAT*GTATAGAAGATAGAAAAATATAAAGCCAAAA
>MTRNR2L12--MALAT1
ACCATTTACCCAAATAAAGTATAGGCGATAG*AAATTTAAACCTGAAAAGTAGGAAGCAGAAG
>MTRNR2L12--MALAT1
CAAGCCATTATTACCCTCACTGTTAACCCAA*CGCAGACGAAAATGGAAAGATTAATTGGGAG
>MTRNR2L12--MALAT1
GACAATTAACAGCCCAATATCTACAATCAAC*CACTTTTCCCTAGCTTTTCCAGAAGCCTGTT
>MTRNR2L12--MALAT1
TAAACCGGCGCAATAGATATAGTACCGCAAG*GAAAAGATAAAAGGTTTCTAAAACATGACGG
>MTRNR2L12--MALAT1
TTAACCAAACCATTTACCCAAATAAAGTATA*GCGAAAAGAAATGAAAATGTTACACTACATT
>MTRNR2L12--MALAT1
TTATTACCCTCACTGTTAACCCAACACAGGC*AAGCTAGGAAACAAAAAGCTAAGGGCAAAAT
>MTRNR2L12--MALAT1
TTGTAAACCGGCGCAATAGATATAGTACCGC*AAGGAAAAGATAAAAGGTTTCTAAAACATGA
>MTRNR2L12--MUC2
AAAGCAGCCACCAATTAAGAAAGCGTTCAAG*CCCAACACCCACCGGCACACAGACCCCAACC
>MTRNR2L12--MUC5AC
AGCCACCAATTAAGAAAGCGTTCAAGCTCAA*CACCAGCACAACCTCTGCTCCCACAACAAGC
>MTRNR2L12--MUC5AC
TCTTACCCCGCCTGTTTACCAAAAACATCAC*CAGCACAACCTCTGCTCCTACAACCAGCACG
>MTRNR2L8--MALAT1
CAAACACATAACTGAACTCCTTGCACCCAAT*TGGACCAGATCAGGATTTGAGCGGAAGAACG
>MTRNR2L8--MALAT1
CTCCTTGCACCCAATTGGACCAATCTATTAC*CTATAGAAGGAGCTTCCAGTTGAATTCACCA
>MTRNR2L8--MALAT1
TGCACCCAATTGGACCAATCTATTACCCTAT*GAAGAACGAATGTAACTTTAAGGCAGGAAAG
>MTRNR2L8--MUC5AC
CAAACCTAGCCCCAAACCCATTCCA*CCCAACAACCAGCACAACCTCTGCCACTACA
>MTRNR2L8--MUC5AC
CAAACCTAGCCCCAAACCCATTCCAC*AGACCAGCATATCCTCTGCCCCTACAAGCAG
>MTRNR2L8--MUC5AC
CAAACCTAGCCCCAAACCCATTCCAC*TACAACCAGAACAACCTCTGCCTCTCCAGCC
>MTRNR2L8--MUC5AC
CAAACCTAGCCCCAAACCCATTCCAC*TACAACCAGCACAACCTTGGCTCCTACAACC
>MTRNR2L8--MUC5AC
CAAACCTAGCCCCAAACCCATTCCAC*TACACCCAGAAGAACCTCAGCCCCTACAACC
>MTRNR2L8--MUC5AC
CAAACCTAGCCCCAAACCCATTCCACCT*ACCACCAGCACAACCTCTGCTCCTACAACCA
>MTRNR2L8--MUC5AC
CAAACCTAGCCCCAAACCCATTCCACCT*ACCACCAGCACAACCTCTGCTCCTACAACTA
>MUC16--MTRNR2L12
CTCTCCTTCTCTTCTCTCTCTCTCTCTCTCT*CACCCGTCTATGTAGCAAAATAGTGGGAAGA
>MUC16--MTRNR2L12
TAATGACAGAATCTGCAGAAATAACCATCAC*CACCCGTCTATGTAGCAAAATAGTGGGAAGA
>MUC16--MTRNR2L12
TGTTCCACTCGCCCTCTCACTCTCTCTCTCT*CCCGTCTATGTAGCAAAATAGTGGGAAGATT
>MUC4--MUC16
ACACTCTGTCAACAGCACTCTCCCCCAGTTC*CACCTCCTCTAGCAGAACATCCATCTCAGGC
>MUC5AC--MTRNR2L12
TCCACTCCACAGACCAGCACAACCTCTGCCC*ACAGAACCTTCTAAATCCCCTTGTAAATTTA
>MUC5AC--MTRNR2L12
TCTGTGGTCTCCAGCAAGCCCACCCCCACGG*AGCCCAATATCTACAATCAACCAACAAGCCA
>MUC5AC--MTRNR2L8
AGTCCAGCCCAGACCACTCCTTCAACAACCT*AAGACCCCCGAAACCAGACGAGCTACCTAAG
>MUC5AC--MTRNR2L8
AGTCCAGCCCAGACCACTCCTTCAACAACCT*CTAAGACCCCCGAAACCAGACGAGCTACCTA
>MUC5AC--MTRNR2L8
CACCAGCACAATCTCTGCTCCTACAACTAGC*AAACCTAGCCCCAAACCCATTCCACCTTACT
>MUC5AC--MTRNR2L8
CTCCCAGCCCTGTTCCTACCACCAGCATAAC*AGCCCCAAACCCATTCCACCTTACTACCGAA
>MUC5AC--MTRNR2L8
GCACGACCTCTGGTCCTGGAACTACTCCCAG*CCCCAAACCCATTCCACCTTACTACCGAACA
>NASP--AC011752.1
TGGAAAAAAGCACAGAACTAAATTCCTCAAA*AAACAAACAAACAAACAAAACCACATCCAGT
>NASP--BCL2
AAAAAAGCACAGAACTAAATTCCTCAAAAAA*CAAACAAACAAACAAAACCACAGCCCTATTG
>NEAT1--PIGR
GAATCCACAGCTGACGGTGGCTGACAGATCA*GAGAAAATGTAAAGGAGGGAGATCTTTCCCA
>NPIPB5--POLR3K
CTACTTTATGCTGTGATAGTTTGTTTGTTTG*TTTATTTATTTATTTATTTATTTAAGACAGA
>NPIPB5--POLR3K
TGTGATAGTTTGTTTGTTTGTTTGTTTGTTT*TTTTATTTATTTATTTATTTATTTAAGACAG
>NUCKS1--LYRM7
AAGTGATGATGGTCTGGGGAGAGATTTTATT*AAAAAAAAAAGAAAAAAAAAGGAAAAAGGAA
>OCLN--NEAT1
TATATGTATGCATGTATGTGTGTGTGTGTGT*TTTTTTTTTTTTTAATTTCAATGGCTTTTGG
>OR14J1--DA750114
GCTTAGGATTATCTTGGCTATGTGGGCTCTT*TTTTTGGTTCCATATGAAATTTAAAGTAGTT
>PHACTR3--AC005261.5
CATTAATCAACAGGGAAATGCAAATCAAAAC*CACAATGAGATATAATTTCACCTCAGTTAGG
>PIGR--C9ORF50
AAGAAAATTGAGGTTTGGAAGAGTGAAGCGT*TTTTCTTTTTCTTTTTTTTTTTCTGATACGG
>PIGR--EML4
AAATTGAGGTTTGGAAGAGTGAAGCGTTTTT*CTTTTTTTTTTTTTTTTTGAGACGGAGTCTC
>PIGR--EML4
TAAGAAAATTGAGGTTTGGAAGAGTGAAGCG*TTTTTTTTTTTTTTTTTTTTTTTTTGAGACG
>PIGR--KNL1
GGTTTGGAAGAGTGAAGCGTTTTTCTTTTTC*TTTCTTTTTTTTTGAGACGGAGTCTTGCTCT
>PIGR--MTO1
CACCGCGTCCGGCCTCTTTTTTTCTTTTCTT*TTTTTGAGACAGAGTCTCACTGTGTAGCCCA
>PIGR--RAD51C
GCCACCGCGTCCGGCCTCTTTTTTTCTTTTC*TTTTTTTGAGACAAAGTCTCACTCTGTCGCC
>PIGR--RAD51C
TCTGGCTAATTTTTTTTTTTTTTTTTTTTTT*TAGAGACAGGGTTTCCCCATGTTGGCCAAGA
>PIGR--RPLP1
TAAGAAAATTGAGGTTTGGAAGAGTGAAGCG*TTTTTCTTTTTCTTTTTTTTTTCTTGAGACG
>PIGR--STMND1
AGAAAATTGAGGTTTGGAAGAGTGAAGCGTT*TTTTTTTTTTTTTTTTTTTTTTTGAGACGGA
>PLAC8--AC003043.2
AATTTTGTTCTTATACTTTGATTTCCTTTTT*TTTTTTTTTTTTTTTTAATAGAGATGGGGGT
>PLAC8--AC003043.2
ATAATTTTGTTCTTATACTTTGATTTCCTTT*TTTTTTTTTTTTTTTTAATAGAGATGGGGGT
>PLAC8--AC003043.2
CATAATTTTGTTCTTATACTTTGATTTCCTT*TTTTCTTTTTTTTTTTTTTTTAATAGAGATG
>PLAC8--AC003043.2
CATAATTTTGTTCTTATACTTTGATTTCCTT*TTTTTTTTTTTTTTTTAATAGAGATGGGGGT
>PLAC8--AC003043.2
CTCATAATTTTGTTCTTATACTTTGATTTCC*TTTTCTTTTTTTTTTTTTTTTAATAGAGATG
>PLAC8--AC003043.2
TAATTTTGTTCTTATACTTTGATTTCCTTTT*TTTTTTTTTTTTTTTTAATAGAGATGGGGGT
>PLAC8--AC003043.2
TTTTGTTCTTATACTTTGATTTCCTTTTTCT*TTTTTTTTTTTTTTTTAATAGAGATGGGGGT
>PLAC8--G6PC
ATAATTTTGTTCTTATACTTTGATTTCCTTT*TCTTTTTTTTTTTTTTAATAGAGATGGGGGT
>PLAC8--SNX5
AATTTTGTTCTTATACTTTGATTTCCTTTTT*TTTTTTTTTTTTTTTTTAATAGAGATGGGGT
>PLAC8--SNX5
ATAATTTTGTTCTTATACTTTGATTTCCTTT*TTTTTTTTTTTTTTTTTAATAGAGATGGGGT
>PLAC8--SNX5
CATAATTTTGTTCTTATACTTTGATTTCCTT*TTTTTTTTTTTTTTTTTAATAGAGATGGGGT
>PSMD6-AS1--CCND3
TCAAATAAATAAATAAATAAATAAATAAAAG*GAAAGAAAGAAAAGAAAGAAAAGATAGCCAA
>RAB3D--VMAC
TTCTCAAGATGCAGTATGGCCTGCTGAGGTT*TTTTTTGTTTTTTTTTTTTTAAGACAGAGTC
>RAB7A--ADGRA1
AAACACAGTTACACCCCACATATCTCTCACA*CACACACACGCACACACACAAACACGTGCAT
>RAB7A--ADGRA1
ACACAGTTACACCCCACATATCTCTCACACA*CACACACACGCACACACACAAACACGTGCAT
>RAC1--NFIB
GGAACCTTTGTACGCTTTGCTCAAAAAAAAA*AAAAACAAAAAACAAAAAAAAAAAAACAAAA
>RBBP9--TRA@
CCAAATTGTAGTTTTTAAATTTGGGGAATGC*ATATATATATATATATATATACACACACACA
>RBM47--CRLF2
AGAAGAAGAAGAGGAAGAAAGAAAAGAAAGA*AAGAAAGAAAGAGAGAAAGAGAGAAAGAAAG
>RBM47--CRLF2
TTGCAAAAAAGAAGAAGAAGAGGAAGAAAGA*AAAGAAAGAAAGAAAGAAAGAAAGAAAGAAA
>RBM47--FCRL4
AAAGAAGAAGAAGAGGAAGAAAGAAAAGAAA*GAAAGAAAGAAAGAAAGAAAGAAAGAAGGAA
>RBM47--FCRL4
AAGAAGAAGAGGAAGAAAGAAAAGAAAGAAA*GAAAGAAAGAAAGAAAGAAAGAAAGAAGGAA
>RBM47--FCRL4
TTGCAAAAAAGAAGAAGAAGAGGAAGAAAGA*AAAGAAAGAAAGAAAGAAAGAAGGAAAGAAA
>RBM47--IGH@
AAAAAAGAAGAAGAAGAGGAAGAAAGAAAAG*AAAGAAAGAAAGAAAGAAAGAAGAAAGAAAG
>RBM47--IGH@
AAGAAGAAGAAGAGGAAGAAAGAAAAGAAAG*AAAGAAAGAAAGAAAGAAAGAAGAAAGAAAG
>RBM47--IGH@
AAGAAGAAGAGGAAGAAAGAAAAGAAAGAAA*GAAAGAAAGAAAGAAGAAAGAAAGAAGGAAG
>RBM47--IGH@
TTGCAAAAAAGAAGAAGAAGAGGAAGAAAGA*AAAGAAAGAAAGAAAGAAAGAAAGAAAGAAA
>RBM47--NUPR1
TAAATTATTTGCAAAAAAGAAGAAGAAGAGG*AGAAAGAGAAGAAAGAAAGAAAGAAAGAAAG
>RBM47--NUPR1
TGCAAAAAAGAAGAAGAAGAGGAAGAAAGAA*AAGAAAGAAAGAAAGAAAGAAAGAAAGAAAG
>RBM47--NUPR1
TTGCAAAAAAGAAGAAGAAGAGGAAGAAAGA*AAAGAAAGAAAGGAAGGAAGGAAAGAAAGAA
>RNF152--AC016705.1
TTTCTGACACAACTGACTTACTACACACACA*CACACACACACACACACACACACACACCACG
>RNF152--BANK1
ACCTGTTTGTTGGGGTGTGCATTCTTGTGTA*TGTGTGTTTGTGTGTGTGTGTGTGTGAGAGA
>RNF152--BANK1
TGTATGTGTGTGTGTGTGTGTGTGTGTGTGT*GAGAGAGAGAGAGAGAGAGAGAGAGAGAGAG
>RNF152--BANK1
TGTGTATGTGTGTGTGTGTGTGTGTGTGTGT*GAGAGAGAGAGAGAGAGAGAGAGAGAGAGAG
>RNF152--BCL2
GTTTGTTGGGGTGTGCATTCTTGTGTATGTG*TGTGTGTGTGTGTGTGTGTGAGAGAGAGAGA
>RNF152--BCL2
TGTGTATGTGTGTGTGTGTGTGTGTGTGTGT*GAGAGAGAGAGAGAGAGAGAGAGAGAGAGAG
>RNF152--BCL2
TGTGTATGTGTGTGTGTGTGTGTGTGTGTGT*GAGAGAGAGAGAGAGAGAGAGAGAGAGAGAT
>RNF152--BCL2
TTGTTGGGGTGTGCATTCTTGTGTATGTGTG*TGTGTGTGTGTGTGTGTGTGAGAGAGAGAGA
>RNF152--BCL2
TTGTTGGGGTGTGCATTCTTGTGTATGTGTG*TTTGTGTGTGTGTGTGTGTGTGAGAGAGAGA
>RNF152--IGK@
TTGTGTACCTGTTTGTTGGGGTGTGCATTCT*TGTGTGTGTGTGTGTGTGTGTGTGTGTGTGT
>RNF152--TAFA2
ACCTGTTTGTTGGGGTGTGCATTCTTGTGTA*TGTGTGTGTGTGTGTGTGTGTGTGTGTGAGA
>RNF152--TAFA2
GTGTATGTGTGTGTGTGTGTGTGTGTGTGTG*GAGAGAGAGAGAGAGAGAGAGAGAAAGAAAA
>RNF152--TAFA2
GTGTATGTGTGTGTGTGTGTGTGTGTGTGTG*GAGAGAGAGAGAGAGAGAGAGAGAGAGAGAT
>SAMHD1--FCRL4
AGGAAAGAGAGAAAGAAAAGAAAGAAAGAAA*GAAAGAAAGAAAGAAAGAAAGAAAGAAGGAA
>SAMHD1--FCRL4
AGGAAGGAAAGAGAGAAAGAAAAGAAAGAAA*GAAAGAAAGAAAGAAAGAAAGAAAGAAGGAA
>SAMHD1--FCRL4
AGGAGGGAAGGAAGGAAGGAAAGAGAGAAAG*AAAGGAAAGAAAGAAAGAAAGAAAGAAAGAA
>SAMHD1--FCRL4
GAAAGAAAAGAAAGAAAGAAAGAAAGAAAGA*AGGAAAGAAAGGAAGAATCTAGCTTTTATTA
>SAMHD1--FCRL4
GGAGGGAAGGAAGGAAGGAAAGAGAGAAAGA*AAAGAAAGAAAGAAAGAAAGAAAGAAAGAAA
>SAMHD1--FCRL4
GGAGGGAAGGAAGGAAGGAAAGAGAGAAAGA*AAAGAAAGAAAGAAAGAAAGAAAGAAGGAAA
>SAMHD1--FCRL4
GGAGGGAAGGAAGGAAGGAAAGAGAGAAAGA*AAGGAAAGAAAGAAAGAAAGAAAGAAAGAAA
>SAMHD1--FCRL4
GGGAAGGAAGGAAGGAAAGAGAGAAAGAAAA*AAAGAAAGAAAGAAAGAAAGAAAGAAAGAAA
>SAMHD1--FCRL4
GGGAAGGAAGGAAGGAAAGAGAGAAAGAAAA*GAAAGAAAGAAAGAAAGAAAGAAAGAAGGAA
>SAMHD1--MIR4500HG
GGGAAGGAAGGAAGGAAAGAGAGAAAGAAAA*GAAAGAAAGAAAGAAAGAAAGAAAGAAAGAG
>SAMHD1--TRA@
AAAGAGAGAAAGAAAAGAAAGAAAGAAAGAA*AGAAAGAAAGAAAGAAAGGAAGGAAGGAAGA
>SAMHD1--TRA@
AGGAGGGAAGGAAGGAAGGAAAGAGAGAAAG*AAAGAAAGAAAGAAAGAAAGAAAGAAAGAAA
>SAMHD1--TRA@
GGAAGGAAAGAGAGAAAGAAAAGAAAGAAAG*AAAGAAAGAAAGAAAGAAAGAAAGAAAAACA
>SAMHD1--TRA@
GGAGGGAAGGAAGGAAGGAAAGAGAGAAAGA*AAAGAAAGAAAGAAAGAAAGAAAGAAAGAAA
>SLC6A14--SOCS2
AATTGTGTTTGACATTCTGAGGGTTTTCTTT*TTTTTTTTTTTTTTTTTTTTTTGGTGGGGGG
>SMARCA4--RCBTB1
GTTCACTGCAGCCTCAACCTCCTGGACTCAG*CAGAGATGTCGGCTTATTTATAGGAATTGCT
>SNX1--AL139106.1
ATTGCCTCTAGGCTCTCACCATACAGTGCAA*AGATATAGATAGATAGATAGATAGATAGATA
>SNX1--AL139106.1
TGCCTCTAGGCTCTCACCATACAGTGCAAAG*ATATAGATAGATAGATAGATAGATAGATAGA
>SORL1--LRIG2
TGAGTTGCAATATGTTATTTTTATATGGGCC*AAAAAAAAAAAAAAAAAAAAAAAAAAATGGA
>SORL1--LRIG2
TGAGTTGCAATATGTTATTTTTATATGGGCC*AAAAAAAAAAAAAAAAAAAAAAAAAATGGAA
>SORL1--RGS17
TGCAATATGTTATTTTTATATGGGCCAAAAA*AAAAAACAAAAAAAAAAAAAAAGGGAAGAAG
>SORL1--SLC4A1
TTGAGTTGCAATATGTTATTTTTATATGGGC*AAAAAAAAAAAAAAAAAAAAAAAAAACGGAA
>SORL1--TFDP2
AGTGAGACTCTGTCTCAAAAAAAAGAAAAAA*AAAGAAAGAAAAATGATTAGAAATTGAGAAC
>SORL1--TFDP2
AGTTGCAATATGTTATTTTTATATGGGCCAA*AAAAAAAAAAAAAAAAAAAAAAAAGAAGAAG
>SORL1--TFDP2
TGAGTTGCAATATGTTATTTTTATATGGGCC*AAAAAAAAAAAAAAAAAAAAAAAAAAAGGTA
>SRGAP2C--IGK@
AATATTTATGGAATACGTTACTGGACCCAAT*AAATAATTTAGTTAATAACATGACAAAGAAC
>SSR1--DIAPH2
CGCTTGGCTCTCTAAATTGACTCAGCTCCAG*ATCAGAGATACTAAATCAGCGGATCAAAAAA
>TAFA2--RNF152
TGTGTGTGTGTGTGTGTGTGTGTGTGAGAGA*GAGAGAGAGAGAGAGAGAGAGAATCTCAGTC
>TFDP2--GK5
GCCTCGGCCTCCCAAGGTGCTGGGATTACAG*GCGTGAGCCACCATGCCTGGCTTGTCCTTTC
>TIMM23B-AGAP6--AC022400.4
AGAGGCTTCCTTCTTGCTGTGTCTTCGCATG*GTGAGTGGTGTCATTGAGCAAACACGAAGTC
>TMEM178B--NEAT1
ATGTGTGTGTGTGGTGTGTGTGTGTGTGTGT*GTGTGTGTGTGTGTGTGTGTGTATGATTTTC
>TMEM189--SPG7
CAGCCTGGGCGACACAGCGAGACTCTGTCTC*AAAAAAAAAATTAAGAAAGAAAAAAGCGACG
>TNFSF10--AC009088.2
TTGGCACATGCCTGTAGTCCCAGCTACTTGA*GAGGCTGAGGCAGGAGAATCGTTTGAACCCG
>TRA@--SAMHD1
CTGGGTGACAGAGTGAGACTCTATCTCAAAA*AAAAAAAAAAGCACAAACACTGCACTGCCTG
>TRB@--AC092683.1
AGACATTTATGCTGCCAACAAACATATGAAA*AAAAAAAAAGCTCAACACTAGTGTTTATTAG
>TRB@--PPIL6
CAGCCTGGCGACAGAGAGAGACTCTGTCTCA*AAAAAAAAAGAAAGGCCAAGTACTGCAGAGA
>TRB@--PPIL6
TTGCACTCCAGCCTGGCGACAGAGAGAGACT*CTGTCTCAAAAAAAAAGAAAGGCCAAGTACT
>TRIOBP--DGKB
ACACACACACACACACACACACACACACACA*CACACACACACATACACACACACTCACACAC
>TRIOBP--DGKB
ACACACACACACACACACACACACACACACA*CACACACACATACACACGCACACACACGCAC
>TRIOBP--DGKB
ACATACGCACACACGTGCACACATGTACACA*CTGATACACACACACACACACACACACACAC
>TRIOBP--DGKB
ACCCACCACACACACACACACACACACACAC*ACACACACACACACACACACACACACACACA
>TRIOBP--DGKB
ACGTGCACTTGCACCCACCACACACACACAC*ACACACACACACACACACACACACACACACA
>TRIOBP--DGKB
ATACGCACACACGTGCACACATGTACACACG*ACACACACACACACACACACACACACACACT
>TRIOBP--DGKB
ATACGCACACACGTGCACACATGTACACACG*ACACACACACACACACACACACACACACTGC
>TRIOBP--DGKB
CGCACACACGTGCACACATGTACACACGGAT*ATACACACACACACACACACACACACACACA
>TRIOBP--DGKB
CGCACACACGTGCACACATGTACACACGGAT*GCACACACACACACACACACACACACACACA
>TRIOBP--DGKB
GCCTGGGTGACAGAGCGAGACTCTGTCTCAA*AAAAAAAAAAAGGAAAATGTACTTTTCATGG
>TRIOBP--DGKB
TACGCACACACGTGCACACATGTACACACGG*ACACACACACACACACACACACACACACTGC
>TUBA4B--DNAJB2
GACCCCAGCCAGCCCCTGTCCAGGCAGCATG*CCCCAAGGAGGCCCGCCTGACGACTGACCAG
>UXT-AS1--GK5
AGTTCGCCTAACTCGGCCTCCCAGGGTGCTG*GGATTACAGGCGTGAGCCACCATGCCTGGCT
>WWOX--NEAT1
AAAAAAAAAAAAAAAAAAAAAAACCAAAAAA*AAAAGGAATGAATTAGAACTTCTTCTGCTTG
>WWOX--NEAT1
CATCTCTACTAAAAATACAAAAATTAGCCAG*GCATGGTGGTGCACGTCTGTAATACCAGCTT
>WWOX--NEAT1
GTCTCTACTAAAAAAAAAAACAAAAACCACA*AAAAAAAAAACAAAACAGAAAAAATAAACTA
>WWOX--NEAT1
GTGTGTGTGTGTGGTGTGTGTGTGTGTGTGT*GTGTGTGTGTGTGTGTGTGTGTATGATTTTC
>WWOX--NEAT1
TCATATATATATATGTGTGTGTGTGTGTGTA*TTTTTTTTTTTAATTTCAATGGCTTTTGGGG
>ZFHX3--NEAT1
AGCCTGGGTGACAGAGCAAGATTCCGTCTCA*AAAAAAAAAAAAAAAAAAAAAACCAAGAAGA
>ZFHX3--NEAT1
GAAGATCAGTCGTGTGTGTGTGTGTGTGCGT*GTGTGTGTGTGTGTGTGTGTGTATGATTTTC
>ZNF268--USP9X
GCCTCAGCCTCCTGAGTAGCTGGGATTACAG*GTTATGCAATGGTCTCTGCAAGATGGTTTAT
>ZNF28--ZNF765
GTGGAATGACGGTCACGCCGCGGCGGGCG*GATTGACTTCTAAAGACTCTTGGTATGTGAG
>ZNF28--ZNF845
AGCAAAGGAGTCAGGGATGGCTCTTCCTCAG*GGTCTATTGACATTCAGGGATGTGGCCATAG
>ZNF28--ZNF845
CCGTGGAATGACGGTGACGCCGCGGCGGGCG*GATTGACTTCTAAAGACTCTTGGTACGTGAG
>ZNF28--ZNF845
CTGGAGAATTATAGGAACCTGGTCTCCCTGG*ATATCTCTTCCAAATGCATGATGAAGGAGTT
>ZNF28--ZNF845
GTGGAATGACGGTCACGCCGCGGCGGGCG*GATTGACTTCTAAAGACTCTTGGTACGTGAG
>ZNF451--PAX5
TCCCCTTCCCTTTGCCTTTTAAATTGTGGTT*AAAAAAAAAAGAAAAAAAAAGGCACAAAAAA
>ZNF451--PAX5
TTCCCCTTCCCTTTGCCTTTTAAATTGTGGT*AAAAAAAAAAGAAAAAAAAAGGCACAAAAAA
>ZNF587B--ZNF880
AGTCATCAGCGCATTCACACTGGAGAGAGAC*CTTACAAATGTAATGAATGTGGCAAAGCATT
>ZNF7--MPV17L
CAGCCTGGGTGACAGAGTGAGACTCCCTCTC*AAAAAAAAAAGTTTCAGTAAATTCCACCTAA
>ZNF7--MPV17L
CAGCCTGGGTGACAGAGTGAGACTCCGTCTC*AAAAAAAAAGTTTCAGTAAATTCCACCTAAG
>ZNF765--ZNF845
ACGGCGTGGAGTGACTATCCCACCGCCGCGG*GATTGACTTCTAAAGACTCTTGGTACGTGAG
>ZNF845--ZNF28
AGCAAAGGAGTCAGGGATGGCTCTTTCTCAG*GGTCTATTGACATTCAGGGACGTGGCCATAG
>ZNF845--ZNF28
CTGGAGAATTATAGGAACCTGGTCTCCCTGG*ATATCTCTTCCAAATGCATGATGAAGACATT
>ZNF845--ZNF765
GATCGCATGGAGTGATGGTCCCACCGCAGCG*GATTGACTTCTAAAGACTCTTGGTATGTGAG
>AC091390.3--PMS2
GAAGGTGAATTTCCATGTCAACATAGTCCAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>AC093307.1--KIAA1217
TGTGATCTTGGACTTCCCAACCTCCAGAACA*AACAAGGCAAAGGCAATCTGCATGTAACATC
>AC140479.3--MALL
AGAGTCTGAGTTGCTTCCCCTGCCTTTCTTG*ATATTTGGGTTCTTGGTCTGGACCATGGTAG
>ACTB--PECAM1
ACCATGGCCTCCCAAAGTGCTGGGATTACAG*ATCTAGGAAAGAAGGACACAGAGACAGTGTA
>ACTB--SDHA
ACCAGTTGAATAAAAGTGCACACCTTAAAAA*AAAAAAAAAAAAAAAAAAAAAAAACAGAGAA
>ACTB--SDHA
GACCAGTTGAATAAAAGTGCACACCTTAAAA*AAAAAAAAAAAAAAAAAAAAAAAACAGAGAA
>ACTB--SDHA
TTGAGACCAGTTGAATAAAAGTGCACACCTT*AAAAAAAAAAAAAAAAAAAAAAAACAGAGAA
>ACTB--STAT6
ACCAGTTGAATAAAAGTGCACACCTTAAAAA*AAAAAAAAAAAAAAAAAAAAAAAAAGGAAAG
>ACTB--STAT6
GACCAGTTGAATAAAAGTGCACACCTTAAAA*AAAAAAAAAAAAAAAAAAAAAAAAAGGAAAG
>ACTB--STAT6
TTGAGACCAGTTGAATAAAAGTGCACACCTT*AAAAAAAAAAAAAAAAAAAAAAAAAGGAAAG
>ACTB--ZNNT1
TGGCTTTATTTGTTTTTTTTGTTTTGTTTTG*TTTTTTTTTTTTTTTTTTGGCTTCCCGTTGT
>ADGRG1--MYH11
GCCTCAGCCTCCCGAGTAGCTGGGATTACAG*AGGGCCCCCCCCACAGGAAACTTCGCAGTGA
>AP3B1--TG
TTAGCAGCGATTTGAAGAAGTGAGGAGAAAG*GAGGAGAGGTGACATGTCTCACTCTGAACAG
>ATP1A1-AS1--HSP90B1
TCCTGGTCTTACAGATGACCAAGTCGGCCCA*TGTTGAGGAGCCCATGGAGGAAGAAGAAGCA
>B2M--ERBB3
TTCCACAAGTTAAATAAATCATAAAACTTGA*AAAAAAAAAAAATTAGCCAGGCGTGGCGGCA
>BLNK--NPM1
CTTTCTCTCCCACCTAAGTGTGTGCCGCCAC*CCGATGGAAGATTCGATGGACATGGACATGA
>C7--EEF1A1
GGAGATTGAACCACTAAAATGTTAGAGCAGA*GCGATTCTTGTGCCTCAGCCACCTGAGTAGC
>CCDC9B--AC089984.2
GAGAAAATTAAAGATAAGGGTATCCAAAATA*GTCTACGGCCATACCACCCTGAATGCACCTG
>CHMP5--AC114402.1
GCAATTCCAGAAGGTGTTCCCACTGATACAA*AAAAACAAGGATGGAGTTCTGGTGGATGAAT
>CIRBP--ATRX
GGTTTTTGTTTTAGTTTTTGGTTGCGTTGCC*TTTTTTTTTTTTTTTTTTTACAACTTTTAAC
>COPDA1--IGHG2
CCAAAACAGGAAGGGCTGAGACCCCCACCTC*GCCTCCACCAAGGGCCCATCGGTCTTCCCCC
>CSNK1G1--TG
TCGAGGCAGCCTCCCCTGGCAAGGACTCAAG*TTGCTCAAAATAATGCTCCCAGTTTTTGCCC
>EEF1A1--DDX24
TTGAGACCCATTAAAAAAGTTAAATGAGAAA*AAAAAAAAAAAAAAAAAAAAGAAATTACCTA
>EML1--TFAM
AATGAGACCCTTCCTCAAAAAAAAAAAAAAA*GAGTTAACACTGCTTGGAAAACCAAAAAGAC
>ETV6--SPRY4
CACACACACACACACACACACACACACACCC*ACACACACCCCTACCCCACACTGACTGTCTA
>FASN--DEPP1
GTCACGGACGATGACCGTCGCTGGAAGGCGG*ACACTGCTCAGGGAAGAGCCTGCTACGGTGG
>FASN--EEF1A1
GGCGAGCGCCCGCTCTGGTTCATCTGCTCTG*GTGATTATCCTGAACCATCCAGGCCAAATAA
>GLG1--HNRNPA2B1
CAGGCAGCTCCGTGTGGAGGAGCTGGAGATG*GCAACTTTGGCTTTGGGGATTCACGTGGTGG
>GRB2--NME6
CCCCGTCTCTACTAAAAATAAGAAATTAGCC*GGGTGTGGTGGCACGCACCTGTAGTCCCAGC
>IGHM--UNC13D
TAAAATTAGAAATAAAAAGATCCATTCAAAA*AAAAAAAAAAAAAAAAAAAAAGGGATGCGGC
>LARS1--TAT
CTGGGTTCAAGCGATTCTCCTGCCTCAACCT*CAAATTTAGATTTGTGTGCATCAGGCTTTCT
>LHFPL5--MYH11
GCCTCAGCCTCCCGAGTAGCTGGGATTACAG*AGGGCCCCCCCCACAGGAAACTTCGCAGTGA
>LINC02379--RNF212B
GGAGGAAGTAACTTGTGGAATGCTGAGAAAG*GTCAATCACACCTCGACCAGTGGGCATTACT
>LINC02379--RNF212B
GGAGGAAGTAACTTGTGGAATGCTGAGAAAG*TTACCCCACGACCCAGTTTCCAGCATAGCAG
>MAPK6--PECAM1
GCCCAGCACTCCCAAAGTGCTGGGATTACAG*ATCTAGGAAAGAAGGACACAGAGACAGTGTA
>MGRN1--PPP1R1A
CCTGCTCCTTAAAACTAAAAGAAAAAAAAAA*AAACTGCAGAATGCATCCCTAAAACTCACGA
>MICOS10--TG
GCTTATTCCAACTGTCAGCATGATTTCCAGG*AAATCCCAACTACCCTTATGAGTTCTCACGG
>MKNK2--FGFR2
GCCATCGGGTCTCTAAAGCTGAGTGGCTAGC*CGTTTGTTTGTTTGTTTGTTTGTTTTTTTTG
>MRPS31P5--THSD1
AGAGTATATTTAAAATTTTTATTAAGGCCCA*TGGGGCAGCCCTTGAGGAAACCAGACCTTGG
>MYH11--SERF2
GCGAGGTGAACGCACTCAAGAGCAAGCTCAG*ACGAAGTCTTGCTCTGTCGCCCAGGCTGGAG
>OBSCN--TG
CTCTGCAAGGCAGAGCTGCTGGTGCTTGGGG*AAGCAGTGGTTTCTCCTCCTTCCTCCCAGGA
>PDE4D--CACNG2
ACTACTTCAGCTACTTGCATTTGATTATTTC*TTTTTTTTTTTTTTTTTAACTCGGGGAGAGT
>PPP2R5A--ZNF680
GCCTCAGCCTCCCGAGTAGCTGGGCTTACAG*TTATATATTCTCATTTCACTGAAGACCTTTG
>RAB7A--NEK9
ATCAAACACAGTTACACCCCACATATCTCTC*ACACACACACACACACACGCACACGCACACA
>RANBP3--TG
TCCGTGGCTCCCTGACATTCTCCAGAGCCAG*GGACTACTTTATCATCTGCCCTATAATCGAC
>RFC2--TG
GCAGAATACCTGAAACTGGAGTTTATCAAGG*ATTCTGACATGGGGTCTCGGCCTGAGTCTAT
>SCD--PDXDC1
TTTGTTCATAATAAAAGTGAAGTGAATCTGA*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>SENP3--TG
TAAAACAAGATTTAAAAAAAAAAAGAAAAAG*CAATTTGAGGAAAGTCGAGGCCGGACCAGTA
>SERPINA1--SCP2
CCTGGATGACATTAAAGAAGGGTTGAGCTGG*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>SPOP--HBM
TTTATTTCTTCTCCCCCACCCCTTTTTCTAA*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>TG--DDX23
TGACAATGTTGCCTGCATGACTTCTGACCAG*TAGAGATGGGGTTTCACCGTGTTGGCCAGGA
>TG--DNAJA1
TGACAATGTTGCCTGCATGACTTCTGACCAG*TAGAGATGGGGTTTCACCATGTTGGCCAGGC
>TG--DYSF
TTGCGTCTTGGAGACCAGGAGTTCATCAAGA*ATCCTGGCATGGGGCCTGCGGAACATGAAGA
>TG--EML2
TGACAATGTTGCCTGCATGACTTCTGACCAG*TAGAGATGGGGTTTCACCATCTTGGCCAGGC
>TG--GCLC
CTCCCAACTTTTCCGGAGAGCTGTGCTGATG*CAATGTGAGGAAAGTGGAATGAACTTGGACT
>TG--OBSCN
TCTGAGTTCCCCGTGATGCAGTGCTTGACAG*GTCTGAGACCCTCAGAGGTGGGGACAGATAC
>TG--PKD1
GAGGCCCCTCTTGAGGACTCACAGTGTTTGA*TGCTGCAGCCAGAGGGAAAGGCGTCCACCAA
>TG--TAPBP
TGACAATGTTGCCTGCATGACTTCTGACCAG*TAGAGATGGGGTTTCACCATGTTGGCCAGGC
>TMEM8B--TXN2
GCCTCGGCCTCCCAAGGTGCTGGGATTACAG*GTGTCAGCGGTGCCCACTGTGCTGGCCATGA
>TMSB4X--STAT6
TTAATATGCAAATAAAAAGTTTAAAAACTTA*AAAAAAAAAAAAAAAAAAAAAAAAAGGAAAG
>TRMT11--CENPW
AAAGAGAAAATGACAGAAATGGAGCACCCTG*GTCCATCTGAACTGTTTACTGTTTGTTCATC
>TTN--MYH11
GAGGAGGAACCAGAGGTTCCACCTCCAGCAG*GACCTTGAAGAACAGCTGGAGGAGGAGGAAG
>ABR--NXN
GGACAAGATCATGGGCAAAGGACAGATCCAG*ATTCTGAGGATGACGGAGAGTCCGAGGCGGC
>AC006064.6--LYZ
TCTCGATCTCCTGACCTTGTGATCCACCCAC*CTCAGCCTCCCGAGTAGCTGGGATTACGGGC
>AC008993.1--MIR1302-9HG
ACTTTTAGAAATCACATATGTGACTTAAATA*TTTCTTTTCTTTTTCTTTCCTCTCACTCAGC
>AC008993.1--MIR1302-9HG
GTGACTTAAATATTTCTTTTCTTTTTCTTTC*CTCTCACTCAGCGTCCTGTGATTCCAAAGAA
>AC008993.1--MIR1302-9HG
TGTGACTTAAATATTTCTTTTCTTTTTCTTT*CTCTCACTCAGCGTCCTGTGATTCCAAAGAA
>AC019205.1--EIF3E
AATTCTCTAGCACAGACTCCCTTTTTTTGGT*AAGATGGCGGAGTACGACTTGACTACTCGCA
>AC019205.1--EIF3E
CTCTAGCACAGACTCCCTTTTTTTGGTAAGA*TGGCGGAGTACGACTTGACTACTCGCATCGC
>AC020656.1--POLH
GAGGTGGAGGTTGCAGTGAGCCGAGATTGCG*CACTGCAATCTCCACCTCCCGGGTTCACGCC
>AC020656.1--POLH
GCACTTTGGGAGGCCGAGGTGGGTGGATCAC*GCGTGAGCCACTGCGCCCGGCCAGGACTGAC
>AC020656.1--POLH
GCCTGTAATCCCAGCACTTTGGGAGGCCGAG*GCGGGCGGATCACAAGGTCAGGAGATCGAGA
>AC020656.1--POLH
GTCTGGGTTATTTCTTTGATAAGAACTGAAT*TGGCCGGGCGCAGTGGCTCACGCCTGCAATC
>AC020656.1--RPL37
GTGGATCACAAGGTCAGGAGATCGAGACCAT*CCTGGCCAACATGGCGAAACCCCGTCTCCAC
>AC020656.1--XACT
CGTGGTGGCGGGCGCCCGTAATCCCAGCTAC*CGGGAGGCTGAGGCAGGAGAAGGGTGTGAAC
>AC020656.1--XACT
CGTGGTGGCGGGCGCCCGTAATCCCAGCTAC*CGGGAGGCTGAGGCAGGAGAATCACTTGAAC
>AC020656.1--XACT
CGTGGTGGCGGGCGCCCGTAATCCCAGCTAC*CGGGAGGCTGAGGCAGGAGAATGGCATGAGC
>AC020656.1--XACT
GCGGGCGCCCGTAATCCCAGCTACTCGGGAG*GGCTGAGGCAGGAGAATGGCGTGAACCCGGG
>AC020656.1--XACT
GCGGGCGCCCGTAATCCCAGCTACTCGGGAG*GGCTGAGGCAGGAGAATGGCGTGAACCGGGG
>AC020656.1--XACT
GCGGGCGCCCGTAATCCCAGCTACTCGGGAG*GGCTGAGGCAGGAGAATGGCGTGAACCTGGG
>AC020907.3--SYT11
TCTCTATCAAAAATACAAAAAAATTAGCCAG*GTGTGGTGCTGCACGCCTGTGGTCCCAGCTA
>AC022210.1--KCMF1
GCTTTTCAGAGGGCTTCTACAGAGCCCACAG*GTGTCAGCTGTGATGCATGTTTAAAAGGAAA
>AC083899.1--AC133644.2
TGGCTATGATCTACTTAAAAACCAATAACAG*GCTACAGAGTGCGGTGGCGCCAGCACAGAAC
>AC083899.1--AC133644.2
TGGCTATGATCTACTTAAAAACCAATAACAG*TGTCTTCTAGGAAGCCCAGACACATGGAGAA
>AC091390.3--AC004980.4
AATGGAAGGTGAATTTCCATGTCAACATAGT*ACAGAACCTGCTAAGGCCATCAAACCTATTG
>AC091390.3--AC004980.4
GAAGGTGAATTTCCATGTCAACATAGTCCAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>AC091390.3--AC211429.1
CCAAGAGCAGCAGAAAATGAACACATTGCAG*GGGCCAGTGTCATTCAAAGATGTGGCTGTGG
>AC091390.3--PMS2
GAAGGTGAATTTCCATGTCAACATAGTCCAG*ATCTAAAGCTTAAGGACTATGGAGTGGATCT
>AC091390.3--PMS2
GAAGGTGAATTTCCATGTCAACATAGTCCAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>AC091390.3--POM121C
CCAAGAGCAGCAGAAAATGAACACATTGCAG*GGGCCAGTGTCATTCAAAGATGTGGCTGTGG
>AC093787.2--AC233263.7
TACAATCCACAGGCAAGGATGTGGCGATTCC*TTTTGCAATATTGGAGGGAATGCCAAACCCT
>AC138123.2--CAP1
CCCTCCAGTCCAGAAAAAAAAAAAAAAAAAC*AAACCCACATGATTTCAAGGAGTCTGGCATT
>AC138409.2--AC138866.1
CGAACACCCCGCCTACCTGTACTCGTGGGAG*GATGGTGATTGCTCACACCAAAGCCTTGGAC
>AC138409.2--AC138866.1
GTAAAACCAGCCTTAGACTATTTTTCAAACA*GATGGTGATTGCTCACACCAAAGCCTTGGAC
>AC138409.2--AC138866.1
GTAAAACCAGCCTTAGACTATTTTTCAAACA*GGCTGGTGAATTACCAGATCTCCGTCAAGTG
>AC138409.2--NAIP
TGGAATTTTGCCGATTTCATGACTAACCAGT*ATGTGAATTTCTTCGGAGTAAGAAATCCTCA
>AC138866.1--AC138409.2
TGGAGTGGAAACGCTTGTAGGGCTTCACCAG*GGCTGGTGAATTACCATATCTCCGTCAAGTG
>AC138866.1--NAIP
TGGAATTTTGCCGATTTCATGACTAACCAGT*ATGTGAATTTCTTCGGAGTAAGAAATCCTCA
>AC211429.1--AC091390.3
CCAAGAGCAGCAGAAAATGAACACATTGCAG*GGGCCAGTGTCATTCAAAGATGTGGCTGTGG
>AC211476.4--PMS2
AACAGTCTGGATGCTGGTGCCACTAATATTG*ATCTAAAGCTTAAGGACTATGGAGTGGATCT
>AC211476.4--PMS2
CAGCCATGCTTCCTGTACAGCCTGCAAAACT*TACAGAACCTGCTAAGGCCATCAAACCTATT
>AC211476.4--PMS2
CCATAAGGAATTTCAAAGGAATATTAAGAAG*AACCTGCTAAGGCCATCAAACCTATTGATCG
>AC211476.4--PMS2
CCATAAGGAATTTCAAAGGAATATTAAGAAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>AC211476.4--PMS2
GCCGGTGGTACTGAGTCTAAGCACTGCGGTG*AAGGAGTTAGTAGAAAACAGTCTGGATGCTG
>AC211476.4--PMS2
GGAAAGCTCTGAGCTCACTTTGTGCACTGAG*CGATGTCACCATTTCTACCTGCCACGCATCG
>AC211476.4--PMS2
GGGGTAGAAGAAGAAAACTTCGAAGGCTTAA*CGATGTCACCATTTCTACCTGCCACGCATCG
>AC211476.4--PMS2
GGGGTAGAAGAAGAAAACTTCGAAGGCTTAA*CTCTGAAACATCACACATCTAAGATTCAAGA
>AC211476.5--PMS2
GCCGGTGGTACTGAGTCTAAGCACTGCGGTG*AAGGAGTTAGTAGAAAACAGTCTGGATGCTG
>AC211476.5--PMS2
GGGGTAGAAGAAGAAAACTTCGAAGGCTTAA*CTCTGAAACATCACACATCTAAGATTCAAGA
>AC239859.5--LINC02802
GGGAGGCTCCAAGACCTCATCAACAATTAAG*ACTCTTTGGCAGTTGCACTGGGTACTGGCTG
>AC239859.5--LINC02802
TCAGTCTGCGGCCAAGACTCCGTCCTGCAAG*ACTCTTTGGCAGTTGCACTGGGTACTGGCTG
>AC239859.5--LINC02802
TCAGTCTGCGGCCAAGACTCCGTCCTGCAAG*GATATTGTGGCTGGAGACATGAGCAAGAAAA
>AC245748.1--ZNF806
CCAAAAGGAAGACGGTATGATTAAGTTCCAG*GAAAGGGTGACATTCAAGGACATAGCTGTTA
>ACBD5--SMCHD1
GCCTCAGCCTCCCAAGTAGCTGGGACTATAG*TCCATTCCAGTGATTAATGGAAGAGATTTAC
>ACTB--HMGB1
CGATCCGCCGCCCGTCCACACCCGCCGCCAG*AAAAATAACTAAACATGGGCAAAGGAGATCC
>AL117335.1--AL109809.4
CTGCCCCAAGAGTTGCTGCTCACCAGGCTGG*AAGGATTTGCTGAGCACCTACTATGCATCAG
>AL121578.1--SYTL5
ATTTCCTACATCTGAGGCATCTGTGTAAAAG*TGTTTTCTGCATTCTCTCCCAGCGGATCTGA
>AL121578.1--SYTL5
GGGTGAGCCACCAAACCAACCTGACATTGAG*TGTTTTCTGCATTCTCTCCCAGCGGATCTGA
>AL138963.3--HBA2
CATCAATGTTACCTTCTGTCCTACTGACCAT*CTTCCCCACCACCAAGACCTACTTCCCGCAC
>AL158066.1--THSD1
AGAGTATATTTAAAATTTTTATTAAGGCCCA*TGGGGCAGCCCTTGAGGAAACCAGACCTTGG
>AL512662.1--BMS1
AGCGGACCTGAGGAGTTGCAGCTGTGAGCAG*GTTAGAGTTACTTGTTATTGGTAAATAGCCA
>ANAPC1--AC133644.2
TGGCTATGATCTACTTAAAAACCAATAACAG*GCTACAGAGTGCGGTGGCGCCAGCACAGAAC
>ANAPC1--AC133644.2
TGGCTATGATCTACTTAAAAACCAATAACAG*TGTCTTCTAGGAAGCCCAGACACATGGAGAA
>ANAPC1--ANAPC1P2
AAGAAATGTTTTGAAATAGAGCTTGAGATTC*TTGTCTTGCCCTCTACTTGAACAGTGTCTTC
>ANAPC1--ANAPC1P2
TGGCTATGATCTACTTAAAAACCAATAACAG*TGTCTTCTAGGAAGCCCAGACACATGGAGAA
>ANAPC1P2--ANAPC1
TGTTTCTCGGGTTGGAAAAGCCATTGATAAG*GATTCACTTTAAGAGATTTGGAAACTCTTCC
>ANXA1--RBM39
CAAAGCAGCATATCTCCAGGAAACAGGAAAG*GATGAGAACAAGTTGAGCAGTGCCAACGGCC
>ARF1--MCM3AP
CAACGCCTGGCTCGGAGCAGCAGCCTCTGAG*CTTGTTCTAAAGACGAAGGGGAACTCCCTAG
>ARHGAP11B--AC091057.1
AGTATTTGCGGATAGCTCCAAACTGGGATAT*TGTTCTCAGGCGCTGTGGTCTCACCTTCCAC
>AZU1--ZNF540
GATTGAGCATTTTCACGGTAGGAGAAACAGT*ATCTTTTTTTTTTTTTTGAGACAGAGTTTAG
>B2M--DDX5
CTCTCTCTTTCTGGCCTGGAGGCTATCCAGC*GTTTGGTGCACCTCGATTTGGAGGAAGTAGG
>BCL11B--PIGR
ATCTCGGCTCACTGCAACCTCCGCCTCCCGG*GTTGACACCATTCTCCTGCCTCACCCTCCCA
>BCL11B--RPL3
AAGCGGGTGCTCACCCTCCGCAAGTCCTTGC*TGGTGGGAACCAAGAAGCGGGTGCTCACCCT
>BCL11B--RPL3
ACTATGGTGAAGTGACCAATGACTTTGTCAT*GCAGAAAGGCTACCATCACCGCACTGAGATC
>BCL11B--RPL3
AGAAGATTGACCTTAAGTTCATTGACACCAC*ACGGCAAGCTGATCAAGAACAATGCCTCCAC
>BCL11B--RPL3
ATGGAGGAGAAGAAAGCATTCATGGGACCAC*GAAGATTGACCTTAAGTTCATTGACACCACC
>BCL11B--RPL3
CAAGAAGATCTATAAGATTGGCCAGGGCTAC*CATCACCGCACTGAGATCAACAAGAAGGTGA
>BCL11B--RPL3
CACCGCACTGAGATCAACAAGAAGATCTATA*AGAAAGGCTACCATCACCGCACTGAGATCAA
>BCL11B--RPL3
CACTGAGATCAACAAGAAGATCTATAAGATT*CAGGGCTACCTTATCAAGGACGGCAAGCTGA
>BCL11B--RPL3
CAGGAAGGCTGTGAGGTGGACCGGCTTGGAC*CATTGTAGAGACACCACCCATGGTGGTTGTG
>BCL11B--RPL3
CATGGAGGAGAAGAAAGCATTCATGGGACCA*AAGATTGACCTTAAGTTCATTGACACCACCT
>BCL11B--RPL3
CCATGGAGGAGAAGAAAGCATTCATGGGACC*GAAGATTGACCTTAAGTTCATTGACACCACC
>BCL11B--RPL3
CTCTGGGTGGCTTTGTCCACTATGGTGAAGT*GAACAATGCCTCCACTGACTATGACCTATCT
>BCL11B--RPL3
CTCTTCACCTTCCCACGATGCCTGCTGCTGC*GCTTCCTGCCTCGGAAGCGCAGCAGCAGGCA
>BCL11B--RPL3
CTGCTGCTGCGCTTCCGAGGCAGGAAGCCGA*GGGAAGGTGAAGAGCTTCCCTAAGGATGACC
>BCL11B--RPL3
GACGGGTCATCCTTAGGGAAGCTCTTCACCT*TCCCTCGGCTTCCTGCCTCGGAAGCGCAGCA
>BCL11B--RPL3
GCCTCCACTGACTATGACCTATCTGACAAGA*AGATTTATAAGATTGGCCAGGGCTACCTTAT
>BCL11B--RPL3
GCTGAAAGGCTGTGTGGTGGGAACCAAGAAG*ATTTATAAGATTGGCCAGGGCTACCTTATCA
>BCL11B--RPL3
GGGAGCGGAGAACTTTCTGTGAGACATCACG*CATGGTGGTTGTGGGCATTGTGGGCTACGTG
>BCL11B--RPL3
GGGTCATCCTTAGGGAAGCTCTTCACCTTCC*CTCGGAAGCGCAGCAGCAGGCATCGTGGGAA
>BCL11B--RPL3
GGTGCAGACGAAGCGGCGGGCTCTGGAGAAG*ATTTATAAGATTGGCCAGGGCTACCTTATCA
>BCL11B--RPL3
GTCTGGGAGCGGAGAACTTTCTGTGAGACAT*CACCCATGGTGGTTGTGGGCATTGTGGGCTA
>BCL11B--RPL3
TGAGATCAACAAGAAGATCTATAAGATTGGC*CAGAAAGGCTACCATCACCGCACTGAGATCA
>BCL11B--RPL3
TGGGACCACTGAAGAAAGACCGAATTGCAAA*AGAAGATTGACCTTAAGTTCATTGACACCAC
>BCL11B--RPL3
TGTAGTCATGCCAGCCTTGTATCCCAGGAAG*GCTGTGACCATTGTAGAGACACCACCCATGG
>BCL11B--RPL3
TTCCCACGATGCCTGCTGCTGCGCTTCCGAG*GGAAGGTGAAGAGCTTCCCTAAGGATGACCC
>BCLAF1--LGALSL
AAATTCCCGGAACCGCGATTCGGCGTGTCAG*AAACTAGATGATGGCCATTTAAACAACTCTT
>BHLHE40--LTF
ATCTGAGGCATGGAGAGCAGATTCAGGGTGT*GTGTGTGTGTGTGTGTGTGTGTGTGTGTGTG
>BRD2--RACK1
TGGTTTTTTGGTGTGGTTTTGTGTTTTGTTT*TTTTTTTGTTTGTTTGTTTTTTGTTTTTTTT
>CAP1--AC138123.2
CCCAGCACCTTCCTATAGAGATGACTTTAAA*AGAAAAAAAAAAAAAAAAACAAACCCAGTAG
>CD24--LINC02321
ATTTGCCTCCCGGGCTCAAGCGATTCTCCTG*CCTGTAATCCCAGCTACTTGGGAGGCTGAGG
>CD24--LINC02321
GCTGGTTTCGAACTCCTGACCTCAGGTGATC*CCACCCGCCTCGGCCTCCCAAAGTGCTTGGG
>COPB2--RNF130
GCCTCGTCCTCCCAAAGTGCTGGGATTATAG*GTGTGAGCCGCACCAGCCTGTTTTTTAATTT
>CTSS--ACAA1
GTAGCTGGAATTATAGGCGTGCACCACCACG*CCCAGCTAATTTTTGTATTTTTGGTGGAGAC
>DA750114--CWF19L1
TTTTGAGACGGGGTCTCGCTCTGTCACCCAG*GCAGGATTGCAGTGGCATAACTGTGGCTCAC
>DDX17--EEF1A1
CCAAAATGATCCAGTCTGGAATAACATTTTG*AAAAAAAAAATATATATATATATATATATAG
>DDX5--ANXA1
GCCATTCCTAGAGAGAGGCGATGGGCCTATT*TGAAGTGCGCCACAAGCAAACCAGCTTTCTT
>DSG3--SYNRG
AATGATGTAAGAATCACAAGGGGCCGGGCGC*GGTGGCTCACGCCTGTAATCCCCAGCACTTT
>DTX2P1-UPK3BP1-PMS2P11--AC004951.1
CACATCCTCCACGTCCCCAGGCCCCACCCAC*CTCCTCCAACTCCTCTGGGGAAACCCAAGCC
>DTX2P1-UPK3BP1-PMS2P11--AC004951.1
CAGCCATGCTTCCTGTACAGCCTGCAAAACT*AGTAAAGAGGACAAATAGGTGAAAGAATAAA
>EDEM3--MNDA
TACAAATCAAAGCATCTCTAAAAAGAACACA*AATCAGGAAACCCAGGCCCAACGGCAGGTGG
>EEF1A1--CHMP3
CAAAAGAACCATTGAAAAATTTGAGAAGGAG*GCTGGGATCATAGAGGAGATGTTAGAGGACA
>EEF1A1--SFPQ
AGCCTGGGCGACAGAGCGAGACTCTGTCTCA*AAAAAAAAAAAAAGGTTTAGTACTGAAGTAG
>EEF1A1--SFPQ
CCTGGGCGACAGAGCGAGACTCTGTCTCAAA*AAAAAAAAAAAAAGGTTTAGTACTGAAGTAG
>EIF3E--AC019205.1
TGGGCAACTCAAGATTCTGGCTTCTACTGAA*GAAACATAAGGAAAAGATTTTAAAAAAACTC
>ELANE--EEF1A1
GCCTGTGTCCTGCCGGCCTTGCTGCTGGGGG*GTGTCGTGAAAACTACCCCTAAAAGCCAAAA
>ETS2--ATP8B4
AAGGCCTGTGGGGGGTAATCCCTGCTTTTTG*TTTTTTTTGTTTGTTTGTTTGTTTGTTTTTG
>ETS2--ATP8B4
TAAGGCCTGTGGGGGGTAATCCCTGCTTTTT*TGTTTTTTTTGTTTGTTTGTTTGTTTGTTTT
>FAT4--LYZ
GGTTCACGCCATTCTCCTGCCTCAGCCTCCC*AGCAAAATATGGCCTTTCTTCTAAGAGATAT
>FCAR--MARK3
ATTTCTACCGGATTTTCATACAAGGAATACA*GGCGTGGTGGCAGGTGCCTGTAGTCCCAGCT
>FCAR--MARK3
GAGAAAAGCTCACTAAGAAGCTTGAATCTAC*TTTTTTTTTTTTTGAGACAGAGTCTGGCTCT
>FOS--AKAP17A
GAGCATTGGCAGGAGGGGCAAGGTGGAACAG*GCCCAAGGTCCCGGAGGCTATGGCAGCGGCT
>FOSB--HMGB1
CGGCAGTCCACCCACCGCCGCCGCCTCCCAG*AAAAATAACTAAACATGGGCAAAGGAGATCC
>FUS--EEF1A1
GTGCGCGCGTGCGCGGACATGGCCTCAAACG*GTGTCGTGAAAACTACCCCTAAAAGCCAAAA
>FUT4--AP000943.3
ACTGTGCATCTCCTTGACTGCCGCATCATGG*ATAACTGAGAAGCGGAGCTACTCACTTCGGA
>HBB--EEF1G
ACTTTGGCAAAGAATTCACCCCACCAGTGCA*GGCTGCCCCTGCTCCTGAGGAGGAGATGGAT
>HBB--GPR160
GCCTAATAAAAAACATTTATTTTCATTGCAA*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>HBB--PLEKHB2
GCCTAATAAAAAACATTTATTTTCATTGCAA*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>HBB--PLEKHB2
TCTGCCTAATAAAAAACATTTATTTTCATTG*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>HIF1A--CALM1
AGGGCGCCGGCGGCGCGAACGACAAGAAAAA*GCTGATCAGCTGACCGAAGAACAGATTGCTG
>HNRNPC--EEF1A1
TGATGATAATGAAGATCGGGGGGATGACCAG*CAGCTCCTTAAAGAAAGAGTACAAATTGTTT
>IFI44L--PDGFA
ATGAAAACTGGAAGTTTTTTTTTGTTTGTTT*TTTTTTTTTTTTTTGTTTTTTTTTTTTTTTT
>IGH@--TUBA1B
GGTCGGTGCAGTGACAGTCGGGGGTGGCAGC*TGAGGGGATGGAGGAAGGCGAGTTTTCAGAG
>KLF8--IGBP1
AGAAGGTGGCTCAATGCAGGTATTCAAGCAG*GCATCAACTTCTAACTCATCTCGCCAGGAGA
>KMT2E--AHCTF1
GGTTCGGGTGTCTCGTGTGTGAACATCACAG*TGGAAGTATGCGAGACTTAAGAGCTCAAGTG
>LINC01031--CEP104
CCTGGGCGACAGAGCGAGACTCCGTCTCGAA*AAAAAAAAAAAAAGTATAAGGAGAGAAATAA
>LINC02321--CD24
CAGGTGATCCACCCGCCTCGGCCTCCCAAAG*GCTGGGATTACAGGCTTGAGCCCCCGCGCCC
>LINC02348--CSDC2
ACACACACACACACACACACACACACACACA*ATCTTCCAGACCCATCCCCTGCCTGCCAGCT
>LINC02802--AC239859.5
GGCCCAGACTCCACTCCCCAGCTGTGAAAGG*ACTGCTGGCCAGACCCCCAAGCTAGCCCGCC
>LINC02802--AC239859.5
TCAGTCTGCGGCCAAGACTCCGTCCTGCAAG*GATATTGTGGCTGGAGACATGAGCAAGAAAA
>LINC02883--AC021739.2
CCAGGACCCAGGGCTCCAAAGCCATTTCCAG*GATTTAGCACACAAATGTCCATAGAGCATTT
>LINC02883--AC021739.2
CTTCTGTCCACTCTGCAGAGTGAACGTTGAG*GATTTAGCACACAAATGTCCATAGAGCATTT
>LYZ--FAT4
AATACATCTCCAGTACATTCCGTTCTTTTTT*TTTTTTTTTTTTTTTGAGACGGAGTCTCGCT
>LYZ--FAT4
GCTAATTTTTTGTATTTTTAGTAGAGACAGG*GTTCACGCCATTCTCCTGCCTCAGCCTCCCG
>LYZ--FAT4
TACATCTCCAGTACATTCCGTTCTTTTTTTT*TTTTTTTTTTTTTTTGAGACAGAGTTTCACT
>LYZ--NCAPH2
CTCCAGTACATTCCGTTCTTTTTTTTTTTGA*GACAGAGTGAGACTCCGTCTCAAACAAAACA
>LYZ--XPO1
GGTTATCAAATACATCTCCAGTACATTCCGT*TTTTTTTTTTTTTTGAGACGGAGCCTGGCTC
>MALT1--STAR
CATGGTGAAACCCCGTCTCTACTAAAATACA*AAAAAATTAGCTGGGTGTGGTGGCGCAAACC
>MARK3--FCAR
GCCTCAGCCTCCTGAGTAAGTGGGATTACAG*GGGACTTTCCCATGCCTTTCATATCTGCCAA
>MRPS31P5--THSD1
AGAGTATATTTAAAATTTTTATTAAGGCCCA*TGGGGCAGCCCTTGAGGAAACCAGACCTTGG
>MT1E--MT1M
AAAGAGTGCAAATGCACCTCCTGCAAGAAGA*GCTGCTGCTCCTGCTGCCCCGTGGGCTGTGC
>MT1M--MT1E
AAAGAGTGCAAATGCACCTCCTGCAAGAAGA*GCTGCTGTTCCTGCTGCCCCGTGGGCTGTGC
>MT1M--MT1E
GAAATGGACCCCAACTGCTCCTGCACCACTG*GTGGCTCCTGCACGTGCGCCGGCTCCTGCAA
>NEB--KCNK3
AGCCTGGGTGACAGGGCAAGACCCTGTCTCA*AAAAAAAAAAAAAAAAATGGCAAAGGGAGAC
>NKG7--AC008750.7
TGGCCAACAGGGCATGGGGACATCATATCAG*ATCCGAGTGAAGCCTGACAGGACCGGTGTGG
>NONO--S100A9
GCCTCAGCCTCCTGAGTAGCTGGGATTACAG*AAGGAGAATAAGAATGAAAAGGTCATAGAAC
>NONO--S100A9
GGCTCAGCCTCCTGAGTAACTGGGATTACAG*AAGGAGAATAAGAATGAAAAGGTCATAGAAC
>NPM1--BLNK
TAAGAAAATAGTTTAAACAATTTGTTAAAAA*TTTTCCATCTTATTTCATTTCTCTAACAGTC
>NPM1--SET
TGATGACGATGATGATGAAGAGGATGATGAT*GAAGATGAAGATGATGATGAAGGGGAGGAAG
>NPM1--SET
TGATGACGATGATGATGAAGAGGATGATGAT*GAAGATGATGATGATGATGAAGAGGAGGAAG
>NPNT--PPA2
CCTGGTTATGCTGGAAAAACCTGTAATCAAG*AGAATGTAACTGGTCACTACATTTCCCCCTT
>NQO2--AL133351.4
AGATTCTCGATACTTCCTGTGGCCACTCCAG*TGGGCTGATGCGCTGGAACGTCGATGCAGCC
>PADI4--S100A9
TGGGCACCTTGACTCAGCTTGACATCTGCAG*ACAGAGTGCAAGACGATGACTTGCAAAATGT
>PDXDC2P-NPIPB14P--AC009065.3
CCCTACTAAAAATACAAAAAATTAGCCGGGC*ACCCTGTCCACAGCCCCCACACAGACTCGAG
>PDXDC2P-NPIPB14P--AC009065.3
TTGCGGTTCAGCCCTTTGATGACCGCAGCAG*GGTGGTCCCACTGTGGGACCACAACCAGGTA
>PGK1--ADSL
GGTCAGGAGATTGAGACCATCCTGGCTAACA*ACGGTGAAACCCTGTCTCTACTAAAAATACA
>PIGH--EXOC8
TTATCACTGGCATTTTTTCCTTCTCTTTCTA*TCTTTTTTCTTTTTTTTTTTTGAGACAGGGT
>PIGR--BCL11B
AAAATTGAGGTTTGGAAGAGTGAAGCGTTTT*TTTTTTTTTTTTTTTTTTTTTGAGACGGAGT
>PIGR--BCL11B
AAGAAAATTGAGGTTTGGAAGAGTGAAGCGT*TTTTCTTTTTCTTTTTTTTTTTTTGAGACGG
>PIGR--BCL11B
GGACTACAGGCGCCTGCCAGCACGCCTGGCT*AACATGGTGAAACCCTGTCTCTACTAAAAAC
>PIGR--BCL11B
GGACTACAGGCGCCTGCCAGCACGCCTGGCT*AACATGGTGAAACCCTGTCTCTACTAAAAAT
>PIGR--BCL11B
GGACTACAGGCGCCTGCCAGCACGCCTGGCT*AACATGGTGAAACCCTGTCTCTACTAAAACT
>PIGR--BCL11B
GTAGCTGGGACTACAGGCGCCTGCCAGCACG*CCTGGCTAACATGGTGAAACCCTGTCTCTAC
>PIGR--BCL11B
TAAGAAAATTGAGGTTTGGAAGAGTGAAGCG*TTTTTTTTTTTTTTTTTTTTTTTTTTGAGAC
>PIGR--BCL11B
TAGCTGGGACTACAGATGTGGGCCACCATGT*CTGTAGTCCCAGCCACTTGAGAGGCGGTGGC
>PIGR--BCL11B
TAGCTGGGACTACAGATGTGGGCCACCATGT*CTGTAGTCCCAGCTACTTGGGAGGCTGAGGC
>PIGR--BCL11B
TAGCTGGGACTACAGATGTGGGCCACCATGT*CTGTATTCCCAGCTACTTGGGAGGCTGATGT
>PIGR--BCL11B
TCTGCCTCCCAAAGTGCTGGGATTACAGGCG*GATCACGAGGTCAGGAGATCGAGACCATCCT
>PIGR--BCL11B
TGCCTCTGCCTCCCAAAGTGCTGGGATTACA*GGCGGATCACGAGGTCAGGAGATCGAGACCA
>PIGR--SI
TAGCTGGGACTACAGATGTGGGCCACCATGT*CTGTAGCCCCAGCTACTTGAGAGGCATATTA
>PKD1--AC138969.1
AGAGTGACTTGTTTCTGGATGATTCTAAGAG*GTTGATGGACGGATGGCTACATGGATGTGAT
>PKD1--AC138969.1
AGAGTGACTTGTTTCTGGATGATTCTAAGAG*TATCAATACTCTGGCTGACCATCGTCATCGT
>PKD1--AC138969.1
AGAGTGACTTGTTTCTGGATGATTCTAAGAG*TGACCTTGAGGAACCCTGGGAGCTCAGGAAG
>PKD1--AC138969.1
CTTCCTCACGTTCTCAGGCCTCCACGCTGAG*GTTATCAATACTCTGGCTGACCATCGTCATC
>PKD1--AC138969.1
CTTCCTCACGTTCTCAGGCCTCCACGCTGAG*GTTGATGGACGGATGGCTACATGGATGTGAT
>PKD1--AC138969.1
CTTCCTCACGTTCTCAGGCCTCCACGCTGAG*TGACCTTGAGGAACCCTGGGAGCTCAGGAAG
>PMS2--AC211476.4
AACAGTCTGGATGCTGGTGCCACTAATATTG*ATCTAAAGCTTAAGGACTATGGAATGGATCT
>PMS2--AC211476.4
CACCATTTCTACCTGCCACGCATCGGCGAAG*GTTGGGACTCGACTGGTGTTTGATCACGATG
>PMS2--AC211476.4
CCATAAGGAATTTCAAAGGAATATTAAGAAG*AAACGTGCCTGCTTCCCCTTCGCCTTCTGCC
>PMS2--AC211476.4
CCATAAGGAATTTCAAAGGAATATTAAGAAG*GACTCCTAGAAGTACCCCACCCCACCCCTGC
>PMS2--AC211476.4
GGGAAGCTCTGAGCTCACTTTGTGCACTGAG*TGATGTCACCATTTCTACCTGCCACGTATCG
>PMS2--AC211476.4
GGGGTAGAAGAAGAAAACTTCGAAGGCTTAA*CTCTGAAACATCACACATCTAAGATTCGAGA
>PMS2--AC211476.4
GGGGTAGAAGAAGAAAACTTCGAAGGCTTAA*TGATGTCACCATTTCTACCTGCCACGTATCG
>PMS2--AC211476.4
TAAGATTCAAGAGTTTGCCGACCTAACTCAG*TGATGTCACCATTTCTACCTGCCACGTATCG
>PMS2--AC211476.4
TCTGGGCAGGTGGTACTGAGTCTAAGCACTG*GGTGAAGAAGATAGTAGGAAACAGTCTGGAT
>PMS2--AC211476.4
TTGCATCCATGGAGCGAGCTGAGAGCTCGAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>PMS2--AC211476.5
AACAGTCTGGATGCTGGTGCCACTAATATTG*ATCTAAAGCTTAAGGACTATGGAATGGATCT
>PMS2--AC211476.5
CACCATTTCTACCTGCCACGCATCGGCGAAG*GTTGGGACTCGACTGGTGTTTGATCACGATG
>PMS2--AC211476.5
CCATAAGGAATTTCAAAGGAATATTAAGAAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>PMS2--AC211476.5
GGGAAGCTCTGAGCTCACTTTGTGCACTGAG*TGATGTCACCATTTCTACCTGCCACGTATCG
>PMS2--AC211476.5
TCTGGGCAGGTGGTACTGAGTCTAAGCACTG*GGTGAAGAAGATAGTAGGAAACAGTCTGGAT
>PMS2--AC211476.5
TTGCATCCATGGAGCGAGCTGAGAGCTCGAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>POM121C--AC091390.3
CCAAGAGCAGCAGAAAATGAACACATTGCAG*GGGCCAGTGTCATTCAAAGATGTGGCTGTGG
>PSMB2--TFAP2E-AS1
AGTGGAACTCCTTAGGAAATGTCTGGAGGAG*GAACTGCTTCACAAGATTAAAGGAGAAACAG
>PSMB2--TFAP2E-AS1
AGTGGAACTCCTTAGGAAATGTCTGGAGGAG*GTGCAGGCTGCCCCATGGATGCCACACGTGG
>PTBP1--MKNK2
CCCGCGGGTCTGCTCTGTGTGCCATGGACGG*ATCATTGAGAAGCAGCCAGGCCACATTCGGA
>RACK1--BRD2
TGTTTTTTTGTTTGTTTGTTTTTTGTTTTTT*TTTTTTTTAAAGACTCCAGGCTGGAGTGCAG
>RBFOX1--SDC3
TTTTGTTTGTTTTTTTTTGTTTGTTTGTTTG*TTTTTTAACACTTCCTGTGCTGTGCCCATTT
>RETSAT--AC093162.1
ACCAACCAGTTCTATCTGGCTGCTCCCCGAG*GTGTCTGCTACAGGGCTGACCATGACCTGGG
>RPL3--BCL11B
AAGAACAATGCCTCCACTGACTATGACCTAT*CTGATCAAGAACAATGCCTCCACTGACTATG
>RPL3--BCL11B
ACTATGGTGAAGTGACCAATGACTTTGTCAT*GCAGAAAGGCTACCATCACCGCACTGAGATC
>RPL3--BCL11B
ATGACCCGTCCAAGCCGGTCCACCTCACAGC*CTTGTATCCCAGGAAGGCTGTGAGGTGGACC
>RPL3--BCL11B
CAGACCATGGAGGAGAAGAAAGCATTCATGG*CACCACCTCCAAGTTTGGCCATGGCCGCTTC
>RPL3--BCL11B
GACCACTGAAGAAAGACCGAATTGCAAAGGA*AGAAGAAAGCATTCATGGGACCACTGAAGAA
>RPL3--BCL11B
GTGTGGTGGGAACCAAGAAGCGGGTGCTCAC*CCTCCACTGACTATGACCTATCTGACAAGAG
>RPL3--BCL11B
TCAAGAACAATGCCTCCACTGACTATGACCT*CTGATCAAGAACAATGCCTCCACTGACTATG
>RPL3--BCL11B
TGTAGCCTTCTCTGTGGCACGCGCTGGGCAG*AAGGCTACCATCACCGCACTGAGATCAACAA
>RPS3A--EEF1A1
ACCAACAGGTCCGCCAAATCCGGAAGAAGAT*GATGGAAAGTCACCCGTAAGGATGGCAATGC
>RPS4X--HBB
AAGGGTCGCTTTGCTGTACATCGTATTACAC*CTGAGGAGAAGTCTGCCGTTACTGCCCTGTG
>SCD5--EEF1A1
TTATCAATGAAAAAACTTTCGTTTTTTTTTT*TTTTTTGTTTTGTTTTTTTTGGGTAGAGAAG
>SDC3--RBFOX1
CTGGTTTTGTTTTTGTTTTTTTTTTTTTTTT*TTTGTTTGTTTGTTTGTTTTTTAGATGGAGT
>SERF2--WSB1
ACGGTGAAACCCTGTCTCTACTAAAAATACA*AAAAAAAAAAAAAATTAGCCAGGCGTGGTGG
>SERF2--WSB1
ATGGTGAAACCCTGTCTCTACTAAAAATACA*AAAAAAAAAAAAAATTAGCCAGGCGTGGTGG
>SH3D19--EEF1A1
GTTGGTGCTGAGCATAAGAGGTCTTCCGTAT*CTGATTGTTGCTGCTGGTGTTGGTGAATTTG
>SMAP2--TNK2
ACGTTTTGAGCCACAACTGTCACAACTCCAG*AGGCTGGGAGGCGGCAGAATGCAGCCAGAGG
>SPEN--S100A9
GCTTCAGCCTCCTGAGTAGCTGGGACTACAG*AAGGAGAATAAGAATGAAAAGGTCATAGAAC
>SRSF3--S100A9
GCCTCAGCCGCCTGAGTAGCTGGGATTACAG*AAGGAGAATAAGAATGAAAAGGTCATAGAAC
>STARD3--TCAP
CTTTGTCTGGATTCTTAATACAGATCTCAAG*CTGCTCCCTGCATGAGGAGGACACCCAGAGA
>STARD3--TCAP
GTGCCAAGCCCCCGACGCACAAATATGTCCG*CTGCTCCCTGCATGAGGAGGACACCCAGAGA
>TAGLN--MACF1
GGAATATTTTTGGGGTTGGAACTCAAAAAAA*CAAAAAAAAAAAAAATTTTTTTTTACTTGTA
>TAX1BP1-AS1--HIBADH
CCACTCCGCCGCGGGACGCCTCATCCCGCGG*TGTGTTCTAGGTCAGTGGCTTCAAAGACTCC
>TMPRSS11B--AC093010.1
AGCCTGGGAAACATGGCTAAATCCGTGTTCA*AAAAATACAAAAAATTAGCTAGGCATGGTGG
>TMPRSS11B--AC093010.1
CAGCCTGGGAAACATGGCTAAATCCGTGTTC*AAAAATACAAAAAATTAGCTAGGCATGGTGG
>TMSB15B--TMSB15B
TACGGCCTTTCACGAGTCTTCAAGCCTTCAG*AAGGAGTCTCGCTCTGTCGCCAGGCTGGAGT
>TMSB15B-AS1--RAB9B
AATACAGGGGAACGCTAGTTTCCGAGCTCAT*CACCCTAAAAAGAAAGAATAAAACAACAACA
>TNRC6B--AL022238.3
AGGCGGCCAAGGCCCAAACTGCACTGCACAT*ACATTGGGTTTGCCTATCACAGATGAACAAA
>TRPM2--ZNF511
CTTCACGGGCCTGAGCCAGAAGGTGAAAAAG*AATACCCTCTACCATCTGCTTTGGTCAGGGT
>WDR1--AC005674.1
TTCCTTTTCTTTTTCAGTGCAGAAATTAAAA*GCTGAGGCTGAGTTGCATCTACATTTTGGTT
>YWHAZ--UBC
CCCTCAAACCTTGCTTCTAGGAGATAAAAAG*ACAATGCAGATCTTCGTGAAGACTCTGACTG
>Z84488.1--BASP1
ACATGGAGAGGATGGCAGCGCAGCCTAGTTG*AACTCCAAGATGGGAGGCAAGCTCAGCAAGA
>ZEB2--AC093525.2
CGCAAACAAGCCAATCCCAGGAGGAAAAACG*CCCTGGGCGCTGCCTATGGCACAGCCAAGAG
>ZMAT4--AC244197.3
GCACCGGCGGCCCCTCTGCCTCGCGGAAAAG*GCAGGCAATCCATGGACCTTGTGGAACTTGT
>ZNF394--ATP5MF
CAGAAGGATTCCGGGAGCACAGTTCCGCCGA*TACCAGTGAAGGACAAGAAACTTCTGGAGGT
>ZNF791--RBM39
CAGGAAACATTCAAGAACCTGGCATCTATAG*GAGGAAAAAAAGCAAGAGCAGAAGTCGTAGT
>ZNF806--ZNF285
GTGCAGCTGTGACCCGTCCCTGTGGACTGAA*TTCTGCTTTTCCAGAATCCTGCCCTTTCCAA
>ABR--NXN
GGACAAGATCATGGGCAAAGGACAGATCCAG*ATTCTGAGGATGACGGAGAGTCCGAGGCGGC
>AC004946.1--UBC
CACCTTGATCTCAGATTTCTAGCCTCCAAAG*ACAATGCAGATCTTCGTGAAGACTCTGACTG
>AC005013.1--AC005162.2
AAGGTTGCCTGCAGAAGGCCATCAGAAGGGG*AAAAATGACTTAATGTCTAAGGAGAAGGCTA
>AC005013.1--AC005162.2
AAGGTTGCCTGCAGAAGGCCATCAGAAGGGG*AGAGTGATCTGCCTGCTCCCCGCAGTTGAGA
>AC005013.1--AC005162.2
CACAAGGTTGCCTGCAGAAGGCCATCAGAAG*GAGAGTGATCTGCCTGCTCCCCGCAGTTGAG
>AC006064.6--LYZ
TCTCGATCTCCTGACCTTGTGATCCACCCAC*CTCAGCCTCCCGAGTAGCTGGGATTACGGGC
>AC006116.8--AC006116.6
ATCTGGCAAATGTGTAATCTGAGTAATAGAG*AATCATCCAGAGCTCCTGTGCTATAATGGCA
>AC008686.1--OLFM1
CAGGAGAGTCGCTTGAACCCAGAAGGCGGAG*GTTGCAGTGAGCCGAGACTGCACCACTGCAC
>AC008993.1--MIR1302-9HG
ACTTTTAGAAATCACATATGTGACTTAAATA*TTTCTTTTCTTTTTCTTTCCTCTCACTCAGC
>AC008993.1--MIR1302-9HG
GTGACTTAAATATTTCTTTTCTTTTTCTTTC*CTCTCACTCAGCGTCCTGTGATTCCAAAGAA
>AC008993.1--MIR1302-9HG
TGTGACTTAAATATTTCTTTTCTTTTTCTTT*CTCTCACTCAGCGTCCTGTGATTCCAAAGAA
>AC015911.9--SLFN13
TTTGTGAAAGTCAGTTTCTGAGGAATTTCAT*CAGGTCTTTCAGGACTGATTTTGGTGGAAGA
>AC015977.1--CIB4
TGCCCAGGTTCTGCCCAGAAGCCAGCTGCAG*GCCCTGACCTTCCTGACCAGAAATGAAATTC
>AC016907.2--AC106870.3
CAAGAAGAAAAAGTCCATGTAGGACTGAAAT*GTCCTGCTCCCTGCCTGCCCATCTTCGACCT
>AC016912.2--RHOQ
AAGGAAGTCCTGTCTTCTGGGATTGAGTTTG*GAAGACTATGACCGTCTGAGGCCTTTATCTT
>AC016912.2--RHOQ
ACAACCAACAAAGGCCTTAACGGCACAGAAG*GAAGACTATGACCGTCTGAGGCCTTTATCTT
>AC017074.1--ACR
AAACCAGTTATTACCAGGAAACAAAAGCAAA*CCCCCAGGCCATCATCTATACTGATGGAGGC
>AC018630.4--AMY2B
TTCTGGTTCCCATCACCAGCAGAGGGTTGAG*ATTGCTCTTGAATGTGAGCGATATTTAGCTC
>AC018630.4--CA6
GGCCTCCCTGTTGGGGTGGTCCTTGTGGCTT*CCCTGGTGCTTCTGCTGTCCCTGTTCCTGCT
>AC018630.4--CCL28
TGTTGTTATTTTGATACGGAGTCTCGCTCTG*TCACACAGGCTGGAGGGCAGTGGCAGGATCT
>AC018630.4--CD44
TGCTCCCAGGCTTTGTTTTTTGTTTTTTGTT*TTTTTTTTTTTTGACACTGTCCAAAGGTTTT
>AC018630.4--CD74
CCCACCTCCTCCTGGAAAGCCCCAGGGACCA*CCTCCCCCATCTCCCACCCTGTACCTCATCC
>AC018630.4--CST1
AGGTGGTCCCTGACGCTCCTCATCTATGAAC*CGCTCTGCCCTCCCTCCTTCCTTCTTGCTTC
>AC018630.4--CST1
CTCAAGGAAGGCCACAAGGACCACCCCAACA*GGGAGGCCTCCCCATGTGCCTGCGCCAAGAG
>AC018630.4--CST1
TGGGGTGGTCCTTGTGGCTTTCCCTGAGGAG*GAGGATAGGATAATCCCGGGTGGCATCTATA
>AC018630.4--CST1
TGTTGGGGTGGTCCTTGTGGCCTTCCTTGAG*GAGGAGGATAGGATAATCCCGGGTGGCATCT
>AC018630.4--CST4
AGCATCTTGCAGAAGGCTCTGGTGTCACTCC*CAGCTTTGTGCTCTGCCTCTGAGGAGACCAT
>AC018630.4--CST4
CCTTGTGGCCTTCCTTGAGGAGGAGGGGGAT*GGCCTCCCCATGTGCCTGTGCCAAGAGACAG
>AC018630.4--CST4
CTTGTGGCCTTCCTCGAGGAGGACGGGGATG*GCCTCCCCATGTGCCTGTGCCAAGAGACAGA
>AC018630.4--CST4
GATGAGGAGCGTCAGGGACCACCTTTGGGAG*GACAGAATGTCCCTGGTGAATTCCAGGTGTC
>AC018630.4--EEF1A1
CTTGTGGCTTTCCCTGAGGAGGTGGTGGACC*TTGTTGCTGCTGGTGTTGGTGAATTTGAAGC
>AC018630.4--EEF1A1
GGTGGTCCTTGTGGCTTTCCCTGAGGAGGTG*GTATTGGTACTGTTCCTGTTGGCCGAGTGGA
>AC018630.4--EEF1A1
TGTGGCTTTCCCTGAGGAGGTGGTGGACCTT*GTTGCTGCTGGTGTTGGTGAATTTGAAGCTG
>AC018630.4--EEF2
GGTCCTTGTGGCTTTCCCTGAGGAGGTGGTG*TGGCATCTACGGGGTTTTGAACAGGAAGCGG
>AC018630.4--GAPDH
CCTTGTGGCTTTCCCTGAGGAGGTGGTGGAC*CTGACCTGCCGTCTAGAAAAACCTGCCAAAT
>AC018630.4--GNAS
ACAAGGAGGCCAGCAGCAACAAGGTCCACCA*CCAGCAGAGGGATGAGATGCTCAAATGACAT
>AC018630.4--GNE
CCTGGGTGACAGAGCAAGACTCCGTCTCAAA*AATAATAATAAATAATAATAATAATAATAAT
>AC018630.4--H1-0
GGGAGGCCATCCCCCTCCTCCTCAAGGAAGG*CCACAGACCACCCCAAGTATTCAGACATGAT
>AC018630.4--IGH@
GGTCCCTGGGGCTTTCCAGGAGGAGGTGGGG*CTGGGCAGGGCACAGTCACATCCTGGCTGGG
>AC018630.4--IGK@
ACCCCAACAAGGAGGCCAGCAGCAACAAGGT*CCACCTGCTCCTCAGTTCCAGCCTGACCCCC
>AC018630.4--IGK@
GGTTGAGATTGCTGTCCTCCCAAAGGTGGTC*CCTGACGCTGAGCAAAGCAGACTACGAGAAA
>AC018630.4--IL6ST
TGTGTGTGTGTATATATATATATACACACAC*ATATACACACACATATCATTGACTTTTCTTA
>AC018630.4--ITPR1
GCTGGTGATGGGAACCAGGATGATGGCCCTC*GTTGGGGACCACCTTTGGGGACTGGCTTCCT
>AC018630.4--MUC7
ACCCCAACAGGGAGGCCATCAGCAAGGTCCT*CACCCACACCTTCTGCAACTACACCAGCTCC
>AC018630.4--MUC7
CAACAAGGAGGCCAGCAGCAACAAGGTCCAC*CATCTTCCTCAGCTCCACCAGAGACCACAGC
>AC018630.4--MUC7
CAACAAGGAGGCCAGCAGCAACAAGGTCCAC*CGTCTTCCCCAGCTCCACAAGAGACCACAGC
>AC018630.4--MUC7
CAGGGAGGCCATCAGCAAGGTCCTCCCCCAC*ACCTTCTGCAACTACACCAGCTCCACCATCT
>AC018630.4--MUC7
CAGGGAGGCCATCAGCAAGGTCCTCCCCCAC*CTCCTGCAACTACACCAGCTCCACCATCTTC
>AC018630.4--MUC7
CCACCACCTCCTCAGGGAAAGCCACAAGGAC*CACAGCTGCCCCACCCACACCTTCTGCAACT
>AC018630.4--MUC7
CCACCCCAACAAGGAGGCCAGCAGCAACAAG*CTCCACCATCTTCCTCAGCTCCACCAGAGAC
>AC018630.4--MUC7
CCCATCACCAGCAGAGGGTTGAGATTGCTGT*CCCCCCAAATTCCCAAATCCTCACCAGCCAC
>AC018630.4--MUC7
CCTCCTCCTCAAGGAAGGCCACAAGGACCAC*CACCTCCTGCAACTACACCAGCTCCACCATC
>AC018630.4--MUC7
CCTCCTCCTCAAGGAAGGCCACAAGGACCAC*CACCTTCTGCAACTACACCAGCTCCACTATC
>AC018630.4--MUC7
CCTCCTCCTGGAAAGCCCCAGGGACCACCTC*CCCAACTACTCTTGCACCTGACACTTCTGAA
>AC018630.4--MUC7
CGTCCTCCTCGAGGAAGGCCACAAGGACCAC*CCACACCTTCTGCAACTACACCAGCTCCACC
>AC018630.4--MUC7
GGAGGCCATCCCCCTCCTCCTCAAGGAAGGC*CACAGCTGTCCCACCCACACCTTCTGCAACT
>AC018630.4--MUC7
GGTCCTTGTGGCTTTCCCTGAGGAGGTGGTG*CTTCTGCAACTACACCAGCTCCACCATCTTC
>AC018630.4--MUC7
GGTTCCCATCACCAGCAGAGGGTTGAGATTG*CTGTCCCACCCACACCTTCTGCAACTACCCT
>AC018630.4--MUC7
TCCTCATCTATGAACTGCTCAGAGTCTCCTC*CATCTTCCTCAGCTCCACCAGAGACCACAGC
>AC018630.4--MUC7
TCTGCTGGTGATGGGAACCAGGATGATGGCC*CTCAGCTCCACCAGAGACCACAGCTGCCCCA
>AC018630.4--MUC7
TGACGCTCCTCATCTATGAACTGCTCAGAGT*CTCCACCATCTTCCTCAGCTCCACCAGAGAC
>AC018630.4--MUC7
TGGTCCCTGCTGAGGGCCATCATTCTGGTTC*CCATCATCCGCCTCAGCTCCACCAGAGACCA
>AC018630.4--MYH9
GGGAGGACAGCAATCTCAACCCTCTGCTGGT*TCCCAACACCAGCAGAGGGGTTGTTCCAGAC
>AC018630.4--PIGR
GGAGGTGGTCCCTGGGGCTTTCCAGGAGGAG*GGAGAAGAGAAAATGTAAAGGAGGGAGATCT
>AC018630.4--PRR27
CTGTCCTCCCAAAGGTGGTCCCTGACGCTCC*CCAACTCAGCTCACTGCAACTTCCACCTCCC
>AC018630.4--RBM47
ACCCCAACAGGGAGGCCATCCCCGTCCTCCT*CAGATTCCCAATCCCACCGCAATGTTTGGCA
>AC018630.4--RPL3
TCCCTGTTGGGGTGGTCCTTGTGGCCTTCCT*GGAGGTGGTGGAGGCTGTGACCATTGTAGAG
>AC018630.4--RPL3
TGAGGAGGTGGTGGACCTTGTTGCTGCTGGC*CATTGTAGAGACACCACCCATGGTGGTTGTG
>AC018630.4--RPL3
TGGGGTGGTCCTTGTGGCTTTCCCTGAGGAG*GTGGTGGAGGCTGTGACCATTGTAGAGACAC
>AC018630.4--SLC20A2
CTGCTGGTGATGGGAACCAGGATGATGGCCC*TCATCCTGGTTCCCCTCACCCGGGCCTGCCT
>AC018630.4--SMR3B
AAAGCCACAAGGACCACCCCAACAGGGAGGC*CACCCCCTCCTCAACCCTAAGGTCCACCACT
>AC018630.4--SMR3B
AAGGACCACCCCAACAGGGAGGCCATCCCCG*CTCCTCAACCCTAAGGTCCACCACTCCATCC
>AC018630.4--SMR3B
AAGGTCCTCCCCCACCTCCTCCTGGAAAGCC*CCTATCCACCTGGACCTCCATTTTTCCCTGT
>AC018630.4--SMR3B
ACAAGGTCCACCACCTCCTCAGGGAAAGCCA*GTTATCCACAGCCTCCTTCCCGACCAAGACC
>AC018630.4--SMR3B
ACCCCAACAGGGAGGCCATCAGCAAGGTCCT*CACCACTCCATCCTGATGCCCCAGGTTATCC
>AC018630.4--SMR3B
ACCTCCTCAGGGAAAGCCACAAGGACCACCC*CAACTGATCCTACCCTCCCTACTCCTGCACC
>AC018630.4--SMR3B
ACGCTCCTCATCTATGAACTGCTCAGAGTCT*CTCCAACTGATCCTACCCTCCCTACTCCTGC
>AC018630.4--SMR3B
AGATTGCTGTCCTCCCAAAGGTGGTCCCTGA*CTCCTCAACCCTAAGGTCCACCACTCCATCC
>AC018630.4--SMR3B
AGGCCATCCCCGTCCTCCTCGAGGAAGGCCA*GTTATCCACAGCCTCCTTCCCGACCAAGACC
>AC018630.4--SMR3B
AGGTCCACCACCTCCTCAGGGAAAGCCACAA*GTTATCCACAGCCTCCTTCCCGACCAAGACC
>AC018630.4--SMR3B
ATGGCCCTCAGCAGGGACCACCCCAACAAGG*ACCACTACCCTTGTAACTACTGCTTCTACTA
>AC018630.4--SMR3B
ATTGCTGTCCTCCCAAAGGTGGTCCCTGACG*CCCAAATATGAACAACTGCAGCAGGTGCCAC
>AC018630.4--SMR3B
ATTGCTGTCCTCCCAAAGGTGGTCCCTGACG*CTCAACCTTTTGGCCCAGGATTTGTTCCACC
>AC018630.4--SMR3B
ATTGCTGTCCTCCCAAAGGTGGTCCCTGACG*CTCCTCAACCCTAAGGTCCACCACTCCATCC
>AC018630.4--SMR3B
ATTGCTGTCCTCCCAAAGGTGGTCCCTGACG*CTCCTCCCGCACCCTATGGTCCAGGGATATT
>AC018630.4--SMR3B
ATTGCTGTCCTCCCAAAGGTGGTCCCTGACG*CTCCTCCTCCCGCACCCTATGGTCCAGGGAT
>AC018630.4--SMR3B
CAACAGGGAGGCCATCAGCAAGGTCCTCCCC*CCCCTCCTCAACCCTAAGGTCCACCACTCCA
>AC018630.4--SMR3B
CAAGGAGGCCAGCAGCAACAAGGTCCACCAC*CACAAAAGACACCACTACCCTTGTAACTACT
>AC018630.4--SMR3B
CAAGGAGGCCAGCAGCAACAAGGTCCACCAC*CTCCTCAACCCTAAGGTCCACCACTCCATCC
>AC018630.4--SMR3B
CAAGGAGGCCAGCAGCAACAAGGTCCACCAC*CTCCTCCACCCTATGGTCCAGGGAGAATCCC
>AC018630.4--SMR3B
CAAGGAGGCCAGCAGCAACAAGGTCCACCAC*CTCCTCCTCCACCCTATGGTCCAGGGAGAAT
>AC018630.4--SMR3B
CAAGGAGGCCAGCAGCAACAAGGTCCACCAC*CTCCTCCTCCCGCACCCTATGGTCCAGGGAT
>AC018630.4--SMR3B
CAGGGAGGCCATCAGCAAGGTCCTCCCCCAC*CACCACAAAAGACACCACTACCCTTGTAACT
>AC018630.4--SMR3B
CAGGGAGGCCATCAGCAAGGTCCTCCCCCAC*CCCCTCCTCAACCCTAAGGTCCACCACTCCA
>AC018630.4--SMR3B
CAGGGAGGCCATCAGCAAGGTCCTCCCCCAC*CTCCTCCACCCTATGGTCCAGGGAGAATCCC
>AC018630.4--SMR3B
CAGGGAGGCCATCAGCAAGGTCCTCCCCCAC*CTCCTCCCGCACCCTATGGTCCAGGGATATT
>AC018630.4--SMR3B
CAGGGAGGCCATCAGCAAGGTCCTCCCCCAC*CTCCTCCTCCACCCTATGGTCCAGGGAGAAT
>AC018630.4--SMR3B
CAGGGAGGCCATCAGCAAGGTCCTCCCCCAC*CTCCTCCTCCCGCACCCTATGGTCCAGGGAT
>AC018630.4--SMR3B
CAGGGAGGCCATCAGCAAGGTCCTCCCCCAC*CTCCTTCCCGACCAAGACCCTATCCACCTGG
>AC018630.4--SMR3B
CATTGAATCCTAGATTACTGAGGAGACTGCC*CCCAAATATGAACAACTGCAGCAGGTGCCAC
>AC018630.4--SMR3B
CCACAAGGACCACCCCAACAGGGAGGCCATC*CCCCTCCTCAACCCTAAGGTCCACCACTCCA
>AC018630.4--SMR3B
CCACCACCTCCTCAGGGAAAGCCACAAGGAC*CACCCCCTCCTCAACCCTAAGGTCCACCACT
>AC018630.4--SMR3B
CCACCCCAACAAGGAGGCCAGCAGCAACAAG*GTCCACCACTCCATCCTGATGCCCCAGGTTA
>AC018630.4--SMR3B
CCACCTCCCCAAGGGGGCCGCCCACAAGGAC*CTCCATTTTTCCCTGTAAATTCTCCAACTGA
>AC018630.4--SMR3B
CCACCTCCTCAGGGAAAGCCACAAGGACCAC*CACTCCATCCTGATGCCCCAGGTTATCCACA
>AC018630.4--SMR3B
CCACCTCCTCAGGGAAAGCCACAAGGACCAC*CCTATGGTCCAGGGATATTTCCACCACCCCC
>AC018630.4--SMR3B
CCACCTCCTCCTGGAAAGCCCCAGGGACCAC*CCCCTCCTCAACCCTAAGGTCCACCACTCCA
>AC018630.4--SMR3B
CCACCTCCTCCTGGAAAGCCCCAGGGACCAC*CTCCTCCTCCACCCTATGGTCCAGGGAGAAT
>AC018630.4--SMR3B
CCACCTCCTCCTGGAAAGCCCCAGGGACCAC*CTCCTCCTCCCGCACCCTATGGTCCAGGGAT
>AC018630.4--SMR3B
CCAGGAGGCCATCACCGCCATCCTCCCCCAC*CTCCTCCTCCACCCTATGGTCCAGGGAGAAT
>AC018630.4--SMR3B
CCCACCTCCTCCTGGAAAGCCCCAGGGACCA*CCTCCCTACTCCTGCACCCCAAATATGAACA
>AC018630.4--SMR3B
CCCCAACAAGGAGGCCAGCAGCAACAAGGTC*CACCCCCTCCTCAACCCTAAGGTCCACCACT
>AC018630.4--SMR3B
CCCCCACCTCCTTTTCAAAATCAGCAACGAC*CACCCCCTCCTCAACCCTAAGGTCCACCACT
>AC018630.4--SMR3B
CCCCCTCCTCCTCAAGGAAGGCCACAAGGAC*CACCACCACAAAAGACACCACTACCCTTGTA
>AC018630.4--SMR3B
CCCCCTCCTCCTCAAGGAAGGCCACAAGGAC*CACCCCCTCCTCAACCCTAAGGTCCACCACT
>AC018630.4--SMR3B
CCCCCTCCTCCTCAAGGAAGGCCACAAGGAC*CACCTCCTCCTCCACCCTATGGTCCAGGGAG
>AC018630.4--SMR3B
CCCCCTCCTCCTCAAGGAAGGCCACAAGGAC*CATATCCACCTGGACCGCTGGCTCCTCCTCA
>AC018630.4--SMR3B
CCCCCTCCTCCTCAAGGAAGGCCACAAGGAC*CTCCATTTTTCCCTGTAAATTCTCCAACTGA
>AC018630.4--SMR3B
CCCCGTCCTCCTCGAGGAAGGCCACAAGGAC*CACCCCCTCCTCAACCCTAAGGTCCACCACT
>AC018630.4--SMR3B
CCCCGTCCTCCTCGAGGAAGGCCACAAGGAC*CATTTTTCCCTGTAAATTCTCCAACTGATCC
>AC018630.4--SMR3B
CCCCGTCCTCCTCGAGGAAGGCCACAAGGAC*CCCCTCCTCAACCCTAAGGTCCACCACTCCA
>AC018630.4--SMR3B
CCCTCTGCTGGTGATGGGAACCAGGATGATG*GCCCCAGGGGACCATATCCACCTGGACCGCT
>AC018630.4--SMR3B
CCTCCTCCTCAAGGAAGGCCACAAGGACCAC*TCCATCCTGATGCCCCAGGTTATCCACAGCC
>AC018630.4--SMR3B
CCTCCTTGTTGGGGTGGTCCCTGCTGAGGGC*CACTCCATCCTGATGCCCCAGGTTATCCACA
>AC018630.4--SMR3B
CCTTGTTGGGGTGGTCCCTGCTGAGGGCCAT*CCTGGACCTCCATTTTTCCCTGTAAATTCTC
>AC018630.4--SMR3B
CGTCAGGGACCACCTTTGGGAGGACAGCAAT*CTCAACCCTAAGGTCCACCACTCCATCCTGA
>AC018630.4--SMR3B
CGTCCTCCTCGAGGAAGGCCACAAGGACCAC*CCCAAATATGAACAACTGCAGCAGGTGCCAC
>AC018630.4--SMR3B
CTCCACAGGGGCAGTCTCCTCAGTAATCTAG*GTCCACCACTCCATCCTGATGCCCCAGGTTA
>AC018630.4--SMR3B
CTGACGCTCCTCATCTATGAACTGCTCAGAG*CTCCTCCACCCTATGGTCCAGGGAGAATCCC
>AC018630.4--SMR3B
CTGACGCTCCTCATCTATGAACTGCTCAGAG*CTCCTCCTCCCGCACCCTATGGTCCAGGGAT
>AC018630.4--SMR3B
GCAACAAGGTCCACCACCTCCTCAGGGAAAG*CCACAGCCTCCTTCCCGACCAAGACCCTATC
>AC018630.4--SMR3B
GCCACAAGGACCACCCCAACAGGGAGGCCAT*CCACCTCCTCCTCCCGCACCCTATGGTCCAG
>AC018630.4--SMR3B
GCCACAAGGACCACCCCAACAGGGAGGCCAT*CCCACCTCCTCCTCCCGCACCCTATGGTCCA
>AC018630.4--SMR3B
GGACCACCCCAACAGGGAGGCCATCCCCCTC*CTCCTCAACCTTTTGGCCCAGGATTTGTTCC
>AC018630.4--SMR3B
GGACCACCCCAACAGGGAGGCCATCCCCGTC*CTCAACCCTAAGGTCCACCACTCCATCCTGA
>AC018630.4--SMR3B
GGACCACCCCAACAGGGAGGCCATCCCCGTC*CTCCTCAACCTTTTGGCCCAGGATTTGTTCC
>AC018630.4--SMR3B
GGACCACCCCAACAGGGAGGCCATCCCCGTC*CTCCTCCCGCACCCTATGGTCCAGGGATATT
>AC018630.4--SMR3B
GGAGGCCATCAGCAAGGTCCTCCCCCACCTC*CATCCTGATGCCCCAGGTTATCCACAGCCTC
>AC018630.4--SMR3B
GGGAACCAGGATGATGGCCCTCAGCAGGGAC*CACCCCCTCCTCAACCCTAAGGTCCACCACT
>AC018630.4--SMR3B
GGTCCCTGCTGAGGGCCATCATTCTGGTTCC*CACCACCACAAAAGACACCACTACCCTTGTA
>AC018630.4--SMR3B
GGTCCCTGCTGAGGGCCATCATTCTGGTTCC*CACCACCCCCTCCTCAACCCTAAGGTCCACC
>AC018630.4--SMR3B
GTTGAGATTGCTGTCCTCCCAAAGGTGGTCC*CTACTCCTGCACCCCAAATATGAACAACTGC
>AC018630.4--SMR3B
GTTGAGATTGCTGTCCTCCCAAAGGTGGTCC*CTGTAAATTCTCCAACTGATCCTACCCTCCC
>AC018630.4--SMR3B
GTTGCTGCTGGCCTCCTTGTTGGGGTGGTCC*CCACCTCCTCCTCCCGCACCCTATGGTCCAG
>AC018630.4--SMR3B
TCAGCAAGGTCCTCCCCCACCTCCTCCTGGA*GCCCCAGGGGACCATATCCACCTGGACCGCT
>AC018630.4--SMR3B
TCATCTATGAACTGCTCAGAGTCTCCTCCAT*CCTGATGCCCCAGGTTATCCACAGCCTCCTT
>AC018630.4--SMR3B
TCATCTATGAACTGCTCAGAGTCTCCTCCAT*CTGATCCTACCCTCCCTACTCCTGCACCCCA
>AC018630.4--SMR3B
TCCTCATCTATGAACTGCTCAGAGTCTCCTC*CACCACAAAAGACACCACTACCCTTGTAACT
>AC018630.4--SMR3B
TCCTCCCCCACCTCCTCCTGGAAAGCCCCAG*GTTATCCACAGCCTCCTTCCCGACCAAGACC
>AC018630.4--SMR3B
TCCTTGTGGCTTTCCCTGAGGAGGTGGTGGA*CCTTGTAACTACTGCTTCTACTACCCAAAAT
>AC018630.4--SMR3B
TCTGCTGGTGATGGGAACCAGGATGATGGCC*GCAGCAGGTGCCACCACCACCACAAAAGACA
>AC018630.4--SMR3B
TGACGCTCCTCATCTATGAACTGCTCAGAGT*CTCCTTCCCGACCAAGACCCTATCCACCTGG
>AC018630.4--SMR3B
TGAGGAGGTGGTGGACCTTGTTGCTGCTGGC*CTCCTTCCCGACCAAGACCCTATCCACCTGG
>AC018630.4--SMR3B
TTGCTGATGGCCTCCCTGTTGGGGTGGTCCT*CCCACCTCCTCCTCCCGCACCCTATGGTCCA
>AC018630.4--SMR3B
TTGTGGCCTTCCTTGAGGAGGAGGGGGATGG*CCTCCCTACTCCTGCACCCCAAATATGAACA
>AC018630.4--SQSTM1
ACCTCCTCCTGGAAAGCCCCAGGGACCACCT*CCCCTTGTGGGCGTGCCCCTCCCATGGCCCC
>AC018630.4--STATH
GGAGGCCATCCCCCTCCTCCTCAAGGAAGGC*CACAATACCAACAATATACCTTTTAATATCA
>AC018630.4--ZG16B
GCTTTCCCTGAGGAGGTGGTGGACCTTGTTG*CTGCTGCTCACGCTTGCCCTCCTGGGGGGCC
>AC018630.4--ZG16B
GGGGTGGTCCTTGTGGCCTTCCTCGAGGAGG*CAAGTATTTCAGCACCACTGAAGACTACGAC
>AC018630.4--ZG16B
TGGGGTGGTCCTTGTGGCTTTCCCTGAGGAG*GTGGTGGCTGATGGTACTGGAGTAACTGAGT
>AC018809.3--PRRT3-AS1
GGCTATCAGCAGGTGGGCTCCAAGTGTCTCG*TGTGGTTGGCAAGTCATTTAACTTTTCCAAA
>AC019205.1--EIF3E
AATTCTCTAGCACAGACTCCCTTTTTTTGGT*AAGATGGCGGAGTACGACTTGACTACTCGCA
>AC019205.1--EIF3E
CTCTAGCACAGACTCCCTTTTTTTGGTAAGA*TGGCGGAGTACGACTTGACTACTCGCATCGC
>AC020656.1--MYLK
TTATATTTTTTACTTAATATTGGGCATAATC*GGCTCACTGCAAGCTCCACCTCCCGGGTTCA
>AC020656.1--POLH
GAGGTGGAGGTTGCAGTGAGCCGAGATTGCG*CACTGCAATCTCCACCTCCCGGGTTCACGCC
>AC020656.1--POLH
GCACTTTGGGAGGCCGAGGTGGGTGGATCAC*GCGTGAGCCACTGCGCCCGGCCAGGACTGAC
>AC020656.1--POLH
GCCTGTAATCCCAGCACTTTGGGAGGCCGAG*GCGGGCGGATCACAAGGTCAGGAGATCGAGA
>AC020656.1--POLH
GTCTGGGTTATTTCTTTGATAAGAACTGAAT*TGGCCGGGCGCAGTGGCTCACGCCTGCAATC
>AC020656.1--RPL37
GTGGATCACAAGGTCAGGAGATCGAGACCAT*CCTGGCCAACATGGCGAAACCCCGTCTCCAC
>AC020656.1--XACT
CGTGGTGGCGGGCGCCCGTAATCCCAGCTAC*CGGGAGGCTGAGGCAGGAGAAGGGTGTGAAC
>AC020656.1--XACT
CGTGGTGGCGGGCGCCCGTAATCCCAGCTAC*CGGGAGGCTGAGGCAGGAGAATCACTTGAAC
>AC020656.1--XACT
CGTGGTGGCGGGCGCCCGTAATCCCAGCTAC*CGGGAGGCTGAGGCAGGAGAATGGCATGAGC
>AC020656.1--XACT
GCGGGCGCCCGTAATCCCAGCTACTCGGGAG*GGCTGAGGCAGGAGAATGGCGTGAACCCGGG
>AC020656.1--XACT
GCGGGCGCCCGTAATCCCAGCTACTCGGGAG*GGCTGAGGCAGGAGAATGGCGTGAACCGGGG
>AC020656.1--XACT
GCGGGCGCCCGTAATCCCAGCTACTCGGGAG*GGCTGAGGCAGGAGAATGGCGTGAACCTGGG
>AC020907.3--SYT11
TCTCTATCAAAAATACAAAAAAATTAGCCAG*GTGTGGTGCTGCACGCCTGTGGTCCCAGCTA
>AC021517.3--SLC14A2-AS1
TCTGGAGAAAGAAGTACAGAACGCTAACAAG*GAAATCCCATCTTCCCAGAATTCAGCTAGGA
>AC022210.1--KCMF1
GCTTTTCAGAGGGCTTCTACAGAGCCCACAG*GTGTCAGCTGTGATGCATGTTTAAAAGGAAA
>AC024560.1--AC026412.4
TCCGGGCACTGCCAGGGGCCACGGACCCCGC*ATCCTTTCCAGGCTGTTCCGCAGCATCTGCC
>AC024940.1--AC092666.1
TCTTGGACACCCCATACCCACAGTCCTCCAG*GGTAGGAGTTCATTCGAATAAAGTTCACTTA
>AC024940.1--SSUH2
TCTTGGACACCCCATACCCACAGTCCTCCAG*AAAGCAGTTCCCTCCTTTGTGCAGTGCAGCT
>AC053503.3--MYH6
TCAGAGGCTGTGAGGGGAGGACTGAGGCTGG*ATGAAATCATCGCTAAGCTGACCAAGGAGAA
>AC053503.3--MYL2
CAGAGGGTCTCTGTCTTTAGAGCACTTCATG*CTGCTGCGCCCTGGGCTCGTCTTTGCAGAGT
>AC053503.3--NPPA
GATGGGCTATGTCGCTGTTGGGGACCACGAG*GGACAGACGTAGGCCAAGAGAGGGGAACCAG
>AC053503.3--NPPA
GCTATGTCGCTGTTGGGGACCACGAGGGATG*GCCAAGAGAGGGGAACCAGAGAGGAACCAGA
>AC053503.3--NPPA
TGGCCAGTGAGGAGGGGTCAGAGGCTGTGAG*GGGAGAGACAGAGCAGCAAGCAGTGGATTGC
>AC053503.3--TPM1
AGGGGTCAGAGGCTGTGAGGGGAGGACTGAG*TTTGCGGAGAGGTCAGTAACTAAATTGGAGA
>AC053503.3--TPM1
TTGGGGACCACGAGGGATGGCCAGTGAGGAG*GTCAGTAACTAAATTGGAGAAAAGCATTGAT
>AC063979.1--LINC02056
AGCCAAGAAGAACCTCTGGAGCAATCAGTCT*AAATTAGTTGATAACGTCTTCCAGGAGACCT
>AC063979.1--LINC02056
AGCCAAGAAGAACCTCTGGAGCAATCAGTCT*GAGACCTACGGCCATCCTACTGATATGAACC
>AC063979.1--LINC02056
AGGTGGAGTCTCCCCCACAGCCCTGGACCAG*AAATTAGTTGATAACGTCTTCCAGGAGACCT
>AC073585.1--DOCK1
TCCTGGAGCAGAGGAGCCATTATTTATGCAG*CTTTTTATAACTATGATGCCAGAGGAGCGGA
>AC073585.1--PNLIP
TTTTAACTCTAATGGGAATAACTTGCAACTA*AAAGACCCCTCCATATATTGCCTTGGTCTCC
>AC083899.1--AC133644.2
TGGCTATGATCTACTTAAAAACCAATAACAG*GCTACAGAGTGCGGTGGCGCCAGCACAGAAC
>AC083899.1--AC133644.2
TGGCTATGATCTACTTAAAAACCAATAACAG*TGTCTTCTAGGAAGCCCAGACACATGGAGAA
>AC087518.1--LINC02866
GGGAAAAGACCTCTTGACCTACCAACAAACA*GTCACTGACACTGTCAAGTTCAGAGGTGGTG
>AC091167.1--PERP
CCTTGGCTGCTGTGTTCCAGATCATCTCCCT*GGTAATTTACCCCGTGAAGTACACCCAGACC
>AC091390.3--AC004980.4
AATGGAAGGTGAATTTCCATGTCAACATAGT*ACAGAACCTGCTAAGGCCATCAAACCTATTG
>AC091390.3--AC004980.4
GAAGGTGAATTTCCATGTCAACATAGTCCAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>AC091390.3--AC211429.1
CCAAGAGCAGCAGAAAATGAACACATTGCAG*GGGCCAGTGTCATTCAAAGATGTGGCTGTGG
>AC091390.3--AC211486.3
GAAGGTGAATTTCCATGTCAACATAGTCCAG*CAGAACCTGCTAAGGCCATCAAACCTATTGA
>AC091390.3--PMS2
GAAGGTGAATTTCCATGTCAACATAGTCCAG*ATCTAAAGCTTAAGGACTATGGAGTGGATCT
>AC091390.3--PMS2
GAAGGTGAATTTCCATGTCAACATAGTCCAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>AC091390.3--POM121
CCAAGAGCAGCAGAAAATGAACACATTGCAG*GGGCCAGTGTCATTCAAAGATGTGGCTGTGG
>AC091390.3--POM121C
CCAAGAGCAGCAGAAAATGAACACATTGCAG*GGGCCAGTGTCATTCAAAGATGTGGCTGTGG
>AC091390.3--RBAK
CGTGGGGCAGAGGCTGCGGAGCCCCAGGAGG*GTCTACCAGCCACAGTCTCTGCACGTTTCCA
>AC091982.1--G3BP1
CAATAACAGATTCTGGAAAGAAGGATTTGTA*GTTTGGACATATTTGACTCTTTTCCCCCCAG
>AC092666.1--AC024940.1
AATTGGACACCTAACACCCACAGTCCTCCAG*GTACCTGCCGTGGGCAGATCTGAGCTTTCTT
>AC092666.1--AC024940.1
ACTTGGACAACTAACACCCACAGTCCTCCAG*GTACCTGCCGTGGGCAGATCTGAGCTTTCTT
>AC092666.1--AC024940.1
ACTTGGACACCTAAAACCCACAGTCCTCCAG*GTACCTGCCGTGGGCAGATCTGAGCTTTCTT
>AC092666.1--AC024940.1
ACTTGGACACCTAATACCCACAGTCCTCCAG*GTACCTGCCGTGGGCAGATCTGAGCTTTCTT
>AC092666.1--AC024940.1
GACCCATGTCTTATTGTAAATCCCACGCAAG*GTACCTGCCGTGGGCAGATCTGAGCTTTCTT
>AC093512.2--ALDOB
CTTGAAGCGCTGCCAGTATGTGACCGAGAAG*GTCCTGGCTGCTGTCTACAAGGCCCTGAATG
>AC093512.2--ALDOB
TCTGGCCCGTTATGCCAGTATCTGCCAGCAG*AATGGACTGGTACCTATTGTTGAACCAGAGG
>AC093512.2--ALDOC
TCTGGCCCGTTATGCCAGTATCTGCCAGCAG*AATGGCATTGTGCCTATTGTGGAACCTGAAA
>AC093787.2--AC233263.7
TACAATCCACAGGCAAGGATGTGGCGATTCC*TTTTGCAATATTGGAGGGAATGCCAAACCCT
>AC098828.1--LAPTM4A
CTCTTCGCGGGATCGTTGGCCACAATAGCAG*GTAGTAAACCTATTGATGGCAATTTTGCTGA
>AC109583.4--PRSS46P
GAAAAGGCCTGCGTGAAGGCACAGAATCCAG*GTTGACTCTGGAAGCTCCCTTGTCTGCCAAA
>AC115220.2--LINC01005
TTTGGATGGAATGCTCATCACCTACAGCAGT*CAGGTTGTATTGTGCCATATATCTGTGTCCA
>AC118758.2--LINC00174
GGCTTAACACGGCCACTGATTCTACATTATG*CCTCAGCCAAGATGTCTCACCCTACTCTCTA
>AC122688.3--H2AZ2
CAGCCTGGGTGACAGAGTGCGACTCTGTCTC*AAAAAAAAAATCCCAACTTTTAGTAGTCTCT
>AC132217.2--H19
TTTCTCCCGCAACCTTCCCTTCGCTCCCTCC*CGTCCCCACCACCTCCCTCTTCTTCTTTTTC
>AC133644.2--PLG
GAGGGTCTCGTCATGGAAGGTTCCAGCAAAT*GTCAAGGAGAGCCTCTGGATGACTATGTGAA
>AC134980.1--OR4N2
ACAGAGTGGTGTTATCACAGAAAAGAACAAG*TGGATTTTTATCTGAGATCTACAACTTAAGA
>AC134980.1--OR4N2
GACTAAGATTCACCTGCAGGTGATTCATTTG*TCCTGAAGATTTTCCTCAGAGAATCTTTCAT
>AC134980.1--OR4N2
GAGAAGAGATTTCAACAAAAGTTTGGAAAAG*GCAGAAGAAATAAAGATAGCAAACAACACAG
>AC134980.1--OR4N2
GAGAAGAGATTTCAACAAAAGTTTGGAAAAG*GCCAGGGAAATGGAAAGCGAGAACAGAACAG
>AC138123.2--CAP1
CCCTCCAGTCCAGAAAAAAAAAAAAAAAAAC*AAACCCACATGATTTCAAGGAGTCTGGCATT
>AC138409.2--AC138866.1
CGAACACCCCGCCTACCTGTACTCGTGGGAG*GATGGTGATTGCTCACACCAAAGCCTTGGAC
>AC138409.2--AC138866.1
GTAAAACCAGCCTTAGACTATTTTTCAAACA*GATGGTGATTGCTCACACCAAAGCCTTGGAC
>AC138409.2--AC138866.1
GTAAAACCAGCCTTAGACTATTTTTCAAACA*GGCTGGTGAATTACCAGATCTCCGTCAAGTG
>AC138409.2--NAIP
TGGAATTTTGCCGATTTCATGACTAACCAGT*ATGTGAATTTCTTCGGAGTAAGAAATCCTCA
>AC138866.1--AC138409.2
TGGAGTGGAAACGCTTGTAGGGCTTCACCAG*GGCTGGTGAATTACCATATCTCCGTCAAGTG
>AC138866.1--AL513548.1
TGGAATTTTGCCGATTTCATGACTAACCAGT*ACAATGAGTAGCCACTGCTGTTTGCCCTGGC
>AC138866.1--NAIP
TGGAATTTTGCCGATTTCATGACTAACCAGT*ATGTGAATTTCTTCGGAGTAAGAAATCCTCA
>AC140479.4--MALL
CTGGAAGGATACAAAAGACATCTGCACCATG*ATATTTGGGTTCTTGGTCTGGACCATGGTAG
>AC140479.4--MALL
GCAGCCCGGCGAGGAGCCGCAGACCCATGAG*ATATTTGGGTTCTTGGTCTGGACCATGGTAG
>AC211429.1--AC091390.3
CCAAGAGCAGCAGAAAATGAACACATTGCAG*GGGCCAGTGTCATTCAAAGATGTGGCTGTGG
>AC211476.4--CCDC146
CAGCCATGCTTCCTGTACAGCCTGCAAAACT*TTACATGCTATGGGAAAACTTCCTGGAACCA
>AC211476.4--CCDC146
CCATAAGGAATTTCAAAGGAATATTAAGAAG*AATCGTGAAAAATGGAAGACAGTAGCACAGA
>AC211476.4--CCDC146
CCATAAGGAATTTCAAAGGAATATTAAGAAG*TTACATGCTATGGGAAAACTTCCTGGAACCA
>AC211476.4--PMS2
AACAGTCTGGATGCTGGTGCCACTAATATTG*ATCTAAAGCTTAAGGACTATGGAGTGGATCT
>AC211476.4--PMS2
CAGCCATGCTTCCTGTACAGCCTGCAAAACT*TACAGAACCTGCTAAGGCCATCAAACCTATT
>AC211476.4--PMS2
CCATAAGGAATTTCAAAGGAATATTAAGAAG*AACCTGCTAAGGCCATCAAACCTATTGATCG
>AC211476.4--PMS2
CCATAAGGAATTTCAAAGGAATATTAAGAAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>AC211476.4--PMS2
GCCGGTGGTACTGAGTCTAAGCACTGCGGTG*AAGGAGTTAGTAGAAAACAGTCTGGATGCTG
>AC211476.4--PMS2
GGAAAGCTCTGAGCTCACTTTGTGCACTGAG*CGATGTCACCATTTCTACCTGCCACGCATCG
>AC211476.4--PMS2
GGGGTAGAAGAAGAAAACTTCGAAGGCTTAA*CGATGTCACCATTTCTACCTGCCACGCATCG
>AC211476.4--PMS2
GGGGTAGAAGAAGAAAACTTCGAAGGCTTAA*CTCTGAAACATCACACATCTAAGATTCAAGA
>AC211476.5--PMS2
GCCGGTGGTACTGAGTCTAAGCACTGCGGTG*AAGGAGTTAGTAGAAAACAGTCTGGATGCTG
>AC211476.5--PMS2
GGGGTAGAAGAAGAAAACTTCGAAGGCTTAA*CTCTGAAACATCACACATCTAAGATTCAAGA
>AC239859.5--LINC02802
GGGAGGCTCCAAGACCTCATCAACAATTAAG*ACTCTTTGGCAGTTGCACTGGGTACTGGCTG
>AC239859.5--LINC02802
TCAGTCTGCGGCCAAGACTCCGTCCTGCAAG*ACTCTTTGGCAGTTGCACTGGGTACTGGCTG
>AC239859.5--LINC02802
TCAGTCTGCGGCCAAGACTCCGTCCTGCAAG*GATATTGTGGCTGGAGACATGAGCAAGAAAA
>AC245452.4--IGLVIV-66-1
TCCCAACACTAAACTCTGTGCCCAGGAGAAG*GTTCCCTCTCTCAGCCTTTGCTTATTCAGCT
>AC245748.1--ZNF806
CCAAAAGGAAGACGGTATGATTAAGTTCCAG*GAAAGGGTGACATTCAAGGACATAGCTGTTA
>ACAP2--KPNA4
TGTGGCAGAATTGGAACTAAAACTTGATAAG*ACTATGAGAAGACAACGAAATGAAGTTGTAG
>ACBD5--SMCHD1
GCCTCAGCCTCCCAAGTAGCTGGGACTATAG*TCCATTCCAGTGATTAATGGAAGAGATTTAC
>ACTB--HMGB1
CGATCCGCCGCCCGTCCACACCCGCCGCCAG*AAAAATAACTAAACATGGGCAAAGGAGATCC
>AGAP11--AL512662.1
AAGAAGTATAGCATTGCTCAAATAACCTGTG*GTTAGAGTTACTTGTTATTGGTAAATAGCCA
>AL021368.4--AC138866.1
ACAGCTGTGGAAAGAAGTCACAAAACACTGG*GGCTGGTGAATTACCAGATCTCCGTCAAGTG
>AL033527.2--AL033527.3
AGACCCCGGCGGGGGAGCCTAATAATGGCAG*ATCCATTATTTATTCTTCTCCATCTTGTTCT
>AL049869.2--ZBTB25
AAGCGCCCACTCTACAGAAGAAGGCATCATG*CGGCCGCAACAGGAACCAGAGCTGCGGAGAT
>AL049869.2--ZBTB25
CACCTTTTCTGGAAGCCAGATCAAGTTTCAA*AACCACAATGGACACTGCCAGCCATAGCCTT
>AL050303.2--LINC01087
AAGGCAATGCAGGCAATGGAAAAATTAGTAC*AAATCCCTTTACAGGTTAGTTTGTAAAATCA
>AL050303.2--LINC01297
GCAATGGAAAAATTAGTACAAATCCCTTTAC*AGTAATGATTCTCACTGTGGAGTCATTTGAT
>AL078621.1--AC016588.2
AGTCATCTATTGTAAGACTTTCTGATTCCTT*TTTTTTTCCTTTGGTCATTATGAATAGGGCT
>AL117335.1--AL109809.4
CTGCCCCAAGAGTTGCTGCTCACCAGGCTGG*AAGGATTTGCTGAGCACCTACTATGCATCAG
>AL121578.1--SYTL5
ATTTCCTACATCTGAGGCATCTGTGTAAAAG*TGTTTTCTGCATTCTCTCCCAGCGGATCTGA
>AL121578.1--SYTL5
GGGTGAGCCACCAAACCAACCTGACATTGAG*TGTTTTCTGCATTCTCTCCCAGCGGATCTGA
>AL138963.3--HBA2
CATCAATGTTACCTTCTGTCCTACTGACCAT*CTTCCCCACCACCAAGACCTACTTCCCGCAC
>AL157834.4--CYP2C8
GATGACACTGACTTCCATGAACTGTTTTCCT*CTTCACCCTGTGATCCCACTTTCATCCTGGG
>AL157834.4--CYP2C8
GATGACACTGACTTCCATGAACTGTTTTCCT*TTCTCAAAAGTCTATGGTCCTGTGTTCACCG
>AL158066.1--THSD1
AGAGTATATTTAAAATTTTTATTAAGGCCCA*TGGGGCAGCCCTTGAGGAAACCAGACCTTGG
>AL355075.4--CCNB1IP1
CCTAACAGGGCTCTCCCTGAGCTTCGGGGAG*GTTGAAGATGAAATCCACTGAGGAGGGAAGT
>AL355312.5--AL355312.2
CCAGAGAGTCAAATCATTCCTGTGATGCCAG*GGTCTCACTTTGTCACCCAGGCTGGAATGCA
>AL355312.5--AL355312.2
GGAGGAGGGGAACACCCAGCCCCAGAAACAG*GGTCTCACTTTGTCACCCAGGCTGGAATGCA
>AL358777.3--PAK1IP1
GACTCACGTCTCCAACAACCAAGCTCCCCAA*CAATGGACTCTTGTGGCTGACTTCACTCACC
>AL365295.1--SENP6
GGACATCTGGCATACTGCTATCAACCAATAT*GAAAGAATACCCACCTCATGTCCAAAAAGTT
>AL445669.1--PPIH
ACAGGCCCAAGAGGCCTTTACCTTTCTTGGC*TCTGTTCTCTTAGTATAGCTTGCTCGACTTC
>AL512590.2--CCDC180
ACCAAATACTGGTGCCTAAATCAATCGGCAG*AGTCCCTTCGGATTTGCGCCATGCGCGGCGG
>AL512662.1--BMS1
AGCGGACCTGAGGAGTTGCAGCTGTGAGCAG*GTTAGAGTTACTTGTTATTGGTAAATAGCCA
>AL590556.3--ATP1A1-AS1
TGCCCGTGGTGGACTATAAGCACTGCTCCAG*GAGAAACCCCACCAGTTCTTAACCTGAATGA
>AL590556.3--INS
ACAGGCTGCATCAGAAGAGGCCATCAAGCAG*ATCACTGTCCTTCTGCCATGGCCCTGTGGAT
>AL590666.1--CRABP2
GTGCCCTCAAGCAACAAGAAAGGTGCACCCA*AACCTGACGACCCGGCGACGGCGACGTCTCT
>AL591848.2--AL591848.1
GATTCACACGCATCTGGTAGCTTGGTTGTTG*ACTGATACAAGTATGGACCCCCTAGAAGGAA
>ALDOB--AC093512.2
CCTGGAACACTGCCAGTATGTTACTGAGAAG*GTGCTGGCTGCTGTCTACAAGGCTCTGAGTG
>ALDOC--AC093512.2
GCTGGCCCGTTATGCCAGTATCTGCCAGCAG*AATGGCATTGTGCCCATCGTGGAGCCTGAGA
>ANAPC1--AC133644.2
TGGCTATGATCTACTTAAAAACCAATAACAG*GCTACAGAGTGCGGTGGCGCCAGCACAGAAC
>ANAPC1--AC133644.2
TGGCTATGATCTACTTAAAAACCAATAACAG*TGTCTTCTAGGAAGCCCAGACACATGGAGAA
>ANAPC1--ANAPC1P2
AAGAAATGTTTTGAAATAGAGCTTGAGATTC*TTGTCTTGCCCTCTACTTGAACAGTGTCTTC
>ANAPC1--ANAPC1P2
TGGCTATGATCTACTTAAAAACCAATAACAG*TGTCTTCTAGGAAGCCCAGACACATGGAGAA
>ANAPC1P2--ANAPC1
TGTTTCTCGGGTTGGAAAAGCCATTGATAAG*GATTCACTTTAAGAGATTTGGAAACTCTTCC
>ANXA1--AC012485.1
CATGCTGCTTTAAAAAAACAAACAAACAAAA*AAACAAAAAAACACTGGCGCTTGGGTCCGTC
>ANXA1--RBM39
CAAAGCAGCATATCTCCAGGAAACAGGAAAG*GATGAGAACAAGTTGAGCAGTGCCAACGGCC
>AP000812.4--FOLR3
GGAACCCCATGGAGGAAGAGAAAGGGTATAG*ACCATGAGCTCCTTGAGGGCAGGGACTGAAT
>APOA1--DPYS
GATGAGCAAGGATCTGGAGGAGGTGAAGGCC*TGAGGCAGGAGATAAGCAGGACTTGTTTACT
>ARF1--MCM3AP
CAACGCCTGGCTCGGAGCAGCAGCCTCTGAG*CTTGTTCTAAAGACGAAGGGGAACTCCCTAG
>ARHGAP11B--AC091057.1
AGTATTTGCGGATAGCTCCAAACTGGGATAT*TGTTCTCAGGCGCTGTGGTCTCACCTTCCAC
>ASB15-AS1--NDUFA5
GAGGAAGAGGAGCTCCCCTAGTTACATTCAG*GAACCAGATGTTAAAAAATTAGAAGACCAAC
>ASS1--RIPOR2
GAGAACCGCTTCATTGGAATGAAGTCCCGAG*TTTTTACAGAGTCAAGAGGATCCTATGAATT
>ATP1B1--USP7
GGAATAAAACTGGCATGGTAATTTTTTTTTT*TTGTGTGTTTTTTGTTTTTTGTTTTTTGTTT
>ATP6V1E1--TG
ACCTCGTCCTCCCCAAGTGCTGGGATTACAG*AAACGAGATGCACTGGGGAACTCAAAGGCCA
>AZU1--ZNF540
GATTGAGCATTTTCACGGTAGGAGAAACAGT*ATCTTTTTTTTTTTTTTGAGACAGAGTTTAG
>B2M--DDX5
CTCTCTCTTTCTGGCCTGGAGGCTATCCAGC*GTTTGGTGCACCTCGATTTGGAGGAAGTAGG
>B4GALT1--SMU1
GCTGGACTATGGCATCTATGTTATCAACCAG*TGTGATCCGCCTTATTATGCAGTACTTGAAG
>BCL11B--PIGR
ATCTCGGCTCACTGCAACCTCCGCCTCCCGG*GTTGACACCATTCTCCTGCCTCACCCTCCCA
>BCL11B--PTP4A2
TATATATGTGTGTATATATATATATATATAT*ATATAATATATATATATATATATATATTTTC
>BCL11B--RPL3
AAGCGGGTGCTCACCCTCCGCAAGTCCTTGC*TGGTGGGAACCAAGAAGCGGGTGCTCACCCT
>BCL11B--RPL3
ACTATGGTGAAGTGACCAATGACTTTGTCAT*GCAGAAAGGCTACCATCACCGCACTGAGATC
>BCL11B--RPL3
AGAAGATTGACCTTAAGTTCATTGACACCAC*ACGGCAAGCTGATCAAGAACAATGCCTCCAC
>BCL11B--RPL3
ATGGAGGAGAAGAAAGCATTCATGGGACCAC*GAAGATTGACCTTAAGTTCATTGACACCACC
>BCL11B--RPL3
CAAGAAGATCTATAAGATTGGCCAGGGCTAC*CATCACCGCACTGAGATCAACAAGAAGGTGA
>BCL11B--RPL3
CACCGCACTGAGATCAACAAGAAGATCTATA*AGAAAGGCTACCATCACCGCACTGAGATCAA
>BCL11B--RPL3
CACTGAGATCAACAAGAAGATCTATAAGATT*CAGGGCTACCTTATCAAGGACGGCAAGCTGA
>BCL11B--RPL3
CAGGAAGGCTGTGAGGTGGACCGGCTTGGAC*CATTGTAGAGACACCACCCATGGTGGTTGTG
>BCL11B--RPL3
CATGGAGGAGAAGAAAGCATTCATGGGACCA*AAGATTGACCTTAAGTTCATTGACACCACCT
>BCL11B--RPL3
CCATGGAGGAGAAGAAAGCATTCATGGGACC*GAAGATTGACCTTAAGTTCATTGACACCACC
>BCL11B--RPL3
CTCTGGGTGGCTTTGTCCACTATGGTGAAGT*GAACAATGCCTCCACTGACTATGACCTATCT
>BCL11B--RPL3
CTCTTCACCTTCCCACGATGCCTGCTGCTGC*GCTTCCTGCCTCGGAAGCGCAGCAGCAGGCA
>BCL11B--RPL3
CTGCTGCTGCGCTTCCGAGGCAGGAAGCCGA*GGGAAGGTGAAGAGCTTCCCTAAGGATGACC
>BCL11B--RPL3
GACGGGTCATCCTTAGGGAAGCTCTTCACCT*TCCCTCGGCTTCCTGCCTCGGAAGCGCAGCA
>BCL11B--RPL3
GCCTCCACTGACTATGACCTATCTGACAAGA*AGATTTATAAGATTGGCCAGGGCTACCTTAT
>BCL11B--RPL3
GCTGAAAGGCTGTGTGGTGGGAACCAAGAAG*ATTTATAAGATTGGCCAGGGCTACCTTATCA
>BCL11B--RPL3
GGGAGCGGAGAACTTTCTGTGAGACATCACG*CATGGTGGTTGTGGGCATTGTGGGCTACGTG
>BCL11B--RPL3
GGGTCATCCTTAGGGAAGCTCTTCACCTTCC*CTCGGAAGCGCAGCAGCAGGCATCGTGGGAA
>BCL11B--RPL3
GGTGCAGACGAAGCGGCGGGCTCTGGAGAAG*ATTTATAAGATTGGCCAGGGCTACCTTATCA
>BCL11B--RPL3
GTCTGGGAGCGGAGAACTTTCTGTGAGACAT*CACCCATGGTGGTTGTGGGCATTGTGGGCTA
>BCL11B--RPL3
TGAGATCAACAAGAAGATCTATAAGATTGGC*CAGAAAGGCTACCATCACCGCACTGAGATCA
>BCL11B--RPL3
TGGGACCACTGAAGAAAGACCGAATTGCAAA*AGAAGATTGACCTTAAGTTCATTGACACCAC
>BCL11B--RPL3
TGTAGTCATGCCAGCCTTGTATCCCAGGAAG*GCTGTGACCATTGTAGAGACACCACCCATGG
>BCL11B--RPL3
TTCCCACGATGCCTGCTGCTGCGCTTCCGAG*GGAAGGTGAAGAGCTTCCCTAAGGATGACCC
>BCL11B--ZNF880
AAACAAAAAACAAACAAACAAACAAAAAAAA*ACAAATGTAATGAATGTGGCAAAGCATTTAG
>BCLAF1--LGALSL
AAATTCCCGGAACCGCGATTCGGCGTGTCAG*AAACTAGATGATGGCCATTTAAACAACTCTT
>BHLHA15--AC018630.4
CCCCAAGCTCTACCAGCACTACCAGCAGCAG*CAAGGTCCACCACCTCCTCAGGGAAAGCCAC
>BHLHA15--PRH2
CCCCAAGCTCTACCAGCACTACCAGCAGCAG*CAAGGTCCACCACCTCCTCAGGGAAAGCCAC
>BHLHE40--LTF
ATCTGAGGCATGGAGAGCAGATTCAGGGTGT*GTGTGTGTGTGTGTGTGTGTGTGTGTGTGTG
>BRD2--RACK1
TGGTTTTTTGGTGTGGTTTTGTGTTTTGTTT*TTTTTTTGTTTGTTTGTTTTTTGTTTTTTTT
>C12ORF42--AC068643.1
GGCCATTTGCTGCTCCTACCTCAAATAGAAG*GCGACTTCTTGGCAGATACTGCATCTTCAGC
>C6ORF201--AL136309.3
AAGAGCCTCCAATGTCCACCATAGTTGTCAG*GCTATACAAGAAGAAGCATGGCACCAGCATC
>C6ORF201--AL136309.3
CTGCCAACAACAGCTAAAGGGACACAAACAG*GCTATACAAGAAGAAGCATGGCACCAGCATC
>CALR--APOA1
AATTTTGTGTCTCAGAGACTGCGAGAAGGAG*GTCCCCCACGGCCCTTCAGGATGAAAGCTGC
>CAP1--AC138123.2
CCCAGCACCTTCCTATAGAGATGACTTTAAA*AGAAAAAAAAAAAAAAAAACAAACCCAGTAG
>CAPZA1--MOV10
AATCATAGAGAATGCAGAAAATGAGTATCAG*TCTTGCCTCCAGCAAATCCATCCTCCAATCC
>CAPZA2--H2AZ2
GAACCCGGGAGGCGGAGCTCGCAGTGAGCCG*AGATGGCGCCACTGCACTCCAGCTTGGCGAC
>CCDC88C--MUC7
CGTGAGAAGGAGGAGCTGAGGAAGAACGTGG*AGCTGCCCCACCCACACCTTCTGCAACTACA
>CCND3--SDC3
TTTTGTTTTTTTTTTGTTTGTTTGTTTGTTT*TTTTTTTAACACTTCCTGTGCTGTGCCCATT
>CD24--EEF1A1
GATCAGTGTACCATTTGCCTCCCGGGCTCAA*GCCTGTAATCCCAGCACTTTGGGAGGCCGAG
>CD24--LINC02321
ATTTGCCTCCCGGGCTCAAGCGATTCTCCTG*CCTGTAATCCCAGCTACTTGGGAGGCTGAGG
>CD24--LINC02321
GCTGGTTTCGAACTCCTGACCTCAGGTGATC*CCACCCGCCTCGGCCTCCCAAAGTGCTTGGG
>CEACAM5--AC020656.1
TCGCTTGAACCCGGGAGGTGGAGGTTGCAGT*GAGAATGGCGTGAACCCGGGAGGTGGAGGTT
>CEACAM5--LINC01553
CACAGTGAAACCCCGTCTCTACTAAAAATAC*AAAAAAAAGTTAGCCGGGCGTGGTGGTGGGT
>CELA2A--TRIP12
GGTCTCCAATTACATCGACTGGATCAATTCG*ACAGAGTCTCACTCTGTCACCCAGGCTGGAG
>CHMP1A--TG
CCTCGGCACCCCCAGAGTGCTGGGATTACAG*AAACGAGATGCACTGGGGAACTCAAAGGCCA
>CLCC1--GBP6
ATGTTGGCCAGGCTGGTCTCGAACTCTTGAC*CTCAAATGATCCACCCGCCTCGGCCTCCCAA
>CLCC1--WFDC3
CCTCAGCCTCCCAAGTAGCTGGGATTTACAG*GGAGGAAAAGAGATTGCCCTAGGGTTATTCG
>CLCC1--WFDC3
CCTCAGCCTCCCAAGTAGCTGGGATTTACAG*GGCGGGGCGGTGATTGTCCAAAAGTTCTGGT
>CLCN6--EXOSC7
CCCTCTGGTTCCTCTCTGGTTCCCCTCTCTT*GGCCTACCTCCTCCTCTGCCTCTTCTCCCTC
>CLCN6--PBX1
CCTCTCTTGGCCTACGTCTGTCCCTGTCTCC*CTCTTTCCTGCTCTTTGTTTTTCTGCCCAGA
>CLEC2D--NPM1
ATTTCTGTAACAGTTGATATCTGGCTGTCCT*TTTTTATAATGCAGAGTGAGAACTTTCCCTA
>CLEC2D--NPM1
CGTCTGCCTTGTCTCCTACCTAAGTGTGTGT*CGCCACCCGATGGAAGATTCGATGGACATGG
>COL4A2--AC132217.2
ACACGTAGACACACACATGCACCCAGCACAC*GCACACACAGCACACACACAAACACAGCACA
>COPB1--RRAS2
AGGATGTGAACTGTACGTTTGCACACTGACA*TCCTATTTTGTAACGGATTATGATCCAACCA
>COPB2--RNF130
GCCTCGTCCTCCCAAAGTGCTGGGATTATAG*GTGTGAGCCGCACCAGCCTGTTTTTTAATTT
>CPA1--FTL
TAACTACGCCACCTACCACACCCTGGAGGAG*CTCTGTGACTTCCTGGAGACTCACTTCCTAG
>CPSF6--NINJ2-AS1
CCGAGAGCGCGAATATCGTCATCGTTAGAAG*ATGGAGTGAAGGTACCTGAGCGCAGGAACCA
>CSHL1--H19
CATTCCTGCATGACTCCCAGACCTCCTTCTG*CTTCTTTTTCATCCTTCTGTCTCTTTGTTTC
>CST1--AC018630.4
AGGTGTCAAGAATCCTAGGGATCTGTGCCAG*GCCAGCAGCAACAAGGTCCACCACCTCCTCA
>CST1--AC018630.4
GACCATGGCCCAGTATCTGAGTACCCTGCTG*AGCAGCAACAAGGTCCACCACCTCCTCAGGG
>CST1--AC018630.4
TAGCCCTGGTACATGGTACACACCCCCCCAC*CTCCTCCTGGAAAGCCCCAGGGACCACCTCC
>CST2--AC018630.4
CCAGGTGTCAAGAAGCCTAGGGATCTGTGCC*AGGCCAGCAGCAACAAGGTCCACCACCTCCT
>CST4--AC018630.4
AGGGGTCTGTGCCAGGCCAGTCACACCGACC*ACCACCCCAACAGGGAGGCCATCCCCGTCCT
>CST4--AC018630.4
CCACCACCCACTCCCACCCACTGTAGTGCTC*CCACCCCAACAGGGAGGCCATCCCCCTCCTC
>CST4--AC018630.4
CCAGGTGTCAAGAAGCCTAGGGGTCTGTGCC*AGGCCAGCAGCAACAAGGTCCACCACCTCCT
>CST4--AC018630.4
CTTCACTTCGCCATCAGCGAGTACAACAAGG*CCACCACCTCCTCAGGGAAAGCCACAAGGAC
>CST4--AC018630.4
CTTCTAATAGACCTGGTACATGGTACACACA*CCCCACCTCCTCCTGGAAAGCCCCAGGGACC
>CST4--AC018630.4
TGATGAGTGGGTACAGCGTGCCCTTCACTTC*GCCATCAGCAAGGTCCTCCCCCACCTCCTCC
>CST4--PPP1R1B
CTGATGGCTACCCTGGCTGGGGCTCTGGCCT*CCTCAGAGGAGGAGGATGAGCTGGGGGAGCT
>CTSB--LIPG
CGCCTCGCCTCCCAAAGTGTTGGGATTACAG*TCCCACTGTGGAGCTTCCCTGAGGGTGCCCG
>CTSS--ACAA1
GTAGCTGGAATTATAGGCGTGCACCACCACG*CCCAGCTAATTTTTGTATTTTTGGTGGAGAC
>CYP2E1--APOA1
TAAATTGACTGAGAGAGACTGCGAGAAGGAG*GTCCCCCACGGCCCTTCAGGATGAAAGCTGC
>DA750114--CWF19L1
TTTTGAGACGGGGTCTCGCTCTGTCACCCAG*GCAGGATTGCAGTGGCATAACTGTGGCTCAC
>DDX17--AC018630.4
AGTCACCAGGGCCTTTTTTCACTTCCTGTCA*TGAATCCTAGATTACTGAGGAGACTGCCCCT
>DDX17--EEF1A1
CCAAAATGATCCAGTCTGGAATAACATTTTG*AAAAAAAAAATATATATATATATATATATAG
>DDX5--ANXA1
GCCATTCCTAGAGAGAGGCGATGGGCCTATT*TGAAGTGCGCCACAAGCAAACCAGCTTTCTT
>DPY19L2--SORD
ATGGATTGGAAATTTTTCTGATTACTAAAAG*GGACATGGATGAAGCTGGAAGCCATCCTTCT
>DPY19L2--SORD
CAGCTAACCCTCCCTTATGTAGCGTCCTGCT*TGAGCCAAGATTGTGCCACTGCACTCCAGCC
>DPYS--APOA1
CAGGGACGCTGGGAGAGACTGCGAGAAGGAG*GTCCCCCACGGCCCTTCAGGATGAAAGCTGC
>DSG3--SYNRG
AATGATGTAAGAATCACAAGGGGCCGGGCGC*GGTGGCTCACGCCTGTAATCCCCAGCACTTT
>DTX2P1-UPK3BP1-PMS2P11--AC004951.1
CACATCCTCCACGTCCCCAGGCCCCACCCAC*CTCCTCCAACTCCTCTGGGGAAACCCAAGCC
>DTX2P1-UPK3BP1-PMS2P11--AC004951.1
CAGCCATGCTTCCTGTACAGCCTGCAAAACT*AGTAAAGAGGACAAATAGGTGAAAGAATAAA
>DTX2P1-UPK3BP1-PMS2P11--EEF1A1
CTATTAAAAATATAAAAAATACGCCAGACGT*GGTGGCTCATGCCTGTAATCCCAGCACTTTG
>DUS4L-BCAP29--SLC26A4
TCAACTCCTGAAAGAACACTCTGAACTTCAG*GTCATGGCAGCGCCAGGCGGCAGGTCGGAGC
>DYNLL2--PABPC1
AGAGAAGGACATTGCTGCCTATATCAAGAAG*GTGGTTTGTGATGAAAATGGTTCCAAGGGCT
>EDEM3--MNDA
TACAAATCAAAGCATCTCTAAAAAGAACACA*AATCAGGAAACCCAGGCCCAACGGCAGGTGG
>EEF1A1--CHMP3
CAAAAGAACCATTGAAAAATTTGAGAAGGAG*GCTGGGATCATAGAGGAGATGTTAGAGGACA
>EEF1A1--PRB3
CCTAGTCTCTGCTGGTGGCTTTCCTGATACT*ACAGAGTTGGGAGTGACTCCAGAGCCTCCAG
>EEF1A1--SFPQ
AGCCTGGGCGACAGAGCGAGACTCTGTCTCA*AAAAAAAAAAAAAGGTTTAGTACTGAAGTAG
>EEF1A1--SFPQ
CCTGGGCGACAGAGCGAGACTCTGTCTCAAA*AAAAAAAAAAAAAGGTTTAGTACTGAAGTAG
>EEF1A1--TNP1
AGAGCTCATACACAGCAAAAGAAACTACCAT*AAGGGAGTCAAGAGAGGTGGCAGCAAAAGAA
>EIF3E--AC019205.1
AAGATGAAATAATAAAACTATTATATAAAGG*GGTGACTTACATTTTGGAAACAACATATTGA
>EIF3E--AC019205.1
TGGGCAACTCAAGATTCTGGCTTCTACTGAA*GAAACATAAGGAAAAGATTTTAAAAAAACTC
>ELANE--EEF1A1
GCCTGTGTCCTGCCGGCCTTGCTGCTGGGGG*GTGTCGTGAAAACTACCCCTAAAAGCCAAAA
>ETS2--AL590004.3
GATAAGGCCTGTGGGGGGTAATCCCTGCTTT*TTGTGTTTTTTTTGTTTGTTTGTTTTGTTTT
>ETS2--ATP8B4
AAGGCCTGTGGGGGGTAATCCCTGCTTTTTG*TTTTTTTTGTTTGTTTGTTTGTTTGTTTTTG
>ETS2--ATP8B4
TAAGGCCTGTGGGGGGTAATCCCTGCTTTTT*TGTTTTTTTTGTTTGTTTGTTTGTTTGTTTT
>ETS2--NOVA1
GGCCTGTGGGGGGTAATCCCTGCTTTTTGTG*TTTTTTTTGTTTGTTTGTTTGTTTGTTTTTT
>FAM133EP--FAM133B
AAAGCAGTGAAGAACGAGAAAAAGCAACAGA*AAAAAACAAAAAAGAAAAAGAAGCATAAGAA
>FAM133EP--FAM133B
AGAATCAGAGTATATGGAGGAGGTACGAGCA*AAAAAAGAAGAAAAGCAGTGAAGAACGAGAA
>FAM133EP--FAM133B
GTGAAGAACGAGAAAAAGCAACAGAAAAAAA*AAAAAAGAAAAAGAAGCATAAGAAACACAGT
>FAM133FP--FAM133B
TTTTGGCTGAATTTGAAGAAAAAAATGAATG*AGAACTGGAAGAAAGAACTGGAAAAACACAG
>FASN--PRB2
ACAGCCTGGCCTCCCGGCTCAGCATCCCCAC*AAGGAGGCAACAAACCTCAAGGTCCCCCACC
>FAT4--LYZ
GGTTCACGCCATTCTCCTGCCTCAGCCTCCC*AGCAAAATATGGCCTTTCTTCTAAGAGATAT
>FCAR--MARK3
ATTTCTACCGGATTTTCATACAAGGAATACA*GGCGTGGTGGCAGGTGCCTGTAGTCCCAGCT
>FCAR--MARK3
GAGAAAAGCTCACTAAGAAGCTTGAATCTAC*TTTTTTTTTTTTTGAGACAGAGTCTGGCTCT
>FKBP8--AC018630.4
CAGGGGGAGTACAGTGAGGCCATCCCCATCC*TCCTTGTGGCCTTCCTTGAGGAGGAGGGGGA
>FOS--AKAP17A
GAGCATTGGCAGGAGGGGCAAGGTGGAACAG*GCCCAAGGTCCCGGAGGCTATGGCAGCGGCT
>FOSB--HMGB1
CGGCAGTCCACCCACCGCCGCCGCCTCCCAG*AAAAATAACTAAACATGGGCAAAGGAGATCC
>FP475955.3--PSLNR
CTGGACATTGCACTTGGCAATGAATTTATGG*GCGCTTTGGAATCCTGCAGATATAATAATGA
>FUS--EEF1A1
GTGCGCGCGTGCGCGGACATGGCCTCAAACG*GTGTCGTGAAAACTACCCCTAAAAGCCAAAA
>FUT11--CHCHD1
GCAATGTGGAAGAGATTCCTGAGAATGACAG*GAAGCCCGAAAGATGAGATCAATACAGGAAA
>FUT4--AP000943.3
ACTGTGCATCTCCTTGACTGCCGCATCATGG*ATAACTGAGAAGCGGAGCTACTCACTTCGGA
>G3BP1--BRD1
GCCCCAGCCTCCTGAGTAGCTGGGACCACAG*GTTCATCAGAATGCAAAGTAGAAAGTGTCTT
>GBF1--ACTR1A
GCTTCTCCGGAGAGAAGAGATCAGTGCTCAG*AGAGCCTGTTACCTATCCATAAACCCCCAAA
>GBF1--TG
GCCTAGGCCTCCCAAAATGCTGGGATTACAG*AAACGAGATGCACTGGGGAACTCAAAGGCCA
>GLCCI1-DT--AC006042.3
TGGACTTCACAGCCTCCAGAATTGTGAGCCA*ATAATCCACTTTAAAGCAAGAAAACAAGATA
>GLCCI1-DT--AC006042.3
TGGACTTCACAGCCTCCAGAATTGTGAGCCA*GGAAAAATGTATGACTCTAAATTTGGAAACC
>GMFG--LRFN1
CAGGCTGGTGCAGACAGCAGAGCTCACAAAG*GTGCTGGCCCAGGCCCTGACTCCCTGAAGAG
>GP2--AC073585.1
ACATCAACTTCCAATGTGCCTACCCACTGGA*GTTTGATGTTGAGGATGGTGGGGTTGGTGAT
>GP2--PPM1G
AAAATACAAAAATTAGCCAGGCATGGTGGCA*GGCGCCTGTGGTCCCAGCTGCTCGGAAGGCT
>HBB--EEF1G
ACTTTGGCAAAGAATTCACCCCACCAGTGCA*GGCTGCCCCTGCTCCTGAGGAGGAGATGGAT
>HBB--GPR160
GCCTAATAAAAAACATTTATTTTCATTGCAA*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>HBB--PLEKHB2
GCCTAATAAAAAACATTTATTTTCATTGCAA*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>HBB--PLEKHB2
TCTGCCTAATAAAAAACATTTATTTTCATTG*AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>HIF1A--CALM1
AGGGCGCCGGCGGCGCGAACGACAAGAAAAA*GCTGATCAGCTGACCGAAGAACAGATTGCTG
>HIKESHI--CCDC81
TGCTGTATCCTCAGTTGACTCATTCACTCAG*AAGTCTCTATTATCTGGGGGAATGTATCAGA
>HNRNPC--EEF1A1
TGATGATAATGAAGATCGGGGGGATGACCAG*CAGCTCCTTAAAGAAAGAGTACAAATTGTTT
>HNRNPH1--STATH
GAGAAAAGGGGTTAATTGAGGCTTGTTTGGA*ATACGACTTCTACATCCATATTCTCATCTTT
>HP--APOA1
TTCTTTTGACCCAAGAGACTGCGAGAAGGAG*GTCCCCCACGGCCCTTCAGGATGAAAGCTGC
>HP--HPX
ATAAGATGTGGTTTGAAGCTGATGGGTGCCA*GGCACCCAGGTATATGTCTTCCTGACAAAGG
>HTN3--EIF3E
ATCTGTATGACAATTGATATCTTCAGTAATC*ATGCTTCATGATTACTGACCTGGTGAATTTT
>HTN3--TMED10
TTTGATTAGATACTTGCAATTTAAATGTTAA*TTTGGAAGGGAAATATCCTAAAATGGAAGAA
>IFI44L--PDGFA
ATGAAAACTGGAAGTTTTTTTTTGTTTGTTT*TTTTTTTTTTTTTTGTTTTTTTTTTTTTTTT
>IFT172--TG
ACCTTGGTCTCGAAAAGTGCTGGGATTACAG*AAACGAGATGCACTGGGGAACTCAAAGGCCA
>IFT80--MOB1A
GCCTCAGCCTCCTGAGTAGCTGGGATTACAG*GAGTTTAATCTGATTGATAGGCGTGAGCTGG
>IGF2--COL1A2
GGCCCCCGCTCTTTCTCCCGCAACCTTCCCT*CGCTCCCCCAAAAATTTGAATTTTTTTTTCA
>IGH@--AC018630.4
CCCAGCCAGGATGTGACTGTGCCCTGCCCAG*CTCCCCCACCTCCTCCTGGAAAGCCCCAGGG
>IGH@--PGA4
AAGACCTTCACTTGCACTGCTGCCTACCCCG*CATTTCCTCCTCCGGGGCCACACCCGTCTTT
>IGH@--PGA4
ACCCCGAGTCCAAGACCCCGCTAACCGCCAC*CGAGACAGTCTCCATCACCTACGGCACCGGC
>IGH@--PGA4
AGAAGCGAACCTCACGTGCACACTGACCGGC*CTGAGCGAGACGGAACCTGGCTCCTTCCTGT
>IGH@--PGA4
AGCGGACAGGGCGTGACCGCCAGAAACTTCC*AACCCAGCCAGAAAGTACTTCCCCCAGTGGG
>IGH@--PGA4
ATGCTCCGTGATGCATGAGGCTCTGCACAAC*CTCAACCCAGCCAGAAAGTACTTCCCCCAGT
>IGH@--PGA4
CACGTGAAGCACTACACGAATCCCAGCCAGG*GTACTTCCCCCAGTGGGAGGCTCCCACCCTG
>IGH@--PGA4
CCAAGGTCTTCCCGCTGAGCCTCTGCAGCAC*CCAGCCAGAAAGTACTTCCCCCAGTGGGAGG
>IGH@--PGA4
CGTGGTCATCGCCTGCCTGGTCCAGGGCTTC*TTCCCCCAGTGGGAGGCTCCCACCCTGGTAG
>IGH@--PGA4
GAAGACCTTCACTTGCACTGCTGCCTACCCC*GAGTCCAGTACCCCGTGCCACCCAGTGCCTA
>IGH@--PGA4
GCACCCAGCCAGATGGGAACGTGGTCATCGC*CTGCGCTGAGGGCTGCCAGGCCATTGTTGAC
>IGH@--PGA4
GCCCAGACGGCAAGTCCGTGACATGCCACGT*GAAGCACAACCTCAACCCAGCCAGAAAGTAC
>IGH@--PGA4
GCCTAGCCGGCAAGTCCGTGACATGCCACGT*GAAGCACAACCTCAACCCAGCCAGAAAGTAC
>IGH@--PGA4
GCTGTGCCGAGCCATGGAACCATGGGAAGAC*CTTCCTCCCATCTTGCCTTCTCCCTCGAGTT
>IGH@--PGA4
GGAAGACCTTCACTTGCACTGCTGCCTACCC*CCGACATCGTCTTCACCATCAATGGAGTCCA
>IGH@--PGA4
GGACAGGGCGTGACCGCCAGAAACTTCCCAC*CCTGGTAGATGAACAGCCCCTGGAGAACTAC
>IGH@--PGA4
GTTCAGAAGCGAACCTCACGTGCACACTGAC*CGGCCCAACCAGCCCCATTGCCAACATCCAG
>IGH@--PGA4
TCCAGTTCCCTCAACTCCACCTACCCCATCT*CTCAACCCAGCCAGAAAGTACTTCCCCCAGT
>IGH@--PGA4
TCTCATGCTCCGTGATGCATGAGGCTCTGCA*CCAACCACAACCGCTTCAACCCTGAGGATTC
>IGH@--PGA4
TGCTGCCACCCCCGACTGTCACTGCACCGAC*CACAACCGCTTCAACCCTGAGGATTCTTCCA
>IGH@--TUBA1B
GGTCGGTGCAGTGACAGTCGGGGGTGGCAGC*TGAGGGGATGGAGGAAGGCGAGTTTTCAGAG
>IGHA1--PGA4
AGAAGCGAACCTCACGTGCACACTGACCGGC*CTGAGCGAGACGGAACCTGGCTCCTTCCTGT
>IGHA1--PGA4
GAAGACCTTCACTTGCACTGCTGCCTACCCC*GAGTCCAGTACCCCGTGCCACCCAGTGCCTA
>IGHA1--PGA4
GTTCAGAAGCGAACCTCACGTGCACACTGAC*CGGCCCAACCAGCCCCATTGCCAACATCCAG
>IGHA1--PGA4
TCCACCTACCCCATCTCCCTCATGCTGCCAC*CCAGTGCCTACATCCTGCAGGTGAGGAGGCT
>IGK@--PIPOX
GAGACTCTGTCTCAAAAAAAAAAACAAAAAA*AAAAAAAACCTCTTTGCTGGTATATTTGAAG
>IGK@--PIPOX
GCCTGGGCGACAGAGTGAGACTCTGTCTCAA*AAAAAAAAAACAAAACAAAAAAAACCTCTTT
>IL9R--RPL23AP53
CCTGTGTGCTTCCAGGCTCCCCAGAGACAAG*GACCCACCCGAGGACGGGTCACCATGGCTTT
>ITGB1--DCAF1
GCCTCAGCCTCCCGAGTACCTGGGATTACAG*GCAAAGCCATGACTACAGTAGTGGTACATGT
>ITM2B--SI
TCAACCTCCCCAGGAGTAGCTAGGACTACAG*AATAATAATAATTTATACGGCCATCAAACAT
>KCCAT198--APPL2
AATTCATCTGACAAGAAAAAGAAAGCAAGAG*ACTCGCTCTTTACTGAGCGTGTTTGAAGAAG
>KHDC4--AC020656.1
CCCTCGGCCTCCCAAAGTGCTGGGATTACAG*GGTCAGGAGATCGAGACCATCCTGGCTAACA
>KLF8--IGBP1
AGAAGGTGGCTCAATGCAGGTATTCAAGCAG*GCATCAACTTCTAACTCATCTCGCCAGGAGA
>KMT2E--AHCTF1
GGTTCGGGTGTCTCGTGTGTGAACATCACAG*TGGAAGTATGCGAGACTTAAGAGCTCAAGTG
>KRT4--HSPG2
CCTGGAATGGGAAGGATGTCAGCTGACCTCT*CCCTACCATGGGCAGAGAGGCAGGTATTGAG
>LAMP2--SMAD1
ACCGTGGATTTGTACTTGCTCCTTTTGCTCT*TTTTTTTTTTGTGTGTGTGTGTGTGTGTGTT
>LAMP2--SMAD1
TACCGTGGATTTGTACTTGCTCCTTTTGCTC*TTTTTTTTTTGTGTGTGTGTGTGTGTGTGTT
>LAMP2--SMAD1
TGGATTTGTACTTGCTCCTTTTGCTCTTTTT*TTTTTGTGTGTGTGTGTGTGTGTGTGTGTGT
>LCOR--TG
CACTTGGCCTCTCAAAGTGCTGGGATTACAG*AAACGAGATGCACTGGGGAACTCAAAGGCCA
>LINC00471--AC017104.4
GAGAGCCACCAGGAGAATGAAGACAGGAACA*AACCTATCAAAGAAACACCTGATAAACAAAA
>LINC00471--AC017104.4
GAGAGCCACCAGGAGAATGAAGACAGGAACA*AGACGGGGTTTCACTGTGTTAGCCAGGATGG
>LINC00471--AC017104.4
GAGAGCCACCAGGAGAATGAAGACAGGAACA*AGACGGGGTTTCACTGTGTTAGCCAGGATGT
>LINC00471--AC017104.4
GAGAGCCACCAGGAGAATGAAGACAGGAACA*TAGAGACGGGGTTTCACTGTGTTAGCCAGGA
>LINC00680--AC138866.1
ACAGCTGTGGAAAGAAGTCACAAAACACTGG*GGCTGGTGAATTACCAGATCTCCGTCAAGTG
>LINC00680--AC138866.1
TAGCATTTTGAAAATCACAGAGCAATTTTGG*GCTGGTGAATTACCAGATCTCCGTCAAGTGC
>LINC01031--CEP104
CCTGGGCGACAGAGCGAGACTCCGTCTCGAA*AAAAAAAAAAAAAGTATAAGGAGAGAAATAA
>LINC01297--PSLNR
CTGGACTTTGCACTTGGCAATGAATTTATGG*GCGCTTTGGAATCCTGCAGATATAATAATGA
>LINC01297--PSLNR
TAAAGAGAGGTTATGGAGTCTGTAGCTTCAG*AATTTCTGGTTTTTCCTGTTGGAGCCCAAGG
>LINC01923--NDUFV1
AGCCTGGGTGACAGAGCAAGAGTCTGTCTCA*AAAAAAAAAAAAATGCTTACTAAGTGGCAGT
>LINC02210_--MORC2-AS1
AAGCTCAGGCTCCCACTGATTCTACATTACA*CCTTGTGCAAAGGACCTTCTCACCACAGATC
>LINC02256--AC055876.5
CCCCCTGGTTTCCCGCAGCGATCCCGCGCAG*GGCTGAAGCAGTGCCTACTCAATAGAACCAG
>LINC02321--CD24
CAGGTGATCCACCCGCCTCGGCCTCCCAAAG*GCTGGGATTACAGGCTTGAGCCCCCGCGCCC
>LINC02348--CSDC2
ACACACACACACACACACACACACACACACA*ATCTTCCAGACCCATCCCCTGCCTGCCAGCT
>LINC02381--AC023794.1
CCTAAAGGATTTTCATCAGCTGCAAGTCTGG*ATGACAAGCTGAGGCCTACGATGATTTAGGC
>LINC02802--AC239859.5
GGCCCAGACTCCACTCCCCAGCTGTGAAAGG*ACTGCTGGCCAGACCCCCAAGCTAGCCCGCC
>LINC02802--AC239859.5
TCAGTCTGCGGCCAAGACTCCGTCCTGCAAG*GATATTGTGGCTGGAGACATGAGCAAGAAAA
>LINC02866--AC022733.1
GCCTCAACTTCCCCAGTAGCTGGGACTACAG*CAAATATCTTCCACTGAGATCGTGTGGTTTT
>LINC02866--AC022733.1
GCCTCAACTTCCCCAGTAGCTGGGACTACAG*CCCTGGTCTTGGCTGTTACATAGTGGGATAG
>LINC02883--AC021739.2
CCAGGACCCAGGGCTCCAAAGCCATTTCCAG*GATTTAGCACACAAATGTCCATAGAGCATTT
>LINC02883--AC021739.2
CTTCTGTCCACTCTGCAGAGTGAACGTTGAG*GATTTAGCACACAAATGTCCATAGAGCATTT
>LMX1A-AS2--LMX1A-AS1
ATGAGAGCCATCTGGAACCCGGCAGAGCCAG*CAGGGGCAGACCTGCCTGACTTGGATATCAG
>LRRC37A6P--ACBD5
GACTTGAAGAATGAAGCCGCAGACCCTGGCG*ACAAACCCGAAGCAAAGCCTCCGGAGGGCGC
>LRRC37A6P--ACBD5
GACTTGAAGAATGAAGCCGCAGACCCTGGCG*TTTCATGCAGGCTCTTGGGAAAGCTGGTGCT
>LYZ--CDK10
GTGTTAGCCAGGATGGTCTCGATCTCCTGAC*CTTGTGATCCGCCCACCTCGGCCTCCCAGAG
>LYZ--FAT4
AATACATCTCCAGTACATTCCGTTCTTTTTT*TTTTTTTTTTTTTTTGAGACGGAGTCTCGCT
>LYZ--FAT4
GCTAATTTTTTGTATTTTTAGTAGAGACAGG*GTTCACGCCATTCTCCTGCCTCAGCCTCCCG
>LYZ--FAT4
TACATCTCCAGTACATTCCGTTCTTTTTTTT*TTTTTTTTTTTTTTTGAGACAGAGTTTCACT
>LYZ--IGF2BP1
TGGTTATCAAATACATCTCCAGTACATTCCG*TTTTTTTTTTTTTTGAGACAGTCTTGCTCTG
>LYZ--NCAPH2
CTCCAGTACATTCCGTTCTTTTTTTTTTTGA*GACAGAGTGAGACTCCGTCTCAAACAAAACA
>LYZ--XPO1
GGTTATCAAATACATCTCCAGTACATTCCGT*TTTTTTTTTTTTTTGAGACGGAGCCTGGCTC
>MAEL--AC115220.2
CGAGGATTTCGATTTCATTGTCAGGCTGCAA*ACCTGACTGGCCAGTGTCTCAAAGACGTGGA
>MALT1--STAR
CATGGTGAAACCCCGTCTCTACTAAAATACA*AAAAAATTAGCTGGGTGTGGTGGCGCAAACC
>MARK3--FCAR
GCCTCAGCCTCCTGAGTAAGTGGGATTACAG*GGGACTTTCCCATGCCTTTCATATCTGCCAA
>MARK3--TG
ATCTTGGCTTCCCAAAGTGCTGGAATTACAG*AAACGAGATGCACTGGGGAACTCAAAGGCCA
>MB--RPS4X
AGGTCCTCTTTCCTTGCCTAACGCAGCCATG*GCTCGTGGTCCCAAGAAGCATCTGAAGCGGG
>MFF--PDE6A
TTAATAGCTGGCTCTGGTTTCGCCGCTAGAG*ATACTGCTGTGGTCTGGGAGCAAAGTCTTTG
>MFF--PDE6A
TTAATAGCTGGCTCTGGTTTCGCCGCTAGAG*GTAACATCAGCTCTCAAAAATATTATCTCAA
>MFN2--MB
GTCTCAGCCTCCCAAGTAGCTGGGATTACAG*ACTGCGCCATGGGGCTCAGCGACGGGGAATG
>MFN2--TG
GCCTTGGCCTCTCAAAGTGCTGGGATTATAG*AAACGAGATGCACTGGGGAACTCAAAGGCCA
>MIA2--AL049828.1
TTCCCAGGTCCACCACCTGCTCCATTTGCAA*ATTGCATTTCTCTGGAATGACCACAGCAATA
>MICALL1--RBM39
AAGATGTTGGAAGCCATGATCAAGAAGAAAG*GTACAGGTTTGCAGATTCCGCCAGCAGCACA
>MIR100HG--RAB29
CTGGGCGACAAGAGTGAAACTCCATCTCCAA*AAAAAAAAAAAACAAAAAGAAAAAGATTTTT
>MIR100HG--RAB29
TGGGAGGCTGAGGCAGGAGAATCGCTTGAAC*CTGGGAGTTAGAGGTTGCAGTGAGCTGAGAT
>MIR100HG--RAB29
TGGGAGGCTGAGGCGGGAGAATCACTTGAAC*CTGGGAGTTAGAGGTTGCAGTGAGCTGAGAT
>MRPS31P5--THSD1
AGAGTATATTTAAAATTTTTATTAAGGCCCA*TGGGGCAGCCCTTGAGGAAACCAGACCTTGG
>MT1E--MT1M
AAAGAGTGCAAATGCACCTCCTGCAAGAAGA*GCTGCTGCTCCTGCTGCCCCGTGGGCTGTGC
>MT1M--MT1E
AAAGAGTGCAAATGCACCTCCTGCAAGAAGA*GCTGCTGTTCCTGCTGCCCCGTGGGCTGTGC
>MT1M--MT1E
GAAATGGACCCCAACTGCTCCTGCACCACTG*GTGGCTCCTGCACGTGCGCCGGCTCCTGCAA
>MUC7--AC018630.4
ACACCTTCTGCAACTACACAAGCTCCACCAT*CTCCTCAGGGAAAGCCACAAGGACCACCCCA
>MUC7--AC018630.4
ACAGCTGCCCCACCCACACCTTCTGCAACTA*AAGGTCCACCACCTCCTCAGGGAAAGCCACA
>MUC7--AC018630.4
ACATCCAGATAAAAATAGCAGTGTGGTCAAC*CCTCTGCTGGTGATGGGAACCAGGATGATGG
>MUC7--AC018630.4
ACTACACCAGCTCCACCGTCTTCCCCAGCTC*CACAAGGACCACCCCAACAGGGAGGCCATCA
>MUC7--AC018630.4
CAGCTGCCCCACCCACACCTTCTGCAACTAC*AAGGTCCACCACCTCCTCAGGGAAAGCCACA
>MUC7--AC018630.4
CCAGAGACCACAGCTGCCCCACCCACACCTC*CAGCAACAAGGTCCACCACCTCCTCAGGGAA
>MUC7--AC018630.4
CCAGAGACCACAGCTGCCCCACCCACACCTT*CAGCAGCAACAAGGTCCACCACCTCCTCAGG
>MUC7--AC018630.4
CCCACACCTCCTGCAACTACACCAGCTCCAC*CACCTCCTCAGGGAAAGCCACAAGGACCACC
>MUC7--AC018630.4
CCCACACCTTCTGCAACTACCCTAGACCCAT*CATCCTCCCCCACCTCCTTTTCAAAATCAGC
>MUC7--AC018630.4
CTCCACCGTCTTCCCCAGCTCCACAAGAGAC*CACCAAAACCAGGAGGCCATCACCGCCATCC
>MUC7--AC018630.4
GCCCCACCCACACCTTCTGCAACTACACCAG*CAACAAGGTCCACCACCTCCTCAGGGAAAGC
>MUC7--AC018630.4
GTAGCTACATTAGCACCAGTGAATTCCCCAG*CAGCAACAAGGTCCACCACCTCCTCAGGGAA
>MUC7--NFIB
TTTCTTTTATATATGAAGAATCTACTAAACA*TCCAATTTCAGCTCCTGGGGAAAAGAGTACA
>NAIP--CHEK2
CATTAGACGATCACACCAGATGTTTTCCCAA*TAGAGACGGGGTTTCACCGTGTTAGCCAGGA
>NDUFS1--HGSNAT
CGCAATCTCAGCTCACTGCAACCTCCACCTC*CTGAGGCAGGAGAATCGCTTGAACCCAGGAG
>NEB--KCNK3
AGCCTGGGTGACAGGGCAAGACCCTGTCTCA*AAAAAAAAAAAAAAAAATGGCAAAGGGAGAC
>NEK4--ZBTB5
ACCTCAGCCTTCTGAGTAGCTGGGATTACAG*GATCATGGATTTTCCTGGTCACTTTGAACAA
>NEK9--MB
GCCTCAGCCTCCCGAGTAGCTGGGACTACAG*ACTGCGCCATGGGGCTCAGCGACGGGGAATG
>NFE2L1-DT--COPZ2
GGCGGAACGTTGGCGGTGTGAGAGGTGTGGG*TATTATGATGACACATTCCCCTCCATGAAGG
>NFE2L1-DT--COPZ2
GTGTCTTCTTAGAAGAGCCCATTATTTATAG*TATTATGATGACACATTCCCCTCCATGAAGG
>NKG7--AC008750.7
TGGCCAACAGGGCATGGGGACATCATATCAG*ATCCGAGTGAAGCCTGACAGGACCGGTGTGG
>NMUR1--AC017104.5
AGCCGGGGTCTCGCGGGCCGCGGGCCGCATG*GGGTGAAAGAGTTCTGGACACTCTCCACTGC
>NONO--S100A9
GCCTCAGCCTCCTGAGTAGCTGGGATTACAG*AAGGAGAATAAGAATGAAAAGGTCATAGAAC
>NONO--S100A9
GGCTCAGCCTCCTGAGTAACTGGGATTACAG*AAGGAGAATAAGAATGAAAAGGTCATAGAAC
>NPM1--BLNK
TAAGAAAATAGTTTAAACAATTTGTTAAAAA*TTTTCCATCTTATTTCATTTCTCTAACAGTC
>NPM1--CLEC2D
ATACATGTGAAATAAAACTCAGTATTTTAAT*AAAAAAAAAAAAAGGTATGGGGAAAACTGCC
>NPM1--SET
TGATGACGATGATGATGAAGAGGATGATGAT*GAAGATGAAGATGATGATGAAGGGGAGGAAG
>NPM1--SET
TGATGACGATGATGATGAAGAGGATGATGAT*GAAGATGATGATGATGATGAAGAGGAGGAAG
>NPNT--PPA2
CCTGGTTATGCTGGAAAAACCTGTAATCAAG*AGAATGTAACTGGTCACTACATTTCCCCCTT
>NQO2--AL133351.4
AGATTCTCGATACTTCCTGTGGCCACTCCAG*TGGGCTGATGCGCTGGAACGTCGATGCAGCC
>NT5C2--TG
GCCTCAGCCTCCCGAAGTGCTGGGATTACAG*AAACGAGATGCACTGGGGAACTCAAAGGCCA
>NT5C2--TG
GTCTCAACCTCCCAAAGTGCTGGGATTACAG*AAACGAGATGCACTGGGGAACTCAAAGGCCA
>NTF4--CGB7
TGAATAGGAGTTCTCCTGGAGCAACTTGAGG*TCCACGGTCGCGGACCCTCTACCTCCGACTG
>NTRK2--PNLIP
AAATCAAGCCAATAGGCATTTCCAGTCCTCA*GTTTGGAATGAGCCAAGTCGTGGGCCACCTA
>NUP153--ADAM3A
CTTTGTTGCTTCTAAACCTCTGGAGGAGGAG*TGATATCCTTTGTGGAAAGATTGTTTGTCAC
>NUP58--METTL6
AGGCGTGGTGGCTCACGCCTGTAATCCCAGC*ACTTTGGGAGGCCGAGGCAGGCGGATCACCT
>NUTM2A--NUTM2B
ACCGGATGATCTTCTACGAGATGGCGGAAAA*TGTACCTTCCCAGCAAGGCCGGCCCCAAGGC
>OBSL1--EEF1A1
GGGCAACATAGCAAGAACCCATCTCCACAAA*AAAAATAAAAATTAGCCGGGCATGGTGGTGG
>OR4N2--AC134980.1
ACAGAGTGGTGTTATCACAGAATAGAACAAG*TGGATTTTTATCTGAGATCTACAACTTATGA
>OR4N2--AC134980.1
CATGTATGGAATAGAGGGGCTTGCACAGTAG*TAAGAAAAAAATGGCTGAGACCCACAGGAAC
>PADI4--S100A9
TGGGCACCTTGACTCAGCTTGACATCTGCAG*ACAGAGTGCAAGACGATGACTTGCAAAATGT
>PAFAH1B1--AC005696.3
TGTCAGTACTGGCATGTGCCTTATGACCCTC*CTTCACAGGAGGCCCTGTTTTGAGAGGTGCA
>PCBP2--IGH@
TTTATAAGCTTCTCCCTGGTTTTTTTTTTTT*TGTGGTTTTTTTTTTTTCTAAATCCCTGACA
>PCK2--DCAF11
TCTGAGTCCACTGCTGCAGCAGAACACAAAG*GATTGGAGAAGGTTTGTGTTCCCGACGCCTT
>PDCD6IP--STATH
TTCAGTTATGAATATGGATGCAGAAGTCCTA*TATACCCACAACCATACCAACCACAATACCA
>PDXDC2P-NPIPB14P--AC009065.3
CCCTACTAAAAATACAAAAAATTAGCCGGGC*ACCCTGTCCACAGCCCCCACACAGACTCGAG
>PDXDC2P-NPIPB14P--AC009065.3
TTGCGGTTCAGCCCTTTGATGACCGCAGCAG*GGTGGTCCCACTGTGGGACCACAACCAGGTA
>PECAM1--EEF1A1
GCCTGTAGTATCAGCTACTCAAGAGGCTGAG*GCGGGCGGATCACCTGAGGTCAGGAGTTTGA
>PGA4--IGH@
ACAACCGCTTCAACCCTGAGGATTCTTCCAC*CTACCCCATCTCCCTCAACTCCACCTACCCC
>PGA4--IGH@
ACAACCGCTTCAACCCTGAGGATTCTTCCAC*CTACCCCATCTCCCTCATGCTGCCACCCCCG
>PGA4--IGH@
ACCATCAATGGAGTCCAGTACCCCGTGCCAC*CCCCGACTGTCACTGCACCGACCGGCCCTCG
>PGA4--IGH@
ACCTACCAGTCCACCAGCGAGACAGTCTCCA*TCATGCTGCCACCCCCGACTGTCACTGCACC
>PGA4--IGH@
ATGCTCCCTTCGATGGCATCCTGGGGCTGGC*CTACCCCGAGTCCAAGACCCCGCTAACCGCC
>PGA4--IGH@
CCATCTTGCCTTCTCCCTCGAGTTGGGACCC*GGGAAGACCTTCACTTGCACTGCTGCCTACC
>PGA4--IGH@
CCATTGTTGACACCGGCACCTCTCTGCTGAC*CGGCCTGAGAGATGCCTCAGGTGTCACCTTC
>PGA4--IGH@
CCTGCCCGACATCGTCTTCACCATCAATGGA*GTCCAAGACCCCGCTAACCGCCACCCTCTCA
>PGA4--IGH@
CGGCACTATCGGCATCGGAACTCCTGCCCAG*TTCCCTCAACTCCACCTACCCCATCTCCCTC
>PGA4--IGH@
CGTCTTTGACACCGGCTCCTCCAACCTGTGG*GTTCCCTCAACTCCACCTACCCCATCTCCCT
>PGA4--IGH@
CTGTTACCGTCGAGGGTTACTGGCAGATCAC*CGTGGACAAGAGCAGGTGGCAGCAGGGGAAC
>PGA4--IGH@
CTTCCACCTACCAGTCCACCAGCGAGACAGT*CTCCCTCAACTCCACCTACCCCATCTCCCTC
>PGA4--IGH@
GAAGGACTTCCTGAAGAAGCACAACCTCAAC*CCAGCCAGATGGGAACGTGGTCATCGCCTGC
>PGA4--IGH@
GGGTGCCCTCAGTCTACTGCTCCAGTCTTGC*CACCTACCCCATCTCCCTCAACTCCACCTAC
>PGA4--IGH@
TCCTGGGGCTGGCCTACCCCAGCATTTCCTC*CTCCGGGGACCTGTACACCACGAGCAGCCAG
>PGA4--IGH@
TGACACCGGCACCTCTCTGCTGACCGGCCCA*CGAGGACCTGCTCTTAGGTTCAGAAGCGAAC
>PGA4--IGH@
TTATGCTCCCTTCGATGGCATCCTGGGGCTG*CTACCCCATCTCCCTCAACTCCACCTACCCC
>PGA4--IGHA1
ATCCTGGGGCTGGCCTACCCCAGCATTTCCT*CTCCGGGGACCTGTACACCACGAGCAGCCAG
>PGA4--IGHA1
CCAACCACAACCGCTTCAACCCTGAGGATTC*CTCCACCTACCCCATCTCCCTCAACTCCACC
>PGA4--IGHA1
CCAACCACAACCGCTTCAACCCTGAGGATTC*CTCCACCTACCCCATCTCCCTCATGCTGCCA
>PGA4--IGHA1
GAAGGACTTCCTGAAGAAGCACAACCTCAAC*CCAGCCAGATGGGAACGTGGTCATCGCCTGC
>PGA4--IGHA1
GAGGCTCCCACCCTGGTAGATGAACAGCCCC*AAGGTCTTCCCGCTGAGCCTCTGCAGCACCC
>PGBD2--RPL23AP25
GAAGAGTTCTACAGCTTTGGCGAGTCTATGT*ATGGCACCGAAAGCGAAGGAAGCTCCTGCTC
>PGK1--ADSL
GGTCAGGAGATTGAGACCATCCTGGCTAACA*ACGGTGAAACCCTGTCTCTACTAAAAATACA
>PIGH--EXOC8
TTATCACTGGCATTTTTTCCTTCTCTTTCTA*TCTTTTTTCTTTTTTTTTTTTGAGACAGGGT
>PIGR--AC018630.4
TCACCCTCCCAAGTAGCTGGGACTACAGGCG*TGATGGCAGGCGCCTGTAGTCCCAGCTACTC
>PIGR--AC018630.4
TGCCTCACCCTCCCAAGTAGCTGGGACTACA*GGCGTGGTGGCAGGCGCCTGTAGTCCCAGCT
>PIGR--BCL11B
AAAATTGAGGTTTGGAAGAGTGAAGCGTTTT*TTTTTTTTTTTTTTTTTTTTTGAGACGGAGT
>PIGR--BCL11B
AAGAAAATTGAGGTTTGGAAGAGTGAAGCGT*TTTTCTTTTTCTTTTTTTTTTTTTGAGACGG
>PIGR--BCL11B
GGACTACAGGCGCCTGCCAGCACGCCTGGCT*AACATGGTGAAACCCTGTCTCTACTAAAAAC
>PIGR--BCL11B
GGACTACAGGCGCCTGCCAGCACGCCTGGCT*AACATGGTGAAACCCTGTCTCTACTAAAAAT
>PIGR--BCL11B
GGACTACAGGCGCCTGCCAGCACGCCTGGCT*AACATGGTGAAACCCTGTCTCTACTAAAACT
>PIGR--BCL11B
GTAGCTGGGACTACAGGCGCCTGCCAGCACG*CCTGGCTAACATGGTGAAACCCTGTCTCTAC
>PIGR--BCL11B
TAAGAAAATTGAGGTTTGGAAGAGTGAAGCG*TTTTTTTTTTTTTTTTTTTTTTTTTTGAGAC
>PIGR--BCL11B
TAGCTGGGACTACAGATGTGGGCCACCATGT*CTGTAGTCCCAGCCACTTGAGAGGCGGTGGC
>PIGR--BCL11B
TAGCTGGGACTACAGATGTGGGCCACCATGT*CTGTAGTCCCAGCTACTTGGGAGGCTGAGGC
>PIGR--BCL11B
TAGCTGGGACTACAGATGTGGGCCACCATGT*CTGTATTCCCAGCTACTTGGGAGGCTGATGT
>PIGR--BCL11B
TCTGCCTCCCAAAGTGCTGGGATTACAGGCG*GATCACGAGGTCAGGAGATCGAGACCATCCT
>PIGR--BCL11B
TGCCTCTGCCTCCCAAAGTGCTGGGATTACA*GGCGGATCACGAGGTCAGGAGATCGAGACCA
>PIGR--SI
TAGCTGGGACTACAGATGTGGGCCACCATGT*CTGTAGCCCCAGCTACTTGAGAGGCATATTA
>PIGS--AP000356.5
GCGGCCGCCGGGGCTGCGGCTACACACCTAG*GCTCTGGCTCCGCTACTGCATCGTGTGCTTC
>PIK3R1--AC018630.4
TGAGATAGCATTAGCTGCCCAGGATGCTGCT*ATATATATATATATATATATATATATATGTG
>PIP5K1B--ZFAND5
AGCGGAGGCACCCGAATGCTGCCCGGGTGAG*TTGTCACTCAGCCCAGTCCATCAGTTTCTCA
>PKD1--AC138969.1
AGAGTGACTTGTTTCTGGATGATTCTAAGAG*GTTGATGGACGGATGGCTACATGGATGTGAT
>PKD1--AC138969.1
AGAGTGACTTGTTTCTGGATGATTCTAAGAG*TATCAATACTCTGGCTGACCATCGTCATCGT
>PKD1--AC138969.1
AGAGTGACTTGTTTCTGGATGATTCTAAGAG*TGACCTTGAGGAACCCTGGGAGCTCAGGAAG
>PKD1--AC138969.1
CTTCCTCACGTTCTCAGGCCTCCACGCTGAG*GTTATCAATACTCTGGCTGACCATCGTCATC
>PKD1--AC138969.1
CTTCCTCACGTTCTCAGGCCTCCACGCTGAG*GTTGATGGACGGATGGCTACATGGATGTGAT
>PKD1--AC138969.1
CTTCCTCACGTTCTCAGGCCTCCACGCTGAG*TGACCTTGAGGAACCCTGGGAGCTCAGGAAG
>PMS2--AC211476.4
AACAGTCTGGATGCTGGTGCCACTAATATTG*ATCTAAAGCTTAAGGACTATGGAATGGATCT
>PMS2--AC211476.4
CACCATTTCTACCTGCCACGCATCGGCGAAG*GTTGGGACTCGACTGGTGTTTGATCACGATG
>PMS2--AC211476.4
CCATAAGGAATTTCAAAGGAATATTAAGAAG*AAACGTGCCTGCTTCCCCTTCGCCTTCTGCC
>PMS2--AC211476.4
CCATAAGGAATTTCAAAGGAATATTAAGAAG*GACTCCTAGAAGTACCCCACCCCACCCCTGC
>PMS2--AC211476.4
GGGAAGCTCTGAGCTCACTTTGTGCACTGAG*TGATGTCACCATTTCTACCTGCCACGTATCG
>PMS2--AC211476.4
GGGGTAGAAGAAGAAAACTTCGAAGGCTTAA*CTCTGAAACATCACACATCTAAGATTCGAGA
>PMS2--AC211476.4
GGGGTAGAAGAAGAAAACTTCGAAGGCTTAA*TGATGTCACCATTTCTACCTGCCACGTATCG
>PMS2--AC211476.4
TAAGATTCAAGAGTTTGCCGACCTAACTCAG*TGATGTCACCATTTCTACCTGCCACGTATCG
>PMS2--AC211476.4
TCTGGGCAGGTGGTACTGAGTCTAAGCACTG*GGTGAAGAAGATAGTAGGAAACAGTCTGGAT
>PMS2--AC211476.4
TTGCATCCATGGAGCGAGCTGAGAGCTCGAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>PMS2--AC211476.5
AACAGTCTGGATGCTGGTGCCACTAATATTG*ATCTAAAGCTTAAGGACTATGGAATGGATCT
>PMS2--AC211476.5
CACCATTTCTACCTGCCACGCATCGGCGAAG*GTTGGGACTCGACTGGTGTTTGATCACGATG
>PMS2--AC211476.5
CCATAAGGAATTTCAAAGGAATATTAAGAAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>PMS2--AC211476.5
GGGAAGCTCTGAGCTCACTTTGTGCACTGAG*TGATGTCACCATTTCTACCTGCCACGTATCG
>PMS2--AC211476.5
TCTGGGCAGGTGGTACTGAGTCTAAGCACTG*GGTGAAGAAGATAGTAGGAAACAGTCTGGAT
>PMS2--AC211476.5
TTGCATCCATGGAGCGAGCTGAGAGCTCGAG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>PMS2--POM121C
CCATAAGGAATTTCAAAGGAATATTAAGAAG*GCTTTGGCTTCTAGTGGACGAGAATCACCGA
>PNLIP--SERPING1
AGAAGTGGCATATTTTGTTGAATTTCTTCAG*AATCTCGCTCTGTCGCCCAGGCTGGAATGCA
>POLA1--PGK1
GGCTAAAGCTGGTGAGAAGTATAAATATGAA*ATTGGCACTTCTCTGTTTGATGAAGAGGGAG
>POM121--AC091390.3
TTGGAGAACTACAGCCATCTAGTTTCCCTGG*GGTATGATTATCACCAAGCCAAACATCATCA
>POM121C--AC091390.3
CCAAGAGCAGCAGAAAATGAACACATTGCAG*GGGCCAGTGTCATTCAAAGATGTGGCTGTGG
>POM121C--PMS2
GGAGGTGAATTTCCATGTCAACATAGTCCTG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>PPM1G--GP2
GCGTAGTGGTGGGCGCCTGTGGTCCCAGCTG*CTCGGGAGGCCAAGGCAGGAGAATTGCTCGA
>PRH1--BCL11B
GGGATGGCCTCCCTGTTGGGGTGGTCCTTGT*GGCTTTCCTGGGGGTGGGTGAGAACTGGATT
>PRR27--NDRG2
TCACTGCAACTTCCACCTCCCGGGTTCAAGT*TGCAGTGAGCTGAGATCGCACCACTGCACTA
>PRSS3--CTRL
CTTCCTTGAGGGAGGCAAGGATTCCTGCCAG*GGTGACTCCGGAGGCCCTCTTGTCTGCCAGA
>PRSS42P--AC109583.1
CGCCCGGGGGAGTCAACTCCTGTGTCTTTGA*GTGAGCCATACCGACCCGAGGAGGATTCCAG
>PRSS44P--PRSS43P
GTGACCGGATGGGGCAAAGTAAATGAAACAG*GGAGATTCTGGGGGGCCCCTGGTCTGTGAAT
>PSLNR--LINC01297
TAAAGAGAGGTTATGGAGTCTGTAGCTTCAG*GTAACCACTGATCCACTTCCAGTCACTATCC
>PSLNR--LINC01297
TGCCAACCCTAGGATGAAGTATATTGTTACT*GCGCTTTGGAATCCTGCAGATATAATAATGA
>PSMB2--TFAP2E-AS1
AGTGGAACTCCTTAGGAAATGTCTGGAGGAG*GAACTGCTTCACAAGATTAAAGGAGAAACAG
>PSMB2--TFAP2E-AS1
AGTGGAACTCCTTAGGAAATGTCTGGAGGAG*GTGCAGGCTGCCCCATGGATGCCACACGTGG
>PTBP1--MKNK2
CCCGCGGGTCTGCTCTGTGTGCCATGGACGG*ATCATTGAGAAGCAGCCAGGCCACATTCGGA
>PTPN14--TSR1
CATGGTGAAATCCCGTCTCTACTAAAAATAC*AAAAAAAAAACAAAAAAAAACTAGCCAGGCA
>RACK1--BRD2
TGTTTTTTTGTTTGTTTGTTTTTTGTTTTTT*TTTTTTTTAAAGACTCCAGGCTGGAGTGCAG
>RBAK--AC091390.3
CGTGGGGCAGAGGCTGCGGAGCCCCAGAAGG*GTCTATCAGCCACAGTCTCTGCAAGTTTCCA
>RBAK--AC091390.3
TTGGAGAACTATAGCCATCTAGTTTCTGTGG*GGTATGATTATCACCAAGCCAAACATCATCA
>RBFOX1--SDC3
TTTTGTTTGTTTTTTTTTGTTTGTTTGTTTG*TTTTTTAACACTTCCTGTGCTGTGCCCATTT
>REG3A--APOA1
TTAGTCTGCACTTAGAGACTGCGAGAAGGAG*CCCCACGGCCCTTCAGGATGAAAGCTGCGGT
>RETSAT--AC093162.1
ACCAACCAGTTCTATCTGGCTGCTCCCCGAG*GTGTCTGCTACAGGGCTGACCATGACCTGGG
>RNF170--THAP1
TAACCGGAGATTCTCAGGGCAACCCAGATCT*GTTTCCTCTTACTCGACCCAGTCTTTGTAAA
>RNF216P1--RBAK
GCTCAAAGGACACTATGCAATCACCCGAAAG*GGGCCAGTGTCATTCAAAGATGTGGCTGTGG
>RNF216P1--RBAK
TATTCTTAAATTGATTTCAAGTTTGAACAAG*GTCTACCAGCCACAGTCTCTGCACGTTTCCA
>RNF38--TG
GATGCAACACACCTGCACGCAACAGAAGAAG*GTTTGACAACAGAACTTTTCTCCCCTGTGGA
>RPE--KANSL1L-AS1
GGATATATTATTTTATTCCAAGAAAGATCAG*TTGTGCAGAAGATTCTGACCTTATTTGAGGC
>RPL3--BCL11B
AAGAACAATGCCTCCACTGACTATGACCTAT*CTGATCAAGAACAATGCCTCCACTGACTATG
>RPL3--BCL11B
ACTATGGTGAAGTGACCAATGACTTTGTCAT*GCAGAAAGGCTACCATCACCGCACTGAGATC
>RPL3--BCL11B
ATGACCCGTCCAAGCCGGTCCACCTCACAGC*CTTGTATCCCAGGAAGGCTGTGAGGTGGACC
>RPL3--BCL11B
CAGACCATGGAGGAGAAGAAAGCATTCATGG*CACCACCTCCAAGTTTGGCCATGGCCGCTTC
>RPL3--BCL11B
GACCACTGAAGAAAGACCGAATTGCAAAGGA*AGAAGAAAGCATTCATGGGACCACTGAAGAA
>RPL3--BCL11B
GTGTGGTGGGAACCAAGAAGCGGGTGCTCAC*CCTCCACTGACTATGACCTATCTGACAAGAG
>RPL3--BCL11B
TCAAGAACAATGCCTCCACTGACTATGACCT*CTGATCAAGAACAATGCCTCCACTGACTATG
>RPL3--BCL11B
TGTAGCCTTCTCTGTGGCACGCGCTGGGCAG*AAGGCTACCATCACCGCACTGAGATCAACAA
>RPS3A--EEF1A1
ACCAACAGGTCCGCCAAATCCGGAAGAAGAT*GATGGAAAGTCACCCGTAAGGATGGCAATGC
>RPS4X--HBB
AAGGGTCGCTTTGCTGTACATCGTATTACAC*CTGAGGAGAAGTCTGCCGTTACTGCCCTGTG
>RTRAF--APLF
TTAGCTGTTAGACTTGAATATGGAGATAATG*GAAGAATCCCCAGCACAAGATAGAATATAGA
>SCD5--EEF1A1
TTATCAATGAAAAAACTTTCGTTTTTTTTTT*TTTTTTGTTTTGTTTTTTTTGGGTAGAGAAG
>SDAD1-AS1--ART3
TGATGAAATGCTCAAACTGACTGTCTAATGG*AAGAGAAAAATGAAGACGGGACATTTTGAAA
>SDAD1-AS1--ART3
TGATGAAATGCTCAAACTGACTGTCTAATGG*GACTAAAAACCGAAAACTGTATTGAGAACCT
>SDC3--CCND3
CCTGTTCCAGCCTTGTGAAGGAACTGGTTTT*GTTTTTTTTTTTGTTTTTTTTTTGTTTGTTT
>SDC3--RBFOX1
CTGGTTTTGTTTTTGTTTTTTTTTTTTTTTT*TTTGTTTGTTTGTTTGTTTTTTAGATGGAGT
>SERF2--WSB1
ACGGTGAAACCCTGTCTCTACTAAAAATACA*AAAAAAAAAAAAAATTAGCCAGGCGTGGTGG
>SERF2--WSB1
ATGGTGAAACCCTGTCTCTACTAAAAATACA*AAAAAAAAAAAAAATTAGCCAGGCGTGGTGG
>SERINC3--C19MC
ATTTTTTTTTTTCAATTTTAATTTTTTTTTT*TTGAGACGGAGTCTTGCTGTGTCTCCCAGGC
>SFRP1--AC018630.4
ATTGAGCATTTGAAAGGTGTGCTAGAACAAG*GTCCACCACCTCCTCAGGGAAAGCCACAAGG
>SH3D19--EEF1A1
GTTGGTGCTGAGCATAAGAGGTCTTCCGTAT*CTGATTGTTGCTGCTGGTGTTGGTGAATTTG
>SHISA4--AL513217.1
GGCTTCCTAAAGGCTTCCTAAAGACAATGAG*ATGTGTCGCTGGTTGTGGAAGCGGAGCTCGG
>SHROOM3--EXOC6
GTCTCAGCCTCCCGAATAGCTGGGACTACAG*CTGGAAAATTAGGGGCCTGTCGAGCGATGTT
>SLC6A8--AC018630.4
GGGCCAGTTCATGAAGGCCGGCAGCATCAAT*GTCCACCACCTCCTCAGGGAAAGCCACAAGG
>SMAP2--TNK2
ACGTTTTGAGCCACAACTGTCACAACTCCAG*AGGCTGGGAGGCGGCAGAATGCAGCCAGAGG
>SMR3B--AC018630.4
ATATTTCCACCACCCCCTCCTCAACCCTAAG*GTCCACCACCTCCTCAGGGAAAGCCACAAGG
>SMR3B--AC018630.4
ATATTTCCACCACCCCCTCCTCAACCCTAAG*GTCCTCCCCCACCTCCTCCTGGAAAGCCCCA
>SMR3B--AC018630.4
CAACCTTTTGGCCCAGGATTTGTTCCACCAC*CTCCTCAGGGAAAGCCACAAGGACCACCCCA
>SMR3B--AC018630.4
CAACCTTTTGGCCCAGGATTTGTTCCACCAC*CTCCTCCTCAAGGAAGGCCACAAGGACCACC
>SMR3B--AC018630.4
CAACCTTTTGGCCCAGGATTTGTTCCACCAC*CTCCTCCTGGAAAGCCCCAGGGACCACCTCC
>SMR3B--AC018630.4
CACCTGGACCGCTGGCTCCTCCTCAACCTTT*GCCAGCAGCAACAAGGTCCACCACCTCCTCA
>SMR3B--AC018630.4
CAGCAGGTGCCACCACCACCACAAAAGACAC*AAGGACCACCCCAACAGGGAGGCCATCCCCC
>SMR3B--AC018630.4
CAGCAGGTGCCACCACCACCACAAAAGACAC*CACCTCCTCAGGGAAAGCCACAAGGACCACC
>SMR3B--AC018630.4
CAGCAGGTGCCACCACCACCACAAAAGACAC*GGACCACCCCAACAGGGAGGCCATCCCCCTC
>SMR3B--AC018630.4
CATATCCACCTGGACCGCTGGCTCCTCCTCA*AGGAAGGCCACAAGGACCACCCCAACAGGGA
>SMR3B--AC018630.4
CATCCTGATGCCCCAGGTTATCCACAGCCTC*CCCCTCCTCCTCAAGGAAGGCCACAAGGACC
>SMR3B--AC018630.4
CCACCACTCCATCCTGATGCCCCAGGTTATC*CACCACCTCCTCAGGGAAAGCCACAAGGACC
>SMR3B--AC018630.4
CCATTTTTCCCTGTAAATTCTCCAACTGATC*CTCCCCCACCTCCTCCTGGAAAGCCCCAGGG
>SMR3B--AC018630.4
CCCAGGGGACCATATCCACCTGGACCGCTGG*CTCCTCCTCAAGGAAGGCCACAAGGACCACC
>SMR3B--AC018630.4
CCCGCACCCTATGGTCCAGGGATATTTCCAC*CACCCCAACAAGGAGGCCAGCAGCAACAAGG
>SMR3B--AC018630.4
CCCGCACCCTATGGTCCAGGGATATTTCCAC*CACCCCAACAGGGAGGCCATCAGCAAGGTCC
>SMR3B--AC018630.4
CCCGCACCCTATGGTCCAGGGATATTTCCAC*CACCTCCTCAGGGAAAGCCACAAGGACCACC
>SMR3B--AC018630.4
CCCTATGGTCCAGGGAGAATCCCACCTCCTC*AAGGGGGCCGCCCACAAGGACCTCCACAGGG
>SMR3B--AC018630.4
CCCTATGGTCCAGGGATATTTCCACCACCCC*AGGGGGCCGCCCACAAGGACCTCCACAGGGG
>SMR3B--AC018630.4
CCCTATGGTCCAGGGATATTTCCACCACCCC*CTCCTCAAGGAAGGCCACAAGGACCACCCCA
>SMR3B--AC018630.4
CCTACTCCTGCACCCCAAATATGAACAACTG*CAACAAGGTCCACCACCTCCTCAGGGAAAGC
>SMR3B--AC018630.4
CCTACTCCTGCACCCCAAATATGAACAACTG*CAGCAAGGTCCTCCCCCACCTCCTCCTGGAA
>SMR3B--AC018630.4
CCTCAACCTTTTGGCCCAGGATTTGTTCCAC*CTCCTCAGGGAAAGCCACAAGGACCACCCCA
>SMR3B--AC018630.4
CCTCCACCCTATGGTCCAGGGAGAATCCCAC*CTCCTCAGGGAAAGCCACAAGGACCACCCCA
>SMR3B--AC018630.4
CCTCCACCCTATGGTCCAGGGAGAATCCCAC*CTCCTCCTCAAGGAAGGCCACAAGGACCACC
>SMR3B--AC018630.4
CCTCCACCCTATGGTCCAGGGAGAATCCCAC*CTCCTCCTGGAAAGCCCCAGGGACCACCTCC
>SMR3B--AC018630.4
CCTCCACCCTATGGTCCAGGGAGAATCCCAC*CTCCTTTTCAAAATCAGCAACGACCACCCCG
>SMR3B--AC018630.4
CCTCCCGCACCCTATGGTCCAGGGATATTTC*CACCTCCTCAGGGAAAGCCACAAGGACCACC
>SMR3B--AC018630.4
CCTGCACCCCAAATATGAACAACTGCAGCAG*GGACCACCCCAACAAGGAGGCCAGCAGCAAC
>SMR3B--AC018630.4
CCTGGACCGCTGGCTCCTCCTCAACCTTTTG*GCCAGCAGCAACAAGGTCCACCACCTCCTCA
>SMR3B--AC018630.4
CCTTCCCGACCAAGACCCTATCCACCTGGAC*CACCCCAACAAGGAGGCCAGCAGCAACAAGG
>SMR3B--AC018630.4
CCTTCCCGACCAAGACCCTATCCACCTGGAC*CTCCACAGGGGCAGTCTCCTCAGTAATCTAG
>SMR3B--AC018630.4
CCTTTTGGCCCAGGATTTGTTCCACCACCTC*CTCCCCCACCTCCTCCTGGAAAGCCCCAGGG
>SMR3B--AC018630.4
CTCCCTACTCCTGCACCCCAAATATGAACAA*CAACAAGGTCCACCACCTCCTCAGGGAAAGC
>SMR3B--AC018630.4
CTCCCTACTCCTGCACCCCAAATATGAACAA*CAGCAACAAGGTCCACCACCTCCTCAGGGAA
>SMR3B--AC018630.4
GGCCCAGGATTTGTTCCACCACCTCCTCCTC*CACCCCAACAGGGAGGCCATCCCCCTCCTCC
>SMR3B--AC018630.4
GTAAATTCTCCAACTGATCCTACCCTCCCTA*CTCCTCAAGGAAGGCCACAAGGACCACCCCA
>SMR3B--AC018630.4
TATGAACAACTGCAGCAGGTGCCACCACCAC*CAGGGAAAGCCACAAGGACCACCCCAACAAG
>SMR3B--AC018630.4
TATGGTCCAGGGATATTTCCACCACCCCCTC*TCAACCCTCTGCTGGTGATGGGAACCAGGAT
>SMR3B--KDM6B
AAATATGAACAACTGCAGCAGGTGCCACCAC*CACCACCACCACCACCACCACCACCACCACC
>SMR3B--KDM6B
ACCCCAAATATGAACAACTGCAGCAGGTGCC*ACCACCACCACCACCACCACCACCACCACCA
>SMR3B--NEK9
GGGCCTTTGGGCTCTTGCAGCGTGTTTCACA*AGTCTCGCTCTGTCACCCAGGCTGGAGTACA
>SMR3B--SEMA6A
GCACCCCAAATATGAACAACTGCAGCAGGTG*CCACCACCACCACCACCACCACCACCACCAC
>SMR3B--SIPA1L3
TACTGCTTCTACTACCCAAAATATGAATTCC*ACCACCACCCCAAAAGATATCCATCTTGTAC
>SORL1--AC018630.4
CTCTTTAAAAACGCCCTTGGGGAAAGCCACA*AGGAGGTGGTGGACCTTGTTGCTGCTGGCCT
>SPAM1--AC004690.2
AGCAGCACCTCATAAGGTCCTTCCTAGCAAG*GGTCTCACTGTGTCACCCAGGCTGGAGTCCA
>SPEM2--AC113189.12
GACAAGATGATTGATTGGGCTACTCAGAAAA*CCCAAAGCGATAGAACCGCATGGCTTCCGCA
>SPEN--S100A9
GCTTCAGCCTCCTGAGTAGCTGGGACTACAG*AAGGAGAATAAGAATGAAAAGGTCATAGAAC
>SPTBN2--WWOX
TGGGAGGCTGAGGCAGGAGCATTGCTTGAAC*CTGGGAAGCAGAGGTTGCAGTGAGCCGAGAT
>SRSF3--S100A9
GCCTCAGCCGCCTGAGTAGCTGGGATTACAG*AAGGAGAATAAGAATGAAAAGGTCATAGAAC
>SSBL4P--SSB
ACAGAAGACCAACAAGAATCCCTAAACATAT*GGAAGTACTAGAAGGAGAGGTGGAAAAAGAA
>SSUH2--AC024940.1
TCTTAGACACCTAATACCCACAGTCCTCCAG*GTACCTGCCGTGGGCAGATCTGAGCTTTCTT
>SSUH2--AC024940.1
TCTTGGACACCTAAAACCCACAGTCCTCCAG*GTACCTGCCGTGGGCAGATCTGAGCTTTCTT
>SSUH2--AC024940.1
TCTTGGACACCTAAAACCCACAGTCCTCCAG*TAAGGTACCTGCCGTGGGCAGATCTGAGCTT
>SSUH2--AC024940.1
TCTTGGACACCTAATACCCACAGTCCTCCAG*GTACCTGCCGTGGGCAGATCTGAGCTTTCTT
>SSX1--SSX8P
CCGACCTCGCAACCACTGCTTTGTCTCTGAA*GTGAGAGTGCTCCTGGTGCCATGAACGGAGA
>SSX8P--SSX1
CCGACCTCGCCACCACTGCTTTGTCTCTGAA*GTGAGACTGCTCCTGGTGCCATGAACGGAGA
>SSX8P--SSX1
TTTGATAATGACCGTAACCGCAGGATTCAGG*TTGAACATCCTCAGATGACTTTCGGCAGGCT
>STAG3--MTHFD2
GATAGAAATCCTTGTGTCCAGTGCCCGGCAA*CTCCTGCCTCAGCCTCCTGAGTAGCTGGGAC
>STARD3--TCAP
CTTTGTCTGGATTCTTAATACAGATCTCAAG*CTGCTCCCTGCATGAGGAGGACACCCAGAGA
>STARD3--TCAP
GTGCCAAGCCCCCGACGCACAAATATGTCCG*CTGCTCCCTGCATGAGGAGGACACCCAGAGA
>STATH--FTX
TGCCTTCATCTTGGCTCTCATGGTTTCCATG*ATAGCCTAGATGAAGGAAAGGTGTGCTTATC
>STAU1--TG
GACTGACTTGAGCAGAGTGCGGTGCCTCACA*TCAGCCATGAGAGGGCTCAGCAGCAGGCAAT
>STRIP2--SMKR1
CGCATGAACGATGACTGGGCTTACGGGAATG*CCAGCTAAAGGGAAAAAAGGAAAAGGCCAGG
>SVIL--SMR3B
GCATCTGAACAGGAGAGGAGGAGGTGGTATT*CTCCTCCTCCACCCTATGGTCCAGGGAGAAT
>TAGLN--MACF1
GGAATATTTTTGGGGTTGGAACTCAAAAAAA*CAAAAAAAAAAAAAATTTTTTTTTACTTGTA
>TAX1BP1-AS1--HIBADH
CCACTCCGCCGCGGGACGCCTCATCCCGCGG*TGTGTTCTAGGTCAGTGGCTTCAAAGACTCC
>TBC1D28--CDRT15L2
GGCAGCGCCGACCCCTAAGGCAGCCGGGCGG*CTCCTGGTCCGTACGCAGACATAGCAGCACT
>TCF3--BAIAP2-DT
GCCTGGGCAACATAGTGAGACCCCATGTCTA*AAAAAAAATTTTTTTTTTAATTAGCCAGGCG
>TG--HSPD1
TGACAATGTTGCCTGCATGACTTCTGACCAG*AGACGGGGTCTCATCATGTTGGCCAGGCTGG
>TG--MTR
TGCATTCCTTGTCCTGTTGGATTCTACCAAG*TATTGAAAAGTGAGGCCAGGCACAGTGGCTC
>TG--SMARCE1
CGGGCTCATCAACAGAGCAAAGGCTGTGAAG*GGGCGGGGCGAGCGCGGGACAAAGGGAAGCG
>TG--UGP2
TGACAATGTTGCCTGCATGACTTCTGACCAG*AGATGGGGTTTCGCCATGTTGCCCAGGCTGG
>TG--VAMP7
ACCTATCCCTTCGGATGGTACCAGAAGCCCA*TCTGTCACCTTCAAAACTACCAGCAGAAATC
>TGFBR1--PTGES2
TCAGGTACTTTTTTTGTGGGGGTTTTTTTTT*TTTTGTTTTTTTTTGTTGTTGTTGTTTTTGT
>TIGD1--RPS19
ACCTTGTCCTCCCAAGGTGCTGGGATTGCAG*GTGTGAGCCACTGTGCCTGGTCTGGTTTGGG
>TMEM154--TAF1L
CCATTTTTACAAAAATGCAAAAATTAGCCAG*GTGTGGTGGTGCACGCCTGTAATCCCCAGAT
>TMPRSS11B--AC093010.1
AGCCTGGGAAACATGGCTAAATCCGTGTTCA*AAAAATACAAAAAATTAGCTAGGCATGGTGG
>TMPRSS11B--AC093010.1
CAGCCTGGGAAACATGGCTAAATCCGTGTTC*AAAAATACAAAAAATTAGCTAGGCATGGTGG
>TMSB15B--H2BW3P
GAAGAAAAAAATACTCTTCCCTCAAAGGAAA*TGGCCACTGCCTCCACCATGGCTGAGGCTTT
>TMSB15B--TMSB15B
CCAGGAGTTCCAGACCAGCCTGGCCAACATG*GCTTTCTTCTAGTCAAGATGAGTGATAAACC
>TMSB15B--TMSB15B
CGCACGGACCAGGCCCATTTCCCTTTCATCT*AAGGAGTCTCGCTCTGTCGCCAGGCTGGAGT
>TMSB15B--TMSB15B
TACGGCCTTTCACGAGTCTTCAAGCCTTCAG*AAGGAGTCTCGCTCTGTCGCCAGGCTGGAGT
>TMSB15B-AS1--RAB9B
AATACAGGGGAACGCTAGTTTCCGAGCTCAT*CACCCTAAAAAGAAAGAATAAAACAACAACA
>TNRC6B--AL022238.3
AGGCGGCCAAGGCCCAAACTGCACTGCACAT*ACATTGGGTTTGCCTATCACAGATGAACAAA
>TRB@--AL391628.1
CTGATCAAGCTCTCCTCACCTGCCGTCATCA*AGCTCTCCTCACCTGCCACAGTTGTCTCTGG
>TRB@--CHMP4B
AACATGATGTCATTGTTCAGAGTCTTCCTGT*GGGCCGAGAGATGGAGGGTGGGGGCGGGGTT
>TRIM2--TG
ACCTTGGCCTTCCAAAGTGCTGGGATTACAG*AAACGAGATGCACTGGGGAACTCAAAGGCCA
>TRIM22--EEF1A1
CAGTAGGGGGATTTTTTTGTTTGTTTGTTTG*TTTTTTTTTTGAGACAGAGTCTCACTCTGTT
>TRIM22--EEF1A1
CAGTAGGGGGATTTTTTTGTTTGTTTGTTTG*TTTTTTTTTTGAGACAGAGTCTCGCTTTGTC
>TRIM22--EEF1A1
GGGGGATTTTTTTGTTTGTTTGTTTGTTTTT*TTTTTGAGACAGAGTCTCATGTTGTCACCCA
>TRPM2--ZNF511
CTTCACGGGCCTGAGCCAGAAGGTGAAAAAG*AATACCCTCTACCATCTGCTTTGGTCAGGGT
>U62317.3--RMI1
GCCTCAGCTTCCCGAGTAGCTGGGATTACAG*GGCCATTTCTGAGCAGTGGAGGTTTCAAGTA
>UBE2D3--RBFOX1
TGGTCTCAGGGGTAGTGAAACACTCACTTTT*TTTTTTTGTTTGTTTTTTTTTGTTTGTTTGT
>UGDH--KIR3DX1
ATTAAATGGTAAATGAACCAAGTGTTTTTTT*TTGTTTTTTTTTTTTGAGACAGAGTCTCACT
>UGDH--KIR3DX1
TCTGCTATTAAATGGTAAATGAACCAAGTGT*TTTTTTTTTTTCTTTTTTTTTGAGACAGAGT
>UGP2--TG
GCCTGAGCCTCCCAAAGTGCTGGGATTACAG*AAACGAGATGCACTGGGGAACTCAAAGGCCA
>UPK3B--AC211476.4
ACGCGCCCCTCCCTGGCCCTGGACCCTATCG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>UPK3B--AC211476.4
CAGGAGGGTTTAGGGGAGGGCCATGGGCAGG*CTGGTCTCGTGATAGTGAGTGAGCGCTCATG
>UPK3B--AC211476.4
CATCCCACCCAGAGAGGCCGCCACACTGCCG*AAACGTGCCTGCTTCCCCTTCGCCTTCTGCC
>UPK3B--AC211476.4
CTTTCTGGTTTCTCTCCCTCTCCAAGCATCT*AAACGTGCCTGCTTCCCCTTCGCCTTCTGCC
>UPK3B--PMS2
ACGCGCCCCTCCCTGGCCCTGGACCCTATCG*TACAGAACCTGCTAAGGCCATCAAACCTATT
>USP15--DDX17
GTGTTTTGTGTTTTTTTTTTTTCCTTTTTTC*TCTTTTTTTTTTTTTAATTTTTCCCCCCAAC
>WDR1--AC005674.1
TTCCTTTTCTTTTTCAGTGCAGAAATTAAAA*GCTGAGGCTGAGTTGCATCTACATTTTGGTT
>WDR70--AC138866.1
ACAGCTGTGGAAAGAAGTCGCAAAACACTGG*GATGGTGATTGCTCACACCAAAGCCTTGGAC
>WDR70--AC138866.1
ACAGCTGTGGAAAGAAGTCGCAAAACACTGG*GGCTGGTGAATTACCAGATCTCCGTCAAGTG
>WWOX--SPTBN2
AGGGGAATTGCTTGAACCTGGGAAGCAGAGG*AAAAAATGGATTACTTGGCCAGGCATGGTGG
>XPO1--MAP3K20
CTAAAAAAAAAAAACAAAACAAAAAAAAACG*AACCCGGTAGCATTGTCCCTTCCCCACTGAC
>XPO1--TGM4
GTCTCAGTCTCCTGAGTAGCTGGGATTACAG*AGCTGCAAGTTCTCCACATTGACTTCTTGAA
>YOD1--C1ORF116
ACCATTCTGGAAGACTTGCCCATCCAATCTG*GCCCGGCTTGGGACATTGTTCACTTCCCCTC
>YWHAZ--UBC
CCCTCAAACCTTGCTTCTAGGAGATAAAAAG*ACAATGCAGATCTTCGTGAAGACTCTGACTG
>Z84488.1--BASP1
ACATGGAGAGGATGGCAGCGCAGCCTAGTTG*AACTCCAAGATGGGAGGCAAGCTCAGCAAGA
>ZEB2--AC093525.2
CGCAAACAAGCCAATCCCAGGAGGAAAAACG*CCCTGGGCGCTGCCTATGGCACAGCCAAGAG
>ZMAT4--AC244197.3
GCACCGGCGGCCCCTCTGCCTCGCGGAAAAG*GCAGGCAATCCATGGACCTTGTGGAACTTGT
>ZNF394--ATP5MF
CAGAAGGATTCCGGGAGCACAGTTCCGCCGA*TACCAGTGAAGGACAAGAAACTTCTGGAGGT
>ZNF665--RAB29
CCTGGGCAACAGGAGTGAAACTCCATCTCCA*AAAAAAAAAAAACAAAAAGAAAAAGATTTTT
>ZNF791--RBM39
CAGGAAACATTCAAGAACCTGGCATCTATAG*GAGGAAAAAAAGCAAGAGCAGAAGTCGTAGT
>ZNF806--ZNF285
GTGCAGCTGTGACCCGTCCCTGTGGACTGAA*TTCTGCTTTTCCAGAATCCTGCCCTTTCCAA"""
    # up is human


    file(os.path.join(options.output_directory,'banned_seq.fa'),'w').write(stub)
    
    data = fusions.get(options.organism.lower(),None)
    if data:
        d = data.replace("\n","\t").replace("\t>","\n>").splitlines()
        s = set()
        r = []
        i = 0
        j = 0
        for e in d:
            i = i + 1
            u = e.split("\t")
            if u and (u[1] not in s):
                j = j + 1
                s.add(u[1])
                x = u[0]+"\n"+u[1]+"\n"
                r.append(x)
        print "Removed",i-j,"out of",i
        file(os.path.join(options.output_directory,'banned_seq.fa'),'w').writelines(r)
    #
