#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of pre-candidate fusion genes. This list is hard coded
in here and it is manually curated from Illumina BodyMap2 RNA-seq database of samples
16 healthy human organs <http://www.ebi.ac.uk/arrayexpress/experiments/E-MTAB-513/>.



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of pre-candidate fusion genes from Illumina BodyMap2 RNA-seq database."""
    version = "%prog 0.12 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of allowed candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of allowed candidate fusion genes is generated. Default is '%default'.""")

    parser.add_option("--skip-filter-overlap",
                      action="store_true",
                      dest="skip_filter_overlap",
                      default = False,
                      help="""If set then it filters out the known fusion genes where the (i) genes are fully overlapping, or (ii) the genes are partially overlapping and are on the same strand. Default is '%default'.""")

    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    print "Generating the list of allowed/known fusion genes..."
    fusions = dict()

    # manual curation from papers

    fusions['rattus_norvegicus'] = []

    fusions['mus_musculus'] = []

    fusions['canis_lupus_familiaris'] = []

    fusions['homo_sapiens'] = [
        ['CSNK1A1','TG'],
        ['SPTB','TG'],
        ['CRYAB','NEAT1'],
        ['IGFBP5','TXNIP'],
        ['MGP','TG'],
        ['MSH5','MSH5-SAPCD1'],
        ['IGFBP4','PLBD2'],
        ['DUT','ZNF534'],
        ['ATP1B1','MALAT1'],
        ['INTS12','RP11-45L9.1'],
        ['SPTBN1','TG'],
        ['CDC42EP4','CTSB'],
        ['IGHA1','IGHV3-48'],
        ['AEBP1','MYH11'],
        ['IGLC1','IGLV7-43'],
        ['TRBV28','TRBV4-1'],
        ['RP11-813N20.3','UGT2B4'],
        ['IGLC1','IGLV2-23'],
        ['RBFOX1','RP11-420N3.2'],
        ['CDK11B','RP1-283E3.8'],
        ['ALB','C3'],
        ['LINC00969','MIR570'],
        ['CCL21','IGKC'],
        ['RP11-159H22.1','RPL37A'],
        ['FGF14','FGF14-IT1'],
        ['AKR1C1','AKR1C2'],
        ['RHOB','TG'],
        ['IGLL5','IGLV1-47'],
        ['IFITM1','IFITM3'],
        ['GTPBP1','RP3-508I15.19'],
        ['AQP1','TG'],
        ['EEF2','GNAS'],
        ['CRYAB','GNAS'],
        ['RPS21P1','TBCE'],
        ['TRBC2','TRBV25-1'],
        ['ITIH4','TMEM110-MUSTN1'],
        ['FLNA','TG'],
        ['CRP','FGB'],
        ['FTH1','MKNK2'],
        ['CMC2','RP11-303E16.3'],
        ['UPP2','UPP2-IT1'],
        ['MAP4','TG'],
        ['TRAC','TRAV38-1'],
        ['AC002398.12','NEAT1'],
        ['SIRPA','SIRPB1'],
        ['RPL13A','RPL13AP7'],
        ['HSPA1B','TG'],
        ['TMBIM6','TPO'],
        ['CD68','CNN2'],
        ['TRBV25-1','TRBV4-2'],
        ['IGHM','IGHV3-30'],
        ['SIDT2','TAGLN'],
        ['DSE','RP1-93H18.1'],
        ['ATP1A1','GPX3'],
        ['LMO7','RP11-29G8.3'],
        ['CLU','MALAT1'],
        ['IGLC2','IGLV6-57'],
        ['CTD-2396E7.9','MALAT1'],
        ['NEK6','RP11-121A14.3'],
        ['FGA','SAA2'],
        ['JUN','TG'],
        ['TRAC','TRAV26-1'],
        ['DES','TCAP'],
        ['IGLC2','IGLV2-23'],
        ['LINC00630','LL0XNC01-237H1.2'],
        ['FPGT-TNNI3K','RP4-650F12.2'],
        ['CLSTN1','CTNNBIP1'],
        ['GOLGA8A','GOLGA8B'],
        ['CCL5','LYZ'],
        ['INPP5J','TPO'],
        ['LINC00968','PENK'],
        ['EDEM2','MMP24-AS1'],
        ['PUS1','RP11-417L19.6'],
        ['TRBC2','TRBV7-9'],
        ['ADPRHL1','DCUN1D2'],
        ['TRAC','TRAV21'],
        ['SYNPO','TG'],
        ['NKX6-2','RP11-288G11.3'],
        ['MALAT1','MAP3K12'],
        ['AEBP1','TG'],
        ['EEF2','TXNIP'],
        ['IGHG1','IGHV3-30'],
        ['PTMS','SMARCA2'],
        ['MYH11','MYH9'],
        ['ZNF580','ZNF581'],
        ['IGHM','IGHV1-3'],
        ['TG','TSC22D3'],
        ['CSDE1','TG'],
        ['ATP1A1','CRYAB'],
        ['ID4','TG'],
        ['CANX','NEAT1'],
        ['IGFBP5','RPL37A'],
        ['IGHA2','IGHV3-15'],
        ['RBP5','RP11-273B20.1'],
        ['ANG','RNASE4'],
        ['PPAP2B','TG'],
        ['IGKV1-8','IGKV3D-11'],
        ['ATF6B','TNXB'],
        ['RP11-321N4.5','SYNCRIP'],
        ['IGHM','IGHV3-20'],
        ['LINC00312','LMCD1'],
        ['COL4A2','TG'],
        ['CYP4A11','CYP4A22'],
        ['COG8','VPS4A'],
        ['IGHG1','IGHV1-46'],
        ['CNN1','MYH11'],
        ['AC007249.3','RP11-320M2.1'],
        ['ACTA1','CLU'],
        ['NDST2','RP11-574K11.31'],
        ['CTBP2P8','WLS'],
        ['ADCK4','NUMBL'],
        ['PTMS','TG'],
        ['RP11-122K13.12','TUBGCP2'],
        ['KMT2D','PRKAG1'],
        ['NUPR1','TG'],
        ['MAOA','TG'],
        ['AC010525.6','AC010525.7'],
        ['IGKV1-8','IGKV5-2'],
        ['RPL5','RPL5P8'],
        ['PCF11','RP11-727A23.4'],
        ['IGHG1','IGKC'],
        ['MYH11','PLEKHO1'],
        ['FLNA','IGKC'],
        ['IGHG1','IGHV4-34'],
        ['APBB3','SRA1'],
        ['IGKV2D-29','IGKV3-11'],
        ['KIAA1257','RP11-723O4.6'],
        ['IGFBP4','NR2F2'],
        ['IGLL5','IGLV3-21'],
        ['DRICH1','KB-1572G7.3'],
        ['CA5B','CA5BP1'],
        ['HSPB8','RP11-64B16.2'],
        ['TRBV28','TRBV3-1'],
        ['IGHG1','IGHV1-3'],
        ['IGHG1','IGHV3-13'],
        ['NR2F2','RP11-522B15.3'],
        ['NKX2-1','SFTA3'],
        ['KLHL41','TG'],
        ['TG','VIM-AS1'],
        ['FLNA','NFIC'],
        ['EHBP1L1','TG'],
        ['IGFBP5','NFIC'],
        ['CRTC2','SLC39A1'],
        ['AC244250.4','IGLV2-18'],
        ['CLU','GFAP'],
        ['GLUL','GRN'],
        ['POLR3GL','RP11-315I20.1'],
        ['FLNC','TG'],
        ['SEC14L1','SNHG20'],
        ['TRBV20-1','TRBV28'],
        ['TG','TNS1'],
        ['TNNT2','TTN'],
        ['GS1-259H13.10','GS1-259H13.2'],
        ['TRAC','TRAV9-2'],
        ['NXF1','TMEM223'],
        ['DUSP1','FLNC'],
        ['IGKV3-11','IGKV3D-11'],
        ['RBM14-RBM4','RBM4'],
        ['RNF149','SNORD89'],
        ['CEBPD','TG'],
        ['IGHG1','IGHV3-11'],
        ['AP000275.65','TCP10L'],
        ['C1R','IGFBP5'],
        ['BTG2','TPO'],
        ['CHRNA5','PSMA4'],
        ['CAV1','TG'],
        ['TRAC','TRAV41'],
        ['IGLC1','IGLV2-14'],
        ['DES','EIF4G1'],
        ['CKLF-CMTM1','CMTM1'],
        ['AC002398.12','TNNT2'],
        ['FLNA','IGFBP5'],
        ['IFT140','LA16C-425C2.1'],
        ['CYP2A6','CYP2A7'],
        ['IGHV1-2','IGHV7-81'],
        ['AP001469.9','MCM3AP'],
        ['IGHG1','IGHV1-58'],
        ['AMZ2','AMZ2P1'],
        ['DSTNP2','RPL13P5'],
        ['C5ORF45','FLNC'],
        ['CTDSP2','DES'],
        ['MYH7','MYL2'],
        ['FASN','TG'],
        ['FRAS1','TG'],
        ['TG','TNNT1'],
        ['CDK11A','CDK11B'],
        ['TRBV28','TRBV4-2'],
        ['AC012309.5','LINC00665'],
        ['HSPB6','TG'],
        ['SLC25A3','SNORA53'],
        ['METTL7A','TG'],
        ['ANP32B','ANP32BP1'],
        ['MYH9','TNNT2'],
        ['CRP','FGA'],
        ['EGLN1','SCAND2P'],
        ['IGFBP4','IGFBP5'],
        ['ASPRV1','PCBP1-AS1'],
        ['ATP1A1','PSAP'],
        ['IGF1R','TG'],
        ['JAK1','RP4-535B20.1'],
        ['IGHG1','IGHV3-53'],
        ['MAVS','TG'],
        ['C10ORF25','CEP164P1'],
        ['CTSB','TIMP3'],
        ['IGHG3','IGHV1-18'],
        ['TG','VCP'],
        ['LGALS2','TMPRSS2'],
        ['IGLL5','IGLV1-51'],
        ['ATP6V0C','IGFBP4'],
        ['IGSF5','PCP4'],
        ['IGKC','TNS1'],
        ['IGKC','IGKV1-17'],
        ['LYZ','TNFRSF1B'],
        ['IGHA1','IGHJ6'],
        ['TG','TMBIM6'],
        ['TRAC','TRAV25'],
        ['CSRP3','TG'],
        ['HOXB3','HOXB4'],
        ['MYH11','MYL9'],
        ['FOXJ3','RP11-223A3.1'],
        ['GPR135','L3HYPDH'],
        ['IGHV3-11','IGHV3-7'],
        ['LAPTM4B','RP11-21I10.2'],
        ['IGKC','IGKV1D-13'],
        ['MYH11','PRM1'],
        ['IGLC2','IGLC3'],
        ['RP3-327A19.7','UPF3B'],
        ['B2M','TG'],
        ['SERPINB3','SERPINB4'],
        ['IGKV3-7','IGKV3D-11'],
        ['CARD16','CASP1'],
        ['NFATC3','PLA2G15'],
        ['TG','ZFP36L1'],
        ['CEBPA-AS1','CTD-2540B15.9'],
        ['TRBC2','TRBV28'],
        ['ALB','SERPINA3'],
        ['IDH2','RP11-617F23.1'],
        ['IGKC','IGKV1-9'],
        ['TRGC1','TRGV2'],
        ['IGHG1','IGHV3-21'],
        ['LIPG','TG'],
        ['CTC-273B12.8','CYTH2'],
        ['CTD-2501E16.2','RP11-219B17.1'],
        ['PLIN4','PSAP'],
        ['IGHA1','IGHV1-3'],
        ['ACTA1','FLNC'],
        ['HHIPL1','RP11-543C4.1'],
        ['RP11-727F15.11','TMEM223'],
        ['LMOD2','TNNT2'],
        ['MB','TG'],
        ['RPL13','RPL13P12'],
        ['RP1-102E24.10','TAPBPL'],
        ['IGKV1-27','IGKV1-6'],
        ['CPSF3L','RP5-890O3.9'],
        ['FGG','PEBP1'],
        ['FNBP4','Y_RNA'],
        ['MIR3654','TG'],
        ['PCBP1','TG'],
        ['IGKV3-11','IGKV4-1'],
        ['HSPA12A','RP11-539I5.1'],
        ['IGLC2','IGLV1-40'],
        ['C2ORF66','PGAP1'],
        ['CALM3','IGFBP5'],
        ['IGLL5','IGLV3-25'],
        ['CTSD','TG'],
        ['AC073046.25','TET3'],
        ['RP11-716O23.1','RP11-716O23.2'],
        ['CTA-38K21.4','CTA-38K21.5'],
        ['RP5-940J5.9','TG'],
        ['IGKV3-11','IGKV3-15'],
        ['ATPAF1','EFCAB14'],
        ['CLU','TPO'],
        ['TRBV20-1','TRBV25-1'],
        ['IGKC','IGKV3-15'],
        ['IGF2','IGFBP4'],
        ['GNAS','TPO'],
        ['ATP1A1','CKM'],
        ['CES1','CES1P1'],
        ['IGKV3-11','IGKV3-20'],
        ['RNASE4','RNASE4'],
        ['IGFBP5','RPL11'],
        ['RP5-1159O4.1','RP5-1159O4.2'],
        ['RP1-178F15.4','S100A13'],
        ['PPIH','RP5-994D16.3'],
        ['GLUL','SBF2'],
        ['DES','GNAS'],
        ['LILRB1','LILRB2'],
        ['TPSAB1','TPSB2'],
        ['PKNOX2','RP11-687M24.7'],
        ['CKLF','CMTM1'],
        ['CDC42EP4','TG'],
        ['JAK1','LINC01359'],
        ['IGHA1','IGHV1-69'],
        ['CLU','CTSB'],
        ['PCBP2','PCBP2-OT1'],
        ['FTX','RP3-368A4.5'],
        ['YWHAZ','YWHAZP6'],
        ['APLP2','TG'],
        ['APOA1','SERPINA1'],
        ['GPX3','SPP1'],
        ['FGF7','RP11-262H14.11'],
        ['TG','TLN1'],
        ['ENG','IGFBP5'],
        ['IGFBP4','LENG8'],
        ['ADSL','SGSM3'],
        ['ATP1A1','VIM'],
        ['CTC-441N14.4','ZNF474'],
        ['IGKV1-16','IGKV3-11'],
        ['CCDC150P1','CLIP1'],
        ['IGKC','IGKV1D-43'],
        ['TRAC','TRAV4'],
        ['ATXN7','PSMD6-AS2'],
        ['IGHG3','IGHG4'],
        ['CLU','TUBB4A'],
        ['CLIC1','DDAH2'],
        ['HERPUD1','RP11-325K4.3'],
        ['NPC2','RP3-449M8.6'],
        ['AF001548.5','NDE1'],
        ['PTMA','PTMAP5'],
        ['IGHA1','IGHV3-43'],
        ['LINC00240','TRNAI2'],
        ['SLC25A6','TG'],
        ['SNHG14','SNRPN'],
        ['CXCR1','CXCR2'],
        ['PEBP1','TG'],
        ['RPL11','TCEB3'],
        ['IGHV3-23','IGHV3-41'],
        ['APITD1-CORT','CORT'],
        ['HK1','TG'],
        ['RPL7L1','WAC-AS1'],
        ['HIAT1','SLC35A3'],
        ['RPL8','TG'],
        ['IGLL5','IGLV9-49'],
        ['PRH1','TAS2R15P'],
        ['MYH7','TNNI3'],
        ['COL1A1','RPL8'],
        ['ROPN1L','RP11-1C1.7'],
        ['CLU','FLNC'],
        ['CLK1','PPIL3'],
        ['IGLC2','IGLV1-51'],
        ['DYNLRB1','ITCH'],
        ['IGHD','IGHJ6'],
        ['AHNAK','TG'],
        ['TRAC','TRAV12-2'],
        ['IGLV2-14','IGLV2-23'],
        ['IGHA1','IGHV3-20'],
        ['CALR','NCL'],
        ['ACTG2','DES'],
        ['MYH11','PCSK7'],
        ['IYD','TTN'],
        ['IGHM','IGHV4-39'],
        ['KIAA1731','SCARNA9'],
        ['TG','TMED2'],
        ['IGFBP5','PTMS'],
        ['IGKV3-11','IGKV3D-20'],
        ['TG','VIM'],
        ['IGLL5','IGLV4-69'],
        ['CTD-3214H19.6','PCP2'],
        ['TG','TMEM66'],
        ['TMEM203','TPRN'],
        ['IGLC1','IGLV1-51'],
        ['ENKD1','GFOD2'],
        ['RP5-991G20.1','RP5-991G20.2'],
        ['RCC1','SNORA73B'],
        ['ACACB','RP11-968O1.5'],
        ['GPX3','TG'],
        ['DDOST','PINK1-AS'],
        ['IGLL5','IGLV1-50'],
        ['ZNF32-AS2','ZNF485'],
        ['TRAC','TRAV14DV4'],
        ['NEAT1','SOD2'],
        ['CD74','IGHM'],
        ['NUMA1','TG'],
        ['IGLL5','IGLV2-23'],
        ['COL1A1','IGFBP4'],
        ['CTD-2008L17.2','RP11-456O19.5'],
        ['GRID1','UC_338'],
        ['ATP1A1','EEF2'],
        ['NAP1L1','NAP1L1P3'],
        ['IGLL5','IGLV1-44'],
        ['SNORA5A','TBRG4'],
        ['ADAP2','RNF135'],
        ['DDX17','TG'],
        ['IGHA1','IGHV3-23'],
        ['IGHG3','IGHV3-23'],
        ['TRBC2','TRBV9'],
        ['CRYAB','EEF2'],
        ['GNAS','RERE'],
        ['ATP1A1','TG'],
        ['C19ORF18','ZNF606'],
        ['IGFBP5','XIRP1'],
        ['IGLC5','IGLL5'],
        ['HNRNPA2B1','MYH11'],
        ['AC006153.3','CTB-13L3.1'],
        ['TG','UBE2G2'],
        ['IGLC1','IGLV3-27'],
        ['IGHM','IGHV3-15'],
        ['FCRL1','FCRL2'],
        ['ATP1A1','GSN'],
        ['TRBV28','TRBV29-1'],
        ['SPINT2','TG'],
        ['RP11-355I22.6','SYT16'],
        ['TG','TXNIP'],
        ['ATP1A1','TPO'],
        ['CDC42EP2','POLA2'],
        ['TRBV25-1','TRBV7-9'],
        ['IGHV1-18','IGHV1-2'],
        ['SLC7A8','TG'],
        ['RP11-54C4.1','RPLP1'],
        ['RBM12','RP1-309K20.6'],
        ['IGHA1','IGKC'],
        ['FLNB','TG'],
        ['AP000320.7','SLC5A3'],
        ['CYP2B6','CYP2B7P'],
        ['IGFBP5','PRKAR1A'],
        ['IDS','TUBB4A'],
        ['ACTG1','TG'],
        ['IGHV3-53','IGHV3-66'],
        ['AC022153.1','LLNLF-65H9.1'],
        ['AKT2','TG'],
        ['HEATR1','RP11-385F5.5'],
        ['ADAMTSL4-AS1','MCL1'],
        ['NACA','NACA3P'],
        ['GBP4','GBP7'],
        ['CDH1','TG'],
        ['IGFBP4','NEAT1'],
        ['IGHM','IGHV3-74'],
        ['ORM1','ORM2'],
        ['RP11-99L13.1','RP11-99L13.2'],
        ['SNAP25','VSNL1'],
        ['TTTY15','USP9Y'],
        ['ADH1B','ADH1C'],
        ['IGHG1','IGHV6-1'],
        ['PLIN4','SPTBN1'],
        ['PPP1R8P1','SPATA6'],
        ['IGLL5','IGLV3-9'],
        ['PRKAR1A','TG'],
        ['IGKV1-17','IGKV1-8'],
        ['CCNI','TG'],
        ['DUSP1','TG'],
        ['CD74','IGKC'],
        ['RP11-307L14.1','RP11-307L14.2'],
        ['MIR6821','PIM3'],
        ['MYH7','TG'],
        ['HSPB6','IGFBP4'],
        ['MT1X','MT2A'],
        ['IGLL5','IGLV10-54'],
        ['NCDN','VSNL1'],
        ['TRAC','TRAV10'],
        ['SLC25A3','TG'],
        ['IGKV1D-43','IGKV3-11'],
        ['RP11-100F15.1','RP11-100F15.2'],
        ['LRG1','TAPBP'],
        ['TPT1','ZFP36L1'],
        ['ZNF32-AS2','ZNF32-AS3'],
        ['FOS','TG'],
        ['RP11-686D22.7','RP11-686D22.8'],
        ['LINC-PINT','MIR29B1'],
        ['NDUFA1','TG'],
        ['CALR','TG'],
        ['MSANTD3','TMEFF1'],
        ['FLNA','PLIN4'],
        ['GAA','SERPINA1'],
        ['TRBC2','TRBV2'],
        ['PSPC1','ZMYM5'],
        ['CTC-786C10.1','RP11-680G10.1'],
        ['BSG','MYH11'],
        ['NKX2-1','TG'],
        ['MACC1-AS1','RP11-486P11.1'],
        ['CMTM4','RP11-403P17.6'],
        ['IGKV2D-30','IGKV3-11'],
        ['IGHG1','IGHV3-72'],
        ['ACTB','TG'],
        ['FLNC','TMBIM6'],
        ['PEX11A','PLIN1'],
        ['GJA9-MYCBP','MYCBP'],
        ['TG','UBA52'],
        ['C4ORF36','SLC10A6'],
        ['ACTA1','CKM'],
        ['AC020956.3','LYPD4'],
        ['ATP5I','MFSD7'],
        ['EIF4G1','TG'],
        ['IGKV1D-8','IGKV3-11'],
        ['MSANTD3','MSANTD3-TMEFF1'],
        ['IGHG1','IGHV3-20'],
        ['C7ORF55-LUC7L2','LUC7L2'],
        ['AP000318.2','AP000320.7'],
        ['ALB','FGG'],
        ['RIMBP2','RP11-662M24.2'],
        ['PSAP','TG'],
        ['AC007557.1','AC007557.3'],
        ['IAH1','RP11-400L8.2'],
        ['CROCC','CROCCP2'],
        ['GSTP1','TG'],
        ['SLFN11','SLFN13'],
        ['TG','TPM2'],
        ['APBB2','Y_RNA'],
        ['LSAMP','RP11-384F7.2'],
        ['RP5-850E9.3','TG'],
        ['IGFBP4','LRP1'],
        ['ALB','FGL1'],
        ['NPM2','XPO7'],
        ['MIR143','MIR143HG'],
        ['IGKC','IGKV3D-15'],
        ['KIF1C','TG'],
        ['AC073063.10','SARNP'],
        ['LRG1','SAA2'],
        ['CLK4','RN7SKP70'],
        ['CREBL2','TPO'],
        ['IGHG3','IGHM'],
        ['NFIC','TG'],
        ['ALB','SAA1'],
        ['FTL','FTLP3'],
        ['CCL21','IGHM'],
        ['TG','UBC'],
        ['OR52N4','TRIM22'],
        ['PCSK2','TG'],
        ['TRAC','TRAV3'],
        ['IGKV1-8','IGKV4-1'],
        ['CLU','EEF2'],
        ['IGHA1','IGHV3-15'],
        ['TG','TTN'],
        ['FGA','HP'],
        ['JUP','TG'],
        ['ATN1','TG'],
        ['EEF2','IGFBP4'],
        ['NRAP','TG'],
        ['LMOD2','MYH7'],
        ['ALDH1A1','TG'],
        ['CLU','IGFBP5'],
        ['TRBV28','TRBV9'],
        ['EEF2','MYBPC1'],
        ['IGHA1','IGHV3-7'],
        ['MYH11','PRM2'],
        ['AC007036.5','MTURN'],
        ['ACADL','KANSL1L'],
        ['AC084082.3','TRIM55'],
        ['ANKRD18A','RP11-392E22.5'],
        ['IGHG1','IGHG4'],
        ['IGHA1','IGHV3-21'],
        ['C11ORF48','UBXN1'],
        ['CCPG1','PIGBOS1'],
        ['RAD23A','TG'],
        ['BACH1','BACH1-IT1'],
        ['IGHM','IGHV3-48'],
        ['IGHA1','IGHV3-73'],
        ['RP11-6O2.3','SYNM'],
        ['NSUN6','RP11-499P20.2'],
        ['MYH7','TNNT2'],
        ['LBH','TG'],
        ['MIA2','RP11-407N17.3'],
        ['IGHV4-28','IGHV4-31'],
        ['C8ORF76','ZHX1'],
        ['FGG','GC'],
        ['GPX3','NEAT1'],
        ['ZNF28','ZNF816'],
        ['ADH1A','ADH1B'],
        ['ENO3','TG'],
        ['IGHG1','IGHM'],
        ['RAP1GAP','TG'],
        ['CTD-3222D19.2','EPS15L1'],
        ['RP11-285G1.15','RSU1P2'],
        ['MSN','TG'],
        ['HSPB1','TG'],
        ['RP4-765C7.2','RPS14'],
        ['IGHA1','IGHM'],
        ['IGHA1','IGHV4-39'],
        ['IGKC','IGKV2-24'],
        ['HNRNPA1','HNRNPA1P48'],
        ['TRBV28','TRBV6-6'],
        ['LINC00963','RP11-492E3.51'],
        ['COL1A1','COL1A2'],
        ['CRYAB','MTSS1L'],
        ['RP11-57H12.2','TMEM56-RWDD3'],
        ['APP','DES'],
        ['MALAT1','MYL2'],
        ['MEG8','SNHG23'],
        ['OR2W3','TRIM58'],
        ['ACTG1','IGFBP5'],
        ['RP11-667K14.3','RP11-667K14.4'],
        ['CTC-246B18.8','GMFG'],
        ['YBX3','YBX3P1'],
        ['GAA','HP'],
        ['SOD2','TPO'],
        ['IGHG1','IGHV4-31'],
        ['KLHL23','SSB'],
        ['AIG1','RP1-95L4.4'],
        ['OTUB1','TG'],
        ['TPSB2','TPSD1'],
        ['NEB','TG'],
        ['SNHG14','SNURF'],
        ['AC018730.4','LINC01159'],
        ['SRXN1','TG'],
        ['CTB-54O9.9','PRR22'],
        ['FLNA','MYL9'],
        ['FGA','FGG'],
        ['TG','VEGFA'],
        ['ZNF197','ZNF660'],
        ['RYR1','TG'],
        ['PAPOLA','RP11-872J21.5'],
        ['IGKV3-11','IGKV3D-15'],
        ['IGHG1','RP11-731F5.2'],
        ['IGHG3','IGHV1-69'],
        ['RP11-258F22.1','ZNF248'],
        ['TG','YWHAZ'],
        ['IGHV3-23','IGHV3-30'],
        ['IGLC1','IGLV3-10'],
        ['TRBC2','TRBV14'],
        ['IGKC','IGKV1D-8'],
        ['SRRM2','TG'],
        ['DES','IGFBP5'],
        ['SYNM','TG'],
        ['NDUFAF6','RP11-347C18.3'],
        ['MACROD1','RP11-21A7A.3'],
        ['CLU','IYD'],
        ['LRP1','MTCH1'],
        ['MLLT6','TG'],
        ['GFAP','MBP'],
        ['AD000092.3','RPS6'],
        ['IGKC','IGKV6-21'],
        ['COPZ2','RP5-890E16.2'],
        ['IGLC1','IGLV3-21'],
        ['LINC00643','RP11-355I22.6'],
        ['CAPN3','RP11-164J13.1'],
        ['TRAC','TRAV8-6'],
        ['RPL24','RPL24P2'],
        ['PPP2R4','RP11-247A12.2'],
        ['CTD-2369P2.2','S1PR2'],
        ['C10ORF10','TIMP3'],
        ['IGHG1','IGHV3-23'],
        ['ADIPOQ','VKORC1L1'],
        ['NKAPP1','RHOXF1'],
        ['AC005152.3','SOX9-AS1'],
        ['IGHG2','IGHG3'],
        ['DHRS1','RABGGTA'],
        ['CAMTA1','CAMTA1-IT1'],
        ['IGHA1','IGHV2-5'],
        ['BEX4','TG'],
        ['IGHG1','IGHV3-62'],
        ['TRBV25-1','TRBV4-1'],
        ['IGHV1-2','IGHV1-24'],
        ['IGKC','IGKV3D-11'],
        ['ANKRD1','TG'],
        ['ACKR1','CADM3'],
        ['IYD','TG'],
        ['AP1G2','JPH4'],
        ['IGKC','IGKV2D-29'],
        ['AP005482.1','PSMG2'],
        ['CD99','CD99P1'],
        ['PLXNB2','TG'],
        ['AFMID','LRG1'],
        ['RP11-203J24.9','ST6GALNAC6'],
        ['TRBC2','TRBV29-1'],
        ['EGR1','TG'],
        ['LL0XNC01-16G2.1','PNMA3'],
        ['INE1','UBA1'],
        ['TSLP','WDR36'],
        ['AC108479.3','KRCC1'],
        ['FLNA','FLNC'],
        ['DISP2','LINC00594'],
        ['FABP3','RP11-266K22.2'],
        ['RP11-159N11.4','TTC12'],
        ['TRBC2','TRBV5-5'],
        ['CTD-2008L17.2','RP11-456O19.4'],
        ['ATP1A1','IGFBP5'],
        ['SESN3','TG'],
        ['CDK14','CTB-13L3.1'],
        ['HS1BP3','HS1BP3-IT1'],
        ['IGHA2','IGHG3'],
        ['RP11-466L17.1','RP11-90C4.1'],
        ['ATP7A','RP5-1000K24.2'],
        ['CTB-50L17.14','PLIN5'],
        ['LIME1','SLC2A4RG'],
        ['IGHG1','IGHV1-69'],
        ['RP11-638I2.6','YY1'],
        ['CRIP3','ZNF318'],
        ['MYL2','TTN'],
        ['CTD-2161F6.1','RP11-445O3.2'],
        ['PLSCR2','RP11-758I14.3'],
        ['IGKC','IGKV5-2'],
        ['IGLC2','IGLV3-1'],
        ['TRBV25-1','TRBV29-1'],
        ['IGHA1','IGHV4-28'],
        ['IGKV3-11','IGKV6-21'],
        ['CXORF65','IL2RG'],
        ['CTR9','SNORD97'],
        ['CTC-498J12.1','RP11-141O11.2'],
        ['ALB','GAA'],
        ['FLNC','TPO'],
        ['IGHG1','IGHV3-73'],
        ['IGKC','IGKV1-8'],
        ['IRS1','TG'],
        ['RP11-166D19.1','RP11-820L6.1'],
        ['ACTG1','FLNC'],
        ['CRP','ORM1'],
        ['ATP1A1','MALAT1'],
        ['IGKV1-8','IGKV3D-20'],
        ['PBXIP1','TG'],
        ['BMS1P20','IGLC1'],
        ['TRBV25-1','TRBV9'],
        ['GSN','TG'],
        ['RP11-599B13.6','TMEM107'],
        ['CALR','FGA'],
        ['MTHFD1','MTHFD1P1'],
        ['LDLRAP1','RP11-70P17.1'],
        ['CRYAB','DES'],
        ['ALB','SERPINA1'],
        ['IGHG1','IGHV1-18'],
        ['CFDP1','RP11-252K23.2'],
        ['FLVCR1-AS1','SPATA45'],
        ['C1ORF86','RP11-181G12.2'],
        ['RP11-385F5.4','RP11-385F5.5'],
        ['CTC-378H22.1','CTC-378H22.2'],
        ['UGT2B15','UGT2B17'],
        ['LDB3','OPN4'],
        ['IGLL5','IGLV7-43'],
        ['IGKC','IGKV1-16'],
        ['GNAS','JUN'],
        ['RP11-649A18.7','SUMO2'],
        ['RPS6','TG'],
        ['HSPB6','TNS1'],
        ['ANO1','RP11-626H12.1'],
        ['SNHG14','SNORD116-21'],
        ['C19ORF24','CIRBP'],
        ['C7','IGFBP5'],
        ['H6PD','LRG1'],
        ['HP','HPR'],
        ['CACNA1C','CACNA1C-IT2'],
        ['IGHA2','IGHV3-74'],
        ['BCAM','TG'],
        ['IGHM','IGHV1-2'],
        ['IGHG1','IGHV3-15'],
        ['ANO3','CTD-2507G9.1'],
        ['TCEAL3','TCEAL6'],
        ['IGHD','IGHG3'],
        ['COX7A2L','KCNG3'],
        ['IGKC','IGKV2D-30'],
        ['GNB2L1','IGFBP5'],
        ['IGKV3-11','IGKV5-2'],
        ['C3','IGFBP5'],
        ['SLC35E1','TG'],
        ['GLUL','GPX3'],
        ['MSRB1','RPL3L'],
        ['H3F3B','H3F3C'],
        ['EEF2','TPO'],
        ['CH17-472G23.4','PDE4DIP'],
        ['PAOX','RP11-108K14.8'],
        ['FEZ1','STT3A-AS1'],
        ['CDH1','TPO'],
        ['AMZ2','RP11-147L13.15'],
        ['AL590762.6','ZCRB1'],
        ['A2M','FGG'],
        ['TRBC2','TRBV4-1'],
        ['MSANTD3-TMEFF1','MURC'],
        ['EEF1DP3','FRY'],
        ['RP4-576H24.4','SIRPD'],
        ['PHB2','SCARNA12'],
        ['C6ORF201','RP3-400B16.4'],
        ['FLNC','IGFBP5'],
        ['IGHG3','IGHV1-46'],
        ['IGLC1','IGLV3-25'],
        ['FLNC','GNAS'],
        ['CKM','TG'],
        ['CECR7','IL17RA'],
        ['AC053503.6','MYH7'],
        ['CYP2A6','LRP1'],
        ['GREB1','RN7SL674P'],
        ['CABIN1','KB-318B8.7'],
        ['ALB','HPX'],
        ['PODN','TG'],
        ['ENTPD4','LOXL2'],
        ['GLIPR1','GLIPR1L2'],
        ['PCED1B','RP11-493L12.4'],
        ['KIAA0754','MACF1'],
        ['C3','FGA'],
        ['AC244250.2','IGLV2-14'],
        ['LYZ','SNORA67'],
        ['RPS12','SNORA33'],
        ['ATP1B1','RP5-1018K9.1'],
        ['EEF2','GSN'],
        ['IGHA1','IGHV3-72'],
        ['RP4-734P14.4','SNRPB'],
        ['AIF1L','TG'],
        ['CTSD','IGFBP5'],
        ['MUSTN1','RP5-966M1.6'],
        ['CTNNA1','CTNNAP1'],
        ['IGHM','IGHV3-7'],
        ['CDC37','TG'],
        ['IGLL5','IGLV8-61'],
        ['HNRNPA2B1','TG'],
        ['IGLL5','IGLV1-40'],
        ['IGLL5','IGLV3-27'],
        ['FKBP11','RP11-302B13.5'],
        ['ZNF607','ZNF781'],
        ['PRSS45','PRSS46'],
        ['FXYD2','FXYD6'],
        ['ATP2A2','TG'],
        ['ITIH4','SERPINA1'],
        ['IQCF1','IQCF5'],
        ['IGHA1','IGHV1-2'],
        ['FBLN2','TG'],
        ['IGKV2-24','IGKV3-11'],
        ['EIF5A','FTH1'],
        ['TRAC','TRAV13-1'],
        ['IGFBP5','NEAT1'],
        ['AC002398.12','FLNC'],
        ['IGHM','IGKC'],
        ['DENR','RP11-290L1.4'],
        ['SNORA31','TPT1'],
        ['AC018766.4','AC018766.5'],
        ['EEF2','FLNC'],
        ['IGHA1','IGHV3-53'],
        ['H6PD','SPSB1'],
        ['SNRNP70','TG'],
        ['CTC-479C5.10','EDC4'],
        ['EYA1','RP11-1102P16.1'],
        ['TRAC','TRAV23DV6'],
        ['FLNA','MYH11'],
        ['GPRIN1','SNCB'],
        ['TRAC','TRAV17'],
        ['NKX2-1','RP11-896J10.3'],
        ['TRBC2','TRBV5-1'],
        ['BMS1P20','IGLL5'],
        ['LDHA','LDHAP7'],
        ['CTSB','FLNC'],
        ['IGHG3','IGHV5-51'],
        ['ALDOA','CRYAB'],
        ['TRAC','TRAV20'],
        ['HSPA1L','LSM2'],
        ['IGKV1-8','IGKV2D-29'],
        ['SLC20A2','TG'],
        ['CALR','IGFBP5'],
        ['FGG','SERPINA3'],
        ['SAA2','SERPINA1'],
        ['IGKV1-8','IGKV3-20'],
        ['IGF2','TSIX'],
        ['FCGR3A','FCGR3B'],
        ['CTD-2619J13.8','ZNF497'],
        ['SMARCA4','TG'],
        ['CTSB','TG'],
        ['AC110769.3','LINC01191'],
        ['C1R','TG'],
        ['CD63','TG'],
        ['CTC-548K16.6','PKN1'],
        ['FBLN1','TG'],
        ['MT1X','TG'],
        ['IGHA1','IGHG1'],
        ['IGLC1','IGLV2-18'],
        ['PRC1-AS1','RCCD1'],
        ['CDC14A','RP5-837M10.2'],
        ['SAA1','SAA2-SAA4'],
        ['IGHA2','IGHV3-7'],
        ['HSPA1A','HSPA1B'],
        ['SEC63','SEC63P1'],
        ['KCTD10','RP11-256L11.3'],
        ['IGLC1','IGLV1-47'],
        ['IGKC','MYH9'],
        ['AC078883.3','AC093818.1'],
        ['RPL28','TMEM190'],
        ['AC015936.3','GFAP'],
        ['MVP','PAGR1'],
        ['CAP1','SPP1'],
        ['BTBD8','KIAA1107'],
        ['BAG6','C6ORF47'],
        ['EIF4A2','EIF4A2P3'],
        ['MIR221','MIR222HG'],
        ['COL1A1','IGFBP5'],
        ['C7ORF55','C7ORF55-LUC7L2'],
        ['EEF2','IGFBP5'],
        ['NME4','Z97634.3'],
        ['CCNH','CTC-428H11.2'],
        ['OSGIN1','RP11-505K9.4'],
        ['AHCTF1','AHCTF1P1'],
        ['LINC00594','RP11-64K12.10'],
        ['IGFBP5','VIM'],
        ['TRBV25-1','TRBV27'],
        ['IGHG1','IGHGP'],
        ['CAMK2N1','VSNL1'],
        ['SULT2A1','TAT'],
        ['ZDHHC11','ZDHHC11B'],
        ['FLNA','IGFBP4'],
        ['TG','TNNT3'],
        ['GJA9','MYCBP'],
        ['AC002398.12','IGFBP5'],
        ['DES','TG'],
        ['EARS2','GGA2'],
        ['ACR','SHANK3'],
        ['IGFBP7','TG'],
        ['EIF2B3','PTCH2'],
        ['TRBC2','TRBV4-2'],
        ['AC073063.10','RP11-762I7.5'],
        ['PSMC2','RP11-64K7.1'],
        ['LILRA2','LILRB1'],
        ['ABCD4','AC005519.4'],
        ['ATP1A1','FLNC'],
        ['IPP','RP11-767N6.7'],
        ['TRBV19','TRBV25-1'],
        ['TRBV28','TRBV5-1'],
        ['CD22','U62631.5'],
        ['IGHV1-2','IGHV1-46'],
        ['CROCCP2','RP4-565E6.1'],
        ['FAM129B','TG'],
        ['IGLC1','IGLV4-69'],
        ['SNX22','TG'],
        ['ARF3','FKBP11'],
        ['CTC-264K15.6','SCN4A'],
        ['TRGC2','TRGV2'],
        ['IGKV1-6','IGKV3-11'],
        ['FGB','FGG'],
        ['P2RY11','PPAN'],
        ['HSP90AB1','HSP90AB3P'],
        ['LINC00484','RP11-305L7.6'],
        ['H1F0','TG'],
        ['ACRBP','LPAR5'],
        ['FBXO44','FBXO6'],
        ['CRYAB','FLNC'],
        ['FLNC','MALAT1'],
        ['CNN2','SNORA67'],
        ['LEMD3','RP11-305O6.4'],
        ['COX14','RP4-605O3.4'],
        ['IFRD1','LSMEM1'],
        ['CTC-326K19.6','IL4I1'],
        ['IGHG4','IGHGP'],
        ['FKBP1A','FKBP1C'],
        ['BST1','FAM200B'],
        ['AL928768.3','IGHA1'],
        ['LILRA6','LILRB3'],
        ['IGKV1-8','IGKV1D-43'],
        ['C16ORF74','GINS2'],
        ['FAM193A','RNF4'],
        ['LPA','PLG'],
        ['ALB','ORM1'],
        ['H1FX','TG'],
        ['CKM','GNAS'],
        ['APLP1','TUBB4A'],
        ['MYL2','TG'],
        ['FKBP8','SPARC'],
        ['HP','SERPINA3'],
        ['IGHA1','IGHV1-18'],
        ['IGHA1','IGHV3-74'],
        ['CTA-250D10.23','MIR378I'],
        ['C3','FGB'],
        ['IGHM','IGHV3-73'],
        ['C6ORF163','SMIM8'],
        ['MDC1','NRM'],
        ['AC069368.3','ANKDD1A'],
        ['FGG','GAA'],
        ['IGHD','IGHM'],
        ['SNORA76C','SNORD104'],
        ['PRH1','PRH2'],
        ['CTSS','LYZ'],
        ['EEF2','TG'],
        ['AMBP','SERPINA1'],
        ['CLCC1','RP11-475E11.9'],
        ['KIAA1614-AS1','STX6'],
        ['ZNF37A','ZNF37BP'],
        ['IGHG3','RP11-731F5.2'],
        ['GNAS','TG'],
        ['ARPC4','ARPC4-TTLL3'],
        ['TRAC','TRAV12-1'],
        ['IGKV2-30','IGKV3-11'],
        ['IGLV3-21','IGLV3-9'],
        ['RP5-966M1.6','TMEM110-MUSTN1'],
        ['IGHA1','IGHV1-46'],
        ['DNAJC9','MRPS16'],
        ['ENO1','IGFBP5'],
        ['IGKC','IGKV2-30'],
        ['TRAC','TRAV2'],
        ['RP11-452F19.3','RP11-455P21.3'],
        ['RP11-383J24.6','TMEM65'],
        ['PLXNB1','TG'],
        ['IGLL5','IGLV3-19'],
        ['IGFBP5','RPL35A'],
        ['IER2','TG'],
        ['TG','USP22'],
        ['IGKV1-8','IGKV2-24'],
        ['CEP152','RP11-227D13.4'],
        ['PDAP1','TG'],
        ['ORM1','SERPINA1'],
        ['NEB','TPM2'],
        ['AL590431.1','UBAP2L'],
        ['RP11-896J10.3','SFTA3'],
        ['TBCEL','TECTA'],
        ['AES','TG'],
        ['RP11-159D12.2','RP11-159D12.5'],
        ['TPT1','TPT1P9'],
        ['IGHA1','IGHV4-34'],
        ['ZNF28','ZNF468'],
        ['AEBP1','IGFBP5'],
        ['TUBB','TUBBP1'],
        ['C3','SERPINA1'],
        ['IGHV3-23','RP11-1166P10.8'],
        ['TRAC','TRAV19'],
        ['APOBEC3A','APOBEC3B'],
        ['IGHM','IGHV3-33'],
        ['CLU','CRYAB'],
        ['C9ORF172','PHPT1'],
        ['GFAP','TUBB4A'],
        ['RPL37A','TG'],
        ['ESRP2','RP11-96D1.11'],
        ['IGLC1','IGLV1-50'],
        ['IGFBP5','TPO'],
        ['AKNA','EMP2'],
        ['RBMX','RBMXL1'],
        ['E2F5','LRRCC1'],
        ['NACA','PRIM1'],
        ['FGG','SERPINA1'],
        ['CTD-2574D22.4','KCTD13'],
        ['EPPIN-WFDC6','RP3-461P17.6'],
        ['CARM1','TG'],
        ['CHD3','TPO'],
        ['AC109642.1','LBH'],
        ['IGFBP5','LUM'],
        ['AC009228.1','FAM228A'],
        ['IGFBP5','SRRM2'],
        ['IGKC','NFIB'],
        ['IGHG3','IGHGP'],
        ['ADAMTSL4','MIR4257'],
        ['IGHM','IGHV3-21'],
        ['PRR4','TAS2R15P'],
        ['STAT1','STAT4'],
        ['MYH7','TPM1'],
        ['IGHV3-48','IGHV3-7'],
        ['ARMCX3','RP4-545K15.5'],
        ['CTD-2023N9.2','LINCR-0003'],
        ['FGA','SAA1'],
        ['MT1B','MT1E'],
        ['IGLC1','IGLV3-9'],
        ['LEAP2','UQCRQ'],
        ['SNHG19','SNORD60'],
        ['RP4-650F12.2','TNNI3K'],
        ['RIN2','RP1-122P22.2'],
        ['PTRF','TG'],
        ['MYH7','TPO'],
        ['PILRA','STAG3L5P-PVRIG2P-PILRB'],
        ['C1QBP','CTC-524C5.2'],
        ['ALB','CRP'],
        ['ATXN1L','IST1'],
        ['TRBV25-1','TRBV5-1'],
        ['LETM2','RP11-350N15.5'],
        ['IGHA1','IGHV3-13'],
        ['IRAK1','TG'],
        ['TG','WDR6'],
        ['AC004893.11','TRRAP'],
        ['IGKV1-5','IGKV3-11'],
        ['BCAT1','RP11-625L16.4'],
        ['RTN4','TG'],
        ['KCNK15','RP4-781B1.5'],
        ['IGHA1','IGHV6-1'],
        ['CDK14','CLDN12'],
        ['IGHG3','IGHV4-39'],
        ['FRG1B','RP11-87H9.2'],
        ['CTC-301O7.4','SLC6A16'],
        ['CLU','POLR2A'],
        ['CFH','CFHR3'],
        ['CUL2','RP11-297A16.2'],
        ['GSTM2','GSTM4'],
        ['IGHM','IGHV1-69'],
        ['MIRLET7I','RP11-631N16.2'],
        ['IGKV1-5','IGKV1-8'],
        ['IGHM','IGHV3-53'],
        ['NBPF14','NOTCH2NL'],
        ['PYGM','TG'],
        ['AFM','RP11-622A1.2'],
        ['IGHD','IGHG1'],
        ['MAP7D1','TG'],
        ['MBP','RP11-386G11.10'],
        ['KRBOX1','ZNF662'],
        ['FCGBP','TG'],
        ['RP11-68I18.10','SEMA6C'],
        ['IGHV3-21','IGHV3-48'],
        ['UBB','UBBP4'],
        ['CTSB','IGFBP5'],
        ['ELFN2','RP1-63G5.5'],
        ['CANX','HMGB3P22'],
        ['CHD3','TG'],
        ['APOC1','APOC1P1'],
        ['C17ORF47','SEPT4'],
        ['AC007566.10','GATAD1'],
        ['INPP5K','MYO1C'],
        ['AC245028.1','IGLL5'],
        ['IGFBP5','TG'],
        ['APOC3','SERPINA1'],
        ['AC244250.2','IGLV2-11'],
        ['SOD2','TG'],
        ['IGHV3-23','IGHV3-35'],
        ['C17ORF85','CTD-3195I5.5'],
        ['P2RX4','P2RX7'],
        ['EMB','EMBP1'],
        ['BRWD1-AS1','BRWD1-AS2'],
        ['MAMDC4','PHPT1'],
        ['APP','TG'],
        ['GPX3','IGFBP5'],
        ['AZGP1','GJC3'],
        ['IGFBP5','NR4A1'],
        ['ALB','FGA'],
        ['SAA1','SAA2'],
        ['RP11-293E1.1','RP11-293E1.2'],
        ['MBP','TUBB4A'],
        ['CLU','DES'],
        ['APOD','TG'],
        ['GGA2','ZFAND5'],
        ['AC074286.1','NFE2L2'],
        ['RP11-444D3.1','SOX5'],
        ['LILRA1','LILRB1'],
        ['CAP1','RP11-550F7.1'],
        ['AC002398.12','TG'],
        ['TG','TIMP3'],
        ['RRP7A','RRP7B'],
        ['IGHV3-11','IGHV3-48'],
        ['AC245369.2','IGHV2-70'],
        ['ANKDD1A','PLEKHO2'],
        ['RP11-124N14.3','VIM-AS1'],
        ['IGFBP4','RP11-20I23.1'],
        ['CSNK1G1','CTD-2116N17.1'],
        ['CLU','HSPB6'],
        ['IGHG1','IGHV3-74'],
        ['IGHM','IGHV3-49'],
        ['AP000783.1','GRAMD1B'],
        ['HSP90AB1','TG'],
        ['TRAC','TRAV27'],
        ['HARS2','ZMAT2'],
        ['IGKC','IGKV3-11'],
        ['CTD-2396E7.10','SYP'],
        ['IGLL5','IGLV2-14'],
        ['SOD1','TG'],
        ['HP','SERPINA1'],
        ['IGKC','IGKV1-6'],
        ['IGLC1','IGLV6-57'],
        ['EPPIN-WFDC6','WFDC6'],
        ['CRYAB','TG'],
        ['ITIH4','RP5-966M1.6'],
        ['RP1','RP11-53M11.3'],
        ['TRGC2','TRGV5'],
        ['IGFBP4','VIM'],
        ['CTD-2611O12.7','CTD-2611O12.8'],
        ['TRAC','TRAV8-3'],
        ['LTBP3','TG'],
        ['CATSPER2','CATSPER2P1'],
        ['CCT6P1','CCT6P3'],
        ['ARFGAP3','PACSIN2'],
        ['EBPL','RP11-744D14.1'],
        ['DES','FLNC'],
        ['CENPL','GAS5'],
        ['CLU','TG'],
        ['ANKRD13A','RP11-946P6.6'],
        ['C3','TG'],
        ['DES','MYH7'],
        ['LCAT','SLC12A4'],
        ['IGHA2','IGHG1'],
        ['AC093838.4','SMPD4'],
        ['RPS24P17','SLC12A3'],
        ['SFTPA1','SFTPA2'],
        ['MIR6753','TCIRG1'],
        ['CRYAB','IGFBP5'],
        ['RP11-268J15.5','SH3D21'],
        ['RP11-57H12.2','TMEM56'],
        ['AC012358.8','PRORSD1P'],
        ['ATP1B1','TG'],
        ['HSPB1','HSPB1P1'],
        ['CD22','MAG'],
        ['TG','TPT1'],
        ['ATP1A1','CLU'],
        ['CTD-2528L19.4','ZNF781'],
        ['ZNF510','ZNF782'],
        ['SOD2','YBX3'],
        ['OSBPL1A','RP11-799B12.4'],
        ['KIAA1147','RP5-894A10.2'],
        ['GSN','PCBP1'],
        ['AEBP1','VIM'],
        ['GPX3','IGFBP4'],
        ['BMS1P20','LL22NC03-80A10.6'],
        ['IGHA1','IGHA2'],
        ['IGKC','IGKV1D-16'],
        ['TG','TSPYL2'],
        ['GLUL','TG'],
        ['EED','RP11-320L11.2'],
        ['GSE1','RP11-680G10.1'],
        ['CST3','TG'],
        ['SH3BGR','WRB'],
        ['IFITM2','IFITM3'],
        ['RP11-727F15.13','STX5'],
        ['TRAC','TRAV1-2'],
        ['IGHG1','IGHV3-48'],
        ['SAP18','TG'],
        ['GUSBP11','KB-1572G7.2'],
        ['LYZ','RSL1D1'],
        ['UGT2B28','UGT2B7'],
        ['MYL6','MYL6B'],
        ['EEF2','FLNA'],
        ['IGHM','IGHV5-51'],
        ['ALB','HP'],
        ['IGKC','IGKV2D-24'],
        ['TRAC','TRAV6'],
        ['TMEM107','VAMP2'],
        ['IGHG1','IGHG3'],
        ['NEAT1','TG'],
        ['CTB-102L5.4','SPINT2'],
        ['GPX3','TXNIP'],
        ['EEF2','PLIN4'],
        ['NRG3-AS1','RP11-202D18.2'],
        ['CTSB','GNAS'],
        ['TVP23C','TVP23C-CDRT4'],
        ['ACTA1','ACTG1'],
        ['DES','TPO'],
        ['TRBC2','TRBV24-1'],
        ['IGHA1','IGHV3-33'],
        ['FGA','FGB'],
        ['IGKC','IGKV1-5'],
        ['IGHM','IGHV2-5'],
        ['IGHA2','IGHV3-21'],
        ['CTSB','TPO'],
        ['ABLIM1','TG'],
        ['IGHA1','IGHG3'],
        ['IGHA1','IGHV5-51'],
        ['IGKV1-16','IGKV1-8'],
        ['CNPY3','RP3-475N16.1'],
        ['PBXIP1','PMVK'],
        ['ATAD3A','ATAD3B'],
        ['CRP','FGG'],
        ['C3','EEF2'],
        ['DES','MYL3'],
        ['RP11-104H15.7','SPEM1'],
        ['FAM175A','HELQ'],
        ['PGAM5','PXMP2'],
        ['CHST7','ZNF674-AS1'],
        ['RP11-142L1.3','SET'],
        ['COL1A2','IGFBP5'],
        ['CHUK','ERLIN1'],
        ['PNO1','RP11-597G23.1'],
        ['GIMAP1','GIMAP2'],
        ['EEF2','LTBP4'],
        ['TG','XIRP1'],
        ['ZNF33A','ZNF33B'],
        ['RP11-264B17.3','SPNS1'],
        ['SERPINA1','SERPINA3'],
        ['CEBPZOS','RP11-423P10.2'],
        ['IGHA1','IGHV3-30'],
        ['FLNC','JUN'],
        ['IGHG3','IGHV3-30'],
        ['EEF1A2','TG'],
        ['IGHG1','IGHV3-49'],
        ['CMYA5','TG'],
        ['DES','TPM2'],
        ['FLNC','TTN'],
        ['TRAC','TRAV26-2'],
        ['AIFM3','LZTR1'],
        ['LRP11','RP11-350J20.5'],
        ['C10ORF32','LYZ'],
        ['CTDSP1','TG'],
        ['IGLC1','IGLV8-61'],
        ['EPPIN','WFDC6'],
        ['TRAC','TRAV39'],
        ['KLHL5','RP11-360F5.3'],
        ['COL6A1','IGFBP4'],
        ['DNAJC25-GNG10','GNG10'],
        ['EIF4G2','TG'],
        ['C3ORF62','USP4'],
        ['FLNC','NUPR1'],
        ['SIGLEC11','SIGLEC16'],
        ['IGLC2','IGLV3-19'],
        ['FLNC','GPX3'],
        ['EEF1G','TG'],
        ['DUS3L','PRR22'],
        ['IGLC1','IGLV3-19'],
        ['SAA1','SERPINA1'],
        ['TG','TPO'],
        ['KB-1440D3.13','PPIL2'],
        ['TRAC','TRAV13-2'],
        ['TRBC2','TRBV7-3'],
        ['GPI','TG'],
        ['FHL1','TG'],
        ['TRAC','TRAV24'],
        ['IGKV1-6','IGKV1-8'],
        ['IGHG3','IGHV3-15'],
        ['IGLL5','IGLV2-28'],
        ['CRP','SERPINA1'],
        ['IGKC','IGKV3-20'],
        ['NAV3','RP1-34H18.1'],
        ['LINC01251','UBE2R2-AS1'],
        ['GPX3','TPO'],
        ['ACTA2','ACTB'],
        ['CALR3','EPS15L1'],
        ['MYH11','TIMP3'],
        ['CTSS','RP11-1143G9.4'],
        ['LSM10','STK40'],
        ['IGKV1-8','IGKV2D-30'],
        ['A2M','ALB'],
        ['EPAS1','TG'],
        ['PILRA','PILRB'],
        ['LYZ','MRPS30'],
        ['ALDOA','TG'],
        ['GHITM','TG'],
        ['PREX1','RP5-906C1.1'],
        ['IGHG1','IGHV4-39'],
        ['FKBP5','TG'],
        ['POLR2E','TG'],
        ['DES','EEF2'],
        ['IGHG1','IGHV3OR16-8'],
        ['RP5-966M1.6','SERPINA1'],
        ['CTC-479C5.17','LCAT'],
        ['GRIP1','RP11-123O10.4'],
        ['ACAD8','GLB1L3'],
        ['TRAC','TRAV35'],
        ['PYGB','TG'],
        ['PDCD4','RP11-313D6.3'],
        ['TRBC2','TRBV3-1'],
        ['PSMD11','ZNF207'],
        ['ACTB','IGFBP5'],
        ['CD300A','CD300C'],
        ['CRY1','MTERFD3'],
        ['IL4I1','NUP62'],
        ['AB015752.3','LINC01030'],
        ['INTS3','SLC27A3'],
        ['GPX3','TIMP3'],
        ['IGKV1-27','IGKV1-8'],
        ['IGKV1-27','IGKV3-11'],
        ['RP11-314N13.3','SDCBP2-AS1'],
        ['AC053503.6','TG'],
        ['SNHG12','SNORD99'],
        ['MRPL19','RP11-342K6.2'],
        ['IGHM','MYH11'],
        ['FAM53B','RP11-12J10.3'],
        ['AOX1','AOX3P'],
        ['MALAT1','TPO'],
        ['MT2A','TG'],
        ['MALAT1','TG'],
        ['TRBC2','TRBV19'],
        ['RP11-168A11.4','THBS4'],
        ['ACTN1','HMGN1'],
        ['IGHG1','IGHV2-26'],
        ['IGFBP5','MYH9'],
        ['RP11-68I3.11','SSH2'],
        ['AC013394.2','CHD2'],
        ['IGHM','IGHV1-18'],
        ['CTC-347C20.1','CTC-347C20.2'],
        ['COL4A2','GPX3'],
        ['CSNK2B','LY6G5B'],
        ['RP11-632C17__A.1','RPL29'],
        ['LINC01089','RHOF'],
        ['NBPF14','SMG8'],
        ['IGHA1','IGHV1-24'],
        ['CLU','NRGN'],
        ['RASSF4','ZNF22'],
        ['LRRC75A','PEBP1'],
        ['FGB','SERPINA1'],
        ['ATF6B','TG'],
        ['AC006946.16','CECR7'],
        ['NBPF2P','NBPF3'],
        ['IGHG1','IGHV3-43'],
        ['AC006153.3','CLDN12'],
        ['GNS','TG'],
        ['IGHG1','IGHV5-51'],
        ['IGKC','IGKV4-1'],
        ['CKM','TPO'],
        ['DLG5','TG'],
        ['DPY19L2','DPY19L2P1'],
        ['CTSB','SOD2'],
        ['IGLL5','IGLV7-46'],
        ['HSPB7','TG'],
        ['BRICD5','PGP'],
        ['ITIH4','MUSTN1'],
        ['BRS3','HTATSF1'],
        ['IGLC2','IGLL5'],
        ['ACTA1','DES'],
        ['HCG15','HCG16'],
        ['ELOVL1','MED8'],
        ['PDIA4','TG'],
        ['HSP90AA1','TG'],
        ['TG','YME1L1'],
        ['MURC','TMEFF1'],
        ['PLVAP','TG'],
        ['LTB4R2','NOP9'],
        ['IGHG1','IGHG2'],
        ['RP11-252I13.1','USP7'],
        ['GPX3','UMOD'],
        ['CKM','CLU'],
        ['IGKV1-8','IGKV1D-8'],
        ['RP11-242D8.1','RP11-242D8.3'],
        ['APITD1','CORT'],
        ['LRR1','RHOQ'],
        ['CTB-119C2.1','HNRNPA2B1'],
        ['RP11-503N18.1','RP11-503N18.3'],
        ['CRYAB','HSPB1'],
        ['IGHV3-30','IGHV3-64'],
        ['EGR1','TPO'],
        ['RP11-446N19.1','RP11-96H19.1'],
        ['VAMP5','VAMP8'],
        ['CLTB','TG'],
        ['RP4-583P15.15','SLC2A4RG'],
        ['RP5-1061H20.3','RP5-1061H20.4'],
        ['BIN1','TG'],
        ['ACTA1','TG'],
        ['IGLC1','IGLV1-40'],
        ['CTD-2396E7.10','MALAT1'],
        ['ST3GAL1','TG'],
        ['GIMAP2','GIMAP5'],
        ['RP4-583P15.14','SLC2A4RG'],
        ['IGHG1','IGHV3-64'],
        ['ZFP2','ZNF354B'],
        ['ACTB','IGHM'],
        ['INO80C','RP11-322E11.5'],
        ['MRPS6','SLC5A3'],
        ['MYO1C','TG'],
        ['IGKC','IGKV1-27'],
        ['IGLL5','IGLV3-10'],
        ['CANX','TG'],
        ['AHNAK','IGFBP5'],
        ['IGF2','PLLP'],
        ['TRAC','TRAV29DV5'],
        ['EEF2','IYD'],
        ['TRBV25-1','TRBV6-6'],
        ['AC108488.4','RNASEH1'],
        ['IGKV2-24','IGKV2D-24'],
        ['IGHA1','IGHV3-49'],
        ['IGHA2','IGHV3-23'],
        ['IGKV1-17','IGKV3-11'],
        ['NPM1','NPM1P6'],
        ['C8ORF44','C8ORF44-SGK3'],
        ['MOCS1','RP11-61I13.3'],
        ['PAGR1','PRRT2'],
        ['EIF4E3','FOXP1'],
        ['PNPLA2','TG'],
        ['MT1E','MT1P3'],
        ['CKM','DES'],
        ['LGMN','LGMNP1'],
        ['MALAT1','NEAT1'],
        ['FGA','SERPINA1'],
        ['IGLV3-16','IGLV3-25'],
        ['MMADHC','RP1-145M24.1'],
        ['PARM1','TG'],
        ['SCNN1A','TNFRSF1A'],
        ['IGHV1-12','IGHV1-2'],
        ['C3ORF17','RP11-572M11.3'],
        ['IGKV1-8','IGKV6-21'],
        ['EEF2','NEAT1'],
        ['C3','COL1A1'],
        ['DCAF4','RP11-109N23.6'],
        ['GAPDH','TG'],
        ['IGHG1','IGHV2-5'],
        ['IGLV3-10','IGLV3-25'],
        ['IGFBP5','TPT1'],
        ['IGLC1','IGLV10-54'],
        ['AKNA','RP11-9M16.2'],
        ['MYH6','MYH7'],
        ['TRBV27','TRBV28'],
        ['CHP1','LRG1'],
        ['AC005037.6','MTND4P23'],
        ['FTX_2','RAB11FIP1P1'],
        ['IGHA2','IGHV5-51'],
        ['TRAC','TRAV12-3'],
        ['RP11-278C7.4','YARS2'],
        ['PTGDR2','RP11-804A23.4'],
        ['ACTB','MYH11'],
        ['ADSS','RP11-518L10.5'],
        ['DES','MYL2'],
        ['CIRBP','EEF2'],
        ['IGHV3-23','IGHV3-74'],
        ['FTL','MYH11'],
        ['IGHV1-46','IGHV1-69'],
        ['SQSTM1','TG'],
        ['IGHG1','IGHV1-24'],
        ['IGKV1-8','IGKV3D-15'],
        ['FTL','TG'],
        ['ATP1A1','GNAS'],
        ['DYNLL2','TG'],
        ['IGHM','IGHV3-72'],
        ['IGLC1','IGLV2-11'],
        ['RP4-800G7.2','ZNF783'],
        ['KLHDC3','RRP36'],
        ['TPT1','TPT1P4'],
        ['AIP','MIR6752'],
        ['IGHG1','IGHV1-67'],
        ['FAM60A','FAM60BP'],
        ['PDIA3','PDIA3P1'],
        ['DES','H1F0'],
        ['SAA1','SAA4'],
        ['COL6A3','IGFBP5'],
        ['IGHG1','IGHV4-28'],
        ['TRBC2','TRBV18'],
        ['LRP10','TG'],
        ['SNHG3','SNORA73B'],
        ['RP11-514O12.4','RPS6KA2'],
        ['BAG3','TG'],
        ['RP11-404G16.2','TF'],
        ['MROH7','MROH7-TTC4'],
        ['DNAJC18','ECSCR'],
        ['IGHG1','IGHV3-33'],
        ['IGKV3-20','IGKV3D-20'],
        ['MPDZ','RP11-272P10.2'],
        ['MYH9','TG'],
        ['S100A6','TG'],
        ['CLEC2D','RP11-705C15.2'],
        ['MYH11','TPM2'],
        ['FZD4','RP11-736K20.5'],
        ['ARID3C','DCTN3'],
        ['CDRT4','TVP23C'],
        ['TRGC1','TRGC2'],
        ['MYH7','TTN'],
        ['IGLL5','IGLV6-57'],
        ['ACSM2A','ACSM2B'],
        ['IGLL5','IGLV2-11'],
        ['TRBV19','TRBV28'],
        ['HIRA','TG'],
        ['LRG1','TAT'],
        ['RP11-285F7.2','TMEM63A'],
        ['SPP1','TIMP3'],
        ['FPGT-TNNI3K','TNNI3K'],
        ['CRYAB','GPX3'],
        ['RP4-613B23.1','SEC22C'],
        ['IGHM','IGHV3-23'],
        ['CRYAB','TPO'],
        ['HSD17B4','TG'],
        ['CPNE1','RBM12'],
        ['RP11-753H16.3','RP11-753H16.5'],
        ['IGKV1-8','IGKV2-30'],
        ['BRI3','RP4-607J23.2'],
        ['EEF2','IGF2'],
        ['SAG','VPS13C'],
        ['CLUH','TG'],
        ['HNRNPH1','HNRNPH1P2'],
        ['PRKRIRP8','RGS7'],
        ['CTD-2396E7.9','SYP'],
        ['IGHV4-28','IGHV4-59'],
        ['IGLC2','IGLV3-21'],
        ['FGG','HP'],
        ['PAX8','TG'],
        ['CTD-3199J23.4','RND2'],
        ['CD68','LYZ'],
        ['IGLC1','IGLV9-49'],
        ['IGLL5','IGLV2-18'],
        ['SYNPO2','TPO'],
        ['IGKC','IGKV3D-20'],
        ['DSTN','DSTNP2'],
        ['CIRBP','TG'],
        ['TRBV2','TRBV25-1'],
        ['METTL2A','METTL2B'],
        ['BHLHB9','LINC00630'],
        ['ACTA1','TPO'],
        ['NR4A1','TG'],
        ['EPOR','RGL3'],
        ['DES','MYL9'],
        ['PER1','TG'],
        ['IGKV1-8','IGKV3-15'],
        ['TRBC2','TRBV20-1'],
        ['RP4-673D20.3','RP5-968J1.1'],
        ['LAMTOR5-AS1','PROK1'],
        ['ERV3-1','ZNF117'],
        ['ARHGAP42','TMEM133'],
        ['KIF27','RP11-213G2.3'],
        ['DLEU2L','EFCAB7'],
        ['DYNC1H1','TG'],
        ['IGFBP5','TIMP2'],
        ['TRBV2','TRBV28'],
        ['PAFAH1B1','RN7SL608P'],
        ['DES','NEAT1'],
        ['AES','GPX3'],
        ['DES','MYH11'],
        ['CTDSP2','TG'],
        ['ALB','FGB'],
        ['RP11-392E22.10','RP11-392E22.5'],
        ['MCL1','TG'],
        ['CTD-2017D11.1','ZNF254'],
        ['ACTB','ACTG1'],
        ['FAN1','RP11-540B6.6'],
        ['TLR1','TLR6'],
        ['AL157871.2','SLC25A29'],
        ['SYNE2','TG'],
        ['RILPL1','RP11-972P1.7'],
        ['AQP7','FP325317.1'],
        ['MBTD1','RP11-481C4.2'],
        ['TG','RP1-309I22.2'],
        ['TIMP3','RP1-309I22.2'],
        ['APOC1','CTB-129P6.7'],
        ['ADSL','RP5-1042K10.14'],
        ['RBFOX1','RP11-420N3.3'],
        ['SYT16','LINC00643'],
        ['SNORD64','PWAR5'],
        ['DLG4','CTD-2545G14.4'],
        ['C10ORF10','RP1-309I22.2'],
        ['SPP1','RP1-309I22.2'],
        ['PARVB','RP4-671O14.6'],
        ['RTN3','RP11-142A22.3'],
        ['GPX3','MALAT1'],
        ['BCL7C','AC106782.20'],
        ['SNHG7','SNHG7'],
        ['IGHA1','IGHV1-69-2'],
        ['CIITA','RP11-876N24.3'],
        ['TAF7','CH17-140K24.2'],
        ['IGHG1','IGHV1-69-2'],
        ['RNASE4','RNASE4'],
        ['CTA-929C8.8','CTA-929C8.9'],
        ['IGLV2-11','IGLV2-8'],
        ['IFT140','LA16C-380F5.3'],
        ['MYH11','RP1-309I22.2'],
        ['CYB5D1','RP11-1099M24.6'],
        ['RP11-344E13.3','RP11-381P6.1'],
        ['GPX3','RP1-309I22.2'],
        ['PDLIM7','RP11-1334A24.5'],
        ['DCXR','RP11-1182P23.5'],
        ['RP11-87H9.2','RP11-87H9.4'],
        ['CTSB','RP1-309I22.2'],
        ['MYO18B','CTA-796E4.5'],
        ['SCGB2B2','CTD-2553L13.9'],
        ['KREMEN1','CHR22-38_28785274-29006793.1'],
        ['DYNLL2','RP11-159D12.8'],
        ['AQP7','AQP7P2'],
        ['VPS37B','RP11-463O12.5'],
        ['RP11-503N18.1','CFAP99'],
        ['SS18','LRG1'],
        ['ZDHHC1','TPPP3'],
        ['ADH1B','C1S'],
        ['TTN','SOD2'],
        ['MYO18B','CTA-796E4.5'],
        ['CNRIP1','WDR92'],
        ['MYBPC1','SOD2'],
        ['SAA4','SERPINA1'],
        ['DCAF7','LYZ'],
        ['ACTC1','POLDIP2'],
        ['MYH11','SPTBN1'],
        ['PTRF','ADIPOQ-AS1'],
        ['RPL4','RPL4P3'],
        ['C1QTNF9','C1QTNF9B'],
        ['RP11-3L10.3','RPL18'],
        ['IGHA2','IGHV3-20'],
        ['IGHG3','IGHV3-7'],
        ['AC022182.2','NASP'],
        ['GPS2','GPS2P1'],
        ['HSD17B12','RP5-933K21.2'],
        ['MSH5','MSH5-SAPCD1'],
        ['DUT','ZNF534'],
        ['ATP1B1','MALAT1'],
        ['IGHA1','IGHV3-48'],
        ['IGLC1','IGLV7-43'],
        ['TRBV28','TRBV4-1'],
        ['RP11-813N20.3','UGT2B4'],
        ['SLC25A5','SLC25A5P2'],
        ['IGLC1','IGLV2-23'],
        ['DDX12P','RP11-22B23.1'],
        ['ZNF417','ZNF587'],
        ['CCDC74A','CCDC74B'],
        ['IGKV2D-28','IGKV3-11'],
        ['LILRA6','LILRB2'],
        ['HILPDA','RP11-155G14.5'],
        ['CDK11B','RP1-283E3.8'],
        ['ALB','C3'],
        ['IGHA2','IGHV3-49'],
        ['CCL21','IGKC'],
        ['RP11-159H22.1','RPL37A'],
        ['AKR1C1','AKR1C2'],
        ['NPIPP1','RP11-419C5.2'],
        ['AC098831.4','PYURF'],
        ['IFITM1','IFITM3'],
        ['CFHR1','CFHR2'],
        ['RN7SK','RN7SKP255'],
        ['IGLC1','IGLV4-60'],
        ['TRBC2','TRBV25-1'],
        ['SORD','SORD2P'],
        ['RPL5','RPL5P9'],
        ['RP3-423B22.5','RPS27'],
        ['CRP','FGB'],
        ['IGLV2-14','IGLV2-18'],
        ['UPP2','UPP2-IT1'],
        ['EIF3LP3','RP5-1014D13.2'],
        ['FTH1','FTH1P11'],
        ['LILRB1','LILRP2'],
        ['TRAC','TRAV38-1'],
        ['IGKV1-27','IGKV1-8'],
        ['RPS7','RPS7P10'],
        ['GUCA1B','MRPS10'],
        ['SIRPA','SIRPB1'],
        ['TPT1','TPT1P11'],
        ['IGHM','IGHV2-26'],
        ['RPL13A','RPL13AP7'],
        ['DENR','RP11-290L1.4'],
        ['RP11-314N13.3','SDCBP2-AS1'],
        ['IGHV3-72','RP11-731F5.2'],
        ['MT1X','MT1XP1'],
        ['CD68','CNN2'],
        ['GLUL','RP11-1H15.1'],
        ['NDUFB8P2','RP11-411B6.6'],
        ['TRBV25-1','TRBV4-2'],
        ['IGHM','IGHV3-30'],
        ['SIDT2','TAGLN'],
        ['RPL3','RPL3P10'],
        ['CTD-2061E19.1','RPS18'],
        ['POM121','POMZP3'],
        ['ALDOA','ALDOAP2'],
        ['LMO7','RP11-29G8.3'],
        ['TMSB4X','TMSB4XP2'],
        ['IGLC2','IGLV6-57'],
        ['TUBA3C','TUBA3E'],
        ['CTD-2396E7.9','MALAT1'],
        ['FGA','SAA2'],
        ['GSTM1','GSTM4'],
        ['DES','TCAP'],
        ['IGLC2','IGLV2-23'],
        ['HSPA8','HSPA8P1'],
        ['RP1-102E24.1','RPL31'],
        ['CLSTN1','CTNNBIP1'],
        ['GOLGA8A','GOLGA8B'],
        ['CCL5','LYZ'],
        ['PARP4','PARP4P1'],
        ['CALM2','CALM2P3'],
        ['ZBTB8OS','ZBTB8OSP2'],
        ['GPX3','MALAT1'],
        ['AC058791.1','LINC-PINT'],
        ['LINC00968','PENK'],
        ['EDEM2','MMP24-AS1'],
        ['RPSA','RPSAP53'],
        ['TRBC2','TRBV7-9'],
        ['RP11-700P18.1','XPOT'],
        ['ADPRHL1','DCUN1D2'],
        ['CALM1','CALM1P1'],
        ['IGH@','IGH@'],
        ['TRAC','TRAV21'],
        ['MALAT1','MAP3K12'],
        ['RP11-564A8.4','RPL13A'],
        ['EEF2','TXNIP'],
        ['MT1G','MT2A'],
        ['HSP90AA1','HSP90AA6P'],
        ['IGHG1','IGHV3-30'],
        ['IGHG1','IGHV2-70'],
        ['IGKC','IGKV2-26'],
        ['MYH11','MYH9'],
        ['IGHM','IGHV1-3'],
        ['RP11-234N17.1','RPL5'],
        ['DDX11','RP11-22B23.1'],
        ['SAA2-SAA4','SERPINA1'],
        ['AC018730.4','LINC01159'],
        ['IGHV2-70','IGHV3-48'],
        ['LMOD2','MYH7'],
        ['IGHA2','IGHV3-15'],
        ['DPY19L2P1','DPY19L2P2'],
        ['RBP5','RP11-273B20.1'],
        ['SBDS','SBDSP1'],
        ['AC000089.3','RPL7A'],
        ['TRBC2','TRBV19'],
        ['IGKV1-8','IGKV3D-11'],
        ['RP11-321N4.5','SYNCRIP'],
        ['IGHM','IGHV3-20'],
        ['CYP4A11','CYP4A22'],
        ['IGHG1','IGHV1-69'],
        ['ZNF221','ZNF224'],
        ['ACTB','DES'],
        ['IGHG1','IGHV1-46'],
        ['CNN1','MYH11'],
        ['RP3-417G15.1','RPS24'],
        ['RP11-112J1.1','RPL31'],
        ['ZNF181','ZNF302'],
        ['CTD-2186M15.1','RPL27'],
        ['IGLV1-41','IGLV1-51'],
        ['RPL35','RPL35P5'],
        ['IGLL5','IGLV2-34'],
        ['PI4KA','PI4KAP2'],
        ['FHL1','RP11-379K17.4'],
        ['ADCK4','NUMBL'],
        ['CD74','IGK@'],
        ['MST1L','MST1P2'],
        ['HNRNPL','HNRNPLP2'],
        ['CP','CTD-2114J12.1'],
        ['EIF3L','EIF3LP3'],
        ['MKRN1','RP11-535M15.2'],
        ['ELFN2','RP1-63G5.5'],
        ['RPL5','RPL5P8'],
        ['PCF11','RP11-727A23.4'],
        ['IGHG1','IGKC'],
        ['MYH11','PLEKHO1'],
        ['FLNA','IGKC'],
        ['IGHG1','IGHV4-34'],
        ['RPL34','RPL34P27'],
        ['IGKV2D-29','IGKV3-11'],
        ['MYH2','MYH7'],
        ['RP1-182O16.1','RPL34'],
        ['IGLC1','IGLV1-47'],
        ['IGHG1','IGHV1-3'],
        ['IGHG1','IGHV3-13'],
        ['NR2F2','RP11-522B15.3'],
        ['NKX2-1','SFTA3'],
        ['IGK@','TPT1'],
        ['GSTM2','GSTM5'],
        ['EARS2','GGA2'],
        ['ARSE','ARSEP1'],
        ['COX17','POPDC2'],
        ['RP4-613B23.1','SEC22C'],
        ['RP11-69L16.5','RPS3'],
        ['CLU','GFAP'],
        ['IGLC3','IGLL5'],
        ['SEC14L1','SNHG20'],
        ['TRBV20-1','TRBV28'],
        ['RP11-129B9.1','RPL14'],
        ['AC016712.2','RPL31'],
        ['EEF1B2','EEF1B2P1'],
        ['TNNT2','TTN'],
        ['C5AR1','CD84'],
        ['TRAC','TRAV9-2'],
        ['AC016708.2','RPL10'],
        ['PABPC1','PABPC1P3'],
        ['GFAP','RP11-386G11.10'],
        ['IGHA2','IGHV2-5'],
        ['IGLV3-21','IGLV3-9'],
        ['CD97','EMR2'],
        ['IGKV3-11','IGKV3D-11'],
        ['RBM14-RBM4','RBM4'],
        ['RNF149','SNORD89'],
        ['GAPDH','GAPDHP61'],
        ['CHRNA5','PSMA4'],
        ['DPY19L2','DPY19L2P4'],
        ['RP11-570G20.1','RPL15'],
        ['TRAC','TRAV41'],
        ['RP11-791G16.2','RPS6'],
        ['IGKC','IGK@'],
        ['MT1B','MT1E'],
        ['CKLF-CMTM1','CMTM1'],
        ['HERC2P2','HERC2P9'],
        ['AC002398.12','TNNT2'],
        ['CYP2A6','CYP2A7'],
        ['CFH','CFHR1'],
        ['IGLC2','IGLV10-54'],
        ['PARVB','RP4-671O14.6'],
        ['NAP1L1','RP11-278C7.1'],
        ['LILRB2','LILRB3'],
        ['AC008427.2','MFF'],
        ['RPL10A','RPL10AP6'],
        ['EIF4A2','EIF4A2P4'],
        ['EEF1B2','EEF1B2P3'],
        ['HP','ORM1'],
        ['MYH7','MYL2'],
        ['HNRNPU','RP11-11N7.5'],
        ['SLC25A3','SLC25A3P2'],
        ['IGKC','IGK@'],
        ['GLUL','GLULP3'],
        ['CDK11A','CDK11B'],
        ['TRBV28','TRBV4-2'],
        ['AQP7','AQP7P2'],
        ['YWHAE','YWHAEP5'],
        ['ZNF419','ZNF773'],
        ['HNRNPA3','HNRNPA3P10'],
        ['IGKC','RPL3'],
        ['IGHG1','IGHV3-48'],
        ['SLC25A3','SNORA53'],
        ['IGHM','IGHV3-48'],
        ['ANP32B','ANP32BP1'],
        ['MYH9','TNNT2'],
        ['CRP','FGA'],
        ['LINC01089','RHOF'],
        ['ASPRV1','PCBP1-AS1'],
        ['LILRB2','LILRB5'],
        ['RP11-778D9.4','RPS3'],
        ['FTH1','FTH1P12'],
        ['XPOT','XPOTP1'],
        ['RP11-247I13.3','RPS18'],
        ['IGHG1','IGHV3-53'],
        ['RP11-395L14.11','WI2-1528O10.1'],
        ['AC078899.3','ZNF737'],
        ['IGKC','IGKV2-29'],
        ['DSE','FAM26F'],
        ['HSP90B1','HSP90B3P'],
        ['CTSB','TIMP3'],
        ['IGHG3','IGHV1-18'],
        ['IGLL5','IGLV1-51'],
        ['AC073869.20','NOC2L'],
        ['PGM5-AS1','PGM5-AS1'],
        ['IFITM3','RP11-467L13.5'],
        ['RP11-6O2.3','SYNM'],
        ['IGSF5','PCP4'],
        ['PABPC1','PABPC3'],
        ['PRELID1','PRELID1P1'],
        ['IGKC','TNS1'],
        ['IGKC','IGKV1-17'],
        ['LYZ','TNFRSF1B'],
        ['EEF1G','EEF1GP5'],
        ['RPS24','RPS24P13'],
        ['FTL','FTLP2'],
        ['AP000347.4','ASLP1'],
        ['TUBA1A','TUBA1B'],
        ['TRAC','TRAV25'],
        ['HOXB3','HOXB4'],
        ['MYH11','MYL9'],
        ['TPT1','TPT1P6'],
        ['FOXJ3','RP11-223A3.1'],
        ['RP11-152F13.10','RPS17P5'],
        ['IGH@','MYL9'],
        ['IGKC','IGKV1D-13'],
        ['RP11-155G14.6','RP11-212P7.3'],
        ['IGKV2-24','IGKV3-11'],
        ['IGLC2','IGLC3'],
        ['SLC2A14','SLC2A3'],
        ['PMS2','PMS2CL'],
        ['RP11-244J10.1','RPL26'],
        ['IGKV1D-13','IGKV3-11'],
        ['RPS4X','RPS4XP8'],
        ['EIF4E3','FOXP1'],
        ['CARD16','CASP1'],
        ['NFATC3','PLA2G15'],
        ['CEBPA-AS1','CTD-2540B15.9'],
        ['TRBC2','TRBV28'],
        ['AP000662.9','PPIA'],
        ['RP11-771F20.1','RPS6'],
        ['EEF1D','EEF1DP1'],
        ['IDH2','RP11-617F23.1'],
        ['EDARADD','ENO1'],
        ['IGKV3-11','IGK@'],
        ['IGHG1','IGHV3-21'],
        ['PLIN4','PSAP'],
        ['IGHA1','IGHV1-3'],
        ['XRCC6','XRCC6P4'],
        ['IGHG2','IGHV3-23'],
        ['HHIPL1','RP11-543C4.1'],
        ['MTCH2','RP11-575C1.1'],
        ['RP11-727F15.11','TMEM223'],
        ['LMOD2','TNNT2'],
        ['ACSM5','ACSM5P1'],
        ['RPL13','RPL13P12'],
        ['RP1-102E24.10','TAPBPL'],
        ['CTD-2576F9.1','GLTSCR2'],
        ['AC074183.3','SNHG7'],
        ['CPSF3L','RP5-890O3.9'],
        ['CTA-929C8.8','CTA-929C8.9'],
        ['IGK@','IGK@'],
        ['IGLC1','IGLV2-28'],
        ['COIL','RP11-1074O12.1'],
        ['SET','SETP14'],
        ['IGKV3-11','IGKV4-1'],
        ['HSPA12A','RP11-539I5.1'],
        ['CYP2C9','RP11-400G3.5'],
        ['IGLC2','IGLV1-40'],
        ['PTPN11','RP11-561O4.1'],
        ['RCN1','RCN1P2'],
        ['RP11-283C24.1','SCD'],
        ['IGLL5','IGLV1-47'],
        ['NOP56','NOP56P3'],
        ['FAM21A','FAM21C'],
        ['RPS16','ZNF90'],
        ['IGLC2','IGLV5-45'],
        ['IGLL5','IGLV3-25'],
        ['IGLC1','IGLV2-34'],
        ['RP11-435F13.2','RPLP1'],
        ['FAM136A','FAM136BP'],
        ['RPL13A','RPL13AP3'],
        ['AC073046.25','TET3'],
        ['TRAC','TRAV26-1'],
        ['HSD17B7','HSD17B7P2'],
        ['LILRB3','LILRB5'],
        ['IGKV3-11','IGKV3-15'],
        ['RP11-715J22.1','RPS16'],
        ['IGHM','IGH@'],
        ['RP11-408P14.1','RPL6'],
        ['PYY','PYY2'],
        ['TRBV20-1','TRBV25-1'],
        ['RPL7A','RPL7AP66'],
        ['GLUD1','GLUD2'],
        ['IGLL5','IGLV3-21'],
        ['B2M','TPT1'],
        ['IGKV3-11','IGKV3-20'],
        ['HNRNPCP1','KIAA0586'],
        ['LINC00969','SDHA'],
        ['IGHG1','IGHV3-11'],
        ['MCAM','TPT1'],
        ['PPIH','RP5-994D16.3'],
        ['FRG1','FRG1B'],
        ['IGH@','IGH@'],
        ['LILRB1','LILRB2'],
        ['TMSB4X','TMSB4XP4'],
        ['AP000580.1','RPS20'],
        ['PTENP1-AS','TRBV26OR9-2'],
        ['TPSAB1','TPSB2'],
        ['IGLC2','IGLV8-61'],
        ['IFITM2','RP11-467L13.5'],
        ['ADSL','RP5-1042K10.14'],
        ['IGHA1','IGHV1-69'],
        ['RPL37A','RPL37AP1'],
        ['RP1-309I22.2','TIMP3'],
        ['PCBP2','PCBP2-OT1'],
        ['FTX','RP3-368A4.5'],
        ['YWHAZ','YWHAZP6'],
        ['AZGP1','GJC3'],
        ['APOA1','SERPINA1'],
        ['GPX3','SPP1'],
        ['EYA1','RP11-1102P16.1'],
        ['CTBS','GNG5'],
        ['AP000568.2','SREK1IP1'],
        ['CTD-2192J16.15','RPL10'],
        ['AC022210.2','RPS2'],
        ['RPL7A','RPL7AP6'],
        ['IGKV1-16','IGKV3-11'],
        ['CTSS','KIRREL3'],
        ['RP11-264F23.1','RPS15'],
        ['TRAC','TRAV4'],
        ['LILRB1','LILRB4'],
        ['RP11-392P7.1','RPS6'],
        ['IGHG3','IGHG4'],
        ['ATP1B1','RP5-1018K9.1'],
        ['CLU','TUBB4A'],
        ['GAPDH','GAPDHP65'],
        ['RPS9','RPS9P1'],
        ['HERPUD1','RP11-325K4.3'],
        ['NPC2','RP3-449M8.6'],
        ['RPL14','RPL14P3'],
        ['AC010677.5','RPL23'],
        ['PTMA','PTMAP5'],
        ['RP4-765C7.2','RPS14'],
        ['CTC-232H10.1','RPL29'],
        ['GJA1','GJA1P1'],
        ['GSDMD','RP11-661A12.4'],
        ['KPNA2','RP13-104F24.1'],
        ['MAP1LC3B','MAP1LC3B2'],
        ['GPRIN1','SNCB'],
        ['SNHG14','SNRPN'],
        ['CXCR1','CXCR2'],
        ['IGHM','IGHV1-69-2'],
        ['CDC20','CDC20P1'],
        ['RPL11','TCEB3'],
        ['DSTNP2','U47924.6'],
        ['PABPC1','PABPC1P1'],
        ['AC009487.6','AHCTF1'],
        ['RPL7L1','WAC-AS1'],
        ['AB019441.29','RPS8'],
        ['HIAT1','SLC35A3'],
        ['NSUN5P1','NSUN5P2'],
        ['SDCBP','SDCBPP2'],
        ['IGLL5','IGLV9-49'],
        ['PRH1','TAS2R15P'],
        ['MYH7','TNNI3'],
        ['CBWD1','CBWD2'],
        ['GAPDH','GAPDHP63'],
        ['IGLL5','IGLV4-60'],
        ['EIF3L','EIF3LP1'],
        ['KAT7','RP5-961K14.1'],
        ['LDHB','LDHBP2'],
        ['RPS17','RPS17P5'],
        ['BCR','BCRP3'],
        ['IGKC','IGKV2-24'],
        ['UQCRFS1','UQCRFS1P1'],
        ['RPL10','RPL10P3'],
        ['TRAC','TRAV12-2'],
        ['IGLV2-14','IGLV2-23'],
        ['PWAR5','RP11-701H24.5'],
        ['IGHA1','IGHV3-20'],
        ['SDHA','SDHAP2'],
        ['ACTG2','DES'],
        ['MYH11','PCSK7'],
        ['EEF1A1','EEF1A1P6'],
        ['AC079250.1','RPL18A'],
        ['IGHM','IGHV4-39'],
        ['INE1','UBA1'],
        ['IGH@','MALAT1'],
        ['CTD-2545G14.7','RP11-6B6.3'],
        ['IGKV3-11','IGKV3D-20'],
        ['GRID1','RNF152'],
        ['UBE2V1','UBE2V1P2'],
        ['IGHG2','IGHG4'],
        ['IGLL5','IGLV4-69'],
        ['PKM','PKMP1'],
        ['DNAJA1','DNAJA1P3'],
        ['EEF1G','EEF1GP1'],
        ['NPIPA2','PKD1P6'],
        ['CH17-189H20.1','CH17-472G23.2'],
        ['CTD-3214H19.6','PCP2'],
        ['TMEM203','TPRN'],
        ['GTF2IRD2','GTF2IRD2P1'],
        ['IGLC1','IGLV1-51'],
        ['ENKD1','GFOD2'],
        ['TUBA1B','TUBAP2'],
        ['COX7C','RP11-480I12.2'],
        ['AC002519.6','PPP2CB'],
        ['BRAF','BRAFP1'],
        ['PPP2R4','RP11-247A12.2'],
        ['AC093162.5','RETSAT'],
        ['SNORA20','TCP1'],
        ['MATR3','RP11-90H3.1'],
        ['ZNF761','ZNF765'],
        ['TRAC','TRAV14DV4'],
        ['RPL5','RPL5P1'],
        ['MTRF1L','RP11-756P10.3'],
        ['CD74','IGHM'],
        ['RP4-635E18.8','TARDBP'],
        ['IPP','RP11-767N6.7'],
        ['TAF9','TAF9'],
        ['NAP1L1','NAP1L1P3'],
        ['IGLL5','IGLV1-44'],
        ['IGHA2','IGHV1-46'],
        ['HNRNPK','HNRNPKP2'],
        ['LDB3','OPN4'],
        ['AC093590.1','POLR3F'],
        ['ATP1A2','ATP1A3'],
        ['IGHA1','IGHV3-23'],
        ['TRBC2','TRBV9'],
        ['GPRASP1','RP4-769N13.6'],
        ['HSPB8','RP11-64B16.2'],
        ['C19ORF18','ZNF606'],
        ['RP4-576H24.4','SIRPD'],
        ['HNRNPK','HNRNPKP1'],
        ['PSAT1','PSAT1P3'],
        ['IGLC5','IGLL5'],
        ['IGLC1','IGLV3-27'],
        ['RP11-350G8.3','RPSA'],
        ['TPMT','TPMTP1'],
        ['RRP7A','RRP7B'],
        ['TRBV28','TRBV29-1'],
        ['ADAMTS7','ADAMTS7P1'],
        ['IGH@','IGH@'],
        ['IGLC7','IGLL5'],
        ['RPS4X','RPS4XP2'],
        ['TRBV25-1','TRBV7-9'],
        ['MT1JP','MT1M'],
        ['TIMM23','TIMM23B'],
        ['CTD-2023N9.2','CTD-2023N9.3'],
        ['HBB','HBD'],
        ['IGHA1','IGKC'],
        ['RPL5','RPL5P24'],
        ['RP11-344H11.4','RPL12'],
        ['AP000320.7','SLC5A3'],
        ['IGLV1-50','IGLV1-51'],
        ['RP11-343B5.1','RPL19'],
        ['IDS','TUBB4A'],
        ['IGHV3-53','IGHV3-66'],
        ['EMR2','EMR3'],
        ['AC022153.1','LLNLF-65H9.1'],
        ['CTC-308K20.3','PPA1'],
        ['HEATR1','RP11-385F5.5'],
        ['FTH1','FTH1P7'],
        ['APOC1','CTB-129P6.7'],
        ['TUBA3D','TUBA3E'],
        ['RP11-353H3.1','RPL18A'],
        ['FHL1','RP11-379K17.5'],
        ['NACA','NACA3P'],
        ['IGHG3','IGH@'],
        ['GBP4','GBP7'],
        ['CCL4','CCL4L1'],
        ['IGHM','IGHV3-74'],
        ['ORM1','ORM2'],
        ['SNAP25','VSNL1'],
        ['TTTY15','USP9Y'],
        ['ADH1B','ADH1C'],
        ['IGHG1','IGHV6-1'],
        ['PLIN4','SPTBN1'],
        ['TRIM16','TRIM16L'],
        ['IGKV1-17','IGKV1-8'],
        ['RP11-142A22.3','RTN3'],
        ['RPL13A','RPL13AP2'],
        ['COX7C','COX7CP1'],
        ['CD74','IGKC'],
        ['NCF1B','NCF1C'],
        ['IGLC2','IGLV7-46'],
        ['IGH@','IGH@'],
        ['RPL3','RPL3P9'],
        ['MT1X','MT2A'],
        ['IGLL5','IGLV10-54'],
        ['IGHV3-11','IGHV3OR16-9'],
        ['TRAC','TRAV10'],
        ['RP11-1143G9.4','SLC2A11'],
        ['IGLC2','IGLV3-27'],
        ['LRG1','TAPBP'],
        ['HP','SAA1'],
        ['AC004692.5','RPL18A'],
        ['RPS4X','RPS4XP16'],
        ['COPS8','COPS8P2'],
        ['NAMPT','NAMPTP1'],
        ['RP11-686D22.7','RP11-686D22.8'],
        ['WDR45B','WDR45BP1'],
        ['IFITM3','RP11-416L21.2'],
        ['CTD-2173L22.4','ZNF658'],
        ['KRT14','KRT16P4'],
        ['EIF4A1P4','RP11-186B7.4'],
        ['CYCS','CYCSP45'],
        ['MSANTD3','TMEFF1'],
        ['NDUFS5','RP11-321A17.3'],
        ['CTD-3035D6.1','RPS18'],
        ['FLNA','PLIN4'],
        ['RP11-462L8.2','RPL7A'],
        ['PFN1','PFN1P1'],
        ['BAZ2B','CAPZA1'],
        ['AC009413.2','RPS12'],
        ['MYL12A','RP11-265N6.3'],
        ['CNN2','RP11-186B7.4'],
        ['CTC-786C10.1','RP11-680G10.1'],
        ['RP11-92K2.2','RPS27A'],
        ['AC099668.5','MST1'],
        ['RPL18A','RPL18AP16'],
        ['MACC1-AS1','RP11-486P11.1'],
        ['IGKV2D-30','IGKV3-11'],
        ['TOMM20','TOMM20P4'],
        ['HSPA8','HSPA8P9'],
        ['IGHG1','IGHV3-72'],
        ['APOL1','APOL2'],
        ['PEX11A','PLIN1'],
        ['RPL10A','RPL10AP2'],
        ['GAPDH','RP11-159J3.1'],
        ['KIAA2013','RP11-680G24.4'],
        ['HILPDA','RP11-155G14.6'],
        ['RP11-343H5.4','RPS14'],
        ['ATP5I','MFSD7'],
        ['IGKV1D-8','IGKV3-11'],
        ['IGHG1','IGHV3-20'],
        ['ALB','FGG'],
        ['RIMBP2','RP11-662M24.2'],
        ['PTMA','PTMAP2'],
        ['EEF1A1','EEF1A1P14'],
        ['RPL7A','RPL7AP11'],
        ['PRB1','PRB2'],
        ['RP11-345K20.2','RPL26'],
        ['SLFN11','SLFN13'],
        ['TRGV2','TRGV4'],
        ['IGHM','IGHV2-70'],
        ['RPS4X','RPS4XP17'],
        ['NOP56','NOP56P1'],
        ['ALB','FGL1'],
        ['KRT14','KRT19'],
        ['IGKC','IGKV3D-15'],
        ['KPNA2','RP13-104F24.2'],
        ['FAM177B','SOD2'],
        ['LRG1','SAA2'],
        ['RP11-307I14.2','RPS5'],
        ['AC016739.2','RPLP1'],
        ['SULT1A1','SULT1A2'],
        ['CLK4','RN7SKP70'],
        ['HSPA8','RP11-488L18.1'],
        ['IGHA2','IGHV4-39'],
        ['IGHG3','IGHM'],
        ['IGHM','IGHV4-28'],
        ['RP11-51O6.1','RPS27A'],
        ['ALB','SAA1'],
        ['AC078883.3','AC093818.1'],
        ['IL6ST','IL6STP1'],
        ['IGLC2','IGLV4-60'],
        ['NEFH','NEFHP1'],
        ['C11ORF58','RP11-305B6.1'],
        ['OR52N4','TRIM22'],
        ['IGHV2-70','IGHV2-70'],
        ['TRAC','TRAV3'],
        ['IGKV1-8','IGKV4-1'],
        ['CTB-33G10.1','RPS9'],
        ['RPL3','RPL3P12'],
        ['IGHA1','IGHV3-15'],
        ['ADH5','RP11-571L19.8'],
        ['RPSA','RPSAP56'],
        ['RN7SK','RN7SKP80'],
        ['PTMS','RPS2'],
        ['CSPG4','CSPG4P13'],
        ['FGA','HP'],
        ['MYH11','TIMP3'],
        ['RP11-760D2.7','YWHAE'],
        ['DDX3X','DDX3Y'],
        ['GS1-124K5.12','KCTD7'],
        ['RP11-562A8.1','RPS20'],
        ['MTIF2','RP11-344P13.3'],
        ['IGHM','IGK@'],
        ['PIP5K1A','PIPSL'],
        ['TRBV28','TRBV9'],
        ['FTH1','FTH1P5'],
        ['AC125232.1','RMND5A'],
        ['IGHA1','IGHV3-7'],
        ['IGHV2-5','IGH@'],
        ['RPS24','RPS24P8'],
        ['AC007036.5','MTURN'],
        ['ATP5E','ATP5EP2'],
        ['KANSL3','SPN'],
        ['IGHG1','IGHG4'],
        ['AP2B1','RP13-88F20.1'],
        ['BMS1P20','IGLC2'],
        ['IGHA1','IGHV3-21'],
        ['RPL30','RPL30P4'],
        ['RP11-889L3.1','RPL19'],
        ['HSPA5','RP11-334L9.1'],
        ['IGHA1','IGHV3-73'],
        ['IGHV3-15','RP11-731F5.2'],
        ['NPIPA3','NPIPP1'],
        ['NSUN6','RP11-499P20.2'],
        ['MYH7','TNNT2'],
        ['MIA2','RP11-407N17.3'],
        ['ACTB','RP11-602M11.3'],
        ['ADIPOQ','IGK@'],
        ['TLE1','TLE1P1'],
        ['RPS13','RPS13P2'],
        ['C8ORF76','ZHX1'],
        ['FGG','GC'],
        ['CATSPER2','CATSPER2P1'],
        ['EEF1A1','EEF1A1P22'],
        ['HSPA8','HSPA8P16'],
        ['ADH1A','ADH1B'],
        ['RPS4X','RPS4XP7'],
        ['NDUFV2','NDUFV2P1'],
        ['IGHG1','IGHM'],
        ['CYP2D7','CYP2D8P'],
        ['RPL39','RPL39P38'],
        ['RAP1A','RAP1AP'],
        ['AC005534.6','RPL26'],
        ['IGHA1','IGHM'],
        ['GPR116','RP11-213O5.4'],
        ['IGLC1','IGLV3-10'],
        ['IGLC2','IGLV1-51'],
        ['HNRNPA1','HNRNPA1P48'],
        ['TRBV28','TRBV6-6'],
        ['LINC00963','RP11-492E3.51'],
        ['EIF2S3','EIF2S3L'],
        ['CYP2A13','CYP2A6'],
        ['AC021016.7','RPL19'],
        ['RP11-57H12.2','TMEM56-RWDD3'],
        ['NT5C3A','NT5C3AP1'],
        ['MALAT1','MYL2'],
        ['GOT2','GOT2P2'],
        ['RPL15','RPL15P2'],
        ['RP11-667K14.3','RP11-667K14.4'],
        ['CTC-246B18.8','GMFG'],
        ['EIF3E','EIF3EP1'],
        ['IGHA1','IGHV2-70'],
        ['PCBP2','RP11-730G20.2'],
        ['SIRPA','SIRPAP1'],
        ['RP11-452G18.2','RPS2P5'],
        ['NCDN','VSNL1'],
        ['GAPDH','SCD'],
        ['EEF2','FTL'],
        ['SEPT7','SEPT7P7'],
        ['CIDEC','CIDECP'],
        ['IGHG1','IGHV4-31'],
        ['KLHL23','SSB'],
        ['AIG1','RP1-95L4.4'],
        ['RP11-175B9.3','RPS27'],
        ['RP1-278E11.3','RPL23'],
        ['TPSB2','TPSD1'],
        ['MSANTD3','MSANTD3-TMEFF1'],
        ['SNHG14','SNURF'],
        ['ZNF350','ZNF615'],
        ['RPS15A','RPS15AP11'],
        ['IGHG2','IGHV3-74'],
        ['MIR100HG','RP11-166D19.1'],
        ['METTL2A','RP11-212P7.3'],
        ['RPL4','RPL4P5'],
        ['FLNA','MYL9'],
        ['IGH@','RPL8'],
        ['FGA','FGG'],
        ['CTD-2575K13.6','RABAC1'],
        ['IGLV2-11','IGLV2-23'],
        ['RP11-173D9.5','SEPT7'],
        ['PAPOLA','RP11-872J21.5'],
        ['IGKV3-11','IGKV3D-15'],
        ['GAPDH','GAPDHP44'],
        ['IGHG1','RP11-731F5.2'],
        ['HRC','MYH7'],
        ['RP11-258F22.1','ZNF248'],
        ['DDX23','RP11-641A6.5'],
        ['IGHA1','IGHV4-39'],
        ['TRBC2','TRBV14'],
        ['ASS1','ASS1P11'],
        ['IGKC','IGKV1D-8'],
        ['IGKC','IGKV1-27'],
        ['IGH@','IGH@'],
        ['NAP1L1','NAP1L4P3'],
        ['PDE4DIP','WI2-1896O14.1'],
        ['RP11-1035H13.3','RPS15AP10'],
        ['NDUFAF6','RP11-347C18.3'],
        ['IGKV2D-24','IGKV3-11'],
        ['RPS2P46','RPS2P5'],
        ['RNF13','RP11-259P15.4'],
        ['UQCRH','UQCRHL'],
        ['TPT1','TPT1P5'],
        ['ATXN1L','IST1'],
        ['IGH@','IGKV1-8'],
        ['CH17-118O6.2','CH17-472G23.2'],
        ['GFAP','MBP'],
        ['AD000092.3','RPS6'],
        ['IGKC','IGKV6-21'],
        ['FTH1','FTH1P3'],
        ['IGHM','IGHV3-13'],
        ['IGLC1','IGLV3-21'],
        ['MT1M','MT2P1'],
        ['CAPN3','RP11-164J13.1'],
        ['TRAC','TRAV8-6'],
        ['RPL24','RPL24P2'],
        ['ACTBP12','ACTG1'],
        ['RPS4X','RPS4XP11'],
        ['C10ORF10','TIMP3'],
        ['H6PD','LRG1'],
        ['ADIPOQ','VKORC1L1'],
        ['RPL5','RPL5P4'],
        ['RP11-397E7.1','RPL6'],
        ['IGKV2-24','IGKV2D-24'],
        ['MYL12A','MYL12B'],
        ['IGHG2','IGHG3'],
        ['ACO2','RP1-101G11.2'],
        ['IGHA1','IGHV2-5'],
        ['TUBA1B','TUBA1C'],
        ['EEF1GP1','RP11-864I4.1'],
        ['RP11-87H9.2','RP11-87H9.4'],
        ['TRBV25-1','TRBV4-1'],
        ['TMEM14B','TMEM14C'],
        ['IGKC','IGKV3D-11'],
        ['RPS18','RPS18P12'],
        ['SDHAP1','SDHAP2'],
        ['PSME2','RP11-815N9.2'],
        ['METAZOA_SRP','RN7SL1'],
        ['ATP5A1','ATP5A1P2'],
        ['RP11-146N23.1','RPS6'],
        ['TRBC2','TRBV5-1'],
        ['CTD-2287O16.1','RPS25'],
        ['AP1G2','JPH4'],
        ['IGKC','IGKV2D-29'],
        ['RP11-278C7.4','YARS2'],
        ['MKRN1','MKRN4P'],
        ['EEF1A1','EEF1A1P13'],
        ['CD99','CD99P1'],
        ['NACA','NACA2'],
        ['HNRNPA1','HNRNPA1L2'],
        ['AFMID','LRG1'],
        ['LL0XNC01-131B10.2','NSA2'],
        ['RP11-203J24.9','ST6GALNAC6'],
        ['TRBC2','TRBV29-1'],
        ['HSPD1','HSPD1P1'],
        ['LL0XNC01-16G2.1','PNMA3'],
        ['TRBV28','TRBV5-1'],
        ['PRKRIR','PRKRIRP1'],
        ['FABP3','RP11-266K22.2'],
        ['P2RY11','PPAN'],
        ['ZDHHC11','ZDHHC11B'],
        ['PRKAR1A','PRKAR1AP'],
        ['FOLH1','FOLH1B'],
        ['C1ORF86','OSMR'],
        ['EIF4A1','EIF4A1P4'],
        ['CDK14','CTB-13L3.1'],
        ['GAPDH','GAPDHP60'],
        ['IGHA2','IGHG3'],
        ['CTA-796E4.5','MYO18B'],
        ['ATP7A','RP5-1000K24.2'],
        ['HDGF','HDGFP1'],
        ['TRGC1','TRGV2'],
        ['LIME1','SLC2A4RG'],
        ['RP11-627K11.1','RPL13A'],
        ['RP11-12M9.3','RPS9'],
        ['MYL2','TTN'],
        ['UBB','UBBP1'],
        ['IGLC1','IGLC3'],
        ['NACA','PRIM1'],
        ['RP11-1029M24.1','RPL17'],
        ['IGKC','IGKV5-2'],
        ['IGLC2','IGLV3-1'],
        ['LDHA','LDHAP3'],
        ['TRBV25-1','TRBV29-1'],
        ['IGHA1','IGHV4-28'],
        ['ACTB','ACTBP9'],
        ['IGKV3-11','IGKV6-21'],
        ['PGK1','PGK1P2'],
        ['EEF1A1','EEF1A1P10'],
        ['RPSA','RPSAP58'],
        ['CXORF65','IL2RG'],
        ['AC078899.1','ACTR2'],
        ['RP11-262D11.2','TPT1'],
        ['CTC-498J12.1','RP11-141O11.2'],
        ['BTN3A1','BTN3A2'],
        ['RP11-179G5.1','RPSA'],
        ['BMS1P20','IGLV1-41'],
        ['HSPA8','HSPA8P8'],
        ['SNX19','SNX19P3'],
        ['NPIPA3','NPIPA5'],
        ['CRYBB2','CRYBB2P1'],
        ['PARP4','PARP4P2'],
        ['C3','HP'],
        ['TPT1','TPT1P1'],
        ['CRP','ORM1'],
        ['ATP1A1','MALAT1'],
        ['IGKV1-8','IGKV3D-20'],
        ['BMS1P20','IGLC1'],
        ['CSDAP1','YBX3'],
        ['RPL19','RPL19P20'],
        ['ATP9A','PLCE1'],
        ['CEP295','SCARNA9'],
        ['RP11-599B13.6','TMEM107'],
        ['AC073063.10','RP11-762I7.5'],
        ['CALR','FGA'],
        ['ATP8B5P','ZFAND6'],
        ['MTHFD1','MTHFD1P1'],
        ['LDLRAP1','RP11-70P17.1'],
        ['IGH@','IGKC'],
        ['IGKV1-16','IGKV1-17'],
        ['ALB','SERPINA1'],
        ['IGHG1','IGHV1-18'],
        ['IGHG1','IGHG3'],
        ['C1ORF86','RP11-181G12.2'],
        ['AC093642.5','FBXO25'],
        ['CTC-378H22.1','CTC-378H22.2'],
        ['UGT2B15','UGT2B17'],
        ['IGLL5','IGLV7-43'],
        ['AC104306.1','RPL31'],
        ['RP11-685B14.1','RPL10A'],
        ['EEF1A1','EEF1A1P24'],
        ['AC005336.4','IPMKP1'],
        ['HSPB6','TNS1'],
        ['CHR22-38_28785274-29006793.1','KREMEN1'],
        ['SAA1','SERPINA1'],
        ['PMS2P3','PMS2P4'],
        ['NPIPA3','PKD1P6'],
        ['IGHG1','IGHV3-23'],
        ['SLC35E2','SLC35E2B'],
        ['HP','HPR'],
        ['GSTM1','GSTM5'],
        ['CACNA1C','CACNA1C-IT2'],
        ['IGHA2','IGHV3-74'],
        ['GLTSCR2','IGK@'],
        ['IGHG1','IGHV3-15'],
        ['ANO3','CTD-2507G9.1'],
        ['PCBP2','UC_338'],
        ['RP11-730G20.2','RP11-793H13.8'],
        ['ATP1A2','ATP1A4'],
        ['LINC00969','SDHAP1'],
        ['TCEAL3','TCEAL6'],
        ['IGHV3-74','RP11-731F5.2'],
        ['IGHD','IGHG3'],
        ['GAPDH','GAPDHP1'],
        ['RPS11','RPS11P5'],
        ['IGKC','IGKV2D-30'],
        ['IGHG3','IGHV1-46'],
        ['YWHAZ','YWHAZP2'],
        ['GPR116','RP11-262H14.7'],
        ['NKAPP1','RHOXF1'],
        ['H3F3B','H3F3C'],
        ['DHRS4','DHRS4L2'],
        ['CH17-472G23.4','PDE4DIP'],
        ['SCD','SCDP1'],
        ['IGK@','MALAT1'],
        ['RPL13A','RPL13AP25'],
        ['AMZ2','RP11-147L13.15'],
        ['CCL21','IGHM'],
        ['XRCC6','XRCC6P2'],
        ['AL590762.6','ZCRB1'],
        ['RP11-122K13.12','TUBGCP2'],
        ['AC024560.3','SDHAP2'],
        ['A2M','FGG'],
        ['TRBC2','TRBV4-1'],
        ['ACTB','POTEM'],
        ['EEF1DP3','FRY'],
        ['LILRA4','LILRB2'],
        ['PHB2','SCARNA12'],
        ['IGHG2','IGHV3-15'],
        ['WBP11','WBP11P1'],
        ['IGLC1','IGLV3-25'],
        ['NDUFB8','NDUFB8P2'],
        ['AC053503.6','MYH7'],
        ['MSN','MSNP1'],
        ['ALB','HPX'],
        ['BBS5','DTD2'],
        ['IGHM','IGHV1-46'],
        ['ANG','RP11-903H12.5'],
        ['RP11-314A20.1','RPS12'],
        ['TTC30A','TTC30B'],
        ['IGHV3-23','RP11-731F5.2'],
        ['IGHV3-21','IGHV3OR16-9'],
        ['ACKR1','IGK@'],
        ['C3','FGA'],
        ['APOBEC3D','APOBEC3F'],
        ['GGT1','RP11-341D18.4'],
        ['PI4KA','PI4KAP1'],
        ['RPS12','SNORA33'],
        ['RPS10','RPS10P2'],
        ['EEF2','GSN'],
        ['IGHA1','IGHV3-72'],
        ['IGHG4','RP11-731F5.2'],
        ['HSP90AA1','HSP90AA2P'],
        ['CTNNA1','CTNNAP1'],
        ['IGHM','IGHV3-7'],
        ['EIF4A1P4','SENP3-EIF4A1'],
        ['IGLL5','IGLV8-61'],
        ['NBPF1','NBPF25P'],
        ['RP11-310N16.1','TPT1'],
        ['RPS15A','RPS15AP1'],
        ['IGLL5','IGLV1-40'],
        ['IGLL5','IGLV3-27'],
        ['IGLC7','IGLJ7'],
        ['FKBP11','RP11-302B13.5'],
        ['LILRA6','LILRB1'],
        ['RPL39','ZNF525'],
        ['IGHM','IGHV3-15'],
        ['NPM1','NPM1P40'],
        ['TCEAL2','TCEAL4'],
        ['IGH@','IGK@'],
        ['CLU','POLR2A'],
        ['IGLC2','IGLV7-43'],
        ['AF127936.9','NRIP1'],
        ['IGLC2','IGLV3-25'],
        ['ITIH4','SERPINA1'],
        ['EEF1A1','EEF1A1P9'],
        ['TRBC2','TRBV11-2'],
        ['IGHA1','IGHV1-2'],
        ['RPS12','RPS12P23'],
        ['TUBG1','TUBG2'],
        ['ZNF561','ZNF562'],
        ['FAM220A','FAM220CP'],
        ['TRAC','TRAV13-1'],
        ['GAPDH','RP11-378G13.2'],
        ['IGKC','IGKV1-9'],
        ['IGLL5','IGLV5-45'],
        ['IGHM','IGKC'],
        ['IGK@','IGK@'],
        ['SNORA31','TPT1'],
        ['RP11-815J4.7','SSBP4'],
        ['RP11-405A12.1','TCP1'],
        ['IGHA1','IGHV3-53'],
        ['H6PD','SPSB1'],
        ['GAPDH','GAPDHP68'],
        ['CTC-479C5.10','EDC4'],
        ['RPS4X','RPS4XP13'],
        ['IGKV1-17','IGKV1D-17'],
        ['IGHA1','IGK@'],
        ['VWF','VWFP1'],
        ['STK24','STK24P1'],
        ['TRAC','TRAV23DV6'],
        ['FLNA','MYH11'],
        ['IGHV2-70','IGHV3-21'],
        ['TRAC','TRAV17'],
        ['AC013717.3','RPL12'],
        ['NKX2-1','RP11-896J10.3'],
        ['IGH@','IGK@'],
        ['SEC14L1','SEC14L1P1'],
        ['BMS1P20','IGLL5'],
        ['LDHA','LDHAP7'],
        ['AOC3','AOC4P'],
        ['IGHG3','IGHV5-51'],
        ['TRAC','TRAV20'],
        ['IGHG2','IGHV3-7'],
        ['RPL4','RPL4P4'],
        ['IGKV1-8','IGKV2D-29'],
        ['RPL15','RPL15P18'],
        ['RPS2','RPS2P5'],
        ['FGG','SERPINA3'],
        ['RNF5','RNF5P1'],
        ['SAA2','SERPINA1'],
        ['RPL13A','RPL13AP20'],
        ['RPL27A','SNORA45B'],
        ['IGKV1-8','IGKV3-20'],
        ['RPL5','RPL5P12'],
        ['FCGR3A','FCGR3B'],
        ['CDC42EP2','POLA2'],
        ['LILRA1','LILRB2'],
        ['KRT5','KRT6A'],
        ['RP11-255B23.1','SRRM1'],
        ['CTC-548K16.6','PKN1'],
        ['CMTM4','RP11-403P17.6'],
        ['IGHA1','IGHG1'],
        ['IGLC1','IGLV2-18'],
        ['MT1E','MT1JP'],
        ['PRC1-AS1','RCCD1'],
        ['SAA1','SAA2-SAA4'],
        ['RP11-297L17.6','TPM3'],
        ['IGHA2','IGHV3-7'],
        ['HSPA1A','HSPA1B'],
        ['SEC63','SEC63P1'],
        ['KCTD10','RP11-256L11.3'],
        ['TRBV28','TRBV3-1'],
        ['IGKC','MYH9'],
        ['FTL','FTLP3'],
        ['HMGB1','SP100'],
        ['TTC3','TTC3P1'],
        ['IGLC2','IGLV2-8'],
        ['GS1-124K5.12','RABGEF1'],
        ['RP11-325P15.2','SSB'],
        ['GLRX','GLRXP3'],
        ['7SK','GSDMD'],
        ['AC015936.3','GFAP'],
        ['LDHA','LDHAP4'],
        ['BAG6','C6ORF47'],
        ['RPS20','RPS20P10'],
        ['EIF4A2','EIF4A2P3'],
        ['ATP1B3','ATP1B3P1'],
        ['IGHG1','IGH@'],
        ['IGKV1-8','IGKV1D-13'],
        ['NME4','Z97634.3'],
        ['AKR7A2','AKR7A3'],
        ['RP11-587D21.1','RPS25'],
        ['RP11-155G14.5','RP11-212P7.3'],
        ['AC005152.3','SOX9-AS1'],
        ['CTA-351J1.1','RPL13A'],
        ['IGHG1','TPT1'],
        ['RP1-309I22.2','SPP1'],
        ['TRBV25-1','TRBV27'],
        ['IGHG1','IGHGP'],
        ['CAMK2N1','VSNL1'],
        ['IGHA1','IGH@'],
        ['RPS23','RPS23P8'],
        ['SULT2A1','TAT'],
        ['RPL34','RPL34P31'],
        ['RP11-578C11.2','YWHAE'],
        ['ACR','SHANK3'],
        ['GAPDH','GAPDHP38'],
        ['TRBC2','TRBV4-2'],
        ['UGT2B11','UGT2B7'],
        ['CTSB','RP1-309I22.2'],
        ['RP11-329A14.1','STRADB'],
        ['PSMC2','RP11-64K7.1'],
        ['LILRA2','LILRB1'],
        ['ABCD4','AC005519.4'],
        ['AP000347.4','IGLL3P'],
        ['RPL7A','RPL7AP8'],
        ['CYP2B6','CYP2B7P'],
        ['TRBV19','TRBV25-1'],
        ['IGH@','IGH@'],
        ['CD22','U62631.5'],
        ['FTH1','FTH1P2'],
        ['RPL7A','RPL7AP60'],
        ['ACTB','IGH@'],
        ['CROCCP2','RP4-565E6.1'],
        ['CYP2A6','LRP1'],
        ['IGLC1','IGLV4-69'],
        ['GUSBP11','IGLL3P'],
        ['CTD-2224J9.8','ZNF765'],
        ['IGHV3-48','RP11-812E19.9'],
        ['RPL3','RPL3P7'],
        ['IGLV2-11','IGLV2-8'],
        ['TRGC2','TRGV2'],
        ['IGKV1-6','IGKV3-11'],
        ['FGB','FGG'],
        ['ACTBP7','ACTG1'],
        ['RPL23A','RPL23AP3'],
        ['HSP90AB1','HSP90AB3P'],
        ['RP11-796G6.1','RPL26'],
        ['AC004453.8','RPL32'],
        ['LINC00484','RP11-305L7.6'],
        ['CTD-2270N23.1','RPL7A'],
        ['OSGIN1','RP11-505K9.4'],
        ['BTBD10','RP11-641A6.9'],
        ['ACTB','ACTG2'],
        ['FBXO44','FBXO6'],
        ['RANBP6','RP11-525E9.1'],
        ['GAPDH','GAPDHP21'],
        ['IGKV1-8','IGKV2D-24'],
        ['IGH@','IGH@'],
        ['RP11-367G18.2','RPS27A'],
        ['RP11-299L17.1','RPS24'],
        ['IGHG4','IGHGP'],
        ['BST1','FAM200B'],
        ['PROS1','PROSP'],
        ['OSTF1','OSTF1P1'],
        ['RP11-85G20.1','RPL18A'],
        ['RPL5','RPL5P18'],
        ['LSAMP','RP11-384F7.2'],
        ['AL928768.3','IGHA1'],
        ['GLUL','RP11-54C4.2'],
        ['RP11-563H6.1','RPL10A'],
        ['LILRA6','LILRB3'],
        ['DPY19L2P1','DPY19L2P3'],
        ['C16ORF74','GINS2'],
        ['IGHA2','IGHV6-1'],
        ['LINC00152','MIR4435-1HG'],
        ['YBX1','YBX1P1'],
        ['FAM193A','RNF4'],
        ['ADAMTSL4-AS1','MCL1'],
        ['ALB','ORM1'],
        ['APLP1','TUBB4A'],
        ['LILRB1','LILRP1'],
        ['IGHG1','IGH@'],
        ['ASLP1','GUSBP11'],
        ['IGHA1','IGHV1-69-2'],
        ['CTD-2086L14.1','RPL13'],
        ['HP','SERPINA3'],
        ['IGHA1','IGHV1-18'],
        ['IGHA1','IGHV3-74'],
        ['IGKV1-8','IGK@'],
        ['C3','FGB'],
        ['IGHM','IGHV3-73'],
        ['CCL3','CCL3L3'],
        ['RP11-79E3.1','RPL31'],
        ['WHAMML1','WHAMMP2'],
        ['AC069368.3','ANKDD1A'],
        ['EIF4B','EIF4BP7'],
        ['ZNF264','ZNF805'],
        ['PRH1','PRH2'],
        ['CTSS','LYZ'],
        ['ACTB','ACTBP8'],
        ['NACA','NACAP1'],
        ['RPL13A','RPL13AP6'],
        ['AMBP','SERPINA1'],
        ['CLCC1','RP11-475E11.9'],
        ['ZNF37A','ZNF37BP'],
        ['IGHG3','RP11-731F5.2'],
        ['PDXDC1','PDXDC2P'],
        ['RP1-283E3.8','SLC35E2B'],
        ['ARPC4','ARPC4-TTLL3'],
        ['TRAC','TRAV12-1'],
        ['ADAMTS7','ADAMTS7P3'],
        ['IGKV2-30','IGKV3-11'],
        ['CTD-2248H3.1','RPS12'],
        ['FKBP1A','FKBP1C'],
        ['RP4-641G12.4','SNW1'],
        ['IGHA1','IGHV1-46'],
        ['DNAJC9','MRPS16'],
        ['POM121','POM121C'],
        ['AC108938.5','LIMS1'],
        ['TRAC','TRAV2'],
        ['IGLC2','IGLV1-47'],
        ['RPL9','RPL9P25'],
        ['NBEAL1','RPL12'],
        ['LINC01513','ROPN1L'],
        ['RPS20','RPS20P24'],
        ['IGLL5','IGLV3-19'],
        ['SNX5','SNX5P1'],
        ['DDX3X','RP11-613H2.2'],
        ['IGKV1-8','IGKV2-24'],
        ['ARHGAP5','RP4-791C19.1'],
        ['AQP7P1','FP325317.1'],
        ['ORM1','SERPINA1'],
        ['CYP2D6','CYP2D7'],
        ['AC002398.12','MYH7'],
        ['LILRB3','LILRB4'],
        ['MST1','MST1P2'],
        ['SVIL','SVILP1'],
        ['STK19','STK19B'],
        ['RP11-159D12.2','RP11-159D12.5'],
        ['TPT1','TPT1P9'],
        ['IGHA1','IGHV4-34'],
        ['IGHV3-33','IGHV3-48'],
        ['ZNF28','ZNF468'],
        ['ANXA2','ANXA2P3'],
        ['RP5-1056L3.3','RPS14'],
        ['TUBB','TUBBP1'],
        ['IGFBP4','IGH@'],
        ['POLRMT','POLRMTP1'],
        ['PLEKHA3','PLEKHA3P1'],
        ['C3','SERPINA1'],
        ['IGHV3-23','RP11-1166P10.8'],
        ['TRAC','TRAV19'],
        ['APOBEC3A','APOBEC3B'],
        ['IGHM','IGHV3-33'],
        ['IGHA2','IGHM'],
        ['ATP5A1','ATP5A1P3'],
        ['RBFOX1','RP11-420N3.3'],
        ['CAMK2N1','RP11-386G11.10'],
        ['GFAP','TUBB4A'],
        ['AC024560.3','LINC00969'],
        ['IGHM','IGHV3OR16-9'],
        ['ESRP2','RP11-96D1.11'],
        ['RPS14','RPS14P8'],
        ['RPS24','RPS24P7'],
        ['AC068491.3','PAFAH1B1'],
        ['RP11-159H3.1','RPL19'],
        ['AKNA','EMP2'],
        ['EEF1A1','EEF1A1P8'],
        ['RBMX','RBMXL1'],
        ['TPT1','TPT1P4'],
        ['IGHV3-23','IGHV3-7'],
        ['LPA','PLG'],
        ['FGG','SERPINA1'],
        ['CTD-2574D22.4','KCTD13'],
        ['IGLC2','IGLV4-69'],
        ['C4A','C4B'],
        ['RPS20','RSF1'],
        ['AC009228.1','FAM228A'],
        ['IGKC','NFIB'],
        ['IGHG3','IGHGP'],
        ['RPL31','RPL31P49'],
        ['PRODH','XXBAC-B444P24.10'],
        ['RPS4X','RPS4XP14'],
        ['RNFT1','RNFT1P3'],
        ['IGHM','IGHV3-21'],
        ['ATP1A1','MLLT6'],
        ['CCL21','IGH@'],
        ['AKR1B10','AKR1B15'],
        ['IGH@','IGH@'],
        ['SDHA','SDHAP3'],
        ['GAPDH','GAPDHP72'],
        ['MYH7','TPM1'],
        ['TRBV25-1','TRBV30'],
        ['IGHA1','IGHV4-31'],
        ['ARMCX3','RP4-545K15.5'],
        ['SEPT7','SEPT7P3'],
        ['NPIPA1','NPIPA5'],
        ['MT1G','MT1M'],
        ['FGA','SAA1'],
        ['ARHGAP42','TMEM133'],
        ['RIN2','RP1-122P22.2'],
        ['GSTA1','GSTA2'],
        ['SCD','SPARC'],
        ['PILRA','STAG3L5P-PVRIG2P-PILRB'],
        ['C1QBP','CTC-524C5.2'],
        ['ALB','CRP'],
        ['RP11-169K16.8','RPS16'],
        ['TRBV25-1','TRBV5-1'],
        ['IGHA1','IGHV3-13'],
        ['FTH1','FTH1P4'],
        ['IGK@','IGK@'],
        ['EEF1D','EEF1DP5'],
        ['TAGLN2','TAGLN2P1'],
        ['IGKV1-5','IGKV3-11'],
        ['FTH1','FTH1P20'],
        ['LILRA6','LILRB4'],
        ['RPL24','RPL24P8'],
        ['RPLP1','RPLP1P6'],
        ['UGT2B10','UGT2B7'],
        ['IGLV2-11','IGLV2-14'],
        ['IGHA1','IGHV6-1'],
        ['HSP90B1','HSP90B2P'],
        ['RP11-113C12.2','RPL15'],
        ['CDK14','CLDN12'],
        ['NBPF10','NBPF8'],
        ['FRG1B','RP11-87H9.2'],
        ['CTC-301O7.4','SLC6A16'],
        ['CD74','IGH@'],
        ['PEBP1','PEBP1P2'],
        ['CFH','CFHR3'],
        ['GSTM2','GSTM4'],
        ['IGHM','IGHV1-69'],
        ['ACTG1','AMY2B'],
        ['IGKV1-5','IGKV1-8'],
        ['IGHM','IGHV3-53'],
        ['NBPF14','NOTCH2NL'],
        ['AFM','RP11-622A1.2'],
        ['IGHD','IGHG1'],
        ['MEIS3P1','MEIS3P2'],
        ['MBP','RP11-386G11.10'],
        ['KRBOX1','ZNF662'],
        ['IFITM3','RP11-396M11.1'],
        ['RPS10-NUDT3','RPS10P2'],
        ['NXF1','TMEM223'],
        ['IGHV3-21','IGHV3-48'],
        ['UBB','UBBP4'],
        ['HNRNPC','HNRNPCP2'],
        ['IGLC1','IGLV2-14'],
        ['AC092798.2','RPL32'],
        ['CYP2E1','SAA1'],
        ['FTH1','FTH1P8'],
        ['RP11-401L13.4','RPL19'],
        ['APOC1','APOC1P1'],
        ['TDGF1','TDGF1P3'],
        ['IGKV1-8','IGKV2-29'],
        ['RPL10','RPL10P15'],
        ['AC007566.10','GATAD1'],
        ['INPP5K','MYO1C'],
        ['ANG','RNASE4'],
        ['APOC3','SERPINA1'],
        ['PPIE','RP11-69E11.4'],
        ['IGH@','IGH@'],
        ['HNRNPA1','HNRNPA1P54'],
        ['RNASE4','RP11-903H12.5'],
        ['EMB','EMBP1'],
        ['OBP2A','OBP2B'],
        ['AMZ2','AMZ2P1'],
        ['CTC-575D19.1','RPL10'],
        ['MAMDC4','PHPT1'],
        ['CES1','CES1P1'],
        ['MXRA5','MXRA5Y'],
        ['LRRFIP1','LRRFIP1P1'],
        ['GPX3','IGFBP5'],
        ['THRAP3','THRAP3P1'],
        ['ZNF354A','ZNF354B'],
        ['ALB','FGA'],
        ['SAA1','SAA2'],
        ['IGLC2','IGLV2-14'],
        ['ACTB','RP11-809F4.2'],
        ['MBP','TUBB4A'],
        ['ACTB','ACTBP1'],
        ['IGHG1','IGHV3-73'],
        ['GGA2','ZFAND5'],
        ['VDAC1','VDAC1P11'],
        ['NBPF11','NBPF12'],
        ['IGKC','IGKV1-8'],
        ['EIF1','EIF1P7'],
        ['AC115617.2','ACAA2'],
        ['AC074286.1','NFE2L2'],
        ['RP11-444D3.1','SOX5'],
        ['LILRA1','LILRB1'],
        ['PSIP1','RP11-54D18.4'],
        ['ICA1L','PYURF'],
        ['CAP1','RP11-550F7.1'],
        ['IGHA1','IGH@'],
        ['FTH1','GPX3'],
        ['IFITM3','IFITM9P'],
        ['IGHV3-11','IGHV3-48'],
        ['RPL5','RPL5P22'],
        ['ANKDD1A','PLEKHO2'],
        ['RP11-124N14.3','VIM-AS1'],
        ['FAM131C','RP11-5P18.10'],
        ['MB','MYH7'],
        ['CTC-260E6.2','RPS16'],
        ['IGHG1','IGHV3-74'],
        ['IGHM','IGHV3-49'],
        ['PI4KAP1','PI4KAP2'],
        ['IGLC2','IGLC6'],
        ['TRAC','TRAV27'],
        ['HARS2','ZMAT2'],
        ['IGKC','IGKV3-11'],
        ['CTD-2396E7.10','SYP'],
        ['IGLL5','IGLV2-14'],
        ['RP11-87H9.2','RP11-88I18.2'],
        ['HP','SERPINA1'],
        ['IGKC','IGKV1-6'],
        ['RP11-829H16.2','RPL27'],
        ['IGLC1','IGLV6-57'],
        ['IGKV1-27','IGKV1OR2-108'],
        ['ITIH4','RP5-966M1.6'],
        ['RP1','RP11-53M11.3'],
        ['ACTB','ACTBP2'],
        ['TRGC2','TRGV5'],
        ['AC093698.2','RPL12'],
        ['FCGR2A','FCGR2C'],
        ['TRAC','TRAV8-3'],
        ['NAMPT','NAMPTP3'],
        ['TUBA1A','TUBA1C'],
        ['RN7SL1','RN7SL674P'],
        ['CCT6P1','CCT6P3'],
        ['RPL3','RPL3P3'],
        ['EBPL','RP11-744D14.1'],
        ['GIMAP1-GIMAP5','GIMAP2'],
        ['IGH@','IGH@'],
        ['IGLC1','IGLC7'],
        ['ANKRD13A','RP11-946P6.6'],
        ['DES','MYH7'],
        ['LCAT','SLC12A4'],
        ['RP11-864N7.2','RPS12'],
        ['IGHA2','IGHG1'],
        ['AC093838.4','SMPD4'],
        ['RPS24P17','SLC12A3'],
        ['SFTPA1','SFTPA2'],
        ['CYP21A1P','CYP21A2'],
        ['DHRS1','RABGGTA'],
        ['RPS20','RPS20P14'],
        ['RP11-57H12.2','TMEM56'],
        ['UQCRB','UQCRBP1'],
        ['FTH1','FTH1P15'],
        ['HSPB1','HSPB1P1'],
        ['RAN','RANP1'],
        ['IGK@','IGK@'],
        ['IGH@','IGKC'],
        ['RPS3','RPS3P6'],
        ['ACTB','CH17-12M21.1'],
        ['RPL9','TIPIN'],
        ['NTAN1','NTAN1P2'],
        ['IGK@','RPS11'],
        ['ZNF510','ZNF782'],
        ['IGLL1','IGLL5'],
        ['FAHD2B','FAHD2CP'],
        ['OSBPL1A','RP11-799B12.4'],
        ['RPL5','RPL5P34'],
        ['KIAA1147','RP5-894A10.2'],
        ['IGKC','IGKV2D-28'],
        ['GSN','PCBP1'],
        ['IGLL5','IGLV2-23'],
        ['GPX3','IGFBP4'],
        ['BMS1P20','LL22NC03-80A10.6'],
        ['IGHA1','IGHA2'],
        ['AC005562.1','LRRC37BP1'],
        ['RPLP0','RPLP0P6'],
        ['IGKC','IGKV1D-16'],
        ['RN7SK','RN7SKP230'],
        ['MT2A','MT2P1'],
        ['HCAR2','HCAR3'],
        ['GAPDH','RP11-321C24.4'],
        ['SH3BGR','WRB'],
        ['IFITM2','IFITM3'],
        ['LYZ','RP11-186B7.4'],
        ['RP11-170M17.2','RPL7A'],
        ['RP11-727F15.13','STX5'],
        ['MAPK8IP1','RP11-259G18.2'],
        ['TRAC','TRAV1-2'],
        ['ACTB','POTEF'],
        ['ACTA1','ACTB'],
        ['GUSBP11','KB-1572G7.2'],
        ['LYZ','RSL1D1'],
        ['UGT2B28','UGT2B7'],
        ['MYL6','MYL6B'],
        ['GLTSCR2','RP11-255M2.3'],
        ['DTX2','DTX2P1'],
        ['EEF2','FLNA'],
        ['IGHM','IGHV5-51'],
        ['ALB','HP'],
        ['IGKC','IGKV2D-24'],
        ['RP4-575N6.2','RPS20'],
        ['TRAC','TRAV6'],
        ['TMEM107','VAMP2'],
        ['FAM35A','FAM35BP'],
        ['CCT8','CCT8P1'],
        ['CFDP1','RP11-252K23.2'],
        ['RPL9P25','UBE2Q2P6'],
        ['FKBP9','FKBP9P1'],
        ['IGHV3-7','RP11-731F5.2'],
        ['LYZ','MROH1'],
        ['IGLV2-23','IGLV2-8'],
        ['GPX3','TXNIP'],
        ['EEF2','PLIN4'],
        ['LILRB1','LILRB3'],
        ['AC005682.7','RPL12'],
        ['ACTB','GPX3'],
        ['CIITA','RP11-876N24.3'],
        ['FRG1','RP11-87H9.2'],
        ['TRBC2','TRBV24-1'],
        ['IGHA1','IGHV3-33'],
        ['FGA','FGB'],
        ['IGKC','IGKV1-5'],
        ['FTH1','FTH1P23'],
        ['IGHM','IGHV2-5'],
        ['NAP1L1','NAP1L1P1'],
        ['ARHGAP21','RP11-296E7.1'],
        ['IGKC','IGKV1-16'],
        ['GLUL','GLULP4'],
        ['IGH@','IGKV3-11'],
        ['RPL5','RPL5P17'],
        ['IGHA2','IGHV3-21'],
        ['IGHA1','IGHG3'],
        ['RPS18','RPS18P9'],
        ['IGHA1','IGHV5-51'],
        ['EEF1A1','EEF1A1P11'],
        ['IGKV1-16','IGKV1-8'],
        ['CNPY3','RP3-475N16.1'],
        ['LDHA','LDHAP5'],
        ['JAK1','RP4-535B20.1'],
        ['HNRNPC','HNRNPCP1'],
        ['ATAD3A','ATAD3B'],
        ['CRP','FGG'],
        ['RPL29','RPL29P12'],
        ['DES','MYL3'],
        ['ACTB','IGKC'],
        ['GOT2','GOT2P3'],
        ['AP000347.4','KB-1572G7.2'],
        ['TRBC2','TRBV2'],
        ['AC007969.5','RPS15'],
        ['CHST7','ZNF674-AS1'],
        ['AC024560.3','SDHAP1'],
        ['FPGT-TNNI3K','TNNI3K'],
        ['EIF5A','EIF5AL1'],
        ['ACOT1','ACOT2'],
        ['NGRN','PERP'],
        ['PNO1','RP11-597G23.1'],
        ['ATP6AP2','RP11-332L8.1'],
        ['GIMAP1','GIMAP2'],
        ['LDHA','LDHAP2'],
        ['IGHG2','IGHV3-72'],
        ['CTSD','IGK@'],
        ['RP11-264B17.3','SPNS1'],
        ['MYL12B','RP11-265N6.3'],
        ['SERPINA1','SERPINA3'],
        ['IGHA1','IGHV3-30'],
        ['RPL37A','RPL37AP8'],
        ['BGN','GADL1'],
        ['RPSA','RPSAP8'],
        ['RPL15','RPL15P20'],
        ['IGHG1','IGHV3-49'],
        ['NDUFV2P1','RP11-21J18.1'],
        ['ZNF286A','ZNF286B'],
        ['DES','TPM2'],
        ['RSL24D1','RSL24D1P11'],
        ['TRAC','TRAV26-2'],
        ['RPL5','RPL5P3'],
        ['AIFM3','LZTR1'],
        ['TPM3','TPM3P1'],
        ['CTC-448F2.3','PPIA'],
        ['C10ORF32','LYZ'],
        ['IGHM','IGH@'],
        ['IGLC1','IGLV8-61'],
        ['BBS5','RP11-176H8.1'],
        ['EEF1A1','RP11-553P9.1'],
        ['HERC2','HERC2P10'],
        ['TRAC','TRAV39'],
        ['KLHL5','RP11-360F5.3'],
        ['DNAJC25-GNG10','GNG10'],
        ['AC092610.12','RPL22'],
        ['C3ORF62','USP4'],
        ['GPS2P1','RP11-542C16.2'],
        ['IGLC2','IGLV3-19'],
        ['DUS3L','PRR22'],
        ['IGLC1','IGLV3-19'],
        ['EPAS1','TPT1'],
        ['PDLIM1','PDLIM1P1'],
        ['GABARAP','RP11-6B6.3'],
        ['ANO1','RP11-626H12.1'],
        ['IGH@','IGH@'],
        ['TRAC','TRAV13-2'],
        ['TRBC2','TRBV7-3'],
        ['ACTB','ACTBP11'],
        ['A2M','PZP'],
        ['TRAC','TRAV24'],
        ['IGKV1-6','IGKV1-8'],
        ['RP11-624L12.1','RPL17'],
        ['IGHG3','IGHV3-15'],
        ['NAT8','NAT8B'],
        ['IGLL5','IGLV2-28'],
        ['EEF1A1','EEF1A1P29'],
        ['CRP','SERPINA1'],
        ['HNRNPA3','HNRNPA3P12'],
        ['IGKC','IGKV3-20'],
        ['AC138517.4','NCOA4'],
        ['NAV3','RP1-34H18.1'],
        ['IGH@','IGH@'],
        ['PRB3','PRB4'],
        ['ACTA2','ACTB'],
        ['AC069213.1','MUC20'],
        ['IGHV3-38','IGHV3-53'],
        ['CTSS','RP11-1143G9.4'],
        ['LSM10','STK40'],
        ['IGKV1-8','IGKV2D-30'],
        ['A2M','ALB'],
        ['RP11-618N24.1','TPM3'],
        ['RP11-204C16.4','YWHAZ'],
        ['PILRA','PILRB'],
        ['RPL7A','RPL7AP30'],
        ['LYZ','MRPS30'],
        ['FAU','FAUP1'],
        ['PREX1','RP5-906C1.1'],
        ['PLEKHM1','PLEKHM1P'],
        ['IGHG1','IGHV4-39'],
        ['IGLC2','IGLV2-28'],
        ['RPL3','RPL3P2'],
        ['RP5-966M1.6','SERPINA1'],
        ['SET','SETP3'],
        ['ACAD8','GLB1L3'],
        ['AC002075.4','RPS27'],
        ['RP11-234A1.1','RPS18'],
        ['TRAC','TRAV35'],
        ['RP11-473N11.2','SUPT16H'],
        ['RPL3','RPL3P1'],
        ['TRBC2','TRBV3-1'],
        ['RPL5','RPL5P29'],
        ['PSMD11','ZNF207'],
        ['CD300A','CD300C'],
        ['RPL14','RPL14P1'],
        ['RPL37','RPL37P6'],
        ['GPX3','TIMP3'],
        ['GRK6','GRK6P1'],
        ['RP11-477G18.1','RPS12'],
        ['IGKV1-27','IGKV3-11'],
        ['KRT8','KRT8P11'],
        ['FTH1','FTH1P1'],
        ['SAA1','SCD'],
        ['HERC2P2','HERC2P3'],
        ['RP4-576H24.4','SIRPA'],
        ['IGHM','MYH11'],
        ['CTD-2014B16.2','FTH1'],
        ['IGHA1','IGHV2-26'],
        ['RILPL1','RP11-972P1.7'],
        ['MPEG1','RP11-1036E20.7'],
        ['PKN2','TCEB1P19'],
        ['XRCC6','XRCC6P1'],
        ['AC007228.9','TOB2'],
        ['ACTN1','HMGN1'],
        ['POM121C','POMZP3'],
        ['RPS15A','RPS15AP10'],
        ['TRBV18','TRBV28'],
        ['IGHG1','IGHV2-26'],
        ['CTB-50E14.5','ZFAND5'],
        ['GAPDH','GAPDHP40'],
        ['KIAA1586','ZNF451'],
        ['NDUFA6-AS1','RP4-669P10.19'],
        ['RPL3','RPL3P6'],
        ['IGHM','IGHV1-18'],
        ['EEF1A1','EEF1A1P4'],
        ['FTH1','FTH1P16'],
        ['YWHAE','YWHAEP1'],
        ['CTD-2021A8.2','RP11-274B21.3'],
        ['COL4A2','GPX3'],
        ['EEF1A1','EEF1A1P19'],
        ['CSNK2B','LY6G5B'],
        ['RP11-632C17__A.1','RPL29'],
        ['RP4-796I17.5','RPL35A'],
        ['RPL36AL','RPL36AP21'],
        ['RP11-20O24.1','RPS27'],
        ['IGHA1','IGHV1-24'],
        ['RP3-486I3.4','RPS5'],
        ['CLU','NRGN'],
        ['RASSF4','ZNF22'],
        ['SKP1','SKP1P1'],
        ['RP11-372E1.1','RPL8'],
        ['FGB','SERPINA1'],
        ['AC006946.16','CECR7'],
        ['CS','CTB-52I2.4'],
        ['NBPF2P','NBPF3'],
        ['C10ORF10','RP1-309I22.2'],
        ['IGHG1','IGHV3-43'],
        ['IGKC','IGKV3-15'],
        ['HSP90AB1','HSP90AB2P'],
        ['IGLV2-11','IGLV2-18'],
        ['LILRB4','LILRB5'],
        ['EEF1GP5','RP11-864I4.1'],
        ['IGHG1','IGHV5-51'],
        ['IGKC','IGKV4-1'],
        ['NPIPP1','PDXDC2P'],
        ['GLUL','SBF2'],
        ['C9ORF172','PHPT1'],
        ['PTGES3','PTGES3P1'],
        ['DPY19L2','DPY19L2P1'],
        ['HMGN2P15','RAMP2-AS1'],
        ['FGF7','RP11-262H14.11'],
        ['AZGP1','AZGP1P1'],
        ['NBEA','NBEAP1'],
        ['RP11-1035H13.3','RPS15AP1'],
        ['IGLC1','IGLV5-45'],
        ['HSPE1','HSPE1P7'],
        ['IGK@','MYH11'],
        ['SEPT7','SEPT7P2'],
        ['IGLL5','IGLV7-46'],
        ['AP002381.2','RPL15'],
        ['BRICD5','PGP'],
        ['BRS3','HTATSF1'],
        ['IGLC2','IGLL5'],
        ['PRPF4B','RP11-78H18.2'],
        ['EEF2','IGH@'],
        ['SMG1P1','SMG1P3'],
        ['GLTP','U52111.14'],
        ['AC067945.3','RAB1A'],
        ['AC008440.5','RP11-626G3.1'],
        ['LTB4R2','NOP9'],
        ['IGHG1','IGHG2'],
        ['FTH1','FTH1P10'],
        ['RP11-252I13.1','USP7'],
        ['IGLV7-43','IGLV7-46'],
        ['RP11-106A1.2','RPL23A'],
        ['IGLC2','IGLV9-49'],
        ['GPX3','UMOD'],
        ['IGKV1-8','IGKV1D-8'],
        ['IGLL5','IGLV3-9'],
        ['ATP5H','ATP5HP4'],
        ['SNORD3A','SNORD3B-2'],
        ['RP11-36C20.1','RPS8'],
        ['IGH@','IGH@'],
        ['LRR1','RHOQ'],
        ['CTB-119C2.1','HNRNPA2B1'],
        ['EXOSC3','EXOSC3P1'],
        ['AQP7P1','AQP7P2'],
        ['IGHM','IGHV4-34'],
        ['AC013470.6','SSB'],
        ['IGFBP7','IGK@'],
        ['SRPK2','SRPK2P'],
        ['EEF1A1','EEF1A1P12'],
        ['RP11-446N19.1','RP11-96H19.1'],
        ['VAMP5','VAMP8'],
        ['RP11-849F2.10','RPS24'],
        ['TRBV18','TRBV25-1'],
        ['TRBV2','TRBV25-1'],
        ['RP1-89D4.1','RPS24'],
        ['MYH1','MYH2'],
        ['AQP7','FP325317.1'],
        ['IGLC1','IGLV1-40'],
        ['CTD-2396E7.10','MALAT1'],
        ['RP4-583P15.14','SLC2A4RG'],
        ['HCG11','LINC00657'],
        ['LINC00665','LINC01535'],
        ['ACTB','IGHM'],
        ['NACA3P','NACAP1'],
        ['INO80C','RP11-322E11.5'],
        ['SEPT7','SEPT7P6'],
        ['MRPS6','SLC5A3'],
        ['TIPIN','UBE2Q2P6'],
        ['BANP','RRN3P2'],
        ['IGLL5','IGLV3-10'],
        ['GPX3','RPL8'],
        ['TRAC','TRAV29DV5'],
        ['BANP','RP11-426C22.5'],
        ['ARL2BP','RP11-407G23.4'],
        ['RPL7A','RPL7AP31'],
        ['IGLC1','IGLC2'],
        ['LILRA6','LILRB5'],
        ['IGHA1','IGHV3-49'],
        ['RP11-10G12.1','RPS27'],
        ['IGHA2','IGHV3-23'],
        ['IGKV1-17','IGKV3-11'],
        ['PTEN','PTENP1'],
        ['NPM1','NPM1P6'],
        ['CTA-276O3.4','RPS29'],
        ['IGLC2','IGLV3-10'],
        ['MOCS1','RP11-61I13.3'],
        ['PAGR1','PRRT2'],
        ['RP11-326A19.5','VAPA'],
        ['GLUL','GPX3'],
        ['MT1E','MT1P3'],
        ['LGMN','LGMNP1'],
        ['HSPA8','HSPA8P5'],
        ['FGA','SERPINA1'],
        ['RPL5','RPL5P23'],
        ['IGLV3-16','IGLV3-25'],
        ['IGKV2-29','IGKV3-11'],
        ['PTMA','PTMAP4'],
        ['MMADHC','RP1-145M24.1'],
        ['HK2','HK2P1'],
        ['PABPC1','PABPC1P4'],
        ['CROCC','CROCCP2'],
        ['IGHG1','IGHV1-69-2'],
        ['C3ORF17','RP11-572M11.3'],
        ['IGKV1-8','IGKV6-21'],
        ['RP11-350G13.1','RPSA'],
        ['HSP90AA1','HSP90AA4P'],
        ['RP11-524O24.2','RPL26'],
        ['AC005251.3','RPL23'],
        ['RP11-364B6.1','THAP3'],
        ['AC092933.3','RPS29'],
        ['RP11-385F5.4','RP11-385F5.5'],
        ['IGHG1','IGHV2-5'],
        ['IGLV3-10','IGLV3-25'],
        ['KCNK15','RP4-781B1.5'],
        ['FGB','HP'],
        ['RP11-423H2.1','THOC3'],
        ['IGLC1','IGLV10-54'],
        ['IGLV3-1','IGLV3-21'],
        ['AKNA','RP11-9M16.2'],
        ['MYH6','MYH7'],
        ['ZNF33A','ZNF33B'],
        ['CTD-2144E22.10','VPS35'],
        ['CHP1','LRG1'],
        ['AC005037.6','MTND4P23'],
        ['IGHG3','IGH@'],
        ['ALB','SERPINA3'],
        ['HSP90AA1','RP11-144L1.8'],
        ['RPL31','RPL31P11'],
        ['IGHA2','IGHV5-51'],
        ['TRAC','TRAV12-3'],
        ['IGKV1-8','IGKV2-26'],
        ['IGKV2-26','IGKV3-11'],
        ['ARPC3','ARPC3P1'],
        ['PTGDR2','RP11-804A23.4'],
        ['ACTB','MYH11'],
        ['CFHR3','CFHR4'],
        ['DES','MYL2'],
        ['IGH@','IGH@'],
        ['ACKR1','CADM3'],
        ['IFITM2','IFITM9P'],
        ['BTF3','BTF3P8'],
        ['IGLC2','IGLV2-11'],
        ['FTL','MYH11'],
        ['ANXA2','ANXA2P1'],
        ['PTMA','RP11-529H20.3'],
        ['RP4-673D20.3','RP5-968J1.1'],
        ['IGKV1-8','IGKV3D-15'],
        ['LAIR1','LAIR2'],
        ['IGHM','IGHV3-72'],
        ['LPL','SCD'],
        ['IGLC1','IGLV2-11'],
        ['AC079922.2','RPS20'],
        ['RPS15A','RPS15AP40'],
        ['RP4-800G7.2','ZNF783'],
        ['NPIPA2','NPIPP1'],
        ['CLCNKA','CLCNKB'],
        ['LAP3','LAP3P2'],
        ['AC068831.6','CERK'],
        ['RP11-385D13.1','TRIM16L'],
        ['IGKV1-5','IGKV1-9'],
        ['IGHG2','IGHGP'],
        ['IGHD','IGHV5-51'],
        ['RP11-274B21.3','RP11-384K6.2'],
        ['PDIA3','PDIA3P1'],
        ['RAD21','RP3-347M6.2'],
        ['SAA1','SAA4'],
        ['PRR4','TAS2R15P'],
        ['GPX1','GPX1P1'],
        ['TRBC2','TRBV18'],
        ['CYP2C19','CYP2C9'],
        ['GPX3','RP1-309I22.2'],
        ['TRBC2','TRBV7-7'],
        ['SNHG3','SNORA73B'],
        ['BEX1','BEX2'],
        ['RPS4X','RPS4XP6'],
        ['IGHV3-15','IGHV3-72'],
        ['APOLD1','SERBP1'],
        ['IGHG1','IGHV3OR16-9'],
        ['RPS4X','RPS4XP22'],
        ['PSME2','PSME2P2'],
        ['FAM212B','PSMB2'],
        ['RP11-404G16.2','TF'],
        ['HNRNPK','HNRNPKP4'],
        ['IGKV1-8','IGKV2D-28'],
        ['DNAJC18','ECSCR'],
        ['IGHG1','IGHV3-33'],
        ['AC073063.10','SARNP'],
        ['IGKV3-20','IGKV3D-20'],
        ['ATF4','ATF4P4'],
        ['CLEC2D','RP11-705C15.2'],
        ['USP32','USP32P1'],
        ['MYH11','TPM2'],
        ['ACTB','ACTG1P14'],
        ['FZD4','RP11-736K20.5'],
        ['IGH@','TPT1'],
        ['CDRT4','TVP23C'],
        ['TRGC1','TRGC2'],
        ['MYH7','TTN'],
        ['IGLL5','IGLV6-57'],
        ['IGLL5','IGLV2-11'],
        ['TRBV19','TRBV28'],
        ['LRG1','TAT'],
        ['RP11-285F7.2','TMEM63A'],
        ['RN7SK','RN7SKP203'],
        ['MYH11','RP1-309I22.2'],
        ['SPP1','TIMP3'],
        ['RP11-120B7.1','RPS20'],
        ['RP1-249H1.3','RPL30'],
        ['IGH@','MYLK'],
        ['IGHM','IGHV3-23'],
        ['SET','SETP4'],
        ['CPNE1','RBM12'],
        ['RPS21','RPS21P4'],
        ['IGKV1-8','IGKV2-30'],
        ['CRLF2','CTSS'],
        ['SAG','VPS13C'],
        ['HNRNPH1','HNRNPH1P2'],
        ['CTD-2396E7.9','SYP'],
        ['RPL12','RPL12L3'],
        ['IGHV4-28','IGHV4-59'],
        ['IGLC2','IGLV3-21'],
        ['RP9','RP9P'],
        ['DSTN','DSTNP4'],
        ['IGH@','IGH@'],
        ['RP11-219A15.1','USP32'],
        ['CD68','LYZ'],
        ['IGLC1','IGLV9-49'],
        ['IGLL5','IGLV2-18'],
        ['IGKC','IGKV3D-20'],
        ['DSTN','DSTNP2'],
        ['CTD-2017D11.1','ZNF254'],
        ['ASAH2','ASAH2B'],
        ['RP4-583P15.15','SLC2A4RG'],
        ['METTL2A','METTL2B'],
        ['XRCC6','XRCC6P5'],
        ['IGHG1','RP11-812E19.9'],
        ['IGLL3P','KB-1572G7.3'],
        ['COMMD1','RPSA'],
        ['CTD-2554C21.1','IST1'],
        ['DES','MYL9'],
        ['HSPA8','HSPA8P7'],
        ['CYP2C8','CYP2C9'],
        ['IGKV1-8','IGKV3-15'],
        ['TRBC2','TRBV20-1'],
        ['SNX19','SNX19P2'],
        ['ERV3-1','ZNF117'],
        ['KIF27','RP11-213G2.3'],
        ['FGB','ORM1'],
        ['TUBA1A','TUBAP2'],
        ['TRBV27','TRBV28'],
        ['RPL31','RPL31P63'],
        ['TRBV2','TRBV28'],
        ['PAFAH1B1','RN7SL608P'],
        ['ACSM2A','ACSM2B'],
        ['FGG','SAA1'],
        ['DES','MYH11'],
        ['CALM2','CALM2P2'],
        ['ALB','FGB'],
        ['FGG','HP'],
        ['EXOSC3P1','RP11-613M10.9'],
        ['RP11-463O12.5','VPS37B'],
        ['AC074183.4','SNHG7'],
        ['ACTB','IGH@'],
        ['PDPR','RP11-252A24.2'],
        ['ACTB','ACTG1'],
        ['PCBP2-OT1','RP11-793H13.8'],
        ['FAN1','RP11-540B6.6'],
        ['MUM1','MYH11'],
        ['GCSAML','FAM129A'],
        ['CTSB','HNRNPH1'],
        ['CKM','CTSB'],
        ['AC138430.4','FAM129A'],
        ['CKM','SYNPO2'],
        ['LPCAT3','LRG1'],
        ['TG','NFE2L2'],
        ['APOL1','SCYL3'],
        ['GMPPB','LRG1'],
        ['PRR4','LYZ'],
        ['MYOZ1','CTSB'],
        ['MUM1','MYH11'],
        ['SLC29A1','HSP90AB1'],
        ['IGKV1D-39','AC096579.13'],
        ['TLDC2','ADIPOQ'],
        ['EMG1','PPP3CA'],
        ['RPL24','ZBTB11'],
        ['SCARB2','MDC1'],
        ['SEPT9','C5AR1'],
        ['ARSD','PIK3C2B'],
        ['IL3RA','GFAP'],
        ['IGKV2D-30','AC096579.13'],
        ['BLVRA','VMA21'],
        ['ADIPOQ','ANKRD31'],
        ['IGKV3D-20','AC096579.13'],
        ['PLIN1','EEF2'],
        ['RPS29','SYN3'],
        ['AC004237.1','BAIAP2-AS1'],
        ['IGKV4-1','AC096579.13'],
        ['GHR','COX10-AS1'],
        ['FOXM1','F11R'],
        ['IGKV2D-29','AC096579.13'],
        ['LYPD5','KCNN4'],
        ['IGKV3D-11','AC096579.13'],
        ['SYN3','PTRF'],
        ['IDI2-AS1','WDR37'],
        ['PLEKHA1','RAB17'],
        ['CWF19L1','ADIPOQ-AS1'],
        ['PLIN4','SYN3'],
        ['APOLD1','DPF3'],
        ['MRPL51','RBM3'],
        ['SUCLA2-AS1','NUDT15'],
        ['LRRC75A-AS1','SYN3'],
        ['KALRN','STOM'],
        ['SYN3','PLIN4'],
        ['CTSB','VSTM1'],
        ['PARL','MAP6D1'],
        ['CDK11B','IGHG1'],
        ['CRTAP','MCAM'],
        ['RPS8','IGH@'],
        ['C7','IGH@'],
        ['ATP1A1','CTSB'],
        ['AC018766.4','PTOV1-AS2'],
        ['ACTB','IGH@'],
        ['AL928768.3','IGH@'],
        ['CCL21','IGH@'],
        ['CCL21','IGK@'],
        ['CD74','IGH@'],
        ['CD74','IGK@'],
        ['ELFN2','ELFN2'],
        ['FLNA','IGK@'],
        ['FOXG1','LINC01551'],
        ['IGHA1','IGH@'],
        ['IGHA2','IGH@'],
        ['IGHD','IGH@'],
        ['IGHG1','IGH@'],
        ['IGHG1','IGHV3-65'],
        ['IGHG2','IGH@'],
        ['IGHG3','IGH@'],
        ['IGHGP','IGH@'],
        ['IGH@','IGH@'],
        ['IGHM','IGH@'],
        ['IGHM','IGK@'],
        ['IGHV1-12','IGH@'],
        ['IGHV1-18','IGH@'],
        ['IGHV1-18','IGHV7-81'],
        ['IGHV1-24','IGH@'],
        ['IGHV1-2','IGH@'],
        ['IGHV1-3','IGH@'],
        ['IGHV1-46','IGH@'],
        ['IGHV1-58','IGH@'],
        ['IGHV1-69-2','IGH@'],
        ['IGHV1-69','IGH@'],
        ['IGHV2-26','IGH@'],
        ['IGHV2-5','IGH@'],
        ['IGHV3-11','IGH@'],
        ['IGHV3-13','IGH@'],
        ['IGHV3-15','IGH@'],
        ['IGHV3-20','IGH@'],
        ['IGHV3-21','IGH@'],
        ['IGHV3-23','IGH@'],
        ['IGHV3-23','IGHV3-43'],
        ['IGHV3-30','IGH@'],
        ['IGHV3-33','IGH@'],
        ['IGHV3-43','IGH@'],
        ['IGHV3-48','IGH@'],
        ['IGHV3-49','IGH@'],
        ['IGHV3-53','IGH@'],
        ['IGHV3-64','IGH@'],
        ['IGHV3-66','IGH@'],
        ['IGHV3-71','IGH@'],
        ['IGHV3-72','IGH@'],
        ['IGHV3-73','IGH@'],
        ['IGHV3-74','IGH@'],
        ['IGHV3-7','IGH@'],
        ['IGHV3OR16-8','IGH@'],
        ['IGHV3OR16-9','IGH@'],
        ['IGHV4-28','IGH@'],
        ['IGHV4-31','IGH@'],
        ['IGHV4-34','IGH@'],
        ['IGHV4-39','IGH@'],
        ['IGHV4-55','IGH@'],
        ['IGHV4-59','IGH@'],
        ['IGHV5-51','IGH@'],
        ['IGHV6-1','IGH@'],
        ['IGHV7-81','IGH@'],
        ['IGKC','IGH@'],
        ['IGKC','IGK@'],
        ['IGK@','IGH@'],
        ['IGK@','IGK@'],
        ['IGKV1-27','IGK@'],
        ['IGKV1-8','IGK@'],
        ['IGKV1D-43','IGK@'],
        ['IGKV1D-8','IGK@'],
        ['IGKV2-24','IGK@'],
        ['IGKV2D-24','IGK@'],
        ['IGKV2D-28','IGK@'],
        ['IGKV2D-29','IGK@'],
        ['IGKV2D-30','IGK@'],
        ['IGKV3-11','IGK@'],
        ['IGKV3-20','IGK@'],
        ['IGKV3-7','IGK@'],
        ['IGKV3D-11','IGK@'],
        ['IGKV3D-15','IGK@'],
        ['IGKV3D-20','IGK@'],
        ['IGKV6-21','IGK@'],
        ['MALAT1','IGH@'],
        ['MYH11','IGH@'],
        ['NFIB','IGK@'],
        ['RP11-1166P10.8','IGH@'],
        ['TNS1','IGK@'],
        ['UGT2B4','UGT2B26P'],
        ['HSPB6','PLIN4'],
        ['APLP2','HLA@'],
        ['BTBD1','SPP1'],
        ['CD99','KANK2'],
        ['GSN','TIMP3'],
        ['IGF2','MYH11'],
        ['IGF2R','IGFBP4'],
        ['IGFBP5','NFIX'],
        ['PHGDH','SCD'],
        ['SVEP1','TIMP3'],
        ['ACTB','SENP3-EIF4A1'],
        ['ADGRL1','SUZ12'],
        ['AGAP11','BMS1'],
        ['BANP','RRN3'],
        ['BRS3','HTATSF1'],
        ['C16ORF46','GCSH'],
        ['ACTB','SENP3-EIF4A1'],
        ['ADGRL1','SUZ12'],
        ['AGAP11','BMS1'],
        ['ATP1A1','SUN1'],
        ['BANP','RRN3'],
        ['BRS3','HTATSF1'],
        ['C16ORF46','GCSH'],
        ['HCG4','XXBAC-BPG248L24.12'],
        ['HSPA8','MALAT1'],
        ['GSTM3','NFIC'],
        ['MYH11','UBE2N'],
        ['CTSB','LTBP2'],
        ['FTL','TPO'],
        ['GNAS','TMBIM6'],
        ['HSPB1','TIMP3'],
        ['HSPE1','MOB4'],
        ['NF2','TG'],
        ['PARG','TIMM23B'],
        ['ST3GAL1','TPO'],
        ['ACTB','EIF4A1'],
        ['AL513122.1','PTRF'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BRS3','HTATSF1'],
        ['COQ8B','NUMBL'],
        ['HSPE1','MOB4'],
        ['SIDT2','TAGLN'],
        ['ANXA1','COL6A1'],
        ['ATP6V0A1','YBX3'],
        ['CD44','CTSS'],
        ['CTSB','PNPLA6'],
        ['DES','FLNA'],
        ['EEF1A1','PLD3'],
        ['FBLN5','MYH11'],
        ['IGFBP5','VCL'],
        ['IGKC','TMED5'],
        ['KLF6','METAP1'],
        ['RNF216','XKR8'],
        ['SARS','TLR8-AS1'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BRS3','HTATSF1'],
        ['COQ8B','NUMBL'],
        ['HSPE1','MOB4'],
        ['SIDT2','TAGLN'],
        ['ADH1B','PTRF'],
        ['ARHGAP27P1-BPTFP1-KPNA2P3','BPTF'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BPTF','KPNA2'],
        ['BRS3','HTATSF1'],
        ['COQ8B','NUMBL'],
        ['HSPE1','MOB4'],
        ['IGLV5-52','LL22NC03-80A10.6'],
        ['SIDT2','TAGLN'],
        ['AC004967.7','OR7E38P'],
        ['AC096574.4','TARDBP'],
        ['AC131056.5','CH17-431G21.1'],
        ['AC145124.2','FAM85B'],
        ['ACTB','HEBP2'],
        ['ACTB','TPO'],
        ['ACTG1','PLPP3'],
        ['ADIPOQ-AS1','GNPTG'],
        ['AGPAT3','SOD2'],
        ['ALK','LYZ'],
        ['ANAPC1','RMND5A'],
        ['ANAPC16','CNOT10'],
        ['ANAPC16','CRLF2'],
        ['ANP32B','CTSS'],
        ['ANXA1','TG'],
        ['APOC2','ISOC2'],
        ['APOL1','LINC01138'],
        ['ARHGEF26-AS1','TRA@'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['AZGP1','GJC3'],
        ['B2M','CDK6'],
        ['BCAP29','DUS4L'],
        ['BCL9','SH3PXD2A'],
        ['BP-21264C1.2','ZNF717'],
        ['BRS3','HTATSF1'],
        ['C12ORF45','PRKCSH'],
        ['C19MC','RPLP1'],
        ['C5','LRG1'],
        ['C7','RPS3'],
        ['C7','TRA@'],
        ['CANX','VEGFA'],
        ['CATSPER2','PPIP5K1'],
        ['CBFA2T3','CTSS'],
        ['CCDC144NL','NBEA'],
        ['CCDC6','MRLN'],
        ['CD24','SMIM14'],
        ['CD63','EEF1A1'],
        ['CD93','SYNPO'],
        ['CDH8','SYNPO'],
        ['CFLAR','METTL7A'],
        ['CLIP1','VHL'],
        ['COPA','FGFR2'],
        ['COQ8B','NUMBL'],
        ['CRLF2','IDS'],
        ['CRLF2','METTL7A'],
        ['CTSB','DNAJC7'],
        ['CTSB','EML4'],
        ['CTSB','FGFR2'],
        ['CTSB','PAX5'],
        ['CWF19L1','RFWD3'],
        ['CXCL5','GLYR1'],
        ['DTX2P1-UPK3BP1-PMS2P11','PMS2'],
        ['EEF1A1','EIF4B'],
        ['EGFR','IGSF6'],
        ['EMC10','MCAM'],
        ['FAM157B','FBXO25'],
        ['FAM98C','MRPS21'],
        ['FBXL18','PTGIS'],
        ['FGB','SERPINA3'],
        ['FGD5-AS1','FRRS1L'],
        ['GON4L','TG'],
        ['H6PD','PMPCA'],
        ['HCG4B','XXBAC-BPG248L24.12'],
        ['HNRNPU','HSP90AA1'],
        ['HP','SDHA'],
        ['HSP90AB1','SLC29A1'],
        ['HSP90B1','PTMS'],
        ['HSPA1B','TPT1'],
        ['HSPE1','MOB4'],
        ['HSPG2','PCBP2'],
        ['HSPG2','SCD'],
        ['IGFBP5','IGH@'],
        ['IGHM','STAT6'],
        ['IGK@','SOD2'],
        ['IL31RA','PRKCSH'],
        ['IL3RA','PTRF'],
        ['ITGB1','MYH11'],
        ['LIMS1','LYZ'],
        ['LINC00266-1','LINC01881'],
        ['LINC02193','RPL23AP82'],
        ['MECOM','RPL22'],
        ['MYBPC1','RYR1'],
        ['MYH11','YWHAZ'],
        ['NAIP','OCLN'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['PAFAH1B1','TG'],
        ['PDIA3','SPP1'],
        ['PDIA3','TG'],
        ['PECAM1','PTRF'],
        ['PIK3R1','TPM1'],
        ['PIPOX','SOD2'],
        ['PTRF','TMOD3'],
        ['RBFOX2','RPL41'],
        ['RPL13A','TG'],
        ['SCD','SORBS1'],
        ['SERF1A','SMN1'],
        ['SIDT2','TAGLN'],
        ['SOD2','TRB@'],
        ['TG','UBE2Z'],
        ['WASH2P','WASH5P'],
        ['AC004520.1','HNRNPA2B1'],
        ['AC004980.4','AC211486.3'],
        ['AC005332.7','AMZ2'],
        ['AC005336.1','IPMK'],
        ['AC006064.4','EEF1A1'],
        ['AC008083.3','PCED1B'],
        ['AC009065.3','PDXDC1'],
        ['AC010273.2','LINC02198'],
        ['AC010329.1','UBXN8'],
        ['AC011450.1','SLC6A16'],
        ['AC011603.3','MBP'],
        ['AC011611.5','DENR'],
        ['AC012485.1','TARDBP'],
        ['AC017083.4','C1D'],
        ['AC017083.4','CNRIP1'],
        ['AC018754.1','CCNH'],
        ['AC019131.2','ADH5'],
        ['AC020656.1','CCL5'],
        ['AC022400.3','PARG'],
        ['AC069208.1','SOX5'],
        ['AC074250.2','AFM'],
        ['AC079921.2','KLHL5'],
        ['AC080162.1','IAH1'],
        ['AC084082.1','TRIM55'],
        ['AC087477.2','NR2F2'],
        ['AC091057.2','GOLGA8N'],
        ['AC092691.1','LSAMP'],
        ['AC092718.3','NDUFS1'],
        ['AC093142.1','RPL34'],
        ['AC093484.4','UBC'],
        ['AC093536.1','EBPL'],
        ['AC100771.1','RPL37A'],
        ['AC106795.2','THOC3'],
        ['AC116535.2','CTR9'],
        ['AC117409.1','PSMC2'],
        ['AC117440.1','MTCH2'],
        ['AC127502.2','AC135983.2'],
        ['AC133065.2','CIITA'],
        ['AC133555.5','SMG1'],
        ['AC134879.2','IGK@'],
        ['AC135983.2','WHAMM'],
        ['AC136944.4','DUSP22'],
        ['AC138207.8','LRRC37B'],
        ['AC138207.8','SMURF2'],
        ['AC145124.1','FAM85B'],
        ['AC243967.3','LYPD4'],
        ['AC244517.11','TAF7'],
        ['AC245060.4','IGLV5-52'],
        ['ADAMTS4','TRA@'],
        ['AF305872.1','RPL21'],
        ['AIG1','AL136116.3'],
        ['AL035587.1','CNPY3'],
        ['AL049839.2','FGB'],
        ['AL049839.2','SERPINA1'],
        ['AL117335.1','AL121760.1'],
        ['AL121761.1','RIN2'],
        ['AL121936.1','NORAD'],
        ['AL133415.1','VIM-AS1'],
        ['AL136982.4','BMS1'],
        ['AL139174.1','MMADHC'],
        ['AL160313.1','HHIPL1'],
        ['AL161782.1','LINC01507'],
        ['AL357078.1','JAK1'],
        ['AL590762.3','ZCRB1'],
        ['ANAPC1','RMND5A'],
        ['ANO1','AP003555.3'],
        ['AP000790.1','YWHAZ'],
        ['AP001528.2','FZD4'],
        ['AP003084.1','EED'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARNT','MYH11'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BMPR1A','MIR100HG'],
        ['BRS3','HTATSF1'],
        ['C19MC','RPLP1'],
        ['CATSPER2','PPIP5K1'],
        ['COQ8B','NUMBL'],
        ['CTSB','PIEZO1'],
        ['CTSB','SLC35E3'],
        ['CTSS','MIR100HG'],
        ['FAM157B','FBXO25'],
        ['GSN','PECAM1'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['MECOM','RPL22'],
        ['NAIP','OCLN'],
        ['NME4','Z97634.1'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['RPS23','ZNF888'],
        ['SERF1A','SMN1'],
        ['SIDT2','TAGLN'],
        ['AC011603.2','MBP'],
        ['AFM','LINC02499'],
        ['ANAPC1','RMND5A'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BOLA2-SMG1P6','SMG1'],
        ['BRS3','HTATSF1'],
        ['C19MC','RPLP1'],
        ['CATSPER2','PPIP5K1'],
        ['COQ8B','NUMBL'],
        ['CTSB','HOOK2'],
        ['CTSB','RAP1GAP'],
        ['EEF1A1','EEF1A2'],
        ['FAM157B','FBXO25'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['MECOM','RPL22'],
        ['NAIP','OCLN'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['SERF1A','SMN1'],
        ['SIDT2','TAGLN'],
        ['AC008083.2','PCED1B'],
        ['AC017083.3','C1D'],
        ['AC017083.3','CNRIP1'],
        ['AC100810.1','AC100810.3'],
        ['ANAPC1','RMND5A'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BRS3','HTATSF1'],
        ['C19MC','RPLP1'],
        ['CATSPER2','PPIP5K1'],
        ['COQ8B','NUMBL'],
        ['FAM157B','FBXO25'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['MECOM','RPL22'],
        ['NAIP','OCLN'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['SERF1A','SMN1'],
        ['SIDT2','TAGLN'],
        ['AC005332.4','AMZ2'],
        ['AC006001.3','AC027644.4'],
        ['AC006453.2','AC245595.1'],
        ['AC068831.7','B2M'],
        ['AC091057.3','GOLGA8N'],
        ['AC093484.3','UBC'],
        ['AC104453.1','CD93'],
        ['AL354740.1','METTL7A'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BRS3','HTATSF1'],
        ['C19MC','RPLP1'],
        ['CATSPER2','PPIP5K1'],
        ['CD59','MYH11'],
        ['COQ8B','NUMBL'],
        ['DNAJC1','MIGA1'],
        ['ELOVL1','MED8'],
        ['FAM157B','FBXO25'],
        ['FTH1','FTL'],
        ['GKAP1','KIF27'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['MECOM','RPL22'],
        ['MUC20-OT1','SDHA'],
        ['NAIP','OCLN'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['SERF1A','SMN1'],
        ['SIDT2','TAGLN'],
        ['ZNF718','ZNF876P'],
        ['AC021087.5','AC132008.2'],
        ['AC021087.5','MUC20-OT1'],
        ['AC022400.6','PARG'],
        ['AC091951.3','AC138649.1'],
        ['AC139256.2','PKD1'],
        ['AC245060.7','IGLC2'],
        ['AC245060.7','IGLL5'],
        ['ADD1','MYH11'],
        ['AP000790.2','YWHAZ'],
        ['AP002956.1','EMC10'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BHMT2','CYP2B6'],
        ['BRS3','HTATSF1'],
        ['C19MC','RPLP1'],
        ['CATSPER2','PPIP5K1'],
        ['COQ8B','NUMBL'],
        ['CTBS','GNG5'],
        ['CTSB','LARP1'],
        ['ELOVL1','MED8'],
        ['FAM157B','FBXO25'],
        ['GKAP1','KIF27'],
        ['HECTD1','TG'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['MECOM','RPL22'],
        ['NAIP','OCLN'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['SERF1A','SMN1'],
        ['SIDT2','TAGLN'],
        ['AC005670.3','ARL17A'],
        ['AC008119.1','ISCU'],
        ['AC015689.1','RRM1'],
        ['AC022034.1','AC022034.3'],
        ['AC092944.1','AC104411.1'],
        ['AC110611.1','AC110611.2'],
        ['AC126182.3','NDUFB4'],
        ['AC145124.1','LINC02018'],
        ['AC245060.7','IGLC1'],
        ['ADIPOQ-AS1','AL139393.1'],
        ['AL096803.2','KCTD10'],
        ['AL109809.5','AL117335.1'],
        ['AL133405.1','STMND1'],
        ['AL353660.1','LINC00402'],
        ['AL354953.1','LAMC2'],
        ['AL391840.3','SH3BGRL2'],
        ['AL445524.2','SCAND2P'],
        ['AL953883.1','FAM27C'],
        ['CTSB','LINC02245'],
        ['DLG1','MYH7'],
        ['LBH','LINC01936'],
        ['LINCADL','LVRN'],
        ['ABO','TRB@'],
        ['AC006230.1','IL6ST'],
        ['AC006453.2','LINC02802'],
        ['AC009404.1','CRTAP'],
        ['AC010273.3','LINC02198'],
        ['AC016907.2','GSN'],
        ['AC018809.2','MORF4L1'],
        ['AC018845.3','PRXL2A'],
        ['AC025594.1','ATP6V1F'],
        ['AC083841.2','MPHOSPH8'],
        ['AC087639.1','C1R'],
        ['AC089984.3','CCDC9B'],
        ['AC104794.3','PRELP'],
        ['AC111188.1','ARPC1B'],
        ['ACSL3','HELLPAR'],
        ['ACSL3','WEE1'],
        ['ACTB','AP003469.4'],
        ['ACTB','CCAR2'],
        ['ADAMTS4','MLXIP'],
        ['ADIPOQ','C19MC'],
        ['AL031315.1','TSPAN15'],
        ['AL109806.1','SCARB2'],
        ['AL110115.2','NPEPPS'],
        ['AL136141.1','PIPOX'],
        ['AL158154.3','SOD2'],
        ['AL161457.2','FRG1DP'],
        ['AMOTL1','ZBTB18'],
        ['ANK2','HIST3H2BB'],
        ['ANKH','MORF4L1'],
        ['ANKRD42','RPL32'],
        ['ANO1','AP003555.2'],
        ['AP001528.1','FZD4'],
        ['AP005482.3','ZNF486'],
        ['APP','MYH11'],
        ['ARGLU1','IGFBP5'],
        ['ARHGEF12','PCBP2'],
        ['ATF5','C1S'],
        ['ATN1','IGFBP5'],
        ['ATP1B1','CRLF2'],
        ['ATP6AP2','SORL1'],
        ['ATXN2','PECAM1'],
        ['BAG1','LYZ'],
        ['BCL11A','ITM2A'],
        ['BMS1P4-AGAP5','PARG'],
        ['BNIP2','PRAMEF17'],
        ['BOLA2','SMG1'],
        ['C19MC','CTSB'],
        ['C19MC','ETS2'],
        ['C19MC','METTL7A'],
        ['C4ORF3','TRA@'],
        ['C7','STAR'],
        ['CAMK2A','CCNI'],
        ['CAMK2A','TRA@'],
        ['CASTOR2','SOD2'],
        ['CAVIN1','MCAM'],
        ['CAVIN1','RBMS3'],
        ['CDS2','PCBP2'],
        ['CEP104','IL6ST'],
        ['CHIC2','WWOX'],
        ['CLUL1','DCUN1D2'],
        ['CMIP','LITAF'],
        ['COA1','EOGT'],
        ['COPA','SLC16A1-AS1'],
        ['COPA','SUGP2'],
        ['CPS1','TF'],
        ['CTSB','TRAPPC12'],
        ['DEFB114','DEFB133'],
        ['DEPTOR','TG'],
        ['DES','PARM1'],
        ['DHX34','IGH@'],
        ['DPYSL2','KDELR1'],
        ['DPYSL2','SPOCK1'],
        ['DTD2','SS18'],
        ['DUSP22','MYH11'],
        ['EDA2R','ROCK2'],
        ['EGR1','MARCKS'],
        ['EIF4B','FNBP1'],
        ['EIF4B','TMCO1'],
        ['ELOC','MORF4L1'],
        ['ELP6','SIK2'],
        ['FAM214A','IL6ST'],
        ['FBXL19','ORAI2'],
        ['FBXO32','LIN7C'],
        ['FBXO32','RAB21'],
        ['FBXO42','MORF4L1'],
        ['GALNT15','WWOX'],
        ['GCOM1','MARCKS'],
        ['GFOD2','RPL7L1'],
        ['GFRA1','SAMD14'],
        ['GNS','IGH@'],
        ['GPBP1L1','SEMA4C'],
        ['GVQW3','VCL'],
        ['HEXIM1','SNRNP40'],
        ['IBA57','SORBS1'],
        ['IL6ST','VPS13C'],
        ['IRS2','KIDINS220'],
        ['ITGA3','SERINC5'],
        ['JADE1','LINC00639'],
        ['JAK1','PECAM1'],
        ['JAK3','METTL7A'],
        ['KAT6A','RAB7A'],
        ['KIRREL3-AS2','MORF4L1'],
        ['KLHL36','LINC02680'],
        ['KLHL42','ZSWIM8'],
        ['LIFR-AS1','SPINK1'],
        ['LINC01358','TMBIM6'],
        ['LINC01740','NENF'],
        ['LINC01765','NFIX'],
        ['LINC02245','PITPNM3'],
        ['LINC02749','RRM1'],
        ['LITAF','TMBIM6'],
        ['LPCAT2','MMP2'],
        ['LPP','MALAT1'],
        ['MAFG','NUCKS1'],
        ['MARCKSL1','PTMS'],
        ['MIR-371-CLUSTER','RPRD1B'],
        ['MORF4L1','SEMA6A'],
        ['MORF4L1','TRIM38'],
        ['MPEG1','ZDHHC20'],
        ['MYBPC1','PRKAR2A'],
        ['MYH11','PCDH15'],
        ['MYH11','RET'],
        ['NCBP3','PTGES'],
        ['NDFIP1','SYNPO2'],
        ['NDUFAB1','PALB2'],
        ['NIBAN1','PLEKHA1'],
        ['OGG1','TMBIM6'],
        ['ONECUT1','RAB7A'],
        ['ORM2','SERPINA1'],
        ['P2RX4','SORBS1'],
        ['PCNP','ZNF26'],
        ['PHF6','PPP1CB'],
        ['PIP4K2A','TRG@'],
        ['PTPRS','SOX4'],
        ['RAB7A','SAMD14'],
        ['RDX','SS18'],
        ['RNF169','TCIM'],
        ['SLC5A12','SOX2-OT'],
        ['SOD2','SOGA1'],
        ['AC089984.2','CCDC9B'],
        ['AC091390.3','PMS2'],
        ['AC093307.1','KIAA1217'],
        ['AC114402.1','CHMP5'],
        ['AC140479.3','MALL'],
        ['ACTB','PECAM1'],
        ['ACTB','SDHA'],
        ['ACTB','STAT6'],
        ['ACTB','ZNNT1'],
        ['ADGRG1','MYH11'],
        ['ANAPC1','RMND5A'],
        ['AP3B1','TG'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ATP1A1-AS1','HSP90B1'],
        ['ATRX','CIRBP'],
        ['AZGP1','GJC3'],
        ['B2M','ERBB3'],
        ['BLNK','NPM1'],
        ['BRS3','HTATSF1'],
        ['C19MC','RPLP1'],
        ['C7','EEF1A1'],
        ['CACNG2','PDE4D'],
        ['CATSPER2','PPIP5K1'],
        ['CENPW','TRMT11'],
        ['COPDA1','IGHG2'],
        ['COQ8B','NUMBL'],
        ['CSNK1G1','TG'],
        ['CTBS','GNG5'],
        ['DDX23','TG'],
        ['DDX24','EEF1A1'],
        ['DEPP1','FASN'],
        ['DNAJA1','TG'],
        ['DYSF','TG'],
        ['EEF1A1','FASN'],
        ['ELOVL1','MED8'],
        ['EML1','TFAM'],
        ['EML2','TG'],
        ['ETV6','SPRY4'],
        ['FAM157B','FBXO25'],
        ['FGFR2','MKNK2'],
        ['GCLC','TG'],
        ['GKAP1','KIF27'],
        ['GLG1','HNRNPA2B1'],
        ['GRB2','NME6'],
        ['HBM','SPOP'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['IGHM','UNC13D'],
        ['LARS1','TAT'],
        ['LHFPL5','MYH11'],
        ['LINC02379','RNF212B'],
        ['MAPK6','PECAM1'],
        ['MECOM','RPL22'],
        ['MGRN1','PPP1R1A'],
        ['MICOS10','TG'],
        ['MIF','SLC2A11'],
        ['MRPS31P5','THSD1'],
        ['MYH11','SERF2'],
        ['MYH11','TTN'],
        ['NAIP','OCLN'],
        ['NEK9','RAB7A'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['OBSCN','TG'],
        ['PDXDC1','SCD'],
        ['PKD1','TG'],
        ['PMS2','POLR2J3'],
        ['PPP2R5A','ZNF680'],
        ['RANBP3','TG'],
        ['RFC2','TG'],
        ['SCP2','SERPINA1'],
        ['SENP3','TG'],
        ['SERF1A','SMN1'],
        ['SIDT2','TAGLN'],
        ['STAT6','TMSB4X'],
        ['TAPBP','TG'],
        ['TMEM8B','TXN2']
        
        
        
]



    data = fusions.get(options.organism.lower(),[])
    if data:

        #file_symbols = os.path.join(options.output_directory,'genes_symbols.txt')
        file_symbols = os.path.join(options.output_directory,'synonyms.txt')
        loci = symbols.generate_loci(file_symbols)

        genes = symbols.read_genes_symbols(file_symbols)

        d = []
        for (g1,g2) in data:
            if g1.upper() != g2.upper():
                ens1 = symbols.ensembl(g1.upper(),genes,loci)
                ens2 = symbols.ensembl(g2.upper(),genes,loci)
                if ens1 and ens2:
                    for e1 in ens1:
                        for e2 in ens2:
                            if e1 != e2:
                                d.append([e1,e2])

        data = ['\t'.join(sorted(line)) + '\n' for line in d]
        data = list(set(data))

        print "%d known fusion genes found in manually currated database" % (len(data),)

        if not options.skip_filter_overlap:
            d1 = []
            overlappings = ['ensembl_fully_overlapping_genes.txt',
                            'ensembl_same_strand_overlapping_genes.txt',
                            'refseq_fully_overlapping_genes.txt',
                            'refseq_same_strand_overlapping_genes.txt',
#                            'ucsc_fully_overlapping_genes.txt',
#                            'ucsc_same_strand_overlapping_genes.txt',
                            'pairs_pseudogenes.txt',
                            'paralogs.txt']
            for ov in overlappings:
                p = os.path.join(options.output_directory,ov)
                print "Parsing file:",p
                if os.path.isfile(p):
                    d2 = sorted(set([tuple(sorted(line.rstrip('\r\n').split('\t'))) for line in file(p,'r').readlines() if line.rstrip('\r\n')]))
                d1.extend(d2)
            d = set()
            for line in d1:
                (a,b) = (line[0],line[1])
                if a > b:
                    (a,b) = (b,a)
                d.add("%s\t%s\n" % (a,b))
            skipped = [line for line in data if line in d]
            data = [line for line in data if not line in d]
            file(os.path.join(options.output_directory,'bodymap2_known_but_overlapping.txt'),'w').writelines(sorted(skipped))

            print "%d known fusion genes left after removing the overlappings" % (len(data),)

    file(os.path.join(options.output_directory,'bodymap2.txt'),'w').writelines(sorted(data))
    #
