#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of candidate fusion genes. This list is hard coded
in here and it is manually curated from:


Vellichirammal et al., Pan-Cancer Analysis Reveals the Diverse Landscape 
of Novel Sense and Antisense Fusion Transcripts,  Mol Ther Nucleic Acids, Vol. 19, 
pp. 1379-1398, March 06, 2020, https://doi.org/10.1016/j.omtn.2020.01.023




Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of pre-candidate fusion genes from CCLE."""
    version = "%prog 0.12 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of allowed candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of allowed candidate fusion genes is generated. Default is '%default'.""")

    parser.add_option("--skip-filter-overlap",
                      action="store_true",
                      dest="skip_filter_overlap",
                      default = False,
                      help="""If set then it filters out the known fusion genes where the (i) genes are fully overlapping, or (ii) the genes are partially overlapping and are on the same strand. Default is '%default'.""")

    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    print "Generating the list of CCLE fusion genes..."
    fusions = dict()

    # manual curation from papers

    fusions['rattus_norvegicus'] = []

    fusions['mus_musculus'] = []

    fusions['canis_lupus_familiaris'] = []

    fusions['homo_sapiens'] = [
        ['HEPHL1','PANX1'],
        ['METTL10','FAM53B'],
        ['ZNF547','TRAPPC2'],
        ['PLEKHO2','ANKDD1A'],
        ['RPL11','TCEB3'],
        ['TIRAP','DCPS'],
        ['ZNF664','SURF4'],
        ['RBFOX2','RPL41'],
        ['TSTD1','F11R'],
        ['ASTN2','ADIPOR2'],
        ['DHRS1','RABGGTA'],
        ['SDHAF2','PPP1R32'],
        ['ARF3','FKBP11'],
        ['CNPY2','CS'],
        ['CTSC','RAB38'],
        ['FEN1','FADS2'],
        ['RAB3IP','SCFD1'],
        ['TRIM58','OR2W3'],
        ['ZNF772','VN1R1'],
        ['ASTN2','SUPT20H'],
        ['CTNNBIP1','CLSTN1'],
        ['FXN','TJP2'],
        ['POLA2','CDC42EP2'],
        ['SLC35A3','MFSD14A'],
        ['TOMM5','FBXO10'],
        ['ZNF655','GS1-259H13.2'],
        ['KMT2C','TPTE'],
        ['LMAN2','MXD3'],
        ['MECOM','RPL22'],
        ['RAD18','OXTR'],
        ['SFT2D2','TBX19'],
        ['SPINT2','C19ORF33'],
        ['UBE2D2','CXXC5'],
        ['UCHL3','LMO7'],
        ['ADORA2B','TTC19'],
        ['APOLD1','DDX47'],
        ['CLTC','VMP1'],
        ['RAB3IP','CMYA5'],
        ['RAB3IP','ME1'],
        ['SURF4','SMURF2'],
        ['BDKRB2','BDKRB1'],
        ['C2CD3','GOSR1'],
        ['C9ORF114','SMURF2'],
        ['C9ORF114','SUB1'],
        ['KRBOX1','FAM198A'],
        ['LARP1','CNOT8'],
        ['MAG','CD22'],
        ['RAB3IP','ACVR1C'],
        ['SURF4','ZNF664'],
        ['CORO1C','SELPLG'],
        ['HOXB9','HOXB7'],
        ['HOXC10','HOXC4'],
        ['HOXD4','HOXD3'],
        ['NF1','AK4'],
        ['PACSIN2','ARFGAP3'],
        ['PEX26','TUBA8'],
        ['RAB3IP','COL28A1'],
        ['TIMM10B','DNHD1'],
        ['CHMP4A','TM9SF1'],
        ['HIST1H2BO','OR2B6'],
        ['LHX6','NDUFA8'],
        ['MTMR8','ASB12'],
        ['NXN','GLOD4'],
        ['RAB3IP','SSBP2'],
        ['TOPORS','DDX58'],
        ['TRIM58','OR2T8'],
        ['ALG1L','USP53'],
        ['CHRM3','ACVR1C'],
        ['CLCF1','POLD4'],
        ['COMMD1','B3GNT2'],
        ['EIF3K','ACTN4'],
        ['EWSR1','TAF15'],
        ['GNG10','UGCG'],
        ['LRRFIP1','RBM44'],
        ['RAB3IP','C14ORF37'],
        ['RAB3IP','IL20RB'],
        ['RAB3IP','MYLK'],
        ['RAB3IP','SVEP1'],
        ['SRP9','EPHX1'],
        ['UMAD1','GLCCI1'],
        ['B4GALT1','SMU1'],
        ['C17ORF100','ALOX15'],
        ['C19ORF60','TMEM59L'],
        ['CHRM3','COL28A1'],
        ['FGF5','C4ORF22'],
        ['HILPDA','EFCAB3'],
        ['HOXB6','HOXB3'],
        ['PDCD7','UBAP1L'],
        ['RAB3IP','ZNF638'],
        ['URB1','MIS18A'],
        ['ZNF343','SNRPB'],
        ['ACKR2','KRBOX1'],
        ['BRK1','VHL'],
        ['C19ORF47','AKT2'],
        ['C2CD3','BBS5'],
        ['CHCHD10','VPREB3'],
        ['CHRM3','CMYA5'],
        ['CHRM3','LRRC69'],
        ['CHRM3','SVEP1'],
        ['CHRM3','TTL'],
        ['CHST9','AQP4'],
        ['EIF4G3','HP1BP3'],
        ['FXYD2','DSCAML1'],
        ['GALNT8','KCNA6'],
        ['LARS2','LIMD1'],
        ['MANBAL','SRC'],
        ['MED8','ELOVL1'],
        ['NAA40','COX8A'],
        ['OR51B5','HBG2'],
        ['PARG','AGAP4'],
        ['PES1','GAL3ST1'],
        ['PLA2G10','PKD1'],
        ['RAB3IP','TDRD5'],
        ['RNASET2','RPS6KA2'],
        ['SLC25A12','GOSR1'],
        ['SLC25A43','SLC25A5'],
        ['SUMO2','HN1'],
        ['WLS','DIRAS3'],
        ['ZBTB1','PPP1R36'],
        ['ZCCHC14','RERE'],
        ['ZNF605','CHFR'],
        ['AIG1','PARL'],
        ['AK4','NF1'],
        ['ARHGEF5','CTAGE8'],
        ['ASTN2','KIF4A'],
        ['BRD9','ZDHHC11'],
        ['C1ORF186','POU5F1'],
        ['CCDC144A','USP32'],
        ['CHRM3','GPHN'],
        ['CHRM3','SLC5A12'],
        ['CHRM3','SPATA6L'],
        ['CHRM3','SSBP2'],
        ['ENTPD4','LOXL2'],
        ['FAM111B','FAM111A'],
        ['FAM234B','EMP1'],
        ['FXYD6-FXYD2','DSCAML1'],
        ['GOSR1','SMG1'],
        ['IL34','VRK2'],
        ['KIAA1324L','ADGRL3'],
        ['NFATC3','PLA2G15'],
        ['NSF','LRRC37A2'],
        ['NSL1','BATF3'],
        ['OPHN1','MFSD1'],
        ['OTUD6B','LRRC69'],
        ['PKD1','NPIPA8'],
        ['PRKAR1A','SLC15A2'],
        ['RAB3IP','CPNE8'],
        ['RAB3IP','GPHN'],
        ['RAB3IP','GPR158'],
        ['RASSF8','SSPN'],
        ['SAV1','GYPE'],
        ['SLC35F2','SLN'],
        ['SMG1','NPIPB3'],
        ['SMURF2','C9ORF114'],
        ['SSBP2','C2ORF91'],
        ['SUPT20H','ASTN2'],
        ['TC2N','CATSPERB'],
        ['TTLL5','IFT43'],
        ['TVP23B','CDRT4'],
        ['UBA2','WTIP'],
        ['UPK3BL','RASA4B'],
        ['WRB','SH3BGR'],
        ['XK','CYBB'],
        ['YARS2','NAP1L1'],
        ['ZNF662','KRBOX1'],
        ['ACTA2','ABCA9'],
        ['AGO2','TRAPPC9'],
        ['ALG1L','AGMO'],
        ['ALG1L','PPP1R1C'],
        ['ANKH','MRPL15'],
        ['ARNTL2','SMCO2'],
        ['BICD2','IPPK'],
        ['BPTF','KPNA2'],
        ['C21ORF59','TCP10L'],
        ['CCDC144NL','NCOR1'],
        ['CCL15','CCL14'],
        ['CDC42EP4','CPSF4L'],
        ['CDYL','C3ORF67'],
        ['CDYL','GNGT1'],
        ['CDYL','SLC44A5'],
        ['CDYL','SNX30'],
        ['CHRM3','ATP8B1'],
        ['CHRM3','C3ORF67'],
        ['CHRM3','CNTLN'],
        ['CHRM3','IGSF11'],
        ['CHRM3','ME1'],
        ['CHRM3','SLC25A43'],
        ['CHRM3','UXS1'],
        ['CNRIP1','PPP3R1'],
        ['COL8A1','CMSS1'],
        ['ELAC1','SMAD4'],
        ['F8','CLIC2'],
        ['FAM200B','BST1'],
        ['FBXL13','CCDC146'],
        ['FUT4','PIWIL4'],
        ['GKAP1','KIF27'],
        ['GLYCTK','DNAH1'],
        ['HACL1','COLQ'],
        ['HDAC8','CITED1'],
        ['HSPE1','MOB4'],
        ['IL34','SLCO1B3'],
        ['KIAA0753','PITPNM3'],
        ['KIAA1324L','IGSF11'],
        ['KIAA1324L','SLC15A2'],
        ['KLHL23','SSB'],
        ['LDLRAD3','PRR5L'],
        ['MAPK7','RNF112'],
        ['MB21D2','FGF12'],
        ['MED20','USP49'],
        ['NMNAT3','DTWD1'],
        ['OR51B5','HBE1'],
        ['ORC4','SRGAP1'],
        ['PDPK1','KCTD5'],
        ['PRKAR1A','ADGRL3'],
        ['PROSC','CHMP1A'],
        ['PRPF3','RPRD2'],
        ['PTPDC1','CHRM3'],
        ['RAB3IP','ADGRL3'],
        ['RAB3IP','HYDIN'],
        ['RAB3IP','SNX30'],
        ['RAD51D','RFFL'],
        ['S1PR2','DNMT1'],
        ['SLC25A12','BBS5'],
        ['SLC35E4','OSBP2'],
        ['SLCO1A2','NUP62CL'],
        ['SLCO1A2','SLC44A5'],
        ['SLCO1A2','WDR72'],
        ['SLFN11','SLFN12'],
        ['SYS1','DBNDD2'],
        ['TMX2','CTNND1'],
        ['TRIM16','FBXW10'],
        ['TSLP','CHRM3'],
        ['TTLL4','CYP27A1'],
        ['TVP23C','CDRT4'],
        ['UPK3B','POM121C'],
        ['WDR77','OVGP1'],
        ['WNK1','ERC1'],
        ['WNT5B','ADIPOR2'],
        ['WWOX','VAT1L'],
        ['XRCC1','ETHE1'],
        ['ACCS','EXT2'],
        ['ACSL4','KCNE5'],
        ['ALDOA','RPS28'],
        ['ALG1L','FGGY'],
        ['ALG1L','LOC100996634'],
        ['ALG1L','MCCC2'],
        ['AP3S1','LVRN'],
        ['ARSG','LRRC37A3'],
        ['ASTN2','AP3B1'],
        ['ASTN2','CNOT2'],
        ['ASTN2','TBC1D32'],
        ['BBS5','HEATR5A'],
        ['BZW2','TSPAN13'],
        ['C11ORF31','CTNND1'],
        ['C1ORF186','PTN'],
        ['C5ORF46','SPINK1'],
        ['C6ORF106','SPDEF'],
        ['C7ORF50','MAD1L1'],
        ['CACNG4','CACNG1'],
        ['CCDC122','ENOX1'],
        ['CD44','PDHX'],
        ['CDK13','SUGCT'],
        ['CDK20','ZNF598'],
        ['CDYL','REG1A'],
        ['CDYL','TDRD5'],
        ['CEP170','RAD51B'],
        ['CHRM3','BCAS3'],
        ['CHRM3','BIRC2'],
        ['CHRM3','CD96'],
        ['CHRM3','ENPEP'],
        ['CHRM3','EYS'],
        ['CHRM3','FEZ1'],
        ['CHRM3','FGGY'],
        ['CHRM3','FOCAD'],
        ['CHRM3','HTR2C'],
        ['CHRM3','IL20RB'],
        ['CHRM3','IMMP2L'],
        ['CHRM3','KIAA1324L'],
        ['CHRM3','LOC100996634'],
        ['CHRM3','NELL2'],
        ['CHRM3','NUP62CL'],
        ['CHRM3','POU6F2'],
        ['CHRM3','SNX30'],
        ['CHRM3','SUPT3H'],
        ['CHRM3','USP53'],
        ['CHRM3','WDR72'],
        ['CLN8','ARHGEF10'],
        ['CPED1','WNT16'],
        ['CTSC','ADIPOR2'],
        ['CUL5','ACAT1'],
        ['DOCK5','CDCA2'],
        ['EFCAB13','BBS5'],
        ['EIF1','RPS28'],
        ['ENTPD7','CUTC'],
        ['FKBP14','SCRN1'],
        ['FKBP1A','SDCBP2'],
        ['GATAD2B','NUP210L'],
        ['GOSR1','AMDHD1'],
        ['GOSR1','GTF2IRD2'],
        ['GOSR1','WDR66'],
        ['IGSF11','FANK1'],
        ['INO80C','TMEM178B'],
        ['INTS1','MICALL2'],
        ['KIAA1324L','HTR2C'],
        ['KIAA1324L','LOC100996634'],
        ['LSM14A','KIAA0355'],
        ['MAP2K5','SKOR1'],
        ['MARC1','HLX'],
        ['MRGPRX3','F2RL1'],
        ['MYL6','AES'],
        ['MYO6','SENP6'],
        ['NANP','NINL'],
        ['NCL','RPL7A'],
        ['NPTXR','DNAL4'],
        ['NSD1','ZNF346'],
        ['NSF','LRRC37A'],
        ['NSUN4','FAAH'],
        ['NSUN7','ARL4A'],
        ['NTF4','CGB8'],
        ['OPRD1','EPB41'],
        ['PAPOLA','AK7'],
        ['PARD6B','BCAS4'],
        ['PDE3A','SLCO1C1'],
        ['PDE4DIP','EIF1AX'],
        ['PEAK1','TSPAN3'],
        ['PFKFB4','SHISA5'],
        ['POLR2J2','RASA4B'],
        ['POLR2J3','RASA4B'],
        ['PPIP5K1','CATSPER2'],
        ['PPP1R12A','PAWR'],
        ['PRKAR1A','CBWD5'],
        ['PRKAR1A','IGSF11'],
        ['PRKAR1A','SPATA6L'],
        ['PTGFRN','CD101'],
        ['RAB3IP','FSTL4'],
        ['RAB3IP','NMU'],
        ['RAB3IP','RAG1'],
        ['RAB3IP','SYT14'],
        ['RAB3IP','TTL'],
        ['RAB9A','OFD1'],
        ['RBAK','PMS2'],
        ['RBM14','C11ORF80'],
        ['RBM14','RBM4B'],
        ['RBM14-RBM4','C11ORF80'],
        ['REL','TMEM212'],
        ['RHBDD2','POR'],
        ['RRN3','CDR2'],
        ['SFPQ','RPS28'],
        ['SHISA9','MAB21L3'],
        ['SLC16A12','LIPA'],
        ['SLC25A51','EXOSC3'],
        ['SLC45A3','ELK4'],
        ['SLC7A5','SMG1'],
        ['SLCO1A2','KIAA1324L'],
        ['SLCO1A2','POU6F2'],
        ['SLCO1A2','VEGFC'],
        ['SNX8','MAD1L1'],
        ['SPSB4','PXYLP1'],
        ['STEAP1','RAPGEF5'],
        ['STEAP1B','RAPGEF5'],
        ['SYTL4','TSPAN6'],
        ['TBCEL','TECTA'],
        ['TMEM233','PRKAB1'],
        ['TNFAIP8','HSD17B4'],
        ['TRPV3','SPATA22'],
        ['UBE2J1','GABRR2'],
        ['UBE2Q2L','CSPG4'],
        ['VEGFC','CHRM3'],
        ['WDR33','GOSR1'],
        ['WDR72','HTR2C'],
        ['YWHAE','CRK'],
        ['ZNF737','GALNT11'],
        ['AADAC','SUCNR1'],
        ['ABHD13','TNFSF13B'],
        ['ABHD17B','TMEM2'],
        ['ABR','HNRNPUL2'],
        ['ACSS1','APMAP'],
        ['ACTN4','CDH1'],
        ['ADGRL1','SUZ12'],
        ['AFF1','PTPN13'],
        ['AGAP4','FRMPD2'],
        ['AGFG1','COL4A3'],
        ['AGRN','MAGED1'],
        ['AGRN','PURB'],
        ['AKAP13','PDE8A'],
        ['ALDOA','CDK5RAP3'],
        ['ALDOA','RPS21'],
        ['ALG1L','ADGRV1'],
        ['ALG1L','APBA1'],
        ['ALG1L','ISPD'],
        ['ALG1L','WDR72'],
        ['ANKHD1','CYSTM1'],
        ['ANKHD1-EIF4EBP3','CYSTM1'],
        ['API5','CHRM3'],
        ['APLP2','EIF5B'],
        ['APOPT1','KLC1'],
        ['ARF4','HSPA1A'],
        ['ARF4','HSPA1B'],
        ['ARHGAP10','TMEM184C'],
        ['ARHGDIA','ZMIZ1'],
        ['ARPP19','CALM2'],
        ['ASB16','MAST2'],
        ['ASCC1','MICU1'],
        ['ASTN2','LINS1'],
        ['ASTN2','PHYHIPL'],
        ['ASTN2','XPR1'],
        ['ASTN2','XRRA1'],
        ['ATF2','CHN1'],
        ['ATP5B','IGFBP3'],
        ['ATP5E','LDHB'],
        ['ATP5I','AES'],
        ['ATP5L','KMT2A'],
        ['ATRX','MAGT1'],
        ['B2M','TMEM30A'],
        ['BACE2','FAM3B'],
        ['BBS5','C2CD3'],
        ['BBS5','HNRNPL'],
        ['BBS5','LDLRAD3'],
        ['BBS5','SLC25A12'],
        ['BBS5','TOM1L2'],
        ['BBS5','VPS37C'],
        ['BBS5','WDR66'],
        ['BCAT1','TSPAN11'],
        ['BCL2L1','OAZ1'],
        ['BGN','CBX3'],
        ['BRD2','PTPRF'],
        ['BSG','STK24'],
        ['BTF3','KRT18'],
        ['BTG3','CXADR'],
        ['C1ORF186','LRRC41'],
        ['C1ORF186','UGT2B17'],
        ['C20ORF203','NOL4L'],
        ['CAP1','PSMA2'],
        ['CAPN2','TPM2'],
        ['CARNMT1','C9ORF40'],
        ['CBARP','SBNO2'],
        ['CBLC','CEACAM16'],
        ['CCDC146','RASA4'],
        ['CCDC6','ANK3'],
        ['CCDC77','B4GALNT3'],
        ['CCT3','CD74'],
        ['CD5L','FCRL1'],
        ['CDYL','CRISPLD1'],
        ['CDYL','DNAJC3'],
        ['CDYL','LOC339862'],
        ['CDYL','NELL2'],
        ['CDYL','PLPPR1'],
        ['CDYL','RABGAP1L'],
        ['CENPA','METTL21A'],
        ['CFAP161','IL16'],
        ['CFL1','MARS'],
        ['CHAF1A','YEATS2'],
        ['CHFR','GOLGA3'],
        ['CHMP4A','ZNF562'],
        ['CHMP6','RPTOR'],
        ['CHRM3','ADAMTS20'],
        ['CHRM3','ARSB'],
        ['CHRM3','CD86'],
        ['CHRM3','CDH8'],
        ['CHRM3','CDK14'],
        ['CHRM3','CPNE8'],
        ['CHRM3','CRISPLD1'],
        ['CHRM3','FAM126A'],
        ['CHRM3','GLDN'],
        ['CHRM3','HFM1'],
        ['CHRM3','IRF1'],
        ['CHRM3','LOC339862'],
        ['CHRM3','MCCC2'],
        ['CHRM3','PKHD1'],
        ['CHRM3','PLPPR1'],
        ['CHRM3','PRB3'],
        ['CHRM3','PTPDC1'],
        ['CHRM3','RABGAP1L'],
        ['CHRM3','SCRN3'],
        ['CHRM3','SLC44A5'],
        ['CHRM3','SLCO1B1'],
        ['CHRM3','SMYD3'],
        ['CHRM3','SP140L'],
        ['CHRM3','SVOP'],
        ['CHRM3','TDRD5'],
        ['CHRM3','TRDN'],
        ['CHRM3','XPR1'],
        ['CHRM3','ZNF622'],
        ['CHRM3','ZNF638'],
        ['CLIC4','RPL28'],
        ['CLTC','RPS6KB1'],
        ['CLTC','TPI1'],
        ['CMTM4','TK2'],
        ['CMTM8','CMTM7'],
        ['CNOT2','ASTN2'],
        ['CNPY3','GNMT'],
        ['COL6A2','MAP4'],
        ['COLQ','PATE4'],
        ['COMMD7','C20ORF203'],
        ['CPSF6','CHMP1A'],
        ['CPSF6','JAG2'],
        ['CPSF6','KBTBD11'],
        ['CPSF6','LYZ'],
        ['CPSF6','TP53BP1'],
        ['CRAMP1','PRRC2B'],
        ['CTBS','GNG5'],
        ['CTNNA1','MATR3'],
        ['CTSD','GDI2'],
        ['DAND5','NFIX'],
        ['DCTN1','SLC4A5'],
        ['DDOST','ALDOA'],
        ['DEPDC1B','ELOVL7'],
        ['DHFR','FLNA'],
        ['DHX40','RNFT1'],
        ['DHX40','VMP1'],
        ['DNM2','YME1L1'],
        ['DNMT1','RPS4X'],
        ['DSE','FAM26F'],
        ['DTWD1','NMNAT3'],
        ['DYNC1H1','PLOD1'],
        ['DYNLRB1','RPL32'],
        ['DZIP1L','CLDN18'],
        ['EARS2','GGA2'],
        ['ECD','P4HA1'],
        ['EDNRB','TMEM200A'],
        ['EIF1','RPL5'],
        ['EIF1AX','TNPO2'],
        ['EIF2S2','FKBP8'],
        ['EIF4A1','GNB1'],
        ['EIF4G1','TXNRD1'],
        ['EIF4H','PYGL'],
        ['EML4','ALK'],
        ['EPHX1','DNAJC15'],
        ['ERBB2IP','NLN'],
        ['ERBB2IP','SREK1'],
        ['EVI5','GFI1'],
        ['EWSR1','HNRNPA3'],
        ['EZR','ARGLU1'],
        ['FAM136A','UBAP2L'],
        ['FAM168A','RAB6A'],
        ['FAM177B','SOD2'],
        ['FARSB','TRIM61'],
        ['FASN','PVR'],
        ['FASN','SUPT5H'],
        ['FBXL18','TNRC18'],
        ['FCHSD2','RAB6A'],
        ['FGF12','CHRM3'],
        ['FGFR2','COL14A1'],
        ['FHOD3','MOCOS'],
        ['FITM2','OSER1'],
        ['FLNB','RPL5'],
        ['FOCAD','REG1A'],
        ['FTH1','CALU'],
        ['FUT8','BBS5'],
        ['GAS6','SEC61A1'],
        ['GIMAP2','GIMAP1'],
        ['GIMAP2','GIMAP1-GIMAP5'],
        ['GLO1','KRT18'],
        ['GNB1','HLA-A'],
        ['GOLGB1','UBC'],
        ['GOLM1','CFL1'],
        ['GOSR1','FAM177A1'],
        ['GOSR1','HNRNPL'],
        ['GOSR1','TIMM23B'],
        ['GOSR1','VPS37C'],
        ['GOSR1','WDR33'],
        ['GPI','PLEC'],
        ['GPI','UBA2'],
        ['GPRC5A','SLC7A5'],
        ['GUSB','NAIP'],
        ['GXYLT2','PPP4R2'],
        ['HAUS4','PRMT5'],
        ['HDDC2','SLC6A20'],
        ['HDLBP','CALR'],
        ['HIC2','PI4KA'],
        ['HIF1A','SNAPC1'],
        ['HLA-A','IRAK1'],
        ['HMGB1','RPS21'],
        ['HNRNPA3','XPO6'],
        ['HNRNPAB','UBE2D3'],
        ['HNRNPK','EIF3E'],
        ['HNRNPK','MYOF'],
        ['HOXD11','HOXD8'],
        ['HOXD11','HOXD9'],
        ['HSD17B4','TNFAIP8'],
        ['HSP90B1','RPS15'],
        ['HSPA5','CDKN1A'],
        ['HSPA8','DBNL'],
        ['HSPA8','PKD1'],
        ['HSPA8','RPS23'],
        ['HSPB1','UBA1'],
        ['HSPD1','RPL7A'],
        ['HSPG2','RAB1B'],
        ['HYOU1','CD24'],
        ['ID2','ID3'],
        ['IGSF5','PCP4'],
        ['IL17RD','ARHGEF3'],
        ['IL34','ASIC2'],
        ['IL34','CALB1'],
        ['IL34','CAPN12'],
        ['IL34','CD177'],
        ['IL34','DSCAML1'],
        ['IL34','LRRTM4'],
        ['IL34','NTN4'],
        ['IL34','SEMA3E'],
        ['IL34','TRPS1'],
        ['IMMP2L','DOCK4'],
        ['INO80C','AXDND1'],
        ['INO80C','SNX9'],
        ['INTS4','GAB2'],
        ['IRGM','NDST4'],
        ['ISCA1','GOLM1'],
        ['ITGB4','PA2G4'],
        ['ITPR2','CCDC91'],
        ['JUNB','RPL14'],
        ['KANSL1','LRRC37A'],
        ['KARS','RPL3'],
        ['KDM5A','SLC6A13'],
        ['KIAA1324L','ATP8B1'],
        ['KIAA1324L','CBWD5'],
        ['KIAA1324L','SPATA6L'],
        ['KIF1B','UBE4B'],
        ['KIRREL','MAGED1'],
        ['KIRREL','RPL10A'],
        ['KLK8','KLK7'],
        ['KLRC4','KLRK1'],
        ['KRT18','RPL8'],
        ['KRT18','RPL9'],
        ['LAMA5','KRT17'],
        ['LAMA5','LGALS3BP'],
        ['LAMB1','CFL1'],
        ['LAMB1','TKT'],
        ['LAMC2','PTMA'],
        ['LDLR','BRK1'],
        ['LIPT1','MRPL30'],
        ['LMNB2','HSP90AB1'],
        ['LMO1','RIC3'],
        ['LOC100289561','PMS2'],
        ['LOC100289561','PRKRIP1'],
        ['LOXL2','TUBB'],
        ['LRRC74B','TUBA3D'],
        ['LRRC74B','TUBA3E'],
        ['LSM3','ANXA2'],
        ['MAEA','CTBP1'],
        ['MAML2','MTMR2'],
        ['MAPK8IP3','AHNAK'],
        ['MATR3','CTNNA1'],
        ['MDH2','EIF1'],
        ['MERTK','ZNF780A'],
        ['METTL22','ABAT'],
        ['METTL24','WASF1'],
        ['MICAL2','KRT5'],
        ['MND1','KIAA0922'],
        ['MOB3B','EQTN'],
        ['MRGPRX3','C1ORF186'],
        ['MRGPRX3','PLCB1'],
        ['MRGPRX3','SMOX'],
        ['MRGPRX3','VPS37C'],
        ['MRPL33','BRE'],
        ['MRPL33','SLC4A1AP'],
        ['MSL1','ATP5B'],
        ['MTFR1','BBS5'],
        ['MTURN','ZNRF2'],
        ['MYL12B','MYL6'],
        ['MYL12B','PABPC1'],
        ['MYLK3','RALA'],
        ['NADSYN1','S100A6'],
        ['NCOA3','EYA2'],
        ['NCOR2','HNRNPH1'],
        ['NF1','SMURF2'],
        ['NFAT5','CYB5B'],
        ['NFE2L1','RRBP1'],
        ['NGRN','PERP'],
        ['NIPBL','WDR70'],
        ['NLK','PYY'],
        ['NLN','ERBB2IP'],
        ['NOTCH2','HSPA8'],
        ['NPEPPS','BANF1'],
        ['NPLOC4','PDE6G'],
        ['NSRP1','ATP5F1'],
        ['NT5E','RPL19'],
        ['NUCKS1','HSPD1'],
        ['PA2G4','CALM2'],
        ['PABPC1','CTTN'],
        ['PABPC1','HDLBP'],
        ['PABPC1','ILF3'],
        ['PACSIN2','TTLL1'],
        ['PAMR1','SLC1A2'],
        ['PCBP1','MLEC'],
        ['PDCD6','SDHA'],
        ['PDIA3','RPS28'],
        ['PDXDC1','PLA2G10'],
        ['PFAS','RANGRF'],
        ['PFN1','BSG'],
        ['PFN1','TMSB10'],
        ['PGAM1','HDLBP'],
        ['PGAP1','C2ORF66'],
        ['PHLDA1','TAGLN2'],
        ['PHOSPHO2-KLHL23','SSB'],
        ['PHRF1','DRD4'],
        ['PIAS2','ST8SIA5'],
        ['PINX1','SOX7'],
        ['PKD1','STRN4'],
        ['PKD1','TUBA1A'],
        ['PKM','RPS28'],
        ['PKM','SRP9'],
        ['PLEC','HSPA8'],
        ['PLEKHA1','TACC2'],
        ['PLXND1','TMCC1'],
        ['POC1B','DUSP6'],
        ['POLR2J2','LOC100289561'],
        ['POLR2J3','LOC100289561'],
        ['PPARD','GOSR1'],
        ['PPFIBP1','SMCO2'],
        ['PPM1D','BCAS3'],
        ['PPP2CA','SKP1'],
        ['PRDX1','HIF1A'],
        ['PRKAR1A','AGMO'],
        ['PRKAR1A','CBWD1'],
        ['PRKAR1A','CBWD3'],
        ['PRKAR1A','HTR2C'],
        ['PRKAR1A','LOC100996634'],
        ['PRKAR1A','LRRC69'],
        ['PRKAR1A','MET'],
        ['PRRC2A','ANLN'],
        ['PRRC2B','SERPINH1'],
        ['PRSS23','PTMA'],
        ['PSAP','RAD23B'],
        ['PTP4A2','RPS15'],
        ['PTPN12','PHTF2'],
        ['PTPN14','RPL13'],
        ['PTPRK','THEMIS'],
        ['PVRIG','PILRB'],
        ['PXMP4','E2F1'],
        ['PXN','MSI1'],
        ['PXN','PLA2G1B'],
        ['QKI','PABPC1'],
        ['QKI','PACRG'],
        ['RAB30','PRCP'],
        ['RAB3IP','C3ORF67'],
        ['RAB3IP','COL24A1'],
        ['RAB3IP','FGGY'],
        ['RAB3IP','IMMP2L'],
        ['RAB3IP','SCRN3'],
        ['RAB3IP','ZNF622'],
        ['RALA','YAE1D1'],
        ['RAN','RPL3'],
        ['RBM39','SUMO2'],
        ['RCC1','ITGB3BP'],
        ['RHEB','FAM35A'],
        ['RHEB','HNRNPU'],
        ['RHOC','HSPD1'],
        ['RHOQ','LRR1'],
        ['RICTOR','FYB'],
        ['RNF115','PDZK1'],
        ['RNF216','RBAK'],
        ['RNF216','RBAK-RBAKDN'],
        ['RPL10','CFL1'],
        ['RPL10','EIF3E'],
        ['RPL13','SF3B1'],
        ['RPL18A','ARHGEF2'],
        ['RPL18A','KPNA2'],
        ['RPL22','MECOM'],
        ['RPL3','RHOT2'],
        ['RPL30','POLR2E'],
        ['RPL30','TPI1'],
        ['RPL35','RPL13'],
        ['RPL41','LRRC37B'],
        ['RPL6','SNU13'],
        ['RPL7','AP2B1'],
        ['RPL7','MAP2K1'],
        ['RPL7A','PPP1CA'],
        ['RPL7A','SAE1'],
        ['RPL8','PGD'],
        ['RPLP0','HMGN2'],
        ['RPLP1','CTNNB1'],
        ['RPN2','C5ORF15'],
        ['RPS10','TXN'],
        ['RPS10-NUDT3','PTMA'],
        ['RPS10-NUDT3','TXN'],
        ['RPS11','PGAM1'],
        ['RPS11','RAB2A'],
        ['RPS14','PSMB4'],
        ['RPS14','RPL23A'],
        ['RPS15A','MLLT6'],
        ['RPS18','BASP1'],
        ['RPS20','SEPT7'],
        ['RPS29','PIGF'],
        ['RPS3','RAB7A'],
        ['RPS6','RDX'],
        ['RPSA','SQLE'],
        ['RPSA','TUBB'],
        ['RRAGA','SAXO1'],
        ['RSF1','TENM4'],
        ['RTN4','RPS2'],
        ['S100Z','CRHBP'],
        ['SCO1','MYH3'],
        ['SCRN2','DDR1'],
        ['SDC3','USP22'],
        ['SDF4','SEMA3F'],
        ['SEC23A','HNRNPU'],
        ['SEC61A1','RPS12'],
        ['SEPT7','YBX1'],
        ['SETD2','CCDC12'],
        ['SETD3','CCDC85C'],
        ['SIL1','CTNNA1'],
        ['SLC1A5','PHLDA1'],
        ['SLC25A16','DNA2'],
        ['SLC43A1','SLC43A3'],
        ['SLC43A3','PRG2'],
        ['SLC7A5','PGAM1'],
        ['SLC7A5','RPL14'],
        ['SLC7A5','RPL39'],
        ['SLCO1A2','ENPEP'],
        ['SLCO1A2','GABRG2'],
        ['SLCO1A2','GNGT1'],
        ['SLCO1A2','LRRC69'],
        ['SLCO1A2','SCRN3'],
        ['SLCO1A2','SNCAIP'],
        ['SLCO1A2','TRDN'],
        ['SMARCA4','C19ORF38'],
        ['SMARCA4','LDLR'],
        ['SMC1A','RPS14'],
        ['SMIM7','MED26'],
        ['SND1','UBA52'],
        ['SNRNP70','HNRNPC'],
        ['SNRNP70','RPL12'],
        ['SNRPD2','HSPA9'],
        ['SNX29','PLA2G10'],
        ['SPECC1','CCDC144A'],
        ['SPIN1','NXNL2'],
        ['SQSTM1','TMSB10'],
        ['SRC','RPN2'],
        ['SRRM2','C1S'],
        ['ST6GALNAC4','ST6GALNAC6'],
        ['ST7','MET'],
        ['STK17B','H2AFZ'],
        ['STRA6','CCDC33'],
        ['STRN3','HECTD1'],
        ['SUB1','SSFA2'],
        ['SUGT1','NOL6'],
        ['SUMO3','KRT8'],
        ['SUZ12','CRLF3'],
        ['TAGLN2','CFAP45'],
        ['TANGO6','SNTB2'],
        ['TCEA1','YWHAE'],
        ['TDRD9','ASPG'],
        ['TFG','TOMM20'],
        ['TGFBI','RPS11'],
        ['THRAP3','AGO3'],
        ['THSD4','LRRC49'],
        ['TIGD1','ERCC6L2'],
        ['TMBIM6','EIF3C'],
        ['TMBIM6','EIF3CL'],
        ['TMC6','TK1'],
        ['TMED3','KIAA1024'],
        ['TMEM107','VAMP2'],
        ['TMEM126B','TMEM126A'],
        ['TMEM56','RWDD3'],
        ['TNFRSF12A','CHPF2'],
        ['TOMM40','RPL12'],
        ['TPI1','TNRC18'],
        ['TPT1','SNRPA1'],
        ['TPT1','TOP1'],
        ['TPX2','HM13'],
        ['TRAP1','ANXA1'],
        ['TRIM58','OR2T33'],
        ['TRIM61','FARSB'],
        ['TRIO','WDR6'],
        ['TRPA1','MSC'],
        ['TSKU','ACER3'],
        ['TUBA1B','PUM1'],
        ['TUBA1B','SRSF6'],
        ['TUBA1C','IRS2'],
        ['TUBGCP2','MICALL1'],
        ['U2AF2','RPS2'],
        ['UBA2','GPI'],
        ['UBB','PABPC1'],
        ['UBC','QSOX1'],
        ['UBE2L3','YBX1'],
        ['UHRF2','KDM4C'],
        ['WDR26','RPS28'],
        ['WDR33','SAP130'],
        ['WDR72','SLC15A2'],
        ['WDR91','C7ORF49'],
        ['WHSC1','C4ORF48'],
        ['XPOT','HNRNPA3'],
        ['XPR1','ASTN2'],
        ['YAP1','MAML2'],
        ['YWHAE','ATP5A1'],
        ['YWHAE','NBPF1'],
        ['YWHAG','ANKRD11'],
        ['ZBTB20','LSAMP'],
        ['ZBTB8OS','YARS'],
        ['ZNF664','C9ORF114'],
        ['ZNF71','SMIM17'],
        ['ZNF737','TRPM4'],
        ['ZNF749','BBS5'],
        ['AAMP','BAG6'],
        ['ABHD12','GPR107'],
        ['ADAP1','C7ORF50'],
        ['ADRBK1','ANXA2'],
        ['AHCY','NEDD8'],
        ['ALDOA','ITGA11'],
        ['ALG1L','C12ORF42'],
        ['ALG1L','SMYD3'],
        ['ANKHD1','PCDH1'],
        ['ANKHD1-EIF4EBP3','PCDH1'],
        ['ANP32A','MRPS18B'],
        ['ANXA3','HNRNPA1'],
        ['APBA2','NSD1'],
        ['APLP2','AARS'],
        ['AREL1','JDP2'],
        ['ARF1','IER2'],
        ['ARHGEF5','OR2A7'],
        ['ARRB1','KLHL35'],
        ['ASTN2','BACH2'],
        ['ASTN2','DISP1'],
        ['ASTN2','OXCT1'],
        ['ATAD2','CYP24A1'],
        ['ATAD5','TLK2'],
        ['ATP5A1','SKP1'],
        ['ATP5B','NUCKS1'],
        ['ATP5B','TRRAP'],
        ['ATP6V0D1','PPIB'],
        ['ATXN7L3','FAM171A2'],
        ['B2M','AARS'],
        ['BAHCC1','MAST2'],
        ['BBS5','IFT81'],
        ['BCAP29','SLC26A4'],
        ['BOP1','ARHGAP39'],
        ['BRD2','AAR2'],
        ['C1ORF186','UGT2B15'],
        ['C8ORF33','TOP2A'],
        ['CACHD1','ROR1'],
        ['CAGE1','SSR1'],
        ['CALU','TPI1'],
        ['CAPNS1','CDCA8'],
        ['CAPZA2','MET'],
        ['CAST','SMG7'],
        ['CAST','WSB1'],
        ['CBLC','NBPF14'],
        ['CCDC144A','PRPSAP2'],
        ['CCDC80','PGAM1'],
        ['CCDC80','TSPAN31'],
        ['CCDC85C','SETD3'],
        ['CCZ1B','ORC4'],
        ['CDCA5','SNRPD1'],
        ['CDH17','GGNBP2'],
        ['CDYL','GAP43'],
        ['CDYL','LAMA2'],
        ['CEP112','AXIN2'],
        ['CERS6','DDX5'],
        ['CFL1','FOSL1'],
        ['CHD2','TP53BP1'],
        ['CHRM3','C14ORF37'],
        ['CHRM3','CDK17'],
        ['CHRM3','EXOC4'],
        ['CHRM3','FAM183A'],
        ['CHRM3','FSTL4'],
        ['CHRM3','GAP43'],
        ['CHRM3','GNGT1'],
        ['CHRM3','INTU'],
        ['CHRM3','KYNU'],
        ['CHRM3','PGAP1'],
        ['CHRM3','ST6GALNAC5'],
        ['CKAP5','RPS2'],
        ['CLCN7','RPL18A'],
        ['CLEC4F','FIGLA'],
        ['CLPP','RPL8'],
        ['CLPTM1','GAA'],
        ['CLPTM1L','RANGAP1'],
        ['CNKSR3','IPCEF1'],
        ['COL3A1','VPS39'],
        ['COL4A3BP','PDCD7'],
        ['COL5A1','CTTN'],
        ['COL5A1','STMN3'],
        ['COL7A1','RPL38'],
        ['COL9A3','NME1'],
        ['COPA','TMBIM6'],
        ['COPG1','SETD3'],
        ['CPS1','CELSR1'],
        ['CPS1','UBR5'],
        ['CTNNB1','SF3B2'],
        ['CTTN','TGFBR2'],
        ['CUTC','ABCC2'],
        ['CYB5B','CHD4'],
        ['CYC1','LAMB1'],
        ['DDX56','CADM1'],
        ['DGKZ','SGK223'],
        ['DHFR','RPL17'],
        ['DHFR','RPL17-C18ORF32'],
        ['DSCR3','TTC3'],
        ['DUSP23','PABPC1'],
        ['EDF1','GRN'],
        ['EEF1G','CLSPN'],
        ['EEF1G','TRMT2B'],
        ['EFEMP1','CD59'],
        ['EIF1AX','PDE4DIP'],
        ['EIF3H','FKBP4'],
        ['EIF3H','LRP8'],
        ['EIF3H','POMP'],
        ['EIF3H','SEPT2'],
        ['EIF4G1','BLVRB'],
        ['EIF4G1','LARP1'],
        ['EIF4G1','TXN'],
        ['EIF4G2','DAZAP2'],
        ['ELP6','CLPP'],
        ['ENOSF1','TEF'],
        ['EPS8L2','FN1'],
        ['ESR1','CCDC170'],
        ['ESRP1','DPY19L4'],
        ['ETF1','RPS7'],
        ['EWSR1','ZNRF3'],
        ['EXOSC1','MARS'],
        ['EXT2','CFL1'],
        ['F3','CNOT1'],
        ['FAF1','AGBL4'],
        ['FAM120A','PTMS'],
        ['FAM126B','ORC2'],
        ['FAM3C','PI4KA'],
        ['FAM45A','GRK5'],
        ['FAR2','MKX'],
        ['FAU','LGALS1'],
        ['FBL','LGALS1'],
        ['FBXO16','ZNF395'],
        ['FBXO34','CDKN3'],
        ['FBXW2','SNRPG'],
        ['FCHSD2','FAM168A'],
        ['FDFT1','PLAU'],
        ['FDPS','PSMC5'],
        ['FNBP1','PLCG1'],
        ['FNIP2','KDM5C'],
        ['FOSL1','MTG1'],
        ['FOXM1','TAF15'],
        ['FTH1','TUBB3'],
        ['FTL','LRP4'],
        ['FUT6','DUS3L'],
        ['FZD1','NEMP1'],
        ['GARS','CALR'],
        ['GARS','CEP120'],
        ['GARS','NHP2'],
        ['GCN1','MSI1'],
        ['GCSH','SPTBN1'],
        ['GGH','NME1'],
        ['GHITM','LRRC59'],
        ['GLTSCR2','CLASP2'],
        ['GLTSCR2','TPM3'],
        ['GMFG','LRFN1'],
        ['GNAS','POLR2E'],
        ['GNB5','CSE1L'],
        ['GOSR1','C1ORF186'],
        ['GOSR1','DEFB124'],
        ['GRB2','PAFAH1B3'],
        ['GTF2E2','SUN1'],
        ['GTPBP4','GNB2'],
        ['GUK1','VIM'],
        ['H2AFV','SRP9'],
        ['H3F3A','SRP9'],
        ['HDAC7','TIMP1'],
        ['HDLBP','PKD1'],
        ['HGS','ARPC2'],
        ['HMGA1','EIF4G2'],
        ['HMGN2','HNRNPA3'],
        ['HMGN2','HTT'],
        ['HNRNPAB','FAM168B'],
        ['HNRNPDL','ALG1'],
        ['HNRNPH2','GANAB'],
        ['HSP90AA1','EPHA2'],
        ['HSP90AB1','IFI30'],
        ['HSP90AB1','PITPNB'],
        ['HSPA4L','GRINA'],
        ['HSPD1','RPL30'],
        ['ID3','ARPC3'],
        ['IL34','COX6C'],
        ['INTS3','WASL'],
        ['IRS1','MTSS1L'],
        ['ISCA1','RPL8'],
        ['ISOC2','VIM'],
        ['ITSN2','OTOF'],
        ['KDM4E','SRSF8'],
        ['KHSRP','COL5A1'],
        ['KIAA0368','SPINT2'],
        ['KIAA1324L','LRRC69'],
        ['KIAA1324L','MRPS31'],
        ['KIAA1324L','PTPDC1'],
        ['KIF2A','IPO11'],
        ['KLC1','APOPT1'],
        ['KLHDC2','SNTB1'],
        ['KMT2D','DDN'],
        ['KRT18','MPRIP'],
        ['KRT8','PRDX4'],
        ['LAMP2','RBM12B'],
        ['LAPTM4A','CYTH3'],
        ['LAPTM4B','SRPRA'],
        ['LDHB','HNRNPH1'],
        ['LEPROT','SNW1'],
        ['LGALS1','DNAJB11'],
        ['LIX1L','APP'],
        ['LMNA','EIF5'],
        ['LOC100289561','CUX1'],
        ['LRP3','CAMSAP1'],
        ['LRP8','THBS1'],
        ['LRRFIP1','NME1-NME2'],
        ['LSM2','RPS27'],
        ['LYPLA1','TCEA1'],
        ['MAP3K1','NDUFAF2'],
        ['MAPKAPK2','ADAM12'],
        ['MARCKSL1','THOP1'],
        ['MCM7','HSPD1'],
        ['MCM7','SSH3'],
        ['MED15','CYFIP1'],
        ['MEF2D','IQGAP3'],
        ['MFNG','ELFN2'],
        ['MLX','ERGIC1'],
        ['MMP14','DHDDS'],
        ['MMP7','LEMD3'],
        ['MRPL15','ANKH'],
        ['MTDH','PPP1R16A'],
        ['MTHFD1L','RPL32'],
        ['MTMR2','GPX2'],
        ['MTRNR2L1','MARCH6'],
        ['MTRNR2L6','MARCH6'],
        ['MUL1','RPL4'],
        ['MYH9','CD47'],
        ['MYO9B','FCHO1'],
        ['MYOF','RPS12'],
        ['NAIP','GTF2H2'],
        ['NAPRT','MGLL'],
        ['NBN','CCT5'],
        ['NCAPD3','TTLL12'],
        ['NDUFA13','PINK1'],
        ['NEXN','DNAJB4'],
        ['NF1','RAB11FIP4'],
        ['NFYC','KCNQ4'],
        ['NOTCH1','TIAL1'],
        ['NOTCH2NL','NBPF1'],
        ['NPC2','SYNDIG1L'],
        ['NPEPPS','COL18A1'],
        ['NPIPB3','HNRNPUL1'],
        ['NPIPB5','HNRNPUL1'],
        ['NPIPB5','SPARC'],
        ['NPM1','ALK'],
        ['NPM1','TOP2B'],
        ['NRROS','PIGX'],
        ['NUMA1','UPF3B'],
        ['NUP98','BEAN1'],
        ['ORC4','LOC400927-CSNK1E'],
        ['PABPC1','POGZ'],
        ['PAX3','FOXO1'],
        ['PBRM1','WDR82'],
        ['PCBP2','RPL8'],
        ['PDIA3','DHX37'],
        ['PFN1','MGAT1'],
        ['PHGDH','RPL13A'],
        ['PHLDA1','RPS5'],
        ['PKD1','RPL6'],
        ['PLD3','RPL29'],
        ['PLEKHA8','MTURN'],
        ['PLEKHG1','PDCL3'],
        ['PLXNB2','CARD19'],
        ['POP1','MATN2'],
        ['PPL','C16ORF89'],
        ['PRF1','CASK'],
        ['PRKAR1A','CBWD2'],
        ['PRKAR1A','MRPS31'],
        ['PRPF3','TUBB'],
        ['PRPF6','ACTN4'],
        ['PRRC2A','FMNL1'],
        ['PRSS23','CLTC'],
        ['PSMA3','HIP1'],
        ['PSMD6','TOR1AIP2'],
        ['PSPH','RPL7A'],
        ['PTK2','AGO2'],
        ['PTMA','RBM25'],
        ['PTMA','RPL17'],
        ['PTMA','RPL17-C18ORF32'],
        ['PTPN11','PPP2R1A'],
        ['PTPRF','SAP30BP'],
        ['PVRL1','HSP90AB1'],
        ['RAB3B','NRDC'],
        ['RAB3IP','EXOC4'],
        ['RAB3IP','IARS2'],
        ['RAB3IP','KAT2B'],
        ['RAB3IP','LRRC42'],
        ['RAB3IP','SCN8A'],
        ['RAB3IP','TYW1B'],
        ['RAPSN','PSMC3'],
        ['RARA','PKIA'],
        ['RARG','AAAS'],
        ['RBM39','SBDS'],
        ['RBM47','APBB2'],
        ['RBM8A','PI4KA'],
        ['RCC1','SERINC2'],
        ['RCN1','RPL37'],
        ['RFX4','RIC8B'],
        ['RLF','COL9A2'],
        ['RLF','HEYL'],
        ['RPL13A','WNK1'],
        ['RPL17','PLEC'],
        ['RPL22','LIF'],
        ['RPL35','SEC23A'],
        ['RPL36A-HNRNPH2','GANAB'],
        ['RPL37A','MCM3'],
        ['RPL5','HM13'],
        ['RPL5','MAPK1'],
        ['RPLP0','ATP5H'],
        ['RPLP0','FSCN1'],
        ['RPLP0','SREBF2'],
        ['RPLP1','CS'],
        ['RPLP1','STIP1'],
        ['RPN1','RPS15A'],
        ['RPS16','MTCH1'],
        ['RPS18','CDK4'],
        ['RPS19','KMT5A'],
        ['RPS2','SURF6'],
        ['RPS20','SEL1L3'],
        ['RPS23','RPL37A'],
        ['RPS25','FLNB'],
        ['RPS25','SEC22B'],
        ['RPS27','POLDIP3'],
        ['RPS27','YWHAE'],
        ['RPS29','CHD4'],
        ['RPS5','RPS13'],
        ['RPSA','MYO10'],
        ['RRP1B','TRPC4AP'],
        ['SASS6','LRRC7'],
        ['SBF1','SRSF2'],
        ['SERF2','HSPA5'],
        ['SERINC3','TGM2'],
        ['SF1','ANKFY1'],
        ['SF3B1','OS9'],
        ['SHMT2','EFTUD2'],
        ['SIPA1L1','STIP1'],
        ['SLC25A13','TPI1'],
        ['SLC25A3','ADGRE5'],
        ['SLC25A5','CYCS'],
        ['SLC2A1','RPS6'],
        ['SLC31A1','YWHAZ'],
        ['SLC35E2','CDK11B'],
        ['SLC3A2','PPIA'],
        ['SLC43A3','RAB3GAP1'],
        ['SLCO1A2','C3ORF67'],
        ['SLCO1A2','XCL1'],
        ['SMARCA1','AATF'],
        ['SMARCA4','C19ORF44'],
        ['SMC4','ARFGEF2'],
        ['SNRNP70','RPL5'],
        ['SNW1','IGFBP4'],
        ['SNX12','MAP4K4'],
        ['SORBS3','RPL29'],
        ['SPEN','NEU1'],
        ['SREBF1','MPP5'],
        ['SRRM4','L2HGDH'],
        ['SSC5D','ACADVL'],
        ['ST13','ADCY1'],
        ['STARD10','FCHSD2'],
        ['STAT1','LNPEP'],
        ['STC2','RPL28'],
        ['STRIP2','SMKR1'],
        ['SULT1A3','PTGES2'],
        ['SULT1A4','PTGES2'],
        ['SUMF1','LRRFIP2'],
        ['SYN2','PPARG'],
        ['TAF2','COLEC10'],
        ['TAF4','BRIP1'],
        ['TARDBP','PFKP'],
        ['TATDN2','EZR'],
        ['TBC1D22A','GRAMD4'],
        ['TBC1D31','ZNF704'],
        ['TCP1','RPL37'],
        ['TCP1','WDR5'],
        ['TCP11X2','NXF2'],
        ['TCP11X2','NXF2B'],
        ['TEAD2','SLC6A16'],
        ['TIAM1','EEF2'],
        ['TIMP1','RPL8'],
        ['TJP2','ADGRG6'],
        ['TLN1','HSF4'],
        ['TMCO3','TFDP1'],
        ['TMED9','EZR'],
        ['TMEM129','ELAC2'],
        ['TMSB10','EIF5A'],
        ['TNKS','ABCC9'],
        ['TPI1','RPL4'],
        ['TRAPPC9','KCNK9'],
        ['TRIM28','ANXA5'],
        ['TRMT12','RNF139'],
        ['TSPAN9','TEAD4'],
        ['TUBB','PLEKHG2'],
        ['TUBB3','DYNC1H1'],
        ['TUBB6','PVRL2'],
        ['TUFM','EIF4B'],
        ['TXN','GLUD1'],
        ['TXNRD1','RPL37A'],
        ['UBAP2L','ZBTB5'],
        ['UBC','TMEM189-UBE2V1'],
        ['UBE2O','PTMA'],
        ['UBN1','UBALD1'],
        ['UPK3B','LOC100289561'],
        ['UQCRC1','CPNE3'],
        ['USMG5','CAPN2'],
        ['USP45','FAXC'],
        ['UTRN','GALNT2'],
        ['VSTM2L','CTNNBL1'],
        ['VWA9','SERINC1'],
        ['WARS2','TBX15'],
        ['WDR60','NCAPG2'],
        ['WLS','PSAP'],
        ['XPO1','PKM'],
        ['XPR1','CTSC'],
        ['YIPF6','BCAP31'],
        ['YWHAZ','PPIA'],
        ['ZBTB17','SLC22A15'],
        ['ZBTB4','CTSD'],
        ['ZFP36L1','CST3'],
        ['ZNF703','ENAH'],
        ['ZNF74','RPLP1'],
        ['ZNF852','ZNF445'],
        ['AAMP','PIGT'],
        ['AARS','LMO7'],
        ['ABCC9','AKR1C1'],
        ['ABCC9','KYNU'],
        ['ABHD12B','IL34'],
        ['ABHD14B','ARF3'],
        ['ACADSB','FBXL12'],
        ['ACADVL','ANAPC5'],
        ['ACADVL','ATP5F1'],
        ['ACADVL','JAK1'],
        ['ACADVL','PKM'],
        ['ACLY','NIPSNAP1'],
        ['ACLY','SIPA1L3'],
        ['ACTA2','GPX1'],
        ['ACTG1','BID'],
        ['ACTN1','RPL9'],
        ['ACTN4','BSG'],
        ['ACTN4','CD59'],
        ['ACTN4','EIF4H'],
        ['ACTN4','HN1L'],
        ['ACTN4','MGST1'],
        ['ACTN4','PKD1'],
        ['ACTN4','PRPF6'],
        ['ACTN4','SLC39A4'],
        ['ACTN4','TKT'],
        ['ACTR1A','TPI1'],
        ['ACTR2','EGR1'],
        ['ACTR2','RPL11'],
        ['ACTR2','RPS2'],
        ['ACTR3','GANAB'],
        ['ADAM15','EMP1'],
        ['ADAM19','CYFIP2'],
        ['ADAM9','ILF3'],
        ['ADAM9','SPATS2L'],
        ['ADAMTS1','STC2'],
        ['ADAMTS18','SYCE1L'],
        ['ADAR','NOLC1'],
        ['ADAR','RPL31'],
        ['ADGRL2','SPAG9'],
        ['ADIPOR1','PNMA1'],
        ['ADIRF','LGALS1'],
        ['ADK','CAPN1'],
        ['ADRBK1','RPS27A'],
        ['AFF1','C4ORF36'],
        ['AFF1','ORC4'],
        ['AGFG1','FLNA'],
        ['AGR2','EPN1'],
        ['AGRN','ITGB4'],
        ['AGRN','PKM'],
        ['AGRN','PLEC'],
        ['AGRN','PLXND1'],
        ['AHCY','RPL32'],
        ['AK2','HNRNPA1'],
        ['AK2','RPS24'],
        ['AK3','TET2'],
        ['AK4','NF1'],
        ['AK5','PIGK'],
        ['AKAP13','CHRM3'],
        ['AKR1C1','RPSA'],
        ['AKR1C2','RPSA'],
        ['AKT1','ALDOA'],
        ['ALDH16A1','KLK14'],
        ['ALDOA','ANXA5'],
        ['ALDOA','MMP14'],
        ['ALDOA','RPS26'],
        ['ALG1L2','FAM90A1'],
        ['ALKBH4','CUX1'],
        ['ANKDD1B','CDYL'],
        ['ANKH','ARHGEF2'],
        ['ANKRD10','MAP1S'],
        ['ANKRD16','FHL2'],
        ['ANXA1','CLIC1'],
        ['ANXA11','SET'],
        ['ANXA2','DCBLD2'],
        ['ANXA2','HDGF'],
        ['ANXA2','NACA'],
        ['ANXA4','NMT1'],
        ['AOX1','KCTD18'],
        ['AP2B1','PLEC'],
        ['AP2M1','HLA-B'],
        ['AP2M1','RAN'],
        ['AP2M1','RPS27'],
        ['APBB1IP','HNRNPA3'],
        ['APLP2','HNRNPUL1'],
        ['APLP2','NBPF12'],
        ['APOL1','EIF5A'],
        ['APOL4','CHMP4A'],
        ['APOL4','ZNF630'],
        ['APP','DYNC1H1'],
        ['AREG','MRPL14'],
        ['ARF1','GNB2L1'],
        ['ARF1','KRT8'],
        ['ARF1','PLEC'],
        ['ARF1','RPL38'],
        ['ARF1','RPS12'],
        ['ARF3','CYR61'],
        ['ARF3','EFCAB14'],
        ['ARHGAP11A','CALU'],
        ['ARHGAP29','NAP1L1'],
        ['ARHGDIA','MET'],
        ['ARHGDIA','RPL37'],
        ['ARHGEF35','CTAGE9'],
        ['ARL17A','LRRC37A3'],
        ['ARL6IP1','CCT4'],
        ['ARL6IP1','EIF1AX'],
        ['ARMC5','PABPC1'],
        ['ARMC5','USP6NL'],
        ['ARPC1B','HMGN2'],
        ['ARPC2','YWHAZ'],
        ['ARPC5L','FTH1'],
        ['ASH1L','IMP4'],
        ['ASPH','EEF1G'],
        ['ASPH','MYO1B'],
        ['ASTN2','SUPT20H'],
        ['ASXL1','H3F3B'],
        ['ATAD2','FXR1'],
        ['ATF4','PKM'],
        ['ATF4','SET'],
        ['ATG10','RPS23'],
        ['ATG101','LGALS1'],
        ['ATN1','HLA-A'],
        ['ATN1','RPS24'],
        ['ATP13A3','SCD'],
        ['ATP1A1','SERPINH1'],
        ['ATP1B1','MLLT1'],
        ['ATP2A2','RPS13'],
        ['ATP2A2','SLC3A2'],
        ['ATP2C2','ORC4'],
        ['ATP5A1','GALE'],
        ['ATP5A1','GANAB'],
        ['ATP5A1','HMGCS1'],
        ['ATP5B','CCT7'],
        ['ATP5B','COL17A1'],
        ['ATP5B','RPSA'],
        ['ATP5B','SEPT7'],
        ['ATP5B','TUBB'],
        ['ATP5G2','PLEC'],
        ['ATP5G3','LY6E'],
        ['ATP5G3','RPL35'],
        ['ATP5G3','RPS18'],
        ['ATP5G3','SREBF2'],
        ['ATP5L','CAST'],
        ['ATP6AP1','CTDNEP1'],
        ['ATP6AP1','CTSD'],
        ['ATP6AP1','RPS3'],
        ['ATP6AP2','VDAC2'],
        ['ATP6V0E1','LRP1'],
        ['ATXN10','RPS28'],
        ['ATXN2L','RPL31'],
        ['AXDND1','CHML'],
        ['AXL','RPL24'],
        ['AXL','RPL8'],
        ['B2M','GPI'],
        ['B2M','MCM7'],
        ['B2M','PRPF40A'],
        ['B4GALT5','EIF4B'],
        ['BAZ2A','EIF5'],
        ['BBS5','FANCI'],
        ['BBS5','ZNF726'],
        ['BCAP31','TLN1'],
        ['BCL2A1','CHM'],
        ['BHMT','DMGDH'],
        ['BIN1','FLOT2'],
        ['BLOC1S6','ORC4'],
        ['BNIP3','KRT18'],
        ['BOP1','MROH1'],
        ['BRD2','HLA-DMA'],
        ['BRD2','S100A2'],
        ['BRK1','RPL10'],
        ['BRK1','RPS28'],
        ['BSG','DYNC1H1'],
        ['BSG','FUS'],
        ['BSG','GLIS2'],
        ['BSG','HSPA5'],
        ['BSG','RNMT'],
        ['BSG','RPS10'],
        ['BSG','RPS10-NUDT3'],
        ['BSG','SUPT16H'],
        ['BTBD1','POM121'],
        ['BTBD1','POM121C'],
        ['BTBD7','YWHAH'],
        ['BTF3','MAP4'],
        ['BTF3','MPRIP'],
        ['BTG3','CXADR'],
        ['BUB3','NONO'],
        ['BZW2','GLS'],
        ['C10ORF35','NEUROG3'],
        ['C11ORF31','PCBP2'],
        ['C14ORF159','RPS6KA5'],
        ['C14ORF2','RPL34'],
        ['C1D','CCDC7'],
        ['C21ORF59','PKM'],
        ['C2ORF27A','TOMM40'],
        ['C5ORF42','NIPBL'],
        ['C6ORF136','PPP1R10'],
        ['C7ORF55-LUC7L2','KLRG2'],
        ['CA12','OPTN'],
        ['CAB39L','GOSR1'],
        ['CABLES1','TMEM241'],
        ['CADPS','GRB10'],
        ['CALM2','HSP90AB1'],
        ['CALM2','RPS14'],
        ['CALM2','RPS24'],
        ['CALM3','HMGN2'],
        ['CALM3','SLC39A7'],
        ['CALM3','SRSF1'],
        ['CALR','PLEC'],
        ['CALR','TUBB'],
        ['CALU','GNAS'],
        ['CALU','RPS5'],
        ['CAMSAP1','RPS8'],
        ['CANX','COX6A1'],
        ['CANX','HDLBP'],
        ['CANX','LDHA'],
        ['CANX','RAC1'],
        ['CANX','RAD23B'],
        ['CAP1','H2AFY'],
        ['CAP1','RPL13A'],
        ['CAPN1','NCL'],
        ['CAPN1','SQSTM1'],
        ['CAPN2','RPL7A'],
        ['CAPNS1','DDX1'],
        ['CAPRIN1','REEP4'],
        ['CAPZA2','RPLP1'],
        ['CAV1','DCTN1'],
        ['CAV1','DHFR'],
        ['CAV1','STAT1'],
        ['CAV2','RPL8'],
        ['CBX3','HMGB1'],
        ['CCDC102B','GJB7'],
        ['CCDC137','SH3D19'],
        ['CCDC15','ZNF562'],
        ['CCDC33','STRA6'],
        ['CCDC80','GRHPR'],
        ['CCDC85A','CHML'],
        ['CCDC85B','RPS4X'],
        ['CCDC88C','NCSTN'],
        ['CCL26','PRELP'],
        ['CCND1','DLG1'],
        ['CCND1','HERC2'],
        ['CCND1','KRT19'],
        ['CCND1','ORAOV1'],
        ['CCND1','PKM'],
        ['CCND1','UQCRH'],
        ['CCNI','DIAPH1'],
        ['CCNK','CD9'],
        ['CCT2','RPL22'],
        ['CCT4','COMMD1'],
        ['CCT5','IMPDH2'],
        ['CCT8','SF3B2'],
        ['CCT8','TAGLN2'],
        ['CD151','RPS2'],
        ['CD164','HNRNPC'],
        ['CD226','DOK6'],
        ['CD37','SLC6A16'],
        ['CD44','LASP1'],
        ['CD44','RPS9'],
        ['CD46','TMBIM6'],
        ['CD53','CUL4A'],
        ['CD53','LRIF1'],
        ['CD55','SPIN1'],
        ['CD59','LAMP1'],
        ['CD63','CFL1'],
        ['CD63','HSPA5'],
        ['CD63','RAB5C'],
        ['CD9','SF3B1'],
        ['CDC20','PKP3'],
        ['CDC37','MAP4K4'],
        ['CDH1','EPS8L2'],
        ['CDK12','TPT1'],
        ['CDV3','NOP10'],
        ['CDX1','LYPLA2'],
        ['CDYL','DCDC1'],
        ['CDYL','FAM65B'],
        ['CDYL','MSANTD3-TMEFF1'],
        ['CDYL','TMEFF1'],
        ['CEBPB','EPAS1'],
        ['CELF1','PTPMT1'],
        ['CELF4','TXNRD1'],
        ['CELSR2','SORT1'],
        ['CEP250','KRT8'],
        ['CEP44','GOSR1'],
        ['CERS2','DDB1'],
        ['CERS6','RPS3A'],
        ['CFAP52','CHRM3'],
        ['CFL1','CRIP1'],
        ['CFL1','CYC1'],
        ['CFL1','FERMT2'],
        ['CFL1','HMGA1'],
        ['CFL1','HSF1'],
        ['CFL1','LDHA'],
        ['CFL1','LUC7L3'],
        ['CFL1','NGRN'],
        ['CFL1','RPL37'],
        ['CFL1','YWHAB'],
        ['CHCHD2','COX6C'],
        ['CHCHD2','RPL23'],
        ['CHCHD2','RPLP1'],
        ['CHCHD2','RPN2'],
        ['CHCHD2','TRA2B'],
        ['CHD1','LAPTM5'],
        ['CHD3','RPS2'],
        ['CHD4','GNL3'],
        ['CHD4','IPO7'],
        ['CHD4','PABPC1'],
        ['CHD4','RPS5'],
        ['CHD5','PSMD3'],
        ['CHIC2','GSX2'],
        ['CHM','PAPPA2'],
        ['CHML','EPG5'],
        ['CHRM3','CLIC2'],
        ['CHRM3','CNTN5'],
        ['CHRM3','CNTNAP5'],
        ['CHRM3','DCLK1'],
        ['CHRM3','DNAH3'],
        ['CHRM3','DPY19L1'],
        ['CHRM3','EI24'],
        ['CHRM3','FAM49A'],
        ['CHRM3','FAM65B'],
        ['CHRM3','FGF12'],
        ['CHRM3','FSIP2'],
        ['CHRM3','GABRA3'],
        ['CHRM3','H2BFWT'],
        ['CHRM3','HECW2'],
        ['CHRM3','HSD17B11'],
        ['CHRM3','ILDR1'],
        ['CHRM3','ITIH6'],
        ['CHRM3','KDM5A'],
        ['CHRM3','KYNU'],
        ['CHRM3','MAST4'],
        ['CHRM3','MYO10'],
        ['CHRM3','MYO1D'],
        ['CHRM3','NEB'],
        ['CHRM3','NKAIN2'],
        ['CHRM3','PHLPP1'],
        ['CHRM3','PRKCA'],
        ['CHRM3','PYROXD1'],
        ['CHRM3','RASEF'],
        ['CHRM3','RFTN2'],
        ['CHRM3','SLC22A15'],
        ['CHRM3','SLC24A2'],
        ['CHRM3','SLC8A1'],
        ['CHRM3','SPTLC1'],
        ['CHRM3','STAG1'],
        ['CHRM3','TCF24'],
        ['CHRM3','TENM4'],
        ['CHRM3','TMEM135'],
        ['CHRM3','TMEM163'],
        ['CHRM3','TMLHE'],
        ['CHRM3','TRPC4'],
        ['CHRM3','TYRP1'],
        ['CIRBP','STX16'],
        ['CKAP4','PPIA'],
        ['CKAP4','SEPT7'],
        ['CKAP5','YWHAE'],
        ['CKB','PDIA3'],
        ['CKLF','CMTM4'],
        ['CKS1B','ZNF106'],
        ['CLDN18','DZIP1L'],
        ['CLIC1','MFSD1'],
        ['CLIC1','MTRNR2L2'],
        ['CLIC2','SLCO1A2'],
        ['CLIC2','TMLHE'],
        ['CLIC4','LDHA'],
        ['CLIC4','PPIA'],
        ['CLSTN1','GNAS'],
        ['CLSTN1','SRRM2'],
        ['CLUH','HNRNPL'],
        ['CMTM6','CMTM7'],
        ['CNGB3','SLCO1A2'],
        ['CNOT1','ITGA3'],
        ['CNOT4','NUP205'],
        ['CNOT7','SHISA5'],
        ['CNPY1','EN2'],
        ['CNTN5','SLCO1A2'],
        ['COA6','RPS15'],
        ['COL4A1','TUBB'],
        ['COL4A3BP','RPS17'],
        ['COL5A1','ERH'],
        ['COL5A1','RPLP0'],
        ['COL5A1','RPS8'],
        ['COL5A1','TPT1'],
        ['COL6A3','RPL22'],
        ['COLGALT1','PLEC'],
        ['COPB2','CTSB'],
        ['COPB2','FAM96B'],
        ['COPB2','MUC16'],
        ['COPS6','H3F3A'],
        ['COPS6','RPL10'],
        ['COPZ1','MRC2'],
        ['COPZ1','NUP93'],
        ['COX16','RPS17'],
        ['COX4I1','TUBB4B'],
        ['COX5A','MYL12A'],
        ['COX6C','RPS18'],
        ['COX7C','MCM7'],
        ['COX7C','TMSB4X'],
        ['CPNE2','IFIH1'],
        ['CPSF7','SEC61A1'],
        ['CRAT','SEPT11'],
        ['CREB3L2','RPS2'],
        ['CRIM1','MYL12A'],
        ['CRLF3','SUZ12'],
        ['CSDE1','RPL18A'],
        ['CSE1L','PKM'],
        ['CSK','EIF5A'],
        ['CSNK1G2','UCKL1'],
        ['CSNK2A1','TUBB'],
        ['CTDSP2','PKM'],
        ['CTDSP2','RAB7A'],
        ['CTDSP2','TARDBP'],
        ['CTNNA1','RPL10'],
        ['CTNNAL1','THOC3'],
        ['CTNND1','SF3B4'],
        ['CTSC','PLAU'],
        ['CTSC','RPL41'],
        ['CTSD','DBNL'],
        ['CTSD','SETD5'],
        ['CTSL','INTS1'],
        ['CTTN','HSPA5'],
        ['CTTN','MCM4'],
        ['CUX1','MAVS'],
        ['CYB5B','RPL15'],
        ['CYC1','HNRNPK'],
        ['CYP2F1','EGLN2'],
        ['CYP3A7-CYP3A51P','ZSCAN25'],
        ['CYR61','RPS3'],
        ['CYTH2','RPL13'],
        ['CYTH2','WDR33'],
        ['DAND5','GADD45GIP1'],
        ['DAZAP1','HSPA5'],
        ['DBN1','SDCBP'],
        ['DBNDD1','HSPA5'],
        ['DCAF11','NRL'],
        ['DCAF15','PODNL1'],
        ['DCBLD2','LMF2'],
        ['DCBLD2','ORC4'],
        ['DCK','RPL3'],
        ['DCTN1','HSPG2'],
        ['DDB1','PRRC2C'],
        ['DDB1','RPS29'],
        ['DDB1','UBB'],
        ['DDOST','TPT1'],
        ['DDRGK1','JDP2'],
        ['DDX17','KIF5B'],
        ['DDX39B','RPS27A'],
        ['DDX43','OOEP'],
        ['DDX47','MSC'],
        ['DDX5','SND1'],
        ['DDX5','SNRPA'],
        ['DDX54','RPS5'],
        ['DEDD','UFC1'],
        ['DEFB126','FOCAD'],
        ['DEK','GPS2'],
        ['DHX30','EMCN'],
        ['DIAPH1','RPL36'],
        ['DKK3','SLC7A5'],
        ['DLAT','U2AF2'],
        ['DNAH3','FOCAD'],
        ['DNAJB6','PFN1'],
        ['DNAJC5','MMP2'],
        ['DNHD1','RRP8'],
        ['DOCK1','PPP2R1A'],
        ['DOCK3','SLCO1A2'],
        ['DONSON','SON'],
        ['DPAGT1','SPRY4'],
        ['DPP3','ITGA5'],
        ['DRAP1','RPL12'],
        ['DRAP1','SQSTM1'],
        ['DTWD1','NMNAT3'],
        ['DYNC1H1','TAOK1'],
        ['DYNC1H1','UCHL1'],
        ['DYNC1LI2','SRP14'],
        ['EDEM1','MKI67'],
        ['EDF1','PHPT1'],
        ['EDF1','PLAU'],
        ['EEF1B2','HSPA8'],
        ['EEF1B2','KRT8'],
        ['EEF1D','PHLDA1'],
        ['EEF1D','RPL9'],
        ['EEF1D','SQSTM1'],
        ['EEF1G','GRINA'],
        ['EEF1G','IRAK1'],
        ['EEF1G','LMAN1'],
        ['EEF1G','RPN2'],
        ['EEF1G','STAU1'],
        ['EEF2','HDGF'],
        ['EEF2','RAN'],
        ['EEF2','RANGAP1'],
        ['EHD2','TAGLN2'],
        ['EIF2AK1','RPL3'],
        ['EIF2B5','SLAMF7'],
        ['EIF2S3','HMGB2'],
        ['EIF2S3','RAB1A'],
        ['EIF3A','SPTAN1'],
        ['EIF3A','STT3B'],
        ['EIF3B','RPL23A'],
        ['EIF3B','RPL39'],
        ['EIF3C','OXA1L'],
        ['EIF3CL','OXA1L'],
        ['EIF3F','RPL10'],
        ['EIF3F','S100A10'],
        ['EIF3H','FASN'],
        ['EIF3H','MTRNR2L2'],
        ['EIF4A1','HSP90AB1'],
        ['EIF4A1','LDHA'],
        ['EIF4A1','MARCKS'],
        ['EIF4A1','PTMA'],
        ['EIF4A1','RPL18'],
        ['EIF4B','RPL23A'],
        ['EIF4G1','ITGA3'],
        ['EIF4G1','RPS6'],
        ['EIF4G1','SHC1'],
        ['EIF4G1','TPX2'],
        ['EIF4H','HSPA5'],
        ['EIF4H','RPL30'],
        ['EIF4H','RPL7A'],
        ['EIF5A','GSTP1'],
        ['EIF5A','HMGN2'],
        ['EIF5A','RPS18'],
        ['EIF5A','S100A6'],
        ['EIF5A','TUBB4B'],
        ['EIF5A','TXNRD1'],
        ['ELF4','RAB33A'],
        ['ELK1','PRKAA1'],
        ['EMC3','SNRPB'],
        ['EMD','MYL6'],
        ['ENO1','HMGA1'],
        ['ENO1','LAMA5'],
        ['ENO1','MFSD14B'],
        ['ENO1','RPL18A'],
        ['ENSA','TMSB10'],
        ['ENTPD1','GVQW2'],
        ['EPCAM','RPS8'],
        ['EPHA2','KRT7'],
        ['EPHB2','RNF213'],
        ['EPHX1','NOMO1'],
        ['EPHX1','NOMO2'],
        ['EPHX1','NOMO3'],
        ['EPN1','WASH1'],
        ['EPRS','RPLP2'],
        ['EPS8L2','S100A6'],
        ['ERH','RPS28'],
        ['ESYT2','RPL37A'],
        ['EVC','FAM129B'],
        ['EWSR1','FUS'],
        ['EWSR1','UBC'],
        ['EXOC5','RPL27A'],
        ['EXOSC10','MAB21L3'],
        ['EYS','GJB7'],
        ['EZR','GNL3'],
        ['EZR','RPL6'],
        ['F8','TMLHE'],
        ['FABP5','S100A6'],
        ['FADS2','SRRT'],
        ['FAM129B','NECAP2'],
        ['FAM129B','S100A6'],
        ['FAM129B','SERF2'],
        ['FAM46C','PNRC2'],
        ['FAM60A','PKM'],
        ['FAM71E2','TMEM190'],
        ['FAM83H','PLCB3'],
        ['FAM84B','RPL17'],
        ['FAM84B','RPL17-C18ORF32'],
        ['FAM91A1','UCHL3'],
        ['FANCI','GOSR1'],
        ['FARSA','GCN1'],
        ['FARSB','USP33'],
        ['FASN','RAB5B'],
        ['FASN','TM7SF3'],
        ['FAT1','HERC2'],
        ['FAT1','RPL3'],
        ['FBL','MAPRE1'],
        ['FBXL14','WNT5B'],
        ['FBXO3','IMPA2'],
        ['FBXW8','RPL37'],
        ['FER','THOC2'],
        ['FGD4','GOSR1'],
        ['FGF12','FOCAD'],
        ['FGF2','SF1'],
        ['FGFRL1','RPL41'],
        ['FHL2','KRI1'],
        ['FJX1','PAMR1'],
        ['FKBP1A','OSBPL8'],
        ['FKBP6','TRIM73'],
        ['FKBP6','TRIM74'],
        ['FKBP8','PRRC2A'],
        ['FKBP8','UBAP2L'],
        ['FKBP9','RPLP0'],
        ['FKTN','GRINA'],
        ['FLII','MAP4'],
        ['FLII','TMED9'],
        ['FLNA','NCEH1'],
        ['FLNB','SLC7A5'],
        ['FLNB','YKT6'],
        ['FOCAD','GABRA3'],
        ['FOCAD','KLHL3'],
        ['FOCAD','TSHZ2'],
        ['FOSL1','PARP4'],
        ['FOXK1','FTH1'],
        ['FOXRED1','IGSF9'],
        ['FURIN','PUF60'],
        ['FURIN','YBX1'],
        ['FUS','PLEC'],
        ['FUS','RPS9'],
        ['FUS','TMEM2'],
        ['FYTTD1','HSP90AB1'],
        ['FZD4','PRSS23'],
        ['G3BP2','RPS18'],
        ['G3BP2','SRSF5'],
        ['GABRR1','KIAA1324L'],
        ['GABRR1','SLC44A5'],
        ['GALNT5','LOC100130451'],
        ['GANAB','PPIA'],
        ['GANAB','RPL19'],
        ['GANAB','RPL39'],
        ['GANAB','RPS16'],
        ['GAPDH','KRT18'],
        ['GARS','MPRIP'],
        ['GARS','RBM39'],
        ['GARS','WDTC1'],
        ['GATAD1','PEX1'],
        ['GBA','MTX1'],
        ['GCN1','HSPA5'],
        ['GCNT3','GTF2A2'],
        ['GDI2','RPSA'],
        ['GIPC1','MRPL37'],
        ['GJB7','ST6GALNAC5'],
        ['GJB7','XDH'],
        ['GLUD1','HNRNPC'],
        ['GNB1','PFN1'],
        ['GNB1','PTX3'],
        ['GNB1','PYGB'],
        ['GNB2L1','LAMB1'],
        ['GNB2L1','SH3PXD2A'],
        ['GNGT1','TFPI2'],
        ['GOLGA2','SNRNP200'],
        ['GOLGA3','RPS11'],
        ['GOLGA8A','HIST2H2AA4'],
        ['GOLGA8B','HIST2H2AA4'],
        ['GOLIM4','HUWE1'],
        ['GOSR1','HSD17B6'],
        ['GOSR1','PTGES3L'],
        ['GOSR1','ZNF726'],
        ['GOT1','RPL30'],
        ['GPI','RPS13'],
        ['GPKOW','UIMC1'],
        ['GPX1','RPL37A'],
        ['GREM1','PPIA'],
        ['GRN','UBE2H'],
        ['GSPT1','PELP1'],
        ['GSTP1','PDXK'],
        ['GSTP1','PLEC'],
        ['GSTP1','RPL8'],
        ['GSTP1','TMED2'],
        ['GUK1','SPTAN1'],
        ['H2AFV','PFN1'],
        ['H2AFY','KPNB1'],
        ['H2AFY','RPS3'],
        ['H2AFZ','RPL22'],
        ['H3F3B','HMGN2'],
        ['H3F3B','SNRNP200'],
        ['HADHA','NR1D2'],
        ['HDAC1','PABPC1'],
        ['HDGF','PPP1R15A'],
        ['HDGF','RPS2'],
        ['HDLBP','PABPC1'],
        ['HDLBP','PRKCSH'],
        ['HDLBP','RPL11'],
        ['HDLBP','RPL23A'],
        ['HDLBP','RPL32'],
        ['HEBP2','PLAC8'],
        ['HECTD3','NME1'],
        ['HERC2','RPL3'],
        ['HIBCH','INPP1'],
        ['HIPK2','UBC'],
        ['HIST1H1C','RABL6'],
        ['HK1','SEC11A'],
        ['HLA-A','LIF'],
        ['HLA-B','OGFR'],
        ['HLA-B','TPD52L2'],
        ['HLA-C','MMP14'],
        ['HLA-DQA1','HLA-DQB1'],
        ['HLA-F','MICB'],
        ['HMG20B','MFSD12'],
        ['HMGA1','RPA1'],
        ['HMGB1','IPO8'],
        ['HMGB1','PPIA'],
        ['HMGB1','RPS11'],
        ['HMGN1','RPL35'],
        ['HMGN2','KRT8'],
        ['HMGN2','RPL35'],
        ['HMGN2','RPS3'],
        ['HMGN2','RPS9'],
        ['HMGN2','TPM4'],
        ['HN1','RPL28'],
        ['HN1','YWHAB'],
        ['HNRNPA0','RPL30'],
        ['HNRNPA0','RPLP2'],
        ['HNRNPA1','SET'],
        ['HNRNPA1','SRF'],
        ['HNRNPA2B1','PCBP1'],
        ['HNRNPA2B1','RPL13'],
        ['HNRNPA3','ITGB4'],
        ['HNRNPAB','NONO'],
        ['HNRNPAB','SLC39A1'],
        ['HNRNPC','HSPA5'],
        ['HNRNPC','ITGA3'],
        ['HNRNPC','SNRPB'],
        ['HNRNPD','MLEC'],
        ['HNRNPD','RPL32'],
        ['HNRNPF','PTMA'],
        ['HNRNPF','RPL37A'],
        ['HNRNPH1','MT2A'],
        ['HNRNPH1','POLR2A'],
        ['HNRNPH1','RPL5'],
        ['HNRNPK','RPS10-NUDT3'],
        ['HNRNPK','SLC1A5'],
        ['HNRNPL','RPS28'],
        ['HNRNPL','VIM'],
        ['HNRNPU','RPLP1'],
        ['HNRNPU','RPN2'],
        ['HNRNPU','RPS28'],
        ['HNRNPUL1','RPL8'],
        ['HNRNPUL1','SLC16A3'],
        ['HNRNPUL1','TGFB1'],
        ['HOOK2','JUNB'],
        ['HSD17B6','PRIM1'],
        ['HSP90AB1','HUWE1'],
        ['HSP90AB1','MFSD14B'],
        ['HSP90AB1','RPS19'],
        ['HSP90AB1','YWHAZ'],
        ['HSP90B1','LASP1'],
        ['HSP90B1','PLEC'],
        ['HSP90B1','SET'],
        ['HSPA1A','PPA1'],
        ['HSPA1B','SLC4A2'],
        ['HSPA5','NME2'],
        ['HSPA8','RPL22L1'],
        ['HSPA9','PABPC1'],
        ['HSPA9','RPLP0'],
        ['HSPB1','HUWE1'],
        ['HSPD1','PAICS'],
        ['HSPD1','PPIA'],
        ['HSPD1','PRKDC'],
        ['HSPD1','PUF60'],
        ['HSPD1','RRP7A'],
        ['HSPD1','UBA1'],
        ['HYLS1','PATE2'],
        ['ID1','SET'],
        ['IFIT5','LIPA'],
        ['IFITM1','LMNA'],
        ['IFNGR2','TMEM50B'],
        ['IGF2R','RPS27A'],
        ['IGF2R','RRBP1'],
        ['IGSF11','KYNU'],
        ['IKZF1','TTC7A'],
        ['IL10','IL19'],
        ['IL15','STAC'],
        ['INA','NT5C2'],
        ['INPPL1','PKM'],
        ['INPPL1','RPLP0'],
        ['INTS1','RPL7A'],
        ['IPO5','RPL10A'],
        ['IPO7','PKM'],
        ['IPO7','TM4SF1'],
        ['IQGAP1','NDST1'],
        ['IRAK1BP1','PHIP'],
        ['ITGA3','RPLP2'],
        ['ITGA3','SRRM2'],
        ['ITGA3','WHSC1'],
        ['ITGB4','RPS3'],
        ['ITM2B','PPP4C'],
        ['ITPR3','MZT2B'],
        ['ITPR3','RPL13A'],
        ['JUND','PIN1'],
        ['KARS','LDHA'],
        ['KAT6A','RPL32'],
        ['KHDRBS1','NAA50'],
        ['KHDRBS1','SERPINB6'],
        ['KHNYN','MTRNR2L2'],
        ['KHSRP','RPS21'],
        ['KIF26B','SMYD3'],
        ['KLF5','RPS2'],
        ['KLHDC10','UBE2Q1'],
        ['KLRG2','LUC7L2'],
        ['KMT2C','TPTE'],
        ['KPNA2','SNRNP200'],
        ['KPNB1','METAP2'],
        ['KPNB1','RPL10A'],
        ['KRT17','RBM39'],
        ['KRT18','NUP188'],
        ['KRT18','PLEC'],
        ['KRT18','PPP2R4'],
        ['KRT18','TXN'],
        ['KRT19','RPL36A'],
        ['KTN1','ZMPSTE24'],
        ['LAMA5','P4HB'],
        ['LAMA5','RPS2'],
        ['LAMA5','SOX13'],
        ['LAMA5','YBX3'],
        ['LAMA5','ZNF680'],
        ['LAMC2','PRRC2A'],
        ['LAMP1','OFD1'],
        ['LARP1','USP22'],
        ['LASP1','LRP5'],
        ['LDHA','PKM'],
        ['LDHA','PSMD11'],
        ['LDHA','RPS23'],
        ['LDHA','TAGLN2'],
        ['LDHA','TRIM28'],
        ['LDHA','UBR4'],
        ['LDHB','NME2'],
        ['LGALS1','NME1-NME2'],
        ['LGALS1','NME2'],
        ['LGALS1','PLEC'],
        ['LGALS1','RPL6'],
        ['LGALS3BP','PKP1'],
        ['LGALS3BP','RPS2'],
        ['LGALS3BP','TNFAIP2'],
        ['LHFPL2','RPL30'],
        ['LIMA1','SEPT7'],
        ['LMAN1','RPS13'],
        ['LMNA','LYPD1'],
        ['LMNA','RPS28'],
        ['LONP1','PKM'],
        ['LRPPRC','YWHAZ'],
        ['LRRC27','STK32C'],
        ['LRRC59','RPS28'],
        ['LSM4','PKM'],
        ['LSM4','RCC2'],
        ['LSM7','RPS15A'],
        ['LTB4R','RCN1'],
        ['MAD2L2','OAT'],
        ['MAF1','PFN1'],
        ['MAGT1','P4HB'],
        ['MAN1B1','RPL18'],
        ['MAP4','PKM'],
        ['MARCKS','SNRPF'],
        ['MARK4','PLEC'],
        ['MATR3','NELFCD'],
        ['MATR3','SDHD'],
        ['MBD3','PTMA'],
        ['MBD3','RPL7A'],
        ['MBD6','RBM3'],
        ['MCL1','RPS19'],
        ['MCM2','RPS5'],
        ['MCM4','PLEKHO1'],
        ['MCM7','MED24'],
        ['MCRS1','RPS5'],
        ['MFI2','NCAPD2'],
        ['MGAT5','RPS6'],
        ['MICAL3','NCOR1'],
        ['MIDN','SLC2A3'],
        ['MIF','PKM'],
        ['MIF','RPL10'],
        ['MIF','RPLP2'],
        ['MIF','RPS18'],
        ['MKL2','TMED9'],
        ['MLEC','RPL9'],
        ['MLF2','NOP56'],
        ['MMP14','SRRM2'],
        ['MMP23B','NADK'],
        ['MNAT1','SIX4'],
        ['MPRIP','ZKSCAN7'],
        ['MRPL15','RPS12'],
        ['MRPL4','RPS10'],
        ['MRPL4','RPS10-NUDT3'],
        ['MSL1','TXLNA'],
        ['MSN','PRRC2B'],
        ['MT2A','PHB'],
        ['MT2A','RND3'],
        ['MT2A','THY1'],
        ['MTHFD1','ZBTB25'],
        ['MTHFD2','RPL41'],
        ['MTRNR2L2','TARDBP'],
        ['MUS81','TPT1'],
        ['MXRA7','PSMD14'],
        ['MYDGF','RPS28'],
        ['MYH14','UBXN1'],
        ['MYH9','PDIA6'],
        ['MYH9','SMARCC1'],
        ['MYL12B','RBM8A'],
        ['MYL6','NUP210'],
        ['MYL6','VDAC1'],
        ['MYO1C','SLC7A5'],
        ['MYO1C','TUBB'],
        ['MYO9B','PLXNB1'],
        ['MYO9B','WDR1'],
        ['MYOF','SEC61A1'],
        ['MYOF','TRNP1'],
        ['MYOF','UBC'],
        ['MZT2A','OXA1L'],
        ['MZT2B','TUBB4B'],
        ['NAA50','RPS28'],
        ['NACC2','PLXNB2'],
        ['NAMPT','PPFIBP1'],
        ['NAMPT','RPS2'],
        ['NAP1L1','PFN1'],
        ['NASP','SPTAN1'],
        ['NAT10','TMSB10'],
        ['NBPF10','YWHAZ'],
        ['NBPF15','YWHAZ'],
        ['NBR1','PLOD1'],
        ['NCAPD2','PDIA6'],
        ['NCAPD2','RPL7'],
        ['NCAPD2','SLC12A7'],
        ['NCL','NSD1'],
        ['NCL','POLR2A'],
        ['NCL','RPS14'],
        ['NCL','RPS21'],
        ['NCL','UBE2C'],
        ['NCLN','PARK7'],
        ['NCOA3','RPL14'],
        ['NCOR2','PRRC2A'],
        ['NCOR2','RPL7A'],
        ['NCOR2','TRAPPC10'],
        ['NCR1','NLRP7'],
        ['NCSTN','NOP56'],
        ['NDST1','PFKP'],
        ['NDUFA10','PFKP'],
        ['NDUFB9','SPTAN1'],
        ['NDUFS8','TPRG1L'],
        ['NDUFV1','RPL7A'],
        ['NFASC','RBBP5'],
        ['NFE2L1','NGFR'],
        ['NFE2L1','RPS2'],
        ['NFE2L1','TCF3'],
        ['NGFRAP1','POLR3K'],
        ['NME1','SLC4A2'],
        ['NME1-NME2','RPL7'],
        ['NME1-NME2','RPS15'],
        ['NME1-NME2','SLC4A2'],
        ['NOL10','RPL37'],
        ['NONO','RPS9'],
        ['NONO','ZMYM3'],
        ['NOP10','RPS10'],
        ['NOP10','RPS10-NUDT3'],
        ['NPIPB3','P4HB'],
        ['NPIPB3','PKM'],
        ['NPIPB5','PKM'],
        ['NPIPB5','PLA2G10'],
        ['NPM1','PLEC'],
        ['NPM1','PRRC2A'],
        ['NPM1','RPL35'],
        ['NPM1','RPS27'],
        ['NPM1','SSRP1'],
        ['NPM1','TMSB10'],
        ['NQO1','RPL38'],
        ['NQO1','UBA1'],
        ['NUDT15','SUCLA2'],
        ['NUMA1','RPS21'],
        ['NUP188','PIEZO1'],
        ['NUP188','RPS18'],
        ['NWD1','PTPRE'],
        ['OAZ1','RPS28'],
        ['OGT','PPIA'],
        ['OGT','TNFRSF10B'],
        ['ORC4','PDIA6'],
        ['ORC4','REL'],
        ['ORC4','RPS15A'],
        ['ORC4','SCEL'],
        ['ORC4','WDR63'],
        ['ORC4','YEATS2'],
        ['ORMDL3','OSBPL10'],
        ['OTUD7B','VPS45'],
        ['P4HB','PSMA2'],
        ['P4HB','RPL36'],
        ['P4HB','SLC3A2'],
        ['PA2G4','RPS12'],
        ['PAFAH1B1','SLC25A3'],
        ['PAICS','RPL37A'],
        ['PALD1','PRF1'],
        ['PAPSS2','TRAM2'],
        ['PARP1','PKM'],
        ['PDE1C','RPS4X'],
        ['PDE5A','SEPT7'],
        ['PDIA6','PSPC1'],
        ['PDPR','PSMD7'],
        ['PDRG1','XKR7'],
        ['PDXDC1','PKD1'],
        ['PEA15','RPS21'],
        ['PEBP1','POLR2A'],
        ['PERP','TUFM'],
        ['PFKL','TNKS1BP1'],
        ['PFKP','SLC7A5'],
        ['PFKP','UBR4'],
        ['PFN1','PLEKHG5'],
        ['PFN1','RPL14'],
        ['PFN1','SLC7A5'],
        ['PFN1','SNRPG'],
        ['PGAM1','RPL23A'],
        ['PGK1','PKM'],
        ['PGK1','TPM4'],
        ['PHPT1','SLC5A6'],
        ['PIGB','RAB27A'],
        ['PILRB','PVRIG'],
        ['PKD1','PLA2G10'],
        ['PKD1','RPL27A'],
        ['PKD1','RPS16'],
        ['PKD1','TUBB'],
        ['PKD1','ZFP36L1'],
        ['PKM','RPS3'],
        ['PKM','TIMP1'],
        ['PLCD3','SRRM2'],
        ['PLEC','PLXNB2'],
        ['PLEC','POM121'],
        ['PLEC','PPIB'],
        ['PLEC','PSMF1'],
        ['PLEC','RPS28'],
        ['PLEC','RPS3'],
        ['PLEC','SDC1'],
        ['PLEC','ZNF217'],
        ['PLEKHH3','ROBO1'],
        ['PLXNB2','RPS27'],
        ['POLE3','TPI1'],
        ['POLR2A','PRRC2A'],
        ['POLR2J2','SPDYE1'],
        ['POLR2J3','SPDYE1'],
        ['POR','WDR75'],
        ['PPCDC','UBE2Z'],
        ['PPHLN1','RPS28'],
        ['PPIA','TRIM8'],
        ['PPIA','YBX1'],
        ['PPIF','ZCCHC24'],
        ['PPP1R15A','THRAP3'],
        ['PPP1R9B','RPS2'],
        ['PPP4C','RPL7A'],
        ['PPP5C','RPL7A'],
        ['PRDM4','PWP1'],
        ['PRELID3B','RAD18'],
        ['PRELP','PRKDC'],
        ['PRELP','ZMYM3'],
        ['PRKCH','TMEM30B'],
        ['PRKCSH','PRPS1'],
        ['PRKDC','RPL37'],
        ['PRKDC','RPL4'],
        ['PRKDC','RPL9'],
        ['PROX2','YLPM1'],
        ['PRRC2A','SH3KBP1'],
        ['PRRC2B','TOMM20'],
        ['PRSS23','RPS2'],
        ['PSAP','RBM39'],
        ['PSAP','RTN4'],
        ['PSAP','S100A11'],
        ['PSAT1','TMBIM6'],
        ['PSMD1','RPL30'],
        ['PSMD13','RPL8'],
        ['PSMD2','RPL10'],
        ['PSMD2','RPL39'],
        ['PSMD2','RPL8'],
        ['PSMD3','RPL5'],
        ['PSMD8','TP53BP1'],
        ['PSPH','SPARC'],
        ['PTBP1','PTRF'],
        ['PTMA','TAF6'],
        ['PTPN14','TMBIM6'],
        ['PTPRF','RPS28'],
        ['PTPRF','RPSA'],
        ['PTPRF','TMSB10'],
        ['QSOX1','RPL32'],
        ['R3HDM4','TIMP1'],
        ['RAB5C','ST13'],
        ['RAB7A','SLC1A5'],
        ['RAB7A','TMBIM6'],
        ['RABAC1','UBR4'],
        ['RABGAP1','RPL7A'],
        ['RABGAP1L','SLC9C2'],
        ['RAD23A','SUMO3'],
        ['RAF1','ZC3H15'],
        ['RAN','TALDO1'],
        ['RASA4B','UBE2D4'],
        ['RBM25','RPL30'],
        ['RFTN2','SLCO1A2'],
        ['RHOA','RPL41'],
        ['RHOC','RPLP0'],
        ['RNF185','TMSB4X'],
        ['RNF19A','SPAG1'],
        ['RNF213','RPL13A'],
        ['RNF213','SLC25A5'],
        ['RNF213','TXNL1'],
        ['RNF31','TPCN1'],
        ['RPL11','WAPL'],
        ['RPL12','U2AF1'],
        ['RPL13','UBA1'],
        ['RPL18A','RPLP2'],
        ['RPL18A','ZNF598'],
        ['RPL19','SLC7A5'],
        ['RPL22','TUBA1B'],
        ['RPL23A','TAGLN2'],
        ['RPL24','RPL6'],
        ['RPL24','RPS15A'],
        ['RPL24','TOP1'],
        ['RPL26','TIMM44'],
        ['RPL3','SMARCC1'],
        ['RPL30','RPS20'],
        ['RPL30','TUBB4B'],
        ['RPL31','TUBA1A'],
        ['RPL31','U2AF2'],
        ['RPL35','RPL8'],
        ['RPL35','RPLP2'],
        ['RPL35','SFPQ'],
        ['RPL35','YBX1'],
        ['RPL36','RPS14'],
        ['RPL37','SZRD1'],
        ['RPL37A','TIMP1'],
        ['RPL37A','TMA7'],
        ['RPL38','RPL6'],
        ['RPL38','TEAD1'],
        ['RPL39L','ST6GAL1'],
        ['RPL4','TCF20'],
        ['RPL4','TGOLN2'],
        ['RPL41','SMARCA4'],
        ['RPL6','SEC61A1'],
        ['RPL7','RPS29'],
        ['RPL7A','SEC61A1'],
        ['RPL7A','SYNGR2'],
        ['RPL8','SAP18'],
        ['RPL8','SDHA'],
        ['RPL8','SURF4'],
        ['RPL8','TOP2A'],
        ['RPL8','UBA3'],
        ['RPL8','UBR4'],
        ['RPL9','SLC7A6'],
        ['RPLP0','SPTAN1'],
        ['RPLP1','TPI1'],
        ['RPLP2','SYNGR2'],
        ['RPN2','TMSB4X'],
        ['RPN2','UBC'],
        ['RPS12','TNPO1'],
        ['RPS16','TINAGL1'],
        ['RPS18','SARS'],
        ['RPS18','SMARCB1'],
        ['RPS2','SNRPD1'],
        ['RPS21','STT3A'],
        ['RPS24','RPS6KA1'],
        ['RPS27','SRSF1'],
        ['RPS28','SNRPF'],
        ['RPS28','SNX3'],
        ['RPS28','TRAM1'],
        ['RPS3','SON'],
        ['RPS3','SPATS2L'],
        ['RPS3','TUBB'],
        ['RPS4X','ZNF207'],
        ['RPS4Y1','TUBB'],
        ['RPS5','TUFM'],
        ['RPS6KA3','SDCCAG3'],
        ['RPS8','RPSA'],
        ['RPS9','SERPINH1'],
        ['RRAGA','SAXO1'],
        ['RRN3','SMG1'],
        ['S100A6','SLC4A2'],
        ['S100A6','SPCS3'],
        ['S100A6','SRP14'],
        ['S100A6','ST14'],
        ['S100A6','TAF15'],
        ['S100A6','TMED9'],
        ['S100A6','UBA52'],
        ['SAMD11','TLN1'],
        ['SAMD4B','SEPHS2'],
        ['SEC14L6','SIRPA'],
        ['SEPT14','ZNF596'],
        ['SEPT2','TUBB'],
        ['SEPT7','TLN1'],
        ['SERBP1','TLN1'],
        ['SETMAR','ZMPSTE24'],
        ['SETX','TRIM28'],
        ['SF3B1','SUZ12'],
        ['SH3D19','TMEM69'],
        ['SHISA9','TENM1'],
        ['SHMT2','UBE3C'],
        ['SLC25A3','SMARCD2'],
        ['SLC25A3','TRIM8'],
        ['SLC25A51','TMX2'],
        ['SLC27A5','ZNF446'],
        ['SLC38A1','TPT1'],
        ['SLC38A1','YBX1'],
        ['SLC39A7','XPO1'],
        ['SLC3A2','VDAC1'],
        ['SMARCC1','XRCC5'],
        ['SNRNP200','YWHAE'],
        ['SNRNP70','SSR2'],
        ['SNRNP70','YBX1'],
        ['SNRPG','TPP2'],
        ['SOD2','WTAP'],
        ['SOX30','THG1L'],
        ['SPINT2','YIF1B'],
        ['SPTAN1','SSR4'],
        ['SPTAN1','TMBIM6'],
        ['SPTAN1','UBE2D2'],
        ['SRPRA','UBC'],
        ['SRRM2','TMED2'],
        ['ST14','TMED2'],
        ['SUPT5H','TUBB4B'],
        ['SWI5','TRUB2'],
        ['SYMPK','WARS'],
        ['TAGLN2','ZNF146'],
        ['TBRG4','TECPR2'],
        ['TDG','TMEM132B'],
        ['TLN1','YKT6'],
        ['TMBIM6','UPF3A'],
        ['TMED2','TUBB4B'],
        ['TNFRSF1A','TPM4'],
        ['TPI1','TXNL4A'],
        ['TPT1','UGGT1'],
        ['TRAF7','YWHAE'],
        ['UPF1','VIMP'],
        ['VCL','WDR1'],
        ['VPS37A','ZNF471'],
        ['SAMD4A','DACT1'],
        ['SCAMP2','WDR72'],
        ['RICTOR','FYB'],
        ['DNHD1','RRP8'],
        ['CLTC','VMP1'],
        ['PRKAR1A','PLCB1'],
        ['SPDYE3','UPK3B'],
        ['BOP1','MROH1'],
        ['NPIPB5','SMG1 '],
        ['RPL39L','ST6GAL1'],
        ['EWSR1','SMCO4'],
        ['PRDM4','PWP1'],
        ['GBA','MTX1'],
        ['SBF1','SRSF2'],
        ['CID','CCDC7']

]



    data = fusions.get(options.organism.lower(),[])
    if data:

        #file_symbols = os.path.join(options.output_directory,'genes_symbols.txt')
        file_symbols = os.path.join(options.output_directory,'synonyms.txt')
        loci = symbols.generate_loci(file_symbols)

        genes = symbols.read_genes_symbols(file_symbols)

        d = []
        for g in data:
            if len(g)>2 or len(g) == 1:
                print "Error:",g
                sys.exit(1)
        for (g1,g2) in data:
            if g1.upper() != g2.upper():
                ens1 = symbols.ensembl(g1.upper(),genes,loci)
                ens2 = symbols.ensembl(g2.upper(),genes,loci)
                if ens1 and ens2:
                    for e1 in ens1:
                        for e2 in ens2:
                            if e1 != e2:
                                d.append([e1,e2])

        data = ['\t'.join(sorted(line)) + '\n' for line in d]
        data = list(set(data))

        print "%d known fusion genes found in manually currated database" % (len(data),)

        if not options.skip_filter_overlap:
            ens2hugo = dict([tuple(line.rstrip('\r\n').split('\t')) for line in file(os.path.join(options.output_directory,'genes_symbols.txt'),'r').readlines() if line.rstrip('\r\n')])

            d1 = []
            overlappings = ['ensembl_fully_overlapping_genes.txt',
                            'ensembl_same_strand_overlapping_genes.txt',
                            'gencode_fully_overlapping_genes.txt',
                            'gencode_same_strand_overlapping_genes.txt',
                            'refseq_fully_overlapping_genes.txt',
                            'refseq_same_strand_overlapping_genes.txt',
                            "ucsc_fully_overlapping_genes.txt",
                            "ucsc_same_strand_overlapping_genes.txt",
                            'pairs_pseudogenes.txt',
                            'banned.txt',
                            'dgd.txt',
                            'healthy.txt',
                            'paralogs.txt']
            ensembls = set(['ensembl_fully_overlapping_genes.txt',
                            'ensembl_same_strand_overlapping_genes.txt'])
            ens = []
            for ov in overlappings:
                p = os.path.join(options.output_directory,ov)
                print "Parsing file:",p
                d2 = []
                if os.path.isfile(p):
                    d2 = sorted(set([tuple(sorted(line.rstrip('\r\n').split('\t'))) for line in file(p,'r').readlines() if line.rstrip('\r\n')]))
                    d3 = set(['\t'.join(line)+'\n' for line in d2])
                    d4 = sorted([line for line in data if line in d3])
                    d4 = [line.rstrip('\r\n').split('\t') for line in d4]
                    d4 = [line+[ens2hugo.get(line[0],'')]+[ens2hugo.get(line[1],'')] for line in d4]
                    d4 = ['\t'.join(line)+'\n' for line in d4]
                    file(os.path.join(options.output_directory,"ccle3___%s"%(ov,)),'w').writelines(d4)
                if ov in ensembls:
                    ens.extend(d2)
                d1.extend(d2)
            d = set()
            for line in d1:
                (a,b) = (line[0],line[1])
                if a > b:
                    (a,b) = (b,a)
                d.add("%s\t%s\n" % (a,b))
            ens = set(['\t'.join(line)+'\n' for line in ens])
            ensembl = [line for line in data if line in ens]
            file(os.path.join(options.output_directory,'ccle3___ensembl.txt'),'w').writelines(sorted(ensembl))
            skipped = [line for line in data if line in d]
            data = [line for line in data if line not in d]
            file(os.path.join(options.output_directory,'ccle3___all.txt'),'w').writelines(sorted(skipped))

            print "%d known fusion genes left after removing the overlappings" % (len(data),)

    file(os.path.join(options.output_directory,'ccle3.txt'),'w').writelines(data)
    #
