#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of candidate fusion genes found in healthy samples. This list is hard coded inhere.



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of candidate fusion genes found in healthy samples."""
    version = "%prog 0.10 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of allowed candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of allowed candidate fusion genes is generated. Default is '%default'.""")


    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    print "Generating the list of candidate fusion genes found in healthy samples..."
    fusions = dict()

    fusions['rattus_norvegicus'] = []

    fusions['mus_musculus'] = [
        ['A930011G23RIK','RASGEF1B'],
        ['Ly6a','Ly6c1'],
        ['Anks1b','AC132131.1'],
        ['Sema4d','Gm15440'],
        ['Fam48a','FAM48A'],
        ['Fbxo21','Gm5403'],
        ['Tonsl','Vps28'],
        ['Prpf19','Gm7701'],
        ['Zc3h11a','Zbed6'],
        ['Mical3','Mical3'],
        ['Sez6l2','RP23-142A14.3'],
        ['Hnrnpc','Gm12696'],
        ['Kctd13','RP23-142A14.3'],
        ['Rgs10','Gm15503'],
        ['Xpot','RP24-215A14.2'],
        ['Gprasp1','Gprasp2'],
        ['Galntl6','AC116875.1'],
        ['Nolc1','Gm5124'],
        ['Gria2','AC163349.1'],
        ['Ipo11','Lrrc70'],
        ['Gprasp1','Armcx5'],
        ['Celf2','Gm13390'],
        ['Flywch1','AC122821.1'],
        ['2610021A01Rik','Zfp141'],
        ['SBSN','C630016N16RIK'],
        ['GM16052','GM16049'],
        ['CPLX2','KIF5A'],
        ['CNRIP1','WDR92'],
        ['4922501L14RIK','BC007180'],
        ['PAM','B230216N24RIK'],
        ['ARF3','SPOCK2'],
        ['THTPA','GM20687'],
        ['PRKAR1B','GM15672'],
        ['ANKS1B','AC132131.1'],
        ['LRRC24','C030006K11RIK'],
        ['SIAH2','FAM194A'],
        ['SFPQ','GM12940'],
        ['CAMK2A','JPH3'],
        ['4933434E20RIK','1700094D03RIK'],
        ['CPLX2','SPARCL1'],
        ['SPARCL1','KIF5A'],
        ['MEG3','ARF3'],
        ['OBSL1','CHPF'],
        ['NFS1','RBM12'],
        ['PARK2','CT009575.1'],
        ['TAF9','TAF9'],
        ['CAMK2A','ARF3'],
        ['RRAGC','MYCBP'],
        ['NDRG4','ATP1A3'],
        ['NRXN1','CAMK2A'],
        ['ACSS2','GM14256'],
        ['CALM1','NPTXR'],
        ['PPP2R4','GM16323'],
        ['APP','KIF5A'],
        ['GM9769','PTGES3'],
        ['SNAP25','VSNL1'],
        ['ARPC5L','WDR38'],
        ['FAM107A','HPCA'],
        ['SNAP25','HPCA'],
        ['CFH','GM5834'],
        ['ZFP617','ZFP882'],
        ['RPL37','GM13826'],
        ['MRI1','D8ERTD738E'],
        ['CAMK2A','ATP1A3'],
        ['MKLN1','2210408F21RIK'],
        ['CLIP2','SYNA'],
        ['ZFP672','GM9900'],
        ['LARP4','RP24-218M19.2'],
        ['MEG3','STX1B'],
        ['TONSL','VPS28'],
        ['2810055G20RIK','2810055G20RIK'],
        ['TRIM17','HIST3H2A'],
        ['IRGQ','ZFP428'],
        ['MYG1','PFDN5'],
        ['PRPF19','GM7701'],
        ['2310028H24RIK','ARF3'],
        ['KCTD13','RP23-142A14.3'],
        ['MOBP','MBP'],
        ['RPS24','GM8292'],
        ['NAT15','GM20695'],
        ['BC022960','GM15246'],
        ['CSNK1G3','GM14584'],
        ['NEUROD1','CERKL'],
        ['MLF2','A230083G16RIK'],
        ['FAM107A','SNAP25'],
        ['RASGEF1B','A930011G23RIK'],
        ['VAMP2','ARF3'],
        ['GRIA2','SNHG11'],
        ['ZFP414','GM20507'],
        ['BC061194','GM13334'],
        ['1700054N08RIK','2810455O05RIK'],
        ['SMARCC1','GM10615'],
        ['CHGB','1110034G24RIK'],
        ['GM14169','GM14174'],
        ['9330159F19RIK','6330407J23RIK'],
        ['GTF2H5','TULP4'],
        ['ARHGEF17','MIR3102'],
        ['UPP2','GM13558'],
        ['PTER','C1QL3'],
        ['ARSG','AMZ2'],
        ['CAMK2A','SNAP25'],
        ['NDUFB9','RNF139'],
        ['CCNJL','4933415A04RIK'],
        ['TAPBP','ZBTB22'],
        ['RHOT2','WDR90'],
        ['SEZ6L2','RP23-142A14.3'],
        ['CPLX2','SNAP25'],
        ['CPNE1','RBM12'],
        ['AXIN2','CCDC46'],
        ['CAMK2A','SLC6A17'],
        ['APBB3','SRA1'],
        ['MICAL3','MICAL3'],
        ['MAP3K13','AC130214.1'],
        ['CDS2','AV099323'],
        ['VOPP1','PIGY'],
        ['LZTS1','AC114995.1'],
        ['SNHG11','CPSF6'],
        ['FLYWCH1','AC122821.1'],
        ['RAB43','ISY1'],
        ['GRIA2','AC163349.1'],
        ['USP9X','5730405O15RIK'],
        ['AGAP2','KIF5A'],
        ['SNHG11','SPNB3'],
        ['CAMK2A','HPCA'],
        ['GM16286','TXNL4A'],
        ['TRABD','PANX2'],
        ['TBC1D24','NTN3'],
        ['ATP1A2','ATP1A3'],
        ['ASB3','GPR75'],
        ['HSP90AB1','CPLX2'],
        ['RIN2','BC039771'],
        ['PPP1R9B','DDN'],
        ['STOX2','AC102862.2'],
        ['SCHIP1','IQCJ'],
        ['C330006D17RIK','BIRC2'],
        ['B230314M03RIK','B230314M03RIK'],
        ['CX3CL1','CCL17'],
        ['SRCIN1','GM11611'],
        ['NCBP2','PIGZ'],
        ['RDH5','BLOC1S1'],
        ['PPP3R1','CNRIP1'],
        ['SASH1','SAMD5'],
        ['MEIS2','2700033N17RIK'],
        ['CPE','KIF5A'],
        ['RCC1','SNHG3'],
        ['TSR1','SNORD91'],
        ['4932411E22RIK','ANKFN1'],
        ['SETD3','GM14026'],
        ['CALM1','CALM3'],
        ['GM1976','GM10324'],
        ['FBXO44','FBXO6'],
        ['KIF5C','KIF5A'],
        ['CAMK2A','DDN'],
        ['KIF17','SH2D5'],
        ['SPAG7','CAMTA2'],
        ['NDUFA13','YJEFN3'],
        ['BZW1','GM11652'],
        ['MBP','KIF5A'],
        ['RBM26','GM17066'],
        ['GPRASP1','ARMCX5'],
        ['PVALB','IFT27'],
        ['KIF1A','CAMK2A'],
        ['STX1B','ARF3'],
        ['SORT1','GM12525'],
        ['1700003E16RIK','DCTN1'],
        ['NOLC1','GM5124'],
        ['RNF5','ARF3'],
        ['1810022K09RIK','GM4540'],
        ['PHB2','SNOU89'],
        ['RPL32','SNORA7A'],
        ['1500011B03RIK','2610524H06RIK'],
        ['CAAA01154301.1','CAAA01217562.1'],
        ['PARK2','PARK2'],
        ['RABGAP1','GPR21'],
        ['PFKM','RP23-268C9.3'],
        ['RPS21','MIR3091'],
        ['FAM53B','METTL10'],
        ['FAM48A','FAM48A'],
        ['TMEFF2','SDPR'],
        ['ACTB','LRRC58'],
        ['STXBP1','ATP1A3'],
        ['SEMA4D','GM15440'],
        ['SLC1A2','ARF3'],
        ['PYGO1','DYX1C1'],
        ['SEZ6L2','KCTD13'],
        ['NKAIN2','NKAIN2'],
        ['XPOT','RP24-215A14.2'],
        ['GALNTL6','AC116875.1'],
        ['PLEKHA2','HTRA4'],
        ['FAM198A','1700048O20RIK'],
        ['SNHG11','SRSF11'],
        ['HNRNPC','GM12696'],
        ['ZC3H11A','ZBED6'],
        ['CELF2','GM13390'],
        ['CAMK2A','CAMK2B'],
        ['AA986860','YOD1'],
        ['ARF3','PNMAL2'],
        ['TRAF7','RAB26'],
        ['RALYL','RALYL'],
        ['PISD-PS2','PISD-PS1'],
        ['NOS1AP','GM7694'],
        ['TAOK2','TMEM219'],
        ['SDCCAG3','GM12221'],
        ['DGKG','SNAP25'],
        ['CPE','PPP1R9B'],
        ['RGS10','GM15503'],
        ['MAMSTR','RASIP1'],
        ['AC125279.1','AC125279.2'],
        ['LY6A','LY6C1'],
        ['FRS3','GM14872'],
        ['NCALD','GM15941'],
        ['TTF1','GM14132'],
        ['PYGO1','GM20510'],
        ['PSPC1','AC154731.1'],
        ['RBM4','RBM4B'],
        ['ARCN1','GM8752'],
        ['FBXO21','GM5403'],
        ['NOP56','SNORA51'],
        ['HYAL1','HYAL3'],
        ['IRF9','RNF31'],
        ['SRCRB4D','YWHAG'],
        ['NRGN','SORT1'],
        ['MEIS2','ARF3'],
        ['NEDD4','SNHG11'],
        ['GPRASP1','GPRASP2'],
        ['HSP90AB1','SPARCL1'],
        ['ETFB','VSIG10L'],
        ['IPO11','LRRC70'],
        ['FAM82A1','4921513D11RIK'],
        ['MKLN1','GM14532'],
        ['ACSL3','UTP14B'],
        ['TCTE2','GM16049'],
        ['AAR2','GM14168'],
        ['PTMS','KIF5A'],
        ['SNHG11','AASDH'],
        ['2610021A01RIK','ZFP141'],
        ['GNAI3','GPR61'],
        ['CCDC136','AC044807.1'],
        ['KCTD12','MIR5130'],
        ['FAM107A','CAMK2A'],
        ['CAMK2A','CPLX2'],
        ['PIGQ','NHLRC4'],
        ['SLC1A2','CAMK2A'],
        ['WDR45L','GM6305'],
        ['FAM120B','AC154378.1'],
        ['CPXM1','PCED1A'],
        ['MATR3','AC121821.1'],
        ['HIRA','GM15797'],
        ['EIF4A1','EIF4A-PS4'],
        ['NAT15','CLUAP1'],
        ['9830147E19RIK','GM7221'],
        ['BRDT','EPHX4'],
        ['WDR77','OVGP1'],
        ['PCDHB21','PCDHB22'],
        ['GM20696','EIF4E3'],
        ['GNB1L','GM16314'],
        ['ARF3','SPOP'],
        ['HYAL3','NAT6'],
        ['KCTD16','2900055J20RIK'],
        ['FTL','TIMP1'],
        ['MFGE8','HAPLN3'],
        ['CLCF1','POLD4'],
        ['CTBS','GNG5'],
        ['MFSD7','ATP5I'],
        ['CLCF1','AP003419.11'],
        ['NOP58','RP11-430C1.2'],
        ['IGFBP3','COL1A2'],
        ['TIMP1','FN1'],
        ['SPARC','IGFBP3'],
        ['C11ORF73','AC007229.3'],
        ['SERPINE1','COL1A1'],
        ['GPX1P1','GPX1'],
        ['COL1A1','TGFBI'],
        ['NACA','NACAP1'],
        ['HSPB6','COL1A1'],
        ['COL1A1','COL6A2'],
        ['NOTCH2','RP11-458D21.5'],
        ['COL1A1','MFGE8'],
        ['ACTN1','HMGN1'],
        ['LGALS1','COL1A1'],
        ['COL1A1','SPARC'],
        ['MTCH2','RP11-575C1.1'],
        ['POLD4','CLCF1'],
        ['POMZP3','ZP3'],
        ['NOTCH2','NOTCH2NL'],
        ['RNU2-2P','RNU2-59P'],
        ['CCDC80','RP11-572C15.6'],
        ['NDUFA11','AC024592.12'],
        ['RP11-423H2.3','RP11-1252I4.2'],
        ['ATP5J2','ATP5J2-PTCD1'],
        ['POLD4','AP003419.11'],
        ['ACTB','COL1A1'],
        ['PTCD1','ATP5J2'],
        ['C8ORF76','ZHX1-C8ORF76'],
        ['PET100','CTD-3214H19.4'],
        ['COL1A1','IGFBP3'],
        ['LGALS1','IGFBP3'],
        ['COL1A1','TAGLN'],
        ['COL1A1','COL6A1'],
        ['TIMP3','COL1A1'],
        ['WDR74','RNU2-59P'],
        ['COL1A1','SH3BGRL3'],
        ['TMSB10','COL6A1'],
        ['FN1','TPM2'],
        ['IGFBP3','TPM2'],
        ['COL6A2','COL1A2'],
        ['HDGF','HDGFP1'],
        ['RP11-773D16.1','RP11-506M13.3'],
        ['COL1A1','PCSK7'],
        ['IGFBP3','S100A6'],
        ['TCF25','RP11-566K11.7'],
        ['COL1A2','ACTG1'],
        ['COL1A1','S100A6'],
        ['COL1A1','GAPDH'],
        ['HNRNPA1','HNRNPA1P48'],
        ['HAPLN3','MFGE8'],
        ['PDCD5','RP11-466P24.4'],
        ['SF3B4','FAM71D'],
        ['MMP24-AS1','RP4-614O4.11'],
        ['FN1','IGFBP3'],
        ['TIMP1','COL1A1'],
        ['MIR199A2','DNM3OS'],
        ['C19ORF10','AC005339.2'],
        ['IGFBP3','PTRF'],
        ['COL1A2','S100A6'],
        ['RPS19','RPS19P1'],
        ['PKM','COL1A1'],
        ['COL1A1','TPM2'],
        ['MYL9','COL1A1']
    ]

    fusions['canis_lupus_familiaris'] = [


    ]


    # manual curation from papers
    fusions['homo_sapiens'] = [
        ['7SK','ACTR5'],
        ['7SK','CLK4'],
        ['7SK','COG7'],
        ['7SK','CTC-498J12.1'],
        ['7SK','GSDMD'],
        ['A2M','ABCA3'],
        ['A2M','AC096579.7'],
        ['A2M','ACTB'],
        ['A2M','ACTG1'],
        ['A2M','AEBP1'],
        ['A2M','AGER'],
        ['A2M','AHNAK'],
        ['A2M','ALB'],
        ['A2M','AP2M1'],
        ['DUS4L','BCAP29'],
        ['A2M','APLP2'],
        ['A2M','APP'],
        ['A2M','AQP1'],
        ['A2M','ATP1A1'],
        ['A2M','B2M'],
        ['A2M','C3'],
        ['A2M','CCDC152'],
        ['A2M','CD34'],
        ['A2M','CD74'],
        ['A2M','CD93'],
        ['A2M','CLU'],
        ['A2M','COL1A1'],
        ['A2M','COL1A2'],
        ['A2M','COL3A1'],
        ['A2M','COL4A2'],
        ['A2M','COL6A2'],
        ['A2M','COL6A3'],
        ['A2M','CRP'],
        ['A2M','CTSD'],
        ['A2M','DUSP1'],
        ['A2M','EEF2'],
        ['A2M','EMP2'],
        ['A2M','EPAS1'],
        ['A2M','FGG'],
        ['A2M','FLNA'],
        ['A2M','FN1'],
        ['A2M','FOS'],
        ['A2M','FOSB'],
        ['A2M','FTL'],
        ['A2M','GNB1'],
        ['A2M','GNB2L1'],
        ['A2M','GSN'],
        ['A2M','HLA-A'],
        ['A2M','HLA-B'],
        ['A2M','HLA-C'],
        ['A2M','HLA-E'],
        ['A2M','HP'],
        ['A2M','HSPG2'],
        ['A2M','IGHA1'],
        ['A2M','IGKC'],
        ['A2M','JUNB'],
        ['A2M','LRP1'],
        ['A2M','LTBP4'],
        ['A2M','MACF1'],
        ['A2M','MALAT1'],
        ['A2M','MCL1'],
        ['A2M','MYH9'],
        ['A2M','NAPSA'],
        ['A2M','NCOR2'],
        ['A2M','NEAT1'],
        ['A2M','NR4A1'],
        ['A2M','PKN1'],
        ['A2M','PLXNB2'],
        ['A2M','PSAP'],
        ['A2M','PTRF'],
        ['A2M','PZP'],
        ['A2M','RNASE1'],
        ['A2M','RPL3'],
        ['A2M','SERPINA1'],
        ['A2M','SFTPA1'],
        ['A2M','SFTPA2'],
        ['A2M','SFTPB'],
        ['A2M','SFTPC'],
        ['A2M','SLC34A2'],
        ['A2M','SMEK2'],
        ['A2M','SPARC'],
        ['A2M','SPTAN1'],
        ['A2M','SPTBN1'],
        ['A2M','SRRM2'],
        ['A2M','STAB1'],
        ['A2M','TGM2'],
        ['A2M','THBS1'],
        ['A2M','TIMP2'],
        ['A2M','TIMP3'],
        ['A2M','TNS1'],
        ['A2M','TPM4'],
        ['A2M','VEGFA'],
        ['A2M','VIM'],
        ['A2M','VWF'],
        ['A2M','ZFP36'],
        ['AAK1','AC114772.1'],
        ['AAK1','RP11-427H3.3'],
        ['AAMP','COL1A1'],
        ['AARS','COL1A1'],
        ['AATK','SFTPC'],
        ['AB015752.3','AB015752.4'],
        ['AB015752.3','LINC01030'],
        ['AB019441.29','RPS8'],
        ['ABAT','METTL22'],
        ['ABCA2','FN1'],
        ['ABCA3','AKAP2'],
        ['ABCA3','CD74'],
        ['ABCA3','CTSD'],
        ['ABCA3','EPAS1'],
        ['ABCA3','FLNA'],
        ['ABCA3','PALM2-AKAP2'],
        ['ABCA3','SFTPA1'],
        ['ABCA3','SFTPA2'],
        ['ABCA3','SFTPB'],
        ['ABCA3','SFTPC'],
        ['ABCA3','SLC34A2'],
        ['ABCB8','ASIC3'],
        ['ABCC1','FN1'],
        ['ABCD4','AC005519.4'],
        ['ABCF1','MIR877'],
        ['ABCF2','COL1A1'],
        ['ABCF2','EHD2'],
        ['ABHD12','COL1A1'],
        ['ABHD15-AS1','INTS7'],
        ['ABHD16A','LY6G5C'],
        ['ABHD17C','CTGF'],
        ['ABHD17C','IGFBP4'],
        ['ABHD2','CNN2'],
        ['ABHD2','COL1A1'],
        ['ABHD2','EHD2'],
        ['ABHD2','FN1'],
        ['ABHD2','SFTPA2'],
        ['ABHD2','SFTPB'],
        ['ABHD2','SFTPC'],
        ['ABI2','FN1'],
        ['ABI3BP','COL1A1'],
        ['ABI3BP','FN1'],
        ['ABL1','COL1A1'],
        ['ABL1','COL1A2'],
        ['ABL1','FN1'],
        ['ABLIM1','SFTPA2'],
        ['ABLIM1','SFTPC'],
        ['ABLIM1','TG'],
        ['ABR','COL1A1'],
        ['AC000089.3','RPL7A'],
        ['AC002075.4','RPS27'],
        ['AC002398.12','ACTG1'],
        ['AC002398.12','CALR'],
        ['AC002398.12','CD151'],
        ['AC002398.12','COL1A1'],
        ['AC002398.12','COL1A2'],
        ['AC002398.12','FLNA'],
        ['AC002398.12','FLNC'],
        ['AC002398.12','FN1'],
        ['AC002398.12','IGFBP5'],
        ['AC002398.12','MYH7'],
        ['AC002398.12','MYH9'],
        ['AC002398.12','NEAT1'],
        ['AC002398.12','PTMS'],
        ['AC002398.12','PTRF'],
        ['AC002398.12','TG'],
        ['AC002398.12','TNNT2'],
        ['AC002398.12','TPM2'],
        ['AC002467.7','CBLL1'],
        ['AC002519.6','PPP2CB'],
        ['AC002550.5','IQCK'],
        ['AC002985.3','COL1A1'],
        ['AC002985.3','COPE'],
        ['AC003002.6','ZNF17'],
        ['AC004410.3','SPPL2B'],
        ['AC004453.8','RPL32'],
        ['AC004692.5','RPL18A'],
        ['AC004893.11','TRRAP'],
        ['AC005037.6','MTND4P23'],
        ['AC005062.2','MACC1'],
        ['AC005077.12','LINC00174'],
        ['AC005152.3','RP11-84E24.2'],
        ['AC005152.3','SOX9-AS1'],
        ['AC005154.6','GGCT'],
        ['AC005251.3','RPL23'],
        ['AC005253.4','COL1A1'],
        ['AC005336.4','IPMKP1'],
        ['AC005339.2','C19ORF10'],
        ['AC005339.2','MYDGF'],
        ['AC005534.6','RPL26'],
        ['AC005540.3','GALNT5'],
        ['AC005540.3','LTBP3'],
        ['AC005540.3','RPL14'],
        ['AC005540.3','SRPX'],
        ['AC005562.1','LRRC37BP1'],
        ['AC005562.1','SMURF2P1'],
        ['AC005682.7','RPL12'],
        ['AC005895.3','PDGFRB'],
        ['AC005943.4','MBD3'],
        ['AC005944.2','PLEC'],
        ['AC005944.2','SFTPA2'],
        ['AC005944.2','SFTPC'],
        ['AC006129.2','CEACAM21'],
        ['AC006153.3','CLDN12'],
        ['AC006153.3','CTB-13L3.1'],
        ['AC006272.1','ZNF577'],
        ['AC006277.2','ZNF77'],
        ['AC006946.16','CECR7'],
        ['AC007036.5','MTURN'],
        ['AC007038.7','RPE'],
        ['AC007228.9','TOB2'],
        ['AC007229.3','C11ORF73'],
        ['AC007246.3','TMEM178A'],
        ['AC007249.3','RP11-320M2.1'],
        ['AC007283.5','CFLAR'],
        ['AC007390.5','AC007390.6'],
        ['AC007390.5','RP11-423P10.2'],
        ['AC007405.6','ERICH2'],
        ['AC007557.1','AC007557.3'],
        ['AC007566.10','GATAD1'],
        ['AC007969.5','RPS15'],
        ['AC008268.2','GPAT2'],
        ['AC008279.1','RPL5'],
        ['AC008280.5','ACYP2'],
        ['AC008427.2','MFF'],
        ['AC008440.10','MYADM'],
        ['AC008440.5','CALR'],
        ['AC008440.5','COL1A1'],
        ['AC008440.5','HLA-E'],
        ['AC008440.5','RP11-626G3.1'],
        ['AC008440.5','SFTPC'],
        ['AC008984.2','LILRA6'],
        ['AC008984.2','LILRB3'],
        ['AC009005.3','FN1'],
        ['AC009005.3','SFTPA2'],
        ['AC009133.12','AC009133.20'],
        ['AC009133.14','COL1A1'],
        ['AC009133.14','COL6A2'],
        ['AC009133.14','FLNA'],
        ['AC009133.14','FN1'],
        ['AC009133.14','IGFBP2'],
        ['AC009228.1','FAM228A'],
        ['AC009362.2','RPL27'],
        ['AC009413.2','RPS12'],
        ['AC009469.1','SFTPA1'],
        ['AC009469.1','SFTPA2'],
        ['AC009469.1','SFTPC'],
        ['AC009469.1','TGM2'],
        ['AC009487.6','AHCTF1'],
        ['AC009487.7','AHCTF1'],
        ['AC009951.1','ZEB2'],
        ['AC009961.3','BAZ2B'],
        ['AC010095.6','COX7B'],
        ['AC010170.1','GNAS'],
        ['AC010525.6','AC010525.7'],
        ['AC010525.6','EPN1'],
        ['AC010677.5','RPL23'],
        ['AC010878.3','STRAP'],
        ['AC010982.1','AC110769.3'],
        ['AC010982.1','ACTR3'],
        ['AC011513.3','CEACAM1'],
        ['AC011513.3','CEACAM6'],
        ['AC011530.4','DMPK'],
        ['AC011558.5','COL1A1'],
        ['AC011558.5','CTSB'],
        ['AC011558.5','FN1'],
        ['AC011558.5','LEPREL4'],
        ['AC011558.5','PTRF'],
        ['AC011558.5','SFTPB'],
        ['AC011558.5','TRAM2'],
        ['AC012309.5','LINC00665'],
        ['AC012358.8','PRORSD1P'],
        ['AC012363.10','MTND3P10'],
        ['AC012363.10','MTND4P26'],
        ['AC012379.1','FN1'],
        ['AC012442.5','CHCHD5'],
        ['AC012485.2','PER2'],
        ['AC012615.1','PTRF'],
        ['AC012615.1','TPM4'],
        ['AC013394.2','CHD2'],
        ['AC013470.6','SSB'],
        ['AC013717.3','RPL12'],
        ['AC015936.3','GFAP'],
        ['AC016586.1','ZBTB7A'],
        ['AC016683.6','RP11-65I12.1'],
        ['AC016708.2','RPL10'],
        ['AC016712.2','RPL31'],
        ['AC016717.1','NYAP2'],
        ['AC016722.2','SOCS5'],
        ['AC016730.1','AC092635.1'],
        ['AC016739.2','RPLP1'],
        ['AC016773.1','TACC3'],
        ['AC017060.1','AC019117.2'],
        ['AC017116.11','DBNL'],
        ['AC018470.4','SP9'],
        ['AC018730.3','AC018730.4'],
        ['AC018730.4','LINC01159'],
        ['AC018766.4','AC018766.5'],
        ['AC018766.4','PNKP'],
        ['AC018766.4','PTOV1-AS2'],
        ['AC019172.2','DCDC2C'],
        ['AC019349.5','FLNA'],
        ['AC020956.3','LYPD4'],
        ['AC021016.7','RPL19'],
        ['AC022153.1','LLNLF-65H9.1'],
        ['AC022182.2','NASP'],
        ['AC022210.2','RPS2'],
        ['AC024560.3','LINC00969'],
        ['AC024560.3','SDHAP1'],
        ['AC024560.3','SDHAP2'],
        ['AC024592.12','NDUFA11'],
        ['AC024704.2','AC104653.1'],
        ['AC027612.6','RP11-14N7.2'],
        ['AC037459.4','COL1A1'],
        ['AC037459.4','PDLIM2'],
        ['AC053503.6','MYH7'],
        ['AC053503.6','TG'],
        ['AC055811.1','MPRIP'],
        ['AC058791.1','AC058791.2'],
        ['AC058791.1','LINC-PINT'],
        ['AC063976.6','P4HA2'],
        ['AC067945.3','RAB1A'],
        ['AC068491.3','PAFAH1B1'],
        ['AC068580.5','CTSD'],
        ['AC068831.6','CERK'],
        ['AC068831.8','RCCD1'],
        ['AC069213.1','LINC00969'],
        ['AC069213.1','MUC20'],
        ['AC069363.1','CCL4'],
        ['AC069368.3','ANKDD1A'],
        ['AC069513.3','SDHAP1'],
        ['AC073046.25','TET3'],
        ['AC073063.10','RP11-762I7.5'],
        ['AC073063.10','SARNP'],
        ['AC073343.13','ZNF316'],
        ['AC073869.1','RAB6A'],
        ['AC073869.20','NOC2L'],
        ['AC074183.3','SNHG7'],
        ['AC074183.4','SNHG7'],
        ['AC074286.1','NFE2L2'],
        ['AC074363.1','B3GALT1'],
        ['AC078883.3','AC093818.1'],
        ['AC078899.1','ACTR2'],
        ['AC078899.3','ZNF737'],
        ['AC079210.1','PRKAR1A'],
        ['AC079233.1','GRIA2'],
        ['AC079250.1','RPL18A'],
        ['AC079922.2','RPS20'],
        ['AC083862.6','RNF14'],
        ['AC083900.1','PLEKHM3'],
        ['AC084018.1','RHOF'],
        ['AC084018.1','RP11-347I19.3'],
        ['AC084082.3','TRIM55'],
        ['AC084117.3','LDHA'],
        ['AC087650.1','RPL29'],
        ['AC090587.2','PGAP2'],
        ['AC090774.2','CCDC144NL-AS1'],
        ['AC092431.1','ANXA4'],
        ['AC092610.12','RPL22'],
        ['AC092798.2','RPL32'],
        ['AC092933.3','RPS29'],
        ['AC093063.2','LGALS17A'],
        ['AC093106.5','NDUFB9'],
        ['AC093142.2','RPL34'],
        ['AC093162.5','RETSAT'],
        ['AC093484.4','TRPV2'],
        ['AC093495.4','RP11-434D12.1'],
        ['AC093590.1','POLR3F'],
        ['AC093642.5','FBXO25'],
        ['AC093673.5','DAP'],
        ['AC093698.2','RPL12'],
        ['AC093734.1','FN1'],
        ['AC093734.13','TMEM184A'],
        ['AC093838.4','SMPD4'],
        ['AC096579.13','B2M'],
        ['AC096579.13','CD74'],
        ['AC096579.13','COL1A1'],
        ['AC096579.13','COL3A1'],
        ['AC096579.13','FLNA'],
        ['AC096579.13','IGHA1'],
        ['AC096579.13','IGHM'],
        ['AC096579.13','IGKV1-16'],
        ['AC096579.13','IGKV1-17'],
        ['AC096579.13','IGKV1-27'],
        ['AC096579.13','IGKV1-5'],
        ['AC096579.13','IGKV1-6'],
        ['AC096579.13','IGKV1-8'],
        ['AC096579.13','IGKV1-9'],
        ['AC096579.13','IGKV1D-13'],
        ['AC096579.13','IGKV1D-16'],
        ['AC096579.13','IGKV1D-17'],
        ['AC096579.13','IGKV1D-43'],
        ['AC096579.13','IGKV1D-8'],
        ['AC096579.13','IGKV2-24'],
        ['AC096579.13','IGKV2D-24'],
        ['AC096579.13','IGKV2D-29'],
        ['AC096579.13','IGKV3-11'],
        ['AC096579.13','IGKV3-20'],
        ['AC096579.13','IGKV3D-11'],
        ['AC096579.13','IGKV3D-15'],
        ['AC096579.13','IGKV3D-20'],
        ['AC096579.13','IGKV4-1'],
        ['AC096579.13','IGKV5-2'],
        ['AC096579.13','IGKV6D-21'],
        ['AC096579.13','NEAT1'],
        ['AC096579.13','PSAP'],
        ['AC096579.13','SFTPA2'],
        ['AC096579.13','SPARC'],
        ['AC096579.7','AHNAK'],
        ['AC096579.7','B2M'],
        ['AC096579.7','BGN'],
        ['AC096579.7','CCL21'],
        ['AC096579.7','CD74'],
        ['AC096579.7','COL1A1'],
        ['AC096579.7','COL3A1'],
        ['AC096579.7','COL4A2'],
        ['AC096579.7','COL6A3'],
        ['AC096579.7','CTSD'],
        ['AC096579.7','EEF2'],
        ['AC096579.7','EGR1'],
        ['AC096579.7','EPAS1'],
        ['AC096579.7','FBLN1'],
        ['AC096579.7','FLNA'],
        ['AC096579.7','FN1'],
        ['AC096579.7','FOSB'],
        ['AC096579.7','FTL'],
        ['AC096579.7','GPRC5A'],
        ['AC096579.7','HLA-A'],
        ['AC096579.7','HLA-B'],
        ['AC096579.7','HLA-DRA'],
        ['AC096579.7','HLA-E'],
        ['AC096579.7','HSPG2'],
        ['AC096579.7','ICAM1'],
        ['AC096579.7','IGHA1'],
        ['AC096579.7','IGHM'],
        ['AC096579.7','IGKV1-12'],
        ['AC096579.7','IGKV1-16'],
        ['AC096579.7','IGKV1-17'],
        ['AC096579.7','IGKV1-27'],
        ['AC096579.7','IGKV1-5'],
        ['AC096579.7','IGKV1-6'],
        ['AC096579.7','IGKV1-8'],
        ['AC096579.7','IGKV1-9'],
        ['AC096579.7','IGKV1D-12'],
        ['AC096579.7','IGKV1D-13'],
        ['AC096579.7','IGKV1D-16'],
        ['AC096579.7','IGKV1D-17'],
        ['AC096579.7','IGKV1D-43'],
        ['AC096579.7','IGKV1D-8'],
        ['AC096579.7','IGKV2-24'],
        ['AC096579.7','IGKV2-26'],
        ['AC096579.7','IGKV2-29'],
        ['AC096579.7','IGKV2-30'],
        ['AC096579.7','IGKV2D-24'],
        ['AC096579.7','IGKV2D-29'],
        ['AC096579.7','IGKV2D-30'],
        ['AC096579.7','IGKV3-11'],
        ['AC096579.7','IGKV3-15'],
        ['AC096579.7','IGKV3-20'],
        ['AC096579.7','IGKV3D-11'],
        ['AC096579.7','IGKV3D-15'],
        ['AC096579.7','IGKV3D-20'],
        ['AC096579.7','IGKV4-1'],
        ['AC096579.7','IGKV5-2'],
        ['AC096579.7','IGKV6-21'],
        ['AC096579.7','IGKV6D-21'],
        ['AC096579.7','LMNA'],
        ['AC096579.7','LRP1'],
        ['AC096579.7','MALAT1'],
        ['AC096579.7','MUC4'],
        ['AC096579.7','MYH9'],
        ['AC096579.7','NEAT1'],
        ['AC096579.7','NFIB'],
        ['AC096579.7','NR4A1'],
        ['AC096579.7','PSAP'],
        ['AC096579.7','PTMS'],
        ['AC096579.7','RP11-1100L3.8'],
        ['AC096579.7','RP11-295K3.1'],
        ['AC096579.7','SCGB1A1'],
        ['AC096579.7','SFTPA1'],
        ['AC096579.7','SFTPA2'],
        ['AC096579.7','SFTPB'],
        ['AC096579.7','SFTPC'],
        ['AC096579.7','SLC34A2'],
        ['AC096579.7','SOCS3'],
        ['AC096579.7','SPARC'],
        ['AC096579.7','SRRM2'],
        ['AC096579.7','VIM'],
        ['AC096579.7','ZFP36'],
        ['AC098831.4','PYURF'],
        ['AC099522.1','CAPNS1'],
        ['AC099522.1','COL5A1'],
        ['AC099522.1','COL5A3'],
        ['AC099522.1','GNAS'],
        ['AC099522.1','PTP4A2'],
        ['AC099522.1','SLC7A5'],
        ['AC099522.1','TXNDC5'],
        ['AC099668.5','MST1'],
        ['AC104135.3','POLE4'],
        ['AC104297.1','PPA1'],
        ['AC104306.1','RPL31'],
        ['AC104698.1','SFTPA2'],
        ['AC105053.3','ATOH8'],
        ['AC105342.1','ANKRD11'],
        ['AC106782.20','BCL7C'],
        ['AC108479.3','KRCC1'],
        ['AC108488.4','RNASEH1'],
        ['AC108938.5','LIMS1'],
        ['AC109642.1','LBH'],
        ['AC110769.3','LINC01191'],
        ['AC111186.1','SLC16A5'],
        ['AC112721.1','AC112721.2'],
        ['AC112721.1','COL6A3'],
        ['AC112721.2','COL6A3'],
        ['AC113607.1','TMEM18'],
        ['AC115617.2','ACAA2'],
        ['AC116366.6','IRF1'],
        ['AC119673.1','IGF2'],
        ['AC121336.2','SLC2A13'],
        ['AC125232.1','RMND5A'],
        ['AC135048.13','FBXL19'],
        ['AC138430.4','FAM129A'],
        ['AC138517.4','NCOA4'],
        ['AC142381.1','IGHV3-23'],
        ['AC195454.1','CHRNA9'],
        ['AC244250.2','IGLV2-11'],
        ['AC244250.2','IGLV2-14'],
        ['AC244250.4','IGLV2-18'],
        ['AC245028.1','IGLL5'],
        ['AC245369.2','IGHV2-70'],
        ['ACACB','RP11-443D10.3'],
        ['ACACB','RP11-968O1.5'],
        ['ACAD8','GLB1L3'],
        ['ACADL','KANSL1L'],
        ['ACADVL','COL1A1'],
        ['ACADVL','FN1'],
        ['ACADVL','LTBP2'],
        ['ACADVL','SFTPA2'],
        ['ACAN','COL1A1'],
        ['ACAN','FN1'],
        ['ACAN','THBS1'],
        ['ACAP3','FN1'],
        ['ACAP3','MARCKS'],
        ['ACBD6','RP5-1180C10.2'],
        ['ACCS','EXT2'],
        ['ACE','CITED2'],
        ['ACE','SFTPA2'],
        ['ACE','SFTPC'],
        ['ACE2','TMEM27'],
        ['ACIN1','FN1'],
        ['ACIN1','SFTPA1'],
        ['ACIN1','SFTPC'],
        ['ACKR1','CADM3'],
        ['ACKR1','IGK@'],
        ['ACKR2','KRBOX1'],
        ['ACKR2','RP11-136C24.3'],
        ['ACLY','COL1A1'],
        ['ACLY','COL1A2'],
        ['ACLY','FN1'],
        ['ACO2','FN1'],
        ['ACO2','RP1-101G11.2'],
        ['ACOT1','ACOT2'],
        ['ACOXL','BCL2L11'],
        ['ACPP','MYLK'],
        ['ACR','SHANK3'],
        ['ACRBP','LPAR5'],
        ['ACSL1','ALB'],
        ['ACSM2A','ACSM2B'],
        ['ACSM5','ACSM5P1'],
        ['ACSS1','APMAP'],
        ['ACTA1','ACTB'],
        ['ACTA1','ACTG1'],
        ['ACTA1','CKM'],
        ['ACTA1','CLU'],
        ['ACTA1','DES'],
        ['ACTA1','FLNC'],
        ['ACTA1','IGFBP5'],
        ['ACTA1','PTRF'],
        ['ACTA1','TG'],
        ['ACTA1','TPO'],
        ['ACTA2','ACTB'],
        ['ACTA2','ACTG1'],
        ['ACTA2','ACTG2'],
        ['ACTA2','COL1A1'],
        ['ACTA2','COL1A2'],
        ['ACTA2','COL6A2'],
        ['ACTA2','FN1'],
        ['ACTA2','HDLBP'],
        ['ACTA2','SFTPC'],
        ['ACTA2','THBS1'],
        ['ACTA2-AS1','ACTB'],
        ['ACTA2-AS1','ACTG1'],
        ['ACTA2-AS1','COL1A1'],
        ['ACTA2-AS1','COL1A2'],
        ['ACTA2-AS1','FN1'],
        ['ACTB','ACTBP1'],
        ['ACTB','ACTBP11'],
        ['ACTB','ACTBP2'],
        ['ACTB','ACTBP8'],
        ['ACTB','ACTBP9'],
        ['ACTB','ACTC1'],
        ['ACTB','ACTG1'],
        ['ACTB','ACTG1P14'],
        ['ACTB','ACTG2'],
        ['ACTB','ACTN1'],
        ['ACTB','ACTN4'],
        ['ACTB','AHNAK'],
        ['ACTB','ALDOA'],
        ['ACTB','ANXA2'],
        ['ACTB','AP2A1'],
        ['ACTB','AP2M1'],
        ['ACTB','ARPC1B'],
        ['ACTB','ATN1'],
        ['ACTB','ATP2B4'],
        ['ACTB','B2M'],
        ['ACTB','BAG6'],
        ['ACTB','BGN'],
        ['ACTB','C3'],
        ['ACTB','CALD1'],
        ['ACTB','CALR'],
        ['ACTB','CCDC80'],
        ['ACTB','CD248'],
        ['ACTB','CD44'],
        ['ACTB','CD74'],
        ['ACTB','CDC37'],
        ['ACTB','CFL1'],
        ['ACTB','CH17-12M21.1'],
        ['ACTB','CKAP4'],
        ['ACTB','CLIP2'],
        ['ACTB','COL12A1'],
        ['ACTB','COL1A1'],
        ['ACTB','COL1A2'],
        ['ACTB','COL3A1'],
        ['ACTB','COL4A2'],
        ['ACTB','COL5A1'],
        ['ACTB','COL5A2'],
        ['ACTB','COL5A3'],
        ['ACTB','COL6A1'],
        ['ACTB','COL6A2'],
        ['ACTB','COL6A3'],
        ['ACTB','CPA1'],
        ['ACTB','CRIM1'],
        ['ACTB','CSRP1'],
        ['ACTB','CTGF'],
        ['ACTB','CTSB'],
        ['ACTB','CXCL12'],
        ['ACTB','DAP'],
        ['ACTB','DBN1'],
        ['ACTB','DBNL'],
        ['ACTB','DES'],
        ['ACTB','DYNC1H1'],
        ['ACTB','EEF2'],
        ['ACTB','EIF4G1'],
        ['ACTB','EIF4G2'],
        ['ACTB','EMILIN1'],
        ['ACTB','ENO1'],
        ['ACTB','EPAS1'],
        ['ACTB','FAM129B'],
        ['ACTB','FARP1'],
        ['ACTB','FASN'],
        ['ACTB','FBLIM1'],
        ['ACTB','FBN1'],
        ['ACTB','FKBP10'],
        ['ACTB','FKBP8'],
        ['ACTB','FLNA'],
        ['ACTB','FLNB'],
        ['ACTB','FLNC'],
        ['ACTB','FN1'],
        ['ACTB','FOSB'],
        ['ACTB','FSCN1'],
        ['ACTB','FTH1'],
        ['ACTB','FTL'],
        ['ACTB','GAPDH'],
        ['ACTB','GNAI2'],
        ['ACTB','GNAS'],
        ['ACTB','GPX3'],
        ['ACTB','GREM1'],
        ['ACTB','H19'],
        ['ACTB','HDGF'],
        ['ACTB','HDLBP'],
        ['ACTB','HLA-A'],
        ['ACTB','HLA-B'],
        ['ACTB','HLA-C'],
        ['ACTB','HLA-E'],
        ['ACTB','HSPB1'],
        ['ACTB','HSPG2'],
        ['ACTB','IGFBP3'],
        ['ACTB','IGFBP4'],
        ['ACTB','IGFBP5'],
        ['ACTB','IGH@'],
        ['ACTB','IGHA1'],
        ['ACTB','IGHG1'],
        ['ACTB','IGHG3'],
        ['ACTB','IGHM'],
        ['ACTB','IGKC'],
        ['ACTB','INTS1'],
        ['ACTB','ITGA5'],
        ['ACTB','KDELR1'],
        ['ACTB','KIAA1199'],
        ['ACTB','LAMB2'],
        ['ACTB','LARP1'],
        ['ACTB','LASP1'],
        ['ACTB','LGALS1'],
        ['ACTB','LMNA'],
        ['ACTB','LOXL1'],
        ['ACTB','LOXL2'],
        ['ACTB','LRP1'],
        ['ACTB','LRP10'],
        ['ACTB','LTBP2'],
        ['ACTB','MAP1A'],
        ['ACTB','MAP4'],
        ['ACTB','MARCH6'],
        ['ACTB','MBOAT7'],
        ['ACTB','MMP2'],
        ['ACTB','MPRIP'],
        ['ACTB','MRC2'],
        ['ACTB','MVP'],
        ['ACTB','MYH11'],
        ['ACTB','MYH9'],
        ['ACTB','MYL6'],
        ['ACTB','MYL9'],
        ['ACTB','NCOR2'],
        ['ACTB','NEAT1'],
        ['ACTB','NFIC'],
        ['ACTB','NME4'],
        ['ACTB','P4HB'],
        ['ACTB','PCSK7'],
        ['ACTB','PDLIM7'],
        ['ACTB','PFN1'],
        ['ACTB','PIEZO1'],
        ['ACTB','PKM'],
        ['ACTB','PLEC'],
        ['ACTB','POTEF'],
        ['ACTB','POTEM'],
        ['ACTB','PRKCSH'],
        ['ACTB','PRRC2A'],
        ['ACTB','PTBP1'],
        ['ACTB','PTMS'],
        ['ACTB','PTRF'],
        ['ACTB','QSOX1'],
        ['ACTB','RHOA'],
        ['ACTB','RP11-498C9.2'],
        ['ACTB','RP11-602M11.3'],
        ['ACTB','RP11-809F4.2'],
        ['ACTB','RP5-977B1.11'],
        ['ACTB','RPS4X'],
        ['ACTB','RRBP1'],
        ['ACTB','S100A6'],
        ['ACTB','SCUBE3'],
        ['ACTB','SEC61A1'],
        ['ACTB','SEPT9'],
        ['ACTB','SERPINE1'],
        ['ACTB','SERPINH1'],
        ['ACTB','SFTPA1'],
        ['ACTB','SFTPA2'],
        ['ACTB','SFTPB'],
        ['ACTB','SFTPC'],
        ['ACTB','SH3BGRL3'],
        ['ACTB','SHC1'],
        ['ACTB','SLC16A3'],
        ['ACTB','SLC34A2'],
        ['ACTB','SLC7A5'],
        ['ACTB','SMARCA4'],
        ['ACTB','SMR3B'],
        ['ACTB','SORBS3'],
        ['ACTB','SPARC'],
        ['ACTB','SPTBN1'],
        ['ACTB','SSR2'],
        ['ACTB','SULF1'],
        ['ACTB','TAGLN'],
        ['ACTB','TBCA'],
        ['ACTB','TG'],
        ['ACTB','TGFBI'],
        ['ACTB','TGM2'],
        ['ACTB','THBS1'],
        ['ACTB','THY1'],
        ['ACTB','TIMP1'],
        ['ACTB','TIMP3'],
        ['ACTB','TLN1'],
        ['ACTB','TMSB10'],
        ['ACTB','TNS1'],
        ['ACTB','TOP1'],
        ['ACTB','TPM2'],
        ['ACTB','TRAM1'],
        ['ACTB','TUBB3'],
        ['ACTB','TUBB4B'],
        ['ACTB','VASN'],
        ['ACTB','VIM'],
        ['ACTB','VIM-AS1'],
        ['ACTB','WDR1'],
        ['ACTB','YWHAG'],
        ['ACTB','ZYX'],
        ['ACTBP12','ACTG1'],
        ['ACTBP7','ACTG1'],
        ['ACTC1','ACTG1'],
        ['ACTC1','COL1A1'],
        ['ACTG1','ACTG2'],
        ['ACTG1','ACTN1'],
        ['ACTG1','AHNAK'],
        ['ACTG1','ALB'],
        ['ACTG1','ALDOA'],
        ['ACTG1','AMY2B'],
        ['ACTG1','ANXA5'],
        ['ACTG1','ANXA6'],
        ['ACTG1','ARHGAP23'],
        ['ACTG1','ARHGDIA'],
        ['ACTG1','ATP2B4'],
        ['ACTG1','BGN'],
        ['ACTG1','BLOC1S5-TXNDC5'],
        ['ACTG1','BSG'],
        ['ACTG1','C3'],
        ['ACTG1','CALM3'],
        ['ACTG1','CALR'],
        ['ACTG1','CAPN1'],
        ['ACTG1','CCDC80'],
        ['ACTG1','CD151'],
        ['ACTG1','CD248'],
        ['ACTG1','CD74'],
        ['ACTG1','CD81'],
        ['ACTG1','CD99'],
        ['ACTG1','CFL1'],
        ['ACTG1','CITED2'],
        ['ACTG1','COL12A1'],
        ['ACTG1','COL1A1'],
        ['ACTG1','COL1A2'],
        ['ACTG1','COL3A1'],
        ['ACTG1','COL4A1'],
        ['ACTG1','COL4A2'],
        ['ACTG1','COL5A1'],
        ['ACTG1','COL6A1'],
        ['ACTG1','COL6A2'],
        ['ACTG1','COL6A3'],
        ['ACTG1','COTL1'],
        ['ACTG1','CSRP1'],
        ['ACTG1','CTD-2313D3.1'],
        ['ACTG1','CTSB'],
        ['ACTG1','CYR61'],
        ['ACTG1','DYNC1H1'],
        ['ACTG1','EEF2'],
        ['ACTG1','EHD2'],
        ['ACTG1','EIF4G1'],
        ['ACTG1','EIF4G2'],
        ['ACTG1','ENO1'],
        ['ACTG1','EPAS1'],
        ['ACTG1','EPN1'],
        ['ACTG1','FBN1'],
        ['ACTG1','FLNA'],
        ['ACTG1','FLNC'],
        ['ACTG1','FN1'],
        ['ACTG1','FTH1'],
        ['ACTG1','FTL'],
        ['ACTG1','GAPDH'],
        ['ACTG1','GNAI2'],
        ['ACTG1','GNAS'],
        ['ACTG1','GNB1'],
        ['ACTG1','GREM1'],
        ['ACTG1','HDLBP'],
        ['ACTG1','HSPB6'],
        ['ACTG1','HSPG2'],
        ['ACTG1','IGFBP3'],
        ['ACTG1','IGFBP4'],
        ['ACTG1','IGFBP5'],
        ['ACTG1','ITGA11'],
        ['ACTG1','ITGB1'],
        ['ACTG1','JUNB'],
        ['ACTG1','KIAA1199'],
        ['ACTG1','KRT10'],
        ['ACTG1','LAMC1'],
        ['ACTG1','LASP1'],
        ['ACTG1','LGALS1'],
        ['ACTG1','LOX'],
        ['ACTG1','LOXL1'],
        ['ACTG1','LOXL2'],
        ['ACTG1','LRP1'],
        ['ACTG1','LTBP2'],
        ['ACTG1','MAP2K3'],
        ['ACTG1','MBOAT7'],
        ['ACTG1','MFGE8'],
        ['ACTG1','MRC2'],
        ['ACTG1','MSN'],
        ['ACTG1','MXRA8'],
        ['ACTG1','MYH9'],
        ['ACTG1','MYL6'],
        ['ACTG1','MYL9'],
        ['ACTG1','MYO1C'],
        ['ACTG1','NCOR2'],
        ['ACTG1','NOTCH2'],
        ['ACTG1','P4HB'],
        ['ACTG1','PCOLCE'],
        ['ACTG1','PCSK7'],
        ['ACTG1','PDIA4'],
        ['ACTG1','PER1'],
        ['ACTG1','PFKL'],
        ['ACTG1','PFN1'],
        ['ACTG1','PKM'],
        ['ACTG1','PLEC'],
        ['ACTG1','PLOD1'],
        ['ACTG1','PPIB'],
        ['ACTG1','PPL'],
        ['ACTG1','PTRF'],
        ['ACTG1','PXDN'],
        ['ACTG1','PYGB'],
        ['ACTG1','RCN3'],
        ['ACTG1','RP11-193H5.1'],
        ['ACTG1','RP11-193H5.2'],
        ['ACTG1','RP5-940J5.9'],
        ['ACTG1','RP5-977B1.11'],
        ['ACTG1','RPL8'],
        ['ACTG1','RRBP1'],
        ['ACTG1','SEC61A1'],
        ['ACTG1','SERPINE1'],
        ['ACTG1','SERPINH1'],
        ['ACTG1','SFTPA1'],
        ['ACTG1','SFTPA2'],
        ['ACTG1','SFTPB'],
        ['ACTG1','SFTPC'],
        ['ACTG1','SLC7A5'],
        ['ACTG1','SPARC'],
        ['ACTG1','SPTAN1'],
        ['ACTG1','TAGLN'],
        ['ACTG1','TG'],
        ['ACTG1','TGFB1'],
        ['ACTG1','TGFBI'],
        ['ACTG1','THBS1'],
        ['ACTG1','TIMP1'],
        ['ACTG1','TIMP2'],
        ['ACTG1','TIMP3'],
        ['ACTG1','TKT'],
        ['ACTG1','TLN1'],
        ['ACTG1','TMEM259'],
        ['ACTG1','TMSB10'],
        ['ACTG1','TNC'],
        ['ACTG1','TPM2'],
        ['ACTG1','TUBB'],
        ['ACTG1','TXNDC5'],
        ['ACTG1','UBXN6'],
        ['ACTG1','VIM'],
        ['ACTG1','VIM-AS1'],
        ['ACTG1','WDR1'],
        ['ACTG1','ZYX'],
        ['ACTG2','DES'],
        ['ACTN1','ACTN4'],
        ['ACTN1','COL1A1'],
        ['ACTN1','COL1A2'],
        ['ACTN1','COL6A1'],
        ['ACTN1','COL6A2'],
        ['ACTN1','COL6A3'],
        ['ACTN1','EEF2'],
        ['ACTN1','FLNA'],
        ['ACTN1','FLNC'],
        ['ACTN1','FN1'],
        ['ACTN1','HMGN1'],
        ['ACTN1','IQCE'],
        ['ACTN1','KIAA1199'],
        ['ACTN1','LASP1'],
        ['ACTN1','MYH9'],
        ['ACTN1','NOC2L'],
        ['ACTN1','PKM'],
        ['ACTN1','PTMS'],
        ['ACTN1','RP5-940J5.9'],
        ['ACTN1','SAMD11'],
        ['ACTN1','SERPINE1'],
        ['ACTN1','SFTPA2'],
        ['ACTN1','SFTPB'],
        ['ACTN1','THBS1'],
        ['ACTN1','TIMP1'],
        ['ACTN1','VIM'],
        ['ACTN4','B2M'],
        ['ACTN4','COL1A1'],
        ['ACTN4','COL1A2'],
        ['ACTN4','COL6A1'],
        ['ACTN4','COL6A2'],
        ['ACTN4','EEF2'],
        ['ACTN4','FASN'],
        ['ACTN4','FLNA'],
        ['ACTN4','FLNC'],
        ['ACTN4','FN1'],
        ['ACTN4','GDI1'],
        ['ACTN4','MYH9'],
        ['ACTN4','PLEC'],
        ['ACTN4','PLIN4'],
        ['ACTN4','PTRF'],
        ['ACTN4','SFTPA1'],
        ['ACTN4','SFTPA2'],
        ['ACTN4','SFTPC'],
        ['ACTN4','TIMP2'],
        ['ACTN4','VIM'],
        ['ACTN4','VIM-AS1'],
        ['ACTN4','VWF'],
        ['ACTR1A','FN1'],
        ['ACTR1B','FN1'],
        ['ACTR2','COL1A1'],
        ['ACTR3','COL1A1'],
        ['ACVRL1','CD74'],
        ['ACVRL1','H19'],
        ['ACVRL1','SFTPB'],
        ['ACVRL1','SFTPC'],
        ['AD000092.3','RPS6'],
        ['ADAM10','PTMS'],
        ['ADAM10','SFTPC'],
        ['ADAM12','FN1'],
        ['ADAM15','FN1'],
        ['ADAM19','COL1A1'],
        ['ADAM19','FLNA'],
        ['ADAM19','FN1'],
        ['ADAMTS1','CD74'],
        ['ADAMTS1','COL1A1'],
        ['ADAMTS1','COL1A2'],
        ['ADAMTS1','COL5A1'],
        ['ADAMTS1','FLNA'],
        ['ADAMTS1','FN1'],
        ['ADAMTS1','SFTPA1'],
        ['ADAMTS1','TRIM28'],
        ['ADAMTS1','VIM'],
        ['ADAMTS2','CHD3'],
        ['ADAMTS2','COL6A2'],
        ['ADAMTS2','FN1'],
        ['ADAMTS2','GNAS'],
        ['ADAMTS2','GSPT1'],
        ['ADAMTS2','TWIST1'],
        ['ADAMTS4','PLIN4'],
        ['ADAMTS4','SFTPB'],
        ['ADAMTS7','ADAMTS7P1'],
        ['ADAMTS7','ADAMTS7P3'],
        ['ADAMTS7','RP11-114H24.5'],
        ['ADAMTS8','SFTPC'],
        ['ADAMTSL1','COL1A1'],
        ['ADAMTSL1','FLNA'],
        ['ADAMTSL1','FN1'],
        ['ADAMTSL4','MIR4257'],
        ['ADAMTSL4','SFTPC'],
        ['ADAMTSL4-AS1','MCL1'],
        ['ADAMTSL5','PTRF'],
        ['ADAP2','RNF135'],
        ['ADAR','COL1A1'],
        ['ADAR','SFTPA2'],
        ['ADAR','SFTPB'],
        ['ADAR','SFTPC'],
        ['ADARB1','SFTPB'],
        ['ADARB1','SFTPC'],
        ['ADAT3','SCAMP4'],
        ['ADCK4','NUMBL'],
        ['ADCY10P1','BGN'],
        ['ADCY10P1','NFYA'],
        ['ADCY3','COL1A1'],
        ['ADCY7','SFTPC'],
        ['ADCY9','COL1A1'],
        ['ADCY9','PTMS'],
        ['ADD1','COL1A1'],
        ['ADD1','ERGIC1'],
        ['ADD1','FN1'],
        ['ADD1','SFTPA2'],
        ['ADD1','SFTPC'],
        ['ADD1','TG'],
        ['ADD2','FIGLA'],
        ['ADGRD1','RP11-243M5.1'],
        ['ADGRE2','ADGRE5'],
        ['ADH1A','ADH1B'],
        ['ADH1B','ADH1C'],
        ['ADH1B','SFTPA2'],
        ['ADH1B','SFTPB'],
        ['ADH1B','SFTPC'],
        ['ADH5','RP11-571L19.8'],
        ['ADIPOQ','CD36'],
        ['ADIPOQ','CLU'],
        ['ADIPOQ','IGK@'],
        ['ADIPOQ','PCYOX1'],
        ['ADIPOQ','VKORC1L1'],
        ['ADIPOQ-AS1','PLIN4'],
        ['ADM','FN1'],
        ['ADM','TMEM214'],
        ['ADPRHL1','DCUN1D2'],
        ['ADPRHL2','TEKT2'],
        ['ADRBK1','IGFBP4'],
        ['ADRM1','FLNA'],
        ['ADRM1','FN1'],
        ['ADRM1','MYH9'],
        ['ADRM1','PCK2'],
        ['ADSL','RP5-1042K10.14'],
        ['ADSL','SGSM3'],
        ['ADSS','RP11-518L10.5'],
        ['AEBP1','CD74'],
        ['AEBP1','COL1A1'],
        ['AEBP1','COL1A2'],
        ['AEBP1','FLNC'],
        ['AEBP1','FN1'],
        ['AEBP1','IGFBP5'],
        ['AEBP1','MYH11'],
        ['AEBP1','SFTPA1'],
        ['AEBP1','SFTPB'],
        ['AEBP1','SFTPC'],
        ['AEBP1','TG'],
        ['AEBP1','THBS1'],
        ['AEBP1','VIM'],
        ['AEN','ISG20'],
        ['AES','CAPN15'],
        ['AES','CCDC85B'],
        ['AES','CD74'],
        ['AES','COL1A1'],
        ['AES','COL5A1'],
        ['AES','EPN1'],
        ['AES','FN1'],
        ['AES','GNAS'],
        ['AES','GPX3'],
        ['AES','IGFBP4'],
        ['AES','NCOR2'],
        ['AES','PRRX2'],
        ['AES','SFTPA2'],
        ['AES','SFTPB'],
        ['AES','SFTPC'],
        ['AES','SLC34A2'],
        ['AES','TG'],
        ['AF001548.5','NDE1'],
        ['AF011889.5','IDS'],
        ['AF011889.5','RP13-507I23.1'],
        ['AF127936.7','NRIP1'],
        ['AF127936.9','NRIP1'],
        ['AFAP1','COL1A1'],
        ['AFAP1','FN1'],
        ['AFF1','TG'],
        ['AFM','RP11-622A1.2'],
        ['AFMID','LRG1'],
        ['AGAP3','FN1'],
        ['AGAP3','SFTPC'],
        ['AGBL5','HBB'],
        ['AGER','AHNAK'],
        ['AGER','B2M'],
        ['AGER','CD74'],
        ['AGER','COL3A1'],
        ['AGER','EEF2'],
        ['AGER','EMP2'],
        ['AGER','EPAS1'],
        ['AGER','FASN'],
        ['AGER','FLNA'],
        ['AGER','FN1'],
        ['AGER','GSN'],
        ['AGER','HLA-DPA1'],
        ['AGER','HLA-DRA'],
        ['AGER','HLA-E'],
        ['AGER','MYH9'],
        ['AGER','NCOR2'],
        ['AGER','PBX2'],
        ['AGER','PSAP'],
        ['AGER','SFTPA1'],
        ['AGER','SFTPA2'],
        ['AGER','SFTPB'],
        ['AGER','SFTPC'],
        ['AGER','SPTBN1'],
        ['AGER','SRRM2'],
        ['AGER','SUSD2'],
        ['AGER','TIMP3'],
        ['AGFG1','MAF1'],
        ['AGO2','RP11-642A1.2'],
        ['AGPAT1','COL1A1'],
        ['AGPAT2','LRP1'],
        ['AGPAT3','TXNIP'],
        ['AGRN','COL1A1'],
        ['AGRN','FN1'],
        ['AGRN','PRNP'],
        ['AGRN','SFTPA2'],
        ['AGRN','SFTPB'],
        ['AGRN','SFTPC'],
        ['AGRN','THBS1'],
        ['AGT','ALB'],
        ['AGT','C1ORF198'],
        ['AGT','FGA'],
        ['AGT','FGG'],
        ['AGXT','ALB'],
        ['AGXT','SERPINA1'],
        ['AHCTF1','AHCTF1P1'],
        ['AHCY','COL1A1'],
        ['AHCY','RP11-292F22.3'],
        ['AHCYL1','SFTPC'],
        ['AHDC1','FN1'],
        ['AHNAK','APLP2'],
        ['AHNAK','AQP1'],
        ['AHNAK','B2M'],
        ['AHNAK','BCAM'],
        ['AHNAK','C3'],
        ['AHNAK','CALR'],
        ['AHNAK','CCDC152'],
        ['AHNAK','CD74'],
        ['AHNAK','CEBPD'],
        ['AHNAK','COL12A1'],
        ['AHNAK','COL1A1'],
        ['AHNAK','COL1A2'],
        ['AHNAK','COL3A1'],
        ['AHNAK','COL4A1'],
        ['AHNAK','COL4A2'],
        ['AHNAK','COL5A1'],
        ['AHNAK','COL6A1'],
        ['AHNAK','COL6A2'],
        ['AHNAK','COL6A3'],
        ['AHNAK','CTNNA1'],
        ['AHNAK','CTSD'],
        ['AHNAK','CTTN'],
        ['AHNAK','DDX17'],
        ['AHNAK','DDX3X'],
        ['AHNAK','DUSP1'],
        ['AHNAK','EEF1G'],
        ['AHNAK','EEF2'],
        ['AHNAK','EMILIN1'],
        ['AHNAK','EPAS1'],
        ['AHNAK','EZR'],
        ['AHNAK','FGFR1'],
        ['AHNAK','FLNA'],
        ['AHNAK','FN1'],
        ['AHNAK','FOS'],
        ['AHNAK','FOSB'],
        ['AHNAK','FTL'],
        ['AHNAK','GPRC5A'],
        ['AHNAK','GREM1'],
        ['AHNAK','GRN'],
        ['AHNAK','HLA-B'],
        ['AHNAK','HLA-DRA'],
        ['AHNAK','HLA-E'],
        ['AHNAK','HUWE1'],
        ['AHNAK','IGFBP3'],
        ['AHNAK','IGFBP5'],
        ['AHNAK','IGKC'],
        ['AHNAK','IST1'],
        ['AHNAK','LRP1'],
        ['AHNAK','MARCO'],
        ['AHNAK','MICAL2'],
        ['AHNAK','MIR3654'],
        ['AHNAK','MSN'],
        ['AHNAK','MYH11'],
        ['AHNAK','MYH9'],
        ['AHNAK','MYL6'],
        ['AHNAK','NCOR2'],
        ['AHNAK','NEGR1'],
        ['AHNAK','NR4A1'],
        ['AHNAK','P4HB'],
        ['AHNAK','PLEC'],
        ['AHNAK','PSAP'],
        ['AHNAK','PTRF'],
        ['AHNAK','RP11-295K3.1'],
        ['AHNAK','SERF2'],
        ['AHNAK','SERPINE1'],
        ['AHNAK','SFTPA1'],
        ['AHNAK','SFTPA2'],
        ['AHNAK','SFTPB'],
        ['AHNAK','SFTPC'],
        ['AHNAK','SLC34A2'],
        ['AHNAK','SLC6A4'],
        ['AHNAK','SOCS3'],
        ['AHNAK','SPARC'],
        ['AHNAK','SPARCL1'],
        ['AHNAK','SPOCK2'],
        ['AHNAK','SPTBN1'],
        ['AHNAK','SRRM2'],
        ['AHNAK','TAOK2'],
        ['AHNAK','TG'],
        ['AHNAK','THBS1'],
        ['AHNAK','TIMP1'],
        ['AHNAK','TIMP3'],
        ['AHNAK','TLN1'],
        ['AHNAK','TNS1'],
        ['AHNAK','TPM1'],
        ['AHNAK','TPM2'],
        ['AHNAK','TPT1'],
        ['AHNAK','TXNIP'],
        ['AHNAK','UBA1'],
        ['AHNAK','VIM'],
        ['AHNAK','VIM-AS1'],
        ['AHNAK','VWF'],
        ['AHNAK','ZFP36'],
        ['AHNAK2','COL1A1'],
        ['AHNAK2','FLNA'],
        ['AHNAK2','FN1'],
        ['AHNAK2','KRT1'],
        ['AHRR','CTD-2228K2.1'],
        ['AHSA2','C2ORF74'],
        ['AIF1L','TG'],
        ['AIFM3','LZTR1'],
        ['AIG1','RP1-95L4.4'],
        ['AIP','MIR6752'],
        ['AJUBA','SERPINE1'],
        ['AK1','ST6GALNAC6'],
        ['AK6','TAF9'],
        ['AKAP13','SFTPA2'],
        ['AKAP13','SFTPB'],
        ['AKAP13','SFTPC'],
        ['AKAP2','FN1'],
        ['AKAP2','SFTPA1'],
        ['AKAP2','SFTPA2'],
        ['AKAP2','SFTPB'],
        ['AKAP2','SFTPC'],
        ['AKNA','EMP2'],
        ['AKNA','RP11-9M16.2'],
        ['AKR1B1','FN1'],
        ['AKR1B10','AKR1B15'],
        ['AKR1C1','AKR1C2'],
        ['AKR7A2','AKR7A3'],
        ['AKT1','COL1A1'],
        ['AKT1','COL6A2'],
        ['AKT1','EXT1'],
        ['AKT1','FLNA'],
        ['AKT1','FN1'],
        ['AKT1','PLEC'],
        ['AKT1','SFTPC'],
        ['AKT1','YBX3'],
        ['AKT1S1','COL1A1'],
        ['AKT1S1','FN1'],
        ['AKT2','COL1A1'],
        ['AKT2','MIR641'],
        ['AKT2','SFTPC'],
        ['AKT2','TG'],
        ['AL031663.2','RP3-461P17.10'],
        ['AL132709.5','MEG8'],
        ['AL132709.8','RP11-70O5.2'],
        ['AL133458.1','RNASET2'],
        ['AL133458.1','RP11-514O12.4'],
        ['AL136376.1','SFTPC'],
        ['AL157871.2','SLC25A29'],
        ['AL163636.6','ANG'],
        ['AL357673.1','CYB5RL'],
        ['AL358175.2','AL592494.5'],
        ['AL512428.1','FAM65B'],
        ['AL590369.1','PTRF'],
        ['AL590431.1','UBAP2L'],
        ['AL590762.6','ZCRB1'],
        ['AL590822.1','C1ORF86'],
        ['AL591845.1','SH3D21'],
        ['AL645728.2','PTRF'],
        ['AL662800.1','PPP1R10'],
        ['AL928768.3','IGH@'],
        ['AL928768.3','IGHA1'],
        ['AL928768.3','IGHG1'],
        ['ALB','AMBP'],
        ['ALB','C3'],
        ['ALB','C4BPA'],
        ['ALB','CRP'],
        ['ALB','CYP2E1'],
        ['ALB','FGA'],
        ['ALB','FGB'],
        ['ALB','FGG'],
        ['ALB','FGL1'],
        ['ALB','GAA'],
        ['ALB','GATM'],
        ['ALB','GP2'],
        ['ALB','HAMP'],
        ['ALB','HP'],
        ['ALB','HPX'],
        ['ALB','LBP'],
        ['ALB','MAT1A'],
        ['ALB','MT2A'],
        ['ALB','ORM1'],
        ['ALB','RP11-986E7.7'],
        ['ALB','SAA1'],
        ['ALB','SAA2'],
        ['ALB','SERPINA1'],
        ['ALB','SERPINA3'],
        ['ALB','TF'],
        ['ALCAM','COL1A1'],
        ['ALCAM','COL6A1'],
        ['ALCAM','THBS1'],
        ['ALDH1A1','TG'],
        ['ALDH2','SFTPA2'],
        ['ALDH2','SFTPC'],
        ['ALDH3A2','SFTPC'],
        ['ALDH3B1','ALDH3B2'],
        ['ALDH6A1','LRG1'],
        ['ALDH6A1','SFTPB'],
        ['ALDOA','ALDOAP2'],
        ['ALDOA','BSG'],
        ['ALDOA','CD74'],
        ['ALDOA','COL1A1'],
        ['ALDOA','COL1A2'],
        ['ALDOA','COL5A1'],
        ['ALDOA','COL6A1'],
        ['ALDOA','COL6A2'],
        ['ALDOA','COL6A3'],
        ['ALDOA','CRYAB'],
        ['ALDOA','FLNA'],
        ['ALDOA','FN1'],
        ['ALDOA','LMNA'],
        ['ALDOA','MYH9'],
        ['ALDOA','PLEC'],
        ['ALDOA','PTRF'],
        ['ALDOA','SFTPA1'],
        ['ALDOA','SFTPA2'],
        ['ALDOA','SFTPC'],
        ['ALDOA','SSC5D'],
        ['ALDOA','TG'],
        ['ALDOA','TGFBI'],
        ['ALDOA','THBS1'],
        ['ALDOA','TPM2'],
        ['ALDOA','VIM'],
        ['ALG14','RP4-639F20.3'],
        ['ALKBH3','SEC14L1P1'],
        ['ALMS1','ALMS1-IT1'],
        ['ALOX15','SFTPB'],
        ['ALOX5','FN1'],
        ['ALPL','SFTPC'],
        ['ALS2CR12','TRAK2'],
        ['ALYREF','COL6A2'],
        ['ALYREF','YBX3'],
        ['AMACR','C1QTNF3'],
        ['AMACR','SLC45A2'],
        ['AMBP','FGA'],
        ['AMBP','SERPINA1'],
        ['AMFR','PTMS'],
        ['AMH','SF3A2'],
        ['AMOTL1','SFTPB'],
        ['AMOTL1','SFTPC'],
        ['AMOTL2','COL1A1'],
        ['AMOTL2','FN1'],
        ['AMT','NICN1'],
        ['AMZ2','AMZ2P1'],
        ['AMZ2','RP11-147L13.15'],
        ['ANAPC7','RP11-478C19.2'],
        ['ANG','RNASE4'],
        ['ANG','RP11-903H12.5'],
        ['ANGEL1','RP11-488C13.5'],
        ['ANGPTL4','PTRF'],
        ['ANK1','S100A9'],
        ['ANKDD1A','PLEKHO2'],
        ['ANKRD1','TG'],
        ['ANKRD11','COL1A1'],
        ['ANKRD11','SFTPA2'],
        ['ANKRD11','SFTPC'],
        ['ANKRD11','TRB@'],
        ['ANKRD12','PTPRM'],
        ['ANKRD13A','RP11-946P6.6'],
        ['ANKRD17','EPAS1'],
        ['ANKRD18A','ANKRD18B'],
        ['ANKRD18A','RP11-392E22.5'],
        ['ANKRD34A','LIX1L'],
        ['ANKRD35','PIAS3'],
        ['ANKRD36','ANKRD36B'],
        ['ANKRD36','ANKRD36C'],
        ['ANKRD36B','ANKRD36C'],
        ['ANKRD40','PTRF'],
        ['ANKRD52','COL1A1'],
        ['ANKRD52','FN1'],
        ['ANKRD54','DAP'],
        ['ANO1','RP11-626H12.1'],
        ['ANO1','SFTPA2'],
        ['ANO2','VWF'],
        ['ANO3','CTD-2507G9.1'],
        ['ANO6','SFTPC'],
        ['ANP32B','ANP32BP1'],
        ['ANP32B','FN1'],
        ['ANP32B','NCL'],
        ['ANP32B','RPLP2'],
        ['ANP32E','NCL'],
        ['ANPEP','COL1A1'],
        ['ANPEP','COL1A2'],
        ['ANPEP','COL6A2'],
        ['ANPEP','FN1'],
        ['ANPEP','THBS1'],
        ['ANPEP','VIM'],
        ['ANTXR1','COL1A1'],
        ['ANTXR1','FN1'],
        ['ANTXR1','SFTPB'],
        ['ANTXR1','SFTPC'],
        ['ANXA1','COL1A1'],
        ['ANXA1','COL1A2'],
        ['ANXA1','SFTPA1'],
        ['ANXA1','SFTPA2'],
        ['ANXA1','SFTPC'],
        ['ANXA11','COL1A1'],
        ['ANXA11','COL3A1'],
        ['ANXA11','FN1'],
        ['ANXA11','PTRF'],
        ['ANXA11','SFTPA2'],
        ['ANXA11','SFTPC'],
        ['ANXA2','ANXA2P1'],
        ['ANXA2','ANXA2P3'],
        ['ANXA2','COL1A1'],
        ['ANXA2','COL1A2'],
        ['ANXA2','FN1'],
        ['ANXA2','SFTPC'],
        ['ANXA4','PRSS1'],
        ['ANXA5','COL1A1'],
        ['ANXA5','COL1A2'],
        ['ANXA5','COL6A2'],
        ['ANXA5','FN1'],
        ['ANXA5','IGFBP3'],
        ['ANXA5','PCSK7'],
        ['ANXA5','SFTPB'],
        ['ANXA5','SFTPC'],
        ['ANXA5','THBS1'],
        ['ANXA6','COL1A1'],
        ['ANXA6','COL1A2'],
        ['ANXA6','FN1'],
        ['ANXA6','MYH9'],
        ['ANXA6','PLIN1'],
        ['ANXA6','SFTPA1'],
        ['ANXA6','SFTPC'],
        ['ANXA6','TNIP1'],
        ['ANXA6','VIM'],
        ['AOC3','AOC4P'],
        ['AOC3','FTL'],
        ['AOC3','SFTPC'],
        ['AOX1','AOX3P'],
        ['AP000275.65','TCP10L'],
        ['AP000318.2','AP000320.7'],
        ['AP000320.7','SLC5A3'],
        ['AP000347.4','ASLP1'],
        ['AP000347.4','IGLL3P'],
        ['AP000347.4','KB-1572G7.2'],
        ['AP000350.10','COL1A1'],
        ['AP000350.4','COL1A1'],
        ['AP000350.4','FN1'],
        ['AP000351.10','KB-226F1.1'],
        ['AP000435.1','EIF4A2'],
        ['AP000568.2','SREK1IP1'],
        ['AP000580.1','RPS20'],
        ['AP000662.9','PPIA'],
        ['AP000783.1','GRAMD1B'],
        ['AP001468.1','LSS'],
        ['AP001469.9','MCM3AP'],
        ['AP002381.2','RPL15'],
        ['AP003068.23','PTRF'],
        ['AP003419.11','CLCF1'],
        ['AP003419.11','POLD4'],
        ['AP005482.1','PSMG2'],
        ['AP006621.1','AP006621.5'],
        ['AP006621.5','CMB9-55F22.1'],
        ['AP1B1','FN1'],
        ['AP1G2','JPH4'],
        ['AP2A1','COL1A1'],
        ['AP2A1','FN1'],
        ['AP2A2','COL1A1'],
        ['AP2B1','FN1'],
        ['AP2B1','RP13-88F20.1'],
        ['AP2B1','SFTPC'],
        ['AP2M1','COL1A1'],
        ['AP2M1','COL1A2'],
        ['AP2M1','FLNA'],
        ['AP2M1','FN1'],
        ['AP2M1','MYH9'],
        ['AP2M1','SFTPC'],
        ['AP2M1','VIM'],
        ['AP2S1','COL1A1'],
        ['AP2S1','SNX17'],
        ['AP3B1','CTD-2179L22.1'],
        ['AP3D1','BCL9L'],
        ['AP3D1','COL1A1'],
        ['AP3D1','COL1A2'],
        ['AP3D1','FLNA'],
        ['AP3D1','FN1'],
        ['AP3D1','KRT4'],
        ['AP3D1','SFTPC'],
        ['AP3S1','COMMD10'],
        ['AP3S2','FBLIM1'],
        ['AP3S2','PTRF'],
        ['AP3S2','SFTPB'],
        ['AP4M1','COL1A1'],
        ['AP5S1','MAVS'],
        ['APBB2','Y_RNA'],
        ['APBB3','SRA1'],
        ['APCDD1L','COL1A1'],
        ['APCDD1L','FN1'],
        ['APH1A','COL1A1'],
        ['APITD1','CORT'],
        ['APITD1-CORT','CORT'],
        ['APLP1','TUBB4A'],
        ['APLP2','APP'],
        ['APLP2','AQP1'],
        ['APLP2','B2M'],
        ['APLP2','CANX'],
        ['APLP2','COL1A1'],
        ['APLP2','COL1A2'],
        ['APLP2','COL6A3'],
        ['APLP2','EPAS1'],
        ['APLP2','FN1'],
        ['APLP2','FTL'],
        ['APLP2','HLA-B'],
        ['APLP2','LAMB2'],
        ['APLP2','MYH9'],
        ['APLP2','SFTPA1'],
        ['APLP2','SFTPA2'],
        ['APLP2','SFTPB'],
        ['APLP2','SFTPC'],
        ['APLP2','TG'],
        ['APLP2','THBS1'],
        ['APLP2','VIM'],
        ['APOA1','FGB'],
        ['APOA1','SERPINA1'],
        ['APOB','CRP'],
        ['APOB','SERPINA1'],
        ['APOB','SERPINA3'],
        ['APOBEC3A','APOBEC3B'],
        ['APOBEC3C','PTRF'],
        ['APOBEC3D','APOBEC3F'],
        ['APOC1','APOC1P1'],
        ['APOC1','CTB-129P6.7'],
        ['APOC1','SERPINA1'],
        ['APOC3','C3'],
        ['APOC3','SERPINA1'],
        ['APOD','TG'],
        ['APOE','CD74'],
        ['APOE','FGG'],
        ['APOE','SFTPC'],
        ['APOE','THBS1'],
        ['APOH','FGB'],
        ['APOH','SERPINA1'],
        ['APOL1','APOL2'],
        ['APOL1','DA750114'],
        ['APOL1','SCYL3'],
        ['APOL3','SFTPC'],
        ['APOL6','SFTPB'],
        ['APOLD1','PARVA'],
        ['APOLD1','SERBP1'],
        ['APP','B2M'],
        ['APP','CD74'],
        ['APP','COL1A1'],
        ['APP','COL1A2'],
        ['APP','DES'],
        ['APP','EPAS1'],
        ['APP','FN1'],
        ['APP','GFAP'],
        ['APP','HLA-B'],
        ['APP','IGKC'],
        ['APP','LRP1'],
        ['APP','LTBP3'],
        ['APP','MYH9'],
        ['APP','NEAT1'],
        ['APP','SFTPA1'],
        ['APP','SFTPA2'],
        ['APP','SFTPB'],
        ['APP','SFTPC'],
        ['APP','SKI'],
        ['APP','TG'],
        ['APP','VIM'],
        ['APPL2','RP11-474B16.1'],
        ['AQP1','CAV1'],
        ['AQP1','CD74'],
        ['AQP1','CTSD'],
        ['AQP1','DCN'],
        ['AQP1','EEF2'],
        ['AQP1','EPAS1'],
        ['AQP1','FN1'],
        ['AQP1','HLA-E'],
        ['AQP1','ITGB2'],
        ['AQP1','LRP1'],
        ['AQP1','SFTPA1'],
        ['AQP1','SFTPA2'],
        ['AQP1','SFTPB'],
        ['AQP1','SFTPC'],
        ['AQP1','TG'],
        ['AQP1','TIMP3'],
        ['AQP3','SFTPC'],
        ['AQP4','CHST9'],
        ['AQP4','SFTPC'],
        ['AQP7','AQP7P2'],
        ['AQP7','FP325317.1'],
        ['AQP7P1','AQP7P2'],
        ['AQP7P1','FP325317.1'],
        ['ARAP1','COL1A1'],
        ['ARAP1','FN1'],
        ['ARAP1','SFTPA2'],
        ['ARAP1','SFTPC'],
        ['ARAP2','RP11-399D2.1'],
        ['ARCN1','COL1A1'],
        ['ARCN1','FN1'],
        ['ARCN1','PTRF'],
        ['ARF1','COL1A1'],
        ['ARF1','COL1A2'],
        ['ARF1','FN1'],
        ['ARF1','SFTPA1'],
        ['ARF1','SFTPC'],
        ['ARF3','COL1A1'],
        ['ARF3','FKBP11'],
        ['ARF3','FN1'],
        ['ARF3','MYH9'],
        ['ARF3','SFTPB'],
        ['ARF3','SFTPC'],
        ['ARF3','WNT10B'],
        ['ARF4','COL1A1'],
        ['ARF4','FN1'],
        ['ARF6','CD151'],
        ['ARFGAP1','COL1A1'],
        ['ARFGAP1','SFTPA2'],
        ['ARFGAP2','RP11-390K5.6'],
        ['ARFGAP3','PACSIN2'],
        ['ARG1','FGG'],
        ['ARG2','CD36'],
        ['ARHGAP1','CD68'],
        ['ARHGAP1','COL6A2'],
        ['ARHGAP1','FBLIM1'],
        ['ARHGAP1','FLG2'],
        ['ARHGAP1','FN1'],
        ['ARHGAP1','PLIN3'],
        ['ARHGAP1','PTRF'],
        ['ARHGAP1','SFTPB'],
        ['ARHGAP1','SNORA67'],
        ['ARHGAP17','COL1A1'],
        ['ARHGAP18','PNLIP'],
        ['ARHGAP21','RP11-296E7.1'],
        ['ARHGAP23','COL1A1'],
        ['ARHGAP23','COL1A2'],
        ['ARHGAP23','COL5A1'],
        ['ARHGAP23','FN1'],
        ['ARHGAP23','IGFBP4'],
        ['ARHGAP23','MARCKS'],
        ['ARHGAP23','SFTPC'],
        ['ARHGAP29','COL1A1'],
        ['ARHGAP30','SFTPC'],
        ['ARHGAP31','SFTPC'],
        ['ARHGAP35','COL1A1'],
        ['ARHGAP35','FN1'],
        ['ARHGAP42','TMEM133'],
        ['ARHGAP5','RP4-791C19.1'],
        ['ARHGDIA','BGN'],
        ['ARHGDIA','COL1A1'],
        ['ARHGDIA','COL1A2'],
        ['ARHGDIA','FAM129B'],
        ['ARHGDIA','FASN'],
        ['ARHGDIA','FLNA'],
        ['ARHGDIA','FLNC'],
        ['ARHGDIA','FN1'],
        ['ARHGDIA','VIM'],
        ['ARHGDIB','ERP27'],
        ['ARHGDIB','SFTPA2'],
        ['ARHGDIB','SFTPC'],
        ['ARHGEF1','COL1A1'],
        ['ARHGEF1','SFTPC'],
        ['ARHGEF11','MIR765'],
        ['ARHGEF17','COL1A1'],
        ['ARHGEF17','FN1'],
        ['ARHGEF17','SFTPC'],
        ['ARHGEF18','CTB-133G6.1'],
        ['ARHGEF2','RP11-336K24.6'],
        ['ARHGEF2','SFTPA2'],
        ['ARHGEF2','SFTPB'],
        ['ARHGEF2','SFTPC'],
        ['ARHGEF38','ARHGEF38-IT1'],
        ['ARID1A','COL1A1'],
        ['ARID1A','CTSB'],
        ['ARID1A','IGFBP4'],
        ['ARID1A','SFTPA1'],
        ['ARID1A','SFTPC'],
        ['ARID1A','TRIM28'],
        ['ARID1B','SFTPC'],
        ['ARID3C','DCTN3'],
        ['ARIH2','PTRF'],
        ['ARIH2','SFTPB'],
        ['ARL10','HIGD2A'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARL2','FN1'],
        ['ARL2BP','RP11-407G23.4'],
        ['ARL6IP4','FLNA'],
        ['ARMCX1','ARMCX4'],
        ['ARMCX3','RP4-545K15.5'],
        ['ARNT','COL1A1'],
        ['ARPC1A','SFTPC'],
        ['ARPC1B','COL1A1'],
        ['ARPC1B','FLNA'],
        ['ARPC1B','FN1'],
        ['ARPC1B','MYH9'],
        ['ARPC1B','SFTPC'],
        ['ARPC2','C3'],
        ['ARPC2','COL1A1'],
        ['ARPC2','FN1'],
        ['ARPC3','ARPC3P1'],
        ['ARPC4','ARPC4-TTLL3'],
        ['ARPC4','COL1A1'],
        ['ARPC4','FN1'],
        ['ARPC4','SFTPC'],
        ['ARPC4','TTLL3'],
        ['ARPC5','COL1A1'],
        ['ARPP19','SFTPB'],
        ['ARRB1','SFTPC'],
        ['ARRDC2','SFTPA2'],
        ['ARRDC2','SFTPC'],
        ['ARSD','SFTPC'],
        ['ARSE','ARSEP1'],
        ['ART4','SMCO3'],
        ['ASAH1','SFTPC'],
        ['ASAH2','ASAH2B'],
        ['ASIC1','MARCKS'],
        ['ASLP1','GUSBP11'],
        ['ASPH','COL1A1'],
        ['ASPH','COL1A2'],
        ['ASPH','FN1'],
        ['ASPRV1','PCBP1-AS1'],
        ['ASRGL1','SCGB1A1'],
        ['ASS1','ASS1P11'],
        ['ASS1','ASS1P12'],
        ['ASS1','ASS1P9'],
        ['ATAD3A','ATAD3B'],
        ['ATAD3A','PHGDH'],
        ['ATF3','SFTPA1'],
        ['ATF3','SFTPA2'],
        ['ATF3','SFTPC'],
        ['ATF4','ATF4P4'],
        ['ATF5','SERPINA1'],
        ['ATF6B','TG'],
        ['ATF6B','TNXB'],
        ['ATG13','COL1A1'],
        ['ATG13','SFTPA1'],
        ['ATG2A','SFTPC'],
        ['ATG9A','COL1A2'],
        ['ATHL1','TRB@'],
        ['ATL3','TPM4'],
        ['ATN1','CD74'],
        ['ATN1','COL1A1'],
        ['ATN1','COL1A2'],
        ['ATN1','COL6A2'],
        ['ATN1','FLNA'],
        ['ATN1','FN1'],
        ['ATN1','SFTPA2'],
        ['ATN1','SFTPB'],
        ['ATN1','TG'],
        ['ATN1','TYROBP'],
        ['ATOH8','COL1A1'],
        ['ATOH8','FN1'],
        ['ATOH8','TMEM54'],
        ['ATOH8','TXNDC5'],
        ['ATP10B','RP11-109J4.1'],
        ['ATP11A','SFTPA1'],
        ['ATP11A','SFTPA2'],
        ['ATP11A','SFTPC'],
        ['ATP13A1','COL6A1'],
        ['ATP13A3','FN1'],
        ['ATP1A1','CKM'],
        ['ATP1A1','CLU'],
        ['ATP1A1','COL1A1'],
        ['ATP1A1','CRYAB'],
        ['ATP1A1','CTSB'],
        ['ATP1A1','DDX17'],
        ['ATP1A1','EEF2'],
        ['ATP1A1','FLNA'],
        ['ATP1A1','FLNC'],
        ['ATP1A1','FN1'],
        ['ATP1A1','GNAS'],
        ['ATP1A1','GPX3'],
        ['ATP1A1','GSN'],
        ['ATP1A1','H19'],
        ['ATP1A1','HLA-DRA'],
        ['ATP1A1','HLA-E'],
        ['ATP1A1','IGFBP5'],
        ['ATP1A1','MALAT1'],
        ['ATP1A1','MLLT6'],
        ['ATP1A1','PSAP'],
        ['ATP1A1','RHOB'],
        ['ATP1A1','SFTPA1'],
        ['ATP1A1','SFTPA2'],
        ['ATP1A1','SFTPB'],
        ['ATP1A1','SFTPC'],
        ['ATP1A1','SLC34A2'],
        ['ATP1A1','TG'],
        ['ATP1A1','TPO'],
        ['ATP1A1','VIM'],
        ['ATP1A2','ATP1A3'],
        ['ATP1A2','ATP1A4'],
        ['ATP1A2','GFAP'],
        ['ATP1B1','MALAT1'],
        ['ATP1B1','RP5-1018K9.1'],
        ['ATP1B1','TG'],
        ['ATP1B3','ATP1B3P1'],
        ['ATP1B4','SOD2'],
        ['ATP2A1','LRP1'],
        ['ATP2A2','COL1A1'],
        ['ATP2A2','COL1A2'],
        ['ATP2A2','COL6A1'],
        ['ATP2A2','MMP14'],
        ['ATP2A2','MYH9'],
        ['ATP2A2','MYO9B'],
        ['ATP2A2','SFTPC'],
        ['ATP2A2','TG'],
        ['ATP2A3','LYZ'],
        ['ATP2A3','PNLIP'],
        ['ATP2B4','COL1A1'],
        ['ATP2B4','COL1A2'],
        ['ATP2B4','FN1'],
        ['ATP2B4','IGFBP4'],
        ['ATP2B4','SFTPA1'],
        ['ATP2B4','SFTPA2'],
        ['ATP2B4','SFTPC'],
        ['ATP5A1','ATP5A1P2'],
        ['ATP5A1','ATP5A1P3'],
        ['ATP5A1','COL1A1'],
        ['ATP5B','COL1A1'],
        ['ATP5B','COL1A2'],
        ['ATP5B','FN1'],
        ['ATP5B','SFTPC'],
        ['ATP5B','SNORD59A'],
        ['ATP5D','FAM96B'],
        ['ATP5D','PLEC'],
        ['ATP5E','ATP5EP2'],
        ['ATP5G3','COL1A1'],
        ['ATP5H','ATP5HP4'],
        ['ATP5I','MFSD7'],
        ['ATP5J2','ATP5J2-PTCD1'],
        ['ATP5J2','PTCD1'],
        ['ATP6AP1','GNAS'],
        ['ATP6AP1','MAP7D1'],
        ['ATP6AP1','PTMS'],
        ['ATP6AP1','RGMB'],
        ['ATP6AP2','RP11-332L8.1'],
        ['ATP6AP2','SFTPA2'],
        ['ATP6V0A1','MIR5010'],
        ['ATP6V0A4','SVOPL'],
        ['ATP6V0C','COL1A1'],
        ['ATP6V0C','FLNA'],
        ['ATP6V0C','FN1'],
        ['ATP6V0C','IGFBP4'],
        ['ATP6V0C','SFTPB'],
        ['ATP6V0C','SFTPC'],
        ['ATP6V0E1','GSN'],
        ['ATP6V1B1','VAX2'],
        ['ATP6V1C1','SFTPB'],
        ['ATP6V1C2','SFTPA1'],
        ['ATP6V1G2-DDX39B','FN1'],
        ['ATP7A','RP5-1000K24.2'],
        ['ATP8B2','COL1A1'],
        ['ATP8B2','FN1'],
        ['ATP8B2','PTRF'],
        ['ATP8B5P','ZFAND6'],
        ['ATP9A','FN1'],
        ['ATP9A','PLCE1'],
        ['ATPAF1','EFCAB14'],
        ['ATRX','MALAT1'],
        ['ATXN1L','IST1'],
        ['ATXN3','MAB21L1'],
        ['ATXN7','PSMD6-AS2'],
        ['ATXN7L3','COL1A1'],
        ['AUP1','COL1A1'],
        ['AUP1','SFTPC'],
        ['AURKA','AURKAPS1'],
        ['AURKA','RAB3GAP2'],
        ['AXL','COL1A1'],
        ['AXL','COL1A2'],
        ['AXL','COL6A2'],
        ['AXL','ECE1'],
        ['AXL','FBLIM1'],
        ['AXL','FN1'],
        ['AXL','HNRNPUL1'],
        ['AXL','LEPREL4'],
        ['AXL','MYH9'],
        ['AXL','PCSK7'],
        ['AXL','PTRF'],
        ['AXL','SFTPB'],
        ['AZGP1','AZGP1P1'],
        ['AZGP1','GJC3'],
        ['AZU1','DDX39B'],
        ['AZU1','ILF3'],
        ['AZU1','LCP1'],
        ['B2M','CD74'],
        ['B2M','COL1A1'],
        ['B2M','COL3A1'],
        ['B2M','CTSB'],
        ['B2M','DCN'],
        ['B2M','DOCK4'],
        ['B2M','EEF2'],
        ['B2M','EMP1'],
        ['B2M','EMP2'],
        ['B2M','EPAS1'],
        ['B2M','FGA'],
        ['B2M','FLNA'],
        ['B2M','FN1'],
        ['B2M','FOSB'],
        ['B2M','FTL'],
        ['B2M','GLUL'],
        ['B2M','GNAS'],
        ['B2M','GRN'],
        ['B2M','HLA-A'],
        ['B2M','HLA-B'],
        ['B2M','HLA-C'],
        ['B2M','HLA-DRA'],
        ['B2M','HLA-E'],
        ['B2M','IGHA1'],
        ['B2M','IGKC'],
        ['B2M','MALAT1'],
        ['B2M','MYH9'],
        ['B2M','NEAT1'],
        ['B2M','PLEC'],
        ['B2M','PSAP'],
        ['B2M','RP11-849H4.4'],
        ['B2M','SFTPA1'],
        ['B2M','SFTPA2'],
        ['B2M','SFTPB'],
        ['B2M','SFTPC'],
        ['B2M','SLC34A2'],
        ['B2M','SOD2'],
        ['B2M','SPTBN1'],
        ['B2M','TG'],
        ['B2M','THBS1'],
        ['B2M','TPT1'],
        ['B2M','TXNIP'],
        ['B2M','VWF'],
        ['B4GALT1','COL1A1'],
        ['B4GALT1','FN1'],
        ['B4GALT2','COL1A1'],
        ['B4GALT2','FN1'],
        ['B4GALT5','SERPINE1'],
        ['B4GALT7','RP11-1277A3.1'],
        ['BACE1','CITED2'],
        ['BACH1','BACH1-IT1'],
        ['BACH1','LINC00189'],
        ['BACH2','CPA4'],
        ['BAG3','FN1'],
        ['BAG3','TG'],
        ['BAG6','C6ORF47'],
        ['BAG6','COL18A1'],
        ['BAG6','COL1A1'],
        ['BAG6','FN1'],
        ['BAG6','SFTPB'],
        ['BAG6','SFTPC'],
        ['BAG6','SOD2'],
        ['BAIAP2','SFTPC'],
        ['BAIAP2L2','SLC16A8'],
        ['BANP','RP11-426C22.5'],
        ['BANP','RRN3P2'],
        ['BASP1','CAPNS1'],
        ['BASP1','CLPTM1L'],
        ['BASP1','COL1A1'],
        ['BAX','LEPREL4'],
        ['BAX','PTRF'],
        ['BAZ2A','SFTPC'],
        ['BAZ2B','CAPZA1'],
        ['BBS5','DTD2'],
        ['BBS5','RP11-176H8.1'],
        ['BCAM','EPAS1'],
        ['BCAM','FN1'],
        ['BCAM','MYH9'],
        ['BCAM','SFTPA1'],
        ['BCAM','SFTPA2'],
        ['BCAM','SFTPB'],
        ['BCAM','SFTPC'],
        ['BCAM','TG'],
        ['BCAN','SPARC'],
        ['BCAP29','DUS4L'],
        ['BCAP29','WBP1LP2'],
        ['BCAR1','COL1A1'],
        ['BCAR1','FN1'],
        ['BCAS2','DENND2C'],
        ['BCAT1','RP11-625L16.4'],
        ['BCKDK','FLNA'],
        ['BCL2L1','FN1'],
        ['BCL2L1','SFTPA1'],
        ['BCL2L12','PRMT1'],
        ['BCL6','SERPINA1'],
        ['BCL7C','EHD2'],
        ['BCL9L','CABIN1'],
        ['BCL9L','COL1A1'],
        ['BCL9L','COL1A2'],
        ['BCL9L','COL6A2'],
        ['BCL9L','FN1'],
        ['BCL9L','MYH9'],
        ['BCL9L','SFTPC'],
        ['BCL9L','SLC2A4RG'],
        ['BCL9L','SMR3B'],
        ['BCR','BCRP3'],
        ['BCR','COL1A1'],
        ['BEST1','COL1A1'],
        ['BEX1','BEX2'],
        ['BEX4','TG'],
        ['BGN','CD74'],
        ['BGN','CD93'],
        ['BGN','COL1A1'],
        ['BGN','COL1A2'],
        ['BGN','FLNA'],
        ['BGN','FN1'],
        ['BGN','GADL1'],
        ['BGN','IGFBP4'],
        ['BGN','IGKC'],
        ['BGN','MYH9'],
        ['BGN','NR4A1'],
        ['BGN','SFTPA1'],
        ['BGN','SFTPA2'],
        ['BGN','SFTPB'],
        ['BGN','SFTPC'],
        ['BGN','THBS1'],
        ['BHLHB9','LINC00630'],
        ['BHLHE40','CD74'],
        ['BHLHE40','SFTPA2'],
        ['BHLHE40','SFTPB'],
        ['BHLHE40','SFTPC'],
        ['BICD2','PTMS'],
        ['BIN1','TG'],
        ['BIRC6','YIPF4'],
        ['BLID','RP11-166D19.1'],
        ['BLOC1S3','PTRF'],
        ['BLOC1S5-TXNDC5','COL1A1'],
        ['BLOC1S5-TXNDC5','COL1A2'],
        ['BLOC1S5-TXNDC5','FN1'],
        ['BMP1','COL1A1'],
        ['BMP1','SFTPC'],
        ['BMS1P10','BMS1P16'],
        ['BMS1P20','IGLC1'],
        ['BMS1P20','IGLC2'],
        ['BMS1P20','IGLL5'],
        ['BMS1P20','IGLV1-41'],
        ['BMS1P20','LL22NC03-80A10.6'],
        ['BMS1P4','RP11-464F9.1'],
        ['BOK','FN1'],
        ['BPIFA2','LYZ'],
        ['BRAF','BRAFP1'],
        ['BRCA1','RP11-242D8.3'],
        ['BRD2','COL1A1'],
        ['BRD2','FN1'],
        ['BRD2','SFTPB'],
        ['BRD2','SFTPC'],
        ['BRD4','COL1A1'],
        ['BRD4','SFTPC'],
        ['BRD9','ZDHHC11B'],
        ['BRDT','EPHX4'],
        ['BRF2','SFTPA1'],
        ['BRI3','RP4-607J23.2'],
        ['BRICD5','PGP'],
        ['BRS3','HTATSF1'],
        ['BRWD1','RP11-707O23.1'],
        ['BRWD1-AS1','BRWD1-AS2'],
        ['BSG','CKAP4'],
        ['BSG','COL1A1'],
        ['BSG','COL1A2'],
        ['BSG','COL6A1'],
        ['BSG','COL6A2'],
        ['BSG','COL6A3'],
        ['BSG','FLNA'],
        ['BSG','FN1'],
        ['BSG','MYH11'],
        ['BSG','NR1H2'],
        ['BSG','PKM'],
        ['BSG','PLEC'],
        ['BSG','SFTPB'],
        ['BSG','SFTPC'],
        ['BSG','SPARC'],
        ['BSG','THBS1'],
        ['BST1','FAM200B'],
        ['BTBD1','COL6A1'],
        ['BTBD10','RP11-641A6.9'],
        ['BTBD2','CAPNS1'],
        ['BTBD2','COL1A1'],
        ['BTBD2','FN1'],
        ['BTBD6','COL1A1'],
        ['BTBD6','LOXL1'],
        ['BTBD8','KIAA1107'],
        ['BTF3','BTF3P8'],
        ['BTG1','SFTPC'],
        ['BTG2','SFTPA1'],
        ['BTG2','TPO'],
        ['BTG4','POU2AF1'],
        ['BTN2A1','BTN3A3'],
        ['BTN3A1','BTN3A2'],
        ['BTN3A2','BTN3A3'],
        ['BTNL3','BTNL8'],
        ['BX470102.3','S100A6'],
        ['BX649563.4','GPR116'],
        ['C10ORF10','EPAS1'],
        ['C10ORF10','RP1-309I22.2'],
        ['C10ORF10','SFTPA1'],
        ['C10ORF10','SFTPA2'],
        ['C10ORF10','SFTPB'],
        ['C10ORF10','SFTPC'],
        ['C10ORF10','TIMP3'],
        ['C10ORF12','LCOR'],
        ['C10ORF25','CEP164P1'],
        ['C10ORF32','LYZ'],
        ['C10ORF68','CCDC7'],
        ['C11ORF24','COL1A1'],
        ['C11ORF24','FN1'],
        ['C11ORF31','CTNND1'],
        ['C11ORF48','UBXN1'],
        ['C11ORF58','RP11-305B6.1'],
        ['C11ORF68','COL1A1'],
        ['C11ORF73','EED'],
        ['C11ORF95','COL1A1'],
        ['C11ORF96','RGCC'],
        ['C12ORF74','PLEKHG7'],
        ['C12ORF75','PTOV1'],
        ['C12ORF76','RP11-946P6.4'],
        ['C12ORF77','FAM129A'],
        ['C14ORF80','CRIP1'],
        ['C15ORF26','IL16'],
        ['C15ORF38-AP3S2','FBLIM1'],
        ['C15ORF38-AP3S2','PTRF'],
        ['C15ORF38-AP3S2','SFTPB'],
        ['C15ORF57','CBX3'],
        ['C16ORF72','RP11-473I1.10'],
        ['C16ORF74','GINS2'],
        ['C16ORF89','SFTPA2'],
        ['C16ORF89','SFTPB'],
        ['C16ORF89','SFTPC'],
        ['C17ORF108','NOS2'],
        ['C17ORF47','SEPT4'],
        ['C17ORF49','RNASEK-C17ORF49'],
        ['C17ORF51','RP11-822E23.8'],
        ['C17ORF61-PLSCR3','COL1A1'],
        ['C17ORF85','CTD-3195I5.5'],
        ['C19MC','TIMM50'],
        ['C19ORF10','COL1A1'],
        ['C19ORF10','COL1A2'],
        ['C19ORF10','FLNA'],
        ['C19ORF10','TNC'],
        ['C19ORF18','ZNF606'],
        ['C19ORF24','CIRBP'],
        ['C19ORF43','LAMC1'],
        ['C19ORF59','CTD-3214H19.16'],
        ['C19ORF59','TRAPPC5'],
        ['C19ORF66','SFTPC'],
        ['C1ORF122','IFFO2'],
        ['C1ORF186','RP11-38J22.6'],
        ['C1ORF198','COL1A1'],
        ['C1ORF227','FLVCR1-AS1'],
        ['C1ORF74','IRF6'],
        ['C1ORF86','OSMR'],
        ['C1ORF86','RP11-181G12.2'],
        ['C1QA','C1QC'],
        ['C1QA','SFTPB'],
        ['C1QA','SFTPC'],
        ['C1QB','SFTPA1'],
        ['C1QB','SFTPA2'],
        ['C1QB','SFTPC'],
        ['C1QBP','CTC-524C5.2'],
        ['C1QC','CD74'],
        ['C1QC','SFTPA1'],
        ['C1QC','SFTPA2'],
        ['C1QC','SFTPB'],
        ['C1QC','SFTPC'],
        ['C1QTNF1','RTN4'],
        ['C1QTNF9','C1QTNF9B'],
        ['C1R','IGFBP5'],
        ['C1R','SERPING1'],
        ['C1R','SFTPB'],
        ['C1R','TG'],
        ['C1RL-AS1','CLSTN3'],
        ['C1RL-AS1','U6'],
        ['C1S','HLA-E'],
        ['C1S','HP'],
        ['C1S','KRT10'],
        ['C1S','SFTPC'],
        ['C2','CFB'],
        ['C21ORF33','PWP2'],
        ['C22ORF29','GNB1L'],
        ['C22ORF43','KB-1572G7.3'],
        ['C2ORF66','PGAP1'],
        ['C2ORF74','KIAA1841'],
        ['C3','CD74'],
        ['C3','COL1A1'],
        ['C3','COL1A2'],
        ['C3','COL3A1'],
        ['C3','COL4A1'],
        ['C3','COL6A2'],
        ['C3','CRP'],
        ['C3','DHCR24'],
        ['C3','EEF2'],
        ['C3','EPAS1'],
        ['C3','FGA'],
        ['C3','FGB'],
        ['C3','FGG'],
        ['C3','FLNA'],
        ['C3','FN1'],
        ['C3','FOSB'],
        ['C3','FTL'],
        ['C3','HLA-B'],
        ['C3','HLA-DRA'],
        ['C3','HLA-E'],
        ['C3','HP'],
        ['C3','IGFBP5'],
        ['C3','MYH9'],
        ['C3','NCOR2'],
        ['C3','NR4A1'],
        ['C3','PSAP'],
        ['C3','SERPINA1'],
        ['C3','SFTPA1'],
        ['C3','SFTPA2'],
        ['C3','SFTPB'],
        ['C3','SFTPC'],
        ['C3','SPARC'],
        ['C3','SPTBN1'],
        ['C3','TG'],
        ['C3','TIMP3'],
        ['C3','TNFSF14'],
        ['C3','ZYX'],
        ['C3ORF17','RP11-572M11.3'],
        ['C3ORF62','USP4'],
        ['C4A','C4B'],
        ['C4ORF3','CD36'],
        ['C4ORF3','FABP2'],
        ['C4ORF36','SLC10A6'],
        ['C4ORF48','EIF5A'],
        ['C4ORF48','RAB2A'],
        ['C5','SERPINA1'],
        ['C5AR1','CD84'],
        ['C5ORF4','CCDC106'],
        ['C5ORF45','CD74'],
        ['C5ORF45','COL1A1'],
        ['C5ORF45','FLNC'],
        ['C5ORF45','FN1'],
        ['C5ORF45','IGF2'],
        ['C5ORF45','SFTPA1'],
        ['C5ORF45','SFTPA2'],
        ['C5ORF45','SFTPC'],
        ['C5ORF45','TBC1D9B'],
        ['C5ORF46','SPINK1'],
        ['C5ORF58','LCP2'],
        ['C6ORF163','C6ORF164'],
        ['C6ORF163','RP1-102H19.6'],
        ['C6ORF163','SMIM8'],
        ['C6ORF201','RP3-400B16.4'],
        ['C6ORF48','FLNA'],
        ['C7','IGF2'],
        ['C7','IGFBP5'],
        ['C7','IGH@'],
        ['C7','SFTPA1'],
        ['C7','SFTPA2'],
        ['C7','SFTPC'],
        ['C7','TIMP2'],
        ['C7ORF49','WDR91'],
        ['C7ORF50','GNA12'],
        ['C7ORF55','C7ORF55-LUC7L2'],
        ['C7ORF55','LUC7L2'],
        ['C7ORF55-LUC7L2','LUC7L2'],
        ['C8ORF44','C8ORF44-SGK3'],
        ['C8ORF58','CCAR2'],
        ['C8ORF58','KIAA1967'],
        ['C8ORF58','PDLIM2'],
        ['C8ORF58','PTMS'],
        ['C8ORF76','ZHX1'],
        ['C8ORF76','ZHX1-C8ORF76'],
        ['C9ORF114','ENDOG'],
        ['C9ORF172','PHPT1'],
        ['C9ORF69','RTN4'],
        ['CA5B','CA5BP1'],
        ['CABIN1','COL1A1'],
        ['CABIN1','KB-318B8.7'],
        ['CABLES2','KRT19'],
        ['CACNA1C','CACNA1C-IT2'],
        ['CACNA1C-AS1','CACNA1C-AS2'],
        ['CACNA1D','SRSF8'],
        ['CACNA2D2','FN1'],
        ['CACNA2D2','SFTPA1'],
        ['CACNA2D2','SFTPA2'],
        ['CACNA2D2','SFTPC'],
        ['CACTIN','PPP1R14B'],
        ['CAD','COL1A1'],
        ['CADM3','DARC'],
        ['CADM4','PLAUR'],
        ['CADPS2','RNF148'],
        ['CALCOCO1','SFTPC'],
        ['CALD1','COL1A1'],
        ['CALD1','COL1A2'],
        ['CALD1','FLNA'],
        ['CALD1','FN1'],
        ['CALD1','GREM1'],
        ['CALD1','IGF2'],
        ['CALD1','SFTPC'],
        ['CALD1','THBS1'],
        ['CALD1','VIM'],
        ['CALHM2','COL1A1'],
        ['CALM1','CALM1P1'],
        ['CALM1','COL1A1'],
        ['CALM1','SFTPA2'],
        ['CALM1','SFTPC'],
        ['CALM2','CALM2P2'],
        ['CALM2','CALM2P3'],
        ['CALM2','COL1A1'],
        ['CALM2','FN1'],
        ['CALM2','SFTPC'],
        ['CALM3','CAMK2N1'],
        ['CALM3','COL1A1'],
        ['CALM3','COL6A2'],
        ['CALM3','FLNA'],
        ['CALM3','FN1'],
        ['CALM3','FTH1'],
        ['CALM3','IGFBP5'],
        ['CALM3','NRGN'],
        ['CALM3','RTN4'],
        ['CALM3','SFTPA1'],
        ['CALM3','SFTPB'],
        ['CALM3','SPARC'],
        ['CALM3','VAMP2'],
        ['CALM3','VAT1'],
        ['CALML4','CLN6'],
        ['CALR','CAPN2'],
        ['CALR','CD248'],
        ['CALR','COL1A1'],
        ['CALR','COL1A2'],
        ['CALR','COL3A1'],
        ['CALR','COL4A2'],
        ['CALR','COL5A1'],
        ['CALR','COL6A1'],
        ['CALR','COL6A2'],
        ['CALR','COL6A3'],
        ['CALR','EEF2'],
        ['CALR','FGA'],
        ['CALR','FLNA'],
        ['CALR','FLNC'],
        ['CALR','FN1'],
        ['CALR','GNAS'],
        ['CALR','IGFBP5'],
        ['CALR','LMNA'],
        ['CALR','LRP1'],
        ['CALR','MDN1'],
        ['CALR','MMP1'],
        ['CALR','MYH9'],
        ['CALR','MYL6'],
        ['CALR','MYL9'],
        ['CALR','NCL'],
        ['CALR','PKM'],
        ['CALR','PLEC'],
        ['CALR','PLIN1'],
        ['CALR','PTMS'],
        ['CALR','PTRF'],
        ['CALR','RCN3'],
        ['CALR','RTN4'],
        ['CALR','SERPINA1'],
        ['CALR','SFTPA1'],
        ['CALR','SFTPA2'],
        ['CALR','SFTPC'],
        ['CALR','SPARC'],
        ['CALR','TG'],
        ['CALR','THBS1'],
        ['CALR','TIMP3'],
        ['CALR','TPM2'],
        ['CALR','UBXN11'],
        ['CALR','VIM'],
        ['CALR','VIM-AS1'],
        ['CALR3','EPS15L1'],
        ['CALU','COL1A1'],
        ['CALU','COL1A2'],
        ['CALU','CTSB'],
        ['CALU','FN1'],
        ['CAMK2A','KIF5A'],
        ['CAMK2N1','MBP'],
        ['CAMK2N1','RP11-386G11.10'],
        ['CAMK2N1','VSNL1'],
        ['CAMKK2','COL1A1'],
        ['CAMTA1','CAMTA1-IT1'],
        ['CAMTA2','SPAG7'],
        ['CAND1','PTMS'],
        ['CANT1','FSCN1'],
        ['CANX','COL1A1'],
        ['CANX','COL1A2'],
        ['CANX','FN1'],
        ['CANX','HMGB3P22'],
        ['CANX','MUC7'],
        ['CANX','MYH9'],
        ['CANX','NEAT1'],
        ['CANX','SFTPB'],
        ['CANX','SFTPC'],
        ['CANX','TG'],
        ['CANX','VIM'],
        ['CAP1','COL1A1'],
        ['CAP1','FN1'],
        ['CAP1','RP11-550F7.1'],
        ['CAP1','SERPINA1'],
        ['CAP1','SFTPC'],
        ['CAP1','SPP1'],
        ['CAPN1','COL1A1'],
        ['CAPN1','COL1A2'],
        ['CAPN1','FN1'],
        ['CAPN1','MYH9'],
        ['CAPN1','THBS1'],
        ['CAPN15','COL1A1'],
        ['CAPN2','COL1A1'],
        ['CAPN2','COL1A2'],
        ['CAPN2','FN1'],
        ['CAPN2','SFTPA1'],
        ['CAPN2','SFTPB'],
        ['CAPN2','SFTPC'],
        ['CAPN3','GANC'],
        ['CAPN3','RP11-164J13.1'],
        ['CAPNS1','CD74'],
        ['CAPNS1','CDK2AP1'],
        ['CAPNS1','COL1A1'],
        ['CAPNS1','COL6A1'],
        ['CAPNS1','COL6A2'],
        ['CAPNS1','FLNA'],
        ['CAPNS1','FN1'],
        ['CAPNS1','FOXD1'],
        ['CAPNS1','FTH1'],
        ['CAPNS1','GNAS'],
        ['CAPNS1','IRF2BPL'],
        ['CAPNS1','LOXL1'],
        ['CAPNS1','MAP7D1'],
        ['CAPNS1','MAZ'],
        ['CAPNS1','PLEC'],
        ['CAPNS1','PRKACA'],
        ['CAPNS1','PTMS'],
        ['CAPNS1','SFTPA2'],
        ['CAPNS1','SFTPB'],
        ['CAPNS1','SFTPC'],
        ['CAPNS1','VEGFB'],
        ['CAPNS1','VIM-AS1'],
        ['CAPNS1','ZNF703'],
        ['CAPRIN1','COL1A1'],
        ['CAPRIN1','KIF5B'],
        ['CAPS','VMAC'],
        ['CAPZB','COL1A1'],
        ['CAPZB','FN1'],
        ['CAPZB','IGF2'],
        ['CAPZB','RIN3'],
        ['CAPZB','SFTPB'],
        ['CARD16','CASP1'],
        ['CARD9','COL6A2'],
        ['CARHSP1','PTRF'],
        ['CARM1','FN1'],
        ['CARM1','GNAS'],
        ['CARM1','PTMS'],
        ['CARM1','TG'],
        ['CARM1','YIPF2'],
        ['CARS','FN1'],
        ['CASC4','PTRF'],
        ['CASC4','SFTPB'],
        ['CASC4','TPM4'],
        ['CASP2','TMEM139'],
        ['CAT','EPAS1'],
        ['CAT','SFTPC'],
        ['CATSPER2','CATSPER2P1'],
        ['CATSPERB','TC2N'],
        ['CAV1','CLU'],
        ['CAV1','COL1A1'],
        ['CAV1','COL1A2'],
        ['CAV1','COL6A1'],
        ['CAV1','EMP2'],
        ['CAV1','FN1'],
        ['CAV1','MYH9'],
        ['CAV1','SFTPA1'],
        ['CAV1','SFTPA2'],
        ['CAV1','SFTPB'],
        ['CAV1','SFTPC'],
        ['CAV1','TG'],
        ['CAV1','TIMP3'],
        ['CAV2','SFTPA2'],
        ['CBWD1','CBWD2'],
        ['CBX4','PTMS'],
        ['CBX5','COL1A1'],
        ['CBX5','PTRF'],
        ['CBX6','FN1'],
        ['CBX7','RP4-742C19.8'],
        ['CC2D2A','CLCC1'],
        ['CCDC11','MBD1'],
        ['CCDC124','SPARC'],
        ['CCDC13','HHATL'],
        ['CCDC144NL-AS1','RP11-381P6.1'],
        ['CCDC149','COL6A2'],
        ['CCDC150P1','CLIP1'],
        ['CCDC152','CD74'],
        ['CCDC152','COL1A1'],
        ['CCDC152','HLA-B'],
        ['CCDC152','MYH9'],
        ['CCDC152','PKM'],
        ['CCDC152','PSAP'],
        ['CCDC152','SFTPA2'],
        ['CCDC152','SFTPB'],
        ['CCDC152','SFTPC'],
        ['CCDC152','SLC34A2'],
        ['CCDC152','SPARC'],
        ['CCDC152','SPARCL1'],
        ['CCDC152','SPTBN1'],
        ['CCDC169','SOHLH2'],
        ['CCDC183','RABL6'],
        ['CCDC183','RP11-216L13.19'],
        ['CCDC183','TMEM141'],
        ['CCDC33','RP11-60L3.1'],
        ['CCDC50','SFTPB'],
        ['CCDC50','SFTPC'],
        ['CCDC58','WDR5B'],
        ['CCDC69','LYZ'],
        ['CCDC7','CCDC7'],
        ['CCDC74A','CCDC74B'],
        ['CCDC8','SFTPB'],
        ['CCDC80','COL1A1'],
        ['CCDC80','COL1A2'],
        ['CCDC80','FN1'],
        ['CCDC80','MYH9'],
        ['CCDC80','RP11-572C15.6'],
        ['CCDC80','SFTPB'],
        ['CCDC85A','RP11-482H16.1'],
        ['CCDC85B','COL1A1'],
        ['CCDC85B','EVA1B'],
        ['CCDC85B','FN1'],
        ['CCDC85C','FN1'],
        ['CCDC86','COL1A1'],
        ['CCDC92','COL1A1'],
        ['CCDC92','RP11-214K3.18'],
        ['CCL21','IGH@'],
        ['CCL21','IGHA1'],
        ['CCL21','IGHG1'],
        ['CCL21','IGHM'],
        ['CCL21','IGK@'],
        ['CCL21','IGKC'],
        ['CCL3','CCL3L3'],
        ['CCL4','CCL4L1'],
        ['CCL5','LYZ'],
        ['CCL5','SFTPC'],
        ['CCM2','GAS6'],
        ['CCND1','COL1A1'],
        ['CCND1','COL1A2'],
        ['CCND1','COL6A1'],
        ['CCND1','COL6A2'],
        ['CCND1','COL6A3'],
        ['CCND1','CXCL12'],
        ['CCND1','CYR61'],
        ['CCND1','FLNA'],
        ['CCND1','FN1'],
        ['CCND1','GREM1'],
        ['CCND1','IGFBP3'],
        ['CCND1','LZTS2'],
        ['CCND1','MYH9'],
        ['CCND1','PLEC'],
        ['CCND1','SERPINE2'],
        ['CCND1','SFTPC'],
        ['CCND1','THBS1'],
        ['CCND1','TIMP1'],
        ['CCND1','VIM'],
        ['CCND1','VIM-AS1'],
        ['CCND3','SFTPC'],
        ['CCNF','PTRF'],
        ['CCNH','CTC-428H11.2'],
        ['CCNI','FLNA'],
        ['CCNI','TG'],
        ['CCNI','TMEM160'],
        ['CCNK','COL1A1'],
        ['CCNK','SFTPA1'],
        ['CCNL2','SFTPC'],
        ['CCP110','TMC5'],
        ['CCPG1','PIGBOS1'],
        ['CCPG1','RP11-139H15.1'],
        ['CCR10','EZH1'],
        ['CCT3','COL1A1'],
        ['CCT5','COL1A1'],
        ['CCT6P1','CCT6P3'],
        ['CCT7','COL1A1'],
        ['CCT7','COL1A2'],
        ['CCT7','FN1'],
        ['CCT8','CCT8P1'],
        ['CD151','COL1A1'],
        ['CD151','COL1A2'],
        ['CD151','COL6A3'],
        ['CD151','FLNA'],
        ['CD151','FLNC'],
        ['CD151','FN1'],
        ['CD151','MYH9'],
        ['CD151','PLEC'],
        ['CD151','PTBP1'],
        ['CD151','SFTPA1'],
        ['CD151','SFTPA2'],
        ['CD151','SFTPB'],
        ['CD151','SFTPC'],
        ['CD151','TGFBI'],
        ['CD163','SFTPC'],
        ['CD22','MAG'],
        ['CD22','U62631.5'],
        ['CD24','COL6A2'],
        ['CD248','COL1A1'],
        ['CD248','COL1A2'],
        ['CD248','COL6A1'],
        ['CD248','COL6A2'],
        ['CD248','COL6A3'],
        ['CD248','FLNA'],
        ['CD248','FN1'],
        ['CD248','IGFBP3'],
        ['CD248','PLEC'],
        ['CD248','RP5-977B1.11'],
        ['CD248','SERPINE1'],
        ['CD248','SPARC'],
        ['CD248','TGFBI'],
        ['CD248','THBS1'],
        ['CD248','VIM'],
        ['CD248','ZYX'],
        ['CD27-AS1','VAMP1'],
        ['CD276','COL1A1'],
        ['CD300A','CD300C'],
        ['CD320','NDUFA7'],
        ['CD36','EHD2'],
        ['CD37','SFTPA1'],
        ['CD3EAP','ERCC1'],
        ['CD4','SFTPA1'],
        ['CD44','CD74'],
        ['CD44','COL1A1'],
        ['CD44','COL1A2'],
        ['CD44','COL6A2'],
        ['CD44','FN1'],
        ['CD44','SFTPA2'],
        ['CD44','SFTPC'],
        ['CD44','SPARC'],
        ['CD44','THBS1'],
        ['CD44','VIM'],
        ['CD47','SFTPA2'],
        ['CD47','SFTPB'],
        ['CD55','SFTPC'],
        ['CD59','COL1A1'],
        ['CD59','COL1A2'],
        ['CD59','FN1'],
        ['CD59','SFTPC'],
        ['CD59','TIMP3'],
        ['CD5L','FCRL1'],
        ['CD63','COL1A1'],
        ['CD63','COL1A2'],
        ['CD63','COL6A1'],
        ['CD63','COL6A2'],
        ['CD63','FLNA'],
        ['CD63','FN1'],
        ['CD63','MYH9'],
        ['CD63','RP5-940J5.9'],
        ['CD63','SFTPA2'],
        ['CD63','SFTPB'],
        ['CD63','SFTPC'],
        ['CD63','TG'],
        ['CD63','TPM2'],
        ['CD68','CNN2'],
        ['CD68','EPAS1'],
        ['CD68','FN1'],
        ['CD68','HLA-B'],
        ['CD68','IGHA1'],
        ['CD68','LAMP3'],
        ['CD68','LYZ'],
        ['CD68','MUC5B'],
        ['CD68','RP11-1143G9.4'],
        ['CD68','SCUBE3'],
        ['CD68','SFTPA1'],
        ['CD68','SFTPA2'],
        ['CD68','SFTPB'],
        ['CD68','SFTPC'],
        ['CD74','CD81'],
        ['CD74','CDKN1A'],
        ['CD74','COL1A1'],
        ['CD74','COL1A2'],
        ['CD74','COL3A1'],
        ['CD74','COL4A1'],
        ['CD74','COL4A2'],
        ['CD74','COL6A1'],
        ['CD74','COL6A2'],
        ['CD74','COL6A3'],
        ['CD74','CTSB'],
        ['CD74','CTSD'],
        ['CD74','CTTN'],
        ['CD74','DHCR24'],
        ['CD74','DUSP1'],
        ['CD74','EEF2'],
        ['CD74','EGR1'],
        ['CD74','EIF4G1'],
        ['CD74','EMP1'],
        ['CD74','EMP2'],
        ['CD74','EPAS1'],
        ['CD74','ERGIC1'],
        ['CD74','EZR'],
        ['CD74','FAM65A'],
        ['CD74','FLNA'],
        ['CD74','FN1'],
        ['CD74','FOS'],
        ['CD74','FOSB'],
        ['CD74','FOSL2'],
        ['CD74','FTL'],
        ['CD74','GLUL'],
        ['CD74','GNAS'],
        ['CD74','GPRC5A'],
        ['CD74','GRN'],
        ['CD74','HDLBP'],
        ['CD74','HEG1'],
        ['CD74','HIF3A'],
        ['CD74','HLA-A'],
        ['CD74','HLA-B'],
        ['CD74','HLA-C'],
        ['CD74','HLA-DPA1'],
        ['CD74','HLA-DRA'],
        ['CD74','HLA-E'],
        ['CD74','HSPG2'],
        ['CD74','ICAM1'],
        ['CD74','IFI30'],
        ['CD74','IGH@'],
        ['CD74','IGHA1'],
        ['CD74','IGHG1'],
        ['CD74','IGHG4'],
        ['CD74','IGHM'],
        ['CD74','IGK@'],
        ['CD74','IGKC'],
        ['CD74','IGKV1-8'],
        ['CD74','IGKV3-11'],
        ['CD74','ITGB2'],
        ['CD74','JUN'],
        ['CD74','JUNB'],
        ['CD74','JUP'],
        ['CD74','LAMA5'],
        ['CD74','LAPTM5'],
        ['CD74','LIMCH1'],
        ['CD74','LMNA'],
        ['CD74','LRP1'],
        ['CD74','LRP10'],
        ['CD74','LTBP3'],
        ['CD74','LYZ'],
        ['CD74','MALAT1'],
        ['CD74','MCL1'],
        ['CD74','MUC4'],
        ['CD74','MYADM'],
        ['CD74','MYH9'],
        ['CD74','NAPSA'],
        ['CD74','NCOR2'],
        ['CD74','NEAT1'],
        ['CD74','NOTCH1'],
        ['CD74','NR4A1'],
        ['CD74','PER1'],
        ['CD74','PFN1'],
        ['CD74','PIK3R2'],
        ['CD74','PLEC'],
        ['CD74','PLXND1'],
        ['CD74','POLR2A'],
        ['CD74','PPP1R9B'],
        ['CD74','PRRC2B'],
        ['CD74','PSAP'],
        ['CD74','PTMS'],
        ['CD74','PTRF'],
        ['CD74','RNASE1'],
        ['CD74','RP11-186B7.4'],
        ['CD74','RP11-831H9.16'],
        ['CD74','RP5-977B1.11'],
        ['CD74','RPL8'],
        ['CD74','S100A6'],
        ['CD74','SELENBP1'],
        ['CD74','SERPING1'],
        ['CD74','SFTPA1'],
        ['CD74','SFTPA2'],
        ['CD74','SFTPB'],
        ['CD74','SFTPC'],
        ['CD74','SFTPD'],
        ['CD74','SLC34A2'],
        ['CD74','SOD2'],
        ['CD74','SPARC'],
        ['CD74','SPTBN1'],
        ['CD74','SRRM2'],
        ['CD74','SUSD2'],
        ['CD74','TAGLN2'],
        ['CD74','TFRC'],
        ['CD74','TGM2'],
        ['CD74','THBS1'],
        ['CD74','TIMP3'],
        ['CD74','TLN1'],
        ['CD74','TMBIM6'],
        ['CD74','TNFAIP2'],
        ['CD74','TNS1'],
        ['CD74','TPT1'],
        ['CD74','TUBB6'],
        ['CD74','UBC'],
        ['CD74','VAT1'],
        ['CD74','VIM'],
        ['CD74','VIPR1'],
        ['CD74','VWF'],
        ['CD74','ZFP36'],
        ['CD74','ZFP36L1'],
        ['CD81','CHSY1'],
        ['CD81','COL1A1'],
        ['CD81','COL1A2'],
        ['CD81','COL6A2'],
        ['CD81','FN1'],
        ['CD81','GNAS'],
        ['CD81','IGFBP3'],
        ['CD81','KIAA2013'],
        ['CD81','SFTPA1'],
        ['CD81','SFTPA2'],
        ['CD81','SFTPB'],
        ['CD81','SFTPC'],
        ['CD81','TSSC4'],
        ['CD81','TWIST1'],
        ['CD81','YKT6'],
        ['CD9','SFTPC'],
        ['CD93','DCN'],
        ['CD93','FN1'],
        ['CD93','IGF2'],
        ['CD93','NCOR2'],
        ['CD93','RP11-1100L3.8'],
        ['CD93','SFTPA1'],
        ['CD93','SFTPA2'],
        ['CD93','SFTPB'],
        ['CD93','SFTPC'],
        ['CD97','EMR2'],
        ['CD97','FN1'],
        ['CD97','SFTPC'],
        ['CD99','CD99P1'],
        ['CD99','COL1A1'],
        ['CD99','COL1A2'],
        ['CD99','COL6A2'],
        ['CD99','FLNA'],
        ['CD99','FLNC'],
        ['CD99','FN1'],
        ['CD99','MYH9'],
        ['CD99L2','FN1'],
        ['CDC14A','RP5-837M10.2'],
        ['CDC20','CDC20P1'],
        ['CDC20','COL1A1'],
        ['CDC25B','COL1A1'],
        ['CDC25B','FN1'],
        ['CDC25B','SFTPC'],
        ['CDC34','VASN'],
        ['CDC37','COL1A1'],
        ['CDC37','SFTPA2'],
        ['CDC37','SFTPC'],
        ['CDC37','TG'],
        ['CDC42BPB','COL1A1'],
        ['CDC42BPB','SFTPC'],
        ['CDC42EP1','COL1A1'],
        ['CDC42EP1','FN1'],
        ['CDC42EP1','TMEM200A'],
        ['CDC42EP2','POLA2'],
        ['CDC42EP3','COL1A1'],
        ['CDC42EP3','FN1'],
        ['CDC42EP4','CTSB'],
        ['CDC42EP4','TG'],
        ['CDCP1','SERPINE1'],
        ['CDH1','TG'],
        ['CDH1','TPO'],
        ['CDH11','COL1A1'],
        ['CDH11','FN1'],
        ['CDH11','VIM'],
        ['CDH2','FN1'],
        ['CDH24','GPC1'],
        ['CDH5','EPAS1'],
        ['CDH5','SFTPA1'],
        ['CDH5','SFTPB'],
        ['CDH5','SFTPC'],
        ['CDIPT','COL1A1'],
        ['CDIPT','RTN4'],
        ['CDK11A','CDK11B'],
        ['CDK11B','IGHG1'],
        ['CDK11B','RP1-283E3.8'],
        ['CDK11B','SLC35E2'],
        ['CDK12','RP11-390P24.1'],
        ['CDK12','SFTPB'],
        ['CDK14','CLDN12'],
        ['CDK14','CTB-13L3.1'],
        ['CDK16','TXNDC5'],
        ['CDK2','RAB5B'],
        ['CDK2AP1','COL1A1'],
        ['CDK2AP1','FN1'],
        ['CDK2AP1','GNAS'],
        ['CDK2AP1','TRIM28'],
        ['CDKL3','PPP2CA'],
        ['CDKN1A','CNPY2'],
        ['CDKN1A','COL1A1'],
        ['CDKN1A','FAM89B'],
        ['CDKN1A','FLNA'],
        ['CDKN1A','FN1'],
        ['CDKN1A','MYH9'],
        ['CDKN1A','RP11-977G19.10'],
        ['CDKN1A','SFTPC'],
        ['CDKN1A','SSSCA1'],
        ['CDKN1B','RP11-180M15.7'],
        ['CDR1-AS','LINC00632'],
        ['CDR2','IDUA'],
        ['CDR2','RRN3P3'],
        ['CDR2L','ICT1'],
        ['CDRT4','FAM18B2'],
        ['CDRT4','TVP23C'],
        ['CDT1','HSPG2'],
        ['CDV3','SCAF1'],
        ['CEA','CEACAM1'],
        ['CEA','CEACAM6'],
        ['CEACAM1','CEACAM5'],
        ['CEACAM1','CEACAM6'],
        ['CEACAM5','CEACAM6'],
        ['CEACAM6','SFTPC'],
        ['CEBPA-AS1','CTD-2540B15.9'],
        ['CEBPB','SFTPB'],
        ['CEBPD','SFTPA1'],
        ['CEBPD','SFTPA2'],
        ['CEBPD','SFTPB'],
        ['CEBPD','SFTPC'],
        ['CEBPD','TG'],
        ['CEBPD','VIM'],
        ['CEBPZOS','RP11-423P10.2'],
        ['CECR1','SFTPB'],
        ['CECR7','IL17RA'],
        ['CEL','CELP'],
        ['CEL','GP2'],
        ['CELA3A','CELA3B'],
        ['CELF1','SFTPC'],
        ['CELF2','SFTPA2'],
        ['CELSR1','SFTPC'],
        ['CENPB','COL1A2'],
        ['CENPB','FN1'],
        ['CENPB','SFTPC'],
        ['CENPB','VEGFB'],
        ['CENPF','COL1A1'],
        ['CENPL','GAS5'],
        ['CENPN','CMC2'],
        ['CENPT','RPL14'],
        ['CEP152','RP11-227D13.4'],
        ['CEP164P1','RSU1P2'],
        ['CEP170B','COL1A1'],
        ['CEP170B','SFTPC'],
        ['CEP295','SCARNA9'],
        ['CERCAM','COL1A1'],
        ['CERCAM','COL1A2'],
        ['CERCAM','FN1'],
        ['CERK','CNN2'],
        ['CERK','EHD2'],
        ['CERS2','FN1'],
        ['CERS2','SFTPC'],
        ['CERS4','SFTPB'],
        ['CES1','CES1P1'],
        ['CES2','H6PD'],
        ['CES3','RP11-361L15.4'],
        ['CFAP45','RP11-190A12.7'],
        ['CFAP52','USP43'],
        ['CFAP53','MBD1'],
        ['CFAP99','RP11-503N18.1'],
        ['CFD','SFTPA1'],
        ['CFD','SFTPB'],
        ['CFD','SFTPC'],
        ['CFDP1','RP11-252K23.2'],
        ['CFH','CFHR1'],
        ['CFH','CFHR3'],
        ['CFHR1','CFHR2'],
        ['CFHR3','CFHR4'],
        ['CFI','LRG1'],
        ['CFI','SULT2A1'],
        ['CFL1','COL1A1'],
        ['CFL1','COL1A2'],
        ['CFL1','COL6A1'],
        ['CFL1','EFEMP2'],
        ['CFL1','FLNA'],
        ['CFL1','FN1'],
        ['CFL1','IGFBP4'],
        ['CFL1','LMNA'],
        ['CFL1','MYH9'],
        ['CFL1','PLEC'],
        ['CFL1','SFTPA2'],
        ['CFL1','SFTPC'],
        ['CFL1','SPARC'],
        ['CFL1','SUSD2'],
        ['CFL1','TAGLN'],
        ['CFL1','THBS1'],
        ['CFL1','VIM'],
        ['CFL1','VIM-AS1'],
        ['CGB7','KCNA7'],
        ['CGN','SFTPB'],
        ['CGNL1','SFTPB'],
        ['CGNL1','SFTPC'],
        ['CH17-118O6.2','CH17-472G23.2'],
        ['CH17-140K24.2','TAF7'],
        ['CH17-189H20.1','CH17-472G23.2'],
        ['CH17-472G23.4','PDE4DIP'],
        ['CHCHD10','VPREB3'],
        ['CHD2','COL1A1'],
        ['CHD2','RP11-437B10.1'],
        ['CHD3','COL1A1'],
        ['CHD3','COL6A2'],
        ['CHD3','FN1'],
        ['CHD3','ITGA5'],
        ['CHD3','PTMS'],
        ['CHD3','SFTPA2'],
        ['CHD3','SFTPB'],
        ['CHD3','SFTPC'],
        ['CHD3','TG'],
        ['CHD3','TPO'],
        ['CHD4','COL1A1'],
        ['CHD4','SFTPA2'],
        ['CHD4','TLN1'],
        ['CHD9','RP11-454F8.3'],
        ['CHFR','GOLGA3'],
        ['CHID1','COL1A1'],
        ['CHKB-CPT1B','SFTPC'],
        ['CHML','OPN3'],
        ['CHMP1A','COL1A1'],
        ['CHMP1A','FN1'],
        ['CHMP1A','PTMS'],
        ['CHMP7','RP11-1149O23.3'],
        ['CHP1','EMP2'],
        ['CHP1','LRG1'],
        ['CHP1','SFTPB'],
        ['CHP1','SULT2A1'],
        ['CHPF','COL1A1'],
        ['CHPF','COL1A2'],
        ['CHPF','COL6A2'],
        ['CHPF','FN1'],
        ['CHPF','IDH3G'],
        ['CHPF','MAPKAPK2'],
        ['CHPF2','COL1A1'],
        ['CHPF2','FN1'],
        ['CHR22-38_28785274-29006793.1','CTD-2233K9.1'],
        ['CHR22-38_28785274-29006793.1','KREMEN1'],
        ['CHRNA5','PSMA4'],
        ['CHST11','IGF2'],
        ['CHST12','VIM'],
        ['CHST3','COL1A1'],
        ['CHST5','CHST6'],
        ['CHST6','RP11-77K12.7'],
        ['CHST7','ZNF674-AS1'],
        ['CHSY1','IGFBP4'],
        ['CHSY3','HSPA8P4'],
        ['CHUK','ERLIN1'],
        ['CIC','COL1A1'],
        ['CIC','FN1'],
        ['CIC','GPC1'],
        ['CIC','SFTPC'],
        ['CIDEC','CIDECP'],
        ['CIITA','RP11-876N24.3'],
        ['CIITA','RP11-876N24.4'],
        ['CIITA','SFTPB'],
        ['CIRBP','EEF2'],
        ['CIRBP','FN1'],
        ['CIRBP','SFTPA1'],
        ['CIRBP','SFTPC'],
        ['CIRBP','TG'],
        ['CITED2','COL1A1'],
        ['CITED2','COL1A2'],
        ['CITED2','COL6A3'],
        ['CITED2','DNAJC5'],
        ['CITED2','EEF2'],
        ['CITED2','FARP1'],
        ['CITED2','FN1'],
        ['CITED2','IER5L'],
        ['CITED2','IGFBP2'],
        ['CITED2','IGFBP3'],
        ['CITED2','ITGA5'],
        ['CITED2','LRP5'],
        ['CITED2','MARCKS'],
        ['CITED2','PLEC'],
        ['CITED2','PTMS'],
        ['CITED2','RGMB'],
        ['CITED2','RNPEPL1'],
        ['CITED2','SFTPC'],
        ['CITED2','VIM'],
        ['CIZ1','COL1A1'],
        ['CKAP4','COL1A1'],
        ['CKAP4','COL1A2'],
        ['CKAP4','COL6A2'],
        ['CKAP4','FLNA'],
        ['CKAP4','FN1'],
        ['CKAP4','HS3ST3B1'],
        ['CKAP4','MT2A'],
        ['CKAP4','MYH9'],
        ['CKAP4','PEA15'],
        ['CKAP4','THBS1'],
        ['CKAP4','TWIST1'],
        ['CKAP4','UBE2J2'],
        ['CKAP4','VIM'],
        ['CKLF','CMTM1'],
        ['CKLF-CMTM1','CMTM1'],
        ['CKM','CLU'],
        ['CKM','CTSB'],
        ['CKM','DAB2'],
        ['CKM','DES'],
        ['CKM','GNAS'],
        ['CKM','NPPA'],
        ['CKM','NUCB1'],
        ['CKM','SYNPO2'],
        ['CKM','TG'],
        ['CKM','TPO'],
        ['CLCC1','RP11-475E11.9'],
        ['CLCF1','POLD4'],
        ['CLCN7','FN1'],
        ['CLCN7','TG'],
        ['CLCNKA','CLCNKB'],
        ['CLDN11','PTRF'],
        ['CLDN11','SERPINE1'],
        ['CLDN18','FAM134C'],
        ['CLDN18','ICAM1'],
        ['CLDN18','SFTPA2'],
        ['CLDN18','SFTPB'],
        ['CLDN18','SFTPC'],
        ['CLDN4','SFTPC'],
        ['CLDN5','SFTPA2'],
        ['CLDN5','SFTPC'],
        ['CLEC11A','COL1A1'],
        ['CLEC14A','SLC25A29'],
        ['CLEC2D','RP11-705C15.2'],
        ['CLEC2D','RP11-705C15.5'],
        ['CLEC3B','SFTPC'],
        ['CLEC4F','FIGLA'],
        ['CLIC1','COL1A1'],
        ['CLIC1','DDAH2'],
        ['CLIC1','FN1'],
        ['CLIC1','RP5-940J5.9'],
        ['CLIC4','COL1A1'],
        ['CLIC4','FN1'],
        ['CLIC5','RP1-8B1.4'],
        ['CLIC5','SFTPA1'],
        ['CLIC5','SFTPC'],
        ['CLIP1','RP11-512M8.3'],
        ['CLIP2','COL1A1'],
        ['CLK1','PPIL3'],
        ['CLK3','SFTPC'],
        ['CLK4','RN7SKP70'],
        ['CLMP','HSPA8'],
        ['CLN6','COL1A2'],
        ['CLPTM1','COL1A1'],
        ['CLPTM1L','COL1A1'],
        ['CLPTM1L','GAS6'],
        ['CLPTM1L','GNAS'],
        ['CLSTN1','COL1A1'],
        ['CLSTN1','COL1A2'],
        ['CLSTN1','CTNNBIP1'],
        ['CLSTN1','EIF3B'],
        ['CLSTN1','FLNA'],
        ['CLSTN1','FN1'],
        ['CLSTN1','MARCKS'],
        ['CLSTN1','MYL9'],
        ['CLSTN1','PRKCDBP'],
        ['CLTA','COL1A1'],
        ['CLTA','SFTPC'],
        ['CLTB','TG'],
        ['CLTC','COL1A1'],
        ['CLTC','COL1A2'],
        ['CLTC','FLNA'],
        ['CLTC','FN1'],
        ['CLTC','PPP1R12B'],
        ['CLTC','PTRF'],
        ['CLTC','SFTPC'],
        ['CLU','COL3A1'],
        ['CLU','CRYAB'],
        ['CLU','CTSB'],
        ['CLU','DES'],
        ['CLU','EEF2'],
        ['CLU','EPAS1'],
        ['CLU','FLNC'],
        ['CLU','GFAP'],
        ['CLU','GRINA'],
        ['CLU','H19'],
        ['CLU','HSPB6'],
        ['CLU','IGFBP5'],
        ['CLU','IYD'],
        ['CLU','MALAT1'],
        ['CLU','NRGN'],
        ['CLU','PHLDB1'],
        ['CLU','PLIN1'],
        ['CLU','PLP1'],
        ['CLU','POLR2A'],
        ['CLU','PTGDS'],
        ['CLU','SERPINA1'],
        ['CLU','SFTPA1'],
        ['CLU','SFTPA2'],
        ['CLU','SFTPB'],
        ['CLU','SFTPC'],
        ['CLU','SLC34A2'],
        ['CLU','SPARCL1'],
        ['CLU','TG'],
        ['CLU','TPO'],
        ['CLU','TUBB4A'],
        ['CLUH','COL1A1'],
        ['CLUH','TG'],
        ['CLUHP3','ZNF720'],
        ['CMC2','RP11-303E16.3'],
        ['CMSS1','COL8A1'],
        ['CMTM4','RP11-403P17.6'],
        ['CMYA5','TG'],
        ['CNBP','SFTPC'],
        ['CNDP2','SFTPB'],
        ['CNN1','FN1'],
        ['CNN1','MYH11'],
        ['CNN2','COL1A1'],
        ['CNN2','COL6A2'],
        ['CNN2','CTSB'],
        ['CNN2','DDX19A'],
        ['CNN2','DEGS1'],
        ['CNN2','ECE1'],
        ['CNN2','EMP2'],
        ['CNN2','ERC1'],
        ['CNN2','FAM212B'],
        ['CNN2','FBLIM1'],
        ['CNN2','FGFR1'],
        ['CNN2','FN1'],
        ['CNN2','GALNT6'],
        ['CNN2','GLTP'],
        ['CNN2','GRB2'],
        ['CNN2','HILPDA'],
        ['CNN2','HLA-E'],
        ['CNN2','ISG20L2'],
        ['CNN2','KIAA1191'],
        ['CNN2','LDLR'],
        ['CNN2','LEPREL4'],
        ['CNN2','MCAM'],
        ['CNN2','MED29'],
        ['CNN2','MYADM'],
        ['CNN2','MYO1C'],
        ['CNN2','ORC6'],
        ['CNN2','PDLIM5'],
        ['CNN2','PIK3R2'],
        ['CNN2','PLIN3'],
        ['CNN2','PPP1R12C'],
        ['CNN2','PTRF'],
        ['CNN2','PXDC1'],
        ['CNN2','RBM3'],
        ['CNN2','RFC2'],
        ['CNN2','RP11-186B7.4'],
        ['CNN2','RPL7L1'],
        ['CNN2','RUNDC1'],
        ['CNN2','SFTPB'],
        ['CNN2','SFTPC'],
        ['CNN2','SLC31A1'],
        ['CNN2','SLC35E4'],
        ['CNN2','SNORA67'],
        ['CNN2','SSH1'],
        ['CNN2','TCTA'],
        ['CNN2','TPM4'],
        ['CNN2','TRAM2'],
        ['CNN2','TSR1'],
        ['CNN2','UBE2G2'],
        ['CNN2','VIM'],
        ['CNN2','VIM-AS1'],
        ['CNN2','WDR82'],
        ['CNN2','YEATS2'],
        ['CNN2','ZBED1'],
        ['CNOT1','IGF2'],
        ['CNOT1','SNORA50'],
        ['CNOT11','HNRNPUL2'],
        ['CNOT11','HNRNPUL2-BSCL2'],
        ['CNOT3','FBXL19'],
        ['CNP','KCTD20'],
        ['CNP','RP11-229E13.2'],
        ['CNP','SFTPB'],
        ['CNPY2','PAN2'],
        ['CNPY3','RP3-475N16.1'],
        ['CNRIP1','HZGJ'],
        ['CNRIP1','PPP3R1'],
        ['CNTNAP3','CNTNAP3B'],
        ['COG7','RN7SKP23'],
        ['COG8','VPS4A'],
        ['COIL','RP11-1074O12.1'],
        ['COL11A1','COL1A1'],
        ['COL12A1','COL1A1'],
        ['COL12A1','COL1A2'],
        ['COL12A1','COL5A1'],
        ['COL12A1','COL6A1'],
        ['COL12A1','COL6A2'],
        ['COL12A1','COL6A3'],
        ['COL12A1','CTSB'],
        ['COL12A1','DGKZ'],
        ['COL12A1','ELN'],
        ['COL12A1','FLNA'],
        ['COL12A1','FN1'],
        ['COL12A1','GAPDH'],
        ['COL12A1','GREM1'],
        ['COL12A1','IGFBP3'],
        ['COL12A1','LOX'],
        ['COL12A1','MYH9'],
        ['COL12A1','PLEC'],
        ['COL12A1','PTRF'],
        ['COL12A1','SEPT11'],
        ['COL12A1','SERPINE1'],
        ['COL12A1','SPARC'],
        ['COL12A1','TAGLN'],
        ['COL12A1','TGFBI'],
        ['COL12A1','THBS1'],
        ['COL12A1','TMBIM6'],
        ['COL12A1','VIM'],
        ['COL16A1','COL1A1'],
        ['COL16A1','EMILIN1'],
        ['COL16A1','FAM83G'],
        ['COL16A1','FN1'],
        ['COL18A1','COL1A1'],
        ['COL18A1','EMC4'],
        ['COL18A1','FLNA'],
        ['COL18A1','FN1'],
        ['COL18A1','SFTPC'],
        ['COL18A1','SPARC'],
        ['COL1A1','COL1A2'],
        ['COL1A1','COL3A1'],
        ['COL1A1','COL4A1'],
        ['COL1A1','COL4A2'],
        ['COL1A1','COL5A1'],
        ['COL1A1','COL5A2'],
        ['COL1A1','COL5A3'],
        ['COL1A1','COL6A1'],
        ['COL1A1','COL6A2'],
        ['COL1A1','COL6A3'],
        ['COL1A1','COL8A1'],
        ['COL1A1','COLGALT1'],
        ['COL1A1','COMP'],
        ['COL1A1','COMT'],
        ['COL1A1','COPA'],
        ['COL1A1','COPE'],
        ['COL1A1','COPG1'],
        ['COL1A1','COPRS'],
        ['COL1A1','COPS6'],
        ['COL1A1','CORO1B'],
        ['COL1A1','CORO1C'],
        ['COL1A1','COTL1'],
        ['COL1A1','COX4I1'],
        ['COL1A1','CPA4'],
        ['COL1A1','CPNE1'],
        ['COL1A1','CPSF1'],
        ['COL1A1','CPSF3L'],
        ['COL1A1','CPSF6'],
        ['COL1A1','CRABP2'],
        ['COL1A1','CREB3L1'],
        ['COL1A1','CREB3L2'],
        ['COL1A1','CRIM1'],
        ['COL1A1','CRTAP'],
        ['COL1A1','CRYAB'],
        ['COL1A1','CSNK1D'],
        ['COL1A1','CSNK1E'],
        ['COL1A1','CSPG4'],
        ['COL1A1','CSRP1'],
        ['COL1A1','CST3'],
        ['COL1A1','CTD-2033D15.1'],
        ['COL1A1','CTD-3126B10.1'],
        ['COL1A1','CTDSP1'],
        ['COL1A1','CTGF'],
        ['COL1A1','CTIF'],
        ['COL1A1','CTNNA1'],
        ['COL1A1','CTNNB1'],
        ['COL1A1','CTNND1'],
        ['COL1A1','CTPS1'],
        ['COL1A1','CTSA'],
        ['COL1A1','CTSB'],
        ['COL1A1','CTSD'],
        ['COL1A1','CTTN'],
        ['COL1A1','CUTA'],
        ['COL1A1','CXCL12'],
        ['COL1A1','CYB5R3'],
        ['COL1A1','CYC1'],
        ['COL1A1','CYFIP1'],
        ['COL1A1','CYR61'],
        ['COL1A1','DAB2'],
        ['COL1A1','DAG1'],
        ['COL1A1','DAP'],
        ['COL1A1','DAPK3'],
        ['COL1A1','DAXX'],
        ['COL1A1','DBN1'],
        ['COL1A1','DBNL'],
        ['COL1A1','DCHS1'],
        ['COL1A1','DCN'],
        ['COL1A1','DCTN1'],
        ['COL1A1','DDAH1'],
        ['COL1A1','DDB1'],
        ['COL1A1','DDOST'],
        ['COL1A1','DDX3X'],
        ['COL1A1','DDX5'],
        ['COL1A1','DDX56'],
        ['COL1A1','DHCR24'],
        ['COL1A1','DHX30'],
        ['COL1A1','DIAPH1'],
        ['COL1A1','DKK1'],
        ['COL1A1','DKK3'],
        ['COL1A1','DLC1'],
        ['COL1A1','DLG5'],
        ['COL1A1','DLGAP4'],
        ['COL1A1','DMPK'],
        ['COL1A1','DNAJC5'],
        ['COL1A1','DNM2'],
        ['COL1A1','DNMT1'],
        ['COL1A1','DPYSL2'],
        ['COL1A1','DRAP1'],
        ['COL1A1','DSP'],
        ['COL1A1','DST'],
        ['COL1A1','DSTN'],
        ['COL1A1','DUSP1'],
        ['COL1A1','DVL3'],
        ['COL1A1','DYNC1H1'],
        ['COL1A1','ECE1'],
        ['COL1A1','ECM1'],
        ['COL1A1','EDF1'],
        ['COL1A1','EEF1D'],
        ['COL1A1','EEF1G'],
        ['COL1A1','EEF2'],
        ['COL1A1','EFEMP1'],
        ['COL1A1','EFEMP2'],
        ['COL1A1','EFHD2'],
        ['COL1A1','EGLN2'],
        ['COL1A1','EGR1'],
        ['COL1A1','EHBP1L1'],
        ['COL1A1','EHD1'],
        ['COL1A1','EHD2'],
        ['COL1A1','EIF1'],
        ['COL1A1','EIF2AK1'],
        ['COL1A1','EIF3B'],
        ['COL1A1','EIF4A3'],
        ['COL1A1','EIF4G1'],
        ['COL1A1','EIF4G2'],
        ['COL1A1','EIF4H'],
        ['COL1A1','EIF5A'],
        ['COL1A1','ELN'],
        ['COL1A1','EMC1'],
        ['COL1A1','EMC10'],
        ['COL1A1','EMILIN1'],
        ['COL1A1','EMP3'],
        ['COL1A1','ENG'],
        ['COL1A1','ENO1'],
        ['COL1A1','EP400'],
        ['COL1A1','EPAS1'],
        ['COL1A1','EPHB4'],
        ['COL1A1','EPN1'],
        ['COL1A1','ERBB2'],
        ['COL1A1','ERCC2'],
        ['COL1A1','ERF'],
        ['COL1A1','ERGIC1'],
        ['COL1A1','ESYT1'],
        ['COL1A1','EWSR1'],
        ['COL1A1','EXOC7'],
        ['COL1A1','EXT1'],
        ['COL1A1','EZR'],
        ['COL1A1','FADS1'],
        ['COL1A1','FADS2'],
        ['COL1A1','FAM120A'],
        ['COL1A1','FAM129B'],
        ['COL1A1','FAM166A'],
        ['COL1A1','FAM195B'],
        ['COL1A1','FAM50A'],
        ['COL1A1','FAM65A'],
        ['COL1A1','FASN'],
        ['COL1A1','FASTK'],
        ['COL1A1','FAT1'],
        ['COL1A1','FAU'],
        ['COL1A1','FBLIM1'],
        ['COL1A1','FBLN1'],
        ['COL1A1','FBLN2'],
        ['COL1A1','FBLN5'],
        ['COL1A1','FBN1'],
        ['COL1A1','FBN2'],
        ['COL1A1','FBXW5'],
        ['COL1A1','FGFR1'],
        ['COL1A1','FGFRL1'],
        ['COL1A1','FHL1'],
        ['COL1A1','FHL2'],
        ['COL1A1','FHL3'],
        ['COL1A1','FHOD1'],
        ['COL1A1','FKBP10'],
        ['COL1A1','FKBP11'],
        ['COL1A1','FKBP1A'],
        ['COL1A1','FKBP8'],
        ['COL1A1','FLII'],
        ['COL1A1','FLNA'],
        ['COL1A1','FLNB'],
        ['COL1A1','FLNC'],
        ['COL1A1','FN1'],
        ['COL1A1','FOSL1'],
        ['COL1A1','FOSL2'],
        ['COL1A1','FOXK1'],
        ['COL1A1','FOXM1'],
        ['COL1A1','FOXP4'],
        ['COL1A1','FRMD4A'],
        ['COL1A1','FRMD6'],
        ['COL1A1','FSCN1'],
        ['COL1A1','FST'],
        ['COL1A1','FSTL1'],
        ['COL1A1','FSTL3'],
        ['COL1A1','FTH1'],
        ['COL1A1','FTL'],
        ['COL1A1','FURIN'],
        ['COL1A1','FUS'],
        ['COL1A1','FXYD5'],
        ['COL1A1','FZD7'],
        ['COL1A1','G6PD'],
        ['COL1A1','GALNT2'],
        ['COL1A1','GANAB'],
        ['COL1A1','GAPDH'],
        ['COL1A1','GAS2L1'],
        ['COL1A1','GAS6'],
        ['COL1A1','GBF1'],
        ['COL1A1','GCN1L1'],
        ['COL1A1','GIPC1'],
        ['COL1A1','GLI2'],
        ['COL1A1','GNA11'],
        ['COL1A1','GNAI2'],
        ['COL1A1','GNAS'],
        ['COL1A1','GNB1'],
        ['COL1A1','GNB2'],
        ['COL1A1','GNB2L1'],
        ['COL1A1','GNG12'],
        ['COL1A1','GOLGA3'],
        ['COL1A1','GPAA1'],
        ['COL1A1','GPC1'],
        ['COL1A1','GPI'],
        ['COL1A1','GPR107'],
        ['COL1A1','GPR124'],
        ['COL1A1','GPR176'],
        ['COL1A1','GPS1'],
        ['COL1A1','GREM1'],
        ['COL1A1','GREM2'],
        ['COL1A1','GRN'],
        ['COL1A1','GSK3A'],
        ['COL1A1','GSN'],
        ['COL1A1','GSTP1'],
        ['COL1A1','GTPBP1'],
        ['COL1A1','GUK1'],
        ['COL1A1','H2AFX'],
        ['COL1A1','HCFC1'],
        ['COL1A1','HDAC5'],
        ['COL1A1','HDAC7'],
        ['COL1A1','HDLBP'],
        ['COL1A1','HEG1'],
        ['COL1A1','HERC4'],
        ['COL1A1','HGS'],
        ['COL1A1','HIF1A'],
        ['COL1A1','HK1'],
        ['COL1A1','HLA-A'],
        ['COL1A1','HLA-B'],
        ['COL1A1','HLA-C'],
        ['COL1A1','HLA-DRA'],
        ['COL1A1','HM13'],
        ['COL1A1','HMGA1'],
        ['COL1A1','HNRNPA0'],
        ['COL1A1','HNRNPA2B1'],
        ['COL1A1','HNRNPF'],
        ['COL1A1','HNRNPU'],
        ['COL1A1','HNRNPUL1'],
        ['COL1A1','HNRNPUL2'],
        ['COL1A1','HNRNPUL2-BSCL2'],
        ['COL1A1','HOMER3'],
        ['COL1A1','HS3ST3A1'],
        ['COL1A1','HSPA5'],
        ['COL1A1','HSPA8'],
        ['COL1A1','HSPB1'],
        ['COL1A1','HSPB6'],
        ['COL1A1','HSPB7'],
        ['COL1A1','HSPG2'],
        ['COL1A1','HTRA1'],
        ['COL1A1','HUWE1'],
        ['COL1A1','HYOU1'],
        ['COL1A1','ID3'],
        ['COL1A1','IDH3G'],
        ['COL1A1','IFITM3'],
        ['COL1A1','IGF2'],
        ['COL1A1','IGF2R'],
        ['COL1A1','IGFBP3'],
        ['COL1A1','IGFBP4'],
        ['COL1A1','IGFBP5'],
        ['COL1A1','IGFBP6'],
        ['COL1A1','IGFBP7'],
        ['COL1A1','IGKC'],
        ['COL1A1','IL17RC'],
        ['COL1A1','ILF3'],
        ['COL1A1','INF2'],
        ['COL1A1','INHBA'],
        ['COL1A1','INTS1'],
        ['COL1A1','IPO4'],
        ['COL1A1','IRAK1'],
        ['COL1A1','IRS1'],
        ['COL1A1','ITGA11'],
        ['COL1A1','ITGA2'],
        ['COL1A1','ITGA3'],
        ['COL1A1','ITGA5'],
        ['COL1A1','ITGB1'],
        ['COL1A1','ITGB5'],
        ['COL1A1','ITM2C'],
        ['COL1A1','JSRP1'],
        ['COL1A1','JUND'],
        ['COL1A1','KANK2'],
        ['COL1A1','KCNIP3'],
        ['COL1A1','KCTD10'],
        ['COL1A1','KDELR1'],
        ['COL1A1','KDELR2'],
        ['COL1A1','KDM2A'],
        ['COL1A1','KIAA1199'],
        ['COL1A1','KIAA1462'],
        ['COL1A1','KIF1C'],
        ['COL1A1','KIRREL'],
        ['COL1A1','KLF2'],
        ['COL1A1','KLHDC3'],
        ['COL1A1','KLHL22'],
        ['COL1A1','KMT2D'],
        ['COL1A1','KPNA2'],
        ['COL1A1','KRT10'],
        ['COL1A1','KRT18'],
        ['COL1A1','KRT19'],
        ['COL1A1','LAMA4'],
        ['COL1A1','LAMB1'],
        ['COL1A1','LAMB2'],
        ['COL1A1','LAMC1'],
        ['COL1A1','LAMP1'],
        ['COL1A1','LARP1'],
        ['COL1A1','LASP1'],
        ['COL1A1','LDHA'],
        ['COL1A1','LDLR'],
        ['COL1A1','LDLRAD2'],
        ['COL1A1','LEPRE1'],
        ['COL1A1','LEPREL2'],
        ['COL1A1','LEPREL4'],
        ['COL1A1','LGALS1'],
        ['COL1A1','LMAN2'],
        ['COL1A1','LMF2'],
        ['COL1A1','LMNA'],
        ['COL1A1','LMNB2'],
        ['COL1A1','LMO7'],
        ['COL1A1','LMOD1'],
        ['COL1A1','LOX'],
        ['COL1A1','LOXL1'],
        ['COL1A1','LOXL1-AS1'],
        ['COL1A1','LOXL2'],
        ['COL1A1','LOXL3'],
        ['COL1A1','LRFN4'],
        ['COL1A1','LRP1'],
        ['COL1A1','LRP10'],
        ['COL1A1','LRPAP1'],
        ['COL1A1','LRRC47'],
        ['COL1A1','LRRC59'],
        ['COL1A1','LRRC8A'],
        ['COL1A1','LTBP1'],
        ['COL1A1','LTBP2'],
        ['COL1A1','LTBP3'],
        ['COL1A1','LY6E'],
        ['COL1A1','MACF1'],
        ['COL1A1','MAGED1'],
        ['COL1A1','MAGED2'],
        ['COL1A1','MALAT1'],
        ['COL1A1','MAP1A'],
        ['COL1A1','MAP1B'],
        ['COL1A1','MAP1S'],
        ['COL1A1','MAP2K2'],
        ['COL1A1','MAP2K3'],
        ['COL1A1','MAP4'],
        ['COL1A1','MAP4K4'],
        ['COL1A1','MAP7D1'],
        ['COL1A1','MAPKAPK2'],
        ['COL1A1','MARCKS'],
        ['COL1A1','MARVELD1'],
        ['COL1A1','MAZ'],
        ['COL1A1','MBD3'],
        ['COL1A1','MBOAT7'],
        ['COL1A1','MBTPS1'],
        ['COL1A1','MCM5'],
        ['COL1A1','MDC1'],
        ['COL1A1','MDH2'],
        ['COL1A1','MED15'],
        ['COL1A1','MEG3'],
        ['COL1A1','MEGF8'],
        ['COL1A1','MEX3D'],
        ['COL1A1','MFAP5'],
        ['COL1A1','MFGE8'],
        ['COL1A1','MFN2'],
        ['COL1A1','MGAT1'],
        ['COL1A1','MGAT4B'],
        ['COL1A1','MGLL'],
        ['COL1A1','MGRN1'],
        ['COL1A1','MICAL2'],
        ['COL1A1','MICALL1'],
        ['COL1A1','MIF'],
        ['COL1A1','MINK1'],
        ['COL1A1','MIR3654'],
        ['COL1A1','MMP1'],
        ['COL1A1','MMP14'],
        ['COL1A1','MMP2'],
        ['COL1A1','MMP24-AS1'],
        ['COL1A1','MMP3'],
        ['COL1A1','MPG'],
        ['COL1A1','MPRIP'],
        ['COL1A1','MPV17'],
        ['COL1A1','MRAS'],
        ['COL1A1','MRC2'],
        ['COL1A1','MSN'],
        ['COL1A1','MTCH1'],
        ['COL1A1','MTSS1L'],
        ['COL1A1','MVP'],
        ['COL1A1','MXD4'],
        ['COL1A1','MXRA8'],
        ['COL1A1','MYADM'],
        ['COL1A1','MYBBP1A'],
        ['COL1A1','MYH11'],
        ['COL1A1','MYH9'],
        ['COL1A1','MYL6'],
        ['COL1A1','MYL9'],
        ['COL1A1','MYO10'],
        ['COL1A1','MYO1B'],
        ['COL1A1','MYO1C'],
        ['COL1A1','MYO9B'],
        ['COL1A1','MYOF'],
        ['COL1A1','NAA10'],
        ['COL1A1','NAB2'],
        ['COL1A1','NACC1'],
        ['COL1A1','NACC2'],
        ['COL1A1','NAPA'],
        ['COL1A1','NAV1'],
        ['COL1A1','NAV2'],
        ['COL1A1','NCDN'],
        ['COL1A1','NCL'],
        ['COL1A1','NCLN'],
        ['COL1A1','NCOR2'],
        ['COL1A1','NCS1'],
        ['COL1A1','NDST1'],
        ['COL1A1','NDUFS8'],
        ['COL1A1','NDUFV1'],
        ['COL1A1','NEK7'],
        ['COL1A1','NFATC4'],
        ['COL1A1','NFE2L1'],
        ['COL1A1','NFIC'],
        ['COL1A1','NFIX'],
        ['COL1A1','NID1'],
        ['COL1A1','NID2'],
        ['COL1A1','NISCH'],
        ['COL1A1','NLRP1'],
        ['COL1A1','NME1'],
        ['COL1A1','NME1-NME2'],
        ['COL1A1','NME2'],
        ['COL1A1','NME4'],
        ['COL1A1','NOTCH2'],
        ['COL1A1','NOTCH3'],
        ['COL1A1','NPAS1'],
        ['COL1A1','NRBP1'],
        ['COL1A1','NRP1'],
        ['COL1A1','NT5E'],
        ['COL1A1','NUCB1'],
        ['COL1A1','NUDCD3'],
        ['COL1A1','NUMA1'],
        ['COL1A1','NUP188'],
        ['COL1A1','NXN'],
        ['COL1A1','OAZ1'],
        ['COL1A1','ORAOV1'],
        ['COL1A1','OS9'],
        ['COL1A1','OTUB1'],
        ['COL1A1','OTUD7B'],
        ['COL1A1','OXTR'],
        ['COL1A1','P4HA2'],
        ['COL1A1','P4HB'],
        ['COL1A1','PABPC1'],
        ['COL1A1','PABPC4'],
        ['COL1A1','PACS1'],
        ['COL1A1','PAK4'],
        ['COL1A1','PALLD'],
        ['COL1A1','PAM'],
        ['COL1A1','PCBP2'],
        ['COL1A1','PCDHGA1'],
        ['COL1A1','PCDHGA10'],
        ['COL1A1','PCDHGA12'],
        ['COL1A1','PCDHGA2'],
        ['COL1A1','PCDHGA3'],
        ['COL1A1','PCDHGA4'],
        ['COL1A1','PCDHGA5'],
        ['COL1A1','PCDHGA6'],
        ['COL1A1','PCDHGA7'],
        ['COL1A1','PCDHGA8'],
        ['COL1A1','PCDHGA9'],
        ['COL1A1','PCDHGB1'],
        ['COL1A1','PCDHGB2'],
        ['COL1A1','PCDHGB3'],
        ['COL1A1','PCDHGB4'],
        ['COL1A1','PCDHGB6'],
        ['COL1A1','PCDHGB7'],
        ['COL1A1','PCDHGC3'],
        ['COL1A1','PCDHGC4'],
        ['COL1A1','PCDHGC5'],
        ['COL1A1','PCNXL3'],
        ['COL1A1','PCOLCE'],
        ['COL1A1','PCSK7'],
        ['COL1A1','PDGFRB'],
        ['COL1A1','PDIA4'],
        ['COL1A1','PDIA6'],
        ['COL1A1','PDLIM2'],
        ['COL1A1','PDLIM7'],
        ['COL1A1','PEA15'],
        ['COL1A1','PELP1'],
        ['COL1A1','PES1'],
        ['COL1A1','PFKL'],
        ['COL1A1','PFKP'],
        ['COL1A1','PFN1'],
        ['COL1A1','PHACTR2'],
        ['COL1A1','PHC2'],
        ['COL1A1','PHLDA3'],
        ['COL1A1','PHLDB1'],
        ['COL1A1','PHPT1'],
        ['COL1A1','PI4KA'],
        ['COL1A1','PICALM'],
        ['COL1A1','PIEZO1'],
        ['COL1A1','PIGT'],
        ['COL1A1','PIK3R2'],
        ['COL1A1','PINK1'],
        ['COL1A1','PKM'],
        ['COL1A1','PKN1'],
        ['COL1A1','PLCB3'],
        ['COL1A1','PLD3'],
        ['COL1A1','PLEC'],
        ['COL1A1','PLEKHM2'],
        ['COL1A1','PLEKHO2'],
        ['COL1A1','PLIN3'],
        ['COL1A1','PLK1'],
        ['COL1A1','PLOD1'],
        ['COL1A1','PLOD3'],
        ['COL1A1','PLP2'],
        ['COL1A1','PLXNA1'],
        ['COL1A1','PLXNA3'],
        ['COL1A1','PLXNB2'],
        ['COL1A1','PLXND1'],
        ['COL1A1','PMM2'],
        ['COL1A1','PODN'],
        ['COL1A1','PODXL'],
        ['COL1A1','POFUT2'],
        ['COL1A1','POLDIP2'],
        ['COL1A1','POLR2A'],
        ['COL1A1','POLR2E'],
        ['COL1A1','POLR2L'],
        ['COL1A1','POLR3D'],
        ['COL1A1','POSTN'],
        ['COL1A1','PPARD'],
        ['COL1A1','PPDPF'],
        ['COL1A1','PPIB'],
        ['COL1A1','PPP1R12C'],
        ['COL1A1','PPP1R18'],
        ['COL1A1','PPP1R9B'],
        ['COL1A1','PPP2R1A'],
        ['COL1A1','PPP2R4'],
        ['COL1A1','PPP4R1'],
        ['COL1A1','PPP6R1'],
        ['COL1A1','PPRC1'],
        ['COL1A1','PRDX1'],
        ['COL1A1','PRDX6'],
        ['COL1A1','PRKCA'],
        ['COL1A1','PRKCDBP'],
        ['COL1A1','PRKCSH'],
        ['COL1A1','PRKDC'],
        ['COL1A1','PRPF8'],
        ['COL1A1','PRRC2A'],
        ['COL1A1','PRRC2B'],
        ['COL1A1','PRSS23'],
        ['COL1A1','PSAP'],
        ['COL1A1','PSMB2'],
        ['COL1A1','PSMD2'],
        ['COL1A1','PTBP1'],
        ['COL1A1','PTK7'],
        ['COL1A1','PTMS'],
        ['COL1A1','PTPRF'],
        ['COL1A1','PTPRS'],
        ['COL1A1','PTRF'],
        ['COL1A1','PTTG1IP'],
        ['COL1A1','PTX3'],
        ['COL1A1','PVR'],
        ['COL1A1','PXDN'],
        ['COL1A1','PXN'],
        ['COL1A1','PYGB'],
        ['COL1A1','QARS'],
        ['COL1A1','QSOX1'],
        ['COL1A1','RAB11B'],
        ['COL1A1','RAB1B'],
        ['COL1A1','RAB34'],
        ['COL1A1','RAB3B'],
        ['COL1A1','RAB4B-EGLN2'],
        ['COL1A1','RAB5C'],
        ['COL1A1','RAB7A'],
        ['COL1A1','RABAC1'],
        ['COL1A1','RALY'],
        ['COL1A1','RANGAP1'],
        ['COL1A1','RBCK1'],
        ['COL1A1','RBFOX2'],
        ['COL1A1','RBM14-RBM4'],
        ['COL1A1','RBM3'],
        ['COL1A1','RBM4'],
        ['COL1A1','RBM42'],
        ['COL1A1','RCN3'],
        ['COL1A1','RELA'],
        ['COL1A1','RER1'],
        ['COL1A1','RERE'],
        ['COL1A1','RGMB'],
        ['COL1A1','RHOA'],
        ['COL1A1','RHOB'],
        ['COL1A1','RHOC'],
        ['COL1A1','RIC8A'],
        ['COL1A1','RNF10'],
        ['COL1A1','RNF187'],
        ['COL1A1','RNF213'],
        ['COL1A1','RNF26'],
        ['COL1A1','RNH1'],
        ['COL1A1','RP1-309I22.2'],
        ['COL1A1','RP11-124N14.3'],
        ['COL1A1','RP11-148K1.12'],
        ['COL1A1','RP11-161H23.5'],
        ['COL1A1','RP11-196G11.1'],
        ['COL1A1','RP11-20I23.1'],
        ['COL1A1','RP11-323N12.5'],
        ['COL1A1','RP11-334E6.12'],
        ['COL1A1','RP11-386G11.10'],
        ['COL1A1','RP11-426L16.10'],
        ['COL1A1','RP11-468E2.2'],
        ['COL1A1','RP11-498C9.2'],
        ['COL1A1','RP11-498C9.3'],
        ['COL1A1','RP11-572C15.6'],
        ['COL1A1','RP11-69I8.3'],
        ['COL1A1','RP11-849H4.4'],
        ['COL1A1','RP5-940J5.9'],
        ['COL1A1','RP5-977B1.11'],
        ['COL1A1','RPL18'],
        ['COL1A1','RPL28'],
        ['COL1A1','RPL3'],
        ['COL1A1','RPL36'],
        ['COL1A1','RPL37A'],
        ['COL1A1','RPL8'],
        ['COL1A1','RPLP1'],
        ['COL1A1','RPLP2'],
        ['COL1A1','RPN1'],
        ['COL1A1','RPN2'],
        ['COL1A1','RPS11'],
        ['COL1A1','RPS14'],
        ['COL1A1','RPS16'],
        ['COL1A1','RPS19'],
        ['COL1A1','RPS5'],
        ['COL1A1','RPS6KA2'],
        ['COL1A1','RRBP1'],
        ['COL1A1','RTN4'],
        ['COL1A1','RUSC2'],
        ['COL1A1','S100A10'],
        ['COL1A1','S100A11'],
        ['COL1A1','S100A16'],
        ['COL1A1','S100A4'],
        ['COL1A1','S100A6'],
        ['COL1A1','SAMD4B'],
        ['COL1A1','SART1'],
        ['COL1A1','SBF1'],
        ['COL1A1','SCAF1'],
        ['COL1A1','SCAMP2'],
        ['COL1A1','SCARF2'],
        ['COL1A1','SCD'],
        ['COL1A1','SCRIB'],
        ['COL1A1','SCRN1'],
        ['COL1A1','SCUBE3'],
        ['COL1A1','SDC3'],
        ['COL1A1','SDF4'],
        ['COL1A1','SEC13'],
        ['COL1A1','SEC16A'],
        ['COL1A1','SEC31A'],
        ['COL1A1','SEC61A1'],
        ['COL1A1','SEMA7A'],
        ['COL1A1','SENP3-EIF4A1'],
        ['COL1A1','SEPN1'],
        ['COL1A1','SEPT11'],
        ['COL1A1','SEPT2'],
        ['COL1A1','SEPT9'],
        ['COL1A1','SERF2'],
        ['COL1A1','SERINC1'],
        ['COL1A1','SERINC2'],
        ['COL1A1','SERPINE1'],
        ['COL1A1','SERPINE2'],
        ['COL1A1','SERPINH1'],
        ['COL1A1','SETD7'],
        ['COL1A1','SF1'],
        ['COL1A1','SF3B2'],
        ['COL1A1','SF3B3'],
        ['COL1A1','SFRP1'],
        ['COL1A1','SFTPA1'],
        ['COL1A1','SFTPA2'],
        ['COL1A1','SFTPB'],
        ['COL1A1','SFTPC'],
        ['COL1A1','SH3BGRL3'],
        ['COL1A1','SH3BP4'],
        ['COL1A1','SH3GL1'],
        ['COL1A1','SH3PXD2A'],
        ['COL1A1','SH3PXD2B'],
        ['COL1A1','SHC1'],
        ['COL1A1','SHISA5'],
        ['COL1A1','SHMT2'],
        ['COL1A1','SIGMAR1'],
        ['COL1A1','SIPA1L3'],
        ['COL1A1','SKI'],
        ['COL1A1','SLC12A4'],
        ['COL1A1','SLC16A3'],
        ['COL1A1','SLC1A5'],
        ['COL1A1','SLC22A17'],
        ['COL1A1','SLC25A1'],
        ['COL1A1','SLC25A3'],
        ['COL1A1','SLC25A6'],
        ['COL1A1','SLC2A1'],
        ['COL1A1','SLC34A2'],
        ['COL1A1','SLC35B2'],
        ['COL1A1','SLC35C1'],
        ['COL1A1','SLC38A10'],
        ['COL1A1','SLC38A2'],
        ['COL1A1','SLC39A1'],
        ['COL1A1','SLC39A13'],
        ['COL1A1','SLC3A2'],
        ['COL1A1','SLC44A2'],
        ['COL1A1','SLC4A2'],
        ['COL1A1','SLC52A2'],
        ['COL1A1','SLC7A1'],
        ['COL1A1','SLC7A5'],
        ['COL1A1','SLIT3'],
        ['COL1A1','SMARCC2'],
        ['COL1A1','SMC1A'],
        ['COL1A1','SMTN'],
        ['COL1A1','SND1'],
        ['COL1A1','SNORA67'],
        ['COL1A1','SNRNP200'],
        ['COL1A1','SNRNP70'],
        ['COL1A1','SNRPB'],
        ['COL1A1','SNX17'],
        ['COL1A1','SNX22'],
        ['COL1A1','SNX32'],
        ['COL1A1','SOGA1'],
        ['COL1A1','SON'],
        ['COL1A1','SORBS3'],
        ['COL1A1','SPARC'],
        ['COL1A1','SPATA21'],
        ['COL1A1','SPEG'],
        ['COL1A1','SPOCD1'],
        ['COL1A1','SPTAN1'],
        ['COL1A1','SPTBN1'],
        ['COL1A1','SQSTM1'],
        ['COL1A1','SRCAP'],
        ['COL1A1','SREBF2'],
        ['COL1A1','SRF'],
        ['COL1A1','SRM'],
        ['COL1A1','SRRM2'],
        ['COL1A1','SRRT'],
        ['COL1A1','SRSF2'],
        ['COL1A1','SRSF9'],
        ['COL1A1','SSH1'],
        ['COL1A1','SSR2'],
        ['COL1A1','STC1'],
        ['COL1A1','STC2'],
        ['COL1A1','STIM1'],
        ['COL1A1','STIP1'],
        ['COL1A1','STK25'],
        ['COL1A1','STRN4'],
        ['COL1A1','STT3A'],
        ['COL1A1','SULF1'],
        ['COL1A1','SUMF2'],
        ['COL1A1','SUPT5H'],
        ['COL1A1','SUPT6H'],
        ['COL1A1','SURF4'],
        ['COL1A1','SYMPK'],
        ['COL1A1','SYNJ2'],
        ['COL1A1','SZRD1'],
        ['COL1A1','TAF10'],
        ['COL1A1','TAGLN'],
        ['COL1A1','TAGLN2'],
        ['COL1A1','TBC1D10B'],
        ['COL1A1','TBC1D9B'],
        ['COL1A1','TBCD'],
        ['COL1A1','TCF25'],
        ['COL1A1','TCHP'],
        ['COL1A1','TCOF1'],
        ['COL1A1','TEAD3'],
        ['COL1A1','TENM2'],
        ['COL1A1','TFE3'],
        ['COL1A1','TFPI'],
        ['COL1A1','TFPI2'],
        ['COL1A1','TFRC'],
        ['COL1A1','TGFB1'],
        ['COL1A1','TGFB1I1'],
        ['COL1A1','TGFBI'],
        ['COL1A1','TGFBR2'],
        ['COL1A1','TGM2'],
        ['COL1A1','THBS1'],
        ['COL1A1','THBS2'],
        ['COL1A1','THOP1'],
        ['COL1A1','THY1'],
        ['COL1A1','TIMP1'],
        ['COL1A1','TIMP2'],
        ['COL1A1','TIMP3'],
        ['COL1A1','TINAGL1'],
        ['COL1A1','TKT'],
        ['COL1A1','TLN1'],
        ['COL1A1','TM4SF1'],
        ['COL1A1','TMBIM6'],
        ['COL1A1','TMED2'],
        ['COL1A1','TMED3'],
        ['COL1A1','TMED9'],
        ['COL1A1','TMEM115'],
        ['COL1A1','TMEM147'],
        ['COL1A1','TMEM184B'],
        ['COL1A1','TMEM189'],
        ['COL1A1','TMEM189-UBE2V1'],
        ['COL1A1','TMEM214'],
        ['COL1A1','TMEM256-PLSCR3'],
        ['COL1A1','TMEM259'],
        ['COL1A1','TMEM50A'],
        ['COL1A1','TMSB10'],
        ['COL1A1','TNC'],
        ['COL1A1','TNFRSF12A'],
        ['COL1A1','TNFRSF1A'],
        ['COL1A1','TNKS1BP1'],
        ['COL1A1','TNPO2'],
        ['COL1A1','TNRC18'],
        ['COL1A1','TNS1'],
        ['COL1A1','TOM1L2'],
        ['COL1A1','TPCN1'],
        ['COL1A1','TPD52L2'],
        ['COL1A1','TPM1'],
        ['COL1A1','TPM2'],
        ['COL1A1','TPM4'],
        ['COL1A1','TPT1'],
        ['COL1A1','TRAF7'],
        ['COL1A1','TRAK1'],
        ['COL1A1','TRAM2'],
        ['COL1A1','TRIM28'],
        ['COL1A1','TRIO'],
        ['COL1A1','TRIP12'],
        ['COL1A1','TRIP6'],
        ['COL1A1','TRNP1'],
        ['COL1A1','TRPC4AP'],
        ['COL1A1','TRPM4'],
        ['COL1A1','TSPAN4'],
        ['COL1A1','TSPO'],
        ['COL1A1','TTC7A'],
        ['COL1A1','TTYH3'],
        ['COL1A1','TUBA1B'],
        ['COL1A1','TUBA1C'],
        ['COL1A1','TUBB'],
        ['COL1A1','TUBB3'],
        ['COL1A1','TUBB4B'],
        ['COL1A1','TUBB6'],
        ['COL1A1','TXLNA'],
        ['COL1A1','TXNDC5'],
        ['COL1A1','TXNRD1'],
        ['COL1A1','U2AF2'],
        ['COL1A1','UBA1'],
        ['COL1A1','UBAP2'],
        ['COL1A1','UBAP2L'],
        ['COL1A1','UBC'],
        ['COL1A1','UBE2V1'],
        ['COL1A1','UBE3B'],
        ['COL1A1','UBE4B'],
        ['COL1A1','UBR4'],
        ['COL1A1','UBTF'],
        ['COL1A1','UBXN11'],
        ['COL1A1','UBXN6'],
        ['COL1A1','UCHL1'],
        ['COL1A1','UNC45A'],
        ['COL1A1','UQCR11'],
        ['COL1A1','USB1'],
        ['COL1A1','USP22'],
        ['COL1A1','USP36'],
        ['COL1A1','USP5'],
        ['COL1A1','VAMP5'],
        ['COL1A1','VARS'],
        ['COL1A1','VASN'],
        ['COL1A1','VASP'],
        ['COL1A1','VAT1'],
        ['COL1A1','VCAN'],
        ['COL1A1','VCL'],
        ['COL1A1','VCP'],
        ['COL1A1','VEGFB'],
        ['COL1A1','VIM'],
        ['COL1A1','VIM-AS1'],
        ['COL1A1','VKORC1'],
        ['COL1A1','VPS4A'],
        ['COL1A1','VPS51'],
        ['COL1A1','WDR1'],
        ['COL1A1','WDTC1'],
        ['COL1A1','WIZ'],
        ['COL1A1','WNT5B'],
        ['COL1A1','XXBAC-BPG252P9.9'],
        ['COL1A1','YIF1A'],
        ['COL1A1','YKT6'],
        ['COL1A1','YWHAE'],
        ['COL1A1','YWHAG'],
        ['COL1A1','YWHAH'],
        ['COL1A1','YWHAQ'],
        ['COL1A1','ZBTB45'],
        ['COL1A1','ZC3H7B'],
        ['COL1A1','ZNF469'],
        ['COL1A1','ZNF579'],
        ['COL1A1','ZNF703'],
        ['COL1A1','ZYX'],
        ['COL1A2','COL3A1'],
        ['COL1A2','COL4A1'],
        ['COL1A2','COL4A2'],
        ['COL1A2','COL5A1'],
        ['COL1A2','COL5A2'],
        ['COL1A2','COL6A1'],
        ['COL1A2','COL6A2'],
        ['COL1A2','COL6A3'],
        ['COL1A2','COL8A1'],
        ['COL1A2','COPG1'],
        ['COL1A2','CORO1C'],
        ['COL1A2','COTL1'],
        ['COL1A2','CRIM1'],
        ['COL1A2','CSRP1'],
        ['COL1A2','CTGF'],
        ['COL1A2','CTSA'],
        ['COL1A2','CTSB'],
        ['COL1A2','CTSD'],
        ['COL1A2','CXCL12'],
        ['COL1A2','CYB5R3'],
        ['COL1A2','CYR61'],
        ['COL1A2','DAPK3'],
        ['COL1A2','DBN1'],
        ['COL1A2','DCTN1'],
        ['COL1A2','DDX17'],
        ['COL1A2','DKK1'],
        ['COL1A2','DKK3'],
        ['COL1A2','DSP'],
        ['COL1A2','DST'],
        ['COL1A2','DVL1'],
        ['COL1A2','DYNC1H1'],
        ['COL1A2','EEF1G'],
        ['COL1A2','EEF2'],
        ['COL1A2','EFEMP2'],
        ['COL1A2','EHD2'],
        ['COL1A2','EIF4G1'],
        ['COL1A2','EIF4G2'],
        ['COL1A2','EIF5A'],
        ['COL1A2','ELN'],
        ['COL1A2','EMILIN1'],
        ['COL1A2','EMILIN2'],
        ['COL1A2','EMP1'],
        ['COL1A2','ENG'],
        ['COL1A2','ENO1'],
        ['COL1A2','EWSR1'],
        ['COL1A2','FAM129B'],
        ['COL1A2','FAT1'],
        ['COL1A2','FBLN1'],
        ['COL1A2','FBLN2'],
        ['COL1A2','FBN1'],
        ['COL1A2','FBN2'],
        ['COL1A2','FGFR1'],
        ['COL1A2','FKBP10'],
        ['COL1A2','FLNA'],
        ['COL1A2','FLNC'],
        ['COL1A2','FN1'],
        ['COL1A2','FRMD6'],
        ['COL1A2','FSCN1'],
        ['COL1A2','FSTL1'],
        ['COL1A2','FSTL3'],
        ['COL1A2','FTL'],
        ['COL1A2','FUS'],
        ['COL1A2','GANAB'],
        ['COL1A2','GAPDH'],
        ['COL1A2','GAS6'],
        ['COL1A2','GLG1'],
        ['COL1A2','GNAI2'],
        ['COL1A2','GNAS'],
        ['COL1A2','GNB1'],
        ['COL1A2','GNB2L1'],
        ['COL1A2','GNS'],
        ['COL1A2','GPC1'],
        ['COL1A2','GPR124'],
        ['COL1A2','GREM1'],
        ['COL1A2','GRN'],
        ['COL1A2','HDLBP'],
        ['COL1A2','HEG1'],
        ['COL1A2','HIF1A'],
        ['COL1A2','HLA-B'],
        ['COL1A2','HMGA1'],
        ['COL1A2','HNRNPA2B1'],
        ['COL1A2','HSPB7'],
        ['COL1A2','HSPG2'],
        ['COL1A2','HTRA1'],
        ['COL1A2','IGFBP3'],
        ['COL1A2','IGFBP4'],
        ['COL1A2','IGFBP5'],
        ['COL1A2','IQGAP1'],
        ['COL1A2','ITGA11'],
        ['COL1A2','ITGA3'],
        ['COL1A2','ITGA5'],
        ['COL1A2','ITGB1'],
        ['COL1A2','ITGB5'],
        ['COL1A2','KDELR2'],
        ['COL1A2','KIAA1199'],
        ['COL1A2','KIF1C'],
        ['COL1A2','KPNA6'],
        ['COL1A2','KRT1'],
        ['COL1A2','KRT19'],
        ['COL1A2','LAMB2'],
        ['COL1A2','LAMC1'],
        ['COL1A2','LAMP1'],
        ['COL1A2','LASP1'],
        ['COL1A2','LDHA'],
        ['COL1A2','LDLRAD2'],
        ['COL1A2','LEPREL4'],
        ['COL1A2','LGALS1'],
        ['COL1A2','LMNA'],
        ['COL1A2','LMO7'],
        ['COL1A2','LOX'],
        ['COL1A2','LOXL2'],
        ['COL1A2','LRP1'],
        ['COL1A2','LRP10'],
        ['COL1A2','LTBP2'],
        ['COL1A2','LUM'],
        ['COL1A2','LY6E'],
        ['COL1A2','MAGED1'],
        ['COL1A2','MAP1A'],
        ['COL1A2','MAP1B'],
        ['COL1A2','MAP7D1'],
        ['COL1A2','MBNL1'],
        ['COL1A2','MEG3'],
        ['COL1A2','MFGE8'],
        ['COL1A2','MIR3654'],
        ['COL1A2','MME'],
        ['COL1A2','MMP1'],
        ['COL1A2','MMP14'],
        ['COL1A2','MMP2'],
        ['COL1A2','MRC2'],
        ['COL1A2','MSN'],
        ['COL1A2','MVP'],
        ['COL1A2','MXRA8'],
        ['COL1A2','MYADM'],
        ['COL1A2','MYH9'],
        ['COL1A2','MYL6'],
        ['COL1A2','MYL9'],
        ['COL1A2','MYO9B'],
        ['COL1A2','MYOF'],
        ['COL1A2','NCKAP1'],
        ['COL1A2','NCOR2'],
        ['COL1A2','NDRG1'],
        ['COL1A2','NDST1'],
        ['COL1A2','NOTCH3'],
        ['COL1A2','NT5E'],
        ['COL1A2','NUMA1'],
        ['COL1A2','P4HB'],
        ['COL1A2','PCDH18'],
        ['COL1A2','PCOLCE'],
        ['COL1A2','PCSK7'],
        ['COL1A2','PDGFRA'],
        ['COL1A2','PDGFRB'],
        ['COL1A2','PDIA6'],
        ['COL1A2','PDLIM7'],
        ['COL1A2','PFN1'],
        ['COL1A2','PIEZO1'],
        ['COL1A2','PKM'],
        ['COL1A2','PKN1'],
        ['COL1A2','PLCB3'],
        ['COL1A2','PLD3'],
        ['COL1A2','PLEC'],
        ['COL1A2','PLEKHM2'],
        ['COL1A2','PLIN3'],
        ['COL1A2','PLOD1'],
        ['COL1A2','PLXNB2'],
        ['COL1A2','PODXL'],
        ['COL1A2','POLDIP3'],
        ['COL1A2','POLR2A'],
        ['COL1A2','PPIB'],
        ['COL1A2','PRPF8'],
        ['COL1A2','PRRC2A'],
        ['COL1A2','PSAP'],
        ['COL1A2','PSMD3'],
        ['COL1A2','PTMS'],
        ['COL1A2','PTPN14'],
        ['COL1A2','PTRF'],
        ['COL1A2','PTTG1IP'],
        ['COL1A2','PTX3'],
        ['COL1A2','PXDN'],
        ['COL1A2','QSOX1'],
        ['COL1A2','RAB7A'],
        ['COL1A2','RBM42'],
        ['COL1A2','RCN3'],
        ['COL1A2','RGMB'],
        ['COL1A2','RHOC'],
        ['COL1A2','RP11-124N14.3'],
        ['COL1A2','RP11-498C9.2'],
        ['COL1A2','RP11-498C9.3'],
        ['COL1A2','RP5-940J5.9'],
        ['COL1A2','RP5-977B1.11'],
        ['COL1A2','RPL28'],
        ['COL1A2','RPL37A'],
        ['COL1A2','RPL4'],
        ['COL1A2','RPL8'],
        ['COL1A2','RRBP1'],
        ['COL1A2','RTN4'],
        ['COL1A2','S100A11'],
        ['COL1A2','S100A6'],
        ['COL1A2','SART1'],
        ['COL1A2','SCUBE3'],
        ['COL1A2','SEC13'],
        ['COL1A2','SEC16A'],
        ['COL1A2','SEC24C'],
        ['COL1A2','SEC31A'],
        ['COL1A2','SEC61A1'],
        ['COL1A2','SEMA7A'],
        ['COL1A2','SEPT9'],
        ['COL1A2','SERPINE1'],
        ['COL1A2','SERPINE2'],
        ['COL1A2','SERPINH1'],
        ['COL1A2','SF3B2'],
        ['COL1A2','SFRP1'],
        ['COL1A2','SFRP2'],
        ['COL1A2','SFTPA1'],
        ['COL1A2','SFTPA2'],
        ['COL1A2','SFTPB'],
        ['COL1A2','SFTPC'],
        ['COL1A2','SH3BGRL3'],
        ['COL1A2','SH3BP4'],
        ['COL1A2','SH3PXD2A'],
        ['COL1A2','SHC1'],
        ['COL1A2','SLC1A5'],
        ['COL1A2','SLC4A2'],
        ['COL1A2','SLC7A5'],
        ['COL1A2','SNRNP70'],
        ['COL1A2','SPARC'],
        ['COL1A2','SPP1'],
        ['COL1A2','SPTAN1'],
        ['COL1A2','SPTBN1'],
        ['COL1A2','SRM'],
        ['COL1A2','SRRM2'],
        ['COL1A2','STC1'],
        ['COL1A2','STC2'],
        ['COL1A2','SURF4'],
        ['COL1A2','TAGLN'],
        ['COL1A2','TFPI2'],
        ['COL1A2','TGFBI'],
        ['COL1A2','TGM2'],
        ['COL1A2','THBS1'],
        ['COL1A2','THBS2'],
        ['COL1A2','THY1'],
        ['COL1A2','TIMP1'],
        ['COL1A2','TIMP2'],
        ['COL1A2','TIMP3'],
        ['COL1A2','TLN1'],
        ['COL1A2','TMEM47'],
        ['COL1A2','TMSB10'],
        ['COL1A2','TNC'],
        ['COL1A2','TNFRSF12A'],
        ['COL1A2','TNS1'],
        ['COL1A2','TPM1'],
        ['COL1A2','TPM2'],
        ['COL1A2','TPM4'],
        ['COL1A2','TPT1'],
        ['COL1A2','TRRAP'],
        ['COL1A2','TSPO'],
        ['COL1A2','TTYH3'],
        ['COL1A2','TUBB'],
        ['COL1A2','TUBB6'],
        ['COL1A2','TXNDC5'],
        ['COL1A2','TXNRD1'],
        ['COL1A2','U2AF2'],
        ['COL1A2','UBA52'],
        ['COL1A2','UBC'],
        ['COL1A2','VCL'],
        ['COL1A2','VIM'],
        ['COL1A2','VIM-AS1'],
        ['COL1A2','WDR1'],
        ['COL1A2','ZC3H7B'],
        ['COL1A2','ZYX'],
        ['COL23A1','TG'],
        ['COL3A1','COL4A1'],
        ['COL3A1','COL5A1'],
        ['COL3A1','COL6A1'],
        ['COL3A1','COL6A2'],
        ['COL3A1','COL6A3'],
        ['COL3A1','EEF2'],
        ['COL3A1','FLNA'],
        ['COL3A1','FLNC'],
        ['COL3A1','FLT1'],
        ['COL3A1','FN1'],
        ['COL3A1','GNB2L1'],
        ['COL3A1','GREM1'],
        ['COL3A1','HNRNPA2B1'],
        ['COL3A1','IGF2'],
        ['COL3A1','IGFBP3'],
        ['COL3A1','IGKC'],
        ['COL3A1','LMNA'],
        ['COL3A1','MYH9'],
        ['COL3A1','PLEC'],
        ['COL3A1','PTMS'],
        ['COL3A1','PTRF'],
        ['COL3A1','RP1-309I22.2'],
        ['COL3A1','SERPINE1'],
        ['COL3A1','SERPINE2'],
        ['COL3A1','SFTPA1'],
        ['COL3A1','SFTPA2'],
        ['COL3A1','SFTPB'],
        ['COL3A1','SFTPC'],
        ['COL3A1','SPARC'],
        ['COL3A1','SPP1'],
        ['COL3A1','SRRM2'],
        ['COL3A1','THBS1'],
        ['COL3A1','TIMP3'],
        ['COL3A1','TLN1'],
        ['COL3A1','TNNT3'],
        ['COL3A1','VIM'],
        ['COL4A1','COL4A2'],
        ['COL4A1','COL5A1'],
        ['COL4A1','COL6A1'],
        ['COL4A1','COL6A2'],
        ['COL4A1','COL6A3'],
        ['COL4A1','FLNA'],
        ['COL4A1','FLNC'],
        ['COL4A1','FN1'],
        ['COL4A1','MYH9'],
        ['COL4A1','SFTPA1'],
        ['COL4A1','SFTPA2'],
        ['COL4A1','SFTPB'],
        ['COL4A1','SFTPC'],
        ['COL4A1','SPARC'],
        ['COL4A1','THBS1'],
        ['COL4A2','COL5A1'],
        ['COL4A2','COL6A1'],
        ['COL4A2','COL6A2'],
        ['COL4A2','COL6A3'],
        ['COL4A2','CTGF'],
        ['COL4A2','FLNA'],
        ['COL4A2','FLNC'],
        ['COL4A2','FN1'],
        ['COL4A2','GPX3'],
        ['COL4A2','HDLBP'],
        ['COL4A2','HSPG2'],
        ['COL4A2','IGKC'],
        ['COL4A2','LMNA'],
        ['COL4A2','MYH9'],
        ['COL4A2','PIEZO1'],
        ['COL4A2','PIN1'],
        ['COL4A2','PLEC'],
        ['COL4A2','PTRF'],
        ['COL4A2','SFTPA1'],
        ['COL4A2','SFTPA2'],
        ['COL4A2','SFTPB'],
        ['COL4A2','SFTPC'],
        ['COL4A2','SPARC'],
        ['COL4A2','TG'],
        ['COL4A2','THBS1'],
        ['COL4A2','TPM2'],
        ['COL4A2','VIM'],
        ['COL4A2','VIM-AS1'],
        ['COL4A3','SFTPA1'],
        ['COL5A1','COL6A1'],
        ['COL5A1','COL6A2'],
        ['COL5A1','COL6A3'],
        ['COL5A1','EEF2'],
        ['COL5A1','EGR1'],
        ['COL5A1','EIF4G1'],
        ['COL5A1','EMILIN1'],
        ['COL5A1','FAM129B'],
        ['COL5A1','FASN'],
        ['COL5A1','FLNA'],
        ['COL5A1','FLNC'],
        ['COL5A1','FN1'],
        ['COL5A1','FOXD1'],
        ['COL5A1','GNAS'],
        ['COL5A1','H1FX'],
        ['COL5A1','HDLBP'],
        ['COL5A1','HMGA1'],
        ['COL5A1','IGFBP3'],
        ['COL5A1','IGFBP4'],
        ['COL5A1','IMPDH1'],
        ['COL5A1','ITGA5'],
        ['COL5A1','ITGB5'],
        ['COL5A1','LASP1'],
        ['COL5A1','LMNA'],
        ['COL5A1','LRP1'],
        ['COL5A1','MXRA8'],
        ['COL5A1','MYH9'],
        ['COL5A1','MYL9'],
        ['COL5A1','NCOR2'],
        ['COL5A1','PCSK7'],
        ['COL5A1','PKM'],
        ['COL5A1','PLEC'],
        ['COL5A1','PPP6R1'],
        ['COL5A1','PTMS'],
        ['COL5A1','PTPRS'],
        ['COL5A1','PTRF'],
        ['COL5A1','QSOX1'],
        ['COL5A1','RGP1'],
        ['COL5A1','RP5-977B1.11'],
        ['COL5A1','RPS2'],
        ['COL5A1','SKI'],
        ['COL5A1','SLC16A3'],
        ['COL5A1','SPARC'],
        ['COL5A1','TAGLN'],
        ['COL5A1','TGFBI'],
        ['COL5A1','THBS1'],
        ['COL5A1','TIMP1'],
        ['COL5A1','TLN1'],
        ['COL5A1','TNC'],
        ['COL5A1','TNFRSF12A'],
        ['COL5A1','TNS1'],
        ['COL5A1','TPM2'],
        ['COL5A1','TXNDC5'],
        ['COL5A1','VIM'],
        ['COL5A1','VIM-AS1'],
        ['COL5A1','ZMIZ2'],
        ['COL5A1','ZYX'],
        ['COL5A2','COL6A1'],
        ['COL5A2','COL6A2'],
        ['COL5A2','FLNA'],
        ['COL5A2','FN1'],
        ['COL5A2','THBS1'],
        ['COL5A3','COL6A1'],
        ['COL5A3','COL6A2'],
        ['COL5A3','COL6A3'],
        ['COL5A3','FLNA'],
        ['COL5A3','FLNC'],
        ['COL5A3','FN1'],
        ['COL5A3','FOXD1'],
        ['COL5A3','SFTPC'],
        ['COL5A3','SPARC'],
        ['COL5A3','TCIRG1'],
        ['COL5A3','THBS1'],
        ['COL6A1','COL6A2'],
        ['COL6A1','COL6A3'],
        ['COL6A1','CSRP1'],
        ['COL6A1','CTC-425F1.4'],
        ['COL6A1','CTSB'],
        ['COL6A1','DRAP1'],
        ['COL6A1','EEF2'],
        ['COL6A1','EHD2'],
        ['COL6A1','EIF3B'],
        ['COL6A1','EIF4G1'],
        ['COL6A1','EMILIN1'],
        ['COL6A1','EMP3'],
        ['COL6A1','ENO1'],
        ['COL6A1','FAM129B'],
        ['COL6A1','FBN1'],
        ['COL6A1','FKBP10'],
        ['COL6A1','FLNA'],
        ['COL6A1','FLNC'],
        ['COL6A1','FN1'],
        ['COL6A1','FSTL1'],
        ['COL6A1','FUS'],
        ['COL6A1','GAPDH'],
        ['COL6A1','GAS6'],
        ['COL6A1','GIT1'],
        ['COL6A1','GNAI2'],
        ['COL6A1','GNAS'],
        ['COL6A1','GNB1'],
        ['COL6A1','GREM1'],
        ['COL6A1','HMGA1'],
        ['COL6A1','HNRNPUL2'],
        ['COL6A1','HSPB1'],
        ['COL6A1','HSPB6'],
        ['COL6A1','HSPG2'],
        ['COL6A1','IGFBP3'],
        ['COL6A1','IGFBP4'],
        ['COL6A1','IGFBP6'],
        ['COL6A1','ITGA11'],
        ['COL6A1','ITGA5'],
        ['COL6A1','KIAA1199'],
        ['COL6A1','KIF1C'],
        ['COL6A1','LAMA4'],
        ['COL6A1','LASP1'],
        ['COL6A1','LGALS1'],
        ['COL6A1','LMNA'],
        ['COL6A1','LOX'],
        ['COL6A1','LOXL1'],
        ['COL6A1','LOXL2'],
        ['COL6A1','LRP1'],
        ['COL6A1','LRRC59'],
        ['COL6A1','MAZ'],
        ['COL6A1','MFGE8'],
        ['COL6A1','MMP14'],
        ['COL6A1','MPRIP'],
        ['COL6A1','MRC2'],
        ['COL6A1','MXRA8'],
        ['COL6A1','MYH9'],
        ['COL6A1','MYL9'],
        ['COL6A1','MYO1C'],
        ['COL6A1','NCOR2'],
        ['COL6A1','NID2'],
        ['COL6A1','P4HB'],
        ['COL6A1','PCOLCE'],
        ['COL6A1','PCSK7'],
        ['COL6A1','PFN1'],
        ['COL6A1','PKM'],
        ['COL6A1','PLEC'],
        ['COL6A1','PLOD1'],
        ['COL6A1','PLXNB2'],
        ['COL6A1','POSTN'],
        ['COL6A1','PRKACA'],
        ['COL6A1','PTBP1'],
        ['COL6A1','PTMS'],
        ['COL6A1','PTRF'],
        ['COL6A1','QSOX1'],
        ['COL6A1','RGP1'],
        ['COL6A1','RP5-940J5.9'],
        ['COL6A1','RPLP2'],
        ['COL6A1','RRBP1'],
        ['COL6A1','SEC16A'],
        ['COL6A1','SEC61A1'],
        ['COL6A1','SEMA7A'],
        ['COL6A1','SEPT9'],
        ['COL6A1','SERPINE1'],
        ['COL6A1','SERPINE2'],
        ['COL6A1','SERPINH1'],
        ['COL6A1','SFTPA2'],
        ['COL6A1','SFTPB'],
        ['COL6A1','SFTPC'],
        ['COL6A1','SPARC'],
        ['COL6A1','TAF15'],
        ['COL6A1','TAGLN'],
        ['COL6A1','TGFBI'],
        ['COL6A1','THBS1'],
        ['COL6A1','TIMP1'],
        ['COL6A1','TIMP3'],
        ['COL6A1','TMSB10'],
        ['COL6A1','TNC'],
        ['COL6A1','TPM2'],
        ['COL6A1','TSPO'],
        ['COL6A1','TTYH3'],
        ['COL6A1','VAT1'],
        ['COL6A1','VIM'],
        ['COL6A1','VIM-AS1'],
        ['COL6A1','WDR1'],
        ['COL6A1','XYLT1'],
        ['COL6A1','ZYX'],
        ['COL6A2','COL6A3'],
        ['COL6A2','CREB3L1'],
        ['COL6A2','CRIM1'],
        ['COL6A2','CSRP1'],
        ['COL6A2','CTGF'],
        ['COL6A2','CTSB'],
        ['COL6A2','DAPK3'],
        ['COL6A2','DNLZ'],
        ['COL6A2','EEF2'],
        ['COL6A2','EFEMP2'],
        ['COL6A2','EHD1'],
        ['COL6A2','EHD2'],
        ['COL6A2','EIF4G1'],
        ['COL6A2','EIF4G2'],
        ['COL6A2','EMILIN1'],
        ['COL6A2','ENO1'],
        ['COL6A2','EPN1'],
        ['COL6A2','ERGIC1'],
        ['COL6A2','FADS2'],
        ['COL6A2','FAM129B'],
        ['COL6A2','FASN'],
        ['COL6A2','FBLN2'],
        ['COL6A2','FBN2'],
        ['COL6A2','FKBP10'],
        ['COL6A2','FLNA'],
        ['COL6A2','FLNC'],
        ['COL6A2','FN1'],
        ['COL6A2','FSTL1'],
        ['COL6A2','FURIN'],
        ['COL6A2','GANAB'],
        ['COL6A2','GAPDH'],
        ['COL6A2','GAS6'],
        ['COL6A2','GNAI2'],
        ['COL6A2','GNAS'],
        ['COL6A2','GNB2L1'],
        ['COL6A2','GPC1'],
        ['COL6A2','GREM1'],
        ['COL6A2','HDLBP'],
        ['COL6A2','HMGA1'],
        ['COL6A2','HNRNPD'],
        ['COL6A2','HNRNPK'],
        ['COL6A2','HSPB6'],
        ['COL6A2','HSPG2'],
        ['COL6A2','HYOU1'],
        ['COL6A2','IGFBP3'],
        ['COL6A2','IGFBP4'],
        ['COL6A2','IGFBP6'],
        ['COL6A2','ITGA11'],
        ['COL6A2','ITGA5'],
        ['COL6A2','ITM2C'],
        ['COL6A2','KIAA1199'],
        ['COL6A2','LAMB2'],
        ['COL6A2','LAMC1'],
        ['COL6A2','LAMP1'],
        ['COL6A2','LARP1'],
        ['COL6A2','LASP1'],
        ['COL6A2','LGALS1'],
        ['COL6A2','LMNA'],
        ['COL6A2','LOX'],
        ['COL6A2','LOXL1'],
        ['COL6A2','LOXL2'],
        ['COL6A2','LRP1'],
        ['COL6A2','MAP7D1'],
        ['COL6A2','MFGE8'],
        ['COL6A2','MFSD10'],
        ['COL6A2','MMP1'],
        ['COL6A2','MMP14'],
        ['COL6A2','MMP2'],
        ['COL6A2','MRC2'],
        ['COL6A2','MXRA8'],
        ['COL6A2','MYH9'],
        ['COL6A2','MYL6'],
        ['COL6A2','MYL9'],
        ['COL6A2','MYO1C'],
        ['COL6A2','NCOR2'],
        ['COL6A2','NFIC'],
        ['COL6A2','NXN'],
        ['COL6A2','OAZ1'],
        ['COL6A2','P4HB'],
        ['COL6A2','PCDHGC3'],
        ['COL6A2','PCOLCE'],
        ['COL6A2','PCSK7'],
        ['COL6A2','PDGFRB'],
        ['COL6A2','PDIA4'],
        ['COL6A2','PDIA6'],
        ['COL6A2','PDLIM7'],
        ['COL6A2','PFN1'],
        ['COL6A2','PHLDB1'],
        ['COL6A2','PIEZO1'],
        ['COL6A2','PIGT'],
        ['COL6A2','PKM'],
        ['COL6A2','PLEC'],
        ['COL6A2','PLOD1'],
        ['COL6A2','PLXNA1'],
        ['COL6A2','PPP1R18'],
        ['COL6A2','PRKACA'],
        ['COL6A2','PRPF8'],
        ['COL6A2','PRRC2A'],
        ['COL6A2','PRRC2B'],
        ['COL6A2','PSAP'],
        ['COL6A2','PTMS'],
        ['COL6A2','PTOV1'],
        ['COL6A2','PTRF'],
        ['COL6A2','PXDN'],
        ['COL6A2','PYCR1'],
        ['COL6A2','QSOX1'],
        ['COL6A2','RAB11B'],
        ['COL6A2','RALY'],
        ['COL6A2','RANGAP1'],
        ['COL6A2','RCN3'],
        ['COL6A2','RP5-940J5.9'],
        ['COL6A2','RP5-977B1.11'],
        ['COL6A2','RPL8'],
        ['COL6A2','RPLP2'],
        ['COL6A2','RRBP1'],
        ['COL6A2','SEPT9'],
        ['COL6A2','SERPINE1'],
        ['COL6A2','SERPINE2'],
        ['COL6A2','SERPINH1'],
        ['COL6A2','SFTPA1'],
        ['COL6A2','SFTPA2'],
        ['COL6A2','SFTPB'],
        ['COL6A2','SFTPC'],
        ['COL6A2','SH3BGRL3'],
        ['COL6A2','SH3BP2'],
        ['COL6A2','SHC1'],
        ['COL6A2','SLC38A2'],
        ['COL6A2','SLC7A5'],
        ['COL6A2','SPARC'],
        ['COL6A2','SRRM2'],
        ['COL6A2','SSH1'],
        ['COL6A2','STC1'],
        ['COL6A2','SULF1'],
        ['COL6A2','TAF10'],
        ['COL6A2','TAGLN'],
        ['COL6A2','TFPI2'],
        ['COL6A2','TGFBI'],
        ['COL6A2','TGM2'],
        ['COL6A2','THBS1'],
        ['COL6A2','THY1'],
        ['COL6A2','TIMP1'],
        ['COL6A2','TIMP2'],
        ['COL6A2','TIMP3'],
        ['COL6A2','TLN1'],
        ['COL6A2','TNC'],
        ['COL6A2','TNRC18'],
        ['COL6A2','TPM1'],
        ['COL6A2','TPM2'],
        ['COL6A2','TPM4'],
        ['COL6A2','TRIO'],
        ['COL6A2','TSC22D4'],
        ['COL6A2','TTYH3'],
        ['COL6A2','TUBB'],
        ['COL6A2','TUBB3'],
        ['COL6A2','TUBB4B'],
        ['COL6A2','TXNDC5'],
        ['COL6A2','USP22'],
        ['COL6A2','VIM'],
        ['COL6A2','VIM-AS1'],
        ['COL6A2','WDR1'],
        ['COL6A2','WNT5B'],
        ['COL6A2','ZYX'],
        ['COL6A3','CSRP1'],
        ['COL6A3','CTGF'],
        ['COL6A3','CTSB'],
        ['COL6A3','CYB5R3'],
        ['COL6A3','EEF2'],
        ['COL6A3','EIF4A1'],
        ['COL6A3','ELN'],
        ['COL6A3','EXT1'],
        ['COL6A3','FAM129B'],
        ['COL6A3','FBN1'],
        ['COL6A3','FLNA'],
        ['COL6A3','FLNC'],
        ['COL6A3','FN1'],
        ['COL6A3','FSTL1'],
        ['COL6A3','FTL'],
        ['COL6A3','FUS'],
        ['COL6A3','GAPDH'],
        ['COL6A3','GNAI2'],
        ['COL6A3','GNAS'],
        ['COL6A3','GREM1'],
        ['COL6A3','HSPG2'],
        ['COL6A3','IGFBP3'],
        ['COL6A3','IGFBP4'],
        ['COL6A3','IGFBP5'],
        ['COL6A3','IGKC'],
        ['COL6A3','IL11'],
        ['COL6A3','ITGA11'],
        ['COL6A3','ITGA5'],
        ['COL6A3','KIAA1199'],
        ['COL6A3','LAMB2'],
        ['COL6A3','LASP1'],
        ['COL6A3','LGALS1'],
        ['COL6A3','LMNA'],
        ['COL6A3','LOXL2'],
        ['COL6A3','LRP1'],
        ['COL6A3','MAGED1'],
        ['COL6A3','MAP1A'],
        ['COL6A3','MMP1'],
        ['COL6A3','MMP14'],
        ['COL6A3','MSN'],
        ['COL6A3','MYH9'],
        ['COL6A3','MYL6'],
        ['COL6A3','MYO1C'],
        ['COL6A3','OAZ1'],
        ['COL6A3','P4HB'],
        ['COL6A3','PIEZO1'],
        ['COL6A3','PKM'],
        ['COL6A3','PLEC'],
        ['COL6A3','PRKCSH'],
        ['COL6A3','PSAP'],
        ['COL6A3','PTK7'],
        ['COL6A3','PTRF'],
        ['COL6A3','PXDN'],
        ['COL6A3','RP11-124N14.3'],
        ['COL6A3','RP5-940J5.9'],
        ['COL6A3','RP5-977B1.11'],
        ['COL6A3','RPL8'],
        ['COL6A3','RTN4'],
        ['COL6A3','SENP3-EIF4A1'],
        ['COL6A3','SERPINE1'],
        ['COL6A3','SERPINE2'],
        ['COL6A3','SFTPA1'],
        ['COL6A3','SFTPA2'],
        ['COL6A3','SFTPB'],
        ['COL6A3','SFTPC'],
        ['COL6A3','SNORA67'],
        ['COL6A3','SPARC'],
        ['COL6A3','STC1'],
        ['COL6A3','TAGLN'],
        ['COL6A3','TFPI2'],
        ['COL6A3','TGFBI'],
        ['COL6A3','THBS1'],
        ['COL6A3','THY1'],
        ['COL6A3','TIMP1'],
        ['COL6A3','TIMP3'],
        ['COL6A3','TNC'],
        ['COL6A3','TPM2'],
        ['COL6A3','VIM'],
        ['COL6A3','VIM-AS1'],
        ['COL6A3','WDR1'],
        ['COL6A3','ZYX'],
        ['COL7A1','FN1'],
        ['COL7A1','KRT10'],
        ['COL7A1','UCN2'],
        ['COL8A1','FN1'],
        ['COL8A1','SPARC'],
        ['COL8A1','THBS1'],
        ['COLGALT1','COPA'],
        ['COLGALT1','FN1'],
        ['COLGALT1','NUP62'],
        ['COLGALT1','PLIN3'],
        ['COLGALT1','PTRF'],
        ['COLQ','HACL1'],
        ['COMMD1','RPSA'],
        ['COMP','FN1'],
        ['COPA','CRTAP'],
        ['COPA','EHD2'],
        ['COPA','FN1'],
        ['COPB1','MUC7'],
        ['COPE','FN1'],
        ['COPG1','FN1'],
        ['COPG1','MYH9'],
        ['COPRS','ITPRIP'],
        ['COPS6','FN1'],
        ['COPS6','FTH1'],
        ['COPS6','ZDHHC7'],
        ['COPS7A','FN1'],
        ['COPS8','COPS8P2'],
        ['COPZ2','RP5-890E16.2'],
        ['CORO1A','SFTPC'],
        ['CORO1B','PTPRCAP'],
        ['CORO1C','FN1'],
        ['CORO1C','SELPLG'],
        ['COTL1','FLNA'],
        ['COTL1','FN1'],
        ['COTL1','HNRNPA0'],
        ['COTL1','MGAT4B'],
        ['COTL1','PTRF'],
        ['COTL1','SCUBE3'],
        ['COTL1','SFTPC'],
        ['COX14','RP4-605O3.4'],
        ['COX16','SYNJ2BP'],
        ['COX17','POPDC2'],
        ['COX7A2L','KCNG3'],
        ['COX7C','COX7CP1'],
        ['COX7C','RP11-480I12.2'],
        ['COX8A','FN1'],
        ['COX8A','SFTPC'],
        ['CP','CTD-2114J12.1'],
        ['CPAMD8','SFTPA1'],
        ['CPM','SFTPB'],
        ['CPM','SFTPC'],
        ['CPM','TMED9'],
        ['CPNE1','RBM12'],
        ['CPQ','SFTPC'],
        ['CPSF1','FN1'],
        ['CPSF3L','RP5-890O3.9'],
        ['CRABP2','FN1'],
        ['CRABP2','RP11-66D17.5'],
        ['CREB3L1','FN1'],
        ['CREB3L1','HSPG2'],
        ['CREB3L1','PRSS1'],
        ['CREB3L1','PSPC1'],
        ['CREB3L2','TPO'],
        ['CREBBP','FN1'],
        ['CREBBP','PTMS'],
        ['CREBL2','TPO'],
        ['CREG1','SFTPA2'],
        ['CRELD1','FN1'],
        ['CRIM1','FN1'],
        ['CRIM1','GNAS'],
        ['CRIM1','MARCKS'],
        ['CRIM1','MN1'],
        ['CRIM1','SFTPB'],
        ['CRIM1','SFTPC'],
        ['CRIM1','VASN'],
        ['CRIP1','CRIP2'],
        ['CRIP2','SFTPA2'],
        ['CRIP2','SFTPB'],
        ['CRIP2','SFTPC'],
        ['CRIP3','ZNF318'],
        ['CRIPAK','UVSSA'],
        ['CRISPLD2','GPNMB'],
        ['CRLF2','CSF2RA'],
        ['CRLF2','CTSS'],
        ['CRLS1','MCM8'],
        ['CRNN','S100A9'],
        ['CROCC','CROCCP2'],
        ['CROCCP2','RNF123'],
        ['CROCCP2','RP4-565E6.1'],
        ['CRP','CYP2E1'],
        ['CRP','FGA'],
        ['CRP','FGB'],
        ['CRP','FGG'],
        ['CRP','HP'],
        ['CRP','ORM1'],
        ['CRP','RDH11'],
        ['CRP','SERPINA1'],
        ['CRP','SERPINA3'],
        ['CRTAP','FN1'],
        ['CRTAP','MCAM'],
        ['CRTAP','MYO1C'],
        ['CRTAP','PLIN3'],
        ['CRTAP','PTRF'],
        ['CRTAP','RSL1D1'],
        ['CRTAP','SCARF2'],
        ['CRTAP','SFTPB'],
        ['CRTAP','SFTPC'],
        ['CRTAP','SMAD3'],
        ['CRTAP','TPM4'],
        ['CRTC2','SLC39A1'],
        ['CRY1','MTERF2'],
        ['CRY1','MTERFD3'],
        ['CRY2','SFTPC'],
        ['CRYAB','DES'],
        ['CRYAB','DGAT2'],
        ['CRYAB','EEF2'],
        ['CRYAB','FLNC'],
        ['CRYAB','FN1'],
        ['CRYAB','GNAS'],
        ['CRYAB','GPX3'],
        ['CRYAB','HSPB1'],
        ['CRYAB','IGFBP5'],
        ['CRYAB','MBP'],
        ['CRYAB','MTSS1L'],
        ['CRYAB','NEAT1'],
        ['CRYAB','PLIN4'],
        ['CRYAB','TG'],
        ['CRYAB','TPO'],
        ['CRYBB2','CRYBB2P1'],
        ['CS','CTB-52I2.4'],
        ['CSDA','SFTPC'],
        ['CSDAP1','YBX3'],
        ['CSDE1','FN1'],
        ['CSDE1','SFTPA2'],
        ['CSDE1','SFTPC'],
        ['CSDE1','TG'],
        ['CSF1R','FN1'],
        ['CSF2RA','CTSB'],
        ['CSF2RB','SFTPC'],
        ['CSF3R','SFTPA2'],
        ['CSF3R','SFTPC'],
        ['CSNK1A1','SFTPA1'],
        ['CSNK1A1','SFTPC'],
        ['CSNK1A1','TG'],
        ['CSNK1D','FN1'],
        ['CSNK1D','SFTPA1'],
        ['CSNK1D','SFTPC'],
        ['CSNK1E','GNAS'],
        ['CSNK1G1','CTD-2116N17.1'],
        ['CSNK2A2','FBXL2'],
        ['CSNK2B','LY6G5B'],
        ['CSPG4','CSPG4P13'],
        ['CSPG4','FN1'],
        ['CSPG4','SDC3'],
        ['CSRNP1','SFTPA2'],
        ['CSRNP1','SFTPC'],
        ['CSRP1','EMILIN1'],
        ['CSRP1','FLNA'],
        ['CSRP1','FN1'],
        ['CSRP1','MYH9'],
        ['CSRP1','NRGN'],
        ['CSRP1','PTRF'],
        ['CSRP1','SERPINE1'],
        ['CSRP1','SERPINH1'],
        ['CSRP1','SFTPA1'],
        ['CSRP1','SFTPA2'],
        ['CSRP1','SFTPB'],
        ['CSRP1','SFTPC'],
        ['CSRP1','SPARC'],
        ['CSRP1','THBS1'],
        ['CSRP1','VIM'],
        ['CSRP3','TG'],
        ['CST1','CST2'],
        ['CST3','FN1'],
        ['CST3','SFTPC'],
        ['CST3','TG'],
        ['CSTB','FN1'],
        ['CTA-221G9.10','STMN3'],
        ['CTA-250D10.23','MIR378I'],
        ['CTA-276O3.4','RPS29'],
        ['CTA-351J1.1','RPL13A'],
        ['CTA-38K21.4','CTA-38K21.5'],
        ['CTA-796E4.5','MYO18B'],
        ['CTA-929C8.8','CTA-929C8.9'],
        ['CTA-963H5.5','DUSP18'],
        ['CTAGE11P','CTAGE5'],
        ['CTAGE12P','CTAGE5'],
        ['CTAGE12P','RP11-407N17.3'],
        ['CTAGE5','MIA2'],
        ['CTB-102L5.4','SPINT2'],
        ['CTB-119C2.1','HNRNPA2B1'],
        ['CTB-133G6.1','CTD-2207O23.3'],
        ['CTB-151G24.1','ZNF585A'],
        ['CTB-179K24.4','ZNF296'],
        ['CTB-191K22.5','SFTPC'],
        ['CTB-191K22.5','TNS1'],
        ['CTB-31N19.3','METTL9'],
        ['CTB-31O20.2','KLF16'],
        ['CTB-33G10.1','RPS9'],
        ['CTB-50E14.5','ZFAND5'],
        ['CTB-50L17.14','PLIN5'],
        ['CTB-54O9.9','PRR22'],
        ['CTBP2P8','WLS'],
        ['CTBS','GNG5'],
        ['CTC-232H10.1','RPL29'],
        ['CTC-246B18.8','GMFG'],
        ['CTC-250I14.6','SFTPB'],
        ['CTC-260E6.2','RPS16'],
        ['CTC-260E6.4','CTC-260E6.6'],
        ['CTC-264K15.6','SCN4A'],
        ['CTC-273B12.10','LMTK3'],
        ['CTC-273B12.8','CYTH2'],
        ['CTC-276P9.1','PITX1'],
        ['CTC-301O7.4','SLC6A16'],
        ['CTC-308K20.3','PPA1'],
        ['CTC-325L16.1','DPY19L3'],
        ['CTC-326K19.6','IL4I1'],
        ['CTC-347C20.1','CTC-347C20.2'],
        ['CTC-378H22.1','CTC-378H22.2'],
        ['CTC-429P9.3','SMIM7'],
        ['CTC-441N14.4','ZNF474'],
        ['CTC-448F2.3','PPIA'],
        ['CTC-471F3.4','PRR13'],
        ['CTC-479C5.10','EDC4'],
        ['CTC-479C5.12','CTRL'],
        ['CTC-479C5.12','PSMB10'],
        ['CTC-479C5.17','LCAT'],
        ['CTC-498J12.1','RP11-141O11.2'],
        ['CTC-518P12.6','EHD2'],
        ['CTC-548K16.6','PKN1'],
        ['CTC-575D19.1','RPL10'],
        ['CTC-786C10.1','GSE1'],
        ['CTC-786C10.1','RP11-680G10.1'],
        ['CTD-2008L17.2','LINC01539'],
        ['CTD-2008L17.2','RP11-456O19.4'],
        ['CTD-2008L17.2','RP11-456O19.5'],
        ['CTD-2014B16.2','FTH1'],
        ['CTD-2017D11.1','ZNF254'],
        ['CTD-2021A8.2','RP11-274B21.3'],
        ['CTD-2023N9.2','CTD-2023N9.3'],
        ['CTD-2023N9.2','LINCR-0003'],
        ['CTD-2033D15.2','THBS1'],
        ['CTD-2047H16.4','RNF213'],
        ['CTD-2061E19.1','RPS18'],
        ['CTD-2066L21.2','CTD-2066L21.3'],
        ['CTD-2086L14.1','RPL13'],
        ['CTD-2119F7.2','HNRNPA1'],
        ['CTD-2144E22.10','VPS35'],
        ['CTD-2154I11.2','SPATA9'],
        ['CTD-2161F6.1','RP11-445O3.2'],
        ['CTD-2173L22.4','ZNF658'],
        ['CTD-2186M15.1','RPL27'],
        ['CTD-2192J16.15','RPL10'],
        ['CTD-2196E14.9','DCTN5'],
        ['CTD-2207P18.2','LTBP2'],
        ['CTD-2224J9.8','ZNF765'],
        ['CTD-2245E15.3','SDHAP3'],
        ['CTD-2248H3.1','RPS12'],
        ['CTD-2270N23.1','RPL7A'],
        ['CTD-2277K2.1','SYT16'],
        ['CTD-2287O16.1','RPS25'],
        ['CTD-2313J17.2','FAM174B'],
        ['CTD-2323K18.1','PTPN9'],
        ['CTD-2323K18.1','SNUPN'],
        ['CTD-2334D19.1','PRR16'],
        ['CTD-2349P21.9','SUZ12P'],
        ['CTD-2349P21.9','SUZ12P1'],
        ['CTD-2369P2.2','PTRF'],
        ['CTD-2369P2.2','S1PR2'],
        ['CTD-2369P2.8','EPAS1'],
        ['CTD-2369P2.8','MALAT1'],
        ['CTD-2369P2.8','SFTPA2'],
        ['CTD-2369P2.8','SFTPB'],
        ['CTD-2369P2.8','SFTPC'],
        ['CTD-2396E7.10','MALAT1'],
        ['CTD-2396E7.10','SYP'],
        ['CTD-2396E7.9','MALAT1'],
        ['CTD-2396E7.9','SYP'],
        ['CTD-2501E16.2','RORA-AS1'],
        ['CTD-2501E16.2','RP11-219B17.1'],
        ['CTD-2528L19.4','ZNF781'],
        ['CTD-2531D15.4','HLA-DRA'],
        ['CTD-2531D15.4','HLA-E'],
        ['CTD-2531D15.4','SFTPA2'],
        ['CTD-2531D15.4','SFTPC'],
        ['CTD-2545G14.4','DLG4'],
        ['CTD-2545G14.7','RP11-6B6.3'],
        ['CTD-2553L13.9','SCGB2B2'],
        ['CTD-2554C21.1','IST1'],
        ['CTD-2561J22.2','ZNF738'],
        ['CTD-2574D22.4','KCTD13'],
        ['CTD-2574D22.6','MVP'],
        ['CTD-2575K13.6','RABAC1'],
        ['CTD-2576F9.1','GLTSCR2'],
        ['CTD-2589H19.4','ZDHHC11B'],
        ['CTD-2611O12.7','CTD-2611O12.8'],
        ['CTD-2619J13.8','ZNF497'],
        ['CTD-2623N2.3','ZNF846'],
        ['CTD-3035D6.1','RPS18'],
        ['CTD-3080F16.3','SFRP1'],
        ['CTD-3093M3.1','TSEN34'],
        ['CTD-3096M3.2','EPB41L3'],
        ['CTD-3195I5.4','RP11-219B4.7'],
        ['CTD-3199J23.4','RND2'],
        ['CTD-3199J23.6','NBR2'],
        ['CTD-3214H19.16','MCEMP1'],
        ['CTD-3214H19.4','PET100'],
        ['CTD-3214H19.6','PCP2'],
        ['CTD-3222D19.2','EPS15L1'],
        ['CTD-3222D19.2','SLC35E1'],
        ['CTD-3252C9.4','MIR27A'],
        ['CTD-3252C9.4','SFTPB'],
        ['CTDSP1','SFTPA1'],
        ['CTDSP1','SFTPC'],
        ['CTDSP1','SLC11A1'],
        ['CTDSP1','TG'],
        ['CTDSP2','DES'],
        ['CTDSP2','SFTPA1'],
        ['CTDSP2','TG'],
        ['CTGF','FLNA'],
        ['CTGF','FLNC'],
        ['CTGF','FN1'],
        ['CTGF','HSPG2'],
        ['CTGF','IGFBP3'],
        ['CTGF','ITGA5'],
        ['CTGF','LTBP2'],
        ['CTGF','MYH9'],
        ['CTGF','PDXP'],
        ['CTGF','PLEC'],
        ['CTGF','RPS19BP1'],
        ['CTGF','SERPINE1'],
        ['CTGF','SFTPA1'],
        ['CTGF','SFTPA2'],
        ['CTGF','SFTPC'],
        ['CTGF','SPARC'],
        ['CTGF','THBS1'],
        ['CTGF','VIM'],
        ['CTNNA1','CTNNAP1'],
        ['CTNNA1','FN1'],
        ['CTNNA1','KRT4'],
        ['CTNNB1','SFTPA2'],
        ['CTNNB1','SFTPC'],
        ['CTNND1','EPAS1'],
        ['CTNND1','FN1'],
        ['CTNND1','RP11-691N7.6'],
        ['CTNND1','SFTPA1'],
        ['CTNND1','SFTPB'],
        ['CTNND1','SFTPC'],
        ['CTR9','RP11-685M7.5'],
        ['CTR9','SNORD97'],
        ['CTSA','FN1'],
        ['CTSA','PLTP'],
        ['CTSA','SFTPB'],
        ['CTSA','SFTPC'],
        ['CTSA','TERF2IP'],
        ['CTSB','EHD2'],
        ['CTSB','EPAS1'],
        ['CTSB','FLNA'],
        ['CTSB','FLNC'],
        ['CTSB','FN1'],
        ['CTSB','GNAS'],
        ['CTSB','H19'],
        ['CTSB','HLA-C'],
        ['CTSB','HNRNPH1'],
        ['CTSB','IGFBP5'],
        ['CTSB','LOXL2'],
        ['CTSB','MMP14'],
        ['CTSB','MYH9'],
        ['CTSB','MYOZ1'],
        ['CTSB','PLEC'],
        ['CTSB','PSAP'],
        ['CTSB','PTRF'],
        ['CTSB','RP1-309I22.2'],
        ['CTSB','SCUBE3'],
        ['CTSB','SERPINE2'],
        ['CTSB','SFTPA2'],
        ['CTSB','SFTPB'],
        ['CTSB','SFTPC'],
        ['CTSB','SOD2'],
        ['CTSB','SPARC'],
        ['CTSB','STC1'],
        ['CTSB','TFPI2'],
        ['CTSB','TG'],
        ['CTSB','THBS1'],
        ['CTSB','TIMP3'],
        ['CTSB','TLN1'],
        ['CTSB','TPO'],
        ['CTSB','VIM'],
        ['CTSC','RAB38'],
        ['CTSC','SFTPC'],
        ['CTSD','EPAS1'],
        ['CTSD','FLNA'],
        ['CTSD','FN1'],
        ['CTSD','FTL'],
        ['CTSD','GPBP1L1'],
        ['CTSD','HLA-B'],
        ['CTSD','ICAM1'],
        ['CTSD','IFITM10'],
        ['CTSD','IGFBP5'],
        ['CTSD','IGHA1'],
        ['CTSD','IGK@'],
        ['CTSD','IGKC'],
        ['CTSD','PSAP'],
        ['CTSD','RP11-295K3.1'],
        ['CTSD','SFTPA1'],
        ['CTSD','SFTPA2'],
        ['CTSD','SFTPB'],
        ['CTSD','SFTPC'],
        ['CTSD','SLC34A2'],
        ['CTSD','SPTBN1'],
        ['CTSD','TG'],
        ['CTSD','TIMP3'],
        ['CTSD','TPT1'],
        ['CTSD','VWF'],
        ['CTSD','ZFP36'],
        ['CTSE','LYZ'],
        ['CTSH','SFTPA1'],
        ['CTSH','SFTPC'],
        ['CTSL','CTSL3P'],
        ['CTSL','FN1'],
        ['CTSS','GPRC5A'],
        ['CTSS','KIRREL3'],
        ['CTSS','LYZ'],
        ['CTSS','RP11-1143G9.4'],
        ['CTSZ','FLNA'],
        ['CTSZ','FN1'],
        ['CTSZ','GAS6'],
        ['CTSZ','PTOV1'],
        ['CTSZ','SFTPA1'],
        ['CTSZ','SFTPA2'],
        ['CTSZ','SFTPC'],
        ['CTTN','FN1'],
        ['CTTN','SFTPC'],
        ['CUL2','RP11-297A16.2'],
        ['CUL7','FN1'],
        ['CUTC','ENTPD7'],
        ['CUZD1','GP2'],
        ['CWF19L1','PTRF'],
        ['CX3CL1','SFTPA2'],
        ['CX3CL1','SFTPB'],
        ['CX3CL1','SFTPC'],
        ['CXCL12','DNAJC5'],
        ['CXCL12','FN1'],
        ['CXCL12','PXDC1'],
        ['CXCL16','SFTPB'],
        ['CXCL16','SFTPC'],
        ['CXCR1','CXCR2'],
        ['CXORF36','SFTPC'],
        ['CXORF65','IL2RG'],
        ['CXXC5','UBE2D2'],
        ['CYB5A','SFTPB'],
        ['CYB5D1','RP11-1099M24.6'],
        ['CYB5R3','FLNA'],
        ['CYB5R3','FN1'],
        ['CYB5R3','FTH1'],
        ['CYB5R3','MAP7D1'],
        ['CYB5R3','PTMS'],
        ['CYB5R3','PTRF'],
        ['CYB5R3','SFTPA1'],
        ['CYB5R3','SFTPA2'],
        ['CYB5R3','SFTPC'],
        ['CYB5R3','SPARC'],
        ['CYB5R3','TNNC2'],
        ['CYB5R3','TUBB4B'],
        ['CYBA','SFTPC'],
        ['CYBA','ZBTB7A'],
        ['CYBRD1','FN1'],
        ['CYBRD1','SFTPB'],
        ['CYBRD1','SFTPC'],
        ['CYC1','FN1'],
        ['CYCS','CYCSP45'],
        ['CYFIP1','SFTPB'],
        ['CYFIP1','SFTPC'],
        ['CYP1B1','RMDN2-AS1'],
        ['CYP21A1P','CYP21A2'],
        ['CYP27A1','SFTPC'],
        ['CYP2A13','CYP2A6'],
        ['CYP2A6','CYP2A7'],
        ['CYP2A6','LRP1'],
        ['CYP2B6','CYP2B7P1'],
        ['CYP2B6','ECE1'],
        ['CYP2B6','LRG1'],
        ['CYP2B6','QPRT'],
        ['CYP2B6','SULT2A1'],
        ['CYP2B7P1','LRG1'],
        ['CYP2C19','CYP2C9'],
        ['CYP2C8','CYP2C9'],
        ['CYP2C9','RP11-400G3.5'],
        ['CYP2D6','CYP2D7'],
        ['CYP2D6','CYP2D7P1'],
        ['CYP2D7','CYP2D8P'],
        ['CYP2E1','FGB'],
        ['CYP2E1','HP'],
        ['CYP2E1','RP11-108K14.4'],
        ['CYP2E1','SAA1'],
        ['CYP2E1','SERPINA1'],
        ['CYP2F1','CYP2F2P'],
        ['CYP4A11','CYP4A22'],
        ['CYP4B1','CYP4X1'],
        ['CYP4B1','SFTPA1'],
        ['CYP4B1','SFTPA2'],
        ['CYP4B1','SFTPC'],
        ['CYP4Z1','CYP4Z2P'],
        ['CYR61','DUSP7'],
        ['CYR61','FLNA'],
        ['CYR61','FN1'],
        ['CYR61','HSPG2'],
        ['CYR61','MYH9'],
        ['CYR61','PLEC'],
        ['CYR61','SFTPC'],
        ['CYR61','ZFP36'],
        ['CYTH1','USP36'],
        ['D86994.1','IGLC1'],
        ['D86994.1','IGLL5'],
        ['D86994.1','IGLV2-11'],
        ['D86994.1','IGLV2-8'],
        ['D86994.2','IGLC1'],
        ['D86994.2','IGLL5'],
        ['D86994.2','IGLV2-11'],
        ['D86994.2','IGLV2-14'],
        ['D86994.2','IGLV2-18'],
        ['D86994.2','IGLV2-23'],
        ['D87007.1','IGLV2-18'],
        ['D87024.1','IGLV2-5'],
        ['DAAM2','MYH9'],
        ['DAB2','FN1'],
        ['DAB2IP','FN1'],
        ['DAG1','FN1'],
        ['DANCR','SNORA26'],
        ['DAP','EHD3'],
        ['DAP','FN1'],
        ['DAP','ID1'],
        ['DAPK3','FLNA'],
        ['DAPK3','FN1'],
        ['DAZAP1','FN1'],
        ['DAZAP2','SFTPC'],
        ['DBET','DUX4'],
        ['DBF4','DBF4P1'],
        ['DBN1','FN1'],
        ['DBN1','MYH9'],
        ['DBN1','NUP62'],
        ['DBN1','PLEC'],
        ['DBN1','TBC1D10B'],
        ['DBNL','SFTPC'],
        ['DCAF13','DCAF13P1'],
        ['DCAF4','RP11-109N23.6'],
        ['DCAF5','SFTPC'],
        ['DCHS1','SFTPC'],
        ['DCN','EEF2'],
        ['DCN','FN1'],
        ['DCN','IGKC'],
        ['DCN','SFTPA1'],
        ['DCN','SFTPA2'],
        ['DCN','SFTPB'],
        ['DCN','SFTPC'],
        ['DCN','VIM'],
        ['DCPS','TIRAP'],
        ['DCTN1','FLNA'],
        ['DCTN1','FN1'],
        ['DCTN1','MYH9'],
        ['DCTN1','SFTPC'],
        ['DCTN1','SPARC'],
        ['DCTN1','VIM'],
        ['DCUN1D3','ERI2'],
        ['DCXR','RP11-1182P23.5'],
        ['DDB1','FLNA'],
        ['DDB1','FN1'],
        ['DDB1','PTRF'],
        ['DDB1','SFTPC'],
        ['DDB1','THBS1'],
        ['DDOST','EHD2'],
        ['DDOST','FN1'],
        ['DDOST','PINK1-AS'],
        ['DDR2','FN1'],
        ['DDRGK1','PROSAPIP1'],
        ['DDX11','RP11-22B23.1'],
        ['DDX12P','RP11-22B23.1'],
        ['DDX17','DMC1'],
        ['DDX17','FN1'],
        ['DDX17','MYH9'],
        ['DDX17','SFTPA1'],
        ['DDX17','SFTPA2'],
        ['DDX17','SFTPB'],
        ['DDX17','SFTPC'],
        ['DDX17','TG'],
        ['DDX17','VWF'],
        ['DDX19A','EHD2'],
        ['DDX23','FN1'],
        ['DDX23','RP11-641A6.5'],
        ['DDX39B','FN1'],
        ['DDX39B','SFTPB'],
        ['DDX3X','DDX3Y'],
        ['DDX3X','RP11-613H2.2'],
        ['DDX3X','SFTPA2'],
        ['DDX3X','SFTPC'],
        ['DDX5','EPAS1'],
        ['DDX5','FN1'],
        ['DDX5','SFTPA2'],
        ['DDX5','SFTPB'],
        ['DDX5','SFTPC'],
        ['DDX50','DDX50P2'],
        ['DDX56','TMED4'],
        ['DEGS1','FN1'],
        ['DENND3','SFTPC'],
        ['DENND4A','SNOU13'],
        ['DENND4B','SFTPC'],
        ['DENND5A','FN1'],
        ['DENND5A','LOX'],
        ['DENND5A','PCBP1'],
        ['DENR','RP11-290L1.4'],
        ['DES','EEF2'],
        ['DES','EIF4G1'],
        ['DES','FLNC'],
        ['DES','GNAS'],
        ['DES','H19'],
        ['DES','H1F0'],
        ['DES','IGFBP5'],
        ['DES','MYH11'],
        ['DES','MYH7'],
        ['DES','MYL2'],
        ['DES','MYL3'],
        ['DES','MYL9'],
        ['DES','NEAT1'],
        ['DES','TCAP'],
        ['DES','TG'],
        ['DES','TIMP2'],
        ['DES','TPM2'],
        ['DES','TPO'],
        ['DGAT2','H19'],
        ['DGAT2','PSAP'],
        ['DGCR11','DGCR2'],
        ['DGCR2','EPN1'],
        ['DGCR8','MRC2'],
        ['DGKZ','FLNC'],
        ['DHCR24','FLNA'],
        ['DHCR24','FN1'],
        ['DHCR24','SFTPA1'],
        ['DHCR24','SFTPA2'],
        ['DHCR24','SFTPB'],
        ['DHCR24','SFTPC'],
        ['DHCR24','ZFP36'],
        ['DHDDS','PTRF'],
        ['DHRS1','RABGGTA'],
        ['DHRS4','DHRS4L2'],
        ['DHX9','DHX9P1'],
        ['DIAPH1','FN1'],
        ['DIAPH1','PRKCSH'],
        ['DIAPH1','SFTPC'],
        ['DIP2A','DIP2A-IT1'],
        ['DIRAS3','WLS'],
        ['DISP1','RP11-452F19.3'],
        ['DISP2','LINC00594'],
        ['DKFZP761J1410','FN1'],
        ['DKK1','FN1'],
        ['DKK3','FN1'],
        ['DKK3','HLA-B'],
        ['DKK3','PLEC'],
        ['DKK3','VIM-AS1'],
        ['DLC1','SFTPA2'],
        ['DLC1','SFTPC'],
        ['DLEU2L','EFCAB7'],
        ['DLG5','TG'],
        ['DLGAP4','FN1'],
        ['DLGAP4','LOXL2'],
        ['DLGAP4','SFTPA2'],
        ['DLGAP4','TRB@'],
        ['DLK1','IGF2'],
        ['DMBT1','PIGR'],
        ['DMC1','TMEM184B'],
        ['DMKN','KRT1'],
        ['DMKN','KRTDAP'],
        ['DMTN','TG'],
        ['DMWD','PTBP1'],
        ['DNA2','SLC25A16'],
        ['DNAH1','GLYCTK'],
        ['DNAI1','RP11-296L22.8'],
        ['DNAJA1','DNAJA1P3'],
        ['DNAJB6','FN1'],
        ['DNAJC14','PTRF'],
        ['DNAJC18','ECSCR'],
        ['DNAJC25','GNG10'],
        ['DNAJC25-GNG10','GNG10'],
        ['DNAJC4','NUDT22'],
        ['DNAJC5','FN1'],
        ['DNAJC5','TPD52L2'],
        ['DNAJC9','MRPS16'],
        ['DNER','GFAP'],
        ['DNLZ','GNB2'],
        ['DNLZ','ITGB5'],
        ['DNM1','DNM1P33'],
        ['DNM2','FN1'],
        ['DNM2','SFTPA2'],
        ['DNM2','SFTPC'],
        ['DNM3OS','MIR199A2'],
        ['DNMT1','FN1'],
        ['DNMT1','S1PR2'],
        ['DOCK6','SFTPC'],
        ['DOCK9','SFTPC'],
        ['DPH1','OVCA2'],
        ['DPM2','PIP5KL1'],
        ['DPP3','LYZ'],
        ['DPP9','FN1'],
        ['DPY19L2','DPY19L2P1'],
        ['DPY19L2','DPY19L2P2'],
        ['DPY19L2','DPY19L2P4'],
        ['DPY19L2P1','DPY19L2P2'],
        ['DPY19L2P1','DPY19L2P3'],
        ['DPYSL2','EPAS1'],
        ['DPYSL2','FN1'],
        ['DPYSL2','SFTPA1'],
        ['DPYSL2','SFTPA2'],
        ['DPYSL2','SFTPB'],
        ['DPYSL2','SFTPC'],
        ['DPYSL3','SFTPC'],
        ['DR1','IGFBP2'],
        ['DR1','PTMS'],
        ['DRAP1','FN1'],
        ['DRAP1','GPR124'],
        ['DRAP1','MARCKS'],
        ['DRAP1','PALLD'],
        ['DRAP1','PLEC'],
        ['DRAP1','SAPCD2'],
        ['DRD4','PHRF1'],
        ['DRICH1','KB-1572G7.3'],
        ['DSC3','FLG2'],
        ['DSC3','KRT10'],
        ['DSCR3','PTRF'],
        ['DSE','FAM26F'],
        ['DSE','FN1'],
        ['DSE','RP1-93H18.1'],
        ['DSE','RP3-486I3.7'],
        ['DSP','FN1'],
        ['DSP','KRT10'],
        ['DSP','KRT2'],
        ['DST','FN1'],
        ['DST','KRT10'],
        ['DST','SFTPA1'],
        ['DST','SFTPC'],
        ['DST','SPARC'],
        ['DST','VIM'],
        ['DSTN','DSTNP2'],
        ['DSTN','DSTNP4'],
        ['DSTN','FN1'],
        ['DSTN','SFTPB'],
        ['DSTNP2','RPL13P5'],
        ['DSTNP2','U47924.6'],
        ['DSTYK','LYZ'],
        ['DTWD1','RP11-353B9.1'],
        ['DTX2','DTX2P1'],
        ['DTX2P1','UPK3B'],
        ['DTX2P1-UPK3BP1-PMS2P11','UPK3B'],
        ['DUOX1','SFTPC'],
        ['DUS3L','PRR22'],
        ['DUSP1','FLNC'],
        ['DUSP1','FN1'],
        ['DUSP1','HLA-B'],
        ['DUSP1','SFTPA1'],
        ['DUSP1','SFTPA2'],
        ['DUSP1','SFTPB'],
        ['DUSP1','SFTPC'],
        ['DUSP1','TG'],
        ['DUSP16','GPR19'],
        ['DUSP22','RP11-23E10.2'],
        ['DUSP3','FGG'],
        ['DUSP6','SFTPA2'],
        ['DUSP6','SFTPC'],
        ['DUSP8P5','GLUD1P3'],
        ['DUT','ZNF534'],
        ['DVL1','FN1'],
        ['DVL1','GNAS'],
        ['DVL1','TNFRSF1A'],
        ['DVL3','EHD2'],
        ['DVL3','IRF2BPL'],
        ['DYNC1H1','EPAS1'],
        ['DYNC1H1','FLNA'],
        ['DYNC1H1','FN1'],
        ['DYNC1H1','MYH9'],
        ['DYNC1H1','PSAP'],
        ['DYNC1H1','SFTPA1'],
        ['DYNC1H1','SFTPA2'],
        ['DYNC1H1','SFTPC'],
        ['DYNC1H1','SLC9A3R2'],
        ['DYNC1H1','TG'],
        ['DYNC1H1','THBS1'],
        ['DYNLL2','FN1'],
        ['DYNLL2','PTRF'],
        ['DYNLL2','RP11-159D12.8'],
        ['DYNLL2','TG'],
        ['DYNLRB1','ITCH'],
        ['DYSF','SFTPA2'],
        ['DYX1C1-CCPG1','PIGBOS1'],
        ['E2F3P2','MRE11A'],
        ['E2F4','RPL14'],
        ['E2F5','LRRCC1'],
        ['EARS2','GGA2'],
        ['EBPL','RP11-744D14.1'],
        ['ECE1','EHD2'],
        ['ECE1','SFTPA2'],
        ['ECE1','TMEM119'],
        ['ECM1','FN1'],
        ['EDARADD','ENO1'],
        ['EDEM2','MMP24-AS1'],
        ['EDN1','SFTPC'],
        ['EDNRA','SFTPC'],
        ['EED','RP11-320L11.2'],
        ['EEF1A1','EEF1A1P10'],
        ['EEF1A1','EEF1A1P11'],
        ['EEF1A1','EEF1A1P12'],
        ['EEF1A1','EEF1A1P13'],
        ['EEF1A1','EEF1A1P14'],
        ['EEF1A1','EEF1A1P16'],
        ['EEF1A1','EEF1A1P19'],
        ['EEF1A1','EEF1A1P22'],
        ['EEF1A1','EEF1A1P24'],
        ['EEF1A1','EEF1A1P29'],
        ['EEF1A1','EEF1A1P4'],
        ['EEF1A1','EEF1A1P6'],
        ['EEF1A1','EEF1A1P8'],
        ['EEF1A1','EEF1A1P9'],
        ['EEF1A1','RP11-553P9.1'],
        ['EEF1A2','TG'],
        ['EEF1B2','EEF1B2P1'],
        ['EEF1B2','EEF1B2P3'],
        ['EEF1D','EEF1DP1'],
        ['EEF1D','EEF1DP5'],
        ['EEF1D','FLNA'],
        ['EEF1D','HDAC5'],
        ['EEF1DP3','FRY'],
        ['EEF1G','EEF1GP1'],
        ['EEF1G','EEF1GP5'],
        ['EEF1G','FLNA'],
        ['EEF1G','FN1'],
        ['EEF1G','MIR3654'],
        ['EEF1G','MYH9'],
        ['EEF1G','TG'],
        ['EEF1GP1','MIR3654'],
        ['EEF1GP1','RP11-864I4.1'],
        ['EEF1GP5','RP11-864I4.1'],
        ['EEF2','EPAS1'],
        ['EEF2','FASN'],
        ['EEF2','FKBP10'],
        ['EEF2','FLNA'],
        ['EEF2','FLNC'],
        ['EEF2','FN1'],
        ['EEF2','FTL'],
        ['EEF2','GNAI2'],
        ['EEF2','GNAS'],
        ['EEF2','GSN'],
        ['EEF2','HDLBP'],
        ['EEF2','HLA-B'],
        ['EEF2','HLA-C'],
        ['EEF2','HLA-E'],
        ['EEF2','HSPG2'],
        ['EEF2','ICAM1'],
        ['EEF2','IGF2'],
        ['EEF2','IGFBP3'],
        ['EEF2','IGFBP4'],
        ['EEF2','IGFBP5'],
        ['EEF2','IGH@'],
        ['EEF2','IGHA1'],
        ['EEF2','IGHG1'],
        ['EEF2','IGHM'],
        ['EEF2','IYD'],
        ['EEF2','KRT10'],
        ['EEF2','LASP1'],
        ['EEF2','LGALS1'],
        ['EEF2','LOXL2'],
        ['EEF2','LRP1'],
        ['EEF2','LTBP4'],
        ['EEF2','MGLL'],
        ['EEF2','MSN'],
        ['EEF2','MUC1'],
        ['EEF2','MYBPC1'],
        ['EEF2','MYH9'],
        ['EEF2','MYL9'],
        ['EEF2','NCOR2'],
        ['EEF2','NEAT1'],
        ['EEF2','NR4A1'],
        ['EEF2','PFN1'],
        ['EEF2','PIEZO1'],
        ['EEF2','PKM'],
        ['EEF2','PLEC'],
        ['EEF2','PLIN4'],
        ['EEF2','PSAP'],
        ['EEF2','PTRF'],
        ['EEF2','RRBP1'],
        ['EEF2','SCUBE3'],
        ['EEF2','SERPINE1'],
        ['EEF2','SFTPA1'],
        ['EEF2','SFTPA2'],
        ['EEF2','SFTPB'],
        ['EEF2','SFTPC'],
        ['EEF2','SLC34A2'],
        ['EEF2','SNORD37'],
        ['EEF2','SPARC'],
        ['EEF2','SPTBN1'],
        ['EEF2','TAGLN'],
        ['EEF2','TG'],
        ['EEF2','TGFBI'],
        ['EEF2','TGM2'],
        ['EEF2','THBS1'],
        ['EEF2','TIMP1'],
        ['EEF2','TIMP2'],
        ['EEF2','TLN1'],
        ['EEF2','TPM2'],
        ['EEF2','TPO'],
        ['EEF2','TXNIP'],
        ['EEF2','VIM'],
        ['EEF2','VIM-AS1'],
        ['EEF2','WDR1'],
        ['EEF2','ZYX'],
        ['EEF2K','SFTPB'],
        ['EFEMP1','FN1'],
        ['EFEMP1','SFTPA2'],
        ['EFEMP1','SFTPC'],
        ['EFEMP2','FLNA'],
        ['EFEMP2','FN1'],
        ['EFEMP2','GNAS'],
        ['EFEMP2','THBS1'],
        ['EFHD2','PPP1R9B'],
        ['EFHD2','SFTPC'],
        ['EGFL7','SFTPC'],
        ['EGFR','SFTPC'],
        ['EGFR','SSR3'],
        ['EGLN1','SCAND2P'],
        ['EGR1','EPAS1'],
        ['EGR1','FLNA'],
        ['EGR1','FN1'],
        ['EGR1','FOSB'],
        ['EGR1','IGKC'],
        ['EGR1','PLIN4'],
        ['EGR1','PSAP'],
        ['EGR1','SFTPA1'],
        ['EGR1','SFTPA2'],
        ['EGR1','SFTPB'],
        ['EGR1','SFTPC'],
        ['EGR1','TG'],
        ['EGR1','TPO'],
        ['EGR1','ZFP36'],
        ['EHBP1L1','FN1'],
        ['EHBP1L1','SFTPC'],
        ['EHBP1L1','TG'],
        ['EHD1','EPAS1'],
        ['EHD1','FN1'],
        ['EHD1','SFTPC'],
        ['EHD2','EIF3I'],
        ['EHD2','EMP2'],
        ['EHD2','FAM212B'],
        ['EHD2','FBLIM1'],
        ['EHD2','FGFR1'],
        ['EHD2','FKBP14'],
        ['EHD2','FLNA'],
        ['EHD2','FLNC'],
        ['EHD2','FN1'],
        ['EHD2','GLTP'],
        ['EHD2','HIP1'],
        ['EHD2','IRAK1'],
        ['EHD2','KIAA0226'],
        ['EHD2','KIAA1191'],
        ['EHD2','LDLR'],
        ['EHD2','LEPREL4'],
        ['EHD2','LRP1'],
        ['EHD2','MCAM'],
        ['EHD2','MSRB3'],
        ['EHD2','MYADM'],
        ['EHD2','MYH9'],
        ['EHD2','MYO1C'],
        ['EHD2','NCAPD2'],
        ['EHD2','NUP62'],
        ['EHD2','ORAI2'],
        ['EHD2','PDLIM5'],
        ['EHD2','PIK3R2'],
        ['EHD2','PLEC'],
        ['EHD2','PLIN3'],
        ['EHD2','PLIN4'],
        ['EHD2','PTRF'],
        ['EHD2','RAB3B'],
        ['EHD2','RBM3'],
        ['EHD2','RFC2'],
        ['EHD2','RGS5'],
        ['EHD2','RRBP1'],
        ['EHD2','SERPINE1'],
        ['EHD2','SFTPA1'],
        ['EHD2','SFTPA2'],
        ['EHD2','SFTPB'],
        ['EHD2','SFTPC'],
        ['EHD2','SLC2A6'],
        ['EHD2','SLC31A1'],
        ['EHD2','SPTLC2'],
        ['EHD2','TCTA'],
        ['EHD2','THBS1'],
        ['EHD2','TPM4'],
        ['EHD2','TPRG1L'],
        ['EHD2','TRAM2'],
        ['EHD2','VIM'],
        ['EHD2','WDR82'],
        ['EHD2','WIPF2'],
        ['EHD2','YIPF5'],
        ['EHD2','ZNF106'],
        ['EHD2','ZNFX1'],
        ['EHD3','FN1'],
        ['EHD3','IGF2'],
        ['EHD3','MYH9'],
        ['EHMT1','EHMT1-IT1'],
        ['EI24','EI24P2'],
        ['EID1','FN1'],
        ['EIF1','EIF1P7'],
        ['EIF1','FN1'],
        ['EIF1','SFTPC'],
        ['EIF2B3','PTCH2'],
        ['EIF2S3','EIF2S3L'],
        ['EIF2S3','PTRF'],
        ['EIF2S3','SFTPB'],
        ['EIF2S3','TPM4'],
        ['EIF3A','FN1'],
        ['EIF3A','PTMS'],
        ['EIF3A','SFTPC'],
        ['EIF3A','SNORA19'],
        ['EIF3B','FBLN2'],
        ['EIF3B','FN1'],
        ['EIF3B','ITGAV'],
        ['EIF3D','SFTPC'],
        ['EIF3E','EIF3EP1'],
        ['EIF3K','SFTPB'],
        ['EIF3L','EIF3LP1'],
        ['EIF3L','EIF3LP3'],
        ['EIF3L','SFTPC'],
        ['EIF3LP3','RP5-1014D13.2'],
        ['EIF4A1','EIF4A1P4'],
        ['EIF4A1P4','RP11-186B7.4'],
        ['EIF4A1P4','SENP3-EIF4A1'],
        ['EIF4A2','EIF4A2P3'],
        ['EIF4A2','EIF4A2P4'],
        ['EIF4A2','SFTPC'],
        ['EIF4B','EIF4BP7'],
        ['EIF4E2','SFTPC'],
        ['EIF4E3','FOXP1'],
        ['EIF4EBP2','PTRF'],
        ['EIF4EBP2','SFTPC'],
        ['EIF4G1','FLNA'],
        ['EIF4G1','FLNC'],
        ['EIF4G1','FN1'],
        ['EIF4G1','LOXL2'],
        ['EIF4G1','MYH9'],
        ['EIF4G1','PLEC'],
        ['EIF4G1','SFTPA1'],
        ['EIF4G1','SFTPA2'],
        ['EIF4G1','SFTPC'],
        ['EIF4G1','SPARC'],
        ['EIF4G1','TG'],
        ['EIF4G1','THBS1'],
        ['EIF4G1','VIM'],
        ['EIF4G2','FN1'],
        ['EIF4G2','SFTPA1'],
        ['EIF4G2','SFTPA2'],
        ['EIF4G2','SFTPB'],
        ['EIF4G2','SFTPC'],
        ['EIF4G2','SNORD97'],
        ['EIF4G2','TG'],
        ['EIF4G2','THBS1'],
        ['EIF4G3','HP1BP3'],
        ['EIF5','SFTPC'],
        ['EIF5','SNORA28'],
        ['EIF5A','EIF5AL1'],
        ['EIF5A','FBRS'],
        ['EIF5A','FN1'],
        ['EIF5A','FTH1'],
        ['EIF5A','IGFBP2'],
        ['EIF5A','IRF2BPL'],
        ['EIF5A','PTMS'],
        ['EIF5A','VAT1'],
        ['EIF6','FN1'],
        ['EIF6','MMP24-AS1'],
        ['ELF3','SFTPC'],
        ['ELFN1','GNAS'],
        ['ELFN1','HNRNPL'],
        ['ELFN2','ELFN2'],
        ['ELFN2','RP1-63G5.5'],
        ['ELK4','SLC45A3'],
        ['ELL2','ELL2P1'],
        ['ELMSAN1','SFTPB'],
        ['ELN','FN1'],
        ['ELN','IGFBP3'],
        ['ELN','PLEC'],
        ['ELN','SFTPC'],
        ['ELN','SPARC'],
        ['ELN','TGFBI'],
        ['ELN','THBS1'],
        ['ELN','U2AF1'],
        ['ELN','VIM'],
        ['ELN','VIM-AS1'],
        ['ELN','ZNF703'],
        ['ELOF1','ITGA11'],
        ['ELOVL1','MED8'],
        ['EMB','EMBP1'],
        ['EMC1','FN1'],
        ['EMC10','FN1'],
        ['EMC8','U1'],
        ['EMILIN1','EWSR1'],
        ['EMILIN1','FLNA'],
        ['EMILIN1','FLNC'],
        ['EMILIN1','FN1'],
        ['EMILIN1','GNAS'],
        ['EMILIN1','MVP'],
        ['EMILIN1','MYH9'],
        ['EMILIN1','PLEC'],
        ['EMILIN1','PTMS'],
        ['EMILIN1','PTRF'],
        ['EMILIN1','SH3PXD2A'],
        ['EMILIN1','TMUB1'],
        ['EMILIN1','TNFRSF12A'],
        ['EMILIN1','TSC22D4'],
        ['EMILIN2','FLNA'],
        ['EMILIN2','FN1'],
        ['EML1','IGFBP4'],
        ['EMP1','FN1'],
        ['EMP1','MALAT1'],
        ['EMP1','SFTPC'],
        ['EMP2','EPAS1'],
        ['EMP2','FLNA'],
        ['EMP2','FN1'],
        ['EMP2','INMT'],
        ['EMP2','LAMP3'],
        ['EMP2','LYZ'],
        ['EMP2','MYH9'],
        ['EMP2','PIGR'],
        ['EMP2','SCARB2'],
        ['EMP2','SFTPA1'],
        ['EMP2','SFTPA2'],
        ['EMP2','SFTPB'],
        ['EMP2','SFTPC'],
        ['EMP2','STOM'],
        ['EMP3','FN1'],
        ['EMP3','VIM'],
        ['EMR2','EMR3'],
        ['EN1','MAZ'],
        ['ENG','FN1'],
        ['ENG','IGFBP5'],
        ['ENG','PRRX2'],
        ['ENG','PTRF'],
        ['ENG','SFTPA1'],
        ['ENG','SFTPA2'],
        ['ENG','SFTPB'],
        ['ENG','SFTPC'],
        ['ENKD1','GFOD2'],
        ['ENO1','FLNA'],
        ['ENO1','FN1'],
        ['ENO1','IGFBP5'],
        ['ENO1','KIAA1199'],
        ['ENO1','MYH9'],
        ['ENO1','PLEC'],
        ['ENO1','SFTPC'],
        ['ENO1','THBS1'],
        ['ENO2','GFAP'],
        ['ENO3','TG'],
        ['ENPP1','FN1'],
        ['ENPP1','KCNE3'],
        ['ENPP1','SERPINE1'],
        ['ENPP2','SFTPA1'],
        ['ENPP2','SFTPC'],
        ['ENTPD4','LOXL2'],
        ['ENTPD6','FN1'],
        ['ENTPD6','UBE2S'],
        ['EOGT','SFTPB'],
        ['EP300','SFTPA1'],
        ['EP300','SFTPC'],
        ['EP400','TBP'],
        ['EPAS1','EZR'],
        ['EPAS1','FASN'],
        ['EPAS1','FLNA'],
        ['EPAS1','FN1'],
        ['EPAS1','FOSB'],
        ['EPAS1','FTL'],
        ['EPAS1','GAA'],
        ['EPAS1','GLUL'],
        ['EPAS1','GNB1'],
        ['EPAS1','GRN'],
        ['EPAS1','HLA-A'],
        ['EPAS1','HLA-B'],
        ['EPAS1','HLA-C'],
        ['EPAS1','HLA-DPA1'],
        ['EPAS1','HLA-DRA'],
        ['EPAS1','HLA-E'],
        ['EPAS1','IGFBP7'],
        ['EPAS1','IGKC'],
        ['EPAS1','LAPTM5'],
        ['EPAS1','LMNA'],
        ['EPAS1','LRP1'],
        ['EPAS1','MACF1'],
        ['EPAS1','MSN'],
        ['EPAS1','MYH9'],
        ['EPAS1','NCOR2'],
        ['EPAS1','NEAT1'],
        ['EPAS1','NEDD9'],
        ['EPAS1','NUMA1'],
        ['EPAS1','PLEC'],
        ['EPAS1','PRRC2C'],
        ['EPAS1','PSAP'],
        ['EPAS1','PTRF'],
        ['EPAS1','RNF5'],
        ['EPAS1','RP11-186B7.4'],
        ['EPAS1','RPL8'],
        ['EPAS1','S100A6'],
        ['EPAS1','SCGB1A1'],
        ['EPAS1','SEPP1'],
        ['EPAS1','SERPINA1'],
        ['EPAS1','SF1'],
        ['EPAS1','SFTPA1'],
        ['EPAS1','SFTPA2'],
        ['EPAS1','SFTPB'],
        ['EPAS1','SFTPC'],
        ['EPAS1','SLC34A2'],
        ['EPAS1','SPARC'],
        ['EPAS1','SPTBN1'],
        ['EPAS1','SPTLC1'],
        ['EPAS1','STOM'],
        ['EPAS1','TG'],
        ['EPAS1','TGM2'],
        ['EPAS1','TIMP3'],
        ['EPAS1','TNS1'],
        ['EPAS1','TPT1'],
        ['EPAS1','TSC22D3'],
        ['EPAS1','TXNIP'],
        ['EPAS1','UBXN11'],
        ['EPAS1','VIM'],
        ['EPAS1','ZFP36'],
        ['EPB41L2','IGFBP5'],
        ['EPB41L2','SFTPB'],
        ['EPB41L2','SFTPC'],
        ['EPB41L4A-AS1','SNORA13'],
        ['EPCAM','RP11-433O3.1'],
        ['EPHB4','MAP7D1'],
        ['EPHX1','FGG'],
        ['EPHX1','SERPINA1'],
        ['EPHX1','SFTPC'],
        ['EPN1','FLNA'],
        ['EPN1','FN1'],
        ['EPN1','ITGA3'],
        ['EPN1','MMP14'],
        ['EPN1','THBS1'],
        ['EPOR','RGL3'],
        ['EPPIN','EPPIN-WFDC6'],
        ['EPPIN','RP3-461P17.6'],
        ['EPPIN','WFDC6'],
        ['EPPIN-WFDC6','RP3-461P17.6'],
        ['EPPIN-WFDC6','WFDC6'],
        ['EPS8L2','SFTPA2'],
        ['EPS8L2','SFTPC'],
        ['EPS8L2','TRIP6'],
        ['ERBB2','FN1'],
        ['ERBB2','GRB7'],
        ['ERC1','FN1'],
        ['ERCC1','FN1'],
        ['ERCC2','FN1'],
        ['ERCC2','SERPINE1'],
        ['ERGIC1','FGB'],
        ['ERGIC1','FGG'],
        ['ERGIC1','FLNA'],
        ['ERGIC1','FN1'],
        ['ERGIC1','RTN4'],
        ['ERGIC1','SFTPC'],
        ['ERGIC1','SPARC'],
        ['ERICH2','ERICH2'],
        ['ERMN','GFAP'],
        ['ERRFI1','SFTPB'],
        ['ERRFI1','SFTPC'],
        ['ERV3-1','ZNF117'],
        ['ERVK13-1','PDPK1'],
        ['ERVV-1','ERVV-2'],
        ['ESAM','VSIG2'],
        ['ESRP2','RP11-96D1.11'],
        ['ESYT1','FN1'],
        ['ESYT1','SFTPC'],
        ['ETF1','LYZ'],
        ['ETFB','FN1'],
        ['ETS1','SFTPC'],
        ['ETS2','SFTPB'],
        ['EVL','SFTPC'],
        ['EWSR1','SFTPC'],
        ['EXOC3L4','RP11-736N17.8'],
        ['EXOC7','FN1'],
        ['EXOSC3','EXOSC3P1'],
        ['EXOSC3P1','RP11-613M10.9'],
        ['EXT1','FN1'],
        ['EXT1','SFTPC'],
        ['EXT1','TRIM28'],
        ['EXT2','PPIC'],
        ['EXTL3','FN1'],
        ['EYA1','RP11-1102P16.1'],
        ['EZR','MYH9'],
        ['EZR','SFTPA1'],
        ['EZR','SFTPB'],
        ['EZR','SFTPC'],
        ['F11R','HLA-E'],
        ['F11R','LRG1'],
        ['F11R','SFTPB'],
        ['FA2H','MLKL'],
        ['FAAH','NSUN4'],
        ['FAAHP1','LINC00505'],
        ['FAAP20','RP11-181G12.2'],
        ['FABP3','RP11-266K22.2'],
        ['FABP4','FAM213A'],
        ['FADS2','FEN1'],
        ['FADS2','FLNA'],
        ['FADS2','FN1'],
        ['FADS3','SLC1A5'],
        ['FAF2','FN1'],
        ['FAHD2B','FAHD2CP'],
        ['FAM101B','PTRF'],
        ['FAM101B','TRAM2'],
        ['FAM102A','IGH@'],
        ['FAM107A','SFTPA2'],
        ['FAM107A','SFTPC'],
        ['FAM107A','SLC34A2'],
        ['FAM107B','HBB'],
        ['FAM127A','FN1'],
        ['FAM129A','GCSAML'],
        ['FAM129A','SFTPB'],
        ['FAM129B','FKBP8'],
        ['FAM129B','FLNA'],
        ['FAM129B','FLNC'],
        ['FAM129B','FN1'],
        ['FAM129B','FSTL3'],
        ['FAM129B','LMNA'],
        ['FAM129B','MYH9'],
        ['FAM129B','MYL9'],
        ['FAM129B','PKM'],
        ['FAM129B','SFTPA2'],
        ['FAM129B','SFTPB'],
        ['FAM129B','SFTPC'],
        ['FAM129B','TG'],
        ['FAM129B','THBS1'],
        ['FAM131C','RP11-5P18.10'],
        ['FAM134A','FN1'],
        ['FAM134C','SFTPB'],
        ['FAM134C','TRAM2'],
        ['FAM135A','SSR3'],
        ['FAM136A','FAM136BP'],
        ['FAM13A-AS1','HERC3'],
        ['FAM157A','FAM157B'],
        ['FAM162B','ZUFSP'],
        ['FAM168B','FN1'],
        ['FAM168B','SFTPC'],
        ['FAM175A','HELQ'],
        ['FAM177B','SOD2'],
        ['FAM182B','MGC39584'],
        ['FAM193A','RNF4'],
        ['FAM198A','RP11-136C24.3'],
        ['FAM211A','PEBP1'],
        ['FAM211A-AS1','LGALS1'],
        ['FAM212B','PSMB2'],
        ['FAM219B','SFTPB'],
        ['FAM21A','FAM21C'],
        ['FAM220A','FAM220CP'],
        ['FAM35A','FAM35BP'],
        ['FAM3A','SFTPB'],
        ['FAM3D-AS1','SMR3B'],
        ['FAM46A','LYZ'],
        ['FAM50A','FN1'],
        ['FAM53B','METTL10'],
        ['FAM53B','RP11-12J10.3'],
        ['FAM58A','RP11-779O18.2'],
        ['FAM60A','FAM60BP'],
        ['FAM65A','FN1'],
        ['FAM65A','SFTPA1'],
        ['FAM65A','SFTPC'],
        ['FAM71D','SF3B4'],
        ['FAM89B','P2RX5-TAX1BP3'],
        ['FAM89B','TAX1BP3'],
        ['FAN1','RP11-540B6.6'],
        ['FANCE','PTRF'],
        ['FARP1','FN1'],
        ['FARP1','SFTPC'],
        ['FARSA','SYCE2'],
        ['FAS','RP11-399O19.8'],
        ['FASN','FN1'],
        ['FASN','HMGA2'],
        ['FASN','MGST1'],
        ['FASN','MYH9'],
        ['FASN','PLEC'],
        ['FASN','PLIN4'],
        ['FASN','SFTPA1'],
        ['FASN','SFTPA2'],
        ['FASN','SFTPB'],
        ['FASN','SFTPC'],
        ['FASN','TG'],
        ['FASN','VIM'],
        ['FAT1','FN1'],
        ['FAT1','GREM1'],
        ['FAT1','TIMP3'],
        ['FAU','FAUP1'],
        ['FBLIM1','NONO'],
        ['FBLIM1','SCUBE3'],
        ['FBLIM1','SFXN3'],
        ['FBLIM1','TAPBP'],
        ['FBLN1','FLNA'],
        ['FBLN1','FN1'],
        ['FBLN1','FSCN1'],
        ['FBLN1','IGKC'],
        ['FBLN1','LRP1'],
        ['FBLN1','MEG3'],
        ['FBLN1','SFTPA1'],
        ['FBLN1','SFTPA2'],
        ['FBLN1','SFTPB'],
        ['FBLN1','SFTPC'],
        ['FBLN1','TG'],
        ['FBLN2','FLNA'],
        ['FBLN2','FN1'],
        ['FBLN2','MBOAT7'],
        ['FBLN2','SFTPC'],
        ['FBLN2','TG'],
        ['FBLN5','FN1'],
        ['FBN1','FLNA'],
        ['FBN1','FN1'],
        ['FBN1','GSPT1'],
        ['FBN1','KIAA1199'],
        ['FBN1','PLEC'],
        ['FBN1','PPP1R14B'],
        ['FBN1','RAB6A'],
        ['FBN1','RTN4'],
        ['FBN1','SERPINE1'],
        ['FBN1','SPARC'],
        ['FBN1','TGFBI'],
        ['FBN1','THBS1'],
        ['FBN1','VIM'],
        ['FBN1','WDR1'],
        ['FBN2','FN1'],
        ['FBN2','PLEC'],
        ['FBN2','SERPINE1'],
        ['FBN2','THBS1'],
        ['FBP1','SFTPC'],
        ['FBRS','FN1'],
        ['FBRS','ZNF703'],
        ['FBXL19','ORAI3'],
        ['FBXL19','SFTPC'],
        ['FBXO10','TOMM5'],
        ['FBXO22','UBE2Q2'],
        ['FBXO44','FBXO6'],
        ['FBXW12','ZNF589'],
        ['FBXW5','FLNA'],
        ['FBXW5','FN1'],
        ['FBXW9','TNPO2'],
        ['FCGBP','PIGR'],
        ['FCGBP','TG'],
        ['FCGR2A','FCGR2C'],
        ['FCGR2B','FCGR2C'],
        ['FCGR3A','FCGR3B'],
        ['FCGRT','SFTPA2'],
        ['FCGRT','SFTPC'],
        ['FCN3','MAP3K6'],
        ['FCRL1','FCRL2'],
        ['FDFT1','FN1'],
        ['FEZ1','STT3A-AS1'],
        ['FGA','FGB'],
        ['FGA','FGG'],
        ['FGA','FGL1'],
        ['FGA','HP'],
        ['FGA','IGF2'],
        ['FGA','KTN1'],
        ['FGA','P4HB'],
        ['FGA','SAA1'],
        ['FGA','SAA2'],
        ['FGA','SERPINA1'],
        ['FGA','SERPINA3'],
        ['FGB','FGG'],
        ['FGB','GC'],
        ['FGB','HP'],
        ['FGB','ORM1'],
        ['FGB','PEBP1'],
        ['FGB','SERPINA1'],
        ['FGD1','PTMS'],
        ['FGD1','YWHAQ'],
        ['FGF14','FGF14-IT1'],
        ['FGF2','FN1'],
        ['FGF2','PTMS'],
        ['FGF5','FN1'],
        ['FGF7','RP11-262H14.11'],
        ['FGFRL1','FN1'],
        ['FGFRL1','PLEC'],
        ['FGFRL1','TRIM28'],
        ['FGG','GAA'],
        ['FGG','GC'],
        ['FGG','HP'],
        ['FGG','HPX'],
        ['FGG','LRG1'],
        ['FGG','PEBP1'],
        ['FGG','RBP4'],
        ['FGG','RP11-986E7.7'],
        ['FGG','SAA1'],
        ['FGG','SERPINA1'],
        ['FGG','SERPINA3'],
        ['FGG','TMBIM6'],
        ['FGL1','HP'],
        ['FGL1','SERPINA3'],
        ['FHL1','FN1'],
        ['FHL1','RP11-379K17.4'],
        ['FHL1','RP11-379K17.5'],
        ['FHL1','SFTPA2'],
        ['FHL1','SFTPC'],
        ['FHL1','TG'],
        ['FHL2','FN1'],
        ['FHL2','TTN'],
        ['FIG4','RP1-249I4.2'],
        ['FKBP10','FLNA'],
        ['FKBP10','FN1'],
        ['FKBP10','JUNB'],
        ['FKBP10','LRP1'],
        ['FKBP10','MYH9'],
        ['FKBP10','MYL9'],
        ['FKBP10','SPARC'],
        ['FKBP10','THBS1'],
        ['FKBP11','PTRF'],
        ['FKBP11','RP11-302B13.5'],
        ['FKBP15','HTN3'],
        ['FKBP15','SFTPA2'],
        ['FKBP1A','FKBP1C'],
        ['FKBP1A','SDCBP2'],
        ['FKBP5','SFTPB'],
        ['FKBP5','TG'],
        ['FKBP8','FN1'],
        ['FKBP8','MYH9'],
        ['FKBP8','PLEC'],
        ['FKBP8','SFTPA2'],
        ['FKBP8','SFTPC'],
        ['FKBP8','SPARC'],
        ['FKBP9','FKBP9L'],
        ['FKBP9','FKBP9P1'],
        ['FKBP9','HLA-E'],
        ['FKBP9','TGOLN2'],
        ['FLG2','KRT1'],
        ['FLG2','KRT10'],
        ['FLG2','TUFT1'],
        ['FLII','FN1'],
        ['FLJ00104','KLHDC4'],
        ['FLJ00388','RP11-543D5.2'],
        ['FLNA','FLNC'],
        ['FLNA','FN1'],
        ['FLNA','FSCN1'],
        ['FLNA','FSTL1'],
        ['FLNA','FTH1'],
        ['FLNA','FTL'],
        ['FLNA','FUS'],
        ['FLNA','G6PD'],
        ['FLNA','GANAB'],
        ['FLNA','GAPDH'],
        ['FLNA','GAS6'],
        ['FLNA','GCN1L1'],
        ['FLNA','GNAI2'],
        ['FLNA','GNAS'],
        ['FLNA','GOLGA2'],
        ['FLNA','GREM1'],
        ['FLNA','GRN'],
        ['FLNA','GSTP1'],
        ['FLNA','HDLBP'],
        ['FLNA','HLA-A'],
        ['FLNA','HLA-B'],
        ['FLNA','HLA-C'],
        ['FLNA','HLA-DPB1'],
        ['FLNA','HLA-E'],
        ['FLNA','HNRNPM'],
        ['FLNA','HNRNPU'],
        ['FLNA','HNRNPUL1'],
        ['FLNA','HNRNPUL2'],
        ['FLNA','HNRNPUL2-BSCL2'],
        ['FLNA','HPS6'],
        ['FLNA','HSP90B1'],
        ['FLNA','HSPB1'],
        ['FLNA','HSPG2'],
        ['FLNA','HYOU1'],
        ['FLNA','IGFBP3'],
        ['FLNA','IGFBP4'],
        ['FLNA','IGFBP5'],
        ['FLNA','IGH@'],
        ['FLNA','IGHM'],
        ['FLNA','IGK@'],
        ['FLNA','IGKC'],
        ['FLNA','IGKV1-8'],
        ['FLNA','IGKV3-11'],
        ['FLNA','ITGA11'],
        ['FLNA','ITGA3'],
        ['FLNA','ITGA5'],
        ['FLNA','ITGB2'],
        ['FLNA','KIAA1199'],
        ['FLNA','KIF1C'],
        ['FLNA','KLF2'],
        ['FLNA','KRT19'],
        ['FLNA','KRT7'],
        ['FLNA','LAMA4'],
        ['FLNA','LAMB2'],
        ['FLNA','LGALS1'],
        ['FLNA','LMF2'],
        ['FLNA','LMNA'],
        ['FLNA','LMNB2'],
        ['FLNA','LOX'],
        ['FLNA','LOXL2'],
        ['FLNA','LRP1'],
        ['FLNA','LTBP2'],
        ['FLNA','LTBP4'],
        ['FLNA','LZTS2'],
        ['FLNA','MAGED1'],
        ['FLNA','MAP4'],
        ['FLNA','MAP7D1'],
        ['FLNA','MAZ'],
        ['FLNA','MFGE8'],
        ['FLNA','MIF'],
        ['FLNA','MIR3654'],
        ['FLNA','MMP14'],
        ['FLNA','MMP2'],
        ['FLNA','MRC2'],
        ['FLNA','MTCH1'],
        ['FLNA','MYH11'],
        ['FLNA','MYH9'],
        ['FLNA','MYL6'],
        ['FLNA','MYL9'],
        ['FLNA','MYO1C'],
        ['FLNA','MYO9B'],
        ['FLNA','NCOR2'],
        ['FLNA','NEAT1'],
        ['FLNA','NFE2L1'],
        ['FLNA','NFIC'],
        ['FLNA','NME4'],
        ['FLNA','NT5DC2'],
        ['FLNA','NUCB1'],
        ['FLNA','P4HB'],
        ['FLNA','PCSK7'],
        ['FLNA','PEA15'],
        ['FLNA','PFN1'],
        ['FLNA','PHLDB1'],
        ['FLNA','PIEZO1'],
        ['FLNA','PIGS'],
        ['FLNA','PKM'],
        ['FLNA','PLEC'],
        ['FLNA','PLIN3'],
        ['FLNA','PLIN4'],
        ['FLNA','PLOD1'],
        ['FLNA','PLXNA3'],
        ['FLNA','PPIB'],
        ['FLNA','PPP2R1A'],
        ['FLNA','PRKCSH'],
        ['FLNA','PRRC2A'],
        ['FLNA','PRRC2B'],
        ['FLNA','PSMD2'],
        ['FLNA','PTMS'],
        ['FLNA','PTPN23'],
        ['FLNA','PTRF'],
        ['FLNA','PUF60'],
        ['FLNA','PXDN'],
        ['FLNA','QSOX1'],
        ['FLNA','RAB1B'],
        ['FLNA','RABL6'],
        ['FLNA','RANGAP1'],
        ['FLNA','RCN3'],
        ['FLNA','RHOB'],
        ['FLNA','RHOC'],
        ['FLNA','RP11-124N14.3'],
        ['FLNA','RP11-197N18.2'],
        ['FLNA','RP11-332H18.4'],
        ['FLNA','RP11-498C9.3'],
        ['FLNA','RP5-940J5.9'],
        ['FLNA','RP5-977B1.11'],
        ['FLNA','RPL28'],
        ['FLNA','RPL37A'],
        ['FLNA','RRBP1'],
        ['FLNA','RTN4'],
        ['FLNA','RUSC2'],
        ['FLNA','SCARF2'],
        ['FLNA','SCUBE3'],
        ['FLNA','SEC13'],
        ['FLNA','SEC61A1'],
        ['FLNA','SEPT9'],
        ['FLNA','SERPINE1'],
        ['FLNA','SERPINE2'],
        ['FLNA','SERPINH1'],
        ['FLNA','SFN'],
        ['FLNA','SFTPA1'],
        ['FLNA','SFTPA2'],
        ['FLNA','SFTPB'],
        ['FLNA','SFTPC'],
        ['FLNA','SH3BGRL3'],
        ['FLNA','SH3GLB2'],
        ['FLNA','SHC1'],
        ['FLNA','SLC16A3'],
        ['FLNA','SLC34A2'],
        ['FLNA','SLC7A1'],
        ['FLNA','SLC7A5'],
        ['FLNA','SMTN'],
        ['FLNA','SNRNP70'],
        ['FLNA','SPARC'],
        ['FLNA','SPARCL1'],
        ['FLNA','SPTAN1'],
        ['FLNA','SRRM2'],
        ['FLNA','TAGLN'],
        ['FLNA','TG'],
        ['FLNA','TGFBI'],
        ['FLNA','TGM2'],
        ['FLNA','THBS1'],
        ['FLNA','THY1'],
        ['FLNA','TIMP1'],
        ['FLNA','TIMP2'],
        ['FLNA','TIMP3'],
        ['FLNA','TLN1'],
        ['FLNA','TMEM203'],
        ['FLNA','TMEM259'],
        ['FLNA','TNC'],
        ['FLNA','TNRC18'],
        ['FLNA','TNXB'],
        ['FLNA','TPM1'],
        ['FLNA','TPM2'],
        ['FLNA','TPM4'],
        ['FLNA','TRNP1'],
        ['FLNA','TRPC4AP'],
        ['FLNA','TUBB'],
        ['FLNA','TUBB3'],
        ['FLNA','TUBB4B'],
        ['FLNA','TUBB6'],
        ['FLNA','TUFM'],
        ['FLNA','UBA1'],
        ['FLNA','UBC'],
        ['FLNA','UBR4'],
        ['FLNA','VIM'],
        ['FLNA','VIM-AS1'],
        ['FLNA','WDR1'],
        ['FLNA','WFS1'],
        ['FLNA','WNT5B'],
        ['FLNA','ZYX'],
        ['FLNB','FN1'],
        ['FLNB','SFTPC'],
        ['FLNB','TG'],
        ['FLNC','FN1'],
        ['FLNC','FSTL1'],
        ['FLNC','GAPDH'],
        ['FLNC','GNAS'],
        ['FLNC','GPX3'],
        ['FLNC','GREM1'],
        ['FLNC','H19'],
        ['FLNC','HSPG2'],
        ['FLNC','IGFBP3'],
        ['FLNC','IGFBP4'],
        ['FLNC','IGFBP5'],
        ['FLNC','JUN'],
        ['FLNC','LAMB2'],
        ['FLNC','LMNA'],
        ['FLNC','LRP1'],
        ['FLNC','MALAT1'],
        ['FLNC','MMP14'],
        ['FLNC','MYH9'],
        ['FLNC','NUPR1'],
        ['FLNC','P4HB'],
        ['FLNC','PIEZO1'],
        ['FLNC','PKM'],
        ['FLNC','PLEC'],
        ['FLNC','PTRF'],
        ['FLNC','PXDN'],
        ['FLNC','RCN3'],
        ['FLNC','RP5-977B1.11'],
        ['FLNC','SERPINE1'],
        ['FLNC','SPARC'],
        ['FLNC','SURF4'],
        ['FLNC','TAGLN'],
        ['FLNC','TG'],
        ['FLNC','TGFBI'],
        ['FLNC','THBS1'],
        ['FLNC','TIMP3'],
        ['FLNC','TLN1'],
        ['FLNC','TMBIM6'],
        ['FLNC','TPM2'],
        ['FLNC','TPO'],
        ['FLNC','TTN'],
        ['FLNC','VIM'],
        ['FLOT1','SFTPC'],
        ['FLOT2','SFTPA1'],
        ['FLOT2','SFTPC'],
        ['FLRT2','TRB@'],
        ['FLT1','IGF2'],
        ['FLT1','IGFBP4'],
        ['FLVCR1-AS1','SPATA45'],
        ['FMN2','FN1'],
        ['FMN2','GREM2'],
        ['FMNL3','SFTPA2'],
        ['FMO6P','NLK'],
        ['FN1','FOS'],
        ['FN1','FOSL1'],
        ['FN1','FOSL2'],
        ['FN1','FOXK1'],
        ['FN1','FOXM1'],
        ['FN1','FRMD6'],
        ['FN1','FSCN1'],
        ['FN1','FST'],
        ['FN1','FSTL1'],
        ['FN1','FTH1'],
        ['FN1','FTL'],
        ['FN1','FURIN'],
        ['FN1','FUS'],
        ['FN1','FXYD5'],
        ['FN1','FYCO1'],
        ['FN1','FZD2'],
        ['FN1','FZD7'],
        ['FN1','G6PD'],
        ['FN1','GALNT10'],
        ['FN1','GALNT2'],
        ['FN1','GANAB'],
        ['FN1','GAPDH'],
        ['FN1','GARS'],
        ['FN1','GAS2L1'],
        ['FN1','GAS6'],
        ['FN1','GCN1L1'],
        ['FN1','GDI1'],
        ['FN1','GFPT1'],
        ['FN1','GIPC1'],
        ['FN1','GLG1'],
        ['FN1','GLIS2'],
        ['FN1','GLUL'],
        ['FN1','GNA11'],
        ['FN1','GNAI2'],
        ['FN1','GNAS'],
        ['FN1','GNB1'],
        ['FN1','GNB2'],
        ['FN1','GNB2L1'],
        ['FN1','GOLGA3'],
        ['FN1','GPC1'],
        ['FN1','GPI'],
        ['FN1','GPR124'],
        ['FN1','GPR176'],
        ['FN1','GPRC5A'],
        ['FN1','GPSM1'],
        ['FN1','GREM1'],
        ['FN1','GREM2'],
        ['FN1','GRINA'],
        ['FN1','GRN'],
        ['FN1','GSE1'],
        ['FN1','GSN'],
        ['FN1','GSPT1'],
        ['FN1','GSTP1'],
        ['FN1','GTSE1'],
        ['FN1','GUCD1'],
        ['FN1','GUK1'],
        ['FN1','GYS1'],
        ['FN1','H1F0'],
        ['FN1','H1FX'],
        ['FN1','H2AFX'],
        ['FN1','HCFC1'],
        ['FN1','HDGF'],
        ['FN1','HDGFRP2'],
        ['FN1','HDLBP'],
        ['FN1','HEG1'],
        ['FN1','HGS'],
        ['FN1','HGSNAT'],
        ['FN1','HIF1A'],
        ['FN1','HK1'],
        ['FN1','HLA-A'],
        ['FN1','HLA-B'],
        ['FN1','HLA-DRA'],
        ['FN1','HLA-E'],
        ['FN1','HM13'],
        ['FN1','HMGA1'],
        ['FN1','HNRNPM'],
        ['FN1','HNRNPU'],
        ['FN1','HNRNPUL1'],
        ['FN1','HNRNPUL2'],
        ['FN1','HNRNPUL2-BSCL2'],
        ['FN1','HSP90B1'],
        ['FN1','HSPA5'],
        ['FN1','HSPA8'],
        ['FN1','HSPB1'],
        ['FN1','HSPB6'],
        ['FN1','HSPB7'],
        ['FN1','HSPG2'],
        ['FN1','HTRA1'],
        ['FN1','HUWE1'],
        ['FN1','HYOU1'],
        ['FN1','IARS'],
        ['FN1','ICAM1'],
        ['FN1','ID3'],
        ['FN1','IGF2'],
        ['FN1','IGFBP3'],
        ['FN1','IGFBP4'],
        ['FN1','IGFBP5'],
        ['FN1','IGFBP6'],
        ['FN1','IGH@'],
        ['FN1','IGHA1'],
        ['FN1','IGKC'],
        ['FN1','IL33'],
        ['FN1','ILF3'],
        ['FN1','ILK'],
        ['FN1','INHBA'],
        ['FN1','INPPL1'],
        ['FN1','INTS1'],
        ['FN1','IPO5'],
        ['FN1','IQGAP1'],
        ['FN1','ISLR'],
        ['FN1','ITGA11'],
        ['FN1','ITGA3'],
        ['FN1','ITGA5'],
        ['FN1','ITGB1'],
        ['FN1','ITGB1BP1'],
        ['FN1','ITGB5'],
        ['FN1','ITGBL1'],
        ['FN1','ITLN1'],
        ['FN1','ITM2C'],
        ['FN1','JMJD8'],
        ['FN1','JUND'],
        ['FN1','KCTD10'],
        ['FN1','KDELR1'],
        ['FN1','KDELR2'],
        ['FN1','KDM6B'],
        ['FN1','KHSRP'],
        ['FN1','KIAA0100'],
        ['FN1','KIAA0930'],
        ['FN1','KIAA1199'],
        ['FN1','KIAA1462'],
        ['FN1','KIF13A'],
        ['FN1','KIF1C'],
        ['FN1','KIF7'],
        ['FN1','KIRREL'],
        ['FN1','KLC2'],
        ['FN1','KMT2D'],
        ['FN1','KRT19'],
        ['FN1','LAMA4'],
        ['FN1','LAMB1'],
        ['FN1','LAMB2'],
        ['FN1','LAMP1'],
        ['FN1','LAPTM5'],
        ['FN1','LASP1'],
        ['FN1','LDHA'],
        ['FN1','LDLRAD2'],
        ['FN1','LEPRE1'],
        ['FN1','LEPREL2'],
        ['FN1','LGALS1'],
        ['FN1','LGALS3'],
        ['FN1','LIMA1'],
        ['FN1','LMF2'],
        ['FN1','LMNA'],
        ['FN1','LMNB2'],
        ['FN1','LMO7'],
        ['FN1','LMOD1'],
        ['FN1','LOX'],
        ['FN1','LOXL1'],
        ['FN1','LOXL2'],
        ['FN1','LRP1'],
        ['FN1','LRP10'],
        ['FN1','LRP5'],
        ['FN1','LRRC15'],
        ['FN1','LRRC41'],
        ['FN1','LRRC59'],
        ['FN1','LTBP1'],
        ['FN1','LTBP2'],
        ['FN1','LTBP4'],
        ['FN1','LUM'],
        ['FN1','LY6E'],
        ['FN1','LZTR1'],
        ['FN1','MACF1'],
        ['FN1','MAFK'],
        ['FN1','MAGED1'],
        ['FN1','MALAT1'],
        ['FN1','MALL'],
        ['FN1','MAN1B1'],
        ['FN1','MAP1A'],
        ['FN1','MAP1B'],
        ['FN1','MAP2K2'],
        ['FN1','MAP2K3'],
        ['FN1','MAP4'],
        ['FN1','MAP7D1'],
        ['FN1','MAPK3'],
        ['FN1','MAPK8IP3'],
        ['FN1','MAPKAP1'],
        ['FN1','MAPKAPK2'],
        ['FN1','MARCKS'],
        ['FN1','MARVELD1'],
        ['FN1','MAZ'],
        ['FN1','MBTPS1'],
        ['FN1','MCL1'],
        ['FN1','MCM3AP'],
        ['FN1','MDH2'],
        ['FN1','MDK'],
        ['FN1','MED15'],
        ['FN1','MED16'],
        ['FN1','MEF2D'],
        ['FN1','MEG3'],
        ['FN1','MEGF8'],
        ['FN1','MESDC2'],
        ['FN1','MFGE8'],
        ['FN1','MGAT1'],
        ['FN1','MGAT4B'],
        ['FN1','MGLL'],
        ['FN1','MICAL2'],
        ['FN1','MICALL1'],
        ['FN1','MIER2'],
        ['FN1','MIF'],
        ['FN1','MINOS1-NBL1'],
        ['FN1','MIR3654'],
        ['FN1','MKI67'],
        ['FN1','MLEC'],
        ['FN1','MLF2'],
        ['FN1','MMP1'],
        ['FN1','MMP14'],
        ['FN1','MMP2'],
        ['FN1','MMP24'],
        ['FN1','MMP3'],
        ['FN1','MPRIP'],
        ['FN1','MPST'],
        ['FN1','MRC2'],
        ['FN1','MSN'],
        ['FN1','MT2A'],
        ['FN1','MTSS1L'],
        ['FN1','MUC1'],
        ['FN1','MVP'],
        ['FN1','MXRA8'],
        ['FN1','MYADM'],
        ['FN1','MYH9'],
        ['FN1','MYL12A'],
        ['FN1','MYL6'],
        ['FN1','MYL9'],
        ['FN1','MYLK'],
        ['FN1','MYO10'],
        ['FN1','MYO1C'],
        ['FN1','MYO1D'],
        ['FN1','MYO9B'],
        ['FN1','MYOF'],
        ['FN1','NACC1'],
        ['FN1','NAV1'],
        ['FN1','NAV2'],
        ['FN1','NBL1'],
        ['FN1','NCL'],
        ['FN1','NCLN'],
        ['FN1','NCOR1'],
        ['FN1','NCOR2'],
        ['FN1','NCS1'],
        ['FN1','NCSTN'],
        ['FN1','NDRG1'],
        ['FN1','NDST1'],
        ['FN1','NDUFV1'],
        ['FN1','NEAT1'],
        ['FN1','NFASC'],
        ['FN1','NFE2L1'],
        ['FN1','NFIC'],
        ['FN1','NFIX'],
        ['FN1','NID1'],
        ['FN1','NID2'],
        ['FN1','NLGN2'],
        ['FN1','NLRP1'],
        ['FN1','NME4'],
        ['FN1','NOL12'],
        ['FN1','NOL6'],
        ['FN1','NOTCH2'],
        ['FN1','NOTCH3'],
        ['FN1','NPTN'],
        ['FN1','NR4A1'],
        ['FN1','NRBP1'],
        ['FN1','NRP1'],
        ['FN1','NRSN2'],
        ['FN1','NT5DC2'],
        ['FN1','NT5E'],
        ['FN1','NUCB1'],
        ['FN1','NUCKS1'],
        ['FN1','NUFIP2'],
        ['FN1','NUMA1'],
        ['FN1','NUP188'],
        ['FN1','NXN'],
        ['FN1','OAF'],
        ['FN1','OAZ1'],
        ['FN1','ODC1'],
        ['FN1','OLFML3'],
        ['FN1','ORAOV1'],
        ['FN1','OS9'],
        ['FN1','OXTR'],
        ['FN1','P2RX5-TAX1BP3'],
        ['FN1','P4HA2'],
        ['FN1','P4HB'],
        ['FN1','PABPC4'],
        ['FN1','PAF1'],
        ['FN1','PALLD'],
        ['FN1','PALM2-AKAP2'],
        ['FN1','PAM'],
        ['FN1','PCBP1'],
        ['FN1','PCDH18'],
        ['FN1','PCDHGA1'],
        ['FN1','PCDHGA10'],
        ['FN1','PCDHGA11'],
        ['FN1','PCDHGA12'],
        ['FN1','PCDHGA2'],
        ['FN1','PCDHGA3'],
        ['FN1','PCDHGA4'],
        ['FN1','PCDHGA5'],
        ['FN1','PCDHGA6'],
        ['FN1','PCDHGA7'],
        ['FN1','PCDHGA8'],
        ['FN1','PCDHGA9'],
        ['FN1','PCDHGB1'],
        ['FN1','PCDHGB2'],
        ['FN1','PCDHGB3'],
        ['FN1','PCDHGB4'],
        ['FN1','PCDHGB6'],
        ['FN1','PCDHGB7'],
        ['FN1','PCDHGC3'],
        ['FN1','PCDHGC4'],
        ['FN1','PCDHGC5'],
        ['FN1','PCOLCE'],
        ['FN1','PCOLCE-AS1'],
        ['FN1','PCSK7'],
        ['FN1','PDGFRA'],
        ['FN1','PDGFRB'],
        ['FN1','PDIA3'],
        ['FN1','PDIA4'],
        ['FN1','PDIA6'],
        ['FN1','PDLIM4'],
        ['FN1','PDLIM7'],
        ['FN1','PEA15'],
        ['FN1','PEAR1'],
        ['FN1','PER1'],
        ['FN1','PFKL'],
        ['FN1','PFKP'],
        ['FN1','PFN1'],
        ['FN1','PFN2'],
        ['FN1','PHGDH'],
        ['FN1','PHLDB1'],
        ['FN1','PIEZO1'],
        ['FN1','PIN1'],
        ['FN1','PIP4K2B'],
        ['FN1','PIP5K1C'],
        ['FN1','PKD1'],
        ['FN1','PKM'],
        ['FN1','PKN1'],
        ['FN1','PLAUR'],
        ['FN1','PLCG1'],
        ['FN1','PLD3'],
        ['FN1','PLEC'],
        ['FN1','PLIN3'],
        ['FN1','PLK1'],
        ['FN1','PLOD1'],
        ['FN1','PLOD3'],
        ['FN1','PLXNA1'],
        ['FN1','PLXNA3'],
        ['FN1','PLXNB2'],
        ['FN1','PLXND1'],
        ['FN1','PNPLA6'],
        ['FN1','PODXL'],
        ['FN1','POLR1A'],
        ['FN1','POLR2A'],
        ['FN1','POLR2C'],
        ['FN1','POLR2E'],
        ['FN1','POSTN'],
        ['FN1','PPIB'],
        ['FN1','PPIF'],
        ['FN1','PPM1G'],
        ['FN1','PPP1CA'],
        ['FN1','PPP1R12C'],
        ['FN1','PPP1R14B'],
        ['FN1','PPP1R15A'],
        ['FN1','PPP1R18'],
        ['FN1','PPP1R9B'],
        ['FN1','PPP2CA'],
        ['FN1','PPP2R1A'],
        ['FN1','PPP2R4'],
        ['FN1','PPP2R5D'],
        ['FN1','PRDX6'],
        ['FN1','PREX1'],
        ['FN1','PRKCDBP'],
        ['FN1','PRKCSH'],
        ['FN1','PRKDC'],
        ['FN1','PRNP'],
        ['FN1','PRPF8'],
        ['FN1','PRRC2A'],
        ['FN1','PRRC2C'],
        ['FN1','PRRX1'],
        ['FN1','PRRX2'],
        ['FN1','PSAP'],
        ['FN1','PSMB4'],
        ['FN1','PSMD2'],
        ['FN1','PSMD3'],
        ['FN1','PSMD8'],
        ['FN1','PTBP1'],
        ['FN1','PTGS2'],
        ['FN1','PTK7'],
        ['FN1','PTMA'],
        ['FN1','PTMS'],
        ['FN1','PTPN14'],
        ['FN1','PTPN23'],
        ['FN1','PTPRB'],
        ['FN1','PTPRF'],
        ['FN1','PTPRS'],
        ['FN1','PTPRU'],
        ['FN1','PTRF'],
        ['FN1','PTX3'],
        ['FN1','PVR'],
        ['FN1','PVRL2'],
        ['FN1','PXDN'],
        ['FN1','PXN'],
        ['FN1','PYCR1'],
        ['FN1','PYGB'],
        ['FN1','QSOX1'],
        ['FN1','RAB11FIP1'],
        ['FN1','RAB11FIP5'],
        ['FN1','RAB1B'],
        ['FN1','RAB31'],
        ['FN1','RAB34'],
        ['FN1','RAB3B'],
        ['FN1','RAB5C'],
        ['FN1','RAB7A'],
        ['FN1','RABAC1'],
        ['FN1','RABL6'],
        ['FN1','RAF1'],
        ['FN1','RALY'],
        ['FN1','RANGAP1'],
        ['FN1','RBM15B'],
        ['FN1','RBM25'],
        ['FN1','RCC2'],
        ['FN1','RCN3'],
        ['FN1','RERE'],
        ['FN1','RHOA'],
        ['FN1','RHOB'],
        ['FN1','RHOC'],
        ['FN1','RNASE1'],
        ['FN1','RNF10'],
        ['FN1','RNF40'],
        ['FN1','RNH1'],
        ['FN1','ROBO4'],
        ['FN1','RP1-56K13.3'],
        ['FN1','RP11-124N14.3'],
        ['FN1','RP11-186B7.4'],
        ['FN1','RP11-197N18.2'],
        ['FN1','RP11-20I23.1'],
        ['FN1','RP11-244F12.2'],
        ['FN1','RP11-290D2.6'],
        ['FN1','RP11-323N12.5'],
        ['FN1','RP11-386G11.10'],
        ['FN1','RP11-426L16.10'],
        ['FN1','RP11-468E2.2'],
        ['FN1','RP11-498C9.2'],
        ['FN1','RP11-498C9.3'],
        ['FN1','RP5-940J5.9'],
        ['FN1','RP5-977B1.11'],
        ['FN1','RPL13'],
        ['FN1','RPL18'],
        ['FN1','RPL27'],
        ['FN1','RPL28'],
        ['FN1','RPL3'],
        ['FN1','RPL30'],
        ['FN1','RPL36'],
        ['FN1','RPL37A'],
        ['FN1','RPL8'],
        ['FN1','RPLP1'],
        ['FN1','RPLP2'],
        ['FN1','RPN1'],
        ['FN1','RPN2'],
        ['FN1','RPS11'],
        ['FN1','RPS19'],
        ['FN1','RPS4X'],
        ['FN1','RPS5'],
        ['FN1','RPS6KA2'],
        ['FN1','RRBP1'],
        ['FN1','RTN4'],
        ['FN1','RUSC2'],
        ['FN1','S100A11'],
        ['FN1','S100A16'],
        ['FN1','S100A4'],
        ['FN1','S100A6'],
        ['FN1','SAE1'],
        ['FN1','SAMD11'],
        ['FN1','SAR1A'],
        ['FN1','SBF1'],
        ['FN1','SBNO2'],
        ['FN1','SCAF1'],
        ['FN1','SCAMP3'],
        ['FN1','SCAMP4'],
        ['FN1','SCARA3'],
        ['FN1','SCARB2'],
        ['FN1','SCARF2'],
        ['FN1','SCUBE3'],
        ['FN1','SCYL1'],
        ['FN1','SDC2'],
        ['FN1','SDC3'],
        ['FN1','SDF4'],
        ['FN1','SDPR'],
        ['FN1','SEC16A'],
        ['FN1','SEC24C'],
        ['FN1','SEC24D'],
        ['FN1','SEC61A1'],
        ['FN1','SEC62'],
        ['FN1','SELENBP1'],
        ['FN1','SEMA7A'],
        ['FN1','SEPN1'],
        ['FN1','SEPT11'],
        ['FN1','SEPT9'],
        ['FN1','SERF2'],
        ['FN1','SERINC1'],
        ['FN1','SERINC2'],
        ['FN1','SERP1'],
        ['FN1','SERPINA1'],
        ['FN1','SERPINB6'],
        ['FN1','SERPINE1'],
        ['FN1','SERPINE2'],
        ['FN1','SERPINH1'],
        ['FN1','SETD3'],
        ['FN1','SF3A1'],
        ['FN1','SF3B2'],
        ['FN1','SF3B3'],
        ['FN1','SFRP1'],
        ['FN1','SFRP2'],
        ['FN1','SFTPA1'],
        ['FN1','SFTPA2'],
        ['FN1','SFTPB'],
        ['FN1','SFTPC'],
        ['FN1','SH3BGRL3'],
        ['FN1','SH3BP2'],
        ['FN1','SH3BP4'],
        ['FN1','SH3GL1'],
        ['FN1','SHC1'],
        ['FN1','SHISA5'],
        ['FN1','SHMT2'],
        ['FN1','SHROOM3'],
        ['FN1','SIGMAR1'],
        ['FN1','SIK1'],
        ['FN1','SIPA1L3'],
        ['FN1','SIRPA'],
        ['FN1','SLC16A3'],
        ['FN1','SLC1A5'],
        ['FN1','SLC20A1'],
        ['FN1','SLC25A1'],
        ['FN1','SLC25A3'],
        ['FN1','SLC25A6'],
        ['FN1','SLC34A2'],
        ['FN1','SLC35A4'],
        ['FN1','SLC35E1'],
        ['FN1','SLC38A10'],
        ['FN1','SLC38A2'],
        ['FN1','SLC39A1'],
        ['FN1','SLC39A13'],
        ['FN1','SLC39A14'],
        ['FN1','SLC39A7'],
        ['FN1','SLC3A2'],
        ['FN1','SLC44A2'],
        ['FN1','SLC4A2'],
        ['FN1','SLC7A1'],
        ['FN1','SLC7A5'],
        ['FN1','SLIT2'],
        ['FN1','SMAD3'],
        ['FN1','SMARCA4'],
        ['FN1','SMARCD2'],
        ['FN1','SMTN'],
        ['FN1','SND1'],
        ['FN1','SNORA67'],
        ['FN1','SNRNP200'],
        ['FN1','SNRNP70'],
        ['FN1','SNX22'],
        ['FN1','SNX32'],
        ['FN1','SNX9'],
        ['FN1','SON'],
        ['FN1','SORBS3'],
        ['FN1','SPARC'],
        ['FN1','SPARCL1'],
        ['FN1','SPEG'],
        ['FN1','SPEN'],
        ['FN1','SPOCD1'],
        ['FN1','SPOCK1'],
        ['FN1','SPOCK2'],
        ['FN1','SPTAN1'],
        ['FN1','SPTBN1'],
        ['FN1','SQSTM1'],
        ['FN1','SRCAP'],
        ['FN1','SREBF2'],
        ['FN1','SRM'],
        ['FN1','SRPR'],
        ['FN1','SRRM2'],
        ['FN1','SSR4'],
        ['FN1','ST5'],
        ['FN1','STC1'],
        ['FN1','STC2'],
        ['FN1','STEAP3'],
        ['FN1','STK25'],
        ['FN1','STUB1'],
        ['FN1','SULF1'],
        ['FN1','SUN1'],
        ['FN1','SUPT6H'],
        ['FN1','SURF4'],
        ['FN1','SUSD2'],
        ['FN1','SYDE1'],
        ['FN1','SYNPO'],
        ['FN1','SZRD1'],
        ['FN1','TAGLN'],
        ['FN1','TAGLN2'],
        ['FN1','TAOK2'],
        ['FN1','TAX1BP3'],
        ['FN1','TBC1D9B'],
        ['FN1','TCIRG1'],
        ['FN1','TENC1'],
        ['FN1','TENM2'],
        ['FN1','TENM3'],
        ['FN1','TFPI2'],
        ['FN1','TGFB1'],
        ['FN1','TGFB1I1'],
        ['FN1','TGFBI'],
        ['FN1','TGFBR2'],
        ['FN1','TGM2'],
        ['FN1','THBS1'],
        ['FN1','THBS2'],
        ['FN1','THOP1'],
        ['FN1','THY1'],
        ['FN1','TIMP1'],
        ['FN1','TIMP2'],
        ['FN1','TIMP3'],
        ['FN1','TKT'],
        ['FN1','TLE3'],
        ['FN1','TLN1'],
        ['FN1','TM4SF1'],
        ['FN1','TMBIM1'],
        ['FN1','TMBIM6'],
        ['FN1','TMED3'],
        ['FN1','TMED9'],
        ['FN1','TMEM119'],
        ['FN1','TMEM214'],
        ['FN1','TMEM259'],
        ['FN1','TMEM262'],
        ['FN1','TMEM30A'],
        ['FN1','TMEM66'],
        ['FN1','TMSB10'],
        ['FN1','TMUB1'],
        ['FN1','TNC'],
        ['FN1','TNFRSF10D'],
        ['FN1','TNFRSF12A'],
        ['FN1','TNKS1BP1'],
        ['FN1','TNRC18'],
        ['FN1','TNS1'],
        ['FN1','TPD52L2'],
        ['FN1','TPI1'],
        ['FN1','TPM1'],
        ['FN1','TPM2'],
        ['FN1','TPM4'],
        ['FN1','TPRG1L'],
        ['FN1','TPT1'],
        ['FN1','TRAM2'],
        ['FN1','TRIM28'],
        ['FN1','TRIO'],
        ['FN1','TRIOBP'],
        ['FN1','TRNP1'],
        ['FN1','TSC22D3'],
        ['FN1','TSPAN4'],
        ['FN1','TSPO'],
        ['FN1','TTYH3'],
        ['FN1','TUBA1B'],
        ['FN1','TUBA1C'],
        ['FN1','TUBB'],
        ['FN1','TUBB3'],
        ['FN1','TUBB4B'],
        ['FN1','TUBB6'],
        ['FN1','TUFM'],
        ['FN1','TXNDC5'],
        ['FN1','TXNIP'],
        ['FN1','TXNRD1'],
        ['FN1','U2AF2'],
        ['FN1','UBA1'],
        ['FN1','UBA52'],
        ['FN1','UBAP2L'],
        ['FN1','UBB'],
        ['FN1','UBC'],
        ['FN1','UBE2Z'],
        ['FN1','UBR4'],
        ['FN1','UBTD1'],
        ['FN1','UBTF'],
        ['FN1','UBXN11'],
        ['FN1','UCHL1'],
        ['FN1','UNC45A'],
        ['FN1','UPF1'],
        ['FN1','USP22'],
        ['FN1','VAMP2'],
        ['FN1','VARS'],
        ['FN1','VASN'],
        ['FN1','VASP'],
        ['FN1','VAT1'],
        ['FN1','VCAN'],
        ['FN1','VCL'],
        ['FN1','VCP'],
        ['FN1','VEGFB'],
        ['FN1','VIM'],
        ['FN1','VIM-AS1'],
        ['FN1','VPS37B'],
        ['FN1','VPS4A'],
        ['FN1','VWF'],
        ['FN1','WASF2'],
        ['FN1','WDR1'],
        ['FN1','WFS1'],
        ['FN1','XRCC5'],
        ['FN1','XXBAC-BPG252P9.9'],
        ['FN1','YARS'],
        ['FN1','YBX1'],
        ['FN1','YBX3'],
        ['FN1','YIPF2'],
        ['FN1','YIPF3'],
        ['FN1','YKT6'],
        ['FN1','ZBTB4'],
        ['FN1','ZDHHC5'],
        ['FN1','ZFP36'],
        ['FN1','ZFP36L1'],
        ['FN1','ZMIZ1'],
        ['FN1','ZMIZ2'],
        ['FN1','ZNF282'],
        ['FN1','ZNF395'],
        ['FN1','ZYX'],
        ['FNBP4','Y_RNA'],
        ['FOLH1','FOLH1B'],
        ['FOLR1','SFTPA1'],
        ['FOLR1','SFTPC'],
        ['FOLR2','FOLR3'],
        ['FOS','FOSB'],
        ['FOS','HLA-B'],
        ['FOS','PTRF'],
        ['FOS','SFTPA1'],
        ['FOS','SFTPA2'],
        ['FOS','SFTPB'],
        ['FOS','SFTPC'],
        ['FOS','SLC34A2'],
        ['FOS','TG'],
        ['FOS','TIMP3'],
        ['FOSB','HLA-B'],
        ['FOSB','IGKC'],
        ['FOSB','RNASE1'],
        ['FOSB','SFTPA1'],
        ['FOSB','SFTPA2'],
        ['FOSB','SFTPB'],
        ['FOSB','SFTPC'],
        ['FOSB','SRRM2'],
        ['FOSL2','NUMA1'],
        ['FOSL2','SFTPA1'],
        ['FOSL2','SFTPA2'],
        ['FOSL2','SFTPB'],
        ['FOSL2','SFTPC'],
        ['FOXC2','IGFBP7'],
        ['FOXC2','PLEC'],
        ['FOXD1','GNAS'],
        ['FOXD1','PTP4A2'],
        ['FOXD1','SLC7A5'],
        ['FOXD1','TXNDC5'],
        ['FOXG1','LINC01551'],
        ['FOXJ2','SFTPA2'],
        ['FOXJ3','RP11-223A3.1'],
        ['FOXK1','MARCKS'],
        ['FOXK2','IGFBP3'],
        ['FOXK2','SRM'],
        ['FPGT','FPGT-TNNI3K'],
        ['FPGT','TNNI3K'],
        ['FPGT-TNNI3K','RP4-650F12.2'],
        ['FPGT-TNNI3K','TNNI3K'],
        ['FRAS1','TG'],
        ['FRG1','FRG1B'],
        ['FRG1','FRG1BP'],
        ['FRG1','RP11-87H9.2'],
        ['FRG1B','RP11-87H9.2'],
        ['FRMD4A','SERPINE1'],
        ['FSCN1','PFKL'],
        ['FSCN1','PHLDA2'],
        ['FSCN1','PTRF'],
        ['FSCN1','THBS1'],
        ['FSTL1','SFTPC'],
        ['FSTL1','SPARC'],
        ['FSTL1','THBS1'],
        ['FSTL1','VIM'],
        ['FSTL3','MPDU1'],
        ['FTH1','FTH1P1'],
        ['FTH1','FTH1P10'],
        ['FTH1','FTH1P11'],
        ['FTH1','FTH1P12'],
        ['FTH1','FTH1P15'],
        ['FTH1','FTH1P16'],
        ['FTH1','FTH1P2'],
        ['FTH1','FTH1P20'],
        ['FTH1','FTH1P23'],
        ['FTH1','FTH1P3'],
        ['FTH1','FTH1P4'],
        ['FTH1','FTH1P5'],
        ['FTH1','FTH1P7'],
        ['FTH1','FTH1P8'],
        ['FTH1','FUS'],
        ['FTH1','GDI1'],
        ['FTH1','GNAS'],
        ['FTH1','GNB1'],
        ['FTH1','GPX3'],
        ['FTH1','HNRNPA0'],
        ['FTH1','HNRNPL'],
        ['FTH1','KRT10'],
        ['FTH1','MKNK2'],
        ['FTH1','MTCH1'],
        ['FTH1','NANS'],
        ['FTH1','PCBP1'],
        ['FTH1','PEA15'],
        ['FTH1','PLEKHM2'],
        ['FTH1','PRELID1'],
        ['FTH1','PTMS'],
        ['FTH1','RALY'],
        ['FTH1','SFTPA2'],
        ['FTH1','SFTPC'],
        ['FTH1','TXNDC5'],
        ['FTH1','UBE2D2'],
        ['FTH1','ZMYND19'],
        ['FTL','FTLP2'],
        ['FTL','FTLP3'],
        ['FTL','GSTP1'],
        ['FTL','HDAC5'],
        ['FTL','HLA-B'],
        ['FTL','IGKC'],
        ['FTL','JUND'],
        ['FTL','MYH11'],
        ['FTL','SFTPA1'],
        ['FTL','SFTPA2'],
        ['FTL','SFTPB'],
        ['FTL','SFTPC'],
        ['FTL','SLC34A2'],
        ['FTL','SPTBN1'],
        ['FTL','SRRM2'],
        ['FTL','TG'],
        ['FTL','THBS1'],
        ['FTL','TIMP1'],
        ['FTL','ZFP36'],
        ['FTO','FTO-IT1'],
        ['FTX','RP3-368A4.5'],
        ['FTX_2','RAB11FIP1P1'],
        ['FURIN','SFTPC'],
        ['FUS','KRT19'],
        ['FUS','LGMN'],
        ['FUS','MYH9'],
        ['FUS','NCL'],
        ['FUS','PITPNM1'],
        ['FUS','PTMS'],
        ['FUS','SFTPC'],
        ['FUS','TMEM158'],
        ['FUS','VIM-AS1'],
        ['FXN','TJP2'],
        ['FXYD2','FXYD6'],
        ['FXYD2','FXYD6-FXYD2'],
        ['FXYD5','SPARC'],
        ['FXYD6','FXYD6-FXYD2'],
        ['FZD4','RP11-736K20.5'],
        ['FZD4','SFTPB'],
        ['FZD8','IGFBP4'],
        ['FZD8','SERPINE1'],
        ['G0S2','MPO'],
        ['G6PD','SFTPC'],
        ['GAA','HP'],
        ['GAA','SERPINA1'],
        ['GAA','SMR3B'],
        ['GAB2','SFTPB'],
        ['GAB2','SFTPC'],
        ['GABARAP','RP11-6B6.3'],
        ['GADD45B','SFTPC'],
        ['GAK','SFTPB'],
        ['GAK','SFTPC'],
        ['GALNT10','SAMD1'],
        ['GALNT2','IGFBP4'],
        ['GALNT2','LAMP1'],
        ['GALNT6','MRPL49'],
        ['GALT','GALT'],
        ['GALT','IL11RA'],
        ['GALT','RP11-195F19.29'],
        ['GANAB','SFTPA2'],
        ['GANAB','SFTPC'],
        ['GANAB','TGFBI'],
        ['GAPDH','GAPDHP1'],
        ['GAPDH','GAPDHP21'],
        ['GAPDH','GAPDHP38'],
        ['GAPDH','GAPDHP40'],
        ['GAPDH','GAPDHP44'],
        ['GAPDH','GAPDHP60'],
        ['GAPDH','GAPDHP61'],
        ['GAPDH','GAPDHP63'],
        ['GAPDH','GAPDHP65'],
        ['GAPDH','GAPDHP68'],
        ['GAPDH','GAPDHP70'],
        ['GAPDH','GAPDHP72'],
        ['GAPDH','IGFBP3'],
        ['GAPDH','IGH@'],
        ['GAPDH','LMNA'],
        ['GAPDH','LOX'],
        ['GAPDH','MAP1A'],
        ['GAPDH','MCAM'],
        ['GAPDH','MYH9'],
        ['GAPDH','PCSK7'],
        ['GAPDH','PLEC'],
        ['GAPDH','RP11-159J3.1'],
        ['GAPDH','RP11-321C24.4'],
        ['GAPDH','RP11-378G13.2'],
        ['GAPDH','RP11-386G11.10'],
        ['GAPDH','SCD'],
        ['GAPDH','SEPT9'],
        ['GAPDH','SERPINE1'],
        ['GAPDH','SFTPA1'],
        ['GAPDH','SFTPC'],
        ['GAPDH','SNX22'],
        ['GAPDH','TG'],
        ['GAPDH','TGFBI'],
        ['GAPDH','THBS1'],
        ['GAPDH','TIMP3'],
        ['GAPDH','TNC'],
        ['GAPDH','VIM'],
        ['GAS5','SNORD74'],
        ['GAS5','SNORD78'],
        ['GAS6','PITPNA'],
        ['GAS6','PTMS'],
        ['GAS6','RP11-486O12.2'],
        ['GAS6','SFTPC'],
        ['GAS6','SLAIN2'],
        ['GAS6','TCFL5'],
        ['GAS6','THBS1'],
        ['GAS6','TRIM28'],
        ['GATC','PTRF'],
        ['GATS','PMS2P1'],
        ['GBP1','GBP1P1'],
        ['GBP3','Y_RNA'],
        ['GBP4','GBP7'],
        ['GC','SERPINA1'],
        ['GCN1L1','SFTPC'],
        ['GCNT4','LINC01336'],
        ['GCNT4','RP11-229C3.2'],
        ['GDI1','GFAP'],
        ['GDI1','SFTPA1'],
        ['GDI1','SFTPC'],
        ['GFAP','HK1'],
        ['GFAP','MBP'],
        ['GFAP','PHYHIP'],
        ['GFAP','RP11-386G11.10'],
        ['GFAP','SCD'],
        ['GFAP','SPARC'],
        ['GFAP','SPARCL1'],
        ['GFAP','SYP'],
        ['GFAP','TUBB4A'],
        ['GGA2','SFTPA2'],
        ['GGA2','ZFAND5'],
        ['GGT1','RP11-341D18.4'],
        ['GHITM','TG'],
        ['GHRLOS','LINC00852'],
        ['GIMAP1','GIMAP2'],
        ['GIMAP1-GIMAP5','GIMAP2'],
        ['GIMAP2','GIMAP5'],
        ['GIMAP5','RP5-1051J4.6'],
        ['GIPC1','PTMS'],
        ['GIPC1','RAB32'],
        ['GJA1','GJA1P1'],
        ['GJA9','MYCBP'],
        ['GJA9-MYCBP','MYCBP'],
        ['GJA9-MYCBP','RHBDL2'],
        ['GLB1','TMPPE'],
        ['GLG1','IGFBP5'],
        ['GLG1','SFTPB'],
        ['GLG1','SFTPC'],
        ['GLG1','TG'],
        ['GLG1','ZMIZ1'],
        ['GLI4','ZFP41'],
        ['GLIPR1','GLIPR1L2'],
        ['GLIPR2','SFTPA2'],
        ['GLOD4','NXN'],
        ['GLRX','GLRXP3'],
        ['GLRX3','PTRF'],
        ['GLT25D1','SFTPB'],
        ['GLT25D1','SFTPC'],
        ['GLTP','U52111.14'],
        ['GLTSCR2','IGK@'],
        ['GLTSCR2','PRSS1'],
        ['GLTSCR2','RP11-255M2.3'],
        ['GLTSCR2','SFTPC'],
        ['GLUD1','GLUD2'],
        ['GLUL','GLULP3'],
        ['GLUL','GLULP4'],
        ['GLUL','GPX3'],
        ['GLUL','GRN'],
        ['GLUL','PSAP'],
        ['GLUL','RP11-1H15.1'],
        ['GLUL','RP11-54C4.2'],
        ['GLUL','SBF2'],
        ['GLUL','SFTPA2'],
        ['GLUL','SFTPB'],
        ['GLUL','SFTPC'],
        ['GLUL','TG'],
        ['GLYR1','TG'],
        ['GM2A','SFTPB'],
        ['GMEB1','TRNAU1AP'],
        ['GMPPB','LRG1'],
        ['GMPPB','TG'],
        ['GNA11','SFTPC'],
        ['GNA12','IGFBP2'],
        ['GNAI2','GNAS'],
        ['GNAI2','H1FX'],
        ['GNAI2','LMNA'],
        ['GNAI2','PLEC'],
        ['GNAI2','SCAF1'],
        ['GNAI2','SFTPA1'],
        ['GNAI2','SFTPA2'],
        ['GNAI2','SFTPC'],
        ['GNAI2','SLC34A2'],
        ['GNAI2','THBS1'],
        ['GNAI2','ZNF428'],
        ['GNAO1','MBP'],
        ['GNAS','GSPT1'],
        ['GNAS','H19'],
        ['GNAS','HNRNPA0'],
        ['GNAS','HNRNPL'],
        ['GNAS','ID4'],
        ['GNAS','IGFBP2'],
        ['GNAS','IGFBP3'],
        ['GNAS','IGFBP4'],
        ['GNAS','IMPAD1'],
        ['GNAS','JUN'],
        ['GNAS','KCNMA1'],
        ['GNAS','KRT10'],
        ['GNAS','MALAT1'],
        ['GNAS','MAP7D1'],
        ['GNAS','MAPK1'],
        ['GNAS','MARCKS'],
        ['GNAS','MAZ'],
        ['GNAS','MICALL1'],
        ['GNAS','MMP2'],
        ['GNAS','MYH11'],
        ['GNAS','MYH9'],
        ['GNAS','NOTCH2'],
        ['GNAS','PRELID1'],
        ['GNAS','PRKACA'],
        ['GNAS','PRRX2'],
        ['GNAS','PTMS'],
        ['GNAS','PTP4A2'],
        ['GNAS','PTRF'],
        ['GNAS','RAB11FIP5'],
        ['GNAS','RAB21'],
        ['GNAS','RABL6'],
        ['GNAS','RERE'],
        ['GNAS','RP11-216L13.19'],
        ['GNAS','RP11-707G14.1'],
        ['GNAS','SDCCAG3'],
        ['GNAS','SERPINE1'],
        ['GNAS','SFTPA1'],
        ['GNAS','SFTPA2'],
        ['GNAS','SFTPB'],
        ['GNAS','SFTPC'],
        ['GNAS','SPARC'],
        ['GNAS','TG'],
        ['GNAS','THBS1'],
        ['GNAS','TMEM158'],
        ['GNAS','TMEM8A'],
        ['GNAS','TNS1'],
        ['GNAS','TPO'],
        ['GNAS','TRIM28'],
        ['GNAS','TTYH3'],
        ['GNAS','TWIST1'],
        ['GNAS','TXNDC5'],
        ['GNAS','URI1'],
        ['GNAS','ZNF358'],
        ['GNB1','MYH11'],
        ['GNB1','PLEC'],
        ['GNB1','SFTPA2'],
        ['GNB1','SFTPB'],
        ['GNB1','SFTPC'],
        ['GNB2','RP11-486O12.2'],
        ['GNB2','SFTPC'],
        ['GNB2L1','IGFBP5'],
        ['GNB2L1','SFTPB'],
        ['GNB2L1','SFTPC'],
        ['GNB3','LEPREL2'],
        ['GNB3','P3H3'],
        ['GNG10','UGCG'],
        ['GNLY','SFTPB'],
        ['GNPTAB','U6'],
        ['GNS','PTRF'],
        ['GNS','SFTPA2'],
        ['GNS','SFTPB'],
        ['GNS','SFTPC'],
        ['GNS','TG'],
        ['GOLGA3','PTRF'],
        ['GOLGA8A','GOLGA8B'],
        ['GOLIM4','MYH9'],
        ['GOLPH3','IGFBP4'],
        ['GOLT1A','KISS1'],
        ['GOT2','GOT2P2'],
        ['GOT2','GOT2P3'],
        ['GP1BB','RELB'],
        ['GP2','UMOD'],
        ['GPC1','MOK'],
        ['GPC1','MZT2B'],
        ['GPC1','NFIC'],
        ['GPC1','TRIM28'],
        ['GPC1','VIM'],
        ['GPC3','SFTPC'],
        ['GPC5','GPC5-IT1'],
        ['GPD1','GSN'],
        ['GPD1','MCAM'],
        ['GPI','LRG1'],
        ['GPI','TG'],
        ['GPNMB','SFTPB'],
        ['GPNMB','SFTPC'],
        ['GPR107','SFTPC'],
        ['GPR116','RP11-213O5.4'],
        ['GPR116','RP11-262H14.7'],
        ['GPR116','SFTPC'],
        ['GPR124','PLEC'],
        ['GPR133','RP11-243M5.1'],
        ['GPR135','L3HYPDH'],
        ['GPR176','H1FX'],
        ['GPR21','RABGAP1'],
        ['GPRASP1','RP4-769N13.6'],
        ['GPRC5A','IGKC'],
        ['GPRC5A','RP11-392P7.6'],
        ['GPRC5A','SFTPA1'],
        ['GPRC5A','SFTPA2'],
        ['GPRC5A','SFTPB'],
        ['GPRC5A','SFTPC'],
        ['GPRIN1','SNCB'],
        ['GPS2','GPS2P1'],
        ['GPS2P1','RP11-542C16.2'],
        ['GPSM1','TSSC4'],
        ['GPX1','GPX1P1'],
        ['GPX3','H19'],
        ['GPX3','HLA-B'],
        ['GPX3','HLA-E'],
        ['GPX3','IGFBP4'],
        ['GPX3','IGFBP5'],
        ['GPX3','LRP1'],
        ['GPX3','LTF'],
        ['GPX3','MALAT1'],
        ['GPX3','NEAT1'],
        ['GPX3','RP1-309I22.2'],
        ['GPX3','RP5-977B1.11'],
        ['GPX3','RPL8'],
        ['GPX3','SFTPA1'],
        ['GPX3','SFTPA2'],
        ['GPX3','SFTPB'],
        ['GPX3','SFTPC'],
        ['GPX3','SPP1'],
        ['GPX3','TG'],
        ['GPX3','TIMP3'],
        ['GPX3','TPO'],
        ['GPX3','TXNIP'],
        ['GPX3','UMOD'],
        ['GPX3','ZFP36'],
        ['GPX8','PTRF'],
        ['GRAMD1A','NACC1'],
        ['GRAMD3','RP11-517I3.2'],
        ['GRB2','SCUBE3'],
        ['GREB1','RN7SL674P'],
        ['GREM1','IGFBP3'],
        ['GREM1','KIAA1199'],
        ['GREM1','LRP1'],
        ['GREM1','MMP1'],
        ['GREM1','MMP2'],
        ['GREM1','MRC2'],
        ['GREM1','MXRA8'],
        ['GREM1','MYH9'],
        ['GREM1','PLEC'],
        ['GREM1','PTRF'],
        ['GREM1','QKI'],
        ['GREM1','RP11-758N13.1'],
        ['GREM1','SERPINE1'],
        ['GREM1','SPARC'],
        ['GREM1','TGFBI'],
        ['GREM1','THBS1'],
        ['GREM1','TIMP3'],
        ['GREM1','TLN1'],
        ['GREM1','VIM'],
        ['GREM1','VIM-AS1'],
        ['GRHPR','MYH9'],
        ['GRHPR','RP11-397D12.6'],
        ['GRID1','RNF152'],
        ['GRID1','UC_338'],
        ['GRINA','SPARC'],
        ['GRIP1','RP11-123O10.4'],
        ['GRK6','GRK6P1'],
        ['GRN','HLA-DRA'],
        ['GRN','MGP'],
        ['GRN','MYH9'],
        ['GRN','SFTPA1'],
        ['GRN','SFTPA2'],
        ['GRN','SFTPB'],
        ['GRN','SFTPC'],
        ['GRN','SLC34A2'],
        ['GRWD1','PLIN3'],
        ['GRWD1','PTRF'],
        ['GS1-124K5.11','GS1-124K5.12'],
        ['GS1-124K5.12','KCTD7'],
        ['GS1-124K5.12','RABGEF1'],
        ['GS1-124K5.2','GS1-124K5.6'],
        ['GS1-259H13.10','GS1-259H13.2'],
        ['GS1-309P15.4','RDX'],
        ['GSDMB','ORMDL3'],
        ['GSDMD','RP11-661A12.4'],
        ['GSDMD','SFTPB'],
        ['GSE1','RP11-680G10.1'],
        ['GSE1','TG'],
        ['GSK3A','MARCKS'],
        ['GSN','ITGB1'],
        ['GSN','PCBP1'],
        ['GSN','PLIN4'],
        ['GSN','PTRF'],
        ['GSN','SFTPA2'],
        ['GSN','SFTPC'],
        ['GSN','SPTBN1'],
        ['GSN','TG'],
        ['GSPT1','PAPSS2'],
        ['GSPT1','PTMS'],
        ['GSPT1','SFTPC'],
        ['GSTA1','GSTA2'],
        ['GSTA1','GSTA7P'],
        ['GSTM1','GSTM4'],
        ['GSTM1','GSTM5'],
        ['GSTM2','GSTM4'],
        ['GSTM2','GSTM5'],
        ['GSTP1','SFTPA1'],
        ['GSTP1','SFTPA2'],
        ['GSTP1','SFTPB'],
        ['GSTP1','SFTPC'],
        ['GSTP1','TG'],
        ['GTF2A1L','STON1'],
        ['GTF2IRD2','GTF2IRD2P1'],
        ['GTF3C1','SFTPA2'],
        ['GTPBP1','LAMC1'],
        ['GTPBP1','RP3-508I15.14'],
        ['GTPBP1','RP3-508I15.19'],
        ['GTPBP2','SFTPB'],
        ['GTSE1','TRMU'],
        ['GUCA1B','MRPS10'],
        ['GUCD1','LDLR'],
        ['GUCD1','LRG1'],
        ['GUCD1','PTRF'],
        ['GUCD1','RFC2'],
        ['GUSBP11','IGLL3P'],
        ['GUSBP11','KB-1572G7.2'],
        ['H19','IGF2'],
        ['H19','LRP1'],
        ['H19','MALAT1'],
        ['H19','MYL2'],
        ['H19','NEAT1'],
        ['H19','SFTPC'],
        ['H19','TG'],
        ['H19','TPO'],
        ['H1F0','TG'],
        ['H1FX','NXN'],
        ['H1FX','RABL6'],
        ['H1FX','RP11-216L13.19'],
        ['H1FX','SCARF2'],
        ['H1FX','TG'],
        ['H1FX','UBE2Q1'],
        ['H2AFJ','HIST1H3J'],
        ['H2AFV','SCAF1'],
        ['H2AFY','IGFBP4'],
        ['H3F3B','H3F3C'],
        ['H6PD','LRG1'],
        ['H6PD','PTRF'],
        ['H6PD','SERPINE1'],
        ['H6PD','SFTPB'],
        ['H6PD','SFTPC'],
        ['H6PD','SPSB1'],
        ['HABP4','PRKACA'],
        ['HAPLN3','MFGE8'],
        ['HARS2','ZMAT2'],
        ['HAUS4','PRMT5'],
        ['HBA2','HERPUD1'],
        ['HBA2','NFE2L1'],
        ['HBB','HBD'],
        ['HBB','ICE1'],
        ['HBB','KLF6'],
        ['HBB','LYST'],
        ['HBB','MTMR14'],
        ['HBB','SFTPC'],
        ['HCAR2','HCAR3'],
        ['HCG11','LINC00657'],
        ['HCG15','HCG16'],
        ['HCG4P3','XXBAC-BPG248L24.12'],
        ['HCG4P3','ZNRD1-AS1'],
        ['HCG4P5','HCG4P7'],
        ['HCG4P5','HLA-C'],
        ['HCG4P5','XXBAC-BPG248L24.12'],
        ['HCG4P7','XXBAC-BPG248L24.12'],
        ['HCN2','PTMS'],
        ['HDAC10','MAPK12'],
        ['HDAC2','ITGA5'],
        ['HDAC5','SFTPC'],
        ['HDAC7','KIF3C'],
        ['HDAC7','SFTPA1'],
        ['HDAC7','SFTPC'],
        ['HDGF','HDGFP1'],
        ['HDGF','MRPL24'],
        ['HDGF','SFTPC'],
        ['HDGFRP2','PTMS'],
        ['HDGFRP2','TAOK2'],
        ['HDLBP','MYH11'],
        ['HDLBP','MYH9'],
        ['HDLBP','MYL9'],
        ['HDLBP','SERPINE1'],
        ['HDLBP','SFTPA1'],
        ['HDLBP','SFTPC'],
        ['HDLBP','SPARC'],
        ['HDLBP','TIMP2'],
        ['HDLBP','VIM'],
        ['HEATR1','RP11-385F5.5'],
        ['HECA','SLC48A1'],
        ['HECTD4','SFTPC'],
        ['HEG1','MAZ'],
        ['HEG1','SFTPA1'],
        ['HEG1','SFTPB'],
        ['HEG1','SFTPC'],
        ['HELZ2','SFTPC'],
        ['HEPHL1','PANX1'],
        ['HERC2','HERC2P10'],
        ['HERC2P2','HERC2P3'],
        ['HERC2P2','HERC2P9'],
        ['HERPUD1','RP11-325K4.3'],
        ['HEXA','RP11-106M3.2'],
        ['HGSNAT','SFTPC'],
        ['HHIPL1','RP11-543C4.1'],
        ['HHLA1','OC90'],
        ['HIAT1','SFTPA1'],
        ['HIAT1','SLC35A3'],
        ['HIATL1','SFTPC'],
        ['HIF3A','PPP5C'],
        ['HILPDA','RP11-155G14.5'],
        ['HILPDA','RP11-155G14.6'],
        ['HILPDA','RP11-212P7.3'],
        ['HIP1','SFTPB'],
        ['HIPK1','SFTPC'],
        ['HIPK2','SFTPC'],
        ['HIPK3','SFTPA1'],
        ['HIPK4','PRX'],
        ['HIRA','TG'],
        ['HIST4H4','RP11-174G6.5'],
        ['HIVEP1','RP11-456H18.2'],
        ['HIVEP2','RP11-439L18.2'],
        ['HK1','TG'],
        ['HK2','HK2P1'],
        ['HLA-A','HLA-B'],
        ['HLA-A','HLA-C'],
        ['HLA-A','HLA-E'],
        ['HLA-A','HLA-F'],
        ['HLA-A','HLA-G'],
        ['HLA-A','HLA-H'],
        ['HLA-A','HLA-J'],
        ['HLA-A','HLA-L'],
        ['HLA-A','HLA-U'],
        ['HLA-A','IGKC'],
        ['HLA-A','MYH9'],
        ['HLA-A','SFTPA1'],
        ['HLA-A','SFTPA2'],
        ['HLA-A','SFTPB'],
        ['HLA-A','SFTPC'],
        ['HLA-A','SLC34A2'],
        ['HLA-A','TG'],
        ['HLA-B','HLA-C'],
        ['HLA-B','HLA-DRA'],
        ['HLA-B','HLA-E'],
        ['HLA-B','HLA-F'],
        ['HLA-B','HLA-G'],
        ['HLA-B','HLA-H'],
        ['HLA-B','HLA-J'],
        ['HLA-B','HLA-L'],
        ['HLA-B','HNRNPA2B1'],
        ['HLA-B','HSPG2'],
        ['HLA-B','IGKC'],
        ['HLA-B','LPCAT1'],
        ['HLA-B','MALAT1'],
        ['HLA-B','MYH9'],
        ['HLA-B','NEAT1'],
        ['HLA-B','PSAP'],
        ['HLA-B','RP11-186B7.4'],
        ['HLA-B','SFTPA1'],
        ['HLA-B','SFTPA2'],
        ['HLA-B','SFTPB'],
        ['HLA-B','SFTPC'],
        ['HLA-B','SPTBN1'],
        ['HLA-B','SRRM2'],
        ['HLA-B','THBS1'],
        ['HLA-C','HLA-E'],
        ['HLA-C','HLA-F'],
        ['HLA-C','HLA-G'],
        ['HLA-C','HLA-H'],
        ['HLA-C','HLA-L'],
        ['HLA-C','MYH11'],
        ['HLA-C','MYH9'],
        ['HLA-C','NEAT1'],
        ['HLA-C','PSAP'],
        ['HLA-C','SFTPA1'],
        ['HLA-C','SFTPA2'],
        ['HLA-C','SFTPB'],
        ['HLA-C','SFTPC'],
        ['HLA-DMA','SFTPA2'],
        ['HLA-DPA1','HLA-DRA'],
        ['HLA-DPA1','SFTPA1'],
        ['HLA-DPA1','SFTPA2'],
        ['HLA-DPA1','SFTPB'],
        ['HLA-DPA1','SFTPC'],
        ['HLA-DPA1','SLC34A2'],
        ['HLA-DPB1','HLA-DQB1'],
        ['HLA-DPB1','HLA-DRB1'],
        ['HLA-DPB1','HLA-DRB5'],
        ['HLA-DPB1','SFTPA1'],
        ['HLA-DPB1','SFTPA2'],
        ['HLA-DPB1','SFTPB'],
        ['HLA-DPB1','SFTPC'],
        ['HLA-DQA1','HLA-DQA2'],
        ['HLA-DQA1','SFTPC'],
        ['HLA-DQB1','HLA-DQB2'],
        ['HLA-DQB1','SFTPC'],
        ['HLA-DQB2','XXBAC-BPG254F23.6'],
        ['HLA-DRA','HLA-E'],
        ['HLA-DRA','IGKC'],
        ['HLA-DRA','MALAT1'],
        ['HLA-DRA','P4HB'],
        ['HLA-DRA','PSAP'],
        ['HLA-DRA','SFTPA1'],
        ['HLA-DRA','SFTPA2'],
        ['HLA-DRA','SFTPB'],
        ['HLA-DRA','SFTPC'],
        ['HLA-DRA','THBS1'],
        ['HLA-DRA','TNS1'],
        ['HLA-DRA','ZYX'],
        ['HLA-DRB1','HLA-DRB5'],
        ['HLA-DRB1','HLA-DRB6'],
        ['HLA-DRB1','HLA-DRB9'],
        ['HLA-DRB1','SFTPA1'],
        ['HLA-DRB1','SFTPA2'],
        ['HLA-DRB1','SFTPC'],
        ['HLA-DRB5','HLA-DRB6'],
        ['HLA-DRB5','HLA-DRB9'],
        ['HLA-E','HLA-F'],
        ['HLA-E','HLA-G'],
        ['HLA-E','HLA-H'],
        ['HLA-E','HLA-J'],
        ['HLA-E','IGKC'],
        ['HLA-E','INMT'],
        ['HLA-E','LMNA'],
        ['HLA-E','LTBP2'],
        ['HLA-E','LYZ'],
        ['HLA-E','MYADM'],
        ['HLA-E','MYH9'],
        ['HLA-E','PIGR'],
        ['HLA-E','PLIN4'],
        ['HLA-E','PSAP'],
        ['HLA-E','RP11-1143G9.4'],
        ['HLA-E','SCARF1'],
        ['HLA-E','SERINC3'],
        ['HLA-E','SFTPA1'],
        ['HLA-E','SFTPA2'],
        ['HLA-E','SFTPB'],
        ['HLA-E','SFTPC'],
        ['HLA-E','SLC34A2'],
        ['HLA-E','SPTBN1'],
        ['HLA-E','STOM'],
        ['HLA-E','TAPBP'],
        ['HLA-E','TGM2'],
        ['HLA-E','TIMP3'],
        ['HLA-E','ZFP36'],
        ['HLA-G','HLA-H'],
        ['HLA-G','HLA-J'],
        ['HLA-G','HLA-L'],
        ['HLA-H','HLA-J'],
        ['HLA-H','HLA-U'],
        ['HLA-J','HLA-K'],
        ['HM13','RP3-324O17.4'],
        ['HM13','SMR3B'],
        ['HMGB1','HMGB1P8'],
        ['HMGB1','SP100'],
        ['HMGN2P15','RAMP2-AS1'],
        ['HMGN2P46','RP11-519G16.3'],
        ['HMOX2','PTRF'],
        ['HMOX2','TPM4'],
        ['HNRNPA0','SIRPA'],
        ['HNRNPA1','HNRNPA1L2'],
        ['HNRNPA1','HNRNPA1P48'],
        ['HNRNPA1','HNRNPA1P54'],
        ['HNRNPA1L2','RP11-64P12.8'],
        ['HNRNPA2B1','MYH11'],
        ['HNRNPA2B1','SFTPA2'],
        ['HNRNPA2B1','SFTPB'],
        ['HNRNPA2B1','SFTPC'],
        ['HNRNPA2B1','TG'],
        ['HNRNPA3','HNRNPA3P10'],
        ['HNRNPA3','HNRNPA3P12'],
        ['HNRNPA3','RP1-167A19.5'],
        ['HNRNPAB','TPBG'],
        ['HNRNPC','HNRNPCP1'],
        ['HNRNPC','HNRNPCP2'],
        ['HNRNPCP1','KIAA0586'],
        ['HNRNPD','IGFBP4'],
        ['HNRNPD','PLEC'],
        ['HNRNPH1','HNRNPH1P2'],
        ['HNRNPH1','SFTPC'],
        ['HNRNPH3','SFTPC'],
        ['HNRNPK','HNRNPKP1'],
        ['HNRNPK','HNRNPKP2'],
        ['HNRNPK','HNRNPKP4'],
        ['HNRNPK','SFTPA2'],
        ['HNRNPL','HNRNPLP2'],
        ['HNRNPL','VIM'],
        ['HNRNPL','YBX1'],
        ['HNRNPM','MARCH2'],
        ['HNRNPM','MYH9'],
        ['HNRNPM','SFTPC'],
        ['HNRNPU','HNRNPU-AS1'],
        ['HNRNPU','RP11-11N7.5'],
        ['HNRNPU','SFTPA1'],
        ['HNRNPU','SFTPC'],
        ['HNRNPU','XBP1'],
        ['HNRNPUL1','SFTPA2'],
        ['HNRNPUL1','SFTPC'],
        ['HNRNPUL2','MEPCE'],
        ['HNRNPUL2','SFTPC'],
        ['HNRNPUL2','SRM'],
        ['HNRNPUL2-BSCL2','MEPCE'],
        ['HNRNPUL2-BSCL2','SRM'],
        ['HNRPCP','KIAA0586'],
        ['HNRPDL','SFTPC'],
        ['HOMER3','PLEC'],
        ['HOXA3','HOXA5'],
        ['HOXA6','RP1-170O19.23'],
        ['HOXB3','HOXB4'],
        ['HOXC10','HOXC4'],
        ['HOXC4','HOXC6'],
        ['HOXC4','HOXC9'],
        ['HOXC4','RP11-834C11.12'],
        ['HOXC8','HOXC9'],
        ['HOXD3','HOXD8'],
        ['HP','HPR'],
        ['HP','ORM1'],
        ['HP','RP11-986E7.7'],
        ['HP','SAA1'],
        ['HP','SERPINA1'],
        ['HP','SERPINA3'],
        ['HP1BP3','PTRF'],
        ['HP1BP3','SFTPB'],
        ['HPS6','IGFBP4'],
        ['HPX','SERPINA1'],
        ['HRAS','RPS2'],
        ['HRC','MYH7'],
        ['HS1BP3','HS1BP3-IT1'],
        ['HSA-MIR-6080','PLEKHM1P'],
        ['HSD11B1L','RPL36'],
        ['HSD17B12','RP5-933K21.2'],
        ['HSD17B4','TG'],
        ['HSD17B7','HSD17B7P2'],
        ['HSP90AA1','HSP90AA2P'],
        ['HSP90AA1','HSP90AA4P'],
        ['HSP90AA1','HSP90AA6P'],
        ['HSP90AA1','RP11-1033A18.1'],
        ['HSP90AA1','RP11-144L1.8'],
        ['HSP90AA1','TG'],
        ['HSP90AB1','HSP90AB2P'],
        ['HSP90AB1','HSP90AB3P'],
        ['HSP90AB1','SFTPA2'],
        ['HSP90AB1','SLC29A1'],
        ['HSP90AB1','TG'],
        ['HSP90AB3P','RP11-759L5.2'],
        ['HSP90B1','HSP90B2P'],
        ['HSP90B1','HSP90B3P'],
        ['HSPA12A','RP11-539I5.1'],
        ['HSPA1A','HSPA1B'],
        ['HSPA1B','TG'],
        ['HSPA1L','LSM2'],
        ['HSPA5','RP11-334L9.1'],
        ['HSPA5','SERPINE1'],
        ['HSPA8','HSPA8P1'],
        ['HSPA8','HSPA8P16'],
        ['HSPA8','HSPA8P5'],
        ['HSPA8','HSPA8P7'],
        ['HSPA8','HSPA8P8'],
        ['HSPA8','HSPA8P9'],
        ['HSPA8','MYH9'],
        ['HSPA8','RP11-488L18.1'],
        ['HSPA8','SFTPA2'],
        ['HSPA8','SFTPC'],
        ['HSPA8','SNORD14E'],
        ['HSPB1','HSPB1P1'],
        ['HSPB1','KRT5'],
        ['HSPB1','SFTPC'],
        ['HSPB1','TG'],
        ['HSPB1','VIM'],
        ['HSPB6','IGFBP4'],
        ['HSPB6','MYH9'],
        ['HSPB6','PLEKHA4'],
        ['HSPB6','PLIN1'],
        ['HSPB6','SFTPA1'],
        ['HSPB6','TG'],
        ['HSPB6','TIMP1'],
        ['HSPB6','TNS1'],
        ['HSPB7','TG'],
        ['HSPB8','RP11-64B16.2'],
        ['HSPD1','HSPD1P1'],
        ['HSPE1','HSPE1P7'],
        ['HSPG2','IGKC'],
        ['HSPG2','ITM2C'],
        ['HSPG2','LGALS1'],
        ['HSPG2','MEGF6'],
        ['HSPG2','MYH9'],
        ['HSPG2','MYL9'],
        ['HSPG2','PCDHGC3'],
        ['HSPG2','PLEC'],
        ['HSPG2','PTRF'],
        ['HSPG2','SERPINE1'],
        ['HSPG2','SFTPA1'],
        ['HSPG2','SFTPA2'],
        ['HSPG2','SFTPB'],
        ['HSPG2','SFTPC'],
        ['HSPG2','TAGLN'],
        ['HSPG2','TGFBI'],
        ['HSPG2','TLN1'],
        ['HSPG2','TNC'],
        ['HSPG2','VIM'],
        ['HSPG2','VIM-AS1'],
        ['HTN3','ITPR2'],
        ['HTN3','RNF150'],
        ['HTN3','TRAM1'],
        ['HTRA1','PLIN4'],
        ['HTT','SFTPB'],
        ['HUWE1','MYH9'],
        ['HUWE1','RPS2'],
        ['HUWE1','SFTPA1'],
        ['HUWE1','SFTPA2'],
        ['HUWE1','THBS2'],
        ['HYAL2','SFTPA2'],
        ['HYAL2','SFTPC'],
        ['HYDIN','HYDIN2'],
        ['HYOU1','SPARC'],
        ['HYPK','SERF2'],
        ['IAH1','RP11-400L8.2'],
        ['ICA1L','PYURF'],
        ['ICAM1','IGKC'],
        ['ICAM1','PCYOX1'],
        ['ICAM1','SFTPA1'],
        ['ICAM1','SFTPA2'],
        ['ICAM1','SFTPB'],
        ['ICAM1','SFTPC'],
        ['ICAM3','RAVER1'],
        ['ID4','PTMS'],
        ['ID4','TG'],
        ['ID4','TMEM158'],
        ['IDH1','RP11-288G3.4'],
        ['IDH2','RP11-617F23.1'],
        ['IDS','RP13-507I23.1'],
        ['IDS','SFTPC'],
        ['IDS','TSPYL1'],
        ['IDS','TUBB4A'],
        ['IER2','TG'],
        ['IER3','SFTPC'],
        ['IFI16','PYHIN1'],
        ['IFI27','SFTPC'],
        ['IFI30','SFTPA1'],
        ['IFI30','SFTPA2'],
        ['IFI30','SFTPC'],
        ['IFITM1','IFITM3'],
        ['IFITM2','IFITM3'],
        ['IFITM2','IFITM9P'],
        ['IFITM2','RP11-467L13.5'],
        ['IFITM2','SFTPC'],
        ['IFITM3','IFITM9P'],
        ['IFITM3','RP11-396M11.1'],
        ['IFITM3','RP11-416L21.2'],
        ['IFITM3','RP11-467L13.5'],
        ['IFITM3','SFTPA2'],
        ['IFITM3','SFTPB'],
        ['IFITM3','SFTPC'],
        ['IFNAR2','IFNAR2'],
        ['IFNAR2','IL10RB'],
        ['IFNE','MIR31HG'],
        ['IFRD1','LSMEM1'],
        ['IFT140','LA16C-380F5.3'],
        ['IFT140','LA16C-425C2.1'],
        ['IFT80','RP11-432B6.3'],
        ['IFT80','TRIM59'],
        ['IGF1R','SFTPA2'],
        ['IGF1R','TG'],
        ['IGF2','IGFBP4'],
        ['IGF2','IL6ST'],
        ['IGF2','KLF7'],
        ['IGF2','LRP1'],
        ['IGF2','MEG3'],
        ['IGF2','PLLP'],
        ['IGF2','RBM6'],
        ['IGF2','RP1-309I22.2'],
        ['IGF2','SERPINE2'],
        ['IGF2','SNX2'],
        ['IGF2','SPARC'],
        ['IGF2','SPP1'],
        ['IGF2','TIMP3'],
        ['IGF2','TSIX'],
        ['IGF2','VSNL1'],
        ['IGF2R','SFTPA1'],
        ['IGF2R','SFTPB'],
        ['IGF2R','SFTPC'],
        ['IGFBP2','LAMC1'],
        ['IGFBP2','MAZ'],
        ['IGFBP2','MBD3'],
        ['IGFBP2','NCOR2'],
        ['IGFBP2','RCC2'],
        ['IGFBP2','SFTPC'],
        ['IGFBP2','TRIM28'],
        ['IGFBP2','UQCR11'],
        ['IGFBP3','IGFBP4'],
        ['IGFBP3','LGALS1'],
        ['IGFBP3','LOX'],
        ['IGFBP3','LRP1'],
        ['IGFBP3','MARCKS'],
        ['IGFBP3','MMP1'],
        ['IGFBP3','MMP14'],
        ['IGFBP3','MYH9'],
        ['IGFBP3','MYL9'],
        ['IGFBP3','PCSK7'],
        ['IGFBP3','PEA15'],
        ['IGFBP3','PKM'],
        ['IGFBP3','PLEC'],
        ['IGFBP3','PTRF'],
        ['IGFBP3','RP5-977B1.11'],
        ['IGFBP3','S100A6'],
        ['IGFBP3','SERPINE1'],
        ['IGFBP3','SERPINE2'],
        ['IGFBP3','SPARC'],
        ['IGFBP3','TGFBI'],
        ['IGFBP3','THBS1'],
        ['IGFBP3','TIMP3'],
        ['IGFBP3','TNC'],
        ['IGFBP3','TPM1'],
        ['IGFBP3','TPM2'],
        ['IGFBP3','VIM'],
        ['IGFBP3','VIM-AS1'],
        ['IGFBP3','WDR1'],
        ['IGFBP4','IGFBP5'],
        ['IGFBP4','IGH@'],
        ['IGFBP4','ITGB5'],
        ['IGFBP4','KLF2'],
        ['IGFBP4','LAMC1'],
        ['IGFBP4','LARP6'],
        ['IGFBP4','LASP1'],
        ['IGFBP4','LENG8'],
        ['IGFBP4','LOXL2'],
        ['IGFBP4','LRP1'],
        ['IGFBP4','MARCKS'],
        ['IGFBP4','MAZ'],
        ['IGFBP4','MYH9'],
        ['IGFBP4','MZT2B'],
        ['IGFBP4','NEAT1'],
        ['IGFBP4','NELFB'],
        ['IGFBP4','NR2F2'],
        ['IGFBP4','NT5DC2'],
        ['IGFBP4','PCSK7'],
        ['IGFBP4','PDLIM4'],
        ['IGFBP4','PKM'],
        ['IGFBP4','PLBD2'],
        ['IGFBP4','PLEC'],
        ['IGFBP4','POLR3H'],
        ['IGFBP4','PQLC1'],
        ['IGFBP4','PTMS'],
        ['IGFBP4','PTRF'],
        ['IGFBP4','PXDN'],
        ['IGFBP4','PXN'],
        ['IGFBP4','RP11-20I23.1'],
        ['IGFBP4','RP5-977B1.11'],
        ['IGFBP4','SCARF2'],
        ['IGFBP4','SCUBE3'],
        ['IGFBP4','SERPINE1'],
        ['IGFBP4','SFTPA1'],
        ['IGFBP4','SFTPA2'],
        ['IGFBP4','SFTPB'],
        ['IGFBP4','SFTPC'],
        ['IGFBP4','SLC35E1'],
        ['IGFBP4','SPARC'],
        ['IGFBP4','SPIRE1'],
        ['IGFBP4','TAGLN'],
        ['IGFBP4','TGFB1'],
        ['IGFBP4','TGFBI'],
        ['IGFBP4','THBS1'],
        ['IGFBP4','TIMP1'],
        ['IGFBP4','TIMP3'],
        ['IGFBP4','TRIM28'],
        ['IGFBP4','TSR3'],
        ['IGFBP4','TXNDC5'],
        ['IGFBP4','UBE2M'],
        ['IGFBP4','UPF1'],
        ['IGFBP4','URI1'],
        ['IGFBP4','VIM'],
        ['IGFBP4','XYLT1'],
        ['IGFBP5','LUM'],
        ['IGFBP5','MYH9'],
        ['IGFBP5','NEAT1'],
        ['IGFBP5','NFIC'],
        ['IGFBP5','NR4A1'],
        ['IGFBP5','PRKAR1A'],
        ['IGFBP5','PTMS'],
        ['IGFBP5','RPL11'],
        ['IGFBP5','RPL35A'],
        ['IGFBP5','RPL37A'],
        ['IGFBP5','SFTPC'],
        ['IGFBP5','SPARC'],
        ['IGFBP5','SRRM2'],
        ['IGFBP5','TG'],
        ['IGFBP5','THBS1'],
        ['IGFBP5','TIMP2'],
        ['IGFBP5','TNC'],
        ['IGFBP5','TPO'],
        ['IGFBP5','TPT1'],
        ['IGFBP5','TXNIP'],
        ['IGFBP5','VIM'],
        ['IGFBP5','XIRP1'],
        ['IGFBP6','MYH9'],
        ['IGFBP6','PLEC'],
        ['IGFBP6','TRIM28'],
        ['IGFBP7','IGK@'],
        ['IGFBP7','IGKC'],
        ['IGFBP7','MYH9'],
        ['IGFBP7','NDNL2'],
        ['IGFBP7','SFTPA2'],
        ['IGFBP7','SFTPC'],
        ['IGFBP7','TG'],
        ['IGFLR1','U2AF1L4'],
        ['IGH@','IGH@'],
        ['IGH@','IGHA1'],
        ['IGH@','IGHA2'],
        ['IGH@','IGHD'],
        ['IGH@','IGHG1'],
        ['IGH@','IGHG2'],
        ['IGH@','IGHG3'],
        ['IGH@','IGHGP'],
        ['IGH@','IGHM'],
        ['IGH@','IGHV1-12'],
        ['IGH@','IGHV1-18'],
        ['IGH@','IGHV1-2'],
        ['IGH@','IGHV1-24'],
        ['IGH@','IGHV1-3'],
        ['IGH@','IGHV1-46'],
        ['IGH@','IGHV1-58'],
        ['IGH@','IGHV1-69'],
        ['IGH@','IGHV1-69-2'],
        ['IGH@','IGHV2-26'],
        ['IGH@','IGHV2-5'],
        ['IGH@','IGHV2-70'],
        ['IGH@','IGHV3-11'],
        ['IGH@','IGHV3-13'],
        ['IGH@','IGHV3-15'],
        ['IGH@','IGHV3-20'],
        ['IGH@','IGHV3-21'],
        ['IGH@','IGHV3-23'],
        ['IGH@','IGHV3-30'],
        ['IGH@','IGHV3-33'],
        ['IGH@','IGHV3-43'],
        ['IGH@','IGHV3-48'],
        ['IGH@','IGHV3-49'],
        ['IGH@','IGHV3-53'],
        ['IGH@','IGHV3-64'],
        ['IGH@','IGHV3-66'],
        ['IGH@','IGHV3-7'],
        ['IGH@','IGHV3-71'],
        ['IGH@','IGHV3-72'],
        ['IGH@','IGHV3-73'],
        ['IGH@','IGHV3-74'],
        ['IGH@','IGHV3OR16-8'],
        ['IGH@','IGHV3OR16-9'],
        ['IGH@','IGHV4-28'],
        ['IGH@','IGHV4-31'],
        ['IGH@','IGHV4-34'],
        ['IGH@','IGHV4-39'],
        ['IGH@','IGHV4-55'],
        ['IGH@','IGHV4-59'],
        ['IGH@','IGHV5-51'],
        ['IGH@','IGHV6-1'],
        ['IGH@','IGHV7-81'],
        ['IGH@','IGK@'],
        ['IGH@','IGKC'],
        ['IGH@','IGKV1-8'],
        ['IGH@','IGKV3-11'],
        ['IGH@','MALAT1'],
        ['IGH@','MUC5B'],
        ['IGH@','MYH11'],
        ['IGH@','MYL9'],
        ['IGH@','MYLK'],
        ['IGH@','RP11-1166P10.8'],
        ['IGH@','RPL8'],
        ['IGH@','RPS8'],
        ['IGH@','TPT1'],
        ['IGHA1','IGHA2'],
        ['IGHA1','IGHG1'],
        ['IGHA1','IGHG3'],
        ['IGHA1','IGHJ6'],
        ['IGHA1','IGHM'],
        ['IGHA1','IGHV1-18'],
        ['IGHA1','IGHV1-2'],
        ['IGHA1','IGHV1-24'],
        ['IGHA1','IGHV1-3'],
        ['IGHA1','IGHV1-45'],
        ['IGHA1','IGHV1-46'],
        ['IGHA1','IGHV1-58'],
        ['IGHA1','IGHV1-69'],
        ['IGHA1','IGHV1-69-2'],
        ['IGHA1','IGHV1-8'],
        ['IGHA1','IGHV2-26'],
        ['IGHA1','IGHV2-5'],
        ['IGHA1','IGHV2-70'],
        ['IGHA1','IGHV3-11'],
        ['IGHA1','IGHV3-13'],
        ['IGHA1','IGHV3-15'],
        ['IGHA1','IGHV3-20'],
        ['IGHA1','IGHV3-21'],
        ['IGHA1','IGHV3-23'],
        ['IGHA1','IGHV3-30'],
        ['IGHA1','IGHV3-33'],
        ['IGHA1','IGHV3-43'],
        ['IGHA1','IGHV3-48'],
        ['IGHA1','IGHV3-49'],
        ['IGHA1','IGHV3-53'],
        ['IGHA1','IGHV3-64'],
        ['IGHA1','IGHV3-66'],
        ['IGHA1','IGHV3-7'],
        ['IGHA1','IGHV3-72'],
        ['IGHA1','IGHV3-73'],
        ['IGHA1','IGHV3-74'],
        ['IGHA1','IGHV3-9'],
        ['IGHA1','IGHV3OR16-9'],
        ['IGHA1','IGHV4-28'],
        ['IGHA1','IGHV4-31'],
        ['IGHA1','IGHV4-34'],
        ['IGHA1','IGHV4-39'],
        ['IGHA1','IGHV4-4'],
        ['IGHA1','IGHV4-55'],
        ['IGHA1','IGHV4-59'],
        ['IGHA1','IGHV4-61'],
        ['IGHA1','IGHV4OR15-8'],
        ['IGHA1','IGHV5-51'],
        ['IGHA1','IGHV6-1'],
        ['IGHA1','IGHV7-27'],
        ['IGHA1','IGHV7-81'],
        ['IGHA1','IGK@'],
        ['IGHA1','IGKC'],
        ['IGHA1','IGKV1-8'],
        ['IGHA1','IGKV3-11'],
        ['IGHA1','MUC4'],
        ['IGHA1','NCOR2'],
        ['IGHA1','PSAP'],
        ['IGHA1','RP11-186B7.4'],
        ['IGHA1','RP11-603B24.5'],
        ['IGHA1','SCGB1A1'],
        ['IGHA1','SFTPA1'],
        ['IGHA1','SFTPA2'],
        ['IGHA1','SFTPB'],
        ['IGHA1','SFTPC'],
        ['IGHA1','SLC34A2'],
        ['IGHA1','TPT1'],
        ['IGHA1','ZFP36'],
        ['IGHA2','IGHG1'],
        ['IGHA2','IGHG3'],
        ['IGHA2','IGHM'],
        ['IGHA2','IGHV1-18'],
        ['IGHA2','IGHV1-2'],
        ['IGHA2','IGHV1-24'],
        ['IGHA2','IGHV1-3'],
        ['IGHA2','IGHV1-46'],
        ['IGHA2','IGHV1-69-2'],
        ['IGHA2','IGHV2-26'],
        ['IGHA2','IGHV2-5'],
        ['IGHA2','IGHV2-70'],
        ['IGHA2','IGHV3-11'],
        ['IGHA2','IGHV3-13'],
        ['IGHA2','IGHV3-15'],
        ['IGHA2','IGHV3-20'],
        ['IGHA2','IGHV3-21'],
        ['IGHA2','IGHV3-23'],
        ['IGHA2','IGHV3-30'],
        ['IGHA2','IGHV3-33'],
        ['IGHA2','IGHV3-43'],
        ['IGHA2','IGHV3-48'],
        ['IGHA2','IGHV3-49'],
        ['IGHA2','IGHV3-53'],
        ['IGHA2','IGHV3-64'],
        ['IGHA2','IGHV3-7'],
        ['IGHA2','IGHV3-72'],
        ['IGHA2','IGHV3-73'],
        ['IGHA2','IGHV3-74'],
        ['IGHA2','IGHV3OR16-9'],
        ['IGHA2','IGHV4-31'],
        ['IGHA2','IGHV4-34'],
        ['IGHA2','IGHV4-39'],
        ['IGHA2','IGHV4-4'],
        ['IGHA2','IGHV4-59'],
        ['IGHA2','IGHV5-51'],
        ['IGHA2','IGHV6-1'],
        ['IGHA2','IGK@'],
        ['IGHA2','IGKC'],
        ['IGHD','IGHG1'],
        ['IGHD','IGHG3'],
        ['IGHD','IGHJ6'],
        ['IGHD','IGHM'],
        ['IGHD','IGHV1-18'],
        ['IGHD','IGHV1-24'],
        ['IGHD','IGHV1-3'],
        ['IGHD','IGHV1-46'],
        ['IGHD','IGHV1-58'],
        ['IGHD','IGHV2-26'],
        ['IGHD','IGHV2-5'],
        ['IGHD','IGHV3-20'],
        ['IGHD','IGHV3-21'],
        ['IGHD','IGHV3-23'],
        ['IGHD','IGHV3-30'],
        ['IGHD','IGHV3-48'],
        ['IGHD','IGHV3-49'],
        ['IGHD','IGHV3-7'],
        ['IGHD','IGHV4-34'],
        ['IGHD','IGHV4-39'],
        ['IGHD','IGHV5-51'],
        ['IGHE','IGHG1'],
        ['IGHE','IGHM'],
        ['IGHG1','IGHG2'],
        ['IGHG1','IGHG3'],
        ['IGHG1','IGHG4'],
        ['IGHG1','IGHGP'],
        ['IGHG1','IGHM'],
        ['IGHG1','IGHV1-14'],
        ['IGHG1','IGHV1-18'],
        ['IGHG1','IGHV1-2'],
        ['IGHG1','IGHV1-24'],
        ['IGHG1','IGHV1-3'],
        ['IGHG1','IGHV1-45'],
        ['IGHG1','IGHV1-46'],
        ['IGHG1','IGHV1-58'],
        ['IGHG1','IGHV1-67'],
        ['IGHG1','IGHV1-69'],
        ['IGHG1','IGHV1-69-2'],
        ['IGHG1','IGHV1-8'],
        ['IGHG1','IGHV1OR16-4'],
        ['IGHG1','IGHV2-26'],
        ['IGHG1','IGHV2-5'],
        ['IGHG1','IGHV2-70'],
        ['IGHG1','IGHV3-11'],
        ['IGHG1','IGHV3-13'],
        ['IGHG1','IGHV3-15'],
        ['IGHG1','IGHV3-20'],
        ['IGHG1','IGHV3-21'],
        ['IGHG1','IGHV3-23'],
        ['IGHG1','IGHV3-30'],
        ['IGHG1','IGHV3-33'],
        ['IGHG1','IGHV3-38'],
        ['IGHG1','IGHV3-43'],
        ['IGHG1','IGHV3-48'],
        ['IGHG1','IGHV3-49'],
        ['IGHG1','IGHV3-53'],
        ['IGHG1','IGHV3-62'],
        ['IGHG1','IGHV3-64'],
        ['IGHG1','IGHV3-65'],
        ['IGHG1','IGHV3-66'],
        ['IGHG1','IGHV3-7'],
        ['IGHG1','IGHV3-72'],
        ['IGHG1','IGHV3-73'],
        ['IGHG1','IGHV3-74'],
        ['IGHG1','IGHV3-9'],
        ['IGHG1','IGHV3OR16-13'],
        ['IGHG1','IGHV3OR16-15'],
        ['IGHG1','IGHV3OR16-7'],
        ['IGHG1','IGHV3OR16-8'],
        ['IGHG1','IGHV3OR16-9'],
        ['IGHG1','IGHV4-28'],
        ['IGHG1','IGHV4-31'],
        ['IGHG1','IGHV4-34'],
        ['IGHG1','IGHV4-39'],
        ['IGHG1','IGHV4-4'],
        ['IGHG1','IGHV4-55'],
        ['IGHG1','IGHV4-59'],
        ['IGHG1','IGHV4-61'],
        ['IGHG1','IGHV4OR15-8'],
        ['IGHG1','IGHV5-51'],
        ['IGHG1','IGHV6-1'],
        ['IGHG1','IGK@'],
        ['IGHG1','IGKC'],
        ['IGHG1','IGKV1-8'],
        ['IGHG1','IGKV3-11'],
        ['IGHG1','RP11-1166P10.8'],
        ['IGHG1','RP11-731F5.2'],
        ['IGHG1','RP11-812E19.9'],
        ['IGHG1','RPS8'],
        ['IGHG1','SFTPB'],
        ['IGHG1','SFTPC'],
        ['IGHG1','TPT1'],
        ['IGHG2','IGHG3'],
        ['IGHG2','IGHG4'],
        ['IGHG2','IGHGP'],
        ['IGHG2','IGHM'],
        ['IGHG2','IGHV1-18'],
        ['IGHG2','IGHV1-2'],
        ['IGHG2','IGHV1-24'],
        ['IGHG2','IGHV1-46'],
        ['IGHG2','IGHV1-69'],
        ['IGHG2','IGHV2-5'],
        ['IGHG2','IGHV3-13'],
        ['IGHG2','IGHV3-15'],
        ['IGHG2','IGHV3-20'],
        ['IGHG2','IGHV3-21'],
        ['IGHG2','IGHV3-23'],
        ['IGHG2','IGHV3-30'],
        ['IGHG2','IGHV3-33'],
        ['IGHG2','IGHV3-49'],
        ['IGHG2','IGHV3-53'],
        ['IGHG2','IGHV3-7'],
        ['IGHG2','IGHV3-72'],
        ['IGHG2','IGHV3-73'],
        ['IGHG2','IGHV3-74'],
        ['IGHG2','IGHV4-39'],
        ['IGHG2','IGHV5-51'],
        ['IGHG2','IGHV6-1'],
        ['IGHG3','IGHG4'],
        ['IGHG3','IGHGP'],
        ['IGHG3','IGHM'],
        ['IGHG3','IGHV1-18'],
        ['IGHG3','IGHV1-2'],
        ['IGHG3','IGHV1-24'],
        ['IGHG3','IGHV1-46'],
        ['IGHG3','IGHV1-58'],
        ['IGHG3','IGHV1-69'],
        ['IGHG3','IGHV1-69-2'],
        ['IGHG3','IGHV1OR15-3'],
        ['IGHG3','IGHV1OR16-4'],
        ['IGHG3','IGHV2-26'],
        ['IGHG3','IGHV2-5'],
        ['IGHG3','IGHV2-70'],
        ['IGHG3','IGHV3-11'],
        ['IGHG3','IGHV3-13'],
        ['IGHG3','IGHV3-15'],
        ['IGHG3','IGHV3-20'],
        ['IGHG3','IGHV3-21'],
        ['IGHG3','IGHV3-23'],
        ['IGHG3','IGHV3-30'],
        ['IGHG3','IGHV3-43'],
        ['IGHG3','IGHV3-48'],
        ['IGHG3','IGHV3-49'],
        ['IGHG3','IGHV3-53'],
        ['IGHG3','IGHV3-64'],
        ['IGHG3','IGHV3-7'],
        ['IGHG3','IGHV3-72'],
        ['IGHG3','IGHV3-73'],
        ['IGHG3','IGHV3-74'],
        ['IGHG3','IGHV3OR16-9'],
        ['IGHG3','IGHV4-31'],
        ['IGHG3','IGHV4-34'],
        ['IGHG3','IGHV4-39'],
        ['IGHG3','IGHV4-59'],
        ['IGHG3','IGHV4-61'],
        ['IGHG3','IGHV5-51'],
        ['IGHG3','IGHV6-1'],
        ['IGHG3','IGK@'],
        ['IGHG3','IGKC'],
        ['IGHG3','IGKV1-8'],
        ['IGHG3','IGKV3-11'],
        ['IGHG3','RP11-731F5.2'],
        ['IGHG3','RPS8'],
        ['IGHG4','IGHGP'],
        ['IGHG4','IGHM'],
        ['IGHG4','IGHV3-49'],
        ['IGHG4','IGHV3-7'],
        ['IGHG4','IGHV4-34'],
        ['IGHG4','IGK@'],
        ['IGHG4','IGKC'],
        ['IGHG4','RP11-731F5.2'],
        ['IGHM','IGHV1-18'],
        ['IGHM','IGHV1-2'],
        ['IGHM','IGHV1-24'],
        ['IGHM','IGHV1-3'],
        ['IGHM','IGHV1-45'],
        ['IGHM','IGHV1-46'],
        ['IGHM','IGHV1-58'],
        ['IGHM','IGHV1-69'],
        ['IGHM','IGHV1-69-2'],
        ['IGHM','IGHV1-8'],
        ['IGHM','IGHV1OR16-4'],
        ['IGHM','IGHV2-26'],
        ['IGHM','IGHV2-5'],
        ['IGHM','IGHV2-70'],
        ['IGHM','IGHV3-11'],
        ['IGHM','IGHV3-13'],
        ['IGHM','IGHV3-15'],
        ['IGHM','IGHV3-19'],
        ['IGHM','IGHV3-20'],
        ['IGHM','IGHV3-21'],
        ['IGHM','IGHV3-23'],
        ['IGHM','IGHV3-30'],
        ['IGHM','IGHV3-33'],
        ['IGHM','IGHV3-41'],
        ['IGHM','IGHV3-43'],
        ['IGHM','IGHV3-48'],
        ['IGHM','IGHV3-49'],
        ['IGHM','IGHV3-53'],
        ['IGHM','IGHV3-64'],
        ['IGHM','IGHV3-66'],
        ['IGHM','IGHV3-7'],
        ['IGHM','IGHV3-71'],
        ['IGHM','IGHV3-72'],
        ['IGHM','IGHV3-73'],
        ['IGHM','IGHV3-74'],
        ['IGHM','IGHV3-9'],
        ['IGHM','IGHV3OR16-13'],
        ['IGHM','IGHV3OR16-15'],
        ['IGHM','IGHV3OR16-7'],
        ['IGHM','IGHV3OR16-9'],
        ['IGHM','IGHV4-28'],
        ['IGHM','IGHV4-31'],
        ['IGHM','IGHV4-34'],
        ['IGHM','IGHV4-39'],
        ['IGHM','IGHV4-4'],
        ['IGHM','IGHV4-59'],
        ['IGHM','IGHV4-61'],
        ['IGHM','IGHV5-51'],
        ['IGHM','IGHV6-1'],
        ['IGHM','IGK@'],
        ['IGHM','IGKC'],
        ['IGHM','IGKV1-8'],
        ['IGHM','IGKV3-11'],
        ['IGHM','MYH11'],
        ['IGHM','RP11-731F5.2'],
        ['IGHM','RPS8'],
        ['IGHM','SFTPA2'],
        ['IGHM','TPT1'],
        ['IGHV1-12','IGHV1-2'],
        ['IGHV1-12','IGHV1-46'],
        ['IGHV1-18','IGHV1-2'],
        ['IGHV1-18','IGHV1-69'],
        ['IGHV1-18','IGHV1OR15-2'],
        ['IGHV1-18','IGHV7-81'],
        ['IGHV1-2','IGHV1-24'],
        ['IGHV1-2','IGHV1-3'],
        ['IGHV1-2','IGHV1-46'],
        ['IGHV1-2','IGHV1OR15-1'],
        ['IGHV1-2','IGHV1OR15-3'],
        ['IGHV1-2','IGHV1OR16-4'],
        ['IGHV1-2','IGHV7-81'],
        ['IGHV1-3','IGHV1-46'],
        ['IGHV1-3','IGHV1OR15-3'],
        ['IGHV1-3','RP11-603B24.5'],
        ['IGHV1-46','IGHV1-67'],
        ['IGHV1-46','IGHV1-69'],
        ['IGHV1-46','IGHV1-69-2'],
        ['IGHV1-46','IGHV1OR16-4'],
        ['IGHV2-26','IGHV2-70'],
        ['IGHV2-26','IGHV2OR16-5'],
        ['IGHV2-5','IGHV2-70'],
        ['IGHV2-70','IGHV2-70'],
        ['IGHV2-70','IGHV3-21'],
        ['IGHV2-70','IGHV3-48'],
        ['IGHV3-11','IGHV3-48'],
        ['IGHV3-11','IGHV3-7'],
        ['IGHV3-11','IGHV3-71'],
        ['IGHV3-11','IGHV3-72'],
        ['IGHV3-11','IGHV3OR16-9'],
        ['IGHV3-15','IGHV3-43'],
        ['IGHV3-15','IGHV3-72'],
        ['IGHV3-15','IGHV3OR16-13'],
        ['IGHV3-15','IGHV3OR16-7'],
        ['IGHV3-15','RP11-731F5.2'],
        ['IGHV3-20','IGHV3-43'],
        ['IGHV3-20','IGHV3-49'],
        ['IGHV3-20','IGHV3-66'],
        ['IGHV3-20','IGHV3OR16-15'],
        ['IGHV3-21','IGHV3-23'],
        ['IGHV3-21','IGHV3-48'],
        ['IGHV3-21','IGHV3-7'],
        ['IGHV3-21','IGHV3OR16-9'],
        ['IGHV3-21','RP11-731F5.2'],
        ['IGHV3-23','IGHV3-30'],
        ['IGHV3-23','IGHV3-35'],
        ['IGHV3-23','IGHV3-38'],
        ['IGHV3-23','IGHV3-41'],
        ['IGHV3-23','IGHV3-43'],
        ['IGHV3-23','IGHV3-48'],
        ['IGHV3-23','IGHV3-49'],
        ['IGHV3-23','IGHV3-7'],
        ['IGHV3-23','IGHV3-74'],
        ['IGHV3-23','IGHV3OR16-13'],
        ['IGHV3-23','IGHV3OR16-9'],
        ['IGHV3-23','RP11-1166P10.8'],
        ['IGHV3-23','RP11-731F5.2'],
        ['IGHV3-30','IGHV3-49'],
        ['IGHV3-30','IGHV3-64'],
        ['IGHV3-30','IGHV3-66'],
        ['IGHV3-33','IGHV3-48'],
        ['IGHV3-38','IGHV3-53'],
        ['IGHV3-43','IGHV3-49'],
        ['IGHV3-43','IGHV3-53'],
        ['IGHV3-43','IGHV3-74'],
        ['IGHV3-43','IGHV3-9'],
        ['IGHV3-48','IGHV3-7'],
        ['IGHV3-48','IGHV3-72'],
        ['IGHV3-48','IGHV3OR16-9'],
        ['IGHV3-48','RP11-812E19.9'],
        ['IGHV3-49','IGHV3-52'],
        ['IGHV3-49','IGHV3-66'],
        ['IGHV3-49','IGHV3-72'],
        ['IGHV3-49','IGHV4-28'],
        ['IGHV3-49','IGHV4-31'],
        ['IGHV3-49','IGHV4-39'],
        ['IGHV3-49','IGHV5-51'],
        ['IGHV3-52','IGHV3-64'],
        ['IGHV3-52','IGHV3-74'],
        ['IGHV3-53','IGHV3-66'],
        ['IGHV3-7','IGHV3-72'],
        ['IGHV3-7','IGHV3-9'],
        ['IGHV3-7','IGHV3OR16-15'],
        ['IGHV3-7','IGHV3OR16-9'],
        ['IGHV3-7','RP11-731F5.2'],
        ['IGHV3-72','RP11-731F5.2'],
        ['IGHV3-74','RP11-731F5.2'],
        ['IGHV4-28','IGHV4-31'],
        ['IGHV4-28','IGHV4-34'],
        ['IGHV4-28','IGHV4-59'],
        ['IGHV4-28','IGHV4-61'],
        ['IGHV4-31','IGHV4-39'],
        ['IGHV4-31','IGHV4-61'],
        ['IGHV4-34','IGHV4-39'],
        ['IGHV4-39','IGHV4-61'],
        ['IGHV5-51','IGHV5-78'],
        ['IGHV5-51','RP11-731F5.2'],
        ['IGHV6-1','RP11-731F5.2'],
        ['IGK@','IGK@'],
        ['IGK@','IGKC'],
        ['IGK@','IGKV1-27'],
        ['IGK@','IGKV1-8'],
        ['IGK@','IGKV1D-43'],
        ['IGK@','IGKV1D-8'],
        ['IGK@','IGKV2-24'],
        ['IGK@','IGKV2D-24'],
        ['IGK@','IGKV2D-28'],
        ['IGK@','IGKV2D-29'],
        ['IGK@','IGKV2D-30'],
        ['IGK@','IGKV3-11'],
        ['IGK@','IGKV3-20'],
        ['IGK@','IGKV3-7'],
        ['IGK@','IGKV3D-11'],
        ['IGK@','IGKV3D-15'],
        ['IGK@','IGKV3D-20'],
        ['IGK@','IGKV6-21'],
        ['IGK@','LL22NC03-2H8.5'],
        ['IGK@','MALAT1'],
        ['IGK@','MUC5AC'],
        ['IGK@','MYH11'],
        ['IGK@','NFIB'],
        ['IGK@','RPS11'],
        ['IGK@','SYNPO2'],
        ['IGK@','TNS1'],
        ['IGK@','TPT1'],
        ['IGKC','IGKV1-12'],
        ['IGKC','IGKV1-16'],
        ['IGKC','IGKV1-17'],
        ['IGKC','IGKV1-27'],
        ['IGKC','IGKV1-5'],
        ['IGKC','IGKV1-6'],
        ['IGKC','IGKV1-8'],
        ['IGKC','IGKV1-9'],
        ['IGKC','IGKV1D-12'],
        ['IGKC','IGKV1D-13'],
        ['IGKC','IGKV1D-16'],
        ['IGKC','IGKV1D-17'],
        ['IGKC','IGKV1D-39'],
        ['IGKC','IGKV1D-43'],
        ['IGKC','IGKV1D-8'],
        ['IGKC','IGKV1OR2-108'],
        ['IGKC','IGKV2-24'],
        ['IGKC','IGKV2-26'],
        ['IGKC','IGKV2-29'],
        ['IGKC','IGKV2-30'],
        ['IGKC','IGKV2D-24'],
        ['IGKC','IGKV2D-26'],
        ['IGKC','IGKV2D-28'],
        ['IGKC','IGKV2D-29'],
        ['IGKC','IGKV2D-30'],
        ['IGKC','IGKV2D-40'],
        ['IGKC','IGKV2OR22-4'],
        ['IGKC','IGKV3-11'],
        ['IGKC','IGKV3-15'],
        ['IGKC','IGKV3-20'],
        ['IGKC','IGKV3-7'],
        ['IGKC','IGKV3D-11'],
        ['IGKC','IGKV3D-15'],
        ['IGKC','IGKV3D-20'],
        ['IGKC','IGKV3D-7'],
        ['IGKC','IGKV4-1'],
        ['IGKC','IGKV5-2'],
        ['IGKC','IGKV6-21'],
        ['IGKC','IGKV6D-21'],
        ['IGKC','LMNA'],
        ['IGKC','MUC1'],
        ['IGKC','MUC4'],
        ['IGKC','MUC5AC'],
        ['IGKC','MYH9'],
        ['IGKC','NEAT1'],
        ['IGKC','NFIB'],
        ['IGKC','NR4A1'],
        ['IGKC','PSAP'],
        ['IGKC','PTMS'],
        ['IGKC','RP11-1100L3.8'],
        ['IGKC','RP11-295K3.1'],
        ['IGKC','RPL3'],
        ['IGKC','SCGB1A1'],
        ['IGKC','SFTPA1'],
        ['IGKC','SFTPA2'],
        ['IGKC','SFTPB'],
        ['IGKC','SFTPC'],
        ['IGKC','SLC34A2'],
        ['IGKC','SPARC'],
        ['IGKC','SRRM2'],
        ['IGKC','TNS1'],
        ['IGKC','TPT1'],
        ['IGKC','VIM'],
        ['IGKC','ZFP36'],
        ['IGKV1-12','MUC5AC'],
        ['IGKV1-16','IGKV1-17'],
        ['IGKV1-16','IGKV1-8'],
        ['IGKV1-16','IGKV3-11'],
        ['IGKV1-17','IGKV1-6'],
        ['IGKV1-17','IGKV1-8'],
        ['IGKV1-17','IGKV3-11'],
        ['IGKV1-27','IGKV1-6'],
        ['IGKV1-27','IGKV1-8'],
        ['IGKV1-27','IGKV1-9'],
        ['IGKV1-27','IGKV1OR2-108'],
        ['IGKV1-27','IGKV3-11'],
        ['IGKV1-5','IGKV1-8'],
        ['IGKV1-5','IGKV1-9'],
        ['IGKV1-5','IGKV3-11'],
        ['IGKV1-6','IGKV1-8'],
        ['IGKV1-6','IGKV3-11'],
        ['IGKV1-8','IGKV1-9'],
        ['IGKV1-8','IGKV1D-13'],
        ['IGKV1-8','IGKV1D-16'],
        ['IGKV1-8','IGKV1D-17'],
        ['IGKV1-8','IGKV1D-39'],
        ['IGKV1-8','IGKV1D-43'],
        ['IGKV1-8','IGKV1D-8'],
        ['IGKV1-8','IGKV1OR2-108'],
        ['IGKV1-8','IGKV2-24'],
        ['IGKV1-8','IGKV2-26'],
        ['IGKV1-8','IGKV2-29'],
        ['IGKV1-8','IGKV2-30'],
        ['IGKV1-8','IGKV2D-24'],
        ['IGKV1-8','IGKV2D-26'],
        ['IGKV1-8','IGKV2D-28'],
        ['IGKV1-8','IGKV2D-29'],
        ['IGKV1-8','IGKV2D-30'],
        ['IGKV1-8','IGKV2D-40'],
        ['IGKV1-8','IGKV3-15'],
        ['IGKV1-8','IGKV3-20'],
        ['IGKV1-8','IGKV3-7'],
        ['IGKV1-8','IGKV3D-11'],
        ['IGKV1-8','IGKV3D-15'],
        ['IGKV1-8','IGKV3D-20'],
        ['IGKV1-8','IGKV3D-7'],
        ['IGKV1-8','IGKV4-1'],
        ['IGKV1-8','IGKV5-2'],
        ['IGKV1-8','IGKV6-21'],
        ['IGKV1-8','IGKV6D-21'],
        ['IGKV1-8','MUC5AC'],
        ['IGKV1-8','TPT1'],
        ['IGKV1-9','IGKV1D-8'],
        ['IGKV1-9','IGKV3-11'],
        ['IGKV1D-13','IGKV3-11'],
        ['IGKV1D-16','IGKV3-11'],
        ['IGKV1D-17','IGKV3-11'],
        ['IGKV1D-39','IGKV3-11'],
        ['IGKV1D-43','IGKV3-11'],
        ['IGKV1D-8','IGKV3-11'],
        ['IGKV1OR2-108','IGKV3-11'],
        ['IGKV2-24','IGKV2D-24'],
        ['IGKV2-24','IGKV3-11'],
        ['IGKV2-26','IGKV3-11'],
        ['IGKV2-29','IGKV2D-29'],
        ['IGKV2-29','IGKV3-11'],
        ['IGKV2-30','IGKV3-11'],
        ['IGKV2D-24','IGKV3-11'],
        ['IGKV2D-26','IGKV3-11'],
        ['IGKV2D-28','IGKV3-11'],
        ['IGKV2D-29','IGKV3-11'],
        ['IGKV2D-30','IGKV3-11'],
        ['IGKV2D-40','IGKV3-11'],
        ['IGKV3-11','IGKV3-15'],
        ['IGKV3-11','IGKV3-20'],
        ['IGKV3-11','IGKV3-7'],
        ['IGKV3-11','IGKV3D-11'],
        ['IGKV3-11','IGKV3D-15'],
        ['IGKV3-11','IGKV3D-20'],
        ['IGKV3-11','IGKV3D-7'],
        ['IGKV3-11','IGKV4-1'],
        ['IGKV3-11','IGKV5-2'],
        ['IGKV3-11','IGKV6-21'],
        ['IGKV3-11','IGKV6D-21'],
        ['IGKV3-11','TPT1'],
        ['IGKV3-15','IGKV3D-11'],
        ['IGKV3-20','IGKV3D-11'],
        ['IGKV3-20','IGKV3D-15'],
        ['IGKV3-20','IGKV3D-20'],
        ['IGKV3-7','IGKV3D-11'],
        ['IGLC1','IGLC2'],
        ['IGLC1','IGLC3'],
        ['IGLC1','IGLC7'],
        ['IGLC1','IGLV1-36'],
        ['IGLC1','IGLV1-40'],
        ['IGLC1','IGLV1-41'],
        ['IGLC1','IGLV1-44'],
        ['IGLC1','IGLV1-47'],
        ['IGLC1','IGLV1-50'],
        ['IGLC1','IGLV1-51'],
        ['IGLC1','IGLV10-54'],
        ['IGLC1','IGLV2-11'],
        ['IGLC1','IGLV2-14'],
        ['IGLC1','IGLV2-18'],
        ['IGLC1','IGLV2-23'],
        ['IGLC1','IGLV2-28'],
        ['IGLC1','IGLV2-34'],
        ['IGLC1','IGLV2-8'],
        ['IGLC1','IGLV3-1'],
        ['IGLC1','IGLV3-10'],
        ['IGLC1','IGLV3-19'],
        ['IGLC1','IGLV3-21'],
        ['IGLC1','IGLV3-25'],
        ['IGLC1','IGLV3-27'],
        ['IGLC1','IGLV3-9'],
        ['IGLC1','IGLV4-60'],
        ['IGLC1','IGLV4-69'],
        ['IGLC1','IGLV5-45'],
        ['IGLC1','IGLV6-57'],
        ['IGLC1','IGLV7-43'],
        ['IGLC1','IGLV7-46'],
        ['IGLC1','IGLV8-61'],
        ['IGLC1','IGLV9-49'],
        ['IGLC1','LL22NC03-80A10.6'],
        ['IGLC2','IGLC3'],
        ['IGLC2','IGLC6'],
        ['IGLC2','IGLL5'],
        ['IGLC2','IGLV1-40'],
        ['IGLC2','IGLV1-47'],
        ['IGLC2','IGLV1-51'],
        ['IGLC2','IGLV10-54'],
        ['IGLC2','IGLV2-11'],
        ['IGLC2','IGLV2-14'],
        ['IGLC2','IGLV2-23'],
        ['IGLC2','IGLV2-28'],
        ['IGLC2','IGLV2-8'],
        ['IGLC2','IGLV3-1'],
        ['IGLC2','IGLV3-10'],
        ['IGLC2','IGLV3-19'],
        ['IGLC2','IGLV3-21'],
        ['IGLC2','IGLV3-25'],
        ['IGLC2','IGLV3-27'],
        ['IGLC2','IGLV4-60'],
        ['IGLC2','IGLV4-69'],
        ['IGLC2','IGLV5-45'],
        ['IGLC2','IGLV6-57'],
        ['IGLC2','IGLV7-43'],
        ['IGLC2','IGLV7-46'],
        ['IGLC2','IGLV8-61'],
        ['IGLC2','IGLV9-49'],
        ['IGLC3','IGLL5'],
        ['IGLC5','IGLL5'],
        ['IGLC7','IGLJ7'],
        ['IGLC7','IGLL5'],
        ['IGLC7','IGLV1-47'],
        ['IGLC7','IGLV3-21'],
        ['IGLC7','IGLV6-57'],
        ['IGLL1','IGLL5'],
        ['IGLL3P','KB-1572G7.3'],
        ['IGLL5','IGLV1-36'],
        ['IGLL5','IGLV1-40'],
        ['IGLL5','IGLV1-41'],
        ['IGLL5','IGLV1-44'],
        ['IGLL5','IGLV1-47'],
        ['IGLL5','IGLV1-50'],
        ['IGLL5','IGLV1-51'],
        ['IGLL5','IGLV10-54'],
        ['IGLL5','IGLV2-11'],
        ['IGLL5','IGLV2-14'],
        ['IGLL5','IGLV2-18'],
        ['IGLL5','IGLV2-23'],
        ['IGLL5','IGLV2-28'],
        ['IGLL5','IGLV2-34'],
        ['IGLL5','IGLV2-8'],
        ['IGLL5','IGLV3-1'],
        ['IGLL5','IGLV3-10'],
        ['IGLL5','IGLV3-19'],
        ['IGLL5','IGLV3-21'],
        ['IGLL5','IGLV3-25'],
        ['IGLL5','IGLV3-27'],
        ['IGLL5','IGLV3-9'],
        ['IGLL5','IGLV4-60'],
        ['IGLL5','IGLV4-69'],
        ['IGLL5','IGLV5-37'],
        ['IGLL5','IGLV5-45'],
        ['IGLL5','IGLV6-57'],
        ['IGLL5','IGLV7-43'],
        ['IGLL5','IGLV7-46'],
        ['IGLL5','IGLV8-61'],
        ['IGLL5','IGLV9-49'],
        ['IGLL5','LL22NC03-80A10.6'],
        ['IGLV1-41','IGLV1-51'],
        ['IGLV1-50','IGLV1-51'],
        ['IGLV2-11','IGLV2-14'],
        ['IGLV2-11','IGLV2-18'],
        ['IGLV2-11','IGLV2-23'],
        ['IGLV2-11','IGLV2-8'],
        ['IGLV2-14','IGLV2-18'],
        ['IGLV2-14','IGLV2-23'],
        ['IGLV2-23','IGLV2-8'],
        ['IGLV3-1','IGLV3-21'],
        ['IGLV3-10','IGLV3-25'],
        ['IGLV3-16','IGLV3-25'],
        ['IGLV3-21','IGLV3-9'],
        ['IGLV7-43','IGLV7-46'],
        ['IGSF5','PCP4'],
        ['IGSF9B','RP11-259P6.1'],
        ['IL11','PTRF'],
        ['IL18BP','SFTPC'],
        ['IL1R1','SFTPC'],
        ['IL1RL1','SFTPA2'],
        ['IL1RL1','SFTPC'],
        ['IL20RB','NCK1'],
        ['IL32','SFTPB'],
        ['IL4I1','NUP62'],
        ['IL4I1','SIGLEC11'],
        ['IL6R','LYZ'],
        ['IL6R','SFTPC'],
        ['IL6ST','IL6STP1'],
        ['IL6ST','SFTPC'],
        ['INE1','UBA1'],
        ['INF2','RP11-982M15.6'],
        ['INF2','SFTPC'],
        ['INHBA','THBS1'],
        ['INHBE','LRG1'],
        ['INMT','SFTPB'],
        ['INMT','SFTPC'],
        ['INO80C','RP11-322E11.5'],
        ['INO80C','RP11-322E11.6'],
        ['INPP5B','SFTPB'],
        ['INPP5J','TPO'],
        ['INPP5K','MYO1C'],
        ['INPPL1','NT5DC2'],
        ['INPPL1','UBA1'],
        ['INSR','SEMG1'],
        ['INTS1','SFTPA2'],
        ['INTS1','SFTPC'],
        ['INTS12','RP11-45L9.1'],
        ['INTS3','SFTPC'],
        ['INTS3','SLC27A3'],
        ['INTS6-AS1','RPS4XP16'],
        ['INVS','METAZOA_SRP'],
        ['INVS','RN7SL75P'],
        ['IPP','RP11-767N6.7'],
        ['IQCF1','IQCF5'],
        ['IQGAP1','SFTPC'],
        ['IQGAP1','VIM'],
        ['IQSEC1','SFTPA1'],
        ['IQSEC1','SFTPC'],
        ['IRAK1','TG'],
        ['IRAK1','USP22'],
        ['IRF1','SFTPC'],
        ['IRF2BP2','SFTPA2'],
        ['IRF2BP2','SFTPC'],
        ['IRGQ','PTRF'],
        ['IRS1','SFTPC'],
        ['IRS1','TG'],
        ['ISG20','SFTPC'],
        ['IST1','IST1'],
        ['ITFG3','SFTPA1'],
        ['ITGA11','VIM'],
        ['ITGA3','MYH9'],
        ['ITGA3','QSOX1'],
        ['ITGA3','SFTPA1'],
        ['ITGA3','SFTPA2'],
        ['ITGA3','SFTPC'],
        ['ITGA3','TAGLN'],
        ['ITGA5','LGALS1'],
        ['ITGA5','MAZ'],
        ['ITGA5','MYH9'],
        ['ITGA5','NR2F2'],
        ['ITGA5','PTOV1'],
        ['ITGA5','PTRF'],
        ['ITGA5','SFTPA1'],
        ['ITGA5','TAGLN'],
        ['ITGA5','TGFBI'],
        ['ITGA5','THBS1'],
        ['ITGA5','VIM'],
        ['ITGA5','VPS37B'],
        ['ITGA5','ZFP36L2'],
        ['ITGAV','PFN1'],
        ['ITGAX','SFTPB'],
        ['ITGB1','PLEC'],
        ['ITGB1','SFTPC'],
        ['ITGB1','TFPI2'],
        ['ITGB1','TG'],
        ['ITGB1','THBS1'],
        ['ITGB1','TIMP3'],
        ['ITGB2','SFTPC'],
        ['ITIH4','MUSTN1'],
        ['ITIH4','RP5-966M1.6'],
        ['ITIH4','SERPINA1'],
        ['ITIH4','TMEM110-MUSTN1'],
        ['ITLN1','SFTPA1'],
        ['ITM2B','SFTPA1'],
        ['ITM2B','SFTPB'],
        ['ITM2B','SFTPC'],
        ['ITPKC','SFTPC'],
        ['ITPR3','SFTPA2'],
        ['ITPR3','SFTPB'],
        ['ITPR3','SFTPC'],
        ['ITPRIP','SFTPC'],
        ['ITPRIPL2','PTRF'],
        ['IWS1','NCL'],
        ['IWS1','SDF4'],
        ['IYD','TG'],
        ['IYD','TTN'],
        ['JAK1','LINC01359'],
        ['JAK1','MYH9'],
        ['JAK1','RP11-182I10.3'],
        ['JAK1','RP4-535B20.1'],
        ['JDP2','SFTPC'],
        ['JMJD8','MYH9'],
        ['JMJD8','STUB1'],
        ['JMJD8','TIMP2'],
        ['JUN','SFTPA1'],
        ['JUN','SFTPA2'],
        ['JUN','SFTPC'],
        ['JUN','TG'],
        ['JUNB','SFTPA1'],
        ['JUNB','SFTPA2'],
        ['JUNB','SFTPB'],
        ['JUNB','SFTPC'],
        ['JUNB','SH3BGRL3'],
        ['JUND','NPAS1'],
        ['JUND','PTMS'],
        ['JUND','SERPINE1'],
        ['JUND','SFTPA1'],
        ['JUND','SFTPA2'],
        ['JUND','SFTPB'],
        ['JUND','SFTPC'],
        ['JUND','TRIM28'],
        ['JUND','VIM'],
        ['JUP','KRT10'],
        ['JUP','SFTPA1'],
        ['JUP','SFTPA2'],
        ['JUP','TG'],
        ['KAL1','SFTPC'],
        ['KANK1','RP11-31F19.1'],
        ['KANK2','SFTPB'],
        ['KANK2','SFTPC'],
        ['KANSL3','SPN'],
        ['KAT6A','SFTPC'],
        ['KAT7','RP5-961K14.1'],
        ['KB-1440D3.13','PPIL2'],
        ['KBTBD4','SFTPC'],
        ['KCNA7','NTF4'],
        ['KCNK15','RP4-781B1.5'],
        ['KCNMA1','PTMS'],
        ['KCNQ3','TREML2'],
        ['KCNQ5','KCNQ5-IT1'],
        ['KCTD10','RP11-256L11.3'],
        ['KCTD12','SFTPA2'],
        ['KCTD12','SFTPC'],
        ['KDELR2','SERPINE1'],
        ['KDELR2','VIM'],
        ['KDM3B','SFTPC'],
        ['KDM4B','SFTPA1'],
        ['KDM5C','KDM5C-IT1'],
        ['KDM5C','SFTPA2'],
        ['KDM5C','SFTPC'],
        ['KDM6B','SFTPC'],
        ['KDM6B','TMEM88'],
        ['KHNYN','SFTPC'],
        ['KHSRP','NAT14'],
        ['KHSRP','SFTPC'],
        ['KHSRP','TRIM28'],
        ['KIAA0100','SFTPC'],
        ['KIAA0195','SFTPA2'],
        ['KIAA0319L','SFTPA2'],
        ['KIAA0754','MACF1'],
        ['KIAA1147','RP5-894A10.2'],
        ['KIAA1147','RP5-894A10.6'],
        ['KIAA1191','PRKCSH'],
        ['KIAA1191','SCUBE3'],
        ['KIAA1199','LAMB2'],
        ['KIAA1199','LRP1'],
        ['KIAA1199','P4HB'],
        ['KIAA1199','PIEZO1'],
        ['KIAA1199','PLEC'],
        ['KIAA1199','SERPINE1'],
        ['KIAA1199','SERPINE2'],
        ['KIAA1199','SPARC'],
        ['KIAA1199','THBS1'],
        ['KIAA1199','VIM'],
        ['KIAA1217','SFTPC'],
        ['KIAA1257','RP11-723O4.6'],
        ['KIAA1462','SFTPB'],
        ['KIAA1462','SFTPC'],
        ['KIAA1586','ZNF451'],
        ['KIAA1614-AS1','STX6'],
        ['KIAA1731','SCARNA9'],
        ['KIAA1984','RABL6'],
        ['KIAA1984','TMEM141'],
        ['KIAA2013','RP11-680G24.4'],
        ['KIF18B','PTRF'],
        ['KIF1C','PTRF'],
        ['KIF1C','SFTPA2'],
        ['KIF1C','SFTPB'],
        ['KIF1C','SFTPC'],
        ['KIF1C','TG'],
        ['KIF22','MCM3AP'],
        ['KIF27','RP11-213G2.3'],
        ['KIF5A','PCNT'],
        ['KIT','SFTPC'],
        ['KLF10','SFTPC'],
        ['KLF13','SFTPA2'],
        ['KLF13','SFTPB'],
        ['KLF13','SFTPC'],
        ['KLF16','REXO1'],
        ['KLF17P2','LINC01085'],
        ['KLF2','LOXL1'],
        ['KLF2','MARCKS'],
        ['KLF2','MAZ'],
        ['KLF2','ORAI2'],
        ['KLF2','SFTPA2'],
        ['KLF2','TRIM28'],
        ['KLF2','UBE2M'],
        ['KLF3','PNLIP'],
        ['KLF4','SFTPC'],
        ['KLF6','SFTPA2'],
        ['KLF6','SFTPC'],
        ['KLF9','SFTPC'],
        ['KLHDC3','RRP36'],
        ['KLHL22','SCARF2'],
        ['KLHL23','SSB'],
        ['KLHL3','RP11-381K20.2'],
        ['KLHL41','TG'],
        ['KLHL5','RP11-360F5.3'],
        ['KLK4','KLKP1'],
        ['KLRC2','KLRC3'],
        ['KLRC3','RP11-277P12.6'],
        ['KMT2D','PRKAG1'],
        ['KPNA2','RP13-104F24.1'],
        ['KPNA2','RP13-104F24.2'],
        ['KPNB1','SFTPC'],
        ['KRBOX1','ZNF662'],
        ['KRT1','KRT10'],
        ['KRT1','KRT14'],
        ['KRT1','KRT2'],
        ['KRT1','KRT5'],
        ['KRT1','NEAT1'],
        ['KRT1','PKP1'],
        ['KRT1','ZFP36L1'],
        ['KRT10','KRT14'],
        ['KRT10','KRT2'],
        ['KRT10','KRT5'],
        ['KRT10','NDUFS7'],
        ['KRT10','NEAT1'],
        ['KRT10','PKP1'],
        ['KRT10','RGMB'],
        ['KRT10','SPINK5'],
        ['KRT13','TIMP2'],
        ['KRT14','KRT16P4'],
        ['KRT14','KRT19'],
        ['KRT14','KRT2'],
        ['KRT14','KRT5'],
        ['KRT14','KRT6A'],
        ['KRT14','MALAT1'],
        ['KRT14','NEAT1'],
        ['KRT14','TRIM29'],
        ['KRT15','KRT5'],
        ['KRT16','KRT5'],
        ['KRT17','MALAT1'],
        ['KRT19','KRT19P1'],
        ['KRT19','SFTPC'],
        ['KRT19','VIM'],
        ['KRT2','KRT5'],
        ['KRT222','SMARCE1'],
        ['KRT4','SVIL'],
        ['KRT4','TGM3'],
        ['KRT5','KRT6A'],
        ['KRT5','PERP'],
        ['KRT5','S100A6'],
        ['KRT6A','KRT6C'],
        ['KRT6A','KRT80'],
        ['KRT7','KRT86'],
        ['KRT7','SFTPA2'],
        ['KRT7','SFTPB'],
        ['KRT7','SFTPC'],
        ['KRT8','KRT8P11'],
        ['KRT8','SFTPC'],
        ['KRTCAP2','MUC1'],
        ['KSR1','LGALS9'],
        ['KSR1','TRB@'],
        ['KTN1','SFTPB'],
        ['LA16C-313D11.10','STUB1'],
        ['LACC1','LINC00284'],
        ['LAIR1','LAIR2'],
        ['LAIR1','LYZ'],
        ['LAMA3','SFTPB'],
        ['LAMA5','RP11-157P1.4'],
        ['LAMA5','SFTPA2'],
        ['LAMA5','SFTPB'],
        ['LAMA5','SFTPC'],
        ['LAMA5','SLC34A2'],
        ['LAMB1','SFTPA2'],
        ['LAMB1','SFTPC'],
        ['LAMB2','MYH9'],
        ['LAMB2','SFTPA2'],
        ['LAMB2','SFTPC'],
        ['LAMB2','THBS1'],
        ['LAMB2','VIM'],
        ['LAMB3','SFTPC'],
        ['LAMC1','PPP1R14B'],
        ['LAMC1','SFTPC'],
        ['LAMC2','SFTPA2'],
        ['LAMP1','MAMDC2'],
        ['LAMP1','SFTPC'],
        ['LAMP3','RP11-186B7.4'],
        ['LAMP3','SFTPA1'],
        ['LAMP3','SFTPA2'],
        ['LAMP3','SFTPB'],
        ['LAMP3','SFTPC'],
        ['LAMP3','SLC22A5'],
        ['LAMTOR5-AS1','PROK1'],
        ['LAMTOR5-AS1','RP5-1074L1.4'],
        ['LAP3','LAP3P2'],
        ['LAPTM4A','SFTPB'],
        ['LAPTM4A','SFTPC'],
        ['LAPTM4B','RP11-21I10.2'],
        ['LAPTM5','SFTPB'],
        ['LAPTM5','SFTPC'],
        ['LASP1','LINC00672'],
        ['LASP1','MYH9'],
        ['LASP1','PCSK7'],
        ['LASP1','PLEC'],
        ['LASP1','SERPINE1'],
        ['LASP1','SFTPA2'],
        ['LASP1','SFTPC'],
        ['LASP1','TAGLN'],
        ['LATS2','SFTPC'],
        ['LBH','SFTPA2'],
        ['LBH','SFTPC'],
        ['LBH','TG'],
        ['LBHD1','UBXN1'],
        ['LBP','RPN2'],
        ['LBX2','PCGF1'],
        ['LCAT','PSMB10'],
        ['LCAT','SLC12A4'],
        ['LCE1D','LCE1E'],
        ['LCP1','SFTPA2'],
        ['LCP1','SFTPC'],
        ['LCP2','SFTPA1'],
        ['LDB3','OPN4'],
        ['LDHA','LDHAP2'],
        ['LDHA','LDHAP3'],
        ['LDHA','LDHAP4'],
        ['LDHA','LDHAP5'],
        ['LDHA','LDHAP7'],
        ['LDHB','LDHBP2'],
        ['LDLR','LYZ'],
        ['LDLR','SFTPC'],
        ['LDLR','TAPBP'],
        ['LDLRAP1','RP11-70P17.1'],
        ['LEAP2','UQCRQ'],
        ['LEFTY2','RP4-559A3.6'],
        ['LEMD3','RP11-305O6.4'],
        ['LENG8','SFTPA1'],
        ['LENG8','SFTPA2'],
        ['LENG8','SFTPB'],
        ['LENG8','SFTPC'],
        ['LEP','SPTBN1'],
        ['LEPRE1','SOX12'],
        ['LEPRE1','TRNP1'],
        ['LEPREL1','NCOR2'],
        ['LEPREL1','PTRF'],
        ['LEPREL4','MRPL49'],
        ['LEPREL4','SCUBE3'],
        ['LETM1','PTRF'],
        ['LETM2','RP11-350N15.5'],
        ['LFNG','MIR4648'],
        ['LGALS1','LMNA'],
        ['LGALS1','MYH9'],
        ['LGALS1','MYL9'],
        ['LGALS1','P4HB'],
        ['LGALS1','PDXP'],
        ['LGALS1','SERPINE1'],
        ['LGALS1','SFTPC'],
        ['LGALS1','SPARC'],
        ['LGALS1','THBS1'],
        ['LGALS1','TLN1'],
        ['LGALS1','VIM'],
        ['LGALS1','VIM-AS1'],
        ['LGALS2','TMPRSS2'],
        ['LGALS3','SFTPC'],
        ['LGALS3BP','SFTPA1'],
        ['LGALS3BP','SFTPB'],
        ['LGALS3BP','SFTPC'],
        ['LGALS9B','LGALS9C'],
        ['LGI1','RP11-437J2.3'],
        ['LGI3','SFTPC'],
        ['LGMN','LGMNP1'],
        ['LHFPL3','METAZOA_SRP'],
        ['LIG3','RP5-837J1.2'],
        ['LILRA1','LILRA2'],
        ['LILRA1','LILRB1'],
        ['LILRA1','LILRB2'],
        ['LILRA2','LILRB1'],
        ['LILRA4','LILRA6'],
        ['LILRA4','LILRB2'],
        ['LILRA4','LILRB3'],
        ['LILRA6','LILRB1'],
        ['LILRA6','LILRB2'],
        ['LILRA6','LILRB3'],
        ['LILRA6','LILRB4'],
        ['LILRA6','LILRB5'],
        ['LILRB1','LILRB2'],
        ['LILRB1','LILRB3'],
        ['LILRB1','LILRB4'],
        ['LILRB1','LILRP1'],
        ['LILRB1','LILRP2'],
        ['LILRB2','LILRB3'],
        ['LILRB2','LILRB5'],
        ['LILRB3','LILRB4'],
        ['LILRB3','LILRB5'],
        ['LILRB4','LILRB5'],
        ['LIMCH1','SFTPA1'],
        ['LIMCH1','SFTPC'],
        ['LIMD1','SFTPC'],
        ['LIME1','SLC2A4RG'],
        ['LINC00094','PTRF'],
        ['LINC00152','MIR4435-1HG'],
        ['LINC00152','MIR4435-2HG'],
        ['LINC00240','TRNAI2'],
        ['LINC00309','PELI1'],
        ['LINC00312','LMCD1'],
        ['LINC00338','SEC14L1'],
        ['LINC00484','RP11-305L7.6'],
        ['LINC00578','RP11-114M1.1'],
        ['LINC00594','RP11-64K12.10'],
        ['LINC00595','LINC00856'],
        ['LINC00612','PZP'],
        ['LINC00630','LL0XNC01-237H1.2'],
        ['LINC00643','RP11-355I22.6'],
        ['LINC00643','SYT16'],
        ['LINC00657','SFTPC'],
        ['LINC00665','LINC01535'],
        ['LINC00665','ZFP14'],
        ['LINC00670','MYOCD'],
        ['LINC00694','ZNF445'],
        ['LINC00856','RP11-90J7.4'],
        ['LINC00963','PTRF'],
        ['LINC00963','RGMA'],
        ['LINC00963','RP11-492E3.51'],
        ['LINC00968','PENK'],
        ['LINC00969','MIR570'],
        ['LINC00969','SDHA'],
        ['LINC00969','SDHAP1'],
        ['LINC00969','SDHAP3'],
        ['LINC01089','RHOF'],
        ['LINC01115','TMEM18'],
        ['LINC01119','SOCS5'],
        ['LINC01251','UBE2R2-AS1'],
        ['LINC01513','ROPN1L'],
        ['LINC01578','RP11-437B10.1'],
        ['LINGO4','RORC'],
        ['LIPG','TG'],
        ['LIX1L','PTRF'],
        ['LL0XNC01-131B10.2','NSA2'],
        ['LL0XNC01-16G2.1','PNMA3'],
        ['LLGL1','MYO1C'],
        ['LLGL1','PTRF'],
        ['LMAN2','MXD3'],
        ['LMNA','MYH9'],
        ['LMNA','NEAT1'],
        ['LMNA','P4HB'],
        ['LMNA','PKM'],
        ['LMNA','PLEC'],
        ['LMNA','PNLIP'],
        ['LMNA','PSAP'],
        ['LMNA','PTRF'],
        ['LMNA','SEMA4A'],
        ['LMNA','SEPT9'],
        ['LMNA','SFTPA1'],
        ['LMNA','SFTPA2'],
        ['LMNA','SFTPB'],
        ['LMNA','SFTPC'],
        ['LMNA','SLC10A3'],
        ['LMNA','SPARC'],
        ['LMNA','TAGLN'],
        ['LMNA','TGFBI'],
        ['LMNA','THBS1'],
        ['LMNA','TIMP1'],
        ['LMNA','TLN1'],
        ['LMNA','TPM2'],
        ['LMNA','VIM'],
        ['LMNA','WIZ'],
        ['LMNB2','THBS1'],
        ['LMO7','RP11-29G8.3'],
        ['LMO7','SFTPA2'],
        ['LMO7','SFTPB'],
        ['LMO7','SFTPC'],
        ['LMOD2','MYH7'],
        ['LMOD2','TNNT2'],
        ['LONP1','SCAF1'],
        ['LOX','MYH9'],
        ['LOX','SH3GL1'],
        ['LOX','SPARC'],
        ['LOX','TAGLN'],
        ['LOX','THBS1'],
        ['LOX','TSR3'],
        ['LOX','VIM'],
        ['LOX','VIM-AS1'],
        ['LOXL1','LOXL1-AS1'],
        ['LOXL1','MAFF'],
        ['LOXL1','MARVELD1'],
        ['LOXL1','OAF'],
        ['LOXL1','RAB11FIP3'],
        ['LOXL1','SPARC'],
        ['LOXL1','UBE2H'],
        ['LOXL1','ZFP36L2'],
        ['LOXL2','MRC2'],
        ['LOXL2','MYH9'],
        ['LOXL2','PCSK7'],
        ['LOXL2','PKN1'],
        ['LOXL2','PLEC'],
        ['LOXL2','PTRF'],
        ['LOXL2','SERPINE1'],
        ['LOXL2','SPARC'],
        ['LOXL2','TGM2'],
        ['LOXL2','THBS1'],
        ['LOXL2','TLN1'],
        ['LOXL2','VIM'],
        ['LPA','PLG'],
        ['LPCAT1','SFTPA1'],
        ['LPCAT1','SFTPA2'],
        ['LPCAT1','SFTPB'],
        ['LPCAT1','SFTPC'],
        ['LPCAT3','LRG1'],
        ['LPHN2','SFTPC'],
        ['LPL','SCD'],
        ['LPL','VIM'],
        ['LPL','ZFP36L1'],
        ['LPO','PRH2'],
        ['LRCH3','PTRF'],
        ['LRCH4','SFTPA2'],
        ['LRCH4','SFTPC'],
        ['LRG1','SAA2'],
        ['LRG1','SERINC3'],
        ['LRG1','TAPBP'],
        ['LRG1','TAT'],
        ['LRIG1','SFTPC'],
        ['LRP1','LTBP4'],
        ['LRP1','MMP1'],
        ['LRP1','MTCH1'],
        ['LRP1','MYH9'],
        ['LRP1','PECAM1'],
        ['LRP1','PLEC'],
        ['LRP1','SFTPA1'],
        ['LRP1','SFTPA2'],
        ['LRP1','SFTPB'],
        ['LRP1','SFTPC'],
        ['LRP1','SLC34A2'],
        ['LRP1','SPARC'],
        ['LRP1','TAGLN'],
        ['LRP1','TGFBI'],
        ['LRP1','TGM2'],
        ['LRP1','THBS1'],
        ['LRP1','TIMP1'],
        ['LRP1','TNC'],
        ['LRP1','VIM'],
        ['LRP1','VIM-AS1'],
        ['LRP1','VWF'],
        ['LRP10','SERPINE1'],
        ['LRP10','SFTPA2'],
        ['LRP10','SFTPB'],
        ['LRP10','SFTPC'],
        ['LRP10','SPARC'],
        ['LRP10','TG'],
        ['LRP11','RP11-350J20.5'],
        ['LRP11','TTYH3'],
        ['LRR1','RHOQ'],
        ['LRRC25','SFTPB'],
        ['LRRC32','SFTPB'],
        ['LRRC32','SFTPC'],
        ['LRRC37A15P','MANBA'],
        ['LRRC37A17P','LRRC37A4P'],
        ['LRRC47','PTRF'],
        ['LRRC59','MYH9'],
        ['LRRC59','PKM'],
        ['LRRC59','VIM-AS1'],
        ['LRRC69','OTUD6B'],
        ['LRRC75A','METTL7A'],
        ['LRRC75A-AS1','LYZ'],
        ['LRRFIP1','LRRFIP1P1'],
        ['LRRK2','SFTPC'],
        ['LSAMP','RP11-384F7.2'],
        ['LSM10','STK40'],
        ['LSM7','SERPINE1'],
        ['LSP1','SFTPB'],
        ['LTB4R2','NOP9'],
        ['LTBP1','SFTPB'],
        ['LTBP2','MYH9'],
        ['LTBP2','SERPINE1'],
        ['LTBP2','SFTPA1'],
        ['LTBP2','SFTPA2'],
        ['LTBP2','SFTPB'],
        ['LTBP2','SFTPC'],
        ['LTBP2','THBS1'],
        ['LTBP2','VIM'],
        ['LTBP3','QSOX1'],
        ['LTBP3','SFTPA1'],
        ['LTBP3','SFTPC'],
        ['LTBP3','TG'],
        ['LTBP4','MYH9'],
        ['LTBP4','SFTPA1'],
        ['LTBP4','SFTPA2'],
        ['LTBP4','SFTPB'],
        ['LTBP4','SFTPC'],
        ['LTBP4','ZFP36'],
        ['LTC4S','MAML1'],
        ['LUM','SFTPA2'],
        ['LUM','SFTPC'],
        ['LY6E','SFTPC'],
        ['LY6G5C','XXBAC-BPG32J3.20'],
        ['LYPD2','SLURP1'],
        ['LYST','LYZ'],
        ['LYZ','MROH1'],
        ['LYZ','MRPS30'],
        ['LYZ','MYADM'],
        ['LYZ','MYH11'],
        ['LYZ','MYO1C'],
        ['LYZ','OSBPL2'],
        ['LYZ','PLEKHM2'],
        ['LYZ','PRR4'],
        ['LYZ','PSMB5'],
        ['LYZ','PTRF'],
        ['LYZ','RP11-186B7.4'],
        ['LYZ','RPL4'],
        ['LYZ','RPL8'],
        ['LYZ','RSL1D1'],
        ['LYZ','SAMHD1'],
        ['LYZ','SFTPA2'],
        ['LYZ','SFTPB'],
        ['LYZ','SFTPC'],
        ['LYZ','SMG1'],
        ['LYZ','SNORA67'],
        ['LYZ','SPARC'],
        ['LYZ','TIMP2'],
        ['LYZ','TNFRSF1B'],
        ['LYZ','UBE2G2'],
        ['LYZ','WDR82'],
        ['LYZ','YIPF5'],
        ['LYZ','ZG16B'],
        ['LZTS1','PTRF'],
        ['LZTS2','PTMS'],
        ['MACC1-AS1','RP11-486P11.1'],
        ['MACF1','MYH9'],
        ['MACF1','NCOR2'],
        ['MACF1','SFTPA1'],
        ['MACF1','SFTPA2'],
        ['MACF1','SFTPB'],
        ['MACF1','SFTPC'],
        ['MACROD1','RP11-21A7A.3'],
        ['MAF1','PCDHGA1'],
        ['MAF1','PCDHGA10'],
        ['MAF1','PCDHGA12'],
        ['MAF1','PCDHGA2'],
        ['MAF1','PCDHGA3'],
        ['MAF1','PCDHGA4'],
        ['MAF1','PCDHGA5'],
        ['MAF1','PCDHGA6'],
        ['MAF1','PCDHGA7'],
        ['MAF1','PCDHGA8'],
        ['MAF1','PCDHGA9'],
        ['MAF1','PCDHGB1'],
        ['MAF1','PCDHGB2'],
        ['MAF1','PCDHGB3'],
        ['MAF1','PCDHGB4'],
        ['MAF1','PCDHGB6'],
        ['MAF1','PCDHGB7'],
        ['MAF1','PCDHGC3'],
        ['MAF1','PCDHGC4'],
        ['MAF1','PCDHGC5'],
        ['MAFG','SIRT7'],
        ['MAGED1','MYH9'],
        ['MAGED1','THBS1'],
        ['MALAT1','MAP1B'],
        ['MALAT1','MAP3K12'],
        ['MALAT1','MUC5AC'],
        ['MALAT1','MYH9'],
        ['MALAT1','MYL2'],
        ['MALAT1','NEAT1'],
        ['MALAT1','NEFM'],
        ['MALAT1','PSAP'],
        ['MALAT1','PSMD4'],
        ['MALAT1','SCD'],
        ['MALAT1','SFTPA1'],
        ['MALAT1','SFTPA2'],
        ['MALAT1','SFTPB'],
        ['MALAT1','SFTPC'],
        ['MALAT1','SLC34A2'],
        ['MALAT1','SPINK5'],
        ['MALAT1','SYNPO'],
        ['MALAT1','TG'],
        ['MALAT1','TNN'],
        ['MALAT1','TNS1'],
        ['MALAT1','TPO'],
        ['MALAT1','VWF'],
        ['MALAT1','WDR74'],
        ['MAMDC4','PHPT1'],
        ['MAN2A2','SFTPC'],
        ['MAN2B1','SFTPC'],
        ['MAOA','TG'],
        ['MAP1A','VIM'],
        ['MAP1B','MYH9'],
        ['MAP1LC3B','MAP1LC3B2'],
        ['MAP2K3','RP11-486O12.2'],
        ['MAP2K3','TRIM28'],
        ['MAP4','MYH9'],
        ['MAP4','PKM'],
        ['MAP4','SFTPA2'],
        ['MAP4','SFTPC'],
        ['MAP4','SPARC'],
        ['MAP4','TG'],
        ['MAP4','TGFBI'],
        ['MAP4','VIM'],
        ['MAP4K2','SFTPC'],
        ['MAP6D1','PARL'],
        ['MAP7D1','NCS1'],
        ['MAP7D1','PLEC'],
        ['MAP7D1','PRELID1'],
        ['MAP7D1','RCN3'],
        ['MAP7D1','TG'],
        ['MAP7D1','THBS1'],
        ['MAP7D1','TRIM28'],
        ['MAP7D1','WHSC1L1'],
        ['MAP7D1','ZYX'],
        ['MAPK3','SFTPC'],
        ['MAPK8IP1','RP11-259G18.2'],
        ['MAPKAP1','PTMS'],
        ['MAPKAPK2','SFTPA2'],
        ['MAPKAPK2','TBC1D10B'],
        ['MARCH6','TG'],
        ['MARCH8','SFTPC'],
        ['MARCKS','N4BP1'],
        ['MARCKS','PINK1'],
        ['MARCKS','RCC2'],
        ['MARCKS','SAMD1'],
        ['MARCKS','TGFB1I1'],
        ['MARCKS','UBE2H'],
        ['MARCO','SFTPA1'],
        ['MARCO','SFTPC'],
        ['MARVELD1','PTX3'],
        ['MAT2A','TG'],
        ['MATR3','RP11-90H3.1'],
        ['MATR3','SNORA74A'],
        ['MAVS','PANK2'],
        ['MAVS','TG'],
        ['MAVS','ZNF444'],
        ['MAZ','PRRT2'],
        ['MAZ','PTMS'],
        ['MAZ','RCC2'],
        ['MAZ','TMEM160'],
        ['MAZ','TRIM28'],
        ['MAZ','UBE2S'],
        ['MB','MYH7'],
        ['MB','NPPA-AS1'],
        ['MB','TG'],
        ['MBD2','PTMS'],
        ['MBD2','YWHAQ'],
        ['MBD3','PTMS'],
        ['MBD3','TIMP2'],
        ['MBD3','TPM2'],
        ['MBOAT7','PCDHGC3'],
        ['MBP','RP11-386G11.10'],
        ['MBP','RTN3'],
        ['MBP','SPARC'],
        ['MBP','SYT11'],
        ['MBP','TUBA1A'],
        ['MBP','TUBB4A'],
        ['MBP','VAMP2'],
        ['MBTD1','RP11-481C4.2'],
        ['MC1R','TCF25'],
        ['MCAM','RPS24'],
        ['MCAM','RPS29'],
        ['MCAM','SFTPA2'],
        ['MCAM','SFTPB'],
        ['MCAM','TPT1'],
        ['MCL1','SFTPA1'],
        ['MCL1','SFTPA2'],
        ['MCL1','SFTPB'],
        ['MCL1','SFTPC'],
        ['MCL1','SPARC'],
        ['MCL1','TG'],
        ['MDC1','NRM'],
        ['MECP2','SFTPC'],
        ['MED13L','SFTPC'],
        ['MED13L','TG'],
        ['MED15','PTRF'],
        ['MED15','RPL14'],
        ['MED20','USP49'],
        ['MED29','TG'],
        ['MEF2B','MEF2BNB'],
        ['MEF2B','MEF2BNB-MEF2B'],
        ['MEF2BNB','PTRF'],
        ['MEG3','MYO1C'],
        ['MEG3','SCARB1'],
        ['MEG3','SFTPB'],
        ['MEG3','TIMP3'],
        ['MEG3','VIM'],
        ['MEG8','SNHG23'],
        ['MEIS3P1','MEIS3P2'],
        ['MEMO1P1','RPS9P1'],
        ['METAZOA_SRP','PAFAH1B1'],
        ['METAZOA_SRP','RN7SL1'],
        ['METAZOA_SRP','RN7SL2'],
        ['METTL10','RP11-12J10.3'],
        ['METTL10','SERPINE1'],
        ['METTL16','RP1-59D14.3'],
        ['METTL21B','RP11-571M6.15'],
        ['METTL21B','TSFM'],
        ['METTL23','MFSD11'],
        ['METTL2A','METTL2B'],
        ['METTL2A','RP11-212P7.3'],
        ['METTL7A','TG'],
        ['METTL9','SRM'],
        ['MFAP4','SFTPA1'],
        ['MFAP4','SFTPA2'],
        ['MFAP4','SFTPB'],
        ['MFAP4','SFTPC'],
        ['MFAP4','SLC34A2'],
        ['MFGE8','MYH9'],
        ['MFGE8','PLEC'],
        ['MFGE8','SPARC'],
        ['MFGE8','THBS1'],
        ['MGAT1','SFTPC'],
        ['MGAT4B','TRIM28'],
        ['MGEA5','SFTPA2'],
        ['MGEA5','SFTPC'],
        ['MGLL','PPP1R1A'],
        ['MGLL','SFTPA1'],
        ['MGLL','SFTPC'],
        ['MGP','SFTPA1'],
        ['MGP','SFTPA2'],
        ['MGP','SFTPB'],
        ['MGP','SFTPC'],
        ['MGP','SLC34A2'],
        ['MGP','TG'],
        ['MGP','VIM'],
        ['MGRN1','SFTPB'],
        ['MIA2','MIA2'],
        ['MIA2','RP11-407N17.3'],
        ['MICA','MICB'],
        ['MICAL2','MICALCL'],
        ['MICALL1','TNRC18'],
        ['MICB','PTRF'],
        ['MIDN','SFTPA2'],
        ['MIDN','SFTPC'],
        ['MINK1','SFTPC'],
        ['MIR100HG','RP11-166D19.1'],
        ['MIR143','MIR143HG'],
        ['MIR205','MIR205HG'],
        ['MIR221','MIR222HG'],
        ['MIR221','RP6-99M1.2'],
        ['MIR29B1','MKLN1-AS1'],
        ['MIR3654','MYH9'],
        ['MIR3654','TG'],
        ['MIR4742','WDR26'],
        ['MIR497HG','SFTPC'],
        ['MIR6753','TCIRG1'],
        ['MIR6821','PIM3'],
        ['MIRLET7I','RP11-631N16.2'],
        ['MKNK2','SFTPA1'],
        ['MKNK2','SFTPA2'],
        ['MKNK2','SFTPC'],
        ['MKRN1','MKRN4P'],
        ['MKRN1','RP11-535M15.2'],
        ['MLL','SFTPC'],
        ['MLL2','SFTPA1'],
        ['MLL2','SFTPB'],
        ['MLL2','SFTPC'],
        ['MLL5','RP11-325F22.2'],
        ['MLLT1','SFTPC'],
        ['MLLT4','SFTPC'],
        ['MLLT6','SFTPA2'],
        ['MLLT6','SFTPC'],
        ['MLLT6','TG'],
        ['MLYCD','OSGIN1'],
        ['MMADHC','RP1-145M24.1'],
        ['MMP1','MMP14'],
        ['MMP1','SERPINE2'],
        ['MMP1','STC1'],
        ['MMP1','TFPI2'],
        ['MMP1','TGFBI'],
        ['MMP1','THBS1'],
        ['MMP1','VIM'],
        ['MMP14','SERPINE2'],
        ['MMP14','THBS1'],
        ['MMP14','TIMP3'],
        ['MMP14','VIM'],
        ['MMP17','RCC2'],
        ['MMP2','MRC2'],
        ['MMP2','RHOB'],
        ['MMP2','SFTPA1'],
        ['MMP2','SFTPA2'],
        ['MMP2','SFTPC'],
        ['MMP2','TGFBI'],
        ['MMP2','THBS1'],
        ['MMP2','VIM'],
        ['MMP24-AS1','RP4-614O4.11'],
        ['MMRN2','RP11-96C23.8'],
        ['MN1','TRIM28'],
        ['MOAP1','SFTPB'],
        ['MOB3A','PTRF'],
        ['MOCS1','RP11-61I13.2'],
        ['MOCS1','RP11-61I13.3'],
        ['MPDZ','RP11-272P10.2'],
        ['MPEG1','RP11-1036E20.7'],
        ['MPHOSPH8','SFTPB'],
        ['MPRIP','SFTPC'],
        ['MPST','TIMP2'],
        ['MRC2','MYH9'],
        ['MRC2','PLEC'],
        ['MRC2','PTRF'],
        ['MRC2','SERPINE1'],
        ['MRC2','SFTPC'],
        ['MRC2','TAGLN'],
        ['MREG','PECR'],
        ['MROH7','MROH7-TTC4'],
        ['MRPL19','RP11-342K6.2'],
        ['MRPL28','VIM'],
        ['MRPL49','PTRF'],
        ['MRPS27','PLIN3'],
        ['MRPS27','PTRF'],
        ['MRPS27','SULT2A1'],
        ['MRPS27','TRAM2'],
        ['MRPS30','MTF2'],
        ['MRPS6','SLC5A3'],
        ['MS4A14','MS4A7'],
        ['MSANTD3','MSANTD3-TMEFF1'],
        ['MSANTD3','TMEFF1'],
        ['MSANTD3-TMEFF1','MURC'],
        ['MSH5','MSH5-SAPCD1'],
        ['MSMB','NCOA4'],
        ['MSN','MSNP1'],
        ['MSN','MYH9'],
        ['MSN','RP11-368D24__A.1'],
        ['MSN','SERPINE1'],
        ['MSN','SFTPA1'],
        ['MSN','SFTPA2'],
        ['MSN','SFTPB'],
        ['MSN','SFTPC'],
        ['MSN','TG'],
        ['MSN','THBS1'],
        ['MSN','VIM'],
        ['MSR1','SFTPC'],
        ['MSRB1','RPL3L'],
        ['MST1','MST1P2'],
        ['MST1L','MST1P2'],
        ['MSTO1','MSTO2P'],
        ['MT1B','MT1E'],
        ['MT1E','MT1JP'],
        ['MT1E','MT1P3'],
        ['MT1G','MT1M'],
        ['MT1G','MT2A'],
        ['MT1JP','MT1M'],
        ['MT1M','MT2P1'],
        ['MT1X','MT1XP1'],
        ['MT1X','MT2A'],
        ['MT1X','TG'],
        ['MT2A','MT2P1'],
        ['MT2A','S100A6'],
        ['MT2A','SFTPC'],
        ['MT2A','TG'],
        ['MTCH1','TG'],
        ['MTCH2','RP11-575C1.1'],
        ['MTG1','RP11-108K14.4'],
        ['MTG1','SCART1'],
        ['MTHFD1','MTHFD1P1'],
        ['MTIF2','RP11-344P13.3'],
        ['MTND4P32','RP11-522L3.6'],
        ['MTR','RYR2'],
        ['MTRF1L','RP11-756P10.3'],
        ['MTSS1L','RANBP3'],
        ['MUC1','MYH9'],
        ['MUC1','SFTPA1'],
        ['MUC1','SFTPA2'],
        ['MUC1','SFTPB'],
        ['MUC1','SFTPC'],
        ['MUC1','SPARC'],
        ['MUC1','ZFP36'],
        ['MUC16','MUC5AC'],
        ['MUC4','MUC5AC'],
        ['MUC4','SFTPA2'],
        ['MUC4','SFTPB'],
        ['MUC5AC','NEAT1'],
        ['MUC5AC','PIGR'],
        ['MUC5B','RP11-186B7.4'],
        ['MUC7','PIGR'],
        ['MUC7','WNK2'],
        ['MUM1','MYH11'],
        ['MURC','TMEFF1'],
        ['MUS81','SNX32'],
        ['MUSTN1','RP5-966M1.6'],
        ['MVP','PAGR1'],
        ['MVP','SFTPA1'],
        ['MVP','SFTPA2'],
        ['MVP','SFTPC'],
        ['MXD4','PTMS'],
        ['MXD4','SFTPC'],
        ['MXRA5','MXRA5Y'],
        ['MXRA7','MYO1C'],
        ['MXRA7','PTRF'],
        ['MXRA7','SFTPB'],
        ['MXRA8','PTMS'],
        ['MXRA8','SBF1'],
        ['MXRA8','SPARC'],
        ['MXRA8','THBS1'],
        ['MXRA8','VIM'],
        ['MXRA8','ZNF768'],
        ['MYADM','MYH9'],
        ['MYADM','RP11-1143G9.4'],
        ['MYADM','SFTPA2'],
        ['MYADM','SFTPB'],
        ['MYADM','SFTPC'],
        ['MYCBP','RP5-864K19.6'],
        ['MYCL','TRIT1'],
        ['MYH1','MYH2'],
        ['MYH10','SFTPA2'],
        ['MYH10','SFTPB'],
        ['MYH10','SFTPC'],
        ['MYH11','MYH9'],
        ['MYH11','MYL9'],
        ['MYH11','PCSK7'],
        ['MYH11','PLEKHO1'],
        ['MYH11','PRM1'],
        ['MYH11','PRM2'],
        ['MYH11','RP1-309I22.2'],
        ['MYH11','SFTPA1'],
        ['MYH11','SFTPA2'],
        ['MYH11','SFTPB'],
        ['MYH11','SFTPC'],
        ['MYH11','SMR3B'],
        ['MYH11','SYNPO2'],
        ['MYH11','TIMP3'],
        ['MYH11','TPM1'],
        ['MYH11','TPM2'],
        ['MYH14','SFTPA2'],
        ['MYH14','SFTPC'],
        ['MYH2','MYH7'],
        ['MYH6','MYH7'],
        ['MYH7','MYL2'],
        ['MYH7','TG'],
        ['MYH7','TNNI3'],
        ['MYH7','TNNT2'],
        ['MYH7','TNS1'],
        ['MYH7','TPM1'],
        ['MYH7','TPO'],
        ['MYH7','TTN'],
        ['MYH9','MYL9'],
        ['MYH9','NBEAL2'],
        ['MYH9','NCOR2'],
        ['MYH9','NEAT1'],
        ['MYH9','NR4A1'],
        ['MYH9','NUMA1'],
        ['MYH9','P4HB'],
        ['MYH9','PCOLCE'],
        ['MYH9','PCSK7'],
        ['MYH9','PIEZO1'],
        ['MYH9','PKM'],
        ['MYH9','PKN1'],
        ['MYH9','PLEC'],
        ['MYH9','PLIN4'],
        ['MYH9','PPP1R15A'],
        ['MYH9','PPP1R9B'],
        ['MYH9','PPP2CA'],
        ['MYH9','PRKCSH'],
        ['MYH9','PRPF8'],
        ['MYH9','PRRC2B'],
        ['MYH9','PSAP'],
        ['MYH9','PTMS'],
        ['MYH9','PTPRF'],
        ['MYH9','PTRF'],
        ['MYH9','PXDN'],
        ['MYH9','QSOX1'],
        ['MYH9','RANGAP1'],
        ['MYH9','RARG'],
        ['MYH9','RCN3'],
        ['MYH9','RHOC'],
        ['MYH9','RNASE1'],
        ['MYH9','RP11-302B13.5'],
        ['MYH9','RP11-498C9.3'],
        ['MYH9','RP5-940J5.9'],
        ['MYH9','RP5-977B1.11'],
        ['MYH9','RPL11'],
        ['MYH9','RPL13'],
        ['MYH9','RPL37'],
        ['MYH9','RRBP1'],
        ['MYH9','RTN4'],
        ['MYH9','SCAF1'],
        ['MYH9','SCRIB'],
        ['MYH9','SDF4'],
        ['MYH9','SEPT9'],
        ['MYH9','SERF2'],
        ['MYH9','SERPINE1'],
        ['MYH9','SERPINH1'],
        ['MYH9','SFTPA1'],
        ['MYH9','SFTPA2'],
        ['MYH9','SFTPB'],
        ['MYH9','SFTPC'],
        ['MYH9','SLC34A2'],
        ['MYH9','SLC7A5'],
        ['MYH9','SND1'],
        ['MYH9','SPARC'],
        ['MYH9','SPARCL1'],
        ['MYH9','SPTAN1'],
        ['MYH9','SPTBN1'],
        ['MYH9','SRCAP'],
        ['MYH9','STC2'],
        ['MYH9','SUPT6H'],
        ['MYH9','SURF4'],
        ['MYH9','TAGLN'],
        ['MYH9','TG'],
        ['MYH9','TGFBI'],
        ['MYH9','THBS1'],
        ['MYH9','TIMP1'],
        ['MYH9','TIMP3'],
        ['MYH9','TLN1'],
        ['MYH9','TNC'],
        ['MYH9','TNNT2'],
        ['MYH9','TNS1'],
        ['MYH9','TPM2'],
        ['MYH9','TPM4'],
        ['MYH9','TPT1'],
        ['MYH9','TRIM28'],
        ['MYH9','TUBB4B'],
        ['MYH9','U2AF2'],
        ['MYH9','UBXN11'],
        ['MYH9','VEGFA'],
        ['MYH9','VIM'],
        ['MYH9','VIM-AS1'],
        ['MYH9','VWF'],
        ['MYH9','WDR1'],
        ['MYH9','XYLT1'],
        ['MYH9','ZYX'],
        ['MYL12A','MYL12B'],
        ['MYL12A','RP11-265N6.3'],
        ['MYL12B','RP11-265N6.3'],
        ['MYL12B','SFTPC'],
        ['MYL2','NDRG1'],
        ['MYL2','TG'],
        ['MYL2','TTN'],
        ['MYL3','PRSS45'],
        ['MYL6','MYL6B'],
        ['MYL6','MYL6P1'],
        ['MYL6','MYL6P3'],
        ['MYL6','PLEC'],
        ['MYL6','PXDN'],
        ['MYL6','RP5-940J5.9'],
        ['MYL6','SFTPA1'],
        ['MYL6','SFTPA2'],
        ['MYL6','SFTPB'],
        ['MYL6','SFTPC'],
        ['MYL6','SPARC'],
        ['MYL6','THBS1'],
        ['MYL6','VIM'],
        ['MYL9','P4HB'],
        ['MYL9','PKM'],
        ['MYL9','PLEC'],
        ['MYL9','SERPINE1'],
        ['MYL9','SFTPA1'],
        ['MYL9','SFTPA2'],
        ['MYL9','SFTPB'],
        ['MYL9','SFTPC'],
        ['MYL9','SPARC'],
        ['MYL9','TAGLN'],
        ['MYL9','TGFBI'],
        ['MYL9','THBS1'],
        ['MYL9','VIM'],
        ['MYLK','SFTPC'],
        ['MYO10','SFTPA2'],
        ['MYO10','VIM-AS1'],
        ['MYO15B','SFTPC'],
        ['MYO1C','NXN'],
        ['MYO1C','PTGIS'],
        ['MYO1C','PTRF'],
        ['MYO1C','SEPN1'],
        ['MYO1C','SERPINE1'],
        ['MYO1C','SFTPA1'],
        ['MYO1C','SFTPA2'],
        ['MYO1C','SFTPB'],
        ['MYO1C','SFTPC'],
        ['MYO1C','TAPBP'],
        ['MYO1C','TG'],
        ['MYO9B','PTRF'],
        ['MYOF','SFTPA1'],
        ['MYRF','SFTPC'],
        ['MZT2A','MZT2B'],
        ['NABP1','UC_338'],
        ['NACA','NACA2'],
        ['NACA','NACA3P'],
        ['NACA','NACAP1'],
        ['NACA','PRIM1'],
        ['NACA3P','NACAP1'],
        ['NACC1','TP53I11'],
        ['NAMPT','NAMPTP1'],
        ['NAMPT','NAMPTP3'],
        ['NAP1L1','NAP1L1P1'],
        ['NAP1L1','NAP1L1P3'],
        ['NAP1L1','NAP1L4P3'],
        ['NAP1L1','RP11-278C7.1'],
        ['NAPSA','SFTPA1'],
        ['NAPSA','SFTPA2'],
        ['NAPSA','SFTPB'],
        ['NAPSA','SFTPC'],
        ['NAPSA','SLC34A2'],
        ['NARF','RP13-20L14.10'],
        ['NAT14','ZNF628'],
        ['NAT8','NAT8B'],
        ['NAV3','RP1-34H18.1'],
        ['NBEA','NBEAP1'],
        ['NBEAL1','RPL12'],
        ['NBEAL2','SFTPB'],
        ['NBEAL2','SFTPC'],
        ['NBL1','SFTPC'],
        ['NBL1','TGFBI'],
        ['NBPF1','NBPF25P'],
        ['NBPF1','NBPF3'],
        ['NBPF10','NBPF8'],
        ['NBPF11','NBPF12'],
        ['NBPF14','NOTCH2NL'],
        ['NBPF14','SMG8'],
        ['NBPF15','NBPF8P'],
        ['NBPF15','RP11-640M9.2'],
        ['NBPF2P','NBPF3'],
        ['NBR1','NBR2'],
        ['NBR1','SFTPB'],
        ['NCAM1','VSNL1'],
        ['NCBP2','PIGZ'],
        ['NCDN','VSNL1'],
        ['NCF1B','NCF1C'],
        ['NCKAP1','PNLIP'],
        ['NCL','NEFM'],
        ['NCL','PSAP'],
        ['NCL','PTMS'],
        ['NCL','RPL31'],
        ['NCL','SFTPA1'],
        ['NCL','SFTPC'],
        ['NCL','SNORA75'],
        ['NCLN','SFTPA2'],
        ['NCLN','THBS1'],
        ['NCOA4','RP11-557J10.3'],
        ['NCOA6','PIGU'],
        ['NCOA6','TBP'],
        ['NCOA7','SFTPC'],
        ['NCOR2','OAF'],
        ['NCOR2','PLAUR'],
        ['NCOR2','PLEC'],
        ['NCOR2','RPL14'],
        ['NCOR2','SFTPA1'],
        ['NCOR2','SFTPA2'],
        ['NCOR2','SFTPB'],
        ['NCOR2','SFTPC'],
        ['NCOR2','SH3BGRL3'],
        ['NCOR2','UBC'],
        ['NCOR2','VIM-AS1'],
        ['NCR3LG1','NUCB2'],
        ['NDE1','PTRF'],
        ['NDEL1','SFTPA2'],
        ['NDRG1','SFTPA2'],
        ['NDRG1','SFTPC'],
        ['NDRG2','SFTPC'],
        ['NDST1','SFTPA1'],
        ['NDST1','SFTPA2'],
        ['NDST1','SFTPC'],
        ['NDST1','ZFP36'],
        ['NDST2','RP11-574K11.31'],
        ['NDUFA1','TG'],
        ['NDUFA6-AS1','RP4-669P10.19'],
        ['NDUFAF6','RP11-347C18.3'],
        ['NDUFB8','NDUFB8P2'],
        ['NDUFB8','SEC31B'],
        ['NDUFB8P2','RP11-411B6.6'],
        ['NDUFS5','RP11-321A17.3'],
        ['NDUFV2','NDUFV2P1'],
        ['NDUFV2P1','RP11-21J18.1'],
        ['NDUFV3','PTRF'],
        ['NEAT1','RPL37A'],
        ['NEAT1','SFTPA1'],
        ['NEAT1','SFTPA2'],
        ['NEAT1','SFTPB'],
        ['NEAT1','SFTPC'],
        ['NEAT1','SOD2'],
        ['NEAT1','TG'],
        ['NEB','TG'],
        ['NEB','TPM2'],
        ['NECAB2','OSGIN1'],
        ['NECAP2','SFTPC'],
        ['NEDD8','SFTPC'],
        ['NEDD8-MDP1','SFTPC'],
        ['NEDD9','SFTPA2'],
        ['NEDD9','SFTPB'],
        ['NEDD9','SFTPC'],
        ['NEFH','NEFHP1'],
        ['NEK10','SLC4A7'],
        ['NEK6','RP11-121A14.3'],
        ['NES','SFTPC'],
        ['NF2','PTRF'],
        ['NFATC3','PLA2G15'],
        ['NFE2L1','SFTPA2'],
        ['NFE2L1','SFTPB'],
        ['NFE2L1','SFTPC'],
        ['NFE2L2','SFTPC'],
        ['NFE2L2','TG'],
        ['NFIC','SFTPB'],
        ['NFIC','TG'],
        ['NFIX','PTMS'],
        ['NFIX','SFTPA1'],
        ['NFIX','SFTPA2'],
        ['NFIX','SFTPB'],
        ['NFIX','SFTPC'],
        ['NFIX','SPTBN1'],
        ['NFIX','VPS26B'],
        ['NFKBIA','SFTPC'],
        ['NFKBIZ','SFTPC'],
        ['NFS1','RBM12'],
        ['NGRN','PERP'],
        ['NGRN','RP11-697E2.12'],
        ['NGRN','RP11-697E2.7'],
        ['NIN','RP11-248J18.3'],
        ['NISCH','SFTPA2'],
        ['NISCH','SFTPC'],
        ['NKAPP1','RHOXF1'],
        ['NKRF','SEPT6'],
        ['NKX2-1','RP11-896J10.3'],
        ['NKX2-1','SFTA3'],
        ['NKX2-1','SFTPA1'],
        ['NKX2-1','SFTPC'],
        ['NKX2-1','TG'],
        ['NKX6-2','RP11-288G11.3'],
        ['NLGN4X','NLGN4Y'],
        ['NLK','PRDM11'],
        ['NLK','RP11-379B18.5'],
        ['NME1-NME2','NME2'],
        ['NME1-NME2','VIM-AS1'],
        ['NME2','VIM-AS1'],
        ['NME4','PCSK7'],
        ['NME4','PTRF'],
        ['NME4','SDC3'],
        ['NME4','Z97634.3'],
        ['NME4','Z97634.5'],
        ['NMNAT1','RP11-84A14.4'],
        ['NMNAT2','VSNL1'],
        ['NOC3L','RP11-76P2.4'],
        ['NOL12','TRIOBP'],
        ['NOL6','SUGT1P'],
        ['NOL9','PTRF'],
        ['NOLC1','PPRC1'],
        ['NONO','PTRF'],
        ['NONO','SFTPB'],
        ['NOP56','NOP56P1'],
        ['NOP56','NOP56P3'],
        ['NOP56','SNORD56'],
        ['NOP58','RP11-430C1.2'],
        ['NOP58','SNORD11'],
        ['NOS2','RP1-66C13.4'],
        ['NOTCH1','SFTPC'],
        ['NOTCH2','NOTCH2NL'],
        ['NOTCH2','RP11-458D21.5'],
        ['NOTCH2','SFTPC'],
        ['NOTCH3','PTMS'],
        ['NOTCH3','SFTPA1'],
        ['NOTCH3','SFTPC'],
        ['NOTCH4','SFTPA2'],
        ['NOTCH4','SFTPC'],
        ['NPC2','RP3-449M8.6'],
        ['NPC2','SFTPA1'],
        ['NPC2','SFTPA2'],
        ['NPC2','SFTPB'],
        ['NPC2','SFTPC'],
        ['NPC2','SYNDIG1L'],
        ['NPEPL1','STX16'],
        ['NPIPA1','NPIPA5'],
        ['NPIPA2','NPIPP1'],
        ['NPIPA2','PKD1P6'],
        ['NPIPA3','NPIPA5'],
        ['NPIPA3','NPIPP1'],
        ['NPIPA3','PKD1P6'],
        ['NPIPA7','PKD1'],
        ['NPIPB15','PDXDC2P'],
        ['NPIPB1P','RP11-166B2.1'],
        ['NPIPL2','PDXDC2P'],
        ['NPIPP1','PDXDC2P'],
        ['NPIPP1','RP11-419C5.2'],
        ['NPLOC4','PRKACA'],
        ['NPLOC4','PTMS'],
        ['NPLOC4','ZMYND19'],
        ['NPM1','NPM1P40'],
        ['NPM1','NPM1P51'],
        ['NPM1','NPM1P6'],
        ['NPM2','XPO7'],
        ['NPNT','SFTPC'],
        ['NPNT','TG'],
        ['NPPA','TIMP3'],
        ['NPR3','VIM'],
        ['NQO2','RP1-90J20.11'],
        ['NQO2','RP1-90J20.7'],
        ['NR2F2','PTMS'],
        ['NR2F2','RP11-522B15.3'],
        ['NR4A1','SFTPA1'],
        ['NR4A1','SFTPA2'],
        ['NR4A1','SFTPB'],
        ['NR4A1','SFTPC'],
        ['NR4A1','SLC34A2'],
        ['NR4A1','TG'],
        ['NR4A1','ZFP36'],
        ['NR4A2','SFTPA1'],
        ['NR4A2','SFTPA2'],
        ['NR4A3','SFTPC'],
        ['NRAP','TG'],
        ['NRG3-AS1','RP11-202D18.2'],
        ['NRGN','TMSB10'],
        ['NRIP3','PTRF'],
        ['NSFL1C','RP4-776F14.3'],
        ['NSFL1C','SIRPB2'],
        ['NSUN5P1','NSUN5P2'],
        ['NSUN6','RP11-499P20.2'],
        ['NSUN7','RP11-632F7.1'],
        ['NT5C1B','RDH14'],
        ['NT5C1B-RDH14','RDH14'],
        ['NT5C3A','NT5C3AP1'],
        ['NT5DC2','PTRF'],
        ['NT5DC3','PTRF'],
        ['NTAN1','NTAN1P2'],
        ['NTN4','RP11-536G4.2'],
        ['NTN4','VASN'],
        ['NUB1','PTRF'],
        ['NUCB1','SFTPA1'],
        ['NUCB1','SFTPA2'],
        ['NUCB1','SFTPC'],
        ['NUCB1','THBS1'],
        ['NUDT10','NUDT11'],
        ['NUDT10','PTMS'],
        ['NUMA1','SFTPA1'],
        ['NUMA1','SFTPA2'],
        ['NUMA1','SFTPB'],
        ['NUMA1','SFTPC'],
        ['NUMA1','TG'],
        ['NUP50','RP11-973N13.2'],
        ['NUPR1','SFTPC'],
        ['NUPR1','TG'],
        ['NUTM2A-AS1','RP11-119F19.2'],
        ['NUTM2A-AS1','RP11-773D16.1'],
        ['NXF1','SFTPC'],
        ['NXF1','TMEM223'],
        ['NXN','PTMS'],
        ['NXN','PTRF'],
        ['NXN','TPM4'],
        ['OAZ1','SFTPA1'],
        ['OAZ1','SFTPA2'],
        ['OAZ1','SFTPC'],
        ['OAZ1','THBS1'],
        ['OAZ1','VIM'],
        ['OAZ1','VIM-AS1'],
        ['OBP2A','OBP2B'],
        ['OBSCN','SFTPC'],
        ['ODF3B','SCO2'],
        ['ODF3B','TYMP'],
        ['OGDH','SFTPA2'],
        ['OGDH','SFTPC'],
        ['OGT','SFTPA2'],
        ['OLFM5P','UBQLNL'],
        ['OR2W3','TRIM58'],
        ['OR52N4','TRIM22'],
        ['ORAI2','PRKRIP1'],
        ['ORAOV1','PTMS'],
        ['ORM1','ORM2'],
        ['ORM1','SERPINA1'],
        ['OS9','SFTPA1'],
        ['OS9','SFTPA2'],
        ['OS9','SFTPB'],
        ['OS9','SFTPC'],
        ['OSBPL1A','RP11-799B12.4'],
        ['OSGIN1','RP11-505K9.4'],
        ['OSMR','RICTOR'],
        ['OSMR','SFTPB'],
        ['OSTF1','OSTF1P1'],
        ['OTUB1','TG'],
        ['OVCH2','RP11-494M8.4'],
        ['OVGP1','WDR77'],
        ['OXR1','RP11-395G23.3'],
        ['P2RX4','P2RX7'],
        ['P2RX5-TAX1BP3','SSSCA1'],
        ['P2RY11','PPAN'],
        ['P4HB','PLEC'],
        ['P4HB','SERPINE1'],
        ['P4HB','SFTPA1'],
        ['P4HB','SFTPA2'],
        ['P4HB','SFTPB'],
        ['P4HB','SFTPC'],
        ['P4HB','SPARC'],
        ['P4HB','TGFBI'],
        ['P4HB','THBS1'],
        ['P4HB','TPM2'],
        ['P4HB','VIM-AS1'],
        ['PABPC1','PABPC1P1'],
        ['PABPC1','PABPC1P3'],
        ['PABPC1','PABPC1P4'],
        ['PABPC1','PABPC3'],
        ['PACS1','THBS1'],
        ['PACS2','SFTPC'],
        ['PAFAH1B1','RN7SL26P'],
        ['PAFAH1B1','RN7SL608P'],
        ['PAFAH1B1','SFTPC'],
        ['PAFAH1B2','YIPF5'],
        ['PAGR1','PAGR1'],
        ['PAGR1','PRRT2'],
        ['PAK2','RP11-32B5.1'],
        ['PAK4','SFTPB'],
        ['PALM2-AKAP2','SFTPA1'],
        ['PALM2-AKAP2','SFTPA2'],
        ['PALM2-AKAP2','SFTPB'],
        ['PALM2-AKAP2','SFTPC'],
        ['PAOX','RP11-108K14.8'],
        ['PAPOLA','RP11-872J21.5'],
        ['PAPSS2','SFTPB'],
        ['PAQR6','SMG5'],
        ['PAQR7','PTRF'],
        ['PARG','PARGP1'],
        ['PARG','PARPG1'],
        ['PARM1','SFTPC'],
        ['PARM1','TG'],
        ['PARP4','PARP4P1'],
        ['PARP4','PARP4P2'],
        ['PARVB','RP4-671O14.6'],
        ['PATL2','SPG11'],
        ['PAX5','RHOBTB3'],
        ['PAX8','TG'],
        ['PAX8-AS1','RP11-65I12.1'],
        ['PBXIP1','PMVK'],
        ['PBXIP1','SFTPC'],
        ['PBXIP1','TG'],
        ['PCBP1','SFTPC'],
        ['PCBP1','TG'],
        ['PCBP2','PCBP2-OT1'],
        ['PCBP2','RP11-730G20.2'],
        ['PCBP2','UC_338'],
        ['PCBP2-OT1','RP11-793H13.8'],
        ['PCBP3','PCBP3-OT1'],
        ['PCDHA10','PCDHA11'],
        ['PCDHB10','PCDHB9'],
        ['PCDHGA12','SFTPC'],
        ['PCDHGA3','PCDHGA6'],
        ['PCDHGC3','SFTPC'],
        ['PCED1B','RP11-493L12.4'],
        ['PCF11','RP11-727A23.4'],
        ['PCOLCE','PLEC'],
        ['PCOLCE','SERPINE1'],
        ['PCSK2','TG'],
        ['PCSK7','PKM'],
        ['PCSK7','PLEC'],
        ['PCSK7','PTRF'],
        ['PCSK7','RRBP1'],
        ['PCSK7','SERPINE1'],
        ['PCSK7','SFTPC'],
        ['PCSK7','SPARC'],
        ['PCSK7','TAF15'],
        ['PCSK7','TGFBI'],
        ['PCSK7','THBS1'],
        ['PCSK7','TPM2'],
        ['PCSK7','VIM'],
        ['PCYOX1','SOD2'],
        ['PDAP1','TG'],
        ['PDCD4','RP11-313D6.3'],
        ['PDCD5','RP11-466P24.4'],
        ['PDCD6','RP11-43F13.1'],
        ['PDCD7','UBAP1L'],
        ['PDE4B','SMR3B'],
        ['PDE4DIP','WI2-1896O14.1'],
        ['PDGFB','SFTPA2'],
        ['PDGFRB','SFTPA2'],
        ['PDGFRB','SFTPC'],
        ['PDGFRB','VIM'],
        ['PDIA3','PDIA3P'],
        ['PDIA3','SEPT9'],
        ['PDIA4','TG'],
        ['PDK4','SFTPC'],
        ['PDLIM1','PDLIM1P1'],
        ['PDLIM1','SFTPC'],
        ['PDLIM2','SFTPA1'],
        ['PDLIM7','RP11-1334A24.5'],
        ['PDLIM7','THBS1'],
        ['PDLIM7','VIM-AS1'],
        ['PDLIM7','ZNF787'],
        ['PDPR','RP11-252A24.2'],
        ['PDXDC1','PDXDC2P'],
        ['PDXK','PTRF'],
        ['PDXK','SFTPB'],
        ['PDXK','SFTPC'],
        ['PEA15','PTMS'],
        ['PEA15','SFTPB'],
        ['PEAK1','PTRF'],
        ['PEAK1','SFTPB'],
        ['PEBP1','PEBP1P2'],
        ['PEBP1','SERPINA1'],
        ['PEBP1','TG'],
        ['PELP1','SFTPB'],
        ['PENK','RP11-16M8.2'],
        ['PER1','SFTPA1'],
        ['PER1','SFTPA2'],
        ['PER1','SFTPB'],
        ['PER1','SFTPC'],
        ['PER1','TG'],
        ['PES1','SFTPC'],
        ['PEX11A','PLIN1'],
        ['PFKFB3','SFTPC'],
        ['PFKP','VIM-AS1'],
        ['PFN1','PFN1P1'],
        ['PFN1','RANBP3'],
        ['PFN1','RP5-940J5.9'],
        ['PFN1','SFTPA1'],
        ['PFN1','SFTPA2'],
        ['PFN1','SFTPB'],
        ['PFN1','SFTPC'],
        ['PFN1','SNX22'],
        ['PFN1','SPARC'],
        ['PFN1','WDTC1'],
        ['PGAM5','PTRF'],
        ['PGAM5','PXMP2'],
        ['PGAM5','RP13-672B3.2'],
        ['PGAM5','SFTPB'],
        ['PGC','POLR2A'],
        ['PGC','SFTPB'],
        ['PGC','SFTPC'],
        ['PGK1','PGK1P2'],
        ['PGM5-AS1','PGM5-AS1'],
        ['PGRMC2','PTMS'],
        ['PHACTR2','SFTPC'],
        ['PHAX','RP11-560B16.5'],
        ['PHB','ZNF607'],
        ['PHB2','SCARNA12'],
        ['PHLDA1','SFTPB'],
        ['PHLDA1','SFTPC'],
        ['PHLDA2','PSPC1'],
        ['PHLDA2','TLE4'],
        ['PHLDA2','TRPM4'],
        ['PHLDA3','SEPT9'],
        ['PHLDB1','SFTPB'],
        ['PI4KA','PI4KAP1'],
        ['PI4KA','PI4KAP2'],
        ['PI4KAP1','PI4KAP2'],
        ['PIEZO1','PLEC'],
        ['PIEZO1','SFTPA2'],
        ['PIEZO1','SFTPC'],
        ['PIEZO1','THBS1'],
        ['PIEZO1','VIM'],
        ['PIGO','TG'],
        ['PIGR','SFTPA1'],
        ['PIGR','SFTPA2'],
        ['PIGR','SFTPB'],
        ['PIGR','SFTPC'],
        ['PIGS','SFTPC'],
        ['PIK3C2A','PNLIP'],
        ['PIK3R2','SFTPA1'],
        ['PIK3R2','SFTPA2'],
        ['PIK3R2','SFTPC'],
        ['PIK3R2','SYDE1'],
        ['PIK3R2','TRIM28'],
        ['PILRA','PILRB'],
        ['PILRA','STAG3L5P-PVRIG2P-PILRB'],
        ['PIP5K1A','PIPSL'],
        ['PKD1','PKD1P6'],
        ['PKD1','RP11-304L19.3'],
        ['PKDCC','PTMS'],
        ['PKM','PKMP1'],
        ['PKM','PLEC'],
        ['PKM','RHOC'],
        ['PKM','SERPINE1'],
        ['PKM','SFTPA1'],
        ['PKM','SFTPA2'],
        ['PKM','SFTPB'],
        ['PKM','SFTPC'],
        ['PKM','SPARC'],
        ['PKM','TAGLN'],
        ['PKM','TGFBI'],
        ['PKM','THBS1'],
        ['PKM','TIMP1'],
        ['PKM','TNC'],
        ['PKM','TPM2'],
        ['PKM','VIM'],
        ['PKM','VIM-AS1'],
        ['PKN1','SFTPB'],
        ['PKN1','UBE2M'],
        ['PKN2','TCEB1P19'],
        ['PKNOX2','RP11-687M24.7'],
        ['PLAUR','SOX11'],
        ['PLCD3','PTRF'],
        ['PLCD3','TPM4'],
        ['PLCG1','RP3-511B24.4'],
        ['PLCG2','RN7SK'],
        ['PLCG2','RN7SKP255'],
        ['PLCXD1','PTRF'],
        ['PLD3','SFTPC'],
        ['PLEC','PLIN3'],
        ['PLEC','PLOD3'],
        ['PLEC','PSAP'],
        ['PLEC','PTMS'],
        ['PLEC','PTRF'],
        ['PLEC','PXDN'],
        ['PLEC','PXN'],
        ['PLEC','RABL6'],
        ['PLEC','RNF187'],
        ['PLEC','RP5-940J5.9'],
        ['PLEC','RRBP1'],
        ['PLEC','RTN4'],
        ['PLEC','SCARF2'],
        ['PLEC','SEPT9'],
        ['PLEC','SERPINE1'],
        ['PLEC','SFTPA1'],
        ['PLEC','SFTPA2'],
        ['PLEC','SFTPB'],
        ['PLEC','SFTPC'],
        ['PLEC','SH3BGRL3'],
        ['PLEC','SH3PXD2A'],
        ['PLEC','SHC1'],
        ['PLEC','SPARC'],
        ['PLEC','SRRM2'],
        ['PLEC','TAGLN'],
        ['PLEC','TFPI2'],
        ['PLEC','TGFBI'],
        ['PLEC','THBS1'],
        ['PLEC','TIMP1'],
        ['PLEC','TIMP2'],
        ['PLEC','TIMP3'],
        ['PLEC','TLN1'],
        ['PLEC','TNC'],
        ['PLEC','TNFRSF12A'],
        ['PLEC','TPM2'],
        ['PLEC','TRIM28'],
        ['PLEC','VIM'],
        ['PLEC','VIM-AS1'],
        ['PLEC','ZYX'],
        ['PLEKHA2','SFTPB'],
        ['PLEKHA3','PLEKHA3P1'],
        ['PLEKHB1','VAMP2'],
        ['PLEKHG5','PTRF'],
        ['PLEKHG5','TNFRSF25'],
        ['PLEKHH1','TG'],
        ['PLEKHH2','SFTPC'],
        ['PLEKHJ1','PTRF'],
        ['PLEKHM1','PLEKHM1P'],
        ['PLEKHM1P','RP11-927P21.4'],
        ['PLEKHM2','SFTPC'],
        ['PLEKHO1','VPS45'],
        ['PLIN3','PRKCSH'],
        ['PLIN3','SCUBE3'],
        ['PLIN3','TAPBP'],
        ['PLIN4','PSAP'],
        ['PLIN4','SERPING1'],
        ['PLIN4','SPARC'],
        ['PLIN4','SPTBN1'],
        ['PLIN4','TBX1'],
        ['PLIN5','SAA2'],
        ['PLOD1','THBS1'],
        ['PLOD2','RP11-274H2.3'],
        ['PLS3','SFTPA2'],
        ['PLSCR2','RP11-758I14.3'],
        ['PLVAP','TG'],
        ['PLXNA1','VIM-AS1'],
        ['PLXNA2','SFTPC'],
        ['PLXNB1','SFTPA1'],
        ['PLXNB1','TG'],
        ['PLXNB2','PTRF'],
        ['PLXNB2','SFTPA2'],
        ['PLXNB2','SFTPC'],
        ['PLXNB2','TG'],
        ['PLXNB3','SRPK3'],
        ['PLXND1','SFTPA2'],
        ['PLXND1','SFTPC'],
        ['PML','SFTPC'],
        ['PMS2','PMS2CL'],
        ['PMS2CL','RBAK-RBAKDN'],
        ['PMS2P3','PMS2P4'],
        ['PNLIP','SLC37A4'],
        ['PNLIP','SLC4A4'],
        ['PNLIP','SPINT2'],
        ['PNLIP','TRIM28'],
        ['PNLIPRP2','PRSS1'],
        ['PNLIPRP2','PRSS2'],
        ['PNO1','RP11-597G23.1'],
        ['PNPLA2','TG'],
        ['PODN','SFTPC'],
        ['PODN','TG'],
        ['PODXL','PTRF'],
        ['PODXL','SFTPC'],
        ['POLR2A','SFTPA1'],
        ['POLR2A','SFTPA2'],
        ['POLR2A','SFTPC'],
        ['POLR2E','TG'],
        ['POLR3GL','RP11-315I20.1'],
        ['POLRMT','POLRMTP1'],
        ['POM121','POM121C'],
        ['POM121','POMZP3'],
        ['POM121C','POMZP3'],
        ['POMK','PTRF'],
        ['POMZP3','ZP3'],
        ['POR','SFTPC'],
        ['POSTN','THBS1'],
        ['POU6F1','SMAGP'],
        ['PPAP2B','TG'],
        ['PPDPF','SDC3'],
        ['PPFIBP1','PTRF'],
        ['PPFIBP1','SFTPB'],
        ['PPHLN1','ZCRB1'],
        ['PPIB','SFTPA2'],
        ['PPIB','SFTPC'],
        ['PPIE','RP11-69E11.4'],
        ['PPIEL','PPIEL'],
        ['PPIEL','RP11-69E11.4'],
        ['PPIH','RP5-994D16.3'],
        ['PPIL2','PTRF'],
        ['PPIL3','RNU6-312P'],
        ['PPL','SFTPC'],
        ['PPM1F','SFTPC'],
        ['PPP1CA','SFTPC'],
        ['PPP1CB','SFTPA1'],
        ['PPP1CB','TG'],
        ['PPP1R15A','SFTPA2'],
        ['PPP1R15A','SFTPC'],
        ['PPP1R15A','SRRM2'],
        ['PPP1R2','PPP1R2P3'],
        ['PPP1R21','STON1'],
        ['PPP1R8P1','SPATA6'],
        ['PPP2CB','TRIM28'],
        ['PPP2R1A','THBS1'],
        ['PPP2R4','RP11-247A12.2'],
        ['PPP6R1','SFTPA1'],
        ['PPP6R3','SFTPC'],
        ['PPRC1','SFTPC'],
        ['PRB1','PRB2'],
        ['PRB3','PRB4'],
        ['PRC1-AS1','RCCD1'],
        ['PRDX1','TDPX2'],
        ['PRELID1','PRELID1P1'],
        ['PRELP','SFTPC'],
        ['PREX1','RP5-906C1.1'],
        ['PREX1','SFTPA1'],
        ['PREX1','SFTPC'],
        ['PRG2','SPARC'],
        ['PRH1','PRH2'],
        ['PRH1','TAS2R15P'],
        ['PRH1-PRR4','PRH2'],
        ['PRKAA1','TTC33'],
        ['PRKAB1','TMEM233'],
        ['PRKAR1A','PRKAR1AP'],
        ['PRKAR1A','TG'],
        ['PRKCA','TPM4'],
        ['PRKCD','TIMM50'],
        ['PRKCH','RP11-47I22.3'],
        ['PRKCQ-AS1','RP11-554I8.1'],
        ['PRKCSH','PTRF'],
        ['PRKCSH','REXO1'],
        ['PRKCSH','SFTPA1'],
        ['PRKCSH','SFTPA2'],
        ['PRKCSH','SFTPB'],
        ['PRKCSH','SFTPC'],
        ['PRKCSH','THBS1'],
        ['PRKCSH','TNRC18'],
        ['PRKCSH','VIM'],
        ['PRKRIR','PRKRIRP1'],
        ['PRKRIRP8','RGS7'],
        ['PRMT5','RP11-298I3.5'],
        ['PRODH','XXBAC-B444P24.10'],
        ['PROS1','PROS2P'],
        ['PROS1','PROSP'],
        ['PRPF40B','SFTPC'],
        ['PRPF4B','RP11-78H18.2'],
        ['PRPF8','SFTPA1'],
        ['PRPF8','SFTPB'],
        ['PRPF8','SFTPC'],
        ['PRPF8','SLC34A2'],
        ['PRR11','ZNF197'],
        ['PRR14','SFTPC'],
        ['PRR24','UBE2Z'],
        ['PRR4','RP11-785H5.2'],
        ['PRR4','TAS2R15'],
        ['PRR4','TAS2R15P'],
        ['PRRC2A','PTMS'],
        ['PRRC2A','SFTPA1'],
        ['PRRC2A','SFTPA2'],
        ['PRRC2A','SFTPB'],
        ['PRRC2A','SFTPC'],
        ['PRRC2A','SLC4A2'],
        ['PRRC2A','THBS1'],
        ['PRRC2B','SFTPA1'],
        ['PRRC2B','SFTPC'],
        ['PRRC2C','SFTPA2'],
        ['PRRC2C','SFTPB'],
        ['PRRX2','YKT6'],
        ['PRSS1','RNASE1'],
        ['PRSS2','RNASE1'],
        ['PRSS44','PRSS45'],
        ['PRSS45','PRSS46'],
        ['PRX','SFTPC'],
        ['PSAP','SFTPA1'],
        ['PSAP','SFTPA2'],
        ['PSAP','SFTPB'],
        ['PSAP','SFTPC'],
        ['PSAP','SLC34A2'],
        ['PSAP','SPARC'],
        ['PSAP','SPTBN1'],
        ['PSAP','SRRM2'],
        ['PSAP','TG'],
        ['PSAP','THBS1'],
        ['PSAP','VIM'],
        ['PSAP','VIM-AS1'],
        ['PSAT1','PSAT1P3'],
        ['PSEN1','SFTPC'],
        ['PSIP1','RP11-54D18.4'],
        ['PSMB7','SFTPC'],
        ['PSMC2','RP11-64K7.1'],
        ['PSMD11','SFTPC'],
        ['PSMD11','ZNF207'],
        ['PSMD3','SFTPC'],
        ['PSME2','PSME2P2'],
        ['PSME2','RP11-815N9.2'],
        ['PSMG3','PTRF'],
        ['PSPC1','ZMYM5'],
        ['PTBP1','SFTPC'],
        ['PTEN','PTENP1'],
        ['PTENP1-AS','TRBV26OR9-2'],
        ['PTGDR2','RP11-804A23.4'],
        ['PTGDS','SFTPA2'],
        ['PTGDS','SFTPC'],
        ['PTGDS','VAMP2'],
        ['PTGES','PTRF'],
        ['PTGES2','RP11-395P17.3'],
        ['PTGES3','PTGES3P1'],
        ['PTK6','ZNF611'],
        ['PTMA','PTMAP2'],
        ['PTMA','PTMAP4'],
        ['PTMA','PTMAP5'],
        ['PTMA','RP11-330M2.4'],
        ['PTMA','RP11-529H20.3'],
        ['PTMA','TMC5'],
        ['PTMS','PTRF'],
        ['PTMS','R3HDM4'],
        ['PTMS','RP5-977B1.11'],
        ['PTMS','RPS2'],
        ['PTMS','RRBP1'],
        ['PTMS','RRP36'],
        ['PTMS','RRP9'],
        ['PTMS','SCAF1'],
        ['PTMS','SDC3'],
        ['PTMS','SEPT9'],
        ['PTMS','SERPINE1'],
        ['PTMS','SFTPA1'],
        ['PTMS','SFTPA2'],
        ['PTMS','SFTPC'],
        ['PTMS','SLC12A2'],
        ['PTMS','SMARCA2'],
        ['PTMS','SRRT'],
        ['PTMS','SULF2'],
        ['PTMS','TAGLN'],
        ['PTMS','TBC1D10B'],
        ['PTMS','TG'],
        ['PTMS','TGFBI'],
        ['PTMS','THBS1'],
        ['PTMS','TMED2'],
        ['PTMS','TPM2'],
        ['PTMS','TRIM28'],
        ['PTMS','TRIO'],
        ['PTMS','TWIST1'],
        ['PTMS','UBAP2L'],
        ['PTMS','UBE2M'],
        ['PTMS','UBE2Q1'],
        ['PTMS','UBXN11'],
        ['PTMS','UPF1'],
        ['PTMS','UQCR11'],
        ['PTMS','VIM'],
        ['PTMS','ZC3H4'],
        ['PTMS','ZNF784'],
        ['PTP4A2','TRIM28'],
        ['PTPN11','RP11-561O4.1'],
        ['PTPN3','SFTPB'],
        ['PTPRB','SFTPA2'],
        ['PTPRB','SFTPB'],
        ['PTPRB','SFTPC'],
        ['PTPRF','SFTPA2'],
        ['PTPRF','SFTPC'],
        ['PTPRQ','RP11-288D9.1'],
        ['PTPRS','STK24'],
        ['PTPRS','TSR3'],
        ['PTRF','PTX3'],
        ['PTRF','RHOA'],
        ['PTRF','RP11-1143G9.4'],
        ['PTRF','RP11-178L8.8'],
        ['PTRF','RP11-399J13.3'],
        ['PTRF','RP11-680A11.5'],
        ['PTRF','RP5-940J5.9'],
        ['PTRF','SCAMP4'],
        ['PTRF','SCARB2'],
        ['PTRF','SCUBE3'],
        ['PTRF','SEPN1'],
        ['PTRF','SERPINE1'],
        ['PTRF','SERPINH1'],
        ['PTRF','SFTPA1'],
        ['PTRF','SFTPA2'],
        ['PTRF','SFTPB'],
        ['PTRF','SFTPC'],
        ['PTRF','SFXN3'],
        ['PTRF','SLC35E2B'],
        ['PTRF','SLC35F6'],
        ['PTRF','SLX4'],
        ['PTRF','SNHG17'],
        ['PTRF','SNX15'],
        ['PTRF','SOD2'],
        ['PTRF','SPARC'],
        ['PTRF','SPC24'],
        ['PTRF','SPG7'],
        ['PTRF','SPTBN1'],
        ['PTRF','SSU72'],
        ['PTRF','STAU1'],
        ['PTRF','SUMF2'],
        ['PTRF','SYT11'],
        ['PTRF','TAGLN'],
        ['PTRF','TANGO2'],
        ['PTRF','TAPBP'],
        ['PTRF','TG'],
        ['PTRF','TGFBI'],
        ['PTRF','THBS1'],
        ['PTRF','TINAGL1'],
        ['PTRF','TLDC1'],
        ['PTRF','TLN1'],
        ['PTRF','TMEM119'],
        ['PTRF','TMEM120B'],
        ['PTRF','TMEM127'],
        ['PTRF','TNFRSF10B'],
        ['PTRF','TOR4A'],
        ['PTRF','TPM2'],
        ['PTRF','TRIM28'],
        ['PTRF','TSPAN31'],
        ['PTRF','TTL'],
        ['PTRF','UAP1L1'],
        ['PTRF','USP22'],
        ['PTRF','VIM'],
        ['PTRF','VIM-AS1'],
        ['PTRF','VPS33A'],
        ['PTRF','WDR55'],
        ['PTRF','WWC3'],
        ['PTRF','ZNF331'],
        ['PTRF','ZSCAN16-AS1'],
        ['PTRF','ZYX'],
        ['PTS','RP11-356J5.12'],
        ['PTTG1IP','SFTPA2'],
        ['PTTG1IP','SFTPC'],
        ['PTX3','SPEG'],
        ['PUM1','SFTPC'],
        ['PUS1','RP11-417L19.6'],
        ['PVR','SERPINE1'],
        ['PVRIG','STAG3'],
        ['PWAR5','RP11-701H24.5'],
        ['PWAR5','SNORD64'],
        ['PXDC1','TNFRSF1A'],
        ['PXDN','SFTPA1'],
        ['PXDN','SFTPC'],
        ['PXDN','THBS1'],
        ['PXDN','TLN1'],
        ['PXDN','VIM'],
        ['PXN','SFTPB'],
        ['PXN','SFTPC'],
        ['PYGB','TG'],
        ['PYGM','TG'],
        ['PYGO2','SHC1'],
        ['PYY','PYY2'],
        ['QSOX1','SFTPA1'],
        ['QSOX1','SFTPC'],
        ['RAB11B','SFTPC'],
        ['RAB11FIP1','SFTPB'],
        ['RAB11FIP1','SFTPC'],
        ['RAB11FIP5','SFXN5'],
        ['RAB11FIP5','THBS1'],
        ['RAB1B','RAB1C'],
        ['RAB34','THBS1'],
        ['RAB3D','SFTPB'],
        ['RAB40C','WFIKKN1'],
        ['RAB5B','SFTPA2'],
        ['RAB5B','SFTPC'],
        ['RAB5C','SFTPB'],
        ['RAB7A','SFTPB'],
        ['RABL2A','RABL2B'],
        ['RABL6','RP11-216L13.19'],
        ['RAD21','RP3-347M6.2'],
        ['RAD23A','TG'],
        ['RALGDS','SFTPA2'],
        ['RALGDS','SFTPB'],
        ['RAN','RANP1'],
        ['RANBP6','RP11-525E9.1'],
        ['RAP1A','RAP1AP'],
        ['RAP1GAP','TG'],
        ['RAPGEF1','SFTPC'],
        ['RARRES2','SFTPC'],
        ['RASA3','SFTPC'],
        ['RASSF4','ZNF22'],
        ['RASSF7','RP11-496I9.1'],
        ['RASSF8','SSPN'],
        ['RAVER1','SHC1'],
        ['RAVER1','SPARC'],
        ['RBFA','RP11-795F19.5'],
        ['RBFOX1','RP11-420N3.2'],
        ['RBFOX1','RP11-420N3.3'],
        ['RBFOX2','SFTPC'],
        ['RBL2','RP11-44F14.2'],
        ['RBM12','RP1-309K20.6'],
        ['RBM14-RBM4','RBM4'],
        ['RBM33','SFTPA1'],
        ['RBM47','RP11-489E7.1'],
        ['RBMX','RBMXL1'],
        ['RBP5','RP11-273B20.1'],
        ['RBX1','XPNPEP3'],
        ['RCC1','SNORA73B'],
        ['RCC2','TLE4'],
        ['RCN1','RCN1P2'],
        ['RCSD1','RP3-455J7.4'],
        ['REEP5','SFTPB'],
        ['RELB','SEPT5'],
        ['RERE','SFTPA2'],
        ['RERE','SFTPB'],
        ['RERE','SFTPC'],
        ['RFT1','RP11-894J14.5'],
        ['RGCC','SFTPB'],
        ['RGCC','SFTPC'],
        ['RGL2','SFTPA2'],
        ['RHBDD2','SFTPC'],
        ['RHOA','SFTPB'],
        ['RHOA','SFTPC'],
        ['RHOB','SFTPA1'],
        ['RHOB','SFTPA2'],
        ['RHOB','SFTPB'],
        ['RHOB','SFTPC'],
        ['RHOB','TG'],
        ['RHOBTB2','SFTPA1'],
        ['RHOBTB2','SFTPB'],
        ['RHOBTB2','SFTPC'],
        ['RHOC','THBS1'],
        ['RHOC','VIM'],
        ['RHOT2','WDR90'],
        ['RILPL1','RP11-972P1.7'],
        ['RIMBP2','RP11-662M24.2'],
        ['RIN2','RP1-122P22.2'],
        ['RIPK1','SERPINB9'],
        ['RN7SK','RN7SKP203'],
        ['RN7SK','RN7SKP230'],
        ['RN7SK','RN7SKP255'],
        ['RN7SK','RN7SKP80'],
        ['RN7SL1','RN7SL674P'],
        ['RN7SL151P','RN7SL2'],
        ['RN7SL2','RN7SL3'],
        ['RN7SL2','RN7SL381P'],
        ['RN7SL2','RN7SL4P'],
        ['RNASE1','SFTPA1'],
        ['RNASE1','SFTPA2'],
        ['RNASE1','SFTPB'],
        ['RNASE1','SFTPC'],
        ['RNASE4','RNASE4'],
        ['RNASE4','RP11-903H12.5'],
        ['RNASEK-C17ORF49','SFTPC'],
        ['RNF11','RP11-296A18.6'],
        ['RNF13','RP11-259P15.4'],
        ['RNF135','RP11-848P1.2'],
        ['RNF149','SNORD89'],
        ['RNF166','TXNDC5'],
        ['RNF213','SFTPA2'],
        ['RNF213','SFTPB'],
        ['RNF213','SFTPC'],
        ['RNF5','RNF5P1'],
        ['RNF5','SFTPA1'],
        ['RNF5','SFTPA2'],
        ['RNF5','SFTPB'],
        ['RNF5','SFTPC'],
        ['RNFT1','RNFT1P3'],
        ['RNH1','SFTPA2'],
        ['RNU2-2P','RNU2-59P'],
        ['RNU2-59P','WDR74'],
        ['RNU4-25P','RP11-55J15.2'],
        ['ROBO4','SFTPA2'],
        ['ROBO4','SFTPB'],
        ['ROBO4','SFTPC'],
        ['ROPN1L','RP11-1C1.7'],
        ['RP1','RP11-53M11.3'],
        ['RP1','RP11-56A10.1'],
        ['RP1-102E24.1','RPL31'],
        ['RP1-102E24.10','TAPBPL'],
        ['RP1-120G22.11','RPL22'],
        ['RP1-178F15.4','RP1-178F15.5'],
        ['RP1-178F15.4','S100A13'],
        ['RP1-178F15.5','S100A13'],
        ['RP1-182O16.1','RPL34'],
        ['RP1-199J3.5','TSTD3'],
        ['RP1-1J6.2','ZHX3'],
        ['RP1-241P17.4','SUMO2'],
        ['RP1-249H1.3','RPL30'],
        ['RP1-278E11.3','RPL23'],
        ['RP1-283E3.8','SLC35E2'],
        ['RP1-283E3.8','SLC35E2B'],
        ['RP1-309I22.2','SPP1'],
        ['RP1-309I22.2','TG'],
        ['RP1-309I22.2','TIMP3'],
        ['RP1-37E16.12','SFTPC'],
        ['RP1-37E16.12','TRIOBP'],
        ['RP1-83M4.2','TPT1'],
        ['RP1-89D4.1','RPS24'],
        ['RP11-100F15.1','RP11-100F15.2'],
        ['RP11-1029M24.1','RPL17'],
        ['RP11-1035H13.3','RPS15AP1'],
        ['RP11-1035H13.3','RPS15AP10'],
        ['RP11-104H15.7','SPEM1'],
        ['RP11-106A1.2','RPL23A'],
        ['RP11-108K14.4','RP11-108K14.8'],
        ['RP11-10G12.1','RPS27'],
        ['RP11-1100L3.8','RPL3'],
        ['RP11-1100L3.8','SFTPA1'],
        ['RP11-1100L3.8','SFTPC'],
        ['RP11-112J1.1','RPL31'],
        ['RP11-113C12.2','RPL15'],
        ['RP11-1143G9.4','RP11-186B7.4'],
        ['RP11-1143G9.4','SFTPB'],
        ['RP11-1143G9.4','SLC2A11'],
        ['RP11-1143G9.4','ZNF106'],
        ['RP11-1149O23.4','TNFRSF10A'],
        ['RP11-114G22.1','RP11-153K16.1'],
        ['RP11-119F19.2','RP11-322M19.1'],
        ['RP11-119F19.2','RP11-773D16.1'],
        ['RP11-120B7.1','RPS20'],
        ['RP11-122K13.12','TUBGCP2'],
        ['RP11-124N14.3','SFTPA2'],
        ['RP11-124N14.3','VIM-AS1'],
        ['RP11-1252I4.2','RP11-423H2.3'],
        ['RP11-125N22.3','RP11-548K12.5'],
        ['RP11-129B9.1','RPL14'],
        ['RP11-12M9.3','RPS9'],
        ['RP11-133O22.6','RP11-384P7.6'],
        ['RP11-13N13.2','SNRPD1'],
        ['RP11-141M1.3','STARD13'],
        ['RP11-142A22.3','RTN3'],
        ['RP11-142L1.3','SET'],
        ['RP11-146N23.1','RPS6'],
        ['RP11-152F13.10','RPS17P5'],
        ['RP11-155G14.5','RP11-212P7.3'],
        ['RP11-155G14.6','RP11-212P7.3'],
        ['RP11-156K13.1','RP11-806O11.1'],
        ['RP11-158M2.3','RP11-158M2.5'],
        ['RP11-159D12.2','RP11-159D12.5'],
        ['RP11-159G9.5','ZNF654'],
        ['RP11-159H22.1','RPL37A'],
        ['RP11-159H3.1','RPL19'],
        ['RP11-159N11.4','TTC12'],
        ['RP11-166D19.1','RP11-820L6.1'],
        ['RP11-168A11.4','THBS4'],
        ['RP11-169K16.8','RPS16'],
        ['RP11-170M17.2','RPL7A'],
        ['RP11-173D9.5','SEPT7'],
        ['RP11-175B9.3','RPS27'],
        ['RP11-179G5.1','RPSA'],
        ['RP11-186B7.4','SFTPA1'],
        ['RP11-186B7.4','SFTPA2'],
        ['RP11-186B7.4','SFTPB'],
        ['RP11-186B7.4','SFTPC'],
        ['RP11-191N8.2','RP11-815M8.1'],
        ['RP11-195E11.2','RP11-373A9.3'],
        ['RP11-203J24.9','ST6GALNAC6'],
        ['RP11-204C16.4','YWHAZ'],
        ['RP11-20G13.2','RP11-20G13.3'],
        ['RP11-20O24.1','RPS27'],
        ['RP11-216L13.17','TMEM141'],
        ['RP11-216L13.18','RP11-216L13.19'],
        ['RP11-219A15.1','USP32'],
        ['RP11-228B15.4','SFTPC'],
        ['RP11-234A1.1','RPS18'],
        ['RP11-234N17.1','RPL5'],
        ['RP11-242D8.1','RP11-242D8.3'],
        ['RP11-244J10.1','RPL26'],
        ['RP11-247I13.3','RPS18'],
        ['RP11-250B2.3','RP11-250B2.4'],
        ['RP11-252I13.1','USP7'],
        ['RP11-255B23.1','SRRM1'],
        ['RP11-255H23.2','ZNF254'],
        ['RP11-258F22.1','ZNF248'],
        ['RP11-262D11.2','TPT1'],
        ['RP11-264B17.3','SPNS1'],
        ['RP11-264F23.1','RPS15'],
        ['RP11-265B8.4','TJP2'],
        ['RP11-265N7.1','RP11-624L4.1'],
        ['RP11-268J15.5','SH3D21'],
        ['RP11-274B21.1','TLK2'],
        ['RP11-274B21.14','TLK2'],
        ['RP11-274B21.3','RP11-384K6.2'],
        ['RP11-276H7.2','RP11-276H7.3'],
        ['RP11-278C7.4','YARS2'],
        ['RP11-27M24.1','SFTPA1'],
        ['RP11-27M24.1','SFTPB'],
        ['RP11-27M24.1','SFTPC'],
        ['RP11-283C24.1','SCD'],
        ['RP11-285F7.2','TMEM63A'],
        ['RP11-285G1.15','RSU1P2'],
        ['RP11-290D2.4','TPT1'],
        ['RP11-293E1.1','RP11-293E1.2'],
        ['RP11-293G6__A.2','TBCE'],
        ['RP11-293G6__A.3','TBCE'],
        ['RP11-295K3.1','SFTPA1'],
        ['RP11-295K3.1','SFTPB'],
        ['RP11-295K3.1','SFTPC'],
        ['RP11-296K13.4','TAOK1'],
        ['RP11-297L17.6','TPM3'],
        ['RP11-299L17.1','RPS24'],
        ['RP11-2C24.4','SRCAP'],
        ['RP11-302B13.5','WNT10B'],
        ['RP11-304L19.5','SNORD60'],
        ['RP11-307I14.2','RPS5'],
        ['RP11-307L14.1','RP11-307L14.2'],
        ['RP11-310N16.1','TPT1'],
        ['RP11-314A20.1','RPS12'],
        ['RP11-314N13.3','SDCBP2-AS1'],
        ['RP11-321N4.5','SYNCRIP'],
        ['RP11-323N12.5','SFTPC'],
        ['RP11-325P15.2','SSB'],
        ['RP11-326A19.5','VAPA'],
        ['RP11-329A14.1','STRADB'],
        ['RP11-334C17.5','SGSH'],
        ['RP11-336K24.4','RP11-336K24.6'],
        ['RP11-343B5.1','RPL19'],
        ['RP11-343C2.11','SNTB2'],
        ['RP11-343C2.3','SNTB2'],
        ['RP11-343H5.4','RPS14'],
        ['RP11-344E13.3','RP11-381P6.1'],
        ['RP11-344H11.4','RPL12'],
        ['RP11-345K20.2','RPL26'],
        ['RP11-350G13.1','RPSA'],
        ['RP11-350G8.3','RPSA'],
        ['RP11-353H3.1','RPL18A'],
        ['RP11-355I22.6','SYT16'],
        ['RP11-356O9.1','TTC6'],
        ['RP11-35J23.5','TLL2'],
        ['RP11-364B6.1','THAP3'],
        ['RP11-367G18.2','RPS27A'],
        ['RP11-36C20.1','RPS8'],
        ['RP11-372E1.1','RPL8'],
        ['RP11-379B18.5','RP11-666A20.4'],
        ['RP11-37O16.6','RP11-37O16.7'],
        ['RP11-383J24.6','TMEM65'],
        ['RP11-385D13.1','TRIM16L'],
        ['RP11-385F5.4','RP11-385F5.5'],
        ['RP11-392E22.10','RP11-392E22.5'],
        ['RP11-392E22.5','RP11-392E22.9'],
        ['RP11-392P7.1','RPS6'],
        ['RP11-395L14.11','WI2-1528O10.1'],
        ['RP11-397E7.1','RPL6'],
        ['RP11-3L10.3','RPL18'],
        ['RP11-400N13.3','RP11-815M8.1'],
        ['RP11-401L13.4','RPL19'],
        ['RP11-404G16.2','TF'],
        ['RP11-405A12.1','TCP1'],
        ['RP11-408P14.1','RPL6'],
        ['RP11-423H2.1','THOC3'],
        ['RP11-435F13.2','RPLP1'],
        ['RP11-442A13.1','TPT1'],
        ['RP11-444D3.1','SOX5'],
        ['RP11-446N19.1','RP11-96H19.1'],
        ['RP11-448G15.3','WDR1'],
        ['RP11-452F19.3','RP11-455P21.3'],
        ['RP11-452G18.2','RPS2P5'],
        ['RP11-45M22.1','SFTPB'],
        ['RP11-462L8.2','RPL7A'],
        ['RP11-463O12.5','VPS37B'],
        ['RP11-465L10.7','SLC12A5'],
        ['RP11-466L17.1','RP11-90C4.1'],
        ['RP11-46A10.4','STX6'],
        ['RP11-473N11.2','SUPT16H'],
        ['RP11-475N22.4','TMED10P2'],
        ['RP11-476E15.3','SORBS1'],
        ['RP11-477G18.1','RPS12'],
        ['RP11-47I22.3','RP11-47I22.4'],
        ['RP11-486O12.2','WSB1'],
        ['RP11-488L18.4','RP11-488L18.8'],
        ['RP11-492E3.1','RP11-492E3.51'],
        ['RP11-493K19.3','SFTPC'],
        ['RP11-498C9.2','THBS1'],
        ['RP11-498C9.3','SFTPC'],
        ['RP11-503N18.1','RP11-503N18.2'],
        ['RP11-503N18.1','RP11-503N18.3'],
        ['RP11-503N18.2','RP11-503N18.3'],
        ['RP11-506M13.3','RP11-773D16.1'],
        ['RP11-510I21.1','SMC5'],
        ['RP11-514O12.4','RPS6KA2'],
        ['RP11-51O6.1','RPS27A'],
        ['RP11-524O24.2','RPL26'],
        ['RP11-534L20.4','RPL7A'],
        ['RP11-536C5.7','TMEM158'],
        ['RP11-537I16.2','SENP2'],
        ['RP11-54C4.1','RPLP1'],
        ['RP11-561C5.5','RP11-561C5.6'],
        ['RP11-562A8.1','RPS20'],
        ['RP11-563H6.1','RPL10A'],
        ['RP11-564A8.4','RPL13A'],
        ['RP11-566K11.7','TCF25'],
        ['RP11-568K15.1','RP4-798P15.3'],
        ['RP11-568K15.1','SEC16B'],
        ['RP11-570G20.1','RPL15'],
        ['RP11-571M6.7','SFTPC'],
        ['RP11-578C11.2','YWHAE'],
        ['RP11-57H12.2','RP11-57H12.6'],
        ['RP11-57H12.2','TMEM56'],
        ['RP11-57H12.2','TMEM56-RWDD3'],
        ['RP11-587D21.1','RPS25'],
        ['RP11-599B13.6','TMEM107'],
        ['RP11-59N23.1','RP11-59N23.3'],
        ['RP11-613M10.8','TOMM5'],
        ['RP11-616M22.5','TPSD1'],
        ['RP11-618N24.1','TPM3'],
        ['RP11-61I13.2','RP11-61I13.3'],
        ['RP11-624L12.1','RPL17'],
        ['RP11-627K11.1','RPL13A'],
        ['RP11-632C17__A.1','RPL29'],
        ['RP11-634B7.4','TRIM58'],
        ['RP11-638I2.6','YY1'],
        ['RP11-649A18.5','RP5-1160K1.3'],
        ['RP11-649A18.7','SUMO2'],
        ['RP11-65L3.4','TTN-AS1'],
        ['RP11-661A12.7','SFTPA1'],
        ['RP11-667K14.3','RP11-667K14.4'],
        ['RP11-66N24.3','RP11-66N24.4'],
        ['RP11-682N22.1','SHFM1'],
        ['RP11-685B14.1','RPL10A'],
        ['RP11-686D22.7','RP11-686D22.8'],
        ['RP11-689P11.2','TRMT44'],
        ['RP11-68I18.10','SEMA6C'],
        ['RP11-68I3.11','SSH2'],
        ['RP11-697E2.6','RP11-697E2.7'],
        ['RP11-69L16.5','RPS3'],
        ['RP11-6O2.3','SYNM'],
        ['RP11-700P18.1','XPOT'],
        ['RP11-701H24.10','SNHG14'],
        ['RP11-701H24.4','SNHG14'],
        ['RP11-702H23.6','SERPINA10'],
        ['RP11-711M9.1','TET3'],
        ['RP11-715J22.1','RPS16'],
        ['RP11-716O23.1','RP11-716O23.2'],
        ['RP11-727F15.11','TMEM223'],
        ['RP11-727F15.13','STX5'],
        ['RP11-730G20.2','RP11-793H13.8'],
        ['RP11-753H16.3','RP11-753H16.5'],
        ['RP11-760D2.7','YWHAE'],
        ['RP11-771F20.1','RPS6'],
        ['RP11-778D9.4','RPS3'],
        ['RP11-77H9.2','RP11-77H9.5'],
        ['RP11-785H5.2','TAS2R14'],
        ['RP11-791G16.2','RPS6'],
        ['RP11-796G6.1','RPL26'],
        ['RP11-79E3.1','RPL31'],
        ['RP11-802E16.3','TCIRG1'],
        ['RP11-813N20.3','UGT2B4'],
        ['RP11-815J4.7','SSBP4'],
        ['RP11-829H16.2','RPL27'],
        ['RP11-82L18.2','SHC3'],
        ['RP11-849F2.10','RPS24'],
        ['RP11-849H4.4','SFTPC'],
        ['RP11-85G20.1','RPL18A'],
        ['RP11-864I4.1','TG'],
        ['RP11-864N7.2','RPS12'],
        ['RP11-875O11.1','SFTPC'],
        ['RP11-875O11.1','TNFRSF10B'],
        ['RP11-879F14.2','RP11-879F14.3'],
        ['RP11-87H9.2','RP11-87H9.4'],
        ['RP11-87H9.2','RP11-88I18.2'],
        ['RP11-889L3.1','RPL19'],
        ['RP11-896J10.3','SFTA3'],
        ['RP11-927P21.5','RP13-104F24.1'],
        ['RP11-92K2.2','RPS27A'],
        ['RP11-93O14.2','VPS35'],
        ['RP11-982M15.8','SYTL4'],
        ['RP11-986E7.7','SERPINA1'],
        ['RP11-989F5.3','STX2'],
        ['RP11-99L13.1','RP11-99L13.2'],
        ['RP13-131K19.2','RP13-131K19.7'],
        ['RP13-152O15.5','TCEA2'],
        ['RP3-327A19.7','UPF3B'],
        ['RP3-395M20.7','TNFRSF14'],
        ['RP3-412A9.10','SMTN'],
        ['RP3-416H24.1','TIMP3'],
        ['RP3-417G15.1','RPS24'],
        ['RP3-423B22.5','RPS27'],
        ['RP3-486I3.4','RPS5'],
        ['RP4-539M6.19','SEC14L2'],
        ['RP4-575N6.2','RPS20'],
        ['RP4-576H24.2','RP4-576H24.4'],
        ['RP4-576H24.4','SIRPA'],
        ['RP4-576H24.4','SIRPD'],
        ['RP4-583P15.14','SLC2A4RG'],
        ['RP4-583P15.15','SLC2A4RG'],
        ['RP4-613B23.1','SEC22C'],
        ['RP4-622L5.7','TMEM234'],
        ['RP4-631H13.2','ZYG11A'],
        ['RP4-635E18.8','TARDBP'],
        ['RP4-641G12.4','SNW1'],
        ['RP4-650F12.2','TNNI3K'],
        ['RP4-673D20.3','RP5-968J1.1'],
        ['RP4-734P14.4','SNRPB'],
        ['RP4-761J14.8','RPL14'],
        ['RP4-765C7.2','RPS14'],
        ['RP4-773N10.5','STRIP1'],
        ['RP4-796I17.5','RPL35A'],
        ['RP4-800G7.2','ZNF783'],
        ['RP5-1056L3.3','RPS14'],
        ['RP5-1061H20.3','RP5-1061H20.4'],
        ['RP5-1159O4.1','RP5-1159O4.2'],
        ['RP5-837O21.4','RP5-837O21.5'],
        ['RP5-837O21.5','RP5-837O21.6'],
        ['RP5-850E9.3','TG'],
        ['RP5-940J5.9','SERPINE1'],
        ['RP5-940J5.9','SFTPC'],
        ['RP5-940J5.9','SPARC'],
        ['RP5-940J5.9','TAGLN'],
        ['RP5-940J5.9','TG'],
        ['RP5-940J5.9','TGFBI'],
        ['RP5-940J5.9','THBS1'],
        ['RP5-940J5.9','TPM2'],
        ['RP5-940J5.9','TUBA1B'],
        ['RP5-966M1.6','SERPINA1'],
        ['RP5-966M1.6','TMEM110-MUSTN1'],
        ['RP5-977B1.11','RPL37A'],
        ['RP5-977B1.11','SFTPA1'],
        ['RP5-977B1.11','SFTPA2'],
        ['RP5-977B1.11','SFTPB'],
        ['RP5-977B1.11','SFTPC'],
        ['RP5-977B1.11','THBS1'],
        ['RP5-977B1.11','VIM'],
        ['RP5-977B1.11','VIM-AS1'],
        ['RP5-991G20.1','RP5-991G20.2'],
        ['RP9','RP9P'],
        ['RPL10','RPL10P15'],
        ['RPL10','RPL10P3'],
        ['RPL10A','RPL10AP2'],
        ['RPL10A','RPL10AP6'],
        ['RPL11','SFTPA2'],
        ['RPL11','SFTPB'],
        ['RPL11','SFTPC'],
        ['RPL11','TCEB3'],
        ['RPL12','RPL12L3'],
        ['RPL13','RPL13P12'],
        ['RPL13A','RPL13AP2'],
        ['RPL13A','RPL13AP20'],
        ['RPL13A','RPL13AP25'],
        ['RPL13A','RPL13AP3'],
        ['RPL13A','RPL13AP6'],
        ['RPL13A','RPL13AP7'],
        ['RPL14','RPL14P1'],
        ['RPL14','RPL14P3'],
        ['RPL14','S100A16'],
        ['RPL14','SMARCA2'],
        ['RPL14','SREBF2'],
        ['RPL14','THAP11'],
        ['RPL15','RPL15P18'],
        ['RPL15','RPL15P2'],
        ['RPL15','RPL15P20'],
        ['RPL18','RPL18P13'],
        ['RPL18','SFTPA2'],
        ['RPL18','SFTPB'],
        ['RPL18','SFTPC'],
        ['RPL18A','RPL18AP16'],
        ['RPL19','RPL19P20'],
        ['RPL23A','RPL23AP3'],
        ['RPL23AP6','RPL23AP64'],
        ['RPL23AP7','RPL23AP82'],
        ['RPL24','RPL24P2'],
        ['RPL24','RPL24P4'],
        ['RPL24','RPL24P8'],
        ['RPL27A','SNORA45'],
        ['RPL28','SFTPC'],
        ['RPL28','TMEM190'],
        ['RPL29','RPL29P12'],
        ['RPL3','RPL3P1'],
        ['RPL3','RPL3P10'],
        ['RPL3','RPL3P12'],
        ['RPL3','RPL3P2'],
        ['RPL3','RPL3P3'],
        ['RPL3','RPL3P6'],
        ['RPL3','RPL3P7'],
        ['RPL3','RPL3P9'],
        ['RPL3','SFTPA1'],
        ['RPL3','SFTPC'],
        ['RPL30','RPL30P4'],
        ['RPL31','RPL31P11'],
        ['RPL31','RPL31P49'],
        ['RPL31','RPL31P63'],
        ['RPL34','RPL34P27'],
        ['RPL34','RPL34P31'],
        ['RPL35','RPL35P5'],
        ['RPL36AL','RPL36AP21'],
        ['RPL36AL','RPL36AP45'],
        ['RPL37','RPL37P6'],
        ['RPL37','TNFSF10'],
        ['RPL37A','RPL37AP1'],
        ['RPL37A','RPL37AP8'],
        ['RPL37A','SFTPB'],
        ['RPL37A','SFTPC'],
        ['RPL37A','TG'],
        ['RPL37A','VEGFA'],
        ['RPL38','SFTPB'],
        ['RPL39','RPL39P38'],
        ['RPL39','ZNF525'],
        ['RPL4','RPL4P3'],
        ['RPL4','RPL4P4'],
        ['RPL4','RPL4P5'],
        ['RPL5','RPL5P1'],
        ['RPL5','RPL5P12'],
        ['RPL5','RPL5P17'],
        ['RPL5','RPL5P18'],
        ['RPL5','RPL5P22'],
        ['RPL5','RPL5P23'],
        ['RPL5','RPL5P24'],
        ['RPL5','RPL5P29'],
        ['RPL5','RPL5P3'],
        ['RPL5','RPL5P34'],
        ['RPL5','RPL5P4'],
        ['RPL5','RPL5P8'],
        ['RPL5','RPL5P9'],
        ['RPL7A','RPL7AP11'],
        ['RPL7A','RPL7AP30'],
        ['RPL7A','RPL7AP31'],
        ['RPL7A','RPL7AP6'],
        ['RPL7A','RPL7AP60'],
        ['RPL7A','RPL7AP66'],
        ['RPL7A','RPL7AP8'],
        ['RPL7L1','WAC-AS1'],
        ['RPL8','SFTPA1'],
        ['RPL8','SFTPA2'],
        ['RPL8','SFTPB'],
        ['RPL8','SFTPC'],
        ['RPL8','TG'],
        ['RPL8','THBS1'],
        ['RPL8','TPM2'],
        ['RPL8','VIM'],
        ['RPL9','RPL9P25'],
        ['RPL9','TIPIN'],
        ['RPL9P25','UBE2Q2P6'],
        ['RPLP0','RPLP0P6'],
        ['RPLP1','RPLP1P6'],
        ['RPLP1','SFTPB'],
        ['RPLP1','SFTPC'],
        ['RPLP2','SFTPC'],
        ['RPN2','SFTPC'],
        ['RPS10','RPS10P2'],
        ['RPS10-NUDT3','RPS10P2'],
        ['RPS11','RPS11P5'],
        ['RPS11','SFTPC'],
        ['RPS12','RPS12P23'],
        ['RPS12','SNORA33'],
        ['RPS12','SNORD101'],
        ['RPS13','RPS13P2'],
        ['RPS14','RPS14P8'],
        ['RPS15A','RPS15AP1'],
        ['RPS15A','RPS15AP10'],
        ['RPS15A','RPS15AP11'],
        ['RPS15A','RPS15AP40'],
        ['RPS16','SFTPC'],
        ['RPS16','ZNF90'],
        ['RPS17','RPS17P5'],
        ['RPS18','RPS18P12'],
        ['RPS18','RPS18P9'],
        ['RPS19','RPS19P1'],
        ['RPS19','SFTPB'],
        ['RPS19','SFTPC'],
        ['RPS19BP1','SFTPC'],
        ['RPS2','RPS2P5'],
        ['RPS2','STK39'],
        ['RPS2','YWHAQ'],
        ['RPS20','RPS20P10'],
        ['RPS20','RPS20P14'],
        ['RPS20','RPS20P24'],
        ['RPS20','RSF1'],
        ['RPS21','RPS21P4'],
        ['RPS21P1','TBCE'],
        ['RPS23','RPS23P8'],
        ['RPS24','RPS24P13'],
        ['RPS24','RPS24P7'],
        ['RPS24','RPS24P8'],
        ['RPS24P17','SLC12A3'],
        ['RPS2P46','RPS2P5'],
        ['RPS3','RPS3P6'],
        ['RPS3','SFTPC'],
        ['RPS3A','RPS3AP49'],
        ['RPS4X','RPS4XP1'],
        ['RPS4X','RPS4XP11'],
        ['RPS4X','RPS4XP13'],
        ['RPS4X','RPS4XP14'],
        ['RPS4X','RPS4XP16'],
        ['RPS4X','RPS4XP17'],
        ['RPS4X','RPS4XP2'],
        ['RPS4X','RPS4XP22'],
        ['RPS4X','RPS4XP6'],
        ['RPS4X','RPS4XP7'],
        ['RPS4X','RPS4XP8'],
        ['RPS5','SPARC'],
        ['RPS6','SFTPC'],
        ['RPS6','TG'],
        ['RPS6KA2','SFTPA2'],
        ['RPS6KA2','SFTPC'],
        ['RPS6KB2','SFTPB'],
        ['RPS7','RPS7P10'],
        ['RPS8','SNORD38A'],
        ['RPS8','SNORD46'],
        ['RPS9','RPS9P1'],
        ['RPS9','SFTPC'],
        ['RPSA','RPSAP53'],
        ['RPSA','RPSAP56'],
        ['RPSA','RPSAP58'],
        ['RPSA','RPSAP8'],
        ['RPSAP64','SNORA62'],
        ['RRAS','SFTPC'],
        ['RRBP1','SFTPA1'],
        ['RRBP1','SFTPC'],
        ['RRBP1','THBS1'],
        ['RRBP1','VIM'],
        ['RREB1','SFTPC'],
        ['RRP7A','RRP7B'],
        ['RRP7A','RRP7BP'],
        ['RSL24D1','RSL24D1P11'],
        ['RTN4','SFTPC'],
        ['RTN4','TG'],
        ['RTN4','THBS1'],
        ['RTN4','VIM-AS1'],
        ['RWDD3','TMEM56'],
        ['RXRA','SERPINA1'],
        ['RXRA','SFTPC'],
        ['RYR1','TG'],
        ['S100A10','SFTPC'],
        ['S100A10','TNS3'],
        ['S100A11','SFTPA2'],
        ['S100A16','S100A2'],
        ['S100A6','SFTPA1'],
        ['S100A6','SFTPB'],
        ['S100A6','SFTPC'],
        ['S100A6','TG'],
        ['S100A6','THBS1'],
        ['S1PR1','SFTPA1'],
        ['SAA1','SAA2'],
        ['SAA1','SAA2-SAA4'],
        ['SAA1','SAA4'],
        ['SAA1','SCD'],
        ['SAA1','SERPINA1'],
        ['SAA1','TIMP3'],
        ['SAA2','SERPINA1'],
        ['SAA2','TMBIM6'],
        ['SAA2-SAA4','SERPINA1'],
        ['SAG','VPS13C'],
        ['SAMHD1','SFTPA2'],
        ['SAP18','TG'],
        ['SAR1A','TYSND1'],
        ['SARAF','TG'],
        ['SASH1','SFTPB'],
        ['SAT1','SFTPA2'],
        ['SAT1','SFTPC'],
        ['SBDS','SBDSP1'],
        ['SBNO2','SFTPB'],
        ['SBNO2','SFTPC'],
        ['SCAF1','SFTPC'],
        ['SCAMP4','SFTPB'],
        ['SCAMP4','TPM4'],
        ['SCARB2','SFTPA2'],
        ['SCARB2','SFTPB'],
        ['SCARF1','SFTPB'],
        ['SCARNA13','SNHG10'],
        ['SCARNA17','SCARNA18'],
        ['SCARNA22','WHSC1'],
        ['SCD','SCDP1'],
        ['SCD','SFTPC'],
        ['SCD','SPARC'],
        ['SCD5','TPM4'],
        ['SCGB1A1','SFTPA2'],
        ['SCGB1A1','SFTPB'],
        ['SCGB1A1','SFTPC'],
        ['SCGB1A1','SLC34A2'],
        ['SCGB1A1','ZFP36'],
        ['SCGB1B2P','SCGB2B2'],
        ['SCNN1A','SFTPC'],
        ['SCNN1A','TNFRSF1A'],
        ['SCO2','TYMP'],
        ['SCP2','TUBB'],
        ['SCUBE3','SLC35E1'],
        ['SCUBE3','TG'],
        ['SCUBE3','TIMP1'],
        ['SDC3','SFTPA1'],
        ['SDC4','SFTPA2'],
        ['SDC4','SFTPC'],
        ['SDCBP','SDCBPP2'],
        ['SDHA','SDHAP2'],
        ['SDHA','SDHAP3'],
        ['SDHAP1','SDHAP2'],
        ['SDPR','SFTPA1'],
        ['SDPR','SFTPC'],
        ['SEC14L1','SEC14L1P1'],
        ['SEC14L1','SFTPA2'],
        ['SEC14L1','SFTPC'],
        ['SEC14L1','SNHG20'],
        ['SEC31A','SFTPC'],
        ['SEC61A1','SERPINE1'],
        ['SEC61A1','SFTPC'],
        ['SEC62','SMR3B'],
        ['SEC63','SEC63P1'],
        ['SEL1L','SFTPC'],
        ['SELENBP1','SFTPA1'],
        ['SELENBP1','SFTPC'],
        ['SELO','SERP1'],
        ['SEMA3B','SFTPC'],
        ['SEMA3G','SFTPC'],
        ['SEMA7A','TBX2'],
        ['SEPN1','SFTPB'],
        ['SEPN1','SFTPC'],
        ['SEPP1','SFTPA2'],
        ['SEPP1','SFTPC'],
        ['SEPT11','SERPINE1'],
        ['SEPT2','SFTPB'],
        ['SEPT2','SFTPC'],
        ['SEPT7','SEPT7P2'],
        ['SEPT7','SEPT7P3'],
        ['SEPT7','SEPT7P6'],
        ['SEPT7','SEPT7P7'],
        ['SEPT9','SERPINE1'],
        ['SEPT9','SFTPA1'],
        ['SEPT9','SFTPA2'],
        ['SEPT9','SFTPC'],
        ['SEPT9','SZRD1'],
        ['SEPT9','TGFBI'],
        ['SEPT9','THBS1'],
        ['SEPT9','VIM'],
        ['SERF2','SFTPC'],
        ['SERINC2','SFTPC'],
        ['SERINC3','TECR'],
        ['SERPINA1','SERPINA3'],
        ['SERPINA1','SERPINC1'],
        ['SERPINA1','SERPINF1'],
        ['SERPINA1','SFTPA1'],
        ['SERPINA1','SFTPA2'],
        ['SERPINA1','SFTPB'],
        ['SERPINA1','SFTPC'],
        ['SERPINA1','SPARC'],
        ['SERPINA1','TAT'],
        ['SERPINA1','TGM2'],
        ['SERPINA1','TIMP3'],
        ['SERPINA1','TMBIM6'],
        ['SERPINA3','SERPINA5'],
        ['SERPINB1','SFTPB'],
        ['SERPINB3','SERPINB4'],
        ['SERPINE1','SERPINH1'],
        ['SERPINE1','SFTPA2'],
        ['SERPINE1','SFTPB'],
        ['SERPINE1','SFTPC'],
        ['SERPINE1','SH3BGRL3'],
        ['SERPINE1','SHROOM3'],
        ['SERPINE1','SPARC'],
        ['SERPINE1','TAGLN'],
        ['SERPINE1','TGFBI'],
        ['SERPINE1','THBS1'],
        ['SERPINE1','TIMP1'],
        ['SERPINE1','TLN1'],
        ['SERPINE1','TNC'],
        ['SERPINE1','TP53'],
        ['SERPINE1','TPM2'],
        ['SERPINE1','TXNDC5'],
        ['SERPINE1','VIM'],
        ['SERPINE1','VIM-AS1'],
        ['SERPINE1','WDR1'],
        ['SERPINE2','TFPI2'],
        ['SERPINE2','THBS1'],
        ['SERPINE2','VIM'],
        ['SERPINE2','VIM-AS1'],
        ['SERPINF2','WDR81'],
        ['SERPING1','SFTPA1'],
        ['SERPING1','SFTPA2'],
        ['SERPING1','SFTPB'],
        ['SERPING1','SFTPC'],
        ['SERPING1','SUSD2'],
        ['SERPINH1','SPARC'],
        ['SERPINH1','THBS1'],
        ['SERPINI2','WDR49'],
        ['SESN3','TG'],
        ['SET','SETP14'],
        ['SET','SETP3'],
        ['SET','SETP4'],
        ['SETD2','SFTPC'],
        ['SF1','SFTPA1'],
        ['SF1','SFTPB'],
        ['SF1','SFTPC'],
        ['SF3A1','SFTPC'],
        ['SF3B1','SFTPC'],
        ['SF3B2','SFTPC'],
        ['SFPQ','SFTPA2'],
        ['SFPQ','SFTPB'],
        ['SFPQ','SFTPC'],
        ['SFRP1','SMAD6'],
        ['SFT2D2','TBX19'],
        ['SFTA2','SFTPA1'],
        ['SFTPA1','SFTPA2'],
        ['SFTPA1','SFTPB'],
        ['SFTPA1','SFTPC'],
        ['SFTPA1','SH2B3'],
        ['SFTPA1','SH2D3C'],
        ['SFTPA1','SIDT2'],
        ['SFTPA1','SIK1'],
        ['SFTPA1','SLC34A2'],
        ['SFTPA1','SLK'],
        ['SFTPA1','SOCS3'],
        ['SFTPA1','SOD2'],
        ['SFTPA1','SPARC'],
        ['SFTPA1','SPOCK2'],
        ['SFTPA1','SPTAN1'],
        ['SFTPA1','SPTBN1'],
        ['SFTPA1','SQSTM1'],
        ['SFTPA1','SRCAP'],
        ['SFTPA1','SREBF2'],
        ['SFTPA1','SRRM2'],
        ['SFTPA1','SSH1'],
        ['SFTPA1','STAT3'],
        ['SFTPA1','STAT6'],
        ['SFTPA1','SUSD2'],
        ['SFTPA1','SYNE1'],
        ['SFTPA1','SYNPO'],
        ['SFTPA1','TACC1'],
        ['SFTPA1','TAGLN2'],
        ['SFTPA1','TAPBP'],
        ['SFTPA1','TGM2'],
        ['SFTPA1','TGOLN2'],
        ['SFTPA1','THBD'],
        ['SFTPA1','THBS1'],
        ['SFTPA1','TIMP2'],
        ['SFTPA1','TIMP3'],
        ['SFTPA1','TLN1'],
        ['SFTPA1','TMBIM6'],
        ['SFTPA1','TNS1'],
        ['SFTPA1','TNXB'],
        ['SFTPA1','TPT1'],
        ['SFTPA1','TSC22D1'],
        ['SFTPA1','TXNIP'],
        ['SFTPA1','UBC'],
        ['SFTPA1','VEGFA'],
        ['SFTPA1','VIM'],
        ['SFTPA1','VWF'],
        ['SFTPA1','WARS'],
        ['SFTPA1','WNK1'],
        ['SFTPA1','XIST'],
        ['SFTPA1','ZBTB4'],
        ['SFTPA1','ZFP36'],
        ['SFTPA1','ZFP36L1'],
        ['SFTPA1','ZNF512B'],
        ['SFTPA2','SFTPB'],
        ['SFTPA2','SFTPC'],
        ['SFTPA2','SH2D3C'],
        ['SFTPA2','SH3BGRL3'],
        ['SFTPA2','SHANK3'],
        ['SFTPA2','SLC25A37'],
        ['SFTPA2','SLC34A2'],
        ['SFTPA2','SLC41A1'],
        ['SFTPA2','SLC44A2'],
        ['SFTPA2','SLC6A4'],
        ['SFTPA2','SLC9A3R2'],
        ['SFTPA2','SLCO2A1'],
        ['SFTPA2','SLCO2B1'],
        ['SFTPA2','SLPI'],
        ['SFTPA2','SMARCC2'],
        ['SFTPA2','SNRNP200'],
        ['SFTPA2','SNRNP70'],
        ['SFTPA2','SNX22'],
        ['SFTPA2','SOCS3'],
        ['SFTPA2','SOD3'],
        ['SFTPA2','SPARC'],
        ['SFTPA2','SPARCL1'],
        ['SFTPA2','SPOCK2'],
        ['SFTPA2','SPTAN1'],
        ['SFTPA2','SPTBN1'],
        ['SFTPA2','SQSTM1'],
        ['SFTPA2','SREBF2'],
        ['SFTPA2','SRGN'],
        ['SFTPA2','SRRM2'],
        ['SFTPA2','STAB1'],
        ['SFTPA2','STARD7'],
        ['SFTPA2','STAT3'],
        ['SFTPA2','STAT6'],
        ['SFTPA2','STOM'],
        ['SFTPA2','SUPT6H'],
        ['SFTPA2','SUSD2'],
        ['SFTPA2','SYNPO'],
        ['SFTPA2','TACC1'],
        ['SFTPA2','TACC2'],
        ['SFTPA2','TAGLN'],
        ['SFTPA2','TAGLN2'],
        ['SFTPA2','TENC1'],
        ['SFTPA2','TGFB1'],
        ['SFTPA2','TGM2'],
        ['SFTPA2','THBD'],
        ['SFTPA2','THBS1'],
        ['SFTPA2','TIMP2'],
        ['SFTPA2','TIMP3'],
        ['SFTPA2','TLN1'],
        ['SFTPA2','TMBIM6'],
        ['SFTPA2','TMPRSS2'],
        ['SFTPA2','TMSB4X'],
        ['SFTPA2','TNFAIP3'],
        ['SFTPA2','TNKS1BP1'],
        ['SFTPA2','TNS1'],
        ['SFTPA2','TNS3'],
        ['SFTPA2','TNXB'],
        ['SFTPA2','TP53I11'],
        ['SFTPA2','TPM3'],
        ['SFTPA2','TPM4'],
        ['SFTPA2','TPP1'],
        ['SFTPA2','TPT1'],
        ['SFTPA2','TSC22D3'],
        ['SFTPA2','TSPAN14'],
        ['SFTPA2','TXNIP'],
        ['SFTPA2','TYMP'],
        ['SFTPA2','UBB'],
        ['SFTPA2','UBC'],
        ['SFTPA2','UPK3B'],
        ['SFTPA2','USP36'],
        ['SFTPA2','VAT1'],
        ['SFTPA2','VEGFA'],
        ['SFTPA2','VIM'],
        ['SFTPA2','VIPR1'],
        ['SFTPA2','VWF'],
        ['SFTPA2','WASF2'],
        ['SFTPA2','WDR1'],
        ['SFTPA2','WNK1'],
        ['SFTPA2','XIST'],
        ['SFTPA2','ZBTB4'],
        ['SFTPA2','ZC3H7B'],
        ['SFTPA2','ZFP36'],
        ['SFTPA2','ZFP36L1'],
        ['SFTPA2','ZFP36L2'],
        ['SFTPA2','ZMIZ1'],
        ['SFTPA2','ZYX'],
        ['SFTPB','SFTPC'],
        ['SFTPB','SFXN3'],
        ['SFTPB','SH2B3'],
        ['SFTPB','SH3BP5-AS1'],
        ['SFTPB','SIK1'],
        ['SFTPB','SKIL'],
        ['SFTPB','SLC11A2'],
        ['SFTPB','SLC27A1'],
        ['SFTPB','SLC34A2'],
        ['SFTPB','SLC35F6'],
        ['SFTPB','SLC38A2'],
        ['SFTPB','SLC4A2'],
        ['SFTPB','SLC9A3R2'],
        ['SFTPB','SNRPN'],
        ['SFTPB','SOCS3'],
        ['SFTPB','SOD2'],
        ['SFTPB','SPARC'],
        ['SFTPB','SPARCL1'],
        ['SFTPB','SPCS3'],
        ['SFTPB','SPEN'],
        ['SFTPB','SPN'],
        ['SFTPB','SPOCK2'],
        ['SFTPB','SPTAN1'],
        ['SFTPB','SPTBN1'],
        ['SFTPB','SQSTM1'],
        ['SFTPB','SRGN'],
        ['SFTPB','SRRM2'],
        ['SFTPB','SRSF9'],
        ['SFTPB','STAB1'],
        ['SFTPB','STAT3'],
        ['SFTPB','STAT6'],
        ['SFTPB','STEAP4'],
        ['SFTPB','STOM'],
        ['SFTPB','SUMF2'],
        ['SFTPB','SUN2'],
        ['SFTPB','SUSD2'],
        ['SFTPB','SYNPO'],
        ['SFTPB','TAGLN'],
        ['SFTPB','TAGLN2'],
        ['SFTPB','TANGO2'],
        ['SFTPB','TAPBP'],
        ['SFTPB','TBC1D9B'],
        ['SFTPB','TCTN2'],
        ['SFTPB','TEK'],
        ['SFTPB','TFPI'],
        ['SFTPB','TGFBR2'],
        ['SFTPB','TGM2'],
        ['SFTPB','THBS1'],
        ['SFTPB','TIMP2'],
        ['SFTPB','TIMP3'],
        ['SFTPB','TJP1'],
        ['SFTPB','TLN1'],
        ['SFTPB','TMBIM1'],
        ['SFTPB','TMBIM6'],
        ['SFTPB','TMEM119'],
        ['SFTPB','TMEM127'],
        ['SFTPB','TMEM259'],
        ['SFTPB','TNFAIP8L1'],
        ['SFTPB','TNFRSF10B'],
        ['SFTPB','TNFRSF14'],
        ['SFTPB','TNFRSF1A'],
        ['SFTPB','TNS1'],
        ['SFTPB','TNS3'],
        ['SFTPB','TOR1AIP2'],
        ['SFTPB','TPM1'],
        ['SFTPB','TPM4'],
        ['SFTPB','TPT1'],
        ['SFTPB','TSC22D1'],
        ['SFTPB','TSPAN14'],
        ['SFTPB','TXNIP'],
        ['SFTPB','UBA1'],
        ['SFTPB','UBC'],
        ['SFTPB','USP22'],
        ['SFTPB','UTRN'],
        ['SFTPB','VIM'],
        ['SFTPB','VIPR1'],
        ['SFTPB','VWF'],
        ['SFTPB','WIZ'],
        ['SFTPB','WWTR1'],
        ['SFTPB','XIST'],
        ['SFTPB','ZFP36'],
        ['SFTPB','ZFP36L1'],
        ['SFTPB','ZFP36L2'],
        ['SFTPB','ZMIZ1'],
        ['SFTPB','ZNF587'],
        ['SFTPB','ZYX'],
        ['SFTPC','SFTPD'],
        ['SFTPC','SGK1'],
        ['SFTPC','SGSM2'],
        ['SFTPC','SH2D3C'],
        ['SFTPC','SH3BP5-AS1'],
        ['SFTPC','SHANK3'],
        ['SFTPC','SHROOM4'],
        ['SFTPC','SIK1'],
        ['SFTPC','SIN3B'],
        ['SFTPC','SIRPA'],
        ['SFTPC','SLC11A1'],
        ['SFTPC','SLC12A7'],
        ['SFTPC','SLC25A3'],
        ['SFTPC','SLC2A4RG'],
        ['SFTPC','SLC34A2'],
        ['SFTPC','SLC39A8'],
        ['SFTPC','SLC44A2'],
        ['SFTPC','SLC44A4'],
        ['SFTPC','SLC4A2'],
        ['SFTPC','SLC6A4'],
        ['SFTPC','SLC6A6'],
        ['SFTPC','SLC9A1'],
        ['SFTPC','SLC9A3R2'],
        ['SFTPC','SLCO2A1'],
        ['SFTPC','SLCO2B1'],
        ['SFTPC','SLIT2'],
        ['SFTPC','SLIT3'],
        ['SFTPC','SMAD3'],
        ['SFTPC','SMARCA2'],
        ['SFTPC','SMARCC2'],
        ['SFTPC','SNED1'],
        ['SFTPC','SNRNP200'],
        ['SFTPC','SNRNP70'],
        ['SFTPC','SOCS3'],
        ['SFTPC','SOD2'],
        ['SFTPC','SORT1'],
        ['SFTPC','SPARC'],
        ['SFTPC','SPARCL1'],
        ['SFTPC','SPOCK2'],
        ['SFTPC','SPTAN1'],
        ['SFTPC','SPTBN1'],
        ['SFTPC','SQSTM1'],
        ['SFTPC','SREBF1'],
        ['SFTPC','SRGN'],
        ['SFTPC','SRRM2'],
        ['SFTPC','SRSF4'],
        ['SFTPC','SRSF5'],
        ['SFTPC','SSU72'],
        ['SFTPC','ST6GALNAC6'],
        ['SFTPC','STAB1'],
        ['SFTPC','STARD7'],
        ['SFTPC','STAT3'],
        ['SFTPC','STAT6'],
        ['SFTPC','STEAP4'],
        ['SFTPC','STOM'],
        ['SFTPC','SUMF2'],
        ['SFTPC','SUN2'],
        ['SFTPC','SUPT5H'],
        ['SFTPC','SUSD2'],
        ['SFTPC','SYMPK'],
        ['SFTPC','SYNGR2'],
        ['SFTPC','SYNPO'],
        ['SFTPC','SYVN1'],
        ['SFTPC','TACC1'],
        ['SFTPC','TACSTD2'],
        ['SFTPC','TAGLN'],
        ['SFTPC','TAGLN2'],
        ['SFTPC','TAOK2'],
        ['SFTPC','TAPBP'],
        ['SFTPC','TBC1D2'],
        ['SFTPC','TBC1D24'],
        ['SFTPC','TBX2'],
        ['SFTPC','TENC1'],
        ['SFTPC','TEP1'],
        ['SFTPC','TGFBI'],
        ['SFTPC','TGFBR2'],
        ['SFTPC','TGM2'],
        ['SFTPC','TGOLN2'],
        ['SFTPC','THBD'],
        ['SFTPC','THBS1'],
        ['SFTPC','TIMP2'],
        ['SFTPC','TIMP3'],
        ['SFTPC','TJP1'],
        ['SFTPC','TLN1'],
        ['SFTPC','TM9SF3'],
        ['SFTPC','TMBIM6'],
        ['SFTPC','TMEM100'],
        ['SFTPC','TMEM184B'],
        ['SFTPC','TMEM259'],
        ['SFTPC','TMEM43'],
        ['SFTPC','TMEM8A'],
        ['SFTPC','TMSB10'],
        ['SFTPC','TNFAIP3'],
        ['SFTPC','TNFRSF14'],
        ['SFTPC','TNFRSF1A'],
        ['SFTPC','TNFRSF1B'],
        ['SFTPC','TNIP1'],
        ['SFTPC','TNKS1BP1'],
        ['SFTPC','TNS1'],
        ['SFTPC','TNS3'],
        ['SFTPC','TNXB'],
        ['SFTPC','TOLLIP'],
        ['SFTPC','TP53I11'],
        ['SFTPC','TPCN1'],
        ['SFTPC','TPGS2'],
        ['SFTPC','TPM4'],
        ['SFTPC','TPP1'],
        ['SFTPC','TPT1'],
        ['SFTPC','TRAPPC12'],
        ['SFTPC','TRIM28'],
        ['SFTPC','TRIM8'],
        ['SFTPC','TRIOBP'],
        ['SFTPC','TRIP10'],
        ['SFTPC','TSC22D1'],
        ['SFTPC','TSC22D3'],
        ['SFTPC','TSPAN14'],
        ['SFTPC','TSPAN3'],
        ['SFTPC','TXNDC11'],
        ['SFTPC','TXNIP'],
        ['SFTPC','UBA1'],
        ['SFTPC','UBA52'],
        ['SFTPC','UBB'],
        ['SFTPC','UBC'],
        ['SFTPC','UBE2Z'],
        ['SFTPC','UBN1'],
        ['SFTPC','UBR1'],
        ['SFTPC','UBR4'],
        ['SFTPC','UBXN11'],
        ['SFTPC','UBXN6'],
        ['SFTPC','UNC45A'],
        ['SFTPC','UPK3B'],
        ['SFTPC','UQCRC2'],
        ['SFTPC','USP22'],
        ['SFTPC','USP54'],
        ['SFTPC','USP9X'],
        ['SFTPC','VAT1'],
        ['SFTPC','VCAN'],
        ['SFTPC','VCP'],
        ['SFTPC','VEGFA'],
        ['SFTPC','VIM'],
        ['SFTPC','VIPR1'],
        ['SFTPC','VPS13D'],
        ['SFTPC','VWF'],
        ['SFTPC','WARS'],
        ['SFTPC','WASF2'],
        ['SFTPC','WBP2'],
        ['SFTPC','WDR1'],
        ['SFTPC','WDR6'],
        ['SFTPC','WDR91'],
        ['SFTPC','WDTC1'],
        ['SFTPC','WFS1'],
        ['SFTPC','WNK1'],
        ['SFTPC','WSB1'],
        ['SFTPC','XIST'],
        ['SFTPC','XRCC6'],
        ['SFTPC','YY1'],
        ['SFTPC','Z98256.1'],
        ['SFTPC','ZBTB40'],
        ['SFTPC','ZC3H7B'],
        ['SFTPC','ZEB2'],
        ['SFTPC','ZFP36'],
        ['SFTPC','ZFP36L1'],
        ['SFTPC','ZFP36L2'],
        ['SFTPC','ZMIZ1'],
        ['SFTPC','ZNF358'],
        ['SFTPC','ZNFX1'],
        ['SFTPC','ZYX'],
        ['SFXN3','TPM4'],
        ['SGK3','UHMK1'],
        ['SH3BGR','WRB'],
        ['SHANK3','TPT1'],
        ['SHC1P2','YIPF5'],
        ['SHISA5','TREX1'],
        ['SIDT2','TAGLN'],
        ['SIGLEC10','SIGLEC16'],
        ['SIGLEC11','SIGLEC16'],
        ['SIRPA','SIRPAP1'],
        ['SIRPA','SIRPB1'],
        ['SIRPAP1','SIRPB1'],
        ['SIRPB1','SIRPD'],
        ['SKP1','SKP1P1'],
        ['SLC10A3','UBL4A'],
        ['SLC12A9','TRIP6'],
        ['SLC13A3','TP53RK'],
        ['SLC1A5','THBS1'],
        ['SLC20A2','TG'],
        ['SLC25A29','YY1'],
        ['SLC25A3','SLC25A3P2'],
        ['SLC25A3','SNORA53'],
        ['SLC25A3','TG'],
        ['SLC25A37','YBX3'],
        ['SLC25A5','SLC25A5P2'],
        ['SLC25A6','TG'],
        ['SLC29A3','UNC5B'],
        ['SLC2A14','SLC2A3'],
        ['SLC34A2','SPTBN1'],
        ['SLC34A2','TIMP3'],
        ['SLC34A2','TNS1'],
        ['SLC34A2','UTRN'],
        ['SLC34A2','VIM'],
        ['SLC34A2','WARS'],
        ['SLC35E1','TG'],
        ['SLC35E2','SLC35E2B'],
        ['SLC35F6','TPM4'],
        ['SLC7A8','TG'],
        ['SLC9A7','SLC9A7P1'],
        ['SLC9A8','ZNFX1-AS1'],
        ['SLFN11','SLFN13'],
        ['SLFN12L','SLFN13'],
        ['SMAD2','SOD2'],
        ['SMARCA4','TG'],
        ['SMG1P1','SMG1P3'],
        ['SMTN','VARS2'],
        ['SNAP25','VSNL1'],
        ['SNHG1','SNORD30'],
        ['SNHG11','SNORA60'],
        ['SNHG12','SNORD99'],
        ['SNHG14','SNORD116-20'],
        ['SNHG14','SNORD116-21'],
        ['SNHG14','SNRPN'],
        ['SNHG14','SNURF'],
        ['SNHG16','SNORD1B'],
        ['SNHG19','SNORD60'],
        ['SNHG3','SNORA73B'],
        ['SNHG7','SNHG7'],
        ['SNORA20','TCP1'],
        ['SNORA31','TPT1'],
        ['SNORA5A','TBRG4'],
        ['SNORA5C','TBRG4'],
        ['SNORA76','SNORD104'],
        ['SNORD12C','ZNFX1-AS1'],
        ['SNORD3A','SNORD3B-2'],
        ['SNRNP70','TG'],
        ['SNRNP70','VIM'],
        ['SNRPN','SNURF'],
        ['SNTB2','VPS4A'],
        ['SNX19','SNX19P2'],
        ['SNX19','SNX19P3'],
        ['SNX22','TG'],
        ['SNX5','SNX5P1'],
        ['SOD1','TG'],
        ['SOD2','TG'],
        ['SOD2','TPO'],
        ['SOD2','YBX3'],
        ['SORBS2','TG'],
        ['SORD','SORD2P'],
        ['SOX12','TXNDC5'],
        ['SP140','SP140L'],
        ['SPARC','STC2'],
        ['SPARC','TAGLN'],
        ['SPARC','TGFBI'],
        ['SPARC','THBS1'],
        ['SPARC','THY1'],
        ['SPARC','TIMP1'],
        ['SPARC','TIMP3'],
        ['SPARC','TLN1'],
        ['SPARC','TNC'],
        ['SPARC','TNS1'],
        ['SPARC','TPM2'],
        ['SPARC','VIM'],
        ['SPARC','VIM-AS1'],
        ['SPARC','VWF'],
        ['SPINT2','TG'],
        ['SPP1','TIMP3'],
        ['SPSB3','TAGLN'],
        ['SPTAN1','THBS1'],
        ['SPTAN1','VIM'],
        ['SPTB','TG'],
        ['SPTBN1','TG'],
        ['SPTBN1','TXNIP'],
        ['SPTBN1','VIM'],
        ['SQSTM1','TG'],
        ['SRP72','SRP72P2'],
        ['SRP9','SRP9P1'],
        ['SRPK2','SRPK2P'],
        ['SRRM2','TG'],
        ['SRRM2','THBS1'],
        ['SRRM2','TIMP3'],
        ['SRRM2','TNKS1BP1'],
        ['SRRM2','VEGFC'],
        ['SRSF1','TOMM40'],
        ['SRXN1','TG'],
        ['SSR3','ZHX3'],
        ['SSSCA1','TAX1BP3'],
        ['ST3GAL1','TG'],
        ['ST6GALNAC4','ST6GALNAC6'],
        ['STAG3L2','STAG3L3'],
        ['STARD7','TPT1'],
        ['STAT1','STAT4'],
        ['STC1','VIM'],
        ['STK19','STK19B'],
        ['STK24','STK24P1'],
        ['STON1','STON1-GTF2A1L'],
        ['STYXL1','TMEM120A'],
        ['SULT1A1','SULT1A2'],
        ['SULT2A1','TAT'],
        ['SUMO1','SUMO1P4'],
        ['SUMO3','UBE2G2'],
        ['SUN2','VAPA'],
        ['SVIL','SVILP1'],
        ['SYNE2','TG'],
        ['SYNJ2BP','SYNJ2BP-COX16'],
        ['SYNM','TG'],
        ['SYNPO','TG'],
        ['SYNPO2','TPO'],
        ['TAF9','TAF9'],
        ['TAGLN','TGFBI'],
        ['TAGLN','THBS1'],
        ['TAGLN','TIMP1'],
        ['TAGLN','TIMP2'],
        ['TAGLN','TPM2'],
        ['TAGLN','VIM'],
        ['TAGLN','VIM-AS1'],
        ['TAGLN2','TAGLN2P1'],
        ['TAOK2','TMEM219'],
        ['TAP1','XXBAC-BPG246D15.8'],
        ['TBC1D9','TNRC18P1'],
        ['TBCEL','TECTA'],
        ['TBX2','ZDHHC7'],
        ['TCEAL2','TCEAL4'],
        ['TCEAL3','TCEAL6'],
        ['TDGF1','TDGF1P3'],
        ['TFF1','TFF2'],
        ['TFPI2','VIM'],
        ['TG','TIMP3'],
        ['TG','TLN1'],
        ['TG','TMBIM6'],
        ['TG','TMED2'],
        ['TG','TMEM66'],
        ['TG','TNNT1'],
        ['TG','TNNT3'],
        ['TG','TNS1'],
        ['TG','TPM2'],
        ['TG','TPO'],
        ['TG','TPT1'],
        ['TG','TSC22D3'],
        ['TG','TSPYL2'],
        ['TG','TTN'],
        ['TG','TXNIP'],
        ['TG','UBA1'],
        ['TG','UBA52'],
        ['TG','UBC'],
        ['TG','UBE2G2'],
        ['TG','USP22'],
        ['TG','VCP'],
        ['TG','VEGFA'],
        ['TG','VIM'],
        ['TG','VIM-AS1'],
        ['TG','VMP1'],
        ['TG','WDR6'],
        ['TG','XIRP1'],
        ['TG','YME1L1'],
        ['TG','YWHAZ'],
        ['TG','ZFP36L1'],
        ['TGFB1','THBS1'],
        ['TGFBI','THBS1'],
        ['TGFBI','TIMP1'],
        ['TGFBI','TIMP3'],
        ['TGFBI','TLN1'],
        ['TGFBI','TNC'],
        ['TGFBI','UBA1'],
        ['TGFBI','VIM'],
        ['TGFBI','VIM-AS1'],
        ['TGM2','THBS1'],
        ['TGM2','TNS1'],
        ['THBS1','THBS2'],
        ['THBS1','THY1'],
        ['THBS1','TIMP1'],
        ['THBS1','TIMP2'],
        ['THBS1','TIMP3'],
        ['THBS1','TLN1'],
        ['THBS1','TMSB10'],
        ['THBS1','TNC'],
        ['THBS1','TPM1'],
        ['THBS1','TUBA1B'],
        ['THBS1','TUBB'],
        ['THBS1','UBA1'],
        ['THBS1','VAT1'],
        ['THBS1','VCAN'],
        ['THBS1','VIM'],
        ['THBS1','VIM-AS1'],
        ['THBS1','WDR1'],
        ['THBS1','ZYX'],
        ['THRAP3','THRAP3P1'],
        ['THY1','VIM-AS1'],
        ['TIMM23','TIMM23B'],
        ['TIMP1','VIM'],
        ['TIMP3','TLN1'],
        ['TIMP3','TPM2'],
        ['TIMP3','TXNIP'],
        ['TIMP3','VIM'],
        ['TIMP3','VIM-AS1'],
        ['TIMP3','VWF'],
        ['TIPIN','UBE2Q2P6'],
        ['TLE1','TLE1P1'],
        ['TLN1','VIM'],
        ['TLN1','VIM-AS1'],
        ['TLR1','TLR6'],
        ['TM7SF2','VPS51'],
        ['TMBIM6','TPO'],
        ['TMED9','VIM-AS1'],
        ['TMEM107','VAMP2'],
        ['TMEM14B','TMEM14C'],
        ['TMEM203','TPRN'],
        ['TMPO','ZFAND5'],
        ['TMSB10','TMSB10P1'],
        ['TMSB10','TPM2'],
        ['TMSB10','VIM'],
        ['TMSB10','VIM-AS1'],
        ['TMSB4X','TMSB4XP2'],
        ['TMSB4X','TMSB4XP4'],
        ['TMSB4X','TMSB4XP8'],
        ['TMSB4X','TRIM28'],
        ['TNC','VIM'],
        ['TNC','VIM-AS1'],
        ['TNFSF10','TVP23B'],
        ['TNNT2','TTN'],
        ['TNS1','VEGFA'],
        ['TOMM20','TOMM20P4'],
        ['TPM1','TPM2'],
        ['TPM1','UBE2M'],
        ['TPM1','VIM'],
        ['TPM2','TUBA1B'],
        ['TPM2','TUBB6'],
        ['TPM2','UBXN11'],
        ['TPM2','VIM'],
        ['TPM3','TPM3P1'],
        ['TPMT','TPMTP1'],
        ['TPO','TTN'],
        ['TPPP3','ZDHHC1'],
        ['TPSAB1','TPSB2'],
        ['TPSAB1','TPSD1'],
        ['TPSB2','TPSD1'],
        ['TPT1','TPT1P1'],
        ['TPT1','TPT1P11'],
        ['TPT1','TPT1P4'],
        ['TPT1','TPT1P5'],
        ['TPT1','TPT1P6'],
        ['TPT1','TPT1P9'],
        ['TPT1','ZFP36L1'],
        ['TPTE2','TPTE2P5'],
        ['TRAC','TRAJ5'],
        ['TRAC','TRAJ52'],
        ['TRAC','TRAV1-1'],
        ['TRAC','TRAV1-2'],
        ['TRAC','TRAV10'],
        ['TRAC','TRAV12-1'],
        ['TRAC','TRAV12-2'],
        ['TRAC','TRAV12-3'],
        ['TRAC','TRAV13-1'],
        ['TRAC','TRAV13-2'],
        ['TRAC','TRAV14DV4'],
        ['TRAC','TRAV17'],
        ['TRAC','TRAV19'],
        ['TRAC','TRAV2'],
        ['TRAC','TRAV20'],
        ['TRAC','TRAV21'],
        ['TRAC','TRAV22'],
        ['TRAC','TRAV23DV6'],
        ['TRAC','TRAV24'],
        ['TRAC','TRAV25'],
        ['TRAC','TRAV26-1'],
        ['TRAC','TRAV26-2'],
        ['TRAC','TRAV27'],
        ['TRAC','TRAV29DV5'],
        ['TRAC','TRAV3'],
        ['TRAC','TRAV35'],
        ['TRAC','TRAV36DV7'],
        ['TRAC','TRAV38-1'],
        ['TRAC','TRAV38-2DV8'],
        ['TRAC','TRAV39'],
        ['TRAC','TRAV4'],
        ['TRAC','TRAV41'],
        ['TRAC','TRAV5'],
        ['TRAC','TRAV6'],
        ['TRAC','TRAV8-1'],
        ['TRAC','TRAV8-3'],
        ['TRAC','TRAV8-4'],
        ['TRAC','TRAV8-6'],
        ['TRAC','TRAV9-2'],
        ['TRAJ39','TRAV12-2'],
        ['TRAJ5','TRAV19'],
        ['TRAV18','TRAV19'],
        ['TRBC2','TRBV10-2'],
        ['TRBC2','TRBV10-3'],
        ['TRBC2','TRBV11-2'],
        ['TRBC2','TRBV13'],
        ['TRBC2','TRBV14'],
        ['TRBC2','TRBV18'],
        ['TRBC2','TRBV19'],
        ['TRBC2','TRBV2'],
        ['TRBC2','TRBV20-1'],
        ['TRBC2','TRBV24-1'],
        ['TRBC2','TRBV25-1'],
        ['TRBC2','TRBV27'],
        ['TRBC2','TRBV28'],
        ['TRBC2','TRBV29-1'],
        ['TRBC2','TRBV3-1'],
        ['TRBC2','TRBV30'],
        ['TRBC2','TRBV4-1'],
        ['TRBC2','TRBV4-2'],
        ['TRBC2','TRBV5-1'],
        ['TRBC2','TRBV5-4'],
        ['TRBC2','TRBV5-5'],
        ['TRBC2','TRBV5-6'],
        ['TRBC2','TRBV6-1'],
        ['TRBC2','TRBV6-5'],
        ['TRBC2','TRBV6-6'],
        ['TRBC2','TRBV7-3'],
        ['TRBC2','TRBV7-7'],
        ['TRBC2','TRBV7-8'],
        ['TRBC2','TRBV7-9'],
        ['TRBC2','TRBV9'],
        ['TRBC2','U66061.31'],
        ['TRBV10-2','TRBV25-1'],
        ['TRBV10-2','TRBV28'],
        ['TRBV10-3','TRBV25-1'],
        ['TRBV12-4','TRBV25-1'],
        ['TRBV12-4','TRBV28'],
        ['TRBV18','TRBV25-1'],
        ['TRBV18','TRBV28'],
        ['TRBV19','TRBV25-1'],
        ['TRBV19','TRBV28'],
        ['TRBV2','TRBV25-1'],
        ['TRBV2','TRBV28'],
        ['TRBV20-1','TRBV25-1'],
        ['TRBV20-1','TRBV28'],
        ['TRBV25-1','TRBV27'],
        ['TRBV25-1','TRBV29-1'],
        ['TRBV25-1','TRBV3-1'],
        ['TRBV25-1','TRBV30'],
        ['TRBV25-1','TRBV4-1'],
        ['TRBV25-1','TRBV4-2'],
        ['TRBV25-1','TRBV5-1'],
        ['TRBV25-1','TRBV6-5'],
        ['TRBV25-1','TRBV6-6'],
        ['TRBV25-1','TRBV7-9'],
        ['TRBV25-1','TRBV9'],
        ['TRBV27','TRBV28'],
        ['TRBV28','TRBV29-1'],
        ['TRBV28','TRBV3-1'],
        ['TRBV28','TRBV4-1'],
        ['TRBV28','TRBV4-2'],
        ['TRBV28','TRBV5-1'],
        ['TRBV28','TRBV6-5'],
        ['TRBV28','TRBV6-6'],
        ['TRBV28','TRBV7-9'],
        ['TRBV28','TRBV9'],
        ['TRBV7-1','TRBV7-3'],
        ['TRBV7-3','TRBV7-8'],
        ['TRDC','TRDV1'],
        ['TRDC','TRDV2'],
        ['TREML2','TREML5P'],
        ['TRGC1','TRGC2'],
        ['TRGC1','TRGV10'],
        ['TRGC1','TRGV2'],
        ['TRGC2','TRGV10'],
        ['TRGC2','TRGV2'],
        ['TRGC2','TRGV5'],
        ['TRGC2','TRGV9'],
        ['TRGV2','TRGV4'],
        ['TRIM16','TRIM16L'],
        ['TRIM28','TRIO'],
        ['TSLP','WDR36'],
        ['TSPAN33','Y_RNA'],
        ['TTC3','TTC3P1'],
        ['TTC30A','TTC30B'],
        ['TTLL11','TTLL11-IT1'],
        ['TTTY15','USP9Y'],
        ['TTYH3','VIM'],
        ['TUBA1A','TUBA1B'],
        ['TUBA1A','TUBA1C'],
        ['TUBA1A','TUBAP2'],
        ['TUBA1B','TUBA1C'],
        ['TUBA1B','TUBAP2'],
        ['TUBA3C','TUBA3E'],
        ['TUBA3D','TUBA3E'],
        ['TUBA4A','TUBA4B'],
        ['TUBA4A','TUBAP2'],
        ['TUBB','TUBBP1'],
        ['TUBB3','TUBB6'],
        ['TUBB4A','TUBB4B'],
        ['TUBB4B','VIM'],
        ['TUBG1','TUBG2'],
        ['TVP23C','TVP23C-CDRT4'],
        ['TWIST1','UBALD2'],
        ['TXN','TXNP6'],
        ['UBA1','VIM'],
        ['UBA52','UBC'],
        ['UBB','UBBP1'],
        ['UBB','UBBP4'],
        ['UBB','UBC'],
        ['UBB','VIM'],
        ['UBE2V1','UBE2V1P2'],
        ['UFSP2','Y_RNA'],
        ['UGT2A1','UGT2A2'],
        ['UGT2B10','UGT2B7'],
        ['UGT2B11','UGT2B7'],
        ['UGT2B15','UGT2B17'],
        ['UGT2B26P','UGT2B4'],
        ['UGT2B28','UGT2B7'],
        ['UPP2','UPP2-IT1'],
        ['UQCRB','UQCRBP1'],
        ['UQCRFS1','UQCRFS1P1'],
        ['UQCRH','UQCRHL'],
        ['USP32','USP32P1'],
        ['USP43','WDR16'],
        ['USP6NL','USP6NL-IT1'],
        ['VAMP5','VAMP8'],
        ['VDAC1','VDAC1P11'],
        ['VIM','WDR1'],
        ['VIM','ZYX'],
        ['VN1R37P','VN1R40P'],
        ['VWF','VWFP1'],
        ['WBP11','WBP11P1'],
        ['WDR45B','WDR45BP1'],
        ['WHAMM','WHAMMP3'],
        ['WHAMML1','WHAMMP2'],
        ['XPOT','XPOTP1'],
        ['XRCC6','XRCC6P1'],
        ['XRCC6','XRCC6P2'],
        ['XRCC6','XRCC6P4'],
        ['XRCC6','XRCC6P5'],
        ['YBX1','YBX1P1'],
        ['YBX3','YBX3P1'],
        ['YWHAE','YWHAEP1'],
        ['YWHAE','YWHAEP5'],
        ['YWHAZ','YWHAZP2'],
        ['YWHAZ','YWHAZP6'],
        ['ZBTB40','ZBTB40-IT1'],
        ['ZBTB8OS','ZBTB8OSP2'],
        ['ZDHHC11','ZDHHC11B'],
        ['ZFP2','ZNF354B'],
        ['ZNF100','ZNF430'],
        ['ZNF137P','ZNF701'],
        ['ZNF141','ZNF876P'],
        ['ZNF165','ZSCAN12P1'],
        ['ZNF177','ZNF559'],
        ['ZNF181','ZNF302'],
        ['ZNF19','ZNF23'],
        ['ZNF197','ZNF660'],
        ['ZNF221','ZNF224'],
        ['ZNF226','ZNF227'],
        ['ZNF264','ZNF805'],
        ['ZNF28','ZNF468'],
        ['ZNF28','ZNF816'],
        ['ZNF28','ZNF860'],
        ['ZNF286A','ZNF286B'],
        ['ZNF32-AS2','ZNF32-AS3'],
        ['ZNF32-AS2','ZNF485'],
        ['ZNF33A','ZNF33B'],
        ['ZNF350','ZNF615'],
        ['ZNF354A','ZNF354B'],
        ['ZNF37A','ZNF37BP'],
        ['ZNF417','ZNF587'],
        ['ZNF419','ZNF773'],
        ['ZNF431','ZNF714'],
        ['ZNF432','ZNF841'],
        ['ZNF510','ZNF782'],
        ['ZNF559','ZNF559-ZNF177'],
        ['ZNF561','ZNF562'],
        ['ZNF577','ZNF649'],
        ['ZNF580','ZNF581'],
        ['ZNF586','ZNF776'],
        ['ZNF587','ZNF814'],
        ['ZNF587B','ZNF814'],
        ['ZNF607','ZNF781'],
        ['ZNF761','ZNF765'],
        ['ABHD11','SNAP23'],
        ['ABLIM1','BTK'],
        ['AC004066.3','PPA2'],
        ['AC005037.3','NDUFB3'],
        ['AC006116.14','AC006116.22'],
        ['AC015849.16','CCL5'],
        ['AC074141.3','ZNF577'],
        ['AC104532.2','SURF1'],
        ['ACAA1','FBP1'],
        ['ACADSB','ARHGAP32'],
        ['ACOX1','DNAL1'],
        ['ACSL1','COL4A2'],
        ['ACSL1','IPO11'],
        ['ACTB','TOMM20'],
        ['ACTN4','BCAP31'],
        ['ADCY10','MPC2'],
        ['AF131217.1','MSMB'],
        ['AIG1','GSK3B'],
        ['AIG1','PHF20L1'],
        ['AIM1','C6ORF203'],
        ['AIMP1','ZNF605'],
        ['AKAP12','RBM39'],
        ['AKR1A1','NBR2'],
        ['ALDOA','NUDC'],
        ['ANO6','BROX'],
        ['ANOS2P','TMSB4Y'],
        ['ANXA3','ZNF350'],
        ['AP003068.18','AP003068.23'],
        ['AP1B1','QRICH1'],
        ['AP3D1','RNF10'],
        ['APCDD1L-AS1','VAPB'],
        ['APPL1','COPA'],
        ['ARFGAP2','CIB1'],
        ['ARHGEF28','MTX2'],
        ['ARL2-SNX15','WSB1'],
        ['ARNT','CTSK'],
        ['ARPC1A','MMD'],
        ['ARPC1B','PDE3A'],
        ['ARPC2','LCORL'],
        ['ASAH1','B2M'],
        ['ASAH1','PCM1'],
        ['ASB12','MTMR8'],
        ['ASCC3','PDE10A'],
        ['ATG13','NRAS'],
        ['ATG13','SMIM14'],
        ['ATG7','RAF1'],
        ['ATL3','RAD17'],
        ['ATM','LINC01091'],
        ['ATP10B','UPK1A'],
        ['ATP13A5-AS1','HRASLS'],
        ['ATP2C1','MBNL1'],
        ['ATP5B','MIR762HG'],
        ['ATP9A','NFATC2'],
        ['ATRAID','GTF3C2'],
        ['B2M','CD36'],
        ['B2M','CD99'],
        ['B2M','ERV3-1'],
        ['B2M','KIF2A'],
        ['B2M','MYCBP'],
        ['B2M','SQRDL'],
        ['B2M','TLN1'],
        ['B2M','TMEM91'],
        ['B2M','TRAK2'],
        ['B2M','WIPF1'],
        ['B3GNT2','COMMD1'],
        ['BAZ2B','MARCH7'],
        ['BCAS1','MID1IP1'],
        ['BCAS1','PRKAA1'],
        ['BCL10','NIPAL1'],
        ['BHLHE41','CTR9'],
        ['BHMT','ZNF414'],
        ['BLOC1S6','GNG4'],
        ['BLOC1S6','SMAD4'],
        ['BPNT1','TSPAN31'],
        ['BZW1','NORAD'],
        ['C11ORF63','GOLGA4'],
        ['C11ORF80','SF1'],
        ['C12ORF45','LYRM2'],
        ['C15ORF52','PGPEP1'],
        ['C16ORF72','GTPBP10'],
        ['C21ORF62-AS1','HOMER1'],
        ['C6ORF25','CCL5'],
        ['C6ORF25','F2R'],
        ['C6ORF25','LIMS1'],
        ['C6ORF25','PLEKHB2'],
        ['C6ORF25','SNAP29'],
        ['C6ORF25','STAT3'],
        ['C6ORF25','TRIM58'],
        ['C7ORF55','TXNDC17'],
        ['C7ORF55-LUC7L2','TXNDC17'],
        ['C7ORF73','NSUN5'],
        ['CA2','MEIS1'],
        ['CA5B','NT5C'],
        ['CALM3','GMPR2'],
        ['CAMK2D','GNAS'],
        ['CAMTA1','CRYL1'],
        ['CASP6','PLA2G12A'],
        ['CASP6','TRA2A'],
        ['CASP8','FAM210B'],
        ['CASP8','LONP2'],
        ['CAT','OGFOD1'],
        ['CCAR1','SP140L'],
        ['CCDC113','CEP70'],
        ['CCDC34','LGR4'],
        ['CCDC69','TPMT'],
        ['CCL5','TMED4'],
        ['CCNL1','SMG1'],
        ['CCSER1','FUBP1'],
        ['CCT6A','CCT6P1'],
        ['CD2AP','GTF3C2'],
        ['CDC37','DDX42'],
        ['CDC42EP3','LINC00211'],
        ['CENPC','NR2F2-AS1'],
        ['CENPU','KIAA0232'],
        ['CERS2','FTH1'],
        ['CFD','ICA1'],
        ['CFL1','LDHA'],
        ['CHCHD2','TCEA3'],
        ['CHD1L','GAMT'],
        ['CHD2','TRIM52'],
        ['CHFR','YKT6'],
        ['CHP1','FBXO27'],
        ['CHP1','NAA40'],
        ['CHRAC1','PAXIP1-AS1'],
        ['CHURC1','GLMP'],
        ['CIAO1','ZNF264'],
        ['CLCC1','ZNF33A'],
        ['CLCN3','EGF'],
        ['CLCN3','RGS18'],
        ['CLHC1','LSMEM1'],
        ['CMBL','CNN2'],
        ['CMBL','LIG3'],
        ['CNPY2','PARP3'],
        ['CNST','MOB1A'],
        ['COL4A3BP','NSA2'],
        ['COMMD2','ICK'],
        ['COMMD7','FTH1'],
        ['COQ10B','R3HDM2'],
        ['COX10-AS1','SH2D4A'],
        ['COX6B1','MDK'],
        ['COX7B','XPR1'],
        ['CRELD2','DPF2'],
        ['CSNK1E','FBXO21'],
        ['CSTB','UBE2L3'],
        ['CTSA','TJP2'],
        ['CTSD','ZYX'],
        ['CTSL','ZC3H14'],
        ['CUL4B','MBD3'],
        ['CYB561D1','VPS53'],
        ['CYCS','ZNF561'],
        ['DAP3','UBAP2L'],
        ['DAPP1','SNAP23'],
        ['DBT','SFXN2'],
        ['DCAF11','PSME1'],
        ['DCAF13','RGS10'],
        ['DCLRE1A','IKZF5'],
        ['DDI2','SUMF2'],
        ['DEGS2','PALLD'],
        ['DHRS2','ZBTB40'],
        ['DMKN','EDEM2'],
        ['DNM1L','GIT2'],
        ['DRAM1','NR3C1'],
        ['DYRK2','EIF1'],
        ['E2F3','VPS53'],
        ['EIF1','HNRNPA0'],
        ['EIF2AK1','TMSB4X'],
        ['EIF2AK4','ZNF121'],
        ['EIF3I','HP1BP3'],
        ['EIF3M','FDFT1'],
        ['ELF1','HMGB1'],
        ['ENC1','UPK1A'],
        ['ERCC6L2','TMEM245'],
        ['ERI3','NOL10'],
        ['ERICH1','ZNF624'],
        ['ERP27','PSMB4'],
        ['ERV3-1','SDPR'],
        ['ESRP1','WRNIP1'],
        ['ESRP2','FMO9P'],
        ['ESRRA','KRT18'],
        ['ESRRG','NEDD9'],
        ['EVI5','PTCD3'],
        ['EXOC1','UPK1A'],
        ['EXOC7','RABEP1'],
        ['EZR','NBEAL2'],
        ['F2R','LPCAT3'],
        ['FAM117B','YWHAB'],
        ['FAM120AOS','LYST'],
        ['FAM210B','MRPL49'],
        ['FAM213A','LIN7C'],
        ['FAM63B','GTPBP3'],
        ['FAM89B','KLC1'],
        ['FASTKD2','ZNF638'],
        ['FBP1','UPF1'],
        ['FBXL20','PTGIR'],
        ['FGFR1OP2','MBNL1'],
        ['FIS1','HSPE1'],
        ['FKBP3','TOP2B'],
        ['FRMD4B','MALAT1'],
        ['FRS2','FTL'],
        ['FTH1','POLR2I'],
        ['FTH1','TSC22D1'],
        ['FTSJ3','SAMD11'],
        ['G3BP1','TNFAIP8'],
        ['GAN','TMEM50B'],
        ['GAPDH','TUBB4B'],
        ['GCNT3','PGPEP1'],
        ['GFOD2','VTI1B'],
        ['GGACT','ZC3H8'],
        ['GGPS1','MAPKAPK5'],
        ['GLIS3','RFX3'],
        ['GLRX3','SLC27A2'],
        ['GLTP','PCBD2'],
        ['GLTP','SAR1B'],
        ['GLTP','SUMF2'],
        ['GMPR','HCG18'],
        ['GNPTG','ZNF502'],
        ['GOLM1','SOCS6'],
        ['GSR','ZNF91'],
        ['GSTK1','UBA52'],
        ['GSTO2','PRRC2C'],
        ['HBP1','KMT2E'],
        ['HECA','XXBAC-BPG32J3.22'],
        ['HERPUD2','MSMO1'],
        ['HERPUD2','SEPT7-AS1'],
        ['HIST1H2AG','HIST1H4I'],
        ['HIST1H2AG','TSC22D1'],
        ['HIST1H2BF','HIST1H4E'],
        ['HIST1H2BH','TRIM13'],
        ['HMG20A','SRSF8'],
        ['HMG20B','TAX1BP3'],
        ['HNRNPA2B1','SKAP2'],
        ['HNRNPF','ZCCHC24'],
        ['HNRNPK','KIF2A'],
        ['HPGD','HSPA8'],
        ['HPS1','SMG1'],
        ['HS1BP3','PDF'],
        ['IFNAR2','NFX1'],
        ['IFT80','TMEM87A'],
        ['IL1R1','MRPS27'],
        ['INO80B-WBP1','MYL12B'],
        ['ISG20L2','SNAP23'],
        ['ITM2B','LINC00534'],
        ['ITM2B','MAPRE2'],
        ['JMJD7','TRAK1'],
        ['JUN','SLAIN2'],
        ['KCTD11','MIR4458HG'],
        ['KDM5B','PPP1R12B'],
        ['KHK','PDZK1IP1'],
        ['KIAA0196','VGLL4'],
        ['KIAA0368','N4BP2L2'],
        ['KIAA0391','NSD1'],
        ['KIAA0513','ORAI2'],
        ['KIAA1217','PPFIA1'],
        ['KIF13A','RPL8'],
        ['KIF2A','NDUFS4'],
        ['KLK4','KLKP1'],
        ['KRT19','MFN2'],
        ['KRT19','TOP1'],
        ['KRT7','P2RX2'],
        ['LEPROT','THOC2'],
        ['LINC00472','RALGPS2'],
        ['LINC00969','PYGL'],
        ['LINC00989','LINC01088'],
        ['LLPH','LYZ'],
        ['LNX2','MTHFR'],
        ['LRRFIP1','PCBD2'],
        ['LYPLAL1','SMYD3'],
        ['MAFF','STAP2'],
        ['MALAT1','SOD2'],
        ['MALAT1','VPS36'],
        ['MAP3K7CL','PPBP'],
        ['MARC2','ZFP91-CNTF'],
        ['MAX','SFI1'],
        ['MBD5','ORC4'],
        ['MCCC2','SMIM14'],
        ['MDH1','TMBIM6'],
        ['MDM2','VPS4B'],
        ['MFSD11','RND3'],
        ['MICU1','UPK1A'],
        ['MID2','PTK2'],
        ['MIF4GD','NFKBIA'],
        ['MKL2','PARN'],
        ['MKRN2OS','NONO'],
        ['MLPH','VPS28'],
        ['MNS1','TEX9'],
        ['MPI','SPINK1'],
        ['MPP7','RAB18'],
        ['MRPL14','RAB40C'],
        ['MRPL9','PI4KA'],
        ['MRPS27','SOCS2-AS1'],
        ['MRPS28','TPD52'],
        ['MTFMT','XXBAC-BPG299F13.17'],
        ['MTURN','ZNRF2'],
        ['NAP1L1','SEMA4D'],
        ['NCBP2','NUDT19'],
        ['NCCRP1','TMF1'],
        ['NDUFB10','USO1'],
        ['NFKBIZ','TGM4'],
        ['NIF3L1','ORC4'],
        ['NME2','SLK'],
        ['NQO1','ZNF486'],
        ['NRIP1','USP25'],
        ['NSD1','PWWP2A'],
        ['NSMCE2','PLCL1'],
        ['NT5C2','SNAP23'],
        ['NUDCD1','SNX16'],
        ['OCEL1','TMEM97'],
        ['OIP5-AS1','RSU1'],
        ['OPA3','ZNF451'],
        ['PACRG','TMEM242'],
        ['PAFAH1B2','USP54'],
        ['PAK2','SMIM14'],
        ['PANK3','TRIM68'],
        ['PAQR7','RPS23'],
        ['PCLO','UCP2'],
        ['PCMTD1','PXDNL'],
        ['PDXK','ZNF593'],
        ['PEA15','S100A11'],
        ['PFDN6','RNF19B'],
        ['PLCB3','TXLNA'],
        ['PLPP1','RBM47'],
        ['PMPCB','UBE2L6'],
        ['POLR3H','PRPSAP2'],
        ['PPARG','TSEN2'],
        ['PPARGC1A','RSF1'],
        ['PRICKLE2','RPRD1A'],
        ['PRKAR1B','RASA3'],
        ['PSPC1','ZMYM2'],
        ['PTN','TPK1'],
        ['QPRT','TMED4'],
        ['RAB2A','SGMS1'],
        ['RAB31','RASA3'],
        ['RAB31','VAPA'],
        ['RAI14','RTN4'],
        ['RASSF8','SSPN'],
        ['RBFA','ZMYND11'],
        ['RBM3','TBC1D25'],
        ['RBM5','RHOA'],
        ['RBM6','RHOA'],
        ['REPS2','SETDB1'],
        ['REXO2','RPUSD4'],
        ['RFFL','RUBCN'],
        ['RNF185','UQCR10'],
        ['RNF220','TRIM28'],
        ['RNF24','SLC35E1'],
        ['RNGTT','UTRN'],
        ['RNH1','VAMP2'],
        ['RNPS1','SRSF5'],
        ['RSPH1','VPS13A'],
        ['S100P','YTHDF2'],
        ['SAMD8','SELT'],
        ['SENP3-EIF4A1','UBL5'],
        ['SEPT2','SKIL'],
        ['SLC2A6','SMC6'],
        ['SLC35E3','THNSL1'],
        ['SNAP23','TRIM58'],
        ['SPP1','UBE2D1'],
        ['SSR2','TRIM25'],
        ['ST3GAL1','TCEA1'],
        ['TBC1D9','TM4SF1'],
        ['THSD4','TPD52'],
        ['TMED4','ZNFX1'],
        ['TMEM198B','YWHAB'],
        ['TMEM41B','ZNF217'],
        ['TOMM70','ZNF490'],
        ['TPI1','TRIM27'],
        ['TPM4','UBXN2B'],
        ['UQCRC2','ZSWIM7'],
        ['VPS8','ZNF699'],
        ['ZNF267','ZNF720'],
        ['ABAT','METTL22'],
        ['AC005062.2','MACC1'],
        ['AC007036.5','MTURN'],
        ['AC007249.3','RP11-320M2.1'],
        ['AC009228.1','FAM228A'],
        ['AC009961.3','BAZ2B'],
        ['AC016730.1','AC092635.1'],
        ['AC022153.1','LLNLF-65H9.1'],
        ['AC024592.12','NDUFA11'],
        ['AC074183.3','SNHG7'],
        ['AC074286.1','NFE2L2'],
        ['AC078883.3','AC093818.1'],
        ['ACRBP','LPAR5'],
        ['ACTB','ACTG1'],
        ['ADAP2','RNF135'],
        ['AF127936.7','NRIP1'],
        ['AF127936.9','NRIP1'],
        ['AK6','TAF9'],
        ['AMZ2','RP11-147L13.15'],
        ['APOBEC3A','APOBEC3B'],
        ['ARHGDIB','ERP27'],
        ['ARHGEF38','ARHGEF38-IT1'],
        ['ATP5E','ATP5EP2'],
        ['ATPAF1','EFCAB14'],
        ['ATXN1L','IST1'],
        ['ATXN7','PSMD6-AS2'],
        ['AZGP1','GJC3'],
        ['BANP','RP11-426C22.5'],
        ['BMS1P20','LL22NC03-80A10.6'],
        ['BST1','FAM200B'],
        ['C15ORF57','CBX3'],
        ['C19ORF18','ZNF606'],
        ['C1QBP','CTC-524C5.2'],
        ['C3ORF62','USP4'],
        ['C6ORF163','SMIM8'],
        ['C9ORF172','PHPT1'],
        ['CADPS2','RNF148'],
        ['CAPS','VMAC'],
        ['CARD16','CASP1'],
        ['CCDC169','SOHLH2'],
        ['CCDC74A','CCDC74B'],
        ['CCPG1','PIGBOS1'],
        ['CEBPA-AS1','CTD-2540B15.9'],
        ['CEBPZOS','RP11-423P10.2'],
        ['CFAP161','IL16'],
        ['CFDP1','RP11-252K23.2'],
        ['CLCNKA','CLCNKB'],
        ['CLEC2D','RP11-705C15.2'],
        ['CLEC2D','RP11-705C15.5'],
        ['CNN2','ISG20L2'],
        ['COX17','POPDC2'],
        ['CPNE1','RBM12'],
        ['CTB-31N19.3','METTL9'],
        ['CTBS','GNG5'],
        ['CTC-301O7.4','SLC6A16'],
        ['CTC-378H22.1','CTC-378H22.2'],
        ['CTC-498J12.1','RP11-141O11.2'],
        ['CTD-2017D11.1','ZNF254'],
        ['CTD-2323K18.1','SNUPN'],
        ['CTD-2561J22.2','ZNF738'],
        ['CTD-3214H19.6','PCP2'],
        ['CTSS','LYZ'],
        ['DNAJC25-GNG10','GNG10'],
        ['DNAJC9','MRPS16'],
        ['DSE','FAM26F'],
        ['DUT','ZNF534'],
        ['DYNLRB1','ITCH'],
        ['DYX1C1-CCPG1','PIGBOS1'],
        ['EARS2','GGA2'],
        ['EDARADD','ENO1'],
        ['ERV3-1','ZNF117'],
        ['ESRP2','RP11-96D1.11'],
        ['FAM53B','METTL10'],
        ['FXYD2','FXYD6-FXYD2'],
        ['FZD4','RP11-736K20.5'],
        ['GALT','IL11RA'],
        ['GALT','RP11-195F19.29'],
        ['GBP4','GBP7'],
        ['GCNT4','LINC01336'],
        ['GGA2','ZFAND5'],
        ['GS1-309P15.4','RDX'],
        ['H3F3B','H3F3C'],
        ['HARS2','ZMAT2'],
        ['HBB','HBD'],
        ['HEPHL1','PANX1'],
        ['HILPDA','RP11-212P7.3'],
        ['HLA-A','HLA-B'],
        ['HLA-A','HLA-C'],
        ['HLA-A','HLA-G'],
        ['HLA-A','HLA-J'],
        ['HLA-B','HLA-C'],
        ['HLA-B','HLA-H'],
        ['HLA-B','HLA-L'],
        ['HLA-C','HLA-G'],
        ['HLA-C','HLA-L'],
        ['HLA-DRB1','HLA-DRB5'],
        ['HLA-E','STOM'],
        ['HMGB1','SP100'],
        ['HSPA1A','HSPA1B'],
        ['IDH2','RP11-617F23.1'],
        ['IFITM2','IFITM3'],
        ['IFRD1','LSMEM1'],
        ['IGH@','IGHM'],
        ['IGH@','RPS8'],
        ['IGSF5','PCP4'],
        ['KCTD10','RP11-256L11.3'],
        ['KLK4','KLKP1'],
        ['KRT7','KRT86'],
        ['LEAP2','UQCRQ'],
        ['LINC00578','RP11-114M1.1'],
        ['LINC00630','LL0XNC01-237H1.2'],
        ['LINC00665','LINC01535'],
        ['LINC00969','SDHA'],
        ['LINC01578','RP11-437B10.1'],
        ['LRR1','RHOQ'],
        ['LRRC69','OTUD6B'],
        ['MAP1LC3B','MAP1LC3B2'],
        ['METTL10','RP11-12J10.3'],
        ['METTL21B','RP11-571M6.15'],
        ['METTL21B','TSFM'],
        ['MFSD14A','SLC35A3'],
        ['MMADHC','RP1-145M24.1'],
        ['MPDZ','RP11-272P10.2'],
        ['MRPS6','SLC5A3'],
        ['MYL6','MYL6B'],
        ['NBPF1','NBPF3'],
        ['NME4','Z97634.3'],
        ['OR2W3','TRIM58'],
        ['OVCH2','RP11-494M8.4'],
        ['PAOX','RP11-108K14.8'],
        ['PDCD6','RP11-43F13.1'],
        ['PHB','ZNF607'],
        ['PILRA','PILRB'],
        ['PILRA','STAG3L5P-PVRIG2P-PILRB'],
        ['PUS1','RP11-417L19.6'],
        ['RBL2','RP11-44F14.2'],
        ['RBMX','RBMXL1'],
        ['RBP5','RP11-273B20.1'],
        ['RP1-199J3.5','TSTD3'],
        ['RP11-1252I4.2','RP11-423H2.3'],
        ['RP11-155G14.6','RP11-212P7.3'],
        ['RP11-156K13.1','RP11-806O11.1'],
        ['RP11-159N11.4','TTC12'],
        ['RP11-314N13.3','SDCBP2-AS1'],
        ['RP11-325P15.2','SSB'],
        ['RP11-343H5.4','RPS14'],
        ['RP11-446N19.1','RP11-96H19.1'],
        ['RP11-452F19.3','RP11-455P21.3'],
        ['RP11-667K14.3','RP11-667K14.4'],
        ['RP11-68I3.11','SSH2'],
        ['RP11-727F15.11','TMEM223'],
        ['RP11-82L18.2','SHC3'],
        ['RP11-879F14.2','RP11-879F14.3'],
        ['RP4-765C7.2','RPS14'],
        ['RP5-1159O4.1','RP5-1159O4.2'],
        ['RPL23AP7','RPL23AP82'],
        ['RPS19','RPS19P1'],
        ['SNHG14','SNRPN'],
        ['STYXL1','TMEM120A'],
        ['TCEAL2','TCEAL4'],
        ['TRAC','TRAV19'],
        ['TRAC','TRAV9-2'],
        ['TRGC1','TRGC2'],
        ['TTTY15','USP9Y'],
        ['TUBA4A','TUBA4B'],
        ['VAMP5','VAMP8'],
        ['ZNF28','ZNF468'],
        ['ZNF33A','ZNF33B'],
        ['ZNF510','ZNF782'],
        ['ZNF577','ZNF649'],
        ['AC005062.2','MACC1'],
        ['AC006153.3','CLDN12'],
        ['AC007036.5','MTURN'],
        ['AC007405.6','ERICH2'],
        ['AC007566.10','GATAD1'],
        ['AC009961.3','BAZ2B'],
        ['AC011513.3','CEACAM1'],
        ['AC011513.3','CEACAM6'],
        ['AC012485.2','PER2'],
        ['AC017116.11','DBNL'],
        ['AC024704.2','AC104653.1'],
        ['AC073046.25','TET3'],
        ['AC073343.13','ZNF316'],
        ['AC090774.2','CCDC144NL-AS1'],
        ['AC098831.4','PYURF'],
        ['ACACB','RP11-968O1.5'],
        ['ADAMTSL4-AS1','MCL1'],
        ['ADGRE2','ADGRE5'],
        ['ADSL','SGSM3'],
        ['AGO2','RP11-642A1.2'],
        ['AK6','TAF9'],
        ['AKR1C1','AKR1C2'],
        ['AL157871.2','SLC25A29'],
        ['AL590762.6','ZCRB1'],
        ['AMZ2','RP11-147L13.15'],
        ['ANKRD13A','RP11-946P6.6'],
        ['ANKRD18A','ANKRD18B'],
        ['ANKRD35','PIAS3'],
        ['ANKRD36','ANKRD36B'],
        ['ANKRD36','ANKRD36C'],
        ['ANKRD36B','ANKRD36C'],
        ['APBB3','SRA1'],
        ['ARHGAP42','TMEM133'],
        ['ARHGEF38','ARHGEF38-IT1'],
        ['ARMCX3','RP4-545K15.5'],
        ['ATAD3A','ATAD3B'],
        ['ATP10B','RP11-109J4.1'],
        ['ATP6V0A4','SVOPL'],
        ['ATP7A','RP5-1000K24.2'],
        ['ATXN3','MAB21L1'],
        ['AURKA','AURKAPS1'],
        ['AURKA','RAB3GAP2'],
        ['AZGP1','GJC3'],
        ['BACH1','BACH1-IT1'],
        ['BAG6','C6ORF47'],
        ['BBS5','DTD2'],
        ['BBS5','RP11-176H8.1'],
        ['BIRC6','YIPF4'],
        ['BMS1P20','LL22NC03-80A10.6'],
        ['BRWD1','RP11-707O23.1'],
        ['BTG4','POU2AF1'],
        ['BTN3A1','BTN3A2'],
        ['C12ORF74','PLEKHG7'],
        ['C1QBP','CTC-524C5.2'],
        ['C3ORF62','USP4'],
        ['C5AR1','CD84'],
        ['C6ORF163','SMIM8'],
        ['CATSPERB','TC2N'],
        ['CCDC74A','CCDC74B'],
        ['CCDC92','RP11-214K3.18'],
        ['CCPG1','PIGBOS1'],
        ['CCT6A','CCT6P1'],
        ['CDRT4','TVP23C'],
        ['CEACAM1','CEACAM5'],
        ['CEACAM1','CEACAM6'],
        ['CEACAM5','CEACAM6'],
        ['CFAP161','IL16'],
        ['CFAP45','RP11-190A12.7'],
        ['CHD2','RP11-437B10.1'],
        ['CHD9','RP11-454F8.3'],
        ['CHMP7','RP11-1149O23.3'],
        ['CLCC1','RP11-475E11.9'],
        ['CLIC5','RP1-8B1.4'],
        ['CLK1','PPIL3'],
        ['CLK4','RN7SKP70'],
        ['CNTNAP3','CNTNAP3B'],
        ['COQ8B','NUMBL'],
        ['COX17','POPDC2'],
        ['CP','CTD-2114J12.1'],
        ['CRIP1','CRIP2'],
        ['CST1','CST2'],
        ['CTBS','GNG5'],
        ['CTC-441N14.4','ZNF474'],
        ['CTC-479C5.12','PSMB10'],
        ['CTC-498J12.1','RP11-141O11.2'],
        ['CTD-2017D11.1','ZNF254'],
        ['CTD-3199J23.6','NBR2'],
        ['CTD-3214H19.4','PET100'],
        ['CYP2B6','CYP2B7P'],
        ['CYP2F1','CYP2F2P'],
        ['CYP4A11','CYP4A22'],
        ['CYP4B1','CYP4X1'],
        ['CYP4Z1','CYP4Z2P'],
        ['DNAI1','RP11-296L22.8'],
        ['DNAJC25-GNG10','GNG10'],
        ['DNAJC9','MRPS16'],
        ['DSE','RP1-93H18.1'],
        ['EEF1A1','RP11-553P9.1'],
        ['EGLN1','SCAND2P'],
        ['ENPP1','KCNE3'],
        ['EPCAM','RP11-433O3.1'],
        ['EPPIN','EPPIN-WFDC6'],
        ['EPPIN','WFDC6'],
        ['EPPIN-WFDC6','WFDC6'],
        ['ESRP2','RP11-96D1.11'],
        ['FAM157A','FAM157B'],
        ['FAM60A','FAM60BP'],
        ['FCGR3A','FCGR3B'],
        ['FPGT','FPGT-TNNI3K'],
        ['FPGT','TNNI3K'],
        ['FXN','TJP2'],
        ['GBP4','GBP7'],
        ['GGA2','ZFAND5'],
        ['GHRLOS','LINC00852'],
        ['GOLGA8A','GOLGA8B'],
        ['GSTA1','GSTA7P'],
        ['H3F3B','H3F3C'],
        ['HARS2','ZMAT2'],
        ['HCAR2','HCAR3'],
        ['HIST4H4','RP11-174G6.5'],
        ['HIVEP1','RP11-456H18.2'],
        ['HLA-A','HLA-B'],
        ['HLA-A','HLA-C'],
        ['HLA-A','HLA-E'],
        ['HLA-A','HLA-G'],
        ['HLA-A','HLA-J'],
        ['HLA-B','HLA-C'],
        ['HLA-B','HLA-H'],
        ['HLA-B','HLA-J'],
        ['HLA-B','HLA-L'],
        ['HLA-C','HLA-E'],
        ['HLA-C','HLA-H'],
        ['HLA-DPB1','HLA-DRB1'],
        ['HLA-DPB1','HLA-DRB5'],
        ['HLA-DQA1','HLA-DQA2'],
        ['HLA-DQB1','HLA-DQB2'],
        ['HLA-DRB1','HLA-DRB5'],
        ['HLA-DRB1','HLA-DRB6'],
        ['HLA-DRB5','HLA-DRB6'],
        ['HLA-DRB5','HLA-DRB9'],
        ['HMGB1','SP100'],
        ['HSP90AA1','HSP90AA2P'],
        ['HSP90AA1','HSP90AA6P'],
        ['HSP90AB1','HSP90AB2P'],
        ['HSP90B1','HSP90B2P'],
        ['HYDIN','HYDIN2'],
        ['ICA1L','PYURF'],
        ['IDH1','RP11-288G3.4'],
        ['IFITM2','IFITM3'],
        ['IFITM3','RP11-467L13.5'],
        ['INTS6-AS1','RPS4XP16'],
        ['IPP','RP11-767N6.7'],
        ['KIAA1147','RP5-894A10.2'],
        ['KIAA1147','RP5-894A10.6'],
        ['KLHL5','RP11-360F5.3'],
        ['KRT5','KRT6A'],
        ['LGALS9B','LGALS9C'],
        ['LGI1','RP11-437J2.3'],
        ['LINC00630','LL0XNC01-237H1.2'],
        ['LINC00969','SDHA'],
        ['LINC01513','ROPN1L'],
        ['LINC01578','RP11-437B10.1'],
        ['LRRC37A17P','LRRC37A4P'],
        ['LRRC69','OTUD6B'],
        ['LTB4R2','NOP9'],
        ['LYPD2','SLURP1'],
        ['MAP1LC3B','MAP1LC3B2'],
        ['METTL2A','METTL2B'],
        ['MRPS6','SLC5A3'],
        ['MUC16','MUC5AC'],
        ['MUC4','MUC5AC'],
        ['MUC5AC','NEAT1'],
        ['MUC5AC','PIGR'],
        ['NACA','NACA3P'],
        ['NBPF1','NBPF3'],
        ['NBPF2P','NBPF3'],
        ['NCOA4','RP11-557J10.3'],
        ['NDUFA6-AS1','RP4-669P10.19'],
        ['NDUFAF6','RP11-347C18.3'],
        ['NGRN','PERP'],
        ['NIN','RP11-248J18.3'],
        ['NSUN6','RP11-499P20.2'],
        ['NT5C1B','RDH14'],
        ['NT5C1B-RDH14','RDH14'],
        ['NUP50','RP11-973N13.2'],
        ['ODF3B','TYMP'],
        ['PAFAH1B1','RN7SL608P'],
        ['PAGR1','PRRT2'],
        ['PCF11','RP11-727A23.4'],
        ['PDCD6','RP11-43F13.1'],
        ['PDXDC1','PDXDC2P'],
        ['PILRA','PILRB'],
        ['PILRA','STAG3L5P-PVRIG2P-PILRB'],
        ['PLCG2','RN7SK'],
        ['PLCG2','RN7SKP255'],
        ['PMS2','PMS2CL'],
        ['PTMA','RP11-330M2.4'],
        ['PTMA','RP11-529H20.3'],
        ['PTPN11','RP11-561O4.1'],
        ['PVRIG','STAG3'],
        ['RBL2','RP11-44F14.2'],
        ['RBM14-RBM4','RBM4'],
        ['RBM47','RP11-489E7.1'],
        ['RBMX','RBMXL1'],
        ['RIN2','RP1-122P22.2'],
        ['RP1-102E24.10','TAPBPL'],
        ['RP1-178F15.4','RP1-178F15.5'],
        ['RP1-178F15.4','S100A13'],
        ['RP1-178F15.5','S100A13'],
        ['RP11-114G22.1','RP11-153K16.1'],
        ['RP11-156K13.1','RP11-806O11.1'],
        ['RP11-159N11.4','TTC12'],
        ['RP11-191N8.2','RP11-815M8.1'],
        ['RP11-203J24.9','ST6GALNAC6'],
        ['RP11-250B2.3','RP11-250B2.4'],
        ['RP11-274B21.1','TLK2'],
        ['RP11-274B21.14','TLK2'],
        ['RP11-276H7.2','RP11-276H7.3'],
        ['RP11-314N13.3','SDCBP2-AS1'],
        ['RP11-321N4.5','SYNCRIP'],
        ['RP11-325P15.2','SSB'],
        ['RP11-343C2.11','SNTB2'],
        ['RP11-444D3.1','SOX5'],
        ['RP11-463O12.5','VPS37B'],
        ['RP11-510I21.1','SMC5'],
        ['RP11-534L20.4','RPL7A'],
        ['RP11-537I16.2','SENP2'],
        ['RP11-638I2.6','YY1'],
        ['RP11-65L3.4','TTN-AS1'],
        ['RP11-702H23.6','SERPINA10'],
        ['RP11-760D2.7','YWHAE'],
        ['RP11-93O14.2','VPS35'],
        ['RP13-131K19.2','RP13-131K19.7'],
        ['RP4-641G12.4','SNW1'],
        ['RP4-765C7.2','RPS14'],
        ['RP5-1159O4.1','RP5-1159O4.2'],
        ['RP5-966M1.6','TMEM110-MUSTN1'],
        ['RPL24','RPL24P4'],
        ['RPL36AL','RPL36AP45'],
        ['RPL7L1','WAC-AS1'],
        ['RPS20','RPS20P10'],
        ['RPS3A','RPS3AP49'],
        ['RRP7A','RRP7BP'],
        ['SAA1','SAA2'],
        ['SAG','VPS13C'],
        ['SCNN1A','TNFRSF1A'],
        ['SERPINB3','SERPINB4'],
        ['SERPINI2','WDR49'],
        ['SLFN11','SLFN13'],
        ['SNHG14','SNRPN'],
        ['SORD','SORD2P'],
        ['TLR1','TLR6'],
        ['TTTY15','USP9Y'],
        ['TUBA4A','TUBA4B'],
        ['TUBG1','TUBG2'],
        ['UGT2A1','UGT2A2'],
        ['VAMP5','VAMP8'],
        ['ZDHHC11','ZDHHC11B'],
        ['ZNF28','ZNF468'],
        ['ZNF510','ZNF782'],
        ['ZNF587B','ZNF814'],
        ['KANSL1','ARL17A'],
        ['AC009228.1','FAM228A'],
        ['AC079250.1','RPL18A'],
        ['AC125232.1','RMND5A'],
        ['AC142381.1','IGHV3-23'],
        ['ACSM2A','ACSM2B'],
        ['ACTA1','ACTB'],
        ['ACTA1','ACTG1'],
        ['ACTA1','CLU'],
        ['ACTA1','TG'],
        ['ACTA2','ACTB'],
        ['ACTB','ACTG1'],
        ['ACTB','ACTG2'],
        ['ACTB','GPX3'],
        ['ACTB','IGFBP5'],
        ['ACTB','IGH@'],
        ['ACTB','IGHM'],
        ['ACTB','NFIC'],
        ['ACTB','POTEF'],
        ['ACTB','TG'],
        ['ACTG1','IGFBP5'],
        ['ACTG1','TG'],
        ['ACTN1','HMGN1'],
        ['ADGRE2','ADGRE3'],
        ['ADH1A','ADH1B'],
        ['ADH1B','ADH1C'],
        ['ADPRHL1','DCUN1D2'],
        ['AKR1C1','AKR1C2'],
        ['AKR7A2','AKR7A3'],
        ['ALB','CRP'],
        ['ALB','FGA'],
        ['ALB','FGB'],
        ['ALB','FGG'],
        ['ALB','HP'],
        ['ALB','ORM1'],
        ['ALB','SAA1'],
        ['ALB','SERPINA1'],
        ['ALDOA','TG'],
        ['ANG','RNASE4'],
        ['ANKRD36','ANKRD36C'],
        ['APOBEC3A','APOBEC3B'],
        ['APOLD1','SERBP1'],
        ['APP','TG'],
        ['AQP7','FP325317.1'],
        ['ARHGAP42','TMEM133'],
        ['ARID3C','DCTN3'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ASPRV1','PCBP1-AS1'],
        ['ATAD3A','ATAD3B'],
        ['ATP1A1','FLNC'],
        ['ATP1A1','TG'],
        ['ATP1B4','SOD2'],
        ['AZGP1','GJC3'],
        ['BCAM','TG'],
        ['BEX1','BEX2'],
        ['BRICD5','PGP'],
        ['BRS3','HTATSF1'],
        ['C12ORF77','FAM129A'],
        ['C17ORF47','SEPT4'],
        ['C3','HP'],
        ['C3','IGFBP5'],
        ['C3ORF62','USP4'],
        ['C4A','C4B'],
        ['C5AR1','CD84'],
        ['C7ORF55-LUC7L2','LUC7L2'],
        ['C9ORF172','PHPT1'],
        ['CACNA1D','SRSF8'],
        ['CALR','TG'],
        ['CALR3','EPS15L1'],
        ['CAMTA1','CAMTA1-IT1'],
        ['CAVIN4','MSANTD3-TMEFF1'],
        ['CAVIN4','TMEFF1'],
        ['CBWD1','CBWD2'],
        ['CCDC74A','CCDC74B'],
        ['CCL21','IGH@'],
        ['CCL5','LYZ'],
        ['CCT6A','CCT6P1'],
        ['CD22','MAG'],
        ['CD300A','CD300C'],
        ['CD68','LYZ'],
        ['CD74','IGH@'],
        ['CDH1','TG'],
        ['CDK11A','CDK11B'],
        ['CDRT4','TVP23C'],
        ['CEP295','SCARNA9'],
        ['CFH','CFHR3'],
        ['CFHR1','CFHR2'],
        ['CFHR3','CFHR4'],
        ['CKM','TG'],
        ['CLSTN1','CTNNBIP1'],
        ['CLU','IGFBP5'],
        ['CLU','TG'],
        ['CLU','TUBB4A'],
        ['CNTNAP3','CNTNAP3B'],
        ['COL1A1','COL1A2'],
        ['COL1A1','IGFBP4'],
        ['COL6A3','THBS1'],
        ['COMMD1','RPSA'],
        ['COQ8B','NUMBL'],
        ['CPNE1','RBM12'],
        ['CRIP3','ZNF318'],
        ['CRLF2','CTSS'],
        ['CRP','FGA'],
        ['CRP','HP'],
        ['CRP','ORM1'],
        ['CRP','SERPINA1'],
        ['CRYAB','DES'],
        ['CRYAB','TG'],
        ['CRYAB','TPO'],
        ['CSDE1','TG'],
        ['CSF2RA','CTSB'],
        ['CTBS','GNG5'],
        ['CTSB','FLNC'],
        ['CTSB','TG'],
        ['CTSS','KIRREL3'],
        ['CTSS','LYZ'],
        ['CXCR1','CXCR2'],
        ['CYP2A6','CYP2A7'],
        ['CYP2B6','CYP2B7P'],
        ['CYP2C19','CYP2C9'],
        ['CYP2D6','CYP2D7'],
        ['CYP2D7','CYP2D8P'],
        ['CYP4A11','CYP4A22'],
        ['DES','MYH11'],
        ['DES','MYH7'],
        ['DES','TG'],
        ['DHRS1','RABGGTA'],
        ['DLEU2L','EFCAB7'],
        ['DNAJC25-GNG10','GNG10'],
        ['DNAJC9','MRPS16'],
        ['DUS3L','PRR22'],
        ['DUT','ZNF534'],
        ['EARS2','GGA2'],
        ['EDARADD','ENO1'],
        ['EEF1G','TG'],
        ['EEF2','IGFBP5'],
        ['EEF2','TG'],
        ['EGR1','TG'],
        ['ELOA','RPL11'],
        ['ENPP1','KCNE3'],
        ['EPOR','RGL3'],
        ['EPPIN','WFDC6'],
        ['EPPIN-WFDC6','WFDC6'],
        ['ERV3-1','ZNF117'],
        ['FAAP20','PRKCZ-AS1'],
        ['FAHD2B','FAHD2CP'],
        ['FAM175A','HELQ'],
        ['FAM177B','SOD2'],
        ['FAM35A','FAM35BP'],
        ['FBXO44','FBXO6'],
        ['FCGR3A','FCGR3B'],
        ['FGA','FGB'],
        ['FGA','HP'],
        ['FGA','SAA1'],
        ['FGA','SERPINA1'],
        ['FGB','FGG'],
        ['FGB','HP'],
        ['FGB','SERPINA1'],
        ['FGG','HP'],
        ['FGG','SERPINA1'],
        ['FHL1','TG'],
        ['FKBP1A','FKBP1C'],
        ['FLNA','IGFBP4'],
        ['FLNA','TG'],
        ['FLNC','GPX3'],
        ['FLNC','H19'],
        ['FLNC','IGFBP5'],
        ['FLNC','TG'],
        ['FLNC','TPO'],
        ['FMC1','LUC7L2'],
        ['FTL','TG'],
        ['GAPDH','TG'],
        ['GBP4','GBP7'],
        ['GGA2','ZFAND5'],
        ['GLUD1','GLUD2'],
        ['GLUL','TG'],
        ['GNAS','TG'],
        ['GOLGA8A','GOLGA8B'],
        ['GPX3','TG'],
        ['GREB1','RN7SL674P'],
        ['GRID1','RNF152'],
        ['GSN','TG'],
        ['GSTM1','GSTM4'],
        ['GSTM1','GSTM5'],
        ['GSTM2','GSTM4'],
        ['H19','TG'],
        ['H1F0','TG'],
        ['H3F3B','H3F3C'],
        ['HARS2','ZMAT2'],
        ['HBB','HBD'],
        ['HCG11','NORAD'],
        ['HECA','SLC48A1'],
        ['HLA-A','HLA-B'],
        ['HLA-A','HLA-C'],
        ['HLA-A','HLA-E'],
        ['HLA-A','HLA-F'],
        ['HLA-A','HLA-G'],
        ['HLA-A','HLA-H'],
        ['HLA-A','HLA-J'],
        ['HLA-A','HLA-L'],
        ['HLA-B','HLA-C'],
        ['HLA-B','HLA-E'],
        ['HLA-B','HLA-F'],
        ['HLA-B','HLA-G'],
        ['HLA-B','HLA-H'],
        ['HLA-B','HLA-J'],
        ['HLA-B','HLA-L'],
        ['HLA-C','HLA-E'],
        ['HLA-C','HLA-F'],
        ['HLA-C','HLA-G'],
        ['HLA-C','HLA-H'],
        ['HLA-C','HLA-L'],
        ['HLA-DPB1','HLA-DRB1'],
        ['HLA-DPB1','HLA-DRB5'],
        ['HLA-DQA1','HLA-DQA2'],
        ['HLA-DQB1','HLA-DQB2'],
        ['HLA-DRB1','HLA-DRB5'],
        ['HLA-DRB1','HLA-DRB6'],
        ['HLA-DRB5','HLA-DRB6'],
        ['HLA-E','HLA-F'],
        ['HLA-E','HLA-G'],
        ['HLA-E','HLA-H'],
        ['HLA-E','HLA-J'],
        ['HLA-E','LYZ'],
        ['HLA-G','HLA-H'],
        ['HLA-G','HLA-J'],
        ['HLA-G','HLA-L'],
        ['HLA-H','HLA-J'],
        ['HLA-J','HLA-K'],
        ['HMGB1','SP100'],
        ['HNRNPA1','HNRNPA1L2'],
        ['HP','HPR'],
        ['HP','ORM1'],
        ['HP','SAA1'],
        ['HP','SERPINA1'],
        ['HP','SERPINA3'],
        ['HSP90AA1','HSP90AA2P'],
        ['HSP90AA1','HSP90AA4P'],
        ['HSP90AA1','TG'],
        ['HSP90AB1','HSP90AB2P'],
        ['HSP90AB1','TG'],
        ['HSPA1A','HSPA1B'],
        ['HSPA1B','TG'],
        ['HSPA1L','LSM2'],
        ['HSPB1','TG'],
        ['IFITM1','IFITM3'],
        ['IFITM2','IFITM3'],
        ['IFITM3','IFITM9P'],
        ['IFRD1','LSMEM1'],
        ['IGFBP4','IGFBP5'],
        ['IGFBP4','IGH@'],
        ['IGFBP5','RPL11'],
        ['IGFBP5','TG'],
        ['IGFBP5','TPO'],
        ['IGFBP5','TPT1'],
        ['IGFBP5','VIM'],
        ['IGH@','IGHA1'],
        ['IGH@','IGHA2'],
        ['IGH@','IGHD'],
        ['IGH@','IGHG1'],
        ['IGH@','IGHG2'],
        ['IGH@','IGHG3'],
        ['IGH@','IGHGP'],
        ['IGH@','IGHM'],
        ['IGH@','IGHV1-12'],
        ['IGH@','IGHV1-18'],
        ['IGH@','IGHV1-2'],
        ['IGH@','IGHV1-24'],
        ['IGH@','IGHV1-3'],
        ['IGH@','IGHV1-46'],
        ['IGH@','IGHV1-58'],
        ['IGH@','IGHV1-69'],
        ['IGH@','IGHV2-26'],
        ['IGH@','IGHV2-5'],
        ['IGH@','IGHV2-70'],
        ['IGH@','IGHV3-11'],
        ['IGH@','IGHV3-13'],
        ['IGH@','IGHV3-15'],
        ['IGH@','IGHV3-20'],
        ['IGH@','IGHV3-21'],
        ['IGH@','IGHV3-23'],
        ['IGH@','IGHV3-30'],
        ['IGH@','IGHV3-33'],
        ['IGH@','IGHV3-43'],
        ['IGH@','IGHV3-48'],
        ['IGH@','IGHV3-49'],
        ['IGH@','IGHV3-53'],
        ['IGH@','IGHV3-64'],
        ['IGH@','IGHV3-66'],
        ['IGH@','IGHV3-7'],
        ['IGH@','IGHV3-71'],
        ['IGH@','IGHV3-72'],
        ['IGH@','IGHV3-73'],
        ['IGH@','IGHV3-74'],
        ['IGH@','IGHV3OR16-8'],
        ['IGH@','IGHV3OR16-9'],
        ['IGH@','IGHV4-28'],
        ['IGH@','IGHV4-31'],
        ['IGH@','IGHV4-34'],
        ['IGH@','IGHV4-39'],
        ['IGH@','IGHV4-55'],
        ['IGH@','IGHV4-59'],
        ['IGH@','IGHV5-51'],
        ['IGH@','IGHV6-1'],
        ['IGH@','IGHV7-81'],
        ['IGH@','IGK@'],
        ['IGH@','IGKC'],
        ['IGH@','RPS8'],
        ['IGH@','TPT1'],
        ['IGHA1','IGHA2'],
        ['IGHA1','IGHM'],
        ['IGHA1','IGHV1-18'],
        ['IGHA1','IGHV1-2'],
        ['IGHA1','IGHV1-24'],
        ['IGHA1','IGHV1-3'],
        ['IGHA1','IGHV1-46'],
        ['IGHA1','IGHV1-69'],
        ['IGHA1','IGHV1OR15-1'],
        ['IGHA1','IGHV2-5'],
        ['IGHA1','IGHV2-70'],
        ['IGHA1','IGHV3-11'],
        ['IGHA1','IGHV3-13'],
        ['IGHA1','IGHV3-15'],
        ['IGHA1','IGHV3-20'],
        ['IGHA1','IGHV3-21'],
        ['IGHA1','IGHV3-23'],
        ['IGHA1','IGHV3-30'],
        ['IGHA1','IGHV3-33'],
        ['IGHA1','IGHV3-43'],
        ['IGHA1','IGHV3-48'],
        ['IGHA1','IGHV3-49'],
        ['IGHA1','IGHV3-53'],
        ['IGHA1','IGHV3-66'],
        ['IGHA1','IGHV3-7'],
        ['IGHA1','IGHV3-71'],
        ['IGHA1','IGHV3-72'],
        ['IGHA1','IGHV3-73'],
        ['IGHA1','IGHV3-74'],
        ['IGHA1','IGHV3OR16-10'],
        ['IGHA1','IGHV3OR16-11'],
        ['IGHA1','IGHV3OR16-6'],
        ['IGHA1','IGHV3OR16-8'],
        ['IGHA1','IGHV3OR16-9'],
        ['IGHA1','IGHV4-28'],
        ['IGHA1','IGHV4-31'],
        ['IGHA1','IGHV4-34'],
        ['IGHA1','IGHV4-39'],
        ['IGHA1','IGHV4-4'],
        ['IGHA1','IGHV4-55'],
        ['IGHA1','IGHV4-59'],
        ['IGHA1','IGHV4-61'],
        ['IGHA1','IGHV4OR15-8'],
        ['IGHA1','IGHV5-51'],
        ['IGHA1','IGHV6-1'],
        ['IGHA1','IGK@'],
        ['IGHA1','IGKC'],
        ['IGHA2','IGHV1-2'],
        ['IGHA2','IGHV1-69'],
        ['IGHA2','IGHV3-11'],
        ['IGHA2','IGHV3-15'],
        ['IGHA2','IGHV3-20'],
        ['IGHA2','IGHV3-21'],
        ['IGHA2','IGHV3-23'],
        ['IGHA2','IGHV3-30'],
        ['IGHA2','IGHV3-33'],
        ['IGHA2','IGHV3-48'],
        ['IGHA2','IGHV3-49'],
        ['IGHA2','IGHV3-53'],
        ['IGHA2','IGHV3-66'],
        ['IGHA2','IGHV3-7'],
        ['IGHA2','IGHV3-74'],
        ['IGHA2','IGHV3OR16-9'],
        ['IGHA2','IGHV4-31'],
        ['IGHA2','IGHV4-34'],
        ['IGHA2','IGHV4-39'],
        ['IGHA2','IGHV4-4'],
        ['IGHA2','IGHV4-59'],
        ['IGHA2','IGHV4-61'],
        ['IGHA2','IGHV5-51'],
        ['IGHD','IGHV3-11'],
        ['IGHD','IGHV3-21'],
        ['IGHD','IGHV3-48'],
        ['IGHD','IGHV3-7'],
        ['IGHG1','IGHG2'],
        ['IGHG1','IGHG3'],
        ['IGHG1','IGHG4'],
        ['IGHG1','IGHGP'],
        ['IGHG1','IGHV1-18'],
        ['IGHG1','IGHV1-2'],
        ['IGHG1','IGHV1-24'],
        ['IGHG1','IGHV1-3'],
        ['IGHG1','IGHV1-46'],
        ['IGHG1','IGHV1-67'],
        ['IGHG1','IGHV1-69'],
        ['IGHG1','IGHV2-26'],
        ['IGHG1','IGHV2-5'],
        ['IGHG1','IGHV2-70'],
        ['IGHG1','IGHV2OR16-5'],
        ['IGHG1','IGHV3-11'],
        ['IGHG1','IGHV3-13'],
        ['IGHG1','IGHV3-15'],
        ['IGHG1','IGHV3-20'],
        ['IGHG1','IGHV3-21'],
        ['IGHG1','IGHV3-23'],
        ['IGHG1','IGHV3-30'],
        ['IGHG1','IGHV3-33'],
        ['IGHG1','IGHV3-43'],
        ['IGHG1','IGHV3-48'],
        ['IGHG1','IGHV3-49'],
        ['IGHG1','IGHV3-53'],
        ['IGHG1','IGHV3-66'],
        ['IGHG1','IGHV3-7'],
        ['IGHG1','IGHV3-71'],
        ['IGHG1','IGHV3-72'],
        ['IGHG1','IGHV3-73'],
        ['IGHG1','IGHV3-74'],
        ['IGHG1','IGHV3OR16-10'],
        ['IGHG1','IGHV3OR16-11'],
        ['IGHG1','IGHV3OR16-8'],
        ['IGHG1','IGHV3OR16-9'],
        ['IGHG1','IGHV4-31'],
        ['IGHG1','IGHV4-34'],
        ['IGHG1','IGHV4-39'],
        ['IGHG1','IGHV4-4'],
        ['IGHG1','IGHV4-55'],
        ['IGHG1','IGHV4-59'],
        ['IGHG1','IGHV4-61'],
        ['IGHG1','IGHV4OR15-8'],
        ['IGHG1','IGHV5-51'],
        ['IGHG1','IGHV6-1'],
        ['IGHG1','IGK@'],
        ['IGHG1','IGKC'],
        ['IGHG2','IGHG3'],
        ['IGHG2','IGHG4'],
        ['IGHG2','IGHGP'],
        ['IGHG2','IGHV1-18'],
        ['IGHG2','IGHV1-2'],
        ['IGHG2','IGHV1-24'],
        ['IGHG2','IGHV1-46'],
        ['IGHG2','IGHV1-69'],
        ['IGHG2','IGHV2-5'],
        ['IGHG2','IGHV2-70'],
        ['IGHG2','IGHV3-11'],
        ['IGHG2','IGHV3-15'],
        ['IGHG2','IGHV3-20'],
        ['IGHG2','IGHV3-21'],
        ['IGHG2','IGHV3-23'],
        ['IGHG2','IGHV3-30'],
        ['IGHG2','IGHV3-33'],
        ['IGHG2','IGHV3-48'],
        ['IGHG2','IGHV3-49'],
        ['IGHG2','IGHV3-53'],
        ['IGHG2','IGHV3-66'],
        ['IGHG2','IGHV3-7'],
        ['IGHG2','IGHV3-71'],
        ['IGHG2','IGHV3-73'],
        ['IGHG2','IGHV3-74'],
        ['IGHG2','IGHV3OR16-9'],
        ['IGHG2','IGHV4-31'],
        ['IGHG2','IGHV4-34'],
        ['IGHG2','IGHV4-39'],
        ['IGHG2','IGHV4-4'],
        ['IGHG2','IGHV4-59'],
        ['IGHG2','IGHV4-61'],
        ['IGHG2','IGHV5-51'],
        ['IGHG2','IGK@'],
        ['IGHG2','IGKC'],
        ['IGHG3','IGHG4'],
        ['IGHG3','IGHGP'],
        ['IGHG3','IGHV1-18'],
        ['IGHG3','IGHV1-46'],
        ['IGHG3','IGHV2-70'],
        ['IGHG3','IGHV3-11'],
        ['IGHG3','IGHV3-13'],
        ['IGHG3','IGHV3-15'],
        ['IGHG3','IGHV3-20'],
        ['IGHG3','IGHV3-21'],
        ['IGHG3','IGHV3-23'],
        ['IGHG3','IGHV3-30'],
        ['IGHG3','IGHV3-33'],
        ['IGHG3','IGHV3-48'],
        ['IGHG3','IGHV3-53'],
        ['IGHG3','IGHV3-7'],
        ['IGHG3','IGHV3-74'],
        ['IGHG3','IGHV3OR16-9'],
        ['IGHG3','IGHV4-59'],
        ['IGHG3','IGHV4-61'],
        ['IGHG3','IGK@'],
        ['IGHG3','IGKC'],
        ['IGHG4','IGHGP'],
        ['IGHG4','IGHV1-18'],
        ['IGHG4','IGHV1-2'],
        ['IGHG4','IGHV1-24'],
        ['IGHG4','IGHV1-46'],
        ['IGHG4','IGHV1-69'],
        ['IGHG4','IGHV2-5'],
        ['IGHG4','IGHV2-70'],
        ['IGHG4','IGHV3-11'],
        ['IGHG4','IGHV3-15'],
        ['IGHG4','IGHV3-20'],
        ['IGHG4','IGHV3-21'],
        ['IGHG4','IGHV3-23'],
        ['IGHG4','IGHV3-30'],
        ['IGHG4','IGHV3-33'],
        ['IGHG4','IGHV3-48'],
        ['IGHG4','IGHV3-53'],
        ['IGHG4','IGHV3-7'],
        ['IGHG4','IGHV3-74'],
        ['IGHG4','IGHV3OR16-9'],
        ['IGHG4','IGHV4-31'],
        ['IGHG4','IGHV4-34'],
        ['IGHG4','IGHV4-39'],
        ['IGHG4','IGHV4-59'],
        ['IGHG4','IGHV5-51'],
        ['IGHG4','IGK@'],
        ['IGHG4','IGKC'],
        ['IGHGP','IGHV1-18'],
        ['IGHGP','IGHV1-46'],
        ['IGHGP','IGHV1-69'],
        ['IGHGP','IGHV3-11'],
        ['IGHGP','IGHV3-15'],
        ['IGHGP','IGHV3-20'],
        ['IGHGP','IGHV3-21'],
        ['IGHGP','IGHV3-23'],
        ['IGHGP','IGHV3-30'],
        ['IGHGP','IGHV3-33'],
        ['IGHGP','IGHV3-48'],
        ['IGHGP','IGHV3-7'],
        ['IGHGP','IGHV3-74'],
        ['IGHGP','IGHV3OR16-9'],
        ['IGHGP','IGHV4-59'],
        ['IGHGP','IGHV4-61'],
        ['IGHGP','IGHV5-51'],
        ['IGHM','IGHV1-18'],
        ['IGHM','IGHV1-46'],
        ['IGHM','IGHV1-69'],
        ['IGHM','IGHV2-26'],
        ['IGHM','IGHV2-5'],
        ['IGHM','IGHV2-70'],
        ['IGHM','IGHV3-11'],
        ['IGHM','IGHV3-15'],
        ['IGHM','IGHV3-20'],
        ['IGHM','IGHV3-21'],
        ['IGHM','IGHV3-23'],
        ['IGHM','IGHV3-30'],
        ['IGHM','IGHV3-33'],
        ['IGHM','IGHV3-48'],
        ['IGHM','IGHV3-49'],
        ['IGHM','IGHV3-53'],
        ['IGHM','IGHV3-66'],
        ['IGHM','IGHV3-7'],
        ['IGHM','IGHV3-71'],
        ['IGHM','IGHV3-74'],
        ['IGHM','IGHV3OR16-9'],
        ['IGHM','IGHV4-31'],
        ['IGHM','IGHV4-34'],
        ['IGHM','IGHV4-39'],
        ['IGHM','IGHV4-4'],
        ['IGHM','IGHV4-59'],
        ['IGHM','IGHV4-61'],
        ['IGHM','IGHV5-51'],
        ['IGHV1-12','IGHV1-2'],
        ['IGHV1-18','IGHV1-2'],
        ['IGHV1-18','IGHV1-69'],
        ['IGHV1-18','IGHV7-81'],
        ['IGHV1-2','IGHV1-46'],
        ['IGHV1-2','IGHV1OR15-1'],
        ['IGHV1-2','IGHV1OR15-3'],
        ['IGHV1-2','IGHV7-81'],
        ['IGHV1-46','IGHV1-69'],
        ['IGHV1-46','IGHV1OR15-1'],
        ['IGHV2-5','IGHV2-70'],
        ['IGHV3-11','IGHV3-21'],
        ['IGHV3-11','IGHV3-23'],
        ['IGHV3-11','IGHV3-48'],
        ['IGHV3-11','IGHV3-64'],
        ['IGHV3-11','IGHV3-66'],
        ['IGHV3-11','IGHV3OR15-7'],
        ['IGHV3-11','IGHV3OR16-8'],
        ['IGHV3-11','IGHV3OR16-9'],
        ['IGHV3-13','IGHV3-48'],
        ['IGHV3-13','IGHV3OR16-10'],
        ['IGHV3-13','IGHV3OR16-11'],
        ['IGHV3-15','IGHV3-20'],
        ['IGHV3-15','IGHV3OR16-7'],
        ['IGHV3-16','IGHV3-20'],
        ['IGHV3-16','IGHV3-48'],
        ['IGHV3-20','IGHV3-22'],
        ['IGHV3-20','IGHV3-23'],
        ['IGHV3-20','IGHV3-35'],
        ['IGHV3-20','IGHV3-38'],
        ['IGHV3-20','IGHV3-41'],
        ['IGHV3-20','IGHV3-43'],
        ['IGHV3-20','IGHV3-48'],
        ['IGHV3-20','IGHV3-53'],
        ['IGHV3-20','IGHV3OR15-7'],
        ['IGHV3-21','IGHV3-48'],
        ['IGHV3-21','IGHV3OR15-7'],
        ['IGHV3-22','IGHV3-48'],
        ['IGHV3-23','IGHV3-41'],
        ['IGHV3-23','IGHV3-43'],
        ['IGHV3-23','IGHV3-48'],
        ['IGHV3-23','IGHV3-53'],
        ['IGHV3-23','IGHV3-62'],
        ['IGHV3-23','IGHV3-64'],
        ['IGHV3-23','IGHV3-66'],
        ['IGHV3-23','IGHV3-7'],
        ['IGHV3-23','IGHV3-71'],
        ['IGHV3-23','IGHV3-74'],
        ['IGHV3-23','IGHV3OR16-10'],
        ['IGHV3-23','IGHV3OR16-11'],
        ['IGHV3-23','IGHV3OR16-15'],
        ['IGHV3-23','IGHV3OR16-16'],
        ['IGHV3-30','IGHV3-48'],
        ['IGHV3-30','IGHV3-53'],
        ['IGHV3-30','IGHV3-62'],
        ['IGHV3-30','IGHV3-64'],
        ['IGHV3-30','IGHV3-66'],
        ['IGHV3-30','IGHV3-7'],
        ['IGHV3-30','IGHV3-71'],
        ['IGHV3-30','IGHV3-74'],
        ['IGHV3-30','IGHV3OR16-10'],
        ['IGHV3-33','IGHV3-48'],
        ['IGHV3-33','IGHV3-53'],
        ['IGHV3-33','IGHV3-62'],
        ['IGHV3-33','IGHV3-64'],
        ['IGHV3-33','IGHV3-66'],
        ['IGHV3-33','IGHV3-7'],
        ['IGHV3-33','IGHV3-71'],
        ['IGHV3-33','IGHV3-74'],
        ['IGHV3-33','IGHV3OR16-10'],
        ['IGHV3-38','IGHV3-53'],
        ['IGHV3-43','IGHV3-7'],
        ['IGHV3-48','IGHV3-65'],
        ['IGHV3-48','IGHV3-7'],
        ['IGHV3-48','IGHV3OR15-7'],
        ['IGHV3-48','IGHV3OR16-8'],
        ['IGHV3-48','IGHV3OR16-9'],
        ['IGHV3-49','IGHV3-53'],
        ['IGHV3-53','IGHV3-66'],
        ['IGHV3-53','IGHV3-73'],
        ['IGHV3-53','IGHV3-74'],
        ['IGHV3-53','IGHV3OR15-7'],
        ['IGHV3-53','IGHV3OR16-13'],
        ['IGHV3-7','IGHV3OR15-7'],
        ['IGHV3-71','IGHV3OR16-9'],
        ['IGHV3OR15-7','IGHV3OR16-9'],
        ['IGHV3OR16-10','IGHV3OR16-9'],
        ['IGHV3OR16-11','IGHV3OR16-9'],
        ['IGHV3OR16-8','IGHV3OR16-9'],
        ['IGHV4-28','IGHV4-34'],
        ['IGHV4-28','IGHV4-55'],
        ['IGHV4-28','IGHV4-59'],
        ['IGHV4-28','IGHV4-61'],
        ['IGHV4-28','IGHV4OR15-8'],
        ['IGHV4-31','IGHV4-34'],
        ['IGHV4-31','IGHV4-55'],
        ['IGHV4-31','IGHV4-59'],
        ['IGHV4-31','IGHV4-61'],
        ['IGHV4-31','IGHV4OR15-8'],
        ['IGHV4-34','IGHV4-55'],
        ['IGHV4-34','IGHV4-59'],
        ['IGHV4-34','IGHV4-61'],
        ['IGHV4-34','IGHV4OR15-8'],
        ['IGHV4-39','IGHV4-55'],
        ['IGHV4-39','IGHV4-59'],
        ['IGHV4-39','IGHV4-61'],
        ['IGHV4-39','IGHV4OR15-8'],
        ['IGHV4-4','IGHV4-59'],
        ['IGHV4-55','IGHV4-59'],
        ['IGHV4-55','IGHV4-61'],
        ['IGK@','IGKC'],
        ['IGK@','IGKV1-27'],
        ['IGK@','IGKV1-8'],
        ['IGK@','IGKV1D-8'],
        ['IGK@','IGKV2-24'],
        ['IGK@','IGKV2D-24'],
        ['IGK@','IGKV2D-28'],
        ['IGK@','IGKV2D-29'],
        ['IGK@','IGKV2D-30'],
        ['IGK@','IGKV3-11'],
        ['IGK@','IGKV3-20'],
        ['IGK@','IGKV3-7'],
        ['IGK@','IGKV3D-11'],
        ['IGK@','IGKV3D-15'],
        ['IGK@','IGKV3D-20'],
        ['IGK@','IGKV6-21'],
        ['IGKC','IGKV1-27'],
        ['IGKC','IGKV1-33'],
        ['IGKC','IGKV1-37'],
        ['IGKC','IGKV1-39'],
        ['IGKC','IGKV1D-27'],
        ['IGKC','IGKV1D-33'],
        ['IGKC','IGKV1D-37'],
        ['IGKC','IGKV1D-39'],
        ['IGKC','IGKV1D-8'],
        ['IGKC','IGKV1OR2-108'],
        ['IGKC','IGKV2-24'],
        ['IGKC','IGKV2-26'],
        ['IGKC','IGKV2-28'],
        ['IGKC','IGKV2-29'],
        ['IGKC','IGKV2-30'],
        ['IGKC','IGKV2-40'],
        ['IGKC','IGKV2D-24'],
        ['IGKC','IGKV2D-26'],
        ['IGKC','IGKV2D-28'],
        ['IGKC','IGKV2D-29'],
        ['IGKC','IGKV2D-30'],
        ['IGKC','IGKV2D-40'],
        ['IGKC','IGKV2OR22-3'],
        ['IGKC','IGKV3D-15'],
        ['IGKC','IGKV3D-20'],
        ['IGKC','IGKV3D-7'],
        ['IGKC','IGKV3OR2-268'],
        ['IGKC','IGKV6-21'],
        ['IGKC','IGKV6D-21'],
        ['IGKC','IGKV6D-41'],
        ['IGKV1-12','IGKV1-8'],
        ['IGKV1-16','IGKV1-17'],
        ['IGKV1-17','IGKV1-6'],
        ['IGKV1-17','IGKV1-8'],
        ['IGKV1-27','IGKV1-39'],
        ['IGKV1-27','IGKV1-6'],
        ['IGKV1-27','IGKV1D-39'],
        ['IGKV1-39','IGKV1-8'],
        ['IGKV1-39','IGKV1D-8'],
        ['IGKV1-5','IGKV1-8'],
        ['IGKV1-5','IGKV1-9'],
        ['IGKV1-5','IGKV1D-8'],
        ['IGKV1-6','IGKV1D-8'],
        ['IGKV1-8','IGKV1D-12'],
        ['IGKV1-8','IGKV1D-39'],
        ['IGKV1-9','IGKV1D-8'],
        ['IGKV1D-39','IGKV1D-8'],
        ['IGKV1D-8','IGKV1OR22-1'],
        ['IGKV2-24','IGKV2-40'],
        ['IGKV2-24','IGKV2D-24'],
        ['IGKV2-24','IGKV2D-40'],
        ['IGKV2-4','IGKV2D-28'],
        ['IGKV2-40','IGKV2D-24'],
        ['IGKV2-40','IGKV2D-28'],
        ['IGKV2-40','IGKV2D-30'],
        ['IGKV2D-24','IGKV2D-40'],
        ['IGKV2D-28','IGKV2D-40'],
        ['IGKV2D-28','IGKV2OR22-4'],
        ['IGKV2D-30','IGKV2D-40'],
        ['IGKV3-11','IGKV3-15'],
        ['IGKV3-11','IGKV3-20'],
        ['IGKV3-11','IGKV3D-11'],
        ['IGKV3-11','IGKV3D-15'],
        ['IGKV3-11','IGKV3D-20'],
        ['IGKV3-15','IGKV3-20'],
        ['IGKV3-15','IGKV3-7'],
        ['IGKV3-15','IGKV3D-11'],
        ['IGKV3-15','IGKV3D-20'],
        ['IGKV3-20','IGKV3-7'],
        ['IGKV3-20','IGKV3D-11'],
        ['IGKV3-20','IGKV3D-15'],
        ['IGKV3-20','IGKV3D-20'],
        ['IGKV3-20','IGKV3OR2-268'],
        ['IGKV3-7','IGKV3D-11'],
        ['IGKV3-7','IGKV3D-15'],
        ['IGKV3-7','IGKV3D-20'],
        ['IGKV3D-11','IGKV3D-15'],
        ['IGKV3D-11','IGKV3D-20'],
        ['IGKV3D-11','IGKV3OR2-268'],
        ['IGKV3D-15','IGKV3D-20'],
        ['IGKV3D-15','IGKV3D-7'],
        ['IGKV3D-15','IGKV3OR2-268'],
        ['IGLC1','IGLC2'],
        ['IGLC1','IGLC3'],
        ['IGLC1','IGLC7'],
        ['IGLC1','IGLV1-36'],
        ['IGLC1','IGLV1-40'],
        ['IGLC1','IGLV1-44'],
        ['IGLC1','IGLV1-47'],
        ['IGLC1','IGLV1-50'],
        ['IGLC1','IGLV1-51'],
        ['IGLC1','IGLV2-11'],
        ['IGLC1','IGLV2-14'],
        ['IGLC1','IGLV2-18'],
        ['IGLC1','IGLV2-23'],
        ['IGLC1','IGLV2-34'],
        ['IGLC1','IGLV2-8'],
        ['IGLC1','IGLV3-1'],
        ['IGLC1','IGLV3-10'],
        ['IGLC1','IGLV3-19'],
        ['IGLC1','IGLV3-21'],
        ['IGLC1','IGLV3-25'],
        ['IGLC1','IGLV3-27'],
        ['IGLC1','IGLV3-9'],
        ['IGLC1','IGLV4-60'],
        ['IGLC1','IGLV4-69'],
        ['IGLC1','IGLV6-57'],
        ['IGLC1','IGLV7-43'],
        ['IGLC1','IGLV8-61'],
        ['IGLC1','IGLV9-49'],
        ['IGLC2','IGLC3'],
        ['IGLC2','IGLC6'],
        ['IGLC2','IGLL5'],
        ['IGLC2','IGLV1-40'],
        ['IGLC2','IGLV1-47'],
        ['IGLC2','IGLV1-51'],
        ['IGLC2','IGLV10-54'],
        ['IGLC2','IGLV2-11'],
        ['IGLC2','IGLV2-14'],
        ['IGLC2','IGLV2-23'],
        ['IGLC2','IGLV2-8'],
        ['IGLC2','IGLV3-1'],
        ['IGLC2','IGLV3-10'],
        ['IGLC2','IGLV3-19'],
        ['IGLC2','IGLV3-21'],
        ['IGLC2','IGLV3-25'],
        ['IGLC2','IGLV3-27'],
        ['IGLC2','IGLV4-60'],
        ['IGLC2','IGLV4-69'],
        ['IGLC2','IGLV5-45'],
        ['IGLC2','IGLV6-57'],
        ['IGLC2','IGLV7-43'],
        ['IGLC2','IGLV7-46'],
        ['IGLC2','IGLV8-61'],
        ['IGLC2','IGLV9-49'],
        ['IGLC3','IGLL5'],
        ['IGLC7','IGLL5'],
        ['IGLC7','IGLV1-47'],
        ['IGLC7','IGLV3-21'],
        ['IGLC7','IGLV6-57'],
        ['IGLL5','IGLV1-36'],
        ['IGLL5','IGLV1-40'],
        ['IGLL5','IGLV1-41'],
        ['IGLL5','IGLV1-44'],
        ['IGLL5','IGLV1-47'],
        ['IGLL5','IGLV1-50'],
        ['IGLL5','IGLV1-51'],
        ['IGLL5','IGLV10-54'],
        ['IGLL5','IGLV2-11'],
        ['IGLL5','IGLV2-14'],
        ['IGLL5','IGLV2-18'],
        ['IGLL5','IGLV2-23'],
        ['IGLL5','IGLV2-34'],
        ['IGLL5','IGLV2-8'],
        ['IGLL5','IGLV3-1'],
        ['IGLL5','IGLV3-10'],
        ['IGLL5','IGLV3-19'],
        ['IGLL5','IGLV3-21'],
        ['IGLL5','IGLV3-25'],
        ['IGLL5','IGLV3-27'],
        ['IGLL5','IGLV3-9'],
        ['IGLL5','IGLV4-60'],
        ['IGLL5','IGLV4-69'],
        ['IGLL5','IGLV5-45'],
        ['IGLL5','IGLV6-57'],
        ['IGLL5','IGLV7-43'],
        ['IGLL5','IGLV8-61'],
        ['IGLL5','IGLV9-49'],
        ['IGLV2-11','IGLV2-14'],
        ['IGLV2-11','IGLV2-18'],
        ['IGLV2-11','IGLV2-23'],
        ['IGLV2-11','IGLV2-8'],
        ['IGLV2-14','IGLV2-18'],
        ['IGLV2-14','IGLV2-23'],
        ['IGLV2-23','IGLV2-8'],
        ['IGLV3-10','IGLV3-25'],
        ['IGLV3-16','IGLV3-25'],
        ['IGLV3-21','IGLV3-9'],
        ['IGLV7-43','IGLV7-46'],
        ['IL4I1','NUP62'],
        ['INE1','UBA1'],
        ['IYD','TG'],
        ['KIAA1614-AS1','STX6'],
        ['KLHL23','SSB'],
        ['KRT5','KRT6A'],
        ['LBHD1','UBXN1'],
        ['LCAT','SLC12A4'],
        ['LILRA1','LILRB1'],
        ['LILRA1','LILRB2'],
        ['LILRA2','LILRB1'],
        ['LILRA6','LILRB3'],
        ['LILRB1','LILRB2'],
        ['LILRB1','LILRB3'],
        ['LILRB1','LILRB4'],
        ['LILRB2','LILRB3'],
        ['LILRB3','LILRB5'],
        ['LINC00665','LINC01535'],
        ['LINC00969','SDHA'],
        ['LINC01089','RHOF'],
        ['LINC01251','UBE2R2-AS1'],
        ['LPA','PLG'],
        ['LRG1','TAPBP'],
        ['LRG1','TAT'],
        ['LRRC37A17P','LRRC37A4P'],
        ['LTB4R2','NOP9'],
        ['LYZ','MROH1'],
        ['LYZ','MRPS30'],
        ['LYZ','TNFRSF1B'],
        ['MALAT1','TG'],
        ['MALAT1','TPO'],
        ['MAP1LC3B','MAP1LC3B2'],
        ['MB','TG'],
        ['MBP','TUBB4A'],
        ['METTL2A','METTL2B'],
        ['METTL7A','TG'],
        ['MICA','MICB'],
        ['MRPS6','SLC5A3'],
        ['MSANTD3','MSANTD3-TMEFF1'],
        ['MSANTD3','TMEFF1'],
        ['MT1B','MT1E'],
        ['MT1E','MT1JP'],
        ['MT1G','MT1M'],
        ['MT1G','MT2A'],
        ['MT1JP','MT1M'],
        ['MT1X','MT2A'],
        ['MYH11','TPM2'],
        ['MYH6','MYH7'],
        ['MYH7','MYL2'],
        ['MYH7','TG'],
        ['MYH7','TNNT2'],
        ['MYH7','TPM1'],
        ['MYH7','TTN'],
        ['MYH9','TG'],
        ['MZT2A','MZT2B'],
        ['NACA','NACA3P'],
        ['NACA','PRIM1'],
        ['NBEAL1','RPL12'],
        ['NBPF1','NBPF25P'],
        ['NBPF1','NBPF3'],
        ['NBPF10','NBPF8'],
        ['NBPF14','NOTCH2NL'],
        ['NBPF2P','NBPF3'],
        ['NEAT1','TG'],
        ['NEB','TG'],
        ['NFIC','TG'],
        ['NGRN','PERP'],
        ['NKX2-1','SFTA3'],
        ['NPIPA2','NPIPP1'],
        ['NPIPA3','NPIPP1'],
        ['NRAP','TG'],
        ['OBP2A','OBP2B'],
        ['OR2W3','TRIM58'],
        ['ORM1','ORM2'],
        ['ORM1','SERPINA1'],
        ['P2RY11','PPAN'],
        ['PABPC1','PABPC3'],
        ['PAFAH1B1','RN7SL608P'],
        ['PAFAH1B2','YIPF5'],
        ['PAGR1','PRRT2'],
        ['PCBP2','PCBP2-OT1'],
        ['PDIA4','TG'],
        ['PEX11A','PLIN1'],
        ['PILRA','PILRB'],
        ['PILRA','STAG3L5P-PVRIG2P-PILRB'],
        ['PMS2','PMS2CL'],
        ['POM121','POM121C'],
        ['PPP1R2','PPP1R2P3'],
        ['PRB1','PRB2'],
        ['PSAP','TG'],
        ['PTMS','RPS2'],
        ['PYGM','TG'],
        ['RABL2A','RABL2B'],
        ['RBMX','RBMXL1'],
        ['RPL10','RPL10P15'],
        ['RPL10','RPL10P3'],
        ['RPL13A','RPL13AP25'],
        ['RPL13A','RPL13AP3'],
        ['RPL13A','RPL13AP6'],
        ['RPL13A','RPL13AP7'],
        ['RPL23A','RPL23AP3'],
        ['RPL24','RPL24P2'],
        ['RPL24','RPL24P8'],
        ['RPL3','RPL3P2'],
        ['RPL3','RPL3P3'],
        ['RPL3','RPL3P6'],
        ['RPL3','RPL3P7'],
        ['RPL3','RPL3P9'],
        ['RPL37','RPL37P6'],
        ['RPL37A','RPL37AP8'],
        ['RPL4','RPL4P3'],
        ['RPL4','RPL4P4'],
        ['RPL4','RPL4P5'],
        ['RPL5','RPL5P1'],
        ['RPL5','RPL5P12'],
        ['RPL5','RPL5P18'],
        ['RPL5','RPL5P22'],
        ['RPL5','RPL5P23'],
        ['RPL5','RPL5P24'],
        ['RPL5','RPL5P29'],
        ['RPL5','RPL5P8'],
        ['RPL7A','RPL7AP11'],
        ['RPL7A','RPL7AP6'],
        ['RPL7A','RPL7AP66'],
        ['RPL7L1','WAC-AS1'],
        ['RPL8','TG'],
        ['RPL9','TIPIN'],
        ['RPLP0','RPLP0P6'],
        ['RPS10','RPS10P2'],
        ['RPS10-NUDT3','RPS10P2'],
        ['RPS15A','RPS15AP10'],
        ['RPS3A','RPS3AP49'],
        ['RPS4X','RPS4XP1'],
        ['RPS4X','RPS4XP13'],
        ['RPS4X','RPS4XP2'],
        ['RPS4X','RPS4XP22'],
        ['RPS4X','RPS4XP6'],
        ['RPS4X','RPS4XP7'],
        ['RPS6','TG'],
        ['RPSA','RPSAP53'],
        ['RRP7A','RRP7BP'],
        ['SAA1','SAA2'],
        ['SAA1','SAA2-SAA4'],
        ['SAA1','SERPINA1'],
        ['SAA2','SERPINA1'],
        ['SAA2-SAA4','SERPINA1'],
        ['SAG','VPS13C'],
        ['SERPINA1','SERPINA3'],
        ['SFTPA1','SFTPA2'],
        ['SIDT2','TAGLN'],
        ['SIGLEC11','SIGLEC16'],
        ['SIRPA','SIRPB1'],
        ['SLC35E2','SLC35E2B'],
        ['SLFN11','SLFN13'],
        ['SMAD2','SOD2'],
        ['SNHG14','SNRPN'],
        ['SOD2','TG'],
        ['SORD','SORD2P'],
        ['SQSTM1','TG'],
        ['SRRM2','TG'],
        ['STAT1','STAT4'],
        ['STK19','STK19B'],
        ['SULT1A1','SULT1A2'],
        ['TCEAL3','TCEAL6'],
        ['TG','TIMP3'],
        ['TG','TMBIM6'],
        ['TG','TPO'],
        ['TG','TPT1'],
        ['TG','TTN'],
        ['TG','TXNIP'],
        ['TG','UBC'],
        ['TG','VIM'],
        ['TG','XIRP1'],
        ['TG','ZFP36L1'],
        ['TMEM203','TPRN'],
        ['TNFSF10','TVP23B'],
        ['TPSAB1','TPSB2'],
        ['TPSAB1','TPSD1'],
        ['TPSB2','TPSD1'],
        ['TRAC','TRAV12-2'],
        ['TRAC','TRAV12-3'],
        ['TRAC','TRAV13-1'],
        ['TRAC','TRAV14DV4'],
        ['TRAC','TRAV2'],
        ['TRAC','TRAV8-4'],
        ['TRAC','TRAV9-2'],
        ['TRGC1','TRGC2'],
        ['TUBA1A','TUBA1B'],
        ['TUBA1A','TUBA1C'],
        ['TUBA1B','TUBA1C'],
        ['TUBA3C','TUBA3E'],
        ['TUBA3D','TUBA3E'],
        ['TUBG1','TUBG2'],
        ['UBB','UBC'],
        ['UGT2B11','UGT2B7'],
        ['UGT2B15','UGT2B17'],
        ['UGT2B28','UGT2B7'],
        ['UQCRH','UQCRHL'],
        ['VAMP5','VAMP8'],
        ['WASHC2A','WASHC2C'],
        ['ZDHHC11','ZDHHC11B'],
        ['ZNF181','ZNF302'],
        ['ZNF197','ZNF660'],
        ['ZNF221','ZNF224'],
        ['ZNF32-AS2','ZNF32-AS3'],
        ['ZNF33A','ZNF33B'],
        ['ZNF37A','ZNF37BP'],
        ['AC104297.1','PPA1'],
        ['ACTN1','HMGN1'],
        ['ADAMTSL4-AS1','MCL1'],
        ['ADH1B','ADH1C'],
        ['AKR1B10','AKR1B15'],
        ['AKR1C1','AKR1C2'],
        ['ANKRD18A','ANKRD18B'],
        ['ANKRD36','ANKRD36B'],
        ['ANKRD36','ANKRD36C'],
        ['ANKRD36B','ANKRD36C'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ASAH2','ASAH2B'],
        ['ATP5E','ATP5EP2'],
        ['ATXN3','MAB21L1'],
        ['AURKA','AURKAPS1'],
        ['BBS5','DTD2'],
        ['BTN3A1','BTN3A2'],
        ['C12ORF74','PLEKHG7'],
        ['C3ORF62','USP4'],
        ['CBWD1','CBWD2'],
        ['CCDC74A','CCDC74B'],
        ['CDK11A','CDK11B'],
        ['CDRT4','TVP23C'],
        ['CEACAM1','CEACAM5'],
        ['CEACAM1','CEACAM6'],
        ['CEACAM5','CEACAM6'],
        ['CFH','CFHR3'],
        ['CNTNAP3','CNTNAP3B'],
        ['COMMD1','RPSA'],
        ['CST1','CST2'],
        ['CTAGE11P','CTAGE5'],
        ['CYP2A6','CYP2A7'],
        ['CYP2B6','CYP2B7P'],
        ['CYP2F1','CYP2F2P'],
        ['CYP4A11','CYP4A22'],
        ['CYP4B1','CYP4X1'],
        ['CYP4Z1','CYP4Z2P'],
        ['DNAJC9','MRPS16'],
        ['ENPP1','KCNE3'],
        ['EPPIN','EPPIN-WFDC6'],
        ['EPPIN','WFDC6'],
        ['ERVK13-1','PDPK1'],
        ['FAM157A','FAM157B'],
        ['FAM35A','FAM35BP'],
        ['FAM60A','FAM60BP'],
        ['FCGR2A','FCGR2C'],
        ['FCGR3A','FCGR3B'],
        ['GBP4','GBP7'],
        ['GGA2','ZFAND5'],
        ['GOLGA8A','GOLGA8B'],
        ['GSTA1','GSTA2'],
        ['GSTA1','GSTA7P'],
        ['HCAR2','HCAR3'],
        ['HLA-A','HLA-B'],
        ['HLA-A','HLA-C'],
        ['HLA-A','HLA-G'],
        ['HLA-A','HLA-H'],
        ['HLA-A','HLA-J'],
        ['HLA-B','HLA-C'],
        ['HLA-B','HLA-E'],
        ['HLA-B','HLA-F'],
        ['HLA-B','HLA-G'],
        ['HLA-B','HLA-H'],
        ['HLA-C','HLA-E'],
        ['HLA-C','HLA-F'],
        ['HLA-C','HLA-H'],
        ['HLA-DPB1','HLA-DRB1'],
        ['HLA-DPB1','HLA-DRB5'],
        ['HLA-DQA1','HLA-DQA2'],
        ['HLA-DRB1','HLA-DRB5'],
        ['HLA-DRB1','HLA-DRB6'],
        ['HLA-DRB5','HLA-DRB6'],
        ['HLA-E','HLA-G'],
        ['HLA-G','HLA-H'],
        ['HMGB1','SP100'],
        ['HSP90AA1','HSP90AA2P'],
        ['HSP90AA1','HSP90AA4P'],
        ['HSP90AB1','HSP90AB2P'],
        ['HSP90B1','HSP90B2P'],
        ['HYDIN','HYDIN2'],
        ['ICA1L','PYURF'],
        ['IFITM2','IFITM3'],
        ['IGH@','IGK@'],
        ['KLRC2','KLRC3'],
        ['KRT5','KRT6A'],
        ['LGALS9B','LGALS9C'],
        ['LINC00969','SDHA'],
        ['LRRC37A17P','LRRC37A4P'],
        ['MAP1LC3B','MAP1LC3B2'],
        ['METTL2A','METTL2B'],
        ['MUC4','MUC5AC'],
        ['NACA','NACA3P'],
        ['NBPF1','NBPF25P'],
        ['NBPF1','NBPF3'],
        ['NBPF10','NBPF8'],
        ['NBPF11','NBPF12'],
        ['NBPF14','NOTCH2NL'],
        ['NBPF2P','NBPF3'],
        ['NGRN','PERP'],
        ['NOTCH2','NOTCH2NL'],
        ['NPIPA1','NPIPA5'],
        ['NPIPA2','NPIPP1'],
        ['NPIPA3','NPIPA5'],
        ['NPIPA3','NPIPP1'],
        ['PABPC1','PABPC3'],
        ['PAGR1','PRRT2'],
        ['PILRA','PILRB'],
        ['PILRA','STAG3L5P-PVRIG2P-PILRB'],
        ['PIP5K1A','PIPSL'],
        ['PLCG2','RN7SK'],
        ['PLCG2','RN7SKP255'],
        ['PMS2','PMS2CL'],
        ['POM121','POM121C'],
        ['PVRIG','STAG3'],
        ['RABL2A','RABL2B'],
        ['RBMX','RBMXL1'],
        ['RN7SK','RN7SKP203'],
        ['RN7SK','RN7SKP230'],
        ['RN7SK','RN7SKP255'],
        ['RN7SK','RN7SKP80'],
        ['RN7SL1','RN7SL674P'],
        ['RN7SL151P','RN7SL2'],
        ['RN7SL2','RN7SL3'],
        ['RN7SL2','RN7SL381P'],
        ['RN7SL2','RN7SL4P'],
        ['RPL23A','RPL23AP3'],
        ['RPL23AP7','RPL23AP82'],
        ['RPL24','RPL24P4'],
        ['RPL3','RPL3P2'],
        ['RPL4','RPL4P5'],
        ['RPL5','RPL5P12'],
        ['RPL5','RPL5P17'],
        ['RPL5','RPL5P34'],
        ['RPL5','RPL5P4'],
        ['RPL7L1','WAC-AS1'],
        ['RPLP0','RPLP0P6'],
        ['RPS20','RPS20P10'],
        ['RPS3A','RPS3AP49'],
        ['RPS4X','RPS4XP22'],
        ['RPSA','RPSAP53'],
        ['RPSA','RPSAP8'],
        ['RRP7A','RRP7BP'],
        ['SAA1','SAA2'],
        ['SCNN1A','TNFRSF1A'],
        ['SERPINB3','SERPINB4'],
        ['SERPINI2','WDR49'],
        ['SLFN11','SLFN13'],
        ['SORD','SORD2P'],
        ['SULT1A1','SULT1A2'],
        ['TPSAB1','TPSB2'],
        ['TUBA1A','TUBA1B'],
        ['TUBA1A','TUBA1C'],
        ['TUBA1B','TUBA1C'],
        ['TUBG1','TUBG2'],
        ['WASHC2A','WASHC2C'],
        ['ZDHHC11','ZDHHC11B'],
        ['ZNF33A','ZNF33B'],
        ['ZNF587B','ZNF814'],
        ['AC104297.1','PPA1'],
        ['ACTN1','HMGN1'],
        ['ADAMTSL4-AS1','MCL1'],
        ['ADGRE2','ADGRE5'],
        ['ADH1B','ADH1C'],
        ['AKR1B10','AKR1B15'],
        ['AKR1C1','AKR1C2'],
        ['ANKRD18A','ANKRD18B'],
        ['ANKRD36','ANKRD36B'],
        ['ANKRD36','ANKRD36C'],
        ['ANKRD36B','ANKRD36C'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ASAH2','ASAH2B'],
        ['ATP5E','ATP5EP2'],
        ['ATXN3','MAB21L1'],
        ['AURKA','AURKAPS1'],
        ['AZGP1','GJC3'],
        ['BBS5','DTD2'],
        ['BTN3A1','BTN3A2'],
        ['C12ORF74','PLEKHG7'],
        ['C3ORF62','USP4'],
        ['CBWD1','CBWD2'],
        ['CCDC74A','CCDC74B'],
        ['CDK11A','CDK11B'],
        ['CDRT4','TVP23C'],
        ['CEACAM1','CEACAM5'],
        ['CEACAM1','CEACAM6'],
        ['CEACAM5','CEACAM6'],
        ['CFH','CFHR3'],
        ['CNTNAP3','CNTNAP3B'],
        ['COMMD1','RPSA'],
        ['CST1','CST2'],
        ['CTAGE11P','CTAGE5'],
        ['CYP2A6','CYP2A7'],
        ['CYP2B6','CYP2B7P'],
        ['CYP2F1','CYP2F2P'],
        ['CYP4A11','CYP4A22'],
        ['CYP4B1','CYP4X1'],
        ['CYP4Z1','CYP4Z2P'],
        ['DNAJC9','MRPS16'],
        ['ENPP1','KCNE3'],
        ['EPPIN','EPPIN-WFDC6'],
        ['EPPIN','WFDC6'],
        ['ERVK13-1','PDPK1'],
        ['FAM157A','FAM157B'],
        ['FAM35A','FAM35BP'],
        ['FAM60A','FAM60BP'],
        ['FCGR2A','FCGR2C'],
        ['FCGR3A','FCGR3B'],
        ['GBP4','GBP7'],
        ['GGA2','ZFAND5'],
        ['GOLGA8A','GOLGA8B'],
        ['GSTA1','GSTA2'],
        ['GSTA1','GSTA7P'],
        ['HCAR2','HCAR3'],
        ['HLA-A','HLA-B'],
        ['HLA-A','HLA-C'],
        ['HLA-A','HLA-G'],
        ['HLA-A','HLA-H'],
        ['HLA-A','HLA-J'],
        ['HLA-B','HLA-C'],
        ['HLA-B','HLA-E'],
        ['HLA-B','HLA-F'],
        ['HLA-B','HLA-G'],
        ['HLA-B','HLA-H'],
        ['HLA-C','HLA-E'],
        ['HLA-C','HLA-F'],
        ['HLA-C','HLA-H'],
        ['HLA-DPB1','HLA-DRB1'],
        ['HLA-DPB1','HLA-DRB5'],
        ['HLA-DQA1','HLA-DQA2'],
        ['HLA-DRB1','HLA-DRB5'],
        ['HLA-DRB1','HLA-DRB6'],
        ['HLA-DRB5','HLA-DRB6'],
        ['HLA-E','HLA-G'],
        ['HLA-G','HLA-H'],
        ['HMGB1','SP100'],
        ['HSP90AA1','HSP90AA2P'],
        ['HSP90AA1','HSP90AA4P'],
        ['HSP90AB1','HSP90AB2P'],
        ['HSP90B1','HSP90B2P'],
        ['HYDIN','HYDIN2'],
        ['ICA1L','PYURF'],
        ['IFITM2','IFITM3'],
        ['IGH@','IGK@'],
        ['KLRC2','KLRC3'],
        ['KRT5','KRT6A'],
        ['LGALS9B','LGALS9C'],
        ['LINC00969','SDHA'],
        ['LRRC37A17P','LRRC37A4P'],
        ['MAP1LC3B','MAP1LC3B2'],
        ['METTL2A','METTL2B'],
        ['MUC4','MUC5AC'],
        ['NACA','NACA3P'],
        ['NBPF1','NBPF25P'],
        ['NBPF1','NBPF3'],
        ['NBPF10','NBPF8'],
        ['NBPF11','NBPF12'],
        ['NBPF14','NOTCH2NL'],
        ['NBPF2P','NBPF3'],
        ['NGRN','PERP'],
        ['NOTCH2','NOTCH2NL'],
        ['NPIPA1','NPIPA5'],
        ['NPIPA2','NPIPP1'],
        ['NPIPA3','NPIPA5'],
        ['NPIPA3','NPIPP1'],
        ['PABPC1','PABPC3'],
        ['PAGR1','PRRT2'],
        ['PILRA','PILRB'],
        ['PILRA','STAG3L5P-PVRIG2P-PILRB'],
        ['PIP5K1A','PIPSL'],
        ['PLCG2','RN7SK'],
        ['PLCG2','RN7SKP255'],
        ['PMS2','PMS2CL'],
        ['POM121','POM121C'],
        ['PVRIG','STAG3'],
        ['RABL2A','RABL2B'],
        ['RBMX','RBMXL1'],
        ['RN7SK','RN7SKP203'],
        ['RN7SK','RN7SKP230'],
        ['RN7SK','RN7SKP255'],
        ['RN7SK','RN7SKP80'],
        ['RN7SL1','RN7SL674P'],
        ['RN7SL151P','RN7SL2'],
        ['RN7SL2','RN7SL3'],
        ['RN7SL2','RN7SL381P'],
        ['RN7SL2','RN7SL4P'],
        ['RPL23A','RPL23AP3'],
        ['RPL23AP7','RPL23AP82'],
        ['RPL24','RPL24P4'],
        ['RPL3','RPL3P2'],
        ['RPL4','RPL4P5'],
        ['RPL5','RPL5P12'],
        ['RPL5','RPL5P17'],
        ['RPL5','RPL5P34'],
        ['RPL5','RPL5P4'],
        ['RPL7L1','WAC-AS1'],
        ['RPLP0','RPLP0P6'],
        ['RPS20','RPS20P10'],
        ['RPS3A','RPS3AP49'],
        ['RPS4X','RPS4XP22'],
        ['RPSA','RPSAP53'],
        ['RPSA','RPSAP8'],
        ['RRP7A','RRP7BP'],
        ['SAA1','SAA2'],
        ['SCNN1A','TNFRSF1A'],
        ['SERPINB3','SERPINB4'],
        ['SERPINI2','WDR49'],
        ['SLFN11','SLFN13'],
        ['SORD','SORD2P'],
        ['SULT1A1','SULT1A2'],
        ['TPSAB1','TPSB2'],
        ['TUBA1A','TUBA1B'],
        ['TUBA1A','TUBA1C'],
        ['TUBA1B','TUBA1C'],
        ['TUBG1','TUBG2'],
        ['WASHC2A','WASHC2C'],
        ['ZDHHC11','ZDHHC11B'],
        ['ZNF33A','ZNF33B'],
        ['ZNF587B','ZNF814'],
        ['A2M','ALB'],
        ['A2M','PZP'],
        ['A2M','SFTPA2'],
        ['A2M','SFTPC'],
        ['ABCB8','ASIC3'],
        ['AC009228.1','FAM228A'],
        ['AC079250.1','RPL18A'],
        ['AC125232.1','RMND5A'],
        ['AC142381.1','IGHV3-23'],
        ['ACAD8','GLB1L3'],
        ['ACCS','EXT2'],
        ['ACKR1','CADM3'],
        ['ACRBP','LPAR5'],
        ['ACSM2A','ACSM2B'],
        ['ACTA1','ACTB'],
        ['ACTA1','ACTG1'],
        ['ACTA1','DES'],
        ['ACTA2','ACTB'],
        ['ACTA2','ACTG1'],
        ['ACTA2','ACTG2'],
        ['ACTA2-AS1','ACTB'],
        ['ACTA2-AS1','ACTG1'],
        ['ACTB','ACTC1'],
        ['ACTB','ACTG1'],
        ['ACTB','ACTG2'],
        ['ACTB','COL3A1'],
        ['ACTB','IGH@'],
        ['ACTB','POTEF'],
        ['ACTB','SFTPA1'],
        ['ACTB','SFTPC'],
        ['ACTB','SMR3B'],
        ['ACTB','TG'],
        ['ACTC1','ACTG1'],
        ['ACTG1','ACTG2'],
        ['ACTG1','GNAS'],
        ['ACTG1','KRT10'],
        ['ACTG1','SFTPA1'],
        ['ACTG1','SFTPA2'],
        ['ACTG1','TG'],
        ['ACTN1','HMGN1'],
        ['ADAMTSL4-AS1','MCL1'],
        ['ADGRE2','ADGRE5'],
        ['ADH1A','ADH1B'],
        ['ADH1B','ADH1C'],
        ['AHNAK','SFTPA2'],
        ['AK1','ST6GALNAC6'],
        ['AK6','TAF9'],
        ['AKR1B10','AKR1B15'],
        ['AKR1C1','AKR1C2'],
        ['AKR7A2','AKR7A3'],
        ['AL591845.1','SH3D21'],
        ['ALB','AMBP'],
        ['ALB','C3'],
        ['ALB','CYP2E1'],
        ['ALB','FGA'],
        ['ALB','FGB'],
        ['ALB','FGG'],
        ['ALB','HP'],
        ['ALB','HPX'],
        ['ALB','ORM1'],
        ['ALB','SAA1'],
        ['ALB','SERPINA1'],
        ['ALB','TF'],
        ['ALDH3B1','ALDH3B2'],
        ['ALMS1','ALMS1-IT1'],
        ['AMACR','C1QTNF3'],
        ['AMBP','SERPINA1'],
        ['ANG','RNASE4'],
        ['ANKDD1A','PLEKHO2'],
        ['ANKRD18A','ANKRD18B'],
        ['ANKRD36','ANKRD36B'],
        ['ANKRD36','ANKRD36C'],
        ['ANKRD36B','ANKRD36C'],
        ['ANXA6','TNIP1'],
        ['AP1G2','JPH4'],
        ['APBB3','SRA1'],
        ['APLP2','TG'],
        ['APOA1','SERPINA1'],
        ['APOB','SERPINA1'],
        ['APOBEC3A','APOBEC3B'],
        ['APOLD1','SERBP1'],
        ['APP','TG'],
        ['AQP7','FP325317.1'],
        ['ARHGAP42','TMEM133'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ASAH2','ASAH2B'],
        ['ASPRV1','PCBP1-AS1'],
        ['ATAD3A','ATAD3B'],
        ['ATF6B','TNXB'],
        ['ATP1A1','TG'],
        ['ATP1A2','ATP1A4'],
        ['ATP5E','ATP5EP2'],
        ['ATP6V1B1','VAX2'],
        ['ATP8B5P','ZFAND6'],
        ['ATPAF1','EFCAB14'],
        ['ATXN3','MAB21L1'],
        ['ATXN7','PSMD6-AS2'],
        ['AZGP1','GJC3'],
        ['B2M','SFTPC'],
        ['B2M','TG'],
        ['BACH1','BACH1-IT1'],
        ['BAG6','C6ORF47'],
        ['BBS5','DTD2'],
        ['BCAP29','DUS4L'],
        ['BCAS2','DENND2C'],
        ['BCR','BCRP3'],
        ['BEX1','BEX2'],
        ['BGN','GADL1'],
        ['BRD9','ZDHHC11B'],
        ['BRICD5','PGP'],
        ['BRWD1-AS1','BRWD1-AS2'],
        ['BTN2A1','BTN3A3'],
        ['BTN3A1','BTN3A2'],
        ['BTNL3','BTNL8'],
        ['C15ORF57','CBX3'],
        ['C17ORF47','SEPT4'],
        ['C19ORF24','CIRBP'],
        ['C1ORF74','IRF6'],
        ['C1QA','C1QC'],
        ['C3','FGA'],
        ['C3','HP'],
        ['C3','SERPINA1'],
        ['C3ORF62','USP4'],
        ['C4A','C4B'],
        ['C4ORF3','FABP2'],
        ['C4ORF36','SLC10A6'],
        ['C5AR1','CD84'],
        ['C5ORF46','SPINK1'],
        ['C9ORF172','PHPT1'],
        ['CALR','TG'],
        ['CALR3','EPS15L1'],
        ['CANX','TG'],
        ['CAPN3','GANC'],
        ['CAPS','VMAC'],
        ['CATSPERB','TC2N'],
        ['CAVIN4','MSANTD3-TMEFF1'],
        ['CAVIN4','TMEFF1'],
        ['CBWD1','CBWD2'],
        ['CCDC169','SOHLH2'],
        ['CCDC58','WDR5B'],
        ['CCDC74A','CCDC74B'],
        ['CCL5','LYZ'],
        ['CCPG1','PIGBOS1'],
        ['CCT6A','CCT6P1'],
        ['CD22','MAG'],
        ['CD74','IGH@'],
        ['CD74','IGHA1'],
        ['CD74','LYZ'],
        ['CD74','SFTPA1'],
        ['CD74','SFTPA2'],
        ['CD74','SFTPC'],
        ['CDK11A','CDK11B'],
        ['CDRT4','TVP23C'],
        ['CEACAM1','CEACAM5'],
        ['CEACAM1','CEACAM6'],
        ['CEACAM5','CEACAM6'],
        ['CEL','CELP'],
        ['CEL','GP2'],
        ['CELA3A','CELA3B'],
        ['CFAP53','MBD1'],
        ['CFH','CFHR1'],
        ['CFH','CFHR3'],
        ['CFHR1','CFHR2'],
        ['CHFR','GOLGA3'],
        ['CHUK','ERLIN1'],
        ['CKLF','CMTM1'],
        ['CKLF-CMTM1','CMTM1'],
        ['CKM','NPPA'],
        ['CLCNKA','CLCNKB'],
        ['CLIC1','DDAH2'],
        ['CLK4','RN7SKP70'],
        ['CNPY2','PAN2'],
        ['CNTNAP3','CNTNAP3B'],
        ['COL1A1','COL1A2'],
        ['COL1A1','COL3A1'],
        ['COL1A1','COL4A1'],
        ['COL1A1','IGF2'],
        ['COL1A1','KRT10'],
        ['COL1A2','COL3A1'],
        ['COL3A1','COL4A1'],
        ['COL3A1','FLNA'],
        ['COL3A1','FLT1'],
        ['COL3A1','FN1'],
        ['COL3A1','IGF2'],
        ['COL3A1','IGFBP3'],
        ['COL3A1','SPARC'],
        ['COL3A1','SPP1'],
        ['COL4A1','FN1'],
        ['COL4A2','TG'],
        ['COL7A1','UCN2'],
        ['COLQ','HACL1'],
        ['COMMD1','RPSA'],
        ['CPNE1','RBM12'],
        ['CRIP3','ZNF318'],
        ['CRIPAK','UVSSA'],
        ['CRTC2','SLC39A1'],
        ['CRY1','MTERF2'],
        ['CRYAB','DES'],
        ['CSF2RA','CTSB'],
        ['CSNK2B','LY6G5B'],
        ['CST1','CST2'],
        ['CTAGE11P','CTAGE5'],
        ['CTBS','GNG5'],
        ['CTSB','TG'],
        ['CUTC','ENTPD7'],
        ['CXORF65','IL2RG'],
        ['CYP21A1P','CYP21A2'],
        ['CYP2A6','CYP2A7'],
        ['CYP2B6','CYP2B7P'],
        ['CYP2D6','CYP2D7'],
        ['CYP2E1','HP'],
        ['CYP4A11','CYP4A22'],
        ['CYTOR','MIR4435-2HG'],
        ['DCUN1D3','ERI2'],
        ['DDX17','TG'],
        ['DDX3X','DDX3Y'],
        ['DDX56','TMED4'],
        ['DES','FLNC'],
        ['DES','MYH11'],
        ['DES','MYH7'],
        ['DES','MYL2'],
        ['DHRS1','RABGGTA'],
        ['DHRS4','DHRS4L2'],
        ['DLK1','IGF2'],
        ['DMKN','KRT1'],
        ['DMKN','KRTDAP'],
        ['DNAAF4-CCPG1','PIGBOS1'],
        ['DNAH1','GLYCTK'],
        ['DNAJC25-GNG10','GNG10'],
        ['DNAJC4','NUDT22'],
        ['DNAJC5','TPD52L2'],
        ['DNAJC9','MRPS16'],
        ['DSC3','KRT10'],
        ['DSP','KRT10'],
        ['DUS3L','PRR22'],
        ['DUSP1','TG'],
        ['DUT','ZNF534'],
        ['DYNLRB1','ITCH'],
        ['E2F4','RPL14'],
        ['EARS2','GGA2'],
        ['EDARADD','ENO1'],
        ['EEF1AKMT2','FAM53B'],
        ['EEF1G','TG'],
        ['EEF2','KRT10'],
        ['EEF2','TG'],
        ['EGR1','TG'],
        ['EIF4G2','TG'],
        ['EIF5A','EIF5AL1'],
        ['ELK4','SLC45A3'],
        ['ELOA','RPL11'],
        ['ELOVL1','MED8'],
        ['ENPP1','KCNE3'],
        ['EP400','TBP'],
        ['EPAS1','FN1'],
        ['EPOR','RGL3'],
        ['EPPIN','WFDC6'],
        ['EPPIN-WFDC6','WFDC6'],
        ['ERV3-1','ZNF117'],
        ['ERVV-1','ERVV-2'],
        ['FAAH','NSUN4'],
        ['FAAP20','PRKCZ-AS1'],
        ['FAHD2B','FAHD2CP'],
        ['FAM13A-AS1','HERC3'],
        ['FAM175A','HELQ'],
        ['FAM35A','FAM35BP'],
        ['FBXO44','FBXO6'],
        ['FCGBP','TG'],
        ['FCGR2A','FCGR2C'],
        ['FCGR2B','FCGR2C'],
        ['FCGR3A','FCGR3B'],
        ['FCRL1','FCRL2'],
        ['FEZ1','STT3A-AS1'],
        ['FGA','FGB'],
        ['FGA','HP'],
        ['FGA','SERPINA1'],
        ['FGB','HP'],
        ['FGB','SERPINA1'],
        ['FGF14','FGF14-IT1'],
        ['FGG','SERPINA1'],
        ['FKBP1A','FKBP1C'],
        ['FLG2','KRT1'],
        ['FLG2','KRT10'],
        ['FLNA','IGH@'],
        ['FMC1','LUC7L2'],
        ['FN1','FTL'],
        ['FN1','IGF2'],
        ['FN1','MEG3'],
        ['FN1','TIMP3'],
        ['FN1','TPT1'],
        ['FOLH1','FOLH1B'],
        ['FOS','TG'],
        ['FRG1','FRG1BP'],
        ['FTH1','SFTPC'],
        ['FTL','SFTPA1'],
        ['FTL','SFTPC'],
        ['FXYD2','FXYD6-FXYD2'],
        ['GALT','IL11RA'],
        ['GAPDH','IGH@'],
        ['GBP4','GBP7'],
        ['GCNT4','LINC01336'],
        ['GGA2','ZFAND5'],
        ['GLIPR1','GLIPR1L2'],
        ['GLUD1','GLUD2'],
        ['GLUL','TG'],
        ['GNAS','TG'],
        ['GOLGA8A','GOLGA8B'],
        ['GPR135','L3HYPDH'],
        ['GPX3','TG'],
        ['GRID1','RNF152'],
        ['GSTA1','GSTA2'],
        ['GSTM1','GSTM4'],
        ['GSTM1','GSTM5'],
        ['GSTM2','GSTM4'],
        ['GTSE1','TRMU'],
        ['GUSBP11','IGLL3P'],
        ['H19','IGF2'],
        ['H3F3B','H3F3C'],
        ['HARS2','ZMAT2'],
        ['HBB','HBD'],
        ['HBB','KLF6'],
        ['HCAR2','HCAR3'],
        ['HCG11','NORAD'],
        ['HECA','SLC48A1'],
        ['HLA-A','HLA-B'],
        ['HLA-A','HLA-C'],
        ['HLA-A','HLA-E'],
        ['HLA-A','HLA-F'],
        ['HLA-A','HLA-G'],
        ['HLA-A','HLA-H'],
        ['HLA-A','HLA-J'],
        ['HLA-A','HLA-L'],
        ['HLA-B','HLA-C'],
        ['HLA-B','HLA-E'],
        ['HLA-B','HLA-F'],
        ['HLA-B','HLA-G'],
        ['HLA-B','HLA-H'],
        ['HLA-B','HLA-J'],
        ['HLA-B','HLA-L'],
        ['HLA-C','HLA-E'],
        ['HLA-C','HLA-F'],
        ['HLA-C','HLA-G'],
        ['HLA-C','HLA-H'],
        ['HLA-C','HLA-L'],
        ['HLA-DPB1','HLA-DRB1'],
        ['HLA-DPB1','HLA-DRB5'],
        ['HLA-DQA1','HLA-DQA2'],
        ['HLA-DQB1','HLA-DQB2'],
        ['HLA-DRB1','HLA-DRB5'],
        ['HLA-DRB1','HLA-DRB6'],
        ['HLA-DRB1','HLA-DRB9'],
        ['HLA-DRB5','HLA-DRB6'],
        ['HLA-DRB5','HLA-DRB9'],
        ['HLA-E','HLA-H'],
        ['HLA-E','HLA-J'],
        ['HLA-G','HLA-H'],
        ['HLA-G','HLA-J'],
        ['HLA-G','HLA-L'],
        ['HLA-H','HLA-J'],
        ['HMGB1','SP100'],
        ['HOXB3','HOXB4'],
        ['HP','HPR'],
        ['HP','ORM1'],
        ['HP','SERPINA1'],
        ['HS1BP3','HS1BP3-IT1'],
        ['HSP90AA1','HSP90AA2P'],
        ['HSP90AA1','HSP90AA4P'],
        ['HSP90AB1','HSP90AB2P'],
        ['HSP90AB1','SLC29A1'],
        ['HSP90AB1','TG'],
        ['HSPA1A','HSPA1B'],
        ['HSPA1L','LSM2'],
        ['HYPK','SERF2'],
        ['IFI16','PYHIN1'],
        ['IFITM1','IFITM3'],
        ['IFITM2','IFITM3'],
        ['IFT80','TRIM59'],
        ['IGF2','MEG3'],
        ['IGF2','MRNIP'],
        ['IGF2','SPARC'],
        ['IGF2','SPP1'],
        ['IGF2','TIMP3'],
        ['IGH@','IGHA1'],
        ['IGH@','IGHA2'],
        ['IGH@','IGHD'],
        ['IGH@','IGHG1'],
        ['IGH@','IGHG2'],
        ['IGH@','IGHG3'],
        ['IGH@','IGHGP'],
        ['IGH@','IGHM'],
        ['IGH@','IGHV1-12'],
        ['IGH@','IGHV1-18'],
        ['IGH@','IGHV1-2'],
        ['IGH@','IGHV1-24'],
        ['IGH@','IGHV1-3'],
        ['IGH@','IGHV1-46'],
        ['IGH@','IGHV1-58'],
        ['IGH@','IGHV1-69'],
        ['IGH@','IGHV2-26'],
        ['IGH@','IGHV2-5'],
        ['IGH@','IGHV2-70'],
        ['IGH@','IGHV3-11'],
        ['IGH@','IGHV3-13'],
        ['IGH@','IGHV3-15'],
        ['IGH@','IGHV3-20'],
        ['IGH@','IGHV3-21'],
        ['IGH@','IGHV3-23'],
        ['IGH@','IGHV3-30'],
        ['IGH@','IGHV3-33'],
        ['IGH@','IGHV3-43'],
        ['IGH@','IGHV3-48'],
        ['IGH@','IGHV3-49'],
        ['IGH@','IGHV3-53'],
        ['IGH@','IGHV3-64'],
        ['IGH@','IGHV3-66'],
        ['IGH@','IGHV3-7'],
        ['IGH@','IGHV3-71'],
        ['IGH@','IGHV3-72'],
        ['IGH@','IGHV3-73'],
        ['IGH@','IGHV3-74'],
        ['IGH@','IGHV3OR16-8'],
        ['IGH@','IGHV3OR16-9'],
        ['IGH@','IGHV4-28'],
        ['IGH@','IGHV4-31'],
        ['IGH@','IGHV4-34'],
        ['IGH@','IGHV4-39'],
        ['IGH@','IGHV4-55'],
        ['IGH@','IGHV4-59'],
        ['IGH@','IGHV5-51'],
        ['IGH@','IGHV6-1'],
        ['IGH@','IGHV7-81'],
        ['IGH@','IGK@'],
        ['IGH@','IGKC'],
        ['IGH@','MYH11'],
        ['IGH@','RPS8'],
        ['IGH@','TPT1'],
        ['IGHA1','IGHA2'],
        ['IGHA1','IGHM'],
        ['IGHA1','IGHV1-18'],
        ['IGHA1','IGHV1-2'],
        ['IGHA1','IGHV1-24'],
        ['IGHA1','IGHV1-3'],
        ['IGHA1','IGHV1-46'],
        ['IGHA1','IGHV1-58'],
        ['IGHA1','IGHV1-69'],
        ['IGHA1','IGHV1OR15-1'],
        ['IGHA1','IGHV1OR16-3'],
        ['IGHA1','IGHV2-26'],
        ['IGHA1','IGHV2-5'],
        ['IGHA1','IGHV2-70'],
        ['IGHA1','IGHV3-11'],
        ['IGHA1','IGHV3-13'],
        ['IGHA1','IGHV3-15'],
        ['IGHA1','IGHV3-16'],
        ['IGHA1','IGHV3-20'],
        ['IGHA1','IGHV3-21'],
        ['IGHA1','IGHV3-22'],
        ['IGHA1','IGHV3-23'],
        ['IGHA1','IGHV3-30'],
        ['IGHA1','IGHV3-33'],
        ['IGHA1','IGHV3-41'],
        ['IGHA1','IGHV3-43'],
        ['IGHA1','IGHV3-48'],
        ['IGHA1','IGHV3-49'],
        ['IGHA1','IGHV3-53'],
        ['IGHA1','IGHV3-64'],
        ['IGHA1','IGHV3-65'],
        ['IGHA1','IGHV3-66'],
        ['IGHA1','IGHV3-7'],
        ['IGHA1','IGHV3-72'],
        ['IGHA1','IGHV3-73'],
        ['IGHA1','IGHV3-74'],
        ['IGHA1','IGHV3OR15-7'],
        ['IGHA1','IGHV3OR16-13'],
        ['IGHA1','IGHV3OR16-7'],
        ['IGHA1','IGHV3OR16-9'],
        ['IGHA1','IGHV4-28'],
        ['IGHA1','IGHV4-31'],
        ['IGHA1','IGHV4-34'],
        ['IGHA1','IGHV4-39'],
        ['IGHA1','IGHV4-4'],
        ['IGHA1','IGHV4-55'],
        ['IGHA1','IGHV4-59'],
        ['IGHA1','IGHV4-61'],
        ['IGHA1','IGHV4OR15-8'],
        ['IGHA1','IGHV5-51'],
        ['IGHA1','IGHV6-1'],
        ['IGHA1','IGK@'],
        ['IGHA1','IGKC'],
        ['IGHA2','IGHV1-12'],
        ['IGHA2','IGHV1-18'],
        ['IGHA2','IGHV1-2'],
        ['IGHA2','IGHV1-3'],
        ['IGHA2','IGHV1-46'],
        ['IGHA2','IGHV1-69'],
        ['IGHA2','IGHV1OR15-1'],
        ['IGHA2','IGHV1OR16-3'],
        ['IGHA2','IGHV1OR21-1'],
        ['IGHA2','IGHV2-26'],
        ['IGHA2','IGHV2-5'],
        ['IGHA2','IGHV2-70'],
        ['IGHA2','IGHV3-11'],
        ['IGHA2','IGHV3-13'],
        ['IGHA2','IGHV3-15'],
        ['IGHA2','IGHV3-16'],
        ['IGHA2','IGHV3-20'],
        ['IGHA2','IGHV3-21'],
        ['IGHA2','IGHV3-22'],
        ['IGHA2','IGHV3-23'],
        ['IGHA2','IGHV3-30'],
        ['IGHA2','IGHV3-33'],
        ['IGHA2','IGHV3-41'],
        ['IGHA2','IGHV3-43'],
        ['IGHA2','IGHV3-48'],
        ['IGHA2','IGHV3-49'],
        ['IGHA2','IGHV3-53'],
        ['IGHA2','IGHV3-64'],
        ['IGHA2','IGHV3-65'],
        ['IGHA2','IGHV3-66'],
        ['IGHA2','IGHV3-7'],
        ['IGHA2','IGHV3-71'],
        ['IGHA2','IGHV3-72'],
        ['IGHA2','IGHV3-73'],
        ['IGHA2','IGHV3-74'],
        ['IGHA2','IGHV3OR15-7'],
        ['IGHA2','IGHV3OR16-13'],
        ['IGHA2','IGHV3OR16-15'],
        ['IGHA2','IGHV3OR16-9'],
        ['IGHA2','IGHV4-28'],
        ['IGHA2','IGHV4-31'],
        ['IGHA2','IGHV4-34'],
        ['IGHA2','IGHV4-39'],
        ['IGHA2','IGHV4-4'],
        ['IGHA2','IGHV4-55'],
        ['IGHA2','IGHV4-59'],
        ['IGHA2','IGHV4-61'],
        ['IGHA2','IGHV4OR15-8'],
        ['IGHA2','IGHV5-51'],
        ['IGHA2','IGHV6-1'],
        ['IGHA2','IGK@'],
        ['IGHA2','IGKC'],
        ['IGHD','IGHV1-18'],
        ['IGHD','IGHV1-3'],
        ['IGHD','IGHV1-46'],
        ['IGHD','IGHV1-69'],
        ['IGHD','IGHV2-26'],
        ['IGHD','IGHV2-5'],
        ['IGHD','IGHV3-11'],
        ['IGHD','IGHV3-15'],
        ['IGHD','IGHV3-20'],
        ['IGHD','IGHV3-21'],
        ['IGHD','IGHV3-23'],
        ['IGHD','IGHV3-30'],
        ['IGHD','IGHV3-33'],
        ['IGHD','IGHV3-48'],
        ['IGHD','IGHV3-53'],
        ['IGHD','IGHV3-66'],
        ['IGHD','IGHV3-7'],
        ['IGHD','IGHV3-74'],
        ['IGHD','IGHV3OR16-9'],
        ['IGHD','IGHV4-31'],
        ['IGHD','IGHV4-34'],
        ['IGHD','IGHV4-39'],
        ['IGHD','IGHV4-4'],
        ['IGHD','IGHV4-59'],
        ['IGHD','IGHV4-61'],
        ['IGHD','IGHV4OR15-8'],
        ['IGHD','IGHV5-51'],
        ['IGHG1','IGHG2'],
        ['IGHG1','IGHG3'],
        ['IGHG1','IGHG4'],
        ['IGHG1','IGHGP'],
        ['IGHG1','IGHM'],
        ['IGHG1','IGHV1-12'],
        ['IGHG1','IGHV1-18'],
        ['IGHG1','IGHV1-2'],
        ['IGHG1','IGHV1-24'],
        ['IGHG1','IGHV1-3'],
        ['IGHG1','IGHV1-45'],
        ['IGHG1','IGHV1-46'],
        ['IGHG1','IGHV1-58'],
        ['IGHG1','IGHV1-67'],
        ['IGHG1','IGHV1-69'],
        ['IGHG1','IGHV1OR15-1'],
        ['IGHG1','IGHV1OR15-3'],
        ['IGHG1','IGHV1OR16-1'],
        ['IGHG1','IGHV1OR16-3'],
        ['IGHG1','IGHV1OR21-1'],
        ['IGHG1','IGHV2-26'],
        ['IGHG1','IGHV2-5'],
        ['IGHG1','IGHV2-70'],
        ['IGHG1','IGHV2OR16-5'],
        ['IGHG1','IGHV3-11'],
        ['IGHG1','IGHV3-13'],
        ['IGHG1','IGHV3-15'],
        ['IGHG1','IGHV3-16'],
        ['IGHG1','IGHV3-20'],
        ['IGHG1','IGHV3-21'],
        ['IGHG1','IGHV3-22'],
        ['IGHG1','IGHV3-23'],
        ['IGHG1','IGHV3-30'],
        ['IGHG1','IGHV3-33'],
        ['IGHG1','IGHV3-41'],
        ['IGHG1','IGHV3-43'],
        ['IGHG1','IGHV3-48'],
        ['IGHG1','IGHV3-49'],
        ['IGHG1','IGHV3-53'],
        ['IGHG1','IGHV3-62'],
        ['IGHG1','IGHV3-64'],
        ['IGHG1','IGHV3-65'],
        ['IGHG1','IGHV3-66'],
        ['IGHG1','IGHV3-7'],
        ['IGHG1','IGHV3-71'],
        ['IGHG1','IGHV3-72'],
        ['IGHG1','IGHV3-73'],
        ['IGHG1','IGHV3-74'],
        ['IGHG1','IGHV3OR15-7'],
        ['IGHG1','IGHV3OR16-13'],
        ['IGHG1','IGHV3OR16-15'],
        ['IGHG1','IGHV3OR16-6'],
        ['IGHG1','IGHV3OR16-7'],
        ['IGHG1','IGHV3OR16-9'],
        ['IGHG1','IGHV4-28'],
        ['IGHG1','IGHV4-31'],
        ['IGHG1','IGHV4-34'],
        ['IGHG1','IGHV4-39'],
        ['IGHG1','IGHV4-4'],
        ['IGHG1','IGHV4-55'],
        ['IGHG1','IGHV4-59'],
        ['IGHG1','IGHV4-61'],
        ['IGHG1','IGHV4OR15-8'],
        ['IGHG1','IGHV5-51'],
        ['IGHG1','IGHV6-1'],
        ['IGHG1','IGK@'],
        ['IGHG1','IGKC'],
        ['IGHG1','SFTPC'],
        ['IGHG2','IGHG3'],
        ['IGHG2','IGHG4'],
        ['IGHG2','IGHGP'],
        ['IGHG2','IGHV1-18'],
        ['IGHG2','IGHV1-2'],
        ['IGHG2','IGHV1-24'],
        ['IGHG2','IGHV1-3'],
        ['IGHG2','IGHV1-46'],
        ['IGHG2','IGHV1-58'],
        ['IGHG2','IGHV1-67'],
        ['IGHG2','IGHV1-69'],
        ['IGHG2','IGHV1OR15-1'],
        ['IGHG2','IGHV1OR16-1'],
        ['IGHG2','IGHV1OR16-3'],
        ['IGHG2','IGHV1OR21-1'],
        ['IGHG2','IGHV2-26'],
        ['IGHG2','IGHV2-5'],
        ['IGHG2','IGHV2-70'],
        ['IGHG2','IGHV2OR16-5'],
        ['IGHG2','IGHV3-11'],
        ['IGHG2','IGHV3-13'],
        ['IGHG2','IGHV3-15'],
        ['IGHG2','IGHV3-16'],
        ['IGHG2','IGHV3-20'],
        ['IGHG2','IGHV3-21'],
        ['IGHG2','IGHV3-22'],
        ['IGHG2','IGHV3-23'],
        ['IGHG2','IGHV3-30'],
        ['IGHG2','IGHV3-33'],
        ['IGHG2','IGHV3-41'],
        ['IGHG2','IGHV3-43'],
        ['IGHG2','IGHV3-48'],
        ['IGHG2','IGHV3-49'],
        ['IGHG2','IGHV3-53'],
        ['IGHG2','IGHV3-64'],
        ['IGHG2','IGHV3-65'],
        ['IGHG2','IGHV3-66'],
        ['IGHG2','IGHV3-7'],
        ['IGHG2','IGHV3-71'],
        ['IGHG2','IGHV3-72'],
        ['IGHG2','IGHV3-73'],
        ['IGHG2','IGHV3-74'],
        ['IGHG2','IGHV3OR15-7'],
        ['IGHG2','IGHV3OR16-13'],
        ['IGHG2','IGHV3OR16-15'],
        ['IGHG2','IGHV3OR16-9'],
        ['IGHG2','IGHV4-28'],
        ['IGHG2','IGHV4-31'],
        ['IGHG2','IGHV4-34'],
        ['IGHG2','IGHV4-39'],
        ['IGHG2','IGHV4-4'],
        ['IGHG2','IGHV4-55'],
        ['IGHG2','IGHV4-59'],
        ['IGHG2','IGHV4-61'],
        ['IGHG2','IGHV4OR15-8'],
        ['IGHG2','IGHV5-51'],
        ['IGHG2','IGHV6-1'],
        ['IGHG2','IGK@'],
        ['IGHG2','IGKC'],
        ['IGHG3','IGHG4'],
        ['IGHG3','IGHGP'],
        ['IGHG3','IGHV1-18'],
        ['IGHG3','IGHV1-2'],
        ['IGHG3','IGHV1-24'],
        ['IGHG3','IGHV1-3'],
        ['IGHG3','IGHV1-46'],
        ['IGHG3','IGHV1-58'],
        ['IGHG3','IGHV1-67'],
        ['IGHG3','IGHV1-69'],
        ['IGHG3','IGHV1OR15-1'],
        ['IGHG3','IGHV1OR16-1'],
        ['IGHG3','IGHV1OR16-3'],
        ['IGHG3','IGHV2-26'],
        ['IGHG3','IGHV2-5'],
        ['IGHG3','IGHV2-70'],
        ['IGHG3','IGHV3-11'],
        ['IGHG3','IGHV3-13'],
        ['IGHG3','IGHV3-15'],
        ['IGHG3','IGHV3-16'],
        ['IGHG3','IGHV3-20'],
        ['IGHG3','IGHV3-21'],
        ['IGHG3','IGHV3-22'],
        ['IGHG3','IGHV3-23'],
        ['IGHG3','IGHV3-30'],
        ['IGHG3','IGHV3-33'],
        ['IGHG3','IGHV3-41'],
        ['IGHG3','IGHV3-43'],
        ['IGHG3','IGHV3-48'],
        ['IGHG3','IGHV3-49'],
        ['IGHG3','IGHV3-53'],
        ['IGHG3','IGHV3-64'],
        ['IGHG3','IGHV3-65'],
        ['IGHG3','IGHV3-66'],
        ['IGHG3','IGHV3-7'],
        ['IGHG3','IGHV3-72'],
        ['IGHG3','IGHV3-73'],
        ['IGHG3','IGHV3-74'],
        ['IGHG3','IGHV3OR15-7'],
        ['IGHG3','IGHV3OR16-13'],
        ['IGHG3','IGHV3OR16-7'],
        ['IGHG3','IGHV3OR16-9'],
        ['IGHG3','IGHV4-28'],
        ['IGHG3','IGHV4-31'],
        ['IGHG3','IGHV4-34'],
        ['IGHG3','IGHV4-39'],
        ['IGHG3','IGHV4-4'],
        ['IGHG3','IGHV4-55'],
        ['IGHG3','IGHV4-59'],
        ['IGHG3','IGHV4-61'],
        ['IGHG3','IGHV4OR15-8'],
        ['IGHG3','IGHV5-51'],
        ['IGHG3','IGHV6-1'],
        ['IGHG4','IGHGP'],
        ['IGHG4','IGHV1-12'],
        ['IGHG4','IGHV1-18'],
        ['IGHG4','IGHV1-2'],
        ['IGHG4','IGHV1-24'],
        ['IGHG4','IGHV1-3'],
        ['IGHG4','IGHV1-46'],
        ['IGHG4','IGHV1-58'],
        ['IGHG4','IGHV1-69'],
        ['IGHG4','IGHV2-26'],
        ['IGHG4','IGHV2-5'],
        ['IGHG4','IGHV2-70'],
        ['IGHG4','IGHV3-11'],
        ['IGHG4','IGHV3-13'],
        ['IGHG4','IGHV3-15'],
        ['IGHG4','IGHV3-20'],
        ['IGHG4','IGHV3-21'],
        ['IGHG4','IGHV3-23'],
        ['IGHG4','IGHV3-30'],
        ['IGHG4','IGHV3-33'],
        ['IGHG4','IGHV3-43'],
        ['IGHG4','IGHV3-48'],
        ['IGHG4','IGHV3-49'],
        ['IGHG4','IGHV3-53'],
        ['IGHG4','IGHV3-64'],
        ['IGHG4','IGHV3-66'],
        ['IGHG4','IGHV3-7'],
        ['IGHG4','IGHV3-71'],
        ['IGHG4','IGHV3-72'],
        ['IGHG4','IGHV3-73'],
        ['IGHG4','IGHV3-74'],
        ['IGHG4','IGHV3OR16-9'],
        ['IGHG4','IGHV4-28'],
        ['IGHG4','IGHV4-31'],
        ['IGHG4','IGHV4-34'],
        ['IGHG4','IGHV4-39'],
        ['IGHG4','IGHV4-55'],
        ['IGHG4','IGHV4-59'],
        ['IGHG4','IGHV5-51'],
        ['IGHG4','IGHV6-1'],
        ['IGHG4','IGK@'],
        ['IGHG4','IGKC'],
        ['IGHGP','IGHV1-18'],
        ['IGHGP','IGHV1-2'],
        ['IGHGP','IGHV1-24'],
        ['IGHGP','IGHV1-3'],
        ['IGHGP','IGHV1-46'],
        ['IGHGP','IGHV1-67'],
        ['IGHGP','IGHV1-69'],
        ['IGHGP','IGHV1OR15-1'],
        ['IGHGP','IGHV1OR16-1'],
        ['IGHGP','IGHV2-26'],
        ['IGHGP','IGHV2-5'],
        ['IGHGP','IGHV2-70'],
        ['IGHGP','IGHV3-11'],
        ['IGHGP','IGHV3-13'],
        ['IGHGP','IGHV3-15'],
        ['IGHGP','IGHV3-20'],
        ['IGHGP','IGHV3-21'],
        ['IGHGP','IGHV3-23'],
        ['IGHGP','IGHV3-30'],
        ['IGHGP','IGHV3-33'],
        ['IGHGP','IGHV3-43'],
        ['IGHGP','IGHV3-48'],
        ['IGHGP','IGHV3-49'],
        ['IGHGP','IGHV3-53'],
        ['IGHGP','IGHV3-66'],
        ['IGHGP','IGHV3-7'],
        ['IGHGP','IGHV3-72'],
        ['IGHGP','IGHV3-73'],
        ['IGHGP','IGHV3-74'],
        ['IGHGP','IGHV3OR16-13'],
        ['IGHGP','IGHV3OR16-9'],
        ['IGHGP','IGHV4-28'],
        ['IGHGP','IGHV4-31'],
        ['IGHGP','IGHV4-34'],
        ['IGHGP','IGHV4-39'],
        ['IGHGP','IGHV4-4'],
        ['IGHGP','IGHV4-55'],
        ['IGHGP','IGHV4-59'],
        ['IGHGP','IGHV4-61'],
        ['IGHGP','IGHV4OR15-8'],
        ['IGHGP','IGHV5-51'],
        ['IGHGP','IGHV6-1'],
        ['IGHM','IGHV1-18'],
        ['IGHM','IGHV1-2'],
        ['IGHM','IGHV1-24'],
        ['IGHM','IGHV1-3'],
        ['IGHM','IGHV1-46'],
        ['IGHM','IGHV1-58'],
        ['IGHM','IGHV1-67'],
        ['IGHM','IGHV1-69'],
        ['IGHM','IGHV1OR16-1'],
        ['IGHM','IGHV2-26'],
        ['IGHM','IGHV2-5'],
        ['IGHM','IGHV2-70'],
        ['IGHM','IGHV2OR16-5'],
        ['IGHM','IGHV3-11'],
        ['IGHM','IGHV3-13'],
        ['IGHM','IGHV3-15'],
        ['IGHM','IGHV3-16'],
        ['IGHM','IGHV3-20'],
        ['IGHM','IGHV3-21'],
        ['IGHM','IGHV3-22'],
        ['IGHM','IGHV3-23'],
        ['IGHM','IGHV3-30'],
        ['IGHM','IGHV3-33'],
        ['IGHM','IGHV3-43'],
        ['IGHM','IGHV3-48'],
        ['IGHM','IGHV3-49'],
        ['IGHM','IGHV3-53'],
        ['IGHM','IGHV3-64'],
        ['IGHM','IGHV3-65'],
        ['IGHM','IGHV3-66'],
        ['IGHM','IGHV3-7'],
        ['IGHM','IGHV3-72'],
        ['IGHM','IGHV3-73'],
        ['IGHM','IGHV3-74'],
        ['IGHM','IGHV3OR16-13'],
        ['IGHM','IGHV3OR16-6'],
        ['IGHM','IGHV3OR16-7'],
        ['IGHM','IGHV3OR16-9'],
        ['IGHM','IGHV4-28'],
        ['IGHM','IGHV4-31'],
        ['IGHM','IGHV4-34'],
        ['IGHM','IGHV4-39'],
        ['IGHM','IGHV4-4'],
        ['IGHM','IGHV4-55'],
        ['IGHM','IGHV4-59'],
        ['IGHM','IGHV4-61'],
        ['IGHM','IGHV4OR15-8'],
        ['IGHM','IGHV5-51'],
        ['IGHV1-12','IGHV1-2'],
        ['IGHV1-12','IGHV1-46'],
        ['IGHV1-18','IGHV1-2'],
        ['IGHV1-18','IGHV1OR15-1'],
        ['IGHV1-18','IGHV1OR15-2'],
        ['IGHV1-18','IGHV1OR16-3'],
        ['IGHV1-18','IGHV1OR21-1'],
        ['IGHV1-18','IGHV7-27'],
        ['IGHV1-18','IGHV7-81'],
        ['IGHV1-2','IGHV1-3'],
        ['IGHV1-2','IGHV1-46'],
        ['IGHV1-2','IGHV1OR15-1'],
        ['IGHV1-2','IGHV1OR15-3'],
        ['IGHV1-2','IGHV1OR16-4'],
        ['IGHV1-3','IGHV1-46'],
        ['IGHV1-3','IGHV1OR15-1'],
        ['IGHV1-3','IGHV1OR15-3'],
        ['IGHV1-3','IGHV1OR16-3'],
        ['IGHV1-3','IGHV1OR21-1'],
        ['IGHV1-46','IGHV1-67'],
        ['IGHV1-46','IGHV1-69'],
        ['IGHV1-46','IGHV1OR15-1'],
        ['IGHV1-46','IGHV1OR16-4'],
        ['IGHV1OR15-1','IGHV7-81'],
        ['IGHV2-26','IGHV2OR16-5'],
        ['IGHV2-5','IGHV2-70'],
        ['IGHV2-5','IGHV2OR16-5'],
        ['IGHV3-11','IGHV3-30'],
        ['IGHV3-11','IGHV3-33'],
        ['IGHV3-11','IGHV3-48'],
        ['IGHV3-11','IGHV3-7'],
        ['IGHV3-11','IGHV3-72'],
        ['IGHV3-11','IGHV3OR16-9'],
        ['IGHV3-13','IGHV3-21'],
        ['IGHV3-13','IGHV3-48'],
        ['IGHV3-13','IGHV3-7'],
        ['IGHV3-15','IGHV3-16'],
        ['IGHV3-15','IGHV3-20'],
        ['IGHV3-15','IGHV3-22'],
        ['IGHV3-15','IGHV3-23'],
        ['IGHV3-15','IGHV3-35'],
        ['IGHV3-15','IGHV3-38'],
        ['IGHV3-15','IGHV3-43'],
        ['IGHV3-15','IGHV3-53'],
        ['IGHV3-15','IGHV3-72'],
        ['IGHV3-15','IGHV3-74'],
        ['IGHV3-15','IGHV3OR15-7'],
        ['IGHV3-15','IGHV3OR16-13'],
        ['IGHV3-15','IGHV3OR16-7'],
        ['IGHV3-16','IGHV3-20'],
        ['IGHV3-16','IGHV3-21'],
        ['IGHV3-16','IGHV3-43'],
        ['IGHV3-16','IGHV3-48'],
        ['IGHV3-16','IGHV3-7'],
        ['IGHV3-19','IGHV3-23'],
        ['IGHV3-20','IGHV3-22'],
        ['IGHV3-20','IGHV3-23'],
        ['IGHV3-20','IGHV3-35'],
        ['IGHV3-20','IGHV3-38'],
        ['IGHV3-20','IGHV3-41'],
        ['IGHV3-20','IGHV3-43'],
        ['IGHV3-20','IGHV3-49'],
        ['IGHV3-20','IGHV3-53'],
        ['IGHV3-20','IGHV3-7'],
        ['IGHV3-20','IGHV3-71'],
        ['IGHV3-20','IGHV3-73'],
        ['IGHV3-20','IGHV3-74'],
        ['IGHV3-20','IGHV3OR15-7'],
        ['IGHV3-21','IGHV3-22'],
        ['IGHV3-21','IGHV3-23'],
        ['IGHV3-21','IGHV3-48'],
        ['IGHV3-21','IGHV3-65'],
        ['IGHV3-21','IGHV3-7'],
        ['IGHV3-21','IGHV3-71'],
        ['IGHV3-22','IGHV3-43'],
        ['IGHV3-22','IGHV3-48'],
        ['IGHV3-22','IGHV3-7'],
        ['IGHV3-23','IGHV3-41'],
        ['IGHV3-23','IGHV3-43'],
        ['IGHV3-23','IGHV3-48'],
        ['IGHV3-23','IGHV3-49'],
        ['IGHV3-23','IGHV3-53'],
        ['IGHV3-23','IGHV3-66'],
        ['IGHV3-23','IGHV3-7'],
        ['IGHV3-23','IGHV3-71'],
        ['IGHV3-23','IGHV3-73'],
        ['IGHV3-23','IGHV3OR15-7'],
        ['IGHV3-23','IGHV3OR16-13'],
        ['IGHV3-23','IGHV3OR16-15'],
        ['IGHV3-23','IGHV3OR16-16'],
        ['IGHV3-23','IGHV3OR16-9'],
        ['IGHV3-30','IGHV3-48'],
        ['IGHV3-30','IGHV3-53'],
        ['IGHV3-30','IGHV3-64'],
        ['IGHV3-30','IGHV3-66'],
        ['IGHV3-30','IGHV3OR16-9'],
        ['IGHV3-33','IGHV3-48'],
        ['IGHV3-33','IGHV3-53'],
        ['IGHV3-33','IGHV3-64'],
        ['IGHV3-33','IGHV3-66'],
        ['IGHV3-33','IGHV3-74'],
        ['IGHV3-33','IGHV3OR16-9'],
        ['IGHV3-35','IGHV3-43'],
        ['IGHV3-35','IGHV3-53'],
        ['IGHV3-38','IGHV3-43'],
        ['IGHV3-38','IGHV3-53'],
        ['IGHV3-38','IGHV3-74'],
        ['IGHV3-41','IGHV3-43'],
        ['IGHV3-41','IGHV3-66'],
        ['IGHV3-41','IGHV3-73'],
        ['IGHV3-43','IGHV3-53'],
        ['IGHV3-43','IGHV3-66'],
        ['IGHV3-43','IGHV3-73'],
        ['IGHV3-43','IGHV3-74'],
        ['IGHV3-43','IGHV3OR15-7'],
        ['IGHV3-43','IGHV3OR16-13'],
        ['IGHV3-48','IGHV3-65'],
        ['IGHV3-48','IGHV3-7'],
        ['IGHV3-48','IGHV3-72'],
        ['IGHV3-48','IGHV3OR16-13'],
        ['IGHV3-48','IGHV3OR16-8'],
        ['IGHV3-48','IGHV3OR16-9'],
        ['IGHV3-49','IGHV3-52'],
        ['IGHV3-49','IGHV3-66'],
        ['IGHV3-49','IGHV3-7'],
        ['IGHV3-49','IGHV3-71'],
        ['IGHV3-49','IGHV3-74'],
        ['IGHV3-49','IGHV3OR15-7'],
        ['IGHV3-49','IGHV3OR16-13'],
        ['IGHV3-52','IGHV3-74'],
        ['IGHV3-53','IGHV3-66'],
        ['IGHV3-53','IGHV3-73'],
        ['IGHV3-53','IGHV3-74'],
        ['IGHV3-53','IGHV3OR15-7'],
        ['IGHV3-53','IGHV3OR16-13'],
        ['IGHV3-62','IGHV3-7'],
        ['IGHV3-64','IGHV3-7'],
        ['IGHV3-64','IGHV3-71'],
        ['IGHV3-64','IGHV3-73'],
        ['IGHV3-65','IGHV3-7'],
        ['IGHV3-66','IGHV3-7'],
        ['IGHV3-66','IGHV3-71'],
        ['IGHV3-66','IGHV3-73'],
        ['IGHV3-66','IGHV3-74'],
        ['IGHV3-66','IGHV3OR15-7'],
        ['IGHV3-66','IGHV3OR16-15'],
        ['IGHV3-66','IGHV3OR16-16'],
        ['IGHV3-7','IGHV3-71'],
        ['IGHV3-7','IGHV3-72'],
        ['IGHV3-7','IGHV3-73'],
        ['IGHV3-7','IGHV3OR15-7'],
        ['IGHV3-7','IGHV3OR16-13'],
        ['IGHV3-7','IGHV3OR16-15'],
        ['IGHV3-7','IGHV3OR16-9'],
        ['IGHV3-71','IGHV3OR15-7'],
        ['IGHV3-73','IGHV3OR15-7'],
        ['IGHV3-73','IGHV3OR16-13'],
        ['IGHV3-74','IGHV3OR15-7'],
        ['IGHV3-74','IGHV3OR16-13'],
        ['IGHV4-28','IGHV4-34'],
        ['IGHV4-28','IGHV4-39'],
        ['IGHV4-28','IGHV4-4'],
        ['IGHV4-28','IGHV4-55'],
        ['IGHV4-28','IGHV4-59'],
        ['IGHV4-28','IGHV4-61'],
        ['IGHV4-28','IGHV4OR15-8'],
        ['IGHV4-31','IGHV4-34'],
        ['IGHV4-31','IGHV4-39'],
        ['IGHV4-31','IGHV4-4'],
        ['IGHV4-31','IGHV4-55'],
        ['IGHV4-31','IGHV4-59'],
        ['IGHV4-31','IGHV4-61'],
        ['IGHV4-31','IGHV4OR15-8'],
        ['IGHV4-34','IGHV4-39'],
        ['IGHV4-34','IGHV4-4'],
        ['IGHV4-34','IGHV4-55'],
        ['IGHV4-34','IGHV4-59'],
        ['IGHV4-34','IGHV4-61'],
        ['IGHV4-34','IGHV4OR15-8'],
        ['IGHV4-39','IGHV4-4'],
        ['IGHV4-39','IGHV4-55'],
        ['IGHV4-39','IGHV4-59'],
        ['IGHV4-39','IGHV4-61'],
        ['IGHV4-39','IGHV4OR15-8'],
        ['IGHV4-4','IGHV4-55'],
        ['IGHV4-4','IGHV4-59'],
        ['IGHV4-55','IGHV4-59'],
        ['IGHV4-55','IGHV4-61'],
        ['IGHV4-55','IGHV4OR15-8'],
        ['IGHV4-59','IGHV4OR15-8'],
        ['IGK@','IGKC'],
        ['IGK@','IGKV1-27'],
        ['IGK@','IGKV1-8'],
        ['IGK@','IGKV1D-43'],
        ['IGK@','IGKV1D-8'],
        ['IGK@','IGKV2-24'],
        ['IGK@','IGKV2D-24'],
        ['IGK@','IGKV2D-28'],
        ['IGK@','IGKV2D-29'],
        ['IGK@','IGKV2D-30'],
        ['IGK@','IGKV3-11'],
        ['IGK@','IGKV3-20'],
        ['IGK@','IGKV3-7'],
        ['IGK@','IGKV3D-11'],
        ['IGK@','IGKV3D-15'],
        ['IGK@','IGKV3D-20'],
        ['IGK@','IGKV6-21'],
        ['IGK@','MUC5AC'],
        ['IGKC','IGKV1-12'],
        ['IGKC','IGKV1-16'],
        ['IGKC','IGKV1-17'],
        ['IGKC','IGKV1-27'],
        ['IGKC','IGKV1-33'],
        ['IGKC','IGKV1-37'],
        ['IGKC','IGKV1-39'],
        ['IGKC','IGKV1-5'],
        ['IGKC','IGKV1-6'],
        ['IGKC','IGKV1-8'],
        ['IGKC','IGKV1-9'],
        ['IGKC','IGKV1D-12'],
        ['IGKC','IGKV1D-13'],
        ['IGKC','IGKV1D-16'],
        ['IGKC','IGKV1D-17'],
        ['IGKC','IGKV1D-27'],
        ['IGKC','IGKV1D-33'],
        ['IGKC','IGKV1D-37'],
        ['IGKC','IGKV1D-39'],
        ['IGKC','IGKV1D-42'],
        ['IGKC','IGKV1D-43'],
        ['IGKC','IGKV1D-8'],
        ['IGKC','IGKV1OR-2'],
        ['IGKC','IGKV1OR-3'],
        ['IGKC','IGKV1OR1-1'],
        ['IGKC','IGKV1OR10-1'],
        ['IGKC','IGKV1OR2-1'],
        ['IGKC','IGKV1OR2-108'],
        ['IGKC','IGKV1OR2-118'],
        ['IGKC','IGKV1OR2-2'],
        ['IGKC','IGKV1OR22-1'],
        ['IGKC','IGKV1OR22-5'],
        ['IGKC','IGKV1OR9-2'],
        ['IGKC','IGKV2-24'],
        ['IGKC','IGKV2-26'],
        ['IGKC','IGKV2-28'],
        ['IGKC','IGKV2-29'],
        ['IGKC','IGKV2-30'],
        ['IGKC','IGKV2-40'],
        ['IGKC','IGKV2D-24'],
        ['IGKC','IGKV2D-26'],
        ['IGKC','IGKV2D-28'],
        ['IGKC','IGKV2D-29'],
        ['IGKC','IGKV2D-30'],
        ['IGKC','IGKV2D-40'],
        ['IGKC','IGKV2OR22-4'],
        ['IGKC','IGKV3-11'],
        ['IGKC','IGKV3-15'],
        ['IGKC','IGKV3-20'],
        ['IGKC','IGKV3-7'],
        ['IGKC','IGKV3D-11'],
        ['IGKC','IGKV3D-15'],
        ['IGKC','IGKV3D-20'],
        ['IGKC','IGKV3D-7'],
        ['IGKC','IGKV3OR2-268'],
        ['IGKC','IGKV4-1'],
        ['IGKC','IGKV6-21'],
        ['IGKC','IGKV6D-21'],
        ['IGKC','IGKV6D-41'],
        ['IGKC','MUC5AC'],
        ['IGKC','SFTPA2'],
        ['IGKV1-12','IGKV1-8'],
        ['IGKV1-12','IGKV1D-8'],
        ['IGKV1-13','IGKV1-8'],
        ['IGKV1-13','IGKV1D-8'],
        ['IGKV1-16','IGKV1-27'],
        ['IGKV1-16','IGKV1-8'],
        ['IGKV1-17','IGKV1-27'],
        ['IGKV1-17','IGKV1-6'],
        ['IGKV1-17','IGKV1-8'],
        ['IGKV1-17','IGKV1D-8'],
        ['IGKV1-27','IGKV1-33'],
        ['IGKV1-27','IGKV1-39'],
        ['IGKV1-27','IGKV1-5'],
        ['IGKV1-27','IGKV1-6'],
        ['IGKV1-27','IGKV1-8'],
        ['IGKV1-27','IGKV1-9'],
        ['IGKV1-27','IGKV1D-33'],
        ['IGKV1-27','IGKV1D-39'],
        ['IGKV1-27','IGKV1D-8'],
        ['IGKV1-27','IGKV1OR2-108'],
        ['IGKV1-33','IGKV1-8'],
        ['IGKV1-33','IGKV1D-8'],
        ['IGKV1-37','IGKV1-8'],
        ['IGKV1-37','IGKV1D-8'],
        ['IGKV1-39','IGKV1-8'],
        ['IGKV1-39','IGKV1D-8'],
        ['IGKV1-5','IGKV1-8'],
        ['IGKV1-5','IGKV1-9'],
        ['IGKV1-5','IGKV1D-43'],
        ['IGKV1-5','IGKV1D-8'],
        ['IGKV1-6','IGKV1-8'],
        ['IGKV1-6','IGKV1D-43'],
        ['IGKV1-8','IGKV1-9'],
        ['IGKV1-8','IGKV1D-12'],
        ['IGKV1-8','IGKV1D-13'],
        ['IGKV1-8','IGKV1D-16'],
        ['IGKV1-8','IGKV1D-27'],
        ['IGKV1-8','IGKV1D-33'],
        ['IGKV1-8','IGKV1D-37'],
        ['IGKV1-8','IGKV1D-39'],
        ['IGKV1-8','IGKV1D-8'],
        ['IGKV1-8','IGKV1OR1-1'],
        ['IGKV1-8','IGKV1OR10-1'],
        ['IGKV1-8','IGKV1OR2-118'],
        ['IGKV1-8','IGKV1OR22-5'],
        ['IGKV1-9','IGKV1D-43'],
        ['IGKV1-9','IGKV1D-8'],
        ['IGKV1D-12','IGKV1D-8'],
        ['IGKV1D-13','IGKV1D-8'],
        ['IGKV1D-16','IGKV1D-8'],
        ['IGKV1D-27','IGKV1D-8'],
        ['IGKV1D-33','IGKV1D-8'],
        ['IGKV1D-37','IGKV1D-8'],
        ['IGKV1D-39','IGKV1D-8'],
        ['IGKV1D-8','IGKV1OR1-1'],
        ['IGKV1D-8','IGKV1OR10-1'],
        ['IGKV1D-8','IGKV1OR2-108'],
        ['IGKV1D-8','IGKV1OR2-118'],
        ['IGKV1D-8','IGKV1OR22-5'],
        ['IGKV2-24','IGKV2-30'],
        ['IGKV2-24','IGKV2-40'],
        ['IGKV2-24','IGKV2D-24'],
        ['IGKV2-24','IGKV2D-30'],
        ['IGKV2-24','IGKV2D-40'],
        ['IGKV2-29','IGKV2D-29'],
        ['IGKV2-4','IGKV2D-28'],
        ['IGKV2-40','IGKV2D-24'],
        ['IGKV2-40','IGKV2D-28'],
        ['IGKV2-40','IGKV2D-30'],
        ['IGKV2D-24','IGKV2D-40'],
        ['IGKV2D-28','IGKV2D-40'],
        ['IGKV2D-28','IGKV2OR22-4'],
        ['IGKV2D-30','IGKV2D-40'],
        ['IGKV3-11','IGKV3-15'],
        ['IGKV3-11','IGKV3-20'],
        ['IGKV3-11','IGKV3-7'],
        ['IGKV3-11','IGKV3D-11'],
        ['IGKV3-11','IGKV3D-15'],
        ['IGKV3-11','IGKV3D-20'],
        ['IGKV3-11','IGKV3D-7'],
        ['IGKV3-11','IGKV3OR2-268'],
        ['IGKV3-15','IGKV3-20'],
        ['IGKV3-15','IGKV3-7'],
        ['IGKV3-15','IGKV3D-11'],
        ['IGKV3-15','IGKV3D-20'],
        ['IGKV3-20','IGKV3-7'],
        ['IGKV3-20','IGKV3D-11'],
        ['IGKV3-20','IGKV3D-15'],
        ['IGKV3-20','IGKV3D-20'],
        ['IGKV3-20','IGKV3D-7'],
        ['IGKV3-20','IGKV3OR2-268'],
        ['IGKV3-7','IGKV3D-11'],
        ['IGKV3-7','IGKV3D-15'],
        ['IGKV3-7','IGKV3D-20'],
        ['IGKV3D-11','IGKV3D-15'],
        ['IGKV3D-11','IGKV3D-20'],
        ['IGKV3D-11','IGKV3D-7'],
        ['IGKV3D-11','IGKV3OR2-268'],
        ['IGKV3D-15','IGKV3D-20'],
        ['IGKV3D-15','IGKV3D-7'],
        ['IGKV3D-15','IGKV3OR2-268'],
        ['IGKV3D-20','IGKV3D-7'],
        ['IGKV3D-20','IGKV3OR2-268'],
        ['IGLC1','IGLC2'],
        ['IGLC1','IGLC3'],
        ['IGLC1','IGLC7'],
        ['IGLC1','IGLV1-36'],
        ['IGLC1','IGLV1-40'],
        ['IGLC1','IGLV1-41'],
        ['IGLC1','IGLV1-44'],
        ['IGLC1','IGLV1-47'],
        ['IGLC1','IGLV1-50'],
        ['IGLC1','IGLV1-51'],
        ['IGLC1','IGLV10-54'],
        ['IGLC1','IGLV2-11'],
        ['IGLC1','IGLV2-14'],
        ['IGLC1','IGLV2-18'],
        ['IGLC1','IGLV2-23'],
        ['IGLC1','IGLV2-34'],
        ['IGLC1','IGLV2-8'],
        ['IGLC1','IGLV3-1'],
        ['IGLC1','IGLV3-10'],
        ['IGLC1','IGLV3-19'],
        ['IGLC1','IGLV3-21'],
        ['IGLC1','IGLV3-25'],
        ['IGLC1','IGLV3-27'],
        ['IGLC1','IGLV3-9'],
        ['IGLC1','IGLV4-60'],
        ['IGLC1','IGLV4-69'],
        ['IGLC1','IGLV5-45'],
        ['IGLC1','IGLV6-57'],
        ['IGLC1','IGLV7-43'],
        ['IGLC1','IGLV7-46'],
        ['IGLC1','IGLV8-61'],
        ['IGLC1','IGLV9-49'],
        ['IGLC2','IGLC3'],
        ['IGLC2','IGLC6'],
        ['IGLC2','IGLL5'],
        ['IGLC2','IGLV1-40'],
        ['IGLC2','IGLV1-47'],
        ['IGLC2','IGLV1-51'],
        ['IGLC2','IGLV10-54'],
        ['IGLC2','IGLV2-11'],
        ['IGLC2','IGLV2-14'],
        ['IGLC2','IGLV2-23'],
        ['IGLC2','IGLV2-8'],
        ['IGLC2','IGLV3-1'],
        ['IGLC2','IGLV3-10'],
        ['IGLC2','IGLV3-19'],
        ['IGLC2','IGLV3-21'],
        ['IGLC2','IGLV3-25'],
        ['IGLC2','IGLV3-27'],
        ['IGLC2','IGLV4-60'],
        ['IGLC2','IGLV4-69'],
        ['IGLC2','IGLV5-45'],
        ['IGLC2','IGLV6-57'],
        ['IGLC2','IGLV7-43'],
        ['IGLC2','IGLV7-46'],
        ['IGLC2','IGLV8-61'],
        ['IGLC2','IGLV9-49'],
        ['IGLC3','IGLL5'],
        ['IGLC5','IGLL5'],
        ['IGLC7','IGLL5'],
        ['IGLC7','IGLV1-47'],
        ['IGLC7','IGLV3-21'],
        ['IGLC7','IGLV6-57'],
        ['IGLL5','IGLV1-36'],
        ['IGLL5','IGLV1-40'],
        ['IGLL5','IGLV1-41'],
        ['IGLL5','IGLV1-44'],
        ['IGLL5','IGLV1-47'],
        ['IGLL5','IGLV1-50'],
        ['IGLL5','IGLV1-51'],
        ['IGLL5','IGLV10-54'],
        ['IGLL5','IGLV2-11'],
        ['IGLL5','IGLV2-14'],
        ['IGLL5','IGLV2-18'],
        ['IGLL5','IGLV2-23'],
        ['IGLL5','IGLV2-34'],
        ['IGLL5','IGLV2-8'],
        ['IGLL5','IGLV3-1'],
        ['IGLL5','IGLV3-10'],
        ['IGLL5','IGLV3-19'],
        ['IGLL5','IGLV3-21'],
        ['IGLL5','IGLV3-25'],
        ['IGLL5','IGLV3-27'],
        ['IGLL5','IGLV3-9'],
        ['IGLL5','IGLV4-60'],
        ['IGLL5','IGLV4-69'],
        ['IGLL5','IGLV5-37'],
        ['IGLL5','IGLV5-45'],
        ['IGLL5','IGLV6-57'],
        ['IGLL5','IGLV7-43'],
        ['IGLL5','IGLV7-46'],
        ['IGLL5','IGLV8-61'],
        ['IGLL5','IGLV9-49'],
        ['IGLV1-41','IGLV1-51'],
        ['IGLV2-11','IGLV2-14'],
        ['IGLV2-11','IGLV2-18'],
        ['IGLV2-11','IGLV2-23'],
        ['IGLV2-11','IGLV2-8'],
        ['IGLV2-14','IGLV2-18'],
        ['IGLV2-14','IGLV2-23'],
        ['IGLV2-23','IGLV2-8'],
        ['IGLV3-10','IGLV3-25'],
        ['IGLV3-16','IGLV3-25'],
        ['IGLV3-21','IGLV3-9'],
        ['IGLV7-43','IGLV7-46'],
        ['IGSF5','PCP4'],
        ['IL4I1','NUP62'],
        ['INE1','UBA1'],
        ['IQCF1','IQCF5'],
        ['ITIH4','MUSTN1'],
        ['ITIH4','SERPINA1'],
        ['ITIH4','TMEM110-MUSTN1'],
        ['IYD','TG'],
        ['JUN','TG'],
        ['KLRC2','KLRC3'],
        ['KRT1','KRT10'],
        ['KRT1','KRT14'],
        ['KRT1','KRT2'],
        ['KRT1','KRT5'],
        ['KRT1','NEAT1'],
        ['KRT1','PKP1'],
        ['KRT10','KRT14'],
        ['KRT10','KRT2'],
        ['KRT10','KRT5'],
        ['KRT10','NEAT1'],
        ['KRT14','KRT2'],
        ['KRT14','KRT5'],
        ['KRT14','NEAT1'],
        ['KRT2','KRT5'],
        ['KRT4','TGM3'],
        ['KRT5','KRT6A'],
        ['KRT6A','KRT6C'],
        ['LAIR1','LAIR2'],
        ['LAMTOR5-AS1','PROK1'],
        ['LCAT','SLC12A4'],
        ['LCE1D','LCE1E'],
        ['LGALS9B','LGALS9C'],
        ['LILRA1','LILRA2'],
        ['LILRA1','LILRB1'],
        ['LILRA1','LILRB2'],
        ['LILRA2','LILRB1'],
        ['LILRA4','LILRA6'],
        ['LILRA4','LILRB3'],
        ['LILRA6','LILRB2'],
        ['LILRA6','LILRB3'],
        ['LILRB1','LILRB2'],
        ['LILRB1','LILRB3'],
        ['LILRB1','LILRB4'],
        ['LILRB1','LILRP1'],
        ['LILRB2','LILRB3'],
        ['LILRB2','LILRB5'],
        ['LILRB3','LILRB4'],
        ['LILRB3','LILRB5'],
        ['LILRB4','LILRB5'],
        ['LIME1','SLC2A4RG'],
        ['LINC00643','SYT16'],
        ['LINC00665','LINC01535'],
        ['LINC00969','SDHA'],
        ['LINC01089','RHOF'],
        ['LINC01251','UBE2R2-AS1'],
        ['LINC01513','ROPN1L'],
        ['LPA','PLG'],
        ['LPO','PRH2'],
        ['LRRC37A15P','MANBA'],
        ['LRRC37A17P','LRRC37A4P'],
        ['LRRC69','OTUD6B'],
        ['LTB4R2','NOP9'],
        ['LYZ','MRPS30'],
        ['LYZ','MYADM'],
        ['LYZ','UBE2G2'],
        ['LYZ','ZG16B'],
        ['MALAT1','TG'],
        ['MAMDC4','PHPT1'],
        ['MAP1LC3B','MAP1LC3B2'],
        ['MAT2A','TG'],
        ['MB','MYH7'],
        ['MDC1','NRM'],
        ['METTL2A','METTL2B'],
        ['MICA','MICB'],
        ['MICAL2','MICALCL'],
        ['MKRN1','MKRN4P'],
        ['MROH7','MROH7-TTC4'],
        ['MRPS6','SLC5A3'],
        ['MSANTD3','MSANTD3-TMEFF1'],
        ['MSANTD3','TMEFF1'],
        ['MSTO1','MSTO2P'],
        ['MT1E','MT1JP'],
        ['MT1G','MT2A'],
        ['MTCH1','TG'],
        ['MUC7','PIGR'],
        ['MVP','PAGR1'],
        ['MXRA5','MXRA5Y'],
        ['MYH11','MYH9'],
        ['MYH11','SMR3B'],
        ['MYH6','MYH7'],
        ['MYH7','MYL2'],
        ['MYH7','TNNI3'],
        ['MYH7','TNNT2'],
        ['MYH7','TPM1'],
        ['MYH9','TG'],
        ['MYL12A','MYL12B'],
        ['MYL6','MYL6B'],
        ['MZT2A','MZT2B'],
        ['NACA','NACA3P'],
        ['NACA','PRIM1'],
        ['NAT8','NAT8B'],
        ['NBEAL1','RPL12'],
        ['NBPF1','NBPF25P'],
        ['NBPF1','NBPF3'],
        ['NBPF10','NBPF8'],
        ['NBPF11','NBPF12'],
        ['NBPF2P','NBPF3'],
        ['NBR1','NBR2'],
        ['NCF1B','NCF1C'],
        ['NEAT1','TG'],
        ['NGRN','PERP'],
        ['NKX2-1','SFTA3'],
        ['NOTCH2','NOTCH2NL'],
        ['NPIPA1','NPIPA5'],
        ['NPIPA2','NPIPP1'],
        ['NPIPA3','NPIPA5'],
        ['NPIPA3','NPIPP1'],
        ['NPIPA7','PKD1'],
        ['NPM2','XPO7'],
        ['NSFL1C','SIRPB2'],
        ['NT5C1B-RDH14','RDH14'],
        ['NUDT10','NUDT11'],
        ['OBP2A','OBP2B'],
        ['ODF3B','SCO2'],
        ['OR2W3','TRIM58'],
        ['ORM1','ORM2'],
        ['ORM1','SERPINA1'],
        ['OVGP1','WDR77'],
        ['P2RY11','PPAN'],
        ['PABPC1','PABPC3'],
        ['PAFAH1B2','YIPF5'],
        ['PAGR1','PRRT2'],
        ['PAQR6','SMG5'],
        ['PAX8','TG'],
        ['PCBP2','PCBP2-OT1'],
        ['PCDHA10','PCDHA11'],
        ['PDCD7','UBAP1L'],
        ['PDIA4','TG'],
        ['PEX11A','PLIN1'],
        ['PHB2','SCARNA12'],
        ['PILRA','PILRB'],
        ['PILRA','STAG3L5P-PVRIG2P-PILRB'],
        ['PIP5K1A','PIPSL'],
        ['PLPP3','TG'],
        ['PMS2','PMS2CL'],
        ['PNLIP','SLC4A4'],
        ['PNLIPRP2','PRSS1'],
        ['PNLIPRP2','PRSS2'],
        ['POM121','POM121C'],
        ['POMZP3','ZP3'],
        ['PPP1R2','PPP1R2P3'],
        ['PRB1','PRB2'],
        ['PRB3','PRB4'],
        ['PRH1','PRH2'],
        ['PRKAA1','TTC33'],
        ['PRSS1','RNASE1'],
        ['PRSS2','RNASE1'],
        ['PSAP','TG'],
        ['PTENP1-AS','TRBV26OR9-2'],
        ['PVRIG','STAG3'],
        ['RABL2A','RABL2B'],
        ['RBM14-RBM4','RBM4'],
        ['RBMX','RBMXL1'],
        ['RHOT2','WDR90'],
        ['RPL10','RPL10P15'],
        ['RPL13A','RPL13AP7'],
        ['RPL18','RPL18P13'],
        ['RPL23AP7','RPL23AP82'],
        ['RPL24','RPL24P2'],
        ['RPL28','TMEM190'],
        ['RPL3','RPL3P7'],
        ['RPL31','RPL31P63'],
        ['RPL4','RPL4P4'],
        ['RPL5','RPL5P3'],
        ['RPL7A','RPL7AP30'],
        ['RPL7L1','WAC-AS1'],
        ['RPL8','TG'],
        ['RPL9','TIPIN'],
        ['RPLP0','RPLP0P6'],
        ['RPS20','RPS20P10'],
        ['RPS20','RPS20P24'],
        ['RPS24','RPS24P8'],
        ['RPS4X','RPS4XP13'],
        ['RPS4X','RPS4XP14'],
        ['RPS4X','RPS4XP16'],
        ['RPS4X','RPS4XP22'],
        ['RPS4X','RPS4XP6'],
        ['RPS6','TG'],
        ['RPSA','RPSAP53'],
        ['RRP7A','RRP7BP'],
        ['SAA1','SAA2'],
        ['SAA1','SAA2-SAA4'],
        ['SCARNA13','SNHG10'],
        ['SCO2','TYMP'],
        ['SEC14L1','SNHG20'],
        ['SERPINA1','SERPINA3'],
        ['SERPINB3','SERPINB4'],
        ['SERPINF2','WDR81'],
        ['SFTPA1','SFTPA2'],
        ['SFTPA1','SFTPB'],
        ['SFTPA1','SFTPC'],
        ['SFTPA2','SFTPB'],
        ['SFTPA2','SFTPC'],
        ['SFTPA2','SLC34A2'],
        ['SFTPB','SFTPC'],
        ['SFTPC','UBC'],
        ['SIDT2','TAGLN'],
        ['SIGLEC10','SIGLEC16'],
        ['SIGLEC11','SIGLEC16'],
        ['SIRPA','SIRPB1'],
        ['SIRPB1','SIRPD'],
        ['SLC2A14','SLC2A3'],
        ['SLC35E2','SLC35E2B'],
        ['SLFN11','SLFN13'],
        ['SLFN12L','SLFN13'],
        ['SORD','SORD2P'],
        ['SP140','SP140L'],
        ['SRRM2','TG'],
        ['ST6GALNAC4','ST6GALNAC6'],
        ['STK19','STK19B'],
        ['STYXL1','TMEM120A'],
        ['SULT1A1','SULT1A2'],
        ['TAOK2','TMEM219'],
        ['TBCEL','TECTA'],
        ['TCEAL2','TCEAL4'],
        ['TCEAL3','TCEAL6'],
        ['TG','TIMP3'],
        ['TG','TMBIM6'],
        ['TG','TPO'],
        ['TG','TPT1'],
        ['TG','UBC'],
        ['TG','VEGFA'],
        ['TG','VIM'],
        ['TIMM23','TIMM23B'],
        ['TM7SF2','VPS51'],
        ['TPM1','TPM2'],
        ['TPSAB1','TPSB2'],
        ['TPSAB1','TPSD1'],
        ['TPSB2','TPSD1'],
        ['TRAC','TRAV1-1'],
        ['TRAC','TRAV1-2'],
        ['TRAC','TRAV10'],
        ['TRAC','TRAV12-1'],
        ['TRAC','TRAV12-2'],
        ['TRAC','TRAV12-3'],
        ['TRAC','TRAV13-1'],
        ['TRAC','TRAV13-2'],
        ['TRAC','TRAV14DV4'],
        ['TRAC','TRAV17'],
        ['TRAC','TRAV19'],
        ['TRAC','TRAV2'],
        ['TRAC','TRAV20'],
        ['TRAC','TRAV21'],
        ['TRAC','TRAV3'],
        ['TRAC','TRAV4'],
        ['TRAC','TRAV5'],
        ['TRAC','TRAV6'],
        ['TRAC','TRAV8-1'],
        ['TRAC','TRAV8-3'],
        ['TRAC','TRAV8-4'],
        ['TRAC','TRAV8-6'],
        ['TRAC','TRAV9-2'],
        ['TRBC2','TRBV28'],
        ['TRBC2','TRBV29-1'],
        ['TRIM16','TRIM16L'],
        ['TTTY15','USP9Y'],
        ['TUBA1A','TUBA1B'],
        ['TUBA1A','TUBA1C'],
        ['TUBA1B','TUBA1C'],
        ['TUBA3C','TUBA3E'],
        ['TUBA3D','TUBA3E'],
        ['TUBA4A','TUBA4B'],
        ['TUBG1','TUBG2'],
        ['TVP23C','TVP23C-CDRT4'],
        ['UBA52','UBC'],
        ['UBB','UBC'],
        ['UGT2B10','UGT2B7'],
        ['UGT2B11','UGT2B7'],
        ['UGT2B15','UGT2B17'],
        ['UGT2B28','UGT2B7'],
        ['UPP2','UPP2-IT1'],
        ['UQCRH','UQCRHL'],
        ['VAMP5','VAMP8'],
        ['WASHC2A','WASHC2C'],
        ['ZDHHC11','ZDHHC11B'],
        ['ZFP2','ZNF354B'],
        ['ZNF137P','ZNF701'],
        ['ZNF181','ZNF302'],
        ['ZNF28','ZNF468'],
        ['ZNF32-AS2','ZNF32-AS3'],
        ['ZNF33A','ZNF33B'],
        ['ZNF37A','ZNF37BP'],
        ['ZNF417','ZNF587'],
        ['ZNF510','ZNF782'],
        ['ZNF577','ZNF649'],
        ['ZNF607','ZNF781'],
        ['ZNF761','ZNF765']

        ]

    # known fusions to exist in healthy patients: TTTY15-USP9Y, SLC45A3-ELK4, MSMB-NCOA4

    data = fusions.get(options.organism.lower(),[])
    if data:

        #file_symbols = os.path.join(options.output_directory,'genes_symbols.txt')
        file_symbols = os.path.join(options.output_directory,'synonyms.txt')
        loci = symbols.generate_loci(file_symbols)

        genes = symbols.read_genes_symbols(file_symbols)

        d = []
        for (g1,g2) in data:
            if g1.upper() != g2.upper():
                ens1 = symbols.ensembl(g1.upper(),genes,loci)
                ens2 = symbols.ensembl(g2.upper(),genes,loci)
                if ens1 and ens2:
                    for e1 in ens1:
                        for e2 in ens2:
                            if e1 != e2:
                                d.append([e1,e2])

        data = ['\t'.join(sorted(line)) + '\n' for line in d]
        data = sorted(set(data))

        print "%d known fusion genes found in healthy samples" % (len(data),)

    file(os.path.join(options.output_directory,'healthy.txt'),'w').writelines(data)
    #
