#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of candidate fusion genes. This list is hard coded
in here and it is manually curated from RNA-seq data from tissue 
samples of 95 human individuals representing 27 different tissues 
<https://www.ebi.ac.uk/arrayexpress/experiments/E-MTAB-1733/>.



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of pre-candidate fusion genes from HPA RNA-seq database of 95 samples from 27 tissues."""
    version = "%prog 0.12 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of allowed candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of allowed candidate fusion genes is generated. Default is '%default'.""")

    parser.add_option("--skip-filter-overlap",
                      action="store_true",
                      dest="skip_filter_overlap",
                      default = False,
                      help="""If set then it filters out the known fusion genes where the (i) genes are fully overlapping, or (ii) the genes are partially overlapping and are on the same strand. Default is '%default'.""")

    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    print "Generating the list of HPA fusion genes..."
    fusions = dict()

    # manual curation from papers

    fusions['rattus_norvegicus'] = []

    fusions['mus_musculus'] = []

    fusions['canis_lupus_familiaris'] = []

    fusions['homo_sapiens'] = [
        ['A2M','MB'],
        ['A2ML1','KRT4'],
        ['A2ML1','MYH11'],
        ['A2ML1','NDE1'],
        ['A2ML1','PHC1'],
        ['A2ML1','RIMKLB'],
        ['A3GALT2','PHC2'],
        ['AACSP1','GBP6'],
        ['AADACL3','C1ORF158'],
        ['AAK1','CTC-513N18.7'],
        ['AAK1','ZNF626'],
        ['AARS','SMR3B'],
        ['AB015752.3','TMEM64'],
        ['ABAT','ALB'],
        ['ABCA13','LYZ'],
        ['ABCA17P','CCNF'],
        ['ABCA8','LYZ'],
        ['ABHD2','CCNY'],
        ['ABHD2','LYZ'],
        ['ABHD2','SMR3B'],
        ['ABI1','LYZ'],
        ['ABI2','CYP20A1'],
        ['ABI2','SMR3B'],
        ['ABI2','TG'],
        ['ABLIM1','HSF2'],
        ['ABR','LYZ'],
        ['ABTB2','MMADHC'],
        ['AC000124.1','ARF5'],
        ['AC002116.7','WDR62'],
        ['AC002398.11','AD000671.6'],
        ['AC002398.12','DES'],
        ['AC002398.12','MYL2'],
        ['AC002398.12','RP11-814P5.1'],
        ['AC002398.12','SPARCL1'],
        ['AC002480.2','STEAP1B'],
        ['AC004381.7','THUMPD1'],
        ['AC004510.3','ZNF861P'],
        ['AC004895.4','USP42'],
        ['AC004967.7','OR7E7P'],
        ['AC005013.5','CTB-113D17.1'],
        ['AC005077.5','SRRM3'],
        ['AC005077.7','GTF2IP7'],
        ['AC005154.7','GGCT'],
        ['AC005237.4','TG'],
        ['AC005253.4','CRLF1'],
        ['AC005487.2','DNAJB9'],
        ['AC005578.3','USP34'],
        ['AC005703.3','PMP22'],
        ['AC005775.2','ODF3L2'],
        ['AC005786.7','FZR1'],
        ['AC006116.15','AC006116.21'],
        ['AC006262.10','AC006262.5'],
        ['AC006538.1','SLC39A3'],
        ['AC006547.14','KB-1269D1.8'],
        ['AC007228.11','CFLAR'],
        ['AC007255.8','ZNRF2P2'],
        ['AC007365.1','AFTPH'],
        ['AC007563.5','MYH11'],
        ['AC008278.2','DDX1'],
        ['AC008440.5','IGF2'],
        ['AC008440.5','MTCH2'],
        ['AC008440.5','NLRP1'],
        ['AC008440.5','SPARCL1'],
        ['AC009014.3','TRPC7'],
        ['AC009133.22','LYZ'],
        ['AC009237.11','RP11-440D17.5'],
        ['AC010136.2','RUFY4'],
        ['AC010149.4','CLASP2'],
        ['AC010149.4','MORC1'],
        ['AC010149.4','RP11-79E3.2'],
        ['AC010504.2','IGF2'],
        ['AC010518.3','LILRA2'],
        ['AC010524.2','TEAD2'],
        ['AC010642.1','ZSCAN22'],
        ['AC010731.4','ADAM23'],
        ['AC011513.3','LYZ'],
        ['AC011513.3','PIGR'],
        ['AC011523.2','KLK3'],
        ['AC012322.1','EBPL'],
        ['AC012442.6','CHCHD5'],
        ['AC013463.2','AC019181.2'],
        ['AC015849.2','RP11-1143G9.4'],
        ['AC015849.2','TAF15'],
        ['AC015971.2','B2M'],
        ['AC015987.1','IGF2'],
        ['AC016582.2','WDR87'],
        ['AC016831.7','RP11-138A9.2'],
        ['AC016907.3','ALK'],
        ['AC018731.3','NEB'],
        ['AC018737.3','TSN'],
        ['AC018890.6','TG'],
        ['AC019349.5','CRNN'],
        ['AC019349.5','GSN'],
        ['AC019349.5','KRT15'],
        ['AC019349.5','KRT4'],
        ['AC019349.5','NDE1'],
        ['AC019349.5','S100A9'],
        ['AC019349.5','SPINK5'],
        ['AC024560.3','AC069213.1'],
        ['AC024592.9','RP1-122P22.4'],
        ['AC025335.1','PLXDC2'],
        ['AC026167.1','AC027119.1'],
        ['AC026202.3','TEX101'],
        ['AC053503.6','CRNN'],
        ['AC053503.6','DLGAP4-AS1'],
        ['AC053503.6','KRT13'],
        ['AC053503.6','MYH6'],
        ['AC053503.6','NPPA'],
        ['AC053503.6','PPP1R12B'],
        ['AC053503.6','SYNPO'],
        ['AC053503.6','TNNC1'],
        ['AC053503.6','TNNI3'],
        ['AC053503.6','TNNT2'],
        ['AC053503.6','TPM1'],
        ['AC068580.7','AC139143.2'],
        ['AC073133.1','LINC01006'],
        ['AC073343.13','IGF2'],
        ['AC073610.5','IGH@'],
        ['AC073842.19','LAMTOR4'],
        ['AC074008.1','ALMS1'],
        ['AC074117.10','GPD1L'],
        ['AC078842.3','DGKI'],
        ['AC078842.3','PTN'],
        ['AC078899.3','ZNF66'],
        ['AC079354.5','DAZAP2P1'],
        ['AC079753.5','IL37'],
        ['AC079767.4','PPP1R14BP2'],
        ['AC084219.2','ZNF224'],
        ['AC087380.14','UBQLNL'],
        ['AC092687.4','RP11-245G13.2'],
        ['AC093698.4','CRYGA'],
        ['AC093698.4','CRYGD'],
        ['AC096579.13','ANPEP'],
        ['AC096579.13','CALD1'],
        ['AC096579.13','CD53'],
        ['AC096579.13','CTGF'],
        ['AC096579.13','DES'],
        ['AC096579.13','DPCR1'],
        ['AC096579.13','EIF2B5'],
        ['AC096579.13','HBB'],
        ['AC096579.13','JCHAIN'],
        ['AC096579.13','MUC6'],
        ['AC096579.13','NDE1'],
        ['AC096579.13','PGC'],
        ['AC096579.13','PIGR'],
        ['AC096579.13','PRH1'],
        ['AC096579.13','PRH1-PRR4'],
        ['AC096579.13','PRM2'],
        ['AC096579.13','PRR4'],
        ['AC096579.13','SMR3B'],
        ['AC096579.13','TFF1'],
        ['AC096579.13','TRBC2'],
        ['AC097468.7','LINC01494'],
        ['AC099344.3','SLC52A3'],
        ['AC099850.1','PRR11'],
        ['AC100802.3','EMP2'],
        ['AC104532.2','MKLN1'],
        ['AC104699.1','LAMP2'],
        ['AC104820.2','ITGA4'],
        ['AC132217.4','SPARC'],
        ['AC141586.5','KCTD5'],
        ['AC226119.4','SFTPB'],
        ['AC226119.5','SFTPB'],
        ['ACACA','EIF3G'],
        ['ACE2','TMEM27'],
        ['ACIN1','MTFR1L'],
        ['ACIN1','MUC7'],
        ['ACIN1','RALGAPA1'],
        ['ACIN1','RP1-317E23.6'],
        ['ACKR2','ALB'],
        ['ACKR2','CALR'],
        ['ACPP','ACTB'],
        ['ACPP','B2M'],
        ['ACPP','CALD1'],
        ['ACPP','MSMB'],
        ['ACPP','MYH11'],
        ['ACPP','NCAPD3'],
        ['ACPP','SGMS1'],
        ['ACRBP','RMI2'],
        ['ACSF2','ALB'],
        ['ACSL1','FTX'],
        ['ACSL1','HBB'],
        ['ACSL1','RP11-351N6.1'],
        ['ACSL1','RP11-701P16.5'],
        ['ACSL4','SULT2A1'],
        ['ACTA1','ACTA2-AS1'],
        ['ACTA1','HSPB6'],
        ['ACTA1','MYL2'],
        ['ACTA1','RP11-814P5.1'],
        ['ACTA1','STAMBPL1'],
        ['ACTA1','TNNI3'],
        ['ACTA2','IGFBP5'],
        ['ACTA2','NPPA-AS1'],
        ['ACTA2-AS1','ACTC1'],
        ['ACTA2-AS1','ACTG2'],
        ['ACTA2-AS1','NPPA'],
        ['ACTA2-AS1','RP11-814P5.1'],
        ['ACTB','ATP8B4'],
        ['ACTB','AZU1'],
        ['ACTB','CAPN12'],
        ['ACTB','KLK3'],
        ['ACTB','LLFOS-48D6.2'],
        ['ACTB','LYZ'],
        ['ACTB','MDK'],
        ['ACTB','MPO'],
        ['ACTB','MUC7'],
        ['ACTB','NDE1'],
        ['ACTB','OAZ1'],
        ['ACTB','RP5-972B16.2'],
        ['ACTB','SNRPD3'],
        ['ACTB','STAMBPL1'],
        ['ACTB','STATH'],
        ['ACTBP8','RP11-459O1.2'],
        ['ACTC1','CRYAB'],
        ['ACTC1','HSPB6'],
        ['ACTC1','NPPA-AS1'],
        ['ACTG1','AMY2A'],
        ['ACTG1','STAMBPL1'],
        ['ACTG1','XIST'],
        ['ACTG2','AHNAK'],
        ['ACTG2','CALR'],
        ['ACTG2','COL1A2'],
        ['ACTG2','GREB1L'],
        ['ACTG2','MYLK'],
        ['ACTG2','MYLK-AS1'],
        ['ACTG2','NDE1'],
        ['ACTG2','RP11-814P5.1'],
        ['ACTG2','STAMBPL1'],
        ['ACTN1','HBB'],
        ['ACTN2','CSRP3'],
        ['ACTN2','TNNT2'],
        ['ACTN2','TPM1'],
        ['ACTN4','LYZ'],
        ['ACTN4','PRSS1'],
        ['ACTN4','TG'],
        ['ACTN4','TRB@'],
        ['ACTR10','CRHR1-IT1_'],
        ['ACTR6','MPO'],
        ['ACTR8','SELK'],
        ['ACYP2','GBP6'],
        ['ADAD1','IL21-AS1'],
        ['ADAM10','MRI1'],
        ['ADAM12','IGF2'],
        ['ADAM32','ADAM9'],
        ['ADAM5','RP11-122L4.2'],
        ['ADAMTS16','ICE1'],
        ['ADAMTS17','SPATA41'],
        ['ADAMTSL3','LYZ'],
        ['ADAMTSL3','SH3GL3'],
        ['ADAMTSL4-AS1','IGF2'],
        ['ADAP2','RNF135'],
        ['ADCY1','ZSCAN5A'],
        ['ADCY10','FAM177A1'],
        ['ADCY5','NPPA'],
        ['ADD1','LYZ'],
        ['ADD1','SH3BP2'],
        ['ADD3','ENAM'],
        ['ADD3','LPP'],
        ['ADD3','TG'],
        ['ADGRA3','TECRL'],
        ['ADGRE4P','RP11-1137G4.4'],
        ['ADGRE5','AK2'],
        ['ADGRG5','HMGB3P32'],
        ['ADGRG7','TFG'],
        ['ADGRL2','FAM134C'],
        ['ADGRL2','ZG16B'],
        ['ADH1B','HP'],
        ['ADH1B','RP11-573D15.8'],
        ['ADH1B','TXNL4B'],
        ['ADH4','ALB'],
        ['ADH4','C3'],
        ['ADIG','ARHGAP40'],
        ['ADIPOQ','CPM'],
        ['ADIPOQ','DNAH14'],
        ['ADIPOR1','CYB5R1'],
        ['ADNP','DPM1'],
        ['ADPRHL1','RP11-102K13.5'],
        ['ADRBK1','LYZ'],
        ['AEBP2','PLEKHA5'],
        ['AES','FLNC'],
        ['AES','MUC7'],
        ['AES','SMR3B'],
        ['AF001548.5','KRT13'],
        ['AF064860.7','IGSF5'],
        ['AF127936.9','HBB'],
        ['AF213884.2','RP11-499E18.1'],
        ['AFF4','F11R'],
        ['AFF4','RP11-544M22.13'],
        ['AFG3L2P1','LYZ'],
        ['AFMID','LYZ'],
        ['AGAP6','TIMM23B'],
        ['AGBL2','MTCH2'],
        ['AGFG2','RP11-44M6.1'],
        ['AGL','HIAT1'],
        ['AGMAT','FBXW12'],
        ['AGPAT3','TG'],
        ['AGR2','LYZ'],
        ['AHCYL2','PRPF4'],
        ['AHNAK','KRT10'],
        ['AHNAK','KRT14'],
        ['AHNAK','KRT2'],
        ['AHNAK','LYZ'],
        ['AHNAK','RP11-1143G9.4'],
        ['AHNAK','SMR3B'],
        ['AHSP','RP11-452L6.7'],
        ['AIF1L','NUP214'],
        ['AK3','RP11-265B8.5'],
        ['AK4P4','ERMP1'],
        ['AKAP13','LYZ'],
        ['AKAP13','STATH'],
        ['AKIRIN1','LYZ'],
        ['AKR1B1','CYP17A1'],
        ['AKR1B1','GML'],
        ['AKR1B10','IGK@'],
        ['AKR1B10','RP11-1143G9.4'],
        ['AKR1C2','AKR1C8P'],
        ['AKR1C5P','AKR1C8P'],
        ['AKT2','C19ORF47'],
        ['AKT3','TRB@'],
        ['AL121578.2','SYTL5'],
        ['AL121656.5','LTBP1'],
        ['ALB','ALDH2'],
        ['ALB','ALDOB'],
        ['ALB','APOA1-AS'],
        ['ALB','APOB'],
        ['ALB','APOH'],
        ['ALB','C9'],
        ['ALB','CANX'],
        ['ALB','CCDC152'],
        ['ALB','CCNY'],
        ['ALB','CFH'],
        ['ALB','CLU'],
        ['ALB','CP'],
        ['ALB','CPS1'],
        ['ALB','CTSB'],
        ['ALB','CTSD'],
        ['ALB','CYP2A6'],
        ['ALB','DMGDH'],
        ['ALB','EIF2B5'],
        ['ALB','F2'],
        ['ALB','F9'],
        ['ALB','FFAR4'],
        ['ALB','FMO5'],
        ['ALB','FN1'],
        ['ALB','GSTA2'],
        ['ALB','HPS3'],
        ['ALB','IGHG1'],
        ['ALB','IGK@'],
        ['ALB','IGKC'],
        ['ALB','INHBE'],
        ['ALB','ITIH2'],
        ['ALB','ITM2B'],
        ['ALB','KRBOX1'],
        ['ALB','LIPA'],
        ['ALB','MYO1B'],
        ['ALB','NDUFS6'],
        ['ALB','NNMT'],
        ['ALB','NUCB1'],
        ['ALB','OPN3'],
        ['ALB','PCBP1-AS1'],
        ['ALB','PCSK6'],
        ['ALB','PDIA4'],
        ['ALB','PIPOX'],
        ['ALB','PLG'],
        ['ALB','PROC'],
        ['ALB','PTGR1'],
        ['ALB','RP11-136C24.3'],
        ['ALB','RP11-141M3.6'],
        ['ALB','RP11-295K3.1'],
        ['ALB','RP11-35N6.6'],
        ['ALB','RP11-573D15.8'],
        ['ALB','RP11-64D24.2'],
        ['ALB','RP11-696N14.1'],
        ['ALB','RP11-903H12.3'],
        ['ALB','RP11-96H19.1'],
        ['ALB','RRBP1'],
        ['ALB','SARM1'],
        ['ALB','SCP2'],
        ['ALB','SERPIND1'],
        ['ALB','SERPINE1'],
        ['ALB','SERPING1'],
        ['ALB','SNAP91'],
        ['ALB','ST6GAL1'],
        ['ALB','STOM'],
        ['ALB','TAT'],
        ['ALB','TAT-AS1'],
        ['ALB','TF'],
        ['ALB','TMBIM6'],
        ['ALB','TXNL4B'],
        ['ALB','UGT1A8'],
        ['ALB','UGT2B11'],
        ['ALDH1L1','SMR3B'],
        ['ALDH1L2','RP11-61E11.2'],
        ['ALDH2','LYZ'],
        ['ALDH2','SFTPB'],
        ['ALDH3A2','SLC47A1'],
        ['ALDH3B2','TMEM99'],
        ['ALDH5A1','LYZ'],
        ['ALDOA','IGF2'],
        ['ALDOA','MUC5AC'],
        ['ALDOB','ANPEP'],
        ['ALDOB','CRP'],
        ['ALDOB','HRG'],
        ['ALDOB','PIGR'],
        ['ALDOB','TXNL4B'],
        ['ALG11','YES1'],
        ['ALG9','OSMR'],
        ['ALOX12','RNASEK'],
        ['ALOX12','RNASEK-C17ORF49'],
        ['ALX1','LRRIQ1'],
        ['AMACR','C1QTNF3'],
        ['AMACR','SLC45A2'],
        ['AMBRA1','CHRM4'],
        ['AMBRA1','GFAP'],
        ['AMOTL1','IGF2'],
        ['AMY2A','CLPS'],
        ['AMY2A','PRSS2'],
        ['AMY2A','TRB@'],
        ['AMY2B','CLPS'],
        ['AMY2B','PRSS2'],
        ['AMY2B','TRB@'],
        ['AMZ1','APOLD1'],
        ['AMZ1','IGF2'],
        ['ANAPC15','LAMTOR1'],
        ['ANAPC16','TG'],
        ['ANAPC16','UGDH'],
        ['ANGPTL3','RP11-696N14.1'],
        ['ANK1','AZU1'],
        ['ANK3','TG'],
        ['ANKIB1','CYP51A1-AS1'],
        ['ANKLE2','GOLGA3'],
        ['ANKMY2','ZNF708'],
        ['ANKRD1','LDB3'],
        ['ANKRD11','LYZ'],
        ['ANKRD11','STOM'],
        ['ANKRD13A','CRISPLD2'],
        ['ANKRD13D','LINC00337'],
        ['ANKRD17','SMR3B'],
        ['ANKRD37','RP11-714G18.1'],
        ['ANO10','TG'],
        ['ANO3','REEP5'],
        ['ANP32B','HBB'],
        ['ANP32B','HMGB2'],
        ['ANPEP','CELA2A'],
        ['ANPEP','IGH@'],
        ['ANPEP','IGK@'],
        ['ANPEP','IGKC'],
        ['ANPEP','LYZ'],
        ['ANPEP','PIGR'],
        ['ANPEP','REG3A'],
        ['ANPEP','TRB@'],
        ['ANTXR1','IGK@'],
        ['ANXA1','CRNN'],
        ['ANXA1','HBB'],
        ['ANXA1','HTN3'],
        ['ANXA11','TRB@'],
        ['AOAH','CXCL8'],
        ['AOX1','SPTLC2'],
        ['AP000295.9','ZNF595'],
        ['AP000304.12','HTN1'],
        ['AP000347.2','KB-1572G7.3'],
        ['AP000438.2','CYP4F3'],
        ['AP000688.14','PIGR'],
        ['AP000695.6','CHAF1B'],
        ['AP000860.2','SLC25A25-AS1'],
        ['AP001350.4','CNTF'],
        ['AP001350.4','ZFP91-CNTF'],
        ['AP006621.8','PRH1'],
        ['AP006621.8','PRH1-PRR4'],
        ['AP006621.8','PRR4'],
        ['AP3B2','SRSF9'],
        ['AP3D1','SMR3B'],
        ['AP3S1','COMMD10'],
        ['AP4B1','PTPN22'],
        ['AP4B1-AS1','RP5-1073O3.2'],
        ['AP5B1','LYZ'],
        ['APBB2','TG'],
        ['APBB3','NDUFA2'],
        ['APLP2','PRSS1'],
        ['APOA1','CLU'],
        ['APOA1','IGHG1'],
        ['APOA1','PTMS'],
        ['APOA1','RP11-573D15.8'],
        ['APOA1','SARM1'],
        ['APOA1','SIK3'],
        ['APOA1-AS','ATF5'],
        ['APOA1-AS','C3'],
        ['APOA1-AS','FGA'],
        ['APOA1-AS','HRG'],
        ['APOA1-AS','MIDN'],
        ['APOA1-AS','SERPINA1'],
        ['APOA4','PIGR'],
        ['APOB','C3'],
        ['APOB','IGKC'],
        ['APOB','PIGR'],
        ['APOC1P1','APOC4-APOC2'],
        ['APOE','FFAR4'],
        ['APOH','C3'],
        ['APOL1','TNFRSF1B'],
        ['APOLD1','RP11-108K3.1'],
        ['APOLD1','TG'],
        ['APOPT1','RP11-894P9.2'],
        ['APP','IGF2'],
        ['APP','LYZ'],
        ['APP','MYH7'],
        ['APP','PIGR'],
        ['APP','SMR3B'],
        ['APP','STATH'],
        ['AQP1','NPPA-AS1'],
        ['AQP2','GPX3'],
        ['AQP3','MYH11'],
        ['AQP4','RP11-862L9.3'],
        ['AQP4-AS1','MBP'],
        ['AQP8','PRSS1'],
        ['AQP8','PRSS2'],
        ['AQP8','TRB@'],
        ['ARCN1','LRRC28'],
        ['ARCN1','NPPA-AS1'],
        ['ARF1','LYZ'],
        ['ARF4','DENND6A'],
        ['ARFGAP1','PNLIP'],
        ['ARHGAP11A','SCG5'],
        ['ARHGAP35','CTSS'],
        ['ARHGAP35','TG'],
        ['ARHGAP4','OPN1LW'],
        ['ARHGEF1','NPPA-AS1'],
        ['ARHGEF2','SSR2'],
        ['ARHGEF3','FGD5-AS1'],
        ['ARID1B','CD24'],
        ['ARID1B','PIGR'],
        ['ARID4B','LYZ'],
        ['ARIH2','LYZ'],
        ['ARL10','KRT13'],
        ['ARL15','KLK3'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARL6','EPHA6'],
        ['ARL6IP1','SMG1'],
        ['ARL8A','PTPN7'],
        ['ARL9','SRP72'],
        ['ARMCX4','TG'],
        ['ARMS2','PLEKHA1'],
        ['ARPC1A','PAPOLA'],
        ['ARPC2','EDC3'],
        ['ARPC4-TTLL3','CNTN4'],
        ['ARPC4-TTLL3','CTD-2267D19.3'],
        ['ARPC5L','RP11-944L7.4'],
        ['ARPIN','RAI14'],
        ['ARPP19','RPL37A'],
        ['ARRB2','LYZ'],
        ['ARRDC1','OSTF1'],
        ['ARRDC3','MPO'],
        ['ARSD','SSH2'],
        ['ARSD','ZNF668'],
        ['ARSE','TAF13'],
        ['ARSG','CARNS1'],
        ['ART3','RP11-630D6.5'],
        ['AS3MT','CNNM2'],
        ['ASB15','LMOD2'],
        ['ASCC1','SPOCK2'],
        ['ASF1B','PRKACA'],
        ['ASH1L','GBP6'],
        ['ASH1L','STATH'],
        ['ASH1L','TG'],
        ['ASIP','EIF5'],
        ['ASLP1','KB-1572G7.3'],
        ['ASMTL','SLC25A6'],
        ['ASNS','RP11-420N3.2'],
        ['ASNS','RP11-420N3.3'],
        ['ASPG','TDRD9'],
        ['ASS1','HMCN2'],
        ['ASXL1','KIF3B'],
        ['ATF3','HNRNPH1'],
        ['ATF3','TG'],
        ['ATF5','CLU'],
        ['ATF5','TG'],
        ['ATF5','TXNL4B'],
        ['ATF6','DUSP12'],
        ['ATG12','CDO1'],
        ['ATG16L1','INPP5D'],
        ['ATG16L1','SAG'],
        ['ATG16L2','DSC3'],
        ['ATM','PRH1'],
        ['ATM','PRH1-PRR4'],
        ['ATM','PRR4'],
        ['ATM','RHOBTB3'],
        ['ATP11B','MYH11'],
        ['ATP12A','TUBGCP2'],
        ['ATP1A1','GP2'],
        ['ATP1A1','IGH@'],
        ['ATP1A1','LYZ'],
        ['ATP1A1','PIGR'],
        ['ATP1A1-AS1','ATP1A3'],
        ['ATP1A1-AS1','IGH@'],
        ['ATP1A1-AS1','MUC7'],
        ['ATP1A1-AS1','TG'],
        ['ATP1A2','NRXN3'],
        ['ATP1B3P1','RP11-416L21.2'],
        ['ATP2A2','LYZ'],
        ['ATP2A3','GANAB'],
        ['ATP2A3','GP2'],
        ['ATP2B4','NPPA-AS1'],
        ['ATP5A1','NPPA-AS1'],
        ['ATP5B','KRT13'],
        ['ATP5L','SLC25A37'],
        ['ATP6V0A2','TCTN2'],
        ['ATP6V0C','DDX5'],
        ['ATP6V1C1','ECE1'],
        ['ATP6V1C1','KB-1507C5.4'],
        ['ATP6V1F','FLNC'],
        ['ATP6V1G2-DDX39B','LYZ'],
        ['ATP6V1G2-DDX39B','XXBAC-BPG299F13.16'],
        ['ATP6V1G3','RP11-553K8.2'],
        ['ATP6V1H','STARD4-AS1'],
        ['ATP7A','LYZ'],
        ['ATP7A','ZC3H12B'],
        ['ATP7B','CTAGE3P'],
        ['ATP8B1','AZGP1'],
        ['ATPIF1','HBB'],
        ['ATXN2','PEX26'],
        ['AUH','RNF4'],
        ['AURKC','ZNF264'],
        ['AUTS2','LYZ'],
        ['AUTS2','SMR3B'],
        ['AVEN','CCDC69'],
        ['AZGP1','CD36'],
        ['AZGP1','LYZ'],
        ['AZGP1','PRB1'],
        ['AZGP1','PRB2'],
        ['AZGP1','PRH1'],
        ['AZGP1','PRH1-PRR4'],
        ['AZGP1','PRR4'],
        ['AZGP1','RP11-573D15.8'],
        ['AZGP1','SMR3B'],
        ['AZGP1','TGM4'],
        ['AZIN1','TG'],
        ['AZIN1','TRB@'],
        ['AZIN1','YWHAZ'],
        ['AZU1','CLTC'],
        ['AZU1','DNM2'],
        ['AZU1','IGHG1'],
        ['AZU1','MYH9'],
        ['AZU1','NARF'],
        ['AZU1','NEMP1'],
        ['AZU1','NR3C1'],
        ['AZU1','OOEP-AS1'],
        ['AZU1','PLEK'],
        ['AZU1','PTP4A1'],
        ['AZU1','RBM3'],
        ['AZU1','RP11-1143G9.4'],
        ['AZU1','S100A9'],
        ['AZU1','SRGN'],
        ['AZU1','TUBGCP3'],
        ['AZU1','ZMYM1'],
        ['B2M','CXCR4'],
        ['B2M','GNAS'],
        ['B2M','HBA2'],
        ['B2M','KLF13'],
        ['B2M','LINC01578'],
        ['B2M','PTP4A1'],
        ['B2M','REG3A'],
        ['B2M','RP11-437B10.1'],
        ['B2M','SYN3'],
        ['B3GNT6','RP11-191A15.2'],
        ['B3GNTL1','TRAF6'],
        ['B4GALNT2','EIF3I'],
        ['BAAT','TRMT112P4'],
        ['BACE1','GP2'],
        ['BACE2','IGF2'],
        ['BAG2','ZNF451'],
        ['BAG6','SMR3B'],
        ['BAG6','TG'],
        ['BAGE2','GBP6'],
        ['BAHD1','IVD'],
        ['BAIAP2','TFB2M'],
        ['BANK1','LYZ'],
        ['BBS2','NUDT21'],
        ['BBS5','KLHL41'],
        ['BCAM','LYZ'],
        ['BCAS1','LYZ'],
        ['BCKDHB','DENND1A'],
        ['BCL2L13','LYZ'],
        ['BCL2L15','PTPN22'],
        ['BCL2L2-PABPN1','KRT1'],
        ['BCL9L','LYZ'],
        ['BCLAF1','LYZ'],
        ['BCO1','TRIM56'],
        ['BDH1','CTSB'],
        ['BDH1','IGF2'],
        ['BEAN1-AS1','PIGR'],
        ['BFAR','SYN3'],
        ['BFSP2','RP11-91K8.4'],
        ['BGN','CKMT2-AS1'],
        ['BGN','IGHG1'],
        ['BGN','IGHM'],
        ['BGN','NPPA-AS1'],
        ['BGN','PRKCH'],
        ['BGN','RP11-452D21.1'],
        ['BHLHB9','RP4-769N13.6'],
        ['BLOC1S5-TXNDC5','HTN3'],
        ['BLOC1S5-TXNDC5','SMR3B'],
        ['BMP10','NPPA-AS1'],
        ['BMPR2','TG'],
        ['BMS1P20','LL22NC03-2H8.4'],
        ['BNIPL','FLG2'],
        ['BPGM','CALD1'],
        ['BPGM','HBB'],
        ['BPI','S100A9'],
        ['BPIFA2','HTN1'],
        ['BPIFA2','MOCOS'],
        ['BPIFA2','PIGR'],
        ['BPIFA2','SMR3B'],
        ['BPIFB1','LYZ'],
        ['BPIFB2','SMR3B'],
        ['BPNT1','PDXDC2P'],
        ['BRD7','SLC22A10'],
        ['BRD9P2','TPPP'],
        ['BRDT','EPHX4'],
        ['BRF1','TRIM22'],
        ['BRI3BP','MYO1D'],
        ['BSCL2','LYZ'],
        ['BSG','LYZ'],
        ['BSG','PTBP1'],
        ['BSG','RNF126'],
        ['BTBD9','RP11-1143G9.4'],
        ['BTG1','HBB'],
        ['BTG2','CEL'],
        ['BTG2','GP2'],
        ['BTG2','SRGN'],
        ['BTG2','TG'],
        ['BTN3A2','SCAMP1-AS1'],
        ['BVES','NUBPL'],
        ['BZW1','CBL'],
        ['C10ORF10','PNLIP'],
        ['C10ORF32-ASMT','CNNM2'],
        ['C10ORF32-ASMT','LYZ'],
        ['C10ORF67','LINC01552'],
        ['C12ORF60','TG'],
        ['C12ORF79','RP11-756G20.1'],
        ['C14ORF166','TG'],
        ['C15ORF38-AP3S2','RAI14'],
        ['C16ORF45','CBX3'],
        ['C16ORF52','VWA3A'],
        ['C16ORF62','PNLIP'],
        ['C16ORF72','HOOK2'],
        ['C16ORF72','LYZ'],
        ['C17ORF100','CTC-281F24.3'],
        ['C17ORF104','DBF4B'],
        ['C17ORF74','TMEM102'],
        ['C17ORF99','SYNGR2'],
        ['C19MC','RPLP1'],
        ['C1D','HZGJ'],
        ['C1D','WDR92'],
        ['C1ORF105','DNM3'],
        ['C1ORF159','RP11-172C16.5'],
        ['C1ORF167','TG'],
        ['C1ORF194','UQCR10'],
        ['C1ORF220','RP5-1098D14.1'],
        ['C1ORF27','RP11-295K2.3'],
        ['C1ORF43','PNLIP'],
        ['C1ORF61','RP11-98G7.1'],
        ['C1QC','GML'],
        ['C1QTNF3-AMACR','RP11-1023L17.1'],
        ['C1QTNF3-AMACR','SLC45A2'],
        ['C1QTNF9B','MIPEP'],
        ['C1S','IGFBP5'],
        ['C20ORF187','RP11-103J8.2'],
        ['C20ORF203','COMMD7'],
        ['C20ORF62','LINC01430'],
        ['C2CD5','ST8SIA1'],
        ['C2ORF50','PECAM1'],
        ['C3','CLU'],
        ['C3','CREB3L3'],
        ['C3','GML'],
        ['C3','HRG'],
        ['C3','ITIH1'],
        ['C3','RBP4'],
        ['C3','RP11-573D15.8'],
        ['C3','SERPING1'],
        ['C3','TMEM176B'],
        ['C3','TXNL4B'],
        ['C3','ZFP36L1'],
        ['C3ORF22','TXNRD3'],
        ['C4BPB','PFKFB2'],
        ['C5AR1','INAFM1'],
        ['C5ORF24','TG'],
        ['C5ORF66','KRT13'],
        ['C6ORF141','KIAA1671'],
        ['C6ORF165','RP4-669H2.1'],
        ['C6ORF165','RP4-678D15.1'],
        ['C6ORF183','CCDC162P'],
        ['C6ORF183','ZNF528-AS1'],
        ['C6ORF201','RP3-400B16.1'],
        ['C6ORF48','HSPA1A'],
        ['C6ORF58','LYZ'],
        ['C6ORF58','PGC'],
        ['C6ORF62','LYZ'],
        ['C6ORF89','LYZ'],
        ['C7','CHGB'],
        ['C7','COL1A2'],
        ['C7','PNLIP'],
        ['C7ORF49','WDR91'],
        ['C7ORF50','COX19'],
        ['C7ORF50','PNLIP'],
        ['C7ORF60','HRAT17'],
        ['C8ORF89','RPL7'],
        ['C9ORF3','IGF2'],
        ['C9ORF3','MIR205HG'],
        ['C9ORF3','PIGR'],
        ['C9ORF43','RGS3'],
        ['C9ORF78','LYZ'],
        ['CA12','FLG-AS1'],
        ['CA6','LYZ'],
        ['CA6','MUC7'],
        ['CA6','PRB2'],
        ['CA6','PRH1'],
        ['CA6','PRR27'],
        ['CA6','SMR3B'],
        ['CA6','STATH'],
        ['CA6','ZG16B'],
        ['CACNA1H','TG'],
        ['CADM1','ST3GAL2'],
        ['CADM3-AS1','IGF2'],
        ['CALCB','INSC'],
        ['CALCOCO1','KRT13'],
        ['CALD1','KRT13'],
        ['CALD1','KRT4'],
        ['CALD1','PNLIP'],
        ['CALM1','CXCR4'],
        ['CALM1','YWHAE'],
        ['CALM2','PNLIP'],
        ['CALML5','SMR3B'],
        ['CALN1','CD24'],
        ['CALR','CTC-326K19.6'],
        ['CALR','KRBOX1'],
        ['CALR','KRT13'],
        ['CALR','RP11-136C24.3'],
        ['CALR','RP11-141M3.6'],
        ['CALR','SMR3B'],
        ['CAMK1D','IL1RAPL1'],
        ['CAMKK1','P2RX1'],
        ['CAMLG','SEC24A'],
        ['CAMTA2','FBXW8'],
        ['CANX','MAML1'],
        ['CAP1','SMR3B'],
        ['CAPN1','TG'],
        ['CAPN2','IGKC'],
        ['CAPN8','IGH@'],
        ['CAPN8','LYZ'],
        ['CAPN8','RP11-1143G9.4'],
        ['CARD8','TG'],
        ['CARM1P1','FZD4'],
        ['CASC1','RP11-713N11.4'],
        ['CASC15','PRSS23'],
        ['CASP14','TMEM99'],
        ['CASQ2','MYH7'],
        ['CAST','CTD-2260A17.1'],
        ['CAST','HBB'],
        ['CAV2','TG'],
        ['CBR4','DES'],
        ['CBX3','FLG-AS1'],
        ['CBX4','IGF2'],
        ['CBX5','TG'],
        ['CC2D2B','RP11-248J23.6'],
        ['CCAR2','PNLIP'],
        ['CCDC113','RP11-459F6.1'],
        ['CCDC142','MOGS'],
        ['CCDC144NL-AS1','TBC1D27'],
        ['CCDC15','SLC37A2'],
        ['CCDC152','IGF2'],
        ['CCDC152','TF'],
        ['CCDC163P','TESK2'],
        ['CCDC17','GPBP1L1'],
        ['CCDC176','LIN52'],
        ['CCDC183','TMEM141'],
        ['CCDC25','PPIE'],
        ['CCDC47','STRADA'],
        ['CCDC83','SLC25A1P1'],
        ['CCDC88C','HBB'],
        ['CCER2','CTC-360G5.8'],
        ['CCIN','GLIPR2'],
        ['CCL28','LYZ'],
        ['CCL28','ORAI2'],
        ['CCL28','SMR3B'],
        ['CCND1','TG'],
        ['CCNG2','RP11-255H23.4'],
        ['CCNO','DHX29'],
        ['CCNT2-AS1','TMEM163'],
        ['CCNY','CREM'],
        ['CCPG1','PNLIP'],
        ['CCPG1','RAB27A'],
        ['CCSER1','RP11-1143G9.4'],
        ['CCT4','FAM161A'],
        ['CD164','RP11-960L18.1'],
        ['CD24','CEBPD'],
        ['CD24','CNTN4'],
        ['CD24','CTSB'],
        ['CD24','ERCC1'],
        ['CD24','FLNA'],
        ['CD24','GLTP'],
        ['CD24','IYD'],
        ['CD24','MACROD2'],
        ['CD24','PCYOX1'],
        ['CD24','PRKCA'],
        ['CD24','PVRL2'],
        ['CD24','RP11-58E21.4'],
        ['CD24','RP11-73B8.1'],
        ['CD24','SAP30L-AS1'],
        ['CD24','ZNF773'],
        ['CD36','IGF2'],
        ['CD44','FLG-AS1'],
        ['CD44','HTN1'],
        ['CD44','PRB1'],
        ['CD44','PRH1'],
        ['CD44','PRH1-PRR4'],
        ['CD44','PRR4'],
        ['CD44','SMR3B'],
        ['CD44','TG'],
        ['CD46','SMR3B'],
        ['CD53','RP11-96K19.2'],
        ['CD55','SMR3B'],
        ['CD59','HTN3'],
        ['CD59','MUC7'],
        ['CD63','HBB'],
        ['CD63','HTN3'],
        ['CD63','PRSS1'],
        ['CD68','MPDU1'],
        ['CD74','PIGR'],
        ['CD74','PSCA'],
        ['CD74','RP11-1143G9.4'],
        ['CD74','SMR3B'],
        ['CD74','STATH'],
        ['CD74','TG'],
        ['CD79A','RPS19'],
        ['CD81','PRSS2'],
        ['CD9','NDE1'],
        ['CD93','SYN3'],
        ['CDA','PINK1'],
        ['CDADC1','GTF2IP23'],
        ['CDADC1','LYZ'],
        ['CDADC1','RP11-203I16.7'],
        ['CDC16','UPF3A'],
        ['CDC40','LYZ'],
        ['CDC40','WDR78'],
        ['CDC42BPA','FGD5-AS1'],
        ['CDC42BPG','EHD1'],
        ['CDH23','CXCL16'],
        ['CDH5','TG'],
        ['CDHR1','FLG-AS1'],
        ['CDK11B','SLC35E2'],
        ['CDK12','HBB'],
        ['CDK12','MGEA5'],
        ['CDK2','RAB5B'],
        ['CDK20','RP11-350E12.5'],
        ['CDK2AP2','TG'],
        ['CDKL4','MAP4K3'],
        ['CDKN1C','FN1'],
        ['CDKN2B-AS1','UBA52'],
        ['CDT1','HBB'],
        ['CDV3','CDV3P1'],
        ['CEACAM1','PRH1'],
        ['CEACAM1','PRH1-PRR4'],
        ['CEACAM20','CEACAM22P'],
        ['CEACAM20','ZNF180'],
        ['CEACAM5','ENAM'],
        ['CEACAM5','IGH@'],
        ['CEACAM5','IGK@'],
        ['CEACAM5','IGKC'],
        ['CEACAM5','PIGR'],
        ['CECR2','TERF2IP'],
        ['CEL','CELA2A'],
        ['CEL','CELA2B'],
        ['CEL','CELP'],
        ['CEL','CLPS'],
        ['CEL','CPA1'],
        ['CEL','CPB1'],
        ['CEL','CTRB1'],
        ['CEL','CTRB2'],
        ['CEL','CTRC'],
        ['CEL','CUZD1'],
        ['CEL','HDLBP'],
        ['CEL','MT1A'],
        ['CEL','PLA2G1B'],
        ['CEL','PNLIP'],
        ['CEL','PNLIPRP1'],
        ['CEL','PRSS1'],
        ['CEL','PRSS2'],
        ['CEL','RP11-680B3.2'],
        ['CEL','SYCN'],
        ['CEL','TMEM59'],
        ['CEL','TRB@'],
        ['CELA2A','CFTR'],
        ['CELA2A','CLPS'],
        ['CELA2A','CPA1'],
        ['CELA2A','CPA2'],
        ['CELA2A','CPB1'],
        ['CELA2A','CTC-479C5.12'],
        ['CELA2A','GP2'],
        ['CELA2A','PNLIP'],
        ['CELA2A','PRSS1'],
        ['CELA2A','PRSS2'],
        ['CELA2A','PRSS3'],
        ['CELA2A','RP11-331F4.4'],
        ['CELA2A','RP11-680B3.2'],
        ['CELA2A','RRBP1'],
        ['CELA2A','SYCN'],
        ['CELA2A','TRB@'],
        ['CELA2A','UBE2R2-AS1'],
        ['CELA2B','CLPS'],
        ['CELA2B','CPA1'],
        ['CELA2B','CPA2'],
        ['CELA2B','CPB1'],
        ['CELA2B','GP2'],
        ['CELA2B','PNLIP'],
        ['CELA2B','PRSS1'],
        ['CELA2B','PRSS2'],
        ['CELA2B','RP11-331F4.4'],
        ['CELA2B','RP11-680B3.2'],
        ['CELA2B','SYCN'],
        ['CELA2B','TRB@'],
        ['CELA3A','CLPS'],
        ['CELA3A','CPA1'],
        ['CELA3A','CPA2'],
        ['CELA3A','CTC-479C5.12'],
        ['CELA3A','GP2'],
        ['CELA3A','KLK1'],
        ['CELA3A','LINC00339'],
        ['CELA3A','PNLIP'],
        ['CELA3A','PRSS1'],
        ['CELA3A','PRSS2'],
        ['CELA3A','PRSS3'],
        ['CELA3A','TRB@'],
        ['CELA3A','UBE2R2-AS1'],
        ['CELA3B','CLPS'],
        ['CELA3B','GP2'],
        ['CELA3B','PNLIP'],
        ['CELA3B','PRSS1'],
        ['CELA3B','PRSS2'],
        ['CELA3B','PRSS3'],
        ['CELA3B','TRB@'],
        ['CELP','CPA1'],
        ['CELP','CPB1'],
        ['CELP','PNLIP'],
        ['CENPM','TNFRSF13C'],
        ['CEP152','RP11-227D13.5'],
        ['CEP63','EPHB1'],
        ['CEP76','GP2'],
        ['CEP78','PSAT1'],
        ['CERK','LCN10'],
        ['CERK','SLC25A51'],
        ['CERS6','SYTL2'],
        ['CES2','PIGR'],
        ['CFAP20','CSNK2A2'],
        ['CFAP54','NEDD1'],
        ['CFB','TXNL4B'],
        ['CFL1','HBB'],
        ['CFL1','LYZ'],
        ['CFL1','MUC7'],
        ['CFLAR','PKP2'],
        ['CFLAR','PTH2R'],
        ['CFLAR','RP11-13J8.1'],
        ['CFTR','HTN3'],
        ['CGNL1','PRSS1'],
        ['CGNL1','PRSS2'],
        ['CH17-212P11.4','IGH@'],
        ['CH17-212P11.5','IGH@'],
        ['CH17-212P11.5','IGHG1'],
        ['CH17-224D4.1','IGH@'],
        ['CH17-224D4.2','IGH@'],
        ['CHCHD10','VPREB3'],
        ['CHCHD2P6','DDI2'],
        ['CHCHD3','SRSF2'],
        ['CHD4','NOP2'],
        ['CHGA','CHGB'],
        ['CHGA','NR4A1'],
        ['CHGA','PRSS2'],
        ['CHGB','CYP11B1'],
        ['CHGB','HNRNPA2B1'],
        ['CHGB','MEG3'],
        ['CHMP3','LYZ'],
        ['CHST5','RP11-77K12.5'],
        ['CIITA','PIK3C2A'],
        ['CIRBP','PNLIP'],
        ['CIRBP','SMR3B'],
        ['CIRH1A','PHACTR4'],
        ['CITF22-1A6.3','ZBED4'],
        ['CIZ1','PNLIP'],
        ['CKM','MYH6'],
        ['CKM','NPPA-AS1'],
        ['CKMT2','ZCCHC9'],
        ['CLASP2','HBB'],
        ['CLCA1','PIGR'],
        ['CLCA2','RAPH1'],
        ['CLCN3','PRSS1'],
        ['CLCN3','TG'],
        ['CLCN3P1','TTC39B'],
        ['CLDN18','LYZ'],
        ['CLDN18','RP11-1143G9.4'],
        ['CLDN4','SMR3B'],
        ['CLEC11A','S100A9'],
        ['CLEC2B','ZNF708'],
        ['CLEC3B','NPPA-AS1'],
        ['CLEC6A','PIGR'],
        ['CLEC6A','UTP23'],
        ['CLEC7A','ERC1'],
        ['CLIC1','HBB'],
        ['CLIP1','RP11-512M8.5'],
        ['CLIP1','VPS33A'],
        ['CLK3','FOXN3'],
        ['CLK3','H2AFZP5'],
        ['CLK3','HEATR4'],
        ['CLK3','MPO'],
        ['CLK3','RP11-33N16.3'],
        ['CLMN','TRB@'],
        ['CLMP','FOXO4'],
        ['CLNK','ZNF518B'],
        ['CLPS','CPA1'],
        ['CLPS','CPA2'],
        ['CLPS','CPB1'],
        ['CLPS','CTRC'],
        ['CLPS','CUZD1'],
        ['CLPS','GP2'],
        ['CLPS','PLA2G1B'],
        ['CLPS','PNLIP'],
        ['CLPS','PNLIPRP2'],
        ['CLPS','PRSS1'],
        ['CLPS','PRSS2'],
        ['CLPS','PRSS3'],
        ['CLPS','RP11-331F4.4'],
        ['CLPS','TRB@'],
        ['CLPS','UBE2R2-AS1'],
        ['CLSTN1','IGFBP5'],
        ['CLSTN1','KRT10'],
        ['CLTCL1','HIRA'],
        ['CLU','CYP4A11'],
        ['CLU','EIF2B5'],
        ['CLU','PRSS2'],
        ['CLU','RMI2'],
        ['CLU','RP11-573D15.8'],
        ['CLU','TRB@'],
        ['CLU','TRBC2'],
        ['CLU','TRBV25-1'],
        ['CLVS1','KRT78'],
        ['CLYBL','PIGR'],
        ['CMAHP','FAM65B'],
        ['CMAHP','RP11-367G6.3'],
        ['CMAHP','RP3-522P13.3'],
        ['CMSS1','TBC1D23'],
        ['CMTM4','RP11-403P17.2'],
        ['CMYA5','TRDN'],
        ['CNDP2','TG'],
        ['CNN3','TG'],
        ['CNOT3','SMR3B'],
        ['CNTN4','TTLL3'],
        ['CNTNAP2','RN7SL72P'],
        ['COBL','MUC7'],
        ['COCH','GP2'],
        ['COG5','CPD'],
        ['COG5','TRB@'],
        ['COL14A1','IGF2'],
        ['COL17A1','KRT1'],
        ['COL18A1','IGF2'],
        ['COL1A1','IGK@'],
        ['COL1A1','PIGR'],
        ['COL1A1','SYN3'],
        ['COL1A2','SMR3B'],
        ['COL3A1','KRT4'],
        ['COL3A1','UBC'],
        ['COL4A1','IGF2'],
        ['COL4A1','MEG3'],
        ['COL4A1','SYN3'],
        ['COL4A2','H19'],
        ['COL4A2','IGF2'],
        ['COL4A2','PRH1'],
        ['COL4A2','PRH1-PRR4'],
        ['COL4A2','PRR4'],
        ['COL5A1','IGF2'],
        ['COL5A1','TMEM99'],
        ['COL6A2','IGF2'],
        ['COL6A2','MUC7'],
        ['COL8A1','RP11-201E8.1'],
        ['COLCA1','POU2AF1'],
        ['COLGALT1','FAM129C'],
        ['COMMD3-BMI1','RP11-573G6.9'],
        ['COMMD6','LYZ'],
        ['COPA','RP11-1143G9.4'],
        ['COPA','RP11-574F21.3'],
        ['COPS5','TCF24'],
        ['COPZ1','RP11-753H16.5'],
        ['COQ10B','LYZ'],
        ['COQ9','POLR2C'],
        ['COTL1','SFTPB'],
        ['COX18','DSG3'],
        ['COX20','TET1'],
        ['COX4I1','RP11-568J23.6'],
        ['COX4I1','TG'],
        ['COX6C','SMR3B'],
        ['COX7A2','TMEM30A'],
        ['COX7B','TCEANC2'],
        ['CPA1','CTRB1'],
        ['CPA1','CTRB2'],
        ['CPA1','CTRC'],
        ['CPA1','CTRL'],
        ['CPA1','DDIT4'],
        ['CPA1','EPAS1'],
        ['CPA1','EZR'],
        ['CPA1','FURIN'],
        ['CPA1','GLTSCR2'],
        ['CPA1','GP2'],
        ['CPA1','GRB10'],
        ['CPA1','LAMB2'],
        ['CPA1','PHLDA1'],
        ['CPA1','PNLIP'],
        ['CPA1','PNLIPRP2'],
        ['CPA1','PRSS1'],
        ['CPA1','PRSS2'],
        ['CPA1','PRSS3'],
        ['CPA1','RBM3'],
        ['CPA1','RBPJL'],
        ['CPA1','REG1A'],
        ['CPA1','RP11-331F4.4'],
        ['CPA1','RP11-680B3.2'],
        ['CPA1','TIMP3'],
        ['CPA1','TRB@'],
        ['CPA1','UBE2R2-AS1'],
        ['CPA2','GP2'],
        ['CPA2','PNLIP'],
        ['CPA2','PRSS1'],
        ['CPA2','PRSS2'],
        ['CPA2','TRB@'],
        ['CPA2','UBE2R2-AS1'],
        ['CPB1','CTRB1'],
        ['CPB1','CYB5A'],
        ['CPB1','GP2'],
        ['CPB1','MYH9'],
        ['CPB1','PNLIP'],
        ['CPB1','PNLIPRP2'],
        ['CPB1','PRSS1'],
        ['CPB1','PRSS2'],
        ['CPB1','PRSS3'],
        ['CPB1','RNASE1'],
        ['CPB1','TRB@'],
        ['CPB1','UBE2R2-AS1'],
        ['CPEB2-AS1','LINC00504'],
        ['CPED1','GBP6'],
        ['CPED1','WNT16'],
        ['CPM','RP11-702H23.4'],
        ['CPNE1','GP2'],
        ['CPT1B','NPPA-AS1'],
        ['CR2','HLA-E'],
        ['CRBN','TG'],
        ['CRCP','LYZ'],
        ['CREB3L2','IGF2'],
        ['CREBBP','TRB@'],
        ['CRIM1','TG'],
        ['CRISP3','HTN1'],
        ['CRISP3','SMR3B'],
        ['CRK','FAHD1'],
        ['CRLF1','HBB'],
        ['CRNN','KRT13'],
        ['CRNN','KRT4'],
        ['CRNN','KRT6A'],
        ['CRNN','MYH11'],
        ['CRNN','PPL'],
        ['CRNN','TAGLN'],
        ['CRNN','TGM3'],
        ['CRP','TXNL4B'],
        ['CRTC1','FZD4'],
        ['CRYAB','MYL2'],
        ['CRYM-AS1','CTD-2547E10.3'],
        ['CS','LYZ'],
        ['CSDE1','DES'],
        ['CSDE1','LYZ'],
        ['CSF2RB','IGF2'],
        ['CSF3R','LYZ'],
        ['CSF3R','MRPS15'],
        ['CSGALNACT1','GYG2'],
        ['CSH1','IGF2'],
        ['CSH1','STAT6'],
        ['CSNK1A1','LYZ'],
        ['CSNK1D','METRNL'],
        ['CSNK1D','TG'],
        ['CSNK1G2','LLFOS-48D6.2'],
        ['CSNK1G2','OAZ1'],
        ['CSRP1','MSLN'],
        ['CSRP1','SMR3B'],
        ['CST1','MUC7'],
        ['CST1','PRH1'],
        ['CST1','PRH1-PRR4'],
        ['CST1','PRH2'],
        ['CST1','PRR4'],
        ['CST1','SMR3B'],
        ['CST2','MUC7'],
        ['CST2','PRH1'],
        ['CST2','PRH1-PRR4'],
        ['CST2','PRH2'],
        ['CST2','PRR4'],
        ['CST2','SMR3B'],
        ['CST3','PRB3'],
        ['CST3','RP11-218C14.5'],
        ['CST3','SMR3B'],
        ['CST4','MUC7'],
        ['CST4','PRH1'],
        ['CST4','PRH1-PRR4'],
        ['CST4','PRH2'],
        ['CST4','PRR4'],
        ['CST4','SMR3B'],
        ['CST5','MUC7'],
        ['CST5','PRH2'],
        ['CST5','SMR3B'],
        ['CSTA','FGD5-AS1'],
        ['CSTB','KRT5'],
        ['CSTF3-AS1','PIGR'],
        ['CSTF3-AS1','RP1-89D4.1'],
        ['CSTL1','RP3-333B15.5'],
        ['CTA-109P11.4','STAM'],
        ['CTA-253N17.1','RPL22'],
        ['CTB-11I22.1','UBLCP1'],
        ['CTB-11I22.2','UBLCP1'],
        ['CTB-120L21.1','ITK'],
        ['CTB-129P6.4','CTC-458I2.2'],
        ['CTB-131B5.5','IGIP'],
        ['CTB-14A14.2','TMTC4'],
        ['CTB-164N12.1','CTB-32H22.1'],
        ['CTB-175E5.7','IGF2'],
        ['CTB-50L17.14','LYZ'],
        ['CTB-60B18.10','NTF4'],
        ['CTB-60B18.6','GINM1'],
        ['CTB-78H18.1','SMIM23'],
        ['CTC-235G5.2','SMS'],
        ['CTC-251I16.1','HRH2'],
        ['CTC-321K16.1','KRT10'],
        ['CTC-321K16.1','KRT14'],
        ['CTC-321K16.1','RP11-119J18.1'],
        ['CTC-338M12.2','TRIM7'],
        ['CTC-338M12.4','STX18-AS1'],
        ['CTC-340A15.2','TMED8'],
        ['CTC-347C20.1','CTD-2631K10.1'],
        ['CTC-360J11.4','ZNF264'],
        ['CTC-463A16.1','NDFIP1'],
        ['CTC-467M3.1','IGF2'],
        ['CTC-479C5.12','PRSS1'],
        ['CTC-479C5.12','PRSS2'],
        ['CTC-479C5.12','TRB@'],
        ['CTC-479C5.12','TRBC2'],
        ['CTC-479C5.12','TRBV25-1'],
        ['CTC-487M23.8','IGFBP4'],
        ['CTC-487M23.8','MUC15'],
        ['CTC-487M23.8','RP11-326C3.16'],
        ['CTC-510F12.4','TMEM205'],
        ['CTC-523E23.4','ZNF599'],
        ['CTC-554D6.1','RP11-326C3.16'],
        ['CTD-2020K17.1','RP13-890H12.2'],
        ['CTD-2031P19.5','TG'],
        ['CTD-2118P12.1','CTD-2134P3.2'],
        ['CTD-2140B24.6','ZNF10'],
        ['CTD-2184D3.5','DGKH'],
        ['CTD-2215E18.1','LINC01340'],
        ['CTD-2231E14.4','LINC00905'],
        ['CTD-2231E14.8','CYP4F3'],
        ['CTD-2231H16.1','RP11-811I15.1'],
        ['CTD-2267D19.3','TTLL3'],
        ['CTD-2270P14.5','TIGD4'],
        ['CTD-2291D10.2','ZNF728'],
        ['CTD-2306A12.1','SQRDL'],
        ['CTD-2308N23.2','S100A9'],
        ['CTD-2315A10.1','LINC01500'],
        ['CTD-2342J14.6','RAB3D'],
        ['CTD-2342J14.6','TMEM205'],
        ['CTD-2528L19.4','ZNF573'],
        ['CTD-2535I10.1','SMR3B'],
        ['CTD-2540M10.1','TNRC6A'],
        ['CTD-2542C24.8','ZNF85'],
        ['CTD-2545G14.7','RP1-4G17.5'],
        ['CTD-2545H1.2','SERINC3'],
        ['CTD-2568P8.1','PPM1A'],
        ['CTD-2583A14.10','SDCBP2-AS1'],
        ['CTD-2583A14.9','TEX41'],
        ['CTD-2616J11.11','NKG7'],
        ['CTD-3065J16.9','OPLAH'],
        ['CTD-3074O7.2','CTSF'],
        ['CTD-3187F8.14','CTD-3187F8.7'],
        ['CTD-3214H19.4','LYZ'],
        ['CTD-3222D19.2','FDPSP7'],
        ['CTD-3222D19.2','LYZ'],
        ['CTD-3233P19.7','ZNF738'],
        ['CTGF','IGHG1'],
        ['CTGF','IGHM'],
        ['CTGF','IGK@'],
        ['CTGF','IGKC'],
        ['CTGF','IGKV1-8'],
        ['CTGF','IGKV3-11'],
        ['CTIF','SLC25A3'],
        ['CTNNA3','KRT8'],
        ['CTRB1','GP2'],
        ['CTRB1','PLA2G1B'],
        ['CTRB1','PNLIP'],
        ['CTRB1','PNLIPRP2'],
        ['CTRB1','PRSS1'],
        ['CTRB1','PRSS2'],
        ['CTRB1','PRSS3'],
        ['CTRB1','TRB@'],
        ['CTRB1','UBE2R2-AS1'],
        ['CTRB2','PNLIP'],
        ['CTRB2','PRSS1'],
        ['CTRB2','PRSS2'],
        ['CTRB2','PRSS3'],
        ['CTRB2','TRB@'],
        ['CTRB2','UBE2R2-AS1'],
        ['CTRC','GP2'],
        ['CTRC','PNLIP'],
        ['CTRC','PRSS1'],
        ['CTRC','PRSS2'],
        ['CTRC','PRSS3'],
        ['CTRC','TRB@'],
        ['CTRC','UBE2R2-AS1'],
        ['CTRL','PRSS1'],
        ['CTRL','PRSS2'],
        ['CTRL','TRB@'],
        ['CTSB','GPD2'],
        ['CTSB','GTF3C1'],
        ['CTSB','IYD'],
        ['CTSB','MT1A'],
        ['CTSB','NEAT1'],
        ['CTSB','P4HB'],
        ['CTSB','PPP1R36'],
        ['CTSB','RP11-479O9.4'],
        ['CTSB','RP11-573D15.8'],
        ['CTSB','RP11-983P16.4'],
        ['CTSB','TMED10'],
        ['CTSB','WWP2'],
        ['CTSD','IGF2'],
        ['CTSD','SMR3B'],
        ['CTSE','DPCR1'],
        ['CTSE','MUC6'],
        ['CTSG','HOOK2'],
        ['CTSG','IGH@'],
        ['CTSG','RP11-1143G9.4'],
        ['CTSG','S100A9'],
        ['CTSS','HORMAD1'],
        ['CTSS','IGH@'],
        ['CTSS','P2RX7'],
        ['CTSS','TBXAS1'],
        ['CTSS','VPRBP'],
        ['CUBN','RSU1'],
        ['CUL4B','EMP2'],
        ['CUZD1','PNLIP'],
        ['CWC25','LYZ'],
        ['CWC25','PIP4K2B'],
        ['CXCL14','TMEM99'],
        ['CXCR4','PTP4A1'],
        ['CXCR4','YPEL5'],
        ['CYB5R2','OVCH2'],
        ['CYB5R3','POLDIP3'],
        ['CYB5R4','RP4-676J13.2'],
        ['CYGB','ST6GALNAC2'],
        ['CYP11B1','DLK1'],
        ['CYP17A1-AS1','WBP1L'],
        ['CYP27A1','TTLL4'],
        ['CYP2B6','RP11-696N14.1'],
        ['CYP2C8','RP11-310E22.5'],
        ['CYP2C8','SERPINA1'],
        ['CYP2U1','HADH'],
        ['CYP3A4','SEPP1'],
        ['CYP3A5','TATDN2P2'],
        ['CYP4F3','LA16C-83F12.6'],
        ['CYP4F3','PLCH1'],
        ['CYP4F3','QRFPR'],
        ['CYP4F3','RSPH14'],
        ['CYP4F3','ZNF536'],
        ['CYP4F32P','SNX18P14'],
        ['CYR61','IGHG1'],
        ['DAB2','ENAM'],
        ['DAB2','MEG3'],
        ['DAB2IP','LYZ'],
        ['DAGLB','PNLIP'],
        ['DAP','HBB'],
        ['DBNDD2','SYS1'],
        ['DBT','RTCA-AS1'],
        ['DCAF8','RP11-536C5.7'],
        ['DCN','IGF2'],
        ['DCTN4','RP11-1143G9.4'],
        ['DDA1','MRPL34'],
        ['DDA1','RP11-857B24.1'],
        ['DDOST','LYZ'],
        ['DDOST','TTL'],
        ['DDR1-AS1','LINC00243'],
        ['DDX17','KHDRBS3'],
        ['DDX17','LYZ'],
        ['DDX17','UBC'],
        ['DDX21','KIF1BP'],
        ['DDX24','HBB'],
        ['DDX39B','LYZ'],
        ['DDX39B','XXBAC-BPG299F13.16'],
        ['DDX3X','NPPA'],
        ['DDX3Y','HBB'],
        ['DDX5','HBB'],
        ['DDX5','POLG2'],
        ['DDX5','PTP4A1'],
        ['DDX5','RP11-20I23.1'],
        ['DEDD','RP11-297K8.2'],
        ['DEF6','PPARD'],
        ['DEFA4','DEFA8P'],
        ['DEFA4','HBB'],
        ['DEFB122','VCL'],
        ['DENND3','HBB'],
        ['DENND4A','LYZ'],
        ['DENND6A-AS1','PDE12'],
        ['DEPDC1B','PSMD12'],
        ['DEPDC5','LYZ'],
        ['DERL2','DHX33'],
        ['DES','DLGAP4-AS1'],
        ['DES','DUSP3'],
        ['DES','EEF1A2'],
        ['DES','FKBP8'],
        ['DES','FLG-AS1'],
        ['DES','HHATL'],
        ['DES','IGH@'],
        ['DES','KLK2'],
        ['DES','MB'],
        ['DES','NPPA'],
        ['DES','NPPA-AS1'],
        ['DES','NPPB'],
        ['DES','OGN'],
        ['DES','RP1-178F15.4'],
        ['DES','RP1-178F15.5'],
        ['DES','RP5-857K21.4'],
        ['DES','S100A13'],
        ['DES','S100A9'],
        ['DES','SRL'],
        ['DES','SYNPO'],
        ['DES','TIMP3'],
        ['DES','TNNI3'],
        ['DES','TNNT2'],
        ['DEXI','FAM102B'],
        ['DFNA5','OSBPL3'],
        ['DGAT2','RN7SL786P'],
        ['DGKD','HBB'],
        ['DGKG','IGF2'],
        ['DHCR24','LYZ'],
        ['DHCR24','STAR'],
        ['DHRS12','LINC00282'],
        ['DHRSX','RP11-617F23.1'],
        ['DHRSX','SORBS2'],
        ['DHTKD1','LYZ'],
        ['DHX9','NPL'],
        ['DISP2','KNSTRN'],
        ['DIXDC1','DLAT'],
        ['DKK3','MYH6'],
        ['DKK3','MYL4'],
        ['DLG2','PIGR'],
        ['DLGAP4','ITGAM'],
        ['DLGAP4-AS1','FLG-AS1'],
        ['DLGAP4-AS1','KRT13'],
        ['DLGAP4-AS1','KRT4'],
        ['DLGAP4-AS1','MYH11'],
        ['DLGAP4-AS1','MYL12A'],
        ['DLGAP4-AS1','NDE1'],
        ['DLGAP4-AS1','TPM2'],
        ['DLGAP4-AS1','WWTR1'],
        ['DLK1','MEG3'],
        ['DMBT1','LYZ'],
        ['DMBT1','PRB1'],
        ['DMBT1','PRH1'],
        ['DMBT1','PRH1-PRR4'],
        ['DMBT1','PRR4'],
        ['DMD','STATH'],
        ['DMTN','LYZ'],
        ['DMWD','RSPH6A'],
        ['DNAAF1','LINC00470'],
        ['DNAAF5','LYZ'],
        ['DNAJC15','LINC00400'],
        ['DNAJC3','HTN3'],
        ['DNASE2','KLF1'],
        ['DNASE2B','SAMD13'],
        ['DNER','SMR3B'],
        ['DNM1L','LYZ'],
        ['DNM1L','STX3'],
        ['DNM2','ZG16B'],
        ['DNMT1','HBB'],
        ['DOCK1','FUS'],
        ['DOCK9','IGF2'],
        ['DOCK9','SMR3B'],
        ['DOPEY2','LYZ'],
        ['DOT1L','LLFOS-48D6.2'],
        ['DOT1L','OAZ1'],
        ['DPCR1','IGK@'],
        ['DPCR1','IGKC'],
        ['DPCR1','IGKV1-8'],
        ['DPCR1','IGKV3-11'],
        ['DPCR1','MUC6'],
        ['DPH6-AS1','TAP2'],
        ['DPH6-AS1','XXBAC-BPG246D15.9'],
        ['DPP6','PABPC1'],
        ['DPY19L1','KCNK3'],
        ['DPY19L2P2','NAPEPLD'],
        ['DPY19L4','INTS8'],
        ['DRAIC','PCAT29'],
        ['DRG1','TG'],
        ['DRICH1','KB-208E9.1'],
        ['DROSHA','TG'],
        ['DSC1','KRT1'],
        ['DSC3','RFX4'],
        ['DSC3','TMEM99'],
        ['DSCAML1','FXYD6-FXYD2'],
        ['DSCAS','KRT10'],
        ['DSCC1','KB-1471A8.1'],
        ['DSE','RP11-274B18.4'],
        ['DSE','TG'],
        ['DSG2','LYZ'],
        ['DSG2','PRH1'],
        ['DSG2','PRH1-PRR4'],
        ['DSG3','IL1RAPL1'],
        ['DSG3','NXN'],
        ['DSP','KRT13'],
        ['DSP','MUC7'],
        ['DSP','TMEM99'],
        ['DTD2','GRB2'],
        ['DUS1L','FASN'],
        ['DUS3L','FUT6'],
        ['DUSP1','HBB'],
        ['DUSP11','TPRKB'],
        ['DUX4L26','DUX4L50'],
        ['DUX4L50','LINC01410'],
        ['DYDC1','MAT1A'],
        ['DYNC1H1','HBB'],
        ['DYNLT1','TATDN2P2'],
        ['DYRK4','RAD51AP1'],
        ['DYX1C1-CCPG1','RAB27A'],
        ['E2F1','NECAB3'],
        ['EBF4','RP4-686C3.7'],
        ['ECE1','LYZ'],
        ['ECH1','HNRNPL'],
        ['ECHDC2','GP2'],
        ['ECHDC2','PNLIP'],
        ['ECM1','KRT13'],
        ['EEF1D','PARK2'],
        ['EEF1D','ZC3H3'],
        ['EEF2','EGR1'],
        ['EEF2','GP2'],
        ['EEF2','HBB'],
        ['EEF2','KRT14'],
        ['EEF2','MUC7'],
        ['EEF2','NDE1'],
        ['EEF2','PIGR'],
        ['EEF2','PRB3'],
        ['EEF2','PRSS1'],
        ['EEF2','PRSS2'],
        ['EEF2','TRB@'],
        ['EFCAB11','SYNC'],
        ['EFCAB14','SFTPC'],
        ['EFCAB3','RP11-212P7.3'],
        ['EFCAB5','NSRP1'],
        ['EGFR','IGF2'],
        ['EGR1','KRT13'],
        ['EGR1','MUC6'],
        ['EGR1','SERPINA1'],
        ['EGR1','SPRR1B'],
        ['EGR2','TG'],
        ['EHBP1L1','FAM89B'],
        ['EHBP1L1','SSSCA1'],
        ['EHD1','SF1'],
        ['EHMT2','SLC44A4'],
        ['EI24','UFM1'],
        ['EIF1','LYZ'],
        ['EIF1','RP11-1143G9.4'],
        ['EIF2B5','KRT1'],
        ['EIF2B5','LYZ'],
        ['EIF2B5','MUC7'],
        ['EIF2B5','MYH7'],
        ['EIF2B5','NPPA-AS1'],
        ['EIF2B5','PRH1'],
        ['EIF2B5','PRH1-PRR4'],
        ['EIF2B5','PRR4'],
        ['EIF2B5','SFTPB'],
        ['EIF2B5','SMR3B'],
        ['EIF2B5','ST3GAL1'],
        ['EIF2B5','TG'],
        ['EIF2D','SFTPC'],
        ['EIF2S1','TG'],
        ['EIF3E','PRB3'],
        ['EIF3E','PRSS1'],
        ['EIF3E','RAB8A'],
        ['EIF3E','SMR3B'],
        ['EIF3E','TG'],
        ['EIF3H','HSP90AA1'],
        ['EIF3L','RP11-666E17.1'],
        ['EIF4E','TMEM30B'],
        ['EIF4G1','KRT13'],
        ['ELANE','HBB'],
        ['ELANE','KBTBD11'],
        ['ELANE','PTBP1'],
        ['ELANE','S100A9'],
        ['ELAVL1','HBB'],
        ['ELAVL1','MUC7'],
        ['ELAVL1','PRB1'],
        ['ELAVL1','SMR3B'],
        ['ELAVL1','TIMM44'],
        ['ELAVL1','TRB@'],
        ['ELF1','TG'],
        ['ELF3','RNPEP'],
        ['ELF5','LYZ'],
        ['ELK1','IGH@'],
        ['ELK2BP','IGH@'],
        ['ELK4','SLC45A3'],
        ['ELL2','PNLIP'],
        ['ELMO1','MPO'],
        ['ELMO1','PNLIP'],
        ['ELN','TNRC18'],
        ['ELP2','GLTPD2'],
        ['EMB','WWC1'],
        ['EMC3','PRRT3'],
        ['EMP1','EVPL'],
        ['EMP2','PTCD1'],
        ['ENAM','IGFBP5'],
        ['ENAM','KLF6'],
        ['ENAM','LYZ'],
        ['ENAM','PIGR'],
        ['ENAM','ZG16B'],
        ['ENO1','LYZ'],
        ['ENO3','NPPA-AS1'],
        ['ENPP5','RP1-8B1.4'],
        ['ENSA','MCL1'],
        ['ENTPD1','TG'],
        ['ENTPD1-AS1','TCTN3'],
        ['ENTPD4','LYZ'],
        ['ENTPD6','MUC7'],
        ['EOGT','TMF1'],
        ['EPAS1','IGF2'],
        ['EPAS1','TMEM247'],
        ['EPB41L1','RP4-550H1.4'],
        ['EPCAM','TG'],
        ['EPCAM','TIAL1'],
        ['EPDR1','IGFBP5'],
        ['EPG5','LYZ'],
        ['EPG5','PSTPIP2'],
        ['EPHX2','GULOP'],
        ['EPRS','LYZ'],
        ['EPS15L1','IGFBP5'],
        ['EPSTI1','LYZ'],
        ['EQTN','MOB3B'],
        ['ERAP1','PNLIP'],
        ['ERBB2','GRB7'],
        ['ERCC1','MYH11'],
        ['ERCC1','S100A9'],
        ['ERGIC1','SMR3B'],
        ['ERGIC2','RP11-996F15.2'],
        ['ERGIC3','ZG16B'],
        ['ERLIN2','RP11-863K10.2'],
        ['ERN1','TEX2'],
        ['ERP27','GP2'],
        ['ERVK3-1','ZSCAN22'],
        ['ERVW-1','PEX1'],
        ['ESD','HTR2A'],
        ['ESRP2','RP11-96D1.5'],
        ['ESRRAP2','TPTE2'],
        ['ESRRG','SMR3B'],
        ['ESYT1','PA2G4'],
        ['ESYT1','RP11-603J24.9'],
        ['ESYT2','LYZ'],
        ['ETF1','HSPA9'],
        ['ETFB','NKG7'],
        ['ETHE1','XRCC1'],
        ['EVPL','TG'],
        ['EVPLL','TNFRSF13B'],
        ['EXD1','RP11-492E3.2'],
        ['EXO1','WDR64'],
        ['EXOC6','HHEX'],
        ['EYS','LYZ'],
        ['EZH1','TG'],
        ['F11-AS1','SLC25A5'],
        ['F11R','RBM22P2'],
        ['F11R','TSTD1'],
        ['F13A1','IGF2'],
        ['F3','PIN1'],
        ['FABP1','FGA'],
        ['FABP3','NPPA'],
        ['FABP6','WSB1'],
        ['FAM107A','FAM3D'],
        ['FAM107B','LYZ'],
        ['FAM111A','PROX2'],
        ['FAM117A','SLC35B1'],
        ['FAM129A','HMCN2'],
        ['FAM129A','MYL2'],
        ['FAM129A','PNLIP'],
        ['FAM129A','RP11-72L22.1'],
        ['FAM129B','KRT13'],
        ['FAM129B','PRSS2'],
        ['FAM129C','MMP23B'],
        ['FAM134B','ZNF622'],
        ['FAM13A','PRH1'],
        ['FAM13A','PRH1-PRR4'],
        ['FAM13A','PRR4'],
        ['FAM149A','TLR3'],
        ['FAM163B','LL09NC01-254D11.1'],
        ['FAM167A-AS1','IYD'],
        ['FAM167A-AS1','TG'],
        ['FAM170B','VSTM4'],
        ['FAM172A','KIAA0825'],
        ['FAM193B','LYZ'],
        ['FAM198A','KRBOX1'],
        ['FAM205BP','FAM205C'],
        ['FAM209B','RTFDC1'],
        ['FAM210B','LYZ'],
        ['FAM221B','HINT2'],
        ['FAM227A','RP3-508I15.9'],
        ['FAM228B','RP1-167A14.2'],
        ['FAM230B','SNX1'],
        ['FAM49B','MPO'],
        ['FAM49B','ZFHX3'],
        ['FAM53C','KDM3B'],
        ['FAM71D','MPP5'],
        ['FAM73A','NEXN'],
        ['FAM8A1','SUMO2P13'],
        ['FAR1','RP11-98J9.1'],
        ['FBLIM1','PDGFRA'],
        ['FBLN1','IGF2'],
        ['FBLN1','MYH11'],
        ['FBLN1','NDE1'],
        ['FBLN1','SERPINH1'],
        ['FBRS','MUC7'],
        ['FBXL2','IGF2'],
        ['FBXL22','USP3'],
        ['FBXL5','METTL16'],
        ['FBXL5','RP11-799M12.2'],
        ['FBXO11','LYZ'],
        ['FBXO16','STOM'],
        ['FBXO21','NOS1'],
        ['FCAR','KIAA1211L'],
        ['FCAR','NRF1'],
        ['FCER1G','TOMM40L'],
        ['FCGBP','PAX8-AS1'],
        ['FCGBP','TPO'],
        ['FDFT1','NEIL2'],
        ['FDFT1','SUB1'],
        ['FDPSP7','SLC35E1'],
        ['FDX1','RP11-347E10.1'],
        ['FER','RP11-1143G9.4'],
        ['FER1L4','IGHG1'],
        ['FEZ2','IGF2'],
        ['FFAR4','RP11-573D15.8'],
        ['FFAR4','SERPINA1'],
        ['FGA','RP11-573D15.8'],
        ['FGA','SARM1'],
        ['FGA','TXNL4B'],
        ['FGB','RP11-573D15.8'],
        ['FGD3','LYZ'],
        ['FGD5-AS1','RCL1'],
        ['FGF14','IGF2'],
        ['FGFR1','PPIC'],
        ['FGFR4','ZNF346'],
        ['FGFRL1','IGF2'],
        ['FGG','RP11-573D15.8'],
        ['FGG','RP11-696N14.1'],
        ['FGG','TXNL4B'],
        ['FHL3','SF3A3'],
        ['FKBP15','LYZ'],
        ['FKBP15','PRB3'],
        ['FKRP','IGF2'],
        ['FLG-AS1','KRT1'],
        ['FLG-AS1','KRT13'],
        ['FLG-AS1','KRT2'],
        ['FLG-AS1','KRT4'],
        ['FLG-AS1','KRT6A'],
        ['FLG-AS1','NUCKS1'],
        ['FLG-AS1','PROM2'],
        ['FLG-AS1','RNF213'],
        ['FLG-AS1','RP5-884C9.2'],
        ['FLG-AS1','S100A2'],
        ['FLG-AS1','SBSN'],
        ['FLG-AS1','TGM3'],
        ['FLG-AS1','VAMP2'],
        ['FLG2','KRT5'],
        ['FLG2','SGPL1'],
        ['FLG2','TSIX'],
        ['FLG2','TTC39C'],
        ['FLJ22447','PRKCH'],
        ['FLJ37505','LINC00507'],
        ['FLJ46284','TRIQK'],
        ['FLNA','HBB'],
        ['FLNA','IGF2'],
        ['FLNA','KRT4'],
        ['FLNA','LYZ'],
        ['FLNA','NPPA'],
        ['FLNA','PRSS2'],
        ['FLNA','RP11-309L24.2'],
        ['FLOT1','HTN3'],
        ['FLOT2','NPPA-AS1'],
        ['FLT1','RP11-27M24.1'],
        ['FMNL2','LYZ'],
        ['FN1','MCAM'],
        ['FN1','PECAM1'],
        ['FNBP1','LYZ'],
        ['FNDC3A','RP11-203I16.7'],
        ['FOS','KLF6'],
        ['FOS','KRT13'],
        ['FOS','MMP8'],
        ['FOSB','RGCC'],
        ['FOSL2','SMR3B'],
        ['FOXC1','PRH1'],
        ['FOXC1','PRH1-PRR4'],
        ['FOXC1','PRR4'],
        ['FOXJ1','RNF157'],
        ['FOXJ3','TRB@'],
        ['FOXP1','HBB'],
        ['FOXP1','PRSS1'],
        ['FPGS','LYZ'],
        ['FRG1HP','RP11-88I18.2'],
        ['FRG1JP','RP11-87H9.4'],
        ['FRMD6-AS2','RP11-255G12.3'],
        ['FRRS1L','RP11-446E24.4'],
        ['FRS2','ZNF529-AS1'],
        ['FRS3','RBBP4'],
        ['FSTL1','TG'],
        ['FTH1','HBB'],
        ['FTH1','PDZD2'],
        ['FTL','S100A9'],
        ['FTO','RP11-1143G9.4'],
        ['FTX','KRT10'],
        ['FTX','PAQR3'],
        ['FURIN','SMR3B'],
        ['FURIN','TG'],
        ['FUT2','SEC1P'],
        ['FUT8','RP11-762H8.4'],
        ['FXYD2','SMR3B'],
        ['GAB1','SMARCA5'],
        ['GABARAP','NPPA-AS1'],
        ['GABARAPL1','NPPA-AS1'],
        ['GABARAPL2','NPPA-AS1'],
        ['GABPB1-AS1','GBP6'],
        ['GABPB2','MLLT11'],
        ['GABRR2','UBE2J1'],
        ['GAK','ZG16B'],
        ['GAL3ST1','PES1'],
        ['GAPDH','RMDN3'],
        ['GAPLINC','TGIF1'],
        ['GAREM','IGH@'],
        ['GAS6','TG'],
        ['GAST','LIPF'],
        ['GAST','PGC'],
        ['GATA2-AS1','TMED10P2'],
        ['GATAD2A','LYZ'],
        ['GATM','GP2'],
        ['GBGT1','RALGDS'],
        ['GBP6','LRP11'],
        ['GBP6','LRP1B'],
        ['GBP6','MYH11'],
        ['GBP6','NRG1'],
        ['GBP6','PTPN12'],
        ['GBP6','RYR2'],
        ['GBP6','SLTM'],
        ['GCA','LYZ'],
        ['GCLC','YES1'],
        ['GCSAM','SLC9C1'],
        ['GDPGP1','RP11-697E2.6'],
        ['GDPGP1','TTLL13P'],
        ['GFAP','MEG3'],
        ['GFAP','PLP1'],
        ['GFAP','RP11-862L9.3'],
        ['GFAP','SERINC1'],
        ['GFM1','MFSD1'],
        ['GFPT2','SIRT3'],
        ['GGACT','TMTC4'],
        ['GGNBP2','LYZ'],
        ['GGT6','IGH@'],
        ['GIMAP8','LINC00996'],
        ['GINM1','LYZ'],
        ['GIPC2','MYADM'],
        ['GK5','LYZ'],
        ['GK5','XRN1'],
        ['GKAP1','KIF27'],
        ['GKN1','IGH@'],
        ['GKN1','IGHA1'],
        ['GKN1','IGK@'],
        ['GKN1','IGKC'],
        ['GKN1','LYZ'],
        ['GKN2','LYZ'],
        ['GKN2','PGC'],
        ['GKN2','TFF2'],
        ['GLIS2','IGF2'],
        ['GLRX','TG'],
        ['GLT8D1','TG'],
        ['GLTP','IDE'],
        ['GLTP','LYZ'],
        ['GLTP','MAP2K5'],
        ['GLTP','MRGPRX4'],
        ['GLTP','POLE2'],
        ['GLTP','TRAPPC9'],
        ['GLUL','IGFBP5'],
        ['GLUL','S100A9'],
        ['GMCL1','SNRNP27'],
        ['GMPR','SPTLC2'],
        ['GNAI2','HBB'],
        ['GNAI2','SFTPB'],
        ['GNAS','KRT13'],
        ['GNAS','MUC7'],
        ['GNAS','PRB3'],
        ['GNAS','PRH1'],
        ['GNAS','PRH1-PRR4'],
        ['GNAS','PRH2'],
        ['GNAS','PRR4'],
        ['GNAS','SLA'],
        ['GNAS','TRB@'],
        ['GNE','LYZ'],
        ['GNE','PRH1'],
        ['GNG12','SMR3B'],
        ['GNG12-AS1','PPP4R3A'],
        ['GNG7','RAB21'],
        ['GNMT','RP3-475N16.1'],
        ['GOLGA2P8','ZNF774'],
        ['GOLGA4','TG'],
        ['GOLGA4','TJP1'],
        ['GOLGA6L2','GOLGA6L7P'],
        ['GOLT1A','KISS1'],
        ['GORASP2','PRH1'],
        ['GORASP2','PRH1-PRR4'],
        ['GORASP2','PRR4'],
        ['GP2','GPT2'],
        ['GP2','LMNA'],
        ['GP2','NPHP3-ACAD11'],
        ['GP2','NTN4'],
        ['GP2','NUPR1'],
        ['GP2','P4HB'],
        ['GP2','PLA2G1B'],
        ['GP2','PNLIP'],
        ['GP2','PNLIPRP2'],
        ['GP2','PRSS1'],
        ['GP2','PRSS2'],
        ['GP2','PRSS3'],
        ['GP2','RP11-680B3.2'],
        ['GP2','SEL1L'],
        ['GP2','SETD2'],
        ['GP2','SLC4A4'],
        ['GP2','TEX11'],
        ['GP2','TRB@'],
        ['GP2','UBE2R2-AS1'],
        ['GPM6B','PIGR'],
        ['GPR107','TG'],
        ['GPR141','NME8'],
        ['GPR65','LINC01146'],
        ['GPX3','HNF4A'],
        ['GPX3','RP11-807H22.7'],
        ['GRAP2','TMA7'],
        ['GRB2','LYZ'],
        ['GRB2','RNF130'],
        ['GRB2','RP11-176H8.1'],
        ['GREB1','MARK4'],
        ['GREB1','PCMT1'],
        ['GREB1','RP11-397A16.1'],
        ['GREM2','RP11-1143G9.4'],
        ['GRHL1','RP11-95D17.1'],
        ['GRID1-AS1','RP11-93H12.4'],
        ['GRIN2D','KDELR1'],
        ['GRIP1','RP11-123O10.3'],
        ['GRK6','PRR7'],
        ['GRN','LYZ'],
        ['GS1-259H13.11','LYZ'],
        ['GS1-259H13.2','ZNF655'],
        ['GS1-259H13.2','ZSCAN25'],
        ['GS1-304P7.3','RP11-295K2.3'],
        ['GSDMB','ORMDL3'],
        ['GSN','KRT13'],
        ['GSN','LYZ'],
        ['GSN','NPPA'],
        ['GSN','PRSS1'],
        ['GSN','PRSS2'],
        ['GSN','TRBC2'],
        ['GSN','TRBV25-1'],
        ['GSN','ZNF106'],
        ['GSTP1','PIGR'],
        ['GTF2IP7','LINC00174'],
        ['GTPBP1','TG'],
        ['GUCY2C','PLBD1'],
        ['GUSB','RP5-1132H15.3'],
        ['GUSBP4','LINC00680'],
        ['GYG2','XG'],
        ['GYS1','MYL2'],
        ['H1F0','LYZ'],
        ['H1F0','S100A9'],
        ['H1FX-AS1','S100A8'],
        ['H2AFJ','HBA1'],
        ['H2AFV','HMGCL'],
        ['H2AFV','RNF128'],
        ['H3F3A','LYZ'],
        ['H3F3B','LYZ'],
        ['H3F3B','MPO'],
        ['H6PD','SMR3B'],
        ['HACD3','IGF2'],
        ['HADH','SGMS2'],
        ['HADHA','HBB'],
        ['HAND2-AS1','NPPA-AS1'],
        ['HAO2','RP5-834N19.1'],
        ['HARS','RP11-259G18.3'],
        ['HAS2-AS1','RP11-3G20.2'],
        ['HBA2','MPO'],
        ['HBA2','SLC4A1'],
        ['HBB','HIPK3'],
        ['HBB','HLA-A'],
        ['HBB','HLA-C'],
        ['HBB','HLA-DRA'],
        ['HBB','HOOK2'],
        ['HBB','IER2'],
        ['HBB','IGH@'],
        ['HBB','IGK@'],
        ['HBB','IGKC'],
        ['HBB','KDM3A'],
        ['HBB','LAPTM5'],
        ['HBB','LMNA'],
        ['HBB','LTF'],
        ['HBB','MBNL3'],
        ['HBB','MCL1'],
        ['HBB','MPO'],
        ['HBB','NCOA6'],
        ['HBB','NUDT4'],
        ['HBB','PDE4B'],
        ['HBB','PINK1'],
        ['HBB','PKM'],
        ['HBB','PLBD1'],
        ['HBB','PNP'],
        ['HBB','POLDIP2'],
        ['HBB','PRTN3'],
        ['HBB','RP11-1334A24.6'],
        ['HBB','RP11-138I1.4'],
        ['HBB','RP11-296A16.1'],
        ['HBB','RP11-404F10.2'],
        ['HBB','RP11-571M6.7'],
        ['HBB','RP11-762H8.4'],
        ['HBB','RP11-968A15.8'],
        ['HBB','RP4-781K5.2'],
        ['HBB','RP5-877J2.1'],
        ['HBB','RP5-940J5.9'],
        ['HBB','RP5-972B16.2'],
        ['HBB','RPL3'],
        ['HBB','RPL37A'],
        ['HBB','RPL8'],
        ['HBB','RPLP1'],
        ['HBB','RPS14'],
        ['HBB','RPS7'],
        ['HBB','RPS8'],
        ['HBB','RRP12'],
        ['HBB','S100A9'],
        ['HBB','SESN3'],
        ['HBB','SLC6A9'],
        ['HBB','SNCA'],
        ['HBB','SNX32'],
        ['HBB','SPPL2B'],
        ['HBB','SPTA1'],
        ['HBB','STAT6'],
        ['HBB','STK35'],
        ['HBB','SULF2'],
        ['HBB','SYNE1'],
        ['HBB','TET3'],
        ['HBB','TMSB4X'],
        ['HBB','TNFAIP3'],
        ['HBB','TPM3'],
        ['HBB','TRAK2'],
        ['HBB','TXNIP'],
        ['HBB','UNC13D'],
        ['HBB','VPS13A'],
        ['HBB','VPS28'],
        ['HBB','XPO6'],
        ['HBB','ZBTB1'],
        ['HBB','ZFAND5'],
        ['HBB','ZWINT'],
        ['HBG2','IGF2'],
        ['HBG2','TRIM5'],
        ['HCG21','IGH@'],
        ['HCG21','IGHG1'],
        ['HCG21','SFTA2'],
        ['HCG23','XXBAC-BPG154L12.5'],
        ['HDAC1','RNF123'],
        ['HDAC2','LYZ'],
        ['HDAC2','NPPA-AS1'],
        ['HDLBP','LYZ'],
        ['HDLBP','PGC'],
        ['HDLBP','PIGR'],
        ['HEATR1','PRSS1'],
        ['HEATR1','PRSS2'],
        ['HEATR1','TRBC2'],
        ['HECTD1','SMR3B'],
        ['HEIH','ZFP62'],
        ['HELLS','MASP2'],
        ['HEMGN','RP11-535C21.3'],
        ['HEPHL1','PANX1'],
        ['HERC1','LYZ'],
        ['HERC1','USP3-AS1'],
        ['HERC2P10','RP11-540B6.3'],
        ['HERC2P3','RP11-603B24.1'],
        ['HERPUD1','SLC12A3'],
        ['HERPUD2','LYZ'],
        ['HGF','IGF2'],
        ['HHIPL2','TAF1A'],
        ['HIF1A','NEK9'],
        ['HIF1A','SF3B1'],
        ['HIF1A-AS2','MYO1F'],
        ['HIF1A-AS2','SFT2D2'],
        ['HIGD1C','METTL7A'],
        ['HILPDA','RCAN3'],
        ['HIP1','IGF2'],
        ['HIPK2','TG'],
        ['HIST1H2AA','SLC17A1'],
        ['HIST1H2APS1','HIST1H2BA'],
        ['HLA-C','LTF'],
        ['HLA-C','LYZ'],
        ['HLA-DRA','TGFBI'],
        ['HLA-E','MPO'],
        ['HLA-E','ZNF638'],
        ['HLX','MARC1'],
        ['HMGB1P20','RP1-34L19.1'],
        ['HMGB2','NCL'],
        ['HMGXB4','TOM1'],
        ['HMHA1','TFE3'],
        ['HMP19','METTL7A'],
        ['HNF1A-AS1','LYZ'],
        ['HNRNPA1','KRT13'],
        ['HNRNPA1P27','XKRX'],
        ['HNRNPA2B1','PTP4A1'],
        ['HNRNPC','LYZ'],
        ['HNRNPD','SMR3B'],
        ['HNRNPDL','RNF128'],
        ['HNRNPH1','KRT4'],
        ['HNRNPH1','LYZ'],
        ['HNRNPK','PRB3'],
        ['HNRNPM','LYZ'],
        ['HNRNPUL2-BSCL2','LYZ'],
        ['HOOK2','HTN3'],
        ['HOOK2','KLK2'],
        ['HOOK2','VIM'],
        ['HOXB7','RP11-357H14.17'],
        ['HOXB9','RP11-357H14.17'],
        ['HP','TF'],
        ['HP1BP3','LYZ'],
        ['HPCAL1','LYZ'],
        ['HPN','TRB@'],
        ['HPS1','LYZ'],
        ['HPS5','SAA2'],
        ['HRG','RBP4'],
        ['HRH4','IMPACT'],
        ['HS2ST1','LINC01140'],
        ['HSF5','RNF43'],
        ['HSP90AA1','LYZ'],
        ['HSP90B1','SYN3'],
        ['HSPA2','ZBTB1'],
        ['HSPA5','TG'],
        ['HSPB7','MYL7'],
        ['HSPB7','NPPA-AS1'],
        ['HSPB8','NPPA'],
        ['HSPE1P22','ZNF512B'],
        ['HSPG2','MYH6'],
        ['HSPG2','TG'],
        ['HTN1','IGH@'],
        ['HTN1','IGK@'],
        ['HTN1','IGKC'],
        ['HTN1','IGKV1-8'],
        ['HTN1','PRB2'],
        ['HTN1','SMR3B'],
        ['HTN3','IGH@'],
        ['HTN3','IGK@'],
        ['HTN3','IGKC'],
        ['HTN3','IGKV1-8'],
        ['HTN3','IGKV3-11'],
        ['HTN3','LYZ'],
        ['HTN3','MAOA'],
        ['HTN3','MCFD2'],
        ['HTN3','NBAS'],
        ['HTN3','PRH1'],
        ['HTN3','PRH1-PRR4'],
        ['HTN3','PRR4'],
        ['HTN3','SLC12A2'],
        ['HTN3','SMR3B'],
        ['HTN3','SSBP2'],
        ['HTN3','TCN1'],
        ['HTN3','TMEM123'],
        ['HTN3','TRIM2'],
        ['HTN3','VPS13B'],
        ['HTRA1','LYZ'],
        ['HUWE1','TG'],
        ['HVCN1','METTL7A'],
        ['HVCN1','PPP1CC'],
        ['HYDIN2','LYZ'],
        ['HYKK','IREB2'],
        ['HYKK','PSMA4'],
        ['HYPK','WSB1'],
        ['IFNAR2','ZNF595'],
        ['IFNG-AS1','LINC01479'],
        ['IFT122','RP11-529F4.1'],
        ['IGF2','INS'],
        ['IGF2','ISM2'],
        ['IGF2','KDM5B'],
        ['IGF2','KIAA0895L'],
        ['IGF2','LAMB1'],
        ['IGF2','LGMN'],
        ['IGF2','LHFP'],
        ['IGF2','MACF1'],
        ['IGF2','MAF'],
        ['IGF2','MED15'],
        ['IGF2','MPEG1'],
        ['IGF2','MPRIP'],
        ['IGF2','MYH9'],
        ['IGF2','NID1'],
        ['IGF2','NPPA-AS1'],
        ['IGF2','OLR1'],
        ['IGF2','OS9'],
        ['IGF2','OSBPL9'],
        ['IGF2','PALM2-AKAP2'],
        ['IGF2','PAPPA'],
        ['IGF2','PCBP2'],
        ['IGF2','PECAM1'],
        ['IGF2','PODN'],
        ['IGF2','PRG2'],
        ['IGF2','PTPRF'],
        ['IGF2','RAB7A'],
        ['IGF2','RAP2C-AS1'],
        ['IGF2','RHOA'],
        ['IGF2','RP11-123M6.2'],
        ['IGF2','RP11-295K3.1'],
        ['IGF2','RP11-537H15.3'],
        ['IGF2','RP11-69L16.4'],
        ['IGF2','RP11-728F11.4'],
        ['IGF2','RP11-779O18.3'],
        ['IGF2','RP11-872D17.8'],
        ['IGF2','RP4-613B23.1'],
        ['IGF2','RP5-972B16.2'],
        ['IGF2','SCARA3'],
        ['IGF2','SCARB1'],
        ['IGF2','SEPP1'],
        ['IGF2','SLC16A3'],
        ['IGF2','SNX24'],
        ['IGF2','SPINT2'],
        ['IGF2','SYN3'],
        ['IGF2','SYNPO'],
        ['IGF2','TIMP2'],
        ['IGF2','TP53I11'],
        ['IGF2','TRIOBP'],
        ['IGF2','TXNL4B'],
        ['IGF2','ZFAT'],
        ['IGF2BP3','RPS6KA2'],
        ['IGFBP4','SERPINA1'],
        ['IGFBP5','NCOR2'],
        ['IGFBP5','NDE1'],
        ['IGFBP7','PRSS2'],
        ['IGFBP7','TRB@'],
        ['IGFBP7-AS1','NPPA'],
        ['IGFBP7-AS1','TNNT2'],
        ['IGH@','JCHAIN'],
        ['IGH@','KRT13'],
        ['IGH@','KRT19'],
        ['IGH@','KRT4'],
        ['IGH@','LCN2'],
        ['IGH@','LCT'],
        ['IGH@','LINC00221'],
        ['IGH@','LYZ'],
        ['IGH@','MEP1B'],
        ['IGH@','MUC13'],
        ['IGH@','MUC5AC'],
        ['IGH@','MUC6'],
        ['IGH@','MUC7'],
        ['IGH@','NCL'],
        ['IGH@','NEAT1'],
        ['IGH@','PGC'],
        ['IGH@','PIGR'],
        ['IGH@','PRB3'],
        ['IGH@','PRH1'],
        ['IGH@','PRH1-PRR4'],
        ['IGH@','PRSS2'],
        ['IGH@','PTPRF'],
        ['IGH@','REG4'],
        ['IGH@','RP11-1166P10.1'],
        ['IGH@','RP11-1166P10.6'],
        ['IGH@','RP11-19N8.2'],
        ['IGH@','RP11-294C11.4'],
        ['IGH@','RP11-302B13.5'],
        ['IGH@','RP11-731F5.2'],
        ['IGH@','RP11-812E19.9'],
        ['IGH@','SFTPB'],
        ['IGH@','SIAE'],
        ['IGH@','SMR3B'],
        ['IGH@','SULT1C2'],
        ['IGH@','TFF2'],
        ['IGH@','TG'],
        ['IGH@','TRB@'],
        ['IGH@','ZG16B'],
        ['IGHA1','MPO'],
        ['IGHA1','MUC6'],
        ['IGHA1','MUC7'],
        ['IGHA1','PGC'],
        ['IGHA1','PIGR'],
        ['IGHA1','PRB1'],
        ['IGHA1','PRB2'],
        ['IGHA1','REG3A'],
        ['IGHA1','SMR3B'],
        ['IGHA2','MUC6'],
        ['IGHA2','MUC7'],
        ['IGHA2','SMR3B'],
        ['IGHG1','LCT'],
        ['IGHG1','LINC01578'],
        ['IGHG1','LIPF'],
        ['IGHG1','MUC7'],
        ['IGHG1','OLFM4'],
        ['IGHG1','PRH1'],
        ['IGHG1','PRH1-PRR4'],
        ['IGHG1','RP11-1143G9.4'],
        ['IGHG1','RP11-731F5.1'],
        ['IGHG3','PRH1'],
        ['IGHG3','PRH1-PRR4'],
        ['IGHM','JCHAIN'],
        ['IGHM','MUC5AC'],
        ['IGHM','MUC7'],
        ['IGHM','PRH1'],
        ['IGHM','PRH1-PRR4'],
        ['IGHM','PTPRF'],
        ['IGHM','RP11-1143G9.4'],
        ['IGHM','RP11-731F5.1'],
        ['IGHV3-49','RP11-731F5.2'],
        ['IGK@','JCHAIN'],
        ['IGK@','LAPTM5'],
        ['IGK@','LYZ'],
        ['IGK@','MUC6'],
        ['IGK@','NDE1'],
        ['IGK@','PGC'],
        ['IGK@','PIGR'],
        ['IGK@','PRH1'],
        ['IGK@','PRH1-PRR4'],
        ['IGK@','PRR4'],
        ['IGK@','RP11-1143G9.4'],
        ['IGK@','SFTPB'],
        ['IGK@','SMR3B'],
        ['IGK@','TFF1'],
        ['IGKC','JCHAIN'],
        ['IGKC','LAPTM5'],
        ['IGKC','LYZ'],
        ['IGKC','MUC6'],
        ['IGKC','MUC7'],
        ['IGKC','NDE1'],
        ['IGKC','PGC'],
        ['IGKC','PIGR'],
        ['IGKC','PRH1'],
        ['IGKC','PRH1-PRR4'],
        ['IGKC','PRR27'],
        ['IGKC','PRR4'],
        ['IGKC','REG3A'],
        ['IGKC','RP11-1143G9.4'],
        ['IGKC','SMR3B'],
        ['IGKC','TFF1'],
        ['IGKC','TMBIM6'],
        ['IGKV1-8','NDRG2'],
        ['IGKV1-8','PIGR'],
        ['IGKV1-8','PRH1'],
        ['IGKV1-8','PRH1-PRR4'],
        ['IGKV1-8','PRR4'],
        ['IGKV1-8','RP11-1143G9.4'],
        ['IGKV1-8','TFF1'],
        ['IGKV3-11','LYZ'],
        ['IGKV3-11','PIGR'],
        ['IGKV3-11','PRH1'],
        ['IGKV3-11','PRH1-PRR4'],
        ['IGKV3-11','PRR4'],
        ['IGKV3-11','RP11-1143G9.4'],
        ['IGKV3-11','TFF1'],
        ['IGLVIV-64','LL22NC03-23C6.13'],
        ['IGLVIV-65','LL22NC03-23C6.13'],
        ['IGLVIV-66-1','LL22NC03-23C6.13'],
        ['IKZF1','LYZ'],
        ['IL18','LITAF'],
        ['IL1RN','RP11-109E10.1'],
        ['IL36A','MRPL11'],
        ['IL6ST','PRH1'],
        ['ILF3','LYZ'],
        ['IMMP2L','PIGR'],
        ['IMP3','PTRF'],
        ['INPP5B','LYZ'],
        ['INPP5D','LYZ'],
        ['INS','TRB@'],
        ['INSL3','JAK3'],
        ['INTS2','MED13'],
        ['INTS6-AS1','WDFY2'],
        ['INTS7','LYZ'],
        ['IQCE','RP11-1143G9.4'],
        ['IQGAP1','LYZ'],
        ['IQGAP1','NSMCE1'],
        ['IQSEC1','RP11-767C1.1'],
        ['IQUB','NDUFA5'],
        ['IRF1','LYZ'],
        ['IRS4','RP6-24A23.7'],
        ['ISPD','SOSTDC1'],
        ['IST1','LYZ'],
        ['IST1','TG'],
        ['ITGA2B','LYZ'],
        ['ITGA3','SFTPB'],
        ['ITGA7','NPPA-AS1'],
        ['ITGAD','RPL37A'],
        ['ITGAL','LYZ'],
        ['ITGAM','LYZ'],
        ['ITGB1','SMR3B'],
        ['ITGB2','SUMO3'],
        ['ITGB4','LYZ'],
        ['ITM2B','LYZ'],
        ['ITM2B','S100A9'],
        ['ITPKB','SMR3B'],
        ['ITPR1','TG'],
        ['ITPR2','STATH'],
        ['IVD','TG'],
        ['IYD','LAMC1'],
        ['IYD','METTL7A'],
        ['IYD','RP11-950C14.7'],
        ['IYD','SLA'],
        ['IYD','TPO'],
        ['JAM3','RP11-713P17.3'],
        ['JCHAIN','LYZ'],
        ['JCHAIN','PIGR'],
        ['JMJD6','SRSF2'],
        ['JUND','KIAA1683'],
        ['JUP','LCE1C'],
        ['JUP','MYH11'],
        ['JUP','TMEM99'],
        ['KANSL2','LYZ'],
        ['KARS','SMR3B'],
        ['KAT8','MYH9'],
        ['KAT8','SNX24'],
        ['KATNAL2','PAX8-AS1'],
        ['KATNAL2','RP11-65I12.1'],
        ['KB-1980E6.3','ODF1'],
        ['KCNA3','RP11-284N8.3'],
        ['KCNE3','ZNF346'],
        ['KCNH8','PRKDC'],
        ['KCNIP3','PROM2'],
        ['KCNIP4','RP4-758J18.10'],
        ['KCNIP4','UBE2V2'],
        ['KCNJ15','TG'],
        ['KCNMA1','KRT13'],
        ['KCNMA1','NDE1'],
        ['KCNMA1','SMR3B'],
        ['KCNN4','MUC7'],
        ['KCNQ1','TG'],
        ['KCP','SENP1'],
        ['KCTD1','PCAT18'],
        ['KCTD14','LARS'],
        ['KCTD14','SMR3B'],
        ['KCTD20','LYZ'],
        ['KCTD7','LYZ'],
        ['KDM2A','LYZ'],
        ['KDM2B','STOM'],
        ['KDM4C','TMEM50B'],
        ['KDM4E','RP11-60C6.3'],
        ['KDM5A','PLD5'],
        ['KDM5A','SLC6A13'],
        ['KDSR','STAG1'],
        ['KIAA0020','SSR3'],
        ['KIAA0319L','LYZ'],
        ['KIAA0355','TRB@'],
        ['KIAA0430','PAX8-AS1'],
        ['KIAA0586','TOMM20L'],
        ['KIAA0753','MED31'],
        ['KIAA0753','PITPNM3'],
        ['KIAA1217','PRB3'],
        ['KIAA1217','TG'],
        ['KIAA1324','PAGE4'],
        ['KIAA1324','PRH1'],
        ['KIAA1324','PRH1-PRR4'],
        ['KIAA1324','PRR4'],
        ['KIAA1522','SMR3B'],
        ['KIAA1841','PEX13'],
        ['KIDINS220','PTH'],
        ['KIF26B','PIGR'],
        ['KIFAP3','TG'],
        ['KIFC3','TG'],
        ['KLC1','SMR3B'],
        ['KLF16','LLFOS-48D6.2'],
        ['KLF16','OAZ1'],
        ['KLF6','TG'],
        ['KLF8','RP13-188A5.1'],
        ['KLHL33','TEP1'],
        ['KLK1','SMR3B'],
        ['KLK2','NDE1'],
        ['KLK2','STAMBPL1'],
        ['KLK3','LLFOS-48D6.2'],
        ['KLK3','MYLK'],
        ['KLK3','NR4A1'],
        ['KLRAP1','MAGOHB'],
        ['KMT2D','S100A8'],
        ['KNOP1P1','TRIM69'],
        ['KPNA1','PARP9'],
        ['KREMEN1','LYZ'],
        ['KRIT1','LRRD1'],
        ['KRR1','RP11-585P4.5'],
        ['KRT1','KRT73-AS1'],
        ['KRT1','LYPD3'],
        ['KRT1','MACF1'],
        ['KRT1','MYH9'],
        ['KRT1','PCGF3'],
        ['KRT1','PERP'],
        ['KRT1','PITPNM3'],
        ['KRT1','POLR2A'],
        ['KRT1','PTPRF'],
        ['KRT1','SBSN'],
        ['KRT1','SRGAP2'],
        ['KRT1','TM6SF1'],
        ['KRT1','TMEM99'],
        ['KRT10','PSORS1C1'],
        ['KRT10','PVRL4'],
        ['KRT128P','KRT73'],
        ['KRT13','KRT78'],
        ['KRT13','MCAM'],
        ['KRT13','MGLL'],
        ['KRT13','MUC4'],
        ['KRT13','MYH11'],
        ['KRT13','MYLK'],
        ['KRT13','NDE1'],
        ['KRT13','PALLD'],
        ['KRT13','PARP9'],
        ['KRT13','PPP1CB'],
        ['KRT13','PRMT2'],
        ['KRT13','RHCG'],
        ['KRT13','RP11-290L1.3'],
        ['KRT13','RP11-779O18.3'],
        ['KRT13','RP11-968A15.8'],
        ['KRT13','S100A14'],
        ['KRT13','S100A9'],
        ['KRT13','SF3B2'],
        ['KRT13','SFN'],
        ['KRT13','SNX32'],
        ['KRT13','SORBS1'],
        ['KRT13','SORT1'],
        ['KRT13','SPARCL1'],
        ['KRT13','SPRR1B'],
        ['KRT13','SREBF2'],
        ['KRT13','SRPK2'],
        ['KRT13','TACSTD2'],
        ['KRT13','TAGLN'],
        ['KRT13','TBCD'],
        ['KRT13','TGFB1'],
        ['KRT13','TMBIM1'],
        ['KRT13','TPM2'],
        ['KRT13','ZBTB4'],
        ['KRT14','TMEM99'],
        ['KRT15','TMEM99'],
        ['KRT18P3','RP11-359G22.2'],
        ['KRT18P59','PKNOX2'],
        ['KRT19','LYZ'],
        ['KRT19','MUC5AC'],
        ['KRT2','SRSF5'],
        ['KRT2','TMEM99'],
        ['KRT20','PIGR'],
        ['KRT23','SMR3B'],
        ['KRT4','LMNA'],
        ['KRT4','MYH11'],
        ['KRT4','NPEPPS'],
        ['KRT4','PERP'],
        ['KRT4','PPP1R12B'],
        ['KRT4','RP11-753H16.3'],
        ['KRT4','RP11-753H16.5'],
        ['KRT4','RP11-779O18.3'],
        ['KRT4','RPS25'],
        ['KRT4','S100A14'],
        ['KRT4','SORBS1'],
        ['KRT4','SPRR3'],
        ['KRT4','SYN3'],
        ['KRT4','SYNPO2'],
        ['KRT4','TNFAIP2'],
        ['KRT4','TRIM29'],
        ['KRT4','ZMIZ1'],
        ['KRT5','LYPD3'],
        ['KRT5','TMEM99'],
        ['KRT6A','RHCG'],
        ['KRT7','TG'],
        ['KRT78','MYH11'],
        ['KRT8','MUC7'],
        ['KRT8P26','RP11-770G2.2'],
        ['KRTAP5-8','NADSYN1'],
        ['KXD1','UBA52'],
        ['L2HGDH','SOS2'],
        ['LAD1','TNNI1'],
        ['LAMA1','TMPO'],
        ['LAMA5','PNLIP'],
        ['LAMP2','LYZ'],
        ['LARP1','MUC7'],
        ['LARS','MUC7'],
        ['LARS','NDUFC2'],
        ['LARS','NDUFC2-KCTD14'],
        ['LARS','TG'],
        ['LBR','LYZ'],
        ['LCLAT1','MLIP'],
        ['LCP1','LYZ'],
        ['LCP1','SRGN'],
        ['LDB1','RP11-862L9.3'],
        ['LDHB','TG'],
        ['LENG8','PAX8-AS1'],
        ['LEPROTL1','LYZ'],
        ['LGALS8','SFTPB'],
        ['LHFPL3','RN7SL8P'],
        ['LILRA1','LILRP2'],
        ['LILRB2','LILRP2'],
        ['LIMCH1','SFTPB'],
        ['LIMK2','LYZ'],
        ['LIN54','SEC31A'],
        ['LIN54','THAP9-AS1'],
        ['LINC00202-1','LINC00264'],
        ['LINC00202-1','LINC00614'],
        ['LINC00452','LINC00453'],
        ['LINC00521','OTUB2'],
        ['LINC00598','RP11-172E9.2'],
        ['LINC00707','RP11-554I8.2'],
        ['LINC00879','RPS18P6'],
        ['LINC00882','RP11-446H18.5'],
        ['LINC00937','RP11-481A20.10'],
        ['LINC00950','TMEM8B'],
        ['LINC00963','SLC12A6'],
        ['LINC00964','ZNF572'],
        ['LINC00969','LYZ'],
        ['LINC00994','PSMD6'],
        ['LINC01057','RP11-86H7.7'],
        ['LINC01091','RP11-381N20.1'],
        ['LINC01091','RP11-381N20.2'],
        ['LINC01128','RP11-63E5.6'],
        ['LINC01227','RP11-109P11.1'],
        ['LINC01278','MUC7'],
        ['LINC01317','LYZ'],
        ['LINC01324','PEX5L'],
        ['LINC01340','RP11-455B3.1'],
        ['LINC01358','RP11-145M4.3'],
        ['LINC01370','RP11-101E14.3'],
        ['LINC01378','SCD5'],
        ['LINC01492','ST6GALNAC2'],
        ['LINC01496','NUDT11'],
        ['LINC01507','RP11-327L9.1'],
        ['LINC01517','RP11-478H13.1'],
        ['LINC01518','ZNF33B'],
        ['LINC01524','RP4-715N11.2'],
        ['LINC01524','ZNF281'],
        ['LINC01605','RP11-527N22.2'],
        ['LINCR-0001','MSRA'],
        ['LIPA','LYZ'],
        ['LIPF','LYZ'],
        ['LIPF','PIGR'],
        ['LIPG','RP11-110H1.8'],
        ['LIX1L','UGT8'],
        ['LL22NC03-2H8.5','RP11-989E6.10'],
        ['LLFOS-48D6.2','SGTA'],
        ['LLGL2','LYZ'],
        ['LMAN1L','TMOD3'],
        ['LMF1','SMR3B'],
        ['LMO7','TG'],
        ['LMO7','UCHL3'],
        ['LMOD2','NPPA-AS1'],
        ['LNP1','TMEM45A'],
        ['LONRF2','RAPH1'],
        ['LPCAT1','SLC6A3'],
        ['LPGAT1','TG'],
        ['LPO','LYZ'],
        ['LPO','PRH1'],
        ['LPO','PRH1-PRR4'],
        ['LPO','PRR4'],
        ['LPO','RP11-1143G9.4'],
        ['LPO','SMR3B'],
        ['LPP','MYH11'],
        ['LPP','RNF128'],
        ['LPP','STXBP4'],
        ['LPP','SUGCT'],
        ['LRG1','LYZ'],
        ['LRIG1','PRSS1'],
        ['LRIG1','PRSS2'],
        ['LRIG1','TRB@'],
        ['LRP10','REM2'],
        ['LRP2','TG'],
        ['LRP4-AS1','LYZ'],
        ['LRP5','NPPA'],
        ['LRP5','TG'],
        ['LRRC14B','PLEKHG4B'],
        ['LRRC56','QRSL1'],
        ['LRRC61','RP4-584D14.7'],
        ['LRRCC1','RP11-146E23.2'],
        ['LRRFIP1','SMR3B'],
        ['LRRK2','MUC19'],
        ['LSG1','RP11-114H23.1'],
        ['LSP1','TNNT3'],
        ['LTA','NFKBIL1'],
        ['LTF','LYZ'],
        ['LTF','NFKBIZ'],
        ['LTF','RBM25'],
        ['LTF','SMR3B'],
        ['LTV1','RP11-718O11.1'],
        ['LYG1','TXNDC9'],
        ['LYRM7','LYZ'],
        ['LYSMD1','SEMA6C'],
        ['LYSMD3','RP1-225E12.2'],
        ['LYSMD3','RP1-225E12.3'],
        ['LYZ','MAFF'],
        ['LYZ','MAGED1'],
        ['LYZ','MAP3K5'],
        ['LYZ','MCM4'],
        ['LYZ','MKNK2'],
        ['LYZ','MLLT6'],
        ['LYZ','MLXIP'],
        ['LYZ','MON1B'],
        ['LYZ','MYH9'],
        ['LYZ','MYO18A'],
        ['LYZ','MYO1A'],
        ['LYZ','MYO1F'],
        ['LYZ','NARF'],
        ['LYZ','NCL'],
        ['LYZ','NCOA6'],
        ['LYZ','NEAT1'],
        ['LYZ','NEBL'],
        ['LYZ','NF1'],
        ['LYZ','NFKBIB'],
        ['LYZ','NFKBIZ'],
        ['LYZ','NONO'],
        ['LYZ','NPC1'],
        ['LYZ','NRAS'],
        ['LYZ','NT5C2'],
        ['LYZ','NXN'],
        ['LYZ','OBSL1'],
        ['LYZ','OPA1'],
        ['LYZ','ORAI2'],
        ['LYZ','OSBPL9'],
        ['LYZ','OTUD5'],
        ['LYZ','P4HB'],
        ['LYZ','PABPC1'],
        ['LYZ','PARP8'],
        ['LYZ','PCBP1-AS1'],
        ['LYZ','PCBP2'],
        ['LYZ','PCDHA1'],
        ['LYZ','PCDHA10'],
        ['LYZ','PCDHA11'],
        ['LYZ','PCDHA12'],
        ['LYZ','PCDHA13'],
        ['LYZ','PCDHA2'],
        ['LYZ','PCDHA3'],
        ['LYZ','PCDHA4'],
        ['LYZ','PCDHA5'],
        ['LYZ','PCDHA7'],
        ['LYZ','PCDHA8'],
        ['LYZ','PCDHA9'],
        ['LYZ','PCDHAC1'],
        ['LYZ','PCDHAC2'],
        ['LYZ','PCYOX1'],
        ['LYZ','PET100'],
        ['LYZ','PGC'],
        ['LYZ','PHACTR4'],
        ['LYZ','PIP'],
        ['LYZ','PITPNA'],
        ['LYZ','PLAUR'],
        ['LYZ','PLEKHS1'],
        ['LYZ','PPP1CB'],
        ['LYZ','PRB1'],
        ['LYZ','PRB2'],
        ['LYZ','PRELP'],
        ['LYZ','PRH1'],
        ['LYZ','PRH1-PRR4'],
        ['LYZ','PRKX'],
        ['LYZ','PRRC2B'],
        ['LYZ','PSAP'],
        ['LYZ','PSKH1'],
        ['LYZ','PSMC6'],
        ['LYZ','PTAFR'],
        ['LYZ','PTPRA'],
        ['LYZ','PTPRC'],
        ['LYZ','PTPRN2'],
        ['LYZ','R3HDM2'],
        ['LYZ','RAB11FIP1'],
        ['LYZ','RAB1A'],
        ['LYZ','RAB8B'],
        ['LYZ','RABGEF1'],
        ['LYZ','RANBP10'],
        ['LYZ','RBM3'],
        ['LYZ','RBM39'],
        ['LYZ','RBM5'],
        ['LYZ','RBSN'],
        ['LYZ','REV1'],
        ['LYZ','RGCC'],
        ['LYZ','RIOK3'],
        ['LYZ','RNF115'],
        ['LYZ','RP11-307N16.6'],
        ['LYZ','RP11-468E2.1'],
        ['LYZ','RP11-473I1.9'],
        ['LYZ','RP11-613M10.9'],
        ['LYZ','RP11-618G20.1'],
        ['LYZ','RP3-331H24.5'],
        ['LYZ','RPL26'],
        ['LYZ','RPL30'],
        ['LYZ','RPL37A'],
        ['LYZ','RPL7L1'],
        ['LYZ','RSPH6A'],
        ['LYZ','RTF1'],
        ['LYZ','RUNDC1'],
        ['LYZ','SAT1'],
        ['LYZ','SBF2'],
        ['LYZ','SCAF11'],
        ['LYZ','SCAMP1'],
        ['LYZ','SF3B1'],
        ['LYZ','SFPQ'],
        ['LYZ','SGSM3'],
        ['LYZ','SLC20A2'],
        ['LYZ','SLC25A37'],
        ['LYZ','SLC25A39'],
        ['LYZ','SLC2A3'],
        ['LYZ','SLC43A2'],
        ['LYZ','SLC4A1'],
        ['LYZ','SLC7A5'],
        ['LYZ','SMIM14'],
        ['LYZ','SMR3B'],
        ['LYZ','SNTB2'],
        ['LYZ','SP3'],
        ['LYZ','SPATA13'],
        ['LYZ','SPC24'],
        ['LYZ','SPINT2'],
        ['LYZ','SPN'],
        ['LYZ','SPTY2D1'],
        ['LYZ','SRGN'],
        ['LYZ','SRRM2'],
        ['LYZ','SRSF10'],
        ['LYZ','SSH1'],
        ['LYZ','STAT3'],
        ['LYZ','STK32C'],
        ['LYZ','STK39'],
        ['LYZ','STT3B'],
        ['LYZ','STX16'],
        ['LYZ','STX16-NPEPL1'],
        ['LYZ','STX8'],
        ['LYZ','SUGCT'],
        ['LYZ','SUGP2'],
        ['LYZ','SYAP1'],
        ['LYZ','TAF15'],
        ['LYZ','TAOK3'],
        ['LYZ','TBRG1'],
        ['LYZ','TENM3'],
        ['LYZ','TFCP2L1'],
        ['LYZ','TFRC'],
        ['LYZ','TGFBRAP1'],
        ['LYZ','THEMIS2'],
        ['LYZ','TIA1'],
        ['LYZ','TIGD6'],
        ['LYZ','TMBIM6'],
        ['LYZ','TMED3'],
        ['LYZ','TMEM220-AS1'],
        ['LYZ','TMEM87A'],
        ['LYZ','TPD52L1'],
        ['LYZ','TPT1'],
        ['LYZ','TRA2A'],
        ['LYZ','TRIP12'],
        ['LYZ','TSC2'],
        ['LYZ','TSG101'],
        ['LYZ','TXNL4A'],
        ['LYZ','TYK2'],
        ['LYZ','UBE2D3'],
        ['LYZ','UBE2H'],
        ['LYZ','UBXN4'],
        ['LYZ','UBXN7'],
        ['LYZ','UFD1L'],
        ['LYZ','UGDH-AS1'],
        ['LYZ','USP4'],
        ['LYZ','VPS16'],
        ['LYZ','VPS4B'],
        ['LYZ','VPS53'],
        ['LYZ','VTI1A'],
        ['LYZ','WDR1'],
        ['LYZ','WIPF2'],
        ['LYZ','WSB1'],
        ['LYZ','YIPF4'],
        ['LYZ','YME1L1'],
        ['LYZ','ZBTB43'],
        ['LYZ','ZKSCAN1'],
        ['LYZ','ZMYND8'],
        ['LYZ','ZNF251'],
        ['LYZ','ZNF655'],
        ['MACF1','PAX8'],
        ['MACF1','TG'],
        ['MAD2L1','MRI1'],
        ['MAF','RP11-679B19.2'],
        ['MAFG-AS1','RP11-498C9.12'],
        ['MALRD1','PLXDC2'],
        ['MALRD1','RP11-1143G9.4'],
        ['MAMDC2-AS1','SMC5-AS1'],
        ['MAML1','SQSTM1'],
        ['MAN1A2','STATH'],
        ['MAOA','MUC7'],
        ['MAP2K2','PIGR'],
        ['MAP4K4','TG'],
        ['MAPK1','TRB@'],
        ['MAPK7','RNF112'],
        ['MAPKAPK3','RP11-804H8.6'],
        ['MARCH1','SNRPA1'],
        ['MARCH6','MUC7'],
        ['MAT1A','RP11-573D15.8'],
        ['MAT2B','STATH'],
        ['MATN2','TG'],
        ['MATN4','SLPI'],
        ['MAVS','PANK2'],
        ['MAVS','RP5-1009E24.9'],
        ['MB','RP11-814P5.1'],
        ['MB','TTN-AS1'],
        ['MBD3','TG'],
        ['MCAM','MYL7'],
        ['MCFD2','TG'],
        ['MCL1','S100A9'],
        ['MCM3AP-AS1','SSR3'],
        ['MCPH1','PIGR'],
        ['MCUR1','SRSF10'],
        ['MDM1','SFTPB'],
        ['MECOM','RPL22'],
        ['MECP2','TG'],
        ['MED15','ZNF890P'],
        ['MED21','RP11-582E3.4'],
        ['MED25','PDLIM1'],
        ['MED6','RP11-486O13.4'],
        ['MEG3','SLC16A3'],
        ['MEG3','SPARC'],
        ['MEG3','STAR'],
        ['MEG3','TFAP2D'],
        ['METTL22','RP11-475D10.4'],
        ['METTL23','MFSD11'],
        ['METTL7A','USP37'],
        ['MFAP5','PIGR'],
        ['MFGE8','NPPA-AS1'],
        ['MFSD1','RP11-170K4.2'],
        ['MGAT4A','PRSS2'],
        ['MGEA5','S100A9'],
        ['MICA','ZDHHC20P2'],
        ['MICAL3','SPAG9'],
        ['MICALL1','POLR2F'],
        ['MICALL1','RP5-1039K5.19'],
        ['MICU2','ZDHHC20'],
        ['MICU3','SERPINB1'],
        ['MINK1','SFTPB'],
        ['MINOS1','NEK10'],
        ['MINOS1-NBL1','NEK10'],
        ['MKLN1','VMAC'],
        ['MKNK2','TG'],
        ['MLLT10P1','NCOA3'],
        ['MLLT6','SMR3B'],
        ['MMD','TG'],
        ['MMD2','RP11-284F21.7'],
        ['MNAT1','SMR3B'],
        ['MOB2','MUC7'],
        ['MORC2-AS1','OSBP2'],
        ['MPDU1','RP11-186B7.4'],
        ['MPHOSPH9','RP11-282O18.3'],
        ['MPO','NUP210L'],
        ['MPO','PDZD2'],
        ['MPO','PIEZO1'],
        ['MPO','SNX32'],
        ['MPO','ZFP36'],
        ['MRE11A','RP11-867G2.8'],
        ['MRPL30','RN7SL377P'],
        ['MRPL30P1','MS4A8'],
        ['MRPL42P3','STX11'],
        ['MRPS31P2','TPTE2'],
        ['MS4A10','RP11-158I13.2'],
        ['MS4A10','ZNF512'],
        ['MS4A12','THCAT158'],
        ['MSI1','PLA2G1B'],
        ['MSMB','NCOA4'],
        ['MSN','MUC7'],
        ['MSN','NPPA-AS1'],
        ['MT1A','RSG1'],
        ['MT1A','TG'],
        ['MT1X','RSG1'],
        ['MTERF4','PASK'],
        ['MTMR6','RP11-271M24.2'],
        ['MTR','RYR2'],
        ['MUC21','PPL'],
        ['MUC5AC','MUC6'],
        ['MUC5AC','PGC'],
        ['MUC5AC','PTBP1'],
        ['MUC5B','MUC7'],
        ['MUC5B','SMR3B'],
        ['MUC5B','STATH'],
        ['MUC6','PGC'],
        ['MUC7','PDLIM5'],
        ['MUC7','PHLDA1'],
        ['MUC7','POLR2A'],
        ['MUC7','PRB1'],
        ['MUC7','PRB2'],
        ['MUC7','PRH1'],
        ['MUC7','PRH1-PRR4'],
        ['MUC7','PRH2'],
        ['MUC7','PRR4'],
        ['MUC7','PSAP'],
        ['MUC7','RP11-432B6.3'],
        ['MUC7','RPL8'],
        ['MUC7','RPS11'],
        ['MUC7','SCARB2'],
        ['MUC7','SECISBP2L'],
        ['MUC7','SQSTM1'],
        ['MUC7','STATH'],
        ['MUC7','SYN3'],
        ['MUC7','TMED10'],
        ['MUC7','TSHZ2'],
        ['MUC7','ZG16B'],
        ['MUM1','NDUFS7'],
        ['MXRA7','RBMS2'],
        ['MYADM','NEDD4L'],
        ['MYADM','PARN'],
        ['MYADM','SEC11A'],
        ['MYBPC3','PPP1R12B'],
        ['MYEF2','RP11-605F22.1'],
        ['MYH11','MYH6'],
        ['MYH11','P4HB'],
        ['MYH11','PIGR'],
        ['MYH11','RHCG'],
        ['MYH11','RP11-2C24.9'],
        ['MYH11','SPINK5'],
        ['MYH11','SRCAP'],
        ['MYH11','SYN3'],
        ['MYH11','SYNM'],
        ['MYH11','TGM3'],
        ['MYH11','TRB@'],
        ['MYH11','TRIM29'],
        ['MYH16','SCD5'],
        ['MYH3','SCO1'],
        ['MYH6','NEXN'],
        ['MYH6','NPPA'],
        ['MYH6','NPPA-AS1'],
        ['MYH6','RP5-857K21.4'],
        ['MYH6','RP5-857K21.6'],
        ['MYH7','NPPA-AS1'],
        ['MYH7','SLC4A3'],
        ['MYH7','TNNC1'],
        ['MYH9','NDE1'],
        ['MYH9','PRH1'],
        ['MYH9','PRH1-PRR4'],
        ['MYH9','PRH2'],
        ['MYHAS','RP11-1143G9.4'],
        ['MYL2','NCOA6'],
        ['MYL2','NPPA-AS1'],
        ['MYL2','PALM2-AKAP2'],
        ['MYL3','PRSS42'],
        ['MYL3','PRSS50'],
        ['MYL4','NPPA'],
        ['MYL6','TG'],
        ['MYL7','MYLK3'],
        ['MYL7','NPPA'],
        ['MYL7','PPP1R12B'],
        ['MYL7','RYR2'],
        ['MYLK','PRB3'],
        ['MYO18A','RP11-22N12.2'],
        ['MYO19','ZG16B'],
        ['MYOM1','RP13-270P17.1'],
        ['MYOZ1','SYNPO2L'],
        ['MYRFL','RAB3IP'],
        ['MYT1','OPRL1'],
        ['N6AMT1','RP11-178H8.3'],
        ['NAA20','RIN2'],
        ['NACA','NPPA-AS1'],
        ['NACA','RAB3IL1'],
        ['NACC1','NFX1'],
        ['NANOGNBP2','ZNF605'],
        ['NAPEPLD','RP11-401L13.7'],
        ['NARS','RP11-706P11.2'],
        ['NBEAL2','S100A9'],
        ['NBEAP1','RP11-32B5.2'],
        ['NBEAP1','RP11-467L19.16'],
        ['NCAM1-AS1','RP11-839D17.3'],
        ['NCBP2-AS1','SENP5'],
        ['NCKAP1L','PDE1B'],
        ['NCLN','S1PR4'],
        ['NCOA1','RP11-1143G9.4'],
        ['NCOA6','NPPA'],
        ['NCOA6','PIGU'],
        ['NCOR2','SMR3B'],
        ['NDE1','PGR'],
        ['NDE1','STAMBPL1'],
        ['NDRG1','ST3GAL1'],
        ['NDRG2','SMR3B'],
        ['NDUFA6-AS1','SMDT1'],
        ['NDUFB8','SEC31B'],
        ['NDUFC2-KCTD14','SMR3B'],
        ['NDUFV1','NPPA-AS1'],
        ['NEAT1','SYN3'],
        ['NEAT1','TMEM99'],
        ['NEBL','SMR3B'],
        ['NECAB2','OSGIN1'],
        ['NEDD4L','TRB@'],
        ['NEDD8','TINF2'],
        ['NEDD8-MDP1','TINF2'],
        ['NEFH','RFPL1'],
        ['NEK7','PIGR'],
        ['NEK9','RP11-618G20.1'],
        ['NEK9','TMED10'],
        ['NEMP1','PLEKHA2'],
        ['NEU3','SPCS2'],
        ['NFE2L1','NPPA-AS1'],
        ['NFIB','PRH1-PRR4'],
        ['NFIC','PNLIP'],
        ['NFIX','SMR3B'],
        ['NFYC','RP11-276H19.4'],
        ['NIN','SAV1'],
        ['NKIRAS1','SMR3B'],
        ['NKTR','TMEM99'],
        ['NMNAT1','RBP7'],
        ['NMT1','PABPC1'],
        ['NNT-AS1','TPT1P2'],
        ['NOL11','PITPNC1'],
        ['NOP10','RP11-290L1.5'],
        ['NOP9','TF'],
        ['NOS1','PKP1'],
        ['NOX4','RP11-643G5.6'],
        ['NOXRED1','VIPAS39'],
        ['NPEPL1','STX16'],
        ['NPHS1','PRODH2'],
        ['NPPA','POLD2'],
        ['NPPA','PPP1R12B'],
        ['NPPA','RP1-178F15.5'],
        ['NPPA','RP11-64D24.2'],
        ['NPPA','RP11-77K12.8'],
        ['NPPA','RP11-814P5.1'],
        ['NPPA','S100A13'],
        ['NPPA','SF3B1'],
        ['NPPA','SPOCK1'],
        ['NPPA','STAMBPL1'],
        ['NPPA','SYNPO'],
        ['NPPA','TECRL'],
        ['NPPA','TGM2'],
        ['NPPA','TNNT2'],
        ['NPPA','TNS1'],
        ['NPPA-AS1','NPPB'],
        ['NPPA-AS1','NUCB2'],
        ['NPPA-AS1','PBX1'],
        ['NPPA-AS1','PLEC'],
        ['NPPA-AS1','PSAP'],
        ['NPPA-AS1','RAB11B'],
        ['NPPA-AS1','RCAN2'],
        ['NPPA-AS1','RILPL1'],
        ['NPPA-AS1','S100A1'],
        ['NPPA-AS1','SGCD'],
        ['NPPA-AS1','SPARC'],
        ['NPPA-AS1','SQSTM1'],
        ['NPPA-AS1','SRRM2'],
        ['NPPA-AS1','SYNPO'],
        ['NPPA-AS1','TGOLN2'],
        ['NPPA-AS1','TNNC1'],
        ['NPPA-AS1','TNNI3'],
        ['NPPA-AS1','TNNT2'],
        ['NPPA-AS1','TNS1'],
        ['NPPA-AS1','TP53INP2'],
        ['NPPA-AS1','TPM1'],
        ['NPPA-AS1','TTN-AS1'],
        ['NPPA-AS1','TXNIP'],
        ['NPPA-AS1','USB1'],
        ['NPPA-AS1','WDPCP'],
        ['NPTXR','RAD52'],
        ['NRBP2','PUF60'],
        ['NRIP3','TMEM9B'],
        ['NRSN2-AS1','TRB@'],
        ['NSUN7','RP11-632F7.3'],
        ['NT5C2','PRICKLE1'],
        ['NTN4','TRB@'],
        ['NTRK2','TG'],
        ['NUCB2','SMR3B'],
        ['NUDT17','POLR3C'],
        ['NUP160','PNLIP'],
        ['NUP35','RP11-438L19.1'],
        ['NUP50','RP11-973N13.3'],
        ['NUPR1','RP11-666O2.2'],
        ['OAZ1','PNLIP'],
        ['OAZ1','SGTA'],
        ['OAZ2','RBPMS2'],
        ['OBFC1','SH3PXD2A'],
        ['OCA2','RP11-30G8.1'],
        ['OGDH','PAX8-AS1'],
        ['OOEP-AS1','PTMA'],
        ['OR2J3','XXBAC-BPG258E24.10'],
        ['OR2T8','TRIM58'],
        ['OR52L1','OR56A4'],
        ['OR52M2P','TRIM21'],
        ['OR6K3','OR6K4P'],
        ['OR6K5P','OR6N2'],
        ['OR6M3P','TMEM225'],
        ['OR7D1P','OR7E24'],
        ['OR8T1P','ZNF641'],
        ['ORC1','ZCCHC11'],
        ['OSMR','RP11-108O10.8'],
        ['OSTF1','SSR3'],
        ['OVGP1','RP11-552M11.4'],
        ['OXR1','PNLIP'],
        ['OXTR','SSUH2'],
        ['P4HB','SMR3B'],
        ['PA2G4','ZC3H10'],
        ['PADI4','TFRC'],
        ['PAICS','SRP72'],
        ['PAK3','TRB@'],
        ['PAK4','SMR3B'],
        ['PAM','PNLIP'],
        ['PANK1','SLC16A12'],
        ['PANK3','PIGR'],
        ['PAPD7','RP11-332J15.3'],
        ['PAPOLA','RP11-433J8.1'],
        ['PAPPA','TIMP3'],
        ['PAQR7','STMN1'],
        ['PARL','RP1-95L4.4'],
        ['PATZ1','PIK3IP1'],
        ['PAX8-AS1','TG'],
        ['PAX8-AS1','TPO'],
        ['PBX1','TG'],
        ['PCAT1','RP11-89K10.1'],
        ['PCBP1-AS1','S100A9'],
        ['PCBP1-AS1','SRGN'],
        ['PCBP1-AS1','TG'],
        ['PCBP2','SMR3B'],
        ['PCBP2','TG'],
        ['PCBP3','PTTG1IP'],
        ['PCCB','PDS5B'],
        ['PCDHGA12','PCDHGB8P'],
        ['PCDHGB8P','PCDHGC3'],
        ['PCDHGB8P','PCDHGC4'],
        ['PCDHGB8P','PCDHGC5'],
        ['PCK1','RP11-573D15.8'],
        ['PDE8B','ZBED3-AS1'],
        ['PDHA1','STATH'],
        ['PDIA6','PRSS2'],
        ['PDIA6','RP11-791G15.2'],
        ['PDIA6','SMR3B'],
        ['PDIA6','STATH'],
        ['PDK4','TRB@'],
        ['PDS5A','TG'],
        ['PDXDC1','PNLIP'],
        ['PDZD2','PVRL4'],
        ['PDZK1IP1','TAL1'],
        ['PEA15','SMR3B'],
        ['PECAM1','SFTPB'],
        ['PECAM1','ZG16B'],
        ['PER1','PRH1'],
        ['PER1','PRH1-PRR4'],
        ['PER1','PRR4'],
        ['PFKFB3','RP11-563J2.2'],
        ['PFKFB4','SHISA5'],
        ['PFKL','TG'],
        ['PGAM1P5','PRSS2'],
        ['PGAM1P5','TRB@'],
        ['PGAM1P5','TRBC2'],
        ['PGAM1P5','TRBV25-1'],
        ['PGC','PIGR'],
        ['PGC','RP11-1143G9.4'],
        ['PGC','RP11-571M6.7'],
        ['PHACTR4','RCC1'],
        ['PHACTR4','SNTB2'],
        ['PHF14','SMR3B'],
        ['PHLDA1','PRH1'],
        ['PHLDA1','PRH1-PRR4'],
        ['PIAS1','SMR3B'],
        ['PICALM','PRB3'],
        ['PIGL','UBB'],
        ['PIGR','PIP'],
        ['PIGR','PLA2G2A'],
        ['PIGR','POC5'],
        ['PIGR','PRB1'],
        ['PIGR','PRB2'],
        ['PIGR','PRH1'],
        ['PIGR','PRH1-PRR4'],
        ['PIGR','PRR4'],
        ['PIGR','RBM20'],
        ['PIGR','RP11-319G6.1'],
        ['PIGR','RP11-403P17.5'],
        ['PIGR','SETD4'],
        ['PIGR','SLC44A4'],
        ['PIGR','SLC5A1'],
        ['PIGR','SPTBN1'],
        ['PIGR','SUSD1'],
        ['PIGR','SYNPO2'],
        ['PIGR','TMBIM6'],
        ['PIGR','ZG16B'],
        ['PIGR','ZKSCAN1'],
        ['PIGR','ZNF232'],
        ['PIM2','SLC35A2'],
        ['PIM3','SCO2'],
        ['PIN4','PNLIP'],
        ['PINK1-AS','TTL'],
        ['PITPNA','SMR3B'],
        ['PITPNA','TG'],
        ['PITPNC1','SLC25A3'],
        ['PIWIL1','RP11-117L5.1'],
        ['PK','RP11-814P5.1'],
        ['PKHD1L1','TG'],
        ['PKM','RP11-1143G9.4'],
        ['PKP1','TMEM99'],
        ['PLA2G1B','PRSS1'],
        ['PLA2G1B','PRSS2'],
        ['PLA2G1B','TRB@'],
        ['PLEKHA5','SRSF11'],
        ['PLEKHB2','SETBP1'],
        ['PLEKHG2','ZFP36'],
        ['PLEKHO1','VPS45'],
        ['PLP1','RP11-862L9.3'],
        ['PLXNA2','PRSS1'],
        ['PLXNA2','PRSS2'],
        ['PM20D2','WDFY3'],
        ['PNLIP','PRSS1'],
        ['PNLIP','PRSS2'],
        ['PNLIP','PRSS3'],
        ['PNLIP','PRSS3P1'],
        ['PNLIP','PTPRN2'],
        ['PNLIP','RP11-1021N1.1'],
        ['PNLIP','RP11-64D24.2'],
        ['PNLIP','RPL30'],
        ['PNLIP','RPS20'],
        ['PNLIP','SERF2'],
        ['PNLIP','SERPINI2'],
        ['PNLIP','SGK1'],
        ['PNLIP','SLC38A10'],
        ['PNLIP','SNTG2'],
        ['PNLIP','SOD2'],
        ['PNLIP','SPATS2L'],
        ['PNLIP','SPTBN1'],
        ['PNLIP','SRRM2'],
        ['PNLIP','ST6GALNAC6'],
        ['PNLIP','SYCN'],
        ['PNLIP','SYN3'],
        ['PNLIP','TCIRG1'],
        ['PNLIP','TMBIM6'],
        ['PNLIP','TMED3'],
        ['PNLIP','TMEM179B'],
        ['PNLIP','TRAM1'],
        ['PNLIP','TRB@'],
        ['PNLIP','TRHDE'],
        ['PNLIP','TRIM44'],
        ['PNLIP','TSC22D3'],
        ['PNLIP','UBE2R2-AS1'],
        ['PNLIP','USP7'],
        ['PNLIP','ZNF664'],
        ['PNLIPRP2','TRB@'],
        ['PNPLA6','TG'],
        ['PNPLA8','THAP5P1'],
        ['PODXL','TG'],
        ['POLR2A','PRM2'],
        ['POLR2A','TMEM99'],
        ['POMGNT1','TG'],
        ['POU2F3','TMEM99'],
        ['PPAP2A','RNF138'],
        ['PPARA','STATH'],
        ['PPDPF','RP4-697K14.3'],
        ['PPIF','RP11-342M3.5'],
        ['PPIL2','TG'],
        ['PPP1R12B','STT3A'],
        ['PPP1R12B','USHBP1'],
        ['PPP1R1B','STARD3'],
        ['PPP1R8','STX12'],
        ['PPP2CB','TG'],
        ['PPP2R1A','PRB3'],
        ['PPP3CB-AS1','TG'],
        ['PPP3CC','SORBS3'],
        ['PPP6R1','TG'],
        ['PPTC7','VPS29'],
        ['PRB1','PRH1'],
        ['PRB1','PRH1-PRR4'],
        ['PRB1','PRH2'],
        ['PRB1','PRR4'],
        ['PRB1','SFRP1'],
        ['PRB1','SMR3B'],
        ['PRB1','STATH'],
        ['PRB1','ZG16B'],
        ['PRB2','PRH1'],
        ['PRB2','PRH1-PRR4'],
        ['PRB2','PRH2'],
        ['PRB2','PRR4'],
        ['PRB2','RP11-1143G9.4'],
        ['PRB2','SMR3B'],
        ['PRB2','STATH'],
        ['PRB3','PRH1'],
        ['PRB3','PRH1-PRR4'],
        ['PRB3','PRH2'],
        ['PRB3','PRR4'],
        ['PRB3','RP11-154D6.1'],
        ['PRB3','SCARB2'],
        ['PRB3','SLPI'],
        ['PRB3','SMR3B'],
        ['PRB3','STATH'],
        ['PRB3','SYVN1'],
        ['PRB4','PRH1'],
        ['PRB4','PRH1-PRR4'],
        ['PRB4','PRH2'],
        ['PRB4','PRR4'],
        ['PRB4','SMR3B'],
        ['PRB4','STATH'],
        ['PRCP','RAB30'],
        ['PRH1','PRKCSH'],
        ['PRH1','RP11-1143G9.4'],
        ['PRH1','RP11-285F7.2'],
        ['PRH1','RPS19'],
        ['PRH1','SMR3B'],
        ['PRH1','STATH'],
        ['PRH1','TMBIM6'],
        ['PRH1','TNS1'],
        ['PRH1','UBC'],
        ['PRH1','ZG16B'],
        ['PRH1-PRR4','PRKCSH'],
        ['PRH1-PRR4','RP11-1143G9.4'],
        ['PRH1-PRR4','RP11-285F7.2'],
        ['PRH1-PRR4','RPS19'],
        ['PRH1-PRR4','SMR3B'],
        ['PRH1-PRR4','STATH'],
        ['PRH1-PRR4','TMBIM6'],
        ['PRH1-PRR4','TNS1'],
        ['PRH1-PRR4','UBC'],
        ['PRH1-PRR4','ZG16B'],
        ['PRH2','SH3BP4'],
        ['PRH2','SMR3B'],
        ['PRH2','STATH'],
        ['PRIM2','SCARB2'],
        ['PRKACA','RP11-120I21.2'],
        ['PRKACB','SAMD13'],
        ['PRKACB','TXN2'],
        ['PRKAR2A','RP11-1143G9.4'],
        ['PRKCH','TG'],
        ['PRKCSH','PRR4'],
        ['PRKX','TG'],
        ['PRPSAP1','QRICH2'],
        ['PRR18','SFT2D1'],
        ['PRR27','SMR3B'],
        ['PRR4','RP11-1143G9.4'],
        ['PRR4','RP11-285F7.2'],
        ['PRR4','RPS19'],
        ['PRR4','SH3BP4'],
        ['PRR4','SMR3B'],
        ['PRR4','STATH'],
        ['PRR4','TACC2'],
        ['PRR4','TMBIM6'],
        ['PRR4','UBC'],
        ['PRR4','ZG16B'],
        ['PRSS1','RP11-290D2.6'],
        ['PRSS1','RP11-331F4.4'],
        ['PRSS1','RP11-680B3.2'],
        ['PRSS1','SLC41A1'],
        ['PRSS1','SNRNP200'],
        ['PRSS1','SYCN'],
        ['PRSS1','TPT1'],
        ['PRSS1','UBE2R2-AS1'],
        ['PRSS2','RP11-290D2.6'],
        ['PRSS2','RP11-331F4.4'],
        ['PRSS2','RP11-680B3.2'],
        ['PRSS2','RRBP1'],
        ['PRSS2','RUVBL1'],
        ['PRSS2','SERPINE1'],
        ['PRSS2','SH3BP4'],
        ['PRSS2','SLC41A1'],
        ['PRSS2','SNRNP200'],
        ['PRSS2','SYCN'],
        ['PRSS2','TNS3'],
        ['PRSS2','TPT1'],
        ['PRSS2','UBE2R2-AS1'],
        ['PRSS3','TRB@'],
        ['PRSS3','UBE2R2'],
        ['PRTN3','RP11-1143G9.4'],
        ['PSMB7','SMR3B'],
        ['PSMD12P','SHQ1'],
        ['PTEN','RPL11'],
        ['PTHLH','TGM1'],
        ['PTMS','TMEM99'],
        ['PTOV1','TG'],
        ['PTPN22','RSBN1'],
        ['PTPN3','YBX1P6'],
        ['PTPRM','TG'],
        ['PTPRN2','TRB@'],
        ['PTRH1','RP11-56D16.8'],
        ['PUM1','SDC3'],
        ['PVRIG','STAG3L5P'],
        ['PVRL1','TMEM99'],
        ['PVRL2','SERINC5'],
        ['PVRL2','TOMM40'],
        ['PWP1','RP11-864J10.4'],
        ['PWRN1','PWRN3'],
        ['PXDC1','RP1-223B1.1'],
        ['PXK','RPP14'],
        ['PXT1','STK38'],
        ['PYY','TMEM101'],
        ['R3HDM1','UBXN4'],
        ['R3HDM2','STAC3'],
        ['RAB1A','STAT1'],
        ['RAB35','TG'],
        ['RAB40C','TECPR1'],
        ['RAB43P1','RABGAP1'],
        ['RAB4A','RP4-803J11.2'],
        ['RAB9A','TCEANC'],
        ['RAD51B','SMR3B'],
        ['RASSF2','RP4-756H11.5'],
        ['RBBP4','TRA@'],
        ['RBM3','RP11-1148L6.5'],
        ['RBM3','RP11-518L10.5'],
        ['RBM3','RP11-60L3.6'],
        ['RBM8A','SEH1L'],
        ['RBP4','RP11-573D15.8'],
        ['RBP4','SERPINA1'],
        ['RBP4','TXNL4B'],
        ['RBP7','VSIG1'],
        ['RBPJL','TRB@'],
        ['RBPMS','TUBBP1'],
        ['RBPMS-AS1','TNNT2'],
        ['RCAN3','RP11-155G14.6'],
        ['RCAN3','RP11-212P7.3'],
        ['RDH10-AS1','RP11-434I12.4'],
        ['RDH16','RP11-74M13.5'],
        ['REG1A','TRB@'],
        ['REG1A','UBE2R2-AS1'],
        ['RFC2','TPM1'],
        ['RFPL3S','RTCB'],
        ['RGCC','SRGN'],
        ['RGP1','RP11-112J3.16'],
        ['RGS6','RP6-114E22.1'],
        ['RHBDL2','RP5-864K19.6'],
        ['RHBDL2','RP5-864K19.7'],
        ['RHBG','TSACC'],
        ['RHCG','SPRR3'],
        ['RIIAD1','RP11-98D18.15'],
        ['RIN2','RP5-999L4.2'],
        ['RIOK3','SRGN'],
        ['RIPK1','SERPINB9'],
        ['RLF','RP11-115D7.3'],
        ['RMND1','ZBTB2'],
        ['RN7SL711P','RP11-359P5.1'],
        ['RN7SL837P','RP5-858B6.3'],
        ['RNASE1','RP11-680B3.2'],
        ['RNF111','RP11-707P17.1'],
        ['RNF130','RTN4RL1'],
        ['RNF168','UBXN7'],
        ['RNF20','TMEM246-AS1'],
        ['RNF212B','TMEM184A'],
        ['RNMTL1','RP11-1143G9.4'],
        ['ROCK2','SFTPB'],
        ['RP1-122P22.2','RP5-999L4.2'],
        ['RP1-137D17.2','WDR27'],
        ['RP1-167F1.2','RP11-686D16.1'],
        ['RP1-178F15.5','S100A16'],
        ['RP1-178F15.5','TNNT2'],
        ['RP1-199J3.6','TSTD3'],
        ['RP1-20N18.4','SPRR3'],
        ['RP1-266L20.2','TCTE3'],
        ['RP1-317E23.6','TG'],
        ['RP1-7G5.6','TCHP'],
        ['RP11-1008C21.1','TMCO5A'],
        ['RP11-1035H13.3','SMG1'],
        ['RP11-108K14.8','SCART1'],
        ['RP11-1099M24.8','RPL21P44'],
        ['RP11-109P14.2','VCPKMT'],
        ['RP11-1143G9.4','RP11-799N11.1'],
        ['RP11-1143G9.4','RPL8'],
        ['RP11-1143G9.4','S100A8'],
        ['RP11-1143G9.4','SESN1'],
        ['RP11-1143G9.4','SLC25A39'],
        ['RP11-1143G9.4','SLC2A3'],
        ['RP11-1143G9.4','SLC4A1'],
        ['RP11-1143G9.4','SMARCA4'],
        ['RP11-1143G9.4','SPTBN1'],
        ['RP11-1143G9.4','SPTY2D1-AS1'],
        ['RP11-1143G9.4','SRGN'],
        ['RP11-1143G9.4','TACC2'],
        ['RP11-1143G9.4','TFF1'],
        ['RP11-1143G9.4','TPT1'],
        ['RP11-1143G9.4','TSPAN33'],
        ['RP11-134O21.1','RP11-378A12.1'],
        ['RP11-135A1.3','RP11-407B7.3'],
        ['RP11-138I1.4','UBC'],
        ['RP11-143A12.3','RP11-315A16.3'],
        ['RP11-145E5.5','UBA52'],
        ['RP11-146B14.1','TMEM50B'],
        ['RP11-151A6.4','TMTC4'],
        ['RP11-159D12.2','SRSF1'],
        ['RP11-15B24.5','SPATA31D3'],
        ['RP11-164H13.1','TCL1B'],
        ['RP11-166O4.6','STAG3L4'],
        ['RP11-173M1.3','WSB1'],
        ['RP11-227G15.10','ZNF207'],
        ['RP11-248J23.7','TG'],
        ['RP11-252M21.6','TEX26-AS1'],
        ['RP11-257I8.2','RP11-508M8.1'],
        ['RP11-278L15.2','TM4SF1'],
        ['RP11-27M24.1','ZNF483'],
        ['RP11-285M22.1','RP11-846F4.11'],
        ['RP11-290D2.6','TRB@'],
        ['RP11-290D2.6','TRBC2'],
        ['RP11-290D2.6','TRBV25-1'],
        ['RP11-290L1.3','TG'],
        ['RP11-302F12.5','SEL1L3'],
        ['RP11-314N13.10','TG'],
        ['RP11-31K23.1','SRSF3'],
        ['RP11-326C3.16','SRP19'],
        ['RP11-331F4.4','TRB@'],
        ['RP11-331F4.4','TRBC2'],
        ['RP11-331F4.4','TRBV25-1'],
        ['RP11-332H14.1','TGFBRAP1'],
        ['RP11-33N16.3','SYNC'],
        ['RP11-340A13.1','RP11-506N2.1'],
        ['RP11-343H19.2','RSG1'],
        ['RP11-357K6.2','SMR3B'],
        ['RP11-359E19.1','RP11-470M17.2'],
        ['RP11-359E19.2','RP11-470M17.2'],
        ['RP11-360D2.1','RP3-436N22.3'],
        ['RP11-364B6.2','THAP3'],
        ['RP11-365F18.1','RPL37A'],
        ['RP11-368M16.3','RP11-548K12.10'],
        ['RP11-369E15.2','RP11-421P23.1'],
        ['RP11-369E15.4','RP11-421P23.1'],
        ['RP11-377D9.3','RP11-392P7.6'],
        ['RP11-383C6.2','TRUB1'],
        ['RP11-392O17.1','RP11-95P13.1'],
        ['RP11-3G20.2','RP11-785H20.1'],
        ['RP11-415A20.1','TMEM236'],
        ['RP11-421L21.2','SLC30A7'],
        ['RP11-424M21.1','RP11-73G16.2'],
        ['RP11-434D2.11','TNFRSF13B'],
        ['RP11-435P24.2','SERF2'],
        ['RP11-448G15.3','SLC2A9'],
        ['RP11-448P19.1','SPATA19'],
        ['RP11-469J4.3','SKIL'],
        ['RP11-479O9.4','RP11-72L22.1'],
        ['RP11-489G11.3','SCRG1'],
        ['RP11-501O2.3','RP11-639B1.1'],
        ['RP11-503L23.1','RP11-73G16.3'],
        ['RP11-510J16.5','SDR42E1'],
        ['RP11-514O12.4','TG'],
        ['RP11-521C20.2','SEMA4D'],
        ['RP11-536C5.7','RP11-574F21.3'],
        ['RP11-545P7.4','WSB1'],
        ['RP11-548P2.2','ST3GAL5-AS1'],
        ['RP11-560G2.1','SET'],
        ['RP11-562A8.4','USP50'],
        ['RP11-571E6.3','SERPINB1'],
        ['RP11-573D15.8','SERPINC1'],
        ['RP11-573D15.8','TF'],
        ['RP11-573D15.8','TFP1'],
        ['RP11-573D15.8','TMEM176B'],
        ['RP11-574K11.27','RP11-96C23.12'],
        ['RP11-58E21.5','VCPKMT'],
        ['RP11-597D13.7','TMEM144'],
        ['RP11-603J24.9','ZC3H10'],
        ['RP11-610P16.1','RP11-731D1.4'],
        ['RP11-613D13.5','SLC9A9'],
        ['RP11-613M10.9','SLC25A52'],
        ['RP11-613M10.9','SMR3B'],
        ['RP11-618G20.1','SF3B1'],
        ['RP11-62H20.1','RTN1'],
        ['RP11-63E5.1','ZNF596'],
        ['RP11-640N20.9','WDR45B'],
        ['RP11-661A12.12','TSTA3'],
        ['RP11-661C8.2','TRIP13'],
        ['RP11-666A8.8','ST6GALNAC2'],
        ['RP11-666E17.1','RP5-1014D13.2'],
        ['RP11-680B3.2','TRB@'],
        ['RP11-680B3.2','TRBC2'],
        ['RP11-680B3.2','TRBV25-1'],
        ['RP11-696N14.1','SERPINA1'],
        ['RP11-697E2.12','ZNF774'],
        ['RP11-69E11.4','TG'],
        ['RP11-6N17.4','SP2-AS1'],
        ['RP11-702H23.4','SPNS3'],
        ['RP11-702H23.4','TG'],
        ['RP11-702H23.6','SPNS3'],
        ['RP11-737G21.1','TRIM2'],
        ['RP11-73M18.2','RP11-894P9.2'],
        ['RP11-742B18.1','SPARCL1'],
        ['RP11-745L13.2','RPL7AP28'],
        ['RP11-746M1.1','TBC1D27'],
        ['RP11-747D18.1','SI'],
        ['RP11-77K12.8','TG'],
        ['RP11-797H7.1','RPL34-AS1'],
        ['RP11-79E3.3','SP100'],
        ['RP11-79N23.1','RP13-476E20.1'],
        ['RP11-814P5.1','TNNT2'],
        ['RP11-81A1.6','SLAIN2'],
        ['RP11-829H16.3','STRN3'],
        ['RP11-88E10.4','TUBGCP3'],
        ['RP11-903H12.3','TF'],
        ['RP11-91K8.4','TMEM108'],
        ['RP11-923I11.6','SCN8A'],
        ['RP11-950C14.7','TG'],
        ['RP11-973N13.4','ZBTB25'],
        ['RP3-492J12.2','ZNF831'],
        ['RP4-595K12.2','RP4-682C21.5'],
        ['RP4-630C24.3','SPAM1'],
        ['RP4-669H2.1','SLC35A1'],
        ['RP4-671G15.2','ST7L'],
        ['RP4-673D20.4','SIRPB3P'],
        ['RP4-678D15.1','SLC35A1'],
        ['RP5-1000K24.2','ZC3H12B'],
        ['RP5-1024N4.2','SLC5A9'],
        ['RP5-1024N4.5','SPATA6'],
        ['RP5-1065J22.8','TAF13'],
        ['RP5-1098D14.1','TEX35'],
        ['RP5-857K21.4','TPM1'],
        ['RP5-928E24.2','ZNF711'],
        ['RP5-972B16.2','TG'],
        ['RPL3','TRB@'],
        ['RPL30','TG'],
        ['RPL37A','TRB@'],
        ['RPL38','TTYH2'],
        ['RPL7L1','RPS2P55'],
        ['RPS12','S100A9'],
        ['RPS19','SNX6'],
        ['RPS29','SLC34A2'],
        ['RPS6KA2','TG'],
        ['RRBP1','TG'],
        ['RSPH4A','RWDD1'],
        ['RUSC2','SMR3B'],
        ['RUVBL1','SMR3B'],
        ['RUVBL1','TRB@'],
        ['RUVBL1','TRBC2'],
        ['RUVBL1','TRBV25-1'],
        ['RUVBL2','SMR3B'],
        ['RYR2','TG'],
        ['S100A13','S100A16'],
        ['S100A13','TNNT2'],
        ['S100A9','SLC25A37'],
        ['S100A9','SLC4A1'],
        ['S100A9','TKT'],
        ['SAMD8','VDAC2'],
        ['SAR1A','TG'],
        ['SARS','SMR3B'],
        ['SBF2','TG'],
        ['SCAND2P','ZSCAN2'],
        ['SCD5','SDC3'],
        ['SCD5','SEPT4-AS1'],
        ['SCG5','TRB@'],
        ['SCUBE3','ZNF76'],
        ['SDCBP2-AS1','ZNF587B'],
        ['SEC11A','SSR3'],
        ['SEC24C','SPAG9'],
        ['SEC61A1','TRB@'],
        ['SEL1L','SFTPB'],
        ['SEMA3B','U73166.2'],
        ['SEPN1','TRB@'],
        ['SEPT2','TG'],
        ['SEPT9','SMR3B'],
        ['SERINC3','TG'],
        ['SERP1','SMR3B'],
        ['SERP1','TG'],
        ['SERPINA1','TXNL4B'],
        ['SERPINA1','UGT2B4'],
        ['SERPINE1','TRBV25-1'],
        ['SERPINF2','WDR81'],
        ['SERPING1','TXNL4B'],
        ['SFN','SLFN5'],
        ['SFPQ','SMR3B'],
        ['SFPQ','SRGN'],
        ['SFTPA2','SYN3'],
        ['SFTPB','SYN3'],
        ['SFTPB','TBCEL'],
        ['SFTPB','XXYLT1'],
        ['SFTPC','TSIX'],
        ['SH3BGRL2','SMR3B'],
        ['SH3BP4','TRB@'],
        ['SH3BP4','TRBC2'],
        ['SH3BP4','TRBV25-1'],
        ['SHISA7','UBE2S'],
        ['SHQ1','TG'],
        ['SIGLEC16','WDR35'],
        ['SIPA1L3','TRB@'],
        ['SLAIN2','SLC10A4'],
        ['SLAIN2','ST20'],
        ['SLAIN2','ST20-MTHFS'],
        ['SLC16A3','TG'],
        ['SLC25A29','TPO'],
        ['SLC25A3','SRGN'],
        ['SLC25A3','SUGCT'],
        ['SLC25A3','TET1'],
        ['SLC26A4','TG'],
        ['SLC2A2','TNIK'],
        ['SLC30A9','TG'],
        ['SLC3A2','TG'],
        ['SLC41A1','TRB@'],
        ['SLC6A16','TEAD2'],
        ['SLPI','SMR3B'],
        ['SLTM','TG'],
        ['SMAD2','TG'],
        ['SMR3B','SPSB3'],
        ['SMR3B','SPTBN1'],
        ['SMR3B','SRRM2'],
        ['SMR3B','STATH'],
        ['SMR3B','SUPT5H'],
        ['SMR3B','SURF4'],
        ['SMR3B','SYN3'],
        ['SMR3B','TMED10'],
        ['SMR3B','TMEM120B'],
        ['SMR3B','TPD52L1'],
        ['SMR3B','TRB@'],
        ['SMR3B','TRPC4AP'],
        ['SMR3B','TXNDC5'],
        ['SMR3B','WWC1'],
        ['SMR3B','YBX3'],
        ['SMR3B','ZCCHC14'],
        ['SMR3B','ZG16B'],
        ['SMR3B','ZNF827'],
        ['SNAP25-AS1','SYN2'],
        ['SNRPD2','TG'],
        ['SNX2','SNX24'],
        ['SOGA1','TG'],
        ['SPARC','TG'],
        ['SPATS2','TMEM99'],
        ['SPINK1','TRB@'],
        ['SPPL3','TG'],
        ['SQLE','ZNF572'],
        ['SREBF2','TG'],
        ['SRGN','TKT'],
        ['SRGN','USP15'],
        ['SRGN','XIRP2'],
        ['SRI','STEAP4'],
        ['SRRM2','TMEM99'],
        ['SRSF10','ZFP69'],
        ['SSR2','UBQLN4'],
        ['SSR3','WIPI2'],
        ['SSR3','ZNF684'],
        ['ST6GAL1','TG'],
        ['STATH','UBXN4'],
        ['STC1','TG'],
        ['STIP1','TAPBP'],
        ['STMN4','TRIM35'],
        ['SUMF2','TG'],
        ['SVEP1','TXNDC8'],
        ['SYBU','TRB@'],
        ['SYCN','TRB@'],
        ['SYN3','TG'],
        ['SYN3','TNS1'],
        ['SYNC','TRA@'],
        ['SYNE1','TG'],
        ['SYT8','TNNI2'],
        ['SYTL4','TSPAN6'],
        ['TAB2','TG'],
        ['TAPBP','TTC39B'],
        ['TAT-AS1','TF'],
        ['TBC1D23','TMEM30C'],
        ['TBCEL','TECTA'],
        ['TCF25','TRB@'],
        ['TECPR1','TRB@'],
        ['TEX41','ZNF417'],
        ['TF','TXNL4B'],
        ['TG','TM2D3'],
        ['TG','TMED10'],
        ['TG','UQCC2'],
        ['TG','USP40'],
        ['TG','USP54'],
        ['TG','UTRN'],
        ['TG','VAC14'],
        ['TG','VAV3'],
        ['TG','WAC'],
        ['TG','WDR86'],
        ['TG','WHSC1'],
        ['TG','XPO1'],
        ['TG','YWHAB'],
        ['TG','ZFAND3'],
        ['TG','ZNF704'],
        ['TLK2','ZNF880'],
        ['TLN1','TRB@'],
        ['TMC6','TRIM69'],
        ['TMEM131','YIPF5'],
        ['TMEM189-UBE2V1','UBE2V1P6'],
        ['TMPRSS5','ZW10'],
        ['TNS3','TRB@'],
        ['TNS3','TRBC2'],
        ['TNS3','TRBV25-1'],
        ['TPST2','TRB@'],
        ['TPT1','TRB@'],
        ['TRB@','TREH'],
        ['TRB@','UBE2R2-AS1'],
        ['TTC38','UQCRB'],
        ['TXNL4B','XXBAC-BPG116M5.17'],
        ['UBE2I','UNKL'],
        ['UGT1A1','UGT1A2P'],
        ['WDR41','WWTR1'],
        ['ZC3H18','ZFPM1'],
        ['ZFP30','ZNF607'],
        ['ZNF224','ZNF225'],
        ['ZNF227','ZNF233'],
        ['ZNF331','ZNF765'],
        ['ZNF345','ZNF790-AS1'],
        ['ZNF517','ZNF7'],
        ['ZNF573','ZNF607'],
        ['ZNF708','ZNF738'],
        ['AC019349.5','FLNA'],
        ['ACPP','MYLK'],
        ['ACSS1','APMAP'],
        ['ACTB','CPA1'],
        ['ACTB','HDGF'],
        ['ACTB','MARCH6'],
        ['ACTB','SMR3B'],
        ['ACTB','TOP1'],
        ['ACTB','TRAM1'],
        ['ACTG1','ALB'],
        ['ADAP2','RNF135'],
        ['ADCY10P1','BGN'],
        ['AGBL5','HBB'],
        ['ALB','C4BPA'],
        ['ALB','FGB'],
        ['ALB','GATM'],
        ['ALB','GP2'],
        ['ALB','SERPINA1'],
        ['ALB','TF'],
        ['AMACR','C1QTNF3'],
        ['AMACR','SLC45A2'],
        ['ANK1','S100A9'],
        ['ANKRD11','TRB@'],
        ['ANXA4','PRSS1'],
        ['AP3D1','KRT4'],
        ['AP3S1','COMMD10'],
        ['ARHGAP1','FLG2'],
        ['ARHGAP18','PNLIP'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ART4','SMCO3'],
        ['ATHL1','TRB@'],
        ['ATP2A3','LYZ'],
        ['ATP2A3','PNLIP'],
        ['AZGP1','GJC3'],
        ['AZU1','DDX39B'],
        ['AZU1','ILF3'],
        ['AZU1','LCP1'],
        ['B2M','GNAS'],
        ['BACH2','CPA4'],
        ['BAIAP2L2','SLC16A8'],
        ['BCAP29','DUS4L'],
        ['BCL9L','SMR3B'],
        ['BPIFA2','LYZ'],
        ['C15ORF57','CBX3'],
        ['C1S','KRT10'],
        ['C3','SERPINA1'],
        ['C7','IGF2'],
        ['C7','TIMP2'],
        ['CALD1','IGF2'],
        ['CANX','MUC7'],
        ['CAPZB','IGF2'],
        ['CCDC183','TMEM141'],
        ['CCP110','TMC5'],
        ['CD24','COL6A2'],
        ['CD74','LYZ'],
        ['CD93','IGF2'],
        ['CDC42EP2','POLA2'],
        ['CEL','CELP'],
        ['CEL','GP2'],
        ['CFH','CFHR1'],
        ['CHCHD10','VPREB3'],
        ['CHD2','COL1A1'],
        ['CHST11','IGF2'],
        ['CKM','NPPA'],
        ['CLTC','PPP1R12B'],
        ['CNOT1','IGF2'],
        ['COL1A1','MYH11'],
        ['COPB1','MUC7'],
        ['CREB3L1','PRSS1'],
        ['CRNN','S100A9'],
        ['CTAGE5','MIA2'],
        ['CTBS','GNG5'],
        ['CTNNA1','KRT4'],
        ['CTSE','LYZ'],
        ['CUZD1','GP2'],
        ['DDX17','MYH9'],
        ['DDX17','SFTPB'],
        ['DHRS1','RABGGTA'],
        ['DLGAP4','TRB@'],
        ['DMBT1','PIGR'],
        ['DNAH1','GLYCTK'],
        ['DPP3','LYZ'],
        ['DSC3','FLG2'],
        ['DST','KRT10'],
        ['DSTYK','LYZ'],
        ['E2F3P2','MRE11A'],
        ['EEF1DP3','FRY'],
        ['EGFR','SSR3'],
        ['EHD3','IGF2'],
        ['ELK4','SLC45A3'],
        ['EPB41L2','IGFBP5'],
        ['ERBB2','GRB7'],
        ['ETF1','LYZ'],
        ['FAM102A','IGH@'],
        ['FAM107B','HBB'],
        ['FAM135A','SSR3'],
        ['FAM13A-AS1','HERC3'],
        ['FAM175A','HELQ'],
        ['FAM3D-AS1','SMR3B'],
        ['FAM46A','LYZ'],
        ['FBLN1','MEG3'],
        ['FHL2','TTN'],
        ['FKBP15','HTN3'],
        ['FLG2','TUFT1'],
        ['FLNA','SFN'],
        ['FLRT2','TRB@'],
        ['G0S2','MPO'],
        ['GAA','SMR3B'],
        ['GFAP','SPARCL1'],
        ['GLTSCR2','PRSS1'],
        ['GNG10','UGCG'],
        ['GOLT1A','KISS1'],
        ['HBB','ICE1'],
        ['HBB','LYST'],
        ['HBB','MTMR14'],
        ['HECA','SLC48A1'],
        ['HEPHL1','PANX1'],
        ['HM13','SMR3B'],
        ['HSP90AB1','SLC29A1'],
        ['HTN3','ITPR2'],
        ['HTN3','RNF150'],
        ['HTN3','TRAM1'],
        ['IGF2','IL6ST'],
        ['IGF2','KLF7'],
        ['IGF2','RBM6'],
        ['IGF2','SERPINE2'],
        ['IGF2','SNX2'],
        ['IGK@','MUC5AC'],
        ['IGKC','MUC5AC'],
        ['IGKV1-12','MUC5AC'],
        ['IGKV1-8','MUC5AC'],
        ['KLF3','PNLIP'],
        ['KLHL3','RP11-381K20.2'],
        ['KLK4','KLKP1'],
        ['KRT1','ZFP36L1'],
        ['KRT13','TIMP2'],
        ['KRT14','KRT2'],
        ['KRT14','NEAT1'],
        ['KRT4','SVIL'],
        ['KRT4','TGM3'],
        ['KSR1','TRB@'],
        ['LAIR1','LYZ'],
        ['LMNA','PNLIP'],
        ['LPO','PRH2'],
        ['LRP1','PECAM1'],
        ['LRRC75A-AS1','LYZ'],
        ['LYST','LYZ'],
        ['LYZ','OSBPL2'],
        ['LYZ','PLEKHM2'],
        ['LYZ','PSMB5'],
        ['LYZ','RPL4'],
        ['LYZ','SMG1'],
        ['LYZ','SPARC'],
        ['LYZ','TIMP2'],
        ['LYZ','WDR82'],
        ['LYZ','YIPF5'],
        ['LYZ','ZG16B'],
        ['MAP6D1','PARL'],
        ['MB','NPPA-AS1'],
        ['MEG3','SCARB1'],
        ['MEG3','TIMP3'],
        ['METTL23','MFSD11'],
        ['MSMB','NCOA4'],
        ['MTR','RYR2'],
        ['MUC7','PIGR'],
        ['MUC7','WNK2'],
        ['MYH11','SMR3B'],
        ['MYH11','TPM1'],
        ['MYH7','TNS1'],
        ['MYL2','NDRG1'],
        ['NCKAP1','PNLIP'],
        ['NDUFB8','SEC31B'],
        ['NECAB2','OSGIN1'],
        ['NPEPL1','STX16'],
        ['NPPA','TIMP3'],
        ['PAQR6','SMG5'],
        ['PAX5','RHOBTB3'],
        ['PDE4B','SMR3B'],
        ['PGC','POLR2A'],
        ['PIK3C2A','PNLIP'],
        ['PLEKHO1','VPS45'],
        ['PNLIP','SLC37A4'],
        ['PNLIP','SLC4A4'],
        ['PNLIP','SPINT2'],
        ['PNLIP','TRIM28'],
        ['PNLIPRP2','PRSS1'],
        ['PNLIPRP2','PRSS2'],
        ['PRSS1','RNASE1'],
        ['PRSS2','RNASE1'],
        ['RIPK1','SERPINB9'],
        ['RP11-356O9.1','TTC6'],
        ['SEC62','SMR3B'],
        ['SERPINF2','WDR81'],
        ['SFTPA2','ZMIZ1'],
        ['SH3BGR','WRB'],
        ['SIDT2','TAGLN'],
        ['TBCEL','TECTA'],
        ['TFF1','TFF2'],
        ['TPPP3','ZDHHC1'],
        ['ACE2','TMEM27'],
        ['ACSS1','APMAP'],
        ['ADD1','TG'],
        ['AFF1','TG'],
        ['AMACR','C1QTNF3'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BRDT','EPHX4'],
        ['C15ORF57','CBX3'],
        ['C7ORF49','WDR91'],
        ['CCDC80','IGK@'],
        ['CDC42EP2','POLA2'],
        ['CDK2','RAB5B'],
        ['CLCN7','TG'],
        ['CREB3L2','TPO'],
        ['CTBS','GNG5'],
        ['DHRS1','RABGGTA'],
        ['DMTN','TG'],
        ['EEF1DP3','FRY'],
        ['ERBB2','GRB7'],
        ['GLYR1','TG'],
        ['GMPPB','TG'],
        ['GSE1','TG'],
        ['ITGB1','TG'],
        ['MAP6D1','PARL'],
        ['MARCH6','TG'],
        ['MED13L','TG'],
        ['MED29','TG'],
        ['METTL23','MFSD11'],
        ['NCOA6','PIGU'],
        ['NDUFB8','SEC31B'],
        ['OLFM5P','UBQLNL'],
        ['PAFAH1B2','YIPF5'],
        ['PAQR6','SMG5'],
        ['PIGO','TG'],
        ['PLEKHH1','TG'],
        ['PLEKHO1','VPS45'],
        ['PPP1CB','TG'],
        ['RIPK1','SERPINB9'],
        ['SCUBE3','TG'],
        ['SH3BGR','WRB'],
        ['SIDT2','TAGLN'],
        ['SLC25A29','YY1'],
        ['TBCEL','TECTA'],
        ['TG','UBA1'],
        ['TG','VMP1'],
        ['ABLIM1','IGHM'],
        ['ACTB','CLU'],
        ['ACTB','IGK@'],
        ['AP1B1','IGK@'],
        ['AP1B1','IGKC'],
        ['ART4','SMCO3'],
        ['B2M','CCDC152'],
        ['B2M','STAT1'],
        ['C15ORF57','CBX3'],
        ['CD68','IGH@'],
        ['CD74','TRB@'],
        ['CD81','IGH@'],
        ['CECR1','IGH@'],
        ['CSK','IGHM'],
        ['CTBS','GNG5'],
        ['CTSB','IGK@'],
        ['CTSB','IGKC'],
        ['CTSS','IGK@'],
        ['DDX39B','IGK@'],
        ['DDX39B','IGKC'],
        ['DHRS1','RABGGTA'],
        ['EEF1DP3','FRY'],
        ['FAM175A','HELQ'],
        ['FTL','IGK@'],
        ['GM2A','IGK@'],
        ['GM2A','IGKC'],
        ['GPX3','IGK@'],
        ['GPX3','IGKC'],
        ['HECA','SLC48A1'],
        ['IGH@','LAMC1'],
        ['IGH@','LIPA'],
        ['IGH@','MYO18A'],
        ['IGH@','PARP14'],
        ['IGH@','PSMA4'],
        ['IGH@','RNH1'],
        ['IGH@','SAMHD1'],
        ['IGH@','SEPT9'],
        ['IGH@','TAP1'],
        ['IGH@','TMBIM6'],
        ['IGH@','RPS11'],
        ['IGH@','MYO18A'],
        ['IGH@','SEPT9'],
        ['IGK@','LIMD1'],
        ['METTL7A','TAB2'],
        ['SH3BGR','WRB'],
        ['SIDT2','TAGLN'],
        ['ABL1','ZG16B'],
        ['AC053503.6','MYL2'],
        ['ACE2','TMEM27'],
        ['ACO2','MYH6'],
        ['ACPP','ACTG2'],
        ['ACSS1','APMAP'],
        ['ACTB','IGF2'],
        ['ACTB','LAPTM5'],
        ['ACTG1','KRT13'],
        ['ACTG1','TRB@'],
        ['ACTN1','KRT4'],
        ['ACVR1B','STATH'],
        ['ADAP2','RNF135'],
        ['ADGRG7','TFG'],
        ['AFAP1L2','TG'],
        ['AHNAK','IGF2'],
        ['AKAP13','CLPS'],
        ['AKAP13','CPB1'],
        ['AL121578.2','SYTL5'],
        ['ALB','APOC3'],
        ['ALB','BAAT'],
        ['ALB','C3'],
        ['ALB','CES1'],
        ['ALB','CYP2C9'],
        ['ALB','FGB'],
        ['ALB','ITPR2'],
        ['ALB','MAOB'],
        ['ALB','MGST1'],
        ['ALB','SCD'],
        ['ALB','SERPINA1'],
        ['ALB','STAU1'],
        ['ALB','TF'],
        ['AMACR','C1QTNF3'],
        ['AMACR','SLC45A2'],
        ['AMY2B','PNLIP'],
        ['ANK1','LYZ'],
        ['ANKRD28','TRB@'],
        ['ANPEP','HBB'],
        ['AP3S1','COMMD10'],
        ['APP','HBB'],
        ['APP','KRT10'],
        ['APP','PNLIP'],
        ['APP','TRB@'],
        ['AQP3','SMR3B'],
        ['ARID1B','SMR3B'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARMS2','PLEKHA1'],
        ['ART4','SMCO3'],
        ['ATF7IP','IGF2'],
        ['ATG14','LYZ'],
        ['AZGP1','GJC3'],
        ['B2M','GNAS'],
        ['BAIAP2L2','SLC16A8'],
        ['BCAP29','DUS4L'],
        ['BCKDHA','TMEM91'],
        ['BHLHE40','TRB@'],
        ['BICD1','DES'],
        ['BNIP3L','TG'],
        ['BRDT','EPHX4'],
        ['C12ORF65','NIPAL1'],
        ['C15ORF57','CBX3'],
        ['C3','SERPINA1'],
        ['C7ORF49','WDR91'],
        ['CALM3','TG'],
        ['CCDC183','TMEM141'],
        ['CCP110','TMC5'],
        ['CD34','NPPA'],
        ['CD68','MPDU1'],
        ['CD82','COLEC12'],
        ['CDC42EP2','POLA2'],
        ['CDK2','RAB5B'],
        ['CEL','CELP'],
        ['CEL','SPINK1'],
        ['CFH','CFHR1'],
        ['CFL1','TG'],
        ['CHCHD10','VPREB3'],
        ['CHD8','SMR3B'],
        ['CHPT1','LYZ'],
        ['CIRBP','MUC7'],
        ['CLDN18','PGC'],
        ['CNOT1','PNLIP'],
        ['COL14A1','IGFBP5'],
        ['COL15A1','IGF2'],
        ['COL15A1','TG'],
        ['COL18A1','TRB@'],
        ['COLEC12','IGF2'],
        ['CPA1','HUWE1'],
        ['CPA1','PLEC'],
        ['CPA1','PTPRN2'],
        ['CREBZF','LYZ'],
        ['CRYAB','TNNT2'],
        ['CSNK1G2','OAZ1'],
        ['CST4','STATH'],
        ['CTAGE5','MIA2'],
        ['CTBS','GNG5'],
        ['CTDSP2','KLK3'],
        ['CYP19A1','IGF2'],
        ['CYP2E1','FGG'],
        ['CYP8B1','HRG'],
        ['CYTH1','USP36'],
        ['DCTD','PNLIP'],
        ['DDIT4','TRB@'],
        ['DDX5','PRSS1'],
        ['DES','HSPB6'],
        ['DES','LDHB'],
        ['DGKI','TG'],
        ['DHRS1','RABGGTA'],
        ['DICER1','HBB'],
        ['DNAH1','GLYCTK'],
        ['DSP','IGF2'],
        ['ECE1','TRB@'],
        ['EEF1DP3','FRY'],
        ['EEF2','LYZ'],
        ['ELK4','SLC45A3'],
        ['EPRS','HTN3'],
        ['ERBB2','GRB7'],
        ['EZR','SMR3B'],
        ['FAM129A','TRB@'],
        ['FAM13A-AS1','HERC3'],
        ['FAM175A','HELQ'],
        ['FARP1','TG'],
        ['FBXL5','LYZ'],
        ['FCAR','LYZ'],
        ['FCN1','MPO'],
        ['FDCSP','SMR3B'],
        ['FHOD3','NPPA'],
        ['FLG2','GSDMA'],
        ['FNDC3A','IGF2'],
        ['FOXO3','LYZ'],
        ['GANAB','LYZ'],
        ['GNB2L1','LYZ'],
        ['GOLT1A','KISS1'],
        ['GP2','HNRNPH1'],
        ['GP2','PNLIPRP1'],
        ['GSDMB','ORMDL3'],
        ['H1F0','HBB'],
        ['H3F3B','HBB'],
        ['HBB','HNRNPAB'],
        ['HBB','RPLP2'],
        ['HBB','RUNX3'],
        ['HBB','VIM-AS1'],
        ['HDLBP','PNLIP'],
        ['HEPHL1','PANX1'],
        ['HNRNPU-AS1','TG'],
        ['HNRNPUL1','PRB3'],
        ['HRG','SERPINA1'],
        ['HSP90AB1','SLC29A1'],
        ['HTN3','MGEA5'],
        ['HTN3','PITPNA'],
        ['HTN3','PRB3'],
        ['HTN3','USO1'],
        ['IGF2','MGRN1'],
        ['IGF2','PAFAH1B1'],
        ['IGF2','REEP5'],
        ['IGF2','SEC31A'],
        ['IGF2','SEPT9'],
        ['IGF2','TREML2'],
        ['IGFBP5','SMR3B'],
        ['IGH@','MYL2'],
        ['IGH@','PRB1'],
        ['IGH@','PRB2'],
        ['IGK@','MUC7'],
        ['IL17RA','LYZ'],
        ['IL1R1','PNLIP'],
        ['INTS1','TG'],
        ['KIAA1109','LYZ'],
        ['KIAA1324','TRB@'],
        ['KLHL3','RP11-381K20.2'],
        ['KLK4','KLKP1'],
        ['KRT13','PTPRF'],
        ['KTN1','TG'],
        ['LAMB1','TG'],
        ['LRG1','LYZ'],
        ['LRP10','LYZ'],
        ['LTF','STATH'],
        ['LYZ','PDLIM5'],
        ['LYZ','PRPF8'],
        ['LYZ','PRR27'],
        ['LYZ','RFWD3'],
        ['LYZ','RPS23'],
        ['LYZ','SMG1'],
        ['LYZ','SPTBN1'],
        ['LYZ','SRSF5'],
        ['LYZ','STAT1'],
        ['LYZ','TEF'],
        ['LYZ','TMEM56'],
        ['LYZ','TRIM58'],
        ['LYZ','ZNF394'],
        ['LYZ','ZNF431'],
        ['MAP6D1','PARL'],
        ['MAPKAPK2','MPO'],
        ['MAVS','PANK2'],
        ['MCL1','PNLIP'],
        ['METTL23','MFSD11'],
        ['MLLT4','PRSS1'],
        ['MSMB','NCOA4'],
        ['MTR','RYR2'],
        ['MTUS1','PNLIP'],
        ['MUC21','MYH11'],
        ['MUC7','PABPC1'],
        ['MUC7','TNS1'],
        ['MYH6','SRRM2'],
        ['NCOA6','PIGU'],
        ['NDRG2','PRH1-PRR4'],
        ['NDUFB8','SEC31B'],
        ['NECAB2','OSGIN1'],
        ['NPEPL1','STX16'],
        ['NQO1','THRIL'],
        ['NUP188','PHYHD1'],
        ['PAQR6','SMG5'],
        ['PARM1','PRH1-PRR4'],
        ['PDE8B','TG'],
        ['PDIA2','TRB@'],
        ['PIGR','STAT1'],
        ['PIM3','SCO2'],
        ['PLEKHO1','VPS45'],
        ['PMEPA1','PNLIP'],
        ['PRPF8','S100A9'],
        ['PRSS1','RPS9'],
        ['PTK7','ZG16B'],
        ['RFFL','TG'],
        ['RIPK1','SERPINB9'],
        ['RNASE1','TRB@'],
        ['RP11-356O9.1','TTC6'],
        ['RPS9','TRB@'],
        ['RYBP','SFTPC'],
        ['SERPINB3','SERPINB4'],
        ['SERPINF2','WDR81'],
        ['SH3BGR','WRB'],
        ['SIDT2','TAGLN'],
        ['SIPA1L3','TG'],
        ['SLC26A7','TG'],
        ['SLC41A3','TG'],
        ['SLC43A3','TG'],
        ['SMR3B','SREBF2'],
        ['SMR3B','TLN2'],
        ['SMR3B','TRIM29'],
        ['SMR3B','WNK2'],
        ['SMR3B','ZFP36L1'],
        ['SYT8','TNNI2'],
        ['TBCEL','TECTA'],
        ['TG','UBE2H'],
        ['TG','XRCC5'],
        ['TPPP3','ZDHHC1'],
        ['VWF','ZG16B'],
        ['AC007192.4','LYZ'],
        ['ACSS1','APMAP'],
        ['ACTB','KRT13'],
        ['ADGRG1','EEF1A1'],
        ['ADGRG7','TFG'],
        ['AL121578.2','SYTL5'],
        ['ALB','EEF1A1'],
        ['ALB','PRAP1'],
        ['ANP32B','NPM1'],
        ['AOX1','TRB@'],
        ['APP','PRSS1'],
        ['APP','PRSS2'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ART4','SMCO3'],
        ['ATP2A2','MYH6'],
        ['AZU1','EEF1A1'],
        ['B2M','EEF1A1'],
        ['BLOC1S6','LYZ'],
        ['CALD1','MYH11'],
        ['CCDC152','EEF1A1'],
        ['CCDC183','TMEM141'],
        ['CEL','CELP'],
        ['CMIP','PNLIP'],
        ['CPA1','ERN1'],
        ['CPA1','SPTBN1'],
        ['CPB1','UBA52'],
        ['CPQ','PNLIP'],
        ['CTSB','LYZ'],
        ['DAZAP2','LYZ'],
        ['DDR2','NPPA'],
        ['DDX5','EEF1A1'],
        ['ECE1','KRT13'],
        ['EEF1A1','G0S2'],
        ['EEF1A1','GP2'],
        ['EEF1A1','HBB'],
        ['EEF1A1','HMGB2'],
        ['EEF1A1','HNRNPA2B1'],
        ['EEF1A1','HNRNPH1'],
        ['EEF1A1','IGH@'],
        ['EEF1A1','KRT13'],
        ['EEF1A1','MPO'],
        ['EEF1A1','MYADM'],
        ['EEF1A1','PIGR'],
        ['EEF1A1','PTP4A1'],
        ['EEF1A1','SFTPB'],
        ['EEF1A1','SFTPC'],
        ['EEF1A1','SLC25A37'],
        ['EEF1A1','SRSF5'],
        ['EEF1A1','SRSF7'],
        ['EEF1A1','TGFBR3'],
        ['EEF1A1','TRB@'],
        ['EEF1DP3','FRY'],
        ['ERBB2','GRB7'],
        ['FN1','PRSS2'],
        ['FTH1','GP2'],
        ['GOLT1A','KISS1'],
        ['GS1-124K5.2','SKP1'],
        ['HDLBP','IGFBP5'],
        ['HSPE1','MOB4'],
        ['IGF2','MBNL1'],
        ['KDELR2','TRB@'],
        ['KIAA1324','PRSS1'],
        ['KRT13','PARD3'],
        ['KRT13','PKP1'],
        ['KRT13','SH3PXD2A'],
        ['KRT13','TNS1'],
        ['KRT4','MYO5B'],
        ['KRT4','RPS11'],
        ['KRT5','MYH9'],
        ['LARP1','TRB@'],
        ['LYZ','PSME4'],
        ['LYZ','SNRNP200'],
        ['METTL23','MFSD11'],
        ['NPPA','PRRC2B'],
        ['NUP188','PHYHD1'],
        ['PIM3','SCO2'],
        ['PNLIP','SND1'],
        ['PNLIP','TM9SF3'],
        ['PRKAA1','TTC33'],
        ['PRSS1','RPL3'],
        ['SERPINB3','SERPINB4'],
        ['SF1','TRB@'],
        ['TBCEL','TECTA'],
        ['AC004980.8','PRKRIP1'],
        ['AC005740.3','NDUFS1'],
        ['ACSS1','APMAP'],
        ['ACTB','AQP3'],
        ['ACTB','ITGB2'],
        ['ACTB','NAMPT'],
        ['ACTB','PTMA'],
        ['ADGRG7','TFG'],
        ['ADGRL1','SUZ12'],
        ['AGAP11','BMS1'],
        ['AL121578.2','SYTL5'],
        ['ANXA1','KRT4'],
        ['ARHGAP27','PLEKHM1'],
        ['ARL17A','KANSL1'],
        ['BANP','RRN3'],
        ['BEST1','LYZ'],
        ['BMS1P4','PARG'],
        ['BNIP3L','HBB'],
        ['C16ORF46','GCSH'],
        ['CCT6P1','VKORC1L1'],
        ['CD24','KRT13'],
        ['CD24','KRT4'],
        ['CD46','KRT4'],
        ['CDR2','RRN3'],
        ['CFL1','PIGR'],
        ['CIDEC','EMC3'],
        ['COX7C','KDM5B'],
        ['CRK','YWHAE'],
        ['CRNN','FLNA'],
        ['CRNN','HSPB1'],
        ['CTSC','RAB38'],
        ['CXCR4','PTMA'],
        ['CXCR4','UBB'],
        ['DCTN6','LYZ'],
        ['DDX5','EEF1D'],
        ['DGKZ','LYZ'],
        ['DPY30','RPS9P1'],
        ['DTX2','UPK3B'],
        ['EDARADD','ENO1'],
        ['EIF1','NAMPT'],
        ['EIF5','HMGB2'],
        ['EMP1','MYH11'],
        ['ERBB2','GRB7'],
        ['EXOSC6','SMG1'],
        ['FTH1','LYZ'],
        ['FTL','TPT1'],
        ['GAPDH','HBB'],
        ['GAPDH','KRT13'],
        ['GATSL2','GTF2I'],
        ['GCSH','NDUFS1'],
        ['GOLT1A','KISS1'],
        ['GTF2I','HBB'],
        ['H1FX-AS1','HBB'],
        ['HBB','NOP56'],
        ['HBB','NSD1'],
        ['HBB','PDIA3'],
        ['HBB','PRR13'],
        ['HBB','UBE2Z'],
        ['HIC2','PI4KA'],
        ['HMGB2','PTMA'],
        ['HMGN2','PTMA'],
        ['HNRNPA3','LYZ'],
        ['HSPE1','MOB4'],
        ['KCNMB3','LRRFIP1'],
        ['KLF6','NAMPT'],
        ['KRT13','TRIM29'],
        ['KRT13','YWHAE'],
        ['KRT4','MCM7'],
        ['KRT4','MYLK'],
        ['LINC00680','WDR70'],
        ['LINC00843','PARG'],
        ['LYZ','MOGS'],
        ['LYZ','PHF2'],
        ['LYZ','PIK3R2'],
        ['LYZ','PLIN3'],
        ['LYZ','SENP3-EIF4A1'],
        ['LYZ','TPM4'],
        ['LYZ','TSN'],
        ['MAP3K9','PGD'],
        ['MBP','TMSB4X'],
        ['NPIPB5','SMG1'],
        ['NPIPP1','PKD1'],
        ['OAZ1','PTMA'],
        ['PDIA3','PIGR'],
        ['PIM3','SCO2'],
        ['PRKAA1','TTC33'],
        ['PTMA','S100A9'],
        ['SLC7A5','SMG1'],
        ['ADGRL1','SUZ12'],
        ['AL122127.25','KIAA0125'],
        ['ALB','LRRD1'],
        ['ALB','SUCLG2'],
        ['AMD1','NNT-AS1'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ART4','SMCO3'],
        ['BANP','SNX29'],
        ['C16ORF46','GCSH'],
        ['CCDC183','TMEM141'],
        ['CCT8','FMO5'],
        ['CTSC','RAB38'],
        ['HNRNPU','NPPA-AS1'],
        ['IGFBP5','SVIL'],
        ['ITGA9-AS1','NDUFAF4'],
        ['METTL23','MFSD11'],
        ['MORF4L1','NPPA-AS1'],
        ['PMS2','PRKRIP1'],
        ['ROBO1','SFTPA2'],
        ['AC025335.1','FLG2'],
        ['ACPP','EEF1A1'],
        ['ACPP','ITGB1'],
        ['ACSS1','APMAP'],
        ['ACTB','HERC2'],
        ['ACTB','PIGR'],
        ['ACTN4','STATH'],
        ['ALDOA','PIGR'],
        ['ALDOA','SMR3B'],
        ['ANXA2','IGF2'],
        ['ANXA2','KRT10'],
        ['ANXA2','PNLIP'],
        ['AOX1','PRSS1'],
        ['ATP5A1','IGF2'],
        ['AZGP1','CA6'],
        ['AZGP1','MYH11'],
        ['BMS1','MBL1P'],
        ['BTG2','IGF2'],
        ['CCDC146','PMS2'],
        ['CEL','CELP'],
        ['CNDP2','LYZ'],
        ['CPA1','FAM162A'],
        ['DPP4','IGF2'],
        ['EEF1A1','KLK3'],
        ['EEF1A1','LYZ'],
        ['EEF1A1','PRH1-PRR4'],
        ['EFCAB14','LYZ'],
        ['EIF3L','IGF2'],
        ['EIF6','LYZ'],
        ['ELK4','SLC45A3'],
        ['FKBP8','LYZ'],
        ['GLTSCR2','TRB@'],
        ['GOLT1A','KISS1'],
        ['GTF2I','HTN1'],
        ['GTF2I','HTN3'],
        ['GTF2I','PNLIP'],
        ['GTF2I','PRSS1'],
        ['GTF2I','STATH'],
        ['HDLBP','IGF2'],
        ['HNRNPL','SPP1'],
        ['HTN1','TTC3'],
        ['HTN3','MSN'],
        ['HTN3','PDIA3'],
        ['IGF2','MSN'],
        ['IGF2','MYH10'],
        ['IGF2','VWF'],
        ['KLK1','TRB@'],
        ['KLK4','KLKP1'],
        ['KRT14','MYH9'],
        ['KRT8','LYZ'],
        ['LYZ','NMD3'],
        ['LYZ','PLPP3'],
        ['LYZ','PPIB'],
        ['LYZ','REXO2'],
        ['LYZ','XRCC6'],
        ['MED13L','PNLIP'],
        ['MED13L','SMR3B'],
        ['MEMO1','RPS9P1'],
        ['MUC7','NFIC'],
        ['MUC7','PFN1'],
        ['MUC7','PRB3'],
        ['MYH9','SMR3B'],
        ['NDRG2','PRH1'],
        ['NIPBL','TRB@'],
        ['NSF','PNLIP'],
        ['NUP188','PHYHD1'],
        ['PARG','TIMM23B'],
        ['PARM1','PRH1'],
        ['PDIA3','PNLIP'],
        ['PDIA3','STATH'],
        ['PNLIP','WLS'],
        ['PRSS1','TIMP2'],
        ['PTMA','SMR3B'],
        ['PYGB','TRB@'],
        ['RNF40','ZG16B'],
        ['SIRPA','SMR3B'],
        ['SMR3B','TNS1'],
        ['STATH','UBE2Z'],
        ['TRB@','TUBB'],
        ['TRB@','ULK1'],
        ['AC004980.9','PMS2'],
        ['AC005562.1','SMURF2'],
        ['AC006050.2','KRT42P'],
        ['ACSS1','APMAP'],
        ['ACTB','TIMM44'],
        ['ACTG1','LYZ'],
        ['AKR1B1','STAG3'],
        ['APLP2','EEF1A1'],
        ['ARL17B','KANSL1'],
        ['BCKDHA','TMEM91'],
        ['BEST1','EEF2'],
        ['BEST1','TG'],
        ['BOLA2B','SMG1'],
        ['CANX','HMGB3'],
        ['CAPN2','TG'],
        ['CDK2','RAB5B'],
        ['CEP97','PDCL3'],
        ['CFL1','TPM2'],
        ['CLU','EEF1A1'],
        ['CTBP2','WLS'],
        ['CTSB','EEF1A1'],
        ['CTSB','TPT1'],
        ['DHX40','TBC1D3P1-DHX40P1'],
        ['EEF1A1','SERPINA3'],
        ['EEF1A1','TG'],
        ['EEF1A1','TMBIM6'],
        ['FAH','ZFAND6'],
        ['FAM167A','TG'],
        ['FAM213A','LYZ'],
        ['FOXP1','TG'],
        ['GAPDH','PGC'],
        ['GOT2','TG'],
        ['IDS','LINC00893'],
        ['IGK@','STAG3'],
        ['JCHAIN','PGC'],
        ['KIAA1671','TG'],
        ['KIT','TG'],
        ['KRT42P','USP6'],
        ['LRBA','TG'],
        ['LRRC37B','SMURF2'],
        ['LYPLA1','TCEA1'],
        ['MLF2','TG'],
        ['MSMB','NCOA4'],
        ['NCOA6','PIGU'],
        ['NOL6','SUGT1'],
        ['NUP188','PHYHD1'],
        ['PSME4','TG'],
        ['PTMA','TG'],
        ['RAC1','TG'],
        ['STAG3','STAG3L5P-PVRIG2P-PILRB'],
        ['A2ML1','CRNN'],
        ['ABCA8','PIGR'],
        ['AC019349.5','AL513122.2'],
        ['AC027682.1','SFTPC'],
        ['AC096579.15','IGKV4-1'],
        ['ACTR2','LYZ'],
        ['AES','LYZ'],
        ['AHNAK','TRB@'],
        ['AL109923.1','DYNLRB1'],
        ['AL513122.2','KRT13'],
        ['ALB','CYP51A1'],
        ['ALB','ZNF511-PRAP1'],
        ['ALDOB','REG3A'],
        ['ARL6IP1','LYZ'],
        ['ATP5A1','LYZ'],
        ['BAIAP2','LYZ'],
        ['C7','HNRNPU'],
        ['CLSTN1','PRSS1'],
        ['CLSTN1','TRB@'],
        ['COL18A1','PRSS1'],
        ['COX6C','LYZ'],
        ['CPA1','SLC3A1'],
        ['CPB1','MAN1A2'],
        ['CPB1','PPIB'],
        ['CTSG','ITGB2'],
        ['CUL3','SMR3B'],
        ['DET1','MRPL46'],
        ['DHX40','RNFT1'],
        ['DNM2','SMR3B'],
        ['DOCK1','TG'],
        ['EEF1A1','PRH1'],
        ['EEF1D','PNLIP'],
        ['ENAH','MUC7'],
        ['ERRFI1','PNLIP'],
        ['FCGBP','HNRNPU'],
        ['H6PD','LYZ'],
        ['HIF1A','LYZ'],
        ['HTN3','IL6ST'],
        ['IGF2','IGFBP5'],
        ['IGF2','ITGB1'],
        ['IGF2','PLPP3'],
        ['IGF2','STAU1'],
        ['KIAA1191','LYZ'],
        ['LYZ','NDRG2'],
        ['LYZ','PLEC'],
        ['LYZ','PPP2CA'],
        ['LYZ','PRTN3'],
        ['LYZ','SLC12A6'],
        ['LYZ','SOCS7'],
        ['PDLIM1','TG'],
        ['PKD1','PKD1P6-NPIPP1'],
        ['PLCB4','SMR3B'],
        ['PLEC','SMR3B'],
        ['PNLIP','SLC25A26'],
        ['PNPLA7','TG'],
        ['PPIB','PRH1-PRR4'],
        ['RPL3','TG'],
        ['SERPINA1','SERPING1'],
        ['SERPINE1','TRB@'],
        ['SVIL','ZG16B'],
        ['AC019349.5','AL513122.1'],
        ['AC024560.3','SDHA'],
        ['ACCS','EXT2'],
        ['ACSS1','APMAP'],
        ['ADGRG7','TFG'],
        ['AL513122.1','KRT13'],
        ['AP3S1','COMMD10'],
        ['APBB3','SRA1'],
        ['ARL17B','KANSL1'],
        ['ART4','SMCO3'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['C19MC','RPLP1'],
        ['C2ORF48','RRM2'],
        ['C4ORF3','LYZ'],
        ['CANX','LYZ'],
        ['COQ8B','NUMBL'],
        ['CTSC','RAB38'],
        ['CYP2A6','CYP2A7'],
        ['DHX9','NPL'],
        ['FAM13A-AS1','HERC3'],
        ['GOLT1A','KISS1'],
        ['HBB','IQSEC1'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['KRT78','TNNT1'],
        ['LRRFIP1','LYZ'],
        ['LYZ','STAT6'],
        ['NACA','PRIM1'],
        ['PBXIP1','PMVK'],
        ['PIM3','SCO2'],
        ['PPP1R12B','TPM1'],
        ['RPL38','TTYH2'],
        ['SCNN1A','TNFRSF1A'],
        ['SIDT2','TAGLN'],
        ['AC019349.5','AL513122.1'],
        ['AC024560.3','SDHA'],
        ['ACCS','EXT2'],
        ['ACSS1','APMAP'],
        ['ADGRG7','TFG'],
        ['AL513122.1','KRT13'],
        ['AP3S1','COMMD10'],
        ['APBB3','SRA1'],
        ['ARL17B','KANSL1'],
        ['ART4','SMCO3'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['C19MC','RPLP1'],
        ['C2ORF48','RRM2'],
        ['C4ORF3','LYZ'],
        ['CANX','LYZ'],
        ['COQ8B','NUMBL'],
        ['CTSC','RAB38'],
        ['CYP2A6','CYP2A7'],
        ['DHX9','NPL'],
        ['FAM13A-AS1','HERC3'],
        ['GOLT1A','KISS1'],
        ['HBB','IQSEC1'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['KRT78','TNNT1'],
        ['LRRFIP1','LYZ'],
        ['LYZ','STAT6'],
        ['NACA','PRIM1'],
        ['PBXIP1','PMVK'],
        ['PIM3','SCO2'],
        ['PPP1R12B','TPM1'],
        ['RPL38','TTYH2'],
        ['SCNN1A','TNFRSF1A'],
        ['SIDT2','TAGLN'],
        ['A2ML1','ANXA2'],
        ['AC096664.1','RPL27'],
        ['ACCS','EXT2'],
        ['ACSS1','APMAP'],
        ['ADGRG7','TFG'],
        ['AP3S1','COMMD10'],
        ['ARL17B','KANSL1'],
        ['ART4','SMCO3'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['C19MC','RPLP1'],
        ['C2ORF48','RRM2'],
        ['CAST','KRT4'],
        ['CD36','MB'],
        ['CDC42EP2','POLA2'],
        ['COLQ','HACL1'],
        ['COQ8B','NUMBL'],
        ['CTSC','RAB38'],
        ['CYBRD1','MYH11'],
        ['DES','TCF4'],
        ['DES','ZBTB38'],
        ['DHX9','NPL'],
        ['EIF4E3','FOXP1'],
        ['FAM13A-AS1','HERC3'],
        ['GOLT1A','KISS1'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['KRT4','SUN2'],
        ['KRT4','TBC1D14'],
        ['LYZ','RAB18'],
        ['MYH11','NUMA1'],
        ['MYH11','PPL'],
        ['MYH6','TTN'],
        ['NACA','PRIM1'],
        ['NPEPL1','STX16'],
        ['NPPA','P4HB'],
        ['PBXIP1','PMVK'],
        ['PIM3','SCO2'],
        ['RPL38','TTYH2'],
        ['SCNN1A','TNFRSF1A'],
        ['SIDT2','TAGLN'],
        ['TBCEL','TECTA'],
        ['A2ML1','ANXA2'],
        ['AADAC','SUCNR1'],
        ['AC005336.4','IPMK'],
        ['AC008753.3','RPLP1'],
        ['AC096664.1','RPL27'],
        ['AC108488.4','ADI1'],
        ['ACPP','ERGIC3'],
        ['ALB','APLP2'],
        ['ASPH','TG'],
        ['CAST','KRT4'],
        ['CD36','MB'],
        ['CTTN','RACK1'],
        ['CYBRD1','MYH11'],
        ['CYP4F12','CYP4F24P'],
        ['DES','TCF4'],
        ['DES','ZBTB38'],
        ['DNAJB1','TG'],
        ['EEF1A1','SCMH1'],
        ['GKN1','NSD1'],
        ['KRT4','SUN2'],
        ['KRT4','TBC1D14'],
        ['LYZ','RAB18'],
        ['MYH11','NUMA1'],
        ['MYH11','PPL'],
        ['MYH6','TTN'],
        ['NOTCH3','TG'],
        ['NPPA','P4HB'],
        ['PCDHB15','PCDHB19P'],
        ['PDXDC1','RAC1'],
        ['SYTL4','TG'],
        ['TACC2','TG'],
        ['ZNF702P','ZNF816'],
        ['ZNF737','ZNF826P'],
        ['A2ML1','ANXA2'],
        ['AADAC','SUCNR1'],
        ['AC005336.4','IPMK'],
        ['AC008753.3','RPLP1'],
        ['AC096664.1','RPL27'],
        ['AC108488.4','ADI1'],
        ['ACCS','EXT2'],
        ['ACPP','ERGIC3'],
        ['ACSS1','APMAP'],
        ['ADGRG7','TFG'],
        ['ALB','APLP2'],
        ['AP3S1','COMMD10'],
        ['APBB3','SRA1'],
        ['ARL17B','KANSL1'],
        ['ART4','SMCO3'],
        ['ASPH','TG'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BCKDHA','TMEM91'],
        ['C19MC','RPLP1'],
        ['C1ORF194','UQCR10'],
        ['C2ORF48','RRM2'],
        ['CAST','KRT4'],
        ['CD36','MB'],
        ['CDC42EP2','POLA2'],
        ['CDK2','RAB5B'],
        ['COLQ','HACL1'],
        ['COQ8B','NUMBL'],
        ['CTSC','RAB38'],
        ['CTTN','RACK1'],
        ['CYBRD1','MYH11'],
        ['CYP4F12','CYP4F24P'],
        ['DES','TCF4'],
        ['DES','ZBTB38'],
        ['DHX9','NPL'],
        ['DNAJB1','TG'],
        ['EEF1A1','SCMH1'],
        ['EEF1A1','TG'],
        ['EIF4E3','FOXP1'],
        ['ELK4','SLC45A3'],
        ['FAM13A-AS1','HERC3'],
        ['GKN1','NSD1'],
        ['GOLT1A','KISS1'],
        ['HGSNAT','TG'],
        ['HSD17B14','PLEKHA4'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['KLK4','KLKP1'],
        ['KRT4','SUN2'],
        ['KRT4','TBC1D14'],
        ['LYZ','RAB18'],
        ['LYZ','SCYL1'],
        ['MSMB','NCOA4'],
        ['MUC7','RRBP1'],
        ['MYH11','NUMA1'],
        ['MYH11','PPL'],
        ['MYH6','TTN'],
        ['NACA','PRIM1'],
        ['NCOA6','PIGU'],
        ['NIT2','TBC1D23'],
        ['NOTCH3','TG'],
        ['NPEPL1','STX16'],
        ['NPPA','P4HB'],
        ['NUP188','PHYHD1'],
        ['PBXIP1','PMVK'],
        ['PCDHB15','PCDHB19P'],
        ['PDXDC1','RAC1'],
        ['PECAM1','TG'],
        ['PGM5','TGM4'],
        ['PIM3','SCO2'],
        ['PSPC1','ZMYM5'],
        ['PTPRJ','TG'],
        ['RASSF3','ZG16B'],
        ['RIPK1','SERPINB9'],
        ['RPL28','TMEM190'],
        ['RPL38','TTYH2'],
        ['SCNN1A','TNFRSF1A'],
        ['SIDT2','TAGLN'],
        ['SLC25A23','TG'],
        ['SLC6A16','TEAD2'],
        ['SPIN1','TG'],
        ['SYTL4','TG'],
        ['TACC2','TG'],
        ['TBCEL','TECTA'],
        ['TGM4','WDR6'],
        ['ZNF702P','ZNF816'],
        ['ZNF737','ZNF826P'],
        ['A2ML1','ANXA2'],
        ['AADAC','SUCNR1'],
        ['AC005336.4','IPMK'],
        ['AC008753.3','RPLP1'],
        ['AC096664.1','RPL27'],
        ['AC108488.4','ADI1'],
        ['ACCS','EXT2'],
        ['ACPP','ERGIC3'],
        ['ACSS1','APMAP'],
        ['ADGRG7','TFG'],
        ['AKAP13','CPA1'],
        ['ALB','APLP2'],
        ['ALDH7A1','CLPS'],
        ['ANXA1','STATH'],
        ['AP3S1','COMMD10'],
        ['APBB3','SRA1'],
        ['ARL17B','KANSL1'],
        ['ART4','SMCO3'],
        ['ASPH','TG'],
        ['ATF6','PNLIP'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BCKDHA','TMEM91'],
        ['BNIP3L','HTN1'],
        ['BNIP3L','STATH'],
        ['C19MC','RPLP1'],
        ['C1ORF194','UQCR10'],
        ['C2ORF48','RRM2'],
        ['CAPN1','SMR3B'],
        ['CAST','KRT4'],
        ['CD36','MB'],
        ['CD44','PRH2'],
        ['CD63','PNLIP'],
        ['CD63','SMR3B'],
        ['CD63','STATH'],
        ['CDC42EP2','POLA2'],
        ['CDK2','RAB5B'],
        ['CEL','CELP'],
        ['COLQ','HACL1'],
        ['COQ8B','NUMBL'],
        ['CPA1','GTF2I'],
        ['CPA1','LARP4B'],
        ['CPA1','MOB3A'],
        ['CPA1','PDIA3'],
        ['CPA1','RGL3'],
        ['CPA1','TSC2'],
        ['CTSC','RAB38'],
        ['CTTN','RACK1'],
        ['CYBRD1','MYH11'],
        ['CYP4F12','CYP4F24P'],
        ['DES','LGMN'],
        ['DES','TCF4'],
        ['DES','ZBTB38'],
        ['DHX9','NPL'],
        ['DNAJB1','TG'],
        ['DNMT1','PRSS1'],
        ['EEF1A1','PRR4'],
        ['EEF1A1','SCMH1'],
        ['EEF1A1','TG'],
        ['EIF4E3','FOXP1'],
        ['ELK4','SLC45A3'],
        ['FAM13A-AS1','HERC3'],
        ['GANC','HTN3'],
        ['GAREM1','TRB@'],
        ['GKN1','NSD1'],
        ['GOLT1A','KISS1'],
        ['GTF2I','SMR3B'],
        ['HEBP2','SMR3B'],
        ['HGSNAT','TG'],
        ['HSD17B14','PLEKHA4'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['HTN1','WWC1'],
        ['HTN3','TOPORS-AS1'],
        ['HTN3','TSC2'],
        ['IFT20','PNLIP'],
        ['KLK4','KLKP1'],
        ['KRT4','SUN2'],
        ['KRT4','TBC1D14'],
        ['LYZ','RAB18'],
        ['LYZ','SCYL1'],
        ['MAOB','PRH2'],
        ['MEG3','SNRNP200'],
        ['METTL21B','TSFM'],
        ['MLPH','SMR3B'],
        ['MSMB','NCOA4'],
        ['MTA3','PNLIP'],
        ['MUC7','RRBP1'],
        ['MYH11','NUMA1'],
        ['MYH11','PPL'],
        ['MYH6','TTN'],
        ['NACA','PRIM1'],
        ['NCOA3','PNLIP'],
        ['NCOA6','PIGU'],
        ['NIT2','TBC1D23'],
        ['NOTCH3','TG'],
        ['NPEPL1','STX16'],
        ['NPPA','P4HB'],
        ['NSD1','PNLIP'],
        ['NSD1','SMR3B'],
        ['NUP188','PHYHD1'],
        ['OS9','TRB@'],
        ['PBXIP1','PMVK'],
        ['PCDHB15','PCDHB19P'],
        ['PDIA3','PRSS1'],
        ['PDIA3','SMR3B'],
        ['PDXDC1','RAC1'],
        ['PECAM1','TG'],
        ['PGM5','TGM4'],
        ['PIM3','SCO2'],
        ['PNLIP','POM121'],
        ['PNLIP','PPARA'],
        ['PNLIP','RALY'],
        ['PNLIP','RNF10'],
        ['PNLIP','SCAMP5'],
        ['PNLIP','SDHA'],
        ['PNLIP','SLIT2'],
        ['PNLIP','VPS28'],
        ['PRB1','PSAP'],
        ['PRB1','PTCH2'],
        ['PRSS1','TSC2'],
        ['PSPC1','ZMYM5'],
        ['PTPRJ','TG'],
        ['RASSF3','ZG16B'],
        ['RIPK1','SERPINB9'],
        ['RPL28','TMEM190'],
        ['RPL38','TTYH2'],
        ['SAMD5','SASH1'],
        ['SCNN1A','TNFRSF1A'],
        ['SDHA','SMR3B'],
        ['SERINC2','SMR3B'],
        ['SH3GLB1','WSB1'],
        ['SIDT2','TAGLN'],
        ['SLC25A23','TG'],
        ['SLC6A16','TEAD2'],
        ['SMR3B','TSC2'],
        ['SMR3B','WFDC2'],
        ['SPIN1','TG'],
        ['SYTL4','TG'],
        ['TACC2','TG'],
        ['TBCEL','TECTA'],
        ['TGM4','WDR6'],
        ['ZNF702P','ZNF816'],
        ['ZNF737','ZNF826P'],
        ['AC010680.1','MYH6'],
        ['ACCS','EXT2'],
        ['ACSL1','HTN1'],
        ['ACSS1','APMAP'],
        ['ADGRG7','TFG'],
        ['ANP32E','HMGB1'],
        ['AP3S1','COMMD10'],
        ['APBB3','SRA1'],
        ['ARL17B','KANSL1'],
        ['ARL4A','NSUN7'],
        ['ART4','SMCO3'],
        ['ATP2A3','HBB'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BCKDHA','TMEM91'],
        ['C19MC','RPLP1'],
        ['C1ORF194','UQCR10'],
        ['C2ORF48','RRM2'],
        ['CDC42EP2','POLA2'],
        ['CDK2','RAB5B'],
        ['CEL','CELP'],
        ['COL14A1','TG'],
        ['COLQ','HACL1'],
        ['COQ8B','NUMBL'],
        ['CSDE1','STATH'],
        ['CTSC','RAB38'],
        ['CXCR4','HMGB1'],
        ['DDX5','HMGB1'],
        ['DHX9','NPL'],
        ['EEF1A1','TG'],
        ['EIF4E3','FOXP1'],
        ['ELK4','SLC45A3'],
        ['FAM13A-AS1','HERC3'],
        ['GOLT1A','KISS1'],
        ['GSTP1','HBB'],
        ['HSD17B14','PLEKHA4'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['HTN3','LGR4'],
        ['KLK4','KLKP1'],
        ['LINC00707','LINP1'],
        ['METTL21B','TSFM'],
        ['MSMB','NCOA4'],
        ['MUC7','RUNX1'],
        ['NACA','PRIM1'],
        ['NCOA6','PIGU'],
        ['NDST1','ZG16B'],
        ['NPEPL1','STX16'],
        ['NUP188','PHYHD1'],
        ['PBXIP1','PMVK'],
        ['PIM3','SCO2'],
        ['PSPC1','ZMYM5'],
        ['RIPK1','SERPINB9'],
        ['RPL28','TMEM190'],
        ['RPL38','TTYH2'],
        ['SAMD5','SASH1'],
        ['SCNN1A','TNFRSF1A'],
        ['SIDT2','TAGLN'],
        ['SLC6A16','TEAD2'],
        ['TG','VWF'],
        ['ZNF208','ZNF676'],
        ['AC006262.10','IGFL2-AS1'],
        ['ACCS','EXT2'],
        ['ACSS1','APMAP'],
        ['ADGRG7','TFG'],
        ['ADH1B','ALB'],
        ['AKT2','C19ORF47'],
        ['AP000344.4','LINC01659'],
        ['AP3S1','COMMD10'],
        ['APBB3','SRA1'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ART4','SMCO3'],
        ['ATG16L1','INPP5D'],
        ['AZGP1','GJC3'],
        ['BCAP29','DUS4L'],
        ['BCKDHA','TMEM91'],
        ['C15ORF57','CBX3'],
        ['C19MC','RPLP1'],
        ['C1ORF194','UQCR10'],
        ['C2ORF48','RRM2'],
        ['CCT3','PRB3'],
        ['CDC42EP2','POLA2'],
        ['CDK2','RAB5B'],
        ['CEL','CELP'],
        ['COLQ','HACL1'],
        ['COQ8B','NUMBL'],
        ['CRISP3','HTN3'],
        ['CTSC','RAB38'],
        ['DHX9','NPL'],
        ['EEF1A1','TG'],
        ['EIF4E3','FOXP1'],
        ['ELK4','SLC45A3'],
        ['ERBB2','GRB7'],
        ['FAM107A','SMR3B'],
        ['FAM13A-AS1','HERC3'],
        ['FGFR4','ZNF346'],
        ['FLJ22447','PRKCH'],
        ['GFAP','SPARC'],
        ['GOLT1A','KISS1'],
        ['HSD17B14','PLEKHA4'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['IGH@','MXI1'],
        ['ITGA4','LINC01934'],
        ['KLK4','KLKP1'],
        ['LINC00174','RABGEF1'],
        ['LINC00174','SKP1'],
        ['LINC01764','ZNF861P'],
        ['LINC01823','TSN'],
        ['LINC01859','ZNF728'],
        ['LINC01979','TBC1D16'],
        ['LINC02009','LTF'],
        ['LINC02019','MAPKAPK3'],
        ['LINC02023','SI'],
        ['LIX1L-AS1','POLR3GL'],
        ['METTL21B','TSFM'],
        ['MSMB','NCOA4'],
        ['NACA','PRIM1'],
        ['NCL','TRB@'],
        ['NCOA6','PIGU'],
        ['NPEPL1','STX16'],
        ['NPIPB5','SMG1'],
        ['NUP188','PHYHD1'],
        ['PBXIP1','PMVK'],
        ['PIM3','SCO2'],
        ['PNLIP','SCARB2'],
        ['PSPC1','ZMYM5'],
        ['RCAN1','TG'],
        ['RIPK1','SERPINB9'],
        ['RPL28','TMEM190'],
        ['RPL38','TTYH2'],
        ['SAMD5','SASH1'],
        ['SCNN1A','TNFRSF1A'],
        ['SIDT2','TAGLN'],
        ['SLC6A16','TEAD2'],
        ['AC008746.10','LYZ'],
        ['AC008993.2','XXYAC-YRM2039.3'],
        ['AC019349.5','KRT6A'],
        ['AC026150.8','GOLGA8M'],
        ['AC027612.6','IGK@'],
        ['ACCS','EXT2'],
        ['ACOT11','MROH7-TTC4'],
        ['ACSL1','IGH@'],
        ['ACSS1','APMAP'],
        ['ACTA1','CRYAB'],
        ['ACTA1','SDHA'],
        ['ACTB','ALB'],
        ['ACTB','C16ORF72'],
        ['ACTB','DDX5'],
        ['ACTB','GLUL'],
        ['ACTB','RPL15'],
        ['ACTB','SF3B1'],
        ['ACTB','SPINK5'],
        ['ACTN1','IGH@'],
        ['ADGRG7','TFG'],
        ['ADH1A','TF'],
        ['ADSL','TNRC6B'],
        ['AGAP5','BMS1P4'],
        ['AHR','LYZ'],
        ['AHSG','HP'],
        ['ALB','CHD4'],
        ['ALB','CYP1A2'],
        ['ALB','CYP4A11'],
        ['ALK','CDK12'],
        ['ANAPC1','RMND5A'],
        ['ANKRD20A21P','BP-2189O9.2'],
        ['ANKRD54','LYZ'],
        ['ANP32B','PTMA'],
        ['AP3S1','COMMD10'],
        ['APBB3','SRA1'],
        ['APP','KRT13'],
        ['AQP7','CH17-469D17.1'],
        ['ARHGAP1','NPPA'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARRDC3','HIST1H2BD'],
        ['ART4','SMCO3'],
        ['ATG16L1','INPP5D'],
        ['ATG16L2','IGK@'],
        ['ATP1B2','LINC00649'],
        ['ATP5G2','TFRC'],
        ['ATP5G3','NPPA-AS1'],
        ['AZGP1','GJC3'],
        ['B2M','CALM1'],
        ['B2M','HMGB1'],
        ['B2M','TMSB4X'],
        ['BANK1','IGH@'],
        ['BANK1','IGK@'],
        ['BANK1','OSGEP'],
        ['BCAP29','DUS4L'],
        ['BCL2','LYZ'],
        ['BCL9','TSPAN14'],
        ['BCLAF1','EEF1A1'],
        ['BGN','FAM102B'],
        ['BORCS7-ASMT','CCND3'],
        ['BPI','NUMA1'],
        ['BPTF','LRRC37A2'],
        ['C15ORF57','CBX3'],
        ['C19MC','RPLP1'],
        ['C1S','ORM2'],
        ['C2ORF48','RRM2'],
        ['C5AR1','LYZ'],
        ['C9ORF78','HBB'],
        ['CA1','MGEA5'],
        ['CATSPER2','PPIP5K1'],
        ['CCDC152','HP'],
        ['CCND3','FCAR'],
        ['CCND3','GRB2'],
        ['CCNG2','LRCH4'],
        ['CCNG2','S100A9'],
        ['CD164','WWOX'],
        ['CD24','IL3RA'],
        ['CD24','RACK1'],
        ['CD84','SLC26A2'],
        ['CDC42','S100A9'],
        ['CDC42EP2','POLA2'],
        ['CDH17','IGHA1'],
        ['CEACAM1','MUC2'],
        ['CES4A','TRA@'],
        ['CH17-164J11.1','IGK@'],
        ['CHGB','IGH@'],
        ['CHRDL1','CHST11'],
        ['CHST11','HIF1A'],
        ['CKMT2-AS1','LYZ'],
        ['CLU','GAPDH'],
        ['CMBL','WWOX'],
        ['CNN2','S100A9'],
        ['CNOT10','HBB'],
        ['COL1A1','PTGDS'],
        ['COLQ','HACL1'],
        ['COQ8B','NUMBL'],
        ['CPLX2','GLUL'],
        ['CSF2RA','LYZ'],
        ['CTSC','RAB38'],
        ['CUX1','HBB'],
        ['CWC25','LSINCT5'],
        ['CWC25','OSBPL2'],
        ['CXCL16','CYB561A3'],
        ['CXCR2','WSB1'],
        ['CXCR4','EEF1A1'],
        ['CXCR4','HNRNPA2B1'],
        ['CYP1A2','MICAL2'],
        ['CYP4V2','NSD2'],
        ['DA750114','TRA@'],
        ['DCTN5','GTDC1'],
        ['DDOST','TAS2R4'],
        ['DDX5','PTMA'],
        ['DENND3','HBA2'],
        ['DHODH','MYSM1'],
        ['DHX9','GLUL'],
        ['DHX9','NPL'],
        ['DKK3','MYH7'],
        ['DMBT1','IGH@'],
        ['DNAJB6','MPO'],
        ['DNAJC25-GNG10','UGCG'],
        ['DNMT1','LYZ'],
        ['DPYSL3','KRT13'],
        ['EBLN3P','ZCCHC7'],
        ['EEF1A1','HBA2'],
        ['EEF1A1','HMGB1'],
        ['EEF1A1','NFKBIZ'],
        ['EEF1A1','SLC4A1'],
        ['EEF1A1','SYNCRIP'],
        ['EEF1A1','UBC'],
        ['EIF4B','KCTD20'],
        ['EIF4E','PIK3C2A'],
        ['EIF4E3','FOXP1'],
        ['ELANE','HBA2'],
        ['EMX2OS','TRA@'],
        ['EPOR','SLC16A12'],
        ['ERBB2','GRB7'],
        ['ERVW-1','LINC01239'],
        ['ETV6','PIGR'],
        ['EWSR1','LYZ'],
        ['FAM13A-AS1','HERC3'],
        ['FAM157B','FBXO25'],
        ['FAM20B','TFDP2'],
        ['FBXL18','TTLL3'],
        ['FBXL5','HBB'],
        ['FCGBP','IGH@'],
        ['FCGBP','IGHA2'],
        ['FGD5-AS1','LINC00969'],
        ['FGFR2','TMPO'],
        ['FOSB','PPM1N'],
        ['FOXP1','GPRC5A'],
        ['FRG1BP','FRG1HP'],
        ['FRG1HP','FRG1JP'],
        ['FUS','PTMA'],
        ['GEMIN6','TSPAN14'],
        ['GINM1','RN7SL470P'],
        ['GLTP','MALL'],
        ['GLUL','PTMA'],
        ['GOLT1A','KISS1'],
        ['GRK2','IGH@'],
        ['GSE1','SLC7A5'],
        ['GUSBP11','LYZ'],
        ['GYPA','HBB'],
        ['HBB','HGS'],
        ['HBB','MICAL2'],
        ['HBB','MYH10'],
        ['HBB','MYO9B'],
        ['HBB','NSUN3'],
        ['HBB','PAFAH1B1'],
        ['HBB','PHLDA1'],
        ['HBB','PLXND1'],
        ['HBB','PTMA'],
        ['HBB','RBMS1'],
        ['HBB','RPS27'],
        ['HBB','RPS3'],
        ['HBB','SDHA'],
        ['HBB','SFI1'],
        ['HBB','SMARCA2'],
        ['HBB','SP100'],
        ['HBB','TACC3'],
        ['HBB','TSC2'],
        ['HBB','ZNF726'],
        ['HHLA3','TESPA1'],
        ['HMGB2','TMSB4X'],
        ['HMGN2','SEPT5'],
        ['HNRNPA1','KRT4'],
        ['HNRNPH1','PTP4A1'],
        ['HP','PLG'],
        ['HRG','ORM1'],
        ['HSP90AB1','SLC29A1'],
        ['HSPA5','MPO'],
        ['HSPA6','HSPA7'],
        ['HSPE1','MOB4'],
        ['IGH@','MUC2'],
        ['IGH@','NFKB2'],
        ['IGH@','PCSK7'],
        ['IGH@','PTMA'],
        ['IGH@','ZFP36L1'],
        ['IGK@','SAMHD1'],
        ['IL3RA','YIPF5'],
        ['KIAA2012','PEX26'],
        ['KRT13','MUC21'],
        ['KRT13','PFKL'],
        ['KRT13','RGS12'],
        ['KRT6A','MYH11'],
        ['KYNU','ZNF528-AS1'],
        ['LINC00491','LINC01807'],
        ['LINC01347','NUPR2'],
        ['LINC01641','RPS18'],
        ['LLFOS-48D6.2','PTMA'],
        ['LMX1B','PAPOLG'],
        ['LRIG2','ZNF106'],
        ['LRRC37A3','NSF'],
        ['LYZ','MAU2'],
        ['LYZ','NHSL2'],
        ['LYZ','NIPAL2'],
        ['LYZ','PAXBP1-AS1'],
        ['LYZ','PRAM1'],
        ['LYZ','SPATS2L'],
        ['LYZ','UBXN2A'],
        ['LYZ','WWOX'],
        ['LYZ','XIST'],
        ['MATR3','PAIP2'],
        ['MC2R','SH3PXD2A'],
        ['MED9','SLC37A2'],
        ['MPO','RPS21'],
        ['MRPS10','PPM1B'],
        ['MS4A10','SP140L'],
        ['MYH11','PABPC1'],
        ['MYH6','PLIN4'],
        ['MYH6','ZNF106'],
        ['MYH7','PSMC4'],
        ['MYH9','RPL13A'],
        ['MYL3','PRSS44'],
        ['NACA','PRIM1'],
        ['NAIP','OCLN'],
        ['NPEPL1','STX16'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['NPPA','RAPGEF1'],
        ['OAZ2','SIK2'],
        ['PBXIP1','PMVK'],
        ['PCDHB13','PCDHB8'],
        ['PIGR','TRA@'],
        ['PIM3','SCO2'],
        ['PLIN1','WWOX'],
        ['PMS2','POLR2J3'],
        ['PPP1R12B','ZNF623'],
        ['RAB21','RNF24'],
        ['RAB2B','TOP3A'],
        ['RASSF8','SSPN'],
        ['RBM25','S100A9'],
        ['RFWD3','RSL1D1'],
        ['RHOA','TKT'],
        ['RHOH','SLC31A1'],
        ['RPL14','SRP14'],
        ['RPL37A','SRGN'],
        ['RPL38','TTYH2'],
        ['RPS14','S100A9'],
        ['S100A9','YWHAZ'],
        ['SCNN1A','TNFRSF1A'],
        ['SELENOP','TPT1'],
        ['SERF1A','SMN1'],
        ['SFN','TNS1'],
        ['SIDT2','TAGLN'],
        ['SLC8A1','TPM1'],
        ['STAT3','TRA@'],
        ['TBCEL','TECTA'],
        ['TRA@','TRIM69'],
        ['TUBB7P','TUBB8'],
        ['ABR','PRH1-PRR4'],
        ['AC003973.4','LINC01859'],
        ['AC005077.7','LINC00174'],
        ['AC006004.1','FLVCR1'],
        ['AC007099.1','RGS6'],
        ['AC008132.13','CALM1'],
        ['AC008132.13','USP18'],
        ['AC008746.10','LYZ'],
        ['AC008993.2','XXYAC-YRM2039.3'],
        ['AC015849.16','LRRC37A9P'],
        ['AC019349.5','KRT6A'],
        ['AC025335.1','DNAH14'],
        ['AC026150.8','GOLGA8M'],
        ['AC027612.6','IGK@'],
        ['AC097374.2','BP-2171C21.2'],
        ['ACCS','EXT2'],
        ['ACIN1','WTIP'],
        ['ACOT11','MROH7-TTC4'],
        ['ACSL1','HTN3'],
        ['ACSL1','IGH@'],
        ['ACSS1','APMAP'],
        ['ACTA1','CRYAB'],
        ['ACTA1','SDHA'],
        ['ACTB','ALB'],
        ['ACTB','C16ORF72'],
        ['ACTB','DDX5'],
        ['ACTB','EEF1A1'],
        ['ACTB','GLUL'],
        ['ACTB','PRB2'],
        ['ACTB','PRH1-PRR4'],
        ['ACTB','RPL15'],
        ['ACTB','SF3B1'],
        ['ACTB','SPINK5'],
        ['ACTB','SPTAN1'],
        ['ACTN1','IGH@'],
        ['ADGRG1','FCGBP'],
        ['ADGRG7','TFG'],
        ['ADH1A','TF'],
        ['ADSL','TNRC6B'],
        ['AFMID','SLC48A1'],
        ['AGAP5','BMS1P4'],
        ['AGPS','USP34'],
        ['AHNAK','LIPF'],
        ['AHR','LYZ'],
        ['AHSA2','TIAL1'],
        ['AHSG','HP'],
        ['AKT2','C19ORF47'],
        ['AKT2','TRB@'],
        ['AL050303.11','LINC01087'],
        ['AL050303.11','LINC01297'],
        ['ALB','CHD4'],
        ['ALB','CYP1A2'],
        ['ALB','CYP4A11'],
        ['ALDH18A1','SMR3B'],
        ['ALDH1L1','HTN3'],
        ['ALG1L5P','TSSC2'],
        ['ALK','CDK12'],
        ['AMY2A','CELA2A'],
        ['AMY2A','CELA3A'],
        ['AMY2A','CELA3B'],
        ['AMY2A','CTRB1'],
        ['AMY2A','PNLIP'],
        ['AMY2A','PRSS1'],
        ['AMY2B','PRH1-PRR4'],
        ['AMY2B','PRH2'],
        ['AMY2B','PRSS1'],
        ['ANAPC1','RMND5A'],
        ['ANAPC7','KDSR'],
        ['ANKRD18CP','ZNF322'],
        ['ANKRD20A21P','BP-2189O9.2'],
        ['ANKRD54','LYZ'],
        ['ANKRD62','VWF'],
        ['ANP32B','IGH@'],
        ['ANP32B','PTMA'],
        ['ANXA1','SMR3B'],
        ['ANXA2','PRB2'],
        ['AP000351.4','GSTTP1'],
        ['AP005530.1','C2ORF27B'],
        ['AP3S1','COMMD10'],
        ['APBB2','SPAG9'],
        ['APBB3','SRA1'],
        ['APLP2','HSP90B1'],
        ['APP','KRT13'],
        ['AQP7','CH17-469D17.1'],
        ['ARAP1','PRB3'],
        ['AREL1','TG'],
        ['ARHGAP1','NPPA'],
        ['ARHGAP44','MAGT1'],
        ['ARID1A','TG'],
        ['ARIH2','TG'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARL6IP1','RPS15A'],
        ['ARRDC3','HIST1H2BD'],
        ['ART4','SMCO3'],
        ['ASCC1','TG'],
        ['ASL','GUSBP11'],
        ['ASPH','PRB3'],
        ['ATG16L1','INPP5D'],
        ['ATG16L2','IGK@'],
        ['ATL2','TG'],
        ['ATP13A1','TG'],
        ['ATP1B2','LINC00649'],
        ['ATP5G2','TFRC'],
        ['ATP5G3','NPPA-AS1'],
        ['AUTS2','PRB3'],
        ['AZGP1','GJC3'],
        ['AZGP1','PRB4'],
        ['B2M','CALM1'],
        ['B2M','HMGB1'],
        ['B2M','TMSB4X'],
        ['B3GAT3','FCGBP'],
        ['BANK1','CTSB'],
        ['BANK1','IGH@'],
        ['BANK1','IGK@'],
        ['BANK1','OSGEP'],
        ['BCAP29','DUS4L'],
        ['BCAP29','SLC26A4'],
        ['BCKDHA','TMEM91'],
        ['BCL2','LYZ'],
        ['BCL9','TSPAN14'],
        ['BCLAF1','EEF1A1'],
        ['BGN','FAM102B'],
        ['BMS1P4','TIMM23B'],
        ['BORCS7-ASMT','CCND3'],
        ['BP-21264C1.2','MAFIP'],
        ['BPI','NUMA1'],
        ['BPIFA2','MUC7'],
        ['BPTF','LRRC37A2'],
        ['C12ORF49','TG'],
        ['C15ORF57','CBX3'],
        ['C19MC','RPLP1'],
        ['C1ORF194','UQCR10'],
        ['C1S','ORM2'],
        ['C2ORF48','RRM2'],
        ['C5AR1','LYZ'],
        ['C7','RPS4X'],
        ['C9ORF78','HBB'],
        ['CA1','MGEA5'],
        ['CA6','PRH1-PRR4'],
        ['CALR','H19'],
        ['CAPZB','SMR3B'],
        ['CARF','IQUB'],
        ['CASP9','PNLIP'],
        ['CATSPER2','PPIP5K1'],
        ['CCDC152','HP'],
        ['CCL28','PRH1-PRR4'],
        ['CCND3','F11R'],
        ['CCND3','FCAR'],
        ['CCND3','GRB2'],
        ['CCNG2','LRCH4'],
        ['CCNG2','S100A9'],
        ['CD164','WWOX'],
        ['CD24','IGH@'],
        ['CD24','IL3RA'],
        ['CD24','MUC7'],
        ['CD24','RACK1'],
        ['CD2AP','DBT'],
        ['CD63','CPB1'],
        ['CD74','PRH1-PRR4'],
        ['CD74','PRH2'],
        ['CD84','SLC26A2'],
        ['CD9','CST4'],
        ['CDC42','S100A9'],
        ['CDC42EP2','POLA2'],
        ['CDH12','LINC02197'],
        ['CDH17','IGHA1'],
        ['CDK2','RAB5B'],
        ['CDK6','HTN3'],
        ['CEACAM1','MUC2'],
        ['CEL','CELA3A'],
        ['CEL','CELA3B'],
        ['CEL','CELP'],
        ['CEL','CPA2'],
        ['CEL','PRSS3'],
        ['CEL','SLC7A2'],
        ['CELA3A','FLNA'],
        ['CELA3B','TPT1'],
        ['CELP','TRB@'],
        ['CES4A','TRA@'],
        ['CGNL1','SFTPA1'],
        ['CH17-164J11.1','IGK@'],
        ['CH17-264L24.1','IGK@'],
        ['CH17-53B9.2','NCOR1'],
        ['CHCHD2','PHKG1'],
        ['CHGB','IGH@'],
        ['CHI3L2','SMR3B'],
        ['CHRDL1','CHST11'],
        ['CHRM3','PRB3'],
        ['CHST11','HIF1A'],
        ['CKMT2-AS1','LYZ'],
        ['CLCC1','KLHDC4'],
        ['CLCN3','LIPG'],
        ['CLEC2D','IGH@'],
        ['CLPS','CTRB1'],
        ['CLU','GAPDH'],
        ['CMBL','WWOX'],
        ['CNN2','S100A9'],
        ['CNNM2','LYZ'],
        ['CNOT10','HBB'],
        ['CNOT4','SPTLC2'],
        ['COASY','HSD17B1'],
        ['COL1A1','PTGDS'],
        ['COL6A2','KRT14'],
        ['COLQ','HACL1'],
        ['COQ8B','NUMBL'],
        ['CORO1A','EEF1A1'],
        ['COX7C','ZG16B'],
        ['CPB1','CTRB2'],
        ['CPB1','DDX24'],
        ['CPB1','GOLGA8B'],
        ['CPB1','REG1A'],
        ['CPB1','STAT6'],
        ['CPD','HTN3'],
        ['CPLX2','GLUL'],
        ['CPT1A','TG'],
        ['CSF2RA','LYZ'],
        ['CSH1','H19'],
        ['CST1','LPO'],
        ['CST2','IGK@'],
        ['CST2','STATH'],
        ['CTBP1','SMR3B'],
        ['CTRB1','SYCN'],
        ['CTRB2','GNAS'],
        ['CTSC','RAB38'],
        ['CTSD','PRB1'],
        ['CTSS','MAN1C1'],
        ['CTTN','IGK@'],
        ['CUX1','HBB'],
        ['CWC25','LSINCT5'],
        ['CWC25','OSBPL2'],
        ['CX3CL1','PRH1-PRR4'],
        ['CX3CL1','PRH2'],
        ['CXCL16','CYB561A3'],
        ['CXCR2','WSB1'],
        ['CXCR4','EEF1A1'],
        ['CXCR4','HNRNPA2B1'],
        ['CXXC5','TG'],
        ['CYBA','EML4'],
        ['CYP1A2','MICAL2'],
        ['CYP3A4','HECW1'],
        ['CYP4F34P','CYP4F35P'],
        ['CYP4V2','NSD2'],
        ['DA750114','TRA@'],
        ['DCTN2','PRB2'],
        ['DCTN5','GTDC1'],
        ['DDOST','SMR3B'],
        ['DDOST','TAS2R4'],
        ['DDX5','PTMA'],
        ['DENND3','HBA2'],
        ['DGCR2','PRH1-PRR4'],
        ['DGCR2','PRH2'],
        ['DGCR6','XXBAC-B444P24.13'],
        ['DGKZ','PRB3'],
        ['DHODH','MYSM1'],
        ['DHRS3','MAPK1'],
        ['DHX9','GLUL'],
        ['DHX9','NPL'],
        ['DIABLO','TG'],
        ['DIS3L2','SMR3B'],
        ['DKK3','MYH7'],
        ['DMBT1','IGH@'],
        ['DMBT1','PRB2'],
        ['DNAJB2','PRH1'],
        ['DNAJB6','MPO'],
        ['DNAJC25-GNG10','UGCG'],
        ['DNMT1','LYZ'],
        ['DPCR1','IGH@'],
        ['DPCR1','MUC5AC'],
        ['DPH5','TG'],
        ['DPYSL3','KRT13'],
        ['DSC3','SKP2'],
        ['DSP','EEF1A1'],
        ['DUXAP8','LINC01297'],
        ['DYM','STATH'],
        ['EBLN3P','ZCCHC7'],
        ['ECI2','PRSS1'],
        ['EEF1A1','FOS'],
        ['EEF1A1','HBA2'],
        ['EEF1A1','HMGB1'],
        ['EEF1A1','ITGB1'],
        ['EEF1A1','NFKBIZ'],
        ['EEF1A1','RPS4X'],
        ['EEF1A1','SLC4A1'],
        ['EEF1A1','SYNCRIP'],
        ['EEF1A1','TG'],
        ['EEF1A1','UBC'],
        ['EEF1A1','XPR1'],
        ['EEF1D','TG'],
        ['EEF2','IGKC'],
        ['EEF2','PRH1-PRR4'],
        ['EFCAB10','YBX1'],
        ['EGFR','ZG16B'],
        ['EIF3B','HTN3'],
        ['EIF3E','HTN1'],
        ['EIF3E','TRB@'],
        ['EIF4B','KCTD20'],
        ['EIF4E','PIK3C2A'],
        ['EIF4E3','FOXP1'],
        ['ELANE','HBA2'],
        ['ELK4','SLC45A3'],
        ['ELOVL6','MUC5AC'],
        ['EMX2OS','TRA@'],
        ['ENAH','LYZ'],
        ['EPOR','SLC16A12'],
        ['ERBB2','GRB7'],
        ['ERVW-1','LINC01239'],
        ['ETV6','PIGR'],
        ['EWSR1','KRT10'],
        ['EWSR1','LYZ'],
        ['FAM129A','HTN3'],
        ['FAM134C','TEDDM1'],
        ['FAM13A-AS1','HERC3'],
        ['FAM157B','FBXO25'],
        ['FAM157C','RPL23AP53'],
        ['FAM20B','TFDP2'],
        ['FARP1','KRT10'],
        ['FBXL18','TTLL3'],
        ['FBXL5','HBB'],
        ['FCGBP','IGH@'],
        ['FCGBP','IGHA2'],
        ['FCGBP','SMR3B'],
        ['FGD5-AS1','LINC00969'],
        ['FGFR2','PNLIP'],
        ['FGFR2','TMPO'],
        ['FGFR4','ZNF346'],
        ['FICD','ISCU'],
        ['FKRP','TG'],
        ['FLJ22447','PRKCH'],
        ['FLOT1','SMR3B'],
        ['FMO4','TOP1'],
        ['FOSB','PPM1N'],
        ['FOXJ2','TG'],
        ['FOXP1','GPRC5A'],
        ['FOXP1','HECA'],
        ['FOXP1','IFNAR2'],
        ['FOXP4','PRH1'],
        ['FRG1BP','FRG1HP'],
        ['FRG1HP','FRG1JP'],
        ['FRG2C','FRG2EP'],
        ['FTH1','PRB3'],
        ['FUS','PTMA'],
        ['GABPB1-AS1','NSD2'],
        ['GALNT10','SFTPA1'],
        ['GAPDH','H19'],
        ['GAPDH','MUC7'],
        ['GAPDH','MYO1A'],
        ['GAPDH','PRH1-PRR4'],
        ['GATM-AS1','SMG1'],
        ['GBA','TG'],
        ['GCC2-AS1','PLGLB1'],
        ['GEMIN6','TSPAN14'],
        ['GGT1','IGSF3'],
        ['GINM1','RN7SL470P'],
        ['GLG1','HTN3'],
        ['GLTP','MALL'],
        ['GLTP','NEK5'],
        ['GLUL','PTMA'],
        ['GNAS','IGF2'],
        ['GNAS','IGH@'],
        ['GNAS','RGS5'],
        ['GNE','PRH1-PRR4'],
        ['GOLT1A','KISS1'],
        ['GP2','GSN'],
        ['GP2','MAN1B1-AS1'],
        ['GP2','MEG3'],
        ['GPR155','OTUB1'],
        ['GPRC5A','MIR100HG'],
        ['GRK2','IGH@'],
        ['GSE1','SLC7A5'],
        ['GSN','SMR3B'],
        ['GSN','TRB@'],
        ['GSTM4','MUC7'],
        ['GUSBP11','LYZ'],
        ['GYG2','TG'],
        ['GYPA','HBB'],
        ['H19','NRP1'],
        ['H19','PFN1'],
        ['H19','PLAC4'],
        ['H19','PLPPR2'],
        ['H19','TBC1D14'],
        ['H19','TPT1'],
        ['H1F0','PRB1'],
        ['H1F0','PRH1-PRR4'],
        ['HBB','HGS'],
        ['HBB','MICAL2'],
        ['HBB','MYH10'],
        ['HBB','MYO9B'],
        ['HBB','NSUN3'],
        ['HBB','PAFAH1B1'],
        ['HBB','PHLDA1'],
        ['HBB','PLXND1'],
        ['HBB','PTMA'],
        ['HBB','RBMS1'],
        ['HBB','RPS27'],
        ['HBB','RPS3'],
        ['HBB','SDHA'],
        ['HBB','SFI1'],
        ['HBB','SMARCA2'],
        ['HBB','SP100'],
        ['HBB','TACC3'],
        ['HBB','TSC2'],
        ['HBB','ZNF726'],
        ['HERC1','TG'],
        ['HHLA3','TESPA1'],
        ['HKR1','LINC00665'],
        ['HKR1','ZNF790-AS1'],
        ['HMGB1','PNLIP'],
        ['HMGB2','TMSB4X'],
        ['HMGN2','SEPT5'],
        ['HNRNPA1','KRT4'],
        ['HNRNPA1L2','MRPS31P4'],
        ['HNRNPA3','SFTPC'],
        ['HNRNPH1','PTP4A1'],
        ['HNRNPUL1','SMR3B'],
        ['HP','PLG'],
        ['HRG','ORM1'],
        ['HSBP1L1','ZNF554'],
        ['HSD17B14','PLEKHA4'],
        ['HSP90AB1','SLC29A1'],
        ['HSPA5','MPO'],
        ['HSPA6','HSPA7'],
        ['HSPE1','MOB4'],
        ['HTN1','NAALADL2'],
        ['HTN1','PDIA6'],
        ['HTN1','PLA2R1'],
        ['HTN1','ZG16B'],
        ['HTN3','LARP1'],
        ['HTN3','LRRC26'],
        ['HTN3','MGP'],
        ['HTN3','PRH2'],
        ['IGF2','NAP1L1'],
        ['IGFBP5','RPL30'],
        ['IGH@','LINC00926'],
        ['IGH@','MUC1'],
        ['IGH@','MUC2'],
        ['IGH@','MXRA7'],
        ['IGH@','NDRG1'],
        ['IGH@','NFKB2'],
        ['IGH@','NPM1'],
        ['IGH@','PCSK7'],
        ['IGH@','PRH2'],
        ['IGH@','PRR4'],
        ['IGH@','PTMA'],
        ['IGH@','PTRF'],
        ['IGH@','RAP1GAP'],
        ['IGH@','SDC1'],
        ['IGH@','SFTPC'],
        ['IGH@','SLC25A6'],
        ['IGH@','SRRM2'],
        ['IGH@','TAGLN2'],
        ['IGH@','TRA@'],
        ['IGH@','ZFP36L1'],
        ['IGHA1','LTF'],
        ['IGHA1','MUC1'],
        ['IGHA1','MVP'],
        ['IGHA1','MYO1A'],
        ['IGHA1','PFN1'],
        ['IGK@','KNL1'],
        ['IGK@','LINC01123'],
        ['IGK@','PRH2'],
        ['IGK@','SAMHD1'],
        ['IGK@','STATH'],
        ['IGK@','TRA@'],
        ['IGLL5','SMR3B'],
        ['IL3RA','YIPF5'],
        ['IL6ST','LYZ'],
        ['IL6ST','PRH1-PRR4'],
        ['ITGB4','SMR3B'],
        ['JADRR','KCNE3'],
        ['KANSL1','LRRC37A3'],
        ['KDM2A','NCALD'],
        ['KIAA1217','SMR3B'],
        ['KIAA2012','PEX26'],
        ['KIF13B','SMR3B'],
        ['KIF1C','SMR3B'],
        ['KLK3','PLPP3'],
        ['KLK4','KLKP1'],
        ['KMT2C','TPTE'],
        ['KRT13','MUC21'],
        ['KRT13','PFKL'],
        ['KRT13','RGS12'],
        ['KRT6A','MYH11'],
        ['KYNU','ZNF528-AS1'],
        ['LENG8','PRSS1'],
        ['LINC00378','LINC01442'],
        ['LINC00491','LINC01807'],
        ['LINC00665','ZNF790-AS1'],
        ['LINC01251','TRB@'],
        ['LINC01347','NUPR2'],
        ['LINC01641','RPS18'],
        ['LLFOS-48D6.2','PTMA'],
        ['LMX1B','PAPOLG'],
        ['LRIG2','ZNF106'],
        ['LRRC37A3','NSF'],
        ['LTBR','SPEF2'],
        ['LYZ','MAU2'],
        ['LYZ','NHSL2'],
        ['LYZ','NIPAL2'],
        ['LYZ','PAM'],
        ['LYZ','PAXBP1-AS1'],
        ['LYZ','PRAM1'],
        ['LYZ','SAMD4B'],
        ['LYZ','SETX'],
        ['LYZ','SPATS2L'],
        ['LYZ','TAPBP'],
        ['LYZ','TATDN2'],
        ['LYZ','TMEM184C'],
        ['LYZ','TNS3'],
        ['LYZ','TRB@'],
        ['LYZ','UBXN2A'],
        ['LYZ','WWOX'],
        ['LYZ','XIST'],
        ['MATR3','PAIP2'],
        ['MBNL1','SMR3B'],
        ['MC2R','SH3PXD2A'],
        ['MDM4','RAB30-AS1'],
        ['MDM4','TRA@'],
        ['MED9','SLC37A2'],
        ['MEG3','MYH9'],
        ['MEIS1','STATH'],
        ['METTL21B','TSFM'],
        ['MFSD13A','SLC25A29'],
        ['MINPP1','NUTM2A-AS1'],
        ['MN1','NCOA3'],
        ['MPO','RPS21'],
        ['MRPS10','PPM1B'],
        ['MS4A10','SP140L'],
        ['MS4A7','TCTN1'],
        ['MSMB','NCOA4'],
        ['MT1E','MT1L'],
        ['MTMR9','YIPF5'],
        ['MUC7','NDRG2'],
        ['MUC7','PTPRF'],
        ['MUC7','SLC35E2B'],
        ['MYH11','PABPC1'],
        ['MYH6','PLIN4'],
        ['MYH6','ZNF106'],
        ['MYH7','PSMC4'],
        ['MYH9','RPL13A'],
        ['MYL3','PRSS44'],
        ['NAA35','RPL7L1'],
        ['NACA','PRIM1'],
        ['NAIP','OCLN'],
        ['NBPF13P','PDE4DIP'],
        ['NCOA6','PIGU'],
        ['NFIX','PRB3'],
        ['NFIX','PRH2'],
        ['NORAD','TRB@'],
        ['NOTCH2','SMR3B'],
        ['NPEPL1','STX16'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['NPPA','RAPGEF1'],
        ['NSUN5','TRIM50'],
        ['NSUN5','TRIM73'],
        ['NUDCD3','TG'],
        ['NUDT4','PNLIP'],
        ['NUMA1','SMR3B'],
        ['NUP188','PHYHD1'],
        ['OAZ2','SIK2'],
        ['OCIAD1','TG'],
        ['ODAM','PRH1-PRR4'],
        ['ODAM','PRH2'],
        ['OPRPN','SMR3B'],
        ['PABPC1','RPS19'],
        ['PAFAH1B2','SUGP2'],
        ['PBX1','PRB2'],
        ['PBXIP1','PMVK'],
        ['PCBP1-AS1','STATH'],
        ['PCBP2','PHF7'],
        ['PCDHB13','PCDHB8'],
        ['PDE1C','SNX1'],
        ['PDXDC1','PRSS1'],
        ['PDXDC1','SMR3B'],
        ['PEX26','PLIN3'],
        ['PIGN','PNLIP'],
        ['PIGR','TRA@'],
        ['PIK3R1','PRH1-PRR4'],
        ['PIM3','SCO2'],
        ['PKD1','PRB3'],
        ['PKP4','PRB3'],
        ['PLCB1','STATH'],
        ['PLEKHM1','TG'],
        ['PLIN1','WWOX'],
        ['PMS2','POLR2J3'],
        ['PNLIP','SMIM19'],
        ['PNLIP','TMSB4X'],
        ['POLR2J4','RASA4'],
        ['PPIP5K1','STRC'],
        ['PPP1R12B','ZNF623'],
        ['PRB1','PRSS8'],
        ['PRB2','ZG16B'],
        ['PRH1-PRR4','PRR27'],
        ['PRH1-PRR4','PSMF1'],
        ['PRH1-PRR4','RBM47'],
        ['PRH1-PRR4','RPL3'],
        ['PRH1-PRR4','SFRP1'],
        ['PRH1-PRR4','SIX4'],
        ['PRH1-PRR4','SLC20A2'],
        ['PRH1-PRR4','SLC6A8'],
        ['PRH1-PRR4','SORL1'],
        ['PRH1-PRR4','SQSTM1'],
        ['PRH2','RBM47'],
        ['PRH2','SFRP1'],
        ['PRH2','SLC6A8'],
        ['PRH2','SQSTM1'],
        ['PRPF8','PRSS1'],
        ['PRRC2A','SMR3B'],
        ['PRSS1','ZNF706'],
        ['PSD4','SMR3B'],
        ['PSPC1','ZMYM5'],
        ['RAB21','RNF24'],
        ['RAB2B','TOP3A'],
        ['RACK1','TRIM52'],
        ['RASSF8','SSPN'],
        ['RBM25','S100A9'],
        ['RCOR3','TG'],
        ['RFWD3','RSL1D1'],
        ['RHOA','TKT'],
        ['RHOH','SLC31A1'],
        ['RIPK1','SERPINB9'],
        ['RNF19A','TG'],
        ['RNF4','TG'],
        ['RPL14','SRP14'],
        ['RPL14','VSIG1'],
        ['RPL15','SMR3B'],
        ['RPL15','TG'],
        ['RPL28','TMEM190'],
        ['RPL37A','SRGN'],
        ['RPL38','TTYH2'],
        ['RPL4','TG'],
        ['RPL7L1','WBP2NL'],
        ['RPS11','SMR3B'],
        ['RPS14','S100A9'],
        ['RPS19','TG'],
        ['RPS25','SMR3B'],
        ['S100A9','YWHAZ'],
        ['SAMD5','SASH1'],
        ['SCNN1A','TNFRSF1A'],
        ['SELENOP','TPT1'],
        ['SENP1','TG'],
        ['SERF1A','SMN1'],
        ['SFI1','SMR3B'],
        ['SFN','TNS1'],
        ['SFTPA2','SLK'],
        ['SIDT2','TAGLN'],
        ['SLC1A4','TG'],
        ['SLC5A5','SMR3B'],
        ['SLC6A16','TEAD2'],
        ['SLC8A1','TPM1'],
        ['SLC9A7','VPS41'],
        ['SMR3B','SNTB1'],
        ['SMR3B','STAT6'],
        ['SMR3B','TBL1XR1'],
        ['SMR3B','TGOLN2'],
        ['SMR3B','TOB1'],
        ['SMR3B','TSIX'],
        ['SMR3B','USP34'],
        ['SMR3B','VIM'],
        ['SQSTM1','STEAP2'],
        ['STAG3','WWOX'],
        ['STAT3','TRA@'],
        ['STATH','WASHC4'],
        ['TBC1D29','USP6'],
        ['TBCEL','TECTA'],
        ['TDRP','TG'],
        ['TG','TMCC3'],
        ['TG','TPCN1'],
        ['TG','TSPAN14'],
        ['TG','USP39'],
        ['TG','VPS45'],
        ['TMEM45B','TRA@'],
        ['TRA@','TRIM69'],
        ['TRB@','ZFP36L1'],
        ['TUBB7P','TUBB8'],
        ['VPS53','WDR70'],
        ['ZNF440','ZNF69'],
        ['ZNF525','ZNF845'],
        ['ZNF702P','ZNF816-ZNF321P'],
        ['A4GNT','DZIP1L'],
        ['AATF','TK2'],
        ['AB015752.1','TMEM64'],
        ['ABAT','CEL'],
        ['ABCA10','GFOD1'],
        ['ABCF2','ICA1'],
        ['ABHD2','AC013565.1'],
        ['ABHD2','AC020656.1'],
        ['ABL1','EXOSC2'],
        ['ABR','AC018630.6'],
        ['ABR','CLUH'],
        ['AC002116.2','WDR62'],
        ['AC002398.2','DES'],
        ['AC002398.2','MYL2'],
        ['AC002480.4','STEAP1B'],
        ['AC003973.2','LINC01859'],
        ['AC004066.2','PPA2'],
        ['AC004080.5','HOXA5'],
        ['AC004160.1','PHF14'],
        ['AC004241.5','BGN'],
        ['AC004381.1','THUMPD1'],
        ['AC004471.2','DGCR2'],
        ['AC004477.1','COPZ2'],
        ['AC004846.2','NUMB'],
        ['AC004865.2','PSMB2'],
        ['AC004889.1','AC005304.2'],
        ['AC004889.1','ARHGEF5'],
        ['AC004895.1','USP42'],
        ['AC004908.2','LINC00115'],
        ['AC004951.2','RASA4'],
        ['AC004951.4','RASA4'],
        ['AC004967.1','OR7E38P'],
        ['AC004980.3','AC091390.2'],
        ['AC004980.3','AC211486.3'],
        ['AC004980.3','PMS2'],
        ['AC005005.4','TXNDC15'],
        ['AC005013.1','AC005162.3'],
        ['AC005035.1','IL1R2'],
        ['AC005077.2','LINC00174'],
        ['AC005091.1','HIBADH'],
        ['AC005154.1','GGCT'],
        ['AC005154.3','GGCT'],
        ['AC005258.1','ACTB'],
        ['AC005258.1','CSNK1G2'],
        ['AC005258.1','DOT1L'],
        ['AC005258.1','KLF16'],
        ['AC005258.1','PTMA'],
        ['AC005258.1','SGTA'],
        ['AC005332.7','BPTF'],
        ['AC005410.1','RPL21'],
        ['AC005479.2','NPC2'],
        ['AC005487.1','DNAJB9'],
        ['AC005540.1','RPL14'],
        ['AC005562.1','AC138207.8'],
        ['AC005586.2','LRRC61'],
        ['AC005674.2','WDR1'],
        ['AC005703.1','PMP22'],
        ['AC005740.1','NDUFS1'],
        ['AC005775.1','ODF3L2'],
        ['AC005832.1','NDUFA9'],
        ['AC005856.1','AC005884.1'],
        ['AC005993.1','RGS6'],
        ['AC006001.3','RABGEF1'],
        ['AC006064.4','HBB'],
        ['AC006064.4','NPPA'],
        ['AC006262.3','IGFL2-AS1'],
        ['AC006269.1','MMD'],
        ['AC006333.1','NDUFA5'],
        ['AC006460.2','NEMP2'],
        ['AC006946.2','CECR7'],
        ['AC007192.1','AC020656.1'],
        ['AC007207.2','AC027288.1'],
        ['AC007207.2','AL117378.1'],
        ['AC007229.1','HIKESHI'],
        ['AC007388.1','AC020656.1'],
        ['AC007570.1','KCTD10'],
        ['AC007663.1','PRODH'],
        ['AC007663.4','DGCR6'],
        ['AC007923.4','SDHA'],
        ['AC008105.3','AC138150.2'],
        ['AC008132.1','CALM1'],
        ['AC008543.1','ZNF627'],
        ['AC008555.5','LINC01801'],
        ['AC008575.1','AC136475.10'],
        ['AC008750.8','NKG7'],
        ['AC008753.1','RPLP1'],
        ['AC008760.2','SERPING1'],
        ['AC008764.1','EPS15L1'],
        ['AC008764.4','SLC35E1'],
        ['AC008770.1','AC011455.2'],
        ['AC008770.1','ZNF625-ZNF20'],
        ['AC008808.2','LINC02106'],
        ['AC008870.1','GTDC1'],
        ['AC009078.2','PRSS1'],
        ['AC009078.2','PRSS2'],
        ['AC009078.2','TRB@'],
        ['AC009107.1','CCDC113'],
        ['AC009126.1','ERAP1'],
        ['AC009135.2','ASNS'],
        ['AC009163.5','CHST5'],
        ['AC009171.2','SRL'],
        ['AC009275.1','FGD4'],
        ['AC009292.2','PIAS1'],
        ['AC009309.1','MRPL35'],
        ['AC009948.1','DFNB59'],
        ['AC010186.2','DCK'],
        ['AC010186.2','GOT2'],
        ['AC010186.4','GOT2'],
        ['AC010203.1','DEPDC4'],
        ['AC010325.1','C19ORF48'],
        ['AC010326.4','ZNF587'],
        ['AC010327.1','ACTA1'],
        ['AC010327.1','DES'],
        ['AC010327.1','NPPA-AS1'],
        ['AC010542.1','CMTM4'],
        ['AC010542.5','CMTM4'],
        ['AC010643.1','TEAD2'],
        ['AC010735.2','RHBDD1'],
        ['AC011379.2','CYSTM1'],
        ['AC011410.1','SMIM23'],
        ['AC011445.1','GMFG'],
        ['AC011446.2','PRB1'],
        ['AC011468.4','TMEM50B'],
        ['AC011487.1','ZNF765'],
        ['AC011487.3','OSTC'],
        ['AC011491.3','ACER1'],
        ['AC011523.1','KLK3'],
        ['AC011591.1','SFTPB'],
        ['AC011767.1','GOLGA6L2'],
        ['AC011825.1','RPS15A'],
        ['AC011840.1','KRT42P'],
        ['AC012170.2','USP50'],
        ['AC012435.2','UBL7-AS1'],
        ['AC012441.2','AC018638.5'],
        ['AC012442.1','CHCHD5'],
        ['AC013244.1','DIP2B'],
        ['AC013394.1','B2M'],
        ['AC013457.1','AP000688.2'],
        ['AC015468.2','LINC02153'],
        ['AC015689.2','RDX'],
        ['AC015691.1','HBE1'],
        ['AC015802.4','ST6GALNAC2'],
        ['AC015802.6','CYGB'],
        ['AC015813.1','SRSF1'],
        ['AC015849.1','TAF15'],
        ['AC015849.5','LRRC37A9P'],
        ['AC015922.4','IL6ST'],
        ['AC016044.1','ONECUT1'],
        ['AC016257.1','ALDH1L2'],
        ['AC016730.1','AC073062.1'],
        ['AC016831.1','LINC-PINT'],
        ['AC016876.2','EIF4A2'],
        ['AC016876.2','IGK@'],
        ['AC016876.2','LYZ'],
        ['AC016876.2','MPDU1'],
        ['AC016907.2','ALK'],
        ['AC017091.1','AC097501.1'],
        ['AC017104.5','LINC00471'],
        ['AC018521.5','SP2-AS1'],
        ['AC018616.1','BEST1'],
        ['AC018630.6','AC020656.1'],
        ['AC018630.6','AMY2B'],
        ['AC018630.6','CA6'],
        ['AC018630.6','CCL28'],
        ['AC018630.6','CD44'],
        ['AC018630.6','CD74'],
        ['AC018630.6','CST1'],
        ['AC018630.6','CST4'],
        ['AC018630.6','CX3CL1'],
        ['AC018630.6','DSG2'],
        ['AC018630.6','EEF1A1'],
        ['AC018630.6','EEF2'],
        ['AC018630.6','GAPDH'],
        ['AC018630.6','GNAS'],
        ['AC018630.6','GNE'],
        ['AC018630.6','H1F0'],
        ['AC018630.6','IGH@'],
        ['AC018630.6','IGK@'],
        ['AC018630.6','IL6ST'],
        ['AC018630.6','ITPR1'],
        ['AC018630.6','LPO'],
        ['AC018630.6','LYZ'],
        ['AC018630.6','MLLT6'],
        ['AC018630.6','MUC7'],
        ['AC018630.6','MYH9'],
        ['AC018630.6','ODAM'],
        ['AC018630.6','PER1'],
        ['AC018630.6','PHLDA1'],
        ['AC018630.6','PIGR'],
        ['AC018630.6','PIK3R1'],
        ['AC018630.6','PRB1'],
        ['AC018630.6','PRB2'],
        ['AC018630.6','PRB3'],
        ['AC018630.6','PRB4'],
        ['AC018630.6','PRR27'],
        ['AC018630.6','PSMF1'],
        ['AC018630.6','RBM47'],
        ['AC018630.6','RPL3'],
        ['AC018630.6','SFRP1'],
        ['AC018630.6','SIX4'],
        ['AC018630.6','SLC6A8'],
        ['AC018630.6','SMR3B'],
        ['AC018630.6','SND1'],
        ['AC018630.6','SORL1'],
        ['AC018630.6','SQSTM1'],
        ['AC018630.6','STATH'],
        ['AC018630.6','ZG16B'],
        ['AC018638.5','AL606534.4'],
        ['AC018688.1','ZNF285'],
        ['AC018695.3','COX4I1'],
        ['AC018797.2','CISD2'],
        ['AC019349.1','CRNN'],
        ['AC019349.1','GSN'],
        ['AC019349.1','KRT4'],
        ['AC019349.1','KRT6A'],
        ['AC019349.1','S100A9'],
        ['AC019349.1','SPINK5'],
        ['AC020656.1','AHNAK'],
        ['AC020656.1','AKR1B10'],
        ['AC020656.1','CEP68'],
        ['AC020656.1','CLDN18'],
        ['AC020656.1','CRISP3'],
        ['AC020656.1','CTSG'],
        ['AC020656.1','DEPDC4'],
        ['AC020656.1','EFCAB11'],
        ['AC020656.1','FAXDC2'],
        ['AC020656.1','FBXO6'],
        ['AC020656.1','HSD17B12'],
        ['AC020656.1','IGK@'],
        ['AC020656.1','IL3RA'],
        ['AC020656.1','LINC01285'],
        ['AC020656.1','LPO'],
        ['AC020656.1','MBTD1'],
        ['AC020656.1','MCAM'],
        ['AC020656.1','MPPED2'],
        ['AC020656.1','PEAK1'],
        ['AC020656.1','PGC'],
        ['AC020656.1','PKM'],
        ['AC020656.1','POMGNT1'],
        ['AC020656.1','PRKAR2A'],
        ['AC020656.1','RPL8'],
        ['AC020656.1','S100A8'],
        ['AC020656.1','SFTPB'],
        ['AC020656.1','SLC25A37'],
        ['AC020656.1','SLC2A3'],
        ['AC020656.1','SLC43A2'],
        ['AC020656.1','SLC4A1'],
        ['AC020656.1','SMARCA4'],
        ['AC020656.1','SPTBN1'],
        ['AC020656.1','TAF15'],
        ['AC020656.1','TARS2'],
        ['AC020656.1','WWOX'],
        ['AC020765.2','NUPR1'],
        ['AC020915.2','ZSCAN22'],
        ['AC020928.1','ZNF567'],
        ['AC020978.4','ESRP2'],
        ['AC021171.1','FAM133B'],
        ['AC022028.2','GRID1-AS1'],
        ['AC022145.1','STK33'],
        ['AC022217.1','FAM58A'],
        ['AC022400.3','BMS1'],
        ['AC022400.5','BMS1'],
        ['AC022733.1','AC087518.1'],
        ['AC023055.1','AC073063.1'],
        ['AC023510.1','CASC1'],
        ['AC023644.1','BEST1'],
        ['AC024084.1','AC078845.1'],
        ['AC024560.3','AC093642.1'],
        ['AC024575.1','TMEM205'],
        ['AC024587.1','USP7'],
        ['AC024592.2','LINC01356'],
        ['AC024937.3','AC026412.1'],
        ['AC024937.3','PDCD6'],
        ['AC025048.2','AC243829.2'],
        ['AC025171.2','ZNF131'],
        ['AC025262.1','TBC1D30'],
        ['AC025279.1','SNX29'],
        ['AC025566.1','LINC02032'],
        ['AC025580.3','SMG1'],
        ['AC025884.1','RERE'],
        ['AC026150.1','GOLGA8M'],
        ['AC026412.1','SDHA'],
        ['AC026471.2','AHSP'],
        ['AC027031.2','OXR1'],
        ['AC027343.1','LINC02123'],
        ['AC027559.1','AC090825.1'],
        ['AC034205.2','SLC36A2'],
        ['AC036108.3','SYNM'],
        ['AC040162.1','CELA2A'],
        ['AC040162.1','CELA2B'],
        ['AC040162.1','CTRB2'],
        ['AC040162.1','PRSS1'],
        ['AC040162.1','PRSS2'],
        ['AC040162.1','TRB@'],
        ['AC040170.1','LINC01081'],
        ['AC044810.2','OVCH2'],
        ['AC044860.1','GOLGA2P3Y'],
        ['AC048337.1','DGKD'],
        ['AC048338.1','CLIP1'],
        ['AC053503.4','KRT13'],
        ['AC053503.4','MYH6'],
        ['AC053503.4','MYH7'],
        ['AC053503.4','MYL2'],
        ['AC053503.4','NPPA'],
        ['AC053503.4','TNNI3'],
        ['AC053503.4','TPM1'],
        ['AC055713.1','WSB1'],
        ['AC058791.1','LINC00513'],
        ['AC060780.1','AC096637.2'],
        ['AC061975.7','AL035661.1'],
        ['AC063926.3','PIWIL1'],
        ['AC063944.1','LINC00882'],
        ['AC063979.2','LINC02056'],
        ['AC067930.6','ANKRD19P'],
        ['AC067930.6','HBB'],
        ['AC068152.1','ARL17B'],
        ['AC068446.1','PAK2'],
        ['AC068491.1','PAFAH1B1'],
        ['AC068580.4','ALB'],
        ['AC068580.4','IGF2'],
        ['AC068580.4','PRB1'],
        ['AC068580.4','PRB2'],
        ['AC068580.4','SMR3B'],
        ['AC068768.1','MPHOSPH9'],
        ['AC068831.3','CERK'],
        ['AC069281.1','AGFG2'],
        ['AC073264.3','AC099548.2'],
        ['AC073288.2','SKIL'],
        ['AC073578.1','LINC01257'],
        ['AC074051.2','ASNS'],
        ['AC074099.1','LINC01876'],
        ['AC078842.1','PTN'],
        ['AC079193.2','MICU3'],
        ['AC079228.1','RTN3'],
        ['AC079316.1','AL139353.3'],
        ['AC079848.1','AC108134.3'],
        ['AC083862.1','RNF14'],
        ['AC083964.2','TDRP'],
        ['AC084024.3','ASH2L'],
        ['AC084032.1','CNOT2'],
        ['AC087071.2','SSMEM1'],
        ['AC087190.4','LYZ'],
        ['AC087463.4','PWRN1'],
        ['AC087463.4','PWRN3'],
        ['AC087481.2','FAN1'],
        ['AC087481.2','HERC2'],
        ['AC087672.3','LY96'],
        ['AC087897.2','CPNE8'],
        ['AC090001.1','NTN4'],
        ['AC090114.3','TLK2'],
        ['AC090409.2','AC105094.2'],
        ['AC091167.3','GDPGP1'],
        ['AC091390.2','AC211486.3'],
        ['AC091390.3','STAG3'],
        ['AC091390.3','STAG3L5P'],
        ['AC091390.3','STAG3L5P-PVRIG2P-PILRB'],
        ['AC091390.4','AC211429.1'],
        ['AC091390.4','AC211486.3'],
        ['AC091390.4','PMS2'],
        ['AC091849.2','SDHA'],
        ['AC092079.1','DDX18'],
        ['AC092162.2','SPOPL'],
        ['AC092284.1','CRLF3'],
        ['AC092376.1','MAF'],
        ['AC092506.1','THAP3'],
        ['AC092718.8','CMC2'],
        ['AC092718.8','NDUFS1'],
        ['AC092746.1','CLEC4E'],
        ['AC092747.2','MED21'],
        ['AC092813.2','ST6GALNAC3'],
        ['AC092821.1','AC141557.1'],
        ['AC092821.1','TPT1-AS1'],
        ['AC092821.2','RANBP6'],
        ['AC092902.2','FAM85B'],
        ['AC092920.1','LINC01208'],
        ['AC093157.2','SLC30A7'],
        ['AC093162.2','RETSAT'],
        ['AC093392.1','SEPT7'],
        ['AC093484.4','ANKRD1'],
        ['AC093484.4','HBA2'],
        ['AC093484.4','HBB'],
        ['AC093484.4','UBA52'],
        ['AC093525.2','DDX5'],
        ['AC093627.3','HABP4'],
        ['AC093720.1','UGT2A3'],
        ['AC093752.1','SEPT7'],
        ['AC093821.1','LINC01091'],
        ['AC093895.1','SPARCL1'],
        ['AC095038.1','ZNF727'],
        ['AC095350.2','RIMBP2'],
        ['AC096664.2','RPL27'],
        ['AC097372.3','LINC01095'],
        ['AC097374.1','AL513478.2'],
        ['AC097374.1','CR392039.5'],
        ['AC098484.2','PPIH'],
        ['AC098798.1','PANCR'],
        ['AC098848.1','CCDC68'],
        ['AC098850.4','AC107983.2'],
        ['AC098850.4','USP32'],
        ['AC098850.5','NOS2'],
        ['AC098850.5','USP6'],
        ['AC099329.1','CCDC13'],
        ['AC099508.2','ZFP1'],
        ['AC099548.2','TCAF1'],
        ['AC099548.2','TCAF2'],
        ['AC099794.1','ATG4C'],
        ['AC100821.2','TCEA1'],
        ['AC100861.1','CHMP7'],
        ['AC103702.2','HOXB9'],
        ['AC103957.1','PRAG1'],
        ['AC104109.3','LINC00174'],
        ['AC104581.1','DNAH14'],
        ['AC104758.1','COMMD4'],
        ['AC105046.1','EGR3'],
        ['AC105052.3','RASA4CP'],
        ['AC105411.1','LINC01227'],
        ['AC106795.3','AC139493.2'],
        ['AC106865.1','TLR2'],
        ['AC106886.5','MYH11'],
        ['AC107983.2','USP32'],
        ['AC107983.2','USP6'],
        ['AC108010.1','TLK2'],
        ['AC108097.1','LINC01227'],
        ['AC108488.1','ADI1'],
        ['AC108688.1','DNAJB6'],
        ['AC108734.3','RNF13'],
        ['AC108751.4','TM4SF1'],
        ['AC109466.1','CCDC7'],
        ['AC110079.2','NPR1'],
        ['AC111149.1','RDH10-AS1'],
        ['AC112719.2','ART3'],
        ['AC113189.4','TMEM102'],
        ['AC113189.6','C17ORF74'],
        ['AC113189.6','TMEM102'],
        ['AC114284.1','PRR16'],
        ['AC114501.2','INTS4'],
        ['AC114781.2','COQ2'],
        ['AC116050.1','IGK@'],
        ['AC116366.1','IRF1'],
        ['AC116563.1','AC131956.2'],
        ['AC116651.1','FBXL5'],
        ['AC117386.2','CHURC1'],
        ['AC117386.2','TGOLN2'],
        ['AC118758.3','LINC00174'],
        ['AC119396.1','ARHGEF18'],
        ['AC119403.1','ZNF77'],
        ['AC120057.3','NPPA-AS1'],
        ['AC123767.1','ADAM5'],
        ['AC123912.3','ZNF100'],
        ['AC125257.1','KLHL11'],
        ['AC125494.1','RPL14'],
        ['AC125634.1','AL353626.3'],
        ['AC126120.1','PPIA'],
        ['AC126335.1','IGK@'],
        ['AC126544.1','HARS'],
        ['AC127502.2','LYZ'],
        ['AC127526.4','LINC00937'],
        ['AC131160.1','AL136116.3'],
        ['AC132008.2','SDHA'],
        ['AC133065.6','CIITA'],
        ['AC133919.3','AL606534.5'],
        ['AC133961.1','SEL1L3'],
        ['AC134772.1','SHISA5'],
        ['AC134878.2','CR381653.1'],
        ['AC134878.2','TEKT4'],
        ['AC135983.1','CHRFAM7A'],
        ['AC136475.10','SRP19'],
        ['AC136628.2','C5ORF60'],
        ['AC136698.1','AC243562.3'],
        ['AC136698.1','LINC00933'],
        ['AC136944.2','IGK@'],
        ['AC136944.3','CHEK2'],
        ['AC137056.1','AC141273.1'],
        ['AC138035.1','AL627309.5'],
        ['AC138207.8','NF1'],
        ['AC138356.3','ERLIN2'],
        ['AC138393.1','SEPT7'],
        ['AC138409.2','AP000347.1'],
        ['AC138409.2','C1QTNF3'],
        ['AC138409.2','C1QTNF3-AMACR'],
        ['AC138409.2','WDR70'],
        ['AC138649.1','PDCD6IP'],
        ['AC138811.2','SMG1'],
        ['AC138915.4','CISD2'],
        ['AC138932.1','NPIPA8'],
        ['AC139256.1','SNX29'],
        ['AC139491.5','FAM153B'],
        ['AC140481.3','FAM95B1'],
        ['AC141586.1','KCTD5'],
        ['AC142381.1','IGH@'],
        ['AC145098.1','DOK3'],
        ['AC211486.3','CCDC146'],
        ['AC211486.3','POM121C'],
        ['AC211486.3','PRKRIP1'],
        ['AC233266.2','TMEM128'],
        ['AC233702.10','FLJ36000'],
        ['AC233702.7','FLJ36000'],
        ['AC239860.1','NBPF13P'],
        ['AC241584.1','SSB'],
        ['AC242376.2','HERC2'],
        ['AC243562.1','GOLGA2P3Y'],
        ['AC243562.3','DNM1'],
        ['AC243829.4','CCL4'],
        ['AC244205.1','IGKV4-1'],
        ['AC244226.1','IGH@'],
        ['AC244517.10','PCDHB9'],
        ['AC244517.2','MALL'],
        ['AC245100.4','LINC00869'],
        ['AC245128.3','AL357055.3'],
        ['AC245128.3','ORC4'],
        ['AC245369.1','IGH@'],
        ['AC245369.3','IGH@'],
        ['AC245369.3','IGHA2'],
        ['AC245369.3','IGHG1'],
        ['AC245369.7','IGH@'],
        ['AC245884.11','LILRA6'],
        ['AC245884.11','LILRB3'],
        ['AC245884.4','LYZ'],
        ['AC246787.2','FAM30A'],
        ['AC247036.4','IGH@'],
        ['AC247036.6','IGH@'],
        ['AC253536.1','GSTTP1'],
        ['ACADSB','MCFD2'],
        ['ACAT1','CUL5'],
        ['ACBD3','ZMYM4'],
        ['ACCS','EXT2'],
        ['ACE2','PIR'],
        ['ACER3','POP1'],
        ['ACSL1','CENPU'],
        ['ACSS1','APMAP'],
        ['ACTB','CCNG2'],
        ['ACTB','CLPS'],
        ['ACTB','CNN2'],
        ['ACTB','CXCR4'],
        ['ACTB','FAM46A'],
        ['ACTB','HMGB2'],
        ['ACTB','HNRNPK'],
        ['ACTB','NUCB1'],
        ['ACTB','PRH1'],
        ['ACTB','YWHAZ'],
        ['ACTG1','TINAGL1'],
        ['ACTN4','EIF3K'],
        ['ACTR10','TG'],
        ['ACVR1B','GRASP'],
        ['ACVR2B','EXOG'],
        ['ADAM23','FAM237A'],
        ['ADAMTS6','CENPK'],
        ['ADAMTS7','AL049757.1'],
        ['ADAP1','MAFK'],
        ['ADAP1','SUN1'],
        ['ADGRF5','AL512625.3'],
        ['ADGRG1','CLPS'],
        ['ADGRG7','TFG'],
        ['ADK','HIBCH'],
        ['ADM','RIN3'],
        ['ADRM1','SS18L1'],
        ['AF001548.2','KRT13'],
        ['AF064860.2','IGSF5'],
        ['AF165147.1','FXR1'],
        ['AFAP1','ANXA7'],
        ['AFDN-AS1','LINC01558'],
        ['AFF1','PTPN13'],
        ['AFP','ALB'],
        ['AFTPH','LINC01805'],
        ['AGAP3','CUX1'],
        ['AGAP4','LINC00843'],
        ['AGAP5','BMS1P4'],
        ['AGGF1','ZBED3-AS1'],
        ['AGMAT','AL121992.1'],
        ['AIM2','CADM3-AS1'],
        ['AKAP13','CD2AP'],
        ['AKAP8L','SMR3B'],
        ['AKNA','AL138895.1'],
        ['AKT2','C19ORF47'],
        ['AL008628.1','FAM120B'],
        ['AL021707.2','FAM227A'],
        ['AL021920.2','EIF1AX'],
        ['AL022322.2','POC1B-GALNT4'],
        ['AL022324.2','PHF10'],
        ['AL024509.3','CMAHP'],
        ['AL031282.2','MALL'],
        ['AL031282.2','SLC35E2B'],
        ['AL031289.1','RIMS3'],
        ['AL031587.6','AL627309.5'],
        ['AL031587.6','LYZ'],
        ['AL031587.6','MICALL1'],
        ['AL034369.1','PNLIPRP2'],
        ['AL035420.1','EPB41L1'],
        ['AL035425.3','IRS4'],
        ['AL035587.1','GNMT'],
        ['AL035701.1','ENPP5'],
        ['AL049795.1','TMEM234'],
        ['AL049830.3','STRN3'],
        ['AL049836.2','LINC02314'],
        ['AL049869.3','ZBTB25'],
        ['AL050303.3','LINC01087'],
        ['AL050303.3','LINC01297'],
        ['AL050320.1','ISM1-AS1'],
        ['AL079341.1','TFG'],
        ['AL080317.1','MFSD4B'],
        ['AL096870.2','AL136295.7'],
        ['AL109804.1','MAVS'],
        ['AL109809.1','AL117335.1'],
        ['AL109809.1','AL121760.1'],
        ['AL109811.3','TARDBP'],
        ['AL109954.2','CSTL1'],
        ['AL117350.1','RAB4A'],
        ['AL121578.3','SYTL5'],
        ['AL121656.1','LTBP1'],
        ['AL121790.1','TTC6'],
        ['AL121987.2','DCAF8'],
        ['AL132639.4','TRAPPC6B'],
        ['AL132656.2','NUTM2A-AS1'],
        ['AL132780.3','PRMT5'],
        ['AL133338.2','MAPK1IP1L'],
        ['AL133353.1','SLC25A28'],
        ['AL133371.1','AL355922.5'],
        ['AL133410.3','TMEM8B'],
        ['AL133481.2','AL359195.2'],
        ['AL136133.1','RPL22'],
        ['AL136309.4','C6ORF201'],
        ['AL136520.1','SRSF10'],
        ['AL136982.1','BMS1'],
        ['AL136982.6','BMS1'],
        ['AL137003.1','ATXN1'],
        ['AL137145.2','PFKFB3'],
        ['AL137786.1','PAPOLA'],
        ['AL137796.1','C1ORF220'],
        ['AL137856.1','AP4B1-AS1'],
        ['AL138963.3','APOH'],
        ['AL138963.3','HP'],
        ['AL138963.3','PNLIP'],
        ['AL138963.3','PRB1'],
        ['AL138963.3','PRB3'],
        ['AL138963.3','PRSS1'],
        ['AL138963.3','SMR3B'],
        ['AL138963.3','TRB@'],
        ['AL139022.1','FNTB'],
        ['AL139142.1','BX248409.2'],
        ['AL139220.2','KLF17'],
        ['AL139300.2','APOPT1'],
        ['AL139353.3','CHST11'],
        ['AL139383.1','LINC02344'],
        ['AL139383.1','STARD13'],
        ['AL139384.2','TUBGCP3'],
        ['AL139412.1','C1ORF61'],
        ['AL139819.1','PKD2L1'],
        ['AL157714.2','RXRG'],
        ['AL157834.1','CYP2C8'],
        ['AL158066.1','MRPS31P4'],
        ['AL158154.2','UPP1'],
        ['AL158212.1','VTI1A'],
        ['AL160291.1','ARMC4'],
        ['AL160408.2','HBB'],
        ['AL162258.2','DES'],
        ['AL162414.1','CLPB'],
        ['AL353148.1','LYZ'],
        ['AL353608.3','PIP5K1B'],
        ['AL353743.1','KIF27'],
        ['AL354710.2','GAPVD1'],
        ['AL354726.1','HEMGN'],
        ['AL354751.1','PTPRVP'],
        ['AL354760.1','ST7L'],
        ['AL354892.2','TCTE3'],
        ['AL354956.1','TCAM1P'],
        ['AL355376.1','LINC01517'],
        ['AL356235.1','ZC3H12B'],
        ['AL356489.2','TRB@'],
        ['AL356585.4','ANKRD36B'],
        ['AL356867.1','SPRR3'],
        ['AL358790.1','JPX'],
        ['AL359075.1','SEC16B'],
        ['AL359715.2','BCKDHB'],
        ['AL360181.3','SCART1'],
        ['AL360270.1','CD53'],
        ['AL360295.1','C1ORF168'],
        ['AL365203.3','ITGB1'],
        ['AL365277.1','PPIE'],
        ['AL390036.1','VAV3-AS1'],
        ['AL390195.1','OVGP1'],
        ['AL390726.6','ANKRD18B'],
        ['AL390726.6','CYP4F26P'],
        ['AL390728.4','RPGRIP1L'],
        ['AL390860.1','TRIM58'],
        ['AL390961.2','FAM238B'],
        ['AL390961.2','FAM238C'],
        ['AL392046.1','CUL2'],
        ['AL392172.1','DISP1'],
        ['AL445238.1','OR7E156P'],
        ['AL445472.1','KLF8'],
        ['AL445623.2','ELAVL2'],
        ['AL450124.1','FAM182B'],
        ['AL450322.2','MYH11'],
        ['AL451085.2','CKS1B'],
        ['AL512310.12','NF1'],
        ['AL512625.3','GLIDR'],
        ['AL512662.2','BMS1'],
        ['AL513314.2','AL627309.5'],
        ['AL513314.2','LINC00174'],
        ['AL513478.2','PARG'],
        ['AL583722.1','INF2'],
        ['AL589935.1','LINC00472'],
        ['AL590705.5','CCDC180'],
        ['AL590787.1','MTMR6'],
        ['AL591242.1','PLA2G7'],
        ['AL591684.2','BMS1'],
        ['AL591806.3','CCND3'],
        ['AL606491.1','LDLRAP1'],
        ['AL645929.1','AL671277.1'],
        ['AL645937.4','AL672167.2'],
        ['AL662899.1','LY6G5C'],
        ['AL669831.4','LINC01881'],
        ['AL671277.1','AL671883.3'],
        ['AL671710.1','ZBED4'],
        ['AL807752.7','COL1A1'],
        ['AL845552.2','WDR25'],
        ['AL928768.1','IGHA1'],
        ['ALB','ORM2'],
        ['ALB','PRDX1'],
        ['ALDH2','ANKHD1'],
        ['ALDOA','PPP4C'],
        ['ALG14','CNN3'],
        ['ALG1L2','ALG1L6P'],
        ['ALKBH5','LLGL1'],
        ['ALMS1','SMOC2'],
        ['ALOX12-AS1','MIR497HG'],
        ['AMD1','EEF1A1'],
        ['AMMECR1L','NUTM2B-AS1'],
        ['AMY2A','CPA2'],
        ['AMY2B','PRSS3'],
        ['ANAPC1','RMND5A'],
        ['ANK2','CAMK2D'],
        ['ANKAR','ASNSD1'],
        ['ANKAR','ZDHHC7'],
        ['ANKEF1','TACC2'],
        ['ANKRD18CP','BX664727.3'],
        ['ANKRD20A11P','AP005212.5'],
        ['ANKRD20A17P','CR381670.1'],
        ['ANO10','NEK7'],
        ['ANP32B','IGH@'],
        ['ANP32B','TMOD1'],
        ['ANP32B','TNNT2'],
        ['ANXA1','MAGT1'],
        ['ANXA1','TAGLN'],
        ['ANXA1','ZNF207'],
        ['ANXA2','CELA2A'],
        ['ANXA2','CRYBG1'],
        ['ANXA2','RNF111'],
        ['ANXA5','CLEC4E'],
        ['AP000317.1','MRPS6'],
        ['AP000317.1','SLC5A3'],
        ['AP000344.1','LINC01659'],
        ['AP000346.1','AP000347.1'],
        ['AP000346.1','AP000347.2'],
        ['AP000346.1','IGLL3P'],
        ['AP000346.2','AP000347.1'],
        ['AP000346.2','AP000347.2'],
        ['AP000356.2','POM121L7P'],
        ['AP000356.2','USP18'],
        ['AP000356.3','BCR'],
        ['AP000473.1','MIR99AHG'],
        ['AP000781.2','IGF2'],
        ['AP000781.2','TG'],
        ['AP000880.1','NCAM1-AS1'],
        ['AP000924.1','FDX1'],
        ['AP000936.1','PCSK7'],
        ['AP000943.3','MRE11'],
        ['AP000943.4','FUT4'],
        ['AP001005.4','TUBB8'],
        ['AP001107.1','EMP2'],
        ['AP001350.2','ZFP91-CNTF'],
        ['AP001931.1','TMX2'],
        ['AP001979.2','SPATA19'],
        ['AP002008.1','BTG4'],
        ['AP002748.1','CTSF'],
        ['AP002884.4','IL18'],
        ['AP003066.1','AP003781.1'],
        ['AP003086.1','GAB2'],
        ['AP003354.2','ATP6V1C1'],
        ['AP003400.1','NOX4'],
        ['AP005329.1','MYOM1'],
        ['AP005435.2','EMB'],
        ['AP005671.1','EPB41L3'],
        ['AP3S1','COMMD10'],
        ['AP3S2','YPEL5'],
        ['APBB1IP','FAM238A'],
        ['APBB2','NR3C2'],
        ['APBB3','SRA1'],
        ['APOL3','MIR100HG'],
        ['APP','HTN3'],
        ['ARHGAP11B','OTUD7A'],
        ['ARHGAP18','DLGAP5'],
        ['ARHGAP18','TMEM244'],
        ['ARHGAP26','NR3C1'],
        ['ARHGAP27P1-BPTFP1-KPNA2P3','KPNA2'],
        ['ARHGEF17','P2RY6'],
        ['ARHGEF5','OR2A1-AS1'],
        ['ARHGEF9','LINC01278'],
        ['ARID4B','B2M'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARL6IP1','RPS15A'],
        ['ARL6IP6','TRNT1'],
        ['ARMC6','SLC25A42'],
        ['ARPC4','OGG1'],
        ['ARRDC3','BCLAF1'],
        ['ART4','SMCO3'],
        ['AS3MT','CCND3'],
        ['ASAP3','TCEA3'],
        ['ASL','CRCP'],
        ['ASXL1','NOL4L'],
        ['ATAD1','PTEN'],
        ['ATG16L1','INPP5D'],
        ['ATP10D','USP32'],
        ['ATP13A3','UBR3'],
        ['ATP1A1-AS1','LINC01762'],
        ['ATP5C1','ITIH2'],
        ['ATP5H','TG'],
        ['ATP5I','MFSD7'],
        ['ATP6V0B','DPH2'],
        ['ATP6V0C','EEF1A1'],
        ['ATP7B','LINC00282'],
        ['ATP8A2','NUP58'],
        ['AZGP1','GJC3'],
        ['AZIN1','KLF10'],
        ['B2M','FBN1'],
        ['B2M','FUS'],
        ['B2M','G3BP2'],
        ['B2M','HSPA9'],
        ['B2M','IGH@'],
        ['B2M','MED13L'],
        ['B2M','PFKFB3'],
        ['B2M','RUNX1'],
        ['B2M','SF1'],
        ['B2M','SUB1'],
        ['B2M','UBC'],
        ['B2M','UBE3A'],
        ['B3GNT5','KLHL24'],
        ['B3GNT5','TIPARP'],
        ['B3GNTL1','METRNL'],
        ['BABAM2','RBKS'],
        ['BANK1','IGH@'],
        ['BASP1','DSE'],
        ['BCAP29','DUS4L'],
        ['BCAS3','TG'],
        ['BCKDHA','TMEM91'],
        ['BCL2L11','CXCR4'],
        ['BCL6','NAMPT'],
        ['BCL7A','RBM38'],
        ['BEST1','HBB'],
        ['BEST1','MYH6'],
        ['BGLAP','PMF1'],
        ['BICD2','IPPK'],
        ['BLOC1S6','SQOR'],
        ['BMI1','DDX5'],
        ['BMPR1B','PDLIM5'],
        ['BPI','CFL1'],
        ['BPIFC','RTCB'],
        ['BPTF','LRRC37A2'],
        ['BRD1','C22ORF34'],
        ['BRD2','SRSF5'],
        ['BRD3','TG'],
        ['BRD4','TG'],
        ['BRK1','VHL'],
        ['BSDC1','FAM229A'],
        ['BSG','ELANE'],
        ['BSG','OAZ1'],
        ['BSG','R3HDM4'],
        ['BSG','SBNO2'],
        ['BSG','TCF3'],
        ['BSG','UBB'],
        ['BX072579.2','LINC00452'],
        ['BX276092.10','BX276092.7'],
        ['BX284632.1','IGK@'],
        ['BX547991.1','LINC00864'],
        ['C10ORF90','LINC00601'],
        ['C11ORF57','DLAT'],
        ['C11ORF70','YAP1'],
        ['C12ORF73','HSP90B1'],
        ['C14ORF166','GNG2'],
        ['C15ORF48','SPATA5L1'],
        ['C15ORF57','CBX3'],
        ['C16ORF87','COL14A1'],
        ['C18ORF8','CAPZA2'],
        ['C19MC','RPLP1'],
        ['C19ORF33','SPINT2'],
        ['C1ORF146','KIAA1107'],
        ['C1ORF194','UQCR10'],
        ['C1ORF43','MB'],
        ['C1ORF61','IFNAR2'],
        ['C20ORF202','PSMF1'],
        ['C2CD3','UCP3'],
        ['C2ORF48','RRM2'],
        ['C2ORF72','SPATA3'],
        ['C2ORF81','HMGA1'],
        ['C2ORF81','RTKN'],
        ['C3ORF67','FAM3D'],
        ['C4ORF51','CALB1'],
        ['C6ORF106','SPDEF'],
        ['C9ORF24','KIF24'],
        ['CA6','JCHAIN'],
        ['CA6','PRB3'],
        ['CA6','PRB4'],
        ['CABIN1','CBFA2T2'],
        ['CABP7','NF2'],
        ['CACNA1C','LINC02371'],
        ['CACNG1','CACNG4'],
        ['CACUL1','SND1'],
        ['CALCRL','NBEAL1'],
        ['CALM2','PIGF'],
        ['CALM2','STATH'],
        ['CALM3','MYL4'],
        ['CALU','FAM71F1'],
        ['CASTOR2','FURIN'],
        ['CATSPER2','PPIP5K1'],
        ['CAVIN1','STAT3'],
        ['CBWD2','DOCK8'],
        ['CCDC117','Z93930.2'],
        ['CCDC175','L3HYPDH'],
        ['CCDC28A','ECT2L'],
        ['CCDC82','JRKL-AS1'],
        ['CCM2L','HCK'],
        ['CCNG2','OAZ1'],
        ['CCNG2','VAV1'],
        ['CD302','TRIP12'],
        ['CD63','TMBIM6'],
        ['CD81','TSSC4'],
        ['CD83','JARID2'],
        ['CD8B','CD8B2'],
        ['CDC34','GZMM'],
        ['CDC34','PTMA'],
        ['CDC42','SLC25A37'],
        ['CDC42','ZEB2'],
        ['CDC42EP2','POLA2'],
        ['CDC42EP4','CPSF4L'],
        ['CDHR4','IP6K1'],
        ['CDK10','SPATA33'],
        ['CDK11B','SLC35E2'],
        ['CDK19','REV3L'],
        ['CDK2','RAB5B'],
        ['CDK5RAP3','PNPO'],
        ['CDKN1A','RAB44'],
        ['CDR1','LINC00632'],
        ['CDRT15L2','TBC1D26'],
        ['CEL','CELP'],
        ['CELA2B','PDIA2'],
        ['CELSR3','NCKIPSD'],
        ['CEP152','FBN1'],
        ['CEP192','SEH1L'],
        ['CFAP157','STXBP1'],
        ['CFL1','MKNK2'],
        ['CFL1','SF1'],
        ['CHST11','IGH@'],
        ['CHSY3','ROCK2'],
        ['CITED1','HDAC8'],
        ['CLCF1','POLD4'],
        ['CLEC2D','USP24'],
        ['CLIC2','F8'],
        ['CLK1','CXCR4'],
        ['CLPS','CTRB2'],
        ['CLTC','VMP1'],
        ['CMTM7','CMTM8'],
        ['CNBD2','PHF20'],
        ['CNN2','FURIN'],
        ['CNN2','RNF126'],
        ['CNN2','SBNO2'],
        ['CNTROB','PET100'],
        ['COL14A1','DEPTOR'],
        ['COL16A1','DES'],
        ['COL18A1','COL3A1'],
        ['COL4A1','SLC29A1'],
        ['COLQ','HACL1'],
        ['COMMD10','FBXL17'],
        ['COMMD2','WWTR1'],
        ['COPA','HTN3'],
        ['COQ4','SLC27A4'],
        ['COQ8B','NUMBL'],
        ['COX19','LINC01002'],
        ['COX6B1','ETV2'],
        ['COX6C','HMGB1'],
        ['COX7A2','MINDY3'],
        ['CPM','LYZ'],
        ['CPSF6','LYZ'],
        ['CRCP','TPST1'],
        ['CREB3','GRB2'],
        ['CREBRF','NSD1'],
        ['CRELD1','PRRT3-AS1'],
        ['CRLF2','LYZ'],
        ['CRYBB2','Z99916.1'],
        ['CRYBG2','ZNF683'],
        ['CST1','STATH'],
        ['CTBP1','MAEA'],
        ['CTBP1','UVSSA'],
        ['CTBP1-AS2','MAEA'],
        ['CTCF','FAT1'],
        ['CTDP1','LINC00969'],
        ['CTDP1','NFATC1'],
        ['CTDP1','PQLC1'],
        ['CTDSP1','DNM3OS'],
        ['CTH','HHLA3'],
        ['CTNNA1','MATR3'],
        ['CTNND1','MAGED1'],
        ['CTNND2','MEX3C'],
        ['CTRB2','P4HB'],
        ['CTSB','NUCB2'],
        ['CTSC','RAB38'],
        ['CTSD','LSP1'],
        ['CU633967.1','CU638689.4'],
        ['CU634019.6','CU638689.4'],
        ['CUEDC1','MALAT1'],
        ['CUX1','KLF10'],
        ['CWF19L2','TMEM41A'],
        ['CXCR4','METRNL'],
        ['CXCR4','MYO15B'],
        ['CXCR4','RARA'],
        ['CXCR4','RBM38'],
        ['CXCR4','SF1'],
        ['CXCR4','THAP4'],
        ['CXCR4','UBC'],
        ['CXCR4','UBXN4'],
        ['CXCR4','ZBTB18'],
        ['CXCR4','ZEB2'],
        ['CYBA','WWOX'],
        ['CYBB','XK'],
        ['CYTIP','ERMN'],
        ['DAND5','NFIX'],
        ['DAZAP1','GLUL'],
        ['DCN','UBE2N'],
        ['DCTN5','UBFD1'],
        ['DCUN1D1','MCCC1'],
        ['DDX27','PARP15'],
        ['DDX39A','JMJD1C'],
        ['DDX3X','EEF1A1'],
        ['DDX3X','UBC'],
        ['DDX42','MAP3K3'],
        ['DDX5','EIF4A1'],
        ['DDX5','FUS'],
        ['DDX5','GRK2'],
        ['DDX5','HMGN2'],
        ['DDX5','HNRNPU'],
        ['DDX5','LIMS2'],
        ['DDX5','METRNL'],
        ['DDX5','NFE2L1'],
        ['DDX5','OAZ1'],
        ['DDX5','TEX2'],
        ['DDX5','TPD52L2'],
        ['DDX5','UGCG'],
        ['DEPDC1B','PDE4D'],
        ['DES','KRT7'],
        ['DGCR8','TANGO2'],
        ['DHCR24','KRT4'],
        ['DHRS4-AS1','NRL'],
        ['DHRS7C','GSG1L2'],
        ['DHX35','FAM83D'],
        ['DHX37','UBC'],
        ['DHX9','NPL'],
        ['DIP2A','LYZ'],
        ['DIP2B','TMBIM6'],
        ['DLD','HNRNPK'],
        ['DLEU2','HMGB1'],
        ['DLEU2','SPRYD7'],
        ['DLEU7-AS1','RNASEH2B'],
        ['DNAAF3','SYT5'],
        ['DNAJA1','POLR2A'],
        ['DNAJB6','LYZ'],
        ['DNAJC7','LYZ'],
        ['DOC2B','LINC02091'],
        ['DOCK1','FAM24B'],
        ['DOCK9','STK24'],
        ['DOT1L','PTBP1'],
        ['DOT1L','ZBTB7A'],
        ['DPH7','PNPLA7'],
        ['DPP9','TG'],
        ['DSC3','TMEM269'],
        ['DTD1','MKL2'],
        ['E2F1','PXMP4'],
        ['EAF2','SLC15A2'],
        ['EEF1A1','HBA1'],
        ['EEF1A1','MUC7'],
        ['EEF1A1','PTMA'],
        ['EEF1A1','TG'],
        ['EEF1A1','TPM3'],
        ['EEF1A2','MYH7'],
        ['EEF1AKMT3','TSFM'],
        ['EEF2','IGK@'],
        ['EFCAB2','KIF26B'],
        ['EFCAB7','PGM1'],
        ['EGF','SEC24B'],
        ['EGFR','H19'],
        ['EIF1AX','MAP7D2'],
        ['EIF1AX','RPS6KA3'],
        ['EIF2AK1','S100A9'],
        ['EIF2S1','RBM25'],
        ['EIF3F','PNLIP'],
        ['EIF4E3','FOXP1'],
        ['EIF4E3','PROK2'],
        ['EIF4G2','PTMA'],
        ['ELANE','HBD'],
        ['ELANE','LINC01578'],
        ['ELANE','SF1'],
        ['ELK4','SLC45A3'],
        ['EMC2','NUDCD1'],
        ['EMG1','MIR200CHG'],
        ['EMP2','MIR100HG'],
        ['ENO2','MYBPC3'],
        ['ENTPD5','FAM161B'],
        ['EPB41L3','ZBTB14'],
        ['EPB42','LYZ'],
        ['EPN2','UBB'],
        ['ERBB2','GRB7'],
        ['ERBB3','LYZ'],
        ['ERP44','UBQLN1'],
        ['ERVK-28','ZNF585A'],
        ['F5','SELP'],
        ['FADS1','TMEM258'],
        ['FAM110A','JAZF1'],
        ['FAM129C','PGLS'],
        ['FAM133B','FAM133CP'],
        ['FAM135A','MAP3K4'],
        ['FAM135A','SDHAF4'],
        ['FAM13A-AS1','HERC3'],
        ['FAM149B1','PTGR1'],
        ['FAM157B','FBXO25'],
        ['FAM160B1','TRUB1'],
        ['FAM174A','ST8SIA4'],
        ['FAM53A','XRRA1'],
        ['FAM53B','MTHFS'],
        ['FAM53B','TSPAN3'],
        ['FAM66A','FAM66C'],
        ['FAM85B','LINC02018'],
        ['FAM86B1','FAM86B3P'],
        ['FBXL5','HMGB1'],
        ['FBXO16','HBB'],
        ['FBXO25','TDRP'],
        ['FCAR','LENG8'],
        ['FES','MAN2A2'],
        ['FGD5','TPRXL'],
        ['FGFR4','ZNF346'],
        ['FITM2','OSER1'],
        ['FKBP5','PNLIP'],
        ['FLJ22447','PRKCH'],
        ['FLNB','PRB3'],
        ['FNIP1','PDE4B'],
        ['FOLR3','LRTOMT'],
        ['FOXK2','SERPINE2'],
        ['FOXO1','NUP98'],
        ['FOXP1','RYBP'],
        ['FP700111.2','IGK@'],
        ['FRG1','FRG1HP'],
        ['FRMD6-AS2','LINC02310'],
        ['FUS','KCTD5'],
        ['G2E3','SCFD1'],
        ['G3BP2','PTP4A1'],
        ['GABPB1','PTP4A1'],
        ['GALK2','SUPT16H'],
        ['GAS7','RCVRN'],
        ['GATC','KDM2A'],
        ['GBP3','KYAT3'],
        ['GCA','SAMSN1'],
        ['GCA','TG'],
        ['GCC1','TG'],
        ['GGT1','IGSF3'],
        ['GLE1','ODF2'],
        ['GLI3','GRB10'],
        ['GLTP','U52111.1'],
        ['GLUL','KCNAB2'],
        ['GNAS','SENP3'],
        ['GNE','SMR3B'],
        ['GNG7','TPM1'],
        ['GOLGA2','SMR3B'],
        ['GOLT1A','KISS1'],
        ['GPR156','LRRC58'],
        ['GPRC5A','OGFRL1'],
        ['GPT','PPP1R16A'],
        ['GPX3','KIAA0319L'],
        ['GRAMD1A','TG'],
        ['GRSF1','RSRP1'],
        ['GS1-124K5.4','TPST1'],
        ['GSN','MMP25'],
        ['GSN','S100A9'],
        ['GUCA1C','MORC1'],
        ['GULP1','SCN9A'],
        ['GYPB','SAV1'],
        ['H1F0','PRB2'],
        ['H3F3B','ZG16B'],
        ['H6PD','SPSB1'],
        ['HACD1','PLS3'],
        ['HAPLN3','MFGE8'],
        ['HAS3','UTP4'],
        ['HBA1','RPS27'],
        ['HBA2','TUBB4B'],
        ['HBB','MOB3A'],
        ['HBB','NLRC5'],
        ['HBB','SEPT2'],
        ['HBB','VTI1A'],
        ['HBD','LTF'],
        ['HBP1','IMMP2L'],
        ['HDLBP','KAT6A'],
        ['HIF1A','PTP4A1'],
        ['HIF1A','RNPC3'],
        ['HIF1A','STAG2'],
        ['HIPK1','OLFML3'],
        ['HKR1','TG'],
        ['HMBS','SBNO2'],
        ['HMGB1','HNRNPH1'],
        ['HMGB1','MXD4'],
        ['HMGB1','SRSF5'],
        ['HMGB2','IVNS1ABP'],
        ['HMGN2','NECAB3'],
        ['HNRNPA2B1','UBB'],
        ['HNRNPC','PTP4A1'],
        ['HNRNPD','RCC1'],
        ['HNRNPDL','TUBA1B'],
        ['HNRNPH1','ITPRIP'],
        ['HNRNPH1','TG'],
        ['HNRNPK','NFIL3'],
        ['HNRNPK','UBQLN1'],
        ['HNRNPUL1','MAEA'],
        ['HOOK3','PPM1A'],
        ['HORMAD1','SLC22A20'],
        ['HSD17B14','PLEKHA4'],
        ['HSD17B4','TNFAIP8'],
        ['HSP90AB1','SLC29A1'],
        ['HSPA9','UBE2D2'],
        ['HSPB11','YIPF1'],
        ['HSPE1','MOB4'],
        ['HTN1','MYH11'],
        ['HTN1','PRR27'],
        ['HTN1','SH3BGRL2'],
        ['HTN3','MAPK1'],
        ['HTN3','PRR27'],
        ['HTN3','PTK7'],
        ['IFT122','RYK'],
        ['IGH@','LAPTM5'],
        ['IGH@','MIR100HG'],
        ['IGH@','MUC1'],
        ['IGH@','NFKB2'],
        ['IGH@','PCSK7'],
        ['IGH@','PPP6R2'],
        ['IGH@','TRA@'],
        ['IGH@','ZFP36L1'],
        ['IGLC1','SMR3B'],
        ['IGSF10','MED12L'],
        ['IL9R','RPL23AP82'],
        ['IMP3','SNUPN'],
        ['INPP4B','PLCZ1'],
        ['INPP5A','PWWP2B'],
        ['IQCG','POLG2'],
        ['IQGAP1','ZNF774'],
        ['ITGA7','PPP1R1A'],
        ['ITGAV','PXK'],
        ['ITGB2','POFUT2'],
        ['ITGB6','PPP4R1'],
        ['ITM2B','RB1'],
        ['ITPKC','SNRPA'],
        ['IVD','KNSTRN'],
        ['JARID2','LINC01578'],
        ['JARID2','MGAT4A'],
        ['JAZF1','ZNF777'],
        ['JKAMP','PCNX4'],
        ['KANSL1','LINC02210'],
        ['KANSL1','LRRC37A3'],
        ['KANSL3','RIPK4'],
        ['KANSL3','TMCO4'],
        ['KAT2B','SGO1-AS1'],
        ['KCNK7','MAP3K11'],
        ['KDM2A','SF1'],
        ['KDM4B','SAFB2'],
        ['KDM7A','LAPTM5'],
        ['KHDRBS1','KIAA0319L'],
        ['KIAA1024','TMED3'],
        ['KIAA1147','WEE2-AS1'],
        ['KIF15','ZNF501'],
        ['KIF1C','KRT13'],
        ['KIFC1','RNH1'],
        ['KLF2','OAZ1'],
        ['KLHL36','ZDHHC7'],
        ['KLK4','KLKP1'],
        ['KMT2C','TPTE'],
        ['KMT2E','SRSF7'],
        ['KMT5A','RFLNA'],
        ['KRT13','SCP2'],
        ['KRT4','LY6D'],
        ['KRT8','VIM'],
        ['LACTB2-AS1','XKR9'],
        ['LAMA3','RBBP8'],
        ['LAMTOR1','NUMA1'],
        ['LAPTM5','RIPOR1'],
        ['LARP1','NR3C1'],
        ['LATS1','NUP43'],
        ['LDAH','PUM2'],
        ['LGMN','MYH11'],
        ['LINC00467','TRAF5'],
        ['LINC00535','UBE2V2'],
        ['LINC00886','TIPARP-AS1'],
        ['LINC01068','NDFIP2'],
        ['LINC01135','LINC01358'],
        ['LINC01169','SMAD6'],
        ['LINC01372','MEF2C'],
        ['LINC01470','NEDD1'],
        ['LINC01578','YPEL5'],
        ['LINC01578','ZBTB7A'],
        ['LINC01608','LINC01609'],
        ['LINC01619','LINC02391'],
        ['LINC01705','LINC02257'],
        ['LINC01872','SIGLECL1'],
        ['LINC02219','PIK3R1'],
        ['LINC02246','NRIP1'],
        ['LINC02345','TMCO5A'],
        ['LMO1','RIC3'],
        ['LPAR2','PBX4'],
        ['LPGAT1','RPS6KC1'],
        ['LRP12','PKP4'],
        ['LRP2BP','UFSP2'],
        ['LRP5','LRP5L'],
        ['LRPAP1','WDR1'],
        ['LRRC37A16P','LRRC37B'],
        ['LRRC37A3','NSF'],
        ['LRRC57','PLEKHA1'],
        ['LRRC59','MRPL27'],
        ['LY6K','THEM6'],
        ['LYST','SETX'],
        ['LYZ','MARF1'],
        ['LYZ','MORF4L1'],
        ['LYZ','PIGQ'],
        ['LYZ','SYNE3'],
        ['LYZ','WAC-AS1'],
        ['LYZ','Z99129.3'],
        ['MAD1L1','MAFK'],
        ['MAML2','MTMR2'],
        ['MAN2A1','UIMC1'],
        ['MAN2C1','PLA2G4F'],
        ['MANBA','UBE2D3'],
        ['MANBAL','SRC'],
        ['MAP3K8','PRKG1'],
        ['MAPKBP1','MGA'],
        ['MARCKS','PTMA'],
        ['MAST2','TNNT2'],
        ['MECOM','RPL22'],
        ['MEF2C','TMEM161B'],
        ['MEIS2','MUC7'],
        ['METRNL','RALGDS'],
        ['MGRN1','NUDT16L1'],
        ['MICAL2','TEAD1'],
        ['MINOS1','MMD'],
        ['MINPP1','NUTM2A-AS1'],
        ['MIR100HG','PIGR'],
        ['MIR100HG','SSR3'],
        ['MIR22HG','TLCD2'],
        ['MLEC','UNC119B'],
        ['MLIP','TINAG'],
        ['MOB1B','VLDLR-AS1'],
        ['MPO','TBC1D9B'],
        ['MRC1','TMEM236'],
        ['MSMB','NCOA4'],
        ['MSRB3','TMBIM4'],
        ['MTA3','THADA'],
        ['MTHFD1L','PLEKHG1'],
        ['MTHFSD','STIM2'],
        ['MTMR10','TRPM1'],
        ['MTSS1','TATDN1'],
        ['MTSS1L','VAC14'],
        ['MTURN','PLEKHA8'],
        ['MUC2','MUC5B'],
        ['MYH11','SCARB2'],
        ['MYH6','TIMP3'],
        ['MYH6','TPT1'],
        ['MYL3','TNNC1'],
        ['MYO10','TSPAN18'],
        ['MYO5C','NDUFS6'],
        ['NACA','PRIM1'],
        ['NAIP','OCLN'],
        ['NANP','NINL'],
        ['NCKAP1','SMR3B'],
        ['NCKAP5','R3HDM1'],
        ['NCOA6','PIGU'],
        ['NCOR2','UBC'],
        ['NDEL1','UBB'],
        ['NDUFV3','PKNOX1'],
        ['NECAP2','SZRD1'],
        ['NFATC1','PQLC1'],
        ['NFATC3','PLA2G15'],
        ['NIN','SAV1'],
        ['NMU','PDCL2'],
        ['NOXRED1','TMED8'],
        ['NPEPL1','STX16'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB15','PDXDC2P-NPIPB14P'],
        ['NPIPB5','SMG1'],
        ['NPPA','SORBS2'],
        ['NQO2','PABPC1L'],
        ['NRAV','SRSF9'],
        ['NRBP1','TG'],
        ['NSMAF','SPTLC3'],
        ['NUCB1-AS1','TULP2'],
        ['NUP188','PHYHD1'],
        ['NUP210','STOM'],
        ['NUSAP1','OIP5-AS1'],
        ['OAZ1','PFKFB3'],
        ['OAZ1','PTBP1'],
        ['OAZ1','RAC2'],
        ['OAZ1','SEPT5'],
        ['OAZ1','TCF3'],
        ['OAZ1','UBTF'],
        ['OSBPL8','UBAC2'],
        ['PACSIN2','TOB1'],
        ['PAN2','STAT2'],
        ['PATJ','TG'],
        ['PBXIP1','PMVK'],
        ['PCM1','SPIDR'],
        ['PCP4L1','SDHC'],
        ['PDZRN3','STATH'],
        ['PEMT','UBB'],
        ['PFDN1','SKP1'],
        ['PFKFB3','S100A9'],
        ['PIAS2','ST8SIA5'],
        ['PICALM','SLC38A1'],
        ['PID1','SFTPA1'],
        ['PIM3','SCO2'],
        ['PIM3','TRABD'],
        ['PINK1-AS','TAS2R4'],
        ['PITPNA','YWHAE'],
        ['PITPNC1','PRKCA'],
        ['PKHD1L1','STX8'],
        ['PLA2G15','WBP1L'],
        ['PLCG1','TG'],
        ['PLEKHB2','ZNF727'],
        ['PLSCR4','RNF13'],
        ['PMS2','POLR2J3'],
        ['PNLIP','STAG3L5P'],
        ['PNLIP','UBA52'],
        ['POLR2A','SMR3B'],
        ['PPM1B','SLC3A1'],
        ['PPM1H','USP15'],
        ['PPP1CB','WSB1'],
        ['PPP1CB','YPEL5'],
        ['PPP1R12C','TSPAN4'],
        ['PRC1','VPS33B'],
        ['PRDX1','TESK2'],
        ['PRH2','SASH1'],
        ['PRH2','TACC2'],
        ['PRKAG2','PSD3'],
        ['PRKAG2','RHEB'],
        ['PRKCH','PRPS2'],
        ['PRKCZ','SKI'],
        ['PRKRIP1','RBAK'],
        ['PRMT7','SLC7A6'],
        ['PRPF39','TOGARAM1'],
        ['PRSS3','TMEM45B'],
        ['PRSS42','PRSS50'],
        ['PRSS51','TNKS'],
        ['PSMA5','SORT1'],
        ['PSMB8','TAP1'],
        ['PSMF1','SIPA1L1'],
        ['PSPC1','ZMYM5'],
        ['PTBP1','RNF126'],
        ['PTBP1','SBNO2'],
        ['PTBP3','SUSD1'],
        ['PTCSC2','RNF144B'],
        ['PTMA','SRSF5'],
        ['PTMA','WDR1'],
        ['PTMA','ZEB2'],
        ['PTP4A1','SRSF5'],
        ['PTPN4','SP140L'],
        ['PVRIG2P','STAG3'],
        ['PXK','RPP14'],
        ['PXYLP1','SPSB4'],
        ['R3HDM1','TRAPPC12'],
        ['RAB11B','SEPHS1'],
        ['RAB35','THEMIS2'],
        ['RARA','RCN1'],
        ['RASA3','TFDP1'],
        ['RASSF8','SSPN'],
        ['RBAK','RNF216'],
        ['RBM38','SUMO2'],
        ['RBMS2','SPRYD4'],
        ['RBMS2','TG'],
        ['RDH11','VTI1B'],
        ['RECQL5','SENP6'],
        ['RHOF','UBC'],
        ['RHOQ','TMEM247'],
        ['RIPK1','SERPINB9'],
        ['RLN1','RLN2'],
        ['RNASET2','SRGN'],
        ['RNF125','RNF138'],
        ['RNF139','TRMT12'],
        ['RNF151','TBL3'],
        ['RNF31','THADA'],
        ['RPL13','SPG7'],
        ['RPL14','SRP14'],
        ['RPL27','RUNDC1'],
        ['RPL28','TMEM190'],
        ['RPL38','TTYH2'],
        ['RPS18','TG'],
        ['RPS19','SNUPN'],
        ['RPS9','UBB'],
        ['RWDD3','TMEM56'],
        ['SAMD5','SASH1'],
        ['SAMSN1','USP25'],
        ['SCAF4','SERGEF'],
        ['SCARB1','UBC'],
        ['SCNN1A','TNFRSF1A'],
        ['SDC2','VMP1'],
        ['SDF4','TNFRSF4'],
        ['SENP3-EIF4A1','TNFSF13'],
        ['SERF1A','SMN1'],
        ['SH3KBP1','TYROBP'],
        ['SIDT2','TAGLN'],
        ['SIRT2','XPNPEP1'],
        ['SLC19A1','SUMO3'],
        ['SLC20A2','WDR45'],
        ['SLC25A3','TOM1'],
        ['SLC25A5','TRB@'],
        ['SLC2A11','SMARCB1'],
        ['SLC33A1','UBP1'],
        ['SLC39A8','TRB@'],
        ['SLC6A16','TEAD2'],
        ['SLFN11','TG'],
        ['SMAP2','ZFP69B'],
        ['SMARCA2','TG'],
        ['SMG1','ZNF623'],
        ['SMR3B','SPINT2'],
        ['SND1','ZNF704'],
        ['SNRPB','SNURF'],
        ['SNRPN','WDR4'],
        ['SNTB2','VPS4A'],
        ['SNX29','TG'],
        ['SPATA3','ZNF483'],
        ['ST3GAL1','UBB'],
        ['STAT3','TRB@'],
        ['STK10','UBE2D2'],
        ['STK4-AS1','TOMM34'],
        ['TAMM41','VGLL4'],
        ['TBCEL','TECTA'],
        ['TDRD10','UBE2Q1-AS1'],
        ['TFDP1','TMCO3'],
        ['TFDP1','ZEB2'],
        ['TFDP2','XRN1'],
        ['TG','TIE1'],
        ['TG','ZNF12'],
        ['TMEM225B','ZNF655'],
        ['TMEM225B','ZSCAN25'],
        ['TOP1','TPRXL'],
        ['TPM1','UQCRB'],
        ['TPM1','XIRP2'],
        ['TRPM7','USP50'],
        ['TTC32','WDR35'],
        ['UBA2','WTIP'],
        ['USP10','ZDHHC7'],
        ['USP34','XPO1'],
        ['VRK2','WDPCP'],
        ['YPEL5','ZEB2'],
        ['Z93930.2','ZNRF3'],
        ['ZC2HC1A','ZFX'],
        ['ZDHHC12','ZER1'],
        ['ZFP30','ZNF571'],
        ['ZNF32-AS3','ZNF485'],
        ['ZNF616','ZNF836'],
        ['ZNF746','ZNF767P'],
        ['ZNF780A','ZNF780B'],
        ['ZNF815P','ZNF890P'],
        ['ABL1','EXOSC2'],
        ['ABR','CLUH'],
        ['AC005258.1','BSG'],
        ['AC005258.1','CCNG2'],
        ['AC005258.1','DDX5'],
        ['AC005258.1','SEPT5'],
        ['AC005258.1','UBTF'],
        ['AC009093.10','RRN3P2'],
        ['AC009093.9','CDR2'],
        ['AC010186.4','DCK'],
        ['AC013394.1','ELANE'],
        ['AC013394.1','ZBTB7A'],
        ['AC016876.2','DDX5'],
        ['AC023790.2','GPRC5D-AS1'],
        ['AC067930.2','HBB'],
        ['AC068580.4','LSP1'],
        ['AC079781.5','LINC00937'],
        ['AC092821.3','CLEC2D'],
        ['AC118758.2','AC138035.3'],
        ['AC244517.10','AC244517.6'],
        ['ACCS','EXT2'],
        ['ACSS1','APMAP'],
        ['ACTN4','EIF3K'],
        ['ADGRG7','TFG'],
        ['AGAP5','BMS1P4'],
        ['AKT2','C19ORF47'],
        ['AL022322.2','GALNT4'],
        ['ALG14','CNN3'],
        ['ANAPC1','RMND5A'],
        ['AP000345.2','DRICH1'],
        ['AP006222.1','DUX4'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARPC4-TTLL3','OGG1'],
        ['ATG16L1','INPP5D'],
        ['ATP5I','MFSD7'],
        ['AZGP1','GJC3'],
        ['B3GNTL1','METRNL'],
        ['BANK1','IGH@'],
        ['BCAP29','DUS4L'],
        ['BLOC1S6','SQOR'],
        ['BOLA2-SMG1P6','SMG1'],
        ['BPTF','LRRC37A2'],
        ['C12ORF73','HSP90B1'],
        ['C15ORF38-AP3S2','YPEL5'],
        ['C19MC','RPLP1'],
        ['C19ORF33','SPINT2'],
        ['C6ORF106','SPDEF'],
        ['CABP7','NF2'],
        ['CATSPER2','PPIP5K1'],
        ['CBWD2','DOCK8'],
        ['CBX3','CCDC32'],
        ['CD81','TSSC4'],
        ['CD83','JARID2'],
        ['CDC42EP2','POLA2'],
        ['CDK2','RAB5B'],
        ['CDR2','RRN3P2'],
        ['CFD','ELANE'],
        ['CITED1','HDAC8'],
        ['COLQ','HACL1'],
        ['COMMD3-BMI1','DDX5'],
        ['COQ8B','NUMBL'],
        ['COX19','MAFK'],
        ['CPM','LYZ'],
        ['CRCP','TPST1'],
        ['CTDP1','PQLC1'],
        ['CTSC','RAB38'],
        ['CXCR4','UBXN4'],
        ['CXCR4','ZEB2'],
        ['CYTIP','ERMN'],
        ['DCUN1D1','MCCC1'],
        ['DDX42','MAP3K3'],
        ['DHX35','FAM83D'],
        ['DHX9','NPL'],
        ['DOCK9','STK24'],
        ['EIF4E3','FOXP1'],
        ['ENTPD5','FAM161B'],
        ['FAM13A-AS1','HERC3'],
        ['FAM157B','FBXO25'],
        ['FOXP1','RYBP'],
        ['GAS7','RCVRN'],
        ['GGT1','IGSF3'],
        ['GNAQ','LYZ'],
        ['GNAS','SENP3-EIF4A1'],
        ['GOLT1A','KISS1'],
        ['HAPLN3','MFGE8'],
        ['HBB','RNF216'],
        ['HBB','TTC7A'],
        ['HBB','TTN'],
        ['HBB','USP15'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['IGH@','LAPTM5'],
        ['IGH@','MIR100HG'],
        ['IGH@','NFKB2'],
        ['IGH@','PCSK7'],
        ['IGH@','ZFP36L1'],
        ['IQGAP1','ZNF774'],
        ['KANSL1','LRRC37A3'],
        ['KLF2','OAZ1'],
        ['KLK4','KLKP1'],
        ['KMT2C','TPTE'],
        ['KMT5A','ZNF664'],
        ['LPGAT1','RPS6KC1'],
        ['LRRC37A3','NSF'],
        ['LYZ','NFATC2IP'],
        ['MANBAL','SRC'],
        ['MAPKBP1','MGA'],
        ['MICAL2','TEAD1'],
        ['MTSS1','TATDN1'],
        ['NACA','PRIM1'],
        ['NAIP','OCLN'],
        ['NCOA6','PIGU'],
        ['NCOR2','UBC'],
        ['NDRG1','ST3GAL1'],
        ['NFATC3','PLA2G15'],
        ['NFKBIZ','SRGN'],
        ['NPEPL1','STX16'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['PIM3','SCO2'],
        ['PIM3','TRABD'],
        ['PMS2','POLR2J3'],
        ['PPP1CB','YPEL5'],
        ['PRDX1','TESK2'],
        ['PRKAG2','RHEB'],
        ['PXK','RPP14'],
        ['RAP1B','RBPJ'],
        ['RBMS2','SPRYD4'],
        ['RPL14','SRP14'],
        ['RPL38','TTYH2'],
        ['SCNN1A','TNFRSF1A'],
        ['SERF1A','SMN1'],
        ['SIDT2','TAGLN'],
        ['STK10','UBE2D2'],
        ['TBCEL','TECTA'],
        ['TFDP2','XRN1'],
        ['UBA2','WTIP'],
        ['USP34','XPO1'],
        ['AATF','AC010542.3'],
        ['ABL1','EXOSC2'],
        ['ABR','CLUH'],
        ['ABR','PRB2'],
        ['AC002398.2','TNNT2'],
        ['AC005041.1','RTKN'],
        ['AC005258.1','BSG'],
        ['AC005258.1','CCNG2'],
        ['AC005258.1','DDX5'],
        ['AC005258.1','SEPT5'],
        ['AC005258.1','UBTF'],
        ['AC007000.3','AC211486.3'],
        ['AC008770.1','ZNF625'],
        ['AC008906.1','CAST'],
        ['AC009093.10','RRN3P2'],
        ['AC009093.9','CDR2'],
        ['AC010186.4','DCK'],
        ['AC013394.1','ELANE'],
        ['AC013394.1','ZBTB7A'],
        ['AC015468.1','LINC02153'],
        ['AC015688.4','AC098850.5'],
        ['AC015688.4','NOS2'],
        ['AC016876.2','DDX5'],
        ['AC016912.1','RHOQ'],
        ['AC018630.6','CST2'],
        ['AC018630.6','DGCR2'],
        ['AC018630.6','KIAA1324'],
        ['AC018630.6','TRPV6'],
        ['AC018638.5','AL606534.5'],
        ['AC019133.1','LINC02494'],
        ['AC019349.1','KRT6B'],
        ['AC020656.1','ACTG1'],
        ['AC020656.1','IVD'],
        ['AC020656.1','NEBL'],
        ['AC020656.1','NT5C2'],
        ['AC020656.1','PIK3R2'],
        ['AC020656.1','PRH1'],
        ['AC020915.6','ZSCAN22'],
        ['AC023790.2','GPRC5D-AS1'],
        ['AC025279.2','SNX29'],
        ['AC026412.1','AC124944.3'],
        ['AC027612.2','IGK@'],
        ['AC067930.2','ANKRD19P'],
        ['AC067930.2','HBB'],
        ['AC068580.4','LSP1'],
        ['AC068587.6','AC079781.5'],
        ['AC073063.1','SARNP'],
        ['AC073610.3','ARF1'],
        ['AC079781.5','LINC00937'],
        ['AC079781.5','RBFOX1'],
        ['AC087203.3','AF228730.7'],
        ['AC091390.4','RBAK'],
        ['AC092128.1','AC138627.1'],
        ['AC092821.3','CLEC2D'],
        ['AC092821.3','GOT2'],
        ['AC092821.3','TPT1-AS1'],
        ['AC104836.1','C1ORF146'],
        ['AC107918.4','DEFB131E'],
        ['AC109583.3','PRSS46'],
        ['AC118758.2','AC138035.3'],
        ['AC119674.2','TG'],
        ['AC124312.1','WDR4'],
        ['AC124944.3','LINC00969'],
        ['AC124944.3','PDCD6'],
        ['AC133919.3','AL606534.6'],
        ['AC244517.10','AC244517.6'],
        ['ACCS','EXT2'],
        ['ACE2','PIR'],
        ['ACSS1','APMAP'],
        ['ACTA1','ACTA2'],
        ['ACTB','XPO1'],
        ['ACTC1','ACTG2'],
        ['ACTN4','EIF3K'],
        ['ADAP1','SUN1'],
        ['ADGRE2','ADGRE5'],
        ['ADGRG7','TFG'],
        ['AFF1','PTPN13'],
        ['AFP','ALB'],
        ['AGAP5','BMS1P4'],
        ['AKT2','C19ORF47'],
        ['AL022322.2','GALNT4'],
        ['AL031432.4','TG'],
        ['AL132639.3','TRAPPC6B'],
        ['AL133481.3','AL359195.2'],
        ['AL136116.3','PARL'],
        ['AL136295.5','THADA'],
        ['AL136295.7','CHMP4A'],
        ['AL136309.2','C6ORF201'],
        ['AL138963.3','PRB2'],
        ['AL139142.1','TEX50'],
        ['AL162400.2','LINC01739'],
        ['AL627309.5','AL732372.2'],
        ['AL645937.4','AL672167.1'],
        ['AL662791.5','OR2J3'],
        ['ALDH2','ANKHD1-EIF4EBP3'],
        ['ALDH6A1','SPAST'],
        ['ALG14','CNN3'],
        ['ALKBH5','LLGL1'],
        ['AMACR','SLC45A2'],
        ['AMY2A','CPA1'],
        ['AMY2B','PRH1'],
        ['ANAPC1','RMND5A'],
        ['ANK2','CAMK2D'],
        ['ANKRD18B','FAM95C'],
        ['ANP32B','IGH@'],
        ['AP000345.2','DRICH1'],
        ['AP000553.4','PPIL2'],
        ['AP001350.2','CNTF'],
        ['AP001372.3','TCF3'],
        ['AP006222.1','DUX4'],
        ['AP3S1','COMMD10'],
        ['APBB3','SRA1'],
        ['ARHGAP18','TMEM244'],
        ['ARHGEF12','SMR3B'],
        ['ARHGEF17','P2RY6'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARL6IP1','RPS15A'],
        ['ARPC4-TTLL3','OGG1'],
        ['ARSD','EPN1'],
        ['ART4','SMCO3'],
        ['ASL','CRCP'],
        ['ASNS','RBFOX1'],
        ['ATAD1','PTEN'],
        ['ATG16L1','INPP5D'],
        ['ATP1B1','CEL'],
        ['ATP5I','MFSD7'],
        ['AZGP1','GJC3'],
        ['B3GNTL1','METRNL'],
        ['BABAM2','RBKS'],
        ['BANK1','IGH@'],
        ['BCAP29','DUS4L'],
        ['BCAS3','TG'],
        ['BCKDHA','TMEM91'],
        ['BEST1','IGF2'],
        ['BGLAP','PMF1'],
        ['BLOC1S6','SQOR'],
        ['BMPR1B','PDLIM5'],
        ['BOLA2-SMG1P6','SMG1'],
        ['BPIFB1','IGH@'],
        ['BPTF','LRRC37A2'],
        ['BRK1','VHL'],
        ['BSDC1','FAM229A'],
        ['C12ORF73','HSP90B1'],
        ['C15ORF38-AP3S2','YPEL5'],
        ['C19MC','RPLP1'],
        ['C19ORF33','SPINT2'],
        ['C1ORF194','UQCR10'],
        ['C2ORF27B','LINC02564'],
        ['C6ORF106','SPDEF'],
        ['CABP7','NF2'],
        ['CATSPER2','PPIP5K1'],
        ['CAVIN1','STAT3'],
        ['CBWD2','DOCK8'],
        ['CBX3','CCDC32'],
        ['CCL28','PRH1'],
        ['CD74','PRH1'],
        ['CD81','TSSC4'],
        ['CD83','JARID2'],
        ['CDC42EP2','POLA2'],
        ['CDK11B','SLC35E2'],
        ['CDK19','REV3L'],
        ['CDK2','RAB5B'],
        ['CDKN1A','RAB44'],
        ['CDR2','RRN3P2'],
        ['CEL','CELP'],
        ['CELA3A','NEAT1'],
        ['CFD','ELANE'],
        ['CHST11','IGH@'],
        ['CITED1','HDAC8'],
        ['CLCF1','POLD4'],
        ['CLTC','VMP1'],
        ['COL14A1','DEPTOR'],
        ['COL3A1','COL4A2'],
        ['COLQ','HACL1'],
        ['COMMD3-BMI1','DDX5'],
        ['COQ8B','NUMBL'],
        ['COX19','MAFK'],
        ['CPA1','DPEP1'],
        ['CPD','STATH'],
        ['CPM','LYZ'],
        ['CRCP','TPST1'],
        ['CTDP1','PQLC1'],
        ['CTSC','RAB38'],
        ['CTSD','PRB2'],
        ['CX3CL1','PRH1'],
        ['CXCR4','UBXN4'],
        ['CXCR4','ZEB2'],
        ['CYTIP','ERMN'],
        ['DCUN1D1','MCCC1'],
        ['DDX42','MAP3K3'],
        ['DEPDC1B','PDE4D'],
        ['DES','PIK3IP1'],
        ['DHX35','FAM83D'],
        ['DHX9','NPL'],
        ['DIP2B','TMBIM6'],
        ['DLGAP4','MUC7'],
        ['DOCK9','STK24'],
        ['DPH7','PNPLA7'],
        ['EEF1A1','TG'],
        ['EEF1AKMT3','TSFM'],
        ['EEF2','PRH1'],
        ['EIF2S1','RBM25'],
        ['EIF4E3','FOXP1'],
        ['ELK4','SLC45A3'],
        ['ENTPD5','FAM161B'],
        ['ERBB2','GRB7'],
        ['ETFRF1','KRAS'],
        ['EZH2','ZNF425'],
        ['FAM129C','PGLS'],
        ['FAM13A-AS1','HERC3'],
        ['FAM157B','FBXO25'],
        ['FAM53B','ST20-MTHFS'],
        ['FBXO25','TDRP'],
        ['FGFR4','ZNF346'],
        ['FLJ22447','PRKCH'],
        ['FOXP1','RYBP'],
        ['FRMD4A','PRB3'],
        ['GAPDH','PRH1'],
        ['GAS7','RCVRN'],
        ['GGT1','IGSF3'],
        ['GNAQ','LYZ'],
        ['GNAS','SENP3-EIF4A1'],
        ['GOLT1A','KISS1'],
        ['GPT','PPP1R16A'],
        ['GYPB','SAV1'],
        ['H1F0','PRH1'],
        ['H6PD','SPSB1'],
        ['HAPLN3','MFGE8'],
        ['HBB','RNF216'],
        ['HBB','TTC7A'],
        ['HBB','TTN'],
        ['HBB','USP15'],
        ['HORMAD1','SLC22A20'],
        ['HSD17B14','PLEKHA4'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['HTN3','MBNL1'],
        ['IGF2','RBM3'],
        ['IGH@','LAPTM5'],
        ['IGH@','MIR100HG'],
        ['IGH@','MUC1'],
        ['IGH@','NFKB2'],
        ['IGH@','PCSK7'],
        ['IGH@','TRA@'],
        ['IGH@','ZFP36L1'],
        ['IGK@','QSOX1'],
        ['IGSF10','MED12L'],
        ['IQGAP1','ZNF774'],
        ['ITM2B','RB1'],
        ['KANSL1','LRRC37A3'],
        ['KLF2','OAZ1'],
        ['KLK1','PRSS1'],
        ['KLK4','KLKP1'],
        ['KLK7','KLK8'],
        ['KMT2C','TPTE'],
        ['KMT5A','ZNF664'],
        ['LAMA3','RBBP8'],
        ['LARP1','SMR3B'],
        ['LATS1','NUP43'],
        ['LDAH','PUM2'],
        ['LPGAT1','RPS6KC1'],
        ['LRPPRC','PNLIP'],
        ['LRRC37A3','NSF'],
        ['LTBP2','PRH2'],
        ['LYNX1','SLURP2'],
        ['LYZ','NFATC2IP'],
        ['LYZ','SFXN1'],
        ['MAML2','MTMR2'],
        ['MANBAL','SRC'],
        ['MAPKBP1','MGA'],
        ['MECOM','RPL22'],
        ['MICAL2','TEAD1'],
        ['MINPP1','NUTM2A-AS1'],
        ['MLEC','UNC119B'],
        ['MRC1','TMEM236'],
        ['MSMB','NCOA4'],
        ['MSRB3','TMBIM4'],
        ['MTA3','THADA'],
        ['MTSS1','TATDN1'],
        ['MUC7','NEBL'],
        ['MYH11','PPT1'],
        ['MYH9','TRB@'],
        ['MYL2','MYL9'],
        ['NACA','PRIM1'],
        ['NAIP','OCLN'],
        ['NCOA6','PIGU'],
        ['NCOR2','UBC'],
        ['NDRG1','ST3GAL1'],
        ['NFATC3','PLA2G15'],
        ['NFKBIZ','SRGN'],
        ['NIN','SAV1'],
        ['NPEPL1','STX16'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['NUP188','PHYHD1'],
        ['ODAM','PRH1'],
        ['OR6D1P','SMG1'],
        ['OR7E161P','OR7E91P'],
        ['PBXIP1','PMVK'],
        ['PIK3R1','PRH1'],
        ['PIM3','SCO2'],
        ['PIM3','TRABD'],
        ['PITPNC1','PRKCA'],
        ['PMS2','POLR2J3'],
        ['PNLIP','RAPGEF1'],
        ['PPM1B','SLC3A1'],
        ['PPP1CB','YPEL5'],
        ['PRB3','SERF2'],
        ['PRDX1','TESK2'],
        ['PRH1','PRR27'],
        ['PRH1','RBM47'],
        ['PRH1','RPL3'],
        ['PRH1','SFRP1'],
        ['PRH1','SIX4'],
        ['PRH1','SLC6A8'],
        ['PRH1','SND1'],
        ['PRH1','SORL1'],
        ['PRH1','SQSTM1'],
        ['PRKAG2','RHEB'],
        ['PRKCZ','SKI'],
        ['PRMT7','SLC7A6'],
        ['PSMB8','TAP1'],
        ['PSPC1','ZMYM5'],
        ['PTBP3','SUSD1'],
        ['PXK','RPP14'],
        ['RAP1B','RBPJ'],
        ['RASSF8','SSPN'],
        ['RBAK-RBAKDN','RNF216'],
        ['RBMS2','SPRYD4'],
        ['RDH11','VTI1B'],
        ['RIPK1','SERPINB9'],
        ['RLN1','RLN2'],
        ['RNF125','RNF138'],
        ['RPL14','SRP14'],
        ['RPL28','TMEM190'],
        ['RPL38','TTYH2'],
        ['RWDD3','TMEM56'],
        ['SAMD5','SASH1'],
        ['SCARB1','UBC'],
        ['SCNN1A','TNFRSF1A'],
        ['SERF1A','SMN1'],
        ['SIDT2','TAGLN'],
        ['SLC39A10','TG'],
        ['SLC6A16','TEAD2'],
        ['SMR3B','STIM1'],
        ['SMR3B','UVSSA'],
        ['SNTB2','VPS4A'],
        ['SREBF2','STATH'],
        ['STATH','TMED3'],
        ['STK10','UBE2D2'],
        ['STXBP4','TOM1L1'],
        ['TBCEL','TECTA'],
        ['TFDP2','XRN1'],
        ['TTTY15','USP9Y'],
        ['UBA2','WTIP'],
        ['USP10','ZDHHC7'],
        ['USP34','XPO1'],
        ['ZNF229','ZNF285'],
        ['ZNF354C','ZNF879'],
        ['ZNF586','ZNF587'],
        ['ABCA10','ABCA6'],
        ['ABCA8','ABCA9'],
        ['AC002480.1','AC002480.2'],
        ['AC004687.2','HSF5'],
        ['AC004967.2','CCZ1'],
        ['AC005154.2','GGCT'],
        ['AC005332.4','AMZ2'],
        ['AC005332.4','BPTF'],
        ['AC005479.1','NPC2'],
        ['AC005832.1','AC005833.1'],
        ['AC006001.3','AC027644.4'],
        ['AC006059.2','HHATL'],
        ['AC006518.7','ARHGAP32'],
        ['AC006518.7','CEACAM1'],
        ['AC006518.7','CST1'],
        ['AC006518.7','CST4'],
        ['AC006518.7','DMBT1'],
        ['AC006518.7','IGH@'],
        ['AC006518.7','IGK@'],
        ['AC006518.7','LPO'],
        ['AC006518.7','MUC7'],
        ['AC006518.7','PRB1'],
        ['AC006518.7','PRB2'],
        ['AC006518.7','PRB3'],
        ['AC006518.7','PRB4'],
        ['AC006518.7','RBM47'],
        ['AC006518.7','SFRP1'],
        ['AC006518.7','SMR3B'],
        ['AC006518.7','STATH'],
        ['AC007207.2','AP000943.2'],
        ['AC007448.2','AC007448.4'],
        ['AC008676.1','AC016571.1'],
        ['AC009078.2','AL590556.3'],
        ['AC009090.6','RSPRY1'],
        ['AC010327.1','CLCN6'],
        ['AC010329.1','ERVE-1'],
        ['AC015908.2','TMEM220-AS1'],
        ['AC015922.3','IL6ST'],
        ['AC017002.6','CYTOR'],
        ['AC020656.1','ACTB'],
        ['AC020656.1','APP'],
        ['AC020656.1','FAM120AOS'],
        ['AC020656.1','TRA@'],
        ['AC020915.1','ZSCAN22'],
        ['AC020978.3','ESRP2'],
        ['AC021097.2','ICA1'],
        ['AC021517.2','SLC14A2-AS1'],
        ['AC023509.1','SMR3B'],
        ['AC024940.1','AC092821.1'],
        ['AC025287.1','CHST5'],
        ['AC026979.4','SLC25A3'],
        ['AC027644.4','LINC00174'],
        ['AC068389.1','SLC5A12'],
        ['AC068587.4','AC079781.5'],
        ['AC068587.4','AC127526.2'],
        ['AC068587.4','LINC00937'],
        ['AC068831.1','CERK'],
        ['AC068896.1','ST20-MTHFS'],
        ['AC087190.3','LYZ'],
        ['AC087203.3','AF228730.5'],
        ['AC087672.2','LY96'],
        ['AC090517.5','ZNF280D'],
        ['AC091057.3','GOLGA8N'],
        ['AC091132.5','LRRC37A3'],
        ['AC091982.3','RPLP1P6'],
        ['AC092745.1','AF228730.5'],
        ['AC093484.3','ANKRD1'],
        ['AC093484.3','HBA2'],
        ['AC093484.3','HBB'],
        ['AC093484.3','UBC'],
        ['AC093512.2','PPP4C'],
        ['AC093512.2','SMR3B'],
        ['AC093827.5','SLC10A6'],
        ['AC097372.2','LINC01095'],
        ['AC097634.4','PROK2'],
        ['AC098588.2','GYPB'],
        ['AC098588.2','GYPE'],
        ['AC098850.3','FAM106A'],
        ['AC098850.3','USP32'],
        ['AC098850.4','NOS2'],
        ['AC098850.4','USP6'],
        ['AC099548.2','TCAF2C'],
        ['AC103718.1','GSDMC'],
        ['AC104472.3','UBP1'],
        ['AC105052.4','RASA4CP'],
        ['AC108010.1','LINC01002'],
        ['AC108134.3','ZNF600'],
        ['AC114490.1','TMEM35B'],
        ['AC120057.2','CLCN6'],
        ['AC124944.3','MUC20-OT1'],
        ['AC126283.2','ALB'],
        ['AC126544.1','BRWD1'],
        ['AC127526.2','LINC00937'],
        ['AC127894.1','CPSF6'],
        ['AC131180.1','NF1'],
        ['AC133644.3','PLG'],
        ['AC138207.8','SMURF2P1-LRRC37BP1'],
        ['AC139491.5','FAM153C'],
        ['AC239809.3','HYDIN2'],
        ['AC241377.4','LINC00869'],
        ['AC242988.2','PLEKHO1'],
        ['AC244033.2','OTUD7B'],
        ['AC245595.1','IGK@'],
        ['ACTA1','ACTC1'],
        ['ACTC1','CLCN6'],
        ['ACTN4','SMR3B'],
        ['ADAMDEC1','CSNK2A1'],
        ['ADH1A','ADH4'],
        ['ADH1B','ADH4'],
        ['ADH4','ADH5'],
        ['AF254983.1','TPTE2'],
        ['AGK','TG'],
        ['AHNAK','AHNAK2'],
        ['AKT1','AKT2'],
        ['AL031281.2','SLC25A37'],
        ['AL031281.2','TRB@'],
        ['AL031281.2','ZEB2'],
        ['AL035078.4','RARA'],
        ['AL035661.1','ERVE-1'],
        ['AL049634.2','SIRPA'],
        ['AL133353.2','SLC25A28'],
        ['AL133371.1','AL355922.4'],
        ['AL135905.2','B2M'],
        ['AL135905.2','CXCR4'],
        ['AL135905.2','DDX5'],
        ['AL135905.2','EEF1A1'],
        ['AL135905.2','G3BP2'],
        ['AL135905.2','GABPB1'],
        ['AL135905.2','HIF1A'],
        ['AL135905.2','HNRNPA2B1'],
        ['AL135905.2','HNRNPH1'],
        ['AL135905.2','SRSF5'],
        ['AL137058.1','AL158066.1'],
        ['AL137058.1','MRPS31P5'],
        ['AL137802.3','BX284668.5'],
        ['AL157935.2','ST6GALNAC4'],
        ['AL158066.1','SUGT1P4-STRA6LP'],
        ['AL161621.1','MRAP2'],
        ['AL162377.3','ATP7B'],
        ['AL162417.1','METRNL'],
        ['AL355312.2','LRP11'],
        ['AL355499.1','AL591485.1'],
        ['AL356275.1','AL356275.2'],
        ['AL358113.1','LYZ'],
        ['AL358777.1','PAK1IP1'],
        ['AL359715.1','BCKDHB'],
        ['AL359762.1','SPATA1'],
        ['AL365203.2','ITGB1'],
        ['AL365394.1','SLC5A12'],
        ['AL390961.1','FAM238B'],
        ['AL390961.1','FAM238C'],
        ['AL392086.2','LINP1'],
        ['AL512310.11','NF1'],
        ['AL513548.1','AL513548.4'],
        ['AL590556.3','AMY2A'],
        ['AL590556.3','CEL'],
        ['AL590556.3','CELA2A'],
        ['AL590556.3','CELA2B'],
        ['AL590556.3','CLPS'],
        ['AL590556.3','CPA1'],
        ['AL590556.3','CPA2'],
        ['AL590556.3','CTRB1'],
        ['AL590556.3','CTRC'],
        ['AL590556.3','EIF4A2'],
        ['AL590556.3','GP2'],
        ['AL590556.3','KRT8'],
        ['AL590556.3','PNLIP'],
        ['AL590556.3','PRSS1'],
        ['AL590556.3','PRSS2'],
        ['AL590556.3','PRSS3'],
        ['AL590556.3','TRB@'],
        ['AL591485.1','DPY19L2'],
        ['AL645937.2','AL672167.1'],
        ['AL662791.2','OR2J3'],
        ['AL928654.3','CRIP2'],
        ['ALB','CTRB1'],
        ['ALOX15','C17ORF100'],
        ['ANKRD20A11P','AP005212.4'],
        ['ANXA1','ANXA2'],
        ['AP000553.5','PPIL2'],
        ['AP000812.5','FOLR3'],
        ['AP001005.3','TUBB8'],
        ['AP002884.2','IL18'],
        ['APOBEC3F','APOBEC3G'],
        ['APP','GLUL'],
        ['AQP5','AQP6'],
        ['ARCN1','CLCN6'],
        ['ARF1','ARF5'],
        ['ASB11','ASB9'],
        ['ATAD3B','ATAD3C'],
        ['ATP1B1','CYP11A1'],
        ['ATP5F1E','CTSZ'],
        ['ATP5MC3','CLCN6'],
        ['ATP6V1C1','AZIN1-AS1'],
        ['AZU1','ELANE'],
        ['B2M','CELA3A'],
        ['BACE1-AS','CLPS'],
        ['BACE1-AS','PRSS1'],
        ['BASP1','Z84488.2'],
        ['BGN','CLCN6'],
        ['BIRC6','TG'],
        ['BTBD8','C1ORF146'],
        ['BTN2A1','BTN2A2'],
        ['C17ORF51','FLJ36000'],
        ['C1R','HP'],
        ['CCDC180','SUGT1P4-STRA6LP'],
        ['CCDC9B','STRN3'],
        ['CD101','PTGFRN'],
        ['CD24','TG'],
        ['CD44','IGKC'],
        ['CELA2A','CTRB1'],
        ['CELA2A','CTRB2'],
        ['CELA2A','CTRC'],
        ['CELA2B','CTRB1'],
        ['CELA2B','CTRC'],
        ['CELA2B','CTRL'],
        ['CELA3A','CTRB1'],
        ['CELA3A','CTRB2'],
        ['CELA3A','CTRC'],
        ['CELA3B','CTRB1'],
        ['CELA3B','CTRB2'],
        ['CELA3B','CTRC'],
        ['CHD4','GTF2I'],
        ['CHD4','LYZ'],
        ['CHRNA3','CHRNB4'],
        ['CHST5','TMEM231'],
        ['CHST6','TMEM231'],
        ['CIITA','CLEC16A'],
        ['CKM','CLCN6'],
        ['CLCN6','DES'],
        ['CLCN6','FLOT2'],
        ['CLCN6','GABARAPL1'],
        ['CLCN6','HNRNPU'],
        ['CLCN6','ITGA7'],
        ['CLCN6','LRRFIP1'],
        ['CLCN6','MB'],
        ['CLCN6','MFGE8'],
        ['CLCN6','MORF4L1'],
        ['CLCN6','MSN'],
        ['CLCN6','MYH6'],
        ['CLCN6','MYH7'],
        ['CLCN6','NDUFV1'],
        ['CLCN6','NFE2L1'],
        ['CLCN6','PSAP'],
        ['CLCN6','S100A1'],
        ['CLCN6','SPARC'],
        ['CLCN6','SRRM2'],
        ['CLCN6','TGOLN2'],
        ['CLCN6','TNNT2'],
        ['CLCN6','TPM1'],
        ['CLCN6','USB1'],
        ['CLEC12A','CLEC12B'],
        ['CLSTN1','TG'],
        ['CNNM3','CNNM4'],
        ['COL3A1','COL5A2'],
        ['COL4A3BP','PNLIP'],
        ['COL4A5','COL4A6'],
        ['CPA1','CPA2'],
        ['CPA1','CPB1'],
        ['CPA1','SLC4A4'],
        ['CPA2','CPA4'],
        ['CPA3','CPB1'],
        ['CR769776.3','PARG'],
        ['CRYAB','HSPB6'],
        ['CRYZL2P','SEC16B'],
        ['CSNK1E','TPTE2'],
        ['CST1','CST3'],
        ['CST3','CST4'],
        ['CST4','TNS1'],
        ['CTDP1','MUC20-OT1'],
        ['CTRB1','CTRL'],
        ['CTRB1','GATM'],
        ['CTRB1','KPNB1'],
        ['CTRB1','MARK3'],
        ['CTRB1','TMBIM6'],
        ['CTRB2','CTRL'],
        ['CTSE','PGC'],
        ['CYP2C8','CYP2E1'],
        ['DCN','LUM'],
        ['DDX3X','LYZ'],
        ['DDX3X','PNLIP'],
        ['DES','VIM'],
        ['DNAH14','RNF227'],
        ['DUXAP8','NBEA'],
        ['DYNC1H1','SERINC3'],
        ['DYNC1H1','TBL2'],
        ['DYNC1H1','TKT'],
        ['DYRK1A','TG'],
        ['EIF1','EIF1B'],
        ['EIF4A1','EIF4A2'],
        ['ERVK9-11','ZNF433-AS1'],
        ['FAM106A','USP32'],
        ['FAM106A','USP6'],
        ['FAM71F1','FAM71F2'],
        ['FGD5-AS1','MUC20-OT1'],
        ['FMNL1','FMNL3'],
        ['FTH1','FTL'],
        ['GBP1','GBP6'],
        ['GFAP','NEFM'],
        ['GSTM4','GSTM5'],
        ['HBA1','HBB'],
        ['HBA2','HBB'],
        ['HBA2','HBD'],
        ['HBA2','HBM'],
        ['HBB','HBM'],
        ['HBB','PGK1'],
        ['HBB','ZNF395'],
        ['HMGB1','HMGB2'],
        ['HMGB1','ZBED3-AS1'],
        ['HNRNPA2B1','IGH@'],
        ['HSD17B11','HSD17B13'],
        ['HTN1','LPO'],
        ['IGH@','MMP2'],
        ['IGH@','PPIA'],
        ['IL1R2','LINC01127'],
        ['INHBC','INHBE'],
        ['ITGA6','PRB3'],
        ['KMT2C','LYZ'],
        ['KRT10','KRT15'],
        ['KRT15','KRT19'],
        ['KRT2','KRT77'],
        ['KRT7','KRT8'],
        ['KRT72','KRT73'],
        ['LIMD1','PTCSC2'],
        ['LINC-PINT','LINC-PINT'],
        ['LINC01281','LINC01282'],
        ['LINC01297','LINC01297'],
        ['LINC01422','LINC01422'],
        ['LINC02583','RHOQ'],
        ['LINC02583','TMEM247'],
        ['LLPH','Z83840.1'],
        ['LRRC8B','LRRC8C'],
        ['LYZ','MUC20-OT1'],
        ['LYZ','PLBD1'],
        ['LYZ','PPIA'],
        ['LYZ','REEP5'],
        ['LYZ','RF00017'],
        ['LYZ','TJP2'],
        ['MED25','PTOV1'],
        ['MGAM','MGAM2'],
        ['MINK1','SMR3B'],
        ['MIR100HG','TG'],
        ['MMD','SMIM36'],
        ['MSX2','OR4D1'],
        ['MUC20-OT1','SDHA'],
        ['MUC5AC','MUC5B'],
        ['MYH7','MYH7B'],
        ['MYH7','MYH9'],
        ['MYL2','MYL7'],
        ['MYO5A','MYO5C'],
        ['NFIC','PRB2'],
        ['NR2F1','NR2F2'],
        ['NUCB2','ZG16B'],
        ['OAZ1','SEPT5-GP1BB'],
        ['PECAM1','SMARCA4'],
        ['PGA3','PGC'],
        ['PGA5','PGC'],
        ['PGK1','SMR3B'],
        ['PGLYRP3','PGLYRP4'],
        ['PGM5','WASHC1'],
        ['PNLIP','PNLIPRP1'],
        ['PNLIP','PNLIPRP2'],
        ['PNLIP','SEC23B'],
        ['PNLIPRP1','PNLIPRP2'],
        ['POU2F1','WDR59'],
        ['PRH2','PRR4'],
        ['PRSS1','SERPINE1'],
        ['PRSS1','SMARCB1'],
        ['PTCSC2','TRG@'],
        ['PTCSC2','XPA'],
        ['PTPRF','PTPRS'],
        ['RAB11B','TG'],
        ['RAB1A','TG'],
        ['RAB27A','RSL24D1'],
        ['RDH16','SDR9C7'],
        ['REG1A','REG3A'],
        ['RHEB','SHLD2'],
        ['RHEX','ZNF888'],
        ['RXRA','TG'],
        ['SAFB','SAFB2'],
        ['SERPINA1','SERPINA11'],
        ['SERPINA1','SERPINA5'],
        ['SERPINB6','TKT'],
        ['SLC17A1','SLC17A3'],
        ['SLC25A23','SLC25A41'],
        ['SLC25A38','TG'],
        ['SLC25A4','SLC25A6'],
        ['SLC25A5','SLC25A6'],
        ['SMURF2','SMURF2P1-LRRC37BP1'],
        ['SNRPB','SNRPN'],
        ['SP100','SP140L'],
        ['SPEM2','SPEM3'],
        ['SPEM3','TMEM102'],
        ['SPRR1A','SPRR3'],
        ['SPRR1B','SPRR3'],
        ['ST3GAL3','TG'],
        ['STATH','TSC22D1'],
        ['SUGT1','SUGT1P4-STRA6LP'],
        ['TBCE','TG'],
        ['TIMP1','TIMP3'],
        ['TMSB10','TMSB4X'],
        ['TNNT1','TNNT2'],
        ['TRIM11','TRIM17'],
        ['TRIM22','TRIM6-TRIM34'],
        ['TUBB','TUBB4B'],
        ['TUBB','TUBB6'],
        ['ZKSCAN7','ZNF852'],
        ['ZNF160','ZNF347'],
        ['ZNF212','ZNF783'],
        ['ZNF226','ZNF234'],
        ['ZNF28','ZNF320'],
        ['ZNF433-AS1','ZNF625-ZNF20'],
        ['ZNF439','ZNF700'],
        ['ZNF525','ZNF83'],
        ['ZNF587','ZNF587B'],
        ['ZNF610','ZNF880'],
        ['ZNF620','ZNF621'],
        ['ZNF708','ZNF85'],
        ['ZSCAN5A','ZSCAN5B'],
        ['ABHD14B','PNLIP'],
        ['ABHD17A','PTMA'],
        ['ABL1','EXOSC2'],
        ['ABLIM1','HTN3'],
        ['ABLIM1','SMR3B'],
        ['AC000093.1','OAZ1'],
        ['AC003973.1','AC073539.1'],
        ['AC003973.1','LINC01859'],
        ['AC004066.2','PPA2'],
        ['AC006064.6','GTF2I'],
        ['AC007950.1','FBXL22'],
        ['AC008267.5','TPST1'],
        ['AC008554.1','ZNF826P'],
        ['AC008555.4','LINC01801'],
        ['AC009022.1','NPIPB15'],
        ['AC009070.1','AC092718.2'],
        ['AC009093.9','AC092338.1'],
        ['AC010329.1','LINC02635'],
        ['AC010618.1','FAM129C'],
        ['AC011462.1','PTMA'],
        ['AC012020.1','CD47'],
        ['AC015911.11','SLFN13'],
        ['AC017015.2','GBE1'],
        ['AC017091.1','LINC02619'],
        ['AC018630.2','CA6'],
        ['AC018630.2','CD44'],
        ['AC018630.2','CST1'],
        ['AC018630.2','CST4'],
        ['AC018630.2','DMBT1'],
        ['AC018630.2','EEF1A1'],
        ['AC018630.2','GAPDH'],
        ['AC018630.2','IGH@'],
        ['AC018630.2','IGHA1'],
        ['AC018630.2','IGK@'],
        ['AC018630.2','LPO'],
        ['AC018630.2','LYZ'],
        ['AC018630.2','MUC5B'],
        ['AC018630.2','MUC7'],
        ['AC018630.2','ODAM'],
        ['AC018630.2','PARM1'],
        ['AC018630.2','PIGR'],
        ['AC018630.2','RBM47'],
        ['AC018630.2','SFRP1'],
        ['AC018630.2','SMR3B'],
        ['AC018630.2','STATH'],
        ['AC018630.2','ZG16B'],
        ['AC020656.1','AP002956.1'],
        ['AC020656.1','MAP4K3-DT'],
        ['AC020656.1','TMEM123'],
        ['AC020915.5','ZSCAN22'],
        ['AC020928.1','HKR1'],
        ['AC020928.1','LINC00665'],
        ['AC020928.2','ZNF567'],
        ['AC021087.5','AC026412.1'],
        ['AC021087.5','AC091849.2'],
        ['AC021087.5','AC132008.2'],
        ['AC021087.5','DES'],
        ['AC021087.5','MUC20-OT1'],
        ['AC021851.1','NUP35'],
        ['AC022137.3','ZNF765'],
        ['AC022400.6','BMS1'],
        ['AC022400.6','PARG'],
        ['AC024560.3','AC024937.3'],
        ['AC024937.3','AC093642.1'],
        ['AC024940.2','AC092821.1'],
        ['AC025287.1','CHST6'],
        ['AC026740.3','BRD9'],
        ['AC067930.5','ANKRD19P'],
        ['AC067930.5','HBB'],
        ['AC069288.1','MAFK'],
        ['AC073263.1','TET3'],
        ['AC073585.2','CEL'],
        ['AC073585.2','DOCK1'],
        ['AC073585.2','GP2'],
        ['AC073585.2','PNLIP'],
        ['AC079790.2','FMNL2'],
        ['AC091057.1','AC091057.6'],
        ['AC091057.1','OTUD7A'],
        ['AC091951.3','AC138649.1'],
        ['AC091951.3','NIPA1'],
        ['AC092338.1','RRN3P2'],
        ['AC093484.3','UBA52'],
        ['AC093838.2','AL929601.1'],
        ['AC093838.2','AL929601.2'],
        ['AC108010.1','AC138035.1'],
        ['AC113189.9','FGF11'],
        ['AC113189.9','SPEM2'],
        ['AC114490.1','AC114490.2'],
        ['AC129492.1','VAMP2'],
        ['AC139256.2','PKD1'],
        ['AC139768.1','SMAGP'],
        ['AC209154.1','C17ORF51'],
        ['AC245033.1','PNLIP'],
        ['AC245041.2','LINC00842'],
        ['AC245060.7','IGLC2'],
        ['AC245060.7','IGLL5'],
        ['AC245060.7','IGLV1-40'],
        ['AC245060.7','IGLV1-41'],
        ['ACAD8','GLB1L3'],
        ['ACCS','EXT2'],
        ['ACE2','PIR'],
        ['ACSS1','APMAP'],
        ['ACTB','CLCN6'],
        ['ACTB','GGT1'],
        ['ACTG2','ZSCAN32'],
        ['ACTN4','EIF3K'],
        ['ADAP1','SUN1'],
        ['ADGRE2','ADGRE5'],
        ['ADGRG7','TFG'],
        ['ADK','KAT6B'],
        ['AFF1','PTPN13'],
        ['AFM','FCN1'],
        ['AFP','ALB'],
        ['AKAP17A','P2RY8'],
        ['AKT2','C19ORF47'],
        ['AL031282.2','DSC3'],
        ['AL109811.2','TARDBP'],
        ['AL117329.1','ARPC2'],
        ['AL121790.1','TTC6'],
        ['AL137186.2','SPAG9'],
        ['AL157955.1','LINC01147'],
        ['AL355312.2','AL355312.6'],
        ['AL355916.1','PRKCH'],
        ['AL360270.1','AL360270.3'],
        ['AL590556.3','CTRB2'],
        ['AL590556.3','LHFPL5'],
        ['AL591806.3','C21ORF62'],
        ['ALDOB','MYO1A'],
        ['ALG14','CNN3'],
        ['ALKBH5','LLGL1'],
        ['AMACR','SLC45A2'],
        ['ANAPC1','RMND5A'],
        ['ANK2','CAMK2D'],
        ['ANP32B','IGH@'],
        ['AP000527.1','LINC01297'],
        ['AP002956.1','KRT13'],
        ['AP3S1','COMMD10'],
        ['APBB3','SRA1'],
        ['APOE','RBP4'],
        ['ARF1','PTMA'],
        ['ARHGAP18','TMEM244'],
        ['ARHGEF17','P2RY6'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARL6IP1','RPS15A'],
        ['ART4','SMCO3'],
        ['ASL','CRCP'],
        ['ASXL1','NOL4L'],
        ['ATAD1','PTEN'],
        ['ATG16L1','INPP5D'],
        ['ATP2B4','TRB@'],
        ['ATP5ME','SLC49A3'],
        ['ATP5MG','KMT2A'],
        ['ATXN3','THAP11'],
        ['AZGP1','GJC3'],
        ['AZIN1','SMR3B'],
        ['B2M','PTMA'],
        ['B2M','S100A8'],
        ['BABAM2','RBKS'],
        ['BAG2','ZNF451'],
        ['BAG6','C6ORF47'],
        ['BANK1','IGH@'],
        ['BCAN','PTPRZ1'],
        ['BCAP29','DUS4L'],
        ['BCAS3','TG'],
        ['BCKDHA','TMEM91'],
        ['BCKDK','KAT8'],
        ['BGLAP','PMF1'],
        ['BHMT2','EBF1'],
        ['BLOC1S6','SQOR'],
        ['BMP10','CLCN6'],
        ['BMPR1B','PDLIM5'],
        ['BMS1','LINC00843'],
        ['BPIFA2','PRH2'],
        ['BPTF','LRRC37A2'],
        ['BRK1','VHL'],
        ['BSDC1','FAM229A'],
        ['BX322557.2','LINC00205'],
        ['C12ORF73','HSP90B1'],
        ['C19MC','RPLP1'],
        ['C19ORF24','CIRBP'],
        ['C19ORF33','SPINT2'],
        ['C1ORF194','UQCR10'],
        ['C6ORF106','SPDEF'],
        ['C9ORF3','MIR205HG'],
        ['CABP7','NF2'],
        ['CATSPER2','PPIP5K1'],
        ['CAVIN1','STAT3'],
        ['CBWD2','DOCK8'],
        ['CBX3','CCDC32'],
        ['CCM2L','HCK'],
        ['CD81','TSSC4'],
        ['CD83','JARID2'],
        ['CDC42','TRB@'],
        ['CDC42EP2','POLA2'],
        ['CDK11B','SLC35E2A'],
        ['CDK19','REV3L'],
        ['CDK2','RAB5B'],
        ['CDK2AP1','MPHOSPH9'],
        ['CDKN1A','RAB44'],
        ['CEL','CELP'],
        ['CEL','LHFPL5'],
        ['CEL','PNLIPRP2'],
        ['CELA2A','CTRL'],
        ['CELA2A','LHFPL5'],
        ['CELA2B','CTRB2'],
        ['CELA3A','LHFPL5'],
        ['CELA3B','LHFPL5'],
        ['CFAP300','YAP1'],
        ['CFL1','PTMA'],
        ['CHST11','IGH@'],
        ['CISD2','UBE2D3-AS1'],
        ['CIZ1','SMR3B'],
        ['CLCN6','GABARAP'],
        ['CLCN6','TNNI3'],
        ['CLPS','TPM3'],
        ['CLSTN1','CTNNBIP1'],
        ['CLTC','VMP1'],
        ['COL14A1','DEPTOR'],
        ['COLQ','HACL1'],
        ['COQ4','SWI5'],
        ['COQ8B','NUMBL'],
        ['CPA1','LHFPL5'],
        ['CPA1','RNASE1'],
        ['CPA2','LHFPL5'],
        ['CPB1','LHFPL5'],
        ['CPM','LYZ'],
        ['CPT1A','LYZ'],
        ['CRCP','TPST1'],
        ['CSNK1G2','OAZ1'],
        ['CST1','IGH@'],
        ['CTBS','GNG5'],
        ['CTDP1','PQLC1'],
        ['CTRB1','LHFPL5'],
        ['CTRB1','P4HB'],
        ['CTRC','LHFPL5'],
        ['CTSC','RAB38'],
        ['CXCR4','UBXN4'],
        ['CYTIP','ERMN'],
        ['DCUN1D1','MCCC1'],
        ['DDX1','NBAS'],
        ['DDX42','MAP3K3'],
        ['DEPDC1B','PDE4D'],
        ['DES','FHL1'],
        ['DHX35','FAM83D'],
        ['DHX9','NPL'],
        ['DIP2B','TMBIM6'],
        ['DLEU1','RNASEH2B'],
        ['DNM3','EEF1AKNMT'],
        ['DOCK9','STK24'],
        ['DPH7','PNPLA7'],
        ['DSP','TRB@'],
        ['DTNBP1','JARID2'],
        ['EED','HIKESHI'],
        ['EEF1A1','TG'],
        ['EEF1A1','TPM4'],
        ['EEF1AKMT3','TSFM'],
        ['EIF2S1','RBM25'],
        ['EIF4E3','FOXP1'],
        ['ELK4','SLC45A3'],
        ['ELOVL1','MED8'],
        ['ENTPD1','MCFD2'],
        ['ENTPD5','FAM161B'],
        ['ERBB2','GRB7'],
        ['ERCC5','S100A9'],
        ['ERGIC1','TRB@'],
        ['ETFRF1','KRAS'],
        ['EZH2','ZNF425'],
        ['FAM129C','SSH2'],
        ['FAM13A-AS1','HERC3'],
        ['FAM157B','FBXO25'],
        ['FAM85B','LINC02614'],
        ['FBXO25','TDRP'],
        ['FBXO44','MUC7'],
        ['FES','MAN2A2'],
        ['FGFR4','ZNF346'],
        ['FLT1','MEG3'],
        ['FOXP1','RYBP'],
        ['FTX','TG'],
        ['FUT8','STATH'],
        ['GAPDH','ZNF516'],
        ['GAS7','RCVRN'],
        ['GGT1','IGSF3'],
        ['GKAP1','KIF27'],
        ['GLE1','PWRN1'],
        ['GNAS','PTMA'],
        ['GOLT1A','KISS1'],
        ['GP2','TOLLIP'],
        ['GPBP1','STATH'],
        ['GPR35','PTMA'],
        ['GPT','PPP1R16A'],
        ['GRK4','HTT'],
        ['GTF2IRD2','LYZ'],
        ['GYPB','SAV1'],
        ['H3F3A','S100A9'],
        ['H6PD','SPSB1'],
        ['HAPLN3','MFGE8'],
        ['HBB','RASA3'],
        ['HBB','SGMS1'],
        ['HDAC4','PTMA'],
        ['HDLBP','PTMA'],
        ['HNRNPA2B1','LYZ'],
        ['HNRNPA2B1','PTMA'],
        ['HNRNPC','KIAA0586'],
        ['HORMAD1','SLC22A20P'],
        ['HPX','TRIM3'],
        ['HSD17B14','PLEKHA4'],
        ['HSP90AA1','IGH@'],
        ['HSP90AB1','SLC29A1'],
        ['HSPE1','MOB4'],
        ['HTN3','PIK3R1'],
        ['ID4','IGH@'],
        ['IDS','LINC00893'],
        ['IGF2','INS'],
        ['IGH@','LAPTM5'],
        ['IGH@','MUC1'],
        ['IGH@','NFKB2'],
        ['IGH@','PCSK7'],
        ['IGH@','S100A8'],
        ['IGH@','ZFP36L1'],
        ['IGHA1','MUC1'],
        ['IGSF10','MED12L'],
        ['IL13RA1','TG'],
        ['INSL3','JAK3'],
        ['IQGAP1','ZNF774'],
        ['ITM2B','RB1'],
        ['KANSL1','LRRC37A3'],
        ['KLF10','PTMA'],
        ['KLF2','OAZ1'],
        ['KLK1','PRSS2'],
        ['KLK4','KLKP1'],
        ['KRT13','S100A8'],
        ['LAMA3','RBBP8'],
        ['LATS1','NUP43'],
        ['LDAH','PUM2'],
        ['LHFPL5','PNLIP'],
        ['LHFPL5','TRB@'],
        ['LINC01422','Z97353.2'],
        ['LINC01513','ROPN1L'],
        ['LINC02624','LYZ'],
        ['LINC02649','PFKFB3'],
        ['LPGAT1','RPS6KC1'],
        ['LRRC37A3','NSF'],
        ['LRRC52-AS1','RXRG'],
        ['LYZ','MEI1'],
        ['LYZ','RPS14'],
        ['LYZ','SGTA'],
        ['MAML2','MTMR2'],
        ['MANBAL','SRC'],
        ['MAPKBP1','MGA'],
        ['MECOM','RPL22'],
        ['METTL14-DT','NPR1'],
        ['MFSD14A','SLC35A3'],
        ['MICAL2','TEAD1'],
        ['MIF','SLC2A11'],
        ['MINPP1','NUTM2A-AS1'],
        ['MLEC','UNC119B'],
        ['MLLT10','TG'],
        ['MPO','RNASE2'],
        ['MPRIP','NPPB'],
        ['MRC1','TMEM236'],
        ['MRPS31P5','SUGT1P4-STRA6LP'],
        ['MSMB','NCOA4'],
        ['MSRB3','TMBIM4'],
        ['MTA3','THADA'],
        ['MTG1','SCART1'],
        ['MTSS1','TATDN1'],
        ['MUC7','RPS21'],
        ['MUC7','ST3GAL1'],
        ['MUC7','TNS4'],
        ['MYADM','TG'],
        ['MYH10','TRB@'],
        ['NAIP','OCLN'],
        ['NCOA6','PIGU'],
        ['NCOR2','UBC'],
        ['NDRG1','ST3GAL1'],
        ['NFATC3','PLA2G15'],
        ['NIN','SAV1'],
        ['NPEPL1','STX16'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB5','SMG1'],
        ['NUP188','PHYHD1'],
        ['NWD1','SLC25A15'],
        ['OAZ1','PTMA'],
        ['PBXIP1','PMVK'],
        ['PCBD1','PRSS1'],
        ['PDCD11','PNLIP'],
        ['PIM3','SCO2'],
        ['PITPNC1','PRKCA'],
        ['PJVK','PRKRA-AS1'],
        ['PLA2G1B','PNLIP'],
        ['PLVAP','RAD23A'],
        ['PMS2','POLR2J3'],
        ['PNLIP','RPS29'],
        ['PRDX1','TESK2'],
        ['PRH2','PTPRF'],
        ['PRH2','SF3B1'],
        ['PRKAA1','TTC33'],
        ['PRKAB1','TMEM233'],
        ['PRKAG2','RHEB'],
        ['PRKCZ','SKI'],
        ['PRMT7','SLC7A6'],
        ['PRSS1','USP9Y'],
        ['PSMB8','TAP1'],
        ['PSPC1','ZMYM5'],
        ['PTBP3','SUSD1'],
        ['PTMA','SRSF7'],
        ['PTMA','TCF25'],
        ['PTMA','ZBTB2'],
        ['PWRN1','PWRN4'],
        ['PXK','RPP14'],
        ['RASSF8','SSPN'],
        ['RBMS2','SPRYD4'],
        ['RDH11','VTI1B'],
        ['RHOA','TKT'],
        ['RIPK1','SERPINB9'],
        ['RLN1','RLN2'],
        ['RNF125','RNF138'],
        ['RPL14','SRP14'],
        ['RPL28','TMEM190'],
        ['RPL38','TTYH2'],
        ['RWDD3','TMEM56'],
        ['SAMD5','SASH1'],
        ['SCARB1','UBC'],
        ['SCNN1A','TNFRSF1A'],
        ['SEPT9','ZG16B'],
        ['SERF1A','SMN1'],
        ['SIDT2','TAGLN'],
        ['SIGLEC6','Z99129.3'],
        ['SLC6A16','TEAD2'],
        ['SMARCB1','TRB@'],
        ['SMR3B','UQCRB'],
        ['SNTB2','VPS4A'],
        ['SPINK5','SPRR3'],
        ['STK10','UBE2D2'],
        ['STRA6LP','SUGT1'],
        ['SYT8','TNNI2'],
        ['TFDP2','XRN1'],
        ['TG','ZSCAN32'],
        ['TOGARAM1','TSTD3'],
        ['TPPP','ZDHHC11B'],
        ['TPTE2','TPTEP2-CSNK1E'],
        ['UBA2','WTIP'],
        ['USP10','ZDHHC7'],
        ['USP34','XPO1'],
        ['ZNF433-AS1','ZNF625'],
        ['ZNF578','ZNF611'],
        ['AC000093.1','AC005258.1'],
        ['AC002057.1','AC002057.2'],
        ['AC003044.1','AC004485.1'],
        ['AC004083.1','TMEM64'],
        ['AC004706.3','MED31'],
        ['AC004921.2','NPLOC4'],
        ['AC005071.1','ZCWPW1'],
        ['AC005532.1','C1GALT1'],
        ['AC005670.3','AC103810.2'],
        ['AC005670.3','ARL17A'],
        ['AC005670.3','ARL17B'],
        ['AC006008.1','ACTR3C'],
        ['AC006115.2','IGF2'],
        ['AC006518.7','CA6'],
        ['AC006538.2','SLC39A3'],
        ['AC007221.1','AC012178.1'],
        ['AC007563.4','LINC01921'],
        ['AC008114.1','PLBD1-AS1'],
        ['AC008119.1','ISCU'],
        ['AC008269.1','FASTKD2'],
        ['AC008467.1','AC116428.1'],
        ['AC008554.2','ZNF93'],
        ['AC008555.4','AC008555.5'],
        ['AC008632.1','AC016573.1'],
        ['AC009053.3','AC009060.1'],
        ['AC009070.1','AC092718.8'],
        ['AC009081.2','APLF'],
        ['AC009704.3','DIPK1C'],
        ['AC010086.3','RBMY2EP'],
        ['AC010168.1','PLBD1-AS1'],
        ['AC010327.2','ACTA1'],
        ['AC010327.2','CLCN6'],
        ['AC010327.2','DES'],
        ['AC010329.1','AC079804.3'],
        ['AC010329.1','AC127526.5'],
        ['AC012101.1','AC012101.2'],
        ['AC012636.1','SERINC5'],
        ['AC013265.1','AC062021.1'],
        ['AC015631.1','AC079380.1'],
        ['AC015689.1','RRM1'],
        ['AC015802.7','ST6GALNAC1'],
        ['AC015911.10','SLFN13'],
        ['AC016821.1','TSPAN15'],
        ['AC017104.6','NMUR1'],
        ['AC018630.2','BPIFA2'],
        ['AC018630.2','CST2'],
        ['AC018630.2','HTN3'],
        ['AC018647.3','HERPUD2'],
        ['AC020656.1','FOXP1'],
        ['AC020656.1','SOX9-AS1'],
        ['AC020656.1','TMBIM6'],
        ['AC020656.1','UBE2G2'],
        ['AC021087.5','AC073210.3'],
        ['AC021660.4','CPOX'],
        ['AC021755.3','SRP14-AS1'],
        ['AC023282.1','PLPP4'],
        ['AC023824.5','AC023824.6'],
        ['AC024028.1','AC073316.1'],
        ['AC024559.2','AC080129.1'],
        ['AC024560.4','AC026412.1'],
        ['AC025171.1','KCNK6'],
        ['AC027088.5','AL355377.4'],
        ['AC053545.1','BBS12'],
        ['AC055720.1','SSPN'],
        ['AC061975.8','NLK'],
        ['AC061975.8','PYY2'],
        ['AC067930.1','ANKRD19P'],
        ['AC067930.1','HBB'],
        ['AC068580.4','AC132217.2'],
        ['AC068700.2','PKIA-AS1'],
        ['AC068756.1','AC108690.1'],
        ['AC068774.1','DENND5B'],
        ['AC068790.8','KMT5A'],
        ['AC068989.1','DDX60'],
        ['AC068993.2','SYT1'],
        ['AC069503.4','HPD'],
        ['AC073210.3','SDHA'],
        ['AC073270.1','LINC01005'],
        ['AC073270.2','ZNF736'],
        ['AC073348.2','AL132800.1'],
        ['AC073349.5','AC092161.1'],
        ['AC073529.1','MID1'],
        ['AC079804.3','AC127526.5'],
        ['AC079804.3','LINC02635'],
        ['AC083862.3','TMEM140'],
        ['AC084734.1','CA2'],
        ['AC087457.1','ACTA1'],
        ['AC087457.1','ACTA2'],
        ['AC087457.1','MYH7'],
        ['AC087457.1','NPPA'],
        ['AC087473.1','PSMB2'],
        ['AC087477.2','ANXA4'],
        ['AC091978.1','LINC00680'],
        ['AC092745.5','AC145124.1'],
        ['AC092944.1','AC104411.1'],
        ['AC093787.2','AC233263.6'],
        ['AC093787.2','IGK@'],
        ['AC093838.1','AL929601.1'],
        ['AC093838.1','AL929601.2'],
        ['AC094105.2','ACTA2'],
        ['AC097374.1','CR392039.6'],
        ['AC097500.1','FSIP2-AS1'],
        ['AC098679.5','TMEM144'],
        ['AC098825.1','PARM1'],
        ['AC099343.3','AC099343.4'],
        ['AC099521.3','GLTP'],
        ['AC099654.1','AL136102.1'],
        ['AC104088.1','ITGA6'],
        ['AC104850.2','CX3CR1'],
        ['AC105052.5','CCDC146'],
        ['AC106791.3','KLHL3'],
        ['AC107071.1','ELF1'],
        ['AC108063.2','PROM1'],
        ['AC108519.1','AF228730.5'],
        ['AC110611.1','AC110611.2'],
        ['AC115220.3','LINC01005'],
        ['AC117834.1','AC117834.2'],
        ['AC119427.2','ACTN2'],
        ['AC119427.2','CLCN6'],
        ['AC119427.2','CRYAB'],
        ['AC119427.2','MYH7'],
        ['AC119427.2','NPPA'],
        ['AC119427.2','TNNT1'],
        ['AC124242.1','AC124242.3'],
        ['AC124944.2','MUC20-OT1'],
        ['AC126182.3','NDUFB4'],
        ['AC127526.2','AC127526.5'],
        ['AC127526.5','LINC02635'],
        ['AC132217.2','ACTB'],
        ['AC132217.2','ADAM12'],
        ['AC132217.2','ADAMTSL4-AS1'],
        ['AC132217.2','ANXA2'],
        ['AC132217.2','AP000781.2'],
        ['AC132217.2','ATP5F1A'],
        ['AC132217.2','BDH1'],
        ['AC132217.2','CALD1'],
        ['AC132217.2','CBX4'],
        ['AC132217.2','CCDC152'],
        ['AC132217.2','CD93'],
        ['AC132217.2','COL5A1'],
        ['AC132217.2','COLEC12'],
        ['AC132217.2','CSF2RB'],
        ['AC132217.2','CSH1'],
        ['AC132217.2','CYP19A1'],
        ['AC132217.2','DCN'],
        ['AC132217.2','EGFR'],
        ['AC132217.2','EHD3'],
        ['AC132217.2','F13A1'],
        ['AC132217.2','FGF14'],
        ['AC132217.2','FGFRL1'],
        ['AC132217.2','FLT1'],
        ['AC132217.2','FN1'],
        ['AC132217.2','HACD3'],
        ['AC132217.2','INS'],
        ['AC132217.2','KIAA0895L'],
        ['AC132217.2','LGMN'],
        ['AC132217.2','LHFPL6'],
        ['AC132217.2','MEG3'],
        ['AC132217.2','MPEG1'],
        ['AC132217.2','MRNIP'],
        ['AC132217.2','NAP1L1'],
        ['AC132217.2','PCBP2'],
        ['AC132217.2','PODN'],
        ['AC132217.2','SEPT9'],
        ['AC132217.2','SPARC'],
        ['AC132217.2','SYNPO'],
        ['AC132217.2','TIMP2'],
        ['AC132217.2','TIMP3'],
        ['AC132217.2','TP53I11'],
        ['AC132217.2','TRIOBP'],
        ['AC138409.2','LINC02241'],
        ['AC138776.1','ANKRD20A21P'],
        ['AC145124.1','LINC02018'],
        ['AC231759.2','AC239585.2'],
        ['AC233702.4','FAM182A'],
        ['AC239804.1','LINC01719'],
        ['AC244197.3','LINC00894'],
        ['AC244197.3','SMR3B'],
        ['ADGRG1','TG'],
        ['AF235103.3','ZNF250'],
        ['AGTPBP1','AL353743.2'],
        ['AKR1A1','CCND3'],
        ['AL031259.1','AL669831.3'],
        ['AL031674.1','LINC01524'],
        ['AL035681.1','CHADL'],
        ['AL050309.1','KLF8'],
        ['AL078587.2','AL450124.1'],
        ['AL080275.1','COL9A1'],
        ['AL080317.1','AL080317.4'],
        ['AL096677.2','CSTL1'],
        ['AL096803.2','ODR4'],
        ['AL096870.3','AL136295.7'],
        ['AL109809.5','AL117335.1'],
        ['AL132656.2','NUTM2B-AS1'],
        ['AL132800.1','AL136018.1'],
        ['AL133304.3','LINC00609'],
        ['AL133405.1','STMND1'],
        ['AL136140.1','CDC5L'],
        ['AL137026.3','CXCL12'],
        ['AL138830.1','AL138830.2'],
        ['AL139158.2','AL390839.2'],
        ['AL157400.3','PANK1'],
        ['AL158066.2','SUGT1P4-STRA6LP'],
        ['AL160154.1','LINC00375'],
        ['AL353660.1','LINC00402'],
        ['AL353693.1','AL591135.2'],
        ['AL354953.1','LAMC2'],
        ['AL355499.1','HULC'],
        ['AL356489.3','TRB@'],
        ['AL356489.4','PRSS3'],
        ['AL357075.4','RELA'],
        ['AL357127.2','LINC02641'],
        ['AL390726.4','CYP4F26P'],
        ['AL390838.1','RMI1'],
        ['AL390839.1','AL390839.2'],
        ['AL391840.3','SH3BGRL2'],
        ['AL392086.1','LINP1'],
        ['AL445524.2','SCAND2P'],
        ['AL449043.1','LINC01388'],
        ['AL591926.2','LINC01597'],
        ['AL645568.1','TNFSF4'],
        ['AL953883.1','FAM27C'],
        ['ALAS2','MPO'],
        ['ANKRD36C','LINC00342'],
        ['AP005242.5','LINC00349'],
        ['AP005901.3','AP005901.5'],
        ['AP006219.2','AP006219.3'],
        ['ARF4-AS1','DENND6A-AS1'],
        ['ARHGDIG','FAM234A'],
        ['ARHGEF12','MUC7'],
        ['ARL17B','BPTF'],
        ['AZU1','S100A8'],
        ['BCAR1','CPA1'],
        ['BX004807.1','EFCAB7'],
        ['BX469938.1','CPXM2'],
        ['C22ORF46','MEI1'],
        ['CCND3','SIGLEC6'],
        ['CD22','PLAT'],
        ['CHMP3','SMR3B'],
        ['CHRDL1','NUTM2A-AS1'],
        ['COL17A1','FLG2'],
        ['COLEC10','SAMD12-AS1'],
        ['CPA1','MPRIP'],
        ['CRYBA4','Z95115.2'],
        ['CTRB2','LHFPL5'],
        ['CTRC','PACSIN2'],
        ['CTRC','PNLIPRP2'],
        ['CTSB','LINC02245'],
        ['CTSS','KAT6A'],
        ['CYP3A5','LINC00662'],
        ['CYP4F26P','FAM95C'],
        ['DES','HTRA1'],
        ['DPY19L2','HULC'],
        ['DPY19L2','MAP4K3-DT'],
        ['EIF1B-AS1','ENTPD3-AS1'],
        ['FAM129C','KAT6A'],
        ['FAM174B','STATH'],
        ['FAM182A','FAM182B'],
        ['FAM245B','LINC02675'],
        ['FAM66C','FAM66D'],
        ['FAN1','HERC2'],
        ['FLG2','IGK@'],
        ['FOXP1','TMPO'],
        ['FP326651.1','LINC01667'],
        ['G0S2','MARCH7'],
        ['GINM1','STXBP5-AS1'],
        ['GIPC1','LINC01484'],
        ['HIF1A','S100A8'],
        ['HMGA1P4','WDR34'],
        ['KLF4','LINC01509'],
        ['KLK1','PRSS3'],
        ['KRT128P','KRT2'],
        ['LINC00271','PDE7B'],
        ['LINC00354','SOX1-OT'],
        ['LINC00494','LINC01522'],
        ['LINC00840','LINC00841'],
        ['LINC01498','SART3'],
        ['LINC01595','SULT2A1'],
        ['LINC01719','PDE4DIP'],
        ['LINC01962','TRIM7'],
        ['LINC02027','LINC02050'],
        ['LINC02087','LINC02088'],
        ['LINC02185','LINC02186'],
        ['LINC02558','SLC5A1'],
        ['LINCADL','LVRN'],
        ['LYZ','PSMA3-AS1'],
        ['METTL7A','PLCG1-AS1'],
        ['NDUFS8','PNLIP'],
        ['NUTM2A-AS1','SYT15'],
        ['PAM','TRB@'],
        ['PKIB','SIGLEC6'],
        ['PLA2G6','TMEM184B'],
        ['PLAT','SPTLC2'],
        ['PRB2','SUN2'],
        ['PTMA','TGFB1'],
        ['RALY-AS1','STAG3'],
        ['RAPGEF1','TRB@'],
        ['SFRP1','SMR3B'],
        ['TESC','ZG16B'],
        ['WAC-AS1','Z99572.1'],
        ['A2ML1','ABI2'],
        ['A2ML1','COA8'],
        ['A2ML1','GIGYF2'],
        ['A2ML1','ZNF665'],
        ['ABHD17B','CEMIP2'],
        ['ABHD2','FANCI'],
        ['ABL1','EXOSC2'],
        ['ABL1','SFTPA2'],
        ['ABO','NLGN4Y'],
        ['ABO','SOX2-OT'],
        ['AC004066.2','PPA2'],
        ['AC004687.2','PRH2'],
        ['AC004967.2','CCZ1B'],
        ['AC005176.1','LGALS13'],
        ['AC005229.5','AC006974.2'],
        ['AC006001.2','RABGEF1'],
        ['AC006062.1','AL079338.1'],
        ['AC007384.1','AP001360.2'],
        ['AC007448.3','AC007448.5'],
        ['AC007562.1','UBE2Q2'],
        ['AC007834.1','TCHP'],
        ['AC008132.1','AC121320.1'],
        ['AC008132.1','RNF223'],
        ['AC008250.1','AC010185.1'],
        ['AC008474.1','NUDT19'],
        ['AC008568.1','SPINK1'],
        ['AC008687.7','NTF4'],
        ['AC009022.1','AC009065.3'],
        ['AC009039.1','LINC02180'],
        ['AC009065.3','PDXDC2P'],
        ['AC009081.2','SPTLC1'],
        ['AC009264.1','HACD4'],
        ['AC009264.1','RAB10'],
        ['AC009271.1','LINC01905'],
        ['AC009812.4','ZBTB10'],
        ['AC010327.2','MYL7'],
        ['AC010359.3','TMSB4X'],
        ['AC010435.1','AMD1'],
        ['AC010538.1','UGDH'],
        ['AC010636.1','LINC02701'],
        ['AC011313.1','APPL2'],
        ['AC012313.1','CNGB3'],
        ['AC012485.1','CSAD'],
        ['AC012560.1','UST'],
        ['AC012645.3','PTMA'],
        ['AC015802.6','CTSS'],
        ['AC015977.2','CIB4'],
        ['AC018629.1','AC110769.2'],
        ['AC018630.2','AC020656.1'],
        ['AC018630.2','ACTB'],
        ['AC018630.2','AMY2B'],
        ['AC018630.2','CCL28'],
        ['AC018630.2','CD74'],
        ['AC018630.2','CX3CL1'],
        ['AC018630.2','DGCR2'],
        ['AC018630.2','DIAPH2'],
        ['AC018630.2','DSG2'],
        ['AC018630.2','DYNC1H1'],
        ['AC018630.2','EEF2'],
        ['AC018630.2','GCC2'],
        ['AC018630.2','GNAS'],
        ['AC018630.2','GNE'],
        ['AC018630.2','H1F0'],
        ['AC018630.2','IL6ST'],
        ['AC018630.2','KIAA1324'],
        ['AC018630.2','MYH9'],
        ['AC018630.2','NCOR2'],
        ['AC018630.2','NFIB'],
        ['AC018630.2','PER1'],
        ['AC018630.2','PHLDA1'],
        ['AC018630.2','PIK3R1'],
        ['AC018630.2','PRR27'],
        ['AC018630.2','PSMF1'],
        ['AC018630.2','RPL3'],
        ['AC018630.2','SIX4'],
        ['AC018630.2','SLC20A2'],
        ['AC018630.2','SLC6A8'],
        ['AC018630.2','SND1'],
        ['AC018630.2','SORL1'],
        ['AC018630.2','SQSTM1'],
        ['AC019077.1','TJP1'],
        ['AC020656.1','FGFR2'],
        ['AC020656.1','GOLGB1'],
        ['AC020656.1','HP1BP3'],
        ['AC020656.1','ZKSCAN1'],
        ['AC020916.1','PRKCSH'],
        ['AC020978.9','NEAT1'],
        ['AC021054.1','CPSF6'],
        ['AC022210.2','KCMF1'],
        ['AC022400.4','BMS1'],
        ['AC023906.3','TRIM69'],
        ['AC024940.2','AC092666.1'],
        ['AC024940.2','SSUH2'],
        ['AC025575.2','TSPAN8'],
        ['AC027288.3','PAWR'],
        ['AC027290.2','LYZ'],
        ['AC027544.2','RPL7L1'],
        ['AC040162.1','CTRB1'],
        ['AC048338.2','TG'],
        ['AC063960.1','ACTA2'],
        ['AC068228.3','TBC1D31'],
        ['AC068279.2','IGK@'],
        ['AC068580.4','SFTPA2'],
        ['AC068672.2','TRMT12'],
        ['AC068987.2','FIGNL2-DT'],
        ['AC069280.2','UST'],
        ['AC069281.2','CCNG2'],
        ['AC072062.1','GOLGB1'],
        ['AC073264.3','TCAF1'],
        ['AC073365.1','BGN'],
        ['AC079142.1','HMGXB4'],
        ['AC079328.2','STARD5'],
        ['AC079414.3','IFI44L'],
        ['AC087280.2','MRPL17'],
        ['AC087283.1','PNO1'],
        ['AC087672.2','STAU2'],
        ['AC090001.1','RNASE10'],
        ['AC090515.4','SLTM'],
        ['AC092435.1','AC092435.2'],
        ['AC092437.1','STX18-AS1'],
        ['AC092691.3','TRIOBP'],
        ['AC092718.8','TWSG1'],
        ['AC092809.4','DEGS1'],
        ['AC092933.2','SLC37A2'],
        ['AC092958.2','LINC02032'],
        ['AC092958.3','LINC02032'],
        ['AC092969.1','IQUB'],
        ['AC092979.2','GYG1'],
        ['AC093010.2','TMPRSS11B'],
        ['AC093157.1','SLC30A7'],
        ['AC093890.1','GYPE'],
        ['AC095038.5','ZNF736'],
        ['AC098483.1','CR589904.2'],
        ['AC098483.1','OR2T2'],
        ['AC098588.3','GYPB'],
        ['AC098588.3','GYPE'],
        ['AC098826.2','ANKRD30A'],
        ['AC098828.2','LAPTM4A'],
        ['AC103957.2','SSBP2'],
        ['AC105094.2','RNF152'],
        ['AC105916.1','LINC02021'],
        ['AC106795.2','AC139493.2'],
        ['AC107021.1','RAB10'],
        ['AC107032.3','HACD4'],
        ['AC107208.1','GUCY1A1'],
        ['AC108488.2','ADI1'],
        ['AC109361.1','TBCK'],
        ['AC114964.2','FBXL7'],
        ['AC114971.1','TECRL'],
        ['AC115220.3','MAEL'],
        ['AC117382.2','GM2A'],
        ['AC118658.1','ANP32A'],
        ['AC119427.2','ANP32B'],
        ['AC119674.1','TMEM127'],
        ['AC127502.1','AC135983.3'],
        ['AC127526.5','ERVE-1'],
        ['AC131160.1','AIG1'],
        ['AC132217.2','AMOTL1'],
        ['AC132217.2','CADM3-AS1'],
        ['AC132217.2','COL1A2'],
        ['AC132217.2','LRP1'],
        ['AC132217.2','VWF'],
        ['AC134504.1','PRSS44P'],
        ['AC134980.2','OR4N2'],
        ['AC138207.8','LRRC37A3'],
        ['AC138409.2','GTF2H2'],
        ['AC138409.2','GTF2H2B'],
        ['AC138409.2','GTF2H2C'],
        ['AC138819.1','AC139493.2'],
        ['AC140134.1','OCLN'],
        ['AC141002.1','LINC02585'],
        ['AC145138.1','OCLN'],
        ['AC234783.1','TMSB15B'],
        ['AC243562.1','AKAP13'],
        ['AC244205.2','IGK@'],
        ['AC245047.1','SSX3'],
        ['AC245060.7','IGLC3'],
        ['AC245452.5','IGLVIV-66-1'],
        ['ACAD8','GLB1L3'],
        ['ACADVL','RACK1'],
        ['ACAP3','ACTB'],
        ['ACCS','EXT2'],
        ['ACE2','PIR'],
        ['ACIN1','PRH1'],
        ['ACIN1','PRH2'],
        ['ACSL1','CT69'],
        ['ACSL3','CPM'],
        ['ACSS1','APMAP'],
        ['ACSS2','TMEM154'],
        ['ACTA2','AL356534.1'],
        ['ACTB','AL022311.1'],
        ['ACTB','B3GNT7'],
        ['ACTB','CLPB'],
        ['ACTB','LINC02210'],
        ['ACTB','MTR'],
        ['ACTN4','EIF3K'],
        ['ACTN4','IGH@'],
        ['ACVR1B','ZG16B'],
        ['ADA2','KCTD13'],
        ['ADAMDEC1','MYH9'],
        ['ADAP1','SUN1'],
        ['ADAR','CCDC80'],
        ['ADCY3','AL160408.3'],
        ['ADGRE1','VAV1'],
        ['ADGRE2','ADGRE5'],
        ['ADGRF1','RNF115'],
        ['ADGRF5','LERFS'],
        ['ADGRG7','TFG'],
        ['ADIPOQ','MRPL10'],
        ['ADIRF','SNCG'],
        ['ADK','KAT6B'],
        ['ADPGK','AL160408.3'],
        ['ADRM1','GSE1'],
        ['AFF1','PTPN13'],
        ['AFF3','ITGB1'],
        ['AFP','ALB'],
        ['AGAP11','AL512662.2'],
        ['AGAP5','BMS1P4'],
        ['AGFG1','CFLAR'],
        ['AGO2','MRE11'],
        ['AGPAT2','ALB'],
        ['AK4','NF1'],
        ['AKAP1','TRB@'],
        ['AKAP11','ETS2'],
        ['AKAP13','FGD4'],
        ['AKR7L','AL035413.2'],
        ['AKT2','C19ORF47'],
        ['AL023802.1','MEG3'],
        ['AL024497.1','AL024497.2'],
        ['AL031056.1','MAP3K20'],
        ['AL031289.1','NFYC-AS1'],
        ['AL033527.3','AL033527.4'],
        ['AL035412.1','DEPDC1-AS1'],
        ['AL049828.1','KCNJ15'],
        ['AL049872.1','CXCR4'],
        ['AL110114.1','YWHAB'],
        ['AL121790.1','TTC6'],
        ['AL121985.1','SET'],
        ['AL132712.2','LINC00637'],
        ['AL133353.2','TBC1D10B'],
        ['AL133367.1','CKB'],
        ['AL133481.1','PPIF'],
        ['AL137072.1','MRPS16'],
        ['AL137803.1','SLC44A5'],
        ['AL138885.2','LINC00518'],
        ['AL138963.4','APOH'],
        ['AL138963.4','HP'],
        ['AL138963.4','PNLIP'],
        ['AL138963.4','PRSS1'],
        ['AL138963.4','SMR3B'],
        ['AL138963.4','TRB@'],
        ['AL138999.2','C9ORF78'],
        ['AL157762.1','KLHDC10'],
        ['AL158154.3','ATG14'],
        ['AL158154.3','MRE11'],
        ['AL158154.3','TTPA'],
        ['AL158154.3','UPP1'],
        ['AL353572.2','CDK20'],
        ['AL353572.2','SMR3B'],
        ['AL354892.3','MOB1A'],
        ['AL354892.3','MORF4L1'],
        ['AL356275.1','LINC02767'],
        ['AL356289.2','FUT4'],
        ['AL359237.1','LINC02330'],
        ['AL390763.1','ANKRD7'],
        ['AL391839.3','PAQR3'],
        ['AL450992.1','TRAF3IP2'],
        ['AL451136.1','CCDC163'],
        ['AL591848.2','AL591848.3'],
        ['AL592310.1','FAM133B'],
        ['ALB','CYP4A22'],
        ['ALB','CYP4F2'],
        ['ALB','EIF3A'],
        ['ALB','FMO3'],
        ['ALB','GAPDH'],
        ['ALDH6A1','C3ORF70'],
        ['ALDH9A1','USF3'],
        ['ALDOB','APOC3'],
        ['ALDOB','ENO1'],
        ['ALG14','CNN3'],
        ['ALK','ETS1'],
        ['ALK','TMED2'],
        ['ALKBH5','HBB'],
        ['ALKBH5','LLGL1'],
        ['AMACR','SLC45A2'],
        ['AMBRA1','HARBI1'],
        ['AMD1','QRSL1'],
        ['AMPD3','ETS1'],
        ['AMZ2','BPTF'],
        ['ANAPC1','RMND5A'],
        ['ANK2','CAMK2D'],
        ['ANKLE2','POLE'],
        ['ANKRD11','HNRNPK'],
        ['ANKRD28','CAST'],
        ['ANKRD52','MUC7'],
        ['ANKS1B','APOL1'],
        ['ANP32B','IGH@'],
        ['ANTXR2','MORC2-AS1'],
        ['ANXA1','MGLL'],
        ['ANXA2','S100A9'],
        ['AOPEP','CPSF6'],
        ['AOPEP','MIR205HG'],
        ['AP000356.2','POM121L4P'],
        ['AP000527.1','FP475955.3'],
        ['AP000662.1','CTSS'],
        ['AP000943.2','MRE11'],
        ['AP001011.1','SGCB'],
        ['AP002748.2','CTSF'],
        ['AP002957.1','FNDC3B'],
        ['AP002990.1','HBB'],
        ['AP003066.1','LINC02737'],
        ['AP003497.1','HMGB3'],
        ['AP005121.1','LINC01906'],
        ['AP1M1','KLHL36'],
        ['AP3S1','COMMD10'],
        ['AP4B1','HIPK1-AS1'],
        ['AP5B1','GLTP'],
        ['APBB2','BRD4'],
        ['APBB3','SRA1'],
        ['APMAP','ID2'],
        ['APOA1','P4HB'],
        ['APOA4','IGH@'],
        ['APOA4','IL9R'],
        ['APOC3','FTL'],
        ['APOE','SCAP'],
        ['APOL1','SFTPA2'],
        ['APP','PTMS'],
        ['ARF1','H3F3A'],
        ['ARF1','HMGB1'],
        ['ARF1','IVNS1ABP'],
        ['ARHGAP18','TMEM244'],
        ['ARHGDIA','MUC7'],
        ['ARHGEF17','P2RY6'],
        ['ARHGEF7','TEX29'],
        ['ARID1A','WSB1'],
        ['ARID3A','LINC01578'],
        ['ARL14EP','MAOB'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARL6IP1','RPS15A'],
        ['ARL8B','TASOR'],
        ['ARMH3','LARP4'],
        ['ARPC2','NABP2'],
        ['ARRB2','WNT4'],
        ['ARRDC3','TNRC6B'],
        ['ART4','SMCO3'],
        ['ASH1L','GABRG2'],
        ['ASL','CRCP'],
        ['ASXL1','NOL4L'],
        ['ATAD1','PTEN'],
        ['ATF5','ORM1'],
        ['ATF7IP','IL3RA'],
        ['ATF7IP','ZNF432'],
        ['ATG14','JPT1'],
        ['ATG16L1','INPP5D'],
        ['ATL2','CXCR4'],
        ['ATP1B1','DTWD1'],
        ['ATP1B1','FNTA'],
        ['ATP2B2','PRB3'],
        ['ATP2C2-AS1','MEAK7'],
        ['ATP5ME','SLC49A3'],
        ['ATP5MG','KMT2A'],
        ['ATP6V0C','ZEB2'],
        ['ATP6V1A','MTREX'],
        ['ATXN1','EEF1A1'],
        ['ATXN1','FABP3'],
        ['ATXN3','THAP11'],
        ['AUTS2','PRH2'],
        ['AXDND1','EPHB1'],
        ['AXDND1','UPRT'],
        ['AZGP1','GJC3'],
        ['AZU1','PTBP1'],
        ['B2M','CSNK1E'],
        ['B2M','HMGB2'],
        ['B2M','IGK@'],
        ['B2M','RASA3'],
        ['B2M','RBBP6'],
        ['B2M','RNF166'],
        ['B2M','SEPTIN9'],
        ['B2M','UBB'],
        ['B3GNTL1','METRNL'],
        ['BABAM2','RBKS'],
        ['BACH2','EEF1A1'],
        ['BACH2','LFNG'],
        ['BAG2','ZNF451'],
        ['BAG6','C6ORF47'],
        ['BAIAP2-DT','BPTF'],
        ['BANK1','IGH@'],
        ['BANK1','LEPROT'],
        ['BASP1','LINC01435'],
        ['BCAP29','DUS4L'],
        ['BCAS3','TG'],
        ['BCKDHA','TMEM91'],
        ['BGLAP','PMF1'],
        ['BHLHE40','WDR3'],
        ['BID','SLC16A3'],
        ['BIN2','CACNB2'],
        ['BIRC6','TRB@'],
        ['BLOC1S6','SQOR'],
        ['BMP10','TNNC1'],
        ['BMP7','ID4'],
        ['BMPR1B','PDLIM5'],
        ['BMS1','BMS1P4'],
        ['BMS1','BMS1P4-AGAP5'],
        ['BMS1P4','TIMM23B-AGAP6'],
        ['BNIPL','IGH@'],
        ['BPIFB1','DDX17'],
        ['BPTF','LRRC37A2'],
        ['BRD4','GNAO1'],
        ['BRK1','VHL'],
        ['BSDC1','FAM229A'],
        ['BSDC1','HDAC1'],
        ['BSG','CDC34'],
        ['BTN2A1','BTN2A3P'],
        ['BTN2A3P','BTN3A2'],
        ['BTNL9','POLR1A'],
        ['BVES','EMC2'],
        ['BZW1','SNX1'],
        ['C11ORF54','SP3'],
        ['C12ORF73','HSP90B1'],
        ['C19MC','CD44'],
        ['C19MC','MDM4'],
        ['C19MC','RPLP1'],
        ['C19MC','RPS19'],
        ['C19ORF33','SPINT2'],
        ['C1GALT1C1','CSNK1D'],
        ['C1GALT1C1','NDST4'],
        ['C1ORF194','UQCR10'],
        ['C1ORF226','NOS1AP'],
        ['C1ORF43','KRT1'],
        ['C1R','IGH@'],
        ['C1R','SERPINA1'],
        ['C20ORF197','MIR646HG'],
        ['C3','PTMS'],
        ['C4ORF3','MYH11'],
        ['C5AR1','NDUFA7'],
        ['C6ORF141','CLEC4M'],
        ['C9ORF72','MOB3B'],
        ['CABP7','NF2'],
        ['CACNA1A','LRRFIP1'],
        ['CACNA2D3','RPS15'],
        ['CACNB1','MRC2'],
        ['CALM1','EEF1A1'],
        ['CALM1','SPG7'],
        ['CALR','ZNF554'],
        ['CAMTA1','VAMP3'],
        ['CANX','PNLIP'],
        ['CAPZA2','SLC38A2'],
        ['CARD8','TOR1AIP2'],
        ['CASC15','CTRB1'],
        ['CASC3','TG'],
        ['CASTOR2','GTF2I'],
        ['CATSPER2','PPIP5K1'],
        ['CAV2','FGFR2'],
        ['CAV2','LINC01923'],
        ['CAV2','NDUFV1'],
        ['CAVIN1','KRT6A'],
        ['CAVIN1','STAT3'],
        ['CBWD2','DOCK8'],
        ['CBX3','CCDC32'],
        ['CBX4','SMR3B'],
        ['CCBE1','GP2'],
        ['CCDC144NL-AS1','SPECC1'],
        ['CCDC15-DT','HEPACAM'],
        ['CCDC27','SMIM1'],
        ['CCDC32','CEACAM8'],
        ['CCDC62','SLC66A2'],
        ['CCDC80','IGFBP5'],
        ['CCDC82','NCKAP1L'],
        ['CCM2L','HCK'],
        ['CCNB1IP1','RPPH1'],
        ['CCND3','PIGR'],
        ['CCND3','UGDH'],
        ['CCND3','ZNF273'],
        ['CCNL1','TOPBP1'],
        ['CD24','ESPL1'],
        ['CD24','FOXP1'],
        ['CD36','CDK6'],
        ['CD44','SAR1B'],
        ['CD44','ZEB2'],
        ['CD5L','EOGT'],
        ['CD63','TAGLN'],
        ['CD74','LIPF'],
        ['CD81','TSSC4'],
        ['CD83','JARID2'],
        ['CDC123','SEC61A2'],
        ['CDC42EP2','POLA2'],
        ['CDHR5','GAPDH'],
        ['CDK11B','SLC35E2A'],
        ['CDK19','REV3L'],
        ['CDK2','RAB5B'],
        ['CDK6','PRELP'],
        ['CDK6','RPL37'],
        ['CDKN1A','RAB44'],
        ['CDR2L','HID1-AS1'],
        ['CEACAM5','OPA3'],
        ['CEL','CELP'],
        ['CELA3B','CPA1'],
        ['CELF4','SOX4'],
        ['CELP','CTRC'],
        ['CELP','PLA2G1B'],
        ['CELP','PRSS1'],
        ['CELP','PRSS3'],
        ['CELSR1','SMR3B'],
        ['CEMIP2','NMRK1'],
        ['CENPC','LEPROT'],
        ['CEP104','GRK4'],
        ['CFAP300','YAP1'],
        ['CFAP58','ITPRIP-AS1'],
        ['CFHR2','CFHR5'],
        ['CFLAR','ERCC4'],
        ['CFLAR','PTMA'],
        ['CHIC2','ROCK2'],
        ['CHMP1A','CPSF6'],
        ['CHST11','NIBAN3'],
        ['CIRBP','FAM174C'],
        ['CLCN6','CLEC3B'],
        ['CLCN6','CLIC4'],
        ['CLCN6','TNNC1'],
        ['CLEC2D','IGH@'],
        ['CLIP1','IGH@'],
        ['CLIP1','ZCCHC8'],
        ['CLSTN1','CTNNBIP1'],
        ['CLTB','KRT4'],
        ['CLTC','VMP1'],
        ['CLU','CTRB2'],
        ['CLU','ORM1'],
        ['CNGA3','VWA3B'],
        ['CNKSR3','IGSF6'],
        ['CNN2','SC5D'],
        ['CNN3','PFDN5'],
        ['CNN3','PPP1R12B'],
        ['CNOT2','LINC02821'],
        ['COL14A1','DEPTOR'],
        ['COL1A1','MCOLN3'],
        ['COL1A2','KRT10'],
        ['COL1A2','NKX2-2'],
        ['COL3A1','RPL35A'],
        ['COL6A1','CSH1'],
        ['COL7A1','KRT14'],
        ['COLQ','HACL1'],
        ['COPA','GOLGA8R'],
        ['COPE','GDF1'],
        ['COPS3','GTF3C2-AS1'],
        ['COQ8B','NUMBL'],
        ['CPM','LYZ'],
        ['CPM','USP9Y'],
        ['CPSF6','EEF1A1'],
        ['CRCP','TPST1'],
        ['CRHR1-IT1_','YIPF5'],
        ['CRTAP','SP4'],
        ['CRTC1','KLHL26'],
        ['CRYAB','GFAP'],
        ['CRYL1','TG'],
        ['CSDC2','HELLPAR'],
        ['CSF2RA','IL3RA'],
        ['CSF2RA','TMEM220'],
        ['CSGALNACT1','CUTALP'],
        ['CSH1','MEG3'],
        ['CSH1','NEAT1'],
        ['CSH2','H19'],
        ['CSH2','IGF2'],
        ['CSNK1G2','OAZ1'],
        ['CSNK2A2','TSHZ2'],
        ['CSPP1','MCMDC2'],
        ['CST1','SMR3A'],
        ['CST4','IGLC2'],
        ['CSTP1','MUC7'],
        ['CTBS','GNG5'],
        ['CTDP1','SLC66A2'],
        ['CTNND1','LYZ'],
        ['CTRC','RPL30'],
        ['CTSB','GCSH'],
        ['CTSB','TRA@'],
        ['CTSC','RAB38'],
        ['CTSS','OR1M1'],
        ['CTSS','SYT14'],
        ['CTSS','WWOX'],
        ['CU634019.5','CU638689.4'],
        ['CUBN','TMEM168'],
        ['CUL1','EZH2'],
        ['CWF19L1','WWOX'],
        ['CXCR4','EIF4G2'],
        ['CXCR4','HNRNPH1'],
        ['CXCR4','SP3'],
        ['CXCR4','UBXN4'],
        ['CXCR4','ZEB2'],
        ['CXORF56','TECRL'],
        ['CYP1A2','IL3RA'],
        ['CYP1A2','KAT6A'],
        ['CYP4F11','DNM1P41'],
        ['CYTH1','DIAPH1'],
        ['CYTH1','UBB'],
        ['CYTIP','ERMN'],
        ['DAPP1','PTP4A2'],
        ['DCAF12','HBA2'],
        ['DCBLD2','PCBP2'],
        ['DCN','TIMP3'],
        ['DCUN1D1','MCCC1'],
        ['DDR1','SFTPA2'],
        ['DDX1','NBAS'],
        ['DDX17','NQO2'],
        ['DDX17','RAB18'],
        ['DDX17','SIAH3'],
        ['DDX25','HYLS1'],
        ['DDX42','MAP3K3'],
        ['DDX5','HBD'],
        ['DDX5','HNRNPH1'],
        ['DDX5','IGH@'],
        ['DDX58','TECRL'],
        ['DDX6','WAC-AS1'],
        ['DEFA6','IGH@'],
        ['DEFA6','REG3A'],
        ['DEPDC1B','PDE4D'],
        ['DES','FABP3'],
        ['DES','FTH1'],
        ['DES','NAP1L4'],
        ['DES','SRRM2'],
        ['DGKD','LYZ'],
        ['DGKH','GRB2'],
        ['DHCR24','TG'],
        ['DHFR2','TTPA'],
        ['DHX35','FAM83D'],
        ['DHX9','NPL'],
        ['DIAPH1','HDAC3'],
        ['DIP2B','TMBIM6'],
        ['DLEU2','KLF5'],
        ['DMKN','KRT10'],
        ['DMKN','SBSN'],
        ['DNAJB14','TRA@'],
        ['DNAJB2','PNLIP'],
        ['DNASE1','UGDH'],
        ['DNM1L','FBLIM1'],
        ['DNM3','EEF1AKNMT'],
        ['DNTTIP2','GNL2'],
        ['DOCK3','MANF'],
        ['DOCK9','STK24'],
        ['DPH7','PNPLA7'],
        ['DPY19L2','MRPS23'],
        ['DSG3','ZBTB11'],
        ['DUSP6','POC1B'],
        ['DYRK4','RAD51AP1'],
        ['E2F4','RPL14P1'],
        ['EDA','SLC35E1'],
        ['EED','HIKESHI'],
        ['EEF1A1','MYL2'],
        ['EEF1A1','RGS5'],
        ['EEF1A1','RPL4'],
        ['EEF1A1','SREK1'],
        ['EEF1A1','TG'],
        ['EEF1AKMT3','TSFM'],
        ['EFCAB10','PUS7'],
        ['EGLN1','NUDT4'],
        ['EHD1','KLF6'],
        ['EIF2S1','RBM25'],
        ['EIF3D','PTMA'],
        ['EIF3D','TNNT2'],
        ['EIF3G','SMR3B'],
        ['EIF4E3','FOXP1'],
        ['ELK4','SLC45A3'],
        ['ELOVL1','MED8'],
        ['ELOVL2-AS1','SMIM13'],
        ['EMB','TCF3'],
        ['EMP2','SCIMP'],
        ['EMP2','VIRMA'],
        ['EMX2','PRKCB'],
        ['ENPP1','SLC12A8'],
        ['ENTPD1','ZNF521'],
        ['ENTPD4','IL3RA'],
        ['ENTPD5','FAM161B'],
        ['EOGT','WWOX'],
        ['EPB41','SLC5A5'],
        ['EPHB2','GNAZ'],
        ['EPHB3','HBG1'],
        ['ERBB2','GRB7'],
        ['ERBB3','GLTP'],
        ['ERICH1','MSI2'],
        ['ESAM','TG'],
        ['ESPL1','UGDH'],
        ['ESRP2','LYZ'],
        ['ETFRF1','KRAS'],
        ['ETS1','NPAP1'],
        ['ETS2','MMP2'],
        ['ETV6','FOSB'],
        ['ETV6','PIPOX'],
        ['EXTL3-AS1','SOD2'],
        ['EZH2','ZNF425'],
        ['FABP6','REG3A'],
        ['FAM13A-AS1','HERC3'],
        ['FAM155A','PCDH12'],
        ['FAM157B','FBXO25'],
        ['FAM167A','ZDHHC20'],
        ['FAM71B','MED7'],
        ['FAM86C1','FAM86C2P'],
        ['FARP2','SEPTIN2'],
        ['FBRSL1','NOC4L'],
        ['FBXL16','GFAP'],
        ['FBXO25','TDRP'],
        ['FBXO30','SHPRH'],
        ['FBXO43','RGS22'],
        ['FCAR','TCF3'],
        ['FCGBP','PNKP'],
        ['FDX1','LINC02732'],
        ['FEM1C','HNRNPK'],
        ['FER1L4','IGH@'],
        ['FES','MAN2A2'],
        ['FGFR4','ZNF346'],
        ['FGG','IGH@'],
        ['FKBP5','TARBP1'],
        ['FMNL2','STATH'],
        ['FN1','SLC6A8'],
        ['FNBP4','SMR3B'],
        ['FNDC3A','TPM4'],
        ['FOXL2NB','MYLK3'],
        ['FOXO1','HMGB1'],
        ['FOXP1','GBP6'],
        ['FOXP1','RYBP'],
        ['FOXRED2','TXN2'],
        ['FRMD1','SERPINF1'],
        ['FRMD6','HERC2'],
        ['FSTL1','NCKAP1'],
        ['FTCD','SPATC1L'],
        ['FTO','ZNF587B'],
        ['FTX','NIBAN1'],
        ['FUS','TCF25'],
        ['FUT11','JMJD6'],
        ['FUT4','PPFIBP1'],
        ['FYTTD1','METRNL'],
        ['FZD4-DT','TMEM135'],
        ['GAA','MUC5AC'],
        ['GAB2','LINC02728'],
        ['GABPB2','PIGR'],
        ['GALNT6','MLKL'],
        ['GALNTL5','MALT1'],
        ['GAS7','RCVRN'],
        ['GBA','MED4'],
        ['GBP2','PTPRC'],
        ['GBP6','SLC27A1'],
        ['GEMIN2','SEC23A-AS1'],
        ['GFAP','NCDN'],
        ['GGT1','IGSF3'],
        ['GGT1','PIGS'],
        ['GHR','GSEC'],
        ['GIPC1','IGHA1'],
        ['GKAP1','KIF27'],
        ['GKN2','IGH@'],
        ['GLTP','GTF2I'],
        ['GLTP','PAFAH1B2'],
        ['GLTP','SORBS1'],
        ['GLUL','TAGLN2'],
        ['GM2A','KAT6A'],
        ['GM2A','LINC00910'],
        ['GNAS','IGFBP7'],
        ['GNB1','NADK'],
        ['GNE','ITGA10'],
        ['GOLGB1','MAGI2'],
        ['GOLT1A','KISS1'],
        ['GP2','REG1A'],
        ['GPR137B','LINC00513'],
        ['GPT','PPP1R16A'],
        ['GRB2','IGF2BP1'],
        ['GRB2','MPO'],
        ['GRHL1','LRRFIP1'],
        ['GRIPAP1','TRA@'],
        ['GRK4','HTT'],
        ['GRN','IGH@'],
        ['GSR','GTF2E2'],
        ['GTF2H2','NAIP'],
        ['GTF2H2B','NAIP'],
        ['GTF2H2C','NAIP'],
        ['GTF2H5','SOX2-OT'],
        ['GTF2I','RNF10'],
        ['GVQW3','PCYOX1'],
        ['GYPB','SAV1'],
        ['H19','QSOX1'],
        ['H2AFY','SFTPA2'],
        ['H3F3A','WWOX'],
        ['H6PD','SPSB1'],
        ['HABP4','LINC01545'],
        ['HAPLN3','MFGE8'],
        ['HAX1','UBAP2L'],
        ['HBA1','MCL1'],
        ['HBA1','SLC4A1'],
        ['HBA1','SNX22'],
        ['HBB','ILF3'],
        ['HBB','SNRPB2'],
        ['HCG18','NPM1'],
        ['HCLS1','TMEM9B'],
        ['HECTD3','ZSWIM5'],
        ['HERC2','PWRN1'],
        ['HIPK1','HMGB1'],
        ['HIPK2','PRB3'],
        ['HMGA1','SRGN'],
        ['HMGB1','HMGN2'],
        ['HMGB1','HSP90AB1'],
        ['HMGB1','IGH@'],
        ['HMGB1','N4BP2L1'],
        ['HMGB1','TMEM18'],
        ['HMGB1','YPEL5'],
        ['HMGXB4','LINC02321'],
        ['HOOK2','PNLIP'],
        ['HORMAD1','SLC22A20P'],
        ['HPR','SERPINA1'],
        ['HPX','TRIM3'],
        ['HRG','ORM2'],
        ['HSD17B14','PLEKHA4'],
        ['HSD17B2','LINC01146'],
        ['HSD17B2','LINC01920'],
        ['HSDL2','KIAA1958'],
        ['HSP90AA1','IGH@'],
        ['HSP90AB1','SLC29A1'],
        ['HSP90AB3P','SLC29A1'],
        ['HSPE1','MOB4'],
        ['HTN1','PRH1'],
        ['ID4','IGH@'],
        ['IDS','LINC00893'],
        ['IDS','ZMAT4'],
        ['IFI44L','WWOX'],
        ['IFRD1','UBE2H'],
        ['IGFBP5','PITHD1'],
        ['IGFBP7','NOA1'],
        ['IGH@','LAPTM5'],
        ['IGH@','MAP3K20'],
        ['IGH@','MUC1'],
        ['IGH@','MUC12'],
        ['IGH@','MYH9'],
        ['IGH@','NFKB2'],
        ['IGH@','PCSK7'],
        ['IGH@','PGA3'],
        ['IGH@','PGA5'],
        ['IGH@','PPP1R12B'],
        ['IGH@','PSAP'],
        ['IGH@','RPS19'],
        ['IGH@','RRBP1'],
        ['IGH@','SH3BGRL3'],
        ['IGH@','SPTBN1'],
        ['IGH@','TMSB4X'],
        ['IGH@','VILL'],
        ['IGH@','ZFP36L1'],
        ['IGHA1','PGA3'],
        ['IGK@','LINC02802'],
        ['IGK@','NCK2'],
        ['IGK@','PNMA8A'],
        ['IGK@','RYBP'],
        ['IGK@','SCAMP4'],
        ['IGLC2','MUCL3'],
        ['IGSF10','MED12L'],
        ['IL6ST','MON1B'],
        ['IL6ST','SLAMF7'],
        ['IL6ST','SLC7A11'],
        ['IL6ST','TRIM2'],
        ['IL6ST','ZEB2'],
        ['ILRUN','SPDEF'],
        ['IMMP2L','TG'],
        ['INPP5B','NKIRAS2'],
        ['INPP5B','WHAMM'],
        ['INSL3','JAK3'],
        ['IQGAP1','ZNF774'],
        ['ITGAE','NCBP3'],
        ['ITGB2','PICALM'],
        ['ITIH1','SERPINA1'],
        ['ITM2B','RB1'],
        ['JADE1','SOX4'],
        ['JAM3','LINC02731'],
        ['JAZF1','RNPS1'],
        ['JCHAIN','RPAIN'],
        ['KANSL1','LRRC37A3'],
        ['KCNE2','SMIM11A'],
        ['KCNE2','SMIM11B'],
        ['KCNK6','SAMHD1'],
        ['KDM1B','LYZ'],
        ['KDM4C','UHRF2'],
        ['KIAA2026','UHRF2'],
        ['KLF16','PTBP1'],
        ['KLF2','OAZ1'],
        ['KLF6','PDK1'],
        ['KLF6','PRLHR'],
        ['KLK11','PRSS30P'],
        ['KLK2','THEM4'],
        ['KLK4','KLKP1'],
        ['KLRA1P','TRA@'],
        ['KMT5A','KRT1'],
        ['KRT1','MYB'],
        ['KRT10','VIM-AS1'],
        ['KRT10','WASF2'],
        ['KRT13','MINK1'],
        ['KRT13','PPP2R1A'],
        ['KRT13','TPM4'],
        ['KRT13','ZBTB16'],
        ['KRT4','MYH9'],
        ['KRT6A','PPL'],
        ['KRT6B','MYH11'],
        ['KRT8','PIGR'],
        ['LAMA3','RBBP8'],
        ['LAMP2','PRRG3'],
        ['LARS','PLAC8L1'],
        ['LATS1','LYZ'],
        ['LATS1','NUP43'],
        ['LAX1','TRA@'],
        ['LDAH','PUM2'],
        ['LDB3','SAMD4B'],
        ['LDLR','TNRC18'],
        ['LGR4','LIN7C'],
        ['LHFPL4','NAP1L4'],
        ['LIG3','ZNF830'],
        ['LIMD1','PKN1'],
        ['LINC-PINT','TG'],
        ['LINC00861','LYZ'],
        ['LINC00893','LINC00894'],
        ['LINC01087','POTEM'],
        ['LINC01186','TMEM178B'],
        ['LINC01297','POTEM'],
        ['LINC01513','ROPN1L'],
        ['LINC01537','TMPRSS11B'],
        ['LINC01649','SLC22A15'],
        ['LINC01761','ZNF528-AS1'],
        ['LINC01808','SCARA5'],
        ['LINC02582','LINC02700'],
        ['LINC02687','MS4A10'],
        ['LINC02743','SPATA19'],
        ['LINC02785','VAV3-AS1'],
        ['LINC02857','MRAP2'],
        ['LIPF','RPL3'],
        ['LMCD1-AS1','SSUH2'],
        ['LMNTD1','TRIM36'],
        ['LNPK','NFIA'],
        ['LPCAT3','PHB2'],
        ['LPGAT1','RPS6KC1'],
        ['LRRC37A3','NSF'],
        ['LRRC49','PIGR'],
        ['LUCAT1','PIGL'],
        ['LYZ','MUC7'],
        ['LYZ','PLD1'],
        ['LYZ','PLEKHB2'],
        ['LYZ','PTP4A2'],
        ['LYZ','PTPN2'],
        ['LYZ','RAD23B'],
        ['LYZ','RPS10-NUDT3'],
        ['LYZ','RPS24'],
        ['MAEA','UVSSA'],
        ['MALRD1','PLXDC2'],
        ['MALT1','RFWD3'],
        ['MAML2','MTMR2'],
        ['MANBAL','SRC'],
        ['MAP3K20','SSTR2'],
        ['MAP7D3','PIGR'],
        ['MAPK10','PTN'],
        ['MAPK10','TRIM36'],
        ['MAPK10','WDR44'],
        ['MAPKBP1','MGA'],
        ['MARK2','ZDHHC24'],
        ['MAT1A','P4HA2'],
        ['MBP','SET'],
        ['MCC','YWHAZ'],
        ['MDM2','NUP107'],
        ['MECOM','RPL22'],
        ['MED12','NUMBL'],
        ['MED21','PPFIBP1'],
        ['MEFV','PLEKHH2'],
        ['METRNL','RASA3'],
        ['METRNL','TBCD'],
        ['METTL7A','PCAT7'],
        ['MFSD14A','SLC35A3'],
        ['MFSD14C','ZNF782'],
        ['MICAL2','TEAD1'],
        ['MICAL3','TLR6'],
        ['MIF','SLC2A11'],
        ['MINPP1','NUTM2A-AS1'],
        ['MIR100HG','STAG1'],
        ['MLEC','UNC119B'],
        ['MLLT6','TRB@'],
        ['MLYCD','UGCG'],
        ['MOCS1','TP53I3'],
        ['MOV10L1','TYK2'],
        ['MPEG1','WWOX'],
        ['MRC1','PICALM'],
        ['MRC1','TMEM236'],
        ['MRNIP','MUC5AC'],
        ['MRNIP','TRA@'],
        ['MRPS16','TM4SF19-AS1'],
        ['MSH2','TTC7A'],
        ['MSMB','NCOA4'],
        ['MSRB3','TMBIM4'],
        ['MTA3','THADA'],
        ['MTG1','SCART1'],
        ['MTMR14','TMPRSS11B'],
        ['MTSS1','TATDN1'],
        ['MTX2','SATB2'],
        ['MUC7','SMR3A'],
        ['MXRA7','TRA@'],
        ['MYH11','WWOX'],
        ['MYO1F','ZNF414'],
        ['MYO6','PNLIP'],
        ['N4BP2','PNPT1'],
        ['N4BP2L2','RTN3'],
        ['NAA16','NKIRAS2'],
        ['NAIP','OCLN'],
        ['NCL','PLBD2'],
        ['NCOA6','PIGU'],
        ['NCOR2','UBC'],
        ['NCSTN','NHLH1'],
        ['NDRG1','ST3GAL1'],
        ['NDRG3','ZFAND6'],
        ['NDRG3','ZNF829'],
        ['NDUFA10','TOX4'],
        ['NDUFS7','SCD5'],
        ['NEAT1','TFF2'],
        ['NEGR1','RBL1'],
        ['NEXN','TGM4'],
        ['NFATC3','PLA2G15'],
        ['NFE2L2','PTMA'],
        ['NFIB','VMP1'],
        ['NFYC','TG'],
        ['NIFK-AS1','TERB1'],
        ['NIN','SAV1'],
        ['NOL9','PPP1R15B'],
        ['NOVA1','SDC3'],
        ['NPEPL1','STX16'],
        ['NPEPPS','TBC1D3'],
        ['NPIPB3','SMG1'],
        ['NPIPB4','SMG1'],
        ['NPIPB5','SMG1'],
        ['NPM1','TJP1'],
        ['NPTXR','PPP1R12B'],
        ['NT5C3B','ZNF814'],
        ['NTRK2','SMR3B'],
        ['NUP188','PHYHD1'],
        ['OAZ1','PTMA'],
        ['OR2T32P','TRIM58'],
        ['PABPC1','RLIM'],
        ['PABPC1L','YWHAB'],
        ['PALM','UBE2J2'],
        ['PATE2','PKP2'],
        ['PAXBP1','PCDH17'],
        ['PBX3','ZNF518A'],
        ['PBXIP1','PMVK'],
        ['PCCA','WDR60'],
        ['PCDH7','RAB29'],
        ['PCNA','TMEM230'],
        ['PGA3','POLR2A'],
        ['PGBD2','RPL23AP7'],
        ['PHF8','PRH2'],
        ['PIGR','RBM3'],
        ['PIGR','SPG7'],
        ['PIGR','VAC14-AS1'],
        ['PILRB','WDR66'],
        ['PIM3','SCO2'],
        ['PITPNC1','PRKCA'],
        ['PIWIL2','STAR'],
        ['PIWIL3','ZFYVE19'],
        ['PLEKHG4B','STX17'],
        ['PML','SDC3'],
        ['PMS2','POLR2J3'],
        ['PNPO','TBC1D31'],
        ['PNRC1','ZNF292'],
        ['PPP1R12B','SOX2-OT'],
        ['PPP3R1','TMEM38B'],
        ['PRB3','SMR3A'],
        ['PRB4','SLPI'],
        ['PRDX1','TESK2'],
        ['PRELP','SFTPA2'],
        ['PRELP','TRG@'],
        ['PRIM2','SERPINB9P1'],
        ['PRKAA1','TTC33'],
        ['PRKAB1','TMEM233'],
        ['PRKAG2','RHEB'],
        ['PRKCSH','TG'],
        ['PRKCZ','SKI'],
        ['PRMT7','SLC7A6'],
        ['PRSS1','RPS19'],
        ['PRSS1','TUBGCP6'],
        ['PSAP','SMR3B'],
        ['PSMA7','SS18L1'],
        ['PSMB8','TAP1'],
        ['PSPC1','XPO4'],
        ['PSPC1','ZMYM5'],
        ['PTBP3','SUSD1'],
        ['PTMA','RNF44'],
        ['PTMA','SLC38A2'],
        ['PTPN14','SMYD2'],
        ['PUM1','SMR3B'],
        ['PWWP2B','SKI'],
        ['PXK','RPP14'],
        ['PYGM','SF1'],
        ['QRICH1','SOD2'],
        ['RAB11FIP1','ZMAT3'],
        ['RAB29','TRABD2A'],
        ['RAB31','VAPA'],
        ['RAB40C','RNF166'],
        ['RAB7A','ZNF793'],
        ['RAC1','TENM1'],
        ['RACK1','SRSF5'],
        ['RASSF8','SSPN'],
        ['RBM14','RBM4B'],
        ['RBM25','STAT5B'],
        ['RBMS2','SPRYD4'],
        ['RDH11','TRG@'],
        ['RDH11','VTI1B'],
        ['RET','ZNF750'],
        ['RFX1','TOB1'],
        ['RHOA','TKT'],
        ['RHOH','TPM4'],
        ['RIPK1','SERPINB9'],
        ['RLN1','RLN2'],
        ['RMI1','U62317.4'],
        ['RMND1','TBPL1'],
        ['RNASEH2B','TG'],
        ['RNF125','RNF138'],
        ['RNF150','TPM1'],
        ['RNF212','TMED11P'],
        ['RNPS1','YWHAE'],
        ['RPL14','SRP14'],
        ['RPL18A','SMR3B'],
        ['RPL18A','TRB@'],
        ['RPL28','TMEM190'],
        ['RPL38','TTYH2'],
        ['RPL7L1','SNX8'],
        ['RPS19','STATH'],
        ['RRN3P2','SNN'],
        ['RWDD3','TLCD4'],
        ['S100A9','TOPBP1'],
        ['SAMD5','SASH1'],
        ['SCARB1','UBC'],
        ['SCNN1A','TNFRSF1A'],
        ['SDC3','WWOX'],
        ['SEC1P','ZNF826P'],
        ['SELPLG','TMEM119'],
        ['SEMA3B','U73166.1'],
        ['SERF1A','SMN1'],
        ['SERPINF1','TRPM8'],
        ['SF3B1','WWOX'],
        ['SFPQ','SUGP2'],
        ['SFTPB','ZNF562'],
        ['SIDT2','TAGLN'],
        ['SIGLEC16','SIGLEC17P'],
        ['SLC14A1','SORCS1'],
        ['SLC26A4','SLC26A4-AS1'],
        ['SLC26A8','SNX29'],
        ['SLC35E3','ZKSCAN1'],
        ['SLC44A1','TG'],
        ['SLC6A16','TEAD2'],
        ['SMAD4','SSB'],
        ['SMR3B','THRA'],
        ['SMR3B','TIMP3'],
        ['SNTB2','VPS4A'],
        ['SNX9','SYNJ2'],
        ['SORL1','ZKSCAN1'],
        ['SPAG9','ZNFX1'],
        ['SPRN','SYCE1'],
        ['SPRY4','TG'],
        ['SRP19','ZRSR2'],
        ['SSBP1','WEE2'],
        ['SSH2','STK4'],
        ['STARD7','ZFHX3'],
        ['STK10','UBE2D2'],
        ['SYT8','TNNI2'],
        ['TAPBP','WWOX'],
        ['TBCD','TBL1XR1'],
        ['TELO2','UBE2I'],
        ['TFDP2','XRN1'],
        ['TG','TTC19'],
        ['TMED2-DT','WSB1'],
        ['TMEM220','TMEM97'],
        ['TMPO','TRB@'],
        ['TPM1','ZNF284'],
        ['TRA@','TXNDC15'],
        ['TRA@','UBE2V2'],
        ['TRIM8','YIPF5'],
        ['UBA2','WTIP'],
        ['UGDH','YWHAZ'],
        ['UMODL1','ZNF295-AS1'],
        ['USP10','ZDHHC7'],
        ['USP34','XPO1'],
        ['WSB1','ZNF264'],
        ['XPOT','ZNF271P'],
        ['ZBTB8A','ZNF440'],
        ['ZKSCAN8','ZNF192P1'],
        ['ZNF257','ZNF429'],
        ['ZNF429','ZNF738'],
        ['ZNF430','ZNF738'],
        ['ZNF486','ZNF826P'],
        ['ZNF826P','ZNF90'],
        ['ABR','NXN'],
        ['AC004690.2','SPAM1'],
        ['AC004946.1','UBC'],
        ['AC004951.1','DTX2P1-UPK3BP1-PMS2P11'],
        ['AC004980.4','AC091390.3'],
        ['AC005013.1','AC005162.2'],
        ['AC005674.1','WDR1'],
        ['AC005696.3','PAFAH1B1'],
        ['AC006042.3','GLCCI1-DT'],
        ['AC006064.6','LYZ'],
        ['AC006116.6','AC006116.8'],
        ['AC008686.1','OLFM1'],
        ['AC008750.7','NKG7'],
        ['AC008993.1','MIR1302-9HG'],
        ['AC009065.3','PDXDC2P-NPIPB14P'],
        ['AC012485.1','ANXA1'],
        ['AC015911.9','SLFN13'],
        ['AC015977.1','CIB4'],
        ['AC016588.2','AL078621.1'],
        ['AC016907.2','AC106870.3'],
        ['AC016912.2','RHOQ'],
        ['AC017074.1','ACR'],
        ['AC017104.4','LINC00471'],
        ['AC017104.5','NMUR1'],
        ['AC018630.4','AMY2B'],
        ['AC018630.4','BHLHA15'],
        ['AC018630.4','CA6'],
        ['AC018630.4','CCL28'],
        ['AC018630.4','CD44'],
        ['AC018630.4','CD74'],
        ['AC018630.4','CST1'],
        ['AC018630.4','CST2'],
        ['AC018630.4','CST4'],
        ['AC018630.4','DDX17'],
        ['AC018630.4','EEF1A1'],
        ['AC018630.4','EEF2'],
        ['AC018630.4','FKBP8'],
        ['AC018630.4','GAPDH'],
        ['AC018630.4','GNAS'],
        ['AC018630.4','GNE'],
        ['AC018630.4','H1-0'],
        ['AC018630.4','IGH@'],
        ['AC018630.4','IGK@'],
        ['AC018630.4','IL6ST'],
        ['AC018630.4','ITPR1'],
        ['AC018630.4','MUC7'],
        ['AC018630.4','MYH9'],
        ['AC018630.4','PIGR'],
        ['AC018630.4','PIK3R1'],
        ['AC018630.4','PRR27'],
        ['AC018630.4','RBM47'],
        ['AC018630.4','RPL3'],
        ['AC018630.4','SFRP1'],
        ['AC018630.4','SLC20A2'],
        ['AC018630.4','SLC6A8'],
        ['AC018630.4','SMR3B'],
        ['AC018630.4','SORL1'],
        ['AC018630.4','SQSTM1'],
        ['AC018630.4','STATH'],
        ['AC018630.4','ZG16B'],
        ['AC018809.3','PRRT3-AS1'],
        ['AC019205.1','EIF3E'],
        ['AC020656.1','CEACAM5'],
        ['AC020656.1','KHDC4'],
        ['AC020656.1','MYLK'],
        ['AC020656.1','POLH'],
        ['AC020656.1','RPL37'],
        ['AC020656.1','XACT'],
        ['AC020907.3','SYT11'],
        ['AC021517.3','SLC14A2-AS1'],
        ['AC021739.2','LINC02883'],
        ['AC022210.1','KCMF1'],
        ['AC022733.1','LINC02866'],
        ['AC023794.1','LINC02381'],
        ['AC024560.1','AC026412.4'],
        ['AC024940.1','AC092666.1'],
        ['AC024940.1','SSUH2'],
        ['AC053503.3','MYH6'],
        ['AC053503.3','MYL2'],
        ['AC053503.3','NPPA'],
        ['AC053503.3','TPM1'],
        ['AC055876.5','LINC02256'],
        ['AC063979.1','LINC02056'],
        ['AC068643.1','C12ORF42'],
        ['AC073585.1','DOCK1'],
        ['AC073585.1','GP2'],
        ['AC073585.1','PNLIP'],
        ['AC083899.1','AC133644.2'],
        ['AC087518.1','LINC02866'],
        ['AC091057.1','ARHGAP11B'],
        ['AC091167.1','PERP'],
        ['AC091390.3','AC211429.1'],
        ['AC091390.3','AC211486.3'],
        ['AC091390.3','PMS2'],
        ['AC091390.3','POM121'],
        ['AC091390.3','POM121C'],
        ['AC091390.3','RBAK'],
        ['AC091982.1','G3BP1'],
        ['AC093010.1','TMPRSS11B'],
        ['AC093162.1','RETSAT'],
        ['AC093512.2','ALDOB'],
        ['AC093512.2','ALDOC'],
        ['AC093525.2','ZEB2'],
        ['AC093787.2','AC233263.7'],
        ['AC098828.1','LAPTM4A'],
        ['AC109583.1','PRSS42P'],
        ['AC109583.4','PRSS46P'],
        ['AC113189.12','SPEM2'],
        ['AC115220.2','LINC01005'],
        ['AC115220.2','MAEL'],
        ['AC118758.2','LINC00174'],
        ['AC122688.3','H2AZ2'],
        ['AC132217.2','COL4A2'],
        ['AC132217.2','H19'],
        ['AC133644.2','ANAPC1'],
        ['AC133644.2','PLG'],
        ['AC134980.1','OR4N2'],
        ['AC138123.2','CAP1'],
        ['AC138409.2','AC138866.1'],
        ['AC138409.2','NAIP'],
        ['AC138866.1','AL021368.4'],
        ['AC138866.1','AL513548.1'],
        ['AC138866.1','LINC00680'],
        ['AC138866.1','NAIP'],
        ['AC138866.1','WDR70'],
        ['AC138969.1','PKD1'],
        ['AC140479.4','MALL'],
        ['AC211476.4','CCDC146'],
        ['AC211476.4','PMS2'],
        ['AC211476.4','UPK3B'],
        ['AC211476.5','PMS2'],
        ['AC239859.5','LINC02802'],
        ['AC244197.3','ZMAT4'],
        ['AC245452.4','IGLVIV-66-1'],
        ['AC245748.1','ZNF806'],
        ['ACAA1','CTSS'],
        ['ACAP2','KPNA4'],
        ['ACBD5','LRRC37A6P'],
        ['ACBD5','SMCHD1'],
        ['ACTB','HMGB1'],
        ['ACTR1A','GBF1'],
        ['ADAM3A','NUP153'],
        ['ADSL','PGK1'],
        ['AGAP11','AL512662.1'],
        ['AHCTF1','KMT2E'],
        ['AKAP17A','FOS'],
        ['AL022238.3','TNRC6B'],
        ['AL033527.2','AL033527.3'],
        ['AL049828.1','MIA2'],
        ['AL049869.2','ZBTB25'],
        ['AL050303.2','LINC01087'],
        ['AL050303.2','LINC01297'],
        ['AL109809.4','AL117335.1'],
        ['AL121578.1','SYTL5'],
        ['AL133351.4','NQO2'],
        ['AL136309.3','C6ORF201'],
        ['AL138963.3','HBA2'],
        ['AL157834.4','CYP2C8'],
        ['AL158066.1','THSD1'],
        ['AL355075.4','CCNB1IP1'],
        ['AL355312.2','AL355312.5'],
        ['AL358777.3','PAK1IP1'],
        ['AL365295.1','SENP6'],
        ['AL391628.1','TRB@'],
        ['AL445669.1','PPIH'],
        ['AL512590.2','CCDC180'],
        ['AL512662.1','BMS1'],
        ['AL513217.1','SHISA4'],
        ['AL590004.3','ETS2'],
        ['AL590556.3','ATP1A1-AS1'],
        ['AL590556.3','INS'],
        ['AL590666.1','CRABP2'],
        ['AL591848.1','AL591848.2'],
        ['ANAPC1','ANAPC1P2'],
        ['ANXA1','DDX5'],
        ['ANXA1','RBM39'],
        ['AP000356.5','PIGS'],
        ['AP000812.4','FOLR3'],
        ['AP000943.3','FUT4'],
        ['APLF','RTRAF'],
        ['APOA1','CALR'],
        ['APOA1','CYP2E1'],
        ['APOA1','DPYS'],
        ['APOA1','HP'],
        ['APOA1','REG3A'],
        ['APPL2','KCCAT198'],
        ['ARF1','MCM3AP'],
        ['ART3','SDAD1-AS1'],
        ['ASB15-AS1','NDUFA5'],
        ['ASS1','RIPOR2'],
        ['ATP1B1','USP7'],
        ['ATP5MF','ZNF394'],
        ['ATP6V1E1','TG'],
        ['ATP8B4','ETS2'],
        ['AZU1','ZNF540'],
        ['B2M','DDX5'],
        ['B4GALT1','SMU1'],
        ['BAIAP2-DT','TCF3'],
        ['BASP1','Z84488.1'],
        ['BCL11B','PIGR'],
        ['BCL11B','PRH1'],
        ['BCL11B','PTP4A2'],
        ['BCL11B','RPL3'],
        ['BCL11B','ZNF880'],
        ['BCLAF1','LGALSL'],
        ['BHLHA15','PRH2'],
        ['BHLHE40','LTF'],
        ['BLNK','NPM1'],
        ['BRD1','G3BP1'],
        ['BRD2','RACK1'],
        ['C19MC','SERINC3'],
        ['C1ORF116','YOD1'],
        ['CALM1','HIF1A'],
        ['CAPZA1','MOV10'],
        ['CAPZA2','H2AZ2'],
        ['CCDC81','HIKESHI'],
        ['CCDC88C','MUC7'],
        ['CCND3','SDC3'],
        ['CD24','EEF1A1'],
        ['CD24','LINC02321'],
        ['CDK10','LYZ'],
        ['CDRT15L2','TBC1D28'],
        ['CEACAM5','LINC01553'],
        ['CELA2A','TRIP12'],
        ['CEP104','LINC01031'],
        ['CGB7','NTF4'],
        ['CHCHD1','FUT11'],
        ['CHEK2','NAIP'],
        ['CHMP1A','TG'],
        ['CHMP3','EEF1A1'],
        ['CHMP4B','TRB@'],
        ['CLCC1','GBP6'],
        ['CLCC1','WFDC3'],
        ['CLCN6','EXOSC7'],
        ['CLCN6','PBX1'],
        ['CLEC2D','NPM1'],
        ['COL1A2','IGF2'],
        ['COPB1','RRAS2'],
        ['COPB2','RNF130'],
        ['COPZ2','NFE2L1-DT'],
        ['CPA1','FTL'],
        ['CPSF6','NINJ2-AS1'],
        ['CSDC2','LINC02348'],
        ['CSHL1','H19'],
        ['CST4','PPP1R1B'],
        ['CTRL','PRSS3'],
        ['CTSB','LIPG'],
        ['CWF19L1','DA750114'],
        ['DCAF1','ITGB1'],
        ['DCAF11','PCK2'],
        ['DDX17','EEF1A1'],
        ['DDX17','USP15'],
        ['DPY19L2','SORD'],
        ['DSG3','SYNRG'],
        ['DTX2P1-UPK3BP1-PMS2P11','EEF1A1'],
        ['DUS4L-BCAP29','SLC26A4'],
        ['DYNLL2','PABPC1'],
        ['EDEM3','MNDA'],
        ['EEF1A1','ELANE'],
        ['EEF1A1','FUS'],
        ['EEF1A1','HNRNPC'],
        ['EEF1A1','OBSL1'],
        ['EEF1A1','PECAM1'],
        ['EEF1A1','PRB3'],
        ['EEF1A1','RPS3A'],
        ['EEF1A1','SCD5'],
        ['EEF1A1','SFPQ'],
        ['EEF1A1','SH3D19'],
        ['EEF1A1','TNP1'],
        ['EEF1A1','TRIM22'],
        ['EEF1G','HBB'],
        ['EIF3E','HTN3'],
        ['ETS2','NOVA1'],
        ['EXOC6','SHROOM3'],
        ['EXOC8','PIGH'],
        ['FAM133B','FAM133EP'],
        ['FAM133B','FAM133FP'],
        ['FASN','PRB2'],
        ['FAT4','LYZ'],
        ['FCAR','MARK3'],
        ['FOSB','HMGB1'],
        ['FP475955.3','PSLNR'],
        ['FTX','STATH'],
        ['GBF1','TG'],
        ['GMFG','LRFN1'],
        ['GP2','PPM1G'],
        ['GPR160','HBB'],
        ['H2BW3P','TMSB15B'],
        ['HBB','PLEKHB2'],
        ['HBB','RPS4X'],
        ['HGSNAT','NDUFS1'],
        ['HIBADH','TAX1BP1-AS1'],
        ['HNRNPH1','STATH'],
        ['HP','HPX'],
        ['HSPD1','TG'],
        ['HSPG2','KRT4'],
        ['HTN3','TMED10'],
        ['IFI44L','PDGFA'],
        ['IFT172','TG'],
        ['IFT80','MOB1A'],
        ['IGBP1','KLF8'],
        ['IGF2BP1','LYZ'],
        ['IGH@','PCBP2'],
        ['IGH@','PGA4'],
        ['IGH@','TUBA1B'],
        ['IGHA1','PGA4'],
        ['IGK@','PIPOX'],
        ['IL9R','RPL23AP53'],
        ['ITM2B','SI'],
        ['KANSL1L-AS1','RPE'],
        ['KCNK3','NEB'],
        ['KDM6B','SMR3B'],
        ['KIR3DX1','UGDH'],
        ['LAMP2','SMAD1'],
        ['LCOR','TG'],
        ['LINC01297','PSLNR'],
        ['LINC01923','NDUFV1'],
        ['LINC02210_','MORC2-AS1'],
        ['LMX1A-AS1','LMX1A-AS2'],
        ['LYZ','NCAPH2'],
        ['LYZ','XPO1'],
        ['MACF1','TAGLN'],
        ['MALT1','STAR'],
        ['MAP3K20','XPO1'],
        ['MARK3','TG'],
        ['MB','MFN2'],
        ['MB','NEK9'],
        ['MB','RPS4X'],
        ['METTL6','NUP58'],
        ['MFF','PDE6A'],
        ['MFN2','TG'],
        ['MICALL1','RBM39'],
        ['MIR100HG','RAB29'],
        ['MKNK2','PTBP1'],
        ['MRPS31P5','THSD1'],
        ['MT1E','MT1M'],
        ['MTHFD2','STAG3'],
        ['MTR','TG'],
        ['MUC7','NFIB'],
        ['NDRG2','PRR27'],
        ['NEK4','ZBTB5'],
        ['NEK9','SMR3B'],
        ['NONO','S100A9'],
        ['NPM1','SET'],
        ['NPNT','PPA2'],
        ['NT5C2','TG'],
        ['NTRK2','PNLIP'],
        ['NUTM2A','NUTM2B'],
        ['PADI4','S100A9'],
        ['PDCD6IP','STATH'],
        ['PGBD2','RPL23AP25'],
        ['PGK1','POLA1'],
        ['PIGR','SI'],
        ['PIP5K1B','ZFAND5'],
        ['PMS2','POM121C'],
        ['PMS2','UPK3B'],
        ['PNLIP','SERPING1'],
        ['PRSS43P','PRSS44P'],
        ['PSMB2','TFAP2E-AS1'],
        ['PTGES2','TGFBR1'],
        ['PTPN14','TSR1'],
        ['RAB29','ZNF665'],
        ['RAB9B','TMSB15B-AS1'],
        ['RBAK','RNF216P1'],
        ['RBFOX1','SDC3'],
        ['RBFOX1','UBE2D3'],
        ['RBM39','ZNF791'],
        ['RMI1','U62317.3'],
        ['RNF170','THAP1'],
        ['RNF38','TG'],
        ['RPS19','TIGD1'],
        ['S100A9','SPEN'],
        ['S100A9','SRSF3'],
        ['SEMA6A','SMR3B'],
        ['SERF2','WSB1'],
        ['SIPA1L3','SMR3B'],
        ['SMAP2','TNK2'],
        ['SMARCE1','TG'],
        ['SMKR1','STRIP2'],
        ['SMR3B','SVIL'],
        ['SPTBN2','WWOX'],
        ['SSB','SSBL4P'],
        ['SSX1','SSX8P'],
        ['STARD3','TCAP'],
        ['STAU1','TG'],
        ['TAF1L','TMEM154'],
        ['TG','TRIM2'],
        ['TG','UGP2'],
        ['TG','VAMP7'],
        ['TGM4','XPO1'],
        ['TMSB15B','TMSB15B'],
        ['TRPM2','ZNF511'],
        ['UBC','YWHAZ'],
        ['ZNF285','ZNF806']

]



    data = fusions.get(options.organism.lower(),[])
    if data:

        #file_symbols = os.path.join(options.output_directory,'genes_symbols.txt')
        file_symbols = os.path.join(options.output_directory,'synonyms.txt')
        loci = symbols.generate_loci(file_symbols)

        genes = symbols.read_genes_symbols(file_symbols)

        d = []
        for (g1,g2) in data:
            if g1.upper() != g2.upper():
                ens1 = symbols.ensembl(g1.upper(),genes,loci)
                ens2 = symbols.ensembl(g2.upper(),genes,loci)
                if ens1 and ens2:
                    for e1 in ens1:
                        for e2 in ens2:
                            if e1 != e2:
                                d.append([e1,e2])

        data = ['\t'.join(sorted(line)) + '\n' for line in d]
        data = list(set(data))

        print "%d known fusion genes found in manually currated database" % (len(data),)

        if not options.skip_filter_overlap:
            d1 = []
            overlappings = ['ensembl_fully_overlapping_genes.txt',
                            'ensembl_same_strand_overlapping_genes.txt',
                            'refseq_fully_overlapping_genes.txt',
                            'refseq_same_strand_overlapping_genes.txt',
#                            'ucsc_fully_overlapping_genes.txt',
#                            'ucsc_same_strand_overlapping_genes.txt',
                            'pairs_pseudogenes.txt',
                            'paralogs.txt']
            for ov in overlappings:
                p = os.path.join(options.output_directory,ov)
                print "Parsing file:",p
                if os.path.isfile(p):
                    d2 = sorted(set([tuple(sorted(line.rstrip('\r\n').split('\t'))) for line in file(p,'r').readlines() if line.rstrip('\r\n')]))
                d1.extend(d2)
            d = set()
            for line in d1:
                (a,b) = (line[0],line[1])
                if a > b:
                    (a,b) = (b,a)
                d.add("%s\t%s\n" % (a,b))
            skipped = [line for line in data if line in d]
            data = [line for line in data if not line in d]
            file(os.path.join(options.output_directory,'hpa_known_but_overlapping.txt'),'w').writelines(sorted(skipped))

            print "%d known fusion genes left after removing the overlappings" % (len(data),)

    file(os.path.join(options.output_directory,'hpa.txt'),'w').writelines(sorted(data))
    #
