#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of allowed candidate fusion genes. This list is hard coded
inhere and it is manually curated from publications.



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of allowed candidate fusion genes (which are manually curated from scientific articles)."""
    version = "%prog 0.14 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of allowed candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of allowed candidate fusion genes is generated. Default is '%default'.""")

    parser.add_option("--skip-filter-overlap",
                      action="store_true",
                      dest="skip_filter_overlap",
                      default = False,
                      help="""If set then it filters out the known fusion genes where the (i) genes are fully overlapping, or (ii) the genes are partially overlapping and are on the same strand. Default is '%default'.""")

    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    print "Generating the list of allowed/known fusion genes..."
    fusions = dict()

    # manual curation from papers

    fusions['rattus_norvegicus'] = []

    fusions['mus_musculus'] = []

    fusions['canis_lupus_familiaris'] = []

    fusions['homo_sapiens'] = [
        ['A121','VEGF'],
        ['A1BG-AS1','APOA2'],
        ['A20','K20'],
        ['A20','TNFAIP3'],
        ['A2M','ALK'],
        ['A2M','NOL10'],
        ['AAC1','AAC2'],
        ['AACSP1','ZNF354A'],
        ['AADACL2','ENSA'],
        ['AADACL2','SLC45A3'],
        ['EWSR1','SSX2'],
        ['AADAT','TPT1'],
        ['AAK1','ARNT2'],
        ['AAK1','C2ORF42'],
        ['AAK1','LHX2'],
        ['AAK1','RPL36A'],
        ['AAK1','TEX261'],
        ['AAK1','WFDC10B'],
        ['AAK1','WFDC13'],
        ['AAMDC','SLC12A7'],
        ['AARSD1','MAP2K6'],
        ['AARSD1','NF1'],
        ['AASDH','LRRC66'],
        ['AASDH','PPAT'],
        ['TNIP1','PDGFRB'],
        ['PCM1','FGFR1'],
        ['FYCO1','RAF1'],
        ['CTTNBP2','BRAF'],
        ['SLC44A1','BRAF'],
        ['KIAA1549','BRAF'],
        ['TBLR1','RARA'],
        ['AATF','TADA2A'],
        ['AATF','ZMYND11'],
        ['ABAT','NAT10'],
        ['ABCA2','FYTTD1'],
        ['ABCA4','BCAR3'],
        ['PGR','NR4A3'],
        ['UBR5','PGR'],
        ['ABCA5','PPP4R1L'],
        ['ABCA5','UNK'],
        ['ABCA8','TTC37'],
        ['ABCA9','MED13'],
        ['ABCB1','DMTF1'],
        ['ABCB1','SEMA3D'],
        ['ABCB1','SLC25A40'],
        ['ABCB11','LRP2'],
        ['ABCB9','ZDHHC7'],
        ['ABCC1','C16ORF45'],
        ['ABCC1','MPV17L'],
        ['ABCC1','MRP1'],
        ['ABCC1','TP53'],
        ['ABCC3','MRP3'],
        ['ABCC4','ETV4'],
        ['ABCC8','VCL'],
        ['ABCD3','C1ORF123'],
        ['ABCD3','CPOX'],
        ['ABCF1','PLVAP'],
        ['ABCG1','SLC37A1'],
        ['ABCG2','BCRP'],
        ['ABCG8','EPCAM'],
        ['ABHD11','BAZ1B'],
        ['ABHD12','CRTC1'],
        ['ABHD12','ZNF337'],
        ['ABHD12B','PYGL'],
        ['ABHD12B','RCC1'],
        ['ABHD12B','SAV1'],
        ['ABHD14B','DHX30'],
        ['ABHD15','TAOK1'],

        ['MEIS1','NCOA1'],
        ['KMT2A','AFDN'],
        ['ZNF532','NUTM1'],
        ['PGAP3','SRPK1'],
        ['CTNNBIP1','CLSTN1'],

        ['GHR','ALK'],
        ['HOOK3','FGFR1'],
        ['GOLGA4','JAK2'],
        ['PGAP3','SRPK1'],
        ['CD63','BCAR4'],
        ['PAFAH1B1','USP6'],
        ['CTNNBIP1','CLSTN1'],

        ['ABHD3','ESCO1'],
        ['ABHD3','ROCK1'],
        ['ABI1','BMI1'],
        ['ABI1','KMT2A'],
        ['ABI1','MLL'],
        ['ABI1','PDSS1'],
        ['ABI2','KMT2A'],
        ['ABI2','MLL'],
        ['ABI2','MTA3'],
        ['ABL','BCR'],
        ['ABL','ETV6'],
        ['ABL','HOXA9'],
        ['ABL','MYC'],
        ['ABL','NUP214'],
        ['ABL','PDGFRA'],
        ['ABL','PDGFRB'],
        ['ABL1','ADAMTSL2'],
        ['ABL1','AIF1L'],
        ['ABL1','ANXA4'],
        ['ABL1','ARNT'],
        ['ABL1','BCR'],
        ['ABL1','CBFB'],
        ['ABL1','CDX2'],
        ['ABL1','CENPC'],
        ['ABL1','CENPC1'],
        ['ABL1','CHIC2'],
        ['ABL1','CNTRL'],
        ['ABL1','DEK'],
        ['ABL1','EML1'],
        ['ABL1','ETV6'],
        ['ABL1','EVI1'],
        ['ABL1','EXOSC2'],
        ['ABL1','FACL6'],
        ['ABL1','FGFR1'],
        ['ABL1','FGFR3'],
        ['ABL1','FOXP1'],
        ['ABL1','GAB1'],
        ['ABL1','HLXB9'],
        ['ABL1','INPP5D'],
        ['ABL1','JAK2'],
        ['ABL1','KIAA1671'],
        ['ABL1','LSM14A'],
        ['ABL1','MDS2'],
        ['ABL1','MN1'],
        ['ASAP1','USP6'],
        ['FAT1','USP6'],
        ['SAR1A','USP6'],
        ['TNC','USP6'], 
        ['ABL1','NUP153'],
        ['ABL1','NUP214'],
        ['ABL1','PAX5'],
        ['ABL1','PER1'],
        ['ABL1','PRDM12'],
        ['ABL1','RALGPS1'],
        ['GOLGA4','JAK2'],
        ['ABL1','RANBP2'],
        ['ABL1','RCSD1'],
        ['ABL1','RNF213'],
        ['ABL1','SEPT9'],
        ['ABL1','SFPQ'],
        ['ABL1','SNX2'],
        ['ABL1','SPTAN1'],
        ['ABL1','STL'],
        ['ABL1','SYK'],
        ['ABL1','TEL'],
        ['ABL1','TNRC6B'],
        ['ABL1','TTL'],
        ['ABL1','ZBTB16'],
        ['ABL1','ZMIZ1'],
        ['ABL2','ETV6'],
        ['ABL2','GAB2'],
        ['ABL2','KMT2A'],
        ['ABL2','PAG1'],
        ['ABL2','RCSD1'],
        ['ABL2','SLC38A2'],
        ['ABL2','ZC3HAV1'],
        ['ABLIM1','PDZD8'],
        ['ABLIM2','DIP2C'],
        ['ABR','CLUH'],
        ['ABR','FAM76A'],
        ['PTBP1','MAML2'],
        ['ABR','GAS7'],
        ['ABR','KCNK1'],
        ['ABR','STAP2'],
        ['ABR','YWHAE'],
        ['ABRACL','MAP3K5'],
        ['AC004066.1','PPA2'],
        ['AC004870.4','EGFR'],
        ['AC005262.1','AES'],
        ['AC006372.2','KMT2C'],
        ['AC011523.1','KDM4A'],
        ['AC011523.1','KDM4B'],
        ['AC011997.1','PLCB1'],
        ['AC012314.2','TRPC5'],
        ['AC012354.6','SIX3'],
        ['AC015849.2','TAF15'],
        ['AC016582.1','TP53'],
        ['AC019197.1','SCN2A'],
        ['AC025048.2','TP53'],
        ['AC027612.3','MRPS6'],
        ['AC079780.1','THADA'],
        ['AC083837.1','IL7'],
        ['AC090627.1','THRA'],
        ['AC090987.1','PVT1'],
        ['AC091132.4','KANSL1'],
        ['AC099850.1','TMEM49'],
        ['AC099850.1','VMP1'],
        ['AC117386.2','OLFM1'],
        ['AC126544.2','EPOR'],
        ['AC138031.1','ADRB2'],
        ['AC141586.5','KCTD5'],
        ['AC243829.2','TP53'],
        ['AC245884.1','EPOR'],
        ['ACA11','WHSC1'],
        ['ACACA','C20ORF72'],
        ['ACACA','DDX52'],
        ['ACACA','ERBB2'],
        ['ACACA','HTRA4'],
        ['ACACA','KMT2A'],
        ['ACACA','MLL'],
        ['ACACA','MSI2'],
        ['ACACA','STAC2'],
        ['ACACA','UTP18'],
        ['ACAD10','MAPKAPK5'],
        ['ACAD11','CPNE4'],
        ['ACAD8','GLB1L3'],
        ['ACAD9','C3ORF46'],
        ['ACADSB','FGFR2'],
        ['ACADVL','FXR2'],
        ['ACAT1','TESK1'],
        ['ACBD3','ESRRG'],
        ['ACBD5','RET'],
        ['ACBD6','IGF1R'],
        ['ACBD6','RRP15'],
        ['ACCN1','NF1'],
        ['ACCS','BCL7A'],
        ['ACCS','EXT2'],
        ['ACCS','KAT6A'],
        ['ACE','KPNB1'],
        ['ACE2','PIR'],
        ['ACER1','MLL'],
        ['ACER3','CALCOCO2'],
        ['ACER3','PAK1'],
        ['ACER3','TAOK1'],
        ['ACIN1','NUTM1'],
        ['ACKR2','KRBOX1'],
        ['ACLY','ERBB2'],
        ['ACLY','NTN1'],
        ['ACO1','GNRHR2P1'],
        ['ACO2','ARHGEF17'],
        ['ACOT11','BLK'],
        ['ACOT2','RXFP1'],
        ['ACOT7','RERE'],
        ['ACOT9','APOO'],
        ['ACOT9','DMD'],
        ['ACOT9','SCHIP1'],
        ['ACP6','AHCTF1'],
        ['ACP6','CNOT2'],
        ['ACPL2','RASA2'],
        ['ACPP','AXL'],
        ['ACPP','CD200'],
        ['ACPP','ETV1'],
        ['ACPP','FGFR1'],
        ['ACPP','PIK3CB'],
        ['ACPP','SEC13'],
        ['ACPP','SKIL'],
        ['ACRBP','ENO2'],
        ['ACS2','ETV6'],
        ['ACSF2','MED1'],
        ['ACSF3','CBFA2T3'],
        ['ACSL3','C15ORF21'],
        ['ACSL3','COL4A3'],
        ['ACSL3','ETV1'],
        ['ACSL3','MOGAT1'],
        ['ACSL4','DCAF6'],
        ['ACSL5','FGFR2'],
        ['ACSL5','GUCY2GP'],
        ['ACSL6','ETV6'],
        ['ACSM1','IKBKE'],
        ['ACSM3','EEF2K'],
        ['ACSS1','C20ORF3'],
        ['ACSS1','RARA'],
        ['ACSS3','PITPNM2'],
        ['ACTA1','CARD11'],
        ['ACTA2','PLAG1'],
        ['ACTB','COL1A1'],
        ['ACTB','FOSB'],
        ['ACTB','GLI1'],
        ['ACTB','MAP7D1'],
        ['ACTB','MITF'],
        ['ACTB','OXSR1'],
        ['ACTB','TFEB'],
        ['ACTG1','MITF'],
        ['ACTG1','PDLIM5'],
        ['ACTG2','ALK'],
        ['ACTG2','DGUOK'],
        ['ACTG2','UFM1'],
        ['ACTL6A','WWTR1'],
        ['ACTN1','MYBL1'],
        ['ACTN2','TBCE'],
        ['ACTN4','AKT2'],
        ['ACTN4','C19ORF10'],
        ['ACTN4','COQ8B'],
        ['ACTN4','EIF3K'],
        ['ACTN4','KMT2A'],
        ['ACTN4','MLL'],
        ['ACTR10','ARID4A'],
        ['ACTR1B','ERVW-1'],
        ['ACTR3','RPS15'],
        ['ACTR3B','DPP6'],
        ['ACTR3B','MLL3'],
        ['ACTR8','IL17RB'],
        ['ACVR1','NFATC1'],
        ['ACVR1B','BAZ2A'],
        ['ACVR1B','PGC'],
        ['ACVR1C','CHN1'],
        ['ACVR2A','AKT3'],
        ['ACVR2A','CASP8'],
        ['ACVR2A','FN1'],
        ['ACVR2A','IGF2R'],
        ['ACVR2A','SUMO1'],
        ['ACVR2B','XYLB'],
        ['ADAM10','NUP37'],
        ['ADAM12','AFF1'],
        ['ADAM17','SLC23A3'],
        ['ADAM17','YWHAQ'],
        ['ADAM18','FGFR1'],
        ['ADAM28','BCR'],
        ['ADAM29','RBMY1J'],
        ['ADAM9','ANK1'],
        ['ADAM9','ASAH1'],
        ['ADAM9','NRG1'],
        ['ADAM9','PLEKHA2'],
        ['ADAM9','TM2D2'],
        ['ADAMTS12','PXDNL'],
        ['ADAMTS12','TRIO'],
        ['ADAMTS13','LHX2'],
        ['ADAMTS14','SNHG11'],
        ['ADAMTS17','ADAMTSL3'],
        ['ADAMTS17','LRRK1'],
        ['ADAMTS17','PML'],
        ['ADAMTS17','RARA'],
        ['ADAMTS19','RUNX1'],
        ['ADAMTS19','SLC27A6'],
        ['ADAMTS6','ARID1B'],
        ['ADAMTSL1','DENND4C'],
        ['ADAMTSL3','MAP2K5'],
        ['ADAP1','SUN1'],
        ['ADAP2','TTC28-AS1'],
        ['ADAR','TPM3'],
        ['ADARB1','PCBP3'],
        ['ADAT3','SPRY4'],
        ['ADC','SASS6'],
        ['ADCK1','KCNQ2'],
        ['ADCK1','OSBPL2'],
        ['ADCK1','TMC1'],
        ['ADCK2','HIPK2'],
        ['ADCK2','TBXAS1'],
        ['ADCK3','CYP2E1'],
        ['ADCK3','EPHX1'],
        ['ADCK3','KCNT2'],
        ['ADCK3','NEK7'],
        ['ADCK3','SPRN'],
        ['ADCK4','AKAP8'],
        ['ADCK4','NUMBL'],
        ['ADCK4','PPM1A'],
        ['ADCY3','SOS1'],
        ['ADCY4','RABGGTA'],
        ['ADCY7','NKD1'],
        ['ADCY8','ASAP1'],
        ['ADCY9','CDK12'],
        ['ADCY9','PRKCB'],
        ['ADCYAP1','CASP9'],
        ['ADD1','FGFR3'],
        ['ADD1','HTT'],
        ['ADD1','POMC'],
        ['ADD2','EXOC6B'],
        ['ADD3','NUP98'],
        ['ADGRF5','THBS1'],
        ['ADGRG6','ROS1'],
        ['ADGRG7','TFG'],
        ['ADH7','CNOT2'],
        ['ADIPOR2','NINJ2'],
        ['ADK','C10ORF11'],
        ['ADK','CD133'],
        ['ADK','DLG5'],
        ['ADK','EIF3A'],
        ['ADK','KAT6B'],
        ['ADK','PI4K2A'],
        ['ADK','SAMD8'],
        ['ADK','VCL'],
        ['ADNP','C20ORF132'],
        ['ADNP2','ITGB3BP'],
        ['ADORA2B','TTC19'],
        ['ADPRHL1','ZP4'],
        ['ADRA1A','GUF1'],
        ['ADRA1A','YIPF7'],
        ['ADRB3','DIS3'],
        ['ADRBK1','BAIAP2'],
        ['ADRBK1','CHGA'],
        ['ADRBK1','IGH@'],
        ['ADRBK1','IGHMBP2'],
        ['ADRBK1','POLD4'],
        ['ADRBK2','AP1B1'],
        ['ADRBK2','EHD1'],
        ['ADRBK2','EP300'],
        ['ADRBK2','RHBDL3'],
        ['ADRBK2','SF3A1'],
        ['ADRBK2','SGSM3'],
        ['ADRBK2','WWC1'],
        ['AE1','AE3'],
        ['AEBP2','PVT1'],
        ['AEBP2','VPS4B'],
        ['AES','FGFR3'],
        ['AF10','AF17'],
        ['AF10','CALM'],
        ['AF10','HEAB'],
        ['AF10','MLL'],
        ['AF10','MLLT10'],
        ['AF10','PICALM'],
        ['AF127577.4','NRIP1'],
        ['AF127936.2','NRIP1'],
        ['AF15Q14','MLL'],
        ['AF17','KMT2A'],
        ['AF17','MLL'],
        ['AF17','MLLT6'],
        ['AF186192.5','AGAP5'],
        ['AF1P','EPS15'],
        ['AF1P','MLL'],
        ['AF1Q','ARHGEF12'],
        ['AF1Q','CBFA2T1'],
        ['AF1Q','CREBBP'],
        ['AF1Q','ELL'],
        ['AF1Q','EP300'],
        ['AF1Q','FNBP1'],
        ['AF1Q','FOXO3A'],
        ['AF1Q','GAS7'],
        ['AF1Q','GMPS'],
        ['AF1Q','GPHN'],
        ['AF1Q','GRAF'],
        ['AF1Q','HEAB'],
        ['AF1Q','LAF4'],
        ['AF1Q','LCX'],
        ['AF1Q','LPP'],
        ['AF1Q','MLL'],
        ['AF1Q','MSF'],
        ['AF1Q','PICALM'],
        ['AF1Q','PNUTL1'],
        ['AF1Q','SEPT6'],
        ['AF1Q','SH3GL1'],
        ['AF1Q','SSH3BP1'],
        ['AF3P21','MLL'],
        ['AF4','AF9'],
        ['AF4','AFF1'],
        ['AF4','FMR2'],
        ['AF4','KMT2A'],
        ['AF4','MLL'],
        ['AF4','NOL4L'],
        ['AF5Q31','MLL'],
        ['AF6','MLL'],
        ['AF6','MLLT4'],
        ['AF9','DOT1L'],
        ['AF9','KMT2A'],
        ['AF9','MLL'],
        ['AF9','MLLT3'],
        ['AFAP','NTRK2'],
        ['AFAP1','NTRK1'],
        ['AFAP1','NTRK2'],
        ['AFAP1','NTRK3'],
        ['AFAP1L1','PDGFRB'],
        ['AFAP1L2','RET'],
        ['AFDN','KMT2A'],
        ['AFF1','ARHGAP17'],
        ['AFF1','C1QTNF1'],
        ['AFF1','DOM'],
        ['AFF1','DSCAML1'],
        ['AFF1','ELF2'],
        ['AFF1','FXYD6'],
        ['AFF1','GAS7'],
        ['AFF1','HOXA13'],
        ['AFF1','IL10RB'],
        ['AFF1','KMT2A'],
        ['AFF1','LYZ'],
        ['AFF1','MLL'],
        ['AFF1','MYC'],
        ['AFF1','PAIP2'],
        ['AFF1','PBX1'],
        ['AFF1','PHACTR2'],
        ['AFF1','PIM1'],
        ['AFF1','PTPN13'],
        ['AFF1','RABGAP1L'],
        ['AFF1','RAD51B'],
        ['AFF1','SRSF11'],
        ['AFF1','TRIM33'],
        ['AFF1','UBE2D3'],
        ['AFF2','DEK'],
        ['AFF2','FMR2'],
        ['AFF3','BCL2'],
        ['AFF3','FGFR2'],
        ['AFF3','KMT2A'],
        ['AFF3','LAF4'],
        ['AFF3','MAP4K4'],
        ['AFF3','MLL'],
        ['AFF3','PHF1'],
        ['AFF3','RUNX1'],
        ['AFF4','FGFR2'],
        ['AFF4','KMT2A'],
        ['AFF4','MLL'],
        ['AFF4','SEPT8'],
        ['AFF4','SLC38A4'],
        ['AFM','ALB'],
        ['AFM','FBF1'],
        ['AFP','ALB'],
        ['AFP','CUX1'],
        ['AFTPH','PRKCE'],
        ['AFX','PAX3'],
        ['AFX1','FOXO4'],
        ['AGAP1','IGFBP2'],
        ['AGAP2','CYP27B1'],
        ['AGAP2','MARS'],
        ['AGAP2','TIE1'],
        ['AGAP2','TMEM67'],
        ['AGAP2','TSFM'],
        ['AGAP3','BRAF'],
        ['AGAP3','HOXD11'],
        ['AGAP3','JAZF1'],
        ['AGAP5','BMS1P4'],
        ['AGBL1','MAGI1'],
        ['AGBL1','NTRK3'],
        ['AGBL3','SLC6A3'],
        ['AGBL4','NTRK1'],
        ['AGBL4','NTRK2'],
        ['AGBL4','NTRK3'],
        ['AGFG1','NTRK1'],
        ['AGGF1','PDGFRB'],
        ['AGGF1','RAF1'],
        ['AGK','BRAF'],
        ['AGK','KIAA1147'],
        ['AGK','WDR60'],
        ['AGL','C9ORF25'],
        ['AGMO','MAP3K1'],
        ['AGO2','PTK2'],
        ['AGO3','SPDYA'],
        ['AGPAT5','MCPH1'],
        ['AGPAT5','MSRA'],
        ['AGT','CLU'],
        ['AGTPBP1','DAPK1'],
        ['AGTPBP1','NAA35'],
        ['AGTRAP','BRAF'],
        ['AGXT2L2','CYB5R1'],
        ['AHCTF1','NAAA'],
        ['AHCY','ITCH'],
        ['AHCY','TAX1BP1'],
        ['AHCYL1','FGFR2'],
        ['AHCYL1','KDM5B'],
        ['AHCYL1','RAD51C'],
        ['AHCYL2','MKLN1'],
        ['AHCYL2','PAX5'],
        ['AHCYL2','TMEM178B'],
        ['AHDC1','MACF1'],
        ['SS18L1','SSX1'],
        ['EWSR1','SSX1'],
        ['SS18','SSX2'],
        ['MN1','SSX1'],
        ['AHDC1','MAP3K6'],
        ['AHDC1','WDFY3'],
        ['AHI1','HBS1L'],
        ['AHI1','MYB'],
        ['AHI1','TRB@'],
        ['AHNAK','HNRNPUL2'],
        ['AHNAK','KAT5'],
        ['AHNAK','MALAT1'],
        ['AHNAK','NXF1'],
        ['AHNAK','RPS6KA4'],
        ['AHNAK2','PLD4'],
        ['AHRR','ETV4'],
        ['AHRR','NCOA2'],
        ['AHRR','NCOA3'],
        ['AHSG','PARP15'],
        ['AHSG','SERPINA1'],
        ['AIDA','KCNH1'],
        ['AIF1','CD163'],
        ['AIF1L','ETV6'],
        ['AIFM1','FGF13'],
        ['AIG1','GOSR1'],
        ['AIG1','NFIB'],
        ['AIM1','SOBP'],
        ['AIP','ATE1'],
        ['AK125726','ETV6'],
        ['AK2','AK8'],
        ['AK2','CD274'],
        ['AK2','PDCD1LG2'],
        ['AK7','CREBRF'],
        ['AK7','EXOC2'],
        ['AK7','PAPOLA'],
        ['AKAP1','RNF43'],
        ['AKAP11','MRPS31'],
        ['AKAP12','ESR1'],
        ['AKAP13','GSN'],
        ['AKAP13','IGF1R'],
        ['AKAP13','NEDD4'],
        ['AKAP13','NRG1'],
        ['AKAP13','NTRK3'],
        ['AKAP13','PDE8A'],
        ['AKAP13','PLDN'],
        ['AKAP13','RET'],
        ['AKAP6','EGLN3'],
        ['AKAP6','ERBB4'],
        ['AKAP6','LEPR'],
        ['AKAP6','NPAS3'],
        ['AKAP6','PRKD1'],
        ['AKAP7','ESR1'],
        ['AKAP8','BRD4'],
        ['AKAP8','CACNA1A'],
        ['AKAP8L','BRD4'],
        ['AKAP8L','ENO1'],
        ['CSF1','CD96'],
        ['RAB3IP','HMGA2'],
        ['SEC16A','KMT2A'],
        ['PAX5','KIDINS220'],
        ['LAMTOR1','AKT1'],
        ['NRIP1','MIR99AHG'],
        ['SPON1','TRIM29'],
        ['MYB','TYK2'],
        ['EWSR1','CTBP1'],
        ['MALAT1','GLI1'],
        ['SPON1','TRIM29'],
        ['CTLA4','CD28'],
        ['AGK','BRAF'],
        ['ZNF532','NUTM1'],
        ['TBLR1','RARA'],
        ['AKAP8L','FGFR1'],
        ['AKAP9','ANKIB1'],
        ['AKAP9','BRAF'],
        ['AKAP9','CDK6'],
        ['AKAP9','KIAA1549'],
        ['AKAP9','LSM14A'],
        ['AKAP9','PON2'],
        ['AKAP9','RUNX1'],
        ['AKAP9','SHFM1'],
        ['AKIRIN1','MACF1'],
        ['AKR1A1','PRDX1'],
        ['AKR1C6P','B4GALT5'],
        ['AKR1D1','BEND7'],
        ['AKR1D1','RIOK1'],
        ['AKR1E2','CCT4P1'],
        ['AKR7A2','LMBRD2'],
        ['AKT1','CPSF2'],
        ['AKT1','LAMTOR1'],
        ['AKT1','PGC'],
        ['AKT1','PHRF1'],
        ['AKT1','PRKACA'],
        ['AKT2','ART3'],
        ['AKT2','BCAM'],
        ['AKT2','C19ORF12'],
        ['AKT2','C19ORF47'],
        ['AKT2','CXCL11'],
        ['AKT2','PAK4'],
        ['AKT2','PRKACA'],
        ['AKT2','SRRM5'],
        ['AKT2','TSHZ3'],
        ['AKT2','YIF1B'],
        ['AKT2','ZNF576'],
        ['AKT3','ARHGAP30'],
        ['AKT3','C1ORF150'],
        ['AKT3','CEP170'],
        ['AKT3','CNOT6L'],
        ['AKT3','MAGI3'],
        ['AKT3','PHIP'],
        ['AKT3','PLD5'],
        ['AKT3','PVT1'],
        ['AKT3','SMYD3'],
        ['AKT3','SSH2'],
        ['AKT3','TAF6L'],
        ['AKT3','XPNPEP1'],
        ['AKT3','ZEB2'],
        ['AL034550.1','PAX5'],
        ['AL035685.1','CSE1L'],
        ['AL121790.1','TTC6'],
        ['AL135923.2','DMAC1'],
        ['AL445647.1','MIR4500HG'],
        ['AL672183.2','CD55'],
        ['AL672183.2','IRF2BP2'],
        ['ALB','APOA2'],
        ['ALB','H19'],
        ['ALB','IGF2'],
        ['ALB','SEPP1'],
        ['ALCAM','RAF1'],
        ['ALDH16A1','PLEKHG5'],
        ['ALDH18A1','DEPDC5'],
        ['ALDH1A3','PTPRG'],
        ['ALDH2','CCNB1IP1'],
        ['ALDH2','CMKOR1'],
        ['ALDH2','EBF1'],
        ['ALDH2','FHIT'],
        ['ALDH2','HMGA2'],
        ['ALDH2','HMGIC'],
        ['ALDH2','WIF1'],
        ['ALDH7A1','SLC9A7'],
        ['ALDOB','KNG1'],
        ['ALG14','BCAR3'],
        ['ALG14','CNN3'],
        ['ALG5','PIGU'],
        ['ALG5','POSTN'],
        ['ALG6','ERBB2'],
        ['ALG8','KHDC1'],
        ['ALG9','PPP2R1B'],
        ['ALK','ALO17'],
        ['ALK','AMBRA1'],
        ['ALK','ANK1'],
        ['ALK','ATG16L1'],
        ['ALK','ATIC'],
        ['ALK','ATRNL1'],
        ['ALK','BCL11A'],
        ['ALK','BEND5'],
        ['ALK','BIRC6'],
        ['ALK','C2ORF22'],
        ['ALK','C2ORF34'],
        ['ALK','C2ORF44'],
        ['ALK','CAD'],
        ['ALK','CARS'],
        ['ALK','CCDC6'],
        ['ALK','CCDC61'],
        ['ALK','CCDC88A'],
        ['ALK','CDK15'],
        ['ALK','CEBPZ'],
        ['ALK','CHRNA7'],
        ['ALK','CLIP2'],
        ['ALK','CLTA'],
        ['ALK','CLTC'],
        ['ALK','CLTCL'],
        ['ALK','CLTCL1'],
        ['ALK','CMTR1'],
        ['ALK','COL1A2'],
        ['ALK','CTLC'],
        ['ALK','CTNNA1'],
        ['ALK','CUX1'],
        ['ALK','DCTN1'],
        ['ALK','DDX6'],
        ['ALK','DES'],
        ['ALK','DIAPH2'],
        ['ALK','DYSF'],
        ['ALK','EEF1G'],
        ['ALK','EEF2'],
        ['ALK','EGFR'],
        ['ALK','EIF2AK3'],
        ['ALK','EIF2AK4'],
        ['ALK','EML4'],
        ['ALK','ERC1'],
        ['ALK','ETV6'],
        ['ALK','FN1'],
        ['ALK','GALM'],
        ['ALK','GALNT17'],
        ['ALK','GCC2'],
        ['ALK','GFPT1'],
        ['ALK','GPC1'],
        ['ALK','GTF2IRD1'],
        ['ALK','HIP'],
        ['ALK','HIP1'],
        ['ALK','HNRNPA1'],
        ['ALK','HNRNPH3'],
        ['ALK','HOOK1'],
        ['ALK','HPCAL1'],
        ['ALK','IGFBP5'],
        ['ALK','IKBKG'],
        ['ALK','ITGAV'],
        ['ALK','KANK1'],
        ['ALK','KANK2'],
        ['ALK','KCNQ1'],
        ['ALK','KCNQ5'],
        ['ALK','KIAA1618'],
        ['ALK','KIF5B'],
        ['ALK','KLC1'],
        ['ALK','KMT2A'],
        ['ALK','KRAS'],
        ['ALK','KTN1'],
        ['ALK','LINC00211'],
        ['ALK','LMNA'],
        ['ALK','LRRFIP1'],
        ['ALK','MAD1L1'],
        ['ALK','MALAT1'],
        ['ALK','MAP2'],
        ['ALK','MCFD2'],
        ['ALK','MET'],
        ['ALK','MIR4432'],
        ['ALK','MLF1'],
        ['ALK','MLPH'],
        ['ALK','MPRIP'],
        ['ALK','MSI2'],
        ['ALK','MSN'],
        ['ALK','MYH9'],
        ['ALK','MYO5A'],
        ['ALK','MZT2B'],
        ['ALK','NCK2'],
        ['ALK','NCOA1'],
        ['ALK','NPM'],
        ['ALK','NPM1'],
        ['ALK','NSF'],
        ['ALK','NTRK1'],
        ['ALK','PARD3'],
        ['ALK','PICALM'],
        ['ALK','PLB1'],
        ['ALK','PLEKHA7'],
        ['ALK','PLEKHH2'],
        ['ALK','PLEKHM2'],
        ['ALK','PPFIBP1'],
        ['ALK','PPM1B'],
        ['ALK','PPP1CB'],
        ['ALK','PPP4R3B'],
        ['ALK','PQLC3'],
        ['ALK','PRKAR1A'],
        ['ALK','PRKAR2A'],
        ['ALK','PRRC2B'],
        ['ALK','PTPN3'],
        ['ALK','RAB10'],
        ['ALK','RANBP1'],
        ['ALK','RANBP2'],
        ['ALK','RARA'],
        ['ALK','RET'],
        ['ALK','RNF213'],
        ['ALK','RNPC3'],
        ['ALK','ROS1'],
        ['ALK','RPA1'],
        ['ALK','RRBP1'],
        ['ALK','SEC31A'],
        ['ALK','SEC31L1'],
        ['ALK','SLMAP'],
        ['ALK','SMEK2'],
        ['ALK','SMPD3'],
        ['ALK','SOCS5'],
        ['ALK','SOS1'],
        ['ALK','SPDYA'],
        ['ALK','SPECC1L'],
        ['ALK','SPTBN1'],
        ['ALK','SQSTM1'],
        ['ALK','SRBD1'],
        ['ALK','SRD5A2'],
        ['ALK','STK32B'],
        ['ALK','STPG4'],
        ['ALK','STRN'],
        ['ALK','SYN3'],
        ['ALK','SYNDIG1L'],
        ['ALK','TACR1'],
        ['ALK','TERT'],
        ['ALK','TFE3'],
        ['ALK','TFG'],
        ['ALK','TGFB1'],
        ['ALK','THBS1'],
        ['ALK','TIMP3'],
        ['ALK','TMED2'],
        ['ALK','TMP3'],
        ['ALK','TMP4'],
        ['ALK','TNIP2'],
        ['ALK','TNS1'],
        ['ALK','TP53'],
        ['ALK','TPM1'],
        ['ALK','TPM3'],
        ['ALK','TPM4'],
        ['ALK','TPR'],
        ['ALK','TRAF1'],
        ['ALK','TRIM33'],
        ['ALK','TRMT61B'],
        ['ALK','VCL'],
        ['ALK','VIT'],
        ['ALK','WDPCP'],
        ['ALK','ZC3H7A'],
        ['ALK','ZEB2'],
        ['ALK5','TFG'],
        ['ALKBH5','LLGL1'],
        ['ALMS1','MRTFB'],
        ['ALO17','MYC'],
        ['ALO17','RNF213'],
        ['ALOX12','MINK1'],
        ['ALOX15B','ETV1'],
        ['ALOXE3','ETV6'],
        ['ALOXE3','TP53'],
        ['ALPK1','ANK2'],
        ['ALPK2','PRKCH'],
        ['ALS2CL','ITGB4'],
        ['ALS2CR16','ALS2CR8'],
        ['ALX3','RECK'],
        ['AMACR','ERG'],
        ['AMACR','FGFR1'],
        ['AMACR','SDK1'],
        ['AMACR','SLC45A2'],
        ['AMD1','CDK19'],
        ['AMD1','KIAA1919'],
        ['AMDHD2','ATP6V0C'],
        ['AMDHD2','TRIM77'],
        ['AMER3','ANKRD42'],
        ['AMFR','IGF2BP2'],
        ['AMH','SF3A2'],
        ['AML1','CBFA2T3'],
        ['AML1','CLCA2'],
        ['AML1','EAP'],
        ['AML1','ETO'],
        ['AML1','ETV6'],
        ['AML1','EVI1'],
        ['AML1','FOG2'],
        ['AML1','JAK2'],
        ['AML1','LAF4'],
        ['AML1','LRP16'],
        ['AML1','MDS1'],
        ['AML1','MTG16'],
        ['AML1','MTG8'],
        ['AML1','PRDM16'],
        ['AML1','PRDX4'],
        ['AML1','PRPF38A'],
        ['AML1','RUNX1'],
        ['AML1','TEL'],
        ['AML1','USP25'],
        ['AMN1','C12ORF35'],
        ['AMOTL2','NTRK1'],
        ['AMPD1','BCAS3'],
        ['AMPD2','FGFR2'],
        ['AMPD3','CCND3'],
        ['AMPH','FDFT1'],
        ['AMPH','ST7'],
        ['AMPK','SIRT1'],
        ['AMZ2','HTRA1'],
        ['ANAPC1','RGPD3'],
        ['ANAPC1','RMND5A'],
        ['ANAPC2','EHMT1'],
        ['ANAPC4','SLC34A2'],
        ['ANAPC5','CAMKK2'],
        ['ANAPC7','SLC2A13'],
        ['ANGPT1','PEAK1'],
        ['ANGPT1','PTK2'],
        ['ANGPTL2','USP6'],
        ['ANGPTL6','GOLGA2P2Y'],
        ['ANGPTL6','KRI1'],
        ['ANK1','FGFR1'],
        ['ANK1','RTN3'],
        ['ANK1','ZMAT4'],
        ['ANK2','ARHGAP10'],
        ['ANK2','CAMK2D'],
        ['ANK2','RRN3P2'],
        ['ANK3','CCDC6'],
        ['ANK3','LDLRAD3'],
        ['ANK3','RET'],
        ['ANKDD1A','SPG21'],
        ['ANKFY1','NLRP1'],
        ['ANKFY1','ZZEF1'],
        ['ANKH','FOS'],
        ['ANKHD1','C5ORF32'],
        ['ANKHD1','PCDH1'],
        ['ANKIB1','LMTK2'],
        ['ANKIB1','PPP6R2'],
        ['ANKIB1','TRRAP'],
        ['ANKRD10','TMPRSS2'],
        ['ANKRD11','DPEP1'],
        ['ANKRD11','EFTUD2'],
        ['ANKRD11','FANCA'],
        ['ANKRD11','KIT'],
        ['ANKRD11','PKD1L3'],
        ['ANKRD11','PTK2'],
        ['ANKRD11','UBQLN1'],
        ['ANKRD12','TWSG1'],
        ['ANKRD13A','FLJ40142'],
        ['ANKRD17','WDFY3'],
        ['ANKRD18B','SIN3A'],
        ['ANKRD23','ANKRD39'],
        ['ANKRD23','CNNM3'],
        ['ANKRD24','NOVA1'],
        ['ANKRD26','RET'],
        ['ANKRD26','WAC'],
        ['ANKRD26','ZNF354C'],
        ['ANKRD27','RHPN2'],
        ['ANKRD27','ZNF790'],
        ['ANKRD28','BCR'],
        ['ANKRD28','FLT3'],
        ['ANKRD28','FNDC3B'],
        ['ANKRD28','NSD1'],
        ['ANKRD28','NUP98'],
        ['ANKRD28','RAF1'],
        ['ANKRD28','STK38'],
        ['ANKRD29','PDS5A'],
        ['ANKRD29','ROCK1'],
        ['ANKRD34A','ATP5O'],
        ['ANKRD34C','EIF2AK4'],
        ['ANKRD42','SRPK1'],
        ['ANKRD65','MMP25'],
        ['ANKRD66','MOK'],
        ['ANKS1A','PPARD'],
        ['ANKS1A','UHRF1BP1'],
        ['ANKS1B','CCDC53'],
        ['ANKS1B','SCYL2'],
        ['ANKS1B','STK24'],
        ['ANKS3','GJB7'],
        ['ANKUB1','WWTR1'],
        ['ANO1','PLA2G16'],
        ['ANO1','ZNF331'],
        ['ANO4','GALNT9'],
        ['ANO4','KIF21A'],
        ['ANO4','TMTC3'],
        ['ANO5','DNAJC24'],
        ['ANO5','RPS6KB2'],
        ['ANO6','DBX2'],
        ['ANO9','CARS'],
        ['ANOS2P','TMSB4Y'],
        ['ANP32A','IGF1R'],
        ['ANP32B','GABBR2'],
        ['ANP32B','SLC35B3'],
        ['ANP32E','SET'],
        ['ANPEP','ERG'],
        ['ANRIL','MTAP'],
        ['ANTXR1','GKN1'],
        ['ANTXR2','IGFBP3'],
        ['ANTXR2','NPLOC4'],
        ['ANTXR2','RNF111'],
        ['ANTXRL','RGS10'],
        ['ANXA1','FNDC1'],
        ['ANXA1','PRKG1'],
        ['ANXA13','PVT1'],
        ['ANXA2','EIF2AK4'],
        ['ANXA2','MYO9A'],
        ['ANXA2','TRIM2'],
        ['ANXA2P2','EIF2AK4'],
        ['ANXA3','GPR107'],
        ['ANXA3','MBNL1'],
        ['ANXA3','UQCRFS1'],
        ['ANXA4','DENND1A'],
        ['ANXA4','PKN1'],
        ['ANXA4','TMEM67'],
        ['ANXA5','CCNA2'],
        ['ANXA6','CPNE5'],
        ['ANXA6','MBLAC1'],
        ['AOAH','RBM28'],
        ['AOF2','C1ORF201'],
        ['AP001931.2','KMT2A'],
        ['AP1G1','PTMS'],
        ['AP1GBP1','TOB1'],
        ['AP1M1','FGFR2'],
        ['AP1M2','HIP1'],
        ['AP1M2','LAG3'],
        ['AP1M2','LINC00662'],
        ['AP1M2','SRP68'],
        ['AP2A1','ZNF544'],
        ['AP2A2','SBF2'],
        ['AP2A2','SBF2-AS1'],
        ['AP2A2','SFMBT1'],
        ['AP2B1','ZNF512'],
        ['AP2M1','SEC63'],
        ['AP3B1','APC'],
        ['AP3B1','BRAF'],
        ['AP3B1','TTTY10'],
        ['AP3S1','COMMD10'],
        ['AP3S1','LMAN2'],
        ['AP4B1','RSBN1'],
        ['AP4B1','VAV3'],
        ['AP4B1-AS1','RSBN1'],
        ['AP4B1-AS1','VAV3'],
        ['AP4E1','CSNK1G1'],
        ['AP5M1','SULT1E1'],
        ['APAF1','TLN1'],
        ['APAF1','UNC13B'],
        ['APBA1','DEGS1'],
        ['APBB1IP','GOLPH3'],
        ['APBB1IP','MICU1'],
        ['APBB1IP','RABEP1'],
        ['APBB1IP','SCARB1'],
        ['APBB3','SRA1'],
        ['APC','DPYSL3'],
        ['APC','EEF1DP3'],
        ['APC','MED20'],
        ['APC','REEP5'],
        ['APC','YTHDC2'],
        ['APC2','C19ORF25'],
        ['APEH','SHISA5'],
        ['APH1B','USP3'],
        ['API2','MALT1'],
        ['API2','MLT1'],
        ['APIP','FGFR2'],
        ['APIP','PVT1'],
        ['APIP','SLC1A2'],
        ['APLP2','CD40'],
        ['APLP2','EEA1'],
        ['APLP2','TNFSF11'],
        ['APMAP','LYN'],
        ['APOA1','APOC3'],
        ['APOA1','RBP4'],
        ['APOA2','APOC2'],
        ['APOA2','FGB'],
        ['APOA2','H19'],
        ['APOA2','OAZ1'],
        ['APOA2','P4HB'],
        ['APOA2','RBP4'],
        ['APOA2','TMEM176B'],
        ['APOA2','XBP1'],
        ['APOA4','APOA5'],
        ['APOBEC2','TMPRSS7'],
        ['APOBEC3B','WEE2'],
        ['APOC1','FGA'],
        ['APOC2','APOC4'],
        ['APOE','SERPINA1'],
        ['APOLD1','RUNX1'],
        ['APOPT1','MARK3'],
        ['APP','C21ORF7'],
        ['APP','FAT1'],
        ['APP','SEC63'],
        ['APPBP2','CUEDC1'],
        ['APPBP2','KIF19'],
        ['APPBP2','PHF20L1'],
        ['APPBP2','PPM1D'],
        ['APPBP2','VMP1'],
        ['APPL2','CIT'],
        ['APRIL','IGH@'],
        ['AQP5','C11ORF85'],
        ['AQP9','TCF12'],
        ['AR','PC3'],
        ['ARCN1','DLG2'],
        ['AREG','EGFR'],
        ['ARF3','PDRG1'],
        ['ARFGAP1','SEC31A'],
        ['ARFGAP2','SLC1A2'],
        ['ARFGAP3','ATM'],
        ['ARFGEF2','CSE1L'],
        ['ARFGEF2','FAM65C'],
        ['ARFGEF2','IGF2R'],
        ['ARFGEF2','ITCH'],
        ['ARFGEF2','SLC9A8'],
        ['ARFGEF2','STAU1'],
        ['ARFGEF2','SULF2'],
        ['ARFGEF2','TMPRSS2'],
        ['ARFIP1','FHDC1'],
        ['ARFIP2','SLC11A1'],
        ['ARGLU1','ENOPH1'],
        ['ARGLU1','TMEM38A'],
        ['ARHGAP1','BMX'],
        ['ARHGAP1','CLDN18'],
        ['ARHGAP1','CPNE3'],
        ['ARHGAP10','CLDN18'],
        ['ARHGAP10','DCLK2'],
        ['ARHGAP10','TMEM184C'],
        ['ARHGAP12','BMX'],
        ['ARHGAP18','TMEM244'],
        ['ARHGAP19','DRG1'],
        ['ARHGAP19','RRP12'],
        ['ARHGAP20','BRWD3'],
        ['ARHGAP21','C15ORF32'],
        ['ARHGAP21','ENKUR'],
        ['ARHGAP21','PIP4K2A'],
        ['ARHGAP21','PRTFDC1'],
        ['ARHGAP24','BANK1'],
        ['ARHGAP24','C4ORF14'],
        ['ARHGAP24','FGD6'],
        ['ARHGAP24','FGFR2'],
        ['ARHGAP24','NENF'],
        ['ARHGAP24','NOA1'],
        ['ARHGAP26','CLDN18'],
        ['ARHGAP26','CTNND1'],
        ['ARHGAP26','KMT2A'],
        ['ARHGAP26','MLL'],
        ['ARHGAP26','NDFIP1'],
        ['ARHGAP26','TP53'],
        ['ARHGAP27','SH3D20'],
        ['ARHGAP29','KDM1A'],
        ['ARHGAP32','TIMM23'],
        ['ARHGAP33','YWHAB'],
        ['ARHGAP35','SMIM7'],
        ['ARHGAP35','UNC13A'],
        ['ARHGAP4','TBC1D27'],
        ['ARHGAP40','CNTF'],
        ['ARHGAP42','CLDN18'],
        ['ARHGAP42','YAP1'],
        ['ARHGAP44','LCE2C'],
        ['ARHGAP6','CLDN18'],
        ['ARHGAP6','MID1'],
        ['ARHGAP9','GLI1'],
        ['ARHGAP9','RUNX2'],
        ['ARHGDIG','ASTN2'],
        ['ARHGEF1','CD79A'],
        ['ARHGEF1','KMT2A'],
        ['ARHGEF10L','PAK4'],
        ['ARHGEF12','KCTD2'],
        ['ARHGEF12','KMT2A'],
        ['ARHGEF12','MLL'],
        ['ARHGEF17','KMT2A'],
        ['ARHGEF17','MLL'],
        ['ARHGEF17','P2RY6'],
        ['ARHGEF17','SHANK2'],
        ['ARHGEF18','INSR'],
        ['ARHGEF2','NTRK1'],
        ['ARHGEF2','NTRK2'],
        ['ARHGEF2','NTRK3'],
        ['ARHGEF2','QSOX2'],
        ['ARHGEF25','B4GALNT1'],
        ['ARHGEF25','NTRK1'],
        ['ARHGEF3','C8ORF38'],
        ['ARHGEF3','PVT1'],
        ['ARHGEF39','NRG1'],
        ['ARHGEF4','IGFBP5'],
        ['ARHGEF4','PLEKHB2'],
        ['ARHGEF5','HIPK2'],
        ['ARHGEF6','ORC4'],
        ['ARHGEF7','CDC16'],
        ['ARHGEF7','EXOC4'],
        ['ARHGEF7','MYO16'],
        ['ARHGEF7','ZDHHC11'],
        ['ARHH','BCL6'],
        ['ARHH','EIF4A2'],
        ['ARHH','SFRS3'],
        ['ARID1A','BEND5'],
        ['ARID1A','DOT1L'],
        ['ARID1A','GPATCH3'],
        ['ARID1A','HOIP'],
        ['ARID1A','MAST2'],
        ['ARID1A','PRKCZ'],
        ['ARID1A','PRKD1'],
        ['ARID1A','RNF31'],
        ['ARID1A','RPS6KA1'],
        ['ARID1A','SLC30A2'],
        ['ARID1A','UBR4'],
        ['ARID1A','WDTC1'],
        ['ARID1B','ESR1'],
        ['ARID1B','KIF26B'],
        ['ARID1B','MRPP3'],
        ['ARID1B','PLAGL1'],
        ['ARID1B','PSMA1'],
        ['ARID1B','PVT1'],
        ['ARID1B','SNX9'],
        ['ARID1B','TFE3'],
        ['ARID1B','TIAM2'],
        ['ARID1B','TRIM68'],
        ['ARID1B','ZNF384'],
        ['ARID2','ASB8'],
        ['ARID2','MRPS30'],
        ['ARID2','TMPRSS2'],
        ['ARID2','TP53'],
        ['ARID4A','RBBP1'],
        ['ARID4B','TARBP1'],
        ['ARID5A','CHD1'],
        ['ARID5B','MLL'],
        ['ARID5B','NFE2L2'],
        ['ARIH2','CAMP'],
        ['ARIH2','PRKAR2A'],
        ['ARL1','CCBL1'],
        ['ARL11','RNASEH2B'],
        ['ARL13A','TRMT2B'],
        ['ARL17A','GSTT2B'],
        ['ARL17A','KANSL1'],
        ['ARL17B','KANSL1'],
        ['ARL3','CAMK1D'],
        ['ARL5A','NEB'],
        ['ARL6','POT1'],
        ['ARL6IP1','RPS15A'],
        ['ARL6IP4','NKX2-1'],
        ['ARL8A','INSRR'],
        ['ARL8B','ITPR1'],
        ['ARMC10','BRAF'],
        ['ARMC4','PATE4'],
        ['ARMC6','DHX57'],
        ['ARMC7','PEX14'],
        ['ARMCX3','FRAS1'],
        ['ARMCX5','B3GNT4'],
        ['ARNT','ETV6'],
        ['ARNT','TEL'],
        ['ARNT2','ESR1'],
        ['ARNT2','MESDC2'],
        ['ARNTL','PTPRD'],
        ['ARNTL2','ETV6'],
        ['ARPC1A','HIPK2'],
        ['ARPC1B','TRRAP'],
        ['ARPC2','TXNDC15'],
        ['ARPC5','SEC31A'],
        ['ARPP19','PSPC1'],
        ['ARRB1','KLHL35'],
        ['ARRDC1','ZMYND19'],
        ['ARSF','SYAP1'],
        ['ARSG','MAP2K6'],
        ['ARSG','PPM1D'],
        ['ART3','CDKL2'],
        ['ART3','GLUL'],
        ['ART3','KCTD8'],
        ['ART3','NUP54'],
        ['ART4','C12ORF69'],
        ['ARTEMIS','DCLRE1C'],
        ['AS3MT','PI4K2A'],
        ['ASAH1','SHROOM3'],
        ['ASAH2','NT5C2'],
        ['ASAP1','PVT1'],
        ['ASAP1','STOM'],
        ['ASAP1','USP6'],
        ['ASAP2','CPSF3'],
        ['ASAP3','USP34'],
        ['ASB11','URM1'],
        ['ASB17','CHST2'],
        ['ASB9','WDR59'],
        ['ASCC1','CBARA1'],
        ['ASCC1','MICU1'],
        ['ASCC1','VCL'],
        ['ASCC2','ZMAT5'],
        ['ASCC3','DST'],
        ['ASCC3','UBE2E1'],
        ['ASGR2','NDEL1'],
        ['ASGR2','TAOK1'],
        ['ASH1L','C1ORF61'],
        ['ASH1L','RSPO3'],
        ['ASH1L','VANGL2'],
        ['ASH1L','YY1AP1'],
        ['ASH2L','CASS4'],
        ['ASIC2','NF1'],
        ['ASIP','ITCH'],
        ['ASL','CRCP'],
        ['ASNA1','RFX1'],
        ['ASNA1','WDR20'],
        ['ASNS','KMT2E'],
        ['ASPH','ESR1'],
        ['ASPH','MRPS28'],
        ['ASPL','TFE3'],
        ['ASPS','TFE3'],
        ['ASPSCR1','CLTC'],
        ['ASPSCR1','CSNK1D'],
        ['ASPSCR1','NONO'],
        ['ASPSCR1','PRCC'],
        ['ASPSCR1','SFPQ'],
        ['ASPSCR1','TFE3'],
        ['ASTN1','PARP1'],
        ['ASTN1','PLXNA2'],
        ['ASTN1','TRIO'],
        ['ASTN2','PAPPA'],
        ['ASTN2','PDIA2'],
        ['ASTN2','PTPRG'],
        ['ASTN2','TBC1D16'],
        ['ASUN','RASSF8'],
        ['ASXL1','BPIL3'],
        ['ASXL1','C20ORF112'],
        ['ASXL1','PAX5'],
        ['ASXL1','PFDN4'],
        ['ASXL1','RUNX1'],
        ['ASXL1','SNX14'],
        ['ASXL2','EPC1'],
        ['ASXL2','KAT6A'],
        ['ASXL2','KMT2A'],
        ['ASZ1','VSTM2L'],
        ['ATAD1','PTEN'],
        ['ATAD1','SLIT1'],
        ['ATAD1','STAMBPL1'],
        ['ATAD2','FAM135B'],
        ['ATAD3A','ATAD3B'],
        ['ATAD3B','ATAD3C'],
        ['ATAD3B','NIPAL3'],
        ['ATAD5','CDK12'],
        ['ATAD5','TLK2'],
        ['ATE1','FGFR2'],
        ['ATE1','HTRA1'],
        ['ATE1','PVT1'],
        ['ATF1','CREB1'],
        ['ATF1','CREB3L1'],
        ['ATF1','CREB3L2'],
        ['ATF1','DDIT3'],
        ['ATF1','ERG'],
        ['ATF1','ETV1'],
        ['ATF1','ETV4'],
        ['ATF1','EWS'],
        ['ATF1','EWSR1'],
        ['ATF1','FUS'],
        ['ATF1','NFATC2'],
        ['ATF1','PBX1'],
        ['ATF1','POU5F1'],
        ['ATF1','WT1'],
        ['ATF1','ZNF384'],
        ['ATF3','PFKFB3'],
        ['ATF3','VASH2'],
        ['ATF5','PRKDC'],
        ['ATF6','FAM78B'],
        ['ATF7','STK24'],
        ['ATF7IP','GNPTAB'],
        ['ATF7IP','JAK2'],
        ['ATF7IP','PDGFRB'],
        ['ATF7IP','TMPRSS2'],
        ['ATF7IP','USP32'],
        ['ATG10','EMB'],
        ['ATG14','FUT8'],
        ['ATG16L1','SHIP1'],
        ['ATG16L2','KMT2A'],
        ['ATG16L2','LRP5'],
        ['ATG4A','PSMD10'],
        ['ATG4B','DTYMK'],
        ['ATG4C','FBXO38'],
        ['ATG4D','C4ORF22'],
        ['ATG5','EIF3E'],
        ['ATG5','PDSS2'],
        ['ATG7','BRAF'],
        ['ATG7','RAF1'],
        ['ATL2','LINC00330'],
        ['ATL3','TNKS'],
        ['ATM','ATR'],
        ['ATM','GRIA4'],
        ['ATM','LRMP'],
        ['ATM','SLC35F2'],
        ['ATN1','KDM5A'],
        ['ATP10A','C12ORF49'],
        ['ATP10B','EBF1'],
        ['ATP10B','LINC02159'],
        ['ATP11A','ING1'],
        ['ATP11A','ULK4'],
        ['ATP11B','GSK3B'],
        ['ATP13A3','XXYLT1'],
        ['ATP1A1','NOTCH2'],
        ['ATP1A1','RASGRP1'],
        ['ATP1A1','ZFP64'],
        ['ATP1A2','CSDE1'],
        ['ATP1A2','DCAF8'],
        ['ATP1B','NTRK1'],
        ['ATP1B1','NRG1'],
        ['ATP1B1','NTRK1'],
        ['ATP1B1','PRKACA'],
        ['ATP1B1','PRKACB'],
        ['ATP1B2','COPZ1'],
        ['ATP1B3','GRK7'],
        ['ATP1B3','PRKAA2'],
        ['ATP1B4','CHST11'],
        ['ATP2B2','ESR1'],
        ['ATP2C1','TANC2'],
        ['ATP5A1','ZNF146'],
        ['ATP5C1','ZNF384'],
        ['ATP5E','HIST1H1E'],
        ['ATP5I','MFSD7'],
        ['ATP5I','SLK'],
        ['ATP5L','KMT2A'],
        ['ATP5L','MLLT10'],
        ['ATP5L','TEAD1'],
        ['ATP5O','TIAM1'],
        ['ATP6V0A1','MB'],
        ['ATP6V0D2','STIP1'],
        ['ATP6V1C1','RIMS2'],
        ['ATP6V1C1','VEGFB'],
        ['ATP7B','FNDC3A'],
        ['ATP8A1','RASGEF1B'],
        ['ATP8A2','FLT1'],
        ['ATP8A2','LATS2'],
        ['ATP8A2','RUNX1'],
        ['ATP8B2','IL6R'],
        ['ATP8B4','ATRX'],
        ['ATP8B5P','PABPC1L2B-AS1'],
        ['ATP9B','ATRX'],
        ['ATP9B','FGFR2'],
        ['ATP9B','LUC7L'],
        ['ATPSCKMT','DAP'],
        ['ATR','CHK1'],
        ['ATR','DBR1'],
        ['ATR','DLG1'],
        ['ATR','EIF2AK1'],
        ['ATR','NMRK1'],
        ['ATR','TOP3B'],
        ['ATR','WWTR1'],
        ['ATRN','ESYT2'],
        ['ATRNL1','EML4'],
        ['ATRX','BCL2'],
        ['ATRX','DACH2'],
        ['ATRX','DAXX'],
        ['ATRX','EGFR'],
        ['ATRX','MAGT1'],
        ['ATRX','PASD1'],
        ['ATRX','RPS6KA6'],
        ['ATRX','STAG2'],
        ['ATRX','ZNF304'],
        ['ATXN1','DNM3'],
        ['ATXN1','DUX4'],
        ['ATXN1','NOTCH4'],
        ['ATXN1','NUTM1'],
        ['ATXN1','SLC24A1'],
        ['ATXN1','TP63'],
        ['ATXN2','RPL6'],
        ['ATXN2L','JAK2'],
        ['ATXN3','THAP11'],
        ['ATXN7','BCAS3'],
        ['ATXN7','FHIT'],
        ['ATXN7','OXSR1'],
        ['ATXN7','RAD51C'],
        ['ATXN7L1','MET'],
        ['ATXN7L3','FAM171A2'],
        ['ATXN7L3','GRN'],
        ['AUNIP','HELZ'],
        ['AURKA','BACH2'],
        ['AURKA','BCAS4'],
        ['AURKA','SOGA1'],
        ['AURKAPS1','RAB3GAP2'],
        ['AURKC','PPP6R1'],
        ['AUTS2','BAZ1B'],
        ['AUTS2','GLI3'],
        ['AUTS2','KMT2C'],
        ['AUTS2','PAX5'],
        ['AUTS2','PRPSAP2'],
        ['AVEN','FAM178A'],
        ['AVEN','NEO1'],
        ['AVEN','NUTM1'],
        ['AVIL','MARS'],
        ['AVIL','PPM1H'],
        ['AVIL','PVT1'],
        ['AVIL','TMEM120B'],
        ['AVIL','TSFM'],
        ['AVIL','VSNL1'],
        ['AXDND1','RNF2'],
        ['AXDND1','UBE2T'],
        ['AXIN1','LMF1'],
        ['AXIN1','SMPD3'],
        ['AXIN1','TGFBI'],
        ['AXL','MBIP'],
        ['AXL','ZNF565'],
        ['AZGP1','GJC3'],
        ['AZIN1','FGFR1'],
        ['AZIN1','PRPF6'],
        ['B2M','HBB'],
        ['B2M','IGFBP5'],
        ['B3GALNT2','GNG4'],
        ['B3GALNT2','TOMM20'],
        ['B3GALT5','FAM217A'],
        ['B3GALTL','STK33'],
        ['B3GAT3','CAPN5'],
        ['B3GNT1','NPSR1'],
        ['B3GNT9','DSCR8'],
        ['B3GNTL1','ERBB2'],
        ['B3GNTL1','METRNL'],
        ['B3GNTL1','SLC9A8'],
        ['B4GALNT1','FAM19A2'],
        ['B4GALNT1','SLC26A10'],
        ['B4GALNT1','TSFM'],
        ['B4GALNT2','THRA'],
        ['SKAP1-AS2','THRA'],
        ['B4GALNT3','DENND5B'],
        ['B4GALNT3','WNK1'],
        ['UBTF','ATXN7L3'],
        ['HMGA2','NCOR2'],
        ['WNK1','PDGFRB'],
        ['QKI','NTRK2'],
        ['NACC2','NTRK2'],
        ['AGBL4','NTRK2'],
        ['TPM3','NTRK1'],
        ['ETV6','NTRK3'],
        ['KIAA1549','BRAF'],
        ['MN1','BEND2'],
        ['PPP1CB','ALK'],
        ['EML4','ALK'],
        ['HIP1','ALK'],
        ['PRKAR2A','ALK'],
        ['SPTBN1','ALK'],
        ['MAD1L1','ALK'],
        ['MAP2','ALK'],
        ['MSI2','ALK'],
        ['SPECC1L','ALK'],
        ['SYNDIG1L','ALK'],
        ['ZC3H7A','ALK'],
        ['CLIP2','ALK'],
        ['KCTD16','NTRK2'],
        ['AGBL4','NTRK2'],
        ['PPP1CB','ALK'],
        ['MAD1L1','ALK'],
        ['KTN1','ALK'],
        ['CCDC88A','ALK'],
        ['ZCCHC8','ROS1'],
        ['B4GALT1','BAG1'],
        ['B4GALT1','CSPP1'],
        ['B4GALT1','MRPS18A'],
        ['B4GALT2','JAK1'],
        ['B4GALT2','ZSWIM5'],
        ['B4GALT5','ERBB4'],
        ['B4GALT5','TERT'],
        ['B7.1','B7.2'],
        ['BAALC','GTF2E2'],
        ['BACE2','CTSB'],
        ['BACE2','ERG'],
        ['BACE2','TMPRSS2'],
        ['RCC1','LCK'],
        ['WWTR1','AFF2'],
        ['CHD4','AFF2'],
        ['DEK','AFF2'],
        ['WWTR1','KDM2A'],
        ['BACH2','BCL2L1'],
        ['BACH2','IGH@'],
        ['BACH2','METTL24'],
        ['BACH2','RNGTT'],
        ['BAF45D','DPF2'],
        ['BAF60B','SMARCD2'],
        ['BAG2','ZNF451'],
        ['BAG4','FGFR1'],
        ['BAG6','C6ORF47'],
        ['BAG6','SLC44A4'],
        ['BAGE2','UBA52'],
        ['BAHD1','IVD'],
        ['BAI2','HS3ST4'],
        ['BAI3','HMGCLL1'],
        ['BAIAP2','DENR'],
        ['BAIAP2','SLC38A10'],
        ['BAIAP2','TMEM66'],
        ['BAIAP2L1','BRAF'],
        ['BAIAP2L1','FGFR3'],
        ['BAIAP2L1','MET'],
        ['BAIAP2L2','SLC16A8'],
        ['BANF1','EIF1AD'],
        ['BANF2','RBM43'],
        ['BANK1','ELMO1'],
        ['BANK1','IGH@'],
        ['BANP','NTRK1'],
        ['BAP1','CCDC66'],
        ['BAP1','PBRM1'],
        ['BAP1','TP53'],
        ['BARD1','FN1'],
        ['BASP1','PVT1'],
        ['BAT3','SLC44A4'],
        ['BATF2','FARS2'],
        ['BATF2','MARK2'],
        ['BAZ1A','NKX2-1'],
        ['BAZ1A','NUMB'],
        ['BAZ1A','RNF115'],
        ['BAZ1A','SFTA3'],
        ['BAZ1A','SNX6'],
        ['BAZ1A','ZNF253'],
        ['BAZ1B','GRID1'],
        ['BAZ2A','ETV6'],
        ['BAZ2A','FSIP1'],
        ['BAZ2A','LRP1'],
        ['BAZ2A','METTL1'],
        ['BBF2H7','FUS'],
        ['BBIP1','RET'],
        ['BBS10','FGFR3'],
        ['BBS2','OGFOD1'],
        ['BBS9','BRAF'],
        ['BBS9','PKD1L1'],
        ['BCAN','FOXJ2'],
        ['BCAN','NTRK1'],
        ['BCAN','NTRK2'],
        ['BCAN','NTRK3'],
        ['BCAP29','BRAF'],
        ['BCAP29','DUS4L'],
        ['BCAP31','TEX28'],
        ['BCAR1','RBM6'],
        ['BCAR1','ZC3H7A'],
        ['BCAR4','CD63'],
        ['BCAR4','CLEC16A'],
        ['BCAS1','BRAF'],
        ['BCAS1','ZBP1'],
        ['BCAS2','IGH@'],
        ['BCAS2P1','PCF11'],
        ['BCAS3','BCAS4'],
        ['BCAS3','CLTC'],
        ['BCAS3','DPY19L2P1'],
        ['BCAS3','DUSP14'],
        ['BCAS3','ERBB2'],
        ['BCAS3','HOXB9'],
        ['BCAS3','MAP3K3'],
        ['BCAS3','MED13'],
        ['BCAS3','NF1'],
        ['BCAS3','NLK'],
        ['BCAS3','PARD6B'],
        ['BCAS3','PRKAA2'],
        ['BCAS3','RARA'],
        ['BCAS3','RSBN1'],
        ['BCAS3','TAOK1'],
        ['BCAS3','TG'],
        ['BCAS3','TRIM37'],
        ['BCAS3','UBE3C'],
        ['BCAS4','C16ORF97'],
        ['BCAS4','PRKCBP1'],
        ['BCAS4','ZMYND8'],
        ['BCAT1','C2CD5'],
        ['BCAT1','FIP1L1'],
        ['BCAT1','MLL'],
        ['BCHE','MTERFD3'],
        ['BCKDHA','TMEM91'],
        ['BCKDHB','PTPRK'],
        ['BCKDHB','RIPK1'],
        ['BCKDHB','SEC31A'],
        ['BCKDHB','TTK'],
        ['BCL1','CCND1'],
        ['BCL1','IGH@'],
        ['BCL10','BCL11A'],
        ['BCL10','BCL2'],
        ['BCL10','C17ORF76-AS1'],
        ['BCL10','CRLF2'],
        ['BCL10','DDX6'],
        ['BCL10','IGH@'],
        ['BCL10','IGK@'],
        ['BCL10','LHX4'],
        ['BCL10','NFKB2'],
        ['BCL10','PAFAH1B2'],
        ['BCL10','PCSK7'],
        ['BCL11A','GRIP2'],
        ['BCL11A','IGH@'],
        ['BCL11A','KLF1'],
        ['BCL11A','REL'],
        ['BCL11B','HELIOS'],
        ['BCL11B','IKZF2'],
        ['BCL11B','NKX2-5'],
        ['BCL11B','RUNX1'],
        ['BCL11B','TLX3'],
        ['BCL11B','TLX3_'],
        ['BCL11B','TRD@'],
        ['BCL11B','TRDC'],
        ['BCL11B','TRDV1'],
        ['BCL11B','ZEB2'],
        ['BCL2','BCL3'],
        ['BCL2','BCL6'],
        ['BCL2','BOB1'],
        ['BCL2','CD10'],
        ['BCL2','IGH@'],
        ['BCL2','IGHG1'],
        ['BCL2','IGK@'],
        ['BCL2','IGL@'],
        ['BCL2','MYC'],
        ['BCL2','RTTN'],
        ['BCL2','TRAF3'],
        ['BCL2A1','IGH@'],
        ['BCL2L1','BMP8B'],
        ['BCL2L1','DEM1'],
        ['BCL2L1','HM13'],
        ['BCL2L1','PPT1'],
        ['BCL2L1','RIMS3'],
        ['BCL2L1','RLF'],
        ['BCL2L1','ZNF643'],
        ['BCL2L1','ZNF684'],
        ['BCL2L11','BRAF'],
        ['BCL2L13','SLC39A9'],
        ['BCL2L14','ETV6'],
        ['BCL2L2','PABPN1'],
        ['BCL3','IGH@'],
        ['BCL3','IGK@'],
        ['BCL3','IGL@'],
        ['BCL3','MYC'],
        ['BCL3','RELB'],
        ['BCL5','BCL7A'],
        ['BCL5','BTG1'],
        ['BCL5','IGH@'],
        ['BCL5','IGK@'],
        ['BCL5','MYC'],
        ['BCL5','TRA@'],
        ['BCL6','BCL8'],
        ['BCL6','C16ORF75'],
        ['BCL6','CBLB'],
        ['BCL6','CIITA'],
        ['BCL6','DMRT1'],
        ['BCL6','EIF4A2'],
        ['BCL6','EIF4G1'],
        ['BCL6','FBXO18'],
        ['BCL6','GAPDH'],
        ['BCL6','GAS5'],
        ['BCL6','GRHPR'],
        ['BCL6','H4'],
        ['BCL6','H4FM'],
        ['BCL6','HIST1H4I'],
        ['BCL6','HSP90AA1'],
        ['BCL6','HSP90AB1'],
        ['BCL6','HSPCA'],
        ['BCL6','HSPCB'],
        ['BCL6','IGH@'],
        ['BCL6','IGHG3'],
        ['BCL6','IGK@'],
        ['BCL6','IGL@'],
        ['BCL6','IKAROS'],
        ['BCL6','IKZF1'],
        ['BCL6','IL21R'],
        ['BCL6','ITPR2'],
        ['BCL6','LAZ3'],
        ['BCL6','LCP1'],
        ['BCL6','LPP'],
        ['BCL6','LRMP'],
        ['BCL6','MBNL1'],
        ['BCL6','MYC'],
        ['BCL6','NACA'],
        ['BCL6','NAPA'],
        ['BCL6','OBF1'],
        ['BCL6','PIM1'],
        ['BCL6','POU2AF1'],
        ['BCL6','RAB29'],
        ['BCL6','RAF1'],
        ['BCL6','RARA'],
        ['BCL6','RBM6'],
        ['BCL6','RCC1'],
        ['BCL6','RHOH'],
        ['BCL6','SFRS3'],
        ['BCL6','SNHG5'],
        ['BCL6','SRSF3'],
        ['BCL6','ST6GAL1'],
        ['BCL6','TFRC'],
        ['BCL6','TTF'],
        ['BCL6','ZNFN1A1'],
        ['BCL6B','TP53'],
        ['BCL7A','C12ORF42'],
        ['BCL7A','IGH@'],
        ['BCL7A','MYC'],
        ['BCL7A','NCOR2'],
        ['BCL7A','S100A6'],
        ['BCL7B','TNFRSF12A'],
        ['BCL8','BCL9'],
        ['BCL8','IGH@'],
        ['BCL9','CCND1'],
        ['BCL9','CCND2'],
        ['BCL9','CLEC2A'],
        ['BCL9','IGH@'],
        ['BCL9','IGL@'],
        ['BCL9','MEF2D'],
        ['BCL9','MYC'],
        ['BCL9','NTRK1'],
        ['BCL9L','CHRNA9'],
        ['BCL9L','KMT2A'],
        ['BCL9L','MLL'],
        ['BCLAF1','GRM1'],
        ['BCOR','CCNB3'],
        ['BCOR','CREBBP'],
        ['BCOR','CREBRF'],
        ['BCOR','DDX3X'],
        ['BCOR','DMD'],
        ['BCOR','EP300'],
        ['BCOR','EPC1'],
        ['BCOR','ING3'],
        ['BCOR','KMT2A'],
        ['BCOR','KMT2D'],
        ['BCOR','L3MBTL2'],
        ['BCOR','MAML3'],
        ['BCOR','MAP7D2'],
        ['BCOR','NUGGC'],
        ['BCOR','NUTM2G'],
        ['BCOR','PAX5'],
        ['BCOR','RALGPS1'],
        ['BCOR','RARA'],
        ['BCOR','RGAG1'],
        ['BCOR','ZC3H7'],
        ['BCOR','ZC3H7B'],
        ['BCORL1','CREBBP'],
        ['BCORL1','ELF4'],
        ['BCORL1','EP300'],
        ['BCORL1','JAZF1'],
        ['BCORL1','NUTM1'],
        ['BCR','C9ORF78'],
        ['BCR','CEP1'],
        ['BCR','ETV6'],
        ['BCR','EXOSC2'],
        ['BCR','FGFR1'],
        ['BCR','FOP'],
        ['BCR','JAK2'],
        ['BCR','LZTR1'],
        ['BCR','MRVI1'],
        ['BCR','MYH11'],
        ['BCR','MYH9'],
        ['BCR','NDUFAF6'],
        ['BCR','NTRK2'],
        ['BCR','PDGFRA'],
        ['BCR','PDGFRB'],
        ['BCR','PPIL2'],
        ['BCR','PTPRD'],
        ['BCR','RALGPS1'],
        ['BCR','RET'],
        ['BCR','TOM1'],
        ['BCR','UPB1'],
        ['BCR','ZNF198'],
        ['BCR','ZNRF3'],
        ['BCS1L','MLL'],
        ['BDH1','NEK11'],
        ['BDH1','PAK2'],
        ['BDNF','CLTC'],
        ['BDP1','TMPRSS2'],
        ['BEND2','CHD7'],
        ['BEND2','EWSR1'],
        ['BEND2','MAMLD1'],
        ['BEND2','MN1'],
        ['BEND5','NTRK2'],
        ['BEST3','NUP107'],
        ['BEST3','TMTC2'],
        ['BET1','FGFR2'],
        ['BF104016','NSMCE2'],
        ['BFSP2','EPHB1'],
        ['BFSP2','FGFR2'],
        ['BFSP2','PIK3CB'],
        ['BGLAP','PMF1'],
        ['BGN','PODXL'],
        ['BGN','THOC7'],
        ['BGN','TSGA13'],
        ['BHMT','ETV6'],
        ['BICC1','FGFR2'],
        ['BICC1','LDB1'],
        ['BICD2','SBNO1'],
        ['BID','NELFE'],
        ['BIN2','CELA1'],
        ['BIN2','PDGFRB'],
        ['BIRC3','MALT1'],
        ['BIRC3','REXO2'],
        ['BIRC5','MSI2'],
        ['BIRC6','EML4'],
        ['BIRC6','IYD'],
        ['BIRC6','LTBP1'],
        ['BIRC6','WTIP'],
        ['BIVM','FGF14'],
        ['BLIMP1','PRDM1'],
        ['BLM','MAN2A2'],
        ['BLM','ROCK2'],
        ['BLMH','FOXN1'],
        ['BLNK','DNTT'],
        ['BLOC1S1','CBFA2T3'],
        ['BMAL1','CLOCK'],
        ['BMI1','COMMD3'],
        ['BMI1','IGH@'],
        ['BMI1','IGL@'],
        ['BMP2K','CCSER1'],
        ['BMP2K','LEKR1'],
        ['BMP2K','NCOR2'],
        ['BMP2K','SHROOM3'],
        ['BMP2K','SLC25A31'],
        ['BMP2K','SLC4A4'],
        ['BMP2K','ZNF384'],
        ['BMP3','TRAF3'],
        ['BMP6','CENPW'],
        ['BMP6','FARS2'],
        ['BMP6','TXNDC5'],
        ['BMP7','COL22A1'],
        ['BMP8A','ZFP69'],
        ['BMPER','ZNF416'],
        ['BMPR1A','C10ORF11'],
        ['BMPR1A','LIPA'],
        ['BMPR1A','ZYG11A'],
        ['BMPR1B','ETV1'],
        ['BMPR1B','GRM4'],
        ['BMPR1B','NPTX1'],
        ['BMPR1B','NRG1'],
        ['BMPR1B','PAWR'],
        ['BMPR1B','PDLIM5'],
        ['BMPR1B','SMARCAD1'],
        ['BMPR1B','ZMIZ1'],
        ['BMPR2','CADM2'],
        ['BMPR2','FAM117B'],
        ['BMPR2','MARCH4'],
        ['BMPR2','SPATS2L'],
        ['BMPR2','WDTC1'],
        ['BMS1','PARG'],
        ['BMX','RPS6KA3'],
        ['BNC2','ESR1'],
        ['BNC2','LPP'],
        ['BNC2','MTAP'],
        ['BNIP2','MYLK2'],
        ['BOB1','OBF1'],
        ['BOC','CTDSP2'],
        ['BOC','PLAG1'],
        ['BOK','PASK'],
        ['BOLA2B','ZNF813'],
        ['BORCS5','ETV6'],
        ['BORCS5','LMO3'],
        ['BORCS5','RUNX1'],
        ['BPESC1','C6ORF10'],
        ['BPGM','FUCA1'],
        ['BPIFA2','DHX35'],
        ['BPIFA2','PIGU'],
        ['BPTF','LRRC37A2'],
        ['BPTF','METTL2A'],
        ['BPTF','NUP98'],
        ['BPTF','PRKCA'],
        ['BPTF','RARA'],
        ['BRAF','BTF3L4'],
        ['BRAF','C16ORF72'],
        ['BRAF','C7ORF10'],
        ['BRAF','CAPZA2'],
        ['BRAF','CCDC6'],
        ['BRAF','CCDC91'],
        ['BRAF','CCNY'],
        ['BRAF','CDC27'],
        ['BRAF','CDC42SE2'],
        ['BRAF','CDK5RAP2'],
        ['BRAF','CEP89'],
        ['BRAF','CIITA'],
        ['BRAF','CLCN6'],
        ['BRAF','CRAF'],
        ['BRAF','CUL1'],
        ['BRAF','CUX1'],
        ['BRAF','DLG1'],
        ['BRAF','DST'],
        ['BRAF','DYNC1I2'],
        ['BRAF','EGFR'],
        ['BRAF','EML4'],
        ['BRAF','EPB41L2'],
        ['BRAF','EPS15'],
        ['BRAF','ERC1'],
        ['BRAF','ETFA'],
        ['BRAF','FAM114A2'],
        ['BRAF','FAM131A'],
        ['BRAF','FAM131B'],
        ['BRAF','FCHSD1'],
        ['BRAF','FOXN3'],
        ['BRAF','FRY'],
        ['BRAF','FXR1'],
        ['BRAF','GATM'],
        ['BRAF','GHR'],
        ['BRAF','GIPC2'],
        ['BRAF','GIT2'],
        ['BRAF','GNAI1'],
        ['BRAF','GOLGB1'],
        ['BRAF','GTF2I'],
        ['BRAF','HERPUD1'],
        ['BRAF','JHDM1D'],
        ['BRAF','KDM7A'],
        ['BRAF','KIAA1549'],
        ['BRAF','KLHL7'],
        ['BRAF','KRAS'],
        ['BRAF','LMO7'],
        ['BRAF','LSM12'],
        ['BRAF','LSM14A'],
        ['BRAF','MACF'],
        ['BRAF','MACF1'],
        ['BRAF','MAD1L1'],
        ['BRAF','MBNL1'],
        ['BRAF','MBNL2'],
        ['BRAF','MBP'],
        ['BRAF','MKRN1'],
        ['BRAF','MS4A6A'],
        ['BRAF','MTAP'],
        ['BRAF','MYLK'],
        ['BRAF','MYRIP'],
        ['BRAF','MZT1'],
        ['BRAF','NFYA'],
        ['BRAF','NPHP3'],
        ['BRAF','NRF1'],
        ['BRAF','NUB1'],
        ['BRAF','NUDCD3'],
        ['BRAF','NUP214'],
        ['BRAF','OPTN'],
        ['BRAF','OSBPL1A'],
        ['BRAF','OSBPL9'],
        ['BRAF','PAPSS1'],
        ['BRAF','PCBP2'],
        ['BRAF','PHTF2'],
        ['BRAF','PICALM'],
        ['BRAF','PJA2'],
        ['BRAF','POR'],
        ['BRAF','PPFIBP2'],
        ['BRAF','PRKAR1B'],
        ['BRAF','PRKAR2B'],
        ['BRAF','QKI'],
        ['BRAF','RAD18'],
        ['BRAF','RBM33'],
        ['BRAF','RBMS3'],
        ['BRAF','RELN'],
        ['BRAF','RNF130'],
        ['BRAF','ROBO1'],
        ['BRAF','RPS25'],
        ['BRAF','RUNDC1'],
        ['BRAF','SCRIB'],
        ['BRAF','SEPT7'],
        ['BRAF','SKAP2'],
        ['BRAF','SLC12A7'],
        ['BRAF','SLC45A3'],
        ['BRAF','SND1'],
        ['BRAF','SOX6'],
        ['BRAF','SRPRA'],
        ['BRAF','STARD9'],
        ['BRAF','STK35'],
        ['BRAF','STRN3'],
        ['BRAF','STXBP6'],
        ['BRAF','SVOPL'],
        ['BRAF','TANK'],
        ['BRAF','TARDBP'],
        ['BRAF','TAX1BP1'],
        ['BRAF','TLK2'],
        ['BRAF','TMEM106B'],
        ['BRAF','TNS3'],
        ['BRAF','TOM1L2'],
        ['BRAF','TPR'],
        ['BRAF','TRIM22'],
        ['BRAF','TRIM24'],
        ['BRAF','TRIM4'],
        ['BRAF','TRIP11'],
        ['BRAF','TTYH3'],
        ['BRAF','UBN2'],
        ['BRAF','VPS41'],
        ['BRAF','ZBTB8A'],
        ['BRAF','ZC3H7A'],
        ['BRAF','ZC3HAV1'],
        ['BRAF','ZKSCAN1'],
        ['BRAF','ZKSCAN5'],
        ['BRAF','ZNF207'],
        ['BRAF','ZNF767'],
        ['BRAF','ZNF84'],
        ['BRAF','ZSCAN23'],
        ['BRAF','ZSCAN30'],
        ['BRAP','FGFR3'],
        ['BRAP','RBM19'],
        ['BRAP','RCAN1'],
        ['BRCA1','BRCA2'],
        ['BRCA1','DNAJC1'],
        ['BRCA1','FAM134C'],
        ['BRCA1','NBR1'],
        ['BRCA1','VAT1'],
        ['BRCA2','TPP2'],
        ['BRCC3','FUNDC2'],
        ['BRD1','MAPK12'],
        ['BRD1','PAX5'],
        ['BRD1','PIM3'],
        ['BRD2','PIGR'],
        ['BRD3','C15ORF55'],
        ['BRD3','NUT'],
        ['BRD3','NUTM1'],
        ['BRD3','PIP5KL1'],
        ['BRD4','C15ORF55'],
        ['BRD4','CD99L2'],
        ['BRD4','EPHX3'],
        ['BRD4','EPS15L1'],
        ['BRD4','FRMD3'],
        ['BRD4','IGH@'],
        ['BRD4','LEUTX'],
        ['BRD4','NOTCH3'],
        ['BRD4','NUT'],
        ['BRD4','NUTM1'],
        ['BRD4','RARA'],
        ['BRD4','RFX1'],
        ['BRD7','CMTM1'],
        ['BRD8','PHF1'],
        ['BRD9','NUTM1'],
        ['BRDT','MTF2'],
        ['BRE','DPYSL5'],
        ['BRE','FOSL2'],
        ['BRE','RBKS'],
        ['BRE','RMND1'],
        ['BRE','SEMA4C'],
        ['BRG1','SMARCA4'],
        ['BRIP1','EGFR'],
        ['BRIP1','FANCJ'],
        ['BRIP1','GNA13'],
        ['BRIP1','GPATCH8'],
        ['BRIP1','KIAA1468'],
        ['BRIP1','POLDIP2'],
        ['BRIP1','TAF4'],
        ['BRIP1','TMEM49'],
        ['BRIP1','TRIM13'],
        ['BRK1','VHL'],
        ['BRPF1','TRIM24'],
        ['BRPF3','CLPSL1'],
        ['BRPF3','NUDT3'],
        ['BRPF3','STARD9'],
        ['BRS3','HTATSF1'],
        ['BRSK1','COX6B1'],
        ['BRSK1','HSPBP1'],
        ['BRSK1','ZNF587'],
        ['BRWD1','DYRK1A'],
        ['BRWD1','LACTBL1'],
        ['BRWD1','LMTK3'],
        ['BSAP','PAX5'],
        ['BSCL2','TMEM260'],
        ['BSDC1','FAM229A'],
        ['BSG','EEA1'],
        ['BSG','NFIX'],
        ['BSG','RAB22A'],
        ['BSG','STK11'],
        ['BSN','RBM6'],
        ['BSND','DDX21'],
        ['BST1','CD38'],
        ['BST2','ZNF14'],
        ['BSX','FDX1L'],
        ['BTAF1','PTEN'],
        ['BTBD1','NTRK1'],
        ['BTBD1','NTRK2'],
        ['BTBD1','NTRK3'],
        ['BTBD11','DGKK'],
        ['BTBD18','KMT2A'],
        ['BTBD7','SLC2A5'],
        ['BTBD9','CDC5L'],
        ['BTBD9','HMGA2'],
        ['BTBD9','PVT1'],
        ['BTC','FAT1'],
        ['BTC','FGD5'],
        ['BTD','DPF3'],
        ['BTF3','TTC37'],
        ['BTF3L4','ZFYVE9'],
        ['BTG1','MYC'],
        ['BTG2','PRMT1'],
        ['BTG3','CCDC40'],
        ['BTG4','REST'],
        ['BTK','KMT2A'],
        ['BTL','ETV6'],
        ['BTNL2','GLCCI1'],
        ['BTRC','MGMT'],
        ['BUB1B','CCDC170'],
        ['BUB1B','EIF2AK4'],
        ['BUB1B','HSD17B6'],
        ['BUB1B','MARK2'],
        ['BUB1B','PCM1'],
        ['BUB1B','SPRED1'],
        ['BUD13','ZW10'],
        ['BX648577','CIITA'],
        ['BZW2','RPH3AL'],
        ['C10ORF11','TMPRSS2'],
        ['C10ORF111','COPS8'],
        ['C10ORF112','PLXDC2'],
        ['C10ORF118','FGFR2'],
        ['C10ORF118','RET'],
        ['C10ORF118','ZFP82'],
        ['C10ORF119','TIAL1'],
        ['C10ORF140','EIF4G3'],
        ['C10ORF18','NET1'],
        ['C10ORF26','SUFU'],
        ['C10ORF32','PI4K2A'],
        ['C10ORF35','EEF2K'],
        ['C10ORF54','RPS6KB2'],
        ['C10ORF59','NFIA'],
        ['C10ORF68','CCDC7'],
        ['C10ORF68','FGFR2'],
        ['C10ORF76','LUC7L3'],
        ['C11ORF1','FCGR1B'],
        ['C11ORF1','KIAA0999'],
        ['C11ORF1','SIK3'],
        ['C11ORF20','ESRRA'],
        ['C11ORF34','TRMT6'],
        ['C11ORF48','CDK6'],
        ['C11ORF49','PHF21A'],
        ['C11ORF52','DIXDC1'],
        ['C11ORF52','SRD5A2'],
        ['C11ORF58','SBSPON'],
        ['C11ORF63','STARD10'],
        ['C11ORF67','SLC12A7'],
        ['C11ORF70','FAM122C'],
        ['C11ORF70','YAP1'],
        ['C11ORF73','EED'],
        ['C11ORF80','MKNK1'],
        ['C11ORF80','SYT12'],
        ['C11ORF82','CELF1'],
        ['C11ORF84','MARK2'],
        ['C11ORF93','HYOU1'],
        ['C11ORF95','LTBP3'],
        ['C11ORF95','MAML2'],
        ['C11ORF95','MAML3'],
        ['C11ORF95','MKL2'],
        ['C11ORF95','NCOA1'],
        ['C11ORF95','NCOA2'],
        ['C11ORF95','PRG2'],
        ['C11ORF95','RELA'],
        ['C11ORF95','SS18'],
        ['C11ORF95','YAP1'],
        ['C12ORF10','CTDSP2'],
        ['C12ORF11','RASSF8'],
        ['C12ORF4','EFCAB4B'],
        ['C12ORF42','PPP1R12A'],
        ['C12ORF42','TM7SF3'],
        ['C12ORF45','CDK7'],
        ['C12ORF49','CNOT2'],
        ['C12ORF49','MDM2'],
        ['C12ORF5','KCNC2'],
        ['C12ORF5','KIF21A'],
        ['C12ORF56','LEMD3'],
        ['C12ORF57','CHD4'],
        ['C12ORF57','LRRC23'],
        ['C12ORF59','TTLL9'],
        ['C12ORF59','XIAP'],
        ['C12ORF65','NUAK1'],
        ['C12ORF73','HSP90B1'],
        ['C12ORF79','SCYL2'],
        ['C14ORF166','ERG'],
        ['C14ORF166','KRAS'],
        ['C14ORF166','SLC25A1'],
        ['C14ORF166','SLC25A21'],
        ['C14ORF177','NGDN'],
        ['C14ORF182','CDC42BPB'],
        ['C14ORF49','IGHV@'],
        ['C14ORF79','EHBP1L1'],
        ['C15ORF21','ETV1'],
        ['C15ORF21','HNRNPA2B1'],
        ['C15ORF21','MYC'],
        ['C15ORF27','CSK'],
        ['C15ORF41','PTPRT'],
        ['C15ORF41','WWOX'],
        ['C15ORF57','CBX3'],
        ['C15ORF59','SLC36A4'],
        ['C16ORF11','RAB40C'],
        ['C16ORF14','RAB40C'],
        ['C16ORF57','ZNF319'],
        ['C16ORF62','C1ORF123'],
        ['C16ORF62','IQCK'],
        ['C16ORF70','ENKD1'],
        ['C16ORF75','CD273'],
        ['C16ORF75','CIITA'],
        ['C17ORF104','ZNF202'],
        ['C17ORF105','TAOK1'],
        ['C17ORF106','CDK3'],
        ['C17ORF57','NSF'],
        ['C17ORF64','TOP1'],
        ['C17ORF75','GOSR1'],
        ['C17ORF75','NR1D1'],
        ['C17ORF77','TRIM60P15'],
        ['C17ORF80','GGNBP2'],
        ['C17ORF85','ITGAE'],
        ['C18ORF1','PARD6G'],
        ['C18ORF32','DYM'],
        ['C18ORF45','HM13'],
        ['C18ORF8','NTRK1'],
        ['C18ORF8','NTRK2'],
        ['C18ORF8','NTRK3'],
        ['C19MC','RPLP1'],
        ['C19MC','STAG3'],
        ['C19MC','TTYH1'],
        ['C19ORF10','MAP2K3'],
        ['C19ORF10','MAST3'],
        ['C19ORF24','CIRBP'],
        ['C19ORF33','MYCBPAP'],
        ['C19ORF33','SPINT2'],
        ['C19ORF38','SMARCA4'],
        ['C19ORF47','MAP3K10'],
        ['C19ORF56','FBXL18'],
        ['C19ORF57','PVT1'],
        ['C19ORF77','DAD1'],
        ['C1ORF100','CEP170'],
        ['C1ORF109','MACF1'],
        ['C1ORF151','RCC2'],
        ['C1ORF173','FKBP11'],
        ['C1ORF177','DHCR24'],
        ['C1ORF185','NRD1'],
        ['C1ORF194','UQCR10'],
        ['C1ORF21','MIXL1'],
        ['C1ORF226','NOS1AP'],
        ['C1ORF43','TBC1D3B'],
        ['C1ORF61','CCT3'],
        ['C1ORF61','RHBG'],
        ['C1ORF86','MUC1'],
        ['C1ORF87','ERBB2'],
        ['C1ORF95','TAF5L'],
        ['C1QBP','HOXB2'],
        ['C1QL4','SLC6A10P'],
        ['C1QTNF1','STK32C'],
        ['C1QTNF2','HS3ST3A1'],
        ['C1QTNF3-AMACR','PLEKHG4B'],
        ['C1QTNF9','PTPRZ1'],
        ['C1RL','LPCAT3'],
        ['C1S','MAP3K13'],
        ['C20ORF112','PAX5'],
        ['C20ORF112','RUNX1'],
        ['C20ORF151','DIDO1'],
        ['C20ORF195','PTK6'],
        ['C20ORF197','STK4'],
        ['C20ORF201','N6AMT1'],
        ['C20ORF203','ZMYND8'],
        ['C20ORF24','LIFR'],
        ['C21ORF37','CXADR'],
        ['C21ORF91','COLGALT2'],
        ['C21ORF91','ZNF638'],
        ['C22ORF25','RANBP1'],
        ['C2CD2','TFF1'],
        ['C2CD2L','ZNF585B'],
        ['C2CD3','KMT2A'],
        ['C2CD3','MLL'],
        ['C2CD3','MS4A7'],
        ['C2CD3','SMAD9'],
        ['C2CD3','TPCN2'],
        ['C2CD4B','LZTFL1'],
        ['C2ORF28','GPR107'],
        ['C2ORF3','CNTNAP2'],
        ['C2ORF42','DIRC2'],
        ['C2ORF43','PUM2'],
        ['C2ORF48','RRM2'],
        ['C2ORF54','CST3'],
        ['C2ORF55','SERINC5'],
        ['C2ORF56','PRKD3'],
        ['C2ORF61','PI4K2B'],
        ['C2ORF66','NRP2'],
        ['C2ORF66','WNT2'],
        ['C2ORF67','KIF1A'],
        ['C2ORF74','USP34'],
        ['C2ORF82','NBAS'],
        ['C2ORF88','DDX27'],
        ['C3','ELK3'],
        ['C3','H19'],
        ['C3','HP'],
        ['C3','MALAT1'],
        ['C3ORF27','EVI1'],
        ['C3ORF27','MECOM'],
        ['C3ORF62','SRF'],
        ['C4ORF22','MYH13'],
        ['C4ORF22','RBFOX3'],
        ['C4ORF22','RERE'],
        ['C4ORF28','INPP4A'],
        ['C5','MLLT10'],
        ['C5ORF22','MAP2K5'],
        ['C5ORF22','PDCD6'],
        ['C5ORF30','CCNH'],
        ['C5ORF47','PDE4D'],
        ['C5ORF63','TMEM138'],
        ['C6ORF106','SPDEF'],
        ['C6ORF106','TMPRSS2'],
        ['C6ORF123','ECHDC1'],
        ['C6ORF138','CAND1'],
        ['C6ORF138','HSP90AB1'],
        ['C6ORF170','RET'],
        ['C6ORF204','ROS1'],
        ['C6ORF204','SYTL3'],
        ['C6ORF211','ESR1'],
        ['C6ORF211','PDPK1'],
        ['C6ORF223','TMEM63B'],
        ['C6ORF225','TLK1'],
        ['C6ORF35','PREP'],
        ['C6ORF35','ZDHHC14'],
        ['C6ORF48','MKNK2'],
        ['C6ORF52','MAST2'],
        ['C6ORF62','SEC31A'],
        ['C6ORF80','NUP98'],
        ['C6ORF97','ESR1'],
        ['C6ORF97','PPP1R14C'],
        ['C7','FGFR2'],
        ['C7ORF10','CDK13'],
        ['C7ORF10','HIBADH'],
        ['C7ORF16','SRRT'],
        ['C7ORF34','TXNRD1'],
        ['C7ORF44','HECW1'],
        ['C7ORF50','MACROD1'],
        ['C7ORF55','MRPS14'],
        ['C7ORF57','COBL'],
        ['C7ORF61','TSC22D4'],
        ['C7ORF72','INSR'],
        ['C8B','PAH'],
        ['C8ORF34','MET'],
        ['C8ORF34','MYBL1'],
        ['C8ORF34','NFIB'],
        ['C8ORF37','CRTC1'],
        ['C8ORF38','PLEC1'],
        ['C8ORF42','FBXO25'],
        ['C8ORF47','PGCP'],
        ['C9ORF163','SEC16A'],
        ['C9ORF167','COBRA1'],
        ['C9ORF24','NFX1'],
        ['C9ORF25','POLN'],
        ['C9ORF25','ZCCHC7'],
        ['C9ORF3','CPSF6'],
        ['C9ORF3','MIR205HG'],
        ['C9ORF3','PRUNE2'],
        ['C9ORF3','PTCH1'],
        ['C9ORF3','SYK'],
        ['C9ORF3','ZNF169'],
        ['C9ORF85','MED6'],
        ['C9ORF9','SNAP91'],
        ['C9ORF9','SUGP2'],
        ['C9ORF92','HMGA2'],
        ['C9ORF92','MTAP'],
        ['C9ORF93','FREM1'],
        ['CA1','LMTK2'],
        ['CA10','ROR1'],
        ['CA14','RASAL2'],
        ['CA2','RUNX1T1'],
        ['CA2','SLC6A17'],
        ['CA4','TANC2'],
        ['CA7','USP22'],
        ['CAAP1','RPGRIP1L'],
        ['CABLES1','TM9SF4'],
        ['CABLES2','LAMA5'],
        ['CABLES2','SNTA1'],
        ['CABP5','STAU1'],
        ['CABP7','NF2'],
        ['CACNA1A','DHX34'],
        ['CACNA1A','GATAD2A'],
        ['CACNA1A','STK11'],
        ['CACNA1G','DGKE'],
        ['CACNA2D1','HGF'],
        ['CACNA2D1','PCLO'],
        ['CACNA2D2','PVT1'],
        ['CACNA2D3','CAPN11'],
        ['CACNA2D3','FLNB'],
        ['CACNA2D3','IFT74'],
        ['CACNA2D4','WDR43'],
        ['CACNB1','CCDC47'],
        ['CACNB1','CDK12'],
        ['CACNB1','PGAP3'],
        ['CACNB4','STAM2'],
        ['CACNG3','PRKCB'],
        ['CACNG6','PAK4'],
        ['CACNG8','PTK2'],
        ['CADM2','MAP2K3'],
        ['CADM2','MITF'],
        ['CADM2','NANS'],
        ['CADM2','TFEB'],
        ['CADM2-AS2','RNF169'],
        ['CADPS','SLC25A26'],
        ['CALB2','CYB5B'],
        ['CALCR','SEPT11'],
        ['CALCRL','MFSD6'],
        ['CALD1','GPATCH4'],
        ['CALD1','MTMR8'],
        ['CALD1','PDK4'],
        ['CALLA','CD10'],
        ['CALM1','CCDC88C'],
        ['CALML4','CLN6'],
        ['CALML4','CUX1'],
        ['CALN1','KMT2C'],
        ['CALU','CAP2'],
        ['CAMK1','SNHG8'],
        ['CAMK1D','FAM188A'],
        ['CAMK1D','FUBP1'],
        ['CAMK1D','KLB'],
        ['CAMK1D','TSPAN18'],
        ['CAMK2A','CD74'],
        ['CAMK2A','FGFR3'],
        ['CAMK2A','TNIP1'],
        ['CAMK2B','SPIDR'],
        ['CAMK2D','KDR'],
        ['CAMK2D','SHROOM3'],
        ['CAMK2D','TLL2'],
        ['CAMK2D','TOP1'],
        ['CAMK2G','FAM102A'],
        ['CAMK2G','NDUFA13'],
        ['CAMK2G','P4HA1'],
        ['CAMK2G','USP54'],
        ['CAMK4','FHIT'],
        ['CAMK4','SEMA6A'],
        ['CAMK4','SFXN1'],
        ['CAMKK1','MAP2K2'],
        ['CAMKK2','DIP2B'],
        ['CAMKK2','HPD'],
        ['CAMKK2','KDM2B'],
        ['CAMKK2','NR2C1'],
        ['CAMKK2','PMCH'],
        ['CAMKMT','MSH2'],
        ['CAMKMT','PRRC1'],
        ['CAMSAP1','NOTCH1'],
        ['CAMTA1','DNAJC11'],
        ['CAMTA1','FOXJ3'],
        ['CAMTA1','GPATCH3'],
        ['CAMTA1','MTOR'],
        ['CAMTA1','PER3'],
        ['CAMTA1','PGD'],
        ['CAMTA1','TAZ'],
        ['CAMTA1','TMPRSS2'],
        ['CAMTA1','WWTR1'],
        ['CAMTA2','LCK'],
        ['CAN','DEK'],
        ['CAN','SET'],
        ['CAND1','EGFR'],
        ['CAND1','PPP1R12A'],
        ['CAND1','PTCHD4'],
        ['CANT1','DDX5'],
        ['CANT1','ETV4'],
        ['CANX','MALAT1'],
        ['CAP1','MACF1'],
        ['CAP2','CSF1R'],
        ['CAPG','RBED1'],
        ['CAPN1','POLA2'],
        ['CAPN11','CPM'],
        ['CAPN11','PPM1H'],
        ['CAPN12','EHD1'],
        ['CAPN2','SMARCB1'],
        ['CAPN7','FGD5'],
        ['CAPRIN1','FGFR1'],
        ['CAPRIN1','NSD1'],
        ['CAPRIN1','PDGFRB'],
        ['CAPS','CCNG2'],
        ['CAPS','JMJD2B'],
        ['CAPSL','ERG'],
        ['CAPZA2','FOXP2'],
        ['CAPZA2','MET'],
        ['CAPZB','EEF2'],
        ['CAPZB','PAFAH2'],
        ['CAPZB','UBR4'],
        ['CAPZB','ZC3H12A'],
        ['CARD17','SIK2'],
        ['CARD6','CXCL9'],
        ['CARM1','SMARCA4'],
        ['CARM1','YIPF2'],
        ['CARMIL1','NIM1K'],
        ['CARMN','NOTCH1'],
        ['CARMN','NOTCH2'],
        ['CARMN','NOTCH3'],
        ['CARS2','XPO4'],
        ['CASC1','ETV6'],
        ['CASC11','PCAT1'],
        ['CASC11','PVT1'],
        ['CASC15','FGFR2'],
        ['CASC19','CASC8'],
        ['CASC20','NFIB'],
        ['CASC3','CCDC46'],
        ['CASC3','RGS9'],
        ['CASC3','SPOP'],
        ['CASC3','SUPT6H'],
        ['CASC3','WIPF2'],
        ['CASC4','COL4A4'],
        ['CASC5','KMT2A'],
        ['CASC5','MLL'],
        ['CASC8','CCDC26'],
        ['CASC8','FGFR2'],
        ['CASC8','IGH@'],
        ['CASC8','PCAT1'],
        ['CASC8','PDHX'],
        ['CASC8','PVT1'],
        ['CASD1','VWC2'],
        ['CASK','EML4'],
        ['CASK','SQRDL'],
        ['CASK','TM9SF2'],
        ['CASKIN1','TSC2'],
        ['CASP14','ERN1'],
        ['CASP7','FGFR2'],
        ['CASP7','SRGAP1'],
        ['CASP8','ERBB4'],
        ['CASP8AP2','KMT2A'],
        ['CASP9','FAM210A'],
        ['CASR','LPP'],
        ['CASZ1','CPSF6'],
        ['CASZ1','DFFA'],
        ['CASZ1','KAZN'],
        ['CASZ1','MASP2'],
        ['CASZ1','MTOR'],
        ['CASZ1','OBSCN'],
        ['CASZ1','PRKD2'],
        ['CAT','HIPK3'],
        ['CAT','TLK1'],
        ['CATSPER2','PPIP5K1'],
        ['CATSPERB','TMEM182'],
        ['CATSPERD','ILF3'],
        ['CATSPERG','KCNK6'],
        ['CAV1','ST7'],
        ['CAV3','SLC6A11'],
        ['CBARP','SBNO2'],
        ['CBFA2','CBFA2T1'],
        ['CBFA2','ETV6'],
        ['CBFA2T1','CBFA2T3'],
        ['CBFA2T1','ETV6'],
        ['CBFA2T1','LAF4'],
        ['CBFA2T1','MLL'],
        ['CBFA2T1','RUNX1'],
        ['CBFA2T2','CHMP4B'],
        ['CBFA2T2','RUNX1'],
        ['CBFA2T3','ETO2'],
        ['CBFA2T3','EVI1'],
        ['CBFA2T3','GALNS'],
        ['CBFA2T3','GLIS2'],
        ['CBFA2T3','IGH@'],
        ['CBFA2T3','KCNIP2'],
        ['CBFA2T3','MAP4'],
        ['CBFA2T3','NFIA'],
        ['CBFA2T3','PAX5'],
        ['CBFA2T3','PIEZO1'],
        ['CBFA2T3','RUNX1'],
        ['CBFB','HMGB1'],
        ['CBFB','MYH11'],
        ['CBFB','RERG'],
        ['CBFB','SMMHC'],
        ['CBFB','TTC3'],
        ['CBL','DDX6'],
        ['CBL','FBXO2'],
        ['CBL','KANK1'],
        ['CBL','KMT2A'],
        ['CBL','MLL'],
        ['CBLC','RELB'],
        ['CBP','KAT6A'],
        ['CBP','MORF'],
        ['CBS','CHI3L1'],
        ['CBS','CTSB'],
        ['CBWD2','DOCK8'],
        ['CBWD3','PNN'],
        ['CBX1','LEPREL1'],
        ['CBX1','RNF169'],
        ['CBX3','CCDC32'],
        ['CBX3','ETV6'],
        ['CBX4','IGF2BP3'],
        ['CBX5','DCD'],
        ['CBX5','SMARCD1'],
        ['CBX5','TFE3'],
        ['CBX7','ENTHD1'],
        ['CCAR1','PAM16'],
        ['CCAR2','CD8A'],
        ['CCAR2','FGFR2'],
        ['CCBL1','GLE1'],
        ['CCDC101','NBPF8'],
        ['CCDC101','NSMCE1'],
        ['CCDC102A','ZNF207'],
        ['CCDC112','CDC25C'],
        ['CCDC117','HSCB'],
        ['CCDC117','MELK'],
        ['CCDC12','SETD2'],
        ['CCDC122','VPS36'],
        ['CCDC126','ETV6'],
        ['CCDC127','TERT'],
        ['CCDC127','ZNF235'],
        ['CCDC129','PTPRG'],
        ['CCDC130','DOCK6'],
        ['CCDC132','CDH1'],
        ['CCDC136','FKBP9L'],
        ['CCDC142','MRPL53'],
        ['CCDC144A','RAP1GAP2'],
        ['CCDC147','FGFR2'],
        ['CCDC148','WFDC10A'],
        ['CCDC152','RICTOR'],
        ['CCDC158','SCARB2'],
        ['CCDC170','ESR1'],
        ['CCDC170','FGFR2'],
        ['CCDC170','RARA'],
        ['CCDC178','ROCK1'],
        ['CCDC178','TAF4B'],
        ['CCDC186','FGFR2'],
        ['CCDC192','SLC12A2'],
        ['CCDC22','KMT2C'],
        ['CCDC26','DUX4'],
        ['CCDC26','IGH@'],
        ['CCDC26','NSMCE2'],
        ['CCDC26','PVT1'],
        ['CCDC26','TLX3'],
        ['CCDC26','TLX3_'],
        ['CCDC28A','IL22RA2'],
        ['CCDC28A','NPM1'],
        ['CCDC28A','NUP98'],
        ['CCDC3','FGFR2'],
        ['CCDC30','PPCS'],
        ['CCDC30','ROS1'],
        ['CCDC33','RARA'],
        ['CCDC39','GPR162'],
        ['CCDC43','YBX2'],
        ['CCDC46','FGFR2'],
        ['CCDC46','SMURF2'],
        ['CCDC46','SUPT4H1'],
        ['CCDC46','TAOK1'],
        ['CCDC47','PRKCA'],
        ['CCDC49','USP32'],
        ['CCDC50','DSCAM'],
        ['CCDC50','MAP3K13'],
        ['CCDC50','VPRBP'],
        ['CCDC57','NKD2'],
        ['CCDC57','NPLOC4'],
        ['CCDC57','TP53'],
        ['CCDC58','RPN1'],
        ['CCDC6','FGFR2'],
        ['CCDC6','GOLGA5'],
        ['CCDC6','HPR'],
        ['CCDC6','PDGFRB'],
        ['CCDC6','RET'],
        ['CCDC6','ROS1'],
        ['CCDC6','VPS13B'],
        ['CCDC67','TMEM135'],
        ['CCDC67','TMEM137'],
        ['CCDC67','VAV2'],
        ['CCDC7','FGFR2'],
        ['CCDC73','GCC2'],
        ['CCDC81','PICALM'],
        ['CCDC82','FAM19A2'],
        ['CCDC85C','SETD3'],
        ['CCDC86','TMEM192'],
        ['CCDC88A','NPM1'],
        ['CCDC88C','FLT3'],
        ['CCDC88C','NAP1L1'],
        ['CCDC88C','PDGFRB'],
        ['CCDC88C','SMEK1'],
        ['CCDC9','KIAA0134'],
        ['CCDC91','KRAS'],
        ['CCDC91','MRPS35'],
        ['CCDC91','STK38L'],
        ['CCDC94','SH3GL1'],
        ['CCKBR','NGLY1'],
        ['CCL17','TARC'],
        ['CCL2','MCP1'],
        ['CCL2','MED1'],
        ['CCL21','CCR7'],
        ['CCL27','DCP1A'],
        ['CCL4','ERBB2'],
        ['CCL4L2','MYO19'],
        ['CCL5','RANTES'],
        ['CCM2','OGDH'],
        ['CCM2L','HCK'],
        ['CCN2','CTGF'],
        ['CCN3','IGH@'],
        ['CCNA1','ORC4'],
        ['CCNB1','CDK7'],
        ['CCNB1','PDE4D'],
        ['CCNB1','TP53'],
        ['CCNB1IP1','HMGA2'],
        ['CCNB1IP1','PLEKHG1'],
        ['CCNB1IP1','PVT1'],
        ['CCNC','USP45'],
        ['CCND1','CDK4'],
        ['CCND1','FGFR3'],
        ['CCND1','FSTL3'],
        ['CCND1','IGH@'],
        ['CCND1','IGHG1'],
        ['CCND1','IGK@'],
        ['CCND1','IGL@'],
        ['CCND1','IGLC1'],
        ['CCND1','MRCK'],
        ['CCND1','MYEOV'],
        ['CCND1','PLAT'],
        ['CCND1','PTH'],
        ['CCND1','TACSTD2'],
        ['CCND1','TROP2'],
        ['CCND2','CCND3'],
        ['CCND2','CDK4'],
        ['CCND2','IGH@'],
        ['CCND2','IGK@'],
        ['CCND2','IGL@'],
        ['CCND2','NINJ2'],
        ['CCND2','TRA@'],
        ['CCND2','TRB@'],
        ['CCND2-AS2','PRMT8'],
        ['CCND3','FGF11'],
        ['CCND3','IGH@'],
        ['CCND3','IGK@'],
        ['CCND3','IGL@'],
        ['CCND3','KCNMB4'],
        ['CCND3','PTCRA'],
        ['CCND3','STIL'],
        ['CCND3','UBR2'],
        ['CCNE','CDK2'],
        ['CCNE1','IGH@'],
        ['CCNE2','FAM82B'],
        ['CCNF','PDPK1'],
        ['CCNG1','IGH@'],
        ['CCNG2','G3BP2'],
        ['CCNG2','GALNS'],
        ['CCNG2','MAML3'],
        ['CCNG2','PDGFRA'],
        ['CCNI','CTSZ'],
        ['CCNI','MALAT1'],
        ['CCNI','SHROOM3'],
        ['CCNI','TXLNG2P'],
        ['CCNK','NPR3'],
        ['CCNL1','NBPF11'],
        ['CCNL1','TRIB1'],
        ['CCNL2','PCMTD1'],
        ['CCNT1','HECTD4'],
        ['CCNY','MAP3K8'],
        ['CCNY','NRG4'],
        ['CCNY','WARS'],
        ['CCNYL1','CREB1'],
        ['CCNYL1','PLOD2'],
        ['CCR10','NR1I3'],
        ['CCR3','DHRS4L2'],
        ['CCR4','NOT'],
        ['CCR6','FGFR1'],
        ['CCR7','DOT1L'],
        ['CCR7','NF1'],
        ['CCR9','SVIP'],
        ['CCRL2','SFMBT1'],
        ['CCRN4L','FAT1'],
        ['CCSER1','FRAS1'],
        ['CCSER1','MAPK10'],
        ['CCT2','NUP107'],
        ['CCT3','PPP2R2B'],
        ['CCZ1B','CNTN1'],
        ['CD10','CD19'],
        ['CD10','CD20'],
        ['CD10','MUM1'],
        ['CD101','SEC31A'],
        ['CD133','PROM1'],
        ['CD15','CD34'],
        ['CD151','DRD4'],
        ['CD151','TSPAN4'],
        ['CD154','CD40'],
        ['CD163','ETV6'],
        ['CD164','SLC15A2'],
        ['CD19','CD22'],
        ['CD19','CD5'],
        ['CD1A','SLAMF1'],
        ['CD1B','GTF2F2'],
        ['CD1C','NEK7'],
        ['CD2','CD34'],
        ['CD2','MYC'],
        ['CD200','CD200R'],
        ['CD200R1','RHOA'],
        ['CD21','CD23'],
        ['CD244','CLK2'],
        ['CD27','CD44'],
        ['CD273','CD274'],
        ['CD273','CIITA'],
        ['CD273','PDL2'],
        ['CD273','RALGDS'],
        ['CD273','RUNDC2A'],
        ['CD274','CIITA'],
        ['CD274','EEF1A1'],
        ['CD274','IGH@'],
        ['CD274','IGL@'],
        ['CD274','JAK2'],
        ['CD274','N4BP2'],
        ['CD274','PDCD1LG2'],
        ['CD274','PDL1'],
        ['CD274','PHACTR4'],
        ['CD28','CD80'],
        ['CD28','CKMT2'],
        ['CD28','CTLA4'],
        ['CD28','ICOS'],
        ['CD2AP','HIST1H2BN'],
        ['CD2AP','MARCH9'],
        ['CD2AP','MGAT4C'],
        ['CD300A','DHRS2'],
        ['CD300E','NPC1'],
        ['CD302','LY75'],
        ['CD302','MARCH7'],
        ['CD34','CD56'],
        ['CD34','PTN'],
        ['CD36','ITK'],
        ['CD36','PVT1'],
        ['CD37','MAP3K13'],
        ['CD3D','MLL'],
        ['CD3D','SIK3'],
        ['CD4','CHD4'],
        ['CD4','M6PR'],
        ['CD4','MED13L'],
        ['CD4','PTPN6'],
        ['CD40','CD40L'],
        ['CD44','EAAT2'],
        ['CD44','FGFR2'],
        ['CD44','IGF1R'],
        ['CD44','IGH@'],
        ['CD44','PDHX'],
        ['CD44','SLC1A2'],
        ['CD46','PAQR3'],
        ['CD46','USH2A'],
        ['CD48','PARD6B'],
        ['CD5','CD7'],
        ['CD5','NTRK1'],
        ['CD53','PDGFRB'],
        ['CD58','COPA'],
        ['CD63','DPM1'],
        ['CD63','PRKCD'],
        ['CD68','PGM1'],
        ['CD69','HIST1H2BG'],
        ['CD70','ETV6'],
        ['CD74','CRP'],
        ['CD74','LTF'],
        ['CD74','NRG1'],
        ['CD74','NTRK1'],
        ['CD74','NTRK2'],
        ['CD74','NTRK3'],
        ['CD74','PDGFRB'],
        ['CD74','ROS1'],
        ['CD79B','MAP2K4'],
        ['CD8','PD1'],
        ['CD81','SPAG6'],
        ['CD81','TSSC4'],
        ['CD83','JARID2'],
        ['CD9','CDKN1A'],
        ['CD9','STAT2'],
        ['CD97','EMR2'],
        ['CD99','MIC2'],
        ['CD99','P2RY8'],
        ['CDADC1','GORASP2'],
        ['CDC14B','HLX'],
        ['CDC16','GPATCH11'],
        ['CDC16','KDM7A'],
        ['CDC16','TP53'],
        ['CDC16','UPF3A'],
        ['CDC25A','GRID1'],
        ['CDC27','ERBB2'],
        ['CDC27','OAT'],
        ['CDC2L1','SLC35E2'],
        ['CDC42BPA','DIP2A'],
        ['CDC42BPA','RASAL1'],
        ['CDC42BPA','WWOX'],
        ['CDC42BPB','PET112'],
        ['CDC42BPB','PUM1'],
        ['CDC42BPB','RRP9'],
        ['CDC42BPB','SYNJ2BP'],
        ['CDC42BPB','ZFYVE1'],
        ['CDC42BPG','MAPKAPK2'],
        ['CDC42EP1','FGFR1'],
        ['CDC42EP1','MAP3K8'],
        ['CDC42EP2','NFAT5'],
        ['CDC42EP2','POLA2'],
        ['CDC42EP5','SAMD1'],
        ['CDC42SE1','PIP5K1A'],
        ['CDC42SE2','FAM114A2'],
        ['CDC42SE2','KIAA0146'],
        ['CDC42SE2','MAP3K1'],
        ['CDC42SE2','TRDMT1'],
        ['CDC6','CDK12'],
        ['CDC6','IKZF3'],
        ['CDC6','PML'],
        ['CDC6','RARA'],
        ['CDC7','LRIG1'],
        ['CDC73','FAM172A'],
        ['CDCA5','PHLDB1'],
        ['CDCA5','SEPT6'],
        ['CDH1','CIRH1A'],
        ['CDH1','CSF1'],
        ['CDH1','LARS'],
        ['CDH1','NRG1'],
        ['CDH1','SF3B3'],
        ['CDH1','TANGO6'],
        ['CDH10','PI4KA'],
        ['CDH10','PTPRH'],
        ['CDH11','COL1A1'],
        ['CDH11','OMD'],
        ['CDH11','THRAP3'],
        ['CDH11','UBFD1'],
        ['CDH11','USP6'],
        ['CDH11','ZNF9'],
        ['CDH12','HTT'],
        ['CDH17','FGFR2'],
        ['CDH17','SLCO2B1'],
        ['CDH18','SUCO'],
        ['CDH18','TFAP4'],
        ['CDH23','KLK5'],
        ['CDH3','TERF2'],
        ['CDH4','TRIO'],
        ['CDH6','PDCD6IP'],
        ['CDHR1','WAPAL'],
        ['CDHR1','WAPL'],
        ['CDHR2','CLTB'],
        ['CDIP1','NHSL1'],
        ['CDK1','WDR11'],
        ['CDK11A','GNB1'],
        ['CDK11A','YWHAB'],
        ['CDK12','CLCN4'],
        ['CDK12','FBXL20'],
        ['CDK12','GRB7'],
        ['CDK12','KCNRG'],
        ['CDK12','LASP1'],
        ['CDK12','MED1'],
        ['CDK12','MIEN1'],
        ['CDK12','MSL1'],
        ['CDK12','PNMT'],
        ['CDK12','PPM1E'],
        ['CDK12','RBM47'],
        ['CDK12','RECQL4'],
        ['CDK12','SMG6'],
        ['CDK12','SYNRG'],
        ['CDK12','TMEM104'],
        ['CDK13','DIP2C'],
        ['CDK13','ESR1'],
        ['CDK13','PDE1C'],
        ['CDK13','POU6F2'],
        ['CDK13','RALA'],
        ['CDK13','TAX1BP1'],
        ['CDK14','MKLN1'],
        ['CDK14','PRKCZ'],
        ['CDK14','TFEC'],
        ['CDK16','USP11'],
        ['CDK17','ELK3'],
        ['CDK17','KCNC2'],
        ['CDK19','LYZ'],
        ['CDK19','PRDM1'],
        ['CDK19','REV3L'],
        ['CDK19','SYNJ2'],
        ['CDK2','IKZF1'],
        ['CDK2','PAN2'],
        ['CDK2','RAB5B'],
        ['CDK2AP1','ETV6'],
        ['CDK4','MDM2'],
        ['CDK4','SUDS3'],
        ['CDK4','TSFM'],
        ['CDK4','TSPAN31'],
        ['CDK5RAP2','JAK2'],
        ['CDK5RAP2','MEGF9'],
        ['CDK5RAP2','PDGFRA'],
        ['CDK6','EVI1'],
        ['CDK6','FAM133B'],
        ['CDK6','FN1'],
        ['CDK6','GATAD1'],
        ['CDK6','HEPACAM2'],
        ['CDK6','IGH@'],
        ['CDK6','IGK@'],
        ['CDK6','IGL@'],
        ['CDK6','KMT2A'],
        ['CDK6','MALAT1'],
        ['CDK6','METTL12'],
        ['CDK6','MLL'],
        ['CDK6','MLLT10'],
        ['CDK6','PIM3'],
        ['CDK6','PPM1H'],
        ['CDK6','RPSA'],
        ['CDK6','STXBP6'],
        ['CDK6','TAF1D'],
        ['CDK6','TCP1'],
        ['CDK6','TLX3'],
        ['CDK6','TLX3_'],
        ['CDK8','CDX2'],
        ['CDK8','GTF2F2'],
        ['CDK8','GTF3A'],
        ['CDK8','IKZF3'],
        ['CDK8','UVRAG'],
        ['CDKAL1','E2F3'],
        ['CDKAL1','EGFR'],
        ['CDKAL1','RANBP9'],
        ['CDKAL1','SAMD9'],
        ['CDKL1','FOXN3'],
        ['CDKL1','TSEN2'],
        ['CDKL2','COMMD9'],
        ['CDKL2','CXCL11'],
        ['CDKL2','G3BP2'],
        ['CDKL2','ULK4'],
        ['CDKN1A','RAB44'],
        ['CDKN1B','ETV6'],
        ['CDKN1B','KIP1'],
        ['CDKN1C','KCNQ1OT1'],
        ['CDKN2A','CDKN2B'],
        ['CDKN2A','FAM230A'],
        ['CDKN2A','FOXJ3'],
        ['CDKN2A','IL1RAPL2'],
        ['CDKN2A','ITCH'],
        ['CDKN2A','KIAA1797'],
        ['CDKN2A','MTAP'],
        ['CDKN2A','P16'],
        ['CDKN2A','RB1'],
        ['CDKN2A','TRA@'],
        ['CDKN2A','UBAP1'],
        ['CDKN2AIP','SYNE1'],
        ['CDKN2B','DGKD'],
        ['CDKN2B','RPS2'],
        ['CDKN2B-AS1','MTAP'],
        ['CDKN2BAS','MTAP'],
        ['CDKN2D','WDFY2'],
        ['CDKN3','FBXL7'],
        ['CDON','NF2'],
        ['CDR2','RRN3P3'],
        ['CDS1','RPLP0'],
        ['CDS2','POLR3E'],
        ['CDX1','IRF2BP2'],
        ['CDX1','SQSTM1'],
        ['CDX2','ETV6'],
        ['CDYL','FARS2'],
        ['CDYL','RERE'],
        ['CDYL2','PLXNA4'],
        ['CDYL2','PRCAT47'],
        ['CEACAM16','CEP89'],
        ['CEACAM6','CEACAM7'],
        ['CEACAM8','RELB'],
        ['CEACAM8','SIRT2'],
        ['CEACAMP1','DIS3L'],
        ['CEBPA','IGH@'],
        ['CEBPB','IGH@'],
        ['CEBPD','IGH@'],
        ['CEBPE','IGH@'],
        ['CEBPE','TRA@'],
        ['CEBPG','IGH@'],
        ['CEBPZ','EIF2AK2'],
        ['CEL','CELP'],
        ['CEL','NTRK1'],
        ['CEL','NTRK2'],
        ['CEL','NTRK3'],
        ['CELF1','DDIAS'],
        ['CELF2','FGFR2'],
        ['CELF2','TNRC6B'],
        ['CELF5','ESPL1'],
        ['CELSR1','SNAP29'],
        ['CELSR1','TP53'],
        ['CELSR3','IP6K1'],
        ['CEN3','SOX2'],
        ['CENPE','HEATR5B'],
        ['CENPE','SLC9B2'],
        ['CENPF','RET'],
        ['CENPK','KMT2A'],
        ['CENPK','LGR5'],
        ['CENPK','MLL'],
        ['CENPP','WNK2'],
        ['CEP1','FGFR1'],
        ['CEP110','FGFR1'],
        ['CEP112','FGFR2'],
        ['CEP112','KRT23'],
        ['CEP120','CSNK1G3'],
        ['CEP126','YAP1'],
        ['CEP128','NOTCH2'],
        ['CEP152','IQGAP1'],
        ['CEP164','DDX6'],
        ['CEP164','MLLT10'],
        ['CEP170','PRKACB'],
        ['CEP170','RAD51B'],
        ['CEP170B','KMT2A'],
        ['CEP192','VAPA'],
        ['CEP250','ZMYND8'],
        ['CEP290','PRKDC'],
        ['CEP55','PIK3C2G'],
        ['CEP55','RET'],
        ['CEP57','NAT10'],
        ['CEP57L1','LACE1'],
        ['CEP57L1','SRPK1'],
        ['CEP72','FNTA'],
        ['CEP72','ROS1'],
        ['CEP76','RUNX1'],
        ['CEP85','TMPRSS2'],
        ['CEP85L','PDGFRB'],
        ['CEP85L','ROS1'],
        ['CEP85L','SYTL3'],
        ['CERS2','PIGC'],
        ['CERS5','TMEM136'],
        ['CES1P2','ZNF317'],
        ['CEV14','PDGFRB'],
        ['CFAP300','YAP1'],
        ['CFAP43','FYN'],
        ['CFAP46','VTI1A'],
        ['CFAP54','NEDD1'],
        ['CFAP70','MRPS16'],
        ['CFAP97','FAT1'],
        ['CFB','ERBB2'],
        ['CFHR1','CFHR3'],
        ['CFHR1','CFHR5'],
        ['CFHR2','CFHR5'],
        ['CFL1','RNASEH2C'],
        ['CFLAR','CTDSP1'],
        ['CFLAR','ETV6'],
        ['CFLAR-AS1','ETV6'],
        ['CFLAR-AS1','ORC6'],
        ['CFTR','PVT1'],
        ['CFTR','SND1'],
        ['CGA','ZFYVE9'],
        ['CGN','NTRK1'],
        ['CGNL1','CYP19A1'],
        ['CGNL1','FGFR2'],
        ['CHAF1A','SH3GL1'],
        ['CHAF1A','SHD'],
        ['CHCHD10','FGA'],
        ['CHCHD10','VPREB3'],
        ['CHCHD2','HES1'],
        ['CHCHD3','ROS1'],
        ['CHCHD5','GUSBP3'],
        ['CHCHD7','CTNNB1'],
        ['CHCHD7','PLAG1'],
        ['CHD1','FRAP1'],
        ['CHD1','MTOR'],
        ['CHD1','RUNX1'],
        ['CHD2','SLCO3A1'],
        ['CHD3','DNAH2'],
        ['CHD3','SHBG'],
        ['CHD4','EP300'],
        ['CHD4','GPX3'],
        ['CHD4','SCNN1A'],
        ['CHD4','ZNF384'],
        ['CHD6','STK35'],
        ['CHD7','IGF1R'],
        ['CHD7','KAT6A'],
        ['CHD7','PVT1'],
        ['CHD7','RAB2A'],
        ['CHD7','TOX'],
        ['CHD7','VGLL3'],
        ['CHD9','CRLF3'],
        ['CHD9','TOX3'],
        ['CHDC2','MAP3K8'],
        ['CHEK1','EXOC6'],
        ['CHEK1','NTM'],
        ['CHEK2','KIAA1984'],
        ['CHEK2','MPG'],
        ['CHEK2','PPP2R2A'],
        ['CHEK2','PPP2RA'],
        ['CHEK2','THOC5'],
        ['CHEK2','XBP1'],
        ['CHERP','SLMO2'],
        ['CHFR','GOLGA3'],
        ['CHFR','PAX5'],
        ['CHFR','SMARCD1'],
        ['CHGA','FGFR2'],
        ['CHGA','MAPKAPK2'],
        ['CHGB','KMT2C'],
        ['CHGB','MLL3'],
        ['CHGN','PSD3'],
        ['CHI3L1','GFAP'],
        ['CHIA','ZNF138'],
        ['CHIC1','MACROD1'],
        ['CHIC2','ETV6'],
        ['CHIC2','FIP1L1'],
        ['CHIC2','PDGFRA'],
        ['CHID1','CORO7'],
        ['CHID1','MAGMAS'],
        ['CHKA','PTK2'],
        ['CHMP1A','ZC3H18'],
        ['CHMP1A','ZC3HAV1L'],
        ['CHMP2B','ZNF197'],
        ['CHMP3','PIGF'],
        ['CHMP4B','PTPN1'],
        ['CHN','EWS'],
        ['CHN','RBP56'],
        ['CHN1','TAF15'],
        ['CHN1','ZNF384'],
        ['CHN2','RALY'],
        ['CHN2','TSPAN33'],
        ['CHODL','TMPRSS2'],
        ['CHOP','DDIT3'],
        ['CHOP','EWS'],
        ['CHOP','EWSR1'],
        ['CHOP','FUS'],
        ['CHOP','GADD153'],
        ['CHOP','TLS'],
        ['CHPF','CPNE5'],
        ['CHPT1','MED13L'],
        ['CHPT1','UTP20'],
        ['CHRM1','MRPS28'],
        ['CHRM3','PBX1'],
        ['CHRM5','NUTM1'],
        ['CHRNA1','SPIRE1'],
        ['CHRNA6','POMK'],
        ['CHRNG','PTBP3'],
        ['CHST11','FAM70A'],
        ['CHST11','IGH@'],
        ['CHST3','PTCH2'],
        ['CHSY3','NINJ2'],
        ['CHTOP','FGFR1'],
        ['CHTOP','INTS3'],
        ['CHTOP','NTRK1'],
        ['CHTOP','NTRK2'],
        ['CHTOP','NTRK3'],
        ['CIB2','PEAK1'],
        ['CIC','CITED1'],
        ['CIC','DUX4'],
        ['CIC','DUX4L1'],
        ['CIC','DUX4L8'],
        ['CIC','FOXO4'],
        ['CIC','LEUTX'],
        ['CIC','NUTM1'],
        ['CIC','NUTM2A'],
        ['CIC','RND3'],
        ['CIC','SREBF1'],
        ['CIC','ZNF526'],
        ['CICP6','EIF4E3'],
        ['CICP6','HTR1F'],
        ['CIDEB','NOP9'],
        ['CIITA','CREBBP'],
        ['CIITA','DEXI'],
        ['CIITA','FLJ27352'],
        ['CIITA','NUBP1'],
        ['CIITA','PDCD1LG2'],
        ['CIITA','RALGDS'],
        ['CIITA','RUNDC2A'],
        ['CIP29','KMT2A'],
        ['CIP29','MLL'],
        ['CIPC','NGFR'],
        ['CIRH1A','TMCO7'],
        ['CIT','ETV6'],
        ['CIT','ETV7'],
        ['CIT','FAM222A'],
        ['CIT','FGFR2'],
        ['CIT','RFC5'],
        ['CITED1','HDAC8'],
        ['CITED1','PC'],
        ['CITED1','SRF'],
        ['CITED2','PRDM10'],
        ['CITED2','VGLL2'],
        ['CIZ1','FAM102A'],
        ['CK7','MUC1'],
        ['CKLF','CMTM4'],
        ['CKM','PRKD2'],
        ['CKS1B','MEST'],
        ['CKS1B','SHC1'],
        ['CLASP1','DTNB'],
        ['CLASP1','INHBB'],
        ['CLASRP','SYMPK'],
        ['CLCA2','RUNX1'],
        ['CLCC1','KIAA0391'],
        ['CLCF1','POLD4'],
        ['CLCN3','NEK1'],
        ['CLCN4','IPO9'],
        ['CLCN6','RAF1'],
        ['CLCN7','CRAMP1L'],
        ['CLDN10','DNAJC3'],
        ['CLDN12','EBF1'],
        ['CLDN14','ERG'],
        ['CLDN6','PKMYT1'],
        ['CLDN7','CUX1'],
        ['CLDN7','SLC34A2'],
        ['CLDN9','THOC6'],
        ['CLEC16A','EMP2'],
        ['CLEC16A','TXNDC11'],
        ['CLEC17A','TBC1D15'],
        ['CLEC19A','ST6GAL1'],
        ['CLEC2D','IGH@'],
        ['CLEC4D','CLEC6A'],
        ['CLEC4E','ZNF765'],
        ['CLEC7A','SUPT5H'],
        ['CLEC9A','TAX1BP1'],
        ['CLHC1','TREML3P'],
        ['CLIC4','DNAJC8'],
        ['CLIC4','PARK7'],
        ['CLIC4','PLOD1'],
        ['CLIC4','SRRM1'],
        ['CLIC5','HNF1A'],
        ['CLIC5','RAI1'],
        ['CLIC5','SCO1'],
        ['CLIC5','SUPT3H'],
        ['CLIC5','TP53'],
        ['CLIC5','TRIO'],
        ['CLINT1','EDIL3'],
        ['CLIP1','ERC1'],
        ['CLIP1','FGFR2'],
        ['CLIP1','KIAA1598'],
        ['CLIP1','PWWP2A'],
        ['CLIP1','ROS1'],
        ['CLIP1','TNIK'],
        ['CLIP1','TPM3'],
        ['CLIP2','EGFR'],
        ['CLIP2','FGFR2'],
        ['CLIP2','FOXK2'],
        ['CLIP2','GTF2I'],
        ['CLIP2','GTF2IRD1'],
        ['CLIP2','KMT2C'],
        ['CLIP2','MET'],
        ['CLIP4','EPHB4'],
        ['CLIP4','MCFD2'],
        ['CLIP4','SAP130'],
        ['CLIP4','VSNL1'],
        ['CLK2','MARS'],
        ['CLK2','PAFAH1B3'],
        ['CLK2','PPP1R12B'],
        ['CLK2','RBM19'],
        ['CLN5','FBXL3'],
        ['CLNK','TRIO'],
        ['CLOCK','SCFD2'],
        ['CLOCK','TMEM165'],
        ['CLP1','MLLT10'],
        ['CLPTM1L','GLIS3'],
        ['CLPTM1L','PVT1'],
        ['CLPTM1L','SERBP1'],
        ['CLPTM1L','TERT'],
        ['CLSTN1','CTNNBIP1'],
        ['CLSTN1','EFCAB7'],
        ['CLSTN1','KAZN'],
        ['CLSTN1','STOX1'],
        ['CLSTN2','FSD1'],
        ['CLTA','UBE2R2'],
        ['CLTC','ETV1'],
        ['CLTC','FOSB'],
        ['CLTC','JAK1'],
        ['CLTC','MLL'],
        ['CLTC','MYO1D'],
        ['CLTC','PPP1R9B'],
        ['CLTC','ROS1'],
        ['CLTC','TEX14'],
        ['CLTC','TFE3'],
        ['CLTC','TFEB'],
        ['CLTC','TMEM49'],
        ['CLTC','VMP1'],
        ['CLTCL1','NF2'],
        ['CLU','DIS3L'],
        ['CLU','MYH14'],
        ['CLU','NRG1'],
        ['CLUHP3','ZNF720'],
        ['CLVS1','CRLF2'],
        ['CLVS1','PVT1'],
        ['CLVS2','MAP3K5'],
        ['CLYBL','TSC22D1'],
        ['CMAHP','MLL'],
        ['CMAS','PIK3C2G'],
        ['CMC4','KANSL1'],
        ['CMIP','DYNLRB2'],
        ['CMIP','WWOX'],
        ['CMKOR1','COX6C'],
        ['CMKOR1','HMGA2'],
        ['CMKOR1','NFIB'],
        ['CMPK2','ZNF90'],
        ['CMTM4','DYRK1A'],
        ['CMTM7','CMTM8'],
        ['CMTM7','GLB1'],
        ['CMTM8','RAF1'],
        ['CMYA5','MIIP'],
        ['CNBP','USP6'],
        ['CNGB1','KIFC3'],
        ['CNIH','MPP7'],
        ['CNIH1','PRKD1'],
        ['CNIH2','RPS6KB2'],
        ['CNIH4','ERBB2'],
        ['CNIH4','PAK1'],
        ['CNKSR3','IPCEF1'],
        ['CNN3','IGH@'],
        ['CNNM2','EGR2'],
        ['CNNM2','SUFU'],
        ['CNNM3','RNF138'],
        ['CNNM4','CTDSP2'],
        ['CNNM4','PARD3B'],
        ['CNOT2','PTK7'],
        ['CNOT2','RAB3IP'],
        ['CNOT2','WT1'],
        ['CNOT6L','FRAS1'],
        ['CNOT8','FOXR1'],
        ['CNOT8','LARP1'],
        ['CNPY3','TPO'],
        ['CNRIP1','DHX57'],
        ['CNTLN','MLLT3'],
        ['CNTLN','RAF1'],
        ['CNTN1','PDZRN4'],
        ['CNTN1','USP15'],
        ['CNTN2','DYRK2'],
        ['CNTN4','SATB1'],
        ['CNTN5','RAB3C'],
        ['CNTN5','YAP1'],
        ['CNTN6','EPHA6'],
        ['CNTNAP2','CUL1'],
        ['CNTNAP2','GCFC2'],
        ['CNTNAP2','GRM8'],
        ['CNTNAP2','KIAA2013'],
        ['CNTNAP2','SORL1'],
        ['CNTNAP2','UBE3C'],
        ['CNTNAP4','RAB11FIP4'],
        ['CNTRL','FGFR1'],
        ['CNTRL','KIT'],
        ['COBL','EGFR'],
        ['COBL','ELMO1'],
        ['COBL','GLI3'],
        ['COBL','IKZF1'],
        ['COBL','SEPT14'],
        ['COBLL1','FGFR2'],
        ['COG3','GPC6'],
        ['COG5','HMGA2'],
        ['COG6','MAPK8'],
        ['COG8','TMED6'],
        ['COL10A1','HIPK1'],
        ['COL12A1','COL4A5'],
        ['COL12A1','GRM1'],
        ['COL12A1','RAC1'],
        ['COL14A1','DEPTOR'],
        ['COL14A1','FAM84B'],
        ['COL14A1','FGFR2'],
        ['COL14A1','MTSS1'],
        ['COL14A1','SKAP1'],
        ['COL15A1','TGFBR1'],
        ['COL17A1','PTEN'],
        ['COL19A1','NLGN4X'],
        ['COL1A1','FYN'],
        ['COL1A1','HSP90AB1'],
        ['COL1A1','PDGFB'],
        ['COL1A1','PEAK1'],
        ['COL1A1','PIKFYVE'],
        ['COL1A1','PPP2R2C'],
        ['COL1A1','USP6'],
        ['COL1A2','EXPH5'],
        ['COL1A2','MALAT1'],
        ['COL1A2','MAPK10'],
        ['COL1A2','MMP14'],
        ['COL1A2','PDGFB'],
        ['COL1A2','PLAG1'],
        ['COL21A1','TFEB'],
        ['COL22A1','FAM135B'],
        ['COL23A1','TKT'],
        ['COL24A1','MCOLN3'],
        ['COL25A1','FGFR2'],
        ['COL25A1','NFKB2'],
        ['COL26A1','YES1'],
        ['COL2A1','HAPLN1'],
        ['COL3A1','MALAT1'],
        ['COL3A1','PDGFB'],
        ['COL3A1','PLAG1'],
        ['COL3A1','PSMC5'],
        ['COL3A1','USP6'],
        ['COL4A1','FAM155A'],
        ['COL4A1','VEGFD'],
        ['COL4A2','FAM155A'],
        ['COL4A5','COL4A6'],
        ['COL4A6','USP20'],
        ['COL5A1','GAPVD1'],
        ['COL6A2','USP6'],
        ['COL6A3','CSF1'],
        ['COL6A3','PDGFD'],
        ['COL8A2','EPHA10'],
        ['COL9A2','RLF'],
        ['COL9A3','ELK3'],
        ['COL9A3','MAN2A2'],
        ['COLCA2','NAA20'],
        ['COLEC10','TAF2'],
        ['COLEC10','TNFRSF11B'],
        ['COLEC12','GATA6'],
        ['COLQ','HACL1'],
        ['COMMD1','EHBP1'],
        ['COMMD1','ERBB4'],
        ['COMMD1','REL'],
        ['COMMD10','SEMA6A'],
        ['COMT','GAPDH'],
        ['COMT','MED15'],
        ['COP1','NTRK1'],
        ['COPG1','RASA2'],
        ['COPG2','TMEM209'],
        ['COPS4','SNRNP25'],
        ['COQ10A','NINJ2'],
        ['COQ10A','USP5'],
        ['COQ6','MAST2'],
        ['COQ6','PVT1'],
        ['CORO1C','VEZT'],
        ['CORO7','VPS13D'],
        ['COX10','NRG1'],
        ['COX10-AS1','NRG1'],
        ['COX4I2','ERBB3'],
        ['COX5A','EDC3'],
        ['COX5A','NTRK3'],
        ['COX5A','PARP16'],
        ['COX6C','HMGA2'],
        ['COX6C','HMGIC'],
        ['COX6C','LAPTM4B'],
        ['COX6C','PCBD2'],
        ['COX7A1','ZNF790'],
        ['COX7B2','SLC30A9'],
        ['COX8C','IAPP'],
        ['CP','MALAT1'],
        ['CP','WWTR1'],
        ['CPA2','FAM69B'],
        ['CPD','ERBB2'],
        ['CPD','PXT1'],
        ['CPD','TAOK1'],
        ['CPE','GNAS'],
        ['CPEB1','GOLGA6L17P'],
        ['CPLX2','MAP3K1'],
        ['CPLX2','UBXD8'],
        ['CPM','LEMD3'],
        ['CPM','LYZ'],
        ['CPM','MDM2'],
        ['CPM','PDZRN4'],
        ['CPM','PRKAA2'],
        ['CPM','RAP1B'],
        ['CPM','STAT6'],
        ['CPN1','PKD2L1'],
        ['CPN2','PTPRC'],
        ['CPNE1','PI3'],
        ['CPNE1','PREX1'],
        ['CPNE2','NFATC3'],
        ['CPNE4','LLGL2'],
        ['CPNE4','STK36'],
        ['CPNE8','RPAP3'],
        ['CPNE8','RUNX1'],
        ['CPPED1','GNPTAB'],
        ['CPPED1','NUP153'],
        ['CPQ','PRKDC'],
        ['CPS1','WNT10B'],
        ['CPSF3L','EPHB2'],
        ['CPSF3L','TWIST2'],
        ['CPSF4L','ERBB4'],
        ['CPSF6','DYRK2'],
        ['CPSF6','FGFR1'],
        ['CPSF6','KCNMB4'],
        ['CPSF6','PDGFRB'],
        ['CPSF6','RARG'],
        ['CPT1A','HRASLS2'],
        ['CPT1A','RPS4XP19'],
        ['CPT1A','SCYL1'],
        ['CPT1C','JAZF1'],
        ['CPTP','PVT1'],
        ['CR848007.2','LEFTY2'],
        ['RARG','HNRNPM'],
        ['CRABP2','RNF146'],
        ['CRABP2','TCEANC2'],
        ['CRADD','ERBB3'],
        ['CRAF','RAF1'],
        ['CRAMP1L','CREBBP'],
        ['CRB1','NEK7'],
        ['CRCP','TPST1'],
        ['CREB1','EWS'],
        ['CREB1','EWSR1'],
        ['CREB1','FUS'],
        ['CREB1','METTL21A'],
        ['CREB1','PARD3B'],
        ['CREB1','TMEM131'],
        ['CREB1','TRAK2'],
        ['CREB3L1','EWS'],
        ['CREB3L1','EWSR1'],
        ['CREB3L1','FUS'],
        ['CREB3L1','HNRNPA1'],
        ['CREB3L1','PAX5'],
        ['CREB3L1','STIP1'],
        ['CREB3L2','EWSR1'],
        ['CREB3L2','FUS'],
        ['CREB3L2','PPARG'],
        ['CREB3L3','EWS'],
        ['CREB3L3','EWSR1'],
        ['CREB3L4','CRTC2'],
        ['CREB5','CWC27'],
        ['CREB5','FGFR2'],
        ['CREB5','KALRN'],
        ['CREB5','KCNH2'],
        ['CREBBP','DEXI'],
        ['CREBBP','EP300'],
        ['CREBBP','ETV6'],
        ['CREBBP','GOLGA6L2'],
        ['CREBBP','KAT6A'],
        ['CREBBP','KMT2A'],
        ['CREBBP','MEFV'],
        ['CREBBP','MLL'],
        ['CREBBP','MORF'],
        ['CREBBP','MYST3'],
        ['CREBBP','MYST4'],
        ['CREBBP','NARFL'],
        ['CREBBP','NCOA2'],
        ['CREBBP','RUNXBP2'],
        ['CREBBP','SLX4'],
        ['CREBBP','SRGAP2B'],
        ['CREBBP','SRRM2'],
        ['CREBBP','TFAP4'],
        ['CREBBP','TRAP1'],
        ['CREBBP','ZNF384'],
        ['CREBZF','DLG2'],
        ['CREG2','TLK1'],
        ['CREM','EWSR1'],
        ['CREM','FUS'],
        ['CREM','MITF'],
        ['CRHR1','EPOR'],
        ['CRHR1','KANSL1'],
        ['CRHR1','SWSAP1'],
        ['CRIM1','MAP4K3'],
        ['CRKL','ERK2'],
        ['CRKL','KLHL22'],
        ['CRKL','MAPK1'],
        ['CRKRS','FBXL20'],
        ['CRKRS','TMEM104'],
        ['CRLF2','CSF2RA'],
        ['CRLF2','IGH@'],
        ['CRLF2','IGHD'],
        ['CRLF2','IGHM'],
        ['CRLF2','JAK2'],
        ['CRLF2','P2RY8'],
        ['CRLF2','TSLP'],
        ['CRLF3','SUZ12'],
        ['CRLF3','UTP6'],
        ['CRLS1','KCNK17'],
        ['CRLS1','PFN1'],
        ['CRLS1','ZFAT'],
        ['CRMP1','JAKMIP1'],
        ['CROCC','NBPF1'],
        ['CRTAC1','EXOSC1'],
        ['CRTC1','DOHH'],
        ['CRTC1','KLHL26'],
        ['CRTC1','MAML1'],
        ['CRTC1','MAML2'],
        ['CRTC1','SS18'],
        ['CRTC1','TRIM11'],
        ['CRTC2','MAML2'],
        ['CRTC3','MAML2'],
        ['CRTC3','PTEN'],
        ['CRTC3','SV2B'],
        ['CRYBA1','TAOK1'],
        ['CRYBB1','PLA2G6'],
        ['CRYBB2','KIAA1671'],
        ['CRYL1','IFT88'],
        ['CRYL1','PSPC1'],
        ['CRYL1','USP31'],
        ['CS','PVT1'],
        ['CS','STAT6'],
        ['CSAG1','F8'],
        ['CSB','ERCC6'],
        ['CSDE1','FAR2P3'],
        ['CSDE1','TERT'],
        ['CSF1','CSF1R'],
        ['CSF1','FN1'],
        ['CSF1','KCNMA1'],
        ['CSF1','S100A10'],
        ['CSF1','TRIM63'],
        ['CSF1','VCAM1'],
        ['CSF1R','IGF2R'],
        ['CSF1R','MEF2D'],
        ['CSF1R','PDGFRB'],
        ['CSF1R','PGC'],
        ['CSF1R','RBM6'],
        ['CSF1R','SSBP2'],
        ['CSF1R','TBL1XR1'],
        ['CSF2RA','IGH@'],
        ['CSF2RA','IGHD'],
        ['CSF2RA','IGHM'],
        ['CSF2RA','IL3RA'],
        ['CSF2RA','IL9R'],
        ['CSF3R','SPTAN1'],
        ['CSGALNACT2','RET'],
        ['CSGALNACT2','SLC16A9'],
        ['CSMD1','MEAF6'],
        ['CSMD1','PRKDC'],
        ['CSMD2','TMEM222'],
        ['CSMD3','MYC'],
        ['CSMD3','SAMD12'],
        ['CSNK1A1','LECT2'],
        ['CSNK1A1','PDE6A'],
        ['CSNK1A1','SCGB3A2'],
        ['CSNK1D','PGC'],
        ['CSNK1D','SCPEP1'],
        ['CSNK1D','SECTM1'],
        ['CSNK1D','ST6GALNAC1'],
        ['CSNK1E','DMC1'],
        ['CSNK1G1','DPYD'],
        ['CSNK1G1','INTS2'],
        ['CSNK1G1','MBOAT2'],
        ['CSNK1G1','ZNF609'],
        ['CSNK1G2','GANAB'],
        ['CSNK1G2','HSD17B13'],
        ['CSNK1G2','OAZ1'],
        ['CSNK1G2','PDE5A'],
        ['CSNK1G2-AS1','LPCAT2'],
        ['CSNK2A1','DDX39B'],
        ['CSNK2A1','PDGFRB'],
        ['CSNK2A1','SNRPB'],
        ['CT45A2','KMT2A'],
        ['CT45A2','MLL'],
        ['CTAGE5','GEMIN2'],
        ['CTAGE5','KHDRBS3'],
        ['CTAGE5','PSMA3'],
        ['CTAGE5','SIP1'],
        ['CTBP1','EWSR1'],
        ['CTBP2','CTNNB1'],
        ['CTBP2','DENND3'],
        ['CTBP2','MGMT'],
        ['CTBS','GNG5'],
        ['CTC1','TMEM117'],
        ['CTCF','ETO2'],
        ['CTCF','NFATC3'],
        ['CTCF','NUTF2'],
        ['CTCF','PARD6A'],
        ['CTCF','SLC7A6'],
        ['CTCF','ZDHHC1'],
        ['CTDP1','PQLC1'],
        ['CTDSP1','DNM3OS'],
        ['CTDSP2','DCUN1D4'],
        ['CTDSP2','DNM3OS'],
        ['CTDSP2','MARS'],
        ['CTDSP2','MON2'],
        ['CTDSP2','PLAG1'],
        ['CTDSP2','RAP1B'],
        ['CTDSP2','SLC6A15'],
        ['CTDSP2','XRCC6BP1'],
        ['CTGLF10P','PFKM'],
        ['CTIF','MOB2'],
        ['CTL4','NEU1'],
        ['CTNNA1','DNAH14'],
        ['CTNNA1','GLRA1'],
        ['CTNNA1','JMJD1B'],
        ['CTNNA1','KDM3B'],
        ['CTNNA1','SEC24A'],
        ['CTNNA2','LCMT1'],
        ['CTNNA3','FGFR2'],
        ['CTNNAL1','TMEM245'],
        ['CTNNB1','EGFR'],
        ['CTNNB1','ETV6'],
        ['CTNNB1','FGFR2'],
        ['CTNNB1','GREB1'],
        ['CTNNB1','GSN'],
        ['CTNNB1','LIFR'],
        ['CTNNB1','PLAG1'],
        ['CTNNB1','ULK4'],
        ['CTNNB1','USP6'],
        ['CTNNBIP1','TRIT1'],
        ['CTNNBIP1','UBE4B'],
        ['CTNNBL1','VSTM2L'],
        ['CTNND1','KMT2A'],
        ['CTNND1','TMEM135'],
        ['CTNND2','NRBP1'],
        ['CTNND2','TERT'],
        ['CTPS1','NFYC'],
        ['CTPS2','SEC31A'],
        ['CTRC','NTRK1'],
        ['CTSB','EIF2AK1'],
        ['CTSB','EIF4A2'],
        ['CTSB','PXK'],
        ['CTSC','RAB38'],
        ['CTSD','IFITM10'],
        ['CTSD','JUP'],
        ['CTSD','KRT19'],
        ['CTSF','PRSS44'],
        ['CTSLP5','RCOR2'],
        ['CTSZ','KRT19'],
        ['CTTN','ERBB2'],
        ['CTTNBP2','ST7'],
        ['CTTNBP2','ST7-AS1'],
        ['CTTNBP2','ST7-OT4'],
        ['CTU2','SP3'],
        ['CTXN3','PITPNM2'],
        ['CUBN','ERRFI1'],
        ['CUBN','MAP3K8'],
        ['CUL1','DPP6'],
        ['CUL4A','GRK1'],
        ['CUL4B','DAPK2'],
        ['CUTA','SLC22A1'],
        ['CUX1','FGFR1'],
        ['CUX1','MAP2K1'],
        ['CUX1','NUTM1'],
        ['CUX1','PRKRIP1'],
        ['CUX1','RAF1'],
        ['CUX1','RET'],
        ['CUX1','RPN2'],
        ['CUX1','ZNF398'],
        ['CUX1','ZNF706'],
        ['CUX2','ZNF140'],
        ['CWC22','MAP4K3'],
        ['CWC22','WNK1'],
        ['CWC25','PIP4K2B'],
        ['CWC25','USP32'],
        ['CWC27','NSD1'],
        ['CXCL11','ZZZ3'],
        ['CXCL12','CXCR4'],
        ['CXCL17','PSG3'],
        ['CXCL9','SOD1'],
        ['CXCR4','CXCR7'],
        ['CXCR4','MAML2'],
        ['CXCR4','PVT1'],
        ['CXCR4','TCF3'],
        ['CXCR4','UBXN4'],
        ['CXCR4','ZEB2'],
        ['CXCR7','HMGA2'],
        ['CXORF15','SYAP1'],
        ['CXORF21','SORT1'],
        ['CXORF25','IL2RG'],
        ['CXORF40A','IDS'],
        ['CXORF67','MBTD1'],
        ['CXXC5','MN1'],
        ['CYB561D2','PBRM1'],
        ['CYB5B','NUP93'],
        ['CYB5R1','NUAK2'],
        ['CYB5R3','FDCSP'],
        ['CYFIP2','XRCC4'],
        ['CYHR1','SEPT9'],
        ['CYP11A1','TFE3'],
        ['CYP11B1','CYP11B2'],
        ['CYP19A1','SCARB2'],
        ['CYP19A1','TMOD3'],
        ['CYP19A1','TRPM7'],
        ['CYP1A2','POLR2G'],
        ['CYP21A','CYP21B'],
        ['CYP21A1P','CYP21A2'],
        ['CYP21A2','IDI1'],
        ['CYP2A6','PAK4'],
        ['CYP2C19','FAM190B'],
        ['CYP2C19','SNX9'],
        ['CYP2E1','FGA'],
        ['CYP2U1','PERLD1'],
        ['CYP39A1','EIF3K'],
        ['CYP4F35P','IFFO1'],
        ['CYP4X1','CYP4Z2P'],
        ['CYP4Z1','MSH2'],
        ['CYP51A1','GBAS'],
        ['CYP7B1','NCOA2'],
        ['CYTH1','EIF3H'],
        ['CYTH1','EXOC7'],
        ['CYTH1','PRPSAP1'],
        ['CYTH3','PKN3'],
        ['CYTIP','ERMN'],
        ['CYTIP','PMP22'],
        ['CYTSB','PDGFRB'],
        ['D10S170','PDGFRB'],
        ['D10S170','RET'],
        ['D2HGDH','GAL3ST2'],
        ['DA750114','TRBC2'],
        ['DAAM1','MNAT1'],
        ['DAAM2','SLC6A6'],
        ['DAB1','RYR3'],
        ['DAB2','ESR1'],
        ['DAB2','FRYL'],
        ['DAB2IP','HCFC1R1'],
        ['DAB2IP','KMT2A'],
        ['DAB2IP','MLL'],
        ['DAB2IP','NTRK2'],
        ['DAB2IP','SUSD1'],
        ['DACH1','PAX5'],
        ['DACH2','PAX5'],
        ['DACT3','TP53'],
        ['DAD1','GPI'],
        ['DAPK1','KCTD2'],
        ['DAPK1','NTRK2'],
        ['DAPK1','TRIM52'],
        ['DAPK3','FERMT1'],
        ['DAXX','KIFC3'],
        ['DAZ4','MAP4K1'],
        ['DAZAP1','DYM'],
        ['DAZAP1','MEF2D'],
        ['DAZAP2','LOC57228'],
        ['DAZL','RAF1'],
        ['DBN1','GRK6'],
        ['DBNDD1','NAT8L'],
        ['DBX1','PAX5'],
        ['DCAF12','UBAP2'],
        ['DCAF5','ETV6'],
        ['DCAF6','MPZL1'],
        ['DCAF8','UHMK1'],
        ['DCBLD2','STXBP5L'],
        ['DCC','IGF1R'],
        ['DCC','ROCK1'],
        ['DCC','SLIT3'],
        ['DCC','SYNJ2'],
        ['DCDC2','ZFAND3'],
        ['DCK','FRAS1'],
        ['DCK','PRRX1'],
        ['DCLK1','ELF1'],
        ['DCLK1','SERPINE3'],
        ['DCLK2','UST'],
        ['DCN','LRRC8A'],
        ['DCP1A','MLL'],
        ['DCP1B','KMT2A'],
        ['DCP2','HIF1A'],
        ['DCP2','IGH@'],
        ['DCP2','UBR1'],
        ['DCPS','KMT2A'],
        ['DCPS','MLL'],
        ['DCPS','ST3GAL4'],
        ['DCPS','YAP1'],
        ['DCST2','PMVK'],
        ['DCTD','ECE2'],
        ['DCTN1','NTRK1'],
        ['DCTN2','FGFR2'],
        ['DCUN1D1','MCCC1'],
        ['DCUN1D3','GSG1L'],
        ['DCUN1D4','RAB3IP'],
        ['DCUN1D4','SCN2A'],
        ['DCX','PASD1'],
        ['DDA1','FAM129C'],
        ['DDAH1','SRPK1'],
        ['DDC','IKZF1'],
        ['DDH','RHEBL1'],
        ['DDHD2','NAT1'],
        ['DDIT2','FUS'],
        ['DDIT3','ERG'],
        ['DDIT3','EWS'],
        ['DDIT3','EWSR1'],
        ['DDIT3','FEV'],
        ['DDIT3','FUS'],
        ['DDIT3','GLI1'],
        ['DDIT3','MARS'],
        ['DDR1','GFAP'],
        ['DDR1','PAK1'],
        ['DDR2','FMO4'],
        ['DDR2','NTRK1'],
        ['DDR2','TRABD'],
        ['DDR2','UAP1'],
        ['DDX1','MALAT1'],
        ['DDX1','NBAS'],
        ['DDX10','HOXA11'],
        ['DDX10','HOXA13'],
        ['DDX10','HOXA9'],
        ['DDX10','HOXC11'],
        ['DDX10','HOXD11'],
        ['DDX10','HOXD13'],
        ['DDX10','NUP98'],
        ['DDX10','PMX1'],
        ['DDX10','RAP1GDS1'],
        ['DDX10','TOP1'],
        ['DDX11L16','MMP23A'],
        ['DDX17','ETV1'],
        ['DDX19A','SPATA31D1'],
        ['DDX19B','SF3B3'],
        ['DDX20','TBX15'],
        ['DDX21','MRPS18C'],
        ['DDX25','ZNF131'],
        ['DDX26B','IRF4'],
        ['DDX3X','DEFB127'],
        ['DDX3X','MLLT10'],
        ['DDX3X','PRKD1'],
        ['DDX3X','USP9X'],
        ['DDX42','MAP3K3'],
        ['DDX42','RPS6KB1'],
        ['DDX47','RUNX1'],
        ['DDX49','NF2'],
        ['DDX5','DEPDC6'],
        ['DDX5','DEPTOR'],
        ['DDX5','ETV4'],
        ['DDX5','IQCG'],
        ['DDX5','KLK2'],
        ['DDX5','LINC00511'],
        ['DDX5','SRSF1'],
        ['DDX52','RPL38'],
        ['DDX55','GTF2H3'],
        ['DDX6','FOXR1'],
        ['DDX6','IGH@'],
        ['DDX6','USP28'],
        ['DDX60','FREM3'],
        ['DDX8','FOXR1'],
        ['DEFB103B','PRKDC'],
        ['DEFB112','NOMO3'],
        ['DEGS2','IGHV@'],
        ['DEK','NUP214'],
        ['DEK','PRRC2B'],
        ['DEK','SET'],
        ['DELTA','NTRK1'],
        ['DENND1A','ZCCHC7'],
        ['DENND1B','ZCCHC7'],
        ['DENND1C','SURF1'],
        ['DENND2A','GTF2IRD1'],
        ['DENND2A','HIPK2'],
        ['DENND2C','HIPK1'],
        ['DENND3','LPCAT1'],
        ['DENND3','PTK2'],
        ['DENND5B','NOTCH2'],
        ['DENND5B','NUP54'],
        ['DEPDC1B','ELOVL7'],
        ['DEPDC1B','PDE4D'],
        ['DEPDC7','NR3C2'],
        ['DEPTOR','TATDN1'],
        ['DERA','GLI1'],
        ['DERL1','TSC1'],
        ['DGAT2','WASH7P'],
        ['DGCR5','GGT2'],
        ['DGCR8','MAGED4B'],
        ['DGKA','ZNF273'],
        ['DGKB','ETV1'],
        ['DGKB','KLK2'],
        ['DGKB','MAPK9'],
        ['DGKB','MIPOL1'],
        ['DGKB','PTEN'],
        ['DGKB','SNX13'],
        ['DGKD','SPP2'],
        ['DGKG','ETV5'],
        ['DGKG','SPERT'],
        ['DGKG','TMPRSS2'],
        ['DGKH','ZFAND3'],
        ['DGKI','EPHA5'],
        ['DGKI','PTN'],
        ['DGUOK','TET3'],
        ['DHCR24','GLIS1'],
        ['DHCR24','MFSD4'],
        ['DHDDS','RPS6KA1'],
        ['DHFR','PABPC1'],
        ['DHH','RHEBL1'],
        ['DHODH','NDUFAB1'],
        ['DHPS','SMARCA4'],
        ['DHRS11','GGNBP2'],
        ['DHRS4','STEAP4'],
        ['DHRS7C','POTEA'],
        ['DHX16','NUP153'],
        ['DHX30','RNASEH2B'],
        ['DHX33','NLRP1'],
        ['DHX35','FAM83D'],
        ['DHX35','ITCH'],
        ['DHX35','LPAR1'],
        ['DHX37','NCOR2'],
        ['DHX57','MAP4K3'],
        ['DHX57','TMEM178'],
        ['DHX8','ETV4'],
        ['DHX9','NPL'],
        ['DHX9','RAF1'],
        ['DHX9','USP25'],
        ['DIAPH1','NTRK1'],
        ['DIAPH1','PDGFRB'],
        ['DIAPH3','RPS6KB1'],
        ['DIDO1','KIAA0406'],
        ['DIDO1','TTI1'],
        ['DIEXF','SYT14'],
        ['DIP2A','UBAC2'],
        ['DIP2B','LINC00330'],
        ['DIP2B','NRG1'],
        ['DIP2B','TAF8'],
        ['DIP2B','TMBIM6'],
        ['DIP2C','LARGE'],
        ['DIP2C','MAP3K8'],
        ['DIP2C','PARD3'],
        ['DIP2C','PDGFRA'],
        ['DIP2C','ZMYND11'],
        ['DIRAS3','GNLY'],
        ['DIRC2','TIA1'],
        ['DIRC3','HSPBAP1'],
        ['DISC1','DISC1FP1'],
        ['DISC1','OBSCN'],
        ['DISC1','TSNAX'],
        ['DISC1FP1','TSNAX'],
        ['DISP1','SNX5'],
        ['DKFZP586P0123','SHANK2'],
        ['DLEC1','ODZ4'],
        ['DLEU1','RB1'],
        ['DLEU2','NDFIP2'],
        ['DLEU2','PSPC1'],
        ['DLEU2','RNASEH2B'],
        ['DLG1','PAK2'],
        ['DLG1','PRKCI'],
        ['DLG2','KPNA1'],
        ['DLG2','MYCBP2'],
        ['DLG2','SEMA6C'],
        ['DLG2','UNC45B'],
        ['DLG2','ZFP91'],
        ['DLG5','RET'],
        ['DLGAP1','GRLF1'],
        ['DLGAP4','SRC'],
        ['DLK2','PTK7'],
        ['DMBX1','MAST2'],
        ['DMC1','KRT18P40'],
        ['DMD','IL1RAPL1'],
        ['DMD','PHEX'],
        ['DMD','SSR1'],
        ['DMD','TAB3'],
        ['DMGDH','JMY'],
        ['DMKN','LGI4'],
        ['DMPK','SYMPK'],
        ['DMRT1','RFX3'],
        ['DMRT3','FBXO4'],
        ['DMRTA2','PAX5'],
        ['DNAH10','KDM2B'],
        ['DNAH11','LANCL2'],
        ['DNAH11','RAPGEF5'],
        ['DNAH14','IKZF1'],
        ['DNAH5','TRA2B'],
        ['DNAH5','TRIO'],
        ['DNAH6','LRRC8A'],
        ['DNAH8','TP53'],
        ['DNAH9','MAP2K4'],
        ['DNAH9','RABEP1'],
        ['DNAJA2','NETO2'],
        ['DNAJA3','ZDHHC16'],
        ['DNAJA3','ZNF555'],
        ['DNAJB1','PRKACA'],
        ['DNAJB1','PRKCA'],
        ['DNAJB6','PLAG1'],
        ['DNAJC1','MLLT10'],
        ['DNAJC1','RERGL'],
        ['DNAJC11','RERE'],
        ['DNAJC12','FGFR2'],
        ['DNAJC13','ZNF212'],
        ['DNAJC18','SIL1'],
        ['DNAJC18','ST8SIA4'],
        ['DNAJC2','POLR2J2'],
        ['DNAJC3','RNASEH2B-AS1'],
        ['DNAJC5','XKR7'],
        ['DNAJC5','ZBTB46'],
        ['DNAJC6','PDE1C'],
        ['DNAJC6','ROR1'],
        ['DNAJC7','ERBB2'],
        ['DNAJC8','RALGPS2'],
        ['DNAL4','PTRH2'],
        ['DNASE1L3','PATL1'],
        ['DNASE2','MAST1'],
        ['DND1','WDR55'],
        ['DNER','ELL2'],
        ['DNER','NTRK1'],
        ['DNER','NTRK2'],
        ['DNER','NTRK3'],
        ['DNER','PSMD1'],
        ['DNM1','FGFR2'],
        ['DNM1P46','ZKSCAN1'],
        ['DNM2','NMRK2'],
        ['DNM3','FGFR2'],
        ['DNM3','METTL13'],
        ['DNM3','UHRF1'],
        ['DNMBP','TACC2'],
        ['DNMT1','KEAP1'],
        ['DNMT3B','MDM4'],
        ['DOCK1','DPCD'],
        ['DOCK1','STK32C'],
        ['DOCK11','THOC2'],
        ['DOCK4','IMMP2L'],
        ['DOCK5','STARD3'],
        ['DOCK6','KDM4B'],
        ['DOCK6','RAVER1'],
        ['DOCK6','ULK2'],
        ['DOCK7','OLR1'],
        ['DOCK8','JAK2'],
        ['DOCK9','STK24'],
        ['DOG1','KIT'],
        ['DOK3','NAV3'],
        ['DOK5','FER1L4'],
        ['DOK5','STARD3'],
        ['DOM','FGFR2'],
        ['DOT1L','HES6'],
        ['DPF2','METTL12'],
        ['DPF2','POLA2'],
        ['DPH7','PNPLA7'],
        ['DPH7','PTP4A3'],
        ['DPM1','GRID1'],
        ['DPM1','MTMR2'],
        ['DPP3','PPP6R3'],
        ['DPP4','PDK1'],
        ['DPP9','PAX2'],
        ['DPP9','PLIN3'],
        ['DPP9','PPP6R3'],
        ['DPRX','TTYH1'],
        ['DPY19L2','PPFIA2'],
        ['DPY19L2','PPM1H'],
        ['DPY19L2','TUBA1A'],
        ['DPY19L4','PVT1'],
        ['DPYD','ETV1'],
        ['DPYS','RIMS2'],
        ['DPYS','SSR1'],
        ['DPYSL2','GFAP'],
        ['DPYSL2','NRG1'],
        ['DR1','DR4'],
        ['DROSHA','NRBP1'],
        ['DSCAM','DYRK1A'],
        ['DSCAM','ERG'],
        ['DSCAM','RUNX1'],
        ['DSCAM','TMPRSS2'],
        ['DSCAML1','FXYD2'],
        ['DSCAML1','KMT2A'],
        ['DSCR3','FBLN7'],
        ['DSE','FHL5'],
        ['DSE','HACE1'],
        ['DSP','RREB1'],
        ['DSP','STK24'],
        ['DST','MALAT1'],
        ['DST','RAB23'],
        ['DSTN','NLK'],
        ['DSTYK','NAV1'],
        ['DSTYK','NEK7'],
        ['DSTYK','NUCKS1'],
        ['DSTYK','PIGK'],
        ['DSTYK','PIK3C2B'],
        ['DSTYK','THSD4'],
        ['DTD1','PDGFRB'],
        ['DTD1','PHRF1'],
        ['DTD1','RRBP1'],
        ['DTNB','FARP2'],
        ['DTNB','SPTLC1'],
        ['DTNB','VEGFA'],
        ['DTNBP1','FRK'],
        ['DUS2L','PSKH1'],
        ['DUSP10','PRDM16'],
        ['DUSP10','TSPAN15'],
        ['DUSP22','IGH@'],
        ['DUSP22','IRF4'],
        ['DUSP22','MUM1'],
        ['DUSP22','PTK7'],
        ['DUSP22','UST'],
        ['DUSP26','GAK'],
        ['DUX4','ERG'],
        ['DUX4','EWSR1'],
        ['DUX4','IGH@'],
        ['DUX4','MIR3687-1'],
        ['DVL2','TFE3'],
        ['DXO','SHISA5'],
        ['DYDC2','TSPAN14'],
        ['DYM','MYO5B'],
        ['DYNC1H1','SIAE'],
        ['DYNC1I1','IMMP2L'],
        ['DYNC1LI1','GADL1'],
        ['DYNC2H1','NTRK1'],
        ['DYNC2H1','NTRK2'],
        ['DYNC2H1','NTRK3'],
        ['DYNC2H1','YAP1'],
        ['DYNLL1','SMS'],
        ['DYRK1A','ERG'],
        ['DYRK1A','HUNK'],
        ['DYRK1A','KDM4B'],
        ['DYRK1A','RUNX1'],
        ['DYRK1A','TTC3'],
        ['DYRK1A','ZNF709'],
        ['DYRK2','HMGA2'],
        ['DYRK2','PKP1'],
        ['DYRK2','SMS'],
        ['DYRK4','RAD51AP1'],
        ['DYX1C1','SGK3'],
        ['DZANK1','FGFR2'],
        ['DZIP1','FGFR2'],
        ['DZIP1','GPR180'],
        ['E1AF','EWS'],
        ['E1AF','EWSR1'],
        ['E2A','HLF'],
        ['E2A','PBX1'],
        ['E2A','TFPT'],
        ['E2F1','SRF'],
        ['E2F1','TAT'],
        ['E2F3','ER'],
        ['E2F3','LINC00340'],
        ['E2F3','TP53'],
        ['E2F5','FER1L6'],
        ['E2F6','FKBP4'],
        ['E2F6','GREB1'],
        ['E2F6','STK36'],
        ['E2F7','NAV3'],
        ['E4F1','RAB26'],
        ['EAAT2','SLC1A2'],
        ['EAPP','REST'],
        ['EBF','HMGA2'],
        ['EBF1','ETV6'],
        ['EBF1','FBXL17'],
        ['EBF1','FEM1C'],
        ['EBF1','HMGA2'],
        ['EBF1','IGH@'],
        ['EBF1','JAK2'],
        ['EBF1','KCNN2'],
        ['EBF1','LOC204010'],
        ['EBF1','PDGFR'],
        ['EBF1','PDGFRB'],
        ['EBF1','PVT1'],
        ['EBF1','XIST'],
        ['EBF3','TSPAN15'],
        ['ECE2','PAK2'],
        ['ECHDC1','FYN'],
        ['ECT2','FABP6'],
        ['ECT2','KALRN'],
        ['EDARADD','RNF10'],
        ['EDC4','KMT2A'],
        ['EDC4','MLL'],
        ['EDDM13','ZNF71'],
        ['EDEM2','OSBPL2'],
        ['EDIL3','FBXW11'],
        ['EDIL3','MKLN1'],
        ['EED','RARA'],
        ['EEF1A1','MALAT1'],
        ['EEF1A1','PDCD1LG2'],
        ['EEF1A1','PLAG1'],
        ['EEF1A1','RPL3'],
        ['EEF1A1','RPL32'],
        ['EEF1A1','SMG1'],
        ['EEF1A1','TG'],
        ['EEF1A1','VIM'],
        ['EEF1D','NAPRT'],
        ['EEF1D','NAPRT1'],
        ['EEF1D','SDC4'],
        ['EEF1DP3','FRY'],
        ['EEF1E1','RREB1'],
        ['EEF2','ENHO'],
        ['EEF2','MALAT1'],
        ['EEF2','MVP'],
        ['EEF2','SLC25A10'],
        ['EEF2','SLC25A42'],
        ['EEF2K','KIAA0430'],
        ['EEF2K','NUP210L'],
        ['EEFSEC','KMT2A'],
        ['EEFSEC','MLL'],
        ['EEFSEC','PDHX'],
        ['EEFSEC','RAB7A'],
        ['EEN','MLL'],
        ['EEN','SH3GL1'],
        ['EFCAB2','SMYD3'],
        ['EFCAB2','ZNF238'],
        ['EFCAB3','RPS6KB1'],
        ['EFCAB4A','WDTC1'],
        ['EFCAB5','KSR1'],
        ['EFCAB6','HMGA2'],
        ['EFCAB6','NF2'],
        ['EFCAB6','SREBF2'],
        ['EFL1','ETV6'],
        ['EFL1','MAGI1'],
        ['EFL1','THSD4'],
        ['EFNA1','PVT1'],
        ['EFNA3','PIK3C2G'],
        ['EFNA5','PCDHB7'],
        ['EFNA5','RICTOR'],
        ['EFNB3','TP53'],
        ['EFR3A','TP53'],
        ['EFTUD1','IGF1R'],
        ['EFTUD2','KIF18B'],
        ['EFTUD2','MAP3K14'],
        ['EFTUD2','NAGLU'],
        ['EFTUD2','SHANK2'],
        ['EGF','FN1'],
        ['EGFL6','RAB9A'],
        ['EGFL7','FOSB'],
        ['EGFR','ERBB2'],
        ['EGFR','ERP44'],
        ['EGFR','FANCC'],
        ['EGFR','FKBP9L'],
        ['EGFR','GASP'],
        ['EGFR','GBAS'],
        ['EGFR','GNS'],
        ['EGFR','HDAC9'],
        ['EGFR','HER2'],
        ['EGFR','HMGA2'],
        ['EGFR','IGH@'],
        ['EGFR','KIAA0573'],
        ['EGFR','KIF5B'],
        ['EGFR','KRAS'],
        ['EGFR','LANCL2'],
        ['EGFR','MDM2'],
        ['EGFR','MED15'],
        ['EGFR','MET'],
        ['EGFR','MYRFL'],
        ['EGFR','PDE1C'],
        ['EGFR','PDGFC'],
        ['EGFR','POLD1'],
        ['EGFR','PPARGC1A'],
        ['EGFR','PSPH'],
        ['EGFR','PURB'],
        ['EGFR','RAD51'],
        ['EGFR','ROS1'],
        ['EGFR','SDK1'],
        ['EGFR','SEC61G'],
        ['EGFR','SEPT14'],
        ['EGFR','SOX2-OT'],
        ['EGFR','TIG1'],
        ['EGFR','TP53'],
        ['EGFR','VOPP1'],
        ['EGFR','VSTM2A'],
        ['EGFR','VWC2'],
        ['EGFR','WIF1'],
        ['EGFR','ZNF267'],
        ['EGLN1','RBBP9'],
        ['EGR1','GRIA1'],
        ['EGR1','RPS14'],
        ['EHBP1','MET'],
        ['EHBP1','NRF1'],
        ['EHF','NFIA'],
        ['EHF','WT1'],
        ['EHMT1','EWSR1'],
        ['EHMT1','ZNF384'],
        ['EHMT2','SLC44A4'],
        ['EHMT2','SRPK1'],
        ['EIF1AX','GLIS2'],
        ['EIF1B','SRPK1'],
        ['EIF2AK1','OGDH'],
        ['EIF2AK3','MYH9'],
        ['EIF2AK3','PIEZO1'],
        ['EIF2AK3','PRKD3'],
        ['EIF2AK4','EPB42'],
        ['EIF2AK4','PKM'],
        ['EIF2C2','PTK2'],
        ['EIF2C3','SPDYA'],
        ['EIF2C3','THRAP3'],
        ['EIF2C3','ZP2'],
        ['EIF2S1','RBM25'],
        ['EIF2S2','NTRK3'],
        ['EIF2S3','TXNDC11'],
        ['EIF3A','POSTN'],
        ['EIF3A','RNF111'],
        ['EIF3A','SFXN4'],
        ['EIF3D','MYH9'],
        ['EIF3D','NF2'],
        ['EIF3D','TXN2'],
        ['EIF3E','PVT1'],
        ['EIF3E','RSPO2'],
        ['EIF3E','RSPO3'],
        ['EIF3E','RUNX1'],
        ['EIF3H','FAM65C'],
        ['EIF3H','MRPS28'],
        ['EIF3H','NUDCD1'],
        ['EIF3H','TPD52'],
        ['EIF3H','TRPS1'],
        ['EIF4A1','PHOSPHO1'],
        ['EIF4A2','ETV5'],
        ['EIF4A2','FGFR2'],
        ['EIF4A2','MECOM'],
        ['EIF4A3','TSPEAR'],
        ['EIF4B','ZNF384'],
        ['EIF4E1B','MTMR12'],
        ['EIF4E2','HJURP'],
        ['EIF4E3','FOXP1'],
        ['EIF4E3','IQSEC1'],
        ['EIF4ENIF1','TEX14'],
        ['EIF4G2','MALAT1'],
        ['EIF4G2','NFATC3'],
        ['EIF4G2','NUCB2'],
        ['EIF4G2','UPK2'],
        ['EIF4G3','GNL2'],
        ['EIF4G3','MAN1C1'],
        ['EIF4G3','SESN2'],
        ['EIF4G3','ZNF683'],
        ['EIF4H','GTF2I'],
        ['EIF5','MARK3'],
        ['EIF5A','HMGN2'],
        ['EIF5A','USP6'],
        ['ELAVL1','FASN'],
        ['ELAVL1','TYK2'],
        ['ELAVL2','IFT74'],
        ['ELAVL2','PLGRKT'],
        ['ELAVL3','FGFR3'],
        ['ELE1','RET'],
        ['ELF1','ETV6'],
        ['ELF1','ROCK1'],
        ['ELF3','RNPEP'],
        ['ELF4','ERG'],
        ['ELF4','FUS'],
        ['ELF4','HERPUD1'],
        ['ELF4','NDRG1'],
        ['ELF5','EWSR1'],
        ['ELF5','ZFPM2'],
        ['ELK1','ETS1'],
        ['ELK1','GPRC5A'],
        ['ELK3','NTN4'],
        ['ELK4','ERG'],
        ['ELK4','ETV1'],
        ['ELK4','REPIN1'],
        ['ELK4','SLC45A3'],
        ['ELK4','TMPRSS2'],
        ['ELKS','EWSR1'],
        ['ELKS','RET'],
        ['ELL','EPS15'],
        ['ELL','HNRNPD'],
        ['ELL','KMT2A'],
        ['ELL','MLL'],
        ['ELMO1','ETV6'],
        ['ELMO1','NLK'],
        ['ELMO2','RAE1'],
        ['ELMOD1','LTB4R2'],
        ['ELMOD3','RB1'],
        ['ELMOD3','RSAD2'],
        ['ELMOD3','SH2D6'],
        ['ELN','ETV6'],
        ['ELN','PAX5'],
        ['ELOVL1','MED8'],
        ['ELOVL2','MED29'],
        ['ELOVL4','USP15'],
        ['ELOVL5','PMP22'],
        ['ELOVL6','POMK'],
        ['EMB','S100A13'],
        ['EMC1','ETV3'],
        ['EMC2','RSPO2'],
        ['EMC7','NUTM1'],
        ['EMID2','GGCT'],
        ['EMILIN2','PDGFD'],
        ['EML1','EVL'],
        ['EML1','RAGE'],
        ['EML4','MET'],
        ['EML4','NTRK1'],
        ['EML4','NTRK2'],
        ['EML4','NTRK3'],
        ['EML4','RET'],
        ['EML5','HMGA2'],
        ['EMP1','RPS24'],
        ['EMP1','RUNX1'],
        ['EMP3','IFT52'],
        ['EMR1','TP53'],
        ['ENAH','MLL'],
        ['ENL','MLL'],
        ['ENO1','ERBB2'],
        ['ENO1','PDK4'],
        ['ENO2','PIEZO2'],
        ['ENOPH1','MRPS18C'],
        ['ENOPH1','TSNAX'],
        ['ENOSF1','YES1'],
        ['ENOX1','TYRO3'],
        ['ENOX2','ODZ1'],
        ['ENPP1','TESK2'],
        ['ENPP1','TP53'],
        ['ENPP2','FGFR2'],
        ['ENPP2','PLEC'],
        ['ENPP2','PLEC1'],
        ['ENTPD1','PDLIM1'],
        ['ENTPD1','RNASEH2B'],
        ['ENTPD3','RBCK1'],
        ['ENTPD4','PSD3'],
        ['ENTPD5','FAM161B'],
        ['ENTPD7','IDE'],
        ['EP','PARM1'],
        ['EP300','KAT6A'],
        ['EP300','KMT2A'],
        ['EP300','MLL'],
        ['EP300','MOZ'],
        ['EP300','RUNXBP2'],
        ['EP300','ZNF384'],
        ['EP400','KIF17'],
        ['EP400','NCOR2'],
        ['EP400','PHF1'],
        ['EP400','TRIT1'],
        ['EP400','YEATS4'],
        ['EPB41L2','NFIB'],
        ['EPB41L2','OSTM1'],
        ['EPB41L2','PTPRK'],
        ['EPB41L2','TBCB'],
        ['EPB41L5','PCDP1'],
        ['EPB41L5','ZDHHC5'],
        ['EPC1','KDM2B'],
        ['EPC1','PHF1'],
        ['EPC1','SUZ12'],
        ['EPC2','MORC1'],
        ['EPC2','MORC1-AS1'],
        ['EPC2','PHF1'],
        ['EPC2','SCN2A'],
        ['EPC2','SYNCRIP'],
        ['EPCAM','MSH2'],
        ['EPCAM','NUPL2'],
        ['EPHA1','USP32'],
        ['EPHA2','FBXO42'],
        ['EPHA2','HCK'],
        ['EPHA2','PREX1'],
        ['EPHA3','LCLAT1'],
        ['EPHA3','LRRFIP2'],
        ['EPHA3','RBFOX1'],
        ['EPHA4','POU2F1'],
        ['EPHA5','HOXB6'],
        ['EPHA5','MGA'],
        ['EPHA6','NPEPPS'],
        ['EPHA6','PPM1D'],
        ['EPHA6','TMEM39A'],
        ['EPHA7','MYH11'],
        ['EPHA7','NDE1'],
        ['EPHA7','NPDC1'],
        ['EPHA7','PIGH'],
        ['EPHB1','GMPS'],
        ['EPHB1','MOBKL1B'],
        ['EPHB1','STAG2'],
        ['EPHB2','NBPF3'],
        ['EPHB2','NTRK1'],
        ['EPHB2','NTRK2'],
        ['EPHB2','NTRK3'],
        ['EPHB2','PDZD4'],
        ['EPHB3','PAX2'],
        ['EPHB3','PGC'],
        ['EPHB4','MARCH10'],
        ['EPHB4','PGC'],
        ['EPN1','NOVA2'],
        ['EPO','TPO'],
        ['EPOR','FGFR2'],
        ['EPOR','IGH@'],
        ['EPOR','IGHV3-73'],
        ['EPOR','IGHV@'],
        ['EPOR','IGK@'],
        ['EPOR','LAIR1'],
        ['EPOR','LINC02210'],
        ['EPOR','LINC02210_'],
        ['EPRS','OTUD7B'],
        ['EPS15','FAF1'],
        ['EPS15','KMT2A'],
        ['EPS15','MLL'],
        ['EPS15','MTA1'],
        ['EPS15','NTRK1'],
        ['EPS15','NTRK2'],
        ['EPS15','NTRK3'],
        ['EPS8L1','OGFR'],
        ['EPT1','GUCY1A3'],
        ['EPYC','UHRF1BP1'],
        ['ERAS','USP9X'],
        ['ERBB','IKZF2'],
        ['ERBB','NRG1'],
        ['ERBB2','ERBB3'],
        ['ERBB2','FKBP10'],
        ['ERBB2','GPATCH8'],
        ['ERBB2','GRB7'],
        ['ERBB2','GTF2E2'],
        ['ERBB2','IFLTD1'],
        ['ERBB2','IFT20'],
        ['ERBB2','IGH@'],
        ['ERBB2','IGHG1'],
        ['ERBB2','IKZF3'],
        ['ERBB2','JUP'],
        ['ERBB2','KRT20'],
        ['ERBB2','KRT39'],
        ['ERBB2','KRTAP1-4'],
        ['ERBB2','LAMA3'],
        ['ERBB2','LTBP4'],
        ['ERBB2','MED1'],
        ['ERBB2','MTSS1'],
        ['ERBB2','MUC6'],
        ['ERBB2','NOS2'],
        ['ERBB2','PARN'],
        ['ERBB2','PGAP3'],
        ['ERBB2','PPP1R1B'],
        ['ERBB2','PSMB3'],
        ['ERBB2','PVT1'],
        ['ERBB2','RARA'],
        ['ERBB2','SDK2'],
        ['ERBB2','SHC1'],
        ['ERBB2','SLC29A3'],
        ['ERBB2','SMIM18'],
        ['ERBB2','SPTBN2'],
        ['ERBB2','SRCIN1'],
        ['ERBB2','TADA2A'],
        ['ERBB2','WIPF2'],
        ['ERBB4','EZR'],
        ['ERBB4','FN1'],
        ['ERBB4','IKZF1'],
        ['ERBB4','IKZF2'],
        ['ERBB4','KIF5B'],
        ['ERBB4','LCLAT1'],
        ['ERBB4','MFF'],
        ['ERBB4','NUMA1'],
        ['ERBB4','PHLDA3'],
        ['ERC1','EZR'],
        ['ERC1','FAM222B'],
        ['ERC1','FGFR2'],
        ['ERC1','NTRK1'],
        ['ERC1','PDGFRB'],
        ['ERC1','PIK3C2G'],
        ['ERC1','RET'],
        ['ERC1','RET1'],
        ['ERC1','ROS1'],
        ['ERC1','WNK1'],
        ['ERC2','RAF1'],
        ['ERCC1','ZNF600'],
        ['ERCC2','KLC3'],
        ['ERCC2','MARK4'],
        ['ERCC2','POU6F2'],
        ['ERF','GSK3A'],
        ['ERG','ETS2'],
        ['ERG','ETV1'],
        ['ERG','ETV4'],
        ['ERG','ETV5'],
        ['ERG','ETV6'],
        ['ERG','EWS'],
        ['ERG','EWSR1'],
        ['ERG','FEV'],
        ['ERG','FKBP5'],
        ['ERG','FOXO1'],
        ['ERG','FUS'],
        ['ERG','GRHL2'],
        ['ERG','HERPUD1'],
        ['ERG','HLCS'],
        ['ERG','HMGN1'],
        ['ERG','HNRNPH1'],
        ['ERG','IFNGR2'],
        ['ERG','KCNJ6'],
        ['ERG','LINC00240'],
        ['ERG','MAP3K7CL'],
        ['ERG','MIR99AHG'],
        ['ERG','MYC'],
        ['ERG','NDRG1'],
        ['ERG','NIPBL'],
        ['ERG','NR4A3'],
        ['ERG','PARP1'],
        ['ERG','PTEN'],
        ['ERG','PTPRM'],
        ['ERG','RBFOX2'],
        ['ERG','RERE'],
        ['ERG','RTN1'],
        ['ERG','RUNX1'],
        ['ERG','SLC45A3'],
        ['ERG','SMG6'],
        ['ERG','SPINK1'],
        ['ERG','SPOP'],
        ['ERG','TLS'],
        ['ERG','TMPRSS2'],
        ['ERG','TRABD2B'],
        ['ERG','ZFAND3'],
        ['ERG','ZNF278'],
        ['ERK1','ERK2'],
        ['ERK2','MET'],
        ['ERLIN2','FGFR1'],
        ['ERLIN2','FGFR2'],
        ['ERN1','ST14'],
        ['ERN2','SMG1'],
        ['ERO1A','FERMT2'],
        ['ERO1L','FERMT2'],
        ['ERP44','MRPL50'],
        ['ERP44','SPTLC1'],
        ['ERRFI1','SSR2'],
        ['ERRFI1','VAMP3'],
        ['ERVK-28','FGFR1'],
        ['ERVK3-1','ETV1'],
        ['ERVK3-1','ETV4'],
        ['ERVK3-1','FGFR1'],
        ['ERVK3-1','LINC00208'],
        ['ERVW-1','FGFR1'],
        ['ERVW-1','IGH@'],
        ['ERVWE1','IGH@'],
        ['ESAM','MSANTD2'],
        ['ESCO2','PTK2B'],
        ['ESD','TDRD3'],
        ['ESPN','ZBTB48'],
        ['ESR1','GNAS'],
        ['ESR1','GYG1'],
        ['ESR1','LATS1'],
        ['ESR1','LPP'],
        ['ESR1','MGP'],
        ['ESR1','MKL1'],
        ['ESR1','MTHFD1L'],
        ['ESR1','MYB'],
        ['ESR1','MYCT1'],
        ['ESR1','NCOA2'],
        ['ESR1','NCOA3'],
        ['ESR1','NKAIN2'],
        ['ESR1','NOP2'],
        ['ESR1','PCDH11X'],
        ['ESR1','PCMT1'],
        ['ESR1','PDE7B'],
        ['ESR1','PKHD1'],
        ['ESR1','PLEKHG1'],
        ['ESR1','POLH'],
        ['ESR1','POLR2E'],
        ['ESR1','SIM1'],
        ['ESR1','SOX9'],
        ['ESR1','SYNE1'],
        ['ESR1','TACR1'],
        ['ESR1','TFB1M'],
        ['ESR1','TFG'],
        ['ESR1','TNRC6B'],
        ['ESR1','TULP4'],
        ['ESR1','USP25'],
        ['ESR1','UTRN'],
        ['ESR1','YAP1'],
        ['ESRP1','RAF1'],
        ['ESRP2','SCARB2'],
        ['ESRRB','PAX5'],
        ['ESRRG','SMYD3'],
        ['EST14','ETV1'],
        ['ESYT1','FN3K'],
        ['ETF1','KDM3B'],
        ['ETHE1','XRCC1'],
        ['ETNK1','IFT81'],
        ['ETO','RUNX1'],
        ['ETO2','GLIS2'],
        ['ETS1','IGH@'],
        ['ETS1','TMPRSS2'],
        ['ETS2','LYL1'],
        ['ETS2','TMPRSS2'],
        ['ETV1','EWS'],
        ['ETV1','EWSR1'],
        ['ETV1','FOXA1'],
        ['ETV1','FOXP1'],
        ['ETV1','FUBP1'],
        ['ETV1','HERVK17'],
        ['ETV1','HMGN2P46'],
        ['ETV1','HNRNPA2B1'],
        ['ETV1','HNRPA2B1'],
        ['ETV1','KLK2'],
        ['ETV1','MIPOL1'],
        ['ETV1','OR51E2'],
        ['ETV1','PCBP2'],
        ['ETV1','PTPRZ1'],
        ['ETV1','SLC45A3'],
        ['ETV1','SMG6'],
        ['ETV1','SNURF'],
        ['ETV1','TMPRSS2'],
        ['ETV1','TP53'],
        ['ETV1','TTC6'],
        ['ETV2','TMPRSS2'],
        ['ETV3','TMPRSS2'],
        ['ETV3L','TMPRSS2'],
        ['ETV4','EWS'],
        ['ETV4','EWSR1'],
        ['ETV4','FASN'],
        ['ETV4','GPR179'],
        ['ETV4','HERVK17'],
        ['ETV4','HMGN2P46'],
        ['ETV4','KLK2'],
        ['ETV4','NCOA2'],
        ['ETV4','SLC45A3'],
        ['ETV4','STAT3'],
        ['ETV4','TMPRSS2'],
        ['ETV4','UBTF'],
        ['ETV5','SCL45A3'],
        ['ETV5','SLC45A3'],
        ['ETV5','TMPRSS2'],
        ['ETV5','TUT1'],
        ['ETV6','EVI1'],
        ['ETV6','FAAP100'],
        ['ETV6','FACL6'],
        ['ETV6','FGFR3'],
        ['ETV6','FLT3'],
        ['ETV6','FOXO1'],
        ['ETV6','FOXO3'],
        ['ETV6','FOXP1'],
        ['ETV6','FRK'],
        ['ETV6','GOT1'],
        ['ETV6','GSX2'],
        ['ETV6','H4'],
        ['ETV6','HCMOGT-1'],
        ['ETV6','HIC1'],
        ['ETV6','HIP1'],
        ['ETV6','HLXB9'],
        ['ETV6','IGH@'],
        ['ETV6','IGK@'],
        ['ETV6','IKZF1'],
        ['ETV6','INO80D'],
        ['ETV6','ITK'],
        ['ETV6','ITPR2'],
        ['ETV6','JAK2'],
        ['ETV6','KIAA1257'],
        ['ETV6','LINC02260'],
        ['ETV6','LOH12CR1'],
        ['ETV6','LPXN'],
        ['ETV6','LRP6'],
        ['ETV6','LYN'],
        ['ETV6','MALAT1'],
        ['ETV6','MAML3'],
        ['ETV6','MDS1'],
        ['ETV6','MDS2'],
        ['ETV6','MECOM'],
        ['ETV6','MET'],
        ['ETV6','MN1'],
        ['ETV6','MNX1'],
        ['ETV6','MSH6'],
        ['ETV6','MST2'],
        ['ETV6','NA'],
        ['ETV6','NCOA2'],
        ['ETV6','NELL2'],
        ['ETV6','NID1'],
        ['ETV6','NIN'],
        ['ETV6','NIPBL'],
        ['ETV6','NKAIN2'],
        ['ETV6','NTRK1'],
        ['ETV6','NTRK2'],
        ['ETV6','NTRK3'],
        ['ETV6','NUFIP1'],
        ['ETV6','PAX5'],
        ['ETV6','PCM1'],
        ['ETV6','PDE4DIP'],
        ['ETV6','PDGFRA'],
        ['ETV6','PDGFRB'],
        ['ETV6','PDZD7'],
        ['ETV6','PER1'],
        ['ETV6','PEX5'],
        ['ETV6','PITX3'],
        ['ETV6','PMEL'],
        ['ETV6','PML'],
        ['ETV6','PRDM16'],
        ['ETV6','PTPRO'],
        ['ETV6','PTPRR'],
        ['ETV6','QSOX1'],
        ['ETV6','RAB5EP'],
        ['ETV6','RET'],
        ['ETV6','RNU6-19P'],
        ['ETV6','RPN1'],
        ['ETV6','RUNX1'],
        ['ETV6','SENP7'],
        ['ETV6','SLC15A5'],
        ['ETV6','SMCO2'],
        ['ETV6','SNRK'],
        ['ETV6','SNUPN'],
        ['ETV6','STL'],
        ['ETV6','SYK'],
        ['ETV6','SYN1'],
        ['ETV6','TMPRSS2'],
        ['ETV6','TNFRSF8'],
        ['ETV6','TRIP11'],
        ['ETV6','TRKC'],
        ['ETV6','TTL'],
        ['ETV6','VMP1'],
        ['ETV6','YTHDF2'],
        ['ETV6','ZNF385A'],
        ['ETV6','ZNF521'],
        ['ETV7','GTF2I'],
        ['ETV7','TMPRSS2'],
        ['EU154352','PLXNA2'],
        ['EVC2','STK32B'],
        ['EVI1','GR6'],
        ['EVI1','MDS1'],
        ['EVI1','MSI2'],
        ['EVI1','NRIP1'],
        ['EVI1','PRDM16'],
        ['EVI1','RPN1'],
        ['EVI1','RUNX1'],
        ['EVI2A','KDM2A'],
        ['EVL','LCK'],
        ['EVL','NKX2-1'],
        ['EVL','SFTA3'],
        ['EVX1','RUNX1'],
        ['EWS','FEV'],
        ['EWS','FIL1'],
        ['EWS','FLI1'],
        ['EWS','NR4A3'],
        ['EWS','POU5F1'],
        ['EWS','SP3'],
        ['EWS','TEC'],
        ['EWS','UQCRH'],
        ['EWS','WT1'],
        ['EWS','ZNF278'],
        ['EWS','ZSG'],
        ['EWSR1','FEV'],
        ['EWSR1','FLI1'],
        ['EWSR1','FUS'],
        ['EWSR1','GORASP2'],
        ['EWSR1','HLF'],
        ['EWSR1','KLF15'],
        ['EWSR1','KLF17'],
        ['EWSR1','MYB'],
        ['EWSR1','NACC1'],
        ['EWSR1','NF2'],
        ['EWSR1','NFATC1'],
        ['EWSR1','NFATC2'],
        ['EWSR1','NR4A3'],
        ['EWSR1','PATZ1'],
        ['EWSR1','PBX1'],
        ['EWSR1','PBX3'],
        ['EWSR1','PKM'],
        ['EWSR1','PLAGL1'],
        ['EWSR1','POU5F1'],
        ['EWSR1','RBFOX2'],
        ['EWSR1','SMAD3'],
        ['EWSR1','SMARCA5'],
        ['EWSR1','SMCO4'],
        ['EWSR1','SP3'],
        ['EWSR1','SSX1'],
        ['EWSR1','TAF15'],
        ['EWSR1','TCF7L2'],
        ['EWSR1','TFCP2'],
        ['EWSR1','TFE3'],
        ['EWSR1','TFEB'],
        ['EWSR1','VEZF1'],
        ['EWSR1','VGLL1'],
        ['EWSR1','WT1'],
        ['EWSR1','YY1'],
        ['EWSR1','ZNF278'],
        ['EWSR1','ZNF384'],
        ['EWSR1','ZNF444'],
        ['EXD1','INO80'],
        ['EXOC1','GABRR3'],
        ['EXOC1','PDGFRA'],
        ['EXOC2','MET'],
        ['EXOC2','TP53'],
        ['EXOC3L2','LIPE'],
        ['EXOC3L2','PLAUR'],
        ['EXOC4','PVT1'],
        ['EXOC4','WDR64'],
        ['EXOC6B','RALGPS1'],
        ['EXOC6B','TNS3'],
        ['EXOD1','IQCK'],
        ['EXOSC10','MTOR'],
        ['EXT1','PLA2G2A'],
        ['EXT1','PVT1'],
        ['EXT1','SAMD12'],
        ['EYA1','PVT1'],
        ['EYA2','NCOA3'],
        ['EYA2','NFATC2'],
        ['EYA3','MST1'],
        ['EYS','LMBRD1'],
        ['EYS','PHF3'],
        ['EYS','ROS1'],
        ['EYS','SCARB2'],
        ['EYS','SH3BGRL2'],
        ['EZH2','PUS7'],
        ['EZH2','ZNF425'],
        ['EZR','GOPC'],
        ['EZR','HLA-A'],
        ['EZR','LRIG3'],
        ['EZR','MYO5A'],
        ['EZR','PPFIBP1'],
        ['EZR','ROS1'],
        ['F11R','NHS'],
        ['F11R','NRG2'],
        ['F13A1','PRPF4B'],
        ['F13A1','TRIB3'],
        ['FAAH','HSPB11'],
        ['FABP3','ZCCHC17'],
        ['FADD','MORT1'],
        ['FADS1','TMEM258'],
        ['FAF1','OSBPL9'],
        ['FAF1','SIKE1'],
        ['FAF1','SLC2A1'],
        ['FAF1','TAS1R2'],
        ['FAF1','UBE4B'],
        ['FAHD1','NUDT16L1'],
        ['FAM102A','PTGES2'],
        ['FAM104A','MYC'],
        ['FAM107A','TEX264'],
        ['FAM107B','FRMD4A'],
        ['FAM109A','SSH1'],
        ['FAM110A','TLK2'],
        ['FAM110B','FGFR1'],
        ['FAM111A','MYB'],
        ['FAM117A','LGALS9'],
        ['FAM117A','RUNX1'],
        ['FAM118B','YAP1'],
        ['FAM119B','MAPK1'],
        ['FAM119B','METTL1'],
        ['FAM119B','SRGAP1'],
        ['FAM120A','MAP4'],
        ['FAM120AOS','PTCH1'],
        ['FAM120C','PAGE1'],
        ['FAM125B','SPTLC1'],
        ['FAM132A','RLF'],
        ['FAM133A','ZNF845'],
        ['FAM135A','PKIB'],
        ['FAM13A','MAPK10'],
        ['FAM13A-AS1','HERC3'],
        ['FAM13C','FGFR2'],
        ['FAM13C','SUFU'],
        ['FAM155A','MAPK1'],
        ['FAM155A','WNK1'],
        ['FAM157B','FBXO25'],
        ['FAM159A','ZCCHC11'],
        ['FAM160B1','NHLRC2'],
        ['FAM161B','PCM1'],
        ['FAM162A','KMO'],
        ['FAM163B','TMEM185AP1'],
        ['FAM167A','PSIP1'],
        ['FAM167B','FAM222B'],
        ['FAM168A','MARK2'],
        ['FAM168A','SLC43A1'],
        ['FAM172A','PDE4D'],
        ['FAM172A','PPP2R2B'],
        ['FAM174A','WWC1'],
        ['FAM175A','PI4KA'],
        ['FAM179A','PSPC1'],
        ['FAM180B','TRIM60'],
        ['FAM186B','NUDT3'],
        ['FAM188B','STEAP4'],
        ['FAM188B','VWC2'],
        ['FAM190A','FRAS1'],
        ['FAM190A','MAPK10'],
        ['FAM193A','FITM2'],
        ['FAM193A','RNF4'],
        ['FAM193A','TNIP2'],
        ['FAM198B','TMEM144'],
        ['FAM19A2','SRGAP1'],
        ['FAM19A2','STRBP'],
        ['FAM19A2','ZNF462'],
        ['FAM201A','PRKCDBP'],
        ['FAM204A','PARS2'],
        ['FAM207A','TMPRSS2'],
        ['FAM20B','RALGPS2'],
        ['FAM20B','SAMD7'],
        ['FAM20B','UNC5A'],
        ['FAM20C','MKRN1'],
        ['FAM20C','SUGCT'],
        ['FAM211B','FIGN'],
        ['FAM217B','PHF20'],
        ['FAM21C','POTEC'],
        ['FAM222A','MVK'],
        ['FAM222A','SIRT4'],
        ['FAM222B','NEK8'],
        ['FAM228B','TP53I3'],
        ['FAM22A','YWHAE'],
        ['FAM22B','YWHAE'],
        ['FAM23A','MRC1'],
        ['FAM24B','FGFR2'],
        ['FAM26D','RWDD1'],
        ['FAM32A','MED26'],
        ['FAM33A','MYO1D'],
        ['FAM3B','MX1'],
        ['FAM47E','SCARB2'],
        ['FAM47E','SHROOM3'],
        ['FAM48A','SLC45A3'],
        ['FAM49A','NBAS'],
        ['FAM49B','MTSS1'],
        ['FAM49B','PSKH2'],
        ['FAM53B','METTL1'],
        ['FAM53B','SLC7A10'],
        ['FAM5C','PLA2G4A'],
        ['FAM62B','PTPRN2'],
        ['FAM65B','VPS54'],
        ['FAM66C','TRIM42'],
        ['FAM69B','MOB2'],
        ['FAM70B','GAS6'],
        ['FAM71C','SCYL2'],
        ['FAM71D','MPP5'],
        ['FAM71F1','SEPT14'],
        ['FAM76A','FGFR2'],
        ['FAM83B','ICK'],
        ['FAM83D','ZNF807'],
        ['FAM83F','GDAP1L1'],
        ['FAM84B','FGFR2'],
        ['FAM84B','PTK2'],
        ['FAM86C2P','SSBP3'],
        ['FAM89B','SSSCA1'],
        ['FAM90A21P','IFNG'],
        ['FAM91A1','NRG1'],
        ['FAM91A1','ZNF343'],
        ['FAM92B','PTTG1IP'],
        ['FANCA','FP236383.1'],
        ['FANCA','VPS9D1'],
        ['FANCD2','PRKCD'],
        ['FANCD2OS','PRKCD'],
        ['FANCE','MRPS18B'],
        ['FANCL','NRXN1'],
        ['FANCL','RFC2'],
        ['FARP1','PVT1'],
        ['FARP1','STK24'],
        ['FARP2','PPP1R7'],
        ['FARP2','TP53'],
        ['FARS2','RUNX1'],
        ['FARSA','ZNF564'],
        ['FAS','PTEN'],
        ['FASN','RGPD2'],
        ['FASN','SGMS1'],
        ['FAT1','FGFR2'],
        ['FAT1','IRF2'],
        ['FAT1','MALAT1'],
        ['FAT1','NPY1R'],
        ['FAT1','NTRK1'],
        ['FAT1','NTRK2'],
        ['FAT1','NTRK3'],
        ['FAT1','SORBS2'],
        ['FAT1','SPOCK3'],
        ['FAT1','STPG2'],
        ['FAT1','USP6'],
        ['FBF1','GTF2I'],
        ['FBF1','RNF157'],
        ['FBLN1','MKL1'],
        ['FBLN1','PDGFRB'],
        ['FBLN2','HDAC11'],
        ['FBLN5','MARK3'],
        ['FBN3','POLR2A'],
        ['FBP17','MLL'],
        ['FBP2','ITPK1'],
        ['FBRS','ZNF771'],
        ['FBRSL1','PAX5'],
        ['FBRSL1','TMTC3'],
        ['FBXL15','TREML1'],
        ['FBXL17','PJA2'],
        ['FBXL18','RNF216'],
        ['FBXL20','GDPD5'],
        ['FBXL20','MIEN1'],
        ['FBXL20','MYO1D'],
        ['FBXL20','NOTCH2'],
        ['FBXL20','NSF'],
        ['FBXL20','NTRK3'],
        ['FBXL20','RARA'],
        ['FBXL20','SHANK2'],
        ['FBXL20','SIRT6'],
        ['FBXL20','SNF8'],
        ['FBXL20','TLK2'],
        ['FBXL20','TOM1L1'],
        ['FBXL7','NRDC'],
        ['FBXL7','TERT'],
        ['FBXL7','TRIO'],
        ['FBXL7','UHMK1'],
        ['FBXO11','MAP2K5'],
        ['FBXO21','TESC'],
        ['FBXO22','UBE2Q2'],
        ['FBXO28','MLLT4'],
        ['FBXO3','RDH13'],
        ['FBXO32','PLAG1'],
        ['FBXO38','TRIM40'],
        ['FBXO42','RUNX1'],
        ['FBXO45','PLGLA'],
        ['FBXO7','PPIL2'],
        ['FBXW7','MLL3'],
        ['FBXW7','NOTCH1'],
        ['FBXW8','ULK1'],
        ['FCAR','TERF2IP'],
        ['FCF1','LYNX1'],
        ['FCGR2B','IGH@'],
        ['FCGR2C','MAST2'],
        ['FCHO1','MYO9B'],
        ['FCHSD2','GLI3'],
        ['FCHSD2','MAP3K14'],
        ['FCHSD2','MAP3K3'],
        ['FCN1','KIAA1217'],
        ['FCRL3','SON'],
        ['FCRL4','IGH@'],
        ['FCRL4','IGHA1'],
        ['FDCSP','MGP'],
        ['FDFT1','FZD3'],
        ['FDX1','MLLT10'],
        ['FEM1C','SEMA6A'],
        ['FER','ITK'],
        ['FER','MAN2A1'],
        ['FER','SSBP2'],
        ['FER1L6','RAD21'],
        ['FER1L6-AS2','PVT1'],
        ['FES','MAN2A2'],
        ['FEV','FUS'],
        ['FGA','FTL'],
        ['FGB','MAT1A'],
        ['FGB','ORM1'],
        ['FGD3','TYK2'],
        ['FGD4','LRRK2'],
        ['FGD4','SP1'],
        ['FGD5','RAF1'],
        ['FGF1','FGFR1'],
        ['FGF1','FN1'],
        ['FGF12','IL1RAP'],
        ['FGF14-AS1','RB1'],
        ['FGF19','SPTBN2'],
        ['FGF23','FGFR1'],
        ['FGF3','RBM14'],
        ['FGF3','RBM4'],
        ['FGF3','SHANK2'],
        ['FGF3','TACC3'],
        ['FGF3','TMEM45B'],
        ['FGF5','HLA-DRB1'],
        ['FGFR1','FGFR1OP'],
        ['FGFR1','FGFR1OP1'],
        ['FGFR1','FGFR1OP2'],
        ['FGFR1','FN1'],
        ['FGFR1','FOP'],
        ['FGFR1','FOXO1'],
        ['FGFR1','FSIP1'],
        ['FGFR1','GPR124'],
        ['FGFR1','HERVK17'],
        ['FGFR1','HOOK3'],
        ['FGFR1','IGFBP5'],
        ['FGFR1','IGH@'],
        ['FGFR1','KAT6A'],
        ['FGFR1','KAT6B'],
        ['FGFR1','KCNU1'],
        ['FGFR1','LRRFIP1'],
        ['FGFR1','LZTS1-AS1'],
        ['FGFR1','MIR1268A'],
        ['FGFR1','MTERF4'],
        ['FGFR1','MYO18A'],
        ['FGFR1','NRG1'],
        ['FGFR1','NSD3'],
        ['FGFR1','NTM'],
        ['FGFR1','NUP358'],
        ['FGFR1','NUP98'],
        ['FGFR1','PCM1'],
        ['FGFR1','PLAG1'],
        ['FGFR1','RAB11FIP1'],
        ['FGFR1','RAD23A'],
        ['FGFR1','RANBP2'],
        ['FGFR1','RBPMS'],
        ['FGFR1','RHOT1'],
        ['FGFR1','RNF5P1'],
        ['FGFR1','SLC20A2'],
        ['FGFR1','SNRNP70'],
        ['FGFR1','SQSTM1'],
        ['FGFR1','TACC1'],
        ['FGFR1','TFG'],
        ['FGFR1','TG'],
        ['FGFR1','TIAF1'],
        ['FGFR1','TIF1'],
        ['FGFR1','TPR'],
        ['FGFR1','TRIM24'],
        ['FGFR1','TRP'],
        ['FGFR1','USP6'],
        ['FGFR1','WHSC1L1'],
        ['FGFR1','WNK1'],
        ['FGFR1','ZMYM2'],
        ['FGFR1','ZNF198'],
        ['FGFR1','ZNF577'],
        ['FGFR1','ZNF703'],
        ['FGFR1','ZNF791'],
        ['FGFR1OP','RET'],
        ['FGFR1OP1','RET'],
        ['FGFR1OP2','ITPR2'],
        ['FGFR2','FN1'],
        ['FGFR2','FOXP1'],
        ['FGFR2','FRK'],
        ['FGFR2','G3BP2'],
        ['FGFR2','GAB2'],
        ['FGFR2','GLB1L2'],
        ['FGFR2','HOOK1'],
        ['FGFR2','IDH1'],
        ['FGFR2','INA'],
        ['FGFR2','KCNH7'],
        ['FGFR2','KCTD1'],
        ['FGFR2','KIAA1217'],
        ['FGFR2','KIAA1524'],
        ['FGFR2','KIAA1598'],
        ['FGFR2','KIAA1967'],
        ['FGFR2','KIF14'],
        ['FGFR2','KLK2'],
        ['FGFR2','KLRAQ1'],
        ['FGFR2','LAMC1'],
        ['FGFR2','LGSN'],
        ['FGFR2','LINC01153'],
        ['FGFR2','LPXN'],
        ['FGFR2','LZTFL1'],
        ['FGFR2','MAN2B2'],
        ['FGFR2','MARVELD3'],
        ['FGFR2','MAST4'],
        ['FGFR2','MCU'],
        ['FGFR2','MGEA5'],
        ['FGFR2','MIR5694'],
        ['FGFR2','MRPL13'],
        ['FGFR2','MTSS1'],
        ['FGFR2','MYH9'],
        ['FGFR2','MYOF'],
        ['FGFR2','MYPN'],
        ['FGFR2','NCALD'],
        ['FGFR2','NDC80'],
        ['FGFR2','NOL4'],
        ['FGFR2','NPM1'],
        ['FGFR2','NRAP'],
        ['FGFR2','NRBF2'],
        ['FGFR2','NSMCE4A'],
        ['FGFR2','OFD1'],
        ['FGFR2','OGDH'],
        ['FGFR2','OPTN'],
        ['FGFR2','PARK2'],
        ['FGFR2','PCM1'],
        ['FGFR2','PDHX'],
        ['FGFR2','PHC1'],
        ['FGFR2','PHGDH'],
        ['FGFR2','PKM'],
        ['FGFR2','PLEKHS1'],
        ['FGFR2','PLPP4'],
        ['FGFR2','POC1B'],
        ['FGFR2','PPAPDC1A'],
        ['FGFR2','PPHLN1'],
        ['FGFR2','PRKN'],
        ['FGFR2','PVT1'],
        ['FGFR2','RAB7L1'],
        ['FGFR2','RABGAP1L'],
        ['FGFR2','RASAL2'],
        ['FGFR2','REEP3'],
        ['FGFR2','RNF41'],
        ['FGFR2','ROCK1'],
        ['FGFR2','SH3GLB1'],
        ['FGFR2','SHC2'],
        ['FGFR2','SHROOM3'],
        ['FGFR2','SHTN1'],
        ['FGFR2','SLC45A3'],
        ['FGFR2','SLMAP'],
        ['FGFR2','SMN1'],
        ['FGFR2','SNX19'],
        ['FGFR2','SORBS1'],
        ['FGFR2','SOX9'],
        ['FGFR2','STK26'],
        ['FGFR2','STK3'],
        ['FGFR2','TACC1'],
        ['FGFR2','TACC2'],
        ['FGFR2','TACC3'],
        ['FGFR2','TBC1D1'],
        ['FGFR2','TEL'],
        ['FGFR2','TFEC'],
        ['FGFR2','TMBIM6'],
        ['FGFR2','TNS1'],
        ['FGFR2','TP73'],
        ['FGFR2','TRA2B'],
        ['FGFR2','TRIM8'],
        ['FGFR2','TXLNA'],
        ['FGFR2','UBP1'],
        ['FGFR2','UBQLN1'],
        ['FGFR2','ULK4'],
        ['FGFR2','USP10'],
        ['FGFR2','VCL'],
        ['FGFR2','VPS35'],
        ['FGFR2','WAC'],
        ['FGFR2','WARS'],
        ['FGFR2','WDR11'],
        ['FGFR2','ZMYM4'],
        ['FGFR2','ZMYND11'],
        ['FGFR2','ZMYND8'],
        ['FGFR3','IGH@'],
        ['FGFR3','IGHG1'],
        ['FGFR3','IGK@'],
        ['FGFR3','IGL@'],
        ['FGFR3','IRF4'],
        ['FGFR3','IRTA1'],
        ['FGFR3','JAKMIP1'],
        ['FGFR3','LETM1'],
        ['FGFR3','MMSET'],
        ['FGFR3','NBR1'],
        ['FGFR3','PAX5'],
        ['FGFR3','PHGDH'],
        ['FGFR3','PHLDB3'],
        ['FGFR3','RANBP17'],
        ['FGFR3','RBBP4'],
        ['FGFR3','SOX13'],
        ['FGFR3','SPDYE4'],
        ['FGFR3','SYNC'],
        ['FGFR3','TACC3'],
        ['FGFR3','TNIP2'],
        ['FGFR3','TPRG1'],
        ['FGFR3','WHSC1'],
        ['FGFR4','NSD1'],
        ['FGFR4','UIMC1'],
        ['FGFR4','ZNF346'],
        ['FGGY','TESK2'],
        ['FGR','WASF2'],
        ['FHIT','FRA3B'],
        ['FHIT','HMGA2'],
        ['FHIT','MACROD2'],
        ['FHIT','PTPRG'],
        ['FHL2','GLI2'],
        ['FIBCD1','TTLL11'],
        ['FIG','ROS1'],
        ['FIG4','SEC63'],
        ['FIGLA','MAP4K3'],
        ['FIGN','STK25'],
        ['FIGNL1','VOPP1'],
        ['FILIP1','PTPRK'],
        ['FIP1L1','FRYL'],
        ['FIP1L1','LHFPL5'],
        ['FIP1L1','MAST4'],
        ['FIP1L1','PDGFA'],
        ['FIP1L1','PDGFR'],
        ['FIP1L1','PDGFRA'],
        ['FIP1L1','RARA'],
        ['FKBP15','PAX5'],
        ['FKBP15','RET'],
        ['FKBP15','SLC31A1'],
        ['FKBP5','TMPRSS2'],
        ['FKBP8','MLL'],
        ['FKBP8','SLC1A6'],
        ['FKBPL','UCHL5'],
        ['FKHR','PAX3'],
        ['FKHR','PAX7'],
        ['FLI1','FUS'],
        ['FLI1','MN1'],
        ['FLI1','SCL45A3'],
        ['FLI1','TCF3'],
        ['FLI1','ZZZ3'],
        ['FLJ10661','THADA'],
        ['FLNA','KMT2A'],
        ['FLNA','MLL'],
        ['FLNB','LGALS12'],
        ['FLNB','PXK'],
        ['FLNC','KMT2A'],
        ['FLT1','GABRA3'],
        ['FLT1','IRS2'],
        ['FLT1','LHFP'],
        ['FLT1','OLFM4'],
        ['FLT1','SRPK2'],
        ['FLT1','ZNF436'],
        ['FLT3','GOLGB1'],
        ['FLT3','IGH@'],
        ['FLT3','NRAS'],
        ['FLT3','PAN3'],
        ['FLT3','RPL21'],
        ['FLT3','SPTBN1'],
        ['FLT3','TRIP11'],
        ['FLT3','USP12'],
        ['FLT3','ZMYM2'],
        ['FLT3LG','RPS11'],
        ['FLT4','HMGXB3'],
        ['FLT4','LBH'],
        ['FLT4','RRP7A'],
        ['FLT4','RUFY1'],
        ['FLVCR1','LYST'],
        ['FLVCR1','TATDN3'],
        ['FLVCR2','TTLL5'],
        ['FMN1','GREM1'],
        ['FMN2','NVL'],
        ['FMNL3','FRS2'],
        ['FN1','ICAM1'],
        ['FN1','IGF1R'],
        ['FN1','MERTK'],
        ['FN1','MKNK2'],
        ['FN1','NTRK1'],
        ['FN1','NUP98'],
        ['FN1','PABPC1'],
        ['FN1','RET'],
        ['FN1','ROS1'],
        ['FN1','TEK'],
        ['FN1','TGFA'],
        ['FN1','TIMP1'],
        ['FN1','USP46'],
        ['FNBP1','KMT2A'],
        ['FNBP1','MLL'],
        ['FNBP1','RUNX1'],
        ['FNBP1','ZFP36L1'],
        ['FNBP4','MTCH2'],
        ['FNDC3B','MECOM'],
        ['FNDC3B','MST1R'],
        ['FNDC3B','PIK3CA'],
        ['FNDC3B','PRKCI'],
        ['FNDC3B','RARA'],
        ['FNDC3B','TNIK'],
        ['FNDC7','TRIM33'],
        ['FNTB','FUT8'],
        ['FO393400.1','MAPRE1'],
        ['FOG2','ZFPM2'],
        ['FOLR2','RNF121'],
        ['FOS','FOSB'],
        ['FOS','LMNA'],
        ['FOS','MBNL1'],
        ['FOS','RUNX2'],
        ['FOS','VIM'],
        ['FOSB','PPP1R10'],
        ['FOSB','SERPINE1'],
        ['FOSB','SETD1B'],
        ['FOSB','WWTR1'],
        ['FOSB','ZFP36'],
        ['FOXA1','GAS6'],
        ['FOXA1','NRG1'],
        ['FOXJ2','MEF2D'],
        ['FOXJ2','TSPAN9'],
        ['FOXJ3','GUCA2B'],
        ['FOXJ3','RLF'],
        ['FOXK2','HEXDC'],
        ['FOXK2','KRT20'],
        ['FOXK2','PIM3'],
        ['FOXM1','SLC2A14'],
        ['FOXN1','PRKDC'],
        ['FOXN1','TNFAIP1'],
        ['FOXN4','IRF2'],
        ['FOXO1','FOXO3'],
        ['FOXO1','GATA6'],
        ['FOXO1','LINC00598'],
        ['FOXO1','MEIS1'],
        ['FOXO1','OGT'],
        ['FOXO1','PAX3'],
        ['FOXO1','PAX7'],
        ['FOXO1','SRF'],
        ['FOXO1','STK24'],
        ['FOXO1A','NCOA1'],
        ['FOXO1A','PAX3'],
        ['FOXO1A','PAX7'],
        ['FOXO3','FXR1'],
        ['FOXO3','KMT2A'],
        ['FOXO3','MLL'],
        ['FOXO3','MYC'],
        ['FOXO3','OGT'],
        ['FOXO3A','MLL'],
        ['FOXO4','GLI1'],
        ['FOXO4','KMT2A'],
        ['FOXO4','MLL'],
        ['FOXO4','OGT'],
        ['FOXO4','PAX3'],
        ['FOXP1','GOLGA4'],
        ['FOXP1','HOXA10'],
        ['FOXP1','IGH@'],
        ['FOXP1','IGHA2'],
        ['FOXP1','LPP'],
        ['FOXP1','MALAT1'],
        ['FOXP1','MIPEP'],
        ['FOXP1','MITF'],
        ['FOXP1','PAX5'],
        ['FOXP1','PDGFRA'],
        ['FOXP1','PLVAP'],
        ['FOXP1','PPP1R2'],
        ['FOXP1','RAB7A'],
        ['FOXP1','RYBP'],
        ['FOXP1','STK4'],
        ['FOXP1','TMPRSS2'],
        ['FOXP2','MET'],
        ['FOXP2','PHF19'],
        ['FOXP4','TGFBR3'],
        ['FOXR1','KMT2A'],
        ['FOXR1','MLL'],
        ['FOXR1','PAFAH1B2'],
        ['FOXR1','RPS25'],
        ['FOXR1','YTHDF2'],
        ['FOXR2','NB'],
        ['FOXR2','PAFAH1B2'],
        ['FPR2','TMEM255A'],
        ['FRA10AC1','HNRNPH1'],
        ['FRAS1','HADH'],
        ['FRAS1','LRRC8D'],
        ['FRAS1','MRPL1'],
        ['FRAS1','SHROOM3'],
        ['FRAS1','WDFY3'],
        ['FREM1','PTPRD'],
        ['FREM2','MTRF1'],
        ['FRG1B','PPP1R1B'],
        ['FRG2','UTP14C'],
        ['FRK','PRIM2'],
        ['FRMD4A','PFKP'],
        ['FRMD4A','PTEN'],
        ['FRMD4A','RET'],
        ['FRMD4B','MITF'],
        ['FRMD8','SCYL1'],
        ['FRS2','KIF5A'],
        ['FRS2','LYZ'],
        ['FRS2','NUP107'],
        ['FRS2','TP53'],
        ['FRYL','KMT2A'],
        ['FRYL','MLL'],
        ['FRYL','OCIAD2'],
        ['FSD1','SPAG16'],
        ['FSTL1','GSK3B'],
        ['FTH1','FTL'],
        ['FTL','SFTPA1'],
        ['FTO','HERPUD1'],
        ['FTX','XIST'],
        ['FUBP1','SEC31A'],
        ['FUBP1','TFE3'],
        ['FUBP3','PRRC2B'],
        ['FUNDC2','RPS6KA6'],
        ['FUS','GABRA3'],
        ['FUS','KLF17'],
        ['FUS','LGMN'],
        ['FUS','NFATC2'],
        ['FUS','NR4A3'],
        ['FUS','POU5F1'],
        ['FUS','PUM1'],
        ['FUS','TFCP2'],
        ['FUT8','KCNH5'],
        ['FVT1','IGK@'],
        ['FXR1','MYC'],
        ['FXR2','SLC30A1'],
        ['FXR2','TP53'],
        ['FXYD6','KMT2A'],
        ['FYCO1','RAF1'],
        ['FYCO1','RET'],
        ['FYN','MLL'],
        ['FYN','SF1'],
        ['FYN','TRAF3IP2'],
        ['FZD4','MYH3'],
        ['FZD6','SDC2'],
        ['G2E3','SPRED2'],
        ['G3BP2','NAAA'],
        ['G3BP2','PDGFC'],
        ['G6PC2','IGRP'],
        ['G6PC3','MRPS31'],
        ['GAB1','HHIP-AS1'],
        ['GAB2','INTS4'],
        ['GAB2','MRPL48'],
        ['GAB2','MYO7A'],
        ['GAB2','PLXDC1'],
        ['GABBR1','UBD'],
        ['GABBR2','NOTCH1'],
        ['GABRA3','SLC6A8'],
        ['GABRB1','MYO1D'],
        ['GABRB3','GAS5'],
        ['GABRB3','RYR3'],
        ['GADL1','GNG12-AS1'],
        ['GADL1','TGFBR2'],
        ['GAGE2E','KLHL25'],
        ['GAK','IDUA'],
        ['GAK','LUC7L3'],
        ['GAK','MAGI1'],
        ['GAK','PDXK'],
        ['GAK','PGC'],
        ['GAK','TG'],
        ['GAL','LRP5'],
        ['GALK2','RAB11A'],
        ['GALNT18','TYSND1'],
        ['GALNT6','NENF'],
        ['GALNT7','ORC4L'],
        ['GALNT7','PDLIM1'],
        ['GALNT9','LMBR1L'],
        ['GALNT9','POLE'],
        ['GALNTL6','NEK1'],
        ['GANAB','NXF1'],
        ['GAP43','MECOM'],
        ['GAPVD1','ZNF483'],
        ['GART','IFNGR2'],
        ['GAS5','MDN1'],
        ['GAS6','RASA3'],
        ['GAS7','KMT2A'],
        ['GAS7','MLL'],
        ['GAS7','MYH1'],
        ['GAS7','RCVRN'],
        ['GAS8','HDAC4'],
        ['GAS8','WDR59'],
        ['GATA1','MYB'],
        ['GATA2','HOXA9'],
        ['GATA2','MECOM'],
        ['GATA4','LIFR'],
        ['GATA4','YTHDC1'],
        ['GATA6','ROCK1'],
        ['GATAD2A','LYN'],
        ['GATAD2A','NFIX'],
        ['GATAD2A','ZNF91'],
        ['GATAD2B','IGFN1'],
        ['GATAD2B','NTRK1'],
        ['GATAD2B','NUP210L'],
        ['GATAD2B','SLC19A2'],
        ['GATAD2B','TNFSF18'],
        ['GATM','RAF1'],
        ['GATSL2','GTF2I'],
        ['GBAS','PCLO'],
        ['GBAS','UBXN2A'],
        ['GC','TTR'],
        ['GCC2','PDGFRB'],
        ['GCGR','LRRC45'],
        ['GCH1','SERPINA6'],
        ['GCM1','MTSS1'],
        ['GCN1L1','MSI1'],
        ['GCN1L1','PLA2G1B'],
        ['GDAP1L1','WASH1'],
        ['GDE1','LIN54'],
        ['GDF15','NRG1'],
        ['GDF5','MGLL'],
        ['GDPD4','PAK1'],
        ['GDPD5','MAP6'],
        ['GDPD5','RELT'],
        ['GEM','PLAG1'],
        ['GEMIN7','ZNF296'],
        ['GFAP','SPARC'],
        ['GFAP','TNF'],
        ['GFAP','TPT1'],
        ['GFAP','VIM'],
        ['GFOD1','JARID2'],
        ['GFRA3','GIGYF2'],
        ['GFRA3','SIL1'],
        ['GGA1','MICALL1'],
        ['GGA2','PRKCB'],
        ['GGA3','VRK2'],
        ['GGNBP2','MYO19'],
        ['GGNBP2','ZDHHC16'],
        ['GGPS1','TMEM63A'],
        ['GGPS1','TP53'],
        ['GGT1','IGSF3'],
        ['GGT1','NCOR1'],
        ['GGT5','USP18'],
        ['GH','LHX2'],
        ['GHR','HDAC4'],
        ['GHR','RICTOR'],
        ['GID4','TP53'],
        ['GIGYF2','NGEF'],
        ['GIGYF2','TRPM8'],
        ['GIMAP8','KMT2A'],
        ['GINS1','ZFP64'],
        ['GIPC1','WNK1'],
        ['GIT1','SHMT1'],
        ['GIT2','PDGFRB'],
        ['GIT2','SCYL2'],
        ['GJA12','GJC2'],
        ['GJB1','PSEN1'],
        ['GJD2','RPN1'],
        ['GK5','METTL15'],
        ['GKAP1','KIF27'],
        ['GKAP1','NTRK2'],
        ['GLB1L','RDH5'],
        ['GLCCI1','LTBR'],
        ['GLCCI1','TRIM7'],
        ['GLCE','TMEM52B'],
        ['GLDC','KDM4C'],
        ['GLDN','WDFY2'],
        ['GLG1','GRM5'],
        ['GLI1','INHBE'],
        ['GLI1','MALAT1'],
        ['GLI1','MARS'],
        ['GLI1','PTCH1'],
        ['GLI1','SEMA4A'],
        ['GLI1','TUBA1A'],
        ['GLIPR2','NUDT2'],
        ['GLIS1','PAX8'],
        ['GLIS2','PHF21A'],
        ['GLIS3','LATS1'],
        ['GLIS3','NF1'],
        ['GLIS3','PAX8'],
        ['GLO1','LMBRD1'],
        ['GLRA1','SPARC'],
        ['GLRB','GRIA2'],
        ['GLRX5','RUNX1'],
        ['GLTP','IGF1R'],
        ['GLTP','VPS29'],
        ['GLTSCR1','MPND'],
        ['GLYAT','MTA2'],
        ['GLYATL1P3','PURA'],
        ['GLYR1','SLC9A8'],
        ['GMDS','PDE8B'],
        ['GMDS','TMEM189'],
        ['GMEB1','YTHDF2'],
        ['GMFG','KLHL15'],
        ['GMPS','KMT2A'],
        ['GMPS','MLL'],
        ['GMPS','SYNPR'],
        ['GNA12','SHANK2'],
        ['GNA13','MYO1D'],
        ['GNAO1','TOX3'],
        ['GNAQ','NTRK2'],
        ['GNAQ','PLEKHA8'],
        ['GNAS','LAPTM5'],
        ['GNB3','METAP2'],
        ['GNG12','IL23R'],
        ['GNG2','MAP3K8'],
        ['GNG3','TAF6L'],
        ['GNGT1','KRIT1'],
        ['GNGT1','SBK1'],
        ['GNL2','SF3A3'],
        ['GNPDA2','OTUD3'],
        ['GNPTAB','MAPKAPK5'],
        ['GNPTG','TNFRSF9'],
        ['GNS','NUP107'],
        ['GNS','PAWR'],
        ['GNS','STK10'],
        ['GOLGA3','MYH6'],
        ['GOLGA4','LRRFIP2'],
        ['GOLGA4','PDGFRB'],
        ['GOLGA4','RAF1'],
        ['GOLGA4','RARA'],
        ['GOLGA5','H4'],
        ['GOLGA5','HOOK3'],
        ['GOLGA5','JAK2'],
        ['GOLGA5','KIF5B'],
        ['GOLGA5','KTN1'],
        ['GOLGA5','RET'],
        ['GOLGA5','TRIM27'],
        ['GOLGA5','TRIM33'],
        ['GOLGA7','PVT1'],
        ['GOLGB1','ILDR1'],
        ['GOLGB1','PDGFRB'],
        ['GOLGB1','ROS1'],
        ['GOLIM4','PPM1L'],
        ['GOLM1','MAK10'],
        ['GOLT1A','KISS1'],
        ['GOLT1A','PLEKHA6'],
        ['GON4L','INSRR'],
        ['GON4L','NTRK1'],
        ['GON4L','RIT1'],
        ['GOPC','KDELR2'],
        ['GOPC','ROS1'],
        ['GOPC','SDC4'],
        ['GOPC','SLC34A2'],
        ['GOPC','TRMT11'],
        ['GOSR1','NF1'],
        ['GOSR1','SSH2'],
        ['GOSR1','TAOK1'],
        ['GOSR1','ZNF207'],
        ['GOT1L1','IKBKB'],
        ['GOT1L1','SLC25A33'],
        ['GP1BA','IQSEC1'],
        ['GPATCH2','N4BP1'],
        ['GPATCH2','PTBP2'],
        ['GPATCH8','HDAC5'],
        ['GPATCH8','KIAA1958'],
        ['GPATCH8','MPP2'],
        ['GPATCH8','PCDHA1'],
        ['GPBP1','YEATS2'],
        ['GPBP1L1','IPP'],
        ['GPBP1L1','MAST2'],
        ['GPC2','MCM7'],
        ['GPC3','NUP85'],
        ['GPC3','XCL1'],
        ['GPC4','HS6ST2'],
        ['GPC5','GPC6'],
        ['GPC5','MYO10'],
        ['GPC6','NBEA'],
        ['GPC6','WIF1'],
        ['GPHN','KMT2A'],
        ['GPHN','MLL'],
        ['GPHN','MPP5'],
        ['GPHN','RET'],
        ['GPIAP1','PDGFRB'],
        ['GPLD1','TDP2'],
        ['GPM6B','RAB9A'],
        ['GPN1','SLC4A1AP'],
        ['GPN2','THRAP3'],
        ['GPNMB','SP4'],
        ['GPR107','MUSK'],
        ['GPR107','MYH10'],
        ['GPR126','PRKCE'],
        ['GPR128','TFG'],
        ['GPR133','TXNRD1'],
        ['GPR137','ZDHHC5'],
        ['GPR149','MAFK'],
        ['GPR156','NFKBIE'],
        ['GPR158','WAC'],
        ['GPR160','NCEH1'],
        ['GPR160','PRKCI'],
        ['GPR3','TP53'],
        ['GPR34','IGH@'],
        ['GPR37L1','IPO9-AS1'],
        ['GPR37L1','NAV1'],
        ['GPR37L1','TESK2'],
        ['GPR4','ZNF787'],
        ['GPR56','TAOK2'],
        ['GPRC5C','RECQL5'],
        ['GPS2','MLL4'],
        ['GPS2','MPP2'],
        ['GPS2','SENP3'],
        ['GPSM3','UBE2QL1'],
        ['GPSN2','STRN4'],
        ['GPT','PPP1R16A'],
        ['GPX3','SEC63'],
        ['GPX3','SQSTM1'],
        ['GPX4','SEC63'],
        ['GRAF','MLL'],
        ['GRAMD2','NUDCD1'],
        ['GRAP2','NF2'],
        ['GRB10','PDE3A'],
        ['GRB10','TRRAP'],
        ['GRB10','VOPP1'],
        ['GRB2','HHATL'],
        ['GRB7','HER2'],
        ['GRB7','PERLD1'],
        ['GRB7','PPP1R1B'],
        ['GRB7','STAT3'],
        ['GREB1','MBOAT2'],
        ['GREB1','NCOA1'],
        ['GREB1','NCOA2'],
        ['GREB1','NR4A3'],
        ['GREB1','PDE1A'],
        ['GREB1','SS18'],
        ['GREM1','SCG5'],
        ['GRHL2','HNRNPD'],
        ['GRHL2','MAP2K2'],
        ['GRHL2','NRBP2'],
        ['GRHL2','RSPO2'],
        ['GRHL2','TMPRSS2'],
        ['GRIA2','ZW10'],
        ['GRIA4','MLLT10'],
        ['GRID2','PDK4'],
        ['GRID2','SCARB2'],
        ['GRID2','UBXN7'],
        ['GRIK1','SLA'],
        ['GRIK2','RUNX1'],
        ['GRIK2','TRMT11'],
        ['GRIK4','GRM5'],
        ['GRIN2B','KRAS'],
        ['GRIN2B','SSPN'],
        ['GRIN2B','TCF12'],
        ['GRIN2B','TMEM87A'],
        ['GRIN3A','TJP2'],
        ['GRIP1','MED13L'],
        ['GRIP1','NUP107'],
        ['GRIP1','PPM1H'],
        ['GRIP1','RNFT2'],
        ['GRIP1','SOX5'],
        ['GRIP1','TBK1'],
        ['GRIP1','TNIP1'],
        ['GRIPAP1','NTRK1'],
        ['GRIPAP1','NTRK2'],
        ['GRIPAP1','NTRK3'],
        ['GRIPAP1','RET'],
        ['GRIPAP1','TFE3'],
        ['GRK1','PCID2'],
        ['GRK1','TFDP1'],
        ['GRK4','HTT'],
        ['GRK4','LRPAP1'],
        ['GRK4','WHSC1'],
        ['GRK5','INPP5F'],
        ['GRK5','STOX1'],
        ['GRK6','UIMC1'],
        ['GRM1','MEF2A'],
        ['GRM1','TBL1XR1'],
        ['GRM1','TP53'],
        ['GRM3','PAX5'],
        ['GRM5','NOX4'],
        ['GRM8','HIPK2'],
        ['GRXCR2','RBM27'],
        ['GSDMA','PPFIA1'],
        ['GSDMA','THRA'],
        ['GSDMB','MED1'],
        ['GSDMB','ORMDL3'],
        ['GSDMB','PVT1'],
        ['GSDMB','TATDN1'],
        ['GSDMC','PVT1'],
        ['GSK3A','ZNF541'],
        ['GSK3B','TIAM1'],
        ['GSN','MEGF9'],
        ['GSN','NTRK1'],
        ['GSR','GTF2E2'],
        ['GSR','PRKDC'],
        ['GSS','SRC'],
        ['GSS','VAV1'],
        ['GSTT1','NF2'],
        ['GSX2','SCFD2'],
        ['GTF2H3','SETD1B'],
        ['GTF2H5','SORBS2'],
        ['GTF2I','GTF2IRD1'],
        ['GTF2I','NCOA2'],
        ['GTF2I','PDGFRB'],
        ['GTF2I','RARA'],
        ['GTF2IRD1','KMT2C'],
        ['GTPBP2','POLH'],
        ['GTSE1','MN1'],
        ['GUCA2B','PID1'],
        ['GUCY2C','LMO3'],
        ['GUCY2C','TSPAN9'],
        ['GUCY2D','RPA1'],
        ['GUF1','MTFR1'],
        ['GUSBP12','PSMC3'],
        ['GXYLT1','LRRK2'],
        ['GXYLT2','NEK3'],
        ['GYG1P1','TAF1B'],
        ['GYLTL1B','HIPK3'],
        ['GYLTL1B','PAK1'],
        ['GYPA','GYPB'],
        ['GYPB','SAV1'],
        ['GYPC','MRPS34'],
        ['GZMK','IL10RB'],
        ['GZMM','RERG'],
        ['H19','HSP90B1'],
        ['H19','IGF2'],
        ['H2AFX','LAMB4'],
        ['H2AFX','WDR18'],
        ['H2AFY','HMHB1'],
        ['H2AFY','MECOM'],
        ['H2AFY','TSLP'],
        ['H2AFZ','TMEM242'],
        ['H3F3B','TP53'],
        ['H3F3B','ZDHHC7'],
        ['H4','NCOA4'],
        ['H4','PCM1'],
        ['H4','PDGFRB'],
        ['H4','PRKAR1A'],
        ['H4','RET'],
        ['H6PD','SPSB1'],
        ['HACL1','RAF1'],
        ['HADHB','RBKS'],
        ['HAO2','VANGL1'],
        ['HAPLN1','PIK3R1'],
        ['HAPLN1','VCAN'],
        ['HAPLN2','RBX1'],
        ['HAPLN3','MFGE8'],
        ['HARS2','ZMAT2'],
        ['HAS2','PLAG1'],
        ['HAUS1','NPM1'],
        ['HAUS6','SLC1A1'],
        ['HBB','HBD'],
        ['HBEGF','RBM27'],
        ['HBG1','HBG2'],
        ['HBP1','LPXN'],
        ['HBS1L','MYB'],
        ['HCFC1R1','THOC6'],
        ['HCG24','TRIM49B'],
        ['HCK','PLAGL2'],
        ['HCK','POFUT1'],
        ['HCMOGT-1','PDGFRB'],
        ['HCMOGT1','PDGFRB'],
        ['HDAC1','SERINC2'],
        ['HDAC4','PDE11A'],
        ['HDAC7','IRAK4'],
        ['HDAC7','VDR'],
        ['HDDC2','STBD1'],
        ['HDDC3','IGLVI-70'],
        ['HDGF','S100A10'],
        ['HDHD1','MACROD1'],
        ['HDHD2','IER3IP1'],
        ['HDHD2','LOXHD1'],
        ['HDLBP','HJURP'],
        ['HDLBP','SCCPDH'],
        ['HDX','RPS6KA6'],
        ['HE1','NPC2'],
        ['HEAB','MLL'],
        ['HEATR1','TMEM27'],
        ['HEATR3','SMEK1'],
        ['HEATR4','NUMB'],
        ['HECTD4','PSD3'],
        ['HECTD4','YAP1'],
        ['HEG1','YAP1'],
        ['HEI10','HMGIC'],
        ['HELZ','MYH10'],
        ['HELZ','PITPNC1'],
        ['HENMT1','RCC1'],
        ['HEPACAM2','ZSCAN21'],
        ['HEPHL1','PANX1'],
        ['HER2','HER3'],
        ['HER2','MDK'],
        ['HER2','WIPF2'],
        ['HER2','ZNF207'],
        ['HERC1','PML'],
        ['HERC4','TMPRSS15'],
        ['HERPUD1','RAF1'],
        ['HES4','TRIM52'],
        ['HEY1','NCOA2'],
        ['HEYL','MACO1'],
        ['HFM1','PPEF2'],
        ['HFM1','SLCO5A1'],
        ['HGF','MET'],
        ['HGSNAT','KIAA0146'],
        ['HHEX','NUP98'],
        ['HHLA1','MECOM'],
        ['HHLA1','OC90'],
        ['HIAT1','SLC35A3'],
        ['HIATL1','RPS26P55'],
        ['HIC2','TEC'],
        ['HIF1A','VRK1'],
        ['HIP1','PDGFRB'],
        ['HIPK1','LRRC42'],
        ['HIPK1','NT5DC1'],
        ['HIPK1','PAX5'],
        ['HIPK2','STAT6'],
        ['HIPK3','NRD1'],
        ['HIST1H2BD','PVT1'],
        ['HIST4H4','MGP'],
        ['HIVEP3','JAK1'],
        ['HIVEP3','SUPT6H'],
        ['HJURP','INPP4A'],
        ['HK1','MYPN'],
        ['HK1','PITRM1'],
        ['HK1','TSPAN15'],
        ['HK2','TEX261'],
        ['HLA-A','ROS1'],
        ['HLA-DRB1','MET'],
        ['HLA-DRB1','PDK2'],
        ['HLF','PBX1'],
        ['HLF','TCF3'],
        ['HLF','TFPT'],
        ['HLF','TOM1L1'],
        ['HLTF','HPS3'],
        ['HM13','RLF'],
        ['HM13','TM9SF4'],
        ['HM13','TPX2'],
        ['HM13','TSPAN31'],
        ['HMBOX1','JAK2'],
        ['HMBOX1','NRG1'],
        ['HMBOX1','NTRK2'],
        ['HMBOX1','ZFAND3'],
        ['HMCES','MECOM'],
        ['HMGA1','LAMA4'],
        ['HMGA1','RAD51L1'],
        ['HMGA1','TBL1XR1'],
        ['HMGA2','KSR2'],
        ['HMGA2','LEMD3'],
        ['HMGA2','LHFP'],
        ['HMGA2','LPP'],
        ['HMGA2','MLL'],
        ['HMGA2','MSRB3'],
        ['HMGA2','MYB'],
        ['HMGA2','NAA11'],
        ['HMGA2','NAV3'],
        ['HMGA2','NCOR2'],
        ['HMGA2','NFIB'],
        ['HMGA2','NUP107'],
        ['HMGA2','PCBP2'],
        ['HMGA2','PLAG1'],
        ['HMGA2','PPAP2B'],
        ['HMGA2','PTPRR'],
        ['HMGA2','RAD51B'],
        ['HMGA2','RAD51L1'],
        ['HMGA2','RDC1'],
        ['HMGA2','SENP1'],
        ['HMGA2','SETBP1'],
        ['HMGA2','SOX5'],
        ['HMGA2','TMTC2'],
        ['HMGA2','TRAF3IP2'],
        ['HMGA2','TSFM'],
        ['HMGA2','WIF1'],
        ['HMGA2','YAP1'],
        ['HMGB1','LATS2'],
        ['HMGB1','PHF6'],
        ['HMGB1','ZDHHC21'],
        ['HMGB3','NUP98'],
        ['HMGB3P22','MATR3'],
        ['HMGCLL1','MAP2K3'],
        ['HMGCLL1','TEC'],
        ['HMGIC','LHFP'],
        ['HMGIC','LPP'],
        ['HMGIC','RAD51L1'],
        ['HMGN2P46','LGSN'],
        ['HMGN2P46','MYC'],
        ['HMGN2P46','SKIL'],
        ['HMGXB3','PPARGC1B'],
        ['HMHA1','POLR3D'],
        ['HMHA1','STK11'],
        ['HMMR','MPDU1'],
        ['HN1','USH1G'],
        ['HN1L','STK10'],
        ['HNF1A','SUPT3H'],
        ['HNF1B','LUC7L3'],
        ['HNF1B','NOTCH1'],
        ['HNF1B','PGAP3'],
        ['HNF4A','TTPAL'],
        ['HNF4G','RSPO2'],
        ['HNF4G','ZMAT4'],
        ['HNRNPA1L2','SUGT1'],
        ['HNRNPA2B1','MYC'],
        ['HNRNPA2B1','NTRK3'],
        ['HNRNPA2B1','PCBD2'],
        ['HNRNPC','PLAG1'],
        ['HNRNPC','RARG'],
        ['HNRNPD','LAMB2'],
        ['HNRNPD','NFE2L2'],
        ['HNRNPD','NHP2L1'],
        ['HNRNPD','NOL6'],
        ['HNRNPD','PDCD10'],
        ['HNRNPD','PPDPF'],
        ['HNRNPD','RASGRF1'],
        ['HNRNPD','RILPL2'],
        ['HNRNPD','TMEM67'],
        ['HNRNPD','VAMP2'],
        ['HNRNPF','RASGEF1A'],
        ['HNRNPH1','MEF2D'],
        ['HNRNPH1','MLLT1'],
        ['HNRNPH1','MLLT10'],
        ['HNRNPH3','SF1'],
        ['HNRNPH3','TFE3'],
        ['HNRNPM','RFX2'],
        ['HNRNPU','PIM3'],
        ['HNRNPUL1','MEF2D'],
        ['HNRNPUL1','TGFB1'],
        ['HNRPDL','PBX1'],
        ['HOMER1','SREK1'],
        ['HOMER2','NTRK1'],
        ['HOMER2','NTRK2'],
        ['HOMER2','NTRK3'],
        ['HOMEZ','MYH6'],
        ['HOMEZ','PRMT5'],
        ['HOOK1','RAB6A'],
        ['HOOK3','IKBKB'],
        ['HOOK3','JAG1'],
        ['HOOK3','NTRK2'],
        ['HOOK3','RET'],
        ['HOPX','PDCL2'],
        ['HORMAD1','SLC22A20'],
        ['HOX11','LCK'],
        ['HOX11','LMO1'],
        ['HOX11','LMO2'],
        ['HOX11','LYL1'],
        ['HOX11','NOTCH1'],
        ['HOX11','RANBP17'],
        ['HOX11','TAL1'],
        ['HOX11','TAL2'],
        ['HOX11','TCL3'],
        ['HOX11','TLX1'],
        ['HOX11','TRB@'],
        ['HOX11L2','TLX3'],
        ['HOX@','NUP98'],
        ['HOXA1','NUP98'],
        ['HOXA1','TRB@'],
        ['HOXA10','NUP98'],
        ['HOXA10','TCRBV@'],
        ['HOXA10','TCRVB@'],
        ['HOXA10','TRB@'],
        ['HOXA10','TRBV@'],
        ['HOXA11','NUP98'],
        ['HOXA11','PUM1'],
        ['HOXA11','TRB@'],
        ['HOXA11-AS','MIR181A1HG'],
        ['HOXA11-AS','TRBC2'],
        ['HOXA13','NUP98'],
        ['HOXA3','NUP98'],
        ['HOXA6','NUP98'],
        ['HOXA9','MED12'],
        ['HOXA9','MEIS1'],
        ['HOXA9','MSI2'],
        ['HOXA9','NSD1'],
        ['HOXA9','NUP98'],
        ['HOXA9','PBX3'],
        ['HOXA9','TRB@'],
        ['HOXA9','WHSC1L1'],
        ['HOXB3','NUP98'],
        ['HOXB3','RARA'],
        ['HOXB4','NUP98'],
        ['HOXB9','NIPBL'],
        ['HOXC10','WIBG'],
        ['HOXC11','NUP98'],
        ['HOXC12','NUP98'],
        ['HOXC13','NUP98'],
        ['HOXD1','NUP98'],
        ['HOXD11','NUP98'],
        ['HOXD13','NUP98'],
        ['HOXD3','NUP98'],
        ['HOXD8','TM9SF4'],
        ['HP','RSPO1'],
        ['HP1BP3','NUP85'],
        ['HPCA','TAT'],
        ['HPCA','TP53'],
        ['HPN-AS1','TP53'],
        ['HPR','MRPS10'],
        ['HPR','RPS10'],
        ['HPRG1','LRRC6'],
        ['HPS3','MYO10'],
        ['HPSE2','PRKD2'],
        ['HPX','TRIM3'],
        ['HRAS','PIK3CA'],
        ['HRAS','RAB3GAP1'],
        ['HRH4','PDGFRB'],
        ['HSD11B1','KCNH1'],
        ['HSD17B14','PLEKHA4'],
        ['HSD17B2','MTMR3'],
        ['HSD3B2','NF2'],
        ['HSF1','RERE'],
        ['HSP90AA1','IGH@'],
        ['HSP90AA1','PIM3'],
        ['HSP90AA1','TMPRSS2'],
        ['HSP90AB1','PTCHD4'],
        ['HSP90AB1','SLC29A1'],
        ['HSP90AB1','SYNGAP1'],
        ['HSP90B1','RDH13'],
        ['HSPA8','NR4A3'],
        ['HSPBP1','PLGRKT'],
        ['HSPE1','MOBKL3'],
        ['HSPG2','TMCO4'],
        ['HTN3','MSANTD3'],
        ['HTR2A','MED4'],
        ['HTR4','ST3GAL1'],
        ['HTRA1','SH3PXD2A'],
        ['HTRA3','TBC1D14'],
        ['HTT','PKN1'],
        ['HUNK','MAMLD1'],
        ['HUNK','SNRPD3'],
        ['HUWE1','RIBC1'],
        ['ICA1','TMPRSS2'],
        ['ICA1L','SRF'],
        ['ID1','WNK1'],
        ['ID3','TCF3'],
        ['ID4','IGH@'],
        ['IDH1','IDH2'],
        ['IDH1','KRAS'],
        ['IDH1','TP53'],
        ['IDO1','ME3'],
        ['IDS','IDSP1'],
        ['IDS','LINC00893'],
        ['IDUA','PTPRC'],
        ['IER2','STARD9'],
        ['IFFO2','UBR4'],
        ['IFI16','NOS1AP'],
        ['IFI44','LPHN2'],
        ['IFITM10','PPP1R12C'],
        ['IFNA2','TA1'],
        ['IFNL1','IL10RA'],
        ['IFRD1','PVT1'],
        ['IFT172','KRTCAP3'],
        ['IFT46','MLL'],
        ['IFT46','TBRG1'],
        ['IFT52','TRIO'],
        ['IFT80','MLH1'],
        ['IGF1R','IGF2'],
        ['IGF1R','MFGE8'],
        ['IGF1R','NBPF1'],
        ['IGF1R','PPP4R1'],
        ['IGF1R','TOLLIP'],
        ['IGF1R','TRIP4'],
        ['IGF1R','TTC23'],
        ['IGF1R','WDR93'],
        ['IGF2','INS'],
        ['IGF2','INS-IGF2'],
        ['IGF2','SNW1'],
        ['IGF2','TF'],
        ['IGF2','WNK2'],
        ['IGF2BP1','IGH@'],
        ['IGF2BP1','KAT7'],
        ['IGF2BP3','PRKCA'],
        ['IGF2BP3','RPS6KA2'],
        ['IGF2BP3','THADA'],
        ['IGF2R','SYTL3'],
        ['IGFBP3','LIMK1'],
        ['IGFBP3','THADA'],
        ['IGFBP5','PLAT'],
        ['IGFBP5','RGS5'],
        ['IGFBP5','SIAE'],
        ['IGFL2','TP53'],
        ['IGH@','IL3'],
        ['IGH@','IL31'],
        ['IGH@','IL31RA'],
        ['IGH@','IL32'],
        ['IGH@','IL33'],
        ['IGH@','IL34'],
        ['IGH@','IL36A'],
        ['IGH@','IL36B'],
        ['IGH@','IL36G'],
        ['IGH@','IL36RN'],
        ['IGH@','IL37'],
        ['IGH@','IL3RA'],
        ['IGH@','IMMP2L'],
        ['IGH@','IRF'],
        ['IGH@','IRF4'],
        ['IGH@','IRF8'],
        ['IGH@','IRTA1'],
        ['IGH@','KDM4C'],
        ['IGH@','KIAA0125'],
        ['IGH@','LAPTM5'],
        ['IGH@','LHX2'],
        ['IGH@','LHX4'],
        ['IGH@','LMO2'],
        ['IGH@','MAF'],
        ['IGH@','MAFA'],
        ['IGH@','MAFB'],
        ['IGH@','MALT1'],
        ['IGH@','MAP3K14'],
        ['IGH@','MECOM'],
        ['IGH@','MIR125B1'],
        ['IGH@','MLT1'],
        ['IGH@','MMSET'],
        ['IGH@','MRPL21'],
        ['IGH@','MUC1'],
        ['IGH@','MUM2'],
        ['IGH@','MYC'],
        ['IGH@','MYCN'],
        ['IGH@','MYEOV'],
        ['IGH@','NBEAP1'],
        ['IGH@','NFKB2'],
        ['IGH@','NKX2-1'],
        ['IGH@','NSD2'],
        ['IGH@','ODZ2'],
        ['IGH@','PAFAH1B2'],
        ['IGH@','PAX5'],
        ['IGH@','PCSK7'],
        ['IGH@','PDCD1LG2'],
        ['IGH@','PDGFRA'],
        ['IGH@','POU5F1B'],
        ['IGH@','PPP4R3A'],
        ['IGH@','PVRL2'],
        ['IGH@','PVT1'],
        ['IGH@','RHOH'],
        ['IGH@','SPIB'],
        ['IGH@','TCF3'],
        ['IGH@','TENM2'],
        ['IGH@','TERT'],
        ['IGH@','TNFSF13'],
        ['IGH@','TOP1MT'],
        ['IGH@','WHSC1'],
        ['IGH@','WWOX'],
        ['IGH@','ZEB2'],
        ['IGH@','ZFP36L1'],
        ['IGHA2','ZFP36L1'],
        ['IGHG1','LHX4'],
        ['IGHG1','MYC'],
        ['IGHMBP2','MRPL21'],
        ['IGK@','IRF4'],
        ['IGK@','KDSR'],
        ['IGK@','MMSET'],
        ['IGK@','MYC'],
        ['IGK@','PVT1'],
        ['IGK@','WHSC1'],
        ['IGL@','IRF4'],
        ['IGL@','MAF'],
        ['IGL@','MYC'],
        ['IGL@','PDCD1LG2'],
        ['IGL@','PRDM16'],
        ['IGL@','PVT1'],
        ['IGL@','WHSC1'],
        ['IGL@','WWOX'],
        ['IGLC','PVT1'],
        ['IGSF10','MED12L'],
        ['IGSF21','OBSCN'],
        ['IGSF9B','TOM1L2'],
        ['IKBKB','TNPO1'],
        ['IKBKB','UNC5D'],
        ['IKBKE','NFASC'],
        ['IKBKG','PLXNB3'],
        ['IKBKG','RET'],
        ['IKZF1','KLF2'],
        ['IKZF1','NOTCH1'],
        ['IKZF1','NUTM1'],
        ['IKZF1','RNASEH2B'],
        ['IKZF1','SETD5'],
        ['IKZF1','STIM2'],
        ['IKZF1','TRPV2'],
        ['IKZF1','TYW1'],
        ['IKZF1','ZEB2'],
        ['IKZF3','MED1'],
        ['IKZF3','NAGLU'],
        ['IKZF3','NF1'],
        ['IKZF3','RARA'],
        ['IKZF3','VAPB'],
        ['IKZF4','RAB40B'],
        ['IL18','IL2'],
        ['IL1RAP','TFRC'],
        ['IL1RAPL1','KDM6A'],
        ['IL1RAPL1','KERA'],
        ['IL1RAPL1','VDAC3'],
        ['IL1RAPL1','ZNF611'],
        ['IL1RAPL2','MYO3B'],
        ['IL1RL1','ZCCHC2'],
        ['IL2','TNFRSF17'],
        ['IL22','VOPP1'],
        ['IL26','PPM1H'],
        ['IL27RA','SMARCA4'],
        ['IL2RB','MYH9'],
        ['IL32','NAA60'],
        ['IL32','TNFRSF12A'],
        ['IL6R','TPM3'],
        ['IMMP2L','LMTK2'],
        ['IMMP2L','SMC5'],
        ['IMMT','SLC25A26'],
        ['INADL','RAF1'],
        ['ING3','NUP98'],
        ['INHBC','METTL1'],
        ['INHBC','TSFM'],
        ['INHBE','TSFM'],
        ['INI1','SMARCB1'],
        ['INO80','PET112'],
        ['INPP1','MFSD6'],
        ['INPP4A','TMEM131'],
        ['INPP4B','MAPK10'],
        ['INPP5A','ROR1'],
        ['INPP5A','TMEM189-UBE2V1'],
        ['INS','INS-IGF2'],
        ['INSL3','JAK3'],
        ['INSL4','JAK2'],
        ['INSL6','JAK2'],
        ['INSR','MAP2K2'],
        ['INSR','RBFOX3'],
        ['INSR','S1PR2'],
        ['INSR','SLC12A2'],
        ['INTS1','PRKAR1B'],
        ['INTS12','MAK'],
        ['INTS12','TBCK'],
        ['INTS2','TMEM49'],
        ['INTS2','ZNF652'],
        ['INTS6','SMIM14'],
        ['IP6K1','MST1R'],
        ['IP6K1','RYBP'],
        ['IP6K1','TRAIP'],
        ['IP6K2','USP19'],
        ['IPO11','LRRC70'],
        ['IPO7','MYH9'],
        ['IQCE','ME2'],
        ['IQCG','NUP98'],
        ['IQCH','MAP2K5'],
        ['IQCJ','SCHIP1'],
        ['IQGAP1','ZNF774'],
        ['IQGAP2','TSLP'],
        ['IQGAP3','PIGR'],
        ['IQSEC3','MAPKAPK2'],
        ['IQSEC3','WNK1'],
        ['IRA1','RGS17'],
        ['IRAK1','LYZ'],
        ['IRAK1','PIGR'],
        ['IRAK2','SRGAP3'],
        ['IRAK3','PPM1H'],
        ['IRAK3','RUNX2'],
        ['IRAK3','TLK2'],
        ['IRF2BP1','RARA'],
        ['IRF2BP2','NTRK1'],
        ['IRF2BP2','NTRK2'],
        ['IRF2BP2','NTRK3'],
        ['IRF2BP2','PPARD'],
        ['IRF2BP2','PVT1'],
        ['IRF2BP2','RARA'],
        ['IRF2BPL','MED12'],
        ['IRF4','MUM1'],
        ['IRF4','TRA@'],
        ['IRF9','RNF31'],
        ['IRS2','PRKCI'],
        ['IRX2','TERT'],
        ['ISG20L2','RFWD3'],
        ['ITCH','RPRD1B'],
        ['ITFG1','NETO2'],
        ['ITFG1','RYK'],
        ['ITFG2','NRIP2'],
        ['ITGA1','PDE4D'],
        ['ITGA11','NARF'],
        ['ITGA11','TRAPPC2'],
        ['ITGA6','PDK1'],
        ['ITGA6','RAPGEF4'],
        ['ITGA8','RET'],
        ['ITGAV','MALAT1'],
        ['ITGB3','TP53'],
        ['ITGB4','UNK'],
        ['ITGB5','SNX4'],
        ['ITGB6','RBMS1'],
        ['ITGB6','TANC1'],
        ['ITGB8','MALAT1'],
        ['ITGB8','ZCWPW1'],
        ['ITIH2','RET'],
        ['ITK','SYK'],
        ['ITLN1','PIP5K1A'],
        ['ITLN1','VANGL2'],
        ['ITM2B','RB1'],
        ['ITPK1','SLC38A1'],
        ['ITPKC','PPFIA3'],
        ['ITPR2','KMT2A'],
        ['ITPR2','TMED10'],
        ['ITPR3','PNPLA1'],
        ['ITPRIPL1','YWHAQ'],
        ['ITSN1','KCNE2'],
        ['ITSN1','PAM'],
        ['IVNS1ABP','KYNU'],
        ['IZUMO3','MLLT3'],
        ['JADE2','NUP98'],
        ['JAG1','SRRM2'],
        ['JAK1','JAK2'],
        ['JAK1','RB1CC1'],
        ['JAK1','SCARB2'],
        ['JAK1','STAT3'],
        ['JAK1','ZC3H7B'],
        ['JAK2','KIAA1432'],
        ['JAK2','LRP8'],
        ['JAK2','MYH9'],
        ['JAK2','NFE2'],
        ['JAK2','OFD1'],
        ['JAK2','PAX5'],
        ['JAK2','PCM1'],
        ['JAK2','PDCD1LG2'],
        ['JAK2','PDL1'],
        ['JAK2','PEX14'],
        ['JAK2','PPFIBP1'],
        ['JAK2','PTPRD'],
        ['JAK2','RFX3'],
        ['JAK2','RIC1'],
        ['JAK2','RNPC3'],
        ['JAK2','RPN1'],
        ['JAK2','SEC31A'],
        ['JAK2','SMU1'],
        ['JAK2','SNX29'],
        ['JAK2','SPAG9'],
        ['JAK2','SSBP2'],
        ['JAK2','STAM'],
        ['JAK2','STAT3'],
        ['JAK2','STAT5'],
        ['JAK2','STRBP'],
        ['JAK2','STRN3'],
        ['JAK2','TEL'],
        ['JAK2','TERF2'],
        ['JAK2','TPM3'],
        ['JAK2','TPR'],
        ['JAK2','TTC13'],
        ['JAK2','USP25'],
        ['JAK2','ZBTB20'],
        ['JAK2','ZNF274'],
        ['JAK2','ZNF340'],
        ['JAK2','ZNF430'],
        ['JAK3','SEC31A'],
        ['JAK3','STAT5B'],
        ['JAM3','MAP4K5'],
        ['JAM3','NCAPD3'],
        ['JARID1A','NUP98'],
        ['JAZF1','JJAZ1'],
        ['JAZF1','PHF1'],
        ['JAZF1','SUZ12'],
        ['JJAZ1','SUZ12'],
        ['JMJD1C','MICU1'],
        ['JMJD1C','NRBF2'],
        ['JMJD1C-AS1','REEP3'],
        ['JMJD7','PLA2G4B'],
        ['JSRP1','THAP9'],
        ['JUNB','SEC63'],
        ['KAL1','PNPLA4'],
        ['KANK1','NTRK2'],
        ['KANK1','NTRK3'],
        ['KANK1','PDGFRB'],
        ['KANSL1','KAT6A'],
        ['KANSL1','KAT6B'],
        ['KANSL1','LINGO2'],
        ['KANSL1','LRRC37A3'],
        ['KANSL1','MTCP1'],
        ['KAT5','PPP1R21'],
        ['KAT6A','LEUTX'],
        ['KAT6A','LRP5'],
        ['KAT6A','NCOA2'],
        ['KAT6A','NCOA3'],
        ['KAT6A','P300'],
        ['KAT6A','PLAT'],
        ['KAT6A','PTK2B'],
        ['KAT6A','TFE3'],
        ['KAT6A','TIF2'],
        ['KATNAL1','ZC3H13'],
        ['KAZN','MTOR'],
        ['KAZN','TRPM7'],
        ['KCNAB1','PIK3CB'],
        ['KCNAB2','MMAB'],
        ['KCNAB2','POLB'],
        ['KCND3','PPM1E'],
        ['KCNH7','MAP3K19'],
        ['KCNIP2','MGEA5'],
        ['KCNIP2','SGSM1'],
        ['KCNJ10','MSTO1'],
        ['KCNJ16','TLK2'],
        ['KCNJ4','MAPK1'],
        ['KCNK3','ZNF512'],
        ['KCNMA1','MDS1'],
        ['KCNMA1','RUNX1'],
        ['KCNMB3P1','NF2'],
        ['KCNQ5','RIMS1'],
        ['KCNS3','MYCNOS'],
        ['KCNS3','ZNF638'],
        ['KCNT2','RUNX1'],
        ['KCTD10','PTGR2'],
        ['KCTD16','NTRK1'],
        ['KCTD16','NTRK2'],
        ['KCTD19','SDHB'],
        ['KCTD20','SORBS2'],
        ['KCTD21','PAK1'],
        ['KCTD3','RPL23AP7'],
        ['KCTD8','NTRK2'],
        ['KDELR2','ROS1'],
        ['KDM1B','TP53'],
        ['KDM2A','LEMD2'],
        ['KDM2A','MRPL21'],
        ['KDM2A','RAD9A'],
        ['KDM2A','RHOD'],
        ['KDM2A','RPS6KB2'],
        ['KDM2A','SYT12'],
        ['KDM2A','WWTR1'],
        ['KDM2B','XIRP2'],
        ['KDM4A','RASSF5'],
        ['KDM4A','ST3GAL3'],
        ['KDM4B','SNTG1'],
        ['KDM4B','STK11'],
        ['KDM4D','ZC3H12D'],
        ['KDM5A','NINJ2'],
        ['KDM5A','NUP98'],
        ['KDM5A','TULP3'],
        ['KDM5A','WNK1'],
        ['KDM5B','LINC01351'],
        ['KDM5B','PVT1'],
        ['KDM6A','LINC01628'],
        ['KDM6A','PTK2B'],
        ['KDM6A','RPS6KA3'],
        ['KDM6A','XIST'],
        ['KDM6B','TP53'],
        ['KDR','PDGFRA'],
        ['KDR','PLCG1'],
        ['KDR','SH3BP2'],
        ['KDR','VEGFR2'],
        ['KHDRBS1','NTRK1'],
        ['KHDRBS1','NTRK3'],
        ['KHDRBS1','PLCH2'],
        ['KHDRBS2','PNISR'],
        ['KHDRBS2','TFE3'],
        ['KHDRBS3','PVT1'],
        ['KHNYN','SDR39U1'],
        ['KHSRP','TFE3'],
        ['KIAA0040','RFWD2'],
        ['KIAA0100','UBE4B'],
        ['KIAA0125','MYEOV'],
        ['KIAA0146','MCM4'],
        ['KIAA0195','MGAT5B'],
        ['KIAA0284','KMT2A'],
        ['KIAA0319','TDP2'],
        ['KIAA0319L','ZMYM4'],
        ['KIAA0355','UNC13A'],
        ['KIAA0368','NF2'],
        ['KIAA0368','ROBO2'],
        ['KIAA0430','SPR'],
        ['KIAA0515','PPAPDC3'],
        ['KIAA0564','SCAF8'],
        ['KIAA0642','TRPM8'],
        ['KIAA0660','TK14'],
        ['KIAA0753','TP53'],
        ['KIAA0756','NTRK1'],
        ['KIAA0825','MGAT5'],
        ['KIAA0833','MTOR'],
        ['KIAA0895','PDE1C'],
        ['KIAA0895L','NPIPA5'],
        ['KIAA0984','PPM1H'],
        ['KIAA0999','PPP2R1B'],
        ['KIAA1033','LGR5'],
        ['KIAA1199','ZYX'],
        ['KIAA1211','NMU'],
        ['KIAA1217','RET'],
        ['KIAA1217','RET51'],
        ['KIAA1244','TMPRSS2'],
        ['KIAA1267','MBTD1'],
        ['KIAA1324L','SRPK2'],
        ['KIAA1324L','TMEM243'],
        ['KIAA1429','SUSD3'],
        ['KIAA1462','KIF5B'],
        ['KIAA1467','TTLL9'],
        ['KIAA1468','RET'],
        ['KIAA1509','PDGFRB'],
        ['KIAA1524','KMT2A'],
        ['KIAA1524','MLL'],
        ['KIAA1549L','PAX5'],
        ['KIAA1549L','RUNX1'],
        ['KIAA1598','ROS1'],
        ['KIAA1598','UTP11L'],
        ['KIAA1618','TBC1D16'],
        ['KIAA1671','NF2'],
        ['KIAA1737','NGFR'],
        ['KIAA1787','SENP3'],
        ['KIAA1797','MOBKL2B'],
        ['KIAA1967','SORBS3'],
        ['KIAA2018','MBNL1'],
        ['KIAA2022','MRX98'],
        ['KIAA2026','MIR31HG'],
        ['KIAA2026','NUDT11'],
        ['KIAA2026','UHRF2'],
        ['KIDINS220','PAX5'],
        ['KIDINS220','PHF20'],
        ['KIF13B','NRG1'],
        ['KIF16B','PCSK2'],
        ['KIF18A','MTFR1L'],
        ['KIF1B','TCF7L1'],
        ['KIF1B','UBE4B'],
        ['KIF1C','RNF167'],
        ['KIF1C','SOAT2'],
        ['KIF21A','MED13L'],
        ['KIF21A','ZNF664'],
        ['KIF26A','MARK3'],
        ['KIF3B','PAX5'],
        ['KIF5A','OS9'],
        ['KIF5B','MET'],
        ['KIF5B','NTRK2'],
        ['KIF5B','PDGFRA'],
        ['KIF5B','RET'],
        ['KIF5C','LYPD6B'],
        ['KIFAP3','SCYL3'],
        ['KIFC1','RPS6KA2'],
        ['KIRREL','PRKCA'],
        ['KIRREL','ZFYVE9'],
        ['KIRREL3','NCOR2'],
        ['KIRREL3','RN7SL167P'],
        ['KIRREL3','VPS37C'],
        ['KISS1','PIK3C2B'],
        ['KIT','PDGFR'],
        ['KIT','PDGFRA'],
        ['KIT','SGCB'],
        ['KIT','SLC4A4'],
        ['KLB','PDS5A'],
        ['KLC1','ROS1'],
        ['KLC1','ZFAT'],
        ['KLC2','LUC7L3'],
        ['KLF10','PVT1'],
        ['KLF2','OAZ1'],
        ['KLF2','TPM4'],
        ['KLF3','SLC30A9'],
        ['KLHDC10','STAC'],
        ['KLHDC2','SNTB1'],
        ['KLHDC4','LRPAP1'],
        ['KLHDC8A','PCMTD1'],
        ['KLHL18','TYRO3'],
        ['KLHL23','PHOSPHO2'],
        ['KLHL29','NR5A2'],
        ['KLHL9','MLLT3'],
        ['KLK2','TMPRSS2'],
        ['KLK3','NUAK1'],
        ['KLK3','TP53'],
        ['KLK4','KLKP1'],
        ['KLKP1','SRPK2'],
        ['KLRD1','LINC00545'],
        ['KMO','RPS6KC1'],
        ['KMT2A','LASP1'],
        ['KMT2A','LPP'],
        ['KMT2A','MAML1'],
        ['KMT2A','MAML2'],
        ['KMT2A','MAPRE1'],
        ['KMT2A','MATR3'],
        ['KMT2A','ME2'],
        ['KMT2A','MEF2C'],
        ['KMT2A','MLL'],
        ['KMT2A','MLLT1'],
        ['KMT2A','MLLT10'],
        ['KMT2A','MLLT11'],
        ['KMT2A','MLLT3'],
        ['KMT2A','MLLT4'],
        ['KMT2A','MLLT6'],
        ['KMT2A','MN1'],
        ['KMT2A','MYH11'],
        ['KMT2A','MYO1F'],
        ['KMT2A','NCBP1'],
        ['KMT2A','NCKIPSD'],
        ['KMT2A','NEBL'],
        ['KMT2A','NRIP3'],
        ['KMT2A','NT5C2'],
        ['KMT2A','NUP98'],
        ['KMT2A','PDS5A'],
        ['KMT2A','PICALM'],
        ['KMT2A','PRRC1'],
        ['KMT2A','PRRX1'],
        ['KMT2A','PTPRC'],
        ['KMT2A','RAD51B'],
        ['KMT2A','RARA'],
        ['KMT2A','RFX2'],
        ['KMT2A','RNF10'],
        ['KMT2A','SARNP'],
        ['KMT2A','SEPT11'],
        ['KMT2A','SEPT2'],
        ['KMT2A','SEPT5'],
        ['KMT2A','SEPT6'],
        ['KMT2A','SEPT9'],
        ['KMT2A','SH3GL1'],
        ['KMT2A','SMAP1'],
        ['KMT2A','SNX9'],
        ['KMT2A','SORBS2'],
        ['KMT2A','SUGP2'],
        ['KMT2A','TCF12'],
        ['KMT2A','TET1'],
        ['KMT2A','TIRAP'],
        ['KMT2A','TMX2'],
        ['KMT2A','TNRC18'],
        ['KMT2A','TOP3A'],
        ['KMT2A','UBE4A'],
        ['KMT2A','USO1'],
        ['KMT2A','USP10'],
        ['KMT2A','USP2'],
        ['KMT2A','USP8'],
        ['KMT2A','VAV1'],
        ['KMT2A','VIM'],
        ['KMT2A','YAP1'],
        ['KMT2A','ZFYVE19'],
        ['KMT2A','ZNF56'],
        ['KMT2C','LIPJ'],
        ['KMT2C','LMTK3'],
        ['KMT2C','MLL3'],
        ['KMT2C','TP53'],
        ['KMT2D','SEC31A'],
        ['KPNA2','OSBPL9'],
        ['KPNA3','SLC2A8'],
        ['KPNA3','TP53'],
        ['KPNA4','WWTR1'],
        ['KRAS','LYRM5'],
        ['KRAS','NRAS'],
        ['KRAS','SOX5'],
        ['KRAS','UBE2L3'],
        ['KRR1','RUNX1'],
        ['KRT14','KRT5'],
        ['KRT14','KRT6'],
        ['KRT14','RARA'],
        ['KRT16','KRT5'],
        ['KRT16','KRT6'],
        ['KRT17','KRT5'],
        ['KRT17','KRT6'],
        ['KRT17','S100A9'],
        ['KRT18P6','MYC'],
        ['KRT18P6','TAL1'],
        ['KRT223P','NIPAL2'],
        ['KRT32','SEPT9'],
        ['KRT6A','ZNF821'],
        ['KRT8','PKM2'],
        ['KRT86','MDM2'],
        ['KRTDAP','NPVF'],
        ['KSR1','ODZ1'],
        ['KSR1','PKN3'],
        ['KSR1','TENM1'],
        ['KSR2','PTPRR'],
        ['KSR2','SETMAR'],
        ['KTN1','PRKD1'],
        ['KTN1','RET'],
        ['L3MBTL4','PAK4'],
        ['LACC1','NRBF2'],
        ['LACE1','LATS1'],
        ['LACTB','USP8'],
        ['LACTB2','NCOA2'],
        ['LAF4','MLL'],
        ['LAF4','RUNX1'],
        ['LAG3','MYH11'],
        ['LAIR1','TTYH1'],
        ['LAMA1','PTPRM'],
        ['LAMA3','RBBP8'],
        ['LAMA3','RIOK3'],
        ['LAMA3','ROCK1'],
        ['LAMB1','MALAT1'],
        ['LAMC1','MALAT1'],
        ['LAMC1','PIK3C2B'],
        ['LAMP1','MCF2L'],
        ['LAMP2','SCARB2'],
        ['LAMTOR1','PRKCD'],
        ['LAMTOR4','SRPK2'],
        ['LANCL2','PLEKHA7'],
        ['LANCL2','PSPH'],
        ['LANCL2','SEC61G'],
        ['LANCL2','SEPT14'],
        ['LANCL2','VOPP1'],
        ['LANCL2','VSTM2A'],
        ['LANCL2','ZNF713'],
        ['LAPTM4A','PRMT8'],
        ['LAPTM5','NFIL3'],
        ['LARG','MLL'],
        ['LARP4B','PPIL4'],
        ['LARP4B','SEC63'],
        ['LAS1L','OPHN1'],
        ['LASP1','MLL'],
        ['LASP1','TRPS1'],
        ['LASP1','WIPF2'],
        ['LASP2','NEBL'],
        ['LASS6','USP32'],
        ['LATS1','NUP43'],
        ['LATS1','PCMT1'],
        ['LATS1','PSEN1'],
        ['LATS2','MYLK'],
        ['LATS2','NUP107'],
        ['LATS2','PAN3'],
        ['LATS2','ZMYM2'],
        ['LAZ3','LCP1'],
        ['LCAT','SLC12A4'],
        ['LCB1','LCB2'],
        ['LCK','TCR@'],
        ['LCK','TLX1'],
        ['LCK','TLX1NB'],
        ['LCK','TRB@'],
        ['LCLAT1','TMEM55A'],
        ['LCX','MLL'],
        ['LDHB','MTX1'],
        ['LDHC','SERGEF'],
        ['LDLR','RPL31P11'],
        ['LDLR','SMARCA4'],
        ['LDLR','TERT'],
        ['LDLRAD3','NCAM1'],
        ['LDLRAD3','RICS'],
        ['LDLRAD3','TCP11L1'],
        ['LDLRAD4','PHACTR3'],
        ['LDLRAD4','RUNX1'],
        ['LDLRAD4','SPECC1'],
        ['LEDGF','NUP98'],
        ['LEF1','METTL23'],
        ['LEKR1','TIPARP'],
        ['LEMD1','NUAK2'],
        ['LEMD3','NUP107'],
        ['LEMD3','PPM1H'],
        ['LEO1','SLC12A1'],
        ['LEP','SND1'],
        ['LGALS12','SYNPO2L'],
        ['LGALS2','RBFOX2'],
        ['LGALS2','RBM9'],
        ['LGALS9','NLK'],
        ['LGALS9','SPOP'],
        ['LGP2','MDA5'],
        ['LGR4','RSPO3'],
        ['LGR5','NUP107'],
        ['LGSN','LMBRD1'],
        ['LHFP','LNX2'],
        ['LHFP','RORA'],
        ['LHFP','SERP2'],
        ['LHFPL2','TERT'],
        ['LHFPL3','MLL5'],
        ['LHFPL3','SND1'],
        ['LHFPL5','NUFIP2'],
        ['LHX6','NDUFA1'],
        ['LHX6','NDUFA8'],
        ['LIF','OSM'],
        ['LIFR','PLAG1'],
        ['LIFR','TRIO'],
        ['LIFR','YTHDF3'],
        ['LIMA1','ROS1'],
        ['LIMA1','SCMH1'],
        ['LIMA1','USP22'],
        ['LIMK1','TYW1'],
        ['LIN28A','SRGAP1'],
        ['LIN28B','STX7'],
        ['LIN52','S100A12'],
        ['LIN54','NAGS'],
        ['LINC00152','RUFY1'],
        ['LINC00176','SLC6A19'],
        ['LINC00243','RASAL1'],
        ['LINC00330','PQLC1'],
        ['LINC00330','ZNF664'],
        ['LINC00462','RB1'],
        ['LINC00476','PTCH1'],
        ['LINC00478','NRIP1'],
        ['LINC00536','PVT1'],
        ['LINC00545','MEDAG'],
        ['LINC00703','MAP3K8'],
        ['LINC00824','PVT1'],
        ['LINC00843','TIMM23B'],
        ['LINC00964','PVT1'],
        ['LINC01054','TP53'],
        ['LINC01169','SMAD6'],
        ['LINC01504','TRIO'],
        ['LINC01513','ROPN1L'],
        ['LINC01545','TFE3'],
        ['LINC02197','PAX5'],
        ['LINC02210','SWSAP1'],
        ['LINC02649','PFKFB3'],
        ['LIPA','RUNX1'],
        ['LIPC','SH3RF3'],
        ['LIPF','MELK'],
        ['LIPI','NTRK1'],
        ['LIPJ','PTEN'],
        ['LITAF','LRP5'],
        ['LKB1','STK11'],
        ['LLGL2','PRKCA'],
        ['LMAN2','PTMA'],
        ['LMBRD1','TFDP2'],
        ['LMBRD2','SLC25A1'],
        ['LMCD1-AS1','MECOM'],
        ['LMF1','TAF15'],
        ['LMF1','ZFHX3'],
        ['LMNA','NTRK1'],
        ['LMNA','NTRK2'],
        ['LMNA','NTRK3'],
        ['LMNA','RAF1'],
        ['LMNA','TRKA'],
        ['LMNB2','MAP2K6'],
        ['LMO1','RIC3'],
        ['LMO1','TLX1'],
        ['LMO1','TLX1NB'],
        ['LMO1','TRA@'],
        ['LMO1','TRAC'],
        ['LMO1','TRB@'],
        ['LMO1','TRBC1'],
        ['LMO1','TRBC2'],
        ['LMO1','TRD@'],
        ['LMO1','YAP1'],
        ['LMO2','RAG1'],
        ['LMO2','TCR@'],
        ['LMO2','TLX1'],
        ['LMO2','TLX1NB'],
        ['LMO2','TRA@'],
        ['LMO2','TRB@'],
        ['LMO2','TRBC2'],
        ['LMO2','TRD@'],
        ['LMO2','TRDC'],
        ['LMTK2','RBFOX3'],
        ['LMX1B','MVB12B'],
        ['LNP1','NUP98'],
        ['LNP1','PAK1'],
        ['LNPEP','RASGEF1B'],
        ['LNX1','NMU'],
        ['LNX1','PDGFRA'],
        ['LNX1','USP46'],
        ['LNX2','POLR1D'],
        ['LOC284100','PSMD3'],
        ['LOC728621','PPCS'],
        ['LONP1','MAP2K7'],
        ['LONP1','SAFB'],
        ['LPAR1','NAA15'],
        ['LPCAT1','TERT'],
        ['LPCAT3','MAP3K13'],
        ['LPGAT1','RPS6KC1'],
        ['LPIN1','MYCN'],
        ['LPL','PEX5'],
        ['LPL','ZNF641'],
        ['LPP','MLL'],
        ['LPP','PRKCZ'],
        ['LPP','RFC4'],
        ['LPP','TPRG1'],
        ['LPP','WWTR1'],
        ['LPPR1','SLC35D2'],
        ['LPXN','PTPRJ'],
        ['LPXN','RUNX1'],
        ['LRBA','SH3D19'],
        ['LRCH3','RAB7A'],
        ['LRCH4','SEC31A'],
        ['LRIG1','TAT'],
        ['LRIG3','PIP4K2C'],
        ['LRIG3','ROCK1'],
        ['LRIG3','ROS1'],
        ['LRIG3','TSFM'],
        ['LRMP','MYC'],
        ['LRMP','PTPRD'],
        ['LRP1','RBMS2'],
        ['LRP1','SNRNP25'],
        ['LRP1','TMBIM4'],
        ['LRP1','TP53'],
        ['LRP1','XPO5'],
        ['LRP16','RUNX1'],
        ['LRP1B','MAST4'],
        ['LRP1B','ZC3H6'],
        ['LRP5','PPP6R3'],
        ['LRP5','SLC22A24'],
        ['LRP5','SRPK2'],
        ['LRP5','TPCN2'],
        ['LRP6','STYK1'],
        ['LRP6','TP53'],
        ['LRP8','PAX8'],
        ['LRP8','TMEM48'],
        ['LRR1','PPIL6'],
        ['LRRC2','PRAMEF13'],
        ['LRRC36','SLC7A6'],
        ['LRRC37A3','NSF'],
        ['LRRC37B','NF1'],
        ['LRRC59','NEUROD2'],
        ['LRRC59','SLC22A10'],
        ['LRRC6','PHF20L1'],
        ['LRRC6','POLB'],
        ['LRRC61','PTPN13'],
        ['LRRC71','NTRK1'],
        ['LRRC8D','ZNF804A'],
        ['LRRFIP1','RBM44'],
        ['LRRFIP2','RAF1'],
        ['LRRFIP2','SUMF1'],
        ['LRRK1','METTL3'],
        ['LRRK1','TOX4'],
        ['LRRN2','NFASC'],
        ['LRWD1','MAGEA9'],
        ['LSAMP','ZBTB20'],
        ['LSAMP','ZNF596'],
        ['LSM14A','PPARG'],
        ['LSM14A','SLIT3'],
        ['LSM14A','TERT'],
        ['LSM14B','TAF4'],
        ['LSM4','PAK4'],
        ['LSS','POMK'],
        ['LSS','UMODL1'],
        ['LST1','LTB'],
        ['LTBP3','RELA'],
        ['LTK','UACA'],
        ['LUC7L3','TFE3'],
        ['LUM','USP6'],
        ['LXN','RPL21'],
        ['LY6G6D','STK19'],
        ['LY6H','PVT1'],
        ['LY9','SDHC'],
        ['LYL1','TLX1'],
        ['LYL1','TLX1NB'],
        ['LYL1','TRB@'],
        ['LYL1','TRMT'],
        ['LYN','NCOR1'],
        ['LYN','NTRK1'],
        ['LYN','NTRK2'],
        ['LYN','NTRK3'],
        ['LYN','SHB'],
        ['LYN','VPS13B'],
        ['LYPD1','SCGB1A1'],
        ['LYPD5','MARK4'],
        ['LYST','RPS6KC1'],
        ['LYZ','PRKCI'],
        ['LYZ','STK10'],
        ['LYZ','STRADA'],
        ['LYZL2','MAP3K8'],
        ['LZTR1','PI4KA'],
        ['MACF','NOTCH2'],
        ['MACF1','MALAT1'],
        ['MACF1','MYCL'],
        ['MACF1','NOTCH2'],
        ['MACF1','RRAGC'],
        ['MACF1','TP53'],
        ['MACROD1','RUNX1'],
        ['MACROD1','SART1'],
        ['MACROD1','VEGFB'],
        ['MACROD2','PDGFRA'],
        ['MACROD2','ROR1'],
        ['MAD1L1','RARS'],
        ['MAD2B','PRCC'],
        ['MAF','MAFB'],
        ['MAFF','STK16'],
        ['MAFG','NELL2'],
        ['MAFK','OAZ'],
        ['MAFK','OAZ1'],
        ['MAFK','PDGFA'],
        ['MAGED2','PLAG1'],
        ['MAGI1','PDZRN3'],
        ['MAGI1','STMN2'],
        ['MAGI2','PARK2'],
        ['MAGI3','NOTCH2'],
        ['MAL','MKL1'],
        ['MALAT1','MUC16'],
        ['MALAT1','PIGR'],
        ['MALAT1','RGPD2'],
        ['MALAT1','RMRP'],
        ['MALAT1','RNF213'],
        ['MALAT1','RPPH1'],
        ['MALAT1','SMARCB1'],
        ['MALAT1','SMG'],
        ['MALAT1','SMG1'],
        ['MALAT1','SPP1'],
        ['MALAT1','SPTAN1'],
        ['MALAT1','SYNE2'],
        ['MALAT1','TFEB'],
        ['MALAT1','TMPRSS2'],
        ['MALAT1','TRPS1'],
        ['MALAT1','USO1'],
        ['MALAT1','USP9X'],
        ['MALAT1','VCAN'],
        ['MALAT1','VMP1'],
        ['MALAT1','VPS13B'],
        ['MALAT1','WWOX'],
        ['MALAT1','XIST'],
        ['MALT1','MAP4'],
        ['MALT1','MLT1'],
        ['MALT1','PMAIP1'],
        ['MALT1','SEC11C'],
        ['MAMDC1','SLC25A21'],
        ['MAML1','NR1D1'],
        ['MAML2','MECT1'],
        ['MAML2','MLL'],
        ['MAML2','MTMR2'],
        ['MAML2','MYB'],
        ['MAML2','PAFAH1B2'],
        ['MAML2','POMC'],
        ['MAML2','PTBP1'],
        ['MAML2','SMARCB1'],
        ['MAML2','TORC1'],
        ['MAML2','WWTR1'],
        ['MAML2','YAP'],
        ['MAML2','YAP1'],
        ['MAML3','MRPL39'],
        ['MAML3','PAX3'],
        ['MAML3','STRN3'],
        ['MAML3','TCF4'],
        ['MAML3','UBTF'],
        ['MAMLD1','MTM1'],
        ['MAMLD1','YAP'],
        ['MAMLD1','YAP1'],
        ['MAN1A1','ROS1'],
        ['MAN1A2','RUNX1'],
        ['MANBA','WNK1'],
        ['MANBAL','SRC'],
        ['MAP2K1','NCOR1'],
        ['MAP2K1','PLEKHO2'],
        ['MAP2K1','SMAD3'],
        ['MAP2K1','TIPIN'],
        ['MAP2K2','PSMD9'],
        ['MAP2K2','RAB8A'],
        ['MAP2K4','MAP2K4P1'],
        ['MAP2K4','MYH3'],
        ['MAP2K5','RAB27A'],
        ['MAP2K5','SNRNP40'],
        ['MAP2K7','PRDM15'],
        ['MAP2K7','SH3GL1'],
        ['MAP2K7','SNAPC2'],
        ['MAP3K1','PARN'],
        ['MAP3K10','SUPT5H'],
        ['MAP3K12','TEAD4'],
        ['MAP3K13','NSUN2'],
        ['MAP3K13','UTS2B'],
        ['MAP3K13','VKORC1'],
        ['MAP3K14','SPECC1'],
        ['MAP3K14','ZNF257'],
        ['MAP3K15','POLA1'],
        ['MAP3K2','SUPT3H'],
        ['MAP3K2','TIAM1'],
        ['MAP3K2','TMEM163'],
        ['MAP3K3','MRC2'],
        ['MAP3K3','MYO15B'],
        ['MAP3K3','TNS3'],
        ['MAP3K4','ZNF292'],
        ['MAP3K5','NKAIN2'],
        ['MAP3K5','PERP'],
        ['MAP3K5','TCF12'],
        ['MAP3K8','MIR3681HG'],
        ['MAP3K8','MROH2B'],
        ['MAP3K8','PCDH7'],
        ['MAP3K8','PIP4K2A'],
        ['MAP3K8','PRKACB'],
        ['MAP3K8','SFMBT2'],
        ['MAP3K8','SLC4A4'],
        ['MAP3K8','SPECC1'],
        ['MAP3K8','STX7'],
        ['MAP3K8','SVIL'],
        ['MAP3K8','UBL3'],
        ['MAP3K9','RBM15'],
        ['MAP3K9','SLC39A9'],
        ['MAP4','RB1'],
        ['MAP4','SMARCC1'],
        ['MAP4K3','SMC6'],
        ['MAP4K3','SOS1'],
        ['MAP4K3','TMEM178'],
        ['MAP4K4','RRBP1'],
        ['MAP4K4','SYF2'],
        ['MAP4K4','TP53'],
        ['MAP4K5','NIN'],
        ['MAP4K5','THSD7A'],
        ['MAP4K5','TRIM9'],
        ['MAP7','NFIB'],
        ['MAP7','PERP'],
        ['MAP7','RNF217'],
        ['MAPK1','METTL21B'],
        ['MAPK1','PRODH'],
        ['MAPK1','SSU72'],
        ['MAPK1','UBE2L3'],
        ['MAPK10','NUP188'],
        ['MAPK10','SPARCL1'],
        ['MAPK10','UGT8'],
        ['MAPK14','MICU2'],
        ['MAPK14','SIN3A'],
        ['MAPK15','MED25'],
        ['MAPK4','RTTN'],
        ['MAPK4','SMAD4'],
        ['MAPK6','USO1'],
        ['MAPK7','NCOR1'],
        ['MAPK7','RNF112'],
        ['MAPK8','RAP1B'],
        ['MAPK8IP3','PDPK1'],
        ['MAPK8IP3','UBE2I'],
        ['MAPK9','RAI14'],
        ['MAPKAPK2','PLA2G2A'],
        ['MAPKBP1','MGA'],
        ['MAPRE1','MLL'],
        ['MARCH1','PTPN13'],
        ['MARCH10','USP32'],
        ['MARCH6','MYO10'],
        ['MARCH6','TXLNA'],
        ['MARCH8','PRKG1'],
        ['MARCH9','METTL1'],
        ['MARCH9','NOL10'],
        ['MARCH9','OS9'],
        ['MARCH9','SLCO1B7'],
        ['MARK1','PA2G4'],
        ['MARK1','RHOU'],
        ['MARK2','PPFIA1'],
        ['MARK2','RTN3'],
        ['MARK3','MIA2'],
        ['MARK3','PDIA4'],
        ['MARK3','SELRC1'],
        ['MARK3','SLC25A21'],
        ['MARK3','SMARCB1'],
        ['MARS','MON2'],
        ['MARS','PPM1H'],
        ['MARVELD1','SYNDIG1'],
        ['MARVELD3','PJA2'],
        ['MAST1','NFIX'],
        ['MAST1','NUP210'],
        ['MAST1','TADA2A'],
        ['MAST1','TADA2L'],
        ['MAST1','ZNF700'],
        ['MAST2','METTL21A'],
        ['MAST2','MMEL1'],
        ['MAST2','PSMA5'],
        ['MAST2','SLC39A6'],
        ['MAST3','TBL1XR1'],
        ['MAST4','NDUFAF2'],
        ['MAST4','NLN'],
        ['MAST4','PLIN2'],
        ['MAST4','SREK1IP1'],
        ['MASTL','SEL1L'],
        ['MAT1A','STK39'],
        ['MAT1A','ZMIZ1'],
        ['MATK','PIP5K1C'],
        ['MATK','ZFR2'],
        ['MATN2','PKHD1L1'],
        ['MATN2','POP1'],
        ['MATR3','NPM1'],
        ['MATR3','TFE3'],
        ['MAX','MYC'],
        ['MAX','TTC7B'],
        ['MAZ','MKL1'],
        ['MB','TP53'],
        ['MBD1','RET'],
        ['MBD2','PERP'],
        ['MBD3','TCF3'],
        ['MBD3','ULK2'],
        ['MBD4','PTPRC'],
        ['MBNL1','MYB'],
        ['MBNL1','PAX5'],
        ['MBNL1','RAF1'],
        ['MBNL1','TAL1'],
        ['MBOAT2','PRKCE'],
        ['MBOAT2','ZBTB16'],
        ['MBOAT7','SIGLEC8'],
        ['MBP','PQLC1'],
        ['MBP','SLC66A2'],
        ['MBTD1','PHF1'],
        ['MBTD1','SPAG9'],
        ['MBTD1','ZMYND11'],
        ['MBTPS1','TCEANC2'],
        ['MBTPS2','YY2'],
        ['MCHR2','RPS6KA2'],
        ['MCL1','TXNIP'],
        ['MCM3AP-AS1','MRFAP1L1'],
        ['MCM4','PRKDC'],
        ['MCM4','SNTB1'],
        ['MCM5','TOM1'],
        ['MCM5','TRPM7'],
        ['MCM9','ROS1'],
        ['MCMBP','PDCD11'],
        ['MCMDC2','NFIB'],
        ['MCTP1','SNTB2'],
        ['MDGA1','PTK7'],
        ['MDGA2','SRPK2'],
        ['MDK','NRG1'],
        ['MDM1','NUP107'],
        ['MDM2','NUP107'],
        ['MDM2','PPM1H'],
        ['MDM2','PVT1'],
        ['MDM2','RUNX2'],
        ['MDM2','SYK'],
        ['MDM2','TMPO'],
        ['MDM2','VOPP1'],
        ['MDM2','ZFC3H1'],
        ['MDM4','NTRK1'],
        ['MDM4','NTRK2'],
        ['MDM4','NTRK3'],
        ['MDM4','STK40'],
        ['MDS1','RPL22'],
        ['MDS1','RUNX1'],
        ['ME2','MLL'],
        ['ME2','SMAD4'],
        ['MEAF6','PHF1'],
        ['MEAF6','SCMH1'],
        ['MEAF6','SUZ12'],
        ['MECOM','NRIP1'],
        ['MECOM','RPN1'],
        ['MECOM','RUNX1'],
        ['MECOM','TCRB@'],
        ['MECOM','THADA'],
        ['MECOM','TTC18'],
        ['MECP2','TEX28'],
        ['MECP2','TMLHE'],
        ['MED1','RARA'],
        ['MED1','SHANK2'],
        ['MED1','STXBP4'],
        ['MED1','USP32'],
        ['MED1','WSB1'],
        ['MED12','PRDM10'],
        ['MED13','TGFBR3'],
        ['MED13L','YEATS4'],
        ['MED15','TFE3'],
        ['MED18','UBE4B'],
        ['MED21','PIP5K1C'],
        ['MED24','PIP4K2B'],
        ['MED24','RARA'],
        ['MED6','RGS6'],
        ['MEF2B','TADA2A'],
        ['MEF2C','SS18'],
        ['MEF2D','NTRK1'],
        ['MEF2D','SS18'],
        ['MEF2D','ZDHHC5'],
        ['MEGF11','SIK2'],
        ['MEGF11','TIPIN'],
        ['MEGF8','MYO9B'],
        ['MEGF8','POU2F2'],
        ['MEGF8','ZNF526'],
        ['MEIS1','NCOA1'],
        ['MEIS1','NCOA2'],
        ['MEIS1','PBX3'],
        ['MEIS1','ROCK2'],
        ['MEIS2','TBC1D22A'],
        ['MELK','NPC1'],
        ['MELK','RNF38'],
        ['MELK','SDK1'],
        ['MELK','SIK3'],
        ['MELK','TMEM88'],
        ['MEMO1','SEC31A'],
        ['MEN2A','RET'],
        ['MEN2B','RET'],
        ['MERTK','MYO5A'],
        ['MERTK','PVT1'],
        ['MERTK','TMEM87B'],
        ['MESDC2','SENP1'],
        ['MET','OXR1'],
        ['MET','PTN'],
        ['MET','PTPRZ'],
        ['MET','PTPRZ1'],
        ['MET','RBPMS'],
        ['MET','SLC1A2'],
        ['MET','SPECC1L'],
        ['MET','SRC'],
        ['MET','ST7'],
        ['MET','STARD3NL'],
        ['MET','TFG'],
        ['MET','TNS3'],
        ['MET','TPR'],
        ['MET','TRIM24'],
        ['MET','TRIM4'],
        ['MET','ZKSCAN1'],
        ['METAP2','PIK3C2G'],
        ['METTL21B','TSFM'],
        ['METTL23','MFSD11'],
        ['METTL5','TLK1'],
        ['METTL7A','SRPK1'],
        ['MFSD1','STARD3NL'],
        ['MFSD2A','ZMPSTE24'],
        ['MFSD2B','ZFYVE9'],
        ['MFSD8','TXK'],
        ['MGA','NUTM1'],
        ['MGA1','NUT'],
        ['MGAM','SSBP1'],
        ['MGAT4C','PIP5K1B'],
        ['MGAT4C','PPP1R12A'],
        ['MGEA5','RNLS'],
        ['MGEA5','TGFBR3'],
        ['MGMT','NFYC'],
        ['MGMT','RPH3A'],
        ['MGMT','SAR1A'],
        ['MGST3','ZMAT5'],
        ['MIB1','YOD1'],
        ['MICA','RAB1B'],
        ['MICAL1','TMEM198B'],
        ['MICAL2','PPAP2A'],
        ['MICAL2','SMAD4'],
        ['MICAL2','TEAD1'],
        ['MICAL3','SPANXN4'],
        ['MICAL3','XBP1'],
        ['MICALL2','RIMKLB'],
        ['MIDN','STK11'],
        ['MIER2','RSRC2'],
        ['MIF','SLC2A11'],
        ['MINK1','NASP'],
        ['MINK1','NQO2'],
        ['MINK1','SLA'],
        ['MINK1','SPNS3'],
        ['MINK1','TG'],
        ['MINK1','USO1'],
        ['MINPP1','NUTM2A-AS1'],
        ['MIOS','RENBP'],
        ['MIPEP','SKIL'],
        ['MIPOL1','NPAS3'],
        ['MIPOL1','SKIL'],
        ['MIPOL1','TTC6'],
        ['MIPOL1','TTTY25P'],
        ['MIR142','MYC'],
        ['MIR143','NOTCH1'],
        ['MIR143','NOTCH2'],
        ['MIR143','NOTCH3'],
        ['MIR143HG','NOTCH1'],
        ['MIR143HG','NOTCH2'],
        ['MIR143HG','NOTCH3'],
        ['MIR3924','RET'],
        ['MIR548F1','NTRK1'],
        ['MIR99AHG','NRIP1'],
        ['MIRO1','RHOT1'],
        ['MITF','PRCC'],
        ['MITF','TFE3'],
        ['MIXL1','PARP1'],
        ['MKL1','NIPA1'],
        ['MKL1','RBM15'],
        ['MKL1','RUNX1'],
        ['MKL1','SRF'],
        ['MKL2','RREB1'],
        ['MKLN1','PTK2'],
        ['MKLN1','SCRN1'],
        ['MKNK2','ZNF598'],
        ['MLEC','UNC119B'],
        ['MLF1','NPM'],
        ['MLF1','NPM1'],
        ['MLF2','SLC25A42'],
        ['MLH1','MSH2'],
        ['MLH1','PMS2'],
        ['MLKL','TMEM231'],
        ['MLL','MLLT1'],
        ['MLL','MLLT10'],
        ['MLL','MLLT11'],
        ['MLL','MLLT2'],
        ['MLL','MLLT3'],
        ['MLL','MLLT4'],
        ['MLL','MLLT6'],
        ['MLL','MLLT7'],
        ['MLL','MSF'],
        ['MLL','MYO1F'],
        ['MLL','NCKIPSD'],
        ['MLL','NEBL'],
        ['MLL','NRIP3'],
        ['MLL','NUP98'],
        ['MLL','PDS5A'],
        ['MLL','PICALM'],
        ['MLL','PNUTL1'],
        ['MLL','PRRC1'],
        ['MLL','RARA'],
        ['MLL','RELA'],
        ['MLL','RFX2'],
        ['MLL','RREB1'],
        ['MLL','SARNP'],
        ['MLL','SEPT11'],
        ['MLL','SEPT2'],
        ['MLL','SEPT3'],
        ['MLL','SEPT5'],
        ['MLL','SEPT6'],
        ['MLL','SEPT9'],
        ['MLL','SH3GL1'],
        ['MLL','SMAP1'],
        ['MLL','SNAPC3'],
        ['MLL','SNX9'],
        ['MLL','SORBS2'],
        ['MLL','TET1'],
        ['MLL','TIRAP'],
        ['MLL','TNRC18'],
        ['MLL','UBE4A'],
        ['MLL','USP2'],
        ['MLL','USP2-AS1'],
        ['MLL','USP8'],
        ['MLL','VAV'],
        ['MLL','ZFYVE19'],
        ['MLL3','PRKAG2'],
        ['MLL3','TPTE'],
        ['MLL4','RB1'],
        ['MLLT10','NAP1L1'],
        ['MLLT10','PICALM'],
        ['MLLT10','PPP2R1B'],
        ['MLLT10','RBBP5'],
        ['MLLT10','XPO1'],
        ['MLLT11','PIP5K1A'],
        ['MLLT3','PAX5'],
        ['MLLT3','PLIN2'],
        ['MLLT3','PPP6R1'],
        ['MLLT6','PIP4K2B'],
        ['MLLT6','TEM7'],
        ['MLLT7','PAX3'],
        ['MLPH','PIKFYVE'],
        ['MLXIP','PTEN'],
        ['MLYCD','RUNX1'],
        ['MMAA','YTHDF3'],
        ['MMP16','RBM12B'],
        ['MMP20','TMEM123'],
        ['MMP27','TMEM123'],
        ['MMSET','WHSC1'],
        ['MN1','PATZ1'],
        ['MN1','TEL'],
        ['MNAT1','RASGEF1B'],
        ['MNDA','NOTCH2'],
        ['MNX1','MYB'],
        ['MOB3A','STK11'],
        ['MOB3C','SLA2'],
        ['MOBKL2B','PCA3'],
        ['MOGAT2','UVRAG'],
        ['MOK','MUCL1'],
        ['MOK','WARS'],
        ['MON2','SLC16A7'],
        ['MORC2','NF2'],
        ['MORN1','SLC6A9'],
        ['MOV10','STK25'],
        ['MPC2','SLCO1A2'],
        ['MPHOSPH8','ZBTB17'],
        ['MPND','RAB8A'],
        ['MPO','TRIB1'],
        ['MPP2','NRXN3'],
        ['MPP3','PYY'],
        ['MPRIP','NTRK1'],
        ['MPRIP','NTRK2'],
        ['MPRIP','NTRK3'],
        ['MPRIP','PDGFRB'],
        ['MPRIP','RAF1'],
        ['MPRIP','TP53'],
        ['MPRIP','UBB'],
        ['MRAS','NME9'],
        ['MRAS','UBA5'],
        ['MRC2','RND2'],
        ['MRE11','RAD50'],
        ['MRE11','YAP1'],
        ['MRM1','ZNF33A'],
        ['MROH1','PARP10'],
        ['MROH9','RCN2'],
        ['MRPL13','MTBP'],
        ['MRPL13','NRG1'],
        ['MRPL21','TADA2A'],
        ['MRPL24','NTRK1'],
        ['MRPL24','NTRK2'],
        ['MRPL24','NTRK3'],
        ['MRPL27','SPAG9'],
        ['MRPL39','NRIP1'],
        ['MRPL45','TRPC4AP'],
        ['MRPL48','STK3'],
        ['MRPL51','NCAPD2'],
        ['MRPS16','TTC18'],
        ['MRPS18C','SIGLEC10'],
        ['MRPS22','ZNF622'],
        ['MRPS23','SECISBP2L'],
        ['MRPS28','PC1'],
        ['MRPS31','SUGT1'],
        ['MRPS6','MYO3A'],
        ['MRTFA','RBM15'],
        ['MRTFB','RREB1'],
        ['MRX58','TSPAN7'],
        ['MS4A2','SLC17A3'],
        ['MSANTD3','XRCC4'],
        ['MSH2','MSH3'],
        ['MSH2','MSH6'],
        ['MSH2','PRKCE'],
        ['MSH2','TACSTD1'],
        ['MSH2','TTC7A'],
        ['MSH5','NFKBIL1'],
        ['MSH6','RUNX1'],
        ['MSI1','PXN'],
        ['MSI2','NEK8'],
        ['MSI2','SETBP1'],
        ['MSI2','TEX14'],
        ['MSI2','TLK2'],
        ['MSI2','TTC40'],
        ['MSL2','PMEPA1'],
        ['MSL2','STAG1'],
        ['MSL2L1','SFRS10'],
        ['MSLN','WDR90'],
        ['MSMB','NCOA4'],
        ['MSMB','PGC'],
        ['MSN','ROS1'],
        ['MSRB2','PTER'],
        ['MSRB3','SRGAP1'],
        ['MSRB3','TMBIM4'],
        ['MST1','OSBPL6'],
        ['MTA1','PIEZO2'],
        ['MTA3','SEC31A'],
        ['MTA3','THADA'],
        ['MTAP','PCDH7'],
        ['MTAP','RAF1'],
        ['MTAP','TYRO3'],
        ['MTBP','SAMD12'],
        ['MTCP1','TRA@'],
        ['MTCP1','TRB@'],
        ['MTF1','SF3A3'],
        ['MTG1','SCART1'],
        ['MTG16','MTGR1'],
        ['MTG16','RUNX1'],
        ['MTG8','RUNX1'],
        ['MTHFD1','PYGL'],
        ['MTMR12','TERT'],
        ['MTMR2','NTRK2'],
        ['MTMR2','SYVN1'],
        ['MTMR2','TMEM135'],
        ['MTMR3','NF2'],
        ['MTMR3','UQCR10'],
        ['MTMR4','TANC1'],
        ['MTOR','RASGEF1B'],
        ['MTOR','TP53BP1'],
        ['MTPN','OSBPL1A'],
        ['MTS1','MTS2'],
        ['MTSS1','TATDN1'],
        ['MTUS2','RIMS1'],
        ['MUC1','VEGFR2'],
        ['MUC16','NCL'],
        ['MUC16','NEAT1'],
        ['MUC16','OGT'],
        ['MUC2','NTRK2'],
        ['MUC4','TNK2'],
        ['MUC6','PCBD2'],
        ['MUC7','RUFY3'],
        ['MUC7','SEC31A'],
        ['MUCL1','SLC16A7'],
        ['MUSK','ZNF618'],
        ['MXD1','NUTM1'],
        ['MXD4','NUTM1'],
        ['MXI1','NUTM1'],
        ['MXI1','SMC3'],
        ['MXI1','STMN1'],
        ['MYB','MYBL1'],
        ['MYB','NFIB'],
        ['MYB','NHSL1'],
        ['MYB','PCDHGA1'],
        ['MYB','PEX7'],
        ['MYB','PLAGL1'],
        ['MYB','QKI'],
        ['MYB','SMR3B'],
        ['MYB','TRB@'],
        ['MYB','TYK2'],
        ['MYBBP1A','TP53'],
        ['MYBL1','NFIB'],
        ['MYBL1','RAD51B'],
        ['MYBL1','YTHDF3'],
        ['MYBPC1','PDZRN4'],
        ['MYC','MYCN'],
        ['MYC','PDLIM5'],
        ['MYC','PVT1'],
        ['MYC','RLF'],
        ['MYC','RNF213'],
        ['MYC','SDF4'],
        ['MYC','SOCS1'],
        ['MYC','TRA@'],
        ['MYC','TXNDC5'],
        ['MYC','ZBTB5'],
        ['MYC','ZCCHC7'],
        ['MYCBP2','RNASEH2B'],
        ['MYCL','RLF'],
        ['MYCL1','RLF'],
        ['MYCL1','SMAP2'],
        ['MYCN','MYCNUT'],
        ['MYCN','ODC1'],
        ['MYD88','OXSR1'],
        ['MYEOV','YAP1'],
        ['MYH10','NDEL1'],
        ['MYH10','RARA'],
        ['MYH10','USP22'],
        ['MYH11','SMG1'],
        ['MYH11','TRIO'],
        ['MYH11','WNK2'],
        ['MYH13','RET'],
        ['MYH14','PRKD2'],
        ['MYH7','PVT1'],
        ['MYH9','MYO18B'],
        ['MYH9','NTRK3'],
        ['MYH9','TXN2'],
        ['MYH9','USP6'],
        ['MYL12B','NLRC5'],
        ['MYL6B','VWA3A'],
        ['MYLK','NXPH1'],
        ['MYLK','REEP5'],
        ['MYLK','TM7SF2'],
        ['MYO15B','PRPF3'],
        ['MYO16','PTK2'],
        ['MYO16','TRIM13'],
        ['MYO18A','PDGFRB'],
        ['MYO18A','SSH2'],
        ['MYO18B','PI4KA'],
        ['MYO18B','UBE2R2'],
        ['MYO19','SKA2'],
        ['MYO19','TRIM37'],
        ['MYO1A','RNFT2'],
        ['MYO1A','XPO5'],
        ['MYO1D','RARA'],
        ['MYO1F','VAV1'],
        ['MYO1G','PAX5'],
        ['MYO3A','SLC5A3'],
        ['MYO3B','TP53'],
        ['MYO3B','TSHZ2'],
        ['MYO5A','NTRK3'],
        ['MYO5A','RET'],
        ['MYO5A','ROS1'],
        ['MYO5C','RET'],
        ['MYO5C','ROS1'],
        ['MYO5C','TNFAIP8L3'],
        ['MYO6','SENP6'],
        ['MYO7A','PAK1'],
        ['MYO7A','SESN3'],
        ['MYO9B','RAB22A'],
        ['MYO9B','UBE2V1'],
        ['MYOD1','PIK3CA'],
        ['MYOF','TNKS2'],
        ['MYOM1','PTPRM'],
        ['MYOM1','TGIF1'],
        ['MYRFL','NCOA2'],
        ['MYST3','NCOA2'],
        ['MYST3','P300'],
        ['MZF1','SMG6'],
        ['N4BP1','VPS35'],
        ['N4BP2','PDCD1LG2'],
        ['N6AMT2','RUNX1'],
        ['NA','RHOA'],
        ['NA','USP25'],
        ['NAA30','TERT'],
        ['NAA38','TP53'],
        ['NAA50','SLC19A2'],
        ['NAAA','PPBPP1'],
        ['NAALADL2','PIK3CA'],
        ['NAB2','STAT6'],
        ['NAB2','TAC3'],
        ['NABP1','RARA'],
        ['NABP2','PVT1'],
        ['NACA','NWD1'],
        ['NACA','PRIM1'],
        ['NACA','RPL41'],
        ['NACC1','NIPBL'],
        ['NACC2','NTRK1'],
        ['NACC2','NTRK2'],
        ['NACC2','NTRK3'],
        ['NADSYN1','SUV420H1'],
        ['NADSYN1','TMEM138'],
        ['NAIF1','SCAND2P'],
        ['NAIP','OCLN'],
        ['NAMPT','TSPY18P'],
        ['NAP1L1','STK38L'],
        ['NAP1L4','NUTM1'],
        ['NAPEPLD','TNFSF12'],
        ['NARG1','NDUFC1'],
        ['NARS2','PAK1'],
        ['NARS2','RARA'],
        ['NARS2','TENM4'],
        ['NARS2','TMPRSS2'],
        ['NAT8','PDK4'],
        ['NAV1','NTRK2'],
        ['NAV2','TCF7L1'],
        ['NAV3','NTRK1'],
        ['NAV3','PDK3'],
        ['NAV3','WDFY3'],
        ['NBAS','SMC6'],
        ['NBEA','PVT1'],
        ['NBEAL2','SETD2'],
        ['NBPF10','NOTCH2NL'],
        ['NBPF25P','SRGAP2-AS1'],
        ['NBPF4','SLC25A24'],
        ['NBPF6','SLC25A24'],
        ['NBPF8','PAX5'],
        ['NBPF9','SEC22B'],
        ['NBR1','WSB1'],
        ['NCALD','PLAG1'],
        ['NCAM2','RUVBL2'],
        ['NCAM2','SGCZ'],
        ['NCAM2','TEC'],
        ['NCAPD2','PRPF3'],
        ['NCAPD2','TNFRSF1A'],
        ['NCAPD3','VPS26B'],
        ['NCAPG2','RBPMS'],
        ['NCBP1','XPA'],
        ['NCBP2','ZNF844'],
        ['NCEH1','TSC22D2'],
        ['NCK2','RAVER1'],
        ['NCKAP5','STK39'],
        ['NCL','NR4A1'],
        ['NCL','RB1'],
        ['NCOA1','PAX3'],
        ['NCOA1','SRF'],
        ['NCOA2','PAX3'],
        ['NCOA2','RUNXBP2'],
        ['NCOA2','SERPINA7'],
        ['NCOA2','SRF'],
        ['NCOA2','TEAD1'],
        ['NCOA2','VGLL2'],
        ['NCOA2','WHSC1L1'],
        ['NCOA2','ZNF704'],
        ['NCOA3','SGK2'],
        ['NCOA3','SULF2'],
        ['NCOA4','RET'],
        ['NCOA5','PAX5'],
        ['NCOA6','PAX5'],
        ['NCOA6','PIGU'],
        ['NCOA6','PRDM14'],
        ['NCOA6','TRPC4AP'],
        ['NCOR1','PIGL'],
        ['NCOR1','PKD1L1'],
        ['NCOR1','RHOU'],
        ['NCOR1','RNF43'],
        ['NCOR2','NTRK1'],
        ['NCOR2','RNF123'],
        ['NCOR2','SCARB1'],
        ['NCOR2','SLC12A7'],
        ['NCOR2','SLC45A1'],
        ['NCOR2','UBC'],
        ['ND3','ND5'],
        ['ND4','PLAG1'],
        ['ND4L','ND6'],
        ['NDC80','SMCHD1'],
        ['NDE1','PDGFRB'],
        ['NDE1','SMG1'],
        ['NDE1','TRIO'],
        ['NDE1','WNK2'],
        ['NDEL1','NISCH'],
        ['NDEL1','PDGFRB'],
        ['NDEL1','TNS1'],
        ['NDEL1','TP53'],
        ['NDFIP1','THEM4'],
        ['NDRG1','PLAG1'],
        ['NDRG1','PVT1'],
        ['NDRG1','SLC30A4'],
        ['NDRG1','ST3GAL1'],
        ['NDRG3','TAF4'],
        ['NDST2','RUNX1'],
        ['NDUFA11','RNF126'],
        ['NDUFA7','PRKCG'],
        ['NDUFA7','TBRG1'],
        ['NDUFB4P12','SDR16C5'],
        ['NDUFB9','PGCP'],
        ['NDUFS4','PDE4D'],
        ['NEAT1','RGPD2'],
        ['NEAT1','TFE3'],
        ['NEAT1','TFEB'],
        ['NEB','ZBTB17'],
        ['NEDD4','SS18'],
        ['NEDD4L','NPAS2'],
        ['NEDD4L','PRPH2'],
        ['NEGR1','ZRANB2'],
        ['NEK1','RUNX2'],
        ['NEK11','TOPBP1'],
        ['NEK11','UBE2H'],
        ['NEK2','NVL'],
        ['NEK4','SETD2'],
        ['NEK6','PSMB7'],
        ['NEK6','PTGS1'],
        ['NEK6','RXRA'],
        ['NEK6','TNRC6B'],
        ['NEK7','RAB3GAP2'],
        ['NEK9','TMED10'],
        ['NELFA','ZNF177'],
        ['NELL1','WEE1'],
        ['NELL2','OS9'],
        ['NELL2','TMEM117'],
        ['NENF','PPP2R5A'],
        ['NEXN-AS1','TINF2'],
        ['NF1','NLE1'],
        ['NF1','NLK'],
        ['NF1','NUBPL'],
        ['NF1','PLCD3'],
        ['NF1','PSMD11'],
        ['NF1','PTPN11'],
        ['NF1','RAB11FIP4'],
        ['NF1','RHOT1'],
        ['NF1','RNF135'],
        ['NF1','SETD2'],
        ['NF1','TAOK1'],
        ['NF1','TEX14'],
        ['NF1','TP53'],
        ['NF1','TSPAN18'],
        ['NF1','VWDE'],
        ['NF1','XPR1'],
        ['NF2','NIPSNAP1'],
        ['NF2','OSBP2'],
        ['NF2','OXCT1'],
        ['NF2','PI4KA'],
        ['NF2','PIAS1'],
        ['NF2','PIEZO2'],
        ['NF2','PPIL2'],
        ['NF2','RHOT1'],
        ['NF2','SPATA13'],
        ['NF2','ZMAT5'],
        ['NF2','ZNRF3'],
        ['NF2','ZPBP2'],
        ['NFAM1','PTPRZ1'],
        ['NFASC','NTRK1'],
        ['NFASC','NTRK2'],
        ['NFASC','NTRK3'],
        ['NFASC','PRELP'],
        ['NFASC','RTN3'],
        ['NFASC','SOX13'],
        ['NFATC3','PLA2G15'],
        ['NFE2','R3HDM2'],
        ['NFE2L2','PAX8'],
        ['NFIA','PDGFRB'],
        ['NFIA','RNLS'],
        ['NFIB','NKAIN2'],
        ['NFIB','PIM2'],
        ['NFIB','PLAG1'],
        ['NFIB','PTPRD'],
        ['NFIB','XRCC4'],
        ['NFIL3','PVT1'],
        ['NFIX','PKN1'],
        ['NFIX','RAD23A'],
        ['NFKBIB','PAK4'],
        ['NFKBIE','NUP98'],
        ['NFKBIL2','RAE1'],
        ['NFS1','PREX1'],
        ['NFS1','PXDC1'],
        ['NFU1','TRIM24'],
        ['NFYA','TDRG1'],
        ['NFYC','TAL1'],
        ['NGF','NTRK1'],
        ['NGFR','SPAG9'],
        ['NICN1','RASA2'],
        ['NIM1K','SLC6A18'],
        ['NIN','PDGFRB'],
        ['NIN','SAV1'],
        ['NIN','SDR39U1'],
        ['NIPAL1','TRPC6'],
        ['NIPBL','SPEF2'],
        ['NIPBL','STK24'],
        ['NISCH','TF'],
        ['NKAIN2','SENP6'],
        ['NKAIN3','POMK'],
        ['NKIRAS1','RAF1'],
        ['NKIRAS2','TP53'],
        ['NKTR','PLAG1'],
        ['NKTR','TMEM79'],
        ['NKX2-1','TRA@'],
        ['NKX2-1','TRB@'],
        ['NKX2-1','TRD@'],
        ['NKX2-1-AS1','TRD@'],
        ['NKX2-2','TMPRSS3'],
        ['NKX2-5','TRD@'],
        ['NKX6-1','SHANK2-AS3'],
        ['NLK','PPP1R1B'],
        ['NLK','RAD51D'],
        ['NLK','TAOK1'],
        ['NLK','TMEM199'],
        ['NLK','WRB'],
        ['NLRP4','TOX'],
        ['NLRP8','PPP2R1A'],
        ['NLRP9P','SRPX2'],
        ['NME2','RPL27'],
        ['NME4','TMC5'],
        ['NME6','TAOK1'],
        ['NMU','PDGFRA'],
        ['NOC4L','SDS'],
        ['NOL1','TCF3'],
        ['NOL4','PVT1'],
        ['NOL4L','PAX5'],
        ['NOL4L','RUNX1'],
        ['NOL4L','TAF4'],
        ['NOL5A','TMC2'],
        ['NOLC1','PTEN'],
        ['NOLC1','SHROOM3'],
        ['NONO','TFE3'],
        ['NOP2','TCF3'],
        ['NOP58','STRADB'],
        ['NOR1','TAF2N'],
        ['NOR1','TFG'],
        ['NOS1AP','NTRK2'],
        ['NOS1AP','SLC7A5'],
        ['NOSIP','STK11'],
        ['NOTCH1','NUP214'],
        ['NOTCH1','PRKCA'],
        ['NOTCH1','ROS1'],
        ['NOTCH1','SEC16A'],
        ['NOTCH1','SNHG7'],
        ['NOTCH1','TCRB@'],
        ['NOTCH1','TLX1'],
        ['NOTCH1','TLX1NB'],
        ['NOTCH1','TRA@'],
        ['NOTCH1','TRB@'],
        ['NOTCH2','NRG1'],
        ['NOTCH2','PARS2'],
        ['NOTCH2','PDE4DIP'],
        ['NOTCH2','PTGFRN'],
        ['NOTCH2','SEC22B'],
        ['NOTCH2','SPAG17'],
        ['NOTCH2','TMEM150C'],
        ['NOTCH3','SETBP1'],
        ['NOTCH3','SLCO1A2'],
        ['NOTCH3','UCA1'],
        ['NOTCH3','VIM'],
        ['NPAS2','RIF1'],
        ['NPAS3','WASH1'],
        ['NPAT','RPS11'],
        ['NPEPL1','STX16'],
        ['NPEPPS','OSBPL9'],
        ['NPEPPS','RSF1'],
        ['NPEPPS','USP32'],
        ['NPHS1','SIRT2'],
        ['NPIPB5','SMG1'],
        ['NPM','RARA'],
        ['NPM1','NUMA1'],
        ['NPM1','PTMA'],
        ['NPM1','RARA'],
        ['NPM1','RARG'],
        ['NPM1','RPP30'],
        ['NPM1','SETBP1'],
        ['NPM1','TRIP12'],
        ['NPM1','TYK2'],
        ['NPSR1','POU6F2'],
        ['NPW','SLC9A3R2'],
        ['NR1D1','THRA'],
        ['NR3C1','SPRY4'],
        ['NR4A1','PRG4'],
        ['NR4A1','TMPRSS2'],
        ['NR4A3','PGR'],
        ['NR4A3','PIK3R1'],
        ['NR4A3','RBP56'],
        ['NR4A3','TAF15'],
        ['NR4A3','TAFII68'],
        ['NR4A3','TCF12'],
        ['NR4A3','TFG'],
        ['NR6A1','OBP2B'],
        ['NRBP1','PMEL'],
        ['NRBP1','RAB10'],
        ['NRG1','ODZ4'],
        ['NRG1','PARP8'],
        ['NRG1','PCM1'],
        ['NRG1','PDE7A'],
        ['NRG1','PMEPA1'],
        ['NRG1','POMK'],
        ['NRG1','RAB2'],
        ['NRG1','RAB3IL1'],
        ['NRG1','RALGAPA1'],
        ['NRG1','RBPMS'],
        ['NRG1','ROCK1'],
        ['NRG1','SDC4'],
        ['NRG1','SETD4'],
        ['NRG1','SLC3A2'],
        ['NRG1','SMAD4'],
        ['NRG1','STMN2'],
        ['NRG1','TBL1XR1'],
        ['NRG1','TENM4'],
        ['NRG1','THAP7'],
        ['NRG1','THBS1'],
        ['NRG1','TMEM66'],
        ['NRG1','TNC'],
        ['NRG1','TSHZ2'],
        ['NRG1','UNC5D'],
        ['NRG1','VAMP2'],
        ['NRG1','VTCN1'],
        ['NRG1','WHSC1L1'],
        ['NRG1','WRN'],
        ['NRG1','ZMYM2'],
        ['NRG1','ZNF704'],
        ['NRG2','ZNF208'],
        ['NRG4','PEAK1'],
        ['NRG4','TRIM68'],
        ['NRG4','TSPAN3'],
        ['NRIP1','RSPO2'],
        ['NRIP1','TIAM1'],
        ['NRIP1','UHRF1'],
        ['NRROS','SGK1'],
        ['NRXN1','PEX13'],
        ['NRXN3','TSHR'],
        ['NSD1','NUP98'],
        ['NSD1','RALBP1'],
        ['NSD1','STIM1'],
        ['NSD1','ZNF250'],
        ['NSD1','ZNF346'],
        ['NSD2','ST6GAL1'],
        ['NSD3','NUP98'],
        ['NSD3','NUT'],
        ['NSD3','NUTM1'],
        ['NSDHL','TP53'],
        ['NSFL1C','SIRPB2'],
        ['NSL1','VASH2'],
        ['NSL1','ZDBF2'],
        ['NSMAF','TOX'],
        ['NSMCE2','PVT1'],
        ['NSMCE2','SQLE'],
        ['NTF3','PLAG1'],
        ['NTN4','TRIM24'],
        ['NTNG2','NUP153'],
        ['NTRK1','P2RY8'],
        ['NTRK1','PAN3'],
        ['NTRK1','PDE4DIP'],
        ['NTRK1','PEAR1'],
        ['NTRK1','PHF20'],
        ['NTRK1','PIP5K1A'],
        ['NTRK1','PLEKHA6'],
        ['NTRK1','PPL'],
        ['NTRK1','PRDX1'],
        ['NTRK1','QKI'],
        ['NTRK1','RABGAP1L'],
        ['NTRK1','RBPMS'],
        ['NTRK1','RET'],
        ['NTRK1','RFWD2'],
        ['NTRK1','RNF213'],
        ['NTRK1','SCYL3'],
        ['NTRK1','SEL1L'],
        ['NTRK1','SQSTM1'],
        ['NTRK1','SSBP2'],
        ['NTRK1','STRN3'],
        ['NTRK1','TBC1'],
        ['NTRK1','TEL'],
        ['NTRK1','TFG'],
        ['NTRK1','TG'],
        ['NTRK1','TMP3'],
        ['NTRK1','TP53'],
        ['NTRK1','TPM3'],
        ['NTRK1','TPR'],
        ['NTRK1','TRIM24'],
        ['NTRK1','TRIM33'],
        ['NTRK1','TRIM63'],
        ['NTRK1','TRP'],
        ['NTRK1','VANGL2'],
        ['NTRK1','VCL'],
        ['NTRK2','PAN3'],
        ['NTRK2','PEAR1'],
        ['NTRK2','PLEKHA6'],
        ['NTRK2','PML'],
        ['NTRK2','PPL'],
        ['NTRK2','PRKAR2A'],
        ['NTRK2','QKI'],
        ['NTRK2','RABGAP1L'],
        ['NTRK2','RBPMS'],
        ['NTRK2','RFWD2'],
        ['NTRK2','RNF213'],
        ['NTRK2','SHC3'],
        ['NTRK2','SLMAP'],
        ['NTRK2','SQSTM1'],
        ['NTRK2','SSBP2'],
        ['NTRK2','STRN'],
        ['NTRK2','TBC1'],
        ['NTRK2','TBC1D2'],
        ['NTRK2','TEL'],
        ['NTRK2','TFG'],
        ['NTRK2','TLE4'],
        ['NTRK2','TP53'],
        ['NTRK2','TPM3'],
        ['NTRK2','TPR'],
        ['NTRK2','TRAF2'],
        ['NTRK2','TRIM24'],
        ['NTRK2','UFD1'],
        ['NTRK2','VCAN'],
        ['NTRK2','VCL'],
        ['NTRK3','PAN3'],
        ['NTRK3','PEAR1'],
        ['NTRK3','PIBF1'],
        ['NTRK3','PLEKHA6'],
        ['NTRK3','POLG'],
        ['NTRK3','PPFIBP1'],
        ['NTRK3','PPL'],
        ['NTRK3','PRPSAP1'],
        ['NTRK3','QKI'],
        ['NTRK3','RABGAP1L'],
        ['NTRK3','RALGPS2'],
        ['NTRK3','RBPMS'],
        ['NTRK3','RFWD2'],
        ['NTRK3','RNF213'],
        ['NTRK3','SCAPER'],
        ['NTRK3','SPECC1L'],
        ['NTRK3','SQSTM1'],
        ['NTRK3','SSBP2'],
        ['NTRK3','STRN'],
        ['NTRK3','STRN3'],
        ['NTRK3','TBC1'],
        ['NTRK3','TEL'],
        ['NTRK3','TFG'],
        ['NTRK3','TMP4'],
        ['NTRK3','TP53'],
        ['NTRK3','TPM3'],
        ['NTRK3','TPM4'],
        ['NTRK3','TPR'],
        ['NTRK3','TRIM24'],
        ['NTRK3','UBE2R2'],
        ['NTRK3','VCL'],
        ['NTRK3','VIM'],
        ['NTRK3','VPS18'],
        ['NTRK3','YWHAE'],
        ['NTRK3','ZBTB10'],
        ['NTRK3','ZNF7'],
        ['NTRK3','ZNF710'],
        ['NTSR2','ROCK2'],
        ['NUAK1','UHRF1BP1L'],
        ['NUAK2','RBM24'],
        ['NUCKS1','PM20D1'],
        ['NUDCD1','SYBU'],
        ['NUDT15','PTMS'],
        ['NUDT19','ZNF841'],
        ['NUDT9','SETD5'],
        ['NUMA1','PML'],
        ['NUMA1','RARA'],
        ['NUMA1','SFMBT1'],
        ['NUMA1','YAP1'],
        ['NUMB','TAOK3'],
        ['NUP107','PDZRN4'],
        ['NUP107','TSFM'],
        ['NUP153','RBM24'],
        ['NUP160','SLC43A3'],
        ['NUP188','PHYHD1'],
        ['NUP188','SPTAN1'],
        ['NUP210','PPARG'],
        ['NUP214','RAC1'],
        ['NUP214','SET'],
        ['NUP214','SQSTM1'],
        ['NUP214','XKR3'],
        ['NUP88','RABEP1'],
        ['NUP98','PDGFRA'],
        ['NUP98','PHF23'],
        ['NUP98','PMX1'],
        ['NUP98','POU1F1'],
        ['NUP98','PRRX1'],
        ['NUP98','PRRX2'],
        ['NUP98','PSIP1'],
        ['NUP98','RAP1GDS1'],
        ['NUP98','RARA'],
        ['NUP98','RARG'],
        ['NUP98','SETBP1'],
        ['NUP98','TOP1'],
        ['NUP98','TOP2B'],
        ['NUP98','VRK1'],
        ['NUP98','WHSC1L1'],
        ['NUP98','XRN1'],
        ['NUS1','SLC25A13'],
        ['NUT','ZNF532'],
        ['NUTM1','SLC12A6'],
        ['NUTM1','TIPIN'],
        ['NUTM1','WWTR1'],
        ['NUTM1','YAP1'],
        ['NUTM1','YWHAE'],
        ['NUTM1','ZNF532'],
        ['NUTM1','ZNF618'],
        ['NUTM2A','YWHAE'],
        ['NUTM2B','YWHAE'],
        ['NUTM2E','YWHAE'],
        ['NXN','VPS53'],
        ['OAZ1','P4HB'],
        ['OAZ1','PTMA'],
        ['OBFC1','SH3PXD2A'],
        ['OBFC2A','RARA'],
        ['OBSCN','RNF187'],
        ['ODF1','STK3'],
        ['ODF2','TCF7L1'],
        ['ODF4','ZFYVE20'],
        ['ODZ2','SRGAP1'],
        ['ODZ4','PACS1'],
        ['ODZ4','SAPS3'],
        ['OGFOD1','TUBD1'],
        ['OGT','SARM1'],
        ['OLFM4','RET'],
        ['OLFML2A','SEPT11'],
        ['OLIG2','TRA@'],
        ['OLR1','XIAP'],
        ['OMA1','TACSTD2'],
        ['OMD','USP6'],
        ['OPA3','SIRT2'],
        ['OPHN1','PTRF'],
        ['OPHN1','RAD51B'],
        ['OPHN1','RRAGB'],
        ['OPHN1','RUNX1'],
        ['OPRD1','UNQ2998'],
        ['OPRD1','ZMYM4'],
        ['OPRL1','SGPP2'],
        ['ORC2L','SF3B1'],
        ['OS9','PPP1R12A'],
        ['OS9','PTGES3'],
        ['OS9','SLC35E3'],
        ['OSBPL2','PLCG1'],
        ['OSBPL2','TTC9C'],
        ['OSBPL9','ZNF326'],
        ['OSGIN1','TCF25'],
        ['OSGIN2','RIPK2'],
        ['OSMR','RICTOR'],
        ['OSTBETA','RASL12'],
        ['OTUB1','SHANK2'],
        ['OVCH2','ZBTB8OS'],
        ['OXCT1','SLCO2B1'],
        ['OXLD1','PDE6G'],
        ['P11','RAPGEF3'],
        ['P2RX3','RTN4RL2'],
        ['P2RY8','PAX5'],
        ['P4HA2','TBCK'],
        ['P54NRB','TFE3'],
        ['P80','PDHB'],
        ['PACRG','QKI'],
        ['PACS1','PTEN'],
        ['PACS1','RBM14'],
        ['PACS1','SCYL1'],
        ['PACS1','SPIRE1'],
        ['PACSIN1','SPDEF'],
        ['PAFAH1B1','USP6'],
        ['PAFAH1B2','SIK3'],
        ['PAFAH1B2','STAU1'],
        ['PAK1','PDGFD'],
        ['PAK1','SPINT2'],
        ['PAK1','STAT5'],
        ['PAK2','RAF1'],
        ['PAK2','RPL19'],
        ['PAK3','STAG2'],
        ['PAK7','PANK2'],
        ['PALM','PIP5K1C'],
        ['PALM','POLRMT'],
        ['PALM','PRSSL1'],
        ['PAM','ST8SIA4'],
        ['PAN2','PAN3'],
        ['PAN3','PSMA2'],
        ['PANK1','PPP2R2D'],
        ['PANX1','TIMP3'],
        ['PAPD7','RAF1'],
        ['PAPSS2','SCGB2A2'],
        ['PAQR3','ULK4'],
        ['PARD3','PRTFDC1'],
        ['PARD6G','TAOK1'],
        ['PARP14','TFE3'],
        ['PARP14','TTBK1'],
        ['PARP4','VRK3'],
        ['PARP4','ZNF331'],
        ['PARVA','TEAD1'],
        ['PASD1','TEC'],
        ['PATL1','TECTA'],
        ['PAX3','PAX7'],
        ['PAX3','STK39'],
        ['PAX3','WWTR1'],
        ['PAX5','PDL1'],
        ['PAX5','PML'],
        ['PAX5','POM121'],
        ['PAX5','RHOXF2'],
        ['PAX5','RNF38'],
        ['PAX5','SLCO1B3'],
        ['PAX5','SOX5'],
        ['PAX5','SP2'],
        ['PAX5','TAF3'],
        ['PAX5','TEL'],
        ['PAX5','TMEM14B'],
        ['PAX5','TMPRSS9'],
        ['PAX5','ZCCHC7'],
        ['PAX5','ZNF276'],
        ['PAX5','ZNF521'],
        ['PAX7','TNKS1BP1'],
        ['PAX7','VPS13D'],
        ['PAX8','PPAR'],
        ['PAX8','PPARG'],
        ['PAX8','PPARG1'],
        ['PAX9','RND2'],
        ['PBRM1','WDR82'],
        ['PBX1','TCF2'],
        ['PBX1','TCF3'],
        ['PBX4','SLC44A2'],
        ['PBXIP1','PMVK'],
        ['PCAT1','PVT1'],
        ['PCBD2','TPM3'],
        ['PCBP2','UHRF1BP1L'],
        ['PCCB','PIK3CB'],
        ['PCCB','TBL1XR1'],
        ['PCDHGC3','QKI'],
        ['PCDHGC4','QKI'],
        ['PCGF2','TAOK1'],
        ['PCGF5','PRKCQ'],
        ['PCIF1','TENM2'],
        ['PCLO','STK3'],
        ['PCLO','TRRAP'],
        ['PCM1','PDGFRB'],
        ['PCM1','RET'],
        ['PCMTD1','PLAG1'],
        ['PCMTD1','SNTG1'],
        ['PCMTD2','SAMD10'],
        ['PCNX','RAD51B'],
        ['PCP4','RUNX1'],
        ['PCSK1','UNC119'],
        ['PCTP','RARA'],
        ['PD1','PDL1'],
        ['PDAP1','SMURF1'],
        ['PDCD10','RET'],
        ['PDCD1LG2','PDL2'],
        ['PDCD1LG2','PHACTR4'],
        ['PDCD1LG2','RNF38'],
        ['PDCD2L','UBA2'],
        ['PDCD5','SIGLEC6'],
        ['PDCD6','PMVK'],
        ['PDCD6','RUNX1'],
        ['PDCD6','SLC12A7'],
        ['PDCD6','TERT'],
        ['PDE10A','POLB'],
        ['PDE10A','PSMB1'],
        ['PDE10A','STARD13'],
        ['PDE11A','ZNF521'],
        ['PDE1C','ZIM2'],
        ['PDE4D','PLK2'],
        ['PDE4D','PPP2R2B'],
        ['PDE4DIP','PDGFRB'],
        ['PDE8A','SCAND2'],
        ['PDE8B','UIMC1'],
        ['PDE8B','WDR41'],
        ['PDE9A','VWF'],
        ['PDGFA','TTYH3'],
        ['PDGFD','YAP1'],
        ['PDGFRA','PDGFRB'],
        ['PDGFRA','RAB3IP'],
        ['PDGFRA','RARA'],
        ['PDGFRA','SCAF11'],
        ['PDGFRA','SCFD2'],
        ['PDGFRA','SF1'],
        ['PDGFRA','SPEN'],
        ['PDGFRA','STRN'],
        ['PDGFRA','TMEM165'],
        ['PDGFRB','PRKG2'],
        ['PDGFRB','RAB5EP'],
        ['PDGFRB','RABEP1'],
        ['PDGFRB','RABEP2'],
        ['PDGFRB','RABPT5'],
        ['PDGFRB','SART3'],
        ['PDGFRB','SDHA'],
        ['PDGFRB','SPARC'],
        ['PDGFRB','SPECC1'],
        ['PDGFRB','SPTBN1'],
        ['PDGFRB','SSBP2'],
        ['PDGFRB','TEL'],
        ['PDGFRB','TNIP1'],
        ['PDGFRB','TP53BP1'],
        ['PDGFRB','TPM3'],
        ['PDGFRB','TRIP11'],
        ['PDGFRB','TSC1'],
        ['PDGFRB','WDR48'],
        ['PDGFRB','ZBTB11'],
        ['PDGFRB','ZEB2'],
        ['PDGFRB','ZMYND8'],
        ['PDHX','PVT1'],
        ['PDIA4','SIAE'],
        ['PDIA5','RAB6B'],
        ['PDK1','S6K1'],
        ['PDK2','TMEM49'],
        ['PDK2','USP54'],
        ['PDLIM1','ZBBX'],
        ['PDLIM4','RAD50'],
        ['PDLIM5','TCEB1'],
        ['PDPK1','PRSS21'],
        ['PDPN','PRKCB'],
        ['PDRG1','RUNX1T1'],
        ['PDS5A','PLAC8'],
        ['PDSS2','SGK1'],
        ['PDZD7','RUNX1'],
        ['PDZK1IP1','TAL1'],
        ['PDZRN3','RAF1'],
        ['PDZRN3','RASSF1'],
        ['PDZRN4','PKM2'],
        ['PDZRN4','SCYL3'],
        ['PDZRN4','SLC35E3'],
        ['PEAK1','PSTPIP1'],
        ['PEAK1','TJP1'],
        ['PEAK1','TM9SF3'],
        ['PEAK1','VPS33B'],
        ['PEBP4','RHOBTB2'],
        ['PECAM1','TEX2'],
        ['PELI2','PGF'],
        ['PELP1','XAGE1B'],
        ['PEMT','SLC35B1'],
        ['PEPD','RHPN2'],
        ['PERLD1','PPM1D'],
        ['PERLD1','SGMS2'],
        ['PEX2','YWHAZ'],
        ['PFKFB3','RGS10'],
        ['PFKM','SLC11A2'],
        ['PFKP','ZNF423'],
        ['PGA4','TSPYL2'],
        ['PGAP3','RARA'],
        ['PGAP3','SRPK1'],
        ['PGAP3','STARD3'],
        ['PGAP3','VMP1'],
        ['PGC','SGK1'],
        ['PGC','WNK1'],
        ['PGGT1B','TERT'],
        ['PGLYRP1','TMEM45B'],
        ['PGM5','PIP5K1B'],
        ['PHACTR4','PIP5K1A'],
        ['PHACTR4','RCC1'],
        ['PHB','SPINK13'],
        ['PHB','ZNF652'],
        ['PHF1','TFE3'],
        ['PHF11','RNASEH2B'],
        ['PHF15','SEC31A'],
        ['PHF15','UBE2B'],
        ['PHF17','ZFHX3'],
        ['PHF20L1','PVT1'],
        ['PHF20L1','SAMD12'],
        ['PHF21A','SLC1A2'],
        ['PHF21B','PI4KA'],
        ['PHF3','PTP4A1'],
        ['PHF3','UBR2'],
        ['PHKG1','PSPH'],
        ['PHLDB1','TP53'],
        ['PHLPP1','ZNF804A'],
        ['PHOSPHO1','SPAG9'],
        ['PHTF2','PPP3CB'],
        ['PI3','RBM12'],
        ['PI3K','PTEN'],
        ['PI3K','STAT'],
        ['PI4K2A','ZDHHC16'],
        ['PI4KB','SELENBP1'],
        ['PIBF1','POR'],
        ['PICALM','RCC1'],
        ['PICALM','RET'],
        ['PICALM','SYTL2'],
        ['PICALM','TEX10'],
        ['PIEZO1','RSPO2'],
        ['PIGG','SPINK2'],
        ['PIK3C2A','TEAD1'],
        ['PIK3C2B','PLEKHA6'],
        ['PIK3C3','RPRD1A'],
        ['PIK3C3','TRPC6'],
        ['PIK3C3','TWSG1'],
        ['PIK3CA','SHROOM3'],
        ['PIK3CA','TBL1XR1'],
        ['PIK3CA','USP13'],
        ['PIK3CB','PLS1'],
        ['PIK3CB','TSPO'],
        ['PIK3CB','VTI1A'],
        ['PIK3CD','RERE'],
        ['PIK3CD','UBR4'],
        ['PIK3R1','PTEN'],
        ['PIK3R2','USP8'],
        ['PIKFYVE','TDRD3'],
        ['PIKFYVE','ZNF124'],
        ['PIM3','SCO2'],
        ['PIM3','SELO'],
        ['PIM3','SIK1'],
        ['PIM3','SKI'],
        ['PIM3','SRSF7'],
        ['PIM3','TRABD'],
        ['PIM3','ZADH2'],
        ['PION','STYXL1'],
        ['PIP4K2A','RIMKLB'],
        ['PIP4K2A','XRN1'],
        ['PIP4K2B','RAD51C'],
        ['PIP4K2C','R3HDM2'],
        ['PIP4K2C','SRI'],
        ['PIP5K1B','PTAR1'],
        ['PIP5K1B','STK24'],
        ['PIP5K1B','ZNF37A'],
        ['PIP5K1C','PPM1H'],
        ['PIP5K2B','RAD51C'],
        ['PIPOX','PPP1R1B'],
        ['PIR','SIK2'],
        ['PIRT','ZNF566'],
        ['PITPNA','RAD51C'],
        ['PITPNC1','PRKCA'],
        ['PITPNC1','RARA'],
        ['PITPNC1','TLK2'],
        ['PIWIL2','SLC39A14'],
        ['PKD1','TSC2'],
        ['PKD1','UBE2J2'],
        ['PKD1L1','RIF1'],
        ['PKD1L1','TNS3'],
        ['PKHD1L1','PTK2'],
        ['PKHD1L1','SBF1'],
        ['PKIA','RARA'],
        ['PKIG','STK4'],
        ['PKMYT1','ZNF707'],
        ['PKN1','TECR'],
        ['PKN2','RAB3B'],
        ['PKN2','TUFT1'],
        ['PKN3','TRIM65'],
        ['PKP2','TNFAIP2'],
        ['PLA2R1','RBMS1'],
        ['PLAG1','PPP2R2A'],
        ['PLAG1','RAB2A'],
        ['PLAG1','RAD51B'],
        ['PLAG1','RAD51L1'],
        ['PLAG1','RUNX1'],
        ['PLAG1','SRSF3'],
        ['PLAG1','TCEA1'],
        ['PLAG1','TGFBR3'],
        ['PLAG1','TRPS1'],
        ['PLAG1','WFS1'],
        ['PLAG1','YWHAZ'],
        ['PLAG1','ZEB2'],
        ['PLCB1','ROR1'],
        ['PLCB3','VEGFB'],
        ['PLCD4','STK16'],
        ['PLCG1','TOP1'],
        ['PLDN','SQRDL'],
        ['PLEKHD1','SLC39A9'],
        ['PLEKHF2','ZEB1'],
        ['PLEKHG3','TCF4'],
        ['PLEKHG5','TNFRSF8'],
        ['PLEKHH2','PRKCE'],
        ['PLEKHM2','TMEM51'],
        ['PLEKHO1','VPS45'],
        ['PLGRKT','RAB3GAP2'],
        ['PLK1','RNF40'],
        ['PLK4','TBC1D9'],
        ['PLK4','UPF3B'],
        ['PLSCR1','SLC25A36'],
        ['PLSCR3','TP53'],
        ['PLXDC1','RARA'],
        ['PLXNA2','RUNX1'],
        ['PLXNB1','PRKAR2A'],
        ['PLXND1','SEMA3A'],
        ['PLXND1','TMCC1'],
        ['PLZF','RARA'],
        ['PMEL','RIPK4'],
        ['PMF1','SMG5'],
        ['PML','PP1A'],
        ['PML','RARA'],
        ['PML','SYK'],
        ['PML','TIF1'],
        ['PML','ZNF106'],
        ['PML','ZNF145'],
        ['PMP22','RAI1'],
        ['PMP22','TP53'],
        ['PMS2','POLR2J3'],
        ['PMVK','WARS2-IT1'],
        ['PNPLA7','WDR34'],
        ['PNPLA7','WDR85'],
        ['POC5','PRKD1'],
        ['POLB','UNC5D'],
        ['POLB','UVRAG'],
        ['POLD1','PRR12'],
        ['POLDIP2','RARA'],
        ['POLN','SH3BP2'],
        ['POLR1A','REEP1'],
        ['POLR1D','SPECC1'],
        ['POLR2A','WRAP53'],
        ['POLR2K','STK3'],
        ['POLR2K','VPS13B'],
        ['POLR3GL','PROM2'],
        ['POM121','POR'],
        ['POMGNT1','TSPAN1'],
        ['POMK','ST6GAL1'],
        ['POMT2','SCN2B'],
        ['PORCN','WNT10B'],
        ['POU2F2','VRK3'],
        ['POU5F1','SS18'],
        ['POU5F1B','PVT1'],
        ['PPA2','TBCK'],
        ['PPAP2C','RTCA'],
        ['PPAPDC1A','PVT1'],
        ['PPAPDC1A','TACC2'],
        ['PPARA','SLC16A7'],
        ['PPARD','TP53'],
        ['PPARG','SYN2'],
        ['PPARG','VGLL4'],
        ['PPARGC1B','SMF'],
        ['PPFIA1','RARA'],
        ['PPFIA1','TSKU'],
        ['PPFIBP1','ROS1'],
        ['PPFIBP1','SMCO2'],
        ['PPFIBP1','STK38L'],
        ['PPFIBP2','RET'],
        ['PPIG','ZNF430'],
        ['PPIL4','SEC63'],
        ['PPM1B','SLC3A1'],
        ['PPM1F','SPECC1L'],
        ['PPM1H','TBC1D30'],
        ['PPOX','UHMK1'],
        ['PPP1CA','RPS6KB2'],
        ['PPP1CB','YPEL5'],
        ['PPP1R10','TFEB'],
        ['PPP1R12A','SEPT10'],
        ['PPP1R12B','SNX27'],
        ['PPP1R13L','ZNF541'],
        ['PPP1R1B','PSMD3'],
        ['PPP1R1B','STARD3'],
        ['PPP1R21','STON1'],
        ['PPP1R37','THRA'],
        ['PPP1R9A','WBSCR17'],
        ['PPP2CA','ROCK2'],
        ['PPP2R2B','YTHDC2'],
        ['PPP2R2C','SEC31A'],
        ['PPP2R5E','PTK2'],
        ['PPP3CA','SEPT11'],
        ['PPP3R1','TTC27'],
        ['PPP4R1-AS1','RAB31'],
        ['PPP5C','TP53'],
        ['PPP5D1','ZNF83'],
        ['PPP6R2','SIM2'],
        ['PPP6R2','SPECC1L'],
        ['PPP6R3','USP6'],
        ['PPP6R3','YAP1'],
        ['PPT1','SPP1'],
        ['PRAME','RPTOR'],
        ['PRAMEF15','PUS1'],
        ['PRAMEF2','PTEN'],
        ['PRAP1','ZNF511'],
        ['PRB3','ZNF217'],
        ['PRCC','TFE3'],
        ['PRCC','TMED5'],
        ['PRDM1','TMEM41B'],
        ['PRDM15','TMPRSS2'],
        ['PRDM16','PRKCZ'],
        ['PRDM16','PSMD2'],
        ['PRDM16','RPN1'],
        ['PRDM16','RUNX1'],
        ['PRDM16','SKI'],
        ['PRDM6','SNX2'],
        ['PRDM7','RUNX1'],
        ['PRDX1','TESK2'],
        ['PRDX4','RUNX1'],
        ['PRDX4','ZNF668'],
        ['PREP','TGFBR2'],
        ['PREX1','SERINC2'],
        ['PREX1','SLC9A8'],
        ['PREX1','ZNF292'],
        ['PRG2','SLC43A3'],
        ['PRICKLE2','SULF2'],
        ['PRICKLE2-AS1','SULF2'],
        ['PRISM','SRM'],
        ['PRKAA1','TBC1D5'],
        ['PRKAA1','TTC33'],
        ['PRKAA1','UBLCP1'],
        ['PRKAA2','SLC35B1'],
        ['PRKAA2','USP24'],
        ['PRKAB1','TMEM233'],
        ['PRKAG2','RHEB'],
        ['PRKAG2','ST7'],
        ['PRKAR1','RET'],
        ['PRKAR1A','RARA'],
        ['PRKAR1A','RET'],
        ['PRKAR2A','RAF1'],
        ['PRKAR2A','RHOA'],
        ['PRKAR2A','SLC26A6'],
        ['PRKCA','SLC44A1'],
        ['PRKCA','TANC2'],
        ['PRKCB','SPNS1'],
        ['PRKCB','TNRC6A'],
        ['PRKCE','TRIM13'],
        ['PRKCH','VWA2'],
        ['PRKCI','PTBP2'],
        ['PRKCZ','SKI'],
        ['PRKD3','TMEM51'],
        ['PRKD3','UBB'],
        ['PRKDC','PXDNL'],
        ['PRKDC','SPIDR'],
        ['PRKDC','ZNF114'],
        ['PRKG1','RET'],
        ['PRKG1','SUPV3L1'],
        ['PRKG1','TRIM22'],
        ['PRKG1','TRIM5'],
        ['PRKG1','VGLL4'],
        ['PRLR','TERT'],
        ['PRMT2','RCAN1'],
        ['PRMT7','SLC7A6'],
        ['PRMT8','RRM2'],
        ['PRMT8','WNK1'],
        ['PROM1','TAPT1'],
        ['PROX1','SMOC2'],
        ['PRPF3','SRGAP2'],
        ['PRPF38A','RUNX1'],
        ['PRPF6','TENM3'],
        ['PRPSAP1','PSCD1'],
        ['PRPSAP1','QRICH2'],
        ['PRR11','RARA'],
        ['PRR11','SMG8'],
        ['PRR11','WDR68'],
        ['PRR12','ZMIZ1'],
        ['PRR15L','RSPO2'],
        ['PRR5','SYT13'],
        ['PRR5-ARHGAP8','SRRD'],
        ['PRRC1','RUNX1'],
        ['PRRC2A','SETX'],
        ['PRSS3','UBE2R2'],
        ['PRSS50','TTTY5'],
        ['PSAP','SFTPC'],
        ['PSD3','WHSC1L1'],
        ['PSEN1','STK4'],
        ['PSF','TFE3'],
        ['PSKH1','RANBP10'],
        ['PSKH1','TSNAXIP1'],
        ['PSMA6','SHMT1'],
        ['PSMB5','RBM23'],
        ['PSMB8','TAP1'],
        ['PSMD8','SIPA1L3'],
        ['PSPC1','ZMYM5'],
        ['PSPH','SEPT7P2'],
        ['PTBP1','USP6'],
        ['PTBP2','UBIAD1'],
        ['PTBP2','VRK2'],
        ['PTBP3','SUSD1'],
        ['PTC','RET'],
        ['PTC3','RET'],
        ['PTC7','RET'],
        ['PTCH1','RET'],
        ['PTDSS1','SPRYD3'],
        ['PTEN','SEC31A'],
        ['PTEN','SEC8L1'],
        ['PTEN','SGMS1'],
        ['PTEN','TAS2R1'],
        ['PTGES3','PTPRB'],
        ['PTGES3','RARG'],
        ['PTHLH','RUNX1'],
        ['PTK2','PVT1'],
        ['PTK2','RNF139'],
        ['PTK2','SP3'],
        ['PTK2','STK3'],
        ['PTK2','THOC2'],
        ['PTK2B','STAG2'],
        ['PTK2B','TMEM2'],
        ['PTK6','ZNF142'],
        ['PTOV1','VRK3'],
        ['PTP4A1','PVT1'],
        ['PTPLB','RSRC1'],
        ['PTPN13','TMEM63A'],
        ['PTPN13','TUBA1A'],
        ['PTPN2','UBLCP1'],
        ['PTPN22','VAV3'],
        ['PTPN9','SIN3A'],
        ['PTPRD','RPS6KC1'],
        ['PTPRD','SPTAN1'],
        ['PTPRF','ST3GAL3'],
        ['PTPRG','TEX14'],
        ['PTPRK','RSPO3'],
        ['PTPRK','THEMIS'],
        ['PTPRO','RUNX1'],
        ['PTPRR','SRGAP1'],
        ['PTPRR','TBX5'],
        ['PTPRT','RALY'],
        ['PTRF','STAT3'],
        ['PU.1','STAT5'],
        ['PUF60','SCRIB'],
        ['PUF60','TYW1'],
        ['PUM1','SAPS3'],
        ['PUM1','TRAF3'],
        ['PUM1','TRERF1'],
        ['PUS7','SRPK2'],
        ['PVRL2','SNTG1'],
        ['PVRL2','TRA@'],
        ['PVT1','QRSL1'],
        ['PVT1','RBBP8NL'],
        ['PVT1','RFTN1'],
        ['PVT1','RGS20'],
        ['PVT1','RGS22'],
        ['PVT1','RIMS2'],
        ['PVT1','RNF139'],
        ['PVT1','RNGTT'],
        ['PVT1','RSPO2'],
        ['PVT1','SLC1A2'],
        ['PVT1','SLC22A3'],
        ['PVT1','SLC30A4'],
        ['PVT1','SLC30A8'],
        ['PVT1','SLC7A7'],
        ['PVT1','SMARCAD1'],
        ['PVT1','SNRPB2'],
        ['PVT1','SNX31'],
        ['PVT1','SPRY2'],
        ['PVT1','TATDN1'],
        ['PVT1','TLE4'],
        ['PVT1','TMEM67'],
        ['PVT1','TMEM87B'],
        ['PVT1','TNFRSF19'],
        ['PVT1','TRD@'],
        ['PVT1','TRPS1'],
        ['PVT1','UBB'],
        ['PVT1','UGCG'],
        ['PVT1','WWOX'],
        ['PVT1','ZC3H3'],
        ['PVT1','ZCCHC7'],
        ['PVT1','ZNF521'],
        ['PWP1','TNFRSF11A'],
        ['PWWP2A','ROS1'],
        ['PXDN','TMPRSS2'],
        ['PXDNL','RNF121'],
        ['PXDNL','UHRF1'],
        ['PXK','RPP14'],
        ['PXT1','SRPK1'],
        ['PYGO1','YAP1'],
        ['PYY','UBE2M'],
        ['QKI','RAF1'],
        ['QRICH1','RHOA'],
        ['R3HDM2','RDH16'],
        ['R3HDM2','SORBS2'],
        ['R3HDM2','TSFM'],
        ['R3HDM2','XRCC6BP1'],
        ['RAB1A','XPO1'],
        ['RAB20','VWC2'],
        ['RAB21','TBC1D15'],
        ['RAB22A','TTC3'],
        ['RAB2A','SNX9'],
        ['RAB31','VAPA'],
        ['RAB3IP','SRGAP1'],
        ['RAB44','TP53'],
        ['RAB6A','RARA'],
        ['RAB6A','ZFP91'],
        ['RAB6IP2','RET'],
        ['RABGAP1L','RBKS'],
        ['RABGAP1L','ZBTB37'],
        ['RABL3','RYK'],
        ['RABL6','RTN3'],
        ['RACGAP1','WIBG'],
        ['RACK1','TRIM52'],
        ['RAD21','SNTB1'],
        ['RAD21','UTP23'],
        ['RAD23A','ZNF562'],
        ['RAD51B','SEMA6D'],
        ['RAD51D','SLCO6A1'],
        ['RAD52','WNK1'],
        ['RAD54B','SSBP4'],
        ['RAD54L2','VPRBP'],
        ['RAD9A','SHANK2'],
        ['RAE1','STX16'],
        ['RAE1','STX16-NPEPL1'],
        ['RAE1','ZMYND8'],
        ['RAF1','RBM35A'],
        ['RAF1','RPL32'],
        ['RAF1','RRBP1'],
        ['RAF1','SASS6'],
        ['RAF1','SH3BP5'],
        ['RAF1','SLC43A1'],
        ['RAF1','SOX5'],
        ['RAF1','SPECC1L'],
        ['RAF1','SRGAP3'],
        ['RAF1','TMEM40'],
        ['RAF1','TRAK1'],
        ['RAF1','TRIM33'],
        ['RAF1','XPC'],
        ['RAG1','RAG2'],
        ['RAI1','SPECC1'],
        ['RALA','ZNF610'],
        ['RALY','SYNGAP1'],
        ['RANBP17','RUNX1'],
        ['RANBP17','TLX1'],
        ['RANBP17','TLX1NB'],
        ['RANBP17','TRD@'],
        ['RAP1B','SCYL2'],
        ['RAP1B','TMBIM4'],
        ['RAPGEF3','SLC48A1'],
        ['RARA','RXRA'],
        ['RARA','SAE1'],
        ['RARA','STAT3'],
        ['RARA','STAT5B'],
        ['RARA','TBL1XR1'],
        ['RARA','TBLR1'],
        ['RARA','TFG'],
        ['RARA','TIF1'],
        ['RARA','TNRC18'],
        ['RARA','TRIM24'],
        ['RARA','USP22'],
        ['RARA','USP32'],
        ['RARA','ZBTB16'],
        ['RARA','ZNF145'],
        ['RARA','ZNF595'],
        ['RARA','ZNF732'],
        ['RARB','RB1'],
        ['RARB','TBL1XR1'],
        ['RARB','UBE2E1'],
        ['RARG','WIPI1'],
        ['RASA2','SPATA21'],
        ['RASAL2','RET'],
        ['RASGRF1','TMEM154'],
        ['RASGRF1','TMEM87A'],
        ['RASGRF2','RNF145'],
        ['RASGRP3','RPS6KA1'],
        ['RASSF2','SRXN1'],
        ['RASSF3','TBK1'],
        ['RASSF3','TTYH2'],
        ['RASSF4','RET'],
        ['RASSF4','ZNF22'],
        ['RASSF7','SCUBE2'],
        ['RASSF8','SSPN'],
        ['RASSF8','WNK1'],
        ['RAX','SMAD4'],
        ['RB1','RFX3'],
        ['RB1','RNASEH2B'],
        ['RB1','SIAH3'],
        ['RB1','SLC15A2'],
        ['RB1','STIM1'],
        ['RB1','TSC22D1'],
        ['RB1','UNC79'],
        ['RBBP4','STK40'],
        ['RBBP8','TP53'],
        ['RBFOX1','SIK2'],
        ['RBFOX1','VASN'],
        ['RBM10','TFE3'],
        ['RBM10','YAP1'],
        ['RBM23','TMEM68'],
        ['RBM28','VWA5B2'],
        ['RBM39','SMARCB1'],
        ['RBM42','TCEAL4'],
        ['RBM47','ROCK1'],
        ['RBM4B','ZCCHC11'],
        ['RBM6','SRGAP1'],
        ['RBMS1','SEMA4C'],
        ['RBMS2','SPRYD4'],
        ['RBMX','TFE3'],
        ['RBP56','TAF15'],
        ['RC3H2','RGS3'],
        ['RCAN1','RUNX1'],
        ['RCC','TFE3'],
        ['RCC1','UBE2D2'],
        ['RCC1','WDR74'],
        ['RCC17','TFE3'],
        ['RCHY1','SFXN3'],
        ['RCHY1','YWHAG'],
        ['RDH11','VTI1B'],
        ['RDM1','TANC2'],
        ['REC114','TP53'],
        ['REEP5','SLC35A2'],
        ['REEP6','STK11'],
        ['REG4','RPS10'],
        ['REG4','RPS10-NUDT3'],
        ['REG4','TFF2'],
        ['REG4','YAP1'],
        ['RELA','SRF'],
        ['RELA','YAP1'],
        ['RELA','ZMYND8'],
        ['RELB','WDR88'],
        ['REN','ZC3H11A'],
        ['RER1','ZC3H7B'],
        ['RERE','SLC2A5'],
        ['RERE','SLC9A1'],
        ['RET','RFG7'],
        ['RET','RFG8'],
        ['RET','ROS1'],
        ['RET','RRBP1'],
        ['RET','RUFY1'],
        ['RET','RUFY2'],
        ['RET','SATB1'],
        ['RET','SLC25A36'],
        ['RET','SLC39A8'],
        ['RET','SPECC1L'],
        ['RET','SPINT1'],
        ['RET','SQSTM1'],
        ['RET','TBC1D32'],
        ['RET','TBL1XR1'],
        ['RET','TFG'],
        ['RET','TIMP3'],
        ['RET','TNIP2'],
        ['RET','TPR'],
        ['RET','TRIM24'],
        ['RET','TRIM27'],
        ['RET','TRIM33'],
        ['RET','UEVLD'],
        ['RET','VCL'],
        ['RET','VIM'],
        ['RET','ZBTB41'],
        ['RFT1','UQCRC2'],
        ['RFTN1','SCD5'],
        ['RFWD3','STK31'],
        ['RFX1','ZSWIM4'],
        ['RFX5','SELENBP1'],
        ['RFX5','ZC3H12B'],
        ['RGNEF','ZFP62'],
        ['RGS10','ZMYM1'],
        ['RGS17','TBL1XR1'],
        ['RGS22','SYCP1'],
        ['RGS3','SLC31A2'],
        ['RGS7','SMYD3'],
        ['RHBDF2','ST6GALNAC2'],
        ['RHOA','TAL1'],
        ['RHOA','TKT'],
        ['RHOA','UBB'],
        ['RHOA','WWTR1'],
        ['RHOC','RPS6KA1'],
        ['RHOT1','TNKS'],
        ['RIC3','TCRBC2'],
        ['RIC3','TMEM38B'],
        ['RIC3','TRB@'],
        ['RIC3','TRBC2'],
        ['RILPL1','SBNO1'],
        ['RIMS2','STK3'],
        ['RIN2','RPL36'],
        ['RIOK1','USP45'],
        ['RIOK3','TAF4B'],
        ['RIOK3','TMEM241'],
        ['RIPK1','SERPINB9'],
        ['RIPK4','TMPRSS2'],
        ['RIT2','TMX3'],
        ['RLF','SMAP2'],
        ['RLF','UBE2J2'],
        ['RLN1','RLN2'],
        ['RMI1','STX17'],
        ['RMND5B','SFXN1'],
        ['RMST','TERT'],
        ['RNASEH2B','RREB1'],
        ['RNASEH2B','TRIM13'],
        ['RNF103','UCK2'],
        ['RNF11','TAOK3'],
        ['RNF111','TCF12'],
        ['RNF111','TP53'],
        ['RNF121','SFRS2IP'],
        ['RNF125','RNF138'],
        ['RNF130','YAP1'],
        ['RNF139','TRC8'],
        ['RNF165','SSR1'],
        ['RNF170','YES1'],
        ['RNF180','ZNF384'],
        ['RNF19A','STK3'],
        ['RNF19B','YARS'],
        ['RNF213','RPS6KB1'],
        ['RNF213','SLC26A11'],
        ['RNF216','TSC2'],
        ['RNF38','ROCK1'],
        ['RNF43','SUPT4H1'],
        ['RNFT1','TRIM37'],
        ['RNGTT','ZC2HC1B'],
        ['ROBO1','SETD2'],
        ['ROBO1','XPC'],
        ['ROCK1','TBL1XR1'],
        ['ROD1','SUSD1'],
        ['ROR1','ZNF362'],
        ['ROR2','TP53'],
        ['RORA','SMARCB1'],
        ['ROS1','RWDD1'],
        ['ROS1','SDC4'],
        ['ROS1','SLC12A2'],
        ['ROS1','SLC16A10'],
        ['ROS1','SLC22A16'],
        ['ROS1','SLC34A2'],
        ['ROS1','SLC4A4'],
        ['ROS1','TBL1XR1'],
        ['ROS1','TFG'],
        ['ROS1','TIMP3'],
        ['ROS1','TJP1'],
        ['ROS1','TMEM106B'],
        ['ROS1','TPM3'],
        ['ROS1','TPR'],
        ['ROS1','WNK1'],
        ['ROS1','YWHAE'],
        ['ROS1','ZCCHC8'],
        ['RPAP3','TRPM2'],
        ['RPGRIP1','SLC7A8'],
        ['RPL14','SLC7A11'],
        ['RPL14','SRP14'],
        ['RPL21','RPS9'],
        ['RPL22','RUNX1'],
        ['RPL28','TMEM190'],
        ['RPL3','SEPT11'],
        ['RPL36','TTYH1'],
        ['RPL38','TTYH2'],
        ['RPL39L','UBXN7'],
        ['RPL5','SFXN3'],
        ['RPN2','STK17A'],
        ['RPN2','UQCC'],
        ['RPRD2','SHC1'],
        ['RPS25','TRAPPC4'],
        ['RPS2P32','THADA'],
        ['RPS6KA3','VPREB1'],
        ['RPS6KB1','SNF8'],
        ['RPS6KB1','TMEM49'],
        ['RPS6KB1','TUBD1'],
        ['RPS6KB1','VMP1'],
        ['RPS6KB2','ST3GAL2'],
        ['RPS6KC1','SDCCAG8'],
        ['RPS6KC1','TFPI'],
        ['RPSAP52','TGFBR3'],
        ['RPTOR','TULP4'],
        ['RRBP1','SIRPA'],
        ['RREB1','TFE3'],
        ['RRM2B','STK3'],
        ['RRP15','USH2A'],
        ['RSBN1','TRBVA'],
        ['RSF1','TENM4'],
        ['RSF1','TPCN2'],
        ['RSPRY1','ZFHX3'],
        ['RTBDN','TP53'],
        ['RTF1','TYRO3'],
        ['RTN1','SSH2'],
        ['RTN3','TRPC6'],
        ['RTP4','ST6GAL1'],
        ['RUFY1','TMEM178B'],
        ['RUNDC2A','SNX29'],
        ['RUNX1','RUNX1T1'],
        ['RUNX1','SH3D19'],
        ['RUNX1','SIN3A'],
        ['RUNX1','SUMO2P13'],
        ['RUNX1','SV2B'],
        ['RUNX1','TACC1'],
        ['RUNX1','TCF12'],
        ['RUNX1','THOC6'],
        ['RUNX1','TMEM50B'],
        ['RUNX1','TRPS1'],
        ['RUNX1','TSPEAR'],
        ['RUNX1','UBL7-AS1'],
        ['RUNX1','UGGT2'],
        ['RUNX1','USP16'],
        ['RUNX1','USP42'],
        ['RUNX1','YTHDF2'],
        ['RUNX1','ZADH2'],
        ['RUNX1','ZFPM2'],
        ['RUNX1','ZNF687'],
        ['RUNX2','USP6'],
        ['RWDD3','TMEM56'],
        ['RXFP2','ZNF828'],
        ['RYR1','SMARCA4'],
        ['RYR2','SHANK3'],
        ['RYR2','USH2A'],
        ['S100A7','VAV1'],
        ['S100A8','S100A9'],
        ['S100B','WBP2NL'],
        ['SACM1L','TP53'],
        ['SACS','SGCG'],
        ['SAE1','TGFB1'],
        ['SALL2','TRA@'],
        ['SAMD12','SNTB1'],
        ['SAMD14','SGCA'],
        ['SAMD5','SASH1'],
        ['SAR1A','USP6'],
        ['SBF1','TSHR'],
        ['SBF2','TMEM99'],
        ['SBK1','TXLNA'],
        ['SBNO2','STK11'],
        ['SCAF1','TRAPPC6A'],
        ['SCAI','ZBTB34'],
        ['SCAMP2','WDR72'],
        ['SCAPER','TM6SF1'],
        ['SCARB1','UBC'],
        ['SCARB2','SERPINE1'],
        ['SCARB2','STAU2'],
        ['SCARB2','TNFAIP8'],
        ['SCD5','TMEM150C'],
        ['SCFD2','YES1'],
        ['SCGB3A1','ZGLP1'],
        ['SCHLAP1','UBE2E3'],
        ['SCL','TRD@'],
        ['SCML2','SH3KBP1'],
        ['SCN2A','STK39'],
        ['SCN4B','SCNN1G'],
        ['SCNN1A','TNFRSF1A'],
        ['SCO1','TTC7A'],
        ['SCUBE2','ZNF195'],
        ['SCYL1','SPATA2L'],
        ['SCYL3','SYN3'],
        ['SCYL3','TIMP3'],
        ['SDC1','SMYD1'],
        ['SDC2','SRRT'],
        ['SDC4','SPCS1'],
        ['SDC4P','SEC14L4'],
        ['SDR16C5','ZNF384'],
        ['SEC31A','SMC1A'],
        ['SEC31A','USP6'],
        ['SECISBP2L','USO1'],
        ['SELT','TRIM36'],
        ['SEMA3C','VRK3'],
        ['SEMA5A','SLC6A19'],
        ['SEMA5A','TGIF2'],
        ['SEMA5B','SLC12A8'],
        ['SEMA6A','TSLP'],
        ['SEMA6B','SLC27A1'],
        ['SENP6','ZNF544'],
        ['SEPP1','SERPINA3'],
        ['SEPT11','SHROOM3'],
        ['SEPT11','SYT11'],
        ['SEPT11','UBE3C'],
        ['SEPT9','ZMYND8'],
        ['SERF1A','SMN1'],
        ['SERHL2','THSD4'],
        ['SERP1','ZBTB41'],
        ['SERPINA1','TF'],
        ['SERPINB9','TBCEL'],
        ['SERPINE3','ZNF22'],
        ['SERPING1','TMX2'],
        ['SETD1B','TFE3'],
        ['SETD2','SPATA12'],
        ['SF1','SF3B1'],
        ['SF3B4','SMYD5'],
        ['SFPQ','TFE3'],
        ['SFPQ','ZFP36L1'],
        ['SFPQ','ZFP36L2'],
        ['SFSWAP','TCP11L2'],
        ['SFSWAP','ULK1'],
        ['SFTPB','ST3GAL5'],
        ['SGCD','SNRPG'],
        ['SGK1','UTRN'],
        ['SGMS1','STK10'],
        ['SGMS1','ZEB1'],
        ['SGPP2','ULK4'],
        ['SGSM2','SMG6'],
        ['SGTA','SLC39A3'],
        ['SH2B1','ZDHHC8'],
        ['SH2B3','WNK1'],
        ['SH3BP5','VGLL4'],
        ['SH3KBP1','TMTC4'],
        ['SH3PXD2B','STK10'],
        ['SHANK2','YAP1'],
        ['SHBG','TP53'],
        ['SHC2','TMEM259'],
        ['SHC2','TYK2'],
        ['SHC3','TBC1D32'],
        ['SHFM1','ZNF569'],
        ['SIAE','TPT1'],
        ['SIDT2','TAGLN'],
        ['SIGLEC17P','SORCS1'],
        ['SIGLEC9','TLK2'],
        ['SIK1','TMPRSS2'],
        ['SIK1','TRAPPC10'],
        ['SIK3','UBR1'],
        ['SIL','TAL1'],
        ['SIP1','ZEB2'],
        ['SIPA1L3','ZNF585A'],
        ['SIRPG','WWOX'],
        ['SIRT2','ZNF45'],
        ['SKAP1','THRA'],
        ['SKI','SSU72'],
        ['SKIL','SLC45A3'],
        ['SKIL','TMPRSS2'],
        ['SKP2','SLC1A3'],
        ['SLAMF1','UVRAG'],
        ['SLC10A4','ZAR1'],
        ['SLC12A2','ZNF322A'],
        ['SLC12A7','TERT'],
        ['SLC16A14','TRIP12'],
        ['SLC1A3','TMPRSS2'],
        ['SLC22A23','TNFRSF21'],
        ['SLC22A9','TMED8'],
        ['SLC25A32','TRIB1'],
        ['SLC25A32','UBR5'],
        ['SLC25A41','THOP1'],
        ['SLC25A44','ZNF395'],
        ['SLC26A10','TSFM'],
        ['SLC26A4','SLC39A9'],
        ['SLC2A5','ZNF292'],
        ['SLC30A6','SPAST'],
        ['SLC30A8','WDR67'],
        ['SLC35A1','TSHZ2'],
        ['SLC35E3','YEATS4'],
        ['SLC35F1','STK24'],
        ['SLC38A2','USP6'],
        ['SLC45A3','TMPRSS2'],
        ['SLC45A3','UBE3A'],
        ['SLC45A3','ZBTB7B'],
        ['SLC4A1AP','SUPT7L'],
        ['SLC5A11','SS18'],
        ['SLC5A3','ZNF252P'],
        ['SLC6A16','TEAD2'],
        ['SLC6A20','URI1'],
        ['SLC7A13','TSSC2'],
        ['SLC7A2','TERT'],
        ['SLC7A9','ZNF614'],
        ['SLC9A3','TRIO'],
        ['SLC9B2','TBCK'],
        ['SMAD2','TFF1'],
        ['SMAGP','TFCP2'],
        ['SMARCA2','VLDLR'],
        ['SMARCA2','ZNF384'],
        ['SMARCA4','SMARCB1'],
        ['SMARCA4','TGIF1'],
        ['SMARCA4','TIMM29'],
        ['SMARCA4','TYK2'],
        ['SMARCA4','ZNF254'],
        ['SMARCC1','TRIM24'],
        ['SMARCC2','TSFM'],
        ['SMARCD2','STK11'],
        ['SMARCE1','TMEM99'],
        ['SMCHD1','USP9X'],
        ['SMEK1','SRM'],
        ['SMG5','TMEM79'],
        ['SMIM14','STK32B'],
        ['SMYD2','TP53BP2'],
        ['SMYD3','XRN1'],
        ['SMYD3','ZNF670'],
        ['SMYD3','ZNF695'],
        ['SNAP91','USP45'],
        ['SND1','TMEM178B'],
        ['SNHG5','SNX14'],
        ['SNTB1','ZNF250'],
        ['SNTB2','VPS4A'],
        ['SNTG2','TRPM6'],
        ['SNX14','SYNCRIP'],
        ['SNX2','STK10'],
        ['SNX29','TP53'],
        ['SNX30','TP53'],
        ['SORCS2','TADA2B'],
        ['SORD','SPG11'],
        ['SOX2-OT','WWTR1'],
        ['SOX5','TCF4'],
        ['SOX8','TRAC'],
        ['SP1','SP3'],
        ['SP4','STK31'],
        ['SPAG16','YES1'],
        ['SPARC','TNKS1BP1'],
        ['SPARC','USP6'],
        ['SPATA48','TAX1BP1'],
        ['SPBP','TCF20'],
        ['SPECC1L','TXNRD2'],
        ['SPI1','STMN1'],
        ['SPI1','TCF7'],
        ['SPIDR','UBE2V2'],
        ['SPINK5','STK32A'],
        ['SPINT2','TCEANC2'],
        ['SPOCK1','TBC1D9B'],
        ['SPRED2','THADA'],
        ['SPRR2B','STK3'],
        ['SPRR2E','STK3'],
        ['SPRYD3','TRAPPC9'],
        ['SPTAN1','TRAF4'],
        ['SRC','VWC2'],
        ['SREBF1','TOM1L2'],
        ['SREBF2','XRCC6'],
        ['SRF','STAT6'],
        ['SRGAP3','TATDN2'],
        ['SRP68','URM1'],
        ['SRP9','USH2A'],
        ['SRSF11','TANGO2'],
        ['SRSF3','TP53'],
        ['SRSF3','USP6'],
        ['SRSF6','SVEP1'],
        ['SS18','SSX'],
        ['SS18','SSX1'],
        ['SS18','SSX2'],
        ['SS18','SSX2B'],
        ['SS18','SSX4'],
        ['SS18','USP6'],
        ['SS18','YAP1'],
        ['SS18','ZBTB7A'],
        ['SS18L1','SSX1'],
        ['SSB','UBR3'],
        ['SSBP1','WEE2'],
        ['SSBP3','TMEM48'],
        ['SSH2','SUZ12'],
        ['SSH2','TCEAL6'],
        ['SSH2','TP53'],
        ['SSX','SYT'],
        ['SSX1','SYT'],
        ['SSX1','SYT4'],
        ['SSX2','SYT'],
        ['SSX2','SYT4'],
        ['SSX4','SYT'],
        ['SSX4','SYT4'],
        ['ST6GAL1','WHSC1'],
        ['ST7','TRIM24'],
        ['ST7L','THAP1'],
        ['STAG1','STXBP5L'],
        ['STAG3','STAG3L5P'],
        ['STAG3L2','TSGA10IP'],
        ['STARD13','TNFRSF8'],
        ['STARD3','STRADA'],
        ['STARD3','TAC4'],
        ['STAU1','TOP1'],
        ['STAU1','TOX2'],
        ['STIL','TAL1'],
        ['STIM1','YAP1'],
        ['STK10','UBE2D2'],
        ['STK11','TYK2'],
        ['STK24','TMEM159'],
        ['STK24','TMTC4'],
        ['STK24','TRIQK'],
        ['STK3','SULT1A3'],
        ['STK3','VPS13B'],
        ['STK3','ZNF706'],
        ['STK38','TDRD7'],
        ['STK38','VEGFA'],
        ['STK38','ZNF184'],
        ['STK38L','WNK1'],
        ['STK39','TMEM245'],
        ['STK4','TSHZ2'],
        ['STK40','THRAP3'],
        ['STRADA','TANC2'],
        ['STRN4','TECR'],
        ['STXBP6','TCF12'],
        ['STYK1','YBX3'],
        ['SUB1','WHSC1'],
        ['SUCO','TP53'],
        ['SULF2','UBE2V1'],
        ['SULF2','ZNF217'],
        ['SUPT4H1','ZBTB40'],
        ['SUSP1','TCBA1'],
        ['SUZ12','TP53'],
        ['SVOPL','TRIM24'],
        ['SYAP1','TXLNG'],
        ['SYCP2','VAPB'],
        ['SYN3','TP53'],
        ['SYNRG','TOB1'],
        ['SYNRG','TTBK2'],
        ['SYNRG','ZNF384'],
        ['SYT8','TNNI2'],
        ['SYT9','TDRD5'],
        ['TAF15','TCF3'],
        ['TAF15','TEC'],
        ['TAF15','ZNF384'],
        ['TAF2N','TEC'],
        ['TAF3','ZDHHC14'],
        ['TAF4B','WDR1'],
        ['TAF6L','TMEM179B'],
        ['TAL1','TCTA'],
        ['TAL1','TLX1'],
        ['TAL1','TLX1NB'],
        ['TAL1','TMEM91'],
        ['TAL1','TRB@'],
        ['TAL1','TRBC1'],
        ['TAL1','TRBC2'],
        ['TAL1','TRD@'],
        ['TAL1','TRDC'],
        ['TAL2','TLX1'],
        ['TAL2','TLX1NB'],
        ['TAL2','TRB@'],
        ['TAOK1','VMP1'],
        ['TAOK1','ZNF207'],
        ['TAP1','TAP2'],
        ['TAP1','ZAP70'],
        ['TARSL2','WFS1'],
        ['TAX1BP3','TRPV3'],
        ['TAZ','YAP1'],
        ['TBC1D15','TRHDE'],
        ['TBC1D19','TMEM199'],
        ['TBC1D22A','TP73'],
        ['TBC1D23','TF'],
        ['TBC1D3','USP32'],
        ['TBC1D31','ZNF704'],
        ['TBCB','UBE4B'],
        ['TBCD','ZNF750'],
        ['TBCEL','TECTA'],
        ['TBL1XR1','TP63'],
        ['TBP','WT1'],
        ['TBRG1','TMEM25'],
        ['TBX2','UBE2V1'],
        ['TBX2-AS1','UBE2V1'],
        ['TCF12','TEC'],
        ['TCF12','TMEM87A'],
        ['TCF21','TMED1'],
        ['TCF3','TEF'],
        ['TCF3','TFPT'],
        ['TCF3','ZNF384'],
        ['TCF4','VTI1A'],
        ['TCF4','ZNF362'],
        ['TCF7L2','VTI1A'],
        ['TCF7L2','VWA2'],
        ['TCL1A','TRA@'],
        ['TCL1A','TRB@'],
        ['TCL1A','TRD@'],
        ['TCL1B','TRA@'],
        ['TCL6','TRA@'],
        ['TCR@','TLX1'],
        ['TCR@','TLX1NB'],
        ['TCR@','TLX3'],
        ['TCR@','TLX3_'],
        ['TCRB@','TLX3'],
        ['TCRB@','TLX3_'],
        ['TDP1','VRK1'],
        ['TEAD4','TSPAN9'],
        ['TEC','TFG'],
        ['TEK','ZDHHC21'],
        ['TENC1','TP53'],
        ['TENC1','TTLL9'],
        ['TERT','TPPP'],
        ['TERT','TRIO'],
        ['TET1','TTC12'],
        ['TEX10','XPA'],
        ['TFDP2','XRN1'],
        ['TFE3','TFEB'],
        ['TFE3','UXT'],
        ['TFE3','VCP'],
        ['TFE3','YAP'],
        ['TFE3','YAP1'],
        ['TFEB','TRERF1'],
        ['TFEB','VEGFA'],
        ['TG','THADA'],
        ['TG','WNK3'],
        ['TGFB1','TP53'],
        ['TGFBR2','XPC'],
        ['TGM5','WDR18'],
        ['TGM7','UBAP1'],
        ['THADA','TRA2A'],
        ['THAP4','VAV1'],
        ['THEM7P','WT1'],
        ['THRAP3','USP6'],
        ['THUMPD3','ZNF713'],
        ['TLCD1','ZNF672'],
        ['TLN1','TMEM8B'],
        ['TLX1','TLX3'],
        ['TLX1','TRA@'],
        ['TLX1','TRB@'],
        ['TLX1','TRD@'],
        ['TLX1NB','TRA@'],
        ['TLX1NB','TRB@'],
        ['TLX1NB','TRD@'],
        ['TLX2','TMEM50B'],
        ['TLX2_','TMEM50B'],
        ['TLX3','TRA@'],
        ['TLX3','TRB@'],
        ['TLX3','TRD@'],
        ['TLX3_','TRA@'],
        ['TLX3_','TRB@'],
        ['TLX3_','TRD@'],
        ['TM4SF18','WWTR1'],
        ['TM9SF2','UBAC2'],
        ['TMC8','ZFP36L1'],
        ['TMCO1','UCK2'],
        ['TMED10','ZNF252'],
        ['TMEM120B','UBR4'],
        ['TMEM123','YAP1'],
        ['TMEM127','ZAP70'],
        ['TMEM138','TMEM216'],
        ['TMEM168','TRRAP'],
        ['TMEM184B','VPS13C'],
        ['TMEM19','TSPAN8'],
        ['TMEM25','USP28'],
        ['TMEM256','TP53'],
        ['TMEM256-PLSCR3','TNK1'],
        ['TMEM38B','TRBC2'],
        ['TMEM49','TRIM37'],
        ['TMEM63B','TTK'],
        ['TMEM91','ZNF550'],
        ['TMOD4','VPS72'],
        ['TMPRSS2','TTC3'],
        ['TMPRSS2','UMODL1'],
        ['TMPRSS2','ZFAT'],
        ['TMPRSS2','ZMIZ1'],
        ['TMPRSS2','ZNF589'],
        ['TMX1','TTYH2'],
        ['TNC','USP6'],
        ['TNK1','ZBTB7A'],
        ['TNR','TP53'],
        ['TNRC18','XPO1'],
        ['TNXA','TNXB'],
        ['TOP2A','TTBK2'],
        ['TP53','VAV1'],
        ['TP53','ZNF565'],
        ['TP53','ZNF76'],
        ['TP53BP2','ZSWIM5'],
        ['TP53RK','ZNF227'],
        ['TPCN2','YAP1'],
        ['TPH2','TRHDE'],
        ['TPM1','UBN2'],
        ['TPM3','TPR'],
        ['TPM3','TRKA'],
        ['TPS1','TPS2'],
        ['TRA2A','UBASH3B'],
        ['TRA2B','VSIG4'],
        ['TRA@','ZFP36L2'],
        ['TRAC','TRDC'],
        ['TRAF4','WSB1'],
        ['TRAP1','UBIAD1'],
        ['TRAP150','USP6'],
        ['TRAPPC10','USP25'],
        ['TRDN','UBR4'],
        ['TRE17','USP6'],
        ['TRGV1','TRGV8'],
        ['TRIM24','ULK4'],
        ['TRIO','ZNF558'],
        ['TRIP10','VAV1'],
        ['TRMT1L','XPR1'],
        ['TRPC6','YAP1'],
        ['TRPM7','USP8'],
        ['TRUB2','VAV2'],
        ['TSC1','TSC2'],
        ['TSEN54','UNC84A'],
        ['TSHZ2','UAP1'],
        ['TSHZ2','ZNF251'],
        ['TTBK2','WRAP53'],
        ['TTC12','YAP1'],
        ['TTC3','ZNF83'],
        ['TTC6','TXK'],
        ['TTR','TYRO3'],
        ['TTYH3','ZNF652'],
        ['TUBGCP2','ZNF511'],
        ['TUT1','ZNF564'],
        ['TXNDC11','ZC3H7A'],
        ['TYK2','UPF1'],
        ['TYK2','ZNF430'],
        ['UBA2','WTIP'],
        ['UBQLN4','ZSWIM4'],
        ['UBR4','ZFP37'],
        ['UBR5','ZNF423'],
        ['UBXN4','ZNF729'],
        ['UFC1','VANGL2'],
        ['ULK2','UQCR11'],
        ['UNK','WBP2'],
        ['UNK','ZDHHC7'],
        ['USO1','XPNPEP3'],
        ['USP10','ZDHHC7'],
        ['USP2','YAP1'],
        ['USP31','WNK1'],
        ['USP32','ZMYND8'],
        ['USP33','ZFYVE9'],
        ['USP34','XPO1'],
        ['USP54','ZMIZ1'],
        ['USP6','USP9X'],
        ['USP6','ZNF9'],
        ['VAPA','ZNF880'],
        ['VGLL2','ZNF84'],
        ['VPS13D','WASL'],
        ['VPS18','ZFYVE19'],
        ['VPS4A','ZDHHC9'],
        ['VSTM2A','ZNF713'],
        ['WDFY3','ZNF578'],
        ['WDR67','ZNF704'],
        ['WFDC13','ZBTB46'],
        ['WNK1','WNT5B'],
        ['XPA','ZNF680'],
        ['XRCC6BP1','YEATS4'],
        ['YAP1','ZNF721'],
        ['ZEB1','ZEB2'],
        ['ZFPM1','ZNRF1'],
        ['ZMPSTE24','ZMYM4'],
        ['ZMPSTE24','ZMYM5'],
        ['MRPS9','ALK'],
        ['TFG','NTRK2'],
        ['PARD3','ALK'],
        ['HIP1','ALK'],
        ['GCC2','ALK'],
        ['PLAG1','CHCHD7'],
        ['PLAG1','COL3A1'],
        ['PLAG1','SRSF3'],
        ['PLAG1','HNRNPC'],
        ['PLAG1','PCMTD1'],
        ['PLAG1','YWHAZ'],
        ['PLAG1','CTDSP2'],
        ['PLAG1','PPP2R2A'],
        ['A2M','ALK'],
        ['AAC1','AAC2'],
        ['ABCG2','BCRP'],
        ['ABI2','MTA3'],
        ['ABL','MYC'],
        ['ABL1','AIF1L'],
        ['ABL1','BCR'],
        ['ABL1','NUP214'],
        ['ABL2','GAB2'],
        ['ACACA','STAC2'],
        ['ACBD5','RET'],
        ['ACIN1','NUTM1'],
        ['ACTB','FOSB'],
        ['ACTB','GLI1'],
        ['ACTB','MITF'],
        ['ACTB','TFEB'],
        ['ACTG1','MITF'],
        ['ACTG2','ALK'],
        ['ADCK4','NUMBL'],
        ['ADD3','NUP98'],
        ['AF10','AF17'],
        ['AF4','AF9'],
        ['AF4','FMR2'],
        ['AF9','DOT1L'],
        ['AFAP1','NTRK2'],
        ['AFAP1L1','PDGFRB'],
        ['AFAP1L2','RET'],
        ['AFF1','AF4'],
        ['AFF1','KMT2A'],
        ['AFF2','FMR2'],
        ['AFF3','LAF4'],
        ['AFF3','PHF1'],
        ['AGAP3','BRAF'],
        ['AGGF1','PDGFRB'],
        ['AGK','BRAF'],
        ['AHRR','NCOA2'],
        ['AIF1L','ETV6'],
        ['AIG1','GOSR1'],
        ['AK2','AK8'],
        ['AKAP13','PDE8A'],
        ['AKAP8','BRD4'],
        ['AKAP9','BRAF'],
        ['AKT3','XPNPEP1'],
        ['ALK','CLTC'],
        ['ALK','EGFR'],
        ['ALK','EML4'],
        ['ALK','KMT2A'],
        ['ALK','KRAS'],
        ['ALK','MET'],
        ['ALK','PTPN3'],
        ['ALK','RANBP2'],
        ['ALK','RET'],
        ['ALK','ROS1'],
        ['ALK','TNS1'],
        ['ALK','TP53'],
        ['ALK','TPM3'],
        ['AMACR','ERG'],
        ['AML1','CLCA2'],
        ['AML1','EVI1'],
        ['AML1','FOG2'],
        ['AML1','LAF4'],
        ['AML1','LRP16'],
        ['AML1','MDS1'],
        ['AML1','MTG16'],
        ['AML1','MTG8'],
        ['AML1','PRDM16'],
        ['AML1','PRDX4'],
        ['AML1','RUNX1'],
        ['AMPK','SIRT1'],
        ['ANKRD26','RET'],
        ['ANKRD28','FNDC3B'],
        ['ANKRD28','NUP98'],
        ['ANXA2','MYO9A'],
        ['ANXA5','CCNA2'],
        ['API2','MALT1'],
        ['API2','MLT1'],
        ['APIP','FGFR2'],
        ['APIP','SLC1A2'],
        ['APOA5','APOA4'],
        ['APPBP2','PHF20L1'],
        ['AREG','EGFR'],
        ['ARHGEF2','NTRK1'],
        ['ARHGEF3','C8ORF38'],
        ['ARHGEF39','NRG1'],
        ['ARID1A','GPATCH3'],
        ['ARID1A','PRKD1'],
        ['ARID1B','PSMA1'],
        ['ARID1B','SNX9'],
        ['ARID1B','TFE3'],
        ['ARID1B','ZNF384'],
        ['ARL6','POT1'],
        ['ARMC10','BRAF'],
        ['ASAP1','USP6'],
        ['ASPL','TFE3'],
        ['ASPS','TFE3'],
        ['ASPSCR1','TFE3'],
        ['ATAD3B','ATAD3A'],
        ['ATAD3C','ATAD3B'],
        ['ATF7IP','PDGFRB'],
        ['ATG7','RAF1'],
        ['ATIC','ALK'],
        ['ATM','ATR'],
        ['ATP1B','NTRK1'],
        ['ATP1B1','NRG1'],
        ['ATP1B1','NTRK1'],
        ['ATP1B1','PRKACA'],
        ['ATP1B1','PRKACB'],
        ['ATP5L','KMT2A'],
        ['ATP8A2','FLT1'],
        ['ATR','ATM'],
        ['ATR','CHK1'],
        ['ATRX','DAXX'],
        ['ATXN1','NUTM1'],
        ['ATXN2L','JAK2'],
        ['AVEN','NUTM1'],
        ['AZGP1','GJC3'],
        ['B4GALT5','TERT'],
        ['B7.1','B7.2'],
        ['BABAM2','ALK'],
        ['BACH2','BCL2L1'],
        ['BAG4','FGFR1'],
        ['BAIAP2L1','BRAF'],
        ['BBIP1','RET'],
        ['BCAM','AKT2'],
        ['BCAN','NTRK1'],
        ['BCAP31','TEX28'],
        ['BCAS3','BCAS4'],
        ['BCAS3','ERBB2'],
        ['BCAS3','HOXB9'],
        ['BCAS4','AURKA'],
        ['BCAS4','BCAS3'],
        ['BCL1','CCND1'],
        ['BCL1','IGH@'],
        ['BCL11A','ALK'],
        ['BCL11B','TRDC'],
        ['BCL2','BCL6'],
        ['BCL2','CD10'],
        ['BCL2','IGH@'],
        ['BCL2','MYC'],
        ['BCL2L14','ETV6'],
        ['BCL3','RELB'],
        ['BCL6','BCL2'],
        ['BCL6','IGH@'],
        ['BCL6','LAZ3'],
        ['BCL6','MYC'],
        ['BCL6','RAF1'],
        ['BCL9','NTRK1'],
        ['BCOR','CCNB3'],
        ['BCOR','CREBBP'],
        ['BCOR','L3MBTL2'],
        ['BCOR','MAML3'],
        ['BCOR','MAP7D2'],
        ['BCOR','NUGGC'],
        ['BCOR','NUTM2G'],
        ['BCOR','RALGPS1'],
        ['BCOR','RARA'],
        ['BCOR','ZC3H7B'],
        ['BCORL1','ELF4'],
        ['BCORL1','NUTM1'],
        ['BCR','ABL'],
        ['BCR','ABL1'],
        ['BCR','FGFR1'],
        ['BCR','JAK2'],
        ['BCR','NTRK2'],
        ['BCR','PDGFRA'],
        ['BCR','RET'],
        ['BIRC3','MALT1'],
        ['BIRC6','ALK'],
        ['BMPR1B','GRM4'],
        ['BOB1','BCL2'],
        ['BOB1','OBF1'],
        ['BOC','PLAG1'],
        ['BORCS5','ETV6'],
        ['BRAF','AKAP9'],
        ['BRAF','CRAF'],
        ['BRAF','KIAA1549'],
        ['BRAF','KRAS'],
        ['BRCA1','BRCA2'],
        ['BRCA1','NBR1'],
        ['BRD3','NUTM1'],
        ['BRD4','LEUTX'],
        ['BRD4','NOTCH3'],
        ['BRD4','NUTM1'],
        ['BRD8','PHF1'],
        ['BRG1','SMARCA4'],
        ['BST1','CD38'],
        ['BTG2','PRMT1'],
        ['BTG3','CCDC40'],
        ['C11ORF95','RELA'],
        ['C2CD2','TFF1'],
        ['CALLA','CD10'],
        ['CAMTA1','WWTR1'],
        ['CANT1','ETV4'],
        ['CAPZA2','MET'],
        ['CARS','ALK'],
        ['CASP8','ERBB4'],
        ['CASZ1','DFFA'],
        ['CASZ1','MASP2'],
        ['CBFA2','CBFA2T1'],
        ['CBFA2','ETV6'],
        ['CBFA2T3','ACSF3'],
        ['CBFA2T3','GLIS2'],
        ['CBFB','MYH11'],
        ['CBFB','SMMHC'],
        ['CCBL1','ARL1'],
        ['CCDC28A','IL22RA2'],
        ['CCDC30','ROS1'],
        ['CCDC32','CBX3'],
        ['CCDC47','PRKCA'],
        ['CCDC6','ALK'],
        ['CCDC6','PDGFRB'],
        ['CCDC6','RET'],
        ['CCDC88C','PDGFRB'],
        ['CCL17','TARC'],
        ['CCL21','CCR7'],
        ['CCN2','CTGF'],
        ['CCNB3','BCOR'],
        ['CCND1','CDK4'],
        ['CCND1','IGH@'],
        ['CCND1','IGK@'],
        ['CCND1','MRCK'],
        ['CCND1','MYEOV'],
        ['CCND2','CCND3'],
        ['CCND2','CDK4'],
        ['CCND2','IGL@'],
        ['CCNE','CDK2'],
        ['CCNY','NRG4'],
        ['CD10','CALLA'],
        ['CD10','MUM1'],
        ['CD15','CD34'],
        ['CD163','AIF1'],
        ['CD19','CD10'],
        ['CD19','CD22'],
        ['CD2','CD34'],
        ['CD2','MYC'],
        ['CD20','CD10'],
        ['CD200','CD200R'],
        ['CD21','CD23'],
        ['CD27','CD44'],
        ['CD273','PDL2'],
        ['CD274','PDCD1LG2'],
        ['CD274','PDL1'],
        ['CD40','CD154'],
        ['CD40','CD40L'],
        ['CD44','IGF1R'],
        ['CD44','SLC1A2'],
        ['CD5','CD19'],
        ['CD53','PDGFRB'],
        ['CD56','CD34'],
        ['CD63','BCAR4'],
        ['CD63','PRKCD'],
        ['CD68','PGM1'],
        ['CD7','CD5'],
        ['CD74','NRG1'],
        ['CD74','NTRK1'],
        ['CD74','ROS1'],
        ['CD80','CD28'],
        ['CD99','MIC2'],
        ['CDC27','OAT'],
        ['CDC42SE2','BRAF'],
        ['CDH11','USP6'],
        ['CDK4','MDM2'],
        ['CDK5RAP2','PDGFRA'],
        ['CDK6','IGK@'],
        ['CDKN1C','KCNQ1OT1'],
        ['CDKN2A','CDKN2B'],
        ['CDKN2A','IL1RAPL2'],
        ['CDKN2A','RB1'],
        ['CDKN2A','UBAP1'],
        ['CDKN2D','WDFY2'],
        ['CEACAM7','CEACAM6'],
        ['CEBPE','IGH@'],
        ['CEP110','FGFR1'],
        ['CEP72','ROS1'],
        ['CEP85L','PDGFRB'],
        ['CEP85L','ROS1'],
        ['CFHR1','CFHR3'],
        ['CFHR1','CFHR5'],
        ['CFHR5','CFHR2'],
        ['CHCHD7','PLAG1'],
        ['CHD7','BEND2'],
        ['CHD7','VGLL3'],
        ['CHFR','GOLGA3'],
        ['CHOP','FUS'],
        ['CHOP','GADD153'],
        ['CHRM5','NUTM1'],
        ['CHRNA7','ALK'],
        ['CHST11','IGH@'],
        ['CHTOP','NTRK1'],
        ['CIC','DUX4'],
        ['CIC','DUX4L8'],
        ['CIC','FOXO4'],
        ['CIC','LEUTX'],
        ['CIC','NUTM1'],
        ['CIC','NUTM2A'],
        ['CIITA','CREBBP'],
        ['CITED2','PRDM10'],
        ['CK7','MUC1'],
        ['CLDN18','ARHGAP26'],
        ['CLEC6A','CLEC4D'],
        ['CLIP2','ALK'],
        ['CLIP2','MET'],
        ['CLIP4','VSNL1'],
        ['CLOCK','BMAL1'],
        ['CLTC','ALK'],
        ['CLTC','FOSB'],
        ['CLTC','TFE3'],
        ['CLTC','TFEB'],
        ['CLTC','VMP1'],
        ['CLTCL','ALK'],
        ['CMTM8','CMTM7'],
        ['CNOT2','RAB3IP'],
        ['CNOT2','WT1'],
        ['CNTRL','ABL1'],
        ['COL14A1','SKAP1'],
        ['COL1A1','FYN'],
        ['COL1A1','PDGFB'],
        ['COL1A1','PPP2R2C'],
        ['COL1A1','USP6'],
        ['COL1A2','ALK'],
        ['COL1A2','PDGFB'],
        ['COL1A2','PLAG1'],
        ['COL3A1','PDGFB'],
        ['COL3A1','PLAG1'],
        ['COL3A1','USP6'],
        ['COL4A1','VEGFD'],
        ['COL4A5','COL4A6'],
        ['COL6A2','USP6'],
        ['COL6A3','CSF1'],
        ['COL6A3','PDGFD'],
        ['COMMD10','AP3S1'],
        ['COMMD3','BMI1'],
        ['CPQ','PRKDC'],
        ['CPSF6','PDGFRB'],
        ['CPSF6','RARG'],
        ['CRAF','RAF1'],
        ['CREB3L2','PPARG'],
        ['CREBBP','BCOR'],
        ['CREBBP','BCORL1'],
        ['CREBBP','GOLGA6L2'],
        ['CREBBP','SRGAP2B'],
        ['CREBBP','SRRM2'],
        ['CREBBP','ZNF384'],
        ['CRHR1','KANSL1'],
        ['CRLF2','IGH@'],
        ['CRLF2','JAK2'],
        ['CRLF2','P2RY8'],
        ['CRTC1','MAML1'],
        ['CRTC1','MAML2'],
        ['CRTC1','SS18'],
        ['CRTC1','TRIM11'],
        ['CRTC2','MAML2'],
        ['CRTC3','MAML2'],
        ['CRYL1','IFT88'],
        ['CSF1','CDH1'],
        ['CSF1','COL6A3'],
        ['CSF1','CSF1R'],
        ['CSF1','FN1'],
        ['CSF1','S100A10'],
        ['CSF1','VCAM1'],
        ['CSF2RA','CRLF2'],
        ['CSF3R','SPTAN1'],
        ['CSGALNACT2','RET'],
        ['CSMD1','MEAF6'],
        ['CSNK2A1','PDGFRB'],
        ['CTAGE5','KHDRBS3'],
        ['CTDSP1','CFLAR'],
        ['CTLA4','CD28'],
        ['CTNNB1','EGFR'],
        ['CTNNB1','PLAG1'],
        ['CTNNB1','USP6'],
        ['CTNND1','ARHGAP26'],
        ['CTRC','NTRK1'],
        ['CTSC','RAB38'],
        ['CTSD','IFITM10'],
        ['CUL1','BRAF'],
        ['CUX1','ALK'],
        ['CUX1','BRAF'],
        ['CUX1','RET'],
        ['CXCL12','CXCR4'],
        ['CXCR4','CXCR7'],
        ['CXCR4','MAML2'],
        ['CYP11B1','CYP11B2'],
        ['CYP11B2','CYP11B1'],
        ['CYP21A1P','CYP21A2'],
        ['CYP4X1','CYP4Z2P'],
        ['D2HGDH','GAL3ST2'],
        ['DAXX','KIFC3'],
        ['DAZAP1','MEF2D'],
        ['DCLRE1C','ARTEMIS'],
        ['DCP1A','CCL27'],
        ['DCTN1','ALK'],
        ['DCTN1','NTRK1'],
        ['DDA1','FAM129C'],
        ['DDIT3','CHOP'],
        ['DDIT3','FUS'],
        ['DDX3X','MLLT10'],
        ['DDX3X','PRKD1'],
        ['DDX5','ETV4'],
        ['DEK','AFF2'],
        ['DEK','NUP214'],
        ['DERA','GLI1'],
        ['DHH','RHEBL1'],
        ['DHX35','BPIFA2'],
        ['DIAPH1','PDGFRB'],
        ['DIAPH2','ALK'],
        ['DLG5','RET'],
        ['DNAH14','IKZF1'],
        ['DNAJB1','PRKACA'],
        ['DNAJB1','PRKCA'],
        ['DNAJB6','PLAG1'],
        ['DOCK4','IMMP2L'],
        ['DOT1L','HES6'],
        ['DPF2','BAF45D'],
        ['DPH7','PTP4A3'],
        ['DPP9','PPP6R3'],
        ['DR4','DR1'],
        ['DST','BRAF'],
        ['DTD1','PDGFRB'],
        ['DUSP22','IRF4'],
        ['DUSP22','MUM1'],
        ['DUX4','CIC'],
        ['DUX4','IGH@'],
        ['DVL2','TFE3'],
        ['DYSF','ALK'],
        ['E2F6','FKBP4'],
        ['EBF1','PDGFRB'],
        ['EEF1A1','PLAG1'],
        ['EEF1DP3','FRY'],
        ['EEF1G','ALK'],
        ['EEF2','ALK'],
        ['EGFL7','FOSB'],
        ['EGFR','ALK'],
        ['EGFR','BRAF'],
        ['EGFR','ERBB2'],
        ['EGFR','HER2'],
        ['EGFR','KRAS'],
        ['EGFR','MET'],
        ['EGFR','PPARGC1A'],
        ['EGFR','PURB'],
        ['EGFR','RAD51'],
        ['EGFR','ROS1'],
        ['EGFR','SEPT14'],
        ['EGFR','TP53'],
        ['EGFR','VWC2'],
        ['EGFR','ZNF267'],
        ['EGR1','GRIA1'],
        ['EGR1','RPS14'],
        ['EHD1','CAPN12'],
        ['EIF2AK1','ATR'],
        ['EIF2AK3','ALK'],
        ['EIF2AK4','ANXA2P2'],
        ['EIF3E','RSPO2'],
        ['EIF3K','ACTN4'],
        ['EIF4A2','MECOM'],
        ['EIF4G1','BCL6'],
        ['EIF5A','USP6'],
        ['ELAVL1','TYK2'],
        ['ELE1','RET'],
        ['ELF4','ERG'],
        ['ELKS','RET'],
        ['ELMOD3','SH2D6'],
        ['EMC7','NUTM1'],
        ['EMILIN2','PDGFD'],
        ['EML1','ABL1'],
        ['EML4','ALK'],
        ['EML4','BIRC6'],
        ['EML4','BRAF'],
        ['EML4','MET'],
        ['EML4','NTRK1'],
        ['EML4','NTRK3'],
        ['EP300','BCOR'],
        ['EP300','BCORL1'],
        ['EP300','ZNF384'],
        ['EP400','PHF1'],
        ['EPB41L2','BRAF'],
        ['EPC1','ASXL2'],
        ['EPC1','BCOR'],
        ['EPC1','PHF1'],
        ['EPC1','SUZ12'],
        ['EPC2','PHF1'],
        ['EPCAM','MSH2'],
        ['EPHB4','MARCH10'],
        ['EPOR','IGH@'],
        ['EPS15','AF1P'],
        ['EPS15','KMT2A'],
        ['ERBB2','CCL4'],
        ['ERBB2','ERBB3'],
        ['ERC1','RET'],
        ['ERG','DUX4'],
        ['ERG','ETV1'],
        ['ERG','MYC'],
        ['ERG','PARP1'],
        ['ERG','PTEN'],
        ['ERG','TMPRSS2'],
        ['ERK1','ERK2'],
        ['ESR1','CCDC170'],
        ['ESR1','DAB2'],
        ['ESR1','NCOA2'],
        ['ESR1','NCOA3'],
        ['ESR1','YAP1'],
        ['ESRP1','RAF1'],
        ['ETO2','CBFA2T3'],
        ['ETO2','GLIS2'],
        ['ETS1','ELK1'],
        ['ETS2','ERG'],
        ['ETV4','AHRR'],
        ['ETV6','ABL'],
        ['ETV6','ABL1'],
        ['ETV6','ABL2'],
        ['ETV6','ACSL6'],
        ['ETV6','ALK'],
        ['ETV6','AML1'],
        ['ETV6','ARNT'],
        ['ETV6','CBFA2'],
        ['ETV6','CDX2'],
        ['ETV6','CTNNB1'],
        ['ETV6','EVI1'],
        ['ETV6','FLT3'],
        ['ETV6','HIC1'],
        ['ETV6','IGH@'],
        ['ETV6','JAK2'],
        ['ETV6','LPXN'],
        ['ETV6','LYN'],
        ['ETV6','MAML3'],
        ['ETV6','MDS2'],
        ['ETV6','MECOM'],
        ['ETV6','MET'],
        ['ETV6','NCOA2'],
        ['ETV6','NTRK3'],
        ['ETV6','NUFIP1'],
        ['ETV6','PDGFRA'],
        ['ETV6','PDGFRB'],
        ['ETV6','PER1'],
        ['ETV6','PITX3'],
        ['ETV6','RET'],
        ['ETV6','RUNX1'],
        ['ETV6','SNUPN'],
        ['ETV6','SYK'],
        ['ETV6','TNFRSF8'],
        ['ETV6','TRKC'],
        ['ETV6','TTL'],
        ['ETV6','ZNF385A'],
        ['EVI1','PRDM16'],
        ['EWSR1','ATF1'],
        ['EWSR1','BEND2'],
        ['EWSR1','CHOP'],
        ['EWSR1','CREB1'],
        ['EWSR1','CREB3L1'],
        ['EWSR1','CREB3L2'],
        ['EWSR1','CREB3L3'],
        ['EWSR1','CREM'],
        ['EWSR1','DDIT3'],
        ['EWSR1','ELF5'],
        ['EWSR1','ERG'],
        ['EWSR1','ETV4'],
        ['EWSR1','FEV'],
        ['EWSR1','FLI1'],
        ['EWSR1','FUS'],
        ['EWSR1','KLF15'],
        ['EWSR1','KLF17'],
        ['EWSR1','NFATC1'],
        ['EWSR1','NFATC2'],
        ['EWSR1','NR4A3'],
        ['EWSR1','PATZ1'],
        ['EWSR1','PBX1'],
        ['EWSR1','PBX3'],
        ['EWSR1','POU5F1'],
        ['EWSR1','SMAD3'],
        ['EWSR1','SMARCA5'],
        ['EWSR1','SSX1'],
        ['EWSR1','TCF7L2'],
        ['EWSR1','TFCP2'],
        ['EWSR1','TFE3'],
        ['EWSR1','TFEB'],
        ['EWSR1','VEZF1'],
        ['EWSR1','VGLL1'],
        ['EWSR1','WT1'],
        ['EWSR1','YY1'],
        ['EWSR1','ZNF444'],
        ['EXOC2','AK7'],
        ['EXOC6B','TNS3'],
        ['EXPH5','COL1A2'],
        ['EZR','ERBB4'],
        ['EZR','ROS1'],
        ['F11R','NRG2'],
        ['FADD','MORT1'],
        ['FAM131A','BRAF'],
        ['FAM131B','BRAF'],
        ['FAM133B','CDK6'],
        ['FAM174A','WWC1'],
        ['FAM22B','YWHAE'],
        ['FAM91A1','NRG1'],
        ['FANCA','VPS9D1'],
        ['FANCJ','BRIP1'],
        ['FARP1','STK24'],
        ['FAT1','USP6'],
        ['FBXO32','PLAG1'],
        ['FGF1','FGFR1'],
        ['FGF23','FGFR1'],
        ['FGF3','TACC3'],
        ['FGFR1','BCR'],
        ['FGFR1','ERLIN2'],
        ['FGFR1','FN1'],
        ['FGFR1','HOOK3'],
        ['FGFR1','PLAG1'],
        ['FGFR1','TACC1'],
        ['FGFR1','USP6'],
        ['FGFR1','ZNF198'],
        ['FGFR1OP','RET'],
        ['FGFR2','BICC1'],
        ['FGFR2','CCDC6'],
        ['FGFR2','CREB5'],
        ['FGFR2','CTNNA3'],
        ['FGFR2','KIAA1598'],
        ['FGFR2','OGDH'],
        ['FGFR2','PPHLN1'],
        ['FGFR2','TACC3'],
        ['FGFR2','ZMYND11'],
        ['FGFR3','BAIAP2L1'],
        ['FGFR3','CAMK2A'],
        ['FGFR3','IGH@'],
        ['FGFR3','MMSET'],
        ['FGFR3','NBR1'],
        ['FGFR3','TACC3'],
        ['FHIT','FRA3B'],
        ['FHL2','GLI2'],
        ['FIP1L1','PDGFA'],
        ['FIP1L1','PDGFR'],
        ['FIP1L1','PDGFRA'],
        ['FIP1L1','RARA'],
        ['FKHR','PAX3'],
        ['FLT3','ETV6'],
        ['FLT3','TRIP11'],
        ['FN1','ACVR2A'],
        ['FN1','ALK'],
        ['FN1','EGF'],
        ['FN1','FGF1'],
        ['FN1','FGFR1'],
        ['FN1','FGFR2'],
        ['FN1','IGF1R'],
        ['FN1','MERTK'],
        ['FN1','NTRK1'],
        ['FN1','RET'],
        ['FN1','ROS1'],
        ['FN1','TEK'],
        ['FN1','TGFA'],
        ['FN1','TIMP1'],
        ['FNDC3B','PRKCI'],
        ['FOG2','AML1'],
        ['FOG2','ZFPM2'],
        ['FOS','ANKH'],
        ['FOS','FOSB'],
        ['FOS','LMNA'],
        ['FOS','RUNX2'],
        ['FOS','VIM'],
        ['FOXN3','BRAF'],
        ['FOXO1','ERG'],
        ['FOXO1','FGFR1'],
        ['FOXO1','FOXO3'],
        ['FOXO1','PAX3'],
        ['FOXO1','PAX7'],
        ['FOXO1','STK24'],
        ['FOXO4','OGT'],
        ['FOXP1','ABL1'],
        ['FOXP1','PDGFRA'],
        ['FUBP1','TFE3'],
        ['FUS','ATF1'],
        ['FUS','BBF2H7'],
        ['FUS','CHOP'],
        ['FUS','CREB1'],
        ['FUS','CREB3L1'],
        ['FUS','CREB3L2'],
        ['FUS','CREM'],
        ['FUS','DDIT3'],
        ['FUS','ERG'],
        ['FUS','KLF17'],
        ['FUS','NFATC2'],
        ['FUS','POU5F1'],
        ['FUS','TFCP2'],
        ['FYCO1','RET'],
        ['FYN','TRAF3IP2'],
        ['GAB1','ABL1'],
        ['GABRA3','FLT1'],
        ['GAS6','FOXA1'],
        ['GATA2','HOXA9'],
        ['GATA6','FOXO1'],
        ['GATM','RAF1'],
        ['GATSL2','GTF2I'],
        ['GCC2','ALK'],
        ['GCC2','PDGFRB'],
        ['GEM','PLAG1'],
        ['GFAP','TNF'],
        ['GFPT1','ALK'],
        ['GJA12','GJC2'],
        ['GLI1','FOXO4'],
        ['GLRB','GRIA2'],
        ['GLYR1','SLC9A8'],
        ['GOLGA4','RAF1'],
        ['GOLGB1','BRAF'],
        ['GOLGB1','PDGFRB'],
        ['GOLM1','MAK10'],
        ['GOLT1A','KISS1'],
        ['GOPC','ROS1'],
        ['GR6','EVI1'],
        ['GREB1','CTNNB1'],
        ['GREB1','NCOA1'],
        ['GREB1','NCOA2'],
        ['GREB1','NR4A3'],
        ['GREB1','SS18'],
        ['GREM1','FMN1'],
        ['GRHL2','ERG'],
        ['GTF2I','BRAF'],
        ['GTF2I','NCOA2'],
        ['GTF2I','PDGFRB'],
        ['GTF2I','RARA'],
        ['GTF3A','CDK8'],
        ['GYPB','GYPA'],
        ['H19','IGF2'],
        ['H2AFY','MECOM'],
        ['HAS2','PLAG1'],
        ['HBB','HBD'],
        ['HBD','HBB'],
        ['HBEGF','RBM27'],
        ['HBG2','HBG1'],
        ['HELIOS','BCL11B'],
        ['HER2','GRB7'],
        ['HER2','HER3'],
        ['HERC1','PML'],
        ['HERPUD1','RAF1'],
        ['HEY1','NCOA2'],
        ['HGF','CACNA2D1'],
        ['HGF','MET'],
        ['HIP1','ALK'],
        ['HLXB9','ETV6'],
        ['HMGA2','DYRK2'],
        ['HMGA2','EFCAB6'],
        ['HMGA2','EGFR'],
        ['HMGA2','FHIT'],
        ['HMGA2','LPP'],
        ['HMGA2','NAA11'],
        ['HMGA2','NCOR2'],
        ['HMGA2','NFIB'],
        ['HMGA2','RAD51L1'],
        ['HMGA2','RDC1'],
        ['HMGA2','SETBP1'],
        ['HMGA2','SOX5'],
        ['HMGA2','TMTC2'],
        ['HMGA2','TRAF3IP2'],
        ['HMGA2','WIF1'],
        ['HMGA2','YAP1'],
        ['HMGIC','LHFP'],
        ['HMGIC','LPP'],
        ['HNRNPA1','ALK'],
        ['HNRNPA1L2','SUGT1'],
        ['HNRNPA2B1','ETV1'],
        ['HNRNPH3','ALK'],
        ['HNRNPH3','TFE3'],
        ['HNRPDL','PBX1'],
        ['HOOK1','ALK'],
        ['HOX11','TCL3'],
        ['HOXA9','ABL'],
        ['HOXA9','MEIS1'],
        ['HOXA9','PBX3'],
        ['HOXC13','NUP98'],
        ['HOXD11','AGAP3'],
        ['HPCAL1','ALK'],
        ['HRAS','PIK3CA'],
        ['HTN3','MSANTD3'],
        ['HTR4','ST3GAL1'],
        ['HTT','PKN1'],
        ['ICOS','CD28'],
        ['IDH1','FGFR2'],
        ['IDH1','IDH2'],
        ['IDH1','KRAS'],
        ['IDS','IDSP1'],
        ['IFNA2','TA1'],
        ['IGF1R','TTC23'],
        ['IGF2','H19'],
        ['IGF2','IGF1R'],
        ['IGFBP5','ALK'],
        ['IGH@','BCL1'],
        ['IGH@','BCL10'],
        ['IGH@','BCL11A'],
        ['IGH@','BCL2'],
        ['IGH@','BCL3'],
        ['IGH@','BCL6'],
        ['IGH@','CCND1'],
        ['IGH@','CCND2'],
        ['IGH@','CCND3'],
        ['IGH@','CEBPE'],
        ['IGH@','CHST11'],
        ['IGH@','CRLF2'],
        ['IGH@','DUX4'],
        ['IGH@','EPOR'],
        ['IGH@','FGFR3'],
        ['IGH@','FOXP1'],
        ['IGH@','IGF2BP1'],
        ['IGH@','IGK@'],
        ['IGH@','IL3'],
        ['IGH@','IRF8'],
        ['IGH@','MAF'],
        ['IGH@','MAFA'],
        ['IGH@','MAFB'],
        ['IGH@','MALT1'],
        ['IGH@','MLT1'],
        ['IGH@','MMSET'],
        ['IGH@','MYC'],
        ['IGH@','WHSC1'],
        ['IGHJ1','IGHJ6'],
        ['IGHM','IGHD'],
        ['IGK@','CCND3'],
        ['IGK@','IGL@'],
        ['IGK@','IRF4'],
        ['IGK@','MYC'],
        ['IGL@','CCND2'],
        ['IGL@','MYC'],
        ['IGLL5','IGH@'],
        ['IGRP','G6PC2'],
        ['IKBKG','RET'],
        ['IKZF1','TYW1'],
        ['IKZF2','ERBB4'],
        ['IL18','IL2'],
        ['IL3','IGH@'],
        ['IMMP2L','DOCK4'],
        ['ING3','BCOR'],
        ['INHBE','GLI1'],
        ['INI1','SMARCB1'],
        ['INPP5D','ABL1'],
        ['INTS3','CHTOP'],
        ['IRA1','RGS17'],
        ['IRF2BP2','CDX1'],
        ['IRF2BP2','NTRK1'],
        ['IRF2BP2','NTRK3'],
        ['IRF2BP2','RARA'],
        ['IRF4','DUSP22'],
        ['IRF4','MUM1'],
        ['IRX2','TERT'],
        ['ITCH','ASIP'],
        ['ITGA8','RET'],
        ['ITGAV','ALK'],
        ['ITIH2','RET'],
        ['ITK','FER'],
        ['ITK','SYK'],
        ['ITPK1','FBP2'],
        ['ITPR2','BCL6'],
        ['JAK1','JAK2'],
        ['JAK1','STAT3'],
        ['JAK2','CRLF2'],
        ['JAK2','PCM1'],
        ['JAK2','PDCD1LG2'],
        ['JAK2','PDL1'],
        ['JAK2','STAT'],
        ['JAK2','STAT3'],
        ['JAK2','STAT5'],
        ['JAK3','INSL3'],
        ['JAK3','STAT5B'],
        ['JAZF1','JJAZ1'],
        ['JAZF1','PHF1'],
        ['JAZF1','SUZ12'],
        ['JJAZ1','SUZ12'],
        ['K20','A20'],
        ['KANK1','ALK'],
        ['KANK1','PDGFRB'],
        ['KANK2','ALK'],
        ['KANSL1','ARL17A'],
        ['KANSL1','CMC4'],
        ['KANSL1','KAT6B'],
        ['KANSL1','MTCP1'],
        ['KAT6A','CREBBP'],
        ['KAT6A','LEUTX'],
        ['KAT6A','TFE3'],
        ['KAT6B','KANSL1'],
        ['KCNMB4','CCND3'],
        ['KCTD16','NTRK1'],
        ['KDM2A','WWTR1'],
        ['KDM3B','ETF1'],
        ['KDR','PDGFRA'],
        ['KHDRBS1','NTRK3'],
        ['KIAA0319','TDP2'],
        ['KIAA1217','RET'],
        ['KIAA1217','RET51'],
        ['KIAA1468','RET'],
        ['KIAA1509','PDGFRB'],
        ['KIAA1549','BRAF'],
        ['KIAA2022','MRX98'],
        ['KIAA2026','NUDT11'],
        ['KIF13B','NRG1'],
        ['KIF5B','ALK'],
        ['KIF5B','EGFR'],
        ['KIF5B','ERBB4'],
        ['KIF5B','MET'],
        ['KIF5B','NTRK2'],
        ['KIF5B','PDGFRA'],
        ['KIF5B','RET'],
        ['KIP1','CDKN1B'],
        ['KIRREL','PRKCA'],
        ['KIT','DOG1'],
        ['KIT','PDGFR'],
        ['KIT','PDGFRA'],
        ['KLC1','ALK'],
        ['KLC1','ROS1'],
        ['KLF1','BCL11A'],
        ['KLHL7','BRAF'],
        ['KLK2','ETV4'],
        ['KLK2','FGFR2'],
        ['KLK4','KLKP1'],
        ['KMT2A','ABI1'],
        ['KMT2A','AF4'],
        ['KMT2A','AF9'],
        ['KMT2A','AFDN'],
        ['KMT2A','AFF1'],
        ['KMT2A','ARHGEF12'],
        ['KMT2A','BCOR'],
        ['KMT2A','ELL'],
        ['KMT2A','EPS15'],
        ['KMT2A','FLNA'],
        ['KMT2A','GIMAP8'],
        ['KMT2A','MAML1'],
        ['KMT2A','MAML2'],
        ['KMT2A','ME2'],
        ['KMT2A','MLLT1'],
        ['KMT2A','MLLT10'],
        ['KMT2A','MLLT11'],
        ['KMT2A','MLLT3'],
        ['KMT2A','MLLT4'],
        ['KMT2A','MLLT6'],
        ['KMT2A','NEBL'],
        ['KMT2A','PICALM'],
        ['KMT2A','PRRX1'],
        ['KMT2A','SEPT5'],
        ['KMT2A','SEPT9'],
        ['KMT2A','SNX9'],
        ['KMT2A','USP10'],
        ['KMT2A','USP2'],
        ['KMT2A','USP8'],
        ['KMT2A','YAP1'],
        ['KMT2C','TP53'],
        ['KMT2D','BCOR'],
        ['KMT2E','ASNS'],
        ['KRAS','NRAS'],
        ['KTN1','ALK'],
        ['LACTB2','NCOA2'],
        ['LAMTOR1','AKT1'],
        ['LAMTOR1','PRKCD'],
        ['LARP1','CNOT8'],
        ['LASP2','NEBL'],
        ['LATS1','PSEN1'],
        ['LAZ3','BCL6'],
        ['LAZ3','LCP1'],
        ['LCB1','LCB2'],
        ['LCP1','LAZ3'],
        ['LDLR','RPL31P11'],
        ['LDLR','TERT'],
        ['LEO1','SLC12A1'],
        ['LHX6','NDUFA8'],
        ['LIFR','PLAG1'],
        ['LINC00211','ALK'],
        ['LIPI','NTRK1'],
        ['LKB1','STK11'],
        ['LMNA','NTRK1'],
        ['LMNA','TRKA'],
        ['LMO3','BORCS5'],
        ['LMO7','BRAF'],
        ['LPCAT1','TERT'],
        ['LPL','PEX5'],
        ['LPP','HMGA2'],
        ['LPP','HMGIC'],
        ['LPP','TPRG1'],
        ['LRIG1','TAT'],
        ['LRP1','SNRNP25'],
        ['LRP16','RUNX1'],
        ['LRRFIP1','ALK'],
        ['LRRFIP2','RAF1'],
        ['LUC7L3','TFE3'],
        ['LUM','USP6'],
        ['MAD2B','PRCC'],
        ['MAF','MAFB'],
        ['MAGI3','AKT3'],
        ['MAL','MKL1'],
        ['MALAT1','GLI1'],
        ['MALAT1','TFEB'],
        ['MALT1','API2'],
        ['MALT1','MAP4'],
        ['MAN1A1','ROS1'],
        ['MAN2A1','FER'],
        ['MAP2K4','MAP2K4P1'],
        ['MAP3K3','TNS3'],
        ['MAP3K7CL','ERG'],
        ['MAP4K3','FIGLA'],
        ['MAPKAPK5','ACAD10'],
        ['MATR3','TFE3'],
        ['MBD4','PTPRC'],
        ['MBNL1','KIAA2018'],
        ['MBTD1','PHF1'],
        ['MDA5','LGP2'],
        ['MDK','HER2'],
        ['MDM1','NUP107'],
        ['MDM2','CDK4'],
        ['MDM2','CPM'],
        ['MDS1','EVI1'],
        ['MDS2','ETV6'],
        ['MEAF6','PHF1'],
        ['MEAF6','SUZ12'],
        ['MECOM','HHLA1'],
        ['MECP2','TEX28'],
        ['MECT1','MAML2'],
        ['MED1','ACSF2'],
        ['MED12','PRDM10'],
        ['MED13','BCAS3'],
        ['MED15','TFE3'],
        ['MEF2C','SS18'],
        ['MEF2D','BCL9'],
        ['MEF2D','CSF1R'],
        ['MEF2D','DAZAP1'],
        ['MEF2D','HNRNPH1'],
        ['MEF2D','HNRNPUL1'],
        ['MEF2D','SS18'],
        ['MEIS1','FOXO1'],
        ['MEIS1','NCOA1'],
        ['MEIS1','NCOA2'],
        ['MEIS1','PBX3'],
        ['MESDC2','SENP1'],
        ['MET','ALK'],
        ['MET','ATXN7L1'],
        ['MET','ERK2'],
        ['MET','HGF'],
        ['MET','SRC'],
        ['MGA','NUTM1'],
        ['MGST3','ZMAT5'],
        ['MIC2','CD99'],
        ['MIR3924','RET'],
        ['MITF','CREM'],
        ['MKL1','SRF'],
        ['MKRN1','BRAF'],
        ['MLH1','MSH2'],
        ['MLH1','PMS2'],
        ['MLL3','KMT2C'],
        ['MLLT10','AF10'],
        ['MLLT10','DNAJC1'],
        ['MLLT10','PICALM'],
        ['MLLT3','AF9'],
        ['MLLT3','KMT2A'],
        ['MLLT4','AF6'],
        ['MLLT6','AF17'],
        ['MLPH','ALK'],
        ['MLT1','MALT1'],
        ['MMSET','FGFR3'],
        ['MN1','BEND2'],
        ['MN1','ETV6'],
        ['MN1','FLI1'],
        ['MN1','GTSE1'],
        ['MN1','KMT2A'],
        ['MN1','PATZ1'],
        ['MNX1','ETV6'],
        ['MPP5','FAM71D'],
        ['MPRIP','ALK'],
        ['MPRIP','NTRK1'],
        ['MPRIP','PDGFRB'],
        ['MRE11','RAD50'],
        ['MRP1','ABCC1'],
        ['MRP3','ABCC3'],
        ['MRPL48','GAB2'],
        ['MRPS30','ARID2'],
        ['MRPS31','SUGT1'],
        ['MS4A6A','BRAF'],
        ['MS4A7','C2CD3'],
        ['MSH2','MLH1'],
        ['MSH2','MSH3'],
        ['MSH2','MSH6'],
        ['MSI2','EVI1'],
        ['MSI2','HOXA9'],
        ['MSMB','NCOA4'],
        ['MSN','ALK'],
        ['MSN','ROS1'],
        ['MTAP','ANRIL'],
        ['MTAP','BRAF'],
        ['MTAP','CDKN2A'],
        ['MTAP','RAF1'],
        ['MTCH2','FNBP4'],
        ['MTG16','MTGR1'],
        ['MTM1','MAMLD1'],
        ['MTMR2','NTRK2'],
        ['MTOR','TP53BP1'],
        ['MUC1','VEGFR2'],
        ['MUC2','NTRK2'],
        ['MUM1','IRF4'],
        ['MXD1','NUTM1'],
        ['MXD4','NUTM1'],
        ['MYB','GATA1'],
        ['MYB','MBNL1'],
        ['MYB','MYBL1'],
        ['MYB','NFIB'],
        ['MYB','NHSL1'],
        ['MYB','QKI'],
        ['MYB','SMR3B'],
        ['MYBL1','ACTN1'],
        ['MYBL1','NFIB'],
        ['MYBL1','RAD51B'],
        ['MYC','BCL2'],
        ['MYC','BCL6'],
        ['MYC','IGH@'],
        ['MYC','IGK@'],
        ['MYC','IGL@'],
        ['MYC','MAX'],
        ['MYC','MYCN'],
        ['MYC','SOCS1'],
        ['MYH11','CBFB'],
        ['MYH9','ALK'],
        ['MYH9','EIF3D'],
        ['MYH9','USP6'],
        ['MYLK','BRAF'],
        ['MYO16','PTK2'],
        ['MYO18A','PDGFRB'],
        ['MYO5A','ALK'],
        ['MYO5A','MERTK'],
        ['MYO5A','NTRK3'],
        ['MYO5A','RET'],
        ['MYO5A','ROS1'],
        ['MYOD1','PIK3CA'],
        ['MYST3','CREBBP'],
        ['MYST3','NCOA2'],
        ['MYST3','P300'],
        ['NAB2','STAT6'],
        ['NACC2','NTRK2'],
        ['NAGLU','IKZF3'],
        ['NCAM2','SGCZ'],
        ['NCL','RB1'],
        ['NCOA2','ETV4'],
        ['NCOA2','ETV6'],
        ['NCOA4','RET'],
        ['NCOR1','LYN'],
        ['NCOR1','MAP2K1'],
        ['NCOR2','NTRK1'],
        ['ND3','ND5'],
        ['ND4','PLAG1'],
        ['ND4L','ND6'],
        ['NDE1','PDGFRB'],
        ['NDEL1','PDGFRB'],
        ['NDRG1','ERG'],
        ['NDRG1','PLAG1'],
        ['NDUFAF2','MAST4'],
        ['NEAT1','TFE3'],
        ['NEAT1','TFEB'],
        ['NF1','AARSD1'],
        ['NF1','PTPN11'],
        ['NF1','TP53'],
        ['NF2','SPATA13'],
        ['NFASC','NTRK1'],
        ['NFATC3','PLA2G15'],
        ['NFIA','CBFA2T3'],
        ['NFIB','AIG1'],
        ['NFIB','PLAG1'],
        ['NFIX','PKN1'],
        ['NIN','PDGFRB'],
        ['NIPBL','ERG'],
        ['NIPBL','HOXB9'],
        ['NIPBL','NACC1'],
        ['NONO','TFE3'],
        ['NOTCH1','FBXW7'],
        ['NOTCH1','PRKCA'],
        ['NOTCH2','MIR143'],
        ['NOTCH3','SETBP1'],
        ['NPC1','MELK'],
        ['NPC2','HE1'],
        ['NPM1','ALK'],
        ['NPM1','CCDC28A'],
        ['NPM1','HAUS1'],
        ['NPM1','MLF1'],
        ['NPM1','RARA'],
        ['NPM1','TYK2'],
        ['NR1D1','MAML1'],
        ['NR4A3','EWSR1'],
        ['NR4A3','PGR'],
        ['NRAS','FLT3'],
        ['NRIP1','MECOM'],
        ['NRIP1','RSPO2'],
        ['NSD1','ANKRD28'],
        ['NSD1','NUP98'],
        ['NSD3','NUP98'],
        ['NSL1','ZDBF2'],
        ['NTF3','PLAG1'],
        ['NTRK1','KHDRBS1'],
        ['NTRK1','NGF'],
        ['NTRK2','TRAF2'],
        ['NTRK3','ETV6'],
        ['NTRK3','SCAPER'],
        ['NUMA1','RARA'],
        ['NUMA1','SFMBT1'],
        ['NUP107','LGR5'],
        ['NUP160','SLC43A3'],
        ['NUP214','ABL'],
        ['NUP214','ABL1'],
        ['NUP214','RAC1'],
        ['NUP214','XKR3'],
        ['NUP93','CYB5B'],
        ['NUP98','ADD3'],
        ['NUP98','BPTF'],
        ['NUP98','CCDC28A'],
        ['NUP98','DDX10'],
        ['NUP98','FN1'],
        ['NUP98','HMGB3'],
        ['NUP98','HOXA10'],
        ['NUP98','HOXA11'],
        ['NUP98','HOXA13'],
        ['NUP98','HOXA9'],
        ['NUP98','HOXB3'],
        ['NUP98','HOXB4'],
        ['NUP98','HOXC11'],
        ['NUP98','HOXC13'],
        ['NUP98','HOXD11'],
        ['NUP98','HOXD13'],
        ['NUP98','IQCG'],
        ['NUP98','JARID1A'],
        ['NUP98','KDM5A'],
        ['NUP98','KMT2A'],
        ['NUP98','LEDGF'],
        ['NUP98','NSD1'],
        ['NUP98','NSD3'],
        ['NUP98','PDGFRA'],
        ['NUP98','PHF23'],
        ['NUP98','PMX1'],
        ['NUP98','POU1F1'],
        ['NUP98','PRRX2'],
        ['NUP98','PSIP1'],
        ['NUP98','RAP1GDS1'],
        ['NUP98','RARA'],
        ['NUP98','RARG'],
        ['NUP98','TOP1'],
        ['NUTM1','BRD4'],
        ['NUTM2A','CIC'],
        ['OAZ1','CSNK1G2'],
        ['OBFC2A','RARA'],
        ['OFD1','JAK2'],
        ['OGT','FOXO1'],
        ['OGT','FOXO3'],
        ['OPTN','BRAF'],
        ['OR51E2','ETV1'],
        ['OSBPL1A','BRAF'],
        ['OSBPL9','BRAF'],
        ['OSM','LIF'],
        ['P16','CDKN2A'],
        ['P2RY8','CRLF2'],
        ['P4HA2','TBCK'],
        ['PAFAH1B1','USP6'],
        ['PAFAH1B2','FOXR1'],
        ['PAFAH1B3','CLK2'],
        ['PAK1','STAT5'],
        ['PAN2','PAN3'],
        ['PAN3','PSMA2'],
        ['PARD3','ALK'],
        ['PARP14','TFE3'],
        ['PARP15','AHSG'],
        ['PAX3','FKHR'],
        ['PAX3','FOXO1'],
        ['PAX3','FOXO1A'],
        ['PAX3','MAML3'],
        ['PAX3','NCOA1'],
        ['PAX3','NCOA2'],
        ['PAX3','PAX7'],
        ['PAX3','WWTR1'],
        ['PAX5','AUTS2'],
        ['PAX5','BSAP'],
        ['PAX5','CREB3L1'],
        ['PAX5','DACH1'],
        ['PAX5','DACH2'],
        ['PAX5','ELN'],
        ['PAX5','ESRRB'],
        ['PAX5','ETV6'],
        ['PAX5','HIPK1'],
        ['PAX5','IGH@'],
        ['PAX5','JAK2'],
        ['PAX5','KIAA1549L'],
        ['PAX5','KIDINS220'],
        ['PAX5','MYO1G'],
        ['PAX5','NCOA6'],
        ['PAX5','PDL1'],
        ['PAX5','PML'],
        ['PAX5','POM121'],
        ['PAX7','FKHR'],
        ['PAX7','FOXO1'],
        ['PAX7','FOXO1A'],
        ['PAX8','GLIS1'],
        ['PAX8','GLIS3'],
        ['PAX8','PPAR'],
        ['PAX8','PPARG'],
        ['PAX8','PPARG1'],
        ['PBX1','TCF3'],
        ['PBX3','MEIS1'],
        ['PC3','AR'],
        ['PCM1','JAK2'],
        ['PCM1','PDGFRB'],
        ['PD1','CD8'],
        ['PD1','PDL1'],
        ['PDCD1LG2','PDL2'],
        ['PDE4D','CCNB1'],
        ['PDE4DIP','NOTCH2'],
        ['PDGFB','COL1A1'],
        ['PDGFRA','ABL'],
        ['PDGFRA','FIP1L1'],
        ['PDGFRA','MACROD2'],
        ['PDGFRA','PDGFRB'],
        ['PDGFRB','ABL'],
        ['PDGFRB','CEV14'],
        ['PDHB','P80'],
        ['PDK1','S6K1'],
        ['PDPN','PRKCB'],
        ['PDZRN3','RAF1'],
        ['PEX14','JAK2'],
        ['PEX5','LPL'],
        ['PGAP3','SRPK1'],
        ['PHF1','BRD8'],
        ['PHF1','JAZF1'],
        ['PHF1','TFE3'],
        ['PHF19','FOXP2'],
        ['PHF20','NTRK1'],
        ['PHOSPHO2','KLHL23'],
        ['PI3K','PTEN'],
        ['PI3K','STAT'],
        ['PICALM','MLLT10'],
        ['PIEZO2','MTA1'],
        ['PIK3R1','PTEN'],
        ['PKD1','TSC2'],
        ['PLAG1','HAS2'],
        ['PLAG1','HMGA2'],
        ['PLAG1','RAD51L1'],
        ['PLAG1','YWHAZ'],
        ['PLB1','ALK'],
        ['PLCG1','KDR'],
        ['PLEC1','ENPP2'],
        ['PLEKHA6','NTRK3'],
        ['PLEKHA7','ALK'],
        ['PLEKHG3','TCF4'],
        ['PLEKHH2','ALK'],
        ['PLEKHM2','ALK'],
        ['PLXNB1','PRKAR2A'],
        ['PLZF','RARA'],
        ['PMAIP1','MALT1'],
        ['PML','NTRK2'],
        ['PML','RARA'],
        ['PMP22','ELOVL5'],
        ['PMP22','RAI1'],
        ['POLA2','CDC42EP2'],
        ['POMK','NRG1'],
        ['POR','BRAF'],
        ['PP1A','PML'],
        ['PPFIBP2','RET'],
        ['PPL','NTRK1'],
        ['PPM1B','ALK'],
        ['PPP1CB','ALK'],
        ['PPP1CB','YPEL5'],
        ['PPP1R10','TFEB'],
        ['PPP1R1B','STARD3'],
        ['PPP6R3','USP6'],
        ['PRB3','ZNF217'],
        ['PRCC','MITF'],
        ['PRCC','TFE3'],
        ['PRDM1','BLIMP1'],
        ['PRDM16','SKI'],
        ['PRDX1','AKR1A1'],
        ['PRISM','SRM'],
        ['PRKAR1A','ALK'],
        ['PRKAR1A','RARA'],
        ['PRKAR1A','RET'],
        ['PRKAR1B','BRAF'],
        ['PRKAR2A','ALK'],
        ['PRKG2','PDGFRB'],
        ['PRLR','TERT'],
        ['PROM1','CD133'],
        ['PRPSAP1','NTRK3'],
        ['PRR15L','RSPO2'],
        ['PRRC2B','DEK'],
        ['PTBP1','MAML2'],
        ['PTBP1','USP6'],
        ['PTCH1','GLI1'],
        ['PTEN','COL17A1'],
        ['PTGES3','PTPRB'],
        ['PTGES3','RARG'],
        ['PTK2','THOC2'],
        ['PTMS','AP1G1'],
        ['PTPLB','RSRC1'],
        ['PTPN11','NF1'],
        ['PTPRC','CPN2'],
        ['PTPRK','RSPO3'],
        ['PTPRZ1','ETV1'],
        ['PTPRZ1','MET'],
        ['PTPRZ1','NFAM1'],
        ['PU.1','STAT5'],
        ['PUF60','SCRIB'],
        ['PUF60','TYW1'],
        ['PVT1','CHD7'],
        ['PVT1','NBEA'],
        ['PVT1','NSMCE2'],
        ['PVT1','WWOX'],
        ['QKI','RAF1'],
        ['R3HDM2','NFE2'],
        ['RAB10','ALK'],
        ['RAB2A','PLAG1'],
        ['RAB3GAP1','HRAS'],
        ['RABGAP1L','NTRK1'],
        ['RAC1','COL12A1'],
        ['RAD50','PDLIM4'],
        ['RAD51AP1','DYRK4'],
        ['RAD51B','OPHN1'],
        ['RAD51B','PLAG1'],
        ['RAD51L1','HMGA2'],
        ['RAF1','QKI'],
        ['RAG1','RAG2'],
        ['RANBP1','ALK'],
        ['RANBP2','ABL1'],
        ['RANBP2','ALK'],
        ['RANTES','CCL5'],
        ['RARA','PLZF'],
        ['RARA','PML'],
        ['RARA','RXRA'],
        ['RARS','MAD1L1'],
        ['RASAL2','RET'],
        ['RASGRP1','ATP1A1'],
        ['RASSF1','PDZRN3'],
        ['RB1','RNASEH2B'],
        ['RBBP1','ARID4A'],
        ['RBM10','TFE3'],
        ['RBM15','MKL1'],
        ['RBM47','CDK12'],
        ['RBMX','TFE3'],
        ['RBPMS','MET'],
        ['RBPMS','NTRK3'],
        ['RCC1','ABHD12B'],
        ['RCSD1','ABL1'],
        ['RCSD1','ABL2'],
        ['REEP1','POLR1A'],
        ['REL','BCL11A'],
        ['REL','COMMD1'],
        ['RET','CCDC6'],
        ['RET','ELE1'],
        ['RET','KIF5B'],
        ['RET','MEN2A'],
        ['RET','MEN2B'],
        ['RET','NTRK1'],
        ['RET','PTC3'],
        ['RET','SPECC1L'],
        ['RFC4','LPP'],
        ['RFG7','RET'],
        ['RGAG1','BCOR'],
        ['RHOT1','MIRO1'],
        ['RIF1','PKD1L1'],
        ['RLN1','RLN2'],
        ['RMST','TERT'],
        ['RNF121','FOLR2'],
        ['RNF130','BRAF'],
        ['RNF213','ALK'],
        ['RNF213','ALO17'],
        ['RNF213','SLC26A11'],
        ['RNF43','SUPT4H1'],
        ['RNPC3','JAK2'],
        ['ROBO1','BRAF'],
        ['ROR1','DNAJC6'],
        ['ROS1','ADGRG6'],
        ['ROS1','ALK'],
        ['ROS1','CCDC30'],
        ['ROS1','EZR'],
        ['ROS1','GOPC'],
        ['ROS1','RET'],
        ['ROS1','SDC4'],
        ['ROS1','SLC34A2'],
        ['RPN1','EVI1'],
        ['RPS6KB1','VMP1'],
        ['RPS9','RPL21'],
        ['RRAGB','OPHN1'],
        ['RRBP1','RAF1'],
        ['RREB1','MKL2'],
        ['RREB1','MRTFB'],
        ['RSF1','TENM4'],
        ['RSPO2','EMC2'],
        ['RSPO3','LGR4'],
        ['RUFY2','RET'],
        ['RUNX1','AML1'],
        ['RUNX1','CBFA2T1'],
        ['RUNX1','CBFA2T2'],
        ['RUNX1','CBFA2T3'],
        ['RUNX1','ETV6'],
        ['RUNX1','EVI1'],
        ['RUNX1','FNBP1'],
        ['RUNX1','GRIK2'],
        ['RUNX1','MECOM'],
        ['RUNX1','MTG16'],
        ['RUNX1','MTG8'],
        ['RUNX1','PDCD6'],
        ['RUNX1','PRDM16'],
        ['RUNX1','PRPF38A'],
        ['RUNX1','RUNX1T1'],
        ['RUNX1','TACC1'],
        ['RUNX1','USP42'],
        ['RUNX1T1','RUNX1'],
        ['RUNX2','USP6'],
        ['RYR2','SHANK3'],
        ['S100A8','S100A9'],
        ['SAMD1','CDC42EP5'],
        ['SAMD12','EXT1'],
        ['SAMD5','SASH1'],
        ['SAR1A','USP6'],
        ['SART3','PDGFRB'],
        ['SAV1','GYPB'],
        ['SBF1','PKHD1L1'],
        ['SCG5','GREM1'],
        ['SCNN1A','TNFRSF1A'],
        ['SCYL3','NTRK1'],
        ['SDC4','NRG1'],
        ['SDC4','ROS1'],
        ['SDK1','AMACR'],
        ['SEC16A','NOTCH1'],
        ['SEC22B','NOTCH2'],
        ['SEC31A','ALK'],
        ['SEC31A','JAK2'],
        ['SEC31L1','ALK'],
        ['SEC61G','EGFR'],
        ['SEL1L','NTRK1'],
        ['SEMA3A','PLXND1'],
        ['SENP1','MESDC2'],
        ['SEPT7','BRAF'],
        ['SEPT9','ABL1'],
        ['SERPINE1','FOSB'],
        ['SET','NUP214'],
        ['SETD1B','FOSB'],
        ['SETD1B','GTF2H3'],
        ['SETD1B','TFE3'],
        ['SETD2','NF1'],
        ['SFPQ','ABL1'],
        ['SFPQ','TFE3'],
        ['SH3PXD2A','HTRA1'],
        ['SHANK3','RYR2'],
        ['SHIP1','ATG16L1'],
        ['SIRPG','WWOX'],
        ['SLC12A2','ROS1'],
        ['SLC1A2','EAAT2'],
        ['SLC25A36','RET'],
        ['SLC2A1','FAF1'],
        ['SLC2A11','MIF'],
        ['SLC34A2','ROS1'],
        ['SLC35A3','HIAT1'],
        ['SLC38A2','USP6'],
        ['SLC39A8','RET'],
        ['SLC3A2','NRG1'],
        ['SLC44A1','PRKCA'],
        ['SLC45A2','AMACR'],
        ['SLC45A3','BRAF'],
        ['SLC45A3','ELK4'],
        ['SLC45A3','ERG'],
        ['SLC45A3','ETV4'],
        ['SLC45A3','ETV5'],
        ['SLC6A3','AGBL3'],
        ['SLMAP','ALK'],
        ['SLMAP','NTRK2'],
        ['SMARCA4','BRG1'],
        ['SMARCB1','INI1'],
        ['SMARCB1','SMARCA4'],
        ['SMARCD2','BAF60B'],
        ['SMC3','MXI1'],
        ['SMOC2','PROX1'],
        ['SMPD3','ALK'],
        ['SND1','BRAF'],
        ['SNX2','ABL1'],
        ['SOCS5','ALK'],
        ['SOS1','ALK'],
        ['SOX2','CEN3'],
        ['SOX5','RAF1'],
        ['SP1','SP3'],
        ['SP3','PTK2'],
        ['SPAG9','JAK2'],
        ['SPARC','PDGFRB'],
        ['SPARC','USP6'],
        ['SPAST','SLC30A6'],
        ['SPECC1L','ALK'],
        ['SPECC1L','NTRK3'],
        ['SPOP','ERG'],
        ['SPTAN1','ABL1'],
        ['SPTBN1','ALK'],
        ['SPTBN1','FLT3'],
        ['SQSTM1','ALK'],
        ['SQSTM1','CDX1'],
        ['SQSTM1','NTRK1'],
        ['SQSTM1','NTRK2'],
        ['SQSTM1','NTRK3'],
        ['SQSTM1','NUP214'],
        ['SRBD1','ALK'],
        ['SRD5A2','ALK'],
        ['SRF','CITED1'],
        ['SRF','E2F1'],
        ['SRF','FOXO1'],
        ['SRF','ICA1L'],
        ['SRF','NCOA1'],
        ['SRF','NCOA2'],
        ['SRF','RELA'],
        ['SRF','STAT6'],
        ['SRGAP3','RAF1'],
        ['SS18','NEDD4'],
        ['SS18','POU5F1'],
        ['SS18','SLC5A11'],
        ['SS18','SSX1'],
        ['SS18','SSX2'],
        ['SS18','SSX2B'],
        ['SS18','SSX4'],
        ['SS18','USP6'],
        ['SS18','ZBTB7A'],
        ['SS18L1','SSX1'],
        ['SSBP2','CSF1R'],
        ['SSBP2','FER'],
        ['SSBP2','JAK2'],
        ['STARD9','BRAF'],
        ['STAT3','JAK2'],
        ['STAT5','PU.1'],
        ['STAT5B','RARA'],
        ['STAT6','NAB2'],
        ['STIL','TAL1'],
        ['STIP1','CREB3L1'],
        ['STK11','LKB1'],
        ['STMN1','SPI1'],
        ['STRBP','JAK2'],
        ['STRN','ALK'],
        ['STRN','PDGFRA'],
        ['SULF2','ZNF217'],
        ['SYN2','PPARG'],
        ['SYN3','ALK'],
        ['SYNCRIP','SNX14'],
        ['SYNRG','ZNF384'],
        ['SYT4','SSX1'],
        ['SYT4','SSX2'],
        ['SYT4','SSX4'],
        ['SYT8','TNNI2'],
        ['TACC2','PPAPDC1A'],
        ['TACR1','ALK'],
        ['TACSTD1','MSH2'],
        ['TACSTD2','OMA1'],
        ['TADA2A','MEF2B'],
        ['TAF15','NR4A3'],
        ['TAF15','RBP56'],
        ['TAF15','ZNF384'],
        ['TAF2N','NOR1'],
        ['TAF2N','TEC'],
        ['TAF4B','WDR1'],
        ['TAOK1','PCGF2'],
        ['TAP1','TAP2'],
        ['TATDN1','GSDMB'],
        ['TAX1BP1','AHCY'],
        ['TAZ','CAMTA1'],
        ['TBC1D15','RAB21'],
        ['TBCK','P4HA2'],
        ['TBL1XR1','HMGA1'],
        ['TBL1XR1','PIK3CA'],
        ['TBL1XR1','RARA'],
        ['TBL1XR1','RARB'],
        ['TBL1XR1','ROS1'],
        ['TBL1XR1','TP63'],
        ['TBLR1','RARA'],
        ['TCEA1','PLAG1'],
        ['TCF12','NR4A3'],
        ['TCF12','TEC'],
        ['TCF20','SPBP'],
        ['TCF3','FLI1'],
        ['TCF3','HLF'],
        ['TCF3','ID3'],
        ['TCF3','PBX1'],
        ['TCF3','TEF'],
        ['TCF3','ZNF384'],
        ['TCF7','SPI1'],
        ['TEAD1','NCOA2'],
        ['TERF2','JAK2'],
        ['TERT','CLPTM1L'],
        ['TEX2','PECAM1'],
        ['TFE3','ASPL'],
        ['TFE3','ASPSCR1'],
        ['TFE3','NONO'],
        ['TFEB','VEGFA'],
        ['TFG','ADGRG7'],
        ['TFG','ALK'],
        ['TFG','ALK5'],
        ['TFG','FGFR1'],
        ['TFG','GPR128'],
        ['TFG','MET'],
        ['TFG','NOR1'],
        ['TFG','NR4A3'],
        ['TFG','NTRK1'],
        ['TFG','RARA'],
        ['TFG','RET'],
        ['TFG','ROS1'],
        ['TFG','TEC'],
        ['TGFBR3','MGEA5'],
        ['TGFBR3','PLAG1'],
        ['THADA','IGF2BP3'],
        ['THBS1','ADGRF5'],
        ['THBS1','ALK'],
        ['TIAM1','NRIP1'],
        ['TIMM23','ARHGAP32'],
        ['TIMP3','ALK'],
        ['TIMP3','RET'],
        ['TIMP3','ROS1'],
        ['TJP1','ROS1'],
        ['TLX1','HOX11'],
        ['TLX1','TLX3'],
        ['TLX3','HOX11L2'],
        ['TMBIM4','MSRB3'],
        ['TMED2','ALK'],
        ['TMED6','COG8'],
        ['TMEM123','MMP27'],
        ['TMEM135','CCDC67'],
        ['TMEM66','BAIAP2'],
        ['TMPRSS2','ERG'],
        ['TMPRSS2','ETV1'],
        ['TMPRSS2','ETV4'],
        ['TMPRSS2','ETV5'],
        ['TNC','USP6'],
        ['TNFAIP3','A20'],
        ['TNIP2','ALK'],
        ['TNRC18','RARA'],
        ['TNRC6B','ABL1'],
        ['TNS3','EXOC6B'],
        ['TNS3','MAP3K3'],
        ['TNXA','TNXB'],
        ['TOM1L2','BRAF'],
        ['TOP1','NUP98'],
        ['TORC1','MAML2'],
        ['TP53','BAP1'],
        ['TP53','IDH1'],
        ['TP53','KMT2C'],
        ['TPCN2','RSF1'],
        ['TPCN2','YAP1'],
        ['TPM3','ALK'],
        ['TPM3','JAK2'],
        ['TPM3','NTRK1'],
        ['TPM3','PDGFRB'],
        ['TPM3','ROS1'],
        ['TPM3','TRKA'],
        ['TPM4','ALK'],
        ['TPO','EPO'],
        ['TPR','ALK'],
        ['TPR','FGFR1'],
        ['TPR','MET'],
        ['TPR','NTRK1'],
        ['TPR','RET'],
        ['TPR','ROS1'],
        ['TPS1','TPS2'],
        ['TRA@','SALL2'],
        ['TRA2B','DNAH5'],
        ['TRAC','TRDC'],
        ['TRAF1','ALK'],
        ['TRAF3','BCL2'],
        ['TRB@','NOTCH1'],
        ['TRC8','RNF139'],
        ['TRDV1','BCL11B'],
        ['TRGV1','TRGV8'],
        ['TRIM24','BRAF'],
        ['TRIM24','BRPF1'],
        ['TRIM24','RET'],
        ['TRIM27','RET'],
        ['TRIM33','ALK'],
        ['TRIM33','NTRK1'],
        ['TRIM33','RAF1'],
        ['TRIM33','RET'],
        ['TRIM4','BRAF'],
        ['TRIM4','MET'],
        ['TRIM52','RACK1'],
        ['TRIO','TERT'],
        ['TRIP11','BRAF'],
        ['TRIP11','FLT3'],
        ['TRIP12','NPM1'],
        ['TRMT','LYL1'],
        ['TRMT11','GRIK2'],
        ['TRPS1','LASP1'],
        ['TRPS1','PLAG1'],
        ['TSC1','TSC2'],
        ['TSC2','PKD1'],
        ['TSC2','RNF216'],
        ['TSLP','CRLF2'],
        ['TSNAX','DISC1'],
        ['TSPAN4','CD151'],
        ['TSPAN7','MRX58'],
        ['TTC6','MIPOL1'],
        ['TTL','ETV6'],
        ['TTYH3','BRAF'],
        ['TUT1','ETV5'],
        ['TXNDC5','BMP6'],
        ['UBA2','PDCD2L'],
        ['UBA2','WTIP'],
        ['UBAP1','TGM7'],
        ['UBE2L3','KRAS'],
        ['UBR5','ZNF423'],
        ['UBTF','ETV4'],
        ['UEVLD','RET'],
        ['UNC5D','NRG1'],
        ['USP15','CNTN1'],
        ['USP32','CCDC49'],
        ['USP42','RUNX1'],
        ['USP6','CDH11'],
        ['USP6','LUM'],
        ['USP6','TRE17'],
        ['USP9X','DDX3X'],
        ['USP9X','ERAS'],
        ['USP9X','USP6'],
        ['UTP6','CRLF3'],
        ['VAMP2','NRG1'],
        ['VAPB','IKZF3'],
        ['VAV1','MYO1F'],
        ['VAV1','S100A7'],
        ['VAV1','THAP4'],
        ['VCAN','HAPLN1'],
        ['VCL','ALK'],
        ['VCL','RET'],
        ['VCP','TFE3'],
        ['VEGF','A121'],
        ['VEGFR2','KDR'],
        ['VGLL2','CITED2'],
        ['VGLL2','NCOA2'],
        ['VGLL4','PRKG1'],
        ['VIM','KMT2A'],
        ['VIM','RET'],
        ['VIT','ALK'],
        ['VPS41','BRAF'],
        ['VTI1A','CFAP46'],
        ['VTI1A','TCF4'],
        ['VTI1A','TCF7L2'],
        ['VWA2','TCF7L2'],
        ['WAC','ANKRD26'],
        ['WASH1','NPAS3'],
        ['WHSC1','MMSET'],
        ['WHSC1L1','NCOA2'],
        ['WNK1','B4GALNT3'],
        ['WNK1','ROS1'],
        ['WNT10B','PORCN'],
        ['WRN','NRG1'],
        ['WT1','TBP'],
        ['WWC1','ADRBK2'],
        ['WWTR1','ACTL6A'],
        ['WWTR1','CAMTA1'],
        ['WWTR1','FOSB'],
        ['WWTR1','MAML2'],
        ['WWTR1','NUTM1'],
        ['XCL1','GPC3'],
        ['YAP','MAMLD1'],
        ['YAP','TFE3'],
        ['YAP1','FAM118B'],
        ['YAP1','KMT2A'],
        ['YAP1','MAML2'],
        ['YAP1','MAMLD1'],
        ['YAP1','NUTM1'],
        ['YAP1','SS18'],
        ['YAP1','TAZ'],
        ['YAP1','TFE3'],
        ['YPEL5','PPP1CB'],
        ['YTHDF2','FOXR1'],
        ['YWHAE','NTRK3'],
        ['YWHAE','NUTM2B'],
        ['YWHAE','ROS1'],
        ['YWHAZ','PLAG1'],
        ['YY1','EWSR1'],
        ['ZBTB10','NTRK3'],
        ['ZBTB16','ABL1'],
        ['ZBTB16','RARA'],
        ['ZBTB20','JAK2'],
        ['ZBTB41','RET'],
        ['ZBTB46','WFDC13'],
        ['ZC3H7','BCOR'],
        ['ZC3H7B','BCOR'],
        ['ZC3HAV1','ABL2'],
        ['ZC3HAV1L','CHMP1A'],
        ['ZCCHC8','ROS1'],
        ['ZDHHC5','GPR137'],
        ['ZEB1','ZEB2'],
        ['ZEB2','ALK'],
        ['ZEB2','BCL11B'],
        ['ZEB2','PLAG1'],
        ['ZFP36','FOSB'],
        ['ZFPM2','ELF5'],
        ['ZKSCAN1','MET'],
        ['ZMIZ1','ABL1'],
        ['ZMIZ1','PRR12'],
        ['ZMYM2','FGFR1'],
        ['ZMYM2','FLT3'],
        ['ZMYM4','OPRD1'],
        ['ZMYND11','MBTD1'],
        ['ZNF198','FGFR1'],
        ['ZNF207','HER2'],
        ['ZNF273','DGKA'],
        ['ZNF362','TCF4'],
        ['ZNF384','TCF3'],
        ['ZNF451','BAG2'],
        ['ZNF532','NUTM1'],
        ['ZNF56','KMT2A'],
        ['ZNF611','IL1RAPL1'],
        ['ZNF704','NRG1'],
        ['ZSWIM5','TP53BP2'],
        ['CBFB','MYH11'],
        ['CREBBP','ZNF384'],
        ['DDX5','GLUL'],
        ['DEK','NUP214'],
        ['EBF1','PDGFRB'],
        ['EP300','ZNF384'],
        ['ETV6','ARNT'],
        ['ETV6','LOH12CR1'],
        ['ETV6','NCOA2'],
        ['ETV6','RUNX1'],
        ['EWSR1','ZNF384'],
        ['EZR','ROS1'],
        ['FOSB','UBC'],
        ['FUS','ZNF362'],
        ['KMT2A','AFF1'],
        ['KMT2A','CARS'],
        ['KMT2A','ELL'],
        ['KMT2A','MAML2'],
        ['KMT2A','MLLT1'],
        ['KMT2A','MLLT10'],
        ['KMT2A','MLLT11'],
        ['KMT2A','MLLT3'],
        ['KMT2A','MLLT4'],
        ['KMT2A','MLLT6'],
        ['KMT2A','MYO1F'],
        ['KMT2A','SEPT5'],
        ['KMT2A','SEPT6'],
        ['KMT2A','USP2'],
        ['MEF2D','BCL9'],
        ['MEF2D','DAZAP1'],
        ['MEF2D','FOXJ2'],
        ['MEF2D','HNRNPM'],
        ['MEF2D','HNRNPUL1'],
        ['MEF2D','STAT6'],
        ['NAP1L1','MLLT10'],
        ['NUP98','NSD1'],
        ['PAX5','AIF1L'],
        ['PAX5','AUTS2'],
        ['PAX5','CBFA2T2'],
        ['PAX5','CBFA2T3'],
        ['PAX5','CPSF7'],
        ['PAX5','DAK'],
        ['PAX5','ELN'],
        ['PAX5','EMSY'],
        ['PAX5','ESRRB'],
        ['PAX5','FBRS'],
        ['PAX5','FBRSL1'],
        ['PAX5','FKBP15'],
        ['PAX5','GNAI1'],
        ['PAX5','NOL4L'],
        ['PAX5','NSD1'],
        ['PAX5','ZCCHC7'],
        ['PICALM','MLLT10'],
        ['PML','RARA'],
        ['PTMA','CXCR4'],
        ['FMR1','ALK'],
        ['ALK','LRRN2'],
        ['LTBP1','ALK'],
        ['HIP1','ALK'],
        ['NPM1','RPP30'],
        ['NPM1','SETBP1'],
        ['RLF','MYCL'],
        ['MYB','GATA1'],
        ['GOLGA4','JAK2'],
        ['RAB3IP','HMGA2'],
        ['OLFM4','RET'],
        ['COL6A3','PDGFD'],
        ['EMILIN2','PDGFD'],
        ['KANSL1','MTCP1'],
        ['KANSL1','CMC4'],
        ['FOXK1','GRHL1'],
        ['GPS2','GRHL1'],
        ['HOOK3','FGFR1'],
        ['ALK','KANK4'],
        ['EIF4A2','PLAG1'],
        ['SPON1','TRIM29'],
        ['C19ORF48','ETV4'],
        ['RYBP','FOXP1'],
        ['CTNNB1','USP6'],
        ['ING2','WTAP'],
        ['MAP4','RAF1'],
        ['PLAG1','COL1A2'],
        ['PLAG1','COL3A1'],
        ['PLAG1','HAS2'],
        ['PLAG1','RAD51L1'],
        ['PLAG1','RAB2A'],
        ['PLAG1','BOC'],
        ['PLAG1','CHCHD7'],
        ['PLAG1','SRSF3'],
        ['PLAG1','HNRNPC'],
        ['PLAG1','PCMTD1'],
        ['PLAG1','EEF1A1'],
        ['PLAG1','YWHAZ'],
        ['PLAG1','CTDSP2'],
        ['PLAG1','PPP2R2A'],
        ['UBTF','ATXN7L3'],
        ['PLAG1','DDX6'],
        ['PLAG1','KLF10'],
        ['PLAG1','KANSL1L'],
        ['FOXK1','GRHL1'],
        ['GPS2','GRHL1'],
        ['ALK','KANK4'],
        ['EIF4A2','PLAG1'],
        ['SPON1','TRIM29'],
        ['C19ORF48','ETV4'],
        ['RYBP','FOXP1'],
        ['EIF4A2','PLAG1'],
        ['ZNF532','NUTM1'],
        ['NIPBL','NACC1'],
        ['SPON1','TRIM29'],
        ['AGK','BRAF'],
        ['FMR1','ALK'],
        ['PGAP3','SRPK1'],
        ['CSF1','CD96'],
        ['FOXK1','GRHL1'],
        ['FOXK1','GRHL2'],
        ['GPS2','GRHL1'],
        ['GPS2','GRHL2'],
        ['GPS2','GRHL3'],
        ['NPM1','CCDC28A'],
        ['MAN2A1','FER'],
        ['SLC45A2','AMACR'],
        ['TRMT11','GRIK2'],
        ['CCNH','C5ORF30'],
        ['MTOR','TP53BP1'],
        ['TMEM135','CCDC67'],
        ['RAB3IP','HMGA2'],
        ['AGK','BRAF'],
        ['RUNX1','MECOM'],
        ['RUNX1','RUNX1T1'],
        ['SET','NUP214'],
        ['SMARCA2','NR4A3'],
        ['STIL','TAL1'],
        ['TAF15','ZNF384'],
        ['TCF3','HLF'],
        ['TCF3','PBX1'],
        ['TCF3','ZNF362'],
        ['TCF3','ZNF384'],
        ['TCF4','ZNF384'],
        ['TRA2B','MECOM'],
        ['ZNF292','PNRC1'],
        ['ARHGEF3','CACNA2D2'],
        ['CBFA2T3','GLIS2'],
        ['CBFB','MYH11'],
        ['CLASP2','IL17RD'],
        ['CNTN4','RHOA'],
        ['CTNNBIP1','CLSTN1'],
        ['DDX5','GLUL'],
        ['EBF1','PDGFRB'],
        ['EP300','BCORL1'],
        ['EWSR1','ELF5'],
        ['FOSB','UBC'],
        ['FYN','TRAF3IP2'],
        ['GPR37L1','PRKCA'],
        ['ITPR1-AS1','NR2C2'],
        ['ITPR1-DT','NR2C2'],
        ['ITPR1','NR2C2'],
        ['ITPR2','FSTL4'],
        ['JAZF1','BCORL1'],
        ['KAT6A','CREBBP'],
        ['KHDRBS1','LCK'],
        ['KMT2A','AFF1'],
        ['KMT2A','CARS'],
        ['KMT2A','ELL'],
        ['KMT2A','MAML2'],
        ['KMT2A','MLLT1'],
        ['KMT2A','MLLT10'],
        ['KMT2A','MLLT11'],
        ['KMT2A','MLLT3'],
        ['KMT2A','MLLT4'],
        ['KMT2A','MLLT6'],
        ['KMT2A','MYO1F'],
        ['KMT2A','SEPT5'],
        ['KMT2A','SEPT6'],
        ['KMT2A','USP2'],
        ['MED14','HOXA9'],
        ['MFGE8','HAPLN3'],
        ['MKRN1','BRAF'],
        ['NR2C2','CFAP410'],
        ['PTMA','CXCR4'],
        ['SMARCA2','NR4A3'],
        ['STAG2','AFF2'],
        ['TNC','PDGFD'],
        ['TRAK1','TIMP4'],
        ['GHR','ALK'],
        ['CRTC1','MAML2'],
        ['ZNF292','B3GAT2'],
        ['ZNF292','CGA'],
        ['ZNF292','EYS'],
        ['ZNF292','FIG4'],
        ['ZNF292','GRM4'],
        ['ZNF292','MAP3K4'],
        ['ZNF292','PNRC1'],
        ['ZNF292','PRSS54'],
        ['ZNF292','PTPRO'],
        ['ZNF621','LARS2'],
        ['ERBIN','RASGRF2'],
        ['ATP2B4','RASGRF2'],
        ['KDM2A','WWTR1'],
        ['CLTC','FOSB'],
        ['YAP1','NUTM1'],
        ['RREB1','MKL2'],
        ['WWTR1','MAML2'],
        ['WWTR1','FOSB'],
        ['KIAA1671','NF2'],
        ['KLHL7','BRAF'],
        ['NF2','PIAS1'],
        ['NF2','CABP7'],
        ['YWHAE','NUTM2A'],
        ['YWHAE','NUTM2B'],
        ['YAP1','KDM2B'],
        ['CHST11','KDM2B'],
        ['FOSB','ACTB'],
        ['BCOR','CCNB3'],
        ['RNF213','SLC26A11'],
        ['HOOK3','FGFR1']





]

    file(os.path.join(options.output_directory,'known_genes.txt'),'w').write("")
    file(os.path.join(options.output_directory,'known.txt'),'w').write("")

    data = fusions.get(options.organism.lower(),[])
    data2 = []
    if data:

        #file_symbols = os.path.join(options.output_directory,'genes_symbols.txt')
        file_symbols = os.path.join(options.output_directory,'synonyms.txt')
        loci = symbols.generate_loci(file_symbols)
        genes = symbols.read_genes_symbols(file_symbols)
        full = set()

        ens2hugo = dict([tuple(line.rstrip('\r\n').split('\t')) for line in file(os.path.join(options.output_directory,'genes_symbols.txt'),'r').readlines() if line.rstrip('\r\n')])

        d = []
        for gg in data:
            if len(gg) != 2:
                print >>sys.stderr, "WARNING: too many genes!",gg
                continue
            g1 = gg[0]
            g2 = gg[1]
            if g1.upper() != g2.upper():
                ens1 = symbols.ensembl(g1.upper(),genes,loci)
                ens2 = symbols.ensembl(g2.upper(),genes,loci)
                if ens1:
                    full.update(ens1)
                if ens2:
                    full.update(ens2)

                if ens1 and ens2:
                    xe1 = set(ens1)
                    xe2 = set(ens2)
                    xe = xe1.intersection(xe2)
#                    if xe:
#                        print "   - ",g1,g2," <-- WARNING: both genes point to same gene!"

                    for e1 in ens1:
                        for e2 in ens2:
                            if e1 != e2:
                                if e2 > e1:
                                    d.append([e2,e1])
                                else:
                                    d.append([e1,e2])

                else:
                    print "   - Original:",g1,g2

        full = sorted(full)
        file(os.path.join(options.output_directory,'known_genes.txt'),'w').writelines([e+"\t"+ens2hugo.get(e,"")+'\n' for e in sorted(full)])
        
        data2 = ['\t'.join(sorted(line)) + '\n' for line in d]
        data2 = sorted(set(data2))

        print "%d known fusion genes found in manually currated database" % (len(data2),)

        if not options.skip_filter_overlap:

            d1 = []
            overlappings = ['ensembl_fully_overlapping_genes.txt',
                            'ensembl_same_strand_overlapping_genes.txt',
                            'refseq_fully_overlapping_genes.txt',
                            'refseq_same_strand_overlapping_genes.txt'#,
#                            "ucsc_fully_overlapping_genes.txt",
#                            "ucsc_same_strand_overlapping_genes.txt",
#                            'pairs_pseudogenes.txt',
                            #'banned.txt',
                            #'paralogs.txt'
                            ]
            ensembls = set(['ensembl_fully_overlapping_genes.txt',
                            'ensembl_same_strand_overlapping_genes.txt'])
            ens = []
            for ov in overlappings:
                p = os.path.join(options.output_directory,ov)
                print "Parsing file:",p
                d2 = []
                if os.path.isfile(p):
                    d2 = sorted(set([tuple(sorted(line.rstrip('\r\n').split('\t'))) for line in file(p,'r').readlines() if line.rstrip('\r\n')]))
                    d3 = set(['\t'.join(line)+'\n' for line in d2])
                    d4 = sorted([line for line in data2 if line in d3])
                    d4 = [line.rstrip('\r\n').split('\t') for line in d4]
                    d4 = [line+[ens2hugo.get(line[0],'')]+[ens2hugo.get(line[1],'')] for line in d4]
                    d4 = ['\t'.join(line)+'\n' for line in d4]
                    file(os.path.join(options.output_directory,"known___%s"%(ov,)),'w').writelines(d4)
                if ov in ensembls:
                    ens.extend(d2)
                d1.extend(d2)
            d = set()
            for line in d1:
                (a,b) = (line[0],line[1])
                if a > b:
                    (a,b) = (b,a)
                d.add("%s\t%s\n" % (a,b))
            ens = set(['\t'.join(line)+'\n' for line in ens])
            ensembl = [line for line in data2 if line in ens]
            file(os.path.join(options.output_directory,'known___ensembl.txt'),'w').writelines(sorted(ensembl))
            skipped = [line for line in data2 if line in d]
            data2 = [line for line in data2 if line not in d]
            file(os.path.join(options.output_directory,'known___all.txt'),'w').writelines(sorted(skipped))
           

            print "%d known fusion genes left after removing the overlappings" % (len(data2),)

            # find the most common genes which partners in fusions
            count = dict()
            for e in data:
                for d in e:
                    x = d.upper()
                    count[x] = count.get(x,0) + 1
            count = sorted([(v,k) for (k,v) in count.items()],reverse=True)
            file(os.path.join(options.output_directory,'known___stat.txt'),'w').writelines(["\t".join(map(str,e))+'\n' for e in count])


    file(os.path.join(options.output_directory,'known.txt'),'w').writelines(sorted(data2))
    #
