#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of candidate fusion genes found in non-tumor cell lines. This list is hard coded inhere.



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of candidate fusion genes found in non-tumor cell lines."""
    version = "%prog 0.10 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of allowed candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of allowed candidate fusion genes is generated. Default is '%default'.""")


    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    print "Generating the list of candidate fusion genes found in non-tumor cell lines..."
    fusions = dict()

    fusions['rattus_norvegicus'] = []

    fusions['mus_musculus'] = [
    ]

    fusions['canis_lupus_familiaris'] = [
    ]


    # manual curation from papers
    fusions['homo_sapiens'] = [
        ['AAED1','HAUS2'],
        ['AAGAB','RPL17'],
        ['AARS','P4HB'],
        ['AARS','PFDN1'],
        ['AARS','UQCRC1'],
        ['AASDHPPT','MAPK1'],
        ['AASDHPPT','RPS16'],
        ['ABAT','FN1'],
        ['ABCA2','UBE2D2'],
        ['ABCA3','SEC13'],
        ['ABCB10','IRF2BP1'],
        ['ABCB6','NIPSNAP3A'],
        ['ABCB7','ENO1'],
        ['ABCB7','METTL3'],
        ['ABCB7','RBBP7'],
        ['ABCD4','SKA2'],
        ['ABCF1','ALDH3A2'],
        ['ABCF1','KDM5C'],
        ['ABCF2','CALR'],
        ['ABCF2','NPLOC4'],
        ['ABCF3','SGSM3'],
        ['ABHD10','POLR2I'],
        ['ABHD10','VIM'],
        ['ABHD14B','RAI1'],
        ['ABHD2','SP3'],
        ['ABI1','DNAJB11'],
        ['ABL1','TAOK2'],
        ['ABRACL','ECHDC1'],
        ['ABRACL','PARPBP'],
        ['AC002116.7','WDR62'],
        ['AC002310.7','DCTPP1'],
        ['AC002985.3','COG4'],
        ['AC003006.7','ZNF530'],
        ['AC004447.2','HNRNPD'],
        ['AC004967.7','OR7E38P'],
        ['AC005785.2','RPL37A'],
        ['AC009133.22','AP2B1'],
        ['AC009133.24','TMED7'],
        ['AC009133.24','TMED7-TICAM2'],
        ['AC010547.9','ATP5L'],
        ['AC010646.3','NLK'],
        ['AC016747.3','PRPF8'],
        ['AC068580.6','RPL8'],
        ['AC091729.9','MPHOSPH8'],
        ['AC092635.1','AC016730.1'],
        ['AC093323.3','HEATR1'],
        ['AC104820.2','ITGA4'],
        ['ACAA2','CUL1'],
        ['ACADM','FAM53C'],
        ['ACAT2','SLC4A7'],
        ['ACBD3','ACTG1'],
        ['ACBD3','MTIF2'],
        ['ACLY','HUWE1'],
        ['ACLY','MED21'],
        ['ACLY','PKM'],
        ['ACOT9','EIF4A3'],
        ['ACOT9','HACL1'],
        ['ACOT9','OAZ1'],
        ['ACOXL','ELP3'],
        ['ACP1','CA11'],
        ['ACSL4','SRSF8'],
        ['ACSL4','TRMT10B'],
        ['ACSM3','LCP1'],
        ['ACSS2','PUS7'],
        ['ACTB','ALDOA'],
        ['ACTB','ARF1'],
        ['ACTB','CD22'],
        ['ACTB','CENPC'],
        ['ACTB','DNAJB4'],
        ['ACTB','FN1'],
        ['ACTB','IP6K2'],
        ['ACTB','IRF3'],
        ['ACTB','KRT5'],
        ['ACTB','KRT6A'],
        ['ACTB','LAPTM5'],
        ['ACTB','MRPL11'],
        ['ACTB','NCOA5'],
        ['ACTB','PA2G4'],
        ['ACTB','RNF139'],
        ['ACTB','RNF20'],
        ['ACTB','SNX22'],
        ['ACTB','TMBIM6'],
        ['ACTB','TRIM25'],
        ['ACTB','UBTF'],
        ['ACTG1','ACVR2B'],
        ['ACTG1','C1QBP'],
        ['ACTG1','GFRA1'],
        ['ACTG1','HM13'],
        ['ACTG1','KRR1'],
        ['ACTG1','LDHB'],
        ['ACTG1','LEMD2'],
        ['ACTG1','NADK'],
        ['ACTG1','NCL'],
        ['ACTG1','NGRN'],
        ['ACTG1','RCN1'],
        ['ACTG1','RPL37A'],
        ['ACTG1','SHMT2'],
        ['ACTG1','SMAP2'],
        ['ACTG1','TIAM2'],
        ['ACTG1','TMEM205'],
        ['ACTG1','ZNF267'],
        ['ACTG1','ZNF343'],
        ['ACTL6A','CELF1'],
        ['ACTL6A','CENPE'],
        ['ACTL6A','PRPF3'],
        ['ACTR2','NDUFAF2'],
        ['ACTR3','C6ORF201'],
        ['ACTR3','DZIP1'],
        ['ACTR5','CALM1'],
        ['ACTR5','PRDX2'],
        ['ACTR6','ATF6B'],
        ['ACVR2A','CAV2'],
        ['ADA','EEF2'],
        ['ADAM15','TTC37'],
        ['ADAM17','CST3'],
        ['ADAM8','EEF2'],
        ['ADAM9','SH2D4A'],
        ['ADAMTS13','ARL4D'],
        ['ADAMTS19','CHSY3'],
        ['ADAMTS4','B4GALT3'],
        ['ADAR','GTF3C5'],
        ['ADAT2','TSC22D1'],
        ['ADCK4','SEPT11'],
        ['ADCY7','ABLIM1'],
        ['ADCY7','RPL30'],
        ['ADD3','PSMB5'],
        ['ADGRA2','DHX40'],
        ['ADGRA3','ZNF91'],
        ['ADH5','GNB2L1'],
        ['ADI1','DEK'],
        ['ADK','FRA10AC1'],
        ['ADNP2','CHSY3'],
        ['ADNP2','SDHB'],
        ['ADNP2','SRSF5'],
        ['ADNP','RPS6KB2'],
        ['ADPGK','P4HB'],
        ['ADRM1','RAB40C'],
        ['ADSL','CHRNA5'],
        ['ADSL','CTBP1'],
        ['ADSL','FBXW11'],
        ['ADSL','SGSM3'],
        ['ADSS','ECHS1'],
        ['AEN','FAM136A'],
        ['AEN','NUDC'],
        ['AES','AZIN1'],
        ['AES','GNB2L1'],
        ['AFF1','LPIN1'],
        ['AFF4','B4GALT6'],
        ['AFF4','IPO8'],
        ['AFG3L2','NDC1'],
        ['AFG3L2','TROVE2'],
        ['AGFG1','SELO'],
        ['AGL','SLC25A36'],
        ['AGL','SNX7'],
        ['AGO1','CCNB1'],
        ['AGO2','BIRC5'],
        ['AGO2','DDX5'],
        ['AGO2','LYN'],
        ['AGPAT3','RNF167'],
        ['AGPAT5','RPL11'],
        ['AGPAT6','GALNT2'],
        ['AGPAT6','POLR1E'],
        ['AGPS','ENO1'],
        ['AGRN','NUDT21'],
        ['AGRN','UBR5'],
        ['AHCY','BRD1'],
        ['AHCY','GPC6'],
        ['AHCY','HSPA1B'],
        ['AHCY','MARCH5'],
        ['AHCY','SAPCD2'],
        ['AHCY','ZNF587B'],
        ['AHNAK','SCAMP1'],
        ['AHRR','UHRF1'],
        ['AHSA1','TMEM241'],
        ['AIFM1','HMGN2'],
        ['AIFM1','KPNA5'],
        ['AIG1','G2E3'],
        ['AIG1','SNRNP40'],
        ['AIMP1','USP5'],
        ['AK2','CAMLG'],
        ['AK2','GEMIN5'],
        ['AKAP11','KIAA0430'],
        ['AKAP11','TRA2B'],
        ['AKAP12','CEBPG'],
        ['AKAP17A','IGH@'],
        ['AKAP17A','IGHA1'],
        ['AKAP8L','ZNF248'],
        ['AKAP9','DNAJC3'],
        ['AKIRIN1','RBBP9'],
        ['AKNA','PHC3'],
        ['AKNA','XRCC5'],
        ['AL121578.2','SYTL5'],
        ['ALCAM','ADCK2'],
        ['ALCAM','GNA12'],
        ['ALDH18A1','APP'],
        ['ALDH18A1','ELL3'],
        ['ALDH18A1','PHF12'],
        ['ALDH1A2','MTRR'],
        ['ALDH1A2','USP24'],
        ['ALDH1A3','RAI14'],
        ['ALDH1B1','MOGS'],
        ['ALDOA','GNB2L1'],
        ['ALDOA','LINC01578'],
        ['ALDOA','PEBP1'],
        ['ALG10','UBXN8'],
        ['ALG13','LDHA'],
        ['ALG9','WRB'],
        ['ALKBH2','KIAA0391'],
        ['ALKBH5','ANXA2'],
        ['ALKBH5','CS'],
        ['ALKBH5','DPYSL5'],
        ['ALKBH5','GALNT18'],
        ['ALKBH5','HSPA5'],
        ['ALKBH5','RIF1'],
        ['ALKBH8','P4HB'],
        ['ALYREF','RPL39'],
        ['AMACR','EXOSC1'],
        ['AMD1','CCT4'],
        ['AMD1','MRPS35'],
        ['AMMECR1','CBX6'],
        ['AMOT','NAA15'],
        ['AMZ2','PHF3'],
        ['ANAPC15','LAMTOR1'],
        ['ANAPC16','DCAF16'],
        ['ANAPC16','STARD3NL'],
        ['ANAPC5','FAM189B'],
        ['ANAPC7','EEF2'],
        ['ANAPC7','PELI2'],
        ['ANGEL2','SLC16A1'],
        ['ANKIB1','PSMA4'],
        ['ANKLE2','ELOVL5'],
        ['ANKRD10','ZNF711'],
        ['ANKRD13C','EHMT1'],
        ['ANKRD13D','AC009120.6'],
        ['ANKRD33B','QRICH1'],
        ['ANKRD37','FUNDC2'],
        ['ANKRD40','ANXA7'],
        ['ANKRD40','ARGLU1'],
        ['ANKRD9','C17ORF75'],
        ['ANKS6','EIF5'],
        ['ANKZF1','SLC25A3'],
        ['ANP32A','COMMD4'],
        ['ANXA5','HEATR1'],
        ['ANXA5','HSPA1A'],
        ['AP000304.12','FRMD5'],
        ['AP000304.12','MRPL18'],
        ['AP000304.12','RNASEH2C'],
        ['AP000350.10','GLRX2'],
        ['AP000350.10','TYRO3'],
        ['AP000721.4','SLC4A11'],
        ['AP1B1','MARS'],
        ['AP1M1','PSMC4'],
        ['AP1S2','SLC27A4'],
        ['AP2A1','ZNF33B'],
        ['AP2A2','PML'],
        ['AP2B1','MTMR3'],
        ['AP2B1','NDUFB11'],
        ['AP2M1','IKBIP'],
        ['AP2M1','ZNF274'],
        ['AP2S1','C1QBP'],
        ['AP3D1','FTSJ2'],
        ['AP3D1','HAGH'],
        ['AP3D1','HNRNPA0'],
        ['AP3D1','KHSRP'],
        ['AP3D1','NUB1'],
        ['APAF1','DDX18'],
        ['APBA1','CCDC14'],
        ['APBA3','PIK3AP1'],
        ['APBB1','CDCA7'],
        ['APEX1','CKB'],
        ['APEX1','POLDIP2'],
        ['APEX1','PSMA3-AS1'],
        ['APEX1','TMEM259'],
        ['API5','MAD1L1'],
        ['APITD1','ARHGAP5-AS1'],
        ['APITD1-CORT','ARHGAP5-AS1'],
        ['APLP2','PPP1R2'],
        ['APOA1BP','RAB11FIP1'],
        ['APOL2','CRLS1'],
        ['APOLD1','HDAC3'],
        ['APPBP2','CLPTM1L'],
        ['APPBP2','COA4'],
        ['APP','CORO7'],
        ['APPL1','CLEC16A'],
        ['APP','PRKDC'],
        ['APP','UAP1'],
        ['APRT','MYBBP1A'],
        ['APRT','PPIB'],
        ['APRT','RPS5'],
        ['AQR','PUM2'],
        ['ARAP1','AHI1'],
        ['ARAP1','TRAF1'],
        ['ARAP3','KRT81'],
        ['ARCN1','POP4'],
        ['ARF1','C7ORF55'],
        ['ARF1','EDF1'],
        ['ARF1','NISCH'],
        ['ARF1','PRDX4'],
        ['ARF1','RPN2'],
        ['ARF1','VBP1'],
        ['ARF4','RPL38'],
        ['ARF6','NCL'],
        ['ARFGAP1','MIAT'],
        ['ARFGAP1','RNF34'],
        ['ARGLU1','DNMT3A'],
        ['ARHGAP11A','PABPC4'],
        ['ARHGAP12','PPP2R4'],
        ['ARHGAP30','SCAF1'],
        ['ARHGAP4','CD74'],
        ['ARHGEF1','PPP1R15A'],
        ['ARHGEF1','STAG2'],
        ['ARID1A','RALY'],
        ['ARID1B','RPS3'],
        ['ARID2','CEP68'],
        ['ARID4B','EIF3L'],
        ['ARID4B','LGALS3BP'],
        ['ARID5B','HPS5'],
        ['ARL10','HNRNPF'],
        ['ARL15','PGAM5'],
        ['ARL1','PDSS2'],
        ['ARL6IP1','ATP11B'],
        ['ARL6IP1','RPS6KL1'],
        ['ARL6IP1','SURF2'],
        ['ARL6IP5','PPP1CC'],
        ['ARMC2','SH3BP5-AS1'],
        ['ARMCX3-AS1','CORO1A'],
        ['ARMT1','SYNCRIP'],
        ['ARNT2','PGRMC1'],
        ['ARPC1A','HSP90AB1'],
        ['ARPC1B','CECR5'],
        ['ARPC2','AKR7A2'],
        ['ARPC2','HACD3'],
        ['ARPIN','LUC7L'],
        ['ARPP19','DBI'],
        ['ARRDC4','SLC25A3'],
        ['ARSI','CAMK2A'],
        ['ASAP2','SMC3'],
        ['ASB6','C17ORF89'],
        ['ASB6','RPS4X'],
        ['ASB6','STON2'],
        ['ASF1A','LIN28B'],
        ['ASF1A','MLEC'],
        ['ASMTL','EIF2B1'],
        ['ASNA1','SLC25A36'],
        ['ASNA1','YBX1'],
        ['ASNS','RPS6'],
        ['ASPSCR1','C19ORF43'],
        ['ASXL1','MGEA5'],
        ['ASXL1','PLEKHA2'],
        ['ASXL1','VEZF1'],
        ['ATAD1','CPVL'],
        ['ATAD2','EIF3M'],
        ['ATAD2','TBPL1'],
        ['ATAD5','DNAJC10'],
        ['ATF6','SLC3A2'],
        ['ATF7IP','GLA'],
        ['ATG12','GID4'],
        ['ATG12','RAF1'],
        ['ATG14','SLC39A14'],
        ['ATG5','LRRC75A-AS1'],
        ['ATG9A','NIPSNAP3A'],
        ['ATG9A','TMX1'],
        ['ATL2','NDUFB7'],
        ['ATL3','NPTX2'],
        ['ATL3','TALDO1'],
        ['ATP11B','FAT1'],
        ['ATP11C','BMP7'],
        ['ATP11C','MFSD10'],
        ['ATP11C','PUM1'],
        ['ATP13A3','ORC1'],
        ['ATP1A1','KIF18A'],
        ['ATP2A2','PSMA6'],
        ['ATP2A2','SMARCD1'],
        ['ATP4A','NUDT1'],
        ['ATP5A1','BTBD2'],
        ['ATP5A1','CBX5'],
        ['ATP5A1','HAUS1'],
        ['ATP5A1','L3MBTL3'],
        ['ATP5A1','STAG2'],
        ['ATP5A1','TMEM198'],
        ['ATP5A1','TOMM40'],
        ['ATP5B','APOA1BP'],
        ['ATP5B','CALR'],
        ['ATP5B','DZIP1'],
        ['ATP5B','FAM131A'],
        ['ATP5B','NIPAL3'],
        ['ATP5B','PAN2'],
        ['ATP5B','SARS'],
        ['ATP5B','SIRT2'],
        ['ATP5C1','MCM4'],
        ['ATP5C1','PLS3'],
        ['ATP5G1','NVL'],
        ['ATP5G1','SPAG9'],
        ['ATP5G3','IGF2BP3'],
        ['ATP5G3','INTS7'],
        ['ATP5G3','MDH1'],
        ['ATP5L','NELFCD'],
        ['ATP5O','FRMD5'],
        ['ATP5O','MRPL18'],
        ['ATP5SL','SEC24C'],
        ['ATP6AP2','TMEM70'],
        ['ATP6V0B','AGAP1'],
        ['ATP6V0B','FBL'],
        ['ATP6V0B','TMEM218'],
        ['ATP6V0D1','SNX1'],
        ['ATP6V1A','C1ORF112'],
        ['ATP6V1A','ILF3'],
        ['ATP6V1A','THAP5'],
        ['ATP6V1D','PCNA'],
        ['ATP6V1E1','SDE2'],
        ['ATP6V1G1','MRPL3'],
        ['ATP6V1H','BOP1'],
        ['ATP8B2','HNRNPAB'],
        ['ATP8B2','NUPL2'],
        ['ATPIF1','FMR1'],
        ['ATPIF1','TMEM214'],
        ['ATRN','RPS29'],
        ['ATXN2L','PDE8A'],
        ['ATXN3','ASXL3'],
        ['ATXN7L3B','PXMP2'],
        ['ATXN7L3','TATDN2'],
        ['AUNIP','DDX17'],
        ['AURKA','ARHGEF7'],
        ['AURKA','SP1'],
        ['AURKB','FARP2'],
        ['AURKB','RAD21'],
        ['AVEN','BAZ1B'],
        ['AVPI1','ATP5SL'],
        ['AXIN1','OSBPL6'],
        ['AXIN1','RPRM'],
        ['AXIN1','WHSC1'],
        ['AZIN1','DNAJC3'],
        ['AZIN1','RPL4'],
        ['B2M','ADAR'],
        ['B2M','AKAP8L'],
        ['B2M','FNBP1'],
        ['B2M','MTPN'],
        ['B2M','SLC9A7'],
        ['B2M','TMEM30A'],
        ['B3GNT2','ANXA11'],
        ['B3GNT5','TMX3'],
        ['B4GALT1','ALDOA'],
        ['B4GALT1','SUGP2'],
        ['B4GAT1','LMAN2'],
        ['B4GAT1','TMED3'],
        ['BACE2','FAM3B'],
        ['BAG2','EDC4'],
        ['BAG5','TUBB4B'],
        ['BAG6','MED14'],
        ['BAG6','PPARD'],
        ['BAIAP2-AS1','IRF2BPL'],
        ['BASP1','SLC39A14'],
        ['BAZ1A','HNRNPH1'],
        ['BAZ1B','BNIP3L'],
        ['BAZ1B','JRK'],
        ['BBS10','FKBP4'],
        ['BBS10','RREB1'],
        ['BBS1','RNF213'],
        ['BBS9','NXT2'],
        ['BCAP31','DHX40'],
        ['BCAR1','NRF1'],
        ['BCAT1','AGPAT5'],
        ['BCAT1','COPS6'],
        ['BCAT2','FAM122A'],
        ['BCCIP','HSP90AB1'],
        ['BCKDK','CSNK1G2'],
        ['BCL2L13','DUSP3'],
        ['BCL2L13','TLN2'],
        ['BCL2L13','UBC'],
        ['BCL2','UBAP2'],
        ['BCL7C','BUB1'],
        ['BCLAF1','MTMR4'],
        ['BCLAF1','SORT1'],
        ['BCOR','POP4'],
        ['BCR','TMEM219'],
        ['BECN1','HNRNPM'],
        ['BEST1','SPARC'],
        ['BET1','HELLS'],
        ['BET1','P3H3'],
        ['BHLHE40','ZBTB14'],
        ['BICD2','ZNF397'],
        ['BIRC5','RMDN1'],
        ['BIRC6','G3BP1'],
        ['BIRC6','GPKOW'],
        ['BIRC6','TUBB'],
        ['BLCAP','PFAS'],
        ['BLM','AMMECR1'],
        ['BLMH','SCML1'],
        ['BLOC1S2','GOSR2'],
        ['BLOC1S5-TXNDC5','PKM'],
        ['BLOC1S5-TXNDC5','UBC'],
        ['BLOC1S5-TXNDC5','UTP18'],
        ['BLVRB','QTRT1'],
        ['BLZF1','ATP6V1A'],
        ['BMI1','SRP72'],
        ['BNIP1','SP110'],
        ['BNIP2','PQLC1'],
        ['BNIP2','REST'],
        ['BOD1L1','ADNP2'],
        ['BOD1L1','TBC1D10C'],
        ['BOLA3','HSP90AA1'],
        ['BOP1','HNRNPL'],
        ['BOP1','RABEP1'],
        ['BRCA1','C19ORF48'],
        ['BRCA1','HEATR3'],
        ['BRCA1','ZNF581'],
        ['BRCA2','COG6'],
        ['BRD2','EPCAM'],
        ['BRD4','SEC24A'],
        ['BRD8','WDR41'],
        ['BRF1','FAM32A'],
        ['BRI3BP','GTF2F1'],
        ['BRI3BP','RPL11'],
        ['BRI3BP','TJP1'],
        ['BRI3BP','UVRAG'],
        ['BRIX1','AES'],
        ['BRIX1','ECHS1'],
        ['BRIX1','SLC35E3'],
        ['BRK1','BABAM1'],
        ['BRK1','RAF1'],
        ['BRK1','UQCR10'],
        ['BROX','ITGAE'],
        ['BSCL2','EEF1G'],
        ['BSG','BLVRA'],
        ['BSG','RAB8B'],
        ['BSG','SNRPD3'],
        ['BTAF1','PPP1R9A'],
        ['BTAF1','STRA13'],
        ['BTBD1','ALKBH5'],
        ['BTBD1','EEF2'],
        ['BTBD2','BSG'],
        ['BTF3','NPM1'],
        ['BTG1','DSN1'],
        ['BTG1','YTHDF2'],
        ['BTG3','CKAP5'],
        ['BUB1','ARF1'],
        ['BUB1','BRIP1'],
        ['BUB1','EEF2'],
        ['BUB1','NSDHL'],
        ['BUB3','SMAP2'],
        ['BUD31','SPC24'],
        ['BZW1','JMJD6'],
        ['BZW1','ORC2'],
        ['C10ORF12','PSMA7'],
        ['C10ORF88','RBFOX2'],
        ['C11ORF58','MRPS35'],
        ['C11ORF98','MAN1A2'],
        ['C12ORF49','STUB1'],
        ['C12ORF49','ZNF674-AS1'],
        ['C12ORF4','SETD7'],
        ['C12ORF66','SPIN1'],
        ['C12ORF75','FTL'],
        ['C12ORF75','MTHFD2'],
        ['C14ORF166','TUBB4B'],
        ['C14ORF37','CSDE1'],
        ['C15ORF38-AP3S2','LUC7L'],
        ['C16ORF72','CAST'],
        ['C16ORF72','HNRNPF'],
        ['C16ORF72','SAMD12'],
        ['C16ORF72','SENP6'],
        ['C16ORF91','LHPP'],
        ['C16ORF93','ZCCHC8'],
        ['C17ORF51','TMBIM6'],
        ['C17ORF75','SUPT5H'],
        ['C18ORF32','TYSND1'],
        ['C19MC','RPLP1'],
        ['C19ORF43','ACTG1'],
        ['C1ORF123','IARS'],
        ['C1ORF123','UQCR10'],
        ['C1ORF174','PTGES3'],
        ['C1ORF174','SLC25A1'],
        ['C1ORF198','CENPF'],
        ['C1ORF43','ARIH1'],
        ['C1ORF43','FNDC3B'],
        ['C1ORF43','KAT6A'],
        ['C1ORF43','SENP2'],
        ['C1ORF54','ACTB'],
        ['C1QBP','BROX'],
        ['C1QBP','DEK'],
        ['C1QBP','FAM136A'],
        ['C1QBP','MED8'],
        ['C1QBP','PIGS'],
        ['C1QBP','PIK3IP1'],
        ['C1QBP','RPS16'],
        ['C1QL4','ELAC2'],
        ['C1QL4','PSMD8'],
        ['C20ORF24','ATP5A1'],
        ['C20ORF24','EMC4'],
        ['C21ORF58','PRKD2'],
        ['C21ORF58','R3HDM4'],
        ['C22ORF39','PCMT1'],
        ['C22ORF39','RAB18'],
        ['C2CD2','LDHA'],
        ['C3ORF52','AGO1'],
        ['C3ORF52','EIF5A'],
        ['C3ORF58','STARD7'],
        ['C4ORF3','ENO1'],
        ['C4ORF46','HAUS7'],
        ['C4ORF46','TREX2'],
        ['C5ORF15','RINT1'],
        ['C5ORF24','MTIF3'],
        ['C5ORF34','USMG5'],
        ['C5ORF46','SPINK1'],
        ['C6ORF48','QARS'],
        ['C6ORF62','CALU'],
        ['C9ORF142','DEK'],
        ['CA5B','ACLY'],
        ['CAB39','CYTH2'],
        ['CABIN1','ARL10'],
        ['CACNA2D1','CSDE1'],
        ['CACTIN','SNUPN'],
        ['CAD','CALR'],
        ['CADM1','FER'],
        ['CADPS2','STARD3NL'],
        ['CALCOCO2','RBBP4'],
        ['CALD1','VPS25'],
        ['CALM1','CYFIP2'],
        ['CALM1','HNRNPC'],
        ['CALM1','PDIA3'],
        ['CALM1','PRRX1'],
        ['CALM2','ATP6V1D'],
        ['CALM2','MKI67'],
        ['CALM2','PA2G4'],
        ['CALM2','SAMD15'],
        ['CALM3','FUBP1'],
        ['CALR','AP1B1'],
        ['CALR','CDC5L'],
        ['CALR','CTNNB1'],
        ['CALR','DYRK1B'],
        ['CALR','FBXO21'],
        ['CALR','POLR2I'],
        ['CALR','RIF1'],
        ['CALU','EIF3G'],
        ['CAMK1','C22ORF39'],
        ['CAMK1','HIRA'],
        ['CAMK2D','EEF2'],
        ['CAMTA2','PLCG2'],
        ['CAND1','DDX6'],
        ['CANX','ARHGAP4'],
        ['CANX','ARL6IP1'],
        ['CANX','CENPV'],
        ['CANX','DGKZ'],
        ['CANX','HLCS'],
        ['CANX','KIDINS220'],
        ['CANX','KPNB1'],
        ['CANX','LPCAT3'],
        ['CANX','NGRN'],
        ['CANX','PTMA'],
        ['CANX','THRB'],
        ['CANX','UHRF1'],
        ['CANX','XPO1'],
        ['CANX','ZADH2'],
        ['CAP1','PRSS23'],
        ['CAPN2','TNKS2'],
        ['CAPNS1','CTBP1'],
        ['CAPNS1','LIMS1'],
        ['CAPNS1','TMEM167A'],
        ['CAPRIN1','ANAPC16'],
        ['CAPRIN1','MAGED2'],
        ['CAPZA1','TRMT2B'],
        ['CAPZA2','SERPINH1'],
        ['CAPZB','FADS2'],
        ['CAPZB','SIPA1L3'],
        ['CAPZB','SPTBN1'],
        ['CARD10','PHGDH'],
        ['CARD11','MAPKBP1'],
        ['CARS2','CTNNBIP1'],
        ['CARS','HDLBP'],
        ['CASC3','ATP6V0E1'],
        ['CASC4','PGAM5'],
        ['CASC4','SRSF7'],
        ['CASC5','ENO1'],
        ['CASK','TSPAN3'],
        ['CASP7','ATP1A1'],
        ['CASP8AP2','COPS3'],
        ['CASP8AP2','HTR3A'],
        ['CASP8','TSPYL1'],
        ['CASP9','RPS20'],
        ['CAST','MTHFD2'],
        ['CAT','TIMM13'],
        ['CAV1','EZR'],
        ['CAV2','HAUS4'],
        ['CBFB','IRAK1'],
        ['CBFB','SFXN1'],
        ['CBFB','ZNF639'],
        ['CBX1','TMX4'],
        ['CBX3','ISYNA1'],
        ['CBX5','CD2AP'],
        ['CBX5','CELF2'],
        ['CBX5','R3HDM1'],
        ['CBX6','RAB35'],
        ['CBY1','RPA1'],
        ['CCAR1','DDX3X'],
        ['CCAR1','SDCCAG8'],
        ['CCDC117','CPSF2'],
        ['CCDC117','GPX4'],
        ['CCDC127','PCNA'],
        ['CCDC174','UBE3C'],
        ['CCDC22','EZR'],
        ['CCDC25','RPL4'],
        ['CCDC34','BDNF'],
        ['CCDC34','MSH6'],
        ['CCDC34','ZNF611'],
        ['CCDC43','CCDC109B'],
        ['CCDC43','SEC23B'],
        ['CCDC50','CCT2'],
        ['CCDC6','ARPC5'],
        ['CCDC85B','PEX5'],
        ['CCDC85C','USP33'],
        ['CCDC86','NDUFV2'],
        ['CCDC88B','GNAI2'],
        ['CCDC88C','BCAT1'],
        ['CCDC88C','FLNA'],
        ['CCDC8','LRRC16A'],
        ['CCDC97','AGPAT5'],
        ['CCDC9','NCOR2'],
        ['CCL22','EIF2B2'],
        ['CCL22','NDUFA3'],
        ['CCM2','ATP6V1G2-DDX39B'],
        ['CCM2','CSNK2A1'],
        ['CCM2','DDX39B'],
        ['CCNA2','TM9SF2'],
        ['CCNA2','ZNF516'],
        ['CCNB1','FAM169A'],
        ['CCNB1IP1','MAN2C1'],
        ['CCNB1','RBBP4'],
        ['CCND2','CBX5'],
        ['CCND2','NRAS'],
        ['CCND2','RLIM'],
        ['CCNF','ASAH1'],
        ['CCNG1','FLYWCH2'],
        ['CCNG1','GALNT7'],
        ['CCNI','BAP1'],
        ['CCNL1','ACSL5'],
        ['CCSER2','CDC123'],
        ['CCT2','HTATSF1'],
        ['CCT2','KIF7'],
        ['CCT3','HSPA9'],
        ['CCT3','ITGA6'],
        ['CCT3','SSBP1'],
        ['CCT3','USP33'],
        ['CCT4','CSDE1'],
        ['CCT4','FZR1'],
        ['CCT4','GTF3A'],
        ['CCT4','LAMTOR4'],
        ['CCT4','MIF4GD'],
        ['CCT4','PDCD6IP'],
        ['CCT4','SYK'],
        ['CCT5','AP2S1'],
        ['CCT5','CAMTA1'],
        ['CCT5','GSR'],
        ['CCT5','NEO1'],
        ['CCT5','ZNF148'],
        ['CCT6A','AKAP11'],
        ['CCT6A','PDZD8'],
        ['CCT6A','RPS19'],
        ['CCT7','AGPS'],
        ['CCT7','CHMP7'],
        ['CCT7','CUL3'],
        ['CCT7','EEF2'],
        ['CCT7','EPT1'],
        ['CCT7','RPL8'],
        ['CCT7','SLC26A11'],
        ['CCT7','STK26'],
        ['CCT7','UBE3C'],
        ['CCT8','SLC16A7'],
        ['CCT8','UQCRC1'],
        ['CD151','P4HB'],
        ['CD164','SAFB'],
        ['CD164','ZNF207'],
        ['CD226','DHX38'],
        ['CD276','HSPA1A'],
        ['CD27-AS1','NUMA1'],
        ['CD2AP','EEF2'],
        ['CD2BP2','GGA2'],
        ['CD302','RBM27'],
        ['CD320','CKB'],
        ['CD3EAP','PRRC2A'],
        ['CD46','ANXA2'],
        ['CD46','MSH6'],
        ['CD47','RHOA'],
        ['CD48','PRICKLE2'],
        ['CD55','CLSTN1'],
        ['CD55','GSK3B'],
        ['CD55','RPL11'],
        ['CD63','SCRIB'],
        ['CD63','TXNDC9'],
        ['CD69','CES4A'],
        ['CD74','ATP2A1'],
        ['CD74','AURKB'],
        ['CD74','ITGAL'],
        ['CD74','PPP1R26'],
        ['CD74','SLC38A2'],
        ['CD99','ATL3'],
        ['CDC123','NCOR1'],
        ['CDC123','PAX5'],
        ['CDC123','RPS11'],
        ['CDC14A','GNAS'],
        ['CDC25A','MDH2'],
        ['CDC25B','PCNX'],
        ['CDC27','SMC1A'],
        ['CDC34','ALDOA'],
        ['CDC34','ERCC6'],
        ['CDC34','SAR1B'],
        ['CDC37','B4GALT3'],
        ['CDC37','DDX41'],
        ['CDC37L1','MRPL23'],
        ['CDC37L1','PKM'],
        ['CDC37','TTC17'],
        ['CDC42BPA','KIAA0513'],
        ['CDC42BPB','EEF2'],
        ['CDC42BPG','MGAT4B'],
        ['CDC42','CLK3'],
        ['CDC42EP3','SPTBN1'],
        ['CDC42SE2','EIF3B'],
        ['CDC42SE2','SARAF'],
        ['CDC5L','GPX4'],
        ['CDC73','RPS20'],
        ['CDCA8','TOR1A'],
        ['CDH2','TSR3'],
        ['CDK10','TMBIM6'],
        ['CDK11B','GLRX5'],
        ['CDK12','RPS4X'],
        ['CDK16','POLR2A'],
        ['CDK1','UMAD1'],
        ['CDK2','NGFRAP1'],
        ['CDK4','BPTF'],
        ['CDK4','TOP2B'],
        ['CDK6','DRG1'],
        ['CDK6','NAT6'],
        ['CDK6','RPS8'],
        ['CDK6','STUB1'],
        ['CDK8','SNRPF'],
        ['CDKN1A','MACF1'],
        ['CDKN2A','EIF3I'],
        ['CDKN2AIPNL','LMNB2'],
        ['CEBPG','SEPHS2'],
        ['CEBPG','WARS2'],
        ['CEBPZ','ENO1'],
        ['CECR5','AMZ2'],
        ['CECR5','HNRNPF'],
        ['CENPB','GOLIM4'],
        ['CENPF','CD99'],
        ['CENPF','GLA'],
        ['CENPF','MSMO1'],
        ['CENPM','ERCC1'],
        ['CENPN','TMEM167A'],
        ['CENPO','IGHA1'],
        ['CENPV','CANX'],
        ['CEP128','C19ORF12'],
        ['CEP128','FAM168B'],
        ['CEP128','LAPTM4B'],
        ['CEP290','NGRN'],
        ['CEP295','WDR41'],
        ['CEP78','SSRP1'],
        ['CEP83','SRP9'],
        ['CEP85L','CXCL16'],
        ['CEP85L','RABL6'],
        ['CEP85','SLC44A1'],
        ['CERS4','SMARCD2'],
        ['CERS6','TIMM50'],
        ['CFLAR','PFDN5'],
        ['CGRRF1','CEP104'],
        ['CH17-373J23.1','PSMA1'],
        ['CHAC2','ERLEC1'],
        ['CHAC2','MMADHC'],
        ['CHAF1B','HN1'],
        ['CHCHD2','CCDC137'],
        ['CHCHD3','CENPV'],
        ['CHCHD3','PABPC4'],
        ['CHCHD3','ZNF317'],
        ['CHCHD4','EIF3L'],
        ['CHD1L','NTPCR'],
        ['CHD1','PPAP2A'],
        ['CHD2','FOXM1'],
        ['CHD3','ABLIM1'],
        ['CHD4','ROMO1'],
        ['CHD4','WDR41'],
        ['CHD7','CCT3'],
        ['CHD8','KRTCAP2'],
        ['CHD8','SUMO1'],
        ['CHD9','ZNF146'],
        ['CHDH','C1GALT1'],
        ['CHEK2','CAPZB'],
        ['CHEK2','MPZL1'],
        ['CHIC2','NASP'],
        ['CHID1','SMC1A'],
        ['CHMP2B','TCOF1'],
        ['CHORDC1','CDH2'],
        ['CHP1','CCT8'],
        ['CHPT1','KLHDC3'],
        ['CHPT1','TCF19'],
        ['CHPT1','TLX1'],
        ['CHRAC1','FGFR2'],
        ['CHRNA5','COX20'],
        ['CHSY3','ADAMTS19'],
        ['CHTF8','SMAD5'],
        ['CHUK','CCDC47'],
        ['CHURC1','IDH3B'],
        ['CIAO1','TPRN'],
        ['CIART','NUP43'],
        ['CIC','CNNM4'],
        ['CIC','LIMD1'],
        ['CIC','ZMAT3'],
        ['CIRBP','BAP1'],
        ['CIRBP','FXR2'],
        ['CIRBP','RBCK1'],
        ['CIRBP','RPL11'],
        ['CIRBP','TIAL1'],
        ['CIRH1A','SCN8A'],
        ['CISD1','GDAP1'],
        ['CISD1','RPS19'],
        ['CISD3','CELF2'],
        ['CISD3','NDUFB8'],
        ['CKAP2','AP000350.10'],
        ['CKAP2','KNOP1'],
        ['CKAP2L','CXORF57'],
        ['CKAP2L','STARD4'],
        ['CKAP2','MIF'],
        ['CKAP2','PNISR'],
        ['CKAP2','SHMT2'],
        ['CKAP4','OAF'],
        ['CKB','BCL7C'],
        ['CKB','PHF3'],
        ['CKB','PRC1'],
        ['CKB','RYBP'],
        ['CKB','TRAPPC2'],
        ['CKB','ZNF207'],
        ['CLCN3','GET4'],
        ['CLCN3','SUN1'],
        ['CLIC1','DNTTIP2'],
        ['CLIC4','ANP32A'],
        ['CLIC4','APOL2'],
        ['CLIC4','MAGT1'],
        ['CLIC4','MRPS30'],
        ['CLIC4','PEBP1'],
        ['CLIC4','SPTBN1'],
        ['CLINT1','CSDE1'],
        ['CLINT1','GCLC'],
        ['CLINT1','PAQR3'],
        ['CLIP2','XYLT1'],
        ['CLIP4','SPG11'],
        ['CLK1','DDX3X'],
        ['CLN5','HBS1L'],
        ['CLNS1A','KHSRP'],
        ['CLNS1A','P4HB'],
        ['CLNS1A','TUBA1B'],
        ['CLPP','UBB'],
        ['CLPTM1L','CHAF1B'],
        ['CLPTM1L','CSE1L'],
        ['CLPTM1L','CTNNB1'],
        ['CLSPN','EEF2'],
        ['CLSPN','PSMB2'],
        ['CLTA','MYC'],
        ['CLTB','HACD3'],
        ['CLTC','CREBZF'],
        ['CLTC','PLEKHG4'],
        ['CLTC','RSL1D1'],
        ['CMAS','RAB11FIP4'],
        ['CMC4','NEDD1'],
        ['CMPK1','PARK7'],
        ['CMTR2','RAD17'],
        ['CNBP','MAVS'],
        ['CNBP','REST'],
        ['CNBP','TRIM28'],
        ['CNDP2','SRF'],
        ['CNIH1','LY6E'],
        ['CNIH4','CDK16'],
        ['CNIH4','KLHL15'],
        ['CNIH4','WDR1'],
        ['CNN3','SLC27A3'],
        ['CNNM2','BBS4'],
        ['CNOT10','ANAPC16'],
        ['CNOT11','HIRIP3'],
        ['CNOT11','OSBP'],
        ['CNOT1','ARAP2'],
        ['CNOT1','MLLT6'],
        ['CNOT1','NAA38'],
        ['CNOT1','NGRN'],
        ['CNOT6','HNRNPU'],
        ['CNOT6L','ZNF268'],
        ['CNPY2','ZNF670'],
        ['CNPY4','MBLAC1'],
        ['CNPY4','TMSB10'],
        ['COA3','GLUL'],
        ['COA6','MIER3'],
        ['COASY','PRDX1'],
        ['COCH','DNAL1'],
        ['COCH','SRPK1'],
        ['COL1A1','ACTB'],
        ['COL1A1','FN1'],
        ['COL1A1','PTPRK'],
        ['COL4A3','QTRTD1'],
        ['COL6A1','AUNIP'],
        ['COL6A2','MYBL2'],
        ['COL7A1','UCN2'],
        ['COMMD2','NME1'],
        ['COMMD3','ACTG1'],
        ['COMMD3-BMI1','SRP72'],
        ['COPB1','NPEPPS'],
        ['COPB2','ATP6AP1'],
        ['COPB2','FANCI'],
        ['COPE','COG4'],
        ['COPE','RPS21'],
        ['COPE','SNRPD2'],
        ['COPG2','LAS1L'],
        ['COPRS','NIFK'],
        ['COPRS','TMTC1'],
        ['COPS3','MAP4K4'],
        ['COPS3','SLC25A11'],
        ['COPS3','TOR1AIP2'],
        ['COPZ1','GAPDH'],
        ['COPZ1','IRF2BP2'],
        ['COQ3','PSMA5'],
        ['CORO1A','PSD4'],
        ['CORO1C','MBOAT7'],
        ['CORO1C','RCHY1'],
        ['CORO1C','TSPAN15'],
        ['CORO7','ERGIC1'],
        ['CORO7-PAM16','ERGIC1'],
        ['COTL1','SLC1A5'],
        ['COX11','CST3'],
        ['COX11','NPTN'],
        ['COX14','ATF6'],
        ['COX15','EXTL3'],
        ['COX20','BCAP31'],
        ['COX4I1','CDK12'],
        ['COX4I1','PHGDH'],
        ['COX4I1','RPL18'],
        ['COX6C','EDF1'],
        ['COX7A2','CORO7-PAM16'],
        ['COX7A2','SENP2'],
        ['COX8A','SLC4A11'],
        ['CPNE3','CCAR2'],
        ['CPNE3','DSC2'],
        ['CPNE3','PRKAR1B'],
        ['CPPED1','FANCD2'],
        ['CPSF3','ASTN1'],
        ['CPSF3L','MKI67'],
        ['CPSF3','TBC1D9B'],
        ['CPSF6','MRPS12'],
        ['CPSF6','TOMM5'],
        ['CRACR2B','PLS3'],
        ['CREB3L2','PBX1'],
        ['CREBBP','SOD1'],
        ['CREBZF','CPSF3L'],
        ['CREG1','ENO1'],
        ['CREG1','RAPGEF3'],
        ['CRELD1','PRRT3-AS1'],
        ['CRK','KRAS'],
        ['CRK','ZNF580'],
        ['CRNKL1','NCAPD2'],
        ['CRTC3','TCEB3'],
        ['CRYBG3','KCTD14'],
        ['CRYBG3','NDUFC2'],
        ['CRYBG3','NDUFC2-KCTD14'],
        ['CRYBG3','SLC25A6'],
        ['CSDE1','ADD3'],
        ['CSDE1','ALDOC'],
        ['CSDE1','ATP5A1'],
        ['CSDE1','CCDC181'],
        ['CSDE1','EIF4G2'],
        ['CSDE1','GRIK5'],
        ['CSDE1','LRRC75A-AS1'],
        ['CSDE1','LSS'],
        ['CSDE1','NOL7'],
        ['CSDE1','SDHB'],
        ['CSDE1','STUB1'],
        ['CSDE1','YEATS4'],
        ['CSE1L','CRCP'],
        ['CSE1L','HSPA9'],
        ['CSE1L','LINC00657'],
        ['CSE1L','MRPL3'],
        ['CSE1L','PDLIM2'],
        ['CSF1','ACTB'],
        ['CS','KDM5B'],
        ['CSK','RAE1'],
        ['CSNK1A1','CA12'],
        ['CSNK1A1','CSTF3'],
        ['CSNK2A1','GSK3B'],
        ['CSRP2','PDK4'],
        ['CSTF1','SART3'],
        ['CSTF3','SLC25A3'],
        ['CS','UTP14A'],
        ['CTBP1','POLR2D'],
        ['CTBS','GNG5'],
        ['CTBS','RPS9'],
        ['CTDSP1','KNOP1'],
        ['CTDSP2','HSP90AB1'],
        ['CTNNB1','MTURN'],
        ['CTNNB1','NR2F6'],
        ['CTSA','CCDC102A'],
        ['CTSB','OAZ1'],
        ['CTSC','OSBPL3'],
        ['CTSC','RAB38'],
        ['CTSD','IFITM10'],
        ['CTSH','UBP1'],
        ['CTTN','PEX1'],
        ['CTTN','QSOX2'],
        ['CUBN','DNAJC1'],
        ['CUEDC2','GSPT1'],
        ['CUL2','CEP76'],
        ['CUL3','CALM2'],
        ['CUL4A','STARD7'],
        ['CUL4B','CPNE3'],
        ['CUL4B','PSMD8'],
        ['CUTC','ABCC2'],
        ['CUX1','DDX24'],
        ['CXORF38','NRIP1'],
        ['CXORF56','SERBP1'],
        ['CXORF56','TTLL12'],
        ['CXORF57','EIF2S3'],
        ['CXORF57','ENO1'],
        ['CXXC1','UBE2G1'],
        ['CYB561D2','ANAPC13'],
        ['CYB5B','CISD2'],
        ['CYBA','JARID2'],
        ['CYBA','SYNCRIP'],
        ['CYBB','EP300'],
        ['CYCS','TIMELESS'],
        ['CYLD','LCP1'],
        ['CYLD','LINC00339'],
        ['CYP51A1','ATP5B'],
        ['CYP51A1','CDC42BPA'],
        ['CYP51A1','FAM208B'],
        ['CYP51A1','ZNF512B'],
        ['CYTH3','ADD3'],
        ['DAP3','HSPA8'],
        ['DAP3','TMBIM6'],
        ['DAPK1','COX5B'],
        ['DAZAP1','CSNK1A1'],
        ['DBNL','MRFAP1L1'],
        ['DBR1','SSR3'],
        ['DBT','GALNT2'],
        ['DCAF16','ZFAND1'],
        ['DCAF17','ENO1'],
        ['DCAF4','ZBTB34'],
        ['DCAF8','DDX3X'],
        ['DCBLD2','CANX'],
        ['DCTN1','C11ORF57'],
        ['DCTN1','RABEP1'],
        ['DCTN4','ANKRD27'],
        ['DCTN4','DOCK11'],
        ['DCTN4','RHOA'],
        ['DCTN5','MGAT1'],
        ['DCTPP1','EXOSC7'],
        ['DCTPP1','RAI14'],
        ['DCXR','EEF2'],
        ['DDAH1','APEH'],
        ['DDAH1','USP22'],
        ['DDAH1','WSB2'],
        ['DDB1','FAM168B'],
        ['DDB1','MATR3'],
        ['DDB1','SPTLC2'],
        ['DDB1','XPO7'],
        ['DDB2','GAK'],
        ['DDIT4','CCT3'],
        ['DDIT4','PDXP'],
        ['DDIT4','SH3BP1'],
        ['DDOST','XRN2'],
        ['DDRGK1','GTF2B'],
        ['DDT','DNAJC10'],
        ['DDT','ZRANB1'],
        ['DDX10','GPRC5C'],
        ['DDX17','ATP11C'],
        ['DDX17','ATP6V0D1'],
        ['DDX17','MRPL40'],
        ['DDX17','NCL'],
        ['DDX17','PDS5A'],
        ['DDX17','QRICH2'],
        ['DDX17','SMG1'],
        ['DDX18','HSPA1A'],
        ['DDX1','CD69'],
        ['DDX20','EIF3H'],
        ['DDX21','CHD8'],
        ['DDX21','COX14'],
        ['DDX21','GFM1'],
        ['DDX21','TOMM20'],
        ['DDX23','MSH6'],
        ['DDX24','CNOT6'],
        ['DDX24','HNRNPU'],
        ['DDX24','SLC25A17'],
        ['DDX39A','NGRN'],
        ['DDX39B','PIEZO2'],
        ['DDX3X','GNPTAB'],
        ['DDX3X','MTA2'],
        ['DDX3X','POLL'],
        ['DDX46','JADE2'],
        ['DDX47','HDAC3'],
        ['DDX52','GOLGA3'],
        ['DDX52','PRKCZ'],
        ['DDX54','TBCA'],
        ['DDX54','UBA1'],
        ['DDX56','LDHB'],
        ['DDX5','DDT'],
        ['DDX5','POLG2'],
        ['DDX6','CKB'],
        ['DDX6','HSPB8'],
        ['DDX6','SLBP'],
        ['DEGS1','RBBP7'],
        ['DEK','CMIP'],
        ['DEK','COX6C'],
        ['DEK','EEF2'],
        ['DEK','NSD1'],
        ['DEK','SRPK1'],
        ['DENR','PERP'],
        ['DESI1','TSPAN6'],
        ['DFFA','HSF1'],
        ['DFFA','TLK1'],
        ['DGCR14','RNF31'],
        ['DGCR2','CALM3'],
        ['DGKD','LRRC14'],
        ['DGKE','MLH1'],
        ['DGKQ','LENG8'],
        ['DGUOK','METTL17'],
        ['DHCR24','MAPK1'],
        ['DHCR24','NDUFA13'],
        ['DHCR7','INTS5'],
        ['DHPS','WDR83OS'],
        ['DHRS1','RABGGTA'],
        ['DHRS2','HOXA5'],
        ['DHRSX','IFNAR2'],
        ['DHTKD1','ATP1A1'],
        ['DHX15','SIRT1'],
        ['DHX30','CTSB'],
        ['DHX30','H2AFZ'],
        ['DHX30','MET'],
        ['DHX32','ETFA'],
        ['DHX38','KDM1A'],
        ['DHX38','TP53'],
        ['DHX57','RAD51C'],
        ['DHX8','IER3IP1'],
        ['DHX9','PTCD3'],
        ['DHX9','ZFR'],
        ['DIAPH1','HAGH'],
        ['DIAPH3','MCOLN2'],
        ['DICER1','BOD1'],
        ['DICER1','DHX15'],
        ['DIP2B','TMEM201'],
        ['DIS3','GLA'],
        ['DIS3','LIMK1'],
        ['DIS3L','PRPS1'],
        ['DLAT','BRD1'],
        ['DLG3','PKP3'],
        ['DLGAP5','ITGB1'],
        ['DLST','TBC1D9B'],
        ['DMC1','DDX17'],
        ['DMTF1','VPS51'],
        ['DMXL2','SERINC1'],
        ['DNA2','CLN6'],
        ['DNAH14','XPO1'],
        ['DNAJA1','ADK'],
        ['DNAJA1','CLCN3'],
        ['DNAJA1','CNOT1'],
        ['DNAJA1','FYTTD1'],
        ['DNAJA2','CALR'],
        ['DNAJA2','LIN9'],
        ['DNAJA2','LRRC49'],
        ['DNAJA2','NIPBL'],
        ['DNAJA2','RPL31'],
        ['DNAJA3','DOCK7'],
        ['DNAJB11','ACTG1'],
        ['DNAJB11','TKT'],
        ['DNAJC10','LRPPRC'],
        ['DNAJC14','HSP90AA1'],
        ['DNAJC16','CSDE1'],
        ['DNAJC21','EIF3I'],
        ['DNAJC21','RPS21'],
        ['DNAJC2','PBK'],
        ['DNAJC2','PRIM2'],
        ['DNAJC2','TBC1D10B'],
        ['DNAJC7','RPL15'],
        ['DNAL1','COCH'],
        ['DNASE2','RCC1'],
        ['DNM1L','AKT2'],
        ['DNM1L','MLLT1'],
        ['DNM1L','MRPL42'],
        ['DNMT1','EHMT1'],
        ['DNMT1','VCP'],
        ['DNMT3A','IGH@'],
        ['DNMT3B','FAM168B'],
        ['DNPH1','CPSF3'],
        ['DNTTIP2','TUBA1A'],
        ['DOCK2','NDUFS3'],
        ['DOCK7','ALDOA'],
        ['DONSON','SIN3A'],
        ['DPCD','FBXW4'],
        ['DPH1','WDTC1'],
        ['DPH3','ZFP82'],
        ['DPYSL3','PSEN1'],
        ['DPYSL3','RNF113A'],
        ['DR1','TSFM'],
        ['DROSHA','ANKRD44'],
        ['DROSHA','SAR1B'],
        ['DSC2','PKM'],
        ['DSCC1','TAF2'],
        ['DSCR3','ZNF503-AS2'],
        ['DSG2','CBX1'],
        ['DSN1','PLEKHA5'],
        ['DSP','CENPJ'],
        ['DSP','FEM1C'],
        ['DST','COPS3'],
        ['DTD1','MRPS10'],
        ['DTL','LUM'],
        ['DTX3L','PDCD4'],
        ['DTX3','ZSWIM8'],
        ['DTX4','TRMT10C'],
        ['DTYMK','BLVRA'],
        ['DUS1L','DYNC1LI2'],
        ['DUS1L','PTGFRN'],
        ['DUS1L','TPT1'],
        ['DUS4L','EXOC4'],
        ['DUSP14','TPR'],
        ['DUSP15','EFTUD1'],
        ['DUSP16','LDAH'],
        ['DUSP22','ACTB'],
        ['DUT','ADD3'],
        ['DYM','COX18'],
        ['DYNC1H1','ACAT2'],
        ['DYNC1H1','BTBD3'],
        ['DYNC1LI2','PRELID1'],
        ['DYNLL1','C4ORF32'],
        ['DYNLRB1','ARHGEF4'],
        ['DYNLRB1','HNRNPDL'],
        ['DYNLRB1','SHTN1'],
        ['DYRK2','NELFB'],
        ['E2F1','TMEM183A'],
        ['E2F2','GPI'],
        ['E2F4','RPL14'],
        ['EAF2','ATP1A1'],
        ['EARS2','EXOC5'],
        ['EARS2','UBE2E1'],
        ['EBNA1BP2','CITED2'],
        ['EBNA1BP2','ORC6'],
        ['EBP','SUFU'],
        ['ECHS1','NOP56'],
        ['ECI1','INSIG1'],
        ['ECI2','GLUL'],
        ['ECT2','MPC1'],
        ['EDC4','SLC1A5'],
        ['EDEM1','SRP54'],
        ['EDEM3','AP1B1'],
        ['EDEM3','ZNF600'],
        ['EDF1','FAM204A'],
        ['EDF1','PPTC7'],
        ['EEA1','GNB2L1'],
        ['EEF1A1','ENO1'],
        ['EEF1A1','KRT81'],
        ['EEF1A1P5','FTH1'],
        ['EEF1A1P5','KRT81'],
        ['EEF1A1P6','FTH1'],
        ['EEF1A1P6','KRT81'],
        ['EEF1A1','PRC1'],
        ['EEF1A1','SNRPB'],
        ['EEF1B2','H3F3B'],
        ['EEF1D','CALR'],
        ['EEF1D','CKB'],
        ['EEF1D','DUSP28'],
        ['EEF1D','NAPRT1'],
        ['EEF1E1-BLOC1S5','NSMCE4A'],
        ['EEF1E1','NSMCE4A'],
        ['EEF1G','CCDC6'],
        ['EEF1G','IGFBP5'],
        ['EEF1G','TMED10'],
        ['EEF1G','TOMM20'],
        ['EEF1G','TYK2'],
        ['EEF2','AC006538.4'],
        ['EEF2','ARMCX3'],
        ['EEF2','ARSK'],
        ['EEF2','ATAD2B'],
        ['EEF2','BCAT1'],
        ['EEF2','CCT7'],
        ['EEF2','CDC25A'],
        ['EEF2','CHD4'],
        ['EEF2','COG6'],
        ['EEF2','DEK'],
        ['EEF2','DKC1'],
        ['EEF2','EIF2AK2'],
        ['EEF2','FLNB'],
        ['EEF2','GINS1'],
        ['EEF2','GSK3A'],
        ['EEF2','GUCD1'],
        ['EEF2','HMBS'],
        ['EEF2','HSPB8'],
        ['EEF2','IFRD1'],
        ['EEF2','INPP5D'],
        ['EEF2K','LMAN1'],
        ['EEF2','MARCKS'],
        ['EEF2','MIF'],
        ['EEF2','ORC1'],
        ['EEF2','PAQR3'],
        ['EEF2','PHLPP2'],
        ['EEF2','PRDX1'],
        ['EEF2','PRMT6'],
        ['EEF2','PSAT1'],
        ['EEF2','PTDSS1'],
        ['EEF2','RBM15'],
        ['EEF2','RNF14'],
        ['EEF2','SEPT2'],
        ['EEF2','SLC39A3'],
        ['EEF2','SNW1'],
        ['EEF2','SOX12'],
        ['EEF2','SRP9'],
        ['EEF2','TMEM181'],
        ['EEF2','UGGT1'],
        ['EEF2','XRCC1'],
        ['EEF2','ZNF8'],
        ['EFCAB7','PGM1'],
        ['EFTUD2','GAS5'],
        ['EFTUD2','MKRN1'],
        ['EGFL7','RBM28'],
        ['EGLN1','ILF2'],
        ['EHD4','MALAT1'],
        ['EHHADH','ARID4B'],
        ['EHMT1','RPS21'],
        ['EIF1AD','HECA'],
        ['EIF1AX','DHX9'],
        ['EIF1AX','GMEB2'],
        ['EIF1','SRRM2'],
        ['EIF2AK1','DDX17'],
        ['EIF2AK1','EHMT2'],
        ['EIF2AK2','DDX20'],
        ['EIF2A','POMP'],
        ['EIF2A','PTBP1'],
        ['EIF2B1','EIF4G2'],
        ['EIF2B4','SQLE'],
        ['EIF2D','TRIP12'],
        ['EIF2S1','ALDOA'],
        ['EIF2S1','SENP2'],
        ['EIF3A','CCNG1'],
        ['EIF3A','KIAA0226'],
        ['EIF3A','SF3B2'],
        ['EIF3A','SH3BGRL'],
        ['EIF3B','ABHD14B'],
        ['EIF3B','ARL5B'],
        ['EIF3B','C17ORF89'],
        ['EIF3B','PTGES3'],
        ['EIF3B','SLC35F6'],
        ['EIF3D','MCL1'],
        ['EIF3D','TPI1'],
        ['EIF3E','DIMT1'],
        ['EIF3E','GFM1'],
        ['EIF3E','LYRM9'],
        ['EIF3E','RPL11'],
        ['EIF3E','TUBB'],
        ['EIF3F','AC142472.6'],
        ['EIF3G','OAZ1'],
        ['EIF3G','TCF19'],
        ['EIF3H','KNOP1'],
        ['EIF3H','NUBP2'],
        ['EIF3H','SMC4'],
        ['EIF3H','SSTR3'],
        ['EIF3I','CDC37'],
        ['EIF3I','MED13L'],
        ['EIF3I','PGAM5'],
        ['EIF3I','VAPA'],
        ['EIF3L','INPP5A'],
        ['EIF3M','ERH'],
        ['EIF4A1','CTBP1'],
        ['EIF4A1','NOLC1'],
        ['EIF4A1','RPAIN'],
        ['EIF4A1','ZNF426'],
        ['EIF4B','DEK'],
        ['EIF4B','EEF2'],
        ['EIF4B','RPS11'],
        ['EIF4E2','COMMD3'],
        ['EIF4E2','VCAN'],
        ['EIF4G1','BUD31'],
        ['EIF4G1','RPL13A'],
        ['EIF4G1','RTFDC1'],
        ['EIF4G1','SERTAD2'],
        ['EIF4G2','CDC6'],
        ['EIF4G2','PHB'],
        ['EIF4G3','HP1BP3'],
        ['EIF4H','PTPN11'],
        ['EIF5A','BARX2'],
        ['EIF5A','COX5B'],
        ['EIF5A','LAMP1'],
        ['EIF5B','JMJD8'],
        ['EIF5B','PTMA'],
        ['EIF5','PSMA4'],
        ['EIF5','TOP2A'],
        ['ELAC2','P4HA1'],
        ['ELAVL1','CCND2'],
        ['ELF1','EID1'],
        ['ELF2','FAM101B'],
        ['ELK1','C19ORF43'],
        ['ELK4','IGF2R'],
        ['ELOVL1','RNF149'],
        ['ELOVL4','EIF3B'],
        ['ELOVL5','RAP2A'],
        ['ELP2','RAD21'],
        ['ELP2','VDAC3'],
        ['EMC2','SEC11C'],
        ['EMC3','SMC3'],
        ['EMC4','FOXK1'],
        ['EMC4','SGTA'],
        ['EMC7','NDC1'],
        ['EMD','ADAT2'],
        ['EMG1','MEX3D'],
        ['EMP2','PLIN5'],
        ['ENAH','AP2A1'],
        ['ENAH','TADA3'],
        ['ENGASE','DACT1'],
        ['ENO1','CNBP'],
        ['ENO1','DHCR24'],
        ['ENO1','DLC1'],
        ['ENO1','GOLGA1'],
        ['ENO1','GUK1'],
        ['ENO1','ITPRIPL2'],
        ['ENO1','MAGT1'],
        ['ENO1','MAP3K3'],
        ['ENO1','MCM5'],
        ['ENO1','MRPL53'],
        ['ENO1','MSH2'],
        ['ENO1','MTR'],
        ['ENO1','MTRR'],
        ['ENO1','NOC2L'],
        ['ENO1P4','DDOST'],
        ['ENO1','PAN2'],
        ['ENO1','PARK7'],
        ['ENO1','PRMT7'],
        ['ENO1','PSMD7'],
        ['ENO1','REXO4'],
        ['ENO1','RPL30'],
        ['ENO1','SCAND1'],
        ['ENO1','SHMT1'],
        ['ENO1','WDR54'],
        ['ENO2','HDLBP'],
        ['ENOX2','MCFD2'],
        ['ENPP2','KIF1C'],
        ['ENPP4','PSMA4'],
        ['ENSA','FAM168A'],
        ['ENTHD2','MAPKAP1'],
        ['ENTPD1','KIAA0922'],
        ['ENTPD1','SOX2-OT'],
        ['ENTPD5','FAM161B'],
        ['ENY2','USF1'],
        ['EOGT','GLUL'],
        ['EP300','L3MBTL2'],
        ['EPB41L3','GPS1'],
        ['EPB41L4B','PKM'],
        ['EPB41','TCP1'],
        ['EPB41','UPF1'],
        ['EPCAM','ARL6IP5'],
        ['EPCAM','ZNF704'],
        ['EPG5','RDH11'],
        ['EPHA3','YWHAH'],
        ['EPOR','TMEM41A'],
        ['EPRS','LRP3'],
        ['EPRS','MPPE1'],
        ['EPRS','NOLC1'],
        ['EPRS','RHOA'],
        ['EPRS','RPS16'],
        ['EPS8','CIRBP'],
        ['ERAL1','RPS17'],
        ['ERCC3','ARID2'],
        ['ERGIC1','MDC1'],
        ['ERGIC2','DHX15'],
        ['ERGIC2','MFSD11'],
        ['ERGIC3','ACTG1'],
        ['ERGIC3','C1QBP'],
        ['ERI1','ASF1B'],
        ['ERI3','ADAM10'],
        ['ERLEC1','UBL4A'],
        ['ERMP1','C12ORF49'],
        ['ERN1','ZNF445'],
        ['ERO1LB','BCR'],
        ['ERO1L','MB21D2'],
        ['ERO1L','THAP10'],
        ['ERP29','NCBP1'],
        ['ERVMER34-1','SPTBN1'],
        ['ESCO1','COPE'],
        ['ESD','JUN'],
        ['ESD','ZBTB18'],
        ['ESF1','TBC1D4'],
        ['ESPL1','GPX4'],
        ['ESPL1','IRF2BP2'],
        ['ESPL1','LIN28B'],
        ['ESPL1','NABP2'],
        ['ESYT1','ILF2'],
        ['ETF1','ATP6AP1'],
        ['ETF1','HNRNPA2B1'],
        ['ETF1','RTCB'],
        ['ETF1','U2SURP'],
        ['ETFA','EEF1G'],
        ['ETFA','EEF2'],
        ['ETFA','FLII'],
        ['ETFA','MED30'],
        ['ETFB','RNF13'],
        ['ETFDH','HAUS6'],
        ['ETFDH','RPS6'],
        ['ETNK1','MCM6'],
        ['ETS1','EIF4G2'],
        ['EVA1A','POLR2B'],
        ['EVC','NIPA2'],
        ['EVL','IGH@'],
        ['EXD2','LATS2'],
        ['EXD2','NOL7'],
        ['EXO1','MICU1'],
        ['EXOC1','PABPC1L'],
        ['EXOC5','ZDBF2'],
        ['EXOC6B','ZNF593'],
        ['EXOC6','MARCH5'],
        ['EXOC6','NDUFA13'],
        ['EXOC8','RAB21'],
        ['EXOSC10','KDM1A'],
        ['EXOSC2','ACTG1'],
        ['EXOSC3','CORO1C'],
        ['EXOSC6','C20ORF24'],
        ['EXOSC6','SLC30A5'],
        ['EZR','CRIM1'],
        ['EZR','EDF1'],
        ['EZR','MFN2'],
        ['F12','GIPC1'],
        ['F2RL1','RSF1'],
        ['FAAHP1','LINC00505'],
        ['FAM101B','TERF2IP'],
        ['FAM103A1','PLS3'],
        ['FAM120AOS','DLG5'],
        ['FAM120AOS','GMCL1'],
        ['FAM129A','LAT2'],
        ['FAM129A','SAC3D1'],
        ['FAM129B','PDCD4'],
        ['FAM129C','COLGALT1'],
        ['FAM134C','CCNB1'],
        ['FAM134C','PPHLN1'],
        ['FAM135A','LGALS3BP'],
        ['FAM160B2','NACA'],
        ['FAM168B','DENND6A'],
        ['FAM168B','TXNL1'],
        ['FAM172A','ANKRD32'],
        ['FAM173A','PALD1'],
        ['FAM177A1','ATP1A1'],
        ['FAM184A','RPL11'],
        ['FAM189B','RER1'],
        ['FAM195A','AASDHPPT'],
        ['FAM199X','VMA21'],
        ['FAM200B','SRSF6'],
        ['FAM208A','CPVL'],
        ['FAM208B','GDI2'],
        ['FAM208B','TOP2B'],
        ['FAM20B','TKT'],
        ['FAM210B','FUCA2'],
        ['FAM217B','YTHDF2'],
        ['FAM220A','FAM134B'],
        ['FAM229B','DNTTIP2'],
        ['FAM49B','FOXP4'],
        ['FAM50A','PPP1CA'],
        ['FAM53C','ANXA5'],
        ['FAM53C','PHF2'],
        ['FAM60A','PYURF'],
        ['FAM73A','VIM'],
        ['FAM73B','PREP'],
        ['FAM98A','LSP1'],
        ['FANCD2','HEBP2'],
        ['FANCI','FAM96B'],
        ['FANCI','WDR43'],
        ['FAR2','SCARB1'],
        ['FARSA','FAM13B'],
        ['FARSA','KDM3A'],
        ['FARSB','AP3D1'],
        ['FASN','DHX38'],
        ['FASN','FUCA2'],
        ['FASN','MAN1A1'],
        ['FASTKD3','USP28'],
        ['FAU','RPL8'],
        ['FBLIM1','NQO1'],
        ['FBL','OCIAD1'],
        ['FBL','ZNF274'],
        ['FBN2','CSNK2A2'],
        ['FBXO17','PSPC1'],
        ['FBXO3','TMPO'],
        ['FBXO41','VCP'],
        ['FBXO46','S1PR1'],
        ['FBXO7','PEBP1'],
        ['FBXW11','HDAC2'],
        ['FBXW11','KPNB1'],
        ['FBXW11','RBSN'],
        ['FBXW2','MAPRE2'],
        ['FBXW9','SELENBP1'],
        ['FCER2','XIST'],
        ['FDFT1','IRF2BP1'],
        ['FDFT1','NDUFB5'],
        ['FDFT1','RBBP7'],
        ['FER','STAG2'],
        ['FEZ2','AHSA1'],
        ['FEZ2','RPL37A'],
        ['FGD4','ARNTL2'],
        ['FGD5-AS1','CTSZ'],
        ['FGF11','RPP14'],
        ['FGFR3','RRAGB'],
        ['FIP1L1','GFPT1'],
        ['FKBP10','SERBP1'],
        ['FKBP14','LBR'],
        ['FKBP3','MCRS1'],
        ['FKBP3','TMEM99'],
        ['FKBP3','TRIM21'],
        ['FKBP4','GAS2L3'],
        ['FKBP4','MCM3'],
        ['FKBP4','SPECC1'],
        ['FKBP4','STIP1'],
        ['FKBP5','PYCRL'],
        ['FKBP8','PCGF6'],
        ['FLAD1','PRDX5'],
        ['FLII','GPR137B'],
        ['FLNA','CAPN1'],
        ['FLNA','HDGFRP2'],
        ['FLNA','MTATP6P1'],
        ['FLNA','NSF'],
        ['FLNA','PARN'],
        ['FLNA','PPIL2'],
        ['FLNA','TAF3'],
        ['FLNA','UBE3C'],
        ['FLNB','CARD8'],
        ['FLNB','FLNA'],
        ['FLYWCH1','ASPRV1'],
        ['FMR1','PAK1IP1'],
        ['FMR1','TMX4'],
        ['FN1','ACTB'],
        ['FN1','MTATP6P1'],
        ['FN1','PTRF'],
        ['FN1','SDCCAG8'],
        ['FNBP1','TCEB2'],
        ['FNBP4','DDX24'],
        ['FNDC3B','CCT7'],
        ['FNIP1','SMCHD1'],
        ['FOPNL','TXN'],
        ['FOXD1','LRPAP1'],
        ['FOXJ2','RPS19'],
        ['FOXJ3','PSMA4'],
        ['FOXM1','MYCBP2'],
        ['FOXM1','PCSK1N'],
        ['FOXN2','GNB2L1'],
        ['FOXO1','HSD17B10'],
        ['FOXO6','SRSF1'],
        ['FRAT2','RPS14'],
        ['FRS2','CPSF6'],
        ['FSCN1','SYMPK'],
        ['FSTL1','BUB3'],
        ['FSTL1','MRPL55'],
        ['FTH1','MT2A'],
        ['FTL','ACSL3'],
        ['FTL','NF2'],
        ['FTL','RHOB'],
        ['FTSJ1','PSMB1'],
        ['FUBP1','HSP90AB1'],
        ['FUBP1','PTPN1'],
        ['FUBP3','CHCHD1'],
        ['FUNDC1','PES1'],
        ['FUS','CCDC91'],
        ['FXR1','GLA'],
        ['FXR1','PINK1'],
        ['FZD8','KATNA1'],
        ['FZR1','DCAF12'],
        ['G2E3','RPL26'],
        ['G2E3','SCFD1'],
        ['G2E3','SEPT11'],
        ['G3BP1','AFAP1'],
        ['G3BP1','ALDH1B1'],
        ['G3BP1','DHX16'],
        ['G3BP1','PSMB5'],
        ['G3BP2','HCCS'],
        ['G6PD','WI2-89031B12.1'],
        ['GABPB2','ARPC5L'],
        ['GABRB3','LAPTM4B'],
        ['GALE','PKM'],
        ['GALM','GNAZ'],
        ['GALNT10','IARS'],
        ['GALNT11','PCDH9'],
        ['GALNT1','TRAP1'],
        ['GALNT2','DAZAP2'],
        ['GANAB','NEAT1'],
        ['GAPDH','IGH@'],
        ['GAPDH','KRT17'],
        ['GAPDH','KRT5'],
        ['GAPDH','MTATP6P1'],
        ['GAPDH','NUP160'],
        ['GAPDH','TCP1'],
        ['GAPDH','TMEM131'],
        ['GAPDH','TMEM242'],
        ['GAPDH','VPS36'],
        ['GAPVD1','EIF3L'],
        ['GARS','OAZ1'],
        ['GARS','TCF20'],
        ['GART','OSBPL1A'],
        ['GART','TPCN1'],
        ['GATB','GOT2'],
        ['GATC','MCUR1'],
        ['GATC','ZCCHC2'],
        ['GBGT1','CRKL'],
        ['GCAT','CALR'],
        ['GCC2','PHIP'],
        ['GCFC2','RRN3'],
        ['GCLM','NOL12'],
        ['GCN1L1','TOMM20'],
        ['GDE1','ARSB'],
        ['GDE1','TYRO3'],
        ['GDF11','EIF3H'],
        ['GDF11','SSB'],
        ['GDI2','ATP6V1A'],
        ['GDI2','EBP'],
        ['GDI2','KCTD20'],
        ['GDI2','PPIB'],
        ['GEMIN2','RBM15B'],
        ['GEMIN4','ABHD17B'],
        ['GEMIN5','HN1'],
        ['GEMIN5','RPL38'],
        ['GEMIN5','USP22'],
        ['GEMIN6','KLHL36'],
        ['GEN1','LDHB'],
        ['GFM1','CDT1'],
        ['GFOD2','AKT1'],
        ['GGA2','GIGYF2'],
        ['GGCX','ATXN1'],
        ['GGH','DDX5'],
        ['GID8','SARS'],
        ['GIGYF2','GNAS-AS1'],
        ['GIGYF2','ZNF292'],
        ['GINS1','AMFR'],
        ['GINS1','EIF4A1'],
        ['GINS1','SENP3-EIF4A1'],
        ['GIT1','AP2A2'],
        ['GLA','ACTG1'],
        ['GLA','HSPH1'],
        ['GLA','LIMD2'],
        ['GLA','N4BP2'],
        ['GLA','PHKA1'],
        ['GLA','SPAG5'],
        ['GLA','TM9SF2'],
        ['GLA','XYLT2'],
        ['GLB1L','PTER'],
        ['GLDC','OSBPL11'],
        ['GLDC','PTPN23'],
        ['GLE1','CAPNS1'],
        ['GLE1','FAM49B'],
        ['GLMN','TCEAL4'],
        ['GLO1','C17ORF51'],
        ['GLO1','RNF34'],
        ['GLO1','STAG2'],
        ['GLO1','TIMM13'],
        ['GLRX5','TMEM43'],
        ['GLS','HLCS'],
        ['GLS','NTN1'],
        ['GLTP','MCCC2'],
        ['GLUD1','IFNAR2'],
        ['GLUD1','MARCKS'],
        ['GLUL','ATP11C'],
        ['GLUL','COX8A'],
        ['GLUL','MAGI2'],
        ['GLUL','STARD7'],
        ['GLYCTK','DNAH1'],
        ['GM2A','HDAC2'],
        ['GMFB','GATAD2A'],
        ['GMFB','PSMA7'],
        ['GMFB','RPL37A'],
        ['GMPPB','MCM2'],
        ['GMPR2','TAF7'],
        ['GMPS','RBBP8'],
        ['GNA11','ATP5A1'],
        ['GNA11','BCAT2'],
        ['GNA11','MRPL12'],
        ['GNA11','ZWINT'],
        ['GNA12','ANXA5'],
        ['GNAI2','CCDC88B'],
        ['GNAI2','IGH@'],
        ['GNAI2','IGHA1'],
        ['GNAI3','HSPB11'],
        ['GNAI3','PIM3'],
        ['GNAI3','PYCR2'],
        ['GNAI3','RMND5A'],
        ['GNAL','DAZAP1'],
        ['GNAS','RPL3'],
        ['GNAS','WDR83OS'],
        ['GNAS','YWHAB'],
        ['GNB1','HNRNPU'],
        ['GNB1','ZFP90'],
        ['GNB2L1','AGFG1'],
        ['GNB2L1','ALDH18A1'],
        ['GNB2L1','CD109'],
        ['GNB2L1','DBP'],
        ['GNB2L1','ENO1'],
        ['GNB2L1','GGH'],
        ['GNB2L1','GIPC1'],
        ['GNB2L1','IL13RA1'],
        ['GNB2L1','KDM5B'],
        ['GNB2L1','KIAA1919'],
        ['GNB2L1','KIF14'],
        ['GNB2L1','MRPL38'],
        ['GNB2L1','MTMR1'],
        ['GNB2L1','NAA38'],
        ['GNB2L1','PHB2'],
        ['GNB2L1','PRPS2'],
        ['GNB2L1','RPN2'],
        ['GNB2L1','SEC13'],
        ['GNB2L1','SNAPIN'],
        ['GNB2L1','SNRPD3'],
        ['GNB2L1','ZNF512'],
        ['GNB4','CBWD2'],
        ['GNB4','CCT4'],
        ['GNB5','ARPP19'],
        ['GNE','SDE2'],
        ['GNG11','RPS15A'],
        ['GNG12','HMGB3'],
        ['GNG12','NCAPD2'],
        ['GNG5','CCT3'],
        ['GNL2','NEAT1'],
        ['GNL2','RPS8'],
        ['GNL3','ADSL'],
        ['GNPAT','SEH1L'],
        ['GOLGA1','PES1'],
        ['GOLGA1','SLC30A5'],
        ['GOLGA3','PSMA3'],
        ['GOLGA3','ZNF445'],
        ['GOLGA4','GSPT1'],
        ['GOLGA4','KREMEN1'],
        ['GOLGA4','RNF8'],
        ['GOLGB1','PRPF31'],
        ['GOLM1','CDC23'],
        ['GOLM1','FLNA'],
        ['GOLPH3','MTDH'],
        ['GOLPH3','MXRA7'],
        ['GOPC','CANX'],
        ['GOPC','SFXN1'],
        ['GORASP2','ELAVL1'],
        ['GPAA1','DDB1'],
        ['GPAA1','GRPEL1'],
        ['GPBP1L1','TOP3A'],
        ['GPI','ARF5'],
        ['GPI','DDAH1'],
        ['GPI','DEDD2'],
        ['GPI','MLLT3'],
        ['GPI','SCD'],
        ['GPI','SUPT4H1'],
        ['GPI','THAP4'],
        ['GPM6B','SLIT2'],
        ['GPN2','MPC2'],
        ['GPR107','GINS1'],
        ['GPR107','NUCKS1'],
        ['GPR107','ZC3H7A'],
        ['GPR157','MESDC1'],
        ['GPR180','HEXDC'],
        ['GPRC5C','MKNK1'],
        ['GPS1','ENOPH1'],
        ['GPS1','GRPEL1'],
        ['GPS1','RPS19'],
        ['GPS2','TIMM17B'],
        ['GRAMD4','MARCKS'],
        ['GRB2','KIAA1109'],
        ['GRB2','TSTA3'],
        ['GRHPR','JMJD4'],
        ['GRK6','ENO2'],
        ['GRK6','TOP2A'],
        ['GRSF1','PSMD8'],
        ['GS1-358P8.4','PBK'],
        ['GSPT1','SAC3D1'],
        ['GSPT1','ZMYND11'],
        ['GSTO1','ENSA'],
        ['GSTO1','RBMX'],
        ['GSTP1','LINC00998'],
        ['GSTP1','PTPN11'],
        ['GSTP1','SMIM12'],
        ['GTF2A1','RRAGD'],
        ['GTF2A2','PCGF5'],
        ['GTF2F1','ZFR'],
        ['GTF2F2','HSPA1A'],
        ['GTF2H1','PHF23'],
        ['GTF3A','ALKBH1'],
        ['GTF3A','CCP110'],
        ['GTF3A','KBTBD4'],
        ['GTF3C1','ALDOA'],
        ['GTF3C1','CLPTM1'],
        ['GTF3C1','TMSB4X'],
        ['GTPBP2','CDK6'],
        ['GTPBP4','NEMF'],
        ['GTPBP4','SRSF1'],
        ['GUF1','ERLIN1'],
        ['H1FX','EIF3B'],
        ['H2AFX','MCM5'],
        ['H2AFY','AARSD1'],
        ['H2AFY','KIRREL2'],
        ['H2AFY','MCM5'],
        ['H2AFY','PTGES3L-AARSD1'],
        ['H2AFZ','MKLN1'],
        ['H2AFZ','PPP2R5C'],
        ['H3F3B','ARL6IP4'],
        ['H3F3B','GHITM'],
        ['HACD1','CCT5'],
        ['HACD2','NRD1'],
        ['HACD3','LCP1'],
        ['HACD3','ZNF146'],
        ['HACL1','RPS12'],
        ['HACL1','SNRPD3'],
        ['HADHA','PAQR3'],
        ['HADHA','SOAT1'],
        ['HADHA','UBA52'],
        ['HADHB','ARID1B'],
        ['HADHB','CBX5'],
        ['HADHB','HEATR5A'],
        ['HADHB','UCHL1'],
        ['HAGH','CALR'],
        ['HARS','MMGT1'],
        ['HARS','NPM3'],
        ['HAUS5','KPNB1'],
        ['HCFC1','TMEM219'],
        ['HCFC1','TMSB10'],
        ['HCFC2','EIF3E'],
        ['HCLS1','IGF2R'],
        ['HDAC10','CYTH1'],
        ['HDAC1','HN1'],
        ['HDAC2','CHCHD2'],
        ['HDAC2','LMAN2'],
        ['HDAC3','DIAPH1'],
        ['HDAC3','TJP2'],
        ['HDAC4','ZNF585B'],
        ['HDAC5','CLCC1'],
        ['HDAC8','CITED1'],
        ['HDGF','PRRC2A'],
        ['HDGFRP2','HRASLS'],
        ['HDGFRP2','WDR18'],
        ['HDGFRP3','BAP1'],
        ['HDGFRP3','C14ORF169'],
        ['HDLBP','BIRC5'],
        ['HDLBP','EAF1'],
        ['HDLBP','HNRNPH1'],
        ['HDLBP','MRPL21'],
        ['HDLBP','PTPRG'],
        ['HDLBP','SIN3B'],
        ['HDLBP','TUBA1B'],
        ['HDLBP','WDR33'],
        ['HEATR1','RPS6'],
        ['HEATR1','TMEM50A'],
        ['HEATR5B','CHCHD5'],
        ['HECA','PABPC4'],
        ['HECA','TBC1D1'],
        ['HECTD1','RPS4X'],
        ['HEG1','B2M'],
        ['HELLS','CDC25C'],
        ['HELLS','COPE'],
        ['HELZ','EIF3I'],
        ['HERC1','NDUFA8'],
        ['HES4','MALAT1'],
        ['HEY1','ACTB'],
        ['HHEX','EIF2S3'],
        ['HIC1','SUPT7L'],
        ['HIF1A','ALKBH5'],
        ['HIF1AN','OGT'],
        ['HINFP','ZBED1'],
        ['HINT1','RBBP9'],
        ['HINT1','STAT5B'],
        ['HIPK3','CDS2'],
        ['HIPK3','PHKB'],
        ['HIPK3','PTGFRN'],
        ['HIRIP3','CLTC'],
        ['HIST1H4D','MCPH1'],
        ['HIST3H2A','MON1B'],
        ['HK1','CCT7'],
        ['HLCS','NUF2'],
        ['HLTF','MKNK2'],
        ['HMCES','GTPBP8'],
        ['HMG20B','SETX'],
        ['HMGA1','HIF1A'],
        ['HMGA2','C11ORF24'],
        ['HMGB1','PIH1D1'],
        ['HMGB2','APRT'],
        ['HMGB2','DARS'],
        ['HMGB2','ZRANB1'],
        ['HMGB3','CCDC122'],
        ['HMGCS1','GRPEL2'],
        ['HMGN3-AS1','RBM39'],
        ['HMGXB3','BRD9'],
        ['HMGXB4','NRDE2'],
        ['HMOX2','TTYH3'],
        ['HN1L','RASSF7'],
        ['HN1L','YARS'],
        ['HN1L','ZNF714'],
        ['HN1','RUFY2'],
        ['HNRNPA0','DOPEY1'],
        ['HNRNPA0','OSBP'],
        ['HNRNPA2B1','C1QBP'],
        ['HNRNPA2B1','DDX21'],
        ['HNRNPA2B1','STAMBP'],
        ['HNRNPA2B1','TXLNG'],
        ['HNRNPAB','ACADSB'],
        ['HNRNPAB','AKT2'],
        ['HNRNPC','MTHFD2'],
        ['HNRNPC','SUCLA2'],
        ['HNRNPDL','TGS1'],
        ['HNRNPF','BIRC5'],
        ['HNRNPF','DNAJC6'],
        ['HNRNPF','GPR137'],
        ['HNRNPF','IMPDH2'],
        ['HNRNPF','MRPL14'],
        ['HNRNPH1','C19ORF43'],
        ['HNRNPH1','VIM'],
        ['HNRNPH2','ALKBH5'],
        ['HNRNPH3','CD74'],
        ['HNRNPH3','DARS'],
        ['HNRNPH3','HNRNPUL2'],
        ['HNRNPH3','HNRNPUL2-BSCL2'],
        ['HNRNPH3','TSPAN14'],
        ['HNRNPK','GPD1L'],
        ['HNRNPK','MCRS1'],
        ['HNRNPLL','EIF3B'],
        ['HNRNPLL','PA2G4'],
        ['HNRNPM','DPH1'],
        ['HNRNPM','OVCA2'],
        ['HNRNPM','P4HB'],
        ['HNRNPM','TPI1'],
        ['HNRNPR','GADD45A'],
        ['HNRNPR','TPD52L2'],
        ['HNRNPU','EXOSC7'],
        ['HNRNPU','FNIP1'],
        ['HNRNPUL1','SSR2'],
        ['HNRNPUL1','TAF6'],
        ['HNRNPUL1','TSN'],
        ['HNRNPUL2-BSCL2','EEF1G'],
        ['HNRNPU','LLFOS-48D6.2'],
        ['HNRNPU','OAZ1'],
        ['HNRNPU','PARP1'],
        ['HNRNPU','RPS4X'],
        ['HNRNPU','WTAP'],
        ['HOMER1','ERMARD'],
        ['HOXA5','NOP10'],
        ['HOXA-AS2','CCT3'],
        ['HPS6','RSPRY1'],
        ['HS2ST1','CRACR2B'],
        ['HS6ST3','UQCRB'],
        ['HSBP1','WDHD1'],
        ['HSCB','VPS26B'],
        ['HSD17B10','YTHDF3'],
        ['HSD17B11','CDKN3'],
        ['HSD17B4','RPL36'],
        ['HSDL2','GABPB1'],
        ['HSDL2','MYC'],
        ['HSF1','RSF1'],
        ['HSP90AA1','BECN1'],
        ['HSP90AA1','CAMK1D'],
        ['HSP90AA1','LIN28B'],
        ['HSP90AA1','RRAGD'],
        ['HSP90AA1','TBC1D23'],
        ['HSP90AA1','TMED9'],
        ['HSP90AA1','TRIT1'],
        ['HSP90AA1','TSPAN3'],
        ['HSP90AB1','BRF1'],
        ['HSP90AB1','EIF3G'],
        ['HSP90AB1','NPM1'],
        ['HSP90AB1','SLC6A15'],
        ['HSP90AB1','SMC1A'],
        ['HSP90AB1','XRCC5'],
        ['HSP90B1','BOP1'],
        ['HSP90B1','PPP2R1A'],
        ['HSP90B1','VCAN'],
        ['HSPA13','RBM27'],
        ['HSPA14','SEC31A'],
        ['HSPA14','TIMM10B'],
        ['HSPA1B','ACTG1'],
        ['HSPA1B','LRPPRC'],
        ['HSPA1B','VCL'],
        ['HSPA4','KIF23'],
        ['HSPA9','ATP11B'],
        ['HSPA9','BUB3'],
        ['HSPA9','CCDC59'],
        ['HSPA9','GIGYF1'],
        ['HSPA9','TMTC4'],
        ['HSPD1','RPL7L1'],
        ['HSPH1','EIF4G2'],
        ['HSPH1','U2SURP'],
        ['HTATIP2','PSMB3'],
        ['HTATSF1','COX5B'],
        ['HTATSF1','MRPS11'],
        ['HTR3A','DAPK1'],
        ['HTR3A','ETFDH'],
        ['HUWE1','FIRRE'],
        ['HUWE1','HNRNPA2B1'],
        ['HUWE1','MTOR'],
        ['HUWE1','PSD4'],
        ['HUWE1','SRSF7'],
        ['HYKK','PSMA4'],
        ['HYLS1','HID1'],
        ['HYOU1','MYH9'],
        ['HYOU1','RPL18'],
        ['IARS2','TMEM259'],
        ['IARS','AC072062.1'],
        ['IARS','CANX'],
        ['IARS','CMTM6'],
        ['IARS','FAIM'],
        ['IARS','GAPDH'],
        ['IARS','MYEOV2'],
        ['ICMT','CCT7'],
        ['ICMT','CRNKL1'],
        ['ICMT','MYL6'],
        ['ICT1','H2AFZ'],
        ['ID2','ASNA1'],
        ['IDH2','MIF'],
        ['IDH2','RPS11'],
        ['IDH3B','ATP2A2'],
        ['IDH3B','CCNL1'],
        ['IDH3B','DSTN'],
        ['IDH3B','GUCD1'],
        ['IDI1','ALDOA'],
        ['IER3IP1','TAF6'],
        ['IER5','ENO1'],
        ['IFI30','MKI67'],
        ['IFNGR1','MRPL28'],
        ['IFNGR1','SAFB'],
        ['IFT27','GCDH'],
        ['IFT52','EIF1AX'],
        ['IGF2BP1','TPI1'],
        ['IGF2R','CSE1L'],
        ['IGF2R','MCU'],
        ['IGF2R','TBC1D23'],
        ['IGHA1','SEC24B'],
        ['IGHA1','SND1'],
        ['IGH@','ADCY3'],
        ['IGH@','ALDOA'],
        ['IGH@','ARPC4-TTLL3'],
        ['IGH@','AUTS2'],
        ['IGH@','CENPW'],
        ['IGH@','CYSLTR1'],
        ['IGH@','DENND5A'],
        ['IGH@','DLGAP4'],
        ['IGH@','DOK3'],
        ['IGH@','FTL'],
        ['IGH@','ITPKB'],
        ['IGH@','IVD'],
        ['IGH@','KCNAB2'],
        ['IGH@','LDHA'],
        ['IGHM','DLGAP4'],
        ['IGHM','DOK3'],
        ['IGH@','P4HB'],
        ['IGH@','PA2G4'],
        ['IGH@','PDCD6IP'],
        ['IGH@','PITPNB'],
        ['IGH@','PKM'],
        ['IGH@','PLBD2'],
        ['IGH@','PREX1'],
        ['IGH@','RRBP1'],
        ['IGH@','SCPEP1'],
        ['IGH@','SEC24B'],
        ['IGH@','SH2D3C'],
        ['IGH@','SLC2A1'],
        ['IGH@','SND1'],
        ['IGH@','SRRM2'],
        ['IGH@','TBC1D10C'],
        ['IGH@','TCIRG1'],
        ['IGH@','TLN1'],
        ['IGH@','TRABD'],
        ['IGH@','TTLL3'],
        ['IGHV3-21','DOK3'],
        ['IGHV3-21','PIGO'],
        ['IGH@','WDFY4'],
        ['IGH@','XIST'],
        ['IGH@','YBX1'],
        ['IGH@','ZNF318'],
        ['IGLV2-23','FBXO11'],
        ['IKBKAP','STAMBP'],
        ['IKZF5','ATP8B2'],
        ['IL10RB-AS1','FLNA'],
        ['IL10RB-AS1','VPS13C'],
        ['IL3RA','SLAMF7'],
        ['IL4R','EIF4A1'],
        ['ILF2','EEF2'],
        ['ILF2','NLRX1'],
        ['ILF3','DNAJA2'],
        ['ILF3','EIF2S3'],
        ['ILF3','YWHAE'],
        ['ILF3','ZBTB4'],
        ['IMMP2L','CSDE1'],
        ['IMMT','PTCD3'],
        ['IMP3','STX10'],
        ['IMPA1','SLBP'],
        ['IMPA2','STIP1'],
        ['IMPAD1','CKAP4'],
        ['IMPAD1','PTMA'],
        ['IMPDH2','DBN1'],
        ['IMPDH2','POR'],
        ['IMPDH2','TIA1'],
        ['ING3','WSB1'],
        ['ING4','TRIB2'],
        ['ING5','RABL6'],
        ['INPP4A','ANKRD11'],
        ['INPP4A','PRSS53'],
        ['INPP5B','HSP90AB1'],
        ['INPP5D','PAGR1'],
        ['INSIG1','FECH'],
        ['INTS10','GOLGA5'],
        ['INTS10','NR2F6'],
        ['INTS8','CCT3'],
        ['INTS8','GALNT1'],
        ['INTS9','HSPA1A'],
        ['INVS','MYDGF'],
        ['IPO5','CLTC'],
        ['IPO5','RRP8'],
        ['IPPK','GBE1'],
        ['IQGAP2','BUB3'],
        ['IQSEC1','OGDH'],
        ['IRAK1','ACTR1A'],
        ['IRAK1','ATP6V1G2-DDX39B'],
        ['IRAK1','DDX39B'],
        ['IRAK1','EID1'],
        ['IRAK1','GPC3'],
        ['IRAK1','SMC1A'],
        ['IREB2','TXLNG'],
        ['IRF2BP2','PREB'],
        ['IRF2BP2','RASEF'],
        ['IRF2BP2','RPL4'],
        ['IRF2BPL','KCNG1'],
        ['IRF2','ZER1'],
        ['IRF6','C1orf74'],
        ['IRS2','DSTN'],
        ['ISG20L2','RPL19'],
        ['ISOC1','CLIC4'],
        ['ITCH','TRMT1'],
        ['ITGA6','GGPS1'],
        ['ITGB1BP1','STARD7'],
        ['ITGB1','RPS14'],
        ['ITGB1','UBE2D3'],
        ['ITPRIPL1','B2M'],
        ['ITPRIPL2','MAZ'],
        ['IWS1','MRPL37'],
        ['IWS1','NENF'],
        ['JAK1','CANX'],
        ['JAK1','LDHB'],
        ['JAK1','NFKB2'],
        ['JKAMP','CCT5'],
        ['JKAMP','PHPT1'],
        ['JMJD6','DAGLB'],
        ['JMJD6','KIF5B'],
        ['JSRP1','CORO1A'],
        ['JTB','PANK3'],
        ['JTB','RPL5'],
        ['JUNB','MIS18BP1'],
        ['KANSL1','SURF4'],
        ['KANSL2','TUFM'],
        ['KARS','CAPZA2'],
        ['KARS','DHX30'],
        ['KAT6A','ACTB'],
        ['KCNC4','RBM15'],
        ['KCNJ8','SLC25A6'],
        ['KCNK12','ETS1'],
        ['KCTD17','FCRLA'],
        ['KCTD2','TRIM80P'],
        ['KCTD3','CHMP2A'],
        ['KDELR1','CLIC4'],
        ['KDELR2','EI24'],
        ['KDELR2','POLR2E'],
        ['KDELR3','PSMD1'],
        ['KDM1A','DDRGK1'],
        ['KDM1A','TMX1'],
        ['KDM2A','INTS8'],
        ['KDM2A','PNISR'],
        ['KDM2B','FZR1'],
        ['KDM4B','G3BP2'],
        ['KDM4B','MAGEF1'],
        ['KDM4C','CCDC9'],
        ['KDM5A','BROX'],
        ['KDM5A','MPZL1'],
        ['KDM5C','CCT6A'],
        ['KDM5C','TMED7'],
        ['KDM5C','TMED7-TICAM2'],
        ['KDM5C','ZDHHC2'],
        ['KDM6A','RPS19'],
        ['KHDC1','EXOC2'],
        ['KHK','RPL39'],
        ['KIAA0020','IPO5'],
        ['KIAA0101','ANGEL1'],
        ['KIAA0101','ASUN'],
        ['KIAA0101','NRD1'],
        ['KIAA0125','IGH@'],
        ['KIAA0391','CNBP'],
        ['KIAA0391','WWC2'],
        ['KIAA0513','SYAP1'],
        ['KIAA1024','STARD7'],
        ['KIAA1107','MYH10'],
        ['KIAA1161','ZBTB41'],
        ['KIAA1191','ALDH3A2'],
        ['KIAA1191','ATAD2'],
        ['KIAA1429','AGO1'],
        ['KIAA1468','PCNA'],
        ['KIAA1524','PGRMC1'],
        ['KIAA1524','RPL38'],
        ['KIDINS220','CAMK4'],
        ['KIF11','RPL11'],
        ['KIF14','PCBP1'],
        ['KIF14','VARS'],
        ['KIF14','ZNF460'],
        ['KIF16B','FEM1B'],
        ['KIF1B','FAM208A'],
        ['KIF1BP','CDCA7'],
        ['KIF1C','ABHD17C'],
        ['KIF1C','CCPG1'],
        ['KIF1C','DYX1C1-CCPG1'],
        ['KIF20A','GTF3C1'],
        ['KIF20A','RBBP7'],
        ['KIF20B','EEF2'],
        ['KIF20B','MFAP1'],
        ['KIF20B','PCM1'],
        ['KIF20B','RPL37A'],
        ['KIF21A','YBX3'],
        ['KIF22','PLS3'],
        ['KIF22','RFC2'],
        ['KIF22','SLC25A3'],
        ['KIF3B','CALR'],
        ['KIF3B','MOAP1'],
        ['KIFC2','EEF2'],
        ['KLF13','PKN1'],
        ['KLF16','TMEM127'],
        ['KLF3','CERS5'],
        ['KLHDC3','ALDOA'],
        ['KLHDC3','FOXK1'],
        ['KLHL15','EEF2'],
        ['KLHL15','HOXD13'],
        ['KLHL15','NRN1'],
        ['KLHL15','NUCB2'],
        ['KLHL15','WDR7'],
        ['KLHL15','ZNF316'],
        ['KLHL36','GNG12'],
        ['KLHL36','PEX5'],
        ['KLHL9','FAM13B'],
        ['KMT2C','NUDT22'],
        ['KMT2E','RBBP7'],
        ['KPNA1','TMEM38B'],
        ['KPNA3','CLPB'],
        ['KPNA3','EIF3E'],
        ['KPNA6','GLO1'],
        ['KPNA6','KCTD20'],
        ['KPNB1','TBL1XR1'],
        ['KPNB1','TMSB4X'],
        ['KPNB1','TNPO1'],
        ['KRBA2','KCTD20'],
        ['KRR1','ACAP2'],
        ['KRR1','FLRT2'],
        ['KRR1','ZNF362'],
        ['KRT14','HSP90B1'],
        ['KRT14','KRT5'],
        ['KRT14','MALAT1'],
        ['KRT14','RPL37'],
        ['KRT19','RXRB'],
        ['KRT19','TSHZ3'],
        ['KRT5','HSPB1'],
        ['KRT5','KRT10'],
        ['KRT5','KRT14'],
        ['KRT5','KRT16'],
        ['KRT5','KRT16P1'],
        ['KRT5','KRT16P3'],
        ['KRT5','KRT16P4'],
        ['KRT5','KRT16P5'],
        ['KRT5','RPL8'],
        ['KRT5','S100A6'],
        ['KRT6A','KRT14'],
        ['KRT6A','KRT80'],
        ['KRT7','LDHB'],
        ['KRT81','GSTP1'],
        ['KRT81','RPL18'],
        ['KRT81','SFN'],
        ['KRT81','SLC7A5'],
        ['KXD1','CANX'],
        ['KXD1','PAFAH1B3'],
        ['L3MBTL2','MCM4'],
        ['LA16C-431H6.6','RASSF7'],
        ['LACTB','UTRN'],
        ['LAMA4','EIF2S3'],
        ['LAMB1','RFK'],
        ['LAMB1','YWHAQ'],
        ['LAMB2','CST3'],
        ['LAMC1','SUDS3'],
        ['LAMP1','FAAP100'],
        ['LAMP1','SUV39H1'],
        ['LAMTOR1','NUMA1'],
        ['LANCL2','FAM178A'],
        ['LAPTM4A','MYL12B'],
        ['LAPTM4B','CNBP'],
        ['LAPTM4B','LPGAT1'],
        ['LAPTM5','UBL7'],
        ['LARP1','SRSF10'],
        ['LARP4B','HNRNPH1'],
        ['LARP4','HNRNPF'],
        ['LARS','EMG1'],
        ['LARS','EXO1'],
        ['LARS','FAM53C'],
        ['LAS1L','LCP1'],
        ['LBHD1','MAN1A2'],
        ['LBR','EIF3E'],
        ['LBR','NARS'],
        ['LBR','PCM1'],
        ['LBR','TRAPPC5'],
        ['LCMT1-AS1','AC074286.1'],
        ['LCP1','IGH@'],
        ['LCP2','SEC24C'],
        ['LDHA','SCO2'],
        ['LDHA','WAC'],
        ['LDHB','CCT3'],
        ['LDHB','CISD2'],
        ['LDHB','CSDE1'],
        ['LDHB','EEF2'],
        ['LDHB','GEMIN5'],
        ['LDHB','MUT'],
        ['LDHB','PSMC4'],
        ['LDHB','RBBP7'],
        ['LDHB','SKP2'],
        ['LDHB','TRAPPC5'],
        ['LDLR','SIPA1L1'],
        ['LENG8','ARAP1'],
        ['LENG8','RAPGEF1'],
        ['LEPR','FAM60A'],
        ['LEPR','FDXR'],
        ['LETM1','EEF2'],
        ['LGALS17A','AC093063.2'],
        ['LGALS3','MSH6'],
        ['LHX4','EFNB1'],
        ['LHX4','TPT1'],
        ['LIG3','HMGB3'],
        ['LIN7A','CAST'],
        ['LIN9','MAX'],
        ['LINC00294','PSMD8'],
        ['LINC00707','MTATP6P1'],
        ['LINC00963','POLR2B'],
        ['LINC00996','GIMAP8'],
        ['LINC00998','TSC22D3'],
        ['LINC00998','TUBG1'],
        ['LINC01184','SAMM50'],
        ['LINC01420','USP45'],
        ['LIX1L','MTHFD2'],
        ['LIX1L','TBC1D10B'],
        ['LLFOS-48D6.2','CNOT4'],
        ['LLFOS-48D6.2','EIF4G1'],
        ['LLFOS-48D6.2','MGEA5'],
        ['LLGL1','B3GAT2'],
        ['LMAN1','ABRACL'],
        ['LMAN1','EPHA3'],
        ['LMAN1','LGALS3BP'],
        ['LMAN1','NAPG'],
        ['LMAN1','NCOA3'],
        ['LMAN1','NUCKS1'],
        ['LMAN2','HAGH'],
        ['LMAN2L','TNPO1'],
        ['LMAN2','RAB12'],
        ['LMAN2','UBFD1'],
        ['LMBR1L','SHROOM3'],
        ['LMNA','GLO1'],
        ['LMNA','KRT5'],
        ['LMNA','SNRNP70'],
        ['LMNB1','NDUFB11'],
        ['LMNB2','CLPX'],
        ['LMNB2','CSNK1E'],
        ['LMNB2','FECH'],
        ['LMNB2','PHLDA3'],
        ['LMTK2','HINT1'],
        ['LNPEP','DDX17'],
        ['LOH12CR1','TRMT2A'],
        ['LONP1','ZNF711'],
        ['LONP2','METAP2'],
        ['LONP2','NF2'],
        ['LONRF2','COCH'],
        ['LPIN1','AFF1'],
        ['LRIF1','PSMD12'],
        ['LRP11','BMP2K'],
        ['LRP11','TMCO1'],
        ['LRPPRC','CCNF'],
        ['LRPPRC','CENPE'],
        ['LRPPRC','LMNB1'],
        ['LRPPRC','PIN1'],
        ['LRPPRC','TJP1'],
        ['LRR1','B3GAT2'],
        ['LRR1','USP31'],
        ['LRRC23','TGFB1'],
        ['LRRC28','NAPG'],
        ['LRRC40','NELFE'],
        ['LRRC47','CDC42'],
        ['LRRC49','ARHGEF2'],
        ['LRRC57','GCN1L1'],
        ['LRRC69','PRIM2'],
        ['LRRC75A','ANKZF1'],
        ['LRRC75A-AS1','ACSL3'],
        ['LRRC75A-AS1','HAT1'],
        ['LRRC75A-AS1','KCNQ1OT1'],
        ['LRRC75A-AS1','LDHB'],
        ['LRRC75A-AS1','RPL18'],
        ['LRRC75A-AS1','TAF6L'],
        ['LRRC8A','ACLY'],
        ['LRRC8B','RPL8'],
        ['LRRD1','FAM208B'],
        ['LRRD1','ZNF512B'],
        ['LRRFIP2','CASP2'],
        ['LSG1','ANXA6'],
        ['LSP1','RNF187'],
        ['LSS','ANP32E'],
        ['LTA4H','SDE2'],
        ['LTBP1','BTAF1'],
        ['LTV1','PCED1A'],
        ['LUC7L3','VDAC1'],
        ['LY9','EIF3E'],
        ['LYAR','EEF1G'],
        ['LYN','C15ORF40'],
        ['LYPLA1','GNA12'],
        ['LYPLA1','MYBBP1A'],
        ['LYPLAL1','POLDIP3'],
        ['LZIC','EIF4G2'],
        ['LZTS2','STAT6'],
        ['LZTS3','CHD1L'],
        ['MACF1','RPS19'],
        ['MACF1','SNRNP200'],
        ['MACF1','TMA16'],
        ['MAD2L1','MRPS22'],
        ['MAD2L1','TPI1'],
        ['MAD2L2','STARD7'],
        ['MAGED1','IRF4'],
        ['MAGOH','SAP18'],
        ['MAGT1','DDB1'],
        ['MAGT1','HOOK1'],
        ['MAGT1','ITGAV'],
        ['MAGT1','RFXANK'],
        ['MALAT1','BCL9'],
        ['MALAT1','EVI5'],
        ['MALAT1','KRT17'],
        ['MALAT1','LLFOS-48D6.2'],
        ['MALAT1','OAZ1'],
        ['MALAT1','RBM12'],
        ['MALAT1','SPINK5'],
        ['MALSU1','FASN'],
        ['MALT1','RCC1'],
        ['MAML1','SLC1A3'],
        ['MAML2','EEF2'],
        ['MAN1A2','KIAA1524'],
        ['MANF','ATP5G2'],
        ['MAP1B','NANP'],
        ['MAP2K1','TMEM65'],
        ['MAP2K2','ARVCF'],
        ['MAP2K3','COPB2'],
        ['MAP2K4','ANXA7'],
        ['MAP3K2','CTSD'],
        ['MAP3K2','ENO1'],
        ['MAP3K2','IWS1'],
        ['MAP3K4','AURKAIP1'],
        ['MAP4K4','SRSF3'],
        ['MAP4','SERTAD2'],
        ['MAP4','SUMO1'],
        ['MAPK1','CACYBP'],
        ['MAPK3','SKIV2L2'],
        ['MAPK9','CENPF'],
        ['MAPRE1','GUK1'],
        ['MAPRE1','HMBOX1'],
        ['MAPRE1','RPL35'],
        ['MAPRE2','RPS19'],
        ['MARC1','RPLP1'],
        ['MARCH6','C15ORF39'],
        ['MARCH6','SLC16A4'],
        ['MARCH6','USP36'],
        ['MARCH7','MSN'],
        ['MARCKS','KRCC1'],
        ['MARCKS','YBX1'],
        ['MARCKS','ZNF7'],
        ['MARK3','TNPO3'],
        ['MARK4','STARD7'],
        ['MASTL','PPM1G'],
        ['MATR3','AK6'],
        ['MATR3','AKR1A1'],
        ['MATR3','GLO1'],
        ['MATR3','GRN'],
        ['MATR3','HEATR5A'],
        ['MATR3','MRPL9'],
        ['MATR3','QARS'],
        ['MAVS','SDC2'],
        ['MAX','RNF144A'],
        ['MAZ','EEF2'],
        ['MAZ','PAFAH1B1'],
        ['MAZ','TEAD1'],
        ['MAZ','TMED7'],
        ['MAZ','TMED7-TICAM2'],
        ['MBD1','CCDC11'],
        ['MBD2','RPL26'],
        ['MBD3','CTNNB1'],
        ['MBD6','MCCC2'],
        ['MBNL1','CARHSP1'],
        ['MBNL1','KIAA0100'],
        ['MBOAT2','RPRD1A'],
        ['MBP','TXLNG'],
        ['MCAT','CCDC90B'],
        ['MCC','SCAF8'],
        ['MCEE','PTPN6'],
        ['MCEE','USP9X'],
        ['MCFD2','TPI1'],
        ['MCFD2','ZBTB1'],
        ['MCL1','MIF'],
        ['MCM10','CDC123'],
        ['MCM10','KIAA1549L'],
        ['MCM10','PRPSAP2'],
        ['MCM2','ATF2'],
        ['MCM2','DYNLL2'],
        ['MCM2','KIAA0391'],
        ['MCM2','MATR3'],
        ['MCM2','SCAMP1'],
        ['MCM2','SRSF4'],
        ['MCM3','DDX17'],
        ['MCM3','RNF10'],
        ['MCM3','SRSF3'],
        ['MCM4','AGPS'],
        ['MCM4','ATP5B'],
        ['MCM4','EIF3G'],
        ['MCM4','ICK'],
        ['MCM5','FDPS'],
        ['MCM6','ADI1'],
        ['MCM6','CDK14'],
        ['MCM6','ICMT'],
        ['MCM6','SERF2'],
        ['MCM6','USP9X'],
        ['MCM7','C19ORF52'],
        ['MCM7','CCT3'],
        ['MCM7','HACD3'],
        ['MCM7','TMTC3'],
        ['MCM8','CTSD'],
        ['MCM8','SLC25A11'],
        ['MCMBP','CSRP2BP'],
        ['MCMBP','PKM'],
        ['MCMBP','RNF25'],
        ['MCMBP','SLC12A6'],
        ['MCOLN2','ZNF644'],
        ['MCRS1','HEATR3'],
        ['MCUR1','GATC'],
        ['MDH1','GTF3C2'],
        ['MDH1','HSPA1B'],
        ['MDH1','NET1'],
        ['MDH2','B2M'],
        ['MDH2','IGH@'],
        ['MDM2','LPGAT1'],
        ['MDM2','NCS1'],
        ['MDM4','KIF5B'],
        ['ME2','MCTP2'],
        ['MEA1','UTP18'],
        ['MEA1','XIST'],
        ['MED10','ATP2B1'],
        ['MED13','TRMT112'],
        ['MED24','ACTN4'],
        ['MED24','KLHL5'],
        ['MED29','TOP2A'],
        ['MED31','KIAA0753'],
        ['MED4','RFX5'],
        ['MED6','LYRM1'],
        ['MED8','TERF2IP'],
        ['MELK','OGFR'],
        ['MERTK','PPM1B'],
        ['MESDC2','ZNF76'],
        ['METAP2','MLK4'],
        ['METAP2','MRPS33'],
        ['METAP2','RHNO1'],
        ['METTL17','HEXA'],
        ['METTL7A','CCNG1'],
        ['METTL7A','ENTPD4'],
        ['MEX3C','SEPT6'],
        ['MFN2','MATR3'],
        ['MFSD11','AC010642.1'],
        ['MFSD11','ERVK3-1'],
        ['MGAT1','KIAA0922'],
        ['MGAT1','MED24'],
        ['MGAT1','NFYC'],
        ['MGAT2','PHF20'],
        ['MGAT2','SEPT11'],
        ['MGAT4B','ACADVL'],
        ['MGEA5','APEX1'],
        ['MGEA5','MYH9'],
        ['MGEA5','PRRC2A'],
        ['MIAT','MALAT1'],
        ['MIB1','CLTC'],
        ['MIB1','EIF2S3'],
        ['MIB1','GIGYF2'],
        ['MICAL3','JUNB'],
        ['MICALL1','TMEM106C'],
        ['MICU2','MEA1'],
        ['MID1IP1','RNF219'],
        ['MID1IP1','SLC25A3'],
        ['MIEF1','CORO1A'],
        ['MIEF1','EIF3L'],
        ['MIEF1','VIM'],
        ['MIER1','XPO5'],
        ['MIF','GLRX2'],
        ['MIF','SERBP1'],
        ['MIF','SLC25A3'],
        ['MIF','STARD7'],
        ['MIF','TYRO3'],
        ['MINA','ANKRD13A'],
        ['MINK1','TMEM194A'],
        ['MINOS1','NUF2'],
        ['MINPP1','EEF2'],
        ['MIOS','HDLBP'],
        ['MIS18A','BID'],
        ['MKI67','PCCB'],
        ['MKLN1','SAV1'],
        ['MKRN1','EEF2'],
        ['MKRN2','MYO6'],
        ['MLEC','CLUH'],
        ['MLF2','PMF1'],
        ['MLF2','PMF1-BGLAP'],
        ['MLF2','RPPH1'],
        ['MLF2','RSL1D1'],
        ['MLH1','POP4'],
        ['MLK4','METAP2'],
        ['MLLT11','CNOT3'],
        ['MLLT6','M6PR'],
        ['MLLT6','SMARCA2'],
        ['MLST8','GNA12'],
        ['MMAB','CCT3'],
        ['MMS22L','NUP88'],
        ['MOAP1','C6ORF106'],
        ['MOAP1','CALM2'],
        ['MOB1A','ATE1'],
        ['MOB1A','PGAM5'],
        ['MOB1A','PPP1CB'],
        ['MOB1A','SPAG5'],
        ['MOB3A','MFSD11'],
        ['MOCS2','LPGAT1'],
        ['MOCS2','MINA'],
        ['MORC3','CD164'],
        ['MORF4L1','PUS7'],
        ['MORF4L2','CKB'],
        ['MORF4L2','CPSF6'],
        ['MORF4L2','DYRK2'],
        ['MORF4L2','TSPAN6'],
        ['MOSPD2','FANCB'],
        ['MOV10','PAM'],
        ['MPC2','HNRNPU'],
        ['MPDZ','CXORF57'],
        ['MPG','TFAM'],
        ['MPHOSPH10','MAP1LC3B'],
        ['MPHOSPH6','ELL2'],
        ['MPHOSPH6','SYAP1'],
        ['MPHOSPH8','CIAO1'],
        ['MPHOSPH8','RAB7A'],
        ['MPHOSPH9','STXBP3'],
        ['MPP1','PRICKLE3'],
        ['MPP3','HNRNPM'],
        ['MPP5','NUP54'],
        ['MPV17','CDC123'],
        ['MPZL1','ENO1'],
        ['MPZL1','MRTO4'],
        ['MRFAP1','ASCC2'],
        ['MRFAP1','ENO1'],
        ['MRGBP','DDX5'],
        ['MRI1','HDLBP'],
        ['MRPL14','CENPH'],
        ['MRPL14','CLPX'],
        ['MRPL16','MRPL28'],
        ['MRPL19','TMEM241'],
        ['MRPL20','HSP90AB1'],
        ['MRPL23','ALMS1'],
        ['MRPL23','TPCN1'],
        ['MRPL37','CLPTM1'],
        ['MRPL40','KMT2A'],
        ['MRPL42','AQR'],
        ['MRPL50','SART3'],
        ['MRPL55','ANXA11'],
        ['MRPL55','RPS6'],
        ['MRPL9','ASCC3'],
        ['MRPL9','P4HA1'],
        ['MRPS11','SPOPL'],
        ['MRPS12','AP000304.12'],
        ['MRPS12','ATP5O'],
        ['MRPS12','ODC1'],
        ['MRPS15','ZNF205'],
        ['MRPS16','HEXB'],
        ['MRPS22','YWHAQ'],
        ['MRPS23','KIFAP3'],
        ['MRPS23','VCP'],
        ['MRPS27','LYSMD3'],
        ['MRPS28','GSR'],
        ['MRPS2','GHDC'],
        ['MRPS2','WBP1L'],
        ['MRPS30','CLPTM1L'],
        ['MRPS31','RPL31'],
        ['MRPS31','SLC25A10'],
        ['MRPS31','WDR41'],
        ['MRPS33','PKM'],
        ['MRPS34','BABAM1'],
        ['MRPS34','MFN2'],
        ['MRPS34','POLE2'],
        ['MRPS35','CSRP2BP'],
        ['MRPS7','CHPT1'],
        ['MRRF','FBXO34'],
        ['MRTO4','FGD5-AS1'],
        ['MRTO4','P4HA1'],
        ['MRTO4','POLA1'],
        ['MSANTD3','FAM175B'],
        ['MSANTD3-TMEFF1','VDAC1'],
        ['MSANTD4','WBSCR16'],
        ['MSC','SEL1L'],
        ['MSH3','UNK'],
        ['MSH5','ORAI2'],
        ['MSH6','IPO5'],
        ['MSH6','MAP2K2'],
        ['MSH6','PRPF31'],
        ['MSMO1','DFFA'],
        ['MSN','ZZEF1'],
        ['MSRB1','FSCN1'],
        ['MSRB3','VIM'],
        ['MT2A','KRT5'],
        ['MT2A','KRT81'],
        ['MTATP6P1','MTRNR2L8'],
        ['MTCP1','NEDD1'],
        ['MTDH','BTRC'],
        ['MTDH','RHOA'],
        ['MTDH','ZNF672'],
        ['MTF2','ACADM'],
        ['MTFR1','BLM'],
        ['MTG2','GPATCH3'],
        ['MTHFD1','ALKBH5'],
        ['MTHFD1','CFL1'],
        ['MTHFD1','RBM15'],
        ['MTHFD2','DEPDC1'],
        ['MTHFD2','PPP2CB'],
        ['MTHFD2','RAD23A'],
        ['MTHFD2','SCML2'],
        ['MTIF2','HINT1'],
        ['MTIF3','SMG9'],
        ['MTMR12','ANKRD46'],
        ['MTMR3','C6ORF48'],
        ['MTOR','INPP4A'],
        ['MTPAP','ZNF850'],
        ['MTPN','LRPPRC'],
        ['MTR','GTF2E1'],
        ['MTR','TMEM201'],
        ['MTX3','CDT1'],
        ['MVB12A','ANKRD10'],
        ['MXD4','ENO1'],
        ['MXI1','PKM'],
        ['MXRA7','FAM101B'],
        ['MXRA7','HIP1'],
        ['MXRA7','IGF2BP1'],
        ['MYBBP1A','EIF2A'],
        ['MYBBP1A','TMEM123'],
        ['MYBL2','ANP32A'],
        ['MYBL2','LUC7L3'],
        ['MYBL2','PFN2'],
        ['MYBL2','SFPQ'],
        ['MYCBP','RYK'],
        ['MYDGF','HNRNPU'],
        ['MYH10','ANKLE2'],
        ['MYH10','RAN'],
        ['MYH10','SLC44A1'],
        ['MYH9','CNPY2'],
        ['MYH9','DICER1'],
        ['MYH9','DOT1L'],
        ['MYH9','NDUFA7'],
        ['MYH9','PIH1D1'],
        ['MYH9','PSMD1'],
        ['MYH9','RBBP7'],
        ['MYH9','RBM10'],
        ['MYH9','ZNF362'],
        ['MYL12B','ACTB'],
        ['MYL6','ABCF2'],
        ['MYL6','SLC38A1'],
        ['MYL6','STRAP'],
        ['MYL6','VRK1'],
        ['MYLIP','RPL4'],
        ['MYO1C','KDM1A'],
        ['MYO1D','VAPA'],
        ['MZB1','CAPN1'],
        ['MZT1','C1ORF115'],
        ['MZT1','DNM1L'],
        ['N4BP2L2','ADCK3'],
        ['NAA10','GPR160'],
        ['NAA10','RPL24'],
        ['NAA15','DDX5'],
        ['NAA15','MAPRE1'],
        ['NAA15','PHGDH'],
        ['NAA30','CCND2'],
        ['NAA30','EPS15L1'],
        ['NAB1','NGRN'],
        ['NABP2','CENPV'],
        ['NABP2','SKIV2L2'],
        ['NACA','ASUN'],
        ['NACA','HSP90AB1'],
        ['NACA','RPS17'],
        ['NACA','TRIM28'],
        ['NACC1','RPN2'],
        ['NADK','MAPKAPK2'],
        ['NAE1','BUB3'],
        ['NAGA','RAB21'],
        ['NAP1L1','IPO11'],
        ['NAP1L1','LARP1'],
        ['NAPG','LANCL2'],
        ['NAPRT','NME1'],
        ['NARF','RCC2'],
        ['NASP','C10ORF35'],
        ['NAT14','RBX1'],
        ['NAT8L','COL4A2'],
        ['NAT8L','GPI'],
        ['NAT9','PSMC4'],
        ['NAV2','IER3IP1'],
        ['NBAS','MYO9B'],
        ['NBN','ATP6V0E2'],
        ['NBN','MFNG'],
        ['NBN','RHOA'],
        ['NCAPG','ODC1'],
        ['NCAPG','TAX1BP1'],
        ['NCAPH2','UACA'],
        ['NCAPH','MAGEH1'],
        ['NCBP2-AS2','CNOT1'],
        ['NCDN','EEF2'],
        ['NCK1','CCDC8'],
        ['NCK2','CDS2'],
        ['NCKAP1','EIF4A1'],
        ['NCKAP1','SENP3-EIF4A1'],
        ['NCKIPSD','C17ORF62'],
        ['NCL','ACTB'],
        ['NCL','CHCHD1'],
        ['NCL','GFOD1'],
        ['NCL','NPM1'],
        ['NCL','PHPT1'],
        ['NCL','PTMA'],
        ['NCL','STEAP3'],
        ['NCL','UBE2E1'],
        ['NCOA1','HNRNPH3'],
        ['NCOA4','FLII'],
        ['NCOR2','SCARB1'],
        ['NCR3LG1','KIAA1324L'],
        ['NCR3LG1','PM20D2'],
        ['NCS1','ALKBH1'],
        ['NCSTN','NHLH1'],
        ['NDC1','DCAF6'],
        ['NDC1','DYNLRB1'],
        ['NDC1','NDUFAF3'],
        ['NDC1','PGK1'],
        ['NDC1','TMED2'],
        ['NDE1','B2M'],
        ['NDEL1','U2SURP'],
        ['NDFIP2','RPS14'],
        ['NDRG1','TRAPPC5'],
        ['NDUFA10','CAPN15'],
        ['NDUFA11','AKR1B1'],
        ['NDUFA12','GCHFR'],
        ['NDUFA13','BOC'],
        ['NDUFA13','BUB3'],
        ['NDUFA13','EEF2'],
        ['NDUFA6','EEF2'],
        ['NDUFA6','LMF2'],
        ['NDUFA6','TIPIN'],
        ['NDUFA9','DSCR3'],
        ['NDUFA9','LSM14A'],
        ['NDUFA9','RPS19'],
        ['NDUFB10','TMTC2'],
        ['NDUFB10','ZP3'],
        ['NDUFB11','DNAJC10'],
        ['NDUFB11','FASN'],
        ['NDUFB11','RPN1'],
        ['NDUFB6','LRRC58'],
        ['NDUFB8','SEC31B'],
        ['NDUFC2','H2AFZ'],
        ['NDUFS1','SNRPB'],
        ['NDUFS1','USP15'],
        ['NDUFS5','HIPK3'],
        ['NDUFS8','MYBBP1A'],
        ['NDUFS8','TCEB3'],
        ['NDUFV1','MIA3'],
        ['NDUFV1','TEX9'],
        ['NDUFV3','DNAJC8'],
        ['NDUFV3','TOR1A'],
        ['NEAT1','PAAF1'],
        ['NECAP2','KPNB1'],
        ['NEDD8','GANAB'],
        ['NEDD9','NCOR1'],
        ['NEFL','MAP1LC3B'],
        ['NEFL','TSTA3'],
        ['NEIL1','CALML4'],
        ['NEIL3','FEM1B'],
        ['NEK7','MCM6'],
        ['NEK8','SERINC1'],
        ['NELFB','CHD8'],
        ['NELFB','TOR4A'],
        ['NEMF','DGCR14'],
        ['NEMP1','EIF2S3'],
        ['NENF','BRD2'],
        ['NENF','RBM20'],
        ['NF1','SCAMP1'],
        ['NF2','HSCB'],
        ['NFU1','RTCB'],
        ['NFYC','TM9SF2'],
        ['NGRN','AP000350.10'],
        ['NGRN','MIF'],
        ['NGRN','NUCKS1'],
        ['NHP2','DCTN2'],
        ['NID2','KRR1'],
        ['NIF3L1','FNDC3B'],
        ['NIFK','FAR2'],
        ['NIPA2','SFSWAP'],
        ['NIPAL3','VCP'],
        ['NIPBL','XPO7'],
        ['NKAPP1','LBR'],
        ['NKIRAS2','C1ORF43'],
        ['NKIRAS2','EIF4G2'],
        ['NKRF','MANF'],
        ['NKTR','SPATC1L'],
        ['NLE1','ATP6V0A2'],
        ['NLRC5','CSAD'],
        ['NMB','SH3GL2'],
        ['NME1','MOGS'],
        ['NME1','SPEN'],
        ['NME2','GSTO1'],
        ['NME2','IFT81'],
        ['NME4','DHX38'],
        ['NMT2','RPS14'],
        ['NOA1','EIF5'],
        ['NOC2L','AP2M1'],
        ['NOC3L','CLEC16A'],
        ['NOC3L','FAM134B'],
        ['NOL11','DLD'],
        ['NOL7','VIM'],
        ['NOL9','IDH1'],
        ['NOLC1','CDC34'],
        ['NOLC1','DIS3L'],
        ['NOLC1','NUP50'],
        ['NOLC1','RPS4X'],
        ['NOM1','PGK1'],
        ['NONO','ABCB1'],
        ['NONO','AHCY'],
        ['NONO','IQCB1'],
        ['NONO','NSUN2'],
        ['NONO','REV1'],
        ['NONO','SLC25A16'],
        ['NONO','SLX4IP'],
        ['NONO','TARS2'],
        ['NOP10','DBN1'],
        ['NOP2','HNRNPA2B1'],
        ['NOP56','GADD45GIP1'],
        ['NOP56','MARVELD2'],
        ['NOP56','MEPCE'],
        ['NOP56','PPP2R5D'],
        ['NOP58','NUP160'],
        ['NOP58','PARD6G'],
        ['NOP58','VBP1'],
        ['NOP9','NF2'],
        ['NOTCH2','AHCY'],
        ['NOTCH2','MYL6'],
        ['NPLOC4','AGGF1'],
        ['NPLOC4','GMPS'],
        ['NPM1','ATAD2'],
        ['NPM1','GLO1'],
        ['NPM1','MELK'],
        ['NPM1','NCL'],
        ['NPM1P27','ATAD2'],
        ['NPM1P32','NCL'],
        ['NPM1P39','NCL'],
        ['NPM1P6','NCL'],
        ['NPM3','SETD6'],
        ['NPM3','USP13'],
        ['NPNT','PPA2'],
        ['NQO2','ATRX'],
        ['NQO2','UBR2'],
        ['NR2F6','CABIN1'],
        ['NR2F6','FARSA'],
        ['NR2F6','NLK'],
        ['NR3C1','DONSON'],
        ['NRBP1','GTF2A2'],
        ['NRBP1','USP5'],
        ['NRBP2','PUF60'],
        ['NREP','APMAP'],
        ['NRIP3','ATP5SL'],
        ['NRP2','ASPM'],
        ['NRROS','HDLBP'],
        ['NSA2','GNB2L1'],
        ['NSD1','JUN'],
        ['NSD1','OAZ1'],
        ['NSD1','PLXNB2'],
        ['NSF','ADD1'],
        ['NSFL1C','GDI2'],
        ['NSFL1C','ZACN'],
        ['NSMAF','CNIH4'],
        ['NSMAF','WHSC1'],
        ['NSMCE4A','HNRNPU'],
        ['NSMCE4A','HSBP1'],
        ['NSUN2','CUL4B'],
        ['NT5C3B','POGK'],
        ['NT5E','RAB11FIP2'],
        ['NUCB1','MTHFD1L'],
        ['NUCB1','TAF9'],
        ['NUCKS1','AKAP12'],
        ['NUCKS1','MYO1B'],
        ['NUCKS1','PAXIP1-AS1'],
        ['NUCKS1','RLIM'],
        ['NUCKS1','SMUG1'],
        ['NUCKS1','ZBTB24'],
        ['NUDCD1','RPS20'],
        ['NUDCD2','CBX5'],
        ['NUDCD2','DCAF13'],
        ['NUDCD2','LASP1'],
        ['NUDCD2','NPM3'],
        ['NUDC','TBC1D24'],
        ['NUDT3','EEF2'],
        ['NUDT4','TMED3'],
        ['NUP133','CEP83'],
        ['NUP133','CSE1L'],
        ['NUP133','SH3BP4'],
        ['NUP153','ATP1A1'],
        ['NUP153','SLC39A9'],
        ['NUP155','CANT1'],
        ['NUP155','COQ3'],
        ['NUP160','THUMPD2'],
        ['NUP205','GTF3C4'],
        ['NUP205','KIAA1524'],
        ['NUP210','PHF19'],
        ['NUP214','HEXB'],
        ['NUP37','SEC62'],
        ['NUP43','CCT2'],
        ['NUP43','IFNGR2'],
        ['NUP43','RPS8'],
        ['NUP54','CD59'],
        ['NUP62','D2HGDH'],
        ['NUP98','PIN1'],
        ['NUPL2','NCOA4'],
        ['NUSAP1','EIF4B'],
        ['NUSAP1','RBBP9'],
        ['NXT1','KITLG'],
        ['OAT','CDC42BPA'],
        ['OAT','DST'],
        ['OAZ1','ANXA6'],
        ['OAZ1','CNOT4'],
        ['OAZ1','EIF4G1'],
        ['OAZ1','MGEA5'],
        ['OAZ1','PM20D2'],
        ['OAZ1','RPL35A'],
        ['OAZ1','SERBP1'],
        ['OAZ1','SLC27A2'],
        ['OCIAD1','HEXB'],
        ['OCIAD2','UQCRH'],
        ['ODC1','LAMP1'],
        ['ODC1','NCK2'],
        ['ODC1','TRMT6'],
        ['ODC1','ZBTB38'],
        ['OFD1','HADHA'],
        ['OGFRL1','ESD'],
        ['OGFRL1','PRDX6'],
        ['OGFRL1','UBE2I'],
        ['OGT','MAP4K1'],
        ['OGT','RERE'],
        ['OIP5-AS1','KTN1'],
        ['OIP5-AS1','RNF8'],
        ['OIP5-AS1','SEPT9'],
        ['OIP5-AS1','TAX1BP1'],
        ['OIP5-AS1','TBC1D7'],
        ['OIP5-AS1','XRN1'],
        ['OLA1','PEG10'],
        ['OMA1','ZNF229'],
        ['OPA1','PTDSS2'],
        ['OPA1','RPL19'],
        ['ORMDL1','PPP1R16A'],
        ['ORMDL1','PSMA7'],
        ['ORMDL2','SLC2A6'],
        ['OSBPL8','PHB'],
        ['OSBPL9','AASDHPPT'],
        ['OSBPL9','SLC25A3'],
        ['OST4','C2ORF69'],
        ['OST4','EIF4G3'],
        ['OST4','PEBP1'],
        ['OTUD3','CALM3'],
        ['OTUD5','GSPT1'],
        ['OTUD6B-AS1','RPL38'],
        ['OTUD6B','LRRC69'],
        ['P2RX5-TAX1BP3','BASP1'],
        ['P2RY10','TNRC6B'],
        ['P2RY8','CDK10'],
        ['P4HB','AGPAT6'],
        ['P4HB','CTDNEP1'],
        ['P4HB','ENO1'],
        ['P4HB','FAM193B'],
        ['P4HB','NUP98'],
        ['PA2G4','DDX3X'],
        ['PABPC1','KRT5'],
        ['PABPC3','KRT5'],
        ['PABPC4','GOLIM4'],
        ['PABPC4','TBC1D8'],
        ['PACSIN3','PSMD12'],
        ['PAF1','SMAD4'],
        ['PAFAH1B1','CALR'],
        ['PAFAH1B1','HNRNPF'],
        ['PAFAH1B2','MPC1'],
        ['PAGR1','CUL3'],
        ['PAGR1','HSPA5'],
        ['PAICS','EXOSC6'],
        ['PAICS','HSPA1B'],
        ['PAK4','CD81'],
        ['PANK2','NUP88'],
        ['PAPD5','SNX1'],
        ['PAPOLA','BCL2L2-PABPN1'],
        ['PAPOLA','EIF2S2'],
        ['PAPOLA','FANCD2'],
        ['PAPOLA','MPC2'],
        ['PAPOLA','MTR'],
        ['PAPOLA','PABPN1'],
        ['PAQR5','POLR3H'],
        ['PARK7','DNTTIP2'],
        ['PARK7','OSER1'],
        ['PARK7','PSMA7'],
        ['PARK7','RBM39'],
        ['PARP1','POMT1'],
        ['PARP1','SON'],
        ['PARP1','ZNF490'],
        ['PARPBP','SIAH1'],
        ['PATZ1','EIF3H'],
        ['PATZ1','LIMK2'],
        ['PAWR','SAR1B'],
        ['PAX9','DLG5'],
        ['PAXIP1-AS1','GALNT11'],
        ['PBXIP1','HNRNPF'],
        ['PBXIP1','RRP9'],
        ['PCBP2','GOSR1'],
        ['PCF11','LMO4'],
        ['PCGF3','RBBP7'],
        ['PCGF5','DDX6'],
        ['PCMT1','FOXJ3'],
        ['PCMTD2','MTMR12'],
        ['PCNA','FEM1A'],
        ['PCNA','FGD4'],
        ['PCNA','PGRMC1'],
        ['PCSK6','NCKAP1'],
        ['PCSK7','DCTN3'],
        ['PCYOX1','ZC3H13'],
        ['PCYT1A','TKT'],
        ['PDAP1','DLC1'],
        ['PDCD2','PA2G4'],
        ['PDCD5','EIF3B'],
        ['PDCD5','THOC5'],
        ['PDCD5','YWHAB'],
        ['PDCL','CFAP20'],
        ['PDE12','FAM46A'],
        ['PDE4D','RRP9'],
        ['PDHA1','RBM39'],
        ['PDHX','MDH1'],
        ['PDIA4','NCL'],
        ['PDIA4','P4HB'],
        ['PDIA4','TOB2'],
        ['PDIA6','GLA'],
        ['PDK1','KRT8'],
        ['PDK4','SSR3'],
        ['PDLIM1','PCYOX1'],
        ['PDLIM5','PSPH'],
        ['PDLIM7','BIRC5'],
        ['PDPR','GNB2L1'],
        ['PDRG1','OSBPL1A'],
        ['PDS5B','THAP4'],
        ['PDZD8','TRIB2'],
        ['PDZD8','TSPAN15'],
        ['PEBP1','HPCAL1'],
        ['PEBP1','KDM3B'],
        ['PEBP1','TFAM'],
        ['PEG10','POLR3C'],
        ['PELI1','TPM4'],
        ['PELO','TMEM256'],
        ['PEX11A','PGRMC1'],
        ['PEX19','SESTD1'],
        ['PEX1','DDX10'],
        ['PEX5','FBXO16'],
        ['PEX5','TUBB4B'],
        ['PEX5','ZNF395'],
        ['PFAS','APLP2'],
        ['PFAS','ATP5O'],
        ['PFDN5','JUN'],
        ['PFDN5','RPL38'],
        ['PFDN5','TPT1'],
        ['PFDN6','ATP2A2'],
        ['PFKL','STT3B'],
        ['PFKM','MKI67'],
        ['PFKM','PLS3'],
        ['PFKP','MAP9'],
        ['PFKP','PPP3CA'],
        ['PFKP','TIMM13'],
        ['PFN1','ARHGEF1'],
        ['PFN1','S1PR2'],
        ['PGAM5','GTF2F1'],
        ['PGAM5','LRRC75A-AS1'],
        ['PGAP1','NRP1'],
        ['PGD','IPO5'],
        ['PGK1','NARS'],
        ['PGK1','TIMM44'],
        ['PGM1','BEND3'],
        ['PGM1','CHMP2A'],
        ['PGM1','MARCH6'],
        ['PGM2','C1ORF43'],
        ['PGM3','ACOT7'],
        ['PGRMC1','B4GALT2'],
        ['PGRMC1','ENAH'],
        ['PGRMC1','SLC46A1'],
        ['PGRMC1','TALDO1'],
        ['PHB2','RPL37A'],
        ['PHF14','PAK2'],
        ['PHF19','SYVN1'],
        ['PHF20','RRM2'],
        ['PHF21A','CCDC186'],
        ['PHF5A','RNF34'],
        ['PHF6','CSDE1'],
        ['PHF7','ZFYVE9'],
        ['PHGDH','NDUFA8'],
        ['PHGDH','PPP4C'],
        ['PHGDH','TMED10'],
        ['PHKB','PGM1'],
        ['PHKB','RETSAT'],
        ['PHTF1','TEAD4'],
        ['PIBF1','KLF5'],
        ['PICALM','ARPP19'],
        ['PIEZO1','CXORF57'],
        ['PIEZO2','AGPAT5'],
        ['PIGT','RBM3'],
        ['PIGV','RPS4X'],
        ['PIK3R2','MKI67'],
        ['PIK3R2','POLR3C'],
        ['PIM1','TFAM'],
        ['PIM2','DUSP22'],
        ['PIM2','KALRN'],
        ['PIP5K1A','EIF5'],
        ['PITPNA','YWHAE'],
        ['PITX1','H3F3B'],
        ['PJA2','MTMR9'],
        ['PKM','ADAMTS2'],
        ['PKM','AHI1'],
        ['PKM','CDK16'],
        ['PKM','CUL5'],
        ['PKM','GNB2L1'],
        ['PKM','HNRNPF'],
        ['PKM','PSMC5'],
        ['PKM','RNF149'],
        ['PKM','RPL35'],
        ['PKM','SEMA4G'],
        ['PKM','UBR4'],
        ['PKN1','EHD4'],
        ['PKN2','OAZ1'],
        ['PKNOX1','CWC15'],
        ['PKP4','GALNT13'],
        ['PK','SLC1A5'],
        ['PLCD3','KDM5C'],
        ['PLCE1','TTF2'],
        ['PLCG1','ATAD5'],
        ['PLCG2','OGFR'],
        ['PLCG2','XAB2'],
        ['PLD3','LSG1'],
        ['PLD3','PARN'],
        ['PLEKHA1','FAM84A'],
        ['PLEKHA2','PDIA4'],
        ['PLEKHA3','ARHGAP19'],
        ['PLEKHA3','ARHGAP19-SLIT1'],
        ['PLEKHG1','ZNF711'],
        ['PLEKHJ1','HINT3'],
        ['PLEKHJ1','YLPM1'],
        ['PLOD1','CCNG1'],
        ['PLOD1','EIF4B'],
        ['PLOD1','PHF3'],
        ['PLOD2','SRSF3'],
        ['PLS3','ALDOA'],
        ['PLS3','CYP51A1'],
        ['PLS3','HMGCS1'],
        ['PLS3','LRRD1'],
        ['PLS3','MPP5'],
        ['PLS3','PDIK1L'],
        ['PLS3','SEPT10'],
        ['PLS3','TBC1D10B'],
        ['PLS3','ZNF678'],
        ['PLXNA1','CDT1'],
        ['PLXNA1','SYNPO'],
        ['PLXNB2','RPS19'],
        ['PLXNC1','TTLL3'],
        ['PM20D2','CIAO1'],
        ['PM20D2','GALNT1'],
        ['PM20D2','YWHAH'],
        ['PMAIP1','TFIP11'],
        ['PMEL','CKB'],
        ['PMM2','AP2S1'],
        ['PMPCA','RPN2'],
        ['PMPCA','SLC6A15'],
        ['PMPCB','CUL3'],
        ['PMPCB','PDIA4'],
        ['PMPCB','SLC30A7'],
        ['PMVK','HSP90B1'],
        ['PNO1','UGP2'],
        ['PNOC','P4HB'],
        ['PODXL2','FAM65A'],
        ['POFUT1','CENPU'],
        ['POGK','IGSF3'],
        ['POGK','VAPB'],
        ['POGLUT1','NUMA1'],
        ['POLA1','MRTO4'],
        ['POLA2','ATAD1'],
        ['POLA2','ATP5A1'],
        ['POLA2','FBXL6'],
        ['POLD1','BPTF'],
        ['POLD2','LMAN2'],
        ['POLD2','VDAC1'],
        ['POLDIP2','HSP90AB1'],
        ['POLDIP2','TDRD3'],
        ['POLDIP3','RNF10'],
        ['POLDIP3','RPL15'],
        ['POLE2','SPECC1'],
        ['POLE3','PRKDC'],
        ['POLG','MBD1'],
        ['POLR1A','MAGEF1'],
        ['POLR1A','VPS25'],
        ['POLR1B','TSNAX'],
        ['POLR1D','C14ORF2'],
        ['POLR1E','UQCR11'],
        ['POLR2A','IQGAP3'],
        ['POLR2B','HSPA9'],
        ['POLR2B','MAPK1'],
        ['POLR2B','MYC'],
        ['POLR2C','MYL6B'],
        ['POLR2K','DNAL4'],
        ['POLR3H','PSPC1'],
        ['POMP','ERBB2IP'],
        ['POMP','HSP90AB1'],
        ['PON2','ATAD1'],
        ['POP1','DPYSL3'],
        ['PPAN','CLTA'],
        ['PPAPDC2','VDAC3'],
        ['PPAT','MBD3'],
        ['PPAT','TMED2'],
        ['PPCS','MRPS23'],
        ['PPHLN1','HNRNPU'],
        ['PPIB','GLCE'],
        ['PPIB','PRMT5'],
        ['PPIB','ZNF143'],
        ['PPIF','CCND2'],
        ['PPIG','UQCRQ'],
        ['PPIL1','MYH10'],
        ['PPIL2','HNRNPL'],
        ['PPIL2','USP42'],
        ['PPIP5K1','PRPF6'],
        ['PPIP5K2','USP12'],
        ['PPM1A','RPL11'],
        ['PPM1F','AXIN2'],
        ['PPM1G','GNB2L1'],
        ['PPM1G','SCD'],
        ['PPM1G','SF3B1'],
        ['PPM1K','EHMT1'],
        ['PPOX','CD74'],
        ['PPOX','IKZF3'],
        ['PPP1CA','COPB1'],
        ['PPP1CA','LMAN1'],
        ['PPP1CA','SSRP1'],
        ['PPP1CB','AP1B1'],
        ['PPP1CB','KATNB1'],
        ['PPP1CB','N4BP1'],
        ['PPP1CB','THRAP3'],
        ['PPP1CC','HM13'],
        ['PPP1CC','LDHB'],
        ['PPP1CC','UBE2D3'],
        ['PPP1R12A','CPSF2'],
        ['PPP1R12A','PAWR'],
        ['PPP1R14C','IARS2'],
        ['PPP1R15B','PAIP2'],
        ['PPP1R15B','PAWR'],
        ['PPP1R3B','GNS'],
        ['PPP1R7','MYH9'],
        ['PPP1R7','OAZ1'],
        ['PPP1R8','RPL7A'],
        ['PPP1R9B','POLD2'],
        ['PPP1R9B','UBA5'],
        ['PPP2CA','GLO1'],
        ['PPP2CA','ODF2L'],
        ['PPP2R1A','ARL1'],
        ['PPP2R1A','CPSF4'],
        ['PPP2R1A','MRPL41'],
        ['PPP2R1A','NAT9'],
        ['PPP2R1A','RPN2'],
        ['PPP2R3B','GLUD1'],
        ['PPP2R3C','CTAGE5'],
        ['PPP2R5A','POLA1'],
        ['PPP3R1','SOD2'],
        ['PPP4C','HCCS'],
        ['PPP4R1','HNRNPL'],
        ['PPP5C','VPS25'],
        ['PPP6C','IDH3B'],
        ['PPT1','TMEM2'],
        ['PQBP1','OS9'],
        ['PRC1','RPL35'],
        ['PRDM2','RPS4X'],
        ['PRDX1','C1QBP'],
        ['PRDX1','CANX'],
        ['PRDX1','CBX2'],
        ['PRDX1','EEF2'],
        ['PRDX1','NDUFB8'],
        ['PRDX1','TBC1D3P1-DHX40P1'],
        ['PRDX1','TMEM98'],
        ['PRDX2','MRPL37'],
        ['PRDX2','NCAPD2'],
        ['PRDX3','TSPAN13'],
        ['PRDX4','AMFR'],
        ['PRDX4','NARS'],
        ['PRDX4','RPS23'],
        ['PRDX4','RPS6'],
        ['PRDX5','ENO1'],
        ['PRDX5','LYN'],
        ['PRDX6','HKR1'],
        ['PRDX6','PSENEN'],
        ['PRDX6','TXNL4A'],
        ['PREP','GINM1'],
        ['PRH1-PRR4','POLK'],
        ['PRKAA1','MTCH1'],
        ['PRKAA1','TTC33'],
        ['PRKAB1','RPS2'],
        ['PRKACB','TARS'],
        ['PRKAG1','KPNA6'],
        ['PRKAR1A','LRP12'],
        ['PRKAR1A','RN7SL1'],
        ['PRKAR1A','SEC61A2'],
        ['PRKAR2A','HSPA1B'],
        ['PRKAR2A','PFAS'],
        ['PRKCB','ALKBH5'],
        ['PRKCB','MOB3A'],
        ['PRKCB','RNF213'],
        ['PRKCH','FLJ22447'],
        ['PRKCQ','YWHAZ'],
        ['PRKD1','EEF2'],
        ['PRKD2','MYO1D'],
        ['PRKDC','MCM4'],
        ['PRKDC','PCBP1'],
        ['PRKDC','TUBGCP6'],
        ['PRMT1','CALR'],
        ['PRMT2','RDH10'],
        ['PRMT5','GANAB'],
        ['PRMT6','CCDC93'],
        ['PRMT6','PKN2'],
        ['PROCR','NDUFS5'],
        ['PRPF38A','SCML2'],
        ['PRPF3','MCUR1'],
        ['PRPF40A','ZNF263'],
        ['PRPF4B','SMARCC2'],
        ['PRPF6','EIF3K'],
        ['PRPF6','HN1L'],
        ['PRPF8','CKB'],
        ['PRPF8','EEA1'],
        ['PRPF8','TM9SF1'],
        ['PRPS2','FNIP2'],
        ['PRPS2','LYPLA1'],
        ['PRPS2','TIMM13'],
        ['PRR4','POLK'],
        ['PRRC2B','DYNLT1'],
        ['PRRC2C','CKB'],
        ['PRRG1','MAPRE1'],
        ['PRRT2','SIPA1'],
        ['PSAP','PGAM5'],
        ['PSENEN','YBX3'],
        ['PSIP1','GDE1'],
        ['PSIP1','MRPS34'],
        ['PSMA1','CKLF'],
        ['PSMA1','CKLF-CMTM1'],
        ['PSMA1','DLG3'],
        ['PSMA1','SPSB1'],
        ['PSMA3','SART3'],
        ['PSMA4','CKAP4'],
        ['PSMA4','DNAJC1'],
        ['PSMA4','FEM1B'],
        ['PSMA4','GDI1'],
        ['PSMA4','GNB2L1'],
        ['PSMA4','HSPA8'],
        ['PSMA4','SLC33A1'],
        ['PSMA4','TMED2'],
        ['PSMA5','CD320'],
        ['PSMA5','CLTC'],
        ['PSMA7','NCAPH2'],
        ['PSMB1','ENO1'],
        ['PSMB1','IMP3'],
        ['PSMB1','LEPR'],
        ['PSMB1','PTBP3'],
        ['PSMB1','USP39'],
        ['PSMB3','SIN3A'],
        ['PSMB3','SLC25A6'],
        ['PSMB4','HOXA10'],
        ['PSMB5','LONP2'],
        ['PSMB5','RPS8'],
        ['PSMB7','DNM2'],
        ['PSMC2','EIF2S3'],
        ['PSMC4','AHI1'],
        ['PSMC5','IGH@'],
        ['PSMD11','UBL5'],
        ['PSMD11','WDR55'],
        ['PSMD12','NT5C'],
        ['PSMD14','BCAT2'],
        ['PSMD14','CHCHD3'],
        ['PSMD1','BRD2'],
        ['PSMD3','HUWE1'],
        ['PSMD6','KCTD15'],
        ['PSMD7','GPS1'],
        ['PSMD7','GUCD1'],
        ['PSMD7','SYAP1'],
        ['PSMD8','DYNLL1'],
        ['PSME1','EEF2'],
        ['PSME1','ENO1'],
        ['PSME3','NAB1'],
        ['PSMG1','MAP4K4'],
        ['PSMG1','RICTOR'],
        ['PSMG1','TOMM7'],
        ['PSTPIP2','ARF1'],
        ['PTBP1','IGH@'],
        ['PTBP1','SPOP'],
        ['PTCD3','FTSJ1'],
        ['PTER','DGCR2'],
        ['PTGER2','DAZAP1'],
        ['PTGES3','ACSL3'],
        ['PTGES3','C16ORF13'],
        ['PTGES3','SLC30A6'],
        ['PTGFRN','PCSK6'],
        ['PTK2B','CD74'],
        ['PTK2B','P4HTM'],
        ['PTK2B','PMFBP1'],
        ['PTK2B','XRCC6'],
        ['PTK2B','ZSCAN30'],
        ['PTMA','NCL'],
        ['PTMA','NPM1'],
        ['PTMA','NPM1P27'],
        ['PTMAP3','PTMA'],
        ['PTMS','KRT81'],
        ['PTP4A2','ACAT2'],
        ['PTP4A2','CENPV'],
        ['PTP4A2','PIM3'],
        ['PTPN11','KCMF1'],
        ['PTPN14','FANCI'],
        ['PTPN18','UBE2D4'],
        ['PTPN1','SEC61A1'],
        ['PTPN3','DDX17'],
        ['PTPN4','POU2F1'],
        ['PTPN4','UTP18'],
        ['PTPN9','MTOR'],
        ['PTPRA','SNRPB'],
        ['PTPRC','RPS6'],
        ['PTPRG','HNRNPDL'],
        ['PTPRS','MCM2'],
        ['PTRHD1','C1ORF27'],
        ['PTRHD1','MPLKIP'],
        ['PTTG1IP','SYAP1'],
        ['PUF60','POLG'],
        ['PUS1','HMX1'],
        ['PUSL1','AAMDC'],
        ['PVRL3','EEF2'],
        ['PWP2','C21orf33'],
        ['PWWP2A','COX6C'],
        ['PXN-AS1','SMARCA5'],
        ['PYCR2','RB1CC1'],
        ['PYGB','LARS'],
        ['PYGL','CDT1'],
        ['PYGL','DDX3X'],
        ['PYURF','LMAN1'],
        ['QARS','NCOR2'],
        ['QKI','IFI30'],
        ['QKI','PIK3R2'],
        ['QPRT','AP2B1'],
        ['QSER1','UTP18'],
        ['QSOX2','PSMD14'],
        ['QTRT1','MOV10'],
        ['QTRTD1','DLG3'],
        ['RAB10','NUCKS1'],
        ['RAB11FIP1','GPI'],
        ['RAB11FIP2','PM20D2'],
        ['RAB11FIP4','FAM134C'],
        ['RAB11FIP4','SOD1'],
        ['RAB12','POLR2F'],
        ['RAB14','POLD2'],
        ['RAB18','NR2F6'],
        ['RAB18','SART1'],
        ['RAB18','SLC39A10'],
        ['RAB1A','EIF4G3'],
        ['RAB23','CALCOCO1'],
        ['RAB23','STARD7'],
        ['RAB27B','NUP88'],
        ['RAB29','TUFM'],
        ['RAB2A','AARS'],
        ['RAB2A','IKBKAP'],
        ['RAB31','SCAF11'],
        ['RAB32','GNB2L1'],
        ['RAB36','COL4A3BP'],
        ['RAB3D','RC3H1'],
        ['RAB3GAP1','DCP2'],
        ['RAB4A','GEMIN7'],
        ['RAB4A','VPS4A'],
        ['RAB5C','NDE1'],
        ['RAB7A','AARSD1'],
        ['RAB7A','PTGES3L-AARSD1'],
        ['RAB7A','RNPEP'],
        ['RAB8B','AKT2'],
        ['RAB8B','UNG'],
        ['RAB9A','HSP90AA1'],
        ['RAB9A','TCEANC'],
        ['RAB9A','TMEM254'],
        ['RABAC1','DPYSL5'],
        ['RABAC1','PDS5B'],
        ['RABEP1','ATL3'],
        ['RABEP1','PAGR1'],
        ['RABEP1','SMG1'],
        ['RABEP1','SPPL3'],
        ['RABGAP1L','EPHA7'],
        ['RABGGTB','MSL3'],
        ['RABL6','IGH@'],
        ['RAC3','EEF2'],
        ['RAC3','PPP1R14C'],
        ['RACGAP1','MARS'],
        ['RACGAP1','PES1'],
        ['RAD18','OXTR'],
        ['RAD23A','TNFRSF10B'],
        ['RAD23B','TIMM13'],
        ['RAD50','PPM1F'],
        ['RAD50','TRPS1'],
        ['RAD51C','SLC22A5'],
        ['RAD51D','C14ORF166'],
        ['RAF1','LBR'],
        ['RAI14','CANX'],
        ['RALBP1','DHX29'],
        ['RALBP1','LARP4B'],
        ['RALGDS','ANXA5'],
        ['RALGPS2','ARL6IP1'],
        ['RALY','QKI'],
        ['RANBP1','LDHB'],
        ['RANBP1','UACA'],
        ['RANBP3','CD300A'],
        ['RANBP3','PFKP'],
        ['RANBP3','PPP2R1B'],
        ['RANBP6','ALG13'],
        ['RAN','DNM2'],
        ['RANGAP1','PELP1'],
        ['RANGRF','SEC23A'],
        ['RAP1GDS1','BTAF1'],
        ['RAP2A','CEP170B'],
        ['RAPGEF1','GPI'],
        ['RAPGEF1','TTC31'],
        ['RAPGEF4','PDK1'],
        ['RAPGEF5','TRIM36'],
        ['RARS','PARK7'],
        ['RASA2','FOXN3'],
        ['RASGRF2','TMEM218'],
        ['RASGRP1','TWF1'],
        ['RB1CC1','DFFA'],
        ['RB1CC1','PFKM'],
        ['RBBP4','TBC1D9B'],
        ['RBBP6','GAPVD1'],
        ['RBBP7','CCT5'],
        ['RBBP7','HSPA1A'],
        ['RBBP7','NCL'],
        ['RBBP7','SEPT8'],
        ['RBBP7','SMARCB1'],
        ['RBM10','CANX'],
        ['RBM12','MDH1'],
        ['RBM15B','FAM178A'],
        ['RBM26','INPP5E'],
        ['RBM38','SP1'],
        ['RBM39','KDM5C'],
        ['RBM3','CASC4'],
        ['RBM3','EIF2S3'],
        ['RBM3','FAM46A'],
        ['RBM3','LGALS3BP'],
        ['RBM3','NSUN2'],
        ['RBM3','OAZ1'],
        ['RBM3','TCF4'],
        ['RBM6','BTBD6'],
        ['RBM6','OCIAD1'],
        ['RBMX','ATXN7L3B'],
        ['RBMX','RNASEH2A'],
        ['RBSN','ACTR10'],
        ['RCC1','CAMK2D'],
        ['RCC1','TMEM230'],
        ['RCC2','CWF19L1'],
        ['RCC2','PARP1'],
        ['RCC2','RPLP1'],
        ['RCC2','TECPR1'],
        ['RCCD1','JUN'],
        ['RCCD1','UBXN6'],
        ['RCHY1','VIM'],
        ['RCN2','STEAP3'],
        ['RCSD1','TUT1'],
        ['RDH10','RPS4X'],
        ['RDH11','VTI1B'],
        ['RDH14','RPA2'],
        ['RECQL4','ROBO3'],
        ['RECQL','EMP2'],
        ['REEP2','GOLIM4'],
        ['REEP2','MRPL28'],
        ['RELA','CD37'],
        ['RELA','SEPT9'],
        ['RELL2','BRD2'],
        ['REST','LINC01547'],
        ['RETSAT','SCAND2P'],
        ['REV3L','RFC3'],
        ['RFC2','DAZAP1'],
        ['RFC2','KTN1'],
        ['RFC2','VAMP4'],
        ['RFESD','TMEM97'],
        ['RFFL','CBX3'],
        ['RFK','PAFAH1B2'],
        ['RFK','SMG5'],
        ['RFWD3','MINOS1'],
        ['RFWD3','TPI1'],
        ['RGL2','STAG2'],
        ['RGS10','ATP5L'],
        ['RGS10','FBXO11'],
        ['RHBDD3','TMEM41B'],
        ['RHOA','DDX17'],
        ['RHOA','TRIT1'],
        ['RHOA','TUBA1C'],
        ['RHOBTB1','ZCCHC7'],
        ['RHOBTB3','HNRNPH1'],
        ['RHOB','ZNF592'],
        ['RHOC','KCNAB2'],
        ['RHOC','MREG'],
        ['RHOC','ZNF527'],
        ['RHOG','TPT1'],
        ['RHOQ','HACD3'],
        ['RHPN1','BRIX1'],
        ['RICTOR','RANGAP1'],
        ['RICTOR','TIMM13'],
        ['RIF1','PAICS'],
        ['RIMS4','DYNLT3'],
        ['RINT1','LINC00843'],
        ['RINT1','TIMM23B'],
        ['RIOK2','PRDX1'],
        ['RIPK2','CAMSAP1'],
        ['RIPK4','NUDT2'],
        ['RLF','BNIP2'],
        ['RLF','DHX30'],
        ['RLIM','FUT1'],
        ['RMDN1','GPBP1L1'],
        ['RMND5A','ARL5B'],
        ['RMND5A','HNRNPR'],
        ['RMND5A','REEP5'],
        ['RMND5A','STX6'],
        ['RMRP','YTHDF1'],
        ['RNASEH1','GNL2'],
        ['RNASEH2A','TOR1A'],
        ['RNASEH2B','ADAT2'],
        ['RNASEH2B','EEF1G'],
        ['RNASET2','EIF3K'],
        ['RNASET2','MAP9'],
        ['RNF114','PRPS2'],
        ['RNF115','UBE2E1'],
        ['RNF11','RSL1D1'],
        ['RNF138','EPRS'],
        ['RNF144A','LRRC8B'],
        ['RNF145','HARS'],
        ['RNF149','CDS2'],
        ['RNF187','MLK4'],
        ['RNF187','RPA2'],
        ['RNF187','UQCC3'],
        ['RNF220','NASP'],
        ['RNF2','UBR3'],
        ['RNF38','BBX'],
        ['RNF6','ABCF1'],
        ['RNF6','MYO1D'],
        ['RNLS','ATP6V0A2'],
        ['RNPC3','EGLN3'],
        ['RNPEPL1','RPL18'],
        ['ROCK1','RIOK3'],
        ['ROCK2','ACTG1'],
        ['ROCK2','FBXW11'],
        ['ROR1','GINM1'],
        ['RPA2','SLC1A5'],
        ['RPA2','TBC1D14'],
        ['RPA2','ZNF275'],
        ['RPE','BMPR2'],
        ['RPF1','PKM'],
        ['RPF1','SNRNP70'],
        ['RPF1','SPAG9'],
        ['RPGR','RPS6KA3'],
        ['RPL10A','KRT81'],
        ['RPL10A','RBPJ'],
        ['RPL11','ACBD6'],
        ['RPL11','DEAF1'],
        ['RPL11','MAP3K2'],
        ['RPL11','MTPN'],
        ['RPL11','NANS'],
        ['RPL11','RHOA'],
        ['RPL11','TCEB3'],
        ['RPL13A','ELL2'],
        ['RPL13A','SCAF11'],
        ['RPL13A','SF3B3'],
        ['RPL13A','SYNGAP1'],
        ['RPL13','ATP6AP1'],
        ['RPL13A','ZBTB9'],
        ['RPL13','MAZ'],
        ['RPL13','RPL31'],
        ['RPL14','TPT1'],
        ['RPL15','NUCKS1'],
        ['RPL15','SENP1'],
        ['RPL15','TMUB2'],
        ['RPL17','RPS21'],
        ['RPL17','STUB1'],
        ['RPL17','TYSND1'],
        ['RPL18A','ITCH'],
        ['RPL18','GIPC1'],
        ['RPL18','HDGF'],
        ['RPL18','KLHL23'],
        ['RPL18','MARS'],
        ['RPL18','MED17'],
        ['RPL18','NR3C1'],
        ['RPL18','POLR1B'],
        ['RPL18','RPL38'],
        ['RPL18','SMC1A'],
        ['RPL18','ZNF91'],
        ['RPL19','ACBD6'],
        ['RPL19','GSS'],
        ['RPL19','HERC2'],
        ['RPL19','PCMT1'],
        ['RPL19','TAB2'],
        ['RPL22','HDAC2'],
        ['RPL22','LDLRAP1'],
        ['RPL23AP82','STARD7'],
        ['RPL23A','RPS27A'],
        ['RPL23','HSP90AA1'],
        ['RPL23','LDHB'],
        ['RPL23','TBCC'],
        ['RPL26','DHCR24'],
        ['RPL26','RNF40'],
        ['RPL27A','RAB10'],
        ['RPL27','CKB'],
        ['RPL27','GRN'],
        ['RPL27','LLGL1'],
        ['RPL27','NEDD4'],
        ['RPL27','NEK11'],
        ['RPL27','RAN'],
        ['RPL29','IQGAP1'],
        ['RPL30','AURKAIP1'],
        ['RPL30','GPR137B'],
        ['RPL30','SMC1A'],
        ['RPL30','TP53'],
        ['RPL31','ARL6IP4'],
        ['RPL31','HSPA4'],
        ['RPL31','SERF2'],
        ['RPL31','USP22'],
        ['RPL35A','CANX'],
        ['RPL35A','RPN2'],
        ['RPL35A','RPS14'],
        ['RPL35A','SCD5'],
        ['RPL35','IDE'],
        ['RPL35','MBTPS2'],
        ['RPL36A','DEGS1'],
        ['RPL36A-HNRNPH2','DEGS1'],
        ['RPL36','COQ4'],
        ['RPL36','IMP3'],
        ['RPL37A','CCDC142'],
        ['RPL37A','EIF3L'],
        ['RPL37A','MARS'],
        ['RPL37A','MRPL53'],
        ['RPL37','ANAPC5'],
        ['RPL37A','NEAT1'],
        ['RPL37A','OAZ1'],
        ['RPL37A','PDIA6'],
        ['RPL37','ENO1'],
        ['RPL37','SEPT8'],
        ['RPL37','UCK1'],
        ['RPL38','AIF1L'],
        ['RPL38','BEND7'],
        ['RPL38','CEP120'],
        ['RPL38','COPE'],
        ['RPL38','FAU'],
        ['RPL38','PDIA6'],
        ['RPL38','RPS17'],
        ['RPL38','SLC30A7'],
        ['RPL38','TTYH2'],
        ['RPL38','WDR54'],
        ['RPL39','ERH'],
        ['RPL3','ATP6V0A1'],
        ['RPL3','KLC3'],
        ['RPL3','NT5C3B'],
        ['RPL3','RPL22'],
        ['RPL41','RABEP2'],
        ['RPL4','AZIN1'],
        ['RPL4','CLNS1A'],
        ['RPL4','KRT14'],
        ['RPL4','NIPSNAP1'],
        ['RPL4','POMGNT1'],
        ['RPL4','STAMBPL1'],
        ['RPL4','ZBTB3'],
        ['RPL5','CSDE1'],
        ['RPL5','TFRC'],
        ['RPL7A','KLF13'],
        ['RPL7A','SREBF2'],
        ['RPL7L1','RPS9'],
        ['RPL7L1','SMC4'],
        ['RPL7L1','TRA@'],
        ['RPL7L1','ZW10'],
        ['RPL8','ENO1'],
        ['RPL8','KDM4A'],
        ['RPL8','KRT5'],
        ['RPL8','MAPK8IP3'],
        ['RPL8','MTATP6P1'],
        ['RPL8','NET1'],
        ['RPL8','POC1B'],
        ['RPL8','TMEM147'],
        ['RPLP1','LIN28B'],
        ['RPLP2','ARL6IP1'],
        ['RPLP2','SH2D4A'],
        ['RPN1','MIF-AS1'],
        ['RPN2','RABL6'],
        ['RPN2','TRIQK'],
        ['RPPH1','POLDIP2'],
        ['RPS12','FKBP4'],
        ['RPS14','FASTK'],
        ['RPS15A','AP2A1'],
        ['RPS15A','DBI'],
        ['RPS15A','PSMC5'],
        ['RPS15A','TRMT5'],
        ['RPS16','CCT3'],
        ['RPS16','CEP85'],
        ['RPS16','EIF5'],
        ['RPS16','NAPG'],
        ['RPS16','PSPH'],
        ['RPS16','SCD'],
        ['RPS17','DDX54'],
        ['RPS17','KRT81'],
        ['RPS17L','KRT81'],
        ['RPS17','MCM10'],
        ['RPS17','RREB1'],
        ['RPS18','G6PD'],
        ['RPS19','AFG3L2'],
        ['RPS19','ATP6V1G2-DDX39B'],
        ['RPS19','CBX5'],
        ['RPS19','CCT6A'],
        ['RPS19','DDRGK1'],
        ['RPS19','DDX39B'],
        ['RPS19','EIF2A'],
        ['RPS19','G3BP2'],
        ['RPS19','HADHB'],
        ['RPS19','IFI6'],
        ['RPS19','NELFCD'],
        ['RPS19','PFDN5'],
        ['RPS19','RASSF8-AS1'],
        ['RPS19','UBE2G1'],
        ['RPS19','YWHAQ'],
        ['RPS20','CCT2'],
        ['RPS20','LRRC75A-AS1'],
        ['RPS20','PPP2R1A'],
        ['RPS20','RNF213'],
        ['RPS21','GSPT1'],
        ['RPS21','IARS'],
        ['RPS21','MT2A'],
        ['RPS23','C1ORF43'],
        ['RPS23','CCNK'],
        ['RPS23','CDC123'],
        ['RPS23','EXOSC5'],
        ['RPS23','MCL1'],
        ['RPS24','EEF2'],
        ['RPS24','RPH3AL'],
        ['RPS24','TLR10'],
        ['RPS25','QPRT'],
        ['RPS25','RASSF8'],
        ['RPS25','ZNF768'],
        ['RPS27A','TM9SF4'],
        ['RPS27','GADD45GIP1'],
        ['RPS27','MED21'],
        ['RPS29','TMUB2'],
        ['RPS4X','GART'],
        ['RPS4X','GCN1L1'],
        ['RPS4X','GNB2L1'],
        ['RPS4X','HEATR5B'],
        ['RPS4X','LMNB2'],
        ['RPS4X','MTMR2'],
        ['RPS4X','WHSC1L1'],
        ['RPS5','HIF1A'],
        ['RPS5','PSMA7'],
        ['RPS5','SKIV2L'],
        ['RPS5','SSFA2'],
        ['RPS6','BFAR'],
        ['RPS6','CEBPA'],
        ['RPS6KA1','RPL31'],
        ['RPS6KA1','WHSC1'],
        ['RPS6KA3','JMJD8'],
        ['RPS6KB2','UROD'],
        ['RPS6KC1','CBR4'],
        ['RPS6KC1','VPS4B'],
        ['RPS6','LHFPL2'],
        ['RPS6','PACS2'],
        ['RPS6','PGK1'],
        ['RPS6','PQBP1'],
        ['RPS6','TMEM181'],
        ['RPS6','ZNF302'],
        ['RPS8','ENO1'],
        ['RPS8','WDR82'],
        ['RPS9','RPL7L1'],
        ['RPUSD1','DDX24'],
        ['RPUSD1','WFS1'],
        ['RPUSD4','GLA'],
        ['RRM1','CENPBD1P1'],
        ['RRM1','MGC2752'],
        ['RRM1','PCBP1'],
        ['RRM1','PGRMC1'],
        ['RRM2B','KPNA6'],
        ['RRM2','C2ORF74'],
        ['RRN3','ZNF562'],
        ['RRP15','G3BP1'],
        ['RRP15','VPS16'],
        ['RRP1B','HSF2BP'],
        ['RSBN1','ATP5A1'],
        ['RSL1D1','CCT2'],
        ['RSL1D1','PSMA5'],
        ['RSL1D1','TMBIM1'],
        ['RSL24D1','C20ORF24'],
        ['RSL24D1','CCDC144NL-AS1'],
        ['RSU1','KIF20B'],
        ['RSU1','WDR1'],
        ['RTCB','EEF2'],
        ['RTCB','LDAH'],
        ['RTCB','RAB32'],
        ['RTFDC1','CNN3'],
        ['RTN4','PTMA'],
        ['RUNDC1','DDOST'],
        ['RUNX1','CD19'],
        ['RUNX3','ISCU'],
        ['RUVBL1','CD99L2'],
        ['RUVBL1','CIAO1'],
        ['RUVBL1','NGRN'],
        ['RUVBL1','TMEM14A'],
        ['RUVBL2','NBR1'],
        ['RXRA','SCAMP4'],
        ['S100A2','KRT7'],
        ['S100A2','KRT81'],
        ['SAC3D1','DAD1'],
        ['SACM1L','KLHDC3'],
        ['SACM1L','PEBP1'],
        ['SAE1','GCN1L1'],
        ['SAE1','RDX'],
        ['SAFB2','ROCK2'],
        ['SAMD12','GGCT'],
        ['SAMD8','CYB5B'],
        ['SAMHD1','ZRANB2'],
        ['SAMM50','LRRC75A-AS1'],
        ['SAMM50','SRSF3'],
        ['SAP18','IDH3B'],
        ['SAR1A','KIF20B'],
        ['SAR1A','MAZ'],
        ['SAR1A','PAFAH1B3'],
        ['SARS2','MMS19'],
        ['SARS','ACTR6'],
        ['SARS','AURKA'],
        ['SARS','IMP4'],
        ['SART3','STAG2'],
        ['SAT2','DYNC1H1'],
        ['SAV1','B4GAT1'],
        ['SAV1','MKLN1'],
        ['SBDS','IER5'],
        ['SBDS','SGOL2'],
        ['SBDS','SMIM8'],
        ['SBF1','ING1'],
        ['SCAF4','UBR4'],
        ['SCAF8','MAGOH'],
        ['SCAF8','ROCK2'],
        ['SCAND2P','ANKLE2'],
        ['SCAP','ENO1'],
        ['SCAPER','HMGN1'],
        ['SCAP','ETFB'],
        ['SCAP','MCRS1'],
        ['SCARA3','HDAC2'],
        ['SCARB1','CCDC18'],
        ['SCARB1','CDIPT'],
        ['SCARB2','CSE1L'],
        ['SCD','CASC3'],
        ['SCD','EFNB2'],
        ['SCD','H3F3B'],
        ['SCD','SMURF1'],
        ['SCD','ZNF107'],
        ['SCFD1','MYO9B'],
        ['SCML1','MRPS34'],
        ['SCML1','RPL36'],
        ['SCNN1A','TNFRSF1A'],
        ['SCO1','GINS4'],
        ['SCO2','EEF2'],
        ['SCP2','IDH3B'],
        ['SCRN1','C6ORF62'],
        ['SCRN1','CADPS2'],
        ['SDCCAG3','CARD11'],
        ['SDCCAG3','CHAF1A'],
        ['SDCCAG8','DLAT'],
        ['SDE2','CHPF'],
        ['SDE2','GNE'],
        ['SDE2','HOXC9'],
        ['SDE2','LIN7C'],
        ['SDF4','SPTBN1'],
        ['SDHA','ZNF148'],
        ['SDHA','ZNF71'],
        ['SDHB','DICER1'],
        ['SDHB','PRKAR1A'],
        ['SDHB','SND1'],
        ['SDK1','PRPF4'],
        ['SEC11A','MORF4L2'],
        ['SEC16A','LMNB2'],
        ['SEC23A','HTATSF1'],
        ['SEC23B','RBL2'],
        ['SEC24A','CLTC'],
        ['SEC24B','OSTC'],
        ['SEC24C','CELF1'],
        ['SEC24C','RNF157'],
        ['SEC31A','MTX3'],
        ['SEC61A1','CD63'],
        ['SEC61A1','RAB7A'],
        ['SEC63','MRPL39'],
        ['SECISBP2L','NDUFS2'],
        ['SEL1L3','EHD1'],
        ['SEL1L3','ZNF146'],
        ['SEL1L','NSF'],
        ['SELO','CHD4'],
        ['SELT','NUP37'],
        ['SELT','PSMA5'],
        ['SEMA3C','SDE2'],
        ['SEMA3E','CFL1'],
        ['SEMA6A','XRN2'],
        ['SENP2','TCEA1'],
        ['SENP3-EIF4A1','RPAIN'],
        ['SENP3-EIF4A1','RPS3'],
        ['SENP3','RPS3'],
        ['SEP15','EDIL3'],
        ['SEPT11','CNPY2'],
        ['SEPT11','LASP1'],
        ['SEPT2','IDH1'],
        ['SEPT2','XRCC5'],
        ['SEPT6','LDHB'],
        ['SEPT7','PSMD3'],
        ['SEPT9','ATG101'],
        ['SEPT9','IGH@'],
        ['SERBP1','GNB2L1'],
        ['SERBP1','MIF4GD'],
        ['SERBP1','PPP4R1'],
        ['SERBP1','SH3YL1'],
        ['SERF2','KBTBD4'],
        ['SERGEF','TES'],
        ['SERINC1','COQ9'],
        ['SERINC1','ZNF430'],
        ['SERINC3','KIF1B'],
        ['SERINC3','ZNF433'],
        ['SERP1','MPHOSPH8'],
        ['SERP1','ZC3HC1'],
        ['SERPINH1','ERGIC3'],
        ['SERTAD1','PRX'],
        ['SESN1','TP53RK'],
        ['SESN3','FBXO11'],
        ['SESN3','SHMT2'],
        ['SET','ALDH16A1'],
        ['SET','COPE'],
        ['SETD5','MRPS11'],
        ['SET','FAXC'],
        ['SET','HSP90AB1'],
        ['SET','PAIP2'],
        ['SET','PRDX4'],
        ['SF1','MOB3A'],
        ['SF1','NUMBL'],
        ['SF3A2','EPG5'],
        ['SF3A3','TAZ'],
        ['SF3B1','COBL'],
        ['SF3B1','PLCL2'],
        ['SF3B2','APAF1'],
        ['SF3B2','PSMC4'],
        ['SF3B5','UBE2J1'],
        ['SF3B6','SMC1A'],
        ['SFPQ','NDUFS3'],
        ['SFPQ','POLE'],
        ['SFPQ','PTPMT1'],
        ['SFXN1','TPR'],
        ['SFXN3','FANCG'],
        ['SFXN4','LETM1'],
        ['SGMS2','PRKD3'],
        ['SGSH','RAC2'],
        ['SGTA','BICC1'],
        ['SH2B1','GNA13'],
        ['SH2D5','STYX'],
        ['SH3GL1','PEX26'],
        ['SHISA3','GSR'],
        ['SHMT2','CYB5D2'],
        ['SHQ1','VDAC1'],
        ['SIGLEC10','KIAA1147'],
        ['SIK2','RPS4X'],
        ['SIKE1','C5ORF24'],
        ['SIKE1','KLHL15'],
        ['SIPA1L2','JUN'],
        ['SIRPB2','NSFL1C'],
        ['SIX1','TDRD3'],
        ['SKA1','GPT2'],
        ['SKA2','CIC'],
        ['SKA2','FTO'],
        ['SKIV2L','TICRR'],
        ['SKP2','ATL3'],
        ['SKP2','SUB1'],
        ['SLAMF7','MEF2C'],
        ['SLC11A2','PAQR4'],
        ['SLC12A7','HMOX2'],
        ['SLC16A1','ATF2'],
        ['SLC16A1','HSP90AA1'],
        ['SLC16A1','NFKBIB'],
        ['SLC16A1','PPP1R9B'],
        ['SLC16A1','USP1'],
        ['SLC16A1','VIM'],
        ['SLC16A4','MKI67'],
        ['SLC16A7','MRPS16'],
        ['SLC19A1','CREB1'],
        ['SLC19A1','TARS'],
        ['SLC19A1','ZCCHC3'],
        ['SLC1A5','ATP5I'],
        ['SLC1A5','CAPZA2'],
        ['SLC1A5','MAN1A2'],
        ['SLC1A5','PSMB4'],
        ['SLC20A1','GTF3C5'],
        ['SLC20A1','TXLNG'],
        ['SLC25A10','CDH7'],
        ['SLC25A10','UBL4A'],
        ['SLC25A13','DOCK6'],
        ['SLC25A13','DTL'],
        ['SLC25A15','RPL38'],
        ['SLC25A16','CCNB1'],
        ['SLC25A19','MATR3'],
        ['SLC25A1','TCERG1'],
        ['SLC25A23','CCNC'],
        ['SLC25A24','TMED2'],
        ['SLC25A25-AS1','ARID2'],
        ['SLC25A25-AS1','LMAN1'],
        ['SLC25A32','LAPTM4B'],
        ['SLC25A32','UBE2W'],
        ['SLC25A3','CARM1'],
        ['SLC25A3','TP53BP2'],
        ['SLC25A3','ZNF768'],
        ['SLC25A42','NAGLU'],
        ['SLC25A5','CTNNB1'],
        ['SLC25A5','DDB1'],
        ['SLC25A5','NYNRIN'],
        ['SLC25A5','PRICKLE4'],
        ['SLC25A5','RBM28'],
        ['SLC25A5','SNX17'],
        ['SLC25A5','TOMM6'],
        ['SLC25A6','C1ORF115'],
        ['SLC25A6','EEF2'],
        ['SLC25A6','MTATP6P1'],
        ['SLC27A2','EEF2'],
        ['SLC2A1','LSP1'],
        ['SLC30A5','RPS19'],
        ['SLC30A6','STRAP'],
        ['SLC35A3','HIAT1'],
        ['SLC35A4','LRCH4'],
        ['SLC35A5','SLC26A2'],
        ['SLC35B1','VIM'],
        ['SLC35B3','ACTN4'],
        ['SLC35D2','TIMELESS'],
        ['SLC35E1','WWC3'],
        ['SLC35F2','LINC00657'],
        ['SLC38A2','AP000295.9'],
        ['SLC38A2','HIST3H2A'],
        ['SLC38A2','HSF1'],
        ['SLC38A2','IL10RB'],
        ['SLC38A2','KAT8'],
        ['SLC38A2','YLPM1'],
        ['SLC39A10','KPNA1'],
        ['SLC39A11','NANS'],
        ['SLC39A14','YWHAH'],
        ['SLC39A1','CRTC2'],
        ['SLC39A1','RPS6'],
        ['SLC39A3','EMG1'],
        ['SLC39A6','DYRK4'],
        ['SLC39A8','SNRPD3'],
        ['SLC41A2','LONP2'],
        ['SLC44A1','SPATS2L'],
        ['SLC47A1','APLP2'],
        ['SLC47A1','ZBTB44'],
        ['SLC4A2','TYW1'],
        ['SLC5A3','SYAP1'],
        ['SLC6A8','ACO2'],
        ['SLC6A9','CDC23'],
        ['SLC7A11','DGKE'],
        ['SLC7A2','CEP350'],
        ['SLC7A2','SPTBN1'],
        ['SLC9A3R1','LMNB2'],
        ['SLC9A6','GARS'],
        ['SLC9A7','ARCN1'],
        ['SLC9B2','WRAP53'],
        ['SLFN11','KRI1'],
        ['SLFN11','TMEM9B'],
        ['SLIRP','CCT7'],
        ['SLIRP','TIA1'],
        ['SLIT2','EIF2AK1'],
        ['SLK','RRAGD'],
        ['SLMO2','FNIP1'],
        ['SLMO2','MICALL1'],
        ['SLMO2','TRAPPC2L'],
        ['SLU7','TIMM13'],
        ['SMAD2','GABRB3'],
        ['SMAD2','GNB2L1'],
        ['SMAD2','PITRM1'],
        ['SMAP1','DDX46'],
        ['SMAP1','KDELR2'],
        ['SMAP2','BTG1'],
        ['SMAP2','ZFP69B'],
        ['SMARCA2','RPF2'],
        ['SMARCB1','GNL1'],
        ['SMARCB1','SERTAD3'],
        ['SMARCB1','TTC38'],
        ['SMARCC1','ZNF362'],
        ['SMARCD1','UQCRB'],
        ['SMARCD2','RPL8'],
        ['SMARCD3','HRSP12'],
        ['SMC1A','ALDH3A2'],
        ['SMC1A','CMPK1'],
        ['SMC1A','DNAJB1'],
        ['SMC1A','EID2'],
        ['SMC1A','GLA'],
        ['SMC1A','SF3B1'],
        ['SMC1A','VDAC3'],
        ['SMC2','H2AFZ'],
        ['SMC4','SQLE'],
        ['SMCHD1','SUPT4H1'],
        ['SMCHD1','VPS33B'],
        ['SMCR8','SMG5'],
        ['SMEK1','ALDH5A1'],
        ['SMG1','FUS'],
        ['SMG1','PRDX1'],
        ['SMG5','PAQR6'],
        ['SMG5','RRP36'],
        ['SMG6','PPP5C'],
        ['SMG7-AS1','IL1A'],
        ['SMG7','TRPC4AP'],
        ['SMIM12','HNRNPUL1'],
        ['SMIM15','ACLY'],
        ['SMIM4','CUEDC2'],
        ['SMIM7','MCM5'],
        ['SMOX','LINC01433'],
        ['SMPD2','MCM10'],
        ['SMS','APP'],
        ['SMS','CCSER2'],
        ['SNAP23','RTF1'],
        ['SNAPIN','EEF2'],
        ['SNAPIN','NPTX1'],
        ['SND1','PPIL4'],
        ['SND1','UBA1'],
        ['SNF8','ENDOG'],
        ['SNF8','ZBTB18'],
        ['SNRNP70','CCDC88C'],
        ['SNRNP70','OAT'],
        ['SNRPB','APPL1'],
        ['SNRPB','ATXN3'],
        ['SNRPB','FAM210B'],
        ['SNRPB','POMGNT1'],
        ['SNRPB','STAT5B'],
        ['SNRPD1','CSDE1'],
        ['SNRPD2','CKB'],
        ['SNRPF','CLTB'],
        ['SNRPF','PTMA'],
        ['SNRPG','SMS'],
        ['SNW1','KRAS'],
        ['SNW1','PABPC4'],
        ['SNX10','ZNHIT1'],
        ['SNX13','CCNI'],
        ['SNX18','ZNF121'],
        ['SNX19','UQCRC2'],
        ['SNX27','CALM1'],
        ['SNX2','AMOT'],
        ['SNX2','HSD17B4'],
        ['SNX2','LRRFIP1'],
        ['SNX2','RNF7'],
        ['SNX2','ZSCAN12'],
        ['SNX32','MDM2'],
        ['SNX3','EIF3D'],
        ['SNX9','MBNL1'],
        ['SOAT1','RCOR1'],
        ['SOAT1','ZYG11A'],
        ['SOCS2','EIF4A3'],
        ['SOCS4','EDF1'],
        ['SOCS4','HDGFRP2'],
        ['SOCS5','ZNF326'],
        ['SOCS6','CDK17'],
        ['SOD1','AHSA1'],
        ['SOD1','BRD2'],
        ['SOD1','GMNN'],
        ['SOD1','IFT20'],
        ['SOD1','PIGX'],
        ['SOD1','PKM'],
        ['SOD1','POLI'],
        ['SOD1','SRSF7'],
        ['SOD2','CANX'],
        ['SOD2','EEF2'],
        ['SOD2','UBA52'],
        ['SORD','FASN'],
        ['SORT1','ACTG1'],
        ['SOS1-IT1','BRMS1'],
        ['SOX4','EIF3B'],
        ['SOX4','SELO'],
        ['SOX4','TMEM141'],
        ['SP140','RTEL1'],
        ['SP140','RTEL1-TNFRSF6B'],
        ['SP1','PHF8'],
        ['SP2-AS1','LMNB2'],
        ['SPAG5','HSPA14'],
        ['SPAG5','LETM1'],
        ['SPAG5','WNK3'],
        ['SPAG9','CIRBP'],
        ['SPAG9','EIF2S3'],
        ['SPAG9','LIN28B'],
        ['SPAG9','STK26'],
        ['SPAST','UBA6'],
        ['SPAST','VPS41'],
        ['SPATA2L','MCTP2'],
        ['SPC24','NVL'],
        ['SPC25','ZNF227'],
        ['SPCS3','STRIP1'],
        ['SPDL1','GSTO1'],
        ['SPDYA','SERINC3'],
        ['SPECC1L-ADORA2A','EPB41L3'],
        ['SPECC1L','CNN2'],
        ['SPECC1L','EPB41L3'],
        ['SPG11','GRWD1'],
        ['SPG20','ABHD15'],
        ['SPIDR','ADGRL2'],
        ['SPIDR','HSP90AB1'],
        ['SPIN1','ATXN7L3B'],
        ['SPIN1','SFSWAP'],
        ['SPINT2','ODC1'],
        ['SPN','LINC00907'],
        ['SPN','NUB1'],
        ['SPN','PCID2'],
        ['SPN','ZMYM2'],
        ['SPOPL','NUDT15'],
        ['SPOP','RPL18'],
        ['SPOP','RPL27'],
        ['SPPL3','SRSF4'],
        ['SPR','GDI2'],
        ['SPRTN','PRDX1'],
        ['SPTAN1','CCND2'],
        ['SPTBN2','NSD1'],
        ['SPTBN2','PPP1R14B'],
        ['SQSTM1','CSNK1E'],
        ['SQSTM1','TNPO1'],
        ['SRA1','SDR39U1'],
        ['SREK1IP1','ENO1'],
        ['SREK1IP1','PES1'],
        ['SRGAP2B','SRGAP2C'],
        ['SRGN','CTC1'],
        ['SRM','EEF2'],
        ['SRP54','ILF3'],
        ['SRP68','RAC3'],
        ['SRP72','ENO1'],
        ['SRP9','DSTN'],
        ['SRP9','MARK3'],
        ['SRPK1','DCAF7'],
        ['SRPK1','DOLK'],
        ['SRPK1','GLS'],
        ['SRPK1','ITM2B'],
        ['SRPK1','KAT8'],
        ['SRPK1','NQO2'],
        ['SRPK1','PODXL'],
        ['SRPK1','RC3H2'],
        ['SRPRB','CCT7'],
        ['SRPRB','FKBP1A'],
        ['SRPR','TPI1'],
        ['SRRM2','ACLY'],
        ['SRRM2','CSDE1'],
        ['SRRM2','CXXC1'],
        ['SRRM2','DENND1C'],
        ['SRRM2','TRIM11'],
        ['SRRT','CD3EAP'],
        ['SRRT','NCL'],
        ['SRSF10','CAPN2'],
        ['SRSF10','PDIA6'],
        ['SRSF11','ARPC2'],
        ['SRSF11','CRKL'],
        ['SRSF11','NDUFA6'],
        ['SRSF11','TMEM30A'],
        ['SRSF1','EIF4A1'],
        ['SRSF1','SENP3-EIF4A1'],
        ['SRSF1','SFPQ'],
        ['SRSF1','TBCB'],
        ['SRSF1','UBA52'],
        ['SRSF1','UBE2L3'],
        ['SRSF2','PEX19'],
        ['SRSF2','PSMD8'],
        ['SRSF6','RPS6KA3'],
        ['SRSF7','DCP2'],
        ['SRSF7','MRPL28'],
        ['SRSF9','GOLPH3L'],
        ['SS18','B4GALT6'],
        ['SSBP1','GGPS1'],
        ['SSBP1','PTDSS2'],
        ['SSB','SPTB'],
        ['SSR3','RPS4X'],
        ['SSR4','ATP6V1A'],
        ['SSR4','EPB41L3'],
        ['SSR4','PPA1'],
        ['SSX2IP','CLSTN1'],
        ['ST13','SBF1'],
        ['ST6GAL1','ZNF384'],
        ['STAG1','DCXR'],
        ['STAG2','GLUL'],
        ['STAG2','HSPA1B'],
        ['STAG2','SHPK'],
        ['STAG2','SUCLG1'],
        ['STAMBPL1','CMSS1'],
        ['STARD10','EEF2'],
        ['STARD4-AS1','NDNL2'],
        ['STARD7','GSTP1'],
        ['STARD7','MAGT1'],
        ['STARD7','PAM'],
        ['STARD7','PGK1'],
        ['STARD9','FRMD4A'],
        ['STAT1','HDGFRP2'],
        ['STAT2','TTC13'],
        ['STAT3','UBXN7'],
        ['STEAP3','SKIV2L2'],
        ['STIP1','SCOC-AS1'],
        ['STIP1','TTC27'],
        ['STK10','ARHGDIB'],
        ['STK10','TMSB10'],
        ['STK10','XPO6'],
        ['STK19','ZRANB2'],
        ['STK24','AP3M1'],
        ['STK25','DDX39A'],
        ['STK26','PABPN1'],
        ['STK38L','ACTR6'],
        ['STK38L','CYP51A1'],
        ['STK40','ARID1A'],
        ['STKLD1','EARS2'],
        ['STMN1','DDX17'],
        ['STMN3','PSMD8'],
        ['STOML2','BDP1'],
        ['STOML2','SUPT16H'],
        ['STOX2','ALDOA'],
        ['STRA13','RPL22'],
        ['STRA6','FEM1B'],
        ['STRAP','STUB1'],
        ['STRIP1','IP6K2'],
        ['STRN4','ETF1'],
        ['STRN4','VARS2'],
        ['STT3A','AKAP1'],
        ['STT3A','ARMCX3'],
        ['STT3A','CANX'],
        ['STT3B','ELOVL2'],
        ['STUB1','KIAA0100'],
        ['STUB1','KIF1C'],
        ['STX12','ADD3'],
        ['STX7','ACLY'],
        ['STX7','PSMC5'],
        ['STYXL1','TMEM120A'],
        ['SUB1','HIF1A'],
        ['SUB1','QRSL1'],
        ['SUB1','SLC1A5'],
        ['SUCLG1','TM9SF2'],
        ['SUCO','CSRP1'],
        ['SUDS3','C9ORF41'],
        ['SUGP1','EMC8'],
        ['SUGT1','OBSL1'],
        ['SUMF1','HNRNPU'],
        ['SUPT16H','MLF2'],
        ['SUPT20HL1','RPL14'],
        ['SUPT3H','ARCN1'],
        ['SUPT4H1','EIF3I'],
        ['SUPT6H','CDC34'],
        ['SUPT6H','GLA'],
        ['SUPV3L1','MRPS21'],
        ['SUPV3L1','PRC1'],
        ['SURF2','STK11'],
        ['SURF4','KHSRP'],
        ['SURF4','NOM1'],
        ['SURF4','SURF1'],
        ['SUSD1','WASF2'],
        ['SUV39H1','EIF3K'],
        ['SUV39H2','HNRNPA3'],
        ['SUV39H2','TM9SF2'],
        ['SYAP1','SERF2'],
        ['SYF2','ATP11C'],
        ['SYF2','NQO2'],
        ['SYF2','SART3'],
        ['SYMPK','REV3L'],
        ['SYNCRIP','EIF5B'],
        ['SYNCRIP','EPHX2'],
        ['SYNCRIP','PSMA4'],
        ['SYNCRIP','UBB'],
        ['SYNE2','ARL11'],
        ['SYNE2','CD74'],
        ['SYNJ2BP','COX16'],
        ['SYNJ2BP-COX16','BRSK2'],
        ['SYNJ2BP','ENO1'],
        ['SYPL1','MAZ'],
        ['SYT1','SRSF1'],
        ['SYT8','TNNI2'],
        ['SYVN1','GAA'],
        ['SYVN1','GRAMD1A'],
        ['TAB2','DST'],
        ['TAB2','RPL27'],
        ['TACC1','ANXA11'],
        ['TACC1','FAN1'],
        ['TACC1','TBCD'],
        ['TADA1','AKAP2'],
        ['TADA1','PALM2-AKAP2'],
        ['TAF11','GTF2A1'],
        ['TAF13','MAGI3'],
        ['TAF1B','TUFM'],
        ['TAF1C','GNA13'],
        ['TAF2','TMED7'],
        ['TAF2','USP22'],
        ['TAF4B','UQCRC2'],
        ['TAF6','GTF2F1'],
        ['TAF6L','AGBL3'],
        ['TAF7','FBL'],
        ['TAGLN2','RAB11FIP5'],
        ['TALDO1','BAG1'],
        ['TANGO6','POMP'],
        ['TAOK1','CDC42'],
        ['TAOK2','SMC1A'],
        ['TAOK3','PDCD4'],
        ['TAP2','DPP9'],
        ['TAPBP','DGKD'],
        ['TARBP1','CDKN3'],
        ['TARDBP','MBTD1'],
        ['TARS','EBNA1BP2'],
        ['TARS','EIF3B'],
        ['TARS','S100A10'],
        ['TAS2R14','POLK'],
        ['TASP1','MARCKS'],
        ['TATDN3','THOC5'],
        ['TAZ','VIM'],
        ['TBC1D1','POLR2B'],
        ['TBC1D20','RANGAP1'],
        ['TBC1D23','RNF149'],
        ['TBC1D32','TPX2'],
        ['TBC1D5','BOP1'],
        ['TBC1D8B','CCNC'],
        ['TBC1D8B','TRIP6'],
        ['TBCB','BRD8'],
        ['TBCB','ZNF428'],
        ['TBCE','NELFB'],
        ['TBCE','NUCKS1'],
        ['TBL1XR1','MTHFD2'],
        ['TBPL1','COLGALT1'],
        ['TC2N','SRSF1'],
        ['TCEAL1','EEF2'],
        ['TCEAL8','EEF2'],
        ['TCEANC','RAB9A'],
        ['TCEB1','GGCX'],
        ['TCEB2','AKAP10'],
        ['TCEB2','KLHL28'],
        ['TCEB3','AZIN1'],
        ['TCEB3','LYPLA1'],
        ['TCEB3','PIK3R2'],
        ['TCEB3','SACS'],
        ['TCERG1','EMC3'],
        ['TCERG1','RBBP7'],
        ['TCF19','ACBD6'],
        ['TCF19','SP3'],
        ['TCF20','HNRNPU'],
        ['TCF20','PCGF6'],
        ['TCF25','MRPS18B'],
        ['TCF3','PRPSAP2'],
        ['TCFL5','CYB5B'],
        ['TCFL5','DDHD2'],
        ['TCFL5','FBXO46'],
        ['TCOF1','HINT3'],
        ['TCP1','GAPDH'],
        ['TCP1','NHP2L1'],
        ['TCP1','NKIRAS2'],
        ['TCP1','PSMB1'],
        ['TCTN1','RPL35A'],
        ['TDP2','SEC23B'],
        ['TDRKH','CD3EAP'],
        ['TECR','DDX39A'],
        ['TELO2','RPS21'],
        ['TENM3','ENO1'],
        ['TERF2IP','GCN1L1'],
        ['TERF2IP','PCMT1'],
        ['TET1','FAM114A2'],
        ['TEX2','SNRNP40'],
        ['TEX2','SOAT1'],
        ['TFG','IFI16'],
        ['TFIP11','CCND2'],
        ['TFRC','CRTAP'],
        ['TFRC','GOLGB1'],
        ['TFRC','PCNXL4'],
        ['TFRC','RANBP9'],
        ['TFRC','SUPT6H'],
        ['TGFBR1','TMCO1'],
        ['TGIF1','MAPKAPK5'],
        ['THAP1','ALKBH5'],
        ['THAP4','NUDT19'],
        ['THAP6','EIF1'],
        ['THAP7','THAP11'],
        ['THEM4','CRIM1'],
        ['THOC2','MAPRE2'],
        ['THRAP3','TIMM17A'],
        ['THRAP3','ZBED1'],
        ['THRIL','IRAK1BP1'],
        ['THUMPD2','KIAA0100'],
        ['TIAL1','SDHB'],
        ['TIAM1','KHSRP'],
        ['TICAM2','HERC1'],
        ['TICRR','RPL8'],
        ['TIGD5','TOM1L1'],
        ['TIGD5','YIPF1'],
        ['TIMELESS','FKBP3'],
        ['TIMM13','APMAP'],
        ['TIMM13','RPL8'],
        ['TIMM17A','PHB'],
        ['TIMM17A','RPL3'],
        ['TIMM17A','SLC6A8'],
        ['TIMM21','RNF40'],
        ['TIMM23B','LINC00843'],
        ['TIMM44','DDX19A'],
        ['TIMM44','RAB11A'],
        ['TIMM9','ERI3'],
        ['TIMMDC1','RTFDC1'],
        ['TIPARP','ACTG1'],
        ['TIPIN','AFG3L2'],
        ['TIPIN','CNBP'],
        ['TIRAP','DCPS'],
        ['TK1','RNF216'],
        ['TKT','RPS19'],
        ['TLE3','AEN'],
        ['TLE3','EIF4B'],
        ['TLK1','EPCAM'],
        ['TLK1','SRSF1'],
        ['TLN1','EEF1A1'],
        ['TLN2','BCL2L13'],
        ['TM2D2','SPDL1'],
        ['TM7SF3','C1QBP'],
        ['TM7SF3','EPPK1'],
        ['TM9SF2','TK1'],
        ['TM9SF3','ANAPC5'],
        ['TM9SF3','DDX3X'],
        ['TM9SF3','EIF3B'],
        ['TM9SF3','GLO1'],
        ['TM9SF3','RYBP'],
        ['TM9SF3','SFRP1'],
        ['TMBIM6','CCDC59'],
        ['TMBIM6','GOPC'],
        ['TMBIM6','MCFD2'],
        ['TMBIM6','SLC16A1'],
        ['TMC8','MAGED1'],
        ['TMCO3','NDUFAF4'],
        ['TMED10','H2AFZ'],
        ['TMED10','TP53'],
        ['TMED1','RTKN'],
        ['TMED2','FOXK2'],
        ['TMED2','HNRNPA2B1'],
        ['TMED2','MAN1A2'],
        ['TMED2','MINK1'],
        ['TMED2','SNRNP200'],
        ['TMED3','RLF'],
        ['TMED3','RPN1'],
        ['TMED3','TTI1'],
        ['TMED5','SNRNP200'],
        ['TMED7','RPLP2'],
        ['TMED9','ALKBH5'],
        ['TMED9','ANKFY1'],
        ['TMEFF1','VDAC1'],
        ['TMEM106B','DDOST'],
        ['TMEM106B','TOMM22'],
        ['TMEM108','ACTG1'],
        ['TMEM109','STK10'],
        ['TMEM117','MTF2'],
        ['TMEM127','XIAP'],
        ['TMEM129','PLEKHJ1'],
        ['TMEM132A','ATP5D'],
        ['TMEM132E','SQSTM1'],
        ['TMEM147','NNT'],
        ['TMEM14B','PARK7'],
        ['TMEM14B','RHOA'],
        ['TMEM14C','HDLBP'],
        ['TMEM14C','MTHFD2'],
        ['TMEM161A','FBXO11'],
        ['TMEM165','TMEM64'],
        ['TMEM167A','CS'],
        ['TMEM167B','AKT1'],
        ['TMEM167B','PRMT1'],
        ['TMEM177','KLHL15'],
        ['TMEM178B','CLIC1'],
        ['TMEM184B','DMC1'],
        ['TMEM185B','RNF14'],
        ['TMEM187','FAM57A'],
        ['TMEM189','DUSP14'],
        ['TMEM189-UBE2V1','DUSP14'],
        ['TMEM201','TRIB3'],
        ['TMEM209','OSBPL8'],
        ['TMEM237','DDOST'],
        ['TMEM248','CIRH1A'],
        ['TMEM251','EIF3B'],
        ['TMEM251','ID4'],
        ['TMEM251','KIAA0100'],
        ['TMEM258','PPP2R1A'],
        ['TMEM260','ALDH18A1'],
        ['TMEM30A','SDAD1'],
        ['TMEM30A','TKFC'],
        ['TMEM41A','SETD4'],
        ['TMEM43','CBX6'],
        ['TMEM50A','GGCT'],
        ['TMEM50A','HNRNPH1'],
        ['TMEM50A','TOP2A'],
        ['TMEM50B','TOMM22'],
        ['TMEM65','FBXO25'],
        ['TMEM65','KCNK2'],
        ['TMEM70','ANKRD11'],
        ['TMEM8A','SRSF7'],
        ['TMEM97','USF2'],
        ['TMEM98','KIAA1715'],
        ['TMOD3','ATP6V1G1'],
        ['TMPO','ABR'],
        ['TMPO','ETFA'],
        ['TMPO','LRP10'],
        ['TMPO','MED8'],
        ['TMPO','SERBP1'],
        ['TMSB10','HTR3A'],
        ['TMSB10','MRPS7'],
        ['TMSB10','TUBB4B'],
        ['TMSB4X','VCP'],
        ['TMTC1','SLC39A14'],
        ['TMUB2','ALDH18A1'],
        ['TMX1','TBL2'],
        ['TMX2','CTNND1'],
        ['TMX2','TRMT112'],
        ['TMX3','STARD7'],
        ['TNFAIP3','ATXN1L'],
        ['TNFAIP3','PSAP'],
        ['TNFAIP3','RPS16'],
        ['TNFAIP3','SPOP'],
        ['TNFRSF14','CCR7'],
        ['TNFRSF17','SNX29'],
        ['TNFRSF1A','ENO1'],
        ['TNIP1','ALDH18A1'],
        ['TNNT1','USP7'],
        ['TNPO1','DDX54'],
        ['TNPO1','G3BP2'],
        ['TOM1L1','PDCD5'],
        ['TOM1L1','PRMT1'],
        ['TOMM20','APEH'],
        ['TOMM20','ARHGEF39'],
        ['TOMM20','RAP1GAP2'],
        ['TOMM22','ACBD6'],
        ['TOMM22','ZMYM4'],
        ['TOMM40','MTMR10'],
        ['TOMM7','DDB1'],
        ['TOMM7','PML'],
        ['TOP1','TNIK'],
        ['TOP2A','GNB1'],
        ['TOP2A','MRFAP1'],
        ['TOP2A','NUDC'],
        ['TOP2A','NUDT16'],
        ['TOP2A','PCNP'],
        ['TOP2A','PHF14'],
        ['TOP2A','PSMD8'],
        ['TOP2B','FN1'],
        ['TOP2B','LDHB'],
        ['TOP2B','SART3'],
        ['TOP2B','ZMPSTE24'],
        ['TOPBP1','ANKRD11'],
        ['TOR1AIP2','SMARCE1'],
        ['TOX4','POC1A'],
        ['TP53BP1','CCDC47'],
        ['TP53BP2','CEBPG'],
        ['TP53','SCLT1'],
        ['TPD52L2','KATNAL1'],
        ['TPI1','CCT7'],
        ['TPI1','DDX55'],
        ['TPI1','PRDX1'],
        ['TPI1','WDR1'],
        ['TPM3','CELF1'],
        ['TPM3','GLUL'],
        ['TPM3','SRSF11'],
        ['TPM4','LTBP1'],
        ['TPM4','SF3B2'],
        ['TPP2','ACVR2B'],
        ['TPR','CKB'],
        ['TPR','HMGB3'],
        ['TPRN','TMSB10'],
        ['TPT1','ELOVL5'],
        ['TPT1','PIM2'],
        ['TPT1','POLR3E'],
        ['TPT1','RNF169'],
        ['TPTE2P1','CENPJ'],
        ['TPX2','FTO'],
        ['TPX2','NUCKS1'],
        ['TRA2A','PSMA7'],
        ['TRA2B','FAIM'],
        ['TRA2B','IL6ST'],
        ['TRA2B','WIPF2'],
        ['TRAF5','SLC12A6'],
        ['TRAM1','FURIN'],
        ['TRANK1','ANP32E'],
        ['TRAP1','IGLV2-23'],
        ['TRAP1','SIRT7'],
        ['TRAPPC13','TCERG1'],
        ['TRAPPC2L','MTMR1'],
        ['TRAPPC3','EIF3D'],
        ['TRAPPC4','LRP3'],
        ['TRAPPC5','APBB3'],
        ['TRAPPC6B','AURKAIP1'],
        ['TRB@','SDHAF4'],
        ['TREML2','TREML5P'],
        ['TRG@','MX1'],
        ['TRIAP1','MCM5'],
        ['TRIB2','HNRNPU'],
        ['TRIB3','GABARAP'],
        ['TRIM11','FAM32A'],
        ['TRIM14','ATP6V0E1'],
        ['TRIM25','HLCS'],
        ['TRIM25','PDCD5'],
        ['TRIM28','ACTB'],
        ['TRIM28','EGLN2'],
        ['TRIM28','RAB4B-EGLN2'],
        ['TRIM2','THRAP3'],
        ['TRIM37','DLD'],
        ['TRIM38','SLC25A1'],
        ['TRIM44','MSN'],
        ['TRIM52','UFL1'],
        ['TRIM56','YLPM1'],
        ['TRIM69','SIGMAR1'],
        ['TRIM8','NUBP2'],
        ['TRIP12','POLR3H'],
        ['TRIP4','U2SURP'],
        ['TRIQK','RAD50'],
        ['TRIT1','CCT2'],
        ['TRMT11','DCTN3'],
        ['TRMT1L','FAM159B'],
        ['TRMT2A','IRAK1'],
        ['TRNAU1AP','GMEB1'],
        ['TROVE2','CNOT1'],
        ['TRPM7','MRPL28'],
        ['TRPM7','USP50'],
        ['TRPS1','FKBP10'],
        ['TRRAP','KDM4C'],
        ['TRRAP','NR1D2'],
        ['TRUB1','ALDH16A1'],
        ['TRUB1','PAFAH1B2'],
        ['TSC22D1','MAGT1'],
        ['TSC2','TRAP1'],
        ['TSEN54','NF1'],
        ['TSFM','FUBP1'],
        ['TSFM','RAB40C'],
        ['TSG101','NAA15'],
        ['TSG101','SPAG9'],
        ['TSIX','SDC2'],
        ['TSNAX-DISC1','YWHAE'],
        ['TSNAX','PABPC1'],
        ['TSNAX','YWHAE'],
        ['TSN','ZFP64'],
        ['TSPAN33','SEPT11'],
        ['TSPAN3','MYH9'],
        ['TSPAN3','RPL9'],
        ['TSPYL1','CCDC132'],
        ['TSPYL4','RPP40'],
        ['TSTD1','F11R'],
        ['TST','MOAP1'],
        ['TTC13','EIF3K'],
        ['TTC13','QARS'],
        ['TTC19','CD79A'],
        ['TTC19','ERI2'],
        ['TTC21B','EEF1D'],
        ['TTC27','SLC25A39'],
        ['TTC28','SECISBP2'],
        ['TTC37','NFATC3'],
        ['TTC37','ZFHX4'],
        ['TTC7A','NUDCD3'],
        ['TTC7B','AKT2'],
        ['TTC7B','MVD'],
        ['TTC8','CLDND1'],
        ['TTF2','MCU'],
        ['TTK','ACO2'],
        ['TTK','EIF4G1'],
        ['TTLL12','C1QL4'],
        ['TTLL12','EIF3A'],
        ['TTLL12','RPL37A'],
        ['TTN-AS1','MYH9'],
        ['TUBA1B','CKB'],
        ['TUBA1B','HDLBP'],
        ['TUBA1B','RPLP0'],
        ['TUBA1B','SEPT9'],
        ['TUBA1B','TNFRSF1A'],
        ['TUBB4B','ZNF260'],
        ['TUBB6','ZDHHC2'],
        ['TUBB','LARP1'],
        ['TUBB','LTA4H'],
        ['TUBB','NDUFV1'],
        ['TUBB','SREK1'],
        ['TUBGCP2','DNPEP'],
        ['TUBGCP2','GAPDH'],
        ['TUBGCP2','PCNA'],
        ['TUSC2','ATF6'],
        ['TUSC3','FASTKD2'],
        ['TUSC3','PLS3'],
        ['TWIST2','SNAP47'],
        ['TWISTNB','CIAO1'],
        ['TXLNA','CUL3'],
        ['TXLNA','YRDC'],
        ['TXLNG','C16ORF72'],
        ['TXLNG','ELOVL1'],
        ['TXLNG','GHITM'],
        ['TXNDC11','ATXN10'],
        ['TXNDC11','LPGAT1'],
        ['TXNDC15','EIF2S1'],
        ['TXNDC5','PKM'],
        ['TXNDC5','UBC'],
        ['TXNDC5','UTP18'],
        ['TXNDC9','UGDH'],
        ['TXNL1','GANAB'],
        ['TXNL1','TRAPPC5'],
        ['TXNL4A','MCM2'],
        ['TXNL4A','NDUFS1'],
        ['TXNL4A','PHF14'],
        ['TXNL4A','RAF1'],
        ['TXNL4A','TMPO'],
        ['TXNRD2','RPL17-C18ORF32'],
        ['TYRO3','ARHGEF10L'],
        ['U2SURP','MRPS7'],
        ['U2SURP','NADK'],
        ['U73166.2','SEMA3B'],
        ['UACA','USP48'],
        ['UAP1L1','PRC1'],
        ['UBA1','CSRP1'],
        ['UBA1','FUNDC1'],
        ['UBA3','SMIM7'],
        ['UBA52','CRCP'],
        ['UBA6','LARP1'],
        ['UBA6','LIN9'],
        ['UBA6','MRPL42'],
        ['UBAC1','BAZ1B'],
        ['UBAC1','TCFL5'],
        ['UBB','COX15'],
        ['UBB','EEF2'],
        ['UBB','MCM3'],
        ['UBB','ZNF252P'],
        ['UBC','GLB1'],
        ['UBE2A','PKM'],
        ['UBE2A','RAB3B'],
        ['UBE2D2','AARS'],
        ['UBE2D2','IFI16'],
        ['UBE2D2','UNC13B'],
        ['UBE2D3','KIAA1467'],
        ['UBE2D3','RPL30'],
        ['UBE2E1','BBS10'],
        ['UBE2E1','NOP10'],
        ['UBE2E2','TOP2B'],
        ['UBE2G2','LIMA1'],
        ['UBE2G2','RABEP1'],
        ['UBE2H','WASL'],
        ['UBE2I','PSMB4'],
        ['UBE2K','NPM3'],
        ['UBE2M','SDE2'],
        ['UBE2O','C17ORF80'],
        ['UBE2Q2','METTL17'],
        ['UBE2R2','APLP2'],
        ['UBE2S','SF3B2'],
        ['UBE2Z','EIF3D'],
        ['UBE3A','CLASRP'],
        ['UBE3D','DDX54'],
        ['UBE3D','TMEM243'],
        ['UBE4A','DCTN2'],
        ['UBE4A','FAM114A2'],
        ['UBE4A','MRPL20'],
        ['UBL7','CYB561D2'],
        ['UBL7','KCNQ2'],
        ['UBLCP1','PKM'],
        ['UBQLN1','DANCR'],
        ['UBQLN2','SPTBN1'],
        ['UBR2','SLC30A9'],
        ['UBR5','EFNB2'],
        ['UBR5','FAM229B'],
        ['UBR5','FGR'],
        ['UBR5','HMGB1'],
        ['UBR7','HEXIM1'],
        ['UBXN2A','C14ORF166'],
        ['UBXN6','AMD1'],
        ['UBXN6','WHSC1'],
        ['UCHL5','PRDX1'],
        ['UCK1','LDHB'],
        ['UCK2','CYFIP2'],
        ['UCK2','MRPL39'],
        ['UFC1','POLR2D'],
        ['UFC1','VAT1'],
        ['UFL1','MTDH'],
        ['UFL1','ZNF592'],
        ['UFM1','MDM2'],
        ['UGCG','CHP1'],
        ['UGCG','PEX3'],
        ['UGGT1','GAS5'],
        ['UGGT1','TMBIM6'],
        ['UGP2','PMPCA'],
        ['UHRF1','DDX5'],
        ['ULK3','AATF'],
        ['UMAD1','RAD51C'],
        ['UNC13B','TJP2'],
        ['UNC13D','MRPL9'],
        ['UNC45A','EIF2AK2'],
        ['UNG','NFRKB'],
        ['UNKL','WIPF1'],
        ['UNK','RPN2'],
        ['UPF1','CDKN2C'],
        ['UPF2','PKN1'],
        ['UPF3B','OPHN1'],
        ['UQCC1','PLEC'],
        ['UQCC2','SEC16A'],
        ['UQCR10','EEF2'],
        ['UQCR10','HN1'],
        ['UQCRC2','IRF2BP2'],
        ['UQCRC2','MOB1A'],
        ['UQCRQ','C6ORF120'],
        ['UQCRQ','MCM6'],
        ['URB1','MAPKAPK5'],
        ['URI1','PLK4'],
        ['UROD','BIVM'],
        ['UROS','C5ORF49'],
        ['USMG5','DDX17'],
        ['USMG5','EIF3L'],
        ['USMG5','MRPL14'],
        ['USP12','LRBA'],
        ['USP13','PAFAH1B2'],
        ['USP16','CYC1'],
        ['USP16','IRAK1'],
        ['USP1','GTF2F2'],
        ['USP1','INO80B'],
        ['USP1','INO80B-WBP1'],
        ['USP22','GRHL1'],
        ['USP22','MTMR4'],
        ['USP22','TMED2'],
        ['USP24','PELI2'],
        ['USP33','TOMM20'],
        ['USP34','KNTC1'],
        ['USP34','SRA1'],
        ['USP37','ZNF768'],
        ['USP38','NACA'],
        ['USP45','ZNF507'],
        ['USP47','RPL8'],
        ['USP48','B2M'],
        ['USP48','HPS4'],
        ['USP5','CLTA'],
        ['USP5','MYBL2'],
        ['USP5','OTUD7B'],
        ['USP9X','CBX5'],
        ['USP9X','DICER1'],
        ['USP9X','DTNA'],
        ['USP9X','GINS1'],
        ['USP9X','HDGFRP2'],
        ['USP9X','UBB'],
        ['USP9X','YY1AP1'],
        ['USPL1','ZFAND5'],
        ['UTP11L','FMNL2'],
        ['UTP14C','MYBL2'],
        ['UTP18','EEF1G'],
        ['UTP18','MTA2'],
        ['UTP6','COPRS'],
        ['UTRN','ATP2A2'],
        ['UXT','RNF2'],
        ['VAMP3','AGPAT1'],
        ['VAMP3','APLP2'],
        ['VAMP3','CPEB4'],
        ['VAV1','C1ORF35'],
        ['VBP1','GPN3'],
        ['VBP1','KAT2A'],
        ['VBP1','NOP56'],
        ['VCAN','MEX3C'],
        ['VCP','CTSD'],
        ['VCP','SCRN1'],
        ['VCP','VIM'],
        ['VDAC3','CALM2'],
        ['VDAC3','RRM1'],
        ['VDAC3','TXLNA'],
        ['VHL','ATXN1'],
        ['VHL','RPL36'],
        ['VIM','C6ORF48'],
        ['VIM','CORO7'],
        ['VIM','CORO7-PAM16'],
        ['VIM','PRKCI'],
        ['VIM','SPECC1L'],
        ['VIM','SPECC1L-ADORA2A'],
        ['VIM','VDAC1'],
        ['VIM','YME1L1'],
        ['VIPAS39','NOXRED1'],
        ['VIPAS39','NUP155'],
        ['VMA21','MRPL43'],
        ['VMA21','ODC1'],
        ['VMA21','RBBP9'],
        ['VMA21','SIKE1'],
        ['VMP1','LRRC59'],
        ['VPS11','SDE2'],
        ['VPS26B','ATXN7L3B'],
        ['VPS35','LIFR'],
        ['VPS35','NELFE'],
        ['VPS37B','AHSA1'],
        ['VPS51','PKN1'],
        ['VPS51','PSMA1'],
        ['VPS8','NRF1'],
        ['VRK1','CTNNB1'],
        ['VRK1','DDX3X'],
        ['VSIG10','SDHB'],
        ['VTI1A','ADK'],
        ['VTI1A','EEF2'],
        ['VWA9','LSM14A'],
        ['VWA9','PTPRF'],
        ['WAC','RBBP7'],
        ['WARS','DDX3X'],
        ['WASF2','HSP90B1'],
        ['WBP11','LEMD2'],
        ['WBP1L','CRTAP'],
        ['WBP1L','STK4'],
        ['WBP2','MIR762HG'],
        ['WBSCR22','C1QBP'],
        ['WDR11','C7ORF49'],
        ['WDR11','HIF1A'],
        ['WDR13','PRPF19'],
        ['WDR18','DDX56'],
        ['WDR1','GOLGA4'],
        ['WDR1','PLCG2'],
        ['WDR1','SCAP'],
        ['WDR1','UBFD1'],
        ['WDR3','EIF4A3'],
        ['WDR43','NRDE2'],
        ['WDR46','MYH9'],
        ['WDR48','CYB5R4'],
        ['WDR48','PPP2R2A'],
        ['WDR59','DDX3X'],
        ['WDR5','LONP2'],
        ['WDR61','HARS'],
        ['WDR62','DYNC1H1'],
        ['WDR6','XPNPEP1'],
        ['WDR70','SKP2'],
        ['WDR70','SUPT16H'],
        ['WDR70','TPT1'],
        ['WDR76','TSR1'],
        ['WDR82','GTF3C4'],
        ['WDR83','OS9'],
        ['WDYHV1','CDIPT'],
        ['WDYHV1','HEXB'],
        ['WEE1','TPI1'],
        ['WHSC1','CENPJ'],
        ['WIPF1','BTRC'],
        ['WIPF3','ENO1'],
        ['WIPI1','CIPC'],
        ['WNK1','ARGLU1'],
        ['WRAP73','CYTH1'],
        ['WRB','MIIP'],
        ['WSB1','NDUFA5'],
        ['WSB2','UBR7'],
        ['WSB2','ZNF629'],
        ['WSCD1','ZC3HAV1'],
        ['WTAP','RPP21'],
        ['WTAP','SMIM20'],
        ['WTAP','TRIM39-RPP21'],
        ['WWC2','YWHAE'],
        ['WWOX','GNAS'],
        ['XBP1','DSE'],
        ['XBP1','SNRPB'],
        ['XIST','AARS'],
        ['XIST','HNRNPU'],
        ['XIST','LIN28B'],
        ['XIST','PIAS3'],
        ['XIST','RPL27'],
        ['XPA','EMP2'],
        ['XPC','ABI2'],
        ['XPNPEP3','AHCY'],
        ['XPO1','GATAD2A'],
        ['XPO1','PIH1D1'],
        ['XPO1','PLEKHG3'],
        ['XPO4','SKP2'],
        ['XPO5','ALG6'],
        ['XPO6','ATP5B'],
        ['XPOT','DNM1L'],
        ['XPOT','FSBP'],
        ['XPOT','RAD54B'],
        ['XRCC1','UBB'],
        ['XRCC2','SPRED2'],
        ['XRCC4','CNOT1'],
        ['XRCC5','ATP2A2'],
        ['XRCC5','CAPNS1'],
        ['XRCC5','CD55'],
        ['XRCC5','GNB2L1'],
        ['XRCC5','NDUFB9'],
        ['XRCC5','OLFM2'],
        ['XRCC5','PGS1'],
        ['XRCC5','SIRT5'],
        ['XRCC5','SLC39A7'],
        ['XRCC5','TMEM62'],
        ['XRCC6','MAZ'],
        ['XRCC6','MSH6'],
        ['XRN2','PRSS12'],
        ['XRN2','TPX2'],
        ['XXBAC-B461K10.4','CAPZB'],
        ['XXBAC-B461K10.4','JUNB'],
        ['XXBAC-BPG252P9.9','IGHA1'],
        ['YAP1','TMEM97'],
        ['YARS','FKBP4'],
        ['YARS','HEATR1'],
        ['YARS','METAP1'],
        ['YBX1','FARS2'],
        ['YBX1','LITAF'],
        ['YBX1','MARCH7'],
        ['YBX1','POGK'],
        ['YBX1','SLC23A2'],
        ['YBX3','BUB3'],
        ['YBX3','FCHO2'],
        ['YBX3','TUFM'],
        ['YDJC','C12ORF75'],
        ['YDJC','HES4'],
        ['YEATS4','CBFB'],
        ['YIF1A','E2F1'],
        ['YIPF2','EEF2'],
        ['YIPF5','KCTD20'],
        ['YKT6','RPS14'],
        ['YKT6','VAMP1'],
        ['YOD1','ACLY'],
        ['YTHDC1','ATP5B'],
        ['YTHDC1','HMGB2'],
        ['YTHDF2','ENO1'],
        ['YWHAB','LARP7'],
        ['YWHAB','RAB10'],
        ['YWHAB','XRCC6'],
        ['YWHAE','PDCD2'],
        ['YWHAE','RPA1'],
        ['YWHAQ','MRPS24'],
        ['YWHAQ','RPN1'],
        ['YWHAQ','TIA1'],
        ['YY1','FOXG1'],
        ['ZADH2','HSPH1'],
        ['ZADH2','JMJD6'],
        ['ZBED4','SRSF3'],
        ['ZBED9','GLO1'],
        ['ZBTB11','DOCK8'],
        ['ZBTB11','TMOD3'],
        ['ZBTB33','NKAP'],
        ['ZBTB33','PNMA2'],
        ['ZBTB41','EZR'],
        ['ZBTB7A','SCPEP1'],
        ['ZBTB8A','GADD45A'],
        ['ZBTB8A','ZNF518A'],
        ['ZC3H18','HNRNPA2B1'],
        ['ZC3H3','IL16'],
        ['ZC3H4','TMEM203'],
        ['ZCCHC14','FBXL5'],
        ['ZCCHC8','STAT3'],
        ['ZDHHC12','SHKBP1'],
        ['ZER1','STIM1'],
        ['ZFAND1','MAGT1'],
        ['ZFAND5','TOPORS'],
        ['ZFAND6','CCDC6'],
        ['ZFHX3','GYS1'],
        ['ZFP14','WDR18'],
        ['ZFP30','VPS41'],
        ['ZFP62','TRAPPC3'],
        ['ZFR','RBSN'],
        ['ZFR','ZNF136'],
        ['ZFX','LEMD3'],
        ['ZFYVE16','LRRC8D'],
        ['ZFYVE19','SNRPB'],
        ['ZKSCAN1','DNAJC7'],
        ['ZKSCAN2','GNB2L1'],
        ['ZMAT2','TET1'],
        ['ZMAT5','C20ORF24'],
        ['ZMAT5','TGIF2-C20ORF24'],
        ['ZMPSTE24','IPMK'],
        ['ZMYM2','NPM1'],
        ['ZMYM2','SPN'],
        ['ZMYM3','CCDC186'],
        ['ZMYND19','RTCB'],
        ['ZNF121','ANXA7'],
        ['ZNF121','RABAC1'],
        ['ZNF121','SRRM1'],
        ['ZNF12','RPS20'],
        ['ZNF131','DIS3'],
        ['ZNF131','MAP3K1'],
        ['ZNF140','PRPF8'],
        ['ZNF146','ARL8A'],
        ['ZNF146','C4ORF32'],
        ['ZNF146','LDHB'],
        ['ZNF207','GSR'],
        ['ZNF222','CNOT1'],
        ['ZNF227','CENPF'],
        ['ZNF227','COTL1'],
        ['ZNF229','SLC25A5'],
        ['ZNF234','CYFIP2'],
        ['ZNF23','ATP5L'],
        ['ZNF24','RHOT1'],
        ['ZNF260','SLC25A16'],
        ['ZNF264','AURKC'],
        ['ZNF268','BTF3'],
        ['ZNF26','ZDHHC17'],
        ['ZNF271P','SHB'],
        ['ZNF280C','SLC39A1'],
        ['ZNF283','ACIN1'],
        ['ZNF283','PRPS2'],
        ['ZNF292','EXOSC9'],
        ['ZNF333','POLR2E'],
        ['ZNF33A','MYL6'],
        ['ZNF347','TMBIM6'],
        ['ZNF395','NDUFB11'],
        ['ZNF397','BICD2'],
        ['ZNF397','PIEZO2'],
        ['ZNF423','ARL3'],
        ['ZNF428','OGFRL1'],
        ['ZNF430','SERINC1'],
        ['ZNF451','BAG2'],
        ['ZNF460','FTL'],
        ['ZNF483','TUBB6'],
        ['ZNF490','GLG1'],
        ['ZNF511','IRS4'],
        ['ZNF516','TOR1A'],
        ['ZNF518B','TUBA1B'],
        ['ZNF542P','MICAL3'],
        ['ZNF542P','PHB2'],
        ['ZNF548','POLR1E'],
        ['ZNF569','PPP5C'],
        ['ZNF587B','CANX'],
        ['ZNF593','CLK2'],
        ['ZNF622','GNB2L1'],
        ['ZNF622','HSCB'],
        ['ZNF655','STARD7'],
        ['ZNF664','DDX24'],
        ['ZNF674-AS1','ACTG1'],
        ['ZNF674-AS1','GMPPA'],
        ['ZNF704','ZNF629'],
        ['ZNF711','GRN'],
        ['ZNF711','P4HA2'],
        ['ZNF711','SFXN1'],
        ['ZNF711','TMEM263'],
        ['ZNF711','TXNDC9'],
        ['ZNF714','EIF3B'],
        ['ZNF730','TTC37'],
        ['ZNF766','ZC3H13'],
        ['ZNF77','MON1A'],
        ['ZNF787','TPX2'],
        ['ZNF793','CCND2'],
        ['ZNF814','VPS53'],
        ['ZNF829','MRPL17'],
        ['ZNF846','FOXJ3'],
        ['ZNF84','CD276'],
        ['ZNHIT6','CSDE1'],
        ['ZNRD1','INTS2'],
        ['ZNRD1','NUP205'],
        ['ZRANB3','ST13'],
        ['ZSCAN18','NAA50'],
        ['ZSCAN29','DNAJC8'],
        ['ZW10','RPL7L1'],
        ['ZWINT','STRAP'],
        ['ZXDC','EEF1G'],
        ['ZYG11B','RBM12'],
        ['ZYX','NUP210'],
        ['ZZZ3','OS9'],
        ['C5ORF34','EEF1A1'],
        ['CAPZB','EEF2'],
        ['EEF1A1','C5ORF34'],
        ['IGH@','GRK2'],
        ['IGH@','LDHA'],
        ['IGH@','WDFY4'],
        ['NCL','PTMA'],
        ['RPL38','TTYH2'],
        ['RRM2','C2ORF48'],
        ['TTC7B','MVD']

        ]



    data = fusions.get(options.organism.lower(),[])
    if data:

        #file_symbols = os.path.join(options.output_directory,'genes_symbols.txt')
        file_symbols = os.path.join(options.output_directory,'synonyms.txt')
        loci = symbols.generate_loci(file_symbols)

        genes = symbols.read_genes_symbols(file_symbols)

        d = []
        for (g1,g2) in data:
            if g1.upper() != g2.upper():
                ens1 = symbols.ensembl(g1.upper(),genes,loci)
                ens2 = symbols.ensembl(g2.upper(),genes,loci)
                if ens1 and ens2:
                    for e1 in ens1:
                        for e2 in ens2:
                            if e1 != e2:
                                d.append([e1,e2])

        data = ['\t'.join(sorted(line)) + '\n' for line in d]
        data = sorted(set(data))

    print "%d known fusion genes found in non-cancer cell lines" % (len(data),)

    file(os.path.join(options.output_directory,'non-tumor_cells.txt'),'w').writelines(data)
    #
