#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of candidate fusion genes. This list is hard coded
in here and it is manually curated from:


TCGA, Integrated genomic characterization of oesophageal carcinoma, Nature, 
Jan. 2017, http://dx.doi.org/10.1038/nature20805



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of pre-candidate fusion genes from TCGA, Integrated genomic characterization of oesophageal carcinoma article."""
    version = "%prog 0.12 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of allowed candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of allowed candidate fusion genes is generated. Default is '%default'.""")

    parser.add_option("--skip-filter-overlap",
                      action="store_true",
                      dest="skip_filter_overlap",
                      default = False,
                      help="""If set then it filters out the known fusion genes where the (i) genes are fully overlapping, or (ii) the genes are partially overlapping and are on the same strand. Default is '%default'.""")

    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    print "Generating the list of fusion genes from oesophagus tumors from TCGA..."
    fusions = dict()

    # manual curation from papers

    fusions['rattus_norvegicus'] = []

    fusions['mus_musculus'] = []

    fusions['canis_lupus_familiaris'] = []

    fusions['homo_sapiens'] = [
        ["AARS","CEP192"],
        ["AARSD1","GLIPR1"],
        ["AATF","INTS2"],
        ["ABHD2","VAPA"],
        ["ABI2","PPP1R1C"],
        ["ABPARTS","ABPARTS"],
        ["ACAA2","MYO5B"],
        ["ACACA","C17ORF78"],
        ["ACACA","LASP1"],
        ["ACBD3","TBC1D17"],
        ["ACER2","TOPORS"],
        ["ACER3","FCHSD2"],
        ["ACLY","NEUROD2"],
        ["ACLY","WEE1"],
        ["ACOT7","PLEKHG5"],
        ["ACOX1","TNRC6A"],
        ["ACP6","BCL9"],
        ["ACSF2","IKZF3"],
        ["ACSL1","FNTA"],
        ["ACTB","ACTG1"],
        ["ACTG1","DICER1"],
        ["ACTL8","ACOX1"],
        ["ACTN4","MAP4K1"],
        ["ACTR2","A2LD1"],
        ["ADAM12","HSPA12A"],
        ["ADAM9","SGK2"],
        ["ADA","RIMS4"],
        ["ADCY1","FOXP4"],
        ["ADCY2","ITCH"],
        ["ADCY7","BRD7"],
        ["ADIPOR2","CASK"],
        ["ADIPOR2","WNT5B"],
        ["ADNP","NFATC2"],
        ["AEBP1","POLD2"],
        ["AFF1","PTPN13"],
        ["AFF4","RPL34"],
        ["AFF4","SPATA21"],
        ["AFG3L2","PIEZO2"],
        ["AFG3L2","TUBB6"],
        ["AGAP1","C1ORF115"],
        ["AGAP1","WRB"],
        ["AGRN","ZC3H18"],
        ["AGTPBP1","GNAQ"],
        ["AHI1","SGK1"],
        ["AHI1","TMEM182"],
        ["AIFM2","LRRC20"],
        ["AK096803","MUC3A"],
        ["AK126266","AK096792"],
        ["AK311005","CTNNB1"],
        ["AK4","TMEM48"],
        ["AK8","C9ORF9"],
        ["AKT1","GPR132"],
        ["ALDH3A2","LYZ"],
        ["ALDH5A1","LYZ"],
        ["ALDOA","BCAR4"],
        ["ALG12","IL17REL"],
        ["ALG13","KDM6B"],
        ["ALYREF","CCDC57"],
        ["ANKFY1","CCDC88C"],
        ["ANKRD11","COPS6"],
        ["ANKRD11","FANCA"],
        ["ANKRD17","REST"],
        ["ANKRD34A","TXNIP"],
        ["ANKS1A","PACSIN1"],
        ["ANKS1B","UHRF1BP1L"],
        ["ANKS3","SEPT12"],
        ["ANLN","SEPT7"],
        ["ANXA13","COPA"],
        ["ANXA2","ANXA2"],
        ["ANXA2","DJ439574"],
        ["AP2A2","POLR2L"],
        ["AP2B1","GAS2L2"],
        ["APBB3","SLC35A4"],
        ["APLP2","OPCML"],
        ["APOO","MTERF"],
        ["ARAP1","IL12RB2"],
        ["ARF1","C1ORF35"],
        ["ARF6","IFI6"],
        ["ARHGAP17","CDC5L"],
        ["ARHGAP17","COL1A1"],
        ["ARHGAP30","PFDN2"],
        ["ARHGAP35","KLF13"],
        ["ARHGEF10L","PADI6"],
        ["ARID3A","PRPF31"],
        ["ARID4B","IGF2BP2"],
        ["ARID4B","MYH11"],
        ["ARL5A","CACNB4"],
        ["ARL8A","ZMYND8"],
        ["ARRB1","FCHSD2"],
        ["ARRB1","GDPD5"],
        ["ASAP1","FGF12"],
        ["ASH1L","ACTB"],
        ["ASH2L","MLF1IP"],
        ["ASPSCR1","PDE6G"],
        ["ASPSCR1","SLC25A10"],
        ["ASXL1","C20ORF160"],
        ["ASXL1","TPX2"],
        ["ASXL2","HADHA"],
        ["ATAD1","PTEN"],
        ["ATAD5","IKZF3"],
        ["ATF5","SIGLEC11"],
        ["ATF7","GLI3"],
        ["ATG4A","VSIG1"],
        ["ATN1","NOTCH4"],
        ["ATP1A1","ATP1A1OS"],
        ["ATP2A3","YAP1"],
        ["ATP2A3","ZZEF1"],
        ["ATP5E","FAM83D"],
        ["ATP5J","IDNK"],
        ["ATP6V0A1","STAT3"],
        ["ATP6V0E1","ERGIC1"],
        ["ATP6V1D","ZFYVE26"],
        ["ATP7B","KLF5"],
        ["ATP8A1","SHISA3"],
        ["ATP9B","ZNF521"],
        ["ATXN2","ARRB1"],
        ["ATXN7","SYNPR"],
        ["AUTS2","IL1RAPL1"],
        ["AVEN","ANO1"],
        ["AX746903","C11ORF87"],
        ["AXIN1","TMEM8A"],
        ["AZI1","NPLOC4"],
        ["AZIN1","YJEFN3"],
        ["B4GALT1","MRPS18A"],
        ["B9D2","REEP6"],
        ["BAIAP2L1","ANKIB1"],
        ["BAIAP2L2","TMEM184B"],
        ["BANP","SNX29P1"],
        ["BARD1","C6ORF222"],
        ["BAX","DHDH"],
        ["BAZ2A","PTPRF"],
        ["BAZ2B","WDR96"],
        ["BC014606","AHI1"],
        ["BC035172","RFXANK"],
        ["BC037918","MMP8"],
        ["BC041434","NIPAL1"],
        ["BC047605","CHN1"],
        ["BC062794","TBCD"],
        ["BC071809","WDR62"],
        ["BC073928","PKIG"],
        ["BC142949","GPR37"],
        ["BCAS3","SCN4A"],
        ["BCAT1","C12ORF77"],
        ["BCDIN3D","LYZ"],
        ["BCKDHA","FKBP3"],
        ["BCKDHA","REEP6"],
        ["BCL11A","PTPLA"],
        ["BCL2L13","BID"],
        ["BCR","CHD7"],
        ["BECN1","AP2B1"],
        ["BIRC5","AFMID"],
        ["BRCA1","CXCL14"],
        ["BRD9","CDK12"],
        ["BRD9","STARD3"],
        ["BRD9","TERT"],
        ["BRI3BP","UBC"],
        ["BRIX1","DNAJC21"],
        ["BTBD9","MANEA"],
        ["BTG3","C21ORF91"],
        ["BTN3A3","BTN3A2"],
        ["BTNL3","BTNL8"],
        ["BV13S6J2.1","TCRBV2S1"],
        ["BV13S6J2.1","TCRBV3S1"],
        ["BX538254","OSBPL6"],
        ["C10ORF76","LDB1"],
        ["C11ORF48","FN1"],
        ["C11ORF52","CRYAB"],
        ["C11ORF67","TMEM135"],
        ["C11ORF74","RAG1"],
        ["C11ORF85","CREB3"],
        ["C11ORF9","PIGN"],
        ["C12ORF51","RPL6"],
        ["C12ORF66","BC031864"],
        ["C14ORF159","CDC42BPB"],
        ["C14ORF178","MAP1LC3A"],
        ["C15ORF57","CBX3"],
        ["C16ORF13","RPL23"],
        ["C16ORF59","CCNF"],
        ["C16ORF70","DNAAF1"],
        ["C18ORF1","ANKRD12"],
        ["C18ORF25","ZFPM2"],
        ["C19ORF42","EIF5B"],
        ["C19ORF55","MARCH2"],
        ["C19ORF70","SCAF11"],
        ["C1ORF159","ZNF286A"],
        ["C1ORF194","UQCR10"],
        ["C1ORF85","C1ORF61"],
        ["C20ORF132","AFAP1L1"],
        ["C2ORF69","NDUFB5"],
        ["C3ORF26","COL8A1"],
        ["C3ORF78","NISCH"],
        ["C3","ZNF556"],
        ["C4ORF3","LYZ"],
        ["C5ORF42","NIPBL"],
        ["C5ORF45","NDUFB7"],
        ["C6ORF132","LYZ"],
        ["C6ORF72","LYZ"],
        ["C7ORF10","ARMC10"],
        ["C7ORF50","AK024243"],
        ["C7ORF50","MAD1L1"],
        ["C7ORF58","WNT16"],
        ["C8ORF76","ATAD2"],
        ["C9ORF3","HIATL1"],
        ["C9ORF78","LYZ"],
        ["CACNA1D","GNL3"],
        ["CADM4","SMG9"],
        ["CALD1","GNG12"],
        ["CAMK1D","ECHDC3"],
        ["CAMK2G","LYZ"],
        ["CAP1","PPT1"],
        ["CAPG","GGCX"],
        ["CAPN1","GTF2IRD1"],
        ["CAPNS1","FRA10AC1"],
        ["CAPZA2","STYXL1"],
        ["CARD8","MYH14"],
        ["CARM1","PLA2G4C"],
        ["CARS","OSBPL5"],
        ["CASP7","TRUB1"],
        ["CASP7","UQCRB"],
        ["CBFB","NAE1"],
        ["CBL","HINFP"],
        ["CBL","PHLDB1"],
        ["CBL","TRAPPC4"],
        ["CCDC117","HSCB"],
        ["CCDC53","NUP37"],
        ["CCDC64B","THOC6"],
        ["CCHCR1","SPINT2"],
        ["CCL25","FBN3"],
        ["CCM2","SLC38A10"],
        ["CCNB2","FAM63B"],
        ["CCND1","ORAOV1"],
        ["CCND1","YAP1"],
        ["CCT4","BIRC6"],
        ["CCT5","TMSB10"],
        ["CD109","IFT80"],
        ["CD24","CEACAM5"],
        ["CD24","DSG2"],
        ["CD2AP","FAM69A"],
        ["CD2AP","GGA2"],
        ["CD2AP","USP49"],
        ["CD44","PDHX"],
        ["CD68","LYZ"],
        ["CD82","LYZ"],
        ["CD93","SYNPO"],
        ["CDC27","CCT6B"],
        ["CDH3","C16ORF70"],
        ["CDH3","EIF4E"],
        ["CDIPT","SEZ6L2"],
        ["CDK12","RHCG"],
        ["CDK8","ATP8A2"],
        ["CDKAL1","ACTB"],
        ["CDKL1","GK5"],
        ["CDON","KIAA1377"],
        ["CDR2L","C17ORF28"],
        ["CEACAM5","CEACAM1"],
        ["CEACAM5","LYZ"],
        ["CEACAM5","PIGR"],
        ["CELSR2","FBXO46"],
        ["CEP192","PIGU"],
        ["CEP72","EXOC3"],
        ["CEP89","GTF2E2"],
        ["CFLAR","TPM3"],
        ["CHD7","KAT6A"],
        ["CHD7","RAB2A"],
        ["CHMP1A","FANCA"],
        ["CHST12","EIF3B"],
        ["CIAO1","LYZ"],
        ["CKMT1A","CKMT1B"],
        ["CLASRP","ZNF317"],
        ["CLDN18","LYZ"],
        ["CLDN4","TPT1"],
        ["CLIC4","MUM1"],
        ["CLK2P","CLK2"],
        ["CLTC","USP32"],
        ["CLU","ATPAF1"],
        ["CMPK1","TAL1"],
        ["CMTM4","DYNC1LI2"],
        ["CNBP","UNC93B1"],
        ["CNOT10","CMTM7"],
        ["CNOT4","EYS"],
        ["CNP","DNAJC7"],
        ["CNP","TTC25"],
        ["CNTN6","CNTN4"],
        ["COBRA1","FBXW5"],
        ["COBRA1","NDOR1"],
        ["COG3","GTF2F2"],
        ["COG3","TPT1-AS1"],
        ["COG5","LRGUK"],
        ["COG5","NRCAM"],
        ["COL27A1","ZNF618"],
        ["COL7A1","CELSR3"],
        ["CORO2A","NCBP1"],
        ["CORO7-PAM16","RMI2"],
        ["CPD","CCDC57"],
        ["CPEB4","SHISA5"],
        ["CPNE8","MGST1"],
        ["CPSF1","PABPC1"],
        ["CPSF6","CREBZF"],
        ["CPSF6","RBM48"],
        ["CPT1A","PPFIA1"],
        ["CPT1A","RNFT1"],
        ["CRAMP1L","IFT140"],
        ["CREBBP","ADCY9"],
        ["CRK","LYZ"],
        ["CSNK1A1","AK304483"],
        ["CSNK1A1","PDE6A"],
        ["CSNK1E","CISD1"],
        ["CSNK1E","CLCF1"],
        ["CSNK1E","KLF6"],
        ["CSNK1G2","SCAMP4"],
        ["CTBP2","TLX1NB"],
        ["CTCF","HNRPDL"],
        ["CTCF","KIAA0895L"],
        ["CTDSP2","MAGEB16"],
        ["CTDSPL","ESRRA"],
        ["CTNNA1","EGR1"],
        ["CTNNBIP1","C1ORF187"],
        ["CTNND1","CLPB"],
        ["CTSB","LYZ"],
        ["CTSB","PIGR"],
        ["CTSS","LYZ"],
        ["CTTN","KIRREL3"],
        ["CTTN","SHANK2"],
        ["CTTN","Y00482"],
        ["CUX1","EMID2"],
        ["CWC25","C17ORF98"],
        ["CWC25","LPCAT1"],
        ["CXORF38","FADS1"],
        ["CYB561D2","SLC38A3"],
        ["CYP51A1","AL133568"],
        ["CYP51A1","ANKIB1"],
        ["CYTH1","KCNJ4"],
        ["D2HGDH","MLPH"],
        ["DAB1","OMA1"],
        ["DAB2IP","TPM2"],
        ["DACH1","KLF12"],
        ["DAGLB","ZNF12"],
        ["DANCR","PQLC2"],
        ["DAP3","MAP6D1"],
        ["DAZAP1","ZNF226"],
        ["DCAF10","LYZ"],
        ["DCAF7","GNG12"],
        ["DCAF7","LYZ"],
        ["DCPS","CCND1"],
        ["DCPS","YAP1"],
        ["DD181999","COL1A1"],
        ["DDAH2","SLC44A4"],
        ["DDB1","CPSF7"],
        ["DDOST","LYZ"],
        ["DDOST","PIGR"],
        ["DDOST","TAPBP"],
        ["DDX49","COPE"],
        ["DDX50","STOX1"],
        ["DDX52","IKZF3"],
        ["DDX6","FGD6"],
        ["DECR1","STAU2"],
        ["DENND4B","BCL11A"],
        ["DEPDC1B","ELOVL7"],
        ["DES","PRKAG1"],
        ["DGAT1","TONSL"],
        ["DGKD","SAG"],
        ["DHX30","SMARCC1"],
        ["DHX40","GH1"],
        ["DIO3AS","DIO3"],
        ["DIP2A","LYZ"],
        ["DIP2B","TMEM185A"],
        ["DIP2C","CCDC3"],
        ["DJ439557","ANXA2"],
        ["DJ439583","ANXA2"],
        ["DKFZP434K1323","TP63"],
        ["DLG1","BDH1"],
        ["DLX6","IRF2BPL"],
        ["DMPK","SLPI"],
        ["DMTF1","C7ORF23"],
        ["DNAJB1","TECR"],
        ["DNAJB1","TRPM4"],
        ["DNAJC14","WIBG"],
        ["DNAJC22","LYZ"],
        ["DNAJC2","CUX1"],
        ["DNAJC6","RPN2"],
        ["DNMT1","AF161365"],
        ["DOCK5","ATCAY"],
        ["DOCK5","MATR3"],
        ["DOCK6","C3P1"],
        ["DOCK6","NFIX"],
        ["DOPEY2","LYZ"],
        ["DPYD","AK094607"],
        ["DQ658414","PTTG1"],
        ["DRAM1","SYCP3"],
        ["DRG1","LIMK2"],
        ["DSC2","TAF4B"],
        ["DSE","RPS5"],
        ["DSG2","CD24"],
        ["DSG2","FANCF"],
        ["DSG2","LYZ"],
        ["DSG2","PIGR"],
        ["DSG3","CD24"],
        ["DSG3","DSC3"],
        ["DSG3","LYZ"],
        ["DSP","LY86"],
        ["DTNB","AK058136"],
        ["DTNB","ASXL2"],
        ["DTX4","FAM111B"],
        ["DVL1","CCNL2"],
        ["DVL3","RFX6"],
        ["DYNC1LI2","NAE1"],
        ["DYNLL1","CASP4"],
        ["DYRK1A","DENND3"],
        ["E2F4","RPL14"],
        ["EAF2","RHOBTB3"],
        ["ECE2","ALG3"],
        ["ECT2","FNDC3B"],
        ["EDARADD","ENO1"],
        ["EEA1","LRRIQ1"],
        ["EEA1","UBE2N"],
        ["EFHA1","CD101"],
        ["EFHD2","MTHFR"],
        ["EHF","BDNF"],
        ["EIF2AK4","TMOD3"],
        ["EIF2S1","GPHN"],
        ["EIF3L","TRIOBP"],
        ["EIF4G3","CCNK"],
        ["ELF2","MGARP"],
        ["ELL","MKNK2"],
        ["EMID2","GTF2IRD1"],
        ["EML4","EPHA3"],
        ["ENO1","EDARADD"],
        ["ENO1","SLC2A5"],
        ["EPB41","POLR2L"],
        ["EPC1","AKR1E2"],
        ["EPC1","ARHGAP28"],
        ["EPC1","L3MBTL4"],
        ["EPC2","MBD5"],
        ["EPHB2","LCN2"],
        ["EPHB2","ZBTB40"],
        ["EPN3","AGPAT3"],
        ["EPSTI1","DNAJC15"],
        ["ERBB2IP","PPWD1"],
        ["ERBB2","JUP"],
        ["ERBB2","KRT10"],
        ["ERBB2","MIEN1"],
        ["ERBB2","PGAP3"],
        ["ERBB2","PSMB3"],
        ["ERC1","WNK1"],
        ["ERI2","DCUN1D3"],
        ["ERLIN2","LYZ"],
        ["ERN1","MYCBPAP"],
        ["ETHE1","PLAUR"],
        ["ETV4","TCAP"],
        ["EVI5L","PTPRS"],
        ["EWSR1","KREMEN1"],
        ["EXT1","TSNARE1"],
        ["EZR","PPP1R1B"],
        ["F2R","LYZ"],
        ["FA2H","KARS"],
        ["FADS2","RPLP0P2"],
        ["FAF1","TESK2"],
        ["FAM102A","C11ORF30"],
        ["FAM102A","WDR93"],
        ["FAM118A","UPK3A"],
        ["FAM126B","ORC2"],
        ["FAM127A","FAM127B"],
        ["FAM129B","IDNK"],
        ["FAM134C","AP2B1"],
        ["FAM134C","NOS2"],
        ["FAM135A","LAMA2"],
        ["FAM188B","HIBADH"],
        ["FAM188B","ZNRF2"],
        ["FAM190A","MMRN1"],
        ["FAM193A","PDGFRA"],
        ["FAM200A","CYP3A7"],
        ["FAM213A","LYZ"],
        ["FAM222B","MYO18A"],
        ["FAM222B","PHF12"],
        ["FAM35A","SEZ6L2"],
        ["FAM53B","CTBP2"],
        ["FAM59A","ACTN4"],
        ["FAM73A","LYZ"],
        ["FAM83G","B9D1"],
        ["FAM98A","RAD21"],
        ["FBLIM1","LYZ"],
        ["FBL","ZNF829"],
        ["FBXL13","LRRC17"],
        ["FBXL18","PHF12"],
        ["FBXL20","CDK12"],
        ["FBXO11","NET1"],
        ["FBXO32","TMEM120B"],
        ["FBXO6","LYZ"],
        ["FBXW11","DPCR1"],
        ["FBXW5","ENTPD2"],
        ["FGFR3","TACC3"],
        ["FKBP10","FAM134C"],
        ["FLJ39080","PAG1"],
        ["FLNB","SLMAP"],
        ["FNDC3B","TNIK"],
        ["FOXJ3","C1ORF141"],
        ["FOXJ3","VSTM4"],
        ["FOXN1","AMZ2"],
        ["FOXN3","C14ORF102"],
        ["FOXO1","FNDC3A"],
        ["FOXP4","UBR2"],
        ["FOXRED1","KRT6A"],
        ["FPGS","ENG"],
        ["FRG1B","RLF"],
        ["FTL","GYS1"],
        ["FTO","ZNF423"],
        ["FTSJD2","CCDC167"],
        ["FURIN","CCNI"],
        ["FUS","COPS6"],
        ["FUS","IL9R"],
        ["FUT10","SH2D4A"],
        ["FUT8","CHURC1-FNTB"],
        ["GAB1","PPP2R2C"],
        ["GADD45A","BCL11A"],
        ["GALM","LYZ"],
        ["GALNT11","ABCB8"],
        ["GALNT11","GALNTL5"],
        ["GAPVD1","RABEPK"],
        ["GATA4","TRIB1"],
        ["GATAD1","LYZ"],
        ["GATAD1","PAFAH1B2"],
        ["GATAD2A","CHST1"],
        ["GATAD2A","ZNF350"],
        ["GDE1","TMC5"],
        ["GGA1","CDC42EP1"],
        ["GIPC1","TMEM185A"],
        ["GIPR","RBM42"],
        ["GIT1","CSDA"],
        ["GLCE","NOX5"],
        ["GLG1","YWHAB"],
        ["GLI3","ADAMTSL3"],
        ["GLRX3","SGMS1"],
        ["GLT8D1","JMJD6"],
        ["GLTP","CD24"],
        ["GMDS","WASF3"],
        ["GNAI2","SLC38A3"],
        ["GNAI3","CYB561D1"],
        ["GNAQ","FBP1"],
        ["GNB1","AK054708"],
        ["GNB1","CEP68"],
        ["GNB1","KIAA1751"],
        ["GNB1","TXN"],
        ["GNPNAT1","LYZ"],
        ["GOLGB1","IQCB1"],
        ["GOLPH3","IKZF3"],
        ["GOSR1","CPD"],
        ["GP2","LYZ"],
        ["GPANK1","UBE4B"],
        ["GPATCH1","TAF3"],
        ["GPATCH8","SUPT6H"],
        ["GPHN","C16ORF57"],
        ["GPR107","PMM2"],
        ["GPR111","TNFRSF21"],
        ["GPR115","CD2AP"],
        ["GPR137","BAD"],
        ["GPRC5A","APOLD1"],
        ["GRAMD4","TNFAIP8L1"],
        ["GRB2","LYZ"],
        ["GRB2","PADI1"],
        ["GRB2","UNC13D"],
        ["GRB7","MIEN1"],
        ["GRB7","PGAP3"],
        ["GRB7","PIP4K2B"],
        ["GRHL2","NACAP1"],
        ["GRK1","ATP4B"],
        ["GRN","SLC25A39"],
        ["GSDMB","AATF"],
        ["GSR","CASC4"],
        ["GSTP1","C11ORF72"],
        ["GTF2H5","LYZ"],
        ["GTF2I","GTF2IRD1"],
        ["GTF3C3","COL5A2"],
        ["GTPBP10","CCDC132"],
        ["GYS1","GALP"],
        ["H1F0","COPG2"],
        ["H3F3B","H3F3C"],
        ["HCFC1R1","PTMS"],
        ["HDDC2","MEIS2"],
        ["HDLBP","NDFIP1"],
        ["HEATR5A","RPS10-NUDT3"],
        ["HELZ","SKAP1"],
        ["HERPUD2","AK057379"],
        ["HIAT1","STRN"],
        ["HIF1A","PRKCH"],
        ["HIF1A","RAD51B"],
        ["HINT2","ABCE1"],
        ["HINT2","ANKRD12"],
        ["HINT3","LYZ"],
        ["HIPK2","CPSF4"],
        ["HIST1H2AE","CLVS2"],
        ["HK1","VPS26A"],
        ["HK2","ZNF121"],
        ["HKR1","CEACAM7"],
        ["HLA-E","TAPBP"],
        ["HMG20B","CXCL14"],
        ["HMOX2","NMRAL1"],
        ["HNF1B","ARHGAP40"],
        ["HNF4G","TERF2"],
        ["HNRNPK","UHMK1"],
        ["HNRNPUL2","ABCC1"],
        ["HOMER2","TP53BP1"],
        ["HOXA10","CSN1S1"],
        ["HOXD10","WIPF1"],
        ["HOXD11","CIR1"],
        ["HPSE2","TIMM23"],
        ["HRSP12","LINC00511"],
        ["HSP90AB1","CEACAM5"],
        ["HSP90AB1","HSP90AB3P"],
        ["HSP90AB1","MKL2"],
        ["HSP90AB1","NFKBIE"],
        ["HSP90AB1","PRKAR1A"],
        ["HSP90AB1","RPLP2"],
        ["HSP90AB1","SLC35B2"],
        ["HSP90AB3P","HSP90AB1"],
        ["HSP90B1","C12ORF73"],
        ["HSPG2","HEBP1"],
        ["HUS1","VOPP1"],
        ["ICAM1","MRPL4"],
        ["IDS","CD24"],
        ["IFFO2","UBR4"],
        ["IFITM2","IFITM3"],
        ["IFNAR1","LYZ"],
        ["IFT80","TNIK"],
        ["IGFBP4","TNS4"],
        ["IGHMBP2","CPT1A"],
        ["IKBKAP","PTPN3"],
        ["IL17RD","LYZ"],
        ["IL18","TAPBP"],
        ["IL4I1","ATF5"],
        ["IL4I1","DYRK1B"],
        ["IMMP2L","DOCK4"],
        ["IMPA2","GNAL"],
        ["IMPACT","PTEN"],
        ["IMPDH2","TDRD3"],
        ["INADL","BC043254"],
        ["INPP5D","C2ORF82"],
        ["INTS1","CTAGE5"],
        ["INTS4","NDUFC2-KCTD14"],
        ["IP6K1","FOXRED2"],
        ["IP6K1","UBA7"],
        ["IPO7","SOX6"],
        ["IQCE","ACTN4"],
        ["IQGAP1","NGRN"],
        ["IQGAP1","SEMA4B"],
        ["IQGAP1","ZNF774"],
        ["IQGAP3","ARHGEF11"],
        ["IRAK1","LYZ"],
        ["IRAK2","BRK1"],
        ["IRF2BPL","ZSWIM6"],
        ["IRF7","RCC1"],
        ["IST1","ATXN1L"],
        ["ITGA6","MYEOV"],
        ["ITGA6","PDK1"],
        ["ITGA7","RNF41"],
        ["ITGB4","GALK1"],
        ["ITGB6","RBMS1"],
        ["ITPK1","BTBD7"],
        ["ITPR2","KIAA0922"],
        ["ITPR2","LRMP"],
        ["JMJD6","CD300LF"],
        ["JMJD6","GLT8D1"],
        ["JOSD1","THUMPD3"],
        ["JUP","C17ORF80"],
        ["JUP","KLHL11"],
        ["JUP","LEPREL4"],
        ["JUP","TTLL6"],
        ["KANSL1","KRT20"],
        ["KANSL2","DDX23"],
        ["KAT2B","VMA21"],
        ["KCNQ1","PRKCD"],
        ["KCTD20","LYZ"],
        ["KDM2A","ARAP1"],
        ["KDM4A","MPL"],
        ["KDM5A","NINJ2"],
        ["KHDC1","SLC17A5"],
        ["KIAA0100","ERBB2"],
        ["KIAA0100","PITPNC1"],
        ["KIAA0146","C3ORF39"],
        ["KIAA0247","SLC39A9"],
        ["KIAA0284","C1ORF135"],
        ["KIAA0319L","LYZ"],
        ["KIAA0368","C9ORF84"],
        ["KIAA1109","FSTL5"],
        ["KIAA1191","LYZ"],
        ["KIAA1432","LARS"],
        ["KIAA1644","CYP19A1"],
        ["KIF13B","C20ORF194"],
        ["KIF16B","C18ORF34"],
        ["KIF18B","C17ORF104"],
        ["KIF22","PRKCB"],
        ["KIF23","PAQR5"],
        ["KIF5B","EPC1"],
        ["KIF5C","STK39"],
        ["KLC4","MRPL2"],
        ["KLF5","PIBF1"],
        ["KLF9","FOXRED2"],
        ["KLHDC4","SLC7A5"],
        ["KLHL11","ASIC2"],
        ["KLK5","CLPTM1"],
        ["KREMEN1","ZNRF3"],
        ["KRIT1","CDK6"],
        ["KRT13","JUP"],
        ["KRT13","KRT15"],
        ["KRT13","S100A8"],
        ["KRT16","DSG1"],
        ["KRT16","KRT17"],
        ["KRT38","ACLY"],
        ["KRT4","KRT13"],
        ["KRT4","KRT79"],
        ["KRT5","MSN"],
        ["KRT6A","KRT5"],
        ["KRT6A","SPRR1B"],
        ["KRT6B","KRT6A"],
        ["KRT6C","KRT5"],
        ["KRT6C","KRT6A"],
        ["LAMA3","APOBEC1"],
        ["LAMA3","CTIF"],
        ["LAMC2","NMNAT2"],
        ["LAMP1","KRT5"],
        ["LANCL2","C20ORF11"],
        ["LARP4","AK055974"],
        ["LARS","KIAA1432"],
        ["LDLRAD3","PVRL2"],
        ["LDLR","LYZ"],
        ["LDLR","METTL2B"],
        ["LEPREL4","LYZ"],
        ["LEPR","MEF2D"],
        ["LETM1","LDLR"],
        ["LETM1","RGS12"],
        ["LFNG","MLEC"],
        ["LGALS3","DLGAP5"],
        ["LGALS3","FTL"],
        ["LGALS3","MAPK1IP1L"],
        ["LIFR","WDR70"],
        ["LILRB5","LILRB1"],
        ["LIMS1","LYZ"],
        ["LINC00483","SPAG9"],
        ["LIPF","BLNK"],
        ["LIPF","PGC"],
        ["LLGL2","FAM222B"],
        ["LLGL2","KIAA0195"],
        ["LLPH","LYZ"],
        ["LONP1","FUT6"],
        ["LPCAT4","SPRED1"],
        ["LPGAT1","INTS7"],
        ["LPHN1","LGALS4"],
        ["LPP","CLDN1"],
        ["LRFN3","FXYD7"],
        ["LRIG2","ZNF250"],
        ["LRRC37B","RAB11FIP4"],
        ["LSM14A","HIPK4"],
        ["LSR","NLRP2"],
        ["LTA4H","CDK17"],
        ["LYZ","ABHD2"],
        ["LYZ","AP1S3"],
        ["LYZ","BMP7"],
        ["LYZ","CD68"],
        ["LYZ","CEACAM5"],
        ["LYZ","CES2"],
        ["LYZ","CTSB"],
        ["LYZ","CYB561"],
        ["LYZ","DDOST"],
        ["LYZ","DSG2"],
        ["LYZ","DVL3"],
        ["LYZ","FAM210B"],
        ["LYZ","HLA-E"],
        ["LYZ","LEPREL4"],
        ["LYZ","LRP10"],
        ["LYZ","LSG1"],
        ["LYZ","MLLT6"],
        ["LYZ","PIGX"],
        ["LYZ","REG4"],
        ["LYZ","RNF141"],
        ["LYZ","SLC44A4"],
        ["LYZ","SPTLC2"],
        ["LYZ","TMEM45B"],
        ["LYZ","TPM4"],
        ["LYZ","UBN2"],
        ["LYZ","ZNF431"],
        ["MACROD1","BAD"],
        ["MAFF","RCOR1"],
        ["MAFK","KCTD5"],
        ["MAGI3","RGS18"],
        ["MAGOHB","CSDA"],
        ["MAN1C1","TTC40"],
        ["MAN2B1","FBXW9"],
        ["MAN2C1","SIN3A"],
        ["MAP1S","CEACAM6"],
        ["MAP2K3","NFIC"],
        ["MAP2K4","DNAH9"],
        ["MAP2K4","PRR11"],
        ["MAP2K7","CACNA1A"],
        ["MAP3K3","DBF4B"],
        ["MAP3K5","IFNGR1"],
        ["MAP4","CDC25A"],
        ["MAP7","AK124173"],
        ["MAP7","NRG3"],
        ["MAPRE3","RAB10"],
        ["MARK3","RPS6KA5"],
        ["MARVELD3","APOL1"],
        ["MAU2","ZNF682"],
        ["MAVS","HSPA12B"],
        ["MBD4","TMCC1"],
        ["MBOAT1","TPMT"],
        ["MBTD1","THRA"],
        ["MBTPS1","TAF1C"],
        ["MCAM","LYZ"],
        ["MCL1","VPS72"],
        ["MCM4","LYZ"],
        ["MECOM","ACTRT3"],
        ["MED12L","DPYD"],
        ["MED13","MYO15B"],
        ["MED1","PGAP3"],
        ["MED1","STARD3"],
        ["MED24","NR1D1"],
        ["MED29","LYZ"],
        ["MEGF6","WRAP73"],
        ["MEX3D","TCF3"],
        ["MGEA5","NFIC"],
        ["MGST3","ALDH9A1"],
        ["MICAL2","C12ORF70"],
        ["MICALL2","CA3"],
        ["MIDN","C19ORF38"],
        ["MKNK2","AK127846"],
        ["MLL3","SND1"],
        ["MLL5","LHFPL3"],
        ["MLL","TTC36"],
        ["MLPH","MARCH2"],
        ["MLXIP","BCL7A"],
        ["MMD","HDAC5"],
        ["MPEG1","GABRE"],
        ["MPP5","GPHN"],
        ["MPP7","ZEB1"],
        ["MPRIP","CCDC144A"],
        ["MRPL15","ACY3"],
        ["MRPL1","FAM190A"],
        ["MRPL21","IGHMBP2"],
        ["MRPL48","PLEKHB1"],
        ["MRPS10","LYZ"],
        ["MRPS16","TTC18"],
        ["MRTO4","LYZ"],
        ["MSH2","EPCAM"],
        ["MSRB3","ARNTL2"],
        ["MST1R","SAPCD2"],
        ["MTDH","LYZ"],
        ["MUC1","GBA"],
        ["MXD3","FAM193B"],
        ["MYH14","GPR19"],
        ["MYL6","SMARCC2"],
        ["MYO15B","TSEN54"],
        ["MZF1","DKFZP434J0226"],
        ["MZT2A","SMPD4"],
        ["NAALADL2","NAALADL2"],
        ["NADKD1","RANBP3L"],
        ["NADSYN1","DHCR7"],
        ["NADSYN1","SPAG9"],
        ["NBL1","MINOS1"],
        ["NCAPD2","MRPL51"],
        ["NCK2","FHL2"],
        ["NCK2","PRDX3"],
        ["NCOA6","DEFB118"],
        ["NCOA6","TRPC4AP"],
        ["NCOA7","NKAIN2"],
        ["NCOA7","RNF217"],
        ["NCOR2","SCARB1"],
        ["NCOR2","UBC"],
        ["NDE1","C16ORF45"],
        ["NDRG1","C9"],
        ["NDUFA4","PHF14"],
        ["NDUFS2","ADAMTS4"],
        ["NDUFS3","FAM32A"],
        ["NDUFS6","SLC9A3"],
        ["NEDD4","RFX7"],
        ["NELF","ZMYND19"],
        ["NEMF","SEPT7"],
        ["NEMF","SOS2"],
        ["NENF","PPP2R5A"],
        ["NFE2L3","C16ORF62"],
        ["NFE2L3","NPY"],
        ["NFIC","B3GAT2"],
        ["NFIC","ZFHX3"],
        ["NFIX","DOCK6"],
        ["NFIX","GCDH"],
        ["NFIX","RNASEH2A"],
        ["NFIX","TRMT1"],
        ["NFU1","AAK1"],
        ["NIN","ABHD12B"],
        ["NIPBL","C5ORF42"],
        ["NISCH","DNAH1"],
        ["NKAPP1","AK123976"],
        ["NLN","CTNNA1"],
        ["NLRC5","FTO"],
        ["NME1-NME2","CBX1"],
        ["NMT2","FAM171A1"],
        ["NOC2L","CCNL2"],
        ["NOL11","KIF19"],
        ["NOL6","SUGT1P1"],
        ["NOM1","RNF32"],
        ["NOS2","MYO18A"],
        ["NOTCH3","EPHX3"],
        ["NPAS2","PACSIN2"],
        ["NPC2","KIAA0317"],
        ["NPR2","BAIAP2"],
        ["NR1D2","UBE2E1"],
        ["NR2C2","PARL"],
        ["NR2F6","SFTPB"],
        ["NR2F6","USHBP1"],
        ["NRIP1","DNAJC15"],
        ["NRP2","KLF7"],
        ["NRSN2","C20ORF96"],
        ["NT5C2","RIPK4"],
        ["NUBPL","NPAS3"],
        ["NUCB1","NUDCD3"],
        ["NUMA1","IQSEC3"],
        ["NUP188","ENDOG"],
        ["NUP214","ABL1"],
        ["NUPL1","LYZ"],
        ["ODZ4","EXT2"],
        ["OLA1","CIR1"],
        ["OLA1","GPR155"],
        ["OPA3","VASP"],
        ["ORC2","C11ORF82"],
        ["ORC2","RPLP2"],
        ["OS9","AGAP2"],
        ["OS9","LYZ"],
        ["OSBP","GIF"],
        ["OSBPL3","ANO1"],
        ["OTUB1","COX8A"],
        ["OTUD4","NAA15"],
        ["PACSIN2","SCUBE1"],
        ["PANK4","TTC34"],
        ["PAPD7","SRD5A1"],
        ["PAPL","ACTN4"],
        ["PARD3B","MYO1B"],
        ["PARD3","NRXN1"],
        ["PARL","COL1A1"],
        ["PARP3","MUC6"],
        ["PASD1","ZFX"],
        ["PATZ1","EIF4ENIF1"],
        ["PAWR","PPP1R12A"],
        ["PBXIP1","HCST"],
        ["PCNP","RPS18"],
        ["PCSK9","USP24"],
        ["PCYOX1","LYZ"],
        ["PDCD4","RBM20"],
        ["PDE3B","SPON1"],
        ["PDE4D","DEPDC1B"],
        ["PDGFA","PRKAR1B"],
        ["PDIA3","MFAP1"],
        ["PDIA3P","PDIA3"],
        ["PDLIM2","SORBS3"],
        ["PDXK","KLF10"],
        ["PDZD2","KCNB2"],
        ["PEBP1","RPL18"],
        ["PGAP3","ERBB2"],
        ["PGC","EPS15L1"],
        ["PGC","LIPF"],
        ["PGC","MT1G"],
        ["PGC","MUC6"],
        ["PGC","PGA3"],
        ["PGM2L1","CHRDL2"],
        ["PGM2L1","LYZ"],
        ["PGS1","BPTF"],
        ["PHC3","FNDC3B"],
        ["PHC3","NCEH1"],
        ["PHF10","RBL1"],
        ["PHF21A","LEUTX"],
        ["PHF2","WNK2"],
        ["PHF2","WWOX"],
        ["PHLDB2","PLCXD2"],
        ["PHLPP2","ZNF821"],
        ["PI4KA","MAN1A2"],
        ["PI4KA","SNAP29"],
        ["PIEZO1","CBFA2T3"],
        ["PIGM","LYZ"],
        ["PIGR","CES2"],
        ["PIGZ","MFI2"],
        ["PIK3R2","BABAM1"],
        ["PILRB","ZKSCAN1"],
        ["PIN1","PDE4C"],
        ["PIP5K1B","PGM5"],
        ["PITPNM3","PELP1"],
        ["PKNOX1","NDUFV3"],
        ["PKP3","SIGIRR"],
        ["PLA2G6","CSNK1E"],
        ["PLCE1","LGI1"],
        ["PLEK2","GNL3L"],
        ["PLEK2","RDH11"],
        ["PLEKHA1","CYP2C19"],
        ["PLEKHA7","C6ORF221"],
        ["PLEKHA7","USH1C"],
        ["PLEKHG5","KHDRBS1"],
        ["PLEKHH1","MPP5"],
        ["PLIN3","LYZ"],
        ["PMPCB","DNAJC2"],
        ["PMS1","RAPGEF4"],
        ["POLB","XYLT1"],
        ["POLDIP2","PIGS"],
        ["POLR3B","ANO4"],
        ["PPAP2C","CDC34"],
        ["PPARD","ANKS1A"],
        ["PPDPF","CTSB"],
        ["PPDPF","PPIE"],
        ["PPDPF","UBL3"],
        ["PPFIA1","AK125463"],
        ["PPFIA1","XRRA1"],
        ["PPIB","CSNK1G1"],
        ["PPIP5K1","KRT6A"],
        ["PPM1G","IFT172"],
        ["PPM1H","DIP2B"],
        ["PPP1CA","TMEM134"],
        ["PPP1R12B","SYT2"],
        ["PPP1R13B","L2HGDH"],
        ["PPP1R13L","MBOAT7"],
        ["PPP1R14D","PPDPF"],
        ["PPP1R21","RAP2C"],
        ["PPP1R2","APOD"],
        ["PPP1R37","DAAM2"],
        ["PPP1R37","RTN2"],
        ["PPP1R9B","BAIAP2"],
        ["PPP3CB","USP54"],
        ["PPP4R1","NUP62"],
        ["PPP4R2","KIDINS220"],
        ["PPP5C","AP2S1"],
        ["PPP6R1","SGK110"],
        ["PPP6R3","ACTB"],
        ["PPT2","EGFL8"],
        ["PQLC1","ZNF407"],
        ["PRDX1","RFX5"],
        ["PRKCZ","IFLTD1"],
        ["PRKD2","KLHL26"],
        ["PRKG1","RNF13"],
        ["PROSC","TTLL10"],
        ["PRPF31","ZNF547"],
        ["PRPF3","VPS45"],
        ["PRRC2B","NUP214"],
        ["PRSS8","PRSS36"],
        ["PSD3","XPO7"],
        ["PSMA3","ACTR10"],
        ["PSMB3","PIP4K2B"],
        ["PSMB4","POGZ"],
        ["PSMD3","MSI2"],
        ["PSME4","ACYP2"],
        ["PSMG2","OSBPL1A"],
        ["PSPH","DNAJC3"],
        ["PTGES3","SPATS2"],
        ["PTK2","SLC45A4"],
        ["PTK6","LYZ"],
        ["PTP4A1","NOTCH3"],
        ["PTP4A2","XPO1"],
        ["PTPN12","PION"],
        ["PTPN12","RHBDD2"],
        ["PTPN2","L3MBTL4"],
        ["PTPN4","NCR2"],
        ["PTPN4","PCDP1"],
        ["PTPRH","HSPBP1"],
        ["PTPRK","ARHGAP18"],
        ["PTPRK","RSPO3"],
        ["PTRF","AIFM2"],
        ["PUM1","NKAIN1"],
        ["PUM1","YARS"],
        ["PVT1","ANXA13"],
        ["PVT1","FARP1"],
        ["PVT1","MYC"],
        ["PVT1","PCAT1"],
        ["PVT1","SNX31"],
        ["PVT1","TMEM67"],
        ["PVT1","TNFRSF11B"],
        ["PXN","PLA2G1B"],
        ["PXN","SRSF9"],
        ["PYGB","CDC20B"],
        ["QRSL1","MBOAT1"],
        ["QTRT1","SLC44A2"],
        ["R3HCC1","GRIK5"],
        ["R3HDM2","MYO1E"],
        ["R3HDM4","JAK3"],
        ["RAB11A","GOT1"],
        ["RAB11FIP1","CD24"],
        ["RAB11FIP1","LYZ"],
        ["RAB21","SYT10"],
        ["RAB25","LAMTOR2"],
        ["RAB6B","PPP2R3A"],
        ["RAB9A","EGFL6"],
        ["RABEP1","ZFP3"],
        ["RABGEF1","KCTD7"],
        ["RAC1","PITX1"],
        ["RAD54L","MAST2"],
        ["RAF1","C9ORF153"],
        ["RAI2","MYH11"],
        ["RALA","HOXA11-AS"],
        ["RALGAPB","CTNNBL1"],
        ["RALGAPB","KCNB1"],
        ["RALGAPB","SNX2"],
        ["RANGAP1","POLR3H"],
        ["RAP1GAP","HSPG2"],
        ["RAPGEF5","TOMM7"],
        ["RASSF3","TBK1"],
        ["RASSF8","CASC1"],
        ["RB1","ITM2B"],
        ["RBL1","C20ORF132"],
        ["RBM33","TMEM120A"],
        ["RBM34","BIRC5"],
        ["RBM39","C22ORF24"],
        ["RBM47","APBB2"],
        ["RBM47","KCTD8"],
        ["RBM47","RAPGEF2"],
        ["RCHY1","EPHA5"],
        ["RCOR1","ISCA2"],
        ["REEP3","RTKN2"],
        ["REEP4","PLTP"],
        ["REG1A","REG1B"],
        ["RELB","ZNF227"],
        ["RERE","ENO1"],
        ["RERE","GNB1"],
        ["RFX1","NPAS1"],
        ["RFX2","MLLT1"],
        ["RG9MTD1","CAMK4"],
        ["RG9MTD1","MGC34034"],
        ["RGNEF","BTF3"],
        ["RGS6","RGCC"],
        ["RHBDL2","MYCBP"],
        ["RHOT2","RAB11FIP3"],
        ["RHPN2","BRSK1"],
        ["RHPN2","CLASRP"],
        ["RICTOR","C9"],
        ["RIN3","IGHG1"],
        ["RIOK3","CD24"],
        ["RLIM","PABPC1"],
        ["RNF103","CD8A"],
        ["RNF111","TLN2"],
        ["RNF123","CHMP2B"],
        ["RNF141","LYZ"],
        ["RNF19A","F10"],
        ["RNF19A","LRP12"],
        ["RNF213","SLC26A11"],
        ["RNF217","NKAIN2"],
        ["RNF24","SNRPD3"],
        ["RNF44","WDR1"],
        ["RNF4","WHSC1"],
        ["RNF5","RNF5P1"],
        ["RNGTT","GABRR2"],
        ["RNPS1","C17ORF65"],
        ["ROCK1","CIDEA"],
        ["ROCK1","TMEM241"],
        ["ROCK2","E2F6"],
        ["RPH3AL","NXN"],
        ["RPL19","LAPTM4A"],
        ["RPL19","STAC2"],
        ["RPL23AP53","SNTG1"],
        ["RPL24","CNBD1"],
        ["RPL36","LONP1"],
        ["RPLP2","TNC"],
        ["RPS21","BC010030"],
        ["RPS5","DSE"],
        ["RPS6KB1","TAOK1"],
        ["RPS6KB1","TUBD1"],
        ["RPS6KB1","VMP1"],
        ["RRP7A","DCAF8"],
        ["RSAD2","SULF1"],
        ["RSL1D1","LYZ"],
        ["RUNDC1","LYZ"],
        ["RUNX1T1","KAT6A"],
        ["RUNX1T1","ZC2HC1A"],
        ["S100A13","MAP3K14"],
        ["S100A16","RPL14"],
        ["S100A2","MTMR2"],
        ["S100A4","AGR2"],
        ["S100A9","KRT4"],
        ["S100PBP","ZBTB8A"],
        ["SAFB","SAFB2"],
        ["SARS2","DEDD2"],
        ["SAT1","APOO"],
        ["SBF1","TSR2"],
        ["SBF2","ANO3"],
        ["SBF2","HBE1"],
        ["SCAF11","LYZ"],
        ["SCARB1","UBC"],
        ["SCD5","SEC31A"],
        ["SCML2","SCML1"],
        ["SCN2A","SCN3A"],
        ["SDHAP3","NELL1"],
        ["SEC13","COL1A1"],
        ["SEC23B","CSRP2BP"],
        ["SEC31A","FAM175A"],
        ["SELS","MED11"],
        ["SEMA3F","RNF123"],
        ["SEMA4C","ACTR2"],
        ["SEMA6D","RTN4"],
        ["SENP5","PAK2"],
        ["SEPT2","RHOH"],
        ["SEPT6","MPEG1"],
        ["SEPT9","BAIAP2"],
        ["SERGEF","SPTY2D1"],
        ["SF3B3","DDX19B"],
        ["SFMBT1","TTLL3"],
        ["SFN","SFN"],
        ["SGK1","AJ606331"],
        ["SGK1","BC040308"],
        ["SH3BP2","BC038204"],
        ["SH3PXD2A","OBFC1"],
        ["SHOC2","RBM20"],
        ["SHROOM3","FAM47E"],
        ["SIM2","MORC3"],
        ["SIPA1L2","ARID4B"],
        ["SIPA1L2","TTC13"],
        ["SIPA1L3","ODF3L2"],
        ["SKIL","LEPREL1"],
        ["SKIV2L","PRRC2A"],
        ["SLC12A2","FLJ33630"],
        ["SLC16A9","CCDC6"],
        ["SLC17A5","LYZ"],
        ["SLC20A2","VDAC3"],
        ["SLC25A11","RNF167"],
        ["SLC25A13","MGC72080"],
        ["SLC25A42","TEX101"],
        ["SLC26A11","CARD14"],
        ["SLC2A12","SAMD3"],
        ["SLC43A2","SCARF1"],
        ["SLC44A4","LYZ"],
        ["SLC45A2","AMACR"],
        ["SLC45A2","C1QTNF3"],
        ["SLC50A1","GDI1"],
        ["SLC50A1","LYZ"],
        ["SLC6A18","SLC6A19"],
        ["SLC6A8","BCAP31"],
        ["SLC7A7","ACIN1"],
        ["SLC7A8","ARCN1"],
        ["SLC9A2","SLC9A4"],
        ["SLMAP","FAF1"],
        ["SLTM","LPHN1"],
        ["SMAD2","SALL3"],
        ["SMAGP","GALNT6"],
        ["SMAGP","TBC1D30"],
        ["SMARCA4","DNM2"],
        ["SMARCA4","SLC44A2"],
        ["SMARCA5","COL17A1"],
        ["SMARCAD1","GRID2"],
        ["SMC3","MXI1"],
        ["SMCR7L","TMBIM6"],
        ["SMG1","ABCC1"],
        ["SMG1","LYZ"],
        ["SMG5","CCT3"],
        ["SMPD4","MZT2A"],
        ["SMURF1","ATP5J2-PTCD1"],
        ["SMURF1","CFLAR"],
        ["SMURF1","PDAP1"],
        ["SNAPIN","FMNL1"],
        ["SND1","MLL3"],
        ["SNHG8","PQLC2"],
        ["SNX27","PIP5K1A"],
        ["SNX30","KIAA1958"],
        ["SOCS6","CTDP1"],
        ["SP2","KPNB1"],
        ["SPAG9","TRIM37"],
        ["SPOP","IKZF3"],
        ["SPOP","ITGB3"],
        ["SPOP","TSPEAR"],
        ["SPPL2B","DOT1L"],
        ["SPPL3","IFNAR2"],
        ["SPRED1","C15ORF39"],
        ["SPRR1B","SPRR1A"],
        ["SPRR2A","SPRR2G"],
        ["SPRR3","KRT13"],
        ["SPSB3","ECI1"],
        ["SPTBN2","PC"],
        ["SPTLC2","LYZ"],
        ["SREBF1","TAF15"],
        ["SRPK2","PUS7"],
        ["SRRM2","PRSS33"],
        ["SRRM2","TCEB2"],
        ["SSBP3","FAM193B"],
        ["SSH2","SLC39A11"],
        ["ST3GAL2","COG4"],
        ["ST3GAL3","PTPRF"],
        ["ST3GAL4","C11ORF70"],
        ["ST3GAL4","FGF19"],
        ["ST5","TMEM9B"],
        ["ST7","KCND2"],
        ["STAMBPL1","ACTA2"],
        ["STARD10","DAGLA"],
        ["STARD13","CAB39L"],
        ["STARD3","CDK12"],
        ["STARD3","FBXL20"],
        ["STARD3","IKZF3"],
        ["STARD3","RAB5C"],
        ["STAT3","CD24"],
        ["STAT3","LYZ"],
        ["STAT3","NOS2"],
        ["STAT5B","NSRP1"],
        ["STAT5B","STAT3"],
        ["STIP1","SYT12"],
        ["STK10","UBTD2"],
        ["STK17A","HECW1"],
        ["STK39","KCNH2"],
        ["STK3","COX6C"],
        ["STK40","PABPC1"],
        ["STK4","YWHAB"],
        ["STRADA","GRB2"],
        ["STRBP","DENND1A"],
        ["SULF2","ZFP64"],
        ["SUMO2","SLC16A5"],
        ["SUN1","AHNAK"],
        ["SUPT5H","PAPL"],
        ["SUPT5H","ZFP36"],
        ["SUPT5H","ZSCAN5A"],
        ["SUPT6H","GPATCH8"],
        ["SUPT6H","PROCA1"],
        ["SUV420H1","RNPS1"],
        ["SYNE2","GEMIN2"],
        ["SYNJ1","GCFC1"],
        ["SYNPO","COL1A1"],
        ["SYNRG","HNF1B"],
        ["SYTL2","PICALM"],
        ["TACC2","DOCK1"],
        ["TACC2","PPAPDC1A"],
        ["TADA2B","LYZ"],
        ["TAF7","KCNQ4"],
        ["TAGLN","PCSK7"],
        ["TALDO1","PDDC1"],
        ["TANC2","TSHZ2"],
        ["TAPBPL","CD9"],
        ["TASP1","SEL1L2"],
        ["TATDN2","BRK1"],
        ["TATDN2","FANCD2"],
        ["TAX1BP1","CHN2"],
        ["TAX1BP3","CTNS"],
        ["TBC1D14","MRFAP1"],
        ["TBC1D15","LGR5"],
        ["TBC1D15","PTPRR"],
        ["TBC1D15","RAB21"],
        ["TBC1D1","PGM2"],
        ["TBC1D22A","TRMU"],
        ["TBC1D23","C3ORF26"],
        ["TBCD","CCDC57"],
        ["TBK1","RBM48"],
        ["TBL1XR1","DCUN1D1"],
        ["TBL1XR1","MCF2L2"],
        ["TBRG1","LYZ"],
        ["TCEA3","DNAJC16"],
        ["TCEANC","MAP4K5"],
        ["TCOF1","PPARGC1B"],
        ["TCRB","TCRB"],
        ["TCRB","TCRBV3S1"],
        ["TCRBV21S1","TCRBV2S1"],
        ["TCRBV5S2","TCRBV2S1"],
        ["TCRBV5S2","TCRBV3S1"],
        ["TESK2","C8A"],
        ["TFDP2","XRN1"],
        ["TFG","GPR128"],
        ["TGOLN2","DSG3"],
        ["TGOLN2","LYZ"],
        ["THRAP3","ARHGEF10L"],
        ["THUMPD3","LYZ"],
        ["TIMM22","SGSM2"],
        ["TIMMDC1","DNAJC30"],
        ["TIMP2","ZBTB40"],
        ["TJP2","FXN"],
        ["TJP2","PIP5K1B"],
        ["TJP3","APBA3"],
        ["TKT","CENPF"],
        ["TKT","KLF7"],
        ["TLE3","YWHAH"],
        ["TLK1","KCNH7"],
        ["TMBIM4","LYZ"],
        ["TMC5","GPRC5B"],
        ["TMEM106C","RPAP3"],
        ["TMEM117","IRAK4"],
        ["TMEM11","C17ORF103"],
        ["TMEM129","LETM1"],
        ["TMEM144","DAPP1"],
        ["TMEM147","BC043213"],
        ["TMEM160","ZC3H4"],
        ["TMEM220","TMEM97"],
        ["TMEM241","SERPINB7"],
        ["TMEM241","TRAPPC8"],
        ["TMEM245","DSP"],
        ["TMEM245","LYZ"],
        ["TMEM30A","FILIP1"],
        ["TMEM38B","SLC44A1"],
        ["TMEM41A","LYZ"],
        ["TMEM60","HIP1"],
        ["TMEM67","TNFRSF11B"],
        ["TMLHE","DERL1"],
        ["TMSB4X","BCAS4"],
        ["TNC","C22ORF25"],
        ["TNFRSF10A","TNFRSF10C"],
        ["TNFRSF11A","KIAA1468"],
        ["TNFSF12-TNFSF13","FHDC1"],
        ["TNIP2","FAM53A"],
        ["TNRC18","ACTB"],
        ["TNRC6C","SEPT9"],
        ["TNS3","SOX18"],
        ["TNS4","WIPF2"],
        ["TOLLIP","MUC5B"],
        ["TONSL","GRB2"],
        ["TP53","ATP1B2"],
        ["TPD52L1","RNF217"],
        ["TPM2","ACTG1"],
        ["TPMT","BTBD9"],
        ["TPRA1","S100A2"],
        ["TPSAB1","TPSB2"],
        ["TPSB2","MAGED1"],
        ["TPT1","S100A8"],
        ["TPX2","RIF1"],
        ["TRABD","TCF7L2"],
        ["TRADD","KCTD19"],
        ["TRAPPC10","AGPAT3"],
        ["TRAPPC4","SLC37A4"],
        ["TRAPPC6A","STC1"],
        ["TRB@","TCRBV3S1"],
        ["TRB@","TCRBV2S1"],
        ["TRB@","TCRBV2S1"],
        ["TRB@","TCRBV3S1"],
        ["TRIM14","CORO2A"],
        ["TRIM26","DHX16"],
        ["TRIM44","PDHX"],
        ["TRIM65","STAT3"],
        ["TRIM66","EHF"],
        ["TRIP11","ABCA11P"],
        ["TRPC4AP","RBM39"],
        ["TRPC4AP","UQCC"],
        ["TRUB2","MAPKAP1"],
        ["TSEN54","KIAA0195"],
        ["TSHZ1","FUT2"],
        ["TSHZ2","SGK2"],
        ["TSPAN15","HK1"],
        ["TTC19","ZSWIM7"],
        ["TTC39C","GATA6"],
        ["TTC5","RNASE9"],
        ["TTC7A","SOCS5"],
        ["TTLL5","JDP2"],
        ["TUBA1C","IGFBP4"],
        ["TUBB3","TCF25"],
        ["TUBG2","TUBG1"],
        ["TXLNG","APPL2"],
        ["TXNDC11","ZC3H7A"],
        ["TXN","GNB1"],
        ["TYK2","RAB3D"],
        ["TYK2","TMEM205"],
        ["TYMP","SCO2"],
        ["TYMS","ANXA2"],
        ["TYMS","SEPT9"],
        ["TYW1","DMTF1"],
        ["U2AF2","KIR3DL2"],
        ["UBA2","MIER2"],
        ["UBAC1","SEC16A"],
        ["UBE2Q2","SNX33"],
        ["UBN1","MGRN1"],
        ["UBQLN1","HEXIM1"],
        ["UBR1","TGM5"],
        ["UBXN7","LYZ"],
        ["UCHL3","KLF5"],
        ["UCHL5","SLC6A17"],
        ["UHRF1BP1","SNRPC"],
        ["UNK","KIAA0195"],
        ["URI1","PDCD5"],
        ["USF2","ZNF30"],
        ["USMG5","TAF5"],
        ["USP15","PZP"],
        ["USP18","ADM2"],
        ["USP32","CDK12"],
        ["USP32","RBFOX3"],
        ["USP47","NPAT"],
        ["USP54","PIK3CD"],
        ["USP9X","PNPLA4"],
        ["UTP18","SPOP"],
        ["UVRAG","RPS3"],
        ["UVRAG","SLCO2B1"],
        ["VAPA","ABHD2"],
        ["VAPA","GNAL"],
        ["VAPA","ZNF718"],
        ["VCL","AP3M1"],
        ["VCPIP1","C8ORF44"],
        ["VEZF1","KRT222"],
        ["VPS13D","TNFRSF1B"],
        ["VPS33A","ABCB9"],
        ["VPS35","ENAH"],
        ["VPS53","LYZ"],
        ["VPS54","PELI1"],
        ["VRK3","AFG3L2"],
        ["WASF2","FGR"],
        ["WASF2","IFI6"],
        ["WASF2","OPRD1"],
        ["WDR45","PLEC"],
        ["WDR59","KARS"],
        ["WIF1","PAG1"],
        ["WIPF2","LYZ"],
        ["WNK2","SUSD3"],
        ["WWP2","CDH13"],
        ["WWP2","GSG1L"],
        ["XPO5","VARS"],
        ["XPR1","NCF2"],
        ["YAF2","GPHN"],
        ["YAP1","CDON"],
        ["YAP1","USP2"],
        ["YJEFN3","AZIN1"],
        ["YJEFN3","GATAD2A"],
        ["YME1L1","LRRC37A6P"],
        ["YTHDC1","GAK"],
        ["ZBTB17","ARHGEF19"],
        ["ZBTB43","LYZ"],
        ["ZC3H7A","BCAR4"],
        ["ZC3H7B","USP49"],
        ["ZC3HAV1","TTPAL"],
        ["ZCCHC11","ENY2"],
        ["ZCCHC8","RSRC2"],
        ["ZDHHC3","CDCP1"],
        ["ZDHHC5","CLP1"],
        ["ZEB1","THOC2"],
        ["ZFAND3","FTSJD2"],
        ["ZFP112","PLEKHA4"],
        ["ZFP36L2","UBOX5"],
        ["ZFP36","PPID"],
        ["ZFP3","PCCB"],
        ["ZKSCAN1","LYZ"],
        ["ZMYM2","LATS2"],
        ["ZNF169","IMMP1L"],
        ["ZNF207","C17ORF75"],
        ["ZNF207","TANC2"],
        ["ZNF212","HDLBP"],
        ["ZNF217","MAVS"],
        ["ZNF235","OR7G2"],
        ["ZNF295","TFF1"],
        ["ZNF318","FOXP4"],
        ["ZNF417","CNFN"],
        ["ZNF418","USP25"],
        ["ZNF420","DPF1"],
        ["ZNF429","RBMS3"],
        ["ZNF519","PKP4"],
        ["ZNF541","C19ORF68"],
        ["ZNF542","ANGPTL6"],
        ["ZNF562","HK2"],
        ["ZNF611","PIGX"],
        ["ZNF680","MIPOL1"],
        ["ZNF713","FMR1"],
        ["ZNF713","SEPT14"],
        ["ZNF718","GAK"],
        ["ZNF790","C19ORF45"],
        ["ZNF805","ZNF264"],
        ["ZNF808","PIGX"],
        ["ZNFX1","EIF2D"],
        ["ZNRF3","SLC5A1"],
        ["ZRANB1","LYRM7"]



]



    data = fusions.get(options.organism.lower(),[])
    if data:

        #file_symbols = os.path.join(options.output_directory,'genes_symbols.txt')
        file_symbols = os.path.join(options.output_directory,'synonyms.txt')
        loci = symbols.generate_loci(file_symbols)

        genes = symbols.read_genes_symbols(file_symbols)

        d = []
        for (g1,g2) in data:
            if g1.upper() != g2.upper():
                ens1 = symbols.ensembl(g1.upper(),genes,loci)
                ens2 = symbols.ensembl(g2.upper(),genes,loci)
                if ens1 and ens2:
                    for e1 in ens1:
                        for e2 in ens2:
                            if e1 != e2:
                                d.append([e1,e2])

        data = ['\t'.join(sorted(line)) + '\n' for line in d]
        data = list(set(data))

        print "%d known fusion genes found in manually currated database" % (len(data),)

        if not options.skip_filter_overlap:
            d1 = []
            overlappings = ['ensembl_fully_overlapping_genes.txt',
                            'ensembl_same_strand_overlapping_genes.txt',
#                            'refseq_fully_overlapping_genes.txt',
#                            'refseq_same_strand_overlapping_genes.txt',
#                            'ucsc_fully_overlapping_genes.txt',
#                            'ucsc_same_strand_overlapping_genes.txt',
#                            'pairs_pseudogenes.txt',
#                            'paralogs.txt'
                            ]
            for ov in overlappings:
                p = os.path.join(options.output_directory,ov)
                print "Parsing file:",p
                if os.path.isfile(p):
                    d2 = sorted(set([tuple(sorted(line.rstrip('\r\n').split('\t'))) for line in file(p,'r').readlines() if line.rstrip('\r\n')]))
                d1.extend(d2)
            d = set()
            for line in d1:
                (a,b) = (line[0],line[1])
                if a > b:
                    (a,b) = (b,a)
                d.add("%s\t%s\n" % (a,b))
            skipped = [line for line in data if line in d]
            data = [line for line in data if not line in d]
            file(os.path.join(options.output_directory,'oesophagus_known_but_overlapping.txt'),'w').writelines(sorted(skipped))

            print "%d known fusion genes left after removing the overlappings" % (len(data),)

    file(os.path.join(options.output_directory,'oesophagus.txt'),'w').writelines(sorted(data))
    #
