#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the sequence of the "Human ribosomal DNA complete repeating unit" which is hard coded inhere.



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the ribosomal DNA complete repeating unit."""
    version = "%prog 0.10 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the ribosomal DNA complete repeating unit is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the ribosomal DNA complete repeating unit is generated. Default is '%default'.""")


    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    seq = dict()

    seq['homo_sapiens'] = """>gi|555853|gb|U13369.1|HSU13369 Human ribosomal DNA complete repeating unit
GCTGACACGCTGTCCTCTGGCGACCTGTCGTCGGAGAGGTTGGGCCTCCGGATGCGCGCGGGGCTCTGGC
CTCACGGTGACCGGCTAGCCGGCCGCGCTCCTGCCTTGAGCCGCCTGCCGCGGCCCGCGGGCCTGCTGTT
CTCTCGCGCGTCCGAGCGTCCCGACTCCCGGTGCCGGCCCGGGTCCGGGTCTCTGACCCACCCGGGGGCG
GCGGGGAAGGCGGCGAGGGCCACCGTGCCCCGTGCGCTCTCCGCTGCGGGCGCCCGGGGCGCCGCACAAC
CCCACCCGCTGGCTCCGTGCCGTGCGTGTCAGGCGTTCTCGTCTCCGCGGGGTTGTCCGCCGCCCCTTCC
CCGGAGTGGGGGGTGGCCGGAGCCGATCGGCTCGCTGGCCGGCCGGCCTCCGCTCCCGGGGGGCTCTTCG
ATCGATGTGGTGACGTCGTGCTCTCCCGGGCCGGGTCCGAGCCGCGACGGGCGAGGGGCGGACGTTCGTG
GCGAACGGGACCGTCCTTCTCGCTCCGCCCGCGCGGTCCCCTCGTCTGCTCCTCTCCCCGCCCGCCGGCC
GGCGTGTGGGAAGGCGTGGGGTGCGGACCCCGGCCCGACCTCGCCGTCCCGCCCGCCGCCTTCGCTTCGC
GGGTGCGGGCCGGCGGGGTCCTCTGACGCGGCAGACAGCCCTGCCTGTCGCCTCCAGTGGTTGTCGACTT
GCGGGCGGCCCCCCTCCGCGGCGGTGGGGGTGCCGTCCCGCCGGCCCGTCGTGCTGCCCTCTCGGGGGGG
GTTTGCGCGAGCGTCGGCTCCGCCTGGGCCCTTGCGGTGCTCCTGGAGCGCTCCGGGTTGTCCCTCAGGT
GCCCGAGGCCGAACGGTGGTGTGTCGTTCCCGCCCCCGGCGCCCCCTCCTCCGGTCGCCGCCGCGGTGTC
CGCGCGTGGGTCCTGAGGGAGCTCGTCGGTGTGGGGTTCGAGGCGGTTTGAGTGAGACGAGACGAGACGC
GCCCCTCCCACGCGGGGAAGGGCGCCCGCCTGCTCTCGGTGAGCGCACGTCCCGTGCTCCCCTCTGGCGG
GTGCGCGCGGGCCGTGTGAGCGATCGCGGTGGGTTCGGGCCGGTGTGACGCGTGCGCCGGCCGGCCGCCG
AGGGGCTGCCGTTCTGCCTCCGACCGGTCGTGTGTGGGTTGACTTCGGAGGCGCTCTGCCTCGGAAGGAA
GGAGGTGGGTGGACGGGGGGGCCTGGTGGGGTTGCGCGCACGCGCGCACCGGCCGGGCCCCCGCCCTGAA
CGCGAACGCTCGAGGTGGCCGCGCGCAGGTGTTTCCTCGTACCGCAGGGCCCCCTCCCTTCCCCAGGCGT
CCCTCGGCGCCTCTGCGGGCCCGAGGAGGAGCGGCTGGCGGGTGGGGGGAGTGTGACCCACCCTCGGTGA
GAAAAGCCTTCTCTAGCGATCTGAGAGGCGTGCCTTGGGGGTACCGGATCCCCCGGGCCGCCGCCTCTGT
CTCTGCCTCCGTTATGGTAGCGCTGCCGTAGCGACCCGCTCGCAGAGGACCCTCCTCCGCTTCCCCCTCG
ACGGGGTTGGGGGGGAGAAGCGAGGGTTCCGCCGGCCACCGCGGTGGTGGCCGAGTGCGGCTCGTCGCCT
ACTGTGGCCCGCGCCTCCCCCTTCCGAGTCGGGGGAGGATCCCGCCGGGCCGGGCCCGGCGCTCCCACCC
AGCGGGTTGGGACGCGGCGGCCGGCGGGCGGTGGGTGTGCGCGCCCGGCGCTCTGTCCGGCGCGTGACCC
CCTCCGTCCGCGAGTCGGCTCTCCGCCCGCTCCCGTGCCGAGTCGTGACCGGTGCCGACGACCGCGTTTG
CGTGGCACGGGGTCGGGCCCGCCTGGCCCTGGGAAAGCGTCCCACGGTGGGGGCGCGCCGGTCTCCCGGA
GCGGGACCGGGTCGGAGGATGGACGAGAATCACGAGCGACGGTGGTGGTGGCGTGTCGGGTTCGTGGCTG
CGGTCGCTCCGGGGCCCCCGGTGGCGGGGCCCCGGGGCTCGCGAGGCGGTTCTCGGTGGGGGCCGAGGGC
CGTCCGGCGTCCCAGGCGGGGCGCCGCGGGACCGCCCTCGTGTCTGTGGCGGTGGGATCCCGCGGCCGTG
TTTTCCTGGTGGCCCGGCCGTGCCTGAGGTTTCTCCCCGAGCCGCCGCCTCTGCGGGCTCCCGGGTGCCC
TTGCCCTCGCGGTCCCCGGCCCTCGCCCGTCTGTGCCCTCTTCCCCGCCCGCCGCCCGCCGATCCTCTTC
TTCCCCCCGAGCGGCTCACCGGCTTCACGTCCGTTGGTGGCCCCGCCTGGGACCGAACCCGGCACCGCCT
CGTGGGGCGCCGCCGCCGGCCACTGATCGGCCCGGCGTCCGCGTCCCCCGGCGCGCGCCTTGGGGACCGG
GTCGGTGGCGCGCCGCGTGGGGCCCGGTGGGCTTCCCGGAGGGTTCCGGGGGTCGGCCTGCGGCGCGTGC
GGGGGAGGAGACGGTTCCGGGGGACCGGCCGCGGCTGCGGCGGCGGCGGTGGTGGGGGGAGCCGCGGGGA
TCGCCGAGGGCCGGTCGGCCGCCCCGGGTGCCCCGCGGTGCCGCCGGCGGCGGTGAGGCCCCGCGCGTGT
GTCCCGGCTGCGGTCGGCCGCGCTCGAGGGGTCCCCGTGGCGTCCCCTTCCCCGCCGGCCGCCTTTCTCG
CGCCTTCCCCGTCGCCCCGGCCTCGCCCGTGGTCTCTCGTCTTCTCCCGGCCCGCTCTTCCGAACCGGGT
CGGCGCGTCCCCCGGGTGCGCCTCGCTTCCCGGGCCTGCCGCGGCCCTTCCCCGAGGCGTCCGTCCCGGG
CGTCGGCGTCGGGGAGAGCCCGTCCTCCCCGCGTGGCGTCGCCCCGTTCGGCGCGCGCGTGCGCCCGAGC
GCGGCCCGGTGGTCCCTCCCGGACAGGCGTTCGTGCGACGTGTGGCGTGGGTCGACCTCCGCCTTGCCGG
TCGCTCGCCCTCTCCCCGGGTCGGGGGGTGGGGCCCGGGCCGGGGCCTCGGCCCCGGTCGCTGCCTCCCG
TCCCGGGCGGGGGCGGGCGCGCCGGCCGGCCTCGGTCGCCCTCCCTTGGCCGTCGTGTGGCGTGTGCCAC
CCCTGCGCCGGCGCCCGCCGGCGGGGCTCGGAGCCGGGCTTCGGCCGGGCCCCGGGCCCTCGACCGGACC
GGCTGCGCGGGCGCTGCGGCCGCACGGCGCGACTGTCCCCGGGCCGGGCACCGCGGTCCGCCTCTCGCTC
GCCGCCCGGACGTCGGGGCCGCCCCGCGGGGCGGGCGGAGCGCCGTCCCCGCCTCGCCGCCGCCCGCGGG
CGCCGGCCGCGCGCGCGCGCGCGTGGCCGCCGGTCCCTCCCGGCCGCCGGGCGCGGGTCGGGCCGTCCGC
CTCCTCGCGGGCGGGCGCGACGAAGAAGCGTCGCGGGTCTGTGGCGCGGGGCCCCCGGTGGTCGTGTCGC
GTGGGGGGCGGGTGGTTGGGGCGTCCGGTTCGCCGCGCCCCGCCCCGGCCCCACCGGTCCCGGCCGCCGC
CCCCGCGCCCGCTCGCTCCCTCCCGTCCGCCCGTCCGCGGCCCGTCCGTCCGTCCGTCCGTCGTCCTCCT
CGCTTGCGGGGCGCCGGGCCCGTCCTCGCGAGGCCCCCCGGCCGGCCGTCCGGCCGCGTCGGGGGCTCGC
CGCGCTCTACCTTACCTACCTGGTTGATCCTGCCAGTAGCATATGCTTGTCTCAAAGATTAAGCCATGCA
TGTCTAAGTACGCACGGCCGGTACAGTGAAACTGCGAATGGCTCATTAAATCAGTTATGGTTCCTTTGGT
CGCTCGCTCCTCTCCTACTTGGATAACTGTGGTAATTCTAGAGCTAATACATGCCGACGGGCGCTGACCC
CCTTCGCGGGGGGGATGCGTGCATTTATCAGATCAAAACCAACCCGGTCAGCCCCTCTCCGGCCCCGGCC
GGGGGGCGGGCGCCGGCGGCTTTGGTGACTCTAGATAACCTCGGGCCGATCGCACGCCCCCCGTGGCGGC
GACGACCCATTCGAACGTCTGCCCTATCAACTTTCGATGGTAGTCGCCGTGCCTACCATGGTGACCACGG
GTGACGGGGAATCAGGGTTCGATTCCGGAGAGGGAGCCTGAGAAACGGCTACCACATCCAAGGAAGGCAG
CAGGCGCGCAAATTACCCACTCCCGACCCGGGGAGGTAGTGACGAAAAATAACAATACAGGACTCTTTCG
AGGCCCTGTAATTGGAATGAGTCCACTTTAAATCCTTTAACGAGGATCCATTGGAGGGCAAGTCTGGTGC
CAGCAGCCGCGGTAATTCCAGCTCCAATAGCGTATATTAAAGTTGCTGCAGTTAAAAAGCTCGTAGTTGG
ATCTTGGGAGCGGGCGGGCGGTCCGCCGCGAGGCGAGCCACCGCCCGTCCCCGCCCCTTGCCTCTCGGCG
CCCCCTCGATGCTCTTAGCTGAGTGTCCCGCGGGGCCCGAAGCGTTTACTTTGAAAAAATTAGAGTGTTC
AAAGCAGGCCCGAGCCGCCTGGATACCGCAGCTAGGAATAATGGAATAGGACCGCGGTTCTATTTTGTTG
GTTTTCGGAACTGAGGCCATGATTAAGAGGGACGGCCGGGGGCATTCGTATTGCGCCGCTAGAGGTGAAA
TTCTTGGACCGGCGCAAGACGGACCAGAGCGAAAGCATTTGCCAAGAATGTTTTCATTAATCAAGAACGA
AAGTCGGAGGTTCGAAGACGATCAGATACCGTCGTAGTTCCGACCATAAACGATGCCGACCGGCGATGCG
GCGGCGTTATTCCCATGACCCGCCGGGCAGCTTCCGGGAAACCAAAGTCTTTGGGTTCCGGGGGGAGTAT
GGTTGCAAAGCTGAAACTTAAAGGAATTGACGGAAGGGCACCACCAGGAGTGGAGCCTGCGGCTTAATTT
GACTCAACACGGGAAACCTCACCCGGCCCGGACACGGACAGGATTGACAGATTGATAGCTCTTTCTCGAT
TCCGTGGGTGGTGGTGCATGGCCGTTCTTAGTTGGTGGAGCGATTTGTCTGGTTAATTCCGATAACGAAC
GAGACTCTGGCATGCTAACTAGTTACGCGACCCCCGAGCGGTCGGCGTCCCCCAACTTCTTAGAGGGACA
AGTGGCGTTCAGCCACCCGAGATTGAGCAATAACAGGTCTGTGATGCCCTTAGATGTCCGGGGCTGCACG
CGCGCTACACTGACTGGCTCAGCGTGTGCCTACCCTACGCCGGCAGGCGCGGGTAACCCGTTGAACCCCA
TTCGTGATGGGGATCGGGGATTGCAATTATTCCCCATGAACGAGGGAATTCCCGAGTAAGTGCGGGTCAT
AAGCTTGCGTTGATTAAGTCCCTGCCCTTTGTACACACCGCCCGTCGCTACTACCGATTGGATGGTTTAG
TGAGGCCCTCGGATCGGCCCCGCCGGGGTCGGCCCACGGCCCTGGCGGAGCGCTGAGAAGACGGTCGAAC
TTGACTATCTAGAGGAAGTAAAAGTCGTAACAAGGTTTCCGTAGGTGAACCTGCGGAAGGATCATTAACG
GAGCCCGGAGGGCGAGGCCCGCGGCGGCGCCGCCGCCGCCGCGCGCTTCCCTCCGCACACCCACCCCCCC
ACCGCGACGCGGCGCGTGCGCGGGCGGGGCCCGCGTGCCCGTTCGTTCGCTCGCTCGTTCGTTCGCCGCC
CGGCCCCGCCGCCGCGAGAGCCGAGAACTCGGGAGGGAGACGGGGGGGAGAGAGAGAGAGAGAGAGAGAG
AGAGAGAGAGAGAGAGAGAAAGAAGGGCGTGTCGTTGGTGTGCGCGTGTCGTGGGGCCGGCGGGCGGCGG
GGAGCGGTCCCCGGCCGCGGCCCCGACGACGTGGGTGTCGGCGGGCGCGGGGGCGGTTCTCGGCGGCGTC
GCGGCGGGTCTGGGGGGGTCTCGGTGCCCTCCTCCCCGCCGGGGCCCGTCGTCCGGCCCCGCCGCGCCGG
CTCCCCGTCTTCGGGGCCGGCCGGATTCCCGTCGCCTCCGCCGCGCCGCTCCGCGCCGCCGGGCACGGCC
CCGCTCGCTCTCCCCGGCCTTCCCGCTAGGGCGTCTCGAGGGTCGGGGGCCGGACGCCGGTCCCCTCCCC
CGCCTCCTCGTCCGCCCCCCCGCCGTCCAGGTACCTAGCGCGTTCCGGCGCGGAGGTTTAAAGACCCCTT
GGGGGGATCGCCCGTCCGCCCGTGGGTCGGGGGCGGTGGTGGGCCCGCGGGGGAGTCCCGTCGGGAGGGG
CCCGGCCCCTCCCGCGCCTCCACCGCGGACTCCGCTCCCCGGCCGGGGCCGCGCCGCCGCCGCCGCCGCG
GCGGCCGTCGGGTGGGGGCTTTACCCGGCGGCCGTCGCGCGCCTGCCGCGCGTGTGGCGTGCGCCCCGCG
CCGTGGGGGCGGGAACCCCCGGGCGCCTGTGGGGTGGTGTCCGCGCTCGCCCCCGCGTGGGCGGCGCGCG
CCTCCCCGTGGTGTGAAACCTTCCGACCCCTCTCCGGAGTCCGGTCCCGTTTGCTGTCTCGTCTGGCCGG
CCTGAGGCAACCCCCTCTCCTCTTGGGCGGGGGGGGCGGGGGGACGTGCCGCGCCAGGAAGGGCCTCCTC
CCGGTGCGTCGTCGGGAGCGCCCTCGCCAAATCGACCTCGTACGACTCTTAGCGGTGGATCACTCGGCTC
GTGCGTCGATGAAGAACGCAGCTAGCTGCGAGAATTAATGTGAATTGCAGGACACATTGATCATCGACAC
TTCGAACGCACTTGCGGCCCCGGGTTCCTCCCGGGGCTACGCCTGTCTGAGCGTCGCTTGCCGATCAATC
GCCCCGGGGGTGCCTCCGGGCTCCTCGGGGTGCGCGGCTGGGGGTTCCCTCGCAGGGCCCGCCGGGGGCC
CTCCGTCCCCCTAAGCGCAGACCCGGCGGCGTCCGCCCTCCTCTTGCCGCCGCGCCCGCCCCTTCCCCCT
CCCCCCGCGGGCCCTGCGTGGTCACGCGTCGGGTGGCGGGGGGGAGAGGGGGGCGCGCCCGGCTGAGAGA
GACGGGGAGGGCGGCGCCGCCGCCGGAAGACGGAGAGGGAAAGAGAGAGCCGGCTCGGGCCGAGTTCCCG
TGGCCGCCGCCTGCGGTCCGGGTTCCTCCCTCGGGGGGCTCCCTCGCGCCGCGCGCGGCTCGGGGTTCGG
GGTTCGTCGGCCCCGGCCGGGTGGAAGGTCCCGTGCCCGTCGTCGTCGTCGTCGCGCGTCGTCGGCGGTG
GGGGCGTGTTGCGTGCGGTGTGGTGGTGGGGGAGGAGGAAGGCGGGTCCGGAAGGGGAAGGGTGCCGGCG
GGGAGAGAGGGTCGGGGGAGCGCGTCCCGGTCGCCGCGGTTCCGCCGCCCGCCCCCGGTGGCGGCCCGGC
GTCCGGCCGACCGGCCGCTCCCCGCGCCCCTCCTCCTCCCCGCCGCCCCTCCTCCGAGGCCCCGCCCGTC
CTCCTCGCCCTCCCCGCGCGTACGCGCGCGCGCCCGCCCGCCCGGCTCGCCTCGCGGCGCGTCGGCCGGG
GCCGGGAGCCCGCCCCGCCGCCCGCCCGTGGCCGCGGCGCCGGGGTTCGCGTGTCCCCGGCGGCGACCCG
CGGGACGCCGCGGTGTCGTCCGCCGTCGCGCGCCCGCCTCCGGCTCGCGGCCGCGCCGCGCCGCGCCGGG
GCCCCGTCCCGAGCTTCCGCGTCGGGGCGGCGCGGCTCCGCCGCCGCGTCCTCGGACCCGTCCCCCCGAC
CTCCGCGGGGGAGACGCGCCGGGGCGTGCGGCGCCCGTCCCGCCCCCGGCCCGTGCCCCTCCCTCCGGTC
GTCCCGCTCCGGCGGGGCGGCGCGGGGGCGCCGTCGGCCGCGCGCTCTCTCTCCCGTCGCCTCTCCCCCT
CGCCGGGCCCGTCTCCCGACGGAGCGTCGGGCGGGCGGTCGGGCCGGCGCGATTCCGTCCGTCCGTCCGC
CGAGCGGCCCGTCCCCCTCCGAGACGCGACCTCAGATCAGACGTGGCGACCCGCTGAATTTAAGCATATT
AGTCAGCGGAGGAAAAGAAACTAACCAGGATTCCCTCAGTAACGGCGAGTGAACAGGGAAGAGCCCAGCG
CCGAATCCCCGCCCCGCGGGGCGCGGGACATGTGGCGTACGGAAGACCCGCTCCCCGGCGCCGCTCGTGG
GGGGCCCAAGTCCTTCTGATCGAGGCCCAGCCCGTGGACGGTGTGAGGCCGGTAGCGGCCGGCGCGCGCC
CGGGTCTTCCCGGAGTCGGGTTGCTTGGGAATGCAGCCCAAAGCGGGTGGTAAACTCCATCTAAGGCTAA
ATACCGGCACGAGACCGATAGTCAACAAGTACCGTAAGGGAAAGTTGAAAAGAACTTTGAAGAGAGAGTT
CAAGAGGGCGTGAAACCGTTAAGAGGTAAACGGGTGGGGTCCGCGCAGTCCGCCCGGAGGATTCAACCCG
GCGGCGGGTCCGGCCGTGTCGGCGGCCCGGCGGATCTTTCCCGCCCCCCGTTCCTCCCGACCCCTCCACC
CGCCCTCCCTTCCCCCGCCGCCCCTCCTCCTCCTCCCCGGAGGGGGCGGGCTCCGGCGGGTGCGGGGGTG
GGCGGGCGGGGCCGGGGGTGGGGTCGGCGGGGGACCGTCCCCCGACCGGCGACCGGCCGCCGCCGGGCGC
ATTTCCACCGCGGCGGTGCGCCGCGACCGGCTCCGGGACGGCTGGGAAGGCCCGGCGGGGAAGGTGGCTC
GGGGGGCCCCGTCCGTCCGTCCGTCCTCCTCCTCCCCCGTCTCCGCCCCCCGGCCCCGCGTCCTCCCTCG
GGAGGGCGCGCGGGTCGGGGCGGCGGCGGCGGCGGCGGTGGCGGCGGCGGCGGGGGCGGCGGGACCGAAA
CCCCCCCCGAGTGTTACAGCCCCCCCGGCAGCAGCACTCGCCGAATCCCGGGGCCGAGGGAGCGAGACCC
GTCGCCGCGCTCTCCCCCCTCCCGGCGCCCACCCCCGCGGGGAATCCCCCGCGAGGGGGGTCTCCCCCGC
GGGGGCGCGCCGGCGTCTCCTCGTGGGGGGGCCGGGCCACCCCTCCCACGGCGCGACCGCTCTCCCACCC
CTCCTCCCCGCGCCCCCGCCCCGGCGACGGGGGGGGTGCCGCGCGCGGGTCGGGGGGCGGGGCGGACTGT
CCCCAGTGCGCCCCGGGCGGGTCGCGCCGTCGGGCCCGGGGGAGGTTCTCTCGGGGCCACGCGCGCGTCC
CCCGAAGAGGGGGACGGCGGAGCGAGCGCACGGGGTCGGCGGCGACGTCGGCTACCCACCCGACCCGTCT
TGAAACACGGACCAAGGAGTCTAACACGTGCGCGAGTCGGGGGCTCGCACGAAAGCCGCCGTGGCGCAAT
GAAGGTGAAGGCCGGCGCGCTCGCCGGCCGAGGTGGGATCCCGAGGCCTCTCCAGTCCGCCGAGGGCGCA
CCACCGGCCCGTCTCGCCCGCCGCGCCGGGGAGGTGGAGCACGAGCGCACGTGTTAGGACCCGAAAGATG
GTGAACTATGCCTGGGCAGGGCGAAGCCAGAGGAAACTCTGGTGGAGGTCCGTAGCGGTCCTGACGTGCA
AATCGGTCGTCCGACCTGGGTATAGGGGCGAAAGACTAATCGAACCATCTAGTAGCTGGTTCCCTCCGAA
GTTTCCCTCAGGATAGCTGGCGCTCTCGCAGACCCGACGCACCCCCGCCACGCAGTTTTATCCGGTAAAG
CGAATGATTAGAGGTCTTGGGGCCGAAACGATCTCAACCTATTCTCAAACTTTAAATGGGTAAGAAGCCC
GGCTCGCTGGCGTGGAGCCGGGCGTGGAATGCGAGTGCCTAGTGGGCCACTTTTGGTAAGCAGAACTGGC
GCTGCGGGATGAACCGAACGCCGGGTTAAGGCGCCCGATGCCGACGCTCATCAGACCCCAGAAAAGGTGT
TGGTTGATATAGACAGCAGGACGGTGGCCATGGAAGTCGGAATCCGCTAAGGAGTGTGTAACAACTCACC
TGCCGAATCAACTAGCCCTGAAAATGGATGGCGCTGGAGCGTCGGGCCCATACCCGGCCGTCGCCGGCAG
TCGAGAGTGGACGGGAGCGGCGGGGGCGGCGCGCGCGCGCGCGCGTGTGGTGTGCGTCGGAGGGCGGCGG
CGGCGGCGGCGGCGGGGGTGTGGGGTCCTTCCCCCGCCCCCCCCCCCACGCCTCCTCCCCTCCTCCCGCC
CACGCCCCGCTCCCCGCCCCCGGAGCCCCGCGGACGCTACGCCGCGACGAGTAGGAGGGCCGCTGCGGTG
AGCCTTGAAGCCTAGGGCGCGGGCCCGGGTGGAGCCGCCGCAGGTGCAGATCTTGGTGGTAGTAGCAAAT
ATTCAAACGAGAACTTTGAAGGCCGAAGTGGAGAAGGGTTCCATGTGAACAGCAGTTGAACATGGGTCAG
TCGGTCCTGAGAGATGGGCGAGCGCCGTTCCGAAGGGACGGGCGATGGCCTCCGTTGCCCTCGGCCGATC
GAAAGGGAGTCGGGTTCAGATCCCCGAATCCGGAGTGGCGGAGATGGGCGCCGCGAGGCGTCCAGTGCGG
TAACGCGACCGATCCCGGAGAAGCCGGCGGGAGCCCCGGGGAGAGTTCTCTTTTCTTTGTGAAGGGCAGG
GCGCCCTGGAATGGGTTCGCCCCGAGAGAGGGGCCCGTGCCTTGGAAAGCGTCGCGGTTCCGGCGGCGTC
CGGTGAGCTCTCGCTGGCCCTTGAAAATCCGGGGGAGAGGGTGTAAATCTCGCGCCGGGCCGTACCCATA
TCCGCAGCAGGTCTCCAAGGTGAACAGCCTCTGGCATGTTGGAACAATGTAGGTAAGGGAAGTCGGCAAG
CCGGATCCGTAACTTCGGGATAAGGATTGGCTCTAAGGGCTGGGTCGGTCGGGCTGGGGCGCGAAGCGGG
GCTGGGCGCGCGCCGCGGCTGGACGAGGCGCGCGCCCCCCCCACGCCCGGGGCACCCCCCTCGCGGCCCT
CCCCCGCCCCACCCGCGCGCGCCGCTCGCTCCCTCCCCACCCCGCGCCCTCTCTCTCTCTCTCTCCCCCG
CTCCCCGTCCTCCCCCCTCCCCGGGGGAGCGCCGCGTGGGGGCGCGGCGGGGGGAGAAGGGTCGGGGCGG
CAGGGGCCGCGCGGCGGCCGCCGGGGCGGCCGGCGGGGGCAGGTCCCCGCGAGGGGGGCCCCGGGGACCC
GGGGGGCCGGCGGCGGCGCGGACTCTGGACGCGAGCCGGGCCCTTCCCGTGGATCGCCCCAGCTGCGGCG
GGCGTCGCGGCCGCCCCCGGGGAGCCCGGCGGCGGCGCGGCGCGCCCCCCACCCCCACCCCACGTCTCGG
TCGCGCGCGCGTCCGCTGGGGGCGGGAGCGGTCGGGCGGCGGCGGTCGGCGGGCGGCGGGGCGGGGCGGT
TCGTCCCCCCGCCCTACCCCCCCGGCCCCGTCCGCCCCCCGTTCCCCCCTCCTCCTCGGCGCGCGGCGGC
GGCGGCGGCAGGCGGCGGAGGGGCCGCGGGCCGGTCCCCCCCGCCGGGTCCGCCCCCGGGGCCGCGGTTC
CGCGCGCGCCTCGCCTCGGCCGGCGCCTAGCAGCCGACTTAGAACTGGTGCGGACCAGGGGAATCCGACT
GTTTAATTAAAACAAAGCATCGCGAAGGCCCGCGGCGGGTGTTGACGCGATGTGATTTCTGCCCAGTGCT
CTGAATGTCAAAGTGAAGAAATTCAATGAAGCGCGGGTAAACGGCGGGAGTAACTATGACTCTCTTAAGG
TAGCCAAATGCCTCGTCATCTAATTAGTGACGCGCATGAATGGATGAACGAGATTCCCACTGTCCCTACC
TACTATCCAGCGAAACCACAGCCAAGGGAACGGGCTTGGCGGAATCAGCGGGGAAAGAAGACCCTGTTGA
GCTTGACTCTAGTCTGGCACGGTGAAGAGACATGAGAGGTGTAGAATAAGTGGGAGGCCCCCGGCGCCCC
CCCGGTGTCCCCGCGAGGGGCCCGGGGCGGGGTCCGCGGCCCTGCGGGCCGCCGGTGAAATACCACTACT
CTGATCGTTTTTTCACTGACCCGGTGAGGCGGGGGGGCGAGCCCGAGGGGCTCTCGCTTCTGGCGCCAAG
CGCCCGCCCGGCCGGGCGCGACCCGCTCCGGGGACAGTGCCAGGTGGGGAGTTTGACTGGGGCGGTACAC
CTGTCAAACGGTAACGCAGGTGTCCTAAGGCGAGCTCAGGGAGGACAGAAACCTCCCGTGGAGCAGAAGG
GCAAAAGCTCGCTTGATCTTGATTTTCAGTACGAATACAGACCGTGAAAGCGGGGCCTCACGATCCTTCT
GACCTTTTGGGTTTTAAGCAGGAGGTGTCAGAAAAGTTACCACAGGGATAACTGGCTTGTGGCGGCCAAG
CGTTCATAGCGACGTCGCTTTTTGATCCTTCGATGTCGGCTCTTCCTATCATTGTGAAGCAGAATTCGCC
AAGCGTTGGATTGTTCACCCACTAATAGGGAACGTGAGCTGGGTTTAGACCGTCGTGAGACAGGTTAGTT
TTACCCTACTGATGATGTGTTGTTGCCATGGTAATCCTGCTCAGTACGAGAGGAACCGCAGGTTCAGACA
TTTGGTGTATGTGCTTGGCTGAGGAGCCAATGGGGCGAAGCTACCATCTGTGGGATTATGACTGAACGCC
TCTAAGTCAGAATCCCGCCCAGGCGAACGATACGGCAGCGCCGCGGAGCCTCGGTTGGCCTCGGATAGCC
GGTCCCCCGCCTGTCCCCGCCGGCGGGCCGCCCCCCCCTCCACGCGCCCCGCCGCGGGAGGGCGCGTGCC
CCGCCGCGCGCCGGGACCGGGGTCCGGTGCGGAGTGCCCTTCGTCCTGGGAAACGGGGCGCGGCCGGAAA
GGCGGCCGCCCCCTCGCCCGTCACGCACCGCACGTTCGTGGGGAACCTGGCGCTAAACCATTCGTAGACG
ACCTGCTTCTGGGTCGGGGTTTCGTACGTAGCAGAGCAGCTCCCTCGCTGCGATCTATTGAAAGTCAGCC
CTCGACACAAGGGTTTGTCCGCGCGCGCGTGCGTGCGGGGGGCCCGGCGGGCGTGCGCGTTCGGCGCCGT
CCGTCCTTCCGTTCGTCTTCCTCCCTCCCGGCCTCTCCCGCCGACCGCGGCGTGGTGGTGGGGTGGGGGG
GAGGGCGCGCGACCCCGGTCGGCCGCCCCGCTTCTTCGGTTCCCGCCTCCTCCCCGTTCACGCCGGGGCG
GCTCGTCCGCTCCGGGCCGGGACGGGGTCCGGGGAGCGTGGTTTGGGAGCCGCGGAGGCGCCGCGCCGAG
CCGGGCCCCGTGGCCCGCCGGTCCCCGTCCCGGGGGTTGGCCGCGCGGCGCGGTGGGGGGCCACCCGGGG
TCCCGGCCCTCGCGCGTCCTTCCTCCTCGCTCCTCCGCACGGGTCGACCGACGAACCGCGGGTGGCGGGC
GGCGGGCGGCGAGCCCCACGGGCGTCCCCGCACCCGGCCGACCTCCGCTCGCGACCTCTCCTCGGTCGGG
CCTCCGGGGTCGACCGCCTGCGCCCGCGGGCGTGAGACTCAGCGGCGTCTCGCCGTGTCCCGGGTCGACC
GCGGCCTTCTCCACCGAGCGGCGGTGTAGGAGTGCCCGTCGGGACGAACCGCAACCGGAGCGTCCCCGTC
TCGGTCGGCACCTCCGGGGTCGACCAGCTGCCGCCCGCGAGCTCCGGACTTAGCCGGCGTCTGCACGTGT
CCCGGGTCGACCAGCAGGCGGCCGCCGGACGCAGCGGCGCACGCACGCGAGGGCGTCGATTCCCCTTCGC
GCGCCCGCGCCTCCACCGGCCTCGGCCCGCGGTGGAGCTGGGACCACGCGGAACTCCCTCTCCCACATTT
TTTTCAGCCCCACCGCGAGTTTGCGTCCGCGGGACCTTTAAGAGGGAGTCACTGCTGCCGTCAGCCAGTA
CTGCCTCCTCCTTTTTCGCTTTTAGGTTTTGCTTGCCTTTTTTTTTTTTTTTTTTTTTTTTTTTTTCTTT
CTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCGCTTGTCTTCTTCTTGTGTTCTCTTCTTG
CTCTTCCTCTGTCTGTCTCTCTCTCTCTCTCTCTCTCTGTCTCTCGCTCTCGCCCTCTCTCTCTTCTCTC
TCTCTCTCTCTCTCTCTCTGTCTCTCGCTCTCGCCCTCTCTCTCTCTCTTCTCTCTGTCTCTCTCTCTCT
CTCTCTCTCTCTCTCTCTCTGTCGCTCTCGCCCTCTCGCTCTCTCTCTGTCTCTGTCTGTGTCTCTCTCT
CTCCCTCCCTCCCTCCCTCCCTCCCTCCCTCCCTCCCCTTCCTTGGCGCCTTCTCGGCTCTTGAGACTTA
GCCGCTGTCTCGCCGTACCCCGGGTCGACCGGCGGGCCTTCTCCACCGAGCGGCGTGCCACAGTGCCCGT
CGGGACGAGCCGGACCCGCCGCGTCCCCGTCTCGGTCGGCACCTCCGGGGTCGACCAGCTGCCGCCCGCG
AGCTCCGGACTTAGCCGGCGTCTGCACGTGTCCCGGGTCGACCAGCAGGCGGCCGCCGGACGCAGCGGCG
CACCGACGGAGGGCGCTGATTCCCGTTCACGCGCCCGCGCCTCCACCGGCCTCGGCCCGCCGTGGAGCTG
GGACCACGCGGAACTCCCTCTCCTACATTTTTTTCAGCCCCACCGCGAGTTTGCGTCCGCGGGACCTTTA
AGAGGGAGTCACTGCTGCCGTCAGCCAGTACTGCCTCCTCCTTTTTCGCTTTTAGGTTTTGCTTGCCTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTT
CTTTCGCTCTCGCTCTCTCGCTCTCTCCCTCGCTCGTTTCTTTCTTTCTCTTTCTCTCTCTCTCTCTCTC
TCTCTCTCTCTCTGTCTCTCGCTCTCGCCCTCTCTCTCTCTTTCTCTCTCTCTCTGTCTCTCTCTCTCTC
TCTCTCTCTCTCTCTCTCTCCCTCCCTCCCTCCCCCTCCCTCCCTCTCTCCCCTTCCTTGGCGCCTTCTC
GGCTCTTGAGACTTAGCCGCTGTCTCGCCGTGTCCCGGGTCGACCGGCGGGCCTTCTCCACCGAGCGGCG
TGCCACAGTGCCCGTCGGGACGAGCCGGACCCGCCGCGTCCCCGTCTCGGTCGGCACCTCCGGGGTCGAC
CAGCTGCCGCCCGCGAGCTCCGGACTTAGCCGGCGTCTGCACGTGTCCCGGGTCGACCAGCAGGCGGCCG
CCGGACGCTGCGGCGCACCGACGCGAGGGCGTCGATTCCGGTTCACGCGCCGGCGACCTCCACCGGCCTC
GGCCCGCGGTGGAGCTGGGACCACGCGGAACTCCCTCTCCCACATTTTTTTCAGCCCCACCGCGAGTTTG
CGTCCGCGGGACTTTTAAGAGGGAGTCACTGCTGCCGTCAGCCAGTAATGCTTCCTCCTTTTTTGCTTTT
TGGTTTTGCCTTGCGTTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTCTCTCTCTC
TCTCTCTCTCTCTCTGTCTCTCTCTCTCTGTCTCTCTCCCCTCCCTCCCTCCTTGGTGCCTTCTCGGCTC
GCTGCTGCTGCTGCCTCTGCCTCCACGGTTCAAGCAAACAGCAAGTTTTCTATTTCGAGTAAAGACGTAA
TTTCACCATTTTGGCCGGGCTGGTCTCGAACTCCCGACCTAGTGATCCGCCCGCCTCGGCCTCCCAAAGA
CTGCTGGGAGTACAGATGTGAGCCACCATGCCCGGCCGATTCCTTCCTTTTTTCAATCTTATTTTCTGAA
CGCTGCCGTGTATGAACATACATCTACACACACACACACACACACACACACACACACACACACACACACA
CACACACCCCGTAGTGATAAAACTATGTAAATGATATTTCCATAATTAATACGTTTATATTATGTTACTT
TTAATGGATGAATATGTATCGAAGCCCCATTTCATTTACATACACGTGTATGTATATCCTTCCTCCCTTC
CTTCATTCATTATTTATTAATAATTTTCGTTTATTTATTTTCTTTTCTTTTGGGGCCGGCCCGCCTGGTC
TTCTGTCTCTGCGCTCTGGTGACCTCAGCCTCCCAAATAGCTGGGACTACAGGGATCTCTTAAGCCCGGG
AGGAGAGGTTAACGTGGGCTGTGATCGCACACTTCCACTCCAGCTTACGTGGGCTGCGGTGCGGTGGGGT
GGGGTGGGGTGGGGTGGGGTGCAGAGAAAACGATTGATTGCGATCTCAATTGCCTTTTAGCTTCATTCAT
ACCCTGTTATTTGCTCGTTTATTCTCATGGGTTCTTCTGTGTCATTGTCACGTTCATCGTTTGCTTGCCT
GCTTGCCTGTTTATTTCCTTCCTTCCTTCCTTCCTTCCTTCCTTCCTTCCTTCCTTCCTTCCCTCCCTTA
CTGGCAGGGTCTTCCTCTGTCTCTGCCGCCCAGGATCACCCCAACCTCAACGCTTTGGACCGACCAAACG
GTCGTTCTGCCTCTGATCCCTCCCATCCCCATTACCTGAGACTACAGGCGCGCACCACCACACCGGCTGA
CTTTTATGTTGTTTCTCATGTTTTCCGTAGGTAGGTATGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGT
GTGTGTGTGTGTGTGTGTGTGTGTGTATCTATGTATGTACGTATGTATGTATGTATGTGAGTGAGATGGG
TTTCGGGGTTCTATCATGTTGCCCACGCTGGTCTCGAACTCCTGTCCTCAAGCAATCCGCCTGCCTGCCT
CGGCCGCCCACACTGCTGCTATTACAGGCGTGAGACGCTGCGCCTGGCTCCTTCTACATTTGCCTGCCTG
CCTGCCTGCCTGCCTGCCTATCAATCGTCTTCTTTTTAGTACGGATGTCGTCTCGCTTTATTGTCCATGC
TCTGGGCACACGTGGTCTCTTTTCAAACTTCTATGATTATTATTATTGTAGGCGTCATCTCACGTGTCGA
GGTGATCTCGAACTTTTAGGCTCCAGAGATCCTCCCGCATCGGCCTCCCGGAGTGCTGTGATGACACGCG
TGGGCACGGTACGCTCTGGTCGTGTTTGTCGTGGGTCGGTTCTTTCCGTTTTTAATACGGGGACTGCGAA
CGAAGAAAATTTTCAGACGCATCTCACCGATCCGCCTTTTCGTTCTTTCTTTTTATTCTCTTTAGACGGA
GTTTCACTCTTGTCGCCCAGGGTGGAGTACGATGGCGGCTCTCGGCTCACCGCACCCTCCGCCTCCCAGG
TTCAAGTGATTCTCCTGCCTCAGCCTTCCCGAGTAGCTGGAATGACAGAGATGAGCCATCGTGCCCGGCT
AATTTTTCTATTTTTAGTACAGATGGGGTTTCTCCATCTTGGTCAGGCTGGTCTTCAACTTCCGACCGTT
GGAGAATCTTAACTTTCTTGGTGGTGGTTGTTTTCCTTTTTCTTTTTTTTTCTTTTCTTTTCTTTCCTTC
TCCTCCCCCCCCCACCCCCCTTGTCGTCGTCCTCCTCCTCCTCCTCCTCCTCCTCCTCCTCCTCCTCCTC
CTCCTCCTCCTCTTTCATTTCTTTCAGCTGGGCTCTCCTACTTGTGTTGCTCTGTTGCTCACGCTGGTCT
CAAACTCCTGGCCTTGACTCTTCTCCCGTCACATCCGCCGTCTGGTTGTTGAAATGAGCATCTCTCGTAA
AATGGAAAAGATGAAAGAAATAAACACGAAGACGGAAAGCACGGTGTGAACGTTTCTCTTGCCGTCTCCC
GGGGTGTACCTTGGACCCGGAAACACGGAGGGAGCTTGGCTGAGTGGGTTTTCGGTGCCGAAACCTCCCG
AGGGCCTCCTTCCCTCTCCCCCTTGTCCCCGCTTCTCCGCCAGCCGAGGCTCCCACCGCCGCCCCTGGCA
TTTTCCATAGGAGAGGTATGGGAGAGGACTGACACGCCTTCCAGATCTATATCCTGCCGGACGTCTCTGG
CTCGGCGTGCCCCACCGGCTACCTGCCACCTTCCAGGGAGCTCTGAGGCGGATGCGACCCCCACCCCCCC
GTCACGTCCCGCTACCCTCCCCCGGCTGGCCTTTGCCGGGCGACCCCAGGGGAACCGCGTTGATGCTGCT
TCGGATCCTCCGGCGAAGACTTCCACCGGATGCCCCGGGTGGGCCGGTTGGGATCAGACTGGACCACCCC
GGACCGTGCTGTTCTTGGGGGTGGGTTGACGTACAGGGTGGACTGGCAGCCCCAGCATTGTAAAGGGTGC
GTGGGTATGGAAATGTCACCTAGGATGCCCTCCTTCCCTTCGGTCTGCCTTCAGCTGCCTCAGGCGTGAA
GACAACTTCCCATCGGAACCTCTTCTCTTCCCTTTCTCCAGCACACAGATGAGACGCACGAGAGGGAGAA
ACAGCTCAATAGATACCGCTGACCTTCATTTGTGGAATCCTCAGTCATCGACACACAAGACAGGTGACTA
GGCAGGGACACAGATCAAACACTATTTCCGGGTCCTCGTGGTGGGATTGGTCTCTCTCTCTCTCTCTCTC
TCTCTCTCTCTCTCTCTCTCTCTCGCACGCGCACGCGCGCACACACACACACAATTTCCATATCTAGTTC
ACAGAGCACACTCACTTCCCCTTTTCACAGTACGCAGGCTGAGTAAAACGCGCCCCACCCTCCACCCGTT
GGCTGACGAAACCCCTTCTCTACAATTGATGAAAAAGATGATCTGGGCCGGGCACGCTAGCTCACGCCTG
TCACTCCGGCACTTTGGGAGGCCGAGGCGGGTGGATCGCTTGGGGCCGGGAGTTCGAGACCAGGCTGGCC
GACGTGGCGAAACCCCGTCTCTCTGAAAAATAGAACGATTAGCCGGGCCTGGTGGCGTGGGCTTGGAATC
ACGACCGCTCGGGAGACTGGGGCGGGCGACTTGTTCCAACCGGGGAGGCCGAGGCCGCGATGAGCTGAGA
TCGTGCCGTGGCGATGCGGCCTGGATGACGGAGCGAGACCCCGTCTCGAGAGAATCATGATGTTATTATA
AGATGAGTTGTGCGCGGTGATGGCCGCCTGTAGTCGCGGCTACTCGGGAGGCTGAGACGAGGAGAAGATC
ACTTGAGGCCCCACAGGTCGAGGCTTCGGTCGGCCGTGACCCACTGTATCCTGGGCAGTCACCGGTCAAG
GAGATATGCCCCTTCCCCGTTTGCTTTTCTTTTCTTCCCTTCTCTTTTCTTCTTTTTGCTTCTCTTTTCT
TTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTTCTTTTTCTCTCTTCCCCTCTTTCTTT
CCTGCCTTCCTGCCTTTCTTCTTTTCTTCTTTCCTCCCTTCCTCCCTTCCTTCTTTCCTCCCGCCTCAGC
CTCCCAAAGTGCTGGGATGACTGGCGGGAGGCACCATGCCTGCTTGGCCCAAAGAGACCCTCTTGGAAAG
TGAGACGCAGAGAGCGCCTTCCAGTGATCTCATTGACTGATTTAGAGACGGCATCTCGCTCCGTCACCCC
GGCAGTGGTGCCGTCGTAACTCACTCCCTGCAGCGTGGACGCTCCTGGACTCGAGCGATCCTTCCACCTC
AGCCTCCAGAGTACAGAGCCTGGGACCGCGGGCACGCGCCACTGTGCCCACACCGTTTTTAATTGTTTTT
TTTTCCCCCGAGACAGAGTTTCACTCTCGTGGCCTAGACTGCAGTGCGGTGGCGCGATCTTGGCTCACCG
CAACCTCTGCCTCCCGGTTTCAAGCGATTCTCCTGCATCGGCCTCCTGAGTAGCCGGGATTGCGGGCATG
CGCTGCCACGTCTGGCTGATTTCGTATTTTTAGTGGAGACGGGGCTTCTCCATGTCGATCGGGCTGGTTT
CGAACTCCCGACCTCAGGTGATCCGCCCTCCCCGGCCTCCGGAAGTGCTGGGATGACAGGCGTGAGCCAC
CGCGCCCGGCCTTCATTTTTAAATGTTTTCCCACAGACGGGGTCTCATCATTTCTTTGCAACCCTCCTGC
CCGGCGTCTCAAAGTGCTGGCGTGACGGGCGTGAGCCACTGCGCCTGGACTCCGGGGAATGACTCACGAC
CACCATCGCTCTACTGATCCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTT
TCTTTCTTGATGAATTATCTTATGATTTATTTGTGTACTTATTTTCAGACGGAGTCTCGCTCTGGGCGGG
GCGAGGCGAGGCGAGGCACAGCGCATCGCTTTGGAAGCCGCGGCAACGCCTTTCAAAGCCCCATTCGTAT
GCACAGAGCCTTATTCCCTTCCTGGAGTTGGAGCTGATGCCTTCCGTAGCCTTGGGCTTCTCTCCATTCG
GAAGCTTGACAGGCGCAGGGCCACCCAGAGGCTGGCTGCGGCTGAGGATTAGGGGGTGTGTTGGGGCTGA
AAACTGGGTCCCCTATTTTTGATACCTCAGCCGACACATCCCCCGACCGCCATCGCTTGCTCGCCCTCTG
AGATCCCCCGCCTCCACCGCCTTGCAGGCTCACCTCTTACTTTCATTTCTTCCTTTCTTGCGTTTGAGGA
GGGGGTGCGGGAATGAGGGTGTGTGTGGGGAGGGGGTGCGGGGTGGGGACGGAGGGGAGCGTCCTAAGGG
TCGATTTAGTGTCATGCCTCTTTCACCACCACCACCACCACCGAAGATGACAGCAAGGATCGGCTAAATA
CCGCGTGTTCTCATCTAGAAGTGGGAACTTACAGATGACAGTTCTTGCATGGGCAGAACGAGGGGGACCG
GGGACGCGGAAGTCTGCTTGAGGGAGGAGGGGTGGAAGGAGAGACAGCTTCAGGAAGAAAACAAAACACG
AATACTGTCGGACACAGCACTGACTACCCGGGTGATGAAATCATCTGCACACTGAACACCCCCGTCACAA
GTTTACCTATGTCACAATCTTGCACATGTATCGCTTGAACGACAAATAAAAGTTAGGGGGGAGAAGAGAG
GAGAGAGAGAGAGAGAGAGAGACAGAGAGAGACAGAGAGAGAGAGAGAGGAGGGAGAGAGGAAAACGAAA
CACCACCTCCTTGACCTGAGTCAGGGGGTTTCTGGCCTTTTGGGAGAACGTTCAGCGACAATGCAGTATT
TGGGCCCGTTCTTTTTTTTTCTTCTTCTTTTCTTTCTTTTTTTTTGGACTGAGTCTCTCTCGCTCTGTCA
CCCAGGCTGCGGTCGCGGTGGCGCTCTCTCGGCTCACTGAAACCTCTGCTTCCCGGGTTCCAGTGATTCT
TCTTCGGTAGCTGGGATTACAGGCGCACACCATGACGGCGGGCTCATATTCCTATTTTCAGTAGAGACGG
GGTTTCTCCACGTTGGCCACGCTGGTCTCGAACTCCTGACCTCAAATGATCCGCCTTCCTGGGCCTCCCA
AAGTGCTGGAAACGACAGGCCTGAGCCGCCGGGATTTCAGCCTTTAAAAGCGCGGCCCTGCCACCTTTCG
CTGTGGCCCTTACGCTCAGAATGACGTGTCCTCTCTGCCGTAGGTTGACTCCTTGAGTCCCCTAGGCCAT
TGCACTGTAGCCTGGGCAGCAAGAGCCAAACTCCGNNCCCCCACCTCCTCGCGCACATAATAACTAACTA
ACAAACTAACTAACTAACTAAACTAACTAACTAACTAAAATCTCTACACGTCACCCATAAGTGTGTGTTC
CCGTGAGAGTGATTTCTAAGAAATGGTACTGTACACTGAACGCAGTGGCTCACGTCTGTCATCCCGAGGT
CAGGAGTTCGAGACCAGCCCGGCCAACGTGGTGAAACCCCGTCTCTACTGAAAATACGAAATGGAGTCAG
GCGCCGTGGGGCAGGCACCTGTAACCCCAGCTACTCGGGAGGCTGGGGTGGAAGAATTGCTTGAACCTGG
CAGGCGGAGGCTGCAGTGACCCAAGATCGCACCACTGCACTACAGCCTGGGCGACAGAGTGAGACCCGGT
CTCCAGATAAATACGTACATAAATAAATACACACATACATACATACATACATACATACATACATACATAC
ATCCATGCATACAGATATACAAGAAAGAAAAAAAGAAAAGAAAAGAAAGAGAAAATGAAAGAAAAGGCAC
TGTATTGCTACTGGGCTAGGGCCTTCTCTCTGTCTGTTTCTCTCTGTTCGTCTCTGTCTTTCTCTCTGTG
TCTCTTTCTCTGTCTGTCTGTCTCTTTCTTTCTCTCTGTCTCTGTCTCTGTCTTTGTCTCTCTCTCTCCC
TCTCTGCCTGTCTCACTGTGTCTGTCTTCTGTCTTACTCTCTTTCTCTCCCCGTCTGTCTCTCTCTCTCT
CTCTCCCTCCCTGTTTGTTTCTCTCTCTCCCTCCCTGTCTGTTTCTCTCTCTCTCTTTCTGTCTGTTTCT
GTCTCTCTCTGTCTGTCTATGTCTTTCTCTGTCTGTCTCTTTCTCTGTCTGTCTGCCTCTCTCTTTCTTT
TTCTGTGTCTCTCTGTCGGTCTCTCTCTCTCTGTCTGTCTGTCTGTCTCTCTCTCTCTCTCTCTGTGCCT
ATCTTCTGTCTTACTCTCTTTCTCTGCCTGTCTGTCTGTCTCTCCCTCCCTTTCTGTTTCTCTCTCTCTC
TCTCTCTCTCTCCCCCTCTCCCTGTCTGTTTCTCTCCGTCTCTCTCTCTTTCTGTCTGTTTCTCACTGTC
TCTCTCTGTCCATCTCTCTCTCTCTCTGTCTGTCTCTTTCGTTCTCTCTGTCTGTCTGTCTCTCTCTCTC
TCTCTCTCTCTCTCTCTCTCTCCCTGTCTGTCTGTTTCTCTCTATCTCTCGCTGTCCATCTCTGTCTTTC
TATGTCTGTCTCTTTCTCTGTCAGTCTGTCAGACACCCCCGTGCCGGGTAGGGCCCTGCCCCTTCCACGA
AAGTGAGAAGCGCGTGCTTCGGTGCTTAGAGAGGCCGAGAGGAATCTAGACAGGCGGGCCTTGCTGGGCT
TCCCCACTCGGTGTATGATTTCGGGAGGTCGAGGCCGGGTCCCCGCTTGGATGCGAGGGGCATTTTCAGA
CTTTTCTCTCGGTCACGTGTGGCGTCCGTACTTCTCCTATTTCCCCGATAAGCTCCTCGACTTCAACATA
AACGGCGTCCTAAGGGTCGATTTAGTGTCATGCCTCTTTCACCGCCACCACCGAAGATGAAAGCAAAGAT
CGGCTAAATACCGCGTGTTCTCATCTAGAAGTGGGAACTTACAGATGACAGTTCTTGCATGGGCAGAACG
AGGGGGACCGGGNACGCGGAAGCCTGCTTGAGGGRGGAGGGGYGGAAGGAGAGACAGCTTCAGGAAGAAA
ACAAAACACGAATACTGTCGGACACAGCACTGACTACCCGGGTGATGAAATCATCTGCACACTGAACACC
CCCGTCACAAGTTTACCTATGTCACAGTCTTGCTCATGTATGCTTGAACGACAAATAAAAGTTCGGGGGG
GAGAAGAGAGGAGAGAGAGAGAGAGACGGGGAGAGAGGGGGGAGAGGGGGGGGGAGAGAGAGAGAGAGAG
AGAGAGAGAGAGAGAGAGAGAGAAAGAGAAGTAAAACCAACCACCACCTCCTTGACCTGAGTCAGGGGGT
TTCTGGCCTTTTGGGAGAACGTTCAGCGACAATGCAGTATTTGGGCCCGTTCTTTTTTTCTTCTTCTTCT
TTTCTTTCTTTTTTTTTGGACTGAGTCTCTCTCGCTCTGTCACCCAGGCTGCGGTGCGGTGGCGCTCTCT
CGGCTCACTGAAACCTCTGCTTCCCGGGTTCCAGTGATTCTTCTTCGGTAGCTGGGATTACAGGTGCGCA
CCATGACGGCCGGCTCATCGTTCTATTTTTAGTAGAGACGGGGTTTCTCCACGTTGGCCACGCTGGTCTC
GAACTCCTGACCACAAATGATCCACCTTCCTGGGCCTCCCAAAGTGCTGGAAACGACAGGCCTGAGCCGC
CGGGATTTCAGCCTTTAAAAGCGCGCGGCCCTGCCACCTTTCGCTGCGGCCCTTACGCTCAGAATGACGT
GTCCTCTCTGCCATAGGTTGACTCCTTGAGTCCCCTAGGCCATTGCACTGTAGCCTGGGCAGCAAGAGCC
AAACTCCGTCCCCCCACCTCCCCGCGCACATAATAACTAACTAACTAACTAACTAACTAAAATCTCTACA
CGTCACCCATAAGTGTGTGTTCCCGTGAGGAGTGATTTCTAAGAAATGGTACTGTACACTGAACGCAGGC
TTCACGTCTGTCATCCCGAGGTCAGGAGTTCGAGACCAGCCCGGCCCACGTGGTGAAACCCCCGTCTCTA
CTGAAAATACGAAATGGAGTCAGGCGCCGTGGGGCAGGCACCTGTAACCCCAGCTACTCGGGAGGCTGGG
GTGGAAGAATTGCTTGAACCTGGCAGGCGGAGGCTGCAGTGACCCAAGATCGCACCACTGCACTACAGCC
TGGGCGACAGAGTGAGACCCGGTCTCCAGATAAATACGTACATAAATAAATACACACATACATACATACA
TACATACAACATACATACATACAGATATACAAGAAAGAAAAAAAGAAAAGAAAAGAAAGAGAAAATGAAA
GAAAAGGCACTGTATTGCTACTGGGCTAGGGCCTTCTCTCTGTCTGTTTCTCTCTGTTCGTCTCTGTCTT
TCTCTCTGTGTCTCTTTCTCTGTCTGTCTGTCTGTCTGTCTGTCTGTCTCTTTCTTTCTTTCTGTCTCTG
TCTTTGTCCCTCTCTCTCCCTCTCTGCCCTGTCTCACTGTGTCTGTCTTCTATCTTACTCTCTTTCTCTC
CCCGTCTGTCTCTCTCTCACTCCCTCCCTGTCTGTTTCTCTCTCTCTCTCTTTCTGTCTGTTTCTGTCTC
TCTCTGTCTGCCTCTCTCTTTCTCTATCTGTCTCTTTCTCTGTCTGTCTGCCCCTCTCTTTCTTTTTCTG
TGTCTCTCTGTCTGTCTCTCTCTCTCTCTGTGCCTATCTTCTGTCTTACTCTCTTTCTCTGCCTGTCTGT
CTGTCTCTCTCTGTCTCTCCCTCCCTTTCTGCTTCTCTCTCTCTCTCTCTCTCTNNNCCCTCCCTGTCTG
TTTCTCTCTGTCTCCCTCTCTTTCTGTCTGTTTCTCACTGTCTCTCTCTGTCTGTCTGTTTCATTCTCTC
TGTCTCTGTCTCTGTCTCTCTCTCTCTCTGTCTCTCCCTCTCTGTGTGTATCTTTTGTCTTACTCTCCTT
CTCTGCCTGTCCGTCTGTCTGTCTGTCTCTCTCTCTCCCTGTCCCTCTCTCTTTCTGTCTGTTTCTCTCT
CTCTCTCTCTCTCTCTCTCTCTGTCTCTGTCTTTCTCTGTCTGTCCCTTTCTCTGTCTGTCTGCCTCTCT
CTTTCTCTTTCTGTGTCTCTCTGTCTCTCTCTCTGTGCCTATCTTCTGTCTTACTCTCTTTCTCTGCCTG
TCTATCTGTCTGTCTCTCTCTGTCTCTCTCCCTGCCTTTCTGTTTCTCTCTCTCTCCCTCTCTCGCTCTC
TCTGTCTTTCTCTCTTTCTCTCTGTTTCTCTGTCTCTCTCTGTCCGTCTCTGTCTTTTTCTGTCTGTCTG
TCTCTCTCTTTCTTTCTGTCGTCTGTCTCTGTCTCTGTCTCTGTCTCTCTCTCTCTCTCTCTCCTTGTCT
CTCTCACTGTGTCTGTCTTCTGTCTTACTCTCCTTCTCTGCCTGTCCATCTGTCTGTCTGTCTCTCTCTC
TCTCTCCCTACCTTTCTGTTTCTCTCTCGCTAGCTCTCTCTCTCTCTGCCTGTTTCTCTCTTTCTCTCTC
TGTCTTTCTCTGTCTGTCTCTTTCTCTGTCTGTCTGTCTCTTTCTCTCTGTCTCTGTCTCTGTCTCTCTC
TCTCTCTCTCTCTCTCTCTCTGCCTCTCTCACTGTGTCTGTCTTCTGTCTTATTCTCTTTCTCTCTCTGT
CTCTCTCTCTCTCTCCTTTACTGTCTGTTTCTCTCTCTCTCTCTCTCTTTCTGCCTGTTTCTCTCTGTCT
GTCTCTGTCTTTCTCTGTCTGTCTGCCTCTCTCTTTCTTTTTCTGCGTCTCTCTGTCTCTCTCTCTCTCT
CTCTGTTCCTATCTTCTGTCTTACTCTGTTTCCTTGCCTGCCTGCCTGTCTGTGTGTCTGTCTCTCTCTC
TCTCTCTCTCTCTCTCTCCCTCCCTTTCTCTTTCTCTGTCTCTCTCTCTCTTTCTGGGTGTTTCTCTCTG
TCTCTCTGTCCATCTCTGTCTTTCTATGTCTGTCTCTCTCTTTCTCTCTGTCTCTGTCTCTGCCTCTCTC
TCTCTCTCTCTCTCTCTCTCTCTGTCTGTCTCTCTCACTGTGTGTGTCTGTCTTCTGTCTTACTCTCCTT
CTCTGCCTGTCCGTCTGTCTGTCTGTCTCTCCCTCTCTCTCCCTCCCTTTCTGTTTCTCTCTCTCTCTCT
TTCTGTCTGTTTCTCTCTTTCTCTCTCTGTCTGTCTCTTTCTCTGTCTGTCTGTCTCTCTCTTTCTTTTT
CTCTGTCTCTCTGTCTCTCTCTGTGTCTGTCTCTCTGTCTGTGCCTATCTTCTGTCTTACTCTCTTTCTC
TGGCTGTCTGCCTGTCTCTCTCTCTCTCTCTGTCTGTCTCCGTCCCTCTCTCCCTGTCTGTCTGTTTCTC
TCTCTGCCTCTCTCTCTCTCTGTCTGTCTCTTTCTCTGTCTGTCTGTCTCTCTCTTTCTTTTTCTCTGTC
TCTCTGTCTCTCTCTGTGTCTGTCTCTCTTTCTGTGCCTATCTTCTGTCTTACTCTCTTTCTCTGGCTGT
CTGCCTGTCTCTCTCTCTCTGCCTGTCTCCGTCCCTCCCTCCCTGTCTGTCTGTTTCTCTCTCTGTCTCT
GTCTCTCTGTCCATCTCTGTCTGTCTCTTTCTCTTTCTCTCTCTCTGTCTCTGTCTCTCTCTCTCTCTGC
CTGTCTCTCTCACTGTGTCTGTCTTCTGTCTTACTCTCTTTCTCTTGCCTGCCTCTCTGTCTGTCTGTCT
CTCTCCCTCCATGTCTCTCTCTCTCTCTCACTCACTCTCTCTCCGTCTCTCTCTCTTTCTGTCTGTTTCT
CTCTCTGTCTGTCTCTCTCCCTCCATGTCTCTCTCTCTCTCTCTCACTCACTCTCTCTCCGTCTCTCTCT
CTCTTTCTGTCTGTTTCTCTCTCTGTCTGTCTCTCTCCCTCCATGTCTCTCTCTCTCCCTCTCACTCACT
CTCTCTCCGTCTCTCTCTCTCTTTCTGTCTGTTTCTTTGTCTGTCTGTCTGTCTGTCTGTCTGTCTCTCT
CTCTCTCTCTCTCTCTCTCTCTCTCTGTTTGTCTTTCTCCCTCCCTGTCTGTCTGTCTGTCTCTCTCTCT
CTGTCTCTGTCTCTGTCTCTCTCTCTTTCTCTTTCTGTCTGTTTCTCTCTATCTCTCGCTGTCCATCTCT
GTCTTTCTATGTCTGTCTCTTTCTCTGTCAGTCTGTCAGACACACCCGTGCCGGTAGGGCCCTGCCCTTC
CACGAGAGTGAGAAGCGCGTGCTTCGGTGCTTAGAGAGGCCGAGAGGAATCTAGACAGGCGGGCCTTGCT
GGGCTTCCCCACTCGGTGTACGATTTCGGGAGGTCGAGGCCGGGTCCCCGCTTGGATGCGAGGGGCATTT
TCAGACTTTTCTCTCGGTCACGTGTGGCGTCCGTACTTCTCCTATTTCCCCGATAAGTCTCCTCGACTTC
AACATAAACTGTTAAGGCCGGACGCCAACACGGCGAAACCCCGTCTCTACTAAAAATACAAAGCTGAGTC
GGGAGCGGTGGGGCAGGCCCTGTAATGCCAGCTCCTCGGGAGGCTGAGGCGGGAGAATCGCTTGAACCAG
GGAAGCGGAGGCTGCAGGGAGCCGAGATCGCGCCACTGCACTACGGCCCAGGCTGTAGAGTGAGTGAGAC
TCGGTCTCTAAATAAATACGGAAATTAATTAATTCATTAATTCTTTTCCCTGCTGACGGACATTTGCAGG
CAGGCATCGGTTGTCTTCGGGCATCACCTAGCGGCCACTGTTATTGAAAGTCGACGTTGACACGGAGGGA
GGTCTCGCCGACTTCACCGAGCCTGGGGCAACGGGTTTCTCTCTCTCCCTTCTGGAGGCCCCTCCCTCTC
TCCCTCGTTGCCTAGGGAACCTCGCCTAGGGAACCTCCGCCCTGGGGGCCCTATTGTTCTTTGATCGGCG
CTTTACTTTTCTTTGTGTTTTGGCGCCTAGACTCTTCTACTTGGGCTTTGGGAAGGGTCAGTTTAATTTT
CAAGTTGCCCCCCGGCTCCCCCCACTACCCACGTCCCTTCACCTTAATTTAGTGAGNCGGTTAGGTGGGT
TTCCCCCAAACCGCCCCCCCCCCCCCGCCTCCCAACACCCTGCTTGGAAACCTTCCAGAGCCACCCCGGT
GTGCCTCCGTCTTCTCTCCCCTTCCCCCACCCCTTGCCGGCGATCTCATTCTTGCCAGGCTGACATTTGC
ATCGGTGGGCGTCAGGCCTCACTCGGGGGCCACCGTTTTTGAAGATGGGGGCGGCACGGTCCCACTTCCC
CGGAGGCAGCTTGGGCCGATGGCATAGCCCCTTGACCCGCGTGGGCAAGCGGGCGGGTCTGCAGTTGTGA
GGCTTTTCCCCCCGCTGCTTCCCGCTCAGGCCTCCCTCCCTAGGAAAGCTTCACCCTGGCTGGGTCTCGG
TCACCTTTTATCACGATGTTTTAGTTTCTCCGCCCTCCGGCCAGCAGAGTTTCACAATGCGAAGGGCGCC
ACGGCTCTAGTCTGGGCCTTCTCAGTACTTGCCCAAAATAGAAACGCTTTCTGAAAACTAATAACTTTNC
TCACTTAAGATTTCCAGGGACGGCGCCTTGGCCCGTGTTTGTTGGCTTGTTTTGTTTCGTTCTGTTTTGT
TTTGTTCGTGTTTTTCCTTTCTCGTATGTCTTTCTTTTCAGGTGAAGTAGAAATCCCCAGTTTTCAGGAA
GACGTCTATTTTCCCCAAGACACGTTAGCTGCCGTTTTTTCCTGTTGTGAACTAGCGCTTTTGTGACTCT
CTCAACGCTGCAGTGAGAGCCGGTTGATGTTTACNATCCTTCATCATGACATCTTATTTTCTAGAAATCC
GTAGGCGAATGCTGCTGCTGCTCTTGTTGCTGTTGTTGTTGTTGTTGTTGTCGTCGTTGCTGTTGTCGTT
GTCGTTGTTGTTGTCGTTGTCGTTGTTTTCAAAGTATACCCCGGCCACCGTTTATGGGATCAAAAGCATT
ATAAAATATGTGTGATTATTTCTTGAGCACGCCCTTCCTCCCCCTCTCTCTGTCTCTCTGTCTGTCTCTG
TCTCTCTCTTTCTCTGTCTGTCTTCTCTCTCTCTCTCTCTCTGTGTCTCTCTCTCTCTGCCTGTCTGTTT
CTCTCTCTCTGCCTCTCTCTCTCTCTCTCTCTCTGCCTGTCTCTCTCACTGTGTCTGTCTTCTGTCTTAC
TCCCTTTCTCTGTCTGTCTGTCGGTCTCTCTCTCTCTCTCTCCCTGTCTGTATGTTTCTCTCTGTCTCTG
TCTCTCTCTCTCTTTCTGTTTCTCTCTCTCCGTCTCTGTCTTTCTCTGACTGTCTCTCTCTTTCCTTCTC
TCTGTCTCTCTCTGCCTGTCTCTCTCACTCTGTCTTCTGTCTTATCTCTCTCTCTGCCTGCCTGTCTCTC
TCACTCTCTCTCTCTGTGTGTCTCTCTCTCTCTTTCTGTTTCTCTCTGTCTCTCTGTCCGTCTCTGTCTT
TCTCTGTCTGTCTCTTTGTCTGTCTGTCTTTGTCTTTCCTTCTCTCTGTCTCTGTCTCTCTCACTGTGTC
TGTCTTCTGTCTTAGTCTCTCTCTCTCTCTCTCCCTGTCTGTCTGTCTCTCTCTCTCTCTCCCCCTGTCT
GTTTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTGTCTTTGTCTTTCTTTCTGTCTCTGTCTCTCTCTCT
CTCTCTGTGTGTCTGTCTTCTGTCTTACTGTCTTTCTCTGCCTGTCTGTCTGTCTGTCTCTCTCTGTCTG
TCTCTCTCTCTCTCTCCCCCTGTCGGCTGTTTCTCTGTCTCTGTCTGTGTCTCTCTTTCTGTCTGTTTCT
CTCTGTCTGTCTTTCTCTCTCTGTCTCTTTCTCTCTGTCTCTCTGTCTGTCTCTGTCTCTCTCTCTGTCT
CTCTCTCTCTGTGGGGGTGTGTGTGTGTGTGTGTATGTGTGTGTGTGTGTGTGTGTGTGTCTGCCTTCTG
TCTTACTCTCTTTCTCTGCCTGTCTGTCTGCCTGTCTGTTTGTCTCTCTCTCTCTGCCTGTCTCTCTCCC
TTCCTGTCTGTTTCTCTCTCTTTCTGTTTCTCTCTGTCTCTGTCCATCTCTGTCTTTCTCCGTCTGTCTC
TTTATCTGTCTCTCTCCGTCTGTCTCTTTATCTGTCTCTCTCTCTCTTTCTGTCTTTCTCTCTCTGTGTA
TCGTTGTCTCTCTCTGTCTGTCTCTGTCTCTGTCTCTCTGTCTCTCTCTCTCTCTCTCTCTCTCTGTCTG
TCTGTCCGTCTGTCTGTCTCGGTCTCTGCGTCTCGCTATCTCCCGCCCTCTCTTTTTTTGCAAAAGAAGC
TCAAGTACATCTAATCTAATCCCTTACCAAGGCCTGAATTCTTCACTTCTGACATCCCAGATTTGATCTC
CCTACAGAATGCTGTACAGAACTGGCGAGTTGATTTCTGGACTTGGATACCTCATAGAAACTACATATGA
ATAAAGATCCAATCCTAAAATCTGGGGTGGCTTCTCCCTCGACTGTCTCGAAAAATCGTACCTCTGTTCC
CCTAGGATGCCGGAAGAGTTTTCTCAATGTGCATCTGCCCGTGTCCTAAGTGATCTGTGACCGAGCCCTG
TCCGTCCTGTCTCAAATATGTACGTGCAAACACTTCTCTCCATTTCCACAACTACCCACGGCCCCTTGTG
GAACCACTGGCTCTTTGAAAAAAATCCCAGAAGTGGTTTTGGCTTTTTGGCTAGGAGGCCTAAGCCTGCT
GAGAACTTTCCTGCCCAGGATCCTCGGGACCATGCTTGCTAGCGCTGGATGAGTCTCTGGAAGGACGCAC
GGGACTCCGCAAAGCTGACCTGTCCCACCGAGGTCAAATGGATACCTCTGCATTGGCCCGAGGCCTCCGA
AGTACATCACCGTCACCAACCGTCACCGTCAGCATCCTTGTGAGCCTGCCCAAGGCCCCGCCTCCGGGGA
GACTCTTGGGAGCCCGGCCTTCGTCGGCTAAAGTCCAAAGGGATGGTGACTTCCACCCACAAGGTCCCAC
TGAACGGCGAAGATGTGGAGCGTAGGTCAGAGAGGGGACCAGGAGGGGAGACGTCCCGACAGGCGACGAG
TTCCCAAGGCTCTGGCCACCCCACCCACGCCCCACGCCCCACGTCCCGGGCACCCGCGGGACACCGCCGC
TTTATCCCCTCCTCTGTCCACAGCCGGCCCCACCCCACCACGCAACCCACGCACACACGCTGGAGGTTCC
AAAACCACACGGTGTGACTAGAGCCTGACGGAGCGAGAGCCCATTTCACGAGGTGGGAGGGGTGGGGGTG
GGGTGGGTTGGGGGTTGTGGGGTCTGTGGCGAGCCCGATTCTCCCTCTTGGGTGGCTACAGGCTAGAAAT
GAATATCGCTTCTTGGGGGGAGGGGCTTCCTTAGGCCATCACCGCTTGCGGGACTACCTCTCAAACCCTC
CCTTGAGGCCACAAAATAGATTCCACCCCACCCATCGACGTTTCCCCCGGGTGCTGGATGTATCCTGTCA
AGAGACCTGAGCCTGACACCGTCGAATTAAACACCTTGACTGGCTTTGTGTGTTTGTTTGTTTCTGAGAT
GGAGTCTTGCTCTGTCCCCCAGGCTGGAGTGCAGTGGCGTGATCTCAGCTCACTGGAACCTCTGCCTCCT
GGGTTCAAGTGATTCTCCTGTCTCAGCGCCACCATGGCCGGCTCATTTTTTTTTTTTTTTTTTTTGGTAG
ACACGGGGTTTCACCCTCTTTCATTGGTTTTCACTGGAGATTCTAGATTCGAGCCACACCTCATTCCGTG
CCACAGAGAGACTTCTTTTTTTTTTTTTTTTTTTTAAGCGCAACGCAACATGTCTGCCTTATTTGAGTGG
CTTCCTATATCATTATAATTGTGTTATAGATGAAGAAACGGTATTAAACACTGTGCTAATGATAGTGAAA
GTGAAGACAAAAGAAAGGCTATCTATTTTGTGGTTAGAATAAAGTTGCTCAGTATTTAGAAGCTACCTAA
ATACGTCAGCATTTACACTCTTCCTAGTAAAAGCTGGCCGATCTGAATAATCCTCCTTTAAACAAACACA
ATTTTTGATAGGGTTAAGATTTTTTTAAGAATGCGACTCCTGCAAAATAGCTGAACAGACGATACACATT
TAAAAAAATAACAACACAAGGATCAACCAGACTTGGGAAAAAATCGAAAACCACACAAGTCTTATGAAGA
ACTGAGTTCTTAAAATAGGACGGAGAACGTAGCTATCGGAAGAGAAGGCAGTATTGGCAAGTTGATTGTT
ACGTTGGTCAGCAGTAGCTGGCACTATCTTTTTGGCCATCTTTCGGGCAATGTAACTACTACAGCAAAAT
GAGATATGATCCATTAAACAACATATTCGCAAATCAAAAAGTGTTTCAGTAATATAATGCTTCAGATTTA
GAAGCAAATCAAATGATAGAACTCCACTGCTGTAATAAGTCACCCCAAAGATCACCGTATCTGACAAAAT
AACTACCACAGGGTTATGACTTCAGAATCATACTTTCTTCTTGATATTTACTTATGTATTTATTTTTTTT
AATTTATTTCTCTTGAGACGCGTCTCGCTCTGTCGCCCAGGCTGGAGTGCGATGGTGTGATCTCGGCTCA
CTGCAACCGCCACCTCCCTGGGTTCAAGCGATTCTCCTGCCTCAGCCTCCCGAGTAGCTGGGACTACAGG
TGCCCGCCACCACGCCCAGCTAATCTTTATACTTTTAATAGAGACGGGGTTTCACCGTGTCGGCCCGGAT
GGTCTCGATCTCTTGACCTCGTGACCCGCCCGCCTCGGCCTCCCAAAGTGCTGGGATGACAGGCGTGAGC
CACTGAGCCCGGCCTTCTCTTGACGTTTAAACTATGAAGTCAGTCCAGAGAAACGCAATAAATGTCAACG
GTGAGGATGGTGTTGAGGCAGAAGTAGGACCACACTTTTTCCTATCTTATTCAGTTGATAACAATATGAC
CTAGGTAGTAATTTCCTATGTGCCTACTTATACACGAGTACAAAAGAGTAAAACAGAGAGACTGCTAAAT
TAAAGGGTACGTGAAGTTCTTCATAGTAACTCCGTAAACTGGAACACTGTCAAAAAGCAGCAGCTAGTGA
ATTGTTTCCATGTATTTTTCTATTATCCAATAAGTGAACTATGCTATTCCTTTCCAGTCTCCCAAGCACT
TCTTGTCCCCATCACCACTTCGGTGCTCGAAGAAAAAGTAAGCAAATCAAGGAACACAAGCTAAAGAAAC
ACACACACAAACCAAAGACAACTACAGCGTCTGCAAAAGTTTGCTAGAAGACTGAAACTGTTGAGTATAA
GGATCTGGTATTCTACGATCATGAGTTCACTTCAGAGTTTGTTCAAGACATACGTTTCGTAAGGAAACAT
CTTAGTTAGAAGTTATTCAGCAGTAGGTACCATCCCTAAGTATTTTTCACCAAATCCGTGACAATAAAGA
GCTATCTAACCAGAAAAATTAGCGAGTACGGGCACCATCCATAGGGCTTTGTCTTTACGCTTCATTAGCA
CTTACCATGCCTTACAATGTCTAGGATTGACCCTGATAGCATTTCGAAAACAAGCTAATGCTTTGTCCAG
TTCTTCAGTGAAGACAACTCACGCCCTAATGCGCTATAGGCATAAGCATCATTTGGATCCACTTCGAGAG
TTCTCTGGAAGAATTGAATCGCAATATCGTGTTCCCGTTTGCAGACCGAAACAGTTTCCCTGCAGCACAC
CAGGCCTCTGGCTGGCGAATTTTTATCCATGTCTGTGAAGTCTTTGGACAGAACTGAAAGAGCAACCTCT
TTCGGAGGATGCCAAAGTGTTGTAGAGTAGATCTCCATGCCTTCGACTCTGTAATTCTCAATCCTCCTAA
CCTCTGAGAATTGTCTTTCAGCTTGCGTGGACTCTGAAAGTTTACAATAGGCCNTTTCCGATTTGGCACA
GTACCCAACCGGTATTGCAGTGGTGAGAAGCTAGATGGCTCAAGATGCTGATAGCTTCTTTGCCGTGGTA
AGAACACAAAGCTAAATAACCTTTCCCCCTTTCACGAAGAAGGCTCATCAAGCCTTCCGCTGCTGCTTTT
TGTAGATTAAAAGCCTGAATCTGAGGCGCGATTGCGGCTATTTTCCCTTCTGAAATGACGGAAGAGTCCA
ATTTTGTCACTTCCAGGCTATCACTTATGTTCGGTGGAGTTATTGCTCCTTTATTAGTTTTACTTTTGGT
TCTTCTGTTTGGGATTTTAGGTGGAAACTTCATTTTTAATTTTCTCCTAATTCTCCTCGGTTGTGGAGCT
GTCACTAGTCAAGAGTCGTGAATTTCTTCGAGGNCGGTGCATTTGGGGGAGATGCCATAGTGGGGCTCAA
TACCTGAGGTGTTGCCCTTGTCGGCGGACCAGAACTTTGTGTTTTTGCAAGGACTGGAGTTACCTTTCGG
CTCTTTCCCCTCTGCGAGAAGACAGACGGTGTTCCGGTTTGGCCGATTCTGGCAACAGGCTTTTCTGAAG
GGGCTCCGGTGGATGGCACGTCAGTGACAGACGGTGTCTCATACCAGTGCAGTTTTGTCAATAGGGTCCG
TCTCCGGGACTTGGGGTTTCTAATGGCAAAATGCCAACACTTGGGGTTAATGGACTAACAGCTGCTGGTC
CTCCTAATAAACTTCGACCAGTTTTTGGTTTATGTTGAACCTGTTTAGATCATATGGAAGTTCCTGTTCC
CAGTGGGACAGTATCAGGTGAAAGGACAGCTGAATCGATAGAAGACACTGGGGAGTCTGTATTCAAGGAG
TACTTTGAATTGGAAGATTCTAAATTCCATCCGTTTCATTCGACGGTGTCCTGGGGTGTTTCCGTAAGAA
CGGTCTCGGGCTGTCTGTGACATAAACTAGGACGAGGTCCAAGTGTTGTGGCGCAACACTTGGACAGGCA
GTTGCTAAAGCTCTCTAGAGAGGTGAATCAAAATGTTTGGTCAGGATCTGGCTTTTCCCCCCTATTTCAC
ATCATGATTCAAAGGGACACCAGAGGAAAGGATTTCAACGAAGGCTCTTTTGGTCACATTCTGATCCTTT
GGTAAGCCGATCTGTCTTGCAATATACATGTCCCGACGATGGAAGGGGAAAGCGAGCTGAATCACCAAAC
TCAGGAACGATAATATCATCGTGGCTTTTCTGCTTATGAAACACTCCACCCGATAAGATTTGATCCCCTT
CTGCAAGCTTGCTGAGATCAACACAACATTTCGCAAGCAGGCATTTGCATTGCGGGGTAGTACAACTGTG
TCCTTTCAAGAGTCTATATGTTTTATAGGCCTTTCCTGAGCGGTAAGAACAGGTCGCCAGTAAGAACAAG
GCTTCTTCTGAGTGTACTTCTGCATAAAGGCGTTCTGCGGGGGAAACCGCATCTCGGTAGGCATAGTGGT
TTAGTGCTTGCCATATAGCAGCCTGGACGGGTCCCTGCAGCACCGCCATCCTCGAGGCTCAGGCCCACTT
TCTGCAGTGCCACAGGCACCCCCCCCCCCCCATAGCGGCTCCGGCCCGGCCAGCCCCGGCTCATTTAAAG
GCACCAGCCGCCGTTACCGGGGGATGGGGGAGTCCGAGACAGAATGACTTCTTTATCCTGCTGACTCTGG
AAAGCCCGGCGCCTTGTGATCCATTGCAAACCGAGAGTCACCTCGTGTTTAGAACACGGATCCACTCCCA
AGTTCAGTGGGGGGATGTGAGGGGTGTGGCAGGTAGGACGAAGGACTCTCTTCCTTCTGATTCGGTCTGC
ACAGTGGGGCCTAGGGCTGGAGCTCTCTCCGTGCGGACCGCTGACTCCCTCTACCTTGGGTTCCCTCGGC
CCCACCCTGGAACGCCGGGCCTTGGCAGATTCTGGCCCTTTCTGGCCCTTCAGTCGCTGTCAGAAACCCC
ATCTCATGCTCGGATGCCCCGAGTGACTGTGGCTCGCACCTCTCCGGAAACATTGGAAATCTCTCCTCTA
CGCGCGGCCACCTGAAACCACAGGAGCTCGGGACACACGTGCTTTCGGGAGAGAATGCTGAGAGTCTCTC
GCCGACTCTCTCTTGACTTGAGTTCTTCGTGGGTGCGTGGTTAAGACGTAGTGAGACCAGATGTATTAAC
TCAGGCCGGGTGCTGGTGGCTCACGCCTGTAACCCCAACACTTTGGGAGGCCGAGGCCGTAGGATCCCTC
GAGGAATCGCCTAACCCTGGGGAGGTTGAGGTTGCAGTGAGTGAGCCATAGTTGTGTCACTGTGCTCCAG
TCTGGGCGAAAGACAGAATGAGGCCCTGCCACAGGCAGGCAGGCAGGCAGGCAGGCAGAAAGACAACAGC
TGTATTATGTTCTTCTCAGGGTAGGAAGCAAAAATAACAGAATACAGCACTTAATTAATTTTTTTTTTTT
CCTTCGGACGGAGTTTCACTCTTGGTGCCCACGCTGGAGTGCAGTGGCACCATCTCGGCTCACCGCAACC
TCCACCTCCCGCGTTCAAGCGATTCTCCTGCCTCAGCCTCCTGAGTAGCTGGGATTACAGGGAGGAGCCA
CCACACCCAGCTGATTTTGTATTGTTAGTAGAGACGGCATTTCTCCATGTGGGTCAGGCTGGTCTCGAAC
TGGCGACCCCAGTGGATCTGCCCGCCCCGGCCTCCCAAAGTGCTGGGGTGACAGGCGTGAGCCATCGTGA
CTGGCCGGCTACGTTTATTTATTTATTTTTTTAATTATTTTACTTTTTTTTAGTTTTCCATTTTAATCTA
TTTATTTATTTACATTTATTTATTTATTTATTTATTTACTTATTTATTTATTTTCGAGACAGACTCTCGC
TCTGCTGCCCAGGCTGGAGTGCAGCGGCGTGATCTCGGCTCACTGCAACGTCCGCCTCCCGGGTTCACGC
CATTCTCCTGCCTCAGCCTCCCAAGTAGCTGGGACTACAGGCGCCCGCCACCGTGCCCGGCTAACTTTTT
GTATTTTGAGTAGAGATGGGGTTTCACTGTGGTAGCCAGGATGGTCTCGATCTCCTGACCCCGTGATCCG
TCCACCTCGGCCTCCCAAAGTGCTGGGATGACAGGCGTGAGCCACCGGCCCCGGCCTATTTATCTATTTA
TTAACTTTGAGTCCAGGTTATGAAACCAGTTAGTTTTTGTAATTTTTTTTTTTTTTTTTTTTTTTTGAGA
CGAGGTTTCACCGTGTTGCCAAGGCTTGGACCGAGGGATCCACCGGCCCTCGGCCTCCCAAAAGTGCGGG
GATGACAGGCGCGAGCCTACCGCGCCCGGACCCCCCCTTTCCCCTTCCCCCGCTTGTCTTCCCGACAGAC
AGTTTCACGGCAGAGCGTTTGGCTGGCGTGCTTAAACTCATTCTAAATAGAAATTTGGGACGTCAGCTTC
TGGCCTCACGGACTCTGAGCCGAGGAGTCCCCTGGTCTGTCTATCACAGGACCGTACACGTAAGGAGGAG
AAAAATCGTAACGTTCAAAGTCAGTCATTTTGTGATACAGAAATACACGGATTCACCCAAAACACAGAAA
CCAGTCTTTTAGAAATGGCCTTAGCCCTGGTGTCCGTGCCAGTGATTCTTTTCGGTTTGGACCTTGACTG
AGAGGATTCCCAGTCGGTCTCTCGTCTCTGGACGGAAGTTCCAGATGATCCGATGGGTGGGGGACTTAGG
CTGCGTCCCCCCAGGAGCCCTGGTCGATTAGTTGTGGGGATCGCCTTGGAGGGCGCGGTGACCCACTGTG
CTGTGGGAGCCTCCATCCTTCCCCCCACCCCCTCCCCAGGGGGATCCCAATTCATTCCGGGCTGACACGC
TCACTGGCAGGCGTCGGGCATCACCTAGCGGTCACTGTTACTCTGAAAACGGAGGCCTCACAGAGGAAGG
GAGCACCAGGCCGCCTGCGCACAGCCTGGGGCAACTGTGTCTTCTCCACCGCCCCCGCCCCCACCTCCAA
GTTCCTCCCTCCCTTGTTGCCTAGGAAATCGCCACTTTGACGACCGGGTCTGATTGACCTTTGATCAGGC
AAAAACGAACAAACAGATAAATAAATAAAATAACACAAAAGTAACTAACTAAATAAAATAAGTCAATACA
ACCCATTACAATACAATAAGATACGATACGATAGGATGCGATAGGATACGATAGGATACAATACAATAGG
ATACGATACAATACAATACAATACAATACAATACAATACAATACAATACAATACAATACAATACAATACG
CCGGGCGCGGTGGCTCATGCCTGTCATCCCGTCACTTTGGGATGCCGAGGTGGACGCATCACCTGAAGTC
GGGAGTTGGAGACAAGCCCGACCAACATGGAGAAATCCCGTCTCAATTGAAAATACAAAACTAGCCGGGC
GCGGTGGCACATGCCTATAATCCCAGCTGCTAGGAAGGCTGAGGCAGGAGAATCGCTTGAACCTGGGAAG
CGGAGGTTGCAGTGAGCCGAGATTGCGCCATCGCACTCCAGTCTGAGCAACAAGAGCGAAACTCCGTCTC
AAAAATAAATACATAAATAAATACATACATACATACATACATACATACATACATACATACATAAATTAAA
ATAAATAAATAAAATAAAATAAATAAATGGGCCCTGCGCGGTGGCTCAAGCCTGTCATCCCCTCACTTTG
GGAGGCCAAGGCCGGTGGATCAAGAGGCGGTCAGACCAACAGGGCCAGTATGGTGAAACCCCGTCTCTAC
TCACAATACACAACATTAGCCGGGCGCTGTGCTGTGCTGTACTGTCTGTAATCCCAGCTACTCGGGAGGC
CGAGCTGAGGCAGGAGAATCGCTTGAACCTGGGAGGCGGAGGTTGCAGTGAGCCGAGATCGCGCCACTGC
AACCCAGCCTGGGCGACAGAGCGAGACTCCGTCTCCAAAAAATGAAAATGAAAATGAAACGCAACAAAAT
AATTAAAAAGTGAGTTTCTGGGGAAAAAGAAGAAAAGAAAAAAGAAAAAAACAACAAAACAGAACAACCC
CACCGTGACATACACGTACGCTTCTCGCCTTTCGAGGCCTCAAACACGTTAGGAATTATGCGTGATTTCT
TTTTTTAACTTCATTTTATGTTATTATCATGATTGATGTTTCGAGACGGAGTCTCGGAGGCCCGCCCTCC
CTGGTTGCCCAGACAACCCCGGGAGACAGACCCTGGCTGGGCCCGATTGTTCTTCTCCTTGGTCAGGGGT
TTCCTTGTCTTTCTTCGTGTCTTTAACCCGCGTGGACTCTTCCGCCTCGGGTTTGACAGATGGCAGCTCC
ACTTTAGGCCTTGTTGTTGTTGGGGACTTTCCTGATTCTCCCCAGATGTAGTGAAAGCAGGTAGATTGCC
TTGCCTGGCCTTGCCTGGCCTTGCCTTTTCTTTCTTTCTTTCTTTCTTTATTACTTTCTCTTTTTCTTCT
TCTTCTTCTTCTTTTTTTTGAGACAGAGTTTCACTCTTGTTGCCCAGGCTAGAGGGCAATGGCGCGATCT
CGGCTCACCGCACCCTCCGCCTCCCAGGTTCAAGCGATTCTCCTGCCTCAGCCTCCTGATTAGCTGGGAT
TACAGGCATGGGCCACCGTGCTGGCTGATGTTTGTACTTTTAGTAGAGACGGTGTTTTTCCATGTTGGTC
AGGCTGGTCTCCCACTCCCAACCTCAGGTGGTCCGCCTGCCTTAGCCTCCCAAAGTGCTGGGATGACAGG
CGTGCAACCGCGCCCAGCCTCTCTCTCTCTCTCTCTCTCTCTCGCTCGCTTGCTTGCTTGCTTTCGTGCT
TTCTTGCTTTCCCGTTTTCTTGCTTTCTTTCTTTCTTTCGTTTCTTTCATGCTTGCTTTCTTGCTTGCTT
GCTTGCTTTCGTGCTTTCTTGCTTTCCTGTTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTTGTTTCTTT
CTTGCTTGCTTTCTTGCTTGCTTGCTTGCTTTCGTGCTTTCTTGCTTTCCTGTTTTCTTTCTTTCTTTCT
TTCTTTTCTTTCTTTCTTGCTTGCTTTCCTGCTTGCTTGCTTTCGTGCTTTCTTGTTTTCTCGATTTCTT
TCTTTCTTTTGTTTCTTTCCTGCTTGCTTTCTTGCTTGCTTGCTTTCGTGCTTCTTGCTTTCCTGTTTTC
TTTCTTTCTTTCTTTCTTTTGTTTCTTTCTTGCTTGCTTTCTTGCTTGCTTGCTTTCGTGCTGTCTTGTT
TCTCGATTTCTTTCTTTCTTTTGTTTCTTTCCTGCTTGCTTTCTTGCTTGATTGCTTTCGTGCTTTCTTG
CTTTCTTGTTTTCTTTCTTTCTTTTGTTTCTTTCTTTCTTGCTTCCTTGTTTTCTTGCTTTCTTGCTTGC
TTGCTTTCGTGCTTTCTTGTTTTCTTGCTTTCTTTCTTTTGTTTCTTTCTTGCTTGCTTTCTTGCTTCCT
TGTTTTCTTGCTTTCTTGCTTGCTTGCTTTCGTGCTTTCTTTCTTGCTTTCTTTTCTTTCTTTCTTTTCT
TTTTCTTTCTTTCTTGCTTTCTTTTCTTTCATCATCATCTTTCTTTCTTTCCTTTCTTTCTTTCTTTCTT
TCTATCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTGTTTCGTCCTTTTGAGACAGAGT
TTCACTCTTGTTTCCACGGCTAGAGTGCAATGGCGCGATCTTGGCTCACCGCACCTTCCGCCTCCCGGGT
TCGAGCGCTTCTCCTGCCTCCAGCCTCCCGATTAGCGGGGATTGACAGGGAGGCACCCCCACGCCTGGCT
TGGCTGATGTTTGTGTTTTTAGTAGGCACGCCGTGTCTCTCCATGTTGCTCAGGCTGGTCTCCAACTCCC
GACCTCCTGTGATGCGCCCACCTCGGCCTCTCGAAGTGCTGGGATGACGGGCGTGACGACCGTGCCCGGC
CTGTTGACTCATTTCGCTTTTTTATTTCTTTCGTTTCCACGCGTTTACTTATATGTATTAATGTAAACGT
TTCTGTACGCTTATATGCAAACAACGACAACGTGTATCTCTGCATTGAATACTCTTGCGTATGGTAAATA
CGTATCGGTTGTATGGAAATAGACTTCTGTATGATAGATGTAGGTGTCTGTGTTATACAAATAAATACAC
ATCGCTCTATAAAGAAGGGATCGTCGATAAAGACGTTTATTTTACGTATGAAAAGCGTCGTATTTATGTG
TGTAAATGAACCGAGCGTACGTAGTTATCTCTGTTTTCTTTCTTCCTCTCCTTCGTGTTTTTCTTCCTTC
CTTTCTTCCTTTCTCTCCTTCTTTAGGTTTTTCTTCCTCTCTTCCTTTCCTTCTTTCTCTCTTTCTGTCC
TTTTTTCCTTCGTGCTTTATTTCTCTTTCGTTCCCTGTGTTTCCTTCTTTTTTCTTTCCTCTCTGTTTCT
TTTTCCCTTCTTTCCTTCGTTTCTTTCCTCATTCTTTCTCTCTTTTTCGTTGTTTCTTTCCTTCCCGTCT
GTCTTTTAAAAAATTGGAGTGTTTCAGAAGTTTACTTTGTGTATCTACGTTTTCTAAATTGTCTCTCTTT
TCTCCATTTTCTTCCTCCCTCCCTCCCTCCCTCCCTGCTCCCTTCCCTCCCTCCTTCCCTTTCGCCATCT
GTCTCTTTTCCCCACTCCCCTCCCCCCGTCTGTCTCTGCGTGGATTCCGGAAGAGCCTACCGATTCTGCC
TCTCCGTGTGTCTGCAGCGACCCCGCGACCGAGTCCTTGTGTGTTCTTTCTCCCTCCCTCCCTCCCTCCC
TCCCTCCCTCCCTCCCTGCTTCCGAGAGGCATCTCCAGAGACCGCGCCGTGGGTTGTCTTCTGACTCTGT
CGCGGTCGAGGCAGAGACGCGTTTTGGGCACCGTTTGTGTGGGGTTGGGGCAGAGGGGCTGCGTTTTCGG
CCTCGGGAAGAGCTTCTCGACTCACGGTTTCGCTTTCGCGGTCCACGGGCCGCCCTGCCAGCCGGATCTG
TCTCGCTGACGTCCGCGGCGGTTGTCGGGCTCCATCTGGCGGCCGCTTTGAGATCGTGCTCTCGGCTTCC
GGAGCTGCGGTGGCAGCTGCCGAGGGAGGGGACCGTCCCCGCTGTGAGCTAGGCAGAGCTCCGGAAAGCC
CGCGGTCGTCAGCCCGGCTGGCCCGGTGGCGCCAGAGCTGTGGCCGGTCGCTTGTGAGTCACAGCTCTGG
CGTGCAGGTTTATGTGGGGGAGAGGCTGTCGCTGCGCTTCTGGGCCCGCGGCGGGCGTGGGGCTGCCCGG
GCCGGTCGACCAGCGCGCCGTAGCTCCCGAGGCCCGAGCCGCGACCCGGCGGACCCGCCGCGCGTGGCGG
AGGCTGGGGACGCCCTTCCCGGCCCGGTCGCGGTCCGCTCATCCTGGCCGTCTGAGGCGGCGGCCGAATT
CGTTTCCGAGATCCCCGTGGGGAGCCGGGGACCGTCCCGCCCCCGTCCCCCGGGTGCCGGGGAGCGGTCC
CCGGGCCGGGCCGCGGTCCCTCTGCCGCGATCCTTTCTGGCGAGTCCCCGTGGCCAGTCGGAGAGCGCTC
CCTGAGCCGGTGCGGCCCGAGAGGTCGCGCTGGCCGGCCTTCGGTCCCTCGTGTGTCCCGGTCGTAGGAG
GGGCCGGCCGAAAATGCTTCCGGCTCCCGCTCTGGAGACACGGGCCGGCCCCTGCGTGTGGCCAGGGCGG
CCGGGAGGGCTCCCCGGCCCGGCGCTGTCCCCGCGTGTGTCCTTGGGTTGACCAGAGGGACCCCGGGCGC
TCCGTGTGTGGCTGCGATGGTGGCGTTTTTGGGGACAGGTGTCCGTGTCCGTGTCGCGCGTCGCCTGGGC
CGGCGGCGTGGTCGGTGACGCGACCTCCCGGCCCCGGGGGAGGTATATCTTTCGCTCCGAGTCGGCAATT
TTGGGCCGCCGGGTTATAT
>gi|374429547|ref|NR_046235.1| Homo sapiens RNA, 45S pre-ribosomal 5 (RNA45S5), ribosomal RNA
GCTGACACGCTGTCCTCTGGCGACCTGTCGCTGGAGAGGTTGGGCCTCCGGATGCGCGCGGGGCTCTGGC
CTACCGGTGACCCGGCTAGCCGGCCGCGCTCCTGCTTGAGCCGCCTGCCGGGGCCCGCGGGCCTGCTGTT
CTCTCGCGCGTCCGAGCGTCCCGACTCCCGGTGCCGGCCCGGGTCCGGGTCTCTGACCCACCCGGGGGCG
GCGGGGAAGGCGGCGAGGGCCACCGTGCCCCCGTGCGCTCTCCGCTGCGGGCGCCCGGGGCGGCCGCGAC
AACCCCACCCCGCTGGCTCCGTGCCGTGCGTGTCAGGCGTTCTCGTCTCCGCGGGGTTGTCCGCCGCCCC
TTCCCCGGAGTGGGGGGTTGGCCGGAGCCGATCGGCTCGCTGGCCGGCCGGCCGGCCTCCGCTCCCGGGG
GGCTCTTCGTGATCGATGTGGTGACGTCGTGCTCTCCCGGGCCGGGTCCGAGCCGCGACGGGCGAGGGGC
GGACGTTCGTGGCGAACGGGACCGTCCTTCTCGCTCCGCCCCGCGGGGGTCCCCTCGTCTCTCCTCTCCC
CGCCCGCCGGCGGTGCGTGTGGGAAGGCGTGGGGTGCGGACCCCGGCCCGACCTCGCCGTCCCGCCCGCC
GCCTTCTGCGTCGCGGGGCGGGCCGGCGGGGTCCTCTGACGCGGCAGACAGCCCTCGCTGTCGCCTCCAG
TGGTTGTCGACTTGCGGGCGGCCCCCCTCCGCGGCGGTGGGGGTGCCGTCCCGCCGGCCCGTCGTGCTGC
CCTCTCGGGGGGTTTGCGCGAGCGTCGGCTCCGCCTGGGCCCTTGCGGTGCTCCTGGAGCGCTCCGGGTT
GTCCCTCAGGTGCCCGAGGCCGAACGGTGGTGTGTCGTTCCCGCCCCCGGCGCCCCCTCCTCCGGTCGCC
GCCGCGGTGTCCGCGCGTGGGTCCTGAGGGAGCTCGTCGGTGTGGGGTTCGAGGCGGTTTGAGTGAGACG
AGACGAGACGCGCCCCTCCCACGCGGGGAAGGGCGCCCGCCTGCTCTCGGTGAGCGCACGTCCCGTGCTC
CCCTCTGGCGGGTGCGCGCGGGCCGTGTGAGCGATCGCGGTGGGTTCGGGCCGGTGTGACGCGTGCGCCG
GCCGGCCGCCGAGGGGCTGCCGTTCTGCCTCCGACCGGTCGTGTGTGGGTTGACTTCGGAGGCGCTCTGC
CTCGGAAGGAAGGAGGTGGGTGGACGGGGGGGCCTGGTGGGGTTGCGCGCACGCGCGCACCGGCCGGGCC
CCCGCCCTGAACGCGAACGCTCGAGGTGGCCGCGCGCAGGTGTTTCCTCGTACCGCAGGGCCCCCTCCCT
TCCCCAGGCGTCCCTCGGCGCCTCTGCGGGCCCGAGGAGGAGCGGCTGGCGGGTGGGGGGAGTGTGACCC
ACCCTCGGTGAGAAAAGCCTTCTCTAGCGATCTGAGAGGCGTGCCTTGGGGGTACCGGATCCCCCGGGCC
GCCGCCTCTGTCTCTGCCTCCGTTATGGTAGCGCTGCCGTAGCGACCCGCTCGCAGAGGACCCTCCTCCG
CTTCCCCCTCGACGGGGTTGGGGGGGAGAAGCGAGGGTTCCGCCGGCCACCGCGGTGGTGGCCGAGTGCG
GCTCGTCGCCTACTGTGGCCCGCGCCTCCCCCTTCCGAGTCGGGGGAGGATCCCGCCGGGCCGGGCCCGG
CGTTCCCAGCGGGTTGGGACGCGGCGGCCGGCGGGCGGTGGGTGTGCGCGCCCGGCGCTCTGTCCGGCGC
GTGACCCCCTCCGCCGCGAGTCGGCTCTCCGCCCGCTCCCGTGCCGAGTCGTGACCGGTGCCGACGACCG
CGTTTGCGTGGCACGGGGTCGGGCCCGCCTGGCCCTGGGAAAGCGTCCCACGGTGGGGGCGCGCCGGTCT
CCCGGAGCGGGACCGGGTCGGAGGATGGACGAGAATCACGAGCGACGGTGGTGCGGGCGTGTCGGGTTCG
TGGCTGCGGTCGCTCCGGGGCCCCCGGTGGCGGGGCCCCGGGGCTCGCGAGGCGGTTCTCGGTGGGGGCC
GAGGGCCGTCCGGCGTCCCAGGCGGGGCGCCGCGGGACCGCCCTCGTGTCTGTGGCGGTGGGATCCCGCG
GCCGTGTTTTCCTGGTGGCCCGGCCGTGCCTGAGGTTTCTCCCCGAGCCGCCGCCTCTGCGGGCTCCCGG
GTGCCCTTGCCCTCGCGGTCCCCGGCCCTCGCCCGTCTGTGCCCTCTTCCCCGCCCGCCGCCCGCCGATC
CTCTTCTTCCCCCCGAGCGGCTCACCGGCTTCACGTCCGTTGGTGGCCCCGCCTGGGACCGAACCCGGCA
CCGCCTCGTGGGGCGCCGCCGCCGGCCACTGATCGGCCCGGCGTCCGCGTCCCCCGGCGCGCGCCTTGGG
GACCGGGTCGGTGGCGCCCCGCGTGGGGCCCGGTGGGCTTCCCGGAGGGTTCCGGGGGTCGGCCTGCGGC
GCGTGCGGGGGAGGAGACGGTTCCGGGGGACCGGCCGCGACTGCGGCGGCGGTGGTGGGGGCAGCCGCGG
GGATCGCCGAGGGCCGGTCGGCCGCCCCGGGTGCCGCGCGGTGCCGCCGGCGGCGGTGAGGCCCCGCGCG
TGTGTCCCGGCCGCGGTCGGCCGCGCTCGAGGGGTCCCCGTGGCGTCCCCTTCCCCGCCGGCCGCCTTTC
TCGCGCCTTCCCCGTCGCCCCGGCCTCGCCCGTGGTCTCTCGTCTTCTCCCGGCCCGCTCTTCCGAACCG
GGTCGGCGCGTCCCCCGGGTGCGCCTCGCTTCCCGGGCCTGCCGCGGCCCTTCCCCGAGGCGTCCGTCCC
GGGCGTCGGCGTCGGGGAGAGCCCGTCCTCCCCGCGTGGCGTCGCCCCGTTCGGCGCGCGCGTGCGCCCG
AGCGCGGCCCGGTGGTCCCTGCCGGACAGGCGTTCGTGCGACGTGTGGCGTGGGTCGACCTCCGCCTTGC
CGGTCGCTCGCCCTTTCCCCGGGTCGGGGGGTGGGGCCCGGGCCGGGGCCTCGGCCCCGGTCGCGGTCCC
CCGTCCCGGGCGGGGGCGGGCGCGCCGGCCGGCCTCGGTCGGCCCTCCCTTGGCCGTCGTGTGGCGTGTG
CCACCCCTGCGCCCGCGCCCGCCGGCGGGGCTCGGAGCCGGGCTTCGGCCGGGCCCCGGGCCCTCGACCG
GACCGGTGCGCGGGCGCTGCGGCCGCACGGCGCGACTGTCCCCGGGCCGGGCACCGCGGTCCGCCTCTCG
CTCGCCGCCCGGACGTCGGGGCCGCCCCGCGGGGCGGGCGGAGCGCCGTCCCCGCCTCGCCGCCGCCCGC
GGGCGCCGGCCGCGCGCGCGCGCGCGTGGCCGCCGGTCCCTCCCGGCCGCCGGGCGCGGGTCGGGCCGTC
CGCCTCCTCGCGGGCGGGCGCGACGAAGAAGCGTCGCGGGTCTGTGGCGCGGGGCCCCGGTGGTCGTGTC
GCGTGGGGGGCGGGTGGTTGGGGCGTCCGGTTCGCCGCGCCCCGCCCCGGCCCCACCGGTCCCGGCCGCC
GCCCCCGCGCCCGCTCGCTCCCTCCCGTCCGCCCGTCCGCGGCCCGTCCGTCCGTCCGTCGTCCTCCTCG
CTTGCGGGGCGCCGGGCCCGTCCTCGCGAGGCCCCCCGGCCGGCCGTCCGGCCGCGTCGGGGCCTCGCCG
CGCTCTACCTTACCTACCTGGTTGATCCTGCCAGTAGCATATGCTTGTCTCAAAGATTAAGCCATGCATG
TCTGAGTACGCACGGCCGGTACAGTGAAACTGCGAATGGCTCATTAAATCAGTTATGGTTCCTTTGGTCG
CTCGCTCCTCTCCTACTTGGATAACTGTGGTAATTCTAGAGCTAATACATGCCGACGGGCGCTGACCCCC
TTCGCGGGGGGGATGCGTGCATTTATCAGATCAAAACCAACCCGGTCAGCCCCTCTCCGGCCCCGGCCGG
GGGGCGGGCGCCGGCGGCTTTGGTGACTCTAGATAACCTCGGGCCGATCGCACGCCCCCCGTGGCGGCGA
CGACCCATTCGAACGTCTGCCCTATCAACTTTCGATGGTAGTCGCCGTGCCTACCATGGTGACCACGGGT
GACGGGGAATCAGGGTTCGATTCCGGAGAGGGAGCCTGAGAAACGGCTACCACATCCAAGGAAGGCAGCA
GGCGCGCAAATTACCCACTCCCGACCCGGGGAGGTAGTGACGAAAAATAACAATACAGGACTCTTTCGAG
GCCCTGTAATTGGAATGAGTCCACTTTAAATCCTTTAACGAGGATCCATTGGAGGGCAAGTCTGGTGCCA
GCAGCCGCGGTAATTCCAGCTCCAATAGCGTATATTAAAGTTGCTGCAGTTAAAAAGCTCGTAGTTGGAT
CTTGGGAGCGGGCGGGCGGTCCGCCGCGAGGCGAGCCACCGCCCGTCCCCGCCCCTTGCCTCTCGGCGCC
CCCTCGATGCTCTTAGCTGAGTGTCCCGCGGGGCCCGAAGCGTTTACTTTGAAAAAATTAGAGTGTTCAA
AGCAGGCCCGAGCCGCCTGGATACCGCAGCTAGGAATAATGGAATAGGACCGCGGTTCTATTTTGTTGGT
TTTCGGAACTGAGGCCATGATTAAGAGGGACGGCCGGGGGCATTCGTATTGCGCCGCTAGAGGTGAAATT
CTTGGACCGGCGCAAGACGGACCAGAGCGAAAGCATTTGCCAAGAATGTTTTCATTAATCAAGAACGAAA
GTCGGAGGTTCGAAGACGATCAGATACCGTCGTAGTTCCGACCATAAACGATGCCGACCGGCGATGCGGC
GGCGTTATTCCCATGACCCGCCGGGCAGCTTCCGGGAAACCAAAGTCTTTGGGTTCCGGGGGGAGTATGG
TTGCAAAGCTGAAACTTAAAGGAATTGACGGAAGGGCACCACCAGGAGTGGAGCCTGCGGCTTAATTTGA
CTCAACACGGGAAACCTCACCCGGCCCGGACACGGACAGGATTGACAGATTGATAGCTCTTTCTCGATTC
CGTGGGTGGTGGTGCATGGCCGTTCTTAGTTGGTGGAGCGATTTGTCTGGTTAATTCCGATAACGAACGA
GACTCTGGCATGCTAACTAGTTACGCGACCCCCGAGCGGTCGGCGTCCCCCAACTTCTTAGAGGGACAAG
TGGCGTTCAGCCACCCGAGATTGAGCAATAACAGGTCTGTGATGCCCTTAGATGTCCGGGGCTGCACGCG
CGCTACACTGACTGGCTCAGCGTGTGCCTACCCTACGCCGGCAGGCGCGGGTAACCCGTTGAACCCCATT
CGTGATGGGGATCGGGGATTGCAATTATTCCCCATGAACGAGGAATTCCCAGTAAGTGCGGGTCATAAGC
TTGCGTTGATTAAGTCCCTGCCCTTTGTACACACCGCCCGTCGCTACTACCGATTGGATGGTTTAGTGAG
GCCCTCGGATCGGCCCCGCCGGGGTCGGCCCACGGCCCTGGCGGAGCGCTGAGAAGACGGTCGAACTTGA
CTATCTAGAGGAAGTAAAAGTCGTAACAAGGTTTCCGTAGGTGAACCTGCGGAAGGATCATTAACGGAGC
CCGGAGGGCGAGGCCCGCGGCGGCGCCGCCGCCGCCGCGCGCTTCCCTCCGCACACCCACCCCCCCACCG
CGACGCGGCGCGTGCGCGGGCGGGGCCCGCGTGCCCGTTCGTTCGCTCGCTCGTTCGTTCGCCGCCCGGC
CCCGCCGGCCGCGAGAGCCGGAGAACTCGGGAGGGAGACGGGGGAGAGAGAGAGAGAGAGAGAAAGAGAA
AGAAGGGCGTGTCGTTGGTGTGCGCGTGTCGTGGGGCCGGCGGGCGGCGGGGAGCGGTCCCCGGCCGCGG
CCCCGACGACGTGGGTGTCGGCGGGCGCGGGGGCGGTTCTCGGCGGCGTCGCGGCGGGTCTGGGGGGGTC
TCGGTGCCCTCCTCCCCGCCGGGGCCCGTCGTCCGGCCCCGCCGCGCCGGCTCCCCGTCTTCGGGGCCGG
CCGGATTCCCGTCGCCTCCGCCGCGCCGCTCCGCGCCGCCGGGCACGGCCCCGCTCGCTCTCCCCGGCCT
TCCCGCTAGGGCGTCTCGAGGGTCGGGGGCCGGACGCCGGTCCCCTCCCCCGCCTCCTCGTCCGCCCCCC
CGCCGTCCAGGTACCTAGCGCGTTCCGGCGCGGAGGTTTAAAGACCCCTTGGGGGGATCGCCCGTCCGCC
CGTGGGTCGGGGGCGGTGGTGGGCCCGCGGGGGAGTCCCGTCGGGAGGGGCCCGGCCCCTCCCGCGCCTC
CACCGCGGACTCCGCTCCCCGGCCGGGGCCGCGCCGCCGCCGCCGCCGCGGCGGCCGTCGGGTGGGGGCT
TTACCCGGCGGCCGTCGCGCGCCTGCCGCGCGTGTGGCGTGCGCCCCGCGCCGTGGGGGCGGGAACCCCC
GGGCGCCTGTGGGGTGGTGTCCGCGCTCGCCCCCGCGTGGGCGGCGCGCGCCTCCCCGTGGTGTGAAACC
TTCCGACCCCTCTCCGGAGTCCGGTCCCGTTTGCTGTCTCGTCTGGCCGGCCTGAGGCAACCCCCTCTCC
TCTTGGGCGGGGGGGGGGGGGACGTGCCGCGCCAGGAAGGGCCTCCTCCCGGTGCGTCGTCGGGAGCGCC
CTCGCCAAATCGACCTCGTACGACTCTTAGCGGTGGATCACTCGGCTCGTGCGTCGATGAAGAACGCAGC
TAGCTGCGAGAATTAATGTGAATTGCAGGACACATTGATCATCGACACTTCGAACGCACTTGCGGCCCCG
GGTTCCTCCCGGGGCTACGCCTGTCTGAGCGTCGCTTGCCGATCAATCGCCCCCGGGGGTGCCTCCGGGC
TCCTCGGGGTGCGCGGCTGGGGGTTCCCTCGCAGGGCCCGCCGGGGGCCCTCCGTCCCCCTAAGCGCAGA
CCCGGCGGCGTCCGCCCTCCTCTTGCCGCCGCGCCCGCCCCTTCCCCCTCCCCCCGCGGGCCCTGCGTGG
TCACGCGTCGGGTGGCGGGGGGGAGAGGGGGGCGCGCCCGGCTGAGAGAGACGGGGAGGGCGGCGCCGCC
GCCGCCCGCGAAGACGGAGAGGGAAAGAGAGAGCCGGCTCGGGCCGAGTTCCCGTGGCCGCCGCCTGCGG
TCCGGGTTCCTCCCTCGGGGGGCTCCCTCGCGCCGCGCGCGGCTCGGGGTTCGGGGTTCGTCGGCCCCGG
CCGGGTGGAAGGTCCCGTGCCCGTCGTCGTCGTCGTCGTCGCGCGTCGTCGGCGGTGGGGGCGTGTTGCG
TGCGGTGTGGTGGTGGGGGAGGAGGAAGGCGGGTCCGGAAGGGGAAGGGTGCCGGCGGGGAGAGAGGGTC
GGGGGAGCGCGTCCCGGTCGCCGCGGTTCGCCGCCCGCCCCCGGTGGCGGCCCGGCGTCCGGCCGACCGC
CGCTCCCGCGCCCCTCCTCCTCCCCGCCGCCCCTCCTCCGAGGCCCCGCCCGTCCTCCTCGCCCTCCCCG
CGCGTACGCGCGCCCGCCCGCCCGGCTCGCCTCGCGGCGCGTCGGCCGGGGCCGGGAGCCCGCCCCGCGG
CCCGCCCGGCCGCGCCCGTGGCCGCGGCGCCGGGGTTCGCGTGTCCCCGGCGGCGACCCGCGGGACGCCG
CGGTGTCGTCCGCCGTCGCGCGCCCGCCTCCGGCTCGCGGCCGCGCCGCGCCGCGCCGGGGCCCCGTCCC
GAGCTTCCGCGTCGGGGCGGGGCGGCTCCGCCGCCGCGTCCTCGGACCCGTCCCCCCGACCTCCGCGGGG
GAGACGGGTCGGGGCGTGCGGCGCCCGTCCCGCCCCCGGCCCGTGCCCCTCCCTCCGGTCGTCCCGCTCC
GGCGGGGCGGCGCGGGGGTGCCGCCGGCCGCGCGCTCTCTCTCCCGTCGCCTCTCCCCCTCGCCGGGCCC
GTCTCCCGACGGAGCGTCGGGCGGGCGGTCGGGCCGGCGCGATTCCGTCCGTCCGTCCGCCGAGCGGCCC
GTCCCCCTCCGAGACGCGACCTCAGATCAGACGTGGCGACCCGCTGAATTTAAGCATATTAGTCAGCGGA
GGAGAAGAAACTAACCAGGATTCCCTCAGTAACGGCGAGTGAACAGGGAAGAGCCCAGCGCCGAATCCCC
GCCCCGCGGCGGGGCGCGGGACATGTGGCGTACGGAAGACCCGCTCCCCGGCGCCGCTCGTGGGGGGCCC
AAGTCCTTCTGATCGAGGCCCAGCCCGTGGACGGTGTGAGGCCGGTAGCGGCCCCCGGCGCGCCGGGCCC
GGGTCTTCCCGGAGTCGGGTTGCTTGGGAATGCAGCCCAAAGCGGGTGGTAAACTCCATCTAAGGCTAAA
TACCGGCACGAGACCGATAGTCAACAAGTACCGTAAGGGAAAGTTGAAAAGAACTTTGAAGAGAGAGTTC
AAGAGGGCGTGAAACCGTTAAGAGGTAAACGGGTGGGGTCCGCGCAGTCCGCCCGGAGGATTCAACCCGG
CGGCGGGTCCGGCCGTGTCGGCGGCCCGGCGGATCTTTCCCGCCCCCCGTTCCTCCCGACCCCTCCACCC
GCCCTCCCTTCCCCCGCCGCCCCTCCTCCTCCTCCCCGGAGGGGGCGGGCTCCGGCGGGTGCGGGGGTGG
GCGGGCGGGGCCGGGGGTGGGGTCGGCGGGGGACCGTCCCCCGACCGGCGACCGGCCGCCGCCGGGCGCA
TTTCCACCGCGGCGGTGCGCCGCGACCGGCTCCGGGACGGCTGGGAAGGCCCGGCGGGGAAGGTGGCTCG
GGGGGCCCCGTCCGTCCGTCCGTCCGTCCTCCTCCTCCCCCGTCTCCGCCCCCCGGCCCCGCGTCCTCCC
TCGGGAGGGCGCGCGGGTCGGGGCGGCGGCGGCGGCGGCGGTGGCGGCGGCGGCGGCGGCGGCGGGACCG
AAACCCCCCCCGAGTGTTACAGCCCCCCCGGCAGCAGCACTCGCCGAATCCCGGGGCCGAGGGAGCGAGA
CCCGTCGCCGCGCTCTCCCCCCTCCCGGCGCCCACCCCCGCGGGGAATCCCCCGCGAGGGGGGTCTCCCC
CGCGGGGGCGCGCCGGCGTCTCCTCGTGGGGGGGCCGGGCCACCCCTCCCACGGCGCGACCGCTCTCCCA
CCCCTCCTCCCCGCGCCCCCGCCCCGGCGACGGGGGGGGTGCCGCGCGCGGGTCGGGGGGCGGGGCGGAC
TGTCCCCAGTGCGCCCCGGGCGGGTCGCGCCGTCGGGCCCGGGGGAGGTTCTCTCGGGGCCACGCGCGCG
TCCCCCGAAGAGGGGGACGGCGGAGCGAGCGCACGGGGTCGGCGGCGACGTCGGCTACCCACCCGACCCG
TCTTGAAACACGGACCAAGGAGTCTAACACGTGCGCGAGTCGGGGGCTCGCACGAAAGCCGCCGTGGCGC
AATGAAGGTGAAGGCCGGCGCGCTCGCCGGCCGAGGTGGGATCCCGAGGCCTCTCCAGTCCGCCGAGGGC
GCACCACCGGCCCGTCTCGCCCGCCGCGCCGGGGAGGTGGAGCACGAGCGCACGTGTTAGGACCCGAAAG
ATGGTGAACTATGCCTGGGCAGGGCGAAGCCAGAGGAAACTCTGGTGGAGGTCCGTAGCGGTCCTGACGT
GCAAATCGGTCGTCCGACCTGGGTATAGGGGCGAAAGACTAATCGAACCATCTAGTAGCTGGTTCCCTCC
GAAGTTTCCCTCAGGATAGCTGGCGCTCTCGCAGACCCGACGCACCCCCGCCACGCAGTTTTATCCGGTA
AAGCGAATGATTAGAGGTCTTGGGGCCGAAACGATCTCAACCTATTCTCAAACTTTAAATGGGTAAGAAG
CCCGGCTCGCTGGCGTGGAGCCGGGCGTGGAATGCGAGTGCCTAGTGGGCCACTTTTGGTAAGCAGAACT
GGCGCTGCGGGATGAACCGAACGCCGGGTTAAGGCGCCCGATGCCGACGCTCATCAGACCCCAGAAAAGG
TGTTGGTTGATATAGACAGCAGGACGGTGGCCATGGAAGTCGGAATCCGCTAAGGAGTGTGTAACAACTC
ACCTGCCGAATCAACTAGCCCTGAAAATGGATGGCGCTGGAGCGTCGGGCCCATACCCGGCCGTCGCCGG
CAGTCGAGAGTGGACGGGAGCGGCGGGGGCGGCGCGCGCGCGCGCGCGTGTGGTGTGCGTCGGAGGGCGG
CGGCGGCGGCGGCGGCGGGGGTGTGGGGTCCTTCCCCCGCCCCCCCCCCCACGCCTCCTCCCCTCCTCCC
GCCCACGCCCCGCTCCCCGCCCCCGGAGCCCCGCGGACGCTACGCCGCGACGAGTAGGAGGGCCGCTGCG
GTGAGCCTTGAAGCCTAGGGCGCGGGCCCGGGTGGAGCCGCCGCAGGTGCAGATCTTGGTGGTAGTAGCA
AATATTCAAACGAGAACTTTGAAGGCCGAAGTGGAGAAGGGTTCCATGTGAACAGCAGTTGAACATGGGT
CAGTCGGTCCTGAGAGATGGGCGAGCGCCGTTCCGAAGGGACGGGCGATGGCCTCCGTTGCCCTCGGCCG
ATCGAAAGGGAGTCGGGTTCAGATCCCCGAATCCGGAGTGGCGGAGATGGGCGCCGCGAGGCGTCCAGTG
CGGTAACGCGACCGATCCCGGAGAAGCCGGCGGGAGCCCCGGGGAGAGTTCTCTTTTCTTTGTGAAGGGC
AGGGCGCCCTGGAATGGGTTCGCCCCGAGAGAGGGGCCCGTGCCTTGGAAAGCGTCGCGGTTCCGGCGGC
GTCCGGTGAGCTCTCGCTGGCCCTTGAAAATCCGGGGGAGAGGGTGTAAATCTCGCGCCGGGCCGTACCC
ATATCCGCAGCAGGTCTCCAAGGTGAACAGCCTCTGGCATGTTGGAACAATGTAGGTAAGGGAAGTCGGC
AAGCCGGATCCGTAACTTCGGGATAAGGATTGGCTCTAAGGGCTGGGTCGGTCGGGCTGGGGCGCGAAGC
GGGGCTGGGCGCGCGCCGCGGCTGGACGAGGCGCCGCCGCCCCCCCCACGCCCGGGGCACCCCCCTCGCG
GCCCTCCCCCGCCCCACCCCGCGCGCGCCGCTCGCTCCCTCCCCGCCCCGCGCCCTCTCTCTCTCTCTCT
CCCCCGCTCCCCGTCCTCCCCCCTCCCCGGGGGAGCGCCGCGTGGGGGCGGCGGCGGGGGGAGAAGGGTC
GGGGCGGCAGGGGCCGGCGGCGGCCCGCCGCGGGGCCCCGGCGGCGGGGGCACGGTCCCCCGCGAGGGGG
GCCCGGGCACCCGGGGGGCCGGCGGCGGCGGCGACTCTGGACGCGAGCCGGGCCCTTCCCGTGGATCGCC
CCAGCTGCGGCGGGCGTCGCGGCCGCCCCCGGGGAGCCCGGCGGGCGCCGGCGCGCCCCCCCCCCCACCC
CACGTCTCGTCGCGCGCGCGTCCGCTGGGGGCGGGGAGCGGTCGGGCGGCGGCGGTCGGCGGGCGGCGGG
GCGGGGCGGTTCGTCCCCCCGCCCTACCCCCCCGGCCCCGTCCGCCCCCCGTTCCCCCCTCCTCCTCGGC
GCGCGGCGGCGGCGGCGGCAGGCGGCGGAGGGGCCGCGGGCCGGTCCCCCCCGCCGGGTCCGCCCCCGGG
GCCGCGGTTCCGCGCGGCGCCTCGCCTCGGCCGGCGCCTAGCAGCCGACTTAGAACTGGTGCGGACCAGG
GGAATCCGACTGTTTAATTAAAACAAAGCATCGCGAAGGCCCGCGGCGGGTGTTGACGCGATGTGATTTC
TGCCCAGTGCTCTGAATGTCAAAGTGAAGAAATTCAATGAAGCGCGGGTAAACGGCGGGAGTAACTATGA
CTCTCTTAAGGTAGCCAAATGCCTCGTCATCTAATTAGTGACGCGCATGAATGGATGAACGAGATTCCCA
CTGTCCCTACCTACTATCCAGCGAAACCACAGCCAAGGGAACGGGCTTGGCGGAATCAGCGGGGAAAGAA
GACCCTGTTGAGCTTGACTCTAGTCTGGCACGGTGAAGAGACATGAGAGGTGTAGAATAAGTGGGAGGCC
CCCGGCGCCCCCCCGGTGTCCCCGCGAGGGGCCCGGGGCGGGGTCCGCCGGCCCTGCGGGCCGCCGGTGA
AATACCACTACTCTGATCGTTTTTTCACTGACCCGGTGAGGCGGGGGGGCGAGCCCCGAGGGGCTCTCGC
TTCTGGCGCCAAGCGCCCGGCCGCGCGCCGGCCGGGCGCGACCCGCTCCGGGGACAGTGCCAGGTGGGGA
GTTTGACTGGGGCGGTACACCTGTCAAACGGTAACGCAGGTGTCCTAAGGCGAGCTCAGGGAGGACAGAA
ACCTCCCGTGGAGCAGAAGGGCAAAAGCTCGCTTGATCTTGATTTTCAGTACGAATACAGACCGTGAAAG
CGGGGCCTCACGATCCTTCTGACCTTTTGGGTTTTAAGCAGGAGGTGTCAGAAAAGTTACCACAGGGATA
ACTGGCTTGTGGCGGCCAAGCGTTCATAGCGACGTCGCTTTTTGATCCTTCGATGTCGGCTCTTCCTATC
ATTGTGAAGCAGAATTCACCAAGCGTTGGATTGTTCACCCACTAATAGGGAACGTGAGCTGGGTTTAGAC
CGTCGTGAGACAGGTTAGTTTTACCCTACTGATGATGTGTTGTTGCCATGGTAATCCTGCTCAGTACGAG
AGGAACCGCAGGTTCAGACATTTGGTGTATGTGCTTGGCTGAGGAGCCAATGGGGCGAAGCTACCATCTG
TGGGATTATGACTGAACGCCTCTAAGTCAGAATCCCGCCCAGGCGGAACGATACGGCAGCGCCGCGGAGC
CTCGGTTGGCCTCGGATAGCCGGTCCCCCGCCTGTCCCCGCCGGCGGGCCGCCCCCCCCCTCCACGCGCC
CCGCGCGCGCGGGAGGGCGCGTGCCCCGCCGCGCGCCGGGACCGGGGTCCGGTGCGGAGTGCCCTTCGTC
CTGGGAAACGGGGCGCGGCCGGAGAGGCGGCCGCCCCCTCGCCCGTCACGCACCGCACGTTCGTGGGGAA
CCTGGCGCTAAACCATTCGTAGACGACCTGCTTCTGGGTCGGGGTTTCGTACGTAGCAGAGCAGCTCCCT
CGCTGCGATCTATTGAAAGTCAGCCCTCGACACAAGGGTTTGTCCGCGCGCGCGCGCGCGCGCGCGTGCG
GGGGGCCCGGCGGGGCGTGCGCGTCCGGCGCCGTCCGTCCTTCCGTTCGTCTTCCTCCCTCCCGGCCTCT
CCCGCCGACCGCGGGCGTGGTGGTGGGGGTGTGGGGGGGAGGGCGCGCGACCCCGGTCGGCGCGCCCCGC
TTCTTCGGTTCCCGCCTCCTCCCCGTTCACCGCCGGGGCGGCTCGTCCGCTCCGGGCCGGGACGGGGTCC
GGGGAGCGTGGTTTGGGAGCCGCGGAGGCGGCCGCGCCGAGCCGGGCCCGTGGCCCGCCGGTCCCCGTCC
CGGGGGTTGGCCGCGCGGGCCCCGGTGGGGCGGCCACCCGGGGTCCCGGCCCTCGCG
>poly-A
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>poly-C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
>poly-G
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
>poly-T
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
"""

    seq['mus_musculus'] = """>gi|38176281|gb|BK000964|BK000964 Mus musculus ribosomal DNA complete repeating unit
ACTGACACGCTGTCCTTTCCCTATTAACACTAAAGGACACTATAAAGAGACCCTTTCGAT
TTAAGGCTGTTTTGCTTGTCCAGCCTATTCTTTTTACTGGCTTGGGTCTGTCGCGGTGCC
TGAAGCTGTCCCCGAGCCACGCTTCCTGCTTTCCCGGGCTTGCTGCTTGCGTGTGCTTGC
TGTGGGCAGCTTGTGACAACTGGGCGCTGTGACTTTGCTGCGTGTCAGACGTTTTTCCCG
ATTTCCCCGAGGTGTCGTTGTCACACCTGTCCCGGTTGGAATGGTGGAGCCAGCTGTGGT
TGAGGGCCACCTTATTTCGGCTCACTTTTTTTTTTTTTTTTTCTCTTGGAGTCCCGAACC
TCCGCTCTTTTCTCTTCCCGGTCTTTCTTCCACATGCCTCCCGAGTGCATTTCTTTTTGT
TTTTTTTCTTTTTTTTTTTTTTTTTTTGGGGAGGTGGAGAGTCCCGAGTACTTCACTCCT
GTCTGTGGTGTCCAAGTGTTCATGCCACGTGCCTCCCGAGTGCACTTTTTTTTGTGGCAG
TCGCTCGTTGTGTTCTCTTGTTCTGTGTCTGCCCGTATCAGTAACTGTCTTGCCCCGCGT
GTAAGACATTCCTATCTCGCTTGTTTCTCCCGATTGCGCGTCGTTGCTCACTCTTAGATC
GATGTGGTGCTCCGGAGTTCTCTTCGGGCCAGGGCCAAGCCGCGCCAGGCGAGGGACGGA
CATTCATGGCGAATGGCGGCCGCTCTTCTCGTTCTGCCAGCGGGCCCTCGTCTCTCCACC
CCATCCGTCTGCCGGTGGTGTGTGGAAGGCAGGGGTGCGGCTCTCCGGCCCGACGCTGCC
CCGCGCGCACTTTTCTCAGTGGTTCGCGTGGTCCTTGTGGATGTGTGAGGCGCCCGGTTG
TGCCCTCACGTGTTTCACTTTGGTCGTGTCTCGCTTGACCATGTTCCCAGAGTCGGTGGA
TGTGGCCGGTGGCGTTGCATACCCTTCCCGTCTGGTGTGTGCACGCGCTGTTTCTTGTAA
GCGTCGAGGTGCTCCTGGAGCGTTCCAGGTTTGTCTCCTAGGTGCCTGCTTCTGAGCTGG
TGGTGGCGCTCCCCATTCCCTGGTGTGCCTCCGGTGCTCCGTCTGGCTGTGTGCCTTCCC
GTTTGTGTCTGAGAAGCCCGTGAGAGGGGGGTCGAGGAGAGAAGGAGGGGCAAGACCCCC
CTTCTTCGTCGGGTGAGGCGCCCACCCCGCGACTAGTACGCCTGTGCGTAGGGCTGGTGC
TGAGCGGTCGCGGCTGGGGTTGGAAAGTTTCTCGAGAGACTCATTGCTTTCCCGTGGGGA
GCTTTGAGAGGCCTGGCTTTCGGGGGGGACCGGTTGCAGGGTCTCCCCTGTCCGCGGATG
CTCAGAATGCCCTTGGAAGAGAACCTTCCTGTTGCCGCAGACCCCCCCGCGCGGTCGCCC
GCGTGTTGGTCTTCTGGTTTCCCTGTGTGCTCGTCGCATGCATCCTCTCTCGGTGGCCGG
GGCTCGTCGGGGTTTTGGGTCCGTCCCGCCCTCAGTGAGAAAGTTTCCTTCTCTAGCTAT
CTTCCGGAAAGGGTGCGGGCTTCTTACGGTCTCGAGGGGTCTCTCCCGAATGGTCCCCTG
GAGGGCTCGCCCCCTGACCGCCTCCCGCGCGCGCAGCGTTTGCTCTCTCGTCTACCGCGG
CCCGCGGCCTCCCCGCTCCGAGTTCGGGGAGGGATCACGCGGGGCAGAGCCTGTCTGTCG
TCCTGCCGTTGCTGCGGAGCATGTGGCTCGGCTTGTGTGGTTGGTGGCTGGGGAGAGGGC
TCCGTGCACACCCCCGCGTGCGCGTACTTTCCTCCCCTCCTGAGGGCCGCCGTGCGGACG
GGGTGTGGGTAGGCGACGGTGGGCTCCCGGGTCCCCACCCGTCTTCCCGTGCCTCACCCG
TGCCTTCCGTCGCGTGCGTCCCTCTCGCTCGCGTCCACGACTTTGGCCGCTCCCGCGACG
GCGGCCTGCGCCGCGCGTGGTGCGTGCTGTGTGCTTCTCGGGCTGTGTGGTTGTGTCGCC
TCGCCCCCCCCTTCCCGCGGCAGCGTTCCCACGGCTGGCGAAATCGCGGGAGTCCTCCTT
CCCCTCCTCGGGGTCGAGAGGGTCCGTGTCTGGCGTTGATTGATCTCGCTCTCGGGGACG
GGACCGTTCTGTGGGAGAACGGCTGTTGGCCGCGTCCGGCGCGACGTCGGACGTGGGGAC
CCACTGCCGCTCGGGGGTCTTCGTCGGTAGGCATCGGTGTGTCGGCATCGGTCTCTCTCT
CGTGTCGGTGTCGCCTCCTCGGGCTCCCGGGGGGCCGTCGTGTTTCGGGTCGGCTCGGCG
CTGCAGGTGTGGTGGGACTGCTCAGGGGAGTGGTGCAGTGTGATTCCCGCCGGTTTTGCC
TCGCGTGCCCTGACCGGTCCGACGCCCGAGCGGTCTCTCGGTCCCTTGTGAGGACCCCCT
TCCGGGAGGGGCCCGTTTCGGCCGCCCTTGCCGTCGTCGCCGGCCCTCGTTCTGCTGTGT
CGTTCCCCCCTCCCCGCTCGCCGCAGCCGGTCTTTTTTCCTCTCTCCCCCCCTCTCCTCT
GACTGACCCGTGGCCGTGCTGTCGGACCCCCCGCATGGGGGCGGCCGGGCACGTACGCGT
CCGGGCGGTCACCGGGGTCTTGGGGGGGGGCCGAGGGGTAAGAAAGTCGGCTCGGCGGGC
GGGAGGAGCTGTGGTTTGGAGGGCGTCCCGGCCCCGCGGCCGTGGCGGTGTCTTGCGCGG
TCTTGGAGAGGGCTGCGTGCGAGGGGAAAAGGTTGCCCCGCGAGGGCAAAGGGAAAGAGG
CTAGCAGTGGTCATTGTCCCGACGGTGTGGTGGTCTGTTGGCCGAGGTGCGTCTGGGGGG
CTCGTCCGGCCCTGTCGTCCGTCGGGAAGGCGCGTGTTGGGGCCTGCCGGAGTGCCGAGG
TGGGTACCCTGGCGGTGGGATTAACCCCGCGCGCGTGTCCCGGTGTGGCGGTGGGGGCTC
CGGTCGATGTCTACCTCCCTCTCCCCGAGGTCTCAGGCCTTCTCCGCGCGGGCTCTCGGC
CCTCCCCTCGTTCCTCCCTCTCGCGGGGTTCAAGTCGCTCGTCGACCTCCCCTCCTCCGT
CCTTCCATCTCTCGCGCAATGGCGCCGCCCGAGTTCACGGTGGGTTCGTCCTCCGCCTCC
GCTTCTCGCCGGGGGCTGGCCGCTGTCCGGTCTCTCCTGCCCGACCCCCGTTGGCGTGGT
CTTCTCTCGCCGGCTTCGCGGACTCCTGGCTTCGCCCGGAGGGTCAGGGGGCTTCCCGGT
TCCCCGACGTTGCGCCTCGCTGCTGTGTGCTTGGGGGGGGCCCGCTGCGGCCTCCGCCCG
CCCGTGAGCCCCTGCCGCACCCGCCGGTGTGCGGTTTCGCGCCGCGGTCAGTTGGGCCCT
GGCGTTGTGTCGCGTCGGGAGCGTGTCCGCCTCGCGGCGGCTAGACGCGGGTGTCGCCGG
GCTCCGACGGGTGGCCTATCCAGGGCTCGCCCCCGCCGACCCCCGCCTGCCCGTCCCGGT
GGTGGTCGTTGGTGTGGGGAGTGAATGGTGCTACCGGTCATTCCCTCCCGCGTGGTTTGA
CTGTCTCGCCGGTGTCGCGCTTCTCTTTCCGCCAACCCCCACGCCAACCCACCACCCTGC
TCTCCCGGCCCGGTGCGGTCGACGTTCCGGCTCTCCCGATGCCGAGGGGTTCGGGATTTG
TGCCGGGGACGGAGGGGAGAGCGGGTAAGAGAGGTGTCGGAGAGCTGTCCCGGGGCGACG
CTCGGGTTGGCTTTGCCGCGTGCGTGTGCTCGCGGACGGGTTTTGTCGGACCCCGACGGG
GTCGGTCCGGCCGCATGCACTCTCCCGTTCCGCGCGAGCGCCCGCCCGGCTCACCCCCGG
TTTGTCCTCCCGCGAGGCTCTCCGCCGCCGCCGCCTCCTCCTCCTCTCTCGCGCTCTCTG
TCCCGCCTGGTCCTGTCCCACCCCCGACGCTCCGCTCGCGCTTCCTTACCTGGTTGATCC
TGCCAGGTAGCATATGCTTGTCTCAAAGATTAAGCCATGCATGTCTAAGTACGCACGGCC
GGTACAGTGAAACTGCGAATGGCTCATTAAATCAGTTATGGTTCCTTTGGTCGCTCGCTC
CTCTCCTACTTGGATAACTGTGGTAATTCTAGAGCTAATACATGCCGACGGGCGCTGACC
CCCCTTCCCGGGGGGGGATGCGTGCATTTATCAGATCAAAACCAACCCGGTGAGCTCCCT
CCCGGCTCCGGCCGGGGGTCGGGCGCCGGCGGCTTGGTGACTCTAGATAACCTCGGGCCG
ATCGCACGCCCCCCGTGGCGGCGACGACCCATTCGAACGTCTGCCCTATCAACTTTCGAT
GGTAGTCGCCGTGCCTACCATGGTGACCACGGGTGACGGGGAATCAGGGTTCGATTCCGG
AGAGGGAGCCTGAGAAACGGCTACCACATCCAAGGAAGGCAGCAGGCGCGCAAATTACCC
ACTCCCGACCCGGGGAGGTAGTGACGAAAAATAACAATACAGGACTCTTTCGAGGCCCTG
TAATTGGAATGAGTCCACTTTAAATCCTTTAACGAGGATCCATTGGAGGGCAAGTCTGGT
GCCAGCAGCCGCGGTAATTCCAGCTCCAATAGCGTATATTAAAGTTGCTGCAGTTAAAAA
GCTCGTAGTTGGATCTTGGGAGCGGGCGGGCGGTCCGCCGCGAGGCGAGTCACCGCCCGT
CCCCGCCCCTTGCCTCTCGGCGCCCCCTCGATGCTCTTAGCTGAGTGTCCCGCGGGGCCC
GAAGCGTTTACTTTGAAAAAATTAGAGTGTTCAAAGCAGGCCCGAGCCGCCTGGATACCG
CAGCTAGGAATAATGGAATAGGACCGCGGTTCTATTTTGTTGGTTTTCGGAACTGAGGCC
ATGATTAAGAGGGACGGCCGGGGGCATTCGTATTGCGCCGCTAGAGGTGAAATTCTTGGA
CCGGCGCAAGACGGACCAGAGCGAAAGCATTTGCCAAGAATGTTTTCATTAATCAAGAAC
GAAAGTCGGAGGTTCGAAGACGATCAGATACCGTCGTAGTTCCGACCATAAACGATGCCG
ACTGGCGATGCGGCGGCGTTATTCCCATGACCCGCCGGGCAGCTTCCGGGAAACCAAAGT
CTTTGGGTTCCGGGGGGAGTATGGTTGCAAAGCTGAAACTTAAAGGAATTGACGGAAGGG
CACCACCAGGAGTGGGCCTGCGGCTTAATTTGACTCAACACGGGAAACCTCACCCGGCCC
GGACACGGACAGGATTGACAGATTGATAGCTCTTTCTCGATTCCGTGGGTGGTGGTGCAT
GGCCGTTCTTAGTTGGTGGAGCGATTTGTCTGGTTAATTCCGATAACGAACGAGACTCTG
GCATGCTAACTAGTTACGCGACCCCCGAGCGGTCGGCGTCCCCCAACTTCTTAGAGGGAC
AAGTGGCGTTCAGCCACCCGAGATTGAGCAATAACAGGTCTGTGATGCCCTTAGATGTCC
GGGGCTGCACGCGCGCTACACTGACTGGCTCAGCGTGTGCCTACCCTGCGCCGGCAGGCG
CGGGTAACCCGTTGAACCCCATTCGTGATGGGGATCGGGGATTGCAATTATTCCCCATGA
ACGAGGAATTCCCAGTAAGTGCGGGTCATAAGCTTGCGTTGATTAAGTCCCTGCCCTTTG
TACACACCGCCCGTCGCTACTACCGATTGGATGGTTTAGTGAGGCCCTCGGATCGGCCCC
GCCGGGGTCGGCCCACGGCCCTGGCGGAGCGCTGAGAAGACGGTCGAACTTGACTATCTA
GAGGAAGTAAAAGTCGTAACAAGGTTTCCGTAGGTGAACCTGCGGAAGGATCATTAAACG
GGAGACTGTGGAGGAGCGGCGGCGTGGCCCGCTCTCCCCGTCTTGTGTGTGTCCTCGCCG
GGAGGCGCGTGCGTCCCGGGTCCCGTCGCCCGCGTGTGGAGCGAGGTGTCTGGAGTGAGG
TGAGAGAAGGGGTGGGTGGGGTCGGTCTGGGTCCGTCTGGGACCGCCTCCGATTTCCCCT
CCCCCTCCCCTCTCCCTCGTCCGGCTCTGACCTCGCCACCCTACCGCGGCGGCGGCTGCT
CGCGGGCGTCTTGCCTCTTTCCCGTCCGGCTCTTCCGTGTCTACGAGGGGCGGTACGTCG
TTACGGGTTTTTGACCCGTCCCGGGGGCGTTCGGTCGTCGGGGCGCGCGCTTTGCTCTCC
CGGCACCCATCCCCGCCGCGGCTCTGGCTTTTCTACGTTGGCTGGGGCGGTTGTCGCGTG
TGGGGGGATGTGAGTGTCGCGTGTGGGCTCGCCCGTCCCGATGCCACGCTTTTCTGGCCT
CGCGTGTCCTCCCCGCTCCTGTCCCGGGTACCTAGCTGTCGCGTTCCGGCGCGGAGGTTT
AAGGACCCCGGGGGGGTCGCCCTGCCGCCCCCAGGGTCGGGGGGCGGTGGGGCCCGTAGG
GAAGTCGGTCGTTCGGGCGGCTCTCCCTCAGACTCCATGACCCTCCTCCCCCCGCTGCCG
CCGTTCCCGAGGCGGCGGTCGTGTGGGGGGGTGGATGTCTGGAGCCCCCTCGGGCGCCGT
GGGGGCCCGACCCGCGCCGCCGGCTTGCCCGATTTCCGCGGGTCGGTCCTGTCGGTGCCG
GTCGTGGGTTCCCGTGTCGTTCCCGTGTTTTTCCGCTCCCGACCCTTTTTTTTTCCTCCC
CCCCACACGTGTCTCGTTTCGTTCCTGCTGGCCGGCCTGAGGCTACCCCTCGGTCCATCT
GTTCTCCTCTCTCTCCGGGGAGAGGAGGGCGGTGGTCGTTGGGGGACTGTGCCGTCGTCA
GCACCCGTGAGTTCGCTCACACCCGAAATACCGATACGACTCTTAGCGGTGGATCACTCG
GCTCGTGCGTCGATGAAGAACGCAGCTAGCTGCGAGAATTAATGTGAATTGCAGGACACA
TTGATCATCGACACTTCGAACGCACTTGCGGCCCCGGGTTCCTCCCGGGGCTACGCCTGT
CTGAGCGTCGGTTGACGATCAATCGCGTCACCCGCTGCGGTGGGTGCTGCGCGGCTGGGA
GTTTGCTCGCAGGGCCAACCCCCCAACCCGGGTCGGGCCCTCCGTCTCCCGAAGTTCAGA
CGTGTGGGCGGTTGTCGGTGTGGCGCGCGCGCCCGCGTCGCGGAGCCTGGTCTCCCCCGC
GCATCCGCGCTCGCGGCTTCTTCCCGCTCCGCCGTTCCCGCCCTCGCCCGTGCACCCCGG
TCCTGGCCTCGCGTCGGCGCCTCCCGGACCGCTGCCTCACCAGTCTTTCTCGGTCCCGTG
CCCCGTGGGAACCCACCGCGCCCCCGTGGCGCCCGGGGGTGGGCGCGTCCGCATCTGCTC
TGGTCGAGGTTGGCGGTTGAGGGTGTGCGTGCGCCGAGGTGGTGGTCGGTCCCCTGCGGC
CGCGGGGTTGTCGGGGTGGCGGTCGACGAGGGCCGGTCGGTCGCCTGCGGTGGTTGTCTG
TGTGTGTTTGGGTCTTGCGCTGGGGGAGGCGGGGTCGACCGCTCGCGGGGTTGGCGCGGT
CGCCCGGCGCCGCGCACCCTCCGGCTTGTGTGGAGGGAGAGCGAGGGCGAGAACGGAGAG
AGGTGGTATCCCCGGTGGCGTTGCGAGGGAGGGTTTGGCGTCCCGCGTCCGTCCGTCCCT
CCCTCCCTCGGTGGGCGCCTTCGCGCCGCACGCGGCCGCTAGGGGCGGTCGGGGCCCGTG
GCCCCCGTGGCTCTTCTTCGTCTCCGCTTCTCCTTCACCCGGGCGGTACCCGCTCCGGCG
CCGGCCCGCGGGACGCCGCGGCGTCCGTGCGCCGATGCGAGTCACCCCCGGGTGTTGCGA
GTTCGGGGAGGGAGAGGGCCTCGCTGACCCGTTGCGTCCCGGCTTCCCTGGGGGGGACCC
GGCGTCTGTGGGCTGTGCGTCCCGGGGGTTGCGTGTGAGTAAGATCCTCCACCCCCGCCG
CCCTCCCCTCCCGCCGGCCTCTCGGGGACCCCCTGAGACGGTTCGCCGGCTCGTCCTCCC
GTGCCGCCGGGTGCCGTCTCTTTCCCGCCCGCCTCCTCGCTCTCTTCTTCCCGCGGCTGG
GCGCGTGTCCCCCCTTTCTGACCGCGACCTCAGATCAGACGTGGCGACCCGCTGAATTTA
AGCATATTAGTCAGCGGAGGAAAAGAAACTAACCAGGATTCCCTCAGTAACGGCGAGTGA
ACAGGGAAGAGCCCAGCGCCGAATCCCCGCCGCGCGTCGCGGCGTGGGAAATGTGGCGTA
CGGAAGACCCACTCCCCGGCGCCGCTCGTGGGGGGCCCAAGTCCTTCTGATCGAGGCCCA
GCCCGTGGACGGTGTGAGGCCGGTAGCGGCCCCGGCGCGCCGGGCTCGGGTCTTCCCGGA
GTCGGGTTGCTTGGGAATGCAGCCCAAAGCGGGTGGTAAACTCCATCTAAGGCTAAATAC
CGGCACGAGACCGATAGTCAACAAGTACCGTAAGGGAAAGTTGAAAAGAACTTTGAAGAG
AGAGTTCAAGAGGGCGTGAAACCGTTAAGAGGTAAACGGGTGGGGTCCGCGCAGTCCGCC
CGGAGGATTCAACCCGGCGGCGCGCGTCCGGCCGTGCCCGGTGGTCCCGGCGGATCTTTC
CCGCTCCCCGTTCCTCCCGACCCCTCCACCCGCGCGTCGTTCCCCTCTTCCTCCCCGCGT
CCGGCGCCTCCGGCGGCGGGCGCGGGGGGTGGTGTGGTGGTGGCGCGCGGGCGGGGCCGG
GGGTGGGGTCGGCGGGGGACCGCCCCCGGCCGGCGACCGGCCGCCGCCGGGCGCACTTCC
ACCGTGGCGGTGCGCCGCGACCGGCTCCGGGACGGCCGGGAAGGCCCGGTGGGGAAGGTG
GCTCGGGGGGGGCGGCGCGTCTCAGGGCGCGCCGAACCACCTCACCCCGAGTGTTACAGC
CCTCCGGCCGCGCTTTCGCCGAATCCCGGGGCCGAGGAAGCCAGATACCCGTCGCCGCGC
TCTCCCTCTCCCCCCGTCCGCCTCCCGGGCGGGCGTGGGGGTGGGGGCCGGGCCGCCCCT
CCCACGGCGCGACCGCTCTCCCACCCCCCTCCGTCGCCTCTCTCGGGGCCCGGTGGGGGG
CGGGGCGGACTGTCCCCAGTGCGCCCCGGGCGTCGTCGCGCCGTCGGGTCCCGGGGGGAC
CGTCGGTCACGCGTCTCCCGACGAAGCCGAGCGCACGGGGTCGGCGGCGATGTCGGCTAC
CCACCCGACCCGTCTTGAAACACGGACCAAGGAGTCTAACGCGTGCGCGAGTCAGGGGCT
CGTCCGAAAGCCGCCGTGGCGCAATGAAGGTGAAGGGCCCCGCCCGGGGGCCCGAGGTGG
GATCCCGAGGCCTCTCCAGTCCGCCGAGGGCGCACCACCGGCCCGTCTCGCCCGCCGCGC
CGGGGAGGTGGAGCACGAGCGTACGCGTTAGGACCCGAAAGATGGTGAACTATGCTTGGG
CAGGGCGAAGCCAGAGGAAACTCTGGTGGAGGTCCGTAGCGGTCCTGACGTGCAAATCGG
TCGTCCGACCTGGGTATAGGGGCGAAAGACTAATCGAACCATCTAGTAGCTGGTTCCCTC
CGAAGTTTCCCTCAGGATAGCTGGCGCTCTCGCTCCCGACGTACGCAGTTTTATCCGGTA
AAGCGAATGATTAGAGGTCTTGGGGCCGAAACGATCTCAACCTATTCTCAAACTTTAAAT
GGGTAAGAAGCCCGGCTCGCTGGCGTGGAGCCGGGCGTGGAATGCGAGTGCCTAGTGGGC
CACTTTTGGTAAGCAGAACTGGCGCTGCGGGATGAACCGAACGCCGGGTTAAGGCGCCCG
ATGCCGACGCTCATCAGACCCCAGAAAAGGTGTTGGTTGATATAGACAGCAGGACGGTGG
CCATGGAAGTCGGAATCCGCTAAGGAGTGTGTAACAACTCACCTGCCGAATCAACTAGCC
CTGAAAATGGATGGCGCTGGAGCGTCGGGCCCATACCCGGCCGTCGCCGCAGTCGGAACG
GAACGGGACGGGAGCGGCCGCGGGTGCGCGTCTCTCGGGGTCGGGGGTGCGTGGCGGGGG
CCCGTCCCCCGCCTCCCCTCCGCGCGCCGGGTTCGCCCCCGCGGCGTCGGGCCCCGCGGA
GCCTACGCCGCGACGAGTAGGAGGGCCGCTGCGGTGAGCCTTGAAGCCTAGGGCGCGGGC
CCGGGTGGAGCCGCCGCAGGTGCAGATCTTGGTGGTAGTAGCAAATATTCAAACGAGAAC
TTTGAAGGCCGAAGTGGAGAAGGGTTCCATGTGAACAGCAGTTGAACATGGGTCAGTCGG
TCCTGAGAGATGGGCGAGTGCCGTTCCGAAGGGACGGGCGATGGCCTCCGTTGCCCTCGG
CCGATCGAAAGGGAGTCGGGTTCAGATCCCCGAATCCGGAGTGGCGGAGATGGGCGCCGC
GAGGCCAGTGCGGTAACGCGACCGATCCCGGAGAAGCCGGCGGGAGGCCTCGGGGAGAGT
TCTCTTTTCTTTGTGAAGGGCAGGGCGCCCTGGAATGGGTTCGCCCCGAGAGAGGGGCCC
GTGCCTTGGAAAGCGTCGCGGTTCCGGCGGCGTCCGGTGAGCTCTCGCTGGCCCTTGAAA
ATCCGGGGGAGAGGGTGTAAATCTCGCGCCGGGCCGTACCCATATCCGCAGCAGGTCTCC
AAGGTGAACAGCCTCTGGCATGTTGGAACAATGTAGGTAAGGGAAGTCGGCAAGCCGGAT
CCGTAACTTCGGGATAAGGATTGGCTCTAAGGGCTGGGTCGGTCGGGCTGGGGCGCGAAG
CGGGGCTGGGCGCGCGCCGCGGCTGGACGAGGCGCCGCCGCCCTCTCCCACGTCCGGGGA
GACCCCCCGTCCTTTCCGCCCGGGCCCGCCCTCCCCTCTTCCCCGCGGGGCCCCGTCGTC
CCCCGCGTCGTCGCCACCTCTCTTCCCCCCTCCTTCTTCCCGTCGGGGGGCGGGTCGGGG
GTCGGCGCGCGGCGCGGGCTCCGGGGCGGCGGGTCCAACCCCGCGGGGGTTCCGGAGCGG
GAGGAACCAGCGGTCCCCGGTGGGGCGGGGGGCCCGGACACTCGGGGGGCCGGCGGCGGC
GGCGACTCTGGACGCGAGCCGGGCCCTTCCCGTGGATCGCCTCAGCTGCGGCGGGCGTCG
CGGCCGCTCCCGGGGAGCCCGGCGGGTGCCGGCGCGGGTCCCCTCCCCGCGGGGCCTCGC
TCCACCCCCCCATCGCCTCTCCCGAGGTGCGTGGCGGGGGCGGGCGGGCGTGTCCCGCGC
GTGTGGGGGGAACCTCCGCGTCGGTGTTCCCCCGCCGGGTCCGCCCCCCGGGCCGCGGTT
TTCCGCGCGGCGCCCCCGCCTCGGCCGGCGCCTAGCAGCCGACTTAGAACTGGTGCGGAC
CAGGGGAATCCGACTGTTTAATTAAAACAAAGCATCGCGAAGGCCCGCGGCGGGTGTTGA
CGCGATGTGATTTCTGCCCAGTGCTCTGAATGTCAAAGTGAAGAAATTCAATGAAGCGCG
GGTAAACGGCGGGAGTAACTATGACTCTCTTAAGGTAGCCAAATGCCTCGTCATCTAATT
AGTGACGCGCATGAATGGATGAACGAGATTCCCACTGTCCCTACCTACTATCCAGCGAAA
CCACAGCCAAGGGAACGGGCTTGGCGGAATCAGCGGGGAAAGAAGACCCTGTTGAGCTTG
ACTCTAGTCTGGCACGGTGAAGAGACATGAGAGGTGTAGAATAAGTGGGAGGCCCCCGGC
GCCCGGCCCCGTCCTCGCGTCGGGGTCGGGGCACGCCGGCCTCGCGGGCCGCCGGTGAAA
TACCACTACTCTCATCGTTTTTTCACTGACCCGGTGAGGCGGGGGGGCGAGCCCCGAGGG
GCTCTCGCTTCTGGCGCCAAGCGTCCGTCCCGCGCGTGCGGGCGGGCGCGACCCGCTCCG
GGGACAGTGCCAGGTGGGGAGTTTGACTGGGGCGGTACACCTGTCAAACGGTAACGCAGG
TGTCCTAAGGCGAGCTCAGGGAGGACAGAAACCTCCCGTGGAGCAGAAGGGCAAAAGCTC
GCTTGATCTTGATTTTCAGTACGAATACAGACCGTGAAAGCGGGGCCTCACGATCCTTCT
GACCTTTTGGGTTTTAAGCAGGAGGTGTCAGAAAAGTTACCACAGGGATAACTGGCTTGT
GGCGGCCAAGCGTTCATAGCGACGTCGCTTTTTGATCCTTCGATGTCGGCTCTTCCTATC
ATTGTGAAGCAGAATTCACCAAGCGTTGGATTGTTCACCCACTAATAGGGAACGTGAGCT
GGGTTTAGACCGTCGTGAGACAGGTTAGTTTTACCCTACTGATGATGTGTTGTTGCCATG
GTAATCCTGCTCAGTACGAGAGGAACCGCAGGTTCAGACATTTGGTGTATGTGCTTGGCT
GAGGAGCCAATGGGGCGAAGCTACCATCTGTGGGATTATGACTGAACGCCTCTAAGTCAG
AATCCCGCCCAGGCGGAACGATACGGCAGCGCCGAAGGAGCCTCGGTTGGCCCCGGATAG
CCGGGTCCCCGTCCGTCCCCGCTCGGCGGGGTCCCCGCGTCGTCCCCGCGGCGGCGCGGG
GTCTCCCCCCGCCGGGCGTCGGGACCGGGGTCCGGTGCGGAGAGCCGTTCGTCTTGGGAA
ACGGGGTGCGGCCGGAAAGGGGGCCGCCCTCTCGCCCGTCACGTTGAACGCACGTTCGTG
TGGAACCTGGCGCTAAACCATTCGTAGACGACCTGCTTCTGGGTCGGGGTTTCGTACGTA
GCAGAGCAGCTCCCTCGCTGCGATCTATTGAAAGTCAGCCCTCGACACAAGGGTTTGTCT
CTGCGGGCTTTCCCGTCGCACGCCCGCTCGCTCGCACGCGACCGTGTCGTCGCCCGGGCG
TCACGGGGGCGGTCGCCTCGGCCCCCGCGCGGTCGCCCGAACGACCGTGTGGTGGTTGGG
GGGGGATCGTCTTCTCCTCCGTCTCCCGAGGACGGTTCGTTTCTCTTTCCCCTTCCGTCG
CTCTCCTTGGGTGTGGGAGCCTCGTGCCGTCGCGACCGCGGCCTGCCGTCGCCTGCCGCC
GCAGCCCCTTGCCCTCCGGCCTTGGCCAAGCCGGAGGGCGGAGGAGGGGGATCGGCGGCG
GCGGCGGCGACCCGGGCCGGTGACCGACGGTGGGATCCCCATCCTCGGCGCGTCCGTCGG
GGACGGCCGGTTGGAGGGGCGGGAGGGGTTTTTCCCGTGAACGCCGCGTTCGGCGCCAGG
CCTCTGGCGGCCGGGGGGGCGCTCTCTCCGCCCGAGCATCCCCACTCCCGCCCCTCCTCT
TCGCGCGCCGCGGCGGCGACGTGCGTACGAGGGGAGGATGTCGCGGTGTGGAGGCGGAGA
GGGTCCGGCGCGGCGCCTCTTCCATTTTTTCCCCCCCAACTTCGGAGGTCGACCAGTACT
CCGGGCGACACTTTGTTTTTTTTTTTTCCCCCGATGCTGGAGGTCGACCAGATGTCCGAA
AGTGTCCCCCCCCCCCCCCCCCCCCCCCCGGCGCGGAGCGGCGGGGCCACCCCGGACCCC
TTTTTTTTTTTTTTTTTTTTTTTTAAATTCCTGGAACCTTTAGGTCGACCAGTTGTCCGT
CTTTTACTCCTTCATATAGGTCGACCAGTACTCCGGGTGGTACTTTGTCTTTTTCTGAAA
ATCCCAGAGGTCGACCAGATATCCGAAAGTCCTCTCTTTCCCTTTACTCTTCCCCACAGC
GATTCTCTTTTTTTTTTTTTTTTTTGGTGTGCCTCTTTTTGACTTATATACATGTAAATA
GTGTGTACGTTTATATACTTATAGGAGGAGGTCGACCAGTACTCCGGGCGACACTTTGTT
TTTTTTTTTTTTTCCACCGATGATGGAGGTCGACCAGATGTCCGAAAGTGTCCCGTCCCC
CCCCTCCCCCCCCCGCGACGCGGCGGGCTCACTCTGGACTCTTTTTTTTTTTTTTTTTTT
TTTTTAAATTTCTGGAACCTTAAGGTCGACCAGTTGTCCGTCTTTCACTCATTCATATAG
GTCGACCGGTGGTACTTTGTCTTTTTCTGAAAATCGCAGAGGTCGACCAGATGTCAGAAA
GTCTGGTGGTCGATAAATTATCTGATCTAGATTTGTTTTTCTGTTTTTCAGTTTTGTGTT
GTTTTGTGTTGTTTTGTTTTGTTTTGTTTTGTTTTGTTTTGTTTTGTTTTGTTTTGTTTT
GTTTTGTTTTGTGTTGTGTTGTGTTGTGTTGTGTTGTGTTGGGTTGGGTTGGGTTGGGTT
GGGTTGGGTTGGGTTGGGTTGGGTTGGGTTGTGTTGTGTTGTGTTGTTTGGTTTTGTGTT
GTTTGGTGTTGTTTGGTTTTGTTTTGTTTGCTGTTGTTTTGTGTTTTGCGGGTCGAACAG
TTGTCCCTAACCGAGTTTTTTTGTACACAAACATGCACTTTTTTTAAAATAAATTTTTAA
AATAAATGCGAAAATCGACCAATTATCCCTTTTCTTCTCTCTCTTTTTTTAAAAATTTTT
CTTTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGCGTGTGTGTGTGTGTGTGCGTGC
ACGCGTGCGCGCGCTCGTTTTATAAATACTTATAATAATAGGTCGCCGGGTGGTGGTAGC
TGTAATTCCAGCACTCCAGAGGCAGAGGCAGGCAGACTTCTGAGTTCGAGGCCAGCCTGG
TCTACAGAGGAACCCTGTCTCGAAAAATGAAAATAAATACATACATACATACATACATAC
ATACATACATACATACATACATACATATGAGGTTGACCAGTTGTCAATCCTTTAGAATTT
TGTTTTTAATTAATGTGATAGAGAGATAGATAATAGATAGATGGATAGAGTGATACAAAT
ATAGGTTTTTTTTTTCAGTAAATATGAGGTTGATTAACCACTTTTCCCTTTTTTAGGTTT
TTTTTTTTTTCCCCTGTCCATGTGGTTGCTGGGATTTGAACTCAGGACCCTGGCAGGTCA
ACTGGCAAAACGTGTTTTCTATATATATAAATAGTGGTCTGTCTGCTGTTTGTTTGTTTG
CTTGCTTGCTTGCTTGCTTGCTTGCTTGCTTGCTTGCTTGCTTTTTTTTTTCTTCTGAGA
CAGTATTTCTCTGTGTAACCTGGCTGCCCTGAAACTCACTCTGTAGACCAGCCTGGCCTC
AATCGAACTCAGAAATCCTCCTGCCTCTGTCTACCTCCCAATTTTGGGAGTAAAGGTGTG
CTACACCACTGCCTGGCATTATTATCATTATCATTATTAATTTTATTATTAGACAGAACG
AAATCAACTAGTTGGTCCTGTTTCGTTAATTCATTTGAAATTAGTTGGACCAATTAGTTG
GCTGGTTTGGGAGGTTTCTTTTGTTTCCGATTTGGGTGTTTGTGGGGCTGGGGATCAGGT
ATCTCAACGGAATGCATGAAGGTTAAGGTGAGATGGCTCGATTTTTGTAAAGATTACTTT
TCTTAGTCTGAGGAAAAAATAAAATATTGGGCTACGTTTCATTGCTTCATTTCTATTTCT
CTTTCTTTCTTTCTTTCTTTCAGATAAGGAGGTCGGCCAGTTCCTCCTGCCTTCTGGAAG
ATGTAGGCATTGCATTGGGAAGCATTGTTTGAGAGATGTGCTAGTGAACCAGAGAGTTTG
GATGTCAAGCTGTATAATGTTTCTTCCAAAAAAGTTCTAACAAAGGGATCTTTAACTTTT
TTTTTCTCCTTCTACTTTTCCTTGTAGGCTATCAGCCTTTGTATAATGAAATGTGAGCTT
TGCTTTGTTTTGTTGTTTTTCTTTTGGTTTTGCTTGACATGGTTTCCCTTTCTATCCGTG
CAGGGTTCCCAGACTGCCCTTTGAGAATAAAATGGGAGGCCACAACCAAAGTCTTTTGGA
TAAAGCACCACAATGGACAATGTAAGGAAGGCAGCCAAACGCTTACGCTTCCCAAGGCAC
AGATCTTTCCCAGCATGGGAAAGATCCTCCCTCCCCCTCCCTACCTGCCTCCCTCCTTCC
ACCATACACTCAATAAAATAAAATAAAATAAGATAAGATAAGATAAAATAAAATAAAATA
AATGAATGAACGAATGAATGACGTTTCCAGTTATTTATTGGTCTAAACTACCTAAATTCC
CCTTTTACACCTCCCCCTACTGCCCCCATGCCGCCATCTCCTTGTGGGGGTAGGGCCAGG
CACATCTGTGTCTTACTCTCTGAACTATTTGTTTTCTGGTCCACATCGCTCCAGTGCAAT
GTAGCCCTTTGGAAATGGTAATGCTAACACACCCCACCAGAAATGGTAATGCTAACACAC
CCTACCAGAAAGAGGAAGGCTCCACTCACTCACTCACTCACTCACTCACTCACTCACTCA
CTCACTCACTCTTAACCAGCCCTGCCCTGCTATTTCAAGGATTTCTAAGGACAGAATATT
CTGGACAGTATTTACATGGCTTGATATCTCTAAATTTTGGTTTTGGTTTTGGTGGTTTTT
TGTTTTTTTGTTTTTGTTTTTCCTTCCAGACAGGGTTTCTCTGTGTAGCCCTGGCTGTCC
TGTACTCTGTGTAGACCAGGCTGTCCTCAAACTCAGAAATCCATCTGCCTCTGCCTCCCA
AGTGCTGGAATTAAAAGTGTGTGTCCACCCACCACTGCCTGGCTCTACTTTTTTTAAACA
TTATGTATGTATTATGTATGTATGTATGTATGTATGTATGCATGCATGCATGCATGCATT
TTTTTTTTAAGATTTATTTATTTATTATATGTAAGTACACTGTAGCTGTCTTCAGACACT
CCAGAAGAGGGAATCAGATCTTGTTACACATGGTTGTGAGCCACCATGTGGTTGCTGGGA
TTTGAACTTCAGACCTTCGGAGAGCAGTCAGGTGCTCTTACCCACTGAGCCATCTCACCA
GCCCGTATGCATGCATTATTATGCATGAAACTACACTGCAGCTGTCTTCAGACACACACC
AGAAGAATACATGGGATCCCACGGCAGATGGTTGTTGGAAATTGAACTCAGGACCTCTGG
AAGAGAGGTCAGTGCTGAGCCATCTCTCAAAATAAAACTTGTGGAGAAATTGAGCATCTT
AAAACCCTAGGCATCCTTCCTCCCTCTCTCCACGTGGCCATGATGACCACCCTCTTTCTA
CCTTCTCTCTTTCCCTCTGCCTTAAAGCTGTAAAACCACAAAACAACAGCAACAACAACC
CCTAGCCATCTGTGTTTGTTTGTTTGTTTATGTGTCTATATTTATTTATTTACTTACTTA
CATTTATTTATTTATTTATTTTTATTTATTCATCTGTATTTATTTATTTATTTATTTATT
TATTTATTTATTTATTTACTTACTTACTTACTTACTTACTTACTTACTTGTTCTAACACC
ACCCAGGTGTGGTGGTGCATGCCTTTGCTCCCAACACTTGGGAGGCAGAGGCAAAAAAAA
CAAAACAAAACAAAACAAAACAACAACAACAACAACATCAAACAAAGTTTGGGTCTTAAT
GAAGTAAGCAAGGCACAGGACTTGATGAATTTGTACGAATTACCCCTTCAGAGTTTACCG
GAGCTTAAAGGACTGAGTGTCTGGGACTGCCATACATATACATGTATATGTATATATATT
TTTTTTCTTTTTTTCTTTCTGAGCTGAGCCATTTCTCTCACCAACCTGGGTCCCTCTTTC
TTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTCCTTCCTTCC
TTCCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCCTTCTTTCTTTCTTTCTTTCTCTC
TTCCTTCACCCTGGGGTCAATCCAAAACACAGACTGTTTTGAGGTAAACTTAATAAACCT
CTAGCTTCTCTCTGGCTCTCTTGCTTCTTTCTCTCTTGCCCTCTTGGGTGCTTCTCTTTC
CCATTTCCTTCCTCTGCTCTCCAGGAGCTCCTGTCCAGCCTCTATTTATCTACTTCTCTA
CTCTCTCTGCCTCTACTAAAAGTGAAGAAGAAGAAGAAGAAGAAGAAGAAGAAGAAGAAG
AAGAAGAAGAAGAAGAAGAAGAAGAAGAAGAAGAAGAAGAAGAAGAAGGGCTGGTGAGAT
GGCTCAGTGGGTTAGAGCACCCGACTGCTCTTCCGAAGGTGCAGAGTTCAGGTCCCAGCA
ACCACATGGTGGCTCACAACCATCCGTAACAAGATCTGACCCCCTCTTCTGGAGTGTCTG
AGGACAGCTGCAGTGTACTTACATATAATAAATAAATAAACAAACAAATAAATAAATAAA
AAAAAAAAAAAGAAGAAGAAGAAGAAGAAGAGGAGGAGGAGGAGGAGGAGGAGGAGGAGG
AGGAGGAGGGGGAGGAGGAGGAAGAGGAGGTTGATGATCTGACTTTTCTCTTAACTGGCA
GGTCAACTGGCAAAATGGGTTTTCTTTTATATATATAATATATATAATATATATATAATT
AAATATATATATATATATTTTAAAGAGTAACTTGTAAATAGTGGTTTGTCTGCTGTTTGT
TTGTTTGTTTGTTTGCTTGCTTGATTTCTTGCTTGCTTTTTTTCTTTTTTCTGAGACAGT
GTTTCTCTGTGTAGCCTTGGCTGCCCTGGAACTCACTCTGTAGACAAGCCTGGCCTCAAT
CGAATTCAGAAATCCTCCTGCCTCTGTCTGCCTCCTAATTGCTGGGAGTTATTATTATTA
TTATTATTATTATTATTATTATTATTATTATTATTATATCAACTAGTTGGCCCTGTTTCA
TTTATTCATTTGAAATTAGTTGGCTGGTTTGGGAGGTTTCTTTCTTTTTTTTTTTTTTAA
AGACTTATTTATTTATTATATGTAAGTACACTGTAGCTGTCTTCAGACACTCCAGAAGAG
GGAGTCAGATCTTGTTACGGATGGTTGTGAGCCACCATGTGGTTGCTGGGATTTGAACTC
TGGACCATCGGAAGAGCAGTCGGGTGCTCTAACCCACTGAGCCATCTCACCAGCCCGGGA
GGTTTCTTTTGTTTTAGATTTGGGTGTTTTCTTAGTCTGAGGAAAAAATAAAATATTGGG
CTACGTTTCATTGCTTCATTTCTATTTCTCTTTTTTTTCTTTCAGATCAGGAGGTCCCGC
TAGTTCCTCCTGTCTTCTGTTTTTCATAACCCACGCACACAAGCATGCTCCATCCACAGT
GGGGAAAAATCATGGGGGAAGACGGTAGACATTGCATTGGGAAGATGTGCTAGTGAACCA
GAGAGTTTGGATGTCAAGCCGTATAATGTTTATTACAATATAGAAAAGTTCTAACAAAGT
GATCTTTAACTTTTTTTTTTTTTTTTCTCCTTCTACTTCTACTTGTTCTCACTCTGCCAC
CAACGTGGCAGGCTTTGTACAAAGAATGTGAGCTTTGTTTTGCTTAACAGACATATATTT
TTTCTTTTGGTTTTGCTTGACATGGTTTCCCTTTCTATCCGTGCAGGGTTCCCAGACTGC
CTTTTGAGAATAAAATGGGAGGCCAGAACCAAAGTCTTTTGAATAAAGCACCACAACTCT
AACCTGTTTGGCTGTTTTCCTTCCCAAGGCACAGATCTTTCCCAGCATGGAAAAGCATGT
AGCAGTTGTAGGACACACTAGACGAGAGCACCAGATCTCATTGTGGGTGGTTGTGAACCA
CCCACCATGTGGTTGCTGGGATTTGAACTCAGGATCTTCAGAAGAGCAGTCAGGGCTCTA
AACCGATGAGCCATCTCTCCAGCCTCCTACATTCCTTCTTAAGGCATGAATGATCCCAGC
ATGGGAAGACAGTCTGCCCTCCCTCCTTTTTGAGCCATTTTCCCTCTTTCACCATATACT
CAATAAAATAAGTAAATGAATGAAGTCTCCAGCTATTTATTTCTTCGAGCTATCTAAATT
CTCTCACAGCACCTCCCCCTCCCCCACACTGCCTTTTCTCCCTATGTTTGGGTGGGGCTG
GGGGAGGGGTGGGGTGGGGGGCAGGGATCTGCATGTCTTCCTTGCAGGTCTGTGAACTAT
TTGCGATGGCCTGGTTCTCTGAACTGTTGAGCCTTGTCTATCCAGAGGCTGACTGGCTAG
TTTTCTACCTGAAGTCCCTGAGTGATGATTTCCCTGTGAATTCTCTGAACTCACTGTCCT
TTCAGGTCCACATTTGCTCTGGCTCTGGCAACTGACTGAACCAAACAATTAGGCTCATAC
CTAGAACTTTGGAAATGGTAATGGAGTTGATAACACACCCTAACAGAAAGAGGAAGGCTC
CACTCACTAGTACACTTTCACTAGAAGCCTCGCCCTGTCAGCAGTTTAGGAATTTGTAAA
GCCAATCTGGACAATATTTAAATGGTTTGGAATTTCTACTTCATGTTTAATTTTTTTTTT
TTTTTTTTTTTTTTGAGACAGGGTTTCTCTGTATAGCCCTGGCTGTCCTGGAACTCACTT
TGTAGACCAGGCTGGCCTCGAACTCAGAAATACGCCTGCCTCTGCCTCCCGAGTGCTGTG
ATTAAAGGCGTGCACCACCACGCCAGGCCCCATGTTTAATGTTTAGTAAACATTTAATAA
ACATTTAATAAACATTGTGAAGGTCTGAAGAGATGGATCAGCAGTTAAGAGCATTGACTT
TTCTTGCAGAGGTCCTGAGTTCAATTCCCAGGAATCACATGGTGCCTTTGGAATCCGATG
CACACTTTTGGTGTGTGTTGGAAGACAGCTGCAGTGTACTCATATAAATAAAAAATAAAT
AAAACATTCAAAAATAAAGTAAAACAAAGATTGTGATGAAATTGAGCATTTTAATTTTAT
TTTATTGTATTGTATTTCACACACACACACACACACACAGACATATATCTACTGGCACAG
AAATTCACATTACCTAGTAAGCTGGGAGTGGTTGGATTGAATATGCAAAGAAGGAACATT
TGCAGTCAGTCAGTAAAGTTTTGCTGTTGCTCTTTTTTCTTTTTTTTTGGGGGGGGGGGG
TGTTTAAATAGTACAGCAGAGCAGTACATGCCATTAATCCCAGCACTTCGGAGGCAGAGG
CAGGCAGATTTCTGAATTGGAGGCCAGCCTGGTCTACAGAGTGAGTTCCAGGACAGCCCA
GGCTACACAGAGAAACCCTGTCTGGAAAAACAAACAAACAGACAAACAAACCAAGCAAAA
TCCAATCCAAACTTCACAAAGGACACTATAACAAATTACAGTTGCTTGCGTCTAATAGAA
AACATTGCCTTCCCCCCAAGTAACCCTGGTTCAAGTAAATACAAAACTAAGTAATCACAC
ATGTACAAACATACAAACATACCAAAATACATACATACATACATACATACATACATACAT
ACATACATACATAAAACAAATAATGATAACTAAGTGGCATGGAGCTGCTCACACAGTTTG
GATTCCACCTACCCACTCCACCTCCATCACATGCCTGCAGAACTTACTTGAGCAGCCGGC
CTGATCTGTCTTGTTTGTTTTTGTCTTGTACTTTGTCTCTCACAAATGAATGAATAGAGA
GATAGATATTCATTCATCTATACATGCATGTGTGATGGCTCAGAGGGCATAGCTGCTGGC
TGTTAAGGTATAGGTAAAATATTAAGGACTGAGGAAACTAACCTCTTCTCTGGTTAACCT
GCTATCTTGTGCTGTTAACTACTATTATAAGGACACTTTATCATAGCTTGTTTCTGTCAT
TACCAAACAATTATGGCTGGCTGCCTGCCTGCCTGCCTGCCTGCCTGCCTGCCTACACAG
AGAAATGATTTCTCCAATCAACTAAAACTACCTCCTAAGCCTGCCTCTTTTTCTCTTCCT
TTTTCTTTTTCTTTCATTTTCTTTCTCTTTCTTTTTCTCTTTCTTTCTCTTTCTCTTTCT
TTTTCATTTTCTCTCTCTCTCTTCCTTTTTTTCTTTCTTTCTTACATTTATTCTTTTCAT
ACATATTTTATTTCATTTCATTTGACTTTTATTTTATTTTATTTTATTTTATTTTATTTT
ATTTTATTTTATTTTATTTTGGTTTTTCGAGACAGGGTTTCTCTGTATACCCCTGGCTGT
CCTGGAACTCAGTTTGCAGACCAGGCTGTCCTGGAACTCTGAAATCCTCCCATCTCTGCC
TCCCTAGTGTTGGGATTAAAGACATGGGCCACCACTCTGGCTACTTTTTAATTTTTTAAT
TTTTATTTTTTTTGTTTTTGTTTTTGTTTTTGTTTTTTCGAGACAGGGTTTCTCTGTATA
ACTCTGGCTGTCCTGGAACTCACTTCATAGACCAGGCTGGCCTTGAACTCAGAAATACGC
CTGCCTCTGCCTCCCAAGTGCTGGGATTAAAGGTGTGCTTTACGCCCGGCTTTTATTTTT
AAATTTTATTTATTTAATTTATTTTTGTTTATTTCAAGATGTGGTTTCTCTGTGTAACCT
TGGCTGTCCTGGAACTCACTGTGTAGACCAGGCTGGCCTTGAACACTTTAATCTGTCTGC
CTCTGCTTCCCAAATGCTGGGATTAAAAGCATCTGCCACCACTGCCCTGACTAATTTTTT
TTTTTTTGGAAAATACCTTTCTTCCTACCTTCCTTTCTTCCTTCCTTCCTTCCTTCATTC
CTTCCTTCCTTCCTTCCTTTCTTTCTTTCTTTCATTATTTTTTCTTTTTTTTGAGATATG
GTTTCTCTGTGTAGCCCTAGCTGCCTTGTAATCCAAGGTGACCTTGAATGCAGAAATCTG
CCTGAGGTTAAAGGTGTGTTCCACAATTGCCCCAGCTCTACTCTAATTCTCTTAAAAAAA
AATATGTGTGTACATATGTATGTATAATTAGAAGTTATATTAATCCATTAAGATCTCTAA
AGTGAGTTCCAGGAGTATTCAGAGAAACCCTGTCACAATAAATAAATAAATAAATAAATA
AATAAATAAATAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGGA
GGAATCCTAACCATCACCCCCAATAAAAAGGCTGTGGAAACTATTTCATGTCCCTCTCCC
CCATAGGTGAGGATGTAGCTGCTTGTAAATGAATACGCCGTGTGTGTGTGTGTATGTGTT
TCCATGGCCAGCTAGTACCAACCCCAGATTTGTGTTTCAGACATATGATTCTTCTAGTAC
TATTGACTATTGAATTTTGATCATGAAGGCACATTGGATTTTTATTTTTTGGTTTTCTTT
TTGTTATGGTTTTTTCAGACAGGGTTTCCCTGTATAGCCCTGGCTGTCCTGGCACATTGG
ATTTTATAACAGAGTTTATGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTG
TGTGTGTGTGTGTAATACTTCTGCTGTGATTGCGACTAATTTTATTTGTTTGAAAATACC
TTTAATTCTTTCTTTCTTCTTCCTTCCTTCCTTCCTTCCTTCCTTCCTTCCTTCCTTCCT
TTCTTTCTCTTTCTTTCTCTCTTTCTATTTTTTTTTTTTGTTTTTTGTTTTTTGTTTTTT
TGAAGATATGGTTTCTCTAGCCCTAGCTGACTTGTAAACCAGGGTGACCTTGAAAGCAGA
AATCTGCCTGAGGTTAAAGGTGGTTTCCACAATTGCCCCAGCTCTACTCTAATTCTCTTT
TAAAAAAATGTGTGTGTGTATGTTTCTATGTATATATGTATATATAGAAGTAATATTAAT
CCATTAATATCTCTTTTTCCTAAAATTCATGTCTTTCTCTCTCTCTTTCTTCTTTCCTTT
CTTTCTTTCTTTTTGCTTTCTCTTTCTTTTTTCTCTCTTTCTCCTTTTCTCTCTTTCTAT
TAGTTTTTATTGTTTTTTCTGAGATATGGTTTCTCTAGCCCTAGCTGACTTGTAAGCCAG
GGTGACCTTGAAAGCAGAAATCTGCCTGAGGTTGAAGGTGGTTTCCACAATTGACCCAGC
TCTACTCTAATTCTCTTTTAAAAAAATGTGTGTGTGTATGTTTGTATGTATATATGTATA
TATAGAAGTAATATTAGGGGCTGGTGAGATGGCTCAGTGGGTAAGAGCACCCGACTGCTC
TTCCGAAGGTCTGGAGTTCAAATCCCAGCAACCACATGGTGGCTCACAACCACCTGTAAT
GAGATCTGACTCTCTCTTCTGGAGTGTCGGAAGACAGCTACAGTGTACTTACATATTAAA
AAAAAAAAAGAAGTAATATTAATCCATTAATATCTCTTTTTCCTAAAATTCATGTCTTTC
TTTCTCTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTCTC
TTTCTTTCTTTCTTTTTACTTTCTCTTTCTTTTTTCTCTCTTTCTCTCTTTCTATTAGTT
TTTATTGTTTTTTCTGAGATATGGTTTCTCTAGTCCTAGCTGACTTGTAAGCCAGGGTGA
CCTTGAAAGCAGAAATCTGCCTGAGGTTGAAGGTGTGTTCCACAATTGCCCCAGCTCTAC
TCTAATTCTCTTTTAAAAAAATGTGTGTGTGTATGTTTCTATGTATATATGTATATATAG
AAGTAATATTAATCCATTAATATCTCTTTTTCCGAAATTTCATGTCTCTCTTCTCTCTCT
CTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTTTCTTTCTTTCT
TACATTTATTCTTTTCATACATAGTTTCTTAGTGTAAGCAGCCCTGGCTGTCTTGAAGAT
ACTTTGTAGGCCTCAATCCTGTAAGAGCCTTCCTCTGCTTTTCAAATGCTGGCATGAATG
TTGTACCTCACTATGACCAGCTAAGATTCTCACCTTTTCTTTTCTTTTTTTCCCCCCCTT
TCTTTTTATGATTCTAGCTATGGCCAGTCAGTGGTGGTGGTGGTGGCAGAGGGAGGTGGA
TCACTGTAACTTTGCAGCTAGCTTAGTCTTCAGTCTGAGTTACTGGAAAGGAGTTCCAAG
AAGACTGGTTATATTTTTCATTTATTATGCATTTTTTTAAGATTTATTTATTATTATATG
TAAGTACACTGTAGCTGTCTTCAGACACACCAGAAGAGGGAGTCAGATCTTGTTATGGAT
GGTTGTGAGCCACCATGTGGTTGCTGGGATTTGAACTCTGGACCTTTGGAAGAGCAGTCG
GGTGCTCTTACCCACTGAGCCATCTCACCAGCCCTTATTATGCATTTTAATTAAGATTTA
ATTACACCAAAAGAATTTAGACTGACCAATTCAGAGTCTGCCGTTTAAAAGCATAAGGAA
AAAGTAGGAGAAAAACGTGAGGCTGTCTGTGGATGGTCGAGGCTGCTTTAGGGAGCCTCC
TCACCATTCTGCACTTGCAAACCGGGCCACTAAAACCCGGTGAAGGGAGAAACCAAAGCG
ACCTGAAACAATAGGTCACATGAAGGCCAGCCACCTCCATCTTGTTGTGCGGGAGTTCAG
TTAGCAGACAAGATGGCTGCCATGCACATGTTGTCTTTCAGCTTGGTGAGGTCAAAGTGC
AACCGAGTGACAGAACAAGGAAGTAGACATGCAGACAACAGACATGCAGGCGAACCATCT
GCCTTCTGTGTTTGGATAAAAGACACACAACAATTTTCATTACTTACACTAAGCCTTAAA
AAGCACTCTGACAGCCTTGCATATATCTACATTCTATGTGATTAGAATTTCCTTTTTTGT
TGTTTTTTCATTTATTCTCAGGAGCTGCATGGTGGTTTCTCCTAGGTTCTGTTGCCCTCT
CCTGGCATGCAGGTATATACGCAGCAGATAAAGCACAAACACACACTTACATTGAAAACT
ATTTGTAATAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAG
AAAGGGAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGGAAGGAAG
GAAGGATTTTCAGACATATGATTCTTCTAGTACTATTGACTATCGAATTTTGATCATGAA
GGCACATTGGATTTTTGTTTTTTGTTTTTTTGCTTGTTTTGGTTTTTCGAGACAGGGTTT
CTCTGTATAGCCCTGGCTGTCCTGGCACATTGGATTTTATGACAGAGTGTGTGTGCATGT
GTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTTTGATACTTCTGCTGTGATTGGGGTA
ACCTGCAATCTAGTGGGCTTATGTTTTATTTCTGTAATCTTTTTTCTAATTCTTTTTCAT
TTCTATTTCTTATTTTATTTTATTTTATTTTATTTTATTTATTTTGTTTTTTTTTTTTCC
AGACAGGGTTTCTCCCTATAGCCCTGGCTGTTCTGGAACTCACTTTGCCGACCATGGCCA
TGGTGGCCTTGAATTCAGAAATCCTCCCATCCCTGCCTCCCTAGGACTGGGAATAAAGGC
TTGAGCCACCACTCCCGGGTATATTTAATTTTTTTAATTTTTAAATTTTATTTGTTTTAT
TTATTTTTCTTTTTTTTCAAGATGTGGTTTCTCTGTGTAACCCTGGCTGTCCTGGAACTC
ACTCTGTAGACCAGGCTGGCCTTGAACCATTAATCTGAACGTTTTTGCTTCCCAAATGCT
GGGATTAGAGGCATGTGACACCACTGCCCGGATTAATTTTATTTGTTTGAAAATACCTTC
CTTCCTTCCTCCTTCCTTCCTCCCTTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTT
CTTTCTTTCTTTCTTTCTTTCTTTCTTTTTTCTCTCTCTCTTTCTCTCTTTCTATTAGTT
TTTTTTTTGTTTTGTTTTGTTTTTTTGGAGATATGGTTTCTCTAGTCCTAGCTGACTTGT
AAGCCAGGGTGACCTTGAAAGCAGAAATCTGCCTTCTTTCTTTCTTTCTTTCTTTCTTTC
TTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTCCAAAAATGCTCAGGGACATCACTCT
CCCTCCAGCCAGAGAAGAGTCTCAGGAAATGTATGTGAGCTTGCTTTGCTTTGTTTTGTG
GGGCCAAATGTGACAGACATTTCTTTTGGCTTTGTGTGTTTTGCCAGCCAGCTCTCTTGC
TCTGCTCACTCACTCACTCACTCACTCACTCACTCACTCACTCATTCACTCATTAGCTTG
CTTACTGTGGTTTTCACTTCTATCTATCTCTGCAGGGTTCCAAGACTGCCCTTTGGGATT
CAAACGGGAGCCGCTTTGGACTGACTGGCTGCCTTCCTCCTCCTCCCCCTCCTCCCCCTT
CTGCTTCAGATTTATTTATTATATGTAAGTACACTGTAGCAGTTGTAGGACACACTAGAC
GAGGGCACCAGATCTCATTGTGGGTGGTTGTGAACCACCCACCATGTGGTTGCTGGGATT
TGAACTCAGGATCTTCAGAAGAGCAGTCAGGGCTCTAAACCGCTGAGCCATCTCTCCAGC
CTCCTGCATTCCTTCTTAAGGCATGAATGATCCCAGCATAGGAAGACAGTCTGCCCTCCC
TCCTTTTCGAGCCATTTTCCCTTTCACCATATACTCAATAAAATCAGTAAATGAATGAAG
TCTCCAGCTATTTATTTCTTCGAGCTATCTAAATTCTCTCACAGCACCTCCCCCTCCCCC
ACACTGCCTTTTCTCCCTATGTTTGGGTGGGGCTGGGGGAGGGGTGGGGTGGGGGGCAGG
GATCTGCATGTCTTCCTTGCAGGTCTGTGAACTATTTGCGATGGCCTGGTTCTCTGAACT
GTTGAGCCTTGTCTATCCAGAGGCTGACTGGCTAGTTTTCTGCCTGAAGTCCCTGAGTGA
TGATTTCCCTGTGAATTCTCTGAACTCACTGTCCTTTCAGGTCCACATTTGCTCTGGCTC
TGGCAACTGACTGAACCAAACAATTAGGCTCATACCTAGAACTTTGGAAATGGTAATGGA
GTTGATAACACACCCTAACAGAAAGAGGAATGCTCCACTAACTAGTACACTTTCACTAGA
AGCCTCACCCTGTCAGCAGTTTAGGAATTTGTAAAGCCAATCTGGACAATATTTAAATGG
TTTGGAATTTCTACCTTCATGTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TGAGACAGGGTTTCTCTGTATAGCCCTGGCTGTCCTGGCACATTGGATTTTATAACAGAC
TTTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTAATACTTCTCCTG
TGATTGCCGTAACTTGCAATCTAGTGGGCTTATGTTTGATTCCTGTAATCTTTTTTTCTA
ATTCTTTTCCCTTTTTATATTTTATTTTATTTTATTTTATTTTATTTTATTTTATTTTAT
TTTATTTTGGTTTTTGAGACAGGGTTTCTCTGTATAGCCCTGGCTGTCCTGGAACTCAGT
TTGCAGACCAGGCTGGCCTAGAACTCAGAAATCCTCCCATCTCTGCCTCCCGAGTGCTGA
GATTAAAGGCGTGCACCACCACACCCGGCTATTTTTAAATTTTTTTATTTTTAAATTTTA
TTTTTATTTTTATTTATTTATTTATTTATTTTTGTTTTTTCTAGACACGGTTGGTTTCTC
TGTATAACTCTGGCTGTCCTGGAACTCACTTCATAGACCAGGCTGGCCTTGATCTCAGAA
ATCCGCCTGCCTCTGCCTCCCAAGTGCTGGGATTAAAGGTGTGCTTTACGCCCGGCTTTT
ATTTTTAAATTTTATTTATTTAATTTATTTTTGTTTATCACTGTGTAGACCAGGCTGGCC
TTGAACACTTTAATCTGTCTGCCTCTGCTTCCCAAATGCTGGGATTAAAAGCATCTGCCA
CCACTGCCCTGACTAATTTTTTTTTTTTTTTTTTTTTAGAAAATACCTTTCTTCCTACCT
TCCTTTCTTCCTTCCTTCCTTCCTTCCTTCCTTCCTTTCTTTCTTTCATTATTTTTTTCT
TTTTTTTGAGATATGGTTTCTCTGTGTAGCCCTAGCTGCCTTGTAATCCAAGGTGACCTT
GAACGCAGAAATCTGCCTGAGGTTAAAGGTGTGTTCCACAATTGCCCCAGCTCTACTCTA
ATTCTCTTTTTAAAAATATGTGTCTTTGTATGTATGTATATATCGAAGTATTCATGTCAT
TCCTGGTCCACAAAGTGAGTTCCAGGAGTACCCAGAGAAACCCTGTCTCCAAAAAAAAAA
AAAAAAAAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAGGAATCCTAACCATC
ACTACCATCACCACCAATAAAAAGGCTGTGGAAACTATTTCAGGTCCCTCTCCCCCACAG
GTGAGGATGTAGCTGCTTGTAAATGAATACGCCGTGTGTGTGTGTGTGTGTGTGTGTGTG
TGTGTGTGTGTGTGTGTGTGTATGTGTTTCCATGGCCAGCTAGTACCAACCCCAGATTTG
TGTTTCAGAGGTCTTTACTTATTTATTTCTCATGCCTGAGCTTGTGTAGGTGGTAGGCCA
CCACACCTCCAGAAAAACAACATAAAATCAACTAACAAACCAACCCGAAAATGTACAAGA
ATGGATTCATCTGGCATTGAAAAGTAATAGACTGCCGGGCAGTGGTGTAACATGCCTTTA
GTCCTACCTACTACTTTGTAGGCAGAGGCAGACAAATTTCTGCGTTCCAAGCCAGCCTGG
TCTACAGAGTAAGTTCCAGGACAGCGAGGGCAACACAGAAAAACACTGTCTTGATCTCAG
CACTGGGAAGAGGCAGAGGCAGAACTTACTTGAGAGGCCGGCCTGGTCTGTCTTGTATGT
TTTTGCACCATGTACAGAGACTTTGTCTCTCATAAATAAATGAATACAGAGAAAGATATT
CATTGATCTGAACATGCATGTGCGATGGCTCAGAGGGCAAAACTGCTGGCTGTTAAGGTA
TAGGTAAAATATTAAGGCCTAGATGGAATGTGAAGGCCTACTTCAGATTAGATAAGGCCT
AGGTAACCTCTTACCTGGCTAGCCTGCCATTTTGTGCTGTTAACTACTATTATAAGGACA
CTTTCTCATAGCTTGTTTCTGCCATTATTGCCAGAATTGACTACATATTCTTGTTGGATG
TGTCCTCGGAAACCAATCAGAATAGAAGCTTCCTTTGTTCTATAGTTATCTGCAATAGGC
TTGGACCCAAACTGACCACCCAGCAGCATTTCCTGCACAAACCCATGTATACACTTTTCT
GGCATTTGCTCTTAGAAGCTGCCCTGGGGACGGACCCCCGCACCCCAACAAAAGAGAGAC
ACCCGCACCACCGATTCAGTGCAGAGACTACTTCGAACCAGACTTCTATTTTGAGTAGGG
TGTGAGTGAGTAAAGTATAAGTTCTTGAGGCCTCCCTGGGAACTGACATCTTACTTGGAA
GAAAGAGACTTGTACATGCACAACTGACTTTCTGGTTTTCAAGTTAAGACATGGCAAGTA
CCCTTGCCACAGTAGAGTACCCCAATTAATGGGAGCAGGATAAGTATACAACAAGGACAT
GTTCCTAAGGAATTCCCCTATCCCTAATCCAGATTGGTGGAATAACTTGGTATAGATGTT
TGTGCATTAAAAACCCTGTAGGATCTTCACTCTAGGTCACTGTTCAGCACTGGAACCTGA
ATTGTGGCCCTGAGTGATAGGTCCTGGGACATATGCAGTTCTGCACAGACAGACAGACAG
ACAGACAGACAGACAGACAGACAGACGTACAAACAAACACGTGAGCGTGTGCAACACACA
CACAAACACCACTCCTGGCCATAATTATTGAGGACGTGATTTTATTATTCTGTGTTTGTG
AGTCTGTCTGTCTGTCTGTCTGTCTGTCTGTCTATCAAACCAAAAGAAACCAAACAATTA
TGGCTGCCTGCCTGCCTGCCTGCCTACACAGAGAAATGATTTCTTCAATCAACTAAAACG
ACCTCCTAAGTTTGCCTTTTTTCTCTTTCTTTATCTTTTTCTTTTTTCTTTTCTTCTTCC
TTCCTTCCTTCCTTCCTTTCTTTCTTTCTTTCTTTCTTTCTTACTTTCTTTCTTTCTTTC
TTACATTTATTCTTTTCATACATAGTTTCTTAGTGTAAGCATCCCTGACTGTCTTGAAGA
CACTTTGTAGGCCTCAATCCTGTAAGAGCCTTCCTCTGCTTTTCAAATGCTGGCATGAAT
GTTGTACCTCACTATGACCAGCTAAGATTCTCACCTTTTCTTTTCTTTTTTTCCCCCCCT
TTCTTTTTATGATTCTAGCTATGGCCAGTCAGTGGTGGTGGTGGTGGCAGAGGGAGGTGG
ATCACTGTAACTTTGCAGCTAGCTTAGTCTTCAGTCTGAGTTACTGGAAAGGAGTTCCAA
GAAGACTGGTTATATTTTTCATTTATTATGCATTTTAATTAAAATTTAATTTCACCAAAA
GAATTTAGACTGACCAATTCAGAGTCTGCCGTTTAAAAGCATAAGGAAAAAGTAGGAGAA
AAACGTGAGGCTGTCTGTGGATGGTCGAGGCTGCTTTAGGGAGCCTCGTCACCATTCTGC
ACTTGCAAACCGGGCCACTAGAACCCGGTGAAGGGAGAAACCAAAGCGACCTGAAACAAT
AGGTCACATGAAGGCCAGCCACCTCCATCTTGTTGTGCAGGAGTTCAGTTAGCAGACAAG
ATGGCTGCCATCCACATGTCACCTTTCATCTTGGTGAGGTCAATGAGCAGCCGAGTCACA
GGACAAGGAAGTAGACATGCAGACAACAGACATGCAGGCGAACCACCTCCCTTCTGTGTT
TGGATAAAAGACACACAACAATTTTTATTATTTACAGTAAGCCTTAAAAAGCACTCTGAC
AGCCACTCAGATATCTACCTTCTATATGATTAGAATTTCCTTTTTTCTCTTTTTAAATTA
ATTCGAGGTTCTGGTGCCCTCTTCTGGCATGCAGGCATATAGGCAGCAGATAAAGCCCAA
ACACACACTTAGATTGAAAACTAATTTTAATAAGAAAGAAAGAAAGAAAGAAAGAAAGAA
AGAAAGAAAGAAAGAAAGGAAGGAAGGAAGGAAGAAAGAAAGGAACGAAAGATTTTCAGA
TATATGATTCTTCTAGTTCTATTGACTGTTGAATTTTGATCATGAAGGCACATTGGATTT
TTGTTTCTTGGTTTTTTTTTTTTTTTTGTTCGTTTGTTTTGGTTTTTCGAGACAGGGTTT
CTCTGTATAGCCCTGGATAAAAGACACACAACAATTTTCATTACTTACACTAAGCCTTAA
AAAGCACTCTGACAGCCTCGCATATATCTACATTCTATGTGATTAGAATTTCCTTTTTTT
GTTGTTTTTTCACTTATTCTCAGGAGCTGCATGGTGGTTTCTCCTAGGTTCTGTTGCCCT
CTCCTGGCAAGCAGGCCTATAGGCAGCAGATAGAGCACAAACACACACTTACATTGAAAA
CTAATTGTAATAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAGAGA
GAAAGAGAGAAAGAAATAAAGATCCTAGGTTCGACTCGGTTGTGTGGGTGGCAGAAGAAG
TGAGGCAGGCAGGCAGGCAGGCAGGCAGGCAGGCAGGCAGGCAGGCAGGCAGAGAGGACC
TCTCTGAGTTTGAGGCTACCAAAGCCTACAAAGCAAGTTCCAGGCAGGGATGCCAGGGAT
ACACTGAGAGAAACCCTATCTTGGGGCAAAAGAAAACAAAAGCAGAAAAGGTTCTGGCTT
TTATTTTCAGACATATGATTCTTCTAGTATTACTGACTATCGAAATTTGATCATGAAGGC
ACATTGGATTTTATGACAGAGTCTGTGTGTGTGTGTGTGTATAATATTTCTGCTATGATT
GCAGTTACTTGCCATCTCGTGGGCTTATGTTTGATTTCTGTAGTTTTTTAAAATTCTTTA
AAATTTTTATTTTATATATTTTTTAGTTTAGTTTAGTTTAATTTAGTTTAGTTTTTCAAG
ACAGGGTTTCTCTGTATAGCCCTGACTGTCCTGGAACTCACTTTGCAGACCAGGCTGGCC
TCAAACTCAGAAATCCTCCCATCTCTGCCTGAAGAGAGCTGGGATTAAAGACATGCGCCA
TCACTCCCGGCTATTTTTAAATTTTTAAATTATATTTATTTAATTTATTTTTGTTTTTTT
CAAGATGTGGTTTCTCTGTGTAACTCTGGCTGACCTGGAACTCACTGTGTAGACCAGGCT
GGCCTTGAACACATTAATCTGTCTGCCTCTGCTTCCCTAATGCTGCGATTAAAGGCATGT
GCCACCACATGCCCGGACTGATTTCTTCTTTTTTTTTTTTTTGGAAAATACCTTTCTTTC
TTTTTCTCTCTCTCTTTCTTCCTTCCTTCCTTCCTTCCTTTCTTTCTATTCTTTTTTTCT
TTCTTTTTTCTTTTTTTTTTTTTTAAAATTTGTTTGAAAATACCTTCCTTCCTTCCTCCT
TCCTTCCTCCCTTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTC
TTTCTTTCTTTCTTTTTTCTCTCTCTCTTTCTCTCTTTCTATTAGTTTTTTTTTTGTTTT
GTTTTGTTTTTTTGGAGATATGGTTTCTCTAGTCCTAGCTGACTTGTAAGCCAGGGTGAC
CTTGAAAGCAGAAATCTGCCTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCT
TTCTTTCTTCTTTCTTTCTTTCTTCCAAAAATGCTCAGGGACATCACTCTCCCTCCAGCC
AGAGAAGAGTCTCAGGAAATGTATGTGAGCTTGCTTTGCTTTGTTTTGTGGGGCCAAATG
TGACAGACATTTCTTTTGGCTTTGTGTGTTTTGCCAGCCAGCTCTCTTGCTCTGCTCACT
CACTCACTCACTCACTCACTCACTCACTCACTCATTCACTCATTAGCTTGCTTACTGTGG
TTTTCACTTCTATCTATCTCTGCAGGGTTCCAAGACTGCCCTTTGGGATTCAAACGGGAG
CCGCTTTGGACTGACTGGCTGCCTTCCTCCTCCTCCCCCTCCTCCCCCTTCTGCTTCAGA
TTTATTTATTATATGTAAGTACACTGTAGCAGTTGTAGGACACACTAGACGAGGGCACCA
GATCTCATTGTGGGTGGTTGTGAACCACCCACCATGTGGTTGCTGGGATTTGAACTCAGG
ATCTTCAGAAGAGCAGTCAGGGCTCTAAACCGCTGAGCCATCTCTCCAGCCTCCTGCATT
CCTTCTTAAGGCATGAATGATCCCAGCATAGGAAGACAGTCTGCCCTCCCTCCTTTTCGA
GCCATTTTCCCTTTCACCATATACTCAATAAAATCAGTAAATGAATGAAGTCTCCAGCTA
TTTATTTCTTCGAGCTATCTAAATTCTCTCACAGCACCTCCCCCTCCCCCACACTGCCTT
TTCTCCCTATGTTTGGGTGGGGCTGGGGGAGGGGTGGGGTGGGGGGCAGGGATCTGCATG
TCTTCCTTGCAGGTCTGTGAACTATTTGCGATGGCCTGGTTCTCTGAACTGTTGAGCCTT
GTCTATCCAGAGGCTGACTGGCTAGTTTTCTGCCTGAAGTCCCTGAGTGATGATTTCCCT
GTGAATTCTCTGAACTCACTGTCCTTTCAGGTCCACATTTGCTCTGGCTCTGGCAACTGA
CTGAACCAAACAATTAGGCTCATACCTAGAACTTTGGAAATGGTAATGGAGTTGATAACA
CACCCTAACAGAAAGAGGAATGCTCCACTAACTAGTACACTTTCACTAGAAGCCTCACCC
TGTCAGCAGTTTAGGAATTTGTAAAGCCAATCTGGACAATATTTAAATGGTTTGGTATTT
CTACCTTCATGTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTGAGACGGGG
TTTCTCTGTATAGCCCTGGCTGTCCTGGCACATTGGATTTTATAACAGACTTTGTGTGTG
TGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTAATACTTCTCCTGTGATTGCCGT
AACTTGCAATCTAGTGGGCTTATGTTTGATTCCTGTAATCTTTTTTTCTAATTCTTTTTC
CTTTTTATATTTTATTTTATTTTATTTTATTTTATTTTATTTTATTTTATTTTATTTTGG
TTTTTGAGACAGGGTTTCTCTGTATAGCCCTGGCTGTCCTGGAACTCAGTTTGCAGACCA
GGCTGGCCTAGAACTCAGAAATCCTCCCATCTCTGCCTCCCGAGTGCTGAGATTAAAGGC
GTGCACCACCACACCCGGCTATTTTTAAATTTTTTTATTTTTAAATTTTATTTTTATTTT
TATTTATTTATTTATTTATTTTTGTTTTTTCTAGACACGGTTGGTTTCTCTGTATAACTC
TGGCTGTCCTGGAACTCACTTCATAGACCAGGCTGGCCTTGATCTCAGAAATCCGCCTGC
CTCTGCCTCCCAAGTGCTGGGATTAAAGGTGTGCTTTACGCCCGGCTTTTATTTTTAAAT
TTTATTTATTTAATTTATTTTTGTTTATCACTGTGTAGACCAGGCTGGCCTTGAACACTT
TAATCTGTCTGCCTCTGCTTCCCAAATGCTGGGATTAAAAGCATCTGCCACCACTGCCCT
GACTAATTTTTTTTTTTTTTTTTTTTTAGAAAATACCTTTCTTCCTACCTTCCTTTCTTC
CTTCCTTCCTTCCTTCCTTCCTTCCTTTCTTTCTTTCATTATTTTTTTCTTTTTTTTGAG
ATATGGTTTCTCTGTGTAGCCCTAGCTGCCTTGTAATCCAAGGTGACCTTGAACGCAGAA
ATCTGCCTGAGGTTAAAGGTGTGTTCCACAATTGCCCCAGCTCTACTCTAATTCTCTTTT
TAAAAATATGTGTCTTTGTATGTATGTATATATCGAAGTATTCATGTCATTCCTGGTCCA
CAAAGTGAGTTCCAGGAGTACCCAGAGAAACCCTGTCTCCAAAAAAAAAAAAAAAAAAAA
GAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAGGAATCCTAACCATCACTACCATCA
CCACCAATAAAAAGGCTGTGGAAACTATTTCAGGTCCCTCTCCCCCACAGGTGAGGATGT
AGCTGCTTGTAAATGAATACGCCGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTGTG
TGTGTGTGTGTATGTGTTTCCATGGCCAGCTAGTACCAACCCCAGATTTGTGTTTCAGAG
GTCTTTACTTATTTATTTCTCATGCCTGAGCTTGTGTAGGTGGTAGGCCACCACACCTCC
AGAAAAACAACATAAAATCAACTAACAAACCAACCCGAAAATGTACAAGAATGGATTCAT
CTGGCATTGAAAAGTAATAGACTGCCGGGCAGTGGTGTAACATGCCTTTAGTCCTACCTA
CTACTTTGTAGGCAGAGGCAGACAAATTTCTGCGTTCCAAGCCAGCCTGGTCTACAGAGT
AAGTTCCAGGACAGCGAGGGCAACACAGAAAAACACTGTCTTGATCTCAGCACTGGGAAG
AGGCAGAGGCAGAACTTACTTGAGAGGCCGGCCTGGTCTGTCTTGTATGTTTTTGCACCA
TGTACAGAGACTTTGTCTCTCATAAATAAATGAATACAGAGAAAGATATTCATTGATCTG
AACATGCATGTGCGATGGCTCAGAGGGCAAAACTGCTGGCTGTTAAGGTATAGGTAAAAT
ATTAAGGCCTAGATGGAATGTGAAGGCCTACTTCAGATTAGATAAGGCCTAGGTAACCTC
TTACCTGGCTAGCCTGCCATTTTGTGCTGTTAACTACTATTATAAGGACACTTTCTCATA
GCTTGTTTCTGCCATTATTGCCAGAATTGACTACATATTCTTGTTGGATGTGTCCTCGGA
AACCAATCAGAATAGAAGCTTCCTTTGTTCTATAGTTATCTGCAATAGGCTTGGACCCAA
ACTTGACCACCCAGCAGCATTTCCTGCACAAACCCATGTATACACTTTTCTGGCATTTGC
TCTTAGAAGCTGCCCTGGGGACGGGACCCCCGCACCCCAACAAAAGAGAGACACCCGCAC
CACCGATTCAGTGCAGAGACTACTTCGAACCAGACTTCTATTTTGAGTAGGGTGTGAGTG
AGTAAAGTATAAGTTCTTGAGGCCTCCCTGGGAACTGACATCTTACTTGGAAGAAAGAGA
CTTGTACATGCACAACTGACTTTCTGGTTTTCAAGTTAAGACATGGCAAGTACCCTTGCC
ACAGTAGAGTACCCCAATTAATGGGAGCAGGATAAGTATACAACAAGGACATGTTCCTAA
GGAATTCCCCTATCCCTAATCCAGATTGGTGGAATAACTTGGTATAGATGTTTGTGCATT
AAAAACCCTGTAGGATCTTCACTCTAGGTCACTGTTCAGCACTGGAACCTGAATTGTGGC
CCTGAGTGATAGGTCCTGGGACATATGCAGTTCTGCACAGACAGACAGACAGACAGACAG
ACAGACAGACAGACAGACGTACAAACAAACACGTGAGCGTGTGCAACACACACACAAACA
CCACTCCTGGCCATAATTATTGAGGACGTGATTTTATTATTCTGTGTTTGTGAGTCTGTC
TGTCTGTCTGTCTGTCTGTCTGTCTATCAAACCAAAAGAAACCAAACAATTATGGCTGCC
TGCCTGCCTGCCTGCCTACACAGAGAAATGATTTCTTCAATCAACTAAAACGACCTCCTA
AGTTTGCCTTTTTTCTCTTTCTTTATCTTTTTCTTTTTTCTTTTCTTCTTCCTTCCTTCC
TTCCTTCCTTTCTTTCTTTCTTTCTTTCTTTCTTACTTTCTTTCTTTCTTTCTTACATTT
ATTCTTTTCATACATAGTTTCTTAGTGTAAGCATCCCTGACTGTCTTGAAGACACTTTGT
AGGCCTCAATCCTGTAAGAGCCTTCCTCTGCTTTTCAAATGCTGGCATGAATGTTGTACC
TCACTATGACCAGCTAAGATTCTCACCTTTTCTTTTCTTTTTTTCCCCCCCTTTCTTTTT
ATGATTCTAGCTATGGCCAGTCAGTGGTGGTGGTGGTGGCAGAGGGAGGTGGATCACTGT
AACTTTGCAGCTAGCTTAGTCTTCAGTCTGAGTTACTGGAAAGGAGTTCCAAGAAGACTG
GTTATATTTTTCATTTATTATGCATTTTAATTAAAATTTAATTTCACCAAAAGAATTTAG
ACTGACCAATTCAGAGTCTGCCGTTTAAAAGCATAAGGAAAAAGTAGGAGAAAAACGTGA
GGCTGTCTGTGGATGGTCGAGGCTGCTTTAGGGAGCCTCGTCACCATTCTGCACTTGCAA
ACCGGGCCACTAGAACCCGGTGAAGGGAGAAACCAAAGCGACCTGAAACAATAGGTCACA
TGAAGGCCAGCCACCTCCATCTTGTTGTGCAGGAGTTCAGTTAGCAGACAAGATGGCTGC
CATCCACATGTCACCTTTCATCTTGGTGAGGTCAATGTGCAGCCGAGTGACAGGACAAGG
AAGTAGACATGCAGACAACAGACATGCAGGCGAACCACCTCCCTTCTGTGTTTGGATAAA
AGACACACAACAATTTTTATTATTTACAGTAAGCCTTAAAAAGCACTCTGACAGCCACTC
AGATATCTACCTTCTATATGATTAGAATTTCCTTTTTTCTCTTTTTAAATTAATTCGAGG
TTCTGGTGCCCTCTTCTGGCATGCAGGCATATAGGCAGCAGATAAAGCCCAAACACACAC
TTAGATTGAAAACTAATTTTAATAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAA
AGAAAGAAAGAAAGGAAGGAAGGAAGGAAGAAAGAAAGGAACGAAAGATTTTCAGATATA
TGATTCTTCTAGTTCTATTGACTGTTGAATTTTGATCATGAAGGCACATTGGATTTTTGT
TTCTTGGTTTTTTTTTTTTTTTTGTTCGTTTGTTTTGGTTTTTCGAGACAGGGTTTCTCT
GTATAGCCCTGGCTGTCCTGGCACATTGGATTTTATAAAAGAGTTTGTGTGTGTGTGTGT
GTGTGTGTGTGTGTGTGTAATACTTCTGTGATTGCGGTAACTTGCAATCTAGTGGGCTTA
TGTTTGATTTCTGTAATATTTTTTTCCAATTTTTTTTCCTTTTTATATTTTATTTTATTT
TATTTTATTTTATTTTATTTTATTTTATTTTATTTTATTTTATTTTGAGACAGGGTTTCT
CTGTATAGCCCTGGCTGTCCTGGAACTCACTTTGCAGACCAGGCTGGCCACAAACTCAGA
AATCCTCCCATCCCTGCCTTCCTAGGGCTGGGATTAAAGGCTTGAGCCACCACTCCCCGG
TATTTTTTAATTTTTTAATTGTATTTATTAAATTCATTTTTGTTTGTTTTTTTTTCATGA
TGCAGTTTCTCTGTGTAACACTGGCTGTCCTGGAACTCACTCTGTAGGCCAGGCTGGCCT
TGAACCCTAATCTGTCTGTTTCTGCTTCCCAAGTGCTGGGATTAAATGCATGTGCCACCA
CTGCCCGGACTAATTTTATTTGTTTGAAAATACCTTCCTTTCTTTCTTTCTTTCTTTCTT
TCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTTTCTCTTTC
TATCTTTCTGTTTTTCTATTTTTTTTTTTCGTTTTTTCCTGAGATATGGTTTCTCTAGCC
CTAGCTGACTTGTAAACCAGGGTGACCTTGAACGCAGAAATCTGCCTGAGGTTAAAGGTG
TGTTCCACAATTGCCCCAGCTCTACTCTAATTCTCTTTAAAAAAATGTGTGTGTGTGTAT
GTATGTATATACGTATATATAGTAGTAATATTAATCCATTAATATCTCTTTTTCCAAAAA
TCATCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTCTTTCTTTCTTACATTTATT
CTTTTCATACATAGTTTCTTAGTGTAAGCAGCCCTGGCTGTCTTGAAGATACTTTGTAGG
CCTCAATCCTGTAAGAGCCTTCCTCTGCTTTTCAAATGCTGGCATAAATGTTGTACCTCA
CTATGACCGGCTGAGATTCTCACCTTTTCTTTTCTCCCCTCCCCCTTTATTTTTTTTTTA
AAGATTTATTTATTTATTATATGTAAGTACACTGTAGCTGTCTTCAGACACACCAGAAGA
GGACATCAGATCTCATTACGGGTGGTTGTGAGCCACCGTGTGGTTGCTGGGATTTGAACT
TCGGACCTTCGGAAGAGCAGTCGGGTGCTCTTACCCACTGAGCCATCTCACCAGCCCTCC
CCCTTCTTTTTATGATTTTAGCTTTGGCCAGGCAGTGGTGGTGGTGGTGGTGGTGGTGTT
GGTGGTGGCAGAGGGAGGTGGATCACTGTAACTTTGCAGCTAGCTTAGTCTTCAGTCTGA
GTTACTCGAAAGGAGTTCCAGGAAGACTGGTTATATTTTTCATTTATTATGCATTTTAAT
TAAGATTTAATTACACCAAAAGAATTTAGACTGACCAATTCAGAGTCTGCCGTTTAAAAG
CATAAGGAAAAAGTAGGAGAAAAACGTGAGGCTGTCTGTGGATGGTCGAGGCTGCTTTAG
GGAGCCTCGTCACCATTCTGCACTTGCAAACCGGGCCACTAGAACCCGGTGAAGGGAGAA
ACCAAAGCGACCTGAAACAATAGGTCACATGAAGGCCAGCCACCTCCATCTTGTTGTGCG
GGAGTTCAGTTAGCAGACAAGATGGCTGCCATGCACATGTTGTCTTTCAGCTTGGTGAGG
TCAAAGTACAACCGAGTGACAGAACAAGGAAGTAGACATGCAGACAACAGACATGCAGGC
GAGCCACCTCTCTTCTGTGTTTGGATAAAAGACACACAACAATTTTCATTACTTACACTA
AGCCTTAAAAAGCACTCTGACAGCCTCGCATATATCTACATTCTATGTGATTAGAATTTC
CTTTTTTTGTTGTTTTTTCACTTATTCTCAGGAGCTGCATGGTGGTTTCTCCTAGGTTCT
GTTGCCCTCTCCTGGCAAGCAGGCCTATAGGCAGCAGATAGAGCACAAACACACACTTAC
ATTGAAAACTAATTGTAATAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAAAGAA
AGAAAGAAAGAAGAGAGAAAGAGAGAAAGAAATAAAGATCCTAGGTTCGACTCGGTTGTG
TGGGTGGCAGAAGAAGTGAGGCAGGCAGGCAGGCAGGCAGGCAGGCAGGCAGGCAGGCAG
GCAGGCAGAGAGGACCTCTCTGAGTTTGAGGCTACCAAAGCCTACAAAGCAAGTTCCAGG
CAGGGATGCCAGGGATACACTGAGAGAAACCCTATCTTGGGGCAAAAGAAAACAAAAGCA
GAAAAGGTTCTGGCTTTTATTTTCAGACATATGATTCTTCTAGTATTACTGACTATCGAA
ATTTGATCATGAAGGCACATTGGATTTTATGACAGAGTCTGTGTGTGTGTGTGTGTATAA
TATTTCTGCTATGATTGCAGTTACTTGCCATCTCGTGGGCTTATGTTTGATTTCTGTAGT
TTTTTAAAATTCTTTAAAATTTTTATTTTATATATTTTTTAGTTTAGTTTAGTTTAATTT
AGTTTAGTTTTTCAAGACAGGGTTTCTCTGTATAGCCCTGACTGTCCTGGAACTCACTTT
GCAGACCAGGCTGGCCTCAAACTCAGAAATCCTCCCATCTCTGCCTGAAGAGAGCTGGGA
TTAAAGACATGCGCCATCACTCCCGGCTATTTTTAAATTTTTAAATTATATTTATTTAAT
TTATTTTTGTTTTTTTCAAGATGTGGTTTCTCTGTGTAACTCTGGCTGACCTGGAACTCA
CTGTGTCTACCAGGCTGGCCTTGAACACATTAATCTGTCTGCCTCTGCTTCCCTAATGCT
GCGATTAAAGGCATGTGCCACCACTGCCCGGACTGATTTCTTCTTTTTTTTTTTTTTGGA
AAATACCTTTCTTTCTTTTTCTCTCTCTCTTTCTTCCTTCCTTCCTTCCTTCCTTTCTTT
CTATTCTTTTTTTCTTTCTTTTTTCTTTTTTTTTTTTTTAAAATTTGCCTAAGGTTAAAG
GTGTGCTCCACAATTGCCTCAGCTCTGCTCTAATTCTCTTTAAAAAAAAACAAACAAAAA
AAAAACCAAAACAGTATGTATGTATGTATATTTAGAAGAAATACTAATCCATTAATAACT
CTTTTTTCCTAAAATTCATGTCATTCTTGTTCCACAAAGTGAGTTCCAGGACTTACCAGA
GAAACCCTGTGTTCAAATTTCTGTGTTCAAGGTCACCCTGGCTTACAAAGTGAGTTCCAA
GTCCGATAGGGCTACACAGAAAAACCATATCTCAGAAAAAAAAAAAGTTCCAAACACACA
CACACACACACACACACACACACACACACACACACACACACACACACAGGCGCGCGCGCG
GCGATGAGGGGAAGTCGTGCCTAAAATAAATATTTTTCTGGCCAAAGTGAAAGCAAATCA
CTATGAAGAGGTACTCCTAGAAAAAATAAATACAAACGGGCTTTTTAATCATTCCAGCAC
TGTTTTAATTTAACTCTGAATTTAGTCTTGGAAAAGGGGGCGGGTGTGGGTGAGTGAGGG
CGAGCGAGCAGACGGGCGGGCGGGCGGGTGAGTGGCCGGCGGGCGGTGGCAGCGAGCACC
AGAAAACAACAAACCCCAAGCGGTAGAGTGTTTTAAAAATGAGACCTAAATGTGGTGGAA
CGGAGGTCGCCGCCACCCTCCTCTTCCACTGCTTAGATGCTCCCTTCCCCTTACTGTGCT
CCCTTCCCCTAACTGTGCCTAACTGTGCCTGTTCCCTCACCCCGCTGATTCGCCAGCGAC
GTACTTTGACTTCAAGAACGATTTTGCCTGTTTTCACCGCTCCCTGTCATACTTTCGTTT
TTGGGTGCCCGAGTCTAGCCCGCGTTCGCTATGTTCGGGCGGGACGATGGGGACCGTTTG
TGCCACTCGGGAGAAGTGGTGGGTGGGTACGCTGCTCCGTCGTGCGTGCGTGAGTGCCGG
AACCTGAGCTCGGGAGACCCTCCGGAGAGACAGAATGAGTGAGTGAATGTGGCGGCGCGT
GAGCGATCTGTATTGGTTTGTATGGTTGATCGAGACCATTGTCGGGCGACACCTAGTGGT
GACAAGTTTCGGGAACGCTCCAGGCCTCTCAGGTTGGTGACACAGGAGAGGGAAGTGCCT
GTGGTGAGGCGACCAGGGTGACAGGAGGCCGGGCAAGCAGGCGGGAGCGTCTCGGAGATG
GTGTCGTGTTTAAGGACGGTCTCTAACAAGGAGGTCGTACAGGGAGATGGCCAAAGCAGA
CCGAGTTGCTGTACGCCCTTTTGGGAAAAATGCTAGGGTTGGTGGCAACGTTACTAGGTC
GACCAGAAGGCTTAAGTCCTACCCCCCCCCCCCTTTTTTTTTTTTTTCCTCCAGAAGCCC
TCTCTTGTCCCCGTCACCGGGGGCACCGTACATCTGAGGCCGAGAGGACGCGATGGGCCC
GGCTTCCAAGCCGGTGTGGCTCGGCCAGCTGGCGCTTCGGGTCTTTTTTTTTTTTTTTTT
TTTTTTTTCCTCCAGAAGCCTTGTCTGTCGCTGTCACCGGGGGCGCTGTACTTCTGAGGC
CGAGAGGACGCGATGGGCCCCGGCTTCCAAGCCGGTGTGGCTCGGCCAGCTGGAGCTTCG
GGTCTTTTTTTTTTTTTTTTTTTTTTTTTTTCTCCAGAAGCCTTGTCTGTCGCTGTCACC
GGGGGCGCTGTACTTCTGAGGCCGAGAGGACGCGATGGGTCGGCTTCCAAGCCGATGTGG
CGGGGCCAGCTGGAGCTTCGGGTTTTTTTTTTTCCTCCAGAAGCCCTCTCTTGTCCCCGT
CACCGGGGGCGCTGTACTTCTGAGGCCGAGAGGACGTGATGGGCCCGGGTTCCAGGCGGA
TGTCGCCCGGTCAGCTGGAGCTTTGGATCTTTTTTTTTTTTTTTCCTCCAGAAGCCCTCT
CTTGTCCCCGTCACCGGGGGCACCTTACATCTGAGGGCGAGAGGACGTGATGGGTCCGGC
TTCCAAGCCGATGTGGCGGGGCCAGCTGGAGCTTCGGGTTTTTTTTTTTTCCTCCAGAAG
CCCTCTCTTGTCCCCGTCACCGGGGGCGCTGTACTTCTGAGGCCGAGAGGACGTGATGGG
CCCGGGTTCCAGGCGGATGTCGCCCGGTCAGCTGGAGCTTTGGATCATTTTTTTTTTTCC
CTCCAGAAGCCCTCTCTTGTCCCCGTCACCGGGGGCACCGTACATCTGAGGCCGAGAGGA
CACGATGGGCCTGTCTTCCAAGCCGATGTGGCCCGGCCAGCTGGAGCTTCGGGTCTTTTT
TTTTTTTTTTCCTCCAGAAGCCTTGTCTGTCGCTGTCACCCGGGGCGCTGTACTTCTGAG
GCCGAGAGGACGCGATGGGCCCGGCTTCCAAGCCGGTGTGGCTCGGCCAGCTGGAGCTTC
GGGTCTTTTTTTTTTTTTTTTTTTTTCCTCCAGAAACCTTGTCTGTCGCTGTCACCCGGG
GCGCTTGTACTTCTGATGCCGAGAGGACGCGATGGGCCCGTCTTCCAGGCCGATGTGGCC
CGGTCAGCTGGAGCTTTGGATCTTTTTTTTTTTTTTTTCCTCCAGAAGCCCTCTCTTGTC
CCCGTCACCGGGGGCACCTTACATCTGAGGCCTAGAGGACACGATGGGCCCGGGTTCCAG
GCCGATGTGGCCCGGTCAGCTGGAGCTTTGGATCTTTTTTTTTTTTTTCTTCCAGAAGCC
CTCTTGTCCCCGTCACCGGTGGCACTGTACATCTGAGGCGGAGAGGACATTATGGGCCCG
GCTTCCAATCCGATGTGGCCCGGTCAGCTGGAGCTTTGGATCTTATTTTTTTTTTAATTT
TTTCTTCCAGAAGCCCTCTTGTCCCTGTCACCGGTGGCACGGTACATCTGAGGCCGAGAG
GACATTATGGGCCCGGCTTCCAGGCCGATGTGGCCCGGTCAGCTGGAGCTTTGGATCTTT
TTTTTTTTTTTTCTTTTTTCCTCCAGAAGCCCTCTCTGTCCCTGTCACCGGGGGCCCTGT
ACGTCTGAGGCCGAGGGAAAGCTATGGGCGCGGTTTTCTTTCATTGACCTGTCGGTCTTA
TCAGTTCTCCGGGTTGTCAGGGTCGACCAGTTGTTCCTTTGAGGTCCGGTTCTTTTCGTT
ATGGGGTCATTTTTGGGCCACCTCCCCAGGTATGACTTCCAGGCGTCGTTGCTCGCCTGT
CACTTTCCTCCCTGTCTCTTTTATGCTTGTGATCTTTTCTATCTGTTCCTATTGGACCTG
GAGATAGGT
>gi|396138|emb|X71804.1| M.musculus gene for 5S rRNA (436 bp)
GAGCTCGTCGTCGTCGTCGGGCATCGGGCGTCGGGCGTCGGGCGGGCGAGTGAGCGAGCGAGCGAGAGCG
CAGCAGGCTCTTGGGGCTTGTCGGCGCCGCCCGCCCTCGTCTACGGCCATACCACCCTGAACGCGCCCGA
TCTCGTCTGATCTCGGAAGCTAAGCAGGGTCGGGCCTGGTTAGTACTTGGATGGGAGACCGCCTGGGAAT
ACCGGGTGCTGTAGGCTTTTTGGACTCCCCTCTGTCTCTCTCTCCCTTTTGGCCACGGCCACGGCAACCC
GGCGACCCAGGCTCAGGCCAGCGCCATCCACACCCCACAACCCCCCAAGGAGCCCACCCTCGTCCTCTCA
CAGGCTTGCCCTCTCACGGGAGGCTCCTGGAGCCCTAAGACAGGGCACACCACGCTGGGCACCTCGGCAC
CACCCTGCTGAGTTAG
>poly-A
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>poly-C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
>poly-G
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
>poly-T
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
"""

    seq['rattus_norvegicus'] = """>gi|2624399|emb|V01270.1| Rattus norvegicus genes for 18S, 5.8S, and 28S ribosomal RNAs
TACCTGGTTGATCCTGCCAGTAGCATATGCTTGTCTCAAAGATTAAGCCATGCATGTCTAAGTACGCACG
GCCGGTACAGTGAAACTGCGAATGGCTCATTAAATCAGTTATGGTTCCTTTGGTCGCTCGCTCCTCTCCT
ACTTGGATAACTGTGGTAATTCTAGAGCTAATACATGCCGACGGGCGCTGACCCCCCTTCCCGTGGGGGG
AACGCGTGCATTTATCAGATCAAAACCAACCCGGTCAGCCCCCTCCCGGCTCCGGCCGGGGGTCGGGCGC
CGGCGGCTTTGGTGACTCTAGATAACCTCGGGCCGATCGCACGTCCCCGTGGCGGCGACGACCCATTCGA
ACGTCTGCCCTATCAACTTTCGATGGTAGTCGCCGTGCCTACCATGGTGACCACGGGTGACGGGGAATCA
GGGTTCGATTCCGGAGAGGGAGCCTGAGAAACGGCTACCACATCCAAGGAAGGCAGCAGGCGCGCAAATT
ACCCACTCCCGACCCGGGGAGGTAGTGACGAAAAATAACAATACAGGACTCTTTCGAGGCCCTGTAATTG
GAATGAGTCCACTTTAAATCCTTTAACGAGGATCCATTGGAGGGCAAGTCTGGTGCCAGCAGCCGCGGTA
ATTCCAGCTCCAATAGCGTATATTAAAGTTGCTGCAGTTAAAAAGCTCGTAGTTGGATCTTGGGAGCGGG
CGGGCGGTCCGCCGCGAGGCGAGCTCACCGCCCTGTCCCCAGCCCCTGCCTCTCGGCGCCCCCTCGATGC
TCTTAGCTGAGTGTCCCGCGGGGCCCGAAGCGTTTACTTTGAAAAAATTAGAGTGTTCAAAGCAGGCCCG
AGCCGCCTGGATACCGCAGCTAGGAATAATGGAATAGGACCGCGGTTCTATTTTGTTGGTTTTCGGAACT
GAGGCCATGATTAAGAGGGACGGCCGGGGGCATTCGTATTGCGCCGCTAGAGGTGAAATTCTTGGACCGG
CGCAAGACGAACCAGAGCGAAAGCATTTGCCAAGAATGTTTTCATTAATCAAGAACGAAAGTCGGAGGTT
CGAAGACGATCAGATACCGTCGTAGTTCCGACCATAAACGATGCCGACTGGCGATGCGGCGGCGTTATTC
CCATGACCCGCCGGGCAGCTTCCGGGAAACCAAAGTCTTTGGGTTCCGGGGGGAGTATGGTTGCAAAGCT
GAAACTTAAAGGAATTGACGGAAGGGCACCACCAGGAGTGGAGCCTGCGGCTTAATTTGACTCAACACGG
GAAACCTCACCCGGCCCGGACACGGACAGGATTGACAGATTGATAGCTCTTTCTCGATTCCGTGGGTGGT
GGTGCATGGCCGTTCTTAGTTGGTGGAGCGATTTGTCTGGTTAATTCCGATAACGAACGAGACTCTCGGC
ATGCTAACTAGTTACGCGACCCCCGAGCGGTCGGCGTCCCCCAACTTCTTAGAGGGACAAGTGGCGTTCA
GCCACCGAGATTGAGCAATAACAGGTCTGTGATGCCCTTAGATGTCCGGGGCTGCACGCGCGCTACACTG
AACTGGCTCAGCGTGTGCCTACCCTACGCCGGCAGGCGCGGGTAACCCGTTGAACCCCATTCGTGATGGG
GATCGGGGATTGCAATTATTCCCCATGAACGAGGAATTCCCAGTAAGTGCGGGTCATAAGCTTGCGTTGA
TTAAGTCCCTGCCCTTTGTACACACCGCCCGTCGCTACTACCGATTGGATGGTTTAGTGAGGCCCTCGGA
TCGGCCCCGCCGGGGTCGGCCCACGGCCTTGGCGGAGGCCTGAGAAGACGGTCGAACTTGACTATCTAGA
GGAAGTAAAAGTCGTAACAAGGTTTCCGTAGGTGAACCTGCGGAAGGATCATTAACGGAGAAGGCCGAGG
GGGGTCGTGCCCGTCCCCTCTTGGGCCTGTGTGAGTGTTCCTCCTTCTCGCCGGGAGGCGCGTCCCCGGG
TGGGTCCCCGTGTCCGGCGTCCTGGCGTCGTGGCGCGTGCCGTGGCCGGCCGGCGGAGGGGGTTTCGGGA
CAGGTGTGCGGTGGCAGGGTGTCGGGTCTGGTCCGCCGCGGGACCTCCTCCGTTTTCTCGCTCTTCCCTC
GACGCCTCCGTGCGCCCGCCTCCGCCTCCGCCGCTGCCGTCGGCAGCGAGGGCGTTCTGCCTCTTTCCCG
ACCGGCTCCGTGATCTCGTGTGCACCGGGGGTGGTACGTGATCTCTCCGGGCGGTGGCCGGGACGCGCTC
GCTCTCCTCGAGGCGCCTGCCGCTCCTCCCCCGCCGCGGCTCTCCGGCACTGACGGTGCCGGTGGCTCCG
CCGCGTGCGGTGGGGGGGGGTGTGTGTGCCGGTGCCGCGGGTGGGCTCGCGCGCCCCGCCGCGTCCTTCT
GGGCCTCCGTGTCCGCCCCGCTCCTGTCCGGGTACCTGCTCTCGCGTTCCGGCGCGGAGGTTTCGCGACC
CCGGGGGGGTCGCCCTACCGTCCCCGGGGTCGGGGGGGCGGTGGGGCCCGTAGGGACGTCGGTCGTCCGC
GGGAGGGCTTCCCGGTCCCTCCCCGCGTCCGGCGCTCCCTCAGACTCAGCGGCGCACCCCTCTGCCGCCG
GCGCTTCCCGAGGCGGCGGTTCGGGGGGGCGTGTCGCGAGCCCCCTCTGGGCGCCCGAGGGGTTCCCGGC
CCGAGCGCCTGGCTTGTCCCGCCCGTTGGCGGGCCGCCGCCGGGTGCCGGGTTCGTGGCGTCCCGTGTCC
CGTGTGTCCCGCCTTTCCGTCTCCGACCCTGTCTTTTTTTTTTTATTTTCTTACGTGTGCCGTTTCGTTT
CGCGCTGGCCGGCCTGAGGCGGAGCCCCCTCGCCGTCCGTCGGGCCCCTCCCATCCCGCGCAGGAGGGGC
GGGCGGGGGCGCTTGTTGCGGTCGTCAGCACCCCGTGTGCGGAGTTCCGCTCACACCTCAGATAACCGTA
CGACTCTTAGCGGTGGATCACTCGGCTCGTGCGTCGATGAAGAACGCAGCTAGCTGCGAGAATTAATGTG
AATTGCAGGACACATTGATCATCGACACTTCGAACGCACTTGCGGCCCCGGGTTCCTCCCGGGGCTACGC
CTGTCTGAGCGTCGCTTGACGATCAATCGCCGCCCCCCTTGCGGTTTCGCCGCCGCGGGGGGGATCGCGT
CGCGCGGCTGGGAGTCTGCTCGCAGGGCCCGCTCCCCGGGGCCCTCCGTCTCCCGAAGTTCCGACGTGGG
CGGCGTCCGGCGTTCGCCCGCGCCCCGTGGGTCCGCGCGGGCTCGTCCTCCCTGCCCGCCGTCGTCGTGC
GGCCGGGCGGCCCTCCTCGCGGCTCCTTCCCCGCGCGCTCCGCCGCTCCCGCTCCCGTCCTCGCCCGCTC
TCCCCCGCCGTGGGACGCGTCGGCGGGGCCGGTCGGGTGCTGGGCGCGGAGAGGGGGGGGAAAGTCGCGT
CGCCGGGGTCGTTCGCCGCGCCTCCGGTGGGGTGGGGGGCCGGGACCCCGCCCGCGGGGCCGGCGCGTCC
GCCGCCGCCGCGCGCCCGCCCGGTGCGTGCGGAGAGAGAGAGAGAGAGAGAGACCCGCGAGGGTCGTTCG
GGGGGGCCGTACCGCGCCCGCGGGGCCGGTCTCGCGTCCCTCTCCTCCCCCGCTCCCTCTACCGGGGCCG
ACCGCTCCGGCGCCGGCCCGCGGCGCCGCGGCGTCCGTGCGCCGACGCGACTTGCCCCCGGGATGTGGGG
CCGGCCGGCGAAGGAGAGGTCGCGGGCTCCGGCCTGGCTCCCCCGGCGCCCTCGCCCCGTCGTCCCGTCG
TTCTCGCTCTCGCTCTCCTCTCCTCTCCTCTCCTTCCGTCGCCCGCGCGCGCCCACCTCTCCTCCTTCTC
CTCCTCTGGACCGCGACCTCAGATCAGACGTGGCGACCCGCTGAATTTAAGCATATTAGTCAGCGGAGGA
AAAGAAACTAACCAGGATTCCCTCAGTAACGGCGAGTGAACAGGGAAGAGCCCAGCGCCGAATCCCCGCC
GCGCGCCGCGGCGCGGGAAATGTGGCGTACGGAAGACCCACTCCCCGGCGCCGCTCGTGGGGGGCCCAAG
TCCTTCTGATCGAGGCCCAGCCCGTGGACGGTGTGAGGCCGGTAGCGGCCCCGGCGCGCCGGGCCCGGGT
CTTCCCGGAGTCGGGTTGCTTGGGAATGCAGCCCAAAGCGGGTGGTAAACTCCATCTAAGGCTAAATACC
GGCACGAGACCGATAGCCAACAAGTACCGTAAGGGAAAGTTGAAAAGAACTTTGAAGAGAGAGTTCAAGA
GGGCGTGAAACCGTTAAGAGGTAAACGGGTGGGGTCCGCGCAGTCCGCCCGGAGGATTCAACCCGGCGGC
GCGCGCCGGCCGGCCGGTGGTCCCGGCGGATCTTTCCCGCTCCCCGTTCCTCCCGACCCCTCCACCCGCG
CGTCTCTCTCCCCCCTCCCCGCGTCCCGCCGTCGCCGTCCCCGCTCCTCCCTCCGGGGGGGTGTCGGCGG
GCGCTCCGGCGGCGGGCGCGGGGTGTGGTGGGGGCGCGCGGGCGGGGCCGGGGGTGGGGTCGGCGGGGGA
CCGCCCCCGGTCGGCGACCGGCCGCCGCCGGGCGCACTTCCACCGTGGCGGTGCGCCGCGACCGGCTCCG
GGACGGCTGGGAAGGCCCGGCGGGGAAGGTGGCTCGGGGGGGGCGGCGTCACCCGTGGGCGCCGGACCAC
CCCGCCCCGAGTGTTACAGCCCCCCGGCAGCAGCGCTCGCCGAATCCCGGGGCCGAGGGAGCCGGATACC
CGTCGCCGCGCTCTCCCCCCGGCCTCTCCCCTCCCGCCCCTCCCCGTGGGGTGGCGGAAAGGGGGGCGGT
CGCGGGGGCCGGGCCGCCCCTCCCACGGCGCGACCGCTCTCCCACCCCCCGTCGCCTCCGTCGTCCCTCT
CGGGGGTCCGGGGGCCCGGGGGGCGGGGCGGACTGTCCCCAGTGCGCCCCGGGCGTCGTCGCGCCGTCGG
GCCCGGGGGGGCCGTCGTCACGCGCTCTCCCTCCCCTTCTCGGGGTGGGGGGGAGCGAAGCCGAGCGCAC
GGGGTCGGCGGCGATGTCGGCTACCCACCCGACCCGTCTTGAAACACGGACCAAGGAGTCTAACGCGTGC
GCGAGTCAGGGGCTCGTCCGAAAGCCGCCGTGGCGCAATGAAGGTGAAGGGCCCCGTTCCCGGGGGCCCC
GAGGTGGGATCCCGAGGCCTCTCCAGTCCGCCGAGGGCGCACCACCGGCCCGTCTCGCCCGCCGCGCCGG
GGAGGTGGAGCACGAGCGTACGCGTTAGGACCCGAAAGATGGTGAACTATGCTTGGGCAGGGCGAAGCAG
AGGAAACTCTGGTGGAGGTCCGTAGCGGTCCTGACGTGCAAATCGGTCGTCCGACTTGGGTATAGGGGCG
AAAGACTAATCGAACCATCTAGTAGCTGGTTCCCTCCGAAGTTTCCCTCAGGATAGCTGGCGCTCTCGCA
ACGCGTTCGCTCGACAACCCGCAGTTTTATCCGGTAAAGCGAATGATTAGAGGTCTTGGGGCCGAAACGA
TCTCAACCTATTCTCAAACTTTAAATGGGTAAGAAGCCCGGCTCGCTGGCGTGGAGCCGGGCGTGGATGC
GAGTGCCTAGTGGGCCACTTTTGGTAAGCAGAACTGGCGCTGCGGGATGAACCGAACGCCGGGTTAAGGC
GCCCGATGCCGACGCTCATCAGACCCCAGAAAAGGTGTTGGTTGATATAGACAGCAGGACGGTGGCCATG
GAAGTCGGAATCCGCTAGGAGTGTGTAACAACTCACCTGCCGAATCAACTAGCCCTGAAAATGGATGGCG
CTGGAGCGTCGGGCCCATACCCGGCCGTCGCCGGCAGTCGGAACGGGACGGGAGCGGCCGCGGGCGCGCG
ACCCCGGGGGCCGGGCGGCGTCGGCTTCGGCCGGCCGCCGCCCGTCCACCCCCGGGGCTCCCCCCGCGGC
GTCGGGCCCCGCGGAGCCTACGCCGCGACGAGTAGGAGGGCCGCTGCGGTGAGCCTTGAAGCCTAGGGCG
CGGGCCCGGGTGGAGCCGCCGCAGGTGCAGATCTTGGTGGTAGTAGCAAATATTCAAACGAGAACTTTGA
AGGCCGAAGTGGAGAAGGGTTCCATGTGAACAGCAGTTGAACATGGGTCAGTCGGTCCTGAGAGATGGGC
GAGTGCCGTTCCGAAGGGACGGGCGATGGCCTCCGTTGCCCTCAGCCGATCGAAAGGGAGTCGGGTTCAG
ATCCCCGAATCCGGAGTGGCGGAGATGGGCGCCGCGAGGCGTCCAGTGCCGGTAACGCGACCGATCCCGG
AGAAGCCGGCGGGAGCCCGGGGAGAGTTCTCTTTTCTTTGTGAAGGGCAGGGCGCCTTGGAATGGGTTCG
CCCCGAGAGAGGGGCCCGTGCCTTGGAAAGCGTCGCAGTTCCGGCGGCGTCCGGTGAGCTCTCGCTGGCC
CTTGAAAATCCGGGGGAGAGGGTGTAAATCTCGCGCCGGGCCGTACCCATATCCGCAGCAGGTCTCCAAG
GTGAACGCCTCTGGCATGTTGGAACAATGTAGGTAAGGGAAGCGGCAAGCCGGATCCGTAACTTCGGGAT
AAGGATTGGCTCTAAGGGCTGGGTCGGTCGGGCTGGGGCGCGAAGGCGGGGCTGGGCGCGCGCCGCGGCT
GGACGAGGCGCCGCCGCCCCCCTCCCACGTCCGGGGAGACCCCCTCCTTTCCGCCCGGGCCCGCCCTCCC
CTCTCCCCGCGGGGCCCCGCCGTCCCCCGCGTCGTCGCCGTGGTCCCCTCCTCCCTCCCTTCTTCCCCGT
CCGCGGGGGGGACGGGGCGGGTGCGGGGGGGCGCGCGCGCGCGCGGCCCAGGGGCGGCGGGTCCAACCCC
GCGCGGGCCGGAGCGGGGGGAACCCGCGGGCCCCCGGTGGGGGGGGGCCCGGACACCCGGGGGGGACCGG
CGGCGGCGGCGACTCTGGACGCGAGCCGGGCCCTTCCCGTGGATCGCCCCAGCTGCGGCGGGCGTCGCGG
CCGCTCCCGGGGAGCCCGGCGGGTCGCCCGGCGGGGTTTTCCTCCGGCCTCGTCCTCCCCCTTCCCCCTC
CGCGGGGTCGGGGGTTCCCGGGGTTCGGGGTTCTCCTCCGCGCGGCGGTTCCCCCGCCGGGTGCGCCCCC
CGGGCGCGGTTTCCCGCGGCCGCCCGCCTCGGCCGGCGCCTAGCAGCCGACTTAGAACTGGTGCGGACTA
GGGGAATCCGACTGTTTAATTAAAACAAAGCATCGCGAAGGCCCGCGGCGGGTGTTGACGCGATGTGATT
TCTGCCCAGTGCTCAGAATGTCAAAGTGAAGAAATTCAATGAAGCGCGGGTAAACGGCGGGAGTAACTAT
GACTCTCTTAAGGTAGCCAAATGCCTCGTCATCTAATTAGTGACGCGCATGAATGGATGAACGAGATTCC
CACTGTCCCTACCTACTATCCAGCGAAACCACAGCCAAGGGAACGGGCTTGGCGGAATCAGCGGGGAAAG
AAGACCCTGTTGAGCTTGACTCTAGTCTGGCACGGTGAAGAGACATGAGAGGTGTAGAATAAGTGGGAGG
CCCCCGGCGCCCCCCCGTTCCCCGCGAGGGGTCGGGGCGGGGTCCGCCGGCCTCGCGGGCCGCCGGTGAA
ATACCACTACTCTCATCGTTTTTTCACTGACCCGGTGAGGCGGGGGGGCGAGCCCCGAGGGGCTCTCGCT
TCTGGCGCCGAACGCGTCCGCGCGCGCGGGCGGGCGCGACCCGCTCCGGGGACAGTGCCAGGTGGGGAGT
TTGACTGGGGCGGTACACCTGTCAAACGGTAACGCAGGTGTCCTAAGGCGAGCTCAGGGAGGACAGAAAC
CTCCCGTGGAGCAGAAGGGCAAAAGCTCGCTTGATCTTGATTTTCAGTACGAATACAGACCGTGAAAGCG
GGGCCTCACGATCCTTCTGACCTTTTGGGTTTTAAGCAGGAGGTGTCAGAAAAGTTACCACAGGGATAAC
TGGCTTGTGGCGGCCAAGCGTTCATAGCGACGTCGCTTTTTGATCCTTCGATGTCGGCTCTTCCTATCAT
TGTGAAGCAGAATTCACCAAGCGTTGGATTGTTCACCCACTAATAGGGAACGTGAGCTGGGTTTAGACCG
TCGTGAGACAGGTTAGTTTTACCCTACTGATGATGTGTTGTTGCCATGGTAATCCTGCTCAGTACGAGAG
GAACCGCAGGTTCAGACATTTGGTGTATGTGCTTGGCTGAGGAGCCAATGGGGCGAAGCTACCATCTGTG
GGATTATGACTGAACGCCTCTAAGTCAGAATCCGCCCAAGCGGAACGATACGGCAGCGCCGAAGGAGCCT
CGGTTGGCCCCGGATAGCCGGCTCCCCGTCCGTCCCCGTCCGGCGGGTCCCCGCCTCGTCGCCCCCCCGG
GTGCGGGGCGGGTCCCCCCGCCGGGCGTCGGGACCGGGGTCCGGTGCGGAGAGCCATTCGTCCCGGGAAA
CGGGGTGCGGCCGGAAAGGGGGCCGCCCTCTCGCCCGTCACGCTTAACGCACGTTCGTGTGGAACTTGGC
GCTAAACCATTCGTAGACGACCTGCTTCTGGGTCGGGGTTTCGTACGTAGCAGAGCAGCTCCCTCGCTGC
GATCTATTGAAAGTCAGCCCTCGACACAAGGGTTTGT
>poly-A
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
>poly-C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
>poly-G
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
>poly-T
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
TTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTTT
"""
    empty = '>empty_A\n'+'A'*500+'\n'+'>empty_C\n'+'C'*500+'\n'+'>empty_G\n'+'G'*500+'\n'+'>empty_T\n'+'T'*500+'\n'
    sequence = seq.get(options.organism.lower(),empty)
    file(os.path.join(options.output_directory,'rrna_unit.fa'),'w').write(sequence)
    
#
