#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of allowed candidate fusion genes. This list is hard coded
in here and it is manually curated from TCGA database (https://tcga-data.nci.nih.gov/tcga/ ).



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of allowed candidate fusion genes (which are manually curated from TCGA database)."""
    version = "%prog 0.12 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of allowed candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of allowed candidate fusion genes is generated. Default is '%default'.""")

    parser.add_option("--skip-filter-overlap",
                      action="store_true",
                      dest="skip_filter_overlap",
                      default = False,
                      help="""If set then it filters out the known fusion genes where the (i) genes are fully overlapping, or (ii) the genes are partially overlapping and are on the same strand. Default is '%default'.""")

    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    print "Generating the list of allowed/known fusion genes..."
    fusions = dict()

    # manual curation from papers

    fusions['rattus_norvegicus'] = []

    fusions['mus_musculus'] = []

    fusions['canis_lupus_familiaris'] = []

    fusions['homo_sapiens'] = [
        ['2-COP','PBX1'],
        ['6C6-AG','FLJ34125'],
        ['6C6-AG','PDZD4'],
        ['A2M','ARFGEF2'],
        ['A2M','FAR1'],
        ['AAK1','C2ORF42'],
        ['AAK1','GALNT14'],
        ['AAK1','RPL36A'],
        ['AAK1','TEX261'],
        ['AAK1','WFDC10B'],
        ['AAK1','WFDC13'],
        ['AAMDC','ARHGEF18'],
        ['AAMDC','SCGB1D1'],
        ['AAMDC','TMEM135'],
        ['AAMP','MARCH4'],
        ['AANAT','AGAP1'],
        ['AARD','KIAA1432'],
        ['AARS','COG4'],
        ['AARS','ST3GAL2'],
        ['AARSD1','FBXL20'],
        ['AASDH','LRRC66'],
        ['AATF','CPD'],
        ['AATF','HOMER2'],
        ['AATF','MLLT6'],
        ['AATF','NUMA1'],
        ['ABAT','C16ORF62'],
        ['ABAT','KIAA0430'],
        ['ABAT','RUSC2'],
        ['ABCA1','HDHD3'],
        ['ABCA1','ZBTB26'],
        ['ABCA1','ZBTB6'],
        ['ABCA10','BRWD1'],
        ['ABCA10','HELZ'],
        ['ABCA13','C2CD3'],
        ['ABCA13','KIAA1324L'],
        ['ABCA2','TRIM14'],
        ['ABCA3','PARN'],
        ['ABCA4','ARHGAP29'],
        ['ABCA5','BCAS3'],
        ['ABCA5','FOXK2'],
        ['ABCA8','BCAS3'],
        ['ABCA8','NCOA3'],
        ['ABCA8','RAC3'],
        ['ABCA9','MED13'],
        ['ABCA9','SOBP'],
        ['ABCB4','ABCC4'],
        ['ABCB5','NF1'],
        ['ABCB7','CHM'],
        ['ABCB8','NANOS3'],
        ['ABCB8','NOS3'],
        ['ABCB8','NUB1'],
        ['ABCB9','MPHOSPH9'],
        ['ABCC10','PPP2R5D'],
        ['ABCC11','ANO1'],
        ['ABCC11','DDX19A'],
        ['ABCC11','SKAP1'],
        ['ABCC3','BCAN'],
        ['ABCC3','BPTF'],
        ['ABCC3','LUC7L3'],
        ['ABCC3','MBD4'],
        ['ABCC3','MED1'],
        ['ABCC3','NURF301'],
        ['ABCC4','COL4A2'],
        ['ABCC4','FGFR1OP'],
        ['ABCC4','PTPN4'],
        ['ABCC5','EHHADH'],
        ['ABCC5','NAALADL2'],
        ['ABCC5','PARL'],
        ['ABCC6','C16ORF70'],
        ['ABCC6P1','C16ORF62'],
        ['ABCC8','GTF2H1'],
        ['ABCC8','PLEKHA7'],
        ['ABCC9','FGD4'],
        ['ABCD3','AK5'],
        ['ABCD3','ARHGAP29'],
        ['ABCD3','PRPF38A'],
        ['ABCF1','PLVAP'],
        ['ABCF1','RPL18'],
        ['ABCF1','S100A6'],
        ['ABCG1','BRWD1'],
        ['ABCG1','PDE9A'],
        ['ABCG1','PLOD2'],
        ['ABCG8','PPM1B'],
        ['ABHD11','BAZ1B'],
        ['ABHD12','CHD6'],
        ['ABHD12','NUMA1'],
        ['ABHD14B','POC1A'],
        ['ABHD15','ANO1'],
        ['ABHD15','TAOK1'],
        ['ABHD17C','KIAA1199'],
        ['ABHD17C','LPP'],
        ['ABHD2','BIRC6'],
        ['ABHD3','ESCO1'],
        ['ABHD3','ROCK1'],
        ['ABHD5','SFMBT1'],
        ['ABHD6','FLNB'],
        ['ABHD6','RPP14'],
        ['ABI3BP','GCFC1'],
        ['ABI3BP','PAXBP1'],
        ['ABI3BP','TBC1D23'],
        ['ABL1','ADAMTSL2'],
        ['ABL1','BCR'],
        ['ABL1','CCBL1'],
        ['ABL1','DDX27'],
        ['ABL1','EXOSC2'],
        ['ABL1','PRDM12'],
        ['ABLIM1','ARID5B'],
        ['ABLIM1','GOPC'],
        ['ABLIM1','KRT5'],
        ['ABLIM1','PIP4K2A'],
        ['ABLIM1','PTCD2'],
        ['ABLIM2','AFAP1'],
        ['ABLIM2','PPP2R2C'],
        ['ABR','CLUH'],
        ['ABR','CRK'],
        ['ABR','EWSR1'],
        ['ABR','FAM76A'],
        ['ABR','GAS7'],
        ['ABR','KCNK1'],
        ['ABR','MAN2B1'],
        ['ABR','NXN'],
        ['ABR','PITPNA'],
        ['ABR','STAP2'],
        ['ABR','YWHAE'],
        ['ABRACL','MAP3K5'],
        ['ABTB2','APIP'],
        ['ABTB2','ATXN3'],
        ['AC002511.1','WDR62'],
        ['AC005863.1','TOM1L2'],
        ['AC007000.1','PTPN12'],
        ['AC008277.1','TANC1'],
        ['AC008745.1','EHD2'],
        ['AC010884.1','PRRG4'],
        ['AC011357.1','CCT3'],
        ['AC011450.1','GLIS3'],
        ['AC011498.1','ELAVL1'],
        ['AC011997.1','LRRC69'],
        ['AC016995.3','RPLP0'],
        ['AC019349.5','ACTB'],
        ['AC019349.5','ACTN4'],
        ['AC019349.5','CALR'],
        ['AC019349.5','COL1A2'],
        ['AC019349.5','COL3A1'],
        ['AC019349.5','DMKN'],
        ['AC019349.5','HNRNPU'],
        ['AC019349.5','KRT14'],
        ['AC019349.5','KRT15'],
        ['AC019349.5','KRT16'],
        ['AC019349.5','KRT4'],
        ['AC019349.5','KRT5'],
        ['AC019349.5','KRT6B'],
        ['AC019349.5','KRT75'],
        ['AC019349.5','LY6D'],
        ['AC019349.5','RP11-1020M18.10'],
        ['AC019349.5','RP11-762I7.5'],
        ['AC019349.5','SARNP'],
        ['AC019349.5','SPARC'],
        ['AC019349.5','TGM3'],
        ['AC021860.1','PCDH7'],
        ['AC023085.1','PHF21A'],
        ['AC034228.4','CDC42SE2'],
        ['AC034243.1','ETF1'],
        ['AC053503.6','H19'],
        ['AC053503.6','KRT6A'],
        ['AC067960.1','LMAN2L'],
        ['AC068020.1','TNKS'],
        ['AC068580.6','S100A9'],
        ['AC068610.3','ZNF277'],
        ['AC068718.1','AC074286.1'],
        ['AC069368.3','PTPLAD1'],
        ['AC074212.3','SMARCA2'],
        ['AC079354.1','BMPR2'],
        ['AC079354.1','NOP58'],
        ['AC079807.4','EXOC6B'],
        ['AC083949.1','MTA3'],
        ['AC087491.2','METRNL'],
        ['AC087491.2','NPLOC4'],
        ['AC087491.2','PSMD3'],
        ['AC090505.3','TFRC'],
        ['AC091167.2','PERP'],
        ['AC092155.4','WDPCP'],
        ['AC096579.13','HSP90AB1'],
        ['AC098820.3','PSMD1'],
        ['AC130352.1','INTS9'],
        ['AC130352.1','LIMCH1'],
        ['AC135352.1','TBL1XR1'],
        ['AC135352.2','TBL1XR1'],
        ['AC141586.5','KCTD5'],
        ['ACACA','ASIC2'],
        ['ACACA','KIAA1684'],
        ['ACACA','MGME1'],
        ['ACACA','MYO1D'],
        ['ACACA','NCOR1'],
        ['ACACA','NF1'],
        ['ACACA','SRCIN1'],
        ['ACACA','TADA2A'],
        ['ACACB','UNG'],
        ['ACAD9','KIAA1257'],
        ['ACADM','EGFR'],
        ['ACAP2','APOD'],
        ['ACAP2','IQCG'],
        ['ACAP3','CPSF3L'],
        ['ACAP3','UBE2J2'],
        ['ACBD3','ESRRG'],
        ['ACCS','PRDM11'],
        ['ACE','GTF2E2'],
        ['ACE','MLLT6'],
        ['ACER3','NCOR1'],
        ['ACER3','NRL'],
        ['ACER3','PPME1'],
        ['ACER3','TAOK1'],
        ['ACER3','THRSP'],
        ['ACER3','UVRAG'],
        ['ACKR3','STAT4'],
        ['ACL','DNAJC7'],
        ['ACL','TTC2'],
        ['ACLY','DNAJC7'],
        ['ACLY','TTC2'],
        ['ACMSD','RAB3GAP1'],
        ['ACMSD','SESTD1'],
        ['ACO1','ITPK1'],
        ['ACO1','NFX1'],
        ['ACO1','TYRP1'],
        ['ACO2','PRAME'],
        ['ACO2','SREBF2'],
        ['ACO2','TANGO2'],
        ['ACO2','TEF'],
        ['ACO2','ZC3H7B'],
        ['ACOT1','ELMSAN1'],
        ['ACOT11','BLK'],
        ['ACOT11','C8B'],
        ['ACOT12','IQGAP2'],
        ['ACOT4','COQ6'],
        ['ACOT7','FUCA1'],
        ['ACOT7','KLHL21'],
        ['ACOT7','NPHP4'],
        ['ACOT7','NSMCE2'],
        ['ACOT7','TTC34'],
        ['ACOT9','APOO'],
        ['ACOX1','SRP68'],
        ['ACOXL','BCL2L11'],
        ['ACOXL','UNC5D'],
        ['ACP5','GTF2F1'],
        ['ACP6','ARHGEF3'],
        ['ACPL2','RASA2'],
        ['ACPL2','ZBTB38'],
        ['ACPP','AXL'],
        ['ACPT','C19ORF48'],
        ['ACSBG2','NCLN'],
        ['ACSF2','SPAG9'],
        ['ACSF3','MYH7B'],
        ['ACSL1','FAT1'],
        ['ACSL3','PHGDH'],
        ['ACSL5','B4GALT4'],
        ['ACSL5','ZDHHC6'],
        ['ACSL6','PLIN1'],
        ['ACSM1','IGF2BP3'],
        ['ACSM1','IKBKE'],
        ['ACSM3','EEF2K'],
        ['ACSS1','PHACTR3'],
        ['ACSS2','FRS2'],
        ['ACSS2','GSS'],
        ['ACSS2','PRNP'],
        ['ACSS3','SYT1'],
        ['ACTA1','ACTB'],
        ['ACTB','KRT13'],
        ['ACTB','KRT14'],
        ['ACTB','KRT5'],
        ['ACTB','OXSR1'],
        ['ACTB','S100A9'],
        ['ACTG','PLEKHA5'],
        ['ACTG1','CORO6'],
        ['ACTG1','KRT14'],
        ['ACTG1','KRT5'],
        ['ACTG1','MYO18A'],
        ['ACTG1','PLEKHA5'],
        ['ACTG2','DGUOK'],
        ['ACTG2','PELI1'],
        ['ACTG2','TET3'],
        ['ACTL6A','ALG3'],
        ['ACTN1','KRT14'],
        ['ACTN1','SCGN'],
        ['ACTN1','SYNE2'],
        ['ACTN4','CAPZB'],
        ['ACTN4','FTO'],
        ['ACTN4','KRT13'],
        ['ACTN4','KRT14'],
        ['ACTN4','LSM14A'],
        ['ACTN4','YIF1B'],
        ['ACTR10','NRXN3'],
        ['ACTR2','TSGA10'],
        ['ACTR3C','CUX1'],
        ['ACTR5','RPRD1B'],
        ['ACVR1','CCDC148'],
        ['ACVR1','NFATC1'],
        ['ACVR1B','BAZ2A'],
        ['ACVR1B','PGC'],
        ['ACVR1B','TENC1'],
        ['ACVR2A','AKT3'],
        ['ACVR2A','CASP8'],
        ['ACVR2A','IGF2R'],
        ['ACVR2A','MBD5'],
        ['ACVR2A','SUMO1'],
        ['ACYP2','SPTBN1'],
        ['ADAM-TS10','MLLT1'],
        ['ADAM10','FIZ1'],
        ['ADAM10','ZNF770'],
        ['ADAM12','C10ORF90'],
        ['ADAM12','CTBP2'],
        ['ADAM17','YWHAQ'],
        ['ADAM18','FGFR1'],
        ['ADAM18','LSM1'],
        ['ADAM19','CCDC14'],
        ['ADAM19','TTC33'],
        ['ADAM2','CPT1A'],
        ['ADAM2','HIRA'],
        ['ADAM2','KAT6A'],
        ['ADAM2','MTMR2'],
        ['ADAM23','CERKL'],
        ['ADAM23','PPAP2A'],
        ['ADAM23','SKIV2L2'],
        ['ADAM28','CDCA2'],
        ['ADAM28','CERKL'],
        ['ADAM28','PPAP2A'],
        ['ADAM32','FBXO16'],
        ['ADAM32','RPAP3'],
        ['ADAM32','WHSC1L1'],
        ['ADAM32','ZNF251'],
        ['ADAM7','ANK1'],
        ['ADAM9','CLYBL'],
        ['ADAM9','DOCK5'],
        ['ADAM9','RNF170'],
        ['ADAM9','TACC1'],
        ['ADAM9','TM2D2'],
        ['ADAM9','WRN'],
        ['ADAMTS1','GCFC1'],
        ['ADAMTS1','PAXBP1'],
        ['ADAMTS10','MLLT1'],
        ['ADAMTS12','C1QTNF3'],
        ['ADAMTS12','HSF1'],
        ['ADAMTS12','HSTF1'],
        ['ADAMTS12','TMEM182'],
        ['ADAMTS12','TRIO'],
        ['ADAMTS12','UQCC1'],
        ['ADAMTS12','WDR70'],
        ['ADAMTS12','ZFR'],
        ['ADAMTS13','FSD1L'],
        ['ADAMTS13','LHX2'],
        ['ADAMTS16','ETV6'],
        ['ADAMTS16','SRD5A1'],
        ['ADAMTS17','CERS3'],
        ['ADAMTS17','LRRK1'],
        ['ADAMTS17','PCDHA8'],
        ['ADAMTS19','KIF2A'],
        ['ADAMTS2','WBP1L'],
        ['ADAMTS3','YT521'],
        ['ADAMTS3','YTHDC1'],
        ['ADAMTS9','CHL1'],
        ['ADAMTS9','DDX11'],
        ['ADAMTS9','ULK4'],
        ['ADAMTSL1','TOPORS'],
        ['ADAMTSL2','MAPKAP1'],
        ['ADAMTSL3','MAP2K5'],
        ['ADAMTSL3','MCCC1'],
        ['ADAMTSL3','NEDD4'],
        ['ADAP1','MIOS'],
        ['ADAP1','NDUFAF6'],
        ['ADAR','GPATCH4'],
        ['ADAR','SNX27'],
        ['ADARB1','COL18A1'],
        ['ADARB1','SBNO2'],
        ['ADARB2','NUMA1'],
        ['ADAT1','BBS2'],
        ['ADC','PHIP'],
        ['ADCK1','KCNQ2'],
        ['ADCK1','OSBPL2'],
        ['ADCK1','TMC1'],
        ['ADCK2','DPP6'],
        ['ADCK2','HIPK2'],
        ['ADCK2','PRKAG2'],
        ['ADCK2','TBXAS1'],
        ['ADCK3','CYP2E1'],
        ['ADCK3','EPHX1'],
        ['ADCK3','SPRN'],
        ['ADCK4','AKAP8'],
        ['ADCK4','PPM1A'],
        ['ADCY1','EYA4'],
        ['ADCY10','ESRRG'],
        ['ADCY2','RPAP2'],
        ['ADCY4','CIDEB'],
        ['ADCY5','PTPLB'],
        ['ADCY5','TSC22D2'],
        ['ADCY8','SCAF11'],
        ['ADCY9','GLYR1'],
        ['ADCY9','PRKCB'],
        ['ADCY9','PRSS27'],
        ['ADCY9','SRL'],
        ['ADCY9','TFAP4'],
        ['ADCYAP1R1','DFNA5'],
        ['ADD1','ADK'],
        ['ADD1','ATP2B4'],
        ['ADD1','FAM193A'],
        ['ADD1','NUDT4'],
        ['ADD1','SH3BP2'],
        ['ADD3','OGT'],
        ['ADD3','XPNPEP1'],
        ['ADH6','TET2'],
        ['ADH7','CXCR5'],
        ['ADHFE1','LAMA3'],
        ['ADI1','KHDRBS3'],
        ['ADIPOR1','CYB5R1'],
        ['ADIPOR1','RGS8'],
        ['ADIPOR2','DCP1B'],
        ['ADIPOR2','ERC1'],
        ['ADIPOR2','FOXJ2'],
        ['ADIPOR2','ITPR2'],
        ['ADIPOR2','NINJ2'],
        ['ADIPOR2','WNT5B'],
        ['ADK','C10ORF11'],
        ['ADK','DNAJC19'],
        ['ADK','FAM149B1'],
        ['ADK','KAT6B'],
        ['ADK','KCNMA1'],
        ['ADK','LRRTM3'],
        ['ADK','PI4K2A'],
        ['ADK','SLC16A9'],
        ['ADK','SPTAN1'],
        ['ADK','TTC23'],
        ['ADM5','YARS'],
        ['ADNP','BMP7'],
        ['ADNP','DDX27'],
        ['ADNP','DPM1'],
        ['ADNP','GHRH'],
        ['ADNP','KCNG1'],
        ['ADNP','MMP24'],
        ['ADNP','PHF20'],
        ['ADNP','SYCP2'],
        ['ADNP','ZNF841'],
        ['ADNP2','L3MBTL4'],
        ['ADNP2','PARD6G'],
        ['ADNP2','PQLC1'],
        ['ADORA2B','SYN3'],
        ['ADORA3','GDAP2'],
        ['ADORA3','TMED5'],
        ['ADPRHL2','SHPK'],
        ['ADRBK1','BAIAP2'],
        ['ADRBK1','CHGA'],
        ['ADRBK1','IQGAP1'],
        ['ADRBK1','POLD4'],
        ['ADRBK2','AP1B1'],
        ['ADRBK2','EHD1'],
        ['ADRBK2','EP300'],
        ['ADRBK2','ERC1'],
        ['ADRBK2','RHBDL3'],
        ['ADRBK2','SF3A1'],
        ['ADSL','MB'],
        ['ADSL','PVALB'],
        ['ADSS','GNA12'],
        ['ADSS','SERTAD4'],
        ['AEBP1','EIF4G1'],
        ['AEBP2','ENAH'],
        ['AEBP2','PRLR'],
        ['AES','ELK3'],
        ['AES','FGFR3'],
        ['AES','KCNH8'],
        ['AES','VPS41'],
        ['AF127936.7','GART'],
        ['AF127936.7','NRIP1'],
        ['AFAP1','NTRK2'],
        ['AFAP1','SLC45A1'],
        ['AFAP1L2','INPP5A'],
        ['AFF1','PIM1'],
        ['AFF1','PTPN13'],
        ['AFF3','ASZ1'],
        ['AFF3','MAP4K4'],
        ['AFF3','NPAS2'],
        ['AFF3','TMEM182'],
        ['AFF4','FSTL4'],
        ['AFG3L2','TIAM1'],
        ['AFMID','BCL2L11'],
        ['AFMID','TNRC6C'],
        ['AFTPH','THRB'],
        ['AGAP1','IGSF21'],
        ['AGAP1','PHIP'],
        ['AGAP1','RINT1'],
        ['AGAP1','TMSB10'],
        ['AGAP1','UBE2F'],
        ['AGAP1','WDR11'],
        ['AGAP1','ZC2HC1A'],
        ['AGAP2','TMEM67'],
        ['AGAP3','GIMAP1'],
        ['AGAP3','GIMAP2'],
        ['AGBL1','PTPN9'],
        ['AGBL1','THBS1'],
        ['AGBL3','ATXN7L1'],
        ['AGBL4','DAB1'],
        ['AGBL4','DTL'],
        ['AGBL4','FAF1'],
        ['AGBL4','FUT8'],
        ['AGBL4','PPT1'],
        ['AGBL4','PTGFRN'],
        ['AGBL4','ZNF267'],
        ['AGER','CINP'],
        ['AGER','HSP90AA1'],
        ['AGER','PRND'],
        ['AGFG1','MFF'],
        ['AGFG1','TRIP12'],
        ['AGGF1','RAF1'],
        ['AGK','BRAF'],
        ['AGMO','MAP3K1'],
        ['AGO2','CNBD1'],
        ['AGO2','PCMTD1'],
        ['AGO2','PTK2'],
        ['AGO2','TRAPPC9'],
        ['AGO3','FRMD5'],
        ['AGO3','THRAP3'],
        ['AGO3','VRK2'],
        ['AGPAT2','CDC42SE1'],
        ['AGPAT3','KSR1'],
        ['AGPAT3','PDXK'],
        ['AGPAT6','HGSNAT'],
        ['AGPS','PRKRA'],
        ['AGRN','NOC4L'],
        ['AGRN','PUSL1'],
        ['AGTPBP1','DAPK1'],
        ['AGTRAP','KIF1B'],
        ['AHCY','ITCH'],
        ['AHCYL1','DENND2D'],
        ['AHCYL1','LRRIQ3'],
        ['AHCYL2','CUX1'],
        ['AHDC1','ALCAM'],
        ['AHDC1','MAP3K6'],
        ['AHDC1','MORN1'],
        ['AHDC1','WASF2'],
        ['AHI1','C12ORF50'],
        ['AHI1','CHEK2'],
        ['AHI1','MYB'],
        ['AHNAK','ARHGAP32'],
        ['AHNAK','RPS6KA4'],
        ['AHNAK','STIP1'],
        ['AHNAK','ZNF827'],
        ['AHNAK2','KRT14'],
        ['AHR','CDK14'],
        ['AHR','CUL4B'],
        ['AIF1','G2'],
        ['AIF1','PRRC2A'],
        ['AIF1','SORBS2'],
        ['AIF1','TRERF1'],
        ['AIF1L','NUP214'],
        ['AIF1L','SYN2'],
        ['AIG1','PHACTR2'],
        ['AIM1','ARMC2'],
        ['AIM1','BEND3'],
        ['AIM1','MYB'],
        ['AIM1','PITRM1'],
        ['AIM1','PREP'],
        ['AIM1','SESN1'],
        ['AIM1','SOBP'],
        ['AIM1L','S100A6'],
        ['AIM2','ASH1L'],
        ['AIM2','TBCE'],
        ['AIMP2','PMS2'],
        ['AK3','DOCK8'],
        ['AK3','RCL1'],
        ['AK4','DOCK8'],
        ['AK4','RCL1'],
        ['AK5','FAM49B'],
        ['AK5','NEXN'],
        ['AK7','PAPOLA'],
        ['AK8','DDX31'],
        ['AK9','FAM184A'],
        ['AK9','MICAL1'],
        ['AKAP1','ANKFN1'],
        ['AKAP1','C17ORF67'],
        ['AKAP1','CDK12'],
        ['AKAP1','GGA3'],
        ['AKAP1','MSI2'],
        ['AKAP1','WIPF2'],
        ['AKAP10','CCNY'],
        ['AKAP10','EPN2'],
        ['AKAP10','RIMS4'],
        ['AKAP11','MRPS31'],
        ['AKAP13','CHD2'],
        ['AKAP13','CHSY3'],
        ['AKAP13','GSN'],
        ['AKAP13','IGF1R'],
        ['AKAP13','NRG1'],
        ['AKAP13','NTRK3'],
        ['AKAP13','RET'],
        ['AKAP6','EGLN3'],
        ['AKAP6','ERBB4'],
        ['AKAP6','KIAA0391'],
        ['AKAP6','NPAS3'],
        ['AKAP6','PRKD1'],
        ['AKAP6','RMDN1'],
        ['AKAP8','VMP1'],
        ['AKAP8L','EFCAB3'],
        ['AKAP8L','KDM4B'],
        ['AKAP8L','WIZ'],
        ['AKAP9','CDK6'],
        ['AKAP9','RUNX1'],
        ['AKIP1','BACH2'],
        ['AKIP1','IPO7'],
        ['AKIRIN1','MACF1'],
        ['AKIRIN1','MEAF6'],
        ['AKIRIN2','ANKRD55'],
        ['AKR1B1','TRMT10B'],
        ['AKR1B15','CALD1'],
        ['AKR1C1','AKR1C4'],
        ['AKR1C1','CCDC88B'],
        ['AKR1CL1','SHB'],
        ['AKR1D1','RIOK1'],
        ['AKR1D1','SYPL1'],
        ['AKR1E2','FARS2'],
        ['AKT1','PGC'],
        ['AKT2','ART3'],
        ['AKT2','CXCL11'],
        ['AKT2','SRRM5'],
        ['AKT2','UPF1'],
        ['AKT2','YIF1B'],
        ['AKT2','ZNF576'],
        ['AKT3','ARHGAP30'],
        ['AKT3','CEP170'],
        ['AKT3','GCSAML'],
        ['AKT3','PLD5'],
        ['AKT3','RGS7'],
        ['AKT3','ZEB2'],
        ['AKT3','ZNF804A'],
        ['AL132709.7','COL3A1'],
        ['AL136419.1','TMPRSS11E'],
        ['AL139022.1','FUT8'],
        ['AL139022.1','PEBP4'],
        ['AL139022.1','RAB15'],
        ['AL353597.1','ZFAND3'],
        ['AL590867.1','ARID1B'],
        ['ALAS1','DNAH1'],
        ['ALAS1','GLYCTK'],
        ['ALB','SIAE'],
        ['ALCAM','EIF2A'],
        ['ALCAM','EIF2S1'],
        ['ALCAM','SH3KBP1'],
        ['ALCAM','YIF1B'],
        ['ALDH16A1','FLT3LG'],
        ['ALDH16A1','PIH1D1'],
        ['ALDH16A1','TAF2'],
        ['ALDH1A2','THSD4'],
        ['ALDH1A3','FYTTD1'],
        ['ALDH1L1','APC'],
        ['ALDH1L1','CNBP'],
        ['ALDH1L1','TMCC1'],
        ['ALDH2','MAPKAPK5'],
        ['ALDH3A1','KRT6A'],
        ['ALDH3A2','IFNK'],
        ['ALDH3B2','KCNMB4'],
        ['ALDH3B2','TMEM135'],
        ['ALDH5A1','MAP3K5'],
        ['ALDH7A1','PIP'],
        ['ALDH9A1','NME7'],
        ['ALDH9A1','S100A8'],
        ['ALDOA','S100A9'],
        ['ALDOC','KIF3A'],
        ['ALG1','SEC14L5'],
        ['ALG10B','IGFBP6'],
        ['ALG12','ZBED4'],
        ['ALG14','C1ORF194'],
        ['ALG14','CNN3'],
        ['ALG14','MYSM1'],
        ['ALG14','NRD1'],
        ['ALG14','SGPP2'],
        ['ALG3','B3GNT5'],
        ['ALG5','WBSCR17'],
        ['ALG6','C1ORF123'],
        ['ALG8','FGF19'],
        ['ALG8','NARS2'],
        ['ALG9','PPP2R1B'],
        ['ALK','C2ORF61'],
        ['ALK','CREAM1'],
        ['ALK','EML4'],
        ['ALK','GTF2IRD1'],
        ['ALK','KCNQ5'],
        ['ALK','SMEK2'],
        ['ALK','STRN'],
        ['ALK','TPM1'],
        ['ALKBH3','ANO3'],
        ['ALKBH3','EXT2'],
        ['ALKBH3','PHF21A'],
        ['ALKBH5','KIAA0100'],
        ['ALKBH5','LLGL1'],
        ['ALKBH5','LRP5'],
        ['ALL-1','ELL'],
        ['ALL-1','FYN'],
        ['ALL-1','MLLT10'],
        ['ALL-1','MLLT3'],
        ['ALL-1','MLLT4'],
        ['ALL-1','SLC22A10'],
        ['ALL-1','ZDHHC7'],
        ['ALMS1','RORA'],
        ['ALOX12','MINK1'],
        ['ALOX12','SLC35G6'],
        ['ALPK1','ANK2'],
        ['ALPK3','ELOVL2'],
        ['ALPK3','TFAP2A'],
        ['ALPL','CD302'],
        ['ALPL','FILIP1L'],
        ['ALS2CR12','CASP8'],
        ['ALS2CR12','FAM126B'],
        ['ALS2CR12','ORC2'],
        ['ALS2CR12','PPTC7'],
        ['ALS2CR4','SLC39A10'],
        ['ALX4','EXT2'],
        ['ALX4','STK33'],
        ['AMAC1L3','KRT14'],
        ['AMACR','SLC45A2'],
        ['AMBRA1','ARHGAP1'],
        ['AMBRA1','ATG13'],
        ['AMBRA1','CCDC61'],
        ['AMBRA1','HARBI1'],
        ['AMBRA1','PHF21A'],
        ['AMD1','CDK19'],
        ['AMD1','RPF2'],
        ['AMDHD1','ELK3'],
        ['AMDHD1','KCNH8'],
        ['AMFR','ITGAM'],
        ['AMMECR1','PLA2G12B'],
        ['AMMECR1L','SAP130'],
        ['AMMECR1L','SF3B3'],
        ['AMN1','CAPRIN2'],
        ['AMN1','FAR2'],
        ['AMOTL1','CCDC88B'],
        ['AMOTL1','MPPED2'],
        ['AMOTL1','SLCO2B1'],
        ['AMPD1','DENND2C'],
        ['AMPD1','TFIP11'],
        ['AMTN','LEC3'],
        ['AMTN','LPHN3'],
        ['ANAPC13','SACM1L'],
        ['ANAPC15','RNF169'],
        ['ANAPC16','MICU1'],
        ['ANAPC2','GIGYF2'],
        ['ANAPC4','TBC1D19'],
        ['ANAPC5','CAMKK2'],
        ['ANAPC5','KDM2B'],
        ['ANGPT1','PEAK1'],
        ['ANGPT1','PTK2'],
        ['ANGPT1','ZFP41'],
        ['ANGPT2','MAK16'],
        ['ANK1','CNEP1R1'],
        ['ANK1','FGFR1'],
        ['ANK1','FUT10'],
        ['ANK1','IKBKB'],
        ['ANK1','NECAP1'],
        ['ANK1','PPFIA3'],
        ['ANK1','WHSC1L1'],
        ['ANK2','ARHGAP10'],
        ['ANK2','ARHGAP21'],
        ['ANK2','CAMK2D'],
        ['ANK3','ATAD1'],
        ['ANK3','CCDC6'],
        ['ANK3','CCNY'],
        ['ANK3','CDK1'],
        ['ANK3','CHN2'],
        ['ANK3','IFIT5'],
        ['ANK3','KIAA0930'],
        ['ANK3','RET'],
        ['ANK3','RHOBTB1'],
        ['ANK3','SLC16A9'],
        ['ANK3','SP2'],
        ['ANK3','TMPRSS15'],
        ['ANK3','TRAPPC8'],
        ['ANK3','WDFY4'],
        ['ANKFN1','ARRB1'],
        ['ANKFN1','GPC1'],
        ['ANKFN1','NF1'],
        ['ANKFN1','RPTOR'],
        ['ANKFN1','TANC2'],
        ['ANKFY1','ZZEF1'],
        ['ANKH','MXD3'],
        ['ANKH','SLC20A2'],
        ['ANKHD1','CRNR7'],
        ['ANKHD1','CYSTM1'],
        ['ANKHD1','PCDHA11'],
        ['ANKHD1','TIAM2'],
        ['ANKIB1','LMTK2'],
        ['ANKIB1','TRRAP'],
        ['ANKLE1','MGA'],
        ['ANKLE1','MGAM'],
        ['ANKLE2','NCOR2'],
        ['ANKLE2','NHP2L1'],
        ['ANKRA2','TMEM232'],
        ['ANKRD10','MYO16'],
        ['ANKRD11','BRMS1'],
        ['ANKRD11','DPEP1'],
        ['ANKRD11','FANCA'],
        ['ANKRD11','KIT'],
        ['ANKRD11','KLHL32'],
        ['ANKRD11','LINGO1'],
        ['ANKRD11','PTK2'],
        ['ANKRD11','TGIF1'],
        ['ANKRD12','MTHFD2L'],
        ['ANKRD13B','NSRP1'],
        ['ANKRD13C','HHLA3'],
        ['ANKRD13C','OR5V1'],
        ['ANKRD17','NIPBL'],
        ['ANKRD17','RPRD1B'],
        ['ANKRD17','SQRDL'],
        ['ANKRD17','WDFY3'],
        ['ANKRD18B','SNX30'],
        ['ANKRD18B','SNX33'],
        ['ANKRD23','CNNM3'],
        ['ANKRD24','SH3GL1'],
        ['ANKRD26','FAM171A1'],
        ['ANKRD26','ZNHIT6'],
        ['ANKRD27','SENP2'],
        ['ANKRD27','ZFP14'],
        ['ANKRD28','BCR'],
        ['ANKRD28','FLT3'],
        ['ANKRD28','RFTN1'],
        ['ANKRD28','STK38'],
        ['ANKRD29','NPC1'],
        ['ANKRD29','ROCK1'],
        ['ANKRD29','SERGEF'],
        ['ANKRD30BL','PDE3A'],
        ['ANKRD31','FAM169A'],
        ['ANKRD31','NDUFAF2'],
        ['ANKRD34C','EIF2AK4'],
        ['ANKRD39','FAM178B'],
        ['ANKRD40','LRRC6'],
        ['ANKRD42','RHOD'],
        ['ANKRD42','SRPK1'],
        ['ANKRD44','PTPN14'],
        ['ANKRD46','NUBPL'],
        ['ANKRD46','ODF1'],
        ['ANKRD46','RNF19A'],
        ['ANKRD46','TMEM222'],
        ['ANKRD46','VPS13B'],
        ['ANKRD55','IL6ST'],
        ['ANKRD55','SNX18'],
        ['ANKRD6','DYNC2H1'],
        ['ANKRD6','ZDHHC14'],
        ['ANKRD66','MOK'],
        ['ANKS1A','PTK7'],
        ['ANKS1B','CCL14'],
        ['ANKS1B','CORO1C'],
        ['ANKS1B','ERP29'],
        ['ANKS1B','HCFC2'],
        ['ANKS1B','IKBIP'],
        ['ANKS1B','NDUFA12'],
        ['ANKS1B','POLE'],
        ['ANKS1B','SCYL2'],
        ['ANKS1B','STK24'],
        ['ANKS1B','UHRF1BP1L'],
        ['ANKS1B','UTP20'],
        ['ANKUB1','EIF2A'],
        ['ANKUB1','EIF2S1'],
        ['ANKUB1','WWTR1'],
        ['ANLN','EEPD1'],
        ['ANLN','POU6F2'],
        ['ANO1','C14ORF43'],
        ['ANO1','CTTN'],
        ['ANO1','CYP2R1'],
        ['ANO1','DGCR2'],
        ['ANO1','PSD3'],
        ['ANO1','RNF121'],
        ['ANO1','SLC22A15'],
        ['ANO1','UNC5D'],
        ['ANO10','ARHGEF3'],
        ['ANO10','ULK4'],
        ['ANO2','ITPR2'],
        ['ANO2','KDM5A'],
        ['ANO3','SMG1'],
        ['ANO6','IAPP'],
        ['ANO7P1','ARHGEF19'],
        ['ANO9','DEAF1'],
        ['ANP32E','CTSK'],
        ['ANPEP','CPT1A'],
        ['ANTXR1','APLF'],
        ['ANTXR1','DBR1'],
        ['ANTXR1','GPR98'],
        ['ANTXR1','MXD1'],
        ['ANXA1','CSTB'],
        ['ANXA1','KRT13'],
        ['ANXA1','PDE3A'],
        ['ANXA1','PDXK'],
        ['ANXA1','PRKG1'],
        ['ANXA1','RRP1B'],
        ['ANXA10','FAM198B'],
        ['ANXA13','PROSAP1'],
        ['ANXA13','SHANK2'],
        ['ANXA2','KRT14'],
        ['ANXA2','RNF114'],
        ['ANXA2','RORA'],
        ['ANXA4','GMCL1'],
        ['ANXA4','MCEE'],
        ['ANXA4','PKN1'],
        ['ANXA4','RBBP8'],
        ['ANXA7','LGALS12'],
        ['ANXA9','MRPL9'],
        ['AOAH','RBM28'],
        ['AP000295.9','C21ORF49'],
        ['AP000318.2','MYO3A'],
        ['AP000431.2','USP25'],
        ['AP1B1','SGSM3'],
        ['AP1G2','FAM204A'],
        ['AP1G2','FARSA'],
        ['AP1G2','IGF2'],
        ['AP1G2','MAPKAP1'],
        ['AP1G2','RPS14'],
        ['AP1G2','SDC4'],
        ['AP1G2','SMPD3'],
        ['AP1G2','SQSTM1'],
        ['AP1M1','CLPTM1'],
        ['AP1M1','CYP4F11'],
        ['AP1M1','DNMT1'],
        ['AP1M1','RAB8A'],
        ['AP1M1','ZNF93'],
        ['AP1M2','SRP68'],
        ['AP2A1','DHDH'],
        ['AP2A1','MED25'],
        ['AP2A1','MED9'],
        ['AP2A1','ZNF544'],
        ['AP2A2','EPS8L2'],
        ['AP2A2','PLEKHA7'],
        ['AP2A2','PSMD13'],
        ['AP2A2','TMEM80'],
        ['AP2A2','TRIM22'],
        ['AP2B1','GOSR1'],
        ['AP2B1','NPL3'],
        ['AP2B1','PPM1E'],
        ['AP2B1','TAF15'],
        ['AP2M1','SEC63'],
        ['AP3AASE','CADM2'],
        ['AP3AASE','CADPS'],
        ['AP3AASE','CAMK4'],
        ['AP3AASE','FAM172A'],
        ['AP3AASE','FAM19A4'],
        ['AP3AASE','MTCH2'],
        ['AP3AASE','PTPRG'],
        ['AP3AASE','RPAIN'],
        ['AP3AASE','SYNPR'],
        ['AP3B1','BRAF'],
        ['AP3B1','DCTN4'],
        ['AP3D1','IRF4'],
        ['AP3D1','MUM1'],
        ['AP3D1','TCF3'],
        ['AP3D1','TCF7L1'],
        ['AP3D1','ZNF317'],
        ['AP3D1','ZNF69'],
        ['AP3S1','COMMD10'],
        ['AP4E1','CSNK1G1'],
        ['AP5S1','RNF24'],
        ['APBA1','DOCK1'],
        ['APBA2','GABRB3'],
        ['APBB1','FAM160A2'],
        ['APBB1IP','DNAJC1'],
        ['APBB1IP','FRMD4A'],
        ['APBB1IP','WAC'],
        ['APBB2','DMD'],
        ['APBB2','LIMCH1'],
        ['APBB2','NSUN7'],
        ['APBB2','RBM47'],
        ['APC2','GIGYF2'],
        ['APC5','KDM2B'],
        ['APCDD1','TRAPPC8'],
        ['APEX1','GAST'],
        ['APEX1','LIG3'],
        ['APH1A','SLAMF1'],
        ['APIP','C10ORF35'],
        ['APLF','PELI1'],
        ['APLP2','CD40'],
        ['APLP2','CIT'],
        ['APLP2','EEA1'],
        ['APLP2','KCNJ5'],
        ['APLP2','PLEKHA5'],
        ['APMAP','LYN'],
        ['APMAP','ZNF337'],
        ['APOA5','BUD13'],
        ['APOBEC3B','WEE2'],
        ['APOC1','IQCE'],
        ['APOD','CLDN7'],
        ['APOL2','ULK2'],
        ['APOL4','MKL1'],
        ['APOL5','BRD1'],
        ['APOLD1','CDKN1B'],
        ['APOM','MYSM1'],
        ['APOO','ZFX'],
        ['APOPT1','MARK3'],
        ['APP','EEA1'],
        ['APP','KRT14'],
        ['APP','RCAN1'],
        ['APP','SEC63'],
        ['APP','TMEM50B'],
        ['APPBP2','BCAS3'],
        ['APPBP2','DNAH17'],
        ['APPBP2','MSF1'],
        ['APPBP2','SEPT9'],
        ['APPBP2','TNRC6C'],
        ['APPBP2','USP32'],
        ['APPBP2','WISP2'],
        ['APPL1','ATXN10'],
        ['APPL1','SPATA12'],
        ['APPL2','CIT'],
        ['APRT','PIEZO1'],
        ['APTX','ARL5B'],
        ['APTX','GNE'],
        ['APTX','NOL6'],
        ['APTX','UBAP2'],
        ['AQP1','CHI3L2'],
        ['AQP2','FOXP1'],
        ['AQP3','KRT14'],
        ['AQP3','SFN'],
        ['AR','TRMT10B'],
        ['ARAP1','DYNC2H1'],
        ['ARAP1','FADD'],
        ['ARAP1','MKL1'],
        ['ARAP1','MOGAT2'],
        ['ARAP1','PDE2A'],
        ['ARAP1','USP35'],
        ['ARCN1','CADM1'],
        ['ARF3','CHFR'],
        ['ARF3','DDX23'],
        ['ARF6','MAPK1IP1L'],
        ['ARFGAP1','SPAG4'],
        ['ARFGAP2','CELF1'],
        ['ARFGAP3','PACSIN2'],
        ['ARFGAP3','SPAG4'],
        ['ARFGEF1','CPA6'],
        ['ARFGEF1','CYP2A6'],
        ['ARFGEF1','NCOA2'],
        ['ARFGEF2','ATP6AP1'],
        ['ARFGEF2','BSG'],
        ['ARFGEF2','CF2'],
        ['ARFGEF2','CLPTM1L'],
        ['ARFGEF2','CNIH2'],
        ['ARFGEF2','CSE1L'],
        ['ARFGEF2','CTSB'],
        ['ARFGEF2','FAM19A5'],
        ['ARFGEF2','IGF2R'],
        ['ARFGEF2','ITM2B'],
        ['ARFGEF2','KRT10'],
        ['ARFGEF2','KRT14'],
        ['ARFGEF2','MBP'],
        ['ARFGEF2','PRG2'],
        ['ARFGEF2','TPT1'],
        ['ARFGEF2','TSHZ2'],
        ['ARFIP1','FGB'],
        ['ARFIP1','FHDC1'],
        ['ARFIP1','MAP3K5'],
        ['ARFIP1','TRIM2'],
        ['ARG1','ATXN1'],
        ['ARG1','MAP7'],
        ['ARG2','IL34'],
        ['ARHGAP1','ATG13'],
        ['ARHGAP1','GLIS3'],
        ['ARHGAP1','PHF21A'],
        ['ARHGAP1','PPP6R3'],
        ['ARHGAP10','FBXW7'],
        ['ARHGAP10','TMEM184C'],
        ['ARHGAP12','DESI1'],
        ['ARHGAP12','SORCS3'],
        ['ARHGAP15','KYNU'],
        ['ARHGAP18','CEP85L'],
        ['ARHGAP18','MAN1A1'],
        ['ARHGAP19','RRP12'],
        ['ARHGAP21','ASPHD2'],
        ['ARHGAP21','ENKUR'],
        ['ARHGAP21','FBXW7'],
        ['ARHGAP21','PIP4K2A'],
        ['ARHGAP21','TMEM184C'],
        ['ARHGAP22','MICU1'],
        ['ARHGAP23','ASIC2'],
        ['ARHGAP23','MBD4'],
        ['ARHGAP23','MED1'],
        ['ARHGAP23','TEX14'],
        ['ARHGAP24','NOA1'],
        ['ARHGAP24','PTPN13'],
        ['ARHGAP25','MBOAT2'],
        ['ARHGAP26','SPOCK1'],
        ['ARHGAP29','MID2'],
        ['ARHGAP32','C20ORF24'],
        ['ARHGAP32','FCHSD2'],
        ['ARHGAP32','RP11-121M22.1'],
        ['ARHGAP32','YWHAB'],
        ['ARHGAP35','COL22A1'],
        ['ARHGAP36','CD99B'],
        ['ARHGAP36','CD99L2'],
        ['ARHGAP39','CYHR1'],
        ['ARHGAP39','USP32'],
        ['ARHGAP39','VPS8'],
        ['ARHGAP40','NECAB3'],
        ['ARHGAP42','PGR'],
        ['ARHGAP44','DNAH2'],
        ['ARHGAP5','NUBPL'],
        ['ARHGAP5','TBK1'],
        ['ARHGAP5','TPGS2'],
        ['ARHGAP6','MID1'],
        ['ARHGDIA','RNF213'],
        ['ARHGDIA','SQSTM1'],
        ['ARHGDIG','ASTN2'],
        ['ARHGEF1','SERINC2'],
        ['ARHGEF1','STXBP2'],
        ['ARHGEF10L','CSDE1'],
        ['ARHGEF10L','PADI4'],
        ['ARHGEF10L','PAK4'],
        ['ARHGEF11','ETV3'],
        ['ARHGEF12','CNTN5'],
        ['ARHGEF12','SC5D'],
        ['ARHGEF12','SIK3'],
        ['ARHGEF17','EP300'],
        ['ARHGEF17','PROSAP1'],
        ['ARHGEF17','SHANK2'],
        ['ARHGEF17','WDR74'],
        ['ARHGEF18','GNG7'],
        ['ARHGEF18','INSR'],
        ['ARHGEF18','MYO9B'],
        ['ARHGEF18','PAK1'],
        ['ARHGEF18','PKN1'],
        ['ARHGEF2','ASH1L'],
        ['ARHGEF2','PSPC1'],
        ['ARHGEF2','SMG5'],
        ['ARHGEF2','SSR2'],
        ['ARHGEF25','REV3L'],
        ['ARHGEF26','KAL1'],
        ['ARHGEF3','ERC2'],
        ['ARHGEF3','HESX1'],
        ['ARHGEF3','LPAP'],
        ['ARHGEF3','RHAG'],
        ['ARHGEF39','NFX1'],
        ['ARHGEF4','CPE'],
        ['ARHGEF4','IGFBP5'],
        ['ARHGEF4','PPHLN1'],
        ['ARHGEF5','HIPK2'],
        ['ARHGEF7','ATP11A'],
        ['ARHGEF7','KLF10'],
        ['ARHGEF7','MAPK1'],
        ['ARID1A','BEND5'],
        ['ARID1A','C1QC'],
        ['ARID1A','NIPBL'],
        ['ARID1A','PRKCZ'],
        ['ARID1A','RNF31'],
        ['ARID1A','RPS6KA1'],
        ['ARID1A','THBS3'],
        ['ARID1A','YTHDF2'],
        ['ARID1B','BM033'],
        ['ARID1B','ENPP4'],
        ['ARID1B','PARK2'],
        ['ARID1B','PITRM1'],
        ['ARID1B','PLEKHG1'],
        ['ARID1B','PREP'],
        ['ARID1B','SASH1'],
        ['ARID1B','SNX9'],
        ['ARID1B','SYNE1'],
        ['ARID1B','TIAM2'],
        ['ARID1B','TMEM242'],
        ['ARID2','SCAF11'],
        ['ARID2','TMPRSS2'],
        ['ARID2','ZNF641'],
        ['ARID3A','PMEPA1'],
        ['ARID4A','L3HYPDH'],
        ['ARID4A','NPAS3'],
        ['ARID4B','B3GALNT2'],
        ['ARID4B','GNPAT'],
        ['ARID4B','NTPCR'],
        ['ARID4B','PCNXL2'],
        ['ARIH2','C13ORF8'],
        ['ARIH2','CAMP'],
        ['ARIH2','CHAMP1'],
        ['ARIH2','DHX30'],
        ['ARIH2','RBM6'],
        ['ARIH2','RHOA'],
        ['ARIH2','SLC25A20'],
        ['ARIH2','TCTA'],
        ['ARIH2','TRIO'],
        ['ARL15','CDC42SE2'],
        ['ARL15','HSPA9'],
        ['ARL15','MOCS2'],
        ['ARL16','C16ORF70'],
        ['ARL3','CAMK1D'],
        ['ARL3','WBP1L'],
        ['ARL5B','CCDC102B'],
        ['ARL5C','COL11A1'],
        ['ARL5C','SPOP'],
        ['ARL8B','BSN'],
        ['ARMC1','PEX19'],
        ['ARMC1','RB1CC1'],
        ['ARMC1','SPAG11B'],
        ['ARMC1','UBE2W'],
        ['ARMC2','KDM4C'],
        ['ARMC2','KIAA1833'],
        ['ARMC2','MROH1'],
        ['ARMC3','PARD3'],
        ['ARMC3','USP6NL'],
        ['ARMC4','CLPB'],
        ['ARMC4','MPP7'],
        ['ARMC4','PARD3'],
        ['ARMC5','SFRP1'],
        ['ARMC7','HN1'],
        ['ARMC8','CLDN18'],
        ['ARMC8','FOXP1'],
        ['ARMC9','EVL'],
        ['ARNT','F5'],
        ['ARNT','RIIAD1'],
        ['ARNT','RRNAD1'],
        ['ARNT2','MS4A3'],
        ['ARNT2','PDLIM5'],
        ['ARNT2','RORA'],
        ['ARNT2','TNFAIP8L3'],
        ['ARNTL','TEAD1'],
        ['ARNTL2','ITPR2'],
        ['ARNTL2','STAT5A'],
        ['ARPC1A','HIPK2'],
        ['ARPC1B','TRRAP'],
        ['ARPC2','TXNDC15'],
        ['ARRB1','CCDC90B'],
        ['ARRB1','DLG2'],
        ['ARRB1','GAB2'],
        ['ARRB1','GDPD5'],
        ['ARRB1','KLHL35'],
        ['ARRB1','MPP2'],
        ['ARRB1','PAK1'],
        ['ARRB1','PICALM'],
        ['ARRB1','PKN1'],
        ['ARRB1','PRCP'],
        ['ARRB1','RASSF3'],
        ['ARRB1','RSF1'],
        ['ARRB1','UVRAG'],
        ['ARRDC1','ZMYND19'],
        ['ARRDC2','KCNN1'],
        ['ARSG','CEP112'],
        ['ARSG','KIAA1738'],
        ['ARSG','MAP2K6'],
        ['ARSG','TEX2'],
        ['ARSG','TMCC1'],
        ['ART1','DDX5'],
        ['ART1','G17P1'],
        ['ART3','CDKL2'],
        ['ART3','NUP54'],
        ['AS','RNASEH2B'],
        ['AS3MT','GBF1'],
        ['AS3MT','PI4K2A'],
        ['ASAH1','TNKS'],
        ['ASAP1','CLVS1'],
        ['ASAP1','EFR3A'],
        ['ASAP1','FAM49B'],
        ['ASAP1','KCNH1'],
        ['ASAP1','LRRCC1'],
        ['ASAP1','RAD51B'],
        ['ASAP1','SAMD12'],
        ['ASB1','CLVS1'],
        ['ASB1','TRAF3IP1'],
        ['ASB15','IQUB'],
        ['ASB2','EFCAB11'],
        ['ASB4','RAI14'],
        ['ASB8','NELL2'],
        ['ASB9','TRIM41'],
        ['ASB9','WDR59'],
        ['ASCC1','MICU1'],
        ['ASCC1','NR2C2'],
        ['ASCC1','PSAP'],
        ['ASCC2','EIF3L'],
        ['ASCC2','ZMAT5'],
        ['ASCC2','ZNRF3'],
        ['ASCC3','ATL1'],
        ['ASCC3','GRIK2'],
        ['ASCC3','GRIK5'],
        ['ASCC3','HIVEP3'],
        ['ASCC3','PPP2R5D'],
        ['ASCC3','SERINC1'],
        ['ASCC3','SIM1'],
        ['ASGR2','TAOK1'],
        ['ASH1L','C1ORF87'],
        ['ASH1L','CLK2'],
        ['ASH1L','FAM78B'],
        ['ASH1L','GATAD2B'],
        ['ASH1L','GPATCH4'],
        ['ASH1L','INSRR'],
        ['ASH1L','P66BETA'],
        ['ASH1L','SHC1'],
        ['ASH2L','C8ORF86'],
        ['ASH2L','MGME1'],
        ['ASH2L','WHSC1L1'],
        ['ASIC2','KIAA1501'],
        ['ASIC2','MYO1D'],
        ['ASIP','ITCH'],
        ['ASPA','RABEP1'],
        ['ASPDH','KCNC3'],
        ['ASPH','CHD7'],
        ['ASPH','ESR1'],
        ['ASPH','RSF1'],
        ['ASPSCR1','COL5A3'],
        ['ASPSCR1','CSNK1D'],
        ['ASPSCR1','TBCD'],
        ['ASS1','NCS1'],
        ['ASTN1','TRIO'],
        ['ASTN2','LRRC37A5P'],
        ['ASTN2','NUP188'],
        ['ASTN2','PNMT'],
        ['ASTN2','PTBP3'],
        ['ASTN2','STRBP'],
        ['ASTN2','ZCCHC7'],
        ['ASXL1','ERBB2IP'],
        ['ASXL1','PDRG1'],
        ['ASXL1','RALY'],
        ['ASXL1','TM9SF4'],
        ['ASXL1','XKR7'],
        ['ASXL2','FKBP8'],
        ['ASXL2','HSPH1'],
        ['ASXL2','KIF3C'],
        ['ASXL2','SRBD1'],
        ['ASZ1','LRP12'],
        ['ASZ1','ST7'],
        ['ASZ1','ST7-OT4'],
        ['ATAD1','ATRNL1'],
        ['ATAD1','BTBD9'],
        ['ATAD1','HNRNPF'],
        ['ATAD1','HNRNPH3'],
        ['ATAD1','KIAA0534'],
        ['ATAD1','PAPSS2'],
        ['ATAD1','STAMBPL1'],
        ['ATAD1','SYN1'],
        ['ATAD2','C8ORF76'],
        ['ATAD2','ERBB4'],
        ['ATAD2','FAM178B'],
        ['ATAD2','LRP12'],
        ['ATAD2','PODXL'],
        ['ATAD2','RAD51B'],
        ['ATAD2','WDYHV1'],
        ['ATAD2B','ITSN2'],
        ['ATAD5','EFCAB5'],
        ['ATAD5','NF1'],
        ['ATAD5','PTP4A1'],
        ['ATAD5','SIPA1L3'],
        ['ATE1','SLTM'],
        ['ATE1','ZRANB1'],
        ['ATF1','GRIP1'],
        ['ATF1','LGALS12'],
        ['ATF1','NCOA2'],
        ['ATF3','CDH20'],
        ['ATF3','PCDHGB4'],
        ['ATF5','PRKDC'],
        ['ATF5','ZNF534'],
        ['ATF6','ATP6V0A1'],
        ['ATF6','GJA5'],
        ['ATF6','MAGI3'],
        ['ATF6','NMNAT2'],
        ['ATF6','NOS1AP'],
        ['ATF6','PCP4L1'],
        ['ATF6','SH2D1B'],
        ['ATF6B','CMAHP'],
        ['ATF7','ATP5G2'],
        ['ATF7','GGA3'],
        ['ATF7','KRT83'],
        ['ATF7','NELL2'],
        ['ATF7','SPATS2'],
        ['ATF7','STK24'],
        ['ATF7IP','COL16A1'],
        ['ATG10','CREBBP'],
        ['ATG13','ELP4'],
        ['ATG13','TSPAN18'],
        ['ATG14','FUT8'],
        ['ATG14','KTN1-AS1'],
        ['ATG14','STRBP'],
        ['ATG16L1','DGKD'],
        ['ATG16L2','LRP5'],
        ['ATG16L2','ZWILCH'],
        ['ATG2A','EFCAB6'],
        ['ATG2A','EHD1'],
        ['ATG2A','MS4A4A'],
        ['ATG2A','MS4A7'],
        ['ATG2B','SLC22A17'],
        ['ATG4B','BMPR2'],
        ['ATG4B','DTYMK'],
        ['ATG4C','ATP1A1'],
        ['ATG4C','INADL'],
        ['ATG4C','NFIA'],
        ['ATG4D','UHRF1'],
        ['ATG5','C6ORF210'],
        ['ATG5','CEACAM16'],
        ['ATG5','LTV1'],
        ['ATG5','PDSS2'],
        ['ATG5','PITRM1'],
        ['ATG5','PREP'],
        ['ATG5','QRSL1'],
        ['ATG5','SYNJ2'],
        ['ATG5','TMEM200A'],
        ['ATG7','BRAF'],
        ['ATG7','FYCO1'],
        ['ATG7','PPARG'],
        ['ATG7','SETD5'],
        ['ATIC','FAM117B'],
        ['ATL1','CDKL1'],
        ['ATL1','SAV1'],
        ['ATL2','DHX57'],
        ['ATL2','HNRNPLL'],
        ['ATL2','HNRPLL'],
        ['ATL3','ZDHHC14'],
        ['ATM','CUL5'],
        ['ATM','JARID2'],
        ['ATOH8','GNLY'],
        ['ATOX1','FAT2'],
        ['ATP10B','RP11-109J4.1'],
        ['ATP10D','GABRB1'],
        ['ATP11A','C13ORF35'],
        ['ATP11A','TUBGCP3'],
        ['ATP11A','ULK4'],
        ['ATP11B','GSK3B'],
        ['ATP11B','MED10'],
        ['ATP11B','NAALADL2'],
        ['ATP11B','TFRC'],
        ['ATP11C','GS1-256O22.5'],
        ['ATP11C','PREX1'],
        ['ATP13A1','MYO9B'],
        ['ATP13A3','CBFB'],
        ['ATP13A4','MED12'],
        ['ATP13A4','NAALADL2'],
        ['ATP13A4','OPA1'],
        ['ATP1A1','NOTCH2'],
        ['ATP1A1','RABEP1'],
        ['ATP1A1','VANGL1'],
        ['ATP1A3','HKR1'],
        ['ATP1A4','NCSTN'],
        ['ATP1B1','RCSD1'],
        ['ATP1B3','GRK7'],
        ['ATP1B3','PRKAA2'],
        ['ATP1B3','RNF7'],
        ['ATP1B4','CHST11'],
        ['ATP2A2','C12ORF49'],
        ['ATP2A2','EP400'],
        ['ATP2A2','IFT81'],
        ['ATP2A2','NIPBL'],
        ['ATP2A3','FGF3'],
        ['ATP2A3','INT2'],
        ['ATP2B1','KCNC2'],
        ['ATP2B2','SRGAP3'],
        ['ATP2B4','ERBB4'],
        ['ATP2B4','NEK7'],
        ['ATP2B4','NFASC'],
        ['ATP2B4','SRGAP3'],
        ['ATP2C1','NEK11'],
        ['ATP2C1','PSMD1'],
        ['ATP4B','GAS6'],
        ['ATP5C1','TAF3'],
        ['ATP5G3','GSTP1'],
        ['ATP6V0A1','MB'],
        ['ATP6V0A1','MSI2'],
        ['ATP6V0A1','STAT5A'],
        ['ATP6V0D1','DUS2'],
        ['ATP6V1A','MED12'],
        ['ATP6V1A','OPA1'],
        ['ATP6V1C1','AZIN1'],
        ['ATP6V1C1','GBA2'],
        ['ATP6V1C2','RHOBTB1'],
        ['ATP6V1E2','GRIK1'],
        ['ATP6V1F','CPED1'],
        ['ATP6V1H','D6S60'],
        ['ATP6V1H','FCHSD2'],
        ['ATP6V1H','MAPRE2'],
        ['ATP6V1H','RP1'],
        ['ATP6V1H','ST8SIA4'],
        ['ATP6V1H','STK19'],
        ['ATP8A1','LDB2'],
        ['ATP8A2','CRYL1'],
        ['ATP8A2','MICU2'],
        ['ATP8A2','RNF13'],
        ['ATP8B4','DPH6'],
        ['ATP8B4','ODF3B'],
        ['ATP8B4','TYMP'],
        ['ATP9A','MACROD2'],
        ['ATP9B','BCL3'],
        ['ATPAF1','OSBPL9'],
        ['ATPAF2','TOM1L2'],
        ['ATR','DBR1'],
        ['ATR','GPR98'],
        ['ATR','WWTR1'],
        ['ATRN','RNF24'],
        ['ATRNL1','IQGAP2'],
        ['ATRX','BCL2'],
        ['ATRX','ZNF304'],
        ['ATXN1','CGA'],
        ['ATXN1','HMGCLL1'],
        ['ATXN1','PCTP'],
        ['ATXN1','RANBP9'],
        ['ATXN1','TINAGL1'],
        ['ATXN10','FBLN1'],
        ['ATXN10','GTSE1'],
        ['ATXN10','LIMK2'],
        ['ATXN10','SH3BP1'],
        ['ATXN10','TNRC6B'],
        ['ATXN10','TUBGCP6'],
        ['ATXN10','Z83844.1'],
        ['ATXN2','C16ORF87'],
        ['ATXN2','CCDC59'],
        ['ATXN2','GIT2'],
        ['ATXN2','RPL6'],
        ['ATXN3','CATSPERB'],
        ['ATXN7','OXSR1'],
        ['ATXN7L1','LAMB1'],
        ['ATXN7L1','SRPK2'],
        ['ATXN7L1','TICRR'],
        ['AUH','SPTAN1'],
        ['AUNIP','PADI1'],
        ['AUNIP','SRRM1'],
        ['AURKA','PCK1'],
        ['AURKA','SOGA1'],
        ['AURKB','MYH10'],
        ['AURKC','PPP6R1'],
        ['AUTS2','EZH2'],
        ['AUTS2','KIAA1033'],
        ['AUTS2','PNLIPRP3'],
        ['AUTS2','SEMA3C'],
        ['AUTS2','TBC1D30'],
        ['AVEN','EIF2AK4'],
        ['AVIL','HIP1R'],
        ['AVIL','NDUFA4L2'],
        ['AVIL','TMEM120B'],
        ['AVIL','UNK'],
        ['AVL9','BBS9'],
        ['AVL9','PSCA'],
        ['AXIN1','BBX'],
        ['AXIN1','IFT140'],
        ['AXIN1','ITCH'],
        ['AXIN1','ITFG3'],
        ['AXIN2','BCAS3'],
        ['AXIN2','INTS2'],
        ['AXIN2','KAT7'],
        ['AXL','ZNF565'],
        ['AZGP1','FAR1'],
        ['AZGP1','SMURF1'],
        ['AZIN1','RNF19A'],
        ['B2M','KRT5'],
        ['B3GALTL','C13ORF26'],
        ['B3GALTL','MTUS2'],
        ['B3GALTL','STK33'],
        ['B3GALTL','TEX26'],
        ['B3GAT2','ZNF292'],
        ['B3GNT5','CAMK2N2'],
        ['B3GNTL1','CLCN4'],
        ['B3GNTL1','LRRC3C'],
        ['B3GNTL1','WDR45B'],
        ['B3GNTL1','ZNF750'],
        ['B4GALNT2','CWC25'],
        ['B4GALNT2','PEMT'],
        ['B4GALNT2','THRA'],
        ['B4GALNT2','UNK'],
        ['B4GALNT3','CCDC77'],
        ['B4GALNT3','SLCO4A1'],
        ['B4GALNT3','WNK1'],
        ['B4GALT1','BAG1'],
        ['B4GALT1','FAM201A'],
        ['B4GALT1','KIAA1045'],
        ['B4GALT1','NFX1'],
        ['B4GALT1','SMU1'],
        ['B4GALT2','JAK1'],
        ['B4GALT3','BCAS3'],
        ['B4GALT5','TAF4'],
        ['B4GALT5','TOX2'],
        ['B4GALT6','ROCK1'],
        ['B4GALT6','SETD4'],
        ['B4GALT7','NSD1'],
        ['B9D1','EPN2'],
        ['BA457M11.3','PPIL6'],
        ['BAALC','RIMS2'],
        ['BAALC','UBR5'],
        ['BAAT','ZNF189'],
        ['BACH1','DSCR3'],
        ['BACH1','MAP3K7CL'],
        ['BACH1','TENM1'],
        ['BACH2','MAP3K7'],
        ['BACH2','RNGTT'],
        ['BACH2','TBC1D22A'],
        ['BAD','PLCB3'],
        ['BAG2','RNF114'],
        ['BAG2','ZNF451'],
        ['BAG4','FGFR1'],
        ['BAG4','NEDD8'],
        ['BAG4','TP53INP1'],
        ['BAG4','ZMAT4'],
        ['BAG5','MARK3'],
        ['BAG6','LY6G5C'],
        ['BAG6','NG33'],
        ['BAG6','TAF8'],
        ['BAI1','EXT1'],
        ['BAI2','HS3ST4'],
        ['BAI2','SNRNP40'],
        ['BAI2','SPOCD1'],
        ['BAIAP2','NLE1'],
        ['BAIAP2','TBCD'],
        ['BAIAP2L1','MET'],
        ['BAIAP2L2','PLA2G6'],
        ['BANF1','EIF1AD'],
        ['BANK1','PPP3CA'],
        ['BARX2','ST14'],
        ['BATF2','EHD1'],
        ['BATF2','FARS2'],
        ['BATF2','MARK2'],
        ['BAX','DHDH'],
        ['BAZ1A','NKX2-1'],
        ['BAZ1A','RNF115'],
        ['BAZ1A','SFTA3'],
        ['BAZ1A','ZNF253'],
        ['BAZ1A','ZNF364'],
        ['BAZ1B','GRID1'],
        ['BAZ2A','BIN2'],
        ['BAZ2A','PRIM1'],
        ['BAZ2B','TANC1'],
        ['BAZ2B','WDSUB1'],
        ['BB1','EMC10'],
        ['BB1','TMC4'],
        ['BBOX1','CELF1'],
        ['BBS1','C11ORF80'],
        ['BBS4','TTC23'],
        ['BBS9','C11ORF30'],
        ['BBX','PRDM16'],
        ['BBX','RB1'],
        ['BBX','TMEM39A'],
        ['BCAP31','FLJ34125'],
        ['BCAP31','PDZD4'],
        ['BCAR1','ZFP1'],
        ['BCAR3','CAMSAP2'],
        ['BCAR3','ODF3L2'],
        ['BCAS1','GFOD1'],
        ['BCAS1','RARG'],
        ['BCAS1','SLC12A8'],
        ['BCAS3','BPTF'],
        ['BCAS3','BRIP1'],
        ['BCAS3','CYTH1'],
        ['BCAS3','DISP1'],
        ['BCAS3','GH2'],
        ['BCAS3','GIT1'],
        ['BCAS3','GPI'],
        ['BCAS3','GRIK2'],
        ['BCAS3','GRIK5'],
        ['BCAS3','IKZF3'],
        ['BCAS3','KAT7'],
        ['BCAS3','MARCH10'],
        ['BCAS3','MRPL36'],
        ['BCAS3','NFS1'],
        ['BCAS3','NLK'],
        ['BCAS3','NURF301'],
        ['BCAS3','PIGU'],
        ['BCAS3','POLD4'],
        ['BCAS3','PRKAA2'],
        ['BCAS3','RAD51C'],
        ['BCAS3','RNF169'],
        ['BCAS3','RNFT1'],
        ['BCAS3','SKAP1'],
        ['BCAS3','TAOK1'],
        ['BCAS3','USP32'],
        ['BCAS3','VMP1'],
        ['BCAS3','WIPI1'],
        ['BCAS3','ZNF217'],
        ['BCAS4','PTPN1'],
        ['BCAS4','ZNRF1'],
        ['BCAT1','ITPR2'],
        ['BCAT1','PRB3'],
        ['BCAT1','RECQL'],
        ['BCAT2','IRF3'],
        ['BCCIP','EDRF1'],
        ['BCKDHA','CARD8'],
        ['BCKDHB','RIPK1'],
        ['BCKDHB','RP11-250B2.3'],
        ['BCKDHB','TTK'],
        ['BCL11A','EHBP1'],
        ['BCL11A','PHACTR1'],
        ['BCL11A','TIA1'],
        ['BCL2','NEDD4L'],
        ['BCL2','XH2'],
        ['BCL2A1','EPB41L1'],
        ['BCL2A1','MTHFS'],
        ['BCL2L11','BRAF'],
        ['BCL2L13','HIRA'],
        ['BCL2L13','SLC39A9'],
        ['BCL2L13','SPECC1L'],
        ['BCL2L13','ZNF383'],
        ['BCL2L14','MSRB3'],
        ['BCL2L2','PABPN1'],
        ['BCL3','IGSF23'],
        ['BCL7A','KCNIP3'],
        ['BCL7A','S100A6'],
        ['BCL7C','CTF1'],
        ['BCL7C','ZNF720'],
        ['BCL9','IVL'],
        ['BCL9','SMCP'],
        ['BCL9','UAP1'],
        ['BCLAF1','MAP7'],
        ['BCMO1','CMIP'],
        ['BCOR','EP300'],
        ['BCOR','L3MBTL2'],
        ['BCR','MOV10L1'],
        ['BCR','MRVI1'],
        ['BCR','MYH11'],
        ['BCR','PPIL2'],
        ['BCR','PTPRD'],
        ['BCR','TOM1'],
        ['BCR','UPB1'],
        ['BDH1','DLG1'],
        ['BDH1','DLGAP5'],
        ['BDH1','NEK11'],
        ['BDH1','PAK2'],
        ['BDH1','ZBBX'],
        ['BDNF','GPBP1'],
        ['BDP1','CWC27'],
        ['BDP1','USP8'],
        ['BEND3','C6ORF203'],
        ['BEND3','MED23'],
        ['BEND5','SPATA6'],
        ['BEST2','RNASEH2A'],
        ['BET1','COL26A1'],
        ['BET1','PHTF2'],
        ['BFAR','SPNS1'],
        ['BFD','D6S204'],
        ['BFD','HLA-C'],
        ['BFD','HSP90B1'],
        ['BFD','RPL18'],
        ['BFD','RPL8'],
        ['BFD','TSPAN3'],
        ['BFSP2','EPHB1'],
        ['BFSP2','PIK3CB'],
        ['BFSP2','SLC12A8'],
        ['BGN','KIAA2026'],
        ['BGN','TTN'],
        ['BGN','ZNF827'],
        ['BHLHE41','SHROOM2'],
        ['BICC1','DPYSL4'],
        ['BICC1','UBE2D1'],
        ['BICD1','CDK6'],
        ['BICD1','SOX5'],
        ['BICD1','ZNF438'],
        ['BIK','TBX3'],
        ['BIN3','PEBP4'],
        ['BIRC5','PGS1'],
        ['BIRC5','TPGS1'],
        ['BIRC6','CRIPT'],
        ['BIRC6','HDLBP'],
        ['BIRC6','IYD'],
        ['BIRC6','LTBP1'],
        ['BLM','CSTF1'],
        ['BLM','RCCD1'],
        ['BLM','ROCK2'],
        ['BLMH','SPAG5'],
        ['BLNK','PIK3AP1'],
        ['BLNK','UBN1'],
        ['BLNK','ZNF536'],
        ['BLOC1S5','CD2AP'],
        ['BLOC1S5','RANBP9'],
        ['BLZF1','MPC2'],
        ['BM033','PITRM1'],
        ['BM033','PREP'],
        ['BM033','ZDHHC14'],
        ['BMI1','GNA13'],
        ['BMP2','PLCB4'],
        ['BMP2','TRIB3'],
        ['BMP2K','CCSER1'],
        ['BMP2K','SHROOM3'],
        ['BMP2K','SLC25A31'],
        ['BMP2K','SLC4A4'],
        ['BMP3','TRAF3'],
        ['BMP4','RPS6KA5'],
        ['BMP5','COL21A1'],
        ['BMP5','ETHE1'],
        ['BMP5','RANBP2'],
        ['BMP7','RBM38'],
        ['BMPER','TBC1D15'],
        ['BMPR1A','C10ORF11'],
        ['BMPR1A','ZYG11A'],
        ['BMPR1B','DDAH1'],
        ['BMPR1B','ETV1'],
        ['BMPR1B','MCC'],
        ['BMPR1B','NPTX1'],
        ['BMPR1B','NRG1'],
        ['BMPR1B','NSUN4'],
        ['BMPR1B','PAWR'],
        ['BMPR1B','PDLIM5'],
        ['BMPR1B','ZMIZ1'],
        ['BMPR2','CADM2'],
        ['BMPR2','CYP20A1'],
        ['BMPR2','KANSL1L'],
        ['BMPR2','MARCH4'],
        ['BMPR2','SLCO3A1'],
        ['BMPR2','SPATS2L'],
        ['BMPR2','WDTC1'],
        ['BMS1','PLEKHA7'],
        ['BMX','RPS6KA3'],
        ['BNC2','CDC37L1'],
        ['BNC2','ESR1'],
        ['BNC2','KIAA1432'],
        ['BNIP2','MYO1E'],
        ['BNIP3L','EPHX2'],
        ['BOK','MYEOV2'],
        ['BOK','PASK'],
        ['BOLA1','HSPC099'],
        ['BOLA1','RPRD2'],
        ['BOP1','CYHR1'],
        ['BOP1','NIPAL2'],
        ['BPG116M5.10','C1ORF43'],
        ['BPI','DLGAP4'],
        ['BPI','GTF2B'],
        ['BPIFA1','CLDN7'],
        ['BPIFA2','MARK2'],
        ['BPIFB1','FRMD6'],
        ['BPIFB1','PLCG1'],
        ['BPIFB1','SNAPC5'],
        ['BPIFB4','SNAPC5'],
        ['BPTF','CCDC40'],
        ['BPTF','FAM117A'],
        ['BPTF','PIP4K2B'],
        ['BPTF','PITPNC1'],
        ['BPTF','RGS9'],
        ['BPTF','RGS9BP'],
        ['BRAF','CCNY'],
        ['BRAF','CDC27'],
        ['BRAF','CIITA'],
        ['BRAF','FAM114A2'],
        ['BRAF','FLJ10842'],
        ['BRAF','HIBADH'],
        ['BRAF','KDM7A'],
        ['BRAF','KIAA1549'],
        ['BRAF','MACF1'],
        ['BRAF','MKRN1'],
        ['BRAF','MRPS33'],
        ['BRAF','SLC26A4'],
        ['BRAF','SND1'],
        ['BRAF','SUGCT'],
        ['BRAF','TAX1BP1'],
        ['BRAF','TRIM24'],
        ['BRAF','ZC3HAV1'],
        ['BRAP','FAM216A'],
        ['BRAT1','HDAC4'],
        ['BRCA1','ZNF675'],
        ['BRCC3','VBP1'],
        ['BRD1','C22ORF34'],
        ['BRD1','GRAP2'],
        ['BRD1','MAPK12'],
        ['BRD1','MLC1'],
        ['BRD1','SLC3A2'],
        ['BRD2','PIGR'],
        ['BRD3','PIP5KL1'],
        ['BRD4','EPS15L1'],
        ['BRD4','FRMD3'],
        ['BRD9','CEP72'],
        ['BRDT','MTF2'],
        ['BRE','CLIP4'],
        ['BRE','DYNC1I2'],
        ['BRE','FOSL2'],
        ['BRE','PLB1'],
        ['BRE','PPP1CB'],
        ['BRE','PRKCE'],
        ['BRE','SLC30A6'],
        ['BRE','UBASH3A'],
        ['BRF1','FOXN3'],
        ['BRF1','UNC79'],
        ['BRINP3','XPR1'],
        ['BRIP1','CUEDC1'],
        ['BRIP1','DDX42'],
        ['BRIP1','DSCR3'],
        ['BRIP1','MAP3K7CL'],
        ['BRIP1','TENM1'],
        ['BRPF3','HLA-DMA'],
        ['BRPF3','STARD9'],
        ['BRPF3','ZFAND3'],
        ['BRS3','HTATSF1'],
        ['BRSK1','COX6B1'],
        ['BRSK1','HSPBP1'],
        ['BRSK1','ZNF587'],
        ['BRWD1','DYRK1A'],
        ['BRWD1','HUNK'],
        ['BRWD1','LCA5L'],
        ['BRWD1','LMTK3'],
        ['BRWD1','SMC1A'],
        ['BRWD1','SMCB'],
        ['BRWD1','TRAPPC10'],
        ['BRWD1','TTC3'],
        ['BRWD1','WDR4'],
        ['BSCL2','TUT1'],
        ['BSCL2','UBXN1'],
        ['BSDC1','HDAC1'],
        ['BSDC1','YARS'],
        ['BSG','EEA1'],
        ['BSG','HCN2'],
        ['BSG','STK11'],
        ['BSPRY','NPM1'],
        ['BSPRY','RGS3'],
        ['BTBD1','MESP1'],
        ['BTBD1','MRPL46'],
        ['BTBD2','LINGO3'],
        ['BTBD2','NIFK'],
        ['BTBD2','OAS2'],
        ['BTBD3','GCNT2'],
        ['BTBD7','ITPK1'],
        ['BTBD7','TARBP1'],
        ['BTBD7','UBR7'],
        ['BTBD9','CLPSL1'],
        ['BTBD9','FARS2'],
        ['BTF3L4','STMN1'],
        ['BTN2A1','HIST1H2AC'],
        ['BTN2A1','HIST1H2BC'],
        ['BTN3A2','HIST1H2AC'],
        ['BTRC','CDH23'],
        ['BTRC','HIF1AN'],
        ['BUB1B','CCDC170'],
        ['BUB1B','EIF2AK4'],
        ['BUB1B','MARK2'],
        ['BUB1B','PCM1'],
        ['BZW2','GUSB'],
        ['C10ORF11','CYP2C19'],
        ['C10ORF11','GLRX3'],
        ['C10ORF11','SON'],
        ['C10ORF11','VCL'],
        ['C10ORF11','ZNF503-AS2'],
        ['C10ORF120','NT5C2'],
        ['C10ORF18','NET1'],
        ['C10ORF25','WDR70'],
        ['C10ORF26','SUFU'],
        ['C10ORF32','PI4K2A'],
        ['C10ORF35','EEF2K'],
        ['C10ORF53','MAPK8'],
        ['C10ORF54','RPS6KB2'],
        ['C10ORF68','CCDC7'],
        ['C10ORF88','PPAPDC1A'],
        ['C10ORF90','SMAD2'],
        ['C10ORF90','VTI1A'],
        ['C10ORF91','INPP5A'],
        ['C11ORF1','SIK3'],
        ['C11ORF30','CACNB1'],
        ['C11ORF30','INTS4'],
        ['C11ORF30','UVRAG'],
        ['C11ORF41','RAB18'],
        ['C11ORF41','WARS2'],
        ['C11ORF48','CDK6'],
        ['C11ORF49','F2'],
        ['C11ORF49','PPME1'],
        ['C11ORF49','PROSAP1'],
        ['C11ORF49','SHANK2'],
        ['C11ORF57','DLAT'],
        ['C11ORF65','PIBF1'],
        ['C11ORF67','SCGB1D1'],
        ['C11ORF67','TMEM135'],
        ['C11ORF70','DYNC2H1'],
        ['C11ORF70','YAP1'],
        ['C11ORF80','F3'],
        ['C11ORF80','KDM2A'],
        ['C11ORF80','MKNK1'],
        ['C11ORF80','PCNXL3'],
        ['C11ORF80','RBM14'],
        ['C11ORF80','TMX2'],
        ['C11ORF80','TSGA10IP'],
        ['C11ORF83','FCHSD2'],
        ['C11ORF84','MARK2'],
        ['C11ORF84','PPP6C'],
        ['C11ORF84','RIC8B'],
        ['C11ORF85','SDHAF2'],
        ['C11ORF85','SNX15'],
        ['C12ORF10','CTDSP2'],
        ['C12ORF23','RFX4'],
        ['C12ORF4','EFCAB4B'],
        ['C12ORF42','CORO1C'],
        ['C12ORF42','TM7SF3'],
        ['C12ORF45','CDK7'],
        ['C12ORF49','MDM2'],
        ['C12ORF49','SLC35E3'],
        ['C12ORF49','SRRM4'],
        ['C12ORF5','TSPAN9'],
        ['C12ORF52','GRHL2'],
        ['C12ORF56','PPM1H'],
        ['C12ORF56','TBK1'],
        ['C12ORF56','XPOT'],
        ['C12ORF65','NUAK1'],
        ['C12ORF66','OS9'],
        ['C12ORF66','PLAT'],
        ['C12ORF70','PPFIBP1'],
        ['C12ORF73','GLT8D2'],
        ['C12ORF75','MKRN2'],
        ['C12ORF79','SCYL2'],
        ['C13ORF8','ZNF589'],
        ['C14ORF132','RPS6KA5'],
        ['C14ORF132','SMAP1'],
        ['C14ORF166B','CTNNA3'],
        ['C14ORF182','CDC42BPB'],
        ['C14ORF182','NEMF'],
        ['C14ORF2','PPP1R13B'],
        ['C14ORF37','KIAA0586'],
        ['C14ORF39','DCAF5'],
        ['C15ORF27','CSK'],
        ['C15ORF32','CLN3'],
        ['C15ORF41','CSMD1'],
        ['C15ORF41','FAM98B'],
        ['C15ORF41','LYSMD4'],
        ['C15ORF41','MEIS2'],
        ['C15ORF41','UBAP2'],
        ['C15ORF41','WWOX'],
        ['C15ORF43','GALK2'],
        ['C15ORF57','CBX3'],
        ['C15ORF59','CLN6'],
        ['C15ORF59','VPS13C'],
        ['C16ORF45','KLHL3'],
        ['C16ORF45','POMT2'],
        ['C16ORF45','RHBDL2'],
        ['C16ORF45','STAG2'],
        ['C16ORF62','FAM192A'],
        ['C16ORF62','TMC5'],
        ['C16ORF62','ZNF598'],
        ['C16ORF87','CNOT1'],
        ['C17ORF101','CSNK1D'],
        ['C17ORF105','TAOK1'],
        ['C17ORF51','CACNG4'],
        ['C17ORF62','HEXDC'],
        ['C17ORF64','HEATR6'],
        ['C17ORF64','LUC7L3'],
        ['C17ORF64','TOP1'],
        ['C17ORF70','NPLOC4'],
        ['C17ORF75','TOP2A'],
        ['C17ORF75','ZNF207'],
        ['C17ORF80','SLC39A11'],
        ['C17ORF85','CAMKK1'],
        ['C17ORF85','TAX1BP3'],
        ['C17ORF98','STARD3'],
        ['C18ORF56','TYMS'],
        ['C19ORF10','MAP2K3'],
        ['C19ORF20','SBNO2'],
        ['C19ORF25','MBD4'],
        ['C19ORF25','MED1'],
        ['C19ORF26','SBNO2'],
        ['C19ORF38','CARM1'],
        ['C19ORF38','LDLR'],
        ['C19ORF38','SMARCA4'],
        ['C19ORF44','KDM4B'],
        ['C19ORF47','MAP3K10'],
        ['C19ORF47','SPTBN4'],
        ['C19ORF54','SNRPA'],
        ['C19ORF55','IGFLR1'],
        ['C19ORF57','CUTC'],
        ['C19ORF57','HAUS8'],
        ['C19ORF66','TRPS1'],
        ['C1D','TMEM194A'],
        ['C1D','XPO5'],
        ['C1ORF100','CEP170'],
        ['C1ORF101','DESI2'],
        ['C1ORF101','RPS6KC1'],
        ['C1ORF101','SDCCAG8'],
        ['C1ORF105','SUCO'],
        ['C1ORF110','SDE2'],
        ['C1ORF116','LIN9'],
        ['C1ORF132','PROX1'],
        ['C1ORF132','RPS6KB1'],
        ['C1ORF143','MARC2'],
        ['C1ORF159','SDF4'],
        ['C1ORF168','HIVEP3'],
        ['C1ORF168','LRRC8D'],
        ['C1ORF172','DNAJC8'],
        ['C1ORF173','NEGR1'],
        ['C1ORF186','PDE4DIP'],
        ['C1ORF194','WDR47'],
        ['C1ORF198','GXYLT2'],
        ['C1ORF21','IRF6'],
        ['C1ORF21','LAMC1'],
        ['C1ORF222','NADK'],
        ['C1ORF222','PTPRF'],
        ['C1ORF43','NME10'],
        ['C1ORF43','NUDT19'],
        ['C1ORF43','RP2'],
        ['C1ORF43','STK19P'],
        ['C1ORF50','EZR'],
        ['C1ORF54','PRUNE'],
        ['C1ORF56','PIP5K1A'],
        ['C1ORF87','OSBPL9'],
        ['C1ORF95','VTI1A'],
        ['C1QL1','EFTUD2'],
        ['C1QL1','SLC16A5'],
        ['C1QTNF1','FAM222B'],
        ['C1S','MAP3K13'],
        ['C20ORF112','COMMD7'],
        ['C20ORF112','MMD'],
        ['C20ORF114','FRMD6'],
        ['C20ORF152','USP16'],
        ['C20ORF194','CEACAM16'],
        ['C20ORF194','EVI5'],
        ['C20ORF194','TDRD10'],
        ['C20ORF196','NCKAP1'],
        ['C20ORF196','SLC1A3'],
        ['C20ORF197','STK4'],
        ['C20ORF70','MARK2'],
        ['C20ORF94','CDS2'],
        ['C20ORF94','SLC24A3'],
        ['C20ORF96','EFNA4'],
        ['C21ORF120','SYNJ1'],
        ['C21ORF62','SYNJ1'],
        ['C21ORF90','GABRA3'],
        ['C22ORF34','ZBED4'],
        ['C2CD2','MID2'],
        ['C2CD2','ZBTB21'],
        ['C2CD2L','DDX10'],
        ['C2CD2L','PTPRJ'],
        ['C2CD2L','ZNF585B'],
        ['C2CD3','CLPB'],
        ['C2CD3','FAM168A'],
        ['C2CD3','NCAM'],
        ['C2CD3','NCAM1'],
        ['C2CD3','P2RY6'],
        ['C2CD3','RNF169'],
        ['C2CD3','SYTL1'],
        ['C2CD3','TENM4'],
        ['C2CD3','ZDHHC24'],
        ['C2CD4C','CNN2'],
        ['C2CD5','ITPR2'],
        ['C2CD5','PLCZ1'],
        ['C2CD5','TEAD4'],
        ['C2CD5','TRAPPC9'],
        ['C2ORF43','PUM2'],
        ['C2ORF43','RTN4'],
        ['C2ORF54','KIF1A'],
        ['C2ORF55','MGAT4A'],
        ['C2ORF61','PI4K2B'],
        ['C2ORF74','ZNF638'],
        ['C2ORF78','TET3'],
        ['C2ORF78','ZNF638'],
        ['C2ORF88','MFSD6'],
        ['C3','ELK3'],
        ['C3','ILF3'],
        ['C3','KCNH8'],
        ['C3','NOTCH3'],
        ['C3','PIP5K1C'],
        ['C3','RABEP1'],
        ['C3','SIAE'],
        ['C3ORF14','ZBTB20'],
        ['C3ORF33','PLCH1'],
        ['C3ORF52','GCSAM'],
        ['C3ORF55','CHRD'],
        ['C3ORF55','GMPS'],
        ['C3ORF67','MAP1B'],
        ['C3ORF70','LSAMP'],
        ['C3ORF70','TPRG1'],
        ['C3ORF71','TRIO'],
        ['C4BPA','CCDC109B'],
        ['C4BPA','CD55'],
        ['C4ORF3','PDE5A'],
        ['C5','CDC42BPA'],
        ['C5ORF13','WDR36'],
        ['C5ORF22','MAP2K5'],
        ['C5ORF38','MYO10'],
        ['C5ORF42','MTMR12'],
        ['C5ORF42','NPR3'],
        ['C5ORF42','NPRL3'],
        ['C5ORF42','NUP155'],
        ['C5ORF45','EIF4G2'],
        ['C5ORF51','FOXP1'],
        ['C6ORF106','LRRC1'],
        ['C6ORF106','SPDEF'],
        ['C6ORF132','DERL1'],
        ['C6ORF132','GUCA1B'],
        ['C6ORF14','GMDS'],
        ['C6ORF203','PTPRK'],
        ['C6ORF210','MAN1A1'],
        ['C6ORF210','SMARCA4'],
        ['C6ORF211','ESR1'],
        ['C6ORF211','PDPK1'],
        ['C6ORF48','TRIM40'],
        ['C6ORF52','MAST2'],
        ['C6ORF52','RHOJ'],
        ['C6ORF57','CTIF'],
        ['C6ORF62','FAM65B'],
        ['C6ORF89','CCND3'],
        ['C6ORF89','CDKAL1'],
        ['C6ORF89','FAM216A'],
        ['C6ORF89','MPHOSPH9'],
        ['C6ORF97','ESR1'],
        ['C7ORF10','CDK13'],
        ['C7ORF42','TYW1'],
        ['C7ORF72','IGH@'],
        ['C7ORF72','IGHMBP2'],
        ['C7ORF72','INSR'],
        ['C8A','RNF11'],
        ['C8ORF12','WARS'],
        ['C8ORF34','MET'],
        ['C8ORF34','PANK1'],
        ['C8ORF34','RGS7'],
        ['C8ORF38','PLEKHF2'],
        ['C8ORF46','LAMA3'],
        ['C8ORF47','GML'],
        ['C8ORF76','NKAIN3'],
        ['C9','FAM157A'],
        ['C9','LIFR'],
        ['C9ORF100','NFX1'],
        ['C9ORF131','VCP'],
        ['C9ORF156','GNA14'],
        ['C9ORF24','NFX1'],
        ['C9ORF3','CORO2A'],
        ['C9ORF3','CPSF6'],
        ['C9ORF3','FANCC'],
        ['C9ORF3','PTPDC1'],
        ['C9ORF46','RCL1'],
        ['C9ORF64','GNAQ'],
        ['C9ORF8','FSD1L'],
        ['C9ORF8','LHX2'],
        ['CA1','LMTK2'],
        ['CA10','MSI2'],
        ['CA10','PLIN2'],
        ['CA10','PRKAG2'],
        ['CA10','PRKCA'],
        ['CA10','RAB38'],
        ['CA10','RARA'],
        ['CA10','RHOT1'],
        ['CA10','ROR1'],
        ['CA13','RBKS'],
        ['CA3','MYBL1'],
        ['CA4','COIL'],
        ['CA6','HHAT'],
        ['CA6','SKI'],
        ['CA8','SCAI'],
        ['CABIN1','GRIN2A'],
        ['CABIN1','HIRA'],
        ['CABIN1','SMARCB1'],
        ['CABLES1','LAMA3'],
        ['CABLES1','LAMA4'],
        ['CABLES1','RBBP8'],
        ['CABLES1','TTC39C'],
        ['CABLES2','LAMA5'],
        ['CABP2','PITPNM1'],
        ['CABYR','ROCK1'],
        ['CACHD1','CLIC4'],
        ['CACHD1','CNN3'],
        ['CACHD1','INADL'],
        ['CACHD1','ROR1'],
        ['CACHD1','RORA'],
        ['CACNA1A','ZNF302'],
        ['CACNA1A','ZNF44'],
        ['CACNA1C','SYNE1'],
        ['CACNA1C','TBL1XR1'],
        ['CACNA1C','TSPAN9'],
        ['CACNA1C','WNT5B'],
        ['CACNA1D','PRKCD'],
        ['CACNA1F','LMO6'],
        ['CACNA1F','PRICKLE3'],
        ['CACNA1G','NF1'],
        ['CACNA1G','ZNF585B'],
        ['CACNA1H','MKL2'],
        ['CACNA1H','UBE2I'],
        ['CACNA1I','SH3BP1'],
        ['CACNA1I','TAB1'],
        ['CACNA2D1','NAALADL2'],
        ['CACNA2D2','RHOA'],
        ['CACNA2D2','THRSP'],
        ['CACNA2D3','FAM19A4'],
        ['CACNA2D3','FOXP1'],
        ['CACNA2D3','IFT74'],
        ['CACNA2D3','PBRM1'],
        ['CACNA2D4','FANCA'],
        ['CACNA2D4','UQCC1'],
        ['CACNA2D4','WDR70'],
        ['CACNA2D4','ZNF274'],
        ['CACNB1','CDC6'],
        ['CACNB1','CDK12'],
        ['CACNB1','KA36'],
        ['CACNB1','KRT40'],
        ['CACNB1','MLLT6'],
        ['CACNB2','FAM188A'],
        ['CACNB2','PRPF18'],
        ['CACNG1','PPP1R37'],
        ['CACNG4','FLJ12977'],
        ['CACNG4','HELZ'],
        ['CACNG4','IQCB1'],
        ['CACNG4','VMP1'],
        ['CACNG6','PAK4'],
        ['CACNG8','PTK2'],
        ['CACTIN','LPCAT2'],
        ['CACUL1','EIF3A'],
        ['CACUL1','EIF6'],
        ['CADM1','COPD'],
        ['CADM2','CPNE8'],
        ['CADM2','FHIT'],
        ['CADM2','HDLBP'],
        ['CADM2','MAP2K3'],
        ['CADM2','MITF'],
        ['CADM2','TEFM'],
        ['CADM3','TADA1'],
        ['CADM4','ETHE1'],
        ['CADPS','FHIT'],
        ['CADPS','RAD50'],
        ['CADPS2','DENND1A'],
        ['CADPS2','DHCR7'],
        ['CADPS2','LMOD2'],
        ['CADPS2','SPTAN1'],
        ['CALCOCO2','PCTP'],
        ['CALCOCO2','RNF169'],
        ['CALCOCO2','SKAP1'],
        ['CALCRL','MFSD6'],
        ['CALD1','CGN'],
        ['CALD1','LRGUK'],
        ['CALD1','PDK4'],
        ['CALM3','CHRNA4'],
        ['CALML3','GSTP1'],
        ['CALML3','KRT10'],
        ['CALML3','KRT13'],
        ['CALML3','KRT14'],
        ['CALML3','KRT5'],
        ['CALML3','KRT6B'],
        ['CALML3','S100A9'],
        ['CALN1','KMT2C'],
        ['CALN1','NTNG1'],
        ['CALN1','NTNG2'],
        ['CALR','KRT13'],
        ['CALR','KRT14'],
        ['CALR','KRT6A'],
        ['CALR3','EPS15L1'],
        ['CAMK1D','FAM188A'],
        ['CAMK1D','FUBP1'],
        ['CAMK1D','PHF21A'],
        ['CAMK1D','PROSER2'],
        ['CAMK1D','TSPAN18'],
        ['CAMK2A','CD74'],
        ['CAMK2A','KIAA1429'],
        ['CAMK2A','PDGFRB'],
        ['CAMK2A','TNIP1'],
        ['CAMK2B','SPIDR'],
        ['CAMK2D','TLL2'],
        ['CAMK2G','NDUFA13'],
        ['CAMK2G','P4HA1'],
        ['CAMK2G','SPG11'],
        ['CAMK4','FHIT'],
        ['CAMKK1','MAP2K2'],
        ['CAMKK2','DIP2B'],
        ['CAMKK2','HPD'],
        ['CAMKK2','KDM2B'],
        ['CAMKK2','NR2C1'],
        ['CAMKK2','PMCH'],
        ['CAMKMT','PRKCE'],
        ['CAMP','ZNF589'],
        ['CAMSAP1','KCNT1'],
        ['CAMTA1','DHRS3'],
        ['CAMTA1','MTOR'],
        ['CAMTA1','PER3'],
        ['CAMTA1','PGD'],
        ['CAMTA1','THAP3'],
        ['CAMTA2','LCK'],
        ['CAMTA2','SPAG7'],
        ['CAND1','TUSC5'],
        ['CANT1','FBXL20'],
        ['CANT1','PSMD3'],
        ['CANX','IL1R2'],
        ['CAP1','KCNQ4'],
        ['CAP1','MACF1'],
        ['CAP2','CSF1R'],
        ['CAP2','MBD4'],
        ['CAP2','MED1'],
        ['CAP2','TEAD1'],
        ['CAPN1','HMGXB3'],
        ['CAPN1','KRT14'],
        ['CAPN12','ZNF607'],
        ['CAPN13','GALNT14'],
        ['CAPN15','WHSC1L1'],
        ['CAPN5','PELI3'],
        ['CAPN5','TBC1D14'],
        ['CAPN6','DCX'],
        ['CAPNS1','CHD1'],
        ['CAPRIN1','COMMD9'],
        ['CAPRIN1','NAT10'],
        ['CAPRIN1','TRIM44'],
        ['CAPRIN2','R3HDM2'],
        ['CAPZA2','MET'],
        ['CAPZA2','SLTM'],
        ['CAPZB','SCEL'],
        ['CAPZB','THRAP3'],
        ['CARD10','RBM34'],
        ['CARD11','NOL11'],
        ['CARD17','SIK2'],
        ['CARKD','USP34'],
        ['CARM1','OLFM2'],
        ['CARM1','SMARCA4'],
        ['CARS','NAP1L4'],
        ['CARS','OSBPL5'],
        ['CASC1','PLEKHA5'],
        ['CASC1','TMTC1'],
        ['CASC3','MBD4'],
        ['CASC3','MED1'],
        ['CASC3','ZNF254'],
        ['CASC4','TGM5'],
        ['CASD1','VWC2'],
        ['CASK','EML4'],
        ['CASK','EYA2'],
        ['CASK','GLIPR1L1'],
        ['CASK','SQRDL'],
        ['CASK','TBL1X'],
        ['CASK','TM9SF2'],
        ['CASKIN2','QRICH2'],
        ['CASP12','LRBA'],
        ['CASP14','ERN1'],
        ['CASP7','EDRF1'],
        ['CASP7','FGFR2'],
        ['CASP8','UBR3'],
        ['CASQ1','CHRM3'],
        ['CASQ2','GNAI3'],
        ['CASQ2','LRP5'],
        ['CASR','FAM162A'],
        ['CASR','LPP'],
        ['CASR','SUCLA2'],
        ['CASS4','ITCH'],
        ['CASS4','VAPB'],
        ['CAST','CDHR3'],
        ['CAST','CTD-2337A12.1'],
        ['CAST','FAM172A'],
        ['CAST','LNPEP'],
        ['CAST','SCGB3A1'],
        ['CASZ1','CTNNBIP1'],
        ['CASZ1','MTOR'],
        ['CASZ1','OBSCN'],
        ['CASZ1','PRKD2'],
        ['CASZ1','RP11-84A14.5'],
        ['CAT','CD3D'],
        ['CAT','ELF5'],
        ['CAT','HIPK3'],
        ['CAT','PDHX'],
        ['CAT','TLK1'],
        ['CATSPERB','FBLN5'],
        ['CATSPERB','SYNE2'],
        ['CATSPERB','TRIP11'],
        ['CATSPERG','PAK4'],
        ['CAV1','LRP12'],
        ['CAV1','ST7'],
        ['CAV2','CUX1'],
        ['CAV2','TES'],
        ['CBFA2T2','CHMP4B'],
        ['CBFA2T2','DOK5'],
        ['CBFA2T2','HM13'],
        ['CBFA2T2','ZNF341'],
        ['CBFA2T3','PIEZO1'],
        ['CBFB','CES4A'],
        ['CBL','KIAA0638'],
        ['CBL','PHLDB1'],
        ['CBL','VWA5A'],
        ['CBLB','FLJ22344'],
        ['CBLB','MCTP1'],
        ['CBLC','RP11-419C5.2'],
        ['CBLL1','COG5'],
        ['CBLL1','SLC26A4'],
        ['CBR3','DOPEY2'],
        ['CBR4','GALNTL6'],
        ['CBS','CHI3L1'],
        ['CBS','CTSB'],
        ['CBS','CTTN'],
        ['CBX1','RNF169'],
        ['CBX1','SSH2'],
        ['CBX3','GARS'],
        ['CBX4','EFCAB5'],
        ['CBX4','RFK'],
        ['CBX5','NFE2'],
        ['CBX7','VPREB1'],
        ['CBY1','EMID1'],
        ['CC2D1A','RNF207'],
        ['CCAR2','FGFR2'],
        ['CCBL1','LRRC8A'],
        ['CCBL2','GTF2B'],
        ['CCDC102B','DTX4'],
        ['CCDC105','MZF1'],
        ['CCDC106','CD37'],
        ['CCDC107','RFX1'],
        ['CCDC109B','COL25A1'],
        ['CCDC11','MYO5B'],
        ['CCDC110','KIAA1430'],
        ['CCDC117','MELK'],
        ['CCDC117','OSTF1'],
        ['CCDC120','HB2E'],
        ['CCDC120','PPP1R3F'],
        ['CCDC126','OSBPL3'],
        ['CCDC127','DSTN'],
        ['CCDC127','TERT'],
        ['CCDC129','DEFB118'],
        ['CCDC129','URGCP'],
        ['CCDC130','DOCK6'],
        ['CCDC130','ZSWIM4'],
        ['CCDC132','CDH1'],
        ['CCDC132','FZR1'],
        ['CCDC132','PPP6R3'],
        ['CCDC132','PRKAG2'],
        ['CCDC134','MEI1'],
        ['CCDC135','USB1'],
        ['CCDC136','NUP205'],
        ['CCDC136','ZKSCAN1'],
        ['CCDC138','GFRAL'],
        ['CCDC148','LY75-CD302'],
        ['CCDC148','PKP4'],
        ['CCDC148','PSMD14'],
        ['CCDC148','WDSUB1'],
        ['CCDC149','GABRB1'],
        ['CCDC151','KRI1'],
        ['CCDC155','PRRG2'],
        ['CCDC167','RRP7A'],
        ['CCDC169','NUPL1'],
        ['CCDC170','ESR1'],
        ['CCDC171','CCNK'],
        ['CCDC171','GNA13'],
        ['CCDC171','USP8'],
        ['CCDC172','KAT6B'],
        ['CCDC176','LIN52'],
        ['CCDC176','RGS6'],
        ['CCDC178','MTERFD1'],
        ['CCDC178','PHF21A'],
        ['CCDC18','RWDD3'],
        ['CCDC180','NDUFA8'],
        ['CCDC19','THADA'],
        ['CCDC23','FOXJ3'],
        ['CCDC25','TMEM259'],
        ['CCDC3','USP6NL'],
        ['CCDC38','MLF2'],
        ['CCDC38','NTN4'],
        ['CCDC40','NURF301'],
        ['CCDC41','RIMBP2'],
        ['CCDC50','DCUN1D1'],
        ['CCDC50','MAP3K13'],
        ['CCDC53','KNTC1'],
        ['CCDC53','LAMC2'],
        ['CCDC57','FASN'],
        ['CCDC57','SIRT7'],
        ['CCDC58','CHD7'],
        ['CCDC58','GOLGB1'],
        ['CCDC6','FGFR2'],
        ['CCDC6','KITLG'],
        ['CCDC6','LIPI'],
        ['CCDC6','RET'],
        ['CCDC6','UBE2D1'],
        ['CCDC6','VPS13B'],
        ['CCDC61','CHST1'],
        ['CCDC61','FUT2'],
        ['CCDC61','ZNF544'],
        ['CCDC64','GALNAC-T9'],
        ['CCDC64','GALNT9'],
        ['CCDC64','PLS3'],
        ['CCDC64','T-PLASTIN'],
        ['CCDC64B','SLC9A8'],
        ['CCDC66','FLNB'],
        ['CCDC67','PANX1'],
        ['CCDC77','PMEPA1'],
        ['CCDC77','STAG1'],
        ['CCDC77','WNK1'],
        ['CCDC83','NELL1'],
        ['CCDC85C','FUT8'],
        ['CCDC86','NTM'],
        ['CCDC88A','SRBD1'],
        ['CCDC88A','ZNF91'],
        ['CCDC88C','PPP1R13B'],
        ['CCDC90B','CLPB'],
        ['CCDC91','FAR2'],
        ['CCDC91','LYRM5'],
        ['CCDC91','NTSR1'],
        ['CCDC91','PDZRN4'],
        ['CCDC91','PTHLH'],
        ['CCDC91','RP11-993B23.3'],
        ['CCDC91','STK38L'],
        ['CCDC94','CHAF1A'],
        ['CCDC97','GTF2A2'],
        ['CCDC97','SIPA1L3'],
        ['CCL11','RPL19'],
        ['CCL13','FRMD8'],
        ['CCL14','FAM71C'],
        ['CCL16','ETV4'],
        ['CCL2','TAOK1'],
        ['CCM2','SUGCT'],
        ['CCM2L','HCK'],
        ['CCNB1','CDK7'],
        ['CCNB3','PPP4R1'],
        ['CCND2','GCN1L1'],
        ['CCND3','GUCA1B'],
        ['CCND3','TBC1D22B'],
        ['CCNF','PDPK1'],
        ['CCNK','EML1'],
        ['CCNK','YY1'],
        ['CCNT2','MAP3K19'],
        ['CCNT2','RAB3GAP1'],
        ['CCNY','CREM'],
        ['CCNY','DENND1A'],
        ['CCNY','PTPRD'],
        ['CCNY','WARS'],
        ['CCR7','SYNRG'],
        ['CCS','RBM14'],
        ['CCSER1','DENND1B'],
        ['CCSER1','FRAS1'],
        ['CCSER1','MAPK10'],
        ['CCSER1','MMRN1'],
        ['CCT2','CXCR5'],
        ['CCT2','PDE3A'],
        ['CCT3','SYT2'],
        ['CCT4','XPO1'],
        ['CCT8','LHX4'],
        ['CCT8','SOGA2'],
        ['CD101','MAN1A2'],
        ['CD151','EFCAB4A'],
        ['CD151','EPS8L2'],
        ['CD177','ZNF709'],
        ['CD19','ZNF430'],
        ['CD1A','SLAMF1'],
        ['CD1C','NEK7'],
        ['CD1C','SLAMF1'],
        ['CD2','RAP1A'],
        ['CD200R1','ZBTB20'],
        ['CD226','SMAD2'],
        ['CD244','CLK2'],
        ['CD247','POU2F1'],
        ['CD27','FAM60A'],
        ['CD276','PARP16'],
        ['CD2AP','NFKBIE'],
        ['CD2AP','PTCHD4'],
        ['CD2AP','SUPT3H'],
        ['CD2AP','USP49'],
        ['CD2AP','ZNF184'],
        ['CD337','G2'],
        ['CD337','PRRC2A'],
        ['CD337','PTK2'],
        ['CD34','HHAT'],
        ['CD34','RPS6KC1'],
        ['CD36','ITK'],
        ['CD36','SEMA3C'],
        ['CD37','MAP3K13'],
        ['CD38','PTGR1'],
        ['CD38','RNASE10'],
        ['CD3D','SIK3'],
        ['CD3EAP','CDHR3'],
        ['CD3EAP','FAM172A'],
        ['CD3EAP','LNPEP'],
        ['CD3EAP','PDS5B'],
        ['CD3EAP','SCGB3A1'],
        ['CD3G','FXYD6'],
        ['CD44','LDLRAD3'],
        ['CD44','RP11-698N11.2'],
        ['CD44','TRIM44'],
        ['CD46','SETDB1'],
        ['CD48','KIRREL'],
        ['CD48','PARD6B'],
        ['CD52','CELSR1'],
        ['CD53','HLCS'],
        ['CD55','PLD5'],
        ['CD59','FBXO3'],
        ['CD59','HIPK3'],
        ['CD59','JUP'],
        ['CD59','KRT17'],
        ['CD69','SLC39A11'],
        ['CD70','KHSRP'],
        ['CD74','KRT14'],
        ['CD74','KRT5'],
        ['CD74','ROS1'],
        ['CD79B','MAP2K4'],
        ['CD83','LAMA2'],
        ['CD86','KPNA1'],
        ['CD86','RAD18'],
        ['CD8B','USP39'],
        ['CD9','IQCE'],
        ['CD96','PHLDB2'],
        ['CD96','PLCXD2'],
        ['CD96','ZBTB20'],
        ['CD99B','CXORF66'],
        ['CD99L2','CXORF66'],
        ['CDADC1','EFTUD1'],
        ['CDADC1','VPS13B'],
        ['CDC14A','TXLNB'],
        ['CDC16','UPF3A'],
        ['CDC20B','DHX29'],
        ['CDC25A','MAP4'],
        ['CDC25A','PRKAR2A'],
        ['CDC25A','SPINK8'],
        ['CDC25C','PAIP2'],
        ['CDC27','ST7L'],
        ['CDC27','USP32'],
        ['CDC34','MIER2'],
        ['CDC37','ELL'],
        ['CDC37','TYK2'],
        ['CDC42BPA','EPRS'],
        ['CDC42BPA','PRKG1'],
        ['CDC42BPA','WWOX'],
        ['CDC42BPB','RGS6'],
        ['CDC42BPB','SMEK1'],
        ['CDC42BPB','SYNJ2BP'],
        ['CDC42BPB','TRAF3'],
        ['CDC42BPB','UNC79'],
        ['CDC42BPB','ZFYVE1'],
        ['CDC42BPG','MAPKAPK2'],
        ['CDC42EP4','NEUROD2'],
        ['CDC42EP4','SDK2'],
        ['CDC42EP4','SEPT10'],
        ['CDC42EP4','VMP1'],
        ['CDC42EP4','WIPF2'],
        ['CDC42EP5','CYTH1'],
        ['CDC42SE1','PIP5K1A'],
        ['CDC42SE1','TNKS'],
        ['CDC45','LIMK2'],
        ['CDC5L','PITPNC1'],
        ['CDC5L','SPATS1'],
        ['CDC5L','TMEM63B'],
        ['CDC6','CDK12'],
        ['CDC6','RARA'],
        ['CDC6','RNFT1'],
        ['CDC73','FAM172A'],
        ['CDH1','CDKL3'],
        ['CDH1','DEXI'],
        ['CDH1','NPFFR2'],
        ['CDH1','TANGO6'],
        ['CDH10','PI4KA'],
        ['CDH11','WDR59'],
        ['CDH12','NUDT3'],
        ['CDH13','HSBP1'],
        ['CDH13','IPO8'],
        ['CDH15','PPP2CA'],
        ['CDH15','SF3B3'],
        ['CDH15','TNC'],
        ['CDH17','FGGY'],
        ['CDH19','DEPDC5'],
        ['CDH19','NEDD4L'],
        ['CDH19','SERPINB5'],
        ['CDH2','KIAA1328'],
        ['CDH2','LAMA3'],
        ['CDH2','LAMA4'],
        ['CDH23','CHST3'],
        ['CDH23','LTBP1'],
        ['CDH23','NSD1'],
        ['CDH23','SSH2'],
        ['CDH26','FAT3'],
        ['CDH26','TMEM25'],
        ['CDH3','CNN2'],
        ['CDH3','PPP2CA'],
        ['CDH3','TNC'],
        ['CDH4','NCOA5'],
        ['CDH4','NLRP9'],
        ['CDH4','OSBPL2'],
        ['CDH4','SLA2'],
        ['CDH4','SS18L1'],
        ['CDH4','TAF4'],
        ['CDH4','TOP1'],
        ['CDH4','TRIO'],
        ['CDH4','ZFC3H1'],
        ['CDH5','MMP15'],
        ['CDH6','NUP133'],
        ['CDH8','PRKCA'],
        ['CDH8','VAC14'],
        ['CDHR2','CLPTM1L'],
        ['CDHR2','WDYHV1'],
        ['CDHR3','ERC2'],
        ['CDIP1','HMOX2'],
        ['CDK1','MCOLN2'],
        ['CDK1','WDR11'],
        ['CDK10','SPATA33'],
        ['CDK11A','GNB1'],
        ['CDK12','CLCN4'],
        ['CDK12','ERBB2'],
        ['CDK12','FBXL20'],
        ['CDK12','JAK2'],
        ['CDK12','MBD4'],
        ['CDK12','MED1'],
        ['CDK12','MIEN1'],
        ['CDK12','PDK2'],
        ['CDK12','RARA'],
        ['CDK12','SMG6'],
        ['CDK12','TUBD1'],
        ['CDK13','DIP2C'],
        ['CDK13','PDE1C'],
        ['CDK13','POU6F2'],
        ['CDK13','TAX1BP1'],
        ['CDK13','YAE1D1'],
        ['CDK14','MKLN1'],
        ['CDK14','TFEC'],
        ['CDK17','ELK3'],
        ['CDK19','LYZ'],
        ['CDK19','PRDM1'],
        ['CDK19','SYNCRIP'],
        ['CDK19','SYNJ2'],
        ['CDK19','ZFAND3'],
        ['CDK2','PAN2'],
        ['CDK4','SUDS3'],
        ['CDK5RAP1','MIER2'],
        ['CDK5RAP2','ENG'],
        ['CDK5RAP2','GPRIN1'],
        ['CDK5RAP2','GRIN1'],
        ['CDK5RAP3','NF1'],
        ['CDK5RAP3','RNF43'],
        ['CDK6','FAM133B'],
        ['CDK6','FN1'],
        ['CDK6','GATAD1'],
        ['CDK6','HEPACAM2'],
        ['CDK6','METTL12'],
        ['CDK6','PIM3'],
        ['CDK6','PPM1H'],
        ['CDK6','RPSA'],
        ['CDK6','STXBP6'],
        ['CDK6','TAF1D'],
        ['CDK6','TCP1'],
        ['CDK7','RGS20'],
        ['CDK8','GTF2F2'],
        ['CDK8','IKZF3'],
        ['CDK8','UVRAG'],
        ['CDKAL1','DPY19L1'],
        ['CDKAL1','GRIK2'],
        ['CDKAL1','GRIK5'],
        ['CDKAL1','HERC4'],
        ['CDKAL1','HSP90AA1'],
        ['CDKAL1','RCAN2'],
        ['CDKAL1','TBC1D1'],
        ['CDKL1','FOXN3'],
        ['CDKL1','NEMF'],
        ['CDKL1','TSEN2'],
        ['CDKL2','COMMD9'],
        ['CDKL2','CXCL11'],
        ['CDKL2','G3BP2'],
        ['CDKL3','FZR1'],
        ['CDKN2A','CDKN2B-AS1'],
        ['CDKN2A','FOCAD'],
        ['CDKN2A','MGA'],
        ['CDKN2A','MGAM'],
        ['CDKN2A','MTAP'],
        ['CDKN2A','NPHP4'],
        ['CDKN2A','SRGAP2'],
        ['CDKN2A','SRGAP3'],
        ['CDKN2B','DGKD'],
        ['CDKN2B-AS1','MTAP'],
        ['CDKN3','GRN'],
        ['CDKN3','SFTA3'],
        ['CDPF1','GRHL2'],
        ['CDPF1','ZMIZ1'],
        ['CDR2','RRN3P3'],
        ['CDR2L','COIL'],
        ['CDR2L','KSR1'],
        ['CDS2','FKBP1A'],
        ['CDS2','RNF24'],
        ['CDS2','SLX4IP'],
        ['CDSN','PSORS1C1'],
        ['CDV3','TMEM108'],
        ['CDYL2','CMIP'],
        ['CEA','TSHZ3'],
        ['CEA','ZFP30'],
        ['CEACAM16','CLPTM1'],
        ['CEACAM19','EMC6'],
        ['CEACAM19','ZNF277'],
        ['CEACAM7','TSHZ3'],
        ['CEACAM7','ZFP30'],
        ['CEBPD','KRT6B'],
        ['CEBPG','TDRD12'],
        ['CEBPZ','EIF2AK2'],
        ['CELA2A','PER3'],
        ['CELF1','LUZP2'],
        ['CELF1','MTCH2'],
        ['CELF1','MYOM1'],
        ['CELF2','SIRT1'],
        ['CELF5','NFIC'],
        ['CELF5','ZMYND8'],
        ['CELF6','MYO9A'],
        ['CELSR1','CERK'],
        ['CELSR1','MAFF'],
        ['CELSR1','MAPK8IP2'],
        ['CELSR1','PACSIN2'],
        ['CELSR1','SEZ6L'],
        ['CELSR2','SARS'],
        ['CELSR2','SARS2'],
        ['CENPE','SPATA5'],
        ['CENPF','SUSD4'],
        ['CENPL','PRRC2C'],
        ['CENPO','UBXN2A'],
        ['CENPP','CRHR1'],
        ['CENPQ','MUT'],
        ['CENPV','FERMT1'],
        ['CEP104','TP73'],
        ['CEP112','EVA1C'],
        ['CEP112','GNA13'],
        ['CEP112','HELZ'],
        ['CEP112','KIAA1001'],
        ['CEP112','MED22'],
        ['CEP112','MED24'],
        ['CEP112','OLFM2'],
        ['CEP112','OTOA'],
        ['CEP112','PSMD11'],
        ['CEP170','PRKACB'],
        ['CEP170','RGS7'],
        ['CEP170','SDCCAG8'],
        ['CEP170B','ITPK1'],
        ['CEP192','GREB1L'],
        ['CEP250','TM9SF4'],
        ['CEP290','PRKDC'],
        ['CEP350','TDRD5'],
        ['CEP57L1','SRPK1'],
        ['CEP70','PMEPA1'],
        ['CEP70','STAG1'],
        ['CEP72','TPPP'],
        ['CEP76','RUNX1'],
        ['CEP85','FAM110D'],
        ['CEP85L','OSTM1'],
        ['CEP85L','ROS1'],
        ['CEP85L','SYTL3'],
        ['CEP89','KLHL24'],
        ['CEP95','MED22'],
        ['CEP95','MED24'],
        ['CEP95','RBM11'],
        ['CERS3','LTBP2'],
        ['CERS3','LTBP3'],
        ['CERS3','SEMA4B'],
        ['CERS4','HNRNPM'],
        ['CERS4','MARCH2'],
        ['CERS4','SIRT6'],
        ['CERS4','SPAG17'],
        ['CERS4','SPEN'],
        ['CERS5','RACGAP1'],
        ['CERS5','SAMD11'],
        ['CES1','CTCF'],
        ['CES2','CTCF'],
        ['CETN3','SKIV2L2'],
        ['CETP','CNOT1'],
        ['CFB','D6S204'],
        ['CFB','ERBB2'],
        ['CFB','HLA-C'],
        ['CFB','HSP90B1'],
        ['CFB','RPL18'],
        ['CFB','RPL8'],
        ['CFB','TSPAN3'],
        ['CFDP1','GLG1'],
        ['CFI','SEC24D'],
        ['CFTR','ELP4'],
        ['CGA','ZNF292'],
        ['CGGBP1','MITF'],
        ['CGGBP1','SCN11A'],
        ['CGI-143','HSPC099'],
        ['CGI-143','RPRD2'],
        ['CGI-78','SLAMF1'],
        ['CGN','TUFT1'],
        ['CGNL1','TCF12'],
        ['CGRRF1','DTD2'],
        ['CHAF1A','KDM4B'],
        ['CHAF1B','RNF169'],
        ['CHAMP1','ZNF589'],
        ['CHCHD6','OSBPL11'],
        ['CHCHD6','PLXNA1'],
        ['CHD1','FUS'],
        ['CHD1','GPM6B'],
        ['CHD1','MTOR'],
        ['CHD1','S100P'],
        ['CHD2','HOMER2'],
        ['CHD3','TP53'],
        ['CHD4','CLEC9A'],
        ['CHD4','DTX1'],
        ['CHD4','GPX3'],
        ['CHD6','CNBD2'],
        ['CHD6','EFCAB11'],
        ['CHD6','FAT3'],
        ['CHD6','LRP5'],
        ['CHD6','MCUR1'],
        ['CHD6','STK35'],
        ['CHD6','TANC2'],
        ['CHD6','WFDC11'],
        ['CHD7','FAM222B'],
        ['CHD7','FOXO3'],
        ['CHD7','IGF1R'],
        ['CHD7','TACC1'],
        ['CHD7','TOX'],
        ['CHD7','WHSC1L1'],
        ['CHD8','IMPG1'],
        ['CHD8','KIAA1211'],
        ['CHD8','RAB2B'],
        ['CHD8','TOX4'],
        ['CHD9','DLC1'],
        ['CHD9','DLEC1'],
        ['CHD9','DYNLL1'],
        ['CHD9','KSR1'],
        ['CHD9','RAD51B'],
        ['CHD9','SPG7'],
        ['CHDC2','MAP3K8'],
        ['CHEK1','EXOC6'],
        ['CHEK1','NTM'],
        ['CHEK1','TMEM135'],
        ['CHERP','CPAMD8'],
        ['CHFR','OSBPL8'],
        ['CHGA','MAPKAPK2'],
        ['CHI3L2','MUCL1'],
        ['CHIC2','MANBA'],
        ['CHID1','RRBP1'],
        ['CHKA','GDPD5'],
        ['CHKA','HNF1B'],
        ['CHKA','PC'],
        ['CHKA','PODXL'],
        ['CHKA','PTK2'],
        ['CHL1','VGLL4'],
        ['CHM','KLHL4'],
        ['CHMP1A','CPSF6'],
        ['CHMP1A','RBM27'],
        ['CHMP1A','ZC3H18'],
        ['CHMP3','PEX13'],
        ['CHMP5','GALT'],
        ['CHMP5','VMP1'],
        ['CHN1','TSGA10'],
        ['CHODL','KIAA1109'],
        ['CHPT1','MYBPC1'],
        ['CHPT1','PTPRR'],
        ['CHRAC1','SNTG1'],
        ['CHRAC1','TRPS1'],
        ['CHRDL2','ERP44'],
        ['CHRM1','SYTL2'],
        ['CHRM3','FMN2'],
        ['CHRNA6','POMK'],
        ['CHRNB3','HEATR3'],
        ['CHRND','GIGYF2'],
        ['CHST1','PARD6G'],
        ['CHST11','FBXW8'],
        ['CHST11','HCFC2'],
        ['CHST11','RFX4'],
        ['CHST4','SNTB2'],
        ['CHST4','SNTL'],
        ['CHST8','KCNJ15'],
        ['CHST9','NDUFV2'],
        ['CHST9','PAPPA-AS1'],
        ['CHST9','TAF4B'],
        ['CHST9','ZNF675'],
        ['CHSY1','SPOCK2'],
        ['CHSY1','VRK2'],
        ['CHSY3','NINJ2'],
        ['CHTOP','FLJ21919'],
        ['CHTOP','INTS3'],
        ['CIB2','DNAJC17'],
        ['CIB2','PEAK1'],
        ['CIC','KLK11'],
        ['CIC','LMNA'],
        ['CIC','ZFP57'],
        ['CIC','ZNF698'],
        ['CINP','MOK'],
        ['CINP','RAGE'],
        ['CIRH1A','DDX19A'],
        ['CIRH1A','SNTL'],
        ['CISD1','GABRA1'],
        ['CISD2','KCNJ10'],
        ['CISD3','ELL'],
        ['CISD3','MLLT3'],
        ['CIT','FAM222A'],
        ['CIZ1','SLC25A25'],
        ['CKAP2','UBL3'],
        ['CKAP4','FBRSL1'],
        ['CKAP5','TMEM135'],
        ['CKAP5','VMP1'],
        ['CKB','SCN2A'],
        ['CKM','PRKD2'],
        ['CKM','SIAE'],
        ['CKS2','PCSK5'],
        ['CLASP1','FRMD8'],
        ['CLASP1','HOOK2'],
        ['CLASP1','TFCP2L1'],
        ['CLASRP','RELB'],
        ['CLCN2','SENP2'],
        ['CLCN3','GARS'],
        ['CLCN3','SMAD1'],
        ['CLCN3','UBE2D3'],
        ['CLCN4','PRKX'],
        ['CLCN5','HUWE1'],
        ['CLCN6','RAF1'],
        ['CLDN10','FLJ20535'],
        ['CLDN10','TTC12'],
        ['CLDN11','CMTM7'],
        ['CLDN14','MAST2'],
        ['CLDN18','PLS1'],
        ['CLDN6','PKMYT1'],
        ['CLDN7','SLC34A2'],
        ['CLEC16A','LITAF'],
        ['CLEC16A','SNX29'],
        ['CLEC16A','TXNDC11'],
        ['CLEC17A','DOK5'],
        ['CLEC2A','PAWR'],
        ['CLGN','MAML3'],
        ['CLIC1','HIST1H2BC'],
        ['CLIC2','VPS39'],
        ['CLIC4','GRHL3'],
        ['CLIC4','PLOD1'],
        ['CLIC5','TRIO'],
        ['CLIC6','FOXJ3'],
        ['CLIC6','ITSN1'],
        ['CLINT1','HARS'],
        ['CLINT1','MARCH11'],
        ['CLIP-115','KMT2C'],
        ['CLIP1','SLC8B1'],
        ['CLIP1','TNIK'],
        ['CLIP2','KMT2C'],
        ['CLIP4','FAM179A'],
        ['CLK2','MARS'],
        ['CLK2','PPP1R12B'],
        ['CLK2','RBM19'],
        ['CLMN','EVL'],
        ['CLMP','TMEM135'],
        ['CLN6','FBXL17'],
        ['CLN6','PAQR5'],
        ['CLNK','TRIO'],
        ['CLNS1A','EPS15L1'],
        ['CLOCK','NMU'],
        ['CLOCK','PDGFRA'],
        ['CLOCK','PHF21A'],
        ['CLOCK','SRD5A3'],
        ['CLOCK','UGT2A3'],
        ['CLPB','LRP5'],
        ['CLPB','MTUS2'],
        ['CLPB','STARD10'],
        ['CLPTM1','CYP2A7'],
        ['CLPTM1','IGSF23'],
        ['CLPTM1L','PDCD6'],
        ['CLSPN','NIPBL'],
        ['CLSTN1','PLEKHG5'],
        ['CLSTN2','GPR126'],
        ['CLSTN2','LPP'],
        ['CLSTN2','RORA'],
        ['CLSTN3','PEX5'],
        ['CLSTN3','PLCZ1'],
        ['CLTA','UBE2R2'],
        ['CLTB','KRT5'],
        ['CLTB','SIL1'],
        ['CLTC','FAM129C'],
        ['CLTC','NF1'],
        ['CLTC','OTOP3'],
        ['CLTC','PTRH2'],
        ['CLTC','ROS1'],
        ['CLTC','TEX14'],
        ['CLTC','USP32'],
        ['CLTC','VMP1'],
        ['CLTCL','DGCR2'],
        ['CLTCL','NF2'],
        ['CLTCL','STARD10'],
        ['CLTCL1','DGCR2'],
        ['CLTCL1','NF2'],
        ['CLTCL1','STARD10'],
        ['CLUAP1','NPTN'],
        ['CLUH','DLG5'],
        ['CLUL1','COLEC12'],
        ['CLUL1','MYO5B'],
        ['CLUL1','TYMS'],
        ['CLVS1','TRAM1'],
        ['CLVS2','MAP3K5'],
        ['CLYBL','PCCA'],
        ['CLYBL','SPAG9'],
        ['CMAHP','CREBL1'],
        ['CMAHP','LRRC16A'],
        ['CMAHP','TNXB'],
        ['CMAHP','TNXBS'],
        ['CMAHP','ZBTB24'],
        ['CMAS','GPATCH9'],
        ['CMAS','RBM10'],
        ['CMBL','DARS2'],
        ['CMC1','GLB1'],
        ['CMC1','VPS37C'],
        ['CMC2','PZP'],
        ['CMC4','CXADR'],
        ['CMC4','STAG2'],
        ['CMPK1','MORN1'],
        ['CMPK1','SH2D1B'],
        ['CMSS1','TBC1D23'],
        ['CMSS1','TMEM45A'],
        ['CMSS1','XRRA1'],
        ['CMSS1','ZBTB20'],
        ['CMSS1','ZNF844'],
        ['CMTM7','CMTM8'],
        ['CMTM7','KIAA2026'],
        ['CMTM8','NR1D2'],
        ['CMTM8','RAF1'],
        ['CMTR1','GPX6'],
        ['CMTR1','GPX7'],
        ['CMYA5','MIIP'],
        ['CNBD1','DCAF12'],
        ['CNBD1','PTPN2'],
        ['CNBD2','RALGAPB'],
        ['CNBD2','USP16'],
        ['CNBD2','USP21'],
        ['CNBD2','ZGPAT'],
        ['CNBD2','ZNFX1'],
        ['CNGB1','IL34'],
        ['CNGB1','KIFC3'],
        ['CNIH2','MS4A4A'],
        ['CNIH3','RB1'],
        ['CNIH4','ERBB2'],
        ['CNIH4','PAK1'],
        ['CNIH4','PKN1'],
        ['CNKSR1','MAN1C1'],
        ['CNKSR2','KDM2B'],
        ['CNKSR2','MBTPS2'],
        ['CNKSR2','P2RX4'],
        ['CNKSR3','DPH3'],
        ['CNKSR3','GAK'],
        ['CNKSR3','MAP7'],
        ['CNN1','FZR1'],
        ['CNN2','PPP1R12C'],
        ['CNNM2','INT3'],
        ['CNNM2','NOTCH4'],
        ['CNNM2','SUFU'],
        ['CNOT1','RASSF8'],
        ['CNOT2','NUP107'],
        ['CNOT2','PDE3A'],
        ['CNOT2','PTK7'],
        ['CNOT2','RAB3IP'],
        ['CNOT2','SLC45A4'],
        ['CNOT4','DTNB'],
        ['CNOT6','H2AFY'],
        ['CNP','NCALD'],
        ['CNPY2','STAT2'],
        ['CNPY3','CXCR5'],
        ['CNPY3','THPO'],
        ['CNPY3','TPO'],
        ['CNST','RGS7'],
        ['CNST','SCCPDH'],
        ['CNTFR','KIF24'],
        ['CNTLN','RFX3'],
        ['CNTLN','SNX30'],
        ['CNTLN','SNX33'],
        ['CNTN1','CPNE8'],
        ['CNTN1','TSPAN8'],
        ['CNTN2','DYRK2'],
        ['CNTN5','HS3ST1'],
        ['CNTN5','MAML2'],
        ['CNTN5','NAV2'],
        ['CNTN5','SART1'],
        ['CNTN5','TMEM123'],
        ['CNTN5','TTYH1'],
        ['CNTN6','CYB5R4'],
        ['CNTN6','EPHA6'],
        ['CNTNAP2','ESYT2'],
        ['CNTNAP2','GRM8'],
        ['CNTNAP2','KMT2C'],
        ['CNTNAP2','MLL3'],
        ['CNTNAP2','SEMA3C'],
        ['COA1','FAM60A'],
        ['COA1','HECW1'],
        ['COA1','MRPS24'],
        ['COBL','DUSP14'],
        ['COBL','SEPT14'],
        ['COBLL1','SCN3A'],
        ['COBLL1','SLC38A11'],
        ['COBLL1','SMYD3'],
        ['COG2','PCNXL2'],
        ['COG4','GLG1'],
        ['COG4','SF3B3'],
        ['COG4','ST3GAL2'],
        ['COG4','TERF2'],
        ['COG5','DNAH7'],
        ['COG5','DPP6'],
        ['COG5','FOXP2'],
        ['COG5','TFDP2'],
        ['COG6','LHFP'],
        ['COG6','MAPK8'],
        ['COL10A1','HIPK1'],
        ['COL10A1','RARA'],
        ['COL14A1','DEPTOR'],
        ['COL15A1','CTDSP2'],
        ['COL15A1','TGFBR1'],
        ['COL17A1','KRT16'],
        ['COL17A1','KRT5'],
        ['COL17A1','KRT6A'],
        ['COL17A1','NEURL'],
        ['COL17A1','TNC'],
        ['COL18A1','HSPA5'],
        ['COL1A1','COL1A2'],
        ['COL1A1','COL3A1'],
        ['COL1A1','COL5A1'],
        ['COL1A1','COL5A2'],
        ['COL1A1','COL6A2'],
        ['COL1A1','ETS1'],
        ['COL1A1','JUP'],
        ['COL1A1','KRT14'],
        ['COL1A1','KRT16'],
        ['COL1A1','KRT17'],
        ['COL1A1','KRT5'],
        ['COL1A1','KRT6A'],
        ['COL1A1','PEAK1'],
        ['COL1A1','PIKFYVE'],
        ['COL1A1','SPARC'],
        ['COL1A1','SPRR3'],
        ['COL1A2','COL3A1'],
        ['COL1A2','KRT13'],
        ['COL1A2','KRT16'],
        ['COL1A2','SFN'],
        ['COL21A1','EFCAB1'],
        ['COL22A1','RAD54B'],
        ['COL22A1','SNTB1'],
        ['COL22A1','ST3GAL1'],
        ['COL22A1','TRAPPC9'],
        ['COL22A1','TSNARE1'],
        ['COL23A1','FAM193B'],
        ['COL23A1','MED13L'],
        ['COL24A1','PKN2'],
        ['COL24A1','TRIM33'],
        ['COL25A1','PPP3CA'],
        ['COL26A1','YES1'],
        ['COL27A1','PDZD2'],
        ['COL27A1','SCNN1G'],
        ['COL3A1','COL6A2'],
        ['COL3A1','CTSB'],
        ['COL3A1','FN1'],
        ['COL3A1','JUP'],
        ['COL3A1','KRT13'],
        ['COL3A1','KRT16'],
        ['COL3A1','KRT17'],
        ['COL3A1','KRT5'],
        ['COL3A1','RSPO1'],
        ['COL4A1','ELK3'],
        ['COL4A1','KCNH8'],
        ['COL4A1','UGGT2'],
        ['COL4A1','ZNF827'],
        ['COL4A2','EEA1'],
        ['COL4A2','ELK3'],
        ['COL4A2','GRTP1'],
        ['COL4A2','KCNH8'],
        ['COL4A3','EPHA4'],
        ['COL4A3','MFF'],
        ['COL4A3BP','ELOVL7'],
        ['COL4A5','RGAG1'],
        ['COL4A5','XRCC4'],
        ['COL5A1','PTGES2'],
        ['COL5A3','OLFM2'],
        ['COL6A1','S100A6'],
        ['COL6A2','DDX27'],
        ['COL6A2','ELK3'],
        ['COL6A2','KCNH8'],
        ['COL6A2','PCNT'],
        ['COL7A1','PFKFB4'],
        ['COL8A1','GSK3B'],
        ['COL8A2','EPHA10'],
        ['COL9A2','HEYL'],
        ['COL9A3','ELK3'],
        ['COL9A3','KCNH8'],
        ['COLEC12','ZNF200'],
        ['COLGALT2','NOS1AP'],
        ['COLQ','NR2C2'],
        ['COMMD1','EHBP1'],
        ['COMMD1','MDH1'],
        ['COMMD10','DGKI'],
        ['COMMD10','NREP'],
        ['COMMD6','TBC1D4'],
        ['COMT','MED15'],
        ['COPA','KIFAP3'],
        ['COPB2','PIK3CB'],
        ['COPE','TMEM59L'],
        ['COPG2','PBX1'],
        ['COPS2','SULT1B1'],
        ['COPS3','TEKT3'],
        ['COPS5','PDE7A'],
        ['COPS5','TPD52'],
        ['COQ10A','NABP2'],
        ['COQ10A','USP5'],
        ['COQ4','DNM1'],
        ['COQ5','DYNLL1'],
        ['COQ6','MAST2'],
        ['CORIN','OSBPL3'],
        ['CORO1C','SART3'],
        ['CORO1C','TBC1D9'],
        ['CORO1C','ZNF664'],
        ['CORO6','SLC20A2'],
        ['CORO7','ZNF500'],
        ['COX20','SDCCAG8'],
        ['COX4I2','ERBB3'],
        ['COX6C','DDX27'],
        ['COX6C','RGS22'],
        ['COX7A2L','EML4'],
        ['CP','SOX5'],
        ['CP','WWTR1'],
        ['CPA3','TFRC'],
        ['CPA4','TFRC'],
        ['CPA6','KIAA1429'],
        ['CPAMD8','HAUS8'],
        ['CPB1','HPS3'],
        ['CPB2','WWP1'],
        ['CPD','ERBB2'],
        ['CPD','ORMDL3'],
        ['CPD','SKAP1'],
        ['CPD','TAOK1'],
        ['CPD','TBX2'],
        ['CPE','PLEKHA5'],
        ['CPEB4','STK32A'],
        ['CPED1','ESYT2'],
        ['CPLX2','MAP3K1'],
        ['CPM','GOLT1B'],
        ['CPM','KCNMA1'],
        ['CPM','PDZRN4'],
        ['CPM','PRKAA2'],
        ['CPM','RAP1B'],
        ['CPNE1','DMD'],
        ['CPNE1','RBL1'],
        ['CPNE4','PZP'],
        ['CPNE4','STK36'],
        ['CPNE8','LCOR'],
        ['CPSF3','HPCAL1'],
        ['CPSF4','RP11-762I7.5'],
        ['CPSF4','SARNP'],
        ['CPSF4L','ERBB4'],
        ['CPSF6','IRF2BP2'],
        ['CPSF6','JAG2'],
        ['CPSF6','MDM1'],
        ['CPSF6','NAV3'],
        ['CPSF6','OSBPL8'],
        ['CPSF6','PTPRB'],
        ['CPSF6','SH3YL1'],
        ['CPT1A','FCHSD2'],
        ['CPT1A','FGF3'],
        ['CPT1A','INT2'],
        ['CPT1A','KCNG1'],
        ['CPT1A','PROSAP1'],
        ['CPT1A','SHANK2'],
        ['CPT1A','SUV420H1'],
        ['CPT1B','SEC14L3'],
        ['CPT1C','JAZF1'],
        ['CPT2','CTDSP2'],
        ['CPT2','SCP2'],
        ['CPT2','SYCP2'],
        ['CPVL','GDI2'],
        ['CR1L','MTOR'],
        ['CR2','DUSP12'],
        ['CRABP1','PDE4D'],
        ['CRADD','EEA1'],
        ['CRAMP1L','MYO9A'],
        ['CRAMP1L','TRIM9'],
        ['CRB1','IGFN1'],
        ['CRB1','NEK7'],
        ['CRB3','TANC2'],
        ['CRBN','GRM7'],
        ['CREAM1','KMT2C'],
        ['CREAM1','MED13L'],
        ['CREB1','FASTKD2'],
        ['CREB3','TLN1'],
        ['CREB3L3','SHC2'],
        ['CREB3L4','HIVEP3'],
        ['CREB3L4','KDM4A'],
        ['CREB5','KALRN'],
        ['CREB5','KCNH2'],
        ['CREB5','SHISA9'],
        ['CREB5','USP42'],
        ['CREBBP','HCN1'],
        ['CREBBP','KAT6A'],
        ['CREBBP','LUC7L'],
        ['CREBBP','NUBP1'],
        ['CREBBP','ZNF500'],
        ['CREBL2','HEBP1'],
        ['CREBL2','PLBD1'],
        ['CREBRF','SLIT1'],
        ['CREBRF','SLIT3'],
        ['CREG2','TLK1'],
        ['CRELD2','PIM3'],
        ['CRIM1','EEF1B2'],
        ['CRIM1','PPIG'],
        ['CRIM1','TACR1'],
        ['CRISP1','TAF8'],
        ['CRISP1','TTC39A'],
        ['CRISPLD2','MPZL2'],
        ['CRISPLD2','UBL3'],
        ['CRKL','DGCR8'],
        ['CRLF1','ELL'],
        ['CRLS1','NFX1'],
        ['CRLS1','SEL1L2'],
        ['CRNN','KRT13'],
        ['CRTAC1','MMS19'],
        ['CRTAC1','RRP12'],
        ['CRTC1','KLHL26'],
        ['CRTC1','LYPD4'],
        ['CRTC2','GATAD2B'],
        ['CRTC2','KIAA0907'],
        ['CRTC2','P66BETA'],
        ['CRTC3','IQGAP1'],
        ['CRYA2','FDXACB1'],
        ['CRYAB','FDXACB1'],
        ['CRYBA1','PRKCA'],
        ['CRYBA4','USP32'],
        ['CRYBG3','IL20RB'],
        ['CRYGS','DGKG'],
        ['CRYL1','PSPC1'],
        ['CRYL1','SPACA7'],
        ['CRYZ','SPATA6'],
        ['CS','MUCL1'],
        ['CS','PRIM1'],
        ['CSAD','ITGB7'],
        ['CSDE1','CYB5B'],
        ['CSDE1','DDX27'],
        ['CSDE1','RBFOX1'],
        ['CSDE1','TRIM33'],
        ['CSF1R','IGF2R'],
        ['CSF1R','PGC'],
        ['CSGALNACT1','MSRA'],
        ['CSGALNACT2','SLC16A9'],
        ['CSMD1','MCPH1'],
        ['CSMD1','PRKDC'],
        ['CSMD1','WHSC1L1'],
        ['CSMD2','FYB'],
        ['CSMD2','IFI44L'],
        ['CSMD2','PHC2'],
        ['CSMD2','TMEM222'],
        ['CSMD3','OXR1'],
        ['CSMD3','RIMS2'],
        ['CSN2','SULT1B1'],
        ['CSNB2A','LMO6'],
        ['CSNB2A','PRICKLE3'],
        ['CSNK1A1','PDE6A'],
        ['CSNK1A1','SCGB3A2'],
        ['CSNK1D','PGC'],
        ['CSNK1D','SCPEP1'],
        ['CSNK1D','SECTM1'],
        ['CSNK1E','DMC1'],
        ['CSNK1E','DNAL4'],
        ['CSNK1E','GNAL'],
        ['CSNK1E','KDELR3'],
        ['CSNK1E','PHF5A'],
        ['CSNK1G1','DPYD'],
        ['CSNK1G1','HERC1'],
        ['CSNK1G1','INTS2'],
        ['CSNK1G1','MBOAT2'],
        ['CSNK1G1','ZNF609'],
        ['CSNK1G2','GANAB'],
        ['CSNK1G2','HSD17B13'],
        ['CSNK1G2','PDE5A'],
        ['CSNK2A1','SNRPB'],
        ['CSPP1','MAN1A2'],
        ['CSPP1','PPP1R42'],
        ['CSTA','INPP5A'],
        ['CSTB','JUP'],
        ['CSTB','KRT13'],
        ['CSTB','KRT14'],
        ['CSTB','KRT17'],
        ['CSTB','KRT5'],
        ['CSTB','KRT6A'],
        ['CSTB','KRT6B'],
        ['CSTB','SPRR1A'],
        ['CSTB','SPRR3'],
        ['CSTF3','GRB10'],
        ['CSTF3','JAK2'],
        ['CT16.7','PSD3'],
        ['CT37','NCOR2'],
        ['CTAGE5','MIA2'],
        ['CTBP1','KIAA1530'],
        ['CTBP1','MAEA'],
        ['CTC-236F12.4','SERINC5'],
        ['CTCF','HMSE1'],
        ['CTD-2006C1.2','PKN1'],
        ['CTD-2547L24.3','GAS2L1'],
        ['CTD-2589C9.4','FOXR1'],
        ['CTDP1','NFATC1'],
        ['CTDP1','PQLC1'],
        ['CTDSP2','DCUN1D4'],
        ['CTDSP2','ECHDC2'],
        ['CTDSP2','MDM2'],
        ['CTDSP2','RAP1B'],
        ['CTDSP2','RTN4IP1'],
        ['CTDSP2','TSPAN31'],
        ['CTDSPL','OXSR1'],
        ['CTDSPL','TMCC1'],
        ['CTDSPL','WRN'],
        ['CTIF','DYM'],
        ['CTIF','GRM4'],
        ['CTIF','GTF2F2'],
        ['CTIF','PSTPIP2'],
        ['CTIF','SMAD2'],
        ['CTIF','WASF2'],
        ['CTNNA1','MATR3'],
        ['CTNNA1','SLC38A9'],
        ['CTNNA2','GLIS3'],
        ['CTNNA2','HIBCH'],
        ['CTNNA2','MRPL35'],
        ['CTNNA2','PIGN'],
        ['CTNNA3','HERC4'],
        ['CTNNA3','LOH12CR1'],
        ['CTNNA3','MGMT'],
        ['CTNNA3','PBX1'],
        ['CTNNB1','GSN'],
        ['CTNNB1','SLC38A5'],
        ['CTNNB1','SN2'],
        ['CTNNB1','THRB'],
        ['CTNNBL1','RALGAPB'],
        ['CTNNBL1','RBL1'],
        ['CTNNBL1','RPRD1B'],
        ['CTNNBL1','TTI1'],
        ['CTNND1','DDAH2'],
        ['CTNND1','DOCK8'],
        ['CTNND1','PACS1'],
        ['CTNND1','TMX2'],
        ['CTNND1','ZDHHC5'],
        ['CTNND1','ZNFX1'],
        ['CTNND2','FAS'],
        ['CTNND2','FASN'],
        ['CTNND2','LMBRD2'],
        ['CTNND2','MTF2'],
        ['CTNND2','NRBP1'],
        ['CTNND2','P4HA2'],
        ['CTNND2','PPARGC1B'],
        ['CTPS','NFYC'],
        ['CTPS2','HPRT1'],
        ['CTPS2','RBBP7'],
        ['CTSB','EIF2AK1'],
        ['CTSB','EIF4A2'],
        ['CTSB','FAR1'],
        ['CTSB','KRT6A'],
        ['CTSB','LYZ'],
        ['CTSB','PXK'],
        ['CTSB','RP11-1143G9.4'],
        ['CTSD','KRT14'],
        ['CTSD','S100A9'],
        ['CTSH','RASGRF1'],
        ['CTSS','MCL1'],
        ['CTSS','PRPF3'],
        ['CTSS','RP18'],
        ['CTSS','TDRD10'],
        ['CTSW','LRRC16A'],
        ['CTTN','DDX27'],
        ['CTTN','DSG1'],
        ['CTTN','ERBB2'],
        ['CTTN','FOLR1'],
        ['CTTN','PLEKHA5'],
        ['CTTN','PSME2'],
        ['CTTNBP2NL','RP11-315O6.2'],
        ['CTTNBP2NL','SUCLA2'],
        ['CTU1','ZNF549'],
        ['CTU2','RBM27'],
        ['CTU2','ZC3H18'],
        ['CTXN3','XXBAC-B461K10.4'],
        ['CUBN','TSHZ2'],
        ['CUEDC1','MRPL36'],
        ['CUEDC1','MRPS23'],
        ['CUL1','GIMAP5'],
        ['CUL1','MUT'],
        ['CUL1','ZNF282'],
        ['CUL2','NRP1'],
        ['CUL2','PARD3'],
        ['CUL3','IRS1'],
        ['CUL3','SCMH1'],
        ['CUL4A','GRK1'],
        ['CUL4A','KATNAL1'],
        ['CUL4A','TDRD3'],
        ['CUL4A','TFDP1'],
        ['CUL4B','DAPK2'],
        ['CUL4B','SEPT6'],
        ['CUX1','EHBP1L1'],
        ['CUX1','LRWD1'],
        ['CUX1','MAP2K1'],
        ['CUX1','MARCH10'],
        ['CUX1','ORAI2'],
        ['CUX1','RAD54L2'],
        ['CUX1','THSD7A'],
        ['CUZD1','TIAL1'],
        ['CWC25','FBXO47'],
        ['CWC25','MED22'],
        ['CWC25','MED24'],
        ['CWC25','PIP4K2B'],
        ['CWC27','HSA238520'],
        ['CWC27','PTPN18'],
        ['CWF19L1','HPSE2'],
        ['CWH43','SLAIN2'],
        ['CXADR','MTCP1'],
        ['CXCL12','ECD'],
        ['CXCL12','IPMK'],
        ['CXCL12','SHFM1'],
        ['CXCL16','TEKT1'],
        ['CXCR5','IGFBP5'],
        ['CXCR5','P4HB'],
        ['CXORF31','JADE3'],
        ['CXORF31','KIAA0215'],
        ['CXORF31','SACM1L'],
        ['CXORF31','TFE3'],
        ['CXORF53','VBP1'],
        ['CXORF58','POLA1'],
        ['CXORF66','MFSD6'],
        ['CXXC4','INTS12'],
        ['CYB561','GNA13'],
        ['CYB561','USP32'],
        ['CYB5A','DCC'],
        ['CYB5R1','NUAK2'],
        ['CYB5R2','SIX1'],
        ['CYB5R4','GLTSCR1L'],
        ['CYB5R4','MDN1'],
        ['CYBA','TLDC1'],
        ['CYCS','KRT14'],
        ['CYFIP1','TEX9'],
        ['CYFIP1','TUBGCP5'],
        ['CYGB','PTPRD'],
        ['CYHR1','KIAA0124'],
        ['CYP19A1','UBE3A'],
        ['CYP26A1','NRBF2'],
        ['CYP27B1','ZNF175'],
        ['CYP2A6','KIAA1429'],
        ['CYP2A6','PAK4'],
        ['CYP2A6','ZNF613'],
        ['CYP2C18','KRT14'],
        ['CYP2S1','VASP'],
        ['CYP39A1','GFOD1'],
        ['CYP39A1','LEMD2'],
        ['CYP46A1','TRIP11'],
        ['CYP4A11','RERE'],
        ['CYP4F12','DPM1'],
        ['CYP4F3','WIZ'],
        ['CYSTM1','NRG2'],
        ['CYTH1','KSR1'],
        ['CYTH1','MRC2'],
        ['CYTH3','GRID1'],
        ['CYTH3','MAD1L1'],
        ['CYTH3','PKN3'],
        ['CYTH3','SPATA18'],
        ['CYTH3','TP63'],
        ['CYTIP','LRPPRC'],
        ['CYTIP','MGAT5'],
        ['D6S204','FOXP1'],
        ['D6S204','TCOF1'],
        ['D6S52E','LY6G5C'],
        ['D6S52E','NG33'],
        ['D6S52E','TAF8'],
        ['D6S81E','PRPH2'],
        ['DAAM2','LRRC1'],
        ['DAAM2','USP49'],
        ['DAB1','FLJ59067'],
        ['DAB1','IL12RB2'],
        ['DAB1','USP24'],
        ['DAB1','ZNF732'],
        ['DAB2','PAIP1'],
        ['DAB2IP','FLJ14491'],
        ['DAB2IP','HTN3'],
        ['DAB2IP','SLC44A4'],
        ['DACH2','FGF13'],
        ['DACT1','RAD51B'],
        ['DACT1','RAD51L1'],
        ['DAG1','ZNF589'],
        ['DAGLA','PPP1R32'],
        ['DAK','TMEM109'],
        ['DAND5','FKBP8'],
        ['DAND5','NFIX'],
        ['DAND5','ZNF121'],
        ['DAND5','ZNF20'],
        ['DAP','LPCAT1'],
        ['DAP','SNX9'],
        ['DAP6','TAPBP'],
        ['DAPK1','KCTD2'],
        ['DAPK1','TGDS'],
        ['DAPK1','TRIM52'],
        ['DAPK2','HERC1'],
        ['DARS','MCM6'],
        ['DARS2','SUCO'],
        ['DARS2','UCK2'],
        ['DAXX','TAPBP'],
        ['DAZL','RFTN1'],
        ['DBF4B','DYNC2H1'],
        ['DBN1','GRK6'],
        ['DBNDD1','NAT8L'],
        ['DBNDD1','TPGS1'],
        ['DBNL','EEA1'],
        ['DBR1','SERPINA2P'],
        ['DCAF10','KIAA1432'],
        ['DCAF10','MCART1'],
        ['DCAF10','RP11-613M10.9'],
        ['DCAF12','UBAP2'],
        ['DCAF5','NOXRED1'],
        ['DCAF5','PLEKHA6'],
        ['DCAF5','RAB5B'],
        ['DCAF5','RAD51B'],
        ['DCAF6','FAF1'],
        ['DCAF6','MPZL1'],
        ['DCAF7','DDX42'],
        ['DCAF7','KDM6A'],
        ['DCAF7','VMP1'],
        ['DCAF8','NME7'],
        ['DCAF8','UHMK1'],
        ['DCAKD','MAP3K14'],
        ['DCAKD','PLCD3'],
        ['DCBLD1','PITRM1'],
        ['DCBLD1','PREP'],
        ['DCBLD2','LGALS3BP'],
        ['DCBLD2','ST3GAL6'],
        ['DCC','DLGAP1'],
        ['DCC','KIAA1328'],
        ['DCC','ROCK1'],
        ['DCDC2','PC'],
        ['DCDC2','PODXL'],
        ['DCHS2','FGG'],
        ['DCK','FRAS1'],
        ['DCK','GRSF1'],
        ['DCLK1','ELF1'],
        ['DCLK1','NFIA'],
        ['DCLK1','RCBTB1'],
        ['DCLK1','SERPINE3'],
        ['DCLK2','MACROD2'],
        ['DCLK2','UST'],
        ['DCLRE1C','DNAJC1'],
        ['DCLRE1C','NSMCE2'],
        ['DCP1B','SCNN1A'],
        ['DCST2','KCNN3'],
        ['DCST2','THEM4'],
        ['DCTD','ECE2'],
        ['DCTN2','DDX23'],
        ['DCTN2','PLEKHA5'],
        ['DCTN6','MEX3D'],
        ['DCUN1D1','MBNL1'],
        ['DCUN1D4','GPM6B'],
        ['DCUN1D5','NCOA2'],
        ['DCX','PASD1'],
        ['DDAH1','RP5-864K19.4'],
        ['DDAH1','SRPK1'],
        ['DDAH1','ZNHIT6'],
        ['DDB1','PRPF19'],
        ['DDB1','TMEM258'],
        ['DDI2','MTOR'],
        ['DDIT3','GLI1'],
        ['DDR1','GFAP'],
        ['DDR1','PAK1'],
        ['DDR1','PKN1'],
        ['DDR2','FMO4'],
        ['DDR2','GAB1'],
        ['DDR2','KDM2A'],
        ['DDR2','PBRM1'],
        ['DDR2','PIGU'],
        ['DDR2','TRABD'],
        ['DDX1','LAP3'],
        ['DDX1','NBAS'],
        ['DDX11','VGLL4'],
        ['DDX19A','NAIC'],
        ['DDX19B','KCTD19'],
        ['DDX19B','SF3B3'],
        ['DDX19B','TGIF1'],
        ['DDX20','TBX15'],
        ['DDX27','GLUL'],
        ['DDX27','GPNMB'],
        ['DDX27','GSN'],
        ['DDX27','HSPH1'],
        ['DDX27','LONP1'],
        ['DDX27','MTSS1'],
        ['DDX27','MXRA8'],
        ['DDX27','NCOR2'],
        ['DDX27','PERP'],
        ['DDX27','SAMHD1'],
        ['DDX27','SLC25A3'],
        ['DDX27','STOM'],
        ['DDX27','VEGFA'],
        ['DDX31','FANCC'],
        ['DDX39B','PRPH2'],
        ['DDX4','KAT6B'],
        ['DDX42','MED13'],
        ['DDX42','MRPL36'],
        ['DDX42','RPS6KB1'],
        ['DDX42','VEZF1'],
        ['DDX5','PFKP'],
        ['DDX52','TMIGD1'],
        ['DDX54','IQCD'],
        ['DDX6','FOXR1'],
        ['DDX6','SORL1'],
        ['DDX6','TSPAN18'],
        ['DDX60','FAM160A1'],
        ['DDX60L','DSCAM'],
        ['DEAF1','PDDC1'],
        ['DEAF1','RNASEH1P1'],
        ['DEAF1','RNH1'],
        ['DECR1','GNB2L1'],
        ['DEDD2','ZNF526'],
        ['DEF8','FBXO31'],
        ['DEF8','FBXO5'],
        ['DEFB103B','PRKDC'],
        ['DEFB118','ITCH'],
        ['DEFB119','MARCH-VIII'],
        ['DEFB119','MARCH8'],
        ['DEFB123','EIF2S2'],
        ['DEFB123','RALY'],
        ['DEFB123','TPX2'],
        ['DEFB125','TRIB3'],
        ['DEFB135','MSRA'],
        ['DEM1','ZNF684'],
        ['DENND1A','LRRC69'],
        ['DENND1A','LRSAM1'],
        ['DENND1A','PRRX2'],
        ['DENND1A','PTGES2'],
        ['DENND1A','SNX30'],
        ['DENND1A','SNX33'],
        ['DENND1A','STRBP'],
        ['DENND1A','UHRF1BP1L'],
        ['DENND1B','RCOR3'],
        ['DENND1B','RGL1'],
        ['DENND1B','RRP15'],
        ['DENND2A','HIPK2'],
        ['DENND2A','MKRN1'],
        ['DENND3','PTK2'],
        ['DENND5A','IFITM10'],
        ['DENND5A','MAGI3'],
        ['DENND5A','PGPEP1L'],
        ['DENND5A','SBF2'],
        ['DENND5A','ST5'],
        ['DENND5B','GALNT18'],
        ['DENND5B','NUP54'],
        ['DENND5B','PLCZ1'],
        ['DENND6B','ZNF786'],
        ['DENR','KNTC1'],
        ['DENR','RP4-785G19.5'],
        ['DEPDC1B','PDE4D'],
        ['DEPDC1B','VBP1'],
        ['DEPDC4','UHRF1BP1L'],
        ['DEPDC5','SFI1'],
        ['DEPTOR','ENPP2'],
        ['DEPTOR','NSMAF'],
        ['DEPTOR','TRPS1'],
        ['DERA','LRP6'],
        ['DES','H19'],
        ['DES','KRT6A'],
        ['DESI1','DGCR2'],
        ['DESI2','HHAT'],
        ['DEXI','FZR1'],
        ['DFNB2','GAB2'],
        ['DFNB2','MBTD1'],
        ['DFNB2','PAK1'],
        ['DFNB2','PKN1'],
        ['DGAT2','UVRAG'],
        ['DGCR2','EHBP1'],
        ['DGCR2','SEC63'],
        ['DGCR8','SGSM1'],
        ['DGCR8','WNT7B'],
        ['DGKB','GFRA1'],
        ['DGKB','MAPK9'],
        ['DGKD','RNF7'],
        ['DGKD','SAG'],
        ['DGKD','SP140'],
        ['DGKE','EPN3'],
        ['DGKE','KRT222'],
        ['DGKG','TPRG1'],
        ['DGKH','MED13L'],
        ['DGKI','EPHA5'],
        ['DGUOK','RBM45'],
        ['DGUOK','STAMBP'],
        ['DGUOK','TET3'],
        ['DHCR24','PCNXL2'],
        ['DHCR24','SIAE'],
        ['DHCR24','THBS1'],
        ['DHRS1','METTL3'],
        ['DHRS12','RASL11A'],
        ['DHRS12','SLC25A24'],
        ['DHRS4','SCFD1'],
        ['DHRS7','PDE9A'],
        ['DHX15','RBPJ'],
        ['DHX35','DNTTIP1'],
        ['DHX37','NCOR2'],
        ['DHX37','SCARB1'],
        ['DHX40','MSI2'],
        ['DHX40','PBX1'],
        ['DHX40','RNFT1'],
        ['DHX40','UNC5D'],
        ['DHX40','USP32'],
        ['DHX40','VMP1'],
        ['DHX57','SOS1'],
        ['DHX8','GPATCH8'],
        ['DHX9','ZBTB37'],
        ['DIAPH2','DXS1253E'],
        ['DIAPH2','F8'],
        ['DIAPH2','GABRA3'],
        ['DIAPH3','PCCA'],
        ['DIAPH3','TDRD3'],
        ['DIBD1','PPP2R1B'],
        ['DIDO1','ERAL1'],
        ['DIEXF','RP11-385M4.1'],
        ['DIP2A','LPRP'],
        ['DIP2A','PRH1'],
        ['DIP2A','PRMT2'],
        ['DIP2A','PRR4'],
        ['DIP2A','PVRL4'],
        ['DIP2A','RUNX1'],
        ['DIP2A','THOC2'],
        ['DIP2B','LARP4'],
        ['DIP2B','RTTN'],
        ['DIP2B','SPATS2'],
        ['DIP2B','TAF8'],
        ['DIP2B','TBC1D15'],
        ['DIP2B','TFEB'],
        ['DIP2B','TMBIM6'],
        ['DIP2C','TECTA'],
        ['DIP2C','YY1'],
        ['DIP2C','ZMYND11'],
        ['DIRAS2','GNA14'],
        ['DIRAS3','GNG12'],
        ['DIRC2','PDIA5'],
        ['DIS3L','SMAD3'],
        ['DIS3L2','EIF4E2'],
        ['DIS3L2','FAM124B'],
        ['DIS3L2','GIGYF2'],
        ['DIS3L2','SPHKAP'],
        ['DIS3L2','UBE2E3'],
        ['DISC1','LYST'],
        ['DISC1','OBSCN'],
        ['DISC1','PBX1'],
        ['DISP2','PHGR1'],
        ['DIXDC1','GAB2'],
        ['DKKL1','EEF1A2'],
        ['DLAT','FLJ10726'],
        ['DLAT','MARK2'],
        ['DLG1','HRG'],
        ['DLG1','PAK2'],
        ['DLG1','PCCB'],
        ['DLG2','FCHSD2'],
        ['DLG2','INPP5A'],
        ['DLG2','LRP5'],
        ['DLG2','ME3'],
        ['DLG2','NOX4'],
        ['DLG2','ODZ4'],
        ['DLG2','PGR'],
        ['DLG2','PLXNB2'],
        ['DLG2','PRCP'],
        ['DLG4','MARCH10'],
        ['DLG5','KCNMA1'],
        ['DLG5','PCDH15'],
        ['DLGAP1','ENOSF1'],
        ['DLGAP1','RAB31'],
        ['DLGAP1','SOGA2'],
        ['DLGAP3','KIAA0319L'],
        ['DLGAP4','OSBPL2'],
        ['DLGAP4','TGIF2'],
        ['DLGAP4','TUBD1'],
        ['DLGAP5','L2HGDH'],
        ['DLGAP5','PCCB'],
        ['DLK2','PTK7'],
        ['DMAP1','RGS9'],
        ['DMAP1','RGS9BP'],
        ['DMBT1','RAB3D'],
        ['DMC1','DNAL4'],
        ['DMD','SSR1'],
        ['DMD','TAB3'],
        ['DMGDH','NMNAT2'],
        ['DMKN','KRT13'],
        ['DMKN','KRT14'],
        ['DMRT1','SMARCA2'],
        ['DMRT2','ZMYM2'],
        ['DMRTA1','SYNE2'],
        ['DMRTC2','RPS19'],
        ['DMTF1','GRM3'],
        ['DMTF1','SEMA3D'],
        ['DMTN','SCN2A'],
        ['DMTN','TNFRSF10A'],
        ['DMXL2','PTPRR'],
        ['DMXL2','TP53BP1'],
        ['DNAH11','KLHL7'],
        ['DNAH14','EPRS'],
        ['DNAH14','RC3H1'],
        ['DNAH17','MSF1'],
        ['DNAH17','SEPT9'],
        ['DNAH17','SKAP1'],
        ['DNAH17','TK1'],
        ['DNAH17','TMC6'],
        ['DNAH2','KDM6B'],
        ['DNAH3','GNL2'],
        ['DNAH3','TMEM159'],
        ['DNAH5','MYO10'],
        ['DNAH5','TRIO'],
        ['DNAH6','TCF7L1'],
        ['DNAH6','USP39'],
        ['DNAH8','DTNBP1'],
        ['DNAH8','GMDS'],
        ['DNAH8','TMEM217'],
        ['DNAH9','MAP2K4'],
        ['DNAI2','TMEM104'],
        ['DNAJA1','GALNTL6'],
        ['DNAJB1','PRKACA'],
        ['DNAJB12','SLC25A16'],
        ['DNAJB13','FAM168A'],
        ['DNAJC1','NEBL'],
        ['DNAJC1','PLEKHF2'],
        ['DNAJC1','TAF3'],
        ['DNAJC1','ZNF462'],
        ['DNAJC11','NOL9'],
        ['DNAJC11','PER3'],
        ['DNAJC17','PPP1R14D'],
        ['DNAJC2','SLC26A5'],
        ['DNAJC22','MLF2'],
        ['DNAJC22','NTN4'],
        ['DNAJC27','HADHA'],
        ['DNAJC3','GPC6'],
        ['DNAJC5','PCMTD2'],
        ['DNAJC5','PRPF6'],
        ['DNAJC5B','SGK3'],
        ['DNAJC5B','TSC22D1'],
        ['DNAJC6','PDE1C'],
        ['DNAJC6','ROR1'],
        ['DNAJC7','ERBB2'],
        ['DNAJC7','FYTTD1'],
        ['DNAL4','HID1'],
        ['DNER','PID1'],
        ['DNER','PSMD1'],
        ['DNER','TRIP12'],
        ['DNER','UNC13B'],
        ['DNM1','LCN2'],
        ['DNM1L','ITPR2'],
        ['DNM1L','PKP2'],
        ['DNM1L','VIM'],
        ['DNM2','FDX1L'],
        ['DNM2','ILF3'],
        ['DNM2','LDLR'],
        ['DNM2','PDE4A'],
        ['DNM2','PPAN-P2RY11'],
        ['DNM2','SLC44A2'],
        ['DNM2','SMARCA4'],
        ['DNM2','ZGLP1'],
        ['DNM3','SUCO'],
        ['DNMBP','TPTE2'],
        ['DNMT1','ICAM3'],
        ['DNMT1','RAVER1'],
        ['DNMT3B','DUSP15'],
        ['DNMT3B','MDM4'],
        ['DNPEP','LPCAT1'],
        ['DNPEP','SNX9'],
        ['DNTTIP2','GCLM'],
        ['DOCK1','DPCD'],
        ['DOCK1','FUOM'],
        ['DOCK1','GALNT16'],
        ['DOCK1','MGMT'],
        ['DOCK1','PCDH15'],
        ['DOCK1','PPAPDC1A'],
        ['DOCK2','MBD2'],
        ['DOCK2','NUBPL'],
        ['DOCK3','SMARCC1'],
        ['DOCK4','IMMP2L'],
        ['DOCK6','RAVER1'],
        ['DOCK6','SMARCA4'],
        ['DOCK6','ULK2'],
        ['DOCK7','INADL'],
        ['DOCK7','PPP1R12B'],
        ['DOCK8','JAK2'],
        ['DOCK8','KANK1'],
        ['DOCK8','OSBP'],
        ['DOCK8','SLC20A2'],
        ['DOCK9','SPACA7'],
        ['DOK5','SYTL2'],
        ['DOK6','TRAPPC8'],
        ['DOLPP1','SPTAN1'],
        ['DOLPP1','TBC1D13'],
        ['DOLPP1','TSC1'],
        ['DOPEY2','FAM222B'],
        ['DOPEY2','MORC3'],
        ['DOPEY2','SH3BGR'],
        ['DOPEY2','TRPM2'],
        ['DOT1L','EPB41L2'],
        ['DOT1L','HIVEP3'],
        ['DOT1L','MUM1'],
        ['DOT1L','ONECUT3'],
        ['DP3','GOSR1'],
        ['DPEP1','SPATA33'],
        ['DPF1','SPINT2'],
        ['DPF2','FRMD8'],
        ['DPF2','METTL12'],
        ['DPF2','POLA2'],
        ['DPF2','TRIM49C'],
        ['DPH3','MAGI1'],
        ['DPH7','PNPLA7'],
        ['DPH7','PPP2R4'],
        ['DPM1','GRID1'],
        ['DPP3','FAT3'],
        ['DPP3','FKBP2'],
        ['DPP3','PACS1'],
        ['DPP4','PDK1'],
        ['DPP6','GALNT11'],
        ['DPP6','INSIG1'],
        ['DPP8','TICRR'],
        ['DPY19L1','NRXN2'],
        ['DPY19L1','SDK1'],
        ['DPY19L2','PPFIA2'],
        ['DPY19L3','GPR68'],
        ['DPY19L4','INTS8'],
        ['DPYD','EDIL3'],
        ['DPYD','RP11-359C24.1'],
        ['DPYD','STAB2'],
        ['DPYD','UNK'],
        ['DPYS','VPS13B'],
        ['DPYSL2','RP11-360L9.4'],
        ['DRAM1','FRS2'],
        ['DRAM1','NUP107'],
        ['DRD2','USP28'],
        ['DRD3','LRBA'],
        ['DRD3','NAALADL2'],
        ['DRG1','MYO18B'],
        ['DRG2','ZNF624'],
        ['DRGX','KTN1'],
        ['DROSHA','NRBP1'],
        ['DROSHA','TERT'],
        ['DSCAM','DYRK1A'],
        ['DSCAM','PLAC1'],
        ['DSCAML1','KDELC2'],
        ['DSCAML1','MPZL2'],
        ['DSCC1','TAF2'],
        ['DSCR3','IGSF5'],
        ['DSCR3','MACROD2'],
        ['DSCR3','NIPBL'],
        ['DSG2','DSG4'],
        ['DSG3','KRT14'],
        ['DSN1','NPEPL1'],
        ['DSP','KRT14'],
        ['DST','EYS'],
        ['DST','FBLN5'],
        ['DST','HCST'],
        ['DST','HIVEP1'],
        ['DST','KCNQ5'],
        ['DST','KIAA0319L'],
        ['DST','SLC35A1'],
        ['DST','TRIM38'],
        ['DSTN','DZANK1'],
        ['DSTN','TCFL5'],
        ['DSTYK','NAV1'],
        ['DSTYK','NUCKS1'],
        ['DSTYK','PIGK'],
        ['DSTYK','THSD4'],
        ['DTD1','DZANK1'],
        ['DTD1','FRYL'],
        ['DTD1','RALGAPA2'],
        ['DTNA','ZNF521'],
        ['DTNB','EXOC6'],
        ['DTNB','RBKS'],
        ['DTNB','ROCK2'],
        ['DTNBP1','FRK'],
        ['DTNBP1','GPR146'],
        ['DTNBP1','RICTOR'],
        ['DTNBP1','RSBN1L'],
        ['DTWD1','FAM227B'],
        ['DTWD2','S100A8'],
        ['DTWD2','SEMA6A'],
        ['DTX3','FRS2'],
        ['DTX4','FCHSD2'],
        ['DTX4','MRPL48'],
        ['DUPD1','KIFAP3'],
        ['DUS2','HSD11B2'],
        ['DUS2','RLTPR'],
        ['DUS2','UNC5C'],
        ['DUSP10','PLXNA2'],
        ['DUSP12','NAV1'],
        ['DUSP13','PCDH15'],
        ['DUSP14','EME1'],
        ['DUSP14','GPI'],
        ['DUSP14','NLK'],
        ['DUSP15','NCOA3'],
        ['DUSP15','TPX2'],
        ['DUSP22','PTK7'],
        ['DUSP24','GSAP'],
        ['DUSP26','WHSC1L1'],
        ['DUSP27','KIFAP3'],
        ['DUSP28','WHSC1L1'],
        ['DUSP7','PBRM1'],
        ['DXS1253E','G6PD'],
        ['DYDC2','TSPAN14'],
        ['DYM','EGFR'],
        ['DYM','MYO5B'],
        ['DYM','SLC14A2'],
        ['DYM','SMAD7'],
        ['DYM','TMC7'],
        ['DYM','TNPO3'],
        ['DYNC1H1','PPP2R5C'],
        ['DYNC1H1','SIAE'],
        ['DYNC1I1','IMMP2L'],
        ['DYNC1I1','MET'],
        ['DYNC1I1','PTPN12'],
        ['DYNC1I1','SLTM'],
        ['DYNC1I1','SRPK2'],
        ['DYNC1LI1','GADL1'],
        ['DYNC2H1','NCKAP5'],
        ['DYNLL2','SEPT4'],
        ['DYNLRB1','PHF20'],
        ['DYNLT3','U2AF2'],
        ['DYRK1A','HUNK'],
        ['DYRK1A','KDM4B'],
        ['DYRK1A','MRPS6'],
        ['DYRK1A','RUNX1'],
        ['DYRK1A','TTC3'],
        ['DYRK1A','ZBTB46'],
        ['DYRK2','PKP1'],
        ['DYRK2','SMS'],
        ['DYT12','HKR1'],
        ['DYX1C1','RFX7'],
        ['DYX1C1','SGK3'],
        ['DZIP1','PCCA'],
        ['DZIP1L','MBNL1'],
        ['E2F3','FBXO9'],
        ['E2F3','USP12'],
        ['E2F6','LYRM4'],
        ['E2F6','STK36'],
        ['EAF2','RGS9'],
        ['EAF2','RGS9BP'],
        ['EAPP','SNX6'],
        ['EAPP','USP13'],
        ['EARS2','VMP1'],
        ['EBF4','TMC2'],
        ['EBPL','LPAR6'],
        ['ECAC2','ECE3'],
        ['ECAC2','KEL'],
        ['ECD','KCNMA1'],
        ['ECD','TTC18'],
        ['ECE1','EIF4G3'],
        ['ECE1','FAM183A'],
        ['ECE1','PLA2G5'],
        ['ECE2','HTR3C'],
        ['ECE2','PAK2'],
        ['ECE2','PSMD2'],
        ['ECE3','TRPV6'],
        ['ECHDC1','FYN'],
        ['ECHDC1','MAP7'],
        ['ECHDC1','SESN1'],
        ['ECHDC1','SLC22A23'],
        ['ECHDC2','SCP2'],
        ['ECHDC2','SYCP2'],
        ['ECSIT','PNPLA6'],
        ['ECSIT','RGL3'],
        ['ECT2','KALRN'],
        ['EDA','SSBP3'],
        ['EDARADD','ENO1'],
        ['EDARADD','LGALS8'],
        ['EDARADD','SMYD3'],
        ['EDARADD','TGFBR3'],
        ['EDC3','SNX29'],
        ['EDEM2','MBTPS2'],
        ['EDEM2','UQCC'],
        ['EDEM2','UQCC1'],
        ['EDEM3','RP11-44H4.1'],
        ['EEA1','LHFPL3'],
        ['EEA1','MPG'],
        ['EEA1','PAH'],
        ['EEA1','PTGDS'],
        ['EEA1','SLC34A2'],
        ['EEA1','TMCC3'],
        ['EEA1','UBE2N'],
        ['EEA1','VIM'],
        ['EEF1B2','GIGYF2'],
        ['EEF1B2','S52'],
        ['EEF1D','TSTA3'],
        ['EEF1E1','EYS'],
        ['EEF1E1','RREB1'],
        ['EEF2','KRT14'],
        ['EEF2','KRT5'],
        ['EEF2K','KIAA0430'],
        ['EEF2K','NUP210L'],
        ['EEF2K','UQCRC2'],
        ['EEF2K','VWA3A'],
        ['EEFSEC','UBAP2L'],
        ['EEPD1','FAM114A1'],
        ['EEPD1','PRPSAP2'],
        ['EFCAB12','MBD4'],
        ['EFCAB13','MYL4'],
        ['EFCAB2','SDCCAG8'],
        ['EFCAB3','MBTD1'],
        ['EFCAB3','RPS6KB1'],
        ['EFCAB3','TANC2'],
        ['EFCAB5','KSR1'],
        ['EFCAB5','MSI2'],
        ['EFCAB5','NSRP1'],
        ['EFCAB6','PROSAP1'],
        ['EFCAB6','SHANK2'],
        ['EFCAB8','PLAT'],
        ['EFHA1','ZDHHC20'],
        ['EFHB','PLCL2'],
        ['EFHC1','LEC3'],
        ['EFHC1','LPHN3'],
        ['EFHD1','EIF4E2'],
        ['EFHD1','GIGYF2'],
        ['EFHD2','SPEN'],
        ['EFNA3','FLAD1'],
        ['EFNA3','HFE2'],
        ['EFNA3','PIK3C2G'],
        ['EFNA3','RGMC'],
        ['EFNA3','SHC1'],
        ['EFNA3','USP21'],
        ['EFNA3','USP25'],
        ['EFNA4','FLAD1'],
        ['EFNA5','FAM172A'],
        ['EFR3A','VPS13B'],
        ['EFR3A','ZFAT'],
        ['EFTUD1','FAM98B'],
        ['EFTUD1','IGF1R'],
        ['EFTUD1','KHDRBS2'],
        ['EFTUD2','KRT24'],
        ['EGFL6','RAB9A'],
        ['EGFR','ERP44'],
        ['EGFR','GNS'],
        ['EGFR','MDM2'],
        ['EGFR','PPM1H'],
        ['EGFR','SEPT14'],
        ['EGLN1','PGBD5'],
        ['EGLN2','PLD3'],
        ['EGLN3','STRN3'],
        ['EGR1','PPP1R37'],
        ['EGR2','HERC4'],
        ['EHBP1','PREX1'],
        ['EHD2','SEPW1'],
        ['EHD4','FSIP1'],
        ['EHD4','GPR176'],
        ['EHF','NAV2'],
        ['EHMT1','ERRFI1'],
        ['EHMT1','GPRIN1'],
        ['EHMT1','GRIN1'],
        ['EHMT1','PPP1R37'],
        ['EHMT2','SLC44A4'],
        ['EHMT2','SRPK1'],
        ['EIF1','MED28'],
        ['EIF2A','RNF139'],
        ['EIF2AK1','RNF216'],
        ['EIF2AK2','RMDN2'],
        ['EIF2AK2','RNF216'],
        ['EIF2AK3','PIEZO1'],
        ['EIF2AK4','EPB42'],
        ['EIF2AK4','PKM'],
        ['EIF2B3','HYI'],
        ['EIF2B3','SCMH1'],
        ['EIF2B3','SZT2'],
        ['EIF2B3','TMEM53'],
        ['EIF2C2','PTK2'],
        ['EIF2S1','GPHN'],
        ['EIF2S1','RNF139'],
        ['EIF3D','NF2'],
        ['EIF3D','SMOX'],
        ['EIF3M','PDHX'],
        ['EIF4A1','PHOSPHO1'],
        ['EIF4B','PDZRN4'],
        ['EIF4E2','GIGYF2'],
        ['EIF4E2','TMOD3'],
        ['EIF4E3','FOXP1'],
        ['EIF4EBP1','TACC1'],
        ['EIF4ENIF1','TEX14'],
        ['EIF4G1','SNW1'],
        ['EIF4G2','NUCB2'],
        ['EIF4G2','ZNFX1'],
        ['EIF4G3','KIF17'],
        ['EIF4G3','LIN28A'],
        ['EIF4G3','NSUN4'],
        ['EIF4G3','PPP2R3A'],
        ['EIF4G3','PUM1'],
        ['EIF4G3','RAP1GAP'],
        ['EIF4H','ELN'],
        ['EIF4H','WBS'],
        ['EIF5','MARK3'],
        ['EIF6','S100A9'],
        ['ELAVL1','ZNF699'],
        ['ELAVL2','ZDHHC21'],
        ['ELAVL3','FGFR3'],
        ['ELAVL4','ZCCHC11'],
        ['ELF1','FOXO1'],
        ['ELK3','LGALS3BP'],
        ['ELK3','REXO2'],
        ['ELK3','SFN'],
        ['ELL','KMT2A'],
        ['ELL','MLL'],
        ['ELL','MVB12A'],
        ['ELL','SMIM7'],
        ['ELMO1','NLK'],
        ['ELMO1','SCIN'],
        ['ELMO1','TNS3'],
        ['ELMO2','STAP1'],
        ['ELMO2','UXT'],
        ['ELMOD3','TCF7L1'],
        ['ELMOD3','TTC27'],
        ['ELMOD3','USP39'],
        ['ELN','KIAA0038'],
        ['ELN','METTL2B'],
        ['ELOF1','FOXO1'],
        ['ELOF1','RDH8'],
        ['ELOF1','USP32'],
        ['ELOVL2','MAK'],
        ['ELOVL5','FAR1'],
        ['ELOVL5','HMGCLL1'],
        ['ELOVL6','POMK'],
        ['ELOVL7','PDE4D'],
        ['ELP2','GALNT1'],
        ['ELP2','MAPRE2'],
        ['ELP3','MYO5C'],
        ['ELP4','LDLRAD3'],
        ['ELP6','PTPN1'],
        ['ELSPBP1','SYMPK'],
        ['ELSPBP1','ZNF460'],
        ['EMC10','MBOAT7'],
        ['EME1','TMPRSS6'],
        ['EMID1','PPP6R2'],
        ['EMILIN3','PMEPA1'],
        ['EML1','EVL'],
        ['EML1','FBLN5'],
        ['EML2','GYS1'],
        ['EML2','OPA3'],
        ['EML4','MTA3'],
        ['EML4','THADA'],
        ['EMP1','ETV6'],
        ['ENDOV','IKZF3'],
        ['ENDOV','SLC26A11'],
        ['ENDOV','TMEM104'],
        ['ENDOV','USP43'],
        ['ENO1','ERBB2'],
        ['ENO1','KRT14'],
        ['ENO1','KRT5'],
        ['ENO1','KRT6B'],
        ['ENO1','PDK4'],
        ['ENO3','KRT5'],
        ['ENOPH1','MRPS18C'],
        ['ENOX2','OSBPL5'],
        ['ENPP1','RIPPLY2'],
        ['ENPP1','TESK2'],
        ['ENPP2','HSD17B3'],
        ['ENTHD1','KREMEN1'],
        ['ENTHD1','POLR3H'],
        ['ENTPD5','MARK3'],
        ['ENTPD7','IDE'],
        ['EP300','FBLN1'],
        ['EP300','IPO5'],
        ['EP300','MCHR1'],
        ['EP300','TNRC6B'],
        ['EP300','TTC28'],
        ['EP400','SFSWAP'],
        ['EP400','YEATS4'],
        ['EPB41','ITPR1'],
        ['EPB41','PIGV'],
        ['EPB41','SNRNP40'],
        ['EPB41L1','GHRH'],
        ['EPB41L1','PHF20'],
        ['EPB41L1','RALGAPB'],
        ['EPB41L1','TGM3'],
        ['EPB41L2','GPR156'],
        ['EPB41L2','LAMA2'],
        ['EPB41L3','LAMA1'],
        ['EPB41L3','LRRC34'],
        ['EPB41L4A','PLAC8L1'],
        ['EPB41L4B','PTPN3'],
        ['EPC1','PTGR2'],
        ['EPC2','MBD5'],
        ['EPC2','THSD7B'],
        ['EPDR1','TRA2A'],
        ['EPG5','ST8SIA5'],
        ['EPHA1','LMBR1'],
        ['EPHA1','USP32'],
        ['EPHA2','HCK'],
        ['EPHA2','NECAP2'],
        ['EPHA2','PREX1'],
        ['EPHA3','LCLAT1'],
        ['EPHA3','LRRFIP2'],
        ['EPHA4','POU2F1'],
        ['EPHA5','HOXB6'],
        ['EPHA5','MGA'],
        ['EPHA5','RUFY3'],
        ['EPHA6','TMEM39A'],
        ['EPHA6','TRMT10C'],
        ['EPHA6','ZBTB20'],
        ['EPHA7','MYH11'],
        ['EPHA7','NDE1'],
        ['EPHA7','PIGH'],
        ['EPHB1','MOBKL1B'],
        ['EPHB1','STAG2'],
        ['EPHB2','NBPF3'],
        ['EPHB2','PDZD4'],
        ['EPHB3','PAX2'],
        ['EPHB3','PGC'],
        ['EPHB3','PIGZ'],
        ['EPHB4','PGC'],
        ['EPHX2','ESCO2'],
        ['EPHX2','PTK2B'],
        ['EPN1','U2AF2'],
        ['EPN2','KCNJ12'],
        ['EPN2','KCNJN1'],
        ['EPN2','KIF13B'],
        ['EPN2','SGSM2'],
        ['EPN3','SLC25A17'],
        ['EPOR','RFX2'],
        ['EPRS','USH2A'],
        ['EPRS','USP19'],
        ['EPS15','INADL'],
        ['EPS15','OSBPL9'],
        ['EPS15L1','ZNF473'],
        ['EPS8','GYS2'],
        ['EPS8','PLEKHA5'],
        ['EPS8','PRH2'],
        ['EPS8L2','TSPAN4'],
        ['EPYC','UHRF1BP1'],
        ['ERAP2','FAM172A'],
        ['ERBB2','FKBP10'],
        ['ERBB2','GTF2E2'],
        ['ERBB2','IFLTD1'],
        ['ERBB2','IKZF3'],
        ['ERBB2','KRT20'],
        ['ERBB2','KRT39'],
        ['ERBB2','KRTAP1-4'],
        ['ERBB2','MED1'],
        ['ERBB2','MTSS1'],
        ['ERBB2','NARS2'],
        ['ERBB2','PARN'],
        ['ERBB2','PPP1R1B'],
        ['ERBB2','PSMB3'],
        ['ERBB2','SLC29A3'],
        ['ERBB2','SLC39A11'],
        ['ERBB2','SMIM18'],
        ['ERBB2','SPTBN2'],
        ['ERBB2','SRCIN1'],
        ['ERBB2','TADA2A'],
        ['ERBB2IP','NLN'],
        ['ERBB2IP','PPWD1'],
        ['ERBB2IP','TMEM232'],
        ['ERBB2IP','TRAPPC13'],
        ['ERBB2IP','ZBTB10'],
        ['ERBB4','FN1'],
        ['ERBB4','KANSL1L'],
        ['ERC1','NAV2'],
        ['ERC1','PIK3C2G'],
        ['ERC1','RAD52'],
        ['ERC1','RET'],
        ['ERC1','TOMM34'],
        ['ERC1','VWF'],
        ['ERC1','WNK1'],
        ['ERC1','WNT5B'],
        ['ERC2','FAM172A'],
        ['ERC2','LNPEP'],
        ['ERC2','RAD54L2'],
        ['ERC2','SCGB3A1'],
        ['ERCC1','ZNF229'],
        ['ERCC3','MAP3K2'],
        ['ERCC6L2','GTF3C4'],
        ['ERCC8','PDE4D'],
        ['ERG','HNRNPH1'],
        ['ERG','NDRG1'],
        ['ERG','PAK1'],
        ['ERG','PKN1'],
        ['ERG','TMPRSS2'],
        ['ERGIC1','RANBP17'],
        ['ERH','SETDB2'],
        ['ERI3','GATAD2B'],
        ['ERI3','LEPRE1'],
        ['ERI3','P66BETA'],
        ['ERICH1','MBD1'],
        ['ERICH1','PCM1'],
        ['ERK7','MED25'],
        ['ERK7','MED9'],
        ['ERLIN2','FGFR1'],
        ['ERLIN2','ROBO2'],
        ['ERLIN2','TBC1D25'],
        ['ERO1LB','HEATR1'],
        ['ERP27','GNE'],
        ['ERP44','MRPL50'],
        ['ERRFI1','LUZP1'],
        ['ERVFRD-1','NOL7'],
        ['ESCO1','FAM65C'],
        ['ESCO1','KANK1'],
        ['ESCO1','LAMA3'],
        ['ESCO1','LAMA4'],
        ['ESCO2','PTK2B'],
        ['ESD','ITM2B'],
        ['ESD','TDRD3'],
        ['ESF1','LRBA'],
        ['ESPN','FBLIM1'],
        ['ESPN','ZBTB48'],
        ['ESR1','MTHFD1L'],
        ['ESR1','TFB1M'],
        ['ESR1','TULP4'],
        ['ESR1','USP25'],
        ['ESRP1','RMDN1'],
        ['ESRP1','WWP1'],
        ['ESRP2','SLC7A6'],
        ['ESRRG','RNF121'],
        ['ESYT1','MED14'],
        ['ESYT2','HTT'],
        ['ESYT2','SLC6A4'],
        ['ETFA','STRA6'],
        ['ETFB','LIM2'],
        ['ETFB','ZNF805'],
        ['ETHE1','QPCTL'],
        ['ETHE1','SIPA1L3'],
        ['ETHE1','ULK1'],
        ['ETHE1','UPK3A'],
        ['ETHE1','ZNF791'],
        ['ETNK1','GLT1D1'],
        ['ETNK1','RCAN2'],
        ['ETV3','IQGAP3'],
        ['ETV4','XPO7'],
        ['ETV5','RNF213'],
        ['ETV6','NTRK3'],
        ['ETV6','PLEKHG6'],
        ['ETV6','YTHDF2'],
        ['ETV7','KCTD20'],
        ['EVA1C','MRAP'],
        ['EVA1C','MROH8'],
        ['EVC','TBC1D14'],
        ['EVC','WDR60'],
        ['EVC2','STK32B'],
        ['EVI5L','LRRC8E'],
        ['EVI5L','VAV1'],
        ['EVL','LCK'],
        ['EVPL','JUP'],
        ['EVPL','KRT17'],
        ['EVPL','MAP2K6'],
        ['EVPL','SRP68'],
        ['EXD2','GMEB2'],
        ['EXD2','PTPRT'],
        ['EXD3','PNPLA7'],
        ['EXOC1','IDO1'],
        ['EXOC1','RBM47'],
        ['EXOC1','SPATA18'],
        ['EXOC1','XKR3'],
        ['EXOC2','SH3RF3'],
        ['EXOC3','SHANK3'],
        ['EXOC3L1','TK2'],
        ['EXOC4','LRGUK'],
        ['EXOC4','PTAFR'],
        ['EXOC4','SLC25A13'],
        ['EXOC4','SUPT3H'],
        ['EXOC6','TBC1D12'],
        ['EXOC6B','HAT1'],
        ['EXOC6B','HNRNPLL'],
        ['EXOC6B','MTAP'],
        ['EXOC6B','SFXN5'],
        ['EXOC6B','ZNF638'],
        ['EXOC7','FOXJ1'],
        ['EXOC7','USP22'],
        ['EXOSC1','RRP12'],
        ['EXOSC5','LEUTX'],
        ['EXT1','FAM155A'],
        ['EXT1','LRRC6'],
        ['EXT1','OC90'],
        ['EXT1','RSF1'],
        ['EXT1','SAMD12'],
        ['EXT1','WDYHV1'],
        ['EXTL3','NDUFS4'],
        ['EXTL3','TMEM66'],
        ['EYA2','NCOA3'],
        ['EYA2','TM9SF4'],
        ['EYA2','ZMYND8'],
        ['EYA3','KIAA0247'],
        ['EYA3','POP4'],
        ['EYA4','PTPRK'],
        ['EYS','GPI'],
        ['EYS','KIDINS220'],
        ['EYS','LMBRD1'],
        ['EYS','MBD4'],
        ['EYS','MED1'],
        ['EYS','PHF3'],
        ['EYS','PRIM2'],
        ['EYS','PTP4A1'],
        ['EYS','SACS'],
        ['EYS','SESN1'],
        ['EYS','TPX2'],
        ['EZH1','TOP2A'],
        ['EZH1','WIPF2'],
        ['EZH1','ZC3H3'],
        ['EZH2','GALNT11'],
        ['EZH2','TOP2A'],
        ['EZH2','WIPF2'],
        ['EZH2','ZC3H3'],
        ['EZR','KRT14'],
        ['EZR','ROS1'],
        ['F-SPONDIN','PCF11'],
        ['F-SPONDIN','SIRT3'],
        ['F11R','NOS1AP'],
        ['F12','LMAN2'],
        ['F13A1','PRPF4B'],
        ['F13B','KIF21B'],
        ['F2','LPP'],
        ['F5','NOS1AP'],
        ['F5','SRF'],
        ['F8','G6PD'],
        ['F8','ODF2L'],
        ['FA2H','RAD54L'],
        ['FAAH','RAD54L'],
        ['FABP3','ZCCHC17'],
        ['FABP5','JUP'],
        ['FABP5','KRT14'],
        ['FABP5','KRT17'],
        ['FADD','IKBKB'],
        ['FADD','PPP6R3'],
        ['FAF1','HCRTR1'],
        ['FAF1','HOOK1'],
        ['FAF1','LRRC7'],
        ['FAF1','MACF1'],
        ['FAF1','NFIA'],
        ['FAF1','PTRH2'],
        ['FAF1','TAF4'],
        ['FAHD1','NUDT16L1'],
        ['FAHD2B','ZNF251'],
        ['FAIM3','MYO1D'],
        ['FAM102A','NAIF1'],
        ['FAM102A','PTGES2'],
        ['FAM102B','SLC25A24'],
        ['FAM102B','TRAPPC8'],
        ['FAM104A','TMC6'],
        ['FAM107B','USP6NL'],
        ['FAM107B','WDR37'],
        ['FAM109B','RAI14'],
        ['FAM110A','TLK2'],
        ['FAM111B','GLYATL1'],
        ['FAM111B','MS4A5'],
        ['FAM114A1','IQCE'],
        ['FAM117A','NURF301'],
        ['FAM117A','SPAG9'],
        ['FAM117A','SPOP'],
        ['FAM122B','GPC4'],
        ['FAM126A','RBBP8'],
        ['FAM129B','OLFM1'],
        ['FAM129B','PYROXD1'],
        ['FAM129B','SEC16A'],
        ['FAM129B','ZNF79'],
        ['FAM131A','NAA50'],
        ['FAM131C','ZBTB17'],
        ['FAM13C','FAM149B1'],
        ['FAM13C','ZNF365'],
        ['FAM149B1','INPP5A'],
        ['FAM149B1','KCNMA1'],
        ['FAM151A','MKNK2'],
        ['FAM155A','MAPK1'],
        ['FAM155A','TUBGCP2'],
        ['FAM155A','WNK1'],
        ['FAM156B','RPS6KA3'],
        ['FAM159A','FNIP2'],
        ['FAM159A','LRRC41'],
        ['FAM160A1','IL15'],
        ['FAM160A2','ZNF195'],
        ['FAM161A','USP34'],
        ['FAM168A','FCHSD2'],
        ['FAM168A','MARK2'],
        ['FAM168A','MRPL21'],
        ['FAM168A','NADC-1'],
        ['FAM168A','PTCRA'],
        ['FAM168A','PXDNL'],
        ['FAM168A','RAB6A'],
        ['FAM168A','SLC13A2'],
        ['FAM168A','SOX10'],
        ['FAM168A','UVRAG'],
        ['FAM168A','XRRA1'],
        ['FAM168B','PLEKHB2'],
        ['FAM171A1','RP11-534G20.3'],
        ['FAM172A','FHIT'],
        ['FAM172A','KIAA0825'],
        ['FAM177A1','IQSEC3'],
        ['FAM177A1','KIAA0391'],
        ['FAM177A1','KIAA1110'],
        ['FAM177A1','PSMA6'],
        ['FAM178B','LMAN2L'],
        ['FAM178B','SEMA4C'],
        ['FAM179A','UBASH3A'],
        ['FAM179B','NIN'],
        ['FAM179B','PRPF39'],
        ['FAM179B','UBE2E2'],
        ['FAM186A','MYO10'],
        ['FAM187B','PLEKHA1'],
        ['FAM190A','MAPK10'],
        ['FAM190A','MMRN1'],
        ['FAM190A','RP11-115D19.1'],
        ['FAM193A','TNIP2'],
        ['FAM195A','RAB11FIP3'],
        ['FAM19A1','SLC25A26'],
        ['FAM19A2','KCNMB4'],
        ['FAM19A2','PPM1H'],
        ['FAM19A2','RAB12'],
        ['FAM19A4','FHIT'],
        ['FAM19A5','PI4KA'],
        ['FAM19A5','SYNE3'],
        ['FAM19A5','TNRC6B'],
        ['FAM19A5','ZFP2'],
        ['FAM208B','NET1'],
        ['FAM208B','PFKFB3'],
        ['FAM208B','PRPF38B'],
        ['FAM208B','SLC6A2'],
        ['FAM208B','SLC6A5'],
        ['FAM20B','RALGPS2'],
        ['FAM20B','SAMD7'],
        ['FAM20C','MKRN1'],
        ['FAM211A','PEMT'],
        ['FAM213A','GRHL2'],
        ['FAM216B','UBE2K'],
        ['FAM217B','NADC-1'],
        ['FAM217B','PRPF6'],
        ['FAM217B','SLC13A2'],
        ['FAM221A','MUC4'],
        ['FAM222A','MSI1'],
        ['FAM222B','GIP'],
        ['FAM222B','GNAI2'],
        ['FAM222B','GYS2'],
        ['FAM222B','NEK8'],
        ['FAM222B','NEK9'],
        ['FAM222B','PHF12'],
        ['FAM222B','SEZ6'],
        ['FAM222B','TANC2'],
        ['FAM222B','TAOK1'],
        ['FAM222B','UNC5D'],
        ['FAM227B','TRPM7'],
        ['FAM24B','KIAA1598'],
        ['FAM32A','TPM4'],
        ['FAM3B','MORC3'],
        ['FAM47E','SHROOM3'],
        ['FAM49A','GRHL2'],
        ['FAM49B','KAT6A'],
        ['FAM49B','PREX2'],
        ['FAM49B','PSKH2'],
        ['FAM49B','RP11-3O20.1'],
        ['FAM53A','WHSC1'],
        ['FAM53B','KCNN1'],
        ['FAM53B','SGK2'],
        ['FAM53B','SGK3'],
        ['FAM53C','HARS'],
        ['FAM65A','LRRC36'],
        ['FAM65B','MRS2'],
        ['FAM65B','WASF2'],
        ['FAM69A','LRRC8C'],
        ['FAM69A','LRRC8D'],
        ['FAM69A','NTNG1'],
        ['FAM69A','NTNG2'],
        ['FAM71C','SCYL2'],
        ['FAM73B','NUP188'],
        ['FAM73B','PPP2R4'],
        ['FAM76A','WDTC1'],
        ['FAM81B','TMEM161B'],
        ['FAM83B','ICK'],
        ['FAM83B','NUMA1'],
        ['FAM83B','RP11-524H19.2'],
        ['FAM83D','SF3B3'],
        ['FAM83D','SPECC1L'],
        ['FAM83D','ZNF107'],
        ['FAM83E','NUCB1'],
        ['FAM83F','RTCB'],
        ['FAM83G','GRAP'],
        ['FAM83G','KRT14'],
        ['FAM83G','TTC19'],
        ['FAM83H','SRGAP3'],
        ['FAM96A','SNX1'],
        ['FAM98A','GALNT14'],
        ['FAM98B','RASGRP1'],
        ['FANCD2','MTMR14'],
        ['FANCD2','PRKCD'],
        ['FANCD2OS','PRKCD'],
        ['FANCI','PCGF3'],
        ['FANK1','LHPP'],
        ['FANK1','WBP4'],
        ['FAR1','FLYWCH1'],
        ['FAR1','FSCN1'],
        ['FAR1','GRN'],
        ['FAR1','GSN'],
        ['FAR1','MPO'],
        ['FAR1','PAEP'],
        ['FAR1','PDGFRB'],
        ['FAR1','UGGT1'],
        ['FAR2','SMAP1'],
        ['FARP1','GPC6'],
        ['FARP1','NALCN'],
        ['FARP1','NME1-NME2'],
        ['FARP1','SIAE'],
        ['FARP1','TUBGCP3'],
        ['FARP1','UCHL3'],
        ['FARP2','ITGAV'],
        ['FARP2','SEPT2'],
        ['FARP2','SEPT6'],
        ['FARS2','GCNT2'],
        ['FARS2','PHACTR1'],
        ['FARS2','RREB1'],
        ['FARS2','SSR1'],
        ['FASLG','PRRC2C'],
        ['FASN','ZNF595'],
        ['FAT1','KIAA1430'],
        ['FAT1','MTNR1A'],
        ['FAT1','RP11-215A19.1'],
        ['FAT2','GLRA1'],
        ['FAT2','NSD1'],
        ['FAT3','MTMR2'],
        ['FATE1','MTMR1'],
        ['FAXDC2','GALNT10'],
        ['FBF1','HELZ'],
        ['FBF1','SIAE'],
        ['FBL','RASGRP4'],
        ['FBL','RYR1'],
        ['FBLIM1','SPEN'],
        ['FBLIM1','TIAM1'],
        ['FBLN1','HACE1'],
        ['FBLN1','PDGFRB'],
        ['FBLN2','NUP210'],
        ['FBLN5','MARK3'],
        ['FBLN5','SMEK1'],
        ['FBLN5','TC2N'],
        ['FBN3','SLC35G6'],
        ['FBP2','PRKAB2'],
        ['FBRSL1','GALNAC-T9'],
        ['FBRSL1','GALNT9'],
        ['FBXL20','TLK2'],
        ['FBXL20','UNC45B'],
        ['FBXL3','KLF5'],
        ['FBXL7','TRIO'],
        ['FBXL7','UHMK1'],
        ['FBXO11','NEBL'],
        ['FBXO15','ZBTB7C'],
        ['FBXO18','ITGA8'],
        ['FBXO21','RNF10'],
        ['FBXO22','SNX33'],
        ['FBXO28','LYZL2'],
        ['FBXO3','KDM2A'],
        ['FBXO3','NCOR1'],
        ['FBXO34','HECTD1'],
        ['FBXO34','LGALS3'],
        ['FBXO34','SMEK1'],
        ['FBXO38','SPINK6'],
        ['FBXO42','RUNX1'],
        ['FBXO45','NAALADL2'],
        ['FBXO46','SYMPK'],
        ['FBXO7','RP11-864G5.3'],
        ['FBXO7','YAP1'],
        ['FBXW7','INPP4B'],
        ['FBXW7','RP11-18H21.2'],
        ['FBXW7','SLC20A2'],
        ['FBXW7','TMEM126B'],
        ['FBXW7','TMEM154'],
        ['FBXW8','ULK1'],
        ['FCAMR','PDZRN3'],
        ['FCER1G','PTPRJ'],
        ['FCF1','TIMM9'],
        ['FCGR2A','SPTA1'],
        ['FCGRT','PRRG2'],
        ['FCHSD2','GLOD4'],
        ['FCHSD2','GUCY1A2'],
        ['FCHSD2','MAP3K3'],
        ['FCHSD2','NUMA1'],
        ['FCHSD2','PLEKHB1'],
        ['FCHSD2','RALYL'],
        ['FCHSD2','RNF169'],
        ['FCHSD2','SPRYD3'],
        ['FDCSP','SAMD12'],
        ['FDFT1','GATA4'],
        ['FDFT1','INTS9'],
        ['FDFT1','NEIL2'],
        ['FECH','NCMAP'],
        ['FER','GNB2L1'],
        ['FER1L5','LMAN2L'],
        ['FERMT1','MCM8'],
        ['FERMT1','RBBP8'],
        ['FERMT1','TVP23C'],
        ['FERMT2','GPR137C'],
        ['FEZ1','SBF2'],
        ['FEZ2','RAB10'],
        ['FGB','PLEKHA5'],
        ['FGD3','TYK2'],
        ['FGD4','LRRK2'],
        ['FGD4','NPC1'],
        ['FGD4','PRICKLE1'],
        ['FGD5','NR2C2'],
        ['FGD6','NUP37'],
        ['FGF10','HEATR6'],
        ['FGF10','NIPBL'],
        ['FGF12','LAMP3'],
        ['FGF12','SYN3'],
        ['FGF12','TSC22D2'],
        ['FGF13','KPNA3'],
        ['FGF13','NLRP3'],
        ['FGF14','TMTC4'],
        ['FGF19','SPTBN2'],
        ['FGF2','KPNA3'],
        ['FGF3','LGR4'],
        ['FGF7','GAB2'],
        ['FGFR1','RHOT1'],
        ['FGFR1','SLC20A2'],
        ['FGFR1','WHSC1L1'],
        ['FGFR1','ZNF577'],
        ['FGFR1','ZNF791'],
        ['FGFR1OP','MCM9'],
        ['FGFR2','OFD1'],
        ['FGFR2','TACC2'],
        ['FGFR2','USP10'],
        ['FGFR2','USP32'],
        ['FGFR2','VCL'],
        ['FGFR3','TACC3'],
        ['FGFR3','TPRG1'],
        ['FGGY','LDLRAD1'],
        ['FGGY','NFIA'],
        ['FGGY','RPS6KA1'],
        ['FGL1','MTUS1'],
        ['FGR','WASF2'],
        ['FHAD1','ZNF148'],
        ['FHIT','MTCH2'],
        ['FHIT','PTPRG'],
        ['FHIT','RPAIN'],
        ['FHIT','SYNPR'],
        ['FHOD1','LRRC29'],
        ['FHOD1','NFAT5'],
        ['FHOD3','MOCOS'],
        ['FHOD3','MS4A15'],
        ['FHOD3','MYOM1'],
        ['FIG4','SEC63'],
        ['FIG4','WASF1'],
        ['FIG4','ZNF292'],
        ['FIGN','STK25'],
        ['FILIP1','TMEM30A'],
        ['FILIP1L','IMPG2'],
        ['FIP1L1','PDGFRA'],
        ['FIP1L1','SBNO2'],
        ['FIP1L1','SORCS1'],
        ['FITM2','STIM1'],
        ['FKBP15','RET'],
        ['FKBP3','GALNT11'],
        ['FKBP7','ZNF385B'],
        ['FKBP8','SLC1A6'],
        ['FLJ10727','NEDD9'],
        ['FLJ20265','PCGF3'],
        ['FLJ20265','SPINK2'],
        ['FLJ20265','ZNF264'],
        ['FLJ20535','LUC7L2'],
        ['FLJ22344','MMAB'],
        ['FLJ22344','PDE4D'],
        ['FLJ22561','ZDHHC16'],
        ['FLJ22561','ZMAT4'],
        ['FLJ26468','PIBF1'],
        ['FLJ30532','WDR19'],
        ['FLJ46072','SRGAP3'],
        ['FLNB','PXK'],
        ['FLNB','SLMAP'],
        ['FLOT2','NCKAP4'],
        ['FLOT2','PEX12'],
        ['FLOT2','SOCS4'],
        ['FLOT2','SOCS7'],
        ['FLOT2','TAOK1'],
        ['FLT1','OLFM4'],
        ['FLT1','SRPK2'],
        ['FLT1','ZNF436'],
        ['FLT3','PAN3'],
        ['FLT3LG','RPS11'],
        ['FLT3LG','SCAF1'],
        ['FLT4','LBH'],
        ['FMNL2','SREBF2'],
        ['FMO5','PDIA3'],
        ['FMOD','POGK'],
        ['FMR1NB','NCOR2'],
        ['FN1','KRT5'],
        ['FN1','MKNK2'],
        ['FN3KRP','REV3L'],
        ['FN3KRP','WSB1'],
        ['FNBP1','RP11-330M2.7'],
        ['FNBP1L','TMEM56'],
        ['FNBP4','MTCH2'],
        ['FNDC1','MAP3K4'],
        ['FNDC3A','RB1'],
        ['FNDC3B','KHDRBS3'],
        ['FNDC3B','MST1R'],
        ['FNDC3B','NAALADL2'],
        ['FNDC3B','NLGN1'],
        ['FNDC3B','PHC3'],
        ['FNDC3B','PIK3CA'],
        ['FNDC3B','SPATA16'],
        ['FNDC3B','UCK2'],
        ['FNDC8','NLE1'],
        ['FNIP1','RASA1'],
        ['FNIP1','SUSD4'],
        ['FNIP2','LARP7'],
        ['FNIP2','LRBA'],
        ['FNTA','OVOL1'],
        ['FNTA','SPIDR'],
        ['FNTA','ZMAT4'],
        ['FNTB','FUT8'],
        ['FOCAD','MOB3B'],
        ['FOCAD','PTPLAD2'],
        ['FOLR1','NUMA1'],
        ['FOS','KRT6B'],
        ['FOSL2','TAF1B'],
        ['FOXI2','TNS3'],
        ['FOXJ3','RABGGTB'],
        ['FOXJ3','ZFYVE9'],
        ['FOXJ3','ZNF362'],
        ['FOXK2','MADCAM1'],
        ['FOXK2','NARF'],
        ['FOXK2','PITPNC1'],
        ['FOXK2','RNF138'],
        ['FOXK2','SKAP1'],
        ['FOXK2','TBCD'],
        ['FOXK2','TIAF1'],
        ['FOXM1','PPY'],
        ['FOXM1','SLC2A14'],
        ['FOXN1','PRKDC'],
        ['FOXN2','PLB1'],
        ['FOXN2','SPTBN1'],
        ['FOXN3','RP11-944C7.1'],
        ['FOXN3','SLC25A21'],
        ['FOXN3','ZFP36L1'],
        ['FOXP1','HLA-C'],
        ['FOXP1','MITF'],
        ['FOXP1','NDRG1'],
        ['FOXP1','PLVAP'],
        ['FOXP1','PPP4R2'],
        ['FOXP1','PRKAR1A'],
        ['FOXP1','PSMA5'],
        ['FOXP1','PXDNL'],
        ['FOXP1','RGAG4'],
        ['FOXP1','RUFY1'],
        ['FOXP1','SHC1'],
        ['FOXP1','STK4'],
        ['FOXP1','TGOLN2'],
        ['FOXR1','RCK'],
        ['FRAS1','MRPL1'],
        ['FRAS1','SHROOM3'],
        ['FREM1','NFIB'],
        ['FRMD3','HSDL2'],
        ['FRMD4A','INPP5A'],
        ['FRMD4A','MKL2'],
        ['FRMD4A','OR13A1'],
        ['FRMD4A','TMEM45B'],
        ['FRMD4B','MITF'],
        ['FRMD4B','PPP2R3A'],
        ['FRMD4B','UBA3'],
        ['FRMD5','MECOM'],
        ['FRMD6','RTN4'],
        ['FRMD6','SCFD1'],
        ['FRMD8','MAST1'],
        ['FRMD8','SCYL1'],
        ['FRRS1L','TMEM245'],
        ['FRS2','GRIP1'],
        ['FRS2','KIF5A'],
        ['FRS2','LEMD3'],
        ['FRS2','LGALS12'],
        ['FRS2','LYZ'],
        ['FRS2','MDM2'],
        ['FRS2','MSRB3'],
        ['FRS2','NCOA2'],
        ['FRS2','NRG3'],
        ['FRS2','NUP107'],
        ['FRS2','OS9'],
        ['FRS2','OSBPL8'],
        ['FRS2','PIP4K2C'],
        ['FRS2','RAB3IP'],
        ['FRS2','TSPAN8'],
        ['FRY','LRRC1'],
        ['FRYL','LCORL'],
        ['FRYL','OCIAD2'],
        ['FRYL','ZNF606'],
        ['FSCN1','KRT14'],
        ['FSIP2','GULP1'],
        ['FSIP2','PCCA'],
        ['FSTL1','GSK3B'],
        ['FSTL3','MKL2'],
        ['FSTL4','SLIT1'],
        ['FSTL4','SLIT3'],
        ['FSTL4','TWF2'],
        ['FTCD','SETD4'],
        ['FTH1','KRT5'],
        ['FTO','PHIP'],
        ['FTO','RBL2'],
        ['FTO','ZNF536'],
        ['FTSJ3','SCN4A'],
        ['FUBP1','LRRC7'],
        ['FUBP1','SLC44A5'],
        ['FUBP1','USP33'],
        ['FUNDC2','RPS6KA6'],
        ['FUS','PRSS36'],
        ['FUS','SF1'],
        ['FUT10','GAB2'],
        ['FUT8','SIPA1L1'],
        ['FXR1','RIMS2'],
        ['FXYD3','TARM1'],
        ['FXYD3','ZNFX1'],
        ['FXYD4','ZNF487'],
        ['FXYD5','TGFB1'],
        ['FXYD6','OSBP2'],
        ['FXYD6','ZEB2'],
        ['FYB','RICTOR'],
        ['FYN','KMT2A'],
        ['FYN','LAMA4'],
        ['FYN','MBOAT1'],
        ['FYN','PTPRZ1'],
        ['FYN','REV3L'],
        ['FYN','SF1'],
        ['FYN','ZFHX3'],
        ['FYTTD1','TTC2'],
        ['FZD4','LAMTOR1'],
        ['FZR1','NCLN'],
        ['FZR1','NFIC'],
        ['FZR1','NPFFR2'],
        ['FZR1','TANGO6'],
        ['G17P1','PFKP'],
        ['G2','NCR3'],
        ['G2','REXO2'],
        ['G2','SFN'],
        ['G2E3','SPRED2'],
        ['G3BP1','GLRA1'],
        ['G3BP1','PDE4D'],
        ['G3BP2','NAAA'],
        ['G3BP2','PDGFC'],
        ['GAB2','INTS4'],
        ['GAB2','MYO7A'],
        ['GAB2','PLXDC1'],
        ['GAB2','SGMS1'],
        ['GAB2','SRGAP1'],
        ['GAB2','TENM4'],
        ['GAB2','THRSP'],
        ['GAB2','WHSC1L1'],
        ['GAB3','IKBKG'],
        ['GABBR2','TBC1D2'],
        ['GABPB1','GALK2'],
        ['GABPB1','MLLT11'],
        ['GABPB2','MLLT11'],
        ['GABRA3','SPATA7'],
        ['GABRA3','VBP1'],
        ['GABRA5','NEDD4'],
        ['GABRB1','LNX1'],
        ['GABRB1','ZNF430'],
        ['GABRG3','OCA2'],
        ['GABRG3','OSBPL3'],
        ['GABRP','RANBP17'],
        ['GAK','LUC7L3'],
        ['GAK','MAGI1'],
        ['GAK','PGC'],
        ['GAK','TG'],
        ['GAL','OSBP'],
        ['GALA','HNRNPH2'],
        ['GALA','HNRPH2'],
        ['GALK1','SEC14L1'],
        ['GALNAC-T9','TMEM132D'],
        ['GALNS','PIEZO1'],
        ['GALNT10','IQCA1'],
        ['GALNT11','LMBR1'],
        ['GALNT11','PRKAG2'],
        ['GALNT11','PTPRN2'],
        ['GALNT13','HEATR6'],
        ['GALNT13','KCNJ3'],
        ['GALNT18','IER3IP1'],
        ['GALNT18','PPP1R1B'],
        ['GALNT18','PRDM2'],
        ['GALNT18','SOX6'],
        ['GALNT2','KIF26B'],
        ['GALNT5','PAPOLA'],
        ['GALNT9','TMEM132D'],
        ['GALNTL4','PRDM2'],
        ['GALNTL5','KMT2C'],
        ['GALNTL5','TRAF3IP1'],
        ['GALNTL6','NEK1'],
        ['GALNTL6','RASAL2'],
        ['GAN','GRM7'],
        ['GAP43','LSAMP'],
        ['GAP43','TMEM108'],
        ['GAREM','GTF2H1'],
        ['GAREM','LAMA3'],
        ['GAREM','LAMA4'],
        ['GARNL3','LRSAM1'],
        ['GARNL3','SLC2A8'],
        ['GART','PCP4'],
        ['GART','RCAN1'],
        ['GART','UBC7'],
        ['GART','UBE2G2'],
        ['GAS2','QSER1'],
        ['GAS6','GPC5'],
        ['GAS7','MYH1'],
        ['GAS8','SPIRE2'],
        ['GAS8','WDR59'],
        ['GAST','HAP1'],
        ['GAST','HAP2'],
        ['GATA3','TAF3'],
        ['GATA4','NCOR2'],
        ['GATA6','ROCK1'],
        ['GATAD2A','NFIX'],
        ['GATAD2A','PRKACA'],
        ['GATAD2B','NUP210L'],
        ['GATAD2B','SLC9C2'],
        ['GATAD2B','TNFSF18'],
        ['GATS','ZNF3'],
        ['GBA','POGZ'],
        ['GBA','SCAMP3'],
        ['GBE1','ZNF385D'],
        ['GBF1','NFKB2'],
        ['GBF1','SUFU'],
        ['GBGT1','PPP2R4'],
        ['GBP6','HS2ST1'],
        ['GCAT','TBC1D22A'],
        ['GCDH','NFIX'],
        ['GCFC1','TP53I3'],
        ['GCG','PMS1'],
        ['GCNT2','NUP153'],
        ['GCNT2','OSM'],
        ['GCNT2','PDE4B'],
        ['GCOM1','LPHN1'],
        ['GDA','SIAE'],
        ['GDAP1L1','VPS26A'],
        ['GDAP2','SPAG17'],
        ['GDF6','TSNARE1'],
        ['GDI2','ZNF526'],
        ['GDNF','GRIP1'],
        ['GDNF','LGALS12'],
        ['GDNF','NCOA2'],
        ['GDPD1','KIAA1738'],
        ['GDPD1','MBD4'],
        ['GDPD1','MED1'],
        ['GDPD1','SMG8'],
        ['GDPD1','TEX2'],
        ['GDPD4','GRM5'],
        ['GDPD4','TADA2A'],
        ['GDPD5','MRPL48'],
        ['GFM1','MFSD1'],
        ['GFM1','MLF1'],
        ['GFPT1','NFU1'],
        ['GGA2','PRKCB'],
        ['GGA3','HN1'],
        ['GGA3','MRPS7'],
        ['GGA3','VRK2'],
        ['GGACT','PCCA'],
        ['GGCT','ZNRF2'],
        ['GGNBP2','ZDHHC16'],
        ['GGNBP2','ZMAT4'],
        ['GHR','OXCT1'],
        ['GID4','TP53'],
        ['GIGYF2','MROH2A'],
        ['GIGYF2','SLC25A23'],
        ['GINM1','SCAF8'],
        ['GIPC1','UBL3'],
        ['GIPC1','WNK1'],
        ['GIPC1','ZNF100'],
        ['GIPC2','SSR3'],
        ['GIPR','PRR12'],
        ['GIT2','SCYL2'],
        ['GIT2','SMYD5'],
        ['GJC3','SMURF1'],
        ['GK5','MBNL1'],
        ['GK5','PAQR9'],
        ['GK5','TFDP2'],
        ['GKAP1','UBQLN1'],
        ['GKN1','TBC1D22A'],
        ['GLA','HNRNPH2'],
        ['GLA','HNRPH2'],
        ['GLB1','INSC'],
        ['GLCCI1','USP42'],
        ['GLCE','NOX5'],
        ['GLCE','SPESP1'],
        ['GLDC','JAK2'],
        ['GLDC','RFX3'],
        ['GLDN','IGDCC3'],
        ['GLE1','ODF2'],
        ['GLE1','SPTAN1'],
        ['GLI3','HIBADH'],
        ['GLI3','SUGCT'],
        ['GLI3','TMEM243'],
        ['GLIPR1L1','RAP1B'],
        ['GLIPR2','NUDT2'],
        ['GLIPR2','RNF38'],
        ['GLIPR2','SLC25A51'],
        ['GLIPR2','ZCCHC7'],
        ['GLIS1','INADL'],
        ['GLIS1','PAX8'],
        ['GLIS2','PHF21A'],
        ['GLIS3','KIAA0020'],
        ['GLIS3','PLGRKT'],
        ['GLIS3','RCL1'],
        ['GLOD4','VPS53'],
        ['GLOD4','YWHAE'],
        ['GLRA4','MORF4L2'],
        ['GLRX3','HECTD2'],
        ['GLS','MYO1B'],
        ['GLS2','MYO1B'],
        ['GLTP','IGF1R'],
        ['GLTP','RP11-113D6.6'],
        ['GLTSCR1L','HIST1H2AC'],
        ['GLTSCR1L','MPHOSPH9'],
        ['GLTSCR1L','UBR2'],
        ['GLUL','SEC63'],
        ['GLUL','UNC13A'],
        ['GLYATL1','TAF6L'],
        ['GLYCTK','NT5DC2'],
        ['GMDS','METTL24'],
        ['GMDS','MRPS18B'],
        ['GMDS','VPS13B'],
        ['GMEB1','TRNAU1AP'],
        ['GMPR','NDUFAF6'],
        ['GMPS','KPNA1'],
        ['GNA13','NR3C1'],
        ['GNA14','RC3H2'],
        ['GNAI3','KIAA1324'],
        ['GNAI3','SARS'],
        ['GNAI3','SARS2'],
        ['GNAL','LAMA3'],
        ['GNAL','LAMA4'],
        ['GNAL','PER3'],
        ['GNAO1','HGS'],
        ['GNAO1','NUP93'],
        ['GNAQ','PRUNE2'],
        ['GNAS','KRT14'],
        ['GNAS','RAD23B'],
        ['GNAZ','TDH'],
        ['GNB1','LRRC45'],
        ['GNB1','NADK'],
        ['GNB1','UBE2J2'],
        ['GNB1L','MED15'],
        ['GNB5','PDE6D'],
        ['GNG3','SLC3A2'],
        ['GNG7','LMNB2'],
        ['GNG7','PLA2G2A'],
        ['GNG7','TACC3'],
        ['GNGT1','KRIT1'],
        ['GNMT','TRERF1'],
        ['GNPAT','TRIM67'],
        ['GNPDA2','GRID2'],
        ['GNPTAB','MAPKAPK5'],
        ['GNS','STK10'],
        ['GOLGA1','KIAA0319L'],
        ['GOLGA1','NR6A1'],
        ['GOLGA4','HSPA8'],
        ['GOLGA4','MLH1'],
        ['GOLGA5','RIN3'],
        ['GOLGA7','RNF170'],
        ['GOLGB1','IQCB1'],
        ['GOLGB1','NCOA1'],
        ['GOLIM4','LPP'],
        ['GOLIM4','NCEH1'],
        ['GOLIM4','SLC40A1'],
        ['GOLPH3','LRTM1'],
        ['GOLPH3L','HORMAD1'],
        ['GON4L','GPR157'],
        ['GON4L','IQGAP3'],
        ['GON4L','LRRC8A'],
        ['GON4L','PIPOX'],
        ['GORASP2','WIPF1'],
        ['GOSR1','JUP'],
        ['GOSR1','TANC2'],
        ['GOSR1','TAOK1'],
        ['GOT1L1','IKBKB'],
        ['GOT1L1','KAT6A'],
        ['GPAA1','KIAA1833'],
        ['GPAA1','MROH1'],
        ['GPATCH1','SPOP'],
        ['GPATCH8','MYO1D'],
        ['GPATCH8','NPLOC4'],
        ['GPATCH8','TMEM132E'],
        ['GPATCH8','TMEM98'],
        ['GPATCH8','ZMYM4'],
        ['GPATCH9','JADE3'],
        ['GPATCH9','KIAA0215'],
        ['GPBP1L1','SCMH1'],
        ['GPC1','MFN1'],
        ['GPC3','GPC4'],
        ['GPC5','GPC6'],
        ['GPC6','PLEKHA5'],
        ['GPCPD1','PLCB1'],
        ['GPD1L','OSBPL10'],
        ['GPD1L','STT3B'],
        ['GPD1L','TMEM135'],
        ['GPD2','WDSUB1'],
        ['GPHN','MPP5'],
        ['GPHN','PLEKHH1'],
        ['GPI','KALRN'],
        ['GPI','KHDC1'],
        ['GPI','NF1'],
        ['GPI','RAD51D'],
        ['GPI','TADA2A'],
        ['GPI','WRB'],
        ['GPLD1','TDP2'],
        ['GPN1','SLC4A1AP'],
        ['GPNMB','NUPL2'],
        ['GPNMB','SP4'],
        ['GPR107','VPS13A'],
        ['GPR110','MEP1A'],
        ['GPR110','TRERF1'],
        ['GPR114','GPR56'],
        ['GPR114','OSBPL5'],
        ['GPR116','MEP1A'],
        ['GPR124','SENP6'],
        ['GPR125','KIAA1109'],
        ['GPR126','PRKCE'],
        ['GPR128','TFG'],
        ['GPR137','PAFAH1B2'],
        ['GPR137','USP32'],
        ['GPR137C','TXNDC16'],
        ['GPR141','ONECUT1'],
        ['GPR143','TBL1X'],
        ['GPR153','ICMT'],
        ['GPR155','SP3'],
        ['GPR156','GSK3B'],
        ['GPR157','RERE'],
        ['GPR158','MALRD1'],
        ['GPR158','MYO16'],
        ['GPR160','PRKCI'],
        ['GPR160','SKIL'],
        ['GPR176','RP11-342D11.2'],
        ['GPR19','LRP6'],
        ['GPR19','SCD'],
        ['GPR19','SLC22A5'],
        ['GPR35','MELK'],
        ['GPR37L1','LGR6'],
        ['GPR37L1','TESK2'],
        ['GPR39','NCKAP5'],
        ['GPR39','ZNF285'],
        ['GPR56','TAOK2'],
        ['GPR68','NUMB'],
        ['GPR98','PJA2'],
        ['GPR98','SERPINA2P'],
        ['GPSM1','NUP214'],
        ['GPSM2','MTF2'],
        ['GPT2','HSD17B2'],
        ['GPX3','SEC63'],
        ['GPX3','SQSTM1'],
        ['GPX4','SEC63'],
        ['GPX5','XRCC6'],
        ['GPX6','RREB1'],
        ['GPX7','RREB1'],
        ['GPX8','TTLL1'],
        ['GRAMD1A','TNFSF14'],
        ['GRAMD1A','ZNF585A'],
        ['GRAMD2','MYO9A'],
        ['GRAMD4','TBC1D22A'],
        ['GRAP2','NF2'],
        ['GRAP2','SLC3A2'],
        ['GRAP2','TBC1D22A'],
        ['GRB10','MAD1L1'],
        ['GRB10','SLC2A13'],
        ['GRB10','STEAP1B'],
        ['GRB2','TCTEX1D2'],
        ['GRB7','PGAP3'],
        ['GRB7','PLXDC1'],
        ['GREB1L','NAPG'],
        ['GRHL2','KCNRG'],
        ['GRHL2','NACAP1'],
        ['GRHL2','NRBP2'],
        ['GRHL2','PAGE3'],
        ['GRHL2','PKDREJ'],
        ['GRHL2','RITA1'],
        ['GRHL2','TRAPPC9'],
        ['GRHL2','TRIM13'],
        ['GRHL2','TSPAN14'],
        ['GRHL2','ZFHX4'],
        ['GRHL3','SLC2A1'],
        ['GRIA1','RAD17'],
        ['GRIA1','SQSTM1'],
        ['GRIA3','STAG2'],
        ['GRIA4','NAALAD2'],
        ['GRID1','HECTD2'],
        ['GRID1','LRP5'],
        ['GRID1','PGPEP1L'],
        ['GRID1','SGMS1'],
        ['GRID1','TRAPPC10'],
        ['GRID2','PDK4'],
        ['GRIK1','HLCS'],
        ['GRIK1','TRAPPC10'],
        ['GRIK3','LRBA'],
        ['GRIK3','MACF1'],
        ['GRIK4','TENM4'],
        ['GRIK4','TMPRSS4'],
        ['GRIK4','UBASH3B'],
        ['GRIK5','ZNF577'],
        ['GRIN2A','MOB1A'],
        ['GRIN2A','SHB'],
        ['GRIN2B','RNF215'],
        ['GRIN2D','RHPN2'],
        ['GRIP1','KDM2A'],
        ['GRIP1','RP11-123O10.4'],
        ['GRIP1','SLC22A23'],
        ['GRIP1','SPTA1'],
        ['GRIP1','TBK1'],
        ['GRIP1','THAP2'],
        ['GRIP2','ITGA9'],
        ['GRK1','PCID2'],
        ['GRK4','HTT'],
        ['GRK4','SLC6A4'],
        ['GRK5','INPP5F'],
        ['GRK5','PDZD8'],
        ['GRK6','RGS7'],
        ['GRK6','UIMC1'],
        ['GRM5','TTC33'],
        ['GRM8','HIPK2'],
        ['GRM8','SND1'],
        ['GRN','NCALD'],
        ['GRXCR1','SLC7A6'],
        ['GSAP','STYXL1'],
        ['GSDMA','THRA'],
        ['GSDMB','STARD3'],
        ['GSDMB','WHSC1'],
        ['GSDMC','RP11-3O20.1'],
        ['GSE1','HSD17B2'],
        ['GSE1','KIAA0513'],
        ['GSE1','METTL23'],
        ['GSE1','NOTCH3'],
        ['GSE1','USP10'],
        ['GSE1','USP32'],
        ['GSK3B','LSAMP'],
        ['GSK3B','MAATS1'],
        ['GSK3B','TIAM1'],
        ['GSN','HLA-DP1B'],
        ['GSN','HLA-DPB1'],
        ['GSN','HLA-F'],
        ['GSN','HSPA8'],
        ['GSPT1','TXNDC11'],
        ['GSR','GTF2E2'],
        ['GSR','PRKDC'],
        ['GSR','TRIM35'],
        ['GSS','NCOA6'],
        ['GSS','SRC'],
        ['GSTA4','NADSYN1'],
        ['GSTCD','RAP1GDS1'],
        ['GSTM3','IQSEC1'],
        ['GSTM4','KCNC4'],
        ['GSTP1','KRT14'],
        ['GSTP1','KRT16'],
        ['GSTP1','KRT6A'],
        ['GSTP1','NADSYN1'],
        ['GSTP1','RP11-116G8.5'],
        ['GSTP1','SF3B2'],
        ['GSX2','SCFD2'],
        ['GTF2A1','KIAA0391'],
        ['GTF2E2','NARS2'],
        ['GTF2E2','RBPMS'],
        ['GTF2F2','NBEA'],
        ['GTF2F2','S100A3'],
        ['GTF2F2','S100A4'],
        ['GTF2H1','HPS5'],
        ['GTF2H1','TNPO1'],
        ['GTF2H3','PGC'],
        ['GTF2IRD1','KMT2C'],
        ['GTF2IRD1','MED13L'],
        ['GTF2IRD1','MLL3'],
        ['GTF3C2','WDR35'],
        ['GTF3C6','LACE1'],
        ['GTF3C6','RPF2'],
        ['GTPBP1','TOMM22'],
        ['GTPBP1','ZMYND8'],
        ['GTPBP2','HIST1H2BJ'],
        ['GTSF1','ITGA5'],
        ['GUCA2A','PTPRF'],
        ['GUCA2B','ROCK1'],
        ['GUCD1','MPDU1'],
        ['GUCY2D','RPA1'],
        ['GUF1','HUWE1'],
        ['GULP1','NUP35'],
        ['GULP1','ZC3H15'],
        ['GXYLT1','LRRK2'],
        ['GXYLT2','NEK3'],
        ['GXYLT2','VAV1'],
        ['GYLTL1B','HIPK3'],
        ['GYS1','HOMER3'],
        ['H19','KRT14'],
        ['H19','KRT16'],
        ['H19','KRT5'],
        ['H19','S100A8'],
        ['H2AFX','WDR18'],
        ['H2BFM','PKNOX1'],
        ['HA-3I','SYNRG'],
        ['HAP1','LIG3'],
        ['HAP2','LIG3'],
        ['HAPLN2','PRCC'],
        ['HARBI1','TNS3'],
        ['HAS2','HHLA1'],
        ['HAUS6','PLIN2'],
        ['HBE1','HBG2'],
        ['HBG2','TRIM22'],
        ['HCFC1','MECP2'],
        ['HCFC1','MGC45419'],
        ['HCFC1','MRFAP1'],
        ['HCFC1','MRX16'],
        ['HCFC1','PNCK'],
        ['HCFC2','TXNRD1'],
        ['HCK','POFUT1'],
        ['HCLC-K2','HUWE1'],
        ['HCN1','RICTOR'],
        ['HCRT','UHMK1'],
        ['HCST','WDR62'],
        ['HDAC1','KPNA6'],
        ['HDAC1','LRRC42'],
        ['HDAC1','PEPD'],
        ['HDAC1','SERINC2'],
        ['HDAC2','TRDN'],
        ['HDAC4','PDE11A'],
        ['HDAC4','PER2'],
        ['HDAC4','SATB2'],
        ['HDAC4','SNX18'],
        ['HDAC7','IRAK4'],
        ['HDAC9','PPL'],
        ['HDAC9','SNX10'],
        ['HDDC2','TPD52L1'],
        ['HDDC3','SLCO3A1'],
        ['HDHD2','SNF8'],
        ['HDLBP','PRAME'],
        ['HDX','RPS6KA6'],
        ['HEATR2','ZNRF3'],
        ['HEATR5B','MYB'],
        ['HEATR5B','NFIB'],
        ['HEATR5B','STRN'],
        ['HEATR6','SH3BP5'],
        ['HEATR6','TOP3A'],
        ['HEATR6','TUBD1'],
        ['HEATR6','WEE1'],
        ['HEATR6','YPEL2'],
        ['HEBP1','ZBTB46'],
        ['HECA','PTPRK'],
        ['HECTD1','PRPF39'],
        ['HECTD2','PCGF5'],
        ['HECTD4','OAS2'],
        ['HECTD4','PEBP1'],
        ['HECTD4','SBNO1'],
        ['HECTD4','SRRM4'],
        ['HECW1','KLHL7'],
        ['HECW1','TRA2A'],
        ['HECW2','LRP1B'],
        ['HECW2','STK17B'],
        ['HELLS','TBC1D12'],
        ['HELZ','PITPNC1'],
        ['HELZ','TTYH2'],
        ['HEMK1','MAPKAPK3'],
        ['HEPHL1','PANX1'],
        ['HERC1','KIAA0101'],
        ['HERC1','USP3'],
        ['HERC4','RUFY2'],
        ['HEXA','MYO9A'],
        ['HEXB','NSA2'],
        ['HEXB','TRIM36'],
        ['HEYL','TOX3'],
        ['HFC1','MECP2'],
        ['HFC1','MGC45419'],
        ['HFC1','MRFAP1'],
        ['HFC1','MRX16'],
        ['HFC1','PNCK'],
        ['HFM1','KRT13'],
        ['HFM1','TMED5'],
        ['HGF','RSBN1L'],
        ['HGS','SLC25A10'],
        ['HGSNAT','MBOAT1'],
        ['HGSNAT','VSTM2B'],
        ['HHAT','KCNT2'],
        ['HHAT','NFAT5'],
        ['HHAT','PRKCZ'],
        ['HHAT','RFWD2'],
        ['HHAT','SERTAD4'],
        ['HHAT','SSU72'],
        ['HHIPL2','SLC30A7'],
        ['HHIPL2','TLR5'],
        ['HHLA1','TMEM64'],
        ['HIBADH','NRIP1'],
        ['HIBADH','SUGCT'],
        ['HIBCH','NAB1'],
        ['HIBCH','PMS1'],
        ['HIC2','TEC'],
        ['HIF1A','MNAT1'],
        ['HIF1A','PRKCH'],
        ['HIF1A','VRK1'],
        ['HINT3','TRMT11'],
        ['HIP1','SLC20A2'],
        ['HIPK1','LRRC42'],
        ['HIPK1','NT5DC1'],
        ['HIPK3','KIF16B'],
        ['HIPK3','MTL5'],
        ['HIPK3','TOLLIP'],
        ['HIRA','LONP2'],
        ['HIRA','NPAS3'],
        ['HIRA','ZBTB16'],
        ['HIST1H2AC','LHFPL2'],
        ['HIST1H2AC','MRPS28'],
        ['HIST1H2AC','MRPS35'],
        ['HIST1H2AC','SLC17A4'],
        ['HIVEP1','MAPK1'],
        ['HIVEP1','OFCC1'],
        ['HIVEP3','PTPRF'],
        ['HIVEP3','ZMYND12'],
        ['HJURP','SNED1'],
        ['HK2','ONECUT3'],
        ['HKR1','PPP2R3A'],
        ['HLA-B','RNASEH2B'],
        ['HLA-C','TCOF1'],
        ['HLA-DRB1','PDK2'],
        ['HLCS','STRA8'],
        ['HLCS','TASP1'],
        ['HLCS','YBEY'],
        ['HM13','TM9SF4'],
        ['HM13','TPX2'],
        ['HMBOX1','SCARA3'],
        ['HMCES','IFT122'],
        ['HMCN1','PRKAR2B'],
        ['HMCN1','RGL1'],
        ['HMCN1','RNF2'],
        ['HMCN1','TMCC2'],
        ['HMGA2','MSRB3'],
        ['HMGA2','NUP107'],
        ['HMGA2','PCBP2'],
        ['HMGA2','RP11-396F22.1'],
        ['HMGA2','RPSAP52'],
        ['HMGA2','SENP1'],
        ['HMGA2','TSFM'],
        ['HMGCLL1','TEC'],
        ['HMGCLL1','TMEM181'],
        ['HMGCS1','OXCT1'],
        ['HMGXB4','TOM1'],
        ['HMHA1','STK11'],
        ['HMHB1','NR3C1'],
        ['HMOX1','MCM5'],
        ['HN1L','STK10'],
        ['HNF4A','SLCO2B1'],
        ['HNRNPA2B1','RABEP1'],
        ['HNRNPA2B1','SIAE'],
        ['HNRNPAB','SAMM50'],
        ['HNRNPF','ZNF569'],
        ['HNRNPH1','RNF130'],
        ['HNRNPH2','NAT8'],
        ['HNRNPL','SIRT2'],
        ['HNRNPM','PC'],
        ['HNRNPM','PODXL'],
        ['HNRNPM','SLC25A23'],
        ['HNRNPU','KRT10'],
        ['HNRNPU','KRT13'],
        ['HNRNPUL2','TTC9C'],
        ['HNRPH2','NAT8'],
        ['HOMER1','MAST2'],
        ['HOMER2','ORMDL3'],
        ['HOMER2','SKAP1'],
        ['HOMER2','TBX2'],
        ['HOOK2','MAST1'],
        ['HOOK2','ONECUT3'],
        ['HOOK3','IKBKB'],
        ['HOOK3','MTFR1'],
        ['HOOK3','PLEKHA2'],
        ['HOOK3','RNF170'],
        ['HOOK3','TM2D2'],
        ['HOPX','NMU'],
        ['HOPX','PDCL2'],
        ['HORMAD1','XPO1'],
        ['HOXA3','KIAA0895'],
        ['HOXB3','RARA'],
        ['HOXB3','SKAP1'],
        ['HOXC10','WIBG'],
        ['HOXC6','WIBG'],
        ['HP1BP3','TMCO4'],
        ['HPN','PPP2R5C'],
        ['HPRT1','LRRC2'],
        ['HPS1','UBE2Q1'],
        ['HPS3','JADE1'],
        ['HPS4','KIF5B'],
        ['HPS5','SETD5'],
        ['HPSE2','PRKD2'],
        ['HRAS','RNASEH1P1'],
        ['HRAS','RNH1'],
        ['HRASLS5','RTN3'],
        ['HRSP12','RNF19A'],
        ['HS1BP3','MTMR1'],
        ['HS1BP3','PUM2'],
        ['HS1BP3','TACC2'],
        ['HS2ST1','MICALCL'],
        ['HS2ST1','NXNL2'],
        ['HS2ST1','PRDX1'],
        ['HS3ST4','NFATC2IP'],
        ['HS3ST5','VPS53'],
        ['HS3ST6','VPS53'],
        ['HS6ST2','KIAA1755'],
        ['HS6ST2','NPFFR2'],
        ['HSA238520','USP8'],
        ['HSBP1L1','PQLC1'],
        ['HSD11B1','JAK1'],
        ['HSD11B1L','SAFB'],
        ['HSD17B12','TP53I11'],
        ['HSD17B4','SLIT1'],
        ['HSD17B4','SLIT3'],
        ['HSD17B4','STX8'],
        ['HSD3B1','JARID2'],
        ['HSD3B2','NF2'],
        ['HSF2BP','ZNF345'],
        ['HSF5','NOL11'],
        ['HSH2D','LDLR'],
        ['HSP70-2','SLC1A3'],
        ['HSP90AA1','MOK'],
        ['HSP90AA1','RAGE'],
        ['HSP90AB1','SLC29A1'],
        ['HSP90AB1','TAF3'],
        ['HSPA12A','KIAA1598'],
        ['HSPA12B','YPEL2'],
        ['HSPA13','RSF1'],
        ['HSPA14','MRPS28'],
        ['HSPA14','MRPS35'],
        ['HSPA1B','SLC1A3'],
        ['HSPA1L','TNS3'],
        ['HSPA4','KRT6A'],
        ['HSPB1','JUP'],
        ['HSPB1','KRT14'],
        ['HSPB1','KRT17'],
        ['HSPB1','MYO1B'],
        ['HSPB1','PDE3A'],
        ['HSPB1','PI3'],
        ['HSPB1','SFN'],
        ['HSPB11','ZCCHC11'],
        ['HSPC099','S100A2'],
        ['HSPC099','SHC1'],
        ['HSPC099','VPS45'],
        ['HSPC099','VPS45A'],
        ['HSPD1','HSPE1'],
        ['HSPG2','KIAA0247'],
        ['HSPG2','KRT6A'],
        ['HSPG2','NTNG1'],
        ['HSPG2','NTNG2'],
        ['HSPG2','SIAE'],
        ['HSPG2','UBR4'],
        ['HSPH1','KIAA0825'],
        ['HSPH1','PLEKHA5'],
        ['HTN3','KCNE2'],
        ['HTR2A','MED4'],
        ['HTR3D','MSN'],
        ['HTRA3','PELI3'],
        ['HTRA3','TBC1D14'],
        ['HTRA4','RAB11FIP1'],
        ['HTT','TBC1D5'],
        ['HUNK','MAMLD1'],
        ['HUNK','SIK1'],
        ['HUNK','SNRPD3'],
        ['HUWE1','KIAA1111'],
        ['HUWE1','PHF8'],
        ['HVCN1','PPTC7'],
        ['HYI','KDM4A'],
        ['HYI','NEDD4L'],
        ['HYI','NFIA'],
        ['IAPP','LPCAT1'],
        ['IAPP','SNX9'],
        ['ICA1','SUN1'],
        ['ICA1L','PTK2B'],
        ['ICAM1','SMARCA4'],
        ['ICAM2','MSI2'],
        ['ICK','TRMT11'],
        ['ICT1','KIAA0195'],
        ['ICT1','SAP30BP'],
        ['ICT1','SLC16A5'],
        ['IDE','VWA2'],
        ['IDO1','PTPRD'],
        ['IFFO2','UBR4'],
        ['IFI16','MTR'],
        ['IFI44L','SLC44A3'],
        ['IFI6','STPG1'],
        ['IFLTD1','KRAS'],
        ['IFLTD1','MBD4'],
        ['IFLTD1','MED1'],
        ['IFLTD1','ST8SIA1'],
        ['IFNL1','TSHZ3'],
        ['IFNL1','URI1'],
        ['IFNLR1','STPG1'],
        ['IFT140','TCEB3'],
        ['IFT20','RAI1'],
        ['IFT43','TTLL10'],
        ['IFT52','TRIO'],
        ['IFT74','LDLRAD3'],
        ['IFT74','WNK2'],
        ['IFT88','IL17D'],
        ['IFT88','RUNX2'],
        ['IFT88','WASF3'],
        ['IFT88','ZMYM5'],
        ['IGF1R','MFGE8'],
        ['IGF1R','PPP4R1'],
        ['IGF1R','TOLLIP'],
        ['IGF1R','TRIP4'],
        ['IGF1R','WDR93'],
        ['IGF2','MCM8'],
        ['IGF2','SNW1'],
        ['IGF2','WNK2'],
        ['IGF2BP1','KAT7'],
        ['IGF2BP1','PRKCA'],
        ['IGF2BP1','RAI1'],
        ['IGF2BP1','ZBTB8OS'],
        ['IGF2BP2','NCEH1'],
        ['IGF2BP3','PRKCA'],
        ['IGF2BP3','RPS6KA2'],
        ['IGF2BP3','THADA'],
        ['IGF2R','PITRM1'],
        ['IGF2R','PREP'],
        ['IGF2R','SYTL3'],
        ['IGF2R','WTAP'],
        ['IGFBP3','LIMK1'],
        ['IGFBP4','SLC35B1'],
        ['IGFBP5','SIAE'],
        ['IGFN1','KANK1'],
        ['IGLL1','SNAP29'],
        ['IGSF11','MFI2'],
        ['IGSF11','PBRM1'],
        ['IGSF21','OBSCN'],
        ['IGSF23','PPP1R37'],
        ['IGSF23','RELB'],
        ['IGSF23','ZNF675'],
        ['IGSF3','KDM2A'],
        ['IGSF3','MAN1A2'],
        ['IGSF3','TBX15'],
        ['IGSF8','USPL1'],
        ['IGSF9','SLAMF9'],
        ['IKBKAP','TBC1D2'],
        ['IKBKB','KAT6A'],
        ['IKBKB','UNC5D'],
        ['IKBKE','RAPL'],
        ['IKBKE','RASSF5'],
        ['IKZF1','LRBA'],
        ['IKZF3','LUC7L3'],
        ['IKZF3','SKAP1'],
        ['IKZF3','SMURF2'],
        ['IL10RA','KIAA0638'],
        ['IL10RA','PHLDB1'],
        ['IL12RB1','TICAM1'],
        ['IL12RB2','PDE4B'],
        ['IL13RA1','WDR44'],
        ['IL17F','USP38'],
        ['IL18R1','MCOLN2'],
        ['IL1R1','IL1R2'],
        ['IL1R2','IL1RL2'],
        ['IL1RAP','TMEM207'],
        ['IL1RAP','TP63'],
        ['IL1RAP','TPRG1'],
        ['IL1RAPL1','MAGEB10'],
        ['IL1RAPL2','MYO3B'],
        ['IL1RN','KRT16'],
        ['IL1RN','PUM2'],
        ['IL1RN','SFN'],
        ['IL20RB','NCK1'],
        ['IL20RB','NR4A1'],
        ['IL22RA1','RNF213'],
        ['IL27RA','RLN3'],
        ['IL29','TSHZ3'],
        ['IL2RB','WIPF2'],
        ['IL32','NAA60'],
        ['IL34','ST3GAL2'],
        ['IL4I1','KCNQ2'],
        ['IL4R','NSMCE1'],
        ['IL6','RSBN1L'],
        ['ILDR1','IQCB1'],
        ['ILDR2','PBX1'],
        ['ILKAP','PER2'],
        ['ILVBL','SYDE1'],
        ['ILWEQ','SLC20A2'],
        ['IMMP2L','LMTK2'],
        ['IMMP2L','MKLN1'],
        ['IMMP2L','NEDD4L'],
        ['IMMP2L','PILRB'],
        ['IMMP2L','SMC5'],
        ['IMMT','SLC25A26'],
        ['IMPA1','ZNF704'],
        ['IMPA2','LDLRAD4'],
        ['IMPAD1','ZCCHC24'],
        ['INADL','MACF1'],
        ['INADL','NFIA'],
        ['INADL','PHIP'],
        ['INADL','PRSS50'],
        ['INADL','ST6GALNAC5'],
        ['INO80','PRKCA'],
        ['INO80','TMBIM4'],
        ['INPP4A','MGAT4A'],
        ['INPP4B','KIAA0232'],
        ['INPP4B','SH3D19'],
        ['INPP5A','JAKMIP3'],
        ['INPP5A','KIF20B'],
        ['INPP5A','MPP2'],
        ['INPP5A','TTC40'],
        ['INPP5D','TUBD1'],
        ['INPP5E','SEC16A'],
        ['INSL3','MFSD2A'],
        ['INSR','MAP2K2'],
        ['INSR','MTMR4'],
        ['INSR','RBFOX3'],
        ['INSR','S1PR2'],
        ['INT2','LGR4'],
        ['INTS12','MAK'],
        ['INTS2','LIMD2'],
        ['INTS4','KCTD14'],
        ['INTS4','NARS2'],
        ['INTS4','POLD3'],
        ['INTS4','PPFIA1'],
        ['INTS4','RSF1'],
        ['INTS4','TENM4'],
        ['INTS4','UVRAG'],
        ['INTS5','SYT12'],
        ['INTS7','NEK7'],
        ['INTS8','RP11-219B4.5'],
        ['IP6K1','TRAIP'],
        ['IP6K2','PRKAR2A'],
        ['IPO11','TNFAIP8'],
        ['IPO7','TMEM41B'],
        ['IQCE','VIM'],
        ['IQCJ-SCHIP1','NCAPG2'],
        ['IQCK','KIAA0430'],
        ['IQGAP1','MESP2'],
        ['IQGAP1','SCGB1D2'],
        ['IQGAP1','TTLL13'],
        ['IQGAP1','UNC45A'],
        ['IQGAP1','ZNF774'],
        ['IQGAP2','KIAA0534'],
        ['IQGAP2','PWWP2A'],
        ['IQGAP3','SH2D2A'],
        ['IQSEC1','WNT7A'],
        ['IQSEC3','MAPKAPK2'],
        ['IQSEC3','WNK1'],
        ['IRAK1','LYZ'],
        ['IRAK1','MECP2'],
        ['IRAK1','MRX16'],
        ['IRAK1','PIGR'],
        ['IRAK2','SRGAP3'],
        ['IRAK2','TATDN2'],
        ['IRAK3','TLK2'],
        ['IRF2','ZNF804A'],
        ['IRF2BP2','NTRK1'],
        ['IRF3','ZNF814'],
        ['IRF4','SH3GL1'],
        ['IRF5','MLL3'],
        ['IRG1','MBD5'],
        ['ISM1','RNF24'],
        ['ISX','RNF185'],
        ['ISY1','MASP1'],
        ['ISY1','PMEPA1'],
        ['ISY1','STAG1'],
        ['ITCH','MYL9'],
        ['ITCH','NIPBL'],
        ['ITCH','PARD3'],
        ['ITCH','RALY'],
        ['ITCH','TAB1'],
        ['ITCH','TSHZ2'],
        ['ITFG1','PHKB'],
        ['ITFG1','RYK'],
        ['ITFG1','TOX3'],
        ['ITFG2','RIMKLB'],
        ['ITFG3','LUC7L'],
        ['ITGA2','NDUFS4'],
        ['ITGA3','MED13'],
        ['ITGA3','MIEN1'],
        ['ITGA3','RP11-1094H24.3'],
        ['ITGA4','UBE2E2'],
        ['ITGA9','SEC13'],
        ['ITGA9','XYLB'],
        ['ITGAV','PDE11A'],
        ['ITGAX','PHKG2'],
        ['ITGB1BP1','MBOAT2'],
        ['ITGB4','MYO15B'],
        ['ITGB4','NARF'],
        ['ITGB4','OTOP2'],
        ['ITGB4','RNF138'],
        ['ITGB4','SAP30BP'],
        ['ITGB4','SLC16A5'],
        ['ITGB4','TSPAN10'],
        ['ITGB5','NCOR2'],
        ['ITGB5','PARP14'],
        ['ITGB6','PLA2R1'],
        ['ITGB6','RBMS1'],
        ['ITGB6','STK39'],
        ['ITLN1','PIP5K1A'],
        ['ITM2B','MATR3'],
        ['ITM2B','RB1'],
        ['ITM2C','MSF1'],
        ['ITM2C','SEPT9'],
        ['ITPK1','SYNE2'],
        ['ITPKB','ZC3HAV1'],
        ['ITPR1','SETMAR'],
        ['ITPR1','SF1'],
        ['ITPR1','SUMF1'],
        ['ITPR2','RASSF3'],
        ['ITSN1','KIAA1211'],
        ['ITSN1','NEBL'],
        ['ITSN2','KIDINS220'],
        ['ITSN2','NMT1'],
        ['IVL','KRT14'],
        ['IVL','KRT6A'],
        ['IWS1','MAP3K2'],
        ['IZUMO2','ZNF569'],
        ['IZUMO3','MLLT3'],
        ['JADE3','RBM10'],
        ['JAK1','PDE4B'],
        ['JAK1','ZC3H7B'],
        ['JAK2','KDM4C'],
        ['JAK2','SIK2'],
        ['JAK2','TTC13'],
        ['JAKMIP1','LUC7L2'],
        ['JAM3','MAP4K5'],
        ['JARID1C','PPP4R1'],
        ['JARID2','KCTD20'],
        ['JARID2','PNUTS'],
        ['JARID2','PPP1R10'],
        ['JAZF1','MSRB3'],
        ['JAZF1','RNF216'],
        ['JAZF1','SNX13'],
        ['JAZF1','TAX1BP1'],
        ['JDP2','VRTN'],
        ['JMJD1C','NRBF2'],
        ['JMJD6','MSF1'],
        ['JMJD6','SEPT9'],
        ['JMJD6','ST6GALNAC2'],
        ['JMJD7','SPG11'],
        ['JMJD7-PLA2G4B','SPG11'],
        ['JPH3','ZNF276'],
        ['JUNB','SEC63'],
        ['JUP','KRT10'],
        ['JUP','KRT13'],
        ['JUP','KRT14'],
        ['JUP','KRT15'],
        ['JUP','KRT16'],
        ['JUP','KRT17P1'],
        ['JUP','KRT19'],
        ['JUP','KRT5'],
        ['JUP','KRT6A'],
        ['JUP','KRT6B'],
        ['JUP','KRT6C'],
        ['JUP','LEPREL4'],
        ['JUP','LY6D'],
        ['JUP','LYPD3'],
        ['JUP','PDE3A'],
        ['JUP','PPIB'],
        ['JUP','RP11-363E6.3'],
        ['JUP','RP4-541C22.4'],
        ['JUP','RPL11'],
        ['JUP','RPL13'],
        ['JUP','S100A9'],
        ['JUP','SBSN'],
        ['JUP','SFN'],
        ['JUP','SNX22'],
        ['JUP','SPRR3'],
        ['JUP','TGFBI'],
        ['JUP','TPM2'],
        ['JUP','TPT1'],
        ['KA36','MED22'],
        ['KA36','MED24'],
        ['KAL1','KDM6A'],
        ['KALRN','NLK'],
        ['KALRN','UMPS'],
        ['KANK1','TPD52L3'],
        ['KANK1','UHRF2'],
        ['KANK3','SPC24'],
        ['KANSL1','LAYN'],
        ['KANSL1','ORMDL3'],
        ['KANSL1','UNC45B'],
        ['KAT2A','STAT5B'],
        ['KAT6A','KLK4'],
        ['KAT6A','MSR1'],
        ['KAT6A','RAB11FIP1'],
        ['KAT6B','RTKN2'],
        ['KAT6B','SAMD8'],
        ['KATNA1','LATS1'],
        ['KATNA1','PPP2R1A'],
        ['KATNAL1','WASF3'],
        ['KATNAL2','TMCO3'],
        ['KAZN','MTOR'],
        ['KAZN','PTBP2'],
        ['KAZN','ST3GAL3'],
        ['KAZN','TMEM51'],
        ['KAZN','TRPM7'],
        ['KCMF1','PRKDC'],
        ['KCNA5','ONECUT3'],
        ['KCNAB1','MLLT4'],
        ['KCNAB1','PHLDB2'],
        ['KCNAB1','PIK3CB'],
        ['KCNAB1','SLC9A9'],
        ['KCNAB2','KIF1B'],
        ['KCNAB2','POLB'],
        ['KCNB2','LRP12'],
        ['KCNC2','MARCH9'],
        ['KCNC2','ROR2'],
        ['KCNC2','RORA'],
        ['KCNE1','RPS19BP1'],
        ['KCNE3','UVRAG'],
        ['KCNH1','LPGAT1'],
        ['KCNH5','LRR1'],
        ['KCNH8','LGALS3BP'],
        ['KCNH8','REXO2'],
        ['KCNH8','SFN'],
        ['KCNIP1','SLC38A9'],
        ['KCNIP4','ZNF430'],
        ['KCNJ12','MMP28'],
        ['KCNJ15','WRB'],
        ['KCNJ16','TLK2'],
        ['KCNJ2','MED13'],
        ['KCNJ4','MAPK1'],
        ['KCNJ6','MLLT10'],
        ['KCNJ6','SIM2'],
        ['KCNJN1','MMP28'],
        ['KCNK3','ZNF512'],
        ['KCNK5','KIF6'],
        ['KCNK5','PDE4DIP'],
        ['KCNK6','LMBR1L'],
        ['KCNMA1','MSMB'],
        ['KCNMA1','SHFM1'],
        ['KCNMB1','LMBRD2'],
        ['KCNN1','SLC27A1'],
        ['KCNN4','SMG9'],
        ['KCNQ1','RP11-22P4.1'],
        ['KCNQ1','TSPAN32'],
        ['KCNRG','RB1'],
        ['KCNT1','TRUB2'],
        ['KCNT2','SKI'],
        ['KCNU1','PLEKHA2'],
        ['KCNU1','RAB11FIP1'],
        ['KCTD1','PTPRM'],
        ['KCTD10','TPCN1'],
        ['KCTD11','PIAS3'],
        ['KCTD11','ZMIZ5'],
        ['KCTD14','PAK1'],
        ['KCTD14','PKN1'],
        ['KCTD14','THRSP'],
        ['KCTD18','SNED1'],
        ['KCTD2','NUP85'],
        ['KCTD2','TMEM104'],
        ['KCTD20','USP49'],
        ['KCTD20','ZFAND3'],
        ['KCTD21','PAK1'],
        ['KCTD21','RSF1'],
        ['KCTD5','SRRM2'],
        ['KCTD8','TOMM70A'],
        ['KDELR2','SNX8'],
        ['KDM2A','LGALS12'],
        ['KDM2A','MAP3K3'],
        ['KDM2A','NCOA2'],
        ['KDM2A','PACS1'],
        ['KDM2A','PCF11'],
        ['KDM2A','PPP2R5B'],
        ['KDM2A','PPP6R3'],
        ['KDM2A','RHOD'],
        ['KDM2A','SYT12'],
        ['KDM2B','KSR2'],
        ['KDM2B','SETD1B'],
        ['KDM3A','PUS10'],
        ['KDM3A','RFX8'],
        ['KDM3B','PRLR'],
        ['KDM4A','PTPRF'],
        ['KDM4A','RYBP'],
        ['KDM4B','MPND'],
        ['KDM4B','SLC25A42'],
        ['KDM4B','STK11'],
        ['KDM4B','UHRF1'],
        ['KDM4C','KLK6'],
        ['KDM4C','KLK7'],
        ['KDM4C','PGPEP1'],
        ['KDM4C','RFX3'],
        ['KDM5A','NINJ2'],
        ['KDM5A','RAE1'],
        ['KDM5A','WNK1'],
        ['KDM5B','LIN9'],
        ['KDM5B','RABGAP1L'],
        ['KDM5B','UAP1'],
        ['KDM5C','PPP4R1'],
        ['KDM6A','METTL20'],
        ['KDM6A','PDHX'],
        ['KDM6A','RPS6KA3'],
        ['KDM6A','USP9X'],
        ['KDM6A','ZNF674'],
        ['KDM7A','SND1'],
        ['KEL','TRPV6'],
        ['KHDC1','NLK'],
        ['KHDC1','OOEP'],
        ['KHDRBS2','LMBRD1'],
        ['KHSRP','PRKAB2'],
        ['KIAA0038','WBS'],
        ['KIAA0100','SEZ6'],
        ['KIAA0100','TEX14'],
        ['KIAA0101','TRIP4'],
        ['KIAA0124','NIPAL2'],
        ['KIAA0146','SNX7'],
        ['KIAA0182','SLC7A5'],
        ['KIAA0195','NUP85'],
        ['KIAA0195','TTYH2'],
        ['KIAA0215','RBM10'],
        ['KIAA0247','RAD51B'],
        ['KIAA0247','TXNDC16'],
        ['KIAA0319L','PHC2'],
        ['KIAA0355','WTIP'],
        ['KIAA0368','MRRF'],
        ['KIAA0368','PACS1'],
        ['KIAA0368','WNK2'],
        ['KIAA0391','SYNE2'],
        ['KIAA0417','KIAA1598'],
        ['KIAA0556','NECAB2'],
        ['KIAA0586','MPP5'],
        ['KIAA0602','MTA1'],
        ['KIAA0825','PGGT1B'],
        ['KIAA0825','RNPC3'],
        ['KIAA0907','ZC3H3'],
        ['KIAA0922','MND1'],
        ['KIAA0922','OVOL2'],
        ['KIAA1001','KIAA1738'],
        ['KIAA1001','TEX2'],
        ['KIAA1001','TMCC1'],
        ['KIAA1033','LGR5'],
        ['KIAA1033','PCDH9'],
        ['KIAA1110','WNK1'],
        ['KIAA1211','NMU'],
        ['KIAA1211','PUM1'],
        ['KIAA1211','SCFD2'],
        ['KIAA1211','STMN1'],
        ['KIAA1211L','MGAT4A'],
        ['KIAA1217','PDE7A'],
        ['KIAA1217','PFKFB3'],
        ['KIAA1217','SPAG6'],
        ['KIAA1324','SARS'],
        ['KIAA1324','SARS2'],
        ['KIAA1324L','SRPK2'],
        ['KIAA1377','PANX1'],
        ['KIAA1377','YAP1'],
        ['KIAA1407','MKRN2'],
        ['KIAA1429','PLRP1'],
        ['KIAA1429','PNLIPRP1'],
        ['KIAA1430','SIAH2'],
        ['KIAA1430','VHL'],
        ['KIAA1468','PTPRM'],
        ['KIAA1468','TNFRSF11A'],
        ['KIAA1501','MBD4'],
        ['KIAA1501','MED1'],
        ['KIAA1501','TEX14'],
        ['KIAA1549L','WARS2'],
        ['KIAA1551','SOX5'],
        ['KIAA1598','PRDX3'],
        ['KIAA1671','PPIL2'],
        ['KIAA1684','NCKAP4'],
        ['KIAA1684','PCYT2'],
        ['KIAA1684','SOCS4'],
        ['KIAA1684','SOCS7'],
        ['KIAA1731','WDR90'],
        ['KIAA1738','MED13'],
        ['KIAA1738','MSI2'],
        ['KIAA1738','MYO1D'],
        ['KIAA1738','RPS6KB1'],
        ['KIAA1738','TTYH2'],
        ['KIAA1755','UQCC1'],
        ['KIAA1833','TCTEX1D2'],
        ['KIAA1833','ZNF623'],
        ['KIAA1875','PCMTD1'],
        ['KIAA1958','SNX30'],
        ['KIAA1958','SNX33'],
        ['KIAA1958','WDR34'],
        ['KIAA1982','RFPL3S'],
        ['KIAA2022','PSMD12'],
        ['KIF13A','PTPRK'],
        ['KIF13A','TPMT'],
        ['KIF13B','TENM4'],
        ['KIF13B','TMEM66'],
        ['KIF14','RFWD2'],
        ['KIF15','ZKSCAN7'],
        ['KIF16B','RRBP1'],
        ['KIF1A','TRIP12'],
        ['KIF1B','MYO5B'],
        ['KIF1B','TTLL1'],
        ['KIF1B','UBE4B'],
        ['KIF21A','LMF1'],
        ['KIF21A','SLC2A13'],
        ['KIF21A','TMTC2'],
        ['KIF21A','TSPAN8'],
        ['KIF22','MVP'],
        ['KIF22','PHLPP2'],
        ['KIF26A','MARK3'],
        ['KIF26B','SMYD3'],
        ['KIF26B','TEF'],
        ['KIF2A','ONECUT3'],
        ['KIF3A','NDRG2'],
        ['KIF3B','NINL'],
        ['KIF3C','RBKS'],
        ['KIF5A','OS9'],
        ['KIF5B','MET'],
        ['KIF5B','SLTM'],
        ['KIF9','SETD2'],
        ['KIFAP3','SCYL3'],
        ['KIFAP3','XPR1'],
        ['KIFC1','RPS6KA2'],
        ['KIFC2','NPRL3'],
        ['KIRREL','RBP7'],
        ['KIRREL2','MAPK8'],
        ['KIRREL3','MLLT6'],
        ['KIRREL3','RP11-680F20.9'],
        ['KIRREL3','UVRAG'],
        ['KISS1','PPP1R12C'],
        ['KIT','PDGFRA'],
        ['KIT','SLC4A4'],
        ['KIT','UGT2B7'],
        ['KITLG','TNNC1'],
        ['KITLG','TNNI3'],
        ['KLB','LIAS'],
        ['KLC2','RSF1'],
        ['KLF3','SCFD2'],
        ['KLF7','PARD3B'],
        ['KLHDC4','SPG7'],
        ['KLHDC8A','PCMTD1'],
        ['KLHL15','ZFX'],
        ['KLHL18','TEX264'],
        ['KLHL22','RPGRIP1L'],
        ['KLHL22','ZNF74'],
        ['KLHL26','KXD1'],
        ['KLHL32','MED23'],
        ['KLHL42','MRPS28'],
        ['KLHL42','MRPS35'],
        ['KLHL5','LNX1'],
        ['KLHL5','SLC16A2'],
        ['KLK3','NUAK1'],
        ['KLK3','ZC3H3'],
        ['KLKB1','ZC3H3'],
        ['KMT2A','MLLT10'],
        ['KMT2A','MLLT3'],
        ['KMT2A','MLLT4'],
        ['KMT2A','SLC22A10'],
        ['KMT2A','ZDHHC7'],
        ['KMT2B','LMBR1L'],
        ['KMT2B','RHEBL1'],
        ['KMT2C','LMTK3'],
        ['KMT2C','PRKAG2'],
        ['KMT2C','RBM28'],
        ['KMT2C','SYTL5'],
        ['KMT2C','XRCC2'],
        ['KMT2D','LMBR1L'],
        ['KMT2D','RHEBL1'],
        ['KMT2E','LHFPL3'],
        ['KNG1','ST6GAL1'],
        ['KNTC1','MLXIP'],
        ['KPNA6','RHCE'],
        ['KPNA6','TMEM130'],
        ['KPNA7','TMEM130'],
        ['KPNB1','SKAP1'],
        ['KRAS','RERGL'],
        ['KRBOX1','POMGNT2'],
        ['KRT1','KRT10'],
        ['KRT1','KRT13'],
        ['KRT1','KRT14'],
        ['KRT1','KRT15'],
        ['KRT1','KRT16'],
        ['KRT1','KRT5'],
        ['KRT1','KRT6A'],
        ['KRT1','KRT6B'],
        ['KRT1','KRT77'],
        ['KRT10','KRT13'],
        ['KRT10','KRT14'],
        ['KRT10','KRT16'],
        ['KRT10','KRT17'],
        ['KRT10','KRT5'],
        ['KRT10','KRT6B'],
        ['KRT10','RP11-116G8.5'],
        ['KRT10','TANC2'],
        ['KRT12','KRT14'],
        ['KRT13','KRT14'],
        ['KRT13','KRT15'],
        ['KRT13','KRT16'],
        ['KRT13','KRT17'],
        ['KRT13','KRT19'],
        ['KRT13','KRT23'],
        ['KRT13','KRT25'],
        ['KRT13','KRT4'],
        ['KRT13','KRT5'],
        ['KRT13','KRT6A'],
        ['KRT13','KRT6B'],
        ['KRT13','KRT75'],
        ['KRT13','LGALS1'],
        ['KRT13','LY6D'],
        ['KRT13','PCBD2'],
        ['KRT13','PDE3A'],
        ['KRT13','PDXK'],
        ['KRT13','PGD'],
        ['KRT13','PI3'],
        ['KRT13','PKP1'],
        ['KRT13','RP11-1020M18.10'],
        ['KRT13','RP11-116G8.5'],
        ['KRT13','RP11-762I7.5'],
        ['KRT13','RRP1B'],
        ['KRT13','RSPO1'],
        ['KRT13','S100A2'],
        ['KRT13','SARNP'],
        ['KRT13','SFN'],
        ['KRT13','SLC2A1'],
        ['KRT13','SPARC'],
        ['KRT13','SPRR1B'],
        ['KRT13','SPRR2E'],
        ['KRT13','TGM3'],
        ['KRT13','TPT1'],
        ['KRT14','KRT15'],
        ['KRT14','KRT16'],
        ['KRT14','KRT17'],
        ['KRT14','KRT19'],
        ['KRT14','KRT3'],
        ['KRT14','KRT4'],
        ['KRT14','KRT42P'],
        ['KRT14','KRT5'],
        ['KRT14','KRT6A'],
        ['KRT14','KRT6B'],
        ['KRT14','KRT75'],
        ['KRT14','KRT9'],
        ['KRT14','LYPD3'],
        ['KRT14','MYH9'],
        ['KRT14','NDRG1'],
        ['KRT14','PDE3A'],
        ['KRT14','PGK1'],
        ['KRT14','PLEC'],
        ['KRT14','PPIB'],
        ['KRT14','PSAP'],
        ['KRT14','RP11-1020M18.10'],
        ['KRT14','RP11-116G8.5'],
        ['KRT14','RP11-363E6.3'],
        ['KRT14','RPL18'],
        ['KRT14','RPL7A'],
        ['KRT14','RPL8'],
        ['KRT14','RPLP2'],
        ['KRT14','S100A8'],
        ['KRT14','S100A9'],
        ['KRT14','SBSN'],
        ['KRT14','SDC1'],
        ['KRT14','SERINC2'],
        ['KRT14','SFN'],
        ['KRT14','SLC35C1'],
        ['KRT14','SNX22'],
        ['KRT14','SPARC'],
        ['KRT14','SPRR1A'],
        ['KRT14','SPRR1B'],
        ['KRT14','SPRR2A'],
        ['KRT14','SPRR2E'],
        ['KRT14','TMSB10'],
        ['KRT14','TUBB2C'],
        ['KRT14','YIF1B'],
        ['KRT15','KRT16'],
        ['KRT15','KRT17'],
        ['KRT15','KRT19'],
        ['KRT15','KRT6A'],
        ['KRT16','KRT17'],
        ['KRT16','KRT19'],
        ['KRT16','KRT24'],
        ['KRT16','KRT42P'],
        ['KRT16','KRT5'],
        ['KRT16','KRT6A'],
        ['KRT16','KRT6B'],
        ['KRT16','KRT6C'],
        ['KRT16','LCN2'],
        ['KRT16','PDE3A'],
        ['KRT16','PI3'],
        ['KRT16','PKM2'],
        ['KRT16','RPL36'],
        ['KRT16','RPL8'],
        ['KRT16','S100A14'],
        ['KRT16','S100A8'],
        ['KRT16','SFN'],
        ['KRT16','SPRR1A'],
        ['KRT16','SPRR1B'],
        ['KRT16','SPRR2A'],
        ['KRT17','KRT17P1'],
        ['KRT17','KRT19'],
        ['KRT17','KRT5'],
        ['KRT17','KRT6A'],
        ['KRT17','KRT6B'],
        ['KRT17','KRT6C'],
        ['KRT17','LY6D'],
        ['KRT17','LYPD3'],
        ['KRT17','PDE3A'],
        ['KRT17','PPIB'],
        ['KRT17','RP11-363E6.3'],
        ['KRT17','RP4-541C22.4'],
        ['KRT17','RPL11'],
        ['KRT17','RPL13'],
        ['KRT17','RPL8'],
        ['KRT17','S100A9'],
        ['KRT17','SBSN'],
        ['KRT17','SFN'],
        ['KRT17','SNX22'],
        ['KRT17','SPRR3'],
        ['KRT17','TGFBI'],
        ['KRT17','TPM2'],
        ['KRT17','TPT1'],
        ['KRT18','MSRB3'],
        ['KRT19','KRT35'],
        ['KRT2','KRT6A'],
        ['KRT222','MBD4'],
        ['KRT222','MED1'],
        ['KRT23','PHOSPHO1'],
        ['KRT3','KRT5'],
        ['KRT3','KRT6A'],
        ['KRT3','KRT6B'],
        ['KRT32','MSF1'],
        ['KRT32','SEPT9'],
        ['KRT33A','SYNRG'],
        ['KRT4','KRT5'],
        ['KRT4','KRT6A'],
        ['KRT4','KRT6B'],
        ['KRT4','S100A8'],
        ['KRT4','SFN'],
        ['KRT40','MED22'],
        ['KRT40','MED24'],
        ['KRT42P','KRT5'],
        ['KRT5','KRT6A'],
        ['KRT5','KRT6B'],
        ['KRT5','KRT6C'],
        ['KRT5','KRT71'],
        ['KRT5','KRT75'],
        ['KRT5','KRT78'],
        ['KRT5','KRT8'],
        ['KRT5','KRT82'],
        ['KRT5','MT2A'],
        ['KRT5','MYH9'],
        ['KRT5','NASP'],
        ['KRT5','NDRG1'],
        ['KRT5','NOTCH1'],
        ['KRT5','P4HB'],
        ['KRT5','PCBD2'],
        ['KRT5','PDE3A'],
        ['KRT5','PDXK'],
        ['KRT5','PFN1'],
        ['KRT5','PI3'],
        ['KRT5','PKM2'],
        ['KRT5','PPL'],
        ['KRT5','RP11-1020M18.10'],
        ['KRT5','RP11-116G8.5'],
        ['KRT5','RPL8'],
        ['KRT5','RPS21'],
        ['KRT5','RRP1B'],
        ['KRT5','S100A2'],
        ['KRT5','S100A8'],
        ['KRT5','S100A9'],
        ['KRT5','SDF4'],
        ['KRT5','SFN'],
        ['KRT5','SMARCC2'],
        ['KRT5','SPARC'],
        ['KRT5','SPRR1B'],
        ['KRT5','SPRR3'],
        ['KRT5','TMSB10'],
        ['KRT6A','KRT6B'],
        ['KRT6A','KRT6C'],
        ['KRT6A','KRT75'],
        ['KRT6A','KRT76'],
        ['KRT6A','KRT78'],
        ['KRT6A','LDLRAD2'],
        ['KRT6A','NDRG1'],
        ['KRT6A','PDXK'],
        ['KRT6A','PI3'],
        ['KRT6A','PTMS'],
        ['KRT6A','RP11-1020M18.10'],
        ['KRT6A','RRP1B'],
        ['KRT6A','S100A8'],
        ['KRT6A','S100A9'],
        ['KRT6A','SFN'],
        ['KRT6A','TAPBP'],
        ['KRT6A','TYMP'],
        ['KRT6B','KRT6C'],
        ['KRT6B','NUMA1'],
        ['KRT6B','PDXK'],
        ['KRT6B','PI3'],
        ['KRT6B','RP11-116G8.5'],
        ['KRT6B','RP11-849H4.4'],
        ['KRT6B','RRP1B'],
        ['KRT6B','S100A11'],
        ['KRT6B','S100A8'],
        ['KRT6B','SPRR1B'],
        ['KRT6C','KRT75'],
        ['KRT6C','LYPD3'],
        ['KRT6C','RP11-1020M18.10'],
        ['KRT6C','TMSB10'],
        ['KRT6C','TPT1'],
        ['KRT78','NRG1'],
        ['KRT8','PELI1'],
        ['KRT9','SGK3'],
        ['KSR1','MKS1'],
        ['KSR1','MSI2'],
        ['KSR1','PIPOX'],
        ['KSR1','TENM1'],
        ['KSR2','P2RX4'],
        ['KTN1','PRKCSH'],
        ['KXD1','LIG3'],
        ['KYNU','PBRM1'],
        ['L2HGDH','LMBRD1'],
        ['L3MBTL2','ZC3H7B'],
        ['L3MBTL3','WDR37'],
        ['L3MBTL4','LAMA1'],
        ['L3MBTL4','PAK4'],
        ['LACE1','LATS1'],
        ['LACE1','ZUFSP'],
        ['LAMA3','PSMA8'],
        ['LAMA3','ROCK1'],
        ['LAMA3','SNRPD1'],
        ['LAMA3','TTC39C'],
        ['LAMA3','ZNF521'],
        ['LAMA4','PSMA8'],
        ['LAMA4','TTC39C'],
        ['LAMA4','VIM'],
        ['LAMA4','ZNF521'],
        ['LAMA5','MYRFL'],
        ['LAMA5','NCOR2'],
        ['LAMA5','TAF4'],
        ['LAMA5','UBN1'],
        ['LAMB1','PUS7'],
        ['LAMB3','NAB1'],
        ['LAMC2','POC1B'],
        ['LAMC3','NAA35'],
        ['LANCL1','TET2'],
        ['LANCL2','SEPT14'],
        ['LANCL2','TFDP2'],
        ['LANCL2','ZNF786'],
        ['LAP3','NTRK2'],
        ['LAPTM4A','SDC1'],
        ['LAPTM4B','LRP12'],
        ['LAPTM4B','NCALD'],
        ['LAPTM5','MFSD2A'],
        ['LARGE','SNW1'],
        ['LARP4','MUCL1'],
        ['LARP4','RYR3'],
        ['LARP4B','PFKFB3'],
        ['LARP4B','SMCO3'],
        ['LARS2','LIMD1'],
        ['LASS3','LTBP3'],
        ['LATS1','NUP43'],
        ['LCA5L','PDXK'],
        ['LCOR','MAN1A1'],
        ['LCOR','PNLIPRP3'],
        ['LCP2','SGCD'],
        ['LCT','ZDHHC5'],
        ['LDB2','TAPT1'],
        ['LDLR','PKN1'],
        ['LDLR','TNPO2'],
        ['LDLRAD3','TM7SF3'],
        ['LDLRAD3','TRIM44'],
        ['LDLRAD4','PITRM1'],
        ['LDLRAD4','PREP'],
        ['LEC3','PPP3CA'],
        ['LEC3','ST3GAL5'],
        ['LELP1','LMX1A'],
        ['LEMD1','NUAK2'],
        ['LEMD3','LYZ'],
        ['LEMD3','XPOT'],
        ['LEPR','MIER1'],
        ['LEPREL1','LPP'],
        ['LEPREL1','TP63'],
        ['LEPREL2','PTPN6'],
        ['LEPROTL1','NCOA2'],
        ['LETM1','SLBP'],
        ['LETM1','TSC22D1'],
        ['LETM2','NSMAF'],
        ['LETM2','OSBPL1A'],
        ['LEUTX','NOVA2'],
        ['LGALS12','PLAC9'],
        ['LGALS12','SLC22A23'],
        ['LGALS12','SPTA1'],
        ['LGALS12','THAP2'],
        ['LGALS2','RBFOX2'],
        ['LGALS2','SSBP2'],
        ['LGALS3','RBFOX2'],
        ['LGALS3','SSBP2'],
        ['LGALS8','RHBG'],
        ['LGALS8','TRIM67'],
        ['LGALS9','NLK'],
        ['LGI4','MGMT'],
        ['LGR6','PPP1R12B'],
        ['LGR6','RABGAP1L'],
        ['LGSN','LMBRD1'],
        ['LHFP','TDRD3'],
        ['LHFP','UNC79'],
        ['LHFPL3','MLL3'],
        ['LHFPL3','MLL5'],
        ['LHPP','TANGO6'],
        ['LHPP','TMCO7'],
        ['LIFR','OSMR'],
        ['LIFR','PNPLA8'],
        ['LIFR','TRIO'],
        ['LIG3','ZFAT'],
        ['LIMA1','TAOK3'],
        ['LIMA1','TRAPPC9'],
        ['LIMCH1','MANBA'],
        ['LIMD1','SACM1L'],
        ['LIMD1','VGLL4'],
        ['LIMK1','TYW1'],
        ['LIMK2','OSBP2'],
        ['LIMK2','RUFY1'],
        ['LIMK2','SFI1'],
        ['LIMK2','SH3BP1'],
        ['LIMK2','Z83844.1'],
        ['LIN28A','RPS6KA1'],
        ['LIN52','SYNE2'],
        ['LIN52','ZNF410'],
        ['LIN7A','TRHDE'],
        ['LINGO1','SCAPER'],
        ['LIPF','MELK'],
        ['LLGL1','MARCH10'],
        ['LMAN2L','MECOM'],
        ['LMBR1','PTPRN2'],
        ['LMBR1','TPRG1'],
        ['LMBR1L','VDR'],
        ['LMCD1','THRB'],
        ['LMF1','PDXDC1'],
        ['LMNA','RAB4B'],
        ['LMNA','RAF1'],
        ['LMNA','SYT11'],
        ['LMO7','MED4'],
        ['LMTK2','RBFOX3'],
        ['LNP1','PAK1'],
        ['LNX1','LURAP1L'],
        ['LNX1','NFIB'],
        ['LNX1','PDGFRA'],
        ['LNX1','PLIN2'],
        ['LNX1','ZNF584'],
        ['LOH12CR1','PTPRR'],
        ['LOH12CR1','RP11-174G6.1'],
        ['LONP1','MAP2K7'],
        ['LONP1','SAFB2'],
        ['LONP1','ZNF780B'],
        ['LONRF1','PSMB7'],
        ['LPAR1','NAA15'],
        ['LPAR1','SNX30'],
        ['LPAR1','SNX33'],
        ['LPAR3','ST6GALNAC3'],
        ['LPCAT1','SDHA'],
        ['LPCAT2','ZNF77'],
        ['LPCAT3','LRPB7'],
        ['LPCAT3','LRRC23'],
        ['LPCAT3','MAP3K13'],
        ['LPHN1','POLR2M'],
        ['LPHN3','PPP3CA'],
        ['LPHN3','ST3GAL5'],
        ['LPIN1','ROCK2'],
        ['LPIN2','MYOM1'],
        ['LPO','SRP68'],
        ['LPP','MSF1'],
        ['LPP','RP11-430L16.1'],
        ['LPP','SEPT9'],
        ['LPP','SPATA16'],
        ['LPP','ST6GAL1'],
        ['LPP','TC2N'],
        ['LPP','TP63'],
        ['LPRP','TSPAN9'],
        ['LPXN','ZFP91'],
        ['LRBA','PPARGC1A'],
        ['LRBA','RXFP1'],
        ['LRBA','SH3D19'],
        ['LRBA','UQCC1'],
        ['LRCH3','MED12L'],
        ['LRIG1','SLC26A7'],
        ['LRIG3','PIP4K2C'],
        ['LRIG3','ROCK1'],
        ['LRP1','TMEM194A'],
        ['LRP1','XPO5'],
        ['LRP11','ULBP2'],
        ['LRP12','MET'],
        ['LRP12','RALGPS1'],
        ['LRP12','RIMS2'],
        ['LRP12','SLTM'],
        ['LRP12','ST14'],
        ['LRP12','TRAPPC9'],
        ['LRP12','TRPS1'],
        ['LRP12','VPS13B'],
        ['LRP1B','MAST4'],
        ['LRP1B','TXNL1'],
        ['LRP2','POFUT2'],
        ['LRP3','THAP8'],
        ['LRP3','VAV1'],
        ['LRP5','MOGAT2'],
        ['LRP5','MPP2'],
        ['LRP5','MRPL21'],
        ['LRP5','NOX4'],
        ['LRP5','PPP6R3'],
        ['LRP5','SRPK2'],
        ['LRP5','UVRAG'],
        ['LRP5','VPS26B'],
        ['LRP5','VPS37C'],
        ['LRP5','ZNF317'],
        ['LRP5','ZNF507'],
        ['LRP6','NINJ2'],
        ['LRP8','PAX8'],
        ['LRP8','RAD51C'],
        ['LRP8','TMEM217'],
        ['LRP8','USP24'],
        ['LRPB7','MBOAT5'],
        ['LRR1','MTHFD1'],
        ['LRRC1','MLIP'],
        ['LRRC1','SUPT3H'],
        ['LRRC16A','MRS2'],
        ['LRRC16A','SLC17A1'],
        ['LRRC19','ZCCHC7'],
        ['LRRC20','NODAL'],
        ['LRRC23','MBOAT5'],
        ['LRRC27','RPAIN'],
        ['LRRC37A3','SMURF2'],
        ['LRRC37B','RAD51C'],
        ['LRRC37B','RHOT1'],
        ['LRRC37B','TANC2'],
        ['LRRC37B','TIAF1'],
        ['LRRC3C','WIPF2'],
        ['LRRC40','WDR78'],
        ['LRRC41','PIK3R3'],
        ['LRRC41','SOGA2'],
        ['LRRC43','MAP4'],
        ['LRRC48','SREBF1'],
        ['LRRC48','XPO6'],
        ['LRRC49','THSD4'],
        ['LRRC4C','RASGRP1'],
        ['LRRC52','NOS1AP'],
        ['LRRC52','PBX1'],
        ['LRRC56','PHRF1'],
        ['LRRC59','NUP107'],
        ['LRRC59','SPAG9'],
        ['LRRC8A','PHYHD1'],
        ['LRRC8E','TJP3'],
        ['LRRCC1','MTG2'],
        ['LRRK1','METTL3'],
        ['LRRK1','SLC9B1'],
        ['LRRK1','TOX4'],
        ['LRRN2','NFASC'],
        ['LRRN4','MCM8'],
        ['LRRTM4','OXNAD1'],
        ['LRTOMT','NFIC'],
        ['LRTOMT','PROSAP1'],
        ['LRTOMT','SHANK2'],
        ['LRTOMT','UVRAG'],
        ['LSAMP','TMCC1'],
        ['LSAMP','ZBTB20'],
        ['LSM1','PHB'],
        ['LSM1','WHSC1L1'],
        ['LSM14A','SIPA1L3'],
        ['LSM4','PAK4'],
        ['LSM6','SEMA6A'],
        ['LSS','POMK'],
        ['LTA4H','WDR17'],
        ['LTBP1','PLB1'],
        ['LTBP1','TTC27'],
        ['LTBP4','MYO1B'],
        ['LTK','UACA'],
        ['LTN1','RWDD2B'],
        ['LTV1','NFIA'],
        ['LUC7L','NPRL3'],
        ['LUC7L2','TTC12'],
        ['LUC7L3','TPT1'],
        ['LURAP1L','MPDZ'],
        ['LURAP1L','TYRP1'],
        ['LUSTR1','VPS13A'],
        ['LUZP1','NEDD4L'],
        ['LXN','TBL1XR1'],
        ['LY6G6D','STK19'],
        ['LYN','MCM5'],
        ['LYN','NTRK3'],
        ['LYN','PXDNL'],
        ['LYN','SHB'],
        ['LYN','TGS1'],
        ['LYN','VPS13B'],
        ['LYPD1','NCKAP5'],
        ['LYPD5','MARK4'],
        ['LYRM4','RNF144B'],
        ['LYRM4','RPP40'],
        ['LYRM4','RREB1'],
        ['LYSMD2','SLC24A5'],
        ['LYST','NID1'],
        ['LYST','RPS6KC1'],
        ['LYZ','PRKCI'],
        ['LYZ','SMG1'],
        ['LYZ','STK10'],
        ['LYZ','STRADA'],
        ['LYZ','YEATS4'],
        ['LYZL2','MAP3K8'],
        ['LZTR1','PI4KA'],
        ['LZTS1','SBF2'],
        ['MACF1','PTPRU'],
        ['MACROD1','NRXN2'],
        ['MACROD2','TASP1'],
        ['MAD1L1','SDK1'],
        ['MAD1L1','SERGEF'],
        ['MAD1L1','TTYH3'],
        ['MAD2L2','SPATA6'],
        ['MAEA','TMEM71'],
        ['MAEA','UVSSA'],
        ['MAEL','UBE2Q1'],
        ['MAFF','STK16'],
        ['MAGEB18','USP9X'],
        ['MAGED2','MPHOSPH8'],
        ['MAGI1','RP11-175F9.2'],
        ['MAGI2','PILRB'],
        ['MAGI2','VWC2'],
        ['MAGI2','ZBTB1'],
        ['MAGI3','NOTCH2'],
        ['MAGI3','RAP1A'],
        ['MAGI3','SPAG17'],
        ['MAK','TFAP2A'],
        ['MALRD1','MSRB2'],
        ['MALRD1','PLXDC2'],
        ['MAMDC2','MRPS28'],
        ['MAMDC2','MRPS35'],
        ['MAMDC2','SMC5'],
        ['MAMDC2','TPD52'],
        ['MAML2','MTMR2'],
        ['MAN1A1','PDSS2'],
        ['MAN1A2','NPL3'],
        ['MAN1A2','REL'],
        ['MAN1A2','TAF15'],
        ['MAN1A2','TTF2'],
        ['MAN1A2','VANGL1'],
        ['MAN2B2','TACC3'],
        ['MANBA','NFKB1'],
        ['MANBA','WNK1'],
        ['MANBAL','SRC'],
        ['MANF','SETD2'],
        ['MANSC1','SOX5'],
        ['MAOA','OPHN1'],
        ['MAP2K1','SMAD3'],
        ['MAP2K2','PSMD9'],
        ['MAP2K2','RAB8A'],
        ['MAP2K3','NKAIN3'],
        ['MAP2K4','MYH3'],
        ['MAP2K5','RAB27A'],
        ['MAP2K5','SMAD6'],
        ['MAP2K5','SNRNP40'],
        ['MAP2K6','SDF2'],
        ['MAP2K7','SH3GL1'],
        ['MAP3K1','PARN'],
        ['MAP3K10','SUPT5H'],
        ['MAP3K12','TEAD4'],
        ['MAP3K13','NSUN2'],
        ['MAP3K13','UTS2B'],
        ['MAP3K13','VKORC1'],
        ['MAP3K13','VPS8'],
        ['MAP3K15','POLA1'],
        ['MAP3K2','SUPT3H'],
        ['MAP3K2','TIAM1'],
        ['MAP3K3','USP32'],
        ['MAP3K4','METTL24'],
        ['MAP3K4','ZNF292'],
        ['MAP3K5','NKAIN2'],
        ['MAP3K5','PERP'],
        ['MAP3K8','MROH2B'],
        ['MAP4','RHOA'],
        ['MAP4','SPINK8'],
        ['MAP4K1','SCAF1'],
        ['MAP4K1','SPINT2'],
        ['MAP4K2','NETO2'],
        ['MAP4K3','PTPLAD2'],
        ['MAP4K3','RAB3GAP1'],
        ['MAP4K3','SMC6'],
        ['MAP4K3','SOS1'],
        ['MAP4K4','MUCL1'],
        ['MAP4K4','RRBP1'],
        ['MAP4K4','SYF2'],
        ['MAP4K5','NIN'],
        ['MAP4K5','SAV1'],
        ['MAP4K5','THSD7A'],
        ['MAP4K5','TRIM9'],
        ['MAP6','UVRAG'],
        ['MAP7','TINAGL1'],
        ['MAP7','ZBTB20'],
        ['MAP7D1','MIDN'],
        ['MAP7D1','THRAP3'],
        ['MAP7D2','SCML2'],
        ['MAP7D2','SH3KBP1'],
        ['MAP9','TAT'],
        ['MAPK1','PRODH'],
        ['MAPK1','SSU72'],
        ['MAPK1','TOP3B'],
        ['MAPK10','UGT8'],
        ['MAPK14','MICU2'],
        ['MAPK14','SIN3A'],
        ['MAPK15','MED25'],
        ['MAPK15','MED9'],
        ['MAPK4','RTTN'],
        ['MAPK6','TMOD3'],
        ['MAPK6','USO1'],
        ['MAPK8','RAP1B'],
        ['MAPK8IP3','PDPK1'],
        ['MAPK9','RAI14'],
        ['MAPK9','UBL7'],
        ['MAPKAP1','PRRC2B'],
        ['MAPKAP1','SLA'],
        ['MAPKAP1','SPTAN1'],
        ['MAPKAP1','UHRF1'],
        ['MAPKAP1','VAV2'],
        ['MAPKAPK2','PLA2G2A'],
        ['MAPKAPK2','SRGAP2'],
        ['MAPKAPK2','SRGAP3'],
        ['MAPRE1','PTPRT'],
        ['MAPT','MARCH10'],
        ['MAPT','RAPGEFL1'],
        ['MARCH-VIII','MSMB'],
        ['MARCH1','PTPN13'],
        ['MARCH1','YWHAQ'],
        ['MARCH10','PITPNC1'],
        ['MARCH10','VMP1'],
        ['MARCH11','TRIO'],
        ['MARCH2','PLVAP'],
        ['MARCH8','MSMB'],
        ['MARCH9','TSFM'],
        ['MARK1','PA2G4'],
        ['MARK1','RHOU'],
        ['MARK2','PPFIA1'],
        ['MARK2','RTN3'],
        ['MARK3','MIA2'],
        ['MARK3','PDIA4'],
        ['MARK3','RP11-407N17.3'],
        ['MARK3','RP11-73M18.2'],
        ['MARK3','SLC25A21'],
        ['MARK4','PLAUR'],
        ['MARK4','PRKRIR'],
        ['MARS','NOP2'],
        ['MARS','YEATS4'],
        ['MARVELD2','WDR19'],
        ['MASP1','RAB43'],
        ['MAST2','MMEL1'],
        ['MAST2','NASP'],
        ['MAST2','PSMA5'],
        ['MAST2','SLC39A6'],
        ['MAST3','TBL1XR1'],
        ['MAST4','MRPS27'],
        ['MAST4','NLN'],
        ['MAST4','PLIN2'],
        ['MAST4','SREK1IP1'],
        ['MASTL','SEL1L'],
        ['MAT1A','STK39'],
        ['MATK','PIP5K1C'],
        ['MATN2','MTDH'],
        ['MATN2','RAD54B'],
        ['MATN2','ZNF461'],
        ['MATR3','MYLK'],
        ['MATR3','PCDHA2'],
        ['MAX','RUNX2'],
        ['MAX','TEX10'],
        ['MBD1','NRG1'],
        ['MBD3','ULK2'],
        ['MBD4','ODF3L2'],
        ['MBD4','PHOSPHO1'],
        ['MBD4','RHBDL3'],
        ['MBD4','SERPINB8'],
        ['MBD4','TMPRSS9'],
        ['MBD4','WSB1'],
        ['MBNL1','NAALADL2'],
        ['MBNL1','SGSM3'],
        ['MBNL2','PCCA'],
        ['MBOAT7','TMC4'],
        ['MBP','MYO1D'],
        ['MBP','SLC1A3'],
        ['MBTD1','MYO7A'],
        ['MBTD1','PITPNC1'],
        ['MBTD1','SPAG9'],
        ['MCC','PSMD6'],
        ['MCCC1','MED10'],
        ['MCF2L','STK24'],
        ['MCFD2','SRBD1'],
        ['MCHR2','RPS6KA2'],
        ['MCM3AP','PTPRG'],
        ['MCM5','RALGAPB'],
        ['MCM5','TOM1'],
        ['MCM7','ZNF3'],
        ['MCMBP','ZNF23'],
        ['MCOLN1','PCNT'],
        ['MCTP1','MMAB'],
        ['MCTP1','PDE4D'],
        ['MCTP1','TNC'],
        ['MCU','NFIX'],
        ['MCUR1','S100A10'],
        ['MDGA1','POLR1C'],
        ['MDGA1','PTK7'],
        ['MDGA2','SRPK2'],
        ['MDM1','RAB21'],
        ['MDM2','MYRFL'],
        ['MDM2','NUP107'],
        ['MDM2','PARP11'],
        ['MDM2','SBF2'],
        ['MDM2','SYK'],
        ['MDM2','UAP1'],
        ['MDM4','STK40'],
        ['ME1','SNAP91'],
        ['ME3','NARS2'],
        ['MECOM','PSMD2'],
        ['MECOM','RPN1'],
        ['MECOM','RUNX1'],
        ['MECP2','PELLE'],
        ['MECR','NAALADL2'],
        ['MED1','ODF3L2'],
        ['MED1','PHOSPHO1'],
        ['MED1','RHBDL3'],
        ['MED1','SERPINB8'],
        ['MED1','TMPRSS9'],
        ['MED1','WSB1'],
        ['MED12L','ROBO1'],
        ['MED13','TANC2'],
        ['MED13','TEX2'],
        ['MED13','TLDC2'],
        ['MED13','VMP1'],
        ['MED13L','METTL9'],
        ['MED13L','MYBPC1'],
        ['MED13L','RP11-493P1.2'],
        ['MED13L','TRAFD1'],
        ['MED15','PI4KA'],
        ['MED15','RP11-39H13.1'],
        ['MED15','ZNRF3'],
        ['MED17','RP11-817J15.2'],
        ['MED18','UBE4B'],
        ['MED19','SSRP1'],
        ['MED20','RNF182'],
        ['MED20','TLL2'],
        ['MED21','PIP5K1C'],
        ['MED22','PIP4K2B'],
        ['MED22','SSH2'],
        ['MED22','TADA2A'],
        ['MED23','PEX7'],
        ['MED24','PIP4K2B'],
        ['MED24','SSH2'],
        ['MED24','TADA2A'],
        ['MED25','SIGLEC12'],
        ['MED29','SAMD4B'],
        ['MED4','POGZ'],
        ['MED4','RB1'],
        ['MED4-AS1','POGZ'],
        ['MED9','SIGLEC12'],
        ['MEF2A','NFATC2'],
        ['MEF2B','SUGP2'],
        ['MEF2D','ZDHHC5'],
        ['MEGF11','SIK2'],
        ['MEGF6','WRAP73'],
        ['MEI1','WBP2NL'],
        ['MEIOB','P4HTM'],
        ['MEIOB','SMPD3'],
        ['MEIOB','SPSB3'],
        ['MEIS1','ROCK2'],
        ['MEIS2','MYO5A'],
        ['MEIS2','ZNF143'],
        ['MELK','SDK1'],
        ['MELK','TRIM32'],
        ['MEP1A','PAK1IP1'],
        ['MERTK','SOX13'],
        ['MERTK','TMEM87B'],
        ['MET','OXR1'],
        ['MET','PTPRZ1'],
        ['MET','ST7'],
        ['MET','TFG'],
        ['MET','WNT2'],
        ['METTL15','SLC1A3'],
        ['METTL21A','PLEKHM3'],
        ['METTL21B','RASSF3'],
        ['METTL24','SLC22A16'],
        ['METTL25','TMTC2'],
        ['METTL2B','WBS'],
        ['METTL3','SALL2'],
        ['METTL3','STRN3'],
        ['METTL4','TMX3'],
        ['METTL5','TLK1'],
        ['METTL6','SH3BP5'],
        ['METTL7A','SRPK1'],
        ['METTL8','OLA1'],
        ['METTL8','RFTN2'],
        ['MFI2','SERINC2'],
        ['MFI2','SF3A3'],
        ['MFN1','PCBP2'],
        ['MFN1','PREX1'],
        ['MFN2','PLOD1'],
        ['MFSD2A','RLF'],
        ['MG61','PIBF1'],
        ['MGA','PBX4'],
        ['MGAM','PBX4'],
        ['MGAT4C','PLD1'],
        ['MGAT4C','PPP1R12A'],
        ['MGAT5B','VMP1'],
        ['MGC16471','VGLL4'],
        ['MGC20702','STAM2'],
        ['MGC26158','PHOSPHO1'],
        ['MGME1','RAB28'],
        ['MGMT','ZSCAN29'],
        ['MGRN1','SNX29'],
        ['MGRN1','UBN1'],
        ['MGST2','TCF12'],
        ['MIB2','PLEKHN1'],
        ['MICA','MICB'],
        ['MICAL3','SLC25A18'],
        ['MICALCL','PARVA'],
        ['MICALL2','SDK1'],
        ['MICU1','TACC2'],
        ['MICU1','TSPAN3'],
        ['MICU2','ZDHHC20'],
        ['MID1','S100A2'],
        ['MIEN1','SLFN11'],
        ['MIER2','SHC2'],
        ['MIER3','RGS3'],
        ['MIF4GD','RSPH3'],
        ['MIF4GD','SDK2'],
        ['MINK1','NQO2'],
        ['MINK1','SLA'],
        ['MINK1','SPNS3'],
        ['MINK1','TG'],
        ['MINK1','USO1'],
        ['MINPP1','PAPSS2'],
        ['MIP','NLRP4'],
        ['MIP','PAN2'],
        ['MIP','RDH14'],
        ['MIPEP','NLRP4'],
        ['MIPEP','PAN2'],
        ['MIPEP','RDH14'],
        ['MIPEP','SACS'],
        ['MIPEP','TBL1XR1'],
        ['MIPEP','ZDHHC20'],
        ['MIPOL1','NPAS3'],
        ['MIPOL1','SLC25A21'],
        ['MIPOL1','SNX6'],
        ['MIPOL1','SYNE2'],
        ['MIPOL1','TTC6'],
        ['MITD1','RTN4'],
        ['MITF','MTERFD1'],
        ['MKKS','RP4-734C18.1'],
        ['MKL1','NTNG1'],
        ['MKL1','NTNG2'],
        ['MKL1','SCGB1C1'],
        ['MKL1','SLC25A17'],
        ['MKL1','SPBP'],
        ['MKL1','SPIDR'],
        ['MKL1','ST13'],
        ['MKL1','TCF20'],
        ['MKL2','RBFOX1'],
        ['MKL2','SNX29'],
        ['MKL2','USP31'],
        ['MKL2','USP48'],
        ['MKL2','ZNF200'],
        ['MKLN1','PTK2'],
        ['MKNK2','RNF126'],
        ['MKNK2','ZFR2'],
        ['MKRN1','MRPS33'],
        ['MKRN2','RP11-474B16.1'],
        ['MKS1','SPAG9'],
        ['MLF2','PTMS'],
        ['MLIP','TMEM260'],
        ['MLKL','TMEM231'],
        ['MLL','MLLT10'],
        ['MLL','MLLT4'],
        ['MLLT1','ZNF439'],
        ['MLLT10','PICALM'],
        ['MLLT10','SNTG1'],
        ['MLLT11','PIP5K1A'],
        ['MLLT3','SLC28A3'],
        ['MLLT4','SASH1'],
        ['MLLT4','UNC93A'],
        ['MLLT6','PIP4K2B'],
        ['MLLT6','PPP1R12A'],
        ['MLPH','PIKFYVE'],
        ['MLXIP','PTGES2'],
        ['MMACHC','RP11-419C5.2'],
        ['MMP20','TMEM123'],
        ['MMP20','TYR'],
        ['MMP25','TMEM123'],
        ['MMP25','TYR'],
        ['MMP25','ZNF500'],
        ['MMP27','TMEM123'],
        ['MMP28','ZNF500'],
        ['MMP7','TMEM123'],
        ['MMRN1','SPATA5'],
        ['MMRN2','PMEPA1'],
        ['MN1','ZMYND8'],
        ['MNAT1','PCNXL4'],
        ['MNAT1','SIX1'],
        ['MNAT1','TANK'],
        ['MNDA','NOTCH2'],
        ['MOB1B','MUC7'],
        ['MOB1B','PPAPDC1A'],
        ['MOB3A','STK11'],
        ['MOB3B','PTPRD'],
        ['MOB3C','PDE4B'],
        ['MOCS1','TRERF1'],
        ['MOK','MUCL1'],
        ['MOK','PRND'],
        ['MOK','WARS'],
        ['MON2','USP15'],
        ['MORC1','SPICE1'],
        ['MORN4','TBC1D12'],
        ['MOV10','STK25'],
        ['MOV10L1','NAV2'],
        ['MPDZ','NFIB'],
        ['MPDZ','PLIN2'],
        ['MPG','SNRNP25'],
        ['MPHOSPH6','OSBPL3'],
        ['MPHOSPH6','PLXDC2'],
        ['MPHOSPH8','PSPC1'],
        ['MPLKIP','TRA2A'],
        ['MPND','ZNF585A'],
        ['MPO','MSF1'],
        ['MPO','SEPT9'],
        ['MPP2','PLXNB2'],
        ['MPP2','PPY'],
        ['MPP5','RAD51B'],
        ['MPP5','RDH11'],
        ['MPP6','OSBPL3'],
        ['MPP7','RNF169'],
        ['MPRIP','P4HA1'],
        ['MPRIP','RAF1'],
        ['MPRIP','TRAT1'],
        ['MPV17L','XPO6'],
        ['MPZL1','PDE4DIP'],
        ['MRAS','NME9'],
        ['MRC2','RND2'],
        ['MRE11A','YAP1'],
        ['MREG','TTLL4'],
        ['MRGPRF','RAI1'],
        ['MRO','SUPT6H'],
        ['MROH1','TCTEX1D2'],
        ['MROH1','ZNF623'],
        ['MROH2B','MRPS30'],
        ['MROH7','PAK4'],
        ['MROH9','NOS1AP'],
        ['MRPL11','ST8SIA1'],
        ['MRPL14','TMEM63B'],
        ['MRPL15','YIPF1'],
        ['MRPL21','MYO1B'],
        ['MRPL21','PPP6R3'],
        ['MRPL21','SHANK2'],
        ['MRPL21','SSH3'],
        ['MRPL33','ZNF512'],
        ['MRPL4','STAU1'],
        ['MRPL43','OSBP2'],
        ['MRPL48','RAB6A'],
        ['MRPL48','STK24'],
        ['MRPL48','STK3'],
        ['MRPS10','XPO5'],
        ['MRPS16','TTC18'],
        ['MRPS2','NAA35'],
        ['MRPS22','SLC9A9'],
        ['MRPS23','VEZF1'],
        ['MRPS28','PPFIBP1'],
        ['MRPS31','USP9X'],
        ['MRPS35','PPFIBP1'],
        ['MRPS6','MYO3A'],
        ['MRPS9','PRDM11'],
        ['MRVI1','SBF2'],
        ['MRX16','PELLE'],
        ['MS4A12','NKAIN2'],
        ['MS4A18','MTA2'],
        ['MSF1','NARF'],
        ['MSF1','RNF138'],
        ['MSF1','SEC14L1'],
        ['MSF1','TANC2'],
        ['MSH2','SLC3A1'],
        ['MSH2','TTC7A'],
        ['MSH4','RABGGTB'],
        ['MSH5','SKI2W'],
        ['MSH5','SKIV2L'],
        ['MSI1','PXDN'],
        ['MSI1','PXN'],
        ['MSI2','MYO18A'],
        ['MSI2','NR2F2'],
        ['MSI2','PCTP'],
        ['MSI2','RAD51C'],
        ['MSI2','SLC39A11'],
        ['MSI2','STAU2'],
        ['MSI2','STXBP4'],
        ['MSI2','SYNGR2'],
        ['MSI2','TEX14'],
        ['MSI2','TEX2'],
        ['MSI2','TM2D2'],
        ['MSI2','TRIM25'],
        ['MSI2','WIPF2'],
        ['MSI2','YPEL2'],
        ['MSL1','RARA'],
        ['MSL1','WIPF2'],
        ['MSL2','PMEPA1'],
        ['MSL2','PPP2R3A'],
        ['MSL2','STAG1'],
        ['MSN','NLRC4'],
        ['MST1','SRRD'],
        ['MTA1','PACS2'],
        ['MTAP','POC1B'],
        ['MTAP','RP11-399D6.2'],
        ['MTBP','PQLC2'],
        ['MTBP','UBXN2A'],
        ['MTCP1','STAG2'],
        ['MTDH','SNX31'],
        ['MTDH','SPIDR'],
        ['MTERF','PHTF2'],
        ['MTF1','SERINC2'],
        ['MTF1','SF3A3'],
        ['MTF2','RPL5'],
        ['MTG1','ZNF511'],
        ['MTHFD1','ZBTB1'],
        ['MTHFD1L','PLEKHG1'],
        ['MTHFD1L','RNF216'],
        ['MTHFD1L','SHANK1'],
        ['MTHFD1L','THEMIS'],
        ['MTHFD1L','TNIK'],
        ['MTL5','PPFIA1'],
        ['MTL5','PPP6R3'],
        ['MTL5','SART1'],
        ['MTL5','SUV420H1'],
        ['MTM1','VBP1'],
        ['MTMR1','NFE2L1'],
        ['MTMR1','NRF1'],
        ['MTMR12','TERT'],
        ['MTMR12','ZFR'],
        ['MTMR2','RAB1B'],
        ['MTMR3','RNF215'],
        ['MTMR3','SCNN1B'],
        ['MTMR6','ZMYM2'],
        ['MTRR','NMD3'],
        ['MTSS1','ST3GAL1'],
        ['MTSS1','WHSC1L1'],
        ['MTSS1','ZC2HC1A'],
        ['MTSS1L','VAC14'],
        ['MTURN','PLEKHA8'],
        ['MTURN','ZNRF2'],
        ['MUC1','PKLR'],
        ['MUC1','SIAE'],
        ['MUC7','SEC31A'],
        ['MUCL1','WHSC1L1'],
        ['MUM1','SH3GL1'],
        ['MUT','SUPT3H'],
        ['MUTYH','THRAP3'],
        ['MVB12B','RALGPS1'],
        ['MVB12B','TMC1'],
        ['MVP','REL'],
        ['MVP','ZNF267'],
        ['MYB','NFIB'],
        ['MYBPC1','PXDN'],
        ['MYBPC1','PXN'],
        ['MYBPH','PHLDA3'],
        ['MYBPHL','PSRC1'],
        ['MYCBP2','RNF219'],
        ['MYEF2','USP8'],
        ['MYH1','MYH2'],
        ['MYH10','USP22'],
        ['MYH11','SMG1'],
        ['MYH11','TRIO'],
        ['MYH11','WNK2'],
        ['MYH14','PRKD2'],
        ['MYH2','TPM2'],
        ['MYH2','VPS53'],
        ['MYL12B','PTPN2'],
        ['MYL6','PDE3A'],
        ['MYL6','SF3B2'],
        ['MYL6','ZNFX1'],
        ['MYLK','NXPH1'],
        ['MYLK','SIAE'],
        ['MYLK','TM7SF2'],
        ['MYO10','NIPBL'],
        ['MYO10','SERINC5'],
        ['MYO10','SLC12A7'],
        ['MYO10','ST6GALNAC3'],
        ['MYO15A','TOP3A'],
        ['MYO16','TRAF3'],
        ['MYO18A','NUFIP2'],
        ['MYO18A','RNF135'],
        ['MYO18A','SERGEF'],
        ['MYO18A','TOP3A'],
        ['MYO18A','ZBTB46'],
        ['MYO18B','NDRG1'],
        ['MYO18B','PI4KA'],
        ['MYO18B','UBE2R2'],
        ['MYO19','STK32C'],
        ['MYO19','USP22'],
        ['MYO1B','STAT4'],
        ['MYO1B','ZNF536'],
        ['MYO1C','PITPNA'],
        ['MYO1C','RNF41'],
        ['MYO1D','PLA2G6'],
        ['MYO1D','PRG2'],
        ['MYO1D','RARA'],
        ['MYO1D','SSH2'],
        ['MYO1D','SUV420H1'],
        ['MYO1D','TADA2A'],
        ['MYO1D','TEX2'],
        ['MYO1D','TMIGD1'],
        ['MYO1E','RNF41'],
        ['MYO1F','ZNF414'],
        ['MYO1H','SFSWAP'],
        ['MYO3A','SLC5A3'],
        ['MYO3B','TSHZ2'],
        ['MYO5A','MYO5C'],
        ['MYO5A','OIP5'],
        ['MYO5A','PRKDC'],
        ['MYO5A','TRPM1'],
        ['MYO5B','TMEM9'],
        ['MYO6','S100A8'],
        ['MYO6','SENP6'],
        ['MYO7A','PAK1'],
        ['MYO7A','PKN1'],
        ['MYO9A','TMEM59'],
        ['MYO9B','PNN'],
        ['MYO9B','VPS13B'],
        ['MYOZ2','USP53'],
        ['MYRIP','PFKFB4'],
        ['MYT1','SERINC2'],
        ['MYT1','SF3A3'],
        ['MYT1','STX16'],
        ['N4BP1','NOA1'],
        ['N4BP1','PTPRA'],
        ['N4BP2','RHOH'],
        ['N6AMT2','RANBP3'],
        ['NAA15','PALLD'],
        ['NAA15','SLC10A7'],
        ['NAA15','STPG2'],
        ['NAA25','SLC25A21'],
        ['NAA25','TBX5'],
        ['NAA30','TERT'],
        ['NAA60','RAB11FIP3'],
        ['NAALADL1','PATL1'],
        ['NAALADL1','SNX15'],
        ['NAALADL2','NCKAP5'],
        ['NAALADL2','NLGN1'],
        ['NAALADL2','PEX5L'],
        ['NAALADL2','SERPINI1'],
        ['NAALADL2','SLC22A23'],
        ['NAALADL2','SMC5'],
        ['NAB1','STEAP3'],
        ['NACA','PTGES3'],
        ['NACA','RP11-511B23.2'],
        ['NACA','RPL41'],
        ['NACAP1','TTC38'],
        ['NADK2','SPEF2'],
        ['NADSYN1','PLET1'],
        ['NADSYN1','SUPT5H'],
        ['NADSYN1','TRIM9'],
        ['NADSYN1','ZNF451'],
        ['NAF1','UNK'],
        ['NAGA','WBP2NL'],
        ['NAIC','SNTB2'],
        ['NAIC','SNTL'],
        ['NAP1L1','STK38L'],
        ['NAP1L4','RPAP1'],
        ['NAPG','SPRED2'],
        ['NARF','SEPT9'],
        ['NARFL','RBBP6'],
        ['NARS2','ORAOV1'],
        ['NARS2','PAK1'],
        ['NARS2','PCDHGA5'],
        ['NARS2','PPME1'],
        ['NARS2','RP11-360L9.4'],
        ['NARS2','SCFD2'],
        ['NAT8','PDK4'],
        ['NAV2','PACSIN3'],
        ['NAV3','PDK3'],
        ['NAV3','RP1-34H18.1'],
        ['NBAS','THADA'],
        ['NBEA','NDFIP2'],
        ['NBEA','PEX1'],
        ['NBEA','TMEM132D'],
        ['NBEAL1','TANK'],
        ['NBEAL2','ROBO2'],
        ['NBEAL2','TMC2'],
        ['NBR1','PALB2'],
        ['NBR1','RUNDC3A'],
        ['NBR1','WSB1'],
        ['NCALD','NCOA2'],
        ['NCALD','NPPC'],
        ['NCALD','NSMCE2'],
        ['NCALD','PLD3'],
        ['NCALD','RPL8'],
        ['NCALD','SLC5A11'],
        ['NCALD','SPARCL1'],
        ['NCALD','TPT1'],
        ['NCAM2','PDE7A'],
        ['NCAM2','TEC'],
        ['NCAM2','ZNF146'],
        ['NCAN','NR2F6'],
        ['NCAN','SNX29'],
        ['NCAPD2','VWF'],
        ['NCEH1','SPATA16'],
        ['NCF4','UBAP2'],
        ['NCK1','RAB6B'],
        ['NCK1','ZZZ3'],
        ['NCK2','SFXN5'],
        ['NCKAP1','PARD3'],
        ['NCKAP4','PHACTR1'],
        ['NCKAP4','RSF1'],
        ['NCKAP4','SRCIN1'],
        ['NCKAP5','OSMR'],
        ['NCKAP5','STK39'],
        ['NCL','RBM25'],
        ['NCL','SQSTM1'],
        ['NCOA1','PSD2'],
        ['NCOA1','RAB10'],
        ['NCOA2','SH2D6'],
        ['NCOA2','SLC22A23'],
        ['NCOA2','SPTA1'],
        ['NCOA2','ST18'],
        ['NCOA2','THAP2'],
        ['NCOA2','XKR9'],
        ['NCOA3','PTPRT'],
        ['NCOA3','SGK2'],
        ['NCOA3','SGK3'],
        ['NCOA4','OR13A1'],
        ['NCOA4','RET'],
        ['NCOA6','PPP4R1L'],
        ['NCOA6','PRNP'],
        ['NCOA6','RALGAPA2'],
        ['NCOA6','SLC10A1'],
        ['NCOA6','TCFL5'],
        ['NCOA7','TRMT11'],
        ['NCOR1','RHOU'],
        ['NCOR1','RSF1'],
        ['NCOR1','ZNF287'],
        ['NCOR2','PGC'],
        ['NCOR2','SCARB1'],
        ['NCOR2','TMEM132D'],
        ['NCOR2','UBC'],
        ['NCR3','PRRC2A'],
        ['NCR3','PTK2'],
        ['NCSTN','SPRR2G'],
        ['NDC80','SMCHD1'],
        ['NDC80','TTC39C'],
        ['NDE1','SMG1'],
        ['NDE1','TRIO'],
        ['NDE1','WNK2'],
        ['NDEL1','WSCD1'],
        ['NDFIP1','NDUFAF2'],
        ['NDRG1','PLEKHA5'],
        ['NDRG1','SFN'],
        ['NDRG1','SNW1'],
        ['NDRG1','ST3GAL1'],
        ['NDRG2','RABEP1'],
        ['NDRG2','SEC63'],
        ['NDRG2','TPPP2'],
        ['NDST1','TCOF1'],
        ['NDST2','ZSWIM8'],
        ['NDST3','TXN'],
        ['NDUFA10','SGPP2'],
        ['NDUFA7','PRKCG'],
        ['NDUFA7','SH2D3A'],
        ['NDUFAF6','PLEKHF2'],
        ['NDUFB7','PIK3R1'],
        ['NDUFB9','OXR1'],
        ['NDUFB9','TDRKH'],
        ['NDUFS2','PCP4L1'],
        ['NDUFS4','UBE2D2'],
        ['NDUFS5','SHPRH'],
        ['NEBL','PRL'],
        ['NEBL','RP11-383H13.1'],
        ['NEBL','SKIDA1'],
        ['NEBL','TMEM26'],
        ['NECAB1','VPS13B'],
        ['NEDD4','RAB27A'],
        ['NEDD4','RFX7'],
        ['NEDD4L','SECISBP2'],
        ['NEDD4L','TLN2'],
        ['NEDD4L','UBE2G1'],
        ['NEDD4L','ZNF396'],
        ['NEDD9','NUP153'],
        ['NEDD9','TMLHE'],
        ['NEGR1','RNF122'],
        ['NEGR1','WHSC1L1'],
        ['NEGR1','ZRANB2'],
        ['NEK11','TOPBP1'],
        ['NEK11','UBE2H'],
        ['NEK2','NVL'],
        ['NEK4','PBRM1'],
        ['NEK4','SETD2'],
        ['NEK6','PPP2R4'],
        ['NEK6','PSMB7'],
        ['NEK6','PTGS1'],
        ['NEK7','RAB3GAP2'],
        ['NEK8','TLCD1'],
        ['NEK9','TLCD1'],
        ['NEK9','TMED10'],
        ['NELFB','RASA2'],
        ['NELL2','PARD3B'],
        ['NEMF','SOS2'],
        ['NEO1','SIN3A'],
        ['NEP','PAK1'],
        ['NEP','PKN1'],
        ['NEU3','PEMT'],
        ['NEU3','SNTG1'],
        ['NEURL1','PDCD11'],
        ['NF1','NLE1'],
        ['NF1','NLK'],
        ['NF1','PDS5B'],
        ['NF1','PSMD11'],
        ['NF1','TEX14'],
        ['NF1','TPMT'],
        ['NF1','TRMT61B'],
        ['NF2','RP11-282I1.1'],
        ['NFASC','NTRK1'],
        ['NFASC','PPP1R12B'],
        ['NFASC','PRELP'],
        ['NFASC','SOX13'],
        ['NFATC3','PRMT7'],
        ['NFATC3','SLC7A6'],
        ['NFE2','STARD7'],
        ['NFE2L1','SLC39A11'],
        ['NFE2L1','STRIP2'],
        ['NFE2L2','PAX8'],
        ['NFE2L2','PDE11A'],
        ['NFE2L2','RFTN2'],
        ['NFE2L2','ZC3H13'],
        ['NFIA','ST6GALNAC3'],
        ['NFIB','PIM2'],
        ['NFIB','STRN'],
        ['NFIB','ZDHHC21'],
        ['NFIC','WDR18'],
        ['NFIC','ZNF653'],
        ['NFIX','RALGPS1'],
        ['NFIX','TBCD'],
        ['NFIX','TNPO2'],
        ['NFKBIA','PPP2R3C'],
        ['NFKBID','ZNF585B'],
        ['NFKBIE','SLC29A1'],
        ['NFRKB','NOX4'],
        ['NFX1','SPINK4'],
        ['NFX1','TMEM8B'],
        ['NFX1','ZNF600'],
        ['NFX1','ZNF616'],
        ['NFXL1','OCIAD2'],
        ['NFYA','TSPO2'],
        ['NGFR','SMURF2'],
        ['NHSL1','VTA1'],
        ['NICN1','RHOA'],
        ['NINJ2','PAH'],
        ['NINJ2','WNT5B'],
        ['NIPAL2','RNF19A'],
        ['NIPAL4','PRDM6'],
        ['NIPBL','SLC1A3'],
        ['NIPBL','SLC30A9'],
        ['NIPBL','SPEF2'],
        ['NIPBL','STK24'],
        ['NIPSNAP3A','TMEFF1'],
        ['NISCH','PBRM1'],
        ['NIT2','RAP1GAP'],
        ['NKAIN1','PFAS'],
        ['NKAIN1','PUM1'],
        ['NKAIN2','SNX14'],
        ['NKAIN3','PDE7A'],
        ['NKAIN3','POMK'],
        ['NKAIN3','SLCO5A1'],
        ['NKRF','UBE2A'],
        ['NKX2-1','SETX'],
        ['NKX2-1','SLC25A21'],
        ['NLGN1','RSRC1'],
        ['NLGN4Y','TXLNG2P'],
        ['NLK','RAD51D'],
        ['NLK','RAD51L3'],
        ['NLK','TADA2A'],
        ['NLK','TAOK1'],
        ['NLK','WRB'],
        ['NLRP11','ZNF331'],
        ['NLRP4','RP11-977G19.10'],
        ['NLRP4','TNPO1'],
        ['NLRP4','ZNF331'],
        ['NLRP5','ZNF492'],
        ['NLRP6','NTRK2'],
        ['NLRP6','UBL3'],
        ['NLRP9','RDH13'],
        ['NLRP9','U2AF2'],
        ['NME1-NME2','NMT1'],
        ['NME4','THBS2'],
        ['NME4','TMC5'],
        ['NMNAT1','UBE4B'],
        ['NMT1','TMEM106A'],
        ['NMUR1','PDE6D'],
        ['NOA1','PPP3CA'],
        ['NOC2','PUM1'],
        ['NOC4L','SBDS'],
        ['NOC4L','SDS'],
        ['NOC4L','STAB2'],
        ['NOC4L','TTYH3'],
        ['NOS1AP','PTPRU'],
        ['NOSTRIN','STAM2'],
        ['NOTCH1','SEC16A'],
        ['NOTCH2','PSMA5'],
        ['NOTCH2','PTGFRN'],
        ['NOTCH2','SPAG17'],
        ['NOTCH2NL','UBE2H'],
        ['NOTUM','PYCR1'],
        ['NOVA1','RPL8'],
        ['NOX4','ZFP91'],
        ['NPAS3','RGS6'],
        ['NPC2','RAD51B'],
        ['NPEPPS','PNPO'],
        ['NPFFR1','SAR1A'],
        ['NPHP3','VPS8'],
        ['NPLOC4','P4HB'],
        ['NPM1','RAB4A'],
        ['NPSR1','VPS41'],
        ['NQO2','RIPK1'],
        ['NQO2','SKAP1'],
        ['NR2C2','TGFBR2'],
        ['NR2F6','USHBP1'],
        ['NR4A2','TANC1'],
        ['NRBP1','PMEL'],
        ['NRBP1','RAB10'],
        ['NRBP2','SCRB1'],
        ['NRBP2','SCRIB'],
        ['NRCAM','TLN1'],
        ['NRD1','USP24'],
        ['NREP','WDR36'],
        ['NRF1','RHEB'],
        ['NRF1','STRIP2'],
        ['NRG1','PCM1'],
        ['NRG1','PDE7A'],
        ['NRG1','RAB3IL1'],
        ['NRG1','SDC4'],
        ['NRG1','THBS1'],
        ['NRG2','SIL1'],
        ['NRG4','PEAK1'],
        ['NRG4','UBE2Q2'],
        ['NRIP1','SHCBP1'],
        ['NRIP1','TIAM1'],
        ['NRP1','SPAG6'],
        ['NRP1','VSTM4'],
        ['NRP2','PARD3B'],
        ['NRROS','SGK1'],
        ['NRXN3','SOS2'],
        ['NSD1','NUP98'],
        ['NSD1','UNC5A'],
        ['NSD1','ZNF346'],
        ['NSMCE2','PACRG'],
        ['NSMCE2','PBX3'],
        ['NSMCE2','SNTB1'],
        ['NSMCE2','TRIB1'],
        ['NSMCE4A','ZMIZ1'],
        ['NSRP1','SKAP1'],
        ['NT5C3A','PDGFA'],
        ['NT5C3A','SLC37A1'],
        ['NT5DC2','WDR82'],
        ['NT5DC2','WDR82P1'],
        ['NTN1','STX8'],
        ['NTRK1','SQSTM1'],
        ['NTRK1','SSBP2'],
        ['NTRK1','TFG'],
        ['NTRK1','TPM3'],
        ['NTRK2','PAN3'],
        ['NTRK2','RASEF'],
        ['NTRK2','SQSTM1'],
        ['NTRK2','TRIM24'],
        ['NTRK3','RBPMS'],
        ['NTSR1','RBFOX2'],
        ['NTSR2','ROCK2'],
        ['NUAK1','UHRF1BP1L'],
        ['NUAK2','PIK3C2B'],
        ['NUAK2','RBM24'],
        ['NUBP1','UMODL1'],
        ['NUBPL','SCFD1'],
        ['NUBPL','WWP1'],
        ['NUCKS1','PM20D1'],
        ['NUDCD1','TMEM184C'],
        ['NUDCD1','VWA8'],
        ['NUDT19','SLC7A9'],
        ['NUF2','TIMM17A'],
        ['NUFIP2','ZFP69B'],
        ['NUMA1','PITPNM1'],
        ['NUMA1','PPP2R5B'],
        ['NUMA1','PROSAP1'],
        ['NUMA1','PTH'],
        ['NUMA1','REXO2'],
        ['NUMA1','SFN'],
        ['NUMA1','SHANK2'],
        ['NUMA1','TRPT1'],
        ['NUMA1','UBA2'],
        ['NUMB','TAOK3'],
        ['NUP107','RAB3IP'],
        ['NUP107','SHD'],
        ['NUP107','TSFM'],
        ['NUP153','RNF2'],
        ['NUP153','ZNF280C'],
        ['NUP155','PRRC2B'],
        ['NUP155','TRIO'],
        ['NUP155','ZFYVE26'],
        ['NUP188','PPP2R4'],
        ['NUP205','WDR91'],
        ['NUP210L','P66BETA'],
        ['NUP210L','RORA'],
        ['NUP214','POMT1'],
        ['NUP214','PRRC2B'],
        ['NUP50','RBFOX2'],
        ['NUP85','PPM1D'],
        ['NUP85','ZDHHC18'],
        ['NUP88','RABEP1'],
        ['NUP88','VTCN1'],
        ['NUP98','RHOG'],
        ['NUP98','STIM1'],
        ['NUP98','TMEM117'],
        ['NUPL2','PACSIN2'],
        ['NURF301','PIP4K2B'],
        ['NURF301','PITPNC1'],
        ['NURF301','RGS9'],
        ['NURF301','RGS9BP'],
        ['NUSAP1','OIP5'],
        ['NUTM1','WHSC1L1'],
        ['NVL','SUSD4'],
        ['NXF1','SLC3A2'],
        ['NXF1','WDR74'],
        ['NXF3','RNF44'],
        ['NXN','SMG6'],
        ['NXN','VPS53'],
        ['NXN','WRAP53'],
        ['NXPE3','ZPLD1'],
        ['NXT1','VPS16'],
        ['OAF','SLC26A11'],
        ['OAS3','RPH3A'],
        ['OBFC1','SH3PXD2A'],
        ['OC90','TATDN1'],
        ['OC90','TMEM64'],
        ['OCEL1','SMARCA4'],
        ['OCIAD2','RAPGEF2'],
        ['ODF1','RIMS2'],
        ['ODF1','TTC28'],
        ['ODF2','SLC2A8'],
        ['ODF3L1','UBE2Q2'],
        ['ODZ2','WWC1'],
        ['ODZ4','SHANK2'],
        ['OFCC1','USP36'],
        ['OGFOD3','WDR45B'],
        ['OLA1','SP3'],
        ['OLFM2','TBC1D21'],
        ['OLFM2','TMEM205'],
        ['OLFM2','ZNF121'],
        ['OLFML1','PPFIBP2'],
        ['OPA3','PPP1R13L'],
        ['OPA3','YWHAG'],
        ['OPCML','PROSAP1'],
        ['OPCML','SHANK2'],
        ['OPRL1','PRPF6'],
        ['OR2H1','TRERF1'],
        ['OR2H6','TRERF1'],
        ['ORC5','PRKAG2'],
        ['ORC6','VPS35'],
        ['ORMDL3','ZPBP2'],
        ['OS9','SLC35E3'],
        ['OS9','TSFM'],
        ['OSBPL1A','RIOK3'],
        ['OSBPL1A','UHRF1BP1'],
        ['OSBPL2','PLD1'],
        ['OSBPL5','RHOG'],
        ['OSBPL8','PTPRR'],
        ['OSBPL8','RAB3IP'],
        ['OSBPL8','TSHZ2'],
        ['OSGIN2','RIPK2'],
        ['OSR2','VPS13B'],
        ['OSTF1','PCSK5'],
        ['OSTM1','SNX3'],
        ['OTOP2','SLC16A5'],
        ['OTUB1','RPS6KA4'],
        ['OTUD7A','TBC1D22A'],
        ['OXCT1','TMEM17'],
        ['OXR1','PTPRK'],
        ['P2RX4','PTPN1'],
        ['P3H3','PTPN6'],
        ['P4HA2','PITX1'],
        ['P4HB','PLEKHA5'],
        ['P4HB','SEC63'],
        ['P4HB','SOS1'],
        ['P4HB','TPT1'],
        ['P4HB','YBX3'],
        ['P4HB','ZNFX1'],
        ['P66BETA','SLC9C2'],
        ['P66BETA','TNFSF18'],
        ['PAAF1','RNFT1'],
        ['PAAF1','RPS6KB2'],
        ['PABPC1','SNX31'],
        ['PACRG','QKI'],
        ['PACS1','PLEKHB1'],
        ['PACS1','RASGRP2'],
        ['PACS1','SCYL1'],
        ['PACS1','SLC22A25'],
        ['PACS1','SYT7'],
        ['PACS1','ZDHHC24'],
        ['PACSIN1','SAYSD1'],
        ['PACSIN2','TTLL1'],
        ['PACSIN2','ZNF823'],
        ['PACSIN3','PHF21A'],
        ['PADI4','RCC2'],
        ['PAEP','SIAE'],
        ['PAFAH1B1','TRPS1'],
        ['PAFAH1B2','RNF214'],
        ['PAFAH1B3','PPP1R37'],
        ['PAG1','ZNF704'],
        ['PAGE-1','SHROOM4'],
        ['PAGE1','SHROOM4'],
        ['PAGE4','PSD3'],
        ['PAIP2','UBE2D2'],
        ['PAK1','PDGFD'],
        ['PAK1','PRSS23'],
        ['PAK1','SHANK2'],
        ['PAK1','SPINT2'],
        ['PAK1','UVRAG'],
        ['PAK1IP1','TMEM14C'],
        ['PAK2','RPL19'],
        ['PAK3','STAG2'],
        ['PAK4','SPINT2'],
        ['PAK7','PANK2'],
        ['PALM','PIP5K1C'],
        ['PALM2-AKAP2','SHB'],
        ['PAN2','RP11-977G19.10'],
        ['PAN2','TNPO1'],
        ['PAN3','UBL3'],
        ['PAOX','ZNF346'],
        ['PAPD7','RAF1'],
        ['PAPD7','SRD5A1'],
        ['PAPOLA','ZNF652'],
        ['PAPPA-AS1','RBBP8'],
        ['PAPPA-AS1','RIOK3'],
        ['PARD3','RBM17'],
        ['PARD3','ZNF33A'],
        ['PARD3','ZNF521'],
        ['PARD3B','VWC2L'],
        ['PARD3B','WDR12'],
        ['PARD3B','WWOX'],
        ['PARD6B','RTFDC1'],
        ['PARD6G','RFTN2'],
        ['PARD6G','RP11-495O11.1'],
        ['PARD6G','TAOK1'],
        ['PARK2','STRADA'],
        ['PARK7','RP11-431K24.3'],
        ['PARN','RRN3'],
        ['PARP10','TAOK1'],
        ['PARP14','PARP15'],
        ['PARP14','TASP1'],
        ['PARP14','TTBK1'],
        ['PARP4','VRK3'],
        ['PARP8','TENC1'],
        ['PARVA','POLD3'],
        ['PARVA','TEAD1'],
        ['PASD1','TEC'],
        ['PATL1','PROSAP1'],
        ['PATL1','SHANK2'],
        ['PATL1','TECTA'],
        ['PATZ1','PKIB'],
        ['PAWR','PPP1R12A'],
        ['PAWR','PTPRR'],
        ['PAX5','TMEM9B'],
        ['PAX7','UBR4'],
        ['PAX8','PPARG'],
        ['PAXBP1','TP53I3'],
        ['PBP','SMARCC1'],
        ['PBRM1','TKT'],
        ['PBRM1','TMEM110-MUSTN1'],
        ['PBX1','RHBG'],
        ['PBX1','UCK2'],
        ['PC','PPP6R3'],
        ['PC','THRSP'],
        ['PCAF-B','STAT5B'],
        ['PCBD2','PKM2'],
        ['PCBD2','SEC24A'],
        ['PCBP2','USP15'],
        ['PCBP2','YWHAE'],
        ['PCBP3','TIAM1'],
        ['PCCA','SNX16'],
        ['PCCB','PIK3CB'],
        ['PCDH15','USP54'],
        ['PCDH15','ZNF83'],
        ['PCDH9','RNF121'],
        ['PCDHB16','SNAP91'],
        ['PCDHGC3','QKI'],
        ['PCF11','SPAG17'],
        ['PCF11','SPON1'],
        ['PCGF3','PIGG'],
        ['PCLO','STK3'],
        ['PCMT1','SPAG9'],
        ['PCMTD1','RB1CC1'],
        ['PCMTD1','RHOT1'],
        ['PCMTD1','ST18'],
        ['PCMTD1','WRN'],
        ['PCMTD2','SNX6'],
        ['PCNT','POC1A'],
        ['PCNT','SLC37A1'],
        ['PCNT','USP25'],
        ['PCNX','RAD51B'],
        ['PCNXL2','PTPN14'],
        ['PCNXL2','TAOK3'],
        ['PCNXL3','RPS6KB2'],
        ['PCP4','PRGS'],
        ['PCP4','SLC37A1'],
        ['PCTP','RARA'],
        ['PCTP','TRIM37'],
        ['PCTP','USP32'],
        ['PCYT1B','SIAH1'],
        ['PCYT2','SRCIN1'],
        ['PDCD2L','UBA2'],
        ['PDCD6','TERT'],
        ['PDCD7','ZNF609'],
        ['PDE10A','POLB'],
        ['PDE10A','PSMB1'],
        ['PDE1A','PKP4'],
        ['PDE1A','PPP1R1C'],
        ['PDE3A','PHKB'],
        ['PDE3A','PI3'],
        ['PDE3A','S100A9'],
        ['PDE3A','SPRR1B'],
        ['PDE3A','ZNF318'],
        ['PDE3B','SBF2'],
        ['PDE4A','S1PR5'],
        ['PDE4B','PTPRF'],
        ['PDE4D','RGS7BP'],
        ['PDE4D','RNF145'],
        ['PDE4D','RP11-325I22.2'],
        ['PDE4D','SEMA6A'],
        ['PDE4DIP','TRPS1'],
        ['PDE4DIP','VANGL2'],
        ['PDE7A','PREX2'],
        ['PDE7A','TRIM55'],
        ['PDE7A','YPEL2'],
        ['PDE9A','SYNJ1'],
        ['PDGFRA','TMEM165'],
        ['PDGFRB','SDHA'],
        ['PDGFRL','VPS37A'],
        ['PDIA4','SIAE'],
        ['PDLIM1','SORBS1'],
        ['PDLIM3','SORBS2'],
        ['PDS5A','RBM47'],
        ['PDS5B','UBR2'],
        ['PDSS2','SGK1'],
        ['PDSS2','SMARCA4'],
        ['PDZD4','SSR4'],
        ['PDZRN3','TMC1'],
        ['PDZRN4','SCYL3'],
        ['PDZRN4','TOMM40'],
        ['PDZRN4','TPD52L2'],
        ['PEAK1','PSTPIP1'],
        ['PEAK1','TJP1'],
        ['PEAK1','TM9SF3'],
        ['PEAK1','VPS33B'],
        ['PEBP4','UNC5D'],
        ['PELI1','VPS54'],
        ['PEMT','RAI1'],
        ['PEPD','RHPN2'],
        ['PEPD','SKAP1'],
        ['PER3','PGD'],
        ['PEX5L','PTRH2'],
        ['PEX5L','SLC25A36'],
        ['PFDN4','SNAI1'],
        ['PFKFB3','ZNF831'],
        ['PFKL','ZBTB21'],
        ['PFKL','ZDHHC16'],
        ['PFKM','SLC11A2'],
        ['PFKP','PSG1'],
        ['PFKP','PSG2'],
        ['PGAP1','SPAG16'],
        ['PGAP1','VWC2L'],
        ['PGAP2','SBF2'],
        ['PGAP3','RARA'],
        ['PGAP3','STARD3'],
        ['PGAP3','WDR70'],
        ['PGBD2','TOX'],
        ['PGC','SGK1'],
        ['PGC','TFEB'],
        ['PGC','WNK1'],
        ['PGM2L1','UVRAG'],
        ['PGM5','PIP5K1B'],
        ['PGM5','SMC2'],
        ['PGMRP','PIP5K1B'],
        ['PGMRP','SMC2'],
        ['PHACTR1','SOCS4'],
        ['PHACTR1','SOCS7'],
        ['PHACTR1','ZNF638'],
        ['PHACTR2','PLAGL1'],
        ['PHACTR4','PIP5K1A'],
        ['PHACTR4','TRNAU1AP'],
        ['PHB','TANC2'],
        ['PHC2','ZSCAN20'],
        ['PHF12','PLXDC1'],
        ['PHF12','RSF1'],
        ['PHF12','SEZ6'],
        ['PHF12','TMEM97'],
        ['PHF20','PSMD1'],
        ['PHF20','RPN2'],
        ['PHF21B','PI4KA'],
        ['PHF3','SACS'],
        ['PHF8','SLCO1C1'],
        ['PHGDH','TTC39A'],
        ['PHGDH','WARS2'],
        ['PHKB','SLC23A1'],
        ['PHKB','SLC23A2'],
        ['PHKB','TOX3'],
        ['PHKB','VPS35'],
        ['PHKG1','PSPH'],
        ['PHLDB2','RIMS2'],
        ['PHLPP1','WDR12'],
        ['PHTF2','RAB1A'],
        ['PHTF2','RUNDC3B'],
        ['PHYHD1','ZER1'],
        ['PI3','WFDC2'],
        ['PI4K2A','ZDHHC16'],
        ['PI4KA','PRODH'],
        ['PI4KA','RREB1'],
        ['PI4KA','SERPIND1'],
        ['PI4KA','ZNF274'],
        ['PI4KB','SELENBP1'],
        ['PI4KB','SYT11'],
        ['PIAS3','REN'],
        ['PIAS4','SBNO2'],
        ['PIAS4','ZFR2'],
        ['PIBF1','POR'],
        ['PIBF1','PORCN'],
        ['PIBF1','WHSC1'],
        ['PICALM','RP11-849H4.2'],
        ['PICK1','POLR2F'],
        ['PID1','SPTLC2'],
        ['PID1','TRIP12'],
        ['PIGG','SPINK2'],
        ['PIGG','ZNF264'],
        ['PIGK','ST6GALNAC3'],
        ['PIGL','RABEP1'],
        ['PIGL','RNF167'],
        ['PIGL','TBC1D26'],
        ['PIGN','RNF152'],
        ['PIGP','PMEPA1'],
        ['PIGS','SPAG5'],
        ['PIGU','PITPNC1'],
        ['PIGU','PTPRT'],
        ['PIGZ','SENP5'],
        ['PIK3AP1','TM9SF3'],
        ['PIK3C2A','RP11-113D6.6'],
        ['PIK3C2G','PPFIBP1'],
        ['PIK3C3','TRPC6'],
        ['PIK3CA','TBL1XR1'],
        ['PIK3CA','USP13'],
        ['PIK3CB','PLS1'],
        ['PIK3CB','TSPO'],
        ['PIK3CB','VTI1A'],
        ['PIK3R3','ZSWIM5'],
        ['PIKFYVE','TDRD3'],
        ['PIKFYVE','ZNF124'],
        ['PIP4K2A','RIMKLB'],
        ['PIP4K2A','XRN1'],
        ['PIP4K2B','SLC6A4'],
        ['PIP4K2B','TMEM132E'],
        ['PIP4K2B','TMEM99'],
        ['PIP4K2C','R3HDM2'],
        ['PIP4K2C','SRI'],
        ['PIP4K2C','WNK1'],
        ['PIP5K1A','SETDB1'],
        ['PIP5K1A','SLC20A2'],
        ['PIP5K1A','TNFAIP8L2'],
        ['PIP5K1B','PTAR1'],
        ['PIP5K1B','STK24'],
        ['PIP5K1B','ZNF37A'],
        ['PIP5K1C','PPM1H'],
        ['PIP5K1C','RANBP3'],
        ['PIPOX','TAOK1'],
        ['PIR','SIK2'],
        ['PITPNA','PRPF8'],
        ['PITPNB','TOM1'],
        ['PITPNB','ZMYND8'],
        ['PITPNC1','PRKCA'],
        ['PITPNC1','PSMD12'],
        ['PITPNC1','SLC39A11'],
        ['PITPNC1','TLK2'],
        ['PITPNM3','STX8'],
        ['PITRM1','PLN'],
        ['PITRM1','TMEM242'],
        ['PIWIL2','PPP3CC'],
        ['PIWIL4','SLC35F2'],
        ['PKD2','TSPAN5'],
        ['PKHD1','RCAN2'],
        ['PKIB','SLC35F1'],
        ['PKIG','STK4'],
        ['PKM','SLC1A3'],
        ['PKMYT1','STX16'],
        ['PKN1','PRSS23'],
        ['PKN1','TECR'],
        ['PKN1','UVRAG'],
        ['PKN2','RAB3B'],
        ['PKN2','TUFT1'],
        ['PKN3','TRIM65'],
        ['PKP4','SETDB1'],
        ['PKP4','TANC1'],
        ['PKP4','USP15'],
        ['PLA2G2A','TMCO4'],
        ['PLA2G6','TMEM184B'],
        ['PLAA','PTPLAD2'],
        ['PLAUR','ZNF576'],
        ['PLB1','PPP1CB'],
        ['PLCB1','ROR1'],
        ['PLCB3','SNX32'],
        ['PLCB4','SLC24A3'],
        ['PLCB4','TNRC6C'],
        ['PLCD4','SEPT10'],
        ['PLCD4','STK16'],
        ['PLCD4','VIL1'],
        ['PLCG1','TAF4'],
        ['PLCG2','WWOX'],
        ['PLCH1','SKIL'],
        ['PLCXD3','PRLR'],
        ['PLCXD3','WDR70'],
        ['PLD1','TFDP2'],
        ['PLD2','RNF167'],
        ['PLD3','SIRT2'],
        ['PLEKHA2','WHSC1L1'],
        ['PLEKHA4','R3HDM4'],
        ['PLEKHA5','TSC22D1'],
        ['PLEKHA5','VIM'],
        ['PLEKHA6','SRBD1'],
        ['PLEKHA7','SERGEF'],
        ['PLEKHA7','SMPD1'],
        ['PLEKHF2','TP53INP1'],
        ['PLEKHG5','TNFRSF8'],
        ['PLEKHH1','PPP2R5E'],
        ['PLEKHH1','VTI1B'],
        ['PLEKHM2','TMEM51'],
        ['PLEKHM2','ZSWIM5'],
        ['PLEKHM3','SEC23B'],
        ['PLEKHO1','SETDB1'],
        ['PLGRKT','RCL1'],
        ['PLK1','RNF40'],
        ['PLK4','TBC1D9'],
        ['PLN','PREP'],
        ['PLN','PTPRK'],
        ['PLN','SDHB'],
        ['PLOD2','PLSCR4'],
        ['PLP1','SGK1'],
        ['PLRG1','SEC24D'],
        ['PLS1','TFDP2'],
        ['PLXDC2','SPAG6'],
        ['PLXDC2','ST8SIA6'],
        ['PLXDC2','SYNE1'],
        ['PLXDC2','VAMP5'],
        ['PLXN4','TKT2'],
        ['PLXN4','TKTL1'],
        ['PLXNA3','TKT2'],
        ['PLXNA3','TKTL1'],
        ['PLXNB2','SHANK3'],
        ['PLXND1','TMCC1'],
        ['PMEL','RIPK4'],
        ['PMEPA1','PTGIS'],
        ['PMEPA1','RAB43'],
        ['PMEPA1','RALY'],
        ['PML','POLN'],
        ['PML','RARA'],
        ['PMM2','UBFD1'],
        ['PMM2','UIMC1'],
        ['PMP22','SLC16A14'],
        ['PMP22','TAX1BP3'],
        ['PMP22','TRIP12'],
        ['PMPCA','SEC16A'],
        ['PNOC','RHOBTB2'],
        ['PNPLA7','WDR85'],
        ['POC1A','VPRBP'],
        ['PODXL','PPP6R3'],
        ['PODXL','THRSP'],
        ['POFUT2','WDR4'],
        ['POGLUT1','SETD2'],
        ['POLA1','TIMELESS'],
        ['POLA2','PPFIA1'],
        ['POLB','UNC5D'],
        ['POLB','UVRAG'],
        ['POLD3','PPP6R3'],
        ['POLD3','TENM4'],
        ['POLD4','PRKRIR'],
        ['POLDIP3','TBC1D22A'],
        ['POLR1A','REEP1'],
        ['POLR2E','ZNF420'],
        ['POLR2G','SLC3A2'],
        ['POLR2I','ZC3H4'],
        ['POLR2K','STK24'],
        ['POLR2K','STK3'],
        ['POLR3E','RMI2'],
        ['POLR3H','RP1-130H16.18'],
        ['POLR3H','TBC1D10A'],
        ['POMK','ST6GAL1'],
        ['POSTN','YY1'],
        ['POU2F2','PTPRS'],
        ['POU2F2','VRK3'],
        ['POU6F2','SUGCT'],
        ['PPAP2A','SPTLC3'],
        ['PPAP2C','SLC39A3'],
        ['PPAPDC1A','RAPGEF5'],
        ['PPAPDC1A','RP11-254K3.1'],
        ['PPAPDC1A','SREBF2'],
        ['PPARD','WNK4'],
        ['PPARG','SYN2'],
        ['PPARG','TSEN2'],
        ['PPCDC','SCAMP2'],
        ['PPFIA2','TMTC2'],
        ['PPFIA3','TRPM4'],
        ['PPFIBP1','SMCO2'],
        ['PPIF','REEP3'],
        ['PPIG','S52'],
        ['PPIL6','ZNF322'],
        ['PPIL6','ZNF322P1'],
        ['PPM1D','STXBP4'],
        ['PPM1D','TRIM37'],
        ['PPM1E','SPAG5'],
        ['PPM1E','TRIM25'],
        ['PPM1E','VMP1'],
        ['PPM1H','SLC6A15'],
        ['PPM1H','SRGAP1'],
        ['PPM1H','TBC1D30'],
        ['PPM1H','USP15'],
        ['PPME1','UVRAG'],
        ['PPOX','UHMK1'],
        ['PPP1CA','RPS6KB2'],
        ['PPP1R12C','PPP6R1'],
        ['PPP1R12C','SEMA3C'],
        ['PPP1R12C','ULK1'],
        ['PPP1R12C','ZNF233'],
        ['PPP1R12C','ZNF460'],
        ['PPP1R16B','TTI1'],
        ['PPP1R17','SRRT'],
        ['PPP1R1B','SIK3'],
        ['PPP1R1B','ZNHIT1'],
        ['PPP1R37','PRR19'],
        ['PPP1R37','RHPN2'],
        ['PPP1R37','ZNF224'],
        ['PPP1R37','ZNF225'],
        ['PPP1R9A','WBSCR17'],
        ['PPP2CA','ROCK2'],
        ['PPP2CB','PURG'],
        ['PPP2R2A','ZDHHC2'],
        ['PPP2R3C','WARS'],
        ['PPP2R4','RALGDS'],
        ['PPP2R5C','RCOR1'],
        ['PPP2R5C','SCN1B'],
        ['PPP2R5C','TTC7B'],
        ['PPP2R5C','UPF2'],
        ['PPP2R5D','UBR2'],
        ['PPP2R5E','PTK2'],
        ['PPP2R5E','TRIO'],
        ['PPP3CA','SPARCL1'],
        ['PPP3R1','SPATS2L'],
        ['PPP3R1','TTC27'],
        ['PPP6C','SCAI'],
        ['PPP6R1','RDH13'],
        ['PPP6R2','SPECC1L'],
        ['PPP6R2','TYMP'],
        ['PPP6R3','RNF121'],
        ['PPP6R3','SPTBN2'],
        ['PPP6R3','SSH2'],
        ['PPP6R3','STRN3'],
        ['PPP6R3','YAP1'],
        ['PPT1','SCMH1'],
        ['PRC1','VPS33B'],
        ['PRCC','TMED5'],
        ['PRCP','RAB30'],
        ['PRDM1','RNF217'],
        ['PRDM1','SLC22A16'],
        ['PRDM16','PRKCZ'],
        ['PRDM16','SPOP'],
        ['PRDM5','RBM39'],
        ['PRDX6','ZFP30'],
        ['PREP','TMEM242'],
        ['PRG2','SLC1A3'],
        ['PRGS','RCAN1'],
        ['PRGS','UBC7'],
        ['PRGS','UBE2G2'],
        ['PRH1','TSPAN9'],
        ['PRICKLE2','RP11-14D22.2'],
        ['PRIM1','RNF41'],
        ['PRKAA1','TBC1D5'],
        ['PRKAA2','USP24'],
        ['PRKACA','TPGS1'],
        ['PRKACB','RPF1'],
        ['PRKAR1B','ULK1'],
        ['PRKAR2A','SLC25A20'],
        ['PRKAR2B','SLC35D1'],
        ['PRKAR2B','TRRAP'],
        ['PRKCA','RNF157'],
        ['PRKCA','SDK1'],
        ['PRKCA','ST6GALNAC2'],
        ['PRKCA','TANC2'],
        ['PRKCA','UNK'],
        ['PRKCB','SPNS1'],
        ['PRKCB','TNRC6A'],
        ['PRKCE','TRIM13'],
        ['PRKCE','XRN1'],
        ['PRKCH','RP11-47I22.3'],
        ['PRKCI','PTBP2'],
        ['PRKCI','TNIK'],
        ['PRKCZ','RP11-769N22.1'],
        ['PRKCZ','SKI'],
        ['PRKD1','PSMA6'],
        ['PRKD2','TSPAN5'],
        ['PRKD3','TMEM51'],
        ['PRKDC','PXDNL'],
        ['PRKDC','SLC39A14'],
        ['PRKDC','SPIDR'],
        ['PRKDC','ZNF114'],
        ['PRKG1','TRIM22'],
        ['PRKG1','TRIM5'],
        ['PRKM4','RTTN'],
        ['PRKRIR','RP11-111M22.2'],
        ['PRMT1','TLE6'],
        ['PRMT2','STX18'],
        ['PRMT3','SBF2'],
        ['PRMT8','WNK1'],
        ['PRND','RAGE'],
        ['PROSAP1','TACC1'],
        ['PROSAP1','WNT11'],
        ['PRPF6','ST7L'],
        ['PRPSAP1','ST6GALNAC2'],
        ['PRPSAP2','ZNF343'],
        ['PRR11','RARA'],
        ['PRR12','PRRG2'],
        ['PRR14L','SOX10'],
        ['PRR4','TSPAN9'],
        ['PRR5-ARHGAP8','XRCC6'],
        ['PRR5L','TRIM44'],
        ['PRRC2A','REXO2'],
        ['PRRC2A','SFN'],
        ['PRRC2B','SET'],
        ['PRRC2C','SPATA13'],
        ['PRRG2','ZNF671'],
        ['PRRG4','QSER1'],
        ['PRRG4','RAPGEFL1'],
        ['PRSS3','RNF19A'],
        ['PRSS3','UBE2R2'],
        ['PRSS3','ZCCHC7'],
        ['PRSS33','SRRM2'],
        ['PRX','ZFP30'],
        ['PSD3','WHSC1L1'],
        ['PSEN1','STK4'],
        ['PSKH2','TRAPPC9'],
        ['PSMA3','ZNFX1'],
        ['PSMA6','VMP1'],
        ['PSMB4','SEC22B'],
        ['PSMD1','RPL37A'],
        ['PSMD11','WIPF2'],
        ['PSMD12','SUPT4H1'],
        ['PSMD3','RARA'],
        ['PSMD3','THRA'],
        ['PSMD3','UNC45B'],
        ['PSME4','STON1-GTF2A1L'],
        ['PSMG4','RPS6KA2'],
        ['PTBP3','SVEP1'],
        ['PTCRA','PTK7'],
        ['PTDSS1','PTK2'],
        ['PTEN','SHC2'],
        ['PTGER3','ZNF644'],
        ['PTGFRN','VTCN1'],
        ['PTGIS','RIMS2'],
        ['PTK2','RNF139'],
        ['PTK2','SLC45A4'],
        ['PTK2','TRAPPC9'],
        ['PTK2','UBE2H'],
        ['PTK2','UGT2A3'],
        ['PTMS','TAF3'],
        ['PTOV1','VRK3'],
        ['PTPN1','SLC9A8'],
        ['PTPN18','USP8'],
        ['PTPN9','SIN3A'],
        ['PTPRA','TADA2A'],
        ['PTPRD','RAC2'],
        ['PTPRD','RPS6KC1'],
        ['PTPRD','UNC13B'],
        ['PTPRF','ST3GAL3'],
        ['PTPRG','SYNPR'],
        ['PTPRG','TRPC1'],
        ['PTPRG','VWA8'],
        ['PTPRJ','SOX6'],
        ['PTPRK','PUM1'],
        ['PTPRK','STXBP6'],
        ['PTPRN2','TMEM87A'],
        ['PTPRN2','UBE3C'],
        ['PTPRO','UIMC1'],
        ['PTPRQ','RP11-288D9.1'],
        ['PTPRR','VWC2'],
        ['PTPRS','TICAM1'],
        ['PTPRU','SBSPON'],
        ['PTPRU','UIMC1'],
        ['PTPRZ1','RP11-286H14.4'],
        ['PTPRZ1','SRRM3'],
        ['PTPRZ1','TSPAN33'],
        ['PTRH2','STXBP4'],
        ['PTRH2','VMP1'],
        ['PTTG1IP','SIM2'],
        ['PUM1','RPH3AL'],
        ['PUM2','WDR35'],
        ['PUS1','SFSWAP'],
        ['PUS7','SRPK2'],
        ['PUS7L','SMCO2'],
        ['PVRL2','TOMM40'],
        ['PVRL2','ZMAT2'],
        ['PVRL3','THRB'],
        ['PVRL4','TSPAN9'],
        ['PVRL4','USP21'],
        ['PVRL4','USP25'],
        ['PWP1','SRGAP1'],
        ['PWP2','SFSWAP'],
        ['PWWP2B','STK32C'],
        ['PXDC1','RAB27B'],
        ['PXDNL','USP8'],
        ['PXK','RPP14'],
        ['PXMP2','TAX1BP3'],
        ['PYCRL','ZC3H3'],
        ['PYGL','TRIM9'],
        ['PZP','TOP1'],
        ['QARS','USP19'],
        ['QKI','ZNF214'],
        ['QSOX2','UBAC1'],
        ['R3HDM1','UBXN4'],
        ['R3HDM2','RP11-571M6.15'],
        ['R3HDM2','SORBS2'],
        ['R3HDM2','SUPT6H'],
        ['R3HDM2','XRCC6BP1'],
        ['R3HDM4','S100A9'],
        ['RAB10','SDC1'],
        ['RAB11FIP1','UNC5D'],
        ['RAB11FIP1','YWHAG'],
        ['RAB11FIP1','ZDHHC24'],
        ['RAB14','STOM'],
        ['RAB15','SPTB'],
        ['RAB1B','XBP1'],
        ['RAB1B','XBP1P1'],
        ['RAB20','VWC2'],
        ['RAB27A','RSL24D1'],
        ['RAB27A','SQRDL'],
        ['RAB28','TBC1D1'],
        ['RAB2A','TRAPPC9'],
        ['RAB34','TECTA'],
        ['RAB37','TMEM104'],
        ['RAB39A','TECTA'],
        ['RAB3IP','SLC2A13'],
        ['RAB40B','WDR45B'],
        ['RAB40C','RAD51B'],
        ['RAB40C','WDR90'],
        ['RAB43','STAG1'],
        ['RAB4A','SNRPB'],
        ['RAB6A','ZFP91'],
        ['RAB7A','RABEP1'],
        ['RABEP1','RORB'],
        ['RABEP1','RPL8'],
        ['RABEP1','SERPINE2'],
        ['RABEP1','SLC1A3'],
        ['RABEP1','SLC9A3R1'],
        ['RABEP1','SNRNP70'],
        ['RABEP1','TFF1'],
        ['RABGAP1L','RP11-160H22.5'],
        ['RABGAP1L','SLC38A11'],
        ['RABGAP1L','SPAG17'],
        ['RABL3','RYK'],
        ['RAD21','TRPS1'],
        ['RAD50','TCOF1'],
        ['RAD51','SLC2A9'],
        ['RAD51AP1','SLCO1C1'],
        ['RAD51AP1','ST8SIA1'],
        ['RAD51B','SLC39A9'],
        ['RAD51B','UVRAG'],
        ['RAD51B','ZFYVE26'],
        ['RAD51C','USP32'],
        ['RAD51D','SLCO6A1'],
        ['RAD52','WNK1'],
        ['RAD54B','TRIQK'],
        ['RAD54L2','RAPGEF4'],
        ['RAE1','RBM38'],
        ['RAE1','SLC9A8'],
        ['RAF1','TMEM40'],
        ['RAF1','TRAK1'],
        ['RAF1','XPC'],
        ['RALA','ZNF610'],
        ['RALBP1','RP11-159G9.5'],
        ['RALBP1','TWSG1'],
        ['RALGAPA2','RPRD1A'],
        ['RALGAPA2','TMEM132D'],
        ['RALGAPB','RPRD1B'],
        ['RALGPS1','TYRP1'],
        ['RALY','TPX2'],
        ['RALY','TRPC4AP'],
        ['RALYL','RNF19A'],
        ['RALYL','ZHX1'],
        ['RAMP1','RBM44'],
        ['RANBP10','RRP15'],
        ['RANBP10','SLC12A4'],
        ['RANBP10','SLC7A6'],
        ['RANBP17','SPDL1'],
        ['RANBP3','RFX2'],
        ['RANBP3L','RICTOR'],
        ['RANBP9','TPMT'],
        ['RANBP9','TXNDC5'],
        ['RAP1B','SCYL2'],
        ['RAP1B','SNUPN'],
        ['RAP1GAP','USP48'],
        ['RAPGEF5','STEAP1B'],
        ['RAPGEFL1','WIPF2'],
        ['RAPL','TIMM17A'],
        ['RARA','RP11-1407O15.2'],
        ['RARA','SKAP1'],
        ['RARA','SNRPB2'],
        ['RARA','STAC2'],
        ['RARA','STAT3'],
        ['RARA','TAOK1'],
        ['RARA','TPCN2'],
        ['RARA','WIPF2'],
        ['RARA','ZNF595'],
        ['RARS2','SMAP1'],
        ['RASA2','SUCNR1'],
        ['RASA2','ZBTB38'],
        ['RASAL2','ZNF695'],
        ['RASGEF1C','RNF130'],
        ['RASGEF1C','SLC12A7'],
        ['RASGRP3','RPS6KA1'],
        ['RASSF3','SNX10'],
        ['RASSF3','TBK1'],
        ['RASSF4','TMEM72'],
        ['RASSF5','TIMM17A'],
        ['RASSF8','RERGL'],
        ['RASSF8','SLC47A2'],
        ['RAVER1','S1PR2'],
        ['RB1','RCBTB2'],
        ['RB1','SH3PXD2A'],
        ['RB1','VWA8'],
        ['RB1CC1','TAOK1'],
        ['RBBP4','STK40'],
        ['RBBP8','TAF4B'],
        ['RBBP8','TP53'],
        ['RBFOX1','SIK2'],
        ['RBFOX1','VASN'],
        ['RBFOX2','SLC25A17'],
        ['RBKS','SUPT7L'],
        ['RBM12B','RSPO2'],
        ['RBM17','TWSG1'],
        ['RBM23','SLC7A8'],
        ['RBM25','VRTN'],
        ['RBM27','SLC35E2B'],
        ['RBM34','TARBP1'],
        ['RBM38','SPO11'],
        ['RBM39','VMP1'],
        ['RBM4','TENM4'],
        ['RBM47','ROCK1'],
        ['RBM6','SEMA3F'],
        ['RBP7','UBE4B'],
        ['RBPMS','RILPL1'],
        ['RCAN1','RUNX1'],
        ['RCAN1','SRP68'],
        ['RCAN1','TNPO3'],
        ['RCC1','SERINC2'],
        ['RCC1','TRNAU1AP'],
        ['RCK','SORL1'],
        ['RCK','TSPAN18'],
        ['RCOR1','TECPR2'],
        ['RCOR1','TRAF3'],
        ['RCOR1','XPC'],
        ['RCOR3','TMEM206'],
        ['RDH13','RUSC1'],
        ['RDH13','ZNF606'],
        ['RDH14','RP11-977G19.10'],
        ['RDH14','TNPO1'],
        ['RDX','SPAG9'],
        ['RDX','WAC'],
        ['RECK','TMEM8B'],
        ['RECQL5','SEC14L1'],
        ['RECQL5','WBP2'],
        ['REG4','WARS2'],
        ['RELB','RFX3'],
        ['REM1','TPX2'],
        ['REN','ZMIZ5'],
        ['REPS2','SHROOM2'],
        ['RERE','SLC2A5'],
        ['RERE','U2AF2'],
        ['RERE','VAMP3'],
        ['RERE','VPS13D'],
        ['RERE','ZSWIM5'],
        ['RET','RNF76'],
        ['RET','SPECC1L'],
        ['RET','TBL1XR1'],
        ['RET','TRIM27'],
        ['RET','TRIM33'],
        ['REV3L','SOBP'],
        ['RFC4','SENP7'],
        ['RFC5','WSB2'],
        ['RFPL3S','ZNF721'],
        ['RFT1','SFMBT1'],
        ['RFWD2','SRGAP2'],
        ['RFWD2','SRGAP3'],
        ['RFWD2','XPR1'],
        ['RFWD3','STK31'],
        ['RFX3','SMARCA2'],
        ['RFX4','RIC8B'],
        ['RGL1','VAV3'],
        ['RGL3','ZNF14'],
        ['RGS12','USP2'],
        ['RGS17','SYNE1'],
        ['RGS7','RYR2'],
        ['RHOA','SMCO1'],
        ['RHPN2','VSTM2B'],
        ['RHPN2','ZNF790'],
        ['RIBC2','TBC1D22A'],
        ['RIC8B','SYT1'],
        ['RILPL1','SARNP'],
        ['RILPL2','SETD8'],
        ['RIMS4','UBR5'],
        ['RIN1','SDHAF2'],
        ['RIN2','TMEM72'],
        ['RING11','VARS'],
        ['RINT1','SWI5'],
        ['RIOK1','USP45'],
        ['RIOK3','TAF4B'],
        ['RIOK3','TMEM241'],
        ['RIPK1','SERPINB9'],
        ['RLF','SMAP2'],
        ['RMI1','STX17'],
        ['RMI2','TEKT5'],
        ['RMND1','ZDHHC14'],
        ['RNASE10','STRN3'],
        ['RNASEH1P1','SCT'],
        ['RNF10','SYPL1'],
        ['RNF11','TAOK3'],
        ['RNF11','TMEM61'],
        ['RNF111','TCF12'],
        ['RNF111','ZNF280D'],
        ['RNF121','RP11-849H4.2'],
        ['RNF121','SF1'],
        ['RNF121','VPS37C'],
        ['RNF121','VWA5A'],
        ['RNF125','RNF138'],
        ['RNF125','TRAPPC8'],
        ['RNF126','SHC2'],
        ['RNF128','TBC1D8B'],
        ['RNF13','TM4SF4'],
        ['RNF138','SEPT9'],
        ['RNF165','SETBP1'],
        ['RNF169','USP35'],
        ['RNF170','SFRP1'],
        ['RNF170','YES1'],
        ['RNF185','SMTN'],
        ['RNF19A','SLC41A3'],
        ['RNF19A','TRIQK'],
        ['RNF19B','YARS'],
        ['RNF213','RPS6KB1'],
        ['RNF213','RPTOR'],
        ['RNF213','SLC26A11'],
        ['RNF213','UBE2O'],
        ['RNF216','ST14'],
        ['RNF216','TNRC18'],
        ['RNF216L','SP1'],
        ['RNF217','TRMT11'],
        ['RNF25','TMEM184B'],
        ['RNF25','USP37'],
        ['RNF38','ROCK1'],
        ['RNF8','ZFAND3'],
        ['RNFT1','VMP1'],
        ['RNFT2','ZDHHC17'],
        ['RNH1','SCT'],
        ['ROBO1','SEMA5B'],
        ['ROBO1','TUBB6'],
        ['ROBO2','VPS72'],
        ['ROCK1','RPL29'],
        ['ROCK1','SLC35F1'],
        ['ROCK1','TBL1XR1'],
        ['RORC','VPS72'],
        ['ROS1','SLC34A2'],
        ['RP1-8B1.4','SUPT3H'],
        ['RP11-108M9.4','VAMP3'],
        ['RP11-111M22.2','STX6'],
        ['RP11-114H23.1','RP11-114H23.2'],
        ['RP11-116G8.5','S100A9'],
        ['RP11-129B22.7','RP11-14D22.2'],
        ['RP11-1398P2.1','RP11-30C8.2'],
        ['RP11-150O12.1','RP11-150O12.5'],
        ['RP11-161H23.5','RP11-386G11.10'],
        ['RP11-161H23.5','TUBA1A'],
        ['RP11-161H23.5','TUBA1B'],
        ['RP11-176H8.1','SOX5'],
        ['RP11-307P5.1','SAMD5'],
        ['RP11-315O6.1','TMEM123'],
        ['RP11-362K2.2','SLC16A7'],
        ['RP11-366L20.2','TYW1'],
        ['RP11-371E8.4','ZNF169'],
        ['RP11-382A18.2','RP11-645E10.1'],
        ['RP11-382E9.1','VMP1'],
        ['RP11-386G11.10','TUBA1C'],
        ['RP11-419C5.2','TBC1D15'],
        ['RP11-430C17.1','ZFAND3'],
        ['RP11-433C9.2','TBL1XR1'],
        ['RP11-433C9.2','VMP1'],
        ['RP11-439L8.3','TMEM50A'],
        ['RP11-451M19.3','TNKS2'],
        ['RP11-455G16.1','SEC24D'],
        ['RP11-478C19.2','SLC2A13'],
        ['RP11-47I22.3','SNAPC1'],
        ['RP11-487E13.1','SORBS2'],
        ['RP11-511B23.2','RPL41'],
        ['RP11-540D14.8','SHC1'],
        ['RP11-557H15.2','RP11-557H15.3'],
        ['RP11-557H15.2','SGK1'],
        ['RP11-557H15.3','RP11-557H15.4'],
        ['RP11-557H15.4','SGK1'],
        ['RP11-571M6.15','TYW1'],
        ['RP11-624D11.2','TTC17'],
        ['RP11-656D10.5','ZNF684'],
        ['RP11-664D7.4','TRPS1'],
        ['RP11-672A2.3','TSKU'],
        ['RP11-698N11.2','TRIM44'],
        ['RP11-69L16.4','ZMIZ1'],
        ['RP11-706D8.3','TBL1XR1'],
        ['RP11-745O10.4','TMEM131'],
        ['RP11-817J15.2','TAF1D'],
        ['RP11-849H4.2','VPS37C'],
        ['RP11-90P5.2','WHSC1L1'],
        ['RP11-977G19.11','STAT2'],
        ['RP3-477O4.14','TM9SF4'],
        ['RP5-921G16.1','TUBGCP3'],
        ['RP5-988G17.1','SLC24A3'],
        ['RPL18','ZNF827'],
        ['RPL28','ZNF44'],
        ['RPL39L','UBXN7'],
        ['RPL8','SNW1'],
        ['RPLP0P2','TRMT112'],
        ['RPLP2','TALDO1'],
        ['RPN2','STK17A'],
        ['RPRD1A','SLC39A6'],
        ['RPRD2','S100A2'],
        ['RPRD2','SHC1'],
        ['RPRD2','TJP2'],
        ['RPRD2','VPS45'],
        ['RPRD2','VPS45A'],
        ['RPS24','STON2'],
        ['RPS4Y1','SRCAP'],
        ['RPS6KA3','VPREB1'],
        ['RPS6KA5','SOGA1'],
        ['RPS6KA5','TTC7B'],
        ['RPS6KB1','TEX2'],
        ['RPS6KB1','TUBD1'],
        ['RPS6KB1','VMP1'],
        ['RPS6KB2','ST3GAL2'],
        ['RPS6KC1','SDCCAG8'],
        ['RPS6KC1','TFPI'],
        ['RPTOR','SDF2'],
        ['RPTOR','STXBP4'],
        ['RPTOR','TBCD'],
        ['RQCD1','TTLL4'],
        ['RRAGD','UBE2J1'],
        ['RRBP1','TASP1'],
        ['RREB1','RRM2'],
        ['RRM1','STIM1'],
        ['RRM2B','STK3'],
        ['RRM2B','ZFPM2'],
        ['RSBN1L','SOS1'],
        ['RSF1','SOCS4'],
        ['RSF1','SOCS7'],
        ['RSF1','TENM4'],
        ['RSPO1','SFN'],
        ['RSPO3','TBC1D32'],
        ['RSRC2','SBNO1'],
        ['RSRC2','ZCCHC8'],
        ['RTCB','XPNPEP3'],
        ['RTN1','SAMD12'],
        ['RTN1','SYNE2'],
        ['RTN4R','SLC35G2'],
        ['RTN4RL1','VPS53'],
        ['RUNDC3B','ZNF3'],
        ['RUNX1','RUNX1T1'],
        ['RUVBL1','XRN1'],
        ['RWDD1','WASF1'],
        ['RYBP','TADA3'],
        ['RYR2','SUSD4'],
        ['RYR2','WLS'],
        ['S100A1','SNAPIN'],
        ['S100A10','THEM4'],
        ['S100A9','YWHAB'],
        ['S100B','TCTN1'],
        ['S52','TACR1'],
        ['SAC3D1','SNX15'],
        ['SAE1','WDR18'],
        ['SAE1','ZBTB45'],
        ['SAMD10','UCKL1'],
        ['SAMD12','SAMHD1'],
        ['SAMD12','SNTB1'],
        ['SAMD12','SYBU'],
        ['SAMD12','USP37'],
        ['SAMD12','YIF1B'],
        ['SAMD4A','VRK1'],
        ['SAMD4B','SIPA1L3'],
        ['SAMD8','TSPAN15'],
        ['SAMHD1','UQCC1'],
        ['SAMHD1','YIF1B'],
        ['SART1','SF3B2'],
        ['SBF2','TMEM99'],
        ['SBK1','TXLNA'],
        ['SBNO2','SERINC2'],
        ['SBNO2','TPGS1'],
        ['SC5D','SORL1'],
        ['SCAF1','TRAPPC6A'],
        ['SCAF8','VWA8'],
        ['SCAI','TOX3'],
        ['SCAMP2','SLC12A8'],
        ['SCAP','SLC25A20'],
        ['SCAP','SMARCC1'],
        ['SCAPER','TNRC6C'],
        ['SCARB2','STAU2'],
        ['SCARF1','SLC43A2'],
        ['SCD5','TMEM150C'],
        ['SCFD2','SPINK2'],
        ['SCFD2','YES1'],
        ['SCG5','SNX1'],
        ['SCG5','ZNF385B'],
        ['SCGB1D1','TMEM179B'],
        ['SCGB2B2','ZNF260'],
        ['SCIMP','ZNF594'],
        ['SCMH1','SHISA5'],
        ['SCML4','TUSC3'],
        ['SCML4','ZBTB24'],
        ['SCN3A','STK39'],
        ['SCN4A','TRIM47'],
        ['SCP2','STXBP3'],
        ['SCRN1','WDR53'],
        ['SCRN2','SKAP1'],
        ['SCYL3','SYN3'],
        ['SCYL3','TIMP3'],
        ['SDC3','SLC1A3'],
        ['SDC3','ZNF827'],
        ['SDC4','SPCS1'],
        ['SDC4','TSHZ2'],
        ['SDC4','YWHAB'],
        ['SDF2','YPEL2'],
        ['SDK1','VPS41'],
        ['SDK2','SLC39A11'],
        ['SDK2','TEX14'],
        ['SEC14L1','SEPT9'],
        ['SEC14L1','SPOP'],
        ['SEC14L3','VARS'],
        ['SEC16A','ZNF133'],
        ['SEC23A','TRAPPC6B'],
        ['SEC23IP','SLC39A9'],
        ['SEC23IP','TTC40'],
        ['SEC24C','VCL'],
        ['SEC63','SPARCL1'],
        ['SECISBP2','WNK2'],
        ['SEL1L','SERTAD2'],
        ['SEMA3C','VRK3'],
        ['SEMA4B','WDR93'],
        ['SEMA4B','ZNF710'],
        ['SEMA4C','WNT5B'],
        ['SEMA6D','SHANK2'],
        ['SENP5','TCTEX1D2'],
        ['SENP6','SLC35A1'],
        ['SENP6','ZNF544'],
        ['SENP8','SMAD3'],
        ['SEPT14','UPP1'],
        ['SEPT4','SUPT4H1'],
        ['SEPT4','WDR45B'],
        ['SEPT5','TBX1'],
        ['SEPT9','TANC2'],
        ['SERINC2','SNHG3'],
        ['SERINC2','ZCCHC17'],
        ['SERINC2','ZNFX1'],
        ['SERINC3','TWSG1'],
        ['SERINC5','SSBP2'],
        ['SERINC5','ZFYVE16'],
        ['SERPINB1','UBE3D'],
        ['SERPINB3','SERPINB4'],
        ['SERPINB8','TEAD1'],
        ['SESTD1','ZNF385B'],
        ['SETD2','ULK4'],
        ['SETD5','TATDN2'],
        ['SETX','TTF1'],
        ['SEZ6','WIPF2'],
        ['SF1','SYT7'],
        ['SFMBT1','WNT7A'],
        ['SFMBT2','TGM2'],
        ['SFN','SLPI'],
        ['SFN','SPRR1B'],
        ['SFN','SPRR2G'],
        ['SFN','WARS'],
        ['SFPQ','TFE3'],
        ['SFRP1','THAP1'],
        ['SFRP4','TSKU'],
        ['SFRP5','SLIT1'],
        ['SFRP5','SLIT3'],
        ['SFSWAP','ULK1'],
        ['SFTPD','TSPAN14'],
        ['SFTPD','VCL'],
        ['SGCB','SPATA18'],
        ['SGCZ','VPS13B'],
        ['SGIP1','USP1'],
        ['SGK1','SLC2A12'],
        ['SGK1','UTRN'],
        ['SGK3','TRIM55'],
        ['SGSM3','TNRC6B'],
        ['SGTA','SLC39A3'],
        ['SH3BP5','VGLL4'],
        ['SH3GL1','TAX1BP1'],
        ['SH3KBP1','VSIG1'],
        ['SH3PXD2B','STK10'],
        ['SH3PXD2B','UBTD2'],
        ['SH3TC1','SORCS2'],
        ['SHANK2','SLC3A2'],
        ['SHANK2','TACC1'],
        ['SHANK2','TRPM7'],
        ['SHANK2','UBAC1'],
        ['SHANK2','WNT11'],
        ['SHANK2','XPNPEP3'],
        ['SHB','SLC28A3'],
        ['SHC2','TJP3'],
        ['SHC2','TYK2'],
        ['SHFM1','TTC18'],
        ['SHISA5','SMARCC1'],
        ['SIAE','SLC25A16'],
        ['SIAE','SLC25A3'],
        ['SIAE','TPT1'],
        ['SIAE','VIM'],
        ['SIGLEC9','TLK2'],
        ['SIGLECL1','ZNF557'],
        ['SIGLECL1','ZNF714'],
        ['SIK1','TRAPPC10'],
        ['SIK3','UBR1'],
        ['SIL1','STK10'],
        ['SIPA1L2','ZNF252P'],
        ['SIPA1L3','SUPT5H'],
        ['SIPA1L3','ZNF835'],
        ['SIRT3','SPON1'],
        ['SKA2','TRIM37'],
        ['SKAP1','SPAG9'],
        ['SKAP1','TMEM100'],
        ['SKAP1','YPEL2'],
        ['SKI','SSU72'],
        ['SKIV2L2','USP28'],
        ['SKP1','VDAC1'],
        ['SLBP','ZSCAN1'],
        ['SLC10A7','TTC29'],
        ['SLC12A1','TRADD'],
        ['SLC12A5','ZNF350'],
        ['SLC12A7','SLC9A3'],
        ['SLC12A7','TAF1A'],
        ['SLC12A7','TMEM117'],
        ['SLC12A7','ZNF354A'],
        ['SLC16A10','TRMT11'],
        ['SLC16A13','SLC35G6'],
        ['SLC16A14','SP110'],
        ['SLC16A14','TRIP12'],
        ['SLC1A6','ZNF253'],
        ['SLC1A6','ZSWIM4'],
        ['SLC20A2','TUSC3'],
        ['SLC20A2','ZMAT4'],
        ['SLC22A15','VPS4B'],
        ['SLC22A23','TNFRSF21'],
        ['SLC22A23','TRAF3'],
        ['SLC22A23','TUBB2A'],
        ['SLC24A3','SLX4IP'],
        ['SLC24A3','TASP1'],
        ['SLC25A10','USP32'],
        ['SLC25A13','STK31'],
        ['SLC25A32','TRIB1'],
        ['SLC27A2','USP8'],
        ['SLC2A13','TERT'],
        ['SLC2A13','ZNF329'],
        ['SLC2A5','TRIM11'],
        ['SLC2A9','SLC30A9'],
        ['SLC35A1','WDR62'],
        ['SLC35B1','SPOP'],
        ['SLC35D2','YME1L1'],
        ['SLC35E3','YEATS4'],
        ['SLC35F1','STK24'],
        ['SLC35F1','TTK'],
        ['SLC36A1','TTC37'],
        ['SLC38A6','TANK'],
        ['SLC39A11','USP22'],
        ['SLC39A9','USP15'],
        ['SLC3A2','VPS37C'],
        ['SLC43A2','SMG6'],
        ['SLC43A2','YWHAE'],
        ['SLC45A4','TRAPPC9'],
        ['SLC46A2','USP42'],
        ['SLC47A1','SPAG5'],
        ['SLC4A10','TRIP12'],
        ['SLC4A4','TMPRSS11E'],
        ['SLC4A7','TYR'],
        ['SLC6A2','VPS35'],
        ['SLC6A4','TBC1D5'],
        ['SLC9A3','TRIO'],
        ['SLC9A3R1','TBCD'],
        ['SLC9A8','SNAI1'],
        ['SLC9A8','ZC3H7B'],
        ['SLC9A8','ZFP28'],
        ['SLC9A9','SMAD3'],
        ['SLC9B1','UBE2D3'],
        ['SLC9B2','TBCK'],
        ['SLIT1','TM9SF3'],
        ['SLIT3','TM9SF3'],
        ['SLITRK4','SUV420H1'],
        ['SLK','ZFHX3'],
        ['SLMO2','TRAPPC10'],
        ['SLTM','ST7'],
        ['SLTM','TFG'],
        ['SLTM','UACA'],
        ['SLTM','WNT2'],
        ['SMAD2','ZBTB7C'],
        ['SMAD3','SMAD6'],
        ['SMARCA5','TBC1D9'],
        ['SMARCC1','UQCRC1'],
        ['SMARCD2','STK11'],
        ['SMEK1','SRM'],
        ['SMEK1','SRMS'],
        ['SMG6','WDR20'],
        ['SMG9','ZNF415'],
        ['SMIM14','STK32B'],
        ['SMO','ZKSCAN5'],
        ['SMOX','ZKSCAN5'],
        ['SMTNL1','ZDHHC5'],
        ['SMURF1','TRIM4'],
        ['SMYD3','TFB2M'],
        ['SMYD3','WLS'],
        ['SMYD3','ZNF670'],
        ['SMYD3','ZNF695'],
        ['SNCAIP','SNX24'],
        ['SND1','UBE2D2'],
        ['SND1','ZFR'],
        ['SNRNP48','STK24'],
        ['SNRNP48','STK3'],
        ['SNRPB2','WIPF2'],
        ['SNTB2','TANGO6'],
        ['SNTB2','TMCO7'],
        ['SNTG2','TRPM6'],
        ['SNTL','TANGO6'],
        ['SNUPN','ZNF233'],
        ['SNX13','STAP2'],
        ['SNX14','SYNCRIP'],
        ['SNX18','WDR93'],
        ['SNX2','STK10'],
        ['SNX20','WHSC1L1'],
        ['SNX27','UCK2'],
        ['SNX29','TMEM241'],
        ['SNX29','TNFRSF17'],
        ['SNX7','SPIDR'],
        ['SNX7','ZCCHC14'],
        ['SOCS4','SRCIN1'],
        ['SOCS7','SRCIN1'],
        ['SORBS3','TUSC3'],
        ['SORL1','SYTL2'],
        ['SORL1','TBCEL'],
        ['SORL1','TECTA'],
        ['SOXN','UBAC1'],
        ['SP110','TRIP12'],
        ['SP3','STK39'],
        ['SPACA3','USP22'],
        ['SPAG1','TRPS1'],
        ['SPAG16','YES1'],
        ['SPAG4','ZBED4'],
        ['SPAG9','TANC2'],
        ['SPAST','TTC27'],
        ['SPATA16','TMEM131'],
        ['SPECC1','TMEM11'],
        ['SPECC1L','TMEM62'],
        ['SPEF2','TNFAIP8L1'],
        ['SPEN','TMEM51'],
        ['SPG20OS','TRPC4AP'],
        ['SPIDR','STIM2'],
        ['SPINK2','TMPRSS11D'],
        ['SPINK5','STK32A'],
        ['SPIRE2','TCF25'],
        ['SPNS1','ZNF668'],
        ['SPNS3','UBE2G1'],
        ['SPOP','TAC4'],
        ['SPOP','TAOK1'],
        ['SPOP','XYLT2'],
        ['SPRED1','TMCO5A'],
        ['SPRR1A','SPRR1B'],
        ['SPRR2A','SPRR2B'],
        ['SPRR2A','SPRR2D'],
        ['SPRR2B','SPRR2E'],
        ['SPRR2B','STK3'],
        ['SPRR2E','STK3'],
        ['SPSB1','TMEM201'],
        ['SPTB','UBAC2'],
        ['SRC','ZNF586'],
        ['SRC','ZNF587'],
        ['SREBF2','TBC1D22A'],
        ['SREK1','ST6GAL1'],
        ['SRFBP1','STK11'],
        ['SRGAP1','TMEM5'],
        ['SRP68','TRIM65'],
        ['SRP68','URM1'],
        ['SRRD','STK4'],
        ['SRRT','TRIP6'],
        ['SS18','TRAPPC8'],
        ['SSB','TLK1'],
        ['SSBP4','ZNF8'],
        ['SSH2','TANC2'],
        ['SSTR5','UBR5'],
        ['ST13','TRMU'],
        ['ST14','ST7'],
        ['ST3GAL3','TAL1'],
        ['ST6GAL1','TBCCD1'],
        ['STAB2','TDG'],
        ['STAG2','XIAP'],
        ['STAM','TMEM127'],
        ['STAM2','THSD7B'],
        ['STAP1','UBA6'],
        ['STAR','WHSC1L1'],
        ['STARD13','TNFRSF8'],
        ['STARD3','STRADA'],
        ['STARD7','UNC50'],
        ['STAT1','STAT4'],
        ['STAT5A','ZNF14'],
        ['STAU2','TP53BP1'],
        ['STAU2','TPD52'],
        ['STAU2','WWP1'],
        ['STIM1','VHL'],
        ['STIM1','YAP1'],
        ['STK10','UBE2D2'],
        ['STK11','TNRC6B'],
        ['STK11','TYK2'],
        ['STK24','TMEM159'],
        ['STK24','TMTC4'],
        ['STK24','TRIQK'],
        ['STK24','ZHX2'],
        ['STK3','SULT1A3'],
        ['STK3','ZHX2'],
        ['STK3','ZNF706'],
        ['STK38','TDRD7'],
        ['STK38','VEGFA'],
        ['STK38','ZNF184'],
        ['STK38L','WNK1'],
        ['STK39','TMEM245'],
        ['STK4','TSHZ2'],
        ['STMN2','TMEM67'],
        ['STON2','TJP2'],
        ['STPG1','USP48'],
        ['STRADA','TANC2'],
        ['STRN','USP49'],
        ['STX12','TMEM222'],
        ['STX5','TMEM223'],
        ['STX5','WDR74'],
        ['SUCO','ZCWPW2'],
        ['SUFU','WBP1L'],
        ['SUFU','ZCCHC24'],
        ['SUGCT','ZBED4'],
        ['SULF2','ZMYND8'],
        ['SULT1B1','YT521'],
        ['SULT1B1','YTHDC1'],
        ['SUPT6H','UTP6'],
        ['SUPV3L1','UBR1'],
        ['SUPV3L1','VPS26A'],
        ['SUSD1','UGCG'],
        ['SUSD1','ZNF618'],
        ['SVOPL','TRIM24'],
        ['SWAP70','WEE1'],
        ['SWAP70','ZNF143'],
        ['SYCP2','ZMYND8'],
        ['SYMPK','ZNF573'],
        ['SYNE1','ZBTB2'],
        ['SYNJ2','TULP4'],
        ['SYNJ2','ZDHHC14'],
        ['SYNPR','VGLL4'],
        ['SYPL1','TMEM168'],
        ['SYT1','USP15'],
        ['SYT1','YAF2'],
        ['SYTL2','TMEM135'],
        ['SYTL5','TCF3'],
        ['SYTL5','TCF7L1'],
        ['TACC2','TCF7L2'],
        ['TACC3','WHSC1'],
        ['TADA2A','THRA'],
        ['TAF3','ZDHHC14'],
        ['TAF3','ZNF197'],
        ['TAF4','TCFL5'],
        ['TAF4','TSHZ2'],
        ['TAF6','TRIM4'],
        ['TAF7L','UNC13B'],
        ['TAF8','USP49'],
        ['TALDO1','ZNFX1'],
        ['TAMM41','VGLL4'],
        ['TANGO6','WWP2'],
        ['TAOK1','VMP1'],
        ['TAOK1','ZNF207'],
        ['TAOK1','ZNF385C'],
        ['TAP2','VARS'],
        ['TARM1','ZNF83'],
        ['TARS','ZFR'],
        ['TARS2','VPS45'],
        ['TARS2','VPS45A'],
        ['TAS1R1','ZBTB48'],
        ['TASP1','ZNF18'],
        ['TBC1D13','ZER1'],
        ['TBC1D22A','TBCE'],
        ['TBC1D22B','YPEL1'],
        ['TBC1D4','TRAPPC12'],
        ['TBC1D5','TOP2B'],
        ['TBC1D9B','TENM2'],
        ['TBC1D9B','TRPS1'],
        ['TBCD','ZNF750'],
        ['TBCK','USP10'],
        ['TBCK','USP32'],
        ['TBL1XR1','USP13'],
        ['TBX4','TRIM25'],
        ['TC2N','TRIP11'],
        ['TCF12','TEX9'],
        ['TCF12','TMEM255B'],
        ['TCF7L1','WIF1'],
        ['TCTEX1D2','TFRC'],
        ['TEAD1','UBAP2'],
        ['TECPR2','UBE2E3'],
        ['TEKT1','TMEM41B'],
        ['TEKT1','TOP3A'],
        ['TENM2','WWC1'],
        ['TENM4','TMTC2'],
        ['TERT','TRIO'],
        ['TERT','TTLL7'],
        ['TESK2','UTS2'],
        ['TEX2','TTYH2'],
        ['TFDP1','TMCO3'],
        ['TFDP1','TUBGCP3'],
        ['TG','WNK3'],
        ['TGFA','TIA1'],
        ['TGFB1','TMEM91'],
        ['TGFBR2','XPC'],
        ['TGFBRAP1','TMEM131'],
        ['TGM5','UBR1'],
        ['TGS1','TNRC18'],
        ['THADA','TRA2A'],
        ['THAP10','TLE3'],
        ['THAP2','TMTC3'],
        ['THAP8','WDR62'],
        ['THUMPD2','VSNL1'],
        ['TICAM1','TLE2'],
        ['TIMP2','UBALD2'],
        ['TIMP3','ZNFX1'],
        ['TJAP1','TRIM26'],
        ['TJAP1','ZNF173'],
        ['TJP2','TMEM38B'],
        ['TJP3','ZNF554'],
        ['TK1','TMC6'],
        ['TM9SF4','TPX2'],
        ['TMBIM4','TMEM131'],
        ['TMCC2','ZC3H11A'],
        ['TMED10','TTLL10'],
        ['TMED10','TTLL5'],
        ['TMEM104','TRPM7'],
        ['TMEM107','WRAP53'],
        ['TMEM11','UBAP1'],
        ['TMEM117','ZCRB1'],
        ['TMEM127','ZAP70'],
        ['TMEM132C','TRERF1'],
        ['TMEM150B','ZNF587'],
        ['TMEM159','UBFD1'],
        ['TMEM168','TRRAP'],
        ['TMEM19','TSPAN8'],
        ['TMEM204','TSC2'],
        ['TMEM242','ZDHHC14'],
        ['TMEM245','TRPC4AP'],
        ['TMEM248','TYW1'],
        ['TMEM254','ZNF717'],
        ['TMEM39B','WDTC1'],
        ['TMEM41B','ZNF143'],
        ['TMEM63B','TTK'],
        ['TNFRSF1B','VPS13D'],
        ['TNIP1','UNK'],
        ['TNK1','ZBTB7A'],
        ['TNNI3','ZNF607'],
        ['TNRC6B','ZC3H7B'],
        ['TNS3','ZNRF2'],
        ['TOB1','TTLL6'],
        ['TOM1L1','TUBD1'],
        ['TOM1L2','TTC19'],
        ['TOM1L2','ULK2'],
        ['TOP1MT','ZC3H3'],
        ['TOP2A','TTBK2'],
        ['TOR3A','XPR1'],
        ['TOX2','TRPC4AP'],
        ['TP53BP2','WDR26'],
        ['TP53RK','ZNF227'],
        ['TP63','ZMAT3'],
        ['TPRG1','XXYLT1'],
        ['TPST1','WARS2'],
        ['TPST1','ZSCAN25'],
        ['TPX2','UNC5D'],
        ['TRAF7','WDR24'],
        ['TRAM1','ZFPM2'],
        ['TRERF1','VARS'],
        ['TRERF1','VARS2'],
        ['TRERF1','VARSL'],
        ['TRIM2','WDR37'],
        ['TRIM24','ULK4'],
        ['TRIM25','TRIM37'],
        ['TRIM28','ZBTB45'],
        ['TRIM29','UVRAG'],
        ['TRIM37','TUBD1'],
        ['TRIM8','WBP1L'],
        ['TRIO','UHRF2'],
        ['TRIOBP','UNC13C'],
        ['TRMU','TTC38'],
        ['TRPM7','USP8'],
        ['TRPS1','ZNRF2'],
        ['TSEN54','WWP2'],
        ['TSHR','TTLL10'],
        ['TSHR','TTLL5'],
        ['TSHZ2','UAP1'],
        ['TSHZ2','USP35'],
        ['TSPAN15','VPS26A'],
        ['TSPAN8','WNT11'],
        ['TSPO','TTLL1'],
        ['TTBK1','ZNF254'],
        ['TTC12','YAP1'],
        ['TTC38','UNC5A'],
        ['TTLL11','VWCE'],
        ['TTLL3','ZC3H4'],
        ['TTR','TYRO3'],
        ['TUBA1A','TUBA1C'],
        ['TUBA1B','TUBA1C'],
        ['TUBAL3','WDR37'],
        ['TUBD1','USP32'],
        ['TUBD1','VMP1'],
        ['TULP4','ZDHHC14'],
        ['TXNDC11','ZC3H7A'],
        ['TXNRD1','USP30'],
        ['TYW3','XPR1'],
        ['UBA6','UXT'],
        ['UBAC2','WAC'],
        ['UBAP1','UBAP2'],
        ['UBAP1','ZCCHC7'],
        ['UBAP2','USP34'],
        ['UBE2H','ZC3HC1'],
        ['UBE2O','USP32'],
        ['UBE2S','ZNF286A'],
        ['UBE2Z','UTP18'],
        ['UBE2Z','YPEL2'],
        ['UBE2Z','ZNF652'],
        ['UBL3','ZNF211'],
        ['UBL3','ZNF552'],
        ['UBXN11','UGCG'],
        ['ULK2','UQCR11'],
        ['UMODL1','WRB'],
        ['UNC79','ZFP36L1'],
        ['UNK','ZNF266'],
        ['URI1','VSTM2B'],
        ['URI1','XRCC1'],
        ['URI1','ZNF383'],
        ['USP10','ZDHHC7'],
        ['USP32','WIPF2'],
        ['USP32','ZDHHC7'],
        ['USP34','WDR92'],
        ['USP34','ZNF638'],
        ['UTP11L','ZMYM4'],
        ['UVRAG','ZMAT4'],
        ['VAC14','WWOX'],
        ['VARS','VMP1'],
        ['VAV1','ZNF823'],
        ['VAV3','ZHX2'],
        ['VCL','ZSWIM8'],
        ['VEPH1','WDR49'],
        ['VIPR2','WDR60'],
        ['VOPP1','ZNF679'],
        ['VPS13D','ZBTB40'],
        ['VPS53','ZZEF1'],
        ['VPS54','WDPCP'],
        ['VPS9D1','ZNF827'],
        ['VRK1','ZFHX4'],
        ['VSIG10','WSB2'],
        ['VWA8','ZMYM2'],
        ['WDR34','ZER1'],
        ['WHSC1L1','ZMAT4'],
        ['WIZ','ZNF431'],
        ['WNK1','WNT5B'],
        ['WWP2','ZNF629'],
        ['XBP1','ZNRF3'],
        ['XBP1P1','ZNRF3'],
        ['XH2','ZNF304'],
        ['XRCC6BP1','YEATS4'],
        ['YIPF1','YTHDF2'],
        ['YPEL2','ZNF148'],
        ['YY1','ZC3H3'],
        ['ZBTB45','ZFR2'],
        ['ZBTB7C','ZCCHC7'],
        ['ZC4H2','ZMYND8'],
        ['ZNF141','ZNF718'],
        ['ZNF182','ZNF81'],
        ['ZNF236','ZNF627'],
        ['ZNF335','ZSWIM3'],
        ['ZNF578','ZNF880'],
        ['ZNF699','ZSWIM4'],
        ['ABCD3','DPYD'],
        ['ACPP','AXL'],
        ['ACTG2','ALK'],
        ['ADAMTS16','TERT'],
        ['AKAP13','RET'],
        ['ATM','CARD18'],
        ['ATRX','BCL2'],
        ['ATXN1','ARG1'],
        ['B2M','GNAS'],
        ['B4GALT1','RAF1'],
        ['B4GALT','RAF1'],
        ['BRAF','SND1'],
        ['BRD3','LCN2'],
        ['CAB39','ERBB2'],
        ['CAPZA2','MET'],
        ['CASZ1','MTOR'],
        ['CBL','UBE4A'],
        ['CCDC132','CDH1'],
        ['CDC27','BRAF'],
        ['CDH1','CCDC132'],
        ['CDKN2A','CAPZB'],
        ['CDKN2A','SRGAP2'],
        ['CEP85L','ROS1'],
        ['CHD1','MTOR'],
        ['CLCN6','RAF1'],
        ['CLIP4','ALK'],
        ['CLU','CST3'],
        ['CPA6','PTK2'],
        ['DAB1','IL12RB2'],
        ['DDI2','MTOR'],
        ['DHX15','ETV1'],
        ['DOCK8','JAK2'],
        ['DOM','TECR'],
        ['DOT1L','TECR'],
        ['ECHD1','FYN'],
        ['ECHDC1','FYN'],
        ['EGFR','C7ORF72'],
        ['EIF2C2','PTK2'],
        ['EPHB2','NTRK1'],
        ['ERBB2','PPP1R1B'],
        ['ERBB2','SLC29A3'],
        ['ERC1','RET'],
        ['ESR1','CCDC170'],
        ['EXOC4','BRAF'],
        ['EXOSC10','MTOR'],
        ['FAT1','NTRK3'],
        ['FGFR2','BICC1'],
        ['FGFR3','AES'],
        ['FGFR3','ELAVL3'],
        ['FGFR3','FBXO28'],
        ['FGFR3','TACC3'],
        ['FHIT','RUNX1'],
        ['FKBP15','RET'],
        ['FRS2','MDM2'],
        ['GFAP','VIM'],
        ['GLIS3','TERT'],
        ['GTF2IRD1','ALK'],
        ['GUCA2A','ERG'],
        ['HIF1A','PRKCH'],
        ['HLA-C','MUC16'],
        ['HMHA1','STK11'],
        ['HNF1B','NOTCH1'],
        ['HOOK3','IKBKB'],
        ['HPRT1','CTPS2'],
        ['HTATSF1','BRS3'],
        ['IGFBP2','SPP1'],
        ['IL12RB2','DAB1'],
        ['IL6R','C1ORF112'],
        ['IRF2BP2','NTRK1'],
        ['JAK2','RCL1'],
        ['JHDM1D','BRAF'],
        ['KANK2','ALK'],
        ['KAZN','MTOR'],
        ['KCMF1','PRKDC'],
        ['KCNQ5','ALK'],
        ['KIF5B','MET'],
        ['KRIT1','CDK6'],
        ['LMNA','RAF1'],
        ['LPP','CASA'],
        ['LPP','CASR'],
        ['LYN','NTRK3'],
        ['MACF1','BRAF'],
        ['MAPK14','EFHA1'],
        ['MDM2','BRI3BP'],
        ['MDM2','EGFR'],
        ['MDM2','SPATS2'],
        ['MEMO1','ALK'],
        ['MET','TFG'],
        ['MKRN1','BRAF'],
        ['MLL','FYN'],
        ['MLLT6','ACE'],
        ['MTMR12','TERT'],
        ['MUC16','OR7G2'],
        ['MX1','ERG'],
        ['NARS2','TOP1'],
        ['NCOA1','ALK'],
        ['NFASC','NTRK1'],
        ['NOTCH2','EIF2B3'],
        ['NOTCH2','MNDA'],
        ['NTRK1','DYNC2H1'],
        ['NUP214','ABL1'],
        ['OXR1','MET'],
        ['PAPD7','SRD5A1'],
        ['PDCD6','TERT'],
        ['PEX1','CDK6'],
        ['PHKB','PDE3A'],
        ['PL51','PIK3CB'],
        ['PLS1','PIK3CB'],
        ['PM1','ALK'],
        ['PRKACA','AKT1'],
        ['PRKACA','AKT2'],
        ['PRKAR1A','ALK'],
        ['PRKAR1','ALK'],
        ['PRKAR2A','RHOA'],
        ['PTK2','TRAPPC9'],
        ['RAB11B','MDK'],
        ['RAF1','C9ORF153'],
        ['RAF1','TMEM40'],
        ['RARA','HOXB3'],
        ['RARA','SLC9A3R1'],
        ['RB1','GADD45GIP1'],
        ['RLINX1','CEP76'],
        ['RPS6','FN1'],
        ['RPS6KB1','VMP1'],
        ['RSBN1L','HGF'],
        ['RUNX1','CEP76'],
        ['SEC16A','NOTCH1'],
        ['SEC16A','RXRA'],
        ['SHANK3','MAPK1'],
        ['SLC12A7','TERT'],
        ['SLC7A2','TERT'],
        ['SMARCA4','EEF2'],
        ['SMEK2','ALK'],
        ['SND1','BRAF'],
        ['SPECC1L','RET'],
        ['SQSTM1','NTRK1'],
        ['SRD5A1','PAPD7'],
        ['SSBP2','NTRK1'],
        ['STARD13','TNFRSF8'],
        ['STAT3','ETV4'],
        ['STK11','HMHA1'],
        ['STRN','ALK'],
        ['TAOK1','RARA'],
        ['TAX1BP1','BRAF'],
        ['TBL1XR1','CTNNB1'],
        ['TBL1XR1','PIK3CA'],
        ['TBL1XR1','RET'],
        ['TFG','MET'],
        ['THRA','CDK12'],
        ['TICAM1','IL12RB1'],
        ['TMPRSS2','BRAF'],
        ['TMPRSS2','GUCA2A'],
        ['TMPRSS2','TMEM109'],
        ['TOP1','C17ORF64'],
        ['TP53','KIAA0753'],
        ['TPM1','ALK'],
        ['TRAK1','RAF1'],
        ['TRAK','RAF1'],
        ['TRAPPC9','PTK2'],
        ['TRIM24','BRAF'],
        ['TRIO','TERT'],
        ['TRPM8','UGT1A9'],
        ['TSPAN3','MDM2'],
        ['TTC13','JAK2'],
        ['TTLL7','TERT'],
        ['UACA','LTK'],
        ['USP46','PDGFRA'],
        ['USP7','PRKCB'],
        ['VEGFA','STK38'],
        ['VIM','GFAP'],
        ['VMP1','RPS6KB1'],
        ['WNK2','BRD3'],
        ['WRN','ADAM9'],
        ['YWHAE','MAP2K2'],
        ['ZBTB17','MTOR'],
        ['ZC3HAV1','BRAF'],
        ['ZNF226','AKT2']


]



    data = fusions.get(options.organism.lower(),[])
    if data:

        #file_symbols = os.path.join(options.output_directory,'genes_symbols.txt')
        file_symbols = os.path.join(options.output_directory,'synonyms.txt')
        loci = symbols.generate_loci(file_symbols)

        genes = symbols.read_genes_symbols(file_symbols)

        d = []
        for g in data:
            if len(g) != 2:
                print "ERROR:",g
                sys.exit(1)
            g1 = g[0]
            g2 = g[1]
            if not (g1 and g2):
                print "ERROR:",g
                sys.exit(1)
            if g1.upper() != g2.upper():
                ens1 = symbols.ensembl(g1.upper(),genes,loci)
                ens2 = symbols.ensembl(g2.upper(),genes,loci)
                if ens1 and ens2:
                    for e1 in ens1:
                        for e2 in ens2:
                            if e1 != e2:
                                d.append([e1,e2])

        data = ['\t'.join(sorted(line)) + '\n' for line in d]
        data = list(set(data))

        print "%d known fusion genes found in manually currated database" % (len(data),)

        if not options.skip_filter_overlap:
            d1 = []
            overlappings = ['ensembl_fully_overlapping_genes.txt',
                            'ensembl_same_strand_overlapping_genes.txt',
                            'refseq_fully_overlapping_genes.txt',
                            'refseq_same_strand_overlapping_genes.txt',
                            'ucsc_fully_overlapping_genes.txt',
                            'ucsc_same_strand_overlapping_genes.txt',
                            'pairs_pseudogenes.txt',
                            'paralogs.txt']
            for ov in overlappings:
                p = os.path.join(options.output_directory,ov)
                print "Parsing file:",p
                if os.path.isfile(p):
                    d2 = sorted(set([tuple(sorted(line.rstrip('\r\n').split('\t'))) for line in file(p,'r').readlines() if line.rstrip('\r\n')]))
                d1.extend(d2)
            d = set()
            for line in d1:
                (a,b) = (line[0],line[1])
                if a > b:
                    (a,b) = (b,a)
                d.add("%s\t%s\n" % (a,b))
            skipped = [line for line in data if line in d]
            data = [line for line in data if not line in d]
            file(os.path.join(options.output_directory,'tcga_known_but_overlapping.txt'),'w').writelines(sorted(skipped))

            print "%d known fusion genes left after removing the overlappings" % (len(data),)

    file(os.path.join(options.output_directory,'tcga.txt'),'w').writelines(sorted(data))
    #
