#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It generates the list of candidate fusion genes from TCGA. This list is hard coded
in here and it is manually curated from:


Vellichirammal et al., Pan-Cancer Analysis Reveals the Diverse Landscape 
of Novel Sense and Antisense Fusion Transcripts,  Mol Ther Nucleic Acids, Vol. 19, 
pp. 1379-1398, March 06, 2020, https://doi.org/10.1016/j.omtn.2020.01.023




Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse
import symbols

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It generates the list of pre-candidate fusion genes from CCLE."""
    version = "%prog 0.12 beta"

    parser = optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--organism",
                      action = "store",
                      type = "string",
                      dest = "organism",
                      default = "homo_sapiens",
                      help="""The name of the organism for which the list of allowed candidate fusion genes is generated, e.g. homo_sapiens, mus_musculus, etc. Default is '%default'.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_directory",
                      default = '.',
                      help="""The output directory where the list of allowed candidate fusion genes is generated. Default is '%default'.""")

    parser.add_option("--skip-filter-overlap",
                      action="store_true",
                      dest="skip_filter_overlap",
                      default = False,
                      help="""If set then it filters out the known fusion genes where the (i) genes are fully overlapping, or (ii) the genes are partially overlapping and are on the same strand. Default is '%default'.""")

    (options,args) = parser.parse_args()

    # validate options
    if not (options.output_directory
            ):
        parser.print_help()
        sys.exit(1)


    #
    #
    #

    print "Generating the list of TCGA fusion genes..."
    fusions = dict()

    # manual curation from papers

    fusions['rattus_norvegicus'] = []

    fusions['mus_musculus'] = []

    fusions['canis_lupus_familiaris'] = []

    fusions['homo_sapiens'] = [
        ['MRGPRX3','SMOX'],
        ['SLCO1A2','LOC100996634'],
        ['CHRM3','TDRD5'],
        ['CHRM3','EYS'],
        ['RAB3IP','MYLK'],
        ['TMPRSS2','ERG'],
        ['RAB3IP','COL28A1'],
        ['MRGPRX3','HYAL1'],
        ['PRKAR1A','CBWD6'],
        ['GBP5','FBXO17'],
        ['RAB3IP','EXOC4'],
        ['KIAA1324L','CBWD6'],
        ['IL34','C6ORF99'],
        ['RAB3IP','TDRD5'],
        ['RAB3IP','GPHN'],
        ['RAB3IP','ZNF638'],
        ['FOCAD','SLCO1A2'],
        ['RAB3IP','ACVR1C'],
        ['ASTN2','TMEM212'],
        ['SLC37A2','GBP5'],
        ['PRKAR1A','ALG1L'],
        ['SLC37A2','ZNF80'],
        ['SLCO1A2','SMYD3'],
        ['ASTN2','ABCC9'],
        ['WDR72','CBWD6'],
        ['CHRM3','C3ORF67'],
        ['CCL22','C14ORF2'],
        ['CHRM3','FGGY'],
        ['MAPKAPK5','ACAD10'],
        ['CDYL','TDRD5'],
        ['CCDC6','RET'],
        ['LY6G5B','APPBP2'],
        ['HIC2','PI4KA'],
        ['CDYL','WDR72'],
        ['CHRM3','MYLK'],
        ['RAB3IP','COMMD10'],
        ['ADGRL1','SUZ12'],
        ['CHRM3','EXOC4'],
        ['HYAL1','MRGPRX3'],
        ['CES3','CHM'],
        ['CHRM3','SLC44A5'],
        ['SLCO1A2','SLC44A5'],
        ['MRGPRX3','F2RL1'],
        ['KIAA1324L','MCCC2'],
        ['FOXK1','SLC25A23'],
        ['ARL15','NDUFS4'],
        ['WDR72','MCCC2'],
        ['FGFR3','TACC3'],
        ['DNAL1','DYX1C1'],
        ['GOSR1','PROCR'],
        ['TESPA1','GBP4'],
        ['CHRM3','BCAS3'],
        ['GOSR1','GTF2IRD2'],
        ['CHRM3','CBWD6'],
        ['CHRM3','MCCC2'],
        ['NSF','LRRC37A2'],
        ['RNF14','KIAA1804'],
        ['AP5Z1','GBP5'],
        ['ZNF404','GOSR1'],
        ['IGSF11','FANK1'],
        ['RAB3IP','SCFD1'],
        ['NUPR2','PSPH'],
        ['CPSF6','C9ORF3'],
        ['C1ORF186','UGT2B15'],
        ['CHRM3','SP140L'],
        ['KIAA1324L','SPATA6L'],
        ['KMT2E','LHFPL3'],
        ['ETV6','NTRK3'],
        ['RAB3IP','DNER'],
        ['UPK3B','LOC100289561'],
        ['CHRM3','CD96'],
        ['IL34','EPHX1'],
        ['CBFB','MYH11'],
        ['COL1A2','ACTB'],
        ['SH3PXD2A','OBFC1'],
        ['RAB3IP','FGGY'],
        ['MPV17L','DYX1C1'],
        ['PRKAR1A','SPATA6L'],
        ['PTPRK','RSPO3'],
        ['SLCO1A2','SCAMP1'],
        ['ALG1L','APBA1'],
        ['ACTB','FN1'],
        ['GAPDH','ACTB'],
        ['PML','RARA'],
        ['ARSF','GNB4'],
        ['FLNB','SLMAP'],
        ['KIF26B','SMYD3'],
        ['GOSR1','C4ORF51'],
        ['KANK4','SCPEP1'],
        ['MRGPRX3','WDR66'],
        ['NPEPPS','TBC1D3B'],
        ['WDR72','SPATA6L'],
        ['PILRB','PVRIG'],
        ['GOSR1','HYAL1'],
        ['GPC6','GPC5'],
        ['MPP5','GPHN'],
        ['NFIC','CELF5'],
        ['PKD1','NPIPA8'],
        ['POLR2J2','LOC100289561'],
        ['ABHD12B','CASP10'],
        ['BBS5','GOLGA7B'],
        ['BPTF','LRRC37A'],
        ['RAB3IP','KAT2B'],
        ['REV3L','FYN'],
        ['CHRM3','SPATA6L'],
        ['COMMD10','AP3S1'],
        ['GAS7','MYH1'],
        ['GOSR1','STAP2'],
        ['KIAA1324L','ALG1L'],
        ['KIAA1324L','TBCK'],
        ['NPEPPS','TBC1D3F'],
        ['NPEPPS','TBC1D3G'],
        ['NPEPPS','TBC1D3H'],
        ['NPEPPS','TBC1D3I'],
        ['NPEPPS','TBC1D3K'],
        ['NPEPPS','TBC1D3L'],
        ['ASTN2','ODF2L'],
        ['COL14A1','DEPTOR'],
        ['GOSR1','C1ORF64'],
        ['ESR1','CCDC170'],
        ['HHLA2','EPHX1'],
        ['SLC37A2','KCNN2'],
        ['TMEM87B','MERTK'],
        ['CPSF6','CHMP1A'],
        ['EHBP1','COMMD1'],
        ['GTF2I','GTF2IRD1'],
        ['NCOR2','SCARB1'],
        ['OPHN1','MAB21L3'],
        ['POLR2J3','LOC100289561'],
        ['PRKAR1A','CBWD3'],
        ['EXT1','SAMD12'],
        ['GJB7','C12ORF56'],
        ['LTBP1','BIRC6'],
        ['TCF7L2','VTI1A'],
        ['WDR72','TBCK'],
        ['EGFR','SEPT14'],
        ['EML4','ALK'],
        ['HYAL1','GOSR1'],
        ['KIAA1324L','ADGRV1'],
        ['PRKAR1A','MCCC2'],
        ['RAB3IP','CNTLN'],
        ['AFF1','PTPN13'],
        ['CDYL','CRISPLD1'],
        ['COL1A1','FTH1'],
        ['GOSR1','C1ORF186'],
        ['PPDPF','FAM83H'],
        ['SEMA4F','FAHD1'],
        ['TPD52L1','RNF217'],
        ['ACTG1','COL1A1'],
        ['COL1A1','KRT19'],
        ['COL1A1','RPLP1'],
        ['CTSC','ASTN2'],
        ['FAM129C','LPGAT1'],
        ['FTH1','ACTG1'],
        ['HLA-G','HLA-B'],
        ['NPIPB9','NPIPA2'],
        ['NPIPB9','NPIPA3'],
        ['NPIPB9','NPIPA5'],
        ['RUNX1','RUNX1T1'],
        ['UBTF','MAML3'],
        ['B4GALT1','BAG1'],
        ['DNAJB6','UBE3C'],
        ['LRBA','SH3D19'],
        ['MKL1','SLC25A17'],
        ['NSF','LRRC37A'],
        ['POLR1A','REEP1'],
        ['PPP1CB','PLB1'],
        ['ACOT9','APOO'],
        ['MECOM','RPL22'],
        ['NOTCH2','SEC22B'],
        ['ZCCHC14','RERE'],
        ['FRYL','OCIAD2'],
        ['KIAA1324L','MRPS31'],
        ['MYO6','SENP6'],
        ['PRKAR1A','ADGRV1'],
        ['SLCO1A2','KIAA1324L'],
        ['TFDP1','CUL4A'],
        ['WDR72','MRPS31'],
        ['ZAK','RAPGEF4'],
        ['ZBTB20','LSAMP'],
        ['ASTN2','OPHN1'],
        ['CHRM3','NEO1'],
        ['PDE4D','DEPDC1B'],
        ['RPL8','COL1A1'],
        ['RPLP0','COL1A1'],
        ['ZNF780A','MRPL15'],
        ['ACTG1','FN1'],
        ['BOLA2B','SMG1'],
        ['BOLA2','SMG1'],
        ['CAMK1D','TMPO'],
        ['CHRM3','SVEP1'],
        ['COL1A1','COL5A2'],
        ['COL1A1','RPL8'],
        ['COL1A1','TXNIP'],
        ['COL1A2','GAPDH'],
        ['ESR1','TMEM212'],
        ['FAM129C','REEP3'],
        ['IL34','DNAL1'],
        ['OPHN1','SHISA9'],
        ['RPL8','FN1'],
        ['SLC45A3','ERG'],
        ['SPP2','ABCC9'],
        ['TMEFF2','EPHX1'],
        ['ANKHD1-EIF4EBP3','CYSTM1'],
        ['BPTF','PITPNC1'],
        ['CCDC6','ANK3'],
        ['CCND3','TRERF1'],
        ['CMIP','BCO1'],
        ['DNM2','SLC44A2'],
        ['IQGAP1','SEMA4B'],
        ['IQGAP1','ZNF774'],
        ['KIF1B','UBE4B'],
        ['MARK2','C11ORF84'],
        ['NFAT5','CYB5B'],
        ['NSD1','ZNF346'],
        ['PDE4DIP','EIF1AX'],
        ['PEAK1','LINGO1'],
        ['RAB3IP','SPATA6L'],
        ['RAF1','TMEM40'],
        ['RASAL1','DDX54'],
        ['RCOR1','TECPR2'],
        ['SMCHD1','NDC80'],
        ['STK24','SLC15A1'],
        ['TRIP12','SLC16A14'],
        ['ACOX1','SRP68'],
        ['AMMECR1L','MINPP1'],
        ['ASTN2','F5'],
        ['C7ORF50','MAD1L1'],
        ['EP400','SFSWAP'],
        ['FCHSD2','FAM168A'],
        ['FRS2','LYZ'],
        ['GOSR1','DEFB124'],
        ['KAT6B','ADK'],
        ['MAP4','SPINK8'],
        ['MLF2','YWHAG'],
        ['MRGPRX3','C4ORF51'],
        ['NADSYN1','CASP10'],
        ['NADSYN1','GBP4'],
        ['PDE4D','ELOVL7'],
        ['PLPP1','SLC38A9'],
        ['PPP6R3','LRP5'],
        ['PRKAR1A','COMMD10'],
        ['PROSC','CHMP1A'],
        ['RBM14','RBM4B'],
        ['RPS29','PIGF'],
        ['ST7','CAPZA2'],
        ['TNIP1','ANXA6'],
        ['TNRC18','RNF216'],
        ['ADGRF1','TNFRSF21'],
        ['ALG1L','ISPD'],
        ['CHD7','TOX'],
        ['CHRM3','SLC25A43'],
        ['COL1A1','CD74'],
        ['EPHX1','IL34'],
        ['FAM83H','PPDPF'],
        ['GOSR1','SLC52A1'],
        ['MTHFS','BCL2A1'],
        ['MUC1','COL1A1'],
        ['PACSIN2','TTLL1'],
        ['RAB3IP','GRM5'],
        ['SFPQ','TFE3'],
        ['SMARCC1','MAP4'],
        ['WASF2','AHDC1'],
        ['ACTB','XBP1'],
        ['B2M','UBC'],
        ['BBS5','GOSR1'],
        ['BRIP1','CPT1A'],
        ['CFAP57','KANK4'],
        ['COL1A1','ACTG1'],
        ['COL1A1','CD24'],
        ['COL1A1','COL6A3'],
        ['COL1A1','FLNB'],
        ['COL1A1','MTRNR2L2'],
        ['COL1A1','POSTN'],
        ['COL1A1','RPL4'],
        ['COL1A1','RPS9'],
        ['COL1A1','XBP1'],
        ['COL1A2','HLA-B'],
        ['COL1A2','HSP90AB1'],
        ['COL3A1','PPIA'],
        ['EEF2','FN1'],
        ['FN1','ACTB'],
        ['FN1','ACTG1'],
        ['FN1','CTSD'],
        ['FN1','RPL41'],
        ['GAPDH','SPARC'],
        ['HSP90AB1','HSP90AA1'],
        ['HSPA1B','HSPA1A'],
        ['KCNE4','FNBP1'],
        ['NPIPB9','NPIPA1'],
        ['RBM8A','ATL3'],
        ['RCC1','UBE2D2'],
        ['RPL13A','XBP1'],
        ['RPS18','FN1'],
        ['SHISA9','MAB21L3'],
        ['SLC39A6','COL3A1'],
        ['SLC7A5','SMG1'],
        ['SLCO1A2','TDRD5'],
        ['SPARC','ACTG1'],
        ['SS18','SSX2'],
        ['BCR','ABL1'],
        ['ABCC9','MAB21L3'],
        ['ACOT7','DNAJC11'],
        ['ADAP1','SUN1'],
        ['AGBL4','FAF1'],
        ['AGFG1','MFF'],
        ['AKAP13','GOLGA6L4'],
        ['ALDH16A1','TRMT2B'],
        ['ANKHD1','CYSTM1'],
        ['ARHGEF10L','PADI4'],
        ['ASTN2','USP33'],
        ['ATAD2','WDYHV1'],
        ['ATF7','ATP5G2'],
        ['BANP','SNX29'],
        ['BAZ1B','MLXIPL'],
        ['BPTF','NOL11'],
        ['CFLAR','WBSCR27'],
        ['CHRM3','CNTLN'],
        ['CNOT1','GOT2'],
        ['CSNK1D','SECTM1'],
        ['DHRS4L1','ODF2L'],
        ['DLG1','BDH1'],
        ['EIF1AX','PDE4DIP'],
        ['EIF4G3','ECE1'],
        ['ERBB2','PPP1R1B'],
        ['ERC1','WNK1'],
        ['FAM162A','CASR'],
        ['FAM168A','RAB6A'],
        ['FBRSL1','NOC4L'],
        ['FBXL20','GSDMB'],
        ['FRA10AC1','TMEM255B'],
        ['GALNT8','KCNA6'],
        ['GIGYF2','EFHD1'],
        ['HEG1','SLC12A8'],
        ['HIF1A','PRKCH'],
        ['ITPK1','BTBD7'],
        ['KDM2A','C11ORF80'],
        ['KDM5A','NINJ2'],
        ['KIAA1324L','SLCO1A2'],
        ['LAMA5','RBBP8NL'],
        ['LMBR1','PTPRN2'],
        ['MGAT4A','KIAA1211L'],
        ['NAIP','GTF2H2'],
        ['NCOA3','EYA2'],
        ['NF1','RNF135'],
        ['NOTCH2NL','SEC22B'],
        ['NOVA1','FUS'],
        ['PARD3','CUL2'],
        ['PHRF1','PTDSS2'],
        ['PRKAR1A','CBWD5'],
        ['PTK2','SLC45A4'],
        ['PTPN12','CCDC146'],
        ['QKI','PACRG'],
        ['RAB3IP','BEST3'],
        ['RAB3IP','PRB3'],
        ['RAB3IP','ZNF622'],
        ['RCAN1','RUNX1'],
        ['RERE','ENO1'],
        ['RHPN2','PEPD'],
        ['SEMA4B','ZNF710'],
        ['SEMA4D','GBP5'],
        ['SIL1','CTNNA1'],
        ['SLC39A14','PIWIL2'],
        ['SLC7A5','CA5A'],
        ['SMAD7','DYM'],
        ['SMARCA4','CARM1'],
        ['SMYD3','ZNF695'],
        ['SNTB2','TANGO6'],
        ['SNTB2','UTP4'],
        ['SRPK2','PUS7'],
        ['ST6GAL1','KNG1'],
        ['TANC1','PKP4'],
        ['TBCD','FOXK2'],
        ['TMEM120B','ORAI1'],
        ['TRAPPC10','AGPAT3'],
        ['UBE2H','ZC3HC1'],
        ['UBE2Q2','GOLGA6L10'],
        ['VAV2','BRD3'],
        ['ZSWIM4','CCDC130'],
        ['ABHD6','FLNB'],
        ['ACACB','UNG'],
        ['ARL8B','ITPR1'],
        ['ARMC5','B3GNT7'],
        ['B3GLCT','TEX26'],
        ['BBS5','PROCR'],
        ['BBS5','SMOX'],
        ['BRAF','MRPS33'],
        ['C1QTNF9','MIPEP'],
        ['CFLAR','NDUFB3'],
        ['CHRM3','SMYD3'],
        ['CSNK1D','OGFOD3'],
        ['CTNND1','TMX2'],
        ['EPB41L1','PHF20'],
        ['EPT1','HADHB'],
        ['ERC1','RET'],
        ['FAM104A','CDC42EP4'],
        ['FBXO34','LGALS3'],
        ['FGFR2','BICC1'],
        ['GALNT9','NOC4L'],
        ['GAPVD1','RABEPK'],
        ['GJB7','NKAIN2'],
        ['HYAL1','ASTN2'],
        ['IL13RA1','DOCK11'],
        ['MAGI1','SUCLG2'],
        ['MB21D1','NWD1'],
        ['METTL24','SLC22A16'],
        ['MRGPRX3','FILIP1'],
        ['NPIPA2','NPIPB9'],
        ['NPIPA3','NPIPB9'],
        ['NXN','ABR'],
        ['OPHN1','CHRM3'],
        ['PABPC1','COL1A1'],
        ['PCMTD1','ST18'],
        ['PDXDC1','NOMO2'],
        ['PDXDC1','NOMO3'],
        ['POLR2J3','RASA4B'],
        ['PPFIA1','ANO1'],
        ['PRKAR1A','TBCK'],
        ['PTEN','RNLS'],
        ['PTPN23','KLHL18'],
        ['RAB3IP','C14ORF37'],
        ['RAB3IP','PRIM2'],
        ['RBM28','KMT2C'],
        ['REPS2','TXLNG'],
        ['SEC23B','CSRP2BP'],
        ['SERINC2','ZCCHC17'],
        ['SLC12A7','TERT'],
        ['SLCO1A2','WDR72'],
        ['SOGA1','SAMHD1'],
        ['SPEN','FBLIM1'],
        ['STK24','DOCK9'],
        ['SULT1C4','GREB1'],
        ['TANC1','OR7E24'],
        ['TSPAN5','METAP1'],
        ['WDHD1','DLGAP5'],
        ['ZBTB7A','MAP2K2'],
        ['ACTA2','ACTB'],
        ['ACTB','DSP'],
        ['ACTG1','HSP90AB1'],
        ['ACTG1','IGLL5'],
        ['AEBP1','ACTG1'],
        ['ALDOA','COL1A2'],
        ['ALDOA','FN1'],
        ['ALDOA','SLC39A6'],
        ['ATL3','RBM8A'],
        ['B2M','HSPA1A'],
        ['BCAR1','CFDP1'],
        ['BGN','ACTB'],
        ['BGN','FN1'],
        ['BSG','EEF1A1'],
        ['CCL3','CCL18'],
        ['COL18A1','COL1A1'],
        ['COL1A1','CFB'],
        ['COL1A1','CTSD'],
        ['COL1A1','HSP90AA1'],
        ['COL1A1','HSP90AB1'],
        ['COL1A1','KRT8'],
        ['COL1A1','PLXNB2'],
        ['COL1A1','RPL10'],
        ['COL1A1','RPL12'],
        ['COL1A1','RPL15'],
        ['COL1A1','RPL7'],
        ['COL1A1','RPS24'],
        ['COL1A1','SERPINA3'],
        ['COL1A1','SPINT2'],
        ['COL1A1','TRIM28'],
        ['COL1A1','ZFP36L1'],
        ['COL1A2','CFL1'],
        ['COL1A2','HLA-A'],
        ['COL1A2','KRT19'],
        ['COL1A2','RPL10'],
        ['COL1A2','SERPINA3'],
        ['COL3A1','PKM'],
        ['COL6A3','COL3A1'],
        ['CTSD','HLA-B'],
        ['DESI1','DAPL1'],
        ['EEF1A1','EEF1A2'],
        ['EPCAM','FTH1'],
        ['FASN','FN1'],
        ['FLNA','FN1'],
        ['FN1','CFL1'],
        ['FN1','FTL'],
        ['FN1','RPL13A'],
        ['FN1','TPM4'],
        ['FTH1','HLA-B'],
        ['FTH1','HSP90AA1'],
        ['FTH1','HSPB1'],
        ['GAPDH','FN1'],
        ['GAPDH','PSAP'],
        ['GAPDH','RPS2'],
        ['GNAS','ACTB'],
        ['GNAS','COL1A1'],
        ['GNAS','COL3A1'],
        ['GNAS','HLA-B'],
        ['GNAS','SPARC'],
        ['GNPNAT1','DHFRL1'],
        ['HLA-A','COL1A2'],
        ['HLA-C','ACTB'],
        ['HLA-C','FN1'],
        ['HLA-C','KRT19'],
        ['HNRNPA1','COL3A1'],
        ['HSP90B1','COL1A1'],
        ['HSPA8','COL1A2'],
        ['KMT2A','ELL'],
        ['KRT19','ACTB'],
        ['KRT19','ENO1'],
        ['KRT19','SCGB2A2'],
        ['KRT8','FN1'],
        ['LANCL2','SEPT14'],
        ['LYZ','CD68'],
        ['MT2A','MT3'],
        ['MTMR2','MAML2'],
        ['MTRNR2L8','CD24'],
        ['MUC1','EEF1A1'],
        ['MXRA5','EEF1A1'],
        ['NPM1','PTMA'],
        ['PSAP','FN1'],
        ['PTMA','ACTB'],
        ['PTMA','NCL'],
        ['RCC1','ABHD12B'],
        ['RPL10','CPB1'],
        ['RPL37','COL1A2'],
        ['RPL7A','COL1A2'],
        ['RPL8','MGP'],
        ['RPLP0','ACTG1'],
        ['RPLP1','FTH1'],
        ['RPS19','COL1A1'],
        ['RPS2','ACTG1'],
        ['RPS2','COL1A1'],
        ['SLC39A6','FN1'],
        ['SMG1','NPIPB3'],
        ['SS18','SSX1'],
        ['TUBA1B','FN1'],
        ['UBC','ACTG1'],
        ['UGT2B17','TMEM212'],
        ['XBP1','ACTG1'],
        ['XBP1','SLC39A6'],
        ['ZFAND3','LYRM4'],
        ['ABCB8','NOS3'],
        ['ABCC4','UGGT2'],
        ['ABHD16A','VARS'],
        ['ABL1','FUBP3'],
        ['ABR','PITPNA'],
        ['ACO2','ZC3H7B'],
        ['ACOT7','PLEKHG5'],
        ['ACTN1','DCAF5'],
        ['ACTN4','EIF3K'],
        ['ADAM9','UNC5D'],
        ['ADAP1','C7ORF50'],
        ['AEBP2','PLEKHA5'],
        ['AFAP1','ABLIM2'],
        ['AGAP4','FRMPD2'],
        ['AGAP5','FRMPD2'],
        ['AGMO','MEOX2'],
        ['AGR2','IL6ST'],
        ['AHSA2','C2ORF74'],
        ['AKT2','YIF1B'],
        ['ALG1L','ADGRV1'],
        ['ANAPC5','KDM2B'],
        ['ANAPC7','ARPC3'],
        ['ANKRD11','CHMP1A'],
        ['ANKRD11','FANCA'],
        ['ANKRD34A','PYCR2'],
        ['ANKRD39','FAM178B'],
        ['ANKRD44','RFTN2'],
        ['AP2A2','DRD4'],
        ['AP2A2','TALDO1'],
        ['ARHGAP15','KYNU'],
        ['ARHGAP19','RRP12'],
        ['ARHGEF2','SSR2'],
        ['ARL17B','LRRC37A2'],
        ['ARMC8','CLDN18'],
        ['ARRB1','GDPD5'],
        ['ASCC1','MICU1'],
        ['ASH1L','GON4L'],
        ['ASTN2','CNOT2'],
        ['ASTN2','SLC2A14'],
        ['ASXL1','TM9SF4'],
        ['ATE1','NSMCE4A'],
        ['ATP11A','SPACA7'],
        ['ATP2A2','IFT81'],
        ['ATP2B4','OPTC'],
        ['ATP6V0E1','ERGIC1'],
        ['B4GALT7','NSD1'],
        ['BAIAP2L1','ANKIB1'],
        ['BBS5','C1ORF64'],
        ['BBS5','PLCB1'],
        ['BCAM','ACTB'],
        ['BCAS3','RNFT1'],
        ['BCAS3','VMP1'],
        ['BCL7A','PSMD9'],
        ['BICD1','KIAA1551'],
        ['BIRC6','SLC30A6'],
        ['BIRC6','YIPF4'],
        ['BUB1B','EIF2AK4'],
        ['C16ORF72','GOSR1'],
        ['C16ORF74','EMC8'],
        ['C2CD3','GOSR1'],
        ['C9ORF3','MFSD14B'],
        ['CALD1','CGN'],
        ['CALD1','EXOC4'],
        ['CAMK1D','CDC123'],
        ['CAMK2D','ANK2'],
        ['CANX','RUFY1'],
        ['CAPN13','GALNT14'],
        ['CAPZB','TMCO4'],
        ['CBX5','NFE2'],
        ['CCAR1','TET1'],
        ['CCSER1','MMRN1'],
        ['CD44','PDHX'],
        ['CDC42','CELA3A'],
        ['CDC42EP4','SDK2'],
        ['CDKL2','G3BP2'],
        ['CDR2L','HID1'],
        ['CDYL','NDUFA5'],
        ['CDYL','ZNF589'],
        ['CELF1','FNBP4'],
        ['CELSR1','CERK'],
        ['CELSR2','KIAA1324'],
        ['CELSR2','SARS'],
        ['CHD7','RAB2A'],
        ['CHMP4B','CBFA2T2'],
        ['CHRM3','BRE'],
        ['CHRM3','CCDC138'],
        ['CHRM3','COMMD10'],
        ['CHRM3','FSTL4'],
        ['CHRM3','IGSF11'],
        ['CHRM3','NDUFA5'],
        ['CHRM3','RPS6KA3'],
        ['CHRM3','SCRN3'],
        ['CHRM3','ST6GALNAC5'],
        ['CHRM3','TBCK'],
        ['CLIC4','SRRM1'],
        ['CLIP1','RSRC2'],
        ['CLIP1','ZCCHC8'],
        ['CLSTN3','C1S'],
        ['CNOT2','ASTN2'],
        ['CNOT2','NUP107'],
        ['CNOT2','ZDHHC17'],
        ['COBLL1','SLC38A11'],
        ['COL1A1','GNAS'],
        ['COL1A1','IRF2BP2'],
        ['CPM','GOLT1B'],
        ['CPSF6','JAG2'],
        ['CPT1A','ANPEP'],
        ['CRIP1','PABPC1'],
        ['CSNK1A1','PDE6A'],
        ['CTDP1','NFATC1'],
        ['CTNNA1','REEP2'],
        ['CTTN','SHANK2'],
        ['CYB5B','SNTB2'],
        ['DCAF5','ERH'],
        ['DDB1','CPSF7'],
        ['DENND1A','PSMB7'],
        ['DGKD','ATG16L1'],
        ['DNAJC3','CLDN10'],
        ['DNM2','ILF3'],
        ['DOCK4','IMMP2L'],
        ['DSTN','DTD1'],
        ['DUSP8','GLUD2'],
        ['DZANK1','DTD1'],
        ['ECT2','FNDC3B'],
        ['EEF2K','VWA3A'],
        ['EHD1','BATF2'],
        ['EIF2S1','GPHN'],
        ['EIF3L','TRIOBP'],
        ['ELL','CRLF1'],
        ['ELMO1','AKR1B15'],
        ['EML4','PKDCC'],
        ['EMSY','UVRAG'],
        ['EP300','RBX1'],
        ['EPB41L1','CNBD2'],
        ['EPB41','YTHDF2'],
        ['EPHX1','CALB1'],
        ['ERBB2','IKZF3'],
        ['ERN1','TEX2'],
        ['ETV6','PRH1'],
        ['EXOC6B','SFXN5'],
        ['EXT2','ALKBH3'],
        ['EYA3','GUCA1C'],
        ['F11R','NR1I3'],
        ['FAM129B','C16ORF62'],
        ['FAM13C','S100P'],
        ['FAM168A','C2CD3'],
        ['FASN','NUCKS1'],
        ['FAXC','ASCC3'],
        ['FBXL3','MYCBP2'],
        ['FBXW4','MGEA5'],
        ['FBXW7','TMEM154'],
        ['FDFT1','GATA4'],
        ['FGFR2','ATE1'],
        ['FGFR2','SHTN1'],
        ['FNDC3B','TMEM212'],
        ['FOXJ3','ZMYND12'],
        ['FOXK2','NARF'],
        ['FRMD8','DPF2'],
        ['FRS2','KIF5A'],
        ['FRS2','LEMD3'],
        ['FTH1','COL1A1'],
        ['FXYD5','FXYD3'],
        ['FYN','CDK19'],
        ['GATAD2B','NUP210L'],
        ['GIGYF2','EIF4E2'],
        ['GIGYF2','NGEF'],
        ['GINS1','GART'],
        ['GLTSCR2','DHX34'],
        ['GNPAT','TRIM67'],
        ['GNS','RASSF3'],
        ['GOLGA4','MLH1'],
        ['GOLGB1','IQCB1'],
        ['GOSR1','B4GALT1'],
        ['GOSR1','FAM177A1'],
        ['GRHL2','NCALD'],
        ['GTF2F2','GPALPP1'],
        ['GTPBP1','TOMM22'],
        ['GTSE1','ATXN10'],
        ['H2AFJ','FTH1'],
        ['HDAC7','VDR'],
        ['HECTD1','HEATR5A'],
        ['HERC1','DAPK2'],
        ['HNRNPR','TCEA3'],
        ['HSD17B12','ZBTB3'],
        ['HSD17B4','TNFAIP8'],
        ['IFI6','ACTG1'],
        ['IKBKB','ANK1'],
        ['IL34','TCAF1'],
        ['IL4R','NSMCE1'],
        ['IMMP2L','DOCK4'],
        ['INADL','NFIA'],
        ['INTS3','CHTOP'],
        ['INTS3','S100A9'],
        ['INTS4','KCTD14'],
        ['INTS4','NDUFC2'],
        ['INTS4','NDUFC2-KCTD14'],
        ['IPO11','KIF2A'],
        ['IPO7','NUAK1'],
        ['ITCH','PIGU'],
        ['ITGB1','FLNB'],
        ['JADE2','UBE2B'],
        ['KANSL1','PYY'],
        ['KDM2A','PACS1'],
        ['KDM2A','RHOD'],
        ['KDM4A','PTPRF'],
        ['KIAA0391','FAM177A1'],
        ['KIAA0907','ARHGEF2'],
        ['KIAA1324L','BCAS3'],
        ['KIAA1324L','CBWD3'],
        ['KIF1A','C2ORF54'],
        ['KLC1','APOPT1'],
        ['KMT2C','LOC400927-CSNK1E'],
        ['KPNA1','PARP9'],
        ['KPNA3','SPRYD7'],
        ['LAPTM4B','MTDH'],
        ['LDLRAD3','TRIM44'],
        ['LDLR','SMARCA4'],
        ['LEMD3','LYZ'],
        ['LIMD1','LARS2'],
        ['LMAN2L','FAM178B'],
        ['LOC100289561','SRRM3'],
        ['LRP6','MANSC1'],
        ['LRRFIP1','MLPH'],
        ['LSM1','WHSC1L1'],
        ['LYST','NID1'],
        ['MANBA','UBE2D3'],
        ['MAP2K5','IQCH'],
        ['MAP4','NME6'],
        ['MAP7','MAP3K5'],
        ['MARCH6','BASP1'],
        ['MAST2','NASP'],
        ['MDM2','NUP107'],
        ['MECOM','LRRC34'],
        ['MED13L','TAOK3'],
        ['MED1','CDK12'],
        ['MED1','STARD3'],
        ['MEF2A','LRRC28'],
        ['MEMO1','DPY30'],
        ['MET','CAV1'],
        ['MGRN1','C16ORF96'],
        ['MICALL1','EIF3L'],
        ['MICA','NCR3'],
        ['MIEF1','TAB1'],
        ['MKL1','SPIDR'],
        ['MMP15','DRC7'],
        ['MRGPRX3','ACAD11'],
        ['MRGPRX3','CNTROB'],
        ['MRPL21','FAM168A'],
        ['MRPS18A','BBS5'],
        ['MRPS35','PPFIBP1'],
        ['MSH2','EPCAM'],
        ['MSL2','STAG1'],
        ['MTF1','SF3A3'],
        ['MTMR12','SPEF2'],
        ['MYO1D','ASIC2'],
        ['MYO9A','HEXA'],
        ['MYO9A','PARP6'],
        ['NAALADL2','NLGN1'],
        ['NBEA','FNDC3A'],
        ['NECAB3','PXMP4'],
        ['NEMF','SOS2'],
        ['NFX1','DNAJA1'],
        ['NIPAL3','GRHL3'],
        ['NIPBL','SPEF2'],
        ['NOL9','DNAJC11'],
        ['NOTCH2NL','FCGR3B'],
        ['NOTCH3','BRD4'],
        ['NPHP4','ACOT7'],
        ['NPLOC4','ALYREF'],
        ['NRF1','STRIP2'],
        ['NSA2','HEXB'],
        ['OLFM4','LYZ'],
        ['ORC4','WFDC11'],
        ['OSGIN2','RIPK2'],
        ['OXSR1','CTDSPL'],
        ['PAGR1','KIF22'],
        ['PAN3','FLT3'],
        ['PAPD7','SRD5A1'],
        ['PBX3','MAPKAP1'],
        ['PCBP2','SP1'],
        ['PC','KDM2A'],
        ['PDHX','EHF'],
        ['PDXDC1','PKD1'],
        ['PEX14','PGD'],
        ['PGM2L1','CHRDL2'],
        ['PHF21A','AMBRA1'],
        ['PHF21A','ARHGAP1'],
        ['PI4KB','SELENBP1'],
        ['PIEZO1','CBFA2T3'],
        ['PLEKHF2','NDUFAF6'],
        ['PLXND1','TMCC1'],
        ['POLR2J2','PMS2'],
        ['POLR2J3','PMS2'],
        ['POM121','PMS2'],
        ['PPARA','ATXN10'],
        ['PPFIBP1','C12ORF71'],
        ['PPFIBP1','SMCO2'],
        ['PPM1G','IFT172'],
        ['PPP1CB','BRE'],
        ['PPP1R13B','C14ORF2'],
        ['PPP2R4','FAM73B'],
        ['PPP2R5D','GLTSCR1L'],
        ['PPP3CC','PIWIL2'],
        ['PPP3R1','CNRIP1'],
        ['PPP4R3A','FBLN5'],
        ['PRCP','RAB30'],
        ['PRKAR1A','BCAS3'],
        ['PRKCA','PITPNC1'],
        ['PRRC2B','NUP214'],
        ['PSAP','GPI'],
        ['PTBP3','SVEP1'],
        ['PTK2','MROH5'],
        ['PTMA','ATF4'],
        ['PTMS','RPLP2'],
        ['PTPN4','CFAP221'],
        ['PTPRB','CHRM3'],
        ['PTPRG','FHIT'],
        ['PUM1','NKAIN1'],
        ['PXK','RPP14'],
        ['PXN','PLA2G1B'],
        ['PYCR2','FAM83H'],
        ['PYGB','VTCN1'],
        ['RAB11FIP3','NME4'],
        ['RAB27A','DYX1C1'],
        ['RAB33B','MYO19'],
        ['RAB3GAP1','ACMSD'],
        ['RAB3IP','COMMD1'],
        ['RAB3IP','DOCK9'],
        ['RAB3IP','FRS2'],
        ['RAB3IP','GCC2'],
        ['RAB3IP','SNX7'],
        ['RAB40C','PRR35'],
        ['RAB9A','EGFL6'],
        ['RAD51B','GPHN'],
        ['RALA','YAE1D1'],
        ['RALGPS2','RASAL2'],
        ['RALY','CHMP4B'],
        ['RALY','SLC24A3'],
        ['RAMP1','RBM44'],
        ['RANBP10','CENPT'],
        ['RAPGEF5','STEAP1B'],
        ['RARS2','HAVCR2'],
        ['RB1','MED4'],
        ['RBM47','APBB2'],
        ['RCC1','ITGB3BP'],
        ['REEP5','MCC'],
        ['RERE','GPR157'],
        ['REV3L','TRAF3IP2'],
        ['RGS3','BSPRY'],
        ['RIC1','PDCD1LG2'],
        ['RICTOR','FYB'],
        ['RIPK2','NAAA'],
        ['RNF111','SLTM'],
        ['RNF130','RASGEF1C'],
        ['RNF138','RNF125'],
        ['RNF19A','ANKRD46'],
        ['RPL13','RPSA'],
        ['RPL18A','RPS2'],
        ['RPL23A','COL1A1'],
        ['RPL23','RPS16'],
        ['RPL36','RPS19'],
        ['RPL4','COL1A1'],
        ['RPL8','HNRNPUL1'],
        ['RPL8','TMSB10'],
        ['RPLP1','COL1A1'],
        ['RPLP2','PTMS'],
        ['RPS24','CLU'],
        ['RPS24','COL1A1'],
        ['RPS3A','COL1A2'],
        ['RPSA','GOSR1'],
        ['RPUSD3','EMC3'],
        ['RQCD1','ARPC2'],
        ['RSF1','INTS4'],
        ['RTN3','HRASLS5'],
        ['SAFB2','LONP1'],
        ['SAMD8','DUPD1'],
        ['SAMD8','KAT6B'],
        ['SARNP','MMP19'],
        ['SBF1','SYCE3'],
        ['SCRIB','NRBP2'],
        ['SCYL1','FRMD8'],
        ['SCYL3','KIFAP3'],
        ['SDHA','CTNND2'],
        ['SHANK2','LRTOMT'],
        ['SHROOM3','STBD1'],
        ['SIL1','DNAJC18'],
        ['SIL1','NRG2'],
        ['SKAP1','HOXB3'],
        ['SKI','PRKCZ'],
        ['SLC25A21','MIPOL1'],
        ['SLC25A44','SEMA4A'],
        ['SLC35F5','DHFRL1'],
        ['SLC38A10','OGFOD3'],
        ['SLC39A11','SDK2'],
        ['SLC45A4','MROH5'],
        ['SLC47A2','ALDH3A1'],
        ['SLCO1A2','SCRN3'],
        ['SLCO2B1','HNF4A'],
        ['SMAD3','SMAD6'],
        ['SMARCA4','DNM2'],
        ['SMG1','DCTPP1'],
        ['SMIM4','NISCH'],
        ['SMURF1','GJC3'],
        ['SMURF2','LRRC37A3'],
        ['SNRPD3','ZNF613'],
        ['SNX22','RPL10'],
        ['SNX22','RPLP1'],
        ['SOGA1','RBL1'],
        ['SOS1','CDKL4'],
        ['SOS1','MAP4K3'],
        ['SPECC1L','CABIN1'],
        ['SPEN','PLEKHM2'],
        ['SPTAN1','GLE1'],
        ['SQSTM1','RPS7'],
        ['SRSF4','MECR'],
        ['ST20-MTHFS','BCL2A1'],
        ['ST7','CAV1'],
        ['ST7','MET'],
        ['STARD3','IKZF3'],
        ['STRN','ALK'],
        ['STRN','HEATR5B'],
        ['STXBP1','LRSAM1'],
        ['SUCO','DNM3'],
        ['SUPT7L','RBKS'],
        ['SUZ12','LRRC37B'],
        ['SWAP70','WEE1'],
        ['SYNJ2','ZDHHC14'],
        ['TALDO1','EPS8L2'],
        ['TARSL2','OR4F4'],
        ['TBC1D14','HTRA3'],
        ['TBC1D15','MARS'],
        ['TBC1D22A','GTSE1'],
        ['TBC1D8','RNF149'],
        ['TC2N','TRIP11'],
        ['THEM4','S100A10'],
        ['THRAP3','AGO3'],
        ['THRAP3','MAP7D1'],
        ['TMEM123','MMP20'],
        ['TMEM192','MAST2'],
        ['TMEM45A','CMSS1'],
        ['TNFAIP8','DMXL1'],
        ['TNFRSF19','MIPEP'],
        ['TPX2','DEFB118'],
        ['TPX2','DEFB123'],
        ['TRAF3IP2','FYN'],
        ['TRAF3','WDR20'],
        ['TRAPPC9','PTK2'],
        ['TRMT2B','ALDH16A1'],
        ['TSLP','CHRM3'],
        ['TSPAN14','DYDC2'],
        ['TSPAN15','HK1'],
        ['TSPAN31','OS9'],
        ['TTC7B','RPS6KA5'],
        ['TUBD1','USP32'],
        ['TXNDC11','NPIPB3'],
        ['TXNDC11','NPIPB5'],
        ['TYW1B','GTF2IRD2'],
        ['UBA3','FRMD4B'],
        ['UBA52','RPL8'],
        ['UBA6','TMPRSS11D'],
        ['UBE2K','LIAS'],
        ['UBE2L3','PI4KA'],
        ['UBE3C','PTPRN2'],
        ['UBR1','TGM5'],
        ['UBXN1','BSCL2'],
        ['UGCG','GNG10'],
        ['UGDH','GNB2L1'],
        ['UIMC1','MXD3'],
        ['UMAD1','MIOS'],
        ['UNC13D','SRP68'],
        ['UNK','TMEM94'],
        ['VGLL4','TAMM41'],
        ['VGLL4','TMEM40'],
        ['VMP1','TUBD1'],
        ['VPS53','GLOD4'],
        ['WDR59','CFDP1'],
        ['WHSC1L1','DDHD2'],
        ['WNK1','B4GALNT3'],
        ['WWOX','VAT1L'],
        ['WWP2','TANGO6'],
        ['YEATS4','CPSF6'],
        ['ZBTB8OS','YARS'],
        ['ZC3H3','MROH6'],
        ['ZHX1','ATAD2'],
        ['ZMAT5','ASCC2'],
        ['ZNF341','C20ORF144'],
        ['ZNF3','MCM7'],
        ['ZNF544','ZNF274'],
        ['ZNF814','TPRKB'],
        ['ZRANB1','LHPP'],
        ['ZSWIM4','C19ORF53'],
        ['ZSWIM6','NDUFAF2'],
        ['ZZZ3','USP33'],
        ['ABCA3','COL1A1'],
        ['ABCB5','BBS5'],
        ['ACACA','SSH2'],
        ['ACKR2','POMGNT2'],
        ['ACLY','RPL8'],
        ['ACPP','ZNF462'],
        ['ACSL3','SGPP2'],
        ['ACTB','ALDOA'],
        ['ACTB','CD24'],
        ['ACTB','CD74'],
        ['ACTB','ENSA'],
        ['ACTB','ERBB2'],
        ['ACTB','GAPDH'],
        ['ACTB','HLA-A'],
        ['ACTB','HNRNPA1'],
        ['ACTB','JUP'],
        ['ACTB','KRT19'],
        ['ACTB','LSR'],
        ['ACTB','POSTN'],
        ['ACTB','RAB11FIP1'],
        ['ACTB','SERF2'],
        ['ACTB','SLC39A6'],
        ['ACTG1','ALDOA'],
        ['ACTG1','DSP'],
        ['ACTG1','FTL'],
        ['ACTG1','HSPB1'],
        ['ACTG1','KRT19'],
        ['ACTG1','PTMA'],
        ['ACTG1','SLC39A6'],
        ['ACTG1','SPARC'],
        ['ACTG1','SRRM2'],
        ['ACTG1','YWHAZ'],
        ['ADAM9','RNF170'],
        ['ADAR','SLC39A6'],
        ['ADGRG6','VTA1'],
        ['AEBP1','KRT19'],
        ['AES','SCGB2A2'],
        ['AFM','AFP'],
        ['AGR2','EEF1A1'],
        ['ALDOA','ACTB'],
        ['ALDOA','GAPDH'],
        ['ALDOA','HSP90AA1'],
        ['ALDOA','KRT19'],
        ['APP','ACTB'],
        ['ATOH8','GNLY'],
        ['AZGP1','ACTB'],
        ['AZGP1','GPRC5A'],
        ['AZGP1','KRT14'],
        ['AZIN1','IL6ST'],
        ['B2M','ENSA'],
        ['B2M','TACSTD2'],
        ['BCAM','KRT19'],
        ['BIRC2','YAP1'],
        ['BPTF','KPNA2'],
        ['BSG','MPRIP'],
        ['C1R','HSP90AB1'],
        ['C6ORF132','TRERF1'],
        ['CALR','RPL41'],
        ['CASC3','CDK12'],
        ['CASP10','IL34'],
        ['CCT3','IQGAP3'],
        ['CD151','EPS8L2'],
        ['CD151','SCGB2A2'],
        ['CD74','CD24'],
        ['CD74','COX6C'],
        ['CD74','FN1'],
        ['CD74','GAPDH'],
        ['CD74','HLA-B'],
        ['CD74','PSAP'],
        ['CD74','RPL8'],
        ['CD74','SCGB2A2'],
        ['CD74','SEC61A1'],
        ['CD74','SPARC'],
        ['CFB','EEF1A1'],
        ['CFL1','ACTG1'],
        ['CFL1','FN1'],
        ['CHRM3','DNER'],
        ['CHRM3','HTR2C'],
        ['CHRM3','SSBP2'],
        ['COL18A1','COL3A1'],
        ['COL1A1','AARS'],
        ['COL1A1','ALDOA'],
        ['COL1A1','BHLHE40'],
        ['COL1A1','BSG'],
        ['COL1A1','CALR'],
        ['COL1A1','CCND1'],
        ['COL1A1','CD151'],
        ['COL1A1','CERS6'],
        ['COL1A1','COL11A1'],
        ['COL1A1','COL4A1'],
        ['COL1A1','EIF4G2'],
        ['COL1A1','FOXA1'],
        ['COL1A1','GJA1'],
        ['COL1A1','GLUL'],
        ['COL1A1','HLA-B'],
        ['COL1A1','HLA-E'],
        ['COL1A1','HSPA1B'],
        ['COL1A1','HSPA5'],
        ['COL1A1','HSPA8'],
        ['COL1A1','HSPB1'],
        ['COL1A1','HTRA3'],
        ['COL1A1','IGF1R'],
        ['COL1A1','IGFBP4'],
        ['COL1A1','KRT18'],
        ['COL1A1','LGALS3BP'],
        ['COL1A1','LUM'],
        ['COL1A1','MACF1'],
        ['COL1A1','MMP11'],
        ['COL1A1','MMP14'],
        ['COL1A1','MXRA8'],
        ['COL1A1','MYL12B'],
        ['COL1A1','P4HB'],
        ['COL1A1','PPDPF'],
        ['COL1A1','PPIA'],
        ['COL1A1','PRRC2A'],
        ['COL1A1','PTPRF'],
        ['COL1A1','RPL3'],
        ['COL1A1','RPL6'],
        ['COL1A1','RPLP0'],
        ['COL1A1','RPS14'],
        ['COL1A1','RPS28'],
        ['COL1A1','S100A9'],
        ['COL1A1','SEMA3C'],
        ['COL1A1','SLC40A1'],
        ['COL1A1','SQSTM1'],
        ['COL1A1','TAGLN2'],
        ['COL1A1','TIMP3'],
        ['COL1A1','TPM3'],
        ['COL1A1','TRPS1'],
        ['COL1A1','TSPAN13'],
        ['COL1A1','TTYH3'],
        ['COL1A1','UBA52'],
        ['COL1A1','YWHAZ'],
        ['COL1A2','ADAR'],
        ['COL1A2','BHLHE40'],
        ['COL1A2','CALM1'],
        ['COL1A2','COPA'],
        ['COL1A2','EEF1G'],
        ['COL1A2','FADS2'],
        ['COL1A2','GLUL'],
        ['COL1A2','HNRNPA1'],
        ['COL1A2','HNRNPK'],
        ['COL1A2','ITGB1'],
        ['COL1A2','LAMP1'],
        ['COL1A2','MTRNR2L2'],
        ['COL1A2','P4HB'],
        ['COL1A2','PABPC1'],
        ['COL1A2','PLEC'],
        ['COL1A2','POSTN'],
        ['COL1A2','PPIA'],
        ['COL1A2','RPL13'],
        ['COL1A2','RPL8'],
        ['COL1A2','RPLP1'],
        ['COL1A2','SCGB2A2'],
        ['COL1A2','SERP1'],
        ['COL1A2','SLC39A6'],
        ['COL1A2','THBS1'],
        ['COL1A2','TUBA1B'],
        ['COL1A2','UBC'],
        ['COL1A2','VCP'],
        ['COL3A1','ALDOA'],
        ['COL3A1','APP'],
        ['COL3A1','CD24'],
        ['COL3A1','CILP'],
        ['COL3A1','CTGF'],
        ['COL3A1','GLUL'],
        ['COL3A1','HNRNPA1'],
        ['COL3A1','HSPA8'],
        ['COL3A1','IGFBP5'],
        ['COL3A1','PABPC1'],
        ['COL3A1','POSTN'],
        ['COL3A1','RPL10'],
        ['COL3A1','RPL15'],
        ['COL3A1','RPS4X'],
        ['COL3A1','VCP'],
        ['COL3A1','XBP1'],
        ['COL3A1','YWHAB'],
        ['COL3A1','YWHAE'],
        ['COL4A2','FN1'],
        ['COL5A1','BGN'],
        ['COX6C','COL1A2'],
        ['COX6C','FTH1'],
        ['COX6C','RPL7'],
        ['CPB1','GNB1'],
        ['CPB1','RPL23A'],
        ['CPS1','MAP2'],
        ['CRABP2','ACTG1'],
        ['CRABP2','COL3A1'],
        ['CRADD','MRPL42'],
        ['CRTC1','KLHL26'],
        ['CTDSP2','LGR5'],
        ['CTSD','GAPDH'],
        ['CTSD','HLA-A'],
        ['CTSD','HLA-C'],
        ['CTSD','RPS2'],
        ['DCN','COL1A1'],
        ['DDOST','FTH1'],
        ['DHCR24','ACTB'],
        ['DHX33','NLRP1'],
        ['DIP2B','SPATS2'],
        ['DMPK','DMWD'],
        ['DNAJC24','GFOD2'],
        ['DNAL1','EPHX1'],
        ['DSCR3','TTC3'],
        ['EEF1A1','ACLY'],
        ['EEF1A1','CFB'],
        ['EEF1A1','COL12A1'],
        ['EEF1A1','EIF4G1'],
        ['EEF1A1','FAM134A'],
        ['EEF1A1','FMO3'],
        ['EEF1A1','IFI6'],
        ['EEF1A1','LASP1'],
        ['EEF1A1','LIMCH1'],
        ['EEF1A1','MMP11'],
        ['EEF1A1','PRKDC'],
        ['EEF1A1','PSMB4'],
        ['EEF1A1','PTGFRN'],
        ['EEF1A1','SRP9'],
        ['EEF1A1','TNRC18'],
        ['EEF1A1','USP24'],
        ['EEF1A2','EEF1A1'],
        ['EEF1G','COL1A2'],
        ['EEF2','COX6C'],
        ['EEF2','IL6ST'],
        ['EEF2','NCL'],
        ['EEF2','SLC39A6'],
        ['EEF2','TUBA1C'],
        ['EHF','ITGB1'],
        ['EIF1','FTH1'],
        ['EIF4G2','ACTG1'],
        ['ENO1','ALDOA'],
        ['ENO1','COL3A1'],
        ['ENO1','FTH1'],
        ['ENO1','GAPDH'],
        ['ENO1','RPL6'],
        ['ERBB2','HLA-DRB1'],
        ['ERBB2','RPLP1'],
        ['ERBB2','RPS4X'],
        ['ERBB2','TMSB10'],
        ['FADS2','AZGP1'],
        ['FAM102B','SLC25A24'],
        ['FAM234B','CCND1'],
        ['FAM83H','NDUFA4L2'],
        ['FASN','COL6A3'],
        ['FASN','IGFBP5'],
        ['FASN','RPLP0'],
        ['FASN','TPT1'],
        ['FHOD3','MOCOS'],
        ['FKBP6','TRIM74'],
        ['FLNA','CD74'],
        ['FLNA','COL1A1'],
        ['FN1','AEBP1'],
        ['FN1','AHNAK'],
        ['FN1','ANXA2'],
        ['FN1','CD24'],
        ['FN1','CD74'],
        ['FN1','CNOT1'],
        ['FN1','HSP90B1'],
        ['FN1','ITGB1'],
        ['FN1','KRT8'],
        ['FN1','MGP'],
        ['FN1','MTRNR2L2'],
        ['FN1','MYH9'],
        ['FN1','P4HB'],
        ['FN1','PTMA'],
        ['FN1','RPLP0'],
        ['FN1','RPS27A'],
        ['FN1','RPS4X'],
        ['FN1','RPS6'],
        ['FN1','TMSB10'],
        ['FN1','TUBB'],
        ['FN1','YWHAZ'],
        ['FTH1','ACTN4'],
        ['FTH1','ALDOA'],
        ['FTH1','COL1A2'],
        ['FTH1','DSP'],
        ['FTH1','KRT8'],
        ['FTH1','RPL12'],
        ['FTH1','RPLP0'],
        ['FTH1','SEC61A1'],
        ['FTH1','TRPS1'],
        ['FTL','ACTG1'],
        ['FTL','GAPDH'],
        ['GAPDH','CD59'],
        ['GAPDH','CDH1'],
        ['GAPDH','ENO1'],
        ['GAPDH','FGFR1'],
        ['GAPDH','HLA-A'],
        ['GAPDH','HLA-E'],
        ['GAPDH','KRT18'],
        ['GAPDH','KRT19'],
        ['GAPDH','MTRNR2L2'],
        ['GAPDH','MX1'],
        ['GAPDH','MYH9'],
        ['GAPDH','PKM'],
        ['GAPDH','PTMA'],
        ['GAPDH','RPS3A'],
        ['GAPDH','SLC39A6'],
        ['GAPDH','TIMP3'],
        ['GAPDH','TPT1'],
        ['GBP5','CHM'],
        ['GFOD2','GINS4'],
        ['GLUL','FLNB'],
        ['GNAQ','VEPH1'],
        ['GNAS','PTMA'],
        ['GNAS','RPLP0'],
        ['GNAS','SCD'],
        ['GNAS','TPT1'],
        ['GNAS','TUBB'],
        ['GNB4','ARSF'],
        ['GOSR1','CENPV'],
        ['GRB7','IKZF3'],
        ['GRHL2','FAM49A'],
        ['GTF2I','COL1A1'],
        ['H3F3A','SLC39A6'],
        ['HAT1','DYNC1I2'],
        ['HDAC4','SATB2'],
        ['HDHD2','SKOR2'],
        ['HDLBP','HSPA1A'],
        ['HDLBP','HSPA1B'],
        ['HERC1','KIAA0101'],
        ['HIPK2','PARP12'],
        ['HLA-A','FN1'],
        ['HLA-B','CD74'],
        ['HLA-B','COL1A1'],
        ['HLA-B','RPS6'],
        ['HLA-C','LTF'],
        ['HLA-C','PTMA'],
        ['HLA-DRA','COL1A1'],
        ['HLA-DRB1','FTL'],
        ['HLA-DRB1','HLA-DQB2'],
        ['HMGN2','SRP9'],
        ['HNRNPA2B1','ACTG1'],
        ['HNRNPK','ACTB'],
        ['HSP90AA1','ACTG1'],
        ['HSP90AA1','COL1A2'],
        ['HSP90AB1','ACTG1'],
        ['HSP90AB1','AHNAK'],
        ['HSP90AB1','EIF3A'],
        ['HSP90AB1','GNAS'],
        ['HSP90AB1','HLA-B'],
        ['HSP90AB1','SPARC'],
        ['HSP90B1','COL1A2'],
        ['HSP90B1','COL3A1'],
        ['HSP90B1','FN1'],
        ['HSPA1B','XBP1'],
        ['HSPA5','MTRNR2L2'],
        ['HSPA5','SLC39A6'],
        ['HSPA8','COL1A1'],
        ['HSPA8','EEF1G'],
        ['HSPA9','EEF1A1'],
        ['HSPB1','CCND1'],
        ['HSPB1','COL1A1'],
        ['HSPB1','KRT19'],
        ['HTRA1','COL1A1'],
        ['HUWE1','COL1A2'],
        ['HYAL1','GPATCH2'],
        ['IGLL5','HSP90AA1'],
        ['IGLL5','RPLP1'],
        ['IRF2BP2','NTRK1'],
        ['IRF2BP2','SP2'],
        ['ITGAX','FBXW8'],
        ['KIAA1324L','CWF19L2'],
        ['KIAA1324L','GBP1'],
        ['KMT2C','CTBP2'],
        ['KMT2D','EEF1A1'],
        ['KRT18','RPL7'],
        ['KRT19','ACTG1'],
        ['KRT19','ACTN4'],
        ['KRT19','COL1A1'],
        ['KRT19','FN1'],
        ['KRT19','GAPDH'],
        ['KRT19','HLA-C'],
        ['KRT19','HSPB1'],
        ['KRT19','KRT14'],
        ['KRT19','RPL13A'],
        ['KRT19','STC2'],
        ['KRT5','DSP'],
        ['KRT5','KRT16'],
        ['KRT5','KRT8'],
        ['KRT7','ACTB'],
        ['KRT7','EEF2'],
        ['KRT7','KRT8'],
        ['KRT8','COL1A1'],
        ['KRT8','FTH1'],
        ['KRT8','HLA-C'],
        ['LDHA','ENO1'],
        ['LGR5','TSPAN8'],
        ['LNPEP','CAST'],
        ['LOC100289561','PRKRIP1'],
        ['LTF','COL3A1'],
        ['LTF','FTH1'],
        ['LUC7L2','ETV1'],
        ['MAD1L1','EIF3B'],
        ['MAGED2','FN1'],
        ['MAN1A2','TTF2'],
        ['MAP7','CD200R1'],
        ['MGP','COL4A2'],
        ['MGP','FN1'],
        ['MGP','HNRNPA2B1'],
        ['MIER2','CDK5RAP1'],
        ['MKRN2','PPARG'],
        ['MMP11','EEF1A1'],
        ['MMP11','MTRNR2L8'],
        ['MMP14','COL3A1'],
        ['MMP2','COL1A2'],
        ['MRGPRX3','ATF7IP'],
        ['MTRNR2L1','RPS4X'],
        ['MTRNR2L2','ACTB'],
        ['MTRNR2L8','COL1A1'],
        ['MTRNR2L8','RPL19'],
        ['MTRNR2L8','RPLP1'],
        ['MTUS1','FGL1'],
        ['MUC1','ACTB'],
        ['MUC1','COL1A2'],
        ['MUC1','FN1'],
        ['MUC5B','ACTB'],
        ['MUCL1','P4HB'],
        ['MYH9','ALDOA'],
        ['MYH9','ERBB2'],
        ['MYH9','PABPC1'],
        ['MYH9','PPIA'],
        ['MYL6B','VWA3A'],
        ['NAAA','PRELP'],
        ['NBAS','C2ORF82'],
        ['NCALD','UBR5'],
        ['NCL','COL1A1'],
        ['NCL','SPARC'],
        ['NCOA4','RET'],
        ['NDRG1','ACTB'],
        ['NEDD1','TMTC2'],
        ['NF1','ATAD5'],
        ['NOS1AP','ATF6'],
        ['NPIPA7','NPIPB9'],
        ['NPIPA8','NPIPB9'],
        ['NPM1','COL1A2'],
        ['NRIP3','LYZ'],
        ['NRP2','PARD3B'],
        ['NSD1','NUP98'],
        ['NUMA1','COL1A1'],
        ['OS9','ACTB'],
        ['PABPC1','COL1A2'],
        ['PABPC1','HSP90AA1'],
        ['PABPC1','SCGB2A2'],
        ['PDLIM1','SORBS1'],
        ['PFN1','ERBB2'],
        ['PKM','KRT8'],
        ['PKM','TUBA1B'],
        ['PLEKHM3','ARSD'],
        ['PLXNB2','SPARC'],
        ['PNPT1','ZNF865'],
        ['POSTN','FN1'],
        ['PPARG','SYN2'],
        ['PPIA','ACTB'],
        ['PPP2R4','ACTB'],
        ['PRDX1','CFL1'],
        ['PRKAR1A','GBP1'],
        ['PRRC2A','COL1A1'],
        ['PTEN','CTNNA3'],
        ['PTMA','HSP90AA1'],
        ['PTMS','HSP90AA1'],
        ['PTPN1','GAPDH'],
        ['PTPRF','COL1A2'],
        ['PTPRK','ECHDC1'],
        ['RABEP1','RNF167'],
        ['RASA2','PXYLP1'],
        ['RBCK1','CLCN7'],
        ['RPL10A','COL1A2'],
        ['RPL10','COL1A1'],
        ['RPL10','COL3A1'],
        ['RPL10','FN1'],
        ['RPL10','RPL3'],
        ['RPL12','ACTB'],
        ['RPL13A','RPS2'],
        ['RPL13A','TMSB10'],
        ['RPL18A','B2M'],
        ['RPL18A','CD74'],
        ['RPL18A','GAPDH'],
        ['RPL18A','SCGB2A2'],
        ['RPL19','HLA-B'],
        ['RPL19','P4HB'],
        ['RPL22','ACTB'],
        ['RPL26','GAPDH'],
        ['RPL31','COL3A1'],
        ['RPL35','COL1A1'],
        ['RPL36','COL1A2'],
        ['RPL3','ALDOA'],
        ['RPL3','COL1A1'],
        ['RPL3','COL3A1'],
        ['RPL3','PIP'],
        ['RPL3','TUBB'],
        ['RPL4','HLA-B'],
        ['RPL6','SLC39A6'],
        ['RPL8','COL1A2'],
        ['RPL8','FKBP4'],
        ['RPL8','RPLP1'],
        ['RPLP0','FASN'],
        ['RPLP0','PTMA'],
        ['RPLP1','APOD'],
        ['RPLP1','CLU'],
        ['RPLP1','GAPDH'],
        ['RPLP2','ATP5E'],
        ['RPS12','COL1A2'],
        ['RPS16','COL1A2'],
        ['RPS17','MYH9'],
        ['RPS18','PTMA'],
        ['RPS18','SPARC'],
        ['RPS19','EEF1G'],
        ['RPS2','B2M'],
        ['RPS2','CAPG'],
        ['RPS2','GNAS'],
        ['RPS2','MGP'],
        ['RPS2','PSAP'],
        ['RPS2','RPLP1'],
        ['RPS2','SLC39A6'],
        ['RPS2','YWHAZ'],
        ['RPS3','FN1'],
        ['RPS3','MYH9'],
        ['RPS4X','FN1'],
        ['RPS4X','TUBA1B'],
        ['RPS6','COL1A2'],
        ['RPS6','MYH9'],
        ['RPS7','COL1A2'],
        ['RPS8','FN1'],
        ['RPS9','FN1'],
        ['RRBP1','SNX5'],
        ['RRP12','EXOSC1'],
        ['SAP18','SCGB2A2'],
        ['SCD','COL3A1'],
        ['SCD','TMED10'],
        ['SCGB1D2','NACA'],
        ['SCGB2A2','EIF4G2'],
        ['SCGB2A2','FTH1'],
        ['SCGB2A2','MYL6'],
        ['SEPT2','COL3A1'],
        ['SEPW1','EHD2'],
        ['SERF2','ACTG1'],
        ['SERPINA3','ACTB'],
        ['SERPINA3','B2M'],
        ['SERPINA3','COL1A1'],
        ['SERPINA3','COL3A1'],
        ['SFPQ','TFEB'],
        ['SLC25A6','COL1A1'],
        ['SLC2A1','ACTB'],
        ['SLC39A3','PLPP2'],
        ['SLC39A6','ACTB'],
        ['SLC39A6','COL1A2'],
        ['SLC39A6','TPT1'],
        ['SLC45A3','ETV1'],
        ['SMG1','NPIPB5'],
        ['SND1','BRAF'],
        ['SNX30','PXT1'],
        ['SOD1','SCGB2A2'],
        ['SPAG9','PCMT1'],
        ['SPARC','MTRNR2L2'],
        ['SPARC','PSAP'],
        ['SPARC','RHOB'],
        ['SPARC','SLC39A6'],
        ['SPC25','ASTN2'],
        ['SPINT2','TUBA1B'],
        ['SPTBN1','COL1A1'],
        ['SRGAP3','LHFPL4'],
        ['SRRM2','ACTG1'],
        ['STC2','COL3A1'],
        ['STK3','CPQ'],
        ['SUB1','COL3A1'],
        ['SUSD6','PLEKHD1'],
        ['TAF4','TSHZ2'],
        ['TAGLN2','HSP90AB1'],
        ['TAGLN2','RPS2'],
        ['TAP1','EEF1A1'],
        ['TFE3','PRCC'],
        ['TGOLN2','MYO5C'],
        ['TLN1','COL3A1'],
        ['TMBIM6','FN1'],
        ['TMEM192','NPY1R'],
        ['TMPRSS2','ETV4'],
        ['TMPRSS6','PPP6R3'],
        ['TMSB10','COL1A2'],
        ['TMSB10','EEF1G'],
        ['TMSB10','KRT8'],
        ['TMSB10','PTMA'],
        ['TMSB10','RPL41'],
        ['TMSB10','TUBB'],
        ['TMSB4X','ACTB'],
        ['TMSB4X','APP'],
        ['TMSB4X','HLA-B'],
        ['TMSB4X','RPL3'],
        ['TMSB4X','SCGB2A2'],
        ['TOP2B','NGLY1'],
        ['TP63','TPRG1'],
        ['TPM3','NTRK1'],
        ['TPT1','HLA-B'],
        ['TPT1','MTRNR2L2'],
        ['TPX2','HM13'],
        ['TRIM28','KRT19'],
        ['TRIM68','APOL1'],
        ['TRIO','TERT'],
        ['TRMT11','NCOA7'],
        ['TUBA1B','COL1A2'],
        ['TUBA1B','HSP90AA1'],
        ['TUBA1B','MTRNR2L2'],
        ['TUBB','ACTG1'],
        ['TWF2','FSTL4'],
        ['UBA1','TMBIM6'],
        ['UBB','COL1A2'],
        ['UBC','HSPA8'],
        ['UBC','KRT18'],
        ['UBC','TUBA1B'],
        ['UBXN4','R3HDM1'],
        ['UGP2','MDH1'],
        ['URI1','PLEKHF1'],
        ['USP34','FAM161A'],
        ['USP6NL','UPF2'],
        ['VIM','COL1A2'],
        ['WBP1L','SUFU'],
        ['WDR34','ZER1'],
        ['WDR72','ADGRV1'],
        ['WIPF2','PSMD3'],
        ['WSCD1','NLRP1'],
        ['XBP1','ACTB'],
        ['XBP1','CALR'],
        ['XBP1','FASN'],
        ['XBP1','HSP90AB1'],
        ['XRCC6','ACO2'],
        ['YAP1','C11ORF70'],
        ['YWHAB','FN1'],
        ['YWHAE','ACTG1'],
        ['YWHAZ','ACTB'],
        ['YWHAZ','COL1A2'],
        ['ZBTB38','PXYLP1'],
        ['ZBTB7B','CKS1B'],
        ['ZDHHC14','CHRM3'],
        ['ZNF736','CHMP1A'],
        ['A2M','FN1'],
        ['ABCC1','EEF1A1'],
        ['ABCC9','ASTN2'],
        ['ABCC9','MAB21L3'],
        ['ABCC9','NEK5'],
        ['ABHD12B','IL34'],
        ['ACAA2','GOSR1'],
        ['ACAA2','MRGPRX3'],
        ['ACACA','EEF1A1'],
        ['ACIN1','BCAM'],
        ['ACIN1','COL1A1'],
        ['ACLY','CPB1'],
        ['ACO2','EEF1A1'],
        ['ACOT6','NUMB'],
        ['ACTA2','ACTG1'],
        ['ACTA2','COL1A1'],
        ['ACTA2','COL3A1'],
        ['ACTB','APP'],
        ['ACTB','ARF1'],
        ['ACTB','ATP1A1'],
        ['ACTB','AZGP1'],
        ['ACTB','B4GALT1'],
        ['ACTB','BGN'],
        ['ACTB','BHLHE40'],
        ['ACTB','CAP1'],
        ['ACTB','CARTPT'],
        ['ACTB','CCND1'],
        ['ACTB','CD164'],
        ['ACTB','CD24'],
        ['ACTB','CFL1'],
        ['ACTB','CLEC3A'],
        ['ACTB','CLU'],
        ['ACTB','COL12A1'],
        ['ACTB','COL1A1'],
        ['ACTB','COL1A2'],
        ['ACTB','COL6A2'],
        ['ACTB','COPA'],
        ['ACTB','CXCL14'],
        ['ACTB','DDR1'],
        ['ACTB','DHCR24'],
        ['ACTB','DSP'],
        ['ACTB','EIF2S3'],
        ['ACTB','EIF4A1'],
        ['ACTB','EIF4B'],
        ['ACTB','EIF4G2'],
        ['ACTB','ERBB2'],
        ['ACTB','ERBB3'],
        ['ACTB','ESR1'],
        ['ACTB','FAM210B'],
        ['ACTB','FASN'],
        ['ACTB','FLNB'],
        ['ACTB','FN1'],
        ['ACTB','GALNT6'],
        ['ACTB','GAPDH'],
        ['ACTB','GNAS'],
        ['ACTB','GNB1'],
        ['ACTB','GNB2L1'],
        ['ACTB','GRINA'],
        ['ACTB','H3F3B'],
        ['ACTB','HDGF'],
        ['ACTB','HDLBP'],
        ['ACTB','HLA-A'],
        ['ACTB','HLA-C'],
        ['ACTB','HLA-DRA'],
        ['ACTB','HNRNPA1'],
        ['ACTB','HNRNPA2B1'],
        ['ACTB','HP1BP3'],
        ['ACTB','HSPA5'],
        ['ACTB','IGFBP4'],
        ['ACTB','KLF6'],
        ['ACTB','KRT16'],
        ['ACTB','KRT18'],
        ['ACTB','KRT19'],
        ['ACTB','KRT7'],
        ['ACTB','LDHA'],
        ['ACTB','LRP1'],
        ['ACTB','MARCKS'],
        ['ACTB','MLLT6'],
        ['ACTB','MORF4L2'],
        ['ACTB','MTRNR2L2'],
        ['ACTB','MUCL1'],
        ['ACTB','NCL'],
        ['ACTB','NFIC'],
        ['ACTB','NPM1'],
        ['ACTB','NUCKS1'],
        ['ACTB','PKM'],
        ['ACTB','PTMA'],
        ['ACTB','PTP4A2'],
        ['ACTB','PTPRF'],
        ['ACTB','PTTG1IP'],
        ['ACTB','RAD21'],
        ['ACTB','RBM39'],
        ['ACTB','RPL10'],
        ['ACTB','RPL13'],
        ['ACTB','RPL15'],
        ['ACTB','RPL19'],
        ['ACTB','RPL23A'],
        ['ACTB','RPL5'],
        ['ACTB','RPS11'],
        ['ACTB','RPS8'],
        ['ACTB','RPS9'],
        ['ACTB','RTN4'],
        ['ACTB','S100A11'],
        ['ACTB','SCGB1D2'],
        ['ACTB','SCGB2A2'],
        ['ACTB','SLC39A6'],
        ['ACTB','SRP9'],
        ['ACTB','STC2'],
        ['ACTB','TAGLN2'],
        ['ACTB','THRAP3'],
        ['ACTB','TIMP3'],
        ['ACTB','TMSB10'],
        ['ACTB','TMSB4X'],
        ['ACTB','UBB'],
        ['ACTB','VCAN'],
        ['ACTB','WASF2'],
        ['ACTB','XBP1'],
        ['ACTB','XRCC5'],
        ['ACTB','YWHAG'],
        ['ACTB','YWHAZ'],
        ['ACTB','ZMIZ1'],
        ['ACTG1','APLP2'],
        ['ACTG1','ATP1A1'],
        ['ACTG1','AZGP1'],
        ['ACTG1','CANX'],
        ['ACTG1','CDH1'],
        ['ACTG1','COL1A1'],
        ['ACTG1','COL6A3'],
        ['ACTG1','CPB1'],
        ['ACTG1','CTSB'],
        ['ACTG1','CTTN'],
        ['ACTG1','DSP'],
        ['ACTG1','EEF1D'],
        ['ACTG1','EIF4H'],
        ['ACTG1','ERBB2'],
        ['ACTG1','FASN'],
        ['ACTG1','FN1'],
        ['ACTG1','FTH1'],
        ['ACTG1','FTL'],
        ['ACTG1','GATA3'],
        ['ACTG1','GLUL'],
        ['ACTG1','GNB2L1'],
        ['ACTG1','HLA-DRA'],
        ['ACTG1','HMGB1'],
        ['ACTG1','HNRNPA1'],
        ['ACTG1','HNRNPF'],
        ['ACTG1','HNRNPM'],
        ['ACTG1','HSP90AA1'],
        ['ACTG1','HSP90AB1'],
        ['ACTG1','HSP90B1'],
        ['ACTG1','HSPA1A'],
        ['ACTG1','HSPB1'],
        ['ACTG1','IGFBP5'],
        ['ACTG1','KRT15'],
        ['ACTG1','KRT18'],
        ['ACTG1','KRT19'],
        ['ACTG1','KRT7'],
        ['ACTG1','LGALS3BP'],
        ['ACTG1','MGP'],
        ['ACTG1','MTRNR2L2'],
        ['ACTG1','MUC1'],
        ['ACTG1','MYH9'],
        ['ACTG1','NCL'],
        ['ACTG1','PABPC1'],
        ['ACTG1','PCBP2'],
        ['ACTG1','PLOD1'],
        ['ACTG1','POSTN'],
        ['ACTG1','PSAP'],
        ['ACTG1','PTMA'],
        ['ACTG1','RPL10'],
        ['ACTG1','RPL15'],
        ['ACTG1','RPL28'],
        ['ACTG1','RPL7A'],
        ['ACTG1','RPLP0'],
        ['ACTG1','RPN1'],
        ['ACTG1','RPS2'],
        ['ACTG1','RPS24'],
        ['ACTG1','RPS27A'],
        ['ACTG1','SCGB2A2'],
        ['ACTG1','SERPINA3'],
        ['ACTG1','SLC39A6'],
        ['ACTG1','SPARC'],
        ['ACTG1','TAGLN2'],
        ['ACTG1','UBB'],
        ['ACTG1','UBC'],
        ['ACTG1','VCL'],
        ['ACTG1','XBP1'],
        ['ACTG1','YWHAZ'],
        ['ACTN1','COL3A1'],
        ['ACTN4','COL1A1'],
        ['ACTN4','COL3A1'],
        ['ACTN4','FASN'],
        ['ACTN4','HSPA1B'],
        ['ACTN4','RPS2'],
        ['ACTR3C','KMT2C'],
        ['ADAM9','DCDC1'],
        ['ADAR','RPS2'],
        ['ADCY2','MED24'],
        ['ADIPOR2','EEF1A1'],
        ['ADIPOR2','MTRNR2L8'],
        ['AEBP1','CD74'],
        ['AEBP1','COL1A1'],
        ['AEBP1','COL1A2'],
        ['AEBP1','COL3A1'],
        ['AEBP1','EEF1G'],
        ['AEBP1','FN1'],
        ['AEBP1','FTH1'],
        ['AEBP1','LGALS3BP'],
        ['AEBP1','MTRNR2L8'],
        ['AEBP1','SCGB2A2'],
        ['AEBP1','SRRM2'],
        ['AES','COL1A1'],
        ['AES','MMP14'],
        ['AGPAT3','ISY1'],
        ['AGR2','COL1A1'],
        ['AGR2','COL1A2'],
        ['AGR2','EEF1A1'],
        ['AGRN','COL3A1'],
        ['AGRN','EEF1A1'],
        ['AHNAK','AZGP1'],
        ['AHNAK','COL1A1'],
        ['AHNAK','HLA-A'],
        ['AHNAK','HSPA1B'],
        ['AHNAK','IGFBP4'],
        ['AHNAK','RPL7A'],
        ['AHNAK','RPS2'],
        ['AHNAK','SCGB2A2'],
        ['AKAP3','CFLAR'],
        ['AKR1B15','SLC2A14'],
        ['AKT1','COL1A1'],
        ['AKT1','CPB1'],
        ['AKT1','EEF1A1'],
        ['ALDH16A1','TRMT2B'],
        ['ALDOA','APLP2'],
        ['ALDOA','CALR'],
        ['ALDOA','COL1A1'],
        ['ALDOA','COL1A2'],
        ['ALDOA','COL3A1'],
        ['ALDOA','EEF1G'],
        ['ALDOA','ENSA'],
        ['ALDOA','FN1'],
        ['ALDOA','FTH1'],
        ['ALDOA','FTL'],
        ['ALDOA','GAPDH'],
        ['ALDOA','GNAS'],
        ['ALDOA','HDLBP'],
        ['ALDOA','HLA-A'],
        ['ALDOA','HLA-B'],
        ['ALDOA','HSP90AB1'],
        ['ALDOA','IGFBP4'],
        ['ALDOA','IGLL5'],
        ['ALDOA','KRT8'],
        ['ALDOA','RPL13A'],
        ['ALDOA','RPL3'],
        ['ALDOA','RPLP0'],
        ['ALDOA','RPS2'],
        ['ALDOA','SLC25A6'],
        ['ALDOA','XBP1'],
        ['ALG1L2','FAM90A1'],
        ['ANKRD34A','SRP68'],
        ['ANKRD36','BAGE'],
        ['ANKRD36','BAGE2'],
        ['ANKRD36','BAGE3'],
        ['ANKRD36','BAGE4'],
        ['ANKRD36','BAGE5'],
        ['ANP32B','RPL39'],
        ['ANP32B','SRRM2'],
        ['ANTXR1','EEF1A1'],
        ['ANXA2','GAPDH'],
        ['ANXA2','MUCL1'],
        ['AP2B1','COL1A1'],
        ['AP2B1','CPB1'],
        ['AP2B1','SLFN13'],
        ['AP2M1','COL1A1'],
        ['APLP2','AZGP1'],
        ['APLP2','COL1A1'],
        ['APLP2','CYBA'],
        ['APLP2','FN1'],
        ['APLP2','IGFBP4'],
        ['APLP2','MYH9'],
        ['APLP2','RPLP0'],
        ['APLP2','SCGB2A2'],
        ['APLP2','SRRM2'],
        ['APMAP','CPB1'],
        ['APMAP','EEF1G'],
        ['APOE','COL1A1'],
        ['APOE','ERBB2'],
        ['APOE','FTH1'],
        ['APOL1','EEF1A1'],
        ['APP','CALR'],
        ['APP','COL1A2'],
        ['APP','COL3A1'],
        ['APP','CPB1'],
        ['APP','FN1'],
        ['APP','GAPDH'],
        ['APP','MYH9'],
        ['AR','RPL14'],
        ['ARF1','COL1A1'],
        ['ARF1','COL3A1'],
        ['ARF1','FASN'],
        ['ARF1','FN1'],
        ['ARF1','SLC39A6'],
        ['ARFGEF2','PREX1'],
        ['ARHGAP11A','CNTLN'],
        ['ARHGDIA','EEF1A1'],
        ['ARHGDIB','COL3A1'],
        ['ARL6IP1','EEF1A1'],
        ['ARL6IP4','SRRM2'],
        ['ARPC2','COL1A1'],
        ['ARPP19','EEF1A1'],
        ['ASAH1','COL1A1'],
        ['ASCL1','RPL14'],
        ['ASPH','COL1A2'],
        ['ASTN2','CTSC'],
        ['ASTN2','F5'],
        ['ASTN2','MAB21L3'],
        ['ASTN2','OPHN1'],
        ['ASTN2','TMEM212'],
        ['ASTN2','USP33'],
        ['ATAD2','CPB1'],
        ['ATF7IP2','CD24'],
        ['ATN1','COL1A1'],
        ['ATP13A4','OPA1'],
        ['ATP1A1','COL1A1'],
        ['ATP1A1','COL1A2'],
        ['ATP1A1','COL3A1'],
        ['ATP1A1','EEF2'],
        ['ATP1A1','KRT14'],
        ['ATP1A1','MTRNR2L8'],
        ['ATP1A1','PPP1R1B'],
        ['ATP1A1','UBC'],
        ['ATP5B','FN1'],
        ['ATP5B','FTH1'],
        ['ATP5B','MUC6'],
        ['ATP5B','RPS18'],
        ['ATP5E','EEF1A1'],
        ['ATP5G2','LIN28A'],
        ['ATP5S','FAM102B'],
        ['ATP6AP1','EEF1A1'],
        ['ATP6V0C','EEF1A1'],
        ['ATP9A','COL1A1'],
        ['AUTS2','EEF1A1'],
        ['AXIN2','RGS9'],
        ['AZGP1','COL1A1'],
        ['AZGP1','COL1A2'],
        ['AZGP1','COL3A1'],
        ['AZGP1','CPB1'],
        ['AZGP1','CTSD'],
        ['AZGP1','GNAS'],
        ['AZGP1','HLA-B'],
        ['AZGP1','HSPA8'],
        ['AZGP1','MUC1'],
        ['AZGP1','NME1-NME2'],
        ['AZGP1','RPLP1'],
        ['AZGP1','SCGB2A2'],
        ['AZGP1','SERPINA3'],
        ['AZGP1','SPARC'],
        ['AZGP1','UBC'],
        ['AZGP1','XBP1'],
        ['AZGP1','ZKSCAN1'],
        ['B2M','CANX'],
        ['B2M','CLU'],
        ['B2M','COL1A2'],
        ['B2M','CPB1'],
        ['B2M','CTSD'],
        ['B2M','ERBB2'],
        ['B2M','FN1'],
        ['B2M','HDLBP'],
        ['B2M','HLA-A'],
        ['B2M','HLA-DPA1'],
        ['B2M','HNRNPA1'],
        ['B2M','HNRNPC'],
        ['B2M','HSP90AA1'],
        ['B2M','HSP90AB1'],
        ['B2M','HSP90B1'],
        ['B2M','KRT17'],
        ['B2M','KRT19'],
        ['B2M','LGALS3BP'],
        ['B2M','LYZ'],
        ['B2M','PKM'],
        ['B2M','PSAP'],
        ['B2M','RPL19'],
        ['B2M','RPL3'],
        ['B2M','RPL7'],
        ['B2M','RPLP0'],
        ['B2M','RPS2'],
        ['B2M','SCGB2A2'],
        ['B2M','SLC39A6'],
        ['B2M','TBC1D9'],
        ['B2M','TMEM123'],
        ['B2M','XBP1'],
        ['B2M','YBX1'],
        ['B4GALT1','NFX1'],
        ['B4GALT1','SPARC'],
        ['B4GALT1','UBC'],
        ['BAG6','CNPY3'],
        ['BBS5','EIF2B3'],
        ['BCAM','COL1A1'],
        ['BCAM','COL1A2'],
        ['BCAM','COL3A1'],
        ['BCAM','GNAS'],
        ['BCAM','KRT19'],
        ['BCAM','SLC39A6'],
        ['BCAP31','CPB1'],
        ['BGN','CALM2'],
        ['BGN','CAMK2N1'],
        ['BGN','CLIC6'],
        ['BGN','COL1A2'],
        ['BGN','COL3A1'],
        ['BGN','CTSE'],
        ['BGN','EEF1G'],
        ['BGN','FASN'],
        ['BGN','FN1'],
        ['BGN','GATA3'],
        ['BGN','GFRA1'],
        ['BGN','GNAS'],
        ['BGN','HACD3'],
        ['BGN','HSP90AB1'],
        ['BGN','IGF2'],
        ['BGN','KRT8'],
        ['BGN','MUC13'],
        ['BGN','MYH9'],
        ['BGN','PFN1'],
        ['BGN','RAB7A'],
        ['BGN','RPL13A'],
        ['BGN','RPL29'],
        ['BGN','SLC25A6'],
        ['BGN','TNS4'],
        ['BHLHE40','COL1A1'],
        ['BHLHE40','CPB1'],
        ['BHLHE40','IGF2'],
        ['BLOC1S2','MRGPRX3'],
        ['BNIP3L','CHRM3'],
        ['BPNT1','KBTBD12'],
        ['BPTF','KPNA2'],
        ['BSG','COL1A1'],
        ['BSG','COL1A2'],
        ['BSG','EEF1A1'],
        ['BSG','FN1'],
        ['BSG','TUBA1B'],
        ['BST2','CALR'],
        ['BST2','IFITM3'],
        ['BTF3','COL1A1'],
        ['BZW1','COL3A1'],
        ['C12ORF56','FOCAD'],
        ['C19ORF35','DNAJC21'],
        ['C1D','RALY'],
        ['C1QB','SRSF3'],
        ['C1QTNF3','C1ORF109'],
        ['C1QTNF9','CFLAR'],
        ['C1S','COL1A1'],
        ['C1S','COL1A2'],
        ['C1S','FN1'],
        ['C3','COL3A1'],
        ['C3','SPARC'],
        ['C3ORF52','MFSD14B'],
        ['C3ORF67','GABRR1'],
        ['C5ORF42','WDR70'],
        ['C6ORF99','DNAL1'],
        ['C6ORF99','HHLA2'],
        ['C6ORF99','OR5H1'],
        ['C8ORF34','CHRM3'],
        ['C9ORF84','CXORF38'],
        ['CA12','COL1A1'],
        ['CA5B','GINM1'],
        ['CALB1','EPHX1'],
        ['CALB1','IL34'],
        ['CALM1','COL1A1'],
        ['CALM1','GPX4'],
        ['CALM2','COL1A1'],
        ['CALM2','COL3A1'],
        ['CALM2','MUCL1'],
        ['CALM2','RPS2'],
        ['CALM2','SPARC'],
        ['CALM3','COL1A1'],
        ['CALR','CD74'],
        ['CALR','COL1A1'],
        ['CALR','COL1A2'],
        ['CALR','CPB1'],
        ['CALR','EEF2'],
        ['CALR','FN1'],
        ['CALR','FTH1'],
        ['CALR','GAPDH'],
        ['CALR','GNAS'],
        ['CALR','LTF'],
        ['CALR','MTRNR2L8'],
        ['CALR','MYH9'],
        ['CALR','PABPC1'],
        ['CALR','PTMA'],
        ['CALR','RPS18'],
        ['CALR','SCD'],
        ['CALR','SLC9A3R1'],
        ['CALR','SPARC'],
        ['CALR','SRRM2'],
        ['CALR','TMSB10'],
        ['CALU','COL1A1'],
        ['CAMK1D','TMPO'],
        ['CANX','COL1A1'],
        ['CANX','COL3A1'],
        ['CANX','CPB1'],
        ['CANX','FN1'],
        ['CANX','ITGB1'],
        ['CANX','KRT19'],
        ['CANX','MGP'],
        ['CANX','MTRNR2L8'],
        ['CANX','SERPINA3'],
        ['CANX','SLC39A6'],
        ['CAP1','IFI6'],
        ['CAPN8','ZNF343'],
        ['CAPZB','COL1A1'],
        ['CAPZB','COL1A2'],
        ['CARM1','YIPF2'],
        ['CASP10','EPHX1'],
        ['CASP10','IL34'],
        ['CCHCR1','ISPD'],
        ['CCL14','CCL15'],
        ['CCL5','SLC25A23'],
        ['CCND1','COL3A1'],
        ['CCND1','DDX17'],
        ['CCND1','EEF1G'],
        ['CCND1','EEF2'],
        ['CCND1','FN1'],
        ['CCND1','HSPA5'],
        ['CCND1','HSPA8'],
        ['CCND1','IGFBP4'],
        ['CCND1','KRT18'],
        ['CCND1','LSM14A'],
        ['CCND1','RAB6A'],
        ['CCND1','SCD'],
        ['CCND1','SCGB2A2'],
        ['CCND1','SLC1A5'],
        ['CCND1','SLC39A6'],
        ['CCND1','SLC9A3R1'],
        ['CCND1','SPARC'],
        ['CCND1','TMBIM6'],
        ['CCND1','TPT1'],
        ['CCND1','UBC'],
        ['CCND1','YWHAZ'],
        ['CCT3','CPB1'],
        ['CCT3','GAPDH'],
        ['CCZ1','CCZ1B'],
        ['CD164','COL1A1'],
        ['CD24','COL1A1'],
        ['CD24','COL1A2'],
        ['CD24','COL3A1'],
        ['CD24','EEF2'],
        ['CD24','ERBB2'],
        ['CD24','FN1'],
        ['CD24','FTH1'],
        ['CD24','GLUL'],
        ['CD24','HSP90AA1'],
        ['CD24','HSP90AB1'],
        ['CD24','HSPA8'],
        ['CD24','MTRNR2L2'],
        ['CD24','PABPC1'],
        ['CD24','RAB11FIP1'],
        ['CD59','COL1A1'],
        ['CD59','COL3A1'],
        ['CD63','COL1A1'],
        ['CD63','SCGB2A2'],
        ['CD63','SPARC'],
        ['CD68','HNRNPA3'],
        ['CD74','CLU'],
        ['CD74','COL1A1'],
        ['CD74','COL1A2'],
        ['CD74','EEF1G'],
        ['CD74','ERBB2'],
        ['CD74','FLNA'],
        ['CD74','FN1'],
        ['CD74','FTH1'],
        ['CD74','FTL'],
        ['CD74','GAPDH'],
        ['CD74','GNAS'],
        ['CD74','HLA-A'],
        ['CD74','HLA-B'],
        ['CD74','HLA-C'],
        ['CD74','HSPB1'],
        ['CD74','LDHA'],
        ['CD74','LTF'],
        ['CD74','MTRNR2L8'],
        ['CD74','MYL6'],
        ['CD74','P4HB'],
        ['CD74','PKM'],
        ['CD74','RPL13A'],
        ['CD74','RPL19'],
        ['CD74','RPLP0'],
        ['CD74','RPS2'],
        ['CD74','RPS6'],
        ['CD74','SLC39A6'],
        ['CD74','SPARC'],
        ['CD74','TMSB10'],
        ['CD81','COL1A1'],
        ['CD9','COL1A1'],
        ['CD9','COL1A2'],
        ['CD9','MTRNR2L8'],
        ['CDC42BPB','TNFAIP2'],
        ['CDH1','COL1A1'],
        ['CDH1','COL1A2'],
        ['CDH1','CPB1'],
        ['CDH1','FASN'],
        ['CDH1','FTH1'],
        ['CDH1','GAPDH'],
        ['CDH1','IGLL5'],
        ['CDH1','MTRNR2L8'],
        ['CDK12','PROX1'],
        ['CDK14','CDYL'],
        ['CDK14','CHRM3'],
        ['CDYL','FGF12'],
        ['CDYL','NKAIN2'],
        ['CDYL','TIGIT'],
        ['CDYL','TSHZ2'],
        ['CEACAM6','COL1A2'],
        ['CEACAM6','EEF1A1'],
        ['CEACAM6','FTH1'],
        ['CELF3','RPL14'],
        ['CEP104','EMP2'],
        ['CEP104','OPA3'],
        ['CERS2','COL1A2'],
        ['CERS2','EEF1A1'],
        ['CFB','COL1A1'],
        ['CFB','COL1A2'],
        ['CFB','COL3A1'],
        ['CFB','CPB1'],
        ['CFB','EEF1A1'],
        ['CFB','EEF2'],
        ['CFB','FTH1'],
        ['CFB','KRT19'],
        ['CFB','RPL9'],
        ['CFB','SCGB2A2'],
        ['CFL1','COL1A2'],
        ['CFL1','COL3A1'],
        ['CFL1','EEF1D'],
        ['CFL1','FN1'],
        ['CFL1','HLA-A'],
        ['CFL1','HLA-C'],
        ['CFL1','MUC5B'],
        ['CFL1','PSAP'],
        ['CFL1','PTMA'],
        ['CFL1','RPL10'],
        ['CFL1','RPL13A'],
        ['CFL1','TMSB10'],
        ['CFLAR','PTGR1'],
        ['CFLAR','SHISA9'],
        ['CGREF1','SUOX'],
        ['CHD4','COL1A1'],
        ['CHD4','EEF1A1'],
        ['CHGA','RPL13A'],
        ['CHM','GLIPR1L2'],
        ['CHM','ZNF365'],
        ['CHM','ZNF589'],
        ['CHMP3','CPB1'],
        ['CHRM3','FAM49A'],
        ['CHRM3','FGF12'],
        ['CHRM3','KDM5A'],
        ['CHRM3','KRT76'],
        ['CHRM3','KYNU'],
        ['CHRM3','MAB21L3'],
        ['CHRM3','MYO10'],
        ['CHRM3','NKAIN2'],
        ['CHRM3','PRKCA'],
        ['CHRM3','TAB3'],
        ['CHRM3','TMEM163'],
        ['CHRM3','TRDN'],
        ['CHRM3','TSHZ2'],
        ['CHST6','HYAL1'],
        ['CLCN7','MLXIP'],
        ['CLEC3A','EEF1A1'],
        ['CLEC3A','HSPB1'],
        ['CLEC3A','MGP'],
        ['CLSTN1','COL1A1'],
        ['CLSTN1','LGALS3BP'],
        ['CLSTN1','PTMA'],
        ['CLTC','FTH1'],
        ['CLTC','SCGB2A2'],
        ['CLTC','SRRM2'],
        ['CLU','COL1A2'],
        ['CLU','COL3A1'],
        ['CLU','CPB1'],
        ['CLU','GAPDH'],
        ['CLU','IGLL5'],
        ['CLU','KRT19'],
        ['CLU','PABPC1'],
        ['CLU','SLC39A6'],
        ['CLU','TMSB4X'],
        ['CNN3','COL1A2'],
        ['CNOT1','CPB1'],
        ['COG3','SLC25A30'],
        ['COL11A1','COL1A1'],
        ['COL11A1','COL1A2'],
        ['COL11A1','EEF1A1'],
        ['COL12A1','COL1A1'],
        ['COL12A1','COL1A2'],
        ['COL12A1','COL3A1'],
        ['COL12A1','EEF1A1'],
        ['COL12A1','FN1'],
        ['COL16A1','COL1A2'],
        ['COL18A1','COL1A1'],
        ['COL18A1','COL1A2'],
        ['COL1A1','COL4A1'],
        ['COL1A1','COL5A1'],
        ['COL1A1','COL5A2'],
        ['COL1A1','COL6A3'],
        ['COL1A1','COL9A3'],
        ['COL1A1','COMP'],
        ['COL1A1','COPG1'],
        ['COL1A1','COX6C'],
        ['COL1A1','CPB1'],
        ['COL1A1','CRABP2'],
        ['COL1A1','CRIP2'],
        ['COL1A1','CSDE1'],
        ['COL1A1','CSRP1'],
        ['COL1A1','CTNNA1'],
        ['COL1A1','CTNNB1'],
        ['COL1A1','CTSA'],
        ['COL1A1','CTSB'],
        ['COL1A1','CTSD'],
        ['COL1A1','CTSK'],
        ['COL1A1','CTTN'],
        ['COL1A1','DCN'],
        ['COL1A1','DDR1'],
        ['COL1A1','DDX3X'],
        ['COL1A1','DDX5'],
        ['COL1A1','DHCR24'],
        ['COL1A1','DNM2'],
        ['COL1A1','DSP'],
        ['COL1A1','DSTN'],
        ['COL1A1','EEF1D'],
        ['COL1A1','EFHD1'],
        ['COL1A1','EIF3A'],
        ['COL1A1','EIF3B'],
        ['COL1A1','EIF4A1'],
        ['COL1A1','EIF4B'],
        ['COL1A1','EIF4G1'],
        ['COL1A1','EIF4G2'],
        ['COL1A1','EIF5A'],
        ['COL1A1','EMP2'],
        ['COL1A1','ENAH'],
        ['COL1A1','ENO1'],
        ['COL1A1','ENSA'],
        ['COL1A1','ERBB2'],
        ['COL1A1','ERGIC1'],
        ['COL1A1','ERGIC3'],
        ['COL1A1','ESR1'],
        ['COL1A1','ESYT1'],
        ['COL1A1','EZR'],
        ['COL1A1','F11R'],
        ['COL1A1','FBLN2'],
        ['COL1A1','FKBP4'],
        ['COL1A1','FKBP8'],
        ['COL1A1','FLII'],
        ['COL1A1','FLNA'],
        ['COL1A1','FLNB'],
        ['COL1A1','FLOT1'],
        ['COL1A1','FTH1'],
        ['COL1A1','FUS'],
        ['COL1A1','GATA3'],
        ['COL1A1','GDE1'],
        ['COL1A1','GDI2'],
        ['COL1A1','GLUL'],
        ['COL1A1','GLYR1'],
        ['COL1A1','GNAS'],
        ['COL1A1','GNB1'],
        ['COL1A1','GNB2'],
        ['COL1A1','GOLGA2'],
        ['COL1A1','GOLM1'],
        ['COL1A1','GPRC5A'],
        ['COL1A1','GPX4'],
        ['COL1A1','GRN'],
        ['COL1A1','GTF3C1'],
        ['COL1A1','H1F0'],
        ['COL1A1','H2AFY'],
        ['COL1A1','H3F3A'],
        ['COL1A1','HDGF'],
        ['COL1A1','HDLBP'],
        ['COL1A1','HLA-B'],
        ['COL1A1','HLA-C'],
        ['COL1A1','HLA-E'],
        ['COL1A1','HMGN1'],
        ['COL1A1','HMGN2'],
        ['COL1A1','HN1L'],
        ['COL1A1','HNRNPA2B1'],
        ['COL1A1','HNRNPA3'],
        ['COL1A1','HNRNPC'],
        ['COL1A1','HNRNPF'],
        ['COL1A1','HNRNPM'],
        ['COL1A1','HNRNPU'],
        ['COL1A1','HNRNPUL1'],
        ['COL1A1','HNRNPUL2'],
        ['COL1A1','HSP90AA1'],
        ['COL1A1','HSP90AB1'],
        ['COL1A1','HSP90B1'],
        ['COL1A1','HSPA1A'],
        ['COL1A1','HSPA5'],
        ['COL1A1','HSPA8'],
        ['COL1A1','HSPB1'],
        ['COL1A1','HSPG2'],
        ['COL1A1','HTRA1'],
        ['COL1A1','HUWE1'],
        ['COL1A1','IDH2'],
        ['COL1A1','IFI6'],
        ['COL1A1','IFITM3'],
        ['COL1A1','IGF2'],
        ['COL1A1','IGFBP4'],
        ['COL1A1','IGLL5'],
        ['COL1A1','IL6ST'],
        ['COL1A1','IQGAP1'],
        ['COL1A1','IRF2BP2'],
        ['COL1A1','ISLR'],
        ['COL1A1','ITGB1'],
        ['COL1A1','JUN'],
        ['COL1A1','JUNB'],
        ['COL1A1','JUP'],
        ['COL1A1','KIAA1324'],
        ['COL1A1','KRT18'],
        ['COL1A1','KRT19'],
        ['COL1A1','KRT6B'],
        ['COL1A1','KRT7'],
        ['COL1A1','KRT8'],
        ['COL1A1','LAMA5'],
        ['COL1A1','LAMB2'],
        ['COL1A1','LAPTM4A'],
        ['COL1A1','LASP1'],
        ['COL1A1','LDHA'],
        ['COL1A1','LGALS3BP'],
        ['COL1A1','LMNA'],
        ['COL1A1','LTBP2'],
        ['COL1A1','LUM'],
        ['COL1A1','LYZ'],
        ['COL1A1','MAGED2'],
        ['COL1A1','MAOB'],
        ['COL1A1','MAP3K1'],
        ['COL1A1','MARCKSL1'],
        ['COL1A1','MAZ'],
        ['COL1A1','MDK'],
        ['COL1A1','MLLT6'],
        ['COL1A1','MLPH'],
        ['COL1A1','MMP11'],
        ['COL1A1','MMP14'],
        ['COL1A1','MMP2'],
        ['COL1A1','MORF4L1'],
        ['COL1A1','MSN'],
        ['COL1A1','MTRNR2L2'],
        ['COL1A1','MUC1'],
        ['COL1A1','MUCL1'],
        ['COL1A1','MXRA5'],
        ['COL1A1','MXRA7'],
        ['COL1A1','MYH14'],
        ['COL1A1','NBL1'],
        ['COL1A1','NBPF9'],
        ['COL1A1','NCL'],
        ['COL1A1','NCOR2'],
        ['COL1A1','NDFIP1'],
        ['COL1A1','NONO'],
        ['COL1A1','NOTCH3'],
        ['COL1A1','NPM1'],
        ['COL1A1','NPY1R'],
        ['COL1A1','NUMA1'],
        ['COL1A1','NUPR1'],
        ['COL1A1','ORMDL3'],
        ['COL1A1','OS9'],
        ['COL1A1','P4HB'],
        ['COL1A1','PABPC1'],
        ['COL1A1','PARP1'],
        ['COL1A1','PBX1'],
        ['COL1A1','PCBP2'],
        ['COL1A1','PDIA3'],
        ['COL1A1','PDIA6'],
        ['COL1A1','PDXDC1'],
        ['COL1A1','PFN1'],
        ['COL1A1','PGK1'],
        ['COL1A1','PKM'],
        ['COL1A1','PLXNB2'],
        ['COL1A1','PLXND1'],
        ['COL1A1','PMEPA1'],
        ['COL1A1','POLR2A'],
        ['COL1A1','POSTN'],
        ['COL1A1','PPDPF'],
        ['COL1A1','PPIA'],
        ['COL1A1','PPIB'],
        ['COL1A1','PPP2R1A'],
        ['COL1A1','PPT1'],
        ['COL1A1','PREX1'],
        ['COL1A1','PRKDC'],
        ['COL1A1','PRLR'],
        ['COL1A1','PRPF8'],
        ['COL1A1','PSME1'],
        ['COL1A1','PTBP1'],
        ['COL1A1','PTMA'],
        ['COL1A1','PTMS'],
        ['COL1A1','PTPRF'],
        ['COL1A1','PTTG1IP'],
        ['COL1A1','PUF60'],
        ['COL1A1','PYGB'],
        ['COL1A1','RAC1'],
        ['COL1A1','RAD21'],
        ['COL1A1','RALY'],
        ['COL1A1','RBP1'],
        ['COL1A1','RHOA'],
        ['COL1A1','RHOB'],
        ['COL1A1','RNF213'],
        ['COL1A1','RPL10'],
        ['COL1A1','RPL12'],
        ['COL1A1','RPL14'],
        ['COL1A1','RPL17'],
        ['COL1A1','RPL18'],
        ['COL1A1','RPL18A'],
        ['COL1A1','RPL21'],
        ['COL1A1','RPL22'],
        ['COL1A1','RPL23'],
        ['COL1A1','RPL23A'],
        ['COL1A1','RPL24'],
        ['COL1A1','RPL26'],
        ['COL1A1','RPL27'],
        ['COL1A1','RPL27A'],
        ['COL1A1','RPL28'],
        ['COL1A1','RPL29'],
        ['COL1A1','RPL3'],
        ['COL1A1','RPL30'],
        ['COL1A1','RPL39'],
        ['COL1A1','RPL4'],
        ['COL1A1','RPL5'],
        ['COL1A1','RPL6'],
        ['COL1A1','RPL7'],
        ['COL1A1','RPL8'],
        ['COL1A1','RPLP0'],
        ['COL1A1','RPLP1'],
        ['COL1A1','RPN1'],
        ['COL1A1','RPN2'],
        ['COL1A1','RPS10'],
        ['COL1A1','RPS11'],
        ['COL1A1','RPS12'],
        ['COL1A1','RPS14'],
        ['COL1A1','RPS15A'],
        ['COL1A1','RPS16'],
        ['COL1A1','RPS19'],
        ['COL1A1','RPS2'],
        ['COL1A1','RPS20'],
        ['COL1A1','RPS25'],
        ['COL1A1','RPS27'],
        ['COL1A1','RPS27A'],
        ['COL1A1','RPS7'],
        ['COL1A1','RPS9'],
        ['COL1A1','RRBP1'],
        ['COL1A1','RTN4'],
        ['COL1A1','S100A11'],
        ['COL1A1','S100A9'],
        ['COL1A1','SCARB2'],
        ['COL1A1','SDC1'],
        ['COL1A1','SDC4'],
        ['COL1A1','SDF4'],
        ['COL1A1','SEC61A1'],
        ['COL1A1','SEC62'],
        ['COL1A1','SEL1L'],
        ['COL1A1','SEPT2'],
        ['COL1A1','SEPT9'],
        ['COL1A1','SERPINA3'],
        ['COL1A1','SERPING1'],
        ['COL1A1','SERPINH1'],
        ['COL1A1','SET'],
        ['COL1A1','SF3B1'],
        ['COL1A1','SGPL1'],
        ['COL1A1','SH3BGRL'],
        ['COL1A1','SLC25A6'],
        ['COL1A1','SLC39A6'],
        ['COL1A1','SLC39A7'],
        ['COL1A1','SLC44A2'],
        ['COL1A1','SLC9A3R1'],
        ['COL1A1','SNRNP200'],
        ['COL1A1','SON'],
        ['COL1A1','SOX4'],
        ['COL1A1','SPINT2'],
        ['COL1A1','SPTAN1'],
        ['COL1A1','SQSTM1'],
        ['COL1A1','SRPRA'],
        ['COL1A1','SRSF5'],
        ['COL1A1','SSR2'],
        ['COL1A1','STAU1'],
        ['COL1A1','STC2'],
        ['COL1A1','SULF2'],
        ['COL1A1','SUN2'],
        ['COL1A1','TAGLN'],
        ['COL1A1','TAGLN2'],
        ['COL1A1','TAPBP'],
        ['COL1A1','THBS1'],
        ['COL1A1','THBS2'],
        ['COL1A1','TIMP1'],
        ['COL1A1','TIMP3'],
        ['COL1A1','TM9SF2'],
        ['COL1A1','TMBIM6'],
        ['COL1A1','TMC4'],
        ['COL1A1','TMEM59'],
        ['COL1A1','TNKS1BP1'],
        ['COL1A1','TNRC18'],
        ['COL1A1','TOMM20'],
        ['COL1A1','TPI1'],
        ['COL1A1','TPM1'],
        ['COL1A1','TPM3'],
        ['COL1A1','TPM4'],
        ['COL1A1','TRIM8'],
        ['COL1A1','TRPS1'],
        ['COL1A1','TTC3'],
        ['COL1A1','TUBA1A'],
        ['COL1A1','TUBA1C'],
        ['COL1A1','TXNDC5'],
        ['COL1A1','UBA1'],
        ['COL1A1','UBTF'],
        ['COL1A1','UCP2'],
        ['COL1A1','VCAN'],
        ['COL1A1','VCP'],
        ['COL1A1','VDAC1'],
        ['COL1A1','WASF2'],
        ['COL1A1','XBP1'],
        ['COL1A1','XRCC5'],
        ['COL1A1','YWHAB'],
        ['COL1A1','YWHAQ'],
        ['COL1A1','YWHAZ'],
        ['COL1A1','ZC3H11A'],
        ['COL1A1','ZFP36L1'],
        ['COL1A2','COL5A1'],
        ['COL1A2','COL5A2'],
        ['COL1A2','COX6C'],
        ['COL1A2','CPB1'],
        ['COL1A2','CSDE1'],
        ['COL1A2','CTSB'],
        ['COL1A2','CTSZ'],
        ['COL1A2','DDR1'],
        ['COL1A2','DDX5'],
        ['COL1A2','DHCR24'],
        ['COL1A2','DSP'],
        ['COL1A2','EEF1G'],
        ['COL1A2','EIF1'],
        ['COL1A2','EIF4G2'],
        ['COL1A2','EIF4H'],
        ['COL1A2','ELN'],
        ['COL1A2','ENO1'],
        ['COL1A2','ENSA'],
        ['COL1A2','ERBB2'],
        ['COL1A2','ERBB3'],
        ['COL1A2','FASN'],
        ['COL1A2','FKBP4'],
        ['COL1A2','FLNA'],
        ['COL1A2','FLNB'],
        ['COL1A2','FTH1'],
        ['COL1A2','GAPDH'],
        ['COL1A2','GLUL'],
        ['COL1A2','GNB1'],
        ['COL1A2','GRINA'],
        ['COL1A2','H3F3A'],
        ['COL1A2','HDLBP'],
        ['COL1A2','HLA-A'],
        ['COL1A2','HLA-B'],
        ['COL1A2','HLA-C'],
        ['COL1A2','HLA-DRA'],
        ['COL1A2','HLA-DRB1'],
        ['COL1A2','HMGB1'],
        ['COL1A2','HNRNPA1'],
        ['COL1A2','HNRNPC'],
        ['COL1A2','HNRNPK'],
        ['COL1A2','HNRNPUL1'],
        ['COL1A2','HSP90AB1'],
        ['COL1A2','HSP90B1'],
        ['COL1A2','HSPA1A'],
        ['COL1A2','HSPA1B'],
        ['COL1A2','HTRA1'],
        ['COL1A2','IFI6'],
        ['COL1A2','IGLL5'],
        ['COL1A2','IQGAP1'],
        ['COL1A2','ITGB1'],
        ['COL1A2','KRT18'],
        ['COL1A2','KRT19'],
        ['COL1A2','KRT7'],
        ['COL1A2','KRT8'],
        ['COL1A2','LAPTM4A'],
        ['COL1A2','LASP1'],
        ['COL1A2','LGALS3BP'],
        ['COL1A2','LTF'],
        ['COL1A2','MAGED2'],
        ['COL1A2','MMP11'],
        ['COL1A2','MMP14'],
        ['COL1A2','MTRNR2L2'],
        ['COL1A2','MTRNR2L8'],
        ['COL1A2','MUC1'],
        ['COL1A2','MUCL1'],
        ['COL1A2','NACA'],
        ['COL1A2','NBL1'],
        ['COL1A2','NONO'],
        ['COL1A2','NUMA1'],
        ['COL1A2','OS9'],
        ['COL1A2','P4HB'],
        ['COL1A2','PABPC1'],
        ['COL1A2','PDCD4'],
        ['COL1A2','PDIA3'],
        ['COL1A2','PFN1'],
        ['COL1A2','PKM'],
        ['COL1A2','POSTN'],
        ['COL1A2','PPIA'],
        ['COL1A2','PRKCSH'],
        ['COL1A2','PTMS'],
        ['COL1A2','RAB7A'],
        ['COL1A2','RHOA'],
        ['COL1A2','RPL10'],
        ['COL1A2','RPL11'],
        ['COL1A2','RPL12'],
        ['COL1A2','RPL13'],
        ['COL1A2','RPL18'],
        ['COL1A2','RPL18A'],
        ['COL1A2','RPL19'],
        ['COL1A2','RPL23'],
        ['COL1A2','RPL23A'],
        ['COL1A2','RPL27'],
        ['COL1A2','RPL27A'],
        ['COL1A2','RPL28'],
        ['COL1A2','RPL3'],
        ['COL1A2','RPL35'],
        ['COL1A2','RPL37'],
        ['COL1A2','RPL5'],
        ['COL1A2','RPL6'],
        ['COL1A2','RPL7'],
        ['COL1A2','RPL7A'],
        ['COL1A2','RPL8'],
        ['COL1A2','RPLP1'],
        ['COL1A2','RPS12'],
        ['COL1A2','RPS14'],
        ['COL1A2','RPS17'],
        ['COL1A2','RPS19'],
        ['COL1A2','RPS2'],
        ['COL1A2','RPS3'],
        ['COL1A2','RPS3A'],
        ['COL1A2','RPS6'],
        ['COL1A2','RPS9'],
        ['COL1A2','RRBP1'],
        ['COL1A2','S100A9'],
        ['COL1A2','SARAF'],
        ['COL1A2','SCGB2A2'],
        ['COL1A2','SDC1'],
        ['COL1A2','SEPT2'],
        ['COL1A2','SERPINA3'],
        ['COL1A2','SFRP2'],
        ['COL1A2','SLC25A6'],
        ['COL1A2','SLC39A6'],
        ['COL1A2','SLC40A1'],
        ['COL1A2','SPTAN1'],
        ['COL1A2','SRCAP'],
        ['COL1A2','SRRM2'],
        ['COL1A2','STC2'],
        ['COL1A2','TIMP3'],
        ['COL1A2','TMBIM6'],
        ['COL1A2','TMSB10'],
        ['COL1A2','TMSB4X'],
        ['COL1A2','TNRC18'],
        ['COL1A2','TPI1'],
        ['COL1A2','TUBA1B'],
        ['COL1A2','UBB'],
        ['COL1A2','UBC'],
        ['COL1A2','VCAN'],
        ['COL1A2','XBP1'],
        ['COL3A1','COL5A1'],
        ['COL3A1','COL5A2'],
        ['COL3A1','COL6A1'],
        ['COL3A1','COL6A3'],
        ['COL3A1','CPB1'],
        ['COL3A1','CSDE1'],
        ['COL3A1','CTSB'],
        ['COL3A1','CTSD'],
        ['COL3A1','DHCR24'],
        ['COL3A1','DYNC1H1'],
        ['COL3A1','EIF4G2'],
        ['COL3A1','ERBB2'],
        ['COL3A1','FASN'],
        ['COL3A1','FLNA'],
        ['COL3A1','FOXA1'],
        ['COL3A1','FTL'],
        ['COL3A1','GAPDH'],
        ['COL3A1','GLTSCR2'],
        ['COL3A1','GLUL'],
        ['COL3A1','GNAS'],
        ['COL3A1','H3F3A'],
        ['COL3A1','HLA-A'],
        ['COL3A1','HMGB1'],
        ['COL3A1','HNRNPA1'],
        ['COL3A1','HNRNPA2B1'],
        ['COL3A1','HNRNPU'],
        ['COL3A1','HSP90AA1'],
        ['COL3A1','HSP90AB1'],
        ['COL3A1','HSPA1A'],
        ['COL3A1','HSPA8'],
        ['COL3A1','HSPB1'],
        ['COL3A1','HSPD1'],
        ['COL3A1','IGLL5'],
        ['COL3A1','IL6ST'],
        ['COL3A1','JUP'],
        ['COL3A1','KRT19'],
        ['COL3A1','KRT8'],
        ['COL3A1','LAMP1'],
        ['COL3A1','LGALS3BP'],
        ['COL3A1','LUM'],
        ['COL3A1','MAGED2'],
        ['COL3A1','MDK'],
        ['COL3A1','MTRNR2L1'],
        ['COL3A1','MTRNR2L2'],
        ['COL3A1','MUC5B'],
        ['COL3A1','MUCL1'],
        ['COL3A1','MYH9'],
        ['COL3A1','NUMA1'],
        ['COL3A1','P4HB'],
        ['COL3A1','PABPC1'],
        ['COL3A1','PDIA3'],
        ['COL3A1','PFN1'],
        ['COL3A1','PKM'],
        ['COL3A1','PLEC'],
        ['COL3A1','PLXNB2'],
        ['COL3A1','POSTN'],
        ['COL3A1','PTMS'],
        ['COL3A1','PTPRF'],
        ['COL3A1','RHOA'],
        ['COL3A1','RPL10'],
        ['COL3A1','RPL12'],
        ['COL3A1','RPL15'],
        ['COL3A1','RPL19'],
        ['COL3A1','RPL21'],
        ['COL3A1','RPL27A'],
        ['COL3A1','RPL29'],
        ['COL3A1','RPL3'],
        ['COL3A1','RPL32'],
        ['COL3A1','RPL36A-HNRNPH2'],
        ['COL3A1','RPS18'],
        ['COL3A1','RPS3'],
        ['COL3A1','RPS4X'],
        ['COL3A1','RPS9'],
        ['COL3A1','S100A9'],
        ['COL3A1','SCD'],
        ['COL3A1','SCGB2A2'],
        ['COL3A1','SDC4'],
        ['COL3A1','SERPINA3'],
        ['COL3A1','SHISA2'],
        ['COL3A1','SLC25A6'],
        ['COL3A1','SLC39A6'],
        ['COL3A1','SRRM2'],
        ['COL3A1','STC2'],
        ['COL3A1','TAGLN2'],
        ['COL3A1','THBS1'],
        ['COL3A1','THBS2'],
        ['COL3A1','THY1'],
        ['COL3A1','TMBIM1'],
        ['COL3A1','TMBIM6'],
        ['COL3A1','TMSB10'],
        ['COL3A1','TPM4'],
        ['COL3A1','TUBA1B'],
        ['COL3A1','TUBB'],
        ['COL3A1','UBB'],
        ['COL3A1','XBP1'],
        ['COL4A2','FN1'],
        ['COL4A2','LASP1'],
        ['COL5A1','FN1'],
        ['COL5A2','FN1'],
        ['COL6A1','CPB1'],
        ['COL6A1','ERBB2'],
        ['COL6A1','FN1'],
        ['COL6A1','HLA-B'],
        ['COL6A1','RPL3'],
        ['COL6A1','SCGB2A2'],
        ['COL6A2','PRRC2B'],
        ['COL6A3','CPB1'],
        ['COL6A3','DHCR24'],
        ['COL6A3','FLNA'],
        ['COL6A3','FN1'],
        ['COL6A3','GAPDH'],
        ['COL6A3','GLUL'],
        ['COL6A3','HLA-B'],
        ['COL6A3','IGLL5'],
        ['COL6A3','KRT18'],
        ['COL6A3','KRT8'],
        ['COL6A3','MTRNR2L8'],
        ['COL6A3','PTMA'],
        ['COL6A3','RPLP0'],
        ['COL6A3','SPARC'],
        ['COL6A3','TPT1'],
        ['COL6A3','XBP1'],
        ['COPG1','EEF1A1'],
        ['COPS6','RPL37A'],
        ['COTL1','EEF1A1'],
        ['COX6A1','EIF3H'],
        ['COX6A1','RPS10'],
        ['COX6B1','IFITM1'],
        ['COX6C','CPB1'],
        ['COX6C','EEF1A1'],
        ['COX6C','EEF2'],
        ['COX6C','EMP2'],
        ['COX6C','FN1'],
        ['COX6C','GAPDH'],
        ['COX6C','GNAS'],
        ['COX6C','HSPB1'],
        ['COX6C','ITM2B'],
        ['COX6C','KRT19'],
        ['COX6C','KRT8'],
        ['COX6C','MGP'],
        ['COX6C','MTRNR2L2'],
        ['COX6C','MTRNR2L8'],
        ['COX6C','PTPRF'],
        ['COX6C','RPL14'],
        ['COX6C','RPL5'],
        ['COX6C','RPL8'],
        ['COX6C','RPS18'],
        ['COX6C','SLC39A6'],
        ['COX6C','TMSB10'],
        ['COX6C','TPT1'],
        ['CPA3','TMSB10'],
        ['CPB1','CRABP2'],
        ['CPB1','CSDE1'],
        ['CPB1','CTSB'],
        ['CPB1','DDX23'],
        ['CPB1','DHX9'],
        ['CPB1','DNAJA1'],
        ['CPB1','EEF1A2'],
        ['CPB1','EEF1D'],
        ['CPB1','EEF1G'],
        ['CPB1','EIF3A'],
        ['CPB1','EIF4A1'],
        ['CPB1','EIF4B'],
        ['CPB1','EIF4G1'],
        ['CPB1','EIF4G2'],
        ['CPB1','EIF4H'],
        ['CPB1','EIF5A'],
        ['CPB1','ENO2'],
        ['CPB1','ESRP1'],
        ['CPB1','FBRS'],
        ['CPB1','FDFT1'],
        ['CPB1','FN1'],
        ['CPB1','FTH1'],
        ['CPB1','FUBP1'],
        ['CPB1','GAPDH'],
        ['CPB1','GLUL'],
        ['CPB1','GNAS'],
        ['CPB1','GNB2L1'],
        ['CPB1','H3F3B'],
        ['CPB1','HLA-B'],
        ['CPB1','HMGN2'],
        ['CPB1','HNRNPC'],
        ['CPB1','HSP90AA1'],
        ['CPB1','HSP90AB1'],
        ['CPB1','HSPD1'],
        ['CPB1','IGLL5'],
        ['CPB1','ITGB1'],
        ['CPB1','JUP'],
        ['CPB1','KPNB1'],
        ['CPB1','LRRC59'],
        ['CPB1','MMP14'],
        ['CPB1','MTDH'],
        ['CPB1','NACA'],
        ['CPB1','NPNT'],
        ['CPB1','NUCB2'],
        ['CPB1','OS9'],
        ['CPB1','P4HB'],
        ['CPB1','PI4KA'],
        ['CPB1','POLR2A'],
        ['CPB1','PPIA'],
        ['CPB1','PPT1'],
        ['CPB1','PSME2'],
        ['CPB1','PTMA'],
        ['CPB1','PTPRF'],
        ['CPB1','RAB7A'],
        ['CPB1','RBM8A'],
        ['CPB1','REEP5'],
        ['CPB1','RNF103-CHMP3'],
        ['CPB1','RPL10A'],
        ['CPB1','RPL13A'],
        ['CPB1','RPL29'],
        ['CPB1','RPL30'],
        ['CPB1','RPL32'],
        ['CPB1','RPL37A'],
        ['CPB1','RPL38'],
        ['CPB1','RPL41'],
        ['CPB1','RPL5'],
        ['CPB1','RPL7'],
        ['CPB1','RPL7A'],
        ['CPB1','RPL9'],
        ['CPB1','RPLP0'],
        ['CPB1','RPS18'],
        ['CPB1','RPS19'],
        ['CPB1','RPS27'],
        ['CPB1','RPS3'],
        ['CPB1','RPS6'],
        ['CPB1','RSF1'],
        ['CPB1','SCARB2'],
        ['CPB1','SCD'],
        ['CPB1','SDC1'],
        ['CPB1','SEC61A1'],
        ['CPB1','SEL1L'],
        ['CPB1','SEPT2'],
        ['CPB1','SERPINA3'],
        ['CPB1','SF3B4'],
        ['CPB1','SLC39A6'],
        ['CPB1','SLC7A2'],
        ['CPB1','SLC7A8'],
        ['CPB1','SMARCA5'],
        ['CPB1','SNRNP200'],
        ['CPB1','SPARC'],
        ['CPB1','SREBF2'],
        ['CPB1','SRP14'],
        ['CPB1','SRRM2'],
        ['CPB1','SSR2'],
        ['CPB1','SYNGR2'],
        ['CPB1','THRAP3'],
        ['CPB1','TLN1'],
        ['CPB1','TMBIM6'],
        ['CPB1','TMSB10'],
        ['CPB1','TMSB4X'],
        ['CPB1','TOP2A'],
        ['CPB1','TPD52'],
        ['CPB1','TPT1'],
        ['CPB1','TUBA1B'],
        ['CPB1','UBC'],
        ['CPB1','XBP1'],
        ['CPT1A','MALL'],
        ['CPVL','PLIN3'],
        ['CRABP2','RPL13A'],
        ['CRABP2','RPL3'],
        ['CRABP2','SLC39A6'],
        ['CRADD','PATE4'],
        ['CS','EEF1A1'],
        ['CSDE1','FN1'],
        ['CSDE1','GAPDH'],
        ['CTSB','EEF2'],
        ['CTSB','FN1'],
        ['CTSD','DHCR24'],
        ['CTSD','ENO1'],
        ['CTSD','ERBB2'],
        ['CTSD','FN1'],
        ['CTSD','GAPDH'],
        ['CTSD','GNAS'],
        ['CTSD','HLA-B'],
        ['CTSD','HLA-C'],
        ['CTSD','IGLL5'],
        ['CTSD','KRT8'],
        ['CTSD','MTRNR2L8'],
        ['CTSD','MYH9'],
        ['CTSD','P4HB'],
        ['CTSD','PSAP'],
        ['CTSD','RPS2'],
        ['CTSD','RPS6'],
        ['CTSD','SCGB2A2'],
        ['CTTN','GAPDH'],
        ['CXCL14','HNRNPL'],
        ['CXCL14','IFI6'],
        ['CYP2B6','EEF2'],
        ['DAP','EEF1A1'],
        ['DDB1','FN1'],
        ['DDIT4','EEF1A1'],
        ['DDOST','FASN'],
        ['DDX17','FN1'],
        ['DDX17','PTMA'],
        ['DENND3','PTK2'],
        ['DHCR24','KRT19'],
        ['DHCR24','PIP'],
        ['DNAJA1','EEF1A1'],
        ['DNAJC1','EEF1A1'],
        ['DNAL1','EPHX1'],
        ['DNAL1','IL34'],
        ['DPY30','SPAST'],
        ['DSG3','IFI44L'],
        ['DSP','FN1'],
        ['DSP','FTH1'],
        ['DSP','IGLL5'],
        ['DSP','ITGB1'],
        ['DSP','KRT14'],
        ['DSP','KRT19'],
        ['DSP','PTMA'],
        ['DSP','YWHAZ'],
        ['DYX1C1','EPHX1'],
        ['DYX1C1','IL34'],
        ['ECE1','GAPDH'],
        ['EEF1A1','EEF1A2'],
        ['EEF1A1','EIF3M'],
        ['EEF1A1','EIF4G1'],
        ['EEF1A1','ELL2'],
        ['EEF1A1','ENSA'],
        ['EEF1A1','FADS2'],
        ['EEF1A1','FBRS'],
        ['EEF1A1','FKBP9'],
        ['EEF1A1','FUS'],
        ['EEF1A1','GDI1'],
        ['EEF1A1','GFRA1'],
        ['EEF1A1','GSPT1'],
        ['EEF1A1','HSPA9'],
        ['EEF1A1','IDH2'],
        ['EEF1A1','IFI6'],
        ['EEF1A1','IFITM1'],
        ['EEF1A1','IGF2'],
        ['EEF1A1','IGSF3'],
        ['EEF1A1','ILF2'],
        ['EEF1A1','KDELR1'],
        ['EEF1A1','KDELR2'],
        ['EEF1A1','KRT6A'],
        ['EEF1A1','LAPTM4B'],
        ['EEF1A1','LASP1'],
        ['EEF1A1','LIMK2'],
        ['EEF1A1','LITAF'],
        ['EEF1A1','LRRC41'],
        ['EEF1A1','MAGED1'],
        ['EEF1A1','MAL2'],
        ['EEF1A1','MCM7'],
        ['EEF1A1','MDH2'],
        ['EEF1A1','MDK'],
        ['EEF1A1','MMP11'],
        ['EEF1A1','MUC1'],
        ['EEF1A1','MUC5B'],
        ['EEF1A1','MUC6'],
        ['EEF1A1','MXRA5'],
        ['EEF1A1','MYL12B'],
        ['EEF1A1','NFIC'],
        ['EEF1A1','NPNT'],
        ['EEF1A1','NPY1R'],
        ['EEF1A1','NUPR1'],
        ['EEF1A1','OS9'],
        ['EEF1A1','PDAP1'],
        ['EEF1A1','PFN2'],
        ['EEF1A1','PGAM1'],
        ['EEF1A1','PRKRIR'],
        ['EEF1A1','PSMB1'],
        ['EEF1A1','PSMB4'],
        ['EEF1A1','PSMC3'],
        ['EEF1A1','PSMD3'],
        ['EEF1A1','PSMF1'],
        ['EEF1A1','RAB1B'],
        ['EEF1A1','RAB25'],
        ['EEF1A1','RPN1'],
        ['EEF1A1','RTN3'],
        ['EEF1A1','SDC1'],
        ['EEF1A1','SERF2'],
        ['EEF1A1','SERPINH1'],
        ['EEF1A1','SLC12A7'],
        ['EEF1A1','SLC38A2'],
        ['EEF1A1','SMC1A'],
        ['EEF1A1','SOD1'],
        ['EEF1A1','SPDEF'],
        ['EEF1A1','SRSF3'],
        ['EEF1A1','SSRP1'],
        ['EEF1A1','STAC2'],
        ['EEF1A1','STC1'],
        ['EEF1A1','SULF2'],
        ['EEF1A1','SYNGR2'],
        ['EEF1A1','TAPBP'],
        ['EEF1A1','TBL1XR1'],
        ['EEF1A1','TCEB2'],
        ['EEF1A1','TFF1'],
        ['EEF1A1','TFF3'],
        ['EEF1A1','TFG'],
        ['EEF1A1','TKT'],
        ['EEF1A1','TMEM189-UBE2V1'],
        ['EEF1A1','TNRC18'],
        ['EEF1A1','TOP2A'],
        ['EEF1A1','TPX2'],
        ['EEF1A1','UBE2I'],
        ['EEF1A1','UBE2J1'],
        ['EEF1A1','UBE3C'],
        ['EEF1A1','USP7'],
        ['EEF1A1','VTCN1'],
        ['EEF1A1','WFDC2'],
        ['EEF1A2','EEF2'],
        ['EEF1A2','FASN'],
        ['EEF1D','FTH1'],
        ['EEF1D','RPL13A'],
        ['EEF1D','TMBIM6'],
        ['EEF1G','ERBB2'],
        ['EEF1G','FDCSP'],
        ['EEF1G','HLA-A'],
        ['EEF1G','HLA-B'],
        ['EEF1G','HSPA1B'],
        ['EEF1G','HSPA8'],
        ['EEF1G','KRT5'],
        ['EEF1G','LTF'],
        ['EEF1G','P4HB'],
        ['EEF1G','PREX1'],
        ['EEF1G','PTMS'],
        ['EEF1G','SCGB2A2'],
        ['EEF1G','SERPINA3'],
        ['EEF1G','SLC39A6'],
        ['EEF1G','TMSB4X'],
        ['EEF1G','TUBA1B'],
        ['EEF1G','TUBB'],
        ['EEF1G','XBP1'],
        ['EEF1G','YWHAZ'],
        ['EEF2','EIF3A'],
        ['EEF2','EIF4G1'],
        ['EEF2','ENSA'],
        ['EEF2','ESR1'],
        ['EEF2','FN1'],
        ['EEF2','HADHA'],
        ['EEF2','HDLBP'],
        ['EEF2','HLA-A'],
        ['EEF2','HSPA8'],
        ['EEF2','HSPB1'],
        ['EEF2','HSPG2'],
        ['EEF2','HUWE1'],
        ['EEF2','IGFBP5'],
        ['EEF2','KRT8'],
        ['EEF2','LTF'],
        ['EEF2','LY6E'],
        ['EEF2','LYZ'],
        ['EEF2','MSN'],
        ['EEF2','MUC1'],
        ['EEF2','P4HB'],
        ['EEF2','PKM'],
        ['EEF2','POSTN'],
        ['EEF2','PTMS'],
        ['EEF2','RPS12'],
        ['EEF2','SFRP2'],
        ['EEF2','SLC39A6'],
        ['EEF2','SLC9A3R1'],
        ['EEF2','TBC1D9'],
        ['EEF2','TMSB10'],
        ['EEF2','TUBB'],
        ['EFHD1','RPL41'],
        ['EHF','ELF5'],
        ['EIF3E','GAPDH'],
        ['EIF3E','P4HB'],
        ['EIF4A1','FTH1'],
        ['EIF4A1','RPL3'],
        ['EIF4B','MTRNR2L8'],
        ['EIF4B','SLC39A6'],
        ['EIF4G1','FASN'],
        ['EIF4G1','MYH9'],
        ['EIF4G2','HLA-C'],
        ['EIF4G2','HSP90AB1'],
        ['EIF4G2','MGP'],
        ['EIF4G2','YWHAZ'],
        ['EIF4H','KRT14'],
        ['EIF5A','FTH1'],
        ['EIF5A','S100A6'],
        ['ENAH','FTH1'],
        ['ENAH','RPS8'],
        ['ENO1','FLNA'],
        ['ENO1','FN1'],
        ['ENO1','GLUL'],
        ['ENO1','HSP90AA1'],
        ['ENO1','KRT5'],
        ['ENO1','MYH9'],
        ['ENO1','PTMA'],
        ['ENO1','RPL11'],
        ['ENO1','RPL22'],
        ['ENO1','RPL7'],
        ['ENO1','TMSB4X'],
        ['ENPP5','SHISA9'],
        ['EPHB4','IGLL5'],
        ['EPHX1','HHLA2'],
        ['EPHX1','HHLA3'],
        ['EPHX1','PRELP'],
        ['EPHX1','ZNF578'],
        ['ERBB2','FASN'],
        ['ERBB2','FN1'],
        ['ERBB2','FTH1'],
        ['ERBB2','FTL'],
        ['ERBB2','GNAS'],
        ['ERBB2','IGLL5'],
        ['ERBB2','KRT19'],
        ['ERBB2','MTRNR2L8'],
        ['ERBB2','PPIA'],
        ['ERBB2','PPP1R1B'],
        ['ERBB2','RPL13A'],
        ['ERBB2','RPL18A'],
        ['ERBB2','RPL19'],
        ['ERBB2','RPL3'],
        ['ERBB2','RPS18'],
        ['ERBB2','SQSTM1'],
        ['ERBB2','UBC'],
        ['ERBB3','IFI6'],
        ['EZR','RPS3A'],
        ['F5','SCD5'],
        ['FABP7','RPS19'],
        ['FAR2','RAB6C'],
        ['FASN','FN1'],
        ['FASN','GAPDH'],
        ['FASN','GNAS'],
        ['FASN','H1F0'],
        ['FASN','HLA-B'],
        ['FASN','HSP90AA1'],
        ['FASN','HSP90AB1'],
        ['FASN','HSPA5'],
        ['FASN','KRT18'],
        ['FASN','KRT8'],
        ['FASN','MUCL1'],
        ['FASN','PGK1'],
        ['FASN','PKM'],
        ['FASN','PPIA'],
        ['FASN','PSAP'],
        ['FASN','PTMA'],
        ['FASN','PTPN1'],
        ['FASN','RPL18A'],
        ['FASN','RPL3'],
        ['FASN','RPL4'],
        ['FASN','RPL8'],
        ['FASN','RRBP1'],
        ['FASN','TUBA1B'],
        ['FASN','XBP1'],
        ['FBLIM1','ZNF528'],
        ['FBXO17','SUOX'],
        ['FBXO6','LOC100130370'],
        ['FDFT1','ZNF589'],
        ['FKBP4','FTH1'],
        ['FKBP5','LY6G5B'],
        ['FKBP6','TRIM73'],
        ['FKBP6','TRIM74'],
        ['FLNA','FN1'],
        ['FLNA','KRT19'],
        ['FLNA','LTF'],
        ['FLNA','MYL6'],
        ['FLNA','RPL3'],
        ['FLNB','FN1'],
        ['FLNB','PTMA'],
        ['FLNB','SCD'],
        ['FN1','FTL'],
        ['FN1','GANAB'],
        ['FN1','GAPDH'],
        ['FN1','GATA3'],
        ['FN1','GATM'],
        ['FN1','GLUL'],
        ['FN1','GNAS'],
        ['FN1','HDGF'],
        ['FN1','HLA-A'],
        ['FN1','HLA-B'],
        ['FN1','HLA-C'],
        ['FN1','HLA-DRA'],
        ['FN1','HLA-E'],
        ['FN1','HNRNPA1'],
        ['FN1','HNRNPA2B1'],
        ['FN1','HNRNPK'],
        ['FN1','HNRNPU'],
        ['FN1','HSP90AA1'],
        ['FN1','HSP90AB1'],
        ['FN1','HSP90B1'],
        ['FN1','HSPA1A'],
        ['FN1','HSPA1B'],
        ['FN1','HSPA8'],
        ['FN1','HUWE1'],
        ['FN1','IGFBP4'],
        ['FN1','IGFBP5'],
        ['FN1','IGLL5'],
        ['FN1','JUP'],
        ['FN1','KRT14'],
        ['FN1','KRT19'],
        ['FN1','KRT8'],
        ['FN1','LDHA'],
        ['FN1','LGALS3BP'],
        ['FN1','LTF'],
        ['FN1','LYZ'],
        ['FN1','MAGED2'],
        ['FN1','MGP'],
        ['FN1','MMP11'],
        ['FN1','MMP14'],
        ['FN1','MSN'],
        ['FN1','MTRNR2L2'],
        ['FN1','MTRNR2L8'],
        ['FN1','MUC1'],
        ['FN1','MUC5B'],
        ['FN1','MYH9'],
        ['FN1','MYL9'],
        ['FN1','NONO'],
        ['FN1','NUMA1'],
        ['FN1','P4HB'],
        ['FN1','PDIA3'],
        ['FN1','PKM'],
        ['FN1','PLOD1'],
        ['FN1','POSTN'],
        ['FN1','PPP1R1B'],
        ['FN1','PRDX1'],
        ['FN1','PRRC2A'],
        ['FN1','PSAP'],
        ['FN1','PTMA'],
        ['FN1','PTMS'],
        ['FN1','PTPRF'],
        ['FN1','RPL10'],
        ['FN1','RPL11'],
        ['FN1','RPL12'],
        ['FN1','RPL13'],
        ['FN1','RPL13A'],
        ['FN1','RPL18'],
        ['FN1','RPL18A'],
        ['FN1','RPL23'],
        ['FN1','RPL27A'],
        ['FN1','RPL3'],
        ['FN1','RPL30'],
        ['FN1','RPL31'],
        ['FN1','RPL4'],
        ['FN1','RPL41'],
        ['FN1','RPL6'],
        ['FN1','RPL7'],
        ['FN1','RPL8'],
        ['FN1','RPLP0'],
        ['FN1','RPLP1'],
        ['FN1','RPS11'],
        ['FN1','RPS14'],
        ['FN1','RPS18'],
        ['FN1','RPS2'],
        ['FN1','RPS24'],
        ['FN1','RPS3'],
        ['FN1','RPS6'],
        ['FN1','RPS8'],
        ['FN1','S100A6'],
        ['FN1','SCGB1D2'],
        ['FN1','SCGB2A2'],
        ['FN1','SFRP1'],
        ['FN1','SRRM2'],
        ['FN1','STAT1'],
        ['FN1','THBS1'],
        ['FN1','TIMP2'],
        ['FN1','TMBIM6'],
        ['FN1','TMSB10'],
        ['FN1','TMSB4X'],
        ['FN1','TPM4'],
        ['FN1','TPT1'],
        ['FN1','TUBA1B'],
        ['FN1','TUBB'],
        ['FN1','UBA1'],
        ['FN1','UBB'],
        ['FN1','XBP1'],
        ['FN1','YWHAE'],
        ['FN1','YWHAZ'],
        ['FOCAD','TSHZ2'],
        ['FTH1','GLUL'],
        ['FTH1','GNAS'],
        ['FTH1','GPI'],
        ['FTH1','H3F3A'],
        ['FTH1','HLA-B'],
        ['FTH1','HLA-C'],
        ['FTH1','HNRNPA1'],
        ['FTH1','HSP90AA1'],
        ['FTH1','HSP90AB1'],
        ['FTH1','HSPA1A'],
        ['FTH1','HSPA5'],
        ['FTH1','ILF3'],
        ['FTH1','KRT19'],
        ['FTH1','KRT5'],
        ['FTH1','KRT7'],
        ['FTH1','KRT8'],
        ['FTH1','MUC1'],
        ['FTH1','NPM1'],
        ['FTH1','PKM'],
        ['FTH1','PPIB'],
        ['FTH1','PRSS8'],
        ['FTH1','PSAP'],
        ['FTH1','PTMA'],
        ['FTH1','PTPRF'],
        ['FTH1','RPL18'],
        ['FTH1','RPL27A'],
        ['FTH1','RPL28'],
        ['FTH1','RPL8'],
        ['FTH1','RPLP0'],
        ['FTH1','RPS16'],
        ['FTH1','SERPINA3'],
        ['FTH1','SLC38A1'],
        ['FTH1','SLC39A6'],
        ['FTH1','TM9SF2'],
        ['FTH1','TPI1'],
        ['FTH1','XBP1'],
        ['FTH1','YWHAZ'],
        ['FTL','GAPDH'],
        ['FTL','KRT16'],
        ['FTL','LTF'],
        ['FTL','PTMA'],
        ['FTL','RPL13A'],
        ['FTL','RPS8'],
        ['FUS','MUCL1'],
        ['GABARAP','SF3B2'],
        ['GALNT6','SCGB2A2'],
        ['GANAB','HSP90AB1'],
        ['GANAB','PTPRF'],
        ['GAPDH','GLUL'],
        ['GAPDH','GNAS'],
        ['GAPDH','GNB2L1'],
        ['GAPDH','H3F3B'],
        ['GAPDH','HLA-B'],
        ['GAPDH','HLA-C'],
        ['GAPDH','HNRNPA1'],
        ['GAPDH','HNRNPA2B1'],
        ['GAPDH','HNRNPU'],
        ['GAPDH','KRT19'],
        ['GAPDH','KRT7'],
        ['GAPDH','KRT8'],
        ['GAPDH','LTF'],
        ['GAPDH','MYH9'],
        ['GAPDH','P4HB'],
        ['GAPDH','PPP1R1B'],
        ['GAPDH','PSAP'],
        ['GAPDH','PTMA'],
        ['GAPDH','PTPRF'],
        ['GAPDH','RAB1B'],
        ['GAPDH','RPL10'],
        ['GAPDH','RPL5'],
        ['GAPDH','RPL8'],
        ['GAPDH','RPS11'],
        ['GAPDH','RPS24'],
        ['GAPDH','RPS4X'],
        ['GAPDH','RPS5'],
        ['GAPDH','SCGB2A2'],
        ['GAPDH','SERF2'],
        ['GAPDH','SERPINA3'],
        ['GAPDH','SLC39A6'],
        ['GAPDH','SLC7A5'],
        ['GAPDH','SNRNP200'],
        ['GAPDH','SSR4'],
        ['GAPDH','TMSB10'],
        ['GAPDH','TPI1'],
        ['GAPDH','TPM4'],
        ['GAPDH','TPT1'],
        ['GAPDH','UBB'],
        ['GAPDH','XBP1'],
        ['GATA3','MTRNR2L8'],
        ['GATA3','RPL8'],
        ['GFRA1','MGP'],
        ['GHITM','MTRNR2L8'],
        ['GLB1L3','MRTO4'],
        ['GLIPR1L2','ZNF80'],
        ['GLUL','GOLPH3'],
        ['GLUL','KRT19'],
        ['GLUL','LTF'],
        ['GLUL','MGP'],
        ['GLUL','MUC5B'],
        ['GLUL','P4HB'],
        ['GLUL','PSAP'],
        ['GLUL','PTMA'],
        ['GLUL','RPL15'],
        ['GLUL','RPL29'],
        ['GLUL','SCGB2A2'],
        ['GLUL','SLC39A6'],
        ['GLUL','SPARC'],
        ['GLUL','TPM4'],
        ['GMDS','SUPT3H'],
        ['GNAS','HLA-A'],
        ['GNAS','HLA-C'],
        ['GNAS','HNRNPK'],
        ['GNAS','HSP90AB1'],
        ['GNAS','IGLL5'],
        ['GNAS','KRT19'],
        ['GNAS','KRT8'],
        ['GNAS','MYH9'],
        ['GNAS','PREX1'],
        ['GNAS','PSAP'],
        ['GNAS','PTMA'],
        ['GNAS','SCGB2A2'],
        ['GNAS','SEPT2'],
        ['GNAS','SLC39A6'],
        ['GNAS','TMSB4X'],
        ['GNAS','TPT1'],
        ['GNAS','XBP1'],
        ['GNB1','LTF'],
        ['GNB2','RPL8'],
        ['GNB2L1','SERPINA3'],
        ['GOSR1','HLA-F'],
        ['GOSR1','IPP'],
        ['GOSR1','TFPI'],
        ['GOSR1','WBSCR27'],
        ['GPC6','SLC37A2'],
        ['GPI','IGLL5'],
        ['GTF2IRD1','RABGEF1'],
        ['H3F3A','IFI6'],
        ['H3F3A','MTRNR2L8'],
        ['H3F3B','H3F3C'],
        ['HBA1','RPS9'],
        ['HBA2','RPS9'],
        ['HDLBP','SCGB2A2'],
        ['HDLBP','SPARC'],
        ['HHLA2','IL34'],
        ['HHLA2','KMT2C'],
        ['HHLA3','IL34'],
        ['HIF1A','SHISA9'],
        ['HIPK2','PPDPF'],
        ['HK2','SCD'],
        ['HLA-A','HLA-DRA'],
        ['HLA-A','HLA-F'],
        ['HLA-A','IFITM3'],
        ['HLA-A','IGFBP4'],
        ['HLA-A','IGLL5'],
        ['HLA-A','MTRNR2L8'],
        ['HLA-A','MYH9'],
        ['HLA-A','PFN1'],
        ['HLA-A','PTMA'],
        ['HLA-A','RPS24'],
        ['HLA-A','SCD'],
        ['HLA-A','SCGB1D2'],
        ['HLA-A','SCGB2A2'],
        ['HLA-A','TMSB10'],
        ['HLA-A','TPT1'],
        ['HLA-A','TUBA1B'],
        ['HLA-A','VIM'],
        ['HLA-B','HSP90AA1'],
        ['HLA-B','HSP90AB1'],
        ['HLA-B','HSP90B1'],
        ['HLA-B','HSPA5'],
        ['HLA-B','IGLL5'],
        ['HLA-B','ITGB4'],
        ['HLA-B','KRT8'],
        ['HLA-B','MUC1'],
        ['HLA-B','P4HB'],
        ['HLA-B','PTMA'],
        ['HLA-B','PTMS'],
        ['HLA-B','RPL10'],
        ['HLA-B','RPL41'],
        ['HLA-B','RPLP0'],
        ['HLA-B','RPS3'],
        ['HLA-B','SCGB2A2'],
        ['HLA-B','SLC25A6'],
        ['HLA-B','SLC39A6'],
        ['HLA-B','SPINT1'],
        ['HLA-B','TAGLN2'],
        ['HLA-B','TMBIM6'],
        ['HLA-B','TMSB10'],
        ['HLA-B','TPT1'],
        ['HLA-B','UBB'],
        ['HLA-B','UBC'],
        ['HLA-B','XBP1'],
        ['HLA-B','YWHAZ'],
        ['HLA-C','IGLL5'],
        ['HLA-C','LGALS3BP'],
        ['HLA-C','MUC1'],
        ['HLA-C','MYH9'],
        ['HLA-C','PIP'],
        ['HLA-C','PSAP'],
        ['HLA-C','RHOB'],
        ['HLA-C','RPL3'],
        ['HLA-C','SRRM2'],
        ['HLA-C','TPT1'],
        ['HLA-C','XBP1'],
        ['HLA-E','LYZ'],
        ['HLA-E','MUCL1'],
        ['HLA-F','MICB'],
        ['HMGB3','SRRM2'],
        ['HN1','IFITM3'],
        ['HN1','RPL28'],
        ['HNRNPA1','IGLL5'],
        ['HNRNPA1','KRT6A'],
        ['HNRNPA1','MUCL1'],
        ['HNRNPA1','PABPC1'],
        ['HNRNPA1','RPL13A'],
        ['HNRNPA1','SCGB2A2'],
        ['HNRNPA1','SLC39A6'],
        ['HNRNPA1','TAGLN2'],
        ['HNRNPA1L2','MUCL1'],
        ['HNRNPA2B1','MTRNR2L8'],
        ['HNRNPA2B1','RRBP1'],
        ['HNRNPA2B1','SCGB2A2'],
        ['HNRNPA2B1','XBP1'],
        ['HNRNPA3','SLC39A6'],
        ['HNRNPC','MGP'],
        ['HNRNPC','PTMA'],
        ['HNRNPC','TPT1'],
        ['HNRNPK','LTF'],
        ['HNRNPK','RPL19'],
        ['HNRNPK','SH3BGRL'],
        ['HNRNPU','PPP1R1B'],
        ['HNRNPUL1','RPL8'],
        ['HSP90AA1','HSP90AB1'],
        ['HSP90AA1','PABPC1'],
        ['HSP90AA1','RPL8'],
        ['HSP90AA1','SPARC'],
        ['HSP90AA1','SRRM2'],
        ['HSP90AA1','STAT1'],
        ['HSP90AA1','UBC'],
        ['HSP90AB1','HSP90B1'],
        ['HSP90AB1','HSPA8'],
        ['HSP90AB1','IGLL5'],
        ['HSP90AB1','ITGB1'],
        ['HSP90AB1','LDHA'],
        ['HSP90AB1','MUCL1'],
        ['HSP90AB1','P4HB'],
        ['HSP90AB1','PTMS'],
        ['HSP90AB1','QSOX1'],
        ['HSP90AB1','RPL4'],
        ['HSP90AB1','RPL8'],
        ['HSP90AB1','RPS8'],
        ['HSP90AB1','SCGB2A2'],
        ['HSP90AB1','SERBP1'],
        ['HSP90AB1','SLC39A6'],
        ['HSP90AB1','SLC9A3R1'],
        ['HSP90AB1','SPARC'],
        ['HSP90AB1','SRCAP'],
        ['HSP90AB1','SRRM2'],
        ['HSP90AB1','TPI1'],
        ['HSP90AB1','TUBB'],
        ['HSP90AB1','UBC'],
        ['HSP90AB1','XBP1'],
        ['HSP90AB1','YWHAZ'],
        ['HSP90B1','RPL13A'],
        ['HSP90B1','SCGB1D2'],
        ['HSP90B1','SPARC'],
        ['HSP90B1','SRRM2'],
        ['HSPA1A','HSPA1B'],
        ['HSPA1A','HSPA8'],
        ['HSPA1A','KRT8'],
        ['HSPA1A','MTRNR2L8'],
        ['HSPA1A','MUC5B'],
        ['HSPA1A','RPL41'],
        ['HSPA1A','RPLP0'],
        ['HSPA1A','SCD'],
        ['HSPA1A','SCGB2A2'],
        ['HSPA1A','SERPINA3'],
        ['HSPA1A','SRRM2'],
        ['HSPA1A','TBC1D9'],
        ['HSPA1A','XBP1'],
        ['HSPA1B','KRT8'],
        ['HSPA1B','RPL41'],
        ['HSPA1B','SCD'],
        ['HSPA5','RPS2'],
        ['HSPA8','PSAP'],
        ['HSPA8','SEPT2'],
        ['HSPA8','SLC39A6'],
        ['HSPA8','XBP1'],
        ['HSPB1','KRT8'],
        ['HSPB1','RPL4'],
        ['HSPB1','RPS2'],
        ['HSPD1','MTRNR2L8'],
        ['HUWE1','PGK1'],
        ['HYAL1','MRGPRX3'],
        ['IFI27','RPS28'],
        ['IFI6','IFITM3'],
        ['IFI6','IGFBP5'],
        ['IFI6','PEBP1'],
        ['IFI6','PPDPF'],
        ['IFI6','PTMA'],
        ['IFI6','RAC1'],
        ['IFI6','SPARC'],
        ['IFI6','YWHAB'],
        ['IFITM1','MIF'],
        ['IFITM2','TRIP13'],
        ['IFITM3','PRR13'],
        ['IFITM3','TRIP13'],
        ['IGF2','TARDBP'],
        ['IGFBP4','SCD'],
        ['IGFBP4','TUBA1B'],
        ['IGFBP4','YWHAQ'],
        ['IGFBP5','LTF'],
        ['IGFBP5','MYH9'],
        ['IGFBP5','PABPC1'],
        ['IGFBP5','RPL19'],
        ['IGFBP5','RPL41'],
        ['IGFBP5','SLC39A6'],
        ['IGFBP5','SPARC'],
        ['IGLL5','MCL1'],
        ['IGLL5','PKP1'],
        ['IGLL5','PTMA'],
        ['IGLL5','RPL8'],
        ['IGLL5','RPLP0'],
        ['IGLL5','RPS18'],
        ['IGLL5','RPS2'],
        ['IGLL5','RPS3'],
        ['IGLL5','RPS7'],
        ['IGLL5','TPT1'],
        ['IGLL5','TUBA1C'],
        ['IGSF11','WDR72'],
        ['IL12RB2','SLC2A4'],
        ['IL32','MT2A'],
        ['IL34','PRELP'],
        ['IPMK','LAMTOR3'],
        ['IPMK','ZNF80'],
        ['IPO7','TMEM41B'],
        ['IPP','MAST2'],
        ['ITGB1','RPS24'],
        ['JCHAIN','TMBIM6'],
        ['JUP','SPARC'],
        ['KDELR2','P4HB'],
        ['KIAA1522','PSAP'],
        ['KLF15','NDUFA4L2'],
        ['KMT5B','PPP6R3'],
        ['KRT14','KRT5'],
        ['KRT14','KRT6A'],
        ['KRT14','RPL13'],
        ['KRT14','RPS2'],
        ['KRT14','S100A14'],
        ['KRT14','S100A2'],
        ['KRT14','SLC25A6'],
        ['KRT17','KRT5'],
        ['KRT18','P4HB'],
        ['KRT18','PRDX1'],
        ['KRT18','RPL14'],
        ['KRT18','RPL19'],
        ['KRT18','RPLP0'],
        ['KRT18','RPS2'],
        ['KRT18','SCGB2A2'],
        ['KRT19','KRT8'],
        ['KRT19','MMP11'],
        ['KRT19','MYH9'],
        ['KRT19','NUCB1'],
        ['KRT19','P4HB'],
        ['KRT19','PCBP2'],
        ['KRT19','PTMA'],
        ['KRT19','RPL13A'],
        ['KRT19','RPL19'],
        ['KRT19','RPS15A'],
        ['KRT19','RPS2'],
        ['KRT19','RPS6'],
        ['KRT19','RPS8'],
        ['KRT19','SCGB2A2'],
        ['KRT19','SERPINA3'],
        ['KRT19','SLC25A6'],
        ['KRT19','SLC39A6'],
        ['KRT19','SPARC'],
        ['KRT19','TPM4'],
        ['KRT19','TPT1'],
        ['KRT19','UBC'],
        ['KRT5','P4HB'],
        ['KRT5','RPL18'],
        ['KRT5','S100A14'],
        ['KRT7','KRT8'],
        ['KRT7','PTMS'],
        ['KRT7','RPLP0'],
        ['KRT7','SFN'],
        ['KRT8','LMNA'],
        ['KRT8','MAGED2'],
        ['KRT8','MTRNR2L2'],
        ['KRT8','MTRNR2L8'],
        ['KRT8','MUC5B'],
        ['KRT8','MYH9'],
        ['KRT8','RPL13A'],
        ['KRT8','RPL29'],
        ['KRT8','RPL39'],
        ['KRT8','RPL41'],
        ['KRT8','RPS19'],
        ['KRT8','RPS4X'],
        ['KRT8','SCGB2A2'],
        ['KRT8','SERPINA3'],
        ['KRT8','SLC9A3R1'],
        ['KRT8','SPARC'],
        ['KRT8','SQSTM1'],
        ['KRT8','TUBA1B'],
        ['KRT8','UBC'],
        ['KRT8','XBP1'],
        ['KRT8','YWHAZ'],
        ['LASP1','RPLP0'],
        ['LASP1','RPS3'],
        ['LDHA','PDIA6'],
        ['LGALS1','RPL12'],
        ['LGALS3BP','P4HB'],
        ['LGALS3BP','SLC39A6'],
        ['LOC403312','NUP43'],
        ['LOC403312','RBM8A'],
        ['LOC403312','ZFP30'],
        ['LRRC59','SCD'],
        ['LTF','MTRNR2L8'],
        ['LTF','MUC1'],
        ['LTF','MYH9'],
        ['LTF','P4HB'],
        ['LTF','PTMS'],
        ['LTF','RPL24'],
        ['LTF','RPS3'],
        ['LTF','RPS3A'],
        ['LTF','SPARC'],
        ['LTF','TMSB10'],
        ['LUC7L3','TPM4'],
        ['LYZ','PSAP'],
        ['LYZ','SPARC'],
        ['MAB21L1','RPL14'],
        ['MAB21L3','ODF2L'],
        ['MAB21L3','OPHN1'],
        ['MAB21L3','SCD5'],
        ['MAB21L3','TMEM212'],
        ['MAB21L3','XRCC2'],
        ['MAGED2','RPL13A'],
        ['MAGED2','RPS2'],
        ['MAGED2','SPARC'],
        ['MAPK10','PTPN13'],
        ['MAST2','PARD6B'],
        ['MCL1','MTRNR2L2'],
        ['MED15','RPL14'],
        ['METTL21A','MRPS18A'],
        ['METTL2A','RARS2'],
        ['METTL2B','RARS2'],
        ['MGP','PTMS'],
        ['MGP','RPL21'],
        ['MGP','RPN1'],
        ['MGP','SERPINA3'],
        ['MGP','SLC39A6'],
        ['MGP','TPM4'],
        ['MGP','TUBA1B'],
        ['MGP','YWHAZ'],
        ['MIDN','SPARC'],
        ['MIF','RPS28'],
        ['MMP14','PTMA'],
        ['MMP14','SCGB2A2'],
        ['MSL2','PCCB'],
        ['MT2A','POLR2G'],
        ['MT2A','RPL13A'],
        ['MTRNR2L2','RPL10'],
        ['MTRNR2L2','S100A11'],
        ['MTRNR2L2','SCGB2A2'],
        ['MTRNR2L2','SLC39A6'],
        ['MTRNR2L2','SPARC'],
        ['MTRNR2L2','TIMP3'],
        ['MTRNR2L2','TPT1'],
        ['MTRNR2L6','RPL9'],
        ['MTRNR2L8','NPM1'],
        ['MTRNR2L8','OS9'],
        ['MTRNR2L8','PABPC1'],
        ['MTRNR2L8','PPIA'],
        ['MTRNR2L8','PRDX1'],
        ['MTRNR2L8','RPL10'],
        ['MTRNR2L8','SCGB2A2'],
        ['MTRNR2L8','SH3BGRL'],
        ['MTRNR2L8','SHFM1'],
        ['MTRNR2L8','SLC39A6'],
        ['MTRNR2L8','TMSB10'],
        ['MTRNR2L8','TPM4'],
        ['MTRNR2L8','UBB'],
        ['MTRNR2L8','XBP1'],
        ['MTRNR2L9','RPL11'],
        ['MUC1','MYH9'],
        ['MUC1','RPL3'],
        ['MUC1','RPLP0'],
        ['MUC1','RPS2'],
        ['MUC1','RPS8'],
        ['MUC1','SCD'],
        ['MUC1','SCGB2A2'],
        ['MUC1','SLC39A6'],
        ['MUC3A','RPL13A'],
        ['MUC5B','PKM'],
        ['MUCL1','MYH9'],
        ['MUCL1','NONO'],
        ['MUCL1','PABPC1'],
        ['MUCL1','RALY'],
        ['MUCL1','RPL13A'],
        ['MUCL1','RPL14'],
        ['MUCL1','RPL23'],
        ['MUCL1','RPL4'],
        ['MUCL1','RPL41'],
        ['MUCL1','RPL6'],
        ['MUCL1','RPS6'],
        ['MUCL1','SPARC'],
        ['MUCL1','TMBIM6'],
        ['MUCL1','TMSB10'],
        ['MUCL1','TPT1'],
        ['MUCL1','UQCRB'],
        ['MXRA5','SLC40A1'],
        ['MYH9','P4HB'],
        ['MYH9','PABPC1'],
        ['MYH9','PKM'],
        ['MYH9','PPP1R1B'],
        ['MYH9','PREX1'],
        ['MYH9','PTMS'],
        ['MYH9','RPL8'],
        ['MYH9','RPS6'],
        ['MYH9','S100A11'],
        ['MYH9','SCD'],
        ['MYH9','SCGB2A2'],
        ['MYH9','SRRM2'],
        ['MYL6','PTMA'],
        ['MYL6','SOD1'],
        ['MYL6','XBP1'],
        ['MYO5C','TGOLN2'],
        ['MYOM3','NIPAL3'],
        ['NAA25','TRAFD1'],
        ['NABP2','PPEF2'],
        ['NAP1L3','RPL14'],
        ['NCL','SCD'],
        ['NCL','SLC9A3R1'],
        ['NCL','TPT1'],
        ['NCOA3','RPL14'],
        ['NDUFA13','UBA52'],
        ['NEK4','ZNF257'],
        ['NME1-NME2','RPS10'],
        ['NME1-NME2','RPS10-NUDT3'],
        ['NME1-NME2','SLC39A6'],
        ['NME2','RPL13A'],
        ['NMNAT1','SALL4'],
        ['NMT2','ZSCAN21'],
        ['NPM1','SLC39A6'],
        ['NPM1','SPARC'],
        ['NUS1','SLC25A13'],
        ['OCM','RNF216'],
        ['OPHN1','PTEN'],
        ['OPHN1','TRDN'],
        ['ORC4','SCD5'],
        ['ORC4','TMEM182'],
        ['OS9','RPL13A'],
        ['P4HB','PTMA'],
        ['P4HB','RAC1'],
        ['P4HB','RPL13A'],
        ['P4HB','RPLP0'],
        ['P4HB','RPS18'],
        ['P4HB','SCGB2A2'],
        ['P4HB','SLC39A6'],
        ['P4HB','TMSB10'],
        ['P4HB','UBC'],
        ['PABPC1','PTMA'],
        ['PABPC1','RPL13A'],
        ['PABPC1','RPL15'],
        ['PABPC1','RPL23A'],
        ['PABPC1','RPL8'],
        ['PABPC1','SERPINA3'],
        ['PABPC1','SPARC'],
        ['PABPC1','TUBB'],
        ['PDIA3','SCGB2A2'],
        ['PGK1','VDAC1'],
        ['PIP','PTMA'],
        ['PIP','RPL7'],
        ['PIP','RPS25'],
        ['PIP','SPARC'],
        ['PKM','RPL3'],
        ['PKM','RPS18'],
        ['PKM','RPS8'],
        ['PLEC','SCD'],
        ['PLEC','SPARC'],
        ['POLR2A','RPL41'],
        ['POSTN','SPARC'],
        ['PPDPF','RPL28'],
        ['PPDPF','RPL8'],
        ['PPDPF','RPS2'],
        ['PPDPF','YWHAG'],
        ['PPIA','RPL13A'],
        ['PPIA','RPL19'],
        ['PPIA','RPL4'],
        ['PPIA','RPS6'],
        ['PPIA','SCGB2A2'],
        ['PPP1R1B','RPL41'],
        ['PRDX5','RPS11'],
        ['PRELID3B','RAD18'],
        ['PRKX','TNR'],
        ['PRR13','UQCRH'],
        ['PRRC2C','YWHAZ'],
        ['PSAP','SCGB2A2'],
        ['PSAP','SRRM2'],
        ['PSAP','TMSB10'],
        ['PSAP','UBB'],
        ['PSAP','XBP1'],
        ['PSMF1','RAB3B'],
        ['PSMF1','SEMA3E'],
        ['PTCHD4','ZNF880'],
        ['PTMA','PTMS'],
        ['PTMA','RPL13A'],
        ['PTMA','RPL19'],
        ['PTMA','RPL41'],
        ['PTMA','RPLP0'],
        ['PTMA','SCD'],
        ['PTMA','SLC39A6'],
        ['PTMA','SPARC'],
        ['PTMA','TAGLN2'],
        ['PTMA','TUBA1B'],
        ['PTMA','TUBB'],
        ['PTMA','YWHAZ'],
        ['PTMS','RPL3'],
        ['PTMS','RPL8'],
        ['PTMS','RPS3'],
        ['PTMS','SPARC'],
        ['PTMS','SRRM2'],
        ['PTMS','TUBA1B'],
        ['PTP4A2','TPT1'],
        ['PTP4A2','XBP1'],
        ['PTPRB','XRCC6BP1'],
        ['PTPRF','RPL7'],
        ['RAB7A','SLC39A6'],
        ['RHOA','SLC39A6'],
        ['RHOB','SLC39A6'],
        ['RNF213','VHL'],
        ['RPL10','RPL41'],
        ['RPL10','SPARC'],
        ['RPL10','TMSB10'],
        ['RPL12','SCGB2A2'],
        ['RPL12','SERF2'],
        ['RPL13A','RPS2'],
        ['RPL13A','SCGB2A2'],
        ['RPL13A','SLC39A6'],
        ['RPL13A','SLC44A2'],
        ['RPL13A','WFDC2'],
        ['RPL14','SCGB2A2'],
        ['RPL14','SREBF2'],
        ['RPL14','TBP'],
        ['RPL14','VEZF1'],
        ['RPL14','ZNF384'],
        ['RPL15','SPARC'],
        ['RPL17','RPS18'],
        ['RPL18','RPS2'],
        ['RPL18A','RPS15'],
        ['RPL18A','RPS28'],
        ['RPL18A','WFDC2'],
        ['RPL19','RPS18'],
        ['RPL19','SCD'],
        ['RPL19','SCGB2A2'],
        ['RPL19','SERF2'],
        ['RPL19','SQSTM1'],
        ['RPL19','SYNGR2'],
        ['RPL19','TUBA1B'],
        ['RPL22','SCD'],
        ['RPL23A','SERPINA3'],
        ['RPL23A','SPARC'],
        ['RPL23A','TBC1D1'],
        ['RPL27A','SCGB2A2'],
        ['RPL27A','SERPINA3'],
        ['RPL27A','TUBB'],
        ['RPL28','TFF2'],
        ['RPL29','RPS3'],
        ['RPL3','SCGB2A2'],
        ['RPL3','SLC44A2'],
        ['RPL3','XBP1'],
        ['RPL32','RPL37A'],
        ['RPL32','SERPINA3'],
        ['RPL35A','RPL8'],
        ['RPL35A','RPLP2'],
        ['RPL37','RPS3'],
        ['RPL37','TMBIM6'],
        ['RPL37A','TIMP1'],
        ['RPL37A','YBX1'],
        ['RPL38','SNRPD2'],
        ['RPL39','TMSB10'],
        ['RPL4','RPL8'],
        ['RPL4','RPSA'],
        ['RPL4','SCGB2A2'],
        ['RPL41','SCGB2A2'],
        ['RPL41','SERPINA3'],
        ['RPL41','SLC39A6'],
        ['RPL5','RPS2'],
        ['RPL5','SCGB2A2'],
        ['RPL5','SPARC'],
        ['RPL6','SERF2'],
        ['RPL7','TIMP3'],
        ['RPL7','TMSB10'],
        ['RPL7A','RPS19'],
        ['RPL7A','RPS2'],
        ['RPL7A','SCGB2A2'],
        ['RPL7A','SPARC'],
        ['RPL8','RPS2'],
        ['RPL8','RPS20'],
        ['RPL8','SCD'],
        ['RPL8','SLC39A6'],
        ['RPL8','SYPL1'],
        ['RPL8','TMSB10'],
        ['RPL8','TUBA1B'],
        ['RPL8','TUBB'],
        ['RPLP0','RPS13'],
        ['RPLP0','RPS8'],
        ['RPLP0','SCGB2A2'],
        ['RPLP0','SLC39A6'],
        ['RPLP0','TPT1'],
        ['RPLP0','TUBB'],
        ['RPLP1','SCGB2A2'],
        ['RPLP1','SERPINA3'],
        ['RPLP2','SCGB2A2'],
        ['RPS10-NUDT3','RPS28'],
        ['RPS14','SPARC'],
        ['RPS15','TUBA1B'],
        ['RPS15A','SLC39A6'],
        ['RPS17','SCGB2A2'],
        ['RPS18','SERPINA3'],
        ['RPS2','RPS24'],
        ['RPS2','RPS4X'],
        ['RPS2','SCGB2A2'],
        ['RPS24','SLC39A6'],
        ['RPS28','RPS3'],
        ['RPS28','UBA52'],
        ['RPS3','SCD'],
        ['RPS3','SCGB2A2'],
        ['RPS3','YBX1'],
        ['RPS4X','SLC39A6'],
        ['RPS8','SCGB2A2'],
        ['RPS8','SLC39A6'],
        ['RPS8','TMSB10'],
        ['RPS9','TMSB10'],
        ['S100A10','TMSB10'],
        ['S100A2','TMSB10'],
        ['S100A6','SRP14'],
        ['S100A6','UBA52'],
        ['S100A6','YWHAE'],
        ['SCAMP3','THBS3'],
        ['SCD','XBP1'],
        ['SCD','YWHAZ'],
        ['SCGB1D2','SCGB2A2'],
        ['SCGB2A2','SLC25A3'],
        ['SCGB2A2','SLC39A6'],
        ['SCGB2A2','SPARC'],
        ['SCGB2A2','SRRM2'],
        ['SCGB2A2','TFF1'],
        ['SCGB2A2','TFF3'],
        ['SCGB2A2','TMSB10'],
        ['SCGB2A2','TPM3'],
        ['SCGB2A2','TPT1'],
        ['SCGB2A2','UBB'],
        ['SCGB2A2','UBC'],
        ['SCGB2A2','XBP1'],
        ['SEMA4D','ZNF365'],
        ['SEMA4D','ZNF589'],
        ['SEPP1','SLC39A6'],
        ['SERPINA3','SLC39A6'],
        ['SERPINA3','TPT1'],
        ['SERPINA3','UBC'],
        ['SFN','ZNF493'],
        ['SFRP2','SPARC'],
        ['SHISA9','TENM1'],
        ['SLC38A1','ULK4'],
        ['SLC39A6','UBB'],
        ['SLC39A6','UBC'],
        ['SLC39A6','XBP1'],
        ['SPARC','TAGLN'],
        ['SPARC','TIMP2'],
        ['SPARC','TUBA1A'],
        ['SPARC','TUBB'],
        ['SPINT2','YWHAZ'],
        ['SPOCK2','TTBK1'],
        ['TBL1X','USP12'],
        ['TDG','TMEM132B'],
        ['TIMP1','UBD'],
        ['TMEM56','ZNF365'],
        ['TMSB10','TUBA1B'],
        ['TMSB10','UBC'],
        ['TNFAIP2','YWHAG'],
        ['TNK2','TPM3'],
        ['TRPS1','XBP1'],
        ['UBC','XBP1'],
        ['UGT8','XRCC2'],
        ['XBP1','YWHAZ'],
        ['RAB3IP','MYLK'],
        ['RAB3IP','ACVR1C'],
        ['MAPKAPK5','ACAD10'],
        ['CCDC6','RET'],
        ['CHRM3','MYLK'],
        ['FGFR3','TACC3'],
        ['ETV6','NTRK3'],
        ['REV3L','FYN'],
        ['KIAA1324L','TBCK'],
        ['TMEM87B','MERTK'],
        ['WDR72','TBCK'],
        ['EGFR','SEPT-14'],
        ['EML4','ALK'],
        ['BOLA2B','SMG1'],
        ['BOLA2','SMG1'],
        ['CAMK1D','TMPO'],
        ['MARK2','C11ORF84'],
        ['PEAK1','LINGO1'],
        ['RAF1','TMEM40'],
        ['STK24','SLC15A1'],
        ['SLC7A5','SMG1'],
        ['BCR','ABL1'],
        ['CSNK1D','SECTM1'],
        ['ERBB2','PPP1R1B'],
        ['ERC1','WNK1'],
        ['HIF1A','PRKCH'],
        ['PTK2','SLC45A4'],
        ['SRPK2','PUS7'],
        ['BRAF','MRPS33'],
        ['CSNK1D','OGFOD3'],
        ['ERC1','RET'],
        ['FGFR2','BICC1'],
        ['METTL24','SLC22A16'],
        ['PRKAR1A','TBCK'],
        ['STK24','DOCK9'],
        ['TSPAN5','METAP1'],
        ['ZBTB7A','MAP2K2'],
        ['SMG1','NPIPB3'],
        ['ABL1','FUBP3'],
        ['AKT2','YIF1B'],
        ['BUB1B','EIF2AK4'],
        ['CAMK1D','CDC123'],
        ['CAMK2D','ANK2'],
        ['CDKL2','G3BP2'],
        ['CHRM3','RPS6KA3'],
        ['CHRM3','TBCK'],
        ['CSNK1A1','PDE6A'],
        ['EEF2K','VWA3A'],
        ['EIF3L','TRIOBP'],
        ['EML4','PKDCC'],
        ['ERBB2','IKZF3'],
        ['ERN1','TEX2'],
        ['EXT2','ALKBH3'],
        ['FGFR2','ATE1'],
        ['FGFR2','SHTN1'],
        ['FYN','CDK19'],
        ['HERC1','DAPK2'],
        ['IKBKB','ANK1'],
        ['IPO7','NUAK1'],
        ['MAP2K5','IQCH'],
        ['MAP7','MAP3K5'],
        ['MAST2','NASP'],
        ['MED13L','TAOK3'],
        ['MED1','CDK12'],
        ['MET','CAV1'],
        ['OSGIN2','RIPK2'],
        ['OXSR1','CTDSPL'],
        ['PAN3','FLT3'],
        ['PRKCA','PITPNC1'],
        ['PTK2','MROH5'],
        ['PXK','RPP14'],
        ['RIPK2','NAAA'],
        ['SCRIB','NRBP2'],
        ['SCYL1','FRMD8'],
        ['SCYL3','KIFAP3'],
        ['SKI','PRKCZ'],
        ['SMG1','DCTPP1'],
        ['SOS1','CDKL4'],
        ['SOS1','MAP4K3'],
        ['ST7','MET'],
        ['STRN','ALK'],
        ['SWAP70','WEE1'],
        ['TMEM192','MAST2'],
        ['TRAF3IP2','FYN'],
        ['TRAPPC9','PTK2'],
        ['TSPAN31','OS9'],
        ['TTC7B','RPS6KA5'],
        ['WNK1','B4GALNT3'],
        ['ACTB','ERBB2'],
        ['CASC3','CDK12'],
        ['COL1A1','IGF1R'],
        ['DMPK','DMWD'],
        ['EEF1A1','PRKDC'],
        ['ERBB2','HLA-DRB1'],
        ['ERBB2','RPLP1'],
        ['ERBB2','RPS4X'],
        ['ERBB2','TMSB10'],
        ['GAPDH','FGFR1'],
        ['HIPK2','PARP12'],
        ['IRF2BP2','NTRK1'],
        ['MYH9','ERBB2'],
        ['NCOA4','RET'],
        ['PFN1','ERBB2'],
        ['SMG1','NPIPB5'],
        ['SND1','BRAF'],
        ['STK3','CPQ'],
        ['TPM3','NTRK1'],
        ['TRIO','TERT'],
        ['ABCC9','NEK5'],
        ['ACTB','DDR1'],
        ['ACTB','ERBB2'],
        ['ACTB','ERBB3'],
        ['ACTG1','ERBB2'],
        ['AKT1','COL1A1'],
        ['AKT1','CPB1'],
        ['AKT1','EEF1A1'],
        ['APOE','ERBB2'],
        ['B2M','ERBB2'],
        ['CAMK1D','TMPO'],
        ['CD24','ERBB2'],
        ['CD74','ERBB2'],
        ['CDC42BPB','TNFAIP2'],
        ['CDK12','PROX1'],
        ['CDK14','CDYL'],
        ['CDK14','CHRM3'],
        ['CHRM3','PRKCA'],
        ['COL1A1','DDR1'],
        ['COL1A1','ERBB2'],
        ['COL1A1','MAP3K1'],
        ['COL1A1','PRKDC'],
        ['COL1A2','DDR1'],
        ['COL1A2','ERBB2'],
        ['COL1A2','ERBB3'],
        ['COL3A1','ERBB2'],
        ['COL6A1','ERBB2'],
        ['CTSD','ERBB2'],
        ['DENND3','PTK2'],
        ['EEF1A1','LIMK2'],
        ['EEF1G','ERBB2'],
        ['EPHB4','IGLL5'],
        ['ERBB2','FASN'],
        ['ERBB2','FN1'],
        ['ERBB2','FTH1'],
        ['ERBB2','FTL'],
        ['ERBB2','GNAS'],
        ['ERBB2','IGLL5'],
        ['ERBB2','KRT19'],
        ['ERBB2','MTRNR2L8'],
        ['ERBB2','PPIA'],
        ['ERBB2','PPP1R1B'],
        ['ERBB2','RPL13A'],
        ['ERBB2','RPL18A'],
        ['ERBB2','RPL19'],
        ['ERBB2','RPL3'],
        ['ERBB2','RPS18'],
        ['ERBB2','SQSTM1'],
        ['ERBB2','UBC'],
        ['ERBB3','IFI6'],
        ['HIPK2','PPDPF'],
        ['IPP','MAST2'],
        ['MAPK10','PTPN13'],
        ['MAST2','PARD6B'],
        ['NEK4','ZNF257'],
        ['PRKX','TNR'],
        ['SLC38A1','ULK4'],
        ['SPOCK2','TTBK1'],
        ['TNK2','TPM3'],
        ['ADGRL1','SUZ12'],
        ['PRKAR1A','ALG1L'],
        ['PRKAR1A','CBWD6'],
        ['FGFR3','TACC3'],
        ['MKL1','SLC25A17'],
        ['NCOR2','SCARB1'],
        ['PRKAR1A','CBWD3'],
        ['CCDC6','ANK3'],
        ['CCND3','TRERF1'],
        ['DNM2','SLC44A2'],
        ['ETV6','NTRK3'],
        ['EXT1','SAMD12'],
        ['MECOM','RPL22'],
        ['NOTCH2','SEC22B'],
        ['PRKAR1A','SPATA6L'],
        ['RAF1','TMEM40'],
        ['TCF7L2','VTI1A'],
        ['CCDC6','RET'],
        ['EGFR','SEPT14'],
        ['EML4','ALK'],
        ['ERBB2','PPP1R1B'],
        ['HIF1A','PRKCH'],
        ['KAT6B','ADK'],
        ['KDM5A','NINJ2'],
        ['NF1','RNF135'],
        ['NOVA1','FUS'],
        ['PRKAR1A','ADGRV1'],
        ['PRKAR1A','CBWD5'],
        ['PRKAR1A','COMMD10'],
        ['PRKAR1A','MCCC2'],
        ['PTPRK','RSPO3'],
        ['QKI','PACRG'],
        ['RCAN1','RUNX1'],
        ['SMARCA4','CARM1'],
        ['VAV2','BRD3'],
        ['ABL1','FUBP3'],
        ['AFF1','PTPN13'],
        ['AGR2','IL6ST'],
        ['AKT2','YIF1B'],
        ['ANKRD11','FANCA'],
        ['ASXL1','TM9SF4'],
        ['BRAF','MRPS33'],
        ['BUB1B','EIF2AK4'],
        ['CCAR1','TET1'],
        ['COL1A1','CD74'],
        ['COL1A1','GNAS'],
        ['DNM2','ILF3'],
        ['ELL','CRLF1'],
        ['EP300','RBX1'],
        ['ERBB2','IKZF3'],
        ['ERC1','RET'],
        ['ESR1','CCDC170'],
        ['ETV6','PRH1'],
        ['EXT2','ALKBH3'],
        ['FBXW7','TMEM154'],
        ['FGFR2','ATE1'],
        ['FGFR2','BICC1'],
        ['FGFR2','SHTN1'],
        ['GOLGA4','MLH1'],
        ['IKBKB','ANK1'],
        ['KMT2C','LOC400927-CSNK1E'],
        ['LDLR','SMARCA4'],
        ['LSM1','WHSC1L1'],
        ['MDM2','NUP107'],
        ['MECOM','LRRC34'],
        ['MED1','CDK12'],
        ['MET','CAV1'],
        ['MKL1','SPIDR'],
        ['MSH2','EPCAM'],
        ['NOTCH3','BRD4'],
        ['PAN3','FLT3'],
        ['PIEZO1','CBFA2T3'],
        ['POLR2J2','PMS2'],
        ['POLR2J3','PMS2'],
        ['POM121','PMS2'],
        ['PRKAR1A','BCAS3'],
        ['PRKAR1A','TBCK'],
        ['PTEN','RNLS'],
        ['PTPRB','CHRM3'],
        ['PTPRG','FHIT'],
        ['RAD51B','GPHN'],
        ['RB1','MED4'],
        ['RBM28','KMT2C'],
        ['RIC1','PDCD1LG2'],
        ['SAMD8','KAT6B'],
        ['SDHA','CTNND2'],
        ['SFPQ','TFE3'],
        ['SLC12A7','TERT'],
        ['SMAD3','SMAD6'],
        ['SMARCA4','DNM2'],
        ['SNX22','RPL10'],
        ['SPEN','FBLIM1'],
        ['SPEN','PLEKHM2'],
        ['ST7','MET'],
        ['STRN','ALK'],
        ['SUZ12','LRRC37B'],
        ['WHSC1L1','DDHD2'],
        ['ZBTB7A','MAP2K2'],
        ['ACTB','CD74'],
        ['ACTB','ERBB2'],
        ['AZIN1','IL6ST'],
        ['B2M','ENSA'],
        ['B2M','HSPA1A'],
        ['B2M','TACSTD2'],
        ['B2M','UBC'],
        ['BRIP1','CPT1A'],
        ['CALR','RPL41'],
        ['CASC3','CDK12'],
        ['CBFB','MYH11'],
        ['CD74','CD24'],
        ['CD74','COX6C'],
        ['CD74','FN1'],
        ['CD74','GAPDH'],
        ['CD74','HLA-B'],
        ['CD74','PSAP'],
        ['CD74','RPL8'],
        ['CD74','SCGB2A2'],
        ['CD74','SEC61A1'],
        ['CD74','SPARC'],
        ['COL1A1','CALR'],
        ['COL1A1','CCND1'],
        ['COL1A1','FOXA1'],
        ['COL1A1','RPL10'],
        ['COL1A1','TPM3'],
        ['COL1A2','RPL10'],
        ['COL3A1','RPL10'],
        ['COL3A1','YWHAE'],
        ['CRTC1','KLHL26'],
        ['EEF2','IL6ST'],
        ['ERBB2','HLA-DRB1'],
        ['ERBB2','RPLP1'],
        ['ERBB2','RPS4X'],
        ['ERBB2','TMSB10'],
        ['ESR1','TMEM212'],
        ['FAM234B','CCND1'],
        ['FLNA','CD74'],
        ['FN1','CD74'],
        ['FN1','MYH9'],
        ['GAPDH','CDH1'],
        ['GAPDH','FGFR1'],
        ['GAPDH','MYH9'],
        ['GNAQ','VEPH1'],
        ['GNAS','ACTB'],
        ['GNAS','COL1A1'],
        ['GNAS','COL3A1'],
        ['GNAS','HLA-B'],
        ['GNAS','PTMA'],
        ['GNAS','RPLP0'],
        ['GNAS','SCD'],
        ['GNAS','SPARC'],
        ['GNAS','TPT1'],
        ['GNAS','TUBB'],
        ['H3F3A','SLC39A6'],
        ['HLA-B','CD74'],
        ['HNRNPA2B1','ACTG1'],
        ['HSP90AB1','GNAS'],
        ['HSPB1','CCND1'],
        ['IRF2BP2','NTRK1'],
        ['KMT2A','ELL'],
        ['KMT2C','CTBP2'],
        ['KMT2D','EEF1A1'],
        ['LUC7L2','ETV1'],
        ['MGP','HNRNPA2B1'],
        ['MKRN2','PPARG'],
        ['MTMR2','MAML2'],
        ['MYH9','ALDOA'],
        ['MYH9','ERBB2'],
        ['MYH9','PABPC1'],
        ['MYH9','PPIA'],
        ['NCALD','UBR5'],
        ['NCOA4','RET'],
        ['NDRG1','ACTB'],
        ['NF1','ATAD5'],
        ['NPM1','COL1A2'],
        ['NPM1','PTMA'],
        ['NSD1','NUP98'],
        ['PFN1','ERBB2'],
        ['PML','RARA'],
        ['PPARG','SYN2'],
        ['PRKAR1A','GBP1'],
        ['PTEN','CTNNA3'],
        ['PTPRK','ECHDC1'],
        ['RPL10','COL1A1'],
        ['RPL10','COL3A1'],
        ['RPL10','CPB1'],
        ['RPL10','FN1'],
        ['RPL10','RPL3'],
        ['RPL18A','B2M'],
        ['RPL18A','CD74'],
        ['RPL22','ACTB'],
        ['RPS17','MYH9'],
        ['RPS2','B2M'],
        ['RPS2','GNAS'],
        ['RPS3','MYH9'],
        ['RPS6','MYH9'],
        ['RUNX1','RUNX1T1'],
        ['SERPINA3','B2M'],
        ['SFPQ','TFEB'],
        ['SLC45A3','ERG'],
        ['SLC45A3','ETV1'],
        ['SND1','BRAF'],
        ['SS18','SSX1'],
        ['SS18','SSX2'],
        ['TFE3','PRCC'],
        ['TMPRSS2','ETV4'],
        ['TP63','TPRG1'],
        ['TPM3','NTRK1'],
        ['TRIO','TERT'],
        ['WBP1L','SUFU'],
        ['XBP1','CALR'],
        ['YWHAE','ACTG1'],
        ['ACTB','ATP1A1'],
        ['ACTB','CCND1'],
        ['ACTB','ERBB2'],
        ['ACTB','ERBB3'],
        ['ACTB','ESR1'],
        ['ACTB','GNAS'],
        ['ACTB','H3F3B'],
        ['ACTB','HNRNPA2B1'],
        ['ACTB','KLF6'],
        ['ACTB','NPM1'],
        ['ACTB','RAD21'],
        ['ACTB','RPL10'],
        ['ACTB','RPL5'],
        ['ACTG1','ATP1A1'],
        ['ACTG1','CDH1'],
        ['ACTG1','ERBB2'],
        ['ACTG1','GATA3'],
        ['ACTG1','MYH9'],
        ['ACTG1','RPL10'],
        ['ACTR3C','KMT2C'],
        ['AEBP1','CD74'],
        ['AKT1','COL1A1'],
        ['AKT1','CPB1'],
        ['AKT1','EEF1A1'],
        ['ALDOA','CALR'],
        ['ALDOA','GNAS'],
        ['APLP2','MYH9'],
        ['APOE','ERBB2'],
        ['APP','CALR'],
        ['APP','MYH9'],
        ['AR','RPL14'],
        ['ATP1A1','COL1A1'],
        ['ATP1A1','COL1A2'],
        ['ATP1A1','COL3A1'],
        ['ATP1A1','EEF2'],
        ['ATP1A1','KRT14'],
        ['ATP1A1','MTRNR2L8'],
        ['ATP1A1','PPP1R1B'],
        ['ATP1A1','UBC'],
        ['AXIN2','RGS9'],
        ['AZGP1','GNAS'],
        ['B2M','CANX'],
        ['B2M','CLU'],
        ['B2M','COL1A2'],
        ['B2M','CPB1'],
        ['B2M','CTSD'],
        ['B2M','ERBB2'],
        ['B2M','FN1'],
        ['B2M','HDLBP'],
        ['B2M','HLA-A'],
        ['B2M','HLA-DPA1'],
        ['B2M','HNRNPA1'],
        ['B2M','HNRNPC'],
        ['B2M','HSP90AA1'],
        ['B2M','HSP90AB1'],
        ['B2M','HSP90B1'],
        ['B2M','KRT17'],
        ['B2M','KRT19'],
        ['B2M','LGALS3BP'],
        ['B2M','LYZ'],
        ['B2M','PKM'],
        ['B2M','PSAP'],
        ['B2M','RPL19'],
        ['B2M','RPL3'],
        ['B2M','RPL7'],
        ['B2M','RPLP0'],
        ['B2M','RPS2'],
        ['B2M','SCGB2A2'],
        ['B2M','SLC39A6'],
        ['B2M','TBC1D9'],
        ['B2M','TMEM123'],
        ['B2M','XBP1'],
        ['B2M','YBX1'],
        ['BCAM','GNAS'],
        ['BGN','GATA3'],
        ['BGN','GNAS'],
        ['BGN','MYH9'],
        ['BST2','CALR'],
        ['C1QB','SRSF3'],
        ['CALR','CD74'],
        ['CALR','COL1A1'],
        ['CALR','COL1A2'],
        ['CALR','CPB1'],
        ['CALR','EEF2'],
        ['CALR','FN1'],
        ['CALR','FTH1'],
        ['CALR','GAPDH'],
        ['CALR','GNAS'],
        ['CALR','LTF'],
        ['CALR','MTRNR2L8'],
        ['CALR','MYH9'],
        ['CALR','PABPC1'],
        ['CALR','PTMA'],
        ['CALR','RPS18'],
        ['CALR','SCD'],
        ['CALR','SLC9A3R1'],
        ['CALR','SPARC'],
        ['CALR','SRRM2'],
        ['CALR','TMSB10'],
        ['CCND1','COL3A1'],
        ['CCND1','DDX17'],
        ['CCND1','EEF1G'],
        ['CCND1','EEF2'],
        ['CCND1','FN1'],
        ['CCND1','HSPA5'],
        ['CCND1','HSPA8'],
        ['CCND1','IGFBP4'],
        ['CCND1','KRT18'],
        ['CCND1','LSM14A'],
        ['CCND1','RAB6A'],
        ['CCND1','SCD'],
        ['CCND1','SCGB2A2'],
        ['CCND1','SLC1A5'],
        ['CCND1','SLC39A6'],
        ['CCND1','SLC9A3R1'],
        ['CCND1','SPARC'],
        ['CCND1','TMBIM6'],
        ['CCND1','TPT1'],
        ['CCND1','UBC'],
        ['CCND1','YWHAZ'],
        ['CD24','ERBB2'],
        ['CD74','CLU'],
        ['CD74','COL1A1'],
        ['CD74','COL1A2'],
        ['CD74','EEF1G'],
        ['CD74','ERBB2'],
        ['CD74','FLNA'],
        ['CD74','FN1'],
        ['CD74','FTH1'],
        ['CD74','FTL'],
        ['CD74','GAPDH'],
        ['CD74','GNAS'],
        ['CD74','HLA-A'],
        ['CD74','HLA-B'],
        ['CD74','HLA-C'],
        ['CD74','HSPB1'],
        ['CD74','LDHA'],
        ['CD74','LTF'],
        ['CD74','MTRNR2L8'],
        ['CD74','MYL6'],
        ['CD74','P4HB'],
        ['CD74','PKM'],
        ['CD74','RPL13A'],
        ['CD74','RPL19'],
        ['CD74','RPLP0'],
        ['CD74','RPS2'],
        ['CD74','RPS6'],
        ['CD74','SLC39A6'],
        ['CD74','SPARC'],
        ['CD74','TMSB10'],
        ['CDH1','COL1A1'],
        ['CDH1','COL1A2'],
        ['CDH1','CPB1'],
        ['CDH1','FASN'],
        ['CDH1','FTH1'],
        ['CDH1','GAPDH'],
        ['CDH1','IGLL5'],
        ['CDH1','MTRNR2L8'],
        ['CDK12','PROX1'],
        ['CFL1','RPL10'],
        ['CHD4','COL1A1'],
        ['CHD4','EEF1A1'],
        ['CHRM3','KDM5A'],
        ['CLTC','FTH1'],
        ['CLTC','SCGB2A2'],
        ['CLTC','SRRM2'],
        ['COL1A1','CTNNB1'],
        ['COL1A1','DDX3X'],
        ['COL1A1','DDX5'],
        ['COL1A1','DNM2'],
        ['COL1A1','ERBB2'],
        ['COL1A1','ESR1'],
        ['COL1A1','FUS'],
        ['COL1A1','GATA3'],
        ['COL1A1','GNAS'],
        ['COL1A1','H3F3A'],
        ['COL1A1','HNRNPA2B1'],
        ['COL1A1','IDH2'],
        ['COL1A1','IL6ST'],
        ['COL1A1','JUN'],
        ['COL1A1','MAP3K1'],
        ['COL1A1','NCOR2'],
        ['COL1A1','NPM1'],
        ['COL1A1','PBX1'],
        ['COL1A1','PPP2R1A'],
        ['COL1A1','RAC1'],
        ['COL1A1','RAD21'],
        ['COL1A1','RHOA'],
        ['COL1A1','RPL10'],
        ['COL1A1','RPL22'],
        ['COL1A1','RPL5'],
        ['COL1A1','SET'],
        ['COL1A1','SF3B1'],
        ['COL1A1','TPM3'],
        ['COL1A2','DDX5'],
        ['COL1A2','ERBB2'],
        ['COL1A2','ERBB3'],
        ['COL1A2','H3F3A'],
        ['COL1A2','RHOA'],
        ['COL1A2','RPL10'],
        ['COL1A2','RPL5'],
        ['COL3A1','ERBB2'],
        ['COL3A1','FOXA1'],
        ['COL3A1','GNAS'],
        ['COL3A1','H3F3A'],
        ['COL3A1','HNRNPA2B1'],
        ['COL3A1','IL6ST'],
        ['COL3A1','MYH9'],
        ['COL3A1','RHOA'],
        ['COL3A1','RPL10'],
        ['COL6A1','ERBB2'],
        ['COX6C','GNAS'],
        ['COX6C','RPL5'],
        ['CPB1','FUBP1'],
        ['CPB1','GNAS'],
        ['CPB1','H3F3B'],
        ['CPB1','RPL5'],
        ['CTSD','ERBB2'],
        ['CTSD','GNAS'],
        ['CTSD','MYH9'],
        ['EEF1A1','FUS'],
        ['EEF1A1','IDH2'],
        ['EEF1A1','SRSF3'],
        ['EEF1A1','TBL1XR1'],
        ['EEF1G','ERBB2'],
        ['EEF2','ESR1'],
        ['EIF3E','GAPDH'],
        ['EIF3E','P4HB'],
        ['EIF4G1','MYH9'],
        ['ENO1','MYH9'],
        ['ENO1','RPL22'],
        ['ERBB2','FASN'],
        ['ERBB2','FN1'],
        ['ERBB2','FTH1'],
        ['ERBB2','FTL'],
        ['ERBB2','GNAS'],
        ['ERBB2','IGLL5'],
        ['ERBB2','KRT19'],
        ['ERBB2','MTRNR2L8'],
        ['ERBB2','PPIA'],
        ['ERBB2','PPP1R1B'],
        ['ERBB2','RPL13A'],
        ['ERBB2','RPL18A'],
        ['ERBB2','RPL19'],
        ['ERBB2','RPL3'],
        ['ERBB2','RPS18'],
        ['ERBB2','SQSTM1'],
        ['ERBB2','UBC'],
        ['ERBB3','IFI6'],
        ['FASN','GNAS'],
        ['FN1','GATA3'],
        ['FN1','GNAS'],
        ['FN1','HNRNPA2B1'],
        ['FN1','MYH9'],
        ['FN1','RPL10'],
        ['FN1','YWHAE'],
        ['FTH1','GNAS'],
        ['FTH1','H3F3A'],
        ['FTH1','NPM1'],
        ['FUS','MUCL1'],
        ['GAPDH','GNAS'],
        ['GAPDH','H3F3B'],
        ['GAPDH','HNRNPA2B1'],
        ['GAPDH','MYH9'],
        ['GAPDH','RPL10'],
        ['GAPDH','RPL5'],
        ['GATA3','MTRNR2L8'],
        ['GATA3','RPL8'],
        ['GNAS','HLA-A'],
        ['GNAS','HLA-C'],
        ['GNAS','HNRNPK'],
        ['GNAS','HSP90AB1'],
        ['GNAS','IGLL5'],
        ['GNAS','KRT19'],
        ['GNAS','KRT8'],
        ['GNAS','MYH9'],
        ['GNAS','PREX1'],
        ['GNAS','PSAP'],
        ['GNAS','PTMA'],
        ['GNAS','SCGB2A2'],
        ['GNAS','SEPT2'],
        ['GNAS','SLC39A6'],
        ['GNAS','TMSB4X'],
        ['GNAS','TPT1'],
        ['GNAS','XBP1'],
        ['H3F3A','IFI6'],
        ['H3F3A','MTRNR2L8'],
        ['H3F3B','H3F3C'],
        ['HHLA2','KMT2C'],
        ['HIF1A','SHISA9'],
        ['HLA-A','MYH9'],
        ['HLA-B','RPL10'],
        ['HLA-C','MYH9'],
        ['HNRNPA2B1','MTRNR2L8'],
        ['HNRNPA2B1','RRBP1'],
        ['HNRNPA2B1','SCGB2A2'],
        ['HNRNPA2B1','XBP1'],
        ['IFI6','RAC1'],
        ['IGFBP5','MYH9'],
        ['KRT19','MYH9'],
        ['KRT8','MYH9'],
        ['LTF','MYH9'],
        ['MAPK10','PTPN13'],
        ['MTRNR2L2','RPL10'],
        ['MTRNR2L8','NPM1'],
        ['MTRNR2L8','RPL10'],
        ['MUC1','MYH9'],
        ['MUCL1','MYH9'],
        ['MYH9','P4HB'],
        ['MYH9','PABPC1'],
        ['MYH9','PKM'],
        ['MYH9','PPP1R1B'],
        ['MYH9','PREX1'],
        ['MYH9','PTMS'],
        ['MYH9','RPL8'],
        ['MYH9','RPS6'],
        ['MYH9','S100A11'],
        ['MYH9','SCD'],
        ['MYH9','SCGB2A2'],
        ['MYH9','SRRM2'],
        ['NMNAT1','SALL4'],
        ['NPM1','SLC39A6'],
        ['NPM1','SPARC'],
        ['OPHN1','PTEN'],
        ['P4HB','RAC1'],
        ['PTPRB','XRCC6BP1'],
        ['RHOA','SLC39A6'],
        ['RNF213','VHL'],
        ['RPL10','RPL41'],
        ['RPL10','SPARC'],
        ['RPL10','TMSB10'],
        ['RPL22','SCD'],
        ['RPL5','RPS2'],
        ['RPL5','SCGB2A2'],
        ['RPL5','SPARC'],
        ['S100A6','YWHAE'],
        ['SCGB2A2','TPM3'],
        ['TNK2','TPM3'],
        ['ADD1','SH3BP2'],
        ['AFG3L2','TUBB6'],
        ['AKR1C2','AKR1C1'],
        ['AMBRA1','HARBI1'],
        ['AMH','SF3A2'],
        ['BAIAP2L2','PLA2G6'],
        ['C16ORF59','CCNF'],
        ['C1ORF226','NOS1AP'],
        ['CAPS','VMAC'],
        ['CARM1','C19ORF38'],
        ['CARM1','YIPF2'],
        ['CCDC134','MEI1'],
        ['CD59','FBXO3'],
        ['CNP','TTC25'],
        ['COL4A1','COL4A2'],
        ['COL4A2','COL4A1'],
        ['COLCA2','COLCA1'],
        ['CPT1B','CHKB'],
        ['DCANP1','TIFAB'],
        ['DMPK','DMWD'],
        ['DNAJC9','CFAP70'],
        ['DTX2','UPK3B'],
        ['EPOR','RGL3'],
        ['ERBB2','MIEN1'],
        ['GBP7','GBP4'],
        ['GOSR1','CPD'],
        ['HECTD4','TRAFD1'],
        ['IFITM10','CTSD'],
        ['IL11RA','GALT'],
        ['IL27RA','RLN3'],
        ['KANK1','DOCK8'],
        ['KRIT1','ANKIB1'],
        ['LOC728392','NLRP1'],
        ['LOC730183','SRCAP'],
        ['LRRC24','C8ORF82'],
        ['LTBR','SCNN1A'],
        ['MACF1','KIAA0754'],
        ['MUC17','MUC12'],
        ['NR2F6','USHBP1'],
        ['NSRP1','EFCAB5'],
        ['NUMBL','ADCK4'],
        ['PAQR6','SMG5'],
        ['PDLIM2','SORBS3'],
        ['PERM1','PLEKHN1'],
        ['PGA5','PGA4'],
        ['PGAP3','ERBB2'],
        ['PHF3','EYS'],
        ['PRR13','PCBP2'],
        ['PSMA3','ACTR10'],
        ['RNF4','CFAP99'],
        ['RPL21','OLFM4'],
        ['RPL21','TPT1'],
        ['RPS11','RPL13A'],
        ['SAP25','LRCH4'],
        ['SAPCD1','MSH5'],
        ['SEMA4C','FAM178B'],
        ['SLBP','TMEM129'],
        ['SLC25A11','RNF167'],
        ['STX5','WDR74'],
        ['SUPT16H','CHD8'],
        ['TBC1D8B','RNF128'],
        ['TBL1X','GPR143'],
        ['TMEM265','SRCAP'],
        ['TMEM86B','PPP6R1'],
        ['TNFRSF1A','SCNN1A'],
        ['TRPM4','PPFIA3'],
        ['TSSK3','FAM229A'],
        ['TXNDC11','ZC3H7A'],
        ['TYMP','ODF3B'],
        ['UBN1','PPL'],
        ['USP34','AHSA2'],
        ['YEATS4','LYZ'],
        ['YIPF2','CARM1'],
        ['ZCCHC8','RSRC2']
        
]



    data = fusions.get(options.organism.lower(),[])
    if data:

        #file_symbols = os.path.join(options.output_directory,'genes_symbols.txt')
        file_symbols = os.path.join(options.output_directory,'synonyms.txt')
        loci = symbols.generate_loci(file_symbols)

        genes = symbols.read_genes_symbols(file_symbols)

        d = []
        for g in data:
            if len(g)>2 or len(g) == 1:
                print "Error:",g
                sys.exit(1)
        for (g1,g2) in data:
            if g1.upper() != g2.upper():
                ens1 = symbols.ensembl(g1.upper(),genes,loci)
                ens2 = symbols.ensembl(g2.upper(),genes,loci)
                if ens1 and ens2:
                    for e1 in ens1:
                        for e2 in ens2:
                            if e1 != e2:
                                d.append([e1,e2])

        data = ['\t'.join(sorted(line)) + '\n' for line in d]
        data = list(set(data))

        print "%d known fusion genes found in manually currated database" % (len(data),)

        if not options.skip_filter_overlap:
            ens2hugo = dict([tuple(line.rstrip('\r\n').split('\t')) for line in file(os.path.join(options.output_directory,'genes_symbols.txt'),'r').readlines() if line.rstrip('\r\n')])

            d1 = []
            overlappings = ['ensembl_fully_overlapping_genes.txt',
                            'ensembl_same_strand_overlapping_genes.txt',
                            'gencode_fully_overlapping_genes.txt',
                            'gencode_same_strand_overlapping_genes.txt',
                            'refseq_fully_overlapping_genes.txt',
                            'refseq_same_strand_overlapping_genes.txt',
                            "ucsc_fully_overlapping_genes.txt",
                            "ucsc_same_strand_overlapping_genes.txt",
                            'pairs_pseudogenes.txt',
                            'banned.txt',
                            'dgd.txt',
                            'healthy.txt',
                            'paralogs.txt']
            ensembls = set(['ensembl_fully_overlapping_genes.txt',
                            'ensembl_same_strand_overlapping_genes.txt'])
            ens = []
            for ov in overlappings:
                p = os.path.join(options.output_directory,ov)
                print "Parsing file:",p
                d2 = []
                if os.path.isfile(p):
                    d2 = sorted(set([tuple(sorted(line.rstrip('\r\n').split('\t'))) for line in file(p,'r').readlines() if line.rstrip('\r\n')]))
                    d3 = set(['\t'.join(line)+'\n' for line in d2])
                    d4 = sorted([line for line in data if line in d3])
                    d4 = [line.rstrip('\r\n').split('\t') for line in d4]
                    d4 = [line+[ens2hugo.get(line[0],'')]+[ens2hugo.get(line[1],'')] for line in d4]
                    d4 = ['\t'.join(line)+'\n' for line in d4]
                    file(os.path.join(options.output_directory,"tcga3___%s"%(ov,)),'w').writelines(d4)
                if ov in ensembls:
                    ens.extend(d2)
                d1.extend(d2)
            d = set()
            for line in d1:
                (a,b) = (line[0],line[1])
                if a > b:
                    (a,b) = (b,a)
                d.add("%s\t%s\n" % (a,b))
            ens = set(['\t'.join(line)+'\n' for line in ens])
            ensembl = [line for line in data if line in ens]
            file(os.path.join(options.output_directory,'tcga3___ensembl.txt'),'w').writelines(sorted(ensembl))
            skipped = [line for line in data if line in d]
            data = [line for line in data if line not in d]
            file(os.path.join(options.output_directory,'tcga3___all.txt'),'w').writelines(sorted(skipped))

            print "%d known fusion genes left after removing the overlappings" % (len(data),)

    file(os.path.join(options.output_directory,'tcga3.txt'),'w').writelines(data)
    #
