#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It labels the candidate list of fusion genes generated by 'find_fusion_genes_trim2.py'.



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""
import sys
import os
import optparse

if __name__ == '__main__':

    #command line parsing

    usage="%prog [options]"
    description="""It labels the candidate list of fusion genes generated by 'find_fusion_genes_trim2.py'."""
    version="%prog 0.10 beta"

    parser=optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--input",
                      action="store",
                      type="string",
                      dest="input_fusion_genes_filename",
                      help="""The input file in text tab delimited format containing the fusion genes candidates produced by 'find_fusion_genes.py'. """)


    parser.add_option("--fusions_pairs",
                      action="store",
                      type="string",
                      dest="input_fusions_pairs_filename",
                      help="""The input text file tab separated format containing gene pairs which are used as filter for labeling (two columns and no header; the order of genes in the gene pairs is ignored).""")


    parser.add_option("--smaller_pairs",
                      action="store",
                      type="string",
                      dest="input_smaller_pairs_filename",
                      help="""The input text file tab separated format containing gene pairs which are used as filter for labeling (two columns and no header; the order of genes in the gene pairs is ignored).""")



    parser.add_option("--output_fusion_genes",
                      action="store",
                      type="string",
                      dest="output_fusion_genes_filename",
                      help="""The output text tab-separated file containing the candidate fusion genes which are found in the filter. The format is as the input file and sorted by counts column.""")





    (options,args) = parser.parse_args()

    # validate options
    if not (options.input_fusion_genes_filename and
            options.input_fusions_pairs_filename and
            options.output_fusion_genes_filename
            ):
        parser.print_help()
        parser.error("One of the options has not been specified.")
        sys.exit(1)

    shorts = set()
    # read the suporting pairs that are shorter
    if options.input_smaller_pairs_filename and os.path.isfile(options.input_smaller_pairs_filename):
        d = [e.rstrip("\r\n") for e in file(options.input_smaller_pairs_filename,"r") if e.rstrip("\r\n")]
        shorts = set(d)

    # read the supporting pairs file
    d = [e.rstrip("\r\n").split("\t") for e in file(options.input_fusions_pairs_filename,"r") if e.rstrip("\r\n")]
    h = d.pop(0) # header
    sp = dict()
    for e in d:
        k = (e[2],e[3])
        r = e[-1]
        if r:
            r = set(r.split(","))
            l = len(r.intersection(shorts))
            if l != 0:
                sp[k] = "t"+str(l)
    
    # read the fusions where are the labels
    d = [e.rstrip("\r\n").split("\t") for e in file(options.input_fusion_genes_filename,"r") if e.rstrip("\r\n")]
    
    r = []
    for e in d:
        k = (e[0],e[1])
        if k in sp:
            l = sp[k]
            f = e[:-1]
            if e[-1]:
                f.append(e[-1] + "," + l)
            else:
                f.append(l)
            r.append(f)
        else:
            r.append(e)
    
    file(options.output_fusion_genes_filename,"w").writelines(['\t'.join(e)+"\n" for e in r])



