#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
It takes as input a PSL format file generated by BLAT and it outputs only the
lines which contains the contigs with the best aligment which must be unique.



Author: Daniel Nicorici, Daniel.Nicorici@gmail.com

Copyright (c) 2009-2022 Daniel Nicorici

This file is part of FusionCatcher.

FusionCatcher is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

FusionCatcher is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with FusionCatcher (see file 'COPYING.txt').  If not, see
<http://www.gnu.org/licenses/>.

By default, FusionCatcher is running BLAT aligner
<http://users.soe.ucsc.edu/~kent/src/> but it offers also the option to disable
all its scripts which make use of BLAT aligner if you choose explicitly to do so.
BLAT's license does not allow to be used for commercial activities. If BLAT
license does not allow to be used in your case then you may still use
FusionCatcher by forcing not use the BLAT aligner by specifying the option
'--skip-blat'. Fore more information regarding BLAT please see its license.

Please, note that FusionCatcher does not require BLAT in order to find
candidate fusion genes!

This file is not running/executing/using BLAT.

"""


# info PSL
"""
========================================================
More about PSL format is here: http://genome.ucsc.edu/FAQ/FAQformat#format2


PSL format

PSL lines represent alignments, and are typically taken from files generated
by BLAT or psLayout. See the BLAT documentation for more details. All of the
following fields are required on each data line within a PSL file:

   1. matches - Number of bases that match that aren't repeats
   2. misMatches - Number of bases that don't match
   3. repMatches - Number of bases that match but are part of repeats
   4. nCount - Number of 'N' bases
   5. qNumInsert - Number of inserts in query
   6. qBaseInsert - Number of bases inserted in query
   7. tNumInsert - Number of inserts in target
   8. tBaseInsert - Number of bases inserted in target
   9. strand - '+' or '-' for query strand. For translated alignments, second '+'or '-' is for genomic strand
  10. qName - Query sequence name
  11. qSize - Query sequence size
  12. qStart - Alignment start position in query
  13. qEnd - Alignment end position in query
  14. tName - Target sequence name
  15. tSize - Target sequence size
  16. tStart - Alignment start position in target
  17. tEnd - Alignment end position in target
  18. blockCount - Number of blocks in the alignment (a block contains no gaps)
  19. blockSizes - Comma-separated list of sizes of each block
  20. qStarts - Comma-separated list of starting positions of each block in query
  21. tStarts - Comma-separated list of starting positions of each block in target

Example:
Here is an example of an annotation track in PSL format. Note that line breaks have been inserted into the PSL lines in this example for documentation display purposes. Click here for a copy of this example that can be pasted into the browser without editing.

track name=fishBlats description="Fish BLAT" useScore=1
59 9 0 0 1 823 1 96 +- FS_CONTIG_48080_1 1955 171 1062 chr22
    47748585 13073589 13073753 2 48,20,  171,1042,  34674832,34674976,
59 7 0 0 1 55 1 55 +- FS_CONTIG_26780_1 2825 2456 2577 chr22
    47748585 13073626 13073747 2 21,45,  2456,2532,  34674838,34674914,
59 7 0 0 1 55 1 55 -+ FS_CONTIG_26780_1 2825 2455 2676 chr22
    47748585 13073727 13073848 2 45,21,  249,349,  13073727,13073827,

Be aware that the coordinates for a negative strand in a PSL line are handled in a special way. In the qStart and qEnd fields, the coordinates indicate the position where the query matches from the point of view of the forward strand, even when the match is on the reverse strand. However, in the qStarts list, the coordinates are reversed.

Example:
Here is a 30-mer containing 2 blocks that align on the minus strand and 2 blocks that align on the plus strand (this sometimes can happen in response to assembly errors):

0         1         2         3 tens position in query
0123456789012345678901234567890 ones position in query
            ++++          +++++ plus strand alignment on query
    --------    ----------      minus strand alignment on query

Plus strand:
     qStart=12
     qEnd=31
     blockSizes=4,5
     qStarts=12,26

Minus strand:
     qStart=4
     qEnd=26
     blockSizes=10,8
     qStarts=5,19

Essentially, the minus strand blockSizes and qStarts are what you would get if you reverse-complemented the query. However, the qStart and qEnd are not reversed. To convert one to the other:

     qStart = qSize - revQEnd
     qEnd = qSize - revQStart
"""

import os
import sys
import optparse
import tempfile
import sort_ttdb
import gc
import shutil
import multiprocessing


#########################
def give_me_temp_filename(tmp_dir = None):
    if tmp_dir and (not os.path.isdir(tmp_dir)) and (not os.path.islink(tmp_dir)):
        os.makedirs(tmp_dir)
    (ft,ft_name) = tempfile.mkstemp(dir = tmp_dir)
    os.close(ft)
    return ft_name

#########################
def delete_file(some_file):
    if os.path.isfile(some_file) or os.path.islink(some_file):
        os.remove(some_file)
    elif os.path.isdir(some_file):
        shutil.rmtree(some_file)

#########################
def lines_from(a_filename):
    # it gives chunks
    fin = open(a_filename,'r')
    while True:
        lines = fin.readlines(10**8)
        if not lines:
            break
        gc.disable()
        lines = [line.rstrip('\r\n').split('\t') for line in lines]
        gc.enable()
        for line in lines:
            if line:
                yield line
    fin.close()

#########################
def lines_none_from(a_filename):
    # it gives chunks
    fin = open(a_filename,'r')
    while True:
        lines = fin.readlines(10**8)
        if not lines:
            break
        gc.disable()
        lines = [line.rstrip('\r\n').split('\t') for line in lines]
        gc.enable()
        for line in lines:
            if line:
                yield line
    yield None
    fin.close()

#########################
def contigs_multiple_from(a_psl_filename):
    last_contig = ''
    chunk = []
    for line in lines_from(a_psl_filename):
        if not chunk:
            last_contig = line[9]
        if last_contig != line[9]: # line[9] is column no 10 which is query name
            # the bin is full and now analyze it
            if len(chunk)<2 or int(chunk[0][0]) > int(chunk[1][0]):
                pass
            else:
                yield chunk[0][9]+'\n'
            last_contig = line[9]
            chunk = []
        chunk.append(line)
    if chunk:
        if len(chunk)<2 or int(chunk[0][0]) > int(chunk[1][0]):
            pass
        else:
            yield chunk[0][9]+'\n'

#########################
def check_ties(string1,string2,ensg09):
    f = False
    a = string1.find(ensg09)
    b = string2.find(ensg09)
    if ensg09 and (a == -1 and b != -1) or (a != -1 and b == -1):
        f = True
    return f

#########################
def contigs_unique_from(a_psl_filename):
    last_contig = ''
    chunk = []
    for line in lines_from(a_psl_filename):
        if not chunk:
            last_contig = line[9] # line[9] is column no 10 which is query name
        if last_contig != line[9]: # line[9] is column no 10 which is query name
            # the bin is full and now analyze it
            n = len(chunk)
            if n == 1:
                yield '\t'.join(chunk[0])+'\n'
            elif chunk[0][0] != chunk[1][0]: #matches for first best mapping and second best mapping
                yield '\t'.join(chunk[0])+'\n'
            last_contig = line[9]
            chunk = []
        chunk.append(line)
    if chunk:
        n = len(chunk)
        if n == 1:
            yield '\t'.join(chunk[0])+'\n'
        elif chunk[0][0] != chunk[1][0]:
            yield '\t'.join(chunk[0])+'\n'

#########################
def contigs_unique_overlapping_from(a_psl_filename):
    # when there is tie best mapping then all the best mappings are outputed if they overlap each other on same reference sequence
    last_contig = ''
    chunk = []
    for line in lines_none_from(a_psl_filename):
        if not line: # signal for last line
            line9 = last_contig+'_'
        else:
            line9 = line[9]
        if not chunk:
            last_contig = line9 # line[9] is column no 10 which is query name
        if last_contig != line9: # line[9] is column no 10 which is query name
            # the bin is full and now analyze it
            n = len(chunk)
            if n == 1:
                yield '\t'.join(chunk[0])+'\n'
            elif chunk[0][0] != chunk[1][0]: #matches for first best mapping and second best mapping
                yield '\t'.join(chunk[0])+'\n'
            else: # allow ties for best mappins if all of them are overllaping each other
                chunk = [el for el in chunk if el[0]==chunk[0][0]] # get top ties
                tseq =  set([el[13] for el in chunk]) # get target sequence for all ties
                if len(tseq) == 1:
                    # test overlapping
                    a0 = int(chunk[0][15])
                    a1 = int(chunk[0][16])
                    flg = True
                    for e in xrange(1,len(chunk)):
                        b0 = int(chunk[e][15])
                        b1 = int(chunk[e][16])
                        if not (b0 <= a1 and b1 >= a0):
                            flg = False
                            break
                    if flg:
                        for el in chunk:
                            yield '\t'.join(el)+'\n'
            last_contig = line9
            chunk = []
        chunk.append(line)


#########################
def contigs_unique_gene_overlapping_ties_from(a_psl_filename, overlapping_genes=set()):
    # when there is a tie best mapping then all the best mappings are outputed if they overlap each other on same reference sequence
    """
    example of overlapping
    
100	0	0	0	0	0	1	12890	+	3/2	100	0	100	ENSG00000187266|ENSG09000000020|7138	44139	6042	19032	2	74,26,	0,74,	6042,19006,
100	0	0	0	0	0	1	10311	-	3/2	100	0	100	ENSG00000204650|ENSG09000000035|27889	44890	18779	29190	2	25,75,	0,25,	18779,29115,
82	0	0	0	0	0	1	12890	-	4/1	82	0	82	ENSG00000187266|ENSG09000000020|7138	44139	6081	19053	2	35,47,	0,35,	6081,19006,
82	0	0	0	0	0	1	10311	+	4/1	82	0	82	ENSG00000204650|ENSG09000000035|27889	44890	18758	29151	2	46,36,	0,46,	18758,29115,
    """
    last_contig = ''
    chunk = []
    for line in lines_none_from(a_psl_filename):
        if not line: # signal for last line
            line9 = last_contig+'_'
        else:
            line9 = line[9]
            
        if not chunk:
            last_contig = line9 # line[9] is column no 10 which is query name
            
        if last_contig != line9: # line[9] is column no 10 which is query name
            # the bin is full and now analyze it
            n = len(chunk)
            if n == 1:
                yield '\t'.join(chunk[0])+'\n'
            elif chunk[0][0] != chunk[1][0]: #matches for first best mapping and second best mapping
                yield '\t'.join(chunk[0])+'\n'
            else: # allow ties for best mappins if all of them are overllaping each other
                chunk = [el for el in chunk if el[0]==chunk[0][0]] # get top ties
                tseq =  set([el[13] for el in chunk]) # get target sequence for all ties
                ltseq = len(tseq)
                if tseq and ltseq == 1:
                    # test overlapping
                    a0 = int(chunk[0][15])
                    a1 = int(chunk[0][16])
                    flg = True
                    for e in xrange(1,len(chunk)):
                        b0 = int(chunk[e][15])
                        b1 = int(chunk[e][16])
                        if not (b0 <= a1 and b1 >= a0):
                            flg = False
                            break
                    if flg:
                        for el in chunk:
                            yield '\t'.join(el)+'\n'
                else: # more than one gene
                    tseq = sorted(set([tuple(sorted(el.split('|')[:2])) for el in tseq]))
                    # are all the genes overlapping each other?
                    flg = True
                    nn = len(tseq)
                    if ltseq != nn:
                        flg = False
                    else:
                        for ax in xrange(nn-1):
                            for bx in xrange(ax+1,nn):
                                cx1 = tseq[ax][0]
                                cx2 = tseq[ax][1]
                                dx1 = tseq[bx][0]
                                dx2 = tseq[bx][1]
                                e1 = '|'.join(sorted([cx1,dx1]))
                                e2 = '|'.join(sorted([cx1,dx2]))
                                e3 = '|'.join(sorted([cx2,dx1]))
                                e4 = '|'.join(sorted([cx2,dx2]))
                                if (e1 in overlapping_genes) or (e2 in overlapping_genes) or (e3 in overlapping_genes) or (e4 in overlapping_genes):
                                    pass
                                else:
                                    flg = False
                                    break
                            if not flg:
                                break
                    if flg:
                        for el in chunk:
                            yield '\t'.join(el)+'\n'
            last_contig = line9
            chunk = []
        chunk.append(line)

#########################
def contigs_unique_overlapping_and_ties_from(a_psl_filename, allowed_ties = ""):
    # when there is a tie best mapping then all the best mappings are outputed if they overlap each other on same reference sequence
    last_contig = ''
    chunk = []
    for line in lines_none_from(a_psl_filename):
        if not line: # signal for last line
            line9 = last_contig+'_'
        else:
            line9 = line[9]
            
        if not chunk:
            last_contig = line9 # line[9] is column no 10 which is query name
            
        if last_contig != line9: # line[9] is column no 10 which is query name
            # the bin is full and now analyze it
            n = len(chunk)
            if n == 1:
                yield '\t'.join(chunk[0])+'\n'
            elif chunk[0][0] != chunk[1][0]: #matches for first best mapping and second best mapping
                yield '\t'.join(chunk[0])+'\n'
            else: # allow ties for best mappins if all of them are overllaping each other
                chunk = [el for el in chunk if el[0]==chunk[0][0]] # get top ties
                if allowed_ties:
                    tseq =  set([el[13] for el in chunk if el[13].find(allowed_ties) == -1]) # get target sequence for all ties
                else:
                    tseq =  set([el[13] for el in chunk]) # get target sequence for all ties
                if len(tseq) == 1:
                    # test overlapping
                    a0 = int(chunk[0][15])
                    a1 = int(chunk[0][16])
                    flg = True
                    for e in xrange(1,len(chunk)):
                        b0 = int(chunk[e][15])
                        b1 = int(chunk[e][16])
                        if not (b0 <= a1 and b1 >= a0):
                            if allowed_ties:
                                if chunk[e][13].find(allowed_ties) == -1:
                                    flg = False
                                    break
                            else:
                                flg = False
                                break
                    if flg:
                        for el in chunk:
                            yield '\t'.join(el)+'\n'
            last_contig = line9
            chunk = []
        chunk.append(line)


#########################
def contigs_unique_and_ties_from(a_psl_filename, allowed_ties = ""):
    last_contig = ''
    chunk = []
    for line in lines_from(a_psl_filename):
        if not chunk:
            last_contig = line[9] # line[9] is column no 10 which is query name
        if last_contig != line[9]: # line[9] is column no 10 which is query name
            # the bin is full and now analyze it
            n = len(chunk)
            if n == 1:
                yield '\t'.join(chunk[0])+'\n'
            else:
                c00 = int(chunk[0][0])
                c10 = int(chunk[1][0])
                if c00 != c10:
                    yield '\t'.join(chunk[0])+'\n'
                elif allowed_ties and c00 == c10 and (n == 2 or int(chunk[2][0]) < c00) and check_ties(chunk[0][13],chunk[1][13],allowed_ties):
                    yield '\t'.join(chunk[0])+'\n'
                    yield '\t'.join(chunk[1])+'\n'
            last_contig = line[9]
            chunk = []
        chunk.append(line)
    if chunk:
        n = len(chunk)
        if n == 1:
            yield '\t'.join(chunk[0])+'\n'
        else:
            c00 = int(chunk[0][0])
            c10 = int(chunk[1][0])
            if c00 != c10:
                yield '\t'.join(chunk[0])+'\n'
            elif allowed_ties and c00 == c10 and (n == 2 or int(chunk[2][0]) < c00) and check_ties(chunk[0][13],chunk[1][13],allowed_ties):
                yield '\t'.join(chunk[0])+'\n'
                yield '\t'.join(chunk[1])+'\n'



########################
def give_gene_name(t):
    #g = [el.split('ge=')[1] for el in t.split(';') if el.startswith('ge=')]
    g = t.partition(';')[2]
    return g


#########################
def contigs_unique_within_same_gene_from(a_psl_filename):
    last_contig=''
    chunk=[]

    for line in lines_from(a_psl_filename):
        if not chunk:
            last_contig = line[9]
        if last_contig != line[9]: # line[9] is column no 10 which is query name
            # the bin is full and analyze it
            if len(chunk)<2 or int(chunk[0][0]) > int(chunk[1][0]):
                yield '\t'.join(chunk[0])+'\n'
            last_contig = line[9]
            chunk=[]
        if len(chunk) == 1 and chunk[0][0] == line[0] and give_gene_name(chunk[0][13]) == give_gene_name(line[13]):# line[13] is coumn 14 which is the target sequence name
            pass
        else:
            chunk.append(line)
    if chunk:
        if len(chunk)<2 or int(chunk[0][0]) > int(chunk[1][0]):
            yield '\t'.join(chunk[0])+'\n'

#########################
def contigs_multiple_within_same_gene_from(a_psl_filename):
    last_contig=''
    chunk=[]

    for line in lines_from(a_psl_filename):
        if not chunk:
            last_contig = line[9]
        if last_contig != line[9]: # line[9] is column no 10 which is query name
            # the bin is full and analyze it
            if len(chunk)<2 or int(chunk[0][0]) > int(chunk[1][0]):
                pass
            else:
                yield chunk[0][9]+'\n'
            last_contig = line[9]
            chunk=[]
        if len(chunk) == 1 and chunk[0][0] == line[0] and give_gene_name(chunk[0][13]) == give_gene_name(line[13]): # line[13] is coumn 14 which is the target sequence name
            pass
        else:
            chunk.append(line)
    if chunk:
        if len(chunk)<2 or int(chunk[0][0]) > int(chunk[1][0]):
            pass
        else:
            yield chunk[0][9]+'\n'

################################
def filter_line(a_line, threshold = 15, mismatches = 1000000):
    # threshold = 15 # bp
    r = a_line.rstrip('\r\n').split('\t')
    a_line = None
    if int(r[1]) > mismatches:
        pass
    else:
        extra_mismatches = sum([el for el in map(int,r[18][:-1].split(',')) if el < threshold])
        if extra_mismatches > 0:
            r[0] = str(int(r[0]) - extra_mismatches)
            r[1] = int(r[1]) + extra_mismatches
            if r[1] > mismatches:
                pass
            else:
                r[1] = str(r[1])
                a_line = '\t'.join(r)+'\n'
        else:
            a_line = '\t'.join(r)+'\n'
    return a_line

############################
def fix_short_blocks(psl_file, fixed_psl_file, threshold = 15, mismatches = 1000000):
    """
    It converts the short blocks (length < 15 bp) into mismatches.
    """

    fid = open(psl_file,'r')
    fod = open(fixed_psl_file,'w')
    flag = True
    while True:
        lines = fid.readlines(10**8)
        if not lines:
            break
        gc.disable()
        lines = [filter_line(line,threshold,mismatches) for line in lines if line.rstrip('\r\n')]
        lines = [line for line in lines if line]
        gc.enable()
        if lines:
            fod.writelines(lines)
            flag = False
    if flag:
        fod.write('')
    fod.close()
    fid.close()

if __name__ == '__main__':

    #command line parsing

    usage = "%prog [options]"
    description = """It takes as input a PSL format file (generated by BLAT) and it outputs only the lines which contains the contigs with the best aligment which must be unique."""
    version = "%prog 0.10 beta"

    parser=optparse.OptionParser(usage=usage,description=description,version=version)

    parser.add_option("--input",
                      action="store",
                      type="string",
                      dest="input_filename",
                      help="""The input file in PSL format.""")

    parser.add_option("--same_gene",
                      action = "store_true",
                      dest = "same_gene",
                      default = False,
                      help="""The gene name appears in the query name as for example ge=ENSG010101.""")

    parser.add_option("--output",
                      action="store",
                      type="string",
                      dest="output_filename",
                      help="""The output PSL file containing the contigs with the best alignment which must be unique.""")

    parser.add_option("--ties",
                      action="store",
                      type="string",
                      dest="input_ties_filename",
                      help="""A input text file containing a string (which is part of a gene) for which ties are allowed.""")

    parser.add_option("--ties-overlappings",
                      action="store",
                      type="string",
                      dest="input_ties_overlappings_filename",
                      help="""A input text file containing a list of pairs of genes which overlap each other.""")

    parser.add_option("--output_multiple_alignments","-m",
                      action="store",
                      type="string",
                      dest="output_multiple_alignments_filename",
                      help="""A text file containing a list with the names of the contigs such that the best of their alignments are not unique.""")

    parser.add_option("--output_unique_alignments","-u",
                      action="store",
                      type="string",
                      dest="output_unique_alignments_filename",
                      help="""A text file containing a list with the names of the contigs such that the best of their alignments are unique.""")

    parser.add_option("--anchor",
                      action = "store",
                      type = "int",
                      dest = "anchor",
                      default = 15,
                      help = """The minimum length of the anchor (i.e. overhang), i.e. shortest part of a mapping sequence. Default is %default.""")

    parser.add_option("-p", "--processes",
                      action = "store",
                      type = "int",
                      dest = "processes",
                      default = 0,
                      help = """Number of parallel threads/processes/CPUs to be used for computations. In case of value 0 then the program will use all the CPUs which are found. The default value is %default.""")

    parser.add_option("--buffer-size",
                  action = "store",
                  type = "string",
                  default = "80%",
                  dest = "buffer_size",
                  help = "The main buffer size which is passed further to GNU sort command. For more see '--buffer-size' of GNU sort command. Default is '%default'.")


    parser.add_option("--mismatches",
                      action="store",
                      type="float",
                      dest="mismatches",
                      default=1000,
                      help="""All alignments having strictly more mismatches will be removed. Default is '%default'.""")

    parser.add_option("--tmp_dir",
                  action="store",
                  type="string",
                  dest="tmp_dir",
                  help="The directory which should be used as temporary directory. By default is the OS temporary directory.")


    (options,args) = parser.parse_args()

    # validate options
    if not (options.input_filename and
            options.output_filename
            ):
        parser.print_help()
        sys.exit(1)

    temporary_dir = None
    if options.tmp_dir:
        temporary_dir = options.tmp_dir

    cpus = options.processes
    if cpus == 0:
        cpus = multiprocessing.cpu_count()

    # running
    print >>sys.stderr,"Fixing PSL..."
    ft_name_1 = give_me_temp_filename(tmp_dir = options.tmp_dir)
    fix_short_blocks(options.input_filename, ft_name_1, threshold = options.anchor, mismatches = options.mismatches)
#    for line in file(ft_name_1,'r'):
#        os.write(2,line)

    print >>sys.stderr,"Sorting PSL..."
    ft_name_2 = give_me_temp_filename(tmp_dir = options.tmp_dir)
    sort_ttdb.sort_columns(ft_name_1,
                           ft_name_2,
                           columns = '10,1nd,2n,18n,13', # sequence name, matches, mismatches, count blocks, target sequence
                           header = False,
                           ignore_case = False,
                           tmp_dir = options.tmp_dir,
                           buffer_size = options.buffer_size,
                           parallel = cpus
                           )
#    for line in file(ft_name_2,'r'):
#        os.write(3,line)
    delete_file(ft_name_1)

    print >>sys.stderr,"Processing PSL..."
    data=[]
    ties = []
    if options.same_gene: # or file(ft_name_2,'r').readline().find(';ge=')>-1:
        data = contigs_unique_within_same_gene_from(ft_name_2)
    else:
        ties = ""
        ties_overlappings = set()
        if options.input_ties_filename:
            ties = file(options.input_ties_filename,"r").readline().rstrip("\r\n")
        elif options.input_ties_overlappings_filename:
            ties_overlappings = set()
            for f1 in options.input_ties_overlappings_filename.split(','):
                if f1:
                    ties_overlappings.update(set(['|'.join(sorted(line.rstrip('\r\n').split('\t'))) for line in file(f1,"r").readlines() if line.rstrip('\r\n')]))
            
        if ties:
            #data = contigs_unique_and_ties_from(ft_name_2,ties)
            data = contigs_unique_overlapping_and_ties_from(ft_name_2,ties)
        elif ties_overlappings:
            data = contigs_unique_gene_overlapping_ties_from(ft_name_2,ties_overlappings)
        else:
            #data = contigs_unique_from(ft_name_2)
            data = contigs_unique_overlapping_from(ft_name_2)

    ft_name_3 = give_me_temp_filename(tmp_dir = options.tmp_dir)
    file(ft_name_3,'w').writelines(data)
#    for line in file(ft_name_3,'r'):
#        os.write(4,line)


    print >>sys.stderr,"Sorting again..."
    ft_name_4 = give_me_temp_filename(tmp_dir = options.tmp_dir)
    if options.output_filename:
        file(options.output_filename,'w').write('')
        sort_ttdb.sort_columns(ft_name_3,
                               options.output_filename,
                               columns = '1nd',
                               header = False,
                               ignore_case = False,
                               tmp_dir = options.tmp_dir,
                               buffer_size = options.buffer_size,
                               parallel = cpus
                               )
    else:
        sort_ttdb.sort_columns(ft_name_3,
                               ft_name_4,
                               columns = '1nd',
                               header = False,
                               ignore_case = False,
                               tmp_dir = options.tmp_dir,
                               buffer_size = options.buffer_size,
                               parallel = cpus
                               )
#    for line in file(options.output_filename,'r'):
#        os.write(5,line)
    delete_file(ft_name_3)


    if options.output_multiple_alignments_filename:
        print >>sys.stderr,"Writing multiple alignments..."
        data = []
        #if options.same_gene or file(ft_name_2,'r').readline().find(';ge=')>-1: # ;gen=?????
        if options.same_gene: # or file(ft_name_2,'r').readline().find(';ENSG')>-1: # ;gen=?????
            data = contigs_multiple_within_same_gene_from(ft_name_2)
        else:
            data = contigs_multiple_from(ft_name_2)
        file(options.output_multiple_alignments_filename,'w').writelines(data)
    delete_file(ft_name_2)

    if options.output_unique_alignments_filename:
        print >>sys.stderr,"Writing unique alignments..."
        d = options.output_filename if options.output_filename else ft_name_4
        data = list(set([line.rstrip('\r\n').split('\t')[9]+'\n' for line in file(d,'r').readlines()])) # [9] is column 10 which is query name
        file(options.output_unique_alignments_filename,'w').writelines(data)

    delete_file(ft_name_4)

    print >>sys.stderr,"The end."

#
