"""Command to regenerate Dockerfile for container building"""

import os
import shutil
import subprocess
from pathlib import Path
import rich_click as click
from harpy.common.conda import create_conda_recipes
from harpy.common.workflow import Workflow

@click.command(hidden = True)
def containerize():
    """
    Configure conda and docker environments

    **INTERNAL USE ONLY**. Used to recreate all the conda environments required
    by the workflows and build a dockerfile from that.
    """
    workflow = Workflow("container", "environments.smk", "container", 1)
    workflow.fetch_snakefile()
    create_conda_recipes("container")
    
    with open("Dockerfile.raw", "w", encoding = "utf-8") as dockerraw:
        _module = subprocess.run(
            'snakemake -s container/workflow/workflow.smk --containerize --directory container'.split(),
            stdout = dockerraw
        )

    with open("Dockerfile.raw", "r") as dockerraw, open("Dockerfile", "w") as dockerfile:
            # copy over the first three lines
            dockerfile.write(dockerraw.readline())
            dockerfile.write(dockerraw.readline())
            dockerfile.write(dockerraw.readline())
            dockerfile.write("\nCOPY container/workflow/envs/*.yaml /\n")
            env_hash = {}
            for line in dockerraw:
                if line.startswith("#"):
                    continue
                if line.startswith("COPY"):
                    dockercmd, env, hashname = line.split()
                    env = Path(env).stem
                    hashname = hashname.split("/")[-2]
                    env_hash[env] = hashname
            runcmds = []
            for env, _hash in env_hash.items():
                runcmds.append(f"conda env create --prefix /conda-envs/{_hash} --file /{env}.yaml && \\")
            runcmds.append("conda clean --all -y")
            dockerfile.write("\nRUN ")
            dockerfile.write(
                "\n\t".join(runcmds)
            )
    os.remove("Dockerfile.raw")

@click.command(hidden = True)
@click.argument('workflows', required = True, type= click.Choice(["all", "align", "assembly", "metassembly", "phase", "qc", "r", "simulations", "stitch", "variants"]), nargs = -1)
def localenv(workflows):
    """
    Install Harpy workflow dependencies via conda

    **INTERNAL USE ONLY**. Used to recreate the conda environments required
    by the workflows. Provide any combination of: 
    - all
    - align
    - assembly
    - metassembly
    - phase
    - qc
    - r
    - simulations
    - stitch
    - variants
    """
    workflow = Workflow("localenv", "environments.smk", "localenv/", 1)
    # if "all" was mixed with other workflows, default to just all and avoid doubling up
    if "all" in workflows:
        create_conda_recipes(workflow.output_directory, workflows)
    else:
        create_conda_recipes(workflow.output_directory)
    workflow.fetch_snakefile()
    workflow.snakemake_cmd_relative = " ".join(["snakemake", "-s", os.path.join(workflow.workflow_directory, "workflow.smk"), "--sdm", "conda", "--cores 2", "--conda-prefix ../.environments", "--conda-cleanup-pkgs cache", "--directory localenv", "--config spades=True"])
    workflow.launch()
    shutil.rmtree(workflow.output_directory, ignore_errors = True)
