#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright 2024-2025 EMBL - European Bioinformatics Institute
#
# Licensed under the Apache License, Version 2.0 (the 'License');
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an 'AS IS' BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import argparse
import csv
import logging
import os
from collections import defaultdict
from pathlib import Path

from mgnify_pipelines_toolkit.analysis.assembly.go_utils import parse_interproscan_tsv

logging.basicConfig(level=logging.INFO, format="%(asctime)s - %(levelname)s: %(message)s")


def parse_args():
    description = "Go slim pipeline."
    parser = argparse.ArgumentParser(description=description)
    parser.add_argument("-go", "--go_obo", help="Gene Ontology basic file.", required=True)
    parser.add_argument("-gb", "--go_banding", help="Subset GO banding file.", required=True)
    parser.add_argument(
        "-gaf",
        "--gaf_input",
        help="GAF file, generated by generate_gaf.py",
        required=True,
    )
    parser.add_argument("-i", "--ips_input", help="InterProScan result file.", required=True)
    parser.add_argument("-o", "--output", help="GO summary output file.", required=True)
    args = parser.parse_args()

    go_obo = args.go_obo
    go_banding = args.go_banding
    gaf_input = args.gaf_input
    ips_input = args.ips_input
    output = args.output

    return go_obo, go_banding, gaf_input, ips_input, output


def parse_mapped_gaf_file(gaf_file: Path) -> defaultdict[set]:
    mapped_go_dict = defaultdict(set)
    if os.path.exists(gaf_file):
        with open(gaf_file, "r") as handle:
            for line in handle:
                if not line.startswith("!"):
                    line = line.strip()
                    splitted_line = line.split("\t")
                    go_id = splitted_line[1]
                    mapped_go_id = splitted_line[4]
                    mapped_go_dict[go_id].add(mapped_go_id)
    return mapped_go_dict


def get_go_slim_summary(go_slim_banding_file, goslims2_protein_count):
    summary = []

    with open(go_slim_banding_file, "r") as fr:
        for line in fr:
            if line.startswith("GO"):
                line = line.strip()
                line_chunks = line.split("\t")
                go_id = line_chunks[0]
                term = line_chunks[1]
                category = line_chunks[2]
                # Default value for the count
                count = 0
                if go_id in goslims2_protein_count:
                    count = goslims2_protein_count[go_id]
                summary.append((go_id, term, category, count))
    return summary


def write_go_summary_to_file(go_summary, output_file):
    """
    Write a sorted GO summary to a TSV file.

    :param go_summary: A list of tuples, where each tuple contains the following
                       elements:
                       - go (str): The GO identifier.
                       - term (str): The GO term description.
                       - category (str): The category of the GO term.
                       - count (int): The count associated with the GO term.
    :param output_file: The path to the output TSV file where the sorted GO
    """
    sorted_go_summary = sorted(go_summary, key=lambda x: x[3], reverse=True)
    with open(output_file, "w", newline="") as fw:
        tsv_writer = csv.writer(fw, delimiter="\t")
        tsv_writer.writerow(["go", "term", "category", "count"])
        for go, term, category, count in sorted_go_summary:
            tsv_writer.writerow([go, term, category, count])


def parse_gene_ontology(obo_file):
    """
    Parses OBO formatted file.
    :param obo_file:
    :return:
    """
    go_term_tuples = []
    with open(obo_file, "r") as fr:
        id, term, category = "", "", ""
        for line in fr:
            line = line.strip()
            split_line = line.split(": ")
            if line.startswith("id:"):
                id = split_line[1]
            elif line.startswith("name:"):
                term = split_line[1]
            elif line.startswith("namespace"):
                category = split_line[1]
            else:
                if id.startswith("GO:") and id and term and category:
                    item = (id, term, category)
                    go_term_tuples.append(item)
                    id, term, category = "", "", ""
    return go_term_tuples


def get_full_go_summary(core_gene_ontology, go2protein_count_dict, top_level_go_ids):
    summary = []

    for go_id, term, category in core_gene_ontology:
        if (go_id in go2protein_count_dict) and (
            go_id not in top_level_go_ids
        ):  # make sure that top level terms are not included (they tell you nothing!)
            count = go2protein_count_dict[go_id]
            summary.append((go_id, term, category, count))
    summary.sort(key=lambda x: (x[2], -x[3]))
    return summary


def main():
    go_obo, go_banding, gaf_input, ips_input, output = parse_args()

    logging.info("Parsing the InterProScan input: " + ips_input)
    go2protein_count_dict = parse_interproscan_tsv(ips_input)
    logging.info("Finished parsing.")

    # Generate GO summary
    logging.info("Loading full Gene ontology: " + go_obo)
    go_term_tuples = parse_gene_ontology(go_obo)
    logging.info("Finished loading.")

    logging.info("Generating full GO summary...")
    top_level_go_ids = ["GO:0008150", "GO:0003674", "GO:0005575"]
    full_go_summary = get_full_go_summary(go_term_tuples, go2protein_count_dict, top_level_go_ids)
    logging.info("Finished generation.")

    logging.info("Writing full GO summary: " + output)
    write_go_summary_to_file(full_go_summary, output)
    logging.info("Finished writing.")

    mapped_go_terms = parse_mapped_gaf_file(gaf_input)
    logging.info("Getting GO slim counts")
    goslims2_protein_count = parse_interproscan_tsv(ips_input, mapped_go_terms)

    go_slim_summary = get_go_slim_summary(go_banding, goslims2_protein_count)
    go_slim_output_file = output + "_slim"
    logging.info("Writing GO slim summary: " + go_slim_output_file)
    write_go_summary_to_file(go_slim_summary, go_slim_output_file)
    logging.info("Finished writing.")


if __name__ == "__main__":
    main()
