#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright 2024-2025 EMBL - European Bioinformatics Institute
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


import argparse
from collections import defaultdict
import json
import pathlib
import logging

import pandas as pd
import pyfastx

from mgnify_pipelines_toolkit.constants.thresholds import MAJORITY_MARKER_PROPORTION

logging.basicConfig(level=logging.DEBUG)


def parse_args():

    parser = argparse.ArgumentParser()
    parser.add_argument(
        "-i",
        "--input_path",
        required=True,
        type=str,
        help="Input directory containing amplicon analysis pipeline results",
    )
    parser.add_argument(
        "-r",
        "--runs",
        required=True,
        type=str,
        help="CSV file containing successful analyses generated by the pipeline",
    )
    parser.add_argument("-p", "--prefix", required=True, type=str, help="Prefix for the output file")

    args = parser.parse_args()

    input_path = args.input_path
    runs = args.runs
    prefix = args.prefix

    return input_path, runs, prefix


def get_read_count(read_path):

    fasta = pyfastx.Fasta(read_path, build_index=False)
    read_count = sum(1 for _ in fasta)

    return read_count


def add_markergene(root_path, run_acc, markergene_dict, markergene):

    if markergene != "ITS":

        bacterial_ssu = list(pathlib.Path(f"{root_path}/{run_acc}/sequence-categorisation").glob(f"*{markergene}*bacteria*"))
        archaeal_ssu = list(pathlib.Path(f"{root_path}/{run_acc}/sequence-categorisation").glob(f"*{markergene}*archaea*"))
        eukarya_ssu = list(pathlib.Path(f"{root_path}/{run_acc}/sequence-categorisation").glob(f"*{markergene}*eukarya*"))

        markergene_dict[markergene] = defaultdict()
        markergene_dict[markergene]["Bacteria"] = defaultdict()
        markergene_dict[markergene]["Archaea"] = defaultdict()
        markergene_dict[markergene]["Eukarya"] = defaultdict()

        markergene_dict[markergene] = add_read_count_to_markergene(markergene_dict[markergene], bacterial_ssu, "Bacteria")
        markergene_dict[markergene] = add_read_count_to_markergene(markergene_dict[markergene], archaeal_ssu, "Archaea")
        markergene_dict[markergene] = add_read_count_to_markergene(markergene_dict[markergene], eukarya_ssu, "Eukarya")
    else:
        its = list(pathlib.Path(f"{root_path}/{run_acc}/sequence-categorisation").glob("*ITS*"))
        markergene_dict["ITS"] = defaultdict()
        markergene_dict["ITS"]["Eukarya"] = defaultdict()
        markergene_dict["ITS"] = add_read_count_to_markergene(markergene_dict["ITS"], its, "Eukarya")

    return markergene_dict


def add_read_count_to_markergene(marker_gene_dict, marker, label):

    if marker:
        read_count = get_read_count(str(marker[0]))
        marker_gene_dict[label]["read_count"] = read_count
    else:
        marker_gene_dict[label]["read_count"] = 0

    return marker_gene_dict


def main():

    input_path, runs, prefix = parse_args()

    root_path = pathlib.Path(input_path)

    if not root_path.exists():
        logging.error(f"Results path does not exist: {root_path}")
        exit(1)

    runs_df = pd.read_csv(runs, names=["run", "status"])

    # Marker gene study summary
    markergene_dict = defaultdict(dict)
    for i in range(0, len(runs_df)):
        run_acc = runs_df.loc[i, "run"]
        markergene_dict[run_acc]["marker_genes"] = defaultdict(dict)
        markergene_dict[run_acc]["marker_genes"] = add_markergene(root_path, run_acc, markergene_dict[run_acc]["marker_genes"], "SSU")
        markergene_dict[run_acc]["marker_genes"] = add_markergene(root_path, run_acc, markergene_dict[run_acc]["marker_genes"], "LSU")
        markergene_dict[run_acc]["marker_genes"] = add_markergene(root_path, run_acc, markergene_dict[run_acc]["marker_genes"], "ITS")

        total_read_counts = sum([markergene["read_count"] for markergene in markergene_dict[run_acc]["marker_genes"]["SSU"].values()])
        total_read_counts += sum([markergene["read_count"] for markergene in markergene_dict[run_acc]["marker_genes"]["LSU"].values()])
        total_read_counts += sum([markergene["read_count"] for markergene in markergene_dict[run_acc]["marker_genes"]["ITS"].values()])

        for markergene in markergene_dict[run_acc]["marker_genes"].keys():
            read_count = 0
            for domain in markergene_dict[run_acc]["marker_genes"][markergene].keys():
                read_count += markergene_dict[run_acc]["marker_genes"][markergene][domain]["read_count"]
                proportion = read_count / float(total_read_counts)
                markergene_dict[run_acc]["marker_genes"][markergene][domain]["majority_marker"] = proportion >= MAJORITY_MARKER_PROPORTION

    if markergene_dict:
        with open(f"{prefix}_markergene_study_summary.json", "w") as fw:
            fw.write(json.dumps(markergene_dict, indent=4))
    else:
        logging.warning("Marker gene data empty for some reason. No summary file created.")

    # Amplified region study summary (only available if ASV results present)

    ampregion_dict = defaultdict(dict)
    for i in range(0, len(runs_df)):
        run_status = runs_df.loc[i, "status"]
        if run_status == "no_asvs":
            continue

        run_acc = runs_df.loc[i, "run"]
        ampregion_dict[run_acc]["amplified_regions"] = []

        amp_regions = sorted(list(pathlib.Path(f"{root_path}/{run_acc}/asv").glob("*S-V*/*.tsv")))

        for amp_region_path in amp_regions:
            amp_dict = defaultdict()
            amp_region = str(amp_region_path).split("/")[-2]
            marker_gene = amp_region.split("-")[0]
            amp_region = "-".join(amp_region.split("-")[1:])

            amp_region_df = pd.read_csv(amp_region_path, sep="\t")
            asv_count = len(amp_region_df)
            read_count = amp_region_df.loc[:, "count"].sum()

            amp_dict["marker_gene"] = marker_gene
            amp_dict["amplified_region"] = amp_region
            amp_dict["asv_count"] = int(asv_count)  # casting needed for JSON serialising
            amp_dict["read_count"] = int(read_count)  # casting needed for JSON serialising

            ampregion_dict[run_acc]["amplified_regions"].append(amp_dict)

    if ampregion_dict:
        with open(f"{prefix}_ampregion_study_summary.json", "w") as fw:
            fw.write(json.dumps(ampregion_dict, indent=4))
    else:
        logging.warning("No amplified region data found. No summary file created.")


if __name__ == "__main__":
    main()
