"""Utility functions for MS²Rescore report generation."""

import logging
from collections import defaultdict
from csv import DictReader
from pathlib import Path
from typing import Optional, Tuple

import pandas as pd
import psm_utils
from mokapot import LinearConfidence, LinearPsmDataset, read_fasta

from ms2rescore.exceptions import ReportGenerationError

logger = logging.getLogger(__name__)


def read_feature_names(feature_names_path: Path) -> dict:
    """Read feature names and mapping with feature generator from file."""
    feature_names = defaultdict(list)
    with open(feature_names_path) as f:
        reader = DictReader(f, delimiter="\t")
        for line in reader:
            feature_names[line["feature_generator"]].append(line["feature_name"])
    return feature_names


def get_feature_values(
    psm_list: psm_utils.PSMList, feature_names: Optional[list] = None
) -> pd.DataFrame:
    """Get feature values for all PSMs in a PSM list."""
    if not feature_names:
        feature_names = list(psm_list[0].rescoring_features.keys())
    features = pd.DataFrame(
        {fname: psm.rescoring_features[fname] for fname in feature_names} for psm in psm_list
    ).astype("float32")
    return features


def get_confidence_estimates(
    psm_list: psm_utils.PSMList, fasta_file: Optional[str] = None
) -> Tuple[LinearConfidence, LinearConfidence]:
    """Return identification confidence before and after rescoring."""
    try:
        score_before = pd.DataFrame.from_records(psm_list["provenance_data"])[
            "before_rescoring_score"
        ].astype(float)
    except KeyError as e:
        raise ReportGenerationError(
            "No `before_rescoring_score` in PSM list provenance data. Ensure that the PSM list "
            "was generated by MS²Rescore. Could not generate report."
        ) from e

    score_after = psm_list["score"]
    peptide = (
        pd.Series(psm_list["peptidoform"]).astype(str).str.replace(r"(/\d+$)", "", n=1, regex=True)
    )
    psms = pd.DataFrame({"peptide": peptide, "is_target": ~psm_list["is_decoy"]}).reset_index()
    lin_psm_dataset = LinearPsmDataset(
        psms=psms,
        target_column="is_target",
        spectrum_columns="index",
        peptide_column="peptide",
    )
    if fasta_file:
        fasta = read_fasta(fasta_file)
        lin_psm_dataset.add_proteins(fasta)

    confidence = dict()
    for when, scores in [("before", score_before), ("after", score_after)]:
        try:
            confidence[when] = lin_psm_dataset.assign_confidence(scores=scores)
        except (RuntimeError, IndexError):
            confidence[when] = None
            logger.warning("Could not assign confidence estimates for %s rescoring.", when)

    return confidence["before"], confidence["after"]
