import { ValidationError, SanitizationResultValidIp } from "./sanitize";
import { NO_HOSTNAME } from "./from-url";
export declare const RESERVED_TOP_LEVEL_DOMAINS: string[];
export declare type Label = string;
export declare enum ParseResultType {
    /**
     * This parse result is returned in case the given hostname does not adhere to [RFC 1034](https://tools.ietf.org/html/rfc1034).
     */
    Invalid = "INVALID",
    /**
     * This parse result is returned if the given hostname was an IPv4 or IPv6.
     */
    Ip = "IP",
    /**
     * This parse result is returned when the given hostname
     * - is the root domain (the empty string `""`)
     * - belongs to the top-level domain `localhost`, `local`, `example`, `invalid` or `test`
     */
    Reserved = "RESERVED",
    /**
     * This parse result is returned when the given hostname is valid and does not belong to a reserved top-level domain, but is not listed in the public suffix list.
     */
    NotListed = "NOT_LISTED",
    /**
     * This parse result is returned when the given hostname belongs to a top-level domain that is listed in the public suffix list.
     */
    Listed = "LISTED"
}
declare type ParseResultCommon<Type extends ParseResultType> = {
    /**
     * The type of the parse result. Use switch or if to distinguish between different results.
     */
    type: Type;
    /**
     * The original hostname that was passed to parseDomain().
     */
    hostname: Type extends ParseResultType.Invalid ? string | typeof NO_HOSTNAME : string;
};
export declare type ParseResultInvalid = ParseResultCommon<ParseResultType.Invalid> & {
    /**
     * An array of validation errors.
     */
    errors: Array<ValidationError>;
};
declare type ParseResultCommonValidDomain = {
    /**
     * An array of labels that were separated by a dot character in the given hostname.
     */
    labels: Array<Label>;
};
export declare type ParseResultIp = ParseResultCommon<ParseResultType.Ip> & Pick<SanitizationResultValidIp, "ipVersion">;
export declare type ParseResultReserved = ParseResultCommon<ParseResultType.Reserved> & ParseResultCommonValidDomain;
export declare type ParseResultNotListed = ParseResultCommon<ParseResultType.NotListed> & ParseResultCommonValidDomain;
declare type ParseResultListedDomains = {
    /**
     * An array of labels that belong to the subdomain. Can be empty if there was no subdomain in the given hostname.
     */
    subDomains: Array<Label>;
    /**
     * The first label that belongs to the user-controlled section of the hostname. Can be undefined if just a top-level domain was passed to parseDomain().
     */
    domain: Label | undefined;
    /**
     * An array of labels that are controlled by the domain registrar.
     */
    topLevelDomains: Array<Label>;
};
export declare type ParseResultListed = ParseResultCommon<ParseResultType.Listed> & ParseResultCommonValidDomain & ParseResultListedDomains & {
    /**
     * The parse result according to ICANN only without private top-level domains.
     */
    icann: ParseResultListedDomains;
};
export declare type ParseResult = ParseResultInvalid | ParseResultIp | ParseResultReserved | ParseResultNotListed | ParseResultListed;
/**
 * Splits the given hostname in topLevelDomains, a domain and subDomains.
 */
export declare const parseDomain: (hostname: string | typeof NO_HOSTNAME) => ParseResult;
export {};
