# Barcode Package

The barcode package provides comprehensive tools for analyzing, demultiplexing, and processing DNA sequence barcodes in high-throughput sequencing data.

## Classes

#### AnalyzeBarcodes (AnalyzeBarcodes.java)
**Purpose**: Analyzes paired barcode counts from input files generated by CountBarcodes2.

- **Core Function**: Process and statistically analyze barcode pair frequencies, detecting cross-contamination and generating comprehensive statistical reports about barcode distribution.

- **Key Features**:
  - Loads barcode statistics from input files with expected barcode specifications
  - Calculates left and right index barcode metrics
  - Generates mismatch analysis comparing total vs bad pair counts
  - Produces detailed statistical reports on barcode pair frequencies
  - Supports compressed input files (.gz, .bz2) with automatic compression handling
  - Configurable output for left, right, and pair barcode statistics
  - Performance tracking with detailed processing time and byte/line metrics

- **Usage**: 
  - Command-line tool for detailed barcode pair frequency analysis
  - Typically used in genomic sequencing to assess sample cross-contamination
  - Generates reports about unexpected barcode pair occurrences
  - Helps identify potential errors in library preparation or sequencing processes

#### Barcode (Barcode.java)
**Purpose**: Represents and manages DNA sequencing barcode sequences with advanced parsing and comparison capabilities.

**Core Function**: Provides a robust data structure for handling barcode sequences, supporting dual-indexed barcodes, sequence validation, and distance calculations.

**Key Features**:
- **Dual-Index Parsing**: Supports extracting left and right components of dual-indexed barcode sequences using various delimiter strategies
- **Sequence Validation**: Comprehensive methods to validate barcode sequences, including homopolymer detection, undefined base identification, and symbol validation
- **Distance Calculations**: 
  - Hamming distance computation between barcode sequences
  - Edit distance calculation using banded alignment
  - Supports comparisons between different sequence representations (string, byte array)
- **Count Tracking**: Maintains a count for each barcode with thread-safe increment methods
- **Spatial Tracking**: Supports tile number association for spatial information preservation
- **Flexible Comparison**: Implements Comparable interface with multi-level sorting (count, tile, name)

**Usage**: 
- DNA sequencing barcode tracking and analysis
- Demultiplexing sequencing data by indexing unique barcode sequences
- Quality control of sequencing library preparation by analyzing barcode characteristics

#### BarcodeCounter (BarcodeCounter.java)
**Purpose**: Static utility class for comprehensive barcode-related operations in sequencing data processing.

- **Core Function**: Provides static methods for counting, loading, processing, and writing barcode sequences from sequencing read files with advanced format detection and filtering capabilities.

- **Key Features**:
  - **Automatic File Format Detection**: Intelligently detects FASTQ/FASTA file formats and barcode delimiters
  - **Flexible Barcode Counting**: Supports counting barcodes from sequencing reads with optional tile information
  - **Barcode Loading**: Loads barcodes from files or comma-separated strings with format standardization
  - **Count File I/O**: Writes and reads barcode count data with minimum count filtering and sorting options
  - **Barcode Assignment Mapping**: Supports creating and writing barcode assignment lookup tables
  - **Statistical Analysis**: Generates barcode count frequency distributions and percentile calculations

- **Usage**: Primarily used in bioinformatics workflows for processing and analyzing barcode sequences from high-throughput sequencing experiments, enabling tasks like sample demultiplexing, quality control, and experimental tracking.

#### BarcodeMappingStats (BarcodeMappingStats.java)
**Purpose**: Collects and manages statistical tracking for barcode mapping in sequencing reads.

- **Core Function**: Tracks barcode frequencies and their mapping sources across sequencing reads, enabling detailed analysis of library composition and potential contamination.

- **Key Features**:
  - Dynamic frequency tracking for barcodes and their reference mappings
  - Supports merging statistics from multiple sequencing runs
  - Handles lazy initialization of barcode and mapping entries
  - Generates comprehensive output reports with read counts and proportions
  - Tracks both individual barcode frequencies and their mapping distributions
  - Supports handling of unmapped reads with "UNKNOWN" reference key

- **Usage**: Used in bioinformatics pipelines to analyze barcode distribution, library quality, and potential cross-contamination in multiplexed sequencing experiments. Particularly useful for:
  - Quantifying reads per barcode
  - Identifying unexpected mappings
  - Generating summary reports of library composition
  - Detecting potential sample mixing or contamination during sequencing

#### BarcodeStats (BarcodeStats.java)
**Purpose**: A comprehensive statistical analysis tool for processing and analyzing barcode sequences in high-throughput sequencing data.

**Core Function**: Tracks, validates, and generates detailed statistics for observed barcode sequences, supporting both single-index and dual-index barcode formats.

**Key Features**:
- **Flexible Barcode Processing**: Supports single and dual-index barcode formats with configurable delimiter and component handling
- **Detailed Frequency Analysis**: Tracks total and unique barcode counts, including expected and unexpected barcodes
- **Mismatch and Distance Metrics**: Calculates Hamming and edit distances between observed and expected barcodes
- **Homopolymer and N-Base Detection**: Identifies and tracks homopolymer sequences and barcodes containing undefined bases
- **Dual-Index Pair Validation**: Analyzes barcode pair compatibility, tracking good and bad pair combinations
- **Comprehensive Statistics Generation**: Produces detailed statistical reports including base transition patterns, match frequencies, and error characterizations
- **Flexible Input Handling**: Supports loading barcodes from files or direct string specifications with optional reverse complement processing

**Usage**: Primarily used for quality control and characterization of barcode libraries in sequencing experiments, particularly for:
- Verifying barcode design and composition
- Detecting sequencing artifacts or library preparation errors
- Generating detailed statistics on barcode distribution and matching
- Supporting complex barcode validation workflows in genomic and transcriptomic studies

#### CompareLabels (CompareLabels.java)
**Purpose**: Perform detailed statistical analysis and comparison of barcode labels across sequencing reads.

**Core Function**: Analyzes read identifiers to compare label assignments across different methods, tracking label agreement, disagreement, and assignment patterns.

**Key Features**:
  - Supports flexible delimiter-based parsing of read identifiers
  - Tracks multiple label assignment scenarios (Unknown vs Assigned, Fully Assigned)
  - Generates comprehensive statistical reports on label matching
  - Supports optional label whitelist filtering via quantification set
  - Calculates detailed metrics like relative yield, absolute yield, and contamination rates
  - Provides per-label detailed statistics with multiple count types
  - Handles paired and single-end sequencing read formats

**Usage**: Used in bioinformatics workflows to validate and compare label assignment methods in sequencing experiments, helping researchers understand label consistency and potential cross-contamination between different labeling techniques.

#### CorrelateBarcodes (CorrelateBarcodes.java)
**Purpose**: A command-line tool for analyzing sequencing reads with barcode quality correlation and statistical reporting.

**Core Function**: Processes sequencing data to correlate barcode quality with read quality, generating comprehensive quality statistics and enabling quality-based read filtering.

**Key Features**:
- **Barcode Quality Analysis**: Calculates probability-weighted average and minimum barcode quality scores
- **Quality Correlation Matrices**: Generates correlation matrices between read quality and barcode quality for Read1 and Read2
- **Adaptive Filtering**: Supports optional read filtering based on barcode quality thresholds
- **Multi-output Histograms**: Generates histograms for:
  - Barcode quality histogram
  - Average barcode quality histogram
  - Minimum barcode quality histogram
- **Parallel Processing**: Uses concurrent input/output streams for efficient read processing
- **Detailed Reporting**: Outputs comprehensive processing statistics including reads processed, bases processed, and quality-based read discard rates

**Usage**: Used in genomic sequencing workflows to assess and filter sequencing data based on barcode and read quality metrics, helping researchers improve data quality and remove low-quality reads from downstream analysis.

#### CountBarcodes (CountBarcodes.java)
**Purpose**: High-performance utility for counting and analyzing unique barcodes in sequencing read files.

**Core Function**: Extracts barcodes from read headers, performs frequency counting, and generates detailed barcode statistics using multithreaded processing.

**Key Features**:
- **Concurrent Read Processing**: Utilizes multithreaded I/O for efficient barcode extraction from FASTQ files
- **Flexible Barcode Filtering**: Supports counting of undefined and defined barcodes with configurable filtering
- **Distance Metrics**: Calculates Hamming and edit distances between observed and expected barcodes
- **Validation Support**: Allows specification of expected and valid barcode sets
- **Performance Tracking**: Provides detailed processing statistics including reads processed, bases processed, and execution time
- **Comprehensive Output**: Generates frequency table with barcode counts, distances, and validation status

**Usage**: Used in bioinformatics workflows to analyze barcode diversity and distribution in sequencing experiments, particularly for:
- Demultiplexing sequencing libraries
- Quality control of barcode-tagged samples
- Tracking sample-specific read frequencies in complex mixtures

#### CountBarcodes2 (CountBarcodes2.java)
**Purpose**: Advanced barcode counting and analysis tool for processing Illumina sequencing data with support for complex barcode tracking and PCR matrix-based quantification.

**Core Function**: Analyzes FASTQ files to count, categorize, and validate barcodes, providing detailed statistical insights into barcode distributions, contamination, and sequencing quality.

**Key Features**:
- **Flexible Barcode Processing**: Supports single and dual-indexed barcode schemes with configurable delimiter and length detection
- **PCR Matrix Analysis**: Implements sophisticated PCR matrix-based barcode assignment and error correction using probabilistic methods
- **Contamination Detection**: Calculates contamination metrics including parts per million (PPM) and yield percentages
- **Comprehensive Reporting**: Generates multiple output files including barcode counts, transition probabilities, assignment maps, and accuracy metrics
- **Tile-Specific Tracking**: Optional per-tile barcode analysis for advanced quality control
- **Multi-File Support**: Can process multiple input files simultaneously with parallel gzip support
- **Quantification Modes**: Supports both source and sink quantification strategies for precise barcode tracking

**Usage**: 
- Primarily used in genomics and sequencing pipelines for:
  - Barcode quality control in multiplexed sequencing experiments
  - Detecting and quantifying sample cross-contamination
  - Validating barcode assignment accuracy in complex library preparation workflows

**Example Command**:
```bash
countbarcodes2.sh pcrmatrix type=hdist in=sample.fq.gz ow expected=expected.txt -Xmx31g t=32 outcontam=contam.txt quantset=quantset.txt
```

#### DemuxClient (DemuxClient.java)
**Purpose**: Sends and receives demultiplexing data to/from a demux server, supporting barcode encoding and decoding for sequencing workflows.

**Core Function**: Manages client-side communication with a demultiplexing server, encoding and decoding barcode data using different encoding strategies (raw and A48 encoding).

**Key Features**:
- **Flexible Barcode Encoding**: Supports both single and dual-index barcode formats with configurable encoding methods (raw and A48)
- **Server Communication**: Sends encoded barcode data chunks to a demux server and processes server responses
- **Advanced Decoding**: Implements sophisticated decoding algorithms for A48 and raw encoding, including delta encoding and error handling
- **Robust Error Checking**: Validates server responses, checks HTTP status codes, and manages error states
- **Debugging Support**: Optional file output for sent data chunks to support troubleshooting
- **Encoding Validation**: Ensures unique barcode assignments and proper line termination

**Usage**: Used in bioinformatics sequencing workflows to manage barcode demultiplexing, allowing precise mapping of observed barcodes to assigned barcodes across different encoding strategies. Primarily utilized in next-generation sequencing (NGS) library preparation and sample tracking processes.

#### DemuxData (DemuxData.java)
**Purpose**: A data container and utility class for encoding, decoding, and managing dual-index barcode data with advanced compression techniques.

**Core Function**: Manages encoding and decoding of barcode information using efficient delta compression and base-5 encoding strategies for genomic sequencing workflows.

**Key Features**:
- **Dual-Index Barcode Handling**: Supports both single and dual-index barcode configurations with configurable component lengths and delimiters
- **Delta Compression**: Advanced encoding that minimizes data transmission by compressing similar barcode entries using delta techniques for counts and barcode codes
- **Flexible Encoding Modes**: Supports multiple encoding strategies including RAW, HEX, and A48 (base-48) encoding methods
- **Polymorphic Parsing**: Robust parsing of configuration flags, expected barcodes, and count data from serialized byte chunks
- **Base Conversion Utilities**: Specialized methods for converting between DNA bases (ACGTN) and numeric representations
- **High-Performance Serialization**: Efficient byte-level encoding and decoding with configurable buffer sizes and transmission strategies
- **Quality Control Parameters**: Integrated support for setting maximum Hamming distances, minimum ratios, and probability thresholds for barcode validation

**Usage**: Primarily used in genomic sequencing pipelines for managing barcode data during demultiplexing processes, enabling efficient storage and transmission of barcode information with minimal overhead. Ideal for processing large-scale sequencing experiments with complex barcode configurations.

#### DemuxServer (DemuxServer.java)
**Purpose**: HTTP server for handling barcode demultiplexing queries with probabilistic sequence assignment.

- **Core Function**: Provides a web server endpoint for processing barcode sequencing data, using probabilistic algorithms to map observed barcodes to expected reference sequences.

- **Key Features**:
  - Configurable HTTP/HTTPS server with custom port and security settings
  - Probabilistic PCR matrix-based barcode demultiplexing
  - Multiple request handlers for different server endpoints (/demux, /stats, /kill)
  - Performance tracking with atomic query and timing counters
  - Supports single and dual-barcode sequencing strategies
  - Flexible encoding methods for barcode representation (RAW and A48 compression)
  - Secure remote shutdown mechanism with configurable kill code

- **Usage**: Deployed in bioinformatics pipelines to process high-throughput sequencing data, particularly for resolving ambiguous barcode assignments in complex sequencing experiments. Enables remote, scalable barcode demultiplexing through a web service interface.

#### LegacyFileWriter (LegacyFileWriter.java)
**Purpose**: Generates legacy statistics files for demultiplexing processes with detailed barcode and sequencing quality metrics.

**Core Function**: Writes comprehensive CSV output files tracking barcode assignment, index hopping, quality metrics, and demultiplexing statistics for sequencing runs.

**Key Features**:
- Writes detailed CSV files tracking top unknown barcodes with read counts and percentages
- Generates index hopping analysis for dual-indexed libraries
- Creates per-sample quality metrics with yield and Q30 statistics
- Produces demultiplexing statistics including mismatch tolerance breakdown
- Tracks per-tile demultiplexing statistics for spatial quality assessment
- Supports loading sample mapping from CSV/TSV files
- Handles both single and dual-indexed barcode configurations

**Usage**: Used in bioinformatics sequencing pipelines to generate comprehensive legacy-format statistics reports during barcode demultiplexing, providing backward compatibility and detailed run diagnostics.

#### NovaDemux (NovaDemux.java)
**Purpose**: A high-performance demultiplexing utility for processing barcoded sequencing reads using PCRMatrix algorithms.

**Core Function**: Automatically routes sequencing reads to their corresponding sample files based on barcode identification, with advanced parsing modes and optional spike-in detection.

**Key Features**:
- **Flexible Barcode Parsing**: Supports multiple extraction modes including barcode, prefix, suffix, delimiter-based, and full header matching
- **PCRMatrix Barcode Assignment**: Uses sophisticated probabilistic algorithms for accurate barcode-to-sample mapping
- **Spike-in Detection**: Optional mapping of unexpected or unassigned reads to a reference genome (e.g., PhiX control)
- **Performance Optimization**: Configurable compression levels, multi-threaded processing, and memory-efficient read handling
- **Comprehensive Statistics**: Generates detailed reports on read processing, including input/output read counts and cardinality tracking
- **Legacy File Generation**: Can produce Illumina-compatible output files for compatibility with existing pipelines
- **Adaptive Processing**: Supports various input formats (FASTQ, SAM/BAM) and configurable output strategies

**Usage**: Primarily used in bioinformatics workflows for demultiplexing next-generation sequencing data, particularly for separating multiplexed sequencing libraries into individual sample files based on barcodes.

#### PCRMatrix (PCRMatrix.java)
**Purpose**: Tracks and analyzes barcode mismatches by position, primarily used for demultiplexing genomic sequencing data.

**Core Function**: Manages a sophisticated position-specific mismatch matrix that enables precise barcode assignment and error analysis, supporting both single and dual-index barcode configurations.

**Key Features**:
- **Flexible Indexing**: Supports single and dual-index barcode configurations with configurable delimiter
- **Distance Calculation**: Implements multiple distance calculation methods (Hamming distance, probabilistic)
- **Contamination Handling**: Can add poly-nucleotide contamination references (A, C, G, T)
- **Comprehensive Tracking**: Maintains detailed counts of base mismatches at each position
- **Assignment Statistics**: Calculates assignment metrics like assigned fraction, expected fraction, and chimeric fraction
- **Threadsafe Operations**: Uses ReadWriteLock for concurrent access and modification
- **Barcode Matching**: Implements advanced closest barcode finding algorithms with clearzone constraints

**Usage**: Used in genomic sequencing workflows for:
- Demultiplexing sequencing reads by barcode
- Error analysis and quality control of barcode assignments
- Tracking base-level mismatches in indexing sequences
- Supporting complex barcode matching strategies in multi-sample sequencing experiments

#### PCRMatrixHDist (PCRMatrixHDist.java)
**Purpose**: Tracks and analyzes bar code mismatches by position using split barcodes with Hamming distance calculations.

- **Core Function**: Performs barcode assignment and distance analysis using Hamming distance metrics, supporting both single and dual barcode scenarios with flexible mismatch thresholds.

- **Key Features**:
  - Multi-threaded barcode distance calculation with stride-based work partitioning
  - Configurable maximum Hamming distance threshold (maxHDist0) for barcode matching
  - Support for single and dual barcode distance calculations
  - Thread-safe accumulation of substitution matrix data
  - Dynamic thread allocation based on workload complexity
  - Detailed tracking of total counted, assigned, and expected barcodes

- **Usage**: 
  - Used in bioinformatics and genomics for precise barcode assignment and mismatch analysis
  - Helps identify and categorize barcodes based on their Hamming distance from reference sequences
  - Supports both single-threaded and multi-threaded processing of barcode lists
  - Enables filtering and mapping of barcodes with configurable distance thresholds

#### TagAndMerge (TagAndMerge.java)
**Purpose**: Processes demultiplexed sequencing lane files by extracting and appending barcodes to read headers, then consolidating reads into a single output file.

- **Core Function**: Extracts barcodes from input filenames, modifies read headers with barcode tags, and merges reads from multiple input files into a single output stream.

- **Key Features**:
  - Barcode Extraction: Automatically parses barcodes from input filenames using `Barcode.parseBarcodeFromFname()`
  - Header Modification: Appends barcode tags to read headers, with optional header compression for space efficiency
  - Flexible Input Handling: Supports multiple input files, paired-end and single-end reads
  - Optional Read Filtering: Can drop R2 reads with `dropR2` parameter
  - Optional Read Trimming: Supports trimming reads to a specified length using `trimLen`
  - Barcode Remapping: Allows character-level remapping of barcodes (e.g., converting hyphens to plus signs)
  - Barcode Tracking: Maintains a unique set of encountered barcodes

- **Usage**: Primarily used in bioinformatics sequencing pipelines to consolidate and tag reads from demultiplexed sequencing lanes, enabling downstream analysis with clear barcode metadata.

#### Transition (Transition.java)
**Purpose**: Represents a nucleotide base transition with position, reference base, query base, and frequency tracking.

**Core Function**: Provides a compact, efficient encoding and decoding mechanism for genetic sequence base changes, enabling statistical analysis of nucleotide transitions.

**Key Features**:
- **Compact Encoding**: Bit-packs position, reference base, and query base into a single integer for efficient storage and comparison
- **Statistical Tracking**: Captures transition frequency through a count field for quantitative genetic analysis
- **Flexible Base Representation**: Supports A, C, G, T, and N bases with numeric encoding
- **Sorting Mechanism**: Implements Comparable interface with multi-level sorting prioritizing frequency, position, and base changes
- **Export Capability**: Provides ByteBuilder-based method for tab-delimited transition data export
- **Reversible Encoding**: Supports both encode() and decode() methods for transforming transition data

**Usage**: Used in genetic sequence analysis to track and analyze base changes, particularly in variant calling, mutation tracking, and comparative genomic studies. Helps researchers understand nucleotide transition patterns by providing a compact representation of base changes with associated frequencies.

### Probabilistic Classes (prob/ subdirectory)

#### PCRMatrixProb (prob/PCRMatrixProb.java)
**Purpose**: Tracks probabilistic data about barcode mismatches by position, supporting both single and split barcodes with advanced error probability modeling.

**Core Function**: Probabilistic barcode assignment using position-specific error probability matrices to identify the most likely expected barcode from observed sequences with potential errors.

**Key Features**:
- **Position-Specific Error Modeling**: Calculates error probabilities for each base at each position in barcode sequences
- **Dual Barcode Support**: Handles both single and split (two-segment) barcodes with independent probability scoring
- **Multi-Threaded Processing**: Enables parallel barcode assignment for large datasets using worker threads
- **Adaptive Matching**: Uses relaxed thresholds for low-count barcodes to improve sensitivity
- **Frequency-Weighted Probability**: Incorporates barcode frequency into matching probability calculations
- **Hybrid Matching**: Supports fast exact and near-exact matching before full probability calculation
- **Comprehensive Error Tracking**: Maintains detailed error count matrices for debugging and model refinement

**Usage**: Primarily used in bioinformatics for:
- Barcode error correction in sequencing data
- Mapping observed barcode sequences to expected references
- Learning and tracking position-specific sequencing error patterns
- Processing large barcode libraries with probabilistic assignment methods

#### PCRMatrixProbAbstract (prob/PCRMatrixProbAbstract.java)
**Purpose**: Abstract probabilistic matrix implementation for maximum likelihood barcode assignment in sequencing data.

- **Core Function**: Implements expectation-maximization (EM) algorithm to learn base transition probabilities and assign barcodes using position-specific error models.

- **Key Features**:
  - 3D probability matrix tracking [position][called_base][reference_base] base transition likelihoods
  - Iterative refinement of probability estimates through EM algorithm (4-5 typical iterations)
  - Dynamic single-threaded and multi-threaded count population strategies
  - Probability-based barcode similarity scoring with position-specific error modeling
  - Laplace smoothing to prevent zero-probability assignments
  - Configurable parameters for base calling accuracy, error thresholds, and frequency estimation

- **Usage**: Used in probabilistic barcode assignment workflows for sequencing data, particularly in situations requiring:
  - High-confidence barcode matching across complex sequencing libraries
  - Handling systematic sequencing errors
  - Probabilistic assignment of barcodes with minimal false positives

#### PCRMatrixProb_old (prob/PCRMatrixProb_old.java)
**Purpose**: Legacy probabilistic matrix for tracking and assigning barcode mismatches during demultiplexing, with advanced probabilistic scoring and refinement.

**Core Function**: Performs probabilistic barcode assignment using an expectation-maximization algorithm that iteratively refines position-specific base substitution probabilities.

**Key Features**:
- **Probabilistic Scoring**: Computes assignment probabilities using a 3D matrix tracking P(observed_base|reference_base) at each position
- **Multi-Pass Refinement**: Supports iterative refinement with configurable passes, distance thresholds, and probability filters
- **Multi-Threaded Processing**: Enables parallel barcode assignment for large datasets using stride-based thread partitioning
- **Unexpected Barcode Handling**: Generates and weights unexpected barcode combinations from legitimate barcode segments
- **Frequency Normalization**: Adjusts barcode frequencies to prevent over-assignment to rare barcodes using pseudocount spoofing
- **Competitive Assignment**: Applies stringent ratio and probability filters to prevent ambiguous assignments

**Usage**: Used in bioinformatics demultiplexing workflows to accurately assign sequencing reads to their most likely barcode origin, accounting for base substitution errors and PCR recombination events.

#### PCRMatrixTile (prob/PCRMatrixTile.java)
**Purpose**: Probabilistic barcode matching system that performs tile-based error probability analysis for precise barcode sequence classification.

**Core Function**: Implements advanced probabilistic alignment and classification of barcode sequences using position-specific error matrices, supporting both single-segment and dual-segment barcode matching strategies.

**Key Features**:
- **Tile-Specific Error Modeling**: Computes probabilistic scores using position-specific error probability matrices that capture base-call error patterns unique to each tile
- **Multi-Threaded Processing**: Supports parallel processing of large barcode collections with round-robin work distribution and thread-local count matrices
- **Flexible Matching Strategies**: 
  - Handles single-segment and dual-segment (split) barcode sequences
  - Applies adaptive matching thresholds based on query sequence count
  - Implements hybrid distance checking with probabilistic scoring
- **Frequency-Weighted Scoring**: Incorporates barcode frequency into probabilistic matching to improve classification accuracy
- **Chimera Detection**: Tracks assignments to expected vs. chimeric reference barcodes
- **Dynamic Probability Matrix Management**: Lazily allocates and manages tile-specific probability matrices to conserve memory
- **Comprehensive Assignment Reporting**: Generates detailed assignment maps with probabilistic confidence scores

**Usage**: Used in high-throughput sequencing and barcode analysis pipelines to accurately classify and assign barcode sequences, particularly in scenarios with complex error patterns across sequencing tiles. Ideal for processing large numbers of molecular barcodes with position-specific error characteristics.

### Stub Classes (stub/ subdirectory)

#### PCRMatrixProb (stub/PCRMatrixProb.java)
**Purpose**: Stub implementation of a probabilistic PCR matrix for barcode processing, inheriting from PCRMatrixProbAbstract.

**Core Function**: Provides a placeholder implementation for PCR matrix probability calculations in the barcode processing workflow, with no active computational logic in this stub version.

**Key Features**:
- Extends PCRMatrixProbAbstract abstract base class
- Minimal constructor accepting four parameters
- Part of the barcode processing package's probabilistic matrix infrastructure
- Serves as a potential placeholder for future implementation of PCR matrix probability calculations

**Usage**: Used as a potential extension point in the barcode processing system, allowing for future implementation of specialized PCR matrix probability calculations without committing to a specific algorithm in the current version.

#### PCRMatrixProbAbstract (stub/PCRMatrixProbAbstract.java)
**Purpose**: Abstract base class for probabilistic PCR (Polymerase Chain Reaction) matrix implementations in the barcode processing system.

**Core Function**: Provides a skeletal implementation of PCR matrix probability calculations with default no-op method implementations for key matrix operations.

**Key Features**:
- Abstract superclass extending PCRMatrix with minimal default method behaviors
- Supports static and instance-level parsing methods for PCR matrix configuration
- Implements placeholder methods for barcode assignment, count population, and probability calculations
- Provides framework for deriving concrete PCR matrix probability implementations
- Supports clientside processing with `clientside()` method
- Offers methods for finding closest barcodes and converting to byte representations
- Allows refinement of barcode collections with lightweight default implementation

**Usage**: Used as a base class for creating specialized probabilistic PCR matrix processing classes in barcode analysis workflows, particularly in DNA/RNA sequencing and barcode tracking applications. Developers can extend this abstract class to implement specific probabilistic matrix calculation strategies for barcode processing and classification.

#### PCRMatrixTile (stub/PCRMatrixTile.java)
**Purpose**: Represents a probabilistic tile-based matrix for PCR (Polymerase Chain Reaction) calculations within the barcode processing framework.

**Core Function**: A stub implementation of a PCR matrix tile with placeholder functionality for probabilistic matrix operations in barcode processing.

**Key Features**:
- Extends PCRMatrixProbAbstract abstract class for PCR matrix probability calculations
- Provides a constructor with parameters for matrix tile configuration
- Part of the barcode processing infrastructure for probabilistic matrix management
- Designed to support flexible PCR matrix representations

**Usage**: Used in barcode processing workflows to create probabilistic matrix tiles for computational biology and genetic sequencing analysis, particularly in scenarios requiring dynamic matrix-based probabilistic calculations.
