# cython: language_level=3

from pathlib import Path
from typing import List
from fgpyo.sam import CigarOp
from fgpyo.sam import CigarElement
from fgpyo.sam import Cigar

from libc.stdint cimport uint8_t
from libc.stdlib cimport calloc, free
from posix.stdlib cimport srand48
from libc.string cimport strncpy
from pysam import FastxRecord, AlignedSegment
from pybwa.libbwaindex cimport force_bytes
from pybwa.libbwaindex cimport BwaIndex
from pysam.libcalignedsegment cimport makeAlignedSegment
from dataclasses import dataclass
from libc.errno cimport errno

__all__ = [
    "BwaAlnOptions",
    "BwaAln",
    "AuxHit",
    "to_xa_hits",
]

cpdef bint _set_bwa_aln_verbosity(int level):
    """Set the BWA C-API verbosity, returning True if changed, false otherwise."""
    global bwa_verbose
    retval = level != bwa_verbose
    bwa_verbose = level
    return retval


cdef class BwaAlnOptions:
    """The container for options for :class:`pybwa.BwaAln`.
    
    Args:
        max_mismatches: :code:`-n <int>`
        max_gap_opens: :code:`-o <int>`
        max_gap_extensions: :code:`-e <int>`
        min_indel_to_end_distance: :code:`-i <int>`
        max_occurrences_for_extending_long_deletion: :code:`-d <int>`
        seed_length: :code:`-l <int>`
        max_mismatches_in_seed: :code:`-k <int>`
        mismatch_penalty: :code:`-M <int>`
        gap_open_penalty: :code:`-O <int>`
        gap_extension_penalty: :code:`-E <int>`
        stop_at_max_best_hits: :code:`-R <int>`
        max_hits: :code:`bwa samse -n <int>`
        log_scaled_gap_penalty: :code:`-L`
        find_all_hits: :code:`-N`
        with_md: output the MD to each alignment in the XA tag, otherwise use :code:`"."`
        max_entries: :code:`-m`
        threads: the number of threads to use
    """

    def __init__(self,
                 max_mismatches: int = -1,
                 max_gap_opens: int = 1,
                 max_gap_extensions: int = 6,
                 min_indel_to_end_distance: int = 5,
                 max_occurrences_for_extending_long_deletion: int = 10,
                 seed_length: int = 32,
                 max_mismatches_in_seed: int = 2,
                 mismatch_penalty: int = 3,
                 gap_open_penalty: int = 11,
                 gap_extension_penalty: int = 4,
                 stop_at_max_best_hits: int = 30,
                 max_hits: int = 3,
                 log_scaled_gap_penalty: bool = False,
                 find_all_hits: bool = False,
                 with_md: bool = False,
                 max_entries: int = 2000000,
                 threads: int = 1
                 ):
        if max_mismatches is not None:
            self.max_mismatches = max_mismatches
        if max_gap_opens is not None:
            self.max_gap_opens = max_gap_opens
        if max_gap_extensions is not None:
            self.max_gap_extensions = max_gap_extensions
            # this mimics the behavior in bwtaln.c, where the default # of gap extensions is 6
            # and BWA_MODE_GAPE is set.  But when a value is given on the command line, then
            # BWA_MODE_GAPE is unset.  So in this constructor if the default value is given, then
            # we keep BWA_MODE_GAPE set, and if this parameter is set via the setter, we unset it.
            if max_gap_extensions == 6:
                self._delegate.mode |= BWA_MODE_GAPE
        if min_indel_to_end_distance is not None:
            self.min_indel_to_end_distance = min_indel_to_end_distance
        if max_occurrences_for_extending_long_deletion is not None:
            self.max_occurrences_for_extending_long_deletion = max_occurrences_for_extending_long_deletion
        if seed_length is not None:
            self.seed_length = seed_length
        if max_mismatches_in_seed is not None:
            self.max_mismatches_in_seed = max_mismatches_in_seed
        if mismatch_penalty is not None:
            self.mismatch_penalty = mismatch_penalty
        if gap_open_penalty is not None:
            self.gap_open_penalty = gap_open_penalty
        if gap_extension_penalty is not None:
            self.gap_extension_penalty = gap_extension_penalty
        if stop_at_max_best_hits is not None:
            self.stop_at_max_best_hits = stop_at_max_best_hits
        if max_hits is not None:
            self.max_hits = max_hits
        if log_scaled_gap_penalty is not None:
            self.log_scaled_gap_penalty = log_scaled_gap_penalty
        if find_all_hits is not None:
            self.find_all_hits = find_all_hits
        if with_md is not None:
            self.with_md = with_md
        if max_entries is not None:
            self.max_entries = max_entries
        if threads is not None:
            self.threads = threads

    def __cinit__(self):
        self._delegate = gap_init_opt()

    def __dealloc__(self):
        free(self._delegate)

    cdef gap_opt_t* gap_opt(self):
        """Returns the options struct to use with the bwa C library methods"""
        return self._delegate

    property max_mismatches:
        """:code:`bwa aln -n <int>`"""
        def __get__(self) -> int:
            return self._delegate.max_diff
        def __set__(self, value: int):
            if value >= 0:
                self._delegate.fnr = -1.0
                self._delegate.max_diff = value
            else:
                self._delegate.max_diff = -1
                self._delegate.fnr = 0.04

    property max_gap_opens:
        """:code:`bwa aln -o <int>`"""
        def __get__(self) -> int:
            return self._delegate.max_gapo
        def __set__(self, value: int):
           self._delegate.max_gapo = value

    property max_gap_extensions:
        """:code:`bwa aln -e <int>`"""
        def __get__(self) -> int:
            return self._delegate.max_gape
        def __set__(self, value: int):
            self._delegate.max_gape = value
            # the BWA_MODE_GAPE mode indicates that gap extensions
            # should count against the maximum # of mismatches
            if self._delegate.max_gape > 0:
                self._delegate.mode &= ~BWA_MODE_GAPE
            else:
                self._delegate.mode |= BWA_MODE_GAPE

    property min_indel_to_end_distance:
        """:code:`bwa aln -i <int>`"""
        def __get__(self) -> int:
            return self._delegate.indel_end_skip
        def __set__(self, value: int):
           self._delegate.indel_end_skip = value

    property max_occurrences_for_extending_long_deletion:
        """:code:`bwa aln -d <int>`"""
        def __get__(self) -> int:
            return self._delegate.max_del_occ
        def __set__(self, value: int):
           self._delegate.max_del_occ = value

    property seed_length:
        """:code:`bwa aln -l <int>`"""
        def __get__(self) -> int:
            return self._delegate.seed_len
        def __set__(self, value: int):
           self._delegate.seed_len = value

    property max_mismatches_in_seed:
        """:code:`bwa aln -k <int>`"""
        def __get__(self) -> int:
            return self._delegate.max_seed_diff
        def __set__(self, value: int):
           self._delegate.max_seed_diff = value

    property mismatch_penalty:
        """:code:`bwa aln -M <int>`"""
        def __get__(self) -> int:
            return self._delegate.s_mm
        def __set__(self, value: int):
           self._delegate.s_mm = value

    property gap_open_penalty:
        """:code:`bwa aln -O <int>`"""
        def __get__(self) -> int:
            return self._delegate.s_gapo
        def __set__(self, value: int):
           self._delegate.s_gapo = value

    property gap_extension_penalty:
        """:code:`bwa aln -E <int>`"""
        def __get__(self) -> int:
            return self._delegate.s_gape
        def __set__(self, value: int):
           self._delegate.s_gape = value

    property stop_at_max_best_hits:
        """:code:`bwa aln -R <int>`"""
        def __get__(self) -> int:
            return self._delegate.max_top2
        def __set__(self, value: int):
           self._delegate.max_top2 = value

    property max_hits:
        """:code:`bwa samse -n <int>`"""
        def __get__(self) -> int:
            return self._max_hits
        def __set__(self, value: int):
           self._max_hits = value

    property log_scaled_gap_penalty:
        """:code:`bwa aln -L`"""
        def __get__(self) -> bool:
            return self._delegate.mode & BWA_MODE_LOGGAP > 0
        def __set__(self, value: bool):
            if value:
                self._delegate.mode |= BWA_MODE_LOGGAP
            else:
                self._delegate.mode &= ~BWA_MODE_LOGGAP

    property find_all_hits:
        """:code:`bwa aln -N`"""
        def __get__(self) -> bool:
            return self._delegate.mode & BWA_MODE_NONSTOP > 0
        def __set__(self, value: bool):
            if value:
                self._delegate.mode |= BWA_MODE_NONSTOP
                self.stop_at_max_best_hits = 0x7fffffff
            else:
                self._delegate.mode &= ~BWA_MODE_NONSTOP

    property with_md:
        """:code:`bwa samse -d`
        
        Output the MD to each alignment in the XA tag, otherwise use :code:`"."`.
        """
        def __get__(self) -> bool:
            return self._with_md
        def __set__(self, value: bool):
           self._with_md = value

    property max_entries:
        """:code:`bwa aln -m`"""
        def __get__(self) -> int:
            return self._delegate.max_entries
        def __set__(self, value: int):
            self._delegate.max_entries = value

    property threads:
        """:code:`bwa aln -t`"""
        def __get__(self) -> int:
            return self._delegate.n_threads
        def __set__(self, value: int):
            self._delegate.n_threads = value


cdef class BwaAln:
    """The class to align reads with :code:`bwa aln`."""

    cdef BwaIndex _index

    def __init__(self, prefix: str | Path | None = None, index: BwaIndex | None = None):
        """Constructs the :code:`bwa aln` aligner.

        One of :code:`prefix` or :code:`index` must be specified.

        Args:
            prefix: the path prefix for the BWA index (typically a FASTA)
            index: the index to use
        """
        if prefix is not None:
            assert Path(prefix).exists()
            self._index = BwaIndex(prefix=prefix)
        elif index is not None:
            self._index = index
        else:
            raise ValueError("Either prefix or index must be given")

        bwase_initialize()

    def align(self, queries: List[FastxRecord] | List[str] | None = None, opt: BwaAlnOptions | None = None) -> List[AlignedSegment]:
        """Align one or more queries with :code:`bwa aln`.

        Args:
            queries: the queries to align
            opt: the alignment options | None = None, or None to use the default options

        Returns:
            one alignment (:class:`~pysam.AlignedSegment`) per query
            :code:`List[List[AlignedSegment]]`.
        """
        if len(queries) == 0:
            return []
        elif isinstance(queries[0], str):
            queries = [
                FastxRecord(name=f"read.{i}", sequence=sequence)
                for i, sequence in enumerate(queries)
            ]
        opt = BwaAlnOptions() if opt is None else opt

        return self._calign(opt,  queries)

    cdef _copy_seq(self, q: FastxRecord, bwa_seq_t* s, is_comp: bool):
        seq_len = len(q.sequence)
        s.len = seq_len
        s.clip_len = seq_len
        s.full_len = seq_len
        s.seq = <uint8_t *> calloc(sizeof(uint8_t), seq_len + 1)
        s.rseq = <uint8_t *> calloc(sizeof(uint8_t), seq_len + 1)

        # convert char into int
        for i, base in enumerate(q.sequence):
            s.seq[i] = nst_nt4_table[ord(base)]
            s.rseq[i] = nst_nt4_table[ord(base)]
        s.seq[seq_len] = b'\0'

        # qualities
        if q.quality is None:
            s.qual = NULL
        else:
            s.qual = <uint8_t *> calloc(sizeof(uint8_t), seq_len + 1)
            qual_str = force_bytes(q.quality)
            for i in range(seq_len):
                s.qual[i] = qual_str[i]
            s.qual[seq_len] = b'\0'

        # use seq_reverse from bwaseqio.c
        seq_reverse(seq_len, s.seq,
                    0)  #  // *IMPORTANT*: will be reversed back in bwa_refine_gapped()
        seq_reverse(seq_len, s.rseq, 1 if is_comp else 0)

        s.name = <char *> calloc(sizeof(char), len(q.name) + 1)
        strncpy(s.name, force_bytes(q.name), len(q.name))
        s.name[len(q.name)] = b'\0'

    cdef _calign(self, opt: BwaAlnOptions, queries: List[FastxRecord]):
        cdef bwa_seq_t* seqs
        cdef bwa_seq_t* s
        cdef char* s_char
        cdef gap_opt_t* gap_opt
        cdef sam_hdr_t *h
        cdef kstring_t hdr_str
        cdef bam1_t **bams
        cdef bwa_hits_t * c_hits

        hdr_str.l = hdr_str.m = 0
        hdr_str.s = NULL
        bwa_format_sam_hdr(self._index.bns(), NULL, &hdr_str)
        h = sam_hdr_parse(hdr_str.l, hdr_str.s)
        h.l_text = hdr_str.l
        h.text = hdr_str.s

        gap_opt = opt.gap_opt()

        # copy FastqProxy into bwa_seq_t
        num_seqs = len(queries)
        seqs = <bwa_seq_t*>calloc(sizeof(bwa_seq_t), num_seqs)
        for i in range(num_seqs):
            self._copy_seq(queries[i], &seqs[i], (opt._delegate.mode & BWA_MODE_COMPREAD) != 0)
            seqs[i].tid = -1

        # this is `bwa aln` and `bwa se` combined to be multi-threaded.
        bams = bwa_aln_and_samse(self._index.bns(), self._index.bwt(),  self._index.pac(), h, num_seqs, seqs, gap_opt, opt.max_hits, opt.with_md)

        # create the AlignedSegment
        recs = []
        for i in range(num_seqs):
            rec = makeAlignedSegment(bams[i], self._index.header)
            bam_destroy1(bams[i])
            recs.append(rec)
        free(bams)

        bwa_free_read_seq(num_seqs, seqs)
        free(hdr_str.s)

        return recs

    cpdef reinitialize_seed(self):
        """Re-initializes the random seed."""
        srand48(self._index.bns().seed)

cdef _to_cigar(uint32_t n_cigar, uint32_t *cigar):
    cdef uint32_t code, length
    elements = []
    cdef uint32_t k = 0
    while k < n_cigar:
        code = cigar[k] & BAM_CIGAR_MASK
        length = cigar[k] >> BAM_CIGAR_SHIFT
        operator = CigarOp.from_code(code)
        elements.append(CigarElement(length, operator))
        k += 1
    return Cigar(tuple(elements))

cpdef to_xa_hits(rec: AlignedSegment | str | bytes, max_hits: int | None = None):
    """Parses the value of the XA SAM tag, returning a list of :code:`AuxHit`\s.
    
    Args:
        rec: one of an :code:`AlignedSegment`, string, or bytes.  If a string or bytes are provided,
            the _value_ of the XA SAM tag must be provided (i.e. not including the leading 
            :code:`XA:Z:`).
    
    Returns:
        An empty list if a provided :code:`AlignedSegment` has no "XA" tag, otherwise a list of
        :code:`AuxHit`\s parsed from the XA SAM tag.
    
    Raises:
        A :code:`ValueError` if the XA SAM tag could not be parsed.
    """
    if isinstance(rec, AlignedSegment):
        if not rec.has_tag("XA"):
            return []
        value = force_bytes(rec.get_tag("XA"))
    elif isinstance(rec, str):
        value = force_bytes(rec)
    else:
        value = rec
    max_hits = 0 if max_hits is None else max_hits
    c_hits = parse_xa(value, max_hits)
    if c_hits == NULL:
        raise ValueError(f"Could not parse XA tag (error: {errno}: {rec}")
    hits: list[AuxHit] = [
        AuxHit(
            refname=c_hits.hits[j].refname[:c_hits.hits[j].refname_len].decode("ascii"),
            start=c_hits.hits[j].pos,
            negative=c_hits.hits[j].negative == '-',
            cigar=_to_cigar(
                n_cigar=c_hits.hits[j].n_cigar, cigar=c_hits.hits[j].cigar
            ),
            edits=c_hits.hits[j].edits,
            md=None if c_hits.hits[j].md == NULL else c_hits.hits[j].md[:c_hits.hits[j].md_len].decode("ascii"),
            rest=None if c_hits.hits[j].rest == NULL else c_hits.hits[j].rest[:c_hits.hits[j].rest_len].decode("ascii"),
        )
        for j in range(c_hits.n)
    ]
    for j in range(c_hits.n):
        free(c_hits.hits[j].refname)
        free(c_hits.hits[j].md)
        free(c_hits.hits[j].rest)
    free(c_hits.hits)
    free(c_hits)

    return hits


@dataclass(frozen=True)
class AuxHit:
    """Represents a single hit or alignment of a sequence to a location in the genome.

    Typically, this is parsed from the XA SAM tag.

    Attributes:
        refname: the reference name of the hit.
        start: the start position of the hit (1-based inclusive).
        negative: whether the hit is on the negative strand.
        cigar: the cigar returned by BWA.
        edits: the number of edits (mismatches only, no indels) between the read and the reference.
            Calculated as the NM value in the XA SAM tag minus the number of indel bases from
            the cigar.
        md: if present, the MD value that's appended to the entry in the XA SAM tag.
        rest: if present, any string that's appended to the entry in the XA SAM tag (after the MD).
    """
    refname: str
    start: int
    negative: bool
    cigar: Cigar
    edits: int
    md: str | None = None
    rest: str | None = None

    @property
    def mismatches(self) -> int:
        """The number of mismatches for the hit (not including indels)."""
        indel_sum = sum(elem.length for elem in self.cigar.elements if elem.operator.is_indel)
        if indel_sum > self.edits:
            raise ValueError(
                f"indel_sum ({indel_sum}) > self.edits ({self.edits}) with cigar: {self.cigar}"
            )
        return self.edits - indel_sum

    @property
    def end(self) -> int:
        """The end position of the hit (1-based inclusive)."""
        return self.start + self.cigar.length_on_target() - 1

    def __str__(self) -> str:
        """Returns the string representation in bwa's XA tag format."""
        # E.g. XA:Z:chr4,-97592047,24M,3;chr8,-32368749,24M,3;
        return ",".join([
            self.refname,
            ("-" if self.negative else "+") + f"{self.start}",
            f"{self.cigar}",
            f"{self.edits}",
        ])


###########################################################
# Below is code to facilitate testing
###########################################################

cdef _assert_gap_opt_are_the_same_c():
    """Tests that the defaults are synced between bwa and pybwa."""
    cdef gap_opt_t *bwa_gap_opt
    cdef gap_opt_t *pybwa_gap_opt
    bwa_gap_opt = gap_init_opt()
    options = BwaAlnOptions()
    pybwa_gap_opt = options.gap_opt()

    # Check member by member for the gap_opt_t struct
    assert pybwa_gap_opt.s_mm == bwa_gap_opt.s_mm
    assert pybwa_gap_opt.s_gape == bwa_gap_opt.s_gape
    assert pybwa_gap_opt.mode == bwa_gap_opt.mode
    assert pybwa_gap_opt.indel_end_skip == bwa_gap_opt.indel_end_skip
    assert pybwa_gap_opt.max_del_occ == bwa_gap_opt.max_del_occ
    assert pybwa_gap_opt.max_entries == bwa_gap_opt.max_entries
    assert pybwa_gap_opt.fnr == bwa_gap_opt.fnr
    assert pybwa_gap_opt.max_diff == bwa_gap_opt.max_diff
    assert pybwa_gap_opt.max_gapo == bwa_gap_opt.max_gapo
    assert pybwa_gap_opt.max_gape == bwa_gap_opt.max_gape
    assert pybwa_gap_opt.max_seed_diff == bwa_gap_opt.max_seed_diff
    assert pybwa_gap_opt.seed_len == bwa_gap_opt.seed_len
    assert pybwa_gap_opt.n_threads == bwa_gap_opt.n_threads
    assert pybwa_gap_opt.max_top2 == bwa_gap_opt.max_top2
    assert pybwa_gap_opt.trim_qual == bwa_gap_opt.trim_qual

    free(bwa_gap_opt)


def _assert_gap_opt_are_the_same() -> None:
    """Tests that the defaults are synced between bwa and pybwa."""
    _assert_gap_opt_are_the_same_c()
