# SPDX-FileCopyrightText: Copyright (c) 2023-2026, NVIDIA CORPORATION & AFFILIATES.  All rights reserved.
# SPDX-License-Identifier: Apache-2.0

import contextlib
import json
import os
import re
import sys
import traceback
from collections import defaultdict
from functools import wraps
from itertools import chain

import pytest


def replace_kwargs(new_kwargs):
    def wrapper(func):
        @wraps(func)
        def wrapped(*args, **kwargs):
            kwargs.update(new_kwargs)
            return func(*args, **kwargs)

        return wrapped

    return wrapper


@contextlib.contextmanager
def null_assert_warnings(*args, **kwargs):
    try:
        yield []
    finally:
        pass


@pytest.fixture(scope="session", autouse=True)
def patch_testing_functions():
    tm.assert_produces_warning = null_assert_warnings  # noqa: F821
    pytest.raises = replace_kwargs({"match": None})(pytest.raises)


# Dictionary to store function call counts
function_call_counts = defaultdict(lambda: defaultdict(int))  # type: ignore[var-annotated]

# The specific functions to track
FUNCTION_NAME = {"_slow_function_call", "_fast_function_call"}


def find_pytest_file(frame) -> str | None:
    for frame in traceback.extract_stack():
        file = frame.filename
        if "pandas-testing/pandas-tests/tests" in file and file.rsplit("/", 1)[
            -1
        ].startswith("test_"):
            return str(file).rsplit("pandas-tests/", 1)[-1]
    return None


def trace_calls(frame, event, arg):
    if (
        event == "call"
        and (func_name := frame.f_code.co_name) in FUNCTION_NAME
        and (filename := find_pytest_file(frame)) is not None
    ):
        function_call_counts[filename][func_name] += 1


def pytest_sessionstart(session):
    # Set the profile function to trace calls
    sys.setprofile(trace_calls)


def pytest_sessionfinish(session, exitstatus):
    # Remove the profile function
    sys.setprofile(None)


@pytest.hookimpl(trylast=True)
def pytest_unconfigure(config):
    if hasattr(config, "workerinput"):
        # Running in xdist worker, write the counts before exiting
        worker_id = config.workerinput["workerid"]
        output_file = f"function_call_counts_worker_{worker_id}.json"
        with open(output_file, "w") as f:
            json.dump(function_call_counts, f, indent=4)
        print(f"Function call counts have been written to {output_file}")  # noqa: T201


@pytest.fixture(autouse=True)
def set_copy_on_write_option():
    import pandas as xpd

    cudf = xpd._fsproxy_fast
    pd = xpd._fsproxy_slow
    with cudf.option_context("copy_on_write", True):
        with pd.option_context("mode.copy_on_write", True):
            yield


# TODO: Pass these tests with cudf.pandas enabled.
NODEIDS_THAT_FAIL_WITH_CUDF_PANDAS = {
    "tests/api/test_api.py::test_pandas_array_alias",
    "tests/apply/test_frame_apply.py::test_apply[python]",
    "tests/apply/test_frame_apply.py::test_apply_function_runs_once",
    "tests/apply/test_frame_apply.py::test_apply_raw_function_runs_once[python]",
    "tests/apply/test_frame_apply.py::test_mixed_column_raises[max-df0]",
    "tests/apply/test_frame_apply.py::test_mixed_column_raises[min-df0]",
    "tests/apply/test_frame_apply.py::test_mixed_column_raises[sum-df0]",
    "tests/apply/test_frame_apply.py::test_nuiscance_columns",
    "tests/apply/test_series_apply.py::test_apply[False]",
    "tests/apply/test_series_apply.py::test_apply[compat]",
    "tests/arithmetic/test_categorical.py::TestCategoricalComparisons::test_categorical_tuple_equality",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64_array_sub_dt_with_different_timezone[DataFrame]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64_array_sub_dt_with_different_timezone[Index]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64_array_sub_dt_with_different_timezone[Series]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64_array_sub_dt_with_different_timezone[array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64_mul_div_numeric_invalid[array-one2-dt64_series2]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'+01:15']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'-02:15']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'Asia/Tokyo']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'US/Eastern']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'UTC']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'UTC+01:15']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'UTC-02:15']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'dateutil/Asia/Singapore']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'dateutil/US/Pacific']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-<UTC>]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-None]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-datetime.timezone(datetime.timedelta(seconds=3600))]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-datetime.timezone.utc]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-pytz.FixedOffset(-300)]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-pytz.FixedOffset(300)]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-tzlocal()]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-tzutc()]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-zoneinfo.ZoneInfo(key='US/Pacific')]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-zoneinfo.ZoneInfo(key='UTC')]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['+01:15'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['+01:15'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['+01:15'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['+01:15'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['-02:15'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['-02:15'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['-02:15'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['-02:15'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['Asia/Tokyo'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['Asia/Tokyo'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['Asia/Tokyo'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['Asia/Tokyo'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['US/Eastern'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['US/Eastern'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['US/Eastern'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['US/Eastern'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC+01:15'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC+01:15'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC+01:15'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC+01:15'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC-02:15'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC-02:15'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC-02:15'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC-02:15'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/Asia/Singapore'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/Asia/Singapore'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/Asia/Singapore'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/Asia/Singapore'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/US/Pacific'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/US/Pacific'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/US/Pacific'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/US/Pacific'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[<UTC>-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[<UTC>-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[<UTC>-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[<UTC>-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[None-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[None-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[None-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[None-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone.utc-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone.utc-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone.utc-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone.utc-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(-300)-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(-300)-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(-300)-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(-300)-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(300)-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(300)-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(300)-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(300)-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzlocal()-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzlocal()-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzlocal()-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzlocal()-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzutc()-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzutc()-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzutc()-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzutc()-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='UTC')-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='UTC')-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='UTC')-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='UTC')-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_sub_dtscalar[array-ts2]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_sub_dtscalar[array-ts3]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64ArrayLikeComparisons::test_dt64arr_cmp_scalar_invalid[None-DataFrame-None]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64ArrayLikeComparisons::test_dt64arr_cmp_scalar_invalid[None-DataFrame-other6]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64ArrayLikeComparisons::test_dt64arr_cmp_scalar_invalid[None-Series-None]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64ArrayLikeComparisons::test_dt64arr_cmp_scalar_invalid[None-Series-other6]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['+01:15'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['-02:15'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['Asia/Tokyo'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['US/Eastern'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['UTC'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['UTC+01:15'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['UTC-02:15'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['dateutil/Asia/Singapore'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['dateutil/US/Pacific'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[<UTC>-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[datetime.timezone(datetime.timedelta(seconds=3600))-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[datetime.timezone.utc-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[pytz.FixedOffset(-300)-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[pytz.FixedOffset(300)-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[tzlocal()-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[tzutc()-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[zoneinfo.ZoneInfo(key='US/Pacific')-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[zoneinfo.ZoneInfo(key='UTC')-array]",
    "tests/arithmetic/test_datetime64.py::test_non_nano_dt64_addsub_np_nat_scalars_unsupported_unit",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-add-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-floordiv-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-mod-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-mul-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-pow-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-sub-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-truediv-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-add-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-floordiv-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-mod-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-mul-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-pow-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-sub-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-truediv-slice]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_df_mod_zero_df[numexpr]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_df_mod_zero_df[python]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-Index0-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-Index0-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-Index1-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-Index1-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-RangeIndex-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-RangeIndex-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-Index0-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-Index0-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-Index1-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-Index1-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-RangeIndex-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-RangeIndex-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-Index0-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-Index0-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-Index1-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-Index1-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-RangeIndex-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-RangeIndex-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-Index0-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-Index0-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-Index1-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-Index1-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-RangeIndex-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-RangeIndex-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr--0.0-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr--0.0-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr--0.0-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr--0.0-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr-zero14-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr-zero14-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr-zero14-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr-zero14-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python--0.0-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python--0.0-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python--0.0-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python--0.0-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python-zero14-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python-zero14-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python-zero14-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python-zero14-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr--0.0-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr--0.0-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr--0.0-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr--0.0-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr-zero14-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr-zero14-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr-zero14-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr-zero14-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python--0.0-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python--0.0-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python--0.0-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python--0.0-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python-zero14-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python-zero14-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python-zero14-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python-zero14-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_inf[numexpr]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_inf[python]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int16-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int16-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int16-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int8-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int8-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int8-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint16-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint16-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint16-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint8-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint8-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint8-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int16-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int16-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int16-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int8-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int8-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int8-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint16-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint16-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint16-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint8-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint8-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint8-uint64]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2[numexpr]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2[python]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx0-2-expected0]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx1--3-expected1]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx4-50-expected4]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx5-3-expected5]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx6-4-expected6]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx7-2-expected7]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx0-2-expected0]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx1--3-expected1]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx4-50-expected4]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx5-3-expected5]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx6-4-expected6]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx7-2-expected7]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index0-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index0-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index0-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index0-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index0-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index0-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta640-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta640-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta640-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta640-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta641-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta641-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta641-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta641-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_0-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_0-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_0-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_0-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_1-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_1-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_1-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_1-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta640-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta640-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta640-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta640-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta641-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta641-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta641-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta641-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_0-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_0-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_0-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_0-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_1-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_1-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_1-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_1-RangeIndex-array]",
    "tests/arithmetic/test_object.py::TestObjectComparisons::test_object_comparisons[True]",
    "tests/arithmetic/test_object.py::test_index_ops_defer_to_unknown_subclasses[datetime]",
    "tests/arithmetic/test_object.py::test_index_ops_defer_to_unknown_subclasses[object]",
    "tests/arithmetic/test_object.py::test_index_ops_defer_to_unknown_subclasses[period]",
    "tests/arithmetic/test_object.py::test_index_ops_defer_to_unknown_subclasses[timedelta]",
    "tests/arithmetic/test_period.py::TestPeriodIndexSeriesMethods::test_pi_sub_period",
    "tests/arithmetic/test_period.py::TestPeriodIndexSeriesMethods::test_pi_sub_period_nat",
    "tests/arithmetic/test_timedelta64.py::TestAddSubNaTMasking::test_tdi_add_overflow",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_addition_ops",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_dti_tdi_numeric_ops",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_subtraction_ops",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_subtraction_ops_with_tz[DataFrame]",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_subtraction_ops_with_tz[Index]",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_subtraction_ops_with_tz[Series]",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_subtraction_ops_with_tz[array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_operators_timedelta64",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_datetime64_nat[array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_datetimelike_scalar[array-None-datetime64]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_td64_nat[array-tdnat0]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names0]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names2]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names4]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names5]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names6]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_timedeltalike[timedelta640-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_timedeltalike[timedelta641-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_timedeltalike[timedelta64_0-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_timedeltalike[timedelta64_1-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_sub_timedeltalike[timedelta640-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_sub_timedeltalike[timedelta641-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_nat_invalid[DataFrame]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_nat_invalid[Series]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_ndarray[array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-D-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-D-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-D-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-h-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-h-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-h-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-m-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-m-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-m-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ms-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ms-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ms-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ns-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ns-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ns-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-s-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-s-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-s-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-us-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-us-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-us-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64nat[array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar[timedelta640-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar[timedelta641-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar[timedelta64_0-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar[timedelta64_1-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar_with_nat[timedelta640-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar_with_nat[timedelta641-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar_with_nat[timedelta64_0-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar_with_nat[timedelta64_1-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_floordiv_td64arr_with_nat[DataFrame]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_floordiv_td64arr_with_nat[array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_floordiv_tdscalar[array-timedelta64]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_mul_numeric_scalar[array-one1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmod_tdscalar[array-timedelta640]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmod_tdscalar[array-timedelta641]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmod_tdscalar[array-timedelta64_0]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmod_tdscalar[array-timedelta64_1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-float-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-float32-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-float64-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-int-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-int16-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-int32-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-int64-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-int8-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-uint16-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-uint32-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-uint64-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-uint8-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_tdi_rmul_arraylike[array-ndarray]",
    "tests/arrays/boolean/test_arithmetic.py::test_div",
    "tests/arrays/boolean/test_construction.py::test_coerce_to_array_from_boolean_array",
    "tests/arrays/boolean/test_construction.py::test_coerce_to_numpy_array",
    "tests/arrays/boolean/test_construction.py::test_to_numpy_copy",
    "tests/arrays/boolean/test_function.py::test_value_counts_with_normalize",
    "tests/arrays/boolean/test_ops.py::TestUnaryOps::test_invert",
    "tests/arrays/boolean/test_reduction.py::test_any_all[values1-False-False-exp_any_noskip1-False]",
    "tests/arrays/boolean/test_reduction.py::test_any_all[values2-False-True-exp_any_noskip2-exp_all_noskip2]",
    "tests/arrays/boolean/test_reduction.py::test_reductions_return_types[count-False]",
    "tests/arrays/boolean/test_reduction.py::test_reductions_return_types[count-True]",
    "tests/arrays/categorical/test_analytics.py::TestCategoricalAnalytics::test_min_max_ordered[array]",
    "tests/arrays/categorical/test_analytics.py::TestCategoricalAnalytics::test_searchsorted[False]",
    "tests/arrays/categorical/test_analytics.py::TestCategoricalAnalytics::test_searchsorted[None]",
    "tests/arrays/categorical/test_analytics.py::TestCategoricalAnalytics::test_searchsorted[True]",
    "tests/arrays/categorical/test_api.py::TestCategoricalAPI::test_rename_categories",
    "tests/arrays/categorical/test_api.py::TestCategoricalAPI::test_set_categories",
    "tests/arrays/categorical/test_api.py::TestPrivateCategoricalAPI::test_codes_immutable",
    "tests/arrays/categorical/test_astype.py::TestAstype::test_astype[False]",
    "tests/arrays/categorical/test_astype.py::TestAstype::test_astype[True]",
    "tests/arrays/categorical/test_astype.py::TestAstype::test_astype_nan_to_int[values0-CategoricalIndex]",
    "tests/arrays/categorical/test_astype.py::TestAstype::test_astype_nan_to_int[values1-CategoricalIndex]",
    "tests/arrays/categorical/test_constructors.py::TestCategoricalConstructors::test_constructor",
    "tests/arrays/categorical/test_constructors.py::TestCategoricalConstructors::test_constructor_preserves_freq",
    "tests/arrays/categorical/test_constructors.py::TestCategoricalConstructors::test_from_sequence_copy",
    "tests/arrays/categorical/test_indexing.py::TestCategoricalIndexing::test_where_new_category_raises",
    "tests/arrays/categorical/test_missing.py::TestCategoricalMissing::test_use_inf_as_na_outside_context[values2-expected2]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical[a-a-result2-None]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical[b-None-result3-Categorical.categories length are different]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical[b-c-result0-Categorical.categories are different]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical_ea_dtype",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical_series[to_replace11-value11-expected11-False]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical_series[to_replace6-4-expected6-False]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical_series[to_replace7-5-expected7-False]",
    "tests/arrays/categorical/test_sorting.py::TestCategoricalSort::test_sort_values",
    "tests/arrays/categorical/test_sorting.py::TestCategoricalSort::test_sort_values_na_position",
    "tests/arrays/datetimes/test_constructors.py::TestDatetimeArrayConstructor::test_copy",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_empty[ms-Europe/Berlin]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_empty[ns-Asia/Kolkata]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_empty[ns-UTC]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_empty[s-UTC]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_empty[us-US/Eastern]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_integers",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[ms-ms-UTC-Europe/Berlin-data1]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[ms-us-US/Eastern-UTC-data6]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[ns-ns-US/Central-Asia/Kolkata-data3]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[ns-s-UTC-UTC-data4]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[s-ns-US/Central-Asia/Kolkata-data7]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[s-s-UTC-UTC-data0]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[us-ms-UTC-Europe/Berlin-data5]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[us-us-US/Eastern-UTC-data2]",
    "tests/arrays/floating/test_astype.py::test_astype_copy",
    "tests/arrays/floating/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Float32Dtype]",
    "tests/arrays/floating/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Float64Dtype]",
    "tests/arrays/floating/test_construction.py::test_floating_array_constructor_copy",
    "tests/arrays/floating/test_function.py::test_ufuncs_single[absolute]",
    "tests/arrays/floating/test_function.py::test_ufuncs_single[sign]",
    "tests/arrays/floating/test_function.py::test_value_counts_empty",
    "tests/arrays/floating/test_repr.py::test_frame_repr[Float32Dtype]",
    "tests/arrays/floating/test_to_numpy.py::test_to_numpy_copy",
    "tests/arrays/integer/test_arithmetic.py::test_values_multiplying_large_series_by_NA",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Int16Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Int32Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Int64Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Int8Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[UInt16Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[UInt32Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[UInt64Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[UInt8Dtype]",
    "tests/arrays/integer/test_construction.py::test_integer_array_constructor_copy",
    "tests/arrays/integer/test_dtypes.py::test_astype_copy",
    "tests/arrays/integer/test_function.py::test_ufuncs_single_int[absolute]",
    "tests/arrays/integer/test_function.py::test_ufuncs_single_int[sign]",
    "tests/arrays/integer/test_function.py::test_value_counts_empty",
    "tests/arrays/interval/test_interval.py::TestSetitem::test_set_na[float64]",
    "tests/arrays/interval/test_interval_pyarrow.py::test_arrow_array",
    "tests/arrays/interval/test_interval_pyarrow.py::test_arrow_array_missing",
    "tests/arrays/interval/test_interval_pyarrow.py::test_from_arrow_from_raw_struct_array",
    "tests/arrays/masked/test_arithmetic.py::test_frame[boolean-__floordiv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[boolean-__pow__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[boolean-__rfloordiv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[boolean-__rpow__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[boolean-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[boolean-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__floordiv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__pow__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__rfloordiv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__rpow__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__truediv__]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Float32]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Float64]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Int16]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Int32]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Int64]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Int8]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[UInt16]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[UInt32]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[UInt64]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[UInt8]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[boolean]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_from_arrow_uint",
    "tests/arrays/masked/test_arrow_compat.py::test_dataframe_from_arrow_types_mapper",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Float32-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Float32-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Float64-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Float64-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int16-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int16-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int32-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int32-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int64-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int64-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int8-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int8-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt16-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt16-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt32-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt32-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt64-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt64-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt8-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt8-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[boolean-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[boolean-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Float32]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Float64]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Int16]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Int32]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Int64]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Int8]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[UInt16]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[UInt32]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[UInt64]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[UInt8]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[boolean]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_float[Float32-1.0]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_float[Float32-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_float[Float64-1.0]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_float[Float64-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[Int16-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[Int32-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[Int64-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[Int8-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[UInt16-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[UInt32-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[UInt64-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[UInt8-1]",
    "tests/arrays/numpy_/test_numpy.py::test_factorize_unsigned",
    "tests/arrays/numpy_/test_numpy.py::test_np_max_nested_tuples",
    "tests/arrays/numpy_/test_numpy.py::test_np_reduce_2d",
    "tests/arrays/numpy_/test_numpy.py::test_setitem_preserves_views",
    "tests/arrays/numpy_/test_numpy.py::test_setitem_series",
    "tests/arrays/numpy_/test_numpy.py::test_to_numpy",
    "tests/arrays/numpy_/test_numpy.py::test_ufunc",
    "tests/arrays/numpy_/test_numpy.py::test_ufunc_unary[absolute]",
    "tests/arrays/numpy_/test_numpy.py::test_ufunc_unary[negative]",
    "tests/arrays/numpy_/test_numpy.py::test_ufunc_unary[positive]",
    "tests/arrays/period/test_arrow_compat.py::test_arrow_array_missing",
    "tests/arrays/period/test_astype.py::test_astype_copies",
    "tests/arrays/sparse/test_accessor.py::TestFrameAccessor::test_from_spmatrix_columns[columns1]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[add-a0-b0]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[add-a1-b1]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[add-a2-b2]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[add-a3-b3]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[add-a4-b4]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[greater-a0-b0]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[greater-a1-b1]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[greater-a2-b2]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[greater-a3-b3]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[greater-a4-b4]",
    "tests/arrays/sparse/test_arithmetics.py::test_ufuncs[arr0-absolute]",
    "tests/arrays/sparse/test_arithmetics.py::test_ufuncs[arr0-exp]",
    "tests/arrays/sparse/test_arithmetics.py::test_ufuncs[arr1-absolute]",
    "tests/arrays/sparse/test_arithmetics.py::test_ufuncs[arr1-exp]",
    "tests/arrays/sparse/test_array.py::TestSparseArrayAnalytics::test_modf[0.0]",
    "tests/arrays/sparse/test_array.py::TestSparseArrayAnalytics::test_modf[nan]",
    "tests/arrays/sparse/test_array.py::TestSparseArrayAnalytics::test_ufunc",
    "tests/arrays/sparse/test_array.py::TestSparseArrayAnalytics::test_ufunc_args",
    "tests/arrays/sparse/test_array.py::test_array_interface",
    "tests/arrays/sparse/test_constructors.py::TestConstructors::test_constructor_copy",
    "tests/arrays/string_/test_string.py::test_arrow_array[pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_array[pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_array[string=str[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_arrow_array[string=string[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow_numpy-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow_numpy-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow_numpy-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[python-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[python-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[python-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=str[pyarrow]-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=str[pyarrow]-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=str[python]-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=str[python]-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=string[pyarrow]-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=string[pyarrow]-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=string[python]-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=string[python]-python]",
    "tests/arrays/string_/test_string.py::test_arrow_roundtrip[pyarrow-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_roundtrip[pyarrow_numpy-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_roundtrip[python-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_astype_float[pyarrow-Float32]",
    "tests/arrays/string_/test_string.py::test_astype_float[pyarrow-Float64]",
    "tests/arrays/string_/test_string.py::test_astype_float[pyarrow_numpy-Float32]",
    "tests/arrays/string_/test_string.py::test_astype_float[pyarrow_numpy-Float64]",
    "tests/arrays/string_/test_string.py::test_astype_float[python-Float32]",
    "tests/arrays/string_/test_string.py::test_astype_float[python-Float64]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[pyarrow]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[python]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[string=str[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[string=str[python]]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[string=string[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[string=string[python]]",
    "tests/arrays/string_/test_string.py::test_isin[pyarrow]",
    "tests/arrays/string_/test_string.py::test_isin[pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_isin[python]",
    "tests/arrays/string_/test_string.py::test_isin[string=str[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_isin[string=str[python]]",
    "tests/arrays/string_/test_string.py::test_isin[string=string[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_isin[string=string[python]]",
    "tests/arrays/string_/test_string.py::test_min_max[pyarrow-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[pyarrow-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[pyarrow_numpy-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[pyarrow_numpy-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[python-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[python-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[string=str[pyarrow]-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[string=str[pyarrow]-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[string=str[python]-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[string=str[python]-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[string=string[pyarrow]-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[string=string[pyarrow]-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[string=string[python]-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[string=string[python]-False-min]",
    "tests/arrays/string_/test_string.py::test_repr[pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_repr[string=str[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_repr[string=str[python]]",
    "tests/arrays/string_/test_string.py::test_to_numpy_returns_pdna_default[pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_value_counts_with_normalize[pyarrow]",
    "tests/arrays/string_/test_string.py::test_value_counts_with_normalize[python]",
    "tests/arrays/test_array.py::test_array_inference[data7-expected7]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_from_array_keeps_base",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-B-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-D-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-ME-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-QE-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-W-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-YE-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-B-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-D-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-ME-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-QE-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-W-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-YE-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-B-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-D-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-ME-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-QE-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-W-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-YE-True]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[B]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[D]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[ME]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[QE]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[W]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[YE]",
    "tests/arrays/test_datetimelike.py::TestTimedeltaArray::test_array_interface",
    "tests/arrays/test_datetimelike.py::TestTimedeltaArray::test_searchsorted_castable_strings[pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::test_searchsorted_datetimelike_with_listlike_invalid_dtype[arg0-values1]",
    "tests/arrays/test_datetimes.py::TestDatetimeArray::test_array_interface",
    "tests/arrays/test_datetimes.py::TestDatetimeArray::test_astype_to_same",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[eq]",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[ge]",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[gt]",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[le]",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[lt]",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[ne]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'Asia/Tokyo'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'Asia/Tokyo'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'US/Eastern'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'US/Eastern'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'dateutil/Asia/Singapore'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'dateutil/Asia/Singapore'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'dateutil/US/Pacific'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'dateutil/US/Pacific'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-<UTC>-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-<UTC>-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-None-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-None-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone(datetime.timedelta(seconds=3600))-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone(datetime.timedelta(seconds=3600))-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone.utc-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone.utc-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-pytz.FixedOffset(-300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-pytz.FixedOffset(-300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-pytz.FixedOffset(300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-pytz.FixedOffset(300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-tzlocal()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-tzlocal()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-tzutc()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-tzutc()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-zoneinfo.ZoneInfo(key='US/Pacific')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-zoneinfo.ZoneInfo(key='US/Pacific')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-zoneinfo.ZoneInfo(key='UTC')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-zoneinfo.ZoneInfo(key='UTC')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'Asia/Tokyo'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'Asia/Tokyo'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'US/Eastern'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'US/Eastern'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'dateutil/Asia/Singapore'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'dateutil/Asia/Singapore'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'dateutil/US/Pacific'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'dateutil/US/Pacific'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-<UTC>-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-<UTC>-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-None-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-None-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone(datetime.timedelta(seconds=3600))-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone(datetime.timedelta(seconds=3600))-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone.utc-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone.utc-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-pytz.FixedOffset(-300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-pytz.FixedOffset(-300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-pytz.FixedOffset(300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-pytz.FixedOffset(300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-tzlocal()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-tzlocal()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-tzutc()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-tzutc()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-zoneinfo.ZoneInfo(key='US/Pacific')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-zoneinfo.ZoneInfo(key='US/Pacific')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-zoneinfo.ZoneInfo(key='UTC')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-zoneinfo.ZoneInfo(key='UTC')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'Asia/Tokyo'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'Asia/Tokyo'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'US/Eastern'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'US/Eastern'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'dateutil/Asia/Singapore'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'dateutil/Asia/Singapore'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'dateutil/US/Pacific'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'dateutil/US/Pacific'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-<UTC>-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-<UTC>-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-None-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-None-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone(datetime.timedelta(seconds=3600))-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone(datetime.timedelta(seconds=3600))-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone.utc-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone.utc-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-pytz.FixedOffset(-300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-pytz.FixedOffset(-300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-pytz.FixedOffset(300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-pytz.FixedOffset(300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-tzlocal()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-tzlocal()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-tzutc()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-tzutc()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-zoneinfo.ZoneInfo(key='US/Pacific')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-zoneinfo.ZoneInfo(key='US/Pacific')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-zoneinfo.ZoneInfo(key='UTC')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-zoneinfo.ZoneInfo(key='UTC')-scalar3]",
    "tests/arrays/test_period.py::test_registered",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[ms-nat0]",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[ms-nat1]",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[s-nat0]",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[s-nat1]",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[us-nat0]",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[us-nat1]",
    "tests/arrays/test_timedeltas.py::TestUnaryOps::test_abs",
    "tests/arrays/test_timedeltas.py::TestUnaryOps::test_neg",
    "tests/arrays/test_timedeltas.py::TestUnaryOps::test_neg_freq",
    "tests/arrays/test_timedeltas.py::TestUnaryOps::test_pos",
    "tests/arrays/timedeltas/test_constructors.py::TestTimedeltaArrayConstructor::test_copy",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_reductions_empty[False-max]",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_reductions_empty[False-min]",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_reductions_empty[True-max]",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_reductions_empty[True-min]",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_std[add0]",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_std[add1]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[DataFrame-array-datetime64[D]]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[DataFrame-dict-datetime64[D]]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[DataFrame-dict-object-datetime.datetime]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[Index-datetime64[D]]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[Series-datetime64[D]]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[Series-object-datetime.datetime]",
    "tests/base/test_conversion.py::test_array[index-arr0-_codes]",
    "tests/base/test_conversion.py::test_array[index-arr1-_ndarray]",
    "tests/base/test_conversion.py::test_array[index-arr2-_data]",
    "tests/base/test_conversion.py::test_array[index-arr3-_left]",
    "tests/base/test_conversion.py::test_array[index-arr4-_sparse_values]",
    "tests/base/test_conversion.py::test_array[index-arr5-_ndarray]",
    "tests/base/test_conversion.py::test_array[index-arr6-_ndarray]",
    "tests/base/test_conversion.py::test_array[series-arr0-_codes]",
    "tests/base/test_conversion.py::test_array[series-arr1-_ndarray]",
    "tests/base/test_conversion.py::test_array[series-arr2-_data]",
    "tests/base/test_conversion.py::test_array[series-arr3-_left]",
    "tests/base/test_conversion.py::test_array[series-arr4-_sparse_values]",
    "tests/base/test_conversion.py::test_array[series-arr5-_ndarray]",
    "tests/base/test_conversion.py::test_array[series-arr6-_ndarray]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr0-expected0-True]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr1-expected1-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr1-expected1]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr2-expected2-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr3-expected3-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr3-expected3]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr4-expected4-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr5-expected5-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr6-expected6-True]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr6-expected6]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr7-expected7-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr8-expected8-True]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr8-expected8]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr9-expected9-False]",
    "tests/base/test_conversion.py::test_to_numpy[series-arr0-expected0-True]",
    "tests/base/test_conversion.py::test_to_numpy[series-arr6-expected6-True]",
    "tests/base/test_conversion.py::test_to_numpy[series-arr8-expected8-True]",
    "tests/base/test_conversion.py::test_to_numpy_copy[arr0-False]",
    "tests/base/test_conversion.py::test_to_numpy_copy[arr0-True]",
    "tests/base/test_conversion.py::test_to_numpy_copy[arr1-False]",
    "tests/base/test_conversion.py::test_to_numpy_copy[arr1-True]",
    "tests/base/test_conversion.py::test_values_consistent[arr0-ndarray-int64]",
    "tests/base/test_conversion.py::test_values_consistent[arr1-ndarray-object]",
    "tests/base/test_fillna.py::test_fillna[datetime]",
    "tests/base/test_fillna.py::test_fillna[multi]",
    "tests/base/test_fillna.py::test_fillna[tuples]",
    "tests/base/test_misc.py::test_ndarray_compat_properties[multi]",
    "tests/base/test_misc.py::test_ndarray_compat_properties[tuples]",
    "tests/base/test_unique.py::test_unique[multi]",
    "tests/base/test_unique.py::test_unique[tuples]",
    "tests/base/test_value_counts.py::test_value_counts[multi]",
    "tests/base/test_value_counts.py::test_value_counts[tuples]",
    "tests/base/test_value_counts.py::test_value_counts_bins[index]",
    "tests/base/test_value_counts.py::test_value_counts_bins[series]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[index-ms]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[index-ns]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[index-s]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[index-us]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[series-ms]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[series-ns]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[series-s]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[series-us]",
    "tests/computation/test_eval.py::TestAlignment::test_performance_warning_for_poor_alignment[numexpr-pandas]",
    "tests/computation/test_eval.py::TestAlignment::test_performance_warning_for_poor_alignment[numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_disallow_python_keywords",
    "tests/computation/test_eval.py::TestMath::test_binary_functions[arctan2]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[abs]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arccos]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arccosh]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arcsin]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arcsinh]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arctan]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arctanh]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[ceil]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[cos]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[cosh]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[exp]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[expm1]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[floor]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[log10]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[log1p]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[log]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[sin]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[sinh]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[sqrt]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[tanh]",
    "tests/computation/test_eval.py::TestOperations::test_query_inplace",
    "tests/computation/test_eval.py::test_eval_no_support_column_name[False]",
    "tests/computation/test_eval.py::test_eval_no_support_column_name[True]",
    "tests/computation/test_eval.py::test_numexpr_option_respected[False-python]",
    "tests/copy_view/index/test_index.py::test_index_from_series_copy",
    "tests/copy_view/test_array.py::test_dataframe_array_ea_dtypes",
    "tests/copy_view/test_array.py::test_dataframe_array_string_dtype",
    "tests/copy_view/test_array.py::test_dataframe_multiple_numpy_dtypes",
    "tests/copy_view/test_array.py::test_dataframe_values[values]",
    "tests/copy_view/test_array.py::test_series_array_ea_dtypes",
    "tests/copy_view/test_array.py::test_series_to_numpy",
    "tests/copy_view/test_array.py::test_series_values[values]",
    "tests/copy_view/test_astype.py::test_astype_arrow_timestamp",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[Int64-Int64]",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[Int64-int64]",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[int64-Int64]",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[int64-int64]",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[int64[pyarrow]-Int64]",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[int64[pyarrow]-int64]",
    "tests/copy_view/test_astype.py::test_astype_dict_dtypes",
    "tests/copy_view/test_astype.py::test_astype_numpy_to_ea",
    "tests/copy_view/test_astype.py::test_astype_single_dtype",
    "tests/copy_view/test_astype.py::test_astype_string_and_object[object-string]",
    "tests/copy_view/test_astype.py::test_astype_string_and_object[string-object]",
    "tests/copy_view/test_astype.py::test_astype_string_and_object_update_original[object-string]",
    "tests/copy_view/test_astype.py::test_astype_string_and_object_update_original[string-object]",
    "tests/copy_view/test_astype.py::test_convert_dtypes",
    "tests/copy_view/test_astype.py::test_convert_dtypes_infer_objects",
    "tests/copy_view/test_chained_assignment_deprecation.py::test_series_setitem[0]",
    "tests/copy_view/test_chained_assignment_deprecation.py::test_series_setitem[indexer1]",
    "tests/copy_view/test_chained_assignment_deprecation.py::test_series_setitem[indexer2]",
    "tests/copy_view/test_chained_assignment_deprecation.py::test_series_setitem[indexer3]",
    "tests/copy_view/test_clip.py::test_clip_inplace",
    "tests/copy_view/test_clip.py::test_clip_inplace_reference_no_op",
    "tests/copy_view/test_constructors.py::test_dataframe_constructor_mgr_or_df[None-False]",
    "tests/copy_view/test_constructors.py::test_dataframe_constructor_mgr_or_df[None-True]",
    "tests/copy_view/test_constructors.py::test_dataframe_constructor_mgr_or_df[columns1-False]",
    "tests/copy_view/test_constructors.py::test_dataframe_constructor_mgr_or_df[columns1-True]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-None-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-None-None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-None-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-index1-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-index1-None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-index1-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-None-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-None-None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-None-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-index1-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-index1-None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-index1-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns2-None-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns2-None-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns2-index1-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns2-index1-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series_with_dtype[None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series_with_dtype[index1]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series_with_reindex[None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series_with_reindex[int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_records_with_dataframe",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data0-None-Index]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data0-None-Series]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data1-int64-Index]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data1-int64-Series]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data2-None-Index]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data2-None-Series]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data2-object-Index]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data2-object-Series]",
    "tests/copy_view/test_constructors.py::test_frame_from_dict_of_index",
    "tests/copy_view/test_constructors.py::test_frame_from_numpy_array[False]",
    "tests/copy_view/test_constructors.py::test_series_from_index[idx0]",
    "tests/copy_view/test_constructors.py::test_series_from_index[idx1]",
    "tests/copy_view/test_constructors.py::test_series_from_index[idx3]",
    "tests/copy_view/test_constructors.py::test_series_from_index_different_dtypes",
    "tests/copy_view/test_constructors.py::test_series_from_series[None]",
    "tests/copy_view/test_constructors.py::test_series_from_series[int64]",
    "tests/copy_view/test_constructors.py::test_series_from_series_with_reindex",
    "tests/copy_view/test_core_functionalities.py::test_assigning_to_same_variable_removes_references",
    "tests/copy_view/test_core_functionalities.py::test_delete",
    "tests/copy_view/test_core_functionalities.py::test_delete_reference",
    "tests/copy_view/test_core_functionalities.py::test_setitem_dont_track_unnecessary_references",
    "tests/copy_view/test_functions.py::test_concat_copy_keyword[False]",
    "tests/copy_view/test_functions.py::test_merge_copy_keyword[False]",
    "tests/copy_view/test_indexing.py::test_column_as_series[nullable]",
    "tests/copy_view/test_indexing.py::test_column_as_series[numpy]",
    "tests/copy_view/test_indexing.py::test_column_as_series_no_item_cache[nullable-getitem]",
    "tests/copy_view/test_indexing.py::test_column_as_series_no_item_cache[numpy-getitem]",
    "tests/copy_view/test_indexing.py::test_del_frame[nullable]",
    "tests/copy_view/test_indexing.py::test_del_frame[numpy]",
    "tests/copy_view/test_indexing.py::test_del_series[nullable]",
    "tests/copy_view/test_indexing.py::test_del_series[numpy]",
    "tests/copy_view/test_indexing.py::test_getitem_midx_slice",
    "tests/copy_view/test_indexing.py::test_loc_enlarging_with_dataframe",
    "tests/copy_view/test_indexing.py::test_midx_read_only_bool_indexer",
    "tests/copy_view/test_indexing.py::test_series_getitem_ellipsis",
    "tests/copy_view/test_indexing.py::test_series_getitem_slice[nullable]",
    "tests/copy_view/test_indexing.py::test_series_getitem_slice[numpy]",
    "tests/copy_view/test_indexing.py::test_series_midx_slice",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer1-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer1-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer3-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer3-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer5-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer5-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer0-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer0-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer2-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer2-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer4-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer4-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer1-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer1-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer3-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer3-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer5-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer5-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer0-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer0-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer2-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer2-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer4-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer4-a]",
    "tests/copy_view/test_indexing.py::test_subset_column_selection_modify_parent[nullable]",
    "tests/copy_view/test_indexing.py::test_subset_column_selection_modify_parent[numpy]",
    "tests/copy_view/test_indexing.py::test_subset_column_slice[nullable-mixed-block]",
    "tests/copy_view/test_indexing.py::test_subset_column_slice[nullable-single-block]",
    "tests/copy_view/test_indexing.py::test_subset_column_slice[numpy-mixed-block]",
    "tests/copy_view/test_indexing.py::test_subset_column_slice[numpy-single-block]",
    "tests/copy_view/test_indexing.py::test_subset_row_slice[nullable]",
    "tests/copy_view/test_indexing.py::test_subset_row_slice[numpy]",
    "tests/copy_view/test_indexing.py::test_subset_set_column_with_loc[nullable-mixed-block]",
    "tests/copy_view/test_indexing.py::test_subset_set_column_with_loc[nullable-single-block]",
    "tests/copy_view/test_interp_fillna.py::test_fillna",
    "tests/copy_view/test_interp_fillna.py::test_fillna_dict",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Float32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Float64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Int16]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Int32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Int64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Int8]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[UInt16]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[UInt32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[UInt64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[UInt8]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[double[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[float[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[int16[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[int32[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[int64[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[int8[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[uint16[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[uint32[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[uint64[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[uint8[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace[False]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace[None]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Float32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Float64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Int16]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Int32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Int64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Int8]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[UInt16]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[UInt32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[UInt64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[UInt8]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[double[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[float[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[int16[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[int32[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[int64[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[int8[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[uint16[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[uint32[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[uint64[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[uint8[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_reference",
    "tests/copy_view/test_interp_fillna.py::test_fillna_series_empty_arg",
    "tests/copy_view/test_interp_fillna.py::test_fillna_series_empty_arg_inplace",
    "tests/copy_view/test_interp_fillna.py::test_interp_fill_functions[bfill]",
    "tests/copy_view/test_interp_fillna.py::test_interp_fill_functions[ffill]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_chained_assignment[bfill]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_cleaned_fill_method",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_downcast",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_inplace_no_reference_no_copy[vals0]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_inplace_no_reference_no_copy[vals1]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_no_op[linear]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_no_op[nearest]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_no_op[pad]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_object_convert_no_op",
    "tests/copy_view/test_methods.py::test_add_prefix",
    "tests/copy_view/test_methods.py::test_add_suffix",
    "tests/copy_view/test_methods.py::test_align_copy_false",
    "tests/copy_view/test_methods.py::test_align_frame[<lambda>0]",
    "tests/copy_view/test_methods.py::test_align_frame[<lambda>1]",
    "tests/copy_view/test_methods.py::test_align_frame[<lambda>2]",
    "tests/copy_view/test_methods.py::test_align_series",
    "tests/copy_view/test_methods.py::test_align_with_series_copy_false",
    "tests/copy_view/test_methods.py::test_assign_drop_duplicates[assign]",
    "tests/copy_view/test_methods.py::test_assign_drop_duplicates[drop_duplicates]",
    "tests/copy_view/test_methods.py::test_between_time[obj0]",
    "tests/copy_view/test_methods.py::test_between_time[obj1]",
    "tests/copy_view/test_methods.py::test_chained_methods[reset_index]",
    "tests/copy_view/test_methods.py::test_copy",
    "tests/copy_view/test_methods.py::test_copy_shallow",
    "tests/copy_view/test_methods.py::test_droplevel",
    "tests/copy_view/test_methods.py::test_dropna[0-5.5]",
    "tests/copy_view/test_methods.py::test_dropna[1-nan]",
    "tests/copy_view/test_methods.py::test_dropna_series[5.5]",
    "tests/copy_view/test_methods.py::test_dropna_series[5]",
    "tests/copy_view/test_methods.py::test_eval",
    "tests/copy_view/test_methods.py::test_eval_inplace",
    "tests/copy_view/test_methods.py::test_frame_set_axis",
    "tests/copy_view/test_methods.py::test_get[a]",
    "tests/copy_view/test_methods.py::test_get[key1]",
    "tests/copy_view/test_methods.py::test_groupby_column_index_in_references",
    "tests/copy_view/test_methods.py::test_infer_objects",
    "tests/copy_view/test_methods.py::test_insert_series",
    "tests/copy_view/test_methods.py::test_isetitem",
    "tests/copy_view/test_methods.py::test_isetitem_frame",
    "tests/copy_view/test_methods.py::test_isetitem_series[mixed-block]",
    "tests/copy_view/test_methods.py::test_isetitem_series[single-block]",
    "tests/copy_view/test_methods.py::test_items",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[align-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[align-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[align-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[astype-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[astype-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[astype-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[infer_objects-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[infer_objects-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[infer_objects-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex_like-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex_like-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex_like-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis0-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis0-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis0-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis1-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis1-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis1-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_axis-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_axis-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_axis-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_flags-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_flags-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_flags-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[swapaxes-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[swapaxes-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[swapaxes-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[truncate-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[truncate-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[truncate-True]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[astype-False]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[astype-None]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[astype-True]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[reindex-False]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[reindex-None]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[reindex-True]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[rename-False]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[rename-None]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[rename-True]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[truncate-True]",
    "tests/copy_view/test_methods.py::test_pipe",
    "tests/copy_view/test_methods.py::test_pipe_modify_df",
    "tests/copy_view/test_methods.py::test_pop",
    "tests/copy_view/test_methods.py::test_putmask_dont_copy_some_blocks[5-False-None]",
    "tests/copy_view/test_methods.py::test_putmask_no_reference[int64]",
    "tests/copy_view/test_methods.py::test_reindex_like",
    "tests/copy_view/test_methods.py::test_rename_axis[kwargs0]",
    "tests/copy_view/test_methods.py::test_rename_axis[kwargs1]",
    "tests/copy_view/test_methods.py::test_reorder_levels",
    "tests/copy_view/test_methods.py::test_round[-1]",
    "tests/copy_view/test_methods.py::test_round[0]",
    "tests/copy_view/test_methods.py::test_round[1]",
    "tests/copy_view/test_methods.py::test_series_reorder_levels",
    "tests/copy_view/test_methods.py::test_set_index",
    "tests/copy_view/test_methods.py::test_shift_columns",
    "tests/copy_view/test_methods.py::test_shift_no_op",
    "tests/copy_view/test_methods.py::test_sort_index",
    "tests/copy_view/test_methods.py::test_sort_values[obj0-kwargs0]",
    "tests/copy_view/test_methods.py::test_sort_values[obj1-kwargs1]",
    "tests/copy_view/test_methods.py::test_sort_values_inplace[obj0-kwargs0]",
    "tests/copy_view/test_methods.py::test_sort_values_inplace[obj1-kwargs1]",
    "tests/copy_view/test_methods.py::test_squeeze",
    "tests/copy_view/test_methods.py::test_swapaxes_noop[columns]",
    "tests/copy_view/test_methods.py::test_swapaxes_noop[index]",
    "tests/copy_view/test_methods.py::test_swapaxes_single_block",
    "tests/copy_view/test_methods.py::test_swaplevel[obj0]",
    "tests/copy_view/test_methods.py::test_swaplevel[obj1]",
    "tests/copy_view/test_methods.py::test_take[obj0]",
    "tests/copy_view/test_methods.py::test_take[obj1]",
    "tests/copy_view/test_methods.py::test_to_frame",
    "tests/copy_view/test_methods.py::test_to_period[obj0]",
    "tests/copy_view/test_methods.py::test_to_period[obj1]",
    "tests/copy_view/test_methods.py::test_to_timestamp[obj0]",
    "tests/copy_view/test_methods.py::test_to_timestamp[obj1]",
    "tests/copy_view/test_methods.py::test_transpose[False]",
    "tests/copy_view/test_methods.py::test_transpose[True]",
    "tests/copy_view/test_methods.py::test_transpose_copy_keyword[False]",
    "tests/copy_view/test_methods.py::test_transpose_copy_keyword[None]",
    "tests/copy_view/test_methods.py::test_truncate[kwargs0]",
    "tests/copy_view/test_methods.py::test_truncate[kwargs1]",
    "tests/copy_view/test_methods.py::test_update_chained_assignment",
    "tests/copy_view/test_methods.py::test_update_frame",
    "tests/copy_view/test_methods.py::test_xs[mixed-block-1-a]",
    "tests/copy_view/test_methods.py::test_xs[single-block-0-0]",
    "tests/copy_view/test_methods.py::test_xs[single-block-1-a]",
    "tests/copy_view/test_methods.py::test_xs_multiindex[l1-0-0]",
    "tests/copy_view/test_methods.py::test_xs_multiindex[l1-0-1]",
    "tests/copy_view/test_replace.py::test_replace[replace_kwargs0]",
    "tests/copy_view/test_replace.py::test_replace[replace_kwargs1]",
    "tests/copy_view/test_replace.py::test_replace[replace_kwargs2]",
    "tests/copy_view/test_replace.py::test_replace[replace_kwargs3]",
    "tests/copy_view/test_replace.py::test_replace_categorical[1.5]",
    "tests/copy_view/test_replace.py::test_replace_categorical[1]",
    "tests/copy_view/test_replace.py::test_replace_categorical_inplace[1.5]",
    "tests/copy_view/test_replace.py::test_replace_categorical_inplace[1]",
    "tests/copy_view/test_replace.py::test_replace_categorical_inplace_reference[1-1]",
    "tests/copy_view/test_replace.py::test_replace_categorical_inplace_reference[1-to_replace1]",
    "tests/copy_view/test_replace.py::test_replace_chained_assignment",
    "tests/copy_view/test_replace.py::test_replace_coerce_single_column",
    "tests/copy_view/test_replace.py::test_replace_columnwise_no_op",
    "tests/copy_view/test_replace.py::test_replace_columnwise_no_op_inplace",
    "tests/copy_view/test_replace.py::test_replace_empty_list",
    "tests/copy_view/test_replace.py::test_replace_inplace[1.5]",
    "tests/copy_view/test_replace.py::test_replace_inplace[to_replace1]",
    "tests/copy_view/test_replace.py::test_replace_inplace[to_replace2]",
    "tests/copy_view/test_replace.py::test_replace_inplace_reference_no_op[100.5]",
    "tests/copy_view/test_replace.py::test_replace_inplace_reference_no_op[a]",
    "tests/copy_view/test_replace.py::test_replace_list_categorical",
    "tests/copy_view/test_replace.py::test_replace_list_multiple_elements_inplace",
    "tests/copy_view/test_replace.py::test_replace_list_none",
    "tests/copy_view/test_replace.py::test_replace_listlike",
    "tests/copy_view/test_replace.py::test_replace_listlike_inplace",
    "tests/copy_view/test_replace.py::test_replace_mask_all_false_second_block",
    "tests/copy_view/test_replace.py::test_replace_object_list_inplace[None]",
    "tests/copy_view/test_replace.py::test_replace_object_list_inplace[d]",
    "tests/copy_view/test_replace.py::test_replace_regex_inplace",
    "tests/copy_view/test_replace.py::test_replace_regex_inplace_no_op",
    "tests/copy_view/test_replace.py::test_replace_to_replace_wrong_dtype",
    "tests/copy_view/test_setitem.py::test_set_column_with_series",
    "tests/copy_view/test_setitem.py::test_set_columns_with_dataframe",
    "tests/copy_view/test_setitem.py::test_setitem_series_no_copy",
    "tests/copy_view/test_setitem.py::test_setitem_series_no_copy_single_block",
    "tests/copy_view/test_setitem.py::test_setitem_series_no_copy_split_block",
    "tests/copy_view/test_util.py::test_get_array_masked",
    "tests/copy_view/test_util.py::test_get_array_numpy",
    "tests/dtypes/cast/test_construct_from_scalar.py::test_cast_1d_array_like_from_timedelta",
    "tests/dtypes/cast/test_construct_from_scalar.py::test_cast_1d_array_like_from_timestamp",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right0-left0]",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right1-left1]",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right2-left2]",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right3-left3]",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right4-left4]",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right5-left5]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes16-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes17-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes18-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes19-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes20-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes21-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes22-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes23-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes24-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes25-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes30-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_raises_empty_input",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtype::test_repr_range_categories",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_equality[None-False]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_equality[None-None]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_equality[None-True]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_from_categorical_dtype_identity",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_nan_invalid",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_non_unique_invalid",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_basic_dtype",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_closed_mismatch",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_closed_must_match",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction[Interval[int64]]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction[interval[int64]]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_allows_closed_none[Interval[int64]]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_allows_closed_none[int64]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_allows_closed_none[interval[int64]]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_allows_closed_none[subtype3]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_from_string",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_generic[Interval]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_generic[interval]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_not_supported[<U10]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_equality",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_hash_vs_equality",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_name_repr_generic[Interval]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_name_repr_generic[interval]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_unpickling_without_closed",
    "tests/dtypes/test_dtypes.py::test_registry[CategoricalDtype]",
    "tests/dtypes/test_dtypes.py::test_registry[DatetimeTZDtype]",
    "tests/dtypes/test_dtypes.py::test_registry[IntervalDtype]",
    "tests/dtypes/test_dtypes.py::test_registry[PeriodDtype]",
    "tests/dtypes/test_dtypes.py::test_registry_find[category-expected5]",
    "tests/dtypes/test_dtypes.py::test_registry_find[datetime64[ns, US/Eastern]-expected6]",
    "tests/dtypes/test_dtypes.py::test_registry_find[interval-expected1]",
    "tests/dtypes/test_dtypes.py::test_registry_find[interval[datetime64[ns], left]-expected3]",
    "tests/dtypes/test_dtypes.py::test_registry_find[interval[int64, neither]-expected2]",
    "tests/dtypes/test_dtypes.py::test_registry_find[period[D]-expected4]",
    "tests/dtypes/test_generic.py::test_setattr_warnings",
    "tests/dtypes/test_inference.py::TestTypeInference::test_date",
    "tests/dtypes/test_inference.py::test_is_scipy_sparse[dok]",
    "tests/dtypes/test_missing.py::test_array_equivalent_series[val5]",
    "tests/dtypes/test_missing.py::test_array_equivalent_series[val6]",
    "tests/dtypes/test_missing.py::test_array_equivalent_series[val7]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_astype_string[string[pyarrow]]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_astype_string[string[python]]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__eq__-Index]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__ge__-Index]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__gt__-Index]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__le__-Index]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__lt__-Index]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__ne__-Index]",
    "tests/extension/json/test_json.py::TestJSONArray::test_astype_string[string[pyarrow]]",
    "tests/extension/json/test_json.py::TestJSONArray::test_astype_string[string[python]]",
    "tests/extension/json/test_json.py::TestJSONArray::test_concat_mixed_dtypes",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cummax-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cummax-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cummin-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cummin-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cumprod-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cumprod-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[decimal128(7, 3)-cummax-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[decimal128(7, 3)-cummin-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[decimal128(7, 3)-cumsum-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[decimal128(7, 3)-string[pyarrow]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[decimal128(7, 3)-string[python]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[ms]-string[pyarrow]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[ms]-string[python]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[ns]-string[pyarrow]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[ns]-string[python]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[s]-string[pyarrow]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[s]-string[python]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[us]-string[pyarrow]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[us]-string[python]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[decimal128(7, 3)-eq]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[decimal128(7, 3)-ge]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[decimal128(7, 3)-gt]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[decimal128(7, 3)-le]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[decimal128(7, 3)-lt]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[decimal128(7, 3)-ne]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[decimal128(7, 3)-mean-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[decimal128(7, 3)-median-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[decimal128(7, 3)-prod-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[decimal128(7, 3)-sum-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[double-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[float-mean-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[float-prod-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[float-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[float-std-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[float-sum-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[float-var-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[int16-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[int32-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[int64-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[int8-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[uint16-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[uint32-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[uint64-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[uint8-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_boolean[timestamp[ms]-any-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_boolean[timestamp[ns]-any-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_boolean[timestamp[s]-any-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_boolean[timestamp[us]-any-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[float-kurt-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int16-kurt-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int32-kurt-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int64-kurt-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int8-kurt-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint16-kurt-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint32-kurt-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint64-kurt-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint8-kurt-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[uint8]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_and",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_and_scalar[False-expected3]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_and_scalar[None-expected0]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_and_scalar[other1-expected1]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_or",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_or_scalar[None-expected0]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_or_scalar[True-expected2]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_or_scalar[other1-expected1]",
    "tests/extension/test_arrow.py::test_arrow_floordiv_floating_0_divisor[float[pyarrow]]",
    "tests/extension/test_arrow.py::test_arrow_floordiv_integral_invalid[pa_type0]",
    "tests/extension/test_arrow.py::test_arrow_floordiv_integral_invalid[pa_type1]",
    "tests/extension/test_arrow.py::test_arrow_floordiv_integral_invalid[pa_type2]",
    "tests/extension/test_arrow.py::test_arrow_floordiv_integral_invalid[pa_type3]",
    "tests/extension/test_arrow.py::test_arrow_floordiv_large_integral_result[uint64[pyarrow]]",
    "tests/extension/test_arrow.py::test_arrow_floordiv_larger_divisor[pa_type0]",
    "tests/extension/test_arrow.py::test_arrow_floordiv_larger_divisor[pa_type1]",
    "tests/extension/test_arrow.py::test_arrow_floordiv_larger_divisor[pa_type2]",
    "tests/extension/test_arrow.py::test_astype_errors_ignore",
    "tests/extension/test_arrow.py::test_bitwise[pa_type0]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type1]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type2]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type3]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type4]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type5]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type6]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type7]",
    "tests/extension/test_arrow.py::test_boolean_reduce_series_all_null[any-False]",
    "tests/extension/test_arrow.py::test_comparison_not_propagating_arrow_error",
    "tests/extension/test_arrow.py::test_dt_day_month_name[day_name-Sunday]",
    "tests/extension/test_arrow.py::test_dt_day_month_name[month_name-January]",
    "tests/extension/test_arrow.py::test_dt_is_month_start_end",
    "tests/extension/test_arrow.py::test_dt_is_quarter_start_end",
    "tests/extension/test_arrow.py::test_dt_is_year_start_end",
    "tests/extension/test_arrow.py::test_dt_isocalendar",
    "tests/extension/test_arrow.py::test_dt_properties[day-2]",
    "tests/extension/test_arrow.py::test_dt_properties[day_of_year-2]",
    "tests/extension/test_arrow.py::test_dt_properties[dayofyear-2]",
    "tests/extension/test_arrow.py::test_dt_properties[hour-3]",
    "tests/extension/test_arrow.py::test_dt_properties[is_leap_year-False]",
    "tests/extension/test_arrow.py::test_dt_properties[microsecond-5]",
    "tests/extension/test_arrow.py::test_dt_properties[minute-4]",
    "tests/extension/test_arrow.py::test_dt_properties[month-1]",
    "tests/extension/test_arrow.py::test_dt_properties[nanosecond-6]",
    "tests/extension/test_arrow.py::test_dt_properties[quarter-1]",
    "tests/extension/test_arrow.py::test_dt_properties[second-7]",
    "tests/extension/test_arrow.py::test_dt_properties[weekday-0]",
    "tests/extension/test_arrow.py::test_dt_properties[year-2023]",
    "tests/extension/test_arrow.py::test_dt_strftime",
    "tests/extension/test_arrow.py::test_dt_to_pydatetime",
    "tests/extension/test_arrow.py::test_dt_to_pytimedelta",
    "tests/extension/test_arrow.py::test_from_arrow_respecting_given_dtype",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[bool]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[double]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[duration[ms]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[duration[ns]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[duration[s]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[duration[us]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[float]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[int16]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[int32]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[int64]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[int8]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[string]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[timestamp[ms]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[timestamp[ns]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[timestamp[s]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[timestamp[us]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[uint16]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[uint32]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[uint64]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[uint8]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[bool]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[duration[ms]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[duration[ns]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[duration[s]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[duration[us]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[timestamp[ms]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[timestamp[ns]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[timestamp[s]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[timestamp[us]]",
    "tests/extension/test_arrow.py::test_pow_missing_operand",
    "tests/extension/test_arrow.py::test_quantile[bool-0.5-higher]",
    "tests/extension/test_arrow.py::test_quantile[bool-0.5-linear]",
    "tests/extension/test_arrow.py::test_quantile[bool-0.5-lower]",
    "tests/extension/test_arrow.py::test_quantile[bool-0.5-midpoint]",
    "tests/extension/test_arrow.py::test_quantile[bool-0.5-nearest]",
    "tests/extension/test_arrow.py::test_quantile[bool-quantile1-higher]",
    "tests/extension/test_arrow.py::test_quantile[bool-quantile1-linear]",
    "tests/extension/test_arrow.py::test_quantile[bool-quantile1-lower]",
    "tests/extension/test_arrow.py::test_quantile[bool-quantile1-midpoint]",
    "tests/extension/test_arrow.py::test_quantile[bool-quantile1-nearest]",
    "tests/extension/test_arrow.py::test_quantile[timestamp[ns]-0.5-nearest]",
    "tests/extension/test_arrow.py::test_quantile[timestamp[ns]-quantile1-higher]",
    "tests/extension/test_arrow.py::test_quantile[timestamp[ns]-quantile1-lower]",
    "tests/extension/test_arrow.py::test_round",
    "tests/extension/test_arrow.py::test_str_extract_non_symbolic",
    "tests/extension/test_arrow.py::test_str_find[ab-0-None-exp0-exp_typ0]",
    "tests/extension/test_arrow.py::test_str_len",
    "tests/extension/test_arrow.py::test_str_repeat_unsupported",
    "tests/extension/test_arrow.py::test_str_replace_unsupported[pat-b]",
    "tests/extension/test_arrow.py::test_str_starts_ends_with_all_nulls_empty_tuple[endswith]",
    "tests/extension/test_arrow.py::test_str_starts_ends_with_all_nulls_empty_tuple[startswith]",
    "tests/extension/test_categorical.py::Test2DCompat::test_copy_order",
    "tests/extension/test_categorical.py::TestCategorical::test_array_interface_copy",
    "tests/extension/test_categorical.py::TestCategorical::test_astype_own_type[False]",
    "tests/extension/test_categorical.py::TestCategorical::test_in_numeric_groupby",
    "tests/extension/test_categorical.py::TestCategorical::test_is_extension_array_dtype",
    "tests/extension/test_categorical.py::TestCategorical::test_reduce_series_boolean[all-False]",
    "tests/extension/test_categorical.py::TestCategorical::test_reduce_series_boolean[all-True]",
    "tests/extension/test_categorical.py::TestCategorical::test_reduce_series_boolean[any-False]",
    "tests/extension/test_categorical.py::TestCategorical::test_reduce_series_boolean[any-True]",
    "tests/extension/test_categorical.py::TestCategorical::test_searchsorted[True]",
    "tests/extension/test_categorical.py::TestCategorical::test_series_constructor_scalar_na_with_index",
    "tests/extension/test_categorical.py::TestCategorical::test_to_numpy",
    "tests/extension/test_common.py::test_ellipsis_index",
    "tests/extension/test_common.py::test_is_extension_array_dtype[dtype0]",
    "tests/extension/test_common.py::test_is_extension_array_dtype[dtype1]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_EA_types[US/Central-c]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_EA_types[US/Central-python]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_array_interface_copy[US/Central]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_astype_own_type[US/Central-False]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_is_extension_array_dtype[US/Central]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_reduce_series_numeric[US/Central-mean-False]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_reduce_series_numeric[US/Central-mean-True]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_reduce_series_numeric[US/Central-std-False]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_reduce_series_numeric[US/Central-std-True]",
    "tests/extension/test_interval.py::TestIntervalArray::test_EA_types[c]",
    "tests/extension/test_interval.py::TestIntervalArray::test_EA_types[python]",
    "tests/extension/test_interval.py::TestIntervalArray::test_array_interface_copy",
    "tests/extension/test_interval.py::TestIntervalArray::test_astype_own_type[False]",
    "tests/extension/test_interval.py::TestIntervalArray::test_compare_array[eq]",
    "tests/extension/test_interval.py::TestIntervalArray::test_compare_array[ne]",
    "tests/extension/test_interval.py::TestIntervalArray::test_getitem_scalar",
    "tests/extension/test_interval.py::TestIntervalArray::test_in_numeric_groupby",
    "tests/extension/test_interval.py::TestIntervalArray::test_is_extension_array_dtype",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[BooleanDtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[BooleanDtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Float32Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Float32Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Float64Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Float64Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int16Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int16Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int32Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int32Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int64Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int64Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int8Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int8Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt16Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt16Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt32Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt32Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt64Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt64Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt8Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt8Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[Float64Dtype-cummax-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[Float64Dtype-cummax-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[Float64Dtype-cummin-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[Float64Dtype-cummin-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt16Dtype-cumprod-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt16Dtype-cumprod-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt16Dtype-cumsum-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt16Dtype-cumsum-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt32Dtype-cumprod-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt32Dtype-cumprod-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt32Dtype-cumsum-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt32Dtype-cumsum-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt64Dtype-cumprod-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt64Dtype-cumprod-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt64Dtype-cumsum-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt64Dtype-cumsum-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt8Dtype-cumprod-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt8Dtype-cumprod-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt8Dtype-cumsum-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt8Dtype-cumsum-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[Float32Dtype-__floordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[Float64Dtype-__floordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__floordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__pow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__rfloordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__rsub__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__sub__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Float32Dtype-__floordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Float64Dtype-__floordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int16Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int32Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int64Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int8Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt16Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt32Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt64Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt8Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[Float32Dtype-__floordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[Float64Dtype-__floordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[BooleanDtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Float32Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Float64Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Int16Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Int32Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Int64Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Int8Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[UInt16Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[UInt32Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[UInt64Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[UInt8Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_str[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_string[Float32Dtype-string[pyarrow]]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_string[Float32Dtype-string[python]]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_string[Float64Dtype-string[pyarrow]]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_string[Float64Dtype-string[python]]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reduce_frame[Float32Dtype-skew-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[UInt8Dtype]",
    "tests/extension/test_numpy.py::Test2DCompat::test_copy_order[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cummax-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cummax-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cummin-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cummin-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cumprod-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cumprod-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cumsum-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cumsum-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cummax-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cummax-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cummin-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cummin-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cumsum-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cumsum-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-argmax-False--1]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-argmax-True-0]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-argmin-False--1]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-argmin-True-2]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-idxmax-False-nan]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-idxmax-True-0]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-idxmin-False-nan]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-idxmin-True-2]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_array_interface[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_array_interface_copy[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_array_interface_copy[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_array_type[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_array_type[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_object_frame[object-data]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_object_frame[object-data_missing]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_own_type[float-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_own_type[object-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_string[object-string[pyarrow]]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_string[object-string[python]]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_concat[float-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_concat[float-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_concat[object-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_concat[object-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_contains[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_constructor_from_dict[float-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_constructor_from_dict[float-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_constructor_from_dict[object-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_constructor_from_dict[object-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_from_series[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_from_series[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize_empty[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize_empty[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize_equivalence[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize_equivalence[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_getitem_boolean_array_mask[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_getitem_boolean_array_mask[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_getitem_mask[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_getitem_mask[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_groupby_extension_transform[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_grouping_grouper[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_grouping_grouper[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_hash_pandas_object[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_hash_pandas_object_works[object-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_hash_pandas_object_works[object-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_index_from_array[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_index_from_array[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_is_extension_array_dtype[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_pandas_array_dtype[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[float-all-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[float-all-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[float-any-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[float-any-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[object-all-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[object-all-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[object-any-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[object-any-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-count-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-count-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-kurt-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-kurt-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-max-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-max-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-mean-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-mean-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-median-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-median-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-min-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-min-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-prod-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-prod-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-sem-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-sem-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-skew-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-skew-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-std-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-std-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-sum-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-sum-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-var-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-var-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-max-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-max-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-min-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-min-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-sum-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-sum-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reindex[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_series_repr[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_set_frame_overwrite_object[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_set_frame_overwrite_object[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_2d_values[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_stack[object-False-columns1]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_stack[object-True-columns1]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_take_series[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_to_numpy[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[float-absolute]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[float-negative]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[float-positive]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[object-absolute]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[object-negative]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[object-positive]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[float-<lambda>-Series]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[float-unique-<lambda>]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[float-unique-Series]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[object-<lambda>-<lambda>]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[object-<lambda>-Series]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[object-unique-<lambda>]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[object-unique-Series]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_value_counts_with_normalize[object]",
    "tests/extension/test_period.py::Test2DCompat::test_copy_order[2D]",
    "tests/extension/test_period.py::Test2DCompat::test_copy_order[D]",
    "tests/extension/test_period.py::TestPeriodArray::test_EA_types[2D-c]",
    "tests/extension/test_period.py::TestPeriodArray::test_EA_types[2D-python]",
    "tests/extension/test_period.py::TestPeriodArray::test_EA_types[D-c]",
    "tests/extension/test_period.py::TestPeriodArray::test_EA_types[D-python]",
    "tests/extension/test_period.py::TestPeriodArray::test_array_interface_copy[2D]",
    "tests/extension/test_period.py::TestPeriodArray::test_array_interface_copy[D]",
    "tests/extension/test_period.py::TestPeriodArray::test_astype_own_type[2D-False]",
    "tests/extension/test_period.py::TestPeriodArray::test_astype_own_type[D-False]",
    "tests/extension/test_period.py::TestPeriodArray::test_is_extension_array_dtype[2D]",
    "tests/extension/test_period.py::TestPeriodArray::test_is_extension_array_dtype[D]",
    "tests/extension/test_sparse.py::TestSparseArray::test_EA_types[0-c]",
    "tests/extension/test_sparse.py::TestSparseArray::test_EA_types[0-python]",
    "tests/extension/test_sparse.py::TestSparseArray::test_EA_types[nan-c]",
    "tests/extension/test_sparse.py::TestSparseArray::test_EA_types[nan-python]",
    "tests/extension/test_sparse.py::TestSparseArray::test_array_interface_copy[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_array_interface_copy[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_astype_own_type[0-False]",
    "tests/extension/test_sparse.py::TestSparseArray::test_astype_own_type[nan-False]",
    "tests/extension/test_sparse.py::TestSparseArray::test_fillna_copy_frame[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_fillna_copy_frame[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_fillna_copy_series[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_fillna_copy_series[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_is_extension_array_dtype[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_is_extension_array_dtype[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_shift_0_periods[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_shift_0_periods[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[0-absolute]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[0-negative]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[0-positive]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[nan-absolute]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[nan-negative]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[nan-positive]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow_numpy-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow_numpy-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow_numpy-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow_numpy-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[python-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[python-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[python-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[python-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[pyarrow]-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[pyarrow]-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[pyarrow]-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[pyarrow]-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[python]-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[python]-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[python]-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[python]-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[pyarrow]-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[pyarrow]-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[pyarrow]-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[pyarrow]-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[python]-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[python]-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[python]-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[python]-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[pyarrow-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[pyarrow-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[pyarrow_numpy-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[pyarrow_numpy-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[python-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[python-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=str[pyarrow]-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=str[pyarrow]-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=str[python]-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=str[python]-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=string[pyarrow]-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=string[pyarrow]-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=string[python]-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=string[python]-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[pyarrow-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[pyarrow-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[pyarrow_numpy-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[pyarrow_numpy-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[python-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[python-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=str[pyarrow]-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=str[pyarrow]-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=str[python]-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=str[python]-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=string[pyarrow]-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=string[pyarrow]-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=string[python]-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=string[python]-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow-False-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow-False-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow-True-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow-True-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow_numpy-False-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow_numpy-False-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow_numpy-True-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow_numpy-True-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[python-False-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[python-False-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[python-True-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[python-True-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[pyarrow-False-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[pyarrow-True-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[pyarrow_numpy-False-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[pyarrow_numpy-True-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[python-False-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[python-True-True]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_unary_ufunc_dunder_equivalence[pyarrow-False-positive]",
    "tests/extension/test_string.py::TestStringArray::test_unary_ufunc_dunder_equivalence[pyarrow-True-positive]",
    "tests/extension/test_string.py::TestStringArray::test_unary_ufunc_dunder_equivalence[pyarrow_numpy-False-positive]",
    "tests/extension/test_string.py::TestStringArray::test_unary_ufunc_dunder_equivalence[pyarrow_numpy-True-positive]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[python-True]",
    "tests/frame/constructors/test_from_records.py::TestFromRecords::test_from_records_misc_brokenness3",
    "tests/frame/constructors/test_from_records.py::TestFromRecords::test_from_records_with_datetimes",
    "tests/frame/indexing/test_getitem.py::TestGetitemCallable::test_loc_multiindex_columns_one_level",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_fancy_getitem_slice_mixed",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_getitem_setitem_boolean_misaligned",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_getitem_setitem_float_labels",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_getitem_setitem_ix_bool_keyerror[False]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_getitem_setitem_ix_bool_keyerror[True]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_iloc_col_slice_view",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_iloc_row_slice_view",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_iloc_setitem_nullable_2d_values",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_loc_bool_multiindex[bool-indexer1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_loc_bool_multiindex[boolean-indexer1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_loc_internals_not_updated_correctly",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_loc_setitem_reordering_with_all_true_indexer[col0-False]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_loc_setitem_reordering_with_all_true_indexer[col0-True]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_loc_setitem_reordering_with_all_true_indexer[col0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_set_2d_casting_date_to_int[col0-indexer0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_set_2d_casting_date_to_int[col1-indexer0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setitem_boolean",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setitem_mixed_datetime",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setitem_single_column_mixed_datetime",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float32-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float32-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float32-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float64-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float64-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float64-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int16-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int16-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int16-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int32-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int32-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int32-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int64-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int64-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int64-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int8-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int8-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int8-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt16-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt16-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt16-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt32-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt32-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt32-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt64-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt64-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt64-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt8-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt8-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt8-null2]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key0]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key1]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key2]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key3]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key4]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key5]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated_multiindex[key2]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated_multiindex[key3]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated_multiindex[key4]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated_multiindex[key5]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key0]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key1]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key2]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key3]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key4]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key5]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_on_multiindex_one_level",
    "tests/frame/indexing/test_indexing.py::test_add_new_column_infer_string",
    "tests/frame/indexing/test_indexing.py::test_adding_new_conditional_column",
    "tests/frame/indexing/test_indexing.py::test_adding_new_conditional_column_with_string[dtype1-True]",
    "tests/frame/indexing/test_indexing.py::test_adding_new_conditional_column_with_string[object-False]",
    "tests/frame/indexing/test_indexing.py::test_adding_new_conditional_column_with_string[string[pyarrow_numpy]-True]",
    "tests/frame/indexing/test_indexing.py::test_object_casting_indexing_wraps_datetimelike",
    "tests/frame/indexing/test_insert.py::TestDataFrameInsert::test_insert",
    "tests/frame/indexing/test_insert.py::TestDataFrameInsert::test_insert_item_cache",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItem::test_setitem_list_missing_columns[columns3-box3-expected3]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemWithExpansion::test_setitem_listlike_views",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetitemCopyViewSemantics::test_iloc_setitem_view_2dblock",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetitemCopyViewSemantics::test_setitem_listlike_key_scalar_value_not_inplace[value2]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetitemCopyViewSemantics::test_setitem_partial_column_inplace[False]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetitemCopyViewSemantics::test_setitem_same_dtype_not_inplace[value2]",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_bug_transposition",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_dataframe_col_match",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_datetime",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_datetimelike_noop[timedelta64[ns]]",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_ndframe_align",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_set[default]",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_set[mixed_int]",
    "tests/frame/indexing/test_where.py::test_where_int_downcasting_deprecated",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Float32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Float64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Int16]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Int32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Int64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Int8]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-UInt16]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-UInt32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-UInt64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-UInt8]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Float32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Float64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Int16]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Int32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Int64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Int8]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-UInt16]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-UInt32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-UInt64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-UInt8]",
    "tests/frame/indexing/test_xs.py::TestXSWithMultiIndex::test_xs_droplevel_false_view",
    "tests/frame/indexing/test_xs.py::TestXSWithMultiIndex::test_xs_integer_key",
    "tests/frame/methods/test_asfreq.py::TestAsFreq::test_asfreq_resample_set_correct_freq[DataFrame]",
    "tests/frame/methods/test_asfreq.py::TestAsFreq::test_asfreq_resample_set_correct_freq[Series]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_arg_for_errors_dictlist",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_cast_nan_inf_int[inf-int32]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_cast_nan_inf_int[inf-int64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_dt64_to_string[DataFrame-None]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_dt64_to_string[Series-None]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_dt64tz",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_dt64tz_to_str",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_str_float",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_td64_to_string[DataFrame]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_td64_to_string[Series]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_with_exclude_string",
    "tests/frame/methods/test_astype.py::test_astype_to_string_not_modifying_input[pyarrow_numpy-None]",
    "tests/frame/methods/test_astype.py::test_astype_to_string_not_modifying_input[pyarrow_numpy-val4]",
    "tests/frame/methods/test_at_time.py::TestAtTime::test_at_time_axis[0]",
    "tests/frame/methods/test_at_time.py::TestAtTime::test_at_time_axis[index]",
    "tests/frame/methods/test_clip.py::TestDataFrameClip::test_clip_int_data_with_float_bound",
    "tests/frame/methods/test_clip.py::TestDataFrameClip::test_dataframe_clip",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_convert_datatime_correctly[data11-data21-data_expected1]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_convert_datatime_correctly[data12-data22-data_expected2]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_convert_datatime_correctly[data13-data23-data_expected3]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_timezone[ms]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_timezone[ns]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_timezone[s]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_timezone[us]",
    "tests/frame/methods/test_combine_first.py::test_combine_first_timestamp_bug[NAType-scalar10-scalar20]",
    "tests/frame/methods/test_combine_first.py::test_combine_first_timestamp_bug[NaTType-scalar10-scalar20]",
    "tests/frame/methods/test_combine_first.py::test_combine_first_timestamp_bug[NaTType-scalar12-scalar22]",
    "tests/frame/methods/test_combine_first.py::test_combine_first_timestamp_bug_NaT",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes[pyarrow-False-expected0]",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes[pyarrow_numpy-False-expected0]",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes[pyarrow_numpy-True-Int32]",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes[python-False-expected0]",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes_avoid_block_splitting",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes_pyarrow_to_np_nullable",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_pyarrow_engine_lines_false",
    "tests/frame/methods/test_copy.py::TestCopy::test_copy_consolidates",
    "tests/frame/methods/test_count.py::TestDataFrameCount::test_count",
    "tests/frame/methods/test_cov_corr.py::TestDataFrameCorr::test_corr_item_cache",
    "tests/frame/methods/test_diff.py::TestDataFrameDiff::test_diff_all_int_dtype[int16]",
    "tests/frame/methods/test_diff.py::TestDataFrameDiff::test_diff_all_int_dtype[int8]",
    "tests/frame/methods/test_drop.py::TestDataFrameDrop::test_drop",
    "tests/frame/methods/test_drop.py::TestDataFrameDrop::test_drop_multiindex_not_lexsorted",
    "tests/frame/methods/test_drop.py::TestDataFrameDrop::test_drop_multiindex_other_level_nan",
    "tests/frame/methods/test_dropna.py::TestDataFrameMissingData::test_drop_and_dropna_caching",
    "tests/frame/methods/test_dropna.py::TestDataFrameMissingData::test_dropna",
    "tests/frame/methods/test_dropna.py::TestDataFrameMissingData::test_dropna_multiple_axes",
    "tests/frame/methods/test_dropna.py::TestDataFrameMissingData::test_no_nans_in_frame[axis='index']",
    "tests/frame/methods/test_dropna.py::TestDataFrameMissingData::test_no_nans_in_frame[axis=0]",
    "tests/frame/methods/test_dtypes.py::TestDataFrameDataTypes::test_dtypes_timedeltas",
    "tests/frame/methods/test_equals.py::TestEquals::test_equals_different_blocks",
    "tests/frame/methods/test_explode.py::test_duplicate_index[input_dict0-input_index0-expected_dict0-expected_index0]",
    "tests/frame/methods/test_explode.py::test_duplicate_index[input_dict1-input_index1-expected_dict1-expected_index1]",
    "tests/frame/methods/test_explode.py::test_duplicate_index[input_dict2-input_index2-expected_dict2-expected_index2]",
    "tests/frame/methods/test_explode.py::test_duplicate_index[input_dict3-input_index3-expected_dict3-expected_index3]",
    "tests/frame/methods/test_explode.py::test_multi_columns_nan_empty",
    "tests/frame/methods/test_fillna.py::TestFillNA::test_fillna_on_column_view",
    "tests/frame/methods/test_info.py::test_info_memory",
    "tests/frame/methods/test_info.py::test_info_memory_usage_deep_not_pypy",
    "tests/frame/methods/test_info.py::test_usage_via_getsizeof",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interp_basic",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interp_inplace_row",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interp_nan_idx",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interp_rowwise",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_datetimelike_values[DataFrame]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_datetimelike_values[Series]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_inplace[DataFrame]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_inplace[Series]",
    "tests/frame/methods/test_join.py::TestDataFrameJoin::test_join",
    "tests/frame/methods/test_join.py::test_join_index",
    "tests/frame/methods/test_join.py::test_join_index_more",
    "tests/frame/methods/test_map.py::test_map",
    "tests/frame/methods/test_map.py::test_map_empty[<lambda>-expected0]",
    "tests/frame/methods/test_map.py::test_map_empty[<lambda>-expected2]",
    "tests/frame/methods/test_map.py::test_map_empty[round-expected0]",
    "tests/frame/methods/test_map.py::test_map_empty[round-expected2]",
    "tests/frame/methods/test_map.py::test_map_function_runs_once",
    "tests/frame/methods/test_map.py::test_map_na_ignore",
    "tests/frame/methods/test_pop.py::TestDataFramePop::test_pop",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_empty[linear-single]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_empty[nearest-table]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_datetime[ms]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_datetime[ns]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_datetime[s]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_datetime[us]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_empty_no_columns[linear-single]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_empty_no_columns[nearest-table]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_empty_no_rows_dt64[linear-single]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_item_cache[linear-single]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_item_cache[nearest-table]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_multi_empty[linear-single]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_multi_empty[nearest-table]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_nat[linear-single-ms]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_nat[linear-single-ns]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_nat[linear-single-s]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_nat[linear-single-us]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_empty_datelike[datetime64[ns]-expected_data1-expected_index1-0-datetime64[ns]]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea[Float64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea[Float64-Series]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea[Int64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea[Int64-Series]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_all_na[Float64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_all_na[Float64-Series]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_all_na[Int64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_all_na[Int64-Series]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_scalar[Float64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_scalar[Int64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_scalar[timedelta64[ns]-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_with_na[Float64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_with_na[Float64-Series]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_with_na[Int64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_with_na[Int64-Series]",
    "tests/frame/methods/test_rank.py::TestRank::test_rank_string_dtype[string=string[pyarrow]]",
    "tests/frame/methods/test_rank.py::TestRank::test_rank_string_dtype[string[pyarrow]-Int64]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_axis_style_raises",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_copies_ea",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx1-src_idx0]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx1-src_idx1]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx3-src_idx0]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx3-src_idx1]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx4-src_idx0]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx4-src_idx1]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx5-src_idx0]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx5-src_idx1]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_fill_value",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_int",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_nearest_tz[zoneinfo.ZoneInfo(key='US/Pacific')]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_nearest_tz[zoneinfo.ZoneInfo(key='UTC')]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_positional_raises",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Float32]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Float64]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Int16]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Int32]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Int64]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Int8]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[UInt16]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[UInt32]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[UInt64]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[UInt8]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_uint_dtypes_fill_value[uint16]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_uint_dtypes_fill_value[uint32]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_uint_dtypes_fill_value[uint64]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_uint_dtypes_fill_value[uint8]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_with_multi_index",
    "tests/frame/methods/test_reindex.py::TestReindexSetIndex::test_setitem_reset_index_dtypes",
    "tests/frame/methods/test_rename.py::TestRename::test_rename",
    "tests/frame/methods/test_rename.py::TestRename::test_rename_axis_style_raises",
    "tests/frame/methods/test_rename.py::TestRename::test_rename_inplace",
    "tests/frame/methods/test_rename.py::TestRename::test_rename_mapper_and_positional_arguments_raises",
    "tests/frame/methods/test_rename.py::TestRename::test_rename_no_mappings_raises",
    "tests/frame/methods/test_rename.py::TestRename::test_rename_nocopy",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_NAT_with_None",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_NA_with_None",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_after_convert_dtypes",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[DataFrame-array-data0-to_replace0-value0-expected0]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[DataFrame-tuple-data0-to_replace0-value0-expected0]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[Series-array-data0-to_replace0-value0-expected0]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[Series-tuple-data0-to_replace0-value0-expected0]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_method[nan-bfill-expected1]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_mixed_int_block_splitting",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_no_replacement_dtypes[nan-int64]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_no_replacement_dtypes[value1-int64]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_value_is_none",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_value_none_dtype_numeric[2.0]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_value_none_dtype_numeric[2]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_value_none_dtype_numeric[nan]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_duplicate_columns_allow[False-False]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_duplicate_columns_allow[False-True]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_duplicate_columns_allow[True-False]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_duplicate_columns_default[False]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_duplicate_columns_default[True]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_empty_rangeindex",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_level_missing[idx_lev0]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_level_missing[idx_lev1]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_multiindex_columns",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_with_datetimeindex_cols[2012-12-31]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_exclude_include_int[include0]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_exclude_include_int[include2]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_float_dtype[expected0-float]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_float_dtype[expected1-float]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-S1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-U1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-bytes_]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-str0]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-str1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-str_]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-S1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-U1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-bytes_]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-str0]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-str1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-str_]",
    "tests/frame/methods/test_set_axis.py::TestDataFrameSetAxis::test_set_axis_copy",
    "tests/frame/methods/test_set_axis.py::TestSeriesSetAxis::test_set_axis_copy",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-False-None-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-False-None-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-B-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-B-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-None-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-None-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-test-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-test-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-False-None-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-False-None-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-B-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-B-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-None-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-None-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-test-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-test-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndexCustomLabelType::test_set_index_custom_label_hashable_iterable",
    "tests/frame/methods/test_shift.py::TestDataFrameShift::test_shift_bool",
    "tests/frame/methods/test_shift.py::TestDataFrameShift::test_shift_dt64values_axis1_invalid_fill[datetime64[ns]-False]",
    "tests/frame/methods/test_shift.py::TestDataFrameShift::test_shift_dt64values_axis1_invalid_fill[timedelta64[ns]-False]",
    "tests/frame/methods/test_shift.py::TestDataFrameShift::test_shift_dt64values_int_fill_deprecated",
    "tests/frame/methods/test_sort_index.py::TestDataFrameSortIndex::test_sort_index_intervalindex",
    "tests/frame/methods/test_sort_index.py::TestDataFrameSortIndex::test_sort_index_nan",
    "tests/frame/methods/test_sort_index.py::TestDataFrameSortIndex::test_sort_index_reorder_on_ops",
    "tests/frame/methods/test_sort_values.py::TestDataFrameSortValues::test_sort_values_by_empty_list",
    "tests/frame/methods/test_sort_values.py::TestDataFrameSortValues::test_sort_values_item_cache",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_gz_lineend",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_multi_index_header",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_from_csv1",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_from_csv2",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_from_csv5",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_headers",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_multiindex",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_na_quoting",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_quoting",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_single_level_multi_index",
    "tests/frame/methods/test_to_dict.py::TestDataFrameToDict::test_to_dict_index_false_error[series]",
    "tests/frame/methods/test_to_dict.py::TestDataFrameToDict::test_to_dict_orient_dtype[data1-Timestamp]",
    "tests/frame/methods/test_to_dict.py::TestDataFrameToDict::test_to_dict_scalar_constructor_orient_dtype[data4-Timestamp]",
    "tests/frame/methods/test_to_dict.py::test_to_dict_list_pd_scalars[val1]",
    "tests/frame/methods/test_to_numpy.py::TestToNumpy::test_to_numpy_copy",
    "tests/frame/methods/test_to_numpy.py::TestToNumpy::test_to_numpy_mixed_dtype_to_str",
    "tests/frame/methods/test_to_records.py::TestDataFrameToRecords::test_to_records_dt64tz_column",
    "tests/frame/methods/test_to_records.py::TestDataFrameToRecords::test_to_records_with_unicode_index",
    "tests/frame/methods/test_truncate.py::TestDataFrameTruncate::test_truncate_multiindex[DataFrame]",
    "tests/frame/methods/test_tz_localize.py::TestTZLocalize::test_tz_localize_copy_inplace_mutate[DataFrame-False]",
    "tests/frame/methods/test_tz_localize.py::TestTZLocalize::test_tz_localize_copy_inplace_mutate[DataFrame-True]",
    "tests/frame/methods/test_tz_localize.py::TestTZLocalize::test_tz_localize_copy_inplace_mutate[Series-False]",
    "tests/frame/methods/test_tz_localize.py::TestTZLocalize::test_tz_localize_copy_inplace_mutate[Series-True]",
    "tests/frame/methods/test_value_counts.py::test_data_frame_value_counts_dropna_false[NoneType]",
    "tests/frame/methods/test_value_counts.py::test_data_frame_value_counts_empty",
    "tests/frame/methods/test_value_counts.py::test_data_frame_value_counts_empty_normalize",
    "tests/frame/methods/test_value_counts.py::test_data_frame_value_counts_subset[NoneType-columns1]",
    "tests/frame/methods/test_value_counts.py::test_value_counts_with_missing_category",
    "tests/frame/methods/test_values.py::TestDataFrameValues::test_interleave_with_tzaware",
    "tests/frame/methods/test_values.py::TestDataFrameValues::test_values",
    "tests/frame/test_api.py::TestDataFrameMisc::test_display_max_dir_items",
    "tests/frame/test_api.py::TestDataFrameMisc::test_get_agg_axis",
    "tests/frame/test_api.py::TestDataFrameMisc::test_get_axis",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[DataFrame-False]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[DataFrame-None]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[DataFrame-True]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[Series-False]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[Series-None]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[Series-True]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_tab_completion",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_arithmetic_midx_cols_different_dtypes_different_order[numexpr]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_arithmetic_midx_cols_different_dtypes_different_order[python]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[numexpr-pow-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[numexpr-sub-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[numexpr-truediv-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[python-pow-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[python-sub-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[python-truediv-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_frame_with_frame_reindex[numexpr]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_frame_with_frame_reindex[python]",
    "tests/frame/test_arithmetic.py::TestFrameArithmeticUnsorted::test_inplace_ops_identity[numexpr]",
    "tests/frame/test_arithmetic.py::TestFrameArithmeticUnsorted::test_inplace_ops_identity[python]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[numexpr-ge-le]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[numexpr-gt-lt]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[numexpr-le-ge]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[numexpr-lt-gt]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[python-ge-le]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[python-gt-lt]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[python-le-ge]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[python-lt-gt]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_arith_flex_frame_corner[numexpr]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_arith_flex_frame_corner[python]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_broadcast_multiindex[numexpr-None]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_broadcast_multiindex[python-None]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_flex_add_scalar_fill_value[numexpr]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_flex_add_scalar_fill_value[python]",
    "tests/frame/test_arithmetic.py::test_arithmetic_multiindex_align[numexpr]",
    "tests/frame/test_arithmetic.py::test_arithmetic_multiindex_align[python]",
    "tests/frame/test_arithmetic.py::test_bool_frame_mult_float[numexpr]",
    "tests/frame/test_arithmetic.py::test_bool_frame_mult_float[python]",
    "tests/frame/test_arithmetic.py::test_frame_with_zero_len_series_corner_cases[numexpr]",
    "tests/frame/test_arithmetic.py::test_frame_with_zero_len_series_corner_cases[python]",
    "tests/frame/test_arithmetic.py::test_pow_with_realignment[numexpr]",
    "tests/frame/test_arithmetic.py::test_pow_with_realignment[python]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_columns_indexes_raise_on_sets",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_construction_from_ndarray_datetimelike",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_construction_from_set_raises[frozenset]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_construction_from_set_raises[set]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['+01:15'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['-02:15'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['Asia/Tokyo'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['US/Eastern'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['UTC'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['UTC+01:15'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['UTC-02:15'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['dateutil/Asia/Singapore'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['dateutil/US/Pacific'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[<UTC>-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[datetime.timezone(datetime.timedelta(seconds=3600))-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[datetime.timezone.utc-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[pytz.FixedOffset(-300)-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[pytz.FixedOffset(300)-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[tzlocal()-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[tzutc()-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[zoneinfo.ZoneInfo(key='US/Pacific')-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[zoneinfo.ZoneInfo(key='UTC')-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDtypeCoercion::test_floating_values_integer_dtype",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_1d_object_array_does_not_copy",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_2d_object_array_does_not_copy",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_construct_from_list_of_datetimes",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_construct_with_two_categoricalindex_series",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_with_nulls[arr0]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_with_nulls[arr1]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_cast",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_datetime64_index",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_multiindex",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_nan_key[None]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_nan_key_and_columns",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_timedelta64_index[<lambda>-pytimedelta]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_with_index",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_with_index_and_columns",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_error_msgs",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_for_list_with_dtypes",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_list_of_2d_raises",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_lists_to_object_dtype",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_maskedarray",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_maskedarray_hardened",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_maskedarray_nonfloat",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_maskedrecarray_dtype",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_mixed_dict_and_Series",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_rec",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_with_datetimes1",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_with_datetimes2",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_with_datetimes3",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_datetimelike_values_with_object_dtype[DataFrame-M]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_datetimelike_values_with_object_dtype[DataFrame-m]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_datetimelike_values_with_object_dtype[Series-M]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_datetimelike_values_with_object_dtype[Series-m]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_frame_string_inference",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_frame_string_inference_array_string_dtype",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_from_datetime_subclass",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_datetime[Series-None-datetime64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_timedelta[Series-None-timedelta64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_datetime64[DataFrame-dict]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_datetime64[Series-None]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_timedelta64[DataFrame-dict-datetime64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_timedelta64[DataFrame-dict-timedelta64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_timedelta64[Series-None-datetime64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_timedelta64[Series-None-timedelta64]",
    "tests/frame/test_iteration.py::TestIteration::test_iteritems",
    "tests/frame/test_iteration.py::TestIteration::test_iterrows_corner",
    "tests/frame/test_iteration.py::TestIteration::test_itertuples",
    "tests/frame/test_iteration.py::TestIteration::test_keys",
    "tests/frame/test_logical_ops.py::TestDataFrameLogicalOperators::test_different_dtypes_different_index_raises",
    "tests/frame/test_logical_ops.py::TestDataFrameLogicalOperators::test_logical_with_nas",
    "tests/frame/test_npfuncs.py::TestAsArray::test_np_sqrt",
    "tests/frame/test_query_eval.py::TestDataFrameQueryNumExprPython::test_query_numexpr_with_min_and_max_columns",
    "tests/frame/test_query_eval.py::TestDataFrameQueryPythonPython::test_query_numexpr_with_min_and_max_columns",
    "tests/frame/test_query_eval.py::TestDataFrameQueryWithMultiIndex::test_query_multiindex_get_index_resolvers",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_bool_frame[any]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_extra",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data1-True]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data11-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data13-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data14-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data16-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data3-True]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data30-True]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data32-True]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data5-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data7-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data9-True]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[any-data0-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[any-data31-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[any-data33-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_idxmax_idxmin_convert_dtypes[idxmax-expected_value0]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_idxmax_idxmin_convert_dtypes[idxmin-expected_value1]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_operators_timedelta64",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[True-ms-values0]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[True-s-values0]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[True-us-values0]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_min_max_dt64_api_consistency_empty_df",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_min_max_dt64_api_consistency_with_NaT",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reduction_timedelta_smallest_unit",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reduction_timestamp_smallest_unit",
    "tests/frame/test_reductions.py::TestNuisanceColumns::test_any_all_categorical_dtype_nuisance_column[all]",
    "tests/frame/test_reductions.py::TestNuisanceColumns::test_any_all_categorical_dtype_nuisance_column[any]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-False-kurt]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-False-mean]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-False-median]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-False-skew]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-True-kurt]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-True-mean]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-True-median]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-True-skew]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-False-kurt]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-False-mean]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-False-median]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-False-skew]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-True-kurt]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-True-mean]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-True-median]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-True-skew]",
    "tests/frame/test_reductions.py::test_sum_timedelta64_skipna_false",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_int_level_names[False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_int_level_names[True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_ints[False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_ints[True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_mixed_levels[False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_mixed_levels[True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[False-labels0-data0-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[False-labels0-data0-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[False-labels1-data1-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[False-labels1-data1-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[True-labels0-data0-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[True-labels0-data0-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[True-labels1-data1-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[True-labels1-data1-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_preserve_categorical_dtype[False-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_preserve_categorical_dtype[False-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_preserve_categorical_dtype[True-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_preserve_categorical_dtype[True-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_unstack[False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_unstack[True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_multi_level_rows_and_cols",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_nan_index2",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_nan_index3",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_non_unique_index_names[False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_non_unique_index_names[True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_not_consolidated",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_swaplevel_sortlevel[0]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_swaplevel_sortlevel[baz]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_to_series",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_unused_levels",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_multi_level_stack_categorical[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_multi_level_stack_categorical[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_level_name[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_level_name[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_mixed_dtype[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_multiple_bug[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_multiple_bug[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_multiple_out_of_bounds[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_multiple_out_of_bounds[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_nan_in_multiindex_columns[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_nan_in_multiindex_columns[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_nan_level[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_nan_level[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_nullable_dtype[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_nullable_dtype[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_order_with_unsorted_levels_multi_row_2[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_multiple[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_multiple[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_preserve_names[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_preserve_names[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_wrong_level_name[False-unstack]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_wrong_level_name[True-unstack]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_unstack_preserve_types",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_unstack_with_level_has_nan",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_unstack_with_missing_int_cast_to_float",
    "tests/frame/test_stack_unstack.py::test_unstack_sort_false[Series-float64]",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_asof",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_equals_subclass",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_frame_subclassing_and_slicing",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_subclass_stack_multi",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_subclass_stack_multi_mixed",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_subclass_unstack_multi",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_subclass_unstack_multi_mixed",
    "tests/frame/test_subclass.py::test_constructor_with_metadata_from_records",
    "tests/frame/test_ufunc.py::test_alignment_deprecation_enforced",
    "tests/frame/test_ufunc.py::test_binary_input_aligns_index[int64]",
    "tests/frame/test_unary.py::TestDataFrameUnaryOperators::test_pos_object_raises[df0]",
    "tests/frame/test_unary.py::TestDataFrameUnaryOperators::test_pos_raises[df0]",
    "tests/frame/test_unary.py::TestDataFrameUnaryOperators::test_unary_nullable",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[1-False-add]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[1-False-sub]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[1-True-add]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[1-True-sub]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[other1-False-add]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[other1-False-sub]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[other1-True-add]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[other1-True-sub]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs0-kwargs0]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs1-kwargs1]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs2-kwargs2]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs3-kwargs3]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs4-kwargs4]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs5-kwargs5]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs6-kwargs6]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_ndframe_getitem_caching_issue",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserve_getitem",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_frame",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[absolute]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[func0]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[func1]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[func2]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[func3]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[func4]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_to_frame",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_concat_raises[objs0-kwargs0]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter0-None]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter1-loc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter2-loc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter3-loc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter4-iloc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter5-iloc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter6-iloc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_series_raises",
    "tests/generic/test_duplicate_labels.py::test_dataframe_insert_raises",
    "tests/generic/test_duplicate_labels.py::test_inplace_raises[method0-True]",
    "tests/generic/test_duplicate_labels.py::test_inplace_raises[method1-True]",
    "tests/generic/test_duplicate_labels.py::test_inplace_raises[method2-False]",
    "tests/generic/test_frame.py::TestDataFrame::test_metadata_propagation_indiv",
    "tests/generic/test_frame.py::TestDataFrame::test_set_attribute",
    "tests/generic/test_generic.py::TestNDFrame::test_flags_identity[DataFrame]",
    "tests/generic/test_generic.py::TestNDFrame::test_flags_identity[Series]",
    "tests/generic/test_generic.py::TestNDFrame::test_numpy_transpose[DataFrame]",
    "tests/generic/test_generic.py::TestNDFrame::test_transpose_frame",
    "tests/generic/test_series.py::TestSeries::test_metadata_propagation_indiv",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[bool-dtype]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[bool-object]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[categorical]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[complex128]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[complex64]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[datetime-tz]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[datetime]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[float32]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[float64]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[int16]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[int32]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[int64]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[int8]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[interval]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[nullable_bool]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[nullable_float]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[nullable_int]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[nullable_uint]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[object]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[period]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[range]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[repeats]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[string-pyarrow]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[string-python]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[string]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[timedelta]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[uint16]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[uint32]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[uint64]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[uint8]",
    "tests/generic/test_to_xarray.py::TestSeriesToXArray::test_to_xarray_index_types[string-pyarrow]",
    "tests/generic/test_to_xarray.py::TestSeriesToXArray::test_to_xarray_index_types[string-python]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_multiple_with_as_index_false_subset_to_a_single_column",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[count]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[first]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[idxmax]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[idxmin]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[last]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[max]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[min]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[nunique]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-bool-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-bool-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-float32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-float32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-float64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-float64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-int32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-int32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-int64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-int64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-bool-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-bool-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-float32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-float32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-float64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-float64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-int32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-int32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-int64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-int64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-bool-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-bool-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-float32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-float32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-float64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-float64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-int32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-int32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-int64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-int64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-bool-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-bool-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-float32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-float32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-float64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-float64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-int32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-int32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-int64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-int64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-bool-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-bool-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-float32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-float32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-float64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-float64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-int32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-int32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-int64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-int64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_groupby_aggregate_directory[size]",
    "tests/groupby/aggregate/test_aggregate.py::test_groupby_aggregate_empty_key_empty_return",
    "tests/groupby/aggregate/test_aggregate.py::test_groupby_aggregation_multi_level_column",
    "tests/groupby/aggregate/test_aggregate.py::test_multiindex_custom_func[<lambda>0]",
    "tests/groupby/aggregate/test_aggregate.py::test_order_aggregate_multiple_funcs",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data1-prod-large_int-False]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data1-prod-large_int-True]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data1-sum-large_int-False]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data1-sum-large_int-True]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-mean-always_float-False]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-mean-always_float-True]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-median-always_float-False]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-median-always_float-True]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-var-always_float-False]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-var-always_float-True]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[mean]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[median]",
    "tests/groupby/aggregate/test_cython.py::test_cythonized_aggers[prod]",
    "tests/groupby/aggregate/test_cython.py::test_cythonized_aggers[sum]",
    "tests/groupby/aggregate/test_other.py::test_agg_over_numpy_arrays",
    "tests/groupby/methods/test_describe.py::test_frame_describe_tupleindex",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_group_diff_bool",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_group_diff_real_frame[int16]",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_group_diff_real_frame[int8]",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_group_diff_real_series[int16]",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_group_diff_real_series[int8]",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_shift_periods_freq",
    "tests/groupby/methods/test_nth.py::test_first_last_nth",
    "tests/groupby/methods/test_nth.py::test_first_last_nth_dtypes",
    "tests/groupby/methods/test_nth.py::test_groupby_last_first_nth_with_none[NoneType-nth]",
    "tests/groupby/methods/test_nth.py::test_negative_step",
    "tests/groupby/methods/test_nth.py::test_np_ints",
    "tests/groupby/methods/test_nth.py::test_nth",
    "tests/groupby/methods/test_nth.py::test_nth2",
    "tests/groupby/methods/test_nth.py::test_nth3",
    "tests/groupby/methods/test_nth.py::test_nth4",
    "tests/groupby/methods/test_nth.py::test_nth5",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[None-b]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[None-selection1]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[None-selection2]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[all-b]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[all-selection1]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[all-selection2]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[any-b]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[any-selection1]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[any-selection2]",
    "tests/groupby/methods/test_nth.py::test_nth_column_order",
    "tests/groupby/methods/test_nth.py::test_nth_indexed",
    "tests/groupby/methods/test_nth.py::test_nth_multi_grouper",
    "tests/groupby/methods/test_nth.py::test_nth_multi_index_as_expected",
    "tests/groupby/methods/test_nth.py::test_nth_with_na_object[NoneType--1]",
    "tests/groupby/methods/test_nth.py::test_nth_with_na_object[float0--1]",
    "tests/groupby/methods/test_nth.py::test_nth_with_na_object[float1--1]",
    "tests/groupby/methods/test_nth.py::test_slice[arg0-expected_rows0]",
    "tests/groupby/methods/test_nth.py::test_slice[arg1-expected_rows1]",
    "tests/groupby/methods/test_nth.py::test_slice[arg2-expected_rows2]",
    "tests/groupby/methods/test_nth.py::test_slice[arg3-expected_rows3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_NA_float[Float32]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_allNA_column[Float32]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float32-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float32-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float32-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float64-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float64-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float64-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int16-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int16-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int16-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int32-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int32-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int32-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int64-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int64-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int64-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int8-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int8-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int8-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt16-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt16-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt16-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt32-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt32-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt32-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt64-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt64-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt64-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt8-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt8-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt8-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Float32-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Float32-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Float64-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Float64-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int16-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int16-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int32-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int32-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int64-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int64-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int8-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int8-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt16-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt16-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt32-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt32-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt64-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt64-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt8-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt8-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_nonmulti_levels_order",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_raises_on_invalid_dtype[True-q1]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_with_arraylike_q_and_int_columns[q0-groupby0-frame_size0]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_with_arraylike_q_and_int_columns[q0-groupby0-frame_size1]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals0-b_vals0-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals0-b_vals0-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals0-b_vals0-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals1-b_vals1-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals1-b_vals1-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals1-b_vals1-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals2-b_vals2-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals2-b_vals2-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals2-b_vals2-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals0-b_vals0-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals0-b_vals0-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals0-b_vals0-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals1-b_vals1-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals1-b_vals1-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals1-b_vals1-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals2-b_vals2-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals2-b_vals2-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals2-b_vals2-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals0-b_vals0-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals0-b_vals0-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals0-b_vals0-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals1-b_vals1-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals1-b_vals1-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals2-b_vals2-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals2-b_vals2-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals0-b_vals0-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals0-b_vals0-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals0-b_vals0-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals1-b_vals1-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals1-b_vals1-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals1-b_vals1-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals2-b_vals2-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals2-b_vals2-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals2-b_vals2-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals0-b_vals0-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals0-b_vals0-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals0-b_vals0-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals1-b_vals1-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals1-b_vals1-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals1-b_vals1-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals2-b_vals2-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals2-b_vals2-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals2-b_vals2-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile_array",
    "tests/groupby/methods/test_quantile.py::test_quantile_array2",
    "tests/groupby/methods/test_quantile.py::test_quantile_missing_group_values_correct_results[key4-val4-expected_key4-expected_val4]",
    "tests/groupby/methods/test_rank.py::test_groupby_axis0_cummax_axis1",
    "tests/groupby/methods/test_rank.py::test_rank_avg_even_vals[True-int32]",
    "tests/groupby/methods/test_rank.py::test_rank_avg_even_vals[True-int64]",
    "tests/groupby/methods/test_rank.py::test_rank_avg_even_vals[True-uint32]",
    "tests/groupby/methods/test_rank.py::test_rank_avg_even_vals[True-uint64]",
    "tests/groupby/methods/test_size.py::test_size_series_masked_type_returns_Int64[Float64]",
    "tests/groupby/methods/test_size.py::test_size_series_masked_type_returns_Int64[Int64]",
    "tests/groupby/methods/test_size.py::test_size_series_masked_type_returns_Int64[boolean]",
    "tests/groupby/methods/test_size.py::test_size_sort[False-A]",
    "tests/groupby/methods/test_size.py::test_size_sort[False-B]",
    "tests/groupby/methods/test_size.py::test_size_sort[False-by2]",
    "tests/groupby/methods/test_size.py::test_size_strings[string=string[pyarrow]]",
    "tests/groupby/methods/test_size.py::test_size_strings[string[pyarrow]]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-False-None-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-False-None-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-True-False-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-True-False-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-True-True-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-True-True-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-False-None-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-False-None-False-count-function]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-False-None-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-False-None-True-proportion-function]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-False-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-False-False-count-function]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-False-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-False-True-proportion-function]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-True-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-True-False-count-function]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-True-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-True-True-proportion-function]",
    "tests/groupby/methods/test_value_counts.py::test_ambiguous_grouping",
    "tests/groupby/methods/test_value_counts.py::test_column_label_duplicates[False-level-columns1-expected_names1]",
    "tests/groupby/methods/test_value_counts.py::test_compound[False-None-expected_rows0-expected_count0-expected_group_size0-False-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_compound[False-None-expected_rows0-expected_count0-expected_group_size0-True-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_compound[True-False-expected_rows1-expected_count1-expected_group_size1-False-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_compound[True-False-expected_rows1-expected_count1-expected_group_size1-True-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_compound[True-True-expected_rows2-expected_count2-expected_group_size2-False-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_compound[True-True-expected_rows2-expected_count2-expected_group_size2-True-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_data_frame_value_counts_dropna[NoneType-False-count-False-expected_data1-expected_index1]",
    "tests/groupby/methods/test_value_counts.py::test_data_frame_value_counts_dropna[NoneType-True-proportion-False-expected_data1-expected_index1]",
    "tests/groupby/methods/test_value_counts.py::test_dropna_combinations[True-False-expected_rows2-expected_values2]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort[False-False-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort[False-True-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort[False-True-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort[True-False-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort[True-True-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[False-False-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[False-False-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[False-True-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[False-True-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[True-False-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[True-False-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[True-True-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[True-True-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_time_grouper[ms-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_time_grouper[ns-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_time_grouper[s-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_time_grouper[us-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[cummax-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[cummin-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[cumprod-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[cumsum-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[nunique-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[nunique-True]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[rank-False]",
    "tests/groupby/test_api.py::test_all_methods_categorized",
    "tests/groupby/test_api.py::test_tab_completion",
    "tests/groupby/test_apply.py::test_apply_concat_preserve_names",
    "tests/groupby/test_apply.py::test_apply_datetime_issue[group_column_dtlike0]",
    "tests/groupby/test_apply.py::test_apply_datetime_issue[group_column_dtlike1]",
    "tests/groupby/test_apply.py::test_apply_frame_concat_series",
    "tests/groupby/test_apply.py::test_apply_func_that_appends_group_to_list_without_copy",
    "tests/groupby/test_apply.py::test_apply_index_has_complex_internals[index3]",
    "tests/groupby/test_apply.py::test_apply_index_key_error_bug[index_values0]",
    "tests/groupby/test_apply.py::test_apply_index_key_error_bug[index_values1]",
    "tests/groupby/test_apply.py::test_apply_multiindex_fail",
    "tests/groupby/test_apply.py::test_apply_nonmonotonic_float_index[arg0-idx0]",
    "tests/groupby/test_apply.py::test_apply_nonmonotonic_float_index[arg2-idx2]",
    "tests/groupby/test_apply.py::test_apply_with_date_in_multiindex_does_not_convert_to_timestamp",
    "tests/groupby/test_apply.py::test_builtins_apply[jim-max]",
    "tests/groupby/test_apply.py::test_builtins_apply[jim-min]",
    "tests/groupby/test_apply.py::test_builtins_apply[keys1-max]",
    "tests/groupby/test_apply.py::test_builtins_apply[keys1-min]",
    "tests/groupby/test_apply.py::test_empty_df[agg-prod]",
    "tests/groupby/test_apply.py::test_empty_df[agg-skew]",
    "tests/groupby/test_apply.py::test_empty_df[agg-sum]",
    "tests/groupby/test_apply.py::test_empty_df[apply-<lambda>0]",
    "tests/groupby/test_apply.py::test_empty_df[apply-<lambda>1]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH10519]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH12155]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH20084]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH21417]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH2656]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH2936]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH7739 & GH10519]",
    "tests/groupby/test_apply.py::test_groupby_apply_all_none",
    "tests/groupby/test_apply.py::test_groupby_apply_none_first",
    "tests/groupby/test_apply.py::test_groupby_as_index_apply",
    "tests/groupby/test_apply.py::test_time_field_bug",
    "tests/groupby/test_apply_mutate.py::test_apply_function_with_indexing",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-count-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-first-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-idxmax-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-idxmin-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-last-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-max-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-mean-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-median-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-min-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-nunique-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-nunique-keys1-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-prod-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-size-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-std-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-sum-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-var-keys0-True]",
    "tests/groupby/test_categorical.py::test_apply[False]",
    "tests/groupby/test_categorical.py::test_apply[True]",
    "tests/groupby/test_categorical.py::test_as_index",
    "tests/groupby/test_categorical.py::test_category_order_apply[False-False-True-False-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[False-False-True-True-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[False-True-True-False-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[False-True-True-True-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-False-True-False-apply-multi]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-False-True-False-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-False-True-True-apply-multi]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-False-True-True-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-True-True-False-apply-multi]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-True-True-False-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-True-True-True-apply-multi]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-True-True-True-apply-range]",
    "tests/groupby/test_categorical.py::test_describe_categorical_columns",
    "tests/groupby/test_categorical.py::test_groupby_categorical_observed_nunique",
    "tests/groupby/test_categorical.py::test_observed[False]",
    "tests/groupby/test_categorical.py::test_observed_codes_remap[False]",
    "tests/groupby/test_categorical.py::test_observed_codes_remap[True]",
    "tests/groupby/test_categorical.py::test_observed_groups[True]",
    "tests/groupby/test_categorical.py::test_observed_groups_with_nan[True]",
    "tests/groupby/test_categorical.py::test_sort_datetimelike[False-False]",
    "tests/groupby/test_categorical.py::test_sort_datetimelike[True-False]",
    "tests/groupby/test_categorical.py::test_unstack_categorical",
    "tests/groupby/test_counting.py::TestCounting::test_ngroup_distinct",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string=string[pyarrow]]",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string=string[python]]",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string[pyarrow]]",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string[pyarrow_numpy]]",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string[python]]",
    "tests/groupby/test_cumulative.py::test_cummin_max_all_nan_column[boolean-cummax]",
    "tests/groupby/test_cumulative.py::test_cummin_max_all_nan_column[boolean-cummin]",
    "tests/groupby/test_cumulative.py::test_cython_api2",
    "tests/groupby/test_cumulative.py::test_groupby_cumprod_nan_influences_other_columns",
    "tests/groupby/test_cumulative.py::test_numpy_compat[cumprod]",
    "tests/groupby/test_cumulative.py::test_numpy_compat[cumsum]",
    "tests/groupby/test_groupby.py::test_as_index_select_column",
    "tests/groupby/test_groupby.py::test_basic_aggregations[float32]",
    "tests/groupby/test_groupby.py::test_basic_aggregations[int32]",
    "tests/groupby/test_groupby.py::test_decimal_na_sort[True]",
    "tests/groupby/test_groupby.py::test_frame_groupby_columns",
    "tests/groupby/test_groupby.py::test_group_name_available_in_inference_pass",
    "tests/groupby/test_groupby.py::test_group_on_two_row_multiindex_returns_one_tuple_key",
    "tests/groupby/test_groupby.py::test_groupby_all_nan_groups_drop",
    "tests/groupby/test_groupby.py::test_groupby_column_index_name_lost[shift]",
    "tests/groupby/test_groupby.py::test_groupby_column_index_name_lost[sum]",
    "tests/groupby/test_groupby.py::test_groupby_crash_on_nunique[axis='index']",
    "tests/groupby/test_groupby.py::test_groupby_crash_on_nunique[axis=0]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Float32-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Float64-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int16-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int16-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int32-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int32-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int64-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int8-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int8-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt16-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt16-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt32-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt32-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt64-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt64-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt8-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt8-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_skipna_false",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_timedelta64",
    "tests/groupby/test_groupby.py::test_groupby_dtype_inference_empty",
    "tests/groupby/test_groupby.py::test_groupby_groups_in_BaseGrouper",
    "tests/groupby/test_groupby.py::test_groupby_level_mapper",
    "tests/groupby/test_groupby.py::test_groupby_level_nonmulti",
    "tests/groupby/test_groupby.py::test_groupby_multiindex_not_lexsorted",
    "tests/groupby/test_groupby.py::test_groupby_nonobject_dtype",
    "tests/groupby/test_groupby.py::test_groupby_nonstring_columns",
    "tests/groupby/test_groupby.py::test_groupby_numerical_stability_cumsum",
    "tests/groupby/test_groupby.py::test_groupby_numerical_stability_sum_mean[mean-values1]",
    "tests/groupby/test_groupby.py::test_groupby_numerical_stability_sum_mean[sum-values0]",
    "tests/groupby/test_groupby.py::test_groupby_one_row",
    "tests/groupby/test_groupby.py::test_groupby_overflow[222-uint]",
    "tests/groupby/test_groupby.py::test_groupby_series_with_name",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int16-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int16-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int32-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int32-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int8-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int8-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt16-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt16-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt32-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt32-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt64-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt64-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt8-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt8-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_with_Time_Grouper[ms]",
    "tests/groupby/test_groupby.py::test_groupby_with_Time_Grouper[ns]",
    "tests/groupby/test_groupby.py::test_groupby_with_Time_Grouper[s]",
    "tests/groupby/test_groupby.py::test_groupby_with_Time_Grouper[us]",
    "tests/groupby/test_groupby.py::test_groupby_with_hier_columns",
    "tests/groupby/test_groupby.py::test_groups_repr_truncates[1-{0: [0], ...}]",
    "tests/groupby/test_groupby.py::test_groups_repr_truncates[4-{0: [0], 1: [1], 2: [2], 3: [3], ...}]",
    "tests/groupby/test_groupby.py::test_groups_repr_truncates[5-{0: [0], 1: [1], 2: [2], 3: [3], 4: [4]}]",
    "tests/groupby/test_groupby.py::test_handle_dict_return_value",
    "tests/groupby/test_groupby.py::test_indices_concatenation_order",
    "tests/groupby/test_groupby.py::test_len_nan_group",
    "tests/groupby/test_groupby.py::test_ops_not_as_index[size]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[DataFrame-as_index-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[DataFrame-dropna-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[DataFrame-group_keys-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[DataFrame-observed-True]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[DataFrame-sort-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[Series-as_index-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[Series-dropna-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[Series-group_keys-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[Series-observed-True]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[Series-sort-False]",
    "tests/groupby/test_groupby.py::test_wrap_aggregated_output_multindex",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[count-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[count-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[count-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[count-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[first-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[first-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[first-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[first-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-False-True-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-True-True-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-False-True-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-True-True-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[last-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[last-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[last-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[last-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[max-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[max-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[max-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[max-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[mean-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[mean-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[mean-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[mean-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[median-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[median-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[median-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[median-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[min-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[min-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[min-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[min-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[nunique-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[nunique-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[nunique-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[nunique-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[prod-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[prod-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[prod-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[prod-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[quantile-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[quantile-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[quantile-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[quantile-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[std-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[std-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[std-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[std-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[sum-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[sum-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[sum-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[sum-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[var-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[var-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[var-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[var-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumcount-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumcount-True-False-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumcount-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumcount-True-True-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cummax-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cummax-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cummin-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cummin-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumprod-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumprod-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumsum-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumsum-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[diff-False-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[diff-False-False-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[diff-False-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[diff-False-True-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[diff-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[diff-True-False-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[diff-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[diff-True-True-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[ngroup-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[ngroup-True-False-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[ngroup-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[ngroup-True-True-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[pct_change-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[pct_change-True-False-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[pct_change-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[pct_change-True-True-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[rank-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[rank-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[shift-False-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[shift-False-False-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[shift-False-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[shift-False-True-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[shift-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[shift-True-False-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[shift-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[shift-True-True-True]",
    "tests/groupby/test_groupby_dropna.py::test_groupby_dropna_multi_index_dataframe_agg[False-tuples1-outputs1]",
    "tests/groupby/test_groupby_dropna.py::test_groupby_dropna_multi_index_dataframe_nan_in_one_group[NoneType-False-tuples1-outputs1]",
    "tests/groupby/test_groupby_dropna.py::test_groupby_dropna_multi_index_dataframe_nan_in_two_groups[NoneType-NoneType-False-tuples1-outputs1]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-Decimal-Decimal-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-Decimal-Decimal-object-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-NoneType-NoneType-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-NoneType-float0-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-NoneType-float1-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-float0-NoneType-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-float1-NoneType-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-Decimal-Decimal-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-Decimal-Decimal-object-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-NoneType-NoneType-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-NoneType-NoneType-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-NoneType-float0-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-NoneType-float0-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-NoneType-float1-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-NoneType-float1-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-float0-NoneType-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-float0-NoneType-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-float1-NoneType-None-True]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_metadata",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[all-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[any-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[bfill-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[corrwith-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[corrwith-obj1]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[count-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[cumcount-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[cummax-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[cummin-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[cumprod-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[cumsum-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[diff-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[ffill-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[fillna-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[first-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[idxmax-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[idxmin-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[last-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[max-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[mean-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[median-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[min-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[ngroup-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[nunique-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[pct_change-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[prod-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[quantile-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[rank-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[sem-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[shift-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[size-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[skew-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[std-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[sum-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[var-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_resample_preserves_subclass[DataFrame]",
    "tests/groupby/test_grouping.py::TestGetGroup::test_get_group_grouped_by_tuple",
    "tests/groupby/test_grouping.py::TestGetGroup::test_get_group_grouped_by_tuple_with_lambda",
    "tests/groupby/test_grouping.py::TestGetGroup::test_groupby_with_single_column",
    "tests/groupby/test_grouping.py::TestGrouping::test_evaluate_with_empty_groups[agg-expected1]",
    "tests/groupby/test_grouping.py::TestGrouping::test_evaluate_with_empty_groups[apply-expected2]",
    "tests/groupby/test_grouping.py::TestGrouping::test_groupby_empty",
    "tests/groupby/test_grouping.py::TestGrouping::test_groupby_level[False]",
    "tests/groupby/test_grouping.py::TestGrouping::test_groupby_level[True]",
    "tests/groupby/test_grouping.py::TestGrouping::test_groupby_level_index_value_all_na",
    "tests/groupby/test_grouping.py::TestGrouping::test_groupby_multiindex_partial_indexing_equivalence",
    "tests/groupby/test_grouping.py::TestGrouping::test_grouper_getting_correct_binner",
    "tests/groupby/test_grouping.py::TestGrouping::test_multiindex_columns_empty_level",
    "tests/groupby/test_grouping.py::TestGrouping::test_multiindex_negative_level",
    "tests/groupby/test_grouping.py::TestSelection::test_indices_grouped_by_tuple_with_lambda",
    "tests/groupby/test_grouping.py::test_grouper_groups",
    "tests/groupby/test_grouping.py::test_grouping_by_key_is_in_axis",
    "tests/groupby/test_indexing.py::test_multiindex",
    "tests/groupby/test_missing.py::test_ffill_handles_nan_groups[False-False-bfill]",
    "tests/groupby/test_missing.py::test_ffill_handles_nan_groups[False-False-ffill]",
    "tests/groupby/test_missing.py::test_ffill_handles_nan_groups[False-True-bfill]",
    "tests/groupby/test_missing.py::test_ffill_handles_nan_groups[False-True-ffill]",
    "tests/groupby/test_missing.py::test_groupby_column_index_name_lost_fill_funcs[bfill]",
    "tests/groupby/test_missing.py::test_groupby_column_index_name_lost_fill_funcs[ffill]",
    "tests/groupby/test_missing.py::test_indices_with_missing",
    "tests/groupby/test_reductions.py::test_empty_categorical[True]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float32-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float32-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float32-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float32-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float64-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float64-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float64-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float64-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int16-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int16-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int16-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int16-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int32-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int32-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int32-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int32-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int64-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int64-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int64-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int64-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int8-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int8-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int8-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int8-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt16-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt16-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt16-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt16-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt32-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt32-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt32-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt32-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt64-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt64-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt64-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt64-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt8-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt8-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt8-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt8-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[double[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[double[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[double[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[double[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float32-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float32-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float32-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float32-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float64-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float64-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float64-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float64-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int16[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int16[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int16[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int16[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int32[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int32[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int32[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int32[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int64[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int64[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int64[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int64[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int8[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int8[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int8[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int8[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint16[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint16[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint16[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint16[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint32[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint32[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint32[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint32[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint64[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint64[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint64[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint64[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint8[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint8[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint8[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint8[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_groupby_mean_no_overflow",
    "tests/groupby/test_reductions.py::test_idxmin_idxmax_axis1",
    "tests/groupby/test_reductions.py::test_masked_kleene_logic[False-data2-any]",
    "tests/groupby/test_reductions.py::test_masked_kleene_logic[False-data3-any]",
    "tests/groupby/test_reductions.py::test_nunique",
    "tests/groupby/test_reductions.py::test_nunique_with_NaT[key0-data0-True-expected0]",
    "tests/groupby/test_reductions.py::test_nunique_with_NaT[key1-data1-True-expected1]",
    "tests/groupby/test_reductions.py::test_nunique_with_NaT[key2-data2-False-expected2]",
    "tests/groupby/test_reductions.py::test_nunique_with_NaT[key3-data3-False-expected3]",
    "tests/groupby/test_reductions.py::test_nunique_with_empty_series",
    "tests/groupby/test_reductions.py::test_nunique_with_object",
    "tests/groupby/test_reductions.py::test_nunique_with_timegrouper",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-all-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-all-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-any-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-any-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-count-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-count-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-count-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-count-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-first-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-first-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmax-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmax-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmax-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmax-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmin-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmin-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmin-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmin-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-last-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-last-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-max-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-max-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-min-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-min-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-nunique-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-nunique-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-nunique-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-nunique-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-sum-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-sum-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-all-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-all-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-any-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-any-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-count-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-count-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-count-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-count-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-first-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-first-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmax-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmax-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmax-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmax-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmin-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmin-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmin-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmin-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-last-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-last-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-max-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-max-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-min-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-min-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-nunique-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-nunique-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-nunique-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-nunique-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-sum-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-sum-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-all-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-all-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-any-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-any-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-count-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-count-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-count-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-count-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-first-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-first-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmax-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmax-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmax-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmax-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmin-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmin-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmin-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmin-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-last-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-last-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-max-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-max-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-min-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-min-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-nunique-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-nunique-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-nunique-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-nunique-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-size-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-size-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-sum-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-sum-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-all-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-all-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-any-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-any-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-count-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-count-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-count-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-count-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-first-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-first-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmax-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmax-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmax-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmax-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmin-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmin-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmin-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmin-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-last-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-last-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-max-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-max-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-min-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-min-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-nunique-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-nunique-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-nunique-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-nunique-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-sum-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-sum-False-1]",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_apply_timegrouper_with_nat_apply_squeeze",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_apply_timegrouper_with_nat_dict_returns",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_apply_timegrouper_with_nat_scalar_returns",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_datetime64_32_bit",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_with_timegrouper",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_with_timegrouper_methods[False]",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_with_timegrouper_methods[True]",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_scalar_call_versus_list_call",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_timegrouper_apply_return_type_series",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_timegrouper_with_reg_groups",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_timezone_info",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumcount-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumcount-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cummax-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cummax-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cummin-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cummin-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumprod-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumprod-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumsum-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumsum-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[nunique-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[nunique-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[size-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[size-keys1]",
    "tests/groupby/transform/test_transform.py::test_cython_transform_series[cumprod-args0-<lambda>]",
    "tests/groupby/transform/test_transform.py::test_cython_transform_series[cumsum-args1-<lambda>]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[False-idxmax]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[False-idxmin]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[False-nunique]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[False-size]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[True-idxmax]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[True-idxmin]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[True-size]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_transformer[False-cumcount]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_transformer[True-cumcount]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-None--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-None-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-bfill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-bfill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-ffill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-ffill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-None--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-None-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-bfill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-bfill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-ffill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-ffill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-None--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-None-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-bfill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-bfill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-ffill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-ffill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-None--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-None-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-bfill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-bfill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-ffill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-ffill-1-None]",
    "tests/groupby/transform/test_transform.py::test_transform_cumcount",
    "tests/groupby/transform/test_transform.py::test_transform_datetime_to_timedelta",
    "tests/groupby/transform/test_transform.py::test_transform_fast",
    "tests/groupby/transform/test_transform.py::test_transform_fastpath_raises",
    "tests/groupby/transform/test_transform.py::test_transform_mixed_type",
    "tests/groupby/transform/test_transform.py::test_transform_numeric_ret[count-a-expected0]",
    "tests/groupby/transform/test_transform.py::test_transform_numeric_ret[count-cols1-expected1]",
    "tests/groupby/transform/test_transform.py::test_transform_numeric_ret[size-a-expected0]",
    "tests/groupby/transform/test_transform.py::test_transform_numeric_ret[size-cols1-expected1]",
    "tests/groupby/transform/test_transform.py::test_transform_transformation_func[cumcount]",
    "tests/indexes/base_class/test_constructors.py::TestIndexConstructor::test_index_string_inference",
    "tests/indexes/base_class/test_reshape.py::TestReshape::test_insert_missing[Decimal]",
    "tests/indexes/categorical/test_astype.py::TestAstype::test_categorical_date_roundtrip[False]",
    "tests/indexes/categorical/test_astype.py::TestAstype::test_categorical_date_roundtrip[True]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex2::test_engine_type[int16-Int16Engine]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex2::test_engine_type[int32-Int32Engine]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex2::test_engine_type[int64-Int64Engine]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex2::test_engine_type[int8-Int8Engine]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex2::test_remove_maintains_order",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex::test_ensure_copied_data",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex::test_is_monotonic[data0-non_lexsorted_data0]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex::test_is_monotonic[data1-non_lexsorted_data1]",
    "tests/indexes/categorical/test_constructors.py::TestCategoricalIndexConstructors::test_construction",
    "tests/indexes/categorical/test_constructors.py::TestCategoricalIndexConstructors::test_construction_disallows_scalar",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string=str[pyarrow]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string=str[python]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string=string[pyarrow]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string=string[python]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string[pyarrow]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string[pyarrow_numpy]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string[python]]",
    "tests/indexes/categorical/test_fillna.py::TestFillNA::test_fillna_copies_with_no_nas",
    "tests/indexes/categorical/test_indexing.py::TestContains::test_contains_na_dtype[False]",
    "tests/indexes/categorical/test_indexing.py::TestContains::test_contains_nan",
    "tests/indexes/categorical/test_indexing.py::TestWhere::test_where_non_categories",
    "tests/indexes/datetimelike_/test_equals.py::TestDatetimeIndexEquals::test_equals2",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values[DatetimeIndex]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values[TimedeltaIndex]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_with_freq_periodindex2[idx1]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_without_freq_datetimeindex[None-index_dates0-expected_dates0]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_without_freq_datetimeindex[None-index_dates1-expected_dates1]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_without_freq_datetimeindex[None-index_dates2-expected_dates2]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_without_freq_periodindex[idx4-expected4]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_without_freq_timedeltaindex",
    "tests/indexes/datetimelike_/test_sort_values.py::test_order_stability_compat",
    "tests/indexes/datetimes/methods/test_astype.py::TestDatetimeIndex::test_astype",
    "tests/indexes/datetimes/methods/test_astype.py::TestDatetimeIndex::test_astype_datetime64",
    "tests/indexes/datetimes/methods/test_astype.py::TestDatetimeIndex::test_astype_raises[float]",
    "tests/indexes/datetimes/methods/test_astype.py::TestDatetimeIndex::test_astype_str_nat",
    "tests/indexes/datetimes/methods/test_astype.py::TestDatetimeIndex::test_astype_uint",
    "tests/indexes/datetimes/methods/test_factorize.py::TestDatetimeIndexFactorize::test_factorize_tz[zoneinfo.ZoneInfo(key='US/Pacific')-index]",
    "tests/indexes/datetimes/methods/test_factorize.py::TestDatetimeIndexFactorize::test_factorize_tz[zoneinfo.ZoneInfo(key='US/Pacific')-series]",
    "tests/indexes/datetimes/methods/test_isocalendar.py::test_dti_timestamp_isocalendar_fields",
    "tests/indexes/datetimes/methods/test_round.py::TestDatetimeIndexRound::test_round_daily",
    "tests/indexes/datetimes/methods/test_unique.py::test_index_unique2",
    "tests/indexes/datetimes/test_arithmetic.py::TestDatetimeIndexArithmetic::test_sub_datetime_preserves_freq[zoneinfo.ZoneInfo(key='US/Pacific')]",
    "tests/indexes/datetimes/test_arithmetic.py::TestDatetimeIndexArithmetic::test_sub_datetime_preserves_freq[zoneinfo.ZoneInfo(key='UTC')]",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_construction_outofbounds",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_constructor_coverage",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_constructor_datetime64_tzformat[W-SUN]",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_constructor_dtype",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_constructor_dtype_tz_mismatch_raises",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_dti_date_out_of_range[data1]",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_integer_values_and_tz_interpreted_as_utc",
    "tests/indexes/datetimes/test_constructors.py::TestTimeSeries::test_constructor_int64_nocopy",
    "tests/indexes/datetimes/test_date_range.py::TestDateRangeNonTickFreq::test_date_range_freqstr_matches_offset[W-SUN-offset2]",
    "tests/indexes/datetimes/test_date_range.py::TestDateRangeTZ::test_date_range_with_fixed_tz",
    "tests/indexes/datetimes/test_date_range.py::TestDateRangeTZ::test_date_range_with_fixedoffset_noname",
    "tests/indexes/datetimes/test_date_range.py::TestDateRanges::test_date_range_unsigned_overflow_handling",
    "tests/indexes/datetimes/test_date_range.py::TestDateRanges::test_range_tz_dateutil",
    "tests/indexes/datetimes/test_date_range.py::TestDateRanges::test_range_tz_pytz",
    "tests/indexes/datetimes/test_date_range.py::TestGenRangeGeneration::test_precision_finer_than_offset",
    "tests/indexes/datetimes/test_datetime.py::TestDatetimeIndex::test_misc_coverage",
    "tests/indexes/datetimes/test_formats.py::TestFormat::test_format",
    "tests/indexes/datetimes/test_indexing.py::TestGetItem::test_getitem[None]",
    "tests/indexes/datetimes/test_indexing.py::TestTake::test_take2[None]",
    "tests/indexes/datetimes/test_indexing.py::TestTake::test_take[None]",
    "tests/indexes/datetimes/test_indexing.py::TestWhere::test_where_invalid_dtypes",
    "tests/indexes/datetimes/test_indexing.py::TestWhere::test_where_mismatched_nat[zoneinfo.ZoneInfo(key='US/Pacific')]",
    "tests/indexes/datetimes/test_indexing.py::TestWhere::test_where_mismatched_nat[zoneinfo.ZoneInfo(key='UTC')]",
    "tests/indexes/datetimes/test_join.py::TestJoin::test_join_preserves_freq[None]",
    "tests/indexes/datetimes/test_join.py::TestJoin::test_join_self[inner]",
    "tests/indexes/datetimes/test_join.py::TestJoin::test_join_self[left]",
    "tests/indexes/datetimes/test_join.py::TestJoin::test_join_self[outer]",
    "tests/indexes/datetimes/test_join.py::TestJoin::test_join_self[right]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[2018-12-02 21:52:00+00:00-2018-12-02 21:50:00+00:00]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[2018-12-02 21:52:00+00:00-start1]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[2018-12-02 21:52:00+00:00-start2]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end1-2018-12-02 21:50:00+00:00]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end1-start1]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end1-start2]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end2-2018-12-02 21:50:00+00:00]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end2-start1]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end2-start2]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_partial_slice_daily",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_partial_slice_minutely",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_partial_slicing_dataframe",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_return_type_doesnt_depend_on_monotonicity",
    "tests/indexes/datetimes/test_scalar_compat.py::TestDatetimeIndexOps::test_day_name_month_name[None]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_difference_freq[False]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_difference_freq[None]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_intersection[False-Asia/Tokyo]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_intersection[False-None]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_intersection[False-US/Eastern]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_intersection[False-dateutil/US/Pacific]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_intersection[None-Asia/Tokyo]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_intersection[None-None]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_intersection[None-US/Eastern]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_intersection[None-dateutil/US/Pacific]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_intersection_empty[zoneinfo.ZoneInfo(key='US/Pacific')-min]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_intersection_empty[zoneinfo.ZoneInfo(key='UTC')-min]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_setops_preserve_freq[None]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_union_freq_infer",
    "tests/indexes/datetimes/test_timezones.py::TestDatetimeIndexTimezones::test_date_range_localize[ms]",
    "tests/indexes/datetimes/test_timezones.py::TestDatetimeIndexTimezones::test_date_range_localize[ns]",
    "tests/indexes/datetimes/test_timezones.py::TestDatetimeIndexTimezones::test_date_range_localize[s]",
    "tests/indexes/datetimes/test_timezones.py::TestDatetimeIndexTimezones::test_date_range_localize[us]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_float[index0]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_float[index1]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_float[index3]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_float[index4]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index0-int64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index0-uint64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index1-int64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index1-uint64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index3-int64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index3-uint64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index4-int64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index4-uint64]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_datetimelike[index0-datetime64[ns]]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_datetimelike[index0-timedelta64[ns]]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_datetimelike[index1-datetime64[ns]]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_datetimelike[index1-timedelta64[ns]]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_integer[int64]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_integer[uint64]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_integer_errors",
    "tests/indexes/interval/test_constructors.py::TestClassConstructors::test_constructor_pass_closed[breaks0]",
    "tests/indexes/interval/test_constructors.py::TestClassConstructors::test_constructor_pass_closed[breaks2]",
    "tests/indexes/interval/test_constructors.py::TestClassConstructors::test_constructor_pass_closed[breaks3]",
    "tests/indexes/interval/test_constructors.py::TestClassConstructors::test_constructor_pass_closed[breaks4]",
    "tests/indexes/interval/test_constructors.py::TestFromArrays::test_constructor_pass_closed[breaks0]",
    "tests/indexes/interval/test_constructors.py::TestFromArrays::test_constructor_pass_closed[breaks1]",
    "tests/indexes/interval/test_constructors.py::TestFromArrays::test_constructor_pass_closed[breaks2]",
    "tests/indexes/interval/test_constructors.py::TestFromArrays::test_constructor_pass_closed[breaks3]",
    "tests/indexes/interval/test_constructors.py::TestFromArrays::test_constructor_pass_closed[breaks4]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_constructor_pass_closed[breaks0]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_constructor_pass_closed[breaks1]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_constructor_pass_closed[breaks2]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_constructor_pass_closed[breaks3]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_constructor_pass_closed[breaks4]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_generic_errors",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_left_right_dont_share_data",
    "tests/indexes/interval/test_constructors.py::TestFromTuples::test_constructor_pass_closed[breaks0]",
    "tests/indexes/interval/test_constructors.py::TestFromTuples::test_constructor_pass_closed[breaks2]",
    "tests/indexes/interval/test_constructors.py::TestFromTuples::test_constructor_pass_closed[breaks3]",
    "tests/indexes/interval/test_constructors.py::TestFromTuples::test_constructor_pass_closed[breaks4]",
    "tests/indexes/interval/test_formats.py::TestIntervalIndexRendering::test_timestamp_with_timezone[ms]",
    "tests/indexes/interval/test_formats.py::TestIntervalIndexRendering::test_timestamp_with_timezone[ns]",
    "tests/indexes/interval/test_formats.py::TestIntervalIndexRendering::test_timestamp_with_timezone[s]",
    "tests/indexes/interval/test_formats.py::TestIntervalIndexRendering::test_timestamp_with_timezone[us]",
    "tests/indexes/interval/test_indexing.py::TestGetIndexer::test_get_indexer_categorical_with_nans",
    "tests/indexes/interval/test_indexing.py::TestGetIndexer::test_get_indexer_datetime",
    "tests/indexes/interval/test_indexing.py::TestGetLoc::test_get_loc_datetimelike_overlapping[datetime64[ns]]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_comparison",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_ensure_copied_data[both]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_ensure_copied_data[left]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_ensure_copied_data[neither]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_ensure_copied_data[right]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_insert[data0]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[both-breaks4]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[left-breaks4]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[neither-breaks4]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[right-breaks4]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_monotonic[both]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_monotonic[left]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_monotonic[neither]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_monotonic[right]",
    "tests/indexes/interval/test_interval.py::test_dir",
    "tests/indexes/interval/test_interval_range.py::TestIntervalRange::test_interval_range_fractional_period",
    "tests/indexes/multi/test_analytics.py::test_map_dictlike[<lambda>1]",
    "tests/indexes/multi/test_astype.py::test_astype",
    "tests/indexes/multi/test_compat.py::test_inplace_mutation_resets_values",
    "tests/indexes/multi/test_constructors.py::test_create_index_existing_name",
    "tests/indexes/multi/test_constructors.py::test_from_arrays_respects_none_names",
    "tests/indexes/multi/test_constructors.py::test_from_frame_dtype_fidelity",
    "tests/indexes/multi/test_constructors.py::test_from_frame_missing_values_multiIndex",
    "tests/indexes/multi/test_constructors.py::test_multiindex_inference_consistency",
    "tests/indexes/multi/test_conversion.py::test_to_frame_dtype_fidelity",
    "tests/indexes/multi/test_copy.py::test_copy",
    "tests/indexes/multi/test_copy.py::test_copy_deep_false_retains_id",
    "tests/indexes/multi/test_copy.py::test_shallow_copy",
    "tests/indexes/multi/test_copy.py::test_view",
    "tests/indexes/multi/test_drop.py::test_drop_not_lexsorted",
    "tests/indexes/multi/test_drop.py::test_droplevel_list",
    "tests/indexes/multi/test_drop.py::test_droplevel_multiindex_one_level",
    "tests/indexes/multi/test_duplicates.py::test_duplicate_multiindex_codes",
    "tests/indexes/multi/test_equivalence.py::test_equals_multi",
    "tests/indexes/multi/test_equivalence.py::test_equals_op",
    "tests/indexes/multi/test_equivalence.py::test_is_",
    "tests/indexes/multi/test_equivalence.py::test_multiindex_compare",
    "tests/indexes/multi/test_formats.py::TestRepr::test_rjust",
    "tests/indexes/multi/test_formats.py::TestRepr::test_tuple_width",
    "tests/indexes/multi/test_get_set.py::test_set_levels_categorical[False]",
    "tests/indexes/multi/test_get_set.py::test_set_levels_categorical[True]",
    "tests/indexes/multi/test_get_set.py::test_set_name_methods",
    "tests/indexes/multi/test_indexing.py::TestContains::test_contains",
    "tests/indexes/multi/test_indexing.py::TestContains::test_contains_with_missing_value",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_categorical_time",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_crossing_levels",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_kwarg_validation",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_methods",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_nan",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_three_or_more_levels",
    "tests/indexes/multi/test_indexing.py::TestGetLoc::test_get_loc",
    "tests/indexes/multi/test_indexing.py::TestGetLoc::test_get_loc_duplicates2",
    "tests/indexes/multi/test_indexing.py::TestGetLoc::test_get_loc_past_lexsort_depth",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[Decimal]",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[NAType]",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[NaTType]",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[NoneType]",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[float0]",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[float1]",
    "tests/indexes/multi/test_indexing.py::test_getitem",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_all[ind20-ind10]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_all[ind20-ind11]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_all[ind21-ind10]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_all[ind21-ind11]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_single[ind20-ind10]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_single[ind20-ind11]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_single[ind21-ind10]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_single[ind21-ind11]",
    "tests/indexes/multi/test_indexing.py::test_getitem_group_select",
    "tests/indexes/multi/test_integrity.py::test_labels_dtypes",
    "tests/indexes/multi/test_integrity.py::test_metadata_immutable",
    "tests/indexes/multi/test_isin.py::test_isin_level_kwarg",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Float32-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Float32-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Float64-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Float64-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int16-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int16-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int32-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int32-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int64-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int64-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int8-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int8-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt16-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt16-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt32-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt32-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt64-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt64-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt8-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt8-5]",
    "tests/indexes/multi/test_join.py::test_join_midx_ea",
    "tests/indexes/multi/test_join.py::test_join_midx_string",
    "tests/indexes/multi/test_join.py::test_join_multi_return_indexers",
    "tests/indexes/multi/test_join.py::test_join_overlapping_interval_level",
    "tests/indexes/multi/test_join.py::test_join_self[inner]",
    "tests/indexes/multi/test_join.py::test_join_self[left]",
    "tests/indexes/multi/test_join.py::test_join_self[outer]",
    "tests/indexes/multi/test_join.py::test_join_self[right]",
    "tests/indexes/multi/test_missing.py::test_dropna",
    "tests/indexes/multi/test_missing.py::test_nulls",
    "tests/indexes/multi/test_names.py::test_copy_names",
    "tests/indexes/multi/test_names.py::test_setting_names_from_levels_raises",
    "tests/indexes/multi/test_reindex.py::test_reindex_base",
    "tests/indexes/multi/test_reshape.py::test_append_names_match[c-None]",
    "tests/indexes/multi/test_setops.py::test_difference[False]",
    "tests/indexes/multi/test_setops.py::test_difference[None]",
    "tests/indexes/multi/test_setops.py::test_intersection_non_object[False]",
    "tests/indexes/multi/test_setops.py::test_intersection_non_object[None]",
    "tests/indexes/multi/test_sorting.py::test_argsort",
    "tests/indexes/multi/test_sorting.py::test_numpy_argsort",
    "tests/indexes/multi/test_sorting.py::test_reconstruct_remove_unused",
    "tests/indexes/multi/test_sorting.py::test_reconstruct_sort",
    "tests/indexes/multi/test_sorting.py::test_remove_unused_levels_large[datetime64[D]-str]",
    "tests/indexes/multi/test_sorting.py::test_remove_unused_levels_large[int64-int64]",
    "tests/indexes/multi/test_sorting.py::test_unsortedindex_doc_examples",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_astype_float64_to_uint64",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[inf-int16]",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[inf-int32]",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[inf-int64]",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[inf-int]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_invalid",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_nan",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_nearest_decreasing[backfill-expected1]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_nearest_decreasing[nearest-expected2]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_nearest_decreasing[pad-expected0]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_nearest_error",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_numeric_index_boolean_target[get_indexer-float64]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_numeric_index_boolean_target[get_indexer-int64]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_numeric_index_boolean_target[get_indexer-range]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_numeric_index_boolean_target[get_indexer-uint64]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_numeric_vs_bool",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_with_method_numeric_vs_bool[backfill]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_with_method_numeric_vs_bool[nearest]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_with_method_numeric_vs_bool[pad]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_equals_numeric",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[int16]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[int32]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[int64]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[int8]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[int]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[uint16]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[uint32]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[uint64]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[uint8]",
    "tests/indexes/numeric/test_numeric.py::TestIntNumericIndex::test_constructor[int16]",
    "tests/indexes/numeric/test_numeric.py::TestIntNumericIndex::test_constructor[int32]",
    "tests/indexes/numeric/test_numeric.py::TestIntNumericIndex::test_constructor[int64]",
    "tests/indexes/object/test_indexing.py::TestSliceLocs::test_slice_locs_negative_step[in_slice13--object]",
    "tests/indexes/object/test_indexing.py::TestSliceLocs::test_slice_locs_negative_step[in_slice13--string[pyarrow_numpy]]",
    "tests/indexes/period/test_constructors.py::TestPeriodIndex::test_constructor_fromarraylike",
    "tests/indexes/period/test_join.py::TestJoin::test_join_self[inner]",
    "tests/indexes/period/test_join.py::TestJoin::test_join_self[left]",
    "tests/indexes/period/test_join.py::TestJoin::test_join_self[outer]",
    "tests/indexes/period/test_join.py::TestJoin::test_join_self[right]",
    "tests/indexes/period/test_period.py::test_dunder_array[True]",
    "tests/indexes/period/test_period_range.py::TestPeriodRange::test_construction_from_string[W-W]",
    "tests/indexes/ranges/test_indexing.py::TestGetIndexer::test_get_indexer_limit",
    "tests/indexes/ranges/test_join.py::TestJoin::test_join_self[inner]",
    "tests/indexes/ranges/test_join.py::TestJoin::test_join_self[left]",
    "tests/indexes/ranges/test_join.py::TestJoin::test_join_self[outer]",
    "tests/indexes/ranges/test_join.py::TestJoin::test_join_self[right]",
    "tests/indexes/ranges/test_range.py::TestRangeIndex::test_cache",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=-10, step=-1)-RangeIndex(start=-10, stop=-20, step=-1)-RangeIndex(start=-19, stop=1, step=1)-RangeIndex(start=0, stop=-20, step=-1)]",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=-100, step=-1)-RangeIndex(start=0, stop=-50, step=-3)-RangeIndex(start=-99, stop=1, step=1)-RangeIndex(start=0, stop=-100, step=-1)]",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=-100, step=-2)-RangeIndex(start=-100, stop=50, step=102)-RangeIndex(start=-100, stop=4, step=2)-expected_notsorted15]",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=-100, step=-5)-RangeIndex(start=5, stop=-100, step=-20)-RangeIndex(start=-95, stop=10, step=5)-expected_notsorted10]",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=-11, step=-1)-RangeIndex(start=1, stop=-12, step=-4)-RangeIndex(start=-11, stop=2, step=1)-expected_notsorted11]",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=10, step=5)-RangeIndex(start=-5, stop=-6, step=-20)-RangeIndex(start=-5, stop=10, step=5)-expected_notsorted18]",
    "tests/indexes/ranges/test_setops.py::test_range_difference",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[pyarrow]-None]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[pyarrow]-nan0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[pyarrow]-nan1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[pyarrow]-null3]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[python]-None]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[python]-nan0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[python]-nan1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[python]-null3]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[pyarrow]-None]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[pyarrow]-nan0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[pyarrow]-nan1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[pyarrow]-null3]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[python]-None]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[python]-nan0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[python]-nan1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[python]-null3]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=str[pyarrow]-backfill-expected1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=str[pyarrow]-pad-expected0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=str[python]-backfill-expected1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=str[python]-pad-expected0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=string[pyarrow]-backfill-expected1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=string[pyarrow]-pad-expected0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=string[python]-backfill-expected1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=string[python]-pad-expected0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings_raises[string=str[pyarrow]]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings_raises[string=str[python]]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings_raises[string=string[pyarrow]]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings_raises[string=string[python]]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[bool-dtype]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[bool-object]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[categorical]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[complex128]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[complex64]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[datetime-tz]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[datetime]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[empty]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[float32]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[float64]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[int16]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[int32]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[int64]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[int8]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[interval]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[mi-with-dt64tz-level]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[multi]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[nullable_bool]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[nullable_float]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[nullable_int]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[nullable_uint]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[object]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[period]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[range]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[repeats]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[string-pyarrow]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[string-python]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[string]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[timedelta]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[tuples]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[uint16]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[uint32]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[uint64]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[uint8]",
    "tests/indexes/test_base.py::TestIndex::test_cached_properties_not_settable",
    "tests/indexes/test_base.py::TestIndex::test_constructor_dtypes_datetime[None-Index-values]",
    "tests/indexes/test_base.py::TestIndex::test_equals_op_mismatched_multiindex_raises[index0]",
    "tests/indexes/test_base.py::TestIndex::test_is_",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NAType-float32]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NAType-float64]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NAType-float]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NoneType-float32]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NoneType-float64]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NoneType-float]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_object[NoneType-Decimal]",
    "tests/indexes/test_base.py::TestIndex::test_map_defaultdict",
    "tests/indexes/test_base.py::TestIndex::test_str_attribute_raises[index2]",
    "tests/indexes/test_base.py::TestIndex::test_str_bool_return",
    "tests/indexes/test_base.py::TestIndex::test_tab_completion[index1-False]",
    "tests/indexes/test_common.py::TestCommon::test_constructor_non_hashable_name[range]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[bool-dtype-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[bool-dtype-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[bool-object-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[bool-object-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[categorical-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[categorical-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[complex128-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[complex128-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[complex64-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[complex64-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[datetime-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[datetime-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[datetime-tz-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[datetime-tz-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[empty-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[empty-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[float32-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[float32-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[float64-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[float64-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int16-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int16-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int32-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int32-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int64-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int64-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int8-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int8-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[interval-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[interval-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_bool-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_bool-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_float-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_float-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_int-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_int-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_uint-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_uint-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[object-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[object-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[period-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[period-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[range-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[range-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[repeats-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[repeats-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-pyarrow-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-pyarrow-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-python-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-python-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[timedelta-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[timedelta-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint16-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint16-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint32-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint32-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint64-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint64-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint8-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint8-new_name]",
    "tests/indexes/test_common.py::test_ndarray_compat_properties[multi]",
    "tests/indexes/test_common.py::test_ndarray_compat_properties[tuples]",
    "tests/indexes/test_datetimelike.py::TestDatetimeLike::test_argsort_matches_array[simple_index1]",
    "tests/indexes/test_datetimelike.py::TestDatetimeLike::test_argsort_matches_array[simple_index2]",
    "tests/indexes/test_indexing.py::TestGetIndexer::test_get_indexer_base[multi]",
    "tests/indexes/test_indexing.py::TestGetIndexer::test_get_indexer_base[tuples]",
    "tests/indexes/test_indexing.py::TestTake::test_take_indexer_type",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arccos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arccosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arcsin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arcsinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arctan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arctanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-cos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-cosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-deg2rad]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-exp2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-exp]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-expm1]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-log10]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-log1p]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-log2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-log]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-rad2deg]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-sin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-sinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-sqrt]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-tan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-tanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arccos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arccosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arcsin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arcsinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arctan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arctanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-cos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-cosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-exp2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-exp]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-expm1]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-log10]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-log1p]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-log2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-log]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-sin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-sinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-sqrt]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-tan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-tanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arccos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arccosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arcsin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arcsinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arctan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arctanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-cos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-cosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-exp2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-exp]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-expm1]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-log10]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-log1p]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-log2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-log]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-sin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-sinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-tan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-tanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[int8-deg2rad]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[int8-rad2deg]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arccos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arccosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arcsin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arcsinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arctan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arctanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-cos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-cosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-deg2rad]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-exp2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-exp]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-expm1]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-log10]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-log1p]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-log2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-log]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-rad2deg]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-sin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-sinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-sqrt]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-tan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-tanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[uint8-deg2rad]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[uint8-rad2deg]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[bool-dtype-isfinite]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[bool-dtype-isinf]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[bool-dtype-isnan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[bool-dtype-signbit]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[nullable_bool-isfinite]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[nullable_bool-isinf]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[nullable_bool-isnan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[nullable_bool-signbit]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_out[interval]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_out[multi]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_out[tuples]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_reductions[multi-maximum]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_reductions[multi-minimum]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_reductions[tuples-maximum]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_reductions[tuples-minimum]",
    "tests/indexes/test_old_base.py::TestBase::test_argsort[nullable_bool]",
    "tests/indexes/test_old_base.py::TestBase::test_equals_op[simple_index13]",
    "tests/indexes/test_old_base.py::TestBase::test_nulls[multi]",
    "tests/indexes/test_old_base.py::TestBase::test_nulls[tuples]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[categorical]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[interval]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[multi]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[tuples]",
    "tests/indexes/test_setops.py::TestSetOps::test_difference_base[False-string-pyarrow]",
    "tests/indexes/test_setops.py::TestSetOps::test_difference_base[False-string-python]",
    "tests/indexes/test_setops.py::TestSetOps::test_difference_base[None-string-pyarrow]",
    "tests/indexes/test_setops.py::TestSetOps::test_difference_base[None-string-python]",
    "tests/indexes/test_setops.py::TestSetOpsUnsorted::test_difference_should_not_compare",
    "tests/indexes/test_setops.py::TestSetOpsUnsorted::test_intersection[False-string]",
    "tests/indexes/test_setops.py::TestSetOpsUnsorted::test_intersection[None-string]",
    "tests/indexes/test_setops.py::TestSetOpsUnsorted::test_union_identity[False-string]",
    "tests/indexes/test_setops.py::TestSetOpsUnsorted::test_union_identity[None-string]",
    "tests/indexes/test_setops.py::test_setop_with_categorical[bool-dtype-False-union]",
    "tests/indexes/test_setops.py::test_setop_with_categorical[bool-dtype-None-union]",
    "tests/indexes/test_setops.py::test_setop_with_categorical[nullable_bool-False-union]",
    "tests/indexes/test_setops.py::test_setop_with_categorical[nullable_bool-None-union]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-complex128]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-complex64]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-float32]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-float64]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-int16]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-int32]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-int64]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-int8]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-range]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-repeats]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-uint16]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-uint32]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-uint64]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-uint8]",
    "tests/indexes/test_setops.py::test_union_different_types[categorical-nullable_float]",
    "tests/indexes/test_setops.py::test_union_different_types[categorical-nullable_uint]",
    "tests/indexes/test_setops.py::test_union_different_types[categorical-string-pyarrow]",
    "tests/indexes/test_setops.py::test_union_different_types[categorical-string-python]",
    "tests/indexes/test_setops.py::test_union_different_types[complex128-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[complex64-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[datetime-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[float32-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[float64-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[int16-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[int16-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[int32-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[int32-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[int64-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[int64-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[int8-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[int8-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[nullable_float-categorical]",
    "tests/indexes/test_setops.py::test_union_different_types[nullable_uint-categorical]",
    "tests/indexes/test_setops.py::test_union_different_types[range-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[range-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[repeats-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[repeats-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[string-pyarrow-categorical]",
    "tests/indexes/test_setops.py::test_union_different_types[string-python-categorical]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-datetime]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-int16]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-int32]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-int64]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-int8]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-range]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-repeats]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-uint16]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-uint32]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-uint8]",
    "tests/indexes/test_setops.py::test_union_different_types[uint16-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[uint16-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[uint32-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[uint32-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[uint64-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[uint8-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[uint8-timedelta]",
    "tests/indexes/test_setops.py::test_union_duplicate_index_different_dtypes",
    "tests/indexes/test_setops.py::test_union_duplicate_index_subsets_of_each_other[category]",
    "tests/indexes/test_setops.py::test_union_same_value_duplicated_in_both",
    "tests/indexes/test_setops.py::test_union_with_duplicate_index_not_subset_and_non_monotonic[category]",
    "tests/indexes/timedeltas/methods/test_astype.py::TestTimedeltaIndex::test_astype",
    "tests/indexes/timedeltas/methods/test_astype.py::TestTimedeltaIndex::test_astype_raises[float]",
    "tests/indexes/timedeltas/methods/test_astype.py::TestTimedeltaIndex::test_astype_timedelta64",
    "tests/indexes/timedeltas/test_constructors.py::TestTimedeltaIndex::test_constructor_coverage",
    "tests/indexes/timedeltas/test_constructors.py::TestTimedeltaIndex::test_int64_nocopy",
    "tests/indexes/timedeltas/test_formats.py::TestTimedeltaIndexRendering::test_representation[__repr__]",
    "tests/indexes/timedeltas/test_formats.py::TestTimedeltaIndexRendering::test_representation[__str__]",
    "tests/indexes/timedeltas/test_indexing.py::TestContains::test_contains",
    "tests/indexes/timedeltas/test_indexing.py::TestWhere::test_where_invalid_dtypes",
    "tests/indexes/timedeltas/test_searchsorted.py::TestSearchSorted::test_searchsorted_invalid_argument_dtype[arg0]",
    "tests/indexes/timedeltas/test_setops.py::TestTimedeltaIndex::test_intersection_equal[False]",
    "tests/indexes/timedeltas/test_setops.py::TestTimedeltaIndex::test_intersection_equal[None]",
    "tests/indexes/timedeltas/test_setops.py::TestTimedeltaIndex::test_intersection_non_monotonic[None-rng2-expected2]",
    "tests/indexes/timedeltas/test_setops.py::TestTimedeltaIndex::test_union_freq_infer",
    "tests/indexes/timedeltas/test_timedelta.py::TestTimedeltaIndex::test_misc_coverage",
    "tests/indexing/interval/test_interval.py::TestIntervalIndexInsideMultiIndex::test_reindex_behavior_with_interval_index[1010]",
    "tests/indexing/interval/test_interval.py::TestIntervalIndexInsideMultiIndex::test_reindex_behavior_with_interval_index[101]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_loc_with_overlap[loc]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_loc_with_overlap[setitem]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_loc_with_slices[loc]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_loc_with_slices[setitem]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_non_unique_moar[loc]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_non_unique_moar[setitem]",
    "tests/indexing/multiindex/test_getitem.py::test_frame_getitem_multicolumn_empty_level",
    "tests/indexing/multiindex/test_getitem.py::test_loc_empty_multiindex",
    "tests/indexing/multiindex/test_getitem.py::test_series_getitem_multiindex[0-expected0-<lambda>0]",
    "tests/indexing/multiindex/test_getitem.py::test_series_getitem_multiindex[0-expected0-<lambda>1]",
    "tests/indexing/multiindex/test_getitem.py::test_series_getitem_multiindex[1-expected1-<lambda>0]",
    "tests/indexing/multiindex/test_getitem.py::test_series_getitem_multiindex[1-expected1-<lambda>1]",
    "tests/indexing/multiindex/test_iloc.py::test_indexing_ambiguity_bug_1678",
    "tests/indexing/multiindex/test_loc.py::TestKeyErrorsWithMultiIndex::test_missing_key_combination",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_int_slice",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_multiindex_missing_label_raises",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-Index]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-Series]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-list]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-ndarray]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-set]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-tuple]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-Index]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-Series]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-list]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-ndarray]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-set]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-tuple]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-Index]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-Series]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-list]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-ndarray]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-set]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-tuple]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-Index]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-Series]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-list]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-ndarray]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-set]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-tuple]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-Index]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-Series]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-list]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-ndarray]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-set]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-tuple]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_multiindex_indexer_none",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_multiindex_missing_label_raises",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_no_second_level_index",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_sorted_multiindex_after_union",
    "tests/indexing/multiindex/test_loc.py::test_getitem_str_slice",
    "tests/indexing/multiindex/test_loc.py::test_loc_get_scalar_casting_to_float",
    "tests/indexing/multiindex/test_loc.py::test_loc_getitem_drops_levels_for_one_row_dataframe",
    "tests/indexing/multiindex/test_loc.py::test_loc_getitem_duplicates_multiindex_missing_indexers[indexer6-pos6]",
    "tests/indexing/multiindex/test_loc.py::test_loc_getitem_index_differently_ordered_slice_none_duplicates[indexer0]",
    "tests/indexing/multiindex/test_loc.py::test_loc_getitem_int_raises_exception",
    "tests/indexing/multiindex/test_loc.py::test_loc_with_mi_indexer",
    "tests/indexing/multiindex/test_multiindex.py::TestMultiIndexBasic::test_rename_multiindex_with_duplicates",
    "tests/indexing/multiindex/test_partial.py::TestMultiIndexPartial::test_getitem_partial_column_select",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_frame_getitem_setitem_boolean",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_frame_getitem_setitem_multislice",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_frame_setitem_multi_column2",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_loc_getitem_setitem_slice_integers[DataFrame]",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_loc_getitem_setitem_slice_integers[Series]",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_series_setitem",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_set_column_scalar_with_loc",
    "tests/indexing/multiindex/test_setitem.py::test_frame_setitem_view_direct",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[int-loc0-iloc0]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[int-loc3-iloc3]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[int-loc4-iloc4]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[int-loc7-iloc7]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[str-loc12-iloc12]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[str-loc8-iloc8]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_per_axis_per_level_doc_examples",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_per_axis_per_level_getitem",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_per_axis_per_level_setitem",
    "tests/indexing/multiindex/test_sorted.py::TestMultiIndexSorted::test_argsort_with_na",
    "tests/indexing/test_at.py::test_at_timezone",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_ix_categorical_index",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[False-idx_values17]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[False-idx_values28]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[None-idx_values17]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[None-idx_values28]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[True-idx_values17]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[True-idx_values28]",
    "tests/indexing/test_coercion.py::TestFillnaSeriesCoercion::test_fillna_float64[index-True-object]",
    "tests/indexing/test_coercion.py::TestFillnaSeriesCoercion::test_fillna_float64[series-True-object]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[dict-bool-float64]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[dict-bool-int64]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[dict-float64-bool]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-bool-datetime64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-bool-float64]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-bool-int64]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-bool-timedelta64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-datetime64-timedelta64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-float64-datetime64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-float64-timedelta64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-int64-datetime64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-int64-timedelta64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-object-datetime64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-object-timedelta64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-timedelta64-datetime64[ns]]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_floating_misc[setitem]",
    "tests/indexing/test_iloc.py::TestILocErrors::test_iloc_getitem_setitem_fancy_exceptions",
    "tests/indexing/test_iloc.py::TestILocErrors::test_iloc_getitem_setitem_fancy_exceptions[False]",
    "tests/indexing/test_iloc.py::TestILocErrors::test_iloc_getitem_setitem_fancy_exceptions[True]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_identity_slice_returns_new_object",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_getitem_array",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_getitem_bool",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_getitem_doc_issue",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_getitem_int_single_ea_block_view",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_getitem_slice",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_getitem_slice_negative_step_ea_block",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_mask",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_axis_argument",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_axis_argument[False]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_axis_argument[True]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_categorical_updates_inplace",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[DataFrame-Series-False]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[DataFrame-Series]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[DataFrame-array-False]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[DataFrame-array]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[Series-Series-False]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[Series-Series]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[Series-array-False]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[Series-array]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_empty_frame_raises_with_3d_ndarray",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key0]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key1]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key2]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key3]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key4]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key5]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key0]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key1]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key2]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key3]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key4]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key5]",
    "tests/indexing/test_indexing.py::TestDataframeNoneCoercion::test_coercion_with_loc[expected2]",
    "tests/indexing/test_indexing.py::TestDataframeNoneCoercion::test_coercion_with_setitem_and_dataframe[expected2]",
    "tests/indexing/test_indexing.py::TestDataframeNoneCoercion::test_none_coercion_loc_and_dataframe[expected2]",
    "tests/indexing/test_indexing.py::TestDataframeNoneCoercion::test_none_coercion_mixed_dtypes",
    "tests/indexing/test_indexing.py::TestFancy::test_index_type_coercion[getitem]",
    "tests/indexing/test_indexing.py::TestFancy::test_index_type_coercion[loc]",
    "tests/indexing/test_indexing.py::TestFancy::test_multi_assign_broadcasting_rhs",
    "tests/indexing/test_indexing.py::TestFancy::test_setitem_ndarray_3d[empty-DataFrame-iloc]",
    "tests/indexing/test_indexing.py::TestFancy::test_setitem_ndarray_3d[empty-DataFrame-loc]",
    "tests/indexing/test_indexing.py::test_scalar_setitem_series_with_nested_value_length1[iloc-value0]",
    "tests/indexing/test_indexing.py::test_scalar_setitem_series_with_nested_value_length1[iloc-value1]",
    "tests/indexing/test_indexing.py::test_scalar_setitem_series_with_nested_value_length1[iloc-value3]",
    "tests/indexing/test_indexing.py::test_scalar_setitem_series_with_nested_value_length1[loc-value0]",
    "tests/indexing/test_indexing.py::test_scalar_setitem_series_with_nested_value_length1[loc-value1]",
    "tests/indexing/test_indexing.py::test_scalar_setitem_series_with_nested_value_length1[loc-value3]",
    "tests/indexing/test_indexing.py::test_scalar_setitem_series_with_nested_value_length1[setitem-value0]",
    "tests/indexing/test_indexing.py::test_scalar_setitem_series_with_nested_value_length1[setitem-value1]",
    "tests/indexing/test_indexing.py::test_scalar_setitem_series_with_nested_value_length1[setitem-value3]",
    "tests/indexing/test_indexing.py::test_ser_tup_indexer_exceeds_dimensions[iloc-ser0-keys0]",
    "tests/indexing/test_indexing.py::test_ser_tup_indexer_exceeds_dimensions[iloc-ser1-keys1]",
    "tests/indexing/test_indexing.py::test_ser_tup_indexer_exceeds_dimensions[loc-ser0-keys0]",
    "tests/indexing/test_loc.py::TestLabelSlicing::test_loc_getitem_label_slice_period_timedelta[index0]",
    "tests/indexing/test_loc.py::TestLoc::test_loc_getitem_single_boolean_arg[obj1-False-exp1]",
    "tests/indexing/test_loc.py::TestLoc::test_loc_getitem_single_boolean_arg[obj2-False-exp2]",
    "tests/indexing/test_loc.py::TestLocBaseIndependent::test_identity_slice_returns_new_object",
    "tests/indexing/test_loc.py::TestLocBaseIndependent::test_loc_modify_datetime",
    "tests/indexing/test_loc.py::TestLocBaseIndependent::test_loc_reverse_assignment",
    "tests/indexing/test_loc.py::TestLocBaseIndependent::test_loc_setitem_consistency[0]",
    "tests/indexing/test_loc.py::TestLocBaseIndependent::test_loc_setitem_consistency[val2]",
    "tests/indexing/test_loc.py::TestLocBaseIndependent::test_loc_setitem_consistency_dt64_to_float",
    "tests/indexing/test_loc.py::TestLocBaseIndependent::test_loc_setitem_empty_append_expands_rows_mixed_dtype",
    "tests/indexing/test_loc.py::TestLocBaseIndependent::test_loc_setitem_frame_with_inverted_slice",
    "tests/indexing/test_loc.py::TestLocBaseIndependent::test_loc_uint64_disallow_negative",
    "tests/indexing/test_loc.py::TestLocBooleanMask::test_loc_setitem_mask_with_datetimeindex_tz[None]",
    "tests/indexing/test_loc.py::TestLocBooleanMask::test_loc_setitem_ndframe_values_alignment",
    "tests/indexing/test_loc.py::TestLocListlike::test_loc_named_index",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_datetime_keys_cast[np.datetime64]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_datetime_keys_cast[to_datetime64]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_datetime_keys_cast[to_pydatetime]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[bool-dtype-False]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[bool-dtype-True]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[bool-dtype]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[nullable_bool-False]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[nullable_bool-True]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[nullable_bool]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[object-False]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[object-True]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[object]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[string-False]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[string-True]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[string]",
    "tests/indexing/test_loc.py::TestLocWithMultiIndex::test_loc_drops_level",
    "tests/indexing/test_loc.py::TestLocWithMultiIndex::test_loc_getitem_multilevel_index_order[index-keys2-expected2]",
    "tests/indexing/test_loc.py::TestLocWithMultiIndex::test_loc_getitem_multilevel_index_order[index-keys3-expected3]",
    "tests/indexing/test_loc.py::TestLocWithMultiIndex::test_loc_getitem_multilevel_index_order[index-keys4-expected4]",
    "tests/indexing/test_loc.py::TestLocWithMultiIndex::test_loc_getitem_multilevel_index_order[index-keys5-expected5]",
    "tests/indexing/test_loc.py::TestLocWithMultiIndex::test_loc_getitem_multilevel_index_order[index-keys6-expected6]",
    "tests/indexing/test_loc.py::TestLocWithMultiIndex::test_loc_multiindex_null_slice_na_level",
    "tests/indexing/test_loc.py::TestLocWithMultiIndex::test_loc_set_nan_in_categorical_series[Float64]",
    "tests/indexing/test_loc.py::test_loc_datetimelike_mismatched_dtypes",
    "tests/indexing/test_loc.py::test_loc_getitem_multiindex_tuple_level",
    "tests/indexing/test_loc.py::test_loc_setitem_uint8_upcast[300]",
    "tests/indexing/test_na_indexing.py::test_series_mask_boolean[True-list-mask0-values0-object]",
    "tests/indexing/test_na_indexing.py::test_series_mask_boolean[True-list-mask1-values0-object]",
    "tests/indexing/test_na_indexing.py::test_series_mask_boolean[True-list-mask2-values0-object]",
    "tests/indexing/test_partial.py::TestEmptyFrameSetitemExpansion::test_empty_frame_setitem_index_name_retained",
    "tests/indexing/test_partial.py::TestEmptyFrameSetitemExpansion::test_partial_set_empty_frame",
    "tests/indexing/test_partial.py::TestEmptyFrameSetitemExpansion::test_partial_set_empty_frame2",
    "tests/indexing/test_partial.py::TestEmptyFrameSetitemExpansion::test_partial_set_empty_frame3",
    "tests/indexing/test_partial.py::TestEmptyFrameSetitemExpansion::test_partial_set_empty_frame5",
    "tests/indexing/test_partial.py::TestPartialSetting::test_loc_with_list_of_strings_representing_datetimes[DataFrame-idx1-labels1-expected_idx1]",
    "tests/indexing/test_partial.py::TestPartialSetting::test_partial_setting",
    "tests/indexing/test_partial.py::TestPartialSetting::test_partial_setting_frame",
    "tests/indexing/test_partial.py::TestPartialSetting::test_partial_setting_mixed_dtype",
    "tests/indexing/test_scalar.py::TestMultiIndexScalar::test_multiindex_at_get_one_level",
    "tests/interchange/test_impl.py::test_multi_chunk_column",
    "tests/interchange/test_impl.py::test_string_validity_buffer",
    "tests/interchange/test_impl.py::test_string_validity_buffer_no_missing",
    "tests/internals/test_api.py::test_create_block_manager_from_blocks_deprecated",
    "tests/internals/test_api.py::test_deprecations[Block]",
    "tests/internals/test_api.py::test_deprecations[DatetimeTZBlock]",
    "tests/internals/test_api.py::test_deprecations[ExtensionBlock]",
    "tests/internals/test_api.py::test_deprecations[NumericBlock]",
    "tests/internals/test_api.py::test_deprecations[ObjectBlock]",
    "tests/internals/test_internals.py::TestBlock::test_delete_datetimelike",
    "tests/internals/test_internals.py::TestBlockManager::test_categorical_block_pickle",
    "tests/internals/test_internals.py::TestBlockManager::test_copy",
    "tests/internals/test_internals.py::TestBlockManager::test_non_unique_pickle[a,a,a:f8]",
    "tests/internals/test_internals.py::TestBlockManager::test_non_unique_pickle[a: f8; a: i8]",
    "tests/internals/test_internals.py::TestBlockManager::test_pickle",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>0-float64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>0-int64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>0-uint64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>1-float64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>1-int64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>1-uint64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>2-float64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>2-int64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>2-uint64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>3-float64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>3-int64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>3-uint64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>4-float64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>4-int64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>4-uint64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>5-float64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>5-int64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>5-uint64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>6-float64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>6-int64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>6-uint64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>7-float64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>7-int64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_interval_can_hold_element[<lambda>7-uint64]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_period_can_hold_element[<lambda>0]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_period_can_hold_element[<lambda>1]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_period_can_hold_element[<lambda>2]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_period_can_hold_element[<lambda>3]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_period_can_hold_element[<lambda>4]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_period_can_hold_element[<lambda>5]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_period_can_hold_element[<lambda>6]",
    "tests/internals/test_internals.py::TestCanHoldElement::test_period_can_hold_element[<lambda>7]",
    "tests/internals/test_internals.py::TestShouldStore::test_should_store_categorical",
    "tests/internals/test_managers.py::test_dataframe_creation",
    "tests/internals/test_managers.py::test_series_creation",
    "tests/io/excel/test_odswriter.py::test_cell_value_type[1-float-value-1]",
    "tests/io/excel/test_odswriter.py::test_cell_value_type[1.5-float-value-1.5]",
    "tests/io/excel/test_odswriter.py::test_cell_value_type[True-boolean-boolean-value-true]",
    "tests/io/excel/test_odswriter.py::test_cell_value_type[test string-string-string-value-test string]",
    "tests/io/excel/test_odswriter.py::test_cell_value_type[value4-date-date-value-2010-10-10T10:10:10]",
    "tests/io/excel/test_odswriter.py::test_cell_value_type[value5-date-date-value-2010-10-10]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[('calamine', '.ods')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[('calamine', '.xls')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[('calamine', '.xlsb')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[('calamine', '.xlsm')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[('calamine', '.xlsx')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[('odf', '.ods')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[('openpyxl', '.xlsm')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[('openpyxl', '.xlsx')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[('pyxlsb', '.xlsb')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[('xlrd', '.xls')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[(None, '.xls')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[(None, '.xlsm')]",
    "tests/io/excel/test_readers.py::TestExcelFileRead::test_engine_used[(None, '.xlsx')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[('calamine', '.ods')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[('calamine', '.xls')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[('calamine', '.xlsb')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[('calamine', '.xlsm')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[('calamine', '.xlsx')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[('odf', '.ods')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[('openpyxl', '.xlsm')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[('openpyxl', '.xlsx')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[('pyxlsb', '.xlsb')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[('xlrd', '.xls')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[(None, '.xls')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[(None, '.xlsm')]",
    "tests/io/excel/test_readers.py::TestReaders::test_engine_used[(None, '.xlsx')]",
    "tests/io/excel/test_writers.py::TestExcelWriter::test_excel_date_datetime_format[odf-.ods]",
    "tests/io/excel/test_writers.py::TestExcelWriter::test_excel_date_datetime_format[openpyxl-.xlsm]",
    "tests/io/excel/test_writers.py::TestExcelWriter::test_excel_date_datetime_format[openpyxl-.xlsx]",
    "tests/io/excel/test_writers.py::TestExcelWriter::test_excel_date_datetime_format[xlsxwriter-.xlsx]",
    "tests/io/excel/test_writers.py::TestExcelWriterEngineTests::test_register_writer",
    "tests/io/excel/test_writers.py::TestFSPath::test_excelfile_fspath",
    "tests/io/formats/style/test_format.py::test_format_clear[format_index-_display_funcs_columns-kwargs2]",
    "tests/io/formats/style/test_html.py::test_from_custom_template_style",
    "tests/io/formats/style/test_html.py::test_from_custom_template_table",
    "tests/io/formats/style/test_html.py::test_html_template_extends_options",
    "tests/io/formats/style/test_html.py::test_replaced_css_class_names",
    "tests/io/formats/style/test_style.py::TestStyler::test_caption",
    "tests/io/formats/style/test_style.py::TestStyler::test_non_reducing_multi_slice_on_multiindex[slice_7]",
    "tests/io/formats/style/test_style.py::TestStyler::test_table_styles",
    "tests/io/formats/style/test_style.py::TestStyler::test_uuid",
    "tests/io/formats/style/test_style.py::test_clear",
    "tests/io/formats/style/test_style.py::test_copy[False-False-False]",
    "tests/io/formats/style/test_style.py::test_copy[False-False-True]",
    "tests/io/formats/style/test_style.py::test_copy[False-True-False]",
    "tests/io/formats/style/test_style.py::test_copy[False-True-True]",
    "tests/io/formats/style/test_style.py::test_copy[True-False-False]",
    "tests/io/formats/style/test_style.py::test_copy[True-False-True]",
    "tests/io/formats/style/test_style.py::test_copy[True-True-False]",
    "tests/io/formats/style/test_style.py::test_copy[True-True-True]",
    "tests/io/formats/test_eng_formatting.py::TestEngFormatter::test_nan",
    "tests/io/formats/test_format.py::TestDataFrameFormatting::test_info_repr",
    "tests/io/formats/test_format.py::TestDataFrameFormatting::test_info_repr_max_cols",
    "tests/io/formats/test_format.py::TestSeriesFormatting::test_east_asian_unicode_series",
    "tests/io/formats/test_ipython_compat.py::TestTableSchemaRepr::test_publishes",
    "tests/io/formats/test_ipython_compat.py::TestTableSchemaRepr::test_publishes_not_implemented",
    "tests/io/formats/test_to_csv.py::TestToCSV::test_to_csv_compression_dict[tar]",
    "tests/io/formats/test_to_csv.py::TestToCSV::test_to_csv_date_format",
    "tests/io/formats/test_to_csv.py::TestToCSV::test_to_csv_different_datetime_formats",
    "tests/io/formats/test_to_csv.py::TestToCSV::test_to_csv_multi_index",
    "tests/io/formats/test_to_csv.py::TestToCSV::test_to_csv_na_rep",
    "tests/io/formats/test_to_csv.py::TestToCSV::test_to_csv_with_single_column",
    "tests/io/formats/test_to_html.py::test_to_html_truncate",
    "tests/io/formats/test_to_string.py::TestDataFrameToString::test_to_string_truncate",
    "tests/io/formats/test_to_string.py::TestSeriesToString::test_to_string_with_datetimeindex",
    "tests/io/json/test_compression.py::test_chunksize_with_compression[bz2]",
    "tests/io/json/test_compression.py::test_chunksize_with_compression[gzip]",
    "tests/io/json/test_compression.py::test_chunksize_with_compression[tar]",
    "tests/io/json/test_compression.py::test_chunksize_with_compression[xz]",
    "tests/io/json/test_compression.py::test_chunksize_with_compression[zip]",
    "tests/io/json/test_compression.py::test_chunksize_with_compression[zstd]",
    "tests/io/json/test_compression.py::test_compression_roundtrip[bz2]",
    "tests/io/json/test_compression.py::test_compression_roundtrip[gzip]",
    "tests/io/json/test_compression.py::test_compression_roundtrip[tar]",
    "tests/io/json/test_compression.py::test_compression_roundtrip[xz]",
    "tests/io/json/test_compression.py::test_compression_roundtrip[zip]",
    "tests/io/json/test_compression.py::test_compression_roundtrip[zstd]",
    "tests/io/json/test_compression.py::test_lines_with_compression[bz2]",
    "tests/io/json/test_compression.py::test_lines_with_compression[gzip]",
    "tests/io/json/test_compression.py::test_lines_with_compression[tar]",
    "tests/io/json/test_compression.py::test_lines_with_compression[xz]",
    "tests/io/json/test_compression.py::test_lines_with_compression[zip]",
    "tests/io/json/test_compression.py::test_lines_with_compression[zstd]",
    "tests/io/json/test_compression.py::test_write_unsupported_compression_type",
    "tests/io/json/test_json_table_schema.py::TestTableOrient::test_build_series",
    "tests/io/json/test_json_table_schema.py::TestTableOrient::test_to_json_categorical_index",
    "tests/io/json/test_json_table_schema.py::TestTableOrient::test_to_json_float_index",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_empty_frame_roundtrip",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_multiindex[index_names0]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_multiindex[index_names1]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_multiindex[index_names2]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_multiindex[index_names3]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_multiindex[index_names4]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals0-None]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals0-idx]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals0-level_0]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals1-None]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals1-idx]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals1-level_0]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals2-None]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals2-idx]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals2-level_0]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals6-None]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals6-idx]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals6-level_0]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals7-None]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals7-idx]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals7-level_0]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals8-None]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals8-idx]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_orient[vals8-level_0]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_timezones_orient[vals0-idx0]",
    "tests/io/json/test_json_table_schema.py::TestTableOrientReader::test_read_json_table_timezones_orient[vals0-idx1]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_blocks_compat_GH9037",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_date_index_and_values[date-False-epoch]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_date_index_and_values[date-False-iso]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_dtypes",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns0-None]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns0-index1]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns0-index2]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns0-index3]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns0-index4]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns0-index5]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns1-None]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns1-index1]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns1-index2]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns1-index3]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns1-index4]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns1-index5]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns2-None]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns2-index1]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns2-index2]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns2-index3]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns2-index4]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_from_json_to_json_table_index_and_columns[columns2-index5]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_index_false_from_json_to_json[False-table]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_index_false_from_json_to_json[True-table]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_index_false_to_json_table[data0]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_index_false_to_json_table[data1]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_index_false_to_json_table[data2]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_index_false_to_json_table[data3]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_index_false_to_json_table[data4]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_index_false_to_json_table[data5]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_read_timezone_information",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_timedelta",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_timedelta2",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_timedelta_to_json[timedelta-epoch-False]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_timedelta_to_json[timedelta-iso-False]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_timedelta_to_json_fractional_precision[timedelta-False]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_timedelta_to_json_fractional_precision[timedelta-True]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_tz_range_is_utc[tz_range0]",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_tz_range_is_utc[tz_range2]",
    "tests/io/json/test_readlines.py::test_chunksize_is_incremental",
    "tests/io/json/test_readlines.py::test_readjson_chunks_closes[1]",
    "tests/io/json/test_readlines.py::test_readjson_chunks_closes[None]",
    "tests/io/json/test_readlines.py::test_to_json_append_output_different_columns",
    "tests/io/json/test_readlines.py::test_to_json_append_output_different_columns_reordered",
    "tests/io/parser/common/test_common_basic.py::test_override_set_noconvert_columns",
    "tests/io/parser/common/test_file_buffer_url.py::test_context_manager[c_high]",
    "tests/io/parser/common/test_file_buffer_url.py::test_context_manager[c_low]",
    "tests/io/parser/common/test_file_buffer_url.py::test_context_manager[python]",
    "tests/io/parser/common/test_file_buffer_url.py::test_context_manageri_user_provided[c_high]",
    "tests/io/parser/common/test_file_buffer_url.py::test_context_manageri_user_provided[c_low]",
    "tests/io/parser/common/test_file_buffer_url.py::test_context_manageri_user_provided[python]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict[c_high-float64]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict[c_high-float]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict[c_low-float64]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict[c_low-float]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict[python-float64]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict[python-float]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict_invalid[c_high]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict_invalid[c_low]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict_invalid[python]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict_mangle_dup_cols[c_high]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict_mangle_dup_cols[c_low]",
    "tests/io/parser/dtypes/test_dtypes_basic.py::test_dtypes_defaultdict_mangle_dup_cols[python]",
    "tests/io/parser/test_compression.py::test_writes_tar_gz[c_high]",
    "tests/io/parser/test_compression.py::test_writes_tar_gz[c_low]",
    "tests/io/parser/test_compression.py::test_writes_tar_gz[python]",
    "tests/io/parser/test_parse_dates.py::test_datetime_fractional_seconds[c_high-date_format-%Y %m %d %H %M %S.%f-None]",
    "tests/io/parser/test_parse_dates.py::test_datetime_fractional_seconds[c_high-date_parser-<lambda>-FutureWarning]",
    "tests/io/parser/test_parse_dates.py::test_datetime_fractional_seconds[c_low-date_format-%Y %m %d %H %M %S.%f-None]",
    "tests/io/parser/test_parse_dates.py::test_datetime_fractional_seconds[c_low-date_parser-<lambda>-FutureWarning]",
    "tests/io/parser/test_parse_dates.py::test_datetime_fractional_seconds[python-date_format-%Y %m %d %H %M %S.%f-None]",
    "tests/io/parser/test_parse_dates.py::test_datetime_fractional_seconds[python-date_parser-<lambda>-FutureWarning]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col[c_high-False]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col[c_high-True]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col[c_low-False]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col[c_low-True]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col[python-False]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col[python-True]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col_custom[c_high-False]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col_custom[c_high-True]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col_custom[c_low-False]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col_custom[c_low-True]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col_custom[python-False]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_col_custom[python-True]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_cols_int_cast[c_high]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_cols_int_cast[c_low]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_cols_int_cast[python]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_cols_with_header[c_high]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_cols_with_header[c_low]",
    "tests/io/parser/test_parse_dates.py::test_multiple_date_cols_with_header[python]",
    "tests/io/parser/test_parse_dates.py::test_nat_parse[c_high]",
    "tests/io/parser/test_parse_dates.py::test_nat_parse[c_low]",
    "tests/io/parser/test_parse_dates.py::test_nat_parse[python]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_all_fields[c_high-date_format-%Y %m %d %H %M %S-None]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_all_fields[c_high-date_parser-<lambda>-FutureWarning]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_all_fields[c_low-date_format-%Y %m %d %H %M %S-None]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_all_fields[c_low-date_parser-<lambda>-FutureWarning]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_all_fields[python-date_format-%Y %m %d %H %M %S-None]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_all_fields[python-date_parser-<lambda>-FutureWarning]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_fields[c_high]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_fields[c_low]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_fields[python]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[c_high-KORD,19990127, 19:00:00, 18:56:00, 0.8100\\\\\\\\nKORD,19990127, 20:00:00, 19:56:00, 0.0100\\\\\\\\nKORD,19990127, 21:00:00, 20:56:00, -0.5900\\\\\\\\nKORD,19990127, 21:00:00, 21:18:00, -0.9900\\\\\\\\nKORD,19990127, 22:00:00, 21:56:00, -0.5900\\\\\\\\nKORD,19990127, 23:00:00, 22:56:00, -0.5900-kwargs1-expected1]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[c_high-KORD,19990127, 19:00:00, 18:56:00, 0.8100\\nKORD,19990127, 20:00:00, 19:56:00, 0.0100\\nKORD,19990127, 21:00:00, 20:56:00, -0.5900\\nKORD,19990127, 21:00:00, 21:18:00, -0.9900\\nKORD,19990127, 22:00:00, 21:56:00, -0.5900\\nKORD,19990127, 23:00:00, 22:56:00, -0.5900-kwargs1-expected1]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[c_high-date,time,a,b\\\\\\\\n2001-01-05, 10:00:00, 0.0, 10.\\\\\\\\n2001-01-05, 00:00:00, 1., 11.\\\\\\\\n-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[c_high-date,time,a,b\\n2001-01-05, 10:00:00, 0.0, 10.\\n2001-01-05, 00:00:00, 1., 11.\\n-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[c_low-KORD,19990127, 19:00:00, 18:56:00, 0.8100\\\\\\\\nKORD,19990127, 20:00:00, 19:56:00, 0.0100\\\\\\\\nKORD,19990127, 21:00:00, 20:56:00, -0.5900\\\\\\\\nKORD,19990127, 21:00:00, 21:18:00, -0.9900\\\\\\\\nKORD,19990127, 22:00:00, 21:56:00, -0.5900\\\\\\\\nKORD,19990127, 23:00:00, 22:56:00, -0.5900-kwargs1-expected1]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[c_low-KORD,19990127, 19:00:00, 18:56:00, 0.8100\\nKORD,19990127, 20:00:00, 19:56:00, 0.0100\\nKORD,19990127, 21:00:00, 20:56:00, -0.5900\\nKORD,19990127, 21:00:00, 21:18:00, -0.9900\\nKORD,19990127, 22:00:00, 21:56:00, -0.5900\\nKORD,19990127, 23:00:00, 22:56:00, -0.5900-kwargs1-expected1]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[c_low-date,time,a,b\\\\\\\\n2001-01-05, 10:00:00, 0.0, 10.\\\\\\\\n2001-01-05, 00:00:00, 1., 11.\\\\\\\\n-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[c_low-date,time,a,b\\n2001-01-05, 10:00:00, 0.0, 10.\\n2001-01-05, 00:00:00, 1., 11.\\n-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[python-KORD,19990127, 19:00:00, 18:56:00, 0.8100\\\\\\\\nKORD,19990127, 20:00:00, 19:56:00, 0.0100\\\\\\\\nKORD,19990127, 21:00:00, 20:56:00, -0.5900\\\\\\\\nKORD,19990127, 21:00:00, 21:18:00, -0.9900\\\\\\\\nKORD,19990127, 22:00:00, 21:56:00, -0.5900\\\\\\\\nKORD,19990127, 23:00:00, 22:56:00, -0.5900-kwargs1-expected1]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[python-KORD,19990127, 19:00:00, 18:56:00, 0.8100\\nKORD,19990127, 20:00:00, 19:56:00, 0.0100\\nKORD,19990127, 21:00:00, 20:56:00, -0.5900\\nKORD,19990127, 21:00:00, 21:18:00, -0.9900\\nKORD,19990127, 22:00:00, 21:56:00, -0.5900\\nKORD,19990127, 23:00:00, 22:56:00, -0.5900-kwargs1-expected1]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[python-date,time,a,b\\\\\\\\n2001-01-05, 10:00:00, 0.0, 10.\\\\\\\\n2001-01-05, 00:00:00, 1., 11.\\\\\\\\n-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time[python-date,time,a,b\\n2001-01-05, 10:00:00, 0.0, 10.\\n2001-01-05, 00:00:00, 1., 11.\\n-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time_multi_level_column_name[c_high]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time_multi_level_column_name[c_low]",
    "tests/io/parser/test_parse_dates.py::test_parse_date_time_multi_level_column_name[python]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_column_list[c_high-parse_dates0]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_column_list[c_high-parse_dates1]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_column_list[c_low-parse_dates0]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_column_list[c_low-parse_dates1]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_column_list[python-parse_dates0]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_column_list[python-parse_dates1]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_high-a,b\\\\\\\\n04.15.2016,09.16.2013-kwargs2-expected2]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_high-a,b\\n04.15.2016,09.16.2013-kwargs2-expected2]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_high-a\\\\\\\\n04.15.2016-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_high-a\\n04.15.2016-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_low-a,b\\\\\\\\n04.15.2016,09.16.2013-kwargs2-expected2]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_low-a,b\\n04.15.2016,09.16.2013-kwargs2-expected2]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_low-a\\\\\\\\n04.15.2016-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_low-a\\n04.15.2016-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[python-a,b\\\\\\\\n04.15.2016,09.16.2013-kwargs2-expected2]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[python-a,b\\n04.15.2016,09.16.2013-kwargs2-expected2]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[python-a\\\\\\\\n04.15.2016-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[python-a\\n04.15.2016-kwargs0-expected0]",
    "tests/io/parser/test_parse_dates.py::test_separator_date_conflict[c_high]",
    "tests/io/parser/test_parse_dates.py::test_separator_date_conflict[c_low]",
    "tests/io/parser/test_parse_dates.py::test_separator_date_conflict[python]",
    "tests/io/parser/test_python_parser_only.py::test_on_bad_lines_callable_write_to_external_list[python]",
    "tests/io/parser/test_textreader.py::TestTextReader::test_integer_thousands_alt",
    "tests/io/pytables/test_timezones.py::test_append_with_timezones[<lambda>0]",
    "tests/io/pytables/test_timezones.py::test_append_with_timezones[<lambda>1]",
    "tests/io/pytables/test_timezones.py::test_py2_created_with_datetimez",
    "tests/io/sas/test_sas7bdat.py::TestSAS7BDAT::test_iterator_read_too_much",
    "tests/io/sas/test_sas7bdat.py::test_date_time",
    "tests/io/sas/test_sas7bdat.py::test_encoding_infer",
    "tests/io/sas/test_sas7bdat.py::test_encoding_options",
    "tests/io/test_common.py::TestCommonIOCapabilities::test_stringify_file_and_path_like",
    "tests/io/test_common.py::test_pickle_reader[read_excel]",
    "tests/io/test_common.py::test_pickle_reader[read_fwf]",
    "tests/io/test_common.py::test_pickle_reader[read_pickle]",
    "tests/io/test_common.py::test_pickle_reader[read_sas]",
    "tests/io/test_common.py::test_pickle_reader[read_stata]",
    "tests/io/test_compression.py::test_ambiguous_archive_tar",
    "tests/io/test_compression.py::test_compression_size[bz2-to_json-obj0]",
    "tests/io/test_compression.py::test_compression_size[bz2-to_json-obj1]",
    "tests/io/test_compression.py::test_compression_size[gzip-to_json-obj0]",
    "tests/io/test_compression.py::test_compression_size[gzip-to_json-obj1]",
    "tests/io/test_compression.py::test_compression_size[tar-to_json-obj0]",
    "tests/io/test_compression.py::test_compression_size[tar-to_json-obj1]",
    "tests/io/test_compression.py::test_compression_size[xz-to_json-obj0]",
    "tests/io/test_compression.py::test_compression_size[xz-to_json-obj1]",
    "tests/io/test_compression.py::test_compression_size[zip-to_json-obj0]",
    "tests/io/test_compression.py::test_compression_size[zip-to_json-obj1]",
    "tests/io/test_compression.py::test_compression_size[zstd-to_json-obj0]",
    "tests/io/test_compression.py::test_compression_size[zstd-to_json-obj1]",
    "tests/io/test_compression.py::test_dataframe_compression_defaults_to_infer[bz2-to_csv-write_kwargs0-read_csv]",
    "tests/io/test_compression.py::test_dataframe_compression_defaults_to_infer[gzip-to_csv-write_kwargs0-read_csv]",
    "tests/io/test_compression.py::test_dataframe_compression_defaults_to_infer[tar-to_csv-write_kwargs0-read_csv]",
    "tests/io/test_compression.py::test_dataframe_compression_defaults_to_infer[xz-to_csv-write_kwargs0-read_csv]",
    "tests/io/test_compression.py::test_dataframe_compression_defaults_to_infer[zip-to_csv-write_kwargs0-read_csv]",
    "tests/io/test_compression.py::test_dataframe_compression_defaults_to_infer[zstd-to_csv-write_kwargs0-read_csv]",
    "tests/io/test_compression.py::test_gzip_compression_level[to_json-obj0]",
    "tests/io/test_compression.py::test_gzip_compression_level[to_json-obj1]",
    "tests/io/test_compression.py::test_xz_compression_level_read[to_json-obj0]",
    "tests/io/test_compression.py::test_xz_compression_level_read[to_json-obj1]",
    "tests/io/test_fsspec.py::test_non_fsspec_options",
    "tests/io/test_orc.py::test_orc_reader_basic",
    "tests/io/test_orc.py::test_orc_roundtrip_file",
    "tests/io/test_orc.py::test_string_inference",
    "tests/io/test_parquet.py::TestBasic::test_read_empty_array[Float64]",
    "tests/io/test_parquet.py::TestBasic::test_read_empty_array[Int64]",
    "tests/io/test_parquet.py::TestBasic::test_read_empty_array[UInt8]",
    "tests/io/test_parquet.py::TestBasic::test_read_empty_array[boolean]",
    "tests/io/test_parquet.py::TestBasic::test_read_empty_array[float]",
    "tests/io/test_parquet.py::TestBasic::test_read_empty_array[object]",
    "tests/io/test_parquet.py::TestBasic::test_read_empty_array[string]",
    "tests/io/test_parquet.py::TestParquetPyArrow::test_empty_dataframe",
    "tests/io/test_parquet.py::TestParquetPyArrow::test_infer_string_large_string_type",
    "tests/io/test_parquet.py::TestParquetPyArrow::test_read_dtype_backend_pyarrow_config",
    "tests/io/test_pickle.py::test_pickles",
    "tests/io/test_sql.py::test_api_execute_sql[sqlite_buildin_iris]",
    "tests/io/test_sql.py::test_api_execute_sql[sqlite_conn_iris]",
    "tests/io/test_sql.py::test_api_execute_sql[sqlite_engine_iris]",
    "tests/io/test_sql.py::test_api_execute_sql[sqlite_str_iris]",
    "tests/io/test_sql.py::test_datetime_date[sqlite_conn]",
    "tests/io/test_sql.py::test_datetime_date[sqlite_engine]",
    "tests/io/test_sql.py::test_datetime_date[sqlite_str]",
    "tests/io/test_sql.py::test_execute_sql[sqlite_buildin_iris]",
    "tests/io/test_sql.py::test_execute_sql[sqlite_conn_iris]",
    "tests/io/test_sql.py::test_execute_sql[sqlite_engine_iris]",
    "tests/io/test_sql.py::test_execute_sql[sqlite_str_iris]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-numpy_nullable-read_sql-sqlite_buildin]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-numpy_nullable-read_sql-sqlite_conn]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-numpy_nullable-read_sql-sqlite_engine]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-numpy_nullable-read_sql-sqlite_str]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-numpy_nullable-read_sql_query-sqlite_buildin]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-numpy_nullable-read_sql_query-sqlite_conn]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-numpy_nullable-read_sql_query-sqlite_engine]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-numpy_nullable-read_sql_query-sqlite_str]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-pyarrow-read_sql-sqlite_buildin]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-pyarrow-read_sql-sqlite_conn]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-pyarrow-read_sql-sqlite_engine]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-pyarrow-read_sql-sqlite_str]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-pyarrow-read_sql_query-sqlite_buildin]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-pyarrow-read_sql_query-sqlite_conn]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-pyarrow-read_sql_query-sqlite_engine]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[pyarrow-pyarrow-read_sql_query-sqlite_str]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-numpy_nullable-read_sql-sqlite_buildin]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-numpy_nullable-read_sql-sqlite_conn]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-numpy_nullable-read_sql-sqlite_engine]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-numpy_nullable-read_sql-sqlite_str]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-numpy_nullable-read_sql_query-sqlite_buildin]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-numpy_nullable-read_sql_query-sqlite_conn]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-numpy_nullable-read_sql_query-sqlite_engine]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-numpy_nullable-read_sql_query-sqlite_str]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-pyarrow-read_sql-sqlite_buildin]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-pyarrow-read_sql-sqlite_conn]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-pyarrow-read_sql-sqlite_engine]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-pyarrow-read_sql-sqlite_str]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-pyarrow-read_sql_query-sqlite_buildin]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-pyarrow-read_sql_query-sqlite_conn]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-pyarrow-read_sql_query-sqlite_engine]",
    "tests/io/test_sql.py::test_read_sql_dtype_backend[python-pyarrow-read_sql_query-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int16-SMALLINT-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int16-SMALLINT-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int16-SMALLINT-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int32-INTEGER-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int32-INTEGER-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int32-INTEGER-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int64-BIGINT-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int64-BIGINT-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int64-BIGINT-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int8-SMALLINT-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int8-SMALLINT-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[Int8-SMALLINT-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[UInt16-INTEGER-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[UInt16-INTEGER-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[UInt16-INTEGER-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[UInt32-BIGINT-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[UInt32-BIGINT-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[UInt32-BIGINT-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[UInt8-SMALLINT-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[UInt8-SMALLINT-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[UInt8-SMALLINT-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int-BIGINT-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int-BIGINT-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int-BIGINT-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int16-SMALLINT-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int16-SMALLINT-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int16-SMALLINT-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int32-INTEGER-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int32-INTEGER-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int32-INTEGER-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int64-BIGINT-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int64-BIGINT-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int64-BIGINT-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int8-SMALLINT-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int8-SMALLINT-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[int8-SMALLINT-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[uint16-INTEGER-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[uint16-INTEGER-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[uint16-INTEGER-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[uint32-BIGINT-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[uint32-BIGINT-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[uint32-BIGINT-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[uint8-SMALLINT-sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[uint8-SMALLINT-sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_integer_mapping[uint8-SMALLINT-sqlite_str]",
    "tests/io/test_sql.py::test_sqlalchemy_type_mapping[sqlite_conn]",
    "tests/io/test_sql.py::test_sqlalchemy_type_mapping[sqlite_engine]",
    "tests/io/test_sql.py::test_sqlalchemy_type_mapping[sqlite_str]",
    "tests/io/test_sql.py::test_sqlite_datetime_date",
    "tests/io/test_sql.py::test_to_sql_callable[sqlite_conn]",
    "tests/io/test_sql.py::test_to_sql_callable[sqlite_engine]",
    "tests/io/test_sql.py::test_to_sql_callable[sqlite_str]",
    "tests/io/test_sql.py::test_xsqlite_if_exists",
    "tests/io/test_sql.py::test_xsqlite_write_row_by_row",
    "tests/io/test_stata.py::TestStata::test_categorical_writing[114]",
    "tests/io/test_stata.py::TestStata::test_categorical_writing[117]",
    "tests/io/test_stata.py::TestStata::test_categorical_writing[118]",
    "tests/io/test_stata.py::TestStata::test_categorical_writing[119]",
    "tests/io/test_stata.py::TestStata::test_categorical_writing[None]",
    "tests/io/test_stata.py::TestStata::test_date_export_formats",
    "tests/io/test_stata.py::TestStata::test_dates_invalid_column",
    "tests/io/test_stata.py::TestStata::test_default_date_conversion",
    "tests/io/test_stata.py::TestStata::test_encoding_latin1_118",
    "tests/io/test_stata.py::TestStata::test_minimal_size_col",
    "tests/io/test_stata.py::TestStata::test_missing_value_generator",
    "tests/io/test_stata.py::TestStata::test_read_dta18",
    "tests/io/test_stata.py::TestStata::test_read_write_ea_dtypes[numpy_nullable]",
    "tests/io/test_stata.py::TestStata::test_read_write_ea_dtypes[pyarrow]",
    "tests/io/test_stata.py::TestStata::test_timestamp_and_label[114]",
    "tests/io/test_stata.py::TestStata::test_timestamp_and_label[117]",
    "tests/io/test_stata.py::TestStata::test_timestamp_and_label[118]",
    "tests/io/test_stata.py::TestStata::test_timestamp_and_label[119]",
    "tests/io/test_stata.py::TestStata::test_timestamp_and_label[None]",
    "tests/io/test_stata.py::TestStata::test_utf8_writer[118]",
    "tests/io/test_stata.py::TestStata::test_utf8_writer[119]",
    "tests/io/test_stata.py::TestStata::test_utf8_writer[None]",
    "tests/io/test_stata.py::TestStata::test_value_labels_old_format",
    "tests/io/test_stata.py::TestStata::test_variable_labels",
    "tests/io/test_stata.py::TestStata::test_write_variable_labels[114]",
    "tests/io/test_stata.py::TestStata::test_write_variable_labels[117]",
    "tests/io/test_stata.py::TestStata::test_write_variable_labels[118]",
    "tests/io/test_stata.py::TestStata::test_write_variable_labels[119]",
    "tests/io/test_stata.py::TestStata::test_write_variable_labels[None]",
    "tests/io/test_stata.py::test_chunked_categorical[114]",
    "tests/io/test_stata.py::test_chunked_categorical[117]",
    "tests/io/test_stata.py::test_chunked_categorical[118]",
    "tests/io/test_stata.py::test_chunked_categorical[119]",
    "tests/io/test_stata.py::test_chunked_categorical[None]",
    "tests/io/test_stata.py::test_chunked_categorical_partial",
    "tests/io/test_stata.py::test_direct_read",
    "tests/io/test_stata.py::test_iterator_errors[-1]",
    "tests/io/test_stata.py::test_iterator_errors[0]",
    "tests/io/test_stata.py::test_iterator_errors[apple]",
    "tests/io/test_stata.py::test_non_categorical_value_label_convert_categoricals_error",
    "tests/io/test_stata.py::test_non_categorical_value_label_name_conversion",
    "tests/io/test_stata.py::test_non_categorical_value_labels",
    "tests/io/test_stata.py::test_statareader_warns_when_used_without_context",
    "tests/io/xml/test_to_xml.py::test_style_to_csv",
    "tests/io/xml/test_xml.py::test_empty_stylesheet[0]",
    "tests/io/xml/test_xml.py::test_empty_stylesheet[1]",
    "tests/io/xml/test_xml.py::test_wrong_file_path[etree]",
    "tests/io/xml/test_xml.py::test_wrong_file_path[lxml]",
    "tests/libs/test_libalgos.py::test_ensure_platform_int",
    "tests/plotting/frame/test_frame.py::TestDataFramePlots::test_unordered_ts",
    "tests/plotting/test_boxplot_method.py::TestDataFramePlots::test_boxplot_legacy1_series",
    "tests/plotting/test_converter.py::TestDateTimeConverter::test_conversion",
    "tests/plotting/test_converter.py::TestDateTimeConverter::test_dateindex_conversion[B]",
    "tests/plotting/test_converter.py::TestDateTimeConverter::test_dateindex_conversion[ms]",
    "tests/plotting/test_converter.py::TestDateTimeConverter::test_dateindex_conversion[s]",
    "tests/plotting/test_converter.py::TestRegistration::test_matplotlib_formatters",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_frame_inferred",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_frame_inferred_n_gt_1",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_high_freq[ms]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_high_freq[us]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_frame[1B30Min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_frame[D]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_frame[ME]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_frame[QE-DEC]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_frame[W]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_frame[YE]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_frame[h]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_frame[min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_frame[s]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_series[1B30Min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_series[D]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_series[ME]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_series[QE-DEC]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_series[W]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_series[YE]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_series[h]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_series[min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_datetime_series[s]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_inferred_freq[D]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_inferred_freq[ME]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_inferred_freq[QE-DEC]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_inferred_freq[W]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_inferred_freq[YE]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_inferred_freq[h]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_inferred_freq[min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_inferred_freq[s]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_frame[D]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_frame[ME]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_frame[QE]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_frame[W]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_frame[YE]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_frame[h]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_frame[min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_frame[s]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_frame[11M]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_frame[1s]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_frame[3Y]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_frame[3s]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_frame[4D]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_frame[5min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_frame[7h]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_frame[8W]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_series[11M]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_series[1s]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_series[3Y]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_series[3s]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_series[4D]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_series[5min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_series[7h]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_mlt_series[8W]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_series[D]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_series[M]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_series[Q]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_series[W]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_series[Y]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_series[h]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_series[min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_line_plot_period_series[s]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_mixed_freq_irregular_first_df",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_plot_multiple_inferred_freq",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_plot_offset_freq",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_plot_offset_freq_business",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz['+01:15']",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz['-02:15']",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz['Asia/Tokyo']",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz['US/Eastern']",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz['UTC']",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz['UTC+01:15']",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz['UTC-02:15']",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz['dateutil/Asia/Singapore']",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz['dateutil/US/Pacific']",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz[<UTC>]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz[datetime.timezone(datetime.timedelta(seconds=3600))]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz[datetime.timezone.utc]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz[pytz.FixedOffset(-300)]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz[pytz.FixedOffset(300)]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz[tzlocal()]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz[tzutc()]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz[zoneinfo.ZoneInfo(key='US/Pacific')]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_ts_plot_with_tz[zoneinfo.ZoneInfo(key='UTC')]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_datetime[1B30Min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_datetime[D]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_datetime[ME]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_datetime[QE-DEC]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_datetime[W]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_datetime[YE]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_datetime[h]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_datetime[min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_datetime[s]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_period[D]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_period[M]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_period[Q]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_period[W]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_period[Y]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_period[h]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_period[min]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_tsplot_period[s]",
    "tests/reductions/test_reductions.py::TestDatetime64SeriesReductions::test_minmax_nat_series[nat_ser0]",
    "tests/reductions/test_reductions.py::TestDatetime64SeriesReductions::test_minmax_nat_series[nat_ser1]",
    "tests/reductions/test_reductions.py::TestDatetime64SeriesReductions::test_minmax_nat_series[nat_ser2]",
    "tests/reductions/test_reductions.py::TestIndexReductions::test_minmax_timedelta_empty_or_na[max]",
    "tests/reductions/test_reductions.py::TestIndexReductions::test_minmax_timedelta_empty_or_na[min]",
    "tests/reductions/test_reductions.py::TestReductions::test_nanminmax[index-datetime64[ns]-val2-max]",
    "tests/reductions/test_reductions.py::TestReductions::test_nanminmax[index-datetime64[ns]-val2-min]",
    "tests/reductions/test_reductions.py::TestReductions::test_nanminmax[series-datetime64[ns]-val2-max]",
    "tests/reductions/test_reductions.py::TestReductions::test_nanminmax[series-datetime64[ns]-val2-min]",
    "tests/reductions/test_reductions.py::TestReductions::test_nanops_empty_object[index-M8[ns]-max]",
    "tests/reductions/test_reductions.py::TestReductions::test_nanops_empty_object[index-M8[ns]-min]",
    "tests/reductions/test_reductions.py::TestReductions::test_nanops_empty_object[series-M8[ns]-max]",
    "tests/reductions/test_reductions.py::TestReductions::test_nanops_empty_object[series-M8[ns]-min]",
    "tests/reductions/test_reductions.py::TestSeriesMode::test_mode_category[False-expected11-expected21-expected31]",
    "tests/reductions/test_reductions.py::TestSeriesMode::test_mode_datetime[False-expected11-expected21]",
    "tests/reductions/test_reductions.py::TestSeriesMode::test_mode_timedelta[False-expected11-expected21]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_any_all_nullable_kleene_logic[data2-expected_data2-False-any-Float64]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_any_all_nullable_kleene_logic[data2-expected_data2-False-any-Int64]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_any_all_nullable_kleene_logic[data2-expected_data2-False-any-UInt64]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_any_all_nullable_kleene_logic[data2-expected_data2-False-any-boolean]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_any_all_nullable_kleene_logic[data3-expected_data3-False-any-Float64]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_any_all_nullable_kleene_logic[data3-expected_data3-False-any-Int64]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_any_all_nullable_kleene_logic[data3-expected_data3-False-any-UInt64]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_any_all_nullable_kleene_logic[data3-expected_data3-False-any-boolean]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_any_all_object_dtype_missing[data0-any]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_any_all_object_dtype_missing[data1-any]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_empty_timeseries_reductions_return_nat[False-M8[ns]]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_empty_timeseries_reductions_return_nat[False-m8[ns]0]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_empty_timeseries_reductions_return_nat[False-m8[ns]1]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_empty_timeseries_reductions_return_nat[True-M8[ns]]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_empty_timeseries_reductions_return_nat[True-m8[ns]0]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_empty_timeseries_reductions_return_nat[True-m8[ns]1]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_ops_consistency_on_empty_nullable[Int64-mean]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_ops_consistency_on_empty_nullable[Int64-var]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_ops_consistency_on_empty_nullable[boolean-mean]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_ops_consistency_on_empty_nullable[boolean-var]",
    "tests/reductions/test_reductions.py::TestSeriesReductions::test_sum_inf",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_dt64_mean[None-Index]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_dt64_mean[None-Series]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_dt64_mean[None-array]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_dt64_mean[zoneinfo.ZoneInfo(key='US/Pacific')-Index]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_dt64_mean[zoneinfo.ZoneInfo(key='US/Pacific')-Series]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_dt64_mean[zoneinfo.ZoneInfo(key='US/Pacific')-array]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_dt64_mean[zoneinfo.ZoneInfo(key='UTC')-Index]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_dt64_mean[zoneinfo.ZoneInfo(key='UTC')-Series]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_dt64_mean[zoneinfo.ZoneInfo(key='UTC')-array]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[B-Index]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[B-Series]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[B-array]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[D-Index]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[D-Series]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[D-array]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[W-Index]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[W-Series]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[W-array]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[h-Index]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[h-Series]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[h-array]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[s-Index]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[s-Series]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_period_mean[s-array]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_td64_mean[Index]",
    "tests/reductions/test_stat_reductions.py::TestDatetimeLikeStatReductions::test_td64_mean[Series]",
    "tests/resample/test_datetime_index.py::test_arrow_timestamp_resample[None]",
    "tests/resample/test_datetime_index.py::test_resample_anchored_intraday2[ms]",
    "tests/resample/test_datetime_index.py::test_resample_anchored_intraday2[ns]",
    "tests/resample/test_datetime_index.py::test_resample_anchored_intraday2[s]",
    "tests/resample/test_datetime_index.py::test_resample_anchored_intraday2[us]",
    "tests/resample/test_datetime_index.py::test_resample_datetime_values[ms]",
    "tests/resample/test_datetime_index.py::test_resample_datetime_values[ns]",
    "tests/resample/test_datetime_index.py::test_resample_datetime_values[s]",
    "tests/resample/test_datetime_index.py::test_resample_datetime_values[us]",
    "tests/resample/test_datetime_index.py::test_resample_empty_series_with_tz",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ms-10-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ms-10-10000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ms-100-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ms-100-10000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ms-1000-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ms-1000-10000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ns-10-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ns-10-10000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ns-100-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ns-100-10000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ns-1000-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[ns-1000-10000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[s-10-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[s-10-10000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[s-100-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[s-100-10000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[s-1000-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[s-1000-10000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[us-10-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[us-10-10000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[us-100-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[us-100-10000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[us-1000-100000]",
    "tests/resample/test_datetime_index.py::test_resample_group_info[us-1000-10000]",
    "tests/resample/test_datetime_index.py::test_resample_nunique[ms]",
    "tests/resample/test_datetime_index.py::test_resample_nunique[ns]",
    "tests/resample/test_datetime_index.py::test_resample_nunique[s]",
    "tests/resample/test_datetime_index.py::test_resample_nunique[us]",
    "tests/resample/test_datetime_index.py::test_resample_origin_epoch_with_tz_day_vs_24h[ms]",
    "tests/resample/test_datetime_index.py::test_resample_origin_epoch_with_tz_day_vs_24h[ns]",
    "tests/resample/test_datetime_index.py::test_resample_origin_epoch_with_tz_day_vs_24h[s]",
    "tests/resample/test_datetime_index.py::test_resample_origin_epoch_with_tz_day_vs_24h[us]",
    "tests/resample/test_datetime_index.py::test_resample_rounding[ms]",
    "tests/resample/test_datetime_index.py::test_resample_rounding[ns]",
    "tests/resample/test_datetime_index.py::test_resample_rounding[us]",
    "tests/resample/test_datetime_index.py::test_resample_size[ms]",
    "tests/resample/test_datetime_index.py::test_resample_size[ns]",
    "tests/resample/test_datetime_index.py::test_resample_size[s]",
    "tests/resample/test_datetime_index.py::test_resample_size[us]",
    "tests/resample/test_datetime_index.py::test_resample_upsampling_picked_but_not_correct[ms]",
    "tests/resample/test_datetime_index.py::test_resample_upsampling_picked_but_not_correct[ns]",
    "tests/resample/test_datetime_index.py::test_resample_upsampling_picked_but_not_correct[s]",
    "tests/resample/test_datetime_index.py::test_resample_upsampling_picked_but_not_correct[us]",
    "tests/resample/test_period_index.py::TestPeriodIndex::test_evenly_divisible_with_no_extra_bins2",
    "tests/resample/test_period_index.py::TestPeriodIndex::test_resample_basic",
    "tests/resample/test_period_index.py::TestPeriodIndex::test_resample_tz_localized[ms]",
    "tests/resample/test_period_index.py::TestPeriodIndex::test_resample_tz_localized[ns]",
    "tests/resample/test_period_index.py::TestPeriodIndex::test_resample_tz_localized[s]",
    "tests/resample/test_period_index.py::TestPeriodIndex::test_resample_tz_localized[us]",
    "tests/resample/test_period_index.py::TestPeriodIndex::test_with_local_timezone[tz0]",
    "tests/resample/test_period_index.py::TestPeriodIndex::test_with_local_timezone[tz1]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_col_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_col_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_col_resample-agg2]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_grouper_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_grouper_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_grouper_resample-agg2]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_mult_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_mult_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_mult_resample-agg2]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_std_named_result[df_resample-agg2]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_sum[df_col_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_sum[df_grouper_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_sum[df_mult_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_both_mean_sum[df_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_mixed_column_aggregation[df_col_resample]",
    "tests/resample/test_resample_api.py::test_agg_mixed_column_aggregation[df_grouper_resample]",
    "tests/resample/test_resample_api.py::test_agg_mixed_column_aggregation[df_resample]",
    "tests/resample/test_resample_api.py::test_agg_specificationerror_series[df_col_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_specificationerror_series[df_grouper_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_specificationerror_series[df_mult_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_specificationerror_series[df_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_col_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_col_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_col_resample-agg2]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_grouper_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_grouper_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_grouper_resample-agg2]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_mult_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_mult_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_mult_resample-agg2]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_resample-agg0]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_resample-agg1]",
    "tests/resample/test_resample_api.py::test_agg_with_lambda[df_resample-agg2]",
    "tests/resample/test_resample_api.py::test_groupby_resample_api",
    "tests/resample/test_resample_api.py::test_resample_group_keys",
    "tests/resample/test_resample_api.py::test_transform_frame[None]",
    "tests/resample/test_resample_api.py::test_transform_frame[date]",
    "tests/resample/test_resampler_grouper.py::test_deferred_with_groupby",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[count]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[first]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[last]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[max]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[mean]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[median]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[min]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[nunique]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[ohlc]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[prod]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[quantile]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[sem]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[size]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[std]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[sum]",
    "tests/resample/test_time_grouper.py::test_aggregate_normal[var]",
    "tests/resample/test_time_grouper.py::test_apply_iteration",
    "tests/resample/test_time_grouper.py::test_repr",
    "tests/reshape/concat/test_append.py::TestAppend::test_append_different_columns_types[CategoricalIndex-MultiIndex0]",
    "tests/reshape/concat/test_append.py::TestAppend::test_append_different_columns_types[CategoricalIndex-MultiIndex1]",
    "tests/reshape/concat/test_append.py::TestAppend::test_append_different_columns_types[Index-MultiIndex2]",
    "tests/reshape/concat/test_append.py::TestAppend::test_append_dtype_coerce[False]",
    "tests/reshape/concat/test_append.py::TestAppend::test_append_dtype_coerce[True]",
    "tests/reshape/concat/test_append_common.py::TestConcatAppendCommon::test_categorical_concat_append",
    "tests/reshape/concat/test_append_common.py::TestConcatAppendCommon::test_concat_categorical_coercion_nan",
    "tests/reshape/concat/test_append_common.py::TestConcatAppendCommon::test_concat_categorical_empty",
    "tests/reshape/concat/test_categorical.py::TestCategoricalConcat::test_categorical_index_preserver",
    "tests/reshape/concat/test_categorical.py::TestCategoricalConcat::test_categorical_index_upcast",
    "tests/reshape/concat/test_categorical.py::TestCategoricalConcat::test_concat_categorical_datetime",
    "tests/reshape/concat/test_concat.py::TestConcatenate::test_concat_copy",
    "tests/reshape/concat/test_concat.py::TestConcatenate::test_concat_keys_specific_levels",
    "tests/reshape/concat/test_concat.py::TestConcatenate::test_concat_order",
    "tests/reshape/concat/test_concat.py::test_concat_empty_and_non_empty_frame_regression",
    "tests/reshape/concat/test_concat.py::test_concat_ignore_empty_object_float[None-int64]",
    "tests/reshape/concat/test_concat.py::test_concat_ignore_empty_object_float[float64-int64]",
    "tests/reshape/concat/test_concat.py::test_concat_ignore_empty_object_float[object-int64]",
    "tests/reshape/concat/test_concat.py::test_concat_retain_attrs[data0]",
    "tests/reshape/concat/test_concat.py::test_concat_retain_attrs[data1]",
    "tests/reshape/concat/test_concat.py::test_concat_retain_attrs[data2]",
    "tests/reshape/concat/test_concat.py::test_concat_retain_attrs[data3]",
    "tests/reshape/concat/test_datetimes.py::TestTimezoneConcat::test_concat_multiindex_with_tz",
    "tests/reshape/concat/test_datetimes.py::test_concat_float_datetime64",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_dataframe_dtypes",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_df_object_dtype",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtype_category_with_array",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes[M8[ns]-bool-object_]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes[M8[ns]-int64-object_]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes[bool-float32-object_]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes[bool-int32-object_]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes[category-object-object]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes[m8[ns]-bool-object_]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes[m8[ns]-int64-object_]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[M8[ns]-float64]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[M8[ns]-int8]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[M8[ns]-m8[ns]]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[M8[ns]-uint8]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[float64-M8[ns]]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[float64-int8]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[float64-m8[ns]]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[float64-uint8]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[int8-M8[ns]]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[int8-m8[ns]]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[int8-uint8]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[m8[ns]-M8[ns]]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[m8[ns]-float64]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[m8[ns]-int8]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[m8[ns]-uint8]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[uint8-M8[ns]]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_roundtrips[uint8-m8[ns]]",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_dtypes_triple",
    "tests/reshape/concat/test_empty.py::TestEmptyConcat::test_concat_empty_series_timelike[values0-None]",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_concat_copy_index_frame[axis='columns']",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_concat_copy_index_frame[axis='index']",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_concat_copy_index_frame[axis=0]",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_concat_copy_index_frame[axis=1]",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_concat_copy_index_series[axis='columns']",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_concat_copy_index_series[axis=1]",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_concat_same_index_names[idx-None-None-None]",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_concat_same_index_names[idx-idx-None-None]",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_concat_same_index_names[idx1-idx1-idx2-None]",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_concat_same_index_names[idx1-idx2-None-None]",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_concat_same_index_names[idx1-idx2-idx3-None]",
    "tests/reshape/concat/test_index.py::TestIndexConcat::test_default_index",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_axis_1_sort_false_rangeindex",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_find_common[Int16]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_find_common[Int32]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_find_common[Int8]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_keep_dtype",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_keep_dtype_ea_numeric[Float32]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_keep_dtype_ea_numeric[Float64]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_keep_dtype_ea_numeric[Int16]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_keep_dtype_ea_numeric[Int32]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_keep_dtype_ea_numeric[Int64]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_keep_dtype_ea_numeric[Int8]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_keep_dtype_ea_numeric[UInt16]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_keep_dtype_ea_numeric[UInt32]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_keep_dtype_ea_numeric[UInt64]",
    "tests/reshape/concat/test_index.py::TestMultiIndexConcat::test_concat_index_keep_dtype_ea_numeric[UInt8]",
    "tests/reshape/concat/test_series.py::TestSeriesConcat::test_concat_series_axis1",
    "tests/reshape/concat/test_sort.py::TestConcatSort::test_concat_sort_none_raises",
    "tests/reshape/merge/test_join.py::TestJoin::test_join_index_mixed[right]",
    "tests/reshape/merge/test_join.py::TestJoin::test_join_many_non_unique_index",
    "tests/reshape/merge/test_join.py::TestJoin::test_join_multi_to_multi[right]",
    "tests/reshape/merge/test_join.py::TestJoin::test_join_multiindex",
    "tests/reshape/merge/test_join.py::test_join_empty[False-cross-empty]",
    "tests/reshape/merge/test_join.py::test_join_empty[False-right-empty]",
    "tests/reshape/merge/test_join.py::test_join_empty[True-cross-empty]",
    "tests/reshape/merge/test_join.py::test_join_empty[True-right-right]",
    "tests/reshape/merge/test_join.py::test_join_multiindex_categorical_output_index_dtype[right-values3]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_index_and_on_parameters_confusion",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_left_empty_right_empty[inner-kwarg1]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_left_empty_right_empty[inner-kwarg2]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_left_empty_right_empty[left-kwarg1]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_left_empty_right_empty[left-kwarg2]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_left_empty_right_empty[outer-kwarg1]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_left_empty_right_empty[outer-kwarg2]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_left_empty_right_empty[right-kwarg0]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_left_empty_right_empty[right-kwarg1]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_left_empty_right_empty[right-kwarg2]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_left_empty_right_empty[right-kwarg3]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_left_empty_right_notempty",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_nocopy",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_non_string_columns",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index0-expected_index0-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index0-expected_index0-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index1-expected_index1-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index1-expected_index1-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index10-expected_index10-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index10-expected_index10-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index11-expected_index11-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index11-expected_index11-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index12-expected_index12-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index12-expected_index12-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index13-expected_index13-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index13-expected_index13-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index14-expected_index14-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index14-expected_index14-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index16-expected_index16-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index16-expected_index16-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index2-expected_index2-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index2-expected_index2-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index3-expected_index3-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index3-expected_index3-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index4-expected_index4-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index4-expected_index4-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index5-expected_index5-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index5-expected_index5-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index6-expected_index6-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index6-expected_index6-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index7-expected_index7-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index7-expected_index7-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index8-expected_index8-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index8-expected_index8-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index9-expected_index9-outer]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_on_index_with_more_values[index9-expected_index9-right]",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_right_index_right",
    "tests/reshape/merge/test_merge.py::TestMerge::test_merge_take_missing_values_from_index_of_other_dtype",
    "tests/reshape/merge/test_merge.py::TestMergeCategorical::test_dtype_on_categorical_dates",
    "tests/reshape/merge/test_merge.py::TestMergeCategorical::test_dtype_on_merged_different[inner-<lambda>1]",
    "tests/reshape/merge/test_merge.py::TestMergeCategorical::test_dtype_on_merged_different[left-<lambda>0]",
    "tests/reshape/merge/test_merge.py::TestMergeCategorical::test_dtype_on_merged_different[left-<lambda>1]",
    "tests/reshape/merge/test_merge.py::TestMergeCategorical::test_dtype_on_merged_different[outer-<lambda>1]",
    "tests/reshape/merge/test_merge.py::TestMergeCategorical::test_dtype_on_merged_different[right-<lambda>1]",
    "tests/reshape/merge/test_merge.py::TestMergeCategorical::test_merge_categorical",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_join_multi_dtypes[float32-int320]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_join_multi_dtypes[float32-int321]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_merge_bool_dtype[right-expected_data3]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_merge_empty[False-right-empty]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_merge_empty[True-right-right]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_merge_incompat_dtypes_are_ok[df1_vals3-df2_vals3]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_merge_incompat_dtypes_error[df1_vals2-df2_vals2]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_merge_incompat_dtypes_error[df1_vals3-df2_vals3]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_merge_incompat_dtypes_error[df1_vals4-df2_vals4]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_merge_on_ints_floats[int_vals0-float_vals0-exp_vals0]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_merge_on_ints_floats[int_vals1-float_vals1-exp_vals1]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_merge_on_ints_floats[int_vals2-float_vals2-exp_vals2]",
    "tests/reshape/merge/test_merge.py::TestMergeDtypes::test_merge_on_ints_floats_warning",
    "tests/reshape/merge/test_merge.py::TestMergeOnIndexes::test_merge_on_indexes[right-False-expected4]",
    "tests/reshape/merge/test_merge.py::TestMergeOnIndexes::test_merge_on_indexes[right-True-expected5]",
    "tests/reshape/merge/test_merge.py::test_merge_datetime_different_resolution[None-right]",
    "tests/reshape/merge/test_merge.py::test_merge_different_index_names",
    "tests/reshape/merge/test_merge.py::test_merge_ea[Float32-right]",
    "tests/reshape/merge/test_merge.py::test_merge_ea[Float64-right]",
    "tests/reshape/merge/test_merge.py::test_merge_ea[Int16-right]",
    "tests/reshape/merge/test_merge.py::test_merge_ea[Int32-right]",
    "tests/reshape/merge/test_merge.py::test_merge_ea[Int64-right]",
    "tests/reshape/merge/test_merge.py::test_merge_ea[Int8-right]",
    "tests/reshape/merge/test_merge.py::test_merge_ea[UInt16-right]",
    "tests/reshape/merge/test_merge.py::test_merge_ea[UInt32-right]",
    "tests/reshape/merge/test_merge.py::test_merge_ea[UInt64-right]",
    "tests/reshape/merge/test_merge.py::test_merge_ea[UInt8-right]",
    "tests/reshape/merge/test_merge.py::test_merge_ea_int_and_float_numpy",
    "tests/reshape/merge/test_merge.py::test_merge_result_empty_index_and_on",
    "tests/reshape/merge/test_merge.py::test_merge_right_left_index",
    "tests/reshape/merge/test_merge.py::test_merge_series[on4-None-None-False-False-None]",
    "tests/reshape/merge/test_merge.py::test_merge_series_multilevel",
    "tests/reshape/merge/test_merge.py::test_merge_suffix[0-0-kwargs1-expected_cols1]",
    "tests/reshape/merge/test_merge.py::test_merge_suffix[0-0-kwargs10-expected_cols10]",
    "tests/reshape/merge/test_merge.py::test_merge_suffix[0.0-0.0-kwargs5-expected_cols5]",
    "tests/reshape/merge/test_merge.py::test_merge_suffix[a-a-kwargs7-expected_cols7]",
    "tests/reshape/merge/test_merge.py::test_merge_suffix[a-a-kwargs9-expected_cols9]",
    "tests/reshape/merge/test_merge.py::test_merge_suffix[b-b-kwargs6-expected_cols6]",
    "tests/reshape/merge/test_merge.py::test_merge_suffix_raises[suffixes0]",
    "tests/reshape/merge/test_merge.py::test_merge_suffix_raises[suffixes1]",
    "tests/reshape/merge/test_merge.py::test_merge_suffixes_produce_dup_columns_raises",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df0-right_df1-left_on0-right_on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df0-right_df1-left_on3-right_on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df0-right_df2-left_on0-right_on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df0-right_df2-left_on3-right_on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df1-right_df0-left_on0-right_on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df1-right_df0-left_on2-right_on2-left]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df1-right_df0-left_on3-right_on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df1-right_df1-left_on2-right_on2-left]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df1-right_df1-left_on3-right_on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df1-right_df2-left_on2-right_on2-left]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df1-right_df2-left_on3-right_on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df2-right_df0-left_on0-right_on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df2-right_df0-left_on2-right_on2-left]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df2-right_df0-left_on3-right_on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df2-right_df1-left_on0-right_on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df2-right_df1-left_on2-right_on2-left]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df2-right_df1-left_on3-right_on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df2-right_df2-left_on0-right_on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df2-right_df2-left_on1-right_on1-right]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_lefton_righton[left_df2-right_df2-left_on3-right_on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df0-right_df1-on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df0-right_df1-on2-right]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df0-right_df1-on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df0-right_df2-on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df0-right_df2-on2-right]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df0-right_df2-on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df1-right_df0-on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df1-right_df0-on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df1-right_df1-on2-right]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df1-right_df1-on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df1-right_df2-on2-right]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df1-right_df2-on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df2-right_df0-on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df2-right_df0-on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df2-right_df1-on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df2-right_df1-on2-right]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df2-right_df1-on3-outer]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df2-right_df2-on0-inner]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df2-right_df2-on1-left]",
    "tests/reshape/merge/test_merge_index_as_string.py::test_merge_indexes_and_columns_on[left_df2-right_df2-on3-outer]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_join_multi_empty_frames[inner]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_join_multi_empty_frames[left]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_join_multi_empty_frames[outer]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_join_multi_empty_frames[right]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_join_multi_multi[inner]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_join_multi_multi[left]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_join_multi_multi[outer]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_join_multi_multi[right]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_join_multi_wrong_order",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_merge_datetime_index[Index]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_merge_datetime_index[None]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_merge_datetime_index[Series]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_merge_datetime_index[asarray]",
    "tests/reshape/merge/test_multi.py::TestJoinMultiMulti::test_single_common_level",
    "tests/reshape/merge/test_multi.py::TestMergeMulti::test_left_join_multi_index[False-True]",
    "tests/reshape/merge/test_multi.py::TestMergeMulti::test_left_join_multi_index[True-True]",
    "tests/reshape/merge/test_multi.py::TestMergeMulti::test_merge_datetime_index[Index]",
    "tests/reshape/merge/test_multi.py::TestMergeMulti::test_merge_datetime_index[None]",
    "tests/reshape/merge/test_multi.py::TestMergeMulti::test_merge_datetime_index[Series]",
    "tests/reshape/merge/test_multi.py::TestMergeMulti::test_merge_datetime_index[asarray]",
    "tests/reshape/merge/test_multi.py::TestMergeMulti::test_merge_on_multikey[right]",
    "tests/reshape/test_crosstab.py::TestCrosstab::test_crosstab_duplicate_names",
    "tests/reshape/test_crosstab.py::TestCrosstab::test_crosstab_multiple",
    "tests/reshape/test_crosstab.py::TestCrosstab::test_crosstab_no_overlap",
    "tests/reshape/test_crosstab.py::TestCrosstab::test_crosstab_with_categorial_columns",
    "tests/reshape/test_crosstab.py::TestCrosstab::test_crosstab_with_empties",
    "tests/reshape/test_cut.py::test_bins[array]",
    "tests/reshape/test_cut.py::test_bins[list]",
    "tests/reshape/test_cut.py::test_bins_from_interval_index",
    "tests/reshape/test_cut.py::test_bins_not_monotonic",
    "tests/reshape/test_cut.py::test_cut_bool_coercion_to_int[array-assert_categorical_equal-6]",
    "tests/reshape/test_cut.py::test_cut_bool_coercion_to_int[array-assert_categorical_equal-7]",
    "tests/reshape/test_cut.py::test_cut_non_unique_labels[data0-bins0-labels0-expected_codes0-expected_labels0]",
    "tests/reshape/test_cut.py::test_cut_non_unique_labels[data1-bins1-labels1-expected_codes1-expected_labels1]",
    "tests/reshape/test_cut.py::test_cut_nullable_integer[False-False-bins1]",
    "tests/reshape/test_cut.py::test_cut_nullable_integer[False-True-bins1]",
    "tests/reshape/test_cut.py::test_cut_nullable_integer[True-False-bins1]",
    "tests/reshape/test_cut.py::test_cut_nullable_integer[True-True-bins1]",
    "tests/reshape/test_cut.py::test_cut_pass_labels[<lambda>-<lambda>0]",
    "tests/reshape/test_cut.py::test_cut_pass_labels_compat",
    "tests/reshape/test_cut.py::test_cut_pass_series_name_to_factor",
    "tests/reshape/test_cut.py::test_cut_unordered_labels[data0-bins0-labels0-expected_codes0-expected_labels0]",
    "tests/reshape/test_cut.py::test_cut_unordered_labels[data1-bins1-labels1-expected_codes1-expected_labels1]",
    "tests/reshape/test_cut.py::test_datetime_nan_error",
    "tests/reshape/test_cut.py::test_datetime_nan_mask",
    "tests/reshape/test_cut.py::test_label_precision",
    "tests/reshape/test_cut.py::test_no_right",
    "tests/reshape/test_cut.py::test_right",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_dataframe_dummies_prefix_str[dense]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_dataframe_dummies_string_dtype",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_dataframe_dummies_string_dtype[string=str[pyarrow]]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_dataframe_dummies_string_dtype[string=str[python]]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_dataframe_dummies_string_dtype[string=string[pyarrow]]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_dataframe_dummies_string_dtype[string=string[python]]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_get_dummies_arrow_dtype",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_get_dummies_basic_drop_first_NA[dense]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_get_dummies_basic_drop_first_one_level[dense]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_get_dummies_ea_dtype",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_get_dummies_ea_dtype[string=str[pyarrow]-string]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_get_dummies_ea_dtype[string=str[python]-string]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_get_dummies_ea_dtype[string=string[pyarrow]-category]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_get_dummies_ea_dtype[string=string[pyarrow]-string]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_get_dummies_ea_dtype[string=string[python]-category]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_get_dummies_ea_dtype[string=string[python]-string]",
    "tests/reshape/test_get_dummies.py::TestGetDummies::test_get_dummies_int_int",
    "tests/reshape/test_melt.py::TestMelt::test_melt_ea_columns",
    "tests/reshape/test_melt.py::TestMelt::test_multiindex",
    "tests/reshape/test_melt.py::TestMelt::test_vars_work_with_multiindex",
    "tests/reshape/test_melt.py::TestWideToLong::test_invalid_separator",
    "tests/reshape/test_melt.py::TestWideToLong::test_invalid_suffixtype",
    "tests/reshape/test_melt.py::TestWideToLong::test_raise_of_column_name_value",
    "tests/reshape/test_pivot.py::TestPivotTable::test_categorical_aggfunc[None]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_categorical_pivot_index_ordering[False]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_daily",
    "tests/reshape/test_pivot.py::TestPivotTable::test_monthly",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_complex_aggfunc",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_datetime_tz",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_dtaccessor",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_multi_functions",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_no_level_overlap",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_string_as_func",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_string_func_vs_func[f3-f_numpy3]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_string_func_vs_func[f4-f_numpy4]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_string_func_vs_func[f5-f_numpy5]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_table_aggfunc_nunique_with_different_values",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_table_categorical_observed_equal[False]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_table_multiindex_columns_doctest_case",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_table_nocols",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_table_not_series",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_table_with_iterator_values",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_table_with_mixed_nested_tuples",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_timegrouper",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_timegrouper_double",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_categorical[False-False]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_categorical[False-None]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_categorical[False-True]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_list_like_values[False-values0]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_list_like_values[True-values0]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_list_like_values_nans[False-values0]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_list_like_values_nans[True-values0]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_tz[ms-False]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_tz[ms-True]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_tz[ns-False]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_tz[ns-True]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_tz[s-False]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_tz[s-True]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_tz[us-False]",
    "tests/reshape/test_pivot.py::TestPivotTable::test_pivot_with_tz[us-True]",
    "tests/reshape/test_pivot_multilevel.py::test_pivot_list_like_columns[input_index1-input_columns1-values-expected_values1-expected_columns1-expected_index1]",
    "tests/reshape/test_pivot_multilevel.py::test_pivot_list_like_columns[input_index3-input_columns3-values-expected_values3-expected_columns3-expected_index3]",
    "tests/reshape/test_pivot_multilevel.py::test_pivot_list_like_columns[lev4-input_columns0-values-expected_values0-expected_columns0-expected_index0]",
    "tests/reshape/test_qcut.py::test_date_like_qcut_bins[arg1-expected_bins1]",
    "tests/scalar/interval/test_arithmetic.py::TestIntervalArithmetic::test_timedelta_add_timestamp_interval[timedelta64]",
    "tests/scalar/period/test_arithmetic.py::TestPeriodArithmetic::test_period_add_sub_td64_nat[m]",
    "tests/scalar/period/test_arithmetic.py::TestPeriodArithmetic::test_period_add_sub_td64_nat[ms]",
    "tests/scalar/period/test_arithmetic.py::TestPeriodArithmetic::test_period_add_sub_td64_nat[ns]",
    "tests/scalar/period/test_arithmetic.py::TestPeriodArithmetic::test_period_add_sub_td64_nat[s]",
    "tests/scalar/period/test_arithmetic.py::TestPeriodArithmetic::test_period_add_sub_td64_nat[us]",
    "tests/scalar/period/test_arithmetic.py::TestPeriodComparisons::test_period_comparison_numpy_zerodim_arr[zerodim_arr0-False]",
    "tests/scalar/period/test_arithmetic.py::TestPeriodComparisons::test_period_comparison_numpy_zerodim_arr[zerodim_arr1-True]",
    "tests/scalar/period/test_period.py::TestPeriodConstruction::test_construct_from_nat_string_and_freq[D]",
    "tests/scalar/period/test_period.py::TestPeriodConstruction::test_construct_from_nat_string_and_freq[M]",
    "tests/scalar/period/test_period.py::TestPeriodConstruction::test_construct_from_nat_string_and_freq[Y]",
    "tests/scalar/period/test_period.py::TestPeriodConstruction::test_construct_from_nat_string_and_freq[h]",
    "tests/scalar/period/test_period.py::TestPeriodConstruction::test_constructor_corner",
    "tests/scalar/period/test_period.py::TestPeriodConstruction::test_period_cons_nat",
    "tests/scalar/period/test_period.py::TestPeriodMethods::test_repr_nat",
    "tests/scalar/test_nat.py::test_identity[-9223372036854775808-Period]",
    "tests/scalar/test_nat.py::test_identity[-Period]",
    "tests/scalar/test_nat.py::test_identity[NAT-Period]",
    "tests/scalar/test_nat.py::test_identity[NaT-Period]",
    "tests/scalar/test_nat.py::test_identity[None-Period]",
    "tests/scalar/test_nat.py::test_identity[nan0-Period]",
    "tests/scalar/test_nat.py::test_identity[nan1-Period]",
    "tests/scalar/test_nat.py::test_identity[nat-Period]",
    "tests/scalar/test_nat.py::test_identity[value4-Period]",
    "tests/scalar/timedelta/methods/test_as_unit.py::TestAsUnit::test_as_unit",
    "tests/scalar/timedelta/test_arithmetic.py::TestTimedeltaAdditionSubtraction::test_td_add_sub_dt64_ndarray",
    "tests/scalar/timedelta/test_arithmetic.py::TestTimedeltaAdditionSubtraction::test_td_add_sub_ndarray_0d",
    "tests/scalar/timedelta/test_arithmetic.py::TestTimedeltaAdditionSubtraction::test_td_add_sub_td64_ndarray",
    "tests/scalar/timedelta/test_arithmetic.py::TestTimedeltaAdditionSubtraction::test_td_sub_timedelta64",
    "tests/scalar/timedelta/test_arithmetic.py::TestTimedeltaComparison::test_compare_pytimedelta_bounds",
    "tests/scalar/timedelta/test_arithmetic.py::TestTimedeltaComparison::test_compare_td64_ndarray",
    "tests/scalar/timedelta/test_arithmetic.py::TestTimedeltaMultiplicationDivision::test_td_div_td64_ndarray",
    "tests/scalar/timedelta/test_arithmetic.py::TestTimedeltaMultiplicationDivision::test_td_mul_numeric_ndarray",
    "tests/scalar/timedelta/test_arithmetic.py::TestTimedeltaMultiplicationDivision::test_td_rdiv_na_scalar",
    "tests/scalar/timedelta/test_arithmetic.py::TestTimedeltaMultiplicationDivision::test_td_rdiv_ndarray",
    "tests/scalar/timedelta/test_arithmetic.py::TestTimedeltaMultiplicationDivision::test_td_rdiv_ndarray_0d",
    "tests/scalar/timedelta/test_constructors.py::test_construction_out_of_bounds_td64s[106752000000000-D]",
    "tests/scalar/timedelta/test_constructors.py::test_construction_out_of_bounds_td64s[15251000000000-W]",
    "tests/scalar/timedelta/test_constructors.py::test_construction_out_of_bounds_td64s[153722868000000000-m]",
    "tests/scalar/timedelta/test_constructors.py::test_construction_out_of_bounds_td64s[2562048000000000-h]",
    "tests/scalar/timedelta/test_constructors.py::test_timedelta_constructor_identity",
    "tests/scalar/timedelta/test_timedelta.py::TestNonNano::test_hash",
    "tests/scalar/timedelta/test_timedelta.py::TestNonNano::test_max[ms]",
    "tests/scalar/timedelta/test_timedelta.py::TestNonNano::test_max[s]",
    "tests/scalar/timedelta/test_timedelta.py::TestNonNano::test_min[ms]",
    "tests/scalar/timedelta/test_timedelta.py::TestNonNano::test_min[s]",
    "tests/scalar/timestamp/methods/test_as_unit.py::TestTimestampAsUnit::test_as_unit",
    "tests/scalar/timestamp/methods/test_replace.py::TestTimestampReplace::test_replace_out_of_pydatetime_bounds",
    "tests/scalar/timestamp/test_arithmetic.py::TestTimestampArithmetic::test_addsub_m8ndarray[shape0]",
    "tests/scalar/timestamp/test_arithmetic.py::TestTimestampArithmetic::test_addsub_m8ndarray[shape1]",
    "tests/scalar/timestamp/test_arithmetic.py::TestTimestampArithmetic::test_addsub_m8ndarray_tzaware[shape0]",
    "tests/scalar/timestamp/test_arithmetic.py::TestTimestampArithmetic::test_addsub_m8ndarray_tzaware[shape1]",
    "tests/scalar/timestamp/test_arithmetic.py::TestTimestampArithmetic::test_rsub_dtscalars[None]",
    "tests/scalar/timestamp/test_comparisons.py::TestTimestampComparison::test_compare_non_nano_dt64",
    "tests/scalar/timestamp/test_comparisons.py::TestTimestampComparison::test_compare_zerodim_array",
    "tests/scalar/timestamp/test_comparisons.py::TestTimestampComparison::test_comparison_dt64_ndarray",
    "tests/scalar/timestamp/test_constructors.py::test_timestamp_constructor_identity",
    "tests/scalar/timestamp/test_timestamp.py::TestNonNano::test_addsub_timedeltalike_non_nano[ms-td2]",
    "tests/scalar/timestamp/test_timestamp.py::TestNonNano::test_addsub_timedeltalike_non_nano[s-td2]",
    "tests/scalar/timestamp/test_timestamp.py::TestNonNano::test_max[ms]",
    "tests/scalar/timestamp/test_timestamp.py::TestNonNano::test_max[s]",
    "tests/scalar/timestamp/test_timestamp.py::TestNonNano::test_max[us]",
    "tests/scalar/timestamp/test_timestamp.py::TestNonNano::test_min[ms]",
    "tests/scalar/timestamp/test_timestamp.py::TestNonNano::test_min[s]",
    "tests/scalar/timestamp/test_timestamp.py::TestNonNano::test_min[us]",
    "tests/scalar/timestamp/test_timestamp.py::TestTimestampProperties::test_dow_historic[-0040-1-1-4]",
    "tests/scalar/timestamp/test_timestamp.py::TestTimestampProperties::test_dow_historic[0000-2-29-1]",
    "tests/scalar/timestamp/test_timestamp.py::TestTimestampProperties::test_dow_historic[0000-3-1-2]",
    "tests/scalar/timestamp/test_timestamp.py::TestTimestampProperties::test_dow_parametric",
    "tests/scalar/timestamp/test_timestamp.py::test_negative_dates",
    "tests/series/accessors/test_cat_accessor.py::TestCatAccessor::test_categorical_delegations",
    "tests/series/accessors/test_cat_accessor.py::TestCatAccessor::test_dt_accessor_api_for_categorical[idx0]",
    "tests/series/accessors/test_cat_accessor.py::TestCatAccessor::test_dt_accessor_api_for_categorical[idx3]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_date_tz",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_dt_accessor_datetime_name_accessors[None]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_dt_accessor_limited_display_api",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_dt_accessor_no_new_attributes",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_dt_namespace_accessor_datetime64[D]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_dt_namespace_accessor_datetime64[ms]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_dt_namespace_accessor_datetime64[s]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_dt_namespace_accessor_index_and_values",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_dt_namespace_accessor_timedelta",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_dt_other_accessors_categorical[day]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_dt_other_accessors_categorical[month]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_dt_other_accessors_categorical[year]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_isocalendar[input_series0-expected_output0]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_isocalendar[input_series1-expected_output1]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_isocalendar[input_series2-expected_output2]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_isocalendar[input_series4-expected_output4]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_isocalendar[input_series5-expected_output5]",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_strftime",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_strftime_dt64_days",
    "tests/series/accessors/test_dt_accessor.py::TestSeriesDatetimeValues::test_valid_dt_with_missing_values",
    "tests/series/accessors/test_list_accessor.py::test_list_getitem_invalid_index[list_dtype0]",
    "tests/series/accessors/test_list_accessor.py::test_list_len",
    "tests/series/accessors/test_struct_accessor.py::test_struct_accessor_dtypes",
    "tests/series/accessors/test_struct_accessor.py::test_struct_accessor_field_expanded[0-int_col]",
    "tests/series/accessors/test_struct_accessor.py::test_struct_accessor_field_expanded[indices1-str_col]",
    "tests/series/accessors/test_struct_accessor.py::test_struct_accessor_field_expanded[indices2-int_col]",
    "tests/series/accessors/test_struct_accessor.py::test_struct_accessor_field_expanded[indices5-string_col]",
    "tests/series/accessors/test_struct_accessor.py::test_struct_accessor_field_expanded[int_col-int_col]",
    "tests/series/accessors/test_struct_accessor.py::test_struct_accessor_field_expanded[string_col-string_col]",
    "tests/series/accessors/test_struct_accessor.py::test_struct_accessor_field_with_invalid_name_or_index",
    "tests/series/indexing/test_datetime.py::test_indexing_over_size_cutoff_period_index",
    "tests/series/indexing/test_datetime.py::test_indexing_unordered",
    "tests/series/indexing/test_getitem.py::TestGetitemBooleanMask::test_getitem_boolean_dt64_copies",
    "tests/series/indexing/test_getitem.py::TestGetitemDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key0]",
    "tests/series/indexing/test_getitem.py::TestGetitemDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key1]",
    "tests/series/indexing/test_getitem.py::TestGetitemDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key0]",
    "tests/series/indexing/test_getitem.py::TestGetitemDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key1]",
    "tests/series/indexing/test_getitem.py::TestSeriesGetitemScalars::test_getitem_float_keys_tuple_values",
    "tests/series/indexing/test_getitem.py::TestSeriesGetitemSlices::test_getitem_partial_str_slice_with_datetimeindex",
    "tests/series/indexing/test_getitem.py::test_getitem_1tuple_slice_without_multiindex",
    "tests/series/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key0]",
    "tests/series/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key1]",
    "tests/series/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated_multiindex[key2]",
    "tests/series/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated_multiindex[key3]",
    "tests/series/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key0]",
    "tests/series/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key1]",
    "tests/series/indexing/test_indexing.py::test_setitem_empty_indexer[indexer0-val0]",
    "tests/series/indexing/test_indexing.py::test_setitem_empty_indexer[indexer0-val1]",
    "tests/series/indexing/test_indexing.py::test_setitem_empty_indexer[indexer1-val0]",
    "tests/series/indexing/test_indexing.py::test_setitem_empty_indexer[indexer1-val1]",
    "tests/series/indexing/test_indexing.py::test_setitem_empty_indexer[indexer2-val0]",
    "tests/series/indexing/test_indexing.py::test_setitem_empty_indexer[indexer2-val1]",
    "tests/series/indexing/test_indexing.py::test_setitem_empty_indexer[indexer3-val0]",
    "tests/series/indexing/test_indexing.py::test_setitem_empty_indexer[indexer3-val1]",
    "tests/series/indexing/test_indexing.py::test_slice_with_negative_step[index1]",
    "tests/series/indexing/test_setitem.py::TestCoercionBool::test_int_key[iloc-True-bool-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionBool::test_int_key[loc-True-bool-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionBool::test_int_key[setitem-True-bool-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionBool::test_series_where[True-bool-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_int_key[iloc-(1+1j)-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_int_key[iloc-1-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_int_key[iloc-1.1-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_int_key[loc-(1+1j)-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_int_key[loc-1-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_int_key[loc-1.1-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_int_key[setitem-(1+1j)-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_int_key[setitem-1-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_int_key[setitem-1.1-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_series_where[(1+1j)-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_series_where[1-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionComplex::test_series_where[1.1-complex128-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64::test_int_key[iloc-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64::test_int_key[iloc-val0-datetime64[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64::test_int_key[loc-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64::test_int_key[loc-val0-datetime64[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64::test_int_key[setitem-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64::test_int_key[setitem-val0-datetime64[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64::test_mask_key[iloc-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64::test_mask_key[loc-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64::test_mask_key[setitem-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64::test_series_where[val0-datetime64[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64TZ::test_int_key[iloc-val0-datetime64[ns, US/Eastern]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64TZ::test_int_key[iloc-val1-datetime64[ns, US/Eastern]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64TZ::test_int_key[loc-val0-datetime64[ns, US/Eastern]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64TZ::test_int_key[loc-val1-datetime64[ns, US/Eastern]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64TZ::test_int_key[setitem-val0-datetime64[ns, US/Eastern]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64TZ::test_int_key[setitem-val1-datetime64[ns, US/Eastern]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64TZ::test_series_where[val0-datetime64[ns, US/Eastern]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionDatetime64TZ::test_series_where[val1-datetime64[ns, US/Eastern]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[iloc-1-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[iloc-1.1-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[iloc-val4-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[iloc-val5-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[iloc-val7-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[iloc-val8-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[loc-1-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[loc-1.1-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[loc-val4-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[loc-val5-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[loc-val7-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[loc-val8-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[setitem-1-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[setitem-1.1-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[setitem-val4-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[setitem-val5-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[setitem-val7-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_int_key[setitem-val8-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_series_where[1-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_series_where[1.1-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_series_where[val4-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_series_where[val5-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_series_where[val7-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat32::test_series_where[val8-float32-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat64::test_int_key[iloc-1-float64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat64::test_int_key[iloc-1.1-float64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat64::test_int_key[loc-1-float64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat64::test_int_key[loc-1.1-float64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat64::test_int_key[setitem-1-float64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat64::test_int_key[setitem-1.1-float64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat64::test_series_where[1-float64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionFloat64::test_series_where[1.1-float64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionInt64::test_int_key[iloc-1-int64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionInt64::test_int_key[loc-1-int64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionInt64::test_int_key[setitem-1-int64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionInt64::test_series_where[1-int64-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionInt8::test_int_key[iloc-val0-int8-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionInt8::test_int_key[loc-val0-int8-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionInt8::test_int_key[setitem-val0-int8-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionInt8::test_series_where[val0-int8-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionInt8::test_series_where[val1-int16-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[iloc-object-(1+1j)]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[iloc-object-1.1]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[iloc-object-1]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[iloc-object-True]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[loc-object-(1+1j)]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[loc-object-1.1]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[loc-object-1]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[loc-object-True]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[setitem-object-(1+1j)]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[setitem-object-1.1]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[setitem-object-1]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_int_key[setitem-object-True]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_series_where[object-(1+1j)]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_series_where[object-1.1]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_series_where[object-1]",
    "tests/series/indexing/test_setitem.py::TestCoercionObject::test_series_where[object-True]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_index_where[1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_int_key[iloc-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_int_key[iloc-val0-timedelta64[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_int_key[loc-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_int_key[loc-val0-timedelta64[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_int_key[setitem-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_int_key[setitem-val0-timedelta64[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_mask_key[iloc-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_mask_key[loc-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_mask_key[setitem-1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_series_where[1-object-FutureWarning]",
    "tests/series/indexing/test_setitem.py::TestCoercionTimedelta64::test_series_where[val0-timedelta64[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_index_where[obj2-expected2-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_int_key[iloc-obj1-expected1-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_int_key[iloc-obj2-expected2-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_int_key[iloc-obj3-expected3-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_int_key[loc-obj1-expected1-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_int_key[loc-obj2-expected2-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_int_key[loc-obj3-expected3-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_int_key[setitem-obj1-expected1-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_int_key[setitem-obj2-expected2-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_int_key[setitem-obj3-expected3-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_mask_key[iloc-obj2-expected2-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_mask_key[loc-obj2-expected2-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_mask_key[setitem-obj2-expected2-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_series_where[obj1-expected1-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_series_where[obj2-expected2-None]",
    "tests/series/indexing/test_setitem.py::TestSeriesNoneCoercion::test_series_where[obj3-expected3-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_boolean_nullable_int_types[Int16]",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_boolean_nullable_int_types[Int32]",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_boolean_nullable_int_types[Int64]",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_boolean_nullable_int_types[Int8]",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_boolean_nullable_int_types[UInt16]",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_boolean_nullable_int_types[UInt32]",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_boolean_nullable_int_types[UInt64]",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_boolean_nullable_int_types[UInt8]",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_mask_align_and_promote",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_mask_promote_strs",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_mask_smallint_no_upcast",
    "tests/series/indexing/test_setitem.py::TestSetitemBooleanMask::test_setitem_mask_smallint_upcast",
    "tests/series/indexing/test_setitem.py::TestSetitemCasting::test_setitem_boolean_array_into_npbool",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_index_where[M8[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_index_where[m8[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[M8[ns]-None-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[M8[ns]-None-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[M8[ns]-None-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[M8[ns]-array-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[M8[ns]-array-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[M8[ns]-array-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[M8[ns]-list-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[M8[ns]-list-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[M8[ns]-list-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[m8[ns]-None-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[m8[ns]-None-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[m8[ns]-None-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[m8[ns]-array-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[m8[ns]-array-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[m8[ns]-array-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[m8[ns]-list-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[m8[ns]-list-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_mask_key[m8[ns]-list-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_series_where[M8[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_series_where[m8[ns]-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[M8[ns]-None-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[M8[ns]-None-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[M8[ns]-None-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[M8[ns]-array-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[M8[ns]-array-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[M8[ns]-array-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[M8[ns]-list-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[M8[ns]-list-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[M8[ns]-list-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[m8[ns]-None-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[m8[ns]-None-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[m8[ns]-None-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[m8[ns]-array-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[m8[ns]-array-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[m8[ns]-array-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[m8[ns]-list-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[m8[ns]-list-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemDT64IntoInt::test_slice_key[m8[ns]-list-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemFloatIntervalWithIntIntervalValues::test_setitem_example",
    "tests/series/indexing/test_setitem.py::TestSetitemFloatNDarrayIntoIntegerSeries::test_slice_key[iloc-val0-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemFloatNDarrayIntoIntegerSeries::test_slice_key[loc-val0-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemFloatNDarrayIntoIntegerSeries::test_slice_key[setitem-val0-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemIntoIntegerSeriesNeedsUpcast::test_int_key[iloc-512]",
    "tests/series/indexing/test_setitem.py::TestSetitemIntoIntegerSeriesNeedsUpcast::test_int_key[loc-512]",
    "tests/series/indexing/test_setitem.py::TestSetitemIntoIntegerSeriesNeedsUpcast::test_int_key[setitem-512]",
    "tests/series/indexing/test_setitem.py::TestSetitemIntoIntegerSeriesNeedsUpcast::test_mask_key[iloc-512]",
    "tests/series/indexing/test_setitem.py::TestSetitemIntoIntegerSeriesNeedsUpcast::test_mask_key[loc-512]",
    "tests/series/indexing/test_setitem.py::TestSetitemIntoIntegerSeriesNeedsUpcast::test_mask_key[setitem-512]",
    "tests/series/indexing/test_setitem.py::TestSetitemIntoIntegerSeriesNeedsUpcast::test_series_where[512]",
    "tests/series/indexing/test_setitem.py::TestSetitemIntoIntegerSeriesNeedsUpcast::test_series_where[val1]",
    "tests/series/indexing/test_setitem.py::TestSetitemMismatchedTZCastsToObject::test_int_key[iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemMismatchedTZCastsToObject::test_int_key[loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemMismatchedTZCastsToObject::test_int_key[setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemMismatchedTZCastsToObject::test_series_where",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_index_putmask[3-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_index_putmask[3-nan]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_index_putmask[key1-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_index_putmask[key1-nan]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_index_where[3-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_index_where[3-nan]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_index_where[key1-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_index_where[key1-nan]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[3-None-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[3-None-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[3-None-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[3-nan-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[3-nan-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[3-nan-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[key1-None-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[key1-None-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[key1-None-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[key1-nan-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[key1-nan-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_int_key[key1-nan-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[3-None-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[3-None-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[3-None-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[3-nan-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[3-nan-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[3-nan-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[key1-None-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[key1-None-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[key1-None-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[key1-nan-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[key1-nan-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_mask_key[key1-nan-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_series_where[3-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_series_where[3-nan]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_series_where[key1-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_series_where[key1-nan]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[3-None-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[3-None-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[3-None-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[3-nan-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[3-nan-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[3-nan-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[key1-None-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[key1-None-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[key1-None-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[key1-nan-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[key1-nan-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemNAPeriodDtype::test_slice_key[key1-nan-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int16-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int16-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int16-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int32-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int32-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int32-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int64-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int64-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int64-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int8-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int8-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[int8-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint16-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint16-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint16-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint32-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint32-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint32-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint64-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint64-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint64-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint8-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint8-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemRangeIntoIntegerSeries::test_slice_key[uint8-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemTimedelta64IntoNumeric::test_index_where[int]",
    "tests/series/indexing/test_setitem.py::TestSetitemTimedelta64IntoNumeric::test_int_key[int-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemTimedelta64IntoNumeric::test_int_key[int-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemTimedelta64IntoNumeric::test_int_key[int-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemTimedelta64IntoNumeric::test_mask_key[int-iloc]",
    "tests/series/indexing/test_setitem.py::TestSetitemTimedelta64IntoNumeric::test_mask_key[int-loc]",
    "tests/series/indexing/test_setitem.py::TestSetitemTimedelta64IntoNumeric::test_mask_key[int-setitem]",
    "tests/series/indexing/test_setitem.py::TestSetitemTimedelta64IntoNumeric::test_series_where[int]",
    "tests/series/indexing/test_setitem.py::TestSetitemViewCopySemantics::test_dt64tz_setitem_does_not_mutate_dti",
    "tests/series/indexing/test_setitem.py::TestSetitemViewCopySemantics::test_setitem_invalidates_datetime_index_freq",
    "tests/series/indexing/test_setitem.py::TestSetitemWithExpansion::test_setitem_enlarge_with_na[na5-target_na5-int64-object-2-None]",
    "tests/series/indexing/test_setitem.py::TestSetitemWithExpansion::test_setitem_enlargement_object_none[NAType]",
    "tests/series/indexing/test_setitem.py::TestSmallIntegerSetitemUpcast::test_int_key[iloc-8589934593.0]",
    "tests/series/indexing/test_setitem.py::TestSmallIntegerSetitemUpcast::test_int_key[loc-8589934593.0]",
    "tests/series/indexing/test_setitem.py::TestSmallIntegerSetitemUpcast::test_int_key[setitem-8589934593.0]",
    "tests/series/indexing/test_setitem.py::TestSmallIntegerSetitemUpcast::test_mask_key[iloc-8589934593.0]",
    "tests/series/indexing/test_setitem.py::TestSmallIntegerSetitemUpcast::test_mask_key[loc-8589934593.0]",
    "tests/series/indexing/test_setitem.py::TestSmallIntegerSetitemUpcast::test_mask_key[setitem-8589934593.0]",
    "tests/series/indexing/test_setitem.py::test_setitem_scalar_into_readonly_backing_data",
    "tests/series/indexing/test_setitem.py::test_setitem_slice_into_readonly_backing_data",
    "tests/series/indexing/test_where.py::test_where_datetimelike_coerce[timedelta64[ns]]",
    "tests/series/indexing/test_xs.py::TestXSWithMultiIndex::test_series_getitem_multiindex_xs",
    "tests/series/indexing/test_xs.py::TestXSWithMultiIndex::test_xs_level_series",
    "tests/series/methods/test_argsort.py::TestSeriesArgsort::test_argsort_axis",
    "tests/series/methods/test_argsort.py::TestSeriesArgsort::test_argsort_dt64[ms]",
    "tests/series/methods/test_argsort.py::TestSeriesArgsort::test_argsort_dt64[ns]",
    "tests/series/methods/test_argsort.py::TestSeriesArgsort::test_argsort_dt64[s]",
    "tests/series/methods/test_argsort.py::TestSeriesArgsort::test_argsort_dt64[us]",
    "tests/series/methods/test_argsort.py::TestSeriesArgsort::test_argsort_stable",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_cast_nan_inf_int[int32-inf]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_cast_nan_inf_int[int64-inf]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float-uint16]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float-uint32]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float-uint64]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float-uint8]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float32-uint16]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float32-uint32]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float32-uint64]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float32-uint8]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float64-uint16]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float64-uint32]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float64-uint64]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_float_to_uint_negatives_raise[float64-uint8]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_from_float_to_str[float32]",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_str_cast_td64",
    "tests/series/methods/test_astype.py::TestAstype::test_astype_to_str_preserves_na[None-None]",
    "tests/series/methods/test_astype.py::TestAstypeCategorical::test_astype_categorical_to_other",
    "tests/series/methods/test_astype.py::TestAstypeString::test_astype_string_to_extension_dtype_roundtrip[string[pyarrow]-data1-category]",
    "tests/series/methods/test_astype.py::TestAstypeString::test_astype_string_to_extension_dtype_roundtrip[string[python]-data1-category]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Float32-Decimal]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Float32-NAType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Float32-NoneType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Float32-float0]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Float32-float1]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Float64-Decimal]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Float64-NAType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Float64-NoneType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Float64-float0]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Float64-float1]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int16-Decimal]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int16-NAType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int16-NoneType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int16-float0]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int16-float1]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int32-Decimal]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int32-NAType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int32-NoneType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int32-float0]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int32-float1]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int64-Decimal]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int64-NAType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int64-NoneType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int64-float0]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int64-float1]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int8-Decimal]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int8-NAType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int8-NoneType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int8-float0]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[Int8-float1]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt16-Decimal]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt16-NAType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt16-NoneType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt16-float0]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt16-float1]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt32-Decimal]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt32-NAType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt32-NoneType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt32-float0]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt32-float1]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt64-Decimal]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt64-NAType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt64-NoneType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt64-float0]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt64-float1]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt8-Decimal]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt8-NAType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt8-NoneType]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt8-float0]",
    "tests/series/methods/test_clip.py::TestSeriesClip::test_series_clipping_with_na_values[UInt8-float1]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases0-params11]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases0-params17]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases0-params19]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases0-params1]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases0-params25]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases0-params27]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases0-params3]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases0-params9]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases1-params11]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases1-params17]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases1-params19]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases1-params1]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases1-params25]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases1-params27]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases1-params3]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases1-params9]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases10-params11]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases10-params17]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases10-params19]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases10-params1]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases10-params25]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases10-params27]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases10-params3]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases10-params9]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases11-params11]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases11-params17]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases11-params19]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases11-params1]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases11-params25]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases11-params27]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases11-params3]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases11-params9]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases12-params12]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases12-params14]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases12-params20]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases12-params22]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases12-params28]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases12-params30]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases12-params4]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases12-params6]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params17]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params19]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params1]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params20]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params21]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params22]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params23]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params3]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params4]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params5]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params6]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases2-params7]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params11]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params12]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params14]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params17]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params19]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params1]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params20]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params22]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params25]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params27]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params28]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params30]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params3]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params4]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params6]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases6-params9]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases7-params12]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases7-params14]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases7-params20]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases7-params22]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases7-params28]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases7-params30]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases7-params4]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes[test_cases7-params6]",
    "tests/series/methods/test_convert_dtypes.py::TestSeriesConvertDtypes::test_convert_dtypes_pyarrow_to_np_nullable",
    "tests/series/methods/test_diff.py::TestSeriesDiff::test_diff_bool[input0-output0-1]",
    "tests/series/methods/test_drop.py::test_drop_exception_raised[data1-index1-drop_labels1-0-KeyError-not found in axis]",
    "tests/series/methods/test_drop.py::test_drop_unique_and_non_unique_index[data1-index1-drop_labels1-rows-expected_data1-expected_index1]",
    "tests/series/methods/test_equals.py::test_equals[arr1-idx1]",
    "tests/series/methods/test_equals.py::test_equals[arr2-idx2]",
    "tests/series/methods/test_equals.py::test_equals_list_array[val5]",
    "tests/series/methods/test_equals.py::test_equals_list_array[val6]",
    "tests/series/methods/test_equals.py::test_equals_list_array[val7]",
    "tests/series/methods/test_equals.py::test_equals_mismatched_nas[Decimal-NoneType]",
    "tests/series/methods/test_equals.py::test_equals_mismatched_nas[Decimal-float0]",
    "tests/series/methods/test_equals.py::test_equals_mismatched_nas[Decimal-float1]",
    "tests/series/methods/test_equals.py::test_equals_mismatched_nas[NoneType-Decimal]",
    "tests/series/methods/test_equals.py::test_equals_mismatched_nas[float0-Decimal]",
    "tests/series/methods/test_equals.py::test_equals_mismatched_nas[float1-Decimal]",
    "tests/series/methods/test_explode.py::test_duplicate_index",
    "tests/series/methods/test_explode.py::test_ignore_index",
    "tests/series/methods/test_fillna.py::TestSeriesFillNA::test_datetime64_fillna_mismatched_reso_no_rounding[None-False]",
    "tests/series/methods/test_fillna.py::TestSeriesFillNA::test_fillna",
    "tests/series/methods/test_fillna.py::TestSeriesFillNA::test_fillna_dont_cast_strings",
    "tests/series/methods/test_fillna.py::TestSeriesFillNA::test_fillna_nat",
    "tests/series/methods/test_fillna.py::TestSeriesFillNA::test_timedelta64_fillna_mismatched_reso_no_rounding[False]",
    "tests/series/methods/test_infer_objects.py::TestInferObjects::test_copy[index]",
    "tests/series/methods/test_infer_objects.py::TestInferObjects::test_copy[series]",
    "tests/series/methods/test_info.py::test_info_memory",
    "tests/series/methods/test_info.py::test_info_memory_usage_deep_not_pypy",
    "tests/series/methods/test_info.py::test_info_series[False]",
    "tests/series/methods/test_info.py::test_info_series[True]",
    "tests/series/methods/test_interpolate.py::TestSeriesInterpolateData::test_interpolate_akima",
    "tests/series/methods/test_interpolate.py::TestSeriesInterpolateData::test_interpolate_cubicspline",
    "tests/series/methods/test_interpolate.py::TestSeriesInterpolateData::test_interpolate_from_derivatives",
    "tests/series/methods/test_interpolate.py::TestSeriesInterpolateData::test_interpolate_piecewise_polynomial",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data0-values0-expected0-Float64]",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data0-values0-expected0-Int64]",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data0-values0-expected0-boolean]",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data2-values2-expected2-Float64]",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data2-values2-expected2-Int64]",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data2-values2-expected2-boolean]",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data4-values4-expected4-Float64]",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data4-values4-expected4-Int64]",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data4-values4-expected4-boolean]",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data5-values5-expected5-Float64]",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data5-values5-expected5-Int64]",
    "tests/series/methods/test_isin.py::TestSeriesIsIn::test_isin_masked_types[data5-values5-expected5-boolean]",
    "tests/series/methods/test_map.py::test_map_callable",
    "tests/series/methods/test_map.py::test_map_counter",
    "tests/series/methods/test_map.py::test_map_defaultdict",
    "tests/series/methods/test_map.py::test_map_defaultdict_ignore_na",
    "tests/series/methods/test_map.py::test_map_defaultdict_missing_key[None]",
    "tests/series/methods/test_map.py::test_map_dict_subclass_with_missing",
    "tests/series/methods/test_map.py::test_map_dict_with_tuple_keys",
    "tests/series/methods/test_map.py::test_map_empty[bool-dtype]",
    "tests/series/methods/test_map.py::test_map_empty[categorical]",
    "tests/series/methods/test_map.py::test_map_empty[empty]",
    "tests/series/methods/test_map.py::test_map_empty[float32]",
    "tests/series/methods/test_map.py::test_map_empty[float64]",
    "tests/series/methods/test_map.py::test_map_empty[int16]",
    "tests/series/methods/test_map.py::test_map_empty[int32]",
    "tests/series/methods/test_map.py::test_map_empty[int64]",
    "tests/series/methods/test_map.py::test_map_empty[int8]",
    "tests/series/methods/test_map.py::test_map_empty[nullable_bool]",
    "tests/series/methods/test_map.py::test_map_empty[nullable_float]",
    "tests/series/methods/test_map.py::test_map_empty[nullable_int]",
    "tests/series/methods/test_map.py::test_map_empty[nullable_uint]",
    "tests/series/methods/test_map.py::test_map_empty[object]",
    "tests/series/methods/test_map.py::test_map_empty[range]",
    "tests/series/methods/test_map.py::test_map_empty[repeats]",
    "tests/series/methods/test_map.py::test_map_empty[string]",
    "tests/series/methods/test_map.py::test_map_empty[uint16]",
    "tests/series/methods/test_map.py::test_map_empty[uint32]",
    "tests/series/methods/test_map.py::test_map_empty[uint64]",
    "tests/series/methods/test_map.py::test_map_empty[uint8]",
    "tests/series/methods/test_map.py::test_map_int",
    "tests/series/methods/test_map.py::test_map_simple_str_callables_same_as_astype[<lambda>]",
    "tests/series/methods/test_map.py::test_map_simple_str_callables_same_as_astype[str]",
    "tests/series/methods/test_nlargest.py::TestSeriesNLargestNSmallest::test_nlargest_boundary_datetimelike[nsmallest-datetime64[ns]]",
    "tests/series/methods/test_nlargest.py::TestSeriesNLargestNSmallest::test_nlargest_boundary_datetimelike[nsmallest-timedelta64[ns]]",
    "tests/series/methods/test_nlargest.py::TestSeriesNLargestNSmallest::test_nlargest_error[r4]",
    "tests/series/methods/test_nlargest.py::TestSeriesNLargestNSmallest::test_nlargest_misc",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_all_na[Int16]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_all_na[Int32]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_all_na[Int64]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_all_na[Int8]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_all_na[UInt16]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_all_na[UInt32]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_all_na[UInt64]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_all_na[UInt8]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_dtype_size[Int16]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_dtype_size[Int32]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_dtype_size[Int64]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_dtype_size[Int8]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_dtype_size[UInt16]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_dtype_size[UInt32]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_dtype_size[UInt64]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_dtype_size[UInt8]",
    "tests/series/methods/test_quantile.py::TestSeriesQuantile::test_quantile_dtypes[Int64]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_categorical",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[float64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[float64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[float64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[float64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[float64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[float64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[float64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[float64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[int64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[int64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[int64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[int64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[int64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[int64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[int64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[int64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[string[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[string[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[string[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[string[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[string[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[string[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[string[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_dense_method[string[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[average-float64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[average-int64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[average-string[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[dense-float64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[dense-int64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[dense-string[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[first-float64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[first-int64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[first-string[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[max-float64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[max-int64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[max-string[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[min-float64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[min-int64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[min-string[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[results0-Int64]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[results1-Int64]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[results2-Int64]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[results3-Int64]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_descending[results4-Int64]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[average-float64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[average-int64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[average-string[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[dense-float64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[dense-int64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[dense-string[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[first-float64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[first-int64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[first-string[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[max-float64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[max-int64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[max-string[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[min-float64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[min-int64[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods[min-string[pyarrow]]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-bottom-average-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-bottom-average-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-bottom-dense-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-bottom-dense-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-bottom-first-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-bottom-first-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-bottom-max-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-bottom-max-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-bottom-min-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-bottom-min-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-keep-average-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-keep-average-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-keep-dense-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-keep-dense-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-keep-first-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-keep-first-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-keep-max-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-keep-max-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-keep-min-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-keep-min-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-top-average-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-top-average-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-top-dense-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-top-dense-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-top-first-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-top-first-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-top-max-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-top-max-True]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-top-min-False]",
    "tests/series/methods/test_rank.py::TestSeriesRank::test_rank_tie_methods_on_infs_nans[float64[pyarrow]-na_value3-inf--inf-top-min-True]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[float64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[float64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[float64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[float64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[float64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[float64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[float64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[float64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[float64[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[int64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[int64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[int64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[int64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[int64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[int64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[int64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[int64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[int64[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[string[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[string[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[string[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[string[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[string[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[string[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[string[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[string[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_average_pct[string[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[float64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[float64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[float64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[float64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[float64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[float64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[float64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[float64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[float64[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[int64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[int64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[int64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[int64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[int64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[int64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[int64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[int64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[int64[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[string[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[string[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[string[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[string[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[string[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[string[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[string[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[string[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_dense_pct[string[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[float64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[float64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[float64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[float64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[float64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[float64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[float64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[float64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[float64[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[int64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[int64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[int64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[int64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[int64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[int64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[int64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[int64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[int64[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[string[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[string[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[string[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[string[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[string[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[string[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[string[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[string[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_first_pct[string[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[float64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[float64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[float64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[float64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[float64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[float64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[float64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[float64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[float64[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[int64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[int64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[int64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[int64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[int64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[int64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[int64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[int64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[int64[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[string[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[string[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[string[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[string[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[string[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[string[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[string[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[string[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_max_pct[string[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[float64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[float64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[float64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[float64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[float64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[float64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[float64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[float64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[float64[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[int64[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[int64[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[int64[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[int64[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[int64[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[int64[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[int64[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[int64[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[int64[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[string[pyarrow]-ser0-exp0]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[string[pyarrow]-ser1-exp1]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[string[pyarrow]-ser2-exp2]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[string[pyarrow]-ser3-exp3]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[string[pyarrow]-ser4-exp4]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[string[pyarrow]-ser5-exp5]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[string[pyarrow]-ser6-exp6]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[string[pyarrow]-ser7-exp7]",
    "tests/series/methods/test_rank.py::test_rank_min_pct[string[pyarrow]-ser8-exp8]",
    "tests/series/methods/test_reindex.py::test_reindex",
    "tests/series/methods/test_reindex.py::test_reindex_fill_value_datetimelike_upcast[0-datetime64[ns]]",
    "tests/series/methods/test_reindex.py::test_reindex_fill_value_datetimelike_upcast[0-timedelta64[ns]]",
    "tests/series/methods/test_reindex.py::test_reindex_pad2",
    "tests/series/methods/test_reindex.py::test_reindexing_with_float64_NA_log",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_Int_with_na[Int16]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_Int_with_na[Int32]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_Int_with_na[Int64]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_Int_with_na[Int8]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_Int_with_na[UInt16]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_Int_with_na[UInt32]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_Int_with_na[UInt64]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_Int_with_na[UInt8]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_datetime64",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_dtype[Float64-input_data4-to_replace4-expected_data4]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_dtype[Int64-input_data2-to_replace2-expected_data2]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_dtype[string-input_data5-to_replace5-expected_data5]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_explicit_none",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_mixed_types",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_na_in_obj_column[Int64]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_no_cast[ser0-exp0]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_nullable_numeric",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_numeric_column_with_na[0.5]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_numeric_column_with_na[0]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_string_dtype",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_string_dtype_list_to_replace",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_value_none_dtype_numeric[2.0]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_value_none_dtype_numeric[2]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_value_none_dtype_numeric[nan]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_with_dictlike_and_string_dtype[string[pyarrow]]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_with_dictlike_and_string_dtype[string[python]]",
    "tests/series/methods/test_replace.py::TestSeriesReplace::test_replace_with_single_list",
    "tests/series/methods/test_reset_index.py::TestResetIndex::test_reset_index_drop_errors",
    "tests/series/methods/test_reset_index.py::TestResetIndex::test_reset_index_level",
    "tests/series/methods/test_reset_index.py::test_column_name_duplicates[False-names0-expected_names0]",
    "tests/series/methods/test_reset_index.py::test_column_name_duplicates[False-names1-expected_names1]",
    "tests/series/methods/test_round.py::TestSeriesRound::test_round_builtin[Float32]",
    "tests/series/methods/test_round.py::TestSeriesRound::test_round_builtin[Float64]",
    "tests/series/methods/test_round.py::TestSeriesRound::test_round_numpy_with_nan[Float32]",
    "tests/series/methods/test_round.py::TestSeriesRound::test_round_numpy_with_nan[Float64]",
    "tests/series/methods/test_searchsorted.py::TestSeriesSearchSorted::test_searchsorted_datetime64_list",
    "tests/series/methods/test_searchsorted.py::TestSeriesSearchSorted::test_searchsorted_datetime64_scalar",
    "tests/series/methods/test_sort_index.py::TestSeriesSortIndex::test_sort_index_na_position",
    "tests/series/methods/test_sort_index.py::TestSeriesSortIndexKey::test_sort_index_na_position_key[None]",
    "tests/series/methods/test_sort_values.py::TestSeriesSortValues::test_sort_values_categorical",
    "tests/series/methods/test_unique.py::TestUnique::test_unique_categorical",
    "tests/series/methods/test_update.py::TestUpdate::test_update_dtypes[other8-int32-expected8-FutureWarning]",
    "tests/series/methods/test_update.py::TestUpdate::test_update_dtypes[other9-int64-expected9-FutureWarning]",
    "tests/series/methods/test_value_counts.py::TestSeriesValueCounts::test_value_counts_categorical_with_nan",
    "tests/series/test_api.py::TestSeriesMisc::test_index_tab_completion[index0]",
    "tests/series/test_api.py::TestSeriesMisc::test_index_tab_completion[index10]",
    "tests/series/test_api.py::TestSeriesMisc::test_index_tab_completion[index11]",
    "tests/series/test_api.py::TestSeriesMisc::test_index_tab_completion[index2]",
    "tests/series/test_api.py::TestSeriesMisc::test_numeric_only[bool-all-False]",
    "tests/series/test_api.py::TestSeriesMisc::test_numeric_only[bool-any-False]",
    "tests/series/test_api.py::TestSeriesMisc::test_numeric_only[float-all-False]",
    "tests/series/test_api.py::TestSeriesMisc::test_numeric_only[float-any-False]",
    "tests/series/test_api.py::TestSeriesMisc::test_numeric_only[int-all-False]",
    "tests/series/test_api.py::TestSeriesMisc::test_numeric_only[int-any-False]",
    "tests/series/test_api.py::TestSeriesMisc::test_tab_completion",
    "tests/series/test_api.py::TestSeriesMisc::test_tab_completion_cat",
    "tests/series/test_api.py::TestSeriesMisc::test_tab_completion_cat_str",
    "tests/series/test_api.py::TestSeriesMisc::test_tab_completion_dt",
    "tests/series/test_arithmetic.py::TestSeriesArithmetic::test_alignment_categorical[numexpr]",
    "tests/series/test_arithmetic.py::TestSeriesArithmetic::test_alignment_categorical[python]",
    "tests/series/test_arithmetic.py::TestSeriesArithmetic::test_alignment_doesnt_change_tz[numexpr]",
    "tests/series/test_arithmetic.py::TestSeriesArithmetic::test_alignment_doesnt_change_tz[python]",
    "tests/series/test_arithmetic.py::TestSeriesArithmetic::test_arithmetic_with_duplicate_index[numexpr]",
    "tests/series/test_arithmetic.py::TestSeriesArithmetic::test_arithmetic_with_duplicate_index[python]",
    "tests/series/test_arithmetic.py::TestSeriesComparison::test_comparison_tuples[numexpr]",
    "tests/series/test_arithmetic.py::TestSeriesComparison::test_comparison_tuples[python]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_add_scalar_fill_value[numexpr]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_add_scalar_fill_value[python]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__add__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__floordiv__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__mod__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__mul__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__pow__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__radd__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__rfloordiv__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__rmod__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__rmul__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__rpow__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__rsub__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__rtruediv__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__sub__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[numexpr-__truediv__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__add__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__floordiv__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__mod__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__mul__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__pow__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__radd__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__rfloordiv__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__rmod__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__rmul__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__rpow__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__rsub__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__rtruediv__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__sub__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_flex_method_subclass_metadata_preservation[python-__truediv__]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-add-add-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-floordiv-floordiv-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-mul-mul-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-pow-pow-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-radd-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-rfloordiv-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-rmul-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-rpow-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-rsub-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-rtruediv-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-rtruediv-rtruediv-1]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-sub-sub-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-truediv-truediv-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[numexpr-truediv-truediv-1]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-add-add-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-floordiv-floordiv-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-mul-mul-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-pow-pow-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-radd-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-rfloordiv-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-rmul-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-rpow-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-rsub-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-rtruediv-<lambda>-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-rtruediv-rtruediv-1]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-sub-sub-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-truediv-truediv-0]",
    "tests/series/test_arithmetic.py::TestSeriesFlexArithmetic::test_operators_combine[python-truediv-truediv-1]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_categorical_sideeffects_free",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_bool_dtype_missing_values",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_coerce_float_fail[int16]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_coerce_float_fail[int32]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_coerce_float_fail[int64]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_coerce_float_fail[int8]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_coerce_float_fail[int]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_coerce_float_fail[uint16]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_coerce_float_fail[uint32]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_coerce_float_fail[uint64]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_coerce_float_fail[uint8]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive['+01:15'-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive['-02:15'-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive['Asia/Tokyo'-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive['US/Eastern'-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive['UTC'-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive['UTC+01:15'-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive['UTC-02:15'-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive['dateutil/Asia/Singapore'-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive['dateutil/US/Pacific'-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive[<UTC>-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive[datetime.timezone(datetime.timedelta(seconds=3600))-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive[datetime.timezone.utc-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive[pytz.FixedOffset(-300)-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive[pytz.FixedOffset(300)-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive[tzlocal()-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive[tzutc()-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive[zoneinfo.ZoneInfo(key='US/Pacific')-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_data_aware_dtype_naive[zoneinfo.ZoneInfo(key='UTC')-True]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_datetimes_with_nulls",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_dtype_datetime64_10",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_dtype_datetime64_11",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_dtype_no_cast",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_dtype_timedelta64",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_floating_data_int_dtype[DataFrame]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_floating_data_int_dtype[Series]",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_list_of_tuples",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_maskedarray",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_no_data_string_type",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_no_partial_datetime_casting",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_pass_none",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_range_overflows",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_tuple_of_tuples",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_convert_non_ns",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_invalid_dtype",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_series_from_index_dtype_equal_does_not_copy",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_series_string_inference",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_series_string_inference_array_string_dtype",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_series_string_inference_scalar",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_series_string_inference_storage_definition",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_series_string_with_na_inference[None]",
    "tests/series/test_formats.py::TestCategoricalRepr::test_categorical_series_repr_datetime",
    "tests/series/test_formats.py::TestCategoricalRepr::test_categorical_series_repr_datetime_ordered",
    "tests/series/test_formats.py::TestCategoricalRepr::test_categorical_series_repr_timedelta",
    "tests/series/test_formats.py::TestCategoricalRepr::test_categorical_series_repr_timedelta_ordered",
    "tests/series/test_formats.py::TestSeriesRepr::test_newline",
    "tests/series/test_formats.py::TestSeriesRepr::test_newline[string=object]",
    "tests/series/test_formats.py::TestSeriesRepr::test_newline[string=str[pyarrow]]",
    "tests/series/test_formats.py::TestSeriesRepr::test_newline[string=str[python]]",
    "tests/series/test_formats.py::TestSeriesRepr::test_newline[string=string[pyarrow]]",
    "tests/series/test_formats.py::TestSeriesRepr::test_newline[string=string[python]]",
    "tests/series/test_formats.py::TestSeriesRepr::test_timeseries_repr_object_dtype",
    "tests/series/test_iteration.py::TestIteration::test_keys",
    "tests/series/test_logical_ops.py::TestSeriesLogicalOps::test_logical_operators_int_dtype_with_bool_dtype_and_reindex",
    "tests/series/test_logical_ops.py::TestSeriesLogicalOps::test_logical_ops_df_compat",
    "tests/series/test_logical_ops.py::TestSeriesLogicalOps::test_logical_ops_label_based",
    "tests/series/test_logical_ops.py::TestSeriesLogicalOps::test_reversed_xor_with_index_returns_series",
    "tests/series/test_missing.py::TestSeriesMissingData::test_timedelta64_nan",
    "tests/series/test_missing.py::test_hasnans_uncached_for_series",
    "tests/series/test_reductions.py::test_td64_summation_overflow",
    "tests/series/test_subclass.py::TestSeriesSubclassing::test_asof",
    "tests/series/test_subclass.py::TestSeriesSubclassing::test_equals",
    "tests/series/test_ufunc.py::TestNumpyReductions::test_max[interval[int64, right]-DataFrame]",
    "tests/series/test_ufunc.py::TestNumpyReductions::test_min[interval[int64, right]-DataFrame]",
    "tests/series/test_ufunc.py::test_binary_ufunc_scalar[add-sparse-False]",
    "tests/series/test_ufunc.py::test_binary_ufunc_scalar[add-sparse-True]",
    "tests/series/test_ufunc.py::test_binary_ufunc_scalar[logaddexp-sparse-False]",
    "tests/series/test_ufunc.py::test_binary_ufunc_scalar[logaddexp-sparse-True]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_array[sparse-add-flipped]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_array[sparse-add-straight]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_array[sparse-logaddexp-flipped]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_array[sparse-logaddexp-straight]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_index[dense-logaddexp-flipped]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_index[sparse-add-flipped]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_index[sparse-add-straight]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_index[sparse-logaddexp-flipped]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_index[sparse-logaddexp-straight]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_series[sparse-add-flipped-aligned]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_series[sparse-add-flipped-unaligned]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_series[sparse-add-straight-aligned]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_series[sparse-add-straight-unaligned]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_series[sparse-logaddexp-flipped-aligned]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_series[sparse-logaddexp-flipped-unaligned]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_series[sparse-logaddexp-straight-aligned]",
    "tests/series/test_ufunc.py::test_binary_ufunc_with_series[sparse-logaddexp-straight-unaligned]",
    "tests/series/test_ufunc.py::test_multiple_output_binary_ufuncs[sparse-False-divmod]",
    "tests/series/test_ufunc.py::test_multiple_output_binary_ufuncs[sparse-True-divmod]",
    "tests/series/test_ufunc.py::test_multiple_output_ufunc[sparse]",
    "tests/series/test_ufunc.py::test_np_fix",
    "tests/series/test_ufunc.py::test_unary_ufunc[dense-floor]",
    "tests/series/test_ufunc.py::test_unary_ufunc[sparse-exp]",
    "tests/series/test_ufunc.py::test_unary_ufunc[sparse-floor]",
    "tests/series/test_ufunc.py::test_unary_ufunc[sparse-positive]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Float32-source0-neg_target0-abs_target0]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Float32-source1-neg_target1-abs_target1]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Float64-source0-neg_target0-abs_target0]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Float64-source1-neg_target1-abs_target1]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Int16-source0-neg_target0-abs_target0]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Int16-source1-neg_target1-abs_target1]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Int32-source0-neg_target0-abs_target0]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Int32-source1-neg_target1-abs_target1]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Int64-source0-neg_target0-abs_target0]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Int64-source1-neg_target1-abs_target1]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Int8-source0-neg_target0-abs_target0]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[Int8-source1-neg_target1-abs_target1]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[UInt16-source0-neg_target0-abs_target0]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[UInt16-source1-neg_target1-abs_target1]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[UInt32-source0-neg_target0-abs_target0]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[UInt32-source1-neg_target1-abs_target1]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[UInt64-source0-neg_target0-abs_target0]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[UInt64-source1-neg_target1-abs_target1]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[UInt8-source0-neg_target0-abs_target0]",
    "tests/series/test_unary.py::TestSeriesUnaryOps::test_all_numeric_unary_operators[UInt8-source1-neg_target1-abs_target1]",
    "tests/strings/test_api.py::test_api_for_categorical[count-object]",
    "tests/strings/test_api.py::test_api_for_categorical[count-string[pyarrow]]",
    "tests/strings/test_api.py::test_api_for_categorical[count-string[pyarrow_numpy]]",
    "tests/strings/test_api.py::test_api_for_categorical[count-string[python]]",
    "tests/strings/test_api.py::test_api_for_categorical[find-object]",
    "tests/strings/test_api.py::test_api_for_categorical[find-string[pyarrow]]",
    "tests/strings/test_api.py::test_api_for_categorical[find-string[pyarrow_numpy]]",
    "tests/strings/test_api.py::test_api_for_categorical[find-string[python]]",
    "tests/strings/test_api.py::test_api_for_categorical[len-object]",
    "tests/strings/test_api.py::test_api_for_categorical[len-string[pyarrow]]",
    "tests/strings/test_api.py::test_api_for_categorical[len-string[pyarrow_numpy]]",
    "tests/strings/test_api.py::test_api_for_categorical[len-string[python]]",
    "tests/strings/test_api.py::test_api_for_categorical[rfind-object]",
    "tests/strings/test_api.py::test_api_for_categorical[rfind-string[pyarrow]]",
    "tests/strings/test_api.py::test_api_for_categorical[rfind-string[pyarrow_numpy]]",
    "tests/strings/test_api.py::test_api_for_categorical[rfind-string[python]]",
    "tests/strings/test_api.py::test_api_mi_raises",
    "tests/strings/test_extract.py::test_extract_expand_capture_groups[object]",
    "tests/strings/test_extract.py::test_extract_expand_capture_groups[string[pyarrow]]",
    "tests/strings/test_extract.py::test_extract_expand_capture_groups[string[pyarrow_numpy]]",
    "tests/strings/test_extract.py::test_extract_expand_capture_groups[string[python]]",
    "tests/strings/test_extract.py::test_extract_series[string[pyarrow]-None]",
    "tests/strings/test_extract.py::test_extract_series[string[pyarrow]-series_name]",
    "tests/strings/test_extract.py::test_extract_series[string[pyarrow_numpy]-None]",
    "tests/strings/test_extract.py::test_extract_series[string[pyarrow_numpy]-series_name]",
    "tests/strings/test_extract.py::test_extract_series[string[python]-None]",
    "tests/strings/test_extract.py::test_extract_series[string[python]-series_name]",
    "tests/strings/test_find_replace.py::test_find[object]",
    "tests/strings/test_find_replace.py::test_flags_kwarg[string[pyarrow]]",
    "tests/strings/test_find_replace.py::test_flags_kwarg[string[pyarrow_numpy]]",
    "tests/strings/test_find_replace.py::test_fullmatch_case_kwarg[string[pyarrow]]",
    "tests/strings/test_find_replace.py::test_fullmatch_case_kwarg[string[pyarrow_numpy]]",
    "tests/strings/test_find_replace.py::test_replace_callable[string[pyarrow]]",
    "tests/strings/test_find_replace.py::test_replace_callable[string[pyarrow_numpy]]",
    "tests/strings/test_find_replace.py::test_replace_callable_named_groups[string[pyarrow]]",
    "tests/strings/test_find_replace.py::test_replace_callable_named_groups[string[pyarrow_numpy]]",
    "tests/strings/test_find_replace.py::test_replace_callable_raises[string[pyarrow]-<lambda>0]",
    "tests/strings/test_find_replace.py::test_replace_callable_raises[string[pyarrow]-<lambda>1]",
    "tests/strings/test_find_replace.py::test_replace_callable_raises[string[pyarrow]-<lambda>2]",
    "tests/strings/test_find_replace.py::test_replace_callable_raises[string[pyarrow_numpy]-<lambda>0]",
    "tests/strings/test_find_replace.py::test_replace_callable_raises[string[pyarrow_numpy]-<lambda>1]",
    "tests/strings/test_find_replace.py::test_replace_callable_raises[string[pyarrow_numpy]-<lambda>2]",
    "tests/strings/test_find_replace.py::test_replace_compiled_regex[string[pyarrow]]",
    "tests/strings/test_find_replace.py::test_replace_compiled_regex[string[pyarrow_numpy]]",
    "tests/strings/test_find_replace.py::test_replace_compiled_regex_callable[string[pyarrow]]",
    "tests/strings/test_find_replace.py::test_replace_compiled_regex_callable[string[pyarrow_numpy]]",
    "tests/strings/test_find_replace.py::test_replace_compiled_regex_unicode[string[pyarrow]]",
    "tests/strings/test_find_replace.py::test_replace_compiled_regex_unicode[string[pyarrow_numpy]]",
    "tests/strings/test_find_replace.py::test_replace_moar[string[pyarrow]]",
    "tests/strings/test_find_replace.py::test_replace_moar[string[pyarrow_numpy]]",
    "tests/strings/test_find_replace.py::test_replace_not_case_sensitive_not_regex[string[pyarrow]]",
    "tests/strings/test_find_replace.py::test_replace_not_case_sensitive_not_regex[string[pyarrow_numpy]]",
    "tests/strings/test_find_replace.py::test_replace_unicode[string[pyarrow]]",
    "tests/strings/test_find_replace.py::test_replace_unicode[string[pyarrow_numpy]]",
    "tests/strings/test_split_partition.py::test_partition_series_stdlib[string=object-partition]",
    "tests/strings/test_split_partition.py::test_partition_series_stdlib[string=object-rpartition]",
    "tests/strings/test_split_partition.py::test_partition_series_stdlib[string=str[pyarrow]-partition]",
    "tests/strings/test_split_partition.py::test_partition_series_stdlib[string=str[pyarrow]-rpartition]",
    "tests/strings/test_split_partition.py::test_partition_series_stdlib[string=str[python]-partition]",
    "tests/strings/test_split_partition.py::test_partition_series_stdlib[string=str[python]-rpartition]",
    "tests/strings/test_split_partition.py::test_partition_series_stdlib[string=string[pyarrow]-partition]",
    "tests/strings/test_split_partition.py::test_partition_series_stdlib[string=string[pyarrow]-rpartition]",
    "tests/strings/test_split_partition.py::test_partition_series_stdlib[string=string[python]-partition]",
    "tests/strings/test_split_partition.py::test_partition_series_stdlib[string=string[python]-rpartition]",
    "tests/strings/test_split_partition.py::test_split_blank_string[object]",
    "tests/strings/test_strings.py::test_empty_str_methods[object]",
    "tests/strings/test_strings.py::test_index_str_accessor_multiindex_raises",
    "tests/test_algos.py::TestFactorize::test_basic",
    "tests/test_algos.py::TestFactorize::test_datetime64_factorize[False]",
    "tests/test_algos.py::TestFactorize::test_datetime64_factorize[True]",
    "tests/test_algos.py::TestFactorize::test_factorize[multi-False]",
    "tests/test_algos.py::TestFactorize::test_factorize[multi-True]",
    "tests/test_algos.py::TestFactorize::test_factorize[tuples-False]",
    "tests/test_algos.py::TestFactorize::test_factorize[tuples-True]",
    "tests/test_algos.py::TestFactorize::test_factorize_rangeindex[False]",
    "tests/test_algos.py::TestFactorize::test_factorize_rangeindex[True]",
    "tests/test_algos.py::TestFactorize::test_factorize_rangeindex_decreasing[False]",
    "tests/test_algos.py::TestFactorize::test_factorize_rangeindex_decreasing[True]",
    "tests/test_algos.py::TestFactorize::test_factorize_use_na_sentinel[numpy_array-False]",
    "tests/test_algos.py::TestFactorize::test_factorize_use_na_sentinel[numpy_array-True]",
    "tests/test_algos.py::TestFactorize::test_float64_factorize[False]",
    "tests/test_algos.py::TestFactorize::test_float64_factorize[True]",
    "tests/test_algos.py::TestFactorize::test_int64_factorize[False]",
    "tests/test_algos.py::TestFactorize::test_int64_factorize[True]",
    "tests/test_algos.py::TestFactorize::test_int_factorize_use_na_sentinel_false[data0-expected_codes0-expected_uniques0]",
    "tests/test_algos.py::TestFactorize::test_int_factorize_use_na_sentinel_false[data1-expected_codes1-expected_uniques1]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[float32]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[float64]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[float]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[int16]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[int32]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[int64]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[int8]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[int]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[uint16]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[uint32]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[uint64]",
    "tests/test_algos.py::TestFactorize::test_numeric_dtype_factorize[uint8]",
    "tests/test_algos.py::TestFactorize::test_object_factorize_use_na_sentinel_false[data0-expected_codes0-expected_uniques0]",
    "tests/test_algos.py::TestFactorize::test_parametrized_factorize_na_value_default[data0]",
    "tests/test_algos.py::TestFactorize::test_parametrized_factorize_na_value_default[data1]",
    "tests/test_algos.py::TestFactorize::test_parametrized_factorize_na_value_default[data2]",
    "tests/test_algos.py::TestFactorize::test_string_factorize[False]",
    "tests/test_algos.py::TestFactorize::test_string_factorize[True]",
    "tests/test_algos.py::TestFactorize::test_uint64_factorize[False]",
    "tests/test_algos.py::TestFactorize::test_uint64_factorize[True]",
    "tests/test_algos.py::TestIsin::test_isin_datetimelike_all_nat[M8[ns, UTC]]",
    "tests/test_algos.py::TestIsin::test_isin_datetimelike_all_nat[M8[ns]]",
    "tests/test_algos.py::TestIsin::test_isin_datetimelike_all_nat[m8[ns]]",
    "tests/test_algos.py::TestIsin::test_isin_datetimelike_all_nat[period[D]]",
    "tests/test_algos.py::TestUnique::test_datetime64_dtype_array_returned",
    "tests/test_algos.py::TestUnique::test_dtype_preservation[M8[ns]]",
    "tests/test_algos.py::TestUnique::test_dtype_preservation[datetime64[ns]]",
    "tests/test_algos.py::TestUnique::test_dtype_preservation[m8[ns]]",
    "tests/test_algos.py::TestUnique::test_dtype_preservation[timedelta64[ns]]",
    "tests/test_algos.py::TestUnique::test_on_index_object",
    "tests/test_algos.py::TestUnique::test_order_of_appearance_dt64tz[ms]",
    "tests/test_algos.py::TestUnique::test_order_of_appearance_dt64tz[ns]",
    "tests/test_algos.py::TestUnique::test_order_of_appearance_dt64tz[s]",
    "tests/test_algos.py::TestUnique::test_order_of_appearance_dt64tz[us]",
    "tests/test_algos.py::TestUnique::test_timedelta64_dtype_array_returned",
    "tests/test_algos.py::TestValueCounts::test_categorical_nans",
    "tests/test_algos.py::TestValueCounts::test_value_counts_dropna",
    "tests/test_common.py::test_serializable[obj0]",
    "tests/test_common.py::test_standardize_mapping",
    "tests/test_common.py::test_temp_setattr[False]",
    "tests/test_common.py::test_temp_setattr[True]",
    "tests/test_downstream.py::test_construct_dask_float_array_int_dtype_match_ndarray",
    "tests/test_downstream.py::test_dask_ufunc",
    "tests/test_downstream.py::test_from_obscure_array[dask-M8[ns]]",
    "tests/test_downstream.py::test_from_obscure_array[dask-m8[ns]]",
    "tests/test_downstream.py::test_statsmodels",
    "tests/test_downstream.py::test_yaml_dump",
    "tests/test_expressions.py::TestExpressions::test_bool_ops_raise_on_arithmetic[**-pow]",
    "tests/test_expressions.py::TestExpressions::test_bool_ops_raise_on_arithmetic[/-truediv]",
    "tests/test_expressions.py::TestExpressions::test_bool_ops_raise_on_arithmetic[//-floordiv]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-False-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-False-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-False-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-False-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-False-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-False-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-False-_mixed]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-True-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-True-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-True-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-True-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-True-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-True-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[eq-True-_mixed]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-False-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-False-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-False-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-False-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-False-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-False-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-False-_mixed]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-True-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-True-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-True-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-True-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-True-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-True-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ge-True-_mixed]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-False-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-False-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-False-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-False-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-False-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-False-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-False-_mixed]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-True-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-True-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-True-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-True-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-True-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-True-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[gt-True-_mixed]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-False-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-False-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-False-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-False-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-False-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-False-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-False-_mixed]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-True-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-True-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-True-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-True-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-True-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-True-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[le-True-_mixed]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-False-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-False-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-False-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-False-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-False-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-False-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-False-_mixed]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-True-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-True-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-True-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-True-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-True-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-True-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[lt-True-_mixed]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-False-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-False-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-False-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-False-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-False-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-False-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-False-_mixed]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-True-_frame2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-True-_frame]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-True-_integer2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-True-_integer]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-True-_integer_integers]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-True-_mixed2]",
    "tests/test_expressions.py::TestExpressions::test_run_binary[ne-True-_mixed]",
    "tests/test_multilevel.py::TestMultiLevel::test_reindex_preserve_levels",
    "tests/test_multilevel.py::TestSorted::test_sort_non_lexsorted",
    "tests/test_nanops.py::TestDatetime64NaNOps::test_nanmean[ns]",
    "tests/test_nanops.py::test_nanmean_overflow[20150515061816532]",
    "tests/test_register_accessor.py::test_register[DataFrame-register_dataframe_accessor]",
    "tests/test_register_accessor.py::test_register[Index-register_index_accessor]",
    "tests/test_register_accessor.py::test_register[Series-register_series_accessor]",
    "tests/tools/test_to_datetime.py::TestOrigin::test_to_datetime_out_of_bounds_with_format_arg[%Y-%d-%m %H:%M:%S-None]",
    "tests/tools/test_to_datetime.py::TestOrigin::test_to_datetime_out_of_bounds_with_format_arg[%Y-%m-%d %H:%M:%S-None]",
    "tests/tools/test_to_datetime.py::TestOrigin::test_unix",
    "tests/tools/test_to_datetime.py::TestTimeConversionFormats::test_int_to_datetime_format_YYYYMMDD_typeerror[199934-expected1]",
    "tests/tools/test_to_datetime.py::TestTimeConversionFormats::test_int_to_datetime_format_YYYYMMDD_typeerror[20121030-expected0]",
    "tests/tools/test_to_datetime.py::TestTimeConversionFormats::test_to_datetime_format_weeks[False-2013020-%Y%U%w-expected1]",
    "tests/tools/test_to_datetime.py::TestTimeConversionFormats::test_to_datetime_format_weeks[True-2013020-%Y%U%w-expected1]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_datetime_bool[False-False]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_datetime_bool[False-True]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_datetime_bool[True-False]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_datetime_bool[True-True]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_datetime_invalid_scalar[%H:%M:%S-00:01:99]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_datetime_invalid_scalar[%H:%M:%S-a]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_datetime_invalid_scalar[None-00:01:99]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_datetime_invalid_scalar[None-a]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_datetime_outofbounds_scalar[%H:%M:%S-3000/12/11 00:00:00]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_datetime_outofbounds_scalar[None-3000/12/11 00:00:00]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-Day of the year directive '%j' is not compatible with ISO year directive '%G'. Use '%Y' instead.-1999 50-%G %j]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-Day of the year directive '%j' is not compatible with ISO year directive '%G'. Use '%Y' instead.-1999 51 6 256-%G %V %u %j]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO week directive '%V' is incompatible with the year directive '%Y'. Use the ISO year '%G' instead.-1999 50-%Y %V]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO week directive '%V' is incompatible with the year directive '%Y'. Use the ISO year '%G' instead.-1999 51 1-%Y %V %u]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO week directive '%V' is incompatible with the year directive '%Y'. Use the ISO year '%G' instead.-1999 51 1-%Y %V %w]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO week directive '%V' is incompatible with the year directive '%Y'. Use the ISO year '%G' instead.-1999 51 Sun-%Y %V %a]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO week directive '%V' is incompatible with the year directive '%Y'. Use the ISO year '%G' instead.-1999 51 Sunday-%Y %V %A]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 51 1-%V %u]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 51 1-%V %w]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 51 Sun-%V %a]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 51 Sunday-%V %A]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-20 Monday-%V %A]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-20-%V]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 51-%G %V]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 6-%G %u]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 6-%G %w]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 Mon-%G %a]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 Monday-%G %A]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[coerce-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-2051-%G]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-Day of the year directive '%j' is not compatible with ISO year directive '%G'. Use '%Y' instead.-1999 50-%G %j]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-Day of the year directive '%j' is not compatible with ISO year directive '%G'. Use '%Y' instead.-1999 51 6 256-%G %V %u %j]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO week directive '%V' is incompatible with the year directive '%Y'. Use the ISO year '%G' instead.-1999 50-%Y %V]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO week directive '%V' is incompatible with the year directive '%Y'. Use the ISO year '%G' instead.-1999 51 1-%Y %V %u]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO week directive '%V' is incompatible with the year directive '%Y'. Use the ISO year '%G' instead.-1999 51 1-%Y %V %w]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO week directive '%V' is incompatible with the year directive '%Y'. Use the ISO year '%G' instead.-1999 51 Sun-%Y %V %a]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO week directive '%V' is incompatible with the year directive '%Y'. Use the ISO year '%G' instead.-1999 51 Sunday-%Y %V %A]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 51 1-%V %u]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 51 1-%V %w]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 51 Sun-%V %a]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 51 Sunday-%V %A]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-20 Monday-%V %A]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO week directive '%V' must be used with the ISO year directive '%G' and a weekday directive '%A', '%a', '%w', or '%u'.-20-%V]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 51-%G %V]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 6-%G %u]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 6-%G %w]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 Mon-%G %a]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-1999 Monday-%G %A]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_error_iso_week_year[ignore-ISO year directive '%G' must be used with the ISO week directive '%V' and a weekday directive '%A', '%a', '%w', or '%u'.-2051-%G]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_mixed_offsets_with_native_datetime_raises",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_arrow[None-False-Index]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_arrow[US/Central-False-Index]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64d_out_of_bounds[False]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64d_out_of_bounds[True]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[False-coerce-dt0]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[False-coerce-dt1]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[False-ignore-dt0]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[False-ignore-dt1]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[False-raise-dt0]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[False-raise-dt1]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[True-coerce-dt0]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[True-coerce-dt1]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[True-ignore-dt0]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[True-ignore-dt1]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[True-raise-dt0]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dt64s_out_of_ns_bounds[True-raise-dt1]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dtarr[None]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_dtarr[US/Central]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_overflow",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_with_format_out_of_bounds[00010101]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_with_format_out_of_bounds[13000101]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_with_format_out_of_bounds[30000101]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_to_datetime_with_format_out_of_bounds[99990101]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_week_without_day_and_calendar_year[2017-20-%Y-%W]",
    "tests/tools/test_to_datetime.py::TestToDatetime::test_week_without_day_and_calendar_year[2017-21-%Y-%U]",
    "tests/tools/test_to_datetime.py::TestToDatetimeDataFrame::test_dataframe_str_dtype[False]",
    "tests/tools/test_to_datetime.py::TestToDatetimeDataFrame::test_dataframe_str_dtype[True]",
    "tests/tools/test_to_datetime.py::TestToDatetimeMisc::test_to_datetime_barely_out_of_bounds",
    "tests/tools/test_to_datetime.py::TestToDatetimeMisc::test_to_datetime_iso8601_exact_fails[2012-01-01 10-%Y-%m-%d]",
    "tests/tools/test_to_datetime.py::TestToDatetimeMisc::test_to_datetime_iso8601_exact_fails[2012-01-01 10:00-%Y-%m-%d %H]",
    "tests/tools/test_to_datetime.py::TestToDatetimeMisc::test_to_datetime_iso8601_exact_fails[2012-01-01 10:00:00-%Y-%m-%d %H:%M]",
    "tests/tools/test_to_datetime.py::TestToDatetimeMisc::test_to_datetime_iso8601_exact_fails[2012-01-01-%Y-%m]",
    "tests/tools/test_to_datetime.py::TestToDatetimeUnit::test_float_to_datetime_raise_near_bounds",
    "tests/tools/test_to_datetime.py::TestToDatetimeUnit::test_to_datetime_errors_ignore_utc_true",
    "tests/tools/test_to_datetime.py::TestToDatetimeUnit::test_to_datetime_unit[int]",
    "tests/tools/test_to_datetime.py::TestToDatetimeUnit::test_to_datetime_unit_fractional_seconds",
    "tests/tools/test_to_datetime.py::TestToDatetimeUnit::test_to_datetime_unit_invalid[111111111]",
    "tests/tools/test_to_datetime.py::TestToDatetimeUnit::test_to_datetime_unit_with_nulls[-9223372036854775808]",
    "tests/tools/test_to_datetime.py::TestToDatetimeUnit::test_unit_rounding[False]",
    "tests/tools/test_to_datetime.py::TestToDatetimeUnit::test_unit_rounding[True]",
    "tests/tools/test_to_datetime.py::test_na_to_datetime[Decimal-list]",
    "tests/tools/test_to_numeric.py::test_downcast_basic[kwargs0-int64-data2]",
    "tests/tools/test_to_numeric.py::test_downcast_basic[kwargs1-int64-data2]",
    "tests/tools/test_to_numeric.py::test_downcast_basic[kwargs2-f-data2]",
    "tests/tools/test_to_numeric.py::test_downcast_basic[kwargs3-exp_dtype3-data2]",
    "tests/tools/test_to_numeric.py::test_downcast_limits[int16-integer-min_max11]",
    "tests/tools/test_to_numeric.py::test_downcast_limits[int16-integer-min_max1]",
    "tests/tools/test_to_numeric.py::test_downcast_limits[int32-integer-min_max12]",
    "tests/tools/test_to_numeric.py::test_downcast_limits[int32-integer-min_max2]",
    "tests/tools/test_to_numeric.py::test_downcast_limits[int8-integer-min_max0]",
    "tests/tools/test_to_numeric.py::test_downcast_limits[uint16-unsigned-min_max5]",
    "tests/tools/test_to_numeric.py::test_downcast_limits[uint32-unsigned-min_max6]",
    "tests/tools/test_to_numeric.py::test_downcast_limits[uint8-unsigned-min_max4]",
    "tests/tools/test_to_numeric.py::test_numeric_dtypes[transform_assert_equal0-data0]",
    "tests/tools/test_to_numeric.py::test_numeric_dtypes[transform_assert_equal0-data1]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[100000000000000000000-False-str-False-None]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[100000000000000000000-False-str-False-coerce]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[100000000000000000000-False-str-False-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[100000000000000000000-False-str-False-raise]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[100000000000000000000-False-str-True-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[100000000000000000000-True-str-False-None]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[100000000000000000000-True-str-False-coerce]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[100000000000000000000-True-str-False-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[100000000000000000000-True-str-False-raise]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[100000000000000000000-True-str-True-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[47393996303418497800-False-str-False-None]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[47393996303418497800-False-str-False-coerce]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[47393996303418497800-False-str-False-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[47393996303418497800-False-str-False-raise]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[47393996303418497800-False-str-True-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[47393996303418497800-True-str-False-None]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[47393996303418497800-True-str-False-coerce]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[47393996303418497800-True-str-False-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[47393996303418497800-True-str-False-raise]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr[47393996303418497800-True-str-True-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[100000000000000000000-False-False-None]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[100000000000000000000-False-False-coerce]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[100000000000000000000-False-False-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[100000000000000000000-False-False-raise]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[100000000000000000000-True-False-None]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[100000000000000000000-True-False-coerce]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[100000000000000000000-True-False-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[100000000000000000000-True-False-raise]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[47393996303418497800-False-False-None]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[47393996303418497800-False-False-coerce]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[47393996303418497800-False-False-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[47393996303418497800-False-False-raise]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[47393996303418497800-True-False-None]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[47393996303418497800-True-False-coerce]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[47393996303418497800-True-False-ignore]",
    "tests/tools/test_to_numeric.py::test_really_large_in_arr_consistent[47393996303418497800-True-False-raise]",
    "tests/tools/test_to_numeric.py::test_signed_downcast[data2-integer]",
    "tests/tools/test_to_numeric.py::test_signed_downcast[data2-signed]",
    "tests/tools/test_to_numeric.py::test_str[transform_assert_equal0-data0-exp0]",
    "tests/tools/test_to_numeric.py::test_str[transform_assert_equal0-data1-exp1]",
    "tests/tools/test_to_numeric.py::test_to_numeric_large_float_not_downcast_to_float_32[9876543210.0]",
    "tests/tools/test_to_timedelta.py::TestTimedeltas::test_to_timedelta_nullable_int64_dtype[None-None]",
    "tests/tools/test_to_timedelta.py::TestTimedeltas::test_to_timedelta_nullable_int64_dtype[expected_val0-2]",
    "tests/tools/test_to_timedelta.py::TestTimedeltas::test_to_timedelta_series",
    "tests/tools/test_to_timedelta.py::TestTimedeltas::test_to_timedelta_via_apply",
    "tests/tseries/holiday/test_calendar.py::test_calendar_2031",
    "tests/tseries/holiday/test_federal.py::test_memorial_day",
    "tests/tseries/holiday/test_federal.py::test_no_mlk_before_1986",
    "tests/tseries/offsets/test_business_day.py::TestBusinessDay::test_with_offset_index[timedelta64]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[day]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[days]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[hour]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[hours]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[microsecond]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[microseconds]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[milliseconds]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[minute]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[minutes]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[month]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[months]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[nanosecond]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[nanoseconds]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[second]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[seconds]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[weekday]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[weeks]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[year]",
    "tests/tseries/offsets/test_offsets.py::TestDateOffset::test_constructor[years]",
    "tests/tseries/offsets/test_offsets.py::test_construct_int_arg_no_kwargs_assumed_days[-1]",
    "tests/tseries/offsets/test_offsets.py::test_construct_int_arg_no_kwargs_assumed_days[1]",
    "tests/tseries/offsets/test_offsets.py::test_construct_int_arg_no_kwargs_assumed_days[3]",
    "tests/tseries/offsets/test_ticks.py::test_tick_addition[Hour-expected0]",
    "tests/tseries/offsets/test_ticks.py::test_tick_addition[Micro-expected4]",
    "tests/tseries/offsets/test_ticks.py::test_tick_addition[Milli-expected3]",
    "tests/tseries/offsets/test_ticks.py::test_tick_addition[Minute-expected1]",
    "tests/tseries/offsets/test_ticks.py::test_tick_addition[Nano-expected5]",
    "tests/tseries/offsets/test_ticks.py::test_tick_addition[Second-expected2]",
    "tests/tseries/offsets/test_ticks.py::test_tick_rdiv[Hour]",
    "tests/tseries/offsets/test_ticks.py::test_tick_rdiv[Micro]",
    "tests/tseries/offsets/test_ticks.py::test_tick_rdiv[Milli]",
    "tests/tseries/offsets/test_ticks.py::test_tick_rdiv[Minute]",
    "tests/tseries/offsets/test_ticks.py::test_tick_rdiv[Nano]",
    "tests/tseries/offsets/test_ticks.py::test_tick_rdiv[Second]",
    "tests/tseries/offsets/test_year.py::test_add_out_of_pydatetime_range",
    "tests/tslibs/test_conversion.py::test_length_zero_copy[M8[ns]-False]",
    "tests/tslibs/test_conversion.py::test_tz_convert_single_matches_tz_convert_hourly[zoneinfo.ZoneInfo(key='US/Pacific')]",
    "tests/tslibs/test_parsing.py::test_guess_datetime_format_with_parseable_formats[27.03.2003 14:55:00.000-%d.%m.%Y %H:%M:%S.%f]",
    "tests/tslibs/test_parsing.py::test_guess_datetime_format_with_parseable_formats[30-12-2011-%d-%m-%Y]",
    "tests/tslibs/test_parsing.py::test_parsing_tzlocal_deprecated",
    "tests/tslibs/test_to_offset.py::test_to_offset_lowercase_frequency_deprecated[2bms]",
    "tests/tslibs/test_to_offset.py::test_to_offset_lowercase_frequency_deprecated[2bqs]",
    "tests/tslibs/test_to_offset.py::test_to_offset_lowercase_frequency_deprecated[2cbme]",
    "tests/tslibs/test_to_offset.py::test_to_offset_lowercase_frequency_deprecated[2me]",
    "tests/tslibs/test_to_offset.py::test_to_offset_lowercase_frequency_deprecated[2qe]",
    "tests/tslibs/test_to_offset.py::test_to_offset_lowercase_frequency_deprecated[2qs-feb]",
    "tests/tslibs/test_to_offset.py::test_to_offset_lowercase_frequency_deprecated[2sms]",
    "tests/tslibs/test_to_offset.py::test_to_offset_lowercase_frequency_deprecated[2w]",
    "tests/tslibs/test_to_offset.py::test_to_offset_lowercase_frequency_deprecated[2ye-mar]",
    "tests/tslibs/test_to_offset.py::test_to_offset_lowercase_frequency_deprecated[2ys]",
    "tests/tslibs/test_to_offset.py::test_to_offset_uppercase_frequency_deprecated[2BH]",
    "tests/tslibs/test_to_offset.py::test_to_offset_uppercase_frequency_deprecated[2H]",
    "tests/tslibs/test_to_offset.py::test_to_offset_uppercase_frequency_deprecated[2MIN]",
    "tests/tslibs/test_to_offset.py::test_to_offset_uppercase_frequency_deprecated[2NS]",
    "tests/tslibs/test_to_offset.py::test_to_offset_uppercase_frequency_deprecated[2S]",
    "tests/tslibs/test_to_offset.py::test_to_offset_uppercase_frequency_deprecated[2Us]",
    "tests/util/test_assert_frame_equal.py::test_allows_duplicate_labels",
    "tests/util/test_assert_frame_equal.py::test_assert_frame_equal_check_like_different_indexes",
    "tests/util/test_assert_frame_equal.py::test_assert_frame_equal_checking_allow_dups_flag",
    "tests/util/test_assert_frame_equal.py::test_assert_frame_equal_extension_dtype_mismatch",
    "tests/util/test_assert_frame_equal.py::test_frame_equal_index_dtype_mismatch[True-df11-df21-MultiIndex level \\\\[0\\\\] are different]",
    "tests/util/test_assert_frame_equal.py::test_frame_equal_index_dtype_mismatch[True-df11-df21-MultiIndex level \\\\\\\\\\\\\\\\[0\\\\\\\\\\\\\\\\] are different]",
    "tests/util/test_assert_index_equal.py::test_index_equal_range_categories[True-True]",
    "tests/util/test_assert_series_equal.py::test_allows_duplicate_labels",
    "tests/util/test_assert_series_equal.py::test_assert_series_equal_extension_dtype_mismatch",
    "tests/util/test_assert_series_equal.py::test_large_unequal_ints[int64]",
    "tests/util/test_assert_series_equal.py::test_series_equal_index_dtype[True-s11-s21-MultiIndex level \\\\[0\\\\] are different]",
    "tests/util/test_assert_series_equal.py::test_series_equal_index_dtype[True-s11-s21-MultiIndex level \\\\\\\\\\\\\\\\[0\\\\\\\\\\\\\\\\] are different]",
    "tests/util/test_assert_series_equal.py::test_series_equal_series_type",
    "tests/util/test_hashing.py::test_hash_with_tuple[data0-result_data0]",
    "tests/util/test_hashing.py::test_hash_with_tuple[data1-result_data1]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_ewm_consistency_consistent[consistent_data1-False-False-0]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_ewm_consistency_consistent[consistent_data3-False-False-0]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_ewm_consistency_mean[all_data15-False-False-0]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_ewm_consistency_mean[all_data16-False-False-0]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_ewm_consistency_mean[all_data17-False-False-0]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_ewm_consistency_mean[all_data5-False-False-0]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_ewm_consistency_mean[all_data6-False-False-0]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_ewm_consistency_mean[all_data7-False-False-0]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_ewm_consistency_series_cov_corr[series_data5-False-False-0-True]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_ewm_consistency_series_cov_corr[series_data6-False-False-0-True]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_ewm_consistency_series_cov_corr[series_data7-False-False-0-True]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_moments_consistency_var[all_data15-False-False-0-True]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_moments_consistency_var[all_data16-False-False-0-True]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_moments_consistency_var[all_data17-False-False-0-True]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_moments_consistency_var[all_data5-False-False-0-True]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_moments_consistency_var[all_data6-False-False-0-True]",
    "tests/window/moments/test_moments_consistency_ewm.py::test_moments_consistency_var[all_data7-False-False-0-True]",
    "tests/window/moments/test_moments_consistency_expanding.py::test_expanding_apply_consistency_sum_nans[all_data1-0-sum]",
    "tests/window/moments/test_moments_consistency_expanding.py::test_expanding_apply_consistency_sum_nans[all_data5-0-sum]",
    "tests/window/moments/test_moments_consistency_expanding.py::test_expanding_apply_consistency_sum_nans[all_data5-2-sum]",
    "tests/window/moments/test_moments_consistency_expanding.py::test_expanding_apply_consistency_sum_nans[all_data6-0-sum]",
    "tests/window/moments/test_moments_consistency_expanding.py::test_expanding_apply_consistency_sum_nans[all_data6-2-sum]",
    "tests/window/moments/test_moments_consistency_expanding.py::test_expanding_apply_consistency_sum_nans[all_data7-0-sum]",
    "tests/window/moments/test_moments_consistency_expanding.py::test_expanding_apply_consistency_sum_nans[all_data7-2-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data5-rolling_consistency_cases1-False-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data5-rolling_consistency_cases1-True-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data6-rolling_consistency_cases1-False-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data6-rolling_consistency_cases1-True-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data7-rolling_consistency_cases1-False-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data7-rolling_consistency_cases1-True-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_series_cov_corr[series_data1-rolling_consistency_cases0-False-0]",
    "tests/window/test_api.py::test_rolling_max_min_periods[None]",
    "tests/window/test_api.py::test_rolling_min_min_periods[None]",
    "tests/window/test_apply.py::test_ragged_apply[engine_and_raw1]",
    "tests/window/test_apply.py::test_ragged_apply[engine_and_raw2]",
    "tests/window/test_base_indexer.py::test_bad_get_window_bounds_signature",
    "tests/window/test_base_indexer.py::test_indexers_are_reusable_after_groupby_rolling[df_data0-1-FixedForwardWindowIndexer]",
    "tests/window/test_base_indexer.py::test_indexers_are_reusable_after_groupby_rolling[df_data0-12-FixedForwardWindowIndexer]",
    "tests/window/test_base_indexer.py::test_indexers_are_reusable_after_groupby_rolling[df_data0-2-FixedForwardWindowIndexer]",
    "tests/window/test_base_indexer.py::test_indexers_are_reusable_after_groupby_rolling[df_data1-1-FixedForwardWindowIndexer]",
    "tests/window/test_base_indexer.py::test_indexers_are_reusable_after_groupby_rolling[df_data1-12-FixedForwardWindowIndexer]",
    "tests/window/test_base_indexer.py::test_indexers_are_reusable_after_groupby_rolling[df_data1-2-FixedForwardWindowIndexer]",
    "tests/window/test_base_indexer.py::test_indexers_are_reusable_after_groupby_rolling[df_data2-1-FixedForwardWindowIndexer]",
    "tests/window/test_base_indexer.py::test_indexers_are_reusable_after_groupby_rolling[df_data2-12-FixedForwardWindowIndexer]",
    "tests/window/test_base_indexer.py::test_indexers_are_reusable_after_groupby_rolling[df_data2-2-FixedForwardWindowIndexer]",
    "tests/window/test_base_indexer.py::test_rolling_forward_cov_corr[corr-expected1]",
    "tests/window/test_base_indexer.py::test_rolling_forward_cov_corr[cov-expected0]",
    "tests/window/test_dtypes.py::test_dataframe_dtypes[M8[ns]-None-max-expected_data1-None]",
    "tests/window/test_dtypes.py::test_dataframe_dtypes[M8[ns]-None-min-expected_data2-None]",
    "tests/window/test_dtypes.py::test_dataframe_dtypes[category-None-mean-expected_data4-None]",
    "tests/window/test_dtypes.py::test_dataframe_dtypes[m8[ns]-None-max-expected_data1-None]",
    "tests/window/test_dtypes.py::test_dataframe_dtypes[m8[ns]-None-mean-expected_data4-None]",
    "tests/window/test_dtypes.py::test_dataframe_dtypes[m8[ns]-None-min-expected_data2-None]",
    "tests/window/test_dtypes.py::test_dataframe_dtypes[m8[ns]-None-sum-expected_data3-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[M8[ns]-None-max-data3-expected_data3-True-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[M8[ns]-None-max-data4-expected_data4-True-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[M8[ns]-None-max-data5-expected_data5-False-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[M8[ns]-None-min-data6-expected_data6-True-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[M8[ns]-None-min-data7-expected_data7-True-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[M8[ns]-None-min-data8-expected_data8-False-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-max-data3-expected_data3-True-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-max-data4-expected_data4-True-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-max-data5-expected_data5-False-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-mean-data12-expected_data12-True-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-mean-data13-expected_data13-True-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-mean-data14-expected_data14-False-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-min-data6-expected_data6-True-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-min-data7-expected_data7-True-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-min-data8-expected_data8-False-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-sum-data10-expected_data10-True-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-sum-data11-expected_data11-False-None]",
    "tests/window/test_dtypes.py::test_series_dtypes[m8[ns]-None-sum-data9-expected_data9-True-None]",
    "tests/window/test_ewm.py::test_ewma_nan_handling_cases[s10-False-False-w10]",
    "tests/window/test_ewm.py::test_ewma_nan_handling_cases[s2-False-False-w2]",
    "tests/window/test_groupby.py::TestRolling::test_by_column_not_in_values[columns0]",
    "tests/window/test_groupby.py::TestRolling::test_by_column_not_in_values[columns1]",
    "tests/window/test_groupby.py::TestRolling::test_datelike_on_monotonic_within_each_group",
    "tests/window/test_groupby.py::TestRolling::test_getitem",
    "tests/window/test_groupby.py::TestRolling::test_getitem_multiple",
    "tests/window/test_groupby.py::TestRolling::test_groupby_monotonic",
    "tests/window/test_groupby.py::TestRolling::test_groupby_rolling_custom_indexer",
    "tests/window/test_groupby.py::TestRolling::test_groupby_rolling_no_sort",
    "tests/window/test_groupby.py::test_rolling_corr_with_single_integer_in_index",
    "tests/window/test_groupby.py::test_rolling_corr_with_tuples_in_index",
    "tests/window/test_rolling.py::test_closed_fixed_binary_col[False-10]",
    "tests/window/test_rolling.py::test_closed_fixed_binary_col[False-2]",
    "tests/window/test_rolling.py::test_closed_fixed_binary_col[False-5]",
    "tests/window/test_rolling.py::test_closed_fixed_binary_col[True-10]",
    "tests/window/test_rolling.py::test_closed_fixed_binary_col[True-2]",
    "tests/window/test_rolling.py::test_closed_fixed_binary_col[True-5]",
    "tests/window/test_rolling.py::test_rolling_non_monotonic[mean-expected1]",
    "tests/window/test_rolling.py::test_rolling_non_monotonic[sum-expected2]",
    "tests/window/test_rolling.py::test_rolling_non_monotonic[var-expected0]",
    "tests/window/test_rolling.py::test_rolling_numerical_accuracy_kahan_mean[ms-0.0]",
    "tests/window/test_rolling.py::test_rolling_numerical_accuracy_kahan_mean[ms-2.0]",
    "tests/window/test_rolling.py::test_rolling_numerical_accuracy_kahan_mean[ns-0.0]",
    "tests/window/test_rolling.py::test_rolling_numerical_accuracy_kahan_mean[ns-2.0]",
    "tests/window/test_rolling.py::test_rolling_numerical_accuracy_kahan_mean[s-0.0]",
    "tests/window/test_rolling.py::test_rolling_numerical_accuracy_kahan_mean[s-2.0]",
    "tests/window/test_rolling.py::test_rolling_numerical_accuracy_kahan_mean[us-0.0]",
    "tests/window/test_rolling.py::test_rolling_numerical_accuracy_kahan_mean[us-2.0]",
    "tests/window/test_rolling.py::test_rolling_var_same_value_count_logic[values0-3-1-expected0]",
    "tests/window/test_rolling.py::test_variable_window_nonunique[DataFrame-right-expected2]",
    "tests/window/test_rolling.py::test_variable_window_nonunique[Series-right-expected2]",
    "tests/window/test_rolling_functions.py::test_nans[max-max-kwargs4]",
    "tests/window/test_rolling_functions.py::test_nans[mean-mean-kwargs0]",
    "tests/window/test_rolling_functions.py::test_nans[min-min-kwargs3]",
    "tests/window/test_rolling_functions.py::test_rolling_max_gh6297[10]",
    "tests/window/test_rolling_functions.py::test_rolling_max_gh6297[1]",
    "tests/window/test_rolling_functions.py::test_rolling_max_gh6297[2]",
    "tests/window/test_rolling_functions.py::test_rolling_max_gh6297[5]",
    "tests/window/test_rolling_functions.py::test_rolling_max_gh6297[None]",
    "tests/window/test_rolling_functions.py::test_rolling_max_resample[10]",
    "tests/window/test_rolling_functions.py::test_rolling_max_resample[1]",
    "tests/window/test_rolling_functions.py::test_rolling_max_resample[2]",
    "tests/window/test_rolling_functions.py::test_rolling_max_resample[5]",
    "tests/window/test_rolling_functions.py::test_rolling_max_resample[None]",
    "tests/window/test_rolling_functions.py::test_rolling_median_resample",
    "tests/window/test_rolling_functions.py::test_rolling_min_resample[10]",
    "tests/window/test_rolling_functions.py::test_rolling_min_resample[1]",
    "tests/window/test_rolling_functions.py::test_rolling_min_resample[2]",
    "tests/window/test_rolling_functions.py::test_rolling_min_resample[5]",
    "tests/window/test_rolling_functions.py::test_rolling_min_resample[None]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_closed[ms]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_closed[s]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_closed[us]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_invalid_minp[1.0]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_invalid_minp[foo]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_invalid_minp[minp2]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_invalid_window_nonfixed[2MS]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_non_monotonic_on",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_ragged_kurt",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_ragged_skew",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_ragged_std",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_ragged_var",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_rolling_on_decreasing_index[ms]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_rolling_on_decreasing_index[ns]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_rolling_on_decreasing_index[s]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_rolling_on_decreasing_index[us]",
    "tests/window/test_timeseries_window.py::TestRollingTS::test_rolling_on_empty",
    "tests/window/test_win_type.py::test_cmov_window_corner[None]",
    "tests/window/test_win_type.py::test_invalid_scipy_arg",
    "tests/window/test_win_type.py::test_win_type_not_implemented",
    'tests/io/json/test_pandas.py::TestPandasContainer::test_json_indent_all_orients[table-{\\n    "schema":{\\n        "fields":[\\n            {\\n                "name":"index",\\n                "type":"integer"\\n            },\\n            {\\n                "name":"a",\\n                "type":"string"\\n            },\\n            {\\n                "name":"b",\\n                "type":"string"\\n            }\\n        ],\\n        "primaryKey":[\\n            "index"\\n        ],\\n        "pandas_version":"1.4.0"\\n    },\\n    "data":[\\n        {\\n            "index":0,\\n            "a":"foo",\\n            "b":"bar"\\n        },\\n        {\\n            "index":1,\\n            "a":"baz",\\n            "b":"qux"\\n        }\\n    ]\\n}]',
    r"tests/io/parser/common/test_index.py::test_multi_index_blank_df[c_high-True-a,b\nc,d-expected1-header1]",
    r"tests/io/parser/common/test_index.py::test_multi_index_blank_df[c_low-True-a,b\nc,d-expected1-header1]",
    r"tests/io/parser/common/test_index.py::test_multi_index_blank_df[python-True-a,b\nc,d-expected1-header1]",
    r"tests/io/parser/test_parse_dates.py::test_parse_date_time[c_high-KORD,19990127, 19:00:00, 18:56:00, 0.8100\nKORD,19990127, 20:00:00, 19:56:00, 0.0100\nKORD,19990127, 21:00:00, 20:56:00, -0.5900\nKORD,19990127, 21:00:00, 21:18:00, -0.9900\nKORD,19990127, 22:00:00, 21:56:00, -0.5900\nKORD,19990127, 23:00:00, 22:56:00, -0.5900-kwargs1-expected1]",
    r"tests/io/parser/test_parse_dates.py::test_parse_date_time[c_high-date,time,a,b\n2001-01-05, 10:00:00, 0.0, 10.\n2001-01-05, 00:00:00, 1., 11.\n-kwargs0-expected0]",
    r"tests/io/parser/test_parse_dates.py::test_parse_date_time[c_low-KORD,19990127, 19:00:00, 18:56:00, 0.8100\nKORD,19990127, 20:00:00, 19:56:00, 0.0100\nKORD,19990127, 21:00:00, 20:56:00, -0.5900\nKORD,19990127, 21:00:00, 21:18:00, -0.9900\nKORD,19990127, 22:00:00, 21:56:00, -0.5900\nKORD,19990127, 23:00:00, 22:56:00, -0.5900-kwargs1-expected1]",
    r"tests/io/parser/test_parse_dates.py::test_parse_date_time[c_low-date,time,a,b\n2001-01-05, 10:00:00, 0.0, 10.\n2001-01-05, 00:00:00, 1., 11.\n-kwargs0-expected0]",
    r"tests/io/parser/test_parse_dates.py::test_parse_date_time[python-KORD,19990127, 19:00:00, 18:56:00, 0.8100\nKORD,19990127, 20:00:00, 19:56:00, 0.0100\nKORD,19990127, 21:00:00, 20:56:00, -0.5900\nKORD,19990127, 21:00:00, 21:18:00, -0.9900\nKORD,19990127, 22:00:00, 21:56:00, -0.5900\nKORD,19990127, 23:00:00, 22:56:00, -0.5900-kwargs1-expected1]",
    r"tests/io/parser/test_parse_dates.py::test_parse_date_time[python-date,time,a,b\n2001-01-05, 10:00:00, 0.0, 10.\n2001-01-05, 00:00:00, 1., 11.\n-kwargs0-expected0]",
    r"tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_high-a,b\n04.15.2016,09.16.2013-kwargs2-expected2]",
    r"tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_high-a\n04.15.2016-kwargs0-expected0]",
    r"tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_low-a,b\n04.15.2016,09.16.2013-kwargs2-expected2]",
    r"tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[c_low-a\n04.15.2016-kwargs0-expected0]",
    r"tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[python-a,b\n04.15.2016,09.16.2013-kwargs2-expected2]",
    r"tests/io/parser/test_parse_dates.py::test_parse_dates_no_convert_thousands[python-a\n04.15.2016-kwargs0-expected0]",
    r"tests/util/test_assert_frame_equal.py::test_frame_equal_index_dtype_mismatch[True-df11-df21-MultiIndex level \\[0\\] are different]",
    r"tests/util/test_assert_series_equal.py::test_series_equal_index_dtype[True-s11-s21-MultiIndex level \\[0\\] are different]",
}

NODEIDS_THAT_FAIL_WITH_CUDF_PANDAS_WITH_REASON: dict[str, set[str]] = {
    "https://github.com/rapidsai/cudf/issues/18659": {
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ms]-__add__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ms]-__radd__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ms]-__rsub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ms]-__sub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[s]-__add__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[s]-__radd__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[s]-__rsub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[s]-__sub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[s]-__sub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[s]-__rsub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ms]-__sub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ms]-__rsub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint8-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint8-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint16-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint16-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint32-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint32-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint64-__add__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint64-__radd__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint64-__sub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint64-__rsub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint64-__mul__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint64-__rmul__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint64-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint64-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint64-__pow__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[int8-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[int8-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[int16-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[int16-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[int32-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[int32-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[int64-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint8-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint8-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint16-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint16-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint32-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint32-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__add__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__radd__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__sub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__rsub__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__mul__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__rmul__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__pow__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int8-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int8-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int16-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int16-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int32-__floordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int32-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int64-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint8-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint16-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint32-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint64-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__rfloordiv__]",
        "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__rfloordiv__]",
    },
    "https://github.com/rapidsai/cudf/issues/18659#issuecomment-3710985854": {
        "tests/extension/test_arrow.py::test_pickle_roundtrip[string]",
        "tests/arrays/string_/test_string_arrow.py::test_pickle_roundtrip[na_value0]",
        "tests/arrays/string_/test_string_arrow.py::test_pickle_roundtrip[nan]",
    },
}

NODEIDS_THAT_XPASS_WITH_CUDF_PANDAS = {
    "tests/arrays/string_/test_string.py::test_add_2d[pyarrow]",
    "tests/arrays/string_/test_string.py::test_add_2d[pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_add_2d[string=str[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_add_2d[string=string[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_reduce[pyarrow-False]",
    "tests/arrays/string_/test_string.py::test_reduce[pyarrow-True]",
    "tests/arrays/string_/test_string.py::test_reduce[pyarrow_numpy-False]",
    "tests/arrays/string_/test_string.py::test_reduce[pyarrow_numpy-True]",
    "tests/arrays/string_/test_string.py::test_reduce[python-False]",
    "tests/arrays/string_/test_string.py::test_reduce[python-True]",
    "tests/arrays/string_/test_string.py::test_reduce_missing[pyarrow-False]",
    "tests/arrays/string_/test_string.py::test_reduce_missing[pyarrow-True]",
    "tests/arrays/string_/test_string.py::test_reduce_missing[pyarrow_numpy-False]",
    "tests/arrays/string_/test_string.py::test_reduce_missing[pyarrow_numpy-True]",
    "tests/arrays/string_/test_string.py::test_reduce_missing[python-False]",
    "tests/arrays/string_/test_string.py::test_reduce_missing[python-True]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrame-DataFrame-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrame-DataFrame-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrame-DataFrameNaN-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrame-DataFrameNaN-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrame-Series-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrame-Series-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrame-SeriesNaN-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrame-SeriesNaN-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrame-float-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrame-float-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrameNaN-DataFrame-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrameNaN-DataFrame-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrameNaN-DataFrameNaN-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrameNaN-DataFrameNaN-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrameNaN-Series-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrameNaN-Series-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrameNaN-SeriesNaN-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrameNaN-SeriesNaN-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrameNaN-float-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[DataFrameNaN-float-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[Series-DataFrame-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[Series-DataFrame-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[Series-DataFrameNaN-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[Series-DataFrameNaN-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[Series-Series-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[Series-Series-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[Series-SeriesNaN-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[Series-SeriesNaN-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[Series-float-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[Series-float-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[SeriesNaN-DataFrame-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[SeriesNaN-DataFrame-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[SeriesNaN-DataFrameNaN-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[SeriesNaN-DataFrameNaN-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[SeriesNaN-Series-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[SeriesNaN-Series-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[SeriesNaN-SeriesNaN-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[SeriesNaN-SeriesNaN-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[SeriesNaN-float-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[SeriesNaN-float-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[float-DataFrame-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[float-DataFrame-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[float-DataFrameNaN-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[float-DataFrameNaN-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[float-Series-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[float-Series-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[float-SeriesNaN-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[float-SeriesNaN-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[float-float-numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_floor_division[float-float-numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_frame_invert[numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_frame_invert[numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_series_invert[numexpr-pandas]",
    "tests/computation/test_eval.py::TestEval::test_series_invert[numexpr-python]",
    "tests/computation/test_eval.py::test_eval_no_support_column_name[Inf]",
    "tests/computation/test_eval.py::test_eval_no_support_column_name[inf]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_getitem_series_integer_with_missing_raises[integer-array]",
    "tests/extension/json/test_json.py::TestJSONArray::test_getitem_series_integer_with_missing_raises[integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cumsum-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cumsum-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_str[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_str[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_str[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_str[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[binary-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[bool-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[date32[day]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[date64[ms]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[decimal128(7, 3)-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[double-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[duration[ms]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[duration[ns]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[duration[s]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[duration[us]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[float-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[int16-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[int32-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[int64-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[int8-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[string-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[time32[ms]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[time32[s]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[time64[ns]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[time64[us]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[ms, tz=US/Eastern]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[ms, tz=US/Pacific]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[ms, tz=UTC]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[ms]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[ns, tz=US/Eastern]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[ns, tz=US/Pacific]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[ns, tz=UTC]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[ns]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[s, tz=US/Eastern]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[s, tz=US/Pacific]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[s, tz=UTC]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[s]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[us, tz=US/Eastern]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[us, tz=US/Pacific]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[us, tz=UTC]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[timestamp[us]-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[uint16-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[uint32-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[uint64-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_series_integer_with_missing_raises[uint8-integer-array]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[bool-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[bool-median-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[bool-median-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[bool-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[bool-std-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[bool-std-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[bool-var-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[bool-var-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[decimal128(7, 3)-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[decimal128(7, 3)-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[double-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[double-kurt-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[double-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[double-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[float-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[float-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[float-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int16-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int16-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int16-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int32-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int32-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int32-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int64-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int64-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int64-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int8-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int8-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[int8-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint16-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint16-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint16-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint32-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint32-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint32-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint64-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint64-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint64-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint8-kurt-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint8-skew-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_numeric[uint8-skew-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_with_index[date64[ms]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[date64[ms]]",
    "tests/extension/test_categorical.py::TestCategorical::test_getitem_series_integer_with_missing_raises[integer-array]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_getitem_series_integer_with_missing_raises[US/Central-integer-array]",
    "tests/extension/test_interval.py::TestIntervalArray::test_getitem_series_integer_with_missing_raises[integer-array]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_getitem_series_integer_with_missing_raises[BooleanDtype-integer-array]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_getitem_series_integer_with_missing_raises[Float32Dtype-integer-array]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_getitem_series_integer_with_missing_raises[Float64Dtype-integer-array]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_getitem_series_integer_with_missing_raises[Int16Dtype-integer-array]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_getitem_series_integer_with_missing_raises[Int32Dtype-integer-array]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_getitem_series_integer_with_missing_raises[Int64Dtype-integer-array]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_getitem_series_integer_with_missing_raises[Int8Dtype-integer-array]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_getitem_series_integer_with_missing_raises[UInt16Dtype-integer-array]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_getitem_series_integer_with_missing_raises[UInt32Dtype-integer-array]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_getitem_series_integer_with_missing_raises[UInt64Dtype-integer-array]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_getitem_series_integer_with_missing_raises[UInt8Dtype-integer-array]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_getitem_series_integer_with_missing_raises[float-integer-array]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_getitem_series_integer_with_missing_raises[object-integer-array]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_integer_array[object-True-list]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_integer_array[object-True-numpy-array]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_loc_iloc_slice[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_mask[object-True-numpy-array]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_mask_broadcast[object-None]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_mask_broadcast[object-loc]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_sequence_broadcasts[object-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_slice[object-True]",
    "tests/extension/test_period.py::TestPeriodArray::test_getitem_series_integer_with_missing_raises[2D-integer-array]",
    "tests/extension/test_period.py::TestPeriodArray::test_getitem_series_integer_with_missing_raises[D-integer-array]",
    "tests/extension/test_sparse.py::TestSparseArray::test_fillna_length_mismatch[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_getitem_series_integer_with_missing_raises[0-integer-array]",
    "tests/extension/test_sparse.py::TestSparseArray::test_getitem_series_integer_with_missing_raises[nan-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[pyarrow-False-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[pyarrow-True-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[pyarrow_numpy-False-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[pyarrow_numpy-True-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[python-False-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[python-True-integer-array]",
    "tests/frame/methods/test_info.py::test_info_memory_usage_deep_pypy",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_datetime[DataFrame-list-datetime]",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_datetime[Series-dict-datetime]",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_datetime[Series-list-datetime]",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_timedelta[DataFrame-list-timedelta]",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_timedelta[Series-dict-timedelta]",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_timedelta[Series-list-timedelta]",
    "tests/frame/test_query_eval.py::TestDataFrameEval::test_extension_array_eval[numexpr-pandas]",
    "tests/frame/test_query_eval.py::TestDataFrameEval::test_extension_array_eval[numexpr-python]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_preserve_dtypes",
    "tests/generic/test_finalize.py::test_categorical_accessor[method0]",
    "tests/generic/test_finalize.py::test_categorical_accessor[method1]",
    "tests/generic/test_finalize.py::test_categorical_accessor[method2]",
    "tests/generic/test_finalize.py::test_categorical_accessor[method4]",
    "tests/generic/test_finalize.py::test_categorical_accessor[method7]",
    "tests/generic/test_finalize.py::test_categorical_accessor[method8]",
    "tests/groupby/test_groupby.py::test_ops_not_as_index[idxmin]",
    "tests/groupby/test_numeric_only.py::test_axis1_numeric_only[corrwith-False]",
    "tests/groupby/test_numeric_only.py::test_axis1_numeric_only[corrwith-None]",
    "tests/groupby/test_numeric_only.py::test_axis1_numeric_only[skew-False]",
    "tests/groupby/test_numeric_only.py::test_axis1_numeric_only[skew-None]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by3-False-cumsum-agg]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by3-True-fillna-method]",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_dti_constructor_with_non_nano_now_today",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[both-breaks3]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[left-breaks3]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[neither-breaks3]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[right-breaks3]",
    "tests/indexing/test_loc.py::TestLocBaseIndependent::test_loc_copy_vs_view",
    "tests/io/json/test_pandas.py::TestPandasContainer::test_json_pandas_nulls[Decimal]",
    "tests/io/json/test_readlines.py::test_readjson_unicode[pyarrow]",
    "tests/io/parser/test_unsupported.py::test_invalid_file_inputs[python]",
    "tests/plotting/test_datetimelike.py::TestTSPlot::test_add_matplotlib_datetime64",
    "tests/series/methods/test_info.py::test_info_memory_usage_deep_pypy",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_dtype_timedelta_ns_s_astype_int64",
    "tests/series/test_logical_ops.py::TestSeriesLogicalOps::test_int_dtype_different_index_not_bool",
    "tests/strings/test_api.py::test_api_per_method[index-empty1-partition1-category]",
    "tests/strings/test_api.py::test_api_per_method[index-empty1-partition1-object]",
    "tests/strings/test_api.py::test_api_per_method[index-empty1-partition2-category]",
    "tests/strings/test_api.py::test_api_per_method[index-empty1-partition2-object]",
    "tests/strings/test_api.py::test_api_per_method[index-empty1-rpartition1-category]",
    "tests/strings/test_api.py::test_api_per_method[index-empty1-rpartition1-object]",
    "tests/strings/test_api.py::test_api_per_method[index-empty1-rpartition2-category]",
    "tests/strings/test_api.py::test_api_per_method[index-empty1-rpartition2-object]",
    "tests/strings/test_api.py::test_api_per_method[index-empty1-split1-category]",
    "tests/strings/test_api.py::test_api_per_method[index-empty1-split1-object]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data1-rolling_consistency_cases0-False-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data1-rolling_consistency_cases0-True-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data11-rolling_consistency_cases0-False-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data11-rolling_consistency_cases0-True-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data15-rolling_consistency_cases0-False-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data15-rolling_consistency_cases0-True-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data16-rolling_consistency_cases0-False-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data16-rolling_consistency_cases0-True-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data17-rolling_consistency_cases0-False-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data17-rolling_consistency_cases0-True-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data5-rolling_consistency_cases0-False-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data5-rolling_consistency_cases0-True-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data6-rolling_consistency_cases0-False-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data6-rolling_consistency_cases0-True-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_apply_consistency_sum[all_data7-rolling_consistency_cases0-False-sum]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data1-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data1-rolling_consistency_cases0-False-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data5-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data5-rolling_consistency_cases0-False-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data6-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data6-rolling_consistency_cases0-False-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data7-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data7-rolling_consistency_cases0-False-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data11-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data11-rolling_consistency_cases0-False-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data15-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data15-rolling_consistency_cases0-False-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data16-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data16-rolling_consistency_cases0-False-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data17-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_moments_consistency_var[all_data17-rolling_consistency_cases0-False-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_series_cov_corr[series_data1-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_series_cov_corr[series_data5-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_series_cov_corr[series_data5-rolling_consistency_cases0-False-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_series_cov_corr[series_data6-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_series_cov_corr[series_data7-rolling_consistency_cases0-True-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_series_cov_corr[series_data6-rolling_consistency_cases0-False-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_series_cov_corr[series_data7-rolling_consistency_cases0-False-0]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data1-rolling_consistency_cases0-True]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data1-rolling_consistency_cases0-False]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data5-rolling_consistency_cases0-True]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data5-rolling_consistency_cases0-False]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data6-rolling_consistency_cases0-True]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data6-rolling_consistency_cases0-False]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data7-rolling_consistency_cases0-True]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data7-rolling_consistency_cases0-False]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data11-rolling_consistency_cases0-True]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data11-rolling_consistency_cases0-False]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data15-rolling_consistency_cases0-True]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data15-rolling_consistency_cases0-False]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data16-rolling_consistency_cases0-True]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data16-rolling_consistency_cases0-False]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data17-rolling_consistency_cases0-True]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_mean[all_data17-rolling_consistency_cases0-False]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_constant[consistent_data1-rolling_consistency_cases0-True]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_constant[consistent_data1-rolling_consistency_cases0-False]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_constant[consistent_data3-rolling_consistency_cases0-True]",
    "tests/window/moments/test_moments_consistency_rolling.py::test_rolling_consistency_constant[consistent_data3-rolling_consistency_cases0-False]",
    "tests/window/test_rolling.py::test_rolling_mean_all_nan_window_floating_artifacts[1-exp_values0]",
    "tests/window/test_rolling.py::test_rolling_mean_all_nan_window_floating_artifacts[2-exp_values1]",
}

# TODO: Investigate why sometimes these fail
NODEIDS_THAT_FLAKY_XFAIL_WITH_CUDF_PANDAS = {
    "tests/groupby/test_groupby.py::test_single_element_list_grouping[selection2]",
    "tests/frame/methods/test_sort_index.py::TestDataFrameSortIndex::test_sort_index_inplace",
    "tests/frame/methods/test_sort_index.py::TestDataFrameSortIndex::test_sort_index",
    "tests/reshape/test_crosstab.py::test_categoricals[category-category]",
    "tests/reshape/test_crosstab.py::test_categoricals[category-int64]",
    "tests/reshape/test_crosstab.py::test_categoricals[int64-category]",
    "tests/reshape/test_crosstab.py::test_categoricals[int64-int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint8-__rpow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint16-__rpow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint32-__rpow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint8-__rpow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint16-__rpow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint32-__rpow__]",
    r"tests/tools/test_to_datetime.py::TestToDatetimeMisc::test_to_datetime_iso8601_fails[True-2012-01-01 10:00-%Y-%m-%d %H:%M:%S]",
    "tests/tools/test_to_datetime.py::TestToDatetimeMisc::test_to_datetime_iso8601_fails[True-2012-01-01-%Y-%m-%d %H]",
    "tests/tools/test_to_datetime.py::TestToDatetimeMisc::test_to_datetime_iso8601_fails[True-2012-01-01 10-%Y-%m-%d %H:%M]",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_warnings_errors",
    "tests/indexes/multi/test_indexing.py::test_pyint_engine",
    "tests/indexes/multi/test_monotonic.py::test_is_monotonic_with_nans[values0-is_monotonic_increasing]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_sub_timedeltalike[timedelta64_0-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_sub_timedeltalike[timedelta64_1-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[string=str[pyarrow]-False-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[string=str[pyarrow]-True-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[string=str[python]-False-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[string=str[python]-True-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[string=string[pyarrow]-False-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[string=string[pyarrow]-True-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[string=string[python]-False-integer-array]",
    "tests/extension/test_string.py::TestStringArray::test_getitem_series_integer_with_missing_raises[string=string[python]-True-integer-array]",
    "tests/indexing/test_indexing.py::TestMisc::test_no_reference_cycle",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_iloc_ea_series_indexer_with_na",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_type_error_multiindex",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_slice_month",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-nullable_int]",
    "tests/indexes/test_setops.py::test_union_different_types[categorical-nullable_int]",
    "tests/indexes/test_setops.py::test_union_different_types[nullable_int-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[nullable_int-categorical]",
    "tests/indexes/test_setops.py::test_union_different_types[nullable_int-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-nullable_int]",
    "tests/indexing/test_loc.py::TestLocBooleanLabelsAndSlices::test_loc_bool_slice_raises[nullable_int-DataFrame]",
    "tests/indexing/test_loc.py::TestLocBooleanLabelsAndSlices::test_loc_bool_slice_raises[nullable_int-Series]",
    "tests/indexing/test_loc.py::TestLocSetitemWithExpansion::test_loc_setitem_with_expansion_nonunique_index[nullable_int]",
    "tests/reshape/merge/test_multi.py::TestMergeMulti::test_merge_na_keys",
    "tests/series/test_constructors.py::TestSeriesConstructors::test_constructor_maskedarray_hardened",
    # We accelerate np.dot via cupy.dot when the proxy array inputs are backed by real
    # cupy arrays. So it is possible for tests asserting equality like test_dot_2d_ndarray
    # fail due to tiny FP differences.
    "tests/frame/methods/test_dot.py::TestSeriesDot::test_dot_2d_ndarray",
    # TODO: These tests pass when run locally: with pandas 2.3.3, numpy 2.2.6, and cupy 13.6.0
    # bash python/cudf/cudf/pandas/scripts/run-pandas-tests.sh
    #      -q tests/indexes/test_common.py::test_sort_values_with_missing[nullable_int-first]
    # Investigate why.
    "tests/indexes/test_common.py::test_sort_values_with_missing[nullable_int-first]",
    "tests/indexes/test_common.py::test_sort_values_with_missing[nullable_int-last]",
}

# TODO: Ideally cudf.pandas should be able to pass assertions with private APIs,
# e.g. `obj._private is other_obj`, but tests with public APIs are more important.
NODEIDS_THAT_ASSERT_PRIVATE_APIS = {
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-0-False-mean-columns]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-0-False-mean-index]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-0-True-mean-columns]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-0-True-mean-index]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-1-False-mean-columns]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-1-False-mean-index]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-1-True-mean-columns]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-1-True-mean-index]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-bfill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-cummax]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-cummin]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-cumprod]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-cumsum]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-diff]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-ffill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-fillna]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-pct_change]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-rank]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-shift]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-bfill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-cummax]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-cummin]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-cumprod]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-cumsum]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-diff]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-ffill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-fillna]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-pct_change]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-rank]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-shift]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-bfill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-cummax]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-cummin]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-cumprod]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-cumsum]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-diff]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-ffill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-fillna]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-pct_change]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-rank]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-shift]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-bfill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-cummax]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-cummin]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-cumprod]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-cumsum]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-diff]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-ffill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-fillna]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-pct_change]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-rank]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-shift]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64OverflowHandling::test_datetimeindex_sub_timestamp_overflow",
    "tests/arithmetic/test_datetime64.py::TestDatetime64OverflowHandling::test_datetimeindex_sub_datetimeindex_overflow",
    "tests/arrays/boolean/test_construction.py::test_boolean_array_constructor_copy",
    "tests/arrays/boolean/test_construction.py::test_coerce_to_array",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[binary]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[date32[day]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[time32[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[time32[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[time64[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[time64[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[uint8]",
    "tests/extension/test_categorical.py::TestCategorical::test_series_constructor",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_series_constructor[US/Central]",
    "tests/extension/test_interval.py::TestIntervalArray::test_series_constructor",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[UInt8Dtype]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_series_constructor[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_series_constructor[object]",
    "tests/extension/test_period.py::TestPeriodArray::test_series_constructor[2D]",
    "tests/extension/test_period.py::TestPeriodArray::test_series_constructor[D]",
    "tests/extension/test_sparse.py::TestSparseArray::test_series_constructor[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_series_constructor[nan]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=string[python]-True]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[None-None-False]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[None-None-True]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[None-int64-False]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[None-int64-True]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[idx1-None-False]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[idx1-None-True]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[idx1-int64-False]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[idx1-int64-True]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[a-0]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[a-1]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[a-2]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[a-3]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[a-4]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[b-0]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[b-1]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[b-2]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[b-3]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[b-4]",
    "tests/indexing/multiindex/test_partial.py::TestMultiIndexPartial::test_getitem_intkey_leading_level[float]",
    "tests/indexing/multiindex/test_partial.py::TestMultiIndexPartial::test_getitem_intkey_leading_level[int]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index0]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index13]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index1]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index2]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index3]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index4]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index5]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index6]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index7]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index8]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[bool-dtype]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[complex128]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[complex64]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[datetime-tz]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[datetime]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[empty]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[float32]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[float64]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[int16]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[int32]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[int64]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[int8]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[nullable_bool]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[nullable_float]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[nullable_int]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[nullable_uint]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[object]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[period]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[repeats]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[string-pyarrow]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[string-python]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[string]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[timedelta]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[uint16]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[uint32]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[uint64]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[uint8]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index13]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index1]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index2]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index3]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index5]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index6]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index7]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index8]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float32-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float32-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float64-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float64-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int16-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int16-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int32-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int32-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int64-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int64-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int8-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int8-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint16-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint16-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint32-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint32-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint64-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint64-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint8-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint8-Interval]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['+01:15'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['+01:15'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['+01:15'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['-02:15'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['-02:15'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['-02:15'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['Asia/Tokyo'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['Asia/Tokyo'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['Asia/Tokyo'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['US/Eastern'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['US/Eastern'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['US/Eastern'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC+01:15'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC+01:15'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC+01:15'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC-02:15'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC-02:15'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC-02:15'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/Asia/Singapore'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/Asia/Singapore'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/Asia/Singapore'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/US/Pacific'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/US/Pacific'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/US/Pacific'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[<UTC>-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[<UTC>-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[<UTC>-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[None-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[None-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[None-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone.utc-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone.utc-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone.utc-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(-300)-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(-300)-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(-300)-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(300)-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(300)-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(300)-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzlocal()-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzlocal()-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzlocal()-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzutc()-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzutc()-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzutc()-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='UTC')-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='UTC')-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='UTC')-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(ke