# Copyright (c) 2025, NVIDIA CORPORATION.
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This example trains a random forest model on the embeddings generated by
# the "bitcoin_mnmg.py" script.

import argparse

from torch_geometric.datasets import EllipticBitcoinDataset

import cudf
import cupy

from cuml.metrics import confusion_matrix, roc_auc_score
from cuml.ensemble import RandomForestClassifier


def train_rf(X_train, y_train):
    rf = RandomForestClassifier()
    rf.fit(X_train, y_train)
    return rf


def show_confusion_matrix(y_test, prob, name):
    cm = confusion_matrix(y_test, prob.argmax(axis=1))
    acc = (y_test == prob.argmax(axis=1)).sum() / len(y_test)
    auc = roc_auc_score(y_test, prob[:, 1])
    print(
        f"=== {name} ===\n"
        f"Confusion Matrix:\n{cm}\n"
        f"Accuracy: {acc}\n"
        f"ROC AUC: {auc}\n"
    )


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--dataset_root", type=str, default="./data")
    parser.add_argument("--embedding_dir", type=str, default="./results")
    args = parser.parse_args()

    dataset = EllipticBitcoinDataset(root=args.dataset_root)
    data = dataset[0]
    data.x = data.x[:, :94]  # Remove pre-generated embeddings
    assert dataset.num_classes == 2

    df = cudf.read_parquet(args.embedding_dir)
    y = df["y"].to_cupy()
    z = df["z"].to_cupy()
    X = df.drop(columns=["y", "z"]).to_cupy()

    X_train = X[data.train_mask]
    X_test = X[data.test_mask]
    y_train = y[data.train_mask]
    y_test = y[data.test_mask]
    z_train = z[data.train_mask]
    z_test = z[data.test_mask]

    rf_with_gnn = train_rf(X_train, y_train)
    show_confusion_matrix(y_test, rf_with_gnn.predict_proba(X_test), "RF with GNN")

    zz = cupy.stack([z_test, 1 - z_test], axis=1)
    show_confusion_matrix(y_test, zz, "GNN Only")

    rf_only = train_rf(
        cupy.asarray(data.x[data.train_mask].cuda()),
        cupy.asarray(data.y[data.train_mask].cuda()),
    )
    show_confusion_matrix(
        y_test,
        rf_only.predict_proba(cupy.asarray(data.x[data.test_mask].cuda())),
        "RF Only",
    )
