# SPDX-FileCopyrightText: Copyright (c) 2023-2025, NVIDIA CORPORATION & AFFILIATES.
# All rights reserved.
# SPDX-License-Identifier: Apache-2.0

import contextlib
import json
import os
import sys
import traceback
from collections import defaultdict
from functools import wraps

import pytest


def replace_kwargs(new_kwargs):
    def wrapper(func):
        @wraps(func)
        def wrapped(*args, **kwargs):
            kwargs.update(new_kwargs)
            return func(*args, **kwargs)

        return wrapped

    return wrapper


@contextlib.contextmanager
def null_assert_warnings(*args, **kwargs):
    try:
        yield []
    finally:
        pass


@pytest.fixture(scope="session", autouse=True)  # type: ignore
def patch_testing_functions():
    tm.assert_produces_warning = null_assert_warnings  # noqa: F821
    pytest.raises = replace_kwargs({"match": None})(pytest.raises)


# Dictionary to store function call counts
function_call_counts = defaultdict(lambda: defaultdict(int))  # type: ignore

# The specific functions to track
FUNCTION_NAME = {"_slow_function_call", "_fast_function_call"}


def find_pytest_file(frame) -> str | None:
    for frame in traceback.extract_stack():
        file = frame.filename
        if "pandas-testing/pandas-tests/tests" in file and file.rsplit("/", 1)[
            -1
        ].startswith("test_"):
            return str(file).rsplit("pandas-tests/", 1)[-1]
    return None


def trace_calls(frame, event, arg):
    if (
        event == "call"
        and (func_name := frame.f_code.co_name) in FUNCTION_NAME
        and (filename := find_pytest_file(frame)) is not None
    ):
        function_call_counts[filename][func_name] += 1


def pytest_sessionstart(session):
    # Set the profile function to trace calls
    sys.setprofile(trace_calls)


def pytest_sessionfinish(session, exitstatus):
    # Remove the profile function
    sys.setprofile(None)


@pytest.hookimpl(trylast=True)
def pytest_unconfigure(config):
    if hasattr(config, "workerinput"):
        # Running in xdist worker, write the counts before exiting
        worker_id = config.workerinput["workerid"]
        output_file = f"function_call_counts_worker_{worker_id}.json"
        with open(output_file, "w") as f:
            json.dump(function_call_counts, f, indent=4)
        print(f"Function call counts have been written to {output_file}")  # noqa: T201


# TODO: Pass these tests with cudf.pandas enabled.
NODEIDS_THAT_FAIL_WITH_CUDF_PANDAS = {
    "tests/arrays/categorical/test_analytics.py::TestCategoricalAnalytics::test_searchsorted[False]",
    "tests/arrays/categorical/test_analytics.py::TestCategoricalAnalytics::test_searchsorted[None]",
    "tests/arrays/categorical/test_analytics.py::TestCategoricalAnalytics::test_searchsorted[True]",
    "tests/arrays/test_datetimelike.py::test_searchsorted_datetimelike_with_listlike_invalid_dtype[arg0-values1]",
    "tests/extension/test_categorical.py::TestCategorical::test_searchsorted[True]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_nearest_decreasing[backfill-expected1]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_nearest_decreasing[nearest-expected2]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_nearest_decreasing[pad-expected0]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_nearest_error",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_with_method_numeric_vs_bool[backfill]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_with_method_numeric_vs_bool[nearest]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_with_method_numeric_vs_bool[pad]",
    "tests/indexes/ranges/test_indexing.py::TestGetIndexer::test_get_indexer_limit",
    "tests/indexes/timedeltas/test_searchsorted.py::TestSearchSorted::test_searchsorted_invalid_argument_dtype[arg0]",
    "tests/indexing/test_loc.py::TestLocBaseIndependent::test_loc_setitem_frame_with_inverted_slice",
    "tests/api/test_api.py::test_pandas_array_alias",
    "tests/apply/test_frame_apply.py::test_agg_transform[axis='columns']",
    "tests/apply/test_frame_apply.py::test_agg_transform[axis='index']",
    "tests/apply/test_frame_apply.py::test_agg_transform[axis=0]",
    "tests/apply/test_frame_apply.py::test_agg_transform[axis=1]",
    "tests/apply/test_frame_apply.py::test_apply[python]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-0-False-mean-columns]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-0-False-mean-index]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-0-True-mean-columns]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-0-True-mean-index]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-1-False-mean-columns]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-1-False-mean-index]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-1-True-mean-columns]",
    "tests/apply/test_frame_apply.py::test_apply_empty_infer_type[python-1-True-mean-index]",
    "tests/apply/test_frame_apply.py::test_apply_function_runs_once",
    "tests/apply/test_frame_apply.py::test_apply_raw_function_runs_once[python]",
    "tests/apply/test_frame_apply.py::test_nuiscance_columns",
    "tests/apply/test_frame_apply.py::test_nunique_empty",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis='columns'-ops0-names0]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis='columns'-ops1-names1]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis='columns'-ops2-names2]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis='columns'-ops3-names3]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis='index'-ops0-names0]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis='index'-ops1-names1]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis='index'-ops2-names2]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis='index'-ops3-names3]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis=0-ops0-names0]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis=0-ops1-names1]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis=0-ops2-names2]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis=0-ops3-names3]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis=1-ops0-names0]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis=1-ops1-names1]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis=1-ops2-names2]",
    "tests/apply/test_frame_transform.py::test_transform_listlike[axis=1-ops3-names3]",
    "tests/apply/test_frame_transform.py::test_transform_ufunc[axis='columns'-DataFrame]",
    "tests/apply/test_frame_transform.py::test_transform_ufunc[axis='index'-DataFrame]",
    "tests/apply/test_frame_transform.py::test_transform_ufunc[axis=0-DataFrame]",
    "tests/apply/test_frame_transform.py::test_transform_ufunc[axis=1-DataFrame]",
    "tests/apply/test_invalid_arg.py::test_map_arg_is_dict_with_invalid_na_action_raises[True]",
    "tests/apply/test_invalid_arg.py::test_map_arg_is_dict_with_invalid_na_action_raises[____]",
    "tests/apply/test_invalid_arg.py::test_map_with_invalid_na_action_raises",
    "tests/apply/test_series_apply.py::test_apply[False]",
    "tests/apply/test_series_apply.py::test_apply[compat]",
    "tests/apply/test_str.py::test_apply_np_transformer[apply-abs]",
    "tests/apply/test_str.py::test_apply_np_transformer[apply-ceil]",
    "tests/apply/test_str.py::test_apply_np_transformer[apply-cos]",
    "tests/apply/test_str.py::test_apply_np_transformer[apply-exp]",
    "tests/apply/test_str.py::test_apply_np_transformer[apply-log]",
    "tests/apply/test_str.py::test_apply_np_transformer[apply-sqrt]",
    "tests/apply/test_str.py::test_apply_np_transformer[apply-square]",
    "tests/apply/test_str.py::test_apply_np_transformer[transform-abs]",
    "tests/apply/test_str.py::test_apply_np_transformer[transform-ceil]",
    "tests/apply/test_str.py::test_apply_np_transformer[transform-cos]",
    "tests/apply/test_str.py::test_apply_np_transformer[transform-exp]",
    "tests/apply/test_str.py::test_apply_np_transformer[transform-log]",
    "tests/apply/test_str.py::test_apply_np_transformer[transform-sqrt]",
    "tests/apply/test_str.py::test_apply_np_transformer[transform-square]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-bfill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-cummax]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-cummin]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-cumprod]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-cumsum]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-diff]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-ffill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-fillna]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-pct_change]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-rank]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='columns'-shift]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-bfill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-cummax]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-cummin]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-cumprod]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-cumsum]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-diff]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-ffill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-fillna]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-pct_change]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-rank]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis='index'-shift]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-bfill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-cummax]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-cummin]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-cumprod]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-cumsum]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-diff]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-ffill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-fillna]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-pct_change]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-rank]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=0-shift]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-bfill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-cummax]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-cummin]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-cumprod]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-cumsum]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-diff]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-ffill]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-fillna]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-pct_change]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-rank]",
    "tests/apply/test_str.py::test_transform_groupby_kernel_frame[axis=1-shift]",
    "tests/arithmetic/test_categorical.py::TestCategoricalComparisons::test_categorical_nan_equality",
    "tests/arithmetic/test_categorical.py::TestCategoricalComparisons::test_categorical_tuple_equality",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64_array_sub_dt_with_different_timezone[DataFrame]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64_array_sub_dt_with_different_timezone[Index]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64_array_sub_dt_with_different_timezone[Series]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64_array_sub_dt_with_different_timezone[array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64_mul_div_numeric_invalid[array-one2-dt64_series2]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'+01:15']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'-02:15']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'Asia/Tokyo']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'US/Eastern']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'UTC']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'UTC+01:15']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'UTC-02:15']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'dateutil/Asia/Singapore']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-'dateutil/US/Pacific']",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-<UTC>]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-None]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-datetime.timezone(datetime.timedelta(seconds=3600))]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-datetime.timezone.utc]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-pytz.FixedOffset(-300)]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-pytz.FixedOffset(300)]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-tzlocal()]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-tzutc()]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-zoneinfo.ZoneInfo(key='US/Pacific')]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_sub_td64_nat[array-zoneinfo.ZoneInfo(key='UTC')]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['+01:15'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['+01:15'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['+01:15'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['+01:15'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['-02:15'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['-02:15'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['-02:15'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['-02:15'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['Asia/Tokyo'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['Asia/Tokyo'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['Asia/Tokyo'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['Asia/Tokyo'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['US/Eastern'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['US/Eastern'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['US/Eastern'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['US/Eastern'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC+01:15'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC+01:15'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC+01:15'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC+01:15'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC-02:15'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC-02:15'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC-02:15'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['UTC-02:15'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/Asia/Singapore'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/Asia/Singapore'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/Asia/Singapore'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/Asia/Singapore'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/US/Pacific'-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/US/Pacific'-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/US/Pacific'-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar['dateutil/US/Pacific'-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[<UTC>-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[<UTC>-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[<UTC>-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[<UTC>-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[None-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[None-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[None-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[None-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone.utc-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone.utc-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone.utc-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[datetime.timezone.utc-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(-300)-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(-300)-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(-300)-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(-300)-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(300)-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(300)-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(300)-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[pytz.FixedOffset(300)-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzlocal()-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzlocal()-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzlocal()-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzlocal()-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzutc()-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzutc()-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzutc()-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[tzutc()-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='UTC')-timedelta640-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='UTC')-timedelta641-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='UTC')-timedelta64_0-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_add_timedeltalike_scalar[zoneinfo.ZoneInfo(key='UTC')-timedelta64_1-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_sub_dtscalar[array-ts2]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64Arithmetic::test_dt64arr_sub_dtscalar[array-ts3]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64ArrayLikeComparisons::test_dt64arr_cmp_scalar_invalid[None-DataFrame-None]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64ArrayLikeComparisons::test_dt64arr_cmp_scalar_invalid[None-DataFrame-other6]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64ArrayLikeComparisons::test_dt64arr_cmp_scalar_invalid[None-Series-None]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64ArrayLikeComparisons::test_dt64arr_cmp_scalar_invalid[None-Series-other6]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['+01:15'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['-02:15'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['Asia/Tokyo'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['US/Eastern'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['UTC'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['UTC+01:15'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['UTC-02:15'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['dateutil/Asia/Singapore'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware['dateutil/US/Pacific'-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[<UTC>-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[datetime.timezone(datetime.timedelta(seconds=3600))-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[datetime.timezone.utc-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[pytz.FixedOffset(-300)-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[pytz.FixedOffset(300)-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[tzlocal()-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[tzutc()-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[zoneinfo.ZoneInfo(key='US/Pacific')-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64DateOffsetArithmetic::test_dti_add_tick_tzaware[zoneinfo.ZoneInfo(key='UTC')-array]",
    "tests/arithmetic/test_datetime64.py::TestDatetime64OverflowHandling::test_datetimeindex_sub_datetimeindex_overflow",
    "tests/arithmetic/test_datetime64.py::TestDatetime64OverflowHandling::test_datetimeindex_sub_timestamp_overflow",
    "tests/arithmetic/test_datetime64.py::test_non_nano_dt64_addsub_np_nat_scalars_unsupported_unit",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_divmod[numexpr-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_divmod[python-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-add-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-floordiv-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-mod-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-mul-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-pow-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-radd-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-rfloordiv-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-rmod-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-rmul-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-rpow-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-rsub-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-rtruediv-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-sub-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[numexpr-truediv-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-add-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-floordiv-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-mod-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-mul-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-pow-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-radd-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-rfloordiv-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-rmod-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-rmul-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-rpow-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-rsub-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-rtruediv-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-sub-slice]",
    "tests/arithmetic/test_numeric.py::TestAdditionSubtraction::test_series_operators_arithmetic[python-truediv-slice]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_df_mod_zero_df[numexpr]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_df_mod_zero_df[python]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-Index0-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-Index0-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-Index1-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-Index1-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-RangeIndex-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr--0.0-RangeIndex-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-Index0-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-Index0-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-Index1-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-Index1-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-RangeIndex-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[numexpr-zero14-RangeIndex-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-Index0-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-Index0-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-Index1-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-Index1-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-RangeIndex-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python--0.0-RangeIndex-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-Index0-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-Index0-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-Index1-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-Index1-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-RangeIndex-floordiv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_negative_zero[python-zero14-RangeIndex-truediv]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr--0.0-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr--0.0-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr--0.0-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr--0.0-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr-zero14-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr-zero14-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr-zero14-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[numexpr-zero14-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python--0.0-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python--0.0-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python--0.0-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python--0.0-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python-zero14-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python-zero14-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python-zero14-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_div_zero[python-zero14-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr--0.0-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr--0.0-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr--0.0-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr--0.0-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr-zero14-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr-zero14-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr-zero14-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[numexpr-zero14-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python--0.0-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python--0.0-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python--0.0-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python--0.0-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python-zero14-Index0]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python-zero14-Index1]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python-zero14-Index2]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_floordiv_zero[python-zero14-RangeIndex]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_div_ser[numexpr-float32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_inf[numexpr]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_inf[python]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-float64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int16-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int16-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int16-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int8-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int8-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-int8-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint16-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint16-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint16-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint8-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint8-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[numexpr-uint8-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-float64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int16-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int16-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int16-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int8-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int8-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-int8-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint16-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint16-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint16-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint32-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint32-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint32-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint64-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint64-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint64-uint64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint8-float64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint8-int64]",
    "tests/arithmetic/test_numeric.py::TestDivisionByZero::test_ser_divmod_zero[python-uint8-uint64]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2[numexpr]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2[python]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx0-2-expected0]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx1--3-expected1]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx4-50-expected4]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx5-3-expected5]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx6-4-expected6]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[numexpr-idx7-2-expected7]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx0-2-expected0]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx1--3-expected1]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx4-50-expected4]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx5-3-expected5]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx6-4-expected6]",
    "tests/arithmetic/test_numeric.py::TestNumericArithmeticUnsorted::test_numeric_compat2_floordiv[python-idx7-2-expected7]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index0-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index0-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index0-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index1-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-Index2-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[numexpr-RangeIndex-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index0-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index0-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index0-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index1-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-Index2-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-DataFrame-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-Index-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-Series-NaT]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.datetime64('2021-01-01T00:00:00')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.datetime64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.timedelta64('NaT')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.timedelta64('NaT','D')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.timedelta64(111600000000000,'ns')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_add_sub_datetimedeltalike_invalid[python-RangeIndex-array-np.timedelta64(31,'h')]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index0-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index1-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-Index2-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[numexpr-RangeIndex-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index0-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index1-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-Index2-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta640]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta641]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta642]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta64_0]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta64_1]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_mul_tdscalar[python-RangeIndex-array-timedelta64_2]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta640-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta640-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta640-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta640-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta641-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta641-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta641-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta641-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_0-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_0-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_0-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_0-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_1-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_1-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_1-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[numexpr-timedelta64_1-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta640-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta640-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta640-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta640-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta641-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta641-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta641-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta641-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_0-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_0-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_0-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_0-RangeIndex-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_1-Index0-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_1-Index1-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_1-Index2-array]",
    "tests/arithmetic/test_numeric.py::TestNumericArraylikeArithmeticWithDatetimeLike::test_numeric_arr_rdiv_tdscalar[python-timedelta64_1-RangeIndex-array]",
    "tests/arithmetic/test_object.py::TestObjectComparisons::test_object_comparisons[True]",
    "tests/arithmetic/test_object.py::test_index_ops_defer_to_unknown_subclasses[datetime]",
    "tests/arithmetic/test_object.py::test_index_ops_defer_to_unknown_subclasses[object]",
    "tests/arithmetic/test_object.py::test_index_ops_defer_to_unknown_subclasses[period]",
    "tests/arithmetic/test_object.py::test_index_ops_defer_to_unknown_subclasses[timedelta]",
    "tests/arithmetic/test_period.py::TestPeriodIndexSeriesMethods::test_pi_sub_period",
    "tests/arithmetic/test_period.py::TestPeriodIndexSeriesMethods::test_pi_sub_period_nat",
    "tests/arithmetic/test_timedelta64.py::TestAddSubNaTMasking::test_tdi_add_overflow",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_addition_ops",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_dti_tdi_numeric_ops",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_subtraction_ops",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_subtraction_ops_with_tz[DataFrame]",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_subtraction_ops_with_tz[Index]",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_subtraction_ops_with_tz[Series]",
    "tests/arithmetic/test_timedelta64.py::TestTimedelta64ArithmeticUnsorted::test_subtraction_ops_with_tz[array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_operators_timedelta64",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_datetime64_nat[array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_datetimelike_scalar[array-None-datetime64]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_td64_nat[array-tdnat0]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names0]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names2]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names4]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names5]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_sub_tdi[array-names6]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_timedeltalike[timedelta640-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_timedeltalike[timedelta641-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_timedeltalike[timedelta64_0-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_add_timedeltalike[timedelta64_1-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_sub_timedeltalike[timedelta640-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeAddSubOps::test_td64arr_sub_timedeltalike[timedelta641-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_nat_invalid[DataFrame]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_nat_invalid[Series]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_ndarray[array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-D-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-D-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-D-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-h-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-h-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-h-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-m-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-m-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-m-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ms-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ms-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ms-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ns-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ns-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-ns-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-s-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-s-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-s-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-us-10]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-us-1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64_scalar[array-us-3]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_td64nat[array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar[timedelta640-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar[timedelta641-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar[timedelta64_0-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar[timedelta64_1-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar_with_nat[timedelta640-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar_with_nat[timedelta641-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar_with_nat[timedelta64_0-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_div_tdlike_scalar_with_nat[timedelta64_1-array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_floordiv_td64arr_with_nat[DataFrame]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_floordiv_td64arr_with_nat[array]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_floordiv_tdscalar[array-timedelta64]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_mul_numeric_scalar[array-one1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmod_tdscalar[array-timedelta640]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmod_tdscalar[array-timedelta641]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmod_tdscalar[array-timedelta64_0]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmod_tdscalar[array-timedelta64_1]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-float-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-float32-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-float64-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-int-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-int16-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-int32-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-int64-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-int8-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-uint16-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-uint32-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-uint64-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_td64arr_rmul_numeric_array[array-uint8-ndarray]",
    "tests/arithmetic/test_timedelta64.py::TestTimedeltaArraylikeMulDivOps::test_tdi_rmul_arraylike[array-ndarray]",
    "tests/arrays/boolean/test_arithmetic.py::test_div",
    "tests/arrays/boolean/test_construction.py::test_boolean_array_constructor_copy",
    "tests/arrays/boolean/test_construction.py::test_coerce_to_array",
    "tests/arrays/boolean/test_construction.py::test_coerce_to_array_from_boolean_array",
    "tests/arrays/boolean/test_construction.py::test_coerce_to_numpy_array",
    "tests/arrays/boolean/test_construction.py::test_to_numpy_copy",
    "tests/arrays/boolean/test_function.py::test_value_counts_with_normalize",
    "tests/arrays/boolean/test_ops.py::TestUnaryOps::test_invert",
    "tests/arrays/boolean/test_reduction.py::test_any_all[values0-True-True-True-exp_all_noskip0]",
    "tests/arrays/boolean/test_reduction.py::test_any_all[values1-False-False-exp_any_noskip1-False]",
    "tests/arrays/boolean/test_reduction.py::test_any_all[values2-False-True-exp_any_noskip2-exp_all_noskip2]",
    "tests/arrays/boolean/test_reduction.py::test_any_all[values3-False-True-False-True]",
    "tests/arrays/boolean/test_reduction.py::test_any_all[values4-True-True-True-True]",
    "tests/arrays/boolean/test_reduction.py::test_any_all[values5-False-False-False-False]",
    "tests/arrays/boolean/test_reduction.py::test_reductions_return_types[count-False]",
    "tests/arrays/boolean/test_reduction.py::test_reductions_return_types[count-True]",
    "tests/arrays/categorical/test_analytics.py::TestCategoricalAnalytics::test_memory_usage",
    "tests/arrays/categorical/test_analytics.py::TestCategoricalAnalytics::test_min_max_ordered[array]",
    "tests/arrays/categorical/test_api.py::TestCategoricalAPI::test_rename_categories",
    "tests/arrays/categorical/test_api.py::TestCategoricalAPI::test_set_categories",
    "tests/arrays/categorical/test_api.py::TestPrivateCategoricalAPI::test_codes_immutable",
    "tests/arrays/categorical/test_astype.py::TestAstype::test_astype[False]",
    "tests/arrays/categorical/test_astype.py::TestAstype::test_astype[True]",
    "tests/arrays/categorical/test_astype.py::TestAstype::test_astype_nan_to_int[values0-CategoricalIndex]",
    "tests/arrays/categorical/test_astype.py::TestAstype::test_astype_nan_to_int[values1-CategoricalIndex]",
    "tests/arrays/categorical/test_constructors.py::TestCategoricalConstructors::test_constructor",
    "tests/arrays/categorical/test_constructors.py::TestCategoricalConstructors::test_constructor_preserves_freq",
    "tests/arrays/categorical/test_constructors.py::TestCategoricalConstructors::test_from_sequence_copy",
    "tests/arrays/categorical/test_indexing.py::TestCategoricalIndexing::test_where_new_category_raises",
    "tests/arrays/categorical/test_missing.py::TestCategoricalMissing::test_compare_categorical_with_missing[a10-a20-categories0]",
    "tests/arrays/categorical/test_missing.py::TestCategoricalMissing::test_compare_categorical_with_missing[a11-a21-categories1]",
    "tests/arrays/categorical/test_missing.py::TestCategoricalMissing::test_use_inf_as_na_outside_context[values2-expected2]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical[a-a-result2-None]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical[b-None-result3-Categorical.categories length are different]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical[b-c-result0-Categorical.categories are different]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical_ea_dtype",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical_series[to_replace11-value11-expected11-False]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical_series[to_replace6-4-expected6-False]",
    "tests/arrays/categorical/test_replace.py::test_replace_categorical_series[to_replace7-5-expected7-False]",
    "tests/arrays/categorical/test_repr.py::TestCategoricalRepr::test_categorical_repr_int_with_nan",
    "tests/arrays/categorical/test_sorting.py::TestCategoricalSort::test_sort_values",
    "tests/arrays/categorical/test_sorting.py::TestCategoricalSort::test_sort_values_na_position",
    "tests/arrays/datetimes/test_constructors.py::TestDatetimeArrayConstructor::test_bool_dtype_raises",
    "tests/arrays/datetimes/test_constructors.py::TestDatetimeArrayConstructor::test_copy",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_empty[ms-Europe/Berlin]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_empty[ns-Asia/Kolkata]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_empty[ns-UTC]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_empty[s-UTC]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_empty[us-US/Eastern]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_from_integers",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[ms-ms-UTC-Europe/Berlin-data1]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[ms-us-US/Eastern-UTC-data6]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[ns-ns-US/Central-Asia/Kolkata-data3]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[ns-s-UTC-UTC-data4]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[s-ns-US/Central-Asia/Kolkata-data7]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[s-s-UTC-UTC-data0]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[us-ms-UTC-Europe/Berlin-data5]",
    "tests/arrays/datetimes/test_constructors.py::test_from_arrow_with_different_units_and_timezones_with[us-us-US/Eastern-UTC-data2]",
    "tests/arrays/floating/test_astype.py::test_astype_copy",
    "tests/arrays/floating/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Float32Dtype]",
    "tests/arrays/floating/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Float64Dtype]",
    "tests/arrays/floating/test_construction.py::test_floating_array_constructor_copy",
    "tests/arrays/floating/test_construction.py::test_series_from_float[Float32Dtype]",
    "tests/arrays/floating/test_function.py::test_stat_method[kurtosis-kwargs4]",
    "tests/arrays/floating/test_function.py::test_stat_method[skew-kwargs5]",
    "tests/arrays/floating/test_function.py::test_ufuncs_single[absolute]",
    "tests/arrays/floating/test_function.py::test_ufuncs_single[sign]",
    "tests/arrays/floating/test_function.py::test_value_counts_empty",
    "tests/arrays/floating/test_function.py::test_value_counts_with_normalize",
    "tests/arrays/floating/test_repr.py::test_frame_repr[Float32Dtype]",
    "tests/arrays/floating/test_to_numpy.py::test_to_numpy_copy",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int16Dtype-__floordiv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int16Dtype-__rtruediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int16Dtype-__truediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int32Dtype-__floordiv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int32Dtype-__rtruediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int32Dtype-__truediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int64Dtype-__floordiv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int64Dtype-__rtruediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int64Dtype-__truediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int8Dtype-__floordiv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int8Dtype-__rtruediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[Int8Dtype-__truediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt16Dtype-__floordiv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt16Dtype-__rtruediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt16Dtype-__truediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt32Dtype-__floordiv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt32Dtype-__rtruediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt32Dtype-__truediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt64Dtype-__floordiv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt64Dtype-__rtruediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt64Dtype-__truediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt8Dtype-__floordiv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt8Dtype-__rtruediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arith_coerce_scalar[UInt8Dtype-__truediv__]",
    "tests/arrays/integer/test_arithmetic.py::test_arithmetic_conversion[__rtruediv__-1.0]",
    "tests/arrays/integer/test_arithmetic.py::test_arithmetic_conversion[__rtruediv__-other1]",
    "tests/arrays/integer/test_arithmetic.py::test_arithmetic_conversion[__truediv__-1.0]",
    "tests/arrays/integer/test_arithmetic.py::test_arithmetic_conversion[__truediv__-other1]",
    "tests/arrays/integer/test_arithmetic.py::test_values_multiplying_large_series_by_NA",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Int16Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Int32Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Int64Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[Int8Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[UInt16Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[UInt32Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[UInt64Dtype]",
    "tests/arrays/integer/test_comparison.py::TestComparisonOps::test_ufunc_with_out[UInt8Dtype]",
    "tests/arrays/integer/test_construction.py::test_from_dtype_from_float[Int16Dtype]",
    "tests/arrays/integer/test_construction.py::test_from_dtype_from_float[Int32Dtype]",
    "tests/arrays/integer/test_construction.py::test_from_dtype_from_float[Int64Dtype]",
    "tests/arrays/integer/test_construction.py::test_from_dtype_from_float[Int8Dtype]",
    "tests/arrays/integer/test_construction.py::test_from_dtype_from_float[UInt16Dtype]",
    "tests/arrays/integer/test_construction.py::test_from_dtype_from_float[UInt32Dtype]",
    "tests/arrays/integer/test_construction.py::test_from_dtype_from_float[UInt64Dtype]",
    "tests/arrays/integer/test_construction.py::test_from_dtype_from_float[UInt8Dtype]",
    "tests/arrays/integer/test_construction.py::test_integer_array_constructor_copy",
    "tests/arrays/integer/test_dtypes.py::test_astype[data-Int16Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data-Int32Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data-Int64Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data-Int8Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data-UInt16Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data-UInt32Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data-UInt64Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data-UInt8Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data_missing-Int16Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data_missing-Int32Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data_missing-Int64Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data_missing-Int8Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data_missing-UInt16Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data_missing-UInt32Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data_missing-UInt64Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype[data_missing-UInt8Dtype]",
    "tests/arrays/integer/test_dtypes.py::test_astype_copy",
    "tests/arrays/integer/test_function.py::test_ufuncs_single_int[absolute]",
    "tests/arrays/integer/test_function.py::test_ufuncs_single_int[sign]",
    "tests/arrays/integer/test_function.py::test_value_counts_empty",
    "tests/arrays/integer/test_function.py::test_value_counts_with_normalize",
    "tests/arrays/interval/test_interval.py::TestSetitem::test_set_na[float64]",
    "tests/arrays/interval/test_interval_pyarrow.py::test_arrow_array",
    "tests/arrays/interval/test_interval_pyarrow.py::test_arrow_array_missing",
    "tests/arrays/interval/test_interval_pyarrow.py::test_from_arrow_from_raw_struct_array",
    "tests/arrays/masked/test_arithmetic.py::test_frame[Int16-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[Int16-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[Int32-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[Int32-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[Int64-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[Int64-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[Int8-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[Int8-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[UInt16-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[UInt16-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[UInt32-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[UInt32-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[UInt64-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[UInt64-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[UInt8-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[UInt8-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[boolean-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_frame[boolean-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[Int16-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[Int16-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[Int32-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[Int32-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[Int64-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[Int64-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[Int8-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[Int8-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[UInt16-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[UInt16-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[UInt32-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[UInt32-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[UInt64-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[UInt64-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[UInt8-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[UInt8-__truediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__floordiv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__mod__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__pow__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__rfloordiv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__rmod__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__rpow__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__rtruediv__]",
    "tests/arrays/masked/test_arithmetic.py::test_series[boolean-__truediv__]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Float32]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Float64]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Int16]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Int32]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Int64]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[Int8]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[UInt16]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[UInt32]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[UInt64]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[UInt8]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_array[boolean]",
    "tests/arrays/masked/test_arrow_compat.py::test_arrow_from_arrow_uint",
    "tests/arrays/masked/test_arrow_compat.py::test_dataframe_from_arrow_types_mapper",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Float32-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Float32-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Float64-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Float64-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int16-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int16-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int32-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int32-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int64-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int64-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int8-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[Int8-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt16-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt16-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt32-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt32-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt64-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt64-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt8-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[UInt8-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[boolean-arr0]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_null[boolean-arr1]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Float32]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Float64]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Int16]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Int32]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Int64]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[Int8]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[UInt16]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[UInt32]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[UInt64]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[UInt8]",
    "tests/arrays/masked/test_arrow_compat.py::test_from_arrow_type_error[boolean]",
    "tests/arrays/masked/test_function.py::test_round[Float32]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_float[Float32-1.0]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_float[Float32-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_float[Float64-1.0]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_float[Float64-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[Int16-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[Int32-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[Int64-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[Int8-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[UInt16-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[UInt32-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[UInt64-1]",
    "tests/arrays/masked/test_indexing.py::TestSetitemValidation::test_setitem_validation_scalar_int[UInt8-1]",
    "tests/arrays/numpy_/test_numpy.py::test_factorize_unsigned",
    "tests/arrays/numpy_/test_numpy.py::test_np_max_nested_tuples",
    "tests/arrays/numpy_/test_numpy.py::test_np_reduce_2d",
    "tests/arrays/numpy_/test_numpy.py::test_setitem_preserves_views",
    "tests/arrays/numpy_/test_numpy.py::test_to_numpy",
    "tests/arrays/numpy_/test_numpy.py::test_ufunc",
    "tests/arrays/numpy_/test_numpy.py::test_ufunc_unary[absolute]",
    "tests/arrays/numpy_/test_numpy.py::test_ufunc_unary[negative]",
    "tests/arrays/numpy_/test_numpy.py::test_ufunc_unary[positive]",
    "tests/arrays/period/test_arrow_compat.py::test_arrow_array_missing",
    "tests/arrays/period/test_astype.py::test_astype_copies",
    "tests/arrays/sparse/test_accessor.py::TestFrameAccessor::test_from_spmatrix_columns[columns1]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[add-a0-b0]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[add-a1-b1]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[add-a2-b2]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[add-a3-b3]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[add-a4-b4]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[greater-a0-b0]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[greater-a1-b1]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[greater-a2-b2]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[greater-a3-b3]",
    "tests/arrays/sparse/test_arithmetics.py::test_binary_ufuncs[greater-a4-b4]",
    "tests/arrays/sparse/test_arithmetics.py::test_ufuncs[arr0-absolute]",
    "tests/arrays/sparse/test_arithmetics.py::test_ufuncs[arr0-exp]",
    "tests/arrays/sparse/test_arithmetics.py::test_ufuncs[arr1-absolute]",
    "tests/arrays/sparse/test_arithmetics.py::test_ufuncs[arr1-exp]",
    "tests/arrays/sparse/test_array.py::TestSparseArrayAnalytics::test_modf[0.0]",
    "tests/arrays/sparse/test_array.py::TestSparseArrayAnalytics::test_modf[nan]",
    "tests/arrays/sparse/test_array.py::TestSparseArrayAnalytics::test_ufunc",
    "tests/arrays/sparse/test_array.py::TestSparseArrayAnalytics::test_ufunc_args",
    "tests/arrays/sparse/test_array.py::test_array_interface",
    "tests/arrays/sparse/test_constructors.py::TestConstructors::test_constructor_copy",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[delete_blocks-add]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[delete_blocks-floordiv]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[delete_blocks-mul]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[delete_blocks-sub]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[delete_blocks-truediv]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[no_intersect-add]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[no_intersect-floordiv]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[no_intersect-mul]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[no_intersect-sub]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[no_intersect-truediv]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[one_empty-add]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[one_empty-floordiv]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[one_empty-mul]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[one_empty-sub]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[one_empty-truediv]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[plain_case-add]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[plain_case-floordiv]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[plain_case-mul]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[plain_case-sub]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[plain_case-truediv]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[skip_block-add]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[skip_block-floordiv]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[skip_block-mul]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[skip_block-sub]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[skip_block-truediv]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[split_blocks-add]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[split_blocks-floordiv]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[split_blocks-mul]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[split_blocks-sub]",
    "tests/arrays/sparse/test_libsparse.py::TestSparseOperators::test_op[split_blocks-truediv]",
    "tests/arrays/string_/test_string.py::test_arrow_array[pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_array[pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_array[string=str[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_arrow_array[string=string[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow_numpy-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow_numpy-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[pyarrow_numpy-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[python-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[python-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[python-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=str[pyarrow]-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=str[pyarrow]-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=str[python]-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=str[python]-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=string[pyarrow]-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=string[pyarrow]-python]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=string[python]-pyarrow]",
    "tests/arrays/string_/test_string.py::test_arrow_load_from_zero_chunks[string=string[python]-python]",
    "tests/arrays/string_/test_string.py::test_arrow_roundtrip[pyarrow-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_roundtrip[pyarrow_numpy-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_arrow_roundtrip[python-pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_astype_float[pyarrow-Float32]",
    "tests/arrays/string_/test_string.py::test_astype_float[pyarrow-Float64]",
    "tests/arrays/string_/test_string.py::test_astype_float[pyarrow_numpy-Float32]",
    "tests/arrays/string_/test_string.py::test_astype_float[pyarrow_numpy-Float64]",
    "tests/arrays/string_/test_string.py::test_astype_float[python-Float32]",
    "tests/arrays/string_/test_string.py::test_astype_float[python-Float64]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[pyarrow]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[python]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[string=str[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[string=str[python]]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[string=string[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_astype_roundtrip[string=string[python]]",
    "tests/arrays/string_/test_string.py::test_isin[pyarrow]",
    "tests/arrays/string_/test_string.py::test_isin[pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_isin[python]",
    "tests/arrays/string_/test_string.py::test_isin[string=str[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_isin[string=str[python]]",
    "tests/arrays/string_/test_string.py::test_isin[string=string[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_isin[string=string[python]]",
    "tests/arrays/string_/test_string.py::test_memory_usage[python]",
    "tests/arrays/string_/test_string.py::test_memory_usage[string=str[python]]",
    "tests/arrays/string_/test_string.py::test_memory_usage[string=string[python]]",
    "tests/arrays/string_/test_string.py::test_min_max[pyarrow-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[pyarrow-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[pyarrow_numpy-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[pyarrow_numpy-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[python-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[python-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[string=str[pyarrow]-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[string=str[pyarrow]-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[string=str[python]-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[string=str[python]-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[string=string[pyarrow]-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[string=string[pyarrow]-False-min]",
    "tests/arrays/string_/test_string.py::test_min_max[string=string[python]-False-max]",
    "tests/arrays/string_/test_string.py::test_min_max[string=string[python]-False-min]",
    "tests/arrays/string_/test_string.py::test_repr[pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_repr[string=str[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_repr[string=str[python]]",
    "tests/arrays/string_/test_string.py::test_to_numpy_returns_pdna_default[pyarrow_numpy]",
    "tests/arrays/string_/test_string.py::test_to_numpy_returns_pdna_default[string=str[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_to_numpy_returns_pdna_default[string=str[python]]",
    "tests/arrays/string_/test_string.py::test_value_counts_sort_false[string=str[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_value_counts_sort_false[string=str[python]]",
    "tests/arrays/string_/test_string.py::test_value_counts_sort_false[string=string[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_value_counts_with_normalize[pyarrow]",
    "tests/arrays/string_/test_string.py::test_value_counts_with_normalize[python]",
    "tests/arrays/string_/test_string.py::test_value_counts_with_normalize[string=string[pyarrow]]",
    "tests/arrays/string_/test_string.py::test_value_counts_with_normalize[string=string[python]]",
    "tests/arrays/test_array.py::test_array_inference[data7-expected7]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['+01:15'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['-02:15'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['Asia/Tokyo'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['US/Eastern'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC+01:15'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['UTC-02:15'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/Asia/Singapore'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype['dateutil/US/Pacific'-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[<UTC>-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[None-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone(datetime.timedelta(seconds=3600))-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[datetime.timezone.utc-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(-300)-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[pytz.FixedOffset(300)-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzlocal()-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[tzutc()-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_i8_dtype[zoneinfo.ZoneInfo(key='UTC')-YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[B]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[D]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[ME]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[QE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[W]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_array_interface[YE]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-B-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-D-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-ME-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-QE-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-W-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='US/Pacific')-YE-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-B-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-D-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-ME-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-QE-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-W-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-False-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-False-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-None-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-None-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-True-True-False]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_compare_categorical_dtype[zoneinfo.ZoneInfo(key='UTC')-YE-True-True-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_from_array_keeps_base",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-B-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-D-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-ME-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-QE-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-W-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_searchsorted_castable_strings[None-YE-pyarrow_numpy-series]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-B-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-D-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-ME-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-QE-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-W-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='US/Pacific')-YE-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-B-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-D-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-ME-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-QE-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-W-True]",
    "tests/arrays/test_datetimelike.py::TestDatetimeArray::test_setitem_categorical[zoneinfo.ZoneInfo(key='UTC')-YE-True]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[B]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[D]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[ME]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[QE]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[W]",
    "tests/arrays/test_datetimelike.py::TestPeriodArray::test_array_interface[YE]",
    "tests/arrays/test_datetimelike.py::TestTimedeltaArray::test_array_interface",
    "tests/arrays/test_datetimelike.py::TestTimedeltaArray::test_searchsorted_castable_strings[pyarrow_numpy-series]",
    "tests/arrays/test_datetimes.py::TestDatetimeArray::test_array_interface",
    "tests/arrays/test_datetimes.py::TestDatetimeArray::test_astype_copies[datetime64[ns]-datetime64[ns]]",
    "tests/arrays/test_datetimes.py::TestDatetimeArray::test_astype_to_same",
    "tests/arrays/test_datetimes.py::TestDatetimeArray::test_shift_fill_value",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[eq]",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[ge]",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[gt]",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[le]",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[lt]",
    "tests/arrays/test_datetimes.py::TestDatetimeArrayComparisons::test_cmp_dt64_arraylike_tznaive[ne]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'Asia/Tokyo'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'Asia/Tokyo'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'US/Eastern'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'US/Eastern'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'UTC-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'dateutil/Asia/Singapore'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'dateutil/Asia/Singapore'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'dateutil/US/Pacific'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-'dateutil/US/Pacific'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-<UTC>-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-<UTC>-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-None-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-None-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone(datetime.timedelta(seconds=3600))-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone(datetime.timedelta(seconds=3600))-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone.utc-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-datetime.timezone.utc-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-pytz.FixedOffset(-300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-pytz.FixedOffset(-300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-pytz.FixedOffset(300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-pytz.FixedOffset(300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-tzlocal()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-tzlocal()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-tzutc()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-tzutc()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-zoneinfo.ZoneInfo(key='US/Pacific')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-zoneinfo.ZoneInfo(key='US/Pacific')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-zoneinfo.ZoneInfo(key='UTC')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[ms-zoneinfo.ZoneInfo(key='UTC')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'Asia/Tokyo'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'Asia/Tokyo'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'US/Eastern'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'US/Eastern'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'UTC-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'dateutil/Asia/Singapore'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'dateutil/Asia/Singapore'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'dateutil/US/Pacific'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-'dateutil/US/Pacific'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-<UTC>-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-<UTC>-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-None-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-None-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone(datetime.timedelta(seconds=3600))-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone(datetime.timedelta(seconds=3600))-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone.utc-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-datetime.timezone.utc-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-pytz.FixedOffset(-300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-pytz.FixedOffset(-300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-pytz.FixedOffset(300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-pytz.FixedOffset(300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-tzlocal()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-tzlocal()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-tzutc()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-tzutc()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-zoneinfo.ZoneInfo(key='US/Pacific')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-zoneinfo.ZoneInfo(key='US/Pacific')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-zoneinfo.ZoneInfo(key='UTC')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[s-zoneinfo.ZoneInfo(key='UTC')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'Asia/Tokyo'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'Asia/Tokyo'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'US/Eastern'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'US/Eastern'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC+01:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC+01:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC-02:15'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'UTC-02:15'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'dateutil/Asia/Singapore'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'dateutil/Asia/Singapore'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'dateutil/US/Pacific'-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-'dateutil/US/Pacific'-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-<UTC>-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-<UTC>-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-None-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-None-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone(datetime.timedelta(seconds=3600))-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone(datetime.timedelta(seconds=3600))-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone.utc-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-datetime.timezone.utc-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-pytz.FixedOffset(-300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-pytz.FixedOffset(-300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-pytz.FixedOffset(300)-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-pytz.FixedOffset(300)-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-tzlocal()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-tzlocal()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-tzutc()-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-tzutc()-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-zoneinfo.ZoneInfo(key='US/Pacific')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-zoneinfo.ZoneInfo(key='US/Pacific')-scalar3]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-zoneinfo.ZoneInfo(key='UTC')-scalar2]",
    "tests/arrays/test_datetimes.py::TestNonNano::test_add_timedeltalike_scalar_mismatched_reso[us-zoneinfo.ZoneInfo(key='UTC')-scalar3]",
    "tests/arrays/test_period.py::test_registered",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[ms-nat0]",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[ms-nat1]",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[s-nat0]",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[s-nat1]",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[us-nat0]",
    "tests/arrays/test_timedeltas.py::TestNonNano::test_add_nat_datetimelike_scalar[us-nat1]",
    "tests/arrays/test_timedeltas.py::TestUnaryOps::test_abs",
    "tests/arrays/test_timedeltas.py::TestUnaryOps::test_neg",
    "tests/arrays/test_timedeltas.py::TestUnaryOps::test_neg_freq",
    "tests/arrays/test_timedeltas.py::TestUnaryOps::test_pos",
    "tests/arrays/timedeltas/test_constructors.py::TestTimedeltaArrayConstructor::test_copy",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_reductions_empty[False-max]",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_reductions_empty[False-min]",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_reductions_empty[True-max]",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_reductions_empty[True-min]",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_std[add0]",
    "tests/arrays/timedeltas/test_reductions.py::TestReductions::test_std[add1]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[DataFrame-array-datetime64[D]]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[DataFrame-dict-datetime64[D]]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[DataFrame-dict-object-datetime.datetime]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[Index-datetime64[D]]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[Series-datetime64[D]]",
    "tests/base/test_constructors.py::TestConstruction::test_constructor_datetime_outofbound[Series-object-datetime.datetime]",
    "tests/base/test_conversion.py::test_array[index-arr0-_codes]",
    "tests/base/test_conversion.py::test_array[index-arr1-_ndarray]",
    "tests/base/test_conversion.py::test_array[index-arr2-_data]",
    "tests/base/test_conversion.py::test_array[index-arr3-_left]",
    "tests/base/test_conversion.py::test_array[index-arr4-_sparse_values]",
    "tests/base/test_conversion.py::test_array[index-arr5-_ndarray]",
    "tests/base/test_conversion.py::test_array[index-arr6-_ndarray]",
    "tests/base/test_conversion.py::test_array[series-arr0-_codes]",
    "tests/base/test_conversion.py::test_array[series-arr1-_ndarray]",
    "tests/base/test_conversion.py::test_array[series-arr2-_data]",
    "tests/base/test_conversion.py::test_array[series-arr3-_left]",
    "tests/base/test_conversion.py::test_array[series-arr4-_sparse_values]",
    "tests/base/test_conversion.py::test_array[series-arr5-_ndarray]",
    "tests/base/test_conversion.py::test_array[series-arr6-_ndarray]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr0-expected0-True]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr1-expected1-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr1-expected1]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr2-expected2-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr3-expected3-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr3-expected3]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr4-expected4-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr5-expected5-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr6-expected6-True]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr6-expected6]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr7-expected7-False]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr8-expected8-True]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr8-expected8]",
    "tests/base/test_conversion.py::test_to_numpy[array-arr9-expected9-False]",
    "tests/base/test_conversion.py::test_to_numpy[series-arr0-expected0-True]",
    "tests/base/test_conversion.py::test_to_numpy[series-arr6-expected6-True]",
    "tests/base/test_conversion.py::test_to_numpy[series-arr8-expected8-True]",
    "tests/base/test_conversion.py::test_to_numpy_copy[arr0-False]",
    "tests/base/test_conversion.py::test_to_numpy_copy[arr0-True]",
    "tests/base/test_conversion.py::test_to_numpy_copy[arr1-False]",
    "tests/base/test_conversion.py::test_to_numpy_copy[arr1-True]",
    "tests/base/test_conversion.py::test_values_consistent[arr0-ndarray-int64]",
    "tests/base/test_conversion.py::test_values_consistent[arr1-ndarray-object]",
    "tests/base/test_fillna.py::test_fillna[multi]",
    "tests/base/test_fillna.py::test_fillna[tuples]",
    "tests/base/test_misc.py::test_memory_usage[bool-object]",
    "tests/base/test_misc.py::test_memory_usage[bool-series]",
    "tests/base/test_misc.py::test_memory_usage[bytearray-series]",
    "tests/base/test_misc.py::test_memory_usage[bytes-series]",
    "tests/base/test_misc.py::test_memory_usage[categorical]",
    "tests/base/test_misc.py::test_memory_usage[complex-series]",
    "tests/base/test_misc.py::test_memory_usage[complex128]",
    "tests/base/test_misc.py::test_memory_usage[complex64]",
    "tests/base/test_misc.py::test_memory_usage[datetime-tz]",
    "tests/base/test_misc.py::test_memory_usage[datetime]",
    "tests/base/test_misc.py::test_memory_usage[dict-series]",
    "tests/base/test_misc.py::test_memory_usage[empty]",
    "tests/base/test_misc.py::test_memory_usage[float-series]",
    "tests/base/test_misc.py::test_memory_usage[float16-series]",
    "tests/base/test_misc.py::test_memory_usage[float32-series]",
    "tests/base/test_misc.py::test_memory_usage[float32]",
    "tests/base/test_misc.py::test_memory_usage[float64]",
    "tests/base/test_misc.py::test_memory_usage[frozenset-series]",
    "tests/base/test_misc.py::test_memory_usage[int-series]",
    "tests/base/test_misc.py::test_memory_usage[int16-series]",
    "tests/base/test_misc.py::test_memory_usage[int16]",
    "tests/base/test_misc.py::test_memory_usage[int32-series]",
    "tests/base/test_misc.py::test_memory_usage[int32]",
    "tests/base/test_misc.py::test_memory_usage[int64]",
    "tests/base/test_misc.py::test_memory_usage[int8-series]",
    "tests/base/test_misc.py::test_memory_usage[interval]",
    "tests/base/test_misc.py::test_memory_usage[list-series]",
    "tests/base/test_misc.py::test_memory_usage[memoryview-series]",
    "tests/base/test_misc.py::test_memory_usage[mi-with-dt64tz-level]",
    "tests/base/test_misc.py::test_memory_usage[multi]",
    "tests/base/test_misc.py::test_memory_usage[nullable_float]",
    "tests/base/test_misc.py::test_memory_usage[nullable_int]",
    "tests/base/test_misc.py::test_memory_usage[nullable_uint]",
    "tests/base/test_misc.py::test_memory_usage[object]",
    "tests/base/test_misc.py::test_memory_usage[period]",
    "tests/base/test_misc.py::test_memory_usage[range-series]",
    "tests/base/test_misc.py::test_memory_usage[series-with-bool-object-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-categorical-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-complex128-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-complex64-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-datetime-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-datetime-tz-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-empty-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-float32-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-float64-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-int16-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-int32-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-int64-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-int8-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-interval-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-mi-with-dt64tz-level-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-multi-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-nullable_bool-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-nullable_float-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-nullable_int-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-nullable_uint-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-object-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-period-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-range-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-string-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-string-pyarrow-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-string-python-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-timedelta-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-tuples-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-uint16-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-uint32-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-uint64-index]",
    "tests/base/test_misc.py::test_memory_usage[series-with-uint8-index]",
    "tests/base/test_misc.py::test_memory_usage[set-series]",
    "tests/base/test_misc.py::test_memory_usage[str-series]",
    "tests/base/test_misc.py::test_memory_usage[string-pyarrow]",
    "tests/base/test_misc.py::test_memory_usage[string-python]",
    "tests/base/test_misc.py::test_memory_usage[string]",
    "tests/base/test_misc.py::test_memory_usage[timedelta]",
    "tests/base/test_misc.py::test_memory_usage[tuple-series]",
    "tests/base/test_misc.py::test_memory_usage[uint16-series]",
    "tests/base/test_misc.py::test_memory_usage[uint16]",
    "tests/base/test_misc.py::test_memory_usage[uint32-series]",
    "tests/base/test_misc.py::test_memory_usage[uint32]",
    "tests/base/test_misc.py::test_memory_usage[uint64]",
    "tests/base/test_misc.py::test_memory_usage[uint8-series]",
    "tests/base/test_misc.py::test_memory_usage_components_narrow_series[float16]",
    "tests/base/test_misc.py::test_ndarray_compat_properties[multi]",
    "tests/base/test_misc.py::test_ndarray_compat_properties[tuples]",
    "tests/base/test_unique.py::test_nunique_null[float32-None]",
    "tests/base/test_unique.py::test_nunique_null[float32-nan]",
    "tests/base/test_unique.py::test_nunique_null[float32-series-None]",
    "tests/base/test_unique.py::test_nunique_null[float32-series-nan]",
    "tests/base/test_unique.py::test_nunique_null[float64-None]",
    "tests/base/test_unique.py::test_nunique_null[float64-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-bool-dtype-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-bool-dtype-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-bool-object-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-bool-object-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-categorical-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-categorical-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-complex128-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-complex128-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-complex64-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-complex64-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-datetime-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-datetime-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-datetime-tz-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-datetime-tz-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-float32-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-float32-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-float64-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-float64-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-int16-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-int16-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-int32-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-int32-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-int64-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-int64-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-int8-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-int8-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-interval-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-interval-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-mi-with-dt64tz-level-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-mi-with-dt64tz-level-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-multi-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-multi-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-nullable_bool-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-nullable_bool-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-nullable_float-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-nullable_float-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-nullable_int-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-nullable_int-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-nullable_uint-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-nullable_uint-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-object-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-object-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-period-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-period-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-range-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-range-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-repeats-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-repeats-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-string-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-string-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-string-pyarrow-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-string-pyarrow-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-string-python-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-string-python-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-timedelta-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-timedelta-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-tuples-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-tuples-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-uint16-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-uint16-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-uint32-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-uint32-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-uint64-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-uint64-index-nan]",
    "tests/base/test_unique.py::test_nunique_null[series-with-uint8-index-None]",
    "tests/base/test_unique.py::test_nunique_null[series-with-uint8-index-nan]",
    "tests/base/test_unique.py::test_unique[multi]",
    "tests/base/test_unique.py::test_unique[tuples]",
    "tests/base/test_value_counts.py::test_value_counts[multi]",
    "tests/base/test_value_counts.py::test_value_counts[tuples]",
    "tests/base/test_value_counts.py::test_value_counts_bins[index]",
    "tests/base/test_value_counts.py::test_value_counts_bins[series]",
    "tests/base/test_value_counts.py::test_value_counts_null[float32-series-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[float32-series-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-bool-dtype-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-bool-dtype-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-bool-object-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-bool-object-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-categorical-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-categorical-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-complex128-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-complex128-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-complex64-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-complex64-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-datetime-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-datetime-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-datetime-tz-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-datetime-tz-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-float32-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-float32-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-float64-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-float64-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-int16-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-int16-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-int32-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-int32-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-int64-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-int64-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-int8-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-int8-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-interval-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-interval-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-mi-with-dt64tz-level-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-mi-with-dt64tz-level-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-multi-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-multi-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-nullable_bool-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-nullable_bool-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-nullable_float-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-nullable_float-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-nullable_int-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-nullable_int-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-nullable_uint-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-nullable_uint-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-object-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-object-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-period-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-period-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-range-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-range-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-repeats-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-repeats-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-string-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-string-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-string-pyarrow-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-string-pyarrow-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-string-python-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-string-python-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-timedelta-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-timedelta-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-tuples-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-tuples-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-uint16-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-uint16-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-uint32-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-uint32-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-uint64-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-uint64-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-uint8-index-None]",
    "tests/base/test_value_counts.py::test_value_counts_null[series-with-uint8-index-nan]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[index-ms]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[index-ns]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[index-s]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[index-us]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[series-ms]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[series-ns]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[series-s]",
    "tests/base/test_value_counts.py::test_value_counts_timedelta64[series-us]",
    "tests/computation/test_eval.py::TestAlignment::test_performance_warning_for_poor_alignment[numexpr-pandas]",
    "tests/computation/test_eval.py::TestAlignment::test_performance_warning_for_poor_alignment[numexpr-python]",
    "tests/computation/test_eval.py::TestEval::test_disallow_python_keywords",
    "tests/computation/test_eval.py::TestMath::test_binary_functions[arctan2]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[abs]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arccos]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arccosh]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arcsin]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arcsinh]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arctan]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[arctanh]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[ceil]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[cos]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[cosh]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[exp]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[expm1]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[floor]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[log10]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[log1p]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[log]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[sin]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[sinh]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[sqrt]",
    "tests/computation/test_eval.py::TestMath::test_unary_functions[tanh]",
    "tests/computation/test_eval.py::TestOperations::test_query_inplace",
    "tests/computation/test_eval.py::test_eval_no_support_column_name[False]",
    "tests/computation/test_eval.py::test_eval_no_support_column_name[True]",
    "tests/computation/test_eval.py::test_numexpr_option_respected[False-python]",
    "tests/computation/test_eval.py::test_set_inplace",
    "tests/config/test_config.py::TestConfig::test_attribute_access",
    "tests/config/test_config.py::TestConfig::test_callback",
    "tests/config/test_config.py::TestConfig::test_case_insensitive",
    "tests/config/test_config.py::TestConfig::test_config_prefix",
    "tests/config/test_config.py::TestConfig::test_deprecate_option",
    "tests/config/test_config.py::TestConfig::test_describe_option",
    "tests/config/test_config.py::TestConfig::test_dictwrapper_getattr",
    "tests/config/test_config.py::TestConfig::test_get_option",
    "tests/config/test_config.py::TestConfig::test_is_one_of_factory",
    "tests/config/test_config.py::TestConfig::test_option_context_scope",
    "tests/config/test_config.py::TestConfig::test_register_option",
    "tests/config/test_config.py::TestConfig::test_reset_option",
    "tests/config/test_config.py::TestConfig::test_reset_option_all",
    "tests/config/test_config.py::TestConfig::test_set_ContextManager",
    "tests/config/test_config.py::TestConfig::test_set_option",
    "tests/config/test_config.py::TestConfig::test_set_option_empty_args",
    "tests/config/test_config.py::TestConfig::test_set_option_invalid_single_argument_type",
    "tests/config/test_config.py::TestConfig::test_set_option_multiple",
    "tests/config/test_config.py::TestConfig::test_set_option_uneven_args",
    "tests/config/test_config.py::TestConfig::test_validation",
    "tests/copy_view/index/test_index.py::test_index_from_series_copy",
    "tests/copy_view/index/test_index.py::test_index_to_frame",
    "tests/copy_view/index/test_index.py::test_set_index_update_column",
    "tests/copy_view/test_array.py::test_dataframe_array_ea_dtypes",
    "tests/copy_view/test_array.py::test_dataframe_array_string_dtype",
    "tests/copy_view/test_array.py::test_dataframe_multiple_numpy_dtypes",
    "tests/copy_view/test_array.py::test_dataframe_single_block_copy_true",
    "tests/copy_view/test_array.py::test_dataframe_values[values]",
    "tests/copy_view/test_array.py::test_series_array_ea_dtypes",
    "tests/copy_view/test_array.py::test_series_to_numpy",
    "tests/copy_view/test_array.py::test_series_values[values]",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[Int64-Int64]",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[Int64-int64]",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[int64-Int64]",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[int64-int64]",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[int64[pyarrow]-Int64]",
    "tests/copy_view/test_astype.py::test_astype_avoids_copy[int64[pyarrow]-int64]",
    "tests/copy_view/test_astype.py::test_astype_dict_dtypes",
    "tests/copy_view/test_astype.py::test_astype_different_datetime_resos",
    "tests/copy_view/test_astype.py::test_astype_different_target_dtype[Int32]",
    "tests/copy_view/test_astype.py::test_astype_different_target_dtype[float64]",
    "tests/copy_view/test_astype.py::test_astype_different_target_dtype[int32[pyarrow]]",
    "tests/copy_view/test_astype.py::test_astype_different_target_dtype[int32]",
    "tests/copy_view/test_astype.py::test_astype_numpy_to_ea",
    "tests/copy_view/test_astype.py::test_astype_single_dtype",
    "tests/copy_view/test_astype.py::test_astype_string_and_object[object-string]",
    "tests/copy_view/test_astype.py::test_astype_string_and_object[string-object]",
    "tests/copy_view/test_astype.py::test_astype_string_and_object_update_original[object-string]",
    "tests/copy_view/test_astype.py::test_astype_string_and_object_update_original[string-object]",
    "tests/copy_view/test_astype.py::test_convert_dtypes",
    "tests/copy_view/test_chained_assignment_deprecation.py::test_series_setitem[0]",
    "tests/copy_view/test_chained_assignment_deprecation.py::test_series_setitem[indexer1]",
    "tests/copy_view/test_chained_assignment_deprecation.py::test_series_setitem[indexer2]",
    "tests/copy_view/test_chained_assignment_deprecation.py::test_series_setitem[indexer3]",
    "tests/copy_view/test_clip.py::test_clip",
    "tests/copy_view/test_clip.py::test_clip_inplace",
    "tests/copy_view/test_clip.py::test_clip_inplace_reference",
    "tests/copy_view/test_clip.py::test_clip_inplace_reference_no_op",
    "tests/copy_view/test_clip.py::test_clip_no_op",
    "tests/copy_view/test_constructors.py::test_dataframe_constructor_mgr_or_df[None-False]",
    "tests/copy_view/test_constructors.py::test_dataframe_constructor_mgr_or_df[None-True]",
    "tests/copy_view/test_constructors.py::test_dataframe_constructor_mgr_or_df[columns1-False]",
    "tests/copy_view/test_constructors.py::test_dataframe_constructor_mgr_or_df[columns1-True]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-None-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-None-None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-None-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-index1-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-index1-None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[None-index1-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-None-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-None-None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-None-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-index1-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-index1-None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns1-index1-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns2-None-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns2-None-None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns2-None-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns2-index1-Int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns2-index1-None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series[columns2-index1-int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series_with_dtype[None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series_with_dtype[index1]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series_with_reindex[None]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_dict_of_series_with_reindex[int64]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_records_with_dataframe",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data0-None-Index]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data0-None-Series]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data1-int64-Index]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data1-int64-Series]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data2-None-Index]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data2-None-Series]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data2-object-Index]",
    "tests/copy_view/test_constructors.py::test_dataframe_from_series_or_index[data2-object-Series]",
    "tests/copy_view/test_constructors.py::test_frame_from_dict_of_index",
    "tests/copy_view/test_constructors.py::test_frame_from_numpy_array[False]",
    "tests/copy_view/test_constructors.py::test_frame_from_numpy_array[None]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr0-None-None-False]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr0-None-None-True]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr0-None-int64-False]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr0-None-int64-True]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr0-idx1-None-False]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr0-idx1-None-True]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr0-idx1-int64-False]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr0-idx1-int64-True]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr1-None-None-False]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr1-None-None-True]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr1-None-int64-False]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr1-None-int64-True]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr1-idx1-None-False]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr1-idx1-None-True]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr1-idx1-int64-False]",
    "tests/copy_view/test_constructors.py::test_series_from_array[arr1-idx1-int64-True]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[None-None-False]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[None-None-True]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[None-int64-False]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[None-int64-True]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[idx1-None-False]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[idx1-None-True]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[idx1-int64-False]",
    "tests/copy_view/test_constructors.py::test_series_from_block_manager[idx1-int64-True]",
    "tests/copy_view/test_constructors.py::test_series_from_series[None]",
    "tests/copy_view/test_constructors.py::test_series_from_series[int64]",
    "tests/copy_view/test_constructors.py::test_series_from_series_with_reindex",
    "tests/copy_view/test_core_functionalities.py::test_assigning_to_same_variable_removes_references",
    "tests/copy_view/test_core_functionalities.py::test_setitem_dont_track_unnecessary_references",
    "tests/copy_view/test_core_functionalities.py::test_setitem_with_view_copies",
    "tests/copy_view/test_core_functionalities.py::test_setitem_with_view_invalidated_does_not_copy",
    "tests/copy_view/test_functions.py::test_concat_copy_keyword[False]",
    "tests/copy_view/test_functions.py::test_concat_copy_keyword[None]",
    "tests/copy_view/test_functions.py::test_concat_copy_keyword[True]",
    "tests/copy_view/test_functions.py::test_concat_frames",
    "tests/copy_view/test_functions.py::test_concat_frames_chained",
    "tests/copy_view/test_functions.py::test_concat_frames_updating_input",
    "tests/copy_view/test_functions.py::test_concat_mixed_series_frame",
    "tests/copy_view/test_functions.py::test_concat_series",
    "tests/copy_view/test_functions.py::test_concat_series_chained",
    "tests/copy_view/test_functions.py::test_concat_series_updating_input",
    "tests/copy_view/test_functions.py::test_join_multiple_dataframes_on_key",
    "tests/copy_view/test_functions.py::test_join_on_key[object]",
    "tests/copy_view/test_functions.py::test_join_on_key[object]",
    "tests/copy_view/test_functions.py::test_join_on_key[str]",
    "tests/copy_view/test_functions.py::test_join_on_key[str]",
    "tests/copy_view/test_functions.py::test_merge_copy_keyword[False]",
    "tests/copy_view/test_functions.py::test_merge_copy_keyword[None]",
    "tests/copy_view/test_functions.py::test_merge_copy_keyword[True]",
    "tests/copy_view/test_functions.py::test_merge_on_index",
    "tests/copy_view/test_functions.py::test_merge_on_key[<lambda>0]",
    "tests/copy_view/test_functions.py::test_merge_on_key[<lambda>1]",
    "tests/copy_view/test_functions.py::test_merge_on_key_enlarging_one[<lambda>-left]",
    "tests/copy_view/test_functions.py::test_merge_on_key_enlarging_one[<lambda>-right]",
    "tests/copy_view/test_indexing.py::test_column_as_series[nullable]",
    "tests/copy_view/test_indexing.py::test_column_as_series[numpy]",
    "tests/copy_view/test_indexing.py::test_column_as_series_no_item_cache[nullable-getitem]",
    "tests/copy_view/test_indexing.py::test_column_as_series_no_item_cache[nullable-iloc]",
    "tests/copy_view/test_indexing.py::test_column_as_series_no_item_cache[nullable-loc]",
    "tests/copy_view/test_indexing.py::test_column_as_series_no_item_cache[numpy-getitem]",
    "tests/copy_view/test_indexing.py::test_column_as_series_no_item_cache[numpy-iloc]",
    "tests/copy_view/test_indexing.py::test_column_as_series_no_item_cache[numpy-loc]",
    "tests/copy_view/test_indexing.py::test_column_as_series_set_with_upcast[nullable]",
    "tests/copy_view/test_indexing.py::test_column_as_series_set_with_upcast[numpy]",
    "tests/copy_view/test_indexing.py::test_dataframe_add_column_from_series[nullable]",
    "tests/copy_view/test_indexing.py::test_dataframe_add_column_from_series[numpy]",
    "tests/copy_view/test_indexing.py::test_del_frame[nullable]",
    "tests/copy_view/test_indexing.py::test_del_frame[numpy]",
    "tests/copy_view/test_indexing.py::test_del_series[nullable]",
    "tests/copy_view/test_indexing.py::test_del_series[numpy]",
    "tests/copy_view/test_indexing.py::test_getitem_midx_slice",
    "tests/copy_view/test_indexing.py::test_loc_enlarging_with_dataframe",
    "tests/copy_view/test_indexing.py::test_midx_read_only_bool_indexer",
    "tests/copy_view/test_indexing.py::test_null_slice[nullable-getitem]",
    "tests/copy_view/test_indexing.py::test_null_slice[nullable-iloc-rows]",
    "tests/copy_view/test_indexing.py::test_null_slice[nullable-iloc]",
    "tests/copy_view/test_indexing.py::test_null_slice[nullable-loc-rows]",
    "tests/copy_view/test_indexing.py::test_null_slice[nullable-loc]",
    "tests/copy_view/test_indexing.py::test_null_slice[numpy-getitem]",
    "tests/copy_view/test_indexing.py::test_null_slice[numpy-iloc-rows]",
    "tests/copy_view/test_indexing.py::test_null_slice[numpy-iloc]",
    "tests/copy_view/test_indexing.py::test_null_slice[numpy-loc-rows]",
    "tests/copy_view/test_indexing.py::test_null_slice[numpy-loc]",
    "tests/copy_view/test_indexing.py::test_null_slice_series[nullable-getitem]",
    "tests/copy_view/test_indexing.py::test_null_slice_series[nullable-iloc]",
    "tests/copy_view/test_indexing.py::test_null_slice_series[nullable-loc]",
    "tests/copy_view/test_indexing.py::test_null_slice_series[numpy-getitem]",
    "tests/copy_view/test_indexing.py::test_null_slice_series[numpy-iloc]",
    "tests/copy_view/test_indexing.py::test_null_slice_series[numpy-loc]",
    "tests/copy_view/test_indexing.py::test_series_getitem_ellipsis",
    "tests/copy_view/test_indexing.py::test_series_getitem_slice[nullable]",
    "tests/copy_view/test_indexing.py::test_series_getitem_slice[numpy]",
    "tests/copy_view/test_indexing.py::test_series_midx_slice",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[nullable-iloc-array]",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[nullable-iloc-mask]",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[nullable-iloc-slice]",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[nullable-setitem-array]",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[nullable-setitem-mask]",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[nullable-setitem-slice]",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[numpy-iloc-array]",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[numpy-iloc-mask]",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[numpy-iloc-slice]",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[numpy-setitem-array]",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[numpy-setitem-mask]",
    "tests/copy_view/test_indexing.py::test_series_subset_set_with_indexer[numpy-setitem-slice]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer1-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer1-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer3-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer3-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer5-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-iloc-indexer5-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer0-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer0-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer2-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer2-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer4-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[mixed-block-loc-indexer4-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer1-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer1-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer3-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer3-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer5-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-iloc-indexer5-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer0-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer0-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer2-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer2-a]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer4-100]",
    "tests/copy_view/test_indexing.py::test_set_value_copy_only_necessary_column[single-block-loc-indexer4-a]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem[numpy-single-block-column-iloc-slice]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem[numpy-single-block-column-loc-slice]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_column[nullable-mixed-block]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_column[nullable-single-block]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_column[numpy-mixed-block]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_column[numpy-single-block]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_series[nullable-getitem]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_series[nullable-iloc]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_series[nullable-loc]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_series[nullable-long-chain]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_series[numpy-getitem]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_series[numpy-iloc]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_series[numpy-loc]",
    "tests/copy_view/test_indexing.py::test_subset_chained_getitem_series[numpy-long-chain]",
    "tests/copy_view/test_indexing.py::test_subset_chained_single_block_row",
    "tests/copy_view/test_indexing.py::test_subset_column_selection[nullable]",
    "tests/copy_view/test_indexing.py::test_subset_column_selection[numpy]",
    "tests/copy_view/test_indexing.py::test_subset_column_selection_modify_parent[nullable]",
    "tests/copy_view/test_indexing.py::test_subset_column_selection_modify_parent[numpy]",
    "tests/copy_view/test_indexing.py::test_subset_column_slice[numpy-single-block]",
    "tests/copy_view/test_indexing.py::test_subset_iloc_rows_columns[numpy-slice-slice-single-block]",
    "tests/copy_view/test_indexing.py::test_subset_loc_rows_columns[numpy-slice-slice-single-block]",
    "tests/copy_view/test_indexing.py::test_subset_row_slice[nullable]",
    "tests/copy_view/test_indexing.py::test_subset_row_slice[numpy]",
    "tests/copy_view/test_indexing.py::test_subset_set_column[nullable]",
    "tests/copy_view/test_indexing.py::test_subset_set_column_with_loc[nullable-mixed-block]",
    "tests/copy_view/test_indexing.py::test_subset_set_column_with_loc[nullable-single-block]",
    "tests/copy_view/test_indexing.py::test_subset_set_column_with_loc[numpy-mixed-block]",
    "tests/copy_view/test_indexing.py::test_subset_set_column_with_loc[numpy-single-block]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_column_indexer[nullable-array]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_column_indexer[nullable-mask]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_column_indexer[nullable-slice]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_mask[nullable]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_mask[numpy]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_row_indexer[nullable-iloc-array]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_row_indexer[nullable-iloc-mask]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_row_indexer[nullable-iloc-slice]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_row_indexer[nullable-setitem-mask]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_row_indexer[nullable-setitem-slice]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_row_indexer[numpy-iloc-array]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_row_indexer[numpy-iloc-mask]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_row_indexer[numpy-iloc-slice]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_row_indexer[numpy-setitem-mask]",
    "tests/copy_view/test_indexing.py::test_subset_set_with_row_indexer[numpy-setitem-slice]",
    "tests/copy_view/test_internals.py::test_consolidate",
    "tests/copy_view/test_internals.py::test_exponential_backoff",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs0-arr0-int64]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs0-arr0-int8]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs1-arr1-int64]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs1-arr1-int8]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs2-arr2-int64]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs2-arr2-int8]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs3-arr3-int64]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs3-arr3-int8]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs4-arr4-int64]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs4-arr4-int8]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs5-arr5-int64]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs5-arr5-int8]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs6-arr6-int64]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs6-arr6-int8]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs7-arr7-int64]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs7-arr7-int8]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs8-arr8-int64]",
    "tests/copy_view/test_internals.py::test_iset_splits_blocks_inplace[locs8-arr8-int8]",
    "tests/copy_view/test_interp_fillna.py::test_fillna",
    "tests/copy_view/test_interp_fillna.py::test_fillna_dict",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Float32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Float64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Int16]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Int32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Int64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[Int8]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[UInt16]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[UInt32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[UInt64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[UInt8]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[double[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[float[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[int16[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[int32[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[int64[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[int8[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[uint16[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[uint32[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[uint64[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_ea_noop_shares_memory[uint8[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace[False]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace[None]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Float32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Float64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Int16]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Int32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Int64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[Int8]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[UInt16]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[UInt32]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[UInt64]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[UInt8]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[double[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[float[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[int16[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[int32[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[int64[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[int8[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[uint16[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[uint32[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[uint64[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_ea_noop_shares_memory[uint8[pyarrow]]",
    "tests/copy_view/test_interp_fillna.py::test_fillna_inplace_reference",
    "tests/copy_view/test_interp_fillna.py::test_fillna_interval_inplace_reference",
    "tests/copy_view/test_interp_fillna.py::test_fillna_series_empty_arg_inplace",
    "tests/copy_view/test_interp_fillna.py::test_interp_fill_functions[bfill]",
    "tests/copy_view/test_interp_fillna.py::test_interp_fill_functions[ffill]",
    "tests/copy_view/test_interp_fillna.py::test_interp_fill_functions_inplace[Float64-bfill]",
    "tests/copy_view/test_interp_fillna.py::test_interp_fill_functions_inplace[Float64-ffill]",
    "tests/copy_view/test_interp_fillna.py::test_interp_fill_functions_inplace[float64-bfill]",
    "tests/copy_view/test_interp_fillna.py::test_interp_fill_functions_inplace[float64-ffill]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_cannot_with_object_dtype",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_cleaned_fill_method",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_downcast",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_downcast_reference_triggers_copy",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_inplace_no_reference_no_copy[vals0]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_inplace_no_reference_no_copy[vals1]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_inplace_with_refs[vals0]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_inplace_with_refs[vals1]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_no_op[linear]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_no_op[nearest]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_no_op[pad]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_object_convert_copies",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_object_convert_no_op",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_triggers_copy[vals0-bfill]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_triggers_copy[vals0-ffill]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_triggers_copy[vals1-bfill]",
    "tests/copy_view/test_interp_fillna.py::test_interpolate_triggers_copy[vals1-ffill]",
    "tests/copy_view/test_methods.py::test_add_prefix",
    "tests/copy_view/test_methods.py::test_add_suffix",
    "tests/copy_view/test_methods.py::test_align_copy_false",
    "tests/copy_view/test_methods.py::test_align_frame[<lambda>0]",
    "tests/copy_view/test_methods.py::test_align_frame[<lambda>1]",
    "tests/copy_view/test_methods.py::test_align_frame[<lambda>2]",
    "tests/copy_view/test_methods.py::test_align_with_series_copy_false",
    "tests/copy_view/test_methods.py::test_asfreq_noop",
    "tests/copy_view/test_methods.py::test_assign_drop_duplicates[assign]",
    "tests/copy_view/test_methods.py::test_assign_drop_duplicates[drop_duplicates]",
    "tests/copy_view/test_methods.py::test_between_time[obj0]",
    "tests/copy_view/test_methods.py::test_between_time[obj1]",
    "tests/copy_view/test_methods.py::test_chained_methods[reset_index]",
    "tests/copy_view/test_methods.py::test_copy",
    "tests/copy_view/test_methods.py::test_copy_shallow",
    "tests/copy_view/test_methods.py::test_drop_on_column",
    "tests/copy_view/test_methods.py::test_droplevel",
    "tests/copy_view/test_methods.py::test_dropna[0-5.5]",
    "tests/copy_view/test_methods.py::test_dropna[1-nan]",
    "tests/copy_view/test_methods.py::test_eval",
    "tests/copy_view/test_methods.py::test_eval_inplace",
    "tests/copy_view/test_methods.py::test_filter[filter_kwargs0]",
    "tests/copy_view/test_methods.py::test_filter[filter_kwargs1]",
    "tests/copy_view/test_methods.py::test_filter[filter_kwargs2]",
    "tests/copy_view/test_methods.py::test_frame_set_axis",
    "tests/copy_view/test_methods.py::test_get[a]",
    "tests/copy_view/test_methods.py::test_groupby_column_index_in_references",
    "tests/copy_view/test_methods.py::test_head_tail[<lambda>0]",
    "tests/copy_view/test_methods.py::test_head_tail[<lambda>1]",
    "tests/copy_view/test_methods.py::test_head_tail[<lambda>2]",
    "tests/copy_view/test_methods.py::test_head_tail[<lambda>3]",
    "tests/copy_view/test_methods.py::test_infer_objects",
    "tests/copy_view/test_methods.py::test_infer_objects_no_reference",
    "tests/copy_view/test_methods.py::test_infer_objects_reference",
    "tests/copy_view/test_methods.py::test_inplace_arithmetic_series",
    "tests/copy_view/test_methods.py::test_inplace_arithmetic_series_with_reference",
    "tests/copy_view/test_methods.py::test_insert_series",
    "tests/copy_view/test_methods.py::test_interpolate_creates_copy",
    "tests/copy_view/test_methods.py::test_isetitem",
    "tests/copy_view/test_methods.py::test_isetitem_frame",
    "tests/copy_view/test_methods.py::test_items",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[align-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[align-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[align-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[astype-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[astype-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[astype-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[infer_objects-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[infer_objects-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[infer_objects-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex_like-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex_like-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[reindex_like-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis0-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis0-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis0-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis1-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis1-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[rename_axis1-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_axis-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_axis-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_axis-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_flags-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_flags-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[set_flags-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[swapaxes-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[swapaxes-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[swapaxes-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[to_period-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[to_period-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[to_period-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[to_timestamp-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[to_timestamp-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[to_timestamp-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[truncate-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[truncate-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[truncate-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[tz_convert-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[tz_convert-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[tz_convert-True]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[tz_localize-False]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[tz_localize-None]",
    "tests/copy_view/test_methods.py::test_methods_copy_keyword[tz_localize-True]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[astype-False]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[reindex-False]",
    "tests/copy_view/test_methods.py::test_methods_series_copy_keyword[rename-False]",
    "tests/copy_view/test_methods.py::test_pipe",
    "tests/copy_view/test_methods.py::test_pipe_modify_df",
    "tests/copy_view/test_methods.py::test_pop",
    "tests/copy_view/test_methods.py::test_putmask[Int64]",
    "tests/copy_view/test_methods.py::test_putmask[int64]",
    "tests/copy_view/test_methods.py::test_putmask_aligns_rhs_no_reference[Float64]",
    "tests/copy_view/test_methods.py::test_putmask_aligns_rhs_no_reference[float64]",
    "tests/copy_view/test_methods.py::test_putmask_dont_copy_some_blocks[5-False-None]",
    "tests/copy_view/test_methods.py::test_putmask_no_reference[Int64]",
    "tests/copy_view/test_methods.py::test_putmask_no_reference[int64]",
    "tests/copy_view/test_methods.py::test_reindex_columns",
    "tests/copy_view/test_methods.py::test_reindex_like",
    "tests/copy_view/test_methods.py::test_reindex_rows[copy]",
    "tests/copy_view/test_methods.py::test_reindex_rows[identical]",
    "tests/copy_view/test_methods.py::test_reindex_rows[values]",
    "tests/copy_view/test_methods.py::test_reindex_rows[view]",
    "tests/copy_view/test_methods.py::test_rename_axis[kwargs0]",
    "tests/copy_view/test_methods.py::test_rename_axis[kwargs1]",
    "tests/copy_view/test_methods.py::test_rename_columns",
    "tests/copy_view/test_methods.py::test_rename_columns_modify_parent",
    "tests/copy_view/test_methods.py::test_reorder_levels",
    "tests/copy_view/test_methods.py::test_reset_index",
    "tests/copy_view/test_methods.py::test_round[-1]",
    "tests/copy_view/test_methods.py::test_round[0]",
    "tests/copy_view/test_methods.py::test_round[1]",
    "tests/copy_view/test_methods.py::test_select_dtypes",
    "tests/copy_view/test_methods.py::test_series_view",
    "tests/copy_view/test_methods.py::test_set_flags",
    "tests/copy_view/test_methods.py::test_set_index",
    "tests/copy_view/test_methods.py::test_shift_columns",
    "tests/copy_view/test_methods.py::test_shift_index",
    "tests/copy_view/test_methods.py::test_shift_no_op",
    "tests/copy_view/test_methods.py::test_shift_rows_freq",
    "tests/copy_view/test_methods.py::test_sort_values[obj1-kwargs1]",
    "tests/copy_view/test_methods.py::test_sort_values_inplace[obj0-kwargs0]",
    "tests/copy_view/test_methods.py::test_sort_values_inplace[obj1-kwargs1]",
    "tests/copy_view/test_methods.py::test_squeeze",
    "tests/copy_view/test_methods.py::test_swapaxes_noop[columns]",
    "tests/copy_view/test_methods.py::test_swapaxes_noop[index]",
    "tests/copy_view/test_methods.py::test_swapaxes_single_block",
    "tests/copy_view/test_methods.py::test_to_frame",
    "tests/copy_view/test_methods.py::test_to_period[obj1]",
    "tests/copy_view/test_methods.py::test_to_timestamp[obj1]",
    "tests/copy_view/test_methods.py::test_transpose[False]",
    "tests/copy_view/test_methods.py::test_transpose[True]",
    "tests/copy_view/test_methods.py::test_transpose_copy_keyword[False]",
    "tests/copy_view/test_methods.py::test_transpose_copy_keyword[None]",
    "tests/copy_view/test_methods.py::test_transpose_copy_keyword[True]",
    "tests/copy_view/test_methods.py::test_transpose_different_dtypes",
    "tests/copy_view/test_methods.py::test_transpose_ea_single_column",
    "tests/copy_view/test_methods.py::test_truncate[kwargs0]",
    "tests/copy_view/test_methods.py::test_truncate[kwargs1]",
    "tests/copy_view/test_methods.py::test_update_frame",
    "tests/copy_view/test_methods.py::test_update_series",
    "tests/copy_view/test_methods.py::test_where_mask_noop_on_single_column[<lambda>0-Int64-10]",
    "tests/copy_view/test_methods.py::test_where_mask_noop_on_single_column[<lambda>0-int64-10.5]",
    "tests/copy_view/test_methods.py::test_where_mask_noop_on_single_column[<lambda>1-Int64-10]",
    "tests/copy_view/test_methods.py::test_where_mask_noop_on_single_column[<lambda>1-int64-10.5]",
    "tests/copy_view/test_methods.py::test_xs[mixed-block-1-a]",
    "tests/copy_view/test_methods.py::test_xs[single-block-0-0]",
    "tests/copy_view/test_methods.py::test_xs[single-block-1-a]",
    "tests/copy_view/test_methods.py::test_xs_multiindex[l1-0-0]",
    "tests/copy_view/test_methods.py::test_xs_multiindex[l1-0-1]",
    "tests/copy_view/test_replace.py::test_masking_inplace[mask]",
    "tests/copy_view/test_replace.py::test_masking_inplace[where]",
    "tests/copy_view/test_replace.py::test_replace[replace_kwargs0]",
    "tests/copy_view/test_replace.py::test_replace[replace_kwargs1]",
    "tests/copy_view/test_replace.py::test_replace_categorical[1.5]",
    "tests/copy_view/test_replace.py::test_replace_categorical[1]",
    "tests/copy_view/test_replace.py::test_replace_categorical_inplace[1.5]",
    "tests/copy_view/test_replace.py::test_replace_categorical_inplace[1]",
    "tests/copy_view/test_replace.py::test_replace_categorical_inplace_reference[1-1]",
    "tests/copy_view/test_replace.py::test_replace_categorical_inplace_reference[1-to_replace1]",
    "tests/copy_view/test_replace.py::test_replace_categorical_inplace_reference[1.5-1]",
    "tests/copy_view/test_replace.py::test_replace_categorical_inplace_reference[1.5-to_replace1]",
    "tests/copy_view/test_replace.py::test_replace_coerce_single_column",
    "tests/copy_view/test_replace.py::test_replace_empty_list",
    "tests/copy_view/test_replace.py::test_replace_inplace[1.5]",
    "tests/copy_view/test_replace.py::test_replace_inplace[to_replace1]",
    "tests/copy_view/test_replace.py::test_replace_inplace[to_replace2]",
    "tests/copy_view/test_replace.py::test_replace_inplace_reference[1.5]",
    "tests/copy_view/test_replace.py::test_replace_inplace_reference[to_replace1]",
    "tests/copy_view/test_replace.py::test_replace_inplace_reference_no_op[100.5]",
    "tests/copy_view/test_replace.py::test_replace_inplace_reference_no_op[a]",
    "tests/copy_view/test_replace.py::test_replace_list_categorical",
    "tests/copy_view/test_replace.py::test_replace_list_inplace_refs_categorical",
    "tests/copy_view/test_replace.py::test_replace_list_multiple_elements_inplace",
    "tests/copy_view/test_replace.py::test_replace_list_none",
    "tests/copy_view/test_replace.py::test_replace_list_none_inplace_refs",
    "tests/copy_view/test_replace.py::test_replace_listlike",
    "tests/copy_view/test_replace.py::test_replace_listlike_inplace",
    "tests/copy_view/test_replace.py::test_replace_mask_all_false_second_block",
    "tests/copy_view/test_replace.py::test_replace_object_list_inplace[None]",
    "tests/copy_view/test_replace.py::test_replace_object_list_inplace[d]",
    "tests/copy_view/test_replace.py::test_replace_regex_inplace",
    "tests/copy_view/test_replace.py::test_replace_regex_inplace_no_op",
    "tests/copy_view/test_replace.py::test_replace_regex_inplace_refs",
    "tests/copy_view/test_replace.py::test_replace_to_replace_wrong_dtype",
    "tests/copy_view/test_setitem.py::test_set_column_with_array",
    "tests/copy_view/test_setitem.py::test_set_column_with_index",
    "tests/copy_view/test_setitem.py::test_set_column_with_series",
    "tests/copy_view/test_setitem.py::test_set_columns_with_dataframe",
    "tests/copy_view/test_setitem.py::test_setitem_series_no_copy",
    "tests/copy_view/test_setitem.py::test_setitem_series_no_copy_single_block",
    "tests/copy_view/test_setitem.py::test_setitem_series_no_copy_split_block",
    "tests/copy_view/test_util.py::test_get_array_masked",
    "tests/copy_view/test_util.py::test_get_array_numpy",
    "tests/dtypes/cast/test_construct_from_scalar.py::test_cast_1d_array_like_from_timedelta",
    "tests/dtypes/cast/test_construct_from_scalar.py::test_cast_1d_array_like_from_timestamp",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right0-left0]",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right1-left1]",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right2-left2]",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right3-left3]",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right4-left4]",
    "tests/dtypes/cast/test_find_common_type.py::test_interval_dtype[right5-left5]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes16-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes17-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes18-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes19-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes20-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes21-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes22-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes23-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes24-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes25-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_numpy_dtypes[source_dtypes30-object]",
    "tests/dtypes/cast/test_find_common_type.py::test_raises_empty_input",
    "tests/dtypes/test_common.py::TestNumpyEADtype::test_categorical_dtype",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtype::test_constructor_invalid",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtype::test_from_values_or_dtype[None-None-None-None-expected0]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtype::test_repr_range_categories",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_dtype_equality_requires_categories",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_equality[False-False]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_equality[False-None]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_equality[False-True]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_equality[None-False]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_equality[None-None]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_equality[None-True]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_equality[True-False]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_categorical_equality[True-True]",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_from_categorical_dtype_identity",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_invalid_raises",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_nan_invalid",
    "tests/dtypes/test_dtypes.py::TestCategoricalDtypeParametrized::test_non_unique_invalid",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_basic_dtype",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_closed_mismatch",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_closed_must_match",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction[Interval[int64]]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction[interval[int64]]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_allows_closed_none[Interval[int64]]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_allows_closed_none[int64]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_allows_closed_none[interval[int64]]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_allows_closed_none[subtype3]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_from_string",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_generic[Interval]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_generic[interval]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_construction_not_supported[<U10]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_equality",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_hash_vs_equality",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_name_repr_generic[Interval]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_name_repr_generic[interval]",
    "tests/dtypes/test_dtypes.py::TestIntervalDtype::test_unpickling_without_closed",
    "tests/dtypes/test_dtypes.py::test_registry[CategoricalDtype]",
    "tests/dtypes/test_dtypes.py::test_registry[DatetimeTZDtype]",
    "tests/dtypes/test_dtypes.py::test_registry[IntervalDtype]",
    "tests/dtypes/test_dtypes.py::test_registry[PeriodDtype]",
    "tests/dtypes/test_dtypes.py::test_registry_find[category-expected5]",
    "tests/dtypes/test_dtypes.py::test_registry_find[datetime64[ns, US/Eastern]-expected6]",
    "tests/dtypes/test_dtypes.py::test_registry_find[interval-expected1]",
    "tests/dtypes/test_dtypes.py::test_registry_find[interval[datetime64[ns], left]-expected3]",
    "tests/dtypes/test_dtypes.py::test_registry_find[interval[int64, neither]-expected2]",
    "tests/dtypes/test_dtypes.py::test_registry_find[period[D]-expected4]",
    "tests/dtypes/test_generic.py::test_setattr_warnings",
    "tests/dtypes/test_inference.py::TestTypeInference::test_date",
    "tests/dtypes/test_inference.py::test_is_scipy_sparse[dok]",
    "tests/dtypes/test_missing.py::test_array_equivalent_series[val5]",
    "tests/dtypes/test_missing.py::test_array_equivalent_series[val6]",
    "tests/dtypes/test_missing.py::test_array_equivalent_series[val7]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_astype_string[string[pyarrow]]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_astype_string[string[python]]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__eq__-Index]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__ge__-Index]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__gt__-Index]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__le__-Index]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__lt__-Index]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_direct_arith_with_ndframe_returns_not_implemented[__ne__-Index]",
    "tests/extension/decimal/test_decimal.py::TestDecimalArray::test_grouping_grouper",
    "tests/extension/json/test_json.py::TestJSONArray::test_astype_string[string[pyarrow]]",
    "tests/extension/json/test_json.py::TestJSONArray::test_astype_string[string[python]]",
    "tests/extension/json/test_json.py::TestJSONArray::test_concat_mixed_dtypes",
    "tests/extension/json/test_json.py::TestJSONArray::test_grouping_grouper",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cummax-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cummax-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cummin-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cummin-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cumprod-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[bool-cumprod-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[decimal128(7, 3)-cummax-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[decimal128(7, 3)-cummin-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_accumulate_series[decimal128(7, 3)-cumsum-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_argsort_missing[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[bool-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[bool-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[date32[day]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[date32[day]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[date64[ms]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[date64[ms]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[decimal128(7, 3)-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[decimal128(7, 3)-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[decimal128(7, 3)-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[decimal128(7, 3)-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[decimal128(7, 3)-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[decimal128(7, 3)-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[decimal128(7, 3)-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[decimal128(7, 3)-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[decimal128(7, 3)-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[decimal128(7, 3)-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[decimal128(7, 3)-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[double-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[double-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[ms]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[ms]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[ms]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[ms]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[ns]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[ns]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[ns]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[ns]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[s]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[s]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[s]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[s]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[us]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[us]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[us]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[duration[us]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[float-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[float-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int16-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int16-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int16-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int32-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int32-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int32-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int64-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int64-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int64-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int8-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int8-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[int8-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ms, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ms, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ms, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ms, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ms, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ms, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ms]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ms]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ns, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ns, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ns, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ns, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ns, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ns, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ns]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[ns]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[s, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[s, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[s, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[s, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[s, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[s, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[s]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[s]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[us, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[us, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[us, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[us, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[us, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[us, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[us]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[timestamp[us]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint16-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint16-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint16-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint32-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint32-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint32-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint64-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint8-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint8-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_frame_with_scalar[uint8-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__rpow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[bool-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[date32[day]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[date32[day]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[date64[ms]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[date64[ms]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[decimal128(7, 3)-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[decimal128(7, 3)-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[decimal128(7, 3)-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[decimal128(7, 3)-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[decimal128(7, 3)-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[decimal128(7, 3)-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[decimal128(7, 3)-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[decimal128(7, 3)-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[decimal128(7, 3)-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[double-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ms]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ms]-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ms]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ms]-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ms]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ms]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ns]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ns]-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ns]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ns]-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ns]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[ns]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[s]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[s]-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[s]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[s]-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[s]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[s]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[us]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[us]-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[us]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[us]-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[us]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[duration[us]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__rfloordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[float-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int16-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int32-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int64-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[int8-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ms, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ms, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ms, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ms, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ms, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ms, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ms]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ms]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ns, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ns, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ns, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ns, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ns, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ns, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ns]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[ns]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[s, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[s, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[s, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[s, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[s, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[s, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[s]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[s]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[us, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[us, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[us, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[us, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[us, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[us, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[us]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[timestamp[us]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint16-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint16-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint16-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint16-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint16-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint16-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint16-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint16-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint16-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint16-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint32-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint32-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint32-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint32-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint32-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint32-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint32-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint32-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint32-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint32-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint64-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint64-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint64-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint64-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint64-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint64-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint64-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint64-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint64-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint64-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint8-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint8-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint8-__mod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint8-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint8-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint8-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint8-__rmod__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint8-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint8-__rtruediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_array[uint8-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[date32[day]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[date32[day]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[date64[ms]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[date64[ms]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[decimal128(7, 3)-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[decimal128(7, 3)-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[decimal128(7, 3)-__mul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[decimal128(7, 3)-__pow__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[decimal128(7, 3)-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[decimal128(7, 3)-__rmul__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[decimal128(7, 3)-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[decimal128(7, 3)-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[decimal128(7, 3)-__truediv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ms]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ms]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ms]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ms]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ns]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ns]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ns]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[ns]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[s]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[s]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[s]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[s]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[us]-__add__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[us]-__radd__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[us]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[duration[us]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[int16-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[int32-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[int64-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[int8-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ms, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ms, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ms, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ms, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ms, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ms, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ms]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ms]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ns, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ns, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ns, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ns, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ns, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ns, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ns]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[ns]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[s, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[s, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[s, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[s, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[s, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[s, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[s]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[s]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[us, tz=US/Eastern]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[us, tz=US/Eastern]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[us, tz=US/Pacific]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[us, tz=US/Pacific]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[us, tz=UTC]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[us, tz=UTC]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[us]-__rsub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[timestamp[us]-__sub__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint16-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint32-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint64-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_arith_series_with_scalar[uint8-__floordiv__]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[binary]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[date32[day]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[time32[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[time32[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[time64[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[time64[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_interface_copy[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[binary]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[date32[day]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[time32[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[time32[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[time64[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[time64[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_array_type[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[binary-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[bool-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[date32[day]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[date64[ms]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[decimal128(7, 3)-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[double-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[duration[ms]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[duration[ns]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[duration[s]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[duration[us]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[float-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[int16-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[int32-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[int64-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[int8-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[string-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[time32[ms]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[time32[s]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[time64[ns]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[time64[us]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[ms, tz=US/Eastern]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[ms, tz=US/Pacific]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[ms, tz=UTC]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[ms]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[ns, tz=US/Eastern]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[ns, tz=US/Pacific]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[ns, tz=UTC]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[ns]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[s, tz=US/Eastern]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[s, tz=US/Pacific]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[s, tz=UTC]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[s]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[us, tz=US/Eastern]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[us, tz=US/Pacific]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[us, tz=UTC]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[timestamp[us]-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[uint16-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[uint32-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[uint64-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_own_type[uint8-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[decimal128(7, 3)-string[pyarrow]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[decimal128(7, 3)-string[python]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[ms]-string[pyarrow]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[ms]-string[python]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[ns]-string[pyarrow]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[ns]-string[python]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[s]-string[pyarrow]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[s]-string[python]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[us]-string[pyarrow]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_astype_string[duration[us]-string[python]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_add[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_add[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_add[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_add[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_combine_le[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ms]-eq]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ms]-ge]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ms]-gt]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ms]-le]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ms]-lt]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ms]-ne]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ns]-eq]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ns]-ge]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ns]-gt]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ns]-le]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ns]-lt]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[ns]-ne]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[s]-eq]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[s]-ge]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[s]-gt]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[s]-le]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[s]-lt]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[s]-ne]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[us]-eq]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[us]-ge]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[us]-gt]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[us]-le]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[us]-lt]",
    "tests/extension/test_arrow.py::TestArrowArray::test_compare_array[duration[us]-ne]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_construct_empty_dataframe[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[bool--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[bool--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[bool-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[bool-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[bool-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[bool-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[decimal128(7, 3)--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[decimal128(7, 3)-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[decimal128(7, 3)-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[double--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[double--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[double-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[double-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[double-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[double-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ms]--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ms]--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ms]-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ms]-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ms]-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ms]-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ns]--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ns]--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ns]-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ns]-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ns]-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[ns]-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[s]--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[s]--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[s]-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[s]-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[s]-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[s]-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[us]--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[us]--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[us]-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[us]-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[us]-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[duration[us]-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[float--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[float--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[float-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[float-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[float-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[float-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int16--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int16--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int16-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int16-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int16-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int16-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int32--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int32--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int32-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int32-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int32-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int32-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int64--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int64--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int64-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int64-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int64-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int64-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int8--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int8--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int8-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int8-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int8-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[int8-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[string--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[string--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[string-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[string-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[string-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[string-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ms]--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ms]--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ms]-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ms]-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ms]-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ms]-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ns]--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ns]--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ns]-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ns]-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ns]-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[ns]-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[s]--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[s]--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[s]-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[s]-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[s]-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[s]-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[us]--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[us]--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[us]-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[us]-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[us]-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[timestamp[us]-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint16--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint16--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint16-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint16-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint16-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint16-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint32--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint32--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint32-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint32-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint32-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint32-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint64--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint64--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint64-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint64-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint64-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint64-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint8--2-indices0-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint8--2-indices0-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint8-0-indices1-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint8-0-indices1-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint8-2-indices2-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_container_shift[uint8-2-indices2-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_getitem_scalar[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[binary]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[date32[day]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[time32[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[time32[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[time64[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[time64[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_grouping_grouper[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_invert[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_memory_usage[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[decimal128(7, 3)-mean-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[decimal128(7, 3)-prod-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[decimal128(7, 3)-sum-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[float-mean-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[float-prod-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[float-std-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[float-sum-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_frame[float-var-True]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_boolean[timestamp[ms]-any-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_boolean[timestamp[ns]-any-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_boolean[timestamp[s]-any-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reduce_series_boolean[timestamp[us]-any-False]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_reindex_non_na_fill_value[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[binary]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[date32[day]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[time32[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[time32[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[time64[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[time64[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_no_data_with_index[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_na_with_index[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_series_constructor_scalar_with_index[date64[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_set_frame_expand_extension_with_regular[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_expand_columns[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[bool]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[string]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_setitem_frame_2d_values[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[double]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[duration[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[duration[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[duration[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[duration[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[float]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[int16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[int32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[int64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[int8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[ms]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[ns]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[s]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[timestamp[us]]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[uint16]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[uint32]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[uint64]",
    "tests/extension/test_arrow.py::TestArrowArray::test_to_numpy[uint8]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[binary-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[bool-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[bool-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[bool-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[date32[day]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[date64[ms]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[decimal128(7, 3)-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[decimal128(7, 3)-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[decimal128(7, 3)-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[double-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[double-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[double-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[ms]-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[ms]-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[ms]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[ns]-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[ns]-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[ns]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[s]-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[s]-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[s]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[us]-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[us]-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[duration[us]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[float-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[float-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[float-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int16-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int16-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int16-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int32-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int32-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int32-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int64-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int64-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int64-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int8-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int8-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[int8-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[string-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[time32[ms]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[time32[s]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[time64[ns]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[time64[us]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[ms, tz=US/Eastern]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[ms, tz=US/Pacific]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[ms, tz=UTC]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[ms]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[ns, tz=US/Eastern]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[ns, tz=US/Pacific]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[ns, tz=UTC]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[ns]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[s, tz=US/Eastern]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[s, tz=US/Pacific]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[s, tz=UTC]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[s]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[us, tz=US/Eastern]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[us, tz=US/Pacific]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[us, tz=UTC]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[timestamp[us]-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint16-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint16-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint16-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint32-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint32-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint32-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint64-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint64-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint64-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint8-absolute]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint8-negative]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unary_ufunc_dunder_equivalence[uint8-positive]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[bool-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[bool-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[bool-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[bool-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[bool-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[bool-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[double-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[double-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[double-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[double-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[double-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[double-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ms]-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ms]-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ms]-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ms]-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ms]-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ms]-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ns]-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ns]-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ns]-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ns]-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ns]-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[ns]-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[s]-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[s]-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[s]-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[s]-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[s]-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[s]-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[us]-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[us]-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[us]-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[us]-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[us]-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[duration[us]-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[float-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[float-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[float-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[float-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[float-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[float-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int16-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int16-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int16-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int16-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int16-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int16-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int32-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int32-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int32-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int32-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int32-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int32-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int64-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int64-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int64-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int64-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int64-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int64-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int8-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int8-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int8-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int8-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int8-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[int8-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[string-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[string-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[string-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[string-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[string-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[string-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ms]-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ms]-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ms]-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ms]-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ms]-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ms]-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ns]-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ns]-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ns]-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ns]-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ns]-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[ns]-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[s]-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[s]-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[s]-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[s]-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[s]-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[s]-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[us]-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[us]-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[us]-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[us]-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[us]-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[timestamp[us]-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint16-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint16-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint16-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint16-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint16-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint16-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint32-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint32-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint32-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint32-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint32-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint32-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint64-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint64-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint64-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint64-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint64-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint64-series-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint8-frame-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint8-frame-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint8-frame-index3]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint8-series-index1]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint8-series-index2]",
    "tests/extension/test_arrow.py::TestArrowArray::test_unstack[uint8-series-index3]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_and",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_and_scalar[None-expected0]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_and_scalar[other1-expected1]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_or",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_or_scalar[None-expected0]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_kleene_or_scalar[other1-expected1]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_logical_masked_numpy[__and__-True]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_logical_masked_numpy[__or__-True]",
    "tests/extension/test_arrow.py::TestLogicalOps::test_logical_masked_numpy[__xor__-False]",
    "tests/extension/test_arrow.py::test_arrow_true_division_large_divisor[int64[pyarrow]]",
    "tests/extension/test_arrow.py::test_arrow_true_division_large_divisor[uint64[pyarrow]]",
    "tests/extension/test_arrow.py::test_astype_errors_ignore",
    "tests/extension/test_arrow.py::test_bitwise[pa_type0]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type1]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type2]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type3]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type4]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type5]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type6]",
    "tests/extension/test_arrow.py::test_bitwise[pa_type7]",
    "tests/extension/test_arrow.py::test_boolean_reduce_series_all_null[all-False]",
    "tests/extension/test_arrow.py::test_boolean_reduce_series_all_null[any-False]",
    "tests/extension/test_arrow.py::test_comparison_not_propagating_arrow_error",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type10]",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type11]",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type13]",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type14]",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type15]",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type1]",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type2]",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type3]",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type5]",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type6]",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type7]",
    "tests/extension/test_arrow.py::test_describe_datetime_data[pa_type9]",
    "tests/extension/test_arrow.py::test_dt_components",
    "tests/extension/test_arrow.py::test_dt_day_month_name[day_name-Sunday]",
    "tests/extension/test_arrow.py::test_dt_day_month_name[month_name-January]",
    "tests/extension/test_arrow.py::test_dt_days_in_month[days_in_month]",
    "tests/extension/test_arrow.py::test_dt_is_month_start_end",
    "tests/extension/test_arrow.py::test_dt_is_quarter_start_end",
    "tests/extension/test_arrow.py::test_dt_is_year_start_end",
    "tests/extension/test_arrow.py::test_dt_isocalendar",
    "tests/extension/test_arrow.py::test_dt_properties[day-2]",
    "tests/extension/test_arrow.py::test_dt_properties[day_of_year-2]",
    "tests/extension/test_arrow.py::test_dt_properties[dayofweek-0]",
    "tests/extension/test_arrow.py::test_dt_properties[dayofyear-2]",
    "tests/extension/test_arrow.py::test_dt_properties[hour-3]",
    "tests/extension/test_arrow.py::test_dt_properties[is_leap_year-False]",
    "tests/extension/test_arrow.py::test_dt_properties[microsecond-5]",
    "tests/extension/test_arrow.py::test_dt_properties[minute-4]",
    "tests/extension/test_arrow.py::test_dt_properties[month-1]",
    "tests/extension/test_arrow.py::test_dt_properties[nanosecond-6]",
    "tests/extension/test_arrow.py::test_dt_properties[quarter-1]",
    "tests/extension/test_arrow.py::test_dt_properties[second-7]",
    "tests/extension/test_arrow.py::test_dt_properties[weekday-0]",
    "tests/extension/test_arrow.py::test_dt_properties[year-2023]",
    "tests/extension/test_arrow.py::test_dt_strftime",
    "tests/extension/test_arrow.py::test_dt_to_pydatetime",
    "tests/extension/test_arrow.py::test_dt_to_pytimedelta",
    "tests/extension/test_arrow.py::test_dt_tz[US/Pacific]",
    "tests/extension/test_arrow.py::test_dt_tz[UTC]",
    "tests/extension/test_arrow.py::test_dt_tz_convert[ns]",
    "tests/extension/test_arrow.py::test_dt_tz_convert[us]",
    "tests/extension/test_arrow.py::test_dt_tz_convert_none",
    "tests/extension/test_arrow.py::test_dt_tz_localize_nonexistent[shift_backward-exp_date1]",
    "tests/extension/test_arrow.py::test_from_arrow_respecting_given_dtype",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[bool]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[double]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[duration[ms]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[duration[ns]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[duration[s]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[duration[us]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[float]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[int16]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[int32]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[int64]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[int8]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[string]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[timestamp[ms]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[timestamp[ns]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[timestamp[s]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[timestamp[us]]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[uint16]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[uint32]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[uint64]",
    "tests/extension/test_arrow.py::test_groupby_series_size_returns_pa_int[uint8]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[bool]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[duration[ms]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[duration[ns]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[duration[s]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[duration[us]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[timestamp[ms]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[timestamp[ns]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[timestamp[s]]",
    "tests/extension/test_arrow.py::test_interpolate_not_numeric[timestamp[us]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[date64[ms]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[ms, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[ms, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[ms, tz=UTC]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[ns, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[ns, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[ns, tz=UTC]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[s, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[s, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[s, tz=UTC]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[us, tz=US/Eastern]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[us, tz=US/Pacific]]",
    "tests/extension/test_arrow.py::test_mode_dropna_false_mode_na[timestamp[us, tz=UTC]]",
    "tests/extension/test_arrow.py::test_pickle_roundtrip[decimal128(7, 3)]",
    "tests/extension/test_arrow.py::test_pow_missing_operand",
    "tests/extension/test_arrow.py::test_quantile[bool-0.5-higher]",
    "tests/extension/test_arrow.py::test_quantile[bool-0.5-linear]",
    "tests/extension/test_arrow.py::test_quantile[bool-0.5-lower]",
    "tests/extension/test_arrow.py::test_quantile[bool-0.5-midpoint]",
    "tests/extension/test_arrow.py::test_quantile[bool-0.5-nearest]",
    "tests/extension/test_arrow.py::test_quantile[bool-quantile1-higher]",
    "tests/extension/test_arrow.py::test_quantile[bool-quantile1-linear]",
    "tests/extension/test_arrow.py::test_quantile[bool-quantile1-lower]",
    "tests/extension/test_arrow.py::test_quantile[bool-quantile1-midpoint]",
    "tests/extension/test_arrow.py::test_quantile[bool-quantile1-nearest]",
    "tests/extension/test_arrow.py::test_quantile[timestamp[ns]-0.5-nearest]",
    "tests/extension/test_arrow.py::test_quantile[timestamp[ns]-quantile1-higher]",
    "tests/extension/test_arrow.py::test_quantile[timestamp[ns]-quantile1-lower]",
    "tests/extension/test_arrow.py::test_round",
    "tests/extension/test_arrow.py::test_str_starts_ends_with_all_nulls_empty_tuple[endswith]",
    "tests/extension/test_arrow.py::test_str_starts_ends_with_all_nulls_empty_tuple[startswith]",
    "tests/extension/test_categorical.py::Test2DCompat::test_copy_order",
    "tests/extension/test_categorical.py::TestCategorical::test_argsort",
    "tests/extension/test_categorical.py::TestCategorical::test_argsort_missing",
    "tests/extension/test_categorical.py::TestCategorical::test_array_interface_copy",
    "tests/extension/test_categorical.py::TestCategorical::test_astype_own_type[False]",
    "tests/extension/test_categorical.py::TestCategorical::test_grouping_grouper",
    "tests/extension/test_categorical.py::TestCategorical::test_in_numeric_groupby",
    "tests/extension/test_categorical.py::TestCategorical::test_is_extension_array_dtype",
    "tests/extension/test_categorical.py::TestCategorical::test_reduce_series_boolean[all-False]",
    "tests/extension/test_categorical.py::TestCategorical::test_reduce_series_boolean[all-True]",
    "tests/extension/test_categorical.py::TestCategorical::test_reduce_series_boolean[any-False]",
    "tests/extension/test_categorical.py::TestCategorical::test_reduce_series_boolean[any-True]",
    "tests/extension/test_categorical.py::TestCategorical::test_reindex_non_na_fill_value",
    "tests/extension/test_categorical.py::TestCategorical::test_series_constructor",
    "tests/extension/test_categorical.py::TestCategorical::test_series_constructor_scalar_na_with_index",
    "tests/extension/test_categorical.py::TestCategorical::test_setitem_frame_2d_values",
    "tests/extension/test_categorical.py::TestCategorical::test_to_numpy",
    "tests/extension/test_categorical.py::TestCategorical::test_unstack[frame-index0]",
    "tests/extension/test_categorical.py::TestCategorical::test_unstack[frame-index1]",
    "tests/extension/test_categorical.py::TestCategorical::test_unstack[frame-index2]",
    "tests/extension/test_categorical.py::TestCategorical::test_unstack[frame-index3]",
    "tests/extension/test_categorical.py::TestCategorical::test_unstack[series-index0]",
    "tests/extension/test_categorical.py::TestCategorical::test_unstack[series-index1]",
    "tests/extension/test_categorical.py::TestCategorical::test_unstack[series-index2]",
    "tests/extension/test_categorical.py::TestCategorical::test_unstack[series-index3]",
    "tests/extension/test_common.py::test_ellipsis_index",
    "tests/extension/test_common.py::test_is_extension_array_dtype[dtype0]",
    "tests/extension/test_common.py::test_is_extension_array_dtype[dtype1]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_EA_types[US/Central-c]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_EA_types[US/Central-python]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_array_interface_copy[US/Central]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_astype_own_type[US/Central-False]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_grouping_grouper[US/Central]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_is_extension_array_dtype[US/Central]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_reduce_series_numeric[US/Central-mean-False]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_reduce_series_numeric[US/Central-mean-True]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_reduce_series_numeric[US/Central-std-False]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_reduce_series_numeric[US/Central-std-True]",
    "tests/extension/test_datetime.py::TestDatetimeArray::test_series_constructor[US/Central]",
    "tests/extension/test_interval.py::TestIntervalArray::test_EA_types[c]",
    "tests/extension/test_interval.py::TestIntervalArray::test_EA_types[python]",
    "tests/extension/test_interval.py::TestIntervalArray::test_argsort",
    "tests/extension/test_interval.py::TestIntervalArray::test_argsort_missing",
    "tests/extension/test_interval.py::TestIntervalArray::test_array_interface_copy",
    "tests/extension/test_interval.py::TestIntervalArray::test_astype_own_type[False]",
    "tests/extension/test_interval.py::TestIntervalArray::test_compare_array[eq]",
    "tests/extension/test_interval.py::TestIntervalArray::test_compare_array[ne]",
    "tests/extension/test_interval.py::TestIntervalArray::test_getitem_scalar",
    "tests/extension/test_interval.py::TestIntervalArray::test_grouping_grouper",
    "tests/extension/test_interval.py::TestIntervalArray::test_in_numeric_groupby",
    "tests/extension/test_interval.py::TestIntervalArray::test_is_extension_array_dtype",
    "tests/extension/test_interval.py::TestIntervalArray::test_series_constructor",
    "tests/extension/test_interval.py::TestIntervalArray::test_setitem_frame_2d_values",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[BooleanDtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[BooleanDtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Float32Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Float32Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Float64Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Float64Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int16Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int16Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int32Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int32Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int64Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int64Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int8Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[Int8Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt16Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt16Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt32Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt32Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt64Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt64Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt8Dtype-c]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_EA_types[UInt8Dtype-python]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[Float64Dtype-cummax-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[Float64Dtype-cummax-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[Float64Dtype-cummin-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[Float64Dtype-cummin-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt16Dtype-cumprod-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt16Dtype-cumprod-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt16Dtype-cumsum-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt16Dtype-cumsum-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt32Dtype-cumprod-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt32Dtype-cumprod-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt32Dtype-cumsum-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt32Dtype-cumsum-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt64Dtype-cumprod-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt64Dtype-cumprod-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt64Dtype-cumsum-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt64Dtype-cumsum-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt8Dtype-cumprod-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt8Dtype-cumprod-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt8Dtype-cumsum-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_accumulate_series[UInt8Dtype-cumsum-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort_missing[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort_missing[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort_missing[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort_missing[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort_missing[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort_missing[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort_missing[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort_missing[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort_missing[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort_missing[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_argsort_missing[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[Int16Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[Int16Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[Int32Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[Int32Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[Int64Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[Int64Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[Int8Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[Int8Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[UInt16Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[UInt16Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[UInt32Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[UInt32Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[UInt64Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[UInt64Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[UInt8Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_frame_with_scalar[UInt8Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__floordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__mod__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__pow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__rfloordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__rmod__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__rsub__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__sub__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[BooleanDtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Float32Dtype-__floordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Float64Dtype-__floordiv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int16Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int16Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int16Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int32Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int32Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int64Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int64Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int8Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int8Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[Int8Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt16Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt16Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt16Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt32Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt32Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt64Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt64Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt8Dtype-__rpow__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt8Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_array[UInt8Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[Int16Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[Int16Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[Int32Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[Int32Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[Int64Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[Int64Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[Int8Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[Int8Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[UInt16Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[UInt16Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[UInt32Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[UInt32Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[UInt64Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[UInt64Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[UInt8Dtype-__rtruediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_arith_series_with_scalar[UInt8Dtype-__truediv__]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_array_interface_copy[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[BooleanDtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Float32Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Float64Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Int16Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Int32Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Int64Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[Int8Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[UInt16Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[UInt32Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[UInt64Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_own_type[UInt8Dtype-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_str[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_string[Float32Dtype-string[pyarrow]]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_string[Float32Dtype-string[python]]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_string[Float64Dtype-string[pyarrow]]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_astype_string[Float64Dtype-string[python]]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[BooleanDtype--2-indices0-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[BooleanDtype--2-indices0-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[BooleanDtype-0-indices1-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[BooleanDtype-0-indices1-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[BooleanDtype-2-indices2-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[BooleanDtype-2-indices2-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float32Dtype--2-indices0-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float32Dtype--2-indices0-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float32Dtype-0-indices1-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float32Dtype-0-indices1-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float32Dtype-2-indices2-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float32Dtype-2-indices2-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float64Dtype--2-indices0-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float64Dtype--2-indices0-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float64Dtype-0-indices1-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float64Dtype-0-indices1-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float64Dtype-2-indices2-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Float64Dtype-2-indices2-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int16Dtype--2-indices0-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int16Dtype--2-indices0-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int16Dtype-0-indices1-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int16Dtype-0-indices1-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int16Dtype-2-indices2-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int16Dtype-2-indices2-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int32Dtype--2-indices0-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int32Dtype--2-indices0-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int32Dtype-0-indices1-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int32Dtype-0-indices1-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int32Dtype-2-indices2-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int32Dtype-2-indices2-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int64Dtype--2-indices0-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int64Dtype--2-indices0-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int64Dtype-0-indices1-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int64Dtype-0-indices1-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int64Dtype-2-indices2-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int64Dtype-2-indices2-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int8Dtype--2-indices0-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int8Dtype--2-indices0-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int8Dtype-0-indices1-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int8Dtype-0-indices1-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int8Dtype-2-indices2-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[Int8Dtype-2-indices2-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt16Dtype--2-indices0-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt16Dtype--2-indices0-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt16Dtype-0-indices1-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt16Dtype-0-indices1-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt16Dtype-2-indices2-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt16Dtype-2-indices2-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt32Dtype--2-indices0-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt32Dtype--2-indices0-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt32Dtype-0-indices1-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt32Dtype-0-indices1-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt32Dtype-2-indices2-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt32Dtype-2-indices2-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt64Dtype--2-indices0-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt64Dtype--2-indices0-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt64Dtype-0-indices1-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt64Dtype-0-indices1-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt64Dtype-2-indices2-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt64Dtype-2-indices2-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt8Dtype--2-indices0-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt8Dtype--2-indices0-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt8Dtype-0-indices1-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt8Dtype-0-indices1-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt8Dtype-2-indices2-False]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_container_shift[UInt8Dtype-2-indices2-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_grouping_grouper[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_grouping_grouper[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_grouping_grouper[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_grouping_grouper[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_grouping_grouper[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_grouping_grouper[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_grouping_grouper[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_grouping_grouper[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_grouping_grouper[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_grouping_grouper[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_grouping_grouper[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_invert[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_is_extension_array_dtype[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_memory_usage[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_memory_usage[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_memory_usage[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_memory_usage[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_memory_usage[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_memory_usage[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_memory_usage[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_memory_usage[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_memory_usage[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_memory_usage[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_memory_usage[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reduce_series_numeric[Float32Dtype-skew-True]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reindex_non_na_fill_value[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reindex_non_na_fill_value[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reindex_non_na_fill_value[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reindex_non_na_fill_value[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reindex_non_na_fill_value[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reindex_non_na_fill_value[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reindex_non_na_fill_value[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reindex_non_na_fill_value[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reindex_non_na_fill_value[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reindex_non_na_fill_value[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_reindex_non_na_fill_value[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_series_constructor[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_set_frame_expand_extension_with_regular[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_set_frame_expand_extension_with_regular[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_set_frame_expand_extension_with_regular[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_set_frame_expand_extension_with_regular[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_set_frame_expand_extension_with_regular[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_set_frame_expand_extension_with_regular[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_set_frame_expand_extension_with_regular[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_set_frame_expand_extension_with_regular[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_set_frame_expand_extension_with_regular[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_set_frame_expand_extension_with_regular[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_set_frame_expand_extension_with_regular[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_expand_columns[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_expand_columns[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_expand_columns[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_expand_columns[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_expand_columns[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_expand_columns[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_expand_columns[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_expand_columns[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_expand_columns[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_expand_columns[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_expand_columns[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_setitem_frame_2d_values[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_to_numpy[UInt8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[BooleanDtype-frame-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[BooleanDtype-frame-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[BooleanDtype-frame-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[BooleanDtype-series-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[BooleanDtype-series-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[BooleanDtype-series-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float32Dtype-frame-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float32Dtype-frame-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float32Dtype-frame-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float32Dtype-series-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float32Dtype-series-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float32Dtype-series-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float64Dtype-frame-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float64Dtype-frame-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float64Dtype-frame-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float64Dtype-series-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float64Dtype-series-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Float64Dtype-series-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int16Dtype-frame-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int16Dtype-frame-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int16Dtype-frame-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int16Dtype-series-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int16Dtype-series-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int16Dtype-series-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int32Dtype-frame-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int32Dtype-frame-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int32Dtype-frame-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int32Dtype-series-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int32Dtype-series-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int32Dtype-series-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int64Dtype-frame-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int64Dtype-frame-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int64Dtype-frame-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int64Dtype-series-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int64Dtype-series-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int64Dtype-series-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int8Dtype-frame-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int8Dtype-frame-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int8Dtype-frame-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int8Dtype-series-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int8Dtype-series-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[Int8Dtype-series-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt16Dtype-frame-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt16Dtype-frame-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt16Dtype-frame-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt16Dtype-series-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt16Dtype-series-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt16Dtype-series-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt32Dtype-frame-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt32Dtype-frame-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt32Dtype-frame-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt32Dtype-series-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt32Dtype-series-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt32Dtype-series-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt64Dtype-frame-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt64Dtype-frame-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt64Dtype-frame-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt64Dtype-series-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt64Dtype-series-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt64Dtype-series-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt8Dtype-frame-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt8Dtype-frame-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt8Dtype-frame-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt8Dtype-series-index1]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt8Dtype-series-index2]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_unstack[UInt8Dtype-series-index3]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_value_counts_with_normalize[BooleanDtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_value_counts_with_normalize[Float32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_value_counts_with_normalize[Float64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_value_counts_with_normalize[Int16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_value_counts_with_normalize[Int32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_value_counts_with_normalize[Int64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_value_counts_with_normalize[Int8Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_value_counts_with_normalize[UInt16Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_value_counts_with_normalize[UInt32Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_value_counts_with_normalize[UInt64Dtype]",
    "tests/extension/test_masked.py::TestMaskedArrays::test_value_counts_with_normalize[UInt8Dtype]",
    "tests/extension/test_numpy.py::Test2DCompat::test_copy_order[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cummax-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cummax-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cummin-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cummin-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cumprod-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cumprod-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cumsum-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[float-cumsum-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cummax-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cummax-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cummin-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cummin-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cumsum-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_accumulate_series[object-cumsum-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-argmax-False--1]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-argmax-True-0]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-argmin-False--1]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-argmin-True-2]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-idxmax-False-nan]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-idxmax-True-0]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-idxmin-False-nan]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argreduce_series[object-idxmin-True-2]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argsort[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argsort[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_argsort_missing[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_array_interface[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_array_interface_copy[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_array_interface_copy[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_array_type[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_array_type[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_object_frame[object-data]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_object_frame[object-data_missing]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_own_type[float-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_own_type[object-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_string[object-string[pyarrow]]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_astype_string[object-string[python]]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_compare_array[object-eq]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_compare_array[object-ne]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_concat[float-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_concat[float-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_concat[object-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_concat[object-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_contains[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_constructor_from_dict[float-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_constructor_from_dict[float-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_constructor_from_dict[object-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_constructor_from_dict[object-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_from_series[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_dataframe_from_series[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize_empty[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize_empty[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize_equivalence[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_factorize_equivalence[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_getitem_boolean_array_mask[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_getitem_boolean_array_mask[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_getitem_mask[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_getitem_mask[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_groupby_extension_transform[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_grouping_grouper[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_grouping_grouper[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_hash_pandas_object[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_hash_pandas_object_works[object-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_hash_pandas_object_works[object-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_index_from_array[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_index_from_array[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_is_extension_array_dtype[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_memory_usage[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_pandas_array_dtype[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[float-all-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[float-all-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[float-any-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[float-any-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[object-all-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[object-all-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[object-any-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_boolean[object-any-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-count-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-count-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-kurt-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-kurt-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-max-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-max-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-mean-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-mean-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-median-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-median-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-min-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-min-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-prod-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-prod-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-sem-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-sem-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-skew-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-skew-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-std-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-std-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-sum-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-sum-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-var-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[float-var-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-max-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-max-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-min-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-min-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-sum-False]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reduce_series_numeric[object-sum-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reindex[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_reindex_non_na_fill_value[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_series_constructor[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_series_constructor[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_series_repr[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_set_frame_overwrite_object[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_set_frame_overwrite_object[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_2d_values[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_frame_2d_values[float]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_setitem_frame_2d_values[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_stack[object-False-columns1]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_stack[object-True-columns1]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_take_series[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_to_numpy[object]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[float-absolute]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[float-negative]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[float-positive]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[object-absolute]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[object-negative]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unary_ufunc_dunder_equivalence[object-positive]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[float-<lambda>-Series]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[float-unique-<lambda>]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[float-unique-Series]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[object-<lambda>-<lambda>]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[object-<lambda>-Series]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[object-unique-<lambda>]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unique[object-unique-Series]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unstack[float-frame-index1]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unstack[float-frame-index2]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unstack[float-frame-index3]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unstack[float-series-index1]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unstack[float-series-index2]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_unstack[float-series-index3]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_value_counts[float-data_missing-True]",
    "tests/extension/test_numpy.py::TestNumpyExtensionArray::test_value_counts_with_normalize[object]",
    "tests/extension/test_period.py::Test2DCompat::test_copy_order[2D]",
    "tests/extension/test_period.py::Test2DCompat::test_copy_order[D]",
    "tests/extension/test_period.py::TestPeriodArray::test_EA_types[2D-c]",
    "tests/extension/test_period.py::TestPeriodArray::test_EA_types[2D-python]",
    "tests/extension/test_period.py::TestPeriodArray::test_EA_types[D-c]",
    "tests/extension/test_period.py::TestPeriodArray::test_EA_types[D-python]",
    "tests/extension/test_period.py::TestPeriodArray::test_array_interface_copy[2D]",
    "tests/extension/test_period.py::TestPeriodArray::test_array_interface_copy[D]",
    "tests/extension/test_period.py::TestPeriodArray::test_astype_own_type[2D-False]",
    "tests/extension/test_period.py::TestPeriodArray::test_astype_own_type[D-False]",
    "tests/extension/test_period.py::TestPeriodArray::test_grouping_grouper[2D]",
    "tests/extension/test_period.py::TestPeriodArray::test_grouping_grouper[D]",
    "tests/extension/test_period.py::TestPeriodArray::test_is_extension_array_dtype[2D]",
    "tests/extension/test_period.py::TestPeriodArray::test_is_extension_array_dtype[D]",
    "tests/extension/test_period.py::TestPeriodArray::test_series_constructor[2D]",
    "tests/extension/test_period.py::TestPeriodArray::test_series_constructor[D]",
    "tests/extension/test_sparse.py::TestSparseArray::test_EA_types[0-c]",
    "tests/extension/test_sparse.py::TestSparseArray::test_EA_types[0-python]",
    "tests/extension/test_sparse.py::TestSparseArray::test_EA_types[nan-c]",
    "tests/extension/test_sparse.py::TestSparseArray::test_EA_types[nan-python]",
    "tests/extension/test_sparse.py::TestSparseArray::test_array_interface_copy[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_array_interface_copy[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_astype_own_type[0-False]",
    "tests/extension/test_sparse.py::TestSparseArray::test_astype_own_type[nan-False]",
    "tests/extension/test_sparse.py::TestSparseArray::test_fillna_copy_frame[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_fillna_copy_frame[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_fillna_copy_series[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_fillna_copy_series[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_grouping_grouper[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_grouping_grouper[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_is_extension_array_dtype[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_is_extension_array_dtype[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_series_constructor[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_series_constructor[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_shift_0_periods[0]",
    "tests/extension/test_sparse.py::TestSparseArray::test_shift_0_periods[nan]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[0-absolute]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[0-negative]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[0-positive]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[nan-absolute]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[nan-negative]",
    "tests/extension/test_sparse.py::TestSparseArray::test_unary_ufunc_dunder_equivalence[nan-positive]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow_numpy-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow_numpy-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow_numpy-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[pyarrow_numpy-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[python-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[python-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[python-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[python-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[pyarrow]-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[pyarrow]-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[pyarrow]-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[pyarrow]-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[python]-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[python]-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[python]-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=str[python]-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[pyarrow]-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[pyarrow]-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[pyarrow]-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[pyarrow]-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[python]-False-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[python]-False-python]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[python]-True-c]",
    "tests/extension/test_string.py::TestStringArray::test_EA_types[string=string[python]-True-python]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_argsort_missing[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[pyarrow-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[pyarrow-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[pyarrow_numpy-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[pyarrow_numpy-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[python-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[python-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=str[pyarrow]-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=str[pyarrow]-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=str[python]-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=str[python]-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=string[pyarrow]-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=string[pyarrow]-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=string[python]-False-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_arith_series_with_array[string=string[python]-True-__radd__]",
    "tests/extension/test_string.py::TestStringArray::test_array_interface_copy[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_array_interface_copy[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_array_interface_copy[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_array_interface_copy[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_array_interface_copy[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_array_interface_copy[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_array_interface_copy[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_array_interface_copy[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[pyarrow-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[pyarrow-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[pyarrow_numpy-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[pyarrow_numpy-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[python-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[python-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=str[pyarrow]-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=str[pyarrow]-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=str[python]-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=str[python]-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=string[pyarrow]-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=string[pyarrow]-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=string[python]-False-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_own_type[string=string[python]-True-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_astype_str[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[pyarrow_numpy-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[pyarrow]-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=str[python]-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_array[string=string[pyarrow]-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[pyarrow_numpy-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[pyarrow]-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=str[python]-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-False-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-False-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-False-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-False-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-False-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-False-ne]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-True-eq]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-True-ge]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-True-gt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-True-le]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-True-lt]",
    "tests/extension/test_string.py::TestStringArray::test_compare_scalar[string=string[pyarrow]-True-ne]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-False-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow-True-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-False-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[pyarrow_numpy-True-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-False-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[python-True-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-False--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-False--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-False-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-False-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-False-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-False-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-True--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-True--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-True-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-True-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-True-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[pyarrow]-True-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-False--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-False--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-False-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-False-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-False-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-False-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-True--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-True--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-True-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-True-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-True-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=str[python]-True-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-False--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-False--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-False-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-False-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-False-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-False-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-True--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-True--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-True-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-True-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-True-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[pyarrow]-True-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-False--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-False--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-False-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-False-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-False-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-False-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-True--2-indices0-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-True--2-indices0-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-True-0-indices1-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-True-0-indices1-True]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-True-2-indices2-False]",
    "tests/extension/test_string.py::TestStringArray::test_container_shift[string=string[python]-True-2-indices2-True]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_grouping_grouper[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_is_extension_array_dtype[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_memory_usage[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow-False-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow-False-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow-True-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow-True-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow_numpy-False-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow_numpy-False-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow_numpy-True-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[pyarrow_numpy-True-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[python-False-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[python-False-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[python-True-sum-False]",
    "tests/extension/test_string.py::TestStringArray::test_reduce_series_numeric[python-True-sum-True]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_reindex_non_na_fill_value[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[pyarrow-False-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[pyarrow-True-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[pyarrow_numpy-False-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[pyarrow_numpy-True-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[python-False-True]",
    "tests/extension/test_string.py::TestStringArray::test_searchsorted[python-True-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_series_constructor[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_set_frame_expand_extension_with_regular[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_expand_columns[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_setitem_frame_2d_values[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[pyarrow_numpy-False]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[pyarrow_numpy-True]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[string=str[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[string=str[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[string=str[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[string=str[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_to_numpy[string=string[python]-True]",
    "tests/extension/test_string.py::TestStringArray::test_unary_ufunc_dunder_equivalence[pyarrow-False-positive]",
    "tests/extension/test_string.py::TestStringArray::test_unary_ufunc_dunder_equivalence[pyarrow-True-positive]",
    "tests/extension/test_string.py::TestStringArray::test_unary_ufunc_dunder_equivalence[pyarrow_numpy-False-positive]",
    "tests/extension/test_string.py::TestStringArray::test_unary_ufunc_dunder_equivalence[pyarrow_numpy-True-positive]",
    "tests/extension/test_string.py::TestStringArray::test_unique[string=str[python]-False-<lambda>-Series]",
    "tests/extension/test_string.py::TestStringArray::test_unique[string=str[python]-False-unique-Series]",
    "tests/extension/test_string.py::TestStringArray::test_unique[string=str[python]-True-<lambda>-Series]",
    "tests/extension/test_string.py::TestStringArray::test_unique[string=str[python]-True-unique-Series]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-False-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-False-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-False-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-False-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-False-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-False-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-True-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-True-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-True-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-True-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-True-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow-True-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-False-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-False-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-False-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-False-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-False-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-False-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-True-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-True-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-True-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-True-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-True-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[pyarrow_numpy-True-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-False-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-False-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-False-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-False-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-False-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-False-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-True-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-True-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-True-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-True-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-True-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[python-True-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-False-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-False-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-False-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-False-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-False-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-False-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-True-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-True-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-True-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-True-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-True-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[pyarrow]-True-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-False-frame-index0]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-False-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-False-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-False-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-False-series-index0]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-False-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-False-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-False-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-True-frame-index0]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-True-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-True-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-True-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-True-series-index0]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-True-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-True-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=str[python]-True-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-False-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-False-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-False-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-False-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-False-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-False-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-True-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-True-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-True-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-True-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-True-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[pyarrow]-True-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-False-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-False-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-False-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-False-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-False-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-False-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-True-frame-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-True-frame-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-True-frame-index3]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-True-series-index1]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-True-series-index2]",
    "tests/extension/test_string.py::TestStringArray::test_unstack[string=string[python]-True-series-index3]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[pyarrow-False]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[pyarrow-True]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[python-False]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[python-True]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[string=string[pyarrow]-False]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[string=string[pyarrow]-True]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[string=string[python]-False]",
    "tests/extension/test_string.py::TestStringArray::test_value_counts_with_normalize[string=string[python]-True]",
    "tests/frame/constructors/test_from_records.py::TestFromRecords::test_from_records_misc_brokenness3",
    "tests/frame/constructors/test_from_records.py::TestFromRecords::test_from_records_series_categorical_index",
    "tests/frame/constructors/test_from_records.py::TestFromRecords::test_from_records_series_list_dict",
    "tests/frame/constructors/test_from_records.py::TestFromRecords::test_from_records_with_datetimes",
    "tests/frame/indexing/test_coercion.py::test_26395[at]",
    "tests/frame/indexing/test_coercion.py::test_26395[loc]",
    "tests/frame/indexing/test_getitem.py::TestGetitemBooleanMask::test_getitem_returns_view_when_column_is_unique_in_df",
    "tests/frame/indexing/test_getitem.py::TestGetitemCallable::test_loc_multiindex_columns_one_level",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_fancy_getitem_slice_mixed",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_getitem_setitem_boolean_misaligned",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_getitem_setitem_float_labels",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_getitem_setitem_ix_bool_keyerror[False]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_getitem_setitem_ix_bool_keyerror[True]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_getitem_setitem_non_ix_labels",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_iloc_col_slice_view",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_iloc_row_slice_view",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_iloc_setitem_nullable_2d_values",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_loc_bool_multiindex[bool-indexer1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_loc_bool_multiindex[boolean-indexer1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_loc_internals_not_updated_correctly",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_loc_setitem_datetimelike_with_inference",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_loc_setitem_reordering_with_all_true_indexer[col0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_set_2d_casting_date_to_int[col0-indexer0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_set_2d_casting_date_to_int[col1-indexer0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setitem",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setitem_boolean",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setitem_mixed_datetime",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setitem_single_column_mixed_datetime",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float32-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float32-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float32-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float64-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float64-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Float64-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int16-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int16-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int16-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int32-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int32-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int32-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int64-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int64-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int64-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int8-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int8-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[Int8-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt16-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt16-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt16-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt32-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt32-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt32-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt64-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt64-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt64-null2]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt8-null0]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt8-null1]",
    "tests/frame/indexing/test_indexing.py::TestDataFrameIndexing::test_setting_mismatched_na_into_nullable_fails[UInt8-null2]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key0]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key1]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key2]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key3]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key4]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated[key5]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated_multiindex[key2]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated_multiindex[key3]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated_multiindex[key4]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_getitem_dict_and_set_deprecated_multiindex[key5]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key0]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key1]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key2]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key3]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key4]",
    "tests/frame/indexing/test_indexing.py::TestDeprecatedIndexers::test_setitem_dict_and_set_disallowed[key5]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_at_iat_setitem_single_value_in_categories[at]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_at_iat_setitem_single_value_in_categories[iat]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_at_iat_setitem_single_value_in_categories[iloc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_at_iat_setitem_single_value_in_categories[loc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_setitem_full_row_non_categorical_rhs[iloc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_setitem_full_row_non_categorical_rhs[loc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_setitem_list_of_lists[iloc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_setitem_list_of_lists[loc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_setitem_mask_single_value_in_categories[iloc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_setitem_mask_single_value_in_categories[loc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_setitem_non_categorical_rhs[iloc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_setitem_non_categorical_rhs[loc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_setitem_partial_col_categorical_rhs[iloc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_iloc_setitem_partial_col_categorical_rhs[loc]",
    "tests/frame/indexing/test_indexing.py::TestLocILocDataFrameCategorical::test_loc_on_multiindex_one_level",
    "tests/frame/indexing/test_indexing.py::test_add_new_column_infer_string",
    "tests/frame/indexing/test_indexing.py::test_adding_new_conditional_column",
    "tests/frame/indexing/test_indexing.py::test_adding_new_conditional_column_with_string[dtype1-True]",
    "tests/frame/indexing/test_indexing.py::test_adding_new_conditional_column_with_string[object-False]",
    "tests/frame/indexing/test_indexing.py::test_adding_new_conditional_column_with_string[string[pyarrow_numpy]-True]",
    "tests/frame/indexing/test_indexing.py::test_object_casting_indexing_wraps_datetimelike",
    "tests/frame/indexing/test_insert.py::TestDataFrameInsert::test_insert",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItem::test_frame_setitem_datetime64_col_other_units[h]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItem::test_frame_setitem_datetime64_col_other_units[m]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItem::test_frame_setitem_existing_datetime64_col_other_units[h]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItem::test_frame_setitem_existing_datetime64_col_other_units[m]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItem::test_setitem_list_missing_columns[columns3-box3-expected3]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItem::test_setitem_multi_index",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[1-Series-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[1-Series-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[1-array0-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[1-array0-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[1-array1-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[1-array1-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[1-list-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[1-list-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[2-Series-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[2-Series-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[2-array0-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[2-array0-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[2-array1-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[2-array1-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[2-list-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[2-list-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[3-Series-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[3-Series-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[3-array0-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[3-array0-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[3-array1-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[3-array1-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[3-list-iloc]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemSlicing::test_setitem_slice_indexer_broadcasting_rhs[3-list-setitem]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetItemWithExpansion::test_setitem_newcol_tuple_key",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetitemCopyViewSemantics::test_setitem_column_update_inplace",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetitemCopyViewSemantics::test_setitem_listlike_key_scalar_value_not_inplace[value2]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetitemCopyViewSemantics::test_setitem_partial_column_inplace[False]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetitemCopyViewSemantics::test_setitem_partial_column_inplace[True]",
    "tests/frame/indexing/test_setitem.py::TestDataFrameSetitemCopyViewSemantics::test_setitem_same_dtype_not_inplace[value2]",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_bug_transposition",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_dataframe_col_match",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_datetime",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_datetimelike_noop[timedelta64[ns]]",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_ndframe_align",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_set[default]",
    "tests/frame/indexing/test_where.py::TestDataFrameIndexingWhere::test_where_set[mixed_int]",
    "tests/frame/indexing/test_where.py::test_where_int_downcasting_deprecated",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Float32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Float64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Int16]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Int32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Int64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-Int8]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-UInt16]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-UInt32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-UInt64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[DataFrame-UInt8]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Float32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Float64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Int16]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Int32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Int64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-Int8]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-UInt16]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-UInt32]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-UInt64]",
    "tests/frame/indexing/test_where.py::test_where_nullable_invalid_na[Series-UInt8]",
    "tests/frame/indexing/test_xs.py::TestXS::test_xs",
    "tests/frame/indexing/test_xs.py::TestXS::test_xs_view",
    "tests/frame/indexing/test_xs.py::TestXSWithMultiIndex::test_xs_droplevel_false_view",
    "tests/frame/indexing/test_xs.py::TestXSWithMultiIndex::test_xs_integer_key",
    "tests/frame/indexing/test_xs.py::TestXSWithMultiIndex::test_xs_setting_with_copy_error",
    "tests/frame/indexing/test_xs.py::TestXSWithMultiIndex::test_xs_setting_with_copy_error_multiple",
    "tests/frame/methods/test_align.py::TestDataFrameAlign::test_align_float",
    "tests/frame/methods/test_asfreq.py::TestAsFreq::test_asfreq_resample_set_correct_freq[DataFrame]",
    "tests/frame/methods/test_asfreq.py::TestAsFreq::test_asfreq_resample_set_correct_freq[Series]",
    "tests/frame/methods/test_asof.py::TestFrameAsof::test_subset",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_arg_for_errors_dictlist",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_cast_nan_inf_int[inf-int32]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_cast_nan_inf_int[inf-int64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_cast_nan_inf_int[nan-int32]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_cast_nan_inf_int[nan-int64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_dt64_to_string[DataFrame-None]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_dt64_to_string[Series-None]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_dt64tz",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_dt64tz_to_str",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_extension_dtypes_duplicate_col[Int64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_from_datetimelike_to_object[h-M8]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_from_datetimelike_to_object[h-m8]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_from_datetimelike_to_object[m-M8]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_from_datetimelike_to_object[m-m8]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_str_float",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_td64_to_string[DataFrame]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_td64_to_string[Series]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_datetime_unit[h]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_datetime_unit[m]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_datetimelike_unit[h-M8-float64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_datetimelike_unit[h-M8-int64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_datetimelike_unit[h-m8-float64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_datetimelike_unit[h-m8-int64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_datetimelike_unit[m-M8-float64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_datetimelike_unit[m-M8-int64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_datetimelike_unit[m-m8-float64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_datetimelike_unit[m-m8-int64]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_incorrect_datetimelike[h]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_incorrect_datetimelike[m]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_timedelta_unit[h]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_to_timedelta_unit[m]",
    "tests/frame/methods/test_astype.py::TestAstype::test_astype_with_exclude_string",
    "tests/frame/methods/test_astype.py::TestAstypeCategorical::test_astype_categorical_to_string_missing",
    "tests/frame/methods/test_astype.py::test_astype_to_string_not_modifying_input[pyarrow_numpy-None]",
    "tests/frame/methods/test_astype.py::test_astype_to_string_not_modifying_input[pyarrow_numpy-val4]",
    "tests/frame/methods/test_clip.py::TestDataFrameClip::test_clip_int_data_with_float_bound",
    "tests/frame/methods/test_clip.py::TestDataFrameClip::test_dataframe_clip",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_convert_datatime_correctly[data11-data21-data_expected1]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_convert_datatime_correctly[data12-data22-data_expected2]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_convert_datatime_correctly[data13-data23-data_expected3]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_timezone[ms]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_timezone[ns]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_timezone[s]",
    "tests/frame/methods/test_combine_first.py::TestDataFrameCombineFirst::test_combine_first_timezone[us]",
    "tests/frame/methods/test_combine_first.py::test_combine_first_timestamp_bug[Decimal-scalar10-scalar20]",
    "tests/frame/methods/test_combine_first.py::test_combine_first_timestamp_bug[NAType-scalar10-scalar20]",
    "tests/frame/methods/test_combine_first.py::test_combine_first_timestamp_bug[NaTType-scalar10-scalar20]",
    "tests/frame/methods/test_combine_first.py::test_combine_first_timestamp_bug[NaTType-scalar12-scalar22]",
    "tests/frame/methods/test_combine_first.py::test_combine_first_timestamp_bug[NoneType-scalar10-scalar20]",
    "tests/frame/methods/test_combine_first.py::test_combine_first_timestamp_bug_NaT",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes[pyarrow-False-expected0]",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes[pyarrow-True-Int32]",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes[pyarrow_numpy-False-expected0]",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes[pyarrow_numpy-True-Int32]",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes[python-False-expected0]",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes[python-True-Int32]",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes_avoid_block_splitting",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_convert_dtypes_pyarrow_to_np_nullable",
    "tests/frame/methods/test_convert_dtypes.py::TestConvertDtypes::test_pyarrow_engine_lines_false",
    "tests/frame/methods/test_copy.py::TestCopy::test_copy_consolidates",
    "tests/frame/methods/test_count.py::TestDataFrameCount::test_count",
    "tests/frame/methods/test_cov_corr.py::TestDataFrameCorr::test_corr_item_cache",
    "tests/frame/methods/test_cov_corr.py::TestDataFrameCorr::test_corr_nooverlap[pearson]",
    "tests/frame/methods/test_cov_corr.py::TestDataFrameCorr::test_corr_numeric_only[False-spearman]",
    "tests/frame/methods/test_cov_corr.py::TestDataFrameCorr::test_corr_scipy_method[pearson]",
    "tests/frame/methods/test_cov_corr.py::TestDataFrameCov::test_cov",
    "tests/frame/methods/test_describe.py::TestDataFrameDescribe::test_describe_datetime_columns",
    "tests/frame/methods/test_diff.py::TestDataFrameDiff::test_diff_all_int_dtype[int16]",
    "tests/frame/methods/test_diff.py::TestDataFrameDiff::test_diff_all_int_dtype[int8]",
    "tests/frame/methods/test_diff.py::TestDataFrameDiff::test_diff_integer_na[0-expected0]",
    "tests/frame/methods/test_diff.py::TestDataFrameDiff::test_diff_integer_na[1-expected1]",
    "tests/frame/methods/test_dot.py::TestDataFrameDot::test_dot_1d_ndarray",
    "tests/frame/methods/test_dot.py::TestDataFrameDot::test_dot_2d_ndarray",
    "tests/frame/methods/test_dot.py::TestDataFrameDot::test_dot_aligns",
    "tests/frame/methods/test_dot.py::TestDataFrameDot::test_dot_equiv_values_dot",
    "tests/frame/methods/test_dot.py::TestDataFrameDot::test_dot_series",
    "tests/frame/methods/test_dot.py::TestDataFrameDot::test_dot_series_alignment",
    "tests/frame/methods/test_dot.py::TestSeriesDot::test_dot_1d_ndarray",
    "tests/frame/methods/test_dot.py::TestSeriesDot::test_dot_2d_ndarray",
    "tests/frame/methods/test_dot.py::TestSeriesDot::test_dot_aligns",
    "tests/frame/methods/test_dot.py::TestSeriesDot::test_dot_equiv_values_dot",
    "tests/frame/methods/test_dot.py::TestSeriesDot::test_dot_series",
    "tests/frame/methods/test_dot.py::TestSeriesDot::test_dot_series_alignment",
    "tests/frame/methods/test_drop.py::TestDataFrameDrop::test_drop",
    "tests/frame/methods/test_drop.py::TestDataFrameDrop::test_drop_multiindex_not_lexsorted",
    "tests/frame/methods/test_drop.py::TestDataFrameDrop::test_drop_multiindex_other_level_nan",
    "tests/frame/methods/test_drop.py::test_drop_with_non_unique_datetime_index_and_invalid_keys",
    "tests/frame/methods/test_dropna.py::TestDataFrameMissingData::test_drop_and_dropna_caching",
    "tests/frame/methods/test_dropna.py::TestDataFrameMissingData::test_dropna",
    "tests/frame/methods/test_dropna.py::TestDataFrameMissingData::test_dropna_multiple_axes",
    "tests/frame/methods/test_dropna.py::TestDataFrameMissingData::test_no_nans_in_frame[axis=0]",
    "tests/frame/methods/test_dtypes.py::TestDataFrameDataTypes::test_dtypes_timedeltas",
    "tests/frame/methods/test_equals.py::TestEquals::test_equals_different_blocks",
    "tests/frame/methods/test_explode.py::test_duplicate_index[input_dict0-input_index0-expected_dict0-expected_index0]",
    "tests/frame/methods/test_explode.py::test_duplicate_index[input_dict1-input_index1-expected_dict1-expected_index1]",
    "tests/frame/methods/test_explode.py::test_duplicate_index[input_dict2-input_index2-expected_dict2-expected_index2]",
    "tests/frame/methods/test_explode.py::test_duplicate_index[input_dict3-input_index3-expected_dict3-expected_index3]",
    "tests/frame/methods/test_fillna.py::TestFillNA::test_fillna_dict_inplace_nonunique_columns",
    "tests/frame/methods/test_fillna.py::TestFillNA::test_fillna_dtype_conversion",
    "tests/frame/methods/test_fillna.py::TestFillNA::test_fillna_on_column_view",
    "tests/frame/methods/test_fillna.py::TestFillNA::test_fillna_with_multi_index_frame",
    "tests/frame/methods/test_fillna.py::TestFillNA::test_inplace_dict_update_view[-1]",
    "tests/frame/methods/test_fillna.py::TestFillNA::test_inplace_dict_update_view[val1]",
    "tests/frame/methods/test_info.py::test_info_memory",
    "tests/frame/methods/test_info.py::test_info_memory_usage_deep_not_pypy",
    "tests/frame/methods/test_info.py::test_memory_usage_empty_no_warning",
    "tests/frame/methods/test_info.py::test_usage_via_getsizeof",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_inerpolate_invalid_downcast",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interp_basic",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interp_combo",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interp_inplace",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interp_inplace_row",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interp_nan_idx",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interp_rowwise",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_arrow[int16]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_arrow[int32]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_arrow[int64]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_arrow[int8]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_arrow[uint16]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_arrow[uint32]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_arrow[uint64]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_arrow[uint8]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_datetimelike_values[DataFrame]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_datetimelike_values[Series]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea[Int16]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea[Int32]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea[Int64]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea[Int8]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea[UInt16]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea[UInt32]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea[UInt64]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea[UInt8]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea_float[Float32]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea_float[Float64]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea_float[float32[pyarrow]]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_ea_float[float64[pyarrow]]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_inplace[DataFrame]",
    "tests/frame/methods/test_interpolate.py::TestDataFrameInterpolate::test_interpolate_inplace[Series]",
    "tests/frame/methods/test_join.py::TestDataFrameJoin::test_join",
    "tests/frame/methods/test_join.py::test_join_index",
    "tests/frame/methods/test_join.py::test_join_index_more",
    "tests/frame/methods/test_map.py::test_map",
    "tests/frame/methods/test_map.py::test_map_empty[<lambda>-expected0]",
    "tests/frame/methods/test_map.py::test_map_empty[<lambda>-expected2]",
    "tests/frame/methods/test_map.py::test_map_empty[round-expected0]",
    "tests/frame/methods/test_map.py::test_map_empty[round-expected2]",
    "tests/frame/methods/test_map.py::test_map_function_runs_once",
    "tests/frame/methods/test_map.py::test_map_na_ignore",
    "tests/frame/methods/test_matmul.py::TestMatMul::test_matmul",
    "tests/frame/methods/test_pop.py::TestDataFramePop::test_pop",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_empty[linear-single]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_empty[nearest-table]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_invalid_method",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_datetime[ms]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_datetime[ns]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_datetime[s]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_datetime[us]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_empty_no_columns[linear-single]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_empty_no_columns[nearest-table]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_empty_no_rows_dt64[linear-single]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_item_cache[linear-single]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_item_cache[nearest-table]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_multi_empty[linear-single]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_multi_empty[nearest-table]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_nat[linear-single-ms]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_nat[linear-single-ns]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_nat[linear-single-s]",
    "tests/frame/methods/test_quantile.py::TestDataFrameQuantile::test_quantile_nat[linear-single-us]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_empty_datelike[datetime64[ns]-expected_data1-expected_index1-0-datetime64[ns]]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea[Float64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea[Float64-Series]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea[Int64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea[Int64-Series]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_all_na[Float64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_all_na[Float64-Series]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_all_na[Int64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_all_na[Int64-Series]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_scalar[Float64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_scalar[Int64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_scalar[timedelta64[ns]-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_with_na[Float64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_with_na[Float64-Series]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_with_na[Int64-DataFrame]",
    "tests/frame/methods/test_quantile.py::TestQuantileExtensionDtype::test_quantile_ea_with_na[Int64-Series]",
    "tests/frame/methods/test_rank.py::TestRank::test_rank",
    "tests/frame/methods/test_rank.py::TestRank::test_rank_na_option",
    "tests/frame/methods/test_rank.py::TestRank::test_rank_string_dtype[string=string[pyarrow]]",
    "tests/frame/methods/test_rank.py::TestRank::test_rank_string_dtype[string[pyarrow]-Int64]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_axis_style_raises",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_date_fill_value",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx1-src_idx0]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx1-src_idx1]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx3-src_idx0]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx3-src_idx1]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx4-src_idx0]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx4-src_idx1]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx5-src_idx0]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_empty[cat_idx5-src_idx1]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_fill_value",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_int",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_nearest_tz[zoneinfo.ZoneInfo(key='US/Pacific')]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_nearest_tz[zoneinfo.ZoneInfo(key='UTC')]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_positional_raises",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Float32]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Float64]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Int16]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Int32]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Int64]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[Int8]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[UInt16]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[UInt32]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[UInt64]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_single_column_ea_index_and_columns[UInt8]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_uint_dtypes_fill_value[uint16]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_uint_dtypes_fill_value[uint32]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_uint_dtypes_fill_value[uint64]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_uint_dtypes_fill_value[uint8]",
    "tests/frame/methods/test_reindex.py::TestDataFrameSelectReindex::test_reindex_with_multi_index",
    "tests/frame/methods/test_reindex.py::TestReindexSetIndex::test_setitem_reset_index_dtypes",
    "tests/frame/methods/test_rename.py::TestRename::test_rename",
    "tests/frame/methods/test_rename.py::TestRename::test_rename_axis_style_raises",
    "tests/frame/methods/test_rename.py::TestRename::test_rename_inplace",
    "tests/frame/methods/test_rename.py::TestRename::test_rename_mapper_and_positional_arguments_raises",
    "tests/frame/methods/test_rename.py::TestRename::test_rename_multiindex",
    "tests/frame/methods/test_rename.py::TestRename::test_rename_no_mappings_raises",
    "tests/frame/methods/test_rename.py::TestRename::test_rename_nocopy",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_NA_with_None",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_after_convert_dtypes",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_dict_tuple_list_ordering_remains_the_same",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[DataFrame-array-data0-to_replace0-value0-expected0]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[DataFrame-tuple-data0-to_replace0-value0-expected0]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[DataFrame-tuple-data1-to_replace1-value1-expected1]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[DataFrame-tuple-data2-to_replace2-value2-expected2]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[DataFrame-tuple-data3-to_replace3-value3-expected3]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[Series-array-data0-to_replace0-value0-expected0]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[Series-tuple-data0-to_replace0-value0-expected0]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[Series-tuple-data1-to_replace1-value1-expected1]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[Series-tuple-data2-to_replace2-value2-expected2]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_list_with_mixed_type[Series-tuple-data3-to_replace3-value3-expected3]",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_mixed_int_block_splitting",
    "tests/frame/methods/test_replace.py::TestDataFrameReplace::test_replace_value_is_none",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_duplicate_columns_allow[False-False]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_duplicate_columns_allow[False-True]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_duplicate_columns_allow[True-False]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_duplicate_columns_default[False]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_duplicate_columns_default[True]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_empty_rangeindex",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_level_missing[idx_lev0]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_level_missing[idx_lev1]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_multiindex_columns",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_with_datetimeindex_cols[2012-12-31]",
    "tests/frame/methods/test_reset_index.py::TestResetIndex::test_reset_index_with_datetimeindex_cols[name5]",
    "tests/frame/methods/test_sample.py::TestSample::test_sample_random_state[DataFrame-np.array-arg0]",
    "tests/frame/methods/test_sample.py::TestSample::test_sample_random_state[Series-np.array-arg0]",
    "tests/frame/methods/test_sample.py::TestSample::test_sample_requires_positive_n_frac[DataFrame]",
    "tests/frame/methods/test_sample.py::TestSample::test_sample_requires_positive_n_frac[Series]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_exclude_include_int[include0]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_exclude_include_int[include2]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_float_dtype[expected0-float]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_float_dtype[expected1-float]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-S1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-U1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-bytes_]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-str0]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-str1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[exclude-str_]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-S1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-U1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-bytes_]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-str0]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-str1]",
    "tests/frame/methods/test_select_dtypes.py::TestSelectDtypes::test_select_dtypes_str_raises[include-str_]",
    "tests/frame/methods/test_set_axis.py::TestDataFrameSetAxis::test_set_axis_copy",
    "tests/frame/methods/test_set_axis.py::TestSeriesSetAxis::test_set_axis_copy",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-False-None-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-False-None-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-B-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-B-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-None-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-None-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-test-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[False-True-test-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-False-None-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-False-None-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-B-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-B-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-None-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-None-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-test-<lambda>1]",
    "tests/frame/methods/test_set_index.py::TestSetIndex::test_set_index_pass_single_array[True-True-test-Index]",
    "tests/frame/methods/test_set_index.py::TestSetIndexCustomLabelType::test_set_index_custom_label_hashable_iterable",
    "tests/frame/methods/test_shift.py::TestDataFrameShift::test_shift_dt64values_axis1_invalid_fill[datetime64[ns]-False]",
    "tests/frame/methods/test_shift.py::TestDataFrameShift::test_shift_dt64values_axis1_invalid_fill[timedelta64[ns]-False]",
    "tests/frame/methods/test_shift.py::TestDataFrameShift::test_shift_dt64values_int_fill_deprecated",
    "tests/frame/methods/test_sort_index.py::TestDataFrameSortIndex::test_sort_index",
    "tests/frame/methods/test_sort_index.py::TestDataFrameSortIndex::test_sort_index_inplace",
    "tests/frame/methods/test_sort_index.py::TestDataFrameSortIndex::test_sort_index_intervalindex",
    "tests/frame/methods/test_sort_index.py::TestDataFrameSortIndex::test_sort_index_nan",
    "tests/frame/methods/test_sort_index.py::TestDataFrameSortIndex::test_sort_index_reorder_on_ops",
    "tests/frame/methods/test_sort_values.py::TestDataFrameSortValues::test_sort_values_by_empty_list",
    "tests/frame/methods/test_sort_values.py::TestDataFrameSortValues::test_sort_values_frame_column_inplace_sort_exception",
    "tests/frame/methods/test_sort_values.py::TestDataFrameSortValues::test_sort_values_item_cache",
    "tests/frame/methods/test_sort_values.py::TestSortValuesLevelAsStr::test_sort_index_level_and_column_label[df_idx1-inner-True]",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_gz_lineend",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_multi_index_header",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_from_csv1",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_from_csv2",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_from_csv5",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_from_csv_w_all_infs",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_from_csv_w_some_infs",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_headers",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_multiindex",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_quoting",
    "tests/frame/methods/test_to_csv.py::TestDataFrameToCSV::test_to_csv_single_level_multi_index",
    "tests/frame/methods/test_to_dict.py::TestDataFrameToDict::test_to_dict_index_false_error[series]",
    "tests/frame/methods/test_to_dict.py::TestDataFrameToDict::test_to_dict_orient_dtype[data1-Timestamp]",
    "tests/frame/methods/test_to_dict.py::TestDataFrameToDict::test_to_dict_scalar_constructor_orient_dtype[data4-Timestamp]",
    "tests/frame/methods/test_to_dict_of_blocks.py::TestToDictOfBlocks::test_no_copy_blocks",
    "tests/frame/methods/test_to_dict_of_blocks.py::test_set_change_dtype_slice",
    "tests/frame/methods/test_to_dict_of_blocks.py::test_to_dict_of_blocks_item_cache",
    "tests/frame/methods/test_to_numpy.py::TestToNumpy::test_to_numpy_copy",
    "tests/frame/methods/test_to_numpy.py::TestToNumpy::test_to_numpy_mixed_dtype_to_str",
    "tests/frame/methods/test_to_records.py::TestDataFrameToRecords::test_to_records_dt64tz_column",
    "tests/frame/methods/test_to_records.py::TestDataFrameToRecords::test_to_records_with_unicode_index",
    "tests/frame/methods/test_transpose.py::TestTranspose::test_transpose_get_view",
    "tests/frame/methods/test_truncate.py::TestDataFrameTruncate::test_truncate_multiindex[DataFrame]",
    "tests/frame/methods/test_tz_localize.py::TestTZLocalize::test_tz_localize_copy_inplace_mutate[DataFrame-False]",
    "tests/frame/methods/test_tz_localize.py::TestTZLocalize::test_tz_localize_copy_inplace_mutate[DataFrame-True]",
    "tests/frame/methods/test_tz_localize.py::TestTZLocalize::test_tz_localize_copy_inplace_mutate[Series-False]",
    "tests/frame/methods/test_tz_localize.py::TestTZLocalize::test_tz_localize_copy_inplace_mutate[Series-True]",
    "tests/frame/methods/test_update.py::TestDataFrameUpdate::test_update_modify_view",
    "tests/frame/methods/test_value_counts.py::test_data_frame_value_counts_dropna_false[NoneType]",
    "tests/frame/methods/test_value_counts.py::test_data_frame_value_counts_empty",
    "tests/frame/methods/test_value_counts.py::test_data_frame_value_counts_empty_normalize",
    "tests/frame/methods/test_value_counts.py::test_data_frame_value_counts_subset[NoneType-columns1]",
    "tests/frame/methods/test_value_counts.py::test_value_counts_with_missing_category",
    "tests/frame/methods/test_values.py::TestDataFrameValues::test_interleave_with_tzaware",
    "tests/frame/methods/test_values.py::TestDataFrameValues::test_values",
    "tests/frame/methods/test_values.py::TestPrivateValues::test_private_values_dt64_multiblock",
    "tests/frame/test_api.py::TestDataFrameMisc::test_attrs",
    "tests/frame/test_api.py::TestDataFrameMisc::test_attrs_deepcopy",
    "tests/frame/test_api.py::TestDataFrameMisc::test_column_contains_raises",
    "tests/frame/test_api.py::TestDataFrameMisc::test_display_max_dir_items",
    "tests/frame/test_api.py::TestDataFrameMisc::test_get_agg_axis",
    "tests/frame/test_api.py::TestDataFrameMisc::test_get_axis",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[DataFrame-False]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[DataFrame-None]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[DataFrame-True]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[Series-False]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[Series-None]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_set_flags[Series-True]",
    "tests/frame/test_api.py::TestDataFrameMisc::test_tab_completion",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_arithmetic_midx_cols_different_dtypes_different_order[numexpr]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_arithmetic_midx_cols_different_dtypes_different_order[python]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[numexpr-pow-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[numexpr-sub-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[numexpr-truediv-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[python-pow-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[python-sub-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_binop_other[python-truediv-True-bool]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_frame_with_frame_reindex[numexpr]",
    "tests/frame/test_arithmetic.py::TestFrameArithmetic::test_frame_with_frame_reindex[python]",
    "tests/frame/test_arithmetic.py::TestFrameArithmeticUnsorted::test_inplace_ops_identity[numexpr]",
    "tests/frame/test_arithmetic.py::TestFrameArithmeticUnsorted::test_inplace_ops_identity[python]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[numexpr-ge-le]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[numexpr-gt-lt]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[numexpr-le-ge]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[numexpr-lt-gt]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[python-ge-le]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[python-gt-lt]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[python-le-ge]",
    "tests/frame/test_arithmetic.py::TestFrameComparisons::test_timestamp_compare[python-lt-gt]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_arith_flex_frame_corner[numexpr]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_arith_flex_frame_corner[python]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_broadcast_multiindex[numexpr-None]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_broadcast_multiindex[python-None]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_flex_add_scalar_fill_value[numexpr]",
    "tests/frame/test_arithmetic.py::TestFrameFlexArithmetic::test_flex_add_scalar_fill_value[python]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[numexpr-eq]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[numexpr-ge]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[numexpr-gt]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[numexpr-le]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[numexpr-lt]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[numexpr-ne]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[python-eq]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[python-ge]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[python-gt]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[python-le]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[python-lt]",
    "tests/frame/test_arithmetic.py::TestFrameFlexComparisons::test_bool_flex_frame[python-ne]",
    "tests/frame/test_arithmetic.py::test_arithmetic_multiindex_align[numexpr]",
    "tests/frame/test_arithmetic.py::test_arithmetic_multiindex_align[python]",
    "tests/frame/test_arithmetic.py::test_bool_frame_mult_float[numexpr]",
    "tests/frame/test_arithmetic.py::test_bool_frame_mult_float[python]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[numexpr-Int16]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[numexpr-Int32]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[numexpr-Int64]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[numexpr-Int8]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[numexpr-UInt16]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[numexpr-UInt32]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[numexpr-UInt64]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[numexpr-UInt8]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[python-Int16]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[python-Int32]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[python-Int64]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[python-Int8]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[python-UInt16]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[python-UInt32]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[python-UInt64]",
    "tests/frame/test_arithmetic.py::test_frame_sub_nullable_int[python-UInt8]",
    "tests/frame/test_arithmetic.py::test_frame_with_zero_len_series_corner_cases[numexpr]",
    "tests/frame/test_arithmetic.py::test_frame_with_zero_len_series_corner_cases[python]",
    "tests/frame/test_arithmetic.py::test_inplace_arithmetic_series_update[numexpr]",
    "tests/frame/test_arithmetic.py::test_inplace_arithmetic_series_update[python]",
    "tests/frame/test_arithmetic.py::test_pow_with_realignment[numexpr]",
    "tests/frame/test_arithmetic.py::test_pow_with_realignment[python]",
    "tests/frame/test_block_internals.py::TestDataFrameBlockInternals::test_consolidate",
    "tests/frame/test_block_internals.py::TestDataFrameBlockInternals::test_construction_with_conversions",
    "tests/frame/test_block_internals.py::TestDataFrameBlockInternals::test_constructor_with_convert",
    "tests/frame/test_block_internals.py::TestDataFrameBlockInternals::test_modify_values",
    "tests/frame/test_block_internals.py::TestDataFrameBlockInternals::test_stale_cached_series_bug_473",
    "tests/frame/test_block_internals.py::test_update_inplace_sets_valid_block_values",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_columns_indexes_raise_on_sets",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_construction_from_ndarray_datetimelike",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_construction_from_set_raises[frozenset]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_construction_from_set_raises[set]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['+01:15'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['-02:15'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['Asia/Tokyo'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['US/Eastern'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['UTC'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['UTC+01:15'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['UTC-02:15'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['dateutil/Asia/Singapore'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive['dateutil/US/Pacific'-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[<UTC>-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[datetime.timezone(datetime.timedelta(seconds=3600))-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[datetime.timezone.utc-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[pytz.FixedOffset(-300)-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[pytz.FixedOffset(300)-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[tzlocal()-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[tzutc()-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[zoneinfo.ZoneInfo(key='US/Pacific')-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDatetimeTZ::test_constructor_data_aware_dtype_naive[zoneinfo.ZoneInfo(key='UTC')-True]",
    "tests/frame/test_constructors.py::TestDataFrameConstructorWithDtypeCoercion::test_floating_values_integer_dtype",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_1d_object_array_does_not_copy",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_2d_object_array_does_not_copy",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_construct_from_list_of_datetimes",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_construct_ndarray_with_nas_and_int_dtype",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_construct_with_two_categoricalindex_series",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_non_ns[h-A]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_non_ns[h-C]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_non_ns[h-F]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_non_ns[h-K]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_non_ns[m-A]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_non_ns[m-C]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_non_ns[m-F]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_non_ns[m-K]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_with_nulls[arr0]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_datetimes_with_nulls[arr1]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_cast",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_datetime64_index",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_multiindex",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_nan_key[None]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_nan_key_and_columns",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_timedelta64_index[<lambda>-pytimedelta]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_with_index",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dict_with_index_and_columns",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_dtype_nocast_view_2d_array",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_error_msgs",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_for_list_with_dtypes",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_list_of_2d_raises",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_lists_to_object_dtype",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_maskedarray",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_maskedarray_hardened",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_maskedarray_nonfloat",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_maskedrecarray_dtype",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_miscast_na_int_dtype",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_mixed_dict_and_Series",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_ndarray_copy",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_rec",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_timedelta_non_ns[h-A]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_timedelta_non_ns[h-C]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_timedelta_non_ns[h-F]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_timedelta_non_ns[h-K]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_timedelta_non_ns[m-A]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_timedelta_non_ns[m-C]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_timedelta_non_ns[m-F]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_timedelta_non_ns[m-K]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_with_datetimes1",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_with_datetimes2",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_constructor_with_datetimes3",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_datetimelike_values_with_object_dtype[DataFrame-M]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_datetimelike_values_with_object_dtype[DataFrame-m]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_datetimelike_values_with_object_dtype[Series-M]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_datetimelike_values_with_object_dtype[Series-m]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float32-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Float64-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int16-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int32-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int64-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[Int8-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt16-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt32-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt64-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-M8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-bool0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-bool1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-complex-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-complex128-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-complex64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-datetime64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-float-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-float32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-float64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-int-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-int16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-int32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-int64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-int8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-m8[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-object0-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-object1-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-timedelta64[ns]-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-uint16-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-uint32-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-uint64-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_dict_nocopy[UInt8-uint8-False]",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_frame_string_inference",
    "tests/frame/test_constructors.py::TestDataFrameConstructors::test_from_datetime_subclass",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_datetime[DataFrame-None-datetime64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_datetime[DataFrame-dict-datetime64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_datetime[Series-None-datetime64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_timedelta[DataFrame-None-timedelta64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_timedelta[DataFrame-dict-timedelta64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_from_out_of_bounds_ns_timedelta[Series-None-timedelta64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_datetime64[DataFrame-dict]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_datetime64[Series-None]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_timedelta64[DataFrame-dict-datetime64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_timedelta64[DataFrame-dict-timedelta64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_timedelta64[Series-None-datetime64]",
    "tests/frame/test_constructors.py::TestFromScalar::test_out_of_s_bounds_timedelta64[Series-None-timedelta64]",
    "tests/frame/test_iteration.py::TestIteration::test_iteritems",
    "tests/frame/test_iteration.py::TestIteration::test_itertuples",
    "tests/frame/test_iteration.py::TestIteration::test_keys",
    "tests/frame/test_logical_ops.py::TestDataFrameLogicalOperators::test_different_dtypes_different_index_raises",
    "tests/frame/test_logical_ops.py::TestDataFrameLogicalOperators::test_int_dtype_different_index_not_bool",
    "tests/frame/test_logical_ops.py::TestDataFrameLogicalOperators::test_logical_with_nas",
    "tests/frame/test_npfuncs.py::TestAsArray::test_np_sqrt",
    "tests/frame/test_query_eval.py::TestDataFrameEval::test_dataframe_sub_numexpr_path",
    "tests/frame/test_query_eval.py::TestDataFrameQueryNumExprPython::test_query_numexpr_with_min_and_max_columns",
    "tests/frame/test_query_eval.py::TestDataFrameQueryPythonPython::test_query_numexpr_with_min_and_max_columns",
    "tests/frame/test_query_eval.py::TestDataFrameQueryWithMultiIndex::test_query_multiindex_get_index_resolvers",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_bool_frame[any]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_extra",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data1-True]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data11-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data13-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data14-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data16-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data3-True]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data30-True]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data32-True]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data5-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data7-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[all-data9-True]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[any-data0-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[any-data31-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_any_all_np_func[any-data33-False]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_idxmax_idxmin_convert_dtypes[idxmax-expected_value0]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_idxmax_idxmin_convert_dtypes[idxmin-expected_value1]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_mode_dropna[False-expected3]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_nunique",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_operators_timedelta64",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_stat_op_calc",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[False-ms-values0]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[False-ms-values1]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[False-ns-values0]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[False-ns-values1]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[False-s-values0]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[False-s-values1]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[False-us-values0]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[False-us-values1]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[True-ms-values0]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[True-ms-values1]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[True-ns-values1]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[True-s-values0]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[True-s-values1]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[True-us-values0]",
    "tests/frame/test_reductions.py::TestDataFrameAnalytics::test_std_datetime64_with_nat[True-us-values1]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_min_max_dt64_api_consistency_empty_df",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_min_max_dt64_api_consistency_with_NaT",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reduction_timedelta_smallest_unit",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reduction_timestamp_smallest_unit",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-all]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-any]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-kurt]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-max]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-mean]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-median]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-min]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-prod]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-skew]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-std]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-sum]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[DataFrame-var]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-all]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-any]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-kurt]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-max]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-mean]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-median]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-min]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-prod]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-skew]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-std]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-sum]",
    "tests/frame/test_reductions.py::TestDataFrameReductions::test_reductions_skipna_none_raises[Series-var]",
    "tests/frame/test_reductions.py::TestEmptyDataFrameReductions::test_df_empty_min_count_0[prod-int8-1-int64]",
    "tests/frame/test_reductions.py::TestEmptyDataFrameReductions::test_df_empty_min_count_0[prod-uint8-1-uint64]",
    "tests/frame/test_reductions.py::TestEmptyDataFrameReductions::test_df_empty_min_count_0[sum-int8-0-int64]",
    "tests/frame/test_reductions.py::TestEmptyDataFrameReductions::test_df_empty_min_count_0[sum-uint8-0-uint64]",
    "tests/frame/test_reductions.py::TestEmptyDataFrameReductions::test_df_empty_nullable_min_count_0[prod-Int8-1-Int640]",
    "tests/frame/test_reductions.py::TestEmptyDataFrameReductions::test_df_empty_nullable_min_count_0[prod-Int8-1-Int641]",
    "tests/frame/test_reductions.py::TestEmptyDataFrameReductions::test_df_empty_nullable_min_count_0[prod-Int8-1-Int64_0]",
    "tests/frame/test_reductions.py::TestEmptyDataFrameReductions::test_df_empty_nullable_min_count_0[prod-Int8-1-Int64_1]",
    "tests/frame/test_reductions.py::TestEmptyDataFrameReductions::test_df_empty_nullable_min_count_0[prod-UInt8-1-UInt64]",
    "tests/frame/test_reductions.py::TestEmptyDataFrameReductions::test_df_empty_nullable_min_count_0[sum-Int8-0-Int64]",
    "tests/frame/test_reductions.py::TestEmptyDataFrameReductions::test_df_empty_nullable_min_count_0[sum-UInt8-0-UInt64]",
    "tests/frame/test_reductions.py::TestNuisanceColumns::test_any_all_categorical_dtype_nuisance_column[all]",
    "tests/frame/test_reductions.py::TestNuisanceColumns::test_any_all_categorical_dtype_nuisance_column[any]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-0-kurt]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-0-kurtosis]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-0-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-0-sem]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-0-skew]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-2-kurt]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-2-kurtosis]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-2-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-2-sem]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-2-skew]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Float32-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int16-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int16-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int16-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int32-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int32-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int32-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int64-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int64-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int64-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int8-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int8-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-Int8-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt16-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt16-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt16-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt32-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt32-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt32-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt64-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt64-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt64-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt8-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt8-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[False-UInt8-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-0-kurt]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-0-kurtosis]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-0-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-0-sem]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-0-skew]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-2-kurt]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-2-kurtosis]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-2-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-2-sem]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-2-skew]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float32-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float64-0-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Float64-2-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int16-0-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int16-2-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int16-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int16-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int16-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int32-0-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int32-2-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int32-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int32-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int32-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int64-0-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int64-2-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int64-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int64-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int64-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int8-0-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int8-2-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int8-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int8-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-Int8-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt16-0-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt16-2-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt16-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt16-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt16-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt32-0-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt32-2-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt32-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt32-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt32-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt64-0-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt64-2-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt64-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt64-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt64-2-sum]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt8-0-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt8-2-median]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt8-2-prod]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt8-2-product]",
    "tests/frame/test_reductions.py::test_numeric_ea_axis_1[True-UInt8-2-sum]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-False-mean]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-False-median]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-True-mean]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[Float64-True-median]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-False-kurt]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-False-mean]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-False-median]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-False-skew]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-True-kurt]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-True-mean]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-True-median]",
    "tests/frame/test_reductions.py::test_reduction_axis_none_returns_scalar[float64-True-skew]",
    "tests/frame/test_reductions.py::test_sum_timedelta64_skipna_false",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_int_level_names[False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_int_level_names[True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_mixed_levels[False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_mixed_levels[True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_columns_non_unique_index[False-index0-columns0]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_columns_non_unique_index[False-index1-columns1]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_columns_non_unique_index[False-index2-columns2]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_columns_non_unique_index[True-index0-columns0]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_columns_non_unique_index[True-index1-columns1]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_columns_non_unique_index[True-index2-columns2]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[False-labels0-data0-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[False-labels0-data0-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[False-labels1-data1-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[False-labels1-data1-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[True-labels0-data0-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[True-labels0-data0-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[True-labels1-data1-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_multi_preserve_categorical_dtype[True-labels1-data1-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_preserve_categorical_dtype[False-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_preserve_categorical_dtype[False-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_preserve_categorical_dtype[True-False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_preserve_categorical_dtype[True-True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_unstack[False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_stack_unstack[True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_bool",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_long_index",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_mixed_extension_types[0]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_mixed_extension_types[1]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_multi_level_cols",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_multi_level_rows_and_cols",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_nan_index2",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_nan_index3",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_non_unique_index_names[False]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_non_unique_index_names[True]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_not_consolidated",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_swaplevel_sortlevel[0]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_swaplevel_sortlevel[baz]",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_to_series",
    "tests/frame/test_stack_unstack.py::TestDataFrameReshape::test_unstack_unused_levels",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_multi_level_stack_categorical[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_multi_level_stack_categorical[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_level_name[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_level_name[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_mixed_dtype[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_multiple_bug[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_multiple_bug[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_multiple_out_of_bounds[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_multiple_out_of_bounds[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_nan_in_multiindex_columns[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_nan_in_multiindex_columns[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_nullable_dtype[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_nullable_dtype[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_order_with_unsorted_levels_multi_row_2[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_multiple[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_multiple[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_preserve_names[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_preserve_names[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_wrong_level_name[False-unstack]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_stack_unstack_wrong_level_name[True-unstack]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_unstack_categorical_columns",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_unstack_preserve_types",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_unstack_unobserved_keys[False]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_unstack_unobserved_keys[True]",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_unstack_with_level_has_nan",
    "tests/frame/test_stack_unstack.py::TestStackUnstackMultiLevel::test_unstack_with_missing_int_cast_to_float",
    "tests/frame/test_stack_unstack.py::test_unstack_non_slice_like_blocks",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_asof",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_equals_subclass",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_frame_subclassing_and_slicing",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_subclass_stack_multi",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_subclass_stack_multi_mixed",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_subclass_unstack_multi",
    "tests/frame/test_subclass.py::TestDataFrameSubclassing::test_subclass_unstack_multi_mixed",
    "tests/frame/test_ufunc.py::test_alignment_deprecation_enforced",
    "tests/frame/test_ufunc.py::test_alignment_deprecation_many_inputs_enforced",
    "tests/frame/test_ufunc.py::test_array_ufuncs_for_many_arguments",
    "tests/frame/test_ufunc.py::test_binary_input_aligns_columns[int64-int64]",
    "tests/frame/test_ufunc.py::test_binary_input_aligns_index[int64]",
    "tests/frame/test_ufunc.py::test_binary_input_dispatch_binop[Int64]",
    "tests/frame/test_ufunc.py::test_binary_input_dispatch_binop[dtype2]",
    "tests/frame/test_ufunc.py::test_binary_input_dispatch_binop[int64]",
    "tests/frame/test_ufunc.py::test_frame_outer_disallowed",
    "tests/frame/test_ufunc.py::test_ufunc_passes_args[add-1-expected0]",
    "tests/frame/test_ufunc.py::test_ufunc_passes_args[func1-arg1-expected1]",
    "tests/frame/test_ufunc.py::test_ufunc_passes_args[func4-None-expected4]",
    "tests/frame/test_ufunc.py::test_ufunc_passes_args[power-arg2-expected2]",
    "tests/frame/test_ufunc.py::test_ufunc_passes_args[subtract-2-expected3]",
    "tests/frame/test_ufunc.py::test_unary_accumulate_axis",
    "tests/frame/test_ufunc.py::test_unary_binary[int64]",
    "tests/frame/test_ufunc.py::test_unary_unary[Int64]",
    "tests/frame/test_ufunc.py::test_unary_unary[dtype2]",
    "tests/frame/test_ufunc.py::test_unary_unary[int64]",
    "tests/frame/test_unary.py::TestDataFrameUnaryOperators::test_pos_object_raises[df0]",
    "tests/frame/test_unary.py::TestDataFrameUnaryOperators::test_pos_raises[df0]",
    "tests/frame/test_unary.py::TestDataFrameUnaryOperators::test_unary_nullable",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[1-drop_duplicates]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[1-dropna]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[1-reset_index]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[1-set_index]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[5.0-drop_duplicates]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[5.0-dropna]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[5.0-reset_index]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[5.0-set_index]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[True-drop_duplicates]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[True-dropna]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[True-reset_index]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[True-set_index]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[inplace2-drop_duplicates]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[inplace2-dropna]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[inplace2-reset_index]",
    "tests/frame/test_validate.py::TestDataFrameValidate::test_validate_bool_args[inplace2-set_index]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[1-False-add]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[1-False-sub]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[1-True-add]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[1-True-sub]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[other1-False-add]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[other1-False-sub]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[other1-True-add]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_binops[other1-True-sub]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs0-kwargs0]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs1-kwargs1]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs2-kwargs2]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs3-kwargs3]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs4-kwargs4]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs5-kwargs5]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_concat[objs6-kwargs6]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_construction_ok[DataFrame-data2]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_construction_ok[DataFrame-data3]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_construction_ok[Series-data0]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_construction_ok[Series-data1]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserve_getitem",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_frame",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[absolute]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[func0]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[func1]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[func2]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[func3]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_preserved_series[func4]",
    "tests/generic/test_duplicate_labels.py::TestPreserves::test_to_frame",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_concat_raises[objs0-kwargs0]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter0-None]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter1-loc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter2-loc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter3-loc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter4-iloc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter5-iloc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_getitem_raises[getter6-iloc]",
    "tests/generic/test_duplicate_labels.py::TestRaises::test_series_raises",
    "tests/generic/test_duplicate_labels.py::test_dataframe_insert_raises",
    "tests/generic/test_duplicate_labels.py::test_inplace_raises[method0-True]",
    "tests/generic/test_duplicate_labels.py::test_inplace_raises[method1-True]",
    "tests/generic/test_duplicate_labels.py::test_inplace_raises[method2-False]",
    "tests/generic/test_finalize.py::test_binops[add-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[add-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[add-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[add-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[add-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[add-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[add-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[add-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[add-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[add-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[add-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[add-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[add-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[add-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[add-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[add-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[and_-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[and_-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[and_-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[and_-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[and_-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[and_-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[and_-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[and_-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[and_-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[and_-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[and_-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[and_-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[and_-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[and_-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[and_-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[and_-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[eq-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[eq-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[eq-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[eq-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[eq-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[eq-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[eq-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[eq-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[eq-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[eq-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[eq-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[eq-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[eq-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[eq-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[eq-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[eq-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[floordiv-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[ge-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[ge-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[ge-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[ge-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[ge-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[ge-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[ge-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[ge-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[ge-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[ge-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[ge-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[ge-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[ge-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[ge-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[ge-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[ge-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[gt-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[gt-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[gt-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[gt-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[gt-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[gt-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[gt-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[gt-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[gt-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[gt-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[gt-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[gt-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[gt-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[gt-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[gt-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[gt-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[le-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[le-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[le-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[le-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[le-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[le-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[le-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[le-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[le-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[le-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[le-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[le-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[le-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[le-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[le-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[le-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[lt-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[lt-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[lt-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[lt-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[lt-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[lt-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[lt-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[lt-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[lt-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[lt-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[lt-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[lt-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[lt-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[lt-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[lt-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[lt-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[mod-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[mod-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[mod-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[mod-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[mod-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[mod-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[mod-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[mod-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[mod-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[mod-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[mod-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[mod-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[mod-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[mod-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[mod-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[mod-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[mul-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[mul-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[mul-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[mul-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[mul-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[mul-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[mul-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[mul-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[mul-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[mul-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[mul-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[mul-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[mul-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[mul-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[mul-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[mul-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[ne-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[ne-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[ne-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[ne-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[ne-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[ne-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[ne-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[ne-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[ne-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[ne-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[ne-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[ne-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[ne-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[ne-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[ne-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[ne-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[or_-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[or_-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[or_-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[or_-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[or_-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[or_-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[or_-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[or_-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[or_-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[or_-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[or_-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[or_-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[or_-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[or_-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[or_-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[or_-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[pow-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[pow-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[pow-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[pow-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[pow-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[pow-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[pow-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[pow-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[pow-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[pow-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[pow-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[pow-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[pow-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[pow-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[pow-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[pow-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[radd-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[radd-(DataFrame,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[radd-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[radd-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[radd-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[radd-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[radd-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[radd-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[radd-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[radd-(Series,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[radd-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[radd-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[radd-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[radd-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[radd-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[radd-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rand_-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rand_-(DataFrame,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rand_-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rand_-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[rand_-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rand_-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rand_-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rand_-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rand_-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rand_-(Series,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rand_-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rand_-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rand_-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rand_-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rand_-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rand_-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(DataFrame,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(Series,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rfloordiv-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rmod-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rmod-(DataFrame,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rmod-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rmod-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[rmod-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rmod-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rmod-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rmod-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rmod-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rmod-(Series,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rmod-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rmod-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rmod-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rmod-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rmod-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rmod-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rmul-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rmul-(DataFrame,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rmul-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rmul-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[rmul-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rmul-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rmul-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rmul-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rmul-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rmul-(Series,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rmul-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rmul-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rmul-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rmul-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rmul-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rmul-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[ror_-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[ror_-(DataFrame,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[ror_-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[ror_-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[ror_-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[ror_-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[ror_-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[ror_-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[ror_-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[ror_-(Series,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[ror_-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[ror_-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[ror_-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[ror_-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[ror_-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[ror_-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rpow-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rpow-(DataFrame,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rpow-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rpow-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[rpow-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rpow-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rpow-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rpow-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rpow-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rpow-(Series,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rpow-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rpow-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rpow-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rpow-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rpow-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rpow-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rsub-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rsub-(DataFrame,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rsub-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rsub-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[rsub-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rsub-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rsub-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rsub-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rsub-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rsub-(Series,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rsub-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rsub-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rsub-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rsub-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rsub-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rsub-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(DataFrame,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(Series,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rtruediv-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rxor-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rxor-(DataFrame,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rxor-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rxor-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[rxor-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rxor-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rxor-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rxor-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rxor-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rxor-(Series,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[rxor-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[rxor-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[rxor-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[rxor-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[rxor-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[rxor-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[sub-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[sub-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[sub-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[sub-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[sub-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[sub-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[sub-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[sub-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[sub-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[sub-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[sub-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[sub-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[sub-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[sub-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[sub-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[sub-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[truediv-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[truediv-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[truediv-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[truediv-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[truediv-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[truediv-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[truediv-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[truediv-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[truediv-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[truediv-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[truediv-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[truediv-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[truediv-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[truediv-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[truediv-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[truediv-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_binops[xor-(DataFrame,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[xor-(DataFrame,DataFrame)-left]",
    "tests/generic/test_finalize.py::test_binops[xor-(DataFrame,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[xor-(DataFrame,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[xor-(DataFrame,int)-both]",
    "tests/generic/test_finalize.py::test_binops[xor-(DataFrame,int)-left]",
    "tests/generic/test_finalize.py::test_binops[xor-(Series,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[xor-(Series,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[xor-(Series,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[xor-(Series,Series)-left]",
    "tests/generic/test_finalize.py::test_binops[xor-(Series,int)-both]",
    "tests/generic/test_finalize.py::test_binops[xor-(Series,int)-left]",
    "tests/generic/test_finalize.py::test_binops[xor-(int,DataFrame)-both]",
    "tests/generic/test_finalize.py::test_binops[xor-(int,DataFrame)-right]",
    "tests/generic/test_finalize.py::test_binops[xor-(int,Series)-both]",
    "tests/generic/test_finalize.py::test_binops[xor-(int,Series)-right]",
    "tests/generic/test_finalize.py::test_datetime_method[ceil', 'h]",
    "tests/generic/test_finalize.py::test_datetime_method[day_name]",
    "tests/generic/test_finalize.py::test_datetime_method[floor', 'h]",
    "tests/generic/test_finalize.py::test_datetime_method[month_name]",
    "tests/generic/test_finalize.py::test_datetime_method[normalize]",
    "tests/generic/test_finalize.py::test_datetime_method[round', 'h]",
    "tests/generic/test_finalize.py::test_datetime_method[strftime', '%Y]",
    "tests/generic/test_finalize.py::test_datetime_method[to_period]",
    "tests/generic/test_finalize.py::test_datetime_method[tz_localize', 'CET]",
    "tests/generic/test_finalize.py::test_datetime_property[date]",
    "tests/generic/test_finalize.py::test_datetime_property[day]",
    "tests/generic/test_finalize.py::test_datetime_property[day_of_week]",
    "tests/generic/test_finalize.py::test_datetime_property[day_of_year]",
    "tests/generic/test_finalize.py::test_datetime_property[dayofweek]",
    "tests/generic/test_finalize.py::test_datetime_property[dayofyear]",
    "tests/generic/test_finalize.py::test_datetime_property[days_in_month]",
    "tests/generic/test_finalize.py::test_datetime_property[daysinmonth]",
    "tests/generic/test_finalize.py::test_datetime_property[hour]",
    "tests/generic/test_finalize.py::test_datetime_property[is_leap_year]",
    "tests/generic/test_finalize.py::test_datetime_property[is_month_end]",
    "tests/generic/test_finalize.py::test_datetime_property[is_month_start]",
    "tests/generic/test_finalize.py::test_datetime_property[is_quarter_end]",
    "tests/generic/test_finalize.py::test_datetime_property[is_quarter_start]",
    "tests/generic/test_finalize.py::test_datetime_property[is_year_end]",
    "tests/generic/test_finalize.py::test_datetime_property[is_year_start]",
    "tests/generic/test_finalize.py::test_datetime_property[microsecond]",
    "tests/generic/test_finalize.py::test_datetime_property[minute]",
    "tests/generic/test_finalize.py::test_datetime_property[month]",
    "tests/generic/test_finalize.py::test_datetime_property[nanosecond]",
    "tests/generic/test_finalize.py::test_datetime_property[quarter]",
    "tests/generic/test_finalize.py::test_datetime_property[second]",
    "tests/generic/test_finalize.py::test_datetime_property[time]",
    "tests/generic/test_finalize.py::test_datetime_property[timetz]",
    "tests/generic/test_finalize.py::test_datetime_property[year]",
    "tests/generic/test_finalize.py::test_finalize_called[<built-in function abs>0]",
    "tests/generic/test_finalize.py::test_finalize_called[<built-in function abs>1]",
    "tests/generic/test_finalize.py::test_finalize_called[<built-in function inv>0]",
    "tests/generic/test_finalize.py::test_finalize_called[<built-in function inv>1]",
    "tests/generic/test_finalize.py::test_finalize_called[<built-in function neg>0]",
    "tests/generic/test_finalize.py::test_finalize_called[<built-in function neg>1]",
    "tests/generic/test_finalize.py::test_finalize_called[<built-in function pos>0]",
    "tests/generic/test_finalize.py::test_finalize_called[<built-in function pos>1]",
    "tests/generic/test_finalize.py::test_finalize_called[<built-in function round>0]",
    "tests/generic/test_finalize.py::test_finalize_called[<built-in function round>1]",
    "tests/generic/test_finalize.py::test_finalize_called[__getitem__', 'A]",
    "tests/generic/test_finalize.py::test_finalize_called[__getitem__', ['A0]",
    "tests/generic/test_finalize.py::test_finalize_called[__getitem__', ['A1]",
    "tests/generic/test_finalize.py::test_finalize_called[__getitem__0]",
    "tests/generic/test_finalize.py::test_finalize_called[__getitem__1]",
    "tests/generic/test_finalize.py::test_finalize_called[add0]",
    "tests/generic/test_finalize.py::test_finalize_called[add1]",
    "tests/generic/test_finalize.py::test_finalize_called[add_prefix', '_0]",
    "tests/generic/test_finalize.py::test_finalize_called[add_prefix', '_1]",
    "tests/generic/test_finalize.py::test_finalize_called[add_suffix', '_0]",
    "tests/generic/test_finalize.py::test_finalize_called[add_suffix', '_1]",
    "tests/generic/test_finalize.py::test_finalize_called[all]",
    "tests/generic/test_finalize.py::test_finalize_called[any]",
    "tests/generic/test_finalize.py::test_finalize_called[apply0]",
    "tests/generic/test_finalize.py::test_finalize_called[apply1]",
    "tests/generic/test_finalize.py::test_finalize_called[assign]",
    "tests/generic/test_finalize.py::test_finalize_called[astype', <class 'float0]",
    "tests/generic/test_finalize.py::test_finalize_called[astype', <class 'float1]",
    "tests/generic/test_finalize.py::test_finalize_called[between]",
    "tests/generic/test_finalize.py::test_finalize_called[clip0]",
    "tests/generic/test_finalize.py::test_finalize_called[clip1]",
    "tests/generic/test_finalize.py::test_finalize_called[combine]",
    "tests/generic/test_finalize.py::test_finalize_called[combine_first]",
    "tests/generic/test_finalize.py::test_finalize_called[convert_dtypes0]",
    "tests/generic/test_finalize.py::test_finalize_called[convert_dtypes1]",
    "tests/generic/test_finalize.py::test_finalize_called[copy0]",
    "tests/generic/test_finalize.py::test_finalize_called[copy1]",
    "tests/generic/test_finalize.py::test_finalize_called[corr]",
    "tests/generic/test_finalize.py::test_finalize_called[corrwith]",
    "tests/generic/test_finalize.py::test_finalize_called[count]",
    "tests/generic/test_finalize.py::test_finalize_called[cov]",
    "tests/generic/test_finalize.py::test_finalize_called[cummax0]",
    "tests/generic/test_finalize.py::test_finalize_called[cummax1]",
    "tests/generic/test_finalize.py::test_finalize_called[cummin0]",
    "tests/generic/test_finalize.py::test_finalize_called[cummin1]",
    "tests/generic/test_finalize.py::test_finalize_called[cumprod0]",
    "tests/generic/test_finalize.py::test_finalize_called[cumprod1]",
    "tests/generic/test_finalize.py::test_finalize_called[cumsum0]",
    "tests/generic/test_finalize.py::test_finalize_called[cumsum1]",
    "tests/generic/test_finalize.py::test_finalize_called[describe0]",
    "tests/generic/test_finalize.py::test_finalize_called[describe1]",
    "tests/generic/test_finalize.py::test_finalize_called[drop', columns=['A]",
    "tests/generic/test_finalize.py::test_finalize_called[drop0]",
    "tests/generic/test_finalize.py::test_finalize_called[drop1]",
    "tests/generic/test_finalize.py::test_finalize_called[drop_duplicates0]",
    "tests/generic/test_finalize.py::test_finalize_called[drop_duplicates1]",
    "tests/generic/test_finalize.py::test_finalize_called[droplevel', 'A]",
    "tests/generic/test_finalize.py::test_finalize_called[dropna]",
    "tests/generic/test_finalize.py::test_finalize_called[duplicated0]",
    "tests/generic/test_finalize.py::test_finalize_called[duplicated1]",
    "tests/generic/test_finalize.py::test_finalize_called[eval', 'A + 1', engine='python]",
    "tests/generic/test_finalize.py::test_finalize_called[explode', 'A]",
    "tests/generic/test_finalize.py::test_finalize_called[fillna', 'A]",
    "tests/generic/test_finalize.py::test_finalize_called[fillna', method='ffill]",
    "tests/generic/test_finalize.py::test_finalize_called[fillna]",
    "tests/generic/test_finalize.py::test_finalize_called[get', 'A]",
    "tests/generic/test_finalize.py::test_finalize_called[head0]",
    "tests/generic/test_finalize.py::test_finalize_called[head1]",
    "tests/generic/test_finalize.py::test_finalize_called[idxmax]",
    "tests/generic/test_finalize.py::test_finalize_called[idxmin]",
    "tests/generic/test_finalize.py::test_finalize_called[infer_objects0]",
    "tests/generic/test_finalize.py::test_finalize_called[infer_objects1]",
    "tests/generic/test_finalize.py::test_finalize_called[interpolate0]",
    "tests/generic/test_finalize.py::test_finalize_called[interpolate1]",
    "tests/generic/test_finalize.py::test_finalize_called[isin0]",
    "tests/generic/test_finalize.py::test_finalize_called[isin1]",
    "tests/generic/test_finalize.py::test_finalize_called[isin2]",
    "tests/generic/test_finalize.py::test_finalize_called[isin3]",
    "tests/generic/test_finalize.py::test_finalize_called[isna0]",
    "tests/generic/test_finalize.py::test_finalize_called[isna1]",
    "tests/generic/test_finalize.py::test_finalize_called[isnull0]",
    "tests/generic/test_finalize.py::test_finalize_called[isnull1]",
    "tests/generic/test_finalize.py::test_finalize_called[kurt]",
    "tests/generic/test_finalize.py::test_finalize_called[map]",
    "tests/generic/test_finalize.py::test_finalize_called[mask0]",
    "tests/generic/test_finalize.py::test_finalize_called[mask1]",
    "tests/generic/test_finalize.py::test_finalize_called[max]",
    "tests/generic/test_finalize.py::test_finalize_called[mean]",
    "tests/generic/test_finalize.py::test_finalize_called[median]",
    "tests/generic/test_finalize.py::test_finalize_called[melt', id_vars=['A'], value_vars=['B]",
    "tests/generic/test_finalize.py::test_finalize_called[min]",
    "tests/generic/test_finalize.py::test_finalize_called[mode0]",
    "tests/generic/test_finalize.py::test_finalize_called[mode1]",
    "tests/generic/test_finalize.py::test_finalize_called[nlargest', 1, 'A]",
    "tests/generic/test_finalize.py::test_finalize_called[notna0]",
    "tests/generic/test_finalize.py::test_finalize_called[notna1]",
    "tests/generic/test_finalize.py::test_finalize_called[notnull0]",
    "tests/generic/test_finalize.py::test_finalize_called[notnull1]",
    "tests/generic/test_finalize.py::test_finalize_called[nsmallest', 1, 'A]",
    "tests/generic/test_finalize.py::test_finalize_called[nunique]",
    "tests/generic/test_finalize.py::test_finalize_called[pct_change0]",
    "tests/generic/test_finalize.py::test_finalize_called[pct_change1]",
    "tests/generic/test_finalize.py::test_finalize_called[pivot', columns='A]",
    "tests/generic/test_finalize.py::test_finalize_called[pivot_table', columns='A', aggfunc=['mean', 'sum]",
    "tests/generic/test_finalize.py::test_finalize_called[pivot_table', columns='A]",
    "tests/generic/test_finalize.py::test_finalize_called[pop', 'A]",
    "tests/generic/test_finalize.py::test_finalize_called[prod]",
    "tests/generic/test_finalize.py::test_finalize_called[quantile0]",
    "tests/generic/test_finalize.py::test_finalize_called[quantile1]",
    "tests/generic/test_finalize.py::test_finalize_called[quantile2]",
    "tests/generic/test_finalize.py::test_finalize_called[quantile3]",
    "tests/generic/test_finalize.py::test_finalize_called[query', 'A == 1]",
    "tests/generic/test_finalize.py::test_finalize_called[rank0]",
    "tests/generic/test_finalize.py::test_finalize_called[rank1]",
    "tests/generic/test_finalize.py::test_finalize_called[reindex0]",
    "tests/generic/test_finalize.py::test_finalize_called[reindex1]",
    "tests/generic/test_finalize.py::test_finalize_called[reindex_like0]",
    "tests/generic/test_finalize.py::test_finalize_called[reindex_like1]",
    "tests/generic/test_finalize.py::test_finalize_called[rename', 'name]",
    "tests/generic/test_finalize.py::test_finalize_called[rename', columns={'A': 'a]",
    "tests/generic/test_finalize.py::test_finalize_called[rename0]",
    "tests/generic/test_finalize.py::test_finalize_called[rename1]",
    "tests/generic/test_finalize.py::test_finalize_called[rename_axis', columns='a]",
    "tests/generic/test_finalize.py::test_finalize_called[rename_axis', index='a]",
    "tests/generic/test_finalize.py::test_finalize_called[repeat]",
    "tests/generic/test_finalize.py::test_finalize_called[replace]",
    "tests/generic/test_finalize.py::test_finalize_called[reset_index0]",
    "tests/generic/test_finalize.py::test_finalize_called[reset_index1]",
    "tests/generic/test_finalize.py::test_finalize_called[reset_index2]",
    "tests/generic/test_finalize.py::test_finalize_called[round0]",
    "tests/generic/test_finalize.py::test_finalize_called[round1]",
    "tests/generic/test_finalize.py::test_finalize_called[sample0]",
    "tests/generic/test_finalize.py::test_finalize_called[sample1]",
    "tests/generic/test_finalize.py::test_finalize_called[select_dtypes', include='int]",
    "tests/generic/test_finalize.py::test_finalize_called[sem]",
    "tests/generic/test_finalize.py::test_finalize_called[set_axis', ['A]",
    "tests/generic/test_finalize.py::test_finalize_called[set_axis', ['a', 'b]",
    "tests/generic/test_finalize.py::test_finalize_called[set_index', 'A]",
    "tests/generic/test_finalize.py::test_finalize_called[shift]",
    "tests/generic/test_finalize.py::test_finalize_called[skew]",
    "tests/generic/test_finalize.py::test_finalize_called[sort_index]",
    "tests/generic/test_finalize.py::test_finalize_called[sort_values', by='A]",
    "tests/generic/test_finalize.py::test_finalize_called[sort_values]",
    "tests/generic/test_finalize.py::test_finalize_called[squeeze', axis='columns]",
    "tests/generic/test_finalize.py::test_finalize_called[squeeze]",
    "tests/generic/test_finalize.py::test_finalize_called[stack]",
    "tests/generic/test_finalize.py::test_finalize_called[std]",
    "tests/generic/test_finalize.py::test_finalize_called[sum]",
    "tests/generic/test_finalize.py::test_finalize_called[swaplevel]",
    "tests/generic/test_finalize.py::test_finalize_called[tail0]",
    "tests/generic/test_finalize.py::test_finalize_called[tail1]",
    "tests/generic/test_finalize.py::test_finalize_called[take0]",
    "tests/generic/test_finalize.py::test_finalize_called[take1]",
    "tests/generic/test_finalize.py::test_finalize_called[to_frame]",
    "tests/generic/test_finalize.py::test_finalize_called[to_period', freq='D]",
    "tests/generic/test_finalize.py::test_finalize_called[to_timestamp0]",
    "tests/generic/test_finalize.py::test_finalize_called[to_timestamp1]",
    "tests/generic/test_finalize.py::test_finalize_called[transform0]",
    "tests/generic/test_finalize.py::test_finalize_called[transform1]",
    "tests/generic/test_finalize.py::test_finalize_called[transpose]",
    "tests/generic/test_finalize.py::test_finalize_called[truncate0]",
    "tests/generic/test_finalize.py::test_finalize_called[truncate1]",
    "tests/generic/test_finalize.py::test_finalize_called[unstack]",
    "tests/generic/test_finalize.py::test_finalize_called[where0]",
    "tests/generic/test_finalize.py::test_finalize_called[where1]",
    "tests/generic/test_finalize.py::test_finalize_called[xs', 'a0]",
    "tests/generic/test_finalize.py::test_finalize_called[xs', 'a1]",
    "tests/generic/test_finalize.py::test_groupby_finalize[<lambda>0-obj0]",
    "tests/generic/test_finalize.py::test_groupby_finalize[<lambda>0-obj1]",
    "tests/generic/test_finalize.py::test_groupby_finalize[<lambda>1-obj0]",
    "tests/generic/test_finalize.py::test_groupby_finalize[<lambda>1-obj1]",
    "tests/generic/test_finalize.py::test_groupby_finalize[<lambda>2-obj0]",
    "tests/generic/test_finalize.py::test_groupby_finalize[<lambda>2-obj1]",
    "tests/generic/test_finalize.py::test_groupby_finalize[<lambda>3-obj0]",
    "tests/generic/test_finalize.py::test_groupby_finalize[<lambda>3-obj1]",
    "tests/generic/test_finalize.py::test_groupby_finalize[method0-obj0]",
    "tests/generic/test_finalize.py::test_groupby_finalize[method0-obj1]",
    "tests/generic/test_finalize.py::test_string_method[capitalize]",
    "tests/generic/test_finalize.py::test_string_method[casefold]",
    "tests/generic/test_finalize.py::test_string_method[cat', ['a]",
    "tests/generic/test_finalize.py::test_string_method[contains', 'a]",
    "tests/generic/test_finalize.py::test_string_method[count', 'a]",
    "tests/generic/test_finalize.py::test_string_method[encode', 'utf-8]",
    "tests/generic/test_finalize.py::test_string_method[endswith', 'a]",
    "tests/generic/test_finalize.py::test_string_method[find', 'a]",
    "tests/generic/test_finalize.py::test_string_method[findall', 'a]",
    "tests/generic/test_finalize.py::test_string_method[get]",
    "tests/generic/test_finalize.py::test_string_method[get_dummies]",
    "tests/generic/test_finalize.py::test_string_method[index', 'a]",
    "tests/generic/test_finalize.py::test_string_method[isalnum]",
    "tests/generic/test_finalize.py::test_string_method[isalpha]",
    "tests/generic/test_finalize.py::test_string_method[isdecimal]",
    "tests/generic/test_finalize.py::test_string_method[isdigit]",
    "tests/generic/test_finalize.py::test_string_method[islower]",
    "tests/generic/test_finalize.py::test_string_method[isnumeric]",
    "tests/generic/test_finalize.py::test_string_method[isspace]",
    "tests/generic/test_finalize.py::test_string_method[istitle]",
    "tests/generic/test_finalize.py::test_string_method[isupper]",
    "tests/generic/test_finalize.py::test_string_method[len]",
    "tests/generic/test_finalize.py::test_string_method[ljust]",
    "tests/generic/test_finalize.py::test_string_method[lower]",
    "tests/generic/test_finalize.py::test_string_method[lstrip]",
    "tests/generic/test_finalize.py::test_string_method[normalize', 'NFC]",
    "tests/generic/test_finalize.py::test_string_method[pad]",
    "tests/generic/test_finalize.py::test_string_method[partition', 'a]",
    "tests/generic/test_finalize.py::test_string_method[repeat]",
    "tests/generic/test_finalize.py::test_string_method[replace', 'a', 'b]",
    "tests/generic/test_finalize.py::test_string_method[rfind', 'a]",
    "tests/generic/test_finalize.py::test_string_method[rindex', 'a]",
    "tests/generic/test_finalize.py::test_string_method[rjust]",
    "tests/generic/test_finalize.py::test_string_method[rpartition', 'a]",
    "tests/generic/test_finalize.py::test_string_method[rstrip]",
    "tests/generic/test_finalize.py::test_string_method[slice]",
    "tests/generic/test_finalize.py::test_string_method[slice_replace', 1, repl='a]",
    "tests/generic/test_finalize.py::test_string_method[startswith', 'a]",
    "tests/generic/test_finalize.py::test_string_method[strip]",
    "tests/generic/test_finalize.py::test_string_method[swapcase]",
    "tests/generic/test_finalize.py::test_string_method[translate', {'a': 'b]",
    "tests/generic/test_finalize.py::test_string_method[upper]",
    "tests/generic/test_finalize.py::test_string_method[wrap]",
    "tests/generic/test_finalize.py::test_string_method[zfill]",
    "tests/generic/test_finalize.py::test_timedelta_methods[method0]",
    "tests/generic/test_finalize.py::test_timedelta_property[components]",
    "tests/generic/test_finalize.py::test_timedelta_property[days]",
    "tests/generic/test_finalize.py::test_timedelta_property[microseconds]",
    "tests/generic/test_finalize.py::test_timedelta_property[nanoseconds]",
    "tests/generic/test_finalize.py::test_timedelta_property[seconds]",
    "tests/generic/test_frame.py::TestDataFrame2::test_validate_bool_args[1]",
    "tests/generic/test_frame.py::TestDataFrame2::test_validate_bool_args[5.0]",
    "tests/generic/test_frame.py::TestDataFrame2::test_validate_bool_args[True]",
    "tests/generic/test_frame.py::TestDataFrame2::test_validate_bool_args[value2]",
    "tests/generic/test_frame.py::TestDataFrame::test_metadata_propagation_indiv",
    "tests/generic/test_frame.py::TestDataFrame::test_set_attribute",
    "tests/generic/test_generic.py::TestGeneric::test_data_deprecated[DataFrame]",
    "tests/generic/test_generic.py::TestGeneric::test_data_deprecated[Series]",
    "tests/generic/test_generic.py::TestNDFrame::test_flags_identity[DataFrame]",
    "tests/generic/test_generic.py::TestNDFrame::test_flags_identity[Series]",
    "tests/generic/test_generic.py::TestNDFrame::test_numpy_transpose[DataFrame]",
    "tests/generic/test_generic.py::TestNDFrame::test_transpose_frame",
    "tests/generic/test_series.py::TestSeries::test_metadata_propagation_indiv",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[bool-dtype]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[bool-object]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[categorical]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[complex128]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[complex64]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[datetime-tz]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[datetime]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[float32]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[float64]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[int16]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[int32]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[int64]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[int8]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[interval]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[nullable_bool]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[nullable_float]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[nullable_int]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[nullable_uint]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[object]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[period]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[range]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[repeats]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[string-pyarrow]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[string-python]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[string]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[timedelta]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[uint16]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[uint32]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[uint64]",
    "tests/generic/test_to_xarray.py::TestDataFrameToXArray::test_to_xarray_index_types[uint8]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_grouping_is_list_tuple",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_multiple_with_as_index_false_subset_to_a_single_column",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[count]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[first]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[idxmax]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[idxmin]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[last]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[max]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[min]",
    "tests/groupby/aggregate/test_aggregate.py::test_agg_str_with_kwarg_axis_1_raises[nunique]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-bool-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-bool-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-float32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-float32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-float64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-float64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-int32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-int32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-int64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-bool-int64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-bool-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-bool-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-float32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-float32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-float64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-float64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-int32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-int32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-int64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float32-int64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-bool-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-bool-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-float32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-float32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-float64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-float64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-int32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-int32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-int64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-float64-int64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-bool-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-bool-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-float32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-float32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-float64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-float64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-int32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-int32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-int64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int32-int64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-bool-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-bool-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-float32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-float32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-float64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-float64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-int32-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-int32-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-int64-keys0-agg_index0]",
    "tests/groupby/aggregate/test_aggregate.py::test_callable_result_dtype_frame[apply-int64-int64-keys1-agg_index1]",
    "tests/groupby/aggregate/test_aggregate.py::test_func_duplicates_raises",
    "tests/groupby/aggregate/test_aggregate.py::test_groupby_aggregate_directory[size]",
    "tests/groupby/aggregate/test_aggregate.py::test_groupby_aggregate_empty_key_empty_return",
    "tests/groupby/aggregate/test_aggregate.py::test_groupby_aggregation_multi_level_column",
    "tests/groupby/aggregate/test_aggregate.py::test_multiindex_custom_func[<lambda>0]",
    "tests/groupby/aggregate/test_aggregate.py::test_nonagg_agg",
    "tests/groupby/aggregate/test_aggregate.py::test_order_aggregate_multiple_funcs",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data1-prod-large_int-False]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data1-prod-large_int-True]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data1-sum-large_int-False]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data1-sum-large_int-True]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-mean-always_float-False]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-mean-always_float-True]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-median-always_float-False]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-median-always_float-True]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-var-always_float-False]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_EA_known_dtypes[data2-var-always_float-True]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[count]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[max]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[mean]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[median]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[min]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[prod]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[sem]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[std]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[sum]",
    "tests/groupby/aggregate/test_cython.py::test_cython_agg_nullable_int[var]",
    "tests/groupby/aggregate/test_cython.py::test_cythonized_aggers[prod]",
    "tests/groupby/aggregate/test_cython.py::test_cythonized_aggers[sum]",
    "tests/groupby/aggregate/test_other.py::test_agg_over_numpy_arrays",
    "tests/groupby/methods/test_describe.py::test_frame_describe_tupleindex",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_group_diff_bool",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_group_diff_real_frame[int16]",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_group_diff_real_frame[int8]",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_group_diff_real_series[int16]",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_group_diff_real_series[int8]",
    "tests/groupby/methods/test_groupby_shift_diff.py::test_shift_periods_freq",
    "tests/groupby/methods/test_nth.py::test_first_last_nth",
    "tests/groupby/methods/test_nth.py::test_first_last_nth_dtypes",
    "tests/groupby/methods/test_nth.py::test_negative_step",
    "tests/groupby/methods/test_nth.py::test_np_ints",
    "tests/groupby/methods/test_nth.py::test_nth",
    "tests/groupby/methods/test_nth.py::test_nth2",
    "tests/groupby/methods/test_nth.py::test_nth3",
    "tests/groupby/methods/test_nth.py::test_nth4",
    "tests/groupby/methods/test_nth.py::test_nth5",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[None-b]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[None-selection1]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[None-selection2]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[all-b]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[all-selection1]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[all-selection2]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[any-b]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[any-selection1]",
    "tests/groupby/methods/test_nth.py::test_nth_after_selection[any-selection2]",
    "tests/groupby/methods/test_nth.py::test_nth_column_order",
    "tests/groupby/methods/test_nth.py::test_nth_indexed",
    "tests/groupby/methods/test_nth.py::test_nth_multi_grouper",
    "tests/groupby/methods/test_nth.py::test_nth_multi_index_as_expected",
    "tests/groupby/methods/test_nth.py::test_nth_with_na_object[NoneType--1]",
    "tests/groupby/methods/test_nth.py::test_nth_with_na_object[float0--1]",
    "tests/groupby/methods/test_nth.py::test_nth_with_na_object[float1--1]",
    "tests/groupby/methods/test_nth.py::test_slice[arg0-expected_rows0]",
    "tests/groupby/methods/test_nth.py::test_slice[arg1-expected_rows1]",
    "tests/groupby/methods/test_nth.py::test_slice[arg2-expected_rows2]",
    "tests/groupby/methods/test_nth.py::test_slice[arg3-expected_rows3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_NA_float[Float32]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_allNA_column[Float32]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float32-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float32-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float32-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float64-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float64-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Float64-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int16-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int16-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int16-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int32-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int32-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int32-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int64-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int64-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int64-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int8-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int8-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[Int8-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt16-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt16-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt16-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt32-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt32-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt32-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt64-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt64-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt64-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt8-higher-2-3]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt8-lower-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked[UInt8-nearest-2-2]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Float32-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Float32-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Float64-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Float64-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int16-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int16-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int32-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int32-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int64-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int64-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int8-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[Int8-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt16-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt16-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt32-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt32-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt64-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt64-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt8-linear]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_all_na_group_masked_interp[UInt8-midpoint]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_nonmulti_levels_order",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_raises_on_invalid_dtype[True-q1]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_with_arraylike_q_and_int_columns[q0-groupby0-frame_size0]",
    "tests/groupby/methods/test_quantile.py::test_groupby_quantile_with_arraylike_q_and_int_columns[q0-groupby0-frame_size1]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals0-b_vals0-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals0-b_vals0-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals0-b_vals0-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals1-b_vals1-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals1-b_vals1-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals1-b_vals1-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals2-b_vals2-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals2-b_vals2-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0-a_vals2-b_vals2-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals0-b_vals0-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals0-b_vals0-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals0-b_vals0-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals1-b_vals1-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals1-b_vals1-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals1-b_vals1-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals2-b_vals2-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals2-b_vals2-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.25-a_vals2-b_vals2-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals0-b_vals0-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals0-b_vals0-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals0-b_vals0-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals1-b_vals1-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals1-b_vals1-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals2-b_vals2-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.5-a_vals2-b_vals2-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals0-b_vals0-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals0-b_vals0-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals0-b_vals0-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals1-b_vals1-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals1-b_vals1-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals1-b_vals1-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals2-b_vals2-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals2-b_vals2-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[0.75-a_vals2-b_vals2-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals0-b_vals0-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals0-b_vals0-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals0-b_vals0-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals1-b_vals1-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals1-b_vals1-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals1-b_vals1-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals2-b_vals2-higher]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals2-b_vals2-lower]",
    "tests/groupby/methods/test_quantile.py::test_quantile[1-a_vals2-b_vals2-nearest]",
    "tests/groupby/methods/test_quantile.py::test_quantile_array",
    "tests/groupby/methods/test_quantile.py::test_quantile_array2",
    "tests/groupby/methods/test_quantile.py::test_quantile_missing_group_values_correct_results[key4-val4-expected_key4-expected_val4]",
    "tests/groupby/methods/test_rank.py::test_groupby_axis0_cummax_axis1",
    "tests/groupby/methods/test_rank.py::test_rank_avg_even_vals[True-int32]",
    "tests/groupby/methods/test_rank.py::test_rank_avg_even_vals[True-int64]",
    "tests/groupby/methods/test_rank.py::test_rank_avg_even_vals[True-uint32]",
    "tests/groupby/methods/test_rank.py::test_rank_avg_even_vals[True-uint64]",
    "tests/groupby/methods/test_size.py::test_size_series_masked_type_returns_Int64[Float64]",
    "tests/groupby/methods/test_size.py::test_size_series_masked_type_returns_Int64[Int64]",
    "tests/groupby/methods/test_size.py::test_size_series_masked_type_returns_Int64[boolean]",
    "tests/groupby/methods/test_size.py::test_size_sort[False-A]",
    "tests/groupby/methods/test_size.py::test_size_sort[False-B]",
    "tests/groupby/methods/test_size.py::test_size_sort[False-by2]",
    "tests/groupby/methods/test_size.py::test_size_strings[string=string[pyarrow]]",
    "tests/groupby/methods/test_size.py::test_size_strings[string[pyarrow]]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-False-None-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-False-None-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-True-False-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-True-False-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-True-True-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-False-True-True-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-False-None-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-False-None-False-count-function]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-False-None-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-False-None-True-proportion-function]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-False-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-False-False-count-function]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-False-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-False-True-proportion-function]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-True-False-count-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-True-False-count-function]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-True-True-proportion-array]",
    "tests/groupby/methods/test_value_counts.py::test_against_frame_and_seriesgroupby[False-True-True-True-True-proportion-function]",
    "tests/groupby/methods/test_value_counts.py::test_ambiguous_grouping",
    "tests/groupby/methods/test_value_counts.py::test_column_label_duplicates[False-level-columns1-expected_names1]",
    "tests/groupby/methods/test_value_counts.py::test_compound[False-None-expected_rows0-expected_count0-expected_group_size0-False-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_compound[False-None-expected_rows0-expected_count0-expected_group_size0-True-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_compound[True-False-expected_rows1-expected_count1-expected_group_size1-False-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_compound[True-False-expected_rows1-expected_count1-expected_group_size1-True-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_compound[True-True-expected_rows2-expected_count2-expected_group_size2-False-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_compound[True-True-expected_rows2-expected_count2-expected_group_size2-True-string[pyarrow_numpy]]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[pyarrow]-False-None-expected_rows0-expected_count0-expected_group_size0-False]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[pyarrow]-False-None-expected_rows0-expected_count0-expected_group_size0-True]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[pyarrow]-True-False-expected_rows1-expected_count1-expected_group_size1-False]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[pyarrow]-True-False-expected_rows1-expected_count1-expected_group_size1-True]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[pyarrow]-True-True-expected_rows2-expected_count2-expected_group_size2-False]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[pyarrow]-True-True-expected_rows2-expected_count2-expected_group_size2-True]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[python]-False-None-expected_rows0-expected_count0-expected_group_size0-False]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[python]-False-None-expected_rows0-expected_count0-expected_group_size0-True]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[python]-True-False-expected_rows1-expected_count1-expected_group_size1-False]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[python]-True-False-expected_rows1-expected_count1-expected_group_size1-True]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[python]-True-True-expected_rows2-expected_count2-expected_group_size2-False]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=str[python]-True-True-expected_rows2-expected_count2-expected_group_size2-True]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=string[python]-False-None-expected_rows0-expected_count0-expected_group_size0-False]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=string[python]-False-None-expected_rows0-expected_count0-expected_group_size0-True]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=string[python]-True-False-expected_rows1-expected_count1-expected_group_size1-False]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=string[python]-True-False-expected_rows1-expected_count1-expected_group_size1-True]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=string[python]-True-True-expected_rows2-expected_count2-expected_group_size2-False]",
    "tests/groupby/methods/test_value_counts.py::test_compound[string=string[python]-True-True-expected_rows2-expected_count2-expected_group_size2-True]",
    "tests/groupby/methods/test_value_counts.py::test_data_frame_value_counts_dropna[NoneType-False-count-False-expected_data1-expected_index1]",
    "tests/groupby/methods/test_value_counts.py::test_data_frame_value_counts_dropna[NoneType-True-proportion-False-expected_data1-expected_index1]",
    "tests/groupby/methods/test_value_counts.py::test_dropna_combinations[True-False-expected_rows2-expected_values2]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort[False-False-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort[False-True-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort[False-True-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort[True-False-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort[True-True-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[False-False-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[False-False-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[False-True-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[False-True-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[True-False-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[True-False-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[True-True-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_sort_categorical[True-True-True]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_time_grouper[ms-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_time_grouper[ns-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_time_grouper[s-False]",
    "tests/groupby/methods/test_value_counts.py::test_value_counts_time_grouper[us-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[cummax-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[cummin-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[cumprod-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[cumsum-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[nunique-False]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[nunique-True]",
    "tests/groupby/test_all_methods.py::test_duplicate_columns[rank-False]",
    "tests/groupby/test_api.py::test_all_methods_categorized",
    "tests/groupby/test_api.py::test_tab_completion",
    "tests/groupby/test_apply.py::test_apply_datetime_issue[group_column_dtlike0]",
    "tests/groupby/test_apply.py::test_apply_datetime_issue[group_column_dtlike1]",
    "tests/groupby/test_apply.py::test_apply_frame_concat_series",
    "tests/groupby/test_apply.py::test_apply_func_that_appends_group_to_list_without_copy",
    "tests/groupby/test_apply.py::test_apply_index_has_complex_internals[index3]",
    "tests/groupby/test_apply.py::test_apply_index_key_error_bug[index_values0]",
    "tests/groupby/test_apply.py::test_apply_index_key_error_bug[index_values1]",
    "tests/groupby/test_apply.py::test_apply_multiindex_fail",
    "tests/groupby/test_apply.py::test_apply_nonmonotonic_float_index[arg0-idx0]",
    "tests/groupby/test_apply.py::test_apply_nonmonotonic_float_index[arg2-idx2]",
    "tests/groupby/test_apply.py::test_apply_with_date_in_multiindex_does_not_convert_to_timestamp",
    "tests/groupby/test_apply.py::test_builtins_apply[jim-max]",
    "tests/groupby/test_apply.py::test_builtins_apply[jim-min]",
    "tests/groupby/test_apply.py::test_builtins_apply[keys1-max]",
    "tests/groupby/test_apply.py::test_builtins_apply[keys1-min]",
    "tests/groupby/test_apply.py::test_empty_df[agg-prod]",
    "tests/groupby/test_apply.py::test_empty_df[agg-skew]",
    "tests/groupby/test_apply.py::test_empty_df[agg-sum]",
    "tests/groupby/test_apply.py::test_empty_df[apply-<lambda>0]",
    "tests/groupby/test_apply.py::test_empty_df[apply-<lambda>1]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group2",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH10519]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH12155]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH20084]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH21417]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH2656]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH2936]",
    "tests/groupby/test_apply.py::test_group_apply_once_per_group[GH7739 & GH10519]",
    "tests/groupby/test_apply.py::test_groupby_apply_none_first",
    "tests/groupby/test_apply.py::test_groupby_as_index_apply",
    "tests/groupby/test_apply.py::test_time_field_bug",
    "tests/groupby/test_apply_mutate.py::test_apply_function_with_indexing",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-count-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-first-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-idxmax-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-idxmin-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-last-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-max-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-mean-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-median-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-min-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-nunique-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-nunique-keys1-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-prod-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-size-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-std-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-sum-keys0-True]",
    "tests/groupby/test_categorical.py::test_agg_list[False-True-var-keys0-True]",
    "tests/groupby/test_categorical.py::test_apply[False]",
    "tests/groupby/test_categorical.py::test_apply[True]",
    "tests/groupby/test_categorical.py::test_as_index",
    "tests/groupby/test_categorical.py::test_category_order_apply[False-False-True-False-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[False-False-True-True-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[False-True-True-False-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[False-True-True-True-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-False-True-False-apply-multi]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-False-True-False-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-False-True-True-apply-multi]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-False-True-True-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-True-True-False-apply-multi]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-True-True-False-apply-range]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-True-True-True-apply-multi]",
    "tests/groupby/test_categorical.py::test_category_order_apply[True-True-True-True-apply-range]",
    "tests/groupby/test_categorical.py::test_dataframe_groupby_on_2_categoricals_when_observed_is_false[size-False]",
    "tests/groupby/test_categorical.py::test_dataframe_groupby_on_2_categoricals_when_observed_is_false[size-None]",
    "tests/groupby/test_categorical.py::test_describe_categorical_columns",
    "tests/groupby/test_categorical.py::test_groupby_categorical_observed_nunique",
    "tests/groupby/test_categorical.py::test_observed[False]",
    "tests/groupby/test_categorical.py::test_observed_codes_remap[False]",
    "tests/groupby/test_categorical.py::test_observed_codes_remap[True]",
    "tests/groupby/test_categorical.py::test_observed_groups[True]",
    "tests/groupby/test_categorical.py::test_observed_groups_with_nan[True]",
    "tests/groupby/test_categorical.py::test_sort_datetimelike[False-False]",
    "tests/groupby/test_categorical.py::test_sort_datetimelike[False-True]",
    "tests/groupby/test_categorical.py::test_sort_datetimelike[True-False]",
    "tests/groupby/test_categorical.py::test_sort_datetimelike[True-True]",
    "tests/groupby/test_counting.py::TestCounting::test_ngroup_distinct",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string=str[pyarrow]]",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string=str[python]]",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string=string[pyarrow]]",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string=string[python]]",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string[pyarrow]]",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string[pyarrow_numpy]]",
    "tests/groupby/test_counting.py::test_count_arrow_string_array[string[python]]",
    "tests/groupby/test_cumulative.py::test_cummin[Int64]",
    "tests/groupby/test_cumulative.py::test_cummin[np.int64]",
    "tests/groupby/test_cumulative.py::test_cummin_max_all_nan_column[Int64-cummax]",
    "tests/groupby/test_cumulative.py::test_cummin_max_all_nan_column[Int64-cummin]",
    "tests/groupby/test_cumulative.py::test_cummin_max_all_nan_column[UInt64-cummax]",
    "tests/groupby/test_cumulative.py::test_cummin_max_all_nan_column[UInt64-cummin]",
    "tests/groupby/test_cumulative.py::test_cummin_max_all_nan_column[boolean-cummax]",
    "tests/groupby/test_cumulative.py::test_cummin_max_all_nan_column[boolean-cummin]",
    "tests/groupby/test_cumulative.py::test_cython_api2",
    "tests/groupby/test_cumulative.py::test_groupby_cumprod_nan_influences_other_columns",
    "tests/groupby/test_cumulative.py::test_numpy_compat[cumprod]",
    "tests/groupby/test_cumulative.py::test_numpy_compat[cumsum]",
    "tests/groupby/test_groupby.py::test_as_index_select_column",
    "tests/groupby/test_groupby.py::test_basic_aggregations[float32]",
    "tests/groupby/test_groupby.py::test_basic_aggregations[int32]",
    "tests/groupby/test_groupby.py::test_decimal_na_sort[False]",
    "tests/groupby/test_groupby.py::test_decimal_na_sort[True]",
    "tests/groupby/test_groupby.py::test_frame_groupby_columns",
    "tests/groupby/test_groupby.py::test_group_name_available_in_inference_pass",
    "tests/groupby/test_groupby.py::test_group_on_two_row_multiindex_returns_one_tuple_key",
    "tests/groupby/test_groupby.py::test_groupby_all_nan_groups_drop",
    "tests/groupby/test_groupby.py::test_groupby_column_index_name_lost[shift]",
    "tests/groupby/test_groupby.py::test_groupby_column_index_name_lost[sum]",
    "tests/groupby/test_groupby.py::test_groupby_crash_on_nunique[axis='index']",
    "tests/groupby/test_groupby.py::test_groupby_crash_on_nunique[axis=0]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Float32-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Float64-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int16-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int16-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int32-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int32-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int64-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int8-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[Int8-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt16-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt16-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt32-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt32-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt64-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt64-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt8-False-val1]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_mask[UInt8-True-3]",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_skipna_false",
    "tests/groupby/test_groupby.py::test_groupby_cumsum_timedelta64",
    "tests/groupby/test_groupby.py::test_groupby_dtype_inference_empty",
    "tests/groupby/test_groupby.py::test_groupby_groups_in_BaseGrouper",
    "tests/groupby/test_groupby.py::test_groupby_level_mapper",
    "tests/groupby/test_groupby.py::test_groupby_level_nonmulti",
    "tests/groupby/test_groupby.py::test_groupby_multiindex_not_lexsorted",
    "tests/groupby/test_groupby.py::test_groupby_nat_exclude",
    "tests/groupby/test_groupby.py::test_groupby_nonobject_dtype",
    "tests/groupby/test_groupby.py::test_groupby_nonstring_columns",
    "tests/groupby/test_groupby.py::test_groupby_numerical_stability_cumsum",
    "tests/groupby/test_groupby.py::test_groupby_numerical_stability_sum_mean[mean-values1]",
    "tests/groupby/test_groupby.py::test_groupby_numerical_stability_sum_mean[sum-values0]",
    "tests/groupby/test_groupby.py::test_groupby_one_row",
    "tests/groupby/test_groupby.py::test_groupby_overflow[222-uint]",
    "tests/groupby/test_groupby.py::test_groupby_series_with_name",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int16-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int16-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int32-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int32-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int8-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[Int8-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt16-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt16-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt32-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt32-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt64-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt64-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt8-prod-2]",
    "tests/groupby/test_groupby.py::test_groupby_sum_support_mask[UInt8-sum-3]",
    "tests/groupby/test_groupby.py::test_groupby_with_Time_Grouper[ms]",
    "tests/groupby/test_groupby.py::test_groupby_with_Time_Grouper[ns]",
    "tests/groupby/test_groupby.py::test_groupby_with_Time_Grouper[s]",
    "tests/groupby/test_groupby.py::test_groupby_with_Time_Grouper[us]",
    "tests/groupby/test_groupby.py::test_groupby_with_hier_columns",
    "tests/groupby/test_groupby.py::test_groups_repr_truncates[1-{0: [0], ...}]",
    "tests/groupby/test_groupby.py::test_groups_repr_truncates[4-{0: [0], 1: [1], 2: [2], 3: [3], ...}]",
    "tests/groupby/test_groupby.py::test_groups_repr_truncates[5-{0: [0], 1: [1], 2: [2], 3: [3], 4: [4]}]",
    "tests/groupby/test_groupby.py::test_handle_dict_return_value",
    "tests/groupby/test_groupby.py::test_indices_concatenation_order",
    "tests/groupby/test_groupby.py::test_len_nan_group",
    "tests/groupby/test_groupby.py::test_ops_not_as_index[size]",
    "tests/groupby/test_groupby.py::test_single_element_list_grouping[a]",
    "tests/groupby/test_groupby.py::test_single_element_list_grouping[selection2]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[DataFrame-as_index-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[DataFrame-dropna-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[DataFrame-group_keys-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[DataFrame-observed-True]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[DataFrame-sort-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[Series-as_index-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[Series-dropna-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[Series-group_keys-False]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[Series-observed-True]",
    "tests/groupby/test_groupby.py::test_subsetting_columns_keeps_attrs[Series-sort-False]",
    "tests/groupby/test_groupby.py::test_wrap_aggregated_output_multindex",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[count-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[count-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[count-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[count-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[first-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[first-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[first-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[first-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-False-True-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmax-True-True-True-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-False-True-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[idxmin-True-True-True-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[last-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[last-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[last-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[last-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[max-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[max-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[max-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[max-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[mean-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[mean-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[mean-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[mean-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[median-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[median-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[median-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[median-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[min-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[min-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[min-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[min-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[nunique-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[nunique-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[nunique-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[nunique-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[prod-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[prod-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[prod-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[prod-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[quantile-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[quantile-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[quantile-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[quantile-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[std-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[std-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[std-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[std-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[sum-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[sum-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[sum-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[sum-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[var-True-False-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[var-True-False-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[var-True-True-False-multi]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_reducers[var-True-True-False-single]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumcount-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumcount-True-False-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumcount-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumcount-True-True-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cummax-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cummax-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cummin-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cummin-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumprod-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumprod-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumsum-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[cumsum-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[ngroup-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[ngroup-True-False-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[ngroup-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[ngroup-True-True-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[pct_change-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[pct_change-True-False-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[pct_change-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[pct_change-True-True-True]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[rank-True-False-False]",
    "tests/groupby/test_groupby_dropna.py::test_categorical_transformers[rank-True-True-False]",
    "tests/groupby/test_groupby_dropna.py::test_groupby_apply_with_dropna_for_multi_index[dropna_false_no_nan]",
    "tests/groupby/test_groupby_dropna.py::test_groupby_apply_with_dropna_for_multi_index[dropna_true_no_nan]",
    "tests/groupby/test_groupby_dropna.py::test_groupby_dropna_multi_index_dataframe_agg[False-tuples1-outputs1]",
    "tests/groupby/test_groupby_dropna.py::test_groupby_dropna_multi_index_dataframe_nan_in_one_group[NoneType-False-tuples1-outputs1]",
    "tests/groupby/test_groupby_dropna.py::test_groupby_dropna_multi_index_dataframe_nan_in_two_groups[NoneType-NoneType-False-tuples1-outputs1]",
    "tests/groupby/test_groupby_dropna.py::test_grouper_dropna_propagation[False]",
    "tests/groupby/test_groupby_dropna.py::test_grouper_dropna_propagation[True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-NoneType-float0-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-NoneType-float0-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-NoneType-float1-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-NoneType-float1-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-float0-NoneType-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-float0-NoneType-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-float1-NoneType-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[False-float1-NoneType-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-NoneType-float0-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-NoneType-float0-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-NoneType-float1-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-NoneType-float1-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-float0-NoneType-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-float0-NoneType-None-True]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-float1-NoneType-None-False]",
    "tests/groupby/test_groupby_dropna.py::test_null_is_null_for_dtype[True-float1-NoneType-None-True]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_metadata",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[all-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[any-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[bfill-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[corrwith-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[corrwith-obj1]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[count-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[cumcount-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[cummax-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[cummin-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[cumprod-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[cumsum-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[diff-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[ffill-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[fillna-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[first-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[idxmax-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[idxmin-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[last-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[max-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[mean-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[median-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[min-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[ngroup-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[nunique-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[pct_change-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[prod-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[quantile-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[rank-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[sem-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[shift-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[size-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[skew-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[std-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[sum-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_preserves_subclass[var-obj0]",
    "tests/groupby/test_groupby_subclass.py::test_groupby_resample_preserves_subclass[DataFrame]",
    "tests/groupby/test_grouping.py::TestGetGroup::test_get_group_grouped_by_tuple",
    "tests/groupby/test_grouping.py::TestGetGroup::test_get_group_grouped_by_tuple_with_lambda",
    "tests/groupby/test_grouping.py::TestGetGroup::test_groupby_with_single_column",
    "tests/groupby/test_grouping.py::TestGrouping::test_evaluate_with_empty_groups[agg-expected1]",
    "tests/groupby/test_grouping.py::TestGrouping::test_evaluate_with_empty_groups[apply-expected2]",
    "tests/groupby/test_grouping.py::TestGrouping::test_groupby_empty",
    "tests/groupby/test_grouping.py::TestGrouping::test_groupby_level[False]",
    "tests/groupby/test_grouping.py::TestGrouping::test_groupby_level[True]",
    "tests/groupby/test_grouping.py::TestGrouping::test_groupby_level_index_value_all_na",
    "tests/groupby/test_grouping.py::TestGrouping::test_groupby_multiindex_partial_indexing_equivalence",
    "tests/groupby/test_grouping.py::TestGrouping::test_grouper_getting_correct_binner",
    "tests/groupby/test_grouping.py::TestGrouping::test_grouping_labels",
    "tests/groupby/test_grouping.py::TestGrouping::test_level_preserve_order[False-labels1]",
    "tests/groupby/test_grouping.py::TestGrouping::test_level_preserve_order[True-labels0]",
    "tests/groupby/test_grouping.py::TestGrouping::test_multiindex_columns_empty_level",
    "tests/groupby/test_grouping.py::TestGrouping::test_multiindex_negative_level",
    "tests/groupby/test_grouping.py::TestIteration::test_grouping_is_iterable",
    "tests/groupby/test_grouping.py::TestIteration::test_grouping_string_repr",
    "tests/groupby/test_grouping.py::TestSelection::test_indices_grouped_by_tuple_with_lambda",
    "tests/groupby/test_grouping.py::test_depr_grouping_attrs[group_arraylike]",
    "tests/groupby/test_grouping.py::test_depr_grouping_attrs[group_index]",
    "tests/groupby/test_grouping.py::test_depr_grouping_attrs[result_index]",
    "tests/groupby/test_grouping.py::test_grouper_groups",
    "tests/groupby/test_grouping.py::test_grouping_by_key_is_in_axis",
    "tests/groupby/test_indexing.py::test_multiindex",
    "tests/groupby/test_missing.py::test_ffill_handles_nan_groups[False-False-bfill]",
    "tests/groupby/test_missing.py::test_ffill_handles_nan_groups[False-False-ffill]",
    "tests/groupby/test_missing.py::test_ffill_handles_nan_groups[False-True-bfill]",
    "tests/groupby/test_missing.py::test_ffill_handles_nan_groups[False-True-ffill]",
    "tests/groupby/test_missing.py::test_groupby_column_index_name_lost_fill_funcs[bfill]",
    "tests/groupby/test_missing.py::test_groupby_column_index_name_lost_fill_funcs[ffill]",
    "tests/groupby/test_missing.py::test_indices_with_missing",
    "tests/groupby/test_raises.py::test_groupby_raises_category[<lambda>-False-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[<lambda>-True-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[a-False-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[a-True-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by1-False-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by1-True-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by2-False-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by2-True-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by6-False-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by6-True-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by7-False-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by7-True-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by8-False-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by8-True-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by9-False-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_category[by9-True-fillna-method]",
    "tests/groupby/test_raises.py::test_groupby_raises_timedelta[var]",
    "tests/groupby/test_reductions.py::test_empty_categorical[True]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float32-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float32-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float32-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float32-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float64-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float64-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float64-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Float64-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int16-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int16-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int16-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int16-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int32-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int32-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int32-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int32-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int64-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int64-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int64-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int64-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int8-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int8-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int8-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[Int8-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt16-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt16-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt16-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt16-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt32-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt32-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt32-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt32-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt64-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt64-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt64-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt64-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt8-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt8-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt8-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[UInt8-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[double[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[double[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[double[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[double[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float32-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float32-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float32-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float32-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float64-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float64-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float64-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float64-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[float[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int16[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int16[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int16[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int16[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int32[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int32[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int32[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int32[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int64[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int64[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int64[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int64[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int8[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int8[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int8[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[int8[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint16[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint16[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint16[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint16[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint32[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint32[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint32[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint32[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint64[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint64[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint64[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint64[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint8[pyarrow]-False-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint8[pyarrow]-False-False-last]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint8[pyarrow]-True-False-first]",
    "tests/groupby/test_reductions.py::test_first_last_skipna[uint8[pyarrow]-True-False-last]",
    "tests/groupby/test_reductions.py::test_groupby_mean_no_overflow",
    "tests/groupby/test_reductions.py::test_groupby_sum_mincount_boolean[0]",
    "tests/groupby/test_reductions.py::test_idxmin_idxmax_axis1",
    "tests/groupby/test_reductions.py::test_idxmin_idxmax_extremes_skipna[False-float-idxmax]",
    "tests/groupby/test_reductions.py::test_idxmin_idxmax_extremes_skipna[False-float-idxmin]",
    "tests/groupby/test_reductions.py::test_idxmin_idxmax_extremes_skipna[False-float32-idxmax]",
    "tests/groupby/test_reductions.py::test_idxmin_idxmax_extremes_skipna[False-float32-idxmin]",
    "tests/groupby/test_reductions.py::test_idxmin_idxmax_extremes_skipna[False-float64-idxmax]",
    "tests/groupby/test_reductions.py::test_idxmin_idxmax_extremes_skipna[False-float64-idxmin]",
    "tests/groupby/test_reductions.py::test_masked_kleene_logic[False-data2-all]",
    "tests/groupby/test_reductions.py::test_masked_kleene_logic[False-data2-any]",
    "tests/groupby/test_reductions.py::test_masked_kleene_logic[False-data3-any]",
    "tests/groupby/test_reductions.py::test_masked_kleene_logic[False-data4-all]",
    "tests/groupby/test_reductions.py::test_nunique",
    "tests/groupby/test_reductions.py::test_nunique_with_NaT[key0-data0-True-expected0]",
    "tests/groupby/test_reductions.py::test_nunique_with_NaT[key1-data1-True-expected1]",
    "tests/groupby/test_reductions.py::test_nunique_with_NaT[key2-data2-False-expected2]",
    "tests/groupby/test_reductions.py::test_nunique_with_NaT[key3-data3-False-expected3]",
    "tests/groupby/test_reductions.py::test_nunique_with_empty_series",
    "tests/groupby/test_reductions.py::test_nunique_with_object",
    "tests/groupby/test_reductions.py::test_nunique_with_timegrouper",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-all-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-all-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-any-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-any-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-count-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-count-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-count-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-count-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-first-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-first-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmax-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmax-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmax-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmax-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmin-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmin-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmin-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-idxmin-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-last-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-last-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-max-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-max-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-min-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-min-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-nunique-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-nunique-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-nunique-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-nunique-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-sum-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[pyarrow]-sum-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-all-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-all-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-any-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-any-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-count-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-count-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-count-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-count-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-first-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-first-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmax-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmax-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmax-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmax-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmin-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmin-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmin-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-idxmin-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-last-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-last-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-max-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-max-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-min-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-min-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-nunique-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-nunique-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-nunique-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-nunique-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-sum-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=str[python]-sum-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-all-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-all-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-any-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-any-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-count-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-count-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-count-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-count-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-first-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-first-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmax-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmax-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmax-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmax-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmin-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmin-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmin-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-idxmin-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-last-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-last-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-max-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-max-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-min-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-min-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-nunique-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-nunique-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-nunique-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-nunique-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-size-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-size-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-sum-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[pyarrow]-sum-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-all-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-all-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-any-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-any-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-count-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-count-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-count-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-count-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-first-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-first-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmax-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmax-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmax-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmax-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmin-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmin-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmin-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-idxmin-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-last-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-last-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-max-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-max-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-min-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-min-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-nunique-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-nunique-False-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-nunique-True-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-nunique-True-1]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-sum-False-0]",
    "tests/groupby/test_reductions.py::test_string_dtype_all_na[string=string[python]-sum-False-1]",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_apply_timegrouper_with_nat_apply_squeeze",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_apply_timegrouper_with_nat_dict_returns",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_apply_timegrouper_with_nat_scalar_returns",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_datetime64_32_bit",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_with_timegrouper",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_with_timegrouper_methods[False]",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_groupby_with_timegrouper_methods[True]",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_scalar_call_versus_list_call",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_timegrouper_apply_return_type_series",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_timegrouper_with_reg_groups",
    "tests/groupby/test_timegrouper.py::TestGroupBy::test_timezone_info",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumcount-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumcount-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cummax-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cummax-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cummin-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cummin-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumprod-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumprod-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumsum-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[cumsum-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[nunique-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[nunique-keys1]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[size-A]",
    "tests/groupby/transform/test_transform.py::test_as_index_no_change[size-keys1]",
    "tests/groupby/transform/test_transform.py::test_cython_transform_series[cumprod-args0-<lambda>]",
    "tests/groupby/transform/test_transform.py::test_cython_transform_series[cumsum-args1-<lambda>]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[False-idxmax]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[False-idxmin]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[False-nunique]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[False-size]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[True-idxmax]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[True-idxmin]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_reducer[True-size]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_transformer[False-cumcount]",
    "tests/groupby/transform/test_transform.py::test_null_group_str_transformer[True-cumcount]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-None--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-None-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-bfill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-bfill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-ffill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-1-ffill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-None--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-None-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-bfill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-bfill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-ffill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[DataFrame-None-ffill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-None--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-None-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-bfill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-bfill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-ffill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-1-ffill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-None--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-None-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-bfill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-bfill-1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-ffill--1-None]",
    "tests/groupby/transform/test_transform.py::test_pct_change[Series-None-ffill-1-None]",
    "tests/groupby/transform/test_transform.py::test_transform_cumcount",
    "tests/groupby/transform/test_transform.py::test_transform_datetime_to_timedelta",
    "tests/groupby/transform/test_transform.py::test_transform_fast",
    "tests/groupby/transform/test_transform.py::test_transform_fastpath_raises",
    "tests/groupby/transform/test_transform.py::test_transform_mixed_type",
    "tests/groupby/transform/test_transform.py::test_transform_numeric_ret[count-a-expected0]",
    "tests/groupby/transform/test_transform.py::test_transform_numeric_ret[count-cols1-expected1]",
    "tests/groupby/transform/test_transform.py::test_transform_numeric_ret[size-a-expected0]",
    "tests/groupby/transform/test_transform.py::test_transform_numeric_ret[size-cols1-expected1]",
    "tests/groupby/transform/test_transform.py::test_transform_transformation_func[cumcount]",
    "tests/indexes/base_class/test_constructors.py::TestIndexConstructor::test_index_string_inference",
    "tests/indexes/categorical/test_astype.py::TestAstype::test_astype_category[False-True-None]",
    "tests/indexes/categorical/test_astype.py::TestAstype::test_astype_category[False-True-foo]",
    "tests/indexes/categorical/test_astype.py::TestAstype::test_astype_category[True-False-None]",
    "tests/indexes/categorical/test_astype.py::TestAstype::test_astype_category[True-False-foo]",
    "tests/indexes/categorical/test_astype.py::TestAstype::test_categorical_date_roundtrip[False]",
    "tests/indexes/categorical/test_astype.py::TestAstype::test_categorical_date_roundtrip[True]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex2::test_engine_type[int16-Int16Engine]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex2::test_engine_type[int32-Int32Engine]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex2::test_engine_type[int64-Int64Engine]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex2::test_engine_type[int8-Int8Engine]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex2::test_remove_maintains_order",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex::test_ensure_copied_data",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex::test_is_monotonic[data0-non_lexsorted_data0]",
    "tests/indexes/categorical/test_category.py::TestCategoricalIndex::test_is_monotonic[data1-non_lexsorted_data1]",
    "tests/indexes/categorical/test_constructors.py::TestCategoricalIndexConstructors::test_construction",
    "tests/indexes/categorical/test_constructors.py::TestCategoricalIndexConstructors::test_construction_disallows_scalar",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_categorical",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_categorical_unordered",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string=str[pyarrow]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string=str[python]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string=string[pyarrow]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string=string[python]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string[pyarrow]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string[pyarrow_numpy]]",
    "tests/indexes/categorical/test_equals.py::TestEquals::test_equals_string_dtype[string[python]]",
    "tests/indexes/categorical/test_fillna.py::TestFillNA::test_fillna_copies_with_no_nas",
    "tests/indexes/categorical/test_indexing.py::TestContains::test_contains_na_dtype[False]",
    "tests/indexes/categorical/test_indexing.py::TestContains::test_contains_nan",
    "tests/indexes/categorical/test_indexing.py::TestWhere::test_where_non_categories",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values[DatetimeIndex]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values[TimedeltaIndex]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_with_freq_periodindex2[idx1]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_without_freq_datetimeindex[None-index_dates0-expected_dates0]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_without_freq_datetimeindex[None-index_dates1-expected_dates1]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_without_freq_datetimeindex[None-index_dates2-expected_dates2]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_without_freq_periodindex[idx4-expected4]",
    "tests/indexes/datetimelike_/test_sort_values.py::TestSortValues::test_sort_values_without_freq_timedeltaindex",
    "tests/indexes/datetimelike_/test_sort_values.py::test_order_stability_compat",
    "tests/indexes/datetimes/methods/test_astype.py::TestDatetimeIndex::test_astype",
    "tests/indexes/datetimes/methods/test_astype.py::TestDatetimeIndex::test_astype_datetime64",
    "tests/indexes/datetimes/methods/test_astype.py::TestDatetimeIndex::test_astype_raises[float]",
    "tests/indexes/datetimes/methods/test_astype.py::TestDatetimeIndex::test_astype_str_nat",
    "tests/indexes/datetimes/methods/test_astype.py::TestDatetimeIndex::test_astype_uint",
    "tests/indexes/datetimes/methods/test_astype.py::TestDatetimeIndex::test_integer_index_astype_datetime[US/Pacific-datetime64[ns, US/Pacific]]",
    "tests/indexes/datetimes/methods/test_factorize.py::TestDatetimeIndexFactorize::test_factorize_tz[zoneinfo.ZoneInfo(key='US/Pacific')-index]",
    "tests/indexes/datetimes/methods/test_factorize.py::TestDatetimeIndexFactorize::test_factorize_tz[zoneinfo.ZoneInfo(key='US/Pacific')-series]",
    "tests/indexes/datetimes/methods/test_isocalendar.py::test_dti_timestamp_isocalendar_fields",
    "tests/indexes/datetimes/methods/test_map.py::TestMap::test_map_bug_1677",
    "tests/indexes/datetimes/methods/test_round.py::TestDatetimeIndexRound::test_round_daily",
    "tests/indexes/datetimes/methods/test_tz_localize.py::TestTZLocalize::test_tz_localize_invalidates_freq",
    "tests/indexes/datetimes/methods/test_unique.py::test_index_unique2",
    "tests/indexes/datetimes/test_arithmetic.py::TestDatetimeIndexArithmetic::test_sub_datetime_preserves_freq[zoneinfo.ZoneInfo(key='US/Pacific')]",
    "tests/indexes/datetimes/test_arithmetic.py::TestDatetimeIndexArithmetic::test_sub_datetime_preserves_freq[zoneinfo.ZoneInfo(key='UTC')]",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_construction_outofbounds",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_constructor_coverage",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_constructor_datetime64_tzformat[W-SUN]",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_constructor_dtype",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_constructor_dtype_tz_mismatch_raises",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_constructor_with_int_tz[US/Pacific-datetime64[ns, US/Pacific]-array-DatetimeIndex]",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_constructor_with_int_tz[US/Pacific-datetime64[ns, US/Pacific]-array-Index]",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_constructor_with_int_tz[US/Pacific-datetime64[ns, US/Pacific]-list-DatetimeIndex]",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_dti_constructor_with_dtype_object_int_matches_int_dtype[M8[us, US/Pacific]]",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_dti_date_out_of_range[data1]",
    "tests/indexes/datetimes/test_constructors.py::TestDatetimeIndex::test_integer_values_and_tz_interpreted_as_utc",
    "tests/indexes/datetimes/test_constructors.py::TestTimeSeries::test_constructor_int64_nocopy",
    "tests/indexes/datetimes/test_date_range.py::TestDateRangeNonTickFreq::test_date_range_freqstr_matches_offset[W-SUN-offset2]",
    "tests/indexes/datetimes/test_date_range.py::TestDateRangeTZ::test_date_range_with_fixed_tz",
    "tests/indexes/datetimes/test_date_range.py::TestDateRangeTZ::test_date_range_with_fixedoffset_noname",
    "tests/indexes/datetimes/test_date_range.py::TestDateRanges::test_date_range_unsigned_overflow_handling",
    "tests/indexes/datetimes/test_date_range.py::TestDateRanges::test_range_tz_dateutil",
    "tests/indexes/datetimes/test_date_range.py::TestDateRanges::test_range_tz_pytz",
    "tests/indexes/datetimes/test_date_range.py::TestGenRangeGeneration::test_precision_finer_than_offset",
    "tests/indexes/datetimes/test_datetime.py::TestDatetimeIndex::test_misc_coverage",
    "tests/indexes/datetimes/test_formats.py::TestFormat::test_format",
    "tests/indexes/datetimes/test_indexing.py::TestGetItem::test_getitem[None]",
    "tests/indexes/datetimes/test_indexing.py::TestTake::test_take2[None]",
    "tests/indexes/datetimes/test_indexing.py::TestTake::test_take[None]",
    "tests/indexes/datetimes/test_indexing.py::TestWhere::test_where_invalid_dtypes",
    "tests/indexes/datetimes/test_indexing.py::TestWhere::test_where_mismatched_nat[zoneinfo.ZoneInfo(key='US/Pacific')]",
    "tests/indexes/datetimes/test_indexing.py::TestWhere::test_where_mismatched_nat[zoneinfo.ZoneInfo(key='UTC')]",
    "tests/indexes/datetimes/test_join.py::TestJoin::test_join_preserves_freq[None]",
    "tests/indexes/datetimes/test_join.py::TestJoin::test_join_self[inner]",
    "tests/indexes/datetimes/test_join.py::TestJoin::test_join_self[left]",
    "tests/indexes/datetimes/test_join.py::TestJoin::test_join_self[outer]",
    "tests/indexes/datetimes/test_join.py::TestJoin::test_join_self[right]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[2018-12-02 21:52:00+00:00-2018-12-02 21:50:00+00:00]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[2018-12-02 21:52:00+00:00-start1]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[2018-12-02 21:52:00+00:00-start2]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end1-2018-12-02 21:50:00+00:00]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end1-start1]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end1-start2]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end2-2018-12-02 21:50:00+00:00]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end2-start1]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_getitem_with_datestring_with_UTC_offset[end2-start2]",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_partial_slice_daily",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_partial_slice_minutely",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_partial_slice_requires_monotonicity",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_partial_slicing_dataframe",
    "tests/indexes/datetimes/test_partial_slicing.py::TestSlicing::test_return_type_doesnt_depend_on_monotonicity",
    "tests/indexes/datetimes/test_scalar_compat.py::TestDatetimeIndexOps::test_day_name_month_name[None]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_difference_freq[False]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_difference_freq[None]",
    "tests/indexes/datetimes/test_setops.py::TestDatetimeIndexSetOps::test_union_freq_infer",
    "tests/indexes/datetimes/test_timezones.py::TestDatetimeIndexTimezones::test_date_range_localize[ms]",
    "tests/indexes/datetimes/test_timezones.py::TestDatetimeIndexTimezones::test_date_range_localize[ns]",
    "tests/indexes/datetimes/test_timezones.py::TestDatetimeIndexTimezones::test_date_range_localize[s]",
    "tests/indexes/datetimes/test_timezones.py::TestDatetimeIndexTimezones::test_date_range_localize[us]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_float[index0]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_float[index1]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_float[index3]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_float[index4]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index0-int64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index0-uint64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index1-int64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index1-uint64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index3-int64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index3-uint64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index4-int64]",
    "tests/indexes/interval/test_astype.py::TestDatetimelikeSubtype::test_subtype_integer[index4-uint64]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_datetimelike[index0-datetime64[ns]]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_datetimelike[index0-timedelta64[ns]]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_datetimelike[index1-datetime64[ns]]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_datetimelike[index1-timedelta64[ns]]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_integer[int64]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_integer[uint64]",
    "tests/indexes/interval/test_astype.py::TestFloatSubtype::test_subtype_integer_errors",
    "tests/indexes/interval/test_constructors.py::TestClassConstructors::test_constructor_pass_closed[breaks0]",
    "tests/indexes/interval/test_constructors.py::TestClassConstructors::test_constructor_pass_closed[breaks2]",
    "tests/indexes/interval/test_constructors.py::TestClassConstructors::test_constructor_pass_closed[breaks3]",
    "tests/indexes/interval/test_constructors.py::TestClassConstructors::test_constructor_pass_closed[breaks4]",
    "tests/indexes/interval/test_constructors.py::TestFromArrays::test_constructor_pass_closed[breaks0]",
    "tests/indexes/interval/test_constructors.py::TestFromArrays::test_constructor_pass_closed[breaks1]",
    "tests/indexes/interval/test_constructors.py::TestFromArrays::test_constructor_pass_closed[breaks2]",
    "tests/indexes/interval/test_constructors.py::TestFromArrays::test_constructor_pass_closed[breaks3]",
    "tests/indexes/interval/test_constructors.py::TestFromArrays::test_constructor_pass_closed[breaks4]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_constructor_pass_closed[breaks0]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_constructor_pass_closed[breaks1]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_constructor_pass_closed[breaks2]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_constructor_pass_closed[breaks3]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_constructor_pass_closed[breaks4]",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_generic_errors",
    "tests/indexes/interval/test_constructors.py::TestFromBreaks::test_left_right_dont_share_data",
    "tests/indexes/interval/test_constructors.py::TestFromTuples::test_constructor_categorical_valid[CategoricalIndex]",
    "tests/indexes/interval/test_constructors.py::TestFromTuples::test_constructor_pass_closed[breaks0]",
    "tests/indexes/interval/test_constructors.py::TestFromTuples::test_constructor_pass_closed[breaks2]",
    "tests/indexes/interval/test_constructors.py::TestFromTuples::test_constructor_pass_closed[breaks3]",
    "tests/indexes/interval/test_constructors.py::TestFromTuples::test_constructor_pass_closed[breaks4]",
    "tests/indexes/interval/test_formats.py::TestIntervalIndexRendering::test_timestamp_with_timezone[ms]",
    "tests/indexes/interval/test_formats.py::TestIntervalIndexRendering::test_timestamp_with_timezone[ns]",
    "tests/indexes/interval/test_formats.py::TestIntervalIndexRendering::test_timestamp_with_timezone[s]",
    "tests/indexes/interval/test_formats.py::TestIntervalIndexRendering::test_timestamp_with_timezone[us]",
    "tests/indexes/interval/test_indexing.py::TestGetIndexer::test_get_indexer_categorical_with_nans",
    "tests/indexes/interval/test_indexing.py::TestGetIndexer::test_get_indexer_datetime",
    "tests/indexes/interval/test_indexing.py::TestGetLoc::test_get_loc_datetimelike_overlapping[datetime64[ns]]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_comparison",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_ensure_copied_data[both]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_ensure_copied_data[left]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_ensure_copied_data[neither]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_ensure_copied_data[right]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_insert[data0]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[both-breaks2]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[both-breaks4]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[left-breaks2]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[left-breaks4]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[neither-breaks2]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[neither-breaks4]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[right-breaks2]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_length[right-breaks4]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float32-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float32-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float64-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[float64-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int16-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int16-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int32-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int32-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int64-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int64-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int8-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[int8-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint16-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint16-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint32-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint32-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint64-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint64-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint8-IntervalIndex]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_maybe_convert_i8_numeric_identical[uint8-Interval]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_monotonic[both]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_monotonic[left]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_monotonic[neither]",
    "tests/indexes/interval/test_interval.py::TestIntervalIndex::test_monotonic[right]",
    "tests/indexes/interval/test_interval.py::test_dir",
    "tests/indexes/interval/test_interval_range.py::TestIntervalRange::test_interval_range_fractional_period",
    "tests/indexes/multi/test_analytics.py::test_map_dictlike[<lambda>1]",
    "tests/indexes/multi/test_astype.py::test_astype",
    "tests/indexes/multi/test_compat.py::test_inplace_mutation_resets_values",
    "tests/indexes/multi/test_constructors.py::test_create_index_existing_name",
    "tests/indexes/multi/test_constructors.py::test_from_arrays_respects_none_names",
    "tests/indexes/multi/test_constructors.py::test_from_frame_dtype_fidelity",
    "tests/indexes/multi/test_constructors.py::test_from_frame_error[non_frame0]",
    "tests/indexes/multi/test_constructors.py::test_from_frame_error[non_frame3]",
    "tests/indexes/multi/test_constructors.py::test_from_frame_missing_values_multiIndex",
    "tests/indexes/multi/test_constructors.py::test_multiindex_inference_consistency",
    "tests/indexes/multi/test_conversion.py::test_to_frame_dtype_fidelity",
    "tests/indexes/multi/test_copy.py::test_copy",
    "tests/indexes/multi/test_copy.py::test_copy_deep_false_retains_id",
    "tests/indexes/multi/test_copy.py::test_shallow_copy",
    "tests/indexes/multi/test_copy.py::test_view",
    "tests/indexes/multi/test_drop.py::test_drop_not_lexsorted",
    "tests/indexes/multi/test_drop.py::test_droplevel_list",
    "tests/indexes/multi/test_drop.py::test_droplevel_multiindex_one_level",
    "tests/indexes/multi/test_duplicates.py::test_duplicate_multiindex_codes",
    "tests/indexes/multi/test_equivalence.py::test_equals",
    "tests/indexes/multi/test_equivalence.py::test_equals_multi",
    "tests/indexes/multi/test_equivalence.py::test_equals_op",
    "tests/indexes/multi/test_equivalence.py::test_is_",
    "tests/indexes/multi/test_equivalence.py::test_multiindex_compare",
    "tests/indexes/multi/test_formats.py::TestRepr::test_rjust",
    "tests/indexes/multi/test_formats.py::TestRepr::test_tuple_width",
    "tests/indexes/multi/test_get_set.py::test_set_levels_categorical[False]",
    "tests/indexes/multi/test_get_set.py::test_set_levels_categorical[True]",
    "tests/indexes/multi/test_get_set.py::test_set_name_methods",
    "tests/indexes/multi/test_indexing.py::TestContains::test_contains",
    "tests/indexes/multi/test_indexing.py::TestContains::test_contains_with_missing_value",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_categorical_time",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_crossing_levels",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_kwarg_validation",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_methods",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_nan",
    "tests/indexes/multi/test_indexing.py::TestGetIndexer::test_get_indexer_three_or_more_levels",
    "tests/indexes/multi/test_indexing.py::TestGetLoc::test_get_loc",
    "tests/indexes/multi/test_indexing.py::TestGetLoc::test_get_loc_duplicates2",
    "tests/indexes/multi/test_indexing.py::TestGetLoc::test_get_loc_past_lexsort_depth",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[Decimal]",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[NAType]",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[NaTType]",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[NoneType]",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[float0]",
    "tests/indexes/multi/test_indexing.py::test_get_indexer_for_multiindex_with_nans[float1]",
    "tests/indexes/multi/test_indexing.py::test_getitem",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_all[ind20-ind10]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_all[ind20-ind11]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_all[ind21-ind10]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_all[ind21-ind11]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_single[ind20-ind10]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_single[ind20-ind11]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_single[ind21-ind10]",
    "tests/indexes/multi/test_indexing.py::test_getitem_bool_index_single[ind21-ind11]",
    "tests/indexes/multi/test_indexing.py::test_getitem_group_select",
    "tests/indexes/multi/test_integrity.py::test_labels_dtypes",
    "tests/indexes/multi/test_integrity.py::test_memory_usage",
    "tests/indexes/multi/test_integrity.py::test_metadata_immutable",
    "tests/indexes/multi/test_isin.py::test_isin_level_kwarg",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Float32-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Float32-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Float64-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Float64-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int16-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int16-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int32-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int32-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int64-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int64-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int8-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[Int8-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt16-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt16-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt32-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt32-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt64-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt64-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt8-0]",
    "tests/indexes/multi/test_join.py::test_join_dtypes[UInt8-5]",
    "tests/indexes/multi/test_join.py::test_join_dtypes_all_nan[Float32]",
    "tests/indexes/multi/test_join.py::test_join_dtypes_all_nan[Float64]",
    "tests/indexes/multi/test_join.py::test_join_dtypes_all_nan[Int16]",
    "tests/indexes/multi/test_join.py::test_join_dtypes_all_nan[Int32]",
    "tests/indexes/multi/test_join.py::test_join_dtypes_all_nan[Int64]",
    "tests/indexes/multi/test_join.py::test_join_dtypes_all_nan[Int8]",
    "tests/indexes/multi/test_join.py::test_join_dtypes_all_nan[UInt16]",
    "tests/indexes/multi/test_join.py::test_join_dtypes_all_nan[UInt32]",
    "tests/indexes/multi/test_join.py::test_join_dtypes_all_nan[UInt64]",
    "tests/indexes/multi/test_join.py::test_join_dtypes_all_nan[UInt8]",
    "tests/indexes/multi/test_join.py::test_join_index_levels",
    "tests/indexes/multi/test_join.py::test_join_midx_ea",
    "tests/indexes/multi/test_join.py::test_join_midx_string",
    "tests/indexes/multi/test_join.py::test_join_multi_return_indexers",
    "tests/indexes/multi/test_join.py::test_join_overlapping_interval_level",
    "tests/indexes/multi/test_join.py::test_join_self[inner]",
    "tests/indexes/multi/test_join.py::test_join_self[left]",
    "tests/indexes/multi/test_join.py::test_join_self[outer]",
    "tests/indexes/multi/test_join.py::test_join_self[right]",
    "tests/indexes/multi/test_missing.py::test_dropna",
    "tests/indexes/multi/test_missing.py::test_nulls",
    "tests/indexes/multi/test_names.py::test_copy_names",
    "tests/indexes/multi/test_names.py::test_setting_names_from_levels_raises",
    "tests/indexes/multi/test_reindex.py::test_reindex_base",
    "tests/indexes/multi/test_reshape.py::test_append_names_match[c-None]",
    "tests/indexes/multi/test_setops.py::test_difference[False]",
    "tests/indexes/multi/test_setops.py::test_difference[None]",
    "tests/indexes/multi/test_setops.py::test_intersection_non_object[False]",
    "tests/indexes/multi/test_setops.py::test_intersection_non_object[None]",
    "tests/indexes/multi/test_setops.py::test_setop_with_categorical[False-difference]",
    "tests/indexes/multi/test_setops.py::test_setop_with_categorical[False-intersection]",
    "tests/indexes/multi/test_setops.py::test_setop_with_categorical[False-symmetric_difference]",
    "tests/indexes/multi/test_setops.py::test_setop_with_categorical[False-union]",
    "tests/indexes/multi/test_setops.py::test_setop_with_categorical[None-difference]",
    "tests/indexes/multi/test_setops.py::test_setop_with_categorical[None-intersection]",
    "tests/indexes/multi/test_setops.py::test_setop_with_categorical[None-symmetric_difference]",
    "tests/indexes/multi/test_setops.py::test_setop_with_categorical[None-union]",
    "tests/indexes/multi/test_sorting.py::test_argsort",
    "tests/indexes/multi/test_sorting.py::test_numpy_argsort",
    "tests/indexes/multi/test_sorting.py::test_reconstruct_remove_unused",
    "tests/indexes/multi/test_sorting.py::test_reconstruct_sort",
    "tests/indexes/multi/test_sorting.py::test_remove_unused_levels_large[datetime64[D]-str]",
    "tests/indexes/multi/test_sorting.py::test_remove_unused_levels_large[int64-int64]",
    "tests/indexes/multi/test_sorting.py::test_unsortedindex_doc_examples",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_astype_float64_to_uint64",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[inf-int16]",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[inf-int32]",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[inf-int64]",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[inf-int]",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[nan-int16]",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[nan-int32]",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[nan-int64]",
    "tests/indexes/numeric/test_astype.py::TestAstype::test_cannot_cast_inf_to_int[nan-int]",
    "tests/indexes/numeric/test_indexing.py::TestContains::test_contains_float64_nans",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_invalid",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_numeric_index_boolean_target[get_indexer-float64]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_numeric_index_boolean_target[get_indexer-int64]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_numeric_index_boolean_target[get_indexer-range]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_numeric_index_boolean_target[get_indexer-uint64]",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_indexer_numeric_vs_bool",
    "tests/indexes/numeric/test_indexing.py::TestGetIndexer::test_get_loc_masked_na_and_nan",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_equals_numeric",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[int16]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[int32]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[int64]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[int8]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[int]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[uint16]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[uint32]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[uint64]",
    "tests/indexes/numeric/test_numeric.py::TestFloatNumericIndex::test_type_coercion_fail[uint8]",
    "tests/indexes/numeric/test_numeric.py::TestIntNumericIndex::test_constructor[int16]",
    "tests/indexes/numeric/test_numeric.py::TestIntNumericIndex::test_constructor[int32]",
    "tests/indexes/numeric/test_numeric.py::TestIntNumericIndex::test_constructor[int64]",
    "tests/indexes/object/test_indexing.py::TestSliceLocs::test_slice_locs_negative_step[in_slice13--object]",
    "tests/indexes/object/test_indexing.py::TestSliceLocs::test_slice_locs_negative_step[in_slice13--string[pyarrow_numpy]]",
    "tests/indexes/period/test_constructors.py::TestPeriodIndex::test_constructor_fromarraylike",
    "tests/indexes/period/test_indexing.py::TestWhere::test_where_invalid_dtypes",
    "tests/indexes/period/test_join.py::TestJoin::test_join_self[inner]",
    "tests/indexes/period/test_join.py::TestJoin::test_join_self[left]",
    "tests/indexes/period/test_join.py::TestJoin::test_join_self[outer]",
    "tests/indexes/period/test_join.py::TestJoin::test_join_self[right]",
    "tests/indexes/period/test_period.py::test_dunder_array[True]",
    "tests/indexes/period/test_period_range.py::TestPeriodRange::test_construction_from_string[W-W]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args0-kwargs0-0-5-1-None]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args0-kwargs0-0-5-1-foo]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args1-kwargs1-1-5-1-None]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args1-kwargs1-1-5-1-foo]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args2-kwargs2-1-5-2-None]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args2-kwargs2-1-5-2-foo]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args3-kwargs3-0-0-1-None]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args3-kwargs3-0-0-1-foo]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args4-kwargs4-0-0-1-None]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args4-kwargs4-0-0-1-foo]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args5-kwargs5-0-0-1-None]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args5-kwargs5-0-0-1-foo]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args6-kwargs6-0-0-1-None]",
    "tests/indexes/ranges/test_constructors.py::TestRangeIndexConstructors::test_constructor[args6-kwargs6-0-0-1-foo]",
    "tests/indexes/ranges/test_join.py::TestJoin::test_join_self[inner]",
    "tests/indexes/ranges/test_join.py::TestJoin::test_join_self[left]",
    "tests/indexes/ranges/test_join.py::TestJoin::test_join_self[outer]",
    "tests/indexes/ranges/test_join.py::TestJoin::test_join_self[right]",
    "tests/indexes/ranges/test_range.py::TestRangeIndex::test_append_len_one[ri0]",
    "tests/indexes/ranges/test_range.py::TestRangeIndex::test_append_len_one[ri1]",
    "tests/indexes/ranges/test_range.py::TestRangeIndex::test_append_len_one[ri2]",
    "tests/indexes/ranges/test_range.py::TestRangeIndex::test_append_len_one[ri3]",
    "tests/indexes/ranges/test_range.py::TestRangeIndex::test_append_len_one[ri4]",
    "tests/indexes/ranges/test_range.py::TestRangeIndex::test_cache",
    "tests/indexes/ranges/test_range.py::TestRangeIndex::test_copy",
    "tests/indexes/ranges/test_range.py::TestRangeIndex::test_engineless_lookup",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=-10, step=-1)-RangeIndex(start=-10, stop=-20, step=-1)-RangeIndex(start=-19, stop=1, step=1)-RangeIndex(start=0, stop=-20, step=-1)]",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=-100, step=-1)-RangeIndex(start=0, stop=-50, step=-3)-RangeIndex(start=-99, stop=1, step=1)-RangeIndex(start=0, stop=-100, step=-1)]",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=-100, step=-2)-RangeIndex(start=-100, stop=50, step=102)-RangeIndex(start=-100, stop=4, step=2)-expected_notsorted15]",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=-100, step=-5)-RangeIndex(start=5, stop=-100, step=-20)-RangeIndex(start=-95, stop=10, step=5)-expected_notsorted10]",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=-11, step=-1)-RangeIndex(start=1, stop=-12, step=-4)-RangeIndex(start=-11, stop=2, step=1)-expected_notsorted11]",
    "tests/indexes/ranges/test_setops.py::TestRangeIndexSetOps::test_union_sorted[RangeIndex(start=0, stop=10, step=5)-RangeIndex(start=-5, stop=-6, step=-20)-RangeIndex(start=-5, stop=10, step=5)-expected_notsorted18]",
    "tests/indexes/ranges/test_setops.py::test_range_difference",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[pyarrow]-None]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[pyarrow]-nan0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[pyarrow]-nan1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[pyarrow]-null3]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[python]-None]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[python]-nan0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[python]-nan1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=str[python]-null3]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[pyarrow]-None]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[pyarrow]-nan0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[pyarrow]-nan1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[pyarrow]-null3]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[python]-None]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[python]-nan0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[python]-nan1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_missing[string=string[python]-null3]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=str[pyarrow]-backfill-expected1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=str[pyarrow]-pad-expected0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=str[python]-backfill-expected1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=str[python]-pad-expected0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=string[pyarrow]-backfill-expected1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=string[pyarrow]-pad-expected0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=string[python]-backfill-expected1]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings[string=string[python]-pad-expected0]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings_raises[string=str[pyarrow]]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings_raises[string=str[python]]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings_raises[string=string[pyarrow]]",
    "tests/indexes/string/test_indexing.py::TestGetIndexer::test_get_indexer_strings_raises[string=string[python]]",
    "tests/indexes/string/test_indexing.py::TestSliceLocs::test_slice_locs_negative_step[string=object-in_slice13-]",
    "tests/indexes/string/test_indexing.py::TestSliceLocs::test_slice_locs_negative_step[string=str[pyarrow]-in_slice13-]",
    "tests/indexes/string/test_indexing.py::TestSliceLocs::test_slice_locs_negative_step[string=str[python]-in_slice13-]",
    "tests/indexes/string/test_indexing.py::TestSliceLocs::test_slice_locs_negative_step[string=string[pyarrow]-in_slice13-]",
    "tests/indexes/string/test_indexing.py::TestSliceLocs::test_slice_locs_negative_step[string=string[python]-in_slice13-]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[bool-dtype]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[bool-object]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[categorical]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[complex128]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[complex64]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[datetime-tz]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[datetime]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[empty]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[float32]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[float64]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[int16]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[int32]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[int64]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[int8]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[interval]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[mi-with-dt64tz-level]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[multi]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[nullable_bool]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[nullable_float]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[nullable_int]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[nullable_uint]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[object]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[period]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[range]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[repeats]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[string-pyarrow]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[string-python]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[string]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[timedelta]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[tuples]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[uint16]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[uint32]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[uint64]",
    "tests/indexes/test_any_index.py::TestConversion::test_to_series_with_arguments[uint8]",
    "tests/indexes/test_any_index.py::TestIndexing::test_getitem_error[multi-2.5]",
    "tests/indexes/test_any_index.py::TestIndexing::test_getitem_error[range-2.5]",
    "tests/indexes/test_any_index.py::TestIndexing::test_getitem_error[tuples-2.5]",
    "tests/indexes/test_base.py::TestIndex::test_cached_properties_not_settable",
    "tests/indexes/test_base.py::TestIndex::test_empty_fancy_raises[float32]",
    "tests/indexes/test_base.py::TestIndex::test_empty_fancy_raises[float64]",
    "tests/indexes/test_base.py::TestIndex::test_empty_fancy_raises[int32]",
    "tests/indexes/test_base.py::TestIndex::test_empty_fancy_raises[int64]",
    "tests/indexes/test_base.py::TestIndex::test_empty_fancy_raises[string]",
    "tests/indexes/test_base.py::TestIndex::test_empty_fancy_raises[uint32]",
    "tests/indexes/test_base.py::TestIndex::test_empty_fancy_raises[uint64]",
    "tests/indexes/test_base.py::TestIndex::test_equals_op_mismatched_multiindex_raises[index0]",
    "tests/indexes/test_base.py::TestIndex::test_is_",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[bool-dtype-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[categorical-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[datetime-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[empty-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[float32-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[float64-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[int16-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[int32-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[int64-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[int8-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[interval-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[nullable_bool-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[nullable_float-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[nullable_int-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[nullable_uint-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[object-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[range-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[repeats-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[string-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[string-pyarrow-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[string-python-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[uint16-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[uint32-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[uint64-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_label_raises[uint8-nan]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[bool-dtype]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[categorical]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[empty]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[float32]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[float64]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[int16]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[int32]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[int64]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[int8]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[interval]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[nullable_bool]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[nullable_float]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[nullable_int]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[nullable_uint]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[object]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[range]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[repeats]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[string-pyarrow]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[string-python]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[string]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[uint16]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[uint32]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[uint64]",
    "tests/indexes/test_base.py::TestIndex::test_isin_level_kwarg_bad_level_raises[uint8]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NAType-float32]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NAType-float64]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NAType-float]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NoneType-float32]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NoneType-float64]",
    "tests/indexes/test_base.py::TestIndex::test_isin_nan_common_float64[NoneType-float]",
    "tests/indexes/test_base.py::TestIndex::test_map_defaultdict",
    "tests/indexes/test_base.py::TestIndex::test_str_attribute_raises[index2]",
    "tests/indexes/test_base.py::TestIndex::test_str_bool_return",
    "tests/indexes/test_base.py::TestIndex::test_tab_completion[index1-False]",
    "tests/indexes/test_common.py::TestCommon::test_constructor_non_hashable_name[range]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[bool-dtype-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[bool-dtype-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[bool-object-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[bool-object-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[categorical-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[categorical-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[complex128-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[complex128-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[complex64-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[complex64-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[datetime-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[datetime-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[datetime-tz-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[datetime-tz-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[empty-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[empty-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[float32-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[float32-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[float64-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[float64-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int16-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int16-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int32-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int32-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int64-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int64-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int8-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[int8-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[interval-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[interval-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_bool-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_bool-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_float-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_float-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_int-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_int-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_uint-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[nullable_uint-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[object-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[object-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[period-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[period-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[range-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[range-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[repeats-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[repeats-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-pyarrow-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-pyarrow-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-python-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[string-python-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[timedelta-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[timedelta-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint16-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint16-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint32-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint32-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint64-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint64-new_name]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint8-None]",
    "tests/indexes/test_common.py::TestCommon::test_to_frame[uint8-new_name]",
    "tests/indexes/test_common.py::test_ndarray_compat_properties[multi]",
    "tests/indexes/test_common.py::test_ndarray_compat_properties[tuples]",
    "tests/indexes/test_common.py::test_sort_values_invalid_na_position[nullable_int-None]",
    "tests/indexes/test_common.py::test_sort_values_invalid_na_position[nullable_int-middle]",
    "tests/indexes/test_common.py::test_sort_values_with_missing[nullable_int-first]",
    "tests/indexes/test_common.py::test_sort_values_with_missing[nullable_int-last]",
    "tests/indexes/test_datetimelike.py::TestDatetimeLike::test_argsort_matches_array[simple_index1]",
    "tests/indexes/test_datetimelike.py::TestDatetimeLike::test_argsort_matches_array[simple_index2]",
    "tests/indexes/test_indexing.py::TestGetIndexer::test_get_indexer_base[multi]",
    "tests/indexes/test_indexing.py::TestGetIndexer::test_get_indexer_base[tuples]",
    "tests/indexes/test_indexing.py::TestTake::test_take_indexer_type",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arccos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arccosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arcsin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arcsinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arctan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-arctanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-cos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-cosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-deg2rad]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-exp2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-exp]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-expm1]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-log10]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-log1p]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-log2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-log]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-rad2deg]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-sin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-sinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-sqrt]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-tan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[bool-dtype-tanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arccos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arccosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arcsin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arcsinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arctan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-arctanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-cos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-cosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-exp2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-exp]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-expm1]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-log10]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-log1p]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-log2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-log]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-sin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-sinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-sqrt]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-tan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex128-tanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arccos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arccosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arcsin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arcsinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arctan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-arctanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-cos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-cosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-exp2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-exp]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-expm1]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-log10]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-log1p]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-log2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-log]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-sin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-sinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-tan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[complex64-tanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[int8-deg2rad]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[int8-rad2deg]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arccos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arccosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arcsin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arcsinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arctan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-arctanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-cos]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-cosh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-deg2rad]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-exp2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-exp]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-expm1]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-log10]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-log1p]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-log2]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-log]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-rad2deg]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-sin]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-sinh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-sqrt]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-tan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[nullable_bool-tanh]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[uint8-deg2rad]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_basic[uint8-rad2deg]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[bool-dtype-isfinite]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[bool-dtype-isinf]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[bool-dtype-isnan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[bool-dtype-signbit]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[nullable_bool-isfinite]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[nullable_bool-isinf]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[nullable_bool-isnan]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_other[nullable_bool-signbit]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_out[interval]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_out[multi]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_out[tuples]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_reductions[multi-maximum]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_reductions[multi-minimum]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_reductions[tuples-maximum]",
    "tests/indexes/test_numpy_compat.py::test_numpy_ufuncs_reductions[tuples-minimum]",
    "tests/indexes/test_old_base.py::TestBase::test_argsort[nullable_bool]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index0]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index10]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index11]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index12]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index13]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index1]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index2]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index3]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index4]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index5]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index6]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index7]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index8]",
    "tests/indexes/test_old_base.py::TestBase::test_copy_shares_cache[simple_index9]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index13]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index1]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index2]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index3]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index5]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index6]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index7]",
    "tests/indexes/test_old_base.py::TestBase::test_engine_reference_cycle[simple_index8]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[bool-dtype]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[complex128]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[complex64]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[datetime-tz]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[datetime]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[empty]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[float32]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[float64]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[int16]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[int32]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[int64]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[int8]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[nullable_bool]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[nullable_float]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[nullable_int]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[nullable_uint]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[object]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[period]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[repeats]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[string-pyarrow]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[string-python]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[string]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[timedelta]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[uint16]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[uint32]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[uint64]",
    "tests/indexes/test_old_base.py::TestBase::test_ensure_copied_data[uint8]",
    "tests/indexes/test_old_base.py::TestBase::test_equals_op[simple_index13]",
    "tests/indexes/test_old_base.py::TestBase::test_getitem_2d_deprecated[simple_index0]",
    "tests/indexes/test_old_base.py::TestBase::test_getitem_2d_deprecated[simple_index4]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[bool-dtype]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[categorical]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[empty]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[float32]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[float64]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[int16]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[int32]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[int64]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[int8]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[multi]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[object]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[repeats]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[string]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[tuples]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[uint16]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[uint32]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[uint64]",
    "tests/indexes/test_old_base.py::TestBase::test_memory_usage[uint8]",
    "tests/indexes/test_old_base.py::TestBase::test_nulls[multi]",
    "tests/indexes/test_old_base.py::TestBase::test_nulls[tuples]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[bool-dtype]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[categorical]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[empty]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[float32]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[float64]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[int16]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[int32]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[int64]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[int8]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[interval]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[multi]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[nullable_bool]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[nullable_float]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[nullable_int]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[nullable_uint]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[object]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[repeats]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[string-pyarrow]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[string-python]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[string]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[tuples]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[uint16]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[uint32]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[uint64]",
    "tests/indexes/test_old_base.py::TestBase::test_numpy_argsort[uint8]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index0]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index10]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index11]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index12]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index13]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index1]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index2]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index3]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index4]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index5]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index6]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index7]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index8]",
    "tests/indexes/test_old_base.py::TestBase::test_shallow_copy_shares_cache[simple_index9]",
    "tests/indexes/test_setops.py::TestSetOps::test_difference_base[False-string-pyarrow]",
    "tests/indexes/test_setops.py::TestSetOps::test_difference_base[False-string-python]",
    "tests/indexes/test_setops.py::TestSetOps::test_difference_base[None-string-pyarrow]",
    "tests/indexes/test_setops.py::TestSetOps::test_difference_base[None-string-python]",
    "tests/indexes/test_setops.py::TestSetOpsUnsorted::test_difference_should_not_compare",
    "tests/indexes/test_setops.py::TestSetOpsUnsorted::test_intersection[False-string]",
    "tests/indexes/test_setops.py::TestSetOpsUnsorted::test_intersection[None-string]",
    "tests/indexes/test_setops.py::TestSetOpsUnsorted::test_union_identity[False-string]",
    "tests/indexes/test_setops.py::TestSetOpsUnsorted::test_union_identity[None-string]",
    "tests/indexes/test_setops.py::test_setop_with_categorical[bool-dtype-False-union]",
    "tests/indexes/test_setops.py::test_setop_with_categorical[bool-dtype-None-union]",
    "tests/indexes/test_setops.py::test_setop_with_categorical[nullable_bool-False-union]",
    "tests/indexes/test_setops.py::test_setop_with_categorical[nullable_bool-None-union]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-complex128]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-complex64]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-float32]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-float64]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-int16]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-int32]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-int64]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-int8]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-range]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-repeats]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-uint16]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-uint32]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-uint64]",
    "tests/indexes/test_setops.py::test_union_different_types[bool-dtype-uint8]",
    "tests/indexes/test_setops.py::test_union_different_types[categorical-nullable_float]",
    "tests/indexes/test_setops.py::test_union_different_types[categorical-nullable_uint]",
    "tests/indexes/test_setops.py::test_union_different_types[categorical-string-pyarrow]",
    "tests/indexes/test_setops.py::test_union_different_types[categorical-string-python]",
    "tests/indexes/test_setops.py::test_union_different_types[complex128-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[complex64-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[datetime-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[float32-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[float64-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[int16-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[int16-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[int32-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[int32-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[int64-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[int64-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[int8-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[int8-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[nullable_float-categorical]",
    "tests/indexes/test_setops.py::test_union_different_types[nullable_uint-categorical]",
    "tests/indexes/test_setops.py::test_union_different_types[range-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[range-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[repeats-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[repeats-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[string-pyarrow-categorical]",
    "tests/indexes/test_setops.py::test_union_different_types[string-python-categorical]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-datetime]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-int16]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-int32]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-int64]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-int8]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-range]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-repeats]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-uint16]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-uint32]",
    "tests/indexes/test_setops.py::test_union_different_types[timedelta-uint8]",
    "tests/indexes/test_setops.py::test_union_different_types[uint16-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[uint16-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[uint32-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[uint32-timedelta]",
    "tests/indexes/test_setops.py::test_union_different_types[uint64-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[uint8-bool-dtype]",
    "tests/indexes/test_setops.py::test_union_different_types[uint8-timedelta]",
    "tests/indexes/test_setops.py::test_union_duplicate_index_different_dtypes",
    "tests/indexes/test_setops.py::test_union_duplicate_index_subsets_of_each_other[category]",
    "tests/indexes/test_setops.py::test_union_same_value_duplicated_in_both",
    "tests/indexes/test_setops.py::test_union_with_duplicate_index_not_subset_and_non_monotonic[category]",
    "tests/indexes/timedeltas/methods/test_astype.py::TestTimedeltaIndex::test_astype",
    "tests/indexes/timedeltas/methods/test_astype.py::TestTimedeltaIndex::test_astype_raises[float]",
    "tests/indexes/timedeltas/methods/test_astype.py::TestTimedeltaIndex::test_astype_timedelta64",
    "tests/indexes/timedeltas/test_constructors.py::TestTimedeltaIndex::test_constructor_coverage",
    "tests/indexes/timedeltas/test_constructors.py::TestTimedeltaIndex::test_int64_nocopy",
    "tests/indexes/timedeltas/test_formats.py::TestTimedeltaIndexRendering::test_representation[__repr__]",
    "tests/indexes/timedeltas/test_formats.py::TestTimedeltaIndexRendering::test_representation[__str__]",
    "tests/indexes/timedeltas/test_indexing.py::TestContains::test_contains",
    "tests/indexes/timedeltas/test_setops.py::TestTimedeltaIndex::test_intersection_equal[False]",
    "tests/indexes/timedeltas/test_setops.py::TestTimedeltaIndex::test_intersection_equal[None]",
    "tests/indexes/timedeltas/test_setops.py::TestTimedeltaIndex::test_intersection_non_monotonic[None-rng2-expected2]",
    "tests/indexes/timedeltas/test_setops.py::TestTimedeltaIndex::test_union_freq_infer",
    "tests/indexes/timedeltas/test_timedelta.py::TestTimedeltaIndex::test_misc_coverage",
    "tests/indexing/interval/test_interval.py::TestIntervalIndexInsideMultiIndex::test_reindex_behavior_with_interval_index[1010]",
    "tests/indexing/interval/test_interval.py::TestIntervalIndexInsideMultiIndex::test_reindex_behavior_with_interval_index[101]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_loc_with_overlap[loc]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_loc_with_overlap[setitem]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_loc_with_slices[loc]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_loc_with_slices[setitem]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_non_unique_moar[loc]",
    "tests/indexing/interval/test_interval_new.py::TestIntervalIndex::test_non_unique_moar[setitem]",
    "tests/indexing/multiindex/test_chaining_and_caching.py::test_cache_updating",
    "tests/indexing/multiindex/test_chaining_and_caching.py::test_detect_chained_assignment",
    "tests/indexing/multiindex/test_getitem.py::test_frame_getitem_multicolumn_empty_level",
    "tests/indexing/multiindex/test_getitem.py::test_loc_empty_multiindex",
    "tests/indexing/multiindex/test_getitem.py::test_series_getitem_multiindex[0-expected0-<lambda>0]",
    "tests/indexing/multiindex/test_getitem.py::test_series_getitem_multiindex[0-expected0-<lambda>1]",
    "tests/indexing/multiindex/test_getitem.py::test_series_getitem_multiindex[1-expected1-<lambda>0]",
    "tests/indexing/multiindex/test_getitem.py::test_series_getitem_multiindex[1-expected1-<lambda>1]",
    "tests/indexing/multiindex/test_iloc.py::test_indexing_ambiguity_bug_1678",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[a-0]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[a-1]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[a-2]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[a-3]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[a-4]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[b-0]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[b-1]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[b-2]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[b-3]",
    "tests/indexing/multiindex/test_indexing_slow.py::test_multiindex_get_loc[b-4]",
    "tests/indexing/multiindex/test_loc.py::TestKeyErrorsWithMultiIndex::test_missing_key_combination",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_int_slice",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_multiindex_missing_label_raises",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-Index]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-Series]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-list]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-ndarray]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-set]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Index-tuple]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-Index]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-Series]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-list]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-ndarray]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-set]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[Series-tuple]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-Index]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-Series]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-list]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-ndarray]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-set]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[list-tuple]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-Index]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-Series]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-list]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-ndarray]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-set]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[set-tuple]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-Index]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-Series]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-list]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-ndarray]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-set]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_getitem_nested_indexer[tuple-tuple]",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_multiindex_indexer_none",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_multiindex_missing_label_raises",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_loc_no_second_level_index",
    "tests/indexing/multiindex/test_loc.py::TestMultiIndexLoc::test_sorted_multiindex_after_union",
    "tests/indexing/multiindex/test_loc.py::test_getitem_str_slice",
    "tests/indexing/multiindex/test_loc.py::test_loc_get_scalar_casting_to_float",
    "tests/indexing/multiindex/test_loc.py::test_loc_getitem_drops_levels_for_one_row_dataframe",
    "tests/indexing/multiindex/test_loc.py::test_loc_getitem_duplicates_multiindex_missing_indexers[indexer6-pos6]",
    "tests/indexing/multiindex/test_loc.py::test_loc_getitem_index_differently_ordered_slice_none_duplicates[indexer0]",
    "tests/indexing/multiindex/test_loc.py::test_loc_getitem_int",
    "tests/indexing/multiindex/test_loc.py::test_loc_getitem_int_raises_exception",
    "tests/indexing/multiindex/test_loc.py::test_loc_with_mi_indexer",
    "tests/indexing/multiindex/test_multiindex.py::TestMultiIndexBasic::test_rename_multiindex_with_duplicates",
    "tests/indexing/multiindex/test_partial.py::TestMultiIndexPartial::test_getitem_intkey_leading_level[float]",
    "tests/indexing/multiindex/test_partial.py::TestMultiIndexPartial::test_getitem_intkey_leading_level[int]",
    "tests/indexing/multiindex/test_partial.py::TestMultiIndexPartial::test_getitem_partial",
    "tests/indexing/multiindex/test_partial.py::TestMultiIndexPartial::test_getitem_partial_column_select",
    "tests/indexing/multiindex/test_partial.py::TestMultiIndexPartial::test_partial_set",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_frame_getitem_setitem_boolean",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_frame_getitem_setitem_multislice",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_frame_setitem_multi_column2",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_loc_getitem_setitem_slice_integers[DataFrame]",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_loc_getitem_setitem_slice_integers[Series]",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_multiindex_assignment_single_dtype",
    "tests/indexing/multiindex/test_setitem.py::TestMultiIndexSetItem::test_series_setitem",
    "tests/indexing/multiindex/test_setitem.py::test_frame_setitem_copy_no_write",
    "tests/indexing/multiindex/test_setitem.py::test_frame_setitem_copy_raises",
    "tests/indexing/multiindex/test_setitem.py::test_frame_setitem_view_direct",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[int-loc0-iloc0]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[int-loc3-iloc3]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[int-loc4-iloc4]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[int-loc7-iloc7]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[str-loc12-iloc12]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_loc_slice_negative_stepsize[str-loc8-iloc8]",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_multiindex_slicers_datetimelike",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_per_axis_per_level_doc_examples",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_per_axis_per_level_getitem",
    "tests/indexing/multiindex/test_slice.py::TestMultiIndexSlicers::test_per_axis_per_level_setitem",
    "tests/indexing/multiindex/test_sorted.py::TestMultiIndexSorted::test_argsort_with_na",
    "tests/indexing/multiindex/test_sorted.py::TestMultiIndexSorted::test_frame_getitem_not_sorted",
    "tests/indexing/multiindex/test_sorted.py::TestMultiIndexSorted::test_series_getitem_not_sorted",
    "tests/indexing/test_at.py::test_at_timezone",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_ix_categorical_index",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[False-idx_values17]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[False-idx_values18]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[False-idx_values19]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[False-idx_values28]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[False-idx_values29]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[False-idx_values30]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[None-idx_values17]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[None-idx_values18]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[None-idx_values19]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[None-idx_values28]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[None-idx_values29]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[None-idx_values30]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[True-idx_values17]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[True-idx_values18]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[True-idx_values19]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[True-idx_values28]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[True-idx_values29]",
    "tests/indexing/test_categorical.py::TestCategoricalIndex::test_loc_getitem_with_non_string_categories[True-idx_values30]",
    "tests/indexing/test_chaining_and_caching.py::TestCaching::test_altering_series_clears_parent_cache",
    "tests/indexing/test_chaining_and_caching.py::TestCaching::test_setitem_cache_updating_slices",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_chained_getitem_with_lists",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_changing_dtype",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_doc_example",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_fails",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_implicit_take",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_implicit_take2",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_object_dtype",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_raises",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_setting_entire_column",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_undefined_column",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_warning_stacklevel[3]",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_detect_chained_assignment_warning_stacklevel[rhs1]",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_getitem_loc_assignment_slice_state",
    "tests/indexing/test_chaining_and_caching.py::TestChaining::test_setitem_chained_setfault",
    "tests/indexing/test_coercion.py::TestFillnaSeriesCoercion::test_fillna_datetime[index-object0]",
    "tests/indexing/test_coercion.py::TestFillnaSeriesCoercion::test_fillna_datetime[series-object0]",
    "tests/indexing/test_coercion.py::TestFillnaSeriesCoercion::test_fillna_float64[index-True-object]",
    "tests/indexing/test_coercion.py::TestFillnaSeriesCoercion::test_fillna_float64[series-True-object]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[dict-bool-float64]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[dict-bool-int64]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[dict-float64-bool]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-bool-datetime64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-bool-float64]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-bool-int64]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-bool-timedelta64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-datetime64-timedelta64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-float64-datetime64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-float64-timedelta64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-int64-datetime64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-int64-timedelta64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-object-datetime64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-object-timedelta64[ns]]",
    "tests/indexing/test_coercion.py::TestReplaceSeriesCoercion::test_replace_series[series-timedelta64-datetime64[ns]]",
    "tests/indexing/test_coercion.py::TestWhereCoercion::test_where_datetime64[index-datetime64]",
    "tests/indexing/test_coercion.py::TestWhereCoercion::test_where_datetime64[index-datetime64tz]",
    "tests/indexing/test_coercion.py::TestWhereCoercion::test_where_datetime64[series-datetime64]",
    "tests/indexing/test_coercion.py::TestWhereCoercion::test_where_datetime64[series-datetime64tz]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_floating_misc[setitem]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_integer[DataFrame-loc-index0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_integer[DataFrame-loc-index1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_integer[DataFrame-setitem-index0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_integer[DataFrame-setitem-index1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_integer[Series-loc-index0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_integer[Series-loc-index1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_integer[Series-setitem-index0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_integer[Series-setitem-index1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_non_numeric[DataFrame-loc-index0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_non_numeric[DataFrame-loc-index1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_non_numeric[DataFrame-setitem-index0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_non_numeric[DataFrame-setitem-index1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_non_numeric[Series-loc-index0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_non_numeric[Series-loc-index1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_non_numeric[Series-setitem-index0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_scalar_non_numeric[Series-setitem-index1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[DataFrame-loc-idx0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[DataFrame-loc-idx1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[DataFrame-loc-idx2]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[DataFrame-setitem-idx0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[DataFrame-setitem-idx1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[DataFrame-setitem-idx2]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[Series-loc-idx0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[Series-loc-idx1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[Series-loc-idx2]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[Series-setitem-idx0]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[Series-setitem-idx1]",
    "tests/indexing/test_floats.py::TestFloatIndexers::test_slice_float[Series-setitem-idx2]",
    "tests/indexing/test_iat.py::test_iat_setitem_item_cache_cleared[iat]",
    "tests/indexing/test_iat.py::test_iat_setitem_item_cache_cleared[iloc]",
    "tests/indexing/test_iloc.py::TestILocErrors::test_iloc_getitem_setitem_fancy_exceptions",
    "tests/indexing/test_iloc.py::TestILocSeries::test_iloc_nullable_int64_size_1_nan",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_identity_slice_returns_new_object",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_getitem_doc_issue",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_getitem_int_single_ea_block_view",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_getitem_slice_negative_step_ea_block",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_mask",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_axis_argument",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_categorical_updates_inplace",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[DataFrame-Series]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[DataFrame-array]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[Series-Series]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_ea_inplace[Series-array]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_empty_frame_raises_with_3d_ndarray",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key0]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key1]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key2]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key3]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key4]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[iloc-key5]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key0]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key1]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key2]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key3]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key4]",
    "tests/indexing/test_iloc.py::TestiLocBaseIndependent::test_iloc_setitem_fullcol_categorical[loc-key5]",
    "tests/indexing/test_indexing.py::TestDataframeNoneCoercion::test_coercion_with_loc[expected2]",
    "tests/indexing/test_indexing.py::TestDataframeNoneCoercion::test_coercion_with_setitem_and_dataframe[expected2]",
    "tests/indexing/test_indexing.py::TestDataframeNoneCoercion::test_none_coercion_loc_and_dataframe[expected2]",
    "tests/indexing/test_indexing.py::TestDataframeNoneCoercion::test_none_coercion_mixed_dtypes",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['+01:15'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['+01:15'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['+01:15'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['-02:15'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['-02:15'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['-02:15'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['Asia/Tokyo'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['Asia/Tokyo'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['Asia/Tokyo'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['US/Eastern'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['US/Eastern'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['US/Eastern'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC+01:15'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC+01:15'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC+01:15'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC-02:15'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC-02:15'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['UTC-02:15'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/Asia/Singapore'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/Asia/Singapore'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/Asia/Singapore'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/US/Pacific'-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/US/Pacific'-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar['dateutil/US/Pacific'-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[<UTC>-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[<UTC>-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[<UTC>-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[None-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[None-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[None-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone(datetime.timedelta(seconds=3600))-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone.utc-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone.utc-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[datetime.timezone.utc-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(-300)-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(-300)-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(-300)-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(300)-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(300)-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[pytz.FixedOffset(300)-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzlocal()-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzlocal()-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzlocal()-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzutc()-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzutc()-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[tzutc()-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='US/Pacific')-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='UTC')-iloc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='UTC')-loc]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_scalar[zoneinfo.ZoneInfo(key='UTC')-setitem]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['+01:15'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['-02:15'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['Asia/Tokyo'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['US/Eastern'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC+01:15'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['UTC-02:15'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/Asia/Singapore'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values['dateutil/US/Pacific'-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[<UTC>-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[None-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(days=-1, seconds=82800), 'foo')-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone(datetime.timedelta(seconds=3600))-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[datetime.timezone.utc-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(-300)-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[pytz.FixedOffset(300)-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzlocal()-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-iloc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-loc-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key2-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key2-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[tzutc()-setitem-key2-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key0-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key0-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key0-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key0-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key0-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key1-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key1-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key1-array0]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key1-array1]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key1-list]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key2-Categorical]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key='US/Pacific')-iloc-key2-Index]",
    "tests/indexing/test_indexing.py::TestDatetimelikeCoercion::test_setitem_dt64_string_values[zoneinfo.ZoneInfo(key=