# Copyright Louis Dionne 2017
# Copyright Bruno Dutra 2016
#
# Distributed under the Boost Software License, Version 1.0.
# See https://github.com/ldionne/metabench/blob/master/LICENSE.md.
#
# This file is part of Metabench, a simple framework for compile-time
# benchmarks. For documentation, questions and other resources please
# visit the home of the project at https://github.com/ldionne/metabench.

cmake_minimum_required(VERSION 3.30.4 FATAL_ERROR)

find_package(Ruby 2.1 QUIET)
if(NOT RUBY_EXECUTABLE)
    message(WARNING "Ruby >= 2.1 was not found; the metabench.cmake module can't be used.")
    return()
endif()

# The directory where all files related to Metabench support are written, to
# avoid polluting the build tree.
set(METABENCH_DIR "${CMAKE_CURRENT_BINARY_DIR}/_metabench")


# metabench_add_dataset(target path_to_template range
#                       [NAME name]
#                       [ENV env]
#                       [COLOR color]
#                       [MEDIAN_OF n]
#                       [OUTPUT path/to/file])
#
#   Registers a dataset with Metabench. When needed, the dataset will be
#   generated by rendering the ERB template at `path_to_template` for each
#   value of n in the `range`. Then, the resulting C++ file will be compiled
#   and benchmarking data will be recorded. The `target` target represents
#   the executable being built for each value of `n`; any property set on
#   that target will take effect when collecting the dataset.
#
#   Note that building the `target` target itself is not supported. To
#   trigger the generation of the dataset, one must either add the dataset
#   to a chart using `metabench_add_chart`, or create a custom CMake target
#   that depends on the JSON file itself. When that target is built, the
#   dataset will be generated if and only if it is outdated.
#
#   Additionally, a CTest target with the same name is also created. When run,
#   this CTest target will run the benchmark for only the first and last
#   elements in `range`, and won't gather any benchmarking data. This is very
#   useful to make sure that benchmarks stay sane as part of continuous
#   integration scripts, for example.
#
#   What is measured?
#   -----------------
#   When generating a dataset, Metabench records information related to
#   several different aspects of the executable. All the different aspects
#   are always stored in the dataset, and the aspect being displayed on a
#   chart can be controlled with the `ASPECT` option of `metabench_add_chart`.
#   Currently, Metabench measures the following aspects:
#       - Compilation time in seconds. This is the time required to generate
#         the object file from the `.cpp` file. This includes preprocessing,
#         but does not include linking time. Compilation time is treated in
#         a slightly special way; the program is first built with the
#         `METABENCH` macro undefined, and then with the macro defined.
#         The compilation time is then taken to be the difference of the
#         measurement with the macro defined and without the macro defined.
#       - Link time in seconds. This is the time required to generate the
#         executable from the object file. This does not include the
#         compilation time.
#       - Executable size in KB. This is the size of the executable created
#         from the `.cpp` file.
#       - Peak memory usage of the compiler in KB. This is the approximate peak
#         size of the RSS (Resident Set Size) used by the compiler (not the
#         linker) when compiling the `.cpp` file.
#
#   Parameters
#   ----------
#   target:
#       The name of the target associated to this dataset.
#
#   path_to_template:
#       The path of the ERB template to use for this dataset. This may be
#       either an absolute path or a path relative to the current source
#       directory.
#
#   range:
#       A sequence of integers representing the values of `n` for which the
#       template will be rendered and measurements will be taken. The sequence
#       must be provided as a Ruby Iterable (such as `(1...100).step(5)`).
#
#   [NAME name]:
#       A name used to identify the dataset. For example, this is used as the
#       name of the curve associated to the dataset inside a chart. Defaults
#       to `target`.
#
#   [ENV env]:
#       An arbitrary Ruby Hash that will be made available to the ERB template
#       in the `env` variable. For example, if you pass `ENV "{foo: 1, bar: 2}"`,
#       `env[:foo]` will be equal to `1` and `env[:bar]` will be equal to `2`
#       inside the ERB template. This argument can be used to parameterize the
#       dataset in more complex ways.
#
#   [COLOR color]:
#       A string specifying the color used to display the dataset on the chart
#       in any valid CSS format.
#
#   [SCALE factor]:
#       A constant multiplicative factor that may be used to scale results.
#       Defaults to `1.0`.
#
#   [MEDIAN_OF n]:
#       The number of times to build and run the ERB template when gathering
#       timing information. The timings are taken `n` times, and the median of
#       these `n` values is retained. This can help reduce variability at the
#       cost of longer benchmarking times. By default, the timings are taken
#       only once. `n` must be a positive (non-zero) integer.
#
#   [OUTPUT path/to/file]:
#       The path of the resulting JSON file containing the benchmark data. If
#       a relative path is used, the path will be considered as being relative
#       to the current CMake binary directory. This defaults to `target.json`,
#       so that the output file will be `target.json` in the current CMake
#       binary directory.
function(metabench_add_dataset target path_to_template range)
    set(options)
    set(one_value_args NAME ENV COLOR SCALE MEDIAN_OF OUTPUT)
    set(multi_value_args)
    cmake_parse_arguments(ARGS "${options}" "${one_value_args}" "${multi_value_args}" ${ARGN})

    if(NOT IS_ABSOLUTE "${path_to_template}")
        set(path_to_template "${CMAKE_CURRENT_SOURCE_DIR}/${path_to_template}")
    endif()
    if(NOT EXISTS ${path_to_template})
        message(FATAL_ERROR "The file specified to metabench_add_dataset (${path_to_template}) does not exist.")
    endif()

    if(NOT ARGS_NAME)
        set(ARGS_NAME ${target})
    endif()

    if(NOT ARGS_ENV)
        set(ARGS_ENV "{}")
    endif()

    if(NOT ARGS_COLOR)
        set(ARGS_COLOR "")
    endif()

    if(NOT ARGS_SCALE)
        set(ARGS_SCALE 1.0)
    endif()

    if(NOT ARGS_MEDIAN_OF)
        set(ARGS_MEDIAN_OF 1)
    endif()

    if(NOT ARGS_OUTPUT)
        set(ARGS_OUTPUT "${target}.json")
    endif()
    if(NOT IS_ABSOLUTE "${ARGS_OUTPUT}")
        set(ARGS_OUTPUT "${CMAKE_CURRENT_BINARY_DIR}/${ARGS_OUTPUT}")
    endif()


    # Add a dummy executable that will be used to collect the dataset.
    # We'll build this executable multiple times for different values
    # of `n`, and collect compilation statistics each time. Compiling
    # through CMake allows us to be portable without additional work.
    #
    # Also, metabench_add_chart needs to be able to find the JSON file
    # containing the benchmark data from the name of the executable target,
    # so we store it in a custom property.
    file(WRITE "${METABENCH_DIR}/${target}.cpp" "")
    add_executable(${target} EXCLUDE_FROM_ALL "${METABENCH_DIR}/${target}.cpp")
    set_target_properties(${target} PROPERTIES
        RULE_LAUNCH_COMPILE "${RUBY_EXECUTABLE} -- \"${COMPILE_RB_PATH}\""
        RULE_LAUNCH_LINK "${RUBY_EXECUTABLE} -- \"${LINK_RB_PATH}\""
        RUNTIME_OUTPUT_DIRECTORY "${METABENCH_DIR}"
        METABENCH_DATASET_PATH "${ARGS_OUTPUT}"
    )
    get_filename_component(template_dir "${path_to_template}" DIRECTORY)
    target_include_directories(${target} PUBLIC "${template_dir}")


    # Add a command to generate the JSON file that will contain the measurements
    # we collect for this dataset when we build the executable above.
    add_custom_command(OUTPUT "${ARGS_OUTPUT}"
        COMMAND "${RUBY_EXECUTABLE}" -r json -r fileutils -r "${METABENCH_RB_PATH}"
            -e "range = (${range}).to_a"
            -e "env = (${ARGS_ENV})"
            -e "data = {}"
            -e "data['key'] = '${ARGS_NAME}'"
            -e "data['scale'] = (${ARGS_SCALE})"
            -e "data['color'] = '${ARGS_COLOR}'"
            -e "data['values'] = measure('${target}', '${path_to_template}', range, env, ${ARGS_MEDIAN_OF})"
            -e "FileUtils.mkdir_p(File.dirname('${ARGS_OUTPUT}'))"
            -e "IO.write('${ARGS_OUTPUT}', JSON.generate(data))"
        DEPENDS "${path_to_template}"
        WORKING_DIRECTORY "${CMAKE_SOURCE_DIR}"
        VERBATIM USES_TERMINAL
    )


    # We also setup a CTest target to test the generation of the dataset.
    # We do not actually collect any data here.
    add_test(NAME ${target}
        COMMAND "${RUBY_EXECUTABLE}" -r "${METABENCH_RB_PATH}"
            -e "range = (${range}).to_a"
            -e "range = [range[0], range[-1]] if range.length >= 2"
            -e "env = (${ARGS_ENV})"
            -e "data = measure('${target}', '${path_to_template}', range, env, 1)"
        WORKING_DIRECTORY "${CMAKE_SOURCE_DIR}"
    )
endfunction()

# metabench_add_chart(target [ALL]
#                     [ASPECT COMPILATION_TIME|LINK_TIME|EXECUTABLE_SIZE|PEAK_MEMORY]
#                     [TITLE title]
#                     [SUBTITLE subtitle]
#                     [XLABEL label] [YLABEL label]
#                     DATASETS dataset1 [dataset2 [dataset3 [...]]]
#                     [OUTPUT path/to/file])
#
#   Creates a target which generates a chart displaying compile-time benchmarks.
#   After issuing this command, running the target named `target` will cause
#   each `dataset` to be generated and a HTML chart to be generated from those
#   datasets. Several aspects of the compilation can be displayed, such as
#   compilation time and executable size. The aspect being plotted on the
#   generated chart can be controlled via the `ASPECT` argument.
#
#   Parameters
#   ----------
#   target:
#       The name of the target associated to this chart.
#
#   [ALL]:
#       If `ALL` is provided, the target will be added to the default target.
#       This is the same behaviour as `add_custom_target` used with the `ALL`
#       keyword.
#
#   [ASPECT COMPILATION_TIME|LINK_TIME|EXECUTABLE_SIZE|PEAK_MEMORY]:
#       The aspect of the datasets to display on the chart. When this argument
#       is provided, the chart will adopt reasonable default values for the
#       axis labels and other similar settings. However, any setting set
#       explicitly using arguments like `TITLE` or `XLABEL` will have priority
#       over anything defaulted by the choice of an `ASPECT`. When no aspect
#       is provided, it defaults to `COMPILATION_TIME`.
#
#   [TITLE title]:
#       A title to use for the generated chart. If this is not provided, the
#       chart has no title.
#
#   [SUBTITLE subtitle]:
#       A subtitle to use for the generated chart. The subtitle will appear
#       just below the title, in a smaller font. If this is not provided, the
#       chart has no subtitle.
#
#   [XLABEL label]:
#       The label to use for the X axis.
#
#   [YLABEL label]:
#       The label to use for the Y axis.
#
#   DATASETS dataset1 [dataset2 [dataset3 [...]]]:
#       A list of datasets from which the benchmark is generated.
#
#   [OUTPUT path/to/file]:
#       The path of the resulting HTML file containing the chart. If a
#       relative path is used, the path will be considered as being relative
#       to the current CMake binary directory. This defaults to `target.html`,
#       so that the output file will be `target.html` in the current CMake
#       binary directory.
function(metabench_add_chart target)
    set(options ALL)
    set(one_value_args OUTPUT ASPECT TITLE SUBTITLE XLABEL YLABEL)
    set(multi_value_args DATASETS)
    cmake_parse_arguments(ARGS "${options}" "${one_value_args}" "${multi_value_args}" ${ARGN})

    if(NOT ARGS_ASPECT)
        set(ARGS_ASPECT "COMPILATION_TIME")
    endif()

    if(NOT ARGS_DATASETS)
        message(FATAL_ERROR "metabench_add_chart requires at least one dataset.")
    endif()

    if(NOT ARGS_OUTPUT)
        set(ARGS_OUTPUT "${target}.html")
    endif()

    if(NOT IS_ABSOLUTE "${ARGS_OUTPUT}")
        set(ARGS_OUTPUT "${CMAKE_CURRENT_BINARY_DIR}/${ARGS_OUTPUT}")
    endif()

    set(data)
    foreach(dataset ${ARGS_DATASETS})
        get_target_property(json_file ${dataset} METABENCH_DATASET_PATH)
        list(APPEND data "${json_file}")
    endforeach()

    add_custom_command(
        OUTPUT "${ARGS_OUTPUT}"
        COMMAND "${RUBY_EXECUTABLE}" -r erb -r json -r fileutils
            -e "options = {}"
            -e "options['TITLE'] = '${ARGS_TITLE}'"
            -e "options['SUBTITLE'] = '${ARGS_SUBTITLE}'"
            -e "options['XLABEL'] = '${ARGS_XLABEL}'"
            -e "options['YLABEL'] = '${ARGS_YLABEL}'"
            -e "aspect = '${ARGS_ASPECT}'"
            -e "data = '${data}'.split(';').map { |datum| JSON.parse(IO.read(datum)) }"
            -e "html = ERB.new(File.read('${CHART_HTML_ERB_PATH}')).result(binding)"
            -e "FileUtils.mkdir_p(File.dirname('${ARGS_OUTPUT}'))"
            -e "IO.write('${ARGS_OUTPUT}', html)"
        DEPENDS ${data} "${ARGS_CHART}"
        VERBATIM
    )

    if(${ARGS_ALL})
        add_custom_target(${target} ALL DEPENDS "${ARGS_OUTPUT}")
    else()
        add_custom_target(${target} DEPENDS "${ARGS_OUTPUT}")
    endif()
endfunction()

################################################################################
# Below this line are scripts and other files that are required by one part or
# another of Metabench. They are hardcoded here so that users only have to
# copy the `metabench.cmake` module to their project, without having to worry
# about dependencies and other implementation details.
################################################################################

################################################################################
# metabench.rb
#
# This script is used to drive CMake when gathering the measurements for a
# single dataset. If takes a `.cpp.erb` file, renders it with different values
# of `n` and then builds the associated executable for each value of `n`.
################################################################################
set(METABENCH_RB_PATH "${METABENCH_DIR}/metabench.rb")
file(WRITE "${METABENCH_RB_PATH}"
"require 'benchmark'                                                                                      \n"
"require 'erb'                                                                                            \n"
"require 'fileutils'                                                                                      \n"
"require 'open3'                                                                                          \n"
"require 'pathname'                                                                                       \n"
"require 'time'                                                                                           \n"
"                                                                                                         \n"
"                                                                                                         \n"
"def report_error(command_line, stdout, stderr, code)                                                     \n"
"  raise [%{\\ncommand line: #{command_line}},                                                            \n"
"         %{stdout\\n#{'-'*80}\\n#{stdout}},                                                              \n"
"         %{stderr\\n#{'-'*80}\\n#{stderr}},                                                              \n"
"         %{code\\n#{'-'*80}\\n#{code}}].join(%{\\n\\n})                                                  \n"
"end                                                                                                      \n"
"                                                                                                         \n"
"# Build the specified CMake target and return the measurements that were taken.                          \n"
"# The exact format of a measurement returned by this function is explained                               \n"
"# in the JavaScript code that loads it, in the chart template file below.                                \n"
"def build(target)                                                                                        \n"
"  command = ['${CMAKE_COMMAND}', '--build', '${CMAKE_BINARY_DIR}', '--target', target]                   \n"
"  exe_file = %{${METABENCH_DIR}/#{target}${CMAKE_EXECUTABLE_SUFFIX}}                                     \n"
"  cpp_file = %{${METABENCH_DIR}/#{target}.cpp}                                                           \n"
"                                                                                                         \n"
"  # We change the timestamp of the source file and remove the executable                                 \n"
"  # to make sure CMake considers the target as outdated; otherwise, we                                   \n"
"  # might skip the compilation and/or link steps. This is because CMake's                                \n"
"  # timestamps are not precise enough.                                                                   \n"
"  FileUtils.touch(cpp_file, mtime: Time.now+1)                                                           \n"
"  File.delete(exe_file) if File.exist?(exe_file)                                                         \n"
"                                                                                                         \n"
"  stdout, stderr, status = Open3.capture3(*command)                                                      \n"
"  compile_cli = stdout.match(/\\[compilation command: (.+)\\]/i)                                         \n"
"  compile_cli = compile_cli ? compile_cli.captures[0] : '(unavailable)'                                  \n"
"  link_cli = stdout.match(/\\[link command: (.+)\\]/i)                                                   \n"
"  link_cli = link_cli ? link_cli.captures[0] : '(unavailable)'                                           \n"
"                                                                                                         \n"
"  if not status.success?                                                                                 \n"
"    cli = %{compile: #{compile_cli}\\nlink: #{link_cli}}                                                 \n"
"    report_error(cli, stdout, stderr, IO.read(cpp_file))                                                 \n"
"  end                                                                                                    \n"
"                                                                                                         \n"
"  result = {}                                                                                            \n"
"                                                                                                         \n"
"  # Compilation and link times in seconds. They are output to stdout because                             \n"
"  # we use the `compile.rb` and `link.rb` scripts below to launch the compiler                           \n"
"  # and linker with CMake.                                                                               \n"
"  result['COMPILATION_TIME'] = stdout.match(/\\[COMPILATION_TIME: (.+)\\]/i).captures[0].to_f            \n"
"  result['LINK_TIME'] = stdout.match(/\\[LINK_TIME: (.+)\\]/i).captures[0].to_f                          \n"
"                                                                                                         \n"
"  # Peak memory usage                                                                                    \n"
"  result['PEAK_MEMORY'] = stdout.match(/\\[PEAK_MEMORY: (.+)\\]/i).captures[0].to_f                      \n"
"                                                                                                         \n"
"  # Size of the generated executable in KB                                                               \n"
"  result['EXECUTABLE_SIZE'] = File.size(exe_file).to_f / 1000                                            \n"
"                                                                                                         \n"
"  return result                                                                                          \n"
"end                                                                                                      \n"
"                                                                                                         \n"
"# Render the ERB template and return the generated code.                                                 \n"
"def render(erb_template, n, env)                                                                         \n"
"  begin                                                                                                  \n"
"    ERB.new(File.read(erb_template)).result(binding)                                                     \n"
"  rescue Exception => e                                                                                  \n"
"    $stderr.puts(%{\\nError while generating a C++ file from the ERB template #{erb_template}:\\n})      \n"
"    raise e                                                                                              \n"
"  end                                                                                                    \n"
"end                                                                                                      \n"
"                                                                                                         \n"
"# Formats the progress bar that we print while we run the benchmark                                      \n"
"def progress_bar(range, index, start_time, filename)                                                     \n"
"   n = range[index]                                                                                      \n"
"   percentage = (index+1) * 100 / range.size                                                             \n"
"   relative = filename.relative_path_from(Pathname.getwd)                                                \n"
"   elapsed = Time.now - start_time                                                                       \n"
"   return %{==> #{percentage}% (#{elapsed.round(2)}s) #{relative} (n = #{n})}                            \n"
"end                                                                                                      \n"
"                                                                                                         \n"
"# Returns an array of measurements representing the compilation of an ERB                                \n"
"# template for the values of `n` in the specified `range`.                                               \n"
"def measure(target, erb_template, range, env, repetitions)                                               \n"
"  erb_template = Pathname.new(erb_template)                                                              \n"
"  cpp_file = %{${METABENCH_DIR}/#{target}.cpp}                                                           \n"
"  data = []                                                                                              \n"
"  range = range.to_a                                                                                     \n"
"  start = Time.now                                                                                       \n"
"  $stderr.write(progress_bar(range, 0, start, erb_template)) # Setup the initial progress bar            \n"
"  range.each_with_index do |n, index|                                                                    \n"
"    compile = -> (code) {                                                                                \n"
"      IO.write(cpp_file, code)                                                                           \n"
"      return repetitions.times.map { build(target) }                                                     \n"
"    }                                                                                                    \n"
"    code = render(erb_template, n, env)                                                                  \n"
"    compile[code] if index == 0 # Fill the cache on the first iteration                                  \n"
"    base = compile[code]                                                                                 \n"
"    total = compile[%{#define METABENCH\\n} + code]                                                      \n"
"    data << {'n' => n, 'base' => base, 'total' => total}                                                 \n"
"    $stderr.write(%{\\r} + progress_bar(range, index, start, erb_template)) # Update the progress bar    \n"
"  end                                                                                                    \n"
"  return data                                                                                            \n"
"ensure                                                                                                   \n"
"  $stderr.puts # Otherwise the output of the next CMake command appears on the same line                 \n"
"  IO.write(cpp_file, '')                                                                                 \n"
"end                                                                                                      \n"
)
################################################################################
# end metabench.rb
################################################################################

################################################################################
# memusg.rb
#
# This script runs the command line given to it as an argument and monitors
# the peak memory usage in KB. It then prints that to standard output.
################################################################################
set(MEMUSG_RB_PATH "${METABENCH_DIR}/memusg.rb")
file(WRITE "${MEMUSG_RB_PATH}"
"module OS                                                                        \n"
"  def OS.windows?                                                                \n"
"    (/cygwin|mswin|mingw|bccwin|wince|emx/ =~ RUBY_PLATFORM) != nil              \n"
"  end                                                                            \n"
"  def OS.mac?                                                                    \n"
"   (/darwin/ =~ RUBY_PLATFORM) != nil                                            \n"
"  end                                                                            \n"
"  def OS.unix?                                                                   \n"
"    !OS.windows?                                                                 \n"
"  end                                                                            \n"
"  def OS.linux?                                                                  \n"
"    OS.unix? and not OS.mac?                                                     \n"
"  end                                                                            \n"
"end                                                                              \n"
"                                                                                 \n"
"if OS.mac?                                                                       \n"
"  def memusg(pgid)                                                               \n"
"    `/bin/ps -o rss= -g #{pgid}`.to_i                                            \n"
"  end                                                                            \n"
"elsif OS.linux?                                                                  \n"
"  def memusg(pgid)                                                               \n"
"    `/bin/ps -o rss= -#{pgid}`.to_i                                              \n"
"  end                                                                            \n"
"else                                                                             \n"
"  throw %{Unsupported platform #{RUBY_PLATFORM}}                                 \n"
"end                                                                              \n"
"                                                                                 \n"
"pid = Process.spawn(*ARGV)                                                       \n"
"Process.detach(pid) # Make sure the child process does not become a zombie       \n"
"peak = 0                                                                         \n"
"# Loop until getpgid throws ESRCH, which means the process is not alive anymore  \n"
"begin                                                                            \n"
"  while true                                                                     \n"
"    pgid = Process.getpgid(pid)                                                  \n"
"    peak = [peak, memusg(pgid)].max                                              \n"
"    sleep 0.01                                                                   \n"
"  end                                                                            \n"
"rescue Errno::ESRCH                                                              \n"
"end                                                                              \n"
"                                                                                 \n"
"puts %{[PEAK_MEMORY: #{peak}]}                                                   \n"
)
################################################################################
# end memusg.rb
################################################################################

################################################################################
# compile.rb
#
# This script is used to launch the compiler whilst measuring compilation time
# and other aspects.
################################################################################
set(COMPILE_RB_PATH "${METABENCH_DIR}/compile.rb")
file(WRITE "${COMPILE_RB_PATH}"
"require 'benchmark'                                                              \n"
"require 'open3'                                                                  \n"
"stdout = stderr = status = nil                                                   \n"
"command_line = ['${RUBY_EXECUTABLE}', '--', '${MEMUSG_RB_PATH}'] + ARGV          \n"
"time = Benchmark.measure {                                                       \n"
"  stdout, stderr, status = Open3.capture3(*command_line)                         \n"
"}.total                                                                          \n"
"$stdout.puts(stdout)                                                             \n"
"$stdout.puts(%{[compilation command: #{command_line.join(' ')}]})                \n"
"$stdout.puts(%{[COMPILATION_TIME: #{time}]})                                     \n"
"$stderr.puts(stderr)                                                             \n"
"exit(status.success?)                                                            \n"
)
################################################################################
# end compile.rb
################################################################################

################################################################################
# link.rb
#
# This script is used to launch the link and measure the link time.
################################################################################
set(LINK_RB_PATH "${METABENCH_DIR}/link.rb")
file(WRITE "${LINK_RB_PATH}"
"require 'benchmark'                                                              \n"
"require 'open3'                                                                  \n"
"stdout = stderr = status = nil                                                   \n"
"time = Benchmark.measure { stdout, stderr, status = Open3.capture3(*ARGV) }.total\n"
"$stdout.puts(stdout)                                                             \n"
"$stdout.puts(%{[link command: #{ARGV.join(' ')}]})                               \n"
"$stdout.puts(%{[LINK_TIME: #{time}]})                                            \n"
"$stderr.puts(stderr)                                                             \n"
"exit(status.success?)                                                            \n"
)
################################################################################
# end link.rb
################################################################################

################################################################################
# chart.html.erb
#
# The following is a ERB template for the HTML files used to visualize the
# benchmarks. The template is completed by filling it with the contents
# of the corresponding JSON file and the required JS libraries.
################################################################################
set(CHART_HTML_ERB_PATH "${METABENCH_DIR}/chart.html.erb")
file(WRITE "${CHART_HTML_ERB_PATH}"
"<!DOCTYPE html>                                                                                                            \n"
"<html>                                                                                                                     \n"
"  <head>                                                                                                                   \n"
"    <meta charset='utf-8'>                                                                                                 \n"
"    <style><%= IO.read('${METABENCH_DIR}/nvd3.css') %></style>                                                             \n"
"    <style>                                                                                                                \n"
"      * {                                                                                                                  \n"
"        box-sizing: border-box;                                                                                            \n"
"      }                                                                                                                    \n"
"      html, body {                                                                                                         \n"
"        position: relative;                                                                                                \n"
"        height: 100%;                                                                                                      \n"
"        width: 100%;                                                                                                       \n"
"        border: 0;                                                                                                         \n"
"        margin: 0;                                                                                                         \n"
"        padding: 0;                                                                                                        \n"
"        font-size: 0;                                                                                                      \n"
"      }                                                                                                                    \n"
"      svg g {                                                                                                              \n"
"        clip-path: none;                                                                                                   \n"
"      }                                                                                                                    \n"
"      svg text {                                                                                                           \n"
"        fill: #333;                                                                                                        \n"
"      }                                                                                                                    \n"
"      .nv-axislabel {                                                                                                      \n"
"        font-size: 16px !important;                                                                                        \n"
"      }                                                                                                                    \n"
"      .control {                                                                                                           \n"
"        cursor: pointer;                                                                                                   \n"
"        visibility: hidden;                                                                                                \n"
"        pointer-events: visible;                                                                                           \n"
"      }                                                                                                                    \n"
"      @media (min-width: 768px) {                                                                                          \n"
"        .control {                                                                                                         \n"
"          visibility: visible;                                                                                             \n"
"        }                                                                                                                  \n"
"      }                                                                                                                    \n"
"    </style>                                                                                                               \n"
"    <script><%= IO.read('${METABENCH_DIR}/d3.js') %></script>                                                              \n"
"    <script><%= IO.read('${METABENCH_DIR}/nvd3.js') %></script>                                                            \n"
"  </head>                                                                                                                  \n"
"  <body>                                                                                                                   \n"
"    <svg id='chart'></svg>                                                                                                 \n"
"    <script type='text/javascript'>                                                                                        \n"
"      // Simple helper to compute the median of a sequence of values                                                       \n"
"      var median = function(values) {                                                                                      \n"
"        values.sort(function(a, b) { return a - b; });                                                                     \n"
"        var half = Math.floor(values.length / 2);                                                                          \n"
"        if (values.length % 2)                                                                                             \n"
"          return values[half];                                                                                             \n"
"        else                                                                                                               \n"
"          return (values[half - 1] + values[half]) / 2.0;                                                                  \n"
"      };                                                                                                                   \n"
"                                                                                                                           \n"
"      var customSettings = <%= options.to_json %>;                                                                         \n"
"      var aspect = '<%= aspect %>';                                                                                        \n"
"      // 'data' is an array of datasets (i.e. curves), each of which is an object of the form                              \n"
"      // {                                                                                                                 \n"
"      //   key: <name of the curve>,                                                                                       \n"
"      //   color: <COLOR argument for that dataset (optional, may be the empty string)>,                                   \n"
"      //   scale: <SCALE argument for that dataset>,                                                                       \n"
"      //   values: [{                                                                                                      \n"
"      //       n: <value of n for that run>,                                                                               \n"
"      //       base: [{ // an array of samples for the same 'n'                                                            \n"
"      //           COMPILATION_TIME: <compilation time in seconds>,                                                        \n"
"      //           LINK_TIME:        <link time in seconds>,                                                               \n"
"      //           EXECUTABLE_SIZE:  <executable size in KB>,                                                              \n"
"      //           PEAK_MEMORY:      <peak memory usage in KB>                                                             \n"
"      //       }],                                                                                                         \n"
"      //       total: <same as 'base', but with METABENCH defined>                                                         \n"
"      //   }]                                                                                                              \n"
"      // }                                                                                                                 \n"
"      var data = <%= data.to_json %>;                                                                                      \n"
"                                                                                                                           \n"
"      // massage the measurements for ingestion by nvd3                                                                    \n"
"      data.forEach(function(dataset) {                                                                                     \n"
"        dataset.values = dataset.values.map(function(value) {                                                              \n"
"          new_value = {'n': value['n'], 'base': {}, 'total': {}};                                                          \n"
"          ['COMPILATION_TIME', 'LINK_TIME', 'EXECUTABLE_SIZE', 'PEAK_MEMORY'].forEach(function(aspect) {                   \n"
"            new_value['base'][aspect] = value.base.map(function(v){ return dataset.scale * v[aspect]; });                  \n"
"            new_value['total'][aspect] = value.total.map(function(v){ return dataset.scale * v[aspect]; });                \n"
"          });                                                                                                              \n"
"          return new_value;                                                                                                \n"
"        });                                                                                                                \n"
"      });                                                                                                                  \n"
"                                                                                                                           \n"
"      // setup per-aspect custom settings                                                                                  \n"
"      var aspectDefaults = {                                                                                               \n"
"        'COMPILATION_TIME': {                                                                                              \n"
"          axisLabel: 'Compilation time',                                                                                   \n"
"          tickFormat: function(val){ return d3.format('.2f')(val) + 's'; }                                                 \n"
"        },                                                                                                                 \n"
"        'EXECUTABLE_SIZE': {                                                                                               \n"
"          axisLabel: 'Executable size',                                                                                    \n"
"          tickFormat: function(val){ return d3.format('.0f')(val) + 'kb'; }                                                \n"
"        },                                                                                                                 \n"
"        'LINK_TIME': {                                                                                                     \n"
"          axisLabel: 'Link time',                                                                                          \n"
"          tickFormat: function(val){ return d3.format('.2f')(val) + 's'; }                                                 \n"
"        },                                                                                                                 \n"
"        'PEAK_MEMORY': {                                                                                                   \n"
"          axisLabel: 'Peak memory usage',                                                                                  \n"
"          tickFormat: function(val){ return d3.format('.0f')(val) + 'kb'; }                                                \n"
"        }                                                                                                                  \n"
"      };                                                                                                                   \n"
"                                                                                                                           \n"
"      // setup the chart and its options                                                                                   \n"
"      var chart = nv.models.lineChart()                                                                                    \n"
"                    .color(d3.scale.category10().range())                                                                  \n"
"                    .margin({left: 75, bottom: 50})                                                                        \n"
"                    .forceX([0]).forceY([0]);                                                                              \n"
"                                                                                                                           \n"
"      chart.x(function(datum){ return datum.n; })                                                                          \n"
"           .xAxis.options({                                                                                                \n"
"             axisLabel: customSettings.XLABEL || 'Number of elements',                                                     \n"
"             tickFormat: d3.format('.0f')                                                                                  \n"
"           });                                                                                                             \n"
"                                                                                                                           \n"
"      var subtract_baseline = function(datum) {                                                                            \n"
"        return Math.max(0, median(datum.total[aspect]) - median(datum.base[aspect]));                                      \n"
"      };                                                                                                                   \n"
"      chart.y(subtract_baseline)                                                                                           \n"
"           .yAxis.options({                                                                                                \n"
"             axisLabel: customSettings.YLABEL || aspectDefaults[aspect].axisLabel,                                         \n"
"             tickFormat: aspectDefaults[aspect].tickFormat                                                                 \n"
"           });                                                                                                             \n"
"                                                                                                                           \n"
"      chart.interpolate('cardinal').useInteractiveGuideline(true);                                                         \n"
"      d3.select('#chart').datum(data).call(chart);                                                                         \n"
"      var plot = d3.select('#chart > g');                                                                                  \n"
"                                                                                                                           \n"
"      // setup the title                                                                                                   \n"
"      plot.append('text')                                                                                                  \n"
"          .style('font-size', '24px')                                                                                      \n"
"          .attr('text-anchor', 'middle').attr('x', '50%').attr('y', '20px')                                                \n"
"          .text(customSettings.TITLE || '');                                                                               \n"
"                                                                                                                           \n"
"      // setup the subtitle                                                                                                \n"
"      plot.append('text')                                                                                                  \n"
"          .style('font-size', '16px')                                                                                      \n"
"          .attr('text-anchor', 'middle').attr('x', '50%').attr('y', '40px')                                                \n"
"          .text(customSettings.SUBTITLE || '');                                                                            \n"
"                                                                                                                           \n"
"      // setup interpolation control                                                                                       \n"
"      var controls = plot.select('.nv-x.nv-axis').append('g').attr('transform', 'translate(24,30)');                       \n"
"      var interpolate = controls.append('g').attr('class', 'control interpolate').datum([chart.interpolate()]);            \n"
"      interpolate.append('circle')                                                                                         \n"
"                 .style('fill', '#333').style('stroke', '#333').style('stroke-width', '2').style('fill-opacity', 1)        \n"
"                 .attr('cx', 0).attr('cy', 0).attr('r', 5);                                                                \n"
"      interpolate.append('text').attr('text-anchor', 'start').attr('dx', '8').attr('dy', '.32em').text('interpolate');     \n"
"      interpolate.on('click', function() {                                                                                 \n"
"        if (chart.interpolate() === interpolate.datum()[0]) {                                                              \n"
"          interpolate.select('circle').style('fill-opacity', 0);                                                           \n"
"          chart.interpolate('linear');                                                                                     \n"
"        } else {                                                                                                           \n"
"          d3.select(this).select('circle').style('fill-opacity', 1);                                                       \n"
"          chart.interpolate(interpolate.datum()[0]);                                                                       \n"
"        }                                                                                                                  \n"
"        chart.update();                                                                                                    \n"
"      });                                                                                                                  \n"
"                                                                                                                           \n"
"      // setup baseline control                                                                                            \n"
"      var baseline = controls.append('g').attr('class', 'control baseline').datum(true /* whether button is checked */);   \n"
"      baseline.attr('transform', 'translate(80,0)').append('circle')                                                       \n"
"              .style('fill', '#333').style('stroke', '#333').style('stroke-width', '2').style('fill-opacity', 1)           \n"
"              .attr('cx', 0).attr('cy', 0).attr('r', 5);                                                                   \n"
"      baseline.append('text').attr('text-anchor', 'start').attr('dx', '8').attr('dy', '.32em').text('subtract baseline');  \n"
"      baseline.on('click', function() {                                                                                    \n"
"        // We subtract the baseline from the total if the button was just checked, otherwise we use the total unchanged.   \n"
"        if (!baseline.datum()) {                                                                                           \n"
"          d3.select(this).select('circle').style('fill-opacity', 1);                                                       \n"
"          chart.y(subtract_baseline);                                                                                      \n"
"        } else {                                                                                                           \n"
"          baseline.select('circle').style('fill-opacity', 0);                                                              \n"
"          chart.y(function(datum) { return median(datum.total[aspect]); });                                                \n"
"        }                                                                                                                  \n"
"        baseline.datum(!baseline.datum()); // toggle the radio button                                                      \n"
"        chart.update();                                                                                                    \n"
"      });                                                                                                                  \n"
"                                                                                                                           \n"
"      // ensure the chart is responsive                                                                                    \n"
"      nv.utils.windowResize(chart.update);                                                                                 \n"
"    </script>                                                                                                              \n"
"  </body>                                                                                                                  \n"
"</html>                                                                                                                    \n"
)
################################################################################
# end chart.html.erb
################################################################################

################################################################################
# nvd3.css
#
# The following is a copy of the nvd3 1.8.5 css file.
# https://github.com/novus/nvd3
################################################################################
file(WRITE "${METABENCH_DIR}/nvd3.css" "\
.nvd3 .nv-axis line,.nvd3 .nv-axis path{fill:none;shape-rendering:crispEdges}.nv-brush .extent,.nvd3 .background path,.nvd3 .nv-axis line,.nvd3 .nv-axis path{shape-rendering:crispEdges}.nv-distx,.nv-disty,.nv-noninteractive,.nvd3 .nv-axis,.nvd3.nv-pie .nv-label,.nvd3.nv-sparklineplus g.nv-hoverValue{pointer-events:none}.nvd3 .nv-axis{opacity:1}.nvd3 .nv-axis.nv-disabled,.nvd3 .nv-controlsWrap .nv-legend .nv-check-box .nv-check{opacity:0}.nvd3 .nv-axis path{stroke:#000;stroke-opacity:.75}.nvd3 .nv-axis path.domain{stroke-opacity:.75}.nvd3 .nv-axis.nv-x path.domain{stroke-opacity:0}.nvd3 .nv-axis line{stroke:#e5e5e5}.nvd3 .nv-axis .zero line, .nvd3 .nv-axis line.zero{stroke-opacity:.75}.nvd3 .nv-axis .nv-axisMaxMin text{font-weight:700}.nvd3 .x .nv-axis .nv-axisMaxMin text,.nvd3 .x2 .nv-axis .nv-axisMaxMin text,.nvd3 .x3 .nv-axis .nv-axisMaxMin text{text-anchor:middle}.nvd3 .nv-bars rect{fill-opacity:.75;transition:fill-opacity 250ms linear}.nvd3 .nv-bars rect.hover{fill-opacity:1}.nvd3 .nv-bars .hover rect{fill:#add8e6}.nvd3 .nv-bars text{fill:transparent}.nvd3 .nv-bars .hover text{fill:rgba(0,0,0,1)}.nvd3 .nv-discretebar .nv-groups rect,.nvd3 .nv-multibar .nv-groups rect,.nvd3 .nv-multibarHorizontal .nv-groups rect{stroke-opacity:0;transition:fill-opacity 250ms linear}.with-transitions .nv-candlestickBar .nv-ticks .nv-tick,.with-transitions .nvd3 .nv-groups .nv-point{transition:stroke-width 250ms linear,stroke-opacity 250ms linear}.nvd3 .nv-candlestickBar .nv-ticks rect:hover,.nvd3 .nv-discretebar .nv-groups rect:hover,.nvd3 .nv-multibar .nv-groups rect:hover,.nvd3 .nv-multibarHorizontal .nv-groups rect:hover{fill-opacity:1}.nvd3 .nv-discretebar .nv-groups text,.nvd3 .nv-multibarHorizontal .nv-groups text{font-weight:700;fill:rgba(0,0,0,1);stroke:transparent}.nvd3 .nv-boxplot circle{fill-opacity:.5}.nvd3 .nv-boxplot circle:hover,.nvd3 .nv-boxplot rect:hover{fill-opacity:1}.nvd3 line.nv-boxplot-median{stroke:#000}.nv-boxplot-tick:hover{stroke-width:2.5px}.nvd3.nv-bullet{font:10px sans-serif}.nvd3.nv-bullet .nv-measure{fill-opacity:.8}.nvd3.nv-bullet \
.nv-measure:hover{fill-opacity:1}.nvd3.nv-bullet .nv-marker{stroke:#000;stroke-width:2px}.nvd3.nv-bullet .nv-markerTriangle{stroke:#000;fill:#fff;stroke-width:1.5px}.nvd3.nv-bullet .nv-markerLine{stroke:#000;stroke-width:1.5px}.nvd3.nv-bullet .nv-tick line{stroke:#666;stroke-width:.5px}.nvd3.nv-bullet .nv-range.nv-s0{fill:#eee}.nvd3.nv-bullet .nv-range.nv-s1{fill:#ddd}.nvd3.nv-bullet .nv-range.nv-s2{fill:#ccc}.nvd3.nv-bullet .nv-title{font-size:14px;font-weight:700}.nvd3.nv-bullet .nv-subtitle{fill:#999}.nvd3.nv-bullet .nv-range{fill:#bababa;fill-opacity:.4}.nvd3.nv-bullet .nv-range:hover{fill-opacity:.7}.nvd3.nv-candlestickBar .nv-ticks .nv-tick{stroke-width:1px}.nvd3.nv-candlestickBar .nv-ticks .nv-tick.hover{stroke-width:2px}.nvd3.nv-candlestickBar .nv-ticks .nv-tick.positive rect{stroke:#2ca02c;fill:#2ca02c}.nvd3.nv-candlestickBar .nv-ticks .nv-tick.negative rect{stroke:#d62728;fill:#d62728}.nvd3.nv-candlestickBar .nv-ticks line{stroke:#333}.nv-force-node{stroke:#fff;stroke-width:1.5px}.nv-force-link{stroke:#999;stroke-opacity:.6}.nv-force-node text{stroke-width:0}.nvd3 .nv-check-box .nv-box{fill-opacity:0;stroke-width:2}.nvd3 .nv-check-box .nv-check{fill-opacity:0;stroke-width:4}.nvd3 .nv-series.nv-disabled .nv-check-box .nv-check{fill-opacity:0;stroke-opacity:0}.nvd3.nv-linePlusBar .nv-bar rect{fill-opacity:.75}.nvd3.nv-linePlusBar .nv-bar rect:hover{fill-opacity:1}.nvd3 .nv-groups path.nv-line{fill:none}.nvd3 .nv-groups path.nv-area{stroke:none}.nvd3.nv-line .nvd3.nv-scatter .nv-groups .nv-point{fill-opacity:0;stroke-opacity:0}.nvd3.nv-scatter.nv-single-point .nv-groups .nv-point{fill-opacity:.5!important;stroke-opacity:.5!important}.nvd3 .nv-groups .nv-point.hover,.nvd3.nv-scatter .nv-groups .nv-point.hover{stroke-width:7px;fill-opacity:.95!important;stroke-opacity:.95!important}.nvd3 .nv-point-paths path{stroke:#aaa;stroke-opacity:0;fill:#eee;fill-opacity:0}.nvd3 .nv-indexLine{cursor:ew-resize}svg.nvd3-svg{-webkit-user-select:none;-moz-user-select:none;-ms-user-select:none;user-select:none;display:block;width:100%;\
height:100%}.nvtooltip.with-3d-shadow,.with-3d-shadow .nvtooltip{box-shadow:0 5px 10px rgba(0,0,0,.2);border-radius:5px}.nvd3 text{font:400 12px Arial,sans-serif}.nvd3 .title{font:700 14px Arial,sans-serif}.nvd3 .nv-background{fill:#fff;fill-opacity:0}.nvd3.nv-noData{font-size:18px;font-weight:700}.nv-brush .extent{fill-opacity:.125}.nv-brush .resize path{fill:#eee;stroke:#666}.nvd3 .nv-legend .nv-series{cursor:pointer}.nvd3 .nv-legend .nv-disabled circle{fill-opacity:0}.nvd3 .nv-brush .extent{fill-opacity:0!important}.nvd3 .nv-brushBackground rect{stroke:#000;stroke-width:.4;fill:#fff;fill-opacity:.7}\@media print{.nvd3 text{stroke-width:0;fill-opacity:1}}.nvd3.nv-ohlcBar .nv-ticks .nv-tick{stroke-width:1px}.nvd3.nv-ohlcBar .nv-ticks .nv-tick.hover{stroke-width:2px}.nvd3.nv-ohlcBar .nv-ticks .nv-tick.positive{stroke:#2ca02c}.nvd3.nv-ohlcBar .nv-ticks .nv-tick.negative{stroke:#d62728}.nvd3 .background path{fill:none;stroke:#EEE;stroke-opacity:.4}.nvd3 .foreground path{fill:none;stroke-opacity:.7}.nvd3 .nv-parallelCoordinates-brush .extent{fill:#fff;fill-opacity:.6;stroke:gray;shape-rendering:crispEdges}.nvd3 .nv-parallelCoordinates .hover{fill-opacity:1;stroke-width:3px}.nvd3 .missingValuesline line{fill:none;stroke:#000;stroke-width:1;stroke-opacity:1;stroke-dasharray:5,5}.nvd3.nv-pie .nv-pie-title{font-size:24px;fill:rgba(19,196,249,.59)}.nvd3.nv-pie .nv-slice text{stroke:#000;stroke-width:0}.nvd3.nv-pie path{transition:fill-opacity 250ms linear,stroke-width 250ms linear,stroke-opacity 250ms linear;stroke:#fff;stroke-width:1px;stroke-opacity:1;fill-opacity:.7}.nvd3.nv-pie .hover path{fill-opacity:1}.nvd3.nv-pie .nv-label rect{fill-opacity:0;stroke-opacity:0}.nvd3 .nv-groups .nv-point.hover{stroke-width:20px;stroke-opacity:.5}.nvd3 .nv-scatter .nv-point.hover{fill-opacity:1}.nvd3.nv-sparkline path{fill:none}.nvd3.nv-sparklineplus .nv-hoverValue line{stroke:#333;stroke-width:1.5px}.nvd3.nv-sparklineplus,.nvd3.nv-sparklineplus g{pointer-events:all}.nvd3 .nv-interactiveGuideLine,.nvtooltip{pointer-events:none}.nvd3 .nv-hoverArea{fill-opacity:0;\
stroke-opacity:0}.nvd3.nv-sparklineplus .nv-xValue,.nvd3.nv-sparklineplus .nv-yValue{stroke-width:0;font-size:.9em;font-weight:400}.nvd3.nv-sparklineplus .nv-yValue{stroke:#f66}.nvd3.nv-sparklineplus .nv-maxValue{stroke:#2ca02c;fill:#2ca02c}.nvd3.nv-sparklineplus .nv-minValue{stroke:#d62728;fill:#d62728}.nvd3.nv-sparklineplus .nv-currentValue{font-weight:700;font-size:1.1em}.nvtooltip h3,.nvtooltip table td.key{font-weight:400}.nvd3.nv-stackedarea path.nv-area{fill-opacity:.7;stroke-opacity:0;transition:fill-opacity 250ms linear,stroke-opacity 250ms linear}.nvd3.nv-stackedarea path.nv-area.hover{fill-opacity:.9}.nvd3.nv-stackedarea .nv-groups .nv-point{stroke-opacity:0;fill-opacity:0}.nvtooltip{position:absolute;color:rgba(0,0,0,1);padding:1px;z-index:10000;display:block;font-family:Arial,sans-serif;font-size:13px;text-align:left;white-space:nowrap;-webkit-user-select:none;-moz-user-select:none;-ms-user-select:none;user-select:none;background:rgba(255,255,255,.8);border:1px solid rgba(0,0,0,.5);border-radius:4px}.nvtooltip h3,.nvtooltip p{margin:0;text-align:center}.nvtooltip.with-transitions,.with-transitions .nvtooltip{transition:opacity 50ms linear;transition-delay:200ms}.nvtooltip.x-nvtooltip,.nvtooltip.y-nvtooltip{padding:8px}.nvtooltip h3{padding:4px 14px;line-height:18px;background-color:rgba(247,247,247,.75);color:rgba(0,0,0,1);border-bottom:1px solid #ebebeb;border-radius:5px 5px 0 0}.nvtooltip p{padding:5px 14px}.nvtooltip span{display:inline-block;margin:2px 0}.nvtooltip table{margin:6px;border-spacing:0}.nvtooltip table td{padding:2px 9px 2px 0;vertical-align:middle}.nvtooltip table td.key.total{font-weight:700}.nvtooltip table td.value{text-align:right;font-weight:700}.nvtooltip table td.percent{color:#a9a9a9}.nvtooltip table tr.highlight td{padding:1px 9px 1px 0;border-bottom-style:solid;border-bottom-width:1px;border-top-style:solid;border-top-width:1px}.nvtooltip table td.legend-color-guide div{vertical-align:middle;width:12px;height:12px;border:1px solid #999}.nvtooltip .footer{padding:3px;text-align:center}.nvtooltip-pending-removal{pointer-events:none;\
display:none}.nvd3 line.nv-guideline{stroke:#ccc}\
")
################################################################################
# end nvd3.css
################################################################################

################################################################################
# nvd3.js
#
# The following is a copy of the nvd3 1.8.5 js file. Note that we write it
# to the file in multiple steps, because CMake exhausts its stack memory when
# parsing too long strings.
# https://github.com/novus/nvd3
################################################################################
file(WRITE "${METABENCH_DIR}/nvd3.js" "\
/* nvd3 version 1.8.5 (https://github.com/novus/nvd3) 2016-12-01 */\
!function(){var a={};a.dev=!1,a.tooltip=a.tooltip||{},a.utils=a.utils||{},a.models=a.models||{},a.charts={},a.logs={},a.dom={},\"undefined\"!=typeof module&&\"undefined\"!=typeof exports&&\"undefined\"==typeof d3&&(d3=require(\"d3\")),a.dispatch=d3.dispatch(\"render_start\",\"render_end\"),Function.prototype.bind||(Function.prototype.bind=function(a){if(\"function\"!=typeof this)throw new TypeError(\"Function.prototype.bind - what is trying to be bound is not callable\");var b=Array.prototype.slice.call(arguments,1),c=this,d=function(){},e=function(){return c.apply(this instanceof d&&a?this:a,b.concat(Array.prototype.slice.call(arguments)))};return d.prototype=this.prototype,e.prototype=new d,e}),a.dev&&(a.dispatch.on(\"render_start\",function(b){a.logs.startTime=+new Date}),a.dispatch.on(\"render_end\",function(b){a.logs.endTime=+new Date,a.logs.totalTime=a.logs.endTime-a.logs.startTime,a.log(\"total\",a.logs.totalTime)})),a.log=function(){if(a.dev&&window.console&&console.log&&console.log.apply)console.log.apply(console,arguments);else if(a.dev&&window.console&&\"function\"==typeof console.log&&Function.prototype.bind){var b=Function.prototype.bind.call(console.log,console);b.apply(console,arguments)}return arguments[arguments.length-1]},a.deprecated=function(a,b){console&&console.warn&&console.warn(\"nvd3 warning: `\"+a+\"` has been deprecated. \",b||\"\")},a.render=function(b){b=b||1,a.render.active=!0,a.dispatch.render_start();var c=function(){for(var d,e,f=0;b>f&&(e=a.render.queue[f]);f++)d=e.generate(),typeof e.callback==typeof Function&&e.callback(d);a.render.queue.splice(0,f),a.render.queue.length?setTimeout(c):(a.dispatch.render_end(),a.render.active=!1)};setTimeout(c)},a.render.active=!1,a.render.queue=[],a.addGraph=function(b){typeof arguments[0]==typeof Function&&(b={generate:arguments[0],callback:arguments[1]}),a.render.queue.push(b),a.render.active||a.render()},\"undefined\"!=typeof module&&\"undefined\"!=typeof exports&&(module.exports=a),\"undefined\"!=typeof window&&(window.nv=a),a.dom.write=function(a){return void 0!==window.fastdom?fastdom.mutate(a):a()},\
a.dom.read=function(a){return void 0!==window.fastdom?fastdom.measure(a):a()},a.interactiveGuideline=function(){\"use strict\";function b(l){l.each(function(l){function m(){var a=d3.mouse(this),d=a[0],e=a[1],h=!0,i=!1;if(k&&(d=d3.event.offsetX,e=d3.event.offsetY,\"svg\"!==d3.event.target.tagName&&(h=!1),d3.event.target.className.baseVal.match(\"nv-legend\")&&(i=!0)),h&&(d-=c.left,e-=c.top),\"mouseout\"===d3.event.type||0>d||0>e||d>o||e>p||d3.event.relatedTarget&&void 0===d3.event.relatedTarget.ownerSVGElement||i){if(k&&d3.event.relatedTarget&&void 0===d3.event.relatedTarget.ownerSVGElement&&(void 0===d3.event.relatedTarget.className||d3.event.relatedTarget.className.match(j.nvPointerEventsClass)))return;return g.elementMouseout({mouseX:d,mouseY:e}),b.renderGuideLine(null),void j.hidden(!0)}j.hidden(!1);var l=\"function\"==typeof f.rangeBands,m=void 0;if(l){var n=d3.bisect(f.range(),d)-1;if(!(f.range()[n]+f.rangeBand()>=d))return g.elementMouseout({mouseX:d,mouseY:e}),b.renderGuideLine(null),void j.hidden(!0);m=f.domain()[d3.bisect(f.range(),d)-1]}else m=f.invert(d);g.elementMousemove({mouseX:d,mouseY:e,pointXValue:m}),\"dblclick\"===d3.event.type&&g.elementDblclick({mouseX:d,mouseY:e,pointXValue:m}),\"click\"===d3.event.type&&g.elementClick({mouseX:d,mouseY:e,pointXValue:m}),\"mousedown\"===d3.event.type&&g.elementMouseDown({mouseX:d,mouseY:e,pointXValue:m}),\"mouseup\"===d3.event.type&&g.elementMouseUp({mouseX:d,mouseY:e,pointXValue:m})}var n=d3.select(this),o=d||960,p=e||400,q=n.selectAll(\"g.nv-wrap.nv-interactiveLineLayer\").data([l]),r=q.enter().append(\"g\").attr(\"class\",\" nv-wrap nv-interactiveLineLayer\");r.append(\"g\").attr(\"class\",\"nv-interactiveGuideLine\"),i&&(i.on(\"touchmove\",m).on(\"mousemove\",m,!0).on(\"mouseout\",m,!0).on(\"mousedown\",m,!0).on(\"mouseup\",m,!0).on(\"dblclick\",m).on(\"click\",m),b.guideLine=null,b.renderGuideLine=function(c){h&&(b.guideLine&&b.guideLine.attr(\"x1\")===c||a.dom.write(function(){var b=q.select(\".nv-interactiveGuideLine\").selectAll(\"line\").data(null!=c?[a.utils.NaNtoZero(c)]:[],String);b.enter().append(\"line\").attr(\"class\",\
\"nv-guideline\").attr(\"x1\",function(a){return a}).attr(\"x2\",function(a){return a}).attr(\"y1\",p).attr(\"y2\",0),b.exit().remove()}))})})}var c={left:0,top:0},d=null,e=null,f=d3.scale.linear(),g=d3.dispatch(\"elementMousemove\",\"elementMouseout\",\"elementClick\",\"elementDblclick\",\"elementMouseDown\",\"elementMouseUp\"),h=!0,i=null,j=a.models.tooltip(),k=window.ActiveXObject;return j.duration(0).hideDelay(0).hidden(!1),b.dispatch=g,b.tooltip=j,b.margin=function(a){return arguments.length?(c.top=\"undefined\"!=typeof a.top?a.top:c.top,c.left=\"undefined\"!=typeof a.left?a.left:c.left,b):c},b.width=function(a){return arguments.length?(d=a,b):d},b.height=function(a){return arguments.length?(e=a,b):e},b.xScale=function(a){return arguments.length?(f=a,b):f},b.showGuideLine=function(a){return arguments.length?(h=a,b):h},b.svgContainer=function(a){return arguments.length?(i=a,b):i},b},a.interactiveBisect=function(a,b,c){\"use strict\";if(!(a instanceof Array))return null;var d;d=\"function\"!=typeof c?function(a){return a.x}:c;var e=function(a,b){return d(a)-b},f=d3.bisector(e).left,g=d3.max([0,f(a,b)-1]),h=d(a[g]);if(\"undefined\"==typeof h&&(h=g),h===b)return g;var i=d3.min([g+1,a.length-1]),j=d(a[i]);return\"undefined\"==typeof j&&(j=i),Math.abs(j-b)>=Math.abs(h-b)?g:i},a.nearestValueIndex=function(a,b,c){\"use strict\";var d=1/0,e=null;return a.forEach(function(a,f){var g=Math.abs(b-a);null!=a&&d>=g&&c>g&&(d=g,e=f)}),e},a.models.tooltip=function(){\"use strict\";function b(){if(!l||!l.node()){var a=[1];l=d3.select(document.body).select(\"#\"+d).data(a),l.enter().append(\"div\").attr(\"class\",\"nvtooltip \"+(i?i:\"xy-tooltip\")).attr(\"id\",d).style(\"top\",0).style(\"left\",0).style(\"opacity\",0).style(\"position\",\"fixed\").selectAll(\"div, table, td, tr\").classed(q,!0).classed(q,!0),l.exit().remove()}}function c(){return n&&w(e)?(a.dom.write(function(){b();var a=u(e);a&&(l.node().innerHTML=a),y()}),c):void 0}var d=\"nvtooltip-\"+Math.floor(1e5*Math.random()),e=null,f=\"w\",g=25,h=0,i=null,j=!0,k=200,l=null,m={left:null,top:null},n=!0,o=100,p=!0,q=\"nv-pointer-events-none\",\
r=function(a,b){return a},s=function(a){return a},t=function(a,b){return a},u=function(a){if(null===a)return\"\";var b=d3.select(document.createElement(\"table\"));if(p){var c=b.selectAll(\"thead\").data([a]).enter().append(\"thead\");c.append(\"tr\").append(\"td\").attr(\"colspan\",3).append(\"strong\").classed(\"x-value\",!0).html(s(a.value))}var d=b.selectAll(\"tbody\").data([a]).enter().append(\"tbody\"),e=d.selectAll(\"tr\").data(function(a){return a.series}).enter().append(\"tr\").classed(\"highlight\",function(a){return a.highlight});e.append(\"td\").classed(\"legend-color-guide\",!0).append(\"div\").style(\"background-color\",function(a){return a.color}),e.append(\"td\").classed(\"key\",!0).classed(\"total\",function(a){return!!a.total}).html(function(a,b){return t(a.key,b)}),e.append(\"td\").classed(\"value\",!0).html(function(a,b){return r(a.value,b)}),e.filter(function(a,b){return void 0!==a.percent}).append(\"td\").classed(\"percent\",!0).html(function(a,b){return\"(\"+d3.format(\"%\")(a.percent)+\")\"}),e.selectAll(\"td\").each(function(a){if(a.highlight){var b=d3.scale.linear().domain([0,1]).range([\"#fff\",a.color]),c=.6;d3.select(this).style(\"border-bottom-color\",b(c)).style(\"border-top-color\",b(c))}});var f=b.node().outerHTML;return void 0!==a.footer&&(f+=\"<div class='footer'>\"+a.footer+\"</div>\"),f},v=function(){var a={left:null!==d3.event?d3.event.clientX:0,top:null!==d3.event?d3.event.clientY:0};if(\"none\"!=getComputedStyle(document.body).transform){var b=document.body.getBoundingClientRect();a.left-=b.left,a.top-=b.top}return a},w=function(b){if(b&&b.series){if(a.utils.isArray(b.series))return!0;if(a.utils.isObject(b.series))return b.series=[b.series],!0}return!1},x=function(a){var b,c,d,e=l.node().offsetHeight,h=l.node().offsetWidth,i=document.documentElement.clientWidth,j=document.documentElement.clientHeight;switch(f){case\"e\":b=-h-g,c=-(e/2),a.left+b<0&&(b=g),(d=a.top+c)<0&&(c-=d),(d=a.top+c+e)>j&&(c-=d-j);break;case\"w\":b=g,c=-(e/2),a.left+b+h>i&&(b=-h-g),(d=a.top+c)<0&&(c-=d),(d=a.top+c+e)>j&&(c-=d-j);break;case\"n\":b=-(h/2)-5,c=g,a.top+c+e>j&&(c=-e-g),\
(d=a.left+b)<0&&(b-=d),(d=a.left+b+h)>i&&(b-=d-i);break;case\"s\":b=-(h/2),c=-e-g,a.top+c<0&&(c=g),(d=a.left+b)<0&&(b-=d),(d=a.left+b+h)>i&&(b-=d-i);break;case\"center\":b=-(h/2),c=-(e/2);break;default:b=0,c=0}return{left:b,top:c}},y=function(){a.dom.read(function(){var a=v(),b=x(a),c=a.left+b.left,d=a.top+b.top;if(j)l.interrupt().transition().delay(k).duration(0).style(\"opacity\",0);else{var e=\"translate(\"+m.left+\"px, \"+m.top+\"px)\",f=\"translate(\"+Math.round(c)+\"px, \"+Math.round(d)+\"px)\",g=d3.interpolateString(e,f),h=l.style(\"opacity\")<.1;l.interrupt().transition().duration(h?0:o).styleTween(\"transform\",function(a){return g},\"important\").styleTween(\"-webkit-transform\",function(a){return g}).style(\"-ms-transform\",f).style(\"opacity\",1)}m.left=c,m.top=d})};return c.nvPointerEventsClass=q,c.options=a.utils.optionsFunc.bind(c),c._options=Object.create({},{duration:{get:function(){return o},set:function(a){o=a}},gravity:{get:function(){return f},set:function(a){f=a}},distance:{get:function(){return g},set:function(a){g=a}},snapDistance:{get:function(){return h},set:function(a){h=a}},classes:{get:function(){return i},set:function(a){i=a}},enabled:{get:function(){return n},set:function(a){n=a}},hideDelay:{get:function(){return k},set:function(a){k=a}},contentGenerator:{get:function(){return u},set:function(a){u=a}},valueFormatter:{get:function(){return r},set:function(a){r=a}},headerFormatter:{get:function(){return s},set:function(a){s=a}},keyFormatter:{get:function(){return t},set:function(a){t=a}},headerEnabled:{get:function(){return p},set:function(a){p=a}},position:{get:function(){return v},set:function(a){v=a}},chartContainer:{get:function(){return document.body},set:function(b){a.deprecated(\"chartContainer\",\"feature removed after 1.8.3\")}},fixedTop:{get:function(){return null},set:function(b){a.deprecated(\"fixedTop\",\"feature removed after 1.8.1\")}},offset:{get:function(){return{left:0,top:0}},set:function(b){a.deprecated(\"offset\",\"use chart.tooltip.distance() instead\")}},hidden:{get:function(){return j},set:function(a){j!=a&&(j=!!a,\
c())}},data:{get:function(){return e},set:function(a){a.point&&(a.value=a.point.x,a.series=a.series||{},a.series.value=a.point.y,a.series.color=a.point.color||a.series.color),e=a}},node:{get:function(){return l.node()},set:function(a){}},id:{get:function(){return d},set:function(a){}}}),a.utils.initOptions(c),c},a.utils.windowSize=function(){var a={width:640,height:480};return window.innerWidth&&window.innerHeight?(a.width=window.innerWidth,a.height=window.innerHeight,a):\"CSS1Compat\"==document.compatMode&&document.documentElement&&document.documentElement.offsetWidth?(a.width=document.documentElement.offsetWidth,a.height=document.documentElement.offsetHeight,a):document.body&&document.body.offsetWidth?(a.width=document.body.offsetWidth,a.height=document.body.offsetHeight,a):a},a.utils.isArray=Array.isArray,a.utils.isObject=function(a){return null!==a&&\"object\"==typeof a},a.utils.isFunction=function(a){return\"function\"==typeof a},a.utils.isDate=function(a){return\"[object Date]\"===toString.call(a)},a.utils.isNumber=function(a){return!isNaN(a)&&\"number\"==typeof a},a.utils.windowResize=function(b){return window.addEventListener?window.addEventListener(\"resize\",b):a.log(\"ERROR: Failed to bind to window.resize with: \",b),{callback:b,clear:function(){window.removeEventListener(\"resize\",b)}}},a.utils.getColor=function(b){if(void 0===b)return a.utils.defaultColor();if(a.utils.isArray(b)){var c=d3.scale.ordinal().range(b);return function(a,b){var d=void 0===b?a:b;return a.color||c(d)}}return b},a.utils.defaultColor=function(){return a.utils.getColor(d3.scale.category20().range())},a.utils.customTheme=function(b,c,d){c=c||function(a){return a.key},d=d||d3.scale.category20().range();var e=d.length;return function(f,g){var h=c(f);return a.utils.isFunction(b[h])?b[h]():void 0!==b[h]?b[h]:(e||(e=d.length),e-=1,d[e])}},a.utils.pjax=function(b,c){var d=function(d){d3.html(d,function(d){var e=d3.select(c).node();e.parentNode.replaceChild(d3.select(d).select(c).node(),e),a.utils.pjax(b,c)})};d3.selectAll(b).on(\"click\",function(){history.pushState(this.href,\
this.textContent,this.href),d(this.href),d3.event.preventDefault()}),d3.select(window).on(\"popstate\",function(){d3.event.state&&d(d3.event.state)})},a.utils.calcApproxTextWidth=function(b){if(a.utils.isFunction(b.style)&&a.utils.isFunction(b.text)){var c=parseInt(b.style(\"font-size\").replace(\"px\",\"\"),10),d=b.text().length;return a.utils.NaNtoZero(d*c*.5)}return 0},a.utils.NaNtoZero=function(b){return!a.utils.isNumber(b)||isNaN(b)||null===b||b===1/0||b===-(1/0)?0:b},d3.selection.prototype.watchTransition=function(a){var b=[this].concat([].slice.call(arguments,1));return a.transition.apply(a,b)},a.utils.renderWatch=function(b,c){if(!(this instanceof a.utils.renderWatch))return new a.utils.renderWatch(b,c);var d=void 0!==c?c:250,e=[],f=this;this.models=function(a){return a=[].slice.call(arguments,0),a.forEach(function(a){a.__rendered=!1,function(a){a.dispatch.on(\"renderEnd\",function(b){a.__rendered=!0,f.renderEnd(\"model\")})}(a),e.indexOf(a)<0&&e.push(a)}),this},this.reset=function(a){void 0!==a&&(d=a),e=[]},this.transition=function(a,b,c){if(b=arguments.length>1?[].slice.call(arguments,1):[],c=b.length>1?b.pop():void 0!==d?d:250,a.__rendered=!1,e.indexOf(a)<0&&e.push(a),0===c)return a.__rendered=!0,a.delay=function(){return this},a.duration=function(){return this},a;0===a.length?a.__rendered=!0:a.every(function(a){return!a.length})?a.__rendered=!0:a.__rendered=!1;var g=0;return a.transition().duration(c).each(function(){++g}).each(\"end\",function(c,d){0===--g&&(a.__rendered=!0,f.renderEnd.apply(this,b))})},this.renderEnd=function(){e.every(function(a){return a.__rendered})&&(e.forEach(function(a){a.__rendered=!1}),b.renderEnd.apply(this,arguments))}},a.utils.deepExtend=function(b){var c=arguments.length>1?[].slice.call(arguments,1):[];c.forEach(function(c){for(var d in c){var e=a.utils.isArray(b[d]),f=a.utils.isObject(b[d]),g=a.utils.isObject(c[d]);f&&!e&&g?a.utils.deepExtend(b[d],c[d]):b[d]=c[d]}})},a.utils.state=function(){if(!(this instanceof a.utils.state))return new a.utils.state;var b={},c=function(){},d=function(){return{}},\
e=null,f=null;this.dispatch=d3.dispatch(\"change\",\"set\"),this.dispatch.on(\"set\",function(a){c(a,!0)}),this.getter=function(a){return d=a,this},this.setter=function(a,b){return b||(b=function(){}),c=function(c,d){a(c),d&&b()},this},this.init=function(b){e=e||{},a.utils.deepExtend(e,b)};var g=function(){var a=d();if(JSON.stringify(a)===JSON.stringify(b))return!1;for(var c in a)void 0===b[c]&&(b[c]={}),b[c]=a[c],f=!0;return!0};this.update=function(){e&&(c(e,!1),e=null),g.call(this)&&this.dispatch.change(b)}},a.utils.optionsFunc=function(b){return b&&d3.map(b).forEach(function(b,c){a.utils.isFunction(this[b])&&this[b](c)}.bind(this)),this},a.utils.calcTicksX=function(b,c){var d=1,e=0;for(e;e<c.length;e+=1){var f=c[e]&&c[e].values?c[e].values.length:0;d=f>d?f:d}return a.log(\"Requested number of ticks: \",b),a.log(\"Calculated max values to be: \",d),b=b>d?b=d-1:b,b=1>b?1:b,b=Math.floor(b),a.log(\"Calculating tick count as: \",b),b},a.utils.calcTicksY=function(b,c){return a.utils.calcTicksX(b,c)},a.utils.initOption=function(a,b){a._calls&&a._calls[b]?a[b]=a._calls[b]:(a[b]=function(c){return arguments.length?(a._overrides[b]=!0,a._options[b]=c,a):a._options[b]},a[\"_\"+b]=function(c){return arguments.length?(a._overrides[b]||(a._options[b]=c),a):a._options[b]})},a.utils.initOptions=function(b){b._overrides=b._overrides||{};var c=Object.getOwnPropertyNames(b._options||{}),d=Object.getOwnPropertyNames(b._calls||{});c=c.concat(d);for(var e in c)a.utils.initOption(b,c[e])},a.utils.inheritOptionsD3=function(a,b,c){a._d3options=c.concat(a._d3options||[]),c.unshift(b),c.unshift(a),d3.rebind.apply(this,c)},a.utils.arrayUnique=function(a){return a.sort().filter(function(b,c){return!c||b!=a[c-1]})},a.utils.symbolMap=d3.map(),a.utils.symbol=function(){function b(b,e){var f=c.call(this,b,e),g=d.call(this,b,e);return-1!==d3.svg.symbolTypes.indexOf(f)?d3.svg.symbol().type(f).size(g)():a.utils.symbolMap.get(f)(g)}var c,d=64;return b.type=function(a){return arguments.length?(c=d3.functor(a),b):c},b.size=function(a){return arguments.length?(d=d3.functor(a),\
b):d},b},a.utils.inheritOptions=function(b,c){var d=Object.getOwnPropertyNames(c._options||{}),e=Object.getOwnPropertyNames(c._calls||{}),f=c._inherited||[],g=c._d3options||[],h=d.concat(e).concat(f).concat(g);h.unshift(c),h.unshift(b),d3.rebind.apply(this,h),b._inherited=a.utils.arrayUnique(d.concat(e).concat(f).concat(d).concat(b._inherited||[])),b._d3options=a.utils.arrayUnique(g.concat(b._d3options||[]))},a.utils.initSVG=function(a){a.classed({\"nvd3-svg\":!0})},a.utils.sanitizeHeight=function(a,b){return a||parseInt(b.style(\"height\"),10)||400},a.utils.sanitizeWidth=function(a,b){return a||parseInt(b.style(\"width\"),10)||960},a.utils.availableHeight=function(b,c,d){return Math.max(0,a.utils.sanitizeHeight(b,c)-d.top-d.bottom)},a.utils.availableWidth=function(b,c,d){return Math.max(0,a.utils.sanitizeWidth(b,c)-d.left-d.right)},a.utils.noData=function(b,c){var d=b.options(),e=d.margin(),f=d.noData(),g=null==f?[\"No Data Available.\"]:[f],h=a.utils.availableHeight(null,c,e),i=a.utils.availableWidth(null,c,e),j=e.left+i/2,k=e.top+h/2;c.selectAll(\"g\").remove();var l=c.selectAll(\".nv-noData\").data(g);l.enter().append(\"text\").attr(\"class\",\"nvd3 nv-noData\").attr(\"dy\",\"-.7em\").style(\"text-anchor\",\"middle\"),l.attr(\"x\",j).attr(\"y\",k).text(function(a){return a})},a.utils.wrapTicks=function(a,b){a.each(function(){for(var a,c=d3.select(this),d=c.text().split(/\\s+/).reverse(),e=[],f=0,g=1.1,h=c.attr(\"y\"),i=parseFloat(c.attr(\"dy\")),j=c.text(null).append(\"tspan\").attr(\"x\",0).attr(\"y\",h).attr(\"dy\",i+\"em\");a=d.pop();)e.push(a),j.text(e.join(\" \")),j.node().getComputedTextLength()>b&&(e.pop(),j.text(e.join(\" \")),e=[a],j=c.append(\"tspan\").attr(\"x\",0).attr(\"y\",h).attr(\"dy\",++f*g+i+\"em\").text(a))})},a.utils.arrayEquals=function(b,c){if(b===c)return!0;if(!b||!c)return!1;if(b.length!=c.length)return!1;for(var d=0,e=b.length;e>d;d++)if(b[d]instanceof Array&&c[d]instanceof Array){if(!a.arrayEquals(b[d],c[d]))return!1}else if(b[d]!=c[d])return!1;return!0},a.models.axis=function(){\"use strict\";function b(g){return t.reset(),g.each(function(b){var \
g=d3.select(this);a.utils.initSVG(g);var q=g.selectAll(\"g.nv-wrap.nv-axis\").data([b]),r=q.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-axis\"),u=(r.append(\"g\"),q.select(\"g\"));null!==n?c.ticks(n):(\"top\"==c.orient()||\"bottom\"==c.orient())&&c.ticks(Math.abs(d.range()[1]-d.range()[0])/100),u.watchTransition(t,\"axis\").call(c),s=s||c.scale();var v=c.tickFormat();null==v&&(v=s.tickFormat());var w=u.selectAll(\"text.nv-axislabel\").data([h||null]);w.exit().remove(),void 0!==p&&u.selectAll(\"g\").select(\"text\").style(\"font-size\",p);var x,y,z;switch(c.orient()){case\"top\":w.enter().append(\"text\").attr(\"class\",\"nv-axislabel\"),z=0,1===d.range().length?z=m?2*d.range()[0]+d.rangeBand():0:2===d.range().length?z=m?d.range()[0]+d.range()[1]+d.rangeBand():d.range()[1]:d.range().length>2&&(z=d.range()[d.range().length-1]+(d.range()[1]-d.range()[0])),w.attr(\"text-anchor\",\"middle\").attr(\"y\",0).attr(\"x\",z/2),i&&(y=q.selectAll(\"g.nv-axisMaxMin\").data(d.domain()),y.enter().append(\"g\").attr(\"class\",function(a,b){return[\"nv-axisMaxMin\",\"nv-axisMaxMin-x\",0==b?\"nv-axisMin-x\":\"nv-axisMax-x\"].join(\" \")}).append(\"text\"),y.exit().remove(),y.attr(\"transform\",function(b,c){return\"translate(\"+a.utils.NaNtoZero(d(b))+\",0)\"}).select(\"text\").attr(\"dy\",\"-0.5em\").attr(\"y\",-c.tickPadding()).attr(\"text-anchor\",\"middle\").text(function(a,b){var c=v(a);return(\"\"+c).match(\"NaN\")?\"\":c}),y.watchTransition(t,\"min-max top\").attr(\"transform\",function(b,c){return\"translate(\"+a.utils.NaNtoZero(d.range()[c])+\",0)\"}));break;case\"bottom\":x=o+36;var A=30,B=0,C=u.selectAll(\"g\").select(\"text\"),D=\"\";if(j%360){C.attr(\"transform\",\"\"),C.each(function(a,b){var c=this.getBoundingClientRect(),d=c.width;B=c.height,d>A&&(A=d)}),D=\"rotate(\"+j+\" 0,\"+(B/2+c.tickPadding())+\")\";var E=Math.abs(Math.sin(j*Math.PI/180));x=(E?E*A:A)+30,C.attr(\"transform\",D).style(\"text-anchor\",j%360>0?\"start\":\"end\")}else l?C.attr(\"transform\",function(a,b){return\"translate(0,\"+(b%2==0?\"0\":\"12\")+\")\"}):C.attr(\"transform\",\"translate(0,0)\");w.enter().append(\"text\").attr(\"class\",\"nv-axislabel\"),z=0,1===d.range().length?z=m?2*d.range()[0]+d.rangeBand():0:2===d.range().length?z=m?d.range()[0]+d.range()[1]+d.rangeBand():d.range()[1]:d.range().length>2&&(z=d.range()[d.range().length-1]+(d.range()[1]-d.range()[0])),\
w.attr(\"text-anchor\",\"middle\").attr(\"y\",x).attr(\"x\",z/2),i&&(y=q.selectAll(\"g.nv-axisMaxMin\").data([d.domain()[0],d.domain()[d.domain().length-1]]),y.enter().append(\"g\").attr(\"class\",function(a,b){return[\"nv-axisMaxMin\",\"nv-axisMaxMin-x\",0==b?\"nv-axisMin-x\":\"nv-axisMax-x\"].join(\" \")}).append(\"text\"),y.exit().remove(),y.attr(\"transform\",function(b,c){return\"translate(\"+a.utils.NaNtoZero(d(b)+(m?d.rangeBand()/2:0))+\",0)\"}).select(\"text\").attr(\"dy\",\".71em\").attr(\"y\",c.tickPadding()).attr(\"transform\",D).style(\"text-anchor\",j?j%360>0?\"start\":\"end\":\"middle\").text(function(a,b){var c=v(a);return(\"\"+c).match(\"NaN\")?\"\":c}),y.watchTransition(t,\"min-max bottom\").attr(\"transform\",function(b,c){return\"translate(\"+a.utils.NaNtoZero(d(b)+(m?d.rangeBand()/2:0))+\",0)\"}));break;case\"right\":w.enter().append(\"text\").attr(\"class\",\"nv-axislabel\"),w.style(\"text-anchor\",k?\"middle\":\"begin\").attr(\"transform\",k?\"rotate(90)\":\"\").attr(\"y\",k?-Math.max(e.right,f)+12-(o||0):-10).attr(\"x\",k?d3.max(d.range())/2:c.tickPadding()),i&&(y=q.selectAll(\"g.nv-axisMaxMin\").data(d.domain()),y.enter().append(\"g\").attr(\"class\",function(a,b){return[\"nv-axisMaxMin\",\"nv-axisMaxMin-y\",0==b?\"nv-axisMin-y\":\"nv-axisMax-y\"].join(\" \")}).append(\"text\").style(\"opacity\",0),y.exit().remove(),y.attr(\"transform\",function(b,c){return\"translate(0,\"+a.utils.NaNtoZero(d(b))+\")\"}).select(\"text\").attr(\"dy\",\".32em\").attr(\"y\",0).attr(\"x\",c.tickPadding()).style(\"text-anchor\",\"start\").text(function(a,b){var c=v(a);return(\"\"+c).match(\"NaN\")?\"\":c}),y.watchTransition(t,\"min-max right\").attr(\"transform\",function(b,c){return\"translate(0,\"+a.utils.NaNtoZero(d.range()[c])+\")\"}).select(\"text\").style(\"opacity\",1));break;case\"left\":w.enter().append(\"text\").attr(\"class\",\"nv-axislabel\"),w.style(\"text-anchor\",k?\"middle\":\"end\").attr(\"transform\",k?\"rotate(-90)\":\"\").attr(\"y\",k?-Math.max(e.left,f)+25-(o||0):-10).attr(\"x\",k?-d3.max(d.range())/2:-c.tickPadding()),i&&(y=q.selectAll(\"g.nv-axisMaxMin\").data(d.domain()),y.enter().append(\"g\").attr(\"class\",function(a,b){return[\"nv-axisMaxMin\",\"nv-axisMaxMin-y\",\
0==b?\"nv-axisMin-y\":\"nv-axisMax-y\"].join(\" \")}).append(\"text\").style(\"opacity\",0),y.exit().remove(),y.attr(\"transform\",function(b,c){return\"translate(0,\"+a.utils.NaNtoZero(s(b))+\")\"}).select(\"text\").attr(\"dy\",\".32em\").attr(\"y\",0).attr(\"x\",-c.tickPadding()).attr(\"text-anchor\",\"end\").text(function(a,b){var c=v(a);return(\"\"+c).match(\"NaN\")?\"\":c}),y.watchTransition(t,\"min-max right\").attr(\"transform\",function(b,c){return\"translate(0,\"+a.utils.NaNtoZero(d.range()[c])+\")\"}).select(\"text\").style(\"opacity\",1))}if(w.text(function(a){return a}),!i||\"left\"!==c.orient()&&\"right\"!==c.orient()||(u.selectAll(\"g\").each(function(a,b){d3.select(this).select(\"text\").attr(\"opacity\",1),(d(a)<d.range()[1]+10||d(a)>d.range()[0]-10)&&((a>1e-10||-1e-10>a)&&d3.select(this).attr(\"opacity\",0),d3.select(this).select(\"text\").attr(\"opacity\",0))}),d.domain()[0]==d.domain()[1]&&0==d.domain()[0]&&q.selectAll(\"g.nv-axisMaxMin\").style(\"opacity\",function(a,b){return b?0:1})),i&&(\"top\"===c.orient()||\"bottom\"===c.orient())){var F=[];q.selectAll(\"g.nv-axisMaxMin\").each(function(a,b){try{b?F.push(d(a)-this.getBoundingClientRect().width-4):F.push(d(a)+this.getBoundingClientRect().width+4)}catch(c){b?F.push(d(a)-4):F.push(d(a)+4)}}),u.selectAll(\"g\").each(function(a,b){(d(a)<F[0]||d(a)>F[1])&&(a>1e-10||-1e-10>a?d3.select(this).remove():d3.select(this).select(\"text\").remove())})}u.selectAll(\".tick\").filter(function(a){return!parseFloat(Math.round(1e5*a)/1e6)&&void 0!==a}).classed(\"zero\",!0),s=d.copy()}),t.renderEnd(\"axis immediate\"),b}var c=d3.svg.axis(),d=d3.scale.linear(),e={top:0,right:0,bottom:0,left:0},f=75,g=60,h=null,i=!0,j=0,k=!0,l=!1,m=!1,n=null,o=0,p=void 0,q=250,r=d3.dispatch(\"renderEnd\");c.scale(d).orient(\"bottom\").tickFormat(function(a){return a});var s,t=a.utils.renderWatch(r,q);return b.axis=c,b.dispatch=r,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{axisLabelDistance:{get:function(){return o},set:function(a){o=a}},staggerLabels:{get:function(){return l},set:function(a){l=a}},rotateLabels:{get:function(){return j},set:function(a){j=a}},\
rotateYLabel:{get:function(){return k},set:function(a){k=a}},showMaxMin:{get:function(){return i},set:function(a){i=a}},axisLabel:{get:function(){return h},set:function(a){h=a}},height:{get:function(){return g},set:function(a){g=a}},ticks:{get:function(){return n},set:function(a){n=a}},width:{get:function(){return f},set:function(a){f=a}},fontSize:{get:function(){return p},set:function(a){p=a}},margin:{get:function(){return e},set:function(a){e.top=void 0!==a.top?a.top:e.top,e.right=void 0!==a.right?a.right:e.right,e.bottom=void 0!==a.bottom?a.bottom:e.bottom,e.left=void 0!==a.left?a.left:e.left}},duration:{get:function(){return q},set:function(a){q=a,t.reset(q)}},scale:{get:function(){return d},set:function(e){d=e,c.scale(d),m=\"function\"==typeof d.rangeBands,a.utils.inheritOptionsD3(b,d,[\"domain\",\"range\",\"rangeBand\",\"rangeBands\"])}}}),a.utils.initOptions(b),a.utils.inheritOptionsD3(b,c,[\"orient\",\"tickValues\",\"tickSubdivide\",\"tickSize\",\"tickPadding\",\"tickFormat\"]),a.utils.inheritOptionsD3(b,d,[\"domain\",\"range\",\"rangeBand\",\"rangeBands\"]),b},a.models.boxPlot=function(){\"use strict\";function b(l){return E.reset(),l.each(function(b){var l=j-i.left-i.right,F=k-i.top-i.bottom;A=d3.select(this),a.utils.initSVG(A),m.domain(c||b.map(function(a,b){return o(a,b)})).rangeBands(d||[0,l],.1);var G=[];if(!e){var H,I,J=[];b.forEach(function(a,b){var c=p(a),d=r(a),e=s(a),f=t(a),g=v(a);g&&g.forEach(function(a,b){J.push(w(a,b,void 0))}),e&&J.push(e),c&&J.push(c),d&&J.push(d),f&&J.push(f)}),H=d3.min(J),I=d3.max(J),G=[H,I]}n.domain(e||G),n.range(f||[F,0]),g=g||m,h=h||n.copy().range([n(0),n(0)]);var K=A.selectAll(\"g.nv-wrap\").data([b]);K.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap\");K.attr(\"transform\",\"translate(\"+i.left+\",\"+i.top+\")\");var L=K.selectAll(\".nv-boxplot\").data(function(a){return a}),M=L.enter().append(\"g\").style(\"stroke-opacity\",1e-6).style(\"fill-opacity\",1e-6);L.attr(\"class\",\"nv-boxplot\").attr(\"transform\",function(a,b,c){return\"translate(\"+(m(o(a,b))+.05*m.rangeBand())+\", 0)\"}).classed(\"hover\",function(a){return a.hover}),\
L.watchTransition(E,\"nv-boxplot: boxplots\").style(\"stroke-opacity\",1).style(\"fill-opacity\",.75).delay(function(a,c){return c*C/b.length}).attr(\"transform\",function(a,b){return\"translate(\"+(m(o(a,b))+.05*m.rangeBand())+\", 0)\"}),L.exit().remove(),M.each(function(a,b){var c=d3.select(this);[s,t].forEach(function(d){if(void 0!==d(a)&&null!==d(a)){var e=d===s?\"low\":\"high\";c.append(\"line\").style(\"stroke\",u(a)||z(a,b)).attr(\"class\",\"nv-boxplot-whisker nv-boxplot-\"+e),c.append(\"line\").style(\"stroke\",u(a)||z(a,b)).attr(\"class\",\"nv-boxplot-tick nv-boxplot-\"+e)}})});var N=function(){return null===D?.9*m.rangeBand():Math.min(75,.9*m.rangeBand())},O=function(){return.45*m.rangeBand()-N()/2},P=function(){return.45*m.rangeBand()+N()/2};[s,t].forEach(function(a){var b=a===s?\"low\":\"high\",c=a===s?p:r;L.select(\"line.nv-boxplot-whisker.nv-boxplot-\"+b).watchTransition(E,\"nv-boxplot: boxplots\").attr(\"x1\",.45*m.rangeBand()).attr(\"y1\",function(b,c){return n(a(b))}).attr(\"x2\",.45*m.rangeBand()).attr(\"y2\",function(a,b){return n(c(a))}),L.select(\"line.nv-boxplot-tick.nv-boxplot-\"+b).watchTransition(E,\"nv-boxplot: boxplots\").attr(\"x1\",O).attr(\"y1\",function(b,c){return n(a(b))}).attr(\"x2\",P).attr(\"y2\",function(b,c){return n(a(b))})}),[s,t].forEach(function(a){var b=a===s?\"low\":\"high\";M.selectAll(\".nv-boxplot-\"+b).on(\"mouseover\",function(b,c,d){d3.select(this).classed(\"hover\",!0),B.elementMouseover({series:{key:a(b),color:u(b)||z(b,d)},e:d3.event})}).on(\"mouseout\",function(b,c,d){d3.select(this).classed(\"hover\",!1),B.elementMouseout({series:{key:a(b),color:u(b)||z(b,d)},e:d3.event})}).on(\"mousemove\",function(a,b){B.elementMousemove({e:d3.event})})}),M.append(\"rect\").attr(\"class\",\"nv-boxplot-box\").on(\"mouseover\",function(a,b){d3.select(this).classed(\"hover\",!0),B.elementMouseover({key:o(a),value:o(a),series:[{key:\"Q3\",value:r(a),color:u(a)||z(a,b)},{key:\"Q2\",value:q(a),color:u(a)||z(a,b)},{key:\"Q1\",value:p(a),color:u(a)||z(a,b)}],data:a,index:b,e:d3.event})}).on(\"mouseout\",function(a,b){d3.select(this).classed(\"hover\",!1),B.elementMouseout({key:o(a),\
value:o(a),series:[{key:\"Q3\",value:r(a),color:u(a)||z(a,b)},{key:\"Q2\",value:q(a),color:u(a)||z(a,b)},{key:\"Q1\",value:p(a),color:u(a)||z(a,b)}],data:a,index:b,e:d3.event})}).on(\"mousemove\",function(a,b){B.elementMousemove({e:d3.event})}),L.select(\"rect.nv-boxplot-box\").watchTransition(E,\"nv-boxplot: boxes\").attr(\"y\",function(a,b){return n(r(a))}).attr(\"width\",N).attr(\"x\",O).attr(\"height\",function(a,b){return Math.abs(n(r(a))-n(p(a)))||1}).style(\"fill\",function(a,b){return u(a)||z(a,b)}).style(\"stroke\",function(a,b){return u(a)||z(a,b)}),M.append(\"line\").attr(\"class\",\"nv-boxplot-median\"),L.select(\"line.nv-boxplot-median\").watchTransition(E,\"nv-boxplot: boxplots line\").attr(\"x1\",O).attr(\"y1\",function(a,b){return n(q(a))}).attr(\"x2\",P).attr(\"y2\",function(a,b){return n(q(a))});var Q=L.selectAll(\".nv-boxplot-outlier\").data(function(a){return v(a)||[]});Q.enter().append(\"circle\").style(\"fill\",function(a,b,c){return y(a,b,c)||z(a,c)}).style(\"stroke\",function(a,b,c){return y(a,b,c)||z(a,c)}).style(\"z-index\",9e3).on(\"mouseover\",function(a,b,c){d3.select(this).classed(\"hover\",!0),B.elementMouseover({series:{key:x(a,b,c),color:y(a,b,c)||z(a,c)},e:d3.event})}).on(\"mouseout\",function(a,b,c){d3.select(this).classed(\"hover\",!1),B.elementMouseout({series:{key:x(a,b,c),color:y(a,b,c)||z(a,c)},e:d3.event})}).on(\"mousemove\",function(a,b){B.elementMousemove({e:d3.event})}),Q.attr(\"class\",\"nv-boxplot-outlier\"),Q.watchTransition(E,\"nv-boxplot: nv-boxplot-outlier\").attr(\"cx\",.45*m.rangeBand()).attr(\"cy\",function(a,b,c){return n(w(a,b,c))}).attr(\"r\",\"3\"),Q.exit().remove(),g=m.copy(),h=n.copy()}),E.renderEnd(\"nv-boxplot immediate\"),b}var c,d,e,f,g,h,i={top:0,right:0,bottom:0,left:0},j=960,k=500,l=Math.floor(1e4*Math.random()),m=d3.scale.ordinal(),n=d3.scale.linear(),o=function(a){return a.label},p=function(a){return a.values.Q1},q=function(a){return a.values.Q2},r=function(a){return a.values.Q3},s=function(a){return a.values.whisker_low},t=function(a){return a.values.whisker_high},u=function(a){return a.color},v=function(a){return a.values.outliers},\
w=function(a,b,c){return a},x=function(a,b,c){return a},y=function(a,b,c){return void 0},z=a.utils.defaultColor(),A=null,B=d3.dispatch(\"elementMouseover\",\"elementMouseout\",\"elementMousemove\",\"renderEnd\"),C=250,D=null,E=a.utils.renderWatch(B,C);return b.dispatch=B,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return j},set:function(a){j=a}},height:{get:function(){return k},set:function(a){k=a}},maxBoxWidth:{get:function(){return D},set:function(a){D=a}},x:{get:function(){return o},set:function(a){o=a}},q1:{get:function(){return p},set:function(a){p=a}},q2:{get:function(){return q},set:function(a){q=a}},q3:{get:function(){return r},set:function(a){r=a}},wl:{get:function(){return s},set:function(a){s=a}},wh:{get:function(){return t},set:function(a){t=a}},itemColor:{get:function(){return u},set:function(a){u=a}},outliers:{get:function(){return v},set:function(a){\
v=a}},outlierValue:{get:function(){return w},set:function(a){w=a}},outlierLabel:{get:function(){return x},set:function(a){x=a}},outlierColor:{get:function(){return y},set:function(a){y=a}},xScale:{get:function(){return m},set:function(a){m=a}},yScale:{get:function(){return n},set:function(a){n=a}},xDomain:{get:function(){return c},set:function(a){c=a}},yDomain:{get:function(){return e},set:function(a){e=a}},xRange:{get:function(){return d},set:function(a){d=a}},yRange:{get:function(){return f},set:function(a){f=a}},id:{get:function(){return l},set:function(a){l=a}},y:{get:function(){return console.warn(\"BoxPlot 'y' chart option is deprecated. Please use model overrides instead.\"),{}},set:function(a){console.warn(\"BoxPlot 'y' chart option is deprecated. Please use model overrides instead.\")}},margin:{get:function(){return i},set:function(a){i.top=void 0!==a.top?a.top:i.top,i.right=void 0!==a.right?a.right:i.right,i.bottom=void 0!==a.bottom?a.bottom:i.bottom,i.left=void 0!==a.left?a.left:i.left}},color:{get:function(){return z},set:function(b){z=a.utils.getColor(b)}},duration:{get:function(){return C},set:function(a){C=a,E.reset(C)}}}),a.utils.initOptions(b),b},a.models.boxPlotChart=function(){\"use strict\";function b(k){return t.reset(),t.models(e),l&&t.models(f),m&&t.models(g),k.each(function(k){var p=d3.select(this);a.utils.initSVG(p);var t=(i||parseInt(p.style(\"width\"))||960)-h.left-h.right,u=(j||parseInt(p.style(\"height\"))||400)-h.top-h.bottom;if(b.update=function(){r.beforeUpdate(),p.transition().duration(s).call(b)},b.container=this,!k||!k.length){var v=p.selectAll(\".nv-noData\").data([q]);return v.enter().append(\"text\").attr(\"class\",\"nvd3 nv-noData\").attr(\"dy\",\"-.7em\").style(\"text-anchor\",\"middle\"),v.attr(\"x\",h.left+t/2).attr(\"y\",h.top+u/2).text(function(a){return a}),b}p.selectAll(\".nv-noData\").remove(),c=e.xScale(),d=e.yScale().clamp(!0);var w=p.selectAll(\"g.nv-wrap.nv-boxPlotWithAxes\").data([k]),x=w.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-boxPlotWithAxes\").append(\"g\"),y=x.append(\"defs\"),z=w.select(\"g\");\
x.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),x.append(\"g\").attr(\"class\",\"nv-y nv-axis\").append(\"g\").attr(\"class\",\"nv-zeroLine\").append(\"line\"),x.append(\"g\").attr(\"class\",\"nv-barsWrap\"),z.attr(\"transform\",\"translate(\"+h.left+\",\"+h.top+\")\"),n&&z.select(\".nv-y.nv-axis\").attr(\"transform\",\"translate(\"+t+\",0)\"),e.width(t).height(u);var A=z.select(\".nv-barsWrap\").datum(k.filter(function(a){return!a.disabled}));if(A.transition().call(e),y.append(\"clipPath\").attr(\"id\",\"nv-x-label-clip-\"+e.id()).append(\"rect\"),z.select(\"#nv-x-label-clip-\"+e.id()+\" rect\").attr(\"width\",c.rangeBand()*(o?2:1)).attr(\"height\",16).attr(\"x\",-c.rangeBand()/(o?1:2)),l){f.scale(c).ticks(a.utils.calcTicksX(t/100,k)).tickSize(-u,0),z.select(\".nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+d.range()[0]+\")\"),z.select(\".nv-x.nv-axis\").call(f);var B=z.select(\".nv-x.nv-axis\").selectAll(\"g\");o&&B.selectAll(\"text\").attr(\"transform\",function(a,b,c){return\"translate(0,\"+(c%2===0?\"5\":\"17\")+\")\"})}m&&(g.scale(d).ticks(Math.floor(u/36)).tickSize(-t,0),z.select(\".nv-y.nv-axis\").call(g)),z.select(\".nv-zeroLine line\").attr(\"x1\",0).attr(\"x2\",t).attr(\"y1\",d(0)).attr(\"y2\",d(0))}),t.renderEnd(\"nv-boxplot chart immediate\"),b}var c,d,e=a.models.boxPlot(),f=a.models.axis(),g=a.models.axis(),h={top:15,right:10,bottom:50,left:60},i=null,j=null,k=a.utils.getColor(),l=!0,m=!0,n=!1,o=!1,p=a.models.tooltip(),q=\"No Data Available.\",r=d3.dispatch(\"beforeUpdate\",\"renderEnd\"),s=250;f.orient(\"bottom\").showMaxMin(!1).tickFormat(function(a){return a}),g.orient(n?\"right\":\"left\").tickFormat(d3.format(\",.1f\")),p.duration(0);var t=a.utils.renderWatch(r,s);return e.dispatch.on(\"elementMouseover.tooltip\",function(a){p.data(a).hidden(!1)}),e.dispatch.on(\"elementMouseout.tooltip\",function(a){p.data(a).hidden(!0)}),e.dispatch.on(\"elementMousemove.tooltip\",function(a){p()}),b.dispatch=r,b.boxplot=e,b.xAxis=f,b.yAxis=g,b.tooltip=p,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return i},set:function(a){i=a}},height:{get:function(){return j},set:function(a){j=a}},\
staggerLabels:{get:function(){return o},set:function(a){o=a}},showXAxis:{get:function(){return l},set:function(a){l=a}},showYAxis:{get:function(){return m},set:function(a){m=a}},tooltipContent:{get:function(){return p},set:function(a){p=a}},noData:{get:function(){return q},set:function(a){q=a}},margin:{get:function(){return h},set:function(a){h.top=void 0!==a.top?a.top:h.top,h.right=void 0!==a.right?a.right:h.right,h.bottom=void 0!==a.bottom?a.bottom:h.bottom,h.left=void 0!==a.left?a.left:h.left}},duration:{get:function(){return s},set:function(a){s=a,t.reset(s),e.duration(s),f.duration(s),g.duration(s)}},color:{get:function(){return k},set:function(b){k=a.utils.getColor(b),e.color(k)}},rightAlignYAxis:{get:function(){return n},set:function(a){n=a,g.orient(a?\"right\":\"left\")}}}),a.utils.inheritOptions(b,e),a.utils.initOptions(b),b},a.models.bullet=function(){\"use strict\";function b(a,b){var c=a.slice();a.sort(function(a,d){var e=c.indexOf(a),f=c.indexOf(d);return d3.descending(b[e],b[f])})}function c(e){return e.each(function(c,e){var s=p-d.left-d.right,y=q-d.top-d.bottom;r=d3.select(this),a.utils.initSVG(r);var z=g.call(this,c,e).slice(),A=h.call(this,c,e).slice(),B=i.call(this,c,e).slice(),C=j.call(this,c,e).slice(),D=k.call(this,c,e).slice(),E=l.call(this,c,e).slice(),F=m.call(this,c,e).slice(),G=n.call(this,c,e).slice();b(D,z),b(E,A),b(F,B),b(G,C),z.sort(d3.descending),A.sort(d3.descending),B.sort(d3.descending),C.sort(d3.descending);var H=d3.scale.linear().domain(d3.extent(d3.merge([o,z]))).range(f?[s,0]:[0,s]);this.__chart__||d3.scale.linear().domain([0,1/0]).range(H.range());this.__chart__=H;for(var I=(d3.min(z),d3.max(z),z[1],r.selectAll(\"g.nv-wrap.nv-bullet\").data([c])),J=I.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-bullet\"),K=J.append(\"g\"),L=I.select(\"g\"),e=0,M=z.length;M>e;e++){var N=\"nv-range nv-range\"+e;2>=e&&(N=N+\" nv-range\"+w[e]),K.append(\"rect\").attr(\"class\",N)}K.append(\"rect\").attr(\"class\",\"nv-measure\"),I.attr(\"transform\",\"translate(\"+d.left+\",\"+d.top+\")\");for(var O=function(a){return Math.abs(H(a)-H(0))},\
P=function(a){return H(0>a?a:0)},e=0,M=z.length;M>e;e++){var Q=z[e];L.select(\"rect.nv-range\"+e).datum(Q).attr(\"height\",y).transition().duration(x).attr(\"width\",O(Q)).attr(\"x\",P(Q))}L.select(\"rect.nv-measure\").style(\"fill\",t).attr(\"height\",y/3).attr(\"y\",y/3).on(\"mouseover\",function(){u.elementMouseover({value:C[0],label:G[0]||\"Current\",color:d3.select(this).style(\"fill\")})}).on(\"mousemove\",function(){u.elementMousemove({value:C[0],label:G[0]||\"Current\",color:d3.select(this).style(\"fill\")})}).on(\"mouseout\",function(){u.elementMouseout({value:C[0],label:G[0]||\"Current\",color:d3.select(this).style(\"fill\")})}).transition().duration(x).attr(\"width\",0>C?H(0)-H(C[0]):H(C[0])-H(0)).attr(\"x\",P(C));var R=y/6,S=A.map(function(a,b){return{value:a,label:E[b]}});K.selectAll(\"path.nv-markerTriangle\").data(S).enter().append(\"path\").attr(\"class\",\"nv-markerTriangle\").attr(\"d\",\"M0,\"+R+\"L\"+R+\",\"+-R+\" \"+-R+\",\"+-R+\"Z\").on(\"mouseover\",function(a){u.elementMouseover({value:a.value,label:a.label||\"Previous\",color:d3.select(this).style(\"fill\"),pos:[H(a.value),y/2]})}).on(\"mousemove\",function(a){u.elementMousemove({value:a.value,label:a.label||\"Previous\",color:d3.select(this).style(\"fill\")})}).on(\"mouseout\",function(a,b){u.elementMouseout({value:a.value,label:a.label||\"Previous\",color:d3.select(this).style(\"fill\")})}),L.selectAll(\"path.nv-markerTriangle\").data(S).transition().duration(x).attr(\"transform\",function(a){return\"translate(\"+H(a.value)+\",\"+y/2+\")\"});var T=B.map(function(a,b){return{value:a,label:F[b]}});K.selectAll(\"line.nv-markerLine\").data(T).enter().append(\"line\").attr(\"cursor\",\"\").attr(\"class\",\"nv-markerLine\").attr(\"x1\",function(a){return H(a.value)}).attr(\"y1\",\"2\").attr(\"x2\",function(a){return H(a.value)}).attr(\"y2\",y-2).on(\"mouseover\",function(a){u.elementMouseover({value:a.value,label:a.label||\"Previous\",color:d3.select(this).style(\"fill\"),pos:[H(a.value),y/2]})}).on(\"mousemove\",function(a){u.elementMousemove({value:a.value,label:a.label||\"Previous\",color:d3.select(this).style(\"fill\")})}).on(\"mouseout\",function(a,b){u.elementMouseout({value:a.value,\
label:a.label||\"Previous\",color:d3.select(this).style(\"fill\")})}),L.selectAll(\"line.nv-markerLine\").data(T).transition().duration(x).attr(\"x1\",function(a){return H(a.value)}).attr(\"x2\",function(a){return H(a.value)}),I.selectAll(\".nv-range\").on(\"mouseover\",function(a,b){var c=D[b]||v[b];u.elementMouseover({value:a,label:c,color:d3.select(this).style(\"fill\")})}).on(\"mousemove\",function(){u.elementMousemove({value:C[0],label:G[0]||\"Previous\",color:d3.select(this).style(\"fill\")})}).on(\"mouseout\",function(a,b){var c=D[b]||v[b];u.elementMouseout({value:a,label:c,color:d3.select(this).style(\"fill\")})})}),c}var d={top:0,right:0,bottom:0,left:0},e=\"left\",f=!1,g=function(a){return a.ranges},h=function(a){return a.markers?a.markers:[]},i=function(a){return a.markerLines?a.markerLines:[0]},j=function(a){return a.measures},k=function(a){return a.rangeLabels?a.rangeLabels:[]},l=function(a){return a.markerLabels?a.markerLabels:[]},m=function(a){return a.markerLineLabels?a.markerLineLabels:[]},n=function(a){return a.measureLabels?a.measureLabels:[]},o=[0],p=380,q=30,r=null,s=null,t=a.utils.getColor([\"#1f77b4\"]),u=d3.dispatch(\"elementMouseover\",\"elementMouseout\",\"elementMousemove\"),v=[\"Maximum\",\"Mean\",\"Minimum\"],w=[\"Max\",\"Avg\",\"Min\"],x=1e3;return c.dispatch=u,c.options=a.utils.optionsFunc.bind(c),c._options=Object.create({},{ranges:{get:function(){return g},set:function(a){g=a}},markers:{get:function(){return h},set:function(a){h=a}},measures:{get:function(){return j},set:function(a){j=a}},forceX:{get:function(){return o},set:function(a){o=a}},width:{get:function(){return p},set:function(a){p=a}},height:{get:function(){return q},set:function(a){q=a}},tickFormat:{get:function(){return s},set:function(a){s=a}},duration:{get:function(){return x},set:function(a){x=a}},margin:{get:function(){return d},set:function(a){d.top=void 0!==a.top?a.top:d.top,d.right=void 0!==a.right?a.right:d.right,d.bottom=void 0!==a.bottom?a.bottom:d.bottom,d.left=void 0!==a.left?a.left:d.left}},orient:{get:function(){return e},set:function(a){e=a,f=\"right\"==e||\"bottom\"==e}},\
color:{get:function(){return t},set:function(b){t=a.utils.getColor(b)}}}),a.utils.initOptions(c),c},a.models.bulletChart=function(){\"use strict\";function b(d){return d.each(function(e,o){var p=d3.select(this);a.utils.initSVG(p);var q=a.utils.availableWidth(k,p,g),r=l-g.top-g.bottom;if(b.update=function(){b(d)},b.container=this,!e||!h.call(this,e,o))return a.utils.noData(b,p),b;p.selectAll(\".nv-noData\").remove();var s=h.call(this,e,o).slice().sort(d3.descending),t=i.call(this,e,o).slice().sort(d3.descending),u=j.call(this,e,o).slice().sort(d3.descending),v=p.selectAll(\"g.nv-wrap.nv-bulletChart\").data([e]),w=v.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-bulletChart\"),x=w.append(\"g\"),y=v.select(\"g\");x.append(\"g\").attr(\"class\",\"nv-bulletWrap\"),x.append(\"g\").attr(\"class\",\"nv-titles\"),v.attr(\"transform\",\"translate(\"+g.left+\",\"+g.top+\")\");var z=d3.scale.linear().domain([0,Math.max(s[0],t[0]||0,u[0])]).range(f?[q,0]:[0,q]),A=this.__chart__||d3.scale.linear().domain([0,1/0]).range(z.range());this.__chart__=z;var B=x.select(\".nv-titles\").append(\"g\").attr(\"text-anchor\",\"end\").attr(\"transform\",\"translate(-6,\"+(l-g.top-g.bottom)/2+\")\");B.append(\"text\").attr(\"class\",\"nv-title\").text(function(a){return a.title}),B.append(\"text\").attr(\"class\",\"nv-subtitle\").attr(\"dy\",\"1em\").text(function(a){return a.subtitle}),c.width(q).height(r);var C=y.select(\".nv-bulletWrap\");d3.transition(C).call(c);var D=m||z.tickFormat(q/100),E=y.selectAll(\"g.nv-tick\").data(z.ticks(n?n:q/50),function(a){return this.textContent||D(a)}),F=E.enter().append(\"g\").attr(\"class\",\"nv-tick\").attr(\"transform\",function(a){return\"translate(\"+A(a)+\",0)\"}).style(\"opacity\",1e-6);F.append(\"line\").attr(\"y1\",r).attr(\"y2\",7*r/6),F.append(\"text\").attr(\"text-anchor\",\"middle\").attr(\"dy\",\"1em\").attr(\"y\",7*r/6).text(D);var G=d3.transition(E).transition().duration(c.duration()).attr(\"transform\",function(a){return\"translate(\"+z(a)+\",0)\"}).style(\"opacity\",1);G.select(\"line\").attr(\"y1\",r).attr(\"y2\",7*r/6),G.select(\"text\").attr(\"y\",7*r/6),d3.transition(E.exit()).transition().duration(c.duration()).attr(\"transform\",\
function(a){return\"translate(\"+z(a)+\",0)\"}).style(\"opacity\",1e-6).remove()}),d3.timer.flush(),b}var c=a.models.bullet(),d=a.models.tooltip(),e=\"left\",f=!1,g={top:5,right:40,bottom:20,left:120},h=function(a){return a.ranges},i=function(a){return a.markers?a.markers:[]},j=function(a){return a.measures},k=null,l=55,m=null,n=null,o=null,p=d3.dispatch();return d.duration(0).headerEnabled(!1),c.dispatch.on(\"elementMouseover.tooltip\",function(a){a.series={key:a.label,value:a.value,color:a.color},d.data(a).hidden(!1)}),c.dispatch.on(\"elementMouseout.tooltip\",function(a){d.hidden(!0)}),c.dispatch.on(\"elementMousemove.tooltip\",function(a){d()}),b.bullet=c,b.dispatch=p,b.tooltip=d,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{ranges:{get:function(){return h},set:function(a){h=a}},markers:{get:function(){return i},set:function(a){i=a}},measures:{get:function(){return j},set:function(a){j=a}},width:{get:function(){return k},set:function(a){k=a}},height:{get:function(){return l},set:function(a){l=a}},tickFormat:{get:function(){return m},set:function(a){m=a}},ticks:{get:function(){return n},set:function(a){n=a}},noData:{get:function(){return o},set:function(a){o=a}},margin:{get:function(){return g},set:function(a){g.top=void 0!==a.top?a.top:g.top,g.right=void 0!==a.right?a.right:g.right,g.bottom=void 0!==a.bottom?a.bottom:g.bottom,g.left=void 0!==a.left?a.left:g.left}},orient:{get:function(){return e},set:function(a){e=a,f=\"right\"==e||\"bottom\"==e}}}),a.utils.inheritOptions(b,c),a.utils.initOptions(b),b},a.models.candlestickBar=function(){\"use strict\";function b(x){return x.each(function(b){c=d3.select(this);var x=a.utils.availableWidth(i,c,h),y=a.utils.availableHeight(j,c,h);a.utils.initSVG(c);var A=x/b[0].values.length*.45;l.domain(d||d3.extent(b[0].values.map(n).concat(t))),v?l.range(f||[.5*x/b[0].values.length,x*(b[0].values.length-.5)/b[0].values.length]):l.range(f||[5+A/2,x-A/2-5]),m.domain(e||[d3.min(b[0].values.map(s).concat(u)),d3.max(b[0].values.map(r).concat(u))]).range(g||[y,0]),l.domain()[0]===l.domain()[1]&&(l.domain()[0]?l.domain([l.domain()[0]-.01*l.domain()[0],\
l.domain()[1]+.01*l.domain()[1]]):l.domain([-1,1])),m.domain()[0]===m.domain()[1]&&(m.domain()[0]?m.domain([m.domain()[0]+.01*m.domain()[0],m.domain()[1]-.01*m.domain()[1]]):m.domain([-1,1]));var B=d3.select(this).selectAll(\"g.nv-wrap.nv-candlestickBar\").data([b[0].values]),C=B.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-candlestickBar\"),D=C.append(\"defs\"),E=C.append(\"g\"),F=B.select(\"g\");E.append(\"g\").attr(\"class\",\"nv-ticks\"),B.attr(\"transform\",\"translate(\"+h.left+\",\"+h.top+\")\"),c.on(\"click\",function(a,b){z.chartClick({data:a,index:b,pos:d3.event,id:k})}),D.append(\"clipPath\").attr(\"id\",\"nv-chart-clip-path-\"+k).append(\"rect\"),B.select(\"#nv-chart-clip-path-\"+k+\" rect\").attr(\"width\",x).attr(\"height\",y),F.attr(\"clip-path\",w?\"url(#nv-chart-clip-path-\"+k+\")\":\"\");var G=B.select(\".nv-ticks\").selectAll(\".nv-tick\").data(function(a){return a});G.exit().remove();var H=G.enter().append(\"g\");G.attr(\"class\",function(a,b,c){return(p(a,b)>q(a,b)?\"nv-tick negative\":\"nv-tick positive\")+\" nv-tick-\"+c+\"-\"+b});H.append(\"line\").attr(\"class\",\"nv-candlestick-lines\").attr(\"transform\",function(a,b){return\"translate(\"+l(n(a,b))+\",0)\"}).attr(\"x1\",0).attr(\"y1\",function(a,b){return m(r(a,b))}).attr(\"x2\",0).attr(\"y2\",function(a,b){return m(s(a,b))}),H.append(\"rect\").attr(\"class\",\"nv-candlestick-rects nv-bars\").attr(\"transform\",function(a,b){return\"translate(\"+(l(n(a,b))-A/2)+\",\"+(m(o(a,b))-(p(a,b)>q(a,b)?m(q(a,b))-m(p(a,b)):0))+\")\"}).attr(\"x\",0).attr(\"y\",0).attr(\"width\",A).attr(\"height\",function(a,b){var c=p(a,b),d=q(a,b);return c>d?m(d)-m(c):m(c)-m(d)});G.select(\".nv-candlestick-lines\").transition().attr(\"transform\",function(a,b){return\"translate(\"+l(n(a,b))+\",0)\"}).attr(\"x1\",0).attr(\"y1\",function(a,b){return m(r(a,b))}).attr(\"x2\",0).attr(\"y2\",function(a,b){return m(s(a,b))}),G.select(\".nv-candlestick-rects\").transition().attr(\"transform\",function(a,b){return\"translate(\"+(l(n(a,b))-A/2)+\",\"+(m(o(a,b))-(p(a,b)>q(a,b)?m(q(a,b))-m(p(a,b)):0))+\")\"}).attr(\"x\",0).attr(\"y\",0).attr(\"width\",A).attr(\"height\",function(a,b){var c=p(a,b),d=q(a,b);return \
c>d?m(d)-m(c):m(c)-m(d)})}),b}var c,d,e,f,g,h={top:0,right:0,bottom:0,left:0},i=null,j=null,k=Math.floor(1e4*Math.random()),l=d3.scale.linear(),m=d3.scale.linear(),n=function(a){return a.x},o=function(a){return a.y},p=function(a){return a.open},q=function(a){return a.close},r=function(a){return a.high},s=function(a){return a.low},t=[],u=[],v=!1,w=!0,x=a.utils.defaultColor(),y=!1,z=d3.dispatch(\"stateChange\",\"changeState\",\"renderEnd\",\"chartClick\",\"elementClick\",\"elementDblClick\",\"elementMouseover\",\"elementMouseout\",\"elementMousemove\");return b.highlightPoint=function(a,d){b.clearHighlights(),c.select(\".nv-candlestickBar .nv-tick-0-\"+a).classed(\"hover\",d)},b.clearHighlights=function(){c.select(\".nv-candlestickBar .nv-tick.hover\").classed(\"hover\",!1)},b.dispatch=z,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return i},set:function(a){i=a}},height:{get:function(){return j},set:function(a){j=a}},xScale:{get:function(){return l},set:function(a){l=a}},yScale:{get:function(){return m},set:function(a){m=a}},xDomain:{get:function(){return d},set:function(a){d=a}},yDomain:{get:function(){return e},set:function(a){e=a}},xRange:{get:function(){return f},set:function(a){f=a}},yRange:{get:function(){return g},set:function(a){g=a}},forceX:{get:function(){return t},set:function(a){t=a}},forceY:{get:function(){return u},set:function(a){u=a}},padData:{get:function(){return v},set:function(a){v=a}},clipEdge:{get:function(){return w},set:function(a){w=a}},id:{get:function(){return k},set:function(a){k=a}},interactive:{get:function(){return y},set:function(a){y=a}},x:{get:function(){return n},set:function(a){n=a}},y:{get:function(){return o},set:function(a){o=a}},open:{get:function(){return p()},set:function(a){p=a}},close:{get:function(){return q()},set:function(a){q=a}},high:{get:function(){return r},set:function(a){r=a}},low:{get:function(){return s},set:function(a){s=a}},margin:{get:function(){return h},set:function(a){h.top=void 0!=a.top?a.top:h.top,h.right=void 0!=a.right?a.right:h.right,\
h.bottom=void 0!=a.bottom?a.bottom:h.bottom,h.left=void 0!=a.left?a.left:h.left}},color:{get:function(){return x},set:function(b){x=a.utils.getColor(b)}}}),a.utils.initOptions(b),b},a.models.cumulativeLineChart=function(){\"use strict\";function b(l){return I.reset(),I.models(f),s&&I.models(g),t&&I.models(h),l.each(function(l){function B(a,c){d3.select(b.container).style(\"cursor\",\"ew-resize\")}function F(a,b){H.x=d3.event.x,H.i=Math.round(G.invert(H.x)),L()}function I(a,c){d3.select(b.container).style(\"cursor\",\"auto\"),z.index=H.i,D.stateChange(z)}function L(){ba.data([H]);var a=b.duration();b.duration(0),b.update(),b.duration(a)}var M=d3.select(this);a.utils.initSVG(M),M.classed(\"nv-chart-\"+y,!0);var N=a.utils.availableWidth(p,M,m),O=a.utils.availableHeight(q,M,m);if(b.update=function(){0===E?M.call(b):M.transition().duration(E).call(b)},b.container=this,z.setter(K(l),b.update).getter(J(l)).update(),z.disabled=l.map(function(a){return!!a.disabled}),!A){var P;A={};for(P in z)z[P]instanceof Array?A[P]=z[P].slice(0):A[P]=z[P]}var Q=d3.behavior.drag().on(\"dragstart\",B).on(\"drag\",F).on(\"dragend\",I);if(!(l&&l.length&&l.filter(function(a){return a.values.length}).length))return a.utils.noData(b,M),b;if(M.selectAll(\".nv-noData\").remove(),d=f.xScale(),e=f.yScale(),x)f.yDomain(null);else{var R=l.filter(function(a){return!a.disabled}).map(function(a,b){var c=d3.extent(a.values,f.y());return c[0]<-.95&&(c[0]=-.95),[(c[0]-c[1])/(1+c[1]),(c[1]-c[0])/(1+c[0])]}),S=[d3.min(R,function(a){return a[0]}),d3.max(R,function(a){return a[1]})];f.yDomain(S)}G.domain([0,l[0].values.length-1]).range([0,N]).clamp(!0);var l=c(H.i,l),T=w?\"none\":\"all\",U=M.selectAll(\"g.nv-wrap.nv-cumulativeLine\").data([l]),V=U.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-cumulativeLine\").append(\"g\"),W=U.select(\"g\");if(V.append(\"g\").attr(\"class\",\"nv-interactive\"),V.append(\"g\").attr(\"class\",\"nv-x nv-axis\").style(\"pointer-events\",\"none\"),V.append(\"g\").attr(\"class\",\"nv-y nv-axis\"),V.append(\"g\").attr(\"class\",\"nv-background\"),V.append(\"g\").attr(\"class\",\"nv-linesWrap\").style(\"pointer-events\",\
T),V.append(\"g\").attr(\"class\",\"nv-avgLinesWrap\").style(\"pointer-events\",\"none\"),V.append(\"g\").attr(\"class\",\"nv-legendWrap\"),V.append(\"g\").attr(\"class\",\"nv-controlsWrap\"),r?(i.width(N),W.select(\".nv-legendWrap\").datum(l).call(i),n||i.height()===m.top||(m.top=i.height(),O=a.utils.availableHeight(q,M,m)),W.select(\".nv-legendWrap\").attr(\"transform\",\"translate(0,\"+-m.top+\")\")):W.select(\".nv-legendWrap\").selectAll(\"*\").remove(),v){var X=[{key:\"Re-scale y-axis\",disabled:!x}];j.width(140).color([\"#444\",\"#444\",\"#444\"]).rightAlign(!1).margin({top:5,right:0,bottom:5,left:20}),W.select(\".nv-controlsWrap\").datum(X).attr(\"transform\",\"translate(0,\"+-m.top+\")\").call(j)}else W.select(\".nv-controlsWrap\").selectAll(\"*\").remove();U.attr(\"transform\",\"translate(\"+m.left+\",\"+m.top+\")\"),u&&W.select(\".nv-y.nv-axis\").attr(\"transform\",\"translate(\"+N+\",0)\");var Y=l.filter(function(a){return a.tempDisabled});U.select(\".tempDisabled\").remove(),Y.length&&U.append(\"text\").attr(\"class\",\"tempDisabled\").attr(\"x\",N/2).attr(\"y\",\"-.71em\").style(\"text-anchor\",\"end\").text(Y.map(function(a){return a.key}).join(\", \")+\" values cannot be calculated for this time period.\"),w&&(k.width(N).height(O).margin({left:m.left,top:m.top}).svgContainer(M).xScale(d),U.select(\".nv-interactive\").call(k)),V.select(\".nv-background\").append(\"rect\"),W.select(\".nv-background rect\").attr(\"width\",N).attr(\"height\",O),f.y(function(a){return a.display.y}).width(N).height(O).color(l.map(function(a,b){return a.color||o(a,b)}).filter(function(a,b){return!l[b].disabled&&!l[b].tempDisabled}));var Z=W.select(\".nv-linesWrap\").datum(l.filter(function(a){return!a.disabled&&!a.tempDisabled}));Z.call(f),l.forEach(function(a,b){a.seriesIndex=b});var $=l.filter(function(a){return!a.disabled&&!!C(a)}),_=W.select(\".nv-avgLinesWrap\").selectAll(\"line\").data($,function(a){return a.key}),aa=function(a){var b=e(C(a));return 0>b?0:b>O?O:b};_.enter().append(\"line\").style(\"stroke-width\",2).style(\"stroke-dasharray\",\"10,10\").style(\"stroke\",function(a,b){return f.color()(a,a.seriesIndex)}).attr(\"x1\",0).attr(\"x2\",\
N).attr(\"y1\",aa).attr(\"y2\",aa),_.style(\"stroke-opacity\",function(a){var b=e(C(a));return 0>b||b>O?0:1}).attr(\"x1\",0).attr(\"x2\",N).attr(\"y1\",aa).attr(\"y2\",aa),_.exit().remove();var ba=Z.selectAll(\".nv-indexLine\").data([H]);ba.enter().append(\"rect\").attr(\"class\",\"nv-indexLine\").attr(\"width\",3).attr(\"x\",-2).attr(\"fill\",\"red\").attr(\"fill-opacity\",.5).style(\"pointer-events\",\"all\").call(Q),ba.attr(\"transform\",function(a){return\"translate(\"+G(a.i)+\",0)\"}).attr(\"height\",O),s&&(g.scale(d)._ticks(a.utils.calcTicksX(N/70,l)).tickSize(-O,0),W.select(\".nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+e.range()[0]+\")\"),W.select(\".nv-x.nv-axis\").call(g)),t&&(h.scale(e)._ticks(a.utils.calcTicksY(O/36,l)).tickSize(-N,0),W.select(\".nv-y.nv-axis\").call(h)),W.select(\".nv-background rect\").on(\"click\",function(){H.x=d3.mouse(this)[0],H.i=Math.round(G.invert(H.x)),z.index=H.i,D.stateChange(z),L()}),f.dispatch.on(\"elementClick\",function(a){H.i=a.pointIndex,H.x=G(H.i),z.index=H.i,D.stateChange(z),L()}),j.dispatch.on(\"legendClick\",function(a,c){a.disabled=!a.disabled,x=!a.disabled,z.rescaleY=x,D.stateChange(z),b.update()}),i.dispatch.on(\"stateChange\",function(a){for(var c in a)z[c]=a[c];D.stateChange(z),b.update()}),k.dispatch.on(\"elementMousemove\",function(c){f.clearHighlights();var d,e,i,j=[];if(l.filter(function(a,b){return a.seriesIndex=b,!a.disabled}).forEach(function(g,h){e=a.interactiveBisect(g.values,c.pointXValue,b.x()),f.highlightPoint(h,e,!0);var k=g.values[e];\"undefined\"!=typeof k&&(\"undefined\"==typeof d&&(d=k),\"undefined\"==typeof i&&(i=b.xScale()(b.x()(k,e))),j.push({key:g.key,value:b.y()(k,e),color:o(g,g.seriesIndex)}))}),j.length>2){var m=b.yScale().invert(c.mouseY),n=Math.abs(b.yScale().domain()[0]-b.yScale().domain()[1]),p=.03*n,q=a.nearestValueIndex(j.map(function(a){return a.value}),m,p);null!==q&&(j[q].highlight=!0)}var r=g.tickFormat()(b.x()(d,e),e);k.tooltip.valueFormatter(function(a,b){return h.tickFormat()(a)}).data({value:r,series:j})(),k.renderGuideLine(i)}),k.dispatch.on(\"elementMouseout\",function(a){f.clearHighlights()}),\
")

file(APPEND "${METABENCH_DIR}/nvd3.js" "\
D.on(\"changeState\",function(a){\"undefined\"!=typeof a.disabled&&(l.forEach(function(b,c){b.disabled=a.disabled[c]}),z.disabled=a.disabled),\"undefined\"!=typeof a.index&&(H.i=a.index,H.x=G(H.i),z.index=a.index,ba.data([H])),\"undefined\"!=typeof a.rescaleY&&(x=a.rescaleY),b.update()})}),I.renderEnd(\"cumulativeLineChart immediate\"),b}function c(a,b){return L||(L=f.y()),b.map(function(b,c){if(!b.values)return b;var d=b.values[a];if(null==d)return b;var e=L(d,a);return-.95>e&&!F?(b.tempDisabled=!0,b):(b.tempDisabled=!1,b.values=b.values.map(function(a,b){return a.display={y:(L(a,b)-e)/(1+e)},a}),b)})}var d,e,f=a.models.line(),g=a.models.axis(),h=a.models.axis(),i=a.models.legend(),j=a.models.legend(),k=a.interactiveGuideline(),l=a.models.tooltip(),m={top:30,right:30,bottom:50,left:60},n=null,o=a.utils.defaultColor(),p=null,q=null,r=!0,s=!0,t=!0,u=!1,v=!0,w=!1,x=!0,y=f.id(),z=a.utils.state(),A=null,B=null,C=function(a){return a.average},D=d3.dispatch(\"stateChange\",\"changeState\",\"renderEnd\"),E=250,F=!1;z.index=0,z.rescaleY=x,g.orient(\"bottom\").tickPadding(7),h.orient(u?\"right\":\"left\"),l.valueFormatter(function(a,b){return h.tickFormat()(a,b)}).headerFormatter(function(a,b){return g.tickFormat()(a,b)}),j.updateState(!1);var G=d3.scale.linear(),H={i:0,x:0},I=a.utils.renderWatch(D,E),J=function(a){return function(){return{active:a.map(function(a){return!a.disabled}),index:H.i,rescaleY:x}}},K=function(a){return function(b){void 0!==b.index&&(H.i=b.index),void 0!==b.rescaleY&&(x=b.rescaleY),void 0!==b.active&&a.forEach(function(a,c){a.disabled=!b.active[c]})}};f.dispatch.on(\"elementMouseover.tooltip\",function(a){var c={x:b.x()(a.point),y:b.y()(a.point),color:a.point.color};a.point=c,l.data(a).hidden(!1)}),f.dispatch.on(\"elementMouseout.tooltip\",function(a){l.hidden(!0)});var L=null;return b.dispatch=D,b.lines=f,b.legend=i,b.controls=j,b.xAxis=g,b.yAxis=h,b.interactiveLayer=k,b.state=z,b.tooltip=l,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return p},set:function(a){p=a}},height:{get:function(){return \
q},set:function(a){q=a}},rescaleY:{get:function(){return x},set:function(a){x=a}},showControls:{get:function(){return v},set:function(a){v=a}},showLegend:{get:function(){return r},set:function(a){r=a}},average:{get:function(){return C},set:function(a){C=a}},defaultState:{get:function(){return A},set:function(a){A=a}},noData:{get:function(){return B},set:function(a){B=a}},showXAxis:{get:function(){return s},set:function(a){s=a}},showYAxis:{get:function(){return t},set:function(a){t=a}},noErrorCheck:{get:function(){return F},set:function(a){F=a}},margin:{get:function(){return m},set:function(a){void 0!==a.top&&(m.top=a.top,n=a.top),m.right=void 0!==a.right?a.right:m.right,m.bottom=void 0!==a.bottom?a.bottom:m.bottom,m.left=void 0!==a.left?a.left:m.left}},color:{get:function(){return o},set:function(b){o=a.utils.getColor(b),i.color(o)}},useInteractiveGuideline:{get:function(){return w},set:function(a){w=a,a===!0&&(b.interactive(!1),b.useVoronoi(!1))}},rightAlignYAxis:{get:function(){return u},set:function(a){u=a,h.orient(a?\"right\":\"left\")}},duration:{get:function(){return E},set:function(a){E=a,f.duration(E),g.duration(E),h.duration(E),I.reset(E)}}}),a.utils.inheritOptions(b,f),a.utils.initOptions(b),b},a.models.discreteBar=function(){\"use strict\";function b(m){return y.reset(),m.each(function(b){var m=k-j.left-j.right,x=l-j.top-j.bottom;c=d3.select(this),a.utils.initSVG(c),b.forEach(function(a,b){a.values.forEach(function(a){a.series=b})});var z=d&&e?[]:b.map(function(a){return a.values.map(function(a,b){return{x:p(a,b),y:q(a,b),y0:a.y0}})});n.domain(d||d3.merge(z).map(function(a){return a.x})).rangeBands(f||[0,m],.1),o.domain(e||d3.extent(d3.merge(z).map(function(a){return a.y}).concat(r))),t?o.range(g||[x-(o.domain()[0]<0?12:0),o.domain()[1]>0?12:0]):o.range(g||[x,0]),h=h||n,i=i||o.copy().range([o(0),o(0)]);var A=c.selectAll(\"g.nv-wrap.nv-discretebar\").data([b]),B=A.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-discretebar\"),C=B.append(\"g\");A.select(\"g\");C.append(\"g\").attr(\"class\",\"nv-groups\"),A.attr(\"transform\",\
\"translate(\"+j.left+\",\"+j.top+\")\");var D=A.select(\".nv-groups\").selectAll(\".nv-group\").data(function(a){return a},function(a){return a.key});D.enter().append(\"g\").style(\"stroke-opacity\",1e-6).style(\"fill-opacity\",1e-6),D.exit().watchTransition(y,\"discreteBar: exit groups\").style(\"stroke-opacity\",1e-6).style(\"fill-opacity\",1e-6).remove(),D.attr(\"class\",function(a,b){return\"nv-group nv-series-\"+b}).classed(\"hover\",function(a){return a.hover}),D.watchTransition(y,\"discreteBar: groups\").style(\"stroke-opacity\",1).style(\"fill-opacity\",.75);var E=D.selectAll(\"g.nv-bar\").data(function(a){return a.values});E.exit().remove();var F=E.enter().append(\"g\").attr(\"transform\",function(a,b,c){return\"translate(\"+(n(p(a,b))+.05*n.rangeBand())+\", \"+o(0)+\")\"}).on(\"mouseover\",function(a,b){d3.select(this).classed(\"hover\",!0),v.elementMouseover({data:a,index:b,color:d3.select(this).style(\"fill\")})}).on(\"mouseout\",function(a,b){d3.select(this).classed(\"hover\",!1),v.elementMouseout({data:a,index:b,color:d3.select(this).style(\"fill\")})}).on(\"mousemove\",function(a,b){v.elementMousemove({data:a,index:b,color:d3.select(this).style(\"fill\")})}).on(\"click\",function(a,b){var c=this;v.elementClick({data:a,index:b,color:d3.select(this).style(\"fill\"),event:d3.event,element:c}),d3.event.stopPropagation()}).on(\"dblclick\",function(a,b){v.elementDblClick({data:a,index:b,color:d3.select(this).style(\"fill\")}),d3.event.stopPropagation()});F.append(\"rect\").attr(\"height\",0).attr(\"width\",.9*n.rangeBand()/b.length),t?(F.append(\"text\").attr(\"text-anchor\",\"middle\"),E.select(\"text\").text(function(a,b){return u(q(a,b))}).watchTransition(y,\"discreteBar: bars text\").attr(\"x\",.9*n.rangeBand()/2).attr(\"y\",function(a,b){return q(a,b)<0?o(q(a,b))-o(0)+12:-4})):E.selectAll(\"text\").remove(),E.attr(\"class\",function(a,b){return q(a,b)<0?\"nv-bar negative\":\"nv-bar positive\"}).style(\"fill\",function(a,b){return a.color||s(a,b)}).style(\"stroke\",function(a,b){return a.color||s(a,b)}).select(\"rect\").attr(\"class\",w).watchTransition(y,\"discreteBar: bars rect\").attr(\"width\",.9*n.rangeBand()/b.length),\
E.watchTransition(y,\"discreteBar: bars\").attr(\"transform\",function(a,b){var c=n(p(a,b))+.05*n.rangeBand(),d=q(a,b)<0?o(0):o(0)-o(q(a,b))<1?o(0)-1:o(q(a,b));return\"translate(\"+c+\", \"+d+\")\"}).select(\"rect\").attr(\"height\",function(a,b){return Math.max(Math.abs(o(q(a,b))-o(0)),1)}),h=n.copy(),i=o.copy()}),y.renderEnd(\"discreteBar immediate\"),b}var c,d,e,f,g,h,i,j={top:0,right:0,bottom:0,left:0},k=960,l=500,m=Math.floor(1e4*Math.random()),n=d3.scale.ordinal(),o=d3.scale.linear(),p=function(a){return a.x},q=function(a){return a.y},r=[0],s=a.utils.defaultColor(),t=!1,u=d3.format(\",.2f\"),v=d3.dispatch(\"chartClick\",\"elementClick\",\"elementDblClick\",\"elementMouseover\",\"elementMouseout\",\"elementMousemove\",\"renderEnd\"),w=\"discreteBar\",x=250,y=a.utils.renderWatch(v,x);return b.dispatch=v,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return k},set:function(a){k=a}},height:{get:function(){return l},set:function(a){l=a}},forceY:{get:function(){return r},set:function(a){r=a;\
}},showValues:{get:function(){return t},set:function(a){t=a}},x:{get:function(){return p},set:function(a){p=a}},y:{get:function(){return q},set:function(a){q=a}},xScale:{get:function(){return n},set:function(a){n=a}},yScale:{get:function(){return o},set:function(a){o=a}},xDomain:{get:function(){return d},set:function(a){d=a}},yDomain:{get:function(){return e},set:function(a){e=a}},xRange:{get:function(){return f},set:function(a){f=a}},yRange:{get:function(){return g},set:function(a){g=a}},valueFormat:{get:function(){return u},set:function(a){u=a}},id:{get:function(){return m},set:function(a){m=a}},rectClass:{get:function(){return w},set:function(a){w=a}},margin:{get:function(){return j},set:function(a){j.top=void 0!==a.top?a.top:j.top,j.right=void 0!==a.right?a.right:j.right,j.bottom=void 0!==a.bottom?a.bottom:j.bottom,j.left=void 0!==a.left?a.left:j.left}},color:{get:function(){return s},set:function(b){s=a.utils.getColor(b)}},duration:{get:function(){return x},set:function(a){x=a,y.reset(x)}}}),a.utils.initOptions(b),b},a.models.discreteBarChart=function(){\"use strict\";function b(i){return y.reset(),y.models(e),p&&y.models(f),q&&y.models(g),i.each(function(i){var n=d3.select(this);a.utils.initSVG(n);var v=a.utils.availableWidth(l,n,j),y=a.utils.availableHeight(m,n,j);if(b.update=function(){w.beforeUpdate(),n.transition().duration(x).call(b)},b.container=this,!(i&&i.length&&i.filter(function(a){return a.values.length}).length))return a.utils.noData(b,n),b;n.selectAll(\".nv-noData\").remove(),c=e.xScale(),d=e.yScale().clamp(!0);var z=n.selectAll(\"g.nv-wrap.nv-discreteBarWithAxes\").data([i]),A=z.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-discreteBarWithAxes\").append(\"g\"),B=A.append(\"defs\"),C=z.select(\"g\");A.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),A.append(\"g\").attr(\"class\",\"nv-y nv-axis\").append(\"g\").attr(\"class\",\"nv-zeroLine\").append(\"line\"),A.append(\"g\").attr(\"class\",\"nv-barsWrap\"),A.append(\"g\").attr(\"class\",\"nv-legendWrap\"),C.attr(\"transform\",\"translate(\"+j.left+\",\"+j.top+\")\"),o?(h.width(v),C.select(\".nv-legendWrap\").datum(i).call(h),\
k||h.height()===j.top||(j.top=h.height(),y=a.utils.availableHeight(m,n,j)),z.select(\".nv-legendWrap\").attr(\"transform\",\"translate(0,\"+-j.top+\")\")):C.select(\".nv-legendWrap\").selectAll(\"*\").remove(),r&&C.select(\".nv-y.nv-axis\").attr(\"transform\",\"translate(\"+v+\",0)\"),e.width(v).height(y);var D=C.select(\".nv-barsWrap\").datum(i.filter(function(a){return!a.disabled}));if(D.transition().call(e),B.append(\"clipPath\").attr(\"id\",\"nv-x-label-clip-\"+e.id()).append(\"rect\"),C.select(\"#nv-x-label-clip-\"+e.id()+\" rect\").attr(\"width\",c.rangeBand()*(s?2:1)).attr(\"height\",16).attr(\"x\",-c.rangeBand()/(s?1:2)),p){f.scale(c)._ticks(a.utils.calcTicksX(v/100,i)).tickSize(-y,0),C.select(\".nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+(d.range()[0]+(e.showValues()&&d.domain()[0]<0?16:0))+\")\"),C.select(\".nv-x.nv-axis\").call(f);var E=C.select(\".nv-x.nv-axis\").selectAll(\"g\");s&&E.selectAll(\"text\").attr(\"transform\",function(a,b,c){return\"translate(0,\"+(c%2==0?\"5\":\"17\")+\")\"}),u&&E.selectAll(\".tick text\").attr(\"transform\",\"rotate(\"+u+\" 0,0)\").style(\"text-anchor\",u>0?\"start\":\"end\"),t&&C.selectAll(\".tick text\").call(a.utils.wrapTicks,b.xAxis.rangeBand())}q&&(g.scale(d)._ticks(a.utils.calcTicksY(y/36,i)).tickSize(-v,0),C.select(\".nv-y.nv-axis\").call(g)),C.select(\".nv-zeroLine line\").attr(\"x1\",0).attr(\"x2\",r?-v:v).attr(\"y1\",d(0)).attr(\"y2\",d(0))}),y.renderEnd(\"discreteBar chart immediate\"),b}var c,d,e=a.models.discreteBar(),f=a.models.axis(),g=a.models.axis(),h=a.models.legend(),i=a.models.tooltip(),j={top:15,right:10,bottom:50,left:60},k=null,l=null,m=null,n=a.utils.getColor(),o=!1,p=!0,q=!0,r=!1,s=!1,t=!1,u=0,v=null,w=d3.dispatch(\"beforeUpdate\",\"renderEnd\"),x=250;f.orient(\"bottom\").showMaxMin(!1).tickFormat(function(a){return a}),g.orient(r?\"right\":\"left\").tickFormat(d3.format(\",.1f\")),i.duration(0).headerEnabled(!1).valueFormatter(function(a,b){return g.tickFormat()(a,b)}).keyFormatter(function(a,b){return f.tickFormat()(a,b)});var y=a.utils.renderWatch(w,x);return e.dispatch.on(\"elementMouseover.tooltip\",function(a){a.series={key:b.x()(a.data),value:b.y()(a.data),\
color:a.color},i.data(a).hidden(!1)}),e.dispatch.on(\"elementMouseout.tooltip\",function(a){i.hidden(!0)}),e.dispatch.on(\"elementMousemove.tooltip\",function(a){i()}),b.dispatch=w,b.discretebar=e,b.legend=h,b.xAxis=f,b.yAxis=g,b.tooltip=i,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return l},set:function(a){l=a}},height:{get:function(){return m},set:function(a){m=a}},showLegend:{get:function(){return o},set:function(a){o=a}},staggerLabels:{get:function(){return s},set:function(a){s=a}},rotateLabels:{get:function(){return u},set:function(a){u=a}},wrapLabels:{get:function(){return t},set:function(a){t=!!a}},showXAxis:{get:function(){return p},set:function(a){p=a}},showYAxis:{get:function(){return q},set:function(a){q=a}},noData:{get:function(){return v},set:function(a){v=a}},margin:{get:function(){return j},set:function(a){void 0!==a.top&&(j.top=a.top,k=a.top),j.right=void 0!==a.right?a.right:j.right,j.bottom=void 0!==a.bottom?a.bottom:j.bottom,j.left=void 0!==a.left?a.left:j.left}},duration:{get:function(){return x},set:function(a){x=a,y.reset(x),e.duration(x),f.duration(x),g.duration(x)}},color:{get:function(){return n},set:function(b){n=a.utils.getColor(b),e.color(n),h.color(n)}},rightAlignYAxis:{get:function(){return r},set:function(a){r=a,g.orient(a?\"right\":\"left\")}}}),a.utils.inheritOptions(b,e),a.utils.initOptions(b),b},a.models.distribution=function(){\"use strict\";function b(k){return m.reset(),k.each(function(b){var k=(e-(\"x\"===g?d.left+d.right:d.top+d.bottom),\"x\"==g?\"y\":\"x\"),l=d3.select(this);a.utils.initSVG(l),c=c||j;var n=l.selectAll(\"g.nv-distribution\").data([b]),o=n.enter().append(\"g\").attr(\"class\",\"nvd3 nv-distribution\"),p=(o.append(\"g\"),n.select(\"g\"));n.attr(\"transform\",\"translate(\"+d.left+\",\"+d.top+\")\");var q=p.selectAll(\"g.nv-dist\").data(function(a){return a},function(a){return a.key});q.enter().append(\"g\"),q.attr(\"class\",function(a,b){return\"nv-dist nv-series-\"+b}).style(\"stroke\",function(a,b){return i(a,b)});var r=q.selectAll(\"line.nv-dist\"+g).data(function(a){return \
a.values});r.enter().append(\"line\").attr(g+\"1\",function(a,b){return c(h(a,b))}).attr(g+\"2\",function(a,b){return c(h(a,b))}),m.transition(q.exit().selectAll(\"line.nv-dist\"+g),\"dist exit\").attr(g+\"1\",function(a,b){return j(h(a,b))}).attr(g+\"2\",function(a,b){return j(h(a,b))}).style(\"stroke-opacity\",0).remove(),r.attr(\"class\",function(a,b){return\"nv-dist\"+g+\" nv-dist\"+g+\"-\"+b}).attr(k+\"1\",0).attr(k+\"2\",f),m.transition(r,\"dist\").attr(g+\"1\",function(a,b){return j(h(a,b))}).attr(g+\"2\",function(a,b){return j(h(a,b))}),c=j.copy()}),m.renderEnd(\"distribution immediate\"),b}var c,d={top:0,right:0,bottom:0,left:0},e=400,f=8,g=\"x\",h=function(a){return a[g]},i=a.utils.defaultColor(),j=d3.scale.linear(),k=250,l=d3.dispatch(\"renderEnd\"),m=a.utils.renderWatch(l,k);return b.options=a.utils.optionsFunc.bind(b),b.dispatch=l,b.margin=function(a){return arguments.length?(d.top=\"undefined\"!=typeof a.top?a.top:d.top,d.right=\"undefined\"!=typeof a.right?a.right:d.right,d.bottom=\"undefined\"!=typeof a.bottom?a.bottom:d.bottom,d.left=\"undefined\"!=typeof a.left?a.left:d.left,b):d},b.width=function(a){return arguments.length?(e=a,b):e},b.axis=function(a){return arguments.length?(g=a,b):g},b.size=function(a){return arguments.length?(f=a,b):f},b.getData=function(a){return arguments.length?(h=d3.functor(a),b):h},b.scale=function(a){return arguments.length?(j=a,b):j},b.color=function(c){return arguments.length?(i=a.utils.getColor(c),b):i},b.duration=function(a){return arguments.length?(k=a,m.reset(k),b):k},b},a.models.focus=function(b){\"use strict\";function c(u){return t.reset(),t.models(b),m&&t.models(f),n&&t.models(g),u.each(function(t){function u(a){var b=+(\"e\"==a),c=b?1:-1,d=z/3;return\"M\"+.5*c+\",\"+d+\"A6,6 0 0 \"+b+\" \"+6.5*c+\",\"+(d+6)+\"V\"+(2*d-6)+\"A6,6 0 0 \"+b+\" \"+.5*c+\",\"+2*d+\"ZM\"+2.5*c+\",\"+(d+8)+\"V\"+(2*d-8)+\"M\"+4.5*c+\",\"+(d+8)+\"V\"+(2*d-8)}function v(){h.empty()||h.extent(p),E.data([h.empty()?d.domain():p]).each(function(a,b){var c=d(a[0])-d.range()[0],e=y-d(a[1]);d3.select(this).select(\".left\").attr(\"width\",0>c?0:c),d3.select(this).select(\".right\").attr(\"x\",\
d(a[1])).attr(\"width\",0>e?0:e)})}function w(a){p=h.empty()?null:h.extent();var b=h.empty()?d.domain():h.extent();r.brush({extent:b,brush:h}),v(),a&&r.onBrush(b)}var x=d3.select(this);a.utils.initSVG(x);var y=a.utils.availableWidth(k,x,i),z=l-i.top-i.bottom;c.update=function(){0===q?x.call(c):x.transition().duration(q).call(c)},c.container=this,d=b.xScale(),e=b.yScale();var A=x.selectAll(\"g.nv-focus\").data([t]),B=A.enter().append(\"g\").attr(\"class\",\"nvd3 nv-focus\").append(\"g\"),C=A.select(\"g\");A.attr(\"transform\",\"translate(\"+i.left+\",\"+i.top+\")\"),B.append(\"g\").attr(\"class\",\"nv-background\").append(\"rect\"),B.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),B.append(\"g\").attr(\"class\",\"nv-y nv-axis\"),B.append(\"g\").attr(\"class\",\"nv-contentWrap\"),B.append(\"g\").attr(\"class\",\"nv-brushBackground\"),B.append(\"g\").attr(\"class\",\"nv-x nv-brush\"),o&&C.select(\".nv-y.nv-axis\").attr(\"transform\",\"translate(\"+y+\",0)\"),C.select(\".nv-background rect\").attr(\"width\",y).attr(\"height\",z),b.width(y).height(z).color(t.map(function(a,b){return a.color||j(a,b)}).filter(function(a,b){return!t[b].disabled}));var D=C.select(\".nv-contentWrap\").datum(t.filter(function(a){return!a.disabled}));d3.transition(D).call(b),h.x(d).on(\"brush\",function(){w(s)}),h.on(\"brushend\",function(){s||r.onBrush(h.empty()?d.domain():h.extent())}),p&&h.extent(p);var E=C.select(\".nv-brushBackground\").selectAll(\"g\").data([p||h.extent()]),F=E.enter().append(\"g\");F.append(\"rect\").attr(\"class\",\"left\").attr(\"x\",0).attr(\"y\",0).attr(\"height\",z),F.append(\"rect\").attr(\"class\",\"right\").attr(\"x\",0).attr(\"y\",0).attr(\"height\",z);var G=C.select(\".nv-x.nv-brush\").call(h);G.selectAll(\"rect\").attr(\"height\",z),G.selectAll(\".resize\").append(\"path\").attr(\"d\",u),w(!0),C.select(\".nv-background rect\").attr(\"width\",y).attr(\"height\",z),m&&(f.scale(d)._ticks(a.utils.calcTicksX(y/100,t)).tickSize(-z,0),C.select(\".nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+e.range()[0]+\")\"),d3.transition(C.select(\".nv-x.nv-axis\")).call(f)),n&&(g.scale(e)._ticks(a.utils.calcTicksY(z/36,t)).tickSize(-y,0),d3.transition(C.select(\".nv-y.nv-axis\")).call(g)),\
C.select(\".nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+e.range()[0]+\")\")}),t.renderEnd(\"focus immediate\"),c}var d,e,b=b||a.models.line(),f=a.models.axis(),g=a.models.axis(),h=d3.svg.brush(),i={top:10,right:0,bottom:30,left:0},j=a.utils.defaultColor(),k=null,l=70,m=!0,n=!1,o=!1,p=null,q=250,r=d3.dispatch(\"brush\",\"onBrush\",\"renderEnd\"),s=!0;b.interactive(!1),b.pointActive(function(a){return!1});var t=a.utils.renderWatch(r,q);return c.dispatch=r,c.content=b,c.brush=h,c.xAxis=f,c.yAxis=g,c.options=a.utils.optionsFunc.bind(c),c._options=Object.create({},{width:{get:function(){return k},set:function(a){k=a}},height:{get:function(){return l},set:function(a){l=a}},showXAxis:{get:function(){return m},set:function(a){m=a}},showYAxis:{get:function(){return n},set:function(a){n=a}},brushExtent:{get:function(){return p},set:function(a){p=a}},syncBrushing:{get:function(){return s},set:function(a){s=a}},margin:{get:function(){return i},set:function(a){i.top=void 0!==a.top?a.top:i.top,i.right=void 0!==a.right?a.right:i.right,i.bottom=void 0!==a.bottom?a.bottom:i.bottom,i.left=void 0!==a.left?a.left:i.left}},duration:{get:function(){return q},set:function(a){q=a,t.reset(q),b.duration(q),f.duration(q),g.duration(q)}},color:{get:function(){return j},set:function(c){j=a.utils.getColor(c),b.color(j)}},interpolate:{get:function(){return b.interpolate()},set:function(a){b.interpolate(a)}},xTickFormat:{get:function(){return f.tickFormat()},set:function(a){f.tickFormat(a)}},yTickFormat:{get:function(){return g.tickFormat()},set:function(a){g.tickFormat(a)}},x:{get:function(){return b.x()},set:function(a){b.x(a)}},y:{get:function(){return b.y()},set:function(a){b.y(a)}},rightAlignYAxis:{get:function(){return o},set:function(a){o=a,g.orient(o?\"right\":\"left\")}}}),a.utils.inheritOptions(c,b),a.utils.initOptions(c),c},a.models.forceDirectedGraph=function(){\"use strict\";function b(g){return u.reset(),g.each(function(g){f=d3.select(this),a.utils.initSVG(f);var j=a.utils.availableWidth(d,f,c),u=a.utils.availableHeight(e,f,c);if(f.attr(\"width\",j).attr(\"height\",\
u),!(g&&g.links&&g.nodes))return a.utils.noData(b,f),b;f.selectAll(\".nv-noData\").remove(),f.selectAll(\"*\").remove();var v=new Set;g.nodes.forEach(function(a){var b=Object.keys(a);b.forEach(function(a){v.add(a)})});var w=d3.layout.force().nodes(g.nodes).links(g.links).size([j,u]).linkStrength(k).friction(l).linkDistance(m).charge(n).gravity(o).theta(p).alpha(q).start(),x=f.selectAll(\".link\").data(g.links).enter().append(\"line\").attr(\"class\",\"nv-force-link\").style(\"stroke-width\",function(a){return Math.sqrt(a.value)}),y=f.selectAll(\".node\").data(g.nodes).enter().append(\"g\").attr(\"class\",\"nv-force-node\").call(w.drag);y.append(\"circle\").attr(\"r\",r).style(\"fill\",function(a){return h(a)}).on(\"mouseover\",function(a){f.select(\".nv-series-\"+a.seriesIndex+\" .nv-distx-\"+a.pointIndex).attr(\"y1\",a.py),f.select(\".nv-series-\"+a.seriesIndex+\" .nv-disty-\"+a.pointIndex).attr(\"x2\",a.px);var b=h(a);a.series=[],v.forEach(function(c){a.series.push({color:b,key:c,value:a[c]})}),i.data(a).hidden(!1)}).on(\"mouseout\",function(a){i.hidden(!0)}),i.headerFormatter(function(a){return\"Node\"}),t(x),s(y),w.on(\"tick\",function(){x.attr(\"x1\",function(a){return a.source.x}).attr(\"y1\",function(a){return a.source.y}).attr(\"x2\",function(a){return a.target.x}).attr(\"y2\",function(a){return a.target.y}),y.attr(\"transform\",function(a){return\"translate(\"+a.x+\", \"+a.y+\")\"})})}),b}var c={top:2,right:0,bottom:2,left:0},d=400,e=32,f=null,g=d3.dispatch(\"renderEnd\"),h=a.utils.getColor([\"#000\"]),i=a.models.tooltip(),j=null,k=.1,l=.9,m=30,n=-120,o=.1,p=.8,q=.1,r=5,s=function(a){},t=function(a){},u=a.utils.renderWatch(g);return b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return d},set:function(a){d=a}},height:{get:function(){return e},set:function(a){e=a}},linkStrength:{get:function(){return k},set:function(a){k=a}},friction:{get:function(){return l},set:function(a){l=a}},linkDist:{get:function(){return m},set:function(a){m=a}},charge:{get:function(){return n},set:function(a){n=a}},gravity:{get:function(){return o},set:function(a){o=a}},\
theta:{get:function(){return p},set:function(a){p=a}},alpha:{get:function(){return q},set:function(a){q=a}},radius:{get:function(){return r},set:function(a){r=a}},x:{get:function(){return getX},set:function(a){getX=d3.functor(a)}},y:{get:function(){return getY},set:function(a){getY=d3.functor(a)}},margin:{get:function(){return c},set:function(a){c.top=void 0!==a.top?a.top:c.top,c.right=void 0!==a.right?a.right:c.right,c.bottom=void 0!==a.bottom?a.bottom:c.bottom,c.left=void 0!==a.left?a.left:c.left}},color:{get:function(){return h},set:function(b){h=a.utils.getColor(b)}},noData:{get:function(){return j},set:function(a){j=a}},nodeExtras:{get:function(){return s},set:function(a){s=a}},linkExtras:{get:function(){return t},set:function(a){t=a}}}),b.dispatch=g,b.tooltip=i,a.utils.initOptions(b),b},a.models.furiousLegend=function(){\"use strict\";function b(r){function s(a,b){return\"furious\"!=q?\"#000\":o?a.disengaged?h(a,b):\"#fff\":o?void 0:a.disabled?h(a,b):\"#fff\"}function t(a,b){return o&&\"furious\"==q?a.disengaged?\"#fff\":h(a,b):a.disabled?\"#fff\":h(a,b)}return r.each(function(b){var r=d-c.left-c.right,u=d3.select(this);a.utils.initSVG(u);var v=u.selectAll(\"g.nv-legend\").data([b]),w=(v.enter().append(\"g\").attr(\"class\",\"nvd3 nv-legend\").append(\"g\"),v.select(\"g\"));v.attr(\"transform\",\"translate(\"+c.left+\",\"+c.top+\")\");var x,y=w.selectAll(\".nv-series\").data(function(a){return\"furious\"!=q?a:a.filter(function(a){return o?!0:!a.disengaged})}),z=y.enter().append(\"g\").attr(\"class\",\"nv-series\");if(\"classic\"==q)z.append(\"circle\").style(\"stroke-width\",2).attr(\"class\",\"nv-legend-symbol\").attr(\"r\",5),x=y.select(\"circle\");else if(\"furious\"==q){z.append(\"rect\").style(\"stroke-width\",2).attr(\"class\",\"nv-legend-symbol\").attr(\"rx\",3).attr(\"ry\",3),x=y.select(\"rect\"),z.append(\"g\").attr(\"class\",\"nv-check-box\").property(\"innerHTML\",'<path d=\"M0.5,5 L22.5,5 L22.5,26.5 L0.5,26.5 L0.5,5 Z\" class=\"nv-box\"></path><path d=\"M5.5,12.8618467 L11.9185089,19.2803556 L31,0.198864511\" class=\"nv-check\"></path>').attr(\"transform\",\"translate(-10,-8)scale(0.5)\");var \
A=y.select(\".nv-check-box\");A.each(function(a,b){d3.select(this).selectAll(\"path\").attr(\"stroke\",s(a,b))})}z.append(\"text\").attr(\"text-anchor\",\"start\").attr(\"class\",\"nv-legend-text\").attr(\"dy\",\".32em\").attr(\"dx\",\"8\");var B=y.select(\"text.nv-legend-text\");y.on(\"mouseover\",function(a,b){p.legendMouseover(a,b)}).on(\"mouseout\",function(a,b){p.legendMouseout(a,b)}).on(\"click\",function(a,b){p.legendClick(a,b);var c=y.data();if(m){if(\"classic\"==q)n?(c.forEach(function(a){a.disabled=!0}),a.disabled=!1):(a.disabled=!a.disabled,c.every(function(a){return a.disabled})&&c.forEach(function(a){a.disabled=!1}));else if(\"furious\"==q)if(o)a.disengaged=!a.disengaged,a.userDisabled=void 0==a.userDisabled?!!a.disabled:a.userDisabled,a.disabled=a.disengaged||a.userDisabled;else if(!o){a.disabled=!a.disabled,a.userDisabled=a.disabled;var d=c.filter(function(a){return!a.disengaged});d.every(function(a){return a.userDisabled})&&c.forEach(function(a){a.disabled=a.userDisabled=!1})}p.stateChange({disabled:c.map(function(a){return!!a.disabled}),disengaged:c.map(function(a){return!!a.disengaged})})}}).on(\"dblclick\",function(a,b){if((\"furious\"!=q||!o)&&(p.legendDblclick(a,b),m)){var c=y.data();c.forEach(function(a){a.disabled=!0,\"furious\"==q&&(a.userDisabled=a.disabled)}),a.disabled=!1,\"furious\"==q&&(a.userDisabled=a.disabled),p.stateChange({disabled:c.map(function(a){return!!a.disabled})})}}),y.classed(\"nv-disabled\",function(a){return a.userDisabled}),y.exit().remove(),B.attr(\"fill\",s).text(function(a){return g(f(a))});var C;switch(q){case\"furious\":C=23;break;case\"classic\":C=20}if(j){var D=[];y.each(function(b,c){var d;if(g(f(b))&&g(f(b)).length>i){var e=g(f(b)).substring(0,i);d=d3.select(this).select(\"text\").text(e+\"...\"),d3.select(this).append(\"svg:title\").text(g(f(b)))}else d=d3.select(this).select(\"text\");var h;try{if(h=d.node().getComputedTextLength(),0>=h)throw Error()}catch(j){h=a.utils.calcApproxTextWidth(d)}D.push(h+k)});for(var E=0,F=0,G=[];r>F&&E<D.length;)G[E]=D[E],F+=D[E++];for(0===E&&(E=1);F>r&&E>1;){G=[],E--;for(var H=0;H<D.length;\
H++)D[H]>(G[H%E]||0)&&(G[H%E]=D[H]);F=G.reduce(function(a,b,c,d){return a+b})}for(var I=[],J=0,K=0;E>J;J++)I[J]=K,K+=G[J];y.attr(\"transform\",function(a,b){return\"translate(\"+I[b%E]+\",\"+(5+Math.floor(b/E)*C)+\")\"}),l?w.attr(\"transform\",\"translate(\"+(d-c.right-F)+\",\"+c.top+\")\"):w.attr(\"transform\",\"translate(0,\"+c.top+\")\"),e=c.top+c.bottom+Math.ceil(D.length/E)*C}else{var L,M=5,N=5,O=0;y.attr(\"transform\",function(a,b){var e=d3.select(this).select(\"text\").node().getComputedTextLength()+k;return L=N,d<c.left+c.right+L+e&&(N=L=5,M+=C),N+=e,N>O&&(O=N),\"translate(\"+L+\",\"+M+\")\"}),w.attr(\"transform\",\"translate(\"+(d-c.right-O)+\",\"+c.top+\")\"),e=c.top+c.bottom+M+15}\"furious\"==q&&x.attr(\"width\",function(a,b){return B[0][b].getComputedTextLength()+27}).attr(\"height\",18).attr(\"y\",-9).attr(\"x\",-15),x.style(\"fill\",t).style(\"stroke\",function(a,b){return a.color||h(a,b)})}),b}var c={top:5,right:0,bottom:5,left:0},d=400,e=20,f=function(a){return a.key},g=function(a){return a},h=a.utils.getColor(),i=20,j=!0,k=28,l=!0,m=!0,n=!1,o=!1,p=d3.dispatch(\"legendClick\",\"legendDblclick\",\"legendMouseover\",\"legendMouseout\",\"stateChange\"),q=\"classic\";return b.dispatch=p,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return d},set:function(a){d=a}},height:{get:function(){return e},set:function(a){e=a}},key:{get:function(){return f},set:function(a){f=a}},keyFormatter:{get:function(){return g},set:function(a){g=a}},align:{get:function(){return j},set:function(a){j=a}},rightAlign:{get:function(){return l},set:function(a){l=a}},maxKeyLength:{get:function(){return i},set:function(a){i=a}},padding:{get:function(){return k},set:function(a){k=a}},updateState:{get:function(){return m},set:function(a){m=a}},radioButtonMode:{get:function(){return n},set:function(a){n=a}},expanded:{get:function(){return o},set:function(a){o=a}},vers:{get:function(){return q},set:function(a){q=a}},margin:{get:function(){return c},set:function(a){c.top=void 0!==a.top?a.top:c.top,c.right=void 0!==a.right?a.right:c.right,c.bottom=void 0!==a.bottom?a.bottom:c.bottom,\
c.left=void 0!==a.left?a.left:c.left}},color:{get:function(){return h},set:function(b){h=a.utils.getColor(b)}}}),a.utils.initOptions(b),b},a.models.historicalBar=function(){\"use strict\";function b(x){return x.each(function(b){w.reset(),k=d3.select(this);var x=a.utils.availableWidth(h,k,g),y=a.utils.availableHeight(i,k,g);a.utils.initSVG(k),l.domain(c||d3.extent(b[0].values.map(n).concat(p))),r?l.range(e||[.5*x/b[0].values.length,x*(b[0].values.length-.5)/b[0].values.length]):l.range(e||[0,x]),m.domain(d||d3.extent(b[0].values.map(o).concat(q))).range(f||[y,0]),l.domain()[0]===l.domain()[1]&&(l.domain()[0]?l.domain([l.domain()[0]-.01*l.domain()[0],l.domain()[1]+.01*l.domain()[1]]):l.domain([-1,1])),m.domain()[0]===m.domain()[1]&&(m.domain()[0]?m.domain([m.domain()[0]+.01*m.domain()[0],m.domain()[1]-.01*m.domain()[1]]):m.domain([-1,1]));var z=k.selectAll(\"g.nv-wrap.nv-historicalBar-\"+j).data([b[0].values]),A=z.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-historicalBar-\"+j),B=A.append(\"defs\"),C=A.append(\"g\"),D=z.select(\"g\");C.append(\"g\").attr(\"class\",\"nv-bars\"),z.attr(\"transform\",\"translate(\"+g.left+\",\"+g.top+\")\"),k.on(\"click\",function(a,b){u.chartClick({data:a,index:b,pos:d3.event,id:j})}),B.append(\"clipPath\").attr(\"id\",\"nv-chart-clip-path-\"+j).append(\"rect\"),z.select(\"#nv-chart-clip-path-\"+j+\" rect\").attr(\"width\",x).attr(\"height\",y),D.attr(\"clip-path\",s?\"url(#nv-chart-clip-path-\"+j+\")\":\"\");var E=z.select(\".nv-bars\").selectAll(\".nv-bar\").data(function(a){return a},function(a,b){return n(a,b)});E.exit().remove(),E.enter().append(\"rect\").attr(\"x\",0).attr(\"y\",function(b,c){return a.utils.NaNtoZero(m(Math.max(0,o(b,c))))}).attr(\"height\",function(b,c){return a.utils.NaNtoZero(Math.abs(m(o(b,c))-m(0)))}).attr(\"transform\",function(a,c){return\"translate(\"+(l(n(a,c))-x/b[0].values.length*.45)+\",0)\"}).on(\"mouseover\",function(a,b){v&&(d3.select(this).classed(\"hover\",!0),u.elementMouseover({data:a,index:b,color:d3.select(this).style(\"fill\")}))}).on(\"mouseout\",function(a,b){v&&(d3.select(this).classed(\"hover\",!1),u.elementMouseout({data:a,\
index:b,color:d3.select(this).style(\"fill\")}))}).on(\"mousemove\",function(a,b){v&&u.elementMousemove({data:a,index:b,color:d3.select(this).style(\"fill\")})}).on(\"click\",function(a,b){if(v){var c=this;u.elementClick({data:a,index:b,color:d3.select(this).style(\"fill\"),event:d3.event,element:c}),d3.event.stopPropagation()}}).on(\"dblclick\",function(a,b){v&&(u.elementDblClick({data:a,index:b,color:d3.select(this).style(\"fill\")}),d3.event.stopPropagation())}),E.attr(\"fill\",function(a,b){return t(a,b)}).attr(\"class\",function(a,b,c){return(o(a,b)<0?\"nv-bar negative\":\"nv-bar positive\")+\" nv-bar-\"+c+\"-\"+b}).watchTransition(w,\"bars\").attr(\"transform\",function(a,c){return\"translate(\"+(l(n(a,c))-x/b[0].values.length*.45)+\",0)\"}).attr(\"width\",x/b[0].values.length*.9),E.watchTransition(w,\"bars\").attr(\"y\",function(b,c){var d=o(b,c)<0?m(0):m(0)-m(o(b,c))<1?m(0)-1:m(o(b,c));return a.utils.NaNtoZero(d)}).attr(\"height\",function(b,c){return a.utils.NaNtoZero(Math.max(Math.abs(m(o(b,c))-m(0)),1))})}),w.renderEnd(\"historicalBar immediate\"),b}var c,d,e,f,g={top:0,right:0,bottom:0,left:0},h=null,i=null,j=Math.floor(1e4*Math.random()),k=null,l=d3.scale.linear(),m=d3.scale.linear(),n=function(a){return a.x},o=function(a){return a.y},p=[],q=[0],r=!1,s=!0,t=a.utils.defaultColor(),u=d3.dispatch(\"chartClick\",\"elementClick\",\"elementDblClick\",\"elementMouseover\",\"elementMouseout\",\"elementMousemove\",\"renderEnd\"),v=!0,w=a.utils.renderWatch(u,0);return b.highlightPoint=function(a,b){k.select(\".nv-bars .nv-bar-0-\"+a).classed(\"hover\",b)},b.clearHighlights=function(){k.select(\".nv-bars .nv-bar.hover\").classed(\"hover\",!1)},b.dispatch=u,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return h},set:function(a){h=a}},height:{get:function(){return i},set:function(a){i=a}},forceX:{get:function(){return p},set:function(a){p=a}},forceY:{get:function(){return q},set:function(a){q=a}},padData:{get:function(){return r},set:function(a){r=a}},x:{get:function(){return n},set:function(a){n=a}},y:{get:function(){return o},set:function(a){o=a}},\
xScale:{get:function(){return l},set:function(a){l=a}},yScale:{get:function(){return m},set:function(a){m=a}},xDomain:{get:function(){return c},set:function(a){c=a}},yDomain:{get:function(){return d},set:function(a){d=a}},xRange:{get:function(){return e},set:function(a){e=a}},yRange:{get:function(){return f},set:function(a){f=a}},clipEdge:{get:function(){return s},set:function(a){s=a}},id:{get:function(){return j},set:function(a){j=a}},interactive:{get:function(){return v},set:function(a){v=a}},margin:{get:function(){return g},set:function(a){g.top=void 0!==a.top?a.top:g.top,g.right=void 0!==a.right?a.right:g.right,g.bottom=void 0!==a.bottom?a.bottom:g.bottom,g.left=void 0!==a.left?a.left:g.left}},color:{get:function(){return t},set:function(b){t=a.utils.getColor(b)}}}),a.utils.initOptions(b),b},a.models.historicalBarChart=function(b){\"use strict\";function c(b){return b.each(function(k){A.reset(),A.models(f),r&&A.models(g),s&&A.models(h);var x=d3.select(this);a.utils.initSVG(x);var B=a.utils.availableWidth(o,x,l),C=a.utils.availableHeight(p,x,l);if(c.update=function(){x.transition().duration(z).call(c)},c.container=this,v.disabled=k.map(function(a){return!!a.disabled}),!w){var D;w={};for(D in v)v[D]instanceof Array?w[D]=v[D].slice(0):w[D]=v[D]}if(!(k&&k.length&&k.filter(function(a){return a.values.length}).length))return a.utils.noData(c,x),c;x.selectAll(\".nv-noData\").remove(),d=f.xScale(),e=f.yScale();var E=x.selectAll(\"g.nv-wrap.nv-historicalBarChart\").data([k]),F=E.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-historicalBarChart\").append(\"g\"),G=E.select(\"g\");F.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),F.append(\"g\").attr(\"class\",\"nv-y nv-axis\"),F.append(\"g\").attr(\"class\",\"nv-barsWrap\"),F.append(\"g\").attr(\"class\",\"nv-legendWrap\"),F.append(\"g\").attr(\"class\",\"nv-interactive\"),q?(i.width(B),G.select(\".nv-legendWrap\").datum(k).call(i),m||i.height()===l.top||(l.top=i.height(),C=a.utils.availableHeight(p,x,l)),E.select(\".nv-legendWrap\").attr(\"transform\",\"translate(0,\"+-l.top+\")\")):G.select(\".nv-legendWrap\").selectAll(\"*\").remove(),\
E.attr(\"transform\",\"translate(\"+l.left+\",\"+l.top+\")\"),t&&G.select(\".nv-y.nv-axis\").attr(\"transform\",\"translate(\"+B+\",0)\"),u&&(j.width(B).height(C).margin({left:l.left,top:l.top}).svgContainer(x).xScale(d),E.select(\".nv-interactive\").call(j)),f.width(B).height(C).color(k.map(function(a,b){return a.color||n(a,b)}).filter(function(a,b){return!k[b].disabled}));var H=G.select(\".nv-barsWrap\").datum(k.filter(function(a){return!a.disabled}));H.transition().call(f),r&&(g.scale(d)._ticks(a.utils.calcTicksX(B/100,k)).tickSize(-C,0),G.select(\".nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+e.range()[0]+\")\"),G.select(\".nv-x.nv-axis\").transition().call(g)),s&&(h.scale(e)._ticks(a.utils.calcTicksY(C/36,k)).tickSize(-B,0),G.select(\".nv-y.nv-axis\").transition().call(h)),j.dispatch.on(\"elementMousemove\",function(b){f.clearHighlights();var d,e,i,l=[];k.filter(function(a,b){return a.seriesIndex=b,!a.disabled}).forEach(function(g,h){e=a.interactiveBisect(g.values,b.pointXValue,c.x()),f.highlightPoint(e,!0);var j=g.values[e];void 0!==j&&(void 0===d&&(d=j),void 0===i&&(i=c.xScale()(c.x()(j,e))),l.push({key:g.key,value:c.y()(j,e),color:n(g,g.seriesIndex),data:g.values[e]}))});var m=g.tickFormat()(c.x()(d,e));j.tooltip.valueFormatter(function(a,b){return h.tickFormat()(a)}).data({value:m,index:e,series:l})(),j.renderGuideLine(i)}),j.dispatch.on(\"elementMouseout\",function(a){y.tooltipHide(),f.clearHighlights()}),i.dispatch.on(\"legendClick\",function(a,d){a.disabled=!a.disabled,k.filter(function(a){return!a.disabled}).length||k.map(function(a){return a.disabled=!1,E.selectAll(\".nv-series\").classed(\"disabled\",!1),a}),v.disabled=k.map(function(a){return!!a.disabled}),y.stateChange(v),b.transition().call(c)}),i.dispatch.on(\"legendDblclick\",function(a){k.forEach(function(a){a.disabled=!0}),a.disabled=!1,v.disabled=k.map(function(a){return!!a.disabled}),y.stateChange(v),c.update()}),y.on(\"changeState\",function(a){\"undefined\"!=typeof a.disabled&&(k.forEach(function(b,c){b.disabled=a.disabled[c]}),v.disabled=a.disabled),c.update()})}),A.renderEnd(\"historicalBarChart \
immediate\"),c}var d,e,f=b||a.models.historicalBar(),g=a.models.axis(),h=a.models.axis(),i=a.models.legend(),j=a.interactiveGuideline(),k=a.models.tooltip(),l={top:30,right:90,bottom:50,left:90},m=null,n=a.utils.defaultColor(),o=null,p=null,q=!1,r=!0,s=!0,t=!1,u=!1,v={},w=null,x=null,y=d3.dispatch(\"tooltipHide\",\"stateChange\",\"changeState\",\"renderEnd\"),z=250;g.orient(\"bottom\").tickPadding(7),h.orient(t?\"right\":\"left\"),k.duration(0).headerEnabled(!1).valueFormatter(function(a,b){return h.tickFormat()(a,b)}).headerFormatter(function(a,b){return g.tickFormat()(a,b)});var A=a.utils.renderWatch(y,0);return f.dispatch.on(\"elementMouseover.tooltip\",function(a){a.series={key:c.x()(a.data),value:c.y()(a.data),color:a.color},k.data(a).hidden(!1)}),f.dispatch.on(\"elementMouseout.tooltip\",function(a){k.hidden(!0)}),f.dispatch.on(\"elementMousemove.tooltip\",function(a){k()}),c.dispatch=y,c.bars=f,c.legend=i,c.xAxis=g,c.yAxis=h,c.interactiveLayer=j,c.tooltip=k,c.options=a.utils.optionsFunc.bind(c),c._options=Object.create({},{width:{get:function(){return o},set:function(a){o=a}},height:{get:function(){return p},set:function(a){p=a}},showLegend:{get:function(){return q},set:function(a){q=a}},showXAxis:{get:function(){return r},set:function(a){r=a}},showYAxis:{get:function(){return s},set:function(a){s=a}},defaultState:{get:function(){return w},set:function(a){w=a}},noData:{get:function(){return x},set:function(a){x=a}},margin:{get:function(){return l},set:function(a){void 0!==a.top&&(l.top=a.top,m=a.top),l.right=void 0!==a.right?a.right:l.right,l.bottom=void 0!==a.bottom?a.bottom:l.bottom,l.left=void 0!==a.left?a.left:l.left}},color:{get:function(){return n},set:function(b){n=a.utils.getColor(b),i.color(n),f.color(n)}},duration:{get:function(){return z},set:function(a){z=a,A.reset(z),h.duration(z),g.duration(z)}},rightAlignYAxis:{get:function(){return t},set:function(a){t=a,h.orient(a?\"right\":\"left\")}},useInteractiveGuideline:{get:function(){return u},set:function(a){u=a,a===!0&&c.interactive(!1)}}}),a.utils.inheritOptions(c,f),a.utils.initOptions(c),\
c},a.models.ohlcBarChart=function(){var b=a.models.historicalBarChart(a.models.ohlcBar());return b.useInteractiveGuideline(!0),b.interactiveLayer.tooltip.contentGenerator(function(a){var c=a.series[0].data,d=c.open<c.close?\"2ca02c\":\"d62728\";return'<h3 style=\"color: #'+d+'\">'+a.value+\"</h3><table><tr><td>open:</td><td>\"+b.yAxis.tickFormat()(c.open)+\"</td></tr><tr><td>close:</td><td>\"+b.yAxis.tickFormat()(c.close)+\"</td></tr><tr><td>high</td><td>\"+b.yAxis.tickFormat()(c.high)+\"</td></tr><tr><td>low:</td><td>\"+b.yAxis.tickFormat()(c.low)+\"</td></tr></table>\"}),b},a.models.candlestickBarChart=function(){var b=a.models.historicalBarChart(a.models.candlestickBar());return b.useInteractiveGuideline(!0),b.interactiveLayer.tooltip.contentGenerator(function(a){var c=a.series[0].data,d=c.open<c.close?\"2ca02c\":\"d62728\";return'<h3 style=\"color: #'+d+'\">'+a.value+\"</h3><table><tr><td>open:</td><td>\"+b.yAxis.tickFormat()(c.open)+\"</td></tr><tr><td>close:</td><td>\"+b.yAxis.tickFormat()(c.close)+\"</td></tr><tr><td>high</td><td>\"+b.yAxis.tickFormat()(c.high)+\"</td></tr><tr><td>low:</td><td>\"+b.yAxis.tickFormat()(c.low)+\"</td></tr></table>\";\
}),b},a.models.legend=function(){\"use strict\";function b(r){function s(a,b){return\"furious\"!=q?\"#000\":o?a.disengaged?\"#000\":\"#fff\":o?void 0:(a.color||(a.color=h(a,b)),a.disabled?a.color:\"#fff\")}function t(a,b){return o&&\"furious\"==q&&a.disengaged?\"#eee\":a.color||h(a,b)}function u(a,b){return o&&\"furious\"==q?1:a.disabled?0:1}return r.each(function(b){var h=d-c.left-c.right,r=d3.select(this);a.utils.initSVG(r);var v=r.selectAll(\"g.nv-legend\").data([b]),w=v.enter().append(\"g\").attr(\"class\",\"nvd3 nv-legend\").append(\"g\"),x=v.select(\"g\");l?v.attr(\"transform\",\"translate(\"+-c.right+\",\"+c.top+\")\"):v.attr(\"transform\",\"translate(\"+c.left+\",\"+c.top+\")\");var y,z,A=x.selectAll(\".nv-series\").data(function(a){return\"furious\"!=q?a:a.filter(function(a){return o?!0:!a.disengaged})}),B=A.enter().append(\"g\").attr(\"class\",\"nv-series\");switch(q){case\"furious\":z=23;break;case\"classic\":z=20}if(\"classic\"==q)B.append(\"circle\").style(\"stroke-width\",2).attr(\"class\",\"nv-legend-symbol\").attr(\"r\",5),y=A.select(\".nv-legend-symbol\");else if(\"furious\"==q){B.append(\"rect\").style(\"stroke-width\",2).attr(\"class\",\"nv-legend-symbol\").attr(\"rx\",3).attr(\"ry\",3),y=A.select(\".nv-legend-symbol\"),B.append(\"g\").attr(\"class\",\"nv-check-box\").property(\"innerHTML\",'<path d=\"M0.5,5 L22.5,5 L22.5,26.5 L0.5,26.5 L0.5,5 Z\" class=\"nv-box\"></path><path d=\"M5.5,12.8618467 L11.9185089,19.2803556 L31,0.198864511\" class=\"nv-check\"></path>').attr(\"transform\",\"translate(-10,-8)scale(0.5)\");var C=A.select(\".nv-check-box\");C.each(function(a,b){d3.select(this).selectAll(\"path\").attr(\"stroke\",s(a,b))})}B.append(\"text\").attr(\"text-anchor\",\"start\").attr(\"class\",\"nv-legend-text\").attr(\"dy\",\".32em\").attr(\"dx\",\"8\");var D=A.select(\"text.nv-legend-text\");A.on(\"mouseover\",function(a,b){p.legendMouseover(a,b)}).on(\"mouseout\",function(a,b){p.legendMouseout(a,b)}).on(\"click\",function(a,b){p.legendClick(a,b);var c=A.data();if(m){if(\"classic\"==q)n?(c.forEach(function(a){a.disabled=!0}),a.disabled=!1):(a.disabled=!a.disabled,c.every(function(a){return a.disabled})&&c.forEach(function(a){a.disabled=!1}));\
else if(\"furious\"==q)if(o)a.disengaged=!a.disengaged,a.userDisabled=void 0==a.userDisabled?!!a.disabled:a.userDisabled,a.disabled=a.disengaged||a.userDisabled;else if(!o){a.disabled=!a.disabled,a.userDisabled=a.disabled;var d=c.filter(function(a){return!a.disengaged});d.every(function(a){return a.userDisabled})&&c.forEach(function(a){a.disabled=a.userDisabled=!1})}p.stateChange({disabled:c.map(function(a){return!!a.disabled}),disengaged:c.map(function(a){return!!a.disengaged})})}}).on(\"dblclick\",function(a,b){if((\"furious\"!=q||!o)&&(p.legendDblclick(a,b),m)){var c=A.data();c.forEach(function(a){a.disabled=!0,\"furious\"==q&&(a.userDisabled=a.disabled)}),a.disabled=!1,\"furious\"==q&&(a.userDisabled=a.disabled),p.stateChange({disabled:c.map(function(a){return!!a.disabled})})}}),A.classed(\"nv-disabled\",function(a){return a.userDisabled}),A.exit().remove(),D.attr(\"fill\",s).text(function(a){return g(f(a))});var E=0;if(j){var F=[];A.each(function(b,c){var d;if(g(f(b))&&g(f(b)).length>i){var e=g(f(b)).substring(0,i);d=d3.select(this).select(\"text\").text(e+\"...\"),d3.select(this).append(\"svg:title\").text(g(f(b)))}else d=d3.select(this).select(\"text\");var h;try{if(h=d.node().getComputedTextLength(),0>=h)throw Error()}catch(j){h=a.utils.calcApproxTextWidth(d)}F.push(h+k)});var G=0,H=[];for(E=0;h>E&&G<F.length;)H[G]=F[G],E+=F[G++];for(0===G&&(G=1);E>h&&G>1;){H=[],G--;for(var I=0;I<F.length;I++)F[I]>(H[I%G]||0)&&(H[I%G]=F[I]);E=H.reduce(function(a,b,c,d){return a+b})}for(var J=[],K=0,L=0;G>K;K++)J[K]=L,L+=H[K];A.attr(\"transform\",function(a,b){return\"translate(\"+J[b%G]+\",\"+(5+Math.floor(b/G)*z)+\")\"}),l?x.attr(\"transform\",\"translate(\"+(d-c.right-E)+\",\"+c.top+\")\"):x.attr(\"transform\",\"translate(0,\"+c.top+\")\"),e=c.top+c.bottom+Math.ceil(F.length/G)*z}else{var M,N=5,O=5,P=0;A.attr(\"transform\",function(a,b){var e=d3.select(this).select(\"text\").node().getComputedTextLength()+k;return M=O,d<c.left+c.right+M+e&&(O=M=5,N+=z),O+=e,O>P&&(P=O),M+P>E&&(E=M+P),\"translate(\"+M+\",\"+N+\")\"}),x.attr(\"transform\",\"translate(\"+(d-c.right-P)+\",\"+c.top+\")\"),\
e=c.top+c.bottom+N+15}if(\"furious\"==q){y.attr(\"width\",function(a,b){return D[0][b].getComputedTextLength()+27}).attr(\"height\",18).attr(\"y\",-9).attr(\"x\",-15),w.insert(\"rect\",\":first-child\").attr(\"class\",\"nv-legend-bg\").attr(\"fill\",\"#eee\").attr(\"opacity\",0);var Q=x.select(\".nv-legend-bg\");Q.transition().duration(300).attr(\"x\",-z).attr(\"width\",E+z-12).attr(\"height\",e+10).attr(\"y\",-c.top-10).attr(\"opacity\",o?1:0)}y.style(\"fill\",t).style(\"fill-opacity\",u).style(\"stroke\",t)}),b}var c={top:5,right:0,bottom:5,left:0},d=400,e=20,f=function(a){return a.key},g=function(a){return a},h=a.utils.getColor(),i=20,j=!0,k=32,l=!0,m=!0,n=!1,o=!1,p=d3.dispatch(\"legendClick\",\"legendDblclick\",\"legendMouseover\",\"legendMouseout\",\"stateChange\"),q=\"classic\";return b.dispatch=p,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return d},set:function(a){d=a}},height:{get:function(){return e},set:function(a){e=a}},key:{get:function(){return f},set:function(a){f=a}},keyFormatter:{get:function(){return g},set:function(a){g=a}},align:{get:function(){return j},set:function(a){j=a}},maxKeyLength:{get:function(){return i},set:function(a){i=a}},rightAlign:{get:function(){return l},set:function(a){l=a}},padding:{get:function(){return k},set:function(a){k=a}},updateState:{get:function(){return m},set:function(a){m=a}},radioButtonMode:{get:function(){return n},set:function(a){n=a}},expanded:{get:function(){return o},set:function(a){o=a}},vers:{get:function(){return q},set:function(a){q=a}},margin:{get:function(){return c},set:function(a){c.top=void 0!==a.top?a.top:c.top,c.right=void 0!==a.right?a.right:c.right,c.bottom=void 0!==a.bottom?a.bottom:c.bottom,c.left=void 0!==a.left?a.left:c.left}},color:{get:function(){return h},set:function(b){h=a.utils.getColor(b)}}}),a.utils.initOptions(b),b},a.models.line=function(){\"use strict\";function b(r){return v.reset(),v.models(e),r.each(function(b){i=d3.select(this);var r=a.utils.availableWidth(g,i,f),s=a.utils.availableHeight(h,i,f);a.utils.initSVG(i),c=e.xScale(),d=e.yScale(),\
t=t||c,u=u||d;var w=i.selectAll(\"g.nv-wrap.nv-line\").data([b]),x=w.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-line\"),y=x.append(\"defs\"),z=x.append(\"g\"),A=w.select(\"g\");z.append(\"g\").attr(\"class\",\"nv-groups\"),z.append(\"g\").attr(\"class\",\"nv-scatterWrap\"),w.attr(\"transform\",\"translate(\"+f.left+\",\"+f.top+\")\"),e.width(r).height(s);var B=w.select(\".nv-scatterWrap\");B.call(e),y.append(\"clipPath\").attr(\"id\",\"nv-edge-clip-\"+e.id()).append(\"rect\"),w.select(\"#nv-edge-clip-\"+e.id()+\" rect\").attr(\"width\",r).attr(\"height\",s>0?s:0),A.attr(\"clip-path\",p?\"url(#nv-edge-clip-\"+e.id()+\")\":\"\"),B.attr(\"clip-path\",p?\"url(#nv-edge-clip-\"+e.id()+\")\":\"\");var C=w.select(\".nv-groups\").selectAll(\".nv-group\").data(function(a){return a},function(a){return a.key});C.enter().append(\"g\").style(\"stroke-opacity\",1e-6).style(\"stroke-width\",function(a){return a.strokeWidth||j}).style(\"fill-opacity\",1e-6),C.exit().remove(),C.attr(\"class\",function(a,b){return(a.classed||\"\")+\" nv-group nv-series-\"+b}).classed(\"hover\",function(a){return a.hover}).style(\"fill\",function(a,b){return k(a,b)}).style(\"stroke\",function(a,b){return k(a,b)}),C.watchTransition(v,\"line: groups\").style(\"stroke-opacity\",1).style(\"fill-opacity\",function(a){return a.fillOpacity||.5});var D=C.selectAll(\"path.nv-area\").data(function(a){return o(a)?[a]:[]});D.enter().append(\"path\").attr(\"class\",\"nv-area\").attr(\"d\",function(b){return d3.svg.area().interpolate(q).defined(n).x(function(b,c){return a.utils.NaNtoZero(t(l(b,c)))}).y0(function(b,c){return a.utils.NaNtoZero(u(m(b,c)))}).y1(function(a,b){return u(d.domain()[0]<=0?d.domain()[1]>=0?0:d.domain()[1]:d.domain()[0])}).apply(this,[b.values])}),C.exit().selectAll(\"path.nv-area\").remove(),D.watchTransition(v,\"line: areaPaths\").attr(\"d\",function(b){return d3.svg.area().interpolate(q).defined(n).x(function(b,d){return a.utils.NaNtoZero(c(l(b,d)))}).y0(function(b,c){return a.utils.NaNtoZero(d(m(b,c)))}).y1(function(a,b){return d(d.domain()[0]<=0?d.domain()[1]>=0?0:d.domain()[1]:d.domain()[0])}).apply(this,[b.values])});var E=C.selectAll(\"path.nv-line\").data(function(a){return[a.values]});\
E.enter().append(\"path\").attr(\"class\",\"nv-line\").attr(\"d\",d3.svg.line().interpolate(q).defined(n).x(function(b,c){return a.utils.NaNtoZero(t(l(b,c)))}).y(function(b,c){return a.utils.NaNtoZero(u(m(b,c)))})),E.watchTransition(v,\"line: linePaths\").attr(\"d\",d3.svg.line().interpolate(q).defined(n).x(function(b,d){return a.utils.NaNtoZero(c(l(b,d)))}).y(function(b,c){return a.utils.NaNtoZero(d(m(b,c)))})),t=c.copy(),u=d.copy()}),v.renderEnd(\"line immediate\"),b}var c,d,e=a.models.scatter(),f={top:0,right:0,bottom:0,left:0},g=960,h=500,i=null,j=1.5,k=a.utils.defaultColor(),l=function(a){return a.x},m=function(a){return a.y},n=function(a,b){return!isNaN(m(a,b))&&null!==m(a,b)},o=function(a){return a.area},p=!1,q=\"linear\",r=250,s=d3.dispatch(\"elementClick\",\"elementMouseover\",\"elementMouseout\",\"renderEnd\");e.pointSize(16).pointDomain([16,256]);var t,u,v=a.utils.renderWatch(s,r);return b.dispatch=s,b.scatter=e,e.dispatch.on(\"elementClick\",function(){s.elementClick.apply(this,arguments)}),e.dispatch.on(\"elementMouseover\",function(){s.elementMouseover.apply(this,arguments)}),e.dispatch.on(\"elementMouseout\",function(){s.elementMouseout.apply(this,arguments)}),b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return g},set:function(a){g=a}},height:{get:function(){return h},set:function(a){h=a}},defined:{get:function(){return n},set:function(a){n=a}},interpolate:{get:function(){return q},set:function(a){q=a}},clipEdge:{get:function(){return p},set:function(a){p=a}},margin:{get:function(){return f},set:function(a){f.top=void 0!==a.top?a.top:f.top,f.right=void 0!==a.right?a.right:f.right,f.bottom=void 0!==a.bottom?a.bottom:f.bottom,f.left=void 0!==a.left?a.left:f.left}},duration:{get:function(){return r},set:function(a){r=a,v.reset(r),e.duration(r)}},isArea:{get:function(){return o},set:function(a){o=d3.functor(a)}},x:{get:function(){return l},set:function(a){l=a,e.x(a)}},y:{get:function(){return m},set:function(a){m=a,e.y(a)}},color:{get:function(){return k},set:function(b){k=a.utils.getColor(b),\
e.color(k)}}}),a.utils.inheritOptions(b,e),a.utils.initOptions(b),b},a.models.lineChart=function(){\"use strict\";function b(j){return C.reset(),C.models(e),s&&C.models(f),t&&C.models(g),j.each(function(j){function z(){s&&M.select(\".nv-focus .nv-x.nv-axis\").transition().duration(B).call(f)}function C(){t&&M.select(\".nv-focus .nv-y.nv-axis\").transition().duration(B).call(g)}function F(a){var b=M.select(\".nv-focus .nv-linesWrap\").datum(j.filter(function(a){return!a.disabled}).map(function(b,c){return{key:b.key,area:b.area,classed:b.classed,values:b.values.filter(function(b,c){return e.x()(b,c)>=a[0]&&e.x()(b,c)<=a[1]}),disableTooltip:b.disableTooltip}}));b.transition().duration(B).call(e),z(),C()}var G=d3.select(this);a.utils.initSVG(G);var H=a.utils.availableWidth(o,G,l),I=a.utils.availableHeight(p,G,l)-(w?k.height():0);if(b.update=function(){0===B?G.call(b):G.transition().duration(B).call(b)},b.container=this,x.setter(E(j),b.update).getter(D(j)).update(),x.disabled=j.map(function(a){return!!a.disabled}),!y){var J;y={};for(J in x)x[J]instanceof Array?y[J]=x[J].slice(0):y[J]=x[J]}if(!(j&&j.length&&j.filter(function(a){return a.values.length}).length))return a.utils.noData(b,G),b;G.selectAll(\".nv-noData\").remove(),k.dispatch.on(\"onBrush\",function(a){F(a)}),c=e.xScale(),d=e.yScale();var K=G.selectAll(\"g.nv-wrap.nv-lineChart\").data([j]),L=K.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-lineChart\").append(\"g\"),M=K.select(\"g\");L.append(\"g\").attr(\"class\",\"nv-legendWrap\");var N=L.append(\"g\").attr(\"class\",\"nv-focus\");N.append(\"g\").attr(\"class\",\"nv-background\").append(\"rect\"),N.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),N.append(\"g\").attr(\"class\",\"nv-y nv-axis\"),N.append(\"g\").attr(\"class\",\"nv-linesWrap\"),N.append(\"g\").attr(\"class\",\"nv-interactive\");L.append(\"g\").attr(\"class\",\"nv-focusWrap\");q?(h.width(H),M.select(\".nv-legendWrap\").datum(j).call(h),\"bottom\"===r?K.select(\".nv-legendWrap\").attr(\"transform\",\"translate(0,\"+I+\")\"):\"top\"===r&&(m||h.height()===l.top||(l.top=h.height(),I=a.utils.availableHeight(p,G,l)-(w?k.height():0)),\
K.select(\".nv-legendWrap\").attr(\"transform\",\"translate(0,\"+-l.top+\")\"))):M.select(\".nv-legendWrap\").selectAll(\"*\").remove(),K.attr(\"transform\",\"translate(\"+l.left+\",\"+l.top+\")\"),u&&M.select(\".nv-y.nv-axis\").attr(\"transform\",\"translate(\"+H+\",0)\"),v&&(i.width(H).height(I).margin({left:l.left,top:l.top}).svgContainer(G).xScale(c),K.select(\".nv-interactive\").call(i)),M.select(\".nv-focus .nv-background rect\").attr(\"width\",H).attr(\"height\",I),e.width(H).height(I).color(j.map(function(a,b){return a.color||n(a,b)}).filter(function(a,b){return!j[b].disabled}));var O=M.select(\".nv-linesWrap\").datum(j.filter(function(a){return!a.disabled}));if(s&&f.scale(c)._ticks(a.utils.calcTicksX(H/100,j)).tickSize(-I,0),t&&g.scale(d)._ticks(a.utils.calcTicksY(I/36,j)).tickSize(-H,0),M.select(\".nv-focus .nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+I+\")\"),w){k.width(H),M.select(\".nv-focusWrap\").attr(\"transform\",\"translate(0,\"+(I+l.bottom+k.margin().top)+\")\").datum(j.filter(function(a){return!a.disabled})).call(k);var P=k.brush.empty()?k.xDomain():k.brush.extent();null!==P&&F(P)}else O.call(e),z(),C();h.dispatch.on(\"stateChange\",function(a){for(var c in a)x[c]=a[c];A.stateChange(x),b.update()}),i.dispatch.on(\"elementMousemove\",function(d){e.clearHighlights();var f,h,l,m=[];if(j.filter(function(a,b){return a.seriesIndex=b,!a.disabled&&!a.disableTooltip}).forEach(function(g,i){var j=w?k.brush.empty()?k.xScale().domain():k.brush.extent():c.domain(),o=g.values.filter(function(a,b){return j[0]<=j[1]?e.x()(a,b)>=j[0]&&e.x()(a,b)<=j[1]:e.x()(a,b)>=j[1]&&e.x()(a,b)<=j[0]});h=a.interactiveBisect(o,d.pointXValue,e.x());var p=o[h],q=b.y()(p,h);null!==q&&e.highlightPoint(i,h,!0),void 0!==p&&(void 0===f&&(f=p),void 0===l&&(l=b.xScale()(b.x()(p,h))),m.push({key:g.key,value:q,color:n(g,g.seriesIndex),data:p}))}),m.length>2){var o=b.yScale().invert(d.mouseY),p=Math.abs(b.yScale().domain()[0]-b.yScale().domain()[1]),q=.03*p,r=a.nearestValueIndex(m.map(function(a){return a.value}),o,q);null!==r&&(m[r].highlight=!0)}var s=function(a,b){return null==a?\"N/A\":g.tickFormat()(a)};\
i.tooltip.valueFormatter(i.tooltip.valueFormatter()||s).data({value:b.x()(f,h),index:h,series:m})(),i.renderGuideLine(l)}),i.dispatch.on(\"elementClick\",function(c){var d,f=[];j.filter(function(a,b){return a.seriesIndex=b,!a.disabled}).forEach(function(e){var g=a.interactiveBisect(e.values,c.pointXValue,b.x()),h=e.values[g];if(\"undefined\"!=typeof h){\"undefined\"==typeof d&&(d=b.xScale()(b.x()(h,g)));var i=b.yScale()(b.y()(h,g));f.push({point:h,pointIndex:g,pos:[d,i],seriesIndex:e.seriesIndex,series:e})}}),e.dispatch.elementClick(f)}),i.dispatch.on(\"elementMouseout\",function(a){e.clearHighlights()}),A.on(\"changeState\",function(a){\"undefined\"!=typeof a.disabled&&j.length===a.disabled.length&&(j.forEach(function(b,c){b.disabled=a.disabled[c]}),x.disabled=a.disabled),b.update()})}),C.renderEnd(\"lineChart immediate\"),b}var c,d,e=a.models.line(),f=a.models.axis(),g=a.models.axis(),h=a.models.legend(),i=a.interactiveGuideline(),j=a.models.tooltip(),k=a.models.focus(a.models.line()),l={top:30,right:20,bottom:50,left:60},m=null,n=a.utils.defaultColor(),o=null,p=null,q=!0,r=\"top\",s=!0,t=!0,u=!1,v=!1,w=!1,x=a.utils.state(),y=null,z=null,A=d3.dispatch(\"tooltipShow\",\"tooltipHide\",\"stateChange\",\"changeState\",\"renderEnd\"),B=250;f.orient(\"bottom\").tickPadding(7),g.orient(u?\"right\":\"left\"),e.clipEdge(!0).duration(0),j.valueFormatter(function(a,b){return g.tickFormat()(a,b)}).headerFormatter(function(a,b){return f.tickFormat()(a,b)}),i.tooltip.valueFormatter(function(a,b){return g.tickFormat()(a,b)}).headerFormatter(function(a,b){return f.tickFormat()(a,b)});var C=a.utils.renderWatch(A,B),D=function(a){return function(){return{active:a.map(function(a){return!a.disabled})}}},E=function(a){return function(b){void 0!==b.active&&a.forEach(function(a,c){a.disabled=!b.active[c]})}};return e.dispatch.on(\"elementMouseover.tooltip\",function(a){a.series.disableTooltip||j.data(a).hidden(!1)}),e.dispatch.on(\"elementMouseout.tooltip\",function(a){j.hidden(!0)}),b.dispatch=A,b.lines=e,b.legend=h,b.focus=k,b.xAxis=f,b.x2Axis=k.xAxis,b.yAxis=g,b.y2Axis=k.yAxis,\
b.interactiveLayer=i,b.tooltip=j,b.state=x,b.dispatch=A,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return o},set:function(a){o=a}},height:{get:function(){return p},set:function(a){p=a}},showLegend:{get:function(){return q},set:function(a){q=a}},legendPosition:{get:function(){return r},set:function(a){r=a}},showXAxis:{get:function(){return s},set:function(a){s=a}},showYAxis:{get:function(){return t},set:function(a){t=a}},defaultState:{get:function(){return y},set:function(a){y=a}},noData:{get:function(){return z},set:function(a){z=a}},focusEnable:{get:function(){return w},set:function(a){w=a}},focusHeight:{get:function(){return k.height()},set:function(a){k.height(a)}},focusShowAxisX:{get:function(){return k.showXAxis()},set:function(a){k.showXAxis(a)}},focusShowAxisY:{get:function(){return k.showYAxis()},set:function(a){k.showYAxis(a)}},brushExtent:{get:function(){return k.brushExtent()},set:function(a){k.brushExtent(a)}},focusMargin:{get:function(){return k.margin},set:function(a){void 0!==a.top&&(l.top=a.top,m=a.top),k.margin.right=void 0!==a.right?a.right:k.margin.right,k.margin.bottom=void 0!==a.bottom?a.bottom:k.margin.bottom,k.margin.left=void 0!==a.left?a.left:k.margin.left}},margin:{get:function(){return l},set:function(a){l.top=void 0!==a.top?a.top:l.top,l.right=void 0!==a.right?a.right:l.right,l.bottom=void 0!==a.bottom?a.bottom:l.bottom,l.left=void 0!==a.left?a.left:l.left}},duration:{get:function(){return B},set:function(a){B=a,C.reset(B),e.duration(B),k.duration(B),f.duration(B),g.duration(B)}},color:{get:function(){return n},set:function(b){n=a.utils.getColor(b),h.color(n),e.color(n),k.color(n)}},interpolate:{get:function(){return e.interpolate()},set:function(a){e.interpolate(a),k.interpolate(a)}},xTickFormat:{get:function(){return f.tickFormat()},set:function(a){f.tickFormat(a),k.xTickFormat(a)}},yTickFormat:{get:function(){return g.tickFormat()},set:function(a){g.tickFormat(a),k.yTickFormat(a)}},x:{get:function(){return e.x()},set:function(a){e.x(a),k.x(a)}},\
y:{get:function(){return e.y()},set:function(a){e.y(a),k.y(a)}},rightAlignYAxis:{get:function(){return u},set:function(a){u=a,g.orient(u?\"right\":\"left\")}},useInteractiveGuideline:{get:function(){return v},set:function(a){v=a,v&&(e.interactive(!1),e.useVoronoi(!1))}}}),a.utils.inheritOptions(b,e),a.utils.initOptions(b),b},a.models.lineWithFocusChart=function(){return a.models.lineChart().margin({bottom:30}).focusEnable(!0)},a.models.linePlusBarChart=function(){\"use strict\";function b(v){return v.each(function(v){function K(a){var b=+(\"e\"==a),c=b?1:-1,d=$/3;return\"M\"+.5*c+\",\"+d+\"A6,6 0 0 \"+b+\" \"+6.5*c+\",\"+(d+6)+\"V\"+(2*d-6)+\"A6,6 0 0 \"+b+\" \"+.5*c+\",\"+2*d+\"ZM\"+2.5*c+\",\"+(d+8)+\"V\"+(2*d-8)+\"M\"+4.5*c+\",\"+(d+8)+\"V\"+(2*d-8)}function S(){u.empty()||u.extent(J),na.data([u.empty()?e.domain():J]).each(function(a,b){var c=e(a[0])-e.range()[0],d=e.range()[1]-e(a[1]);d3.select(this).select(\".left\").attr(\"width\",0>c?0:c),d3.select(this).select(\".right\").attr(\"x\",e(a[1])).attr(\"width\",0>d?0:d)})}function T(){J=u.empty()?null:u.extent(),c=u.empty()?e.domain():u.extent(),L.brush({extent:c,brush:u}),S(),l.width(Y).height(Z).color(v.map(function(a,b){return a.color||D(a,b)}).filter(function(a,b){return!v[b].disabled&&v[b].bar})),j.width(Y).height(Z).color(v.map(function(a,b){return a.color||D(a,b)}).filter(function(a,b){return!v[b].disabled&&!v[b].bar}));var b=ga.select(\".nv-focus .nv-barsWrap\").datum(aa.length?aa.map(function(a,b){return{key:a.key,values:a.values.filter(function(a,b){return l.x()(a,b)>=c[0]&&l.x()(a,b)<=c[1]})}}):[{values:[]}]),h=ga.select(\".nv-focus .nv-linesWrap\").datum(W(ba)?[{values:[]}]:ba.filter(function(a){return!a.disabled}).map(function(a,b){return{area:a.area,fillOpacity:a.fillOpacity,strokeWidth:a.strokeWidth,key:a.key,values:a.values.filter(function(a,b){return j.x()(a,b)>=c[0]&&j.x()(a,b)<=c[1]})}}));d=aa.length&&!R?l.xScale():j.xScale(),n.scale(d)._ticks(a.utils.calcTicksX(Y/100,v)).tickSize(-Z,0),n.domain([Math.ceil(c[0]),Math.floor(c[1])]),ga.select(\".nv-x.nv-axis\").transition().duration(M).call(n),b.transition().duration(M).call(l),\
")

file(APPEND "${METABENCH_DIR}/nvd3.js" "\
h.transition().duration(M).call(j),ga.select(\".nv-focus .nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+f.range()[0]+\")\"),p.scale(f)._ticks(a.utils.calcTicksY(Z/36,v)).tickSize(-Y,0),q.scale(g)._ticks(a.utils.calcTicksY(Z/36,v)),R?q.tickSize(ba.length?0:-Y,0):q.tickSize(aa.length?0:-Y,0);var i=aa.length?1:0,k=ba.length&&!W(ba)?1:0,m=R?k:i,o=R?i:k;ga.select(\".nv-focus .nv-y1.nv-axis\").style(\"opacity\",m),ga.select(\".nv-focus .nv-y2.nv-axis\").style(\"opacity\",o).attr(\"transform\",\"translate(\"+d.range()[1]+\",0)\"),ga.select(\".nv-focus .nv-y1.nv-axis\").transition().duration(M).call(p),ga.select(\".nv-focus .nv-y2.nv-axis\").transition().duration(M).call(q)}var X=d3.select(this);a.utils.initSVG(X);var Y=a.utils.availableWidth(z,X,w),Z=a.utils.availableHeight(A,X,w)-(F?I:0),$=I-y.top-y.bottom;if(b.update=function(){X.transition().duration(M).call(b)},b.container=this,N.setter(V(v),b.update).getter(U(v)).update(),N.disabled=v.map(function(a){return!!a.disabled}),!O){var _;O={};for(_ in N)N[_]instanceof Array?O[_]=N[_].slice(0):O[_]=N[_]}if(!(v&&v.length&&v.filter(function(a){return a.values.length}).length))return a.utils.noData(b,X),b;X.selectAll(\".nv-noData\").remove();var aa=v.filter(function(a){return!a.disabled&&a.bar}),ba=v.filter(function(a){return!a.bar});d=aa.length&&!R?l.xScale():j.xScale(),e=o.scale(),f=R?j.yScale():l.yScale(),g=R?l.yScale():j.yScale(),h=R?k.yScale():m.yScale(),i=R?m.yScale():k.yScale();var ca=v.filter(function(a){return!a.disabled&&(R?!a.bar:a.bar)}).map(function(a){return a.values.map(function(a,b){return{x:B(a,b),y:C(a,b)}})}),da=v.filter(function(a){return!a.disabled&&(R?a.bar:!a.bar)}).map(function(a){return a.values.map(function(a,b){return{x:B(a,b),y:C(a,b)}})});d.range([0,Y]),e.domain(d3.extent(d3.merge(ca.concat(da)),function(a){return a.x})).range([0,Y]);var ea=X.selectAll(\"g.nv-wrap.nv-linePlusBar\").data([v]),fa=ea.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-linePlusBar\").append(\"g\"),ga=ea.select(\"g\");fa.append(\"g\").attr(\"class\",\"nv-legendWrap\");var ha=fa.append(\"g\").attr(\"class\",\"nv-focus\");\
ha.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),ha.append(\"g\").attr(\"class\",\"nv-y1 nv-axis\"),ha.append(\"g\").attr(\"class\",\"nv-y2 nv-axis\"),ha.append(\"g\").attr(\"class\",\"nv-barsWrap\"),ha.append(\"g\").attr(\"class\",\"nv-linesWrap\");var ia=fa.append(\"g\").attr(\"class\",\"nv-context\");if(ia.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),ia.append(\"g\").attr(\"class\",\"nv-y1 nv-axis\"),ia.append(\"g\").attr(\"class\",\"nv-y2 nv-axis\"),ia.append(\"g\").attr(\"class\",\"nv-barsWrap\"),ia.append(\"g\").attr(\"class\",\"nv-linesWrap\"),ia.append(\"g\").attr(\"class\",\"nv-brushBackground\"),ia.append(\"g\").attr(\"class\",\"nv-x nv-brush\"),E){var ja=t.align()?Y/2:Y,ka=t.align()?ja:0;t.width(ja),ga.select(\".nv-legendWrap\").datum(v.map(function(a){return a.originalKey=void 0===a.originalKey?a.key:a.originalKey,R?a.key=a.originalKey+(a.bar?Q:P):a.key=a.originalKey+(a.bar?P:Q),a})).call(t),x||t.height()===w.top||(w.top=t.height(),Z=a.utils.availableHeight(A,X,w)-I),ga.select(\".nv-legendWrap\").attr(\"transform\",\"translate(\"+ka+\",\"+-w.top+\")\")}else ga.select(\".nv-legendWrap\").selectAll(\"*\").remove();ea.attr(\"transform\",\"translate(\"+w.left+\",\"+w.top+\")\"),ga.select(\".nv-context\").style(\"display\",F?\"initial\":\"none\"),m.width(Y).height($).color(v.map(function(a,b){return a.color||D(a,b)}).filter(function(a,b){return!v[b].disabled&&v[b].bar})),k.width(Y).height($).color(v.map(function(a,b){return a.color||D(a,b)}).filter(function(a,b){return!v[b].disabled&&!v[b].bar}));var la=ga.select(\".nv-context .nv-barsWrap\").datum(aa.length?aa:[{values:[]}]),ma=ga.select(\".nv-context .nv-linesWrap\").datum(W(ba)?[{values:[]}]:ba.filter(function(a){return!a.disabled}));ga.select(\".nv-context\").attr(\"transform\",\"translate(0,\"+(Z+w.bottom+y.top)+\")\"),la.transition().call(m),ma.transition().call(k),H&&(o._ticks(a.utils.calcTicksX(Y/100,v)).tickSize(-$,0),ga.select(\".nv-context .nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+h.range()[0]+\")\"),ga.select(\".nv-context .nv-x.nv-axis\").transition().call(o)),G&&(r.scale(h)._ticks($/36).tickSize(-Y,0),s.scale(i)._ticks($/36).tickSize(aa.length?0:-Y,0),ga.select(\".nv-context \
.nv-y3.nv-axis\").style(\"opacity\",aa.length?1:0).attr(\"transform\",\"translate(0,\"+e.range()[0]+\")\"),ga.select(\".nv-context .nv-y2.nv-axis\").style(\"opacity\",ba.length?1:0).attr(\"transform\",\"translate(\"+e.range()[1]+\",0)\"),ga.select(\".nv-context .nv-y1.nv-axis\").transition().call(r),ga.select(\".nv-context .nv-y2.nv-axis\").transition().call(s)),u.x(e).on(\"brush\",T),J&&u.extent(J);var na=ga.select(\".nv-brushBackground\").selectAll(\"g\").data([J||u.extent()]),oa=na.enter().append(\"g\");oa.append(\"rect\").attr(\"class\",\"left\").attr(\"x\",0).attr(\"y\",0).attr(\"height\",$),oa.append(\"rect\").attr(\"class\",\"right\").attr(\"x\",0).attr(\"y\",0).attr(\"height\",$);var pa=ga.select(\".nv-x.nv-brush\").call(u);pa.selectAll(\"rect\").attr(\"height\",$),pa.selectAll(\".resize\").append(\"path\").attr(\"d\",K),t.dispatch.on(\"stateChange\",function(a){for(var c in a)N[c]=a[c];L.stateChange(N),b.update()}),L.on(\"changeState\",function(a){\"undefined\"!=typeof a.disabled&&(v.forEach(function(b,c){b.disabled=a.disabled[c]}),N.disabled=a.disabled),b.update()}),T()}),b}var c,d,e,f,g,h,i,j=a.models.line(),k=a.models.line(),l=a.models.historicalBar(),m=a.models.historicalBar(),n=a.models.axis(),o=a.models.axis(),p=a.models.axis(),q=a.models.axis(),r=a.models.axis(),s=a.models.axis(),t=a.models.legend(),u=d3.svg.brush(),v=a.models.tooltip(),w={top:30,right:30,bottom:30,left:60},x=null,y={top:0,right:30,bottom:20,left:60},z=null,A=null,B=function(a){return a.x},C=function(a){return a.y},D=a.utils.defaultColor(),E=!0,F=!0,G=!1,H=!0,I=50,J=null,K=null,L=d3.dispatch(\"brush\",\"stateChange\",\"changeState\"),M=0,N=a.utils.state(),O=null,P=\" (left axis)\",Q=\" (right axis)\",R=!1;j.clipEdge(!0),k.interactive(!1),k.pointActive(function(a){return!1}),n.orient(\"bottom\").tickPadding(5),p.orient(\"left\"),q.orient(\"right\"),o.orient(\"bottom\").tickPadding(5),r.orient(\"left\"),s.orient(\"right\"),v.headerEnabled(!0).headerFormatter(function(a,b){return n.tickFormat()(a,b)});var S=function(){return R?{main:q,focus:s}:{main:p,focus:r}},T=function(){return R?{main:p,focus:r}:{main:q,focus:s}},U=function(a){return \
function(){return{active:a.map(function(a){return!a.disabled})}}},V=function(a){return function(b){void 0!==b.active&&a.forEach(function(a,c){a.disabled=!b.active[c]})}},W=function(a){return a.every(function(a){return a.disabled})};return j.dispatch.on(\"elementMouseover.tooltip\",function(a){v.duration(100).valueFormatter(function(a,b){return T().main.tickFormat()(a,b)}).data(a).hidden(!1)}),j.dispatch.on(\"elementMouseout.tooltip\",function(a){v.hidden(!0)}),l.dispatch.on(\"elementMouseover.tooltip\",function(a){a.value=b.x()(a.data),a.series={value:b.y()(a.data),color:a.color},v.duration(0).valueFormatter(function(a,b){return S().main.tickFormat()(a,b)}).data(a).hidden(!1)}),l.dispatch.on(\"elementMouseout.tooltip\",function(a){v.hidden(!0)}),l.dispatch.on(\"elementMousemove.tooltip\",function(a){v()}),b.dispatch=L,b.legend=t,b.lines=j,b.lines2=k,b.bars=l,b.bars2=m,b.xAxis=n,b.x2Axis=o,b.y1Axis=p,b.y2Axis=q,b.y3Axis=r,b.y4Axis=s,b.tooltip=v,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return z},set:function(a){z=a}},height:{get:function(){return A},set:function(a){A=a}},showLegend:{get:function(){return E},set:function(a){E=a}},brushExtent:{get:function(){return J},set:function(a){J=a}},noData:{get:function(){return K},set:function(a){K=a}},focusEnable:{get:function(){return F},set:function(a){F=a}},focusHeight:{get:function(){return I},set:function(a){I=a}},focusShowAxisX:{get:function(){return H},set:function(a){H=a}},focusShowAxisY:{get:function(){return G},set:function(a){G=a}},legendLeftAxisHint:{get:function(){return P},set:function(a){P=a}},legendRightAxisHint:{get:function(){return Q},set:function(a){Q=a}},margin:{get:function(){return w},set:function(a){void 0!==a.top&&(w.top=a.top,x=a.top),w.right=void 0!==a.right?a.right:w.right,w.bottom=void 0!==a.bottom?a.bottom:w.bottom,w.left=void 0!==a.left?a.left:w.left}},focusMargin:{get:function(){return y},set:function(a){y.top=void 0!==a.top?a.top:y.top,y.right=void 0!==a.right?a.right:y.right,y.bottom=void 0!==a.bottom?a.bottom:y.bottom,\
y.left=void 0!==a.left?a.left:y.left}},duration:{get:function(){return M},set:function(a){M=a}},color:{get:function(){return D},set:function(b){D=a.utils.getColor(b),t.color(D)}},x:{get:function(){return B},set:function(a){B=a,j.x(a),k.x(a),l.x(a),m.x(a)}},y:{get:function(){return C},set:function(a){C=a,j.y(a),k.y(a),l.y(a),m.y(a)}},switchYAxisOrder:{get:function(){return R},set:function(a){if(R!==a){var b=p;p=q,q=b;var c=r;r=s,s=c}R=a,p.orient(\"left\"),q.orient(\"right\"),r.orient(\"left\"),s.orient(\"right\")}}}),a.utils.inheritOptions(b,j),a.utils.initOptions(b),b},a.models.multiBar=function(){\"use strict\";function b(F){return D.reset(),F.each(function(b){var F=k-j.left-j.right,G=l-j.top-j.bottom;p=d3.select(this),a.utils.initSVG(p);var H=0;if(x&&b.length&&(x=[{values:b[0].values.map(function(a){return{x:a.x,y:0,series:a.series,size:.01}})}]),u){var I=d3.layout.stack().offset(v).values(function(a){return a.values}).y(r)(!b.length&&x?x:b);I.forEach(function(a,c){a.nonStackable?(b[c].nonStackableSeries=H++,I[c]=b[c]):c>0&&I[c-1].nonStackable&&I[c].values.map(function(a,b){a.y0-=I[c-1].values[b].y,a.y1=a.y0+a.y})}),b=I}b.forEach(function(a,b){a.values.forEach(function(c){c.series=b,c.key=a.key})}),u&&b.length>0&&b[0].values.map(function(a,c){var d=0,e=0;b.map(function(a,f){if(!b[f].nonStackable){var g=a.values[c];g.size=Math.abs(g.y),g.y<0?(g.y1=e,e-=g.size):(g.y1=g.size+d,d+=g.size)}})});var J=d&&e?[]:b.map(function(a,b){return a.values.map(function(a,c){return{x:q(a,c),y:r(a,c),y0:a.y0,y1:a.y1,idx:b}})});m.domain(d||d3.merge(J).map(function(a){return a.x})).rangeBands(f||[0,F],A),n.domain(e||d3.extent(d3.merge(J).map(function(a){var c=a.y;return u&&!b[a.idx].nonStackable&&(c=a.y>0?a.y1:a.y1+a.y),c}).concat(s))).range(g||[G,0]),m.domain()[0]===m.domain()[1]&&(m.domain()[0]?m.domain([m.domain()[0]-.01*m.domain()[0],m.domain()[1]+.01*m.domain()[1]]):m.domain([-1,1])),n.domain()[0]===n.domain()[1]&&(n.domain()[0]?n.domain([n.domain()[0]+.01*n.domain()[0],n.domain()[1]-.01*n.domain()[1]]):n.domain([-1,1])),h=h||m,i=i||n;var \
K=p.selectAll(\"g.nv-wrap.nv-multibar\").data([b]),L=K.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-multibar\"),M=L.append(\"defs\"),N=L.append(\"g\"),O=K.select(\"g\");N.append(\"g\").attr(\"class\",\"nv-groups\"),K.attr(\"transform\",\"translate(\"+j.left+\",\"+j.top+\")\"),M.append(\"clipPath\").attr(\"id\",\"nv-edge-clip-\"+o).append(\"rect\"),K.select(\"#nv-edge-clip-\"+o+\" rect\").attr(\"width\",F).attr(\"height\",G),O.attr(\"clip-path\",t?\"url(#nv-edge-clip-\"+o+\")\":\"\");var P=K.select(\".nv-groups\").selectAll(\".nv-group\").data(function(a){return a},function(a,b){return b});P.enter().append(\"g\").style(\"stroke-opacity\",1e-6).style(\"fill-opacity\",1e-6);var Q=D.transition(P.exit().selectAll(\"rect.nv-bar\"),\"multibarExit\",Math.min(100,z)).attr(\"y\",function(a,c,d){var e=i(0)||0;return u&&b[a.series]&&!b[a.series].nonStackable&&(e=i(a.y0)),e}).attr(\"height\",0).remove();Q.delay&&Q.delay(function(a,b){var c=b*(z/(E+1))-b;return c}),P.attr(\"class\",function(a,b){return\"nv-group nv-series-\"+b}).classed(\"hover\",function(a){return a.hover}).style(\"fill\",function(a,b){return w(a,b)}).style(\"stroke\",function(a,b){\
return w(a,b)}),P.style(\"stroke-opacity\",1).style(\"fill-opacity\",B);var R=P.selectAll(\"rect.nv-bar\").data(function(a){return x&&!b.length?x.values:a.values});R.exit().remove();R.enter().append(\"rect\").attr(\"class\",function(a,b){return r(a,b)<0?\"nv-bar negative\":\"nv-bar positive\"}).attr(\"x\",function(a,c,d){return u&&!b[d].nonStackable?0:d*m.rangeBand()/b.length}).attr(\"y\",function(a,c,d){return i(u&&!b[d].nonStackable?a.y0:0)||0}).attr(\"height\",0).attr(\"width\",function(a,c,d){return m.rangeBand()/(u&&!b[d].nonStackable?1:b.length)}).attr(\"transform\",function(a,b){return\"translate(\"+m(q(a,b))+\",0)\"});R.style(\"fill\",function(a,b,c){return w(a,c,b)}).style(\"stroke\",function(a,b,c){return w(a,c,b)}).on(\"mouseover\",function(a,b){d3.select(this).classed(\"hover\",!0),C.elementMouseover({data:a,index:b,color:d3.select(this).style(\"fill\")})}).on(\"mouseout\",function(a,b){d3.select(this).classed(\"hover\",!1),C.elementMouseout({data:a,index:b,color:d3.select(this).style(\"fill\")})}).on(\"mousemove\",function(a,b){C.elementMousemove({data:a,index:b,color:d3.select(this).style(\"fill\")})}).on(\"click\",function(a,b){var c=this;C.elementClick({data:a,index:b,color:d3.select(this).style(\"fill\"),event:d3.event,element:c}),d3.event.stopPropagation()}).on(\"dblclick\",function(a,b){C.elementDblClick({data:a,index:b,color:d3.select(this).style(\"fill\")}),d3.event.stopPropagation()}),R.attr(\"class\",function(a,b){return r(a,b)<0?\"nv-bar negative\":\"nv-bar positive\"}).attr(\"transform\",function(a,b){return\"translate(\"+m(q(a,b))+\",0)\"}),y&&(c||(c=b.map(function(){return!0})),R.style(\"fill\",function(a,b,d){return d3.rgb(y(a,b)).darker(c.map(function(a,b){return b}).filter(function(a,b){return!c[b]})[d]).toString()}).style(\"stroke\",function(a,b,d){return d3.rgb(y(a,b)).darker(c.map(function(a,b){return b}).filter(function(a,b){return!c[b]})[d]).toString()}));var S=R.watchTransition(D,\"multibar\",Math.min(250,z)).delay(function(a,c){return c*z/b[0].values.length});u?S.attr(\"y\",function(a,c,d){var e=0;return e=b[d].nonStackable?r(a,c)<0?n(0):n(0)-n(r(a,c))<-1?n(0)-1:n(r(a,\
c))||0:n(a.y1)}).attr(\"height\",function(a,c,d){return b[d].nonStackable?Math.max(Math.abs(n(r(a,c))-n(0)),0)||0:Math.max(Math.abs(n(a.y+a.y0)-n(a.y0)),0)}).attr(\"x\",function(a,c,d){var e=0;return b[d].nonStackable&&(e=a.series*m.rangeBand()/b.length,b.length!==H&&(e=b[d].nonStackableSeries*m.rangeBand()/(2*H))),e}).attr(\"width\",function(a,c,d){if(b[d].nonStackable){var e=m.rangeBand()/H;return b.length!==H&&(e=m.rangeBand()/(2*H)),e}return m.rangeBand()}):S.attr(\"x\",function(a,c){return a.series*m.rangeBand()/b.length}).attr(\"width\",m.rangeBand()/b.length).attr(\"y\",function(a,b){return r(a,b)<0?n(0):n(0)-n(r(a,b))<1?n(0)-1:n(r(a,b))||0}).attr(\"height\",function(a,b){return Math.max(Math.abs(n(r(a,b))-n(0)),1)||0}),h=m.copy(),i=n.copy(),b[0]&&b[0].values&&(E=b[0].values.length)}),D.renderEnd(\"multibar immediate\"),b}var c,d,e,f,g,h,i,j={top:0,right:0,bottom:0,left:0},k=960,l=500,m=d3.scale.ordinal(),n=d3.scale.linear(),o=Math.floor(1e4*Math.random()),p=null,q=function(a){return a.x},r=function(a){return a.y},s=[0],t=!0,u=!1,v=\"zero\",w=a.utils.defaultColor(),x=!1,y=null,z=500,A=.1,B=.75,C=d3.dispatch(\"chartClick\",\"elementClick\",\"elementDblClick\",\"elementMouseover\",\"elementMouseout\",\"elementMousemove\",\"renderEnd\"),D=a.utils.renderWatch(C,z),E=0;return b.dispatch=C,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return k},set:function(a){k=a}},height:{get:function(){return l},set:function(a){l=a}},x:{get:function(){return q},set:function(a){q=a}},y:{get:function(){return r},set:function(a){r=a}},xScale:{get:function(){return m},set:function(a){m=a}},yScale:{get:function(){return n},set:function(a){n=a}},xDomain:{get:function(){return d},set:function(a){d=a}},yDomain:{get:function(){return e},set:function(a){e=a}},xRange:{get:function(){return f},set:function(a){f=a}},yRange:{get:function(){return g},set:function(a){g=a}},forceY:{get:function(){return s},set:function(a){s=a}},stacked:{get:function(){return u},set:function(a){u=a}},stackOffset:{get:function(){return v},set:function(a){v=a}},\
clipEdge:{get:function(){return t},set:function(a){t=a}},disabled:{get:function(){return c},set:function(a){c=a}},id:{get:function(){return o},set:function(a){o=a}},hideable:{get:function(){return x},set:function(a){x=a}},groupSpacing:{get:function(){return A},set:function(a){A=a}},fillOpacity:{get:function(){return B},set:function(a){B=a}},margin:{get:function(){return j},set:function(a){j.top=void 0!==a.top?a.top:j.top,j.right=void 0!==a.right?a.right:j.right,j.bottom=void 0!==a.bottom?a.bottom:j.bottom,j.left=void 0!==a.left?a.left:j.left}},duration:{get:function(){return z},set:function(a){z=a,D.reset(z)}},color:{get:function(){return w},set:function(b){w=a.utils.getColor(b)}},barColor:{get:function(){return y},set:function(b){y=b?a.utils.getColor(b):null}}}),a.utils.initOptions(b),b},a.models.multiBarChart=function(){\"use strict\";function b(C){return H.reset(),H.models(e),t&&H.models(f),u&&H.models(g),C.each(function(C){var H=d3.select(this);a.utils.initSVG(H);var L=a.utils.availableWidth(n,H,l),M=a.utils.availableHeight(o,H,l);if(b.update=function(){0===F?H.call(b):H.transition().duration(F).call(b)},b.container=this,A.setter(K(C),b.update).getter(J(C)).update(),A.disabled=C.map(function(a){return!!a.disabled}),!B){var N;B={};for(N in A)A[N]instanceof Array?B[N]=A[N].slice(0):B[N]=A[N]}if(!(C&&C.length&&C.filter(function(a){return a.values.length}).length))return a.utils.noData(b,H),b;H.selectAll(\".nv-noData\").remove(),c=e.xScale(),d=e.yScale();var O=H.selectAll(\"g.nv-wrap.nv-multiBarWithLegend\").data([C]),P=O.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-multiBarWithLegend\").append(\"g\"),Q=O.select(\"g\");if(P.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),P.append(\"g\").attr(\"class\",\"nv-y nv-axis\"),P.append(\"g\").attr(\"class\",\"nv-barsWrap\"),P.append(\"g\").attr(\"class\",\"nv-legendWrap\"),P.append(\"g\").attr(\"class\",\"nv-controlsWrap\"),P.append(\"g\").attr(\"class\",\"nv-interactive\"),s?(i.width(L-E()),Q.select(\".nv-legendWrap\").datum(C).call(i),m||i.height()===l.top||(l.top=i.height(),M=a.utils.availableHeight(o,H,l)),Q.select(\".nv-legendWrap\").attr(\"transform\",\
\"translate(\"+E()+\",\"+-l.top+\")\")):Q.select(\".nv-legendWrap\").selectAll(\"*\").remove(),q){var R=[{key:r.grouped||\"Grouped\",disabled:e.stacked()},{key:r.stacked||\"Stacked\",disabled:!e.stacked()}];j.width(E()).color([\"#444\",\"#444\",\"#444\"]),Q.select(\".nv-controlsWrap\").datum(R).attr(\"transform\",\"translate(0,\"+-l.top+\")\").call(j)}else Q.select(\".nv-controlsWrap\").selectAll(\"*\").remove();O.attr(\"transform\",\"translate(\"+l.left+\",\"+l.top+\")\"),v&&Q.select(\".nv-y.nv-axis\").attr(\"transform\",\"translate(\"+L+\",0)\"),e.disabled(C.map(function(a){return a.disabled})).width(L).height(M).color(C.map(function(a,b){return a.color||p(a,b)}).filter(function(a,b){return!C[b].disabled}));var S=Q.select(\".nv-barsWrap\").datum(C.filter(function(a){return!a.disabled}));if(S.call(e),t){f.scale(c)._ticks(a.utils.calcTicksX(L/100,C)).tickSize(-M,0),Q.select(\".nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+d.range()[0]+\")\"),Q.select(\".nv-x.nv-axis\").call(f);var T=Q.select(\".nv-x.nv-axis > g\").selectAll(\"g\");if(T.selectAll(\"line, text\").style(\"opacity\",1),x){var U=function(a,b){return\"translate(\"+a+\",\"+b+\")\"},V=5,W=17;T.selectAll(\"text\").attr(\"transform\",function(a,b,c){return U(0,c%2==0?V:W)});var X=d3.selectAll(\".nv-x.nv-axis .nv-wrap g g text\")[0].length;Q.selectAll(\".nv-x.nv-axis .nv-axisMaxMin text\").attr(\"transform\",function(a,b){return U(0,0===b||X%2!==0?W:V)})}y&&Q.selectAll(\".tick text\").call(a.utils.wrapTicks,b.xAxis.rangeBand()),w&&T.filter(function(a,b){return b%Math.ceil(C[0].values.length/(L/100))!==0}).selectAll(\"text, line\").style(\"opacity\",0),z&&T.selectAll(\".tick text\").attr(\"transform\",\"rotate(\"+z+\" 0,0)\").style(\"text-anchor\",z>0?\"start\":\"end\"),Q.select(\".nv-x.nv-axis\").selectAll(\"g.nv-axisMaxMin text\").style(\"opacity\",1)}u&&(g.scale(d)._ticks(a.utils.calcTicksY(M/36,C)).tickSize(-L,0),Q.select(\".nv-y.nv-axis\").call(g)),G&&(h.width(L).height(M).margin({left:l.left,top:l.top}).svgContainer(H).xScale(c),O.select(\".nv-interactive\").call(h)),i.dispatch.on(\"stateChange\",function(a){for(var c in a)A[c]=a[c];D.stateChange(A),b.update()}),\
j.dispatch.on(\"legendClick\",function(a,c){if(a.disabled){switch(R=R.map(function(a){return a.disabled=!0,a}),a.disabled=!1,a.key){case\"Grouped\":case r.grouped:e.stacked(!1);break;case\"Stacked\":case r.stacked:e.stacked(!0)}A.stacked=e.stacked(),D.stateChange(A),b.update()}}),D.on(\"changeState\",function(a){\"undefined\"!=typeof a.disabled&&(C.forEach(function(b,c){b.disabled=a.disabled[c]}),A.disabled=a.disabled),\"undefined\"!=typeof a.stacked&&(e.stacked(a.stacked),A.stacked=a.stacked,I=a.stacked),b.update()}),G?(h.dispatch.on(\"elementMousemove\",function(a){if(void 0!=a.pointXValue){var d,e,f,g,i=[];C.filter(function(a,b){return a.seriesIndex=b,!a.disabled}).forEach(function(h,j){e=c.domain().indexOf(a.pointXValue);var k=h.values[e];void 0!==k&&(g=k.x,void 0===d&&(d=k),void 0===f&&(f=a.mouseX),i.push({key:h.key,value:b.y()(k,e),color:p(h,h.seriesIndex),data:h.values[e]}))}),h.tooltip.data({value:g,index:e,series:i})(),h.renderGuideLine(f)}}),h.dispatch.on(\"elementMouseout\",function(a){h.tooltip.hidden(!0)})):(e.dispatch.on(\"elementMouseover.tooltip\",function(a){a.value=b.x()(a.data),a.series={key:a.data.key,value:b.y()(a.data),color:a.color},k.data(a).hidden(!1)}),e.dispatch.on(\"elementMouseout.tooltip\",function(a){k.hidden(!0)}),e.dispatch.on(\"elementMousemove.tooltip\",function(a){k()}))}),H.renderEnd(\"multibarchart immediate\"),b}var c,d,e=a.models.multiBar(),f=a.models.axis(),g=a.models.axis(),h=a.interactiveGuideline(),i=a.models.legend(),j=a.models.legend(),k=a.models.tooltip(),l={top:30,right:20,bottom:50,left:60},m=null,n=null,o=null,p=a.utils.defaultColor(),q=!0,r={},s=!0,t=!0,u=!0,v=!1,w=!0,x=!1,y=!1,z=0,A=a.utils.state(),B=null,C=null,D=d3.dispatch(\"stateChange\",\"changeState\",\"renderEnd\"),E=function(){return q?180:0},F=250,G=!1;A.stacked=!1,e.stacked(!1),f.orient(\"bottom\").tickPadding(7).showMaxMin(!1).tickFormat(function(a){return a}),g.orient(v?\"right\":\"left\").tickFormat(d3.format(\",.1f\")),k.duration(0).valueFormatter(function(a,b){return g.tickFormat()(a,b)}).headerFormatter(function(a,b){return f.tickFormat()(a,\
b)}),h.tooltip.valueFormatter(function(a,b){return null==a?\"N/A\":g.tickFormat()(a,b)}).headerFormatter(function(a,b){return f.tickFormat()(a,b)}),h.tooltip.valueFormatter(function(a,b){return null==a?\"N/A\":g.tickFormat()(a,b)}).headerFormatter(function(a,b){return f.tickFormat()(a,b)}),h.tooltip.duration(0).valueFormatter(function(a,b){return g.tickFormat()(a,b)}).headerFormatter(function(a,b){return f.tickFormat()(a,b)}),j.updateState(!1);var H=a.utils.renderWatch(D),I=!1,J=function(a){return function(){return{active:a.map(function(a){return!a.disabled}),stacked:I}}},K=function(a){return function(b){void 0!==b.stacked&&(I=b.stacked),void 0!==b.active&&a.forEach(function(a,c){a.disabled=!b.active[c]})}};return b.dispatch=D,b.multibar=e,b.legend=i,b.controls=j,b.xAxis=f,b.yAxis=g,b.state=A,b.tooltip=k,b.interactiveLayer=h,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return n},set:function(a){n=a}},height:{get:function(){return o},set:function(a){o=a}},showLegend:{get:function(){return s},set:function(a){s=a}},showControls:{get:function(){return q},set:function(a){q=a}},controlLabels:{get:function(){return r},set:function(a){r=a}},showXAxis:{get:function(){return t},set:function(a){t=a}},showYAxis:{get:function(){return u},set:function(a){u=a}},defaultState:{get:function(){return B},set:function(a){B=a}},noData:{get:function(){return C},set:function(a){C=a}},reduceXTicks:{get:function(){return w},set:function(a){w=a}},rotateLabels:{get:function(){return z},set:function(a){z=a}},staggerLabels:{get:function(){return x},set:function(a){x=a}},wrapLabels:{get:function(){return y},set:function(a){y=!!a}},margin:{get:function(){return l},set:function(a){void 0!==a.top&&(l.top=a.top,m=a.top),l.right=void 0!==a.right?a.right:l.right,l.bottom=void 0!==a.bottom?a.bottom:l.bottom,l.left=void 0!==a.left?a.left:l.left}},duration:{get:function(){return F},set:function(a){F=a,e.duration(F),f.duration(F),g.duration(F),H.reset(F)}},color:{get:function(){return p},set:function(b){p=a.utils.getColor(b),\
i.color(p)}},rightAlignYAxis:{get:function(){return v},set:function(a){v=a,g.orient(v?\"right\":\"left\")}},useInteractiveGuideline:{get:function(){return G},set:function(a){G=a}},barColor:{get:function(){return e.barColor},set:function(a){e.barColor(a),i.color(function(a,b){return d3.rgb(\"#ccc\").darker(1.5*b).toString()})}}}),a.utils.inheritOptions(b,e),a.utils.initOptions(b),b},a.models.multiBarHorizontal=function(){\"use strict\";function b(m){return F.reset(),m.each(function(b){var m=k-j.left-j.right,D=l-j.top-j.bottom;n=d3.select(this),a.utils.initSVG(n),w&&(b=d3.layout.stack().offset(\"zero\").values(function(a){return a.values}).y(r)(b)),b.forEach(function(a,b){a.values.forEach(function(c){c.series=b,c.key=a.key})}),w&&b[0].values.map(function(a,c){var d=0,e=0;b.map(function(a){var b=a.values[c];b.size=Math.abs(b.y),b.y<0?(b.y1=e-b.size,e-=b.size):(b.y1=d,d+=b.size)})});var G=d&&e?[]:b.map(function(a){return a.values.map(function(a,b){return{x:q(a,b),y:r(a,b),y0:a.y0,y1:a.y1}})});o.domain(d||d3.merge(G).map(function(a){return a.x})).rangeBands(f||[0,D],A),p.domain(e||d3.extent(d3.merge(G).map(function(a){return w?a.y>0?a.y1+a.y:a.y1:a.y}).concat(t))),x&&!w?p.range(g||[p.domain()[0]<0?z:0,m-(p.domain()[1]>0?z:0)]):p.range(g||[0,m]),h=h||o,i=i||d3.scale.linear().domain(p.domain()).range([p(0),p(0)]);var H=d3.select(this).selectAll(\"g.nv-wrap.nv-multibarHorizontal\").data([b]),I=H.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-multibarHorizontal\"),J=(I.append(\"defs\"),I.append(\"g\"));H.select(\"g\");J.append(\"g\").attr(\"class\",\"nv-groups\"),H.attr(\"transform\",\"translate(\"+j.left+\",\"+j.top+\")\");var K=H.select(\".nv-groups\").selectAll(\".nv-group\").data(function(a){return a},function(a,b){return b});K.enter().append(\"g\").style(\"stroke-opacity\",1e-6).style(\"fill-opacity\",1e-6),K.exit().watchTransition(F,\"multibarhorizontal: exit groups\").style(\"stroke-opacity\",1e-6).style(\"fill-opacity\",1e-6).remove(),K.attr(\"class\",function(a,b){return\"nv-group nv-series-\"+b}).classed(\"hover\",function(a){return a.hover}).style(\"fill\",function(a,\
b){return u(a,b)}).style(\"stroke\",function(a,b){return u(a,b)}),K.watchTransition(F,\"multibarhorizontal: groups\").style(\"stroke-opacity\",1).style(\"fill-opacity\",B);var L=K.selectAll(\"g.nv-bar\").data(function(a){return a.values});L.exit().remove();var M=L.enter().append(\"g\").attr(\"transform\",function(a,c,d){return\"translate(\"+i(w?a.y0:0)+\",\"+(w?0:d*o.rangeBand()/b.length+o(q(a,c)))+\")\"});M.append(\"rect\").attr(\"width\",0).attr(\"height\",o.rangeBand()/(w?1:b.length)),L.on(\"mouseover\",function(a,b){d3.select(this).classed(\"hover\",!0),E.elementMouseover({data:a,index:b,color:d3.select(this).style(\"fill\")})}).on(\"mouseout\",function(a,b){d3.select(this).classed(\"hover\",!1),E.elementMouseout({data:a,index:b,color:d3.select(this).style(\"fill\")})}).on(\"mouseout\",function(a,b){E.elementMouseout({data:a,index:b,color:d3.select(this).style(\"fill\")})}).on(\"mousemove\",function(a,b){E.elementMousemove({data:a,index:b,color:d3.select(this).style(\"fill\")})}).on(\"click\",function(a,b){var c=this;E.elementClick({data:a,index:b,color:d3.select(this).style(\"fill\"),event:d3.event,element:c}),d3.event.stopPropagation()}).on(\"dblclick\",function(a,b){E.elementDblClick({data:a,index:b,color:d3.select(this).style(\"fill\")}),d3.event.stopPropagation()}),s(b[0],0)&&(M.append(\"polyline\"),L.select(\"polyline\").attr(\"fill\",\"none\").attr(\"points\",function(a,c){var d=s(a,c),e=.8*o.rangeBand()/(2*(w?1:b.length));d=d.length?d:[-Math.abs(d),Math.abs(d)],d=d.map(function(a){return p(a)-p(0)});var f=[[d[0],-e],[d[0],e],[d[0],0],[d[1],0],[d[1],-e],[d[1],e]];return f.map(function(a){return a.join(\",\")}).join(\" \")}).attr(\"transform\",function(a,c){var d=o.rangeBand()/(2*(w?1:b.length));return\"translate(\"+(r(a,c)<0?0:p(r(a,c))-p(0))+\", \"+d+\")\"})),M.append(\"text\"),x&&!w?(L.select(\"text\").attr(\"text-anchor\",function(a,b){return r(a,b)<0?\"end\":\"start\"}).attr(\"y\",o.rangeBand()/(2*b.length)).attr(\"dy\",\".32em\").text(function(a,b){var c=C(r(a,b)),d=s(a,b);return void 0===d?c:d.length?c+\"+\"+C(Math.abs(d[1]))+\"-\"+C(Math.abs(d[0])):c+\"±\"+C(Math.abs(d))}),L.watchTransition(F,\
\"multibarhorizontal: bars\").select(\"text\").attr(\"x\",function(a,b){return r(a,b)<0?-4:p(r(a,b))-p(0)+4})):L.selectAll(\"text\").text(\"\"),y&&!w?(M.append(\"text\").classed(\"nv-bar-label\",!0),L.select(\"text.nv-bar-label\").attr(\"text-anchor\",function(a,b){return r(a,b)<0?\"start\":\"end\"}).attr(\"y\",o.rangeBand()/(2*b.length)).attr(\"dy\",\".32em\").text(function(a,b){return q(a,b)}),L.watchTransition(F,\"multibarhorizontal: bars\").select(\"text.nv-bar-label\").attr(\"x\",function(a,b){return r(a,b)<0?p(0)-p(r(a,b))+4:-4})):L.selectAll(\"text.nv-bar-label\").text(\"\"),L.attr(\"class\",function(a,b){return r(a,b)<0?\"nv-bar negative\":\"nv-bar positive\"}),v&&(c||(c=b.map(function(){return!0})),L.style(\"fill\",function(a,b,d){return d3.rgb(v(a,b)).darker(c.map(function(a,b){return b}).filter(function(a,b){return!c[b]})[d]).toString()}).style(\"stroke\",function(a,b,d){return d3.rgb(v(a,b)).darker(c.map(function(a,b){return b}).filter(function(a,b){return!c[b]})[d]).toString()})),w?L.watchTransition(F,\"multibarhorizontal: bars\").attr(\"transform\",function(a,b){return\"translate(\"+p(a.y1)+\",\"+o(q(a,b))+\")\"}).select(\"rect\").attr(\"width\",function(a,b){return Math.abs(p(r(a,b)+a.y0)-p(a.y0))||0}).attr(\"height\",o.rangeBand()):L.watchTransition(F,\"multibarhorizontal: bars\").attr(\"transform\",function(a,c){return\"translate(\"+p(r(a,c)<0?r(a,c):0)+\",\"+(a.series*o.rangeBand()/b.length+o(q(a,c)))+\")\"}).select(\"rect\").attr(\"height\",o.rangeBand()/b.length).attr(\"width\",function(a,b){return Math.max(Math.abs(p(r(a,b))-p(0)),1)||0}),h=o.copy(),i=p.copy()}),F.renderEnd(\"multibarHorizontal immediate\"),b}var c,d,e,f,g,h,i,j={top:0,right:0,bottom:0,left:0},k=960,l=500,m=Math.floor(1e4*Math.random()),n=null,o=d3.scale.ordinal(),p=d3.scale.linear(),q=function(a){return a.x},r=function(a){return a.y},s=function(a){return a.yErr},t=[0],u=a.utils.defaultColor(),v=null,w=!1,x=!1,y=!1,z=60,A=.1,B=.75,C=d3.format(\",.2f\"),D=250,E=d3.dispatch(\"chartClick\",\"elementClick\",\"elementDblClick\",\"elementMouseover\",\"elementMouseout\",\"elementMousemove\",\"renderEnd\"),F=a.utils.renderWatch(E,D);\
return b.dispatch=E,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return k},set:function(a){k=a}},height:{get:function(){return l},set:function(a){l=a}},x:{get:function(){return q},set:function(a){q=a}},y:{get:function(){return r},set:function(a){r=a}},yErr:{get:function(){return s},set:function(a){s=a}},xScale:{get:function(){return o},set:function(a){o=a}},yScale:{get:function(){return p},set:function(a){p=a}},xDomain:{get:function(){return d},set:function(a){d=a}},yDomain:{get:function(){return e},set:function(a){e=a}},xRange:{get:function(){return f},set:function(a){f=a}},yRange:{get:function(){return g},set:function(a){g=a}},forceY:{get:function(){return t},set:function(a){t=a}},stacked:{get:function(){return w},set:function(a){w=a}},showValues:{get:function(){return x},set:function(a){x=a}},disabled:{get:function(){return c},set:function(a){c=a}},id:{get:function(){return m},set:function(a){m=a}},valueFormat:{get:function(){return C},set:function(a){C=a}},valuePadding:{get:function(){return z},set:function(a){z=a}},groupSpacing:{get:function(){return A},set:function(a){A=a}},fillOpacity:{get:function(){return B},set:function(a){B=a}},margin:{get:function(){return j},set:function(a){j.top=void 0!==a.top?a.top:j.top,j.right=void 0!==a.right?a.right:j.right,j.bottom=void 0!==a.bottom?a.bottom:j.bottom,j.left=void 0!==a.left?a.left:j.left}},duration:{get:function(){return D},set:function(a){D=a,F.reset(D)}},color:{get:function(){return u},set:function(b){u=a.utils.getColor(b)}},barColor:{get:function(){return v},set:function(b){v=b?a.utils.getColor(b):null}}}),a.utils.initOptions(b),b},a.models.multiBarHorizontalChart=function(){\"use strict\";function b(j){return D.reset(),D.models(e),s&&D.models(f),t&&D.models(g),j.each(function(j){var x=d3.select(this);a.utils.initSVG(x);var D=a.utils.availableWidth(m,x,k),E=a.utils.availableHeight(n,x,k);if(b.update=function(){x.transition().duration(A).call(b)},b.container=this,u=e.stacked(),v.setter(C(j),b.update).getter(B(j)).update(),\
v.disabled=j.map(function(a){return!!a.disabled}),!w){var F;w={};for(F in v)v[F]instanceof Array?w[F]=v[F].slice(0):w[F]=v[F]}if(!(j&&j.length&&j.filter(function(a){return a.values.length}).length))return a.utils.noData(b,x),b;x.selectAll(\".nv-noData\").remove(),c=e.xScale(),d=e.yScale().clamp(!0);var G=x.selectAll(\"g.nv-wrap.nv-multiBarHorizontalChart\").data([j]),H=G.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-multiBarHorizontalChart\").append(\"g\"),I=G.select(\"g\");if(H.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),H.append(\"g\").attr(\"class\",\"nv-y nv-axis\").append(\"g\").attr(\"class\",\"nv-zeroLine\").append(\"line\"),H.append(\"g\").attr(\"class\",\"nv-barsWrap\"),H.append(\"g\").attr(\"class\",\"nv-legendWrap\"),H.append(\"g\").attr(\"class\",\"nv-controlsWrap\"),r?(h.width(D-z()),I.select(\".nv-legendWrap\").datum(j).call(h),l||h.height()===k.top||(k.top=h.height(),E=a.utils.availableHeight(n,x,k)),I.select(\".nv-legendWrap\").attr(\"transform\",\"translate(\"+z()+\",\"+-k.top+\")\")):I.select(\".nv-legendWrap\").selectAll(\"*\").remove(),p){var J=[{key:q.grouped||\"Grouped\",disabled:e.stacked()},{key:q.stacked||\"Stacked\",disabled:!e.stacked()}];i.width(z()).color([\"#444\",\"#444\",\"#444\"]),I.select(\".nv-controlsWrap\").datum(J).attr(\"transform\",\"translate(0,\"+-k.top+\")\").call(i)}else I.select(\".nv-controlsWrap\").selectAll(\"*\").remove();G.attr(\"transform\",\"translate(\"+k.left+\",\"+k.top+\")\"),e.disabled(j.map(function(a){return a.disabled})).width(D).height(E).color(j.map(function(a,b){return a.color||o(a,b)}).filter(function(a,b){return!j[b].disabled}));var K=I.select(\".nv-barsWrap\").datum(j.filter(function(a){return!a.disabled}));if(K.transition().call(e),s){f.scale(c)._ticks(a.utils.calcTicksY(E/24,j)).tickSize(-D,0),I.select(\".nv-x.nv-axis\").call(f);var L=I.select(\".nv-x.nv-axis\").selectAll(\"g\");L.selectAll(\"line, text\")}t&&(g.scale(d)._ticks(a.utils.calcTicksX(D/100,j)).tickSize(-E,0),I.select(\".nv-y.nv-axis\").attr(\"transform\",\"translate(0,\"+E+\")\"),I.select(\".nv-y.nv-axis\").call(g)),I.select(\".nv-zeroLine line\").attr(\"x1\",d(0)).attr(\"x2\",d(0)).attr(\"y1\",0).attr(\"y2\",\
-E),h.dispatch.on(\"stateChange\",function(a){for(var c in a)v[c]=a[c];y.stateChange(v),b.update()}),i.dispatch.on(\"legendClick\",function(a,c){if(a.disabled){switch(J=J.map(function(a){return a.disabled=!0,a}),a.disabled=!1,a.key){case\"Grouped\":case q.grouped:e.stacked(!1);break;case\"Stacked\":case q.stacked:e.stacked(!0)}v.stacked=e.stacked(),y.stateChange(v),u=e.stacked(),b.update()}}),y.on(\"changeState\",function(a){\"undefined\"!=typeof a.disabled&&(j.forEach(function(b,c){b.disabled=a.disabled[c]}),v.disabled=a.disabled),\"undefined\"!=typeof a.stacked&&(e.stacked(a.stacked),v.stacked=a.stacked,u=a.stacked),b.update()})}),D.renderEnd(\"multibar horizontal chart immediate\"),b}var c,d,e=a.models.multiBarHorizontal(),f=a.models.axis(),g=a.models.axis(),h=a.models.legend().height(30),i=a.models.legend().height(30),j=a.models.tooltip(),k={top:30,right:20,bottom:50,left:60},l=null,m=null,n=null,o=a.utils.defaultColor(),p=!0,q={},r=!0,s=!0,t=!0,u=!1,v=a.utils.state(),w=null,x=null,y=d3.dispatch(\"stateChange\",\"changeState\",\"renderEnd\"),z=function(){return p?180:0},A=250;v.stacked=!1,e.stacked(u),f.orient(\"left\").tickPadding(5).showMaxMin(!1).tickFormat(function(a){return a}),g.orient(\"bottom\").tickFormat(d3.format(\",.1f\")),j.duration(0).valueFormatter(function(a,b){return g.tickFormat()(a,b)}).headerFormatter(function(a,b){return f.tickFormat()(a,b)}),i.updateState(!1);var B=function(a){return function(){return{active:a.map(function(a){return!a.disabled}),stacked:u}}},C=function(a){return function(b){void 0!==b.stacked&&(u=b.stacked),void 0!==b.active&&a.forEach(function(a,c){a.disabled=!b.active[c]})}},D=a.utils.renderWatch(y,A);return e.dispatch.on(\"elementMouseover.tooltip\",function(a){a.value=b.x()(a.data),a.series={key:a.data.key,value:b.y()(a.data),color:a.color},j.data(a).hidden(!1)}),e.dispatch.on(\"elementMouseout.tooltip\",function(a){j.hidden(!0)}),e.dispatch.on(\"elementMousemove.tooltip\",function(a){j()}),b.dispatch=y,b.multibar=e,b.legend=h,b.controls=i,b.xAxis=f,b.yAxis=g,b.state=v,b.tooltip=j,b.options=a.utils.optionsFunc.bind(b),\
b._options=Object.create({},{width:{get:function(){return m},set:function(a){m=a}},height:{get:function(){return n},set:function(a){n=a}},showLegend:{get:function(){return r},set:function(a){r=a}},showControls:{get:function(){return p},set:function(a){p=a}},controlLabels:{get:function(){return q},set:function(a){q=a}},showXAxis:{get:function(){return s},set:function(a){s=a}},showYAxis:{get:function(){return t},set:function(a){t=a}},defaultState:{get:function(){return w},set:function(a){w=a}},noData:{get:function(){return x},set:function(a){x=a}},margin:{get:function(){return k},set:function(a){void 0!==a.top&&(k.top=a.top,l=a.top),k.right=void 0!==a.right?a.right:k.right,k.bottom=void 0!==a.bottom?a.bottom:k.bottom,k.left=void 0!==a.left?a.left:k.left}},duration:{get:function(){return A},set:function(a){A=a,D.reset(A),e.duration(A),f.duration(A),g.duration(A)}},color:{get:function(){return o},set:function(b){o=a.utils.getColor(b),h.color(o)}},barColor:{get:function(){return e.barColor},set:function(a){e.barColor(a),h.color(function(a,b){return d3.rgb(\"#ccc\").darker(1.5*b).toString()})}}}),a.utils.inheritOptions(b,e),a.utils.initOptions(b),b},a.models.multiChart=function(){\"use strict\";function b(k){return k.each(function(k){function o(a){var b=2===k[a.seriesIndex].yAxis?G:F;a.value=a.point.x,a.series={value:a.point.y,color:a.point.color,key:a.series.key},I.duration(0).headerFormatter(function(a,b){return E.tickFormat()(a,b)}).valueFormatter(function(a,c){return b.tickFormat()(a,c)}).data(a).hidden(!1)}function s(a){var b=2===k[a.seriesIndex].yAxis?G:F;a.value=a.point.x,a.series={value:a.point.y,color:a.point.color,key:a.series.key},I.duration(100).headerFormatter(function(a,b){return E.tickFormat()(a,b)}).valueFormatter(function(a,c){return b.tickFormat()(a,c)}).data(a).hidden(!1)}function J(a){var b=2===k[a.seriesIndex].yAxis?G:F;a.point.x=C.x()(a.point),a.point.y=C.y()(a.point),I.duration(0).headerFormatter(function(a,b){return E.tickFormat()(a,b)}).valueFormatter(function(a,c){return b.tickFormat()(a,c)}).data(a).hidden(!1)}function \
L(a){var b=2===k[a.data.series].yAxis?G:F;a.value=A.x()(a.data),a.series={value:A.y()(a.data),color:a.color,key:a.data.key},I.duration(0).headerFormatter(function(a,b){return E.tickFormat()(a,b)}).valueFormatter(function(a,c){return b.tickFormat()(a,c)}).data(a).hidden(!1)}function M(){for(var a=0,b=K.length;b>a;a++){var c=K[a];try{c.clearHighlights()}catch(d){}}}function N(a,b,c){for(var d=0,e=K.length;e>d;d++){var f=K[d];try{f.highlightPoint(a,b,c)}catch(g){}}}var O=d3.select(this);a.utils.initSVG(O),b.update=function(){O.transition().call(b)},b.container=this;var P=a.utils.availableWidth(h,O,e),Q=a.utils.availableHeight(i,O,e),R=k.filter(function(a){return\"line\"==a.type&&1==a.yAxis}),S=k.filter(function(a){return\"line\"==a.type&&2==a.yAxis}),T=k.filter(function(a){return\"scatter\"==a.type&&1==a.yAxis}),U=k.filter(function(a){return\"scatter\"==a.type&&2==a.yAxis}),V=k.filter(function(a){return\"bar\"==a.type&&1==a.yAxis}),W=k.filter(function(a){return\"bar\"==a.type&&2==a.yAxis}),X=k.filter(function(a){return\"area\"==a.type&&1==a.yAxis}),Y=k.filter(function(a){return\"area\"==a.type&&2==a.yAxis});if(!(k&&k.length&&k.filter(function(a){return a.values.length}).length))return a.utils.noData(b,O),b;O.selectAll(\".nv-noData\").remove();var Z=k.filter(function(a){return!a.disabled&&1==a.yAxis}).map(function(a){return a.values.map(function(a,b){return{x:l(a),y:m(a)}})}),$=k.filter(function(a){return!a.disabled&&2==a.yAxis}).map(function(a){return a.values.map(function(a,b){return{x:l(a),y:m(a)}})});t.domain(d3.extent(d3.merge(Z.concat($)),function(a){return a.x})).range([0,P]);var _=O.selectAll(\"g.wrap.multiChart\").data([k]),aa=_.enter().append(\"g\").attr(\"class\",\"wrap nvd3 multiChart\").append(\"g\");aa.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),aa.append(\"g\").attr(\"class\",\"nv-y1 nv-axis\"),aa.append(\"g\").attr(\"class\",\"nv-y2 nv-axis\"),aa.append(\"g\").attr(\"class\",\"stack1Wrap\"),aa.append(\"g\").attr(\"class\",\"stack2Wrap\"),aa.append(\"g\").attr(\"class\",\"bars1Wrap\"),aa.append(\"g\").attr(\"class\",\"bars2Wrap\"),aa.append(\"g\").attr(\"class\",\"scatters1Wrap\"),\
aa.append(\"g\").attr(\"class\",\"scatters2Wrap\"),aa.append(\"g\").attr(\"class\",\"lines1Wrap\"),aa.append(\"g\").attr(\"class\",\"lines2Wrap\"),aa.append(\"g\").attr(\"class\",\"legendWrap\"),aa.append(\"g\").attr(\"class\",\"nv-interactive\");var ba=_.select(\"g\"),ca=k.map(function(a,b){return k[b].color||g(a,b)});if(j){var da=H.align()?P/2:P,ea=H.align()?da:0;H.width(da),H.color(ca),ba.select(\".legendWrap\").datum(k.map(function(a){return a.originalKey=void 0===a.originalKey?a.key:a.originalKey,a.key=a.originalKey+(1==a.yAxis?\"\":r),a})).call(H),f||H.height()===e.top||(e.top=H.height(),Q=a.utils.availableHeight(i,O,e)),ba.select(\".legendWrap\").attr(\"transform\",\"translate(\"+ea+\",\"+-e.top+\")\")}else ba.select(\".legendWrap\").selectAll(\"*\").remove();w.width(P).height(Q).interpolate(n).color(ca.filter(function(a,b){return!k[b].disabled&&1==k[b].yAxis&&\"line\"==k[b].type})),x.width(P).height(Q).interpolate(n).color(ca.filter(function(a,b){return!k[b].disabled&&2==k[b].yAxis&&\"line\"==k[b].type})),y.width(P).height(Q).color(ca.filter(function(a,b){return!k[b].disabled&&1==k[b].yAxis&&\"scatter\"==k[b].type})),z.width(P).height(Q).color(ca.filter(function(a,b){return!k[b].disabled&&2==k[b].yAxis&&\"scatter\"==k[b].type})),A.width(P).height(Q).color(ca.filter(function(a,b){return!k[b].disabled&&1==k[b].yAxis&&\"bar\"==k[b].type})),B.width(P).height(Q).color(ca.filter(function(a,b){return!k[b].disabled&&2==k[b].yAxis&&\"bar\"==k[b].type})),C.width(P).height(Q).interpolate(n).color(ca.filter(function(a,b){return!k[b].disabled&&1==k[b].yAxis&&\"area\"==k[b].type})),D.width(P).height(Q).interpolate(n).color(ca.filter(function(a,b){return!k[b].disabled&&2==k[b].yAxis&&\"area\"==k[b].type})),ba.attr(\"transform\",\"translate(\"+e.left+\",\"+e.top+\")\");var fa=ba.select(\".lines1Wrap\").datum(R.filter(function(a){return!a.disabled})),ga=ba.select(\".scatters1Wrap\").datum(T.filter(function(a){return!a.disabled})),ha=ba.select(\".bars1Wrap\").datum(V.filter(function(a){return!a.disabled})),ia=ba.select(\".stack1Wrap\").datum(X.filter(function(a){return!a.disabled})),ja=ba.select(\".lines2Wrap\").datum(S.filter(function(a){return!a.disabled})),\
ka=ba.select(\".scatters2Wrap\").datum(U.filter(function(a){return!a.disabled})),la=ba.select(\".bars2Wrap\").datum(W.filter(function(a){return!a.disabled})),ma=ba.select(\".stack2Wrap\").datum(Y.filter(function(a){return!a.disabled})),na=X.length?X.map(function(a){return a.values}).reduce(function(a,b){return a.map(function(a,c){return{x:a.x,y:a.y+b[c].y}})}).concat([{x:0,y:0}]):[],oa=Y.length?Y.map(function(a){return a.values}).reduce(function(a,b){return a.map(function(a,c){return{x:a.x,y:a.y+b[c].y}})}).concat([{x:0,y:0}]):[];u.domain(c||d3.extent(d3.merge(Z).concat(na),function(a){return a.y})).range([0,Q]),v.domain(d||d3.extent(d3.merge($).concat(oa),function(a){return a.y})).range([0,Q]),w.yDomain(u.domain()),y.yDomain(u.domain()),A.yDomain(u.domain()),C.yDomain(u.domain()),x.yDomain(v.domain()),z.yDomain(v.domain()),B.yDomain(v.domain()),D.yDomain(v.domain()),X.length&&d3.transition(ia).call(C),Y.length&&d3.transition(ma).call(D),V.length&&d3.transition(ha).call(A),W.length&&d3.transition(la).call(B),R.length&&d3.transition(fa).call(w),\
S.length&&d3.transition(ja).call(x),T.length&&d3.transition(ga).call(y),U.length&&d3.transition(ka).call(z),E._ticks(a.utils.calcTicksX(P/100,k)).tickSize(-Q,0),ba.select(\".nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+Q+\")\"),d3.transition(ba.select(\".nv-x.nv-axis\")).call(E),F._ticks(a.utils.calcTicksY(Q/36,k)).tickSize(-P,0),d3.transition(ba.select(\".nv-y1.nv-axis\")).call(F),G._ticks(a.utils.calcTicksY(Q/36,k)).tickSize(-P,0),d3.transition(ba.select(\".nv-y2.nv-axis\")).call(G),ba.select(\".nv-y1.nv-axis\").classed(\"nv-disabled\",Z.length?!1:!0).attr(\"transform\",\"translate(\"+t.range()[0]+\",0)\"),ba.select(\".nv-y2.nv-axis\").classed(\"nv-disabled\",$.length?!1:!0).attr(\"transform\",\"translate(\"+t.range()[1]+\",0)\"),H.dispatch.on(\"stateChange\",function(a){b.update()}),q&&(p.width(P).height(Q).margin({left:e.left,top:e.top}).svgContainer(O).xScale(t),_.select(\".nv-interactive\").call(p)),q?(p.dispatch.on(\"elementMousemove\",function(c){M();var d,e,f,h=[];k.filter(function(a,b){return a.seriesIndex=b,!a.disabled}).forEach(function(i,j){var k=t.domain(),l=i.values.filter(function(a,c){return b.x()(a,c)>=k[0]&&b.x()(a,c)<=k[1]});e=a.interactiveBisect(l,c.pointXValue,b.x());var m=l[e],n=b.y()(m,e);null!==n&&N(j,e,!0),void 0!==m&&(void 0===d&&(d=m),void 0===f&&(f=t(b.x()(m,e))),h.push({key:i.key,value:n,color:g(i,i.seriesIndex),data:m,yAxis:2==i.yAxis?G:F}))});var i=function(a,b){var c=h[b].yAxis;return null==a?\"N/A\":c.tickFormat()(a)};p.tooltip.headerFormatter(function(a,b){return E.tickFormat()(a,b)}).valueFormatter(p.tooltip.valueFormatter()||i).data({value:b.x()(d,e),index:e,series:h})(),p.renderGuideLine(f)}),p.dispatch.on(\"elementMouseout\",function(a){M()})):(w.dispatch.on(\"elementMouseover.tooltip\",o),x.dispatch.on(\"elementMouseover.tooltip\",o),w.dispatch.on(\"elementMouseout.tooltip\",function(a){I.hidden(!0)}),x.dispatch.on(\"elementMouseout.tooltip\",function(a){I.hidden(!0)}),y.dispatch.on(\"elementMouseover.tooltip\",s),z.dispatch.on(\"elementMouseover.tooltip\",s),y.dispatch.on(\"elementMouseout.tooltip\",function(a){I.hidden(!0)}),\
z.dispatch.on(\"elementMouseout.tooltip\",function(a){I.hidden(!0)}),C.dispatch.on(\"elementMouseover.tooltip\",J),D.dispatch.on(\"elementMouseover.tooltip\",J),C.dispatch.on(\"elementMouseout.tooltip\",function(a){I.hidden(!0)}),D.dispatch.on(\"elementMouseout.tooltip\",function(a){I.hidden(!0)}),A.dispatch.on(\"elementMouseover.tooltip\",L),B.dispatch.on(\"elementMouseover.tooltip\",L),A.dispatch.on(\"elementMouseout.tooltip\",function(a){I.hidden(!0)}),B.dispatch.on(\"elementMouseout.tooltip\",function(a){I.hidden(!0)}),A.dispatch.on(\"elementMousemove.tooltip\",function(a){I()}),B.dispatch.on(\"elementMousemove.tooltip\",function(a){I()}))}),b}var c,d,e={top:30,right:20,bottom:50,left:60},f=null,g=a.utils.defaultColor(),h=null,i=null,j=!0,k=null,l=function(a){return a.x},m=function(a){return a.y},n=\"linear\",o=!0,p=a.interactiveGuideline(),q=!1,r=\" (right axis)\",s=250,t=d3.scale.linear(),u=d3.scale.linear(),v=d3.scale.linear(),w=a.models.line().yScale(u).duration(s),x=a.models.line().yScale(v).duration(s),y=a.models.scatter().yScale(u).duration(s),z=a.models.scatter().yScale(v).duration(s),A=a.models.multiBar().stacked(!1).yScale(u).duration(s),B=a.models.multiBar().stacked(!1).yScale(v).duration(s),C=a.models.stackedArea().yScale(u).duration(s),D=a.models.stackedArea().yScale(v).duration(s),E=a.models.axis().scale(t).orient(\"bottom\").tickPadding(5).duration(s),F=a.models.axis().scale(u).orient(\"left\").duration(s),G=a.models.axis().scale(v).orient(\"right\").duration(s),H=a.models.legend().height(30),I=a.models.tooltip(),J=d3.dispatch(),K=[w,x,y,z,A,B,C,D];return b.dispatch=J,b.legend=H,b.lines1=w,b.lines2=x,b.scatters1=y,b.scatters2=z,b.bars1=A,b.bars2=B,b.stack1=C,b.stack2=D,b.xAxis=E,b.yAxis1=F,b.yAxis2=G,b.tooltip=I,b.interactiveLayer=p,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return h},set:function(a){h=a}},height:{get:function(){return i},set:function(a){i=a}},showLegend:{get:function(){return j},set:function(a){j=a}},yDomain1:{get:function(){return c},set:function(a){c=a}},yDomain2:{get:function(){return \
d},set:function(a){d=a}},noData:{get:function(){return k},set:function(a){k=a}},interpolate:{get:function(){return n},set:function(a){n=a}},legendRightAxisHint:{get:function(){return r},set:function(a){r=a}},margin:{get:function(){return e},set:function(a){void 0!==a.top&&(e.top=a.top,f=a.top),e.right=void 0!==a.right?a.right:e.right,e.bottom=void 0!==a.bottom?a.bottom:e.bottom,e.left=void 0!==a.left?a.left:e.left}},color:{get:function(){return g},set:function(b){g=a.utils.getColor(b)}},x:{get:function(){return l},set:function(a){l=a,w.x(a),x.x(a),y.x(a),z.x(a),A.x(a),B.x(a),C.x(a),D.x(a)}},y:{get:function(){return m},set:function(a){m=a,w.y(a),x.y(a),y.y(a),z.y(a),C.y(a),D.y(a),A.y(a),B.y(a)}},useVoronoi:{get:function(){return o},set:function(a){o=a,w.useVoronoi(a),x.useVoronoi(a),C.useVoronoi(a),D.useVoronoi(a)}},useInteractiveGuideline:{get:function(){return q},set:function(a){q=a,q&&(w.interactive(!1),w.useVoronoi(!1),x.interactive(!1),x.useVoronoi(!1),C.interactive(!1),C.useVoronoi(!1),D.interactive(!1),D.useVoronoi(!1),y.interactive(!1),z.interactive(!1))}},duration:{get:function(){return s},set:function(a){s=a,[w,x,C,D,y,z,E,F,G].forEach(function(a){a.duration(s)})}}}),a.utils.initOptions(b),b},a.models.ohlcBar=function(){\"use strict\";function b(y){return y.each(function(b){k=d3.select(this);var y=a.utils.availableWidth(h,k,g),A=a.utils.availableHeight(i,k,g);a.utils.initSVG(k);var B=y/b[0].values.length*.9;l.domain(c||d3.extent(b[0].values.map(n).concat(t))),v?l.range(e||[.5*y/b[0].values.length,y*(b[0].values.length-.5)/b[0].values.length]):l.range(e||[5+B/2,y-B/2-5]),m.domain(d||[d3.min(b[0].values.map(s).concat(u)),d3.max(b[0].values.map(r).concat(u))]).range(f||[A,0]),l.domain()[0]===l.domain()[1]&&(l.domain()[0]?l.domain([l.domain()[0]-.01*l.domain()[0],l.domain()[1]+.01*l.domain()[1]]):l.domain([-1,1])),m.domain()[0]===m.domain()[1]&&(m.domain()[0]?m.domain([m.domain()[0]+.01*m.domain()[0],m.domain()[1]-.01*m.domain()[1]]):m.domain([-1,1]));var C=d3.select(this).selectAll(\"g.nv-wrap.nv-ohlcBar\").data([b[0].values]),\
D=C.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-ohlcBar\"),E=D.append(\"defs\"),F=D.append(\"g\"),G=C.select(\"g\");F.append(\"g\").attr(\"class\",\"nv-ticks\"),C.attr(\"transform\",\"translate(\"+g.left+\",\"+g.top+\")\"),k.on(\"click\",function(a,b){z.chartClick({data:a,index:b,pos:d3.event,id:j})}),E.append(\"clipPath\").attr(\"id\",\"nv-chart-clip-path-\"+j).append(\"rect\"),C.select(\"#nv-chart-clip-path-\"+j+\" rect\").attr(\"width\",y).attr(\"height\",A),G.attr(\"clip-path\",w?\"url(#nv-chart-clip-path-\"+j+\")\":\"\");var H=C.select(\".nv-ticks\").selectAll(\".nv-tick\").data(function(a){return a});H.exit().remove(),H.enter().append(\"path\").attr(\"class\",function(a,b,c){return(p(a,b)>q(a,b)?\"nv-tick negative\":\"nv-tick positive\")+\" nv-tick-\"+c+\"-\"+b}).attr(\"d\",function(a,b){return\"m0,0l0,\"+(m(p(a,b))-m(r(a,b)))+\"l\"+-B/2+\",0l\"+B/2+\",0l0,\"+(m(s(a,b))-m(p(a,b)))+\"l0,\"+(m(q(a,b))-m(s(a,b)))+\"l\"+B/2+\",0l\"+-B/2+\",0z\"}).attr(\"transform\",function(a,b){return\"translate(\"+l(n(a,b))+\",\"+m(r(a,b))+\")\"}).attr(\"fill\",function(a,b){return x[0]}).attr(\"stroke\",function(a,b){return x[0]}).attr(\"x\",0).attr(\"y\",function(a,b){return m(Math.max(0,o(a,b)))}).attr(\"height\",function(a,b){return Math.abs(m(o(a,b))-m(0))}),H.attr(\"class\",function(a,b,c){return(p(a,b)>q(a,b)?\"nv-tick negative\":\"nv-tick positive\")+\" nv-tick-\"+c+\"-\"+b}),d3.transition(H).attr(\"transform\",function(a,b){return\"translate(\"+l(n(a,b))+\",\"+m(r(a,b))+\")\"}).attr(\"d\",function(a,c){var d=y/b[0].values.length*.9;return\"m0,0l0,\"+(m(p(a,c))-m(r(a,c)))+\"l\"+-d/2+\",0l\"+d/2+\",0l0,\"+(m(s(a,c))-m(p(a,c)))+\"l0,\"+(m(q(a,c))-m(s(a,c)))+\"l\"+d/2+\",0l\"+-d/2+\",0z\"})}),b}var c,d,e,f,g={top:0,right:0,bottom:0,left:0},h=null,i=null,j=Math.floor(1e4*Math.random()),k=null,l=d3.scale.linear(),m=d3.scale.linear(),n=function(a){return a.x},o=function(a){return a.y},p=function(a){return a.open},q=function(a){return a.close},r=function(a){return a.high},s=function(a){return a.low},t=[],u=[],v=!1,w=!0,x=a.utils.defaultColor(),y=!1,z=d3.dispatch(\"stateChange\",\"changeState\",\"renderEnd\",\"chartClick\",\"elementClick\",\"elementDblClick\",\"elementMouseover\",\
\"elementMouseout\",\"elementMousemove\");return b.highlightPoint=function(a,c){b.clearHighlights(),k.select(\".nv-ohlcBar .nv-tick-0-\"+a).classed(\"hover\",c)},b.clearHighlights=function(){k.select(\".nv-ohlcBar .nv-tick.hover\").classed(\"hover\",!1)},b.dispatch=z,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return h},set:function(a){h=a}},height:{get:function(){return i},set:function(a){i=a}},xScale:{get:function(){return l},set:function(a){l=a}},yScale:{get:function(){return m},set:function(a){m=a}},xDomain:{get:function(){return c},set:function(a){c=a}},yDomain:{get:function(){return d},set:function(a){d=a}},xRange:{get:function(){return e},set:function(a){e=a}},yRange:{get:function(){return f},set:function(a){f=a}},forceX:{get:function(){return t},set:function(a){t=a}},forceY:{get:function(){return u},set:function(a){u=a}},padData:{get:function(){return v},set:function(a){v=a}},clipEdge:{get:function(){return w},set:function(a){w=a}},id:{get:function(){return j},set:function(a){j=a}},interactive:{get:function(){return y},set:function(a){y=a}},x:{get:function(){return n},set:function(a){n=a}},y:{get:function(){return o},set:function(a){o=a}},open:{get:function(){return p()},set:function(a){p=a}},close:{get:function(){return q()},set:function(a){q=a}},high:{get:function(){return r},set:function(a){r=a}},low:{get:function(){return s},set:function(a){s=a}},margin:{get:function(){return g},set:function(a){g.top=void 0!=a.top?a.top:g.top,g.right=void 0!=a.right?a.right:g.right,g.bottom=void 0!=a.bottom?a.bottom:g.bottom,g.left=void 0!=a.left?a.left:g.left}},color:{get:function(){return x},set:function(b){x=a.utils.getColor(b)}}}),a.utils.initOptions(b),b},a.models.parallelCoordinates=function(){\"use strict\";function b(B){return A.reset(),B.each(function(b){function A(a){return x(o.map(function(b){if(isNaN(a.values[b.key])||isNaN(parseFloat(a.values[b.key]))||O){var c=l[b.key].domain(),d=l[b.key].range(),e=c[0]-(c[1]-c[0])/9;if(v.indexOf(b.key)<0){var f=d3.scale.linear().domain([e,c[1]]).range([j-12,\
d[1]]);l[b.key].brush.y(f),v.push(b.key)}if(isNaN(a.values[b.key])||isNaN(parseFloat(a.values[b.key])))return[k(b.key),l[b.key](e)]}return void 0!==U&&(v.length>0||O?(U.style(\"display\",\"inline\"),V.style(\"display\",\"inline\")):(U.style(\"display\",\"none\"),V.style(\"display\",\"none\"))),[k(b.key),l[b.key](a.values[b.key])]}))}function B(a){s.forEach(function(b){var c=l[b.dimension].brush.y().domain();b.hasOnlyNaN&&(b.extent[1]=(l[b.dimension].domain()[1]-c[0])*(b.extent[1]-b.extent[0])/(N[b.dimension]-b.extent[0])+c[0]),b.hasNaN&&(b.extent[0]=c[0]),a&&l[b.dimension].brush.extent(b.extent)}),e.select(\".nv-brushBackground\").each(function(a){d3.select(this).call(l[a.key].brush)}).selectAll(\"rect\").attr(\"x\",-8).attr(\"width\",16),F()}function C(){q===!1&&(q=!0,B(!0))}function D(){$=p.filter(function(a){return!l[a].brush.empty()}),_=$.map(function(a){return l[a].brush.extent()}),s=[],$.forEach(function(a,b){s[b]={dimension:a,extent:_[b],hasNaN:!1,hasOnlyNaN:!1}}),t=[],c.style(\"display\",function(a){var b=$.every(function(b,c){return(isNaN(a.values[b])||isNaN(parseFloat(a.values[b])))&&_[c][0]==l[b].brush.y().domain()[0]?!0:_[c][0]<=a.values[b]&&a.values[b]<=_[c][1]&&!isNaN(parseFloat(a.values[b]))});return b&&t.push(a),b?null:\"none\"}),F(),z.brush({filters:s,active:t})}function E(){var a=$.length>0?!0:!1;s.forEach(function(a){a.extent[0]===l[a.dimension].brush.y().domain()[0]&&v.indexOf(a.dimension)>=0&&(a.hasNaN=!0),a.extent[1]<l[a.dimension].domain()[0]&&(a.hasOnlyNaN=!0)}),z.brushEnd(t,a)}function F(){e.select(\".nv-axis\").each(function(a,b){var c=s.filter(function(b){return b.dimension==a.key});P[a.key]=l[a.key].domain(),0!=c.length&&q&&(P[a.key]=[],c[0].extent[1]>l[a.key].domain()[0]&&(P[a.key]=[c[0].extent[1]]),c[0].extent[0]>=l[a.key].domain()[0]&&P[a.key].push(c[0].extent[0])),d3.select(this).call(y.scale(l[a.key]).tickFormat(a.format).tickValues(P[a.key]))})}function G(a){u[a.key]=this.parentNode.__origin__=k(a.key),d.attr(\"visibility\",\"hidden\")}function H(a){u[a.key]=Math.min(i,Math.max(0,this.parentNode.__origin__+=d3.event.x)),\
c.attr(\"d\",A),o.sort(function(a,b){return J(a.key)-J(b.key)}),o.forEach(function(a,b){return a.currentPosition=b}),k.domain(o.map(function(a){return a.key})),e.attr(\"transform\",function(a){return\"translate(\"+J(a.key)+\")\"})}function I(a,b){delete this.parentNode.__origin__,delete u[a.key],d3.select(this.parentNode).attr(\"transform\",\"translate(\"+k(a.key)+\")\"),c.attr(\"d\",A),d.attr(\"d\",A).attr(\"visibility\",null),z.dimensionsOrder(o)}function J(a){var b=u[a];return null==b?k(a):b}var K=d3.select(this);if(i=a.utils.availableWidth(g,K,f),j=a.utils.availableHeight(h,K,f),a.utils.initSVG(K),void 0===b[0].values){var L=[];b.forEach(function(a){var b={},c=Object.keys(a);c.forEach(function(c){\"name\"!==c&&(b[c]=a[c])}),L.push({key:a.name,values:b})}),b=L}var M=b.map(function(a){return a.values});0===t.length&&(t=b),p=n.sort(function(a,b){return a.currentPosition-b.currentPosition}).map(function(a){return a.key}),o=n.filter(function(a){return!a.disabled}),k.rangePoints([0,i],1).domain(o.map(function(a){return a.key}));var N={},O=!1,P=[];p.forEach(function(a){var b=d3.extent(M,function(b){return+b[a]}),c=b[0],d=b[1],e=!1;(isNaN(c)||isNaN(d))&&(e=!0,c=0,d=0),c===d&&(c-=1,d+=1);var f=s.filter(function(b){return b.dimension==a});0!==f.length&&(e?(c=l[a].domain()[0],d=l[a].domain()[1]):!f[0].hasOnlyNaN&&q?(c=c>f[0].extent[0]?f[0].extent[0]:c,d=d<f[0].extent[1]?f[0].extent[1]:d):f[0].hasNaN&&(d=d<f[0].extent[1]?f[0].extent[1]:d,N[a]=l[a].domain()[1],O=!0)),l[a]=d3.scale.linear().domain([c,d]).range([.9*(j-12),0]),v=[],l[a].brush=d3.svg.brush().y(l[a]).on(\"brushstart\",C).on(\"brush\",D).on(\"brushend\",E)});var Q=K.selectAll(\"g.nv-wrap.nv-parallelCoordinates\").data([b]),R=Q.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-parallelCoordinates\"),S=R.append(\"g\"),T=Q.select(\"g\");S.append(\"g\").attr(\"class\",\"nv-parallelCoordinates background\"),S.append(\"g\").attr(\"class\",\"nv-parallelCoordinates foreground\"),S.append(\"g\").attr(\"class\",\"nv-parallelCoordinates missingValuesline\"),Q.attr(\"transform\",\"translate(\"+f.left+\",\"+f.top+\")\"),x.interpolate(\"cardinal\").tension(w),\
y.orient(\"left\");var U,V,W=d3.behavior.drag().on(\"dragstart\",G).on(\"drag\",H).on(\"dragend\",I),X=k.range()[1]-k.range()[0];if(X=isNaN(X)?k.range()[0]:X,!isNaN(X)){var Y=[0+X/2,j-12,i-X/2,j-12];U=Q.select(\".missingValuesline\").selectAll(\"line\").data([Y]),U.enter().append(\"line\"),U.exit().remove(),U.attr(\"x1\",function(a){return a[0]}).attr(\"y1\",function(a){return a[1]}).attr(\"x2\",function(a){return a[2]}).attr(\"y2\",function(a){return a[3]}),V=Q.select(\".missingValuesline\").selectAll(\"text\").data([m]),V.append(\"text\").data([m]),V.enter().append(\"text\"),V.exit().remove(),V.attr(\"y\",j).attr(\"x\",i-92-X/2).text(function(a){return a})}d=Q.select(\".background\").selectAll(\"path\").data(b),d.enter().append(\"path\"),d.exit().remove(),d.attr(\"d\",A),c=Q.select(\".foreground\").selectAll(\"path\").data(b),c.enter().append(\"path\"),c.exit().remove(),c.attr(\"d\",A).style(\"stroke-width\",function(a,b){return isNaN(a.strokeWidth)&&(a.strokeWidth=1),a.strokeWidth}).attr(\"stroke\",function(a,b){return a.color||r(a,b)}),c.on(\"mouseover\",function(a,b){d3.select(this).classed(\"hover\",!0).style(\"stroke-width\",a.strokeWidth+2+\"px\").style(\"stroke-opacity\",1),z.elementMouseover({label:a.name,color:a.color||r(a,b),values:a.values,dimensions:o})}),c.on(\"mouseout\",function(a,b){d3.select(this).classed(\"hover\",!1).style(\"stroke-width\",a.strokeWidth+\"px\").style(\"stroke-opacity\",.7),z.elementMouseout({label:a.name,index:b})}),c.on(\"mousemove\",function(a,b){z.elementMousemove()}),c.on(\"click\",function(a){z.elementClick({id:a.id})}),e=T.selectAll(\".dimension\").data(o);var Z=e.enter().append(\"g\").attr(\"class\",\"nv-parallelCoordinates dimension\");e.attr(\"transform\",function(a){return\"translate(\"+k(a.key)+\",0)\"}),Z.append(\"g\").attr(\"class\",\"nv-axis\"),Z.append(\"text\").attr(\"class\",\"nv-label\").style(\"cursor\",\"move\").attr(\"dy\",\"-1em\").attr(\"text-anchor\",\"middle\").on(\"mouseover\",function(a,b){z.elementMouseover({label:a.tooltip||a.key,color:a.color})}).on(\"mouseout\",function(a,b){z.elementMouseout({label:a.tooltip})}).on(\"mousemove\",function(a,b){z.elementMousemove()}).call(W),\
")

file(APPEND "${METABENCH_DIR}/nvd3.js" "\
Z.append(\"g\").attr(\"class\",\"nv-brushBackground\"),e.exit().remove(),e.select(\".nv-label\").text(function(a){return a.key}),B(q);var $=p.filter(function(a){return!l[a].brush.empty()}),_=$.map(function(a){return l[a].brush.extent()}),aa=t.slice(0);t=[],c.style(\"display\",function(a){var b=$.every(function(b,c){return(isNaN(a.values[b])||isNaN(parseFloat(a.values[b])))&&_[c][0]==l[b].brush.y().domain()[0]?!0:_[c][0]<=a.values[b]&&a.values[b]<=_[c][1]&&!isNaN(parseFloat(a.values[b]))});return b&&t.push(a),b?null:\"none\"}),(s.length>0||!a.utils.arrayEquals(t,aa))&&z.activeChanged(t)}),b}var c,d,e,f={top:30,right:0,bottom:10,left:0},g=null,h=null,i=null,j=null,k=d3.scale.ordinal(),l={},m=\"undefined values\",n=[],o=[],p=[],q=!0,r=a.utils.defaultColor(),s=[],t=[],u=[],v=[],w=1,x=d3.svg.line(),y=d3.svg.axis(),z=d3.dispatch(\"brushstart\",\"brush\",\"brushEnd\",\"dimensionsOrder\",\"stateChange\",\"elementClick\",\"elementMouseover\",\"elementMouseout\",\"elementMousemove\",\"renderEnd\",\"activeChanged\"),A=a.utils.renderWatch(z);return b.dispatch=z,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return g},set:function(a){g=a}},height:{get:function(){return h},set:function(a){h=a}},dimensionData:{get:function(){return n},set:function(a){n=a}},displayBrush:{get:function(){return q},set:function(a){q=a}},filters:{get:function(){return s},set:function(a){s=a}},active:{get:function(){return t},set:function(a){t=a}},lineTension:{get:function(){return w},set:function(a){w=a}},undefinedValuesLabel:{get:function(){return m},set:function(a){m=a}},dimensions:{get:function(){return n.map(function(a){return a.key})},set:function(b){a.deprecated(\"dimensions\",\"use dimensionData instead\"),0===n.length?b.forEach(function(a){n.push({key:a})}):b.forEach(function(a,b){n[b].key=a})}},dimensionNames:{get:function(){return n.map(function(a){return a.key})},set:function(b){a.deprecated(\"dimensionNames\",\"use dimensionData instead\"),p=[],0===n.length?b.forEach(function(a){n.push({key:a})}):b.forEach(function(a,b){n[b].key=a})}},dimensionFormats:{get:function(){return \
n.map(function(a){return a.format})},set:function(b){a.deprecated(\"dimensionFormats\",\"use dimensionData instead\"),0===n.length?b.forEach(function(a){n.push({format:a})}):b.forEach(function(a,b){n[b].format=a})}},margin:{get:function(){return f},set:function(a){f.top=void 0!==a.top?a.top:f.top,f.right=void 0!==a.right?a.right:f.right,f.bottom=void 0!==a.bottom?a.bottom:f.bottom,f.left=void 0!==a.left?a.left:f.left}},color:{get:function(){return r},set:function(b){r=a.utils.getColor(b)}}}),a.utils.initOptions(b),b},a.models.parallelCoordinatesChart=function(){\"use strict\";function b(e){return s.reset(),s.models(c),e.each(function(e){var k=d3.select(this);a.utils.initSVG(k);var p=a.utils.availableWidth(h,k,f),q=a.utils.availableHeight(i,k,f);if(b.update=function(){k.call(b)},b.container=this,l.setter(u(m),b.update).getter(t(m)).update(),l.disabled=m.map(function(a){return!!a.disabled}),m=m.map(function(a){return a.disabled=!!a.disabled,a}),m.forEach(function(a,b){a.originalPosition=isNaN(a.originalPosition)?b:a.originalPosition,a.currentPosition=isNaN(a.currentPosition)?b:a.currentPosition}),!o){var s;o={};for(s in l)l[s]instanceof Array?o[s]=l[s].slice(0):o[s]=l[s]}if(!e||!e.length)return a.utils.noData(b,k),b;k.selectAll(\".nv-noData\").remove();var v=k.selectAll(\"g.nv-wrap.nv-parallelCoordinatesChart\").data([e]),w=v.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-parallelCoordinatesChart\").append(\"g\"),x=v.select(\"g\");w.append(\"g\").attr(\"class\",\"nv-parallelCoordinatesWrap\"),w.append(\"g\").attr(\"class\",\"nv-legendWrap\"),x.select(\"rect\").attr(\"width\",p).attr(\"height\",q>0?q:0),j?(d.width(p).color(function(a){return\"rgb(188,190,192)\"}),x.select(\".nv-legendWrap\").datum(m.sort(function(a,b){return a.originalPosition-b.originalPosition})).call(d),g||d.height()===f.top||(f.top=d.height(),q=a.utils.availableHeight(i,k,f)),v.select(\".nv-legendWrap\").attr(\"transform\",\"translate( 0 ,\"+-f.top+\")\")):x.select(\".nv-legendWrap\").selectAll(\"*\").remove(),v.attr(\"transform\",\"translate(\"+f.left+\",\"+f.top+\")\"),c.width(p).height(q).dimensionData(m).displayBrush(n);\
var y=x.select(\".nv-parallelCoordinatesWrap \").datum(e);y.transition().call(c),c.dispatch.on(\"brushEnd\",function(a,b){b?(n=!0,r.brushEnd(a)):n=!1}),d.dispatch.on(\"stateChange\",function(a){for(var c in a)l[c]=a[c];r.stateChange(l),b.update()}),c.dispatch.on(\"dimensionsOrder\",function(a){m.sort(function(a,b){return a.currentPosition-b.currentPosition});var b=!1;m.forEach(function(a,c){a.currentPosition=c,a.currentPosition!==a.originalPosition&&(b=!0)}),r.dimensionsOrder(m,b)}),r.on(\"changeState\",function(a){\"undefined\"!=typeof a.disabled&&(m.forEach(function(b,c){b.disabled=a.disabled[c]}),l.disabled=a.disabled),b.update()})}),s.renderEnd(\"parraleleCoordinateChart immediate\"),b}var c=a.models.parallelCoordinates(),d=a.models.legend(),e=a.models.tooltip(),f=(a.models.tooltip(),{top:0,right:0,bottom:0,left:0}),g=null,h=null,i=null,j=!0,k=a.utils.defaultColor(),l=a.utils.state(),m=[],n=!0,o=null,p=null,q=\"undefined\",r=d3.dispatch(\"dimensionsOrder\",\"brushEnd\",\"stateChange\",\"changeState\",\"renderEnd\"),s=a.utils.renderWatch(r),t=function(a){return function(){return{active:a.map(function(a){return!a.disabled})}}},u=function(a){return function(b){void 0!==b.active&&a.forEach(function(a,c){a.disabled=!b.active[c]})}};return e.contentGenerator(function(a){var b='<table><thead><tr><td class=\"legend-color-guide\"><div style=\"background-color:'+a.color+'\"></div></td><td><strong>'+a.key+\"</strong></td></tr></thead>\";return 0!==a.series.length&&(b+='<tbody><tr><td height =\"10px\"></td></tr>',a.series.forEach(function(a){b=b+'<tr><td class=\"legend-color-guide\"><div style=\"background-color:'+a.color+'\"></div></td><td class=\"key\">'+a.key+'</td><td class=\"value\">'+a.value+\"</td></tr>\"}),b+=\"</tbody>\"),b+=\"</table>\"}),c.dispatch.on(\"elementMouseover.tooltip\",function(a){var b={key:a.label,color:a.color,series:[]};a.values&&(Object.keys(a.values).forEach(function(c){var d=a.dimensions.filter(function(a){return a.key===c})[0];if(d){var e;e=isNaN(a.values[c])||isNaN(parseFloat(a.values[c]))?q:d.format(a.values[c]),b.series.push({idx:d.currentPosition,\
key:c,value:e,color:d.color})}}),b.series.sort(function(a,b){return a.idx-b.idx})),e.data(b).hidden(!1)}),c.dispatch.on(\"elementMouseout.tooltip\",function(a){e.hidden(!0)}),c.dispatch.on(\"elementMousemove.tooltip\",function(){e()}),b.dispatch=r,b.parallelCoordinates=c,b.legend=d,b.tooltip=e,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return h},set:function(a){h=a}},height:{get:function(){return i},set:function(a){i=a}},showLegend:{get:function(){return j},set:function(a){j=a}},defaultState:{get:function(){return o},set:function(a){o=a}},dimensionData:{get:function(){return m},set:function(a){m=a}},displayBrush:{get:function(){return n},set:function(a){n=a}},noData:{get:function(){return p},set:function(a){p=a}},nanValue:{get:function(){return q},set:function(a){q=a}},margin:{get:function(){return f},set:function(a){void 0!==a.top&&(f.top=a.top,g=a.top),f.right=void 0!==a.right?a.right:f.right,f.bottom=void 0!==a.bottom?a.bottom:f.bottom,f.left=void 0!==a.left?a.left:f.left}},color:{get:function(){return k},set:function(b){k=a.utils.getColor(b),d.color(k),c.color(k)}}}),a.utils.inheritOptions(b,c),a.utils.initOptions(b),b},a.models.pie=function(){\"use strict\";function b(F){return E.reset(),F.each(function(b){function F(a,b){a.endAngle=isNaN(a.endAngle)?0:a.endAngle,a.startAngle=isNaN(a.startAngle)?0:a.startAngle,p||(a.innerRadius=0);var c=d3.interpolate(this._current,a);return this._current=c(0),function(a){return C[b](c(a))}}var G=d-c.left-c.right,H=e-c.top-c.bottom,I=Math.min(G,H)/2,J=[],K=[];if(i=d3.select(this),0===A.length)for(var L=I-I/5,M=y*I,N=0;N<b[0].length;N++)J.push(L),K.push(M);else r?(J=A.map(function(a){return(a.outer-a.outer/5)*I}),K=A.map(function(a){return(a.inner-a.inner/5)*I}),y=d3.min(A.map(function(a){return a.inner-a.inner/5}))):(J=A.map(function(a){return a.outer*I}),K=A.map(function(a){return a.inner*I}),y=d3.min(A.map(function(a){return a.inner})));a.utils.initSVG(i);var O=i.selectAll(\".nv-wrap.nv-pie\").data(b),P=O.enter().append(\"g\").attr(\"class\",\
\"nvd3 nv-wrap nv-pie nv-chart-\"+h),Q=P.append(\"g\"),R=O.select(\"g\"),S=Q.append(\"g\").attr(\"class\",\"nv-pie\");Q.append(\"g\").attr(\"class\",\"nv-pieLabels\"),O.attr(\"transform\",\"translate(\"+c.left+\",\"+c.top+\")\"),R.select(\".nv-pie\").attr(\"transform\",\"translate(\"+G/2+\",\"+H/2+\")\"),R.select(\".nv-pieLabels\").attr(\"transform\",\"translate(\"+G/2+\",\"+H/2+\")\"),i.on(\"click\",function(a,b){B.chartClick({data:a,index:b,pos:d3.event,id:h})}),C=[],D=[];for(var N=0;N<b[0].length;N++){var T=d3.svg.arc().outerRadius(J[N]),U=d3.svg.arc().outerRadius(J[N]+5);u!==!1&&(T.startAngle(u),U.startAngle(u)),w!==!1&&(T.endAngle(w),U.endAngle(w)),p&&(T.innerRadius(K[N]),U.innerRadius(K[N])),T.cornerRadius&&x&&(T.cornerRadius(x),U.cornerRadius(x)),C.push(T),D.push(U)}var V=d3.layout.pie().sort(null).value(function(a){return a.disabled?0:g(a)});V.padAngle&&v&&V.padAngle(v),p&&q&&(S.append(\"text\").attr(\"class\",\"nv-pie-title\"),O.select(\".nv-pie-title\").style(\"text-anchor\",\"middle\").text(function(a){return q}).style(\"font-size\",Math.min(G,H)*y*2/(q.length+2)+\"px\").attr(\"dy\",\"0.35em\").attr(\"transform\",function(a,b){return\"translate(0, \"+s+\")\"}));var W=O.select(\".nv-pie\").selectAll(\".nv-slice\").data(V),X=O.select(\".nv-pieLabels\").selectAll(\".nv-label\").data(V);W.exit().remove(),X.exit().remove();var Y=W.enter().append(\"g\");Y.attr(\"class\",\"nv-slice\"),Y.on(\"mouseover\",function(a,b){d3.select(this).classed(\"hover\",!0),r&&d3.select(this).select(\"path\").transition().duration(70).attr(\"d\",D[b]),B.elementMouseover({data:a.data,index:b,color:d3.select(this).style(\"fill\"),percent:(a.endAngle-a.startAngle)/(2*Math.PI)})}),Y.on(\"mouseout\",function(a,b){d3.select(this).classed(\"hover\",!1),r&&d3.select(this).select(\"path\").transition().duration(50).attr(\"d\",C[b]),B.elementMouseout({data:a.data,index:b})}),Y.on(\"mousemove\",function(a,b){B.elementMousemove({data:a.data,index:b})}),Y.on(\"click\",function(a,b){var c=this;B.elementClick({data:a.data,index:b,color:d3.select(this).style(\"fill\"),event:d3.event,element:c})}),Y.on(\"dblclick\",function(a,b){B.elementDblClick({data:a.data,\
index:b,color:d3.select(this).style(\"fill\")})}),W.attr(\"fill\",function(a,b){return j(a.data,b)}),W.attr(\"stroke\",function(a,b){return j(a.data,b)});Y.append(\"path\").each(function(a){this._current=a});if(W.select(\"path\").transition().duration(z).attr(\"d\",function(a,b){return C[b](a)}).attrTween(\"d\",F),l){for(var Z=[],N=0;N<b[0].length;N++)Z.push(C[N]),m?p&&(Z[N]=d3.svg.arc().outerRadius(C[N].outerRadius()),u!==!1&&Z[N].startAngle(u),w!==!1&&Z[N].endAngle(w)):p||Z[N].innerRadius(0);X.enter().append(\"g\").classed(\"nv-label\",!0).each(function(a,b){var c=d3.select(this);c.attr(\"transform\",function(a,b){if(t){a.outerRadius=J[b]+10,a.innerRadius=J[b]+15;var c=(a.startAngle+a.endAngle)/2*(180/Math.PI);return(a.startAngle+a.endAngle)/2<Math.PI?c-=90:c+=90,\"translate(\"+Z[b].centroid(a)+\") rotate(\"+c+\")\"}return a.outerRadius=I+10,a.innerRadius=I+15,\"translate(\"+Z[b].centroid(a)+\")\"}),c.append(\"rect\").style(\"stroke\",\"#fff\").style(\"fill\",\"#fff\").attr(\"rx\",3).attr(\"ry\",3),c.append(\"text\").style(\"text-anchor\",t?(a.startAngle+a.endAngle)/2<Math.PI?\"start\":\"end\":\"middle\").style(\"fill\",\"#000\")});var $={},_=14,aa=140,ba=function(a){return Math.floor(a[0]/aa)*aa+\",\"+Math.floor(a[1]/_)*_},ca=function(a){return(a.endAngle-a.startAngle)/(2*Math.PI)};X.watchTransition(E,\"pie labels\").attr(\"transform\",function(a,b){if(t){a.outerRadius=J[b]+10,a.innerRadius=J[b]+15;var c=(a.startAngle+a.endAngle)/2*(180/Math.PI);return(a.startAngle+a.endAngle)/2<Math.PI?c-=90:c+=90,\"translate(\"+Z[b].centroid(a)+\") rotate(\"+c+\")\"}a.outerRadius=I+10,a.innerRadius=I+15;var d=Z[b].centroid(a),e=ca(a);if(a.value&&e>=o){var f=ba(d);$[f]&&(d[1]-=_),$[ba(d)]=!0}return\"translate(\"+d+\")\"}),X.select(\".nv-label text\").style(\"text-anchor\",function(a,b){return t?(a.startAngle+a.endAngle)/2<Math.PI?\"start\":\"end\":\"middle\"}).text(function(a,b){var c=ca(a),d=\"\";if(!a.value||o>c)return\"\";if(\"function\"==typeof n)d=n(a,b,{key:f(a.data),value:g(a.data),percent:k(c)});else switch(n){case\"key\":d=f(a.data);break;case\"value\":d=k(g(a.data));break;case\"percent\":d=d3.format(\"%\")(c)}return \
d})}}),E.renderEnd(\"pie immediate\"),b}var c={top:0,right:0,bottom:0,left:0},d=500,e=500,f=function(a){return a.x},g=function(a){return a.y},h=Math.floor(1e4*Math.random()),i=null,j=a.utils.defaultColor(),k=d3.format(\",.2f\"),l=!0,m=!1,n=\"key\",o=.02,p=!1,q=!1,r=!0,s=0,t=!1,u=!1,v=!1,w=!1,x=0,y=.5,z=250,A=[],B=d3.dispatch(\"chartClick\",\"elementClick\",\"elementDblClick\",\"elementMouseover\",\"elementMouseout\",\"elementMousemove\",\"renderEnd\"),C=[],D=[],E=a.utils.renderWatch(B);return b.dispatch=B,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{arcsRadius:{get:function(){return A},set:function(a){A=a}},width:{get:function(){return d},set:function(a){d=a}},height:{get:function(){return e},set:function(a){e=a}},showLabels:{get:function(){return l},set:function(a){l=a}},title:{get:function(){return q},set:function(a){q=a}},titleOffset:{get:function(){return s},set:function(a){s=a}},labelThreshold:{get:function(){return o},set:function(a){o=a}},valueFormat:{get:function(){return k},set:function(a){k=a}},x:{get:function(){return f},set:function(a){f=a}},id:{get:function(){return h},set:function(a){h=a}},endAngle:{get:function(){return w},set:function(a){w=a}},startAngle:{get:function(){return u},set:function(a){u=a}},padAngle:{get:function(){return v},set:function(a){v=a}},cornerRadius:{get:function(){return x},set:function(a){x=a}},donutRatio:{get:function(){return y},set:function(a){y=a}},labelsOutside:{get:function(){return m},set:function(a){m=a}},labelSunbeamLayout:{get:function(){return t},set:function(a){t=a}},donut:{get:function(){return p},set:function(a){p=a}},growOnHover:{get:function(){return r},set:function(a){r=a}},pieLabelsOutside:{get:function(){return m},set:function(b){m=b,a.deprecated(\"pieLabelsOutside\",\"use labelsOutside instead\")}},donutLabelsOutside:{get:function(){return m},set:function(b){m=b,a.deprecated(\"donutLabelsOutside\",\"use labelsOutside instead\")}},labelFormat:{get:function(){return k},set:function(b){k=b,a.deprecated(\"labelFormat\",\"use valueFormat instead\")}},margin:{get:function(){return \
c},set:function(a){c.top=\"undefined\"!=typeof a.top?a.top:c.top,c.right=\"undefined\"!=typeof a.right?a.right:c.right,c.bottom=\"undefined\"!=typeof a.bottom?a.bottom:c.bottom,c.left=\"undefined\"!=typeof a.left?a.left:c.left}},duration:{get:function(){return z},set:function(a){z=a,E.reset(z)}},y:{get:function(){return g},set:function(a){g=d3.functor(a)}},color:{get:function(){return j},set:function(b){j=a.utils.getColor(b)}},labelType:{get:function(){return n},set:function(a){n=a||\"key\"}}}),a.utils.initOptions(b),b},a.models.pieChart=function(){\"use strict\";function b(e){return s.reset(),s.models(c),e.each(function(e){var j=d3.select(this);a.utils.initSVG(j);var m=a.utils.availableWidth(h,j,f),p=a.utils.availableHeight(i,j,f);if(b.update=function(){j.transition().call(b)},b.container=this,n.setter(u(e),b.update).getter(t(e)).update(),n.disabled=e.map(function(a){return!!a.disabled}),!o){var q;o={};for(q in n)n[q]instanceof Array?o[q]=n[q].slice(0):o[q]=n[q]}if(!e||!e.length)return a.utils.noData(b,j),b;j.selectAll(\".nv-noData\").remove();var s=j.selectAll(\"g.nv-wrap.nv-pieChart\").data([e]),v=s.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-pieChart\").append(\"g\"),w=s.select(\"g\");\
if(v.append(\"g\").attr(\"class\",\"nv-pieWrap\"),v.append(\"g\").attr(\"class\",\"nv-legendWrap\"),k){if(\"top\"===l)d.width(m).key(c.x()),s.select(\".nv-legendWrap\").datum(e).call(d),g||d.height()===f.top||(f.top=d.height(),p=a.utils.availableHeight(i,j,f)),s.select(\".nv-legendWrap\").attr(\"transform\",\"translate(0,\"+-f.top+\")\");else if(\"right\"===l){var x=a.models.legend().width();x>m/2&&(x=m/2),d.height(p).key(c.x()),d.width(x),m-=d.width(),s.select(\".nv-legendWrap\").datum(e).call(d).attr(\"transform\",\"translate(\"+m+\",0)\")}}else w.select(\".nv-legendWrap\").selectAll(\"*\").remove();s.attr(\"transform\",\"translate(\"+f.left+\",\"+f.top+\")\"),c.width(m).height(p);var y=w.select(\".nv-pieWrap\").datum([e]);d3.transition(y).call(c),d.dispatch.on(\"stateChange\",function(a){for(var c in a)n[c]=a[c];r.stateChange(n),b.update()}),r.on(\"changeState\",function(a){\"undefined\"!=typeof a.disabled&&(e.forEach(function(b,c){b.disabled=a.disabled[c]}),n.disabled=a.disabled),b.update()})}),s.renderEnd(\"pieChart immediate\"),b}var c=a.models.pie(),d=a.models.legend(),e=a.models.tooltip(),f={top:30,right:20,bottom:20,left:20},g=null,h=null,i=null,j=!1,k=!0,l=\"top\",m=a.utils.defaultColor(),n=a.utils.state(),o=null,p=null,q=250,r=d3.dispatch(\"stateChange\",\"changeState\",\"renderEnd\");e.duration(0).headerEnabled(!1).valueFormatter(function(a,b){return c.valueFormat()(a,b)});var s=a.utils.renderWatch(r),t=function(a){return function(){return{active:a.map(function(a){return!a.disabled})}}},u=function(a){return function(b){void 0!==b.active&&a.forEach(function(a,c){a.disabled=!b.active[c]})}};return c.dispatch.on(\"elementMouseover.tooltip\",function(a){a.series={key:b.x()(a.data),value:b.y()(a.data),color:a.color,percent:a.percent},j||(delete a.percent,delete a.series.percent),e.data(a).hidden(!1)}),c.dispatch.on(\"elementMouseout.tooltip\",function(a){e.hidden(!0)}),c.dispatch.on(\"elementMousemove.tooltip\",function(a){e()}),b.legend=d,b.dispatch=r,b.pie=c,b.tooltip=e,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return h},set:function(a){h=a}},\
height:{get:function(){return i},set:function(a){i=a}},noData:{get:function(){return p},set:function(a){p=a}},showTooltipPercent:{get:function(){return j},set:function(a){j=a}},showLegend:{get:function(){return k},set:function(a){k=a}},legendPosition:{get:function(){return l},set:function(a){l=a}},defaultState:{get:function(){return o},set:function(a){o=a}},color:{get:function(){return m},set:function(a){m=a,d.color(m),c.color(m)}},duration:{get:function(){return q},set:function(a){q=a,s.reset(q),c.duration(q)}},margin:{get:function(){return f},set:function(a){void 0!==a.top&&(f.top=a.top,g=a.top),f.right=void 0!==a.right?a.right:f.right,f.bottom=void 0!==a.bottom?a.bottom:f.bottom,f.left=void 0!==a.left?a.left:f.left}}}),a.utils.inheritOptions(b,c),a.utils.initOptions(b),b},a.models.sankey=function(){\"use strict\";function b(){n.forEach(function(a){a.sourceLinks=[],a.targetLinks=[]}),o.forEach(function(a){var b=a.source,c=a.target;\"number\"==typeof b&&(b=a.source=n[a.source]),\"number\"==typeof c&&(c=a.target=n[a.target]),b.sourceLinks.push(a),c.targetLinks.push(a)})}function c(){n.forEach(function(a){a.value=Math.max(d3.sum(a.sourceLinks,i),d3.sum(a.targetLinks,i))})}function d(){for(var a,b=n,c=0;b.length&&c<n.length;)a=[],b.forEach(function(b){b.x=c,b.dx=k,b.sourceLinks.forEach(function(b){a.indexOf(b.target)<0&&a.push(b.target)})}),b=a,++c;p&&e(c),f((m[0]-k)/(c-1))}function e(a){n.forEach(function(b){b.sourceLinks.length||(b.x=a-1)})}function f(a){n.forEach(function(b){b.x*=a})}function g(a){function b(){var a=d3.min(g,function(a){return(m[1]-(a.length-1)*l)/d3.sum(a,i)});g.forEach(function(b){b.forEach(function(b,c){b.y=c,b.dy=b.value*a})}),o.forEach(function(b){b.dy=b.value*a})}function c(a){function b(a){return(a.source.y+a.sy+a.dy/2)*a.value}g.forEach(function(c,d){c.forEach(function(c){if(c.targetLinks.length){var d=d3.sum(c.targetLinks,b)/d3.sum(c.targetLinks,i);c.y+=(d-t(c))*a}})})}function d(a){function b(a){return(a.target.y+a.ty+a.dy/2)*a.value}g.slice().reverse().forEach(function(c){c.forEach(function(c){if(c.sourceLinks.length){var \
d=d3.sum(c.sourceLinks,b)/d3.sum(c.sourceLinks,i);c.y+=(d-t(c))*a}})})}function e(){g.forEach(function(a){var b,c,d,e=0,g=a.length;for(a.sort(f),d=0;g>d;++d)b=a[d],c=e-b.y,c>0&&(b.y+=c),e=b.y+b.dy+l;if(c=e-l-m[1],c>0)for(e=b.y-=c,d=g-2;d>=0;--d)b=a[d],c=b.y+b.dy+l-e,c>0&&(b.y-=c),e=b.y})}function f(a,b){return a.y-b.y}var g=d3.nest().key(function(a){return a.x}).sortKeys(d3.ascending).entries(n).map(function(a){return a.values});b(),e(),h();for(var j=1;a>0;--a)d(j*=.99),e(),h(),c(j),e(),h()}function h(){function a(a,b){return a.source.y-b.source.y}function b(a,b){return a.target.y-b.target.y}n.forEach(function(c){c.sourceLinks.sort(b),c.targetLinks.sort(a)}),n.forEach(function(a){var b=0,c=0;a.sourceLinks.forEach(function(a){a.sy=b,b+=a.dy}),a.targetLinks.forEach(function(a){a.ty=c,c+=a.dy})})}function i(a){return a.value}var j={},k=24,l=8,m=[1,1],n=[],o=[],p=!0,q=function(a){b(),c(),d(),g(a)},r=function(){h()},s=function(){function a(a){var c=a.source.x+a.source.dx,d=a.target.x,e=d3.interpolateNumber(c,d),f=e(b),g=e(1-b),h=a.source.y+a.sy+a.dy/2,i=a.target.y+a.ty+a.dy/2,j=\"M\"+c+\",\"+h+\"C\"+f+\",\"+h+\" \"+g+\",\"+i+\" \"+d+\",\"+i;return j}var b=.5;return a.curvature=function(c){return arguments.length?(b=+c,a):b},a},t=function(a){return a.y+a.dy/2};return j.options=a.utils.optionsFunc.bind(j),j._options=Object.create({},{nodeWidth:{get:function(){return k},set:function(a){k=+a}},nodePadding:{get:function(){return l},set:function(a){l=a}},nodes:{get:function(){return n},set:function(a){n=a}},links:{get:function(){return o},set:function(a){o=a}},size:{get:function(){return m},set:function(a){m=a}},sinksRight:{get:function(){return p},set:function(a){p=a}},layout:{get:function(){q(32)},set:function(a){q(a)}},relayout:{get:function(){r()},set:function(a){}},center:{get:function(){return t()},set:function(a){\"function\"==typeof a&&(t=a)}},link:{get:function(){return s()},set:function(a){return\"function\"==typeof a&&(s=a),s()}}}),a.utils.initOptions(j),j},a.models.sankeyChart=function(){\"use strict\";function b(a){return a.each(function(b){function \
c(a){d3.select(this).attr(\"transform\",\"translate(\"+a.x+\",\"+(a.y=Math.max(0,Math.min(f-a.dy,d3.event.y)))+\")\"),d.relayout(),t.attr(\"d\",s)}var i={nodes:[{node:1,name:\"Test 1\"},{node:2,name:\"Test 2\"},{node:3,name:\"Test 3\"},{node:4,name:\"Test 4\"},{node:5,name:\"Test 5\"},{node:6,name:\"Test 6\"}],links:[{source:0,target:1,value:2295},{source:0,target:5,value:1199},{source:1,target:2,value:1119},{source:1,target:5,value:1176},{source:2,target:3,value:487},{source:2,target:5,value:632},{source:3,target:4,value:301},{source:3,target:5,value:186}]},k=!1,l=!1;if((\"object\"==typeof b.nodes&&b.nodes.length)>=0&&(\"object\"==typeof b.links&&b.links.length)>=0&&(k=!0),b.nodes&&b.nodes.length>0&&b.links&&b.links.length>0&&(l=!0),!k)return console.error(\"NVD3 Sankey chart error:\",\"invalid data format for\",b),console.info(\"Valid data format is: \",i,JSON.stringify(i)),r(a,\"Error loading chart, data is invalid\"),!1;if(!l)return r(a,\"No data available\"),!1;var m=a.append(\"svg\").attr(\"width\",e).attr(\"height\",f).append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-sankeyChart\");d.nodeWidth(g).nodePadding(h).size([e,f]);var s=d.link();d.nodes(b.nodes).links(b.links).layout(32).center(j);var t=m.append(\"g\").selectAll(\".link\").data(b.links).enter().append(\"path\").attr(\"class\",\"link\").attr(\"d\",s).style(\"stroke-width\",function(a){return Math.max(1,a.dy)}).sort(function(a,b){return b.dy-a.dy});t.append(\"title\").text(n);var u=m.append(\"g\").selectAll(\".node\").data(b.nodes).enter().append(\"g\").attr(\"class\",\"node\").attr(\"transform\",function(a){return\"translate(\"+a.x+\",\"+a.y+\")\"}).call(d3.behavior.drag().origin(function(a){return a}).on(\"dragstart\",function(){this.parentNode.appendChild(this)}).on(\"drag\",c));u.append(\"rect\").attr(\"height\",function(a){return a.dy}).attr(\"width\",d.nodeWidth()).style(\"fill\",o).style(\"stroke\",p).append(\"title\").text(q),u.append(\"text\").attr(\"x\",-6).attr(\"y\",function(a){return a.dy/2}).attr(\"dy\",\".35em\").attr(\"text-anchor\",\"end\").attr(\"transform\",null).text(function(a){return a.name}).filter(function(a){return a.x<e/2}).attr(\"x\",6+d.nodeWidth()).attr(\"text-anchor\",\
\"start\")}),b}var c={top:5,right:0,bottom:5,left:0},d=a.models.sankey(),e=600,f=400,g=36,h=40,i=\"units\",j=void 0,k=d3.format(\",.0f\"),l=function(a){return k(a)+\" \"+i},m=d3.scale.category20(),n=function(a){return a.source.name+\" → \"+a.target.name+\"\\n\"+l(a.value)},o=function(a){return a.color=m(a.name.replace(/ .*/,\"\"))},p=function(a){return d3.rgb(a.color).darker(2)},q=function(a){return a.name+\"\\n\"+l(a.value)},r=function(a,b){a.append(\"text\").attr(\"x\",0).attr(\"y\",0).attr(\"class\",\"nvd3-sankey-chart-error\").attr(\"text-anchor\",\"middle\").text(b)};return b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{units:{get:function(){return i},set:function(a){i=a}},width:{get:function(){return e},set:function(a){e=a}},height:{get:function(){return f},set:function(a){f=a}},format:{get:function(){return l},set:function(a){l=a}},linkTitle:{get:function(){return n},set:function(a){n=a}},nodeWidth:{get:function(){return g},set:function(a){g=a}},nodePadding:{get:function(){return h},set:function(a){h=a}},center:{get:function(){return j},set:function(a){j=a}},margin:{get:function(){return c},set:function(a){c.top=void 0!==a.top?a.top:c.top,c.right=void 0!==a.right?a.right:c.right,c.bottom=void 0!==a.bottom?a.bottom:c.bottom,c.left=void 0!==a.left?a.left:c.left}},nodeStyle:{get:function(){return{}},set:function(a){o=void 0!==a.fillColor?a.fillColor:o,p=void 0!==a.strokeColor?a.strokeColor:p,q=void 0!==a.title?a.title:q}}}),a.utils.initOptions(b),b},a.models.scatter=function(){\"use strict\";function b(a){var b,c;return b=a[0].series+\":\"+a[1],c=Z[b]=Z[b]||{}}function c(a){var b;b=a[0].series+\":\"+a[1],delete Z[b]}function d(a){var c,d,e,f=b(a),g=!1;for(c=1;c<arguments.length;c+=2)d=arguments[c],e=arguments[c+1](a[0],a[1]),f[d]===e&&f.hasOwnProperty(d)||(f[d]=e,g=!0);return g}function e(b){return X.reset(),b.each(function(b){function T(){if(W=!1,!C)return!1;if(S===!0){var c=d3.merge(b.map(function(b,c){return b.values.map(function(b,d){var e=v(b,d),f=w(b,d);return[a.utils.NaNtoZero(s(e))+1e-4*Math.random(),a.utils.NaNtoZero(t(f))+1e-4*Math.random(),\
c,d,b]}).filter(function(a,b){return D(a[4],b)})}));if(0==c.length)return!1;c.length<3&&(c.push([s.range()[0]-20,t.range()[0]-20,null,null]),c.push([s.range()[1]+20,t.range()[1]+20,null,null]),c.push([s.range()[0]-20,t.range()[0]+20,null,null]),c.push([s.range()[1]+20,t.range()[1]-20,null,null]));var d=d3.geom.polygon([[-10,-10],[-10,n+10],[m+10,n+10],[m+10,-10]]),e=d3.geom.voronoi(c).map(function(a,b){return{data:d.clip(a),series:c[b][2],point:c[b][3]}});ea.select(\".nv-point-paths\").selectAll(\"path\").remove();var f=ea.select(\".nv-point-paths\").selectAll(\"path\").data(e),g=f.enter().append(\"svg:path\").attr(\"d\",function(a){return a&&a.data&&0!==a.data.length?\"M\"+a.data.join(\",\")+\"Z\":\"M 0 0\"}).attr(\"id\",function(a,b){return\"nv-path-\"+b}).attr(\"clip-path\",function(a,b){return\"url(#nv-clip-\"+q+\"-\"+b+\")\"});if(I&&g.style(\"fill\",d3.rgb(230,230,230)).style(\"fill-opacity\",.4).style(\"stroke-opacity\",1).style(\"stroke\",d3.rgb(200,200,200)),H){ea.select(\".nv-point-clips\").selectAll(\"*\").remove();var h=ea.select(\".nv-point-clips\").selectAll(\"clipPath\").data(c);h.enter().append(\"svg:clipPath\").attr(\"id\",function(a,b){return\"nv-clip-\"+q+\"-\"+b}).append(\"svg:circle\").attr(\"cx\",function(a){return a[0]}).attr(\"cy\",function(a){return a[1]}).attr(\"r\",J)}var i=function(a,c,d){if(W)return 0;var e=b[c.series];if(void 0!==e){var f=e.values[c.point];f.color=o(e,c.series),f.x=v(f),f.y=w(f);var g=r.node().getBoundingClientRect(),h=window.pageYOffset||document.documentElement.scrollTop,i=window.pageXOffset||document.documentElement.scrollLeft,j={left:s(v(f,c.point))+g.left+i+l.left+10,top:t(w(f,c.point))+g.top+h+l.top+10};d({point:f,series:e,pos:j,relativePos:[s(v(f,c.point))+l.left,t(w(f,c.point))+l.top],seriesIndex:c.series,pointIndex:c.point,event:d3.event,element:a})}};f.on(\"click\",function(a){i(this,a,R.elementClick)}).on(\"dblclick\",function(a){i(this,a,R.elementDblClick)}).on(\"mouseover\",function(a){i(this,a,R.elementMouseover)}).on(\"mouseout\",function(a,b){i(this,a,R.elementMouseout)})}else ea.select(\".nv-groups\").selectAll(\".nv-group\").selectAll(\".nv-point\").on(\"click\",\
function(a,c){if(W||!b[a.series])return 0;var d=b[a.series],e=d.values[c],f=this;R.elementClick({point:e,series:d,pos:[s(v(e,c))+l.left,t(w(e,c))+l.top],relativePos:[s(v(e,c))+l.left,t(w(e,c))+l.top],seriesIndex:a.series,pointIndex:c,event:d3.event,element:f})}).on(\"dblclick\",function(a,c){if(W||!b[a.series])return 0;var d=b[a.series],e=d.values[c];R.elementDblClick({point:e,series:d,pos:[s(v(e,c))+l.left,t(w(e,c))+l.top],relativePos:[s(v(e,c))+l.left,t(w(e,c))+l.top],seriesIndex:a.series,pointIndex:c})}).on(\"mouseover\",function(a,c){if(W||!b[a.series])return 0;var d=b[a.series],e=d.values[c];R.elementMouseover({point:e,series:d,pos:[s(v(e,c))+l.left,t(w(e,c))+l.top],relativePos:[s(v(e,c))+l.left,t(w(e,c))+l.top],seriesIndex:a.series,pointIndex:c,color:o(a,c)})}).on(\"mouseout\",function(a,c){if(W||!b[a.series])return 0;var d=b[a.series],e=d.values[c];R.elementMouseout({point:e,series:d,pos:[s(v(e,c))+l.left,t(w(e,c))+l.top],relativePos:[s(v(e,c))+l.left,t(w(e,c))+l.top],seriesIndex:a.series,pointIndex:c,color:o(a,c)})})}r=d3.select(this);var Z=a.utils.availableWidth(m,r,l),$=a.utils.availableHeight(n,r,l);a.utils.initSVG(r),b.forEach(function(a,b){a.values.forEach(function(a){a.series=b})});var _=e.yScale().name===d3.scale.log().name?!0:!1,aa=K&&L&&O?[]:d3.merge(b.map(function(a){return a.values.map(function(a,b){return{x:v(a,b),y:w(a,b),size:x(a,b)}})}));if(s.domain(K||d3.extent(aa.map(function(a){return a.x}).concat(z))),E&&b[0]?s.range(M||[(Z*F+Z)/(2*b[0].values.length),Z-Z*(1+F)/(2*b[0].values.length)]):s.range(M||[0,Z]),_){var ba=d3.min(aa.map(function(a){return 0!==a.y?a.y:void 0}));t.clamp(!0).domain(L||d3.extent(aa.map(function(a){return 0!==a.y?a.y:.1*ba}).concat(A))).range(N||[$,0])}else t.domain(L||d3.extent(aa.map(function(a){return a.y}).concat(A))).range(N||[$,0]);u.domain(O||d3.extent(aa.map(function(a){return a.size}).concat(B))).range(P||Y),Q=s.domain()[0]===s.domain()[1]||t.domain()[0]===t.domain()[1],s.domain()[0]===s.domain()[1]&&(s.domain()[0]?s.domain([s.domain()[0]-.01*s.domain()[0],s.domain()[1]+.01*s.domain()[1]]):s.domain([-1,\
1])),t.domain()[0]===t.domain()[1]&&(t.domain()[0]?t.domain([t.domain()[0]-.01*t.domain()[0],t.domain()[1]+.01*t.domain()[1]]):t.domain([-1,1])),isNaN(s.domain()[0])&&s.domain([-1,1]),isNaN(t.domain()[0])&&t.domain([-1,1]),f=f||s,g=g||t,h=h||u;var ca=s(1)!==f(1)||t(1)!==g(1)||u(1)!==h(1);i=i||m,j=j||n;var da=i!==m||j!==n,ea=r.selectAll(\"g.nv-wrap.nv-scatter\").data([b]),fa=ea.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-scatter nv-chart-\"+q),ga=fa.append(\"defs\"),ha=fa.append(\"g\"),ia=ea.select(\"g\");ea.classed(\"nv-single-point\",Q),ha.append(\"g\").attr(\"class\",\"nv-groups\"),ha.append(\"g\").attr(\"class\",\"nv-point-paths\"),fa.append(\"g\").attr(\"class\",\"nv-point-clips\"),ea.attr(\"transform\",\"translate(\"+l.left+\",\"+l.top+\")\"),ga.append(\"clipPath\").attr(\"id\",\"nv-edge-clip-\"+q).append(\"rect\").attr(\"transform\",\"translate( -10, -10)\"),ea.select(\"#nv-edge-clip-\"+q+\" rect\").attr(\"width\",Z+20).attr(\"height\",$>0?$+20:0),ia.attr(\"clip-path\",G?\"url(#nv-edge-clip-\"+q+\")\":\"\"),W=!0;var ja=ea.select(\".nv-groups\").selectAll(\".nv-group\").data(function(a){return a},function(a){return a.key});ja.enter().append(\"g\").style(\"stroke-opacity\",1e-6).style(\"fill-opacity\",1e-6),ja.exit().remove(),ja.attr(\"class\",function(a,b){return(a.classed||\"\")+\" nv-group nv-series-\"+b}).classed(\"nv-noninteractive\",!C).classed(\"hover\",function(a){return a.hover}),ja.watchTransition(X,\"scatter: groups\").style(\"fill\",function(a,b){return o(a,b)}).style(\"stroke\",function(a,b){return a.pointBorderColor||p||o(a,b)}).style(\"stroke-opacity\",1).style(\"fill-opacity\",.5);var ka=ja.selectAll(\"path.nv-point\").data(function(a){return a.values.map(function(a,b){return[a,b]}).filter(function(a,b){return D(a[0],b)})});if(ka.enter().append(\"path\").attr(\"class\",function(a){return\"nv-point nv-point-\"+a[1]}).style(\"fill\",function(a){return a.color}).style(\"stroke\",function(a){return a.color}).attr(\"transform\",function(b){return\"translate(\"+a.utils.NaNtoZero(f(v(b[0],b[1])))+\",\"+a.utils.NaNtoZero(g(w(b[0],b[1])))+\")\"}).attr(\"d\",a.utils.symbol().type(function(a){return y(a[0])}).size(function(a){return \
u(x(a[0],a[1]))})),ka.exit().each(c).remove(),ja.exit().selectAll(\"path.nv-point\").watchTransition(X,\"scatter exit\").attr(\"transform\",function(b){return\"translate(\"+a.utils.NaNtoZero(s(v(b[0],b[1])))+\",\"+a.utils.NaNtoZero(t(w(b[0],b[1])))+\")\"}).remove(),ka.filter(function(a){return ca||da||d(a,\"x\",v,\"y\",w)}).watchTransition(X,\"scatter points\").attr(\"transform\",function(b){return\"translate(\"+a.utils.NaNtoZero(s(v(b[0],b[1])))+\",\"+a.utils.NaNtoZero(t(w(b[0],b[1])))+\")\"}),ka.filter(function(a){return ca||da||d(a,\"shape\",y,\"size\",x)}).watchTransition(X,\"scatter points\").attr(\"d\",a.utils.symbol().type(function(a){return y(a[0])}).size(function(a){return u(x(a[0],a[1]))})),V){var la=ja.selectAll(\".nv-label\").data(function(a){return a.values.map(function(a,b){return[a,b]}).filter(function(a,b){return D(a[0],b)})});la.enter().append(\"text\").style(\"fill\",function(a,b){return a.color}).style(\"stroke-opacity\",0).style(\"fill-opacity\",1).attr(\"transform\",function(b){var c=a.utils.NaNtoZero(f(v(b[0],b[1])))+Math.sqrt(u(x(b[0],b[1]))/Math.PI)+2;return\"translate(\"+c+\",\"+a.utils.NaNtoZero(g(w(b[0],b[1])))+\")\"}).text(function(a,b){return a[0].label}),la.exit().remove(),ja.exit().selectAll(\"path.nv-label\").watchTransition(X,\"scatter exit\").attr(\"transform\",function(b){var c=a.utils.NaNtoZero(s(v(b[0],b[1])))+Math.sqrt(u(x(b[0],b[1]))/Math.PI)+2;return\"translate(\"+c+\",\"+a.utils.NaNtoZero(t(w(b[0],b[1])))+\")\"}).remove(),la.each(function(a){d3.select(this).classed(\"nv-label\",!0).classed(\"nv-label-\"+a[1],!1).classed(\"hover\",!1)}),la.watchTransition(X,\"scatter labels\").attr(\"transform\",function(b){var c=a.utils.NaNtoZero(s(v(b[0],b[1])))+Math.sqrt(u(x(b[0],b[1]))/Math.PI)+2;return\"translate(\"+c+\",\"+a.utils.NaNtoZero(t(w(b[0],b[1])))+\")\"})}U?(clearTimeout(k),k=setTimeout(T,U)):T(),f=s.copy(),g=t.copy(),h=u.copy(),i=m,j=n}),X.renderEnd(\"scatter immediate\"),e}var f,g,h,i,j,k,l={top:0,right:0,bottom:0,left:0},m=null,n=null,o=a.utils.defaultColor(),p=null,q=Math.floor(1e5*Math.random()),r=null,s=d3.scale.linear(),t=d3.scale.linear(),u=d3.scale.linear(),\
v=function(a){return a.x},w=function(a){return a.y},x=function(a){return a.size||1},y=function(a){return a.shape||\"circle\"},z=[],A=[],B=[],C=!0,D=function(a){return!a.notActive},E=!1,F=.1,G=!1,H=!0,I=!1,J=function(){return 25},K=null,L=null,M=null,N=null,O=null,P=null,Q=!1,R=d3.dispatch(\"elementClick\",\"elementDblClick\",\"elementMouseover\",\"elementMouseout\",\"renderEnd\"),S=!0,T=250,U=300,V=!1,W=!1,X=a.utils.renderWatch(R,T),Y=[16,256],Z={};return e.dispatch=R,e.options=a.utils.optionsFunc.bind(e),e._calls=new function(){this.clearHighlights=function(){return a.dom.write(function(){r.selectAll(\".nv-point.hover\").classed(\"hover\",!1)}),null},this.highlightPoint=function(b,c,d){a.dom.write(function(){r.select(\".nv-groups\").selectAll(\".nv-series-\"+b).selectAll(\".nv-point-\"+c).classed(\"hover\",d)})}},R.on(\"elementMouseover.point\",function(a){C&&e._calls.highlightPoint(a.seriesIndex,a.pointIndex,!0)}),R.on(\"elementMouseout.point\",function(a){C&&e._calls.highlightPoint(a.seriesIndex,a.pointIndex,!1)}),e._options=Object.create({},{width:{get:function(){return m},set:function(a){m=a}},height:{get:function(){return n},set:function(a){n=a}},xScale:{get:function(){return s},set:function(a){s=a}},yScale:{get:function(){return t},set:function(a){t=a}},pointScale:{get:function(){return u},set:function(a){u=a}},xDomain:{get:function(){return K},set:function(a){K=a}},yDomain:{get:function(){return L},set:function(a){L=a}},pointDomain:{get:function(){return O},set:function(a){O=a}},xRange:{get:function(){return M},set:function(a){M=a}},yRange:{get:function(){return N},set:function(a){N=a}},pointRange:{get:function(){return P},set:function(a){P=a}},forceX:{get:function(){return z},set:function(a){z=a}},forceY:{get:function(){return A},set:function(a){A=a}},forcePoint:{get:function(){return B},set:function(a){B=a}},interactive:{get:function(){return C},set:function(a){C=a}},pointActive:{get:function(){return D},set:function(a){D=a}},padDataOuter:{get:function(){return F},set:function(a){F=a}},padData:{get:function(){return E},set:function(a){E=a}},\
clipEdge:{get:function(){return G},set:function(a){G=a}},clipVoronoi:{get:function(){return H},set:function(a){H=a}},clipRadius:{get:function(){return J},set:function(a){J=a}},showVoronoi:{get:function(){return I},set:function(a){I=a}},id:{get:function(){return q},set:function(a){q=a}},interactiveUpdateDelay:{get:function(){return U},set:function(a){U=a}},showLabels:{get:function(){return V},set:function(a){V=a}},pointBorderColor:{get:function(){return p},set:function(a){p=a}},x:{get:function(){return v},set:function(a){v=d3.functor(a)}},y:{get:function(){return w},set:function(a){w=d3.functor(a)}},pointSize:{get:function(){return x},set:function(a){x=d3.functor(a)}},pointShape:{get:function(){return y},set:function(a){y=d3.functor(a)}},margin:{get:function(){return l},set:function(a){l.top=void 0!==a.top?a.top:l.top,l.right=void 0!==a.right?a.right:l.right,l.bottom=void 0!==a.bottom?a.bottom:l.bottom,l.left=void 0!==a.left?a.left:l.left}},duration:{get:function(){return T},set:function(a){T=a,X.reset(T)}},color:{get:function(){return o},set:function(b){o=a.utils.getColor(b)}},useVoronoi:{get:function(){return S},set:function(a){S=a,S===!1&&(H=!1)}}}),a.utils.initOptions(e),e},a.models.scatterChart=function(){\"use strict\";function b(A){return F.reset(),F.models(c),u&&F.models(d),v&&F.models(e),r&&F.models(g),s&&F.models(h),A.each(function(A){n=d3.select(this),a.utils.initSVG(n);var I=a.utils.availableWidth(l,n,j),J=a.utils.availableHeight(m,n,j);if(b.update=function(){0===B?n.call(b):n.transition().duration(B).call(b)},b.container=this,x.setter(H(A),b.update).getter(G(A)).update(),x.disabled=A.map(function(a){return!!a.disabled}),!y){var K;y={};for(K in x)x[K]instanceof Array?y[K]=x[K].slice(0):y[K]=x[K]}if(!(A&&A.length&&A.filter(function(a){return a.values.length}).length))return a.utils.noData(b,n),F.renderEnd(\"scatter immediate\"),b;n.selectAll(\".nv-noData\").remove(),p=c.xScale(),q=c.yScale();var L=n.selectAll(\"g.nv-wrap.nv-scatterChart\").data([A]),M=L.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-scatterChart \
nv-chart-\"+c.id()),N=M.append(\"g\"),O=L.select(\"g\");if(N.append(\"rect\").attr(\"class\",\"nvd3 nv-background\").style(\"pointer-events\",\"none\"),N.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),N.append(\"g\").attr(\"class\",\"nv-y nv-axis\"),N.append(\"g\").attr(\"class\",\"nv-scatterWrap\"),N.append(\"g\").attr(\"class\",\"nv-regressionLinesWrap\"),N.append(\"g\").attr(\"class\",\"nv-distWrap\"),N.append(\"g\").attr(\"class\",\"nv-legendWrap\"),w&&O.select(\".nv-y.nv-axis\").attr(\"transform\",\"translate(\"+I+\",0)\"),t){var P=I;f.width(P),L.select(\".nv-legendWrap\").datum(A).call(f),k||f.height()===j.top||(j.top=f.height(),J=a.utils.availableHeight(m,n,j)),L.select(\".nv-legendWrap\").attr(\"transform\",\"translate(0,\"+-j.top+\")\")}else O.select(\".nv-legendWrap\").selectAll(\"*\").remove();L.attr(\"transform\",\"translate(\"+j.left+\",\"+j.top+\")\"),c.width(I).height(J).color(A.map(function(a,b){return a.color=a.color||o(a,b),a.color}).filter(function(a,b){return!A[b].disabled})).showLabels(C),L.select(\".nv-scatterWrap\").datum(A.filter(function(a){return!a.disabled})).call(c),L.select(\".nv-regressionLinesWrap\").attr(\"clip-path\",\"url(#nv-edge-clip-\"+c.id()+\")\");var Q=L.select(\".nv-regressionLinesWrap\").selectAll(\".nv-regLines\").data(function(a){return a});Q.enter().append(\"g\").attr(\"class\",\"nv-regLines\");var R=Q.selectAll(\".nv-regLine\").data(function(a){return[a]});R.enter().append(\"line\").attr(\"class\",\"nv-regLine\").style(\"stroke-opacity\",0),R.filter(function(a){return a.intercept&&a.slope}).watchTransition(F,\"scatterPlusLineChart: regline\").attr(\"x1\",p.range()[0]).attr(\"x2\",p.range()[1]).attr(\"y1\",function(a,b){return q(p.domain()[0]*a.slope+a.intercept)}).attr(\"y2\",function(a,b){return q(p.domain()[1]*a.slope+a.intercept)}).style(\"stroke\",function(a,b,c){return o(a,c)}).style(\"stroke-opacity\",function(a,b){return a.disabled||\"undefined\"==typeof a.slope||\"undefined\"==typeof a.intercept?0:1}),u&&(d.scale(p)._ticks(a.utils.calcTicksX(I/100,A)).tickSize(-J,0),O.select(\".nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+q.range()[0]+\")\").call(d)),v&&(e.scale(q)._ticks(a.utils.calcTicksY(J/36,\
A)).tickSize(-I,0),O.select(\".nv-y.nv-axis\").call(e)),r&&(g.getData(c.x()).scale(p).width(I).color(A.map(function(a,b){return a.color||o(a,b)}).filter(function(a,b){return!A[b].disabled})),N.select(\".nv-distWrap\").append(\"g\").attr(\"class\",\"nv-distributionX\"),O.select(\".nv-distributionX\").attr(\"transform\",\"translate(0,\"+q.range()[0]+\")\").datum(A.filter(function(a){return!a.disabled})).call(g)),s&&(h.getData(c.y()).scale(q).width(J).color(A.map(function(a,b){return a.color||o(a,b)}).filter(function(a,b){return!A[b].disabled})),N.select(\".nv-distWrap\").append(\"g\").attr(\"class\",\"nv-distributionY\"),O.select(\".nv-distributionY\").attr(\"transform\",\"translate(\"+(w?I:-h.size())+\",0)\").datum(A.filter(function(a){return!a.disabled})).call(h)),f.dispatch.on(\"stateChange\",function(a){for(var c in a)x[c]=a[c];z.stateChange(x),b.update()}),z.on(\"changeState\",function(a){\"undefined\"!=typeof a.disabled&&(A.forEach(function(b,c){b.disabled=a.disabled[c]}),x.disabled=a.disabled),b.update()}),c.dispatch.on(\"elementMouseout.tooltip\",function(a){i.hidden(!0),n.select(\".nv-chart-\"+c.id()+\" .nv-series-\"+a.seriesIndex+\" .nv-distx-\"+a.pointIndex).attr(\"y1\",0),n.select(\".nv-chart-\"+c.id()+\" .nv-series-\"+a.seriesIndex+\" .nv-disty-\"+a.pointIndex).attr(\"x2\",h.size())}),c.dispatch.on(\"elementMouseover.tooltip\",function(a){n.select(\".nv-series-\"+a.seriesIndex+\" .nv-distx-\"+a.pointIndex).attr(\"y1\",a.relativePos[1]-J),n.select(\".nv-series-\"+a.seriesIndex+\" .nv-disty-\"+a.pointIndex).attr(\"x2\",a.relativePos[0]+g.size()),i.data(a).hidden(!1)}),D=p.copy(),E=q.copy()}),F.renderEnd(\"scatter with line immediate\"),b}var c=a.models.scatter(),d=a.models.axis(),e=a.models.axis(),f=a.models.legend(),g=a.models.distribution(),h=a.models.distribution(),i=a.models.tooltip(),j={top:30,right:20,bottom:50,left:75},k=null,l=null,m=null,n=null,o=a.utils.defaultColor(),p=c.xScale(),q=c.yScale(),r=!1,s=!1,t=!0,u=!0,v=!0,w=!1,x=a.utils.state(),y=null,z=d3.dispatch(\"stateChange\",\"changeState\",\"renderEnd\"),A=null,B=250,C=!1;c.xScale(p).yScale(q),d.orient(\"bottom\").tickPadding(10),\
e.orient(w?\"right\":\"left\").tickPadding(10),g.axis(\"x\"),h.axis(\"y\"),i.headerFormatter(function(a,b){return d.tickFormat()(a,b)}).valueFormatter(function(a,b){return e.tickFormat()(a,b)});var D,E,F=a.utils.renderWatch(z,B),G=function(a){return function(){return{active:a.map(function(a){return!a.disabled})}}},H=function(a){return function(b){void 0!==b.active&&a.forEach(function(a,c){a.disabled=!b.active[c]})}};return b.dispatch=z,b.scatter=c,b.legend=f,b.xAxis=d,b.yAxis=e,b.distX=g,b.distY=h,b.tooltip=i,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return l},set:function(a){l=a}},height:{get:function(){return m},set:function(a){m=a}},container:{get:function(){return n},set:function(a){n=a}},showDistX:{get:function(){return r},set:function(a){r=a}},showDistY:{get:function(){return s},set:function(a){s=a}},showLegend:{get:function(){return t},set:function(a){t=a}},showXAxis:{get:function(){return u},set:function(a){u=a}},showYAxis:{get:function(){return v},set:function(a){v=a}},defaultState:{get:function(){return y},set:function(a){y=a}},noData:{get:function(){return A},set:function(a){A=a}},duration:{get:function(){return B},set:function(a){B=a}},showLabels:{get:function(){return C},set:function(a){C=a}},margin:{get:function(){return j},set:function(a){void 0!==a.top&&(j.top=a.top,k=a.top),j.right=void 0!==a.right?a.right:j.right,j.bottom=void 0!==a.bottom?a.bottom:j.bottom,j.left=void 0!==a.left?a.left:j.left}},rightAlignYAxis:{get:function(){return w},set:function(a){w=a,e.orient(a?\"right\":\"left\")}},color:{get:function(){return o},set:function(b){o=a.utils.getColor(b),f.color(o),g.color(o),h.color(o)}}}),a.utils.inheritOptions(b,c),a.utils.initOptions(b),b},a.models.sparkline=function(){\"use strict\";function b(k){return t.reset(),k.each(function(b){var k=h-g.left-g.right,s=i-g.top-g.bottom;j=d3.select(this),a.utils.initSVG(j),l.domain(c||d3.extent(b,n)).range(e||[0,k]),m.domain(d||d3.extent(b,o)).range(f||[s,0]);var t=j.selectAll(\"g.nv-wrap.nv-sparkline\").data([b]),u=t.enter().append(\"g\").attr(\"class\",\
\"nvd3 nv-wrap nv-sparkline\");u.append(\"g\"),t.select(\"g\");t.attr(\"transform\",\"translate(\"+g.left+\",\"+g.top+\")\");var v=t.selectAll(\"path\").data(function(a){return[a]});v.enter().append(\"path\"),v.exit().remove(),v.style(\"stroke\",function(a,b){return a.color||p(a,b)}).attr(\"d\",d3.svg.line().x(function(a,b){return l(n(a,b))}).y(function(a,b){return m(o(a,b))}));var w=t.selectAll(\"circle.nv-point\").data(function(a){function b(b){if(-1!=b){var c=a[b];return c.pointIndex=b,c}return null}var c=a.map(function(a,b){return o(a,b)}),d=b(c.lastIndexOf(m.domain()[1])),e=b(c.indexOf(m.domain()[0])),f=b(c.length-1);return[q?e:null,q?d:null,r?f:null].filter(function(a){return null!=a})});w.enter().append(\"circle\"),w.exit().remove(),w.attr(\"cx\",function(a,b){return l(n(a,a.pointIndex))}).attr(\"cy\",function(a,b){return m(o(a,a.pointIndex))}).attr(\"r\",2).attr(\"class\",function(a,b){return n(a,a.pointIndex)==l.domain()[1]?\"nv-point nv-currentValue\":o(a,a.pointIndex)==m.domain()[0]?\"nv-point nv-minValue\":\"nv-point nv-maxValue\"})}),t.renderEnd(\"sparkline immediate\"),b}var c,d,e,f,g={top:2,right:0,bottom:2,left:0},h=400,i=32,j=null,k=!0,l=d3.scale.linear(),m=d3.scale.linear(),n=function(a){return a.x},o=function(a){return a.y},p=a.utils.getColor([\"#000\"]),q=!0,r=!0,s=d3.dispatch(\"renderEnd\"),t=a.utils.renderWatch(s);return b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return h},set:function(a){h=a}},height:{get:function(){return i},set:function(a){i=a}},xDomain:{get:function(){return c},set:function(a){c=a}},yDomain:{get:function(){return d},set:function(a){d=a}},xRange:{get:function(){return e},set:function(a){e=a}},yRange:{get:function(){return f},set:function(a){f=a}},xScale:{get:function(){return l},set:function(a){l=a}},yScale:{get:function(){return m},set:function(a){m=a}},animate:{get:function(){return k},set:function(a){k=a}},showMinMaxPoints:{get:function(){return q},set:function(a){q=a}},showCurrentPoint:{get:function(){return r},set:function(a){r=a}},x:{get:function(){return n},set:function(a){n=d3.functor(a)}},\
y:{get:function(){return o},set:function(a){o=d3.functor(a)}},margin:{get:function(){return g},set:function(a){g.top=void 0!==a.top?a.top:g.top,g.right=void 0!==a.right?a.right:g.right,g.bottom=void 0!==a.bottom?a.bottom:g.bottom,g.left=void 0!==a.left?a.left:g.left}},color:{get:function(){return p},set:function(b){p=a.utils.getColor(b)}}}),b.dispatch=s,a.utils.initOptions(b),b},a.models.sparklinePlus=function(){\"use strict\";function b(p){return r.reset(),r.models(e),p.each(function(p){function q(){if(!j){var a=z.selectAll(\".nv-hoverValue\").data(i),b=a.enter().append(\"g\").attr(\"class\",\"nv-hoverValue\").style(\"stroke-opacity\",0).style(\"fill-opacity\",0);a.exit().transition().duration(250).style(\"stroke-opacity\",0).style(\"fill-opacity\",0).remove(),a.attr(\"transform\",function(a){return\"translate(\"+c(e.x()(p[a],a))+\",0)\"}).transition().duration(250).style(\"stroke-opacity\",1).style(\"fill-opacity\",1),i.length&&(b.append(\"line\").attr(\"x1\",0).attr(\"y1\",-f.top).attr(\"x2\",0).attr(\"y2\",u),\
b.append(\"text\").attr(\"class\",\"nv-xValue\").attr(\"x\",-6).attr(\"y\",-f.top).attr(\"text-anchor\",\"end\").attr(\"dy\",\".9em\"),z.select(\".nv-hoverValue .nv-xValue\").text(k(e.x()(p[i[0]],i[0]))),b.append(\"text\").attr(\"class\",\"nv-yValue\").attr(\"x\",6).attr(\"y\",-f.top).attr(\"text-anchor\",\"start\").attr(\"dy\",\".9em\"),z.select(\".nv-hoverValue .nv-yValue\").text(l(e.y()(p[i[0]],i[0]))))}}function r(){function a(a,b){for(var c=Math.abs(e.x()(a[0],0)-b),d=0,f=0;f<a.length;f++)Math.abs(e.x()(a[f],f)-b)<c&&(c=Math.abs(e.x()(a[f],f)-b),d=f);return d}if(!j){var b=d3.mouse(this)[0]-f.left;i=[a(p,Math.round(c.invert(b)))],q()}}var s=d3.select(this);a.utils.initSVG(s);var t=a.utils.availableWidth(g,s,f),u=a.utils.availableHeight(h,s,f);if(b.update=function(){s.call(b)},b.container=this,!p||!p.length)return a.utils.noData(b,s),b;s.selectAll(\".nv-noData\").remove();var v=e.y()(p[p.length-1],p.length-1);c=e.xScale(),d=e.yScale();var w=s.selectAll(\"g.nv-wrap.nv-sparklineplus\").data([p]),x=w.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-sparklineplus\"),y=x.append(\"g\"),z=w.select(\"g\");y.append(\"g\").attr(\"class\",\"nv-sparklineWrap\"),y.append(\"g\").attr(\"class\",\"nv-valueWrap\"),y.append(\"g\").attr(\"class\",\"nv-hoverArea\"),w.attr(\"transform\",\"translate(\"+f.left+\",\"+f.top+\")\");var A=z.select(\".nv-sparklineWrap\");if(e.width(t).height(u),A.call(e),m){var B=z.select(\".nv-valueWrap\"),C=B.selectAll(\".nv-currentValue\").data([v]);C.enter().append(\"text\").attr(\"class\",\"nv-currentValue\").attr(\"dx\",o?-8:8).attr(\"dy\",\".9em\").style(\"text-anchor\",o?\"end\":\"start\"),C.attr(\"x\",t+(o?f.right:0)).attr(\"y\",n?function(a){return d(a)}:0).style(\"fill\",e.color()(p[p.length-1],p.length-1)).text(l(v))}y.select(\".nv-hoverArea\").append(\"rect\").on(\"mousemove\",r).on(\"click\",function(){j=!j}).on(\"mouseout\",function(){i=[],q()}),z.select(\".nv-hoverArea rect\").attr(\"transform\",function(a){return\"translate(\"+-f.left+\",\"+-f.top+\")\"}).attr(\"width\",t+f.left+f.right).attr(\"height\",u+f.top)}),r.renderEnd(\"sparklinePlus immediate\"),b}var c,d,e=a.models.sparkline(),f={top:15,right:100,bottom:10,\
left:50},g=null,h=null,i=[],j=!1,k=d3.format(\",r\"),l=d3.format(\",.2f\"),m=!0,n=!0,o=!1,p=null,q=d3.dispatch(\"renderEnd\"),r=a.utils.renderWatch(q);return b.dispatch=q,b.sparkline=e,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return g},set:function(a){g=a}},height:{get:function(){return h},set:function(a){h=a}},xTickFormat:{get:function(){return k},set:function(a){k=a}},yTickFormat:{get:function(){return l},set:function(a){l=a}},showLastValue:{get:function(){return m},set:function(a){m=a}},alignValue:{get:function(){return n},set:function(a){n=a}},rightAlignValue:{get:function(){return o},set:function(a){o=a}},noData:{get:function(){return p},set:function(a){p=a}},margin:{get:function(){return f},set:function(a){f.top=void 0!==a.top?a.top:f.top,f.right=void 0!==a.right?a.right:f.right,f.bottom=void 0!==a.bottom?a.bottom:f.bottom,f.left=void 0!==a.left?a.left:f.left}}}),a.utils.inheritOptions(b,e),a.utils.initOptions(b),b},a.models.stackedArea=function(){\"use strict\";function b(n){return v.reset(),v.models(s),n.each(function(n){var t=f-e.left-e.right,w=g-e.top-e.bottom;j=d3.select(this),a.utils.initSVG(j),c=s.xScale(),d=s.yScale();var x=n;n.forEach(function(a,b){a.seriesIndex=b,a.values=a.values.map(function(a,c){return a.index=c,a.seriesIndex=b,a})});var y=n.filter(function(a){return!a.disabled});n=d3.layout.stack().order(p).offset(o).values(function(a){return a.values}).x(k).y(l).out(function(a,b,c){a.display={y:c,y0:b}})(y);var z=j.selectAll(\"g.nv-wrap.nv-stackedarea\").data([n]),A=z.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-stackedarea\"),B=A.append(\"defs\"),C=A.append(\"g\"),D=z.select(\"g\");C.append(\"g\").attr(\"class\",\"nv-areaWrap\"),C.append(\"g\").attr(\"class\",\"nv-scatterWrap\"),z.attr(\"transform\",\"translate(\"+e.left+\",\"+e.top+\")\"),0==s.forceY().length&&s.forceY().push(0),s.width(t).height(w).x(k).y(function(a){return void 0!==a.display?a.display.y+a.display.y0:void 0}).color(n.map(function(a,b){return a.color=a.color||h(a,a.seriesIndex),a.color}));var E=D.select(\".nv-scatterWrap\").datum(n);\
E.call(s),B.append(\"clipPath\").attr(\"id\",\"nv-edge-clip-\"+i).append(\"rect\"),z.select(\"#nv-edge-clip-\"+i+\" rect\").attr(\"width\",t).attr(\"height\",w),D.attr(\"clip-path\",r?\"url(#nv-edge-clip-\"+i+\")\":\"\");var F=d3.svg.area().defined(m).x(function(a,b){return c(k(a,b))}).y0(function(a){return d(a.display.y0)}).y1(function(a){return d(a.display.y+a.display.y0)}).interpolate(q),G=d3.svg.area().defined(m).x(function(a,b){return c(k(a,b))}).y0(function(a){return d(a.display.y0)}).y1(function(a){return d(a.display.y0)}),H=D.select(\".nv-areaWrap\").selectAll(\"path.nv-area\").data(function(a){return a});H.enter().append(\"path\").attr(\"class\",function(a,b){return\"nv-area nv-area-\"+b}).attr(\"d\",function(a,b){return G(a.values,a.seriesIndex)}).on(\"mouseover\",function(a,b){d3.select(this).classed(\"hover\",!0),u.areaMouseover({point:a,series:a.key,pos:[d3.event.pageX,d3.event.pageY],seriesIndex:a.seriesIndex})}).on(\"mouseout\",function(a,b){d3.select(this).classed(\"hover\",!1),u.areaMouseout({point:a,series:a.key,pos:[d3.event.pageX,d3.event.pageY],seriesIndex:a.seriesIndex})}).on(\"click\",function(a,b){d3.select(this).classed(\"hover\",!1),u.areaClick({point:a,series:a.key,pos:[d3.event.pageX,d3.event.pageY],seriesIndex:a.seriesIndex})}),H.exit().remove(),H.style(\"fill\",function(a,b){return a.color||h(a,a.seriesIndex)}).style(\"stroke\",function(a,b){return a.color||h(a,a.seriesIndex)}),H.watchTransition(v,\"stackedArea path\").attr(\"d\",function(a,b){return F(a.values,b)}),s.dispatch.on(\"elementMouseover.area\",function(a){D.select(\".nv-chart-\"+i+\" .nv-area-\"+a.seriesIndex).classed(\"hover\",!0)}),s.dispatch.on(\"elementMouseout.area\",function(a){D.select(\".nv-chart-\"+i+\" .nv-area-\"+a.seriesIndex).classed(\"hover\",!1)}),b.d3_stackedOffset_stackPercent=function(a){var b,c,d,e=a.length,f=a[0].length,g=[];for(c=0;f>c;++c){for(b=0,d=0;b<x.length;b++)d+=l(x[b].values[c]);if(d)for(b=0;e>b;b++)a[b][c][1]/=d;else for(b=0;e>b;b++)a[b][c][1]=0}for(c=0;f>c;++c)g[c]=0;return g}}),v.renderEnd(\"stackedArea immediate\"),b}var c,d,e={top:0,right:0,bottom:0,left:0},f=960,\
g=500,h=a.utils.defaultColor(),i=Math.floor(1e5*Math.random()),j=null,k=function(a){return a.x},l=function(a){return a.y},m=function(a,b){return!isNaN(l(a,b))&&null!==l(a,b)},n=\"stack\",o=\"zero\",p=\"default\",q=\"linear\",r=!1,s=a.models.scatter(),t=250,u=d3.dispatch(\"areaClick\",\"areaMouseover\",\"areaMouseout\",\"renderEnd\",\"elementClick\",\"elementMouseover\",\"elementMouseout\");s.pointSize(2.2).pointDomain([2.2,2.2]);var v=a.utils.renderWatch(u,t);return b.dispatch=u,b.scatter=s,s.dispatch.on(\"elementClick\",function(){u.elementClick.apply(this,arguments)}),s.dispatch.on(\"elementMouseover\",function(){u.elementMouseover.apply(this,arguments)}),s.dispatch.on(\"elementMouseout\",function(){u.elementMouseout.apply(this,arguments)}),b.interpolate=function(a){return arguments.length?(q=a,b):q},b.duration=function(a){return arguments.length?(t=a,v.reset(t),s.duration(t),b):t},b.dispatch=u,b.scatter=s,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return f},set:function(a){f=a}},height:{get:function(){return g},set:function(a){g=a}},defined:{get:function(){return m},set:function(a){m=a}},clipEdge:{get:function(){return r},set:function(a){r=a}},offset:{get:function(){return o},set:function(a){o=a}},order:{get:function(){return p},set:function(a){p=a}},interpolate:{get:function(){return q},set:function(a){q=a}},x:{get:function(){return k},set:function(a){k=d3.functor(a)}},y:{get:function(){return l},set:function(a){l=d3.functor(a)}},margin:{get:function(){return e},set:function(a){e.top=void 0!==a.top?a.top:e.top,e.right=void 0!==a.right?a.right:e.right,e.bottom=void 0!==a.bottom?a.bottom:e.bottom,e.left=void 0!==a.left?a.left:e.left}},color:{get:function(){return h},set:function(b){h=a.utils.getColor(b)}},style:{get:function(){return n},set:function(a){switch(n=a){case\"stack\":b.offset(\"zero\"),b.order(\"default\");break;case\"stream\":b.offset(\"wiggle\"),b.order(\"inside-out\");break;case\"stream-center\":b.offset(\"silhouette\"),b.order(\"inside-out\");break;case\"expand\":b.offset(\"expand\"),b.order(\"default\");\
break;case\"stack_percent\":b.offset(b.d3_stackedOffset_stackPercent),b.order(\"default\")}}},duration:{get:function(){return t},set:function(a){t=a,v.reset(t),s.duration(t)}}}),a.utils.inheritOptions(b,s),a.utils.initOptions(b),b},a.models.stackedAreaChart=function(){\"use strict\";function b(k){return L.reset(),L.models(e),u&&L.models(f),v&&L.models(g),k.each(function(k){function D(){u&&X.select(\".nv-focus .nv-x.nv-axis\").attr(\"transform\",\"translate(0,\"+T+\")\").transition().duration(I).call(f)}function L(){if(v){if(\"expand\"===e.style()||\"stack_percent\"===e.style()){var a=g.tickFormat();J&&a===P||(J=a),g.tickFormat(P)}else J&&(g.tickFormat(J),J=null);X.select(\".nv-focus .nv-y.nv-axis\").transition().duration(0).call(g)}}function Q(a){var b=X.select(\".nv-focus .nv-stackedWrap\").datum(k.filter(function(a){return!a.disabled}).map(function(b,c){return{key:b.key,area:b.area,classed:b.classed,values:b.values.filter(function(b,c){return e.x()(b,c)>=a[0]&&e.x()(b,c)<=a[1]}),disableTooltip:b.disableTooltip}}));b.transition().duration(I).call(e),D(),L()}var R=d3.select(this);a.utils.initSVG(R);var S=a.utils.availableWidth(o,R,m),T=a.utils.availableHeight(p,R,m)-(x?l.height():0);if(b.update=function(){R.transition().duration(I).call(b)},b.container=this,B.setter(O(k),b.update).getter(N(k)).update(),B.disabled=k.map(function(a){return!!a.disabled}),!C){var U;C={};for(U in B)B[U]instanceof Array?C[U]=B[U].slice(0):C[U]=B[U]}if(!(k&&k.length&&k.filter(function(a){return a.values.length}).length))return a.utils.noData(b,R),b;R.selectAll(\".nv-noData\").remove(),c=e.xScale(),d=e.yScale();var V=R.selectAll(\"g.nv-wrap.nv-stackedAreaChart\").data([k]),W=V.enter().append(\"g\").attr(\"class\",\"nvd3 nv-wrap nv-stackedAreaChart\").append(\"g\"),X=V.select(\"g\");W.append(\"g\").attr(\"class\",\"nv-legendWrap\"),W.append(\"g\").attr(\"class\",\"nv-controlsWrap\");var Y=W.append(\"g\").attr(\"class\",\"nv-focus\");Y.append(\"g\").attr(\"class\",\"nv-background\").append(\"rect\"),Y.append(\"g\").attr(\"class\",\"nv-x nv-axis\"),Y.append(\"g\").attr(\"class\",\"nv-y nv-axis\"),Y.append(\"g\").attr(\"class\",\
\"nv-stackedWrap\"),Y.append(\"g\").attr(\"class\",\"nv-interactive\");W.append(\"g\").attr(\"class\",\"nv-focusWrap\");if(s){var Z=r&&\"top\"===t?S-F:S;if(h.width(Z),X.select(\".nv-legendWrap\").datum(k).call(h),\"bottom\"===t){var $=(u?12:0)+10;m.bottom=Math.max(h.height()+$,m.bottom),T=a.utils.availableHeight(p,R,m)-(x?l.height():0);var _=T+$;X.select(\".nv-legendWrap\").attr(\"transform\",\"translate(0,\"+_+\")\")}else\"top\"===t&&(n||m.top==h.height()||(m.top=h.height(),T=a.utils.availableHeight(p,R,m)-(x?l.height():0)),X.select(\".nv-legendWrap\").attr(\"transform\",\"translate(\"+(S-Z)+\",\"+-m.top+\")\"))}else X.select(\".nv-legendWrap\").selectAll(\"*\").remove();if(r){var aa=[{key:H.stacked||\"Stacked\",metaKey:\"Stacked\",disabled:\"stack\"!=e.style(),style:\"stack\"},{key:H.stream||\"Stream\",metaKey:\"Stream\",disabled:\"stream\"!=e.style(),style:\"stream\"},{key:H.expanded||\"Expanded\",metaKey:\"Expanded\",disabled:\"expand\"!=e.style(),style:\"expand\"},{key:H.stack_percent||\"Stack %\",metaKey:\"Stack_Percent\",disabled:\"stack_percent\"!=e.style(),style:\"stack_percent\"}];F=G.length/3*260,aa=aa.filter(function(a){return-1!==G.indexOf(a.metaKey)}),i.width(F).color([\"#444\",\"#444\",\"#444\"]),X.select(\".nv-controlsWrap\").datum(aa).call(i);var ba=Math.max(i.height(),s&&\"top\"===t?h.height():0);m.top!=ba&&(m.top=ba,T=a.utils.availableHeight(p,R,m)-(x?l.height():0)),X.select(\".nv-controlsWrap\").attr(\"transform\",\"translate(0,\"+-m.top+\")\")}else X.select(\".nv-controlsWrap\").selectAll(\"*\").remove();V.attr(\"transform\",\"translate(\"+m.left+\",\"+m.top+\")\"),w&&X.select(\".nv-y.nv-axis\").attr(\"transform\",\"translate(\"+S+\",0)\"),y&&(j.width(S).height(T).margin({left:m.left,top:m.top}).svgContainer(R).xScale(c),V.select(\".nv-interactive\").call(j)),X.select(\".nv-focus .nv-background rect\").attr(\"width\",S).attr(\"height\",T),e.width(S).height(T).color(k.map(function(a,b){return a.color||q(a,b)}).filter(function(a,b){return!k[b].disabled}));var ca=X.select(\".nv-focus .nv-stackedWrap\").datum(k.filter(function(a){return!a.disabled}));if(u&&f.scale(c)._ticks(a.utils.calcTicksX(S/100,k)).tickSize(-T,0),v){var \
da;da=\"wiggle\"===e.offset()?0:a.utils.calcTicksY(T/36,k),g.scale(d)._ticks(da).tickSize(-S,0)}if(x){l.width(S),X.select(\".nv-focusWrap\").attr(\"transform\",\"translate(0,\"+(T+m.bottom+l.margin().top)+\")\").datum(k.filter(function(a){return!a.disabled})).call(l);var ea=l.brush.empty()?l.xDomain():l.brush.extent();null!==ea&&Q(ea)}else ca.transition().call(e),D(),L();e.dispatch.on(\"areaClick.toggle\",function(a){1===k.filter(function(a){return!a.disabled}).length?k.forEach(function(a){a.disabled=!1}):k.forEach(function(b,c){b.disabled=c!=a.seriesIndex}),B.disabled=k.map(function(a){return!!a.disabled}),E.stateChange(B),b.update()}),h.dispatch.on(\"stateChange\",function(a){for(var c in a)B[c]=a[c];E.stateChange(B),b.update()}),i.dispatch.on(\"legendClick\",function(a,c){a.disabled&&(aa=aa.map(function(a){return a.disabled=!0,a}),a.disabled=!1,e.style(a.style),B.style=e.style(),E.stateChange(B),b.update())}),j.dispatch.on(\"elementMousemove\",function(c){e.clearHighlights();var d,f,g,h=[],i=0,l=!0;if(k.filter(function(a,b){return a.seriesIndex=b,!a.disabled}).forEach(function(j,k){f=a.interactiveBisect(j.values,c.pointXValue,b.x());var m=j.values[f],n=b.y()(m,f);if(null!=n&&e.highlightPoint(k,f,!0),\"undefined\"!=typeof m){\"undefined\"==typeof d&&(d=m),\"undefined\"==typeof g&&(g=b.xScale()(b.x()(m,f)));var o=\"expand\"==e.style()?m.display.y:b.y()(m,f);h.push({key:j.key,value:o,color:q(j,j.seriesIndex),point:m}),z&&\"expand\"!=e.style()&&null!=o&&(i+=o,l=!1)}}),h.reverse(),h.length>2){var m=b.yScale().invert(c.mouseY),n=null;h.forEach(function(a,b){m=Math.abs(m);var c=Math.abs(a.point.display.y0),d=Math.abs(a.point.display.y);return m>=c&&d+c>=m?void(n=b):void 0}),null!=n&&(h[n].highlight=!0)}z&&\"expand\"!=e.style()&&h.length>=2&&!l&&h.push({key:A,value:i,total:!0});var o=b.x()(d,f),p=j.tooltip.valueFormatter();\"expand\"===e.style()||\"stack_percent\"===e.style()?(K||(K=p),p=d3.format(\".1%\")):K&&(p=K,K=null),j.tooltip.valueFormatter(p).data({value:o,series:h})(),j.renderGuideLine(g)}),j.dispatch.on(\"elementMouseout\",function(a){e.clearHighlights()}),\
l.dispatch.on(\"onBrush\",function(a){Q(a)}),E.on(\"changeState\",function(a){\"undefined\"!=typeof a.disabled&&k.length===a.disabled.length&&(k.forEach(function(b,c){b.disabled=a.disabled[c]}),B.disabled=a.disabled),\"undefined\"!=typeof a.style&&(e.style(a.style),M=a.style),b.update()})}),L.renderEnd(\"stacked Area chart immediate\"),b}var c,d,e=a.models.stackedArea(),f=a.models.axis(),g=a.models.axis(),h=a.models.legend(),i=a.models.legend(),j=a.interactiveGuideline(),k=a.models.tooltip(),l=a.models.focus(a.models.stackedArea()),m={top:10,right:25,bottom:50,left:60},n=null,o=null,p=null,q=a.utils.defaultColor(),r=!0,s=!0,t=\"top\",u=!0,v=!0,w=!1,x=!1,y=!1,z=!0,A=\"TOTAL\",B=a.utils.state(),C=null,D=null,E=d3.dispatch(\"stateChange\",\"changeState\",\"renderEnd\"),F=250,G=[\"Stacked\",\"Stream\",\"Expanded\"],H={},I=250;B.style=e.style(),f.orient(\"bottom\").tickPadding(7),g.orient(w?\"right\":\"left\"),k.headerFormatter(function(a,b){return f.tickFormat()(a,b)}).valueFormatter(function(a,b){return g.tickFormat()(a,b)}),j.tooltip.headerFormatter(function(a,b){return f.tickFormat()(a,b)}).valueFormatter(function(a,b){return null==a?\"N/A\":g.tickFormat()(a,b)});var J=null,K=null;i.updateState(!1);var L=a.utils.renderWatch(E),M=e.style(),N=function(a){return function(){return{active:a.map(function(a){return!a.disabled}),style:e.style()}}},O=function(a){return function(b){void 0!==b.style&&(M=b.style),void 0!==b.active&&a.forEach(function(a,c){a.disabled=!b.active[c]})}},P=d3.format(\"%\");return e.dispatch.on(\"elementMouseover.tooltip\",function(a){a.point.x=e.x()(a.point),a.point.y=e.y()(a.point),k.data(a).hidden(!1)}),e.dispatch.on(\"elementMouseout.tooltip\",function(a){k.hidden(!0)}),b.dispatch=E,b.stacked=e,b.legend=h,b.controls=i,b.xAxis=f,b.x2Axis=l.xAxis,b.yAxis=g,b.y2Axis=l.yAxis,b.interactiveLayer=j,b.tooltip=k,b.focus=l,b.dispatch=E,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{width:{get:function(){return o},set:function(a){o=a}},height:{get:function(){return p},set:function(a){p=a}},showLegend:{get:function(){return s},\
set:function(a){s=a}},legendPosition:{get:function(){return t},set:function(a){t=a}},showXAxis:{get:function(){return u},set:function(a){u=a}},showYAxis:{get:function(){return v},set:function(a){v=a}},defaultState:{get:function(){return C},set:function(a){C=a}},noData:{get:function(){return D},set:function(a){D=a}},showControls:{get:function(){return r},set:function(a){r=a}},controlLabels:{get:function(){return H},set:function(a){H=a}},controlOptions:{get:function(){return G},set:function(a){G=a}},showTotalInTooltip:{get:function(){return z},set:function(a){z=a}},totalLabel:{get:function(){return A},set:function(a){A=a}},focusEnable:{get:function(){return x},set:function(a){x=a}},focusHeight:{get:function(){return l.height()},set:function(a){l.height(a)}},brushExtent:{get:function(){return l.brushExtent()},set:function(a){l.brushExtent(a)}},margin:{get:function(){return m},set:function(a){void 0!==a.top&&(m.top=a.top,n=a.top),m.right=void 0!==a.right?a.right:m.right,m.bottom=void 0!==a.bottom?a.bottom:m.bottom,m.left=void 0!==a.left?a.left:m.left}},focusMargin:{get:function(){return l.margin},set:function(a){l.margin.top=void 0!==a.top?a.top:l.margin.top,l.margin.right=void 0!==a.right?a.right:l.margin.right,l.margin.bottom=void 0!==a.bottom?a.bottom:l.margin.bottom,l.margin.left=void 0!==a.left?a.left:l.margin.left}},duration:{get:function(){return I},set:function(a){I=a,L.reset(I),e.duration(I),f.duration(I),g.duration(I)}},color:{get:function(){return q},set:function(b){q=a.utils.getColor(b),h.color(q),e.color(q),l.color(q)}},x:{get:function(){return e.x()},set:function(a){e.x(a),l.x(a)}},y:{get:function(){return e.y()},set:function(a){e.y(a),l.y(a)}},rightAlignYAxis:{get:function(){return w},set:function(a){w=a,g.orient(w?\"right\":\"left\")}},useInteractiveGuideline:{get:function(){return y},set:function(a){y=!!a,b.interactive(!a),b.useVoronoi(!a),e.scatter.interactive(!a)}}}),a.utils.inheritOptions(b,e),a.utils.initOptions(b),b},a.models.stackedAreaWithFocusChart=function(){return a.models.stackedAreaChart().margin({bottom:30}).focusEnable(!0)},\
a.models.sunburst=function(){\"use strict\";function b(a){var b=c(a);return b>90?180:0}function c(a){var b=Math.max(0,Math.min(2*Math.PI,F(a.x))),c=Math.max(0,Math.min(2*Math.PI,F(a.x+a.dx))),d=(b+c)/2*(180/Math.PI)-90;return d}function d(a){var b=Math.max(0,Math.min(2*Math.PI,F(a.x))),c=Math.max(0,Math.min(2*Math.PI,F(a.x+a.dx)));return(c-b)/(2*Math.PI)}function e(a){var b=Math.max(0,Math.min(2*Math.PI,F(a.x))),c=Math.max(0,Math.min(2*Math.PI,F(a.x+a.dx))),d=c-b;return d>z}function f(a,b){var c=d3.interpolate(F.domain(),[l.x,l.x+l.dx]),d=d3.interpolate(G.domain(),[l.y,1]),e=d3.interpolate(G.range(),[l.y?20:0,o]);return 0===b?function(){return J(a)}:function(b){return F.domain(c(b)),G.domain(d(b)).range(e(b)),J(a)}}function g(a){var b=d3.interpolate({x:a.x0,dx:a.dx0,y:a.y0,dy:a.dy0},a);return function(c){var d=b(c);return a.x0=d.x,a.dx0=d.dx,a.y0=d.y,a.dy0=d.dy,J(d)}}function h(a){var b=B(a);I[b]||(I[b]={});var c=I[b];c.dx=a.dx,c.x=a.x,c.dy=a.dy,c.y=a.y}function i(a){a.forEach(function(a){var b=B(a),c=I[b];c?(a.dx0=c.dx,a.x0=c.x,a.dy0=c.dy,a.y0=c.y):(a.dx0=a.dx,a.x0=a.x,a.dy0=a.dy,a.y0=a.y),h(a)})}function j(a){var d=v.selectAll(\"text\"),g=v.selectAll(\"path\");d.transition().attr(\"opacity\",0),l=a,g.transition().duration(D).attrTween(\"d\",f).each(\"end\",function(d){if(d.x>=a.x&&d.x<a.x+a.dx&&d.depth>=a.depth){var f=d3.select(this.parentNode),g=f.select(\"text\");g.transition().duration(D).text(function(a){return y(a)}).attr(\"opacity\",function(a){return e(a)?1:0}).attr(\"transform\",function(){var e=this.getBBox().width;if(0===d.depth)return\"translate(\"+e/2*-1+\",0)\";if(d.depth===a.depth)return\"translate(\"+(G(d.y)+5)+\",0)\";var f=c(d),g=b(d);return 0===g?\"rotate(\"+f+\")translate(\"+(G(d.y)+5)+\",0)\":\"rotate(\"+f+\")translate(\"+(G(d.y)+e+5)+\",0)rotate(\"+g+\")\"})}})}function k(f){return K.reset(),f.each(function(f){v=d3.select(this),m=a.utils.availableWidth(q,v,p),n=a.utils.availableHeight(r,v,p),o=Math.min(m,n)/2,G.range([0,o]);var h=v.select(\"g.nvd3.nv-wrap.nv-sunburst\");h[0][0]?h.attr(\"transform\",\"translate(\"+(m/2+p.left+p.right)+\",\"+(n/2+p.top+p.bottom)+\")\"):h=v.append(\"g\").attr(\"class\",\
\"nvd3 nv-wrap nv-sunburst nv-chart-\"+u).attr(\"transform\",\"translate(\"+(m/2+p.left+p.right)+\",\"+(n/2+p.top+p.bottom)+\")\"),v.on(\"click\",function(a,b){E.chartClick({data:a,index:b,pos:d3.event,id:u})}),H.value(t[s]||t.count);var k=H.nodes(f[0]).reverse();i(k);var l=h.selectAll(\".arc-container\").data(k,B),z=l.enter().append(\"g\").attr(\"class\",\"arc-container\");z.append(\"path\").attr(\"d\",J).style(\"fill\",function(a){return a.color?a.color:w(C?(a.children?a:a.parent).name:a.name)}).style(\"stroke\",\"#FFF\").on(\"click\",function(a,b){j(a),E.elementClick({data:a,index:b})}).on(\"mouseover\",function(a,b){d3.select(this).classed(\"hover\",!0).style(\"opacity\",.8),E.elementMouseover({data:a,color:d3.select(this).style(\"fill\"),percent:d(a)})}).on(\"mouseout\",function(a,b){d3.select(this).classed(\"hover\",!1).style(\"opacity\",1),E.elementMouseout({data:a})}).on(\"mousemove\",function(a,b){E.elementMousemove({data:a})}),l.each(function(a){d3.select(this).select(\"path\").transition().duration(D).attrTween(\"d\",g)}),x&&(l.selectAll(\"text\").remove(),l.append(\"text\").text(function(a){return y(a)}).transition().duration(D).attr(\"opacity\",function(a){return e(a)?1:0}).attr(\"transform\",function(a){var d=this.getBBox().width;if(0===a.depth)return\"rotate(0)translate(\"+d/2*-1+\",0)\";var e=c(a),f=b(a);return 0===f?\"rotate(\"+e+\")translate(\"+(G(a.y)+5)+\",0)\":\"rotate(\"+e+\")translate(\"+(G(a.y)+d+5)+\",0)rotate(\"+f+\")\"})),j(k[k.length-1]),l.exit().transition().duration(D).attr(\"opacity\",0).each(\"end\",function(a){var b=B(a);I[b]=void 0}).remove()}),K.renderEnd(\"sunburst immediate\"),k}var l,m,n,o,p={top:0,right:0,bottom:0,left:0},q=600,r=600,s=\"count\",t={count:function(a){return 1},value:function(a){return a.value||a.size},size:function(a){return a.value||a.size}},u=Math.floor(1e4*Math.random()),v=null,w=a.utils.defaultColor(),x=!1,y=function(a){return\"count\"===s?a.name+\" #\"+a.value:a.name+\" \"+(a.value||a.size)},z=.02,A=function(a,b){return a.name>b.name},B=function(a,b){return a.name},C=!0,D=500,E=d3.dispatch(\"chartClick\",\"elementClick\",\"elementDblClick\",\"elementMousemove\",\
\"elementMouseover\",\"elementMouseout\",\"renderEnd\"),F=d3.scale.linear().range([0,2*Math.PI]),G=d3.scale.sqrt(),H=d3.layout.partition().sort(A),I={},J=d3.svg.arc().startAngle(function(a){return Math.max(0,Math.min(2*Math.PI,F(a.x)))}).endAngle(function(a){return Math.max(0,Math.min(2*Math.PI,F(a.x+a.dx)))}).innerRadius(function(a){return Math.max(0,G(a.y))}).outerRadius(function(a){return Math.max(0,G(a.y+a.dy))}),K=a.utils.renderWatch(E);return k.dispatch=E,k.options=a.utils.optionsFunc.bind(k),k._options=Object.create({},{width:{get:function(){return q},set:function(a){q=a}},height:{get:function(){return r},set:function(a){r=a}},mode:{get:function(){return s},set:function(a){s=a}},id:{get:function(){return u},set:function(a){u=a}},duration:{get:function(){return D},set:function(a){D=a}},groupColorByParent:{get:function(){return C},set:function(a){C=!!a}},showLabels:{get:function(){return x},set:function(a){x=!!a}},labelFormat:{get:function(){return y},set:function(a){y=a}},labelThreshold:{get:function(){return z},set:function(a){z=a}},sort:{get:function(){return A},set:function(a){A=a}},key:{get:function(){return B},set:function(a){B=a}},margin:{get:function(){return p},set:function(a){p.top=void 0!=a.top?a.top:p.top,p.right=void 0!=a.right?a.right:p.right,p.bottom=void 0!=a.bottom?a.bottom:p.bottom,p.left=void 0!=a.left?a.left:p.left}},color:{get:function(){return w},set:function(b){w=a.utils.getColor(b)}}}),a.utils.initOptions(k),k},a.models.sunburstChart=function(){\"use strict\";function b(d){return n.reset(),n.models(c),d.each(function(d){var h=d3.select(this);a.utils.initSVG(h);var i=a.utils.availableWidth(f,h,e),j=a.utils.availableHeight(g,h,e);return b.update=function(){0===l?h.call(b):h.transition().duration(l).call(b)},b.container=h,d&&d.length?(h.selectAll(\".nv-noData\").remove(),c.width(i).height(j).margin(e),void h.call(c)):(a.utils.noData(b,h),b)}),n.renderEnd(\"sunburstChart immediate\"),b}var c=a.models.sunburst(),d=a.models.tooltip(),e={top:30,right:20,bottom:20,left:20},f=null,g=null,h=a.utils.defaultColor(),\
i=!1,j=(Math.round(1e5*Math.random()),null),k=null,l=250,m=d3.dispatch(\"stateChange\",\"changeState\",\"renderEnd\"),n=a.utils.renderWatch(m);return d.duration(0).headerEnabled(!1).valueFormatter(function(a){return a}),c.dispatch.on(\"elementMouseover.tooltip\",function(a){a.series={key:a.data.name,value:a.data.value||a.data.size,color:a.color,percent:a.percent},i||(delete a.percent,delete a.series.percent),d.data(a).hidden(!1)}),c.dispatch.on(\"elementMouseout.tooltip\",function(a){d.hidden(!0)}),c.dispatch.on(\"elementMousemove.tooltip\",function(a){d()}),b.dispatch=m,b.sunburst=c,b.tooltip=d,b.options=a.utils.optionsFunc.bind(b),b._options=Object.create({},{noData:{get:function(){return k},set:function(a){k=a}},defaultState:{get:function(){return j},set:function(a){j=a}},showTooltipPercent:{get:function(){return i},set:function(a){i=a}},color:{get:function(){return h},set:function(a){h=a,c.color(h)}},duration:{get:function(){return l},set:function(a){l=a,n.reset(l),c.duration(l)}},margin:{get:function(){return e},set:function(a){e.top=void 0!==a.top?a.top:e.top,e.right=void 0!==a.right?a.right:e.right,e.bottom=void 0!==a.bottom?a.bottom:e.bottom,e.left=void 0!==a.left?a.left:e.left,c.margin(e)}}}),a.utils.inheritOptions(b,c),a.utils.initOptions(b),b},a.version=\"1.8.5\"}();\
")
################################################################################
# end nvd3.js
################################################################################

################################################################################
# d3.js
#
# The following is a copy of the d3 3.5.17 js file. Note that we write it
# to the file in multiple steps, because CMake exhausts its stack memory when
# parsing too long strings.
# https://github.com/mbostock/d3
################################################################################
file(WRITE "${METABENCH_DIR}/d3.js" "\
!function(){function n(n){return n&&(n.ownerDocument||n.document||n).documentElement}function t(n){return n&&(n.ownerDocument&&n.ownerDocument.defaultView||n.document&&n||n.defaultView)}function e(n,t){return t>n?-1:n>t?1:n>=t?0:NaN}function r(n){return null===n?NaN:+n}function i(n){return!isNaN(n)}function u(n){return{left:function(t,e,r,i){for(arguments.length<3&&(r=0),arguments.length<4&&(i=t.length);i>r;){var u=r+i>>>1;n(t[u],e)<0?r=u+1:i=u}return r},right:function(t,e,r,i){for(arguments.length<3&&(r=0),arguments.length<4&&(i=t.length);i>r;){var u=r+i>>>1;n(t[u],e)>0?i=u:r=u+1}return r}}}function o(n){return n.length}function a(n){for(var t=1;n*t%1;)t*=10;return t}function l(n,t){for(var e in t)Object.defineProperty(n.prototype,e,{value:t[e],enumerable:!1})}function c(){this._=Object.create(null)}function f(n){return(n+=\"\")===bo||n[0]===_o?_o+n:n}function s(n){return(n+=\"\")[0]===_o?n.slice(1):n}function h(n){return f(n)in this._}function p(n){return(n=f(n))in this._&&delete this._[n]}function g(){var n=[];for(var t in this._)n.push(s(t));return n}function v(){var n=0;for(var t in this._)++n;return n}function d(){for(var n in this._)return!1;return!0}function y(){this._=Object.create(null)}function m(n){return n}function M(n,t,e){return function(){var r=e.apply(t,arguments);return r===t?n:r}}function x(n,t){if(t in n)return t;t=t.charAt(0).toUpperCase()+t.slice(1);for(var e=0,r=wo.length;r>e;++e){var i=wo[e]+t;if(i in n)return i}}function b(){}function _(){}function w(n){function t(){for(var t,r=e,i=-1,u=r.length;++i<u;)(t=r[i].on)&&t.apply(this,arguments);return n}var e=[],r=new c;return t.on=function(t,i){var u,o=r.get(t);return arguments.length<2?o&&o.on:(o&&(o.on=null,e=e.slice(0,u=e.indexOf(o)).concat(e.slice(u+1)),r.remove(t)),i&&e.push(r.set(t,{on:i})),n)},t}function S(){ao.event.preventDefault()}function k(){for(var n,t=ao.event;n=t.sourceEvent;)t=n;return t}function N(n){for(var t=new _,e=0,r=arguments.length;++e<r;)t[arguments[e]]=w(t);return t.of=function(e,r){return function(i){try{var u=i.sourceEvent=ao.event;\
i.target=n,ao.event=i,t[i.type].apply(e,r)}finally{ao.event=u}}},t}function E(n){return ko(n,Co),n}function A(n){return\"function\"==typeof n?n:function(){return No(n,this)}}function C(n){return\"function\"==typeof n?n:function(){return Eo(n,this)}}function z(n,t){function e(){this.removeAttribute(n)}function r(){this.removeAttributeNS(n.space,n.local)}function i(){this.setAttribute(n,t)}function u(){this.setAttributeNS(n.space,n.local,t)}function o(){var e=t.apply(this,arguments);null==e?this.removeAttribute(n):this.setAttribute(n,e)}function a(){var e=t.apply(this,arguments);null==e?this.removeAttributeNS(n.space,n.local):this.setAttributeNS(n.space,n.local,e)}return n=ao.ns.qualify(n),null==t?n.local?r:e:\"function\"==typeof t?n.local?a:o:n.local?u:i}function L(n){return n.trim().replace(/\\s+/g,\" \")}function q(n){return new RegExp(\"(?:^|\\\\s+)\"+ao.requote(n)+\"(?:\\\\s+|$)\",\"g\")}function T(n){return(n+\"\").trim().split(/^|\\s+/)}function R(n,t){function e(){for(var e=-1;++e<i;)n[e](this,t)}function r(){for(var e=-1,r=t.apply(this,arguments);++e<i;)n[e](this,r)}n=T(n).map(D);var i=n.length;return\"function\"==typeof t?r:e}function D(n){var t=q(n);return function(e,r){if(i=e.classList)return r?i.add(n):i.remove(n);var i=e.getAttribute(\"class\")||\"\";r?(t.lastIndex=0,t.test(i)||e.setAttribute(\"class\",L(i+\" \"+n))):e.setAttribute(\"class\",L(i.replace(t,\" \")))}}function P(n,t,e){function r(){this.style.removeProperty(n)}function i(){this.style.setProperty(n,t,e)}function u(){var r=t.apply(this,arguments);null==r?this.style.removeProperty(n):this.style.setProperty(n,r,e)}return null==t?r:\"function\"==typeof t?u:i}function U(n,t){function e(){delete this[n]}function r(){this[n]=t}function i(){var e=t.apply(this,arguments);null==e?delete this[n]:this[n]=e}return null==t?e:\"function\"==typeof t?i:r}function j(n){function t(){var t=this.ownerDocument,e=this.namespaceURI;return e===zo&&t.documentElement.namespaceURI===zo?t.createElement(n):t.createElementNS(e,n)}function e(){return this.ownerDocument.createElementNS(n.space,n.local)}return\"function\"==typeof \
n?n:(n=ao.ns.qualify(n)).local?e:t}function F(){var n=this.parentNode;n&&n.removeChild(this)}function H(n){return{__data__:n}}function O(n){return function(){return Ao(this,n)}}function I(n){return arguments.length||(n=e),function(t,e){return t&&e?n(t.__data__,e.__data__):!t-!e}}function Y(n,t){for(var e=0,r=n.length;r>e;e++)for(var i,u=n[e],o=0,a=u.length;a>o;o++)(i=u[o])&&t(i,o,e);return n}function Z(n){return ko(n,qo),n}function V(n){var t,e;return function(r,i,u){var o,a=n[u].update,l=a.length;for(u!=e&&(e=u,t=0),i>=t&&(t=i+1);!(o=a[t])&&++t<l;);return o}}function X(n,t,e){function r(){var t=this[o];t&&(this.removeEventListener(n,t,t.$),delete this[o])}function i(){var i=l(t,co(arguments));r.call(this),this.addEventListener(n,this[o]=i,i.$=e),i._=t}function u(){var t,e=new RegExp(\"^__on([^.]+)\"+ao.requote(n)+\"$\");for(var r in this)if(t=r.match(e)){var i=this[r];this.removeEventListener(t[1],i,i.$),delete this[r]}}var o=\"__on\"+n,a=n.indexOf(\".\"),l=$;a>0&&(n=n.slice(0,a));var c=To.get(n);return c&&(n=c,l=B),a?t?i:r:t?b:u}function $(n,t){return function(e){var r=ao.event;ao.event=e,t[0]=this.__data__;try{n.apply(this,t)}finally{ao.event=r}}}function B(n,t){var e=$(n,t);return function(n){var t=this,r=n.relatedTarget;r&&(r===t||8&r.compareDocumentPosition(t))||e.call(t,n)}}function W(e){var r=\".dragsuppress-\"+ ++Do,i=\"click\"+r,u=ao.select(t(e)).on(\"touchmove\"+r,S).on(\"dragstart\"+r,S).on(\"selectstart\"+r,S);if(null==Ro&&(Ro=\"onselectstart\"in e?!1:x(e.style,\"userSelect\")),Ro){var o=n(e).style,a=o[Ro];o[Ro]=\"none\"}return function(n){if(u.on(r,null),Ro&&(o[Ro]=a),n){var t=function(){u.on(i,null)};u.on(i,function(){S(),t()},!0),setTimeout(t,0)}}}function J(n,e){e.changedTouches&&(e=e.changedTouches[0]);var r=n.ownerSVGElement||n;if(r.createSVGPoint){var i=r.createSVGPoint();if(0>Po){var u=t(n);if(u.scrollX||u.scrollY){r=ao.select(\"body\").append(\"svg\").style({position:\"absolute\",top:0,left:0,margin:0,padding:0,border:\"none\"},\"important\");var o=r[0][0].getScreenCTM();Po=!(o.f||o.e),r.remove()}}return Po?(i.x=e.pageX,i.y=e.pageY):(i.x=e.clientX,\
i.y=e.clientY),i=i.matrixTransform(n.getScreenCTM().inverse()),[i.x,i.y]}var a=n.getBoundingClientRect();return[e.clientX-a.left-n.clientLeft,e.clientY-a.top-n.clientTop]}function G(){return ao.event.changedTouches[0].identifier}function K(n){return n>0?1:0>n?-1:0}function Q(n,t,e){return(t[0]-n[0])*(e[1]-n[1])-(t[1]-n[1])*(e[0]-n[0])}function nn(n){return n>1?0:-1>n?Fo:Math.acos(n)}function tn(n){return n>1?Io:-1>n?-Io:Math.asin(n)}function en(n){return((n=Math.exp(n))-1/n)/2}function rn(n){return((n=Math.exp(n))+1/n)/2}function un(n){return((n=Math.exp(2*n))-1)/(n+1)}function on(n){return(n=Math.sin(n/2))*n}function an(){}function ln(n,t,e){return this instanceof ln?(this.h=+n,this.s=+t,void(this.l=+e)):arguments.length<2?n instanceof ln?new ln(n.h,n.s,n.l):_n(\"\"+n,wn,ln):new ln(n,t,e)}function cn(n,t,e){function r(n){return n>360?n-=360:0>n&&(n+=360),60>n?u+(o-u)*n/60:180>n?o:240>n?u+(o-u)*(240-n)/60:u}function i(n){return Math.round(255*r(n))}var u,o;return n=isNaN(n)?0:(n%=360)<0?n+360:n,t=isNaN(t)?0:0>t?0:t>1?1:t,e=0>e?0:e>1?1:e,o=.5>=e?e*(1+t):e+t-e*t,u=2*e-o,new mn(i(n+120),i(n),i(n-120))}function fn(n,t,e){return this instanceof fn?(this.h=+n,this.c=+t,void(this.l=+e)):arguments.length<2?n instanceof fn?new fn(n.h,n.c,n.l):n instanceof hn?gn(n.l,n.a,n.b):gn((n=Sn((n=ao.rgb(n)).r,n.g,n.b)).l,n.a,n.b):new fn(n,t,e)}function sn(n,t,e){return isNaN(n)&&(n=0),isNaN(t)&&(t=0),new hn(e,Math.cos(n*=Yo)*t,Math.sin(n)*t)}function hn(n,t,e){return this instanceof hn?(this.l=+n,this.a=+t,void(this.b=+e)):arguments.length<2?n instanceof hn?new hn(n.l,n.a,n.b):n instanceof fn?sn(n.h,n.c,n.l):Sn((n=mn(n)).r,n.g,n.b):new hn(n,t,e)}function pn(n,t,e){var r=(n+16)/116,i=r+t/500,u=r-e/200;return i=vn(i)*na,r=vn(r)*ta,u=vn(u)*ea,new mn(yn(3.2404542*i-1.5371385*r-.4985314*u),yn(-.969266*i+1.8760108*r+.041556*u),yn(.0556434*i-.2040259*r+1.0572252*u))}function gn(n,t,e){return n>0?new fn(Math.atan2(e,t)*Zo,Math.sqrt(t*t+e*e),n):new fn(NaN,NaN,n)}function vn(n){return n>.206893034?n*n*n:(n-4/29)/7.787037}function dn(n){return n>.008856?Math.pow(n,\
1/3):7.787037*n+4/29}function yn(n){return Math.round(255*(.00304>=n?12.92*n:1.055*Math.pow(n,1/2.4)-.055))}function mn(n,t,e){return this instanceof mn?(this.r=~~n,this.g=~~t,void(this.b=~~e)):arguments.length<2?n instanceof mn?new mn(n.r,n.g,n.b):_n(\"\"+n,mn,cn):new mn(n,t,e)}function Mn(n){return new mn(n>>16,n>>8&255,255&n)}function xn(n){return Mn(n)+\"\"}function bn(n){return 16>n?\"0\"+Math.max(0,n).toString(16):Math.min(255,n).toString(16)}function _n(n,t,e){var r,i,u,o=0,a=0,l=0;if(r=/([a-z]+)\\((.*)\\)/.exec(n=n.toLowerCase()))switch(i=r[2].split(\",\"),r[1]){case\"hsl\":return e(parseFloat(i[0]),parseFloat(i[1])/100,parseFloat(i[2])/100);case\"rgb\":return t(Nn(i[0]),Nn(i[1]),Nn(i[2]))}return(u=ua.get(n))?t(u.r,u.g,u.b):(null==n||\"#\"!==n.charAt(0)||isNaN(u=parseInt(n.slice(1),16))||(4===n.length?(o=(3840&u)>>4,o=o>>4|o,a=240&u,a=a>>4|a,l=15&u,l=l<<4|l):7===n.length&&(o=(16711680&u)>>16,a=(65280&u)>>8,l=255&u)),t(o,a,l))}function wn(n,t,e){var r,i,u=Math.min(n/=255,t/=255,e/=255),o=Math.max(n,t,e),a=o-u,l=(o+u)/2;return a?(i=.5>l?a/(o+u):a/(2-o-u),r=n==o?(t-e)/a+(e>t?6:0):t==o?(e-n)/a+2:(n-t)/a+4,r*=60):(r=NaN,i=l>0&&1>l?0:r),new ln(r,i,l)}function Sn(n,t,e){n=kn(n),t=kn(t),e=kn(e);var r=dn((.4124564*n+.3575761*t+.1804375*e)/na),i=dn((.2126729*n+.7151522*t+.072175*e)/ta),u=dn((.0193339*n+.119192*t+.9503041*e)/ea);return hn(116*i-16,500*(r-i),200*(i-u))}function kn(n){return(n/=255)<=.04045?n/12.92:Math.pow((n+.055)/1.055,2.4)}function Nn(n){var t=parseFloat(n);return\"%\"===n.charAt(n.length-1)?Math.round(2.55*t):t}function En(n){return\"function\"==typeof n?n:function(){return n}}function An(n){return function(t,e,r){return 2===arguments.length&&\"function\"==typeof e&&(r=e,e=null),Cn(t,e,n,r)}}function Cn(n,t,e,r){function i(){var n,t=l.status;if(!t&&Ln(l)||t>=200&&300>t||304===t){try{n=e.call(u,l)}catch(r){return void o.error.call(u,r)}o.load.call(u,n)}else o.error.call(u,l)}var u={},o=ao.dispatch(\"beforesend\",\"progress\",\"load\",\"error\"),a={},l=new XMLHttpRequest,c=null;return!this.XDomainRequest||\"withCredentials\"in l||!/^(http(s)?:)?\\/\\//.test(n)||(l=new \
XDomainRequest),\"onload\"in l?l.onload=l.onerror=i:l.onreadystatechange=function(){l.readyState>3&&i()},l.onprogress=function(n){var t=ao.event;ao.event=n;try{o.progress.call(u,l)}finally{ao.event=t}},u.header=function(n,t){return n=(n+\"\").toLowerCase(),arguments.length<2?a[n]:(null==t?delete a[n]:a[n]=t+\"\",u)},u.mimeType=function(n){return arguments.length?(t=null==n?null:n+\"\",u):t},u.responseType=function(n){return arguments.length?(c=n,u):c},u.response=function(n){return e=n,u},[\"get\",\"post\"].forEach(function(n){u[n]=function(){return u.send.apply(u,[n].concat(co(arguments)))}}),u.send=function(e,r,i){if(2===arguments.length&&\"function\"==typeof r&&(i=r,r=null),l.open(e,n,!0),null==t||\"accept\"in a||(a.accept=t+\",*/*\"),l.setRequestHeader)for(var f in a)l.setRequestHeader(f,a[f]);return null!=t&&l.overrideMimeType&&l.overrideMimeType(t),null!=c&&(l.responseType=c),null!=i&&u.on(\"error\",i).on(\"load\",function(n){i(null,n)}),o.beforesend.call(u,l),l.send(null==r?null:r),u},u.abort=function(){return l.abort(),u},ao.rebind(u,o,\"on\"),null==r?u:u.get(zn(r))}function zn(n){return 1===n.length?function(t,e){n(null==t?e:null)}:n}function Ln(n){var t=n.responseType;return t&&\"text\"!==t?n.response:n.responseText}function qn(n,t,e){var r=arguments.length;2>r&&(t=0),3>r&&(e=Date.now());var i=e+t,u={c:n,t:i,n:null};return aa?aa.n=u:oa=u,aa=u,la||(ca=clearTimeout(ca),la=1,fa(Tn)),u}function Tn(){var n=Rn(),t=Dn()-n;t>24?(isFinite(t)&&(clearTimeout(ca),ca=setTimeout(Tn,t)),la=0):(la=1,fa(Tn))}function Rn(){for(var n=Date.now(),t=oa;t;)n>=t.t&&t.c(n-t.t)&&(t.c=null),t=t.n;return n}function Dn(){for(var n,t=oa,e=1/0;t;)t.c?(t.t<e&&(e=t.t),t=(n=t).n):t=n?n.n=t.n:oa=t.n;return aa=n,e}function Pn(n,t){return t-(n?Math.ceil(Math.log(n)/Math.LN10):1)}function Un(n,t){var e=Math.pow(10,3*xo(8-t));return{scale:t>8?function(n){return n/e}:function(n){return n*e},symbol:n}}function jn(n){var t=n.decimal,e=n.thousands,r=n.grouping,i=n.currency,u=r&&e?function(n,t){for(var i=n.length,u=[],o=0,a=r[0],l=0;i>0&&a>0&&(l+a+1>t&&(a=Math.max(1,t-l)),u.push(n.substring(i-=a,\
i+a)),!((l+=a+1)>t));)a=r[o=(o+1)%r.length];return u.reverse().join(e)}:m;return function(n){var e=ha.exec(n),r=e[1]||\" \",o=e[2]||\">\",a=e[3]||\"-\",l=e[4]||\"\",c=e[5],f=+e[6],s=e[7],h=e[8],p=e[9],g=1,v=\"\",d=\"\",y=!1,m=!0;switch(h&&(h=+h.substring(1)),(c||\"0\"===r&&\"=\"===o)&&(c=r=\"0\",o=\"=\"),p){case\"n\":s=!0,p=\"g\";break;case\"%\":g=100,d=\"%\",p=\"f\";break;case\"p\":g=100,d=\"%\",p=\"r\";break;case\"b\":case\"o\":case\"x\":case\"X\":\"#\"===l&&(v=\"0\"+p.toLowerCase());case\"c\":m=!1;case\"d\":y=!0,h=0;break;case\"s\":g=-1,p=\"r\"}\"$\"===l&&(v=i[0],d=i[1]),\"r\"!=p||h||(p=\"g\"),null!=h&&(\"g\"==p?h=Math.max(1,Math.min(21,h)):\"e\"!=p&&\"f\"!=p||(h=Math.max(0,Math.min(20,h)))),p=pa.get(p)||Fn;var M=c&&s;return function(n){var e=d;if(y&&n%1)return\"\";var i=0>n||0===n&&0>1/n?(n=-n,\"-\"):\"-\"===a?\"\":a;if(0>g){var l=ao.formatPrefix(n,h);n=l.scale(n),e=l.symbol+d}else n*=g;n=p(n,h);var x,b,_=n.lastIndexOf(\".\");if(0>_){var w=m?n.lastIndexOf(\"e\"):-1;0>w?(x=n,b=\"\"):(x=n.substring(0,w),b=n.substring(w))}else x=n.substring(0,_),b=t+n.substring(_+1);!c&&s&&(x=u(x,1/0));var S=v.length+x.length+b.length+(M?0:i.length),k=f>S?new Array(S=f-S+1).join(r):\"\";return M&&(x=u(k+x,k.length?f-b.length:1/0)),i+=v,n=x+b,(\"<\"===o?i+n+k:\">\"===o?k+i+n:\"^\"===o?k.substring(0,S>>=1)+i+n+k.substring(S):i+(M?n:k+n))+e}}}function Fn(n){return n+\"\"}function Hn(){this._=new Date(arguments.length>1?Date.UTC.apply(this,arguments):arguments[0])}function On(n,t,e){function r(t){var e=n(t),r=u(e,1);return r-t>t-e?e:r}function i(e){return t(e=n(new va(e-1)),1),e}function u(n,e){return t(n=new va(+n),e),n}function o(n,r,u){var o=i(n),a=[];if(u>1)for(;r>o;)e(o)%u||a.push(new Date(+o)),t(o,1);else for(;r>o;)a.push(new Date(+o)),t(o,1);return a}function a(n,t,e){try{va=Hn;var r=new Hn;return r._=n,o(r,t,e)}finally{va=Date}}n.floor=n,n.round=r,n.ceil=i,n.offset=u,n.range=o;var l=n.utc=In(n);return l.floor=l,l.round=In(r),l.ceil=In(i),l.offset=In(u),l.range=a,n}function In(n){return function(t,e){try{va=Hn;var r=new Hn;return r._=t,n(r,e)._}finally{va=Date}}}function Yn(n){function t(n){function t(t){for(var e,i,u,\
o=[],a=-1,l=0;++a<r;)37===n.charCodeAt(a)&&(o.push(n.slice(l,a)),null!=(i=ya[e=n.charAt(++a)])&&(e=n.charAt(++a)),(u=A[e])&&(e=u(t,null==i?\"e\"===e?\" \":\"0\":i)),o.push(e),l=a+1);return o.push(n.slice(l,a)),o.join(\"\")}var r=n.length;return t.parse=function(t){var r={y:1900,m:0,d:1,H:0,M:0,S:0,L:0,Z:null},i=e(r,n,t,0);if(i!=t.length)return null;\"p\"in r&&(r.H=r.H%12+12*r.p);var u=null!=r.Z&&va!==Hn,o=new(u?Hn:va);return\"j\"in r?o.setFullYear(r.y,0,r.j):\"W\"in r||\"U\"in r?(\"w\"in r||(r.w=\"W\"in r?1:0),o.setFullYear(r.y,0,1),o.setFullYear(r.y,0,\"W\"in r?(r.w+6)%7+7*r.W-(o.getDay()+5)%7:r.w+7*r.U-(o.getDay()+6)%7)):o.setFullYear(r.y,r.m,r.d),o.setHours(r.H+(r.Z/100|0),r.M+r.Z%100,r.S,r.L),u?o._:o},t.toString=function(){return n},t}function e(n,t,e,r){for(var i,u,o,a=0,l=t.length,c=e.length;l>a;){if(r>=c)return-1;if(i=t.charCodeAt(a++),37===i){if(o=t.charAt(a++),u=C[o in ya?t.charAt(a++):o],!u||(r=u(n,e,r))<0)return-1}else if(i!=e.charCodeAt(r++))return-1}return r}function r(n,t,e){_.lastIndex=0;var r=_.exec(t.slice(e));return r?(n.w=w.get(r[0].toLowerCase()),e+r[0].length):-1}function i(n,t,e){x.lastIndex=0;var r=x.exec(t.slice(e));return r?(n.w=b.get(r[0].toLowerCase()),e+r[0].length):-1}function u(n,t,e){N.lastIndex=0;var r=N.exec(t.slice(e));return r?(n.m=E.get(r[0].toLowerCase()),e+r[0].length):-1}function o(n,t,e){S.lastIndex=0;var r=S.exec(t.slice(e));return r?(n.m=k.get(r[0].toLowerCase()),e+r[0].length):-1}function a(n,t,r){return e(n,A.c.toString(),t,r)}function l(n,t,r){return e(n,A.x.toString(),t,r)}function c(n,t,r){return e(n,A.X.toString(),t,r)}function f(n,t,e){var r=M.get(t.slice(e,e+=2).toLowerCase());return null==r?-1:(n.p=r,e)}var s=n.dateTime,h=n.date,p=n.time,g=n.periods,v=n.days,d=n.shortDays,y=n.months,m=n.shortMonths;t.utc=function(n){function e(n){try{va=Hn;var t=new va;return t._=n,r(t)}finally{va=Date}}var r=t(n);return e.parse=function(n){try{va=Hn;var t=r.parse(n);return t&&t._}finally{va=Date}},e.toString=r.toString,e},t.multi=t.utc.multi=ct;var M=ao.map(),x=Vn(v),b=Xn(v),_=Vn(d),w=Xn(d),S=Vn(y),k=Xn(y),\
N=Vn(m),E=Xn(m);g.forEach(function(n,t){M.set(n.toLowerCase(),t)});var A={a:function(n){return d[n.getDay()]},A:function(n){return v[n.getDay()]},b:function(n){return m[n.getMonth()]},B:function(n){return y[n.getMonth()]},c:t(s),d:function(n,t){return Zn(n.getDate(),t,2)},e:function(n,t){return Zn(n.getDate(),t,2)},H:function(n,t){return Zn(n.getHours(),t,2)},I:function(n,t){return Zn(n.getHours()%12||12,t,2)},j:function(n,t){return Zn(1+ga.dayOfYear(n),t,3)},L:function(n,t){return Zn(n.getMilliseconds(),t,3)},m:function(n,t){return Zn(n.getMonth()+1,t,2)},M:function(n,t){return Zn(n.getMinutes(),t,2)},p:function(n){return g[+(n.getHours()>=12)]},S:function(n,t){return Zn(n.getSeconds(),t,2)},U:function(n,t){return Zn(ga.sundayOfYear(n),t,2)},w:function(n){return n.getDay()},W:function(n,t){return Zn(ga.mondayOfYear(n),t,2)},x:t(h),X:t(p),y:function(n,t){return Zn(n.getFullYear()%100,t,2)},Y:function(n,t){return Zn(n.getFullYear()%1e4,t,4)},Z:at,\"%\":function(){return\"%\"}},C={a:r,A:i,b:u,B:o,c:a,d:tt,e:tt,H:rt,I:rt,j:et,L:ot,m:nt,M:it,p:f,S:ut,U:Bn,w:$n,W:Wn,x:l,X:c,y:Gn,Y:Jn,Z:Kn,\"%\":lt};return t}function Zn(n,t,e){var r=0>n?\"-\":\"\",i=(r?-n:n)+\"\",u=i.length;return r+(e>u?new Array(e-u+1).join(t)+i:i)}function Vn(n){return new RegExp(\"^(?:\"+n.map(ao.requote).join(\"|\")+\")\",\"i\")}function Xn(n){for(var t=new c,e=-1,r=n.length;++e<r;)t.set(n[e].toLowerCase(),e);return t}function $n(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e,e+1));return r?(n.w=+r[0],e+r[0].length):-1}function Bn(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e));return r?(n.U=+r[0],e+r[0].length):-1}function Wn(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e));return r?(n.W=+r[0],e+r[0].length):-1}function Jn(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e,e+4));return r?(n.y=+r[0],e+r[0].length):-1}function Gn(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e,e+2));return r?(n.y=Qn(+r[0]),e+r[0].length):-1}function Kn(n,t,e){return/^[+-]\\d{4}$/.test(t=t.slice(e,e+5))?(n.Z=-t,e+5):-1}function Qn(n){return n+(n>68?1900:2e3)}function nt(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e,\
e+2));return r?(n.m=r[0]-1,e+r[0].length):-1}function tt(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e,e+2));return r?(n.d=+r[0],e+r[0].length):-1}function et(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e,e+3));return r?(n.j=+r[0],e+r[0].length):-1}function rt(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e,e+2));return r?(n.H=+r[0],e+r[0].length):-1}function it(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e,e+2));return r?(n.M=+r[0],e+r[0].length):-1}function ut(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e,e+2));return r?(n.S=+r[0],e+r[0].length):-1}function ot(n,t,e){ma.lastIndex=0;var r=ma.exec(t.slice(e,e+3));return r?(n.L=+r[0],e+r[0].length):-1}function at(n){var t=n.getTimezoneOffset(),e=t>0?\"-\":\"+\",r=xo(t)/60|0,i=xo(t)%60;return e+Zn(r,\"0\",2)+Zn(i,\"0\",2)}function lt(n,t,e){Ma.lastIndex=0;var r=Ma.exec(t.slice(e,e+1));return r?e+r[0].length:-1}function ct(n){for(var t=n.length,e=-1;++e<t;)n[e][0]=this(n[e][0]);return function(t){for(var e=0,r=n[e];!r[1](t);)r=n[++e];return r[0](t)}}function ft(){}function st(n,t,e){var r=e.s=n+t,i=r-n,u=r-i;e.t=n-u+(t-i)}function ht(n,t){n&&wa.hasOwnProperty(n.type)&&wa[n.type](n,t)}function pt(n,t,e){var r,i=-1,u=n.length-e;for(t.lineStart();++i<u;)r=n[i],t.point(r[0],r[1],r[2]);t.lineEnd()}function gt(n,t){var e=-1,r=n.length;for(t.polygonStart();++e<r;)pt(n[e],t,1);t.polygonEnd()}function vt(){function n(n,t){n*=Yo,t=t*Yo/2+Fo/4;var e=n-r,o=e>=0?1:-1,a=o*e,l=Math.cos(t),c=Math.sin(t),f=u*c,s=i*l+f*Math.cos(a),h=f*o*Math.sin(a);ka.add(Math.atan2(h,s)),r=n,i=l,u=c}var t,e,r,i,u;Na.point=function(o,a){Na.point=n,r=(t=o)*Yo,i=Math.cos(a=(e=a)*Yo/2+Fo/4),u=Math.sin(a)},Na.lineEnd=function(){n(t,e)}}function dt(n){var t=n[0],e=n[1],r=Math.cos(e);return[r*Math.cos(t),r*Math.sin(t),Math.sin(e)]}function yt(n,t){return n[0]*t[0]+n[1]*t[1]+n[2]*t[2]}function mt(n,t){return[n[1]*t[2]-n[2]*t[1],n[2]*t[0]-n[0]*t[2],n[0]*t[1]-n[1]*t[0]]}function Mt(n,t){n[0]+=t[0],n[1]+=t[1],n[2]+=t[2]}function xt(n,t){return[n[0]*t,n[1]*t,n[2]*t]}function bt(n){var t=Math.sqrt(n[0]*n[0]+n[1]*n[1]+n[2]*n[2]);\
n[0]/=t,n[1]/=t,n[2]/=t}function _t(n){return[Math.atan2(n[1],n[0]),tn(n[2])]}function wt(n,t){return xo(n[0]-t[0])<Uo&&xo(n[1]-t[1])<Uo}function St(n,t){n*=Yo;var e=Math.cos(t*=Yo);kt(e*Math.cos(n),e*Math.sin(n),Math.sin(t))}function kt(n,t,e){++Ea,Ca+=(n-Ca)/Ea,za+=(t-za)/Ea,La+=(e-La)/Ea}function Nt(){function n(n,i){n*=Yo;var u=Math.cos(i*=Yo),o=u*Math.cos(n),a=u*Math.sin(n),l=Math.sin(i),c=Math.atan2(Math.sqrt((c=e*l-r*a)*c+(c=r*o-t*l)*c+(c=t*a-e*o)*c),t*o+e*a+r*l);Aa+=c,qa+=c*(t+(t=o)),Ta+=c*(e+(e=a)),Ra+=c*(r+(r=l)),kt(t,e,r)}var t,e,r;ja.point=function(i,u){i*=Yo;var o=Math.cos(u*=Yo);t=o*Math.cos(i),e=o*Math.sin(i),r=Math.sin(u),ja.point=n,kt(t,e,r)}}function Et(){ja.point=St}function At(){function n(n,t){n*=Yo;var e=Math.cos(t*=Yo),o=e*Math.cos(n),a=e*Math.sin(n),l=Math.sin(t),c=i*l-u*a,f=u*o-r*l,s=r*a-i*o,h=Math.sqrt(c*c+f*f+s*s),p=r*o+i*a+u*l,g=h&&-nn(p)/h,v=Math.atan2(h,p);Da+=g*c,Pa+=g*f,Ua+=g*s,Aa+=v,qa+=v*(r+(r=o)),Ta+=v*(i+(i=a)),Ra+=v*(u+(u=l)),kt(r,i,u)}var t,e,r,i,u;ja.point=function(o,a){t=o,e=a,ja.point=n,o*=Yo;var l=Math.cos(a*=Yo);r=l*Math.cos(o),i=l*Math.sin(o),u=Math.sin(a),kt(r,i,u)},ja.lineEnd=function(){n(t,e),ja.lineEnd=Et,ja.point=St}}function Ct(n,t){function e(e,r){return e=n(e,r),t(e[0],e[1])}return n.invert&&t.invert&&(e.invert=function(e,r){return e=t.invert(e,r),e&&n.invert(e[0],e[1])}),e}function zt(){return!0}function Lt(n,t,e,r,i){var u=[],o=[];if(n.forEach(function(n){if(!((t=n.length-1)<=0)){var t,e=n[0],r=n[t];if(wt(e,r)){i.lineStart();for(var a=0;t>a;++a)i.point((e=n[a])[0],e[1]);return void i.lineEnd()}var l=new Tt(e,n,null,!0),c=new Tt(e,null,l,!1);l.o=c,u.push(l),o.push(c),l=new Tt(r,n,null,!1),c=new Tt(r,null,l,!0),l.o=c,u.push(l),o.push(c)}}),o.sort(t),qt(u),qt(o),u.length){for(var a=0,l=e,c=o.length;c>a;++a)o[a].e=l=!l;for(var f,s,h=u[0];;){for(var p=h,g=!0;p.v;)if((p=p.n)===h)return;f=p.z,i.lineStart();do{if(p.v=p.o.v=!0,p.e){if(g)for(var a=0,c=f.length;c>a;++a)i.point((s=f[a])[0],s[1]);else r(p.x,p.n.x,1,i);p=p.n}else{if(g){f=p.p.z;for(var a=f.length-1;a>=0;--a)i.point((s=f[a])[0],\
s[1])}else r(p.x,p.p.x,-1,i);p=p.p}p=p.o,f=p.z,g=!g}while(!p.v);i.lineEnd()}}}function qt(n){if(t=n.length){for(var t,e,r=0,i=n[0];++r<t;)i.n=e=n[r],e.p=i,i=e;i.n=e=n[0],e.p=i}}function Tt(n,t,e,r){this.x=n,this.z=t,this.o=e,this.e=r,this.v=!1,this.n=this.p=null}function Rt(n,t,e,r){return function(i,u){function o(t,e){var r=i(t,e);n(t=r[0],e=r[1])&&u.point(t,e)}function a(n,t){var e=i(n,t);d.point(e[0],e[1])}function l(){m.point=a,d.lineStart()}function c(){m.point=o,d.lineEnd()}function f(n,t){v.push([n,t]);var e=i(n,t);x.point(e[0],e[1])}function s(){x.lineStart(),v=[]}function h(){f(v[0][0],v[0][1]),x.lineEnd();var n,t=x.clean(),e=M.buffer(),r=e.length;if(v.pop(),g.push(v),v=null,r)if(1&t){n=e[0];var i,r=n.length-1,o=-1;if(r>0){for(b||(u.polygonStart(),b=!0),u.lineStart();++o<r;)u.point((i=n[o])[0],i[1]);u.lineEnd()}}else r>1&&2&t&&e.push(e.pop().concat(e.shift())),p.push(e.filter(Dt))}var p,g,v,d=t(u),y=i.invert(r[0],r[1]),m={point:o,lineStart:l,lineEnd:c,polygonStart:function(){m.point=f,m.lineStart=s,m.lineEnd=h,p=[],g=[]},polygonEnd:function(){m.point=o,m.lineStart=l,m.lineEnd=c,p=ao.merge(p);var n=Ot(y,g);p.length?(b||(u.polygonStart(),b=!0),Lt(p,Ut,n,e,u)):n&&(b||(u.polygonStart(),b=!0),u.lineStart(),e(null,null,1,u),u.lineEnd()),b&&(u.polygonEnd(),b=!1),p=g=null},sphere:function(){u.polygonStart(),u.lineStart(),e(null,null,1,u),u.lineEnd(),u.polygonEnd()}},M=Pt(),x=t(M),b=!1;return m}}function Dt(n){return n.length>1}function Pt(){var n,t=[];return{lineStart:function(){t.push(n=[])},point:function(t,e){n.push([t,e])},lineEnd:b,buffer:function(){var e=t;return t=[],n=null,e},rejoin:function(){t.length>1&&t.push(t.pop().concat(t.shift()))}}}function Ut(n,t){return((n=n.x)[0]<0?n[1]-Io-Uo:Io-n[1])-((t=t.x)[0]<0?t[1]-Io-Uo:Io-t[1])}function jt(n){var t,e=NaN,r=NaN,i=NaN;return{lineStart:function(){n.lineStart(),t=1},point:function(u,o){var a=u>0?Fo:-Fo,l=xo(u-e);xo(l-Fo)<Uo?(n.point(e,r=(r+o)/2>0?Io:-Io),n.point(i,r),n.lineEnd(),n.lineStart(),n.point(a,r),n.point(u,r),t=0):i!==a&&l>=Fo&&(xo(e-i)<Uo&&(e-=i*Uo),\
xo(u-a)<Uo&&(u-=a*Uo),r=Ft(e,r,u,o),n.point(i,r),n.lineEnd(),n.lineStart(),n.point(a,r),t=0),n.point(e=u,r=o),i=a},lineEnd:function(){n.lineEnd(),e=r=NaN},clean:function(){return 2-t}}}function Ft(n,t,e,r){var i,u,o=Math.sin(n-e);return xo(o)>Uo?Math.atan((Math.sin(t)*(u=Math.cos(r))*Math.sin(e)-Math.sin(r)*(i=Math.cos(t))*Math.sin(n))/(i*u*o)):(t+r)/2}function Ht(n,t,e,r){var i;if(null==n)i=e*Io,r.point(-Fo,i),r.point(0,i),r.point(Fo,i),r.point(Fo,0),r.point(Fo,-i),r.point(0,-i),r.point(-Fo,-i),r.point(-Fo,0),r.point(-Fo,i);else if(xo(n[0]-t[0])>Uo){var u=n[0]<t[0]?Fo:-Fo;i=e*u/2,r.point(-u,i),r.point(0,i),r.point(u,i)}else r.point(t[0],t[1])}function Ot(n,t){var e=n[0],r=n[1],i=[Math.sin(e),-Math.cos(e),0],u=0,o=0;ka.reset();for(var a=0,l=t.length;l>a;++a){var c=t[a],f=c.length;if(f)for(var s=c[0],h=s[0],p=s[1]/2+Fo/4,g=Math.sin(p),v=Math.cos(p),d=1;;){d===f&&(d=0),n=c[d];var y=n[0],m=n[1]/2+Fo/4,M=Math.sin(m),x=Math.cos(m),b=y-h,_=b>=0?1:-1,w=_*b,S=w>Fo,k=g*M;if(ka.add(Math.atan2(k*_*Math.sin(w),v*x+k*Math.cos(w))),u+=S?b+_*Ho:b,S^h>=e^y>=e){var N=mt(dt(s),dt(n));bt(N);var E=mt(i,N);bt(E);var A=(S^b>=0?-1:1)*tn(E[2]);(r>A||r===A&&(N[0]||N[1]))&&(o+=S^b>=0?1:-1)}if(!d++)break;h=y,g=M,v=x,s=n}}return(-Uo>u||Uo>u&&-Uo>ka)^1&o}function It(n){function t(n,t){return Math.cos(n)*Math.cos(t)>u}function e(n){var e,u,l,c,f;return{lineStart:function(){c=l=!1,f=1},point:function(s,h){var p,g=[s,h],v=t(s,h),d=o?v?0:i(s,h):v?i(s+(0>s?Fo:-Fo),h):0;if(!e&&(c=l=v)&&n.lineStart(),v!==l&&(p=r(e,g),(wt(e,p)||wt(g,p))&&(g[0]+=Uo,g[1]+=Uo,v=t(g[0],g[1]))),v!==l)f=0,v?(n.lineStart(),p=r(g,e),n.point(p[0],p[1])):(p=r(e,g),n.point(p[0],p[1]),n.lineEnd()),e=p;else if(a&&e&&o^v){var y;d&u||!(y=r(g,e,!0))||(f=0,o?(n.lineStart(),n.point(y[0][0],y[0][1]),n.point(y[1][0],y[1][1]),n.lineEnd()):(n.point(y[1][0],y[1][1]),n.lineEnd(),n.lineStart(),n.point(y[0][0],y[0][1])))}!v||e&&wt(e,g)||n.point(g[0],g[1]),e=g,l=v,u=d},lineEnd:function(){l&&n.lineEnd(),e=null},clean:function(){return f|(c&&l)<<1}}}function r(n,t,e){var r=dt(n),i=dt(t),o=[1,0,0],\
a=mt(r,i),l=yt(a,a),c=a[0],f=l-c*c;if(!f)return!e&&n;var s=u*l/f,h=-u*c/f,p=mt(o,a),g=xt(o,s),v=xt(a,h);Mt(g,v);var d=p,y=yt(g,d),m=yt(d,d),M=y*y-m*(yt(g,g)-1);if(!(0>M)){var x=Math.sqrt(M),b=xt(d,(-y-x)/m);if(Mt(b,g),b=_t(b),!e)return b;var _,w=n[0],S=t[0],k=n[1],N=t[1];w>S&&(_=w,w=S,S=_);var E=S-w,A=xo(E-Fo)<Uo,C=A||Uo>E;if(!A&&k>N&&(_=k,k=N,N=_),C?A?k+N>0^b[1]<(xo(b[0]-w)<Uo?k:N):k<=b[1]&&b[1]<=N:E>Fo^(w<=b[0]&&b[0]<=S)){var z=xt(d,(-y+x)/m);return Mt(z,g),[b,_t(z)]}}}function i(t,e){var r=o?n:Fo-n,i=0;return-r>t?i|=1:t>r&&(i|=2),-r>e?i|=4:e>r&&(i|=8),i}var u=Math.cos(n),o=u>0,a=xo(u)>Uo,l=ve(n,6*Yo);return Rt(t,e,l,o?[0,-n]:[-Fo,n-Fo])}function Yt(n,t,e,r){return function(i){var u,o=i.a,a=i.b,l=o.x,c=o.y,f=a.x,s=a.y,h=0,p=1,g=f-l,v=s-c;if(u=n-l,g||!(u>0)){if(u/=g,0>g){if(h>u)return;p>u&&(p=u)}else if(g>0){if(u>p)return;u>h&&(h=u)}if(u=e-l,g||!(0>u)){if(u/=g,0>g){if(u>p)return;u>h&&(h=u)}else if(g>0){if(h>u)return;p>u&&(p=u)}if(u=t-c,v||!(u>0)){if(u/=v,0>v){if(h>u)return;p>u&&(p=u)}else if(v>0){if(u>p)return;u>h&&(h=u)}if(u=r-c,v||!(0>u)){if(u/=v,0>v){if(u>p)return;u>h&&(h=u)}else if(v>0){if(h>u)return;p>u&&(p=u)}return h>0&&(i.a={x:l+h*g,y:c+h*v}),1>p&&(i.b={x:l+p*g,y:c+p*v}),i}}}}}}function Zt(n,t,e,r){function i(r,i){return xo(r[0]-n)<Uo?i>0?0:3:xo(r[0]-e)<Uo?i>0?2:1:xo(r[1]-t)<Uo?i>0?1:0:i>0?3:2}function u(n,t){return o(n.x,t.x)}function o(n,t){var e=i(n,1),r=i(t,1);return e!==r?e-r:0===e?t[1]-n[1]:1===e?n[0]-t[0]:2===e?n[1]-t[1]:t[0]-n[0]}return function(a){function l(n){for(var t=0,e=d.length,r=n[1],i=0;e>i;++i)for(var u,o=1,a=d[i],l=a.length,c=a[0];l>o;++o)u=a[o],c[1]<=r?u[1]>r&&Q(c,u,n)>0&&++t:u[1]<=r&&Q(c,u,n)<0&&--t,c=u;return 0!==t}function c(u,a,l,c){var f=0,s=0;if(null==u||(f=i(u,l))!==(s=i(a,l))||o(u,a)<0^l>0){do c.point(0===f||3===f?n:e,f>1?r:t);while((f=(f+l+4)%4)!==s)}else c.point(a[0],a[1])}function f(i,u){return i>=n&&e>=i&&u>=t&&r>=u}function s(n,t){f(n,t)&&a.point(n,t)}function h(){C.point=g,d&&d.push(y=[]),S=!0,w=!1,b=_=NaN}function p(){v&&(g(m,M),x&&w&&E.rejoin(),v.push(E.buffer())),C.point=s,\
w&&a.lineEnd()}function g(n,t){n=Math.max(-Ha,Math.min(Ha,n)),t=Math.max(-Ha,Math.min(Ha,t));var e=f(n,t);if(d&&y.push([n,t]),S)m=n,M=t,x=e,S=!1,e&&(a.lineStart(),a.point(n,t));else if(e&&w)a.point(n,t);else{var r={a:{x:b,y:_},b:{x:n,y:t}};A(r)?(w||(a.lineStart(),a.point(r.a.x,r.a.y)),a.point(r.b.x,r.b.y),e||a.lineEnd(),k=!1):e&&(a.lineStart(),a.point(n,t),k=!1)}b=n,_=t,w=e}var v,d,y,m,M,x,b,_,w,S,k,N=a,E=Pt(),A=Yt(n,t,e,r),C={point:s,lineStart:h,lineEnd:p,polygonStart:function(){a=E,v=[],d=[],k=!0},polygonEnd:function(){a=N,v=ao.merge(v);var t=l([n,r]),e=k&&t,i=v.length;(e||i)&&(a.polygonStart(),e&&(a.lineStart(),c(null,null,1,a),a.lineEnd()),i&&Lt(v,u,t,c,a),a.polygonEnd()),v=d=y=null}};return C}}function Vt(n){var t=0,e=Fo/3,r=ae(n),i=r(t,e);return i.parallels=function(n){return arguments.length?r(t=n[0]*Fo/180,e=n[1]*Fo/180):[t/Fo*180,e/Fo*180]},i}function Xt(n,t){function e(n,t){var e=Math.sqrt(u-2*i*Math.sin(t))/i;return[e*Math.sin(n*=i),o-e*Math.cos(n)]}var r=Math.sin(n),i=(r+Math.sin(t))/2,u=1+r*(2*i-r),o=Math.sqrt(u)/i;return e.invert=function(n,t){var e=o-t;return[Math.atan2(n,e)/i,tn((u-(n*n+e*e)*i*i)/(2*i))]},e}function $t(){function n(n,t){Ia+=i*n-r*t,r=n,i=t}var t,e,r,i;$a.point=function(u,o){$a.point=n,t=r=u,e=i=o},$a.lineEnd=function(){n(t,e)}}function Bt(n,t){Ya>n&&(Ya=n),n>Va&&(Va=n),Za>t&&(Za=t),t>Xa&&(Xa=t)}function Wt(){function n(n,t){o.push(\"M\",n,\",\",t,u)}function t(n,t){o.push(\"M\",n,\",\",t),a.point=e}function e(n,t){o.push(\"L\",n,\",\",t)}function r(){a.point=n}function i(){o.push(\"Z\")}var u=Jt(4.5),o=[],a={point:n,lineStart:function(){a.point=t},lineEnd:r,polygonStart:function(){a.lineEnd=i},polygonEnd:function(){a.lineEnd=r,a.point=n},pointRadius:function(n){return u=Jt(n),a},result:function(){if(o.length){var n=o.join(\"\");return o=[],n}}};return a}function Jt(n){return\"m0,\"+n+\"a\"+n+\",\"+n+\" 0 1,1 0,\"+-2*n+\"a\"+n+\",\"+n+\" 0 1,1 0,\"+2*n+\"z\"}function Gt(n,t){Ca+=n,za+=t,++La}function Kt(){function n(n,r){var i=n-t,u=r-e,o=Math.sqrt(i*i+u*u);qa+=o*(t+n)/2,Ta+=o*(e+r)/2,Ra+=o,Gt(t=n,e=r)}var t,e;Wa.point=function(r,\
i){Wa.point=n,Gt(t=r,e=i)}}function Qt(){Wa.point=Gt}function ne(){function n(n,t){var e=n-r,u=t-i,o=Math.sqrt(e*e+u*u);qa+=o*(r+n)/2,Ta+=o*(i+t)/2,Ra+=o,o=i*n-r*t,Da+=o*(r+n),Pa+=o*(i+t),Ua+=3*o,Gt(r=n,i=t)}var t,e,r,i;Wa.point=function(u,o){Wa.point=n,Gt(t=r=u,e=i=o)},Wa.lineEnd=function(){n(t,e)}}function te(n){function t(t,e){n.moveTo(t+o,e),n.arc(t,e,o,0,Ho)}function e(t,e){n.moveTo(t,e),a.point=r}function r(t,e){n.lineTo(t,e)}function i(){a.point=t}function u(){n.closePath()}var o=4.5,a={point:t,lineStart:function(){a.point=e},lineEnd:i,polygonStart:function(){a.lineEnd=u},polygonEnd:function(){a.lineEnd=i,a.point=t},pointRadius:function(n){return o=n,a},result:b};return a}function ee(n){function t(n){return(a?r:e)(n)}function e(t){return ue(t,function(e,r){e=n(e,r),t.point(e[0],e[1])})}function r(t){function e(e,r){e=n(e,r),t.point(e[0],e[1])}function r(){M=NaN,S.point=u,t.lineStart()}function u(e,r){var u=dt([e,r]),o=n(e,r);i(M,x,m,b,_,w,M=o[0],x=o[1],m=e,b=u[0],_=u[1],w=u[2],a,t),t.point(M,x)}function o(){S.point=e,t.lineEnd()}function l(){\
r(),S.point=c,S.lineEnd=f}function c(n,t){u(s=n,h=t),p=M,g=x,v=b,d=_,y=w,S.point=u}function f(){i(M,x,m,b,_,w,p,g,s,v,d,y,a,t),S.lineEnd=o,o()}var s,h,p,g,v,d,y,m,M,x,b,_,w,S={point:e,lineStart:r,lineEnd:o,polygonStart:function(){t.polygonStart(),S.lineStart=l},polygonEnd:function(){t.polygonEnd(),S.lineStart=r}};return S}function i(t,e,r,a,l,c,f,s,h,p,g,v,d,y){var m=f-t,M=s-e,x=m*m+M*M;if(x>4*u&&d--){var b=a+p,_=l+g,w=c+v,S=Math.sqrt(b*b+_*_+w*w),k=Math.asin(w/=S),N=xo(xo(w)-1)<Uo||xo(r-h)<Uo?(r+h)/2:Math.atan2(_,b),E=n(N,k),A=E[0],C=E[1],z=A-t,L=C-e,q=M*z-m*L;(q*q/x>u||xo((m*z+M*L)/x-.5)>.3||o>a*p+l*g+c*v)&&(i(t,e,r,a,l,c,A,C,N,b/=S,_/=S,w,d,y),y.point(A,C),i(A,C,N,b,_,w,f,s,h,p,g,v,d,y))}}var u=.5,o=Math.cos(30*Yo),a=16;return t.precision=function(n){return arguments.length?(a=(u=n*n)>0&&16,t):Math.sqrt(u)},t}function re(n){var t=ee(function(t,e){return n([t*Zo,e*Zo])});return function(n){return le(t(n))}}function ie(n){this.stream=n}function ue(n,t){return{point:t,sphere:function(){n.sphere()},lineStart:function(){n.lineStart()},lineEnd:function(){n.lineEnd()},polygonStart:function(){n.polygonStart()},polygonEnd:function(){n.polygonEnd()}}}function oe(n){return ae(function(){return n})()}function ae(n){function t(n){return n=a(n[0]*Yo,n[1]*Yo),[n[0]*h+l,c-n[1]*h]}function e(n){return n=a.invert((n[0]-l)/h,(c-n[1])/h),n&&[n[0]*Zo,n[1]*Zo]}function r(){a=Ct(o=se(y,M,x),u);var n=u(v,d);return l=p-n[0]*h,c=g+n[1]*h,i()}function i(){return f&&(f.valid=!1,f=null),t}var u,o,a,l,c,f,s=ee(function(n,t){return n=u(n,t),[n[0]*h+l,c-n[1]*h]}),h=150,p=480,g=250,v=0,d=0,y=0,M=0,x=0,b=Fa,_=m,w=null,S=null;return t.stream=function(n){return f&&(f.valid=!1),f=le(b(o,s(_(n)))),f.valid=!0,f},t.clipAngle=function(n){return arguments.length?(b=null==n?(w=n,Fa):It((w=+n)*Yo),i()):w},t.clipExtent=function(n){return arguments.length?(S=n,_=n?Zt(n[0][0],n[0][1],n[1][0],n[1][1]):m,i()):S},t.scale=function(n){return arguments.length?(h=+n,r()):h},t.translate=function(n){return arguments.length?(p=+n[0],g=+n[1],r()):[p,g]},t.center=function(n){return \
arguments.length?(v=n[0]%360*Yo,d=n[1]%360*Yo,r()):[v*Zo,d*Zo]},t.rotate=function(n){return arguments.length?(y=n[0]%360*Yo,M=n[1]%360*Yo,x=n.length>2?n[2]%360*Yo:0,r()):[y*Zo,M*Zo,x*Zo]},ao.rebind(t,s,\"precision\"),function(){return u=n.apply(this,arguments),t.invert=u.invert&&e,r()}}function le(n){return ue(n,function(t,e){n.point(t*Yo,e*Yo)})}function ce(n,t){return[n,t]}function fe(n,t){return[n>Fo?n-Ho:-Fo>n?n+Ho:n,t]}function se(n,t,e){return n?t||e?Ct(pe(n),ge(t,e)):pe(n):t||e?ge(t,e):fe}function he(n){return function(t,e){return t+=n,[t>Fo?t-Ho:-Fo>t?t+Ho:t,e]}}function pe(n){var t=he(n);return t.invert=he(-n),t}function ge(n,t){function e(n,t){var e=Math.cos(t),a=Math.cos(n)*e,l=Math.sin(n)*e,c=Math.sin(t),f=c*r+a*i;return[Math.atan2(l*u-f*o,a*r-c*i),tn(f*u+l*o)]}var r=Math.cos(n),i=Math.sin(n),u=Math.cos(t),o=Math.sin(t);return e.invert=function(n,t){var e=Math.cos(t),a=Math.cos(n)*e,l=Math.sin(n)*e,c=Math.sin(t),f=c*u-l*o;return[Math.atan2(l*u+c*o,a*r+f*i),tn(f*r-a*i)]},e}function ve(n,t){var e=Math.cos(n),r=Math.sin(n);return function(i,u,o,a){var l=o*t;null!=i?(i=de(e,i),u=de(e,u),(o>0?u>i:i>u)&&(i+=o*Ho)):(i=n+o*Ho,u=n-.5*l);for(var c,f=i;o>0?f>u:u>f;f-=l)a.point((c=_t([e,-r*Math.cos(f),-r*Math.sin(f)]))[0],c[1])}}function de(n,t){var e=dt(t);e[0]-=n,bt(e);var r=nn(-e[1]);return((-e[2]<0?-r:r)+2*Math.PI-Uo)%(2*Math.PI)}function ye(n,t,e){var r=ao.range(n,t-Uo,e).concat(t);return function(n){return r.map(function(t){return[n,t]})}}function me(n,t,e){var r=ao.range(n,t-Uo,e).concat(t);return function(n){return r.map(function(t){return[t,n]})}}function Me(n){return n.source}function xe(n){return n.target}function be(n,t,e,r){var i=Math.cos(t),u=Math.sin(t),o=Math.cos(r),a=Math.sin(r),l=i*Math.cos(n),c=i*Math.sin(n),f=o*Math.cos(e),s=o*Math.sin(e),h=2*Math.asin(Math.sqrt(on(r-t)+i*o*on(e-n))),p=1/Math.sin(h),g=h?function(n){var t=Math.sin(n*=h)*p,e=Math.sin(h-n)*p,r=e*l+t*f,i=e*c+t*s,o=e*u+t*a;return[Math.atan2(i,r)*Zo,Math.atan2(o,Math.sqrt(r*r+i*i))*Zo]}:function(){return[n*Zo,t*Zo]};return g.distance=h,\
g}function _e(){function n(n,i){var u=Math.sin(i*=Yo),o=Math.cos(i),a=xo((n*=Yo)-t),l=Math.cos(a);Ja+=Math.atan2(Math.sqrt((a=o*Math.sin(a))*a+(a=r*u-e*o*l)*a),e*u+r*o*l),t=n,e=u,r=o}var t,e,r;Ga.point=function(i,u){t=i*Yo,e=Math.sin(u*=Yo),r=Math.cos(u),Ga.point=n},Ga.lineEnd=function(){Ga.point=Ga.lineEnd=b}}function we(n,t){function e(t,e){var r=Math.cos(t),i=Math.cos(e),u=n(r*i);return[u*i*Math.sin(t),u*Math.sin(e)]}return e.invert=function(n,e){var r=Math.sqrt(n*n+e*e),i=t(r),u=Math.sin(i),o=Math.cos(i);return[Math.atan2(n*u,r*o),Math.asin(r&&e*u/r)]},e}function Se(n,t){function e(n,t){o>0?-Io+Uo>t&&(t=-Io+Uo):t>Io-Uo&&(t=Io-Uo);var e=o/Math.pow(i(t),u);return[e*Math.sin(u*n),o-e*Math.cos(u*n)]}var r=Math.cos(n),i=function(n){return Math.tan(Fo/4+n/2)},u=n===t?Math.sin(n):Math.log(r/Math.cos(t))/Math.log(i(t)/i(n)),o=r*Math.pow(i(n),u)/u;return u?(e.invert=function(n,t){var e=o-t,r=K(u)*Math.sqrt(n*n+e*e);return[Math.atan2(n,e)/u,2*Math.atan(Math.pow(o/r,1/u))-Io]},e):Ne}function ke(n,t){function e(n,t){var e=u-t;return[e*Math.sin(i*n),u-e*Math.cos(i*n)]}var r=Math.cos(n),i=n===t?Math.sin(n):(r-Math.cos(t))/(t-n),u=r/i+n;return xo(i)<Uo?ce:(e.invert=function(n,t){var e=u-t;return[Math.atan2(n,e)/i,u-K(i)*Math.sqrt(n*n+e*e)]},e)}function Ne(n,t){return[n,Math.log(Math.tan(Fo/4+t/2))]}function Ee(n){var t,e=oe(n),r=e.scale,i=e.translate,u=e.clipExtent;return e.scale=function(){var n=r.apply(e,arguments);return n===e?t?e.clipExtent(null):e:n},e.translate=function(){var n=i.apply(e,arguments);return n===e?t?e.clipExtent(null):e:n},e.clipExtent=function(n){var o=u.apply(e,arguments);if(o===e){if(t=null==n){var a=Fo*r(),l=i();u([[l[0]-a,l[1]-a],[l[0]+a,l[1]+a]])}}else t&&(o=null);return o},e.clipExtent(null)}function Ae(n,t){return[Math.log(Math.tan(Fo/4+t/2)),-n]}function Ce(n){return n[0]}function ze(n){return n[1]}function Le(n){for(var t=n.length,e=[0,1],r=2,i=2;t>i;i++){for(;r>1&&Q(n[e[r-2]],n[e[r-1]],n[i])<=0;)--r;e[r++]=i}return e.slice(0,r)}function qe(n,t){return n[0]-t[0]||n[1]-t[1]}function Te(n,t,e){return(e[0]-t[0])*(n[1]-t[1])<(e[1]-t[1])*(n[0]-t[0])}function \
Re(n,t,e,r){var i=n[0],u=e[0],o=t[0]-i,a=r[0]-u,l=n[1],c=e[1],f=t[1]-l,s=r[1]-c,h=(a*(l-c)-s*(i-u))/(s*o-a*f);return[i+h*o,l+h*f]}function De(n){var t=n[0],e=n[n.length-1];return!(t[0]-e[0]||t[1]-e[1])}function Pe(){rr(this),this.edge=this.site=this.circle=null}function Ue(n){var t=cl.pop()||new Pe;return t.site=n,t}function je(n){Be(n),ol.remove(n),cl.push(n),rr(n)}function Fe(n){var t=n.circle,e=t.x,r=t.cy,i={x:e,y:r},u=n.P,o=n.N,a=[n];je(n);for(var l=u;l.circle&&xo(e-l.circle.x)<Uo&&xo(r-l.circle.cy)<Uo;)u=l.P,a.unshift(l),je(l),l=u;a.unshift(l),Be(l);for(var c=o;c.circle&&xo(e-c.circle.x)<Uo&&xo(r-c.circle.cy)<Uo;)o=c.N,a.push(c),je(c),c=o;a.push(c),Be(c);var f,s=a.length;for(f=1;s>f;++f)c=a[f],l=a[f-1],nr(c.edge,l.site,c.site,i);l=a[0],c=a[s-1],c.edge=Ke(l.site,c.site,null,i),$e(l),$e(c)}function He(n){for(var t,e,r,i,u=n.x,o=n.y,a=ol._;a;)if(r=Oe(a,o)-u,r>Uo)a=a.L;else{if(i=u-Ie(a,o),!(i>Uo)){r>-Uo?(t=a.P,e=a):i>-Uo?(t=a,e=a.N):t=e=a;break}if(!a.R){t=a;break}a=a.R}var l=Ue(n);if(ol.insert(t,l),t||e){if(t===e)return Be(t),e=Ue(t.site),ol.insert(l,e),l.edge=e.edge=Ke(t.site,l.site),$e(t),void $e(e);if(!e)return void(l.edge=Ke(t.site,l.site));Be(t),Be(e);var c=t.site,f=c.x,s=c.y,h=n.x-f,p=n.y-s,g=e.site,v=g.x-f,d=g.y-s,y=2*(h*d-p*v),m=h*h+p*p,M=v*v+d*d,x={x:(d*m-p*M)/y+f,y:(h*M-v*m)/y+s};nr(e.edge,c,g,x),l.edge=Ke(c,n,null,x),e.edge=Ke(n,g,null,x),$e(t),$e(e)}}function Oe(n,t){var e=n.site,r=e.x,i=e.y,u=i-t;if(!u)return r;var o=n.P;if(!o)return-(1/0);e=o.site;var a=e.x,l=e.y,c=l-t;if(!c)return a;var f=a-r,s=1/u-1/c,h=f/c;return s?(-h+Math.sqrt(h*h-2*s*(f*f/(-2*c)-l+c/2+i-u/2)))/s+r:(r+a)/2}function Ie(n,t){var e=n.N;if(e)return Oe(e,t);var r=n.site;return r.y===t?r.x:1/0}function Ye(n){this.site=n,this.edges=[]}function Ze(n){for(var t,e,r,i,u,o,a,l,c,f,s=n[0][0],h=n[1][0],p=n[0][1],g=n[1][1],v=ul,d=v.length;d--;)if(u=v[d],u&&u.prepare())for(a=u.edges,l=a.length,o=0;l>o;)f=a[o].end(),r=f.x,i=f.y,c=a[++o%l].start(),t=c.x,e=c.y,(xo(r-t)>Uo||xo(i-e)>Uo)&&(a.splice(o,0,new tr(Qe(u.site,f,xo(r-s)<Uo&&g-i>Uo?{x:s,y:xo(t-s)<Uo?e:g}:xo(i-g)<Uo&&h-r>Uo?{x:xo(e-g)<Uo?t:h,\
y:g}:xo(r-h)<Uo&&i-p>Uo?{x:h,y:xo(t-h)<Uo?e:p}:xo(i-p)<Uo&&r-s>Uo?{x:xo(e-p)<Uo?t:s,y:p}:null),u.site,null)),++l)}function Ve(n,t){return t.angle-n.angle}function Xe(){rr(this),this.x=this.y=this.arc=this.site=this.cy=null}function $e(n){var t=n.P,e=n.N;if(t&&e){var r=t.site,i=n.site,u=e.site;if(r!==u){var o=i.x,a=i.y,l=r.x-o,c=r.y-a,f=u.x-o,s=u.y-a,h=2*(l*s-c*f);if(!(h>=-jo)){var p=l*l+c*c,g=f*f+s*s,v=(s*p-c*g)/h,d=(l*g-f*p)/h,s=d+a,y=fl.pop()||new Xe;y.arc=n,y.site=i,y.x=v+o,y.y=s+Math.sqrt(v*v+d*d),y.cy=s,n.circle=y;for(var m=null,M=ll._;M;)if(y.y<M.y||y.y===M.y&&y.x<=M.x){if(!M.L){m=M.P;break}M=M.L}else{if(!M.R){m=M;break}M=M.R}ll.insert(m,y),m||(al=y)}}}}function Be(n){var t=n.circle;t&&(t.P||(al=t.N),ll.remove(t),fl.push(t),rr(t),n.circle=null)}function We(n){for(var t,e=il,r=Yt(n[0][0],n[0][1],n[1][0],n[1][1]),i=e.length;i--;)t=e[i],(!Je(t,n)||!r(t)||xo(t.a.x-t.b.x)<Uo&&xo(t.a.y-t.b.y)<Uo)&&(t.a=t.b=null,e.splice(i,1))}function Je(n,t){var e=n.b;if(e)return!0;var r,i,u=n.a,o=t[0][0],a=t[1][0],l=t[0][1],c=t[1][1],f=n.l,s=n.r,h=f.x,p=f.y,g=s.x,v=s.y,d=(h+g)/2,y=(p+v)/2;if(v===p){if(o>d||d>=a)return;if(h>g){if(u){if(u.y>=c)return}else u={x:d,y:l};e={x:d,y:c}}else{if(u){if(u.y<l)return}else u={x:d,y:c};e={x:d,y:l}}}else if(r=(h-g)/(v-p),i=y-r*d,-1>r||r>1)if(h>g){if(u){if(u.y>=c)return}else u={x:(l-i)/r,y:l};e={x:(c-i)/r,y:c}}else{if(u){if(u.y<l)return}else u={x:(c-i)/r,y:c};e={x:(l-i)/r,y:l}}else if(v>p){if(u){if(u.x>=a)return}else u={x:o,y:r*o+i};e={x:a,y:r*a+i}}else{if(u){if(u.x<o)return}else u={x:a,y:r*a+i};e={x:o,y:r*o+i}}return n.a=u,n.b=e,!0}function Ge(n,t){this.l=n,this.r=t,this.a=this.b=null}function Ke(n,t,e,r){var i=new Ge(n,t);return il.push(i),e&&nr(i,n,t,e),r&&nr(i,t,n,r),ul[n.i].edges.push(new tr(i,n,t)),ul[t.i].edges.push(new tr(i,t,n)),i}function Qe(n,t,e){var r=new Ge(n,null);return r.a=t,r.b=e,il.push(r),r}function nr(n,t,e,r){n.a||n.b?n.l===e?n.b=r:n.a=r:(n.a=r,n.l=t,n.r=e)}function tr(n,t,e){var r=n.a,i=n.b;this.edge=n,this.site=t,this.angle=e?Math.atan2(e.y-t.y,e.x-t.x):n.l===t?Math.atan2(i.x-r.x,\
r.y-i.y):Math.atan2(r.x-i.x,i.y-r.y)}function er(){this._=null}function rr(n){n.U=n.C=n.L=n.R=n.P=n.N=null}function ir(n,t){var e=t,r=t.R,i=e.U;i?i.L===e?i.L=r:i.R=r:n._=r,r.U=i,e.U=r,e.R=r.L,e.R&&(e.R.U=e),r.L=e}function ur(n,t){var e=t,r=t.L,i=e.U;i?i.L===e?i.L=r:i.R=r:n._=r,r.U=i,e.U=r,e.L=r.R,e.L&&(e.L.U=e),r.R=e}function or(n){for(;n.L;)n=n.L;return n}function ar(n,t){var e,r,i,u=n.sort(lr).pop();for(il=[],ul=new Array(n.length),ol=new er,ll=new er;;)if(i=al,u&&(!i||u.y<i.y||u.y===i.y&&u.x<i.x))u.x===e&&u.y===r||(ul[u.i]=new Ye(u),He(u),e=u.x,r=u.y),u=n.pop();else{if(!i)break;Fe(i.arc)}t&&(We(t),Ze(t));var o={cells:ul,edges:il};return ol=ll=il=ul=null,o}function lr(n,t){return t.y-n.y||t.x-n.x}function cr(n,t,e){return(n.x-e.x)*(t.y-n.y)-(n.x-t.x)*(e.y-n.y)}function fr(n){return n.x}function sr(n){return n.y}function hr(){return{leaf:!0,nodes:[],point:null,x:null,y:null}}function pr(n,t,e,r,i,u){if(!n(t,e,r,i,u)){var o=.5*(e+i),a=.5*(r+u),l=t.nodes;l[0]&&pr(n,l[0],e,r,o,a),l[1]&&pr(n,l[1],o,r,i,a),l[2]&&pr(n,l[2],e,a,o,u),l[3]&&pr(n,l[3],o,a,i,u)}}function gr(n,t,e,r,i,u,o){var a,l=1/0;return function c(n,f,s,h,p){if(!(f>u||s>o||r>h||i>p)){if(g=n.point){var g,v=t-n.x,d=e-n.y,y=v*v+d*d;if(l>y){var m=Math.sqrt(l=y);r=t-m,i=e-m,u=t+m,o=e+m,a=g}}for(var M=n.nodes,x=.5*(f+h),b=.5*(s+p),_=t>=x,w=e>=b,S=w<<1|_,k=S+4;k>S;++S)if(n=M[3&S])switch(3&S){case 0:c(n,f,s,x,b);break;case 1:c(n,x,s,h,b);break;case 2:c(n,f,b,x,p);break;case 3:c(n,x,b,h,p)}}}(n,r,i,u,o),a}function vr(n,t){n=ao.rgb(n),t=ao.rgb(t);var e=n.r,r=n.g,i=n.b,u=t.r-e,o=t.g-r,a=t.b-i;return function(n){return\"#\"+bn(Math.round(e+u*n))+bn(Math.round(r+o*n))+bn(Math.round(i+a*n))}}function dr(n,t){var e,r={},i={};for(e in n)e in t?r[e]=Mr(n[e],t[e]):i[e]=n[e];for(e in t)e in n||(i[e]=t[e]);return function(n){for(e in r)i[e]=r[e](n);return i}}function yr(n,t){return n=+n,t=+t,function(e){return n*(1-e)+t*e}}function mr(n,t){var e,r,i,u=hl.lastIndex=pl.lastIndex=0,o=-1,a=[],l=[];for(n+=\"\",t+=\"\";(e=hl.exec(n))&&(r=pl.exec(t));)(i=r.index)>u&&(i=t.slice(u,i),a[o]?a[o]+=i:a[++o]=i),\
(e=e[0])===(r=r[0])?a[o]?a[o]+=r:a[++o]=r:(a[++o]=null,l.push({i:o,x:yr(e,r)})),u=pl.lastIndex;return u<t.length&&(i=t.slice(u),a[o]?a[o]+=i:a[++o]=i),a.length<2?l[0]?(t=l[0].x,function(n){return t(n)+\"\"}):function(){return t}:(t=l.length,function(n){for(var e,r=0;t>r;++r)a[(e=l[r]).i]=e.x(n);return a.join(\"\")})}function Mr(n,t){for(var e,r=ao.interpolators.length;--r>=0&&!(e=ao.interpolators[r](n,t)););return e}function xr(n,t){var e,r=[],i=[],u=n.length,o=t.length,a=Math.min(n.length,t.length);for(e=0;a>e;++e)r.push(Mr(n[e],t[e]));for(;u>e;++e)i[e]=n[e];for(;o>e;++e)i[e]=t[e];return function(n){for(e=0;a>e;++e)i[e]=r[e](n);return i}}function br(n){return function(t){return 0>=t?0:t>=1?1:n(t)}}function _r(n){return function(t){return 1-n(1-t)}}function wr(n){return function(t){return.5*(.5>t?n(2*t):2-n(2-2*t))}}function Sr(n){return n*n}function kr(n){return n*n*n}function Nr(n){if(0>=n)return 0;if(n>=1)return 1;var t=n*n,e=t*n;return 4*(.5>n?e:3*(n-t)+e-.75)}function Er(n){return function(t){return Math.pow(t,n)}}function Ar(n){return 1-Math.cos(n*Io)}function Cr(n){return Math.pow(2,10*(n-1))}function zr(n){return 1-Math.sqrt(1-n*n)}function Lr(n,t){var e;return arguments.length<2&&(t=.45),arguments.length?e=t/Ho*Math.asin(1/n):(n=1,e=t/4),function(r){return 1+n*Math.pow(2,-10*r)*Math.sin((r-e)*Ho/t)}}function qr(n){return n||(n=1.70158),function(t){return t*t*((n+1)*t-n)}}function Tr(n){return 1/2.75>n?7.5625*n*n:2/2.75>n?7.5625*(n-=1.5/2.75)*n+.75:2.5/2.75>n?7.5625*(n-=2.25/2.75)*n+.9375:7.5625*(n-=2.625/2.75)*n+.984375}function Rr(n,t){n=ao.hcl(n),t=ao.hcl(t);var e=n.h,r=n.c,i=n.l,u=t.h-e,o=t.c-r,a=t.l-i;return isNaN(o)&&(o=0,r=isNaN(r)?t.c:r),isNaN(u)?(u=0,e=isNaN(e)?t.h:e):u>180?u-=360:-180>u&&(u+=360),function(n){return sn(e+u*n,r+o*n,i+a*n)+\"\"}}function Dr(n,t){n=ao.hsl(n),t=ao.hsl(t);var e=n.h,r=n.s,i=n.l,u=t.h-e,o=t.s-r,a=t.l-i;return isNaN(o)&&(o=0,r=isNaN(r)?t.s:r),isNaN(u)?(u=0,e=isNaN(e)?t.h:e):u>180?u-=360:-180>u&&(u+=360),function(n){return cn(e+u*n,r+o*n,i+a*n)+\"\"}}function Pr(n,t){n=ao.lab(n),t=ao.lab(t);\
var e=n.l,r=n.a,i=n.b,u=t.l-e,o=t.a-r,a=t.b-i;return function(n){return pn(e+u*n,r+o*n,i+a*n)+\"\"}}function Ur(n,t){return t-=n,function(e){return Math.round(n+t*e)}}function jr(n){var t=[n.a,n.b],e=[n.c,n.d],r=Hr(t),i=Fr(t,e),u=Hr(Or(e,t,-i))||0;t[0]*e[1]<e[0]*t[1]&&(t[0]*=-1,t[1]*=-1,r*=-1,i*=-1),this.rotate=(r?Math.atan2(t[1],t[0]):Math.atan2(-e[0],e[1]))*Zo,this.translate=[n.e,n.f],this.scale=[r,u],this.skew=u?Math.atan2(i,u)*Zo:0}function Fr(n,t){return n[0]*t[0]+n[1]*t[1]}function Hr(n){var t=Math.sqrt(Fr(n,n));return t&&(n[0]/=t,n[1]/=t),t}function Or(n,t,e){return n[0]+=e*t[0],n[1]+=e*t[1],n}function Ir(n){return n.length?n.pop()+\",\":\"\"}function Yr(n,t,e,r){if(n[0]!==t[0]||n[1]!==t[1]){var i=e.push(\"translate(\",null,\",\",null,\")\");r.push({i:i-4,x:yr(n[0],t[0])},{i:i-2,x:yr(n[1],t[1])})}else(t[0]||t[1])&&e.push(\"translate(\"+t+\")\")}function Zr(n,t,e,r){n!==t?(n-t>180?t+=360:t-n>180&&(n+=360),r.push({i:e.push(Ir(e)+\"rotate(\",null,\")\")-2,x:yr(n,t)})):t&&e.push(Ir(e)+\"rotate(\"+t+\")\")}function Vr(n,t,e,r){n!==t?r.push({i:e.push(Ir(e)+\"skewX(\",null,\")\")-2,x:yr(n,t)}):t&&e.push(Ir(e)+\"skewX(\"+t+\")\")}function Xr(n,t,e,r){if(n[0]!==t[0]||n[1]!==t[1]){var i=e.push(Ir(e)+\"scale(\",null,\",\",null,\")\");r.push({i:i-4,x:yr(n[0],t[0])},{i:i-2,x:yr(n[1],t[1])})}else 1===t[0]&&1===t[1]||e.push(Ir(e)+\"scale(\"+t+\")\")}function $r(n,t){var e=[],r=[];return n=ao.transform(n),t=ao.transform(t),Yr(n.translate,t.translate,e,r),Zr(n.rotate,t.rotate,e,r),Vr(n.skew,t.skew,e,r),Xr(n.scale,t.scale,e,r),n=t=null,function(n){for(var t,i=-1,u=r.length;++i<u;)e[(t=r[i]).i]=t.x(n);return e.join(\"\")}}function Br(n,t){return t=(t-=n=+n)||1/t,function(e){return(e-n)/t}}function Wr(n,t){return t=(t-=n=+n)||1/t,function(e){return Math.max(0,Math.min(1,(e-n)/t))}}function Jr(n){for(var t=n.source,e=n.target,r=Kr(t,e),i=[t];t!==r;)t=t.parent,i.push(t);for(var u=i.length;e!==r;)i.splice(u,0,e),e=e.parent;return i}function Gr(n){for(var t=[],e=n.parent;null!=e;)t.push(n),n=e,e=e.parent;return t.push(n),t}function Kr(n,t){if(n===t)return n;for(var e=Gr(n),r=Gr(t),\
i=e.pop(),u=r.pop(),o=null;i===u;)o=i,i=e.pop(),u=r.pop();return o}function Qr(n){n.fixed|=2}function ni(n){n.fixed&=-7}function ti(n){n.fixed|=4,n.px=n.x,n.py=n.y}function ei(n){n.fixed&=-5}function ri(n,t,e){var r=0,i=0;if(n.charge=0,!n.leaf)for(var u,o=n.nodes,a=o.length,l=-1;++l<a;)u=o[l],null!=u&&(ri(u,t,e),n.charge+=u.charge,r+=u.charge*u.cx,i+=u.charge*u.cy);if(n.point){n.leaf||(n.point.x+=Math.random()-.5,n.point.y+=Math.random()-.5);var c=t*e[n.point.index];n.charge+=n.pointCharge=c,r+=c*n.point.x,i+=c*n.point.y}n.cx=r/n.charge,n.cy=i/n.charge}function ii(n,t){return ao.rebind(n,t,\"sort\",\"children\",\"value\"),n.nodes=n,n.links=fi,n}function ui(n,t){for(var e=[n];null!=(n=e.pop());)if(t(n),(i=n.children)&&(r=i.length))for(var r,i;--r>=0;)e.push(i[r])}function oi(n,t){for(var e=[n],r=[];null!=(n=e.pop());)if(r.push(n),(u=n.children)&&(i=u.length))for(var i,u,o=-1;++o<i;)e.push(u[o]);for(;null!=(n=r.pop());)t(n)}function ai(n){return n.children}function li(n){return n.value}function ci(n,t){return t.value-n.value}function fi(n){return ao.merge(n.map(function(n){return(n.children||[]).map(function(t){return{source:n,target:t}})}))}function si(n){return n.x}function hi(n){return n.y}function pi(n,t,e){n.y0=t,n.y=e}function gi(n){return ao.range(n.length)}function vi(n){for(var t=-1,e=n[0].length,r=[];++t<e;)r[t]=0;return r}function di(n){for(var t,e=1,r=0,i=n[0][1],u=n.length;u>e;++e)(t=n[e][1])>i&&(r=e,i=t);return r}function yi(n){return n.reduce(mi,0)}function mi(n,t){return n+t[1]}function Mi(n,t){return xi(n,Math.ceil(Math.log(t.length)/Math.LN2+1))}function xi(n,t){for(var e=-1,r=+n[0],i=(n[1]-r)/t,u=[];++e<=t;)u[e]=i*e+r;return u}function bi(n){return[ao.min(n),ao.max(n)]}function _i(n,t){return n.value-t.value}function wi(n,t){var e=n._pack_next;n._pack_next=t,t._pack_prev=n,t._pack_next=e,e._pack_prev=t}function Si(n,t){n._pack_next=t,t._pack_prev=n}function ki(n,t){var e=t.x-n.x,r=t.y-n.y,i=n.r+t.r;return.999*i*i>e*e+r*r}function Ni(n){function t(n){f=Math.min(n.x-n.r,f),s=Math.max(n.x+n.r,s),h=Math.min(n.y-n.r,\
h),p=Math.max(n.y+n.r,p)}if((e=n.children)&&(c=e.length)){var e,r,i,u,o,a,l,c,f=1/0,s=-(1/0),h=1/0,p=-(1/0);if(e.forEach(Ei),r=e[0],r.x=-r.r,r.y=0,t(r),c>1&&(i=e[1],i.x=i.r,i.y=0,t(i),c>2))for(u=e[2],zi(r,i,u),t(u),wi(r,u),r._pack_prev=u,wi(u,i),i=r._pack_next,o=3;c>o;o++){zi(r,i,u=e[o]);var g=0,v=1,d=1;for(a=i._pack_next;a!==i;a=a._pack_next,v++)if(ki(a,u)){g=1;break}if(1==g)for(l=r._pack_prev;l!==a._pack_prev&&!ki(l,u);l=l._pack_prev,d++);g?(d>v||v==d&&i.r<r.r?Si(r,i=a):Si(r=l,i),o--):(wi(r,u),i=u,t(u))}var y=(f+s)/2,m=(h+p)/2,M=0;for(o=0;c>o;o++)u=e[o],u.x-=y,u.y-=m,M=Math.max(M,u.r+Math.sqrt(u.x*u.x+u.y*u.y));n.r=M,e.forEach(Ai)}}function Ei(n){n._pack_next=n._pack_prev=n}function Ai(n){delete n._pack_next,delete n._pack_prev}function Ci(n,t,e,r){var i=n.children;if(n.x=t+=r*n.x,n.y=e+=r*n.y,n.r*=r,i)for(var u=-1,o=i.length;++u<o;)Ci(i[u],t,e,r)}function zi(n,t,e){var r=n.r+e.r,i=t.x-n.x,u=t.y-n.y;if(r&&(i||u)){var o=t.r+e.r,a=i*i+u*u;o*=o,r*=r;var l=.5+(r-o)/(2*a),c=Math.sqrt(Math.max(0,2*o*(r+a)-(r-=a)*r-o*o))/(2*a);e.x=n.x+l*i+c*u,e.y=n.y+l*u-c*i}else e.x=n.x+r,e.y=n.y}function Li(n,t){return n.parent==t.parent?1:2}function qi(n){var t=n.children;return t.length?t[0]:n.t}function Ti(n){var t,e=n.children;return(t=e.length)?e[t-1]:n.t}function Ri(n,t,e){var r=e/(t.i-n.i);t.c-=r,t.s+=e,n.c+=r,t.z+=e,t.m+=e}function Di(n){for(var t,e=0,r=0,i=n.children,u=i.length;--u>=0;)t=i[u],t.z+=e,t.m+=e,e+=t.s+(r+=t.c)}function Pi(n,t,e){return n.a.parent===t.parent?n.a:e}function Ui(n){return 1+ao.max(n,function(n){return n.y})}function ji(n){return n.reduce(function(n,t){return n+t.x},0)/n.length}function Fi(n){var t=n.children;return t&&t.length?Fi(t[0]):n}function Hi(n){var t,e=n.children;return e&&(t=e.length)?Hi(e[t-1]):n}function Oi(n){return{x:n.x,y:n.y,dx:n.dx,dy:n.dy}}function Ii(n,t){var e=n.x+t[3],r=n.y+t[0],i=n.dx-t[1]-t[3],u=n.dy-t[0]-t[2];return 0>i&&(e+=i/2,i=0),0>u&&(r+=u/2,u=0),{x:e,y:r,dx:i,dy:u}}function Yi(n){var t=n[0],e=n[n.length-1];return e>t?[t,e]:[e,t]}function Zi(n){return n.rangeExtent?n.rangeExtent():Yi(n.range())}function \
Vi(n,t,e,r){var i=e(n[0],n[1]),u=r(t[0],t[1]);return function(n){return u(i(n))}}function Xi(n,t){var e,r=0,i=n.length-1,u=n[r],o=n[i];return u>o&&(e=r,r=i,i=e,e=u,u=o,o=e),n[r]=t.floor(u),n[i]=t.ceil(o),n}function $i(n){return n?{floor:function(t){return Math.floor(t/n)*n},ceil:function(t){return Math.ceil(t/n)*n}}:Sl}function Bi(n,t,e,r){var i=[],u=[],o=0,a=Math.min(n.length,t.length)-1;for(n[a]<n[0]&&(n=n.slice().reverse(),t=t.slice().reverse());++o<=a;)i.push(e(n[o-1],n[o])),u.push(r(t[o-1],t[o]));return function(t){var e=ao.bisect(n,t,1,a)-1;return u[e](i[e](t))}}function Wi(n,t,e,r){function i(){var i=Math.min(n.length,t.length)>2?Bi:Vi,l=r?Wr:Br;return o=i(n,t,l,e),a=i(t,n,l,Mr),u}function u(n){return o(n)}var o,a;return u.invert=function(n){return a(n)},u.domain=function(t){return arguments.length?(n=t.map(Number),i()):n},u.range=function(n){return arguments.length?(t=n,i()):t},u.rangeRound=function(n){return u.range(n).interpolate(Ur)},u.clamp=function(n){return arguments.length?(r=n,i()):r},u.interpolate=function(n){return arguments.length?(e=n,i()):e},u.ticks=function(t){return Qi(n,t)},u.tickFormat=function(t,e){return nu(n,t,e)},u.nice=function(t){return Gi(n,t),i()},u.copy=function(){return Wi(n,t,e,r)},i()}function Ji(n,t){return ao.rebind(n,t,\"range\",\"rangeRound\",\"interpolate\",\"clamp\")}function Gi(n,t){return Xi(n,$i(Ki(n,t)[2])),Xi(n,$i(Ki(n,t)[2])),n}function Ki(n,t){null==t&&(t=10);var e=Yi(n),r=e[1]-e[0],i=Math.pow(10,Math.floor(Math.log(r/t)/Math.LN10)),u=t/r*i;return.15>=u?i*=10:.35>=u?i*=5:.75>=u&&(i*=2),e[0]=Math.ceil(e[0]/i)*i,e[1]=Math.floor(e[1]/i)*i+.5*i,e[2]=i,e}function Qi(n,t){return ao.range.apply(ao,Ki(n,t))}function nu(n,t,e){var r=Ki(n,t);if(e){var i=ha.exec(e);if(i.shift(),\"s\"===i[8]){var u=ao.formatPrefix(Math.max(xo(r[0]),xo(r[1])));return i[7]||(i[7]=\".\"+tu(u.scale(r[2]))),i[8]=\"f\",e=ao.format(i.join(\"\")),function(n){return e(u.scale(n))+u.symbol}}i[7]||(i[7]=\".\"+eu(i[8],r)),e=i.join(\"\")}else e=\",.\"+tu(r[2])+\"f\";return ao.format(e)}function tu(n){return-Math.floor(Math.log(n)/Math.LN10+.01)}function \
eu(n,t){var e=tu(t[2]);return n in kl?Math.abs(e-tu(Math.max(xo(t[0]),xo(t[1]))))+ +(\"e\"!==n):e-2*(\"%\"===n)}function ru(n,t,e,r){function i(n){return(e?Math.log(0>n?0:n):-Math.log(n>0?0:-n))/Math.log(t)}function u(n){return e?Math.pow(t,n):-Math.pow(t,-n)}function o(t){return n(i(t))}return o.invert=function(t){return u(n.invert(t))},o.domain=function(t){return arguments.length?(e=t[0]>=0,n.domain((r=t.map(Number)).map(i)),o):r},o.base=function(e){return arguments.length?(t=+e,n.domain(r.map(i)),o):t},o.nice=function(){var t=Xi(r.map(i),e?Math:El);return n.domain(t),r=t.map(u),o},o.ticks=function(){var n=Yi(r),o=[],a=n[0],l=n[1],c=Math.floor(i(a)),f=Math.ceil(i(l)),s=t%1?2:t;if(isFinite(f-c)){if(e){for(;f>c;c++)for(var h=1;s>h;h++)o.push(u(c)*h);o.push(u(c))}else for(o.push(u(c));c++<f;)for(var h=s-1;h>0;h--)o.push(u(c)*h);for(c=0;o[c]<a;c++);for(f=o.length;o[f-1]>l;f--);o=o.slice(c,f)}return o},o.tickFormat=function(n,e){if(!arguments.length)return Nl;arguments.length<2?e=Nl:\"function\"!=typeof e&&(e=ao.format(e));var r=Math.max(1,t*n/o.ticks().length);return function(n){var o=n/u(Math.round(i(n)));return t-.5>o*t&&(o*=t),r>=o?e(n):\"\"}},o.copy=function(){return ru(n.copy(),t,e,r)},Ji(o,n)}function iu(n,t,e){function r(t){return n(i(t))}var i=uu(t),u=uu(1/t);return r.invert=function(t){return u(n.invert(t))},r.domain=function(t){return arguments.length?(n.domain((e=t.map(Number)).map(i)),r):e},r.ticks=function(n){return Qi(e,n)},r.tickFormat=function(n,t){return nu(e,n,t)},r.nice=function(n){return r.domain(Gi(e,n))},r.exponent=function(o){return arguments.length?(i=uu(t=o),u=uu(1/t),n.domain(e.map(i)),r):t},r.copy=function(){return iu(n.copy(),t,e)},Ji(r,n)}function uu(n){return function(t){return 0>t?-Math.pow(-t,n):Math.pow(t,n)}}function ou(n,t){function e(e){return u[((i.get(e)||(\"range\"===t.t?i.set(e,n.push(e)):NaN))-1)%u.length]}function r(t,e){return ao.range(n.length).map(function(n){return t+e*n})}var i,u,o;return e.domain=function(r){if(!arguments.length)return n;n=[],i=new c;for(var u,o=-1,a=r.length;++o<a;\
)i.has(u=r[o])||i.set(u,n.push(u));return e[t.t].apply(e,t.a)},e.range=function(n){return arguments.length?(u=n,o=0,t={t:\"range\",a:arguments},e):u},e.rangePoints=function(i,a){arguments.length<2&&(a=0);var l=i[0],c=i[1],f=n.length<2?(l=(l+c)/2,0):(c-l)/(n.length-1+a);return u=r(l+f*a/2,f),o=0,t={t:\"rangePoints\",a:arguments},e},e.rangeRoundPoints=function(i,a){arguments.length<2&&(a=0);var l=i[0],c=i[1],f=n.length<2?(l=c=Math.round((l+c)/2),0):(c-l)/(n.length-1+a)|0;return u=r(l+Math.round(f*a/2+(c-l-(n.length-1+a)*f)/2),f),o=0,t={t:\"rangeRoundPoints\",a:arguments},e},e.rangeBands=function(i,a,l){arguments.length<2&&(a=0),arguments.length<3&&(l=a);var c=i[1]<i[0],f=i[c-0],s=i[1-c],h=(s-f)/(n.length-a+2*l);return u=r(f+h*l,h),c&&u.reverse(),o=h*(1-a),t={t:\"rangeBands\",a:arguments},e},e.rangeRoundBands=function(i,a,l){arguments.length<2&&(a=0),arguments.length<3&&(l=a);var c=i[1]<i[0],f=i[c-0],s=i[1-c],h=Math.floor((s-f)/(n.length-a+2*l));return u=r(f+Math.round((s-f-(n.length-a)*h)/2),h),c&&u.reverse(),o=Math.round(h*(1-a)),t={t:\"rangeRoundBands\",a:arguments},e},e.rangeBand=function(){return o},e.rangeExtent=function(){return Yi(t.a[0])},e.copy=function(){return ou(n,t)},e.domain(n)}function au(n,t){function u(){var e=0,r=t.length;for(a=[];++e<r;)a[e-1]=ao.quantile(n,e/r);return o}function o(n){return isNaN(n=+n)?void 0:t[ao.bisect(a,n)]}var a;return o.domain=function(t){return arguments.length?(n=t.map(r).filter(i).sort(e),u()):n},o.range=function(n){return arguments.length?(t=n,u()):t},o.quantiles=function(){return a},o.invertExtent=function(e){return e=t.indexOf(e),0>e?[NaN,NaN]:[e>0?a[e-1]:n[0],e<a.length?a[e]:n[n.length-1]]},o.copy=function(){return au(n,t)},u()}function lu(n,t,e){function r(t){return e[Math.max(0,Math.min(o,Math.floor(u*(t-n))))]}function i(){return u=e.length/(t-n),o=e.length-1,r}var u,o;return r.domain=function(e){return arguments.length?(n=+e[0],t=+e[e.length-1],i()):[n,t]},r.range=function(n){return arguments.length?(e=n,i()):e},r.invertExtent=function(t){return t=e.indexOf(t),t=0>t?NaN:t/u+n,\
[t,t+1/u]},r.copy=function(){return lu(n,t,e)},i()}function cu(n,t){function e(e){return e>=e?t[ao.bisect(n,e)]:void 0}return e.domain=function(t){return arguments.length?(n=t,e):n},e.range=function(n){return arguments.length?(t=n,e):t},e.invertExtent=function(e){return e=t.indexOf(e),[n[e-1],n[e]]},e.copy=function(){return cu(n,t)},e}function fu(n){function t(n){return+n}return t.invert=t,t.domain=t.range=function(e){return arguments.length?(n=e.map(t),t):n},t.ticks=function(t){return Qi(n,t)},t.tickFormat=function(t,e){return nu(n,t,e)},t.copy=function(){return fu(n)},t}function su(){return 0}function hu(n){return n.innerRadius}function pu(n){return n.outerRadius}function gu(n){return n.startAngle}function vu(n){return n.endAngle}function du(n){return n&&n.padAngle}function yu(n,t,e,r){return(n-e)*t-(t-r)*n>0?0:1}function mu(n,t,e,r,i){var u=n[0]-t[0],o=n[1]-t[1],a=(i?r:-r)/Math.sqrt(u*u+o*o),l=a*o,c=-a*u,f=n[0]+l,s=n[1]+c,h=t[0]+l,p=t[1]+c,g=(f+h)/2,v=(s+p)/2,d=h-f,y=p-s,m=d*d+y*y,M=e-r,x=f*p-h*s,b=(0>y?-1:1)*Math.sqrt(Math.max(0,M*M*m-x*x)),_=(x*y-d*b)/m,w=(-x*d-y*b)/m,S=(x*y+d*b)/m,k=(-x*d+y*b)/m,N=_-g,E=w-v,A=S-g,C=k-v;return N*N+E*E>A*A+C*C&&(_=S,w=k),[[_-l,w-c],[_*e/M,w*e/M]]}function Mu(n){function t(t){function o(){c.push(\"M\",u(n(f),a))}for(var l,c=[],f=[],s=-1,h=t.length,p=En(e),g=En(r);++s<h;)i.call(this,l=t[s],s)?f.push([+p.call(this,l,s),+g.call(this,l,s)]):f.length&&(o(),f=[]);return f.length&&o(),c.length?c.join(\"\"):null}var e=Ce,r=ze,i=zt,u=xu,o=u.key,a=.7;return t.x=function(n){return arguments.length?(e=n,t):e},t.y=function(n){return arguments.length?(r=n,t):r},t.defined=function(n){return arguments.length?(i=n,t):i},t.interpolate=function(n){return arguments.length?(o=\"function\"==typeof n?u=n:(u=Tl.get(n)||xu).key,t):o},t.tension=function(n){return arguments.length?(a=n,t):a},t}function xu(n){return n.length>1?n.join(\"L\"):n+\"Z\"}function bu(n){return n.join(\"L\")+\"Z\"}function _u(n){for(var t=0,e=n.length,r=n[0],i=[r[0],\",\",r[1]];++t<e;)i.push(\"H\",(r[0]+(r=n[t])[0])/2,\"V\",r[1]);return e>1&&i.push(\"H\",\
r[0]),i.join(\"\")}function wu(n){for(var t=0,e=n.length,r=n[0],i=[r[0],\",\",r[1]];++t<e;)i.push(\"V\",(r=n[t])[1],\"H\",r[0]);return i.join(\"\")}function Su(n){for(var t=0,e=n.length,r=n[0],i=[r[0],\",\",r[1]];++t<e;)i.push(\"H\",(r=n[t])[0],\"V\",r[1]);return i.join(\"\")}function ku(n,t){return n.length<4?xu(n):n[1]+Au(n.slice(1,-1),Cu(n,t))}function Nu(n,t){return n.length<3?bu(n):n[0]+Au((n.push(n[0]),n),Cu([n[n.length-2]].concat(n,[n[1]]),t))}function Eu(n,t){return n.length<3?xu(n):n[0]+Au(n,Cu(n,t))}function Au(n,t){if(t.length<1||n.length!=t.length&&n.length!=t.length+2)return xu(n);var e=n.length!=t.length,r=\"\",i=n[0],u=n[1],o=t[0],a=o,l=1;if(e&&(r+=\"Q\"+(u[0]-2*o[0]/3)+\",\"+(u[1]-2*o[1]/3)+\",\"+u[0]+\",\"+u[1],i=n[1],l=2),t.length>1){a=t[1],u=n[l],l++,r+=\"C\"+(i[0]+o[0])+\",\"+(i[1]+o[1])+\",\"+(u[0]-a[0])+\",\"+(u[1]-a[1])+\",\"+u[0]+\",\"+u[1];for(var c=2;c<t.length;c++,l++)u=n[l],a=t[c],r+=\"S\"+(u[0]-a[0])+\",\"+(u[1]-a[1])+\",\"+u[0]+\",\"+u[1]}if(e){var f=n[l];r+=\"Q\"+(u[0]+2*a[0]/3)+\",\"+(u[1]+2*a[1]/3)+\",\"+f[0]+\",\"+f[1]}return r}function Cu(n,t){for(var e,r=[],i=(1-t)/2,u=n[0],o=n[1],a=1,l=n.length;++a<l;)e=u,u=o,o=n[a],r.push([i*(o[0]-e[0]),i*(o[1]-e[1])]);return r}function zu(n){if(n.length<3)return xu(n);var t=1,e=n.length,r=n[0],i=r[0],u=r[1],o=[i,i,i,(r=n[1])[0]],a=[u,u,u,r[1]],l=[i,\",\",u,\"L\",Ru(Pl,o),\",\",Ru(Pl,a)];for(n.push(n[e-1]);++t<=e;)r=n[t],o.shift(),o.push(r[0]),a.shift(),a.push(r[1]),Du(l,o,a);return n.pop(),l.push(\"L\",r),l.join(\"\")}function Lu(n){if(n.length<4)return xu(n);for(var t,e=[],r=-1,i=n.length,u=[0],o=[0];++r<3;)t=n[r],u.push(t[0]),o.push(t[1]);for(e.push(Ru(Pl,u)+\",\"+Ru(Pl,o)),--r;++r<i;)t=n[r],u.shift(),u.push(t[0]),o.shift(),o.push(t[1]),Du(e,u,o);return e.join(\"\")}function qu(n){for(var t,e,r=-1,i=n.length,u=i+4,o=[],a=[];++r<4;)e=n[r%i],o.push(e[0]),a.push(e[1]);for(t=[Ru(Pl,o),\",\",Ru(Pl,a)],--r;++r<u;)e=n[r%i],o.shift(),o.push(e[0]),a.shift(),a.push(e[1]),Du(t,o,a);return t.join(\"\")}function Tu(n,t){var e=n.length-1;if(e)for(var r,i,u=n[0][0],o=n[0][1],a=n[e][0]-u,l=n[e][1]-o,c=-1;++c<=e;)r=n[c],i=c/e,r[0]=t*r[0]+(1-t)*(u+i*a),\
r[1]=t*r[1]+(1-t)*(o+i*l);return zu(n)}function Ru(n,t){return n[0]*t[0]+n[1]*t[1]+n[2]*t[2]+n[3]*t[3]}function Du(n,t,e){n.push(\"C\",Ru(Rl,t),\",\",Ru(Rl,e),\",\",Ru(Dl,t),\",\",Ru(Dl,e),\",\",Ru(Pl,t),\",\",Ru(Pl,e))}function Pu(n,t){return(t[1]-n[1])/(t[0]-n[0])}function Uu(n){for(var t=0,e=n.length-1,r=[],i=n[0],u=n[1],o=r[0]=Pu(i,u);++t<e;)r[t]=(o+(o=Pu(i=u,u=n[t+1])))/2;return r[t]=o,r}function ju(n){for(var t,e,r,i,u=[],o=Uu(n),a=-1,l=n.length-1;++a<l;)t=Pu(n[a],n[a+1]),xo(t)<Uo?o[a]=o[a+1]=0:(e=o[a]/t,r=o[a+1]/t,i=e*e+r*r,i>9&&(i=3*t/Math.sqrt(i),o[a]=i*e,o[a+1]=i*r));for(a=-1;++a<=l;)i=(n[Math.min(l,a+1)][0]-n[Math.max(0,a-1)][0])/(6*(1+o[a]*o[a])),u.push([i||0,o[a]*i||0]);return u}function Fu(n){return n.length<3?xu(n):n[0]+Au(n,ju(n))}function Hu(n){for(var t,e,r,i=-1,u=n.length;++i<u;)t=n[i],e=t[0],r=t[1]-Io,t[0]=e*Math.cos(r),t[1]=e*Math.sin(r);return n}function Ou(n){function t(t){function l(){v.push(\"M\",a(n(y),s),f,c(n(d.reverse()),s),\"Z\")}for(var h,p,g,v=[],d=[],y=[],m=-1,M=t.length,x=En(e),b=En(i),_=e===r?function(){\
return p}:En(r),w=i===u?function(){return g}:En(u);++m<M;)o.call(this,h=t[m],m)?(d.push([p=+x.call(this,h,m),g=+b.call(this,h,m)]),y.push([+_.call(this,h,m),+w.call(this,h,m)])):d.length&&(l(),d=[],y=[]);return d.length&&l(),v.length?v.join(\"\"):null}var e=Ce,r=Ce,i=0,u=ze,o=zt,a=xu,l=a.key,c=a,f=\"L\",s=.7;return t.x=function(n){return arguments.length?(e=r=n,t):r},t.x0=function(n){return arguments.length?(e=n,t):e},t.x1=function(n){return arguments.length?(r=n,t):r},t.y=function(n){return arguments.length?(i=u=n,t):u},t.y0=function(n){return arguments.length?(i=n,t):i},t.y1=function(n){return arguments.length?(u=n,t):u},t.defined=function(n){return arguments.length?(o=n,t):o},t.interpolate=function(n){return arguments.length?(l=\"function\"==typeof n?a=n:(a=Tl.get(n)||xu).key,c=a.reverse||a,f=a.closed?\"M\":\"L\",t):l},t.tension=function(n){return arguments.length?(s=n,t):s},t}function Iu(n){return n.radius}function Yu(n){return[n.x,n.y]}function Zu(n){return function(){var t=n.apply(this,arguments),e=t[0],r=t[1]-Io;return[e*Math.cos(r),e*Math.sin(r)]}}function Vu(){return 64}function Xu(){return\"circle\"}function $u(n){var t=Math.sqrt(n/Fo);return\"M0,\"+t+\"A\"+t+\",\"+t+\" 0 1,1 0,\"+-t+\"A\"+t+\",\"+t+\" 0 1,1 0,\"+t+\"Z\"}function Bu(n){return function(){var t,e,r;(t=this[n])&&(r=t[e=t.active])&&(r.timer.c=null,r.timer.t=NaN,--t.count?delete t[e]:delete this[n],t.active+=.5,r.event&&r.event.interrupt.call(this,this.__data__,r.index))}}function Wu(n,t,e){return ko(n,Yl),n.namespace=t,n.id=e,n}function Ju(n,t,e,r){var i=n.id,u=n.namespace;return Y(n,\"function\"==typeof e?function(n,o,a){n[u][i].tween.set(t,r(e.call(n,n.__data__,o,a)))}:(e=r(e),function(n){n[u][i].tween.set(t,e)}))}function Gu(n){return null==n&&(n=\"\"),function(){this.textContent=n}}function Ku(n){return null==n?\"__transition__\":\"__transition_\"+n+\"__\"}function Qu(n,t,e,r,i){function u(n){var t=v.delay;return f.t=t+l,n>=t?o(n-t):void(f.c=o)}function o(e){var i=g.active,u=g[i];u&&(u.timer.c=null,u.timer.t=NaN,--g.count,delete g[i],u.event&&u.event.interrupt.call(n,n.__data__,\
u.index));for(var o in g)if(r>+o){var c=g[o];c.timer.c=null,c.timer.t=NaN,--g.count,delete g[o]}f.c=a,qn(function(){return f.c&&a(e||1)&&(f.c=null,f.t=NaN),1},0,l),g.active=r,v.event&&v.event.start.call(n,n.__data__,t),p=[],v.tween.forEach(function(e,r){(r=r.call(n,n.__data__,t))&&p.push(r)}),h=v.ease,s=v.duration}function a(i){for(var u=i/s,o=h(u),a=p.length;a>0;)p[--a].call(n,o);return u>=1?(v.event&&v.event.end.call(n,n.__data__,t),--g.count?delete g[r]:delete n[e],1):void 0}var l,f,s,h,p,g=n[e]||(n[e]={active:0,count:0}),v=g[r];v||(l=i.time,f=qn(u,0,l),v=g[r]={tween:new c,time:l,timer:f,delay:i.delay,duration:i.duration,ease:i.ease,index:t},i=null,++g.count)}function no(n,t,e){n.attr(\"transform\",function(n){var r=t(n);return\"translate(\"+(isFinite(r)?r:e(n))+\",0)\"})}function to(n,t,e){n.attr(\"transform\",function(n){var r=t(n);return\"translate(0,\"+(isFinite(r)?r:e(n))+\")\"})}function eo(n){return n.toISOString()}function ro(n,t,e){function r(t){return n(t)}function i(n,e){var r=n[1]-n[0],i=r/e,u=ao.bisect(Kl,i);return u==Kl.length?[t.year,Ki(n.map(function(n){return n/31536e6}),e)[2]]:u?t[i/Kl[u-1]<Kl[u]/i?u-1:u]:[tc,Ki(n,e)[2]]}return r.invert=function(t){return io(n.invert(t))},r.domain=function(t){return arguments.length?(n.domain(t),r):n.domain().map(io)},r.nice=function(n,t){function e(e){return!isNaN(e)&&!n.range(e,io(+e+1),t).length}var u=r.domain(),o=Yi(u),a=null==n?i(o,10):\"number\"==typeof n&&i(o,n);return a&&(n=a[0],t=a[1]),r.domain(Xi(u,t>1?{floor:function(t){for(;e(t=n.floor(t));)t=io(t-1);return t},ceil:function(t){for(;e(t=n.ceil(t));)t=io(+t+1);return t}}:n))},r.ticks=function(n,t){var e=Yi(r.domain()),u=null==n?i(e,10):\"number\"==typeof n?i(e,n):!n.range&&[{range:n},t];return u&&(n=u[0],t=u[1]),n.range(e[0],io(+e[1]+1),1>t?1:t)},r.tickFormat=function(){return e},r.copy=function(){return ro(n.copy(),t,e)},Ji(r,n)}function io(n){return new Date(n)}function uo(n){return JSON.parse(n.responseText)}function oo(n){var t=fo.createRange();return t.selectNode(fo.body),t.createContextualFragment(n.responseText)}var \
ao={version:\"3.5.17\"},lo=[].slice,co=function(n){return lo.call(n)},fo=this.document;if(fo)try{co(fo.documentElement.childNodes)[0].nodeType}catch(so){co=function(n){for(var t=n.length,e=new Array(t);t--;)e[t]=n[t];return e}}if(Date.now||(Date.now=function(){return+new Date}),fo)try{fo.createElement(\"DIV\").style.setProperty(\"opacity\",0,\"\")}catch(ho){var po=this.Element.prototype,go=po.setAttribute,vo=po.setAttributeNS,yo=this.CSSStyleDeclaration.prototype,mo=yo.setProperty;po.setAttribute=function(n,t){go.call(this,n,t+\"\")},po.setAttributeNS=function(n,t,e){vo.call(this,n,t,e+\"\")},yo.setProperty=function(n,t,e){mo.call(this,n,t+\"\",e)}}ao.ascending=e,ao.descending=function(n,t){return n>t?-1:t>n?1:t>=n?0:NaN},ao.min=function(n,t){var e,r,i=-1,u=n.length;if(1===arguments.length){for(;++i<u;)if(null!=(r=n[i])&&r>=r){e=r;break}for(;++i<u;)null!=(r=n[i])&&e>r&&(e=r)}else{for(;++i<u;)if(null!=(r=t.call(n,n[i],i))&&r>=r){e=r;break}for(;++i<u;)null!=(r=t.call(n,n[i],i))&&e>r&&(e=r)}return e},ao.max=function(n,t){var e,r,i=-1,u=n.length;if(1===arguments.length){for(;++i<u;)if(null!=(r=n[i])&&r>=r){e=r;break}for(;++i<u;)null!=(r=n[i])&&r>e&&(e=r)}else{for(;++i<u;)if(null!=(r=t.call(n,n[i],i))&&r>=r){e=r;break}for(;++i<u;)null!=(r=t.call(n,n[i],i))&&r>e&&(e=r)}return e},ao.extent=function(n,t){var e,r,i,u=-1,o=n.length;if(1===arguments.length){for(;++u<o;)if(null!=(r=n[u])&&r>=r){e=i=r;break}for(;++u<o;)null!=(r=n[u])&&(e>r&&(e=r),r>i&&(i=r))}else{for(;++u<o;)if(null!=(r=t.call(n,n[u],u))&&r>=r){e=i=r;break}for(;++u<o;)null!=(r=t.call(n,n[u],u))&&(e>r&&(e=r),r>i&&(i=r))}return[e,i]},ao.sum=function(n,t){var e,r=0,u=n.length,o=-1;if(1===arguments.length)for(;++o<u;)i(e=+n[o])&&(r+=e);else for(;++o<u;)i(e=+t.call(n,n[o],o))&&(r+=e);return r},ao.mean=function(n,t){var e,u=0,o=n.length,a=-1,l=o;if(1===arguments.length)for(;++a<o;)i(e=r(n[a]))?u+=e:--l;else for(;++a<o;)i(e=r(t.call(n,n[a],a)))?u+=e:--l;return l?u/l:void 0},ao.quantile=function(n,t){var e=(n.length-1)*t+1,r=Math.floor(e),i=+n[r-1],u=e-r;return u?i+u*(n[r]-i):i},ao.median=function(n,\
t){var u,o=[],a=n.length,l=-1;if(1===arguments.length)for(;++l<a;)i(u=r(n[l]))&&o.push(u);else for(;++l<a;)i(u=r(t.call(n,n[l],l)))&&o.push(u);return o.length?ao.quantile(o.sort(e),.5):void 0},ao.variance=function(n,t){var e,u,o=n.length,a=0,l=0,c=-1,f=0;if(1===arguments.length)for(;++c<o;)i(e=r(n[c]))&&(u=e-a,a+=u/++f,l+=u*(e-a));else for(;++c<o;)i(e=r(t.call(n,n[c],c)))&&(u=e-a,a+=u/++f,l+=u*(e-a));return f>1?l/(f-1):void 0},ao.deviation=function(){var n=ao.variance.apply(this,arguments);return n?Math.sqrt(n):n};var Mo=u(e);ao.bisectLeft=Mo.left,ao.bisect=ao.bisectRight=Mo.right,ao.bisector=function(n){return u(1===n.length?function(t,r){return e(n(t),r)}:n)},ao.shuffle=function(n,t,e){(u=arguments.length)<3&&(e=n.length,2>u&&(t=0));for(var r,i,u=e-t;u;)i=Math.random()*u--|0,r=n[u+t],n[u+t]=n[i+t],n[i+t]=r;return n},ao.permute=function(n,t){for(var e=t.length,r=new Array(e);e--;)r[e]=n[t[e]];return r},ao.pairs=function(n){for(var t,e=0,r=n.length-1,i=n[0],u=new Array(0>r?0:r);r>e;)u[e]=[t=i,i=n[++e]];return u},ao.transpose=function(n){if(!(i=n.length))return[];for(var t=-1,e=ao.min(n,o),r=new Array(e);++t<e;)for(var i,u=-1,a=r[t]=new Array(i);++u<i;)a[u]=n[u][t];return r},ao.zip=function(){return ao.transpose(arguments)},ao.keys=function(n){var t=[];for(var e in n)t.push(e);return t},ao.values=function(n){var t=[];for(var e in n)t.push(n[e]);return t},ao.entries=function(n){var t=[];for(var e in n)t.push({key:e,value:n[e]});return t},ao.merge=function(n){for(var t,e,r,i=n.length,u=-1,o=0;++u<i;)o+=n[u].length;for(e=new Array(o);--i>=0;)for(r=n[i],t=r.length;--t>=0;)e[--o]=r[t];return e};var xo=Math.abs;ao.range=function(n,t,e){if(arguments.length<3&&(e=1,arguments.length<2&&(t=n,n=0)),(t-n)/e===1/0)throw new Error(\"infinite range\");var r,i=[],u=a(xo(e)),o=-1;if(n*=u,t*=u,e*=u,0>e)for(;(r=n+e*++o)>t;)i.push(r/u);else for(;(r=n+e*++o)<t;)i.push(r/u);return i},ao.map=function(n,t){var e=new c;if(n instanceof c)n.forEach(function(n,t){e.set(n,t)});else if(Array.isArray(n)){var r,i=-1,u=n.length;if(1===arguments.length)for(;\
++i<u;)e.set(i,n[i]);else for(;++i<u;)e.set(t.call(n,r=n[i],i),r)}else for(var o in n)e.set(o,n[o]);return e};var bo=\"__proto__\",_o=\"\\x00\";l(c,{has:h,get:function(n){return this._[f(n)]},set:function(n,t){return this._[f(n)]=t},remove:p,keys:g,values:function(){var n=[];for(var t in this._)n.push(this._[t]);return n},entries:function(){var n=[];for(var t in this._)n.push({key:s(t),value:this._[t]});return n},size:v,empty:d,forEach:function(n){for(var t in this._)n.call(this,s(t),this._[t])}}),ao.nest=function(){function n(t,o,a){if(a>=u.length)return r?r.call(i,o):e?o.sort(e):o;for(var l,f,s,h,p=-1,g=o.length,v=u[a++],d=new c;++p<g;)(h=d.get(l=v(f=o[p])))?h.push(f):d.set(l,[f]);return t?(f=t(),s=function(e,r){f.set(e,n(t,r,a))}):(f={},s=function(e,r){f[e]=n(t,r,a)}),d.forEach(s),f}function t(n,e){if(e>=u.length)return n;var r=[],i=o[e++];return n.forEach(function(n,i){r.push({key:n,values:t(i,e)})}),i?r.sort(function(n,t){return i(n.key,t.key)}):r}var e,r,i={},u=[],o=[];return i.map=function(t,e){return n(e,t,0)},i.entries=function(e){return t(n(ao.map,e,0),0)},i.key=function(n){return u.push(n),i},i.sortKeys=function(n){return o[u.length-1]=n,i},i.sortValues=function(n){return e=n,i},i.rollup=function(n){return r=n,i},i},ao.set=function(n){var t=new y;if(n)for(var e=0,r=n.length;r>e;++e)t.add(n[e]);return t},l(y,{has:h,add:function(n){return this._[f(n+=\"\")]=!0,n},remove:p,values:g,size:v,empty:d,forEach:function(n){for(var t in this._)n.call(this,s(t))}}),ao.behavior={},ao.rebind=function(n,t){for(var e,r=1,i=arguments.length;++r<i;)n[e=arguments[r]]=M(n,t,t[e]);return n};var wo=[\"webkit\",\"ms\",\"moz\",\"Moz\",\"o\",\"O\"];ao.dispatch=function(){for(var n=new _,t=-1,e=arguments.length;++t<e;)n[arguments[t]]=w(n);return n},_.prototype.on=function(n,t){var e=n.indexOf(\".\"),r=\"\";if(e>=0&&(r=n.slice(e+1),n=n.slice(0,e)),n)return arguments.length<2?this[n].on(r):this[n].on(r,t);if(2===arguments.length){if(null==t)for(n in this)this.hasOwnProperty(n)&&this[n].on(r,null);return this}},ao.event=null,ao.requote=function(n){return \
n.replace(So,\"\\\\$&\")};var So=/[\\\\\\^\\$\\*\\+\\?\\|\\[\\]\\(\\)\\.\\{\\}]/g,ko={}.__proto__?function(n,t){n.__proto__=t}:function(n,t){for(var e in t)n[e]=t[e]},No=function(n,t){return t.querySelector(n)},Eo=function(n,t){return t.querySelectorAll(n)},Ao=function(n,t){var e=n.matches||n[x(n,\"matchesSelector\")];return(Ao=function(n,t){return e.call(n,t)})(n,t)};\"function\"==typeof Sizzle&&(No=function(n,t){return Sizzle(n,t)[0]||null},Eo=Sizzle,Ao=Sizzle.matchesSelector),ao.selection=function(){return ao.select(fo.documentElement)};var Co=ao.selection.prototype=[];Co.select=function(n){var t,e,r,i,u=[];n=A(n);for(var o=-1,a=this.length;++o<a;){u.push(t=[]),t.parentNode=(r=this[o]).parentNode;for(var l=-1,c=r.length;++l<c;)(i=r[l])?(t.push(e=n.call(i,i.__data__,l,o)),e&&\"__data__\"in i&&(e.__data__=i.__data__)):t.push(null)}return E(u)},Co.selectAll=function(n){var t,e,r=[];n=C(n);for(var i=-1,u=this.length;++i<u;)for(var o=this[i],a=-1,l=o.length;++a<l;)(e=o[a])&&(r.push(t=co(n.call(e,e.__data__,a,i))),t.parentNode=e);return E(r)};var zo=\"http://www.w3.org/1999/xhtml\",Lo={svg:\"http://www.w3.org/2000/svg\",xhtml:zo,xlink:\"http://www.w3.org/1999/xlink\",xml:\"http://www.w3.org/XML/1998/namespace\",xmlns:\"http://www.w3.org/2000/xmlns/\"};ao.ns={prefix:Lo,qualify:function(n){var t=n.indexOf(\":\"),e=n;return t>=0&&\"xmlns\"!==(e=n.slice(0,t))&&(n=n.slice(t+1)),Lo.hasOwnProperty(e)?{space:Lo[e],local:n}:n}},Co.attr=function(n,t){if(arguments.length<2){if(\"string\"==typeof n){var e=this.node();return n=ao.ns.qualify(n),n.local?e.getAttributeNS(n.space,n.local):e.getAttribute(n)}for(t in n)this.each(z(t,n[t]));return this}return this.each(z(n,t))},Co.classed=function(n,t){if(arguments.length<2){if(\"string\"==typeof n){var e=this.node(),r=(n=T(n)).length,i=-1;if(t=e.classList){for(;++i<r;)if(!t.contains(n[i]))return!1}else for(t=e.getAttribute(\"class\");++i<r;)if(!q(n[i]).test(t))return!1;return!0}for(t in n)this.each(R(t,n[t]));return this}return this.each(R(n,t))},Co.style=function(n,e,r){var i=arguments.length;if(3>i){if(\"string\"!=typeof n){2>i&&(e=\"\");\
")

file(APPEND "${METABENCH_DIR}/d3.js" "\
for(r in n)this.each(P(r,n[r],e));return this}if(2>i){var u=this.node();return t(u).getComputedStyle(u,null).getPropertyValue(n)}r=\"\"}return this.each(P(n,e,r))},Co.property=function(n,t){if(arguments.length<2){if(\"string\"==typeof n)return this.node()[n];for(t in n)this.each(U(t,n[t]));return this}return this.each(U(n,t))},Co.text=function(n){return arguments.length?this.each(\"function\"==typeof n?function(){var t=n.apply(this,arguments);this.textContent=null==t?\"\":t}:null==n?function(){this.textContent=\"\"}:function(){this.textContent=n}):this.node().textContent},Co.html=function(n){return arguments.length?this.each(\"function\"==typeof n?function(){var t=n.apply(this,arguments);this.innerHTML=null==t?\"\":t}:null==n?function(){this.innerHTML=\"\"}:function(){this.innerHTML=n}):this.node().innerHTML},Co.append=function(n){return n=j(n),this.select(function(){return this.appendChild(n.apply(this,arguments))})},Co.insert=function(n,t){return n=j(n),t=A(t),this.select(function(){return this.insertBefore(n.apply(this,arguments),t.apply(this,arguments)||null)})},Co.remove=function(){return this.each(F)},Co.data=function(n,t){function e(n,e){var r,i,u,o=n.length,s=e.length,h=Math.min(o,s),p=new Array(s),g=new Array(s),v=new Array(o);if(t){var d,y=new c,m=new Array(o);for(r=-1;++r<o;)(i=n[r])&&(y.has(d=t.call(i,i.__data__,r))?v[r]=i:y.set(d,i),m[r]=d);for(r=-1;++r<s;)(i=y.get(d=t.call(e,u=e[r],r)))?i!==!0&&(p[r]=i,i.__data__=u):g[r]=H(u),y.set(d,!0);for(r=-1;++r<o;)r in m&&y.get(m[r])!==!0&&(v[r]=n[r])}else{for(r=-1;++r<h;)i=n[r],u=e[r],i?(i.__data__=u,p[r]=i):g[r]=H(u);for(;s>r;++r)g[r]=H(e[r]);for(;o>r;++r)v[r]=n[r]}g.update=p,g.parentNode=p.parentNode=v.parentNode=n.parentNode,a.push(g),l.push(p),f.push(v)}var r,i,u=-1,o=this.length;if(!arguments.length){for(n=new Array(o=(r=this[0]).length);++u<o;)(i=r[u])&&(n[u]=i.__data__);return n}var a=Z([]),l=E([]),f=E([]);if(\"function\"==typeof n)for(;++u<o;)e(r=this[u],n.call(r,r.parentNode.__data__,u));else for(;++u<o;)e(r=this[u],n);return l.enter=function(){return a},l.exit=function(){return \
f},l},Co.datum=function(n){return arguments.length?this.property(\"__data__\",n):this.property(\"__data__\")},Co.filter=function(n){var t,e,r,i=[];\"function\"!=typeof n&&(n=O(n));for(var u=0,o=this.length;o>u;u++){i.push(t=[]),t.parentNode=(e=this[u]).parentNode;for(var a=0,l=e.length;l>a;a++)(r=e[a])&&n.call(r,r.__data__,a,u)&&t.push(r)}return E(i)},Co.order=function(){for(var n=-1,t=this.length;++n<t;)for(var e,r=this[n],i=r.length-1,u=r[i];--i>=0;)(e=r[i])&&(u&&u!==e.nextSibling&&u.parentNode.insertBefore(e,u),u=e);return this},Co.sort=function(n){n=I.apply(this,arguments);for(var t=-1,e=this.length;++t<e;)this[t].sort(n);return this.order()},Co.each=function(n){return Y(this,function(t,e,r){n.call(t,t.__data__,e,r)})},Co.call=function(n){var t=co(arguments);return n.apply(t[0]=this,t),this},Co.empty=function(){return!this.node()},Co.node=function(){for(var n=0,t=this.length;t>n;n++)for(var e=this[n],r=0,i=e.length;i>r;r++){var u=e[r];if(u)return u}return null},Co.size=function(){var n=0;return Y(this,function(){++n}),n};var qo=[];ao.selection.enter=Z,ao.selection.enter.prototype=qo,qo.append=Co.append,qo.empty=Co.empty,qo.node=Co.node,qo.call=Co.call,qo.size=Co.size,qo.select=function(n){for(var t,e,r,i,u,o=[],a=-1,l=this.length;++a<l;){r=(i=this[a]).update,o.push(t=[]),t.parentNode=i.parentNode;for(var c=-1,f=i.length;++c<f;)(u=i[c])?(t.push(r[c]=e=n.call(i.parentNode,u.__data__,c,a)),e.__data__=u.__data__):t.push(null)}return E(o)},qo.insert=function(n,t){return arguments.length<2&&(t=V(this)),Co.insert.call(this,n,t)},ao.select=function(t){var e;return\"string\"==typeof t?(e=[No(t,fo)],e.parentNode=fo.documentElement):(e=[t],e.parentNode=n(t)),E([e])},ao.selectAll=function(n){var t;return\"string\"==typeof n?(t=co(Eo(n,fo)),t.parentNode=fo.documentElement):(t=co(n),t.parentNode=null),E([t])},Co.on=function(n,t,e){var r=arguments.length;if(3>r){if(\"string\"!=typeof n){2>r&&(t=!1);for(e in n)this.each(X(e,n[e],t));return this}if(2>r)return(r=this.node()[\"__on\"+n])&&r._;e=!1}return this.each(X(n,t,e))};var To=ao.map({mouseenter:\"mouseover\",\
mouseleave:\"mouseout\"});fo&&To.forEach(function(n){\"on\"+n in fo&&To.remove(n)});var Ro,Do=0;ao.mouse=function(n){return J(n,k())};var Po=this.navigator&&/WebKit/.test(this.navigator.userAgent)?-1:0;ao.touch=function(n,t,e){if(arguments.length<3&&(e=t,t=k().changedTouches),t)for(var r,i=0,u=t.length;u>i;++i)if((r=t[i]).identifier===e)return J(n,r)},ao.behavior.drag=function(){function n(){this.on(\"mousedown.drag\",u).on(\"touchstart.drag\",o)}function e(n,t,e,u,o){return function(){function a(){var n,e,r=t(h,v);r&&(n=r[0]-M[0],e=r[1]-M[1],g|=n|e,M=r,p({type:\"drag\",x:r[0]+c[0],y:r[1]+c[1],dx:n,dy:e}))}function l(){t(h,v)&&(y.on(u+d,null).on(o+d,null),m(g),p({type:\"dragend\"}))}var c,f=this,s=ao.event.target.correspondingElement||ao.event.target,h=f.parentNode,p=r.of(f,arguments),g=0,v=n(),d=\".drag\"+(null==v?\"\":\"-\"+v),y=ao.select(e(s)).on(u+d,a).on(o+d,l),m=W(s),M=t(h,v);i?(c=i.apply(f,arguments),c=[c.x-M[0],c.y-M[1]]):c=[0,0],p({type:\"dragstart\"})}}var r=N(n,\"drag\",\"dragstart\",\"dragend\"),i=null,u=e(b,ao.mouse,t,\"mousemove\",\"mouseup\"),o=e(G,ao.touch,m,\"touchmove\",\"touchend\");return n.origin=function(t){return arguments.length?(i=t,n):i},ao.rebind(n,r,\"on\")},ao.touches=function(n,t){return arguments.length<2&&(t=k().touches),t?co(t).map(function(t){var e=J(n,t);return e.identifier=t.identifier,e}):[]};var Uo=1e-6,jo=Uo*Uo,Fo=Math.PI,Ho=2*Fo,Oo=Ho-Uo,Io=Fo/2,Yo=Fo/180,Zo=180/Fo,Vo=Math.SQRT2,Xo=2,$o=4;ao.interpolateZoom=function(n,t){var e,r,i=n[0],u=n[1],o=n[2],a=t[0],l=t[1],c=t[2],f=a-i,s=l-u,h=f*f+s*s;if(jo>h)r=Math.log(c/o)/Vo,e=function(n){return[i+n*f,u+n*s,o*Math.exp(Vo*n*r)]};else{var p=Math.sqrt(h),g=(c*c-o*o+$o*h)/(2*o*Xo*p),v=(c*c-o*o-$o*h)/(2*c*Xo*p),d=Math.log(Math.sqrt(g*g+1)-g),y=Math.log(Math.sqrt(v*v+1)-v);r=(y-d)/Vo,e=function(n){var t=n*r,e=rn(d),a=o/(Xo*p)*(e*un(Vo*t+d)-en(d));return[i+a*f,u+a*s,o*e/rn(Vo*t+d)]}}return e.duration=1e3*r,e},ao.behavior.zoom=function(){function n(n){n.on(L,s).on(Wo+\".zoom\",p).on(\"dblclick.zoom\",g).on(R,h)}function e(n){return[(n[0]-k.x)/k.k,(n[1]-k.y)/k.k]}function r(n){return[n[0]*k.k+k.x,\
n[1]*k.k+k.y]}function i(n){k.k=Math.max(A[0],Math.min(A[1],n))}function u(n,t){t=r(t),k.x+=n[0]-t[0],k.y+=n[1]-t[1]}function o(t,e,r,o){t.__chart__={x:k.x,y:k.y,k:k.k},i(Math.pow(2,o)),u(d=e,r),t=ao.select(t),C>0&&(t=t.transition().duration(C)),t.call(n.event)}function a(){b&&b.domain(x.range().map(function(n){return(n-k.x)/k.k}).map(x.invert)),w&&w.domain(_.range().map(function(n){return(n-k.y)/k.k}).map(_.invert))}function l(n){z++||n({type:\"zoomstart\"})}function c(n){a(),n({type:\"zoom\",scale:k.k,translate:[k.x,k.y]})}function f(n){--z||(n({type:\"zoomend\"}),d=null)}function s(){function n(){a=1,u(ao.mouse(i),h),c(o)}function r(){s.on(q,null).on(T,null),p(a),f(o)}var i=this,o=D.of(i,arguments),a=0,s=ao.select(t(i)).on(q,n).on(T,r),h=e(ao.mouse(i)),p=W(i);Il.call(i),l(o)}function h(){function n(){var n=ao.touches(g);return p=k.k,n.forEach(function(n){n.identifier in d&&(d[n.identifier]=e(n))}),n}function t(){var t=ao.event.target;ao.select(t).on(x,r).on(b,a),_.push(t);for(var e=ao.event.changedTouches,i=0,u=e.length;u>i;++i)d[e[i].identifier]=null;var l=n(),c=Date.now();if(1===l.length){if(500>c-M){var f=l[0];o(g,f,d[f.identifier],Math.floor(Math.log(k.k)/Math.LN2)+1),S()}M=c}else if(l.length>1){var f=l[0],s=l[1],h=f[0]-s[0],p=f[1]-s[1];y=h*h+p*p}}function r(){var n,t,e,r,o=ao.touches(g);Il.call(g);for(var a=0,l=o.length;l>a;++a,r=null)if(e=o[a],r=d[e.identifier]){if(t)break;n=e,t=r}if(r){var f=(f=e[0]-n[0])*f+(f=e[1]-n[1])*f,s=y&&Math.sqrt(f/y);n=[(n[0]+e[0])/2,(n[1]+e[1])/2],t=[(t[0]+r[0])/2,(t[1]+r[1])/2],i(s*p)}M=null,u(n,t),c(v)}function a(){if(ao.event.touches.length){for(var t=ao.event.changedTouches,e=0,r=t.length;r>e;++e)delete d[t[e].identifier];for(var i in d)return void n()}ao.selectAll(_).on(m,null),w.on(L,s).on(R,h),N(),f(v)}var p,g=this,v=D.of(g,arguments),d={},y=0,m=\".zoom-\"+ao.event.changedTouches[0].identifier,x=\"touchmove\"+m,b=\"touchend\"+m,_=[],w=ao.select(g),N=W(g);t(),l(v),w.on(L,null).on(R,t)}function p(){var n=D.of(this,arguments);m?clearTimeout(m):(Il.call(this),v=e(d=y||ao.mouse(this)),l(n)),\
m=setTimeout(function(){m=null,f(n)},50),S(),i(Math.pow(2,.002*Bo())*k.k),u(d,v),c(n)}function g(){var n=ao.mouse(this),t=Math.log(k.k)/Math.LN2;o(this,n,e(n),ao.event.shiftKey?Math.ceil(t)-1:Math.floor(t)+1)}var v,d,y,m,M,x,b,_,w,k={x:0,y:0,k:1},E=[960,500],A=Jo,C=250,z=0,L=\"mousedown.zoom\",q=\"mousemove.zoom\",T=\"mouseup.zoom\",R=\"touchstart.zoom\",D=N(n,\"zoomstart\",\"zoom\",\"zoomend\");return Wo||(Wo=\"onwheel\"in fo?(Bo=function(){return-ao.event.deltaY*(ao.event.deltaMode?120:1)},\"wheel\"):\"onmousewheel\"in fo?(Bo=function(){return ao.event.wheelDelta},\"mousewheel\"):(Bo=function(){return-ao.event.detail},\"MozMousePixelScroll\")),n.event=function(n){n.each(function(){var n=D.of(this,arguments),t=k;Hl?ao.select(this).transition().each(\"start.zoom\",function(){k=this.__chart__||{x:0,y:0,k:1},l(n)}).tween(\"zoom:zoom\",function(){var e=E[0],r=E[1],i=d?d[0]:e/2,u=d?d[1]:r/2,o=ao.interpolateZoom([(i-k.x)/k.k,(u-k.y)/k.k,e/k.k],[(i-t.x)/t.k,(u-t.y)/t.k,e/t.k]);return function(t){var r=o(t),a=e/r[2];this.__chart__=k={x:i-r[0]*a,y:u-r[1]*a,k:a},c(n)}}).each(\"interrupt.zoom\",function(){f(n)}).each(\"end.zoom\",function(){f(n)}):(this.__chart__=k,l(n),c(n),f(n))})},n.translate=function(t){return arguments.length?(k={x:+t[0],y:+t[1],k:k.k},a(),n):[k.x,k.y]},n.scale=function(t){return arguments.length?(k={x:k.x,y:k.y,k:null},i(+t),a(),n):k.k},n.scaleExtent=function(t){return arguments.length?(A=null==t?Jo:[+t[0],+t[1]],n):A},n.center=function(t){return arguments.length?(y=t&&[+t[0],+t[1]],n):y},n.size=function(t){return arguments.length?(E=t&&[+t[0],+t[1]],n):E},n.duration=function(t){return arguments.length?(C=+t,n):C},n.x=function(t){return arguments.length?(b=t,x=t.copy(),k={x:0,y:0,k:1},n):b},n.y=function(t){return arguments.length?(w=t,_=t.copy(),k={x:0,y:0,k:1},n):w},ao.rebind(n,D,\"on\")};var Bo,Wo,Jo=[0,1/0];ao.color=an,an.prototype.toString=function(){return this.rgb()+\"\"},ao.hsl=ln;var Go=ln.prototype=new an;Go.brighter=function(n){return n=Math.pow(.7,arguments.length?n:1),new ln(this.h,this.s,this.l/n)},Go.darker=function(n){return \
n=Math.pow(.7,arguments.length?n:1),new ln(this.h,this.s,n*this.l)},Go.rgb=function(){return cn(this.h,this.s,this.l)},ao.hcl=fn;var Ko=fn.prototype=new an;Ko.brighter=function(n){return new fn(this.h,this.c,Math.min(100,this.l+Qo*(arguments.length?n:1)))},Ko.darker=function(n){return new fn(this.h,this.c,Math.max(0,this.l-Qo*(arguments.length?n:1)))},Ko.rgb=function(){return sn(this.h,this.c,this.l).rgb()},ao.lab=hn;var Qo=18,na=.95047,ta=1,ea=1.08883,ra=hn.prototype=new an;ra.brighter=function(n){return new hn(Math.min(100,this.l+Qo*(arguments.length?n:1)),this.a,this.b)},ra.darker=function(n){return new hn(Math.max(0,this.l-Qo*(arguments.length?n:1)),this.a,this.b)},ra.rgb=function(){return pn(this.l,this.a,this.b)},ao.rgb=mn;var ia=mn.prototype=new an;ia.brighter=function(n){n=Math.pow(.7,arguments.length?n:1);var t=this.r,e=this.g,r=this.b,i=30;return t||e||r?(t&&i>t&&(t=i),e&&i>e&&(e=i),r&&i>r&&(r=i),new mn(Math.min(255,t/n),Math.min(255,e/n),Math.min(255,r/n))):new mn(i,i,i)},ia.darker=function(n){return n=Math.pow(.7,arguments.length?n:1),new mn(n*this.r,n*this.g,n*this.b)},ia.hsl=function(){return wn(this.r,this.g,this.b)},ia.toString=function(){return\"#\"+bn(this.r)+bn(this.g)+bn(this.b)};var ua=ao.map({aliceblue:15792383,antiquewhite:16444375,aqua:65535,aquamarine:8388564,azure:15794175,beige:16119260,bisque:16770244,black:0,blanchedalmond:16772045,blue:255,blueviolet:9055202,brown:10824234,burlywood:14596231,cadetblue:6266528,chartreuse:8388352,chocolate:13789470,coral:16744272,cornflowerblue:6591981,cornsilk:16775388,crimson:14423100,cyan:65535,darkblue:139,darkcyan:35723,darkgoldenrod:12092939,darkgray:11119017,darkgreen:25600,darkgrey:11119017,darkkhaki:12433259,darkmagenta:9109643,darkolivegreen:5597999,darkorange:16747520,darkorchid:10040012,darkred:9109504,darksalmon:15308410,darkseagreen:9419919,darkslateblue:4734347,darkslategray:3100495,darkslategrey:3100495,darkturquoise:52945,darkviolet:9699539,deeppink:16716947,deepskyblue:49151,dimgray:6908265,dimgrey:6908265,dodgerblue:2003199,firebrick:11674146,\
floralwhite:16775920,forestgreen:2263842,fuchsia:16711935,gainsboro:14474460,ghostwhite:16316671,gold:16766720,goldenrod:14329120,gray:8421504,green:32768,greenyellow:11403055,grey:8421504,honeydew:15794160,hotpink:16738740,indianred:13458524,indigo:4915330,ivory:16777200,khaki:15787660,lavender:15132410,lavenderblush:16773365,lawngreen:8190976,lemonchiffon:16775885,lightblue:11393254,lightcoral:15761536,lightcyan:14745599,lightgoldenrodyellow:16448210,lightgray:13882323,lightgreen:9498256,lightgrey:13882323,lightpink:16758465,lightsalmon:16752762,lightseagreen:2142890,lightskyblue:8900346,lightslategray:7833753,lightslategrey:7833753,lightsteelblue:11584734,lightyellow:16777184,lime:65280,limegreen:3329330,linen:16445670,magenta:16711935,maroon:8388608,mediumaquamarine:6737322,mediumblue:205,mediumorchid:12211667,mediumpurple:9662683,mediumseagreen:3978097,mediumslateblue:8087790,mediumspringgreen:64154,mediumturquoise:4772300,mediumvioletred:13047173,midnightblue:1644912,mintcream:16121850,mistyrose:16770273,moccasin:16770229,navajowhite:16768685,navy:128,oldlace:16643558,olive:8421376,olivedrab:7048739,orange:16753920,orangered:16729344,orchid:14315734,palegoldenrod:15657130,palegreen:10025880,paleturquoise:11529966,palevioletred:14381203,papayawhip:16773077,peachpuff:16767673,peru:13468991,pink:16761035,plum:14524637,powderblue:11591910,purple:8388736,rebeccapurple:6697881,red:16711680,rosybrown:12357519,royalblue:4286945,saddlebrown:9127187,salmon:16416882,sandybrown:16032864,seagreen:3050327,seashell:16774638,sienna:10506797,silver:12632256,skyblue:8900331,slateblue:6970061,slategray:7372944,slategrey:7372944,snow:16775930,springgreen:65407,steelblue:4620980,tan:13808780,teal:32896,thistle:14204888,tomato:16737095,turquoise:4251856,violet:15631086,wheat:16113331,white:16777215,whitesmoke:16119285,yellow:16776960,yellowgreen:10145074});ua.forEach(function(n,t){ua.set(n,Mn(t))}),ao.functor=En,ao.xhr=An(m),ao.dsv=function(n,t){function e(n,e,u){arguments.length<3&&(u=e,e=null);var o=Cn(n,t,null==e?r:i(e),u);return \
o.row=function(n){return arguments.length?o.response(null==(e=n)?r:i(n)):e},o}function r(n){return e.parse(n.responseText)}function i(n){return function(t){return e.parse(t.responseText,n)}}function u(t){return t.map(o).join(n)}function o(n){return a.test(n)?'\"'+n.replace(/\\\"/g,'\"\"')+'\"':n}var a=new RegExp('[\"'+n+\"\\n]\"),l=n.charCodeAt(0);return e.parse=function(n,t){var r;return e.parseRows(n,function(n,e){if(r)return r(n,e-1);var i=new Function(\"d\",\"return {\"+n.map(function(n,t){return JSON.stringify(n)+\": d[\"+t+\"]\"}).join(\",\")+\"}\");r=t?function(n,e){return t(i(n),e)}:i})},e.parseRows=function(n,t){function e(){if(f>=c)return o;if(i)return i=!1,u;var t=f;if(34===n.charCodeAt(t)){for(var e=t;e++<c;)if(34===n.charCodeAt(e)){if(34!==n.charCodeAt(e+1))break;++e}f=e+2;var r=n.charCodeAt(e+1);return 13===r?(i=!0,10===n.charCodeAt(e+2)&&++f):10===r&&(i=!0),n.slice(t+1,e).replace(/\"\"/g,'\"')}for(;c>f;){var r=n.charCodeAt(f++),a=1;if(10===r)i=!0;else if(13===r)i=!0,10===n.charCodeAt(f)&&(++f,++a);else if(r!==l)continue;return n.slice(t,f-a)}return n.slice(t)}for(var r,i,u={},o={},a=[],c=n.length,f=0,s=0;(r=e())!==o;){for(var h=[];r!==u&&r!==o;)h.push(r),r=e();t&&null==(h=t(h,s++))||a.push(h)}return a},e.format=function(t){if(Array.isArray(t[0]))return e.formatRows(t);var r=new y,i=[];return t.forEach(function(n){for(var t in n)r.has(t)||i.push(r.add(t))}),[i.map(o).join(n)].concat(t.map(function(t){return i.map(function(n){return o(t[n])}).join(n)})).join(\"\\n\")},e.formatRows=function(n){return n.map(u).join(\"\\n\")},e},ao.csv=ao.dsv(\",\",\"text/csv\"),ao.tsv=ao.dsv(\"	\",\"text/tab-separated-values\");var oa,aa,la,ca,fa=this[x(this,\"requestAnimationFrame\")]||function(n){setTimeout(n,17)};ao.timer=function(){qn.apply(this,arguments)},ao.timer.flush=function(){Rn(),Dn()},ao.round=function(n,t){return t?Math.round(n*(t=Math.pow(10,t)))/t:Math.round(n)};var sa=[\"y\",\"z\",\"a\",\"f\",\"p\",\"n\",\"\\xb5\",\"m\",\"\",\"k\",\"M\",\"G\",\"T\",\"P\",\"E\",\"Z\",\"Y\"].map(Un);ao.formatPrefix=function(n,t){var e=0;return(n=+n)&&(0>n&&(n*=-1),t&&(n=ao.round(n,Pn(n,t))),e=1+Math.floor(1e-12+Math.log(n)/Math.LN10),\
e=Math.max(-24,Math.min(24,3*Math.floor((e-1)/3)))),sa[8+e/3]};var ha=/(?:([^{])?([<>=^]))?([+\\- ])?([$#])?(0)?(\\d+)?(,)?(\\.-?\\d+)?([a-z%])?/i,pa=ao.map({b:function(n){return n.toString(2)},c:function(n){return String.fromCharCode(n)},o:function(n){return n.toString(8)},x:function(n){return n.toString(16)},X:function(n){return n.toString(16).toUpperCase()},g:function(n,t){return n.toPrecision(t)},e:function(n,t){return n.toExponential(t)},f:function(n,t){return n.toFixed(t)},r:function(n,t){return(n=ao.round(n,Pn(n,t))).toFixed(Math.max(0,Math.min(20,Pn(n*(1+1e-15),t))))}}),ga=ao.time={},va=Date;Hn.prototype={getDate:function(){return this._.getUTCDate()},getDay:function(){return this._.getUTCDay()},getFullYear:function(){return this._.getUTCFullYear()},getHours:function(){return this._.getUTCHours()},getMilliseconds:function(){return this._.getUTCMilliseconds()},getMinutes:function(){return this._.getUTCMinutes()},getMonth:function(){return this._.getUTCMonth()},getSeconds:function(){return this._.getUTCSeconds()},getTime:function(){return this._.getTime()},getTimezoneOffset:function(){return 0},valueOf:function(){return this._.valueOf()},setDate:function(){da.setUTCDate.apply(this._,arguments)},setDay:function(){da.setUTCDay.apply(this._,arguments)},setFullYear:function(){da.setUTCFullYear.apply(this._,arguments)},setHours:function(){da.setUTCHours.apply(this._,arguments)},setMilliseconds:function(){da.setUTCMilliseconds.apply(this._,arguments)},setMinutes:function(){da.setUTCMinutes.apply(this._,arguments)},setMonth:function(){da.setUTCMonth.apply(this._,arguments)},setSeconds:function(){da.setUTCSeconds.apply(this._,arguments)},setTime:function(){da.setTime.apply(this._,arguments)}};var da=Date.prototype;ga.year=On(function(n){return n=ga.day(n),n.setMonth(0,1),n},function(n,t){n.setFullYear(n.getFullYear()+t)},function(n){return n.getFullYear()}),ga.years=ga.year.range,ga.years.utc=ga.year.utc.range,ga.day=On(function(n){var t=new va(2e3,0);return t.setFullYear(n.getFullYear(),n.getMonth(),n.getDate()),t},function(n,\
t){n.setDate(n.getDate()+t)},function(n){return n.getDate()-1}),ga.days=ga.day.range,ga.days.utc=ga.day.utc.range,ga.dayOfYear=function(n){var t=ga.year(n);return Math.floor((n-t-6e4*(n.getTimezoneOffset()-t.getTimezoneOffset()))/864e5)},[\"sunday\",\"monday\",\"tuesday\",\"wednesday\",\"thursday\",\"friday\",\"saturday\"].forEach(function(n,t){t=7-t;var e=ga[n]=On(function(n){return(n=ga.day(n)).setDate(n.getDate()-(n.getDay()+t)%7),n},function(n,t){n.setDate(n.getDate()+7*Math.floor(t))},function(n){var e=ga.year(n).getDay();return Math.floor((ga.dayOfYear(n)+(e+t)%7)/7)-(e!==t)});ga[n+\"s\"]=e.range,ga[n+\"s\"].utc=e.utc.range,ga[n+\"OfYear\"]=function(n){var e=ga.year(n).getDay();return Math.floor((ga.dayOfYear(n)+(e+t)%7)/7)}}),ga.week=ga.sunday,ga.weeks=ga.sunday.range,ga.weeks.utc=ga.sunday.utc.range,ga.weekOfYear=ga.sundayOfYear;var ya={\"-\":\"\",_:\" \",0:\"0\"},ma=/^\\s*\\d+/,Ma=/^%/;ao.locale=function(n){return{numberFormat:jn(n),timeFormat:Yn(n)}};var xa=ao.locale({decimal:\".\",thousands:\",\",grouping:[3],currency:[\"$\",\"\"],dateTime:\"%a %b %e %X %Y\",date:\"%m/%d/%Y\",time:\"%H:%M:%S\",periods:[\"AM\",\"PM\"],days:[\"Sunday\",\"Monday\",\"Tuesday\",\"Wednesday\",\"Thursday\",\"Friday\",\"Saturday\"],\
shortDays:[\"Sun\",\"Mon\",\"Tue\",\"Wed\",\"Thu\",\"Fri\",\"Sat\"],months:[\"January\",\"February\",\"March\",\"April\",\"May\",\"June\",\"July\",\"August\",\"September\",\"October\",\"November\",\"December\"],shortMonths:[\"Jan\",\"Feb\",\"Mar\",\"Apr\",\"May\",\"Jun\",\"Jul\",\"Aug\",\"Sep\",\"Oct\",\"Nov\",\"Dec\"]});ao.format=xa.numberFormat,ao.geo={},ft.prototype={s:0,t:0,add:function(n){st(n,this.t,ba),st(ba.s,this.s,this),this.s?this.t+=ba.t:this.s=ba.t},reset:function(){this.s=this.t=0},valueOf:function(){return this.s}};var ba=new ft;ao.geo.stream=function(n,t){n&&_a.hasOwnProperty(n.type)?_a[n.type](n,t):ht(n,t)};var _a={Feature:function(n,t){ht(n.geometry,t)},FeatureCollection:function(n,t){for(var e=n.features,r=-1,i=e.length;++r<i;)ht(e[r].geometry,t)}},wa={Sphere:function(n,t){t.sphere()},Point:function(n,t){n=n.coordinates,t.point(n[0],n[1],n[2])},MultiPoint:function(n,t){for(var e=n.coordinates,r=-1,i=e.length;++r<i;)n=e[r],t.point(n[0],n[1],n[2])},LineString:function(n,t){pt(n.coordinates,t,0)},MultiLineString:function(n,t){for(var e=n.coordinates,r=-1,i=e.length;++r<i;)pt(e[r],t,0)},Polygon:function(n,t){gt(n.coordinates,t)},MultiPolygon:function(n,t){for(var e=n.coordinates,r=-1,i=e.length;++r<i;)gt(e[r],t)},GeometryCollection:function(n,t){for(var e=n.geometries,r=-1,i=e.length;++r<i;)ht(e[r],t)}};ao.geo.area=function(n){return Sa=0,ao.geo.stream(n,Na),Sa};var Sa,ka=new ft,Na={sphere:function(){Sa+=4*Fo},point:b,lineStart:b,lineEnd:b,polygonStart:function(){ka.reset(),Na.lineStart=vt},polygonEnd:function(){var n=2*ka;Sa+=0>n?4*Fo+n:n,Na.lineStart=Na.lineEnd=Na.point=b}};ao.geo.bounds=function(){function n(n,t){M.push(x=[f=n,h=n]),s>t&&(s=t),t>p&&(p=t)}function t(t,e){var r=dt([t*Yo,e*Yo]);if(y){var i=mt(y,r),u=[i[1],-i[0],0],o=mt(u,i);bt(o),o=_t(o);var l=t-g,c=l>0?1:-1,v=o[0]*Zo*c,d=xo(l)>180;if(d^(v>c*g&&c*t>v)){var m=o[1]*Zo;m>p&&(p=m)}else if(v=(v+360)%360-180,d^(v>c*g&&c*t>v)){var m=-o[1]*Zo;s>m&&(s=m)}else s>e&&(s=e),e>p&&(p=e);d?g>t?a(f,t)>a(f,h)&&(h=t):a(t,h)>a(f,h)&&(f=t):h>=f?(f>t&&(f=t),t>h&&(h=t)):t>g?a(f,t)>a(f,h)&&(h=t):a(t,h)>a(f,h)&&(f=t)}else \
n(t,e);y=r,g=t}function e(){b.point=t}function r(){x[0]=f,x[1]=h,b.point=n,y=null}function i(n,e){if(y){var r=n-g;m+=xo(r)>180?r+(r>0?360:-360):r}else v=n,d=e;Na.point(n,e),t(n,e)}function u(){Na.lineStart()}function o(){i(v,d),Na.lineEnd(),xo(m)>Uo&&(f=-(h=180)),x[0]=f,x[1]=h,y=null}function a(n,t){return(t-=n)<0?t+360:t}function l(n,t){return n[0]-t[0]}function c(n,t){return t[0]<=t[1]?t[0]<=n&&n<=t[1]:n<t[0]||t[1]<n}var f,s,h,p,g,v,d,y,m,M,x,b={point:n,lineStart:e,lineEnd:r,polygonStart:function(){b.point=i,b.lineStart=u,b.lineEnd=o,m=0,Na.polygonStart()},polygonEnd:function(){Na.polygonEnd(),b.point=n,b.lineStart=e,b.lineEnd=r,0>ka?(f=-(h=180),s=-(p=90)):m>Uo?p=90:-Uo>m&&(s=-90),x[0]=f,x[1]=h}};return function(n){p=h=-(f=s=1/0),M=[],ao.geo.stream(n,b);var t=M.length;if(t){M.sort(l);for(var e,r=1,i=M[0],u=[i];t>r;++r)e=M[r],c(e[0],i)||c(e[1],i)?(a(i[0],e[1])>a(i[0],i[1])&&(i[1]=e[1]),a(e[0],i[1])>a(i[0],i[1])&&(i[0]=e[0])):u.push(i=e);for(var o,e,g=-(1/0),t=u.length-1,r=0,i=u[t];t>=r;i=e,++r)e=u[r],(o=a(i[1],e[0]))>g&&(g=o,f=e[0],h=i[1])}return M=x=null,f===1/0||s===1/0?[[NaN,NaN],[NaN,NaN]]:[[f,s],[h,p]]}}(),ao.geo.centroid=function(n){Ea=Aa=Ca=za=La=qa=Ta=Ra=Da=Pa=Ua=0,ao.geo.stream(n,ja);var t=Da,e=Pa,r=Ua,i=t*t+e*e+r*r;return jo>i&&(t=qa,e=Ta,r=Ra,Uo>Aa&&(t=Ca,e=za,r=La),i=t*t+e*e+r*r,jo>i)?[NaN,NaN]:[Math.atan2(e,t)*Zo,tn(r/Math.sqrt(i))*Zo]};var Ea,Aa,Ca,za,La,qa,Ta,Ra,Da,Pa,Ua,ja={sphere:b,point:St,lineStart:Nt,lineEnd:Et,polygonStart:function(){ja.lineStart=At},polygonEnd:function(){ja.lineStart=Nt}},Fa=Rt(zt,jt,Ht,[-Fo,-Fo/2]),Ha=1e9;ao.geo.clipExtent=function(){var n,t,e,r,i,u,o={stream:function(n){return i&&(i.valid=!1),i=u(n),i.valid=!0,i},extent:function(a){return arguments.length?(u=Zt(n=+a[0][0],t=+a[0][1],e=+a[1][0],r=+a[1][1]),i&&(i.valid=!1,i=null),o):[[n,t],[e,r]]}};return o.extent([[0,0],[960,500]])},(ao.geo.conicEqualArea=function(){return Vt(Xt)}).raw=Xt,ao.geo.albers=function(){return ao.geo.conicEqualArea().rotate([96,0]).center([-.6,38.7]).parallels([29.5,45.5]).scale(1070)},ao.geo.albersUsa=function(){function \
n(n){var u=n[0],o=n[1];return t=null,e(u,o),t||(r(u,o),t)||i(u,o),t}var t,e,r,i,u=ao.geo.albers(),o=ao.geo.conicEqualArea().rotate([154,0]).center([-2,58.5]).parallels([55,65]),a=ao.geo.conicEqualArea().rotate([157,0]).center([-3,19.9]).parallels([8,18]),l={point:function(n,e){t=[n,e]}};return n.invert=function(n){var t=u.scale(),e=u.translate(),r=(n[0]-e[0])/t,i=(n[1]-e[1])/t;return(i>=.12&&.234>i&&r>=-.425&&-.214>r?o:i>=.166&&.234>i&&r>=-.214&&-.115>r?a:u).invert(n)},n.stream=function(n){var t=u.stream(n),e=o.stream(n),r=a.stream(n);return{point:function(n,i){t.point(n,i),e.point(n,i),r.point(n,i)},sphere:function(){t.sphere(),e.sphere(),r.sphere()},lineStart:function(){t.lineStart(),e.lineStart(),r.lineStart()},lineEnd:function(){t.lineEnd(),e.lineEnd(),r.lineEnd()},polygonStart:function(){t.polygonStart(),e.polygonStart(),r.polygonStart()},polygonEnd:function(){t.polygonEnd(),e.polygonEnd(),r.polygonEnd()}}},n.precision=function(t){return arguments.length?(u.precision(t),o.precision(t),a.precision(t),n):u.precision()},n.scale=function(t){return arguments.length?(u.scale(t),o.scale(.35*t),a.scale(t),n.translate(u.translate())):u.scale()},n.translate=function(t){if(!arguments.length)return u.translate();var c=u.scale(),f=+t[0],s=+t[1];return e=u.translate(t).clipExtent([[f-.455*c,s-.238*c],[f+.455*c,s+.238*c]]).stream(l).point,r=o.translate([f-.307*c,s+.201*c]).clipExtent([[f-.425*c+Uo,s+.12*c+Uo],[f-.214*c-Uo,s+.234*c-Uo]]).stream(l).point,i=a.translate([f-.205*c,s+.212*c]).clipExtent([[f-.214*c+Uo,s+.166*c+Uo],[f-.115*c-Uo,s+.234*c-Uo]]).stream(l).point,n},n.scale(1070)};var Oa,Ia,Ya,Za,Va,Xa,$a={point:b,lineStart:b,lineEnd:b,polygonStart:function(){Ia=0,$a.lineStart=$t},polygonEnd:function(){$a.lineStart=$a.lineEnd=$a.point=b,Oa+=xo(Ia/2)}},Ba={point:Bt,lineStart:b,lineEnd:b,polygonStart:b,polygonEnd:b},Wa={point:Gt,lineStart:Kt,lineEnd:Qt,polygonStart:function(){Wa.lineStart=ne},polygonEnd:function(){Wa.point=Gt,Wa.lineStart=Kt,Wa.lineEnd=Qt}};ao.geo.path=function(){function n(n){return n&&(\"function\"==typeof \
a&&u.pointRadius(+a.apply(this,arguments)),o&&o.valid||(o=i(u)),ao.geo.stream(n,o)),u.result()}function t(){return o=null,n}var e,r,i,u,o,a=4.5;return n.area=function(n){return Oa=0,ao.geo.stream(n,i($a)),Oa},n.centroid=function(n){return Ca=za=La=qa=Ta=Ra=Da=Pa=Ua=0,ao.geo.stream(n,i(Wa)),Ua?[Da/Ua,Pa/Ua]:Ra?[qa/Ra,Ta/Ra]:La?[Ca/La,za/La]:[NaN,NaN]},n.bounds=function(n){return Va=Xa=-(Ya=Za=1/0),ao.geo.stream(n,i(Ba)),[[Ya,Za],[Va,Xa]]},n.projection=function(n){return arguments.length?(i=(e=n)?n.stream||re(n):m,t()):e},n.context=function(n){return arguments.length?(u=null==(r=n)?new Wt:new te(n),\"function\"!=typeof a&&u.pointRadius(a),t()):r},n.pointRadius=function(t){return arguments.length?(a=\"function\"==typeof t?t:(u.pointRadius(+t),+t),n):a},n.projection(ao.geo.albersUsa()).context(null)},ao.geo.transform=function(n){return{stream:function(t){var e=new ie(t);for(var r in n)e[r]=n[r];return e}}},ie.prototype={point:function(n,t){this.stream.point(n,t)},sphere:function(){this.stream.sphere()},lineStart:function(){this.stream.lineStart()},lineEnd:function(){this.stream.lineEnd()},polygonStart:function(){this.stream.polygonStart()},polygonEnd:function(){this.stream.polygonEnd()}},ao.geo.projection=oe,ao.geo.projectionMutator=ae,(ao.geo.equirectangular=function(){return oe(ce)}).raw=ce.invert=ce,ao.geo.rotation=function(n){function t(t){return t=n(t[0]*Yo,t[1]*Yo),t[0]*=Zo,t[1]*=Zo,t}return n=se(n[0]%360*Yo,n[1]*Yo,n.length>2?n[2]*Yo:0),t.invert=function(t){return t=n.invert(t[0]*Yo,t[1]*Yo),t[0]*=Zo,t[1]*=Zo,t},t},fe.invert=ce,ao.geo.circle=function(){function n(){var n=\"function\"==typeof r?r.apply(this,arguments):r,t=se(-n[0]*Yo,-n[1]*Yo,0).invert,i=[];return e(null,null,1,{point:function(n,e){i.push(n=t(n,e)),n[0]*=Zo,n[1]*=Zo}}),{type:\"Polygon\",coordinates:[i]}}var t,e,r=[0,0],i=6;return n.origin=function(t){return arguments.length?(r=t,n):r},n.angle=function(r){return arguments.length?(e=ve((t=+r)*Yo,i*Yo),n):t},n.precision=function(r){return arguments.length?(e=ve(t*Yo,(i=+r)*Yo),n):i},n.angle(90)},ao.geo.distance=function(n,\
t){var e,r=(t[0]-n[0])*Yo,i=n[1]*Yo,u=t[1]*Yo,o=Math.sin(r),a=Math.cos(r),l=Math.sin(i),c=Math.cos(i),f=Math.sin(u),s=Math.cos(u);return Math.atan2(Math.sqrt((e=s*o)*e+(e=c*f-l*s*a)*e),l*f+c*s*a)},ao.geo.graticule=function(){function n(){return{type:\"MultiLineString\",coordinates:t()}}function t(){return ao.range(Math.ceil(u/d)*d,i,d).map(h).concat(ao.range(Math.ceil(c/y)*y,l,y).map(p)).concat(ao.range(Math.ceil(r/g)*g,e,g).filter(function(n){return xo(n%d)>Uo}).map(f)).concat(ao.range(Math.ceil(a/v)*v,o,v).filter(function(n){return xo(n%y)>Uo}).map(s))}var e,r,i,u,o,a,l,c,f,s,h,p,g=10,v=g,d=90,y=360,m=2.5;return n.lines=function(){return t().map(function(n){return{type:\"LineString\",coordinates:n}})},n.outline=function(){return{type:\"Polygon\",coordinates:[h(u).concat(p(l).slice(1),h(i).reverse().slice(1),p(c).reverse().slice(1))]}},n.extent=function(t){return arguments.length?n.majorExtent(t).minorExtent(t):n.minorExtent()},n.majorExtent=function(t){return arguments.length?(u=+t[0][0],i=+t[1][0],c=+t[0][1],l=+t[1][1],u>i&&(t=u,u=i,i=t),c>l&&(t=c,c=l,l=t),n.precision(m)):[[u,c],[i,l]]},n.minorExtent=function(t){return arguments.length?(r=+t[0][0],e=+t[1][0],a=+t[0][1],o=+t[1][1],r>e&&(t=r,r=e,e=t),a>o&&(t=a,a=o,o=t),n.precision(m)):[[r,a],[e,o]]},n.step=function(t){return arguments.length?n.majorStep(t).minorStep(t):n.minorStep()},n.majorStep=function(t){return arguments.length?(d=+t[0],y=+t[1],n):[d,y]},n.minorStep=function(t){return arguments.length?(g=+t[0],v=+t[1],n):[g,v]},n.precision=function(t){return arguments.length?(m=+t,f=ye(a,o,90),s=me(r,e,m),h=ye(c,l,90),p=me(u,i,m),n):m},n.majorExtent([[-180,-90+Uo],[180,90-Uo]]).minorExtent([[-180,-80-Uo],[180,80+Uo]])},ao.geo.greatArc=function(){function n(){return{type:\"LineString\",coordinates:[t||r.apply(this,arguments),e||i.apply(this,arguments)]}}var t,e,r=Me,i=xe;return n.distance=function(){return ao.geo.distance(t||r.apply(this,arguments),e||i.apply(this,arguments))},n.source=function(e){return arguments.length?(r=e,t=\"function\"==typeof e?null:e,n):r},n.target=function(t){return \
arguments.length?(i=t,e=\"function\"==typeof t?null:t,n):i},n.precision=function(){return arguments.length?n:0},n},ao.geo.interpolate=function(n,t){return be(n[0]*Yo,n[1]*Yo,t[0]*Yo,t[1]*Yo)},ao.geo.length=function(n){return Ja=0,ao.geo.stream(n,Ga),Ja};var Ja,Ga={sphere:b,point:b,lineStart:_e,lineEnd:b,polygonStart:b,polygonEnd:b},Ka=we(function(n){return Math.sqrt(2/(1+n))},function(n){return 2*Math.asin(n/2)});(ao.geo.azimuthalEqualArea=function(){return oe(Ka)}).raw=Ka;var Qa=we(function(n){var t=Math.acos(n);return t&&t/Math.sin(t)},m);(ao.geo.azimuthalEquidistant=function(){return oe(Qa)}).raw=Qa,(ao.geo.conicConformal=function(){return Vt(Se)}).raw=Se,(ao.geo.conicEquidistant=function(){return Vt(ke)}).raw=ke;var nl=we(function(n){return 1/n},Math.atan);(ao.geo.gnomonic=function(){return oe(nl)}).raw=nl,Ne.invert=function(n,t){return[n,2*Math.atan(Math.exp(t))-Io]},(ao.geo.mercator=function(){return Ee(Ne)}).raw=Ne;var tl=we(function(){return 1},Math.asin);(ao.geo.orthographic=function(){return oe(tl)}).raw=tl;var el=we(function(n){return 1/(1+n)},function(n){return 2*Math.atan(n)});(ao.geo.stereographic=function(){return oe(el)}).raw=el,Ae.invert=function(n,t){return[-t,2*Math.atan(Math.exp(n))-Io]},(ao.geo.transverseMercator=function(){var n=Ee(Ae),t=n.center,e=n.rotate;return n.center=function(n){return n?t([-n[1],n[0]]):(n=t(),[n[1],-n[0]])},n.rotate=function(n){return n?e([n[0],n[1],n.length>2?n[2]+90:90]):(n=e(),[n[0],n[1],n[2]-90])},e([0,0,90])}).raw=Ae,ao.geom={},ao.geom.hull=function(n){function t(n){if(n.length<3)return[];var t,i=En(e),u=En(r),o=n.length,a=[],l=[];for(t=0;o>t;t++)a.push([+i.call(this,n[t],t),+u.call(this,n[t],t),t]);for(a.sort(qe),t=0;o>t;t++)l.push([a[t][0],-a[t][1]]);var c=Le(a),f=Le(l),s=f[0]===c[0],h=f[f.length-1]===c[c.length-1],p=[];for(t=c.length-1;t>=0;--t)p.push(n[a[c[t]][2]]);for(t=+s;t<f.length-h;++t)p.push(n[a[f[t]][2]]);return p}var e=Ce,r=ze;return arguments.length?t(n):(t.x=function(n){return arguments.length?(e=n,t):e},t.y=function(n){return arguments.length?(r=n,t):r},\
t)},ao.geom.polygon=function(n){return ko(n,rl),n};var rl=ao.geom.polygon.prototype=[];rl.area=function(){for(var n,t=-1,e=this.length,r=this[e-1],i=0;++t<e;)n=r,r=this[t],i+=n[1]*r[0]-n[0]*r[1];return.5*i},rl.centroid=function(n){var t,e,r=-1,i=this.length,u=0,o=0,a=this[i-1];for(arguments.length||(n=-1/(6*this.area()));++r<i;)t=a,a=this[r],e=t[0]*a[1]-a[0]*t[1],u+=(t[0]+a[0])*e,o+=(t[1]+a[1])*e;return[u*n,o*n]},rl.clip=function(n){for(var t,e,r,i,u,o,a=De(n),l=-1,c=this.length-De(this),f=this[c-1];++l<c;){for(t=n.slice(),n.length=0,i=this[l],u=t[(r=t.length-a)-1],e=-1;++e<r;)o=t[e],Te(o,f,i)?(Te(u,f,i)||n.push(Re(u,o,f,i)),n.push(o)):Te(u,f,i)&&n.push(Re(u,o,f,i)),u=o;a&&n.push(n[0]),f=i}return n};var il,ul,ol,al,ll,cl=[],fl=[];Ye.prototype.prepare=function(){for(var n,t=this.edges,e=t.length;e--;)n=t[e].edge,n.b&&n.a||t.splice(e,1);return t.sort(Ve),t.length},tr.prototype={start:function(){return this.edge.l===this.site?this.edge.a:this.edge.b},end:function(){return this.edge.l===this.site?this.edge.b:this.edge.a}},er.prototype={insert:function(n,t){var e,r,i;if(n){if(t.P=n,t.N=n.N,n.N&&(n.N.P=t),n.N=t,n.R){for(n=n.R;n.L;)n=n.L;n.L=t}else n.R=t;e=n}else this._?(n=or(this._),t.P=null,t.N=n,n.P=n.L=t,e=n):(t.P=t.N=null,this._=t,e=null);for(t.L=t.R=null,t.U=e,t.C=!0,n=t;e&&e.C;)r=e.U,e===r.L?(i=r.R,i&&i.C?(e.C=i.C=!1,r.C=!0,n=r):(n===e.R&&(ir(this,e),n=e,e=n.U),e.C=!1,r.C=!0,ur(this,r))):(i=r.L,i&&i.C?(e.C=i.C=!1,r.C=!0,n=r):(n===e.L&&(ur(this,e),n=e,e=n.U),e.C=!1,r.C=!0,ir(this,r))),e=n.U;this._.C=!1},remove:function(n){n.N&&(n.N.P=n.P),n.P&&(n.P.N=n.N),n.N=n.P=null;var t,e,r,i=n.U,u=n.L,o=n.R;if(e=u?o?or(o):u:o,i?i.L===n?i.L=e:i.R=e:this._=e,u&&o?(r=e.C,e.C=n.C,e.L=u,u.U=e,e!==o?(i=e.U,e.U=n.U,n=e.R,i.L=n,e.R=o,o.U=e):(e.U=i,i=e,n=e.R)):(r=n.C,n=e),n&&(n.U=i),!r){if(n&&n.C)return void(n.C=!1);do{if(n===this._)break;if(n===i.L){if(t=i.R,t.C&&(t.C=!1,i.C=!0,ir(this,i),t=i.R),t.L&&t.L.C||t.R&&t.R.C){t.R&&t.R.C||(t.L.C=!1,t.C=!0,ur(this,t),t=i.R),t.C=i.C,i.C=t.R.C=!1,ir(this,i),n=this._;break}}else if(t=i.L,t.C&&(t.C=!1,\
i.C=!0,ur(this,i),t=i.L),t.L&&t.L.C||t.R&&t.R.C){t.L&&t.L.C||(t.R.C=!1,t.C=!0,ir(this,t),t=i.L),t.C=i.C,i.C=t.L.C=!1,ur(this,i),n=this._;break}t.C=!0,n=i,i=i.U}while(!n.C);n&&(n.C=!1)}}},ao.geom.voronoi=function(n){function t(n){var t=new Array(n.length),r=a[0][0],i=a[0][1],u=a[1][0],o=a[1][1];return ar(e(n),a).cells.forEach(function(e,a){var l=e.edges,c=e.site,f=t[a]=l.length?l.map(function(n){var t=n.start();return[t.x,t.y]}):c.x>=r&&c.x<=u&&c.y>=i&&c.y<=o?[[r,o],[u,o],[u,i],[r,i]]:[];f.point=n[a]}),t}function e(n){return n.map(function(n,t){return{x:Math.round(u(n,t)/Uo)*Uo,y:Math.round(o(n,t)/Uo)*Uo,i:t}})}var r=Ce,i=ze,u=r,o=i,a=sl;return n?t(n):(t.links=function(n){return ar(e(n)).edges.filter(function(n){return n.l&&n.r}).map(function(t){return{source:n[t.l.i],target:n[t.r.i]}})},t.triangles=function(n){var t=[];return ar(e(n)).cells.forEach(function(e,r){for(var i,u,o=e.site,a=e.edges.sort(Ve),l=-1,c=a.length,f=a[c-1].edge,s=f.l===o?f.r:f.l;++l<c;)i=f,u=s,f=a[l].edge,s=f.l===o?f.r:f.l,r<u.i&&r<s.i&&cr(o,u,s)<0&&t.push([n[r],n[u.i],n[s.i]])}),t},t.x=function(n){return arguments.length?(u=En(r=n),t):r},t.y=function(n){return arguments.length?(o=En(i=n),t):i},t.clipExtent=function(n){return arguments.length?(a=null==n?sl:n,t):a===sl?null:a},t.size=function(n){return arguments.length?t.clipExtent(n&&[[0,0],n]):a===sl?null:a&&a[1]},t)};var sl=[[-1e6,-1e6],[1e6,1e6]];ao.geom.delaunay=function(n){return ao.geom.voronoi().triangles(n)},ao.geom.quadtree=function(n,t,e,r,i){function u(n){function u(n,t,e,r,i,u,o,a){if(!isNaN(e)&&!isNaN(r))if(n.leaf){var l=n.x,f=n.y;if(null!=l)if(xo(l-e)+xo(f-r)<.01)c(n,t,e,r,i,u,o,a);else{var s=n.point;n.x=n.y=n.point=null,c(n,s,l,f,i,u,o,a),c(n,t,e,r,i,u,o,a)}else n.x=e,n.y=r,n.point=t}else c(n,t,e,r,i,u,o,a)}function c(n,t,e,r,i,o,a,l){var c=.5*(i+a),f=.5*(o+l),s=e>=c,h=r>=f,p=h<<1|s;n.leaf=!1,n=n.nodes[p]||(n.nodes[p]=hr()),s?i=c:a=c,h?o=f:l=f,u(n,t,e,r,i,o,a,l)}var f,s,h,p,g,v,d,y,m,M=En(a),x=En(l);if(null!=t)v=t,d=e,y=r,m=i;else if(y=m=-(v=d=1/0),s=[],h=[],g=n.length,o)for(p=0;g>p;\
++p)f=n[p],f.x<v&&(v=f.x),f.y<d&&(d=f.y),f.x>y&&(y=f.x),f.y>m&&(m=f.y),s.push(f.x),h.push(f.y);else for(p=0;g>p;++p){var b=+M(f=n[p],p),_=+x(f,p);v>b&&(v=b),d>_&&(d=_),b>y&&(y=b),_>m&&(m=_),s.push(b),h.push(_)}var w=y-v,S=m-d;w>S?m=d+w:y=v+S;var k=hr();if(k.add=function(n){u(k,n,+M(n,++p),+x(n,p),v,d,y,m)},k.visit=function(n){pr(n,k,v,d,y,m)},k.find=function(n){return gr(k,n[0],n[1],v,d,y,m)},p=-1,null==t){for(;++p<g;)u(k,n[p],s[p],h[p],v,d,y,m);--p}else n.forEach(k.add);return s=h=n=f=null,k}var o,a=Ce,l=ze;return(o=arguments.length)?(a=fr,l=sr,3===o&&(i=e,r=t,e=t=0),u(n)):(u.x=function(n){return arguments.length?(a=n,u):a},u.y=function(n){return arguments.length?(l=n,u):l},u.extent=function(n){return arguments.length?(null==n?t=e=r=i=null:(t=+n[0][0],e=+n[0][1],r=+n[1][0],i=+n[1][1]),u):null==t?null:[[t,e],[r,i]]},u.size=function(n){return arguments.length?(null==n?t=e=r=i=null:(t=e=0,r=+n[0],i=+n[1]),u):null==t?null:[r-t,i-e]},u)},ao.interpolateRgb=vr,ao.interpolateObject=dr,ao.interpolateNumber=yr,ao.interpolateString=mr;var hl=/[-+]?(?:\\d+\\.?\\d*|\\.?\\d+)(?:[eE][-+]?\\d+)?/g,pl=new RegExp(hl.source,\"g\");ao.interpolate=Mr,ao.interpolators=[function(n,t){var e=typeof t;return(\"string\"===e?ua.has(t.toLowerCase())||/^(#|rgb\\(|hsl\\()/i.test(t)?vr:mr:t instanceof an?vr:Array.isArray(t)?xr:\"object\"===e&&isNaN(t)?dr:yr)(n,t)}],ao.interpolateArray=xr;var gl=function(){return m},vl=ao.map({linear:gl,poly:Er,quad:function(){return Sr},cubic:function(){return kr},sin:function(){return Ar},exp:function(){return Cr},circle:function(){return zr},elastic:Lr,back:qr,bounce:function(){return Tr}}),dl=ao.map({\"in\":m,out:_r,\"in-out\":wr,\"out-in\":function(n){return wr(_r(n))}});ao.ease=function(n){var t=n.indexOf(\"-\"),e=t>=0?n.slice(0,t):n,r=t>=0?n.slice(t+1):\"in\";return e=vl.get(e)||gl,r=dl.get(r)||m,br(r(e.apply(null,lo.call(arguments,1))))},ao.interpolateHcl=Rr,ao.interpolateHsl=Dr,ao.interpolateLab=Pr,ao.interpolateRound=Ur,ao.transform=function(n){var t=fo.createElementNS(ao.ns.prefix.svg,\"g\");return(ao.transform=function(n){if(null!=n){t.setAttribute(\"transform\",\
n);var e=t.transform.baseVal.consolidate()}return new jr(e?e.matrix:yl)})(n)},jr.prototype.toString=function(){return\"translate(\"+this.translate+\")rotate(\"+this.rotate+\")skewX(\"+this.skew+\")scale(\"+this.scale+\")\"};var yl={a:1,b:0,c:0,d:1,e:0,f:0};ao.interpolateTransform=$r,ao.layout={},ao.layout.bundle=function(){return function(n){for(var t=[],e=-1,r=n.length;++e<r;)t.push(Jr(n[e]));return t}},ao.layout.chord=function(){function n(){var n,c,s,h,p,g={},v=[],d=ao.range(u),y=[];for(e=[],r=[],n=0,h=-1;++h<u;){for(c=0,p=-1;++p<u;)c+=i[h][p];v.push(c),y.push(ao.range(u)),n+=c}for(o&&d.sort(function(n,t){return o(v[n],v[t])}),a&&y.forEach(function(n,t){n.sort(function(n,e){return a(i[t][n],i[t][e])})}),n=(Ho-f*u)/n,c=0,h=-1;++h<u;){for(s=c,p=-1;++p<u;){var m=d[h],M=y[m][p],x=i[m][M],b=c,_=c+=x*n;g[m+\"-\"+M]={index:m,subindex:M,startAngle:b,endAngle:_,value:x}}r[m]={index:m,startAngle:s,endAngle:c,value:v[m]},c+=f}for(h=-1;++h<u;)for(p=h-1;++p<u;){var w=g[h+\"-\"+p],S=g[p+\"-\"+h];(w.value||S.value)&&e.push(w.value<S.value?{source:S,target:w}:{source:w,target:S})}l&&t()}function t(){e.sort(function(n,t){return l((n.source.value+n.target.value)/2,(t.source.value+t.target.value)/2)})}var e,r,i,u,o,a,l,c={},f=0;return c.matrix=function(n){return arguments.length?(u=(i=n)&&i.length,e=r=null,c):i},c.padding=function(n){return arguments.length?(f=n,e=r=null,c):f},c.sortGroups=function(n){return arguments.length?(o=n,e=r=null,c):o},c.sortSubgroups=function(n){return arguments.length?(a=n,e=null,c):a},c.sortChords=function(n){return arguments.length?(l=n,e&&t(),c):l},c.chords=function(){return e||n(),e},c.groups=function(){return r||n(),r},c},ao.layout.force=function(){function n(n){return function(t,e,r,i){if(t.point!==n){var u=t.cx-n.x,o=t.cy-n.y,a=i-e,l=u*u+o*o;if(l>a*a/y){if(v>l){var c=t.charge/l;n.px-=u*c,n.py-=o*c}return!0}if(t.point&&l&&v>l){var c=t.pointCharge/l;n.px-=u*c,n.py-=o*c}}return!t.charge}}function t(n){n.px=ao.event.x,n.py=ao.event.y,l.resume()}var e,r,i,u,o,a,l={},c=ao.dispatch(\"start\",\"tick\",\"end\"),f=[1,1],s=.9,h=ml,\
p=Ml,g=-30,v=xl,d=.1,y=.64,M=[],x=[];return l.tick=function(){if((i*=.99)<.005)return e=null,c.end({type:\"end\",alpha:i=0}),!0;var t,r,l,h,p,v,y,m,b,_=M.length,w=x.length;for(r=0;w>r;++r)l=x[r],h=l.source,p=l.target,m=p.x-h.x,b=p.y-h.y,(v=m*m+b*b)&&(v=i*o[r]*((v=Math.sqrt(v))-u[r])/v,m*=v,b*=v,p.x-=m*(y=h.weight+p.weight?h.weight/(h.weight+p.weight):.5),p.y-=b*y,h.x+=m*(y=1-y),h.y+=b*y);if((y=i*d)&&(m=f[0]/2,b=f[1]/2,r=-1,y))for(;++r<_;)l=M[r],l.x+=(m-l.x)*y,l.y+=(b-l.y)*y;if(g)for(ri(t=ao.geom.quadtree(M),i,a),r=-1;++r<_;)(l=M[r]).fixed||t.visit(n(l));for(r=-1;++r<_;)l=M[r],l.fixed?(l.x=l.px,l.y=l.py):(l.x-=(l.px-(l.px=l.x))*s,l.y-=(l.py-(l.py=l.y))*s);c.tick({type:\"tick\",alpha:i})},l.nodes=function(n){return arguments.length?(M=n,l):M},l.links=function(n){return arguments.length?(x=n,l):x},l.size=function(n){return arguments.length?(f=n,l):f},l.linkDistance=function(n){return arguments.length?(h=\"function\"==typeof n?n:+n,l):h},l.distance=l.linkDistance,l.linkStrength=function(n){return arguments.length?(p=\"function\"==typeof n?n:+n,l):p},l.friction=function(n){return arguments.length?(s=+n,l):s},l.charge=function(n){return arguments.length?(g=\"function\"==typeof n?n:+n,l):g},l.chargeDistance=function(n){return arguments.length?(v=n*n,l):Math.sqrt(v)},l.gravity=function(n){return arguments.length?(d=+n,l):d},l.theta=function(n){return arguments.length?(y=n*n,l):Math.sqrt(y)},l.alpha=function(n){return arguments.length?(n=+n,i?n>0?i=n:(e.c=null,e.t=NaN,e=null,c.end({type:\"end\",alpha:i=0})):n>0&&(c.start({type:\"start\",alpha:i=n}),e=qn(l.tick)),l):i},l.start=function(){function n(n,r){if(!e){for(e=new Array(i),l=0;i>l;++l)e[l]=[];for(l=0;c>l;++l){var u=x[l];e[u.source.index].push(u.target),e[u.target.index].push(u.source)}}for(var o,a=e[t],l=-1,f=a.length;++l<f;)if(!isNaN(o=a[l][n]))return o;return Math.random()*r}var t,e,r,i=M.length,c=x.length,s=f[0],v=f[1];for(t=0;i>t;++t)(r=M[t]).index=t,r.weight=0;for(t=0;c>t;++t)r=x[t],\"number\"==typeof r.source&&(r.source=M[r.source]),\"number\"==typeof r.target&&(r.target=M[r.target]),\
++r.source.weight,++r.target.weight;for(t=0;i>t;++t)r=M[t],isNaN(r.x)&&(r.x=n(\"x\",s)),isNaN(r.y)&&(r.y=n(\"y\",v)),isNaN(r.px)&&(r.px=r.x),isNaN(r.py)&&(r.py=r.y);if(u=[],\"function\"==typeof h)for(t=0;c>t;++t)u[t]=+h.call(this,x[t],t);else for(t=0;c>t;++t)u[t]=h;if(o=[],\"function\"==typeof p)for(t=0;c>t;++t)o[t]=+p.call(this,x[t],t);else for(t=0;c>t;++t)o[t]=p;if(a=[],\"function\"==typeof g)for(t=0;i>t;++t)a[t]=+g.call(this,M[t],t);else for(t=0;i>t;++t)a[t]=g;return l.resume()},l.resume=function(){return l.alpha(.1)},l.stop=function(){return l.alpha(0)},l.drag=function(){return r||(r=ao.behavior.drag().origin(m).on(\"dragstart.force\",Qr).on(\"drag.force\",t).on(\"dragend.force\",ni)),arguments.length?void this.on(\"mouseover.force\",ti).on(\"mouseout.force\",ei).call(r):r},ao.rebind(l,c,\"on\")};var ml=20,Ml=1,xl=1/0;ao.layout.hierarchy=function(){function n(i){var u,o=[i],a=[];for(i.depth=0;null!=(u=o.pop());)if(a.push(u),(c=e.call(n,u,u.depth))&&(l=c.length)){for(var l,c,f;--l>=0;)o.push(f=c[l]),f.parent=u,f.depth=u.depth+1;r&&(u.value=0),u.children=c}else r&&(u.value=+r.call(n,u,u.depth)||0),delete u.children;return oi(i,function(n){var e,i;t&&(e=n.children)&&e.sort(t),r&&(i=n.parent)&&(i.value+=n.value)}),a}var t=ci,e=ai,r=li;return n.sort=function(e){return arguments.length?(t=e,n):t},n.children=function(t){return arguments.length?(e=t,n):e},n.value=function(t){return arguments.length?(r=t,n):r},n.revalue=function(t){return r&&(ui(t,function(n){n.children&&(n.value=0)}),oi(t,function(t){var e;t.children||(t.value=+r.call(n,t,t.depth)||0),(e=t.parent)&&(e.value+=t.value)})),t},n},ao.layout.partition=function(){function n(t,e,r,i){var u=t.children;if(t.x=e,t.y=t.depth*i,t.dx=r,t.dy=i,u&&(o=u.length)){var o,a,l,c=-1;for(r=t.value?r/t.value:0;++c<o;)n(a=u[c],e,l=a.value*r,i),e+=l}}function t(n){var e=n.children,r=0;if(e&&(i=e.length))for(var i,u=-1;++u<i;)r=Math.max(r,t(e[u]));return 1+r}function e(e,u){var o=r.call(this,e,u);return n(o[0],0,i[0],i[1]/t(o[0])),o}var r=ao.layout.hierarchy(),i=[1,1];return e.size=function(n){return \
arguments.length?(i=n,e):i},ii(e,r)},ao.layout.pie=function(){function n(o){var a,l=o.length,c=o.map(function(e,r){return+t.call(n,e,r)}),f=+(\"function\"==typeof r?r.apply(this,arguments):r),s=(\"function\"==typeof i?i.apply(this,arguments):i)-f,h=Math.min(Math.abs(s)/l,+(\"function\"==typeof u?u.apply(this,arguments):u)),p=h*(0>s?-1:1),g=ao.sum(c),v=g?(s-l*p)/g:0,d=ao.range(l),y=[];return null!=e&&d.sort(e===bl?function(n,t){return c[t]-c[n]}:function(n,t){return e(o[n],o[t])}),d.forEach(function(n){y[n]={data:o[n],value:a=c[n],startAngle:f,endAngle:f+=a*v+p,padAngle:h}}),y}var t=Number,e=bl,r=0,i=Ho,u=0;return n.value=function(e){return arguments.length?(t=e,n):t},n.sort=function(t){return arguments.length?(e=t,n):e},n.startAngle=function(t){return arguments.length?(r=t,n):r},n.endAngle=function(t){return arguments.length?(i=t,n):i},n.padAngle=function(t){return arguments.length?(u=t,n):u},n};var bl={};ao.layout.stack=function(){function n(a,l){if(!(h=a.length))return a;var c=a.map(function(e,r){return t.call(n,e,r)}),f=c.map(function(t){return t.map(function(t,e){return[u.call(n,t,e),o.call(n,t,e)]})}),s=e.call(n,f,l);c=ao.permute(c,s),f=ao.permute(f,s);var h,p,g,v,d=r.call(n,f,l),y=c[0].length;for(g=0;y>g;++g)for(i.call(n,c[0][g],v=d[g],f[0][g][1]),p=1;h>p;++p)i.call(n,c[p][g],v+=f[p-1][g][1],f[p][g][1]);return a}var t=m,e=gi,r=vi,i=pi,u=si,o=hi;return n.values=function(e){return arguments.length?(t=e,n):t},n.order=function(t){return arguments.length?(e=\"function\"==typeof t?t:_l.get(t)||gi,n):e},n.offset=function(t){return arguments.length?(r=\"function\"==typeof t?t:wl.get(t)||vi,n):r},n.x=function(t){return arguments.length?(u=t,n):u},n.y=function(t){return arguments.length?(o=t,n):o},n.out=function(t){return arguments.length?(i=t,n):i},n};var _l=ao.map({\"inside-out\":function(n){var t,e,r=n.length,i=n.map(di),u=n.map(yi),o=ao.range(r).sort(function(n,t){return i[n]-i[t]}),a=0,l=0,c=[],f=[];for(t=0;r>t;++t)e=o[t],l>a?(a+=u[e],c.push(e)):(l+=u[e],f.push(e));return f.reverse().concat(c)},reverse:function(n){return ao.range(n.length).reverse()},\
\"default\":gi}),wl=ao.map({silhouette:function(n){var t,e,r,i=n.length,u=n[0].length,o=[],a=0,l=[];for(e=0;u>e;++e){for(t=0,r=0;i>t;t++)r+=n[t][e][1];r>a&&(a=r),o.push(r)}for(e=0;u>e;++e)l[e]=(a-o[e])/2;return l},wiggle:function(n){var t,e,r,i,u,o,a,l,c,f=n.length,s=n[0],h=s.length,p=[];for(p[0]=l=c=0,e=1;h>e;++e){for(t=0,i=0;f>t;++t)i+=n[t][e][1];for(t=0,u=0,a=s[e][0]-s[e-1][0];f>t;++t){for(r=0,o=(n[t][e][1]-n[t][e-1][1])/(2*a);t>r;++r)o+=(n[r][e][1]-n[r][e-1][1])/a;u+=o*n[t][e][1]}p[e]=l-=i?u/i*a:0,c>l&&(c=l)}for(e=0;h>e;++e)p[e]-=c;return p},expand:function(n){var t,e,r,i=n.length,u=n[0].length,o=1/i,a=[];for(e=0;u>e;++e){for(t=0,r=0;i>t;t++)r+=n[t][e][1];if(r)for(t=0;i>t;t++)n[t][e][1]/=r;else for(t=0;i>t;t++)n[t][e][1]=o}for(e=0;u>e;++e)a[e]=0;return a},zero:vi});ao.layout.histogram=function(){function n(n,u){for(var o,a,l=[],c=n.map(e,this),f=r.call(this,c,u),s=i.call(this,f,c,u),u=-1,h=c.length,p=s.length-1,g=t?1:1/h;++u<p;)o=l[u]=[],o.dx=s[u+1]-(o.x=s[u]),o.y=0;if(p>0)for(u=-1;++u<h;)a=c[u],a>=f[0]&&a<=f[1]&&(o=l[ao.bisect(s,a,1,p)-1],o.y+=g,o.push(n[u]));return l}var t=!0,e=Number,r=bi,i=Mi;return n.value=function(t){return arguments.length?(e=t,n):e},n.range=function(t){return arguments.length?(r=En(t),n):r},n.bins=function(t){return arguments.length?(i=\"number\"==typeof t?function(n){return xi(n,t)}:En(t),n):i},n.frequency=function(e){return arguments.length?(t=!!e,n):t},n},ao.layout.pack=function(){function n(n,u){var o=e.call(this,n,u),a=o[0],l=i[0],c=i[1],f=null==t?Math.sqrt:\"function\"==typeof t?t:function(){return t};if(a.x=a.y=0,oi(a,function(n){n.r=+f(n.value)}),oi(a,Ni),r){var s=r*(t?1:Math.max(2*a.r/l,2*a.r/c))/2;oi(a,function(n){n.r+=s}),oi(a,Ni),oi(a,function(n){n.r-=s})}return Ci(a,l/2,c/2,t?1:1/Math.max(2*a.r/l,2*a.r/c)),o}var t,e=ao.layout.hierarchy().sort(_i),r=0,i=[1,1];return n.size=function(t){return arguments.length?(i=t,n):i},n.radius=function(e){return arguments.length?(t=null==e||\"function\"==typeof e?e:+e,n):t},n.padding=function(t){return arguments.length?(r=+t,n):r},ii(n,e)},ao.layout.tree=function(){function \
n(n,i){var f=o.call(this,n,i),s=f[0],h=t(s);if(oi(h,e),h.parent.m=-h.z,ui(h,r),c)ui(s,u);else{var p=s,g=s,v=s;ui(s,function(n){n.x<p.x&&(p=n),n.x>g.x&&(g=n),n.depth>v.depth&&(v=n)});var d=a(p,g)/2-p.x,y=l[0]/(g.x+a(g,p)/2+d),m=l[1]/(v.depth||1);ui(s,function(n){n.x=(n.x+d)*y,n.y=n.depth*m})}return f}function t(n){for(var t,e={A:null,children:[n]},r=[e];null!=(t=r.pop());)for(var i,u=t.children,o=0,a=u.length;a>o;++o)r.push((u[o]=i={_:u[o],parent:t,children:(i=u[o].children)&&i.slice()||[],A:null,a:null,z:0,m:0,c:0,s:0,t:null,i:o}).a=i);return e.children[0]}function e(n){var t=n.children,e=n.parent.children,r=n.i?e[n.i-1]:null;if(t.length){Di(n);var u=(t[0].z+t[t.length-1].z)/2;r?(n.z=r.z+a(n._,r._),n.m=n.z-u):n.z=u}else r&&(n.z=r.z+a(n._,r._));n.parent.A=i(n,r,n.parent.A||e[0])}function r(n){n._.x=n.z+n.parent.m,n.m+=n.parent.m}function i(n,t,e){if(t){for(var r,i=n,u=n,o=t,l=i.parent.children[0],c=i.m,f=u.m,s=o.m,h=l.m;o=Ti(o),i=qi(i),o&&i;)l=qi(l),u=Ti(u),u.a=n,r=o.z+s-i.z-c+a(o._,i._),r>0&&(Ri(Pi(o,n,e),n,r),c+=r,f+=r),s+=o.m,c+=i.m,h+=l.m,f+=u.m;o&&!Ti(u)&&(u.t=o,u.m+=s-f),i&&!qi(l)&&(l.t=i,l.m+=c-h,e=n)}return e}function u(n){n.x*=l[0],n.y=n.depth*l[1]}var o=ao.layout.hierarchy().sort(null).value(null),a=Li,l=[1,1],c=null;return n.separation=function(t){return arguments.length?(a=t,n):a},n.size=function(t){return arguments.length?(c=null==(l=t)?u:null,n):c?null:l},n.nodeSize=function(t){return arguments.length?(c=null==(l=t)?null:u,n):c?l:null},ii(n,o)},ao.layout.cluster=function(){function n(n,u){var o,a=t.call(this,n,u),l=a[0],c=0;oi(l,function(n){var t=n.children;t&&t.length?(n.x=ji(t),n.y=Ui(t)):(n.x=o?c+=e(n,o):0,n.y=0,o=n)});var f=Fi(l),s=Hi(l),h=f.x-e(f,s)/2,p=s.x+e(s,f)/2;return oi(l,i?function(n){n.x=(n.x-l.x)*r[0],n.y=(l.y-n.y)*r[1]}:function(n){n.x=(n.x-h)/(p-h)*r[0],n.y=(1-(l.y?n.y/l.y:1))*r[1]}),a}var t=ao.layout.hierarchy().sort(null).value(null),e=Li,r=[1,1],i=!1;return n.separation=function(t){return arguments.length?(e=t,n):e},n.size=function(t){return arguments.length?(i=null==(r=t),n):i?null:r},\
n.nodeSize=function(t){return arguments.length?(i=null!=(r=t),n):i?r:null},ii(n,t)},ao.layout.treemap=function(){function n(n,t){for(var e,r,i=-1,u=n.length;++i<u;)r=(e=n[i]).value*(0>t?0:t),e.area=isNaN(r)||0>=r?0:r}function t(e){var u=e.children;if(u&&u.length){var o,a,l,c=s(e),f=[],h=u.slice(),g=1/0,v=\"slice\"===p?c.dx:\"dice\"===p?c.dy:\"slice-dice\"===p?1&e.depth?c.dy:c.dx:Math.min(c.dx,c.dy);for(n(h,c.dx*c.dy/e.value),f.area=0;(l=h.length)>0;)f.push(o=h[l-1]),f.area+=o.area,\"squarify\"!==p||(a=r(f,v))<=g?(h.pop(),g=a):(f.area-=f.pop().area,i(f,v,c,!1),v=Math.min(c.dx,c.dy),f.length=f.area=0,g=1/0);f.length&&(i(f,v,c,!0),f.length=f.area=0),u.forEach(t)}}function e(t){var r=t.children;if(r&&r.length){var u,o=s(t),a=r.slice(),l=[];for(n(a,o.dx*o.dy/t.value),l.area=0;u=a.pop();)l.push(u),l.area+=u.area,null!=u.z&&(i(l,u.z?o.dx:o.dy,o,!a.length),l.length=l.area=0);r.forEach(e)}}function r(n,t){for(var e,r=n.area,i=0,u=1/0,o=-1,a=n.length;++o<a;)(e=n[o].area)&&(u>e&&(u=e),e>i&&(i=e));return r*=r,t*=t,r?Math.max(t*i*g/r,r/(t*u*g)):1/0}function i(n,t,e,r){var i,u=-1,o=n.length,a=e.x,c=e.y,f=t?l(n.area/t):0;\
if(t==e.dx){for((r||f>e.dy)&&(f=e.dy);++u<o;)i=n[u],i.x=a,i.y=c,i.dy=f,a+=i.dx=Math.min(e.x+e.dx-a,f?l(i.area/f):0);i.z=!0,i.dx+=e.x+e.dx-a,e.y+=f,e.dy-=f}else{for((r||f>e.dx)&&(f=e.dx);++u<o;)i=n[u],i.x=a,i.y=c,i.dx=f,c+=i.dy=Math.min(e.y+e.dy-c,f?l(i.area/f):0);i.z=!1,i.dy+=e.y+e.dy-c,e.x+=f,e.dx-=f}}function u(r){var i=o||a(r),u=i[0];return u.x=u.y=0,u.value?(u.dx=c[0],u.dy=c[1]):u.dx=u.dy=0,o&&a.revalue(u),n([u],u.dx*u.dy/u.value),(o?e:t)(u),h&&(o=i),i}var o,a=ao.layout.hierarchy(),l=Math.round,c=[1,1],f=null,s=Oi,h=!1,p=\"squarify\",g=.5*(1+Math.sqrt(5));return u.size=function(n){return arguments.length?(c=n,u):c},u.padding=function(n){function t(t){var e=n.call(u,t,t.depth);return null==e?Oi(t):Ii(t,\"number\"==typeof e?[e,e,e,e]:e)}function e(t){return Ii(t,n)}if(!arguments.length)return f;var r;return s=null==(f=n)?Oi:\"function\"==(r=typeof n)?t:\"number\"===r?(n=[n,n,n,n],e):e,u},u.round=function(n){return arguments.length?(l=n?Math.round:Number,u):l!=Number},u.sticky=function(n){return arguments.length?(h=n,o=null,u):h},u.ratio=function(n){return arguments.length?(g=n,u):g},u.mode=function(n){return arguments.length?(p=n+\"\",u):p},ii(u,a)},ao.random={normal:function(n,t){var e=arguments.length;return 2>e&&(t=1),1>e&&(n=0),function(){var e,r,i;do e=2*Math.random()-1,r=2*Math.random()-1,i=e*e+r*r;while(!i||i>1);return n+t*e*Math.sqrt(-2*Math.log(i)/i)}},logNormal:function(){var n=ao.random.normal.apply(ao,arguments);return function(){return Math.exp(n())}},bates:function(n){var t=ao.random.irwinHall(n);return function(){return t()/n}},irwinHall:function(n){return function(){for(var t=0,e=0;n>e;e++)t+=Math.random();return t}}},ao.scale={};var Sl={floor:m,ceil:m};ao.scale.linear=function(){return Wi([0,1],[0,1],Mr,!1)};var kl={s:1,g:1,p:1,r:1,e:1};ao.scale.log=function(){return ru(ao.scale.linear().domain([0,1]),10,!0,[1,10])};var Nl=ao.format(\".0e\"),El={floor:function(n){return-Math.ceil(-n)},ceil:function(n){return-Math.floor(-n)}};ao.scale.pow=function(){return iu(ao.scale.linear(),1,[0,1])},ao.scale.sqrt=function(){return \
ao.scale.pow().exponent(.5)},ao.scale.ordinal=function(){return ou([],{t:\"range\",a:[[]]})},ao.scale.category10=function(){return ao.scale.ordinal().range(Al)},ao.scale.category20=function(){return ao.scale.ordinal().range(Cl)},ao.scale.category20b=function(){return ao.scale.ordinal().range(zl)},ao.scale.category20c=function(){return ao.scale.ordinal().range(Ll)};var Al=[2062260,16744206,2924588,14034728,9725885,9197131,14907330,8355711,12369186,1556175].map(xn),Cl=[2062260,11454440,16744206,16759672,2924588,10018698,14034728,16750742,9725885,12955861,9197131,12885140,14907330,16234194,8355711,13092807,12369186,14408589,1556175,10410725].map(xn),zl=[3750777,5395619,7040719,10264286,6519097,9216594,11915115,13556636,9202993,12426809,15186514,15190932,8666169,11356490,14049643,15177372,8077683,10834324,13528509,14589654].map(xn),Ll=[3244733,7057110,10406625,13032431,15095053,16616764,16625259,16634018,3253076,7652470,10607003,13101504,7695281,10394312,12369372,14342891,6513507,9868950,12434877,14277081].map(xn);ao.scale.quantile=function(){return au([],[])},ao.scale.quantize=function(){return lu(0,1,[0,1])},ao.scale.threshold=function(){return cu([.5],[0,1])},ao.scale.identity=function(){return fu([0,1])},ao.svg={},ao.svg.arc=function(){function n(){var n=Math.max(0,+e.apply(this,arguments)),c=Math.max(0,+r.apply(this,arguments)),f=o.apply(this,arguments)-Io,s=a.apply(this,arguments)-Io,h=Math.abs(s-f),p=f>s?0:1;if(n>c&&(g=c,c=n,n=g),h>=Oo)return t(c,p)+(n?t(n,1-p):\"\")+\"Z\";var g,v,d,y,m,M,x,b,_,w,S,k,N=0,E=0,A=[];if((y=(+l.apply(this,arguments)||0)/2)&&(d=u===ql?Math.sqrt(n*n+c*c):+u.apply(this,arguments),p||(E*=-1),c&&(E=tn(d/c*Math.sin(y))),n&&(N=tn(d/n*Math.sin(y)))),c){m=c*Math.cos(f+E),M=c*Math.sin(f+E),x=c*Math.cos(s-E),b=c*Math.sin(s-E);var C=Math.abs(s-f-2*E)<=Fo?0:1;if(E&&yu(m,M,x,b)===p^C){var z=(f+s)/2;m=c*Math.cos(z),M=c*Math.sin(z),x=b=null}}else m=M=0;if(n){_=n*Math.cos(s-N),w=n*Math.sin(s-N),S=n*Math.cos(f+N),k=n*Math.sin(f+N);var L=Math.abs(f-s+2*N)<=Fo?0:1;if(N&&yu(_,w,S,k)===1-p^L){var q=(f+s)/2;_=n*Math.cos(q),\
w=n*Math.sin(q),S=k=null}}else _=w=0;if(h>Uo&&(g=Math.min(Math.abs(c-n)/2,+i.apply(this,arguments)))>.001){v=c>n^p?0:1;var T=g,R=g;if(Fo>h){var D=null==S?[_,w]:null==x?[m,M]:Re([m,M],[S,k],[x,b],[_,w]),P=m-D[0],U=M-D[1],j=x-D[0],F=b-D[1],H=1/Math.sin(Math.acos((P*j+U*F)/(Math.sqrt(P*P+U*U)*Math.sqrt(j*j+F*F)))/2),O=Math.sqrt(D[0]*D[0]+D[1]*D[1]);R=Math.min(g,(n-O)/(H-1)),T=Math.min(g,(c-O)/(H+1))}if(null!=x){var I=mu(null==S?[_,w]:[S,k],[m,M],c,T,p),Y=mu([x,b],[_,w],c,T,p);g===T?A.push(\"M\",I[0],\"A\",T,\",\",T,\" 0 0,\",v,\" \",I[1],\"A\",c,\",\",c,\" 0 \",1-p^yu(I[1][0],I[1][1],Y[1][0],Y[1][1]),\",\",p,\" \",Y[1],\"A\",T,\",\",T,\" 0 0,\",v,\" \",Y[0]):A.push(\"M\",I[0],\"A\",T,\",\",T,\" 0 1,\",v,\" \",Y[0])}else A.push(\"M\",m,\",\",M);if(null!=S){var Z=mu([m,M],[S,k],n,-R,p),V=mu([_,w],null==x?[m,M]:[x,b],n,-R,p);g===R?A.push(\"L\",V[0],\"A\",R,\",\",R,\" 0 0,\",v,\" \",V[1],\"A\",n,\",\",n,\" 0 \",p^yu(V[1][0],V[1][1],Z[1][0],Z[1][1]),\",\",1-p,\" \",Z[1],\"A\",R,\",\",R,\" 0 0,\",v,\" \",Z[0]):A.push(\"L\",V[0],\"A\",R,\",\",R,\" 0 0,\",v,\" \",Z[0])}else A.push(\"L\",_,\",\",w)}else A.push(\"M\",m,\",\",M),null!=x&&A.push(\"A\",c,\",\",c,\" 0 \",C,\",\",p,\" \",x,\",\",b),A.push(\"L\",_,\",\",w),null!=S&&A.push(\"A\",n,\",\",n,\" 0 \",L,\",\",1-p,\" \",S,\",\",k);return A.push(\"Z\"),A.join(\"\")}function t(n,t){return\"M0,\"+n+\"A\"+n+\",\"+n+\" 0 1,\"+t+\" 0,\"+-n+\"A\"+n+\",\"+n+\" 0 1,\"+t+\" 0,\"+n}var e=hu,r=pu,i=su,u=ql,o=gu,a=vu,l=du;return n.innerRadius=function(t){return arguments.length?(e=En(t),n):e},n.outerRadius=function(t){return arguments.length?(r=En(t),n):r},n.cornerRadius=function(t){return arguments.length?(i=En(t),n):i},n.padRadius=function(t){return arguments.length?(u=t==ql?ql:En(t),n):u},n.startAngle=function(t){return arguments.length?(o=En(t),n):o},n.endAngle=function(t){return arguments.length?(a=En(t),n):a},n.padAngle=function(t){return arguments.length?(l=En(t),n):l},n.centroid=function(){var n=(+e.apply(this,arguments)+ +r.apply(this,arguments))/2,t=(+o.apply(this,arguments)+ +a.apply(this,arguments))/2-Io;return[Math.cos(t)*n,Math.sin(t)*n]},n};var ql=\"auto\";ao.svg.line=function(){return Mu(m)};var Tl=ao.map({linear:xu,\
\"linear-closed\":bu,step:_u,\"step-before\":wu,\"step-after\":Su,basis:zu,\"basis-open\":Lu,\"basis-closed\":qu,bundle:Tu,cardinal:Eu,\"cardinal-open\":ku,\"cardinal-closed\":Nu,monotone:Fu});Tl.forEach(function(n,t){t.key=n,t.closed=/-closed$/.test(n)});var Rl=[0,2/3,1/3,0],Dl=[0,1/3,2/3,0],Pl=[0,1/6,2/3,1/6];ao.svg.line.radial=function(){var n=Mu(Hu);return n.radius=n.x,delete n.x,n.angle=n.y,delete n.y,n},wu.reverse=Su,Su.reverse=wu,ao.svg.area=function(){return Ou(m)},ao.svg.area.radial=function(){var n=Ou(Hu);return n.radius=n.x,delete n.x,n.innerRadius=n.x0,delete n.x0,n.outerRadius=n.x1,delete n.x1,n.angle=n.y,delete n.y,n.startAngle=n.y0,delete n.y0,n.endAngle=n.y1,delete n.y1,n},ao.svg.chord=function(){function n(n,a){var l=t(this,u,n,a),c=t(this,o,n,a);return\"M\"+l.p0+r(l.r,l.p1,l.a1-l.a0)+(e(l,c)?i(l.r,l.p1,l.r,l.p0):i(l.r,l.p1,c.r,c.p0)+r(c.r,c.p1,c.a1-c.a0)+i(c.r,c.p1,l.r,l.p0))+\"Z\"}function t(n,t,e,r){var i=t.call(n,e,r),u=a.call(n,i,r),o=l.call(n,i,r)-Io,f=c.call(n,i,r)-Io;return{r:u,a0:o,a1:f,p0:[u*Math.cos(o),u*Math.sin(o)],p1:[u*Math.cos(f),u*Math.sin(f)]}}function e(n,t){return n.a0==t.a0&&n.a1==t.a1}function r(n,t,e){return\"A\"+n+\",\"+n+\" 0 \"+ +(e>Fo)+\",1 \"+t}function i(n,t,e,r){return\"Q 0,0 \"+r}var u=Me,o=xe,a=Iu,l=gu,c=vu;return n.radius=function(t){return arguments.length?(a=En(t),n):a},n.source=function(t){return arguments.length?(u=En(t),n):u},n.target=function(t){return arguments.length?(o=En(t),n):o},n.startAngle=function(t){return arguments.length?(l=En(t),n):l},n.endAngle=function(t){return arguments.length?(c=En(t),n):c},n},ao.svg.diagonal=function(){function n(n,i){var u=t.call(this,n,i),o=e.call(this,n,i),a=(u.y+o.y)/2,l=[u,{x:u.x,y:a},{x:o.x,y:a},o];return l=l.map(r),\"M\"+l[0]+\"C\"+l[1]+\" \"+l[2]+\" \"+l[3]}var t=Me,e=xe,r=Yu;return n.source=function(e){return arguments.length?(t=En(e),n):t},n.target=function(t){return arguments.length?(e=En(t),n):e},n.projection=function(t){return arguments.length?(r=t,n):r},n},ao.svg.diagonal.radial=function(){var n=ao.svg.diagonal(),t=Yu,e=n.projection;return n.projection=function(n){return \
arguments.length?e(Zu(t=n)):t},n},ao.svg.symbol=function(){function n(n,r){return(Ul.get(t.call(this,n,r))||$u)(e.call(this,n,r))}var t=Xu,e=Vu;return n.type=function(e){return arguments.length?(t=En(e),n):t},n.size=function(t){return arguments.length?(e=En(t),n):e},n};var Ul=ao.map({circle:$u,cross:function(n){var t=Math.sqrt(n/5)/2;return\"M\"+-3*t+\",\"+-t+\"H\"+-t+\"V\"+-3*t+\"H\"+t+\"V\"+-t+\"H\"+3*t+\"V\"+t+\"H\"+t+\"V\"+3*t+\"H\"+-t+\"V\"+t+\"H\"+-3*t+\"Z\"},diamond:function(n){var t=Math.sqrt(n/(2*Fl)),e=t*Fl;return\"M0,\"+-t+\"L\"+e+\",0 0,\"+t+\" \"+-e+\",0Z\"},square:function(n){var t=Math.sqrt(n)/2;return\"M\"+-t+\",\"+-t+\"L\"+t+\",\"+-t+\" \"+t+\",\"+t+\" \"+-t+\",\"+t+\"Z\"},\"triangle-down\":function(n){var t=Math.sqrt(n/jl),e=t*jl/2;return\"M0,\"+e+\"L\"+t+\",\"+-e+\" \"+-t+\",\"+-e+\"Z\"},\"triangle-up\":function(n){var t=Math.sqrt(n/jl),e=t*jl/2;return\"M0,\"+-e+\"L\"+t+\",\"+e+\" \"+-t+\",\"+e+\"Z\"}});ao.svg.symbolTypes=Ul.keys();var jl=Math.sqrt(3),Fl=Math.tan(30*Yo);Co.transition=function(n){for(var t,e,r=Hl||++Zl,i=Ku(n),u=[],o=Ol||{time:Date.now(),ease:Nr,delay:0,duration:250},a=-1,l=this.length;++a<l;){u.push(t=[]);for(var c=this[a],f=-1,s=c.length;++f<s;)(e=c[f])&&Qu(e,f,i,r,o),t.push(e)}return Wu(u,i,r)},Co.interrupt=function(n){return this.each(null==n?Il:Bu(Ku(n)))};var Hl,Ol,Il=Bu(Ku()),Yl=[],Zl=0;Yl.call=Co.call,Yl.empty=Co.empty,Yl.node=Co.node,Yl.size=Co.size,ao.transition=function(n,t){return n&&n.transition?Hl?n.transition(t):n:ao.selection().transition(n)},ao.transition.prototype=Yl,Yl.select=function(n){var t,e,r,i=this.id,u=this.namespace,o=[];n=A(n);for(var a=-1,l=this.length;++a<l;){o.push(t=[]);for(var c=this[a],f=-1,s=c.length;++f<s;)(r=c[f])&&(e=n.call(r,r.__data__,f,a))?(\"__data__\"in r&&(e.__data__=r.__data__),Qu(e,f,u,i,r[u][i]),t.push(e)):t.push(null)}return Wu(o,u,i)},Yl.selectAll=function(n){var t,e,r,i,u,o=this.id,a=this.namespace,l=[];n=C(n);for(var c=-1,f=this.length;++c<f;)for(var s=this[c],h=-1,p=s.length;++h<p;)if(r=s[h]){u=r[a][o],e=n.call(r,r.__data__,h,c),l.push(t=[]);for(var g=-1,v=e.length;++g<v;)(i=e[g])&&Qu(i,g,a,o,u),t.push(i)}return Wu(l,\
a,o)},Yl.filter=function(n){var t,e,r,i=[];\"function\"!=typeof n&&(n=O(n));for(var u=0,o=this.length;o>u;u++){i.push(t=[]);for(var e=this[u],a=0,l=e.length;l>a;a++)(r=e[a])&&n.call(r,r.__data__,a,u)&&t.push(r)}return Wu(i,this.namespace,this.id)},Yl.tween=function(n,t){var e=this.id,r=this.namespace;return arguments.length<2?this.node()[r][e].tween.get(n):Y(this,null==t?function(t){t[r][e].tween.remove(n)}:function(i){i[r][e].tween.set(n,t)})},Yl.attr=function(n,t){function e(){this.removeAttribute(a)}function r(){this.removeAttributeNS(a.space,a.local)}function i(n){return null==n?e:(n+=\"\",function(){var t,e=this.getAttribute(a);return e!==n&&(t=o(e,n),function(n){this.setAttribute(a,t(n))})})}function u(n){return null==n?r:(n+=\"\",function(){var t,e=this.getAttributeNS(a.space,a.local);return e!==n&&(t=o(e,n),function(n){this.setAttributeNS(a.space,a.local,t(n))})})}if(arguments.length<2){for(t in n)this.attr(t,n[t]);return this}var o=\"transform\"==n?$r:Mr,a=ao.ns.qualify(n);return Ju(this,\"attr.\"+n,t,a.local?u:i)},Yl.attrTween=function(n,t){function e(n,e){var r=t.call(this,n,e,this.getAttribute(i));return r&&function(n){this.setAttribute(i,r(n))}}function r(n,e){var r=t.call(this,n,e,this.getAttributeNS(i.space,i.local));return r&&function(n){this.setAttributeNS(i.space,i.local,r(n))}}var i=ao.ns.qualify(n);return this.tween(\"attr.\"+n,i.local?r:e)},Yl.style=function(n,e,r){function i(){this.style.removeProperty(n)}function u(e){return null==e?i:(e+=\"\",function(){var i,u=t(this).getComputedStyle(this,null).getPropertyValue(n);return u!==e&&(i=Mr(u,e),function(t){this.style.setProperty(n,i(t),r)})})}var o=arguments.length;if(3>o){if(\"string\"!=typeof n){2>o&&(e=\"\");for(r in n)this.style(r,n[r],e);return this}r=\"\"}return Ju(this,\"style.\"+n,e,u)},Yl.styleTween=function(n,e,r){function i(i,u){var o=e.call(this,i,u,t(this).getComputedStyle(this,null).getPropertyValue(n));return o&&function(t){this.style.setProperty(n,o(t),r)}}return arguments.length<3&&(r=\"\"),this.tween(\"style.\"+n,i)},Yl.text=function(n){return Ju(this,\"text\",\
n,Gu)},Yl.remove=function(){var n=this.namespace;return this.each(\"end.transition\",function(){var t;this[n].count<2&&(t=this.parentNode)&&t.removeChild(this)})},Yl.ease=function(n){var t=this.id,e=this.namespace;return arguments.length<1?this.node()[e][t].ease:(\"function\"!=typeof n&&(n=ao.ease.apply(ao,arguments)),Y(this,function(r){r[e][t].ease=n}))},Yl.delay=function(n){var t=this.id,e=this.namespace;return arguments.length<1?this.node()[e][t].delay:Y(this,\"function\"==typeof n?function(r,i,u){r[e][t].delay=+n.call(r,r.__data__,i,u)}:(n=+n,function(r){r[e][t].delay=n}))},Yl.duration=function(n){var t=this.id,e=this.namespace;return arguments.length<1?this.node()[e][t].duration:Y(this,\"function\"==typeof n?function(r,i,u){r[e][t].duration=Math.max(1,n.call(r,r.__data__,i,u))}:(n=Math.max(1,n),function(r){r[e][t].duration=n}))},Yl.each=function(n,t){var e=this.id,r=this.namespace;if(arguments.length<2){var i=Ol,u=Hl;try{Hl=e,Y(this,function(t,i,u){Ol=t[r][e],n.call(t,t.__data__,i,u)})}finally{Ol=i,Hl=u}}else Y(this,function(i){var u=i[r][e];(u.event||(u.event=ao.dispatch(\"start\",\"end\",\"interrupt\"))).on(n,t)});return this},Yl.transition=function(){for(var n,t,e,r,i=this.id,u=++Zl,o=this.namespace,a=[],l=0,c=this.length;c>l;l++){a.push(n=[]);for(var t=this[l],f=0,s=t.length;s>f;f++)(e=t[f])&&(r=e[o][i],Qu(e,f,o,u,{time:r.time,ease:r.ease,delay:r.delay+r.duration,duration:r.duration})),n.push(e)}return Wu(a,o,u)},ao.svg.axis=function(){function n(n){n.each(function(){var n,c=ao.select(this),f=this.__chart__||e,s=this.__chart__=e.copy(),h=null==l?s.ticks?s.ticks.apply(s,a):s.domain():l,p=null==t?s.tickFormat?s.tickFormat.apply(s,a):m:t,g=c.selectAll(\".tick\").data(h,s),v=g.enter().insert(\"g\",\".domain\").attr(\"class\",\"tick\").style(\"opacity\",Uo),d=ao.transition(g.exit()).style(\"opacity\",Uo).remove(),y=ao.transition(g.order()).style(\"opacity\",1),M=Math.max(i,0)+o,x=Zi(s),b=c.selectAll(\".domain\").data([0]),_=(b.enter().append(\"path\").attr(\"class\",\"domain\"),ao.transition(b));v.append(\"line\"),v.append(\"text\");var w,S,k,N,E=v.select(\"line\"),\
A=y.select(\"line\"),C=g.select(\"text\").text(p),z=v.select(\"text\"),L=y.select(\"text\"),q=\"top\"===r||\"left\"===r?-1:1;if(\"bottom\"===r||\"top\"===r?(n=no,w=\"x\",k=\"y\",S=\"x2\",N=\"y2\",C.attr(\"dy\",0>q?\"0em\":\".71em\").style(\"text-anchor\",\"middle\"),_.attr(\"d\",\"M\"+x[0]+\",\"+q*u+\"V0H\"+x[1]+\"V\"+q*u)):(n=to,w=\"y\",k=\"x\",S=\"y2\",N=\"x2\",C.attr(\"dy\",\".32em\").style(\"text-anchor\",0>q?\"end\":\"start\"),_.attr(\"d\",\"M\"+q*u+\",\"+x[0]+\"H0V\"+x[1]+\"H\"+q*u)),E.attr(N,q*i),z.attr(k,q*M),A.attr(S,0).attr(N,q*i),L.attr(w,0).attr(k,q*M),s.rangeBand){var T=s,R=T.rangeBand()/2;f=s=function(n){return T(n)+R}}else f.rangeBand?f=s:d.call(n,s,f);v.call(n,f,s),y.call(n,s,s)})}var t,e=ao.scale.linear(),r=Vl,i=6,u=6,o=3,a=[10],l=null;return n.scale=function(t){return arguments.length?(e=t,n):e},n.orient=function(t){return arguments.length?(r=t in Xl?t+\"\":Vl,n):r},n.ticks=function(){return arguments.length?(a=co(arguments),n):a},n.tickValues=function(t){return arguments.length?(l=t,n):l},n.tickFormat=function(e){return arguments.length?(t=e,n):t},n.tickSize=function(t){var e=arguments.length;return e?(i=+t,u=+arguments[e-1],n):i},n.innerTickSize=function(t){return arguments.length?(i=+t,n):i},n.outerTickSize=function(t){return arguments.length?(u=+t,n):u},n.tickPadding=function(t){return arguments.length?(o=+t,n):o},n.tickSubdivide=function(){return arguments.length&&n},n};var Vl=\"bottom\",Xl={top:1,right:1,bottom:1,left:1};ao.svg.brush=function(){function n(t){t.each(function(){var t=ao.select(this).style(\"pointer-events\",\"all\").style(\"-webkit-tap-highlight-color\",\"rgba(0,0,0,0)\").on(\"mousedown.brush\",u).on(\"touchstart.brush\",u),o=t.selectAll(\".background\").data([0]);o.enter().append(\"rect\").attr(\"class\",\"background\").style(\"visibility\",\"hidden\").style(\"cursor\",\"crosshair\"),t.selectAll(\".extent\").data([0]).enter().append(\"rect\").attr(\"class\",\"extent\").style(\"cursor\",\"move\");var a=t.selectAll(\".resize\").data(v,m);a.exit().remove(),a.enter().append(\"g\").attr(\"class\",function(n){return\"resize \"+n}).style(\"cursor\",function(n){return $l[n]}).append(\"rect\").attr(\"x\",function(n){return/[ew]$/.test(n)?-3:null}).attr(\"y\",\
function(n){return/^[ns]/.test(n)?-3:null}).attr(\"width\",6).attr(\"height\",6).style(\"visibility\",\"hidden\"),a.style(\"display\",n.empty()?\"none\":null);var l,s=ao.transition(t),h=ao.transition(o);c&&(l=Zi(c),h.attr(\"x\",l[0]).attr(\"width\",l[1]-l[0]),r(s)),f&&(l=Zi(f),h.attr(\"y\",l[0]).attr(\"height\",l[1]-l[0]),i(s)),e(s)})}function e(n){n.selectAll(\".resize\").attr(\"transform\",function(n){return\"translate(\"+s[+/e$/.test(n)]+\",\"+h[+/^s/.test(n)]+\")\"})}function r(n){n.select(\".extent\").attr(\"x\",s[0]),n.selectAll(\".extent,.n>rect,.s>rect\").attr(\"width\",s[1]-s[0])}function i(n){n.select(\".extent\").attr(\"y\",h[0]),n.selectAll(\".extent,.e>rect,.w>rect\").attr(\"height\",h[1]-h[0])}function u(){function u(){32==ao.event.keyCode&&(C||(M=null,L[0]-=s[1],L[1]-=h[1],C=2),S())}function v(){32==ao.event.keyCode&&2==C&&(L[0]+=s[1],L[1]+=h[1],C=0,S())}function d(){var n=ao.mouse(b),t=!1;x&&(n[0]+=x[0],n[1]+=x[1]),C||(ao.event.altKey?(M||(M=[(s[0]+s[1])/2,(h[0]+h[1])/2]),L[0]=s[+(n[0]<M[0])],L[1]=h[+(n[1]<M[1])]):M=null),E&&y(n,c,0)&&(r(k),t=!0),A&&y(n,f,1)&&(i(k),t=!0),t&&(e(k),w({type:\"brush\",mode:C?\"move\":\"resize\"}))}function y(n,t,e){var r,i,u=Zi(t),l=u[0],c=u[1],f=L[e],v=e?h:s,d=v[1]-v[0];return C&&(l-=f,c-=d+f),r=(e?g:p)?Math.max(l,Math.min(c,n[e])):n[e],C?i=(r+=f)+d:(M&&(f=Math.max(l,Math.min(c,2*M[e]-r))),r>f?(i=r,r=f):i=f),v[0]!=r||v[1]!=i?(e?a=null:o=null,v[0]=r,v[1]=i,!0):void 0}function m(){d(),k.style(\"pointer-events\",\"all\").selectAll(\".resize\").style(\"display\",n.empty()?\"none\":null),ao.select(\"body\").style(\"cursor\",null),q.on(\"mousemove.brush\",null).on(\"mouseup.brush\",null).on(\"touchmove.brush\",null).on(\"touchend.brush\",null).on(\"keydown.brush\",null).on(\"keyup.brush\",null),z(),w({type:\"brushend\"})}var M,x,b=this,_=ao.select(ao.event.target),w=l.of(b,arguments),k=ao.select(b),N=_.datum(),E=!/^(n|s)$/.test(N)&&c,A=!/^(e|w)$/.test(N)&&f,C=_.classed(\"extent\"),z=W(b),L=ao.mouse(b),q=ao.select(t(b)).on(\"keydown.brush\",u).on(\"keyup.brush\",v);if(ao.event.changedTouches?q.on(\"touchmove.brush\",d).on(\"touchend.brush\",m):q.on(\"mousemove.brush\",\
d).on(\"mouseup.brush\",m),k.interrupt().selectAll(\"*\").interrupt(),C)L[0]=s[0]-L[0],L[1]=h[0]-L[1];else if(N){var T=+/w$/.test(N),R=+/^n/.test(N);x=[s[1-T]-L[0],h[1-R]-L[1]],L[0]=s[T],L[1]=h[R]}else ao.event.altKey&&(M=L.slice());k.style(\"pointer-events\",\"none\").selectAll(\".resize\").style(\"display\",null),ao.select(\"body\").style(\"cursor\",_.style(\"cursor\")),w({type:\"brushstart\"}),d()}var o,a,l=N(n,\"brushstart\",\"brush\",\"brushend\"),c=null,f=null,s=[0,0],h=[0,0],p=!0,g=!0,v=Bl[0];return n.event=function(n){n.each(function(){var n=l.of(this,arguments),t={x:s,y:h,i:o,j:a},e=this.__chart__||t;this.__chart__=t,Hl?ao.select(this).transition().each(\"start.brush\",function(){o=e.i,a=e.j,s=e.x,h=e.y,n({type:\"brushstart\"})}).tween(\"brush:brush\",function(){var e=xr(s,t.x),r=xr(h,t.y);return o=a=null,function(i){s=t.x=e(i),h=t.y=r(i),n({type:\"brush\",mode:\"resize\"})}}).each(\"end.brush\",function(){o=t.i,a=t.j,n({type:\"brush\",mode:\"resize\"}),n({type:\"brushend\"})}):(n({type:\"brushstart\"}),n({type:\"brush\",mode:\"resize\"}),n({type:\"brushend\"}))})},n.x=function(t){return arguments.length?(c=t,v=Bl[!c<<1|!f],n):c},n.y=function(t){return arguments.length?(f=t,v=Bl[!c<<1|!f],n):f},n.clamp=function(t){return arguments.length?(c&&f?(p=!!t[0],g=!!t[1]):c?p=!!t:f&&(g=!!t),n):c&&f?[p,g]:c?p:f?g:null},n.extent=function(t){var e,r,i,u,l;return arguments.length?(c&&(e=t[0],r=t[1],f&&(e=e[0],r=r[0]),o=[e,r],c.invert&&(e=c(e),r=c(r)),e>r&&(l=e,e=r,r=l),e==s[0]&&r==s[1]||(s=[e,r])),f&&(i=t[0],u=t[1],c&&(i=i[1],u=u[1]),a=[i,u],f.invert&&(i=f(i),u=f(u)),i>u&&(l=i,i=u,u=l),i==h[0]&&u==h[1]||(h=[i,u])),n):(c&&(o?(e=o[0],r=o[1]):(e=s[0],r=s[1],c.invert&&(e=c.invert(e),r=c.invert(r)),e>r&&(l=e,e=r,r=l))),f&&(a?(i=a[0],u=a[1]):(i=h[0],u=h[1],f.invert&&(i=f.invert(i),u=f.invert(u)),i>u&&(l=i,i=u,u=l))),c&&f?[[e,i],[r,u]]:c?[e,r]:f&&[i,u])},n.clear=function(){return n.empty()||(s=[0,0],h=[0,0],o=a=null),n},n.empty=function(){return!!c&&s[0]==s[1]||!!f&&h[0]==h[1]},ao.rebind(n,l,\"on\")};var $l={n:\"ns-resize\",e:\"ew-resize\",s:\"ns-resize\",w:\"ew-resize\",nw:\"nwse-resize\",\
ne:\"nesw-resize\",se:\"nwse-resize\",sw:\"nesw-resize\"},Bl=[[\"n\",\"e\",\"s\",\"w\",\"nw\",\"ne\",\"se\",\"sw\"],[\"e\",\"w\"],[\"n\",\"s\"],[]],Wl=ga.format=xa.timeFormat,Jl=Wl.utc,Gl=Jl(\"%Y-%m-%dT%H:%M:%S.%LZ\");Wl.iso=Date.prototype.toISOString&&+new Date(\"2000-01-01T00:00:00.000Z\")?eo:Gl,eo.parse=function(n){var t=new Date(n);return isNaN(t)?null:t},eo.toString=Gl.toString,ga.second=On(function(n){return new va(1e3*Math.floor(n/1e3))},function(n,t){n.setTime(n.getTime()+1e3*Math.floor(t))},function(n){return n.getSeconds()}),ga.seconds=ga.second.range,ga.seconds.utc=ga.second.utc.range,ga.minute=On(function(n){return new va(6e4*Math.floor(n/6e4))},function(n,t){n.setTime(n.getTime()+6e4*Math.floor(t))},function(n){return n.getMinutes()}),ga.minutes=ga.minute.range,ga.minutes.utc=ga.minute.utc.range,ga.hour=On(function(n){var t=n.getTimezoneOffset()/60;return new va(36e5*(Math.floor(n/36e5-t)+t))},function(n,t){n.setTime(n.getTime()+36e5*Math.floor(t))},function(n){return n.getHours()}),ga.hours=ga.hour.range,ga.hours.utc=ga.hour.utc.range,ga.month=On(function(n){return n=ga.day(n),n.setDate(1),n},function(n,t){n.setMonth(n.getMonth()+t)},function(n){return n.getMonth()}),ga.months=ga.month.range,ga.months.utc=ga.month.utc.range;var Kl=[1e3,5e3,15e3,3e4,6e4,3e5,9e5,18e5,36e5,108e5,216e5,432e5,864e5,1728e5,6048e5,2592e6,7776e6,31536e6],Ql=[[ga.second,1],[ga.second,5],[ga.second,15],[ga.second,30],[ga.minute,1],[ga.minute,5],[ga.minute,15],[ga.minute,30],[ga.hour,1],[ga.hour,3],[ga.hour,6],[ga.hour,12],[ga.day,1],[ga.day,2],[ga.week,1],[ga.month,1],[ga.month,3],[ga.year,1]],nc=Wl.multi([[\".%L\",function(n){return n.getMilliseconds()}],[\":%S\",function(n){return n.getSeconds()}],[\"%I:%M\",function(n){return n.getMinutes()}],[\"%I %p\",function(n){return n.getHours()}],[\"%a %d\",function(n){return n.getDay()&&1!=n.getDate()}],[\"%b %d\",function(n){return 1!=n.getDate()}],[\"%B\",function(n){return n.getMonth()}],[\"%Y\",zt]]),tc={range:function(n,t,e){return ao.range(Math.ceil(n/e)*e,+t,e).map(io)},floor:m,ceil:m};Ql.year=ga.year,ga.scale=function(){return \
ro(ao.scale.linear(),Ql,nc)};var ec=Ql.map(function(n){return[n[0].utc,n[1]]}),rc=Jl.multi([[\".%L\",function(n){return n.getUTCMilliseconds()}],[\":%S\",function(n){return n.getUTCSeconds()}],[\"%I:%M\",function(n){return n.getUTCMinutes()}],[\"%I %p\",function(n){return n.getUTCHours()}],[\"%a %d\",function(n){return n.getUTCDay()&&1!=n.getUTCDate()}],[\"%b %d\",function(n){return 1!=n.getUTCDate()}],[\"%B\",function(n){return n.getUTCMonth()}],[\"%Y\",zt]]);ec.year=ga.year.utc,ga.scale.utc=function(){return ro(ao.scale.linear(),ec,rc)},ao.text=An(function(n){return n.responseText}),ao.json=function(n,t){return Cn(n,\"application/json\",uo,t)},ao.html=function(n,t){return Cn(n,\"text/html\",oo,t)},ao.xml=An(function(n){return n.responseXML}),\"function\"==typeof define&&define.amd?(this.d3=ao,define(ao)):\"object\"==typeof module&&module.exports?module.exports=ao:this.d3=ao}();
")
################################################################################
# end d3.js
################################################################################
