/*
 * SPDX-FileCopyrightText: Copyright (c) 2019-2022, NVIDIA CORPORATION.
 * SPDX-License-Identifier: Apache-2.0
 */
#ifndef __SPARSE_DEGREE_H
#define __SPARSE_DEGREE_H

#pragma once

#include <raft/sparse/coo.hpp>
#include <raft/sparse/linalg/detail/degree.cuh>

namespace raft {
namespace sparse {
namespace linalg {

/**
 * @brief Count the number of values for each row
 * @tparam TPB_X: number of threads to use per block
 * @param rows: rows array of the COO matrix
 * @param nnz: size of the rows array
 * @param results: output result array
 * @param stream: cuda stream to use
 */
template <typename T = int, typename nnz_type, typename outT>
void coo_degree(const T* rows, nnz_type nnz, outT* results, cudaStream_t stream)
{
  detail::coo_degree<64, T>(rows, nnz, results, stream);
}

/**
 * @brief Count the number of values for each row
 * @tparam TPB_X: number of threads to use per block
 * @tparam T: type name of underlying values array
 * @param in: input COO object for counting rows
 * @param results: output array with row counts (size=in->n_rows)
 * @param stream: cuda stream to use
 */
template <typename T, typename outT>
void coo_degree(COO<T>* in, outT* results, cudaStream_t stream)
{
  coo_degree(in->rows(), in->nnz, results, stream);
}

/**
 * @brief Count the number of values for each row that doesn't match a particular scalar
 * @tparam TPB_X: number of threads to use per block
 * @tparam T: the type name of the underlying value arrays
 * @param rows: Input COO row array
 * @param vals: Input COO val arrays
 * @param nnz: size of input COO arrays
 * @param scalar: scalar to match for counting rows
 * @param results: output row counts
 * @param stream: cuda stream to use
 */
template <typename T, typename idx_t, typename nnz_type, typename outT>
void coo_degree_scalar(
  const idx_t* rows, const T* vals, nnz_type nnz, T scalar, outT* results, cudaStream_t stream = 0)
{
  detail::coo_degree_scalar<64, T, idx_t, outT, nnz_type>(rows, vals, nnz, scalar, results, stream);
}

/**
 * @brief Count the number of values for each row that doesn't match a particular scalar
 * @tparam TPB_X: number of threads to use per block
 * @tparam T: the type name of the underlying value arrays
 * @param in: Input COO array
 * @param scalar: scalar to match for counting rows
 * @param results: output row counts
 * @param stream: cuda stream to use
 */
template <typename T, typename outT>
void coo_degree_scalar(COO<T>* in, T scalar, outT* results, cudaStream_t stream)
{
  coo_degree_scalar(in->rows(), in->vals(), in->nnz, scalar, results, stream);
}

/**
 * @brief Count the number of nonzeros for each row
 * @tparam TPB_X: number of threads to use per block
 * @tparam T: the type name of the underlying value arrays
 * @param rows: Input COO row array
 * @param vals: Input COO val arrays
 * @param nnz: size of input COO arrays
 * @param results: output row counts
 * @param stream: cuda stream to use
 */
template <typename T>
void coo_degree_nz(const int* rows, const T* vals, int nnz, int* results, cudaStream_t stream)
{
  detail::coo_degree_nz<64>(rows, vals, nnz, results, stream);
}

/**
 * @brief Count the number of nonzero values for each row
 * @tparam TPB_X: number of threads to use per block
 * @tparam T: the type name of the underlying value arrays
 * @param in: Input COO array
 * @param results: output row counts
 * @param stream: cuda stream to use
 */
template <typename T>
void coo_degree_nz(COO<T>* in, int* results, cudaStream_t stream)
{
  coo_degree_nz(in->rows(), in->vals(), in->nnz, results, stream);
}

};  // end NAMESPACE linalg
};  // end NAMESPACE sparse
};  // end NAMESPACE raft

#endif
