# SPDX-FileCopyrightText: Copyright (c) 2019-2025, NVIDIA CORPORATION.
# SPDX-License-Identifier: Apache-2.0

from collections.abc import Buffer
from typing import Any, NotRequired, Optional, Protocol, TypedDict, overload

import numpy as np
from numpy.typing import NDArray

from rmm.pylibrmm.memory_resource import DeviceMemoryResource
from rmm.pylibrmm.stream import Stream

class _CUDAArrayInterface(TypedDict):
    data: tuple[int, bool]
    shape: tuple[int, ...]
    typestr: str
    version: int
    strides: NotRequired[tuple[int, ...] | None]
    descr: NotRequired[list[tuple]]

class SupportsCUDAArrayInterface(Protocol):
    """Protocol for objects that implement the CUDA array interface."""

    @property
    def __cuda_array_interface__(self) -> dict[str, Any]: ...

class DeviceBuffer:
    mr: DeviceMemoryResource
    stream: Stream

    def __init__(
        self,
        *,
        ptr: int = ...,
        size: int = ...,
        stream: Stream = ...,
        mr: Optional[DeviceMemoryResource] = None,
    ) -> None: ...
    def __len__(self) -> int: ...
    def __sizeof__(self) -> int: ...
    def __bytes__(self) -> bytes: ...
    @property
    def nbytes(self) -> int: ...
    @property
    def ptr(self) -> int: ...
    @property
    def size(self) -> int: ...
    @property
    def __cuda_array_interface__(self) -> _CUDAArrayInterface: ...
    def prefetch(
        self, device: Optional[int] = ..., stream: Optional[Stream] = ...
    ) -> None: ...
    def copy(self) -> DeviceBuffer: ...
    def __copy__(self) -> DeviceBuffer: ...
    @staticmethod
    def to_device(b: Buffer, stream: Stream = ...) -> DeviceBuffer: ...
    @overload
    def copy_to_host(
        self, ary: None = None, stream: Stream = ...
    ) -> NDArray[np.uint8]: ...
    @overload
    def copy_to_host(self, ary: Buffer, stream: Stream = ...) -> Buffer: ...
    def copy_from_host(self, ary: Buffer, stream: Stream = ...) -> None: ...
    def copy_from_device(
        self, cuda_ary: SupportsCUDAArrayInterface, stream: Stream = ...
    ) -> None: ...
    def tobytes(self, stream: Stream = ...) -> bytes: ...
    def reserve(self, new_capacity: int, stream: Stream = ...) -> None: ...
    def resize(self, new_size: int, stream: Stream = ...) -> None: ...
    def capacity(self) -> int: ...

def to_device(b: Buffer, stream: Stream = ...) -> DeviceBuffer: ...
def copy_ptr_to_host(db: int, hb: Buffer, stream: Stream = ...) -> None: ...
def copy_host_to_ptr(hb: Buffer, db: int, stream: Stream = ...) -> None: ...
def copy_device_to_ptr(
    d_src: int, d_dst: int, count: int, stream: Stream = ...
) -> None: ...
