from typing import Optional

from anaconda_auth import login as anaconda_login
from anaconda_auth import logout as anaconda_logout
from anaconda_auth.actions import get_api_key
from anaconda_auth.exceptions import TokenNotFoundError
from anaconda_auth.handlers import shutdown_all_servers
from anaconda_auth.token import TokenInfo as TokenInfo

from aext_shared.config import SHARED_CONFIG
from aext_shared.consts import UserAccessCredentials
from aext_shared.errors import UnauthorizedError

config = SHARED_CONFIG


class AuthService:
    def login(self):
        """
        Attempts to retrieve an API key for authentication.

        This method first tries to obtain an API key from the current session.
        If this fails, it forcibly shuts down all servers, retries the login process,
        and then attempts to retrieve the API key again. If both attempts fail,
        it returns None to indicate that a valid API key cannot be obtained.

        Returns:
            str: The retrieved API key if successful, or None if authentication fails.
        """
        if token_info := self.get_token():
            return token_info.api_key

        try:
            shutdown_all_servers()
            # Logs in on ananconda.com, not anaconda.cloud
            anaconda_login()

            if token_info := self.get_token():
                return token_info.api_key
        except (TokenNotFoundError, Exception):
            return None

    def logout(self) -> None:
        """
        Logs out of anaconda.com and fallback on anaconda.cloud.

        NOTE: we don't use anaconda_auth.logout because this one fails silently.

        :return: None
        """
        token_info = self.get_token()
        if token_info and hasattr(token_info, "api_key"):
            anaconda_logout()
        return None

    def get_token(self) -> Optional[str]:
        """
        Retrieves the API key for anaconda.com and fallback on anaconda.cloud.

        Returns:
            str: The loaded API key or None.
        """
        try:
            return TokenInfo.load()
        except TokenNotFoundError:
            return None

    def create_token(self, access_credentials: UserAccessCredentials, save_to_keyring: bool = False) -> Optional[str]:
        """
        Creates an API key and stores it in the keyring file. It generates keys according to
        the domain configured within the global configuration object.

        Args:
            access_credentials (UserAccessCredentials): An object containing the user's auth
                information (i.e. access token, username, etc.);
            save_to_keyring (bool): A flag that determines whether the generated token should
                be added to the user's keyring file.

        Returns:
            str: The newly-generated API key.
        """
        if not access_credentials.access_token:
            raise UnauthorizedError({"reason": "missing access token"})

        token = get_api_key(access_credentials.access_token)

        if save_to_keyring:
            domain = SHARED_CONFIG["anaconda_auth"]["domain"]
            TokenInfo(domain=domain, api_key=token, username=access_credentials.username).save()

        return token


auth_service = AuthService()
