import sys
from datetime import datetime
from typing import Any

import pandas as pd
import pytest

import fugue.api as fe
import fugue.test as ft
from fugue import ArrowDataFrame
from fugue.exceptions import FugueDataFrameOperationError
from fugue_test.dataframe_suite import DataFrameTests

from .mock.dataframe import MockDuckDataFrame
from .mock.tester import mockibisduck_session  # noqa: F401  # pylint: disable-all
from uuid import uuid4


@ft.fugue_test_suite("mockibisduck", mark_test=True)
class IbisDataFrameTests(DataFrameTests.Tests):
    def df(self, data: Any = None, schema: Any = None) -> MockDuckDataFrame:
        df = ArrowDataFrame(data, schema)
        name = "_" + str(uuid4())[:5]
        con = self.context.engine.sql_engine.backend
        con.create_table(name, df.native, overwrite=True)
        return MockDuckDataFrame(con.table(name), schema=schema)

    def test_init_df(self):
        df = self.df([["x", 1]], "a:str,b:int")
        df = MockDuckDataFrame(df.native, "a:str,b:long")
        assert df.schema == "a:str,b:long"

    def test_is_local(self):
        df = self.df([["x", 1]], "a:str,b:int")
        assert not fe.is_local(df)
        assert fe.is_bounded(df)

    def test_map_type(self):
        pass

    def test_as_arrow(self):
        # empty
        df = self.df([], "a:int,b:int")
        assert [] == list(ArrowDataFrame(df.as_arrow()).as_dict_iterable())
        # pd.Nat
        df = self.df([[pd.NaT, 1]], "a:datetime,b:int")
        assert [dict(a=None, b=1)] == list(
            ArrowDataFrame(df.as_arrow()).as_dict_iterable()
        )
        # pandas timestamps
        df = self.df([[pd.Timestamp("2020-01-01"), 1]], "a:datetime,b:int")
        assert [dict(a=datetime(2020, 1, 1), b=1)] == list(
            ArrowDataFrame(df.as_arrow()).as_dict_iterable()
        )

    def test_deep_nested_types(self):
        pass

    def test_list_type(self):
        pass

    def test_native_table(self):
        df = self.df([["x", 1]], "a:str,b:int").native
        assert fe.get_schema(fe.rename(df, dict())) == "a:str,b:int"
        assert fe.get_schema(fe.rename(df, dict(a="c"))) == "c:str,b:int"

        with pytest.raises(Exception):
            fe.rename(df, dict(a="b"))

        with pytest.raises(FugueDataFrameOperationError):
            fe.rename(df, dict(x="y"))

        assert fe.get_schema(fe.drop_columns(df, [])) == "a:str,b:int"
        assert fe.get_schema(fe.drop_columns(df, ["a"])) == "b:int"

        with pytest.raises(FugueDataFrameOperationError):
            fe.get_schema(fe.drop_columns(df, ["a", "b"]))

        with pytest.raises(FugueDataFrameOperationError):
            fe.get_schema(fe.drop_columns(df, ["a", "c"]))
