#
# Copyright (c) 2012-2025 Snowflake Computing Inc. All rights reserved.
#
"""
Scala UDF utilities for Snowpark Connect.

This module provides utilities for creating and managing Scala User-Defined Functions (UDFs)
in Snowflake through Snowpark Connect. It handles the conversion between different type systems
(Snowpark, Scala, Snowflake, Spark protobuf) and generates the necessary SQL DDL statements
for UDF creation.

Key components:
- ScalaUdf: Reference class for Scala UDFs
- ScalaUDFDef: Definition class for Scala UDF creation
- Type mapping functions for different type systems
- UDF creation and management utilities
"""
from dataclasses import dataclass
from typing import List, Union

import snowflake.snowpark.types as snowpark_type
import snowflake.snowpark_connect.includes.python.pyspark.sql.connect.proto.types_pb2 as types_proto
from snowflake.snowpark_connect.error.error_codes import ErrorCodes
from snowflake.snowpark_connect.error.error_utils import attach_custom_error_code
from snowflake.snowpark_connect.type_mapping import map_type_to_snowflake_type
from snowflake.snowpark_connect.utils.jvm_udf_utils import (
    NullHandling,
    Param,
    ReturnType,
    Signature,
    build_jvm_udxf_imports,
)
from snowflake.snowpark_connect.utils.snowpark_connect_logging import logger
from snowflake.snowpark_connect.utils.udf_utils import (
    ProcessCommonInlineUserDefinedFunction,
)

# Prefix used for internally generated Scala UDF names to avoid conflicts
CREATE_SCALA_UDF_PREFIX = "__SC_BUILD_IN_CREATE_UDF_SCALA_"


class ScalaUdf:
    """
    Reference class for Scala UDFs, providing similar properties like Python UserDefinedFunction.

    This class serves as a lightweight reference to a Scala UDF that has been created
    in Snowflake, storing the essential metadata needed for function calls.
    """

    def __init__(
        self,
        name: str,
        input_types: List[snowpark_type.DataType],
        return_type: snowpark_type.DataType,
    ) -> None:
        """
        Initialize a Scala UDF reference.

        Args:
            name: The name of the UDF in Snowflake
            input_types: List of input parameter types
            return_type: The return type of the UDF
        """
        self.name = name
        self._input_types = input_types
        self._return_type = return_type


@dataclass(frozen=True)
class ScalaUDFDef:
    """
    Complete definition for creating a Scala UDF in Snowflake.

    Contains all the information needed to generate the CREATE FUNCTION SQL statement
    and the Scala code body for the UDF.

    Attributes:
        name: UDF name
        signature: SQL signature (for Snowflake function definition)
        scala_signature: Scala signature (for Scala code generation)
        imports: List of JAR files to import
        null_handling: Null handling behavior (defaults to RETURNS_NULL_ON_NULL_INPUT)
    """

    name: str
    signature: Signature
    scala_signature: Signature
    scala_invocation_args: List[str]
    imports: List[str]
    null_handling: NullHandling = NullHandling.RETURNS_NULL_ON_NULL_INPUT

    # -------------------- DDL Emitter --------------------

    def _gen_body_scala(self) -> str:
        """
        Generate the Scala code body for the UDF.

        Creates a Scala object that loads the serialized function from a binary file
        and provides a run method to execute it.

        Returns:
            String containing the complete Scala code for the UDF body
        """
        # Convert Array to Seq for Scala compatibility in function signatures.
        # Replace each "Variant" type with "Any" in the function signature since fromVariant returns Any
        udf_func_input_types = ", ".join(
            "Any" if p.data_type == "Variant" else p.data_type.replace("Array", "Seq")
            for p in self.scala_signature.params
        )
        udf_func_return_type = self.scala_signature.returns.data_type.replace(
            "Array", "Seq"
        )

        # Create the Scala arguments and input types string: "arg0: Type0, arg1: Type1, ...".
        joined_wrapper_arg_and_input_types_str = ", ".join(
            f"{p.name}: {p.data_type}" for p in self.scala_signature.params
        )

        # All Scala UDFs return Variant to ensure consistency and avoid type conversion issues.
        wrapper_return_type = "Variant"
        wrapped_args = [
            f"udfPacket.fromVariant({arg}, {i})" if p.data_type == "Variant" else arg
            for i, (arg, p) in enumerate(
                zip(self.scala_invocation_args, self.scala_signature.params)
            )
        ]
        invocation_args = ", ".join(wrapped_args)
        invoke_udf_func = f"func({invocation_args})"

        # Always wrap the result in Utils.toVariant() to ensure all Scala UDFs return Variant
        invoke_udf_func = f"Utils.toVariant({invoke_udf_func})"

        return f"""
import org.apache.spark.sql.connect.common.UdfPacket
import com.snowflake.sas.scala.UdfPacketUtils._
import com.snowflake.sas.scala.Utils
import com.snowflake.snowpark_java.types.Variant

object __RecreatedSparkUdf {{
  private lazy val udfPacket: UdfPacket = Utils.deserializeUdfPacket("{self.name}.bin")
  private lazy val func: ({udf_func_input_types}) => {udf_func_return_type} = udfPacket.function.asInstanceOf[({udf_func_input_types}) => {udf_func_return_type}]

  def __wrapperFunc({joined_wrapper_arg_and_input_types_str}): {wrapper_return_type} = {{
    {invoke_udf_func}
  }}
}}
"""

    def to_create_function_sql(self) -> str:
        """
        Generate the complete CREATE FUNCTION SQL statement for the Scala UDF.

        Creates a Snowflake CREATE OR REPLACE TEMPORARY FUNCTION statement with
        all necessary clauses including language, runtime version, packages,
        imports, and the Scala code body.

        Returns:
            Complete SQL DDL statement for creating the UDF
        """
        # self.validate()

        args = ", ".join(f"{p.name} {p.data_type}" for p in self.signature.params)
        ret_type = self.signature.returns.data_type

        def quote_single(s: str) -> str:
            """Helper function to wrap strings in single quotes for SQL."""
            return "'" + s + "'"

        # Handler and imports
        imports_sql = f"IMPORTS = ({', '.join(quote_single(x) for x in self.imports)})"

        return f"""
CREATE OR REPLACE TEMPORARY FUNCTION {self.name}({args})
RETURNS {ret_type}
LANGUAGE SCALA
{self.null_handling.value}
RUNTIME_VERSION = 2.12
PACKAGES = ('com.snowflake:snowpark:latest')
{imports_sql}
HANDLER = '__RecreatedSparkUdf.__wrapperFunc'
AS
$$
{self._gen_body_scala()}
$$;"""


def create_scala_udf(pciudf: ProcessCommonInlineUserDefinedFunction) -> ScalaUdf:
    """
    Create a Scala UDF in Snowflake from a ProcessCommonInlineUserDefinedFunction object.

    This function handles the complete process of creating a Scala UDF:
    1. Generates a unique function name if not provided
    2. Checks for existing UDFs in the session cache
    3. Creates the necessary imports list
    4. Maps types between different systems (Snowpark, Scala, Snowflake)
    5. Generates and executes the CREATE FUNCTION SQL statement

    If the UDF already exists in the session cache, it will be reused.

    Args:
        pciudf: The ProcessCommonInlineUserDefinedFunction object containing UDF details.

    Returns:
        A ScalaUdf object representing the created or cached Scala UDF.
    """
    from snowflake.snowpark_connect.resources_initializer import (
        ensure_scala_udf_jars_uploaded,
    )

    # Lazily upload Scala UDF jars on-demand when a Scala UDF is actually created.
    # This is thread-safe and will only upload once even if multiple threads call it.
    ensure_scala_udf_jars_uploaded()

    function_name = pciudf._function_name
    # If a function name is not provided, hash the binary file and use the first ten characters as the function name.
    if not function_name:
        import hashlib

        function_name = hashlib.sha256(pciudf._payload).hexdigest()[:10]
    udf_name = CREATE_SCALA_UDF_PREFIX + function_name

    # In case the Scala UDF was created with `spark.udf.register`, the Spark Scala input types (from protobuf) are
    # stored in pciudf.scala_input_types.
    # We cannot rely solely on the inputTypes field from the Scala UDF or the Snowpark input types, since:
    # - spark.udf.register arguments come from the inputTypes field
    # - UDFs created with a data type (like below) do not populate the inputTypes field. This requires the input types
    #   inferred by Snowpark. e.g.: udf((i: Long) => (i + 1).toInt, IntegerType)
    input_types = (
        pciudf._scala_input_types if pciudf._scala_input_types else pciudf._input_types
    )

    scala_input_params: List[Param] = []
    sql_input_params: List[Param] = []
    scala_invocation_args: List[str] = []  # arguments passed into the udf function
    if input_types:  # input_types can be None when no arguments are provided
        for i, input_type in enumerate(input_types):
            param_name = "arg" + str(i)
            # Create the Scala arguments and input types string: "arg0: Type0, arg1: Type1, ...".
            scala_input_params.append(
                Param(param_name, _map_type_to_scala_type(input_type, is_input=True))
            )
            # Create the Snowflake SQL arguments and input types string: "arg0 TYPE0, arg1 TYPE1, ...".
            # For arrays and structs, use VARIANT type in SQL signature
            is_snowpark_type = isinstance(input_type, snowpark_type.DataType)
            is_array = (
                is_snowpark_type and isinstance(input_type, snowpark_type.ArrayType)
            ) or (not is_snowpark_type and input_type.WhichOneof("kind") == "array")
            is_map = (
                is_snowpark_type and isinstance(input_type, snowpark_type.MapType)
            ) or (not is_snowpark_type and input_type.WhichOneof("kind") == "map")
            sql_type = (
                "VARIANT"
                if is_array or is_map
                else map_type_to_snowflake_type(input_type)
            )
            sql_input_params.append(Param(param_name, sql_type))
            # In the case of Map input types, we need to cast the argument to the correct type in Scala.
            scala_invocation_args.append(param_name)

    scala_return_type = _map_type_to_scala_type(
        pciudf._original_return_type, is_input=False
    )
    # All Scala UDFs now return VARIANT to ensure consistency and avoid type conversion issues.
    # The actual type conversion is handled after the UDF is called.
    from snowflake.snowpark_connect.utils.session import get_or_create_snowpark_session

    session = get_or_create_snowpark_session()
    imports = build_jvm_udxf_imports(
        session,
        pciudf._payload,
        udf_name,
    )
    sql_return_type = "VARIANT"

    udf_def = ScalaUDFDef(
        name=udf_name,
        signature=Signature(
            params=sql_input_params, returns=ReturnType(sql_return_type)
        ),
        imports=imports,
        scala_signature=Signature(
            params=scala_input_params, returns=ReturnType(scala_return_type)
        ),
        scala_invocation_args=scala_invocation_args,
    )
    create_udf_sql = udf_def.to_create_function_sql()
    logger.info(f"Creating Scala UDF: {create_udf_sql}")
    session.sql(create_udf_sql).collect()
    return ScalaUdf(udf_name, pciudf._input_types, pciudf._return_type)


def _map_type_to_scala_type(
    t: Union[snowpark_type.DataType, types_proto.DataType], is_input: bool = False
) -> str:
    """Maps a Snowpark or Spark protobuf type to a Scala type string.

    Args:
        t: The type to map
        is_input: If True, maps array types to Variant (for UDF inputs).
                  If False, maps array types to Array[ElementType] (for UDF outputs).
    """
    if not t:
        return "String"
    is_snowpark_type = isinstance(t, snowpark_type.DataType)
    condition = type(t) if is_snowpark_type else t.WhichOneof("kind")
    match condition:
        case snowpark_type.ArrayType | "array":
            if is_input:
                return "Variant"
            else:
                return (
                    f"Array[{_map_type_to_scala_type(t.element_type, is_input=False)}]"
                    if is_snowpark_type
                    else f"Array[{_map_type_to_scala_type(t.array.element_type, is_input=False)}]"
                )
        case snowpark_type.BinaryType | "binary":
            return "Array[Byte]"
        case snowpark_type.BooleanType | "boolean":
            return "Boolean"
        case snowpark_type.ByteType | "byte":
            return "Byte"
        case snowpark_type.DateType | "date":
            return "java.sql.Date"
        case snowpark_type.DecimalType | "decimal":
            return "java.math.BigDecimal"
        case snowpark_type.DoubleType | "double":
            return "Double"
        case snowpark_type.FloatType | "float":
            return "Float"
        case snowpark_type.GeographyType:
            return "Geography"
        case snowpark_type.IntegerType | "integer":
            return "Int"
        case snowpark_type.LongType | "long":
            return "Long"
        case snowpark_type.MapType | "map":
            if is_input:
                return "Variant"
            key_type = (
                _map_type_to_scala_type(t.key_type)
                if is_snowpark_type
                else _map_type_to_scala_type(t.map.key_type)
            )
            value_type = (
                _map_type_to_scala_type(t.value_type)
                if is_snowpark_type
                else _map_type_to_scala_type(t.map.value_type)
            )
            return f"Map[{key_type}, {value_type}]"
        case snowpark_type.NullType | "null":
            return "String"  # cannot set the return type to Null in Snowpark Scala UDFs
        case snowpark_type.ShortType | "short":
            return "Short"
        case snowpark_type.StringType | "string" | "char" | "varchar":
            return "String"
        case snowpark_type.StructType | "struct":
            return "Variant"
        case snowpark_type.TimestampType | "timestamp" | "timestamp_ntz":
            return "java.sql.Timestamp"
        case snowpark_type.VariantType:
            return "Variant"
        case _:
            exception = ValueError(f"Unsupported Snowpark type: {t}")
            attach_custom_error_code(exception, ErrorCodes.UNSUPPORTED_TYPE)
            raise exception
