import asyncio
import functools
import json
import os
import platform
import shutil
import subprocess
from pathlib import Path
from typing import Optional, Tuple

from aext_environments_server.consts import (
    CONDA_KERNEL_SPEC_PREFIX,
    WINDOWS_STORAGE_DRIVER,
)
from aext_environments_server.utils import snake_to_human

from aext_shared import Environment
from aext_shared.config import SHARED_CONFIG


class CondaCommandsWrapper:
    @classmethod
    def find_conda(cls) -> str:
        """
        This function attempts to locate the best candidate for the conda executable path.

        On Windows, due to potential issues with bat files initialized via shell hooks,
        finding the conda executable can be problematic.
        The method checks common installation paths such as ProgramData, Users directory
        under Anaconda3 and Miniconda3, and raises a RuntimeError if it fails to determine
        the conda path.

        Returns:
            str: The full path to the conda executable if found; otherwise, raises a RuntimeError.

        Raises:
            RuntimeError: If the conda executable cannot be located on Windows.
        """
        conda_path = shutil.which("conda")
        if conda_path:
            return conda_path

        # Windows Common installation path - this is a best effort approach, we may fail
        if platform.system().upper() == "Windows".upper():
            # Get the system drive letter (typically C:)
            system_drive = os.environ.get("SystemDrive", WINDOWS_STORAGE_DRIVER)

            possible_paths = [
                f"{system_drive}\\ProgramData\\Anaconda3\\Scripts\\conda.exe",
                f"{system_drive}\\Users\\%USERNAME%\\Anaconda3\\Scripts\\conda.exe",
                f"{system_drive}\\Miniconda3\\Scripts\\conda.exe",
                f"{system_drive}\\Users\\%USERNAME%\\Miniconda3\\Scripts\\conda.exe",
            ]

            # Add additional possible paths with other common drive letters if needed
            # This is optional and only needed if installations on non-system drives are common
            if system_drive != WINDOWS_STORAGE_DRIVER:
                possible_paths.extend(
                    [
                        f"{WINDOWS_STORAGE_DRIVER}\\ProgramData\\Anaconda3\\Scripts\\conda.exe",
                        f"{WINDOWS_STORAGE_DRIVER}\\Users\\%USERNAME%\\Anaconda3\\Scripts\\conda.exe",
                        f"{WINDOWS_STORAGE_DRIVER}\\Miniconda3\\Scripts\\conda.exe",
                        f"{WINDOWS_STORAGE_DRIVER}\\Users\\%USERNAME%\\Miniconda3\\Scripts\\conda.exe",
                    ]
                )

            for path in possible_paths:
                resolved = os.path.expandvars(path)
                if os.path.exists(resolved):
                    return resolved

        raise RuntimeError("Failed to determine conda path")

    @classmethod
    def list_environments(cls) -> subprocess.CompletedProcess:
        """List all environments in the local machine"""
        result = subprocess.run([cls.find_conda(), "env", "list", "--json"], capture_output=True, text=True, check=True)
        return result

    @classmethod
    def is_installed(cls, listed_environments: subprocess.CompletedProcess, env_name: str) -> bool:
        """
        Check if an environment is installed in the local machine
        """
        try:
            data = json.loads(listed_environments.stdout)
            env_paths = data.get("envs", [])
            # Each entry is a full path to an environment. Compare the basename with env_name.
            for path in env_paths:
                if os.path.basename(path) == env_name:
                    return True
            return False
        except subprocess.CalledProcessError as e:
            print("Error running conda command:", e)
            return False

    @classmethod
    async def _run_in_thread(cls, func, *args, **kwargs) -> Optional[subprocess.Popen]:
        """Run blocking subprocess in a thread to keep async behavior."""
        try:
            loop = asyncio.get_running_loop()
            return await loop.run_in_executor(None, functools.partial(func, *args, **kwargs))
        except RuntimeError:
            return None

    @classmethod
    async def _create_subprocess(cls, command_args: Tuple) -> subprocess.Popen:
        """
        Runs a subprocess using Popen in a background thread.
        Args:
            command_args: tuple containing the process command arguments. It can be one single string or multiple arguments
        """

        process = await cls._run_in_thread(
            subprocess.Popen,
            " ".join(command_args),
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
            bufsize=1,
            universal_newlines=True,
            text=True,
            shell=True,
        )
        return process

    @classmethod
    async def _get_env_prefix(cls, env_name: str):
        if Environment.is_cloud(SHARED_CONFIG["environment"]):
            # PythonAnywhere locked the volume where conda stores are created by default
            # for custom conda environments it has to be installed at users' home
            home = Path.home()
            return str(home / ".conda" / "envs" / env_name)

        default_path_result = await cls.get_default_conda_envs_path()
        if default_path_result.returncode is not None and default_path_result.returncode != 0:
            raise RuntimeError(f"Failed to get conda envs_dirs: {default_path_result.stderr}")

        info = json.loads(default_path_result.stdout.read())
        try:
            dir_env = info.get("envs_dirs", [])[0]
            dir_env = Path(dir_env)
        except (IndexError, TypeError):
            raise RuntimeError("Failed to determine conda envs_dirs")

        return str(dir_env / env_name)

    @classmethod
    async def install_environment(cls, env_name: str, file_path: str) -> Optional[subprocess.Popen]:
        env_prefix = await cls._get_env_prefix(env_name)
        command_args = (cls.find_conda(), "env", "create", "-v", "-f", file_path, "--prefix", env_prefix)
        return await cls._create_subprocess(command_args)

    @classmethod
    async def remove_environment(cls, env_name: str) -> Optional[subprocess.Popen]:
        env_prefix = await cls._get_env_prefix(env_name)
        command_args = (cls.find_conda(), "env", "remove", "-v", "-y", "--prefix", env_prefix)
        return await cls._create_subprocess(command_args)

    @classmethod
    async def install_kernel(cls, env_name: str) -> Optional[subprocess.Popen]:
        env_prefix = await cls._get_env_prefix(env_name)
        command_args = (
            cls.find_conda(),
            "run",
            "--prefix",
            env_prefix,
            "python",
            "-m",
            "ipykernel",
            "install",
            "--user",
            "--name",
            f"{CONDA_KERNEL_SPEC_PREFIX}{env_name}",
            "--display-name",
            snake_to_human(env_name),
        )
        return await cls._create_subprocess(command_args)

    @classmethod
    async def remove_kernel(cls, env_name: str) -> Optional[subprocess.Popen]:
        command_args = ("python", "-m", "jupyter", "kernelspec", "uninstall", f"conda-env-{env_name}", "-y")
        return await cls._create_subprocess(command_args)

    @classmethod
    async def conda_clean(cls) -> Optional[subprocess.Popen]:
        command_args = (cls.find_conda(), "clean", "--all")
        return await cls._create_subprocess(command_args)

    @classmethod
    async def get_default_conda_envs_path(cls):
        command_args = (cls.find_conda(), "info", "--json")
        return await cls._create_subprocess(command_args)
