import os
from typing import Dict, Optional

import yaml


def get_conda_environment_file_path(environment_name) -> str:
    current_dir = os.path.dirname(os.path.abspath(__file__))
    # Build the file path relative to the current file's location
    file_path = os.path.join(current_dir, "conda_environments", f"{environment_name}.yml")
    return file_path


def extract_packages_from_env_file(environment_name, file_content: Optional[bytes] = None) -> Dict:
    """
    Extracts the list of packages from a conda environment YAML file.

    Returns a dictionary with two keys:
      - "conda": a list of packages specified for conda installation.
      - "pip": a list of pip packages (if any).
    """
    if file_content:
        env_data = yaml.safe_load(file_content.decode("utf-8"))
    else:
        file_path = get_conda_environment_file_path(environment_name)
        try:
            with open(file_path, "r") as f:
                env_data = yaml.safe_load(f)
        except FileNotFoundError:
            return {"conda": [], "pip": []}

    dependencies = env_data.get("dependencies", [])
    conda_packages = []
    pip_packages = []

    for dep in dependencies:
        # If dependency is a simple string, it is a conda package.
        if isinstance(dep, str):
            conda_packages.append(dep)
        # If dependency is a dictionary, it might be a pip section or something similar.
        elif isinstance(dep, dict):
            for key, pkgs in dep.items():
                if key.lower() == "pip":
                    # Assume pkgs is a list of pip packages.
                    pip_packages.extend(pkgs)

    return {"conda": conda_packages, "pip": pip_packages}


def snake_to_human(snake_str: str) -> str:
    """
    Convert a snake_case string to a human-readable string.

    Example:
        "hello_world" -> "Hello world"
    """
    # Replace underscores with spaces
    return snake_str.replace("_", " ")
