import asyncio
import os
from datetime import datetime
from typing import Dict

from aext_project_filebrowser_server.consts import (
    ANACONDA_PROJECTS_FOLDER,
    ENVIRONMENT,
    FILE_MAX_SIZE_100_MB,
    USER_JLAB_HOME_PATH,
)
from aext_project_filebrowser_server.schemas.local_project import FileInfo
from aext_project_filebrowser_server.utils import (
    human_readable_size,
    is_hidden_folder_or_file,
)

from aext_shared import Environment
from aext_shared import logger as custom_logger

logger = custom_logger.logger


class FilesystemOperations:
    # Limit how deep we can walk in folders
    MAX_DEPTH = {
        Environment.local_production: 10,
        Environment.local_development: 10,
        Environment.cloud_production: -1,
        Environment.cloud_development: -1,
        Environment.github_actions: -1,
    }.get(ENVIRONMENT)

    MAX_CONCURRENT_TASKS = 10

    async def process_file_info(self, file_path: str, abs_file_path: str) -> FileInfo:
        """
        Process file information asynchronously.
        file_path: relative file path
        abs_file_path: absolute path to the file
        """
        try:
            file_size_bytes = os.path.getsize(abs_file_path)
            if file_size_bytes >= FILE_MAX_SIZE_100_MB:
                logger.debug(f"Skipping {abs_file_path}: too big, max size is {FILE_MAX_SIZE_100_MB}")
                return None

            file_size_human_readable = human_readable_size(file_size_bytes)
            last_modified = datetime.fromtimestamp(os.path.getmtime(abs_file_path)).strftime("%Y-%m-%d %H:%M:%S.%f")
            return FileInfo(path=file_path, size=file_size_human_readable, last_modified=last_modified)
        except FileNotFoundError:
            logger.debug("Get file size: FileNotFound. Skipping it")
            return None

    async def list_all_user_files(self) -> Dict:
        """
        Lists all files in the specified base path using concurrent, batched processing.
        """
        all_files_and_dirs = {"files": []}
        current_depth = 0
        semaphore = asyncio.Semaphore(self.MAX_CONCURRENT_TASKS)  # concurrent taskss are limited

        async def process_directory(root, dirs, files):
            """
            Goes through all directories and find files within them, creating tasks that will be processed
            concurrently in order to get file information.
            """
            tasks = []
            # Filter directories
            dirs[:] = [
                d for d in dirs if d != ANACONDA_PROJECTS_FOLDER and not is_hidden_folder_or_file(os.path.join(root, d))
            ]

            for file in files:
                abs_file_path = os.path.join(root, file)
                if not is_hidden_folder_or_file(abs_file_path):
                    file_path = os.path.relpath(abs_file_path, USER_JLAB_HOME_PATH)
                    # use semaphore to limit concurrent tasks
                    tasks.append(self._process_file_semaphore(semaphore, file_path, abs_file_path))

            # Batch processing
            results = await asyncio.gather(*tasks)
            filtered_results = filter(None, results)
            all_files_and_dirs["files"].extend(filtered_results)  # Add non-None results

        try:
            for root, dirs, files in os.walk(USER_JLAB_HOME_PATH):
                await process_directory(root, dirs, files)
                current_depth += 1
                if 0 < self.MAX_DEPTH <= current_depth:
                    break
        except Exception as e:
            logger.exception(f"Error accessing directory {USER_JLAB_HOME_PATH}: {e}")
            return {}

        return all_files_and_dirs

    async def _process_file_semaphore(self, semaphore, file_path, abs_file_path):
        """
        Process files using asyncio semaphore in order to limit the number of concurrent tasks.

        semaphore: asyncio semaphore
        file_path: file relative path
        abs_file_path: absolute file path
        """
        async with semaphore:
            return await self.process_file_info(file_path, abs_file_path)


filesystem_operations = FilesystemOperations()
