import asyncio
import os
from pathlib import Path
from typing import Dict, Optional, Tuple, cast
from uuid import UUID

import httpx
from aext_project_filebrowser_server.consts import (
    FILEBROWSER_PROJECT_URL_RELATIVE,
    IS_WINDOWS,
    PROJECTS_FILES_DIR,
    InternalOrigins,
)
from aext_project_filebrowser_server.schemas.local_project import LocalProject
from watchdog.events import FileSystemEvent, FileSystemEventHandler
from watchdog.observers import Observer

from aext_shared import logger as custom_logger
from aext_shared.backend_proxy import CERTIFICATES_LOCATION

logger = custom_logger.logger


class WatchdogService(FileSystemEventHandler):
    """
    Wrapper for the Watchdog library (https://pypi.org/project/watchdog/).

    This class is mainly used to override the following watchdog methods:

        - on_created
        - on_deleted

    , which are responsible for handling file creation/deletion events.

    The watcher runs on a separate thread and can be initiated by calling the `watch` method (defined below).
    """

    active_observers = []

    @staticmethod
    def get_project_id_and_file_path_from_event(event: FileSystemEvent) -> Tuple[Optional[LocalProject], Optional[str]]:
        """
        Reads a watchdog event and retrieves both the associated project id and path to the file in question.

        Args:
            event: watchdog.events.FileSystemEvent (https://pythonhosted.org/watchdog/api.html#watchdog.events.FileSystemEvent)

        Returns: a tuple containing both the project id and file path.
        """
        (project_id, file_path) = (None, None)

        path_elements = cast(str, event.src_path).split(os.sep)
        for index, element in enumerate(path_elements):
            try:
                # force the cast, expect failure for non-UUIDs
                project_id = str(UUID(element))
                file_path = str(Path("/".join(path_elements[index + 1 :])))
            except ValueError:
                pass

        # do nothing if there is an error retrieving project id
        if not project_id:
            logger.warning(f"Error determining project_id: {event.src_path}")

        if not file_path:
            logger.warning(f"Error determining file path: {event.src_path}")

        return (project_id, file_path)

    @staticmethod
    def event_captures_checkpoint_file(event: FileSystemEvent) -> bool:
        """
        Reads a watchdog event and checks whether the file in question is a checkpoint file.

        Args:
            event: watchdog.events.FileSystemEvent

        Returns: a boolean indicating if the file is a checkpoint file.
        """
        # define the list of checkpoint names to look for
        checkpoint_names = ["checkpoint", "Untitled", "project_filebrowser.db-journal"]

        # initialize a flag for checkpoint files
        is_checkpoint_file = False

        # check if the event's source path contains any of the checkpoint names
        for checkpoint in checkpoint_names:
            if checkpoint in event.src_path:
                is_checkpoint_file = True
                break

        # check if the file name starts with a dot (hidden file)
        is_hidden_file = event.src_path.split("/")[-1].startswith(".")

        # return True if it's a checkpoint file or a hidden file
        return is_checkpoint_file or is_hidden_file

    @staticmethod
    def event_captures_directory(event: FileSystemEvent) -> bool:
        """
        Reads a watchdog event and checks whether the file in question is a directory.

        Args:
            event: watchdog.events.FileSystemEvent (https://pythonhosted.org/watchdog/api.html#watchdog.events.FileSystemEvent)

        Returns: a boolean indicating if the file is a directory.
        """
        return event.is_directory

    @staticmethod
    def get_server_info() -> Tuple[Optional[str], Optional[Dict]]:
        """
        Retrieves the server's url and token either via environment variables (priority) or jupyter's
        `serverapp` object (fallback).

        Args: (None)

        Returns: a tuple with both the server's `url` and `token`.
        """
        # default
        base_url = os.getenv("JUPYTERHUB_SERVICE_URL", None)
        auth_token = os.getenv("JUPYTERHUB_API_TOKEN", None)

        # fallback
        if not base_url or not auth_token:
            from jupyter_server import serverapp

            # The order of the elements change accordingly with the OS
            serverapps_list_index = -1
            if IS_WINDOWS:
                serverapps_list_index = 0

            logger.info(f"Serverapps list index: {serverapps_list_index}")
            serverapps = list(serverapp.list_running_servers())
            if serverapps:
                serverapp = serverapps[serverapps_list_index]
                base_url = serverapp.get("url", None)
                auth_token = serverapp.get("token", None)

        if not base_url or not auth_token:
            logger.warning(f"Server configuration information is missing: ({base_url, auth_token})")
            return (base_url, None)

        base_url = base_url.rstrip("/") + FILEBROWSER_PROJECT_URL_RELATIVE.rstrip("/")
        headers = {"Authorization": f"token {auth_token}", "X-Origin-Internal": InternalOrigins.WATCHDOG.value}

        logger.info(f"[Watchdog] Server info {base_url} {headers}")
        return (base_url, headers)

    def on_created(self, event: FileSystemEvent) -> None:
        """
        Reads a watchdog event, checks whether it represents a creation event, and acts as a handler if it is.

        This method performs a simple validation to ensure the file is not a checkpoint file or directory, and
        respectively inserts the file into the database.

        Args:
            event: watchdog.events.FileSystemEvent

        Returns: (None)
        """

        async def on_created_async():
            # do nothing for checkpoint files or directories
            if self.event_captures_directory(event) or self.event_captures_checkpoint_file(event):
                logger.debug(f"Either a checkpoint file or directory was created ({event.src_path}). Skipping...")
                return

            (project_id, file_path) = self.get_project_id_and_file_path_from_event(event)
            if not project_id or not file_path:
                return

            (base_url, headers) = self.get_server_info()
            if not base_url or not headers:
                return

            res = await self.http_client.post(
                url=f"{base_url}/{project_id}/file/create",
                headers=headers,
                json={"filename": file_path},
            )
            logger.info(f"[Watchdog] On Create {res}")
            # remove file if db insertion failed
            if res.status_code != httpx.codes.OK:
                Path(file_path).unlink()

        try:
            task = self.loop.create_task(on_created_async())
            self.loop.run_until_complete(task)
        except Exception as err:
            logger.warning(f"Watchdog exception raised while creating a file: {err}")
            logger.info("Continuing watcher thread..")

    def on_deleted(self, event: FileSystemEvent) -> None:
        """
        Reads a watchdog event, checks whether it represents a deletion event, and acts as a handler if it is.

        This method performs a simple validation to ensure the file is not a checkpoint file or directory, and
        respectively deletes the file from the database.

        Args:
            event: watchdog.events.FileSystemEvent

        Returns: (None)
        """

        async def on_deleted_async():
            # do nothing for checkpoint files or directories
            if self.event_captures_directory(event) or self.event_captures_checkpoint_file(event):
                logger.debug(f"Either a checkpoint file or directory was deleted ({event.src_path}). Skipping...")
                return

            (project_id, file_path) = self.get_project_id_and_file_path_from_event(event)
            if not project_id or not file_path:
                return

            (base_url, headers) = self.get_server_info()
            if not base_url or not headers:
                return

            res = await self.http_client.post(
                url=f"{base_url}/{project_id}/file/delete",
                headers=headers,
                json={"filenames": [file_path]},
            )
            logger.info(f"[Watchdog] On Delete {res}")

        try:
            task = self.loop.create_task(on_deleted_async())
            self.loop.run_until_complete(task)
        except Exception as err:
            logger.warning(f"Watchdog exception raised while deleting a file: {err}")
            logger.info("Continuing watcher thread..")

    def watch(self, path=PROJECTS_FILES_DIR, restart_on_stop: bool = True) -> None:
        """
        Defines the watcher's file path (PROJECTS_FILES_DIR), and initializes an observer on a separate thread.

        Args: (None)

        Returns: (None)
        """
        if self.active_observers:
            logger.warning(f"Unable to start watchdog thread as one is already active: {self.active_observers}")
            return None

        logger.info(f"Initializing a new watchdog thread: {self.active_observers}")
        self.http_client = httpx.AsyncClient(verify=CERTIFICATES_LOCATION)
        self.loop = asyncio.new_event_loop()
        self.observer = Observer()

        if restart_on_stop:
            self.observer.on_thread_stop = self._create_new_observer_thread

        self.observer.schedule(self, path=path, recursive=True)
        self.observer.start()

        self.active_observers.append(self.observer.ident)
        logger.info(f"Successfully initialized a watchdog thread: {self.active_observers}")

    def stop(self) -> None:
        """
        Stops the watcher and closes all relevant entities.

        Args: (None)

        Returns: (None)
        """
        if not self.active_observers:
            logger.warning("Attempted to stop a watchdog thread but no threads are currently running")
            return None

        watcher_id = self.observer.ident
        if watcher_id not in self.active_observers:
            logger.warning(f"Unable to stop non-existent watchdog thread: {watcher_id}")
            return None

        asyncio.run(self.http_client.aclose())
        self.loop.close()
        self.observer.stop()

        try:
            self.observer.join()
        except RuntimeError as err:
            logger.warning(f"Error while joining watchdog thread {err}")

        self.active_observers.remove(watcher_id)

    def _create_new_observer_thread(self) -> None:
        """
        Creates a new watcher thread by overriding self.observer.on_thread_stop.

        Args: (None)

        Returns: (None)
        """
        self.observer.unschedule_all()
        self.watch()


watchdog_service = WatchdogService()
