import json
import os
import shutil
import sys
import unittest
from pathlib import Path

import pytest
import tornado
from aext_project_filebrowser_server.consts import PROJECTS_FILES_DIR, FileOperations
from aext_project_filebrowser_server.schemas.cloud_project import CloudProjectFileList
from aext_project_filebrowser_server.schemas.local_project import LocalProject
from aext_project_filebrowser_server.services.conflict_resolution import FileDiff
from aext_project_filebrowser_server.services.local_storage import local_storage

from .setup_tests import BasicDBAppHTTPTests


class FileDiffTest(unittest.IsolatedAsyncioTestCase):

    def setUp(self):
        self.current_path = Path.cwd()
        self.test_assets = "tests/test_assets"
        self.file_1 = f"{self.current_path}/{self.test_assets}/file_diff_1.txt"
        self.file_2 = f"{self.current_path}/{self.test_assets}/file_diff_2.txt"
        self.identical_to_file_1 = f"{self.current_path}/{self.test_assets}/file_diff_1_identical.txt"

    async def test_two_identical_files_should_return_true(self):
        md5_identical_file = await FileDiff.calculate_md5(self.identical_to_file_1)
        self.assertTrue(await FileDiff.are_files_equal(self.file_1, md5_identical_file))

    async def test_two_non_identical_files_should_return_false(self):
        md5_file_2 = await FileDiff.calculate_md5(self.file_2)
        self.assertFalse(await FileDiff.are_files_equal(self.file_1, md5_file_2))


@pytest.mark.skipif(sys.platform.startswith("win"), reason="needs to be adapted")
class LocalStorageDiffTest(BasicDBAppHTTPTests):

    def setUp(self):
        super(LocalStorageDiffTest, self).setUp()
        self.current_path = Path.cwd()
        self.test_assets = "tests/test_assets"
        project_local_content = self.read_file(
            os.path.join(
                self.current_path,
                self.test_assets,
                "project-175bb011-b88b-4e04-b812-f0c2893aca05-local-content.json",
            )
        )
        project_cloud_files = json.loads(
            self.read_file(
                os.path.join(
                    self.current_path,
                    self.test_assets,
                    "project-175bb011-b88b-4e04-b812-f0c2893aca05-cloud-list-files.json",
                )
            )
        )
        insert_statement = (
            """INSERT INTO projects (id, data, owner, name, default_environment)
        VALUES ('175bb011-b88b-4e04-b812-f0c2893aca05', '%s', '3c01c0e2-1825-49c0-a371-ce9395c25f03', 'my-project', 'anaconda-kernel');
        """
            % project_local_content
        )
        self.db.execute(insert_statement)
        self.cloud_file_list = CloudProjectFileList.create_from_cloud_response(project_cloud_files)

        self.project_id = "175bb011-b88b-4e04-b812-f0c2893aca05"
        self.destination_project_dir = Path(f"{PROJECTS_FILES_DIR}/{self.project_id}")
        self.source_project_dir = Path(f"{self.current_path}/{self.test_assets}/projects/{self.project_id}")
        if self.destination_project_dir.exists() and self.destination_project_dir.is_dir():
            shutil.rmtree(self.destination_project_dir)
        shutil.copytree(self.source_project_dir, self.destination_project_dir)

    @tornado.testing.gen_test
    async def test_create_project(self):
        """
        Testing scenario
        Locally the project contains the following files:
        - images/image-a.jpeg
        - txt/test-a.txt

        On cloud the project contains:
        - images/image-a.jpeg
        - images/image-b.jpeg
        The only common file is images/image-a.jpeg and the content is the same.
        Therefore actions should be taken for txt/test-a.txt and images/image-b.jpeg
        depending on the operation.
        For PUSH: txt/test-a.txt should be written on cloud and images/image-b.jpeg deleted on cloud
        For PULL: images/image-b.jpeg should be written locally and txt/test-a.txt should be deleted locally
        """
        project = LocalProject.get(self.project_id)
        actions = await local_storage.diff(project.id, self.cloud_file_list)
        push = actions.push_operations
        pull = actions.pull_operations
        assert push["images/jpeg/image-b.jpeg"] == FileOperations.CLOUD_DELETE
        assert push["txt/test-a.txt"] == FileOperations.CLOUD_WRITE
        assert pull["images/jpeg/image-b.jpeg"] == FileOperations.LOCAL_WRITE
        assert pull["txt/test-a.txt"] == FileOperations.LOCAL_DELETE
