import sys

import pytest
import tornado
from aext_project_filebrowser_server.consts import FileOperations
from aext_project_filebrowser_server.handlers import project_service
from aext_project_filebrowser_server.schemas.local_project import LocalProject
from aext_project_filebrowser_server.services.local_storage import (
    FileOperationsDict,
    local_storage,
)

from .setup_tests import BasicDBAppHTTPTests
from .test_assets.project_data import EMPTY_LOCAL_PROJECT
from .test_assets.project_files_data import PAYLOAD_1


class TestDiffFileOperations:
    def setup_method(self):
        self.operations_dict = FileOperationsDict(
            {
                "a.txt": FileOperations.LOCAL_DELETE,
                "b.txt": FileOperations.LOCAL_WRITE,
                "c.txt": FileOperations.CLOUD_DELETE,
                "d.txt": FileOperations.CLOUD_WRITE,
            }
        )

    def test_filter_by_operation_type(self):
        ops_and_expected = zip(list(FileOperations), ["a.txt", "b.txt", "c.txt", "d.txt"])
        for operation, expected in ops_and_expected:
            assert len(self.operations_dict.filter_by_operation_type(operation)) == 1
            assert expected in self.operations_dict.filter_by_operation_type(operation)

    def test_filter_local_operations(self):
        expected = {"a.txt", "b.txt"}
        assert len(self.operations_dict.filter_local_operations()) == 2
        assert expected - set(self.operations_dict.filter_local_operations()) == set()

    def test_filter_cloud_operations(self):
        expected = {"a.txt", "b.txt"}
        assert len(self.operations_dict.filter_cloud_operations()) == 2
        assert expected - set(self.operations_dict.filter_local_operations()) == set()


@pytest.mark.skipif(sys.platform.startswith("win"), reason="needs to be adapted")
class TestUpdateProjectMetadata(BasicDBAppHTTPTests):
    metadata = {"user_client": "TEST_USER", "hidden": False, "tags": ["TEST_TAG"]}

    def test_update_project_metadata(self):
        local_project = self.create_empty_project()
        assert local_project.data.metadata.to_dict() == EMPTY_LOCAL_PROJECT["metadata"]

        assert local_storage.update_project_metadata(local_project.id, self.metadata)

        local_project = LocalProject.get(local_project.id)
        assert local_project is not None
        assert local_project.data.metadata.to_dict() == self.metadata


@pytest.mark.skipif(sys.platform.startswith("win"), reason="needs to be adapted")
class TestContainsSnippetFiles(BasicDBAppHTTPTests):
    @tornado.testing.gen_test
    async def test_contains_snippet_files(self):
        local_project = self.create_project_with_code_snippet_file()
        assert local_project.contains_snippet_files()

        local_project.remove_file_or_folder("tmp/project/foo.txt")
        assert not local_project.contains_snippet_files()
