import contextlib
import json
import sys
from unittest import mock

import pytest
from aext_project_filebrowser_server.schemas.local_project import LocalProject
from tornado.httpclient import HTTPRequest
from tornado.testing import gen_test
from tornado.websocket import websocket_connect

from .setup_tests import BasicAppWebSocketsTests
from .test_assets.project_data import LOCAL_PROJECT
from .test_assets.responses import EXPECTED_LIST_PROJECTS

pytestmark = pytest.mark.skipif(sys.platform.startswith("win"), reason="Skipping tests on Windows")


class WebSocketTest(BasicAppWebSocketsTests):
    @pytest.fixture(scope="function", autouse=True)
    def pre_populated_db(self):
        self.setUp()

        # Add pre-populated data
        project = LocalProject.create(LOCAL_PROJECT)
        project.save()

    @gen_test
    def test_list_projects(self):
        with mock.patch("aext_project_filebrowser_server.handlers.get_username") as m:
            m.return_value = "62e920db-1a14-41ca-93a2-98b36dd695c1"

            with contextlib.closing(
                (
                    yield websocket_connect(
                        HTTPRequest(
                            f"ws://127.0.0.1:{self.get_http_port()}/project/list",
                            headers={"Authorization": "token abc"},
                        )
                    )
                )
            ) as ws:

                ws.write_message("start")
                response = yield ws.read_message()
                response_parsed = json.loads(response)

                self.assertEqual(response_parsed.get("http_status", ""), 200)
                self.assertEqual(response_parsed.get("message", ""), "start")
                self.assertEqual(response_parsed.get("data", []), EXPECTED_LIST_PROJECTS)

    @gen_test
    def test_list_projects_wrong_user(self):
        with mock.patch("aext_project_filebrowser_server.handlers.get_username") as m:
            m.return_value = "unknown"

            with contextlib.closing(
                (
                    yield websocket_connect(
                        HTTPRequest(
                            f"ws://127.0.0.1:{self.get_http_port()}/project/list",
                            headers={"Authorization": "token abc"},
                        )
                    )
                )
            ) as ws:

                ws.write_message("start")
                response = yield ws.read_message()
                response_parsed = json.loads(response)

                self.assertEqual(response_parsed.get("http_status", -1), 101)
                self.assertEqual(
                    response_parsed.get("http_message", ""), "User not found or token not valid. Closing connection."
                )
                self.assertEqual(response_parsed.get("message", None), None)
                self.assertEqual(response_parsed.get("data", None), None)
