from abc import ABCMeta, abstractmethod
from typing import Any, ClassVar, Dict, List, Optional

from aext_project_filebrowser_server.schemas.cloud_project import (
    CloudProject,
    CloudProjectFileList,
    CloudProjectFileTree,
    CloudProjectPermissions,
    MyProjectPermissions,
    ProjectPermissionUpdate,
    UserProjectPermissions,
)
from aext_project_filebrowser_server.schemas.organizations import OrganizationMembers


class ProjectAPIClientInterface(metaclass=ABCMeta):
    """
    An abstract interface for a Project API client.

    Potential concrete implementations:
        - HTTP client;
        - Anaconda Cloud Storage client (https://github.com/anaconda/anaconda-cloud-tools/tree/main/libs/anaconda-cloud-storage).
    """

    methods: ClassVar[List[str]] = [
        "get_project",
        "get_projects",
        "get_project_filetree",
        "list_project_files",
        "download_file",
        "create_project",
        "delete_project",
        "update_project",
    ]

    @classmethod
    def __subclasshook__(cls, __subclass: type) -> bool:
        return (
            hasattr(__subclass, method) and callable(__subclass, method) for method in cls.methods
        ) or NotImplemented

    @abstractmethod
    async def get_project(self, project_id: str, **kwargs) -> Optional[CloudProject]:
        """
        Retrieves a project associated with the user.

        :raises: NotImplementedError
        """
        raise NotImplementedError

    @abstractmethod
    async def get_projects(self, **kwargs) -> List[CloudProject]:
        """
        Retrieves all projects associated with the user.

        :raises: NotImplementedError
        """
        raise NotImplementedError

    @abstractmethod
    async def get_project_filetree(self, project_id: str, **kwargs) -> Optional[CloudProjectFileTree]:
        """
        Get project file tree representation.

        Parameters
        __________
        project_id: a str representing a project id.
        project_name: a str representing a project name.

        :raises: NotImplementedError
        """
        raise NotImplementedError

    @abstractmethod
    async def list_project_files(self, project_id: str, **kwargs) -> Optional[CloudProjectFileList]:
        """
        Retrieves the list of files belonging to the project
        Args:
            project_id: id of the project
            **kwargs:

        Returns: Optionally returns the list of files

        """
        raise NotImplementedError

    @abstractmethod
    async def download_file(self, project_id: str, filename: str, **kwargs) -> Optional[bytes]:
        """
        Download a specific file belonging to the project.

        Parameters
        __________
        project_id: a UUID representing a user project id
        filename: name of the file to be downloaded

        :raises: NotImplementedError
        """
        raise NotImplementedError

    @abstractmethod
    async def create_project(
        self, project_title: str, metadata: Optional[Dict] = None, **kwargs
    ) -> Optional[CloudProject]:
        """
        Create a new project.

        Parameters
        __________
        project_title: project's title

        :raises: NotImplementedError
        """
        raise NotImplementedError

    @abstractmethod
    async def delete_project(self, project_id: str, **kwargs) -> bool:
        """
        Delete a project.

        Parameters
        __________
        project_id: The unique identifier of the project to be deleted in UUID format

        :raises: NotImplementedError
        """
        raise NotImplementedError

    @abstractmethod
    async def update_project(self, project_id: str, payload: Any, **kwargs) -> CloudProject:
        """
        Update a project.

        Parameters
        __________
        project_id: The unique identifier of the project to be updated in UUID format
        payload: Request body containing the following optional fields: name, title, description, metadata

        :raises: NotImplementedError
        """
        raise NotImplementedError

    @abstractmethod
    async def list_organization_users(self, organization_name: str) -> OrganizationMembers:
        """
        List all members of an organization
        Args:
            organization_name: organization id

        Returns: OrganizationMembers object

        """
        raise NotImplementedError

    @abstractmethod
    async def list_project_permissions(self, project_id: str) -> CloudProjectPermissions:
        """
        List a project's set of permissions (sharing feature)
        Args:
            project_id: id of the project

        Returns: CloudProjectPermissions

        Raises: NotImplementedError

        """
        raise NotImplementedError

    @abstractmethod
    async def update_project_permissions(self, project_id: str, permissions: List[ProjectPermissionUpdate]) -> bool:
        """
        Update a project's set of permissions (sharing feature)
        Args:
            project_id: id of the project
            permissions: a set of permission update object

        Returns: bool

        Raises: NotImplementedError

        """
        raise NotImplementedError

    @abstractmethod
    async def update_file_metadata(
        self, project_id: str, filename: str, metadata: Dict, description: Optional[str] = ""
    ) -> bool:
        """
        Updates file's metadata
        Args:
            project_id: id of the project that the file belongs to
            filename: file's filename (key)
            metadata: dict containing all the metadata
            description: file description

        Returns: bool

        Raises: NotImplementedError
        """
        raise NotImplementedError

    @abstractmethod
    async def list_user_permissions(self, project_id: str) -> UserProjectPermissions:
        """
        List all the permissions from all users related to a project. Only owners can fetch this info.
        Args:
            project_id: id of the project that the file belongs to

        Returns: List

        Raises: NotImplementedError
        """
        raise NotImplementedError

    @abstractmethod
    async def list_my_permissions(self, project_id: str) -> MyProjectPermissions:
        """
        List user's permissions for a particular project

        Return: MyProjectPermissions

        Raises: NotImplementedError
        """
        raise NotImplementedError
