""" Logging configs and extensions. """

from __future__ import annotations

import logging.config
import sys
import threading
import types
import typing

from anaconda_navigator.utils.logs.common import *
from anaconda_navigator.utils.logs.config import LOGGER_CONFIG
from anaconda_navigator.utils.logs.loggers import *


def global_exception_logger(
        exc_type: typing.Type[BaseException],
        exception: BaseException,
        traceback: types.TracebackType | None,
) -> None:
    """Handle an exception by logs it."""
    logger.exception(exception, exc_info=(exc_type, exception, traceback))


def _sys_unraisablehook(args: 'sys.UnraisableHookArgs') -> None:
    """Adapter for sys.unraisablehook."""
    if args.exc_value:
        global_exception_logger(args.exc_type, args.exc_value, args.exc_traceback)


def _threading_excepthook(args: threading.ExceptHookArgs) -> None:
    """Adapter for threading.excepthook."""
    if args.exc_value and args.exc_type not in (SystemExit, KeyboardInterrupt):
        global_exception_logger(args.exc_type, args.exc_value, args.exc_traceback)


def setup_logger() -> None:
    """Setup, create, and set logger."""
    logging.config.dictConfig(LOGGER_CONFIG.dict_config)

    # Note: threading.excepthook is only supported since Python 3.8
    sys.excepthook = global_exception_logger
    sys.unraisablehook = _sys_unraisablehook
    threading.excepthook = _threading_excepthook

    logger.debug('Setting up logger')
