# -*- coding: utf-8 -*-

# pylint: disable=import-outside-toplevel

"""Tests for utilities used by CloudAPI."""

from __future__ import annotations

__all__ = ()

import itertools
import json
import os
import tempfile
import typing
from unittest import mock
from urllib import parse
import uuid
import pytest


def test_endpoints() -> None:
    """Check :class:`~anaconda_navigator.api.cloud.utilities.endpoints.EndpointCollection` behavior."""
    from anaconda_navigator.api.cloud import utilities

    base_url: str = 'https://example.org/'
    source: typing.Any = {
        'top_level_link': 'first',
        'group': {
            'group_link': 'second',
            'extra_link': 'third',
        },
        'extra_link': 'fourth',
    }

    instance = utilities.EndpointCollection(base_url=base_url, endpoints=source)
    assert instance.base_url == base_url
    assert instance.group.base_url == base_url

    assert len(instance) == 3
    assert len(instance.group) == 2

    assert set(iter(instance)) == {'top_level_link', 'group', 'extra_link'}
    assert set(iter(instance.group)) == {'group_link', 'extra_link'}

    assert instance['top_level_link'] == parse.urljoin(base_url, source['top_level_link'])
    assert instance.top_level_link == parse.urljoin(base_url, source['top_level_link'])

    assert instance['extra_link'] == parse.urljoin(base_url, source['extra_link'])
    assert instance.extra_link == parse.urljoin(base_url, source['extra_link'])

    assert instance['group']['group_link'] == parse.urljoin(base_url, source['group']['group_link'])
    assert instance['group'].group_link == parse.urljoin(base_url, source['group']['group_link'])
    assert instance.group['group_link'] == parse.urljoin(base_url, source['group']['group_link'])
    assert instance.group.group_link == parse.urljoin(base_url, source['group']['group_link'])

    assert instance['group']['extra_link'] == parse.urljoin(base_url, source['group']['extra_link'])
    assert instance['group'].extra_link == parse.urljoin(base_url, source['group']['extra_link'])
    assert instance.group['extra_link'] == parse.urljoin(base_url, source['group']['extra_link'])
    assert instance.group.extra_link == parse.urljoin(base_url, source['group']['extra_link'])

    with pytest.raises(AttributeError):
        _ = instance.some_non_existing_attribute


def test_json_streaming() -> None:
    """Check :mod:`~anaconda_navigator.api.cloud.utilities.json_streaming` behavior."""
    from anaconda_navigator.api.cloud import utilities

    text_path: str = os.path.join(tempfile.gettempdir(), 'AAA_')  # uuid.uuid4().hex
    text_value: str = ''.join(uuid.uuid4().hex for _ in range(8192))
    serialized_path: str = os.path.join(tempfile.gettempdir(), 'BBB_')

    stream: typing.TextIO
    with open(text_path, 'wt', encoding='utf-8') as temp_stream:
        temp_stream.write(text_value)

    with open(text_path, 'rt', encoding='utf-8') as stream:
        original_value = {
            'str': 'He\tllo\nWorld!',
            'mapping': {
                'int': 12,
                'sequence': [True, False, None, 3.14, 2.71],
            },
            'text': stream,
        }

        writer: utilities.JsonWriter = utilities.JsonWriter(serialized_path)
        writer.serialize(original_value)
        writer.close()

    deserialized_value: typing.Any
    original_value['text'] = text_value
    with open(serialized_path, 'rt', encoding='utf-8') as temp_stream:
        deserialized_value = json.load(temp_stream)
    assert deserialized_value == original_value

    os.remove(text_path)
    os.remove(serialized_path)


def test_sessions() -> None:
    """Check :mod:`~anaconda_navigator.api.cloud.utilities.sessions` behavior."""
    from anaconda_navigator.api.cloud import utilities

    request = mock.Mock()
    with mock.patch('anaconda_navigator.api.cloud.utilities.sessions.requests.request', request):
        instance: utilities.Session = utilities.Session()

        initial: typing.Set[str] = set(instance.headers.copy())

        # Check collections

        instance.headers['Content-Length'] = '123456'
        assert set(instance.headers.copy()) == initial | {'Content-Length'}
        assert instance.headers['Content-Length'] == '123456'

        instance.headers['Content-Type'] = 'application/json'
        assert set(instance.headers.copy()) == initial | {'Content-Length', 'Content-Type'}
        assert instance.headers['Content-Length'] == '123456'
        assert instance.headers['Content-Type'] == 'application/json'

        del instance.headers['Content-Length']
        assert set(instance.headers.copy()) == initial | {'Content-Type'}
        assert instance.headers['Content-Type'] == 'application/json'
        with pytest.raises(KeyError):
            _ = instance.headers['Content-Length']

        # Check requests

        method: str = 'GET'
        url: str = 'https://example.org/'

        instance.request(method, url)

        request.assert_called_with(
            method,
            url,
            cookies=instance.cookies.copy(),
            headers=instance.headers.copy(),
            timeout=30,
        )
        request.reset_mock()

        # --- #

        method = 'POST'
        url = 'https://anaconda.org/'
        headers: typing.Dict[str, str] = {'X-Property': uuid.uuid4().hex}

        assert instance.request(method, url, headers=headers) is request.return_value

        headers = dict(itertools.chain(instance.headers.copy().items(), headers.items()))
        request.assert_called_with(
            method,
            url,
            cookies=instance.cookies.copy(),
            headers=headers,
            timeout=30,
        )
        request.reset_mock()

        # --- #

        method = 'PUT'
        url = 'https://anaconda.com/'
        cookies: typing.Dict[str, str] = {'Session': uuid.uuid4().hex}

        assert instance.request(method, url, cookies=cookies) is request.return_value

        cookies = dict(itertools.chain(instance.cookies.copy().items(), cookies.items()))
        request.assert_called_with(
            method,
            url,
            cookies=cookies,
            headers=instance.headers.copy(),
            timeout=30,
        )
        request.reset_mock()

        # --- #

        method = 'PATCH'
        url = 'https://anaconda.com/'
        cookies = {'Session': uuid.uuid4().hex}
        headers = {'Authorization': uuid.uuid4().hex}

        assert instance.request(
            method,
            url,
            cookies=cookies,
            headers=headers,
            raise_for_status=True,
            extra_a=1,
            extra_b='yes',
        ) is request.return_value

        cookies = dict(itertools.chain(instance.cookies.copy().items(), cookies.items()))
        headers = dict(itertools.chain(instance.headers.copy().items(), headers.items()))
        request.assert_called_with(
            method,
            url,
            cookies=cookies,
            headers=headers,
            timeout=30,
            extra_a=1,
            extra_b='yes',
        )
        request.return_value.raise_for_status.assert_called_with()
        request.reset_mock()
